//********************************************************************************
//* File       : BuildCart.cpp                                                   *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2024      Mahlon R. Smith, The Software Samurai   *
//*                 GNU GPL copyright notice below                               *
//* Date       : 20-Jun-2024                                                     *
//* Version    : (see appVersion member)                                         *
//*                                                                              *
//* Description: Build a pseudo-cartouche around the user-specified text.        *
//*              This is used in Texinfo documentation (.info format).           *
//*                                                                              *
//*                                                                              *
//*                                                                              *
//*                                                                              *
//*                                                                              *
//* Development Tools:                                                           *
//* -----------------------------------                                          *
//*  GNU G++ v:13.3.1 20240522 (C++17)                                           *
//*  Fedora Linux v:39                                                           *
//*  GNOME Terminal v:3.50.1 (GNOME 45)                                          *
//*                                                                              *
//********************************************************************************
//* Copyright Notice:                                                            *
//* This program is free software: you can redistribute it and/or modify it      *
//* under the terms of the GNU General Public License as published by the Free   *
//* Software Foundation, either version 3 of the License, or (at your option)    *
//* any later version.                                                           *
//*                                                                              *
//* This program is distributed in the hope that it will be useful, but          *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License     *
//* for more details.                                                            *
//*                                                                              *
//* You should have received a copy of the GNU General Public License along      *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.              *
//*                                                                              *
//********************************************************************************
//* Version History:                                                             *
//*                                                                              *
//* v: 0.00.01 14-Jun-2024                                                       *
//*    -- This application, including the documentation was written in six days, *
//*       part-time. It is therefore possible (despite our magnificent skills)   *
//*       that there may be a few typos, rough edges, or (gasp!) bugs.           *
//*       These will be addressed as the application is integrated into the      *
//*       larger Infodoc Styles package, available as a seperate download.       *
//*    -- This application was created to scratch an itch the author has had for *
//*       some time regarding the unrealized potential of the Texinfo @cartouche *
//*       command.                                                               *
//*       While the cartouche object looks great in the HTML format, it is       *
//*       literally worthless in the '.info' document format.                    *
//*       Our Infodoc Styles package includes two macros to address the issue:   *
//*       @Cartouche and @CartHtml; however, we found that the '.info' cartouche *
//*       object could be significantly improved by simply replacing it with a   *
//*       manually-constructed box to enclose the the selected text.             *
//*       Unfortunately, this manual operation is tedious and time-consuming;    *
//*       therefore, welcome to Build Cartouche.                                 *
//*    -- This application relies heavily on our gString text-analysis tool which*
//*       parses the source data, reformats the data to the specified dimensions,*
//*       and then _almost_ effortlessly draws the border around the text block. *
//*                                                                              *
//*                                                                              *
//********************************************************************************

//*****************
//* Include Files *
//*****************
#include "BuildCart.hpp"      //* Application class and misc.definitions

//***************
//* Definitions *
//***************

//**************
//* Local data *
//**************

//********************
//* Local prototypes *
//********************


//*************************
//*        main           *
//*************************
//********************************************************************************
//* Program entry point.                                                         *
//*                                                                              *
//********************************************************************************

int main ( int argc, const char* argv[], const char* argenv[] )
{
   short status = ZERO ;                  // exit value

   //* Create the application class.*
   BuildCart* bc = new BuildCart ( argc, argv, argenv ) ;

   //* Get operational status.*
   status = bc->exitStatus () ;

   //* Exit, returning control to the shell.*
   exit ( status ) ;

}  //* End main() *

//*************************
//*       BuildCart       *
//*************************
//********************************************************************************
//* Constructor.                                                                 *
//*                                                                              *
//* Input  :                                                                     *
//*                                                                              *
//* Returns:                                                                     *
//********************************************************************************

BuildCart::BuildCart ( int argc, const char* argv[], const char* argenv[] )
{
   gString gs ;                           // text formatting

   //* Initialize data members.*
   this->ioLocale = NULL ;
   *this->srcText = NULLCHAR ;
   *this->trgText = NULLCHAR ;
   *this->sFile   = NULLCHAR ;
   *this->tFile   = NULLCHAR ;
   this->stylePtr = lnSingle ;
   this->width    = wMIN ;
   this->minHgt   = hMIN ;
   this->pad      = padDFLT ;
   this->status   = ZERO ;
   this->leadRow  = false ;
   this->goodtxt  = false ;

   //* Set the locale from the environment.*
   this->ioLocale = new locale("") ;
   this->ioLocale->global( *this->ioLocale ) ;

   //* Clear the terminal window.*
   this->clearTerm () ;

   //* If valid command-line arguments, and  *
   //* not 'help' or 'version'.              *
   if ( (this->GetCommlineArgs ( argc, argv )) )
   {
      if ( this->goodtxt )
         this->Cart () ;
      else
         this->EmptyCart () ;
   }

}  //* End BuildCart() *

//*************************
//*         Cart          *
//*************************
//********************************************************************************
//* Build a cartouche around the specified text.                                 *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void BuildCart::Cart ( void )
{
   #define DEBUG_CART (0)                 // for debugging only

   gString gsPad, gs ;                    // text formatting
   short tindx,                           // target index
         ri ;                             // row index
   gsPad.padCols( this->pad ) ;           // horizontal padding

   //* Instantiate and initialize text-analysis data. *
   //* If pre-formatted text is wider that maximum,  *
   //* AND if target width not specified, set width to maximum.
   AzSource azs( this->pad ) ;
   azs.Analyze ( this->srcText ) ;
   if ( (azs.maxtxtCols > azs.maxtxtWidth) && (this->width == wMIN) )
      this->width = wMAX ;

   //* If a target width specified, reformat the source text to fit.*
   if ( this->width > wMIN )
      this->ReformatSource ( azs ) ;

   #if DEBUG_CART != 0
   gString gsdbg ;
   gsdbg.compose( "Cart: azs: charCnt:%hd rowCnt:%hd maxtxtWidth:%hd maxtxtCols:%hd "
                  "nontxtCols:%hd cartWidth:%hd\n", 
                  &azs.charCnt, &azs.rowCnt, &azs.maxtxtWidth, &azs.maxtxtCols, 
                  &azs.nontxtCols, &azs.cartWidth ) ;
   this->write ( gsdbg ) ;
   #if 0    // Verbose Output
   for ( short i = ZERO ; i < azs.rowCnt ; ++i )
   {
      gsdbg.compose( "  %02hd) '%S'", &i, azs.wbuff[i] ) ;
      gsdbg.padCols( azs.maxtxtCols + 9 ) ;
      gsdbg.append( "cols:%02hd\n", &azs.rowCols[i] ) ;
      this->write ( gsdbg ) ;
   }
   #endif   // Verbose Output
   #endif   // DEBUG_CART

   //* Top Border *
   tindx = ZERO ;
   gs.compose( "%C", &this->stylePtr[ciUL] ) ;
   gs.padCols( azs.cartWidth - 1, this->stylePtr[ciHZ] ) ;
   gs.append( "%C\n", &this->stylePtr[ciUR] ) ;
   gs.copy( &this->trgText[tindx], gsMAXCHARS ) ;
   tindx = gs.gschars() - 1 ;

   //* Text Rows *
   for ( ri = ZERO ; ri < azs.rowCnt ; ++ri )
   {
      gs.compose( "%C%S%S", 
                  &this->stylePtr[ciVT],     // vertical bar
                  gsPad.gstr(),              // left-side padding
                  azs.wbuff[ri] ) ;          // row text
      gs.padCols( azs.maxtxtCols + this->pad + 1 ) ;
      gs.append( "%S%C\n",
                  gsPad.gstr(),              // right-side padding
                  &this->stylePtr[ciVT] ) ;  // vertical bar
      gs.copy( &this->trgText[tindx], gsMAXCHARS ) ;
      tindx += gs.gschars() - 1 ;
   }

   //* If specified, append empty rows.*
   if ( (azs.rowCnt + 2) < this->minHgt )
   {
      gs.compose( "%C", &this->stylePtr[ciVT] ) ;
      gs.padCols( azs.maxtxtCols + (this->pad * 2) + 1 ) ;
      gs.append( "%C\n", &this->stylePtr[ciVT] ) ;
      ri = azs.rowCnt + 2 ;
      while ( ri < this->minHgt )
      {
         gs.copy( &this->trgText[tindx], gsMAXCHARS ) ;
         tindx += gs.gschars() - 1 ;
         ++ri ;
      }
   }

   //* Bottom Border *
   gs.compose( "%C", &this->stylePtr[ciLL] ) ;
   gs.padCols( azs.cartWidth - 1, this->stylePtr[ciHZ] ) ;
   gs.append( "%C\n", &this->stylePtr[ciLR] ) ;
   gs.copy( &this->trgText[tindx], gsMAXCHARS ) ;
   tindx += gs.gschars() ;    // (total character written to output)

   //* Write the cartouche to stdout.*
   this->write ( this->trgText ) ;
   this->write ( L"\n" ) ;

   //* If a file specified, also write to target file.*
   if ( *this->tFile != NULLCHAR )
      this->WriteTargetFile () ;

   #undef DEBUG_CART
}  //* End Cart() *

//*************************
//*       EmptyCart       *
//*************************
//********************************************************************************
//* Create an empty cartouche object.                                            *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void BuildCart::EmptyCart ( void )
{
   short tindx = ZERO ;

   //* Top Border *
   gString gs( "%C", &this->stylePtr[ciUL] ) ;
   gs.padCols( this->width - 1, this->stylePtr[ciHZ] ) ;
   gs.append( "%C\n", &this->stylePtr[ciUR] ) ;
   gs.copy( &this->trgText[tindx], gsMAXCHARS ) ;
   tindx = gs.gschars() - 1 ;

   //* Text Rows *
   gs.compose( "%C", &this->stylePtr[ciVT] ) ;
   gs.padCols( this->width - 1 ) ;
   gs.append( "%C\n", &this->stylePtr[ciVT] ) ;
   for ( short r = this->minHgt - 2 ; r > ZERO ; --r )
   {
      gs.copy( &this->trgText[tindx], gsMAXCHARS ) ;
      tindx += gs.gschars() - 1 ;
   }

   //* Bottom Border *
   gs.compose( "%C", &this->stylePtr[ciLL] ) ;
   gs.padCols( this->width - 1, this->stylePtr[ciHZ] ) ;
   gs.append( "%C\n", &this->stylePtr[ciLR] ) ;
   gs.copy( &this->trgText[tindx], gsMAXCHARS ) ;

   //* Write the cartouche to stdout.*
   this->write ( this->trgText ) ;
   this->write ( L"\n" ) ;

   //* If a file specified, also write to target file.*
   if ( *this->tFile != NULLCHAR )
      this->WriteTargetFile () ;

}  //* End EmptyCart() *

//*************************
//*    ReformatSource     *
//*************************
//********************************************************************************
//* Re-flow source text to fit specified width 'width' - nontext-columns.        *
//* Remove all existing newline characters, then insert line breaks to ensure    *
//* that each row has <= max row columns.                                        *
//*                                                                              *
//* Input  : azs : (by reference) contains analysis of current data format       *
//*                receives statistical data for reformatted text.               *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void BuildCart::ReformatSource ( AzSource& azs )
{
   gString gs ;                           // text analysis and reformatting
   short rowwidth = (this->width - azs.nontxtCols) ; // max width of a text ros
   short stail ;                          // char count for partial row

   //* If the source data fit within a gString *
   //* object, then reformat in one step.      *
   if ( azs.charCnt < gsMAXCHARS )
   {
      gs = this->srcText ;
      gs.formatParagraph( (hMAX - 2), rowwidth, false ) ;
      gs.copy( this->srcText, gsMAXCHARS ) ;
   }
   //* Reformat the data in multiple segments. *
// CZONE - ReformatSource()
   else
   {
      wchar_t* tmpbuff = new wchar_t[srcBUFFER + 1] ; // temporary buffer
      short indx,                         // newline index
            si = ZERO, ti = ZERO ;        // source/target indices

      //* Copy source data to temp buffer *
      for ( si = ZERO ; this->srcText[si] != NULLCHAR ; ++si )
         tmpbuff[si] = this->srcText[si] ;
      tmpbuff[si] = NULLCHAR ;

//* TEMP */ short gsch ; gString gsdbg( "charCnt:%hd\n", &azs.charCnt ) ; this->write ( gsdbg ) ;
      gs.clear() ;                        // clear the work buffer
      si = ZERO ;                         // reset the index
      while ( si < azs.charCnt )
      {
// calculate 'si' advance: written chars MINUS inserted newlines
         stail = gs.gschars() - 1 ;
         gs.loadChars( &this->srcText[si], 400, true ) ;
//* TEMP */ gsch = gs.gschars() ;
//* TEMP */ gsdbg.compose( "READ: si:%03hd gsch:%03hd\n'%S'*\n", &si, &gsch, gs.gstr() ) ;  this->write ( gsdbg ) ;
         si += gs.gschars() - stail - 1 ;
         gs.formatParagraph( (hMAX - 2), rowwidth, false ) ;
//* TEMP */ gsch = gs.gschars() ;
//* TEMP */ gsdbg.compose( "FORM: si:%03hd gsch:%03hd\n'%S'*\n", &si, &gsch, gs.gstr() ) ;  this->write ( gsdbg ) ;
         if ( (indx = gs.findlast( NEWLINE )) > ZERO )
         {
            ++indx ;                      // step past the newline
// strip trailing newline (if any).
            gs.copy( &this->srcText[ti], (indx + 1) ) ;
//* TEMP */ gsdbg = gs ; gsdbg.limitChars( indx ) ; gsdbg.insert( L"COPY:\n'" ) ;
//* TEMP */ gsdbg.append( L"'*\n" ) ; this->write ( gsdbg ) ;
            ti += indx ;
            gs.shiftChars( -(indx) ) ;    // copy full rows
//* TEMP */ gsch = gs.gschars() ;
//* TEMP */ gsdbg.compose( "SHFT: si:%03hd gsch:%03hd\n'%S'*\n", &si, &gsch, gs.gstr() ) ;  this->write ( gsdbg ) ;
         }
         else  // copy the last (partial) row
         {
            gs.copy( &this->srcText[ti], gsMAXCHARS ) ;
            break ;
         }
      }
//* TEMP */ gsch = gs.gschars() ;
//* TEMP */ gsdbg.compose( "LAST: si:%03hd gsch:%03hd '%S'*\n\n", &si, &gsch, gs.gstr() ) ;  this->write ( gsdbg ) ;

      delete [] tmpbuff ;                 // release the dynamic allocation
   }

   //* Analyze the reformatted data.*
   azs.Analyze ( this->srcText ) ;

}  //* End ReformatSource()  *

//*************************
//*         write         *
//*************************
//********************************************************************************
//* Write text to stdout (wide stream only) ;                                    *
//*                                                                              *
//*                                                                              *
//* Input  : txt : text data to be written                                       *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void BuildCart::write ( const uint8_t* txt )
{
   wcout << txt ; wcout.flush() ;
}  //* End write() *
void BuildCart::write ( const wchar_t* txt )
{
   wcout << txt ; wcout.flush() ;
}  //* End write() *
void BuildCart::write ( const gString& txt )
{
   wcout << txt.gstr() ; wcout.flush() ;
}  //* End write() *

//*************************
//*    GetCommlineArgs    *
//*************************
//********************************************************************************
//* Parse the command-line arguments.                                            *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* Programmer's Note: Traditionally, the lowercase 'h' is used to invoke Help.  *
//* However, here it is used here for 'height'. Note however, that if the user   *
//* invokes 'bcart -h' hoping for help, they will actually be invoking the height*
//* option, but without an argument. This error will trigger Help. Cool!         *
//*                                                                              *
//********************************************************************************

bool BuildCart::GetCommlineArgs ( int argc, const char* argv[] )
{
   #define DEBUG_GCLA (0)                 // for debugging only
   short tmpval,                          // capture numeric arguments
         validArgs = 1 ;                  // number of valid arguments (incl. app name)
   bool ver    = false,                   // set if version # requested
        hlp    = false,                   // set if help requested
        status = false ;                  // return value

   if ( argc > 1 )                        // if user provided arguments
   {
      gString gs ;                        // text formatting
      char option ;                       // command-line option
      short j ;

      #if DEBUG_GCLA != 0
      gs.compose( "argc:%hd\n", &argc ) ;
      this->write ( gs ) ;
      #endif   // DEBUG_GCLA

      for ( short i = 1 ; i < argc ; ++i )
      {
         #if DEBUG_GCLA != 0
         gs.compose( "%hd) %s\n", &i, argv[i] ) ;
         this->write ( gs ) ;
         #endif   // DEBUG_GCLA

         j = ZERO ;
         if ( argv[i][j] == DASH )
         {
            ++j ;                         // index the next char of option
            if ( argv[i][j] == DASH )     // double-dash ('--')
            {
               gs = &argv[i][++j] ;
               if ( (gs.find( "version" )) == ZERO )     // request for version number
               { ver = true ; ++validArgs ; break ; }
               else if ( (gs.find( "help" )) == ZERO )   // request for help
               { hlp = true ; ++validArgs ; break ; }
               else if ( (gs.find( "sfile" )) == ZERO )  // specify source file
               {
                  short indx ;
                  if ( (indx = gs.after( EQU )) > ZERO ) // reference first char of filename
                     gs.shiftChars( -(indx) ) ;
                  else { gs.clear() ; }                  // syntax error
                  // Note: Isolate and use filename only (read from current directory).
                  if ( (indx = gs.findlast( L'/' )) >= ZERO )
                     gs.shiftChars( -(indx + 1) ) ;
                  if ( (gs.gschars()) > 1 )
                  {
                     gs.insert( L"./" ) ;
                     gs.copy( this->sFile, MAX_FNAME ) ;
                     ++validArgs ;
                  }
                  #if DEBUG_GCLA != 0
                  gs.compose( "   '%s'\n", this->sFile ) ;
                  this->write ( gs ) ;
                  #endif   // DEBUG_GCLA
               }
               else if ( (gs.find( "tfile" )) == ZERO )  // specify source file
               {
                  short indx ;
                  if ( (indx = gs.after( EQU )) > ZERO ) // reference first char of filename
                     gs.shiftChars( -(indx) ) ;
                  else { gs.clear() ; }                  // syntax error
                  // Note: Isolate and use filename only (written to current directory).
                  if ( (indx = gs.findlast( L'/' )) >= ZERO )
                     gs.shiftChars( -(indx + 1) ) ;
                  if ( (gs.gschars()) > 1 )
                  {
                     gs.insert( L"./" ) ;
                     gs.copy( this->tFile, MAX_FNAME ) ;
                     ++validArgs ;
                  }
                  #if DEBUG_GCLA != 0
                  gs.compose( "   '%s'\n", this->tFile ) ;
                  this->write ( gs ) ;
                  #endif   // DEBUG_GCLA
               }
            }
            else
            {
               option = argv[i][j++] ;
               if ( argv[i][j] == EQU )   // smart user
                  gs = &argv[i][++j] ;
               else                       // dumb user
               { gs = argv[++i] ; j = ZERO ; }
               if ( gs.gschars() > 1 )    // if an argument captured
               {
                  switch ( option )
                  {
                     case 't':
                        if ( gs.gschars() > 1 )
                        {
                           //* If not a code for "empty cartouche",  *
                           //* capture and convert the source text.  *
                           if ( gs.compare( L"-" ) != ZERO )
                              this->goodtxt = this->gclaChar2Word ( &argv[i][j] ) ; 
                           ++validArgs ;
                        }
                        break ;
                     case 'w':
                        if ( ((gs.gscanf( "%hd", &tmpval )) == 1) &&
                             (tmpval > wMIN) && (tmpval <= wMAX) )
                        { this->width = tmpval ; ++validArgs ; }
                        break ;
                     case 'h':
                        if ( ((gs.gscanf( "%hd", &tmpval )) == 1) &&
                             (tmpval >= hMIN) )
                        {  // Note: If specified value would tend to overflow the 
                           //       output buffer, silently set to maximum.
                           if ( tmpval > hMAX ) { tmpval = hMAX ; } 
                           this->minHgt = tmpval ; ++validArgs ;
                        }
                        break ;
                     case 'p':
                        if ( ((gs.gscanf( "%hd", &tmpval )) == 1) &&
                             (tmpval >= ZERO) && (tmpval < (wMAX - wMIN - 2)) )
                        { this->pad = tmpval ; ++validArgs ; }
                        break ;
                     case 's':      // line style
                        ++validArgs ;        // hope for the best
                        switch ( argv[i][j] )
                        {
                           case 's': this->stylePtr = lnSingle ;     break ;
                           case 'b': this->stylePtr = lnSingBold ;   break ;
                           case 'd': this->stylePtr = lnDual ;       break ;
                           case 'D': this->stylePtr = lnDash ;       break ;
                           case 'B': this->stylePtr = lnDashBold ;   break ;
                           default: --validArgs ; break ;
                        }
                        break ;
                     default:             // unknown option
                        break ;
                  }
               }
            }
         }
         else     // all arguments must begin with a dash
         { break ; }
      }

      if ( validArgs == argc )
      {
         status = true ;

         //* If source data are in a file *
         if ( ! this->goodtxt && (*this->sFile != NULLCHAR) )
         {
            this->goodtxt = this->ReadSourceFile () ;
            if ( *this->srcText == NULLCHAR ) // if no source data, call user a dumbguy
            { hlp = true ; }
         }
      }
      else        // set application exit code
         this->status = -1 ;
   }

   if ( ver )
   { this->Version () ; status = false ;}
   else if ( hlp || !status )
   { this->Help () ; status = false ; }

   return status ;

}  //* End GetCommlineArgs() *

//*************************
//*     gclaChar2Word     *
//*************************
//********************************************************************************
//* Convert source text on command line from byte data to UTF-32 data, and       *
//* save it to 'srcText'.                                                        *
//*                                                                              *
//* Input  : src   : pointer to source data from command line.                   *
//*                                                                              *
//* Returns: 'true' if all source data successfully captured and converted.      *
//*          'false' if 1) no data, 2) conversion error                          *
//********************************************************************************

bool BuildCart::gclaChar2Word ( const char* src )
{
   #define DEBUG_C2W (0)                  // for debugging only
   gString gs( src ) ;                    // text formatting
   bool status = false ;                  // return value

   //* If all data fit into a single gString, copy to target directly.*
   if ( gs.gschars() < (gsMAXCHARS - 4) )
   {
      gs.copy( this->srcText, gsMAXCHARS ) ;
      status = true ;
   }

   //* Else, data may be too long for the gString, *
   //* so capture the data in segments.            *
   else
   {
      short si = ZERO, ti = ZERO ;

      #if DEBUG_C2W != 0
      short tot, gsc ; bool trunc = false ;
      for ( tot = ZERO ; src[tot] != NULLCHAR ; ++tot ) { } ++tot ;
      gString gsdbg( "Big source text: srcBuffer:%hd raw src bytes:%hd\n", &srcBUFFER, &tot ) ;
      this->write ( gsdbg ) ;
      #endif   // DEBUG_C2W

      do
      {
         gs.loadChars( &src[si], 1000 ) ;
         si += gs.utfbytes() - 1 ;
         if ( (ti + gs.gschars()) >= srcBUFFER )
         {
            gs.limitChars( srcBUFFER - ti -1 ) ;

            #if DEBUG_C2W != 0
            trunc = true ;
            #endif   // DEBUG_C2W
         }
         gs.copy( &this->srcText[ti], gsMAXCHARS ) ;
         ti += gs.gschars() - 1 ;

         #if DEBUG_C2W != 0
         gsc = gs.gschars() ;
         gsdbg.compose( " gsc:%04hd ti:%03hd\n", &gsc, &ti ) ; this->write ( gsdbg ) ;
         #endif   // DEBUG_C2W
      }
      while ( gs.gschars() >= 1000 && (ti < (srcBUFFER - 2)) ) ;
// CZONE - ++ti ; //count the null terminator. Would it be helpful to save this value?

      #if DEBUG_C2W != 0
      ++ti ; gsdbg.compose( "Total Source Chars: %hd%s", 
                            &ti, (trunc ? " (truncated)\n" : "\n") ) ;
      this->write ( gsdbg ) ;
      #endif   // DEBUG_C2W

      status = true ;
   }
   return status ;

}  //* End gclaChar2Word() *

//*************************
//*    ReadSourceFile     *
//*************************
//********************************************************************************
//* Read source data for cartouche from the specified source file.               *
//*                                                                              *
//* Data are read, one line at a time until:                                     *
//*  1) end of file is reached                                                   *
//*  2) the line contain the literal string: "#EOF" (without quotes).            *
//*                                                                              *
//* Linefeeds: Each line is expected to terminate with a linefeed ('\n').        *
//*            A linefeed is optional for the last line of the file.             *
//*                                                                              *
//* Maximum line length, including linefeed, is gsMAXBYTES (4096 bytes).         *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: 'true' if 'srcText' member initialized, else 'false'                *
//********************************************************************************

bool BuildCart::ReadSourceFile ( void )
{
   const wchar_t* EndOfData = L"#EOF" ;   // token signalling end-of-data
   char ibuff[gsMAXBYTES] ;               // input buffer
   gString gs ;                           // text formatting
   short chChunk = gsMAXCHARS,            // max characters per write
         freeChars = srcBUFFER,           // number of free buffer words
         si = ZERO ;                      // index 'srcText' array
   bool done = false,                     // loop control
        status = false ;                  // return value

   ifstream ifs ( this->sFile, ifstream::in ) ;
   if ( ifs.is_open() )
   {
      while ( ! done )
      {
         ifs.getline ( ibuff, gsMAXBYTES, NEWLINE ) ;
         if ( (ifs.good()) || (ifs.gcount() > ZERO) )
         {
            gs.compose( "%s\n", ibuff ) ; // restore the newline
            //* Test for end-of-data token.*
            if ( (gs.find( EndOfData )) == ZERO )
            { done = true ; }
            else
            {
               gs.copy( &this->srcText[si], chChunk ) ;
               si += gs.gschars() - 1 ;
               if ( (freeChars -= gs.gschars() - 1) < gsMAXCHARS )
               { if ( (chChunk = freeChars - 1) <= ZERO ) { done = true ; } }
            }
         }
         else
            done = true ;
      }
      ifs.close() ;
      if ( si > 1 )
      {
         status = true ;

         //* Remove trailing newline if present.*
         if ( this->srcText[--si] == NEWLINE )
            this->srcText[si] = NULLCHAR ;
      }
   }
   return status ;

}  //* End ReadSourceFile() *

//*************************
//*    WriteTargetFile    *
//*************************
//********************************************************************************
//* Write a copy of the cartouche to the specified target file.                  *
//* If file does not exist, it will be created.                                  *
//* If file exists, data will be appended to existing contents.                  *
//* -- Note: Source file and target may be the same file without loss of data.   *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: 'true' if data written to target, else 'false'                      *
//********************************************************************************

bool BuildCart::WriteTargetFile ( void )
{
   gString gs ;                           // text formatting
   short indx,                            // character index
         si = ZERO ;                      // source-data index
   bool done = false,                     // loop control
        status = false ;                  // return value

   ofstream ofs ( this->tFile, ofstream::out | ofstream::app ) ;
   if ( ofs.is_open() )
   {
      ofs << "\nBuildCart: Create Texinfo Cartouche." << endl ; // write header

      while ( ! done )
      {
         gs = &this->trgText[si] ;        // convert to byte data
         if ( gs.gschars() > 1 )
         {
            if ( (indx = gs.after( NEWLINE )) >= ZERO )
               gs.limitChars( indx ) ;
            ofs << gs.ustr() ;
            si += gs.gschars() - 1 ;
         }
         else                            // 'si' references the null terminator
         { ofs << endl ; done = true ; } // append a newline and exit the loop
      }
      ofs.close() ;
   }
   return status ;

}  //* End WriteTargetFile() *

//*************************
//*         Help          *
//*************************
//********************************************************************************
//* Display application help.                                                    *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void BuildCart::Help ( void )
{
   const char* const Options1 = 
   " Build a plain-text cartouche object for Texinfo documentation.\n"
   " All options are optional except: specify source text using either\n"
   "'-t' OR '--sfile' option, but not both.\n"
   " -t=TEXT    Text data to be displayed. Enclose in quotes and escape\n"
   "            embedded quotes. ('-' for an empty cartouche)\n" ;
   const char* const Options2 = 
   " -w=WIDTH   Width (number of columns) for the cartouche object. (optional)\n"
   "            By default, text data are assumed to be pre-formatted, and the\n"
   "            cartouche is dimensioned according to the number of rows and\n"
   "            columns of the the source text. If a width value is specified,\n"
   "            text will be dynamically reformatted to meet the specification.\n"
   "            Range: %hd to %hd.\n"
   " -h=HEIGHT  Minimum height in rows. Range: 3 to %hd (default: text rows + 2)\n" ;
   const char* const Options3 =
   " -p=PADDING Number of pad characters on left and right interior.(default:1)\n"
   " -s=STYLE   Line style: 's' single (default), 'd' double, 'D' dashed\n"
   "                        'b' single bold,                  'B' dashed bold\n"
   " --sfile=FILENAME Name of source file containing text for the cartouche.\n"
   "                  object. By default, object is written only to stdout.\n"
   " --tfile=FILENAME Name of target file to receive a COPY of the cartouche\n"
   "                  object. By default, object is written only to stdout.\n"
   " --help     Display command-line help, then exit.\n"
   " --version  Display application version and copyright info, then exit.\n"
   "\n" ;

   gString gs( Options2, &wMIN, &wMAX, &hMAX ) ;
   this->write ( Options1 ) ;
   this->write ( gs ) ;
   this->write ( Options3 ) ;

}  //* End Help() *

//*************************
//*        Version        *
//*************************
//********************************************************************************
//* Display application version number and copyright message.                    *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void BuildCart::Version ( void )
{
   static const char* const freeSoftware = 
    "License GPLv3+: GNU GPL version 3 <http://gnu.org/licenses/gpl.html>\n"
    "This is free software: you are free to modify and/or redistribute it\n"
    "under the terms set out in the license.\n"
    "There is NO WARRANTY, to the extent permitted by law.\n\n" ;

   this->clearTerm () ;
   this->write ( freeSoftware ) ;

}  //* End Version() *

//*************************
//*                       *
//*************************
//********************************************************************************
//*                                                                              *
//*                                                                              *
//*                                                                              *
//* Input  :                                                                     *
//*                                                                              *
//* Returns:                                                                     *
//********************************************************************************

