//********************************************************************************
//* File       : BuildCart.hpp                                                   *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2024      Mahlon R. Smith, The Software Samurai   *
//*                 GNU GPL copyright notice below                               *
//* Date       : 20-Jun-2024                                                     *
//* Version    : (see appVersion member, below)                                  *
//*                                                                              *
//* Description: Build a pseudo-cartouche around the user-specified text.        *
//*              This is used in Texinfo documentation (.info format).           *
//*                                                                              *
//*                                                                              *
//********************************************************************************
//* Copyright Notice:                                                            *
//* This program is free software: you can redistribute it and/or modify it      *
//* under the terms of the GNU General Public License as published by the Free   *
//* Software Foundation, either version 3 of the License, or (at your option)    *
//* any later version.                                                           *
//*                                                                              *
//* This program is distributed in the hope that it will be useful, but          *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License     *
//* for more details.                                                            *
//*                                                                              *
//* You should have received a copy of the GNU General Public License along      *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.              *
//*                                                                              *
//********************************************************************************

//*****************
//* Include Files *
//*****************
#include <unistd.h>           //* UNIX system interface
#include <cstdlib>            //* Misc. functionality
#include <cstdint>            //* [u]int*_t definitions (C++13)
#include <locale>             //* Locale handling for character encoding, etc. 
#include <fstream>            //* C++ file I/O
#include "gString.hpp"        //* Text analysis and formatting

using namespace std ;                  //* Scope quailfier

//***************
//* Definitions *
//***************
const wchar_t NEWLINE   = L'\n' ;
const wchar_t NULLCHAR  = L'\0' ;
const char    DASH      = '-' ;
const char    EQU       = '=' ;
const short   ZERO      = 0 ;
const short   MAX_FNAME = 256 ;     //* size of buffer to hold a filename
const short   padDFLT   = 1 ;       //* default horizontal padding (at each margin)
const short   wMIN      = 8 ;       //* minimum object width (columns)
const short   wMAX      = 132 ;     //* maximum object width (columns)
const short   hMIN      = 3 ;       //* minimum object height (top+bottom+one data row)
//* Maximum object height in rows, incl. top and bottom borders. *
//* Calculated to fit within the target buffer: trgBUFFER.       *
//* [132 cols x 30 rows == 3960 + 30('\n') + '\0' == 3991 chars] *
const short   hMAX      = 30 ;      //* This is the max data for the output bufer.
const short srcBUFFER = gsMAXCHARS * 3 ;  // size of source buffer
const short trgBUFFER = gsMAXCHARS * 4 ;  // size of target buffer

//* Indices for border character arrays: UpperLeft, HoriZontal,  *
//* UpperRight, VerTical, LowerLeft and LowerRight, respectively.*
enum chIndex { ciUL = ZERO, ciHZ, ciUR, ciVT, ciLL, ciLR, ciCNT } ;

//* Arrays of border characters for each border type. *
static const wchar_t lnSingle[ciCNT]   = { L'┌', L'─', L'┐', L'│', L'└', L'┘' } ;
static const wchar_t lnDual[ciCNT]     = { L'╔', L'═', L'╗', L'║', L'╚', L'╝' } ;
static const wchar_t lnDash[ciCNT]     = { L'┌', L'╌', L'┐', L'┆', L'└', L'┘' } ;
static const wchar_t lnSingBold[ciCNT] = { L'┏', L'━', L'┓', L'┃', L'┗', L'┛' } ;
static const wchar_t lnDashBold[ciCNT] = { L'┏', L'╍', L'┓', L'╏', L'┗', L'┛' } ;

//* Source text analysis data *
class AzSource
{
   public:
   AzSource ( short horizpad )   // constructor
   {
      this->pad = horizpad ;
      this->reset () ;
   }

   //* Count the number of source characters, and text rows.         *
   //* ATTENTION, DUMBASS! This method allocates dynamic memory!     *
   //* It is the caller's responsibility to release this allocation. *
   //* Input  : wptr : pointer to null-terminated source text        *
   //*          cptr : (by reference)                                *
   //*                 receives address of dynamic allocation        *
   //*          rcnt : (by reference) receives number of text rows   *
   //*          chcnt: (by reference) receives number of text chars  *
   //* Returns: pointer to dynamic allocation (same as 'cptr')       *
   short* colAllocate ( const wchar_t* wptr, short*& cptr, short& rcnt, short&chcnt )
   {
      cptr = NULL ;                       // initialize pointer
      rcnt = chcnt = ZERO ;               // initialize accumulators
      for ( ; wptr[chcnt] != NULLCHAR ; ++chcnt )
      { if ( wptr[chcnt] == NEWLINE ) { ++rcnt ; } }
      ++chcnt ;                           // include the null terminator

      cptr = new short[chcnt] ;           // allocate dynamic storage
      return cptr ;

   }  //* End colAllocate() *

   //* Input  : wptr : pointer to null-terminated source text  *
   //* Returns: nothing                                        *
   void Analyze ( const wchar_t* wptr )
   {
      this->reset () ;                    // reset the accumulators

      short*   cptr = NULL ;              // pointer to array of character widths
      const short* segcptr ;              // pointer to segment column widths
      gString gs ;                        // text analysis
      wchar_t testChar ;                  // character under test
      short si = ZERO,                    // source index
            ti,                           // target index
            segchars,                     // segment character count
            segi ;                        // segmentcolumn-width index

      //* Count the number of source characters, and text rows, *
      //* and allocate an array to hold the character widths.   *
      short rcnt ;                        // initial row count
      this->colAllocate ( wptr, cptr, rcnt, this->charCnt ) ;

      //* Initialize the array of character widths.*
      si = ti = ZERO ;                    // initialize the indices
      while ( ti < (this->charCnt - 1) )
      {
         gs.loadChars( &wptr[si], 1000 ) ;// get a manageable number of characters
         si += gs.gschars() - 1 ;         // advance the source index
         segcptr = gs.gscols( segchars ) ;// get char widths for segment
         --segchars ;                     // exclude null terminator
         segi = ZERO ;                    // reset segment index
         while ( segi < segchars )        // concatenate character widths
            cptr[ti++] = segcptr[segi++] ;
      }
      cptr[ti] = ZERO ;                   // width of null terminator

      //* Separate the individual rows, and count the number of *
      //* columns for each row. Detarmine the maximun row width.*
      si = ZERO ;                         // initialize the indices
      for ( short ri = ZERO ; ri < hMAX ; ++ri )
      {
         ti = ZERO ;
         while ( ((testChar = wptr[si]) != NEWLINE) && (testChar != NULLCHAR) )
         {  // Programmer's Note: If row is too long for the row buffer, the
            // data will not be copied; however, the column count greater than the 
            // maximum will indicate that the data are not correctly formatted.
            if ( ti < wMAX )  // if free space in the row buffer, save character
               this->wbuff[ri][ti++] = testChar ;
            this->rowCols[ri] += cptr[si] ;  // add to total character width
            ++si ;
         }
         if ( testChar == NEWLINE )       // discard the newline
            ++si ;
         this->wbuff[ri][ti] = NULLCHAR ; // terminate the row
         if ( this->rowCols[ri] > this->maxtxtCols ) // new widest row?
            this->maxtxtCols = this->rowCols[ri] ;
         ++this->rowCnt ;                 // row analyzed
         if ( testChar == NULLCHAR )      // all rows analyzed
            break ;
      }
      delete [] cptr ;                    // release the dynamic allocation

      //* Full width of the cartouche is: widest text row plus non-text *
      //* columns per row, but not more than defined maximum.           *
      if ( (this->cartWidth = this->maxtxtCols + this->nontxtCols) > wMAX )
         this->cartWidth = wMAX ;

   }  //* End Analyze() *

   void reset ( void )
   {
      for ( short i = ZERO ; i < hMAX ; ++i )
      { this->wbuff[i][0] = L'\0' ; this->rowCols[i] = 0 ; }
      this->rowCnt = this->charCnt = this->maxtxtCols = 
      this->maxtxtWidth = this->cartWidth = 0 ;
      this->nontxtCols = this->pad * 2 + 2 ;
      this->maxtxtWidth = wMAX - nontxtCols ;
   }  //* End reset() *

   //* Public Data Members *
   wchar_t wbuff[hMAX][wMAX+1] ; // parsing buffer max_rows x max_chars_per_row + 1
   short rowCols[hMAX] ;         // number of columns for each text row
   short rowCnt ;                // actual number of rows
   short charCnt ;               // number of wchar_t source characters
   short pad ;                   // text-padding columns for left and right ends of row
   short maxtxtCols ;            // number of display columns in widest row
   short nontxtCols ;            // columns per row IN ADDITION TO the text data
   short maxtxtWidth ;           // maximum columns on the row for text data
   short cartWidth ;             // total width of cartouche object

   private:
   AzSource ( void ) ;           // hide the default constructor
} ;

//*********************
//* Application class *
//*********************
class BuildCart
{
   public:
   ~BuildCart ( void )                    // destructor
   {
      if ( this->ioLocale != NULL )       // release the locale object
         delete ( this->ioLocale ) ;
   }

   BuildCart ( int argc, const char* argv[], const char* argenv[] ) ; // constructor
   void Cart ( void ) ;                   // build a cartouche around the text
   void EmptyCart ( void ) ;              // create an empty cartouche
   void ReformatSource ( AzSource& azs ) ;// re-flow source text to fit specified width

   void write ( const uint8_t* txt ) ;    // write to stdout
   void write ( const wchar_t* txt ) ;    // write to stdout
   void write ( const gString& txt ) ;    // write to stdout

   void clearTerm ( void )
   {
      system ( "clear" ) ;
      gString gsOut( "BuildCart (bcart) v:%s  Copyright (c) %s The Software Samurai\n",
                     this->appVersion, this->appYears ) ;
      gsOut.padCols( gsOut.gscols() * 2, L'=' ) ;
      gsOut.append( L'\n' ) ;
      this->write ( gsOut ) ;
   }

   bool GetCommlineArgs ( int argc, const char* argv[] ) ;
   bool gclaChar2Word ( const char* src ) ;
   bool ReadSourceFile ( void ) ;
   bool WriteTargetFile ( void ) ;
   void Help ( void ) ;
   void Version ( void ) ;

   short exitStatus ( void )
   { return ( this->status ) ; }

   //** Data Members **
   private:
   const locale* ioLocale ;
   const char* appVersion = "0.0.01" ;
   const char* appYears   = "2024-...." ;

   wchar_t srcText[srcBUFFER+1] ;   // input buffer
   wchar_t trgText[trgBUFFER+1] ;   // output buffer
   char    sFile[MAX_FNAME] ;       // filename for reading source data
   char    tFile[MAX_FNAME] ;       // filename for writing output data
   const wchar_t* stylePtr ;        // pointer to array of line-drawing characters
   short width ;        // target width for object
   short minHgt ;       // minimum object height (rows)
   short pad ;          // padding columns at left and right margins
   short status ;       // set if all parameters are valid (exit code)
   bool  leadRow ;      // set if an empty text row preceeds the actual text
   bool  goodtxt ;      // set if valid text received from user

} ;   // BuiltCart

