//********************************************************************************
//* File       : CommExtract.hpp                                                 *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2024 Mahlon R. Smith, The Software Samurai        *
//*                 GNU GPL copyright notice below                               *
//* Date       : 30-Sep-2024                                                     *
//* Version    : (see AppVersion string in EarthPoints.cpp)                      *
//*                                                                              *
//* Description: CommExt class definition.                                       *
//*                                                                              *
//*                                                                              *
//********************************************************************************

//****************
//* Header Files *
//****************
#define COMMEX_DEFINED        //* Be sure file is included only once

using namespace std ;         //* C++ Scope quailfier

//*****************
//* Include Files *
//*****************
#include <unistd.h>           //* UNIX system interface
#include <cstdlib>            //* Misc. functionality
#include <cstdint>            //* [u]int*_t definitions (C++13)
#include <fstream>            //* C++ file I/O
#include "CxFile.hpp"         //* file management utilities
#include "gString.hpp"        //* gString class definition


//***************************
//* General constant values *
//***************************
const wchar_t NULLCHAR  = L'\0' ;   // ASCII null character  U+0000
const wchar_t NEWLINE   = L'\n' ;   // ASCII linefeed        U+000A
const wchar_t SPACE     = L' ' ;    // ASCII space           U+0020
const wchar_t DASH      = L'-' ;    // ASCII hyphen/minus    U+002D
const wchar_t CR        = L'\r' ;   // ASCII carriage return U+000D (Windoze uses this)
const wchar_t fSLASH    = L'/' ;    // ASCII forward slash   U+002F
const wchar_t STAR      = L'*' ;    // ASCII asterisk        U+002A
const short   ZERO      = (0) ;     // numeric zero
const wchar_t* const CCOMM = L"//" ;// C++ Comment token

//* Return values for GetCommandLineArgs(). ( 'userOption' ) *
enum clArgs : short 
{ clQHelp, clHelp, clVersion, clExtract } ;

//*********************************************
//* Gather and process command-line arguments *
//*********************************************
class commArgs
{
   public:
      //* Initialization constructor *
   commArgs ( int argc, char** argv, char** argenv ) :
               argCount(argc), argList(argv), envList(argenv)
   {
      *this->sFile = NULLCHAR ;
      *this->tFile = NULLCHAR ;
      *this->parmErr = NULLCHAR ;
      this->lnBeg = this->lnEnd = this->lnCnt = ZERO ;
      this->strAll = this->delTrg = this->trmEcho = false ;
      this->verFlag = this->helpFlag = false ;
   }

   //** Public Data Members **
   short    argCount ;           // application's 'argc'
   char**   argList ;            // application's 'argv'
   char**   envList ;            // application's 'argenv'

   char     sFile[MAX_FNAME] ;   // source filename
   char     tFile[MAX_FNAME] ;   // target filename
   char     parmErr[MAX_FNAME] ; // error message, invalid parameter
   short    lnBeg ;              // line number to begin scan
   short    lnEnd ;              // line number to end scan (inclusive)
   short    lnCnt ;              // number of lines to scan
   bool     strAll ;             // if set, strip ALL leading/trailing whitespace
                                 // if reset, remove only 1 leading space
   bool     delTrg ;             // if set, delete existing target file
                                 // if reset, append scanned data to target file (if present)
   bool     trmEcho ;            // if set, echo reformatted data to terminal window
   bool     verFlag ;            // --version: 'true' if display command-line version
   bool     helpFlag ;           // -H option: 'true' if display command-line help
} ;


//********************************
//* Application class definition *
//********************************
class CommExt
{
   public:
   ~CommExt ( void ) ;              // destructor
   CommExt ( commArgs& ca ) ;       // constructor

   //* General support methods *
   //* ----------------------- *
   protected:
   bool scanSourceData ( void ) ;
   clArgs getCommandLineArgs ( commArgs& ca ) ;
   bool validateArgs ( commArgs& ca ) ;
   void composeTitle ( gString& gs, bool tailNL = false, bool headNL = false ) ;
   bool setLocale ( const char * localeName = NULL ) ;

   //* File Management Methods *
   short getFileStats ( tnFName& tnFile, const gString& fPath ) ;

   void reset ( void )              // set all data members to default values
   {
      *this->srcFile = NULLCHAR ;   // filespec unknown
      *this->trgFile = NULLCHAR ;   // filespec unknown
      this->ioLocale = NULL ;       // locale unknown
      this->lnBeg = this->lnCnt = ZERO ;
      this->wspAll = this->trmEcho = false ;
   }

   char srcFile[MAX_PATH] ;         // filespec for source file
   char trgFile[MAX_PATH] ;         // filespec for target file
   tnFName srctnf ;                 // source file stats
   tnFName trgtnf ;                 // target file stats
   locale *ioLocale ;               // application locale (language and text processing)
   short   lnBeg ;                  // line number of first source line to be scanned
   short   lnCnt ;                  // number of source lines to be scanned
   bool    wspAll ;                 // if set, remove ALL leading whitespace
   bool    trmEcho ;                // if set, echo reformatted data to terminal window

} ;   // CommExt class
