//******************************************************************************
//* File       : crcmodel.c                                                    *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2017      Mahlon R. Smith, The Software Samurai *
//*                  GNU GPL copyright notice below.                           *
//* Date       : 16-Feb-2017                                                   *
//* Version    : (see AppVersion string)                                       *
//*                                                                            *
//* Mahlon's Notes:                                                            *
//* 1) The base code for this application comes from:                          *
//*    "A Painless Guide To CRC Error Detection Algorithms," by Ross Williams  *
//*    http://www.ross.net/crc/download/crc_v3.txt                             *
//*    Ross describes this code as a reference model for generating CRC        *
//*    checksums, and he placed the document and the sample code in the        *
//*    public domain.                                                          *
//*                                                                            *
//* 2) We located this reference from a note in the OGG/Vorbis I audio         *
//*    format specification at: https://xiph.org/vorbis/doc/framing.html       *
//*    OGG/Vorbis uses a CRC checksum to validate its metadata (tag data)      *
//*    for audio files.                                                        *
//*                                                                            *
//* 3) We needed to understand CRC for our Taggit tag data editor, and this    *
//*    seemed to be a likely candidate; however, in the original form this     *
//*    code had multiple issues.                                               *
//*    a) Ross's science is quite good, and the step-by-step description of    *
//*       how CRC works is clear and easy to follow (at least for math geeks.) *
//*    b) The example code is embarrassingly poor, and shows no professionalism*
//*       whatsoever. It is likely that it was written with microcontrollers   *
//*       or other low-level hardware in mind, since the use of 'unsigned long'*
//*       indicates a 32-bit value. Still, from the two source modules and     *
//*       the header file we think it may be possible to determine what's      *
//*       going on under the hood.                                             *
//*    c) As provided, crcmodel.c had no main() so we needed investigation to  *
//*       determine how the application is intended to function.               *
//*    d) crctable had a main(), but I had to tweek it before it would compile.*
//*    e) I wish the code weren't implemented with so many macros and unhelpful*
//*       typedefs. A reference implementation should be more about clarity    *
//*       than cuteness.                                                       *
//*    f) The "parameters" for generating checksums and lookup tables were     *
//*       actually hard-coded #define's and while we understand doing this at  *
//        the proof-of-concept stage, it is unnecessarily inflexible for an    *
//*       actual software tool.                                                *
//*    Well, that's enough kvetching from me, so let's get on with it,         *
//*    shall we?                                                               *
//*                                                                            *
//* 4) The new implementation:                                                 *
//*    a) A new main() provides branches to one of three main operating modes  *
//*       plus a call to Help if parameters are not properly formatted.        *
//*    b) GetCommlineArgs() interprets simple command-line switches to provide *
//*       some flexibility for the setup parameters.                           *
//*    c) Register width has been updated for modern hardware:                 *
//*       from: int   == 16-bit  and  long == 32-bit                           *
//*       to  : short == 16-bit  and  int  == 32-bit                           *
//*       This, by itself does not change functionality in any way, but was    *
//*       done for reasons of clarity for readers who have never heard of      *
//*       a Z80, M6502 or i8088.                                               *
//*    d) Mahlon's comments are written in the more convenient and readable    *
//*       C++ comment style.                                                   *
//*    e) Ross Williams' original notes are retained (as C-style comments)     *
//*       except where they were misleading, wrong or made obsolete by the     *
//*       code update.                                                         *
//*    f) Except for the bit-shifting, all macros have been banished.          *
//*       To paraphrase Gollum, "We can't stands the filthy macroses."         *
//*    g) We considered porting the whole thing to a C++ class definition      *
//*       because we firmly believe that C is (or should be) a dying language. *
//*       We retained the C-language implementation because we're too          *
//*       busy (lazy) to do it right. (Maybe later :)                          *
//*       (Note that the algorithm IS implemented in C++ in our Taggit app.)   *
//*    h) Basically, command line options are as follows:                      *
//*         i) Primary option (required): exactly one of the three possible:   *
//*            --test  --file  --table                                         *
//*        ii) Secondary options (all optional)                                *
//*            These options allow optional specification of all setup         *
//*            parameters for CRC checksum and table generation.               *
//*            (All optional parameters are ignored for the)                   *
//*            (algorithm-validation test.                 )                   *
//*            All CRC processing parameters have default values that          *
//*            correspond to the generic 32-bit algorithm.                     *
//*       iii) See the HelpMe() function for a complete list of options.       *
//*                                                                            *
//* 5) The reference model checksum generator does not use a table lookup.     *
//*    Instead it directly calculates the CRC byte-by-byte. This is a simple   *
//*    and reliable model for algorithm verification tests, but it is          *
//*    comparatively slow.                                                     *
//*                                                                            *
//*    For high-speed commercial-grade applications such as Ethernet packets,  *
//*    the CRC algorithm can be enhanced by using a pre-generated lookup table *
//*    to reduce the number of necessary runtime calculations.                 *
//*                                                                            *
//*    For this reason, we have enhanced Ross's original code for 'crcmodel'   *
//*    to include an example implementation of the table-lookup speed          *
//*    enhancement. In this example we dynamically build a lookup table and    *
//*    then process the input stream using that table. A much larger data set  *
//*    than our simple test data would be required to properly evaluate the    *
//*    difference in processing performance between the direct calculation     *
//*    method and the table lookup method. We leave that evaluation as an      *
//*    exercise.                                                               *
//*    While this implementation is certainly faster, it is also less reliable.*
//*    However, for all the likely combinations of setup parameters, the       *
//*    faster algorithm produces exactly the same results as the reference     *
//*    model. We hope that your hot implementation will be even faster, AND    *
//*    every bit as stable. Good luck!                                         *
//*    See the GenRamTable() and FileChecksumT() functions for details of our  *
//*    lookup-table implementation.                                            *
//*                                                                            *
//* 6) Unlike most of our work, this application's documentation is quite      *
//*    simple. This is not a commercial product; it is simply a tool for       *
//*    students who are trying to integrate CRC checksums into their own work. *
//*    If this application generates the same CRC as your program for a given  *
//*    byte stream and parameter setup, then you can be reasonably sure that   *
//*    your implementation is correct.                                         *
//*                                                                            *
//******************************************************************************
//* Copyright Notice:                                                          *
//* This program is free software: you can redistribute it and/or modify it    *
//* under the terms of the GNU General Public License as published by the Free *
//* Software Foundation, either version 3 of the License, or (at your option)  *
//* any later version.                                                         *
//*                                                                            *
//* This program is distributed in the hope that it will be useful, but        *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
//* for more details.                                                          *
//*                                                                            *
//* You should have received a copy of the GNU General Public License along    *
//* with this program.  Full text of the GPL License may be found at:          *
//*                   <http://www.gnu.org/licenses/>.                          *
//*                                                                            *
//******************************************************************************
/*                                                                            */
/* Author : Ross Williams (ross@guest.adelaide.edu.au.).                      */
/* Date   : 3 June 1993.                                                      */
/* Status : Public domain.                                                    */
/*                                                                            */
/* Description : This is the implementation (.c) file for the reference       */
/* implementation of the Rocksoft^tm Model CRC Algorithm. For more            */
/* information on the Rocksoft^tm Model CRC Algorithm, see the document       */
/* titled "A Painless Guide to CRC Error Detection Algorithms" by Ross        */
/* Williams (ross@guest.adelaide.edu.au.). This document is likely to be in   */
/* "ftp.adelaide.edu.au/pub/rocksoft".                                        */
/*                                                                            */
/* Note: Rocksoft is a trademark of Rocksoft Pty Ltd, Adelaide, Australia.    */
/*                                                                            */
/******************************************************************************/
/*                                                                            */
/* Implementation Notes                                                       */
/* --------------------                                                       */
/* To avoid inconsistencies, the specification of each function is not echoed */
/* here. See the header file for a description of these functions.            */
/* This package is light on checking because I want to keep it short and      */
/* simple and portable (i.e. it would be too messy to distribute my entire    */
/* C culture (e.g. assertions package) with this package.                     */
/*                                                                            */
/******************************************************************************/
/* A popular form of the CRC-16 algorithm.                                    */
/*                                                                            */
/*  Name   : "CRC-16"                                                         */
/*  Width  : 16                                                               */
/*  Poly   : 8005                                                             */
/*  Init   : 0000                                                             */
/*  RefIn  : True                                                             */
/*  RefOut : True                                                             */
/*  XorOut : 0000                                                             */
/*  Check  : BB3D                                                             */
/*                                                                            */
/* Here is the specification for the CRC-32 algorithm which is reportedly     */
/* used in PKZip, AUTODIN II, Ethernet, and FDDI.                             */
/*                                                                            */
/*   Name   : "CRC-32"                                                        */
/*   Width  : 32                                                              */
/*   Poly   : 04C11DB7                                                        */
/*   Init   : FFFFFFFF                                                        */
/*   RefIn  : True                                                            */
/*   RefOut : True                                                            */
/*   XorOut : FFFFFFFF                                                        */
/*   Check  : CBF43926                                                        */
/*                                                                            */
/*                                                                            */
/* To ensure that the following code is working, configure it for the CRC-16  */
/* and CRC-32 algorithms given above and ensure that they produce the         */
/* specified "check" checksum when fed the test string "123456789"            */
/* (i.e. 313233... (hexadecimal)) (see earlier).                              */
/******************************************************************************/

#define DEFINE_GLOBALS_HERE

#include "crcmodel.h"

//* Application version *
const char* const AppVersion = "1.5.02" ;

//* List of application operating modes *
enum OpMode { opVerify, opGenerate, opFile, opFileT, opHelp } ;

//* Default filename for generating a lookup table *
const char* DFLT_TABLE_FILE = "crctable.out" ;

//* Default parameters for call to VerifyAlgorithm *
const unsigned char  CRC_REF_MESSAGE[] = "123456789" ;
const unsigned       CRC_16_CHECK = 0x0000BB3D ;
const unsigned       CRC_32_CHECK = 0xCBF43926 ;

//* Macro for bitshifting *
#define BITMASK(X) (1 << (X))

//*******************
//* Local functions *
//*******************
enum OpMode GetCommlineArgs ( int argc, char** argv, cm_t* cmtPtr, char* fName ) ;
unsigned int VerifyAlgorithm ( const unsigned char* cPtr, 
                               unsigned expectedCksm16, unsigned expectedCksm32 ) ;
unsigned int FileChecksum ( const char* fName, cm_t* cmtPtr, unsigned int expCksum ) ;
unsigned int FileChecksumT ( const char* fName, cm_t* cmtPtr, unsigned int expCksum ) ;
static void cksumIntBlock ( cm_t* cmtPtr, const unsigned int* lookupTbl, 
                            const unsigned char* blkAddr, unsigned int blkLen ) ;
static void cksumShortBlock ( cm_t* cmtPtr, const unsigned short* lookupTbl, 
                              const unsigned char* blkAddr, unsigned int blkLen ) ;
void DumpParms ( const cm_t* cmPtr ) ;
void HelpMe ( void ) ;
static unsigned int reflect ( unsigned v, int b) ;
static unsigned int widmask ( cm_t* p_cm ) ;
static void cm_ini ( cm_t* p_cm ) ;
static void cm_blk ( cm_t* p_cm, const unsigned char* blk_adr, unsigned int blk_len ) ;
static void cm_nxt ( cm_t* p_cm, unsigned int uch ) ;
static unsigned int cm_crc ( cm_t* p_cm ) ;


//*************************
//*        main           *
//*************************
//******************************************************************************
//* Program entry point.                                                       *
//*                                                                            *
//* Command-line Usage: crcmodel                                               *
//*                                                                            *
//* Command Line Arguments:                                                    *
//*                                                                            *
//*                                                                            *
//******************************************************************************

int main ( int argc, char* argv[], char* argenv[] )
{
   char fName[64] ;           // receives string arguments
   int exitCode = ZERO ;
   //* Default setup parameters. According to the documentation this  *
   //* setup is used for PKZip, AUTODIN II, Ethernet, FDDI and others.*
   cm_t cm =            // default setup parameters 
   {
      32,               // cm_width
      0x04C11DB7,       // cm_poly
      0xFFFFFFFF,       // cm_init
      true,             // cm_refin
      true,             // cm_refot
      0xFFFFFFFF        // cm_xorot
                        // cm_reg (initialized later)
   } ;

   system ( "clear" ) ;
   printf ( "crcmodel : v:%s  2017 The Software Samurai  (http://www.SoftwareSam.us/)\n"
            "           Written by Ross Williams, 1993 (http://www.ross.net/)\n"
            "           Heavily modified for student use by The Software Samurai.\n"
            "-------------------------------------------------------------------------------\n",
            AppVersion ) ;

   enum OpMode option = GetCommlineArgs ( argc, argv, &cm, fName ) ;

   if ( option == opVerify )
   {
      exitCode = VerifyAlgorithm ( CRC_REF_MESSAGE, CRC_16_CHECK, CRC_32_CHECK ) ;
   }
   else if ( option == opGenerate )
   {
      GenTable ( fName, cm.cm_width, cm.cm_poly, cm.cm_refin ) ;
   }
   else if ( option == opFile )
   {
      unsigned int expCksum = cm.cm_reg ; // get the 'expected checksum' (if any)
      cm.cm_reg = ZERO ;                  // reset interim register value
      exitCode = FileChecksum ( fName, &cm, expCksum ) ;
   }
   else if ( option == opFileT )
   {
      unsigned int expCksum = cm.cm_reg ; // get the 'expected checksum' (if any)
      cm.cm_reg = ZERO ;                  // reset interim register value
      exitCode = FileChecksumT ( fName, &cm, expCksum ) ;
   }
   else
   {
      HelpMe () ;
   }

   exit ( exitCode ) ;

}  //* End main() *

//******************************************************************************
//* Parse the user's command-line options.                                     *
//*                                                                            *
//* Input  : argc   : number of command-line tokens                            *
//*          argv   : pointer to an array of command-line tokens               *
//*          cmPtr  : pointer to a pre-initialized CRC Model (crc_m) structure *
//*          fName  : buffer to hold user-supplied filenames (or other strings)*
//*                                                                            *
//* Returns: 'false' if all parameters verified                                *
//*          'true'  if user needs help                                        *
//******************************************************************************
enum OpMode GetCommlineArgs ( int argc, char** argv, cm_t* cmtPtr, char* fName )
{
   #define DEBUG_OPTIONS (0)           // For debugging only

   unsigned int userNum ;              // user numeric input
   enum OpMode status = opHelp ;       // Return value

   //* If at least one argument AND it is a long-form switch *
   if ( (argc > 1) && (argv[1][0] == '-') && (argv[1][1] == '-') )
   {
      //******************
      //* Primary Option *
      //******************
      short tindex = 2 ;      // index for parsing token

      //* Verify that the model is working correctly *
      if ( (strncasecmp ( &argv[1][tindex], "test", 4 )) == ZERO )
      {
         status = opVerify ;
      }
      //* Generate a CRC lookup table *
      else if ( (strncasecmp ( &argv[1][tindex], "table", 5 )) == ZERO )
      {
         status = opGenerate ;
         tindex += 5 ;
         strncpy ( fName, DFLT_TABLE_FILE, 64 ) ;  // default output file *
         //* Set default parameters. Note: we use the members of the cm_t *
         //* structure in unorthodox ways here to reduce the number of    *
         //* parameters needed:                                           *
         //* cm.cm_width == register BIT width                            *
         //* cm.cm_poly  == poly value                                    *
         //* cm.cm_refin == indicates whether bits are reversed           *
         cmtPtr->cm_width = (TB_WIDTH * 8) ;// 32 bits (four bytes)
         cmtPtr->cm_poly  = TB_POLY ;       // 32-bit poly
         cmtPtr->cm_refin = TB_REVER ;      // reverse == 'true'

         //* If alternate output file specified *
         if ( (argv[1][tindex] == '=') && (argv[1][tindex + 1] != '\0') )
            strncpy ( fName, &argv[1][++tindex], 64 ) ;
         else                                      // use default
            ;
      }

      //* Generate a CRC checksum for the specified file *
      else if ( (strncasecmp ( &argv[1][tindex], "file", 4 )) == ZERO )
      {
         status = opFile ;
         tindex += 4 ;

         if ( (argv[1][tindex] == '=') && (argv[1][tindex + 1] != '\0') )
         {
            userNum = 0x00000000 ;

            //* Separate the filename from the optional 'expected checksum' *
            short indx = ++tindex ;
            for ( ; argv[1][indx] != '\0' ; ++indx )
            {
               if ( argv[1][indx] == '/' )
               {
                  argv[1][indx++] = '\0' ;    // isolate the filename
                  if ( (sscanf ( &argv[1][indx], "%X", &userNum )) == 1 )
                  {
                     cmtPtr->cm_reg = userNum ;

                     //* Continue scanning for optional table-lookup flag *
                     for ( ; argv[1][indx] != '\0' ; ++indx )
                     {
                        if ( (argv[1][indx] == '/') && 
                             ((argv[1][indx + 1] == 't') || (argv[1][indx + 1] == 'T')) )
                        {
                           status = opFileT ;
                           break ;
                        }
                     }
                  }
                  break ;
               }
            }

            //* Save the source filename *
            strncpy ( fName, &argv[1][tindex], 64 ) ;
         }
         else
            status = opHelp ;                   // syntax error
      }

      //* A cry for help *
      else if ( (strncasecmp ( &argv[1][tindex], "help", 4 )) == ZERO )
      {
         status = opHelp ;
      }
      else                    // unrecognized option
         ;


      //*******************************
      //* Scan for optional arguments *
      //*******************************
      if ( (status != opHelp) && (argc > 2) )
      {
         #if DEBUG_OPTIONS != 0
         printf ( "Optional arguments: argc:%d\n", argc ) ;
         #endif   // DEBUG_OPTIONS

         for ( int avIndex = 2 ; avIndex < argc ; ++avIndex )
         {
            #if DEBUG_OPTIONS != 0
            printf ( "   argv[%d]: '%s'\n", avIndex, argv[avIndex] ) ;
            #endif   // DEBUG_OPTIONS

            if ( (argv[avIndex][0] == '-') && (argv[avIndex][1] == '-') )
            {
               tindex = 2 ;

               if ( (strncasecmp ( &argv[avIndex][tindex], "poly", 4 )) == ZERO )
               {
                  tindex += 4 ;
                  if ( argv[avIndex][tindex++] == '=' )
                  {
                     if ( (sscanf( &argv[avIndex][tindex], "%X", &userNum )) == 1 )
                        cmtPtr->cm_poly = userNum ;
                     else           // syntax error
                        status = opHelp ;
                  }
                  else           // syntax error
                     status = opHelp ;

                  #if DEBUG_OPTIONS != 0
                  if ( status != opHelp )
                     printf ( "      cm_poly: %08Xh\n", cmtPtr->cm_poly ) ;
                  else
                     printf ( "      cm_poly: ERROR!\n" ) ;
                  #endif   // DEBUG_OPTIONS

                  if ( status == opHelp )
                     break ;
               }

               else if ( (strncasecmp ( &argv[avIndex][tindex], "reflect", 7 )) == ZERO )
               {
                  tindex += 7 ;
                  if ( argv[avIndex][tindex++] == '=' )
                  {
                     if ( (argv[avIndex][tindex] == 'f') || (argv[avIndex][tindex] == 'F') )
                        cmtPtr->cm_refin = false ;
                     else if ( (argv[avIndex][tindex] == 't') || (argv[avIndex][tindex] == 'T') )
                        cmtPtr->cm_refin = true ;
                     else        // syntax error
                        status = opHelp ;
                  }
                  else
                     status = opHelp ;

                  #if DEBUG_OPTIONS != 0
                  if ( status != opHelp )
                     printf ( "      cm_refin: %s\n", cmtPtr->cm_refin ? "true" : "false" ) ;
                  else
                     printf ( "      cm_refin: ERROR!\n" ) ;
                  #endif   // DEBUG_OPTIONS

                  if ( status == opHelp )
                     break ;
               }

               else if ( (strncasecmp ( &argv[avIndex][tindex], "reflout", 7 )) == ZERO )
               {
                  tindex += 7 ;
                  if ( argv[avIndex][tindex++] == '=' )
                  {
                     if ( (argv[avIndex][tindex] == 'f') || (argv[avIndex][tindex] == 'F') )
                        cmtPtr->cm_refot = false ;
                     else if ( (argv[avIndex][tindex] == 't') || (argv[avIndex][tindex] == 'T') )
                        cmtPtr->cm_refot = true ;
                     else        // syntax error
                        status = opHelp ;
                  }
                  else
                     status = opHelp ;

                  #if DEBUG_OPTIONS != 0
                  if ( status != opHelp )
                     printf ( "      cm_refot: %s\n", cmtPtr->cm_refot ? "true" : "false" ) ;
                  else
                     printf ( "      cm_refot: ERROR!\n" ) ;
                  #endif   // DEBUG_OPTIONS

                  if ( status == opHelp )
                     break ;
               }

               else if ( (strncasecmp ( &argv[avIndex][tindex], "regbits", 7 )) == ZERO )
               {
                  tindex += 7 ;
                  if ( argv[avIndex][tindex++] == '=' )
                  {
                     if ( ((sscanf( &argv[avIndex][tindex], "%u", &userNum )) == 1)
                         && (userNum == 16 || userNum == 32) )
                     {
                        cmtPtr->cm_width = userNum ;
                     }
                     else        // syntax error
                        status = opHelp ;
                  }
                  else           // syntax error
                     status = opHelp ;

                  #if DEBUG_OPTIONS != 0
                  if ( status != opHelp )
                     printf ( "      cm_width: %d\n", cmtPtr->cm_width ) ;
                  else
                     printf ( "      cm_width: ERROR!\n" ) ;
                  #endif   // DEBUG_OPTIONS

                  if ( status == opHelp )
                     break ;
               }

               else if ( (strncasecmp ( &argv[avIndex][tindex], "regfinal", 8 )) == ZERO )
               {
                  tindex += 8 ;
                  if ( argv[avIndex][tindex++] == '=' )
                  {
                     if ( (sscanf( &argv[avIndex][tindex], "%X", &userNum )) == 1 )
                        cmtPtr->cm_xorot = userNum ;
                     else           // syntax error
                        status = opHelp ;
                  }
                  else           // syntax error
                     status = opHelp ;

                  #if DEBUG_OPTIONS != 0
                  if ( status != opHelp )
                     printf ( "      cm_xorot: %08Xh\n", cmtPtr->cm_xorot ) ;
                  else
                     printf ( "      cm_xorot: ERROR!\n" ) ;
                  #endif   // DEBUG_OPTIONS

                  if ( status == opHelp )
                     break ;
               }

               else if ( (strncasecmp ( &argv[avIndex][tindex], "reginit", 7 )) == ZERO )
               {
                  tindex += 7 ;
                  if ( argv[avIndex][tindex++] == '=' )
                  {
                     if ( (sscanf( &argv[avIndex][tindex], "%X", &userNum )) == 1 )
                        cmtPtr->cm_init = userNum ;
                     else           // syntax error
                        status = opHelp ;
                  }
                  else           // syntax error
                     status = opHelp ;

                  #if DEBUG_OPTIONS != 0
                  if ( status != opHelp )
                     printf ( "      cm_init: %08Xh\n", cmtPtr->cm_init ) ;
                  else
                     printf ( "      cm_init: ERROR!\n" ) ;
                  #endif   // DEBUG_OPTIONS

                  if ( status == opHelp )
                     break ;
               }

               //* SPECIAL CASE: Setup parameters for OGG/Vorbis CRC checksum *
               else if ( (strncasecmp ( &argv[avIndex][tindex], "oggvorbis", 9 )) == ZERO )
               {
                  //* This setup has been verified to produce the same CRC as the *
                  //* fully-functional Page 2 CRC for two Ogg/Vorbis audio files, *
                  //* 1) "None Of Us Are Free.oga" by Simon Burke                 *
                  //*    Page 2 (comments) extracted as "NoneOfUs_01.bin".        *
                  //*    Checksum generated by VLC Media Player: C274CC0Ah        *
                  //* 2) "Ogg_Sample_1.ogg" from the Xiph.org website.            *
                  //*    Page 2 (comments) extracted as "OggSamp1_01.bin".        *
                  //*    Checksum generated by the libvorbis C library: 632E7789h *
                  cm_t crc_32 =
                  {
                     32,               // cm_width
                     0x04C11DB7,       // cm_poly
                     0x00000000,       // cm_init
                     false,            // cm_refin
                     false,            // cm_refot
                     0x00000000        // cm_xorot
                                       // cm_reg (non initialized)
                  } ;
                  cmtPtr->cm_width  = crc_32.cm_width ;
                  cmtPtr->cm_poly   = crc_32.cm_poly ;
                  cmtPtr->cm_init   = crc_32.cm_init ;
                  cmtPtr->cm_refin  = crc_32.cm_refin ;
                  cmtPtr->cm_refot  = crc_32.cm_refot ;
                  cmtPtr->cm_xorot  = crc_32.cm_xorot ;
               }
               
               else        // unrecognized option
                  status = opHelp ;
            }
            else           // syntax error
            {
               status = opHelp ;
               break ;
            }
         }        // for(;;)
      }           // argc > 2
   }              // argc > 1
   return status ;

}  //* End GetCommlineArgs() *

//******************************************************************************
//* Run a test on the algoritm for both 16-bit and 32-bit CRC.                 *
//*                                                                            *
//* Input  : cPtr           : pointer to binary byte sequence to be checksummed*
//*          expectedCksm16 : expected checksum for the 16-bit algorithm       *
//*          expectedCksm32 : expected checksum for the 32-bit algorithm       *
//*                                                                            *
//* Returns: the generated CRC checkskum                                       *
//******************************************************************************
unsigned int VerifyAlgorithm ( const unsigned char* cPtr, 
                               unsigned expectedCksm16, unsigned expectedCksm32 )
{
   cm_t crc_16 =
   {
      16,               // cm_width
      0x8005,           // cm_poly
      0x0000,           // cm_init
      true,             // cm_refin
      true,             // cm_refot
      0x0000,           // cm_xorot
                        // cm_reg (initialized below)
   } ;
   cm_t crc_32 =
   {
      32,               // cm_width
      0x04C11DB7,       // cm_poly
      0xFFFFFFFF,       // cm_init
      true,             // cm_refin
      true,             // cm_refot
      0xFFFFFFFF        // cm_xorot
                        // cm_reg (initialized below)
   } ;

   int seqLength = strlen ( (const char*)cPtr ) ;

   printf ( "\nTesting the reference byte sequence: '%s'\n", cPtr ) ;
   //* 16-bit test *
   printf ( "CRC Test (16-bit):\n" ) ;
   DumpParms ( &crc_16 ) ;
   cm_ini ( &crc_16 ) ;
   cm_blk ( &crc_16, cPtr, seqLength ) ;

   unsigned int cksm = cm_crc ( &crc_16 ) ;
   printf ( "   :: Expected: %04Xh  Calculated: %04Xh  %s\n",
            expectedCksm16, cksm, (cksm == expectedCksm16 ? "Match!" : "Error!") ) ;

   //* 32-bit test *
   printf ( "CRC Test (32-bit):\n" ) ;
   DumpParms ( &crc_32 ) ;
   cm_ini ( &crc_32 ) ;
   cm_blk ( &crc_32, cPtr, seqLength ) ;

   cksm = cm_crc ( &crc_32 ) ;
   printf ( "   :: Expected: %08Xh  Calculated: %08Xh  %s\n",
            expectedCksm32, cksm, (cksm == expectedCksm32 ? "Match!" : "Error!") ) ;

   return cksm ;

}

//******************************************************************************
//* Scan the specified file and generate a CRC checksum.                       *
//*                                                                            *
//* Input  : fName    : name of file to be scanned                             *
//*          cmtPTr   : pointer to an initialized parameter structure          *
//*          expCksum : expected checksum                                      *
//*                                                                            *
//* Returns: the generated CRC checkskum                                       *
//******************************************************************************
unsigned int FileChecksum ( const char* fName, cm_t* cmtPtr, unsigned int expCksum )
{
   const int bChunk = 1024 ;     // bytes processed for each iteration
   unsigned char b[bChunk] ;     // input buffer
   int bytesRead,                // bytes read per iteration
       totalBytes = ZERO ;       // total bytes read
   unsigned int cksm = ZERO ;    // return value

   printf ( "Processing: '%s' :: ", fName ) ;

   FILE* inPtr = fopen ( fName, "r" ) ;
   if ( inPtr != NULL )
   {
      cm_ini ( cmtPtr ) ;        // initialize the register

      do
      {
         if ( (bytesRead = fread ( b, 1, bChunk, inPtr )) > ZERO )
         {
            totalBytes += bytesRead ;
            cm_blk ( cmtPtr, b, bytesRead ) ;
         }
      }
      while ( bytesRead > ZERO ) ;

      fclose ( inPtr ) ;            // close the file

      cksm = cm_crc ( cmtPtr ) ;    // get the final checksum value

      //* For 16-bit calculation, clear the top 16 bits of the return value. *
      if ( cmtPtr->cm_width == 16 )
         cksm &= 0x0000FFFF ;

      printf ( "Bytes read: %u  Expected: %08Xh  Checksum: %08Xh\n",
               totalBytes, expCksum, cksm ) ;
   }
   else
      printf ( "Error! Unable to open the file.\n" ) ;
   return cksm ;

}  //* End FileChecksum() *

//******************************************************************************
//* Scan the specified file and generate a CRC checksum.                       *
//* This function produces a CRC identical to the FileChecksum() function      *
//* above, EXCEPT that it first creates a lookup table instead of doing the    *
//* calculation directly.                                                      *
//*                                                                            *
//* Input  : fName    : name of file to be scanned                             *
//*          cmtPTr   : pointer to an initialized parameter structure          *
//*          expCksum : expected checksum                                      *
//*                                                                            *
//* Returns: the generated CRC checkskum                                       *
//******************************************************************************
      /*
      Despite all the fuss I've made about understanding and defining CRC
      algorithms, the mechanics of their high-speed implementation remains
      trivial. There are really only two forms: normal and reflected. Normal
      shifts to the left and covers the case of algorithms with Refin=FALSE
      and Refot=FALSE. Reflected shifts to the right and covers algorithms
      with both those parameters true. (If you want one parameter true and
      the other false, you'll have to figure it out for yourself!) The
      polynomial is embedded in the lookup table (to be discussed). The
      other parameters, Init and XorOt can be coded as macros. Here is the
      32-bit normal form (the 16-bit form is similar).
      
      Note: I have carefully checked the above two code fragments, but I
      haven't actually compiled or tested them. This shouldn't matter to
      you, as, no matter WHAT you code, you will always be able to tell if
      you have got it right by running whatever you have created against the
      reference model given earlier. The code fragments above are really
      just a rough guide. The reference model is the definitive guide.
      
      Note: If you don't care much about speed, just use the reference model
      code!
      
         unsigned long crc_normal ();
         unsigned long crc_normal (blk_adr,blk_len)
         unsigned char *blk_adr;
         unsigned long  blk_len;
         {
          unsigned long crc = INIT;
          while (blk_len--)
             crc = crctable[((crc>>24) ^ *blk_adr++) & 0xFFL] ^ (crc << 8);
          return crc ^ XOROT;
         }
      
      Here is the reflected form:
      
         unsigned long crc_reflected ();
         unsigned long crc_reflected (blk_adr,blk_len)
         unsigned char *blk_adr;
         unsigned long  blk_len;
         {
          unsigned long crc = INIT_REFLECTED;
          while (blk_len--)
             crc = crctable[(crc ^ *blk_adr++) & 0xFFL] ^ (crc >> 8));
          return crc ^ XOROT;
         }
      */
//******************************************************************************
//* Using Ross's notes above, we have implemented 32-bit and 16-bit            *
//* algorithms which use lookup tables instead of direct calculation.          *
//* The actual performance delta is unknown, but roughly 25-30% fewer runtime  *
//* calculations are necessary with a static lookup table.                     *
//******************************************************************************
unsigned int FileChecksumT ( const char* fName, cm_t* cmtPtr, unsigned int expCksum )
{
   unsigned int cksm = ZERO ;    // return value

   #if 0    // FOR DEBUGGING ONLY
   DumpParms ( cmtPtr ) ;
   #endif   // FOR DEBUGGING ONLY

   printf ( "Processing: '%s' (T):: ", fName ) ;
   FILE* inPtr = fopen ( fName, "r" ) ;
   if ( inPtr != NULL )
   {
      const int bChunk = 1024 ;     // bytes processed for each iteration
      unsigned char b[bChunk] ;     // input buffer
      int bytesRead,                // bytes read per iteration
          totalBytes = ZERO ;       // total bytes read

      cm_ini ( cmtPtr ) ;           // initialize the register

      if ( cmtPtr->cm_width == 32 ) // for a 32-bit CRC
      {
         //* Generate a 32-bit lookup table *
         unsigned int lookupTbl[256] ;
         GenRamTable ( lookupTbl, cmtPtr->cm_width, cmtPtr->cm_poly, cmtPtr->cm_refin ) ;

         //* Read and process the source data *
         do
         {
            if ( (bytesRead = fread ( b, 1, bChunk, inPtr )) > ZERO )
            {
               totalBytes += bytesRead ;
               cksumIntBlock ( cmtPtr, lookupTbl, b, bytesRead ) ;
            }
         }
         while ( bytesRead > ZERO ) ;
      }

      else                          // for a 16-bit CRC
      {
         //* Generate a 16-bit lookup table *
         unsigned short lookupTbl[256] ;
         GenRamTable ( lookupTbl, cmtPtr->cm_width, cmtPtr->cm_poly, cmtPtr->cm_refin ) ;

         //* Read and process the source data *
         do
         {
            if ( (bytesRead = fread ( b, 1, bChunk, inPtr )) > ZERO )
            {
               totalBytes += bytesRead ;
               cksumShortBlock ( cmtPtr, lookupTbl, b, bytesRead ) ;
            }
         }
         while ( bytesRead > ZERO ) ;
      }

      fclose ( inPtr ) ;            // close the file

      //* Get the final checksum value.                                        *
      //* Note: If cm_refin == cm_refot, then no final reflection is necessary.*
      //*       If cm_refin != cm_refot, then we must do final reflection.     *
      //* This is because the input reflection is built into the lookup table. *
      if ( cmtPtr->cm_refin != cmtPtr->cm_refot )
         cksm = (reflect ( cmtPtr->cm_reg, cmtPtr->cm_width )) ^ cmtPtr->cm_xorot ;
      else
         cksm = cmtPtr->cm_reg ^ cmtPtr->cm_xorot ;

      //* For 16-bit calculation, clear the top 16 bits of the return value. *
      if ( cmtPtr->cm_width == 16 )
         cksm &= 0x0000FFFF ;

      printf ( "Bytes read: %u  Expected: %08Xh  Checksum: %08Xh\n",
               totalBytes, expCksum, cksm ) ;
   }
   else
      printf ( "Error! Unable to open the file.\n" ) ;

   return cksm ;

}  //* End FileChecksumT() *

//******************************************************************************
//* Processes a block of bytes from the input stream, generating a             *
//* 32-bit CRC.                                                                *
//*                                                                            *
//* Input  : cmtPtr   : pointer to the parameter structure                     *
//*          lookupTbl: pointer to a table containing 256 entries              *
//*          blkAddr  : pointer to source data                                 *
//*          blkLen   : number of elements in source arrray                    *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
static void cksumIntBlock ( cm_t* cmtPtr, const unsigned int* lookupTbl, 
                            const unsigned char* blkAddr, unsigned int blkLen )
{
   unsigned int crc = cmtPtr->cm_reg ;
   const unsigned char mask = 0xFF ;

   for ( int indx = ZERO ; indx < blkLen ; ++indx )
   {
      if ( cmtPtr->cm_refin )
         crc = lookupTbl[(crc ^ blkAddr[indx]) & mask] ^ (crc >> 8) ;
      else
         crc = lookupTbl[((crc >> 24) ^ blkAddr[indx]) & mask] ^ (crc << 8) ;
   }
   cmtPtr->cm_reg = crc ;

}  //* End cksumBlock() *

//******************************************************************************
//* Processes a block of bytes from the input stream, generating a             *
//* 16-bit CRC.                                                                *
//*                                                                            *
//* Input  : cmtPtr   : pointer to the parameter structure                     *
//*          lookupTbl: pointer to a table containing 256 entries              *
//*          blkAddr  : pointer to source data                                 *
//*          blkLen   : number of elements in source arrray                    *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
static void cksumShortBlock ( cm_t* cmtPtr, const unsigned short* lookupTbl, 
                              const unsigned char* blkAddr, unsigned int blkLen )
{
   unsigned short crc = cmtPtr->cm_reg ;
   const unsigned char mask = 0xFF ;

   for ( int indx = ZERO ; indx < blkLen ; ++indx )
   {
      if ( cmtPtr->cm_refin )
         crc = lookupTbl[(crc ^ blkAddr[indx]) & mask] ^ (crc >> 8) ;
      else
         crc = lookupTbl[((crc >> 8) ^ blkAddr[indx]) & mask] ^ (crc << 8) ;
   }
   cmtPtr->cm_reg = crc ;

}  //* End cksumBlock() *

//******************************************************************************
//* Write contents of the parameter structure to stdout.                       *
//*                                                                            *
//* Input  : pointer to the parameter structure                                *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
void DumpParms ( const cm_t* cmPtr )
{
   if ( cmPtr->cm_width <= 16 )     // format output for 16-bit values
   {
      printf ( "   cm_width: %d\n"
               "   cm_poly : %04Xh\n"
               "   cm_init : %04Xh\n"
               "   cm_refin: %s\n"
               "   cm_refot: %s\n"
               "   cm_xorot: %04Xh\n",
               cmPtr->cm_width, cmPtr->cm_poly, cmPtr->cm_init, 
               (cmPtr->cm_refin ? "true" : "false"),
               (cmPtr->cm_refot ? "true" : "false"),
               cmPtr->cm_xorot ) ;
   }
   else                             // format output for 32-bit values
   {
      printf ( "   cm_width: %d\n"
               "   cm_poly : %08Xh\n"
               "   cm_init : %08Xh\n"
               "   cm_refin: %s\n"
               "   cm_refot: %s\n"
               "   cm_xorot: %08Xh\n",
               cmPtr->cm_width, cmPtr->cm_poly, cmPtr->cm_init, 
               (cmPtr->cm_refin ? "true" : "false"),
               (cmPtr->cm_refot ? "true" : "false"),
               cmPtr->cm_xorot ) ;
   }
}  //* End DumpParms() *

//******************************************************************************
//* Returns the value v with the bottom b [0,32] bits reflected.               *
//* Example: reflect(0x3e23,3) yields 0x3e26                                   *
//*          0011 1110 0010 0011 yields 0011 1110 0010 0110                    *
//*          3    E    2    3---        3    E    2    6---                    *
//*                                                                            *
//* Input  : val  : value to be modified                                       *
//*          bits : number of LSB bits to be reflected                         *
//*                                                                            *
//* Returns: reflected value                                                   *
//******************************************************************************
static unsigned int reflect ( unsigned int val, int bits )
{
   unsigned int t = val ;           // single-bit mask

   for ( int i = 0 ; i < bits ; i++ )
   {
      if ( t & 1 )
         val |=  BITMASK((bits-1)-i) ;
      else
         val &= ~BITMASK((bits-1)-i) ;
      t >>= 1 ;
   }
   return val ;
}

//******************************************************************************
//* Returns a integer whose value is (2^p_cm->cm_width)-1.                     *
//* The trick is to do this portably (e.g. without doing <<32).                *
//*                                                                            *
//* Input  : pointer to parameter structure                                    *
//*                                                                            *
//* Returns: the bit mask for eliminating carry bits                           *
//******************************************************************************
//* Notes:                                                                     *
//* Creates a bit mask with the exact width of the target register.            *
//*                                                                            *
//* For a 32-bit register width, this yields:                                  *
//*            1111 1111 1111 1111 1111 1111 1111 1111                         *
//* For a 16-bit register width, this yields:                                  *
//*            0000 0000 0000 0000 1111 1111 1111 1111                         *
//* For a  9-bit register width, this would yield:                             *
//*            0000 0000 0000 0000 0000 0001 1111 1111                         *
//*                                                                            *
//******************************************************************************
static unsigned int widmask ( cm_t* p_cm )
{
   return (((1 << (p_cm->cm_width-1)) - 1) << 1) | 1 ;
}

//******************************************************************************
//* Sets the initial value of the accumulator register.                        *
//* (This function is nearly useless.)                                         *
//*                                                                            *
//* Input  : pointer to parameter structure                                    *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
static void cm_ini ( cm_t* p_cm )
{
   p_cm->cm_reg = p_cm->cm_init;
}

//******************************************************************************
//* Processes a single message byte [0,255].                                   *
//*                                                                            *
//******************************************************************************
static void cm_nxt ( cm_t* p_cm, unsigned int uch )
{
   int   i;
   unsigned int topbit = BITMASK(p_cm->cm_width-1) ;
   
   if ( p_cm->cm_refin )            // if specified, reflect the source bits
      uch = reflect ( uch, 8 ) ;

   //* Shift LSByte into MSByte position and XOR with the register value. *
   p_cm->cm_reg ^= (uch << (p_cm->cm_width-8));

   for ( i = 0 ; i < 8 ; i++ )
   {
      //* If MSB of the register is set, then     *
      //* shift out the MSB and XOR with the poly.*
      if ( p_cm->cm_reg & topbit )
         p_cm->cm_reg = (p_cm->cm_reg << 1) ^ p_cm->cm_poly;
      //* Otherwise, simply shift register left by one bit.*
      else
         p_cm->cm_reg <<= 1 ;

      //* Apply the width mask to eliminate carry.*
      p_cm->cm_reg &= widmask ( p_cm ) ;
   }
}

//******************************************************************************
//* Processes a block of bytes from the input stream.                          *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
static void cm_blk ( cm_t* p_cm, const unsigned char* blk_adr, unsigned int blk_len )
{

   while (blk_len--) cm_nxt ( p_cm, *blk_adr++ ) ;

}

//******************************************************************************
//* Returns the CRC value for the message bytes processed so far.              *
//*                                                                            *
//******************************************************************************
static unsigned int cm_crc ( cm_t* p_cm )
{
   if (p_cm->cm_refot)
      return p_cm->cm_xorot ^ reflect( p_cm->cm_reg, p_cm->cm_width ) ;
   else
      return p_cm->cm_xorot ^ p_cm->cm_reg ;
}

//******************************************************************************
//* Generate a table entry.                                                    *
//* Although Ross's note below indicates that this function can also be used   *
//* at runtime, the reference model checksum generator does not use a table    *
//* lookup (see FileChecksum()). However, in addition to the direct calculation*
//* done by the reference model code, we dynamically build a lookup table      *
//* as an additional test of the algorithm (see FileChecksumT()).              *
//*                                                                            *
//* As mentioned elsewhere, we have converted 'unsigned long' to 'unsigned int'*
//* throughout the application. This function has been verified to produce     *
//* identical lookup tables as the original.                                   *
//*                                                                            *
//*                                                                            *
/* The following function can be used to calculate a CRC lookup table.        */
/* It can also be used at run-time to create or check static tables.          */
/* Returns the i'th entry for the lookup table for the specified algorithm.   */
/* The function examines the field's cm_width, cm_poly, cm_refin, and the     */
/* argument table index in the range [0,255] and returns the table entry in   */
/* the bottom cm_width bytes of the return value.                             */
//******************************************************************************
unsigned int cm_tab ( cm_t* p_cm, int index )
{
   int          i ;
   unsigned int r,
                topbit = BITMASK(p_cm->cm_width-1),
                inbyte = (unsigned int) index ;
   
   if ( p_cm->cm_refin )
      inbyte = reflect ( inbyte, 8 ) ;
   r = inbyte << (p_cm->cm_width - 8) ;
   for ( i = 0 ; i < 8 ; i++ )
   {
      if (r & topbit)
         r = (r << 1) ^ p_cm->cm_poly ;
      else
         r <<= 1 ;
   }
   if ( p_cm->cm_refin )
      r = reflect (r, p_cm->cm_width ) ;
   return r & widmask(p_cm) ;
}

//******************************************************************************
//* Display command-line help.                                                 *
//******************************************************************************
void HelpMe ( void )
{
   printf (
      "\n"
      "Usage: crcmodel --[test | file | table] OPTIONS   \n"
      "                (If arguments contain spaces, enclose in quotes.)\n"
      "                Exit value: For options that generate a checksum, the 32-bit\n"
      "                            (hex) checksum will be returned, otherwise zero.\n"
      "  --test       : Test the algorithm for both 16-bit and 32-bit checksums.\n"
      "                 Test sequence: \"123456789\" (9 ASCII bytes).\n"
      "  --file=FILENAME[/EXPECTED_CHECKSUM[/T]]\n"
      "               : Scan the specified file and create a CRC checksum for it.\n"
      "                 Optionally, specify the expected checksum.\n"
      "                 Optionally, specify that a table-based algorithm be used.\n"
      "  --table[=FILENAME] : Generate a CRC lookup table and write it to a file.\n"
      "              Optionally specify output filename.\n"
      "              Default output file will be \"crctable.out\"\n"
      "              Default setup will be regbits==32, poly==04C11DB7h\n"
      "              and reflect==true,\n"
      "              The following options may be used to modify the format\n"
      "              of the table: '--regbits', '--poly', and '--reflect'\n"
      "  Options:\n"
      "\n"
      "  --poly=POLY  : The so-called 'generator polynomial' or hexadecimal XOR value.\n"
      "                 Default: 04C11DB7h  This is the value used in many\n"
      "                 popular algorithms including the one of particular\n"
      "                 interest to us, which is the CRC checksum used by\n"
      "                 OGG/Vorbis audio tag data. Any hexadecimal value will work,\n"
      "                 so long as it is the same width as the register which is\n"
      "                 specified by the '--regbits' option below.\n"
      "  --reflect=[true|false] : Direction of bit processing for input bytes.\n"
      "                 'true'  for algorithms which shift bits right (default)\n"
      "                 'false' for algorithms which shift bits left\n"
      "  --reflout=[true|false] : Reverse the order of bits for the final CRC\n"
      "                  checksum value before returning it.\n"
      "  --regbits=[16|32] : Number of bits in accumulator register.\n"
      "                  16 == 16-bit calculations\n"
      "                  32 == 32-bit calculations (default)\n"
      "  --regfinal=HEXVALUE : Post-calculation XOR value (in hexadecimal).\n"
      "                  Perform an XOR operation on the final CRC checksum value\n"
      "                  before returning it.\n"
      "  --reginit=HEXVALUE : Initial register value (in hexadecimal).\n"
      "                  This will be the register value when processing begins.\n"
      "  --oggvorbis  : Special Case: Set up processing parameters for OGG/Vorbis\n"
      "                 metadata (tag data) checksum generation.\n"
      "  --help       : Display a list of command-line options (this list)\n"
      ) ;
}  //* End HelpMe() *

#undef DEFINE_GLOBALS_HERE
