//******************************************************************************
//* File       : crcmodel.h                                                    *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2017      Mahlon R. Smith, The Software Samurai *
//*                  GNU GPL copyright notice at the top of crcmodel.c         *
//* Date       : 13-Feb-2017                                                   *
//* Version    : (see AppVersion string)                                       *
//*                                                                            *
//* Description: Definition of the CRC checksum parameter structure and        *
//*              miscellaneous constants.                                      *
//*              Most of Ross Williams' original notes have been retained      *
//*              as C-style comments, even though some of them no longer apply.*
//*              See also the notes at the top of crcmodel.c                   *
//******************************************************************************

/******************************************************************************/
/*                                                                            */
/* Author : Ross Williams (ross@guest.adelaide.edu.au.).                      */
/* Date   : 3 June 1993.                                                      */
/* Status : Public domain.                                                    */
/*                                                                            */
/* Description : This is the header (.h) file for the reference               */
/* implementation of the Rocksoft^tm Model CRC Algorithm. For more            */
/* information on the Rocksoft^tm Model CRC Algorithm, see the document       */
/* titled "A Painless Guide to CRC Error Detection Algorithms" by Ross        */
/* Williams (ross@guest.adelaide.edu.au.). This document is likely to be in   */
/* "ftp.adelaide.edu.au/pub/rocksoft".                                        */
/*                                                                            */
/* Note: Rocksoft is a trademark of Rocksoft Pty Ltd, Adelaide, Australia.    */
/*                                                                            */
/******************************************************************************/
/*                                                                            */
/* How to Use This Package                                                    */
/* -----------------------                                                    */
/* Step 1: Declare a variable of type cm_t. Declare another variable          */
/*         (p_cm say) of type p_cm_t and initialize it to point to the first  */
/*         variable (e.g. p_cm_t p_cm = &cm_t).                               */
/*                                                                            */
/* Step 2: Assign values to the parameter fields of the structure.            */
/*         If you don't know what to assign, see the document cited earlier.  */
/*         For example:                                                       */
/*            p_cm->cm_width = 16;                                            */
/*            p_cm->cm_poly  = 0x8005L;                                       */
/*            p_cm->cm_init  = 0L;                                            */
/*            p_cm->cm_refin = true;                                          */
/*            p_cm->cm_refot = true;                                          */
/*            p_cm->cm_xorot = 0L;                                            */
/*         Note: Poly is specified without its top bit (18005 becomes 8005).  */
/*         Note: Width is one bit less than the raw poly width.               */
/*                                                                            */
/* Step 3: Initialize the instance with a call cm_ini(p_cm);                  */
/*                                                                            */
/* Step 4: Process zero or more message bytes by placing zero or more         */
/*         successive calls to cm_nxt. Example: cm_nxt(p_cm, ch);             */
/*                                                                            */
/* Step 5: Extract the CRC value at any time by calling crc = cm_crc(p_cm);   */
/*         If the CRC is a 16-bit value, it will be in the bottom 16 bits.    */
/*                                                                            */
/******************************************************************************/
/*                                                                            */
/* Design Notes                                                               */
/* ------------                                                               */
/* PORTABILITY: This package has been coded very conservatively so that       */
/* it will run on as many machines as possible. For example, all external     */
/* identifiers have been restricted to 6 characters and all internal ones to  */
/* 8 characters. The prefix cm (for Crc Model) is used as an attempt to avoid */
/* namespace collisions. This package is endian independent.                  */
/*                                                                            */
/* EFFICIENCY: This package (and its interface) is not designed for           */
/* speed. The purpose of this package is to act as a well-defined reference   */
/* model for the specification of CRC algorithms. If you want speed, cook up  */
/* a specific table-driven implementation as described in the document cited  */
/* above. This package is designed for validation only; if you have found or  */
/* implemented a CRC algorithm and wish to describe it as a set of parameters */
/* to the Rocksoft^tm Model CRC Algorithm, your CRC algorithm implementation  */
/* should behave identically to this package under those parameters.          */
/*                                                                            */
/******************************************************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

//* Define convenient boolean reference *
typedef unsigned char   bool;
#ifndef false
#define false (0)
#define true (1)
#endif
#define ZERO (0)

/******************************************************************************/
/* CRC Model Abstract Type */
/* ----------------------- */
/* The following type stores the context of an executing instance of the  */
/* model algorithm. Most of the fields are model parameters which must be */
/* set before the first initializing call to cm_ini.                      */
#if 1    // UNDER CONSTRUCTION
// NOTES: a) Both 16-bit and 32-bit table generation have been verified identical
//           with old and new formats.
//        b) Test sequence for both 16-bit and 32-bit generate the expected
//           checksum with old and new formats.
//        c) Scan of OGG/Vorbis source file generate the expected checksum
//           with both old and new formats.
typedef struct
{
int            cm_width ;  /* Parameter: Width in bits [8,32].       */
unsigned int   cm_poly ;   /* Parameter: The algorithm's polynomial. */
unsigned int   cm_init ;   /* Parameter: Initial register value.     */
bool           cm_refin ;  /* Parameter: Reflect input bytes?        */
bool           cm_refot ;  /* Parameter: Reflect output CRC?         */
unsigned int   cm_xorot ;  /* Parameter: XOR this to output CRC.     */

unsigned int   cm_reg;     /* Context: Context during execution.     */
} cm_t;
#else    // FUNCTIONAL
// OBSOLETE - LONG INTS HAVE BEEN CONVERTED unsigned int
typedef struct
{
int   cm_width;   /* Parameter: Width in bits [8,32].       */
xlong cm_poly;    /* Parameter: The algorithm's polynomial. */
xlong cm_init;    /* Parameter: Initial register value.     */
bool  cm_refin;   /* Parameter: Reflect input bytes?        */
bool  cm_refot;   /* Parameter: Reflect output CRC?         */
xlong cm_xorot;   /* Parameter: XOR this to output CRC.     */

xlong cm_reg;     /* Context: Context during execution.     */
} cm_t;
typedef cm_t *x_cm_t;
#endif   // FUNCTIONAL

/* NAME: This is a name given to the algorithm. A string value.

   WIDTH: This is the width of the algorithm expressed in bits. This
   is one less than the width of the Poly.

   POLY: This parameter is the poly. This is a binary value that
   should be specified as a hexadecimal number. The top bit of the
   poly should be omitted. For example, if the poly is 10110, you
   should specify 0110 (06 hexadecimal). An important aspect of this 
   parameter is that it represents the unreflected poly; the bottom 
   bit of this parameter is always the LSB of the divisor during 
   the division regardless of whether the algorithm being modelled 
   is reflected.

   INIT: This parameter specifies the initial value of the register
   when the algorithm starts. This is the value that is to be assigned
   to the register in the direct table algorithm. In the table
   algorithm, we may think of the register always commencing with the
   value zero, and this value being XORed into the register after the
   N'th bit iteration. This parameter should be specified as a
   hexadecimal number.

   REFIN: This is a boolean parameter. If it is FALSE, input bytes are
   processed with bit 7 being treated as the most significant bit
   (MSB) and bit 0 being treated as the least significant bit. If this
   parameter is TRUE, each byte is reflected before being processed.

   REFOUT: This is a boolean parameter. If it is set to FALSE, the
   final value in the register is fed into the XOROUT stage directly,
   otherwise, if this parameter is TRUE, the final register value is
   reflected first.

   XOROUT: This is an W-bit value that should be specified as a
   hexadecimal number. It is XORed to the final register value (after
   the REFOUT) stage before the value is returned as the official
   checksum.

   CHECK: This field is not strictly part of the definition, and, in
   the event of an inconsistency between this field and the other
   fields, the other fields take precedence. This field is a check
   value that can be used as a weak validator of implementations of
   the algorithm. The field contains the checksum obtained when the
   ASCII string "123456789" is fed through the specified algorithm
   i.e. 31h 32h 33h ... in hexadecimal.
*/
/******************************************************************************/


//* Global data defined in crcmodel.c and declared extern elsewhere *
#ifdef DEFINE_GLOBALS_HERE
//* Default output filename for generated lookup table.                        *
const char* const TB_FILE = "crctable.out" ;
//* Default "width" number of bytes for table entries.                         *
//* 2 == 16-bit lookup, 4 == 32-bit lookup (default)                           *
const unsigned int TB_WIDTH = 4 ;
//* Default "polynomial" for generating the lookup table.                      *
const unsigned int TB_POLY = 0x04C11DB7 ;
//* Default "reverse" flag. 'true' == shift right, 'false' == shift left.      *
const bool TB_REVER = true ;

#else    // ! DEFINE_GLOBALS_HERE
extern const char* const TB_FILE ;
extern const unsigned int TB_WIDTH ;
const unsigned int TB_POLY ;
extern const bool TB_REVER ;
#endif   // ! DEFINE_GLOBALS_HERE


//* Control functions for generating a CRC lookup table.                       *
void GenTable ( const char* outFile, int regBits, unsigned int poly, bool reverse ) ;
void GenRamTable ( void* tblPtr, int regBits, unsigned int poly, bool reflect ) ;

//* Generate a table entry.                                                    *
unsigned int cm_tab ( cm_t* p_cm, int index ) ;

