//******************************************************************************
//* File       : crctable.c                                                    *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2017      Mahlon R. Smith, The Software Samurai *
//*                  GNU GPL copyright notice at the top of crcmodel.c         *
//* Date       : 13-Feb-2017                                                   *
//* Version    : (see AppVersion string)                                       *
//*                                                                            *
//* Description: Generate a 16-bit or 32-bit CRC lookup table.                 *
//*              See below, and see also the notes at the top of crcmodel.c    *
//******************************************************************************

/******************************************************************************/
/*                                                                            */
/* Author  : Ross Williams (ross@guest.adelaide.edu.au.).                     */
/* Date    : 3 June 1993.                                                     */
/* Version : 1.0.                                                             */
/* Status  : Public domain.                                                   */
/*                                                                            */
/* Description : This program writes a CRC lookup table (suitable for         */
/* inclusion in a C program) to a designated output file. The program can be  */
/* statically configured to produce any table covered by the Rocksoft^tm      */
/* Model CRC Algorithm. For more information on the Rocksoft^tm Model CRC     */
/* Algorithm, see the document titled "A Painless Guide to CRC Error          */
/* Detection Algorithms" by Ross Williams (ross@guest.adelaide.edu.au.). This */
/* document is likely to be in "ftp.adelaide.edu.au/pub/rocksoft".            */
/*                                                                            */
/* Note: Rocksoft is a trademark of Rocksoft Pty Ltd, Adelaide, Australia.    */
/*                                                                            */
/******************************************************************************/

#include "crcmodel.h"

static void gentable ( FILE* outfile, int regBits, unsigned int poly, bool reflect ) ;


//******************************************************************************
//* Generate the lookup table.                                                 *
//* Note the risk: no error checking is done on the file I/O.                  *
//*                                                                            *
//* Input  : outFile   : name of output file                                   *
//*          regBits   : register (and table entry) width in binary bits       *
//*                      16 bits == 2-byte register, 32 bits == 4-byte register*
//*          poly      : "polynomial" value to use for generating the table    *
//*                      if regBits == 16, only lower 16-bits are used         *
//*                      if regBits == 32, 32 bits are used                    *
//*          reflect   : 'false' bits are processed high-to-low (bit 7,6...0)  *
//*                      'true'  bits are processed low-to-high                *
//*                              bits are swapped (7->0, 6->1, 5->2, etc.)     *
//*                              then the bits are processed (bit 7, 6,...0)   *
//* Returns: nothing                                                           *
//******************************************************************************
void GenTable ( const char* outFile, int regBits, unsigned int poly, bool reflect )
{
   const char* regBitsMsg = "Error! Width parameter must be either 16 bits or 32 bits.\n" ;
   const char* polyMsg    = "Error! Specified polynomial is wider than 16 bits.\n" ;
   printf( "\n"
           "Generating CRC Algorithm Table\n"
           "------------------------------\n"
           "Register    : %d bits\n"
           "Polynomial  : %X\n"
           "Reflect     : %s\n"
           "Output file : '%s'", regBits, poly, (reflect ? "true" : "false"), outFile);

   FILE* outfile = fopen ( outFile,"w" ) ; // open the file

   if ( outfile != NULL )                 // if file was opened successfully
   {
      bool goodParms = true ;
      if ( (regBits != 16) && (regBits != 32) )
      {
         printf ( "\n%s (%d)", regBitsMsg, regBits ) ;
         fprintf ( outfile, regBitsMsg ) ;
         goodParms = false ;
      }
      if ( (regBits == 16) && (poly & 0x0FFFF0000) )
      {
         printf ( "\n%s", polyMsg ) ;
         fprintf ( outfile, polyMsg ) ;
         goodParms = false ;
      }

      //* Generate the table and write it to file *
      if ( goodParms )
         gentable ( outfile, regBits, poly, reflect ) ;

      fclose ( outfile ) ;                   // close the file

      printf( "   DONE\n\n");
   }
   else
      printf ( "Error! Unable to open file for writing.\n" ) ;

}  //* End GenTable() *

//******************************************************************************
//* Generate the lookup table.                                                 *
//*                                                                            *
//* Input  : outfile   : stream pointer to file opened for writing             *
//*          regBits   : register (and table entry) width in binary bits       *
//*                      16 bits == 2-byte register, 32 bits == 4-byte register*
//*          poly      : "polynomial" value to use for generating the table    *
//*                      if regBits == 16, only lower 16-bits are used         *
//*                      if regBits == 32, 32 bits are used                    *
//*          reflect   : 'false' bits are processed high-to-low (bit 7,6...0)  *
//*                      'true'  bits are processed low-to-high                *
//*                              bits are swapped (7->0, 6->1, 5->2, etc.)     *
//*                              then the bits are processed (bit 7, 6,...0)   *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Notes:                                                                     *
//* -- In Ross's original code, all setup parameters were #define's.           *
//*    (See Ross's notes below.)                                               *
//*    The setup values are now call parameters.                               *
//*    All parameters have been range-checked by the calling function.         *
//* -- As noted elsewhere, the meanings of 'short int', 'int', and 'long int'  *
//*    have been adjusted for modern (> 16-bit) hardware                       *
//* -- No error checking is done on the file I/O.                              *
//******************************************************************************
/******************************************************************************/
/* TABLE PARAMETERS                                                           */
/* ================                                                           */
/* The following parameters entirely determine the table to be generated. You */
/* should need to modify only the definitions in this section before running  */
/* this program.                                                              */
/*                                                                            */
/*    TB_FILE  is the name of the output file.                                */
/*    TB_WIDTH is the table width in bytes (either 2 or 4).                   */
/*    TB_POLY  is the "polynomial", which must be TB_WIDTH bytes wide.        */
/*    TB_REVER indicates whether the table is to be reversed (reflected).     */
/*                                                                            */
/* Example:                                                                   */
/*                                                                            */
/*    #define TB_FILE   "crctable.out"                                        */
/*    #define TB_WIDTH  2                                                     */
/*    #define TB_POLY   0x8005L                                               */
/*    #define TB_REVER  true                                                  */
/******************************************************************************/

void gentable ( FILE* outfile, int regBits, unsigned int poly, bool reflect )
{

   fprintf ( outfile, 
   "/*****************************************************************/\n"
   "/*                                                               */\n"
   "/* CRC LOOKUP TABLE                                              */\n"
   "/* ================                                              */\n"
   "/* The following CRC lookup table was generated automagically    */\n"
   "/* by the Rocksoft^tm Model CRC Algorithm Table Generation       */\n"
   "/* Program V1.0 using the following model parameters:            */\n"
   "/*                                                               */\n"
   "/*    Width   : %d bytes.                                         */\n",
   (regBits / 8) ) ;

   if ( regBits == 16 )
      fprintf( outfile, 
      "/*    Poly    : 0x%04X                                           */\n",
      poly ) ;
   else
      fprintf ( outfile, 
      "/*    Poly    : 0x%08X                                       */\n",
      poly ) ;
   fprintf ( outfile,
   "/*    Reverse : %s                                            */\n",
   reflect ? "true " : "false" ) ;

   fprintf ( outfile,
      "/*                                                               */\n"
      "/* For more information on the Rocksoft^tm Model CRC Algorithm,  */\n"
      "/* see the document titled \"A Painless Guide to CRC Error        */\n"
      "/* Detection Algorithms\" by Ross Williams                        */\n"
      "/* (ross@guest.adelaide.edu.au.). This document is likely to be  */\n"
      "/* in the FTP archive \"ftp.adelaide.edu.au/pub/rocksoft\".        */\n"
      "/*                                                               */\n"
      "/*****************************************************************/\n"
      "\n");

   fprintf ( outfile, "unsigned %s crctable[256] =\n{\n", regBits == 16 ? "short" : "int" ) ;

   cm_t cm ;    // parameters for generating an entry
   char *form    = (regBits == 16) ? " 0x%04X" : " 0x%08X";  // entry format
   int   perline = (regBits == 16) ? 8 : 4;                  // entries per output line

   cm.cm_width = regBits ;           // register width in BITS
   cm.cm_poly  = poly ;              // "polynomial" value
   cm.cm_refin = reflect ;           // direction of shift

   //* Generate a table of 256 entries, one entry for each possible 8-bit value.*
   for ( int i = 0 ; i < 256 ; ++i )
   {
     fprintf ( outfile, form, cm_tab ( &cm, i ) ) ;// write an entry
     if (i != 255)                                 // no ',' after last line
        fprintf ( outfile, "," ) ;
     if (((i+1) % perline) == 0)                   // line break
        fprintf( outfile, "\n");
   }

   fprintf (outfile, 
    "};\n"
    "\n"
    "/*****************************************************************/\n"
    "/*                   End of CRC Lookup Table                     */\n"
    "/*****************************************************************/\n"
    "\n");
   fflush ( outfile ) ;    // be sure all data are written to the file

}

//******************************************************************************
//* This function is the same as the gentable() function above, except that    *
//* it writes the table to RAM memory rather that to a file.                   *
//*                                                                            *
//* Input  : tblPtr    : pointer to an allocated memory space which is large   *
//*                      enough to hold the table:                             *
//*                      256 x 2 bytes for a 16-bit table,                     *
//*                      256 x 4 bytes for a 32-bit table,                     *
//*          regBits   : register (and table entry) width in binary bits       *
//*                      16 bits == 2-byte register, 32 bits == 4-byte register*
//*          poly      : "polynomial" value to use for generating the table    *
//*                      if regBits == 16, only lower 16-bits are used         *
//*                      if regBits == 32, 32 bits are used                    *
//*          reflect   : 'false' bits are processed high-to-low (bit 7,6...0)  *
//*                      'true'  bits are processed low-to-high                *
//*                              bits are swapped (7->0, 6->1, 5->2, etc.)     *
//*                              then the bits are processed (bit 7, 6,...0)   *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
void GenRamTable ( void* tblPtr, int regBits, unsigned int poly, bool reflect )
{
   cm_t cm ;    // parameters for generating an entry
   cm.cm_width = regBits ;           // register width in BITS
   cm.cm_poly  = poly ;              // "polynomial" value
   cm.cm_refin = reflect ;           // direction of shift

   //* Generate a table of 256 entries, one entry for each possible 8-bit value.*
   if ( regBits == 16 )
   {
      unsigned short* sPtr = (unsigned short*)tblPtr ;
      for ( int indx = 0 ; indx < 256 ; ++indx )
      {
         sPtr[indx] = (unsigned short)(cm_tab ( &cm, indx )) ; // write an entry
      }
   }
   else     // (regBits == 32)
   {
      unsigned int* iPtr = (unsigned int*)tblPtr ;
      for ( int indx = 0 ; indx < 256 ; ++indx )
      {
         iPtr[indx] = cm_tab ( &cm, indx ) ;    // write an entry
      }
   }

}  //* End GenRamTable() *

//******************************************************************************
//* Error status on output file.                                               *
//* This function is unnecessary and has been removed.                         *
//*                                                                            *
//* Note to students:                                                          *
//*  -- This code in this function is a masterpiece of what not to do.         *
//*     -- First, there should be only one call to exit() in an application.   *
//*     -- Next, the actual purpose of this function is to check for I/O       *
//*        errors, which should be done in-line with the output call for       *
//*        purposes of self-documenting code. To hide the exit() call in a     *
//*        low-level call is confusing, and even frightening. This is worse    *
//*        than letting the application crash...                               *
//******************************************************************************
/*
static void chk_err ( char* mess, FILE* outfile )
 * If mess is non-empty, write it out and abort. Otherwise, check the error   *
 * status of outfile and abort if an error has occurred.                      *
{
 if (mess[0] != 0   ) {printf("%s\n",mess); exit(EXIT_FAILURE);}
 if (ferror(outfile)) {perror("chk_err");   exit(EXIT_FAILURE);}
}
*/

//******************************************************************************
//* Validate parameters for generating lookup table.                           *
//* This function does nothing useful and has been removed.                    *
//* See error checking in GenTable(), above.                                   *
//*                                                                            *
//* Note to students:                                                          *
//*  -- This code in this function shows one of the major logical errors of    *
//*     coding in the C-language family, namely that a boolean value is either *
//*     zero or not-zero.                                                      *
//*     A boolean value is tested as:                                          *
//*        bool flag = false ;                                                 *
//*        if ( flag == false ) { /* do stuff */ }  or                         *
//*        if ( ! flag )        { /* do stuff */ }                             *
//*           --OR--                                                           *
//*        if ( flag != false ) { /* do stuff */ }  or                         *
//*        if ( flag )          { /* do stuff */ }                             *
//*     NEVER test a boolean value for 'true'                                  *
//******************************************************************************
/*
static void chkparam ( FILE* outfile )
{
 if ((TB_WIDTH != 2) && (TB_WIDTH != 4))
    chk_err("chkparam: Width parameter is illegal.", outfile);
 if ((TB_WIDTH == 2) && (TB_POLY & 0xFFFF0000L))
    chk_err("chkparam: Poly parameter is too wide.", outfile);
 if ((TB_REVER != false) && (TB_REVER != true))
    chk_err("chkparam: Reverse parameter is not boolean.", outfile);
}
*/

//******************************************************************************
//* Old main() from reference code. See new main() in crcmodel.c               *
//* Note to students:                                                          *
//*  -- In REAL code, main() is ALWAYS at the top of the module.               *
//*     This forces us to declare prototypes for all functions. This isn't     *
//*     UNIX, and it isn't 1975.                                               *
//******************************************************************************
/*
int main ()
{
 printf("\n");
 printf("Rocksoft^tm Model CRC Algorithm Table Generation Program V1.0\n");
 printf("-------------------------------------------------------------\n");
 printf("Output file is \"%s\".\n",TB_FILE);
 outfile = fopen(TB_FILE,"w"); chk_err("");
 chkparam();
 gentable();
 if (fclose(outfile) != 0)
    chk_err("main: Couldn't close output file.");
 printf("\nSUCCESS: The table has been successfully written.\n");
 exit ( 0 ) ;
}
*/

