//******************************************************************************
//* File       : cTrashFile.hpp                                                *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2015-2020 Mahlon R. Smith, The Software Samurai *
//*                 GNU GPL copyright notice below                             *
//* Date       : 17-Jan-2020                                                   *
//* Version    : (see AppVersion string)                                       *
//*                                                                            *
//* Description: This module contains methods for accessing files to be        *
//* processed, extracting information about each item and creating the         *
//* modified output.                                                           *
//*                                                                            *
//* These methods are adapted from a small subset of the FMgr class written    *
//* for the FileMangler file-management project by the same author.            *
//*                                                                            *
//*                                                                            *
//* Copyright Notice:                                                          *
//* =================                                                          *
//* This program is free software: you can redistribute it and/or modify it    *
//* under the terms of the GNU General Public License as published by the Free *
//* Software Foundation, either version 3 of the License, or (at your option)  *
//* any later version.                                                         *
//*                                                                            *
//* This program is distributed in the hope that it will be useful, but        *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
//* for more details.                                                          *
//*                                                                            *
//* You should have received a copy of the GNU General Public License along    *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.            *
//*                                                                            *
//*         Full text of the GPL License may be found in the Texinfo           *
//*         documentation for this program under 'Copyright Notice'.           *
//******************************************************************************
//* Version History (see Trsh.cpp)                                             *
//*                                                                            *
//******************************************************************************

//****************
//* Header Files *
//****************
#include "cTrash.hpp"      //* General definitions

//*********************
//* Local Definitions *
//*********************


//****************
//* Local Data   *
//****************


//*************************
//*       ctfGetCWD       *
//*************************
//******************************************************************************
//* Returns the path of user's current working directory.                      *
//*                                                                            *
//* Input  : dPath:(by reference, initial value ignored)                       *
//*                receives path string                                        *
//*                                                                            *
//* Returns: OK if successful, ERR if path too long to fit the buffer          *
//******************************************************************************

short cTrash::ctfGetCWD ( gString& dPath )
{
char     buff[MAX_PATH] ;
short    status = OK ;

   if ( (getcwd ( buff, MAX_PATH )) != NULL )
      dPath = buff ;
   else
      status = ERR ;    // (this is very unlikely)

   return status ;

}  //* End ctfGetCWD() *

//*************************
//*    ctfTargetExists    *
//*************************
//******************************************************************************
//* 'stat' the specified path/filename to see if it exists.                    *
//* Target must be of file type: 'regular', 'directory', 'symbolic link' or    *
//* 'FIFO'. For other file types report as does-not-exist.                     *
//*                                                                            *
//* For non-directory files, test for read/write access.                       *
//* For directory files, test for read/write/exec access.                      *
//*                                                                            *
//*                                                                            *
//* Input  : fPath: target path/filename                                       *
//*          isDir: (optional, false by default)                               *
//*                 if 'true',  test for:                                      *
//*                             a) existence                                   *
//*                             b) whether target is of type 'directory'       *
//*                             c) whether user has 'r/w/x' permission         *
//*                 if 'false', test for:                                      *
//*                             a) existence                                   *
//*                             b) whether user has 'r/w' permission           *
//*          phys : (optional, false by default)                               *
//*                 if 'true',  test only for physical existence of file       *
//*                 if 'false', parameter ignored                              *
//*          fsPtr: (optional, NULL pointer by default)                        *
//*                 if specified, target will receive file size in bytes       *
//*          ftPtr: (optional, NULL pointer by default)                        *
//*                 if specified, target will receive file type                *
//*                                                                            *
//* Returns: 'true'  if target file passes indicated tests                     *
//*          'false' otherwise                                                 *
//******************************************************************************
//* Programmer's Note: Because we report an existing file of unsupported type  *
//* as not existing, this is not a genera-purpose method; so be careful how    *
//* it is used. Specifically, do not use it during restore operations to test  *
//* for existing targets.                                                      *
//******************************************************************************

bool cTrash::ctfTargetExists ( const gString& fPath, bool isDir, bool phys, 
                               UINT64* fsPtr, fmFType* ftPtr )
{
   bool exists = false ;
   FileStats rawStats ;

   //* If the file exists *
   if ( (lstat64 ( fPath.ustr(), &rawStats )) == OK )
   {
      //* Target physically exists. *
      if ( fsPtr != NULL )
         *fsPtr = rawStats.st_size ;
      //* Decode the filetype.      *
      fmFType ft = this->ctfDecodeFileType ( rawStats.st_mode ) ;
      if ( ftPtr != NULL )
         *ftPtr = ft ;

      //* If testing whether target is a directory *
      if ( (isDir != false) && (ft == fmDIR_TYPE) )
         exists = true ;

      //* We operate only on certain file types (see note above) *
      else
         exists = this->cttTestFiletype ( ft ) ;

      //* If target physically exists, test for read/write access *
      if ( phys == false && exists != false )
      {
         bool readAcc  = true,
              writeAcc = true,
              exeAcc   = true ;
         //* Directories must have r/w/x access *
         if ( ft == fmDIR_TYPE )
            exeAcc = bool((access ( fPath.ustr(), X_OK )) == ZERO) ;

         //* Other file types need only r/w access.        *
         //* Note: Symbolic links always have r/w/x access *
         //*       so we shouldn't (and can't) test them.  *
         if ( ft != fmLINK_TYPE )
         {
            readAcc = bool((access ( fPath.ustr(), R_OK )) == ZERO) ;
            writeAcc = bool((access ( fPath.ustr(), W_OK )) == ZERO) ;
         }

         if ( !readAcc || !writeAcc || !exeAcc )
            exists = false ;
      }
   }
   return exists ;

}  //* End ctfTargetExists() *

//*************************
//*    ctfGetFileStats    *
//*************************
//******************************************************************************
//* Stat (lstat) the target file, and return the formatted information.        *
//*                                                                            *
//* Input  : trgPath : (by reference) full path/filename specification         *
//*          dfs     : (by reference) receives decoded stat information        *
//*                                                                            *
//* Returns: 'true'  if successful                                             *
//*          'false' if system call failed                                     *
//******************************************************************************

bool cTrash::ctfGetFileStats ( const gString& trgPath, DFileStats& dfs )
{
   bool status ;

   if ( (status = bool((lstat64 ( trgPath.ustr(), &dfs.rawStats )) == OK)) != false )
   {
      dfs.fType  = this->ctfDecodeFileType ( dfs.rawStats.st_mode ) ;
      dfs.fBytes = dfs.rawStats.st_size ;
      this->ctfDecodeTimestamp ( dfs.modTime, dfs.rawStats.st_mtime ) ;
      if ( dfs.fType != fmLINK_TYPE )
      {
         dfs.rAccess = bool((access ( trgPath.ustr(), R_OK )) == ZERO) ;
         dfs.wAccess = bool((access ( trgPath.ustr(), W_OK )) == ZERO) ;
         dfs.xAccess = bool((access ( trgPath.ustr(), X_OK )) == ZERO) ;
      }
      else     // symbolic link permissions are bogus (and always 'true')
         dfs.rAccess = dfs.wAccess = dfs.xAccess = true ;
   }
   return status ;

}  //* End ctfGetFileStats() *

//*************************
//*   ctfDecodeFileType   *
//*************************
//******************************************************************************
//* Extract the file type from the "st_mode" element of the stat{} structure   *
//* and return the equivalent member of enum fmFType.                          *
//*                                                                            *
//* PRIVATE METHOD.                                                            *
//*                                                                            *
//* The POSIX standard defines certain macros to access the file "st_mode".    *
//* There are several layers to the macros, but for Linux EXT2, EXT3 we        *
//* finally get down to the actual bits in /usr/include/bits/stat.h:           *
//* See also page 135 of "Linux Programming by Example."                       *
//* The actual macros that use these bitmasks are used in this routine.        *
//*                                                                            *
//* Encoding of the file mode.                                                 *
//* #define	__S_IFMT	0170000	/* These bits determine file type.               *
//* File types.                                                                *
//* #define	__S_IFDIR	0040000	 Directory.                                   *
//* #define	__S_IFCHR	0020000	 Character device.                            *
//* #define	__S_IFBLK	0060000	 Block device.                                *
//* #define	__S_IFREG	0100000	 Regular file.                                *
//* #define	__S_IFIFO	0010000	 FIFO.                                        *
//* #define	__S_IFLNK	0120000	 Symbolic link.                               *
//* #define	__S_IFSOCK	0140000	 Socket.                                      *
//*                                                                            *
//* Input Values : mode: st_mode field of file's stat{} strucutre              *
//*                                                                            *
//* Return Value : member of enum fmFType                                      *
//******************************************************************************

fmFType cTrash::ctfDecodeFileType ( UINT mode )
{
   fmFType ftype ;

   if ( (S_ISREG(mode)) != ZERO )         ftype = fmREG_TYPE ;
   else if ( (S_ISDIR(mode)) != ZERO )    ftype = fmDIR_TYPE ;
   else if ( (S_ISLNK(mode)) != ZERO )    ftype = fmLINK_TYPE ;
   else if ( (S_ISCHR(mode)) != ZERO )    ftype = fmCHDEV_TYPE ;
   else if ( (S_ISBLK(mode)) != ZERO )    ftype = fmBKDEV_TYPE ;
   else if ( (S_ISFIFO(mode)) != ZERO )   ftype = fmFIFO_TYPE ;
   else if ( (S_ISSOCK(mode)) != ZERO )   ftype = fmSOCK_TYPE ;
   else ftype = fmUNKNOWN_TYPE ;
   
   return ftype ;
   
}  //* End ctfDecodeFileType() *

//*************************
//*  ctfDecodeTimestamp   *
//*************************
//******************************************************************************
//* Convert an 'epoch' (time_t) time code to its date/time components.      *
//*                                                                            *
//* Input  : rawPath  : (by reference) caller's path specification             *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note:                                                         *
//* We do not currently convert the daylight-savings flag, the time zone, the  *
//* time-zone offset or Julian date returned by the localtime_r function.      *
//* Since this application doesn't use these fields, it seems a waste of       *
//* CPU cycles.                                                                *
//*  const char *tm_zone: ("CST" for China) Note that these abbreviations are  *
//*                       not unique, and are therefore rather useless.        *
//*                       Instead, use something like "UTC+hh:mm"              *
//*                       For a complete list of the time-zone abbreviations:  *
//*                             http://www.timeanddate.com/time/zones/         *
//*  long int tm_gmtoff : seconds East of UTC                                  *
//*                       Example: China is UTC+8 (and never observes DST),    *
//*                       so: +8 * 60 * 60 == 28,800 seconds                   *
//*  long int tm_isdst  : > ZERO == yes on DST                                 *
//*                         ZERO == no on DST                                  *
//*                       < ZERO == DST info not available                     *
//******************************************************************************

void cTrash::ctfDecodeTimestamp ( localTime& lTime, time_t epochTime )
{
   lTime.reset() ;            // initialize caller's data
   if ( epochTime >= ZERO )
   {  //* Width of time_t is system dependent *
      if ( (sizeof(epochTime)) == (sizeof(UINT64)) )
         lTime.epoch = lTime.sysepoch = epochTime ;
      else
      {
         lTime.sysepoch = epochTime ;
         lTime.epoch    = (UINT64)epochTime ;
      }
      //* Receives the broken-down time.              *
      Tm bdt ;       // receives broken-down time
      if ( (localtime_r ( &epochTime, &bdt )) != NULL )
      {
         //* Translate to localTime format *
         lTime.date     = bdt.tm_mday ;          // today's date
         lTime.month    = bdt.tm_mon + 1 ;       // month
         lTime.year     = bdt.tm_year + 1900 ;   // year
         lTime.hours    = bdt.tm_hour ;          // hour
         lTime.minutes  = bdt.tm_min ;           // minutes
         lTime.seconds  = bdt.tm_sec ;           // seconds
         lTime.day      = bdt.tm_wday ;          // day-of-week (0 == Sunday)

         #if 0    // Decode time zone data (not currently used)
         gString gs( bdt.tm_zone ) ;
         gs.copy( lTime.timezone, ltFMT_LEN ) ; // string value (not very useful)
         lTime.gmtoffset = bdt.tm_gmtoff ;      // offset as number of seconds
         lTime.dst = (bdt.tm_isdst > ZERO ) ? true : false ; // DST in effect
         int absOffset = abs(lTime.gmtoffset) ;
         int hOffset = absOffset / (60*60),
             mOffset = absOffset % (60*60) ;
         if ( lTime.gmtoffset < ZERO )
            hOffset = -(hOffset) ;
         gs.compose( L"UTC%+d:%02d", &hOffset, &mOffset ) ;
         gs.copy( lTime.utc_zone, ltFMT_LEN ) ;
         #endif   // Decode time zond data
      }
   }

}  //* End ctfDecodeTimestamp() *

//*************************
//*  ctfVerifyFileStats   *
//*************************
//******************************************************************************
//* Verify that the file's attributes meet the criteria.                       *
//* a) file type  : Must be one of the file types supported by the application.*
//* b) user access: User must have read/write access to the file, and if       *
//*                 the file is a directory, read/write/execute access.        *
//*                                                                            *
//* Input  : dfs  : (by reference) contains file stats                         *
//*                                                                            *
//* Returns: 'true'  if file meets all criteria                                *
//*          'false' if the file cannot be processed                           *
//******************************************************************************

bool cTrash::ctfVerifyFileStats ( const DFileStats& dfs )
{
   bool status = false ;
   if ( dfs.rAccess && dfs.wAccess )
   {
      if ( dfs.fType == fmDIR_TYPE )
      {
         if ( dfs.xAccess != false )
            status = true ;
      }
      else
         status = this->cttTestFiletype ( dfs.fType ) ;
   }
   return status ;

}  //* End ctfVerifyFileStats()

//*************************
//*     ctfRealpath       *
//*************************
//******************************************************************************
//* Decode the relative or aliased path for the specified target.              *
//*                                                                            *
//*                                                                            *
//* Input  : rawPath  : (by reference) caller's path specification             *
//*          realPath : (by reference) receives decoded path specification     *
//*                                                                            *
//* Returns: 'true'  if conversion successful                                  *
//*          'false' if invalid path or if system call fails                   *
//******************************************************************************
//* Programmer's Notes:                                                        *
//* a) The 'realpath' C library function cannot handle a symbolic substitution *
//*    such as: ${HOME}/Docs/filename.txt                                      *
//*    It returns: NULL, and if the optional buffer is specified, then it      *
//*    receives something like: /home/Sam/Docs/(CWD)/${HOME}                   *
//*    This is a little bit nuts, but (sort-of) matches 'realpath'             *
//*    documentation. For this reason, we use the 'wordexp' library function   *
//*    to expand any symbolic substitutions.                                   *
//*                                                                            *
//* b) Note that the 'realpath' C library function always follows symbolic     *
//*    links, which we DO NOT WANT in this application. We want to follow      *
//*    symlinks EXCEPT for the terminal filename.                              *
//*                                                                            *
//* c) In order to get the full path to a symlink file, we use the coreutils   *
//*    'realpath' utility which can be told NOT to follow simlinks; however,   *
//*    it is very slow, AND symlink directory names on the path will not be    *
//*    resolved.                                                               *
//*                                                                            *
//*    Therefore, we 'lstat' to test for an existing symlink file.             *
//*     i) If it is a symlink file, we use the coreutils 'realpath'.           *
//*        '-s' option: don't follow symlinks                                  *
//*        '-m' option: operate only on filename                               *
//*        Example:                                                            *
//*            realpath -sm altTarg/x.lnk                                      *
//*        Yields:                                                             *
//*            /home/sam/Software/cTrash/altTarg/x.lnk                         *
//*    ii) If existing non-symlink OR if non-existing file, we use the         *
//*        C library 'realpath' function instead.                              *
//* Yes, we realize that the solution is clunky, and that we are doing a       *
//* redundant lstat, and if we can think of a better way, we will implement it.*
//* However, this solution works for all but the strangest of path/filename    *
//* specifications, so it stands for now. (05 Oct, 2015)                       *
//*                                                                            *
//*                                                                            *
//* Note also that for systems which do not place restrictions on the length   *
//* of a path string, realpath MUST be called with NULL as the second          *
//* parameter. We're not sure that any such systems still exist, but be safe.  *
//******************************************************************************

bool cTrash::ctfRealpath ( const gString& rawPath, gString& realPath )
{
   bool status = false ;

   gString rp = rawPath ;
   this->ctfEnvExpansion ( rp ) ;

   //* If the expanded path refers to a symbolic link, get the full path *
   //* of the link itself, NOT the link target. (see note above).        *
   FileStats   rawStats ;
   if ( ((lstat64 ( rp.ustr(), &rawStats )) == OK)
        && ((S_ISLNK(rawStats.st_mode)) != false) )
   {  //* Target is a symbolic link.                        *
      //* Get a new tempfile name, and direct the coreutils *
      //* 'realpath' output to it. Then read the file.      *
      status = true ;      // link file exists
      realPath = rp ;      // in case the expansion sequence fails

      gString stdoutCapture ;
      if ( (this->ctfCreateTempname ( stdoutCapture )) != false )
      {
         char cmdBuff[MAX_PATH*3] ;
         snprintf ( cmdBuff, (MAX_PATH*3), "realpath -sm %s 1>%s 2>/dev/null", 
                    rp.ustr(), stdoutCapture.ustr() );
         if ( (system ( cmdBuff )) == OK )
         {
            //* Read the only line in the file *
            ifstream ifsIn ( stdoutCapture.ustr(), ifstream::in ) ;
            ifsIn.getline ( cmdBuff, (MAX_PATH*3) ) ;
            ifsIn.close () ;     // close the temp file
            //* If we have something other than an empty string *
            rp = cmdBuff ;
            if ( rp.gschars() > 1 )
               realPath = rp ;
         }
      }
   }

   //* If file is not a symlink (or if symlink test failed) *
   if ( status == false )
   {
      const char* rpath ;
      if ( (rpath = realpath ( rp.ustr(), NULL )) != NULL )
      {
         realPath = rpath ;
         free ( (void*)rpath ) ;    // release the temp storage
         status = true ;
      }
   }

   return status ;

}  //* End ctfRealpath() *

//*************************
//* ctfValidateTrashPath  *
//*************************
//******************************************************************************
//* Verify:                                                                    *
//* 1) that the specified base directory for the trashcan exists and that      *
//*    user has read/write access to it.                                       *
//* 2) that the two necessary subdirectories 'files' and 'info' exist within   *
//*    the base directory and that user has read/write access to both.         *
//*    If the necessary subdirectories do not exist, try to create them.       *
//*      - We do not attempt to create a missing base trashcan directory, NOR  *
//*        do we attempt to modify the permissions on an existing directory,   *
//*        so either the default path must exist, OR the user needs to specify *
//*        a working alternate path.                                           *
//*                                                                            *
//* Input  : rawPath : (by reference) base path for Trashcan directory         *
//*          realPath: (by reference) receives true Trashcan path              *
//*                                                                            *
//* Returns: 'true' if target directory AND its necessary subdirectories       *
//*                 exist AND if user has read/write access to them            *
//*                 realPath receives the true base path                       *
//*          'false' otherwise (realPath undefined)                            *
//******************************************************************************

bool cTrash::ctfValidateTrashPath ( const gString& rawPath, gString& realPath )
{
   bool status = false ;            // return value

   //* Expand any environment variables and get the real path spec.*
   gString rp = rawPath ;
   this->ctfEnvExpansion ( rp ) ;
   if ( (this->ctfRealpath ( rp, realPath )) != false )
   {
      if ( (this->ctfTargetExists ( realPath, true )) != false )
      {  //* Base directory is accessible. Now test for subdirectories  *
         //* under base directory, and if they are missing, create them.*
         rp.compose( "%S/%S", realPath.gstr(), tcfileDir ) ;
         if ( (this->ctfTargetExists ( rp, true )) != false )
         {
            status = true ;
         }
         else
         {  //* 'files' subdirectory either doesn't exist OR     *
            //* it has insufficient user permissions. If it does *
            //* not exist, create it. Otherwise return failure.  *
            if ( (this->ctfTargetExists ( rp, false, true )) == false )
            {
               if ( ((mkdir ( rp.ustr(), dirMode )) == ZERO) && 
                    ((this->ctfTargetExists ( rp, true )) != false) )
               {
                  status = true ;
               }
            }
         }
         if ( status != false )
         {
            status = false ;        // reset the status flag

            rp.compose( "%S/%S", realPath.gstr(), tcinfoDir ) ;
            if ( (this->ctfTargetExists ( rp, true )) != false )
            {
               status = true ;
            }
            else
            {  //* 'info' subdirectory either doesn't exist OR      *
               //* it has insufficient user permissions. If it does *
               //* not exist, create it. Otherwise return failure.  *
               if ( (this->ctfTargetExists ( rp, false, true )) == false )
               {
                  if ( ((mkdir ( rp.ustr(), dirMode )) == ZERO) && 
                       ((this->ctfTargetExists ( rp, true )) != false) )
                  {
                     status = true ;
                  }
               }
            }
         }
      }
   }
   return status ;

}  //* End ctfValidateTrashPath() *

//*************************
//*   ctfEnvExpansion     *
//*************************
//******************************************************************************
//* Perform environment-variable expansion ${XYZ} or tilde ('~') expansion     *
//* on the specified string.                                                   *
//*                                                                            *
//* Input  : gsPath : (by reference) contains the string to be scanned         *
//*                   and receives the expanded path                           *
//*                                                                            *
//* Returns: 'true'  if expansion successful (or no expansion needed)          *
//*          'false' if invalid or mis-used character(s) in the string         *
//******************************************************************************
//* Programmer's Notes:                                                        *
//* The 'wordexp' function is a pretty cool, but watch out:                    *
//*  a) wordexp returns ZERO on success or WRDE_BADCHAR (2) if an invalid      *
//*     character is detected in the stream.                                   *
//*     - Note that an empty string will pass the scan, but then               *
//*       'wexp.we_wordc' will be ZERO.                                        *
//*  b) Dynamic memory allocation happens, so remember to free it.             *
//*     - If a bad character in the stream, then freeing the dynamic           *
//*       allocation will cause a segmentation fault. This is a Standard       *
//*       Library bug, so the work-around is to call 'wordfree' ONLY if        *
//*      'wordexp' returns success.                                            *
//*  c) SPACE characters delimit the parsing, so if the path contains spaces,  *
//*     then we must concatenate the resulting substrings, reinserting the     *
//*     space characters. Leading and trailing spaces are ignored.             *
//*     (We assume that a  path will never contain a TAB character.)           *
//*  d) wordexp will choke on the following characters in the stream:          *
//*             & | ; < >  \n     (unless they are quoted)                     *
//*     - Parentheses and braces should appear ONLY as part of a token to be   *
//*       expanded (or if they are quoted).                                    *
//*  e) The tokens we are most likely to see are '${HOME}' and '~'.            *
//*     These are both expanded as '/home/sam' or the equivalent.              *
//*                                                                            *
//******************************************************************************

bool cTrash::ctfEnvExpansion ( gString& gsPath )
{
   bool status = false ;         // return value

   wordexp_t wexp ;              // target structure
   if ( (wordexp ( gsPath.ustr(), &wexp, ZERO )) == ZERO )
   {
      if ( wexp.we_wordc > ZERO )   // if we have at least one element
      {
         gsPath.clear() ;
         for ( UINT i = ZERO ; i < wexp.we_wordc ; )
         {
            gsPath.append( wexp.we_wordv[i++] ) ;
            if ( i < wexp.we_wordc )
               gsPath.append( L' ' ) ;
         }
      }
      wordfree ( &wexp ) ;
      status = true ;
   }
   return status ;

}  //* End ctfEnvExpansion() *

//*************************
//*   ctfCreateTemppath   *
//*************************
//******************************************************************************
//* Create a unique path/filename prefix for creating the application's        *
//* temporary files. Resulting string is stored in the 'tmpDir' data member.   *
//*                                                                            *
//* This method calls the 'tmpnam_r' function to create a unique name, then    *
//* we extract the path and call 'mkdtemp' to create a unique directory name.  *
//*                                                                            *
//* Finally, we create the path/filename for the file which will contain the   *
//* list of files to be processed and store it in the 'listFile' data member.  *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true'  if path/filename prefix created successfully              *
//*          'false' if library call failed                                    *
//******************************************************************************
//* Programmer's Note:                                                         *
//* The update to g++ v:5.4.0 caused a new warning that two library functions, *
//* tempnam() and tmpnam_r() are "dangerous" because a black-hat might be      *
//* able to use the milliseconds between creation of the name and creation of  *
//* the file to subvert our design.                                            *
//* -- While this is technically true, it is rather unlikely.                  *
//* -- We have eliminated the call to tmpnam_r() in this method which was used *
//*    to locate the system's temp directory. We have instead implemented the  *
//*    ctfGetTempdirPath() method which does (approximately) what we believe   *
//*    the temp_directory_path() library function is designed to do. See       *
//*    notes in that method for details.                                       *
//* -- Our use of the tempnam() library function is more difficult to supplant.*
//*    The compiler warning recommends _STRONGLY_ that mkstemp() be used       *
//*    instead of tempnam(). Okay, but mkstemp() returns the descriptor for a  *
//*    file which has been created and opened in binary mode, which is a       *
//*    complete waste of our time.                                             *
//*    -- Hey!:                                                                *
//*       1) This is C++, and                                                  *
//*       2) This is the 21st century.                                         *
//*       We aren't going to do low-level file I/O.                            *
//******************************************************************************

bool cTrash::ctfCreateTemppath ( void )
{
   //* Name of temporary file containing names of files to process *
   const char* const listFile = "CTRASH_list" ;

   char tn[gsMAXBYTES] ;               // receives the particularized filespec
   bool  status = false ;              // return value

   //* Get path of temp directory*
   gString basePath ;
   if ( (this->ctfGetTempdirPath ( basePath )) != false )
   {
      basePath.append( "/CTRASH_XXXXXX" ) ;
      basePath.copy( tn, gsMAXBYTES ) ;
      if ( (mkdtemp ( tn )) != NULL )
      {
         this->tmpDir = tn ;
         this->ctfCatPathFilename ( this->listFile, this->tmpDir, listFile ) ;
         status = true ;
      }
      else
         basePath.clear() ;
   }
   return status ;

}  //* End ctfCreateTemppath() *

//*************************
//*   ctfCreateTempname   *
//*************************
//******************************************************************************
//* Create a unique path/filename for a temporary file.                        *
//*                                                                            *
//* Input  : tmpPath: (by reference) receives the path/filename                *
//*                                                                            *
//* Returns: 'true'  if path/filename created successfully                     *
//*          'false' if library call failed                                    *
//******************************************************************************
//* Programmer's Note:                                                         *
//* The application's temporary files live in a uniquely-named subdirectory    *
//* within the system's temporary-file directory. The path to this directory   *
//* is located in our 'tmpDir' data member.                                    *
//*                                                                            *
//* The unique filename is concatenated with 'tmpDir' and returned to caller.  *
//*                                                                            *
//* NOTE: We have replaced the call to tempnam() with a call to mkstemp().     *
//*       For the tempnam() function, because the target file is not created   *
//*       immediately, there is a VERY SMALL chance (literally 1 in a million) *
//*       that another process will receive an identical filename, either      *
//*       accidentally or through black-hat malice.                            *
//*       This is overly-abundant caution on our part, but it silences the     *
//*       compiler warning about tempnam() being "dangerous".                  *
//*       See notes in ctfCreateTemppath(), above.                             *
//******************************************************************************

bool cTrash::ctfCreateTempname ( gString& tmpPath )
{
   bool  status = false ;

   char tn[gsMAXBYTES] ;
   gString gstmp( "%S/CT_XXXXXX", this->tmpDir.gstr() ) ;
   gstmp.copy( tn, gsMAXBYTES ) ;
   int descriptor ;
   if ( (descriptor = mkstemp ( tn )) != (-1) )
   {
      close ( descriptor ) ;     // close the file
      tmpPath = tn ;             // copy target path to caller's buffer
      status = true ;            // declare success
   }
   else
      tmpPath.clear() ;
   return status ;

}  //* End ctfCreateTempname() *

//*************************
//*   ctfGetTempdirPath   *
//*************************
//******************************************************************************
//* This is a stub function which provides access to the C++17 function:       *
//*                       temp_directory_path()                                *
//*                                                                            *
//* Because most programmer's will not reach C++17 for at least several        *
//* months, we isolate the call here and implement what we expect is the way   *
//* the library function will be implemented.                                  *
//*         (This WILL NOT work under Windoze--and we don't care.)             *
//*         (Screw Windoze, and the horse it rode in on.         )             *
//* -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- *
//* From <http://en.cppreference.com>                                          *
//* ---------------------------------                                          *
//* These functions are available in C++17 (released Jan. 2018):               *
//* const char* fs::temp_directory_path ( void ) ;                             *
//* const char* fs::temp_directory_path ( std::error_code& ec ) ;              *
//*                                                                            *
//* Returns A directory suitable for temporary files. The path is guaranteed   *
//* to exist and to be a directory. The overload that takes error_code&        *
//* argument returns an empty path on error.                                   *
//*                                                                            *
//* Exceptions: The overload that does not take a std::error_code& parameter   *
//* throws filesystem_error on underlying OS API errors, constructed with path *
//* to be returned as the first argument and the OS error code as the error    *
//* code argument. std::bad_alloc may be thrown if memory allocation fails.    *
//* The overload taking a std::error_code& parameter sets it to the OS API     *
//* error code if an OS API call fails, and executes ec.clear() if no errors   *
//* occur.                                                                     *
//*                                                                            *
//* On POSIX systems, the path may be the one specified in the environment     *
//* variables TMPDIR, TMP, TEMP, TEMPDIR, and, if none of them are specified,  *
//* the path "/tmp" is returned.                                               *
//*                                                                            *
//* Input  : tdPath : (by reference) receives the path string                  *
//*                                                                            *
//* Returns: 'true'  if successful, 'false' if system error                    *
//******************************************************************************

bool cTrash::ctfGetTempdirPath ( gString& tdPath )
{
   //* Default path of temp directory on GNU/Linux systems. *
   //* (Used only if environment variable not set.)         *
   const char* const dfltPath = "/tmp" ;

   const char* envPath ;            // returned by getenv()
   DFileStats  fStats ;             // target filestats
   bool        status = false ;     // return value

   if ( (envPath = std::getenv ( "TMPDIR" )) == NULL )
      if ( (envPath = std::getenv ( "TMP" )) == NULL )
         if ( (envPath = std::getenv ( "TEMP" )) == NULL )
            if ( (envPath = std::getenv ( "TEMPDIR" )) == NULL )
               envPath = dfltPath ;
   tdPath = envPath ;
   if ( (this->ctfGetFileStats ( tdPath, fStats )) != false )
   {
      if ( (fStats.fType == fmDIR_TYPE) && 
           ((access ( tdPath.ustr(), R_OK )) == ZERO) && 
           ((access ( tdPath.ustr(), W_OK )) == ZERO) )
         status = true ;
   }
   return status ;

}  //* End ctfGetTempdirPath()

//************************
//*    GetLocalTime      *
//************************
//******************************************************************************
//* Get the system timecode and convert it to localTime format.                *
//*                                                                            *
//* Input  : lt  : (by reference) receives the decoded local date/time         *
//*                                                                            *
//* Returns: true if successful, false if system call fails                    *
//******************************************************************************
//* Note that we do a bit of defensive programming in anticipation of the      *
//* dreaded year 2038 overflow of the 32-bit time_t type.                      *
//******************************************************************************

bool cTrash::ctfGetLocalTime ( localTime& lt )
{
   bool success = false ;                       // return value
  
   lt.reset() ;                                 // set to default values
   if ( (time ( &lt.sysepoch )) != (-1) )
   {
      if ( lt.sysepoch >= ZERO )                // reported time is AFTER the epoch
      {
         lt.epoch = (int64_t)lt.sysepoch ;      // promote to 64-bit
         //* Decode the system time *
         Tm    tm ;                             // Linux time structure
         if ( (localtime_r ( &lt.sysepoch, &tm )) != NULL )
         {
            //* Translate to localTime format *
            lt.day      = ZERO ;                // day-of-week is not initialized
            lt.date     = tm.tm_mday ;          // today's date
            lt.month    = tm.tm_mon + 1 ;       // month
            lt.year     = tm.tm_year + 1900 ;   // year
            lt.hours    = tm.tm_hour ;          // hour
            lt.minutes  = tm.tm_min ;           // minutes
            lt.seconds  = tm.tm_sec ;           // seconds
            success = true ;
         }
      }
      else
      {  /* SYSTEM ERROR - time_t OVERFLOW */ }
   }
   return success ;

}  //* End ctfGetLocalTime() *

//*************************
//*  ctfCatPathFilename   *
//*************************
//******************************************************************************
//* Concatenate a path string with a filename string to create a path/filename *
//* specification.                                                             *
//*                                                                            *
//* Input  : pgs  : gString object (by reference) to hold the path/filename    *
//*                 On return, pgs contains the path/filename string in both   *
//*                 UTF-8 and wchar_t formats.                                 *
//*          wPath: gString object (by reference) path string                  *
//*          uFile: pointer to UTF-8 filename string                           *
//*               OR                                                           *
//*          wFile: pointer to wchar_t filename string                         *
//*                                                                            *
//* Returns: OK if success                                                     *
//*          ERR if string truncated                                           *
//******************************************************************************

bool cTrash::ctfCatPathFilename ( gString& pgs, const gString& wPath, const char* uFile )
{
   bool status = true ;

   pgs.compose( L"%S/%s", wPath.gstr(), uFile ) ;
   if ( pgs.gschars() >= (gsMAXCHARS-1) )
      status = false ;  // (this is very unlikely)
   return status ;

}  //* End ctfCatPathFilename() *

bool cTrash::ctfCatPathFilename ( gString& pgs, const gString& wPath, const wchar_t* wFile )
{
   bool status = true ;

   pgs.compose( L"%S/%S", wPath.gstr(), wFile ) ;
   if ( pgs.gschars() >= (gsMAXCHARS-1) )
      status = false ;  // (this is very unlikely)
   return status ;

}  //* End ctfCatPathFilename() *

//*************************
//*   ctfTrimPathString   *
//*************************
//******************************************************************************
//* Shorten the path string so it will fit into the available display space.   *
//* This is done by replacing one or more directory names with the ellipsis.   *
//*     Example: /home/mongoose/Applications/CoffeeCups   --becomes--          *
//*              /home/mongoose/.../CoffeeCups                                 *
//*                                                                            *
//* Input  : gsPath  : gString object (by reference) containing path string    *
//*          maxWidth: maximum number of columns in final display string       *
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
//******************************************************************************
//* Programmer's Note: Hidden-directory names (names beginning with a PERIOD)  *
//* may exist in the path, but are correctly handled.                          *
//*                                                                            *
//******************************************************************************

bool cTrash::ctfTrimPathString ( gString& gsPath, short maxWidth )
{
   bool status = true ;

   if ( (gsPath.gscols()) > maxWidth )
   {
      gString eString = "/...",
              pString, fString ;
      const wchar_t* wPtr ;
      short   eWidth = eString.gscols(),  // column count for ellipsis string
              nWidth = ZERO,              // column count for filename string
              pWidth = ZERO,              // column count for path in progress
              xWidth = ZERO,              // column count in progress
              bLen = gsPath.find( L'/', 1), // length of base directory string
              pi,                         // path length in progress
              failsafe = 15 ;             // when zero, break out of loop

      this->ctfExtractPathname ( pString, gsPath ) ;
      pWidth = pString.gscols() ;
      this->ctfExtractFilename ( fString, gsPath ) ;
      fString.insert( L'/' ) ;
      nWidth = fString.gscols() ;
      xWidth = pWidth + nWidth ;

      while ( xWidth > maxWidth )
      {
         wPtr = pString.gstr( pi ) ;
         --pi ;         // reference the NULLCHAR
         while ( (wPtr[pi] != L'/') && (pi > bLen) )
            --pi ;
         pString.limitChars( pi ) ;
         pWidth = pString.gscols() ;
   
         xWidth = pWidth + eWidth + nWidth ;
         if ( (pi == bLen) || (--failsafe <= ZERO) )
            break ;
      }

      //* If loop not aborted, create the trimmed filespec.*
      if ( xWidth <= maxWidth )
      {
         gsPath.compose( L"%S%S%S", pString.gstr(), eString.gstr(), fString.gstr() ) ;
      }
      else
         status = false ;
   }
   return status ;

}  //* End of ctfTrimPathString() *

//*************************
//*  ctfFilesystemStats   *
//*************************
//******************************************************************************
//* Read file system stats for filesystem containing specified path/filename.  *
//*                                                                            *
//* Input  : trgPath: full path/filename for any file on the target file system*
//*          fsStats: (by reference) unitialized instance of                   *
//*                   fileSystemStats class: receives stat data                *
//*                                                                            *
//* Returns: 'true'  if successful (all fields of 'fsStats' will be initialized*
//*          'false' if unable to create temp file or if system call fails     *
//******************************************************************************
//* Valid format sequences for file systems:                                   *
//*  %a   Free blocks available to non-superuser                               *
//*  %b   Total data blocks in file system                                     *
//*  %c   Total file nodes in file system                                      *
//*  %d   Free file nodes in file system                                       *
//*  %f   Free blocks in file system                                           *
//*  %C   SELinux security context string                                      *
//*  %i   File System ID in hex                                                *
//*  %l   Maximum length of filenames                                          *
//*  %n   File name                                                            *
//*  %s   Block size (for faster transfers)                                    *
//*  %S   Fundamental block size (for block counts)                            *
//*  %t   Type in hex                                                          *
//*  %T   Type in human readable form (not very accurate)                      *
//*                                                                            *
//* See 'info stat' and 'stat --help' for more information.                    *
//* See also 'info df' to get accurate filesystem type in human-readable form. *
//* See also 'info ls --context' to get seLinux context strings if not         *
//*                    available from 'stat' command. This avoids dependency   *
//*                    on the seLinux library which may be unavailable or      *
//*                    restricted on some systems. (see info lgetfilecon)      *
//*                                                                            *
//* Programmer's Note: In the unlikely event that the fsType field or          *
//* seLinuxCode field contains a non-ASCII character, the conversion below     *
//* from string to field values may fail.                                      *
//******************************************************************************

bool cTrash::ctfFilesystemStats ( const gString& trgPath, fileSystemStats& fsStats )
{
static const char fssCmd[] = "stat -f --printf=\"%b %f %a %c %d %s %S %l %i %t %T %C \n\"" ;

   bool status = true ;

   //* Generate a path/filename for the temporary file *
   gString stdoutCapture ;
   if ( (this->ctfCreateTempname ( stdoutCapture )) != false )
   {
      char cmdBuff[MAX_PATH*3] ;
      snprintf ( cmdBuff, (MAX_PATH*3), "%s %s 1>%s 2>/dev/null", 
                 fssCmd, trgPath.ustr(), stdoutCapture.ustr() );
      if ( (system ( cmdBuff )) == OK )
      {
         //* Read the only line in the file *
         ifstream ifsIn ( stdoutCapture.ustr(), ifstream::in ) ;
         ifsIn.getline ( cmdBuff, (MAX_PATH*3) ) ;
         ifsIn.close () ;     // close the temp file
         gString fsData( cmdBuff ) ;
         swscanf ( fsData.gstr(), L"%lu %lu %lu %lu %lu %hu %hu %hu %llX %x %63s %63s", 
            &fsStats.blockTotal, &fsStats.blockFree,  &fsStats.blockAvail, 
            &fsStats.inodeTotal, &fsStats.inodeAvail,
            &fsStats.blockSize,  &fsStats.fblockSize, &fsStats.nameLen,
            &fsStats.systemID,   &fsStats.fsTypeCode,
            fsStats.fsType, fsStats.seLinuxCode ) ;

         fsStats.freeBytes = (UINT64)fsStats.fblockSize * fsStats.blockAvail ;
         fsStats.usedBytes = (UINT64)fsStats.fblockSize * 
                                     (fsStats.blockTotal - fsStats.blockFree) ;

         //* Because the 'stat' command returns an unreliable string describing*
         //* the filesystem, we use the 'df -T' command to get a good string.  *
         snprintf ( cmdBuff, (MAX_PATH*3), "df -T %s 1>%s 2>/dev/null", 
                    trgPath.ustr(), stdoutCapture.ustr() ) ;
         if ( (system ( cmdBuff )) == OK )
         {
            ifsIn.open( stdoutCapture.ustr(), ifstream::in ) ;
            if ( ifsIn.is_open() )
            {  //* Find the column where type string begins *
               ifsIn.getline ( cmdBuff, (MAX_PATH*3) ) ;
               short colIndex = ZERO ;
               while ( cmdBuff[colIndex] != 'T' && cmdBuff[colIndex] != NULLCHAR )
                     colIndex++ ;
               if ( cmdBuff[colIndex] == 'T' )
               {
                  //* Scan the data line *
                  ifsIn.getline ( cmdBuff, (MAX_PATH*3) ) ;
                  fsData = &cmdBuff[colIndex] ;
                  swscanf ( fsData.gstr(), L"%63s", fsStats.fsType ) ;
               }
               ifsIn.close () ;     // close the temp file
            }
         }

         //* If we got nothing from the SELinux Security Context field *
         if ( fsStats.seLinuxCode[0] == '?' || fsStats.seLinuxCode[0] == NULLCHAR )
         {
            const char selCmd[] = "ls -Zd" ;
            snprintf ( cmdBuff, (MAX_PATH*3), "%s %s 1>%s 2>/dev/null", 
                       selCmd, trgPath.ustr(), stdoutCapture.ustr() ) ;
            if ( (system ( cmdBuff )) == OK )
            {
               //* Read the only line in the file *
               ifsIn.open ( stdoutCapture.ustr(), ifstream::in ) ;
               if ( ifsIn.is_open() )
               {
                  ifsIn.getline ( cmdBuff, (MAX_PATH*3) ) ;
                  ifsIn.close () ;     // close the temp file
                  char bitBucket[64] ;
                  fsData = cmdBuff ;
                  swscanf ( fsData.gstr(), L"%63s %63s %63s %63s", 
                            bitBucket, bitBucket, bitBucket, fsStats.seLinuxCode ) ;
               }
            }
         }
      }
      else     // system call unsuccessful, send up a flare
         status = false ;
      remove ( stdoutCapture.ustr() ) ;   // delete the temp file
   }
   else           // unable to create temporary file
      status = false ;
   return status ;

}  //* ctfFilesystemStats() *

//*************************
//*    ctfFileSystemID    *
//*************************
//******************************************************************************
//* Read file system ID (hex)                                                  *
//*                                                                            *
//* Input  : trgPath: full path/filename for any file on the target file system*
//*                                                                            *
//* Returns: filesystem ID if successful                                       *
//*          returns FULL64 (i.e. -1) if stat of filesystem failed             *
//******************************************************************************
//* Programmer's Note: A ZERO '0' value indicates a 'tempfs' filesystem.       *
//* We aren't entirely sure that we want to consider a 'tempfs' filesystem     *
//* as a valid source. For instance, /run/media/sam is a 'temfs' filesystem,   *
//* but the SD card at /run/media/sam/BACKUP32G is 'vfat32' and has a valid    *
//* filesystem ID. This situation needs more thought.                          *
//******************************************************************************

UINT64 cTrash::ctfFileSystemID ( const gString& trgPath )
{
   fileSystemStats fsStats ;
   if ( (this->ctfFilesystemStats ( trgPath, fsStats )) == false )
      fsStats.systemID = FULL64 ;
   return fsStats.systemID ;

}  //* End ctfFileSystemID() *

//*************************
//*  ctfDirectorySummary  *
//*************************
//******************************************************************************
//* Get a summary of the contents of the specified directory.                  *
//* Summary contains: the number of files and subdirectories in the            *
//* directory and the combined size (in bytes) of all files and                *
//* subdirectories in the specified top-level directory (but see 'recurse'     *
//* parameter, below.)                                                         *
//*                                                                            *
//* Input  : dPath : full path specification of directory to be scanned        *
//*          fCount (by reference): initial value ignored                      *
//*          totBytes (by reference): initial value ignored                    *
//*          recurse (optional, 'false' by default)                            *
//*            if 'true' read specified directory and contents of all          *
//*            subdirectories it contains                                      *
//*                                                                            *
//* Returns: 'true' if directory read successfully                             *
//*                 fCount and totBytes are initialized                        *
//*          'false' if directory does not exist or no read access or path     *
//*                 links not resolved. fCount and totBytes == ZERO            *
//******************************************************************************
//* Programmer's Note: This method is primarily for sizing a recursive         *
//* copy/cut/paste operation, but it may have other uses.                      *
//*                                                                            *
//* Note that 'fCount' does not include the target directory itself,           *
//* and that 'totBytes' does not include the size of target directory file.    *
//*                                                                            *
//* Programmer's Note:                                                         *
//* The 'readdir' function is not fully thread safe, but so long as two        *
//* threads are not simultaneously accessing the same stream, there is no      *
//* problem.                                                                   *
//*                                                                            *
//* struct stat64             // supports file size > 2*31 bytes               *
//* {                                                                          *
//*    dev_t      st_dev ;    // device                            (8 bytes)   *
//*    ino64_t    st_ino ;    // inode  (__u_quat_t type)          (8 bytes)   *
//*    mode_t     st_mode ;   // mode i.e. permissions & file type (4 bytes)   *
//*    nlink_t    st_nlink ;  // number of hard links              (4 bytes)   *
//*    uid_t      st_uid ;    // user ID of file's owner           (4 bytes)   *
//*    gid_t      st_gid ;    // group ID of file's owner          (4 bytes)   *
//*    dev_t      st_rdev ;   // device type (if an inode device)  (8 bytes)   *
//*    off64_t    st_size ;   // total size, in bytes (__quad_t)   (8 bytes)   *
//*    blksize_t  st_blksize; // blocksize for file system i/o     (4 bytes)   *
//*    blkcnt64_t st_blocks ; // blocks allocated (__quat_t type)  (8 bytes)   *
//*    time_t     st_atime ;  // time of last access (file opened) (4 bytes)   *
//*    time_t     st_mtime ;  // time of last modification         (4 bytes)   *
//*    time_t     st_ctime ;  // time of last change (in status)   (4 bytes)   *
//*    - - -                                                                   *
//*    __ino_t    __st_ino    // legacy 32-bit inode                           *
//* } ;                                                                        *
//*                                                                            *
//******************************************************************************

bool cTrash::ctfDirectorySummary ( const gString& dPath, UINT& fCount, 
                                   UINT64& totBytes, bool recurse )
{
   DIR*   dirPtr ;
   static USHORT recursionLevel = ZERO ;
   bool   status = true ;

   //* Initialize caller's variables *
   if ( recursionLevel == ZERO )
   {
      fCount   = ZERO ;
      totBytes = ZERO ;
   }
   ++recursionLevel ;

   //* Open the directory *
   if ( (dirPtr = this->ctfOpendir ( dPath )) != NULL )
   {
      gString     fnPath ;
      FileStats   rawStats ;
      deStats*  destat ;
      while ( (destat = readdir64 ( dirPtr )) != NULL )
      {
         //* Do not include 'current dir' and 'parent dir' names.              *
         if ( destat->d_name[ZERO] == PERIOD )
         {
            if (   destat->d_name[1] == NULLCHAR 
                || (destat->d_name[1] == PERIOD && destat->d_name[2] == NULLCHAR) )
               continue ;           // go to next record
         }

         //* Do an 'lstat' on the specified file for file size and type *
         this->ctfCatPathFilename ( fnPath, dPath, destat->d_name ) ;
         if ( (lstat64 ( fnPath.ustr(), &rawStats )) == OK )
         {
            totBytes += (UINT64)rawStats.st_size ; // update byte count
            ++fCount ;                             // and file count
            if ( recurse != false && ((S_ISDIR(rawStats.st_mode)) != false) )
            {
               this->ctfDirectorySummary ( fnPath, fCount, totBytes, true ) ;
            }
         }
         else
         {
            /* lstat failed. DO WE NEED TO DO ANYTHING HERE? */
         }
      }
      closedir ( dirPtr ) ;      // Close the directory
   }
   else
      status = false ;

   --recursionLevel ;
   return status ;

}  //* End ctfDirectorySummary() *

//**************************
//* ctfValidateDirContents *
//**************************
//******************************************************************************
//* Verify the contents of the specified subdirectory.                         *
//*   a) that user has read/write access to all files and subdirectories       *
//*      in specified directory                                                *
//*   b) that the file type is one that we can process.                        *
//*   c) that all contents be on the same filesystem as the base subdirectory. *
//*                                                                            *
//*                                                                            *
//* Input  : fspec : (by reference) specifies the path/filename of the         *
//*                  subdirectory, and receives summary statistics of its      *
//*                  contents                                                  *
//*                                                                            *
//*                                                                            *
//* Returns: number of validated files                                         *
//*            If returned value == fspec.dFiles, then success                 *
//*            Else, one or more files failed validation                       *
//******************************************************************************
//* Programmer's Note:                                                         *
//* This is a recursive scan. If we get an error at a lower level, we can't    *
//* just stop, we need to walk back up the chain. Fortunately, the indication  *
//* of success or failure is a comparison of files tested vs. files validated. *
//*                                                                            *
//* Note that we do filesystem comparison only on directory names. Because we  *
//* do not follow symbolic links, it is very unlikely that a non-directory     *
//* file would be on a different filesystem than the directory which contains  *
//* it. It's not clear whether that is even possible. Of course, if we         *
//* followed a symbolic link to its target, that target could be anywhere.     *
//* So that's one reason why we don't follow symlinks in this application.     *
//*                                                                            *
//* If the LSTAT on a directory entry fails, it means that the directory file  *
//* itself is corrupted. This is unlikely, but if it happens, we abort         *
//* immediately.                                                               *
//*                                                                            *
//******************************************************************************

UINT cTrash::ctfValidateDirContents ( FSpec& fspec )
{
   DIR*  dirPtr ;             // for reading contents of directory file
   UINT  fPass = ZERO ;       // return value
   static USHORT recursionLevel = ZERO ;

   //* Initialize caller's data *
   if ( recursionLevel == ZERO )
   {
      fspec.dFiles = ZERO ;
      fspec.dBytes = ZERO ;
   }
   ++recursionLevel ;

   //* Open the directory *
   if ( (dirPtr = this->ctfOpendir ( fspec.fSpec )) != NULL )
   {
      gString     fPath ;     // target filespec
      DFileStats  fStats ;    // target file decoded stats
      deStats*    destat ;    // directory entry (record)
      bool        showFile ;  // 'true' if file to be included in scan

      while ( (destat = readdir64 ( dirPtr )) != NULL )
      {
         //* Do not include 'current dir' and 'parent dir' names. *
         showFile = true ;
         if ( destat->d_name[ZERO] == PERIOD )
         {
            if (   destat->d_name[1] == NULLCHAR 
                || (destat->d_name[1] == PERIOD && destat->d_name[2] == NULLCHAR) )
               showFile = false ;
         }

         if ( showFile != false )
         {
            this->ctfCatPathFilename ( fPath, fspec.fSpec, destat->d_name ) ;
            ++fspec.dFiles ;        // assume that the file exists

            if ( (this->ctfGetFileStats ( fPath, fStats )) != false )
            {  //* Verify that file is one of our supported types.*
               if ( (this->cttTestFiletype ( fStats.fType )) != false )
               {  //* Directories must have r/w/x access *
                  //* All others must have r/w access.   *
                  if ( fStats.rAccess && fStats.wAccess )
                  {
                     fspec.dBytes += fStats.fBytes ;  // update the accumulator

                     if ( fStats.fType == fmDIR_TYPE )
                     {
                        //* Verify that this directory is on the *
                        //* same filesystem as the parent.       *
                        UINT64 sid = this->ctfFileSystemID ( fPath ) ;
                        if ( fStats.xAccess && sid == fspec.fsysID )
                        {
                           ++fPass ;

                           //* Read subdirectory recursively *
                           FSpec subfspec = { fPath, fStats, sid, ZERO, ZERO } ;
                           fPass += this->ctfValidateDirContents ( subfspec ) ;
                           fspec.dFiles += subfspec.dFiles ;
                           fspec.dBytes += subfspec.dBytes ;
                        }
                        else     // subdirectory not validated
                           break ;
                     }
                     else
                        ++fPass ;
                  }
               }
            }
            else
            {  //* If file not found, then we must abort on the grounds   *
               //* that the directory is incorrect about its own contents.*
               break ;
            }
         }
      }
      closedir ( dirPtr ) ;      // Close the directory
   }

   --recursionLevel ;
   return fPass ;

}  //* End ctfValidateDirContents() *

//*************************
//*      ctfOpendir       *
//*************************
//******************************************************************************
//* Open a directory file for reading.                                         *
//* Before opening the file, verify that we have read access to it.            *
//*                                                                            *
//* NOTE: It is assumed that caller is smart enough to have verified that the  *
//*       path actually points to a directory file. If not, then fail.         *
//*                                                                            *
//* NOTE: If source is a symbolic link to the directory, the read will fail.   *
//*       We report only what is ACTUALLY there, not some distant data.        *
//*                                                                            *
//* Input  : dirPath : full path of directory to be scanned                    *
//*                                                                            *
//* Returns: pointer to open directory file --OR--                             *
//*          returns NULL* if no read access or otherwise unable to open file  *
//******************************************************************************

DIR* cTrash::ctfOpendir ( const gString& dirPath )
{
   DIR* dirPtr = NULL ;

   if ( (access ( dirPath.ustr(), R_OK )) == ZERO )
      dirPtr = opendir ( dirPath.ustr() ) ;

   return dirPtr ;

}  //* End ctfOpendir() *

//*************************
//*  ctfExtractPathname   *
//*************************
//******************************************************************************
//* Extract the path from the path/filename provided.                          *
//* (i.e. truncate the string to remove the filename)                          *
//*                                                                            *
//* Input  : eName : (by reference) receives the extracted path                *
//*          fPath : source path/filename string                               *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void cTrash::ctfExtractPathname ( gString& ePath, const gString& fPath )
{
   ePath = fPath ;
   short slashIndex = ZERO, i ;
   while ( (i = ePath.find( L'/', slashIndex )) >= ZERO )
      slashIndex = i + 1 ;

   ePath.limitChars( slashIndex > ZERO ? --slashIndex : ZERO ) ;

}  //* End of ctfExtractPathname() *

//*************************
//*  ctfExtractFilename   *
//*************************
//******************************************************************************
//* Extract the filename from the path/filename provided.                      *
//*                                                                            *
//* Input  : eName : (by reference) receives the extracted filename            *
//*          fPath : source path/filename string                               *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void cTrash::ctfExtractFilename ( gString& eName, const gString& fPath )
{
   short slashIndex = ZERO, i ;
   while ( (i = fPath.find( L'/', slashIndex )) >= ZERO )
      slashIndex = i + 1 ;

   eName = &fPath.gstr()[slashIndex] ;

}  //* End ctfExtractFilename() *

//***************************
//* ctfExtractFileExtension *
//***************************
//******************************************************************************
//* Extract the filename extension (including the '.') from the path/filename  *
//* provided.                                                                  *
//*                                                                            *
//* Input  : eExt  : (by reference, initial contents ignored)                  *
//*                  receives the extracted filename extension                 *
//*          fPath : source filename or path/filename string                   *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void cTrash::ctfExtractFileExtension ( gString& eExt, const gString& fPath )
{
   eExt.clear() ;                   // initialize the target buffer

   short pIndex ;
   const wchar_t* wptr = fPath.gstr( pIndex ) ;
   --pIndex ;                       // reference the NULLCHAR
   while ( wptr[pIndex] != PERIOD && wptr[pIndex] != SLASH && pIndex > ZERO )
      --pIndex ;

   //* If we found a period character, we have a filename extension.*
   if ( wptr[pIndex] == PERIOD )
      eExt = &wptr[pIndex] ;
   else
      eExt = L"" ;

}  //* End ctfExtractFileExtension() *

//*************************
//*      ctfCopyFile      *
//*************************
//******************************************************************************
//* Copy a file, retaining all timestamps and permissions.                     *
//* Call this method if the file type of the source is unknown.                *
//*                                                                            *
//*                                                                            *
//* Input  : srcPath: (by reference) full path/filename of source file         *
//*          trgPath: (by reference) full path/filename of target file         *
//*                                                                            *
//* Returns: 'true'  if successful                                             *
//*          'false; if copy fails                                             *
//******************************************************************************

bool cTrash::ctfCopyFile ( const gString& srcPath, const gString& trgPath )
{
   DFileStats dfs ;
   bool status = false ;

   //* Stat the source file for its file type.*
   if ( (this->ctfGetFileStats ( srcPath, dfs )) != false )
   {
      //* Copy source to target *
      status = this->ctfCopyFile ( srcPath, trgPath, dfs ) ;
   }
   return status ;

}  //* End ctfCopyFile() *

//*************************
//*      ctfCopyFile      *
//*************************
//******************************************************************************
//* Copy a file, retaining all timestamps and permissions.                     *
//*                                                                            *
//* It is assumed that caller is smart enough to have verified the existence   *
//* of srcPath, and that he/she/it doesn't care about any existing trgPath.    *
//* (After all, the caller is me, and I never make misteaks. :)                *
//*                                                                            *
//* Filetype is determined and the appropriate type of copy is performed.      *
//*  a) regular files                                                          *
//*  b) directory files                                                        *
//*  c) FIFO files                                                             *
//*  d) symbolic links (link is copied, NOT the link target)                   *
//*  e) If srcPath is a Character device, Block device, Socket or unknown      *
//*     type, copy will not be performed.                                      *
//*                                                                            *
//* Input  : srcPath: (by reference) full path/filename of source file         *
//*          trgPath: (by reference) full path/filename of target file         *
//*          dfs    : (by reference) source file stat info                     *
//*                                                                            *
//* Returns: 'true'  if successful                                             *
//*          'false; if copy fails                                             *
//******************************************************************************

bool cTrash::ctfCopyFile ( const gString& srcPath, const gString& trgPath, 
                           const DFileStats& dfs )
{
   bool status = false ;

   switch ( dfs.fType )
   {
      case fmREG_TYPE:
      case fmLINK_TYPE:
         status = this->ctfCopyRegFile ( srcPath, trgPath, false ) ;
         break ;
      case fmDIR_TYPE:
         status = this->ctfCopyDirName ( srcPath, trgPath, dfs ) ;
         break ;
      case fmFIFO_TYPE:
         status = this->ctfCopyFifo ( srcPath, trgPath, dfs ) ;
         break ;
      default:    // unsupported file type
         break ;
   }
   return status ;

}  //* End ctfCopyFile() *

//*************************
//*    ctfCopyRegFile     *
//*************************
//******************************************************************************
//* Create a copy of the specified Regular or Symbolic Link file.              *
//* Preserves all permission bits and date/timestamps.                         *
//* If an existing target, it will be overwritten (if user has write access).  *
//*                                                                            *
//* Call this method only for 'regular' files and symbolic-link files.         *
//* WARNING: Do not call this method to copy a 'special' file. (see below)     *
//*                                                                            *
//* Input  : srcPath : full path/filename specification of source              *
//*          trgPath : full path/filename specification of target              *
//*          followLink: (optional, false by default)                          *
//*                      If source is a symbolic link the following            *
//*                      applies; else the 'followLink' has no effect.         *
//*                      if false, copy the link itself                        *
//*                      if true, copy the file link points to                 *
//*                                                                            *
//* Returns: 'true'  if file copied successfully                               *
//*          'false' if cp command returns an error                            *
//******************************************************************************
//* Programmer's Note: WARNING: Do not call this method to copy a 'special'    *
//*                    file. The cp command opens the pipe and waits forever   *
//*                    for data to be sent through the pipe, OR will refuse to *
//*                    operate on the file at all - both bad results.          *
//*                                                                            *
//* Programmer's Note: If we are copying a symbolic link, all three file       *
//* dates are set to the CURRENT date/timestamp, ignoring our command          *
//* option to retain all file settings. Using the individual parameters        *
//* instead of the '--preserve=all' option doesn't work since 'links' is not   *
//* recognized in the list, nor is 'links' included in 'all' -- hence:         *
//* '--preserve=all -d'.                                                       *
//* This is not a major problem, but it offends my sense of order.             *
//*                               MRS 25 MAR 2007                              *
//*                                                                            *
//* From the 'cp' man page:                                                    *
//* -----------------------                                                    *
//*    When copying from a symbolic link, `cp' normally follows the link       *
//* only when not copying recursively.  This default can be overridden with    *
//* the `--archive' (`-a'), `-d', `--dereference' (`-L'),                      *
//* `--no-dereference' (`-P'), and `-H' options.  If more than one of these    *
//* options is specified, the last one silently overrides the others.          *
//*    When copying to a symbolic link, `cp' follows the link only when it     *
//* refers to an existing regular file.  However, when copying to a            *
//* dangling symbolic link, `cp' refuses by default, and fails with a          *
//* diagnostic, since the operation is inherently dangerous.  This behavior    *
//* is contrary to historical practice and to POSIX.  Set `POSIXLY_CORRECT'    *
//* to make `cp' attempt to create the target of a dangling destination        *
//* symlink, in spite of the possible risk.  Also, when an option like         *
//* `--backup' or `--link' acts to rename or remove the destination before     *
//* copying, `cp' renames or removes the symbolic link rather than the file    *
//* it points to.                                                              *
//*    By default, `cp' copies the contents of special files only when not     *
//* copying recursively.  This default can be overridden with the              *
//* `--copy-contents' option. [BUT THIS WORKS _ONLY_ WHEN RECURSIVE COPY]      *
//*                                                                            *
//    cp -H src trg    - copy link target                                      *
//                       (cp does this automagically unless in recursive mode )*
//*                      (and we use cp only for single files, not recursively)*
//******************************************************************************

bool cTrash::ctfCopyRegFile ( const gString& srcPath, const gString& trgPath, bool followLink )
{
   const char* nfTemplate =      // do not follow symlinks
               "cp --preserve=all -d '%s' '%s' 1>/dev/null 2>/dev/null" ;
   const char* flTemplate =      // follow symlinks
               "cp --preserve=all '%s' '%s' 1>/dev/null 2>/dev/null" ;

   const short bSIZE = MAX_PATH * 3 ;
   char     cmdBuff[bSIZE] ;           // holds command string
   bool status = false ;

   if ( followLink == false )
      snprintf ( cmdBuff, bSIZE, nfTemplate, srcPath.ustr(), trgPath.ustr() );
   else
      snprintf ( cmdBuff, bSIZE, flTemplate, srcPath.ustr(), trgPath.ustr() );

   status = bool( (system ( cmdBuff )) == OK ) ;

   return status ;

}  //* End ctfCopyRegFile() *

//*************************
//*    ctfCopyDirName     *
//*************************
//******************************************************************************
//* Copy a top-level directory file (not its contents).                        *
//*                                                                            *
//* Input  : srcPath : (by reference) full path/filename of source             *
//*                      (Note: This parameter is not currently used.)         *
//*          trgPath : (by reference) full path/filename of target             *
//*          srcStats: (by reference) contains stats on srcPath                *
//*                                                                            *
//* Returns: 'true'  if target created OR if target exists and user has write  *
//*                  permission                                                *
//*          'false' if:                                                       *
//*             a) unable to create target                                     *
//*             b) existig target but user does not have write permission      *
//*             c) existing target is not a directory                          *
//******************************************************************************
//* Notes:                                                                     *
//* - If target directory does not exist, create the directory, preserving     *
//*   the source file's permission bits as well as both 'modified' and         *
//*   'access' timestamps (if possible).                                       *
//* - If target directory already exists, do not overwrite it.                 *
//*   - If existing target IS a directory:                                     *
//*     - Verify that user has write access on the assumption that data will   *
//*       later be written into it. Call may later be surprised to find that   *
//*       the supposedly new directory may already have data in it.            *
//*   - If existing target NOT a directory OR user does not have write access, *
//*     return error.                                                          *
//******************************************************************************

bool cTrash::ctfCopyDirName ( const gString& srcPath, const gString& trgPath, 
                              const DFileStats& srcStats )
{
   bool status = false ;

   DFileStats trgStats ;
   bool trgExists = this->ctfGetFileStats ( trgPath, trgStats ) ;

   //* If target does not exist, create it *
   if ( !trgExists )
   {
      status = this->ctfCreateDirectory ( trgPath, &srcStats ) ;
   }

   //* Target exists. Verify that it is a directory.*
   else if ( trgExists && trgStats.fType == fmDIR_TYPE && trgStats.wAccess )
   {
      status = true ;
   }
   return status ;

}  //* End ctfCopyDirName() *

//*************************
//*      ctfCopyFifo      *
//*************************
//******************************************************************************
//* Copy a FIFO-type file, retaining all timestamps and persmissions.          *
//* If an existing target, it will be overwritten (if user has write access).  *
//*                                                                            *
//* Input  : srcPath : (by reference) full path/filename spec of source        *
//*          trgPath : (by reference) full path/filename spec of target        *
//*          srcStats: (by reference) contains stats on srcPath                *
//*                                                                            *
//* Returns: 'true'  if file copied successfully                               *
//*          'false' if unable to create target                                *
//*                  a) user does not have write access to target directory    *
//*                  b) existing target file is write protected                *
//*                  c) other access violation                                 *
//******************************************************************************
//* Programmer's Note: Creating a FIFO is straightforward because it has no    *
//* special properties.                                                        *
//*                                                                            *
//******************************************************************************

bool cTrash::ctfCopyFifo ( const gString& srcPath, const gString& trgPath, 
                           const DFileStats& srcStats )
{
   bool status = false ;

   //* Create a new FIFO with the same permissions as the source *
   if ( (status = bool((mkfifo ( trgPath.ustr(), srcStats.rawStats.st_mode )) == ZERO)) )
   {
      //* Adjust the mod/access timestamps to match the source *
      // (an error status could be masked here, but it's unlikely)
      status = this->ctfSetTimestamps ( trgPath, srcStats.rawStats.st_mtime, 
                                        srcStats.rawStats.st_atime ) ;
   }
   return status ;

}  //* End ctfCopyFifo()

//*************************
//*    ctfRenameFile      *
//*************************
//******************************************************************************
//* Rename the specified file.                                                 *
//*                                                                            *
//* Input  : srcPath : full path/filename specification of source              *
//*          trgPath : full path/filename specification of target              *
//*                                                                            *
//* Returns: 'true'  if successful, else 'false'                               *
//******************************************************************************
//* -- Instead of a straight rename, we actually do a copy-and-delete because  *
//*    we want to be independent of the filesystem, and the 'rename(); function*
//*    cannot cross filesystem boundaries.                                     *
//* -- Permisisons are preserved.                                              *
//* -- Modification and Access dates are preserved.                            *
//* -- Owner MAY change, it depends on whether the user owns the source.       *
//* -- Stat date is updated to current local time (this is out of our control).*
//* -- An existing target WILL be overwritten.                                 *
//* See the '--preserve=all' and '-d' options for 'cp'.                        *
//*                                                                            *
//*                                                                            *
//******************************************************************************

bool cTrash::ctfRenameFile ( const gString& srcPath, const gString& trgPath )
{
   bool status = false ;
   
   if ( (this->ctfCopyFile ( srcPath, trgPath )) != false )
   {
      status = this->ctfDeleteFile ( srcPath ) ;
   }
   return status ;

}  //* End ctfRenameFile() *

//*************************
//*    ctfDeleteFile      *
//*************************
//******************************************************************************
//* Delete (unlink) the specified file.                                        *
//*                                                                            *
//* Input  : trgPath : full path/filename specification of target              *
//*                                                                            *
//* Returns: 'true' if successful, else 'false'                                *
//******************************************************************************

bool cTrash::ctfDeleteFile ( const gString& trgPath )
{

   return ( bool((unlink ( trgPath.ustr() )) == ZERO) ) ;

}  //* End ctfDeleteFile() *

//*************************
//*  ctfCreateDirectory   *
//*************************
//******************************************************************************
//* Create an empty directory file.                                            *
//*                                                                            *
//*                                                                            *
//* Input  : trgPath  : (by reference) full path/filename spec for target      *
//*          dfsPtr   : (optional, NULL pointer by default)                    *
//*                     If specified, contains the permission bits and         *
//*                     timestamps to be applied.                              *
//*                                                                            *
//* Returns: 'true' if successful, 'false' if unable to create target          *
//******************************************************************************
//* Programmer's Note: Note that the shell's 'umask' setting screws with our   *
//* permission bits during the mkdir() call. The 'umask' is generally set by   *
//* the user's login shell script, and it is dangerous to change the umask     *
//* definition, but we can often compensate by following the directory         *
//* creation with an explicit setting of the file's permission bits.           *
//*                                                                            *
//* If permissions are passed in 'dfsPtr', we may, or may not get the expected *
//* result, depending on what the 'umask' does. If we use the default          *
//* permissions, then we SHOULD get what we ask for, but this cannot be        *
//* guaranteed.                                                                *
//*                                                                            *
//* Default Permissions:                                                       *
//* If a user creates a directory using the core utils, 'mkdir' command, the   *
//* permissions will (usually) be: rwxrwxr-x meaning full permissions for      *
//* owner and group,  with read/execute for others; so, that is what we        *
//* strive for here.                                                           *
//*                                                                            *
//* Notes on directory timestamps:                                             *
//* -- The Mod timestamp is set when the directory is created OR when its      *
//*    contents are modified.                                                  *
//* -- The Access timestamp is set when the directory is created OR when the   *
//*    contents of the directory are read.                                     *
//* -- The Ctime timestamp is updated with every change in status and is out   *
//*    of our control.                                                         *
//* We can set these timestamps here, but they may be changed by any activity  *
//* in the target directory.                                                   *
//*                                                                            *
//******************************************************************************

bool cTrash::ctfCreateDirectory ( const gString& trgPath, const DFileStats* dfsPtr )
{
   bool status ;

   //* If permission bits provided, use them, otherwise use defaults *
   mode_t trg_mode = (dfsPtr == NULL) ? (S_IRWXU | S_IRWXG | S_IROTH | S_IXOTH)
                                      : dfsPtr->rawStats.st_mode ;

   if ( (mkdir ( trgPath.ustr(), trg_mode )) == ZERO )
   {
      status = true ;
      if ( dfsPtr != NULL )
         this->ctfSetTimestamps ( trgPath, dfsPtr->rawStats.st_mtime,
                                  dfsPtr->rawStats.st_atime ) ;
   }
   return status ;

}  //* End ctfCreateDirectory() *

//*************************
//*  ctfCreateDirectory   *
//*************************
//******************************************************************************
//* Delete an empty directory file.                                            *
//* Optionally, delete the contents of the directory before deleting target.   *
//*                                                                            *
//* Input  : trgPath  : (by reference) full path/filename of target directory  *
//*          delContents : (optional, 'false' by default)                      *
//*                     If specified, first delete all data contained in the   *
//*                     target directory; THEN delete the target directory.    *
//*                                                                            *
//* Returns: 'true'  if successful,                                            *
//*          'false' if unable to remove target                                *
//*                  a) target directory is not empty (but see 'delContents')  *
//*                  b) target does not specify a directory name               *
//******************************************************************************
//* Programmer's Note.                                                         *
//*  a) A directory name cannot be deleted directly if it contains other data. *
//*  b) It is dangerous to delete a directory and all its contents UNLESS,     *
//*     you are very sure what those contents are. You have been warned!       *
//*  c) The 'DeleteDirContents' method returns the number of files deleted.    *
//*     -- ZERO if target is empty OR if user does not have access             *
//*     -- > ZERO if one or more files deleted, BUT it DOES NOT indicate       *
//*        whether ALL files were deleted.                                     *
//*     Therefore, the success or failure of 'rmdir' indicates whether         *
//*     all directory contents were successfully deleted.                      *
//*                                                                            *
//******************************************************************************

bool cTrash::ctfDeleteDirectory ( const gString& trgPath, bool delContents )
{
   if ( delContents != false )
      this->ctfDeleteDirContents ( trgPath ) ;
   return ( bool((rmdir ( trgPath.ustr() )) == ZERO) ) ;

}  //* End ctfDeleteDirectory() *

//*************************
//*  ctfCopyDirContents   *
//*************************
//******************************************************************************
//* Copy the contents of the source directory to the target directory.         *
//* a) Both source and target directories must already exist.                  *
//* b) User must have read/write access to all source files.                   *
//*    This must be verified BEFORE calling this method.                       *
//* c) User must have write access to target directory.                        *
//*    This must be verified BEFORE calling this method.                       *
//* d) If an existing target file, (unlikely in our controlled environment),   *
//*    it will be silently overwritten.                                        *
//* e) We do not abort on errors, but copy as much of the directory contents   *
//*    as we can access.                                                       *
//*                                                                            *
//*                                                                            *
//* Input  : srcPath  : (by reference) full path/filename of source directory  *
//*          trgPath  : (by reference) full path/filename of target directory  *
//*                                                                            *
//* Returns: number of files successfully copied                               *
//*          Caller should compare this value with the number of files counted *
//*          during the validation scan.                                       *
//******************************************************************************

UINT cTrash::ctfCopyDirContents ( const gString& srcPath, const gString& trgPath ) 
{
   #define DEBUG_CDC (0)      // DEBUGGING

   UINT  filesCopied = ZERO ;       // return value
   DIR*   dirPtr ;                  // directory-file access

   //* Open the source directory.*
   if ( (dirPtr = this->ctfOpendir ( srcPath )) != NULL )
   {
      gString   sspec, tspec ;      // source and target filespecs
      FileStats rawStats ;          // source stats
      deStats*  destat ;            // directory-entry record
      bool      showFile ;          // 'true' if file is to be copied

      //* Read each item in the directory *
      while ( (destat = readdir64 ( dirPtr )) != NULL )
      {
         //* Do not include 'current dir' and 'parent dir' names.              *
         showFile = true ;
         if ( destat->d_name[ZERO] == PERIOD )
         {
            if (   destat->d_name[1] == NULLCHAR 
                || (destat->d_name[1] == PERIOD && destat->d_name[2] == NULLCHAR) )
               showFile = false ;
         }

         if ( showFile != false )
         {
            #if DEBUG_CDC != 0      // DEBUGGING
            wcout << "NAME: '" << destat->d_name << L"'" << endl ;
            #endif   // DEBUG_CDC

            //* Build full path/filename for source and target *
            this->ctfCatPathFilename ( sspec, srcPath, destat->d_name ) ;
            this->ctfCatPathFilename ( tspec, trgPath, destat->d_name ) ;

            #if DEBUG_CDC != 0      // DEBUGGING
            wcout << L" SRC: '" << sspec.gstr() << L"'\n"
                  << L" TRG: '" << tspec.gstr() << L"'" << endl ;
            #endif   // DEBUG_CDC

            //* Do an 'lstat' on the source for filetype *
            if ( (lstat64 ( sspec.ustr(), &rawStats )) == OK )
            {
               if ( (this->ctfCopyFile ( sspec, tspec )) != false )
                  ++filesCopied ;
               //* If source is a directory name, *
               //* recursively copy its contents. *
               if ( (S_ISDIR(rawStats.st_mode)) != false )
               {
                  filesCopied += this->ctfCopyDirContents ( sspec, tspec ) ;
               }
            }
         }
      }
      closedir ( dirPtr ) ;      // Close the directory
   }
   return filesCopied ;

   #undef DEBUG_CDC     // DEBUGGING
}  //* End cftCopyDirContents() *

//*************************
//* ctfDeleteDirContents  *
//*************************
//******************************************************************************
//* Delete the contents of the target directory. Directory itself not deleted. *
//* a) Target directory must exist.                                            *
//* b) User must have read/write access to all directory contents.             *
//*    This must be verified BEFORE calling this method.                       *
//* c) We do not abort on errors, but unlink as much of the directory contents *
//*    as we can access.                                                       *
//*                                                                            *
//* Input  : trgPath  : (by reference) full path/filename of target directory  *
//*                                                                            *
//* Returns: number of files successfully unlinked                             *
//*          Caller should compare this value with the number of files counted *
//*          during the validation scan.                                       *
//******************************************************************************

UINT cTrash::ctfDeleteDirContents ( const gString& trgPath )
{
   #define DEBUG_DDC (0)      // DEBUGGING

   UINT  filesDeleted = ZERO ;      // return value
   DIR*   dirPtr ;                  // directory-file access

   //* Open the target directory.*
   if ( (dirPtr = this->ctfOpendir ( trgPath )) != NULL )
   {
      gString   tspec ;             // target filespec
      FileStats rawStats ;          // source stats
      deStats*  destat ;            // directory-entry record
      bool      showFile ;          // 'true' if file is to be copied

      //* Read each item in the directory *
      while ( (destat = readdir64 ( dirPtr )) != NULL )
      {
         //* Do not include 'current dir' and 'parent dir' names.              *
         showFile = true ;
         if ( destat->d_name[ZERO] == PERIOD )
         {
            if (   destat->d_name[1] == NULLCHAR 
                || (destat->d_name[1] == PERIOD && destat->d_name[2] == NULLCHAR) )
               showFile = false ;
         }

         if ( showFile != false )
         {
            #if DEBUG_DDC != 0      // DEBUGGING
            wcout << "NAME: '" << destat->d_name << L"'" << endl ;
            #endif   // DEBUG_DDC

            //* Build full path/filename for target *
            this->ctfCatPathFilename ( tspec, trgPath, destat->d_name ) ;

            #if DEBUG_DDC != 0      // DEBUGGING
            wcout << L" TRG: '" << tspec.gstr() << L"'" << endl ;
            #endif   // DEBUG_CDC

            //* Do an 'lstat' on the target for filetype *
            if ( (lstat64 ( tspec.ustr(), &rawStats )) == OK )
            {
               //* If target is a directory name,   *
               //* recursively delete its contents. *
               if ( (S_ISDIR(rawStats.st_mode)) != false )
               {
                  filesDeleted += this->ctfDeleteDirContents ( tspec ) ;
                  if ( (this->ctfDeleteDirectory ( tspec )) != false )
                     ++filesDeleted ;
               }
               else if ( (this->ctfDeleteFile ( tspec )) != false )
                  ++filesDeleted ;
            }
         }
      }
      closedir ( dirPtr ) ;      // Close the directory
   }
   return filesDeleted ;

   #undef DEBUG_DDC     // DEBUGGING
}  //* End ctfDeleteDirContents() *

//*************************
//*   ctfSetTimestamps    *
//*************************
//******************************************************************************
//* Set both Modify and Access timestamps for the target file.                 *
//* a) As a side-effect, 'Change time' will be set to the system current time. *
//* b) Operation requires that user have write permission on the target.       *
//*                                                                            *
//*                                                                            *
//* Input  : trgPath   : full path/filename specification of target            *
//*          modTime   : 'modified' timestamp code                             *
//*          accTime   : 'accessed' timestamp code                             *
//*                                                                            *
//* Returns: 'true'  if successful                                             *
//*          'false' if no write access OR if system call fails                *
//******************************************************************************
//*                                                                            *
//* Aliased as typedef struct utimbuf uTime:                                   *
//* struct utimbuf        // used for call to utime()                          *
//* {                                                                          *
//*    time_t actime ;    // access time                                       *
//*    time_t modtime ;   // modification time                                 *
//* } ;                                                                        *
//*                                                                            *
//******************************************************************************

bool cTrash::ctfSetTimestamps ( const gString& trgPath, time_t modTime, time_t accTime  )
{
   bool status = false ;

   uTime ut = { accTime, modTime } ;
   status = bool( (utime ( trgPath.ustr(), &ut )) == ZERO) ;

   return status ;

}  //* End ctfSetTimestamps() *

//*************************
//*       ctfCdPath       *
//*************************
//******************************************************************************
//* Change directory to directory specified by full path string.               *
//*                                                                            *
//* Input  : dPath: full path specification for new target directory           *
//*                                                                            *
//* Returns: If successful, returns 'true'                                     *
//*           If operation fails, returns 'false'                              *
//******************************************************************************

bool cTrash::ctfCdPath ( const gString& dPath )
{
   bool  status = false ;

   if ( (chdir ( dPath.ustr() )) == OK )
   {
      status = true ;
   }
   return status ;

}  //* End ctfCdPath() *

//*************************
//*      ctfReadLine      *
//*************************
//******************************************************************************
//* Read one line of text from the specified input stream.                     *
//* This is a generic ReadLine.                                                *
//*                                                                            *
//* Input  : ifs    : open input stream (by reference)                         *
//*          gs     : (by reference) receives text data                        *
//*                                                                            *
//* Returns: 'true'  if line read successfully, else 'false'                   *
//******************************************************************************

bool cTrash::ctfReadLine ( ifstream& ifs, gString& gs )
{
   char  tmp[gsMAXBYTES] ; // UTF-8 line input from file
   bool  status = false ;  // return value

   ifs.getline ( tmp, gsMAXBYTES, NEWLINE ) ;   // read a source line
   if ( ifs.good() || ifs.gcount() > ZERO )     // if a good read
   {
      gs = tmp ;                                // convert to wide data
      status = true ;                           // return with good news
   }
   return status ;

}  //* End ctfReadLine() *


//**************************************
//*** Development and debugging only ***
//**************************************
#if DEBUG_METHODS != 0
//*****************************
//* ctfDisplayFilesystemStats *
//*****************************
//******************************************************************************
//* DEVELOPMENT ONLY.                                                          *
//* Display filesystem stats                                                   *
//*                                                                            *
//* Input  : fPath : path/filename of any file on the filesystem               *
//*          fss   : initialized filesystem data                               *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void cTrash::ctfDisplayFilesystemStats ( const gString& fPath, const fileSystemStats& fss )
{
   this->textOut ( L"Filesystem containing:\n"
                    "  ", false ) ;
   this->textOut ( fPath ) ;
   gString gsOut( "File System Type: %0X - %s\n"
                  "  File System ID: %llX", 
                  &fss.fsTypeCode, fss.fsType, &fss.systemID ) ;
   this->textOut ( gsOut ) ;

   gString tbk, fbk, abk, tin, ain, bbs, fbs, uby, fby, fnl ;
   tbk.formatInt( fss.blockTotal, 11 ) ;
   fbk.formatInt( fss.blockFree, 11 ) ;
   abk.formatInt( fss.blockAvail, 11 ) ;
   tin.formatInt( fss.inodeTotal, 10, true ) ;
   ain.formatInt( fss.inodeAvail, 10, true ) ;
   bbs.formatInt( (ULONG)fss.blockSize, 10, true ) ;
   fbs.formatInt( (ULONG)fss.fblockSize, 10, true ) ;
   uby.formatInt( fss.usedBytes, 8, true ) ;
   fby.formatInt( fss.freeBytes, 8, true ) ;
   fnl.formatInt( (ULONG)fss.nameLen, 5, true ) ;

   gsOut.compose( "Total Blocks:%S    Total Inodes: %S\n"
                  " Free Blocks:%S    Avail Inodes: %S\n"
                  "Avail Blocks:%S    Block Size  : %S\n"
                  "  Used Bytes: %S      FBlock Size : %S\n"
                  "  Free Bytes: %S      Max Filename: %S",
                  tbk.gstr(), tin.gstr(),
                  fbk.gstr(), ain.gstr(),
                  abk.gstr(), bbs.gstr(),
                  uby.gstr(), fbs.gstr(),
                  fby.gstr(), fnl.gstr()
                  ) ;
   this->textOut ( gsOut ) ;

   gsOut.compose( "SELinux Security context: %s\n", fss.seLinuxCode ) ;
   this->textOut ( gsOut ) ;

}  //* End ctfDisplayFilesystemStats() *

#endif      //*** END DEBUG_METHODS ***

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

