//******************************************************************************
//* File       : gString.hpp                                                   *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2011-2020 Mahlon R. Smith, The Software Samurai *
//*                 GNU GPL copyright notice below                             *
//* Date       : 12-Dec-2019                                                   *
//* Version    : (see gStringVersion string in gString.cpp)                    *
//*                                                                            *
//* Description: This class is used to convert between UTF-8 character         *
//* encoding and wchar_t character encoding. Statistical information for       *
//* analysis and display of the data is generated.                             *
//*                                                                            *
//* This is not a comprehensive string class, but a highly-specialized,        *
//* single-purpose string class for text encoding conversions. The gString     *
//* class was developed to support the NcDialog class, NCurses class, NcWindow *
//* classes and their descendants, and is a compromise between the full        *
//* functionality of Glib::ustring and the need to keep the NCurses, NcWindow, *
//* and NcDialog classes independent from third-party libraries.               *
//*                                                                            *
//* The gString class is a stand-alone class, that is, it requires no third-   *
//* party source or libraries beyond the standard C and C++ libraries found in *
//* all GNU compiler releases. Note however that the flag '_XOPEN_SOURCE'      *
//* must be defined (see below).                                               *
//*                                                                            *
//* Dynamic memory allocation is not performed within the gString class, and   *
//* The capacity of gString (see gsMAXCHARS and gsMAXBYTES) was chosen to      *
//* accomodate the maximum length of a path/filename string,                   *
//* i.e. 'PATH_MAX' (4096 bytes) as defined in the POSIX standard and found    *
//* in 'limits.h'.                                                             *
//*                                                                            *
//* This class is relatively fast and is useful for text data analysis and     *
//* manipulation, but is rather inefficient for data storage (although perhaps *
//* no more inefficient than std::string()).                                   *
//*                                                                            *
//*                                                                            *
//* Copyright Notice:                                                          *
//* -----------------                                                          *
//* This program is free software: you can redistribute it and/or modify it    *
//* under the terms of the GNU General Public License as published by the Free *
//* Software Foundation, either version 3 of the License, or (at your option)  *
//* any later version, PROVIDED THAT the copyright notices for both code and   *
//* documentation are included and unmodified.                                 *
//*                                                                            *
//* This program is distributed in the hope that it will be useful, but        *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
//* for more details.                                                          *
//*                                                                            *
//* You should have received a copy of the GNU General Public License along    *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.            *
//*                                                                            *
//*         Full text of the GPL License may be found in the Texinfo           *
//*         documentation for this program under 'Copyright Notice'.           *
//******************************************************************************
//* Version History (most recent first):                                       *
//*   See version history in gString.cpp.                                      *
//*                                                                            *
//******************************************************************************
//* Programmer's Notes:                                                        *
//* 1) To assign data from a std::string class object to a gString object,     *
//*    use one of the following:                                               *
//*      std::string st("Hello World!") ;                                      *
//*      gString gs(st.c_str()) ;                                              *
//*      OR                                                                    *
//*      gString gs ;                                                          *
//*      gs = st.c_str() ;                                                     *
//*                                                                            *
//* 2) To assign data from a Glib::ustring class object to a gString object,   *
//*    use one of the following:                                               *
//*      Glib::ustring ust("Good Morning!") ;                                  *
//*      gString gs(ust.c_str()) ;                                             *
//*      OR                                                                    *
//*      gString gs ;                                                          *
//*      gs = ust.c_str() ;                                                    *
//*                                                                            *
//******************************************************************************

#ifndef GSTRING_INCLUDED
#define GSTRING_INCLUDED

//* _XOPEN_SOURCE must be defined for including cwchar header file *
#ifndef _XOPEN_SOURCE
#define _XOPEN_SOURCE
#endif   // _XOPEN_SOURCE

//*****************
//* Include Files *
//*****************
#include <cstdlib>      //* Misc. functionality including character conversion
#include <cstdarg>      //* Variadic-function macro definitions
#include <cwchar>       //* For wcwidth function
#include <iostream>     //* For insertion operator only

//** For Debugging Only - Disable In Production **
#define ENABLE_GS_DEBUG (0)

const short gsMAXCHARS  = (1025) ;  // maximum number of converted characters
const short gsMAXBYTES  = (gsMAXCHARS * 4) ; // max bytes of source data scanned

//* Max field-width parameter for the formatInt methods *
const short FI_MAX_4BYTE_WIDTH = 13 ;  // four-byte integer with comma separators
const short FI_MAX_8BYTE_WIDTH = 18 ;  // eight-byte integer with comma separators
const short FI_MAX_FIELDWIDTH = FI_MAX_8BYTE_WIDTH ;

//* For the 'formatInt' methods, the format of the units suffix.               *
//*     METRIC                       BINARY                                    *
//*     kilobytes (x/1000)           kibibytes (x/1024),                       *
//*     megabytes (x/1000000)        mibibytes (x/1024000)                     *
//*     gigabytes (x/1000000000)     gibibytes (x/1024000000).                 *
//*     terabytes (x/1000000000000)  tebibytes (x/1024000000000)               *
//* The 'formatInt' methods use metric (powers of 10) calculations by default. *
//*                                                                            *
//* The IEC (International System of Quantities) recommends lower-case for     *
//* metric (powers of 10) and upper-case for binary (powers of 2).             *
//* However, unless you must be accurate in presenting the data according to   *
//* IEC standard, it is recommended that you choose the format according to:   *
//* your space requirements, visual appeal, and clear communication with your  *
//* users.                                                                     *
enum fiUnits : short
{
   fiK,     // (default) single, upper-case character:   'K'   'M'   'G'   'T'
   fik,     // single, lower-case character          :   'k'   'm'   'g'   't'
   fiKb,    // two-character, Uppercase/Lowercase    :  'Kb'  'Mb'  'Gb'  'Tb'
   fikB,    // "official" metric 'kilo' designation  :  'kB'  'mB'  'gB'  'tB'
   fiKiB,   // "official" binary 'kibi' designation  : 'KiB' 'MiB' 'GiB' 'TiB'
} ;

//* Use the gsForm class with the gString constructor or operator= to create   *
//* a formatted string.                                                        *
//* Example:                                                                   *
//* char utfString[] = { "Hello! " } ;   short sVal = 21 ;   int iVal = 523 ;  *
//* gsForm gsf { L"%s: %hd %d", { utfString, &sVal, &iVal } } ;                *
//*    gString gs ;                                                            *
//*    gs = gsf ;                                                              *
//*       OR                                                                   *
//*    gString gs( gsf ) ;                                                     *
//* See also the description of the compose() method below.                    *
const short gsFormMAXARGS = 24 ;    // Maximum number of conversion parameters
class gsForm
{
   public:
   //* Formatting string that specifies data type and format for each of       *
   //* the parameters in argList.                                              *
   const wchar_t* fmtSpec ;

   //* Pointers to each value to be converted                                  *
   const void*    argList[gsFormMAXARGS] ;

} ;


//******************************
//** Define the gString class **
//******************************
class gString
{
public:
virtual ~gString() ;             // destructor

//* Constructor: Initialize members to default values (NULL string).           *
//*                                                                            *
//* Input  : none                                                              *
//* Returns: nothing                                                           *
gString ( void ) ;

//* Constructor: Convert specified UTF-8-encoded source to gString.            *
//*                                                                            *
//* Input  : usrc  : pointer to a UTF-8-encoded, null-terminated string        *
//*          charLimit: (optional, gsMAXCHARS by default)                      *
//*                  maximum number of characters from source array to convert *
//* Returns: nothing                                                           *
gString ( const char* usrc, short charLimit=gsMAXCHARS ) ;

//* Constructor: Convert specified wchar_t ('wide') source to gString.         *
//*                                                                            *
//* Input  : wsrc  : pointer to a wchar_t-encoded, null-terminated string      *
//*          charLimit: (optional, gsMAXCHARS by default)                      *
//*                  maximum number of characters from source array to convert *
//* Returns: nothing                                                           *
gString ( const wchar_t* wsrc, short charLimit=gsMAXCHARS ) ;

//* Constructor: Convert formatting specification and its arguments to gString.*
//*              Please refer to compose() method for more information.        *
//*                                                                            *
//* Input  : fmt  : a format specification string in the style of sprintf(),   *
//*                 swprintf() and related formatting C/C++ functions.         *
//*          arg1 : pointer to first value to be converted by 'fmt'            *
//*          ...  : optional arguments (between ZERO and gsfMAXARGS - 1)       *
//*                 Each optional argument is a POINTER (address of) the value *
//*                 to be formatted.                                           *
//*                                                                            *
//* Returns: nothing                                                           *
gString ( const char* fmt, const void* arg1, ... ) 
          __attribute__ ((format (gnu_printf, 2, 0))) ;

//* Constructor: Convert specified integer value to gString.                   *
//*              Please see the 'formatInt' method for formatting details.     *
//*                                                                            *
//* Input  : iVal  : integer value to be converted                             *
//*                  Supported value range: plus/minus 9.999999999999 terabytes*
//*          fWidth: field width (number of display columns)                   *
//*                  range: 1 to FI_MAX_FIELDWIDTH                             *
//*          lJust : (optional, false by default)                              *
//*                  if true, strip leading spaces to left-justify the value   *
//*                  in the field. (resulting string may be less than fWidth)  *
//*          sign  : (optional, false by default)                              *
//*                  'false' : only negative values are signed                 *
//*                  'true'  : always prepend a '+' or '-' sign.               *
//*          kibi  : (optional, false by default)                              *
//*                  'false' : calculate as a decimal value (powers of 10)     *
//*                            kilobyte, megabyte, gigabyte, terabyte          *
//*                  'true'  : calculate as a binary value (powers of 2)       *
//*                            kibibyte, mebibyte, gibibyte, tebibyte          *
//*          units : (optional) member of enum fiUnits (fiK by default)        *
//*                  specifies the format for the units suffix.                *
//*                  Note that if the uncompressed value fits in the field,    *
//*                  then this parameter is ignored.                           *
//*                                                                            *
//* Returns: nothing                                                           *
//*          Note: if field overflow, field will be filled with '#' characters.*
gString ( short iVal, short fWidth, bool lJust = false, 
          bool sign = false, bool kibi = false, fiUnits units = fiK ) ;
gString ( unsigned short iVal, short fWidth, bool lJust = false, 
          bool sign = false, bool kibi = false, fiUnits units = fiK ) ;
gString ( int iVal, short fWidth, bool lJust = false, 
          bool sign = false, bool kibi = false, fiUnits units = fiK ) ;
gString ( unsigned int iVal, short fWidth, bool lJust = false, 
          bool sign = false, bool kibi = false, fiUnits units = fiK ) ;
gString ( long iVal, short fWidth, bool lJust = false, 
          bool sign = false, bool kibi = false, fiUnits units = fiK ) ;
gString ( unsigned long iVal, short fWidth, bool lJust = false, 
          bool sign = false, bool kibi = false, fiUnits units = fiK ) ;
gString ( long long iVal, short fWidth, bool lJust = false, 
          bool sign = false, bool kibi = false, fiUnits units = fiK ) ;
gString ( unsigned long long iVal, short fWidth, bool lJust = false, 
          bool sign = false, bool kibi = false, fiUnits units = fiK ) ;

//* Constructor: Convert formatting instructions in gsForm class object        *
//*              to gString. See compose() for more information.               *
//*                                                                            *
//* Input  : gsf   : initialized gsForm class object containing parameters for *
//*                  creating a formatted text string.                         *
//*          charLimit: (optional, gsMAXCHARS by default)                      *
//*                  maximum number of characters from source array to convert *
//* Returns: nothing                                                           *
gString ( const gsForm& gsf, short charLimit=gsMAXCHARS ) ;

//* Assignment operator: converts UTF-8-encoded source to gString.             *
void operator = ( const char* usrc ) ;

//* Assignment operator: converts wchar_t ('wide') source to gString.          *
void operator = ( const wchar_t* wsrc ) ;

//* Assignment operator: Converts gsForm-class instructions to gString.        *
void operator = ( const gsForm& gsf ) ;

//* Assignment operator. Copies one gString object to another.       .         *
void operator = ( const gString& gssrc ) ;

//* Create formatted text data from a format specification string including    *
//* between ZERO and gsfMAXARGS format specifications and their corresponding  *
//* argument pointers..                                                        *
//*                                                                            *
//* Supported data types:                                                      *
//*  %d, %i  integer (decimal)                                                 *
//*  %o      integer (octal)                                                   *
//*  %u      integer (unsigned)                                                *
//*  %x, %X  integer (hex lower or upper case)                                 *
//*  %f      floating point (fixed point)                                      *
//*  %e, %E  floating point (scientific notation, lower/uppercase)             *
//*  %g, %G  floating point (normal/exponential, lower/uppercase)              *
//*  %a, %A  floating point (hex fraction)                                     *
//*  %c      character                                                         *
//*  %C      character (alias for %lc)                                         *
//*  %s      string                                                            *
//*  %S      string (alias for %ls)                                            *
//*  %p      pointer                                                           *
//*  %b, %B  (extension to swprintf - see description below)                   *
//*  %m      capture 'errno' description string (see /usr/include/errno.h)     *
//*  %n      number of characters printed so far                               *
//*          (value written to corresponding argument's location)              *
//*  %%      literal '%'                                                       *
//*                                                                            *
//* See man pages for the C/C++ function 'swprintf' or                         *
//* 'Table of Output Conversions' for additional details.                      *
//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *
//*                                                                            *
//* Input  : fmt  : a format specification string in the style of sprintf(),   *
//*                 swprintf() and related formatting C/C++ functions.         *
//*          ...  : optional arguments (between ZERO and gsfMAXARGS)           *
//*                 Each optional argument is a POINTER (address of) the value *
//*                 to be formatted.                                           *
//*                 - Important Note: There must be AT LEAST as many optional  *
//*                   arguments as the number of format specifiers defined in  *
//*                   the formatting string. Excess arguments will be ignored; *
//*                   HOWEVER, too few arguments will result in an application *
//*                   crash and ridicule from your peers.                      *
//*                                                                            *
//* Returns: const wchar_t* to formatted data                                  *
//*                                                                            *
//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *
//* Implementation Notes:                                                      *
//* =====================                                                      *
//*                                                                            *
//* 1) Unsupported data types                                                  *
//*    ======================                                                  *
//*    Conversion modifiers that are not fully supported at this time:         *
//*                       'j', 'z', 't', '%['                                  *
//*    Note: The '*' field-width specification or precision specification      *
//*          which uses the following argument as the width/precision value    *
//*          IS NOT supported.                                                 *
//*                                                                            *
//* 2) Binary-format specifier                                                 *
//*    =======================                                                 *
//*    We implement an extension to the swprintf output-conversion-specifiers  *
//*    for binary formatted output. We have found this formatting option useful*
//*    when working with bit masks and for verifying bit-shifting operations.  *
//*    - Base formatting specifier:                                            *
//*       %b , %B                                                              *
//*    - Format modifiers for DATA SIZE are the same as for swprintf:          *
//*       hh , h , l , ll , L , q                                              *
//*    - Format modifier for prepending of a type indicator.                   *
//*       '#' (hash character)                                                 *
//*       This is the same principle as for prepending a '0x' indicator to hex *
//*       output, and will place either a 'b' or 'B' character at the beginning*
//*       of the output. Examples:  %#hhb -> b0101.1010   %#hhB -> B0101.1010  *
//*    - Format modifier for appending of a type indicator.                    *
//*       '-#' (minus sign and hash character)                                 *
//*       Rather than prepending the indicator, the indicator will be append   *
//*       to the end of the output.                                            *
//*       Examples: %-#hhb -> 0101.1010b   %-#hhB -> 0101.1010B                *
//*    - Format modifier for specifying the group-seperator character.         *
//*       By default, the bit groups are seperated by a '.' (fullstop)         *
//*       character. To specify an alternate seperator character:              *
//*       % hB -> 0111 0101 1010 0001    (' ' (space) character as seperator)  *
//*       %_hB -> 0111_0101_1010_0001    ('_' (underscore) char as seperator)  *
//*       %#/hB -> B0111/0101/1010/0001  ('/' (slash) character as seperator)  *
//*       %-#-hB -> 0111-0101-1010-0001B ('-' (dash) character as seperator)   *
//*       Valid seperator characters are any printable ASCII character         *
//*       that IS NOT alphabetical, IS NOT a number, and IS NOT a '.'(fullstop)*
//*    - Format modifier for specifying bit grouping.                          *
//*       By default, bits are formatted in groups of four (4 nybble); however,*
//*       if desired, bits can be formatted in groups of eight (8 byte):       *
//*       %.8hB -> 01110101.10100001                                           *
//*       %-.8hB -> 01110101-10100001                                          *
//*       %# .8hB -> B01110101 10100001                                        *
//*       %-#`.8hb -> 01110101`10100001b                                       *
//*                                                                            *
//* 3) Format-specification Deficiencies                                       *
//*    =================================                                       *
//*    The compiler will scan your UTF-8 formatting string for valid           *
//*    formatting specifications and warn you of potential problems.           *
//*      HOWEVER:                                                              *
//*    The ANSI-C/C++ specification does not yet support parsing of wchar_t    *
//*    (swprintf-style) formatting strings. If you pass a wchar_t formatting   *
//*    string, the compiler will not be able to warn of potential problems,    *
//*    so construct your formatting string carefully. See gcc documentation    *
//*    for more information:  info -f gcc.info -n 'Function Attributes'        *
//*      ADDITIONALLY:                                                         *
//*    Because the optional parameters are passed as POINTERS, similar to the  *
//*    C/C++ library function 'sscanf' and friends, the compiler cannot perform*
//*    automatic promotions from short int* to int* or from float* to double*, *
//*    and so-on. For this reason, please use care that the data type you      *
//*    specify in the formatting string matches the data type of the variable  *
//*    referenced by its parameter pointer. For example, the conversion        *
//*    specifications:  %d, %i, %u, %x, and %X  assume a parameter of 'int'    *
//*    size If your actual variable declaration is for a 8-bit int, short int, *
//*    long int or long long int, then explicitly indicate this in the         *
//*    corresponding conversion specification.                                 *
//*    Examples:                                                               *
//*      char      Greeting[] = { "Hello!" } ;                                 *
//*      int       iValue ;                                                    *
//*      long long int qValue ;                                                *
//*      long int  lValue ;                                                    *
//*      short int sValue1, sValue2, sValue3 ;                                 *
//*      bool      flagValue ;                                                 *
//*      float     fltValue ;                                                  *
//*      double    dblValue ;                                                  *
//*      gString gs ;                                                          *
//*      gs.compose( "%s - %d %12hd, %-hi, %#hx %08lX %llu %hhd",              *
//*                  Greeting, &iValue, &sValue1, &sValue2, &sValue3,          *
//*                  &lValue, &qValue, &flagValue );                           *
//*      gs.compose( "floating downstream:%10.2f and doubling our pace:%.4lf", *
//*                  &fltValue, &dblValue ) ;                                  *
//*                                                                            *
//*      Note: Use care when formatting an 8-bit integer: 'char' ,             *
//*            'unsigned char' or a boolean value declared with the 'bool'     *
//*            keyword. For these types, use the 'hh' conversion modifier.     *
//*            Examples: %hhd , %hhu , %hhX                                    *
//*                                                                            *
//* 4) 'swprintf' bug-fix                                                      *
//*    ==================                                                      *
//*    The standard library 'swprintf' function has a design flaw for format   *
//*    specifications that include a field-width specifier.                    *
//*    'swprintf' pads the string to the specified number of CHARACTERS, not   *
//*    the number of COLUMNS as it should do. For (Arabic) numeric source      *
//*    values this is not a problem because one character == one display       *
//*    column. For string source data, however, if the source string contains  *
//*    characters that require more than one display column each, then the     *
//*    output may be too wide.                                                 *
//*    Therefore, for string-source-formatting specifications ONLY             *
//*          (examples: "%12s"  "%-6s"  "%16ls"  "%5S"  "%-24S")               *
//*    we compensate for this anomalous behavior by interpreting the           *
//*    field-width specifier as number-of-columns, NOT number-of-characters.   *
//*    This will result in output that appears different (and better) than     *
//*    output created directly by the 'swprintf' function.                     *
//*                                                                            *
//* 5) Parameter type checking                                                 *
//*    =======================                                                 *
//*    Unfortunately, type-checking of wchar_t formatting strings is not yet   *
//*    supported by the gnu (v:4.8.0) compiler, (but see wchar.h which is      *
//*    preparing for future expansion). Thus, use care when constructing your  *
//*    'wchar_t fmt' formatting string. The 'char fmt' string is type-checked. *
//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *
const wchar_t* compose ( const wchar_t* fmt, ... ) 
                         /*__attribute__ ((format (gnu_wprintf, 2, 0)))*/ ;
const wchar_t* compose ( const char* fmt, ... ) 
                         __attribute__ ((format (gnu_printf, 2, 0))) ;

//* Convert an integer value into a formatted display string of the specified  *
//* width. Value is right-justified in the field, with leading spaces added    *
//* if necessary (but see 'lJust' parameter).                                  *
//* Maximum field width is FI_MAX_FIELDWIDTH. This is wide enough to display   *
//* a 13-digit, signed and comma-formatted value: '+9,876,543,210,777'         *
//*                                                                            *
//* Actual formatting of the value depends on the combination of:              *
//*   a) magnitude of the value                                                *
//*   b) whether it is a signed value                                          *
//*   c) the specified field-width                                             *
//*   d) the specified suffix format                                           *
//*   e) locale-specific grouping of digits according the LC_NUMERIC locale    *
//*      environment variable                                                  *
//* Examples are for the 'C' and U.S. English locale with default suffix.      *
//* 1) Simple comma formatted output if specified field-width is sufficient.   *
//*    Examples:  345    654,345    782,654,345    4,294,967,295               *
//* 2) Compression of the value to fit a specified field width.                *
//*    Examples:  12.3K    999K    12.345M    1.234G    4.3G                   *
//* Please see NcDialog test application, 'Dialogw', 'gString class            *
//* functionality test' for more examples.                                     *
//*                                                                            *
//* Input  : iVal  : integer value to be converted                             *
//*                  Supported value range: plus/minus 9.999999999999 terabytes*
//*          fWidth: field width (number of display columns)                   *
//*                  range: 1 to FI_MAX_FIELDWIDTH                             *
//*          lJust : (optional, false by default)                              *
//*                  if true, strip leading spaces to left-justify the value   *
//*                  in the field. (resulting string may be less than fWidth)  *
//*          sign  : (optional, false by default)                              *
//*                  'false' : only negative values are signed                 *
//*                  'true'  : always prepend a '+' or '-' sign.               *
//*          kibi  : (optional, false by default)                              *
//*                  'false' : calculate as a decimal value (powers of 10)     *
//*                            kilobyte, megabyte, gigabyte, terabyte          *
//*                  'true'  : calculate as a binary value (powers of 2)       *
//*                            kibibyte, mebibyte, gibibyte, tebibyte          *
//*         units  : (optional) member of enum fiUnits (fiK by default)        *
//*                  specifies the format for the units suffix.                *
//*                  Note that if the uncompressed value fits in the field,    *
//*                  then this parameter is ignored.                           *
//*                                                                            *
//* Returns: 'true' if successful                                              *
//*          'false' if field overflow (field will be filled with '#' chars)   *
//*                  note that oveflow can occur ONLY:                         *
//*                  if fWidth <= 3 OR                                         *
//*                  if iVal <= -10.0 terabytes OR iVal >= 10.0 terabytes      *
//*                  if units is a multi-column substring, then some values    *
//*                     will overflow 4-column, 5-column, and 6-column fields  *
bool  formatInt ( short iVal, short fWidth, bool lJust = false, 
                  bool sign = false, bool kibi = false, fiUnits units = fiK ) ;
bool  formatInt ( unsigned short iVal, short fWidth, bool lJust = false, 
                  bool sign = false, bool kibi = false, fiUnits units = fiK ) ;
bool  formatInt ( int iVal, short fWidth, bool lJust = false, 
                  bool sign = false, bool kibi = false, fiUnits units = fiK ) ;
bool  formatInt ( unsigned int iVal, short fWidth, bool lJust = false, 
                  bool sign = false, bool kibi = false, fiUnits units = fiK ) ;
bool  formatInt ( long iVal, short fWidth, bool lJust = false,
                  bool sign = false, bool kibi = false, fiUnits units = fiK ) ;
bool  formatInt ( unsigned long iVal, short fWidth, bool lJust = false, 
                  bool sign = false, bool kibi = false, fiUnits units = fiK ) ;
bool  formatInt ( long long iVal, short fWidth, bool lJust = false, 
                  bool sign = false, bool kibi = false, fiUnits units = fiK ) ;
bool  formatInt ( unsigned long long iVal, short fWidth, bool lJust = false, 
                  bool sign = false, bool kibi = false, fiUnits units = fiK ) ;

//* Return a const pointer to the array of wchar_t (wide) characters.          *
//*                                                                            *
//* Input  : none                                                              *
//* Returns: pointer to array of wchar_t characters                            *
const wchar_t* gstr ( void ) const ;

//* Return a const pointer to the array of wchar_t (wide) characters.          *
//*                                                                            *
//* Input  : charCount : (by reference, initial value ignored)                 *
//*                      receives number of characters in array,               *
//*                      including null terminator                             *
//* Returns: pointer to array of wchar_t characters                            *
const wchar_t* gstr ( short& charCount ) const ;

//* Return a const pointer to the array of UTF-8-encoded char characters.      *
//*                                                                            *
//* Input  : none                                                              *
//* Returns: pointer to array of const char characters                         *
const char* ustr ( void ) const ;

//* Return a const pointer to the array of UTF-8-encoded char characters.      *
//*                                                                            *
//* Input  : charCount:(by reference, initial value ignored)                   *
//*          receives number of characters in array, including null terminator *
//*        : byteCount:(by reference, initial value ignored)                   *
//*          receives number of data bytes in array                            *
//* Returns: pointer to array of const char characters                         *
const char* ustr ( short& charCount, short& byteCount ) const ;

//* Comparison operator: Compares the text content of two gString objects.     *
//* The comparison is performed against the objects' wchar_t character arrays, *
//* and is case-sensitive.                                                     *
//*                                                                            *
//* Returns: 'true' if the strings are identical, else 'false'                 *
bool operator == ( const gString& gs2 ) const ;

//* Comparison operator: Compares the text content of two gString objects.     *
//* The comparison is performed against the objects' wchar_t character arrays, *
//* and is case-sensitive.                                                     *
//*                                                                            *
//* Returns: 'true' if the strings are different, else 'false'                 *
bool operator != ( const gString& gs2 ) const ;

//* Compares the text content of the gString object with the specified text.   *
//* The comparison is performed against the gString object's wchar_t character *
//* array. The relationship between upper-case and lower-case characters       *
//* is locale dependent.                                                       *
//*                                                                            *
//* Input  : uStr     : (UTF-8 string) to be compared                          *
//*             OR                                                             *
//*          wStr     : (wchar_t string) to be compared                        *
//*          casesen  : (optional, 'true' by default)                          *
//*                     if 'true' perform case-sensitive comparison            *
//*                     if 'false' perform case-insensitive comparison         *
//*          length   : (optional, gsMAXCHARS by default. i.e. compare to end) *
//*                     maximum number of characters to compare                *
//*          offset   : (optional, ZERO by default)                            *
//*                     If specified, equals the character offset into the     *
//*                     gString character array at which to begin comparison.  *
//*                                                                            *
//* Returns: using the rules of the 'wcsncmp' (or 'wcsncasecmp') library       *
//*          function (see string.h):                                          *
//*     ZERO, text data are identical                                          *
//*   > ZERO, first differing char of gString object is numerically larger.    *
//*   < ZERO, first differing char of gString object is numerically smaller.   *
short compare ( const char* uStr, bool casesen = true, 
                short length = gsMAXCHARS, short offset = 0 ) const ;
short compare ( const wchar_t* wStr, bool casesen = true, 
                short length = gsMAXCHARS, short offset = 0 ) const ;

//* Compares the text content of two gString objects.                          *
//* The comparison is performed against the gString objects' wchar_t character *
//* arrays. The relationship between upper-case and lower-case characters      *
//* is locale dependent.                                                       *
//*                                                                            *
//* Input  : gs       : (by reference) object whose text is to be compared     *
//*          casesen  : (optional, 'true' by default)                          *
//*                     if 'true' perform case-sensitive comparison            *
//*                     if 'false' perform case-insensitive comparison         *
//*                                                                            *
//* Returns: using the rules of the 'wcsncmp' (or 'wcsncasecmp') library       *
//*          function (see string.h):                                          *
//*     ZERO, text data are identical                                          *
//*   > ZERO, first differing char of gString object is numerically larger.    *
//*   < ZERO, first differing char of gString object is numerically smaller.   *
short compare ( const gString& gs, bool casesen = true ) const ;

//* Scan the data for a matching substring and if found, return the index at   *
//* which the first substring match begins.                                    *
//*                                                                            *
//* Actual comparison is always performed against the 'wide' character array   *
//* using wcsncasecmp (or wcsncmp). Null terminator character IS NOT included  *
//* in the comparison. These comparisons ARE locale dependent.                 *
//*                                                                            *
//* Input  : src    : source data to be matched, one of the following:         *
//*                   -- pointer to a UTF-8 string (max length==gsMAXBYTES)    *
//*                   -- pointer to a wchar_t string (max length==gsMAXCHARS)  *
//*                   -- a gString object containing the source (by reference) *
//*                   -- a single, wchar_t character                           *
//*          offset : (optional, ZERO by default)                              *
//*                   character index at which to begin search                 *
//*                   -- if out-of-range, then same as if not specified        *
//*          casesen: (optional, 'false' by default)                           *
//*                   if 'false', then scan IS NOT case sensitive              *
//*                   if 'true, then scan IS case sensitive                    *
//*                   The way upper/lowercase are related is locale dependent; *
//*          maxcmp : (optional, (-1) by default)                              *
//*                   -- if not specified, then scan for a match of all        *
//*                      characters in 'src' (not including null terminator)   *
//*                   -- if specified, then scan for a match of only the first *
//*                      'maxcmp' characters of 'src'                          *
//*                   -- if out-of-range, then same as if not specified        *
//* Returns: index of matching substring or (-1) if no match found             *
//*          Note: This is the wchar_t character index, NOT a byte index       *
short find ( const char* src, short offset = 0, 
             bool casesen = false, short maxcmp = -1 ) const ;
short find ( const wchar_t* src, short offset = 0, 
             bool casesen = false, short maxcmp = -1 ) const ;
short find ( const gString& src, short offset = 0, 
             bool casesen = false, short maxcmp = -1 ) const ;
short find ( const wchar_t src, short offset = 0, bool casesen = false ) const ;

//* Scan the data for the last occurance of the matching substring and if      *
//* found, return the index at which the substring match occurs.               *
//*                                                                            *
//* Actual comparison is always performed against the 'wide' character array   *
//* using wcsncasecmp (or wcsncmp). Null terminator character IS NOT included  *
//* in the comparison. These comparisons ARE locale dependent.                 *
//*                                                                            *
//* Input  : src    : source data to be matched, one of the following:         *
//*                   -- pointer to a UTF-8 string (max length==gsMAXBYTES)    *
//*                   -- pointer to a wchar_t string (max length==gsMAXCHARS)  *
//*                   -- a gString object containing the source (by reference) *
//*                   -- a single, wchar_t character                           *
//*          casesen: (optional, 'false' by default)                           *
//*                   if 'false', then scan IS NOT case sensitive              *
//*                   if 'true, then scan IS case sensitive                    *
//*                   The way upper/lowercase are related is locale dependent; *
//* Returns: index of last matching substring or (-1) if no match found        *
//*          Note: This is the wchar_t character index, NOT a byte index       *
short findlast ( const char* src, bool casesen = false ) const ;
short findlast ( const wchar_t* src, bool casesen = false ) const ;
short findlast ( const gString& src, bool casesen = false ) const ;
short findlast ( const wchar_t src, bool casesen = false ) const ;

//* This method is very similar to the 'find()' method above, but instead of   *
//* returning the index to the beginning of the substring, returns the index   *
//* of the character which FOLLOWS the substring.                              *
//*                                                                            *
//* Input  : src    : source data to be matched, one of the following:         *
//*                   -- pointer to a UTF-8 string (max length==gsMAXBYTES)    *
//*                   -- pointer to a wchar_t string (max length==gsMAXCHARS)  *
//*                   -- a gString object containing the source (by reference) *
//*                   -- a single, wchar_t character                           *
//*          offset : (optional, ZERO by default)                              *
//*                   character index at which to begin search                 *
//*                   -- if out-of-range, then same as if not specified        *
//*          casesen: (optional, 'false' by default)                           *
//*                   if 'false', then scan IS NOT case sensitive              *
//*                   if 'true, then scan IS case sensitive                    *
//*                   The way upper/lowercase are related is locale dependent; *
//* Returns: index of the character which follows the matching substring       *
//*            or (-1) if no match found                                       *
//*          Note: This is the wchar_t character index, NOT a byte index.      *
//*          Note: Index will never reference data beyond the NULL terminator. *
short after ( const char* src, short offset = 0, bool casesen = false ) const ;
short after ( const wchar_t* src, short offset = 0, bool casesen = false ) const ;
short after ( const gString& src, short offset = 0, bool casesen = false ) const ;
short after ( const wchar_t src, short offset = 0, bool casesen = false ) const ;

//* Scan the text and locate the first character which DOES NOT match the      *
//* specified character. This method is used primarily to scan past the end of *
//* a sequence of space (' ') (20 hex) characters, but may be used to skip     *
//* over any sequence of a single, repeated character.                         *
//*                                                                            *
//* See also the 'scan' method which scans to the first non-whitespace         *
//* character.                                                                 *
//*                                                                            *
//* Input  : srcChar: (optional, L' ' by default) character to be skipped over *
//*          offset : (optional, ZERO by default)                              *
//*                   if specified, equals the character offset into the       *
//*                   character array at which to begin the scan.              *
//*                                                                            *
//* Returns: a) If successful, returns the index of first character which      *
//*             DOES NOT match specified character.                            *
//*          b) If the scan finds no character which is different from the     *
//*             specified character, OR if 'offset' is out of range, OR if the *
//*             specified character is the null character, the return value    *
//*             indexes the null terminator of the array.                      *
//*          Note: This is the wchar_t character index, NOT a byte index       *
short findx ( wchar_t srcChar = L' ', short offset = 0 ) const ;

//* Scans the text array and returns the index of the first non-whitespace     *
//* character found.                                                           *
//* Whitespace characters are:                                                 *
//*    0x20   single-column ASCII space                                        *
//*    0x3000 two-column CJK space                                             *
//*    0x0A   linefeed character                                               *
//*    0x0D   carriage-return character                                        *
//*    0x09   tab character                                                    *
//*    0x0B   vertical-tab character                                           *
//*    0x0C   formfeed character                                               *
//*                                                                            *
//* Input  : offset : (optional, ZERO by default)                              *
//*                   if specified, equals the character offset into the       *
//*                   character array at which to begin the scan.              *
//*                                                                            *
//* Returns: a) If successful, returns the index of first non-whitespace       *
//*             character                                                      *
//*          b) If the scan finds no non-whitespace character OR if 'offset'   *
//*             is out of range, the return value indexes the null terminator  *
//*             of the array.                                                  *
//*          Note: This is the wchar_t character index, NOT a byte index       *
short scan ( short offset = 0 ) const ;

//* Scan the data for a matching substring, and if found, erase (delete) the   *
//* first occurance of the substring.                                          *
//*                                                                            *
//* Actual comparison is always performed against the 'wide' character array   *
//* using wcsncasecmp (or wcsncmp). Null terminator character IS NOT included  *
//* in the comparison. These comparisons ARE locale dependent.                 *
//*                                                                            *
//* Input  : src    : source data to be matched, one of the following:         *
//*                   -- pointer to a UTF-8 string (max length==gsMAXBYTES)    *
//*                   -- pointer to a wchar_t string (max length==gsMAXCHARS)  *
//*                   -- a gString object containing the source (by reference) *
//*                   -- a single, wchar_t character                           *
//*          offset : (optional, ZERO by default)                              *
//*                   character index at which to begin search                 *
//*                   NOTE: This is a character index, NOT a byte offset.      *
//*          casesen: (optional, 'false' by default)                           *
//*                   if 'false', then scan IS NOT case sensitive              *
//*                   if 'true, then scan IS case sensitive                    *
//*                   The way upper/lowercase are related is locale dependent; *
//* Returns: index of first character following the deleted sequence           *
//*          Note: This is the wchar_t character index, NOT a byte index       *
//*          Returns (-1) if:                                                  *
//*            a) no matching substring found                                  *
//*            b) 'offset' out-of-range                                        *
//*            c) 'src' is an empty string or a NULL character                 *
short erase ( const gString& src, short offset = 0, bool casesen = false ) ;
short erase ( const wchar_t* src, short offset = 0, bool casesen = false ) ;
short erase ( const char* src, short offset = 0, bool casesen = false ) ;
short erase ( const wchar_t src, short offset = 0, bool casesen = false ) ;

//* Erase (delete) the data sequence specified by 'offset' and 'length'.       *
//* 'offset' is the index of the first character to be deleted, and 'length'   *
//* specifies the number of characters to delete.                              *
//*   1) If the 'length' specifies deletion of more characters than remain,    *
//*      then the 'erase' method has the same effect as calling the            *
//*      'limitChars' method.                                                  *
//*   2) If the defaults for both 'offset' and length' are used, then the      *
//*      'erase' method has the same effect as calling the 'clear' method.     *
//*   3) Note that the NULL terminator will never be deleted.                  *
//*   4) If:  a) offset < ZERO,                                                *
//*           b) offset >= number of characters,                               *
//*           c) length <= ZERO                                                *
//*      then data will not be modified and (-1) will be returned.             *
//*                                                                            *
//* Input  : offset : (optional, ZERO by default)                              *
//*                   index of first character of sequence to be erased        *
//*                   NOTE: This is a character index, NOT a byte offset.      *
//*          length : (optional, gsMAXCHARS by default)                        *
//*                   if not specified, then erase all characters from         *
//*                      'offset' to end of data                               *
//*                   if specified, then erase the specified number of         *
//*                      characters beginning at 'offset'                      *
//* Returns: index of first character following the deleted sequence           *
//*          Note: This is the wchar_t character index, NOT a byte index       *
//*          Returns (-1) if parameter out-of-range (data not modified)        *
//* -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  *
short erase ( short offset = 0, short length = gsMAXCHARS ) ;
   // Programmer's Note: The stubs with 'int' arguments, below, are included
   // to silence a compiler warning that erase(short, short) is 'ambiguous'.
   // Actually, it is not ambiguous at all, but we believe that the use of 
   // pragmas is bad design, so it's easier to include the stubs than to 
   // argue with the compiler. Internally, gString uses 'short int' exclusively.
   short erase ( int offset = 0, int length = gsMAXCHARS ) ;
   short erase ( short offset = 0, int length = gsMAXCHARS ) ;
   short erase ( int offset = 0, short length = gsMAXCHARS ) ;

//* Replace the specified source substring with the provided substring.        *
//*                                                                            *
//* Actual comparison is always performed against the 'wide' character array   *
//* using wcsncasecmp (or wcsncmp). Null terminator character IS NOT included  *
//* in the comparison. These comparisons ARE locale dependent.                 *
//*                                                                            *
//* Input  : src    : source data to be matched, one of the following:         *
//*                   -- pointer to a UTF-8 string                             *
//*                   -- pointer to a wchar_t string                           *
//*                   -- a single, wchar_t character                           *
//*          newtxt : data to overwrite existing text, one of the following:   *
//*                   -- pointer to a UTF-8 string                             *
//*                   -- pointer to a wchar_t string                           *
//*                   -- a single, wchar_t character                           *
//*          offset : (optional, ZERO by default)                              *
//*                   character index at which to begin search                 *
//*                   NOTE: This is a character index, NOT a byte offset.      *
//*          casesen: (optional, 'false' by default)                           *
//*                   if 'false', then scan IS NOT case sensitive              *
//*                   if 'true, then scan IS case sensitive                    *
//*                   The way upper/lowercase are related is locale dependent; *
//*          all    : (optional, 'false' by default)                           *
//*                   if 'false', then replace only the first occurance found  *
//*                   if 'true',  then replace all occurances of the specified *
//*                               substring                                    *
//* Returns: 'true' if successful                                              *
//*          returns 'false' if error (existing data not modified):            *
//*            a) no matching source substring found                           *
//*            b) 'src' is a empty string or a null character                  *
//*            c) offset < ZERO or offset is beyond existing data              *
//*            d) modifying the data would cause buffer overflow (>gsMAXCHARS) *
bool replace ( const wchar_t* src, const wchar_t* newtxt, short offset = 0, 
               bool casesen = false, bool all = false ) ;
bool replace ( const wchar_t* src, const char* newtxt, short offset = 0,
               bool casesen = false, bool all = false ) ;
bool replace ( const wchar_t* src, const wchar_t newtxt, short offset = 0,
               bool casesen = false, bool all = false ) ;
bool replace ( const char* src, const wchar_t* newtxt, short offset = 0,
               bool casesen = false, bool all = false ) ;
bool replace ( const char* src, const char* newtxt, short offset = 0,
               bool casesen = false, bool all = false ) ;
bool replace ( const char* src, const wchar_t newtxt, short offset = 0,
               bool casesen = false, bool all = false ) ;
bool replace ( const wchar_t src, const wchar_t* newtxt, short offset = 0,
               bool casesen = false, bool all = false ) ;
bool replace ( const wchar_t src, const char* newtxt, short offset = 0,
               bool casesen = false, bool all = false ) ;
bool replace ( const wchar_t src, const wchar_t newtxt, short offset = 0,
               bool casesen = false, bool all = false ) ;

//* Strip leading and/or trailing whitespace.                                  *
//* Recognizes both the ASCII space 0x20 and the CJK (two-column) space 0x3000 *
//* as well as TAB (0x09), VTAB (0x0B), Newline (0x0A), Carriage-return (0x0D),*
//* and Form-feed (0x0C).                                                      *
//*                                                                            *
//* Input  : leading  : (optional, 'true' by default)                          *
//*                     If 'true', remove whitespace from beginning of string. *
//*          trailing : (optional, 'true' by default)                          *
//*                     If 'true', remove whitespace from end of string.       *
//* Returns: number of characters in modified string (incl. NULL terminator)   *
short strip ( bool leading = true, bool trailing = true ) ;

//* Copy gString text to specified target buffer. (source data unchanged)      *
//*                                                                            *
//* Input  : uTarget : pointer to target array to receive UTF-8-encoded text   *
//*          maxBytes: maximum number of bytes to copy (incl. NULL terminator) *
//*          maxCols : (optional, default == gsMAXCHARS*2)                     *
//*                    maximum number of display-columns to copy               *
//* Returns: number of bytes copied (incl. NULL terminator)                    *
short copy ( char* uTarget, short maxBytes, short maxCols=(gsMAXCHARS*2) ) const ;

//* Copy gString text to specified target buffer. (source data unchanged)      *
//*                                                                            *
//* Input  : wTarget : pointer to target array to receive wchar_t 'wide' text  *
//*          maxChars: maximum number of characters to copy (incl. NULL)       *
//*          maxCols : (optional, default == gsMAXCHARS*2)                     *
//*                    maximum number of display-columns to copy               *
//* Returns: number of characters copied (incl. NULL terminator)               *
short copy ( wchar_t* wTarget, short maxChars, short maxCols=(gsMAXCHARS*2) ) const ;

//* Insertion operator: Insert the contents of the gString object into the     *
//* standard output stream.                                                    *
//* Overloaded operator allows access to the 'wcout' (wide) standard output    *
//* stream OR access to the 'cout' (narrow) standard output stream.            *
//* IMPORTANT NOTE: It is recommended that the 'wcout' (wide) stream version   *
//*                 be used exclusively.                                       *
//*                      !! NON-MEMBER METHOD !!                               *
//*                                                                            *
//* Input  : implied reference to output stream                                *
//*          implied reference to gString object                               *
//* Returns: reference to the specified output stream                          *
friend std::wostream& operator << ( std::wostream& os, const gString& gsref ) ;
friend std::ostream& operator << ( std::ostream& os, const gString& gsref ) ;

//* Copy the specified character range to specified target buffer.             *
//*                                                                            *
//* If target buffer is a char*, then data returned is a UTF-8 text string.    *
//* If target buffer is a wchar_t*, then data returned is a wchar_t (wide)     *
//*   text string.                                                             *
//*   IMPORTANT NOTE: It is the caller's responsibility to specify             *
//*   a target buffer large enough to hold the data. Recommended:              *
//*          wchar_t wbuff[gsMAXCHARS]; or char ubuff[gsMAXBYTES];             *
//* If target buffer is a gString object, then both UTF-8 and wchar_t data     *
//*   are returned (with no chance of target buffer overflow).                 *
//*                                                                            *
//* Input  : targ    : (by reference, initial contents ignored)                *
//*                    receives null-terminated contents of specified          *
//*                    character range                                         *
//*          offset  : character index at which substring begins               *
//*                    (this IS NOT a byte index)                              *
//*          charCnt : number of characters to copy (not incl. NULL terminator)*
//* Returns: if target is a wchar_t* or gString object, then returns number of *
//*            characters written to target (not including the NULL terminator)*
//*          if target is a char*, then returns number of bytes written to     *
//*            target (not including the NULL terminator)                      *
//*                                                                            *
//*          Note: returns ZERO if either 'offset' or 'charCnt' out of range   *
//*              Note that if 'charCnt' extends beyond the end of the source   *
//*              data, then the available data are returned.                   *
short substr ( char* uTarget, short offset, short charCnt ) const ;
short substr ( wchar_t* wTarget, short offset, short charCnt ) const ;
short substr ( gString& wTarget, short offset, short charCnt ) const ;


//* Append text to existing gString text data up to a combined length of       *
//* gsMAXCHARS. Characters in excess of the maximum will not be appended.      *
//*                                                                            *
//* Input  : wPtr: pointer to array of wchar_t 'wide' text to be appended      *
//*           OR                                                               *
//*          uPtr: pointer to array of char UTF-8 text to be appended          *
//*           OR                                                               *
//*          wChar: a single, 'wide' character                                 *
//* Returns: number of characters in resulting string (incl. NULL terminator)  *
//*          Note: if return == gsMAXCHARS, some data MAY HAVE BEEN discarded. *
short append ( const wchar_t* wPtr ) ;
short append ( const char* uPtr ) ;
short append ( const wchar_t wChar ) ;

//* Append formatted text data to existing gString text data up to a combined  *
//* length of gsMAXCHARS. Characters in excess of the maxmum will not be       *
//* appended.                                                                  *
//*                                                                            *
//* Please refer to compose() method for more information on converting data   *
//* using a format-specification string.                                       *
//*                                                                            *
//* Input  : fmt  : a format specification string in the style of sprintf(),   *
//*                 swprintf() and related formatting C/C++ functions.         *
//*          arg1 : pointer to first value to be converted by 'fmt'            *
//*          ...  : optional arguments (between ZERO and gsfMAXARGS - 1)       *
//*                 Each optional argument is a POINTER (address of) the value *
//*                 to be formatted.                                           *
//*                                                                            *
//* Returns: number of characters in resulting string (incl. NULL terminator)  *
//*          Note: if return == gsMAXCHARS, some data MAY HAVE BEEN discarded. *
short append ( const wchar_t* fmt, const void* arg1, ... ) 
               /*__attribute__ ((format (gnu_wprintf, 2, 0)))*/ ;
short append ( const char* fmt, const void* arg1, ... ) 
               __attribute__ ((format (gnu_printf, 2, 0))) ;

//* Insert text into existing gString text data up to a combined length of     *
//* gsMAXCHARS. Characters in excess of the maximum will be truncated.         *
//*                                                                            *
//* Input  : wPtr  : pointer to array of wchar_t 'wide' text to be inserted    *
//*           OR                                                               *
//*          uPtr  : pointer to array of char UTF-8 text to be inserted        *
//*           OR                                                               *
//*          wChar : a single wchar_t 'wide' character                         *
//*          offset: (optional, ZERO by default)                               *
//*                  character offset at which to insert specified text into   *
//*                  existing text.                                            *
//*                  Note: if specified 'offset' > number of characters in     *
//*                        existing text, then acts like 'append' method.      *
//* Returns: number of characters in resulting string (incl. NULL terminator)  *
//*          Note: if return == gsMAXCHARS, some data MAY HAVE BEEN discarded. *
short insert ( const wchar_t* wPtr, short offset = 0 ) ;
short insert ( const char* uPtr, short offset = 0 ) ;
short insert ( wchar_t wChar, short offset = 0 ) ;

//* Load the specified number of characters from the source data.              *
//* By default, the new text REPLACES the existing text; however, the 'append' *
//* parameter allows the new text to be appended to the existing text.         *
//*                                                                            *
//* Input  : usrc     : pointer to a UTF-8 encoded string                      *
//*                     OR                                                     *
//*          wsrc     : pointer to a wchar_t encoded string                    *
//*          charLimit: maximum number of characters (not bytes) from source   *
//*                     array to load. Range: 1 through gsMAXCHARS.            *
//*                     The count should not include the NULL terminator       *
//*          append   : (optional, 'false' by default)                         *
//*                     if 'false', replace existing text with specified text  *
//*                     if 'true', append new text to existing text            *
//* Returns: number of characters in modified string (incl. NULL terminator)   *
short loadChars ( const wchar_t* wsrc, short charLimit, bool append = false ) ;
short loadChars ( const char* usrc, short charLimit, bool append = false ) ;

//* Shift text data by the specified number of characters.                     *
//*                                                                            *
//* Input  : shiftCount: < ZERO: shift data to the left, discarding the        *
//*                              specified number of characters from the       *
//*                              beginning of the array                        *
//*                      > ZERO: shift data to the right, padding the vacated  *
//*                              positions on the left with 'padChar'          *
//*                      ==ZERO: do nothing                                    *
//*          padChar   : (optional, SPACE character, 0x20 by default)          *
//*                      when shifting data to the right, use this character   *
//*                      to fill the vacated character positions               *
//* Returns: number of characters in adjusted array                            *
short shiftChars ( short shiftCount, wchar_t padChar = L' ' ) ;

//* Shift text data by the specified number of display columns.                *
//*                                                                            *
//* Input  : shiftCount: < ZERO: shift data to the left, discarding the        *
//*                              number of characters equivalent to the        *
//*                              specified number of display columns           *
//*                              NOTE: May discard one extra column if count   *
//*                              falls within a multi-column character.        *
//*                      > ZERO: shift data to the right, padding the vacated  *
//*                              positions on the left with 'padChar'          *
//*                      ==ZERO: do nothing                                    *
//*          padChar   : (optional, SPACE character, 0x20 by default)          *
//*                      when shifting data to the right, use this character   *
//*                      to fill the vacated column positions                  *
//*                      NOTE: Specify a one-column character ONLY as the      *
//*                      padding character. (multi-column characters ignored)  *
//* Returns: number of columns in adjusted array                               *
short shiftCols ( short shiftCount, wchar_t padChar = L' ' ) ;

//* Add padding to the end of the existing data to equal the specified number  *
//* of columns.                                                                *
//* a) Padding will be added until either the specified number of columns is   *
//*    reached, OR until the array contains the maximum number of characters   *
//*    (gsMAXCHARS).                                                           *
//* b) If 'fieldWidth' <= current width, then data will not be modified.       *
//*                                                                            *
//* Input  : fieldWidth: number of columns for the adjusted data array         *
//*          padChar   : (optional, SPACE character, 0x20 by default)          *
//*                      character with which to pad the data                  *
//*                      NOTE: Specify a one-column character ONLY as the      *
//*                      padding character. (multi-column characters ignored)  *
//* Returns: number of columns in adjusted array                               *
short padCols ( short fieldWidth, wchar_t padChar = L' ' ) ;

//* Returns the number of characters in the string including null terminator   *
short gschars ( void ) const ;

//* Returns the number of columns required to display the string.              *
short gscols ( void ) const ;

//* Returns a pointer to an array of column counts, one for each character     *
//* of the text data. charCount gets number of characters in display string.  .*
const short* gscols ( short& charCount ) const ;

//* Returns the number of bytes in the UTF-8-encoded string (incl. null)       *
short utfbytes ( void ) const ;

//* Returns 'true' if data are pure 7-bit ASCII, 'false' otherwise.            *
bool  isASCII ( void ) ;

//* Truncate the existing data to no more than charCount display characters.   *
//* Insert a null terminator after the specified number of characters          *
//*                                                                            *
//* Input  : maximum number of characters allowed in formatted data            *
//*          (not including NULL) Range: 1 to gsMAXCHARS-1                     *
//*                                                                            *
//* Returns: number of characters in the adjusted data (including NULL)        *
short limitChars ( short charCount ) ;

//* Truncate the existing data to no more than colCount display columns.       *
//* Insert a null terminator after the number of characters required to fill   *
//* the specified number of display columns. Range: 1 to (gsMAXCHARS * 2).     *
//*                                                                            *
//* Input  : maximum number of display columns available for data display      *
//*                                                                            *
//* Returns: number of columns needed to display the adjusted data             *
short limitCols ( short colCount ) ;

//* Reset contents to an empty string i.e. "". The data will consist of a      *
//* single, NULLCHAR character. The character and byte counts are set to       *
//* 1 (one), and the column count is zero.                                     *
void  clear ( void ) ;

//* Returns a pointer to version number string for the gString class.          *
const char* Get_gString_Version ( void ) const ;

#if ENABLE_GS_DEBUG != 0
void dbMsg ( gString& gsmsg ) ;  // for debugging only
#endif   // ENABLE_GS_DEBUG


private:
void  ConvertUTF8toWide ( const char* usrc, short charLimit=gsMAXCHARS ) ;
void  Bytewise_u2w ( const char* usrc, short charLimit ) ;
void  ConvertWidetoUTF8 ( const wchar_t* wsrc=NULL, short charLimit=gsMAXCHARS ) ;
void  Bytewise_w2u ( short charLimit ) ;
void  Convert_gsForm2gs ( const gsForm& gsf, wchar_t wsrc[gsMAXCHARS] ) ;
void  limitBytes ( short maxBytes ) ;
void  ColumnCount ( short maxCols=(gsMAXCHARS * 2) ) ;
void  Reinit ( bool initAll=true ) ;
void  binaryFormat ( wchar_t* convBuff, wchar_t convType, wchar_t sepType, 
                     long long int intData, short dataType, short grpType, 
                     bool append ) ;
bool  formatQInt ( unsigned long long iVal, short fWidth, bool lJust, 
                   bool sVal, bool sign, bool kibi, fiUnits units ) ;
bool  fqiRound ( gString& gt, short fWidth ) ;
bool  fqiThreeColumn ( gString&gt, gString& suffix, double dblVal, short vRange,
                       bool sVal, bool sign, bool kibi ) ;
const wchar_t* fqiAppendSuffix ( short& vRange, double dblVal, fiUnits units, bool sVal ) ; 
bool  fqiBumpSuffix ( wchar_t* wsrc, gString& gtSuffix ) ;
bool  replace ( const gString& src, const gString& newtxt, 
                short offset, bool casesen, bool all ) ;

short composeFieldwidthBugFix ( char convType, wchar_t* wsrcPtr, short wsrcSpace, 
                                const wchar_t* fmt, const gString& gsSrc ) ;

wchar_t  gStr[gsMAXCHARS] ;   // text data in wchar_t 'wide' format
char     uStr[gsMAXBYTES] ;   // text data in UTF-8 encoded format
short    cWidth[gsMAXCHARS] ; // column width for each character in gStr
short    gsCols ;             // number of columns needed to display gStr
short    gsChars ;            // number of characters represented (incl. null)
short    utfBytes ;           // number of bytes in UTF-8 data (incl. null)
bool     isAscii ;            // 'true' if data are pure, 7-bit ASCII, else 'false'
// Note that isAscii is unitialized by default and is valid only after 
// a call to the isASCII() method
} ;

#endif   // GSTRING_INCLUDED


