//******************************************************************************
//* File       : Chart.cpp                                                     *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2020-2021 Mahlon R. Smith, The Software Samurai *
//*                 GNU GPL copyright notice below                             *
//* Date       : 01-Jul-2021                                                   *
//* Version    : (see ChartVersion string)                                     *
//*                                                                            *
//* Description: Implementation of the "Chart" class.                          *
//*              This is a widget built on the NcDialog API by the same author.*
//*              Draw a bar chart representing the specified data.             *
//*                                                                            *
//******************************************************************************
//* Copyright Notice:                                                          *
//* This program is free software: you can redistribute it and/or modify it    *
//* under the terms of the GNU General Public License as published by the Free *
//* Software Foundation, either version 3 of the License, or (at your option)  *
//* any later version.                                                         *
//*                                                                            *
//* This program is distributed in the hope that it will be useful, but        *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
//* for more details.                                                          *
//*                                                                            *
//* You should have received a copy of the GNU General Public License along    *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.            *
//*                                                                            *
//*         Full text of the GPL License may be found in the Texinfo           *
//*         documentation for this program under 'Copyright Notice'.           *
//******************************************************************************
//* Version History (most recent first):                                       *
//*                                                                            *
//* v: 0.0.01 01-Mar-2021                                                      *
//*   -- First effort is an extension and generalization of chart generation   *
//*      code in an early release of Exercalc, an exercise tracking application*
//*      by the same author.                                                   *
//*   -- Development of the Chart class is assisted by the Exercalc method     *
//*      "DebugChartWidget()". This method will be integrated into a future    *
//*      version of one of the NcDialog test applications.                     *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//******************************************************************************
//* Notes:                                                                     *
//* ------                                                                     *
//* Arithmetic Mean:                                                           *
//* The arithmetic mean value is the sum of all elements of a data set,        *
//* divided by the number of elements in the set.                              *
//*   (Wikipedia)                                                              *
//*                                                                            *
//* Median Value:                                                              *
//* The median is the middle number of the group when they are ranked in order.*
//* (If there are an even number of numbers, the mean of the middle two is     *
//* taken.)                                                                    *
//* Thus to find the median, order the list according to its elements'         *
//* magnitude and then repeatedly remove the pair consisting of the highest    *
//* and lowest values until either one or two values are left. If exactly one  *
//* value is left, it is the median; if two values, the median is the          *
//* arithmetic mean of these two.                                              *
//*   (Wikipedia)                                                              *
//*                                                                            *
//* The median is sometimes used as opposed to the mean when there are         *
//* outliers in the sequence that might skew the average of the values.        *
//* The median of a sequence can be less affected by outliers than the mean.   *
//*                                                                            *
//*                                                                            *
//******************************************************************************
//****************
//* Header Files *
//****************
#include "Chart.hpp"          // class definition and support data

//***************
//* Definitions *
//***************

//* For debugging only: enable display of config data *
#define DEBUG_CONFIG (1)
#if DEBUG_CONFIG != 0
#define DEBUG_SHIFT (0)    // Display shift offset
static winPos wpShift ;
#endif   // DEBUG_CONFIG

//* Class version number *
static const char* const ChartVersion = "0.0.01" ;

//* Percentage of chart area for display of minimum and maximum values *
static const double minBAR = 2.0 ;  // number of divisions displayed for minimum value

//* For Cartesian charts only: scaling multiplier for X coordinate used *
//* to make the aspect ratio of the plot appear (approximately) square. *
static const double xCONST = 1.5 ;

//***************
//* Prototypes  *
//***************
static short InvertFractionalCell ( short blockIndex ) ;
static attr_t InvertColorAttribute ( attr_t normColor ) ;

//***************
//* Local Data  *
//***************

//* Character definitions for drawing graphs *
static const short wcINDX = 7 ;        // index of whole-cell character
static const wchar_t hBlock[cellDIV] = // for horizontal bar graphs
{
   0x0258F,    // ▏
   0x0258E,    // ▎
   0x0258D,    // ▍
   0x0258C,    // ▌
   0x0258B,    // ▋
   0x0258A,    // ▊
   0x02589,    // ▉
   0x02588,    // █
} ;
static const wchar_t vBlock[cellDIV] = // for vertical bar graphs
{
   0x02581,    // ▁
   0x02582,    // ▂
   0x02583,    // ▃
   0x02584,    // ▄
   0x02585,    // ▅
   0x02586,    // ▆
   0x02587,    // ▇
   0x02588,    // █
} ;

//* Character used when full bar is not displayed, only the terminal point.*
static const wchar_t hTipChar = 0x025AE ;  // character '▮' for horizontal bars
static const wchar_t vTipChar = 0x025AC ;  // character '▬' for vertical bars


//*************************
//*        ~Chart         *
//*************************
//******************************************************************************
//* Destructor. Return all resources to the system.                            *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

Chart::~Chart ( void )
{
   //* If dynamic allocation of a data array to hold   *
   //* converted data, return the memory to the system.*
   if ( this->dynoData && (this->dataPtr != NULL) )
   { delete [] this->dataPtr ; this->dataPtr = NULL ; this->dynoData = false ; }

   //* If locally-instantiated dialog, delete it now. *
   if ( this->localDlg && this->dp != NULL )
   { delete this->dp ; this->dp = NULL ; }

}  //* End ~Chart() *

//*************************
//*         Chart         *
//*************************
//******************************************************************************
//* Full-initialization constructor. All data members set according to         *
//* provided parameters.                                                       *
//*                                                                            *
//* Input  : cdef   : (by reference) a fully-initialized instance of the       *
//*                   'chartDef' class.                                        *
//*          refresh: (optional, 'false' by default)                           *
//*                   if 'false', setup is performed, but display not updated  *
//*                   if 'true',  after setup, update display immediately      *
//*                                                                            *
//* Returns: implicitly returns a pointer to instantiated object               *
//******************************************************************************

Chart::Chart ( const chartDef& cdef, bool refresh )
{
   //* Configure all data members *
   this->Configure ( cdef ) ;

   bool isOpen = false ;            // 'true' if target window is open

   if ( this->localDlg && this->dp == NULL ) // if we are to open a dialog locally
   {
      InitNcDialog id
         {
            this->ldRows,              // dialog rows (expanded to include borders)
            this->ldCols,              // dialog columns (expanded to include borders)
            this->wpTerm.ypos,         // dialog position Y
            this->wpTerm.xpos,         // dialog position X
            this->titText,             // title (if any)
            this->bStyle,              // border style
            this->bdrColor,            // border color
            this->dColor,              // interior color
            NULL                       // no controls
         } ;
      this->dp = new NcDialog ( id ) ;
      if ( (this->dp->OpenWindow ()) == OK )
         isOpen = true ;
   }
   else                             // if caller has provided a dialog
   {
      //* Defensive programming: test whether dialog is accessible *
      short py = -1, px = -1 ;
      this->dp->GetDialogDimensions ( py, px) ;
      if ( py >= ZERO && px >= ZERO ) 
         isOpen = true ;
   }

   //* If we have a valid target window *
   if ( isOpen )
   {
      //* If specified, set the dialog title *
      if ( this->titText != NULL )
         this->dp->SetDialogTitle ( this->titText, this->titColor ) ;

      //* Clear the display area,and if specified, draw a border.*
      this->ClearArea () ;

      //* Draw the static text (if any) *
      if ( this->hdrText != NULL )
         this->DrawHeaderText ( this->hdrText, this->bColor ) ;
      if ( this->ftrText != NULL )
         this->DrawFooterText ( this->ftrText, this->bColor ) ;

      //* Draw the vertical and horizontal axes *
      //* and axis labels (if provided).        *
      this->DrawGrid ( true ) ;

      //* Display the data *
      if ( (this->dataPtr != NULL) && (this->dataCount > ZERO) )
         this->MapData2Grid () ;

      #if DEBUG_CONFIG != 0      // (assumes space in caller's dialog window)
      // UNDER CONSTRUCTION - When moved to NcDialog test, we can use the same
      // Y offset for both parent and child window.
      //* Positioned two(2) lines below the footer text position.*
      winPos wpDump( this->wpBase.ypos + this->dispRows - this->yFooter 
                     + (this->localDlg ? 4 : 3), 
                     this->wpBase.xpos + 1 ) ;
      this->DumpCfg ( wpDump ) ;
      #endif   // DEBUG_CONFIG

      //* If specified, make everything visible *
      if ( refresh )
         this->dp->RefreshWin () ;
   }

}  //* End Chart() *

//*************************
//*         reset         *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* Reset all data members to default values.                                  *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Chart::reset ( void )
{
   this->dp = NULL ;
   this->wpTerm.ypos = this->wpTerm.xpos = ZERO ;
   this->wpBase.ypos = this->wpBase.xpos = ZERO ;
   this->wpOrig.ypos = this->wpOrig.xpos = ZERO ;
   this->wpYlab.ypos = this->wpYlab.xpos = ZERO ;
   this->wpXlab.ypos = this->wpXlab.xpos = ZERO ;
   this->jYlab = this->jXlab = tjLeft ;
   this->dispRows = this->dispCols = ZERO ;
   this->gridRows = this->gridCols = ZERO ;
   this->yOffset  = 1 ;             // default offset from top of area to top of grid
   this->xOffset  = 1 ;             // default freespace above grid (including line for axis label)
   this->yFooter  = 1 ;             // default freespace below grid (including line for axis label)
   this->barWidth = cellDIV ;       // one full cell
   this->barSpace = ZERO ;          // no space between bars
   this->chType   = ctLowerLeft ;   // default chart type
   this->vCells   = this->hCells = ZERO ;
   this->bdrColor = this->titColor = this->barColor = this->negColor = 
   this->dColor   = this->bColor   = this->gColor   = ZERO ;
   this->dataCount = ZERO ;
   this->dataOffset = ZERO ;
   this->dataPtr  = NULL ;
   this->attrPtr  = NULL ;
   this->titText = this->hdrText = this->ftrText = NULL ;
   this->vLabel  = this->hLabel  = NULL ;
   this->bStyle  = ncltSINGLE ;     // single-line border style
   this->gStyle  = ncltSINGLE ;     // single-line grid style

   this->minVal = this->maxVal = this->rngVal = this->meaVal = this->medVal = 0.0 ;
   this->verDiv = this->horDiv = this->minDiv = this->maxDiv = this->rngDiv = 0.0 ;
   this->perDiv = 0.0 ;
   this->cartRng.minValX = this->cartRng.maxValX = 
   this->cartRng.minValY = this->cartRng.maxValY = 0.0 ;
   this->vaxisChar = wcsLTEEs ;     // default: L'├'
   this->haxisChar = wcsBTEEs ;     // default: L'┴'
   this->axisCross = wcsLLs ;       // default: L'└'
   this->axisCap  = wcsVERTs ;      // default: L'│' (for vertical axis)
   this->cartChar  = dblDiamond ;   // default: Cartesian datapoint marker
   this->dynoData  = false ;        // no dynamic allocation
   this->localDlg  = false ;        // assume caller's dialog
   this->bdrFlag   = false ;        // assume no border
   this->hBars     = false ;        // assume vertical bars
   this->bTips     = false ;        // display entire bar (not just tip)
   this->audible   = true ;         // audible shift alert enabled by default

}  //* End reset() *

//*************************
//*         Chart         *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* Default constructor. All data members set to default values.               *
//* This method is defined as private to prevent the calling application from  *
//* embarrassment resulting from an incomplete setup sequence.                 *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: implicitly returns a pointer to instantiated object               *
//******************************************************************************

Chart::Chart ( void )
{

   this->reset () ;           // initialize the data members

}  //* End Chart() *

//*************************
//*       Configure       *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* 1) Calculate dimensions, layout and color scheme of display area.          *
//* 2) Calculate the number of character Y/X character cells for the grid.     *
//* 3) Scan the data for max, min, median, average.                            *
//* 4) Size the data layout to fit the grid matrix.                            *
//*                                                                            *
//* Input  : cdef  : (by reference) a fully-initialized definition-class object*
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Chart::Configure ( const chartDef& cdef )
{
   this->reset () ;                 // initialize all data members

   this->dp = cdef.dPtr ;                 // dialog pointer (or NULL pointer)
   if ( this->dp == NULL )                // if local dialog to be created
   {
      this->wpTerm.ypos = cdef.ulY ;      // local dialog absolute terminal Y offset
      this->wpTerm.xpos = cdef.ulX ;      // local dialog absolute terminal X offset
      this->ldRows      = cdef.rows ;     // local dialog height
      this->ldCols      = cdef.cols ;     // local dialog width
      this->dispRows    = cdef.rows - 2 ; // rows in display area (inside the borders)
      this->dispCols    = cdef.cols - 2 ; // columns in display area (inside the borders)
      this->wpBase.ypos = 1 ;             // Y origin inside upper left corner
      this->wpBase.xpos = 1 ;             // X origin inside upper left corner
      this->localDlg    = true ;          // set the local-dialog-definition flag
   }
   else
   {
      this->wpBase.ypos = cdef.ulY ;         // Y origin
      this->wpBase.xpos = cdef.ulX ;         // X origin
      this->dispRows    = cdef.rows ;        // rows in display area
      this->dispCols    = cdef.cols ;        // columns in display area
   }
   this->yOffset     = cdef.yOff ;        // space above grid
   this->xOffset     = cdef.xOff ;        // space at left of grid
   this->yFooter     = cdef.yFoot ;       // space below grid
   this->barWidth    = cdef.barWidth ;    // width of each bar (divisions)
   this->barSpace    = cdef.barSpace ;    // spacing between bars (0 or 1 columns)
   this->chType      = cdef.chType ;      // chart type (member of enum chartType)
   this->bdrColor    = cdef.borderColor ; // border color attribute
   this->titColor    = cdef.titleColor ;  // title-text color attribute
   this->barColor    = cdef.barColor ;    // default bar color attribute
   this->negColor    = cdef.negColor ;    // bar color for negative values
   this->dColor      = cdef.textColor ;   // normal text color attribute
   this->bColor      = cdef.boldColor ;   // bold text color attribute
   this->gColor      = cdef.gridColor ;   // vertical/horizontal axes color attribute
   this->dataCount   = cdef.dataCount ;   // number of data elements in 'dataPtr'
   this->dataOffset  = cdef.dataOffset ;  // index of first data item to be displayed
   //* NOTE: The 'dataPtr' member is initialized by the call to ConvertData().*
   this->attrPtr     = cdef.attrPtr ;     // array of color attributes for bars (optional)
   this->bStyle      = cdef.borderStyle ; // line style for border
   this->gStyle      = cdef.gridStyle ;   // line style for vertical/horizontal axes
   this->vLabel      = cdef.vaxisLabel ;  // Y-axis label text
   this->hLabel      = cdef.haxisLabel ;  // X-axis label text
   this->titText     = cdef.title ;       // title text
   this->hdrText     = cdef.headText ;    // header text
   this->ftrText     = cdef.footText ;    // footer text
   this->hBars       = cdef.horizBars ;   // horizontal bars == 'true', vertical == 'false'
   this->bTips       = cdef.barTips ;     // display full bars or only tips of bars
   this->bdrFlag     = cdef.border ;      // border flag
   this->localDlg    = this->dp == NULL ? true : false ; // locally-instantiated-dialog flag
   // All other members will be calculated below

   //* Perform sanity check on data provided by caller.          *
   //* NOTE: This test IS NOT a guarantee against user stupidity.*
   if ( this->wpBase.ypos < ZERO || this->wpBase.xpos < ZERO )
      this->wpBase.ypos = this->wpBase.xpos = ZERO ;
   if ( this->dispRows < 1 )
      this->dispRows = 1 ;
   if ( this->dispCols < 1 )
      this->dispRows = 1 ;
   if ( this->yOffset < 1 || this->yOffset >= this->dispRows )
      this->yOffset = 1 ;
   if ( this->xOffset < ZERO || this->xOffset >= this->dispCols )
      this->xOffset = 1 ;
   if ( this->yFooter < 1 || this->yFooter >= this->dispRows )
      this->yFooter = 1 ;
   if ( this->barWidth < 1 || this->barWidth > cellDIV )
      this->barWidth = cellDIV ;
   if ( this->barSpace < ZERO || this->barSpace > 1 )
      this->barSpace = ZERO ;
   if ( this->chType < ctLowerLeft || this->chType > ctCartesian )
      this->chType = ctLowerLeft ;
   if ( this->dataCount < ZERO )
      this->dataCount = ZERO ;
   if ( this->bStyle != ncltSINGLE && this->bStyle != ncltDUAL )
      this->bStyle = ncltSINGLE ;
   if ( this->gStyle != ncltSINGLE && this->gStyle != ncltDUAL )
      this->gStyle = ncltSINGLE ;
   //* For locally-defined dialog window, the border *
   //* is built in, so disable the border flag.      *
   if ( this->dp == NULL )
      this->bdrFlag = false ;
   //* These chart types support only vertical bars.*
   if ( (this->chType == ctCenterLeft) || (this->chType == ctCenterRight) )
      this->hBars = false ;
   //* These chart types support only horizontal bars.*
   if ( (this->chType == ctLowerCenter) || (this->chType == ctUpperCenter) )
      this->hBars = true ;
   // (all non-null char* members are assumed to point at actual text)
   // (color attributes and 'dataAttr' if specified, are assumed to be valid)
   // (dimensions are assumed to fit within the terminal window and/or dialog window)

   //* Initialize the source data as an array of *
   //* double-precision floating point values.   *
   this->ConvertData ( cdef.dataPtr, cdef.dataCount, cdef.dataType, this->dataPtr ) ;

   //* Set grid characters for specified grid style *
   this->SetStyle () ;

   //* Calculate the number of character cells available for the chart. *
   //* (See notes in method header about this calculation.)             *
   this->SetDimensions () ;

   //* Set base position of chart grid. This identifies *
   //* the cell where vertical and horizontal axes meet.*
   this->SetOrigin () ;

   //* Do a preliminary scan of the data *
   if ( this->dataCount > ZERO )
   {
      //* Establish value range *
      meaVal = 0.0 ;
      minVal = maxVal = this->dataPtr[ZERO] ;
      for ( int32_t indx = ZERO ; indx < this->dataCount ; ++indx )
      {
         if ( this->dataPtr[indx] < minVal )
            minVal = this->dataPtr[indx] ;         // minimum value
         if ( this->dataPtr[indx] > maxVal )
            maxVal = this->dataPtr[indx] ;         // maximum value
         meaVal += this->dataPtr[indx] ;           // accumulate the values
      }
      meaVal /= double(this->dataCount) ;          // arithmetic mean value
      this->rngVal = this->maxVal - this->minVal ; // data range
      // Note: The median value 'medVal' is calculated on demand.
      //       This is because it requires sorting the source data.
   }

   //* Calculate available grid divisions.                                 *
   //* A "division" is the smallest visible bar element (1/8 of one cell). *
   this->verDiv = this->vCells * cellDIV ;   // total vertical divisions
   this->horDiv = this->hCells * cellDIV ;   // total horizontal divisions

   //* Set the maximum number of divisions available for a bar.            *
   //* -- For chart types which allow the data to extend the full width of *
   //*    the grid, all physical divisions are available.                  *
   //* -- For chart types with a centered axis, positive values may extend *
   //*    from the axis across half the chart, while negative values extend*
   //*    in the opposite direction across the other half of the chart.    *
   //*    -- Note that for chart types with a centered axis, the 'minDiv'  *
   //*       limit is ignored, so: minDiv==ZERO and rngDiv==maxDiv         *
   int32_t tmpDiv ;                    // integer result of floating-point math
   this->minDiv = minBAR ;             // divisions displayed for minimum value
   switch ( this->chType )             // maximum display divisions per bar
   {
      case ctLowerLeft:
      case ctLowerRight:
      case ctUpperLeft:
      case ctUpperRight:
         this->maxDiv = (this->hBars ? this->horDiv : this->verDiv) ;
         break ;
      case ctLowerCenter:
      case ctUpperCenter:
         //* Note that these chart types are designed for *
         //* horizontal bars ONLY, but safety first.      *
         //* Whole cells only. Discard fractional cell.   *
         tmpDiv = (this->hBars ? (this->horDiv / 2) : (this->verDiv / 2)) ;
         this->maxDiv = (tmpDiv - (tmpDiv % cellDIV)) ;
         this->minDiv = ZERO ;
         break ;
      case ctCenterLeft:
      case ctCenterRight:
         //* Note that these chart types are designed for *
         //* vertical bars ONLY, but safety first.        *
         //* Whole cells only. Discard fractional cell.   *
         tmpDiv = (this->hBars ? (this->horDiv / 2) : (this->verDiv / 2)) ;
         this->maxDiv = (tmpDiv - (tmpDiv % cellDIV)) ;
         this->minDiv = ZERO ;
         break ;
      case ctCartesian:
         /* Note that Cartesian charts do not use cell-division data. */
         break ;
   } ;
   //* Range of divisions for graph orientation *
   this->rngDiv = this->maxDiv - this->minDiv ;
   //* Data-units per division *
   this->perDiv = this->rngVal / (maxDiv - minDiv) ;
   //* If data range is less than division range, promote 'perDiv' to 1.0 *
   if ( this->perDiv < 1.0 )
      this->perDiv = 1.0 ;

}  //* End Configure() *

//*************************
//*      ConvertData      *
//*************************
//******************************************************************************
//* If the user has provided data in a format other than as doubles, convert   *
//* the data to doubles for processing.                                        *
//* The 'dataPtr' member is initialized to point to the working copy of the    *
//* input data (double-precision floating-point values).                       *
//*                                                                            *
//* Be Aware!! : If application provides an invalid pointer or a data type     *
//*              which doesn't match the provided data, then the most likely   *
//*              result will be crash-and-burn, or at best, garbled data.      *
//*                                                                            *
//* Input  : vdPtr  : void pointer to input data ('dType' == source format)    *
//*          dCount : number of elements referenced by 'vdPtr'                 *
//*          dType  : type of data referenced by 'vdPtr'                       *
//*          trgPtr : (by reference) receives a pointer to the converted data  *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Chart::ConvertData ( const void* vdPtr, int32_t dCount, idataType dType, 
                          const double*& trgPtr )
{
//   //* Defensive programming: If invalid pointer, set count to zero.*
//   if ( vdPtr == NULL )
//      this->dataCount = ZERO ;

   //* Verify that source data were provided, AND *
   //* that a supported data type was specified.  *
   if ( (vdPtr != NULL) && (dCount > ZERO) &&
        ((dType == idtDouble)     || (dType == idtFloat)   ||
         (dType == idtByte_s)     || (dType == idtByte_u)  ||
         (dType == idtShort_s)    || (dType == idtShort_u) ||
         (dType == idtInt_s)      || (dType == idtInt_u)   ||
         (dType == idtLong_s)     || (dType == idtLong_u)  ||
         (dType == idtLongLong_s) || (dType == idtLongLong_u)) )
   {
      //* If the data are already in the correct format, *
      //* simply copy the pointer.                       *
      if ( dType == idtDouble )
         trgPtr = (const double*)vdPtr ;

      //* Perform data-type conversion *
      else
      {
         //* Allocate the new data array (non-const pointer) *
         double *dptr = new double[dCount] ;
         this->dynoData = true ;    // set the dynamic-allocation flag

         if ( dType == idtFloat )
         {
            //* Use the specified pointer type.*
            const float *aptr = (const float*)vdPtr ;
            for ( int32_t i = ZERO ; i < dCount ; ++i )
               dptr[i] = double(aptr[i]) ;
         }
         else if ( dType == idtShort_s )
         {
            //* Use the specified pointer type.*
            const short *aptr = (const short*)vdPtr ;
            for ( int32_t i = ZERO ; i < dCount ; ++i )
               dptr[i] = double(aptr[i]) ;
         }
         else if ( dType == idtShort_u )
         {
            //* Use the specified pointer type.*
            const unsigned short *aptr = (const unsigned short*)vdPtr ;
            for ( int32_t i = ZERO ; i < dCount ; ++i )
               dptr[i] = double(aptr[i]) ;
         }
         else if ( dType == idtInt_s )
         {
            const int *aptr = (const int*)vdPtr ;
            for ( int32_t i = ZERO ; i < dCount ; ++i )
               dptr[i] = double(aptr[i]) ;
         }
         else if ( dType == idtInt_u )
         {
            const unsigned int *aptr = (const unsigned int*)vdPtr ;
            for ( int32_t i = ZERO ; i < dCount ; ++i )
               dptr[i] = double(aptr[i]) ;
         }
         else if ( dType == idtLong_s )
         {
            const long *aptr = (const long*)vdPtr ;
            for ( int32_t i = ZERO ; i < dCount ; ++i )
               dptr[i] = double(aptr[i]) ;
         }
         else if ( dType == idtLong_u )
         {
            const unsigned long *aptr = (const unsigned long*)vdPtr ;
            for ( int32_t i = ZERO ; i < dCount ; ++i )
               dptr[i] = double(aptr[i]) ;
         }
         else if ( dType == idtLongLong_s )
         {
            const long long *aptr = (const long long*)vdPtr ;
            for ( int32_t i = ZERO ; i < dCount ; ++i )
               dptr[i] = double(aptr[i]) ;
         }
         else if ( dType == idtLongLong_u )
         {
            const unsigned long long *aptr = (const unsigned long long*)vdPtr ;
            for ( int32_t i = ZERO ; i < dCount ; ++i )
               dptr[i] = double(aptr[i]) ;
         }
         else if ( dType == idtByte_s )
         {
            const char *aptr = (const char*)vdPtr ;
            for ( int32_t i = ZERO ; i < dCount ; ++i )
               dptr[i] = double(aptr[i]) ;
         }
         else if ( dType == idtByte_u )
         {
            const unsigned char *aptr = (const unsigned char*)vdPtr ;
            for ( int32_t i = ZERO ; i < dCount ; ++i )
               dptr[i] = double(aptr[i]) ;
         }

         trgPtr = dptr ;      // initialize caller's pointer
      }
   }

}  //* End ConvertData() *

//*************************
//*       SetStyle        *
//*************************
//******************************************************************************
//* Set the X and Y axis-line style for the chart.                             *
//*                                                                            *
//* Input  : none (references the 'chType' and 'gStyle' members)               *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Chart::SetStyle ( void )
{
   switch ( this->chType )
   {
      case ctLowerLeft:    // origin at lower-left (default type)
         if ( this->gStyle == ncltDUAL )
         {
            this->vaxisChar = wcsLTEEdv ;    // L'╟'
            this->haxisChar = wcsBTEEdh;     // L'╧'
            this->axisCross = wcsLLd ;       // L'╚'
            this->axisCap   = (this->hBars ? wcsHORIZd : wcsVERTd) ; // L'═' or L'║'
         }
         else  // this->gStyle == ncltSINGLE || invalid value
         {
            this->vaxisChar = wcsLTEEs ;     // L'├'
            this->haxisChar = wcsBTEEs ;     // L'┴'
            this->axisCross = wcsLLs ;       // L'└'
            this->axisCap   = (this->hBars ? wcsHORIZs : wcsVERTs) ; // L'─' or L'│'
         }
         break ;
      case ctLowerRight:   // origin at lower-right
         if ( this->gStyle == ncltDUAL )
         {
            this->vaxisChar = wcsRTEEdv ;    // L'╢'
            this->haxisChar = wcsBTEEdh;     // L'╧'
            this->axisCross = wcsLRd ;       // L'╝'
            this->axisCap   = (this->hBars ? wcsHORIZd : wcsVERTd) ; // L'═' or L'║'
         }
         else  // this->gStyle == ncltSINGLE
         {
            this->vaxisChar = wcsRTEEs ;     // L'┤'
            this->haxisChar = wcsBTEEs ;     // L'┴'
            this->axisCross = wcsLRs ;       // L'┘'
            this->axisCap   = (this->hBars ? wcsHORIZs : wcsVERTs) ; // L'─' or L'│'
         }
         break ;
      case ctUpperLeft:    // origin at upper-left
         if ( this->gStyle == ncltDUAL )
         {
            this->vaxisChar = wcsLTEEdv ;    // L'╟'
            this->haxisChar = wcsTTEEdh;     // L'╤'
            this->axisCross = wcsULd ;       // L'╔'
            this->axisCap   = (this->hBars ? wcsHORIZd : wcsVERTd) ; // L'═' or L'║'
         }
         else  // this->gStyle == ncltSINGLE || invalid value
         {
            this->vaxisChar = wcsLTEEs ;     // L'├'
            this->haxisChar = wcsTTEEs ;     // L'┬'
            this->axisCross = wcsULs ;       // L'┌'
            this->axisCap   = (this->hBars ? wcsHORIZs : wcsVERTs) ; // L'─' or L'│'
         }
         break ;
      case ctUpperRight:   // origin at upper-right
         if ( this->gStyle == ncltDUAL )
         {
            this->vaxisChar = wcsRTEEdv ;    // L'╢'
            this->haxisChar = wcsTTEEdh;     // L'╤'
            this->axisCross = wcsURd ;       // L'╗'
            this->axisCap   = (this->hBars ? wcsHORIZd : wcsVERTd) ; // L'═' or L'║'
         }
         else  // this->gStyle == ncltSINGLE
         {
            this->vaxisChar = wcsRTEEs ;     // L'┤'
            this->haxisChar = wcsTTEEs ;     // L'┬'
            this->axisCross = wcsURs ;       // L'┐'
            this->axisCap   = (this->hBars ? wcsHORIZs : wcsVERTs) ; // L'─' or L'│'
         }
         break ;
      case ctLowerCenter:  // origin at lower-center
         if ( this->gStyle == ncltDUAL )
         {
            this->vaxisChar = wcsINSECTdv ;  // L'╫'
            this->haxisChar = wcsBTEEdh ;    // L'╧'
            this->axisCross = wcsBTEEd ;     // L'╩'
            this->axisCap   = (this->hBars ? wcsHORIZd : wcsVERTd) ; // L'═' or L'║'
         }
         else  // this->gStyle == ncltSINGLE
         {
            this->vaxisChar = wcsINSECTs ;   // L'┼'
            this->haxisChar = wcsBTEEs ;     // L'┴'
            this->axisCross = wcsBTEEs ;     // L'┴'
            this->axisCap   = (this->hBars ? wcsHORIZs : wcsVERTs) ; // L'─' or L'│'
         }
         break ;
      case ctUpperCenter:  // origin at upper-center
         if ( this->gStyle == ncltDUAL )
         {
            this->vaxisChar = wcsINSECTdv ;  // L'╫'
            this->haxisChar = wcsTTEEdh ;    // L'╤'
            this->axisCross = wcsTTEEd ;     // L'╦'
            this->axisCap   = (this->hBars ? wcsHORIZd : wcsVERTd) ; // L'═' or L'║'
         }
         else  // this->gStyle == ncltSINGLE
         {
            this->vaxisChar = wcsINSECTs ;   // L'┼'
            this->haxisChar = wcsTTEEs ;     // L'┬'
            this->axisCross = wcsTTEEs ;     // L'┬'
            this->axisCap   = (this->hBars ? wcsHORIZs : wcsVERTs) ; // L'─' or L'│'
         }
         break ;
      case ctCenterLeft:   // origin at center-left
         if ( this->gStyle == ncltDUAL )
         {
            this->vaxisChar = wcsLTEEdv ;    // L'╟'
            this->haxisChar = wcsINSECTdh ;  // L'╪'
            this->axisCross = wcsLTEEd ;     // L'╠'
            this->axisCap   = (this->hBars ? wcsHORIZd : wcsVERTd) ; // L'═' or L'║'
         }
         else  // this->gStyle == ncltSINGLE
         {
            this->vaxisChar = wcsLTEEs ;     // L'├'
            this->haxisChar = wcsINSECTs ;   // L'┼'
            this->axisCross = wcsLTEEs ;     // L'├'
            this->axisCap   = (this->hBars ? wcsHORIZs : wcsVERTs) ; // L'─' or L'│'
         }
         break ;
      case ctCenterRight:  // origin at center-right
         if ( this->gStyle == ncltDUAL )
         {
            this->vaxisChar = wcsRTEEdv ;    // L'╢'
            this->haxisChar = wcsINSECTdh ;  // L'╪'
            this->axisCross = wcsRTEEd ;     // L'╣'
            this->axisCap   = (this->hBars ? wcsHORIZd : wcsVERTd) ; // L'═' or L'║'
         }
         else  // this->gStyle == ncltSINGLE
         {
            this->vaxisChar = wcsRTEEs ;     // L'┤'
            this->haxisChar = wcsINSECTs ;   // L'┼'
            this->axisCross = wcsRTEEs ;     // L'┤'
            this->axisCap   = (this->hBars ? wcsHORIZs : wcsVERTs) ; // L'─' or L'│'
         }
         break ;
      case ctCartesian:    // Cartesian coordinates
         if ( this->gStyle == ncltDUAL )
         {
            this->vaxisChar = wcsINSECTdv ;  // L'╫'
            this->haxisChar = wcsINSECTdh ;  // L'╪'
            this->axisCross = wcsINSECTd ;   // L'╬'
            this->axisCap   = wcsINSECTdv ;  // L'╫'
         }
         else  // this->gStyle == ncltSINGLE
         {
            this->vaxisChar = wcsINSECTs ;   // L'┼'
            this->haxisChar = wcsINSECTs ;   // L'┼'
            this->axisCross = wcsINSECTs ;   // L'┼'
            this->axisCap   = wcsINSECTs ;   // L'┼'
         }
         break ;
      default:             // origin at lower-left
         if ( this->gStyle == ncltDUAL )
         {
            this->vaxisChar = wcsLTEEdv ;    // L'╟'
            this->haxisChar = wcsBTEEdh;     // L'╧'
            this->axisCross = wcsLLd ;       // L'╚'
            this->axisCap   = (this->hBars ? wcsHORIZd : wcsVERTd) ; // L'═' or L'║'
         }
         else  // this->gStyle == ncltSINGLE || invalid value
         {
            this->vaxisChar = wcsLTEEs ;     // L'├'
            this->haxisChar = wcsBTEEs ;     // L'┴'
            this->axisCross = wcsLLs ;       // L'└'
            this->axisCap   = (this->hBars ? wcsHORIZs : wcsVERTs) ; // L'─' or L'│'
         }
         break ;
   }
}  //* End SetStyle() *

//*************************
//*     SetDimensions     *
//*************************
//******************************************************************************
//* Set the X and Y axis dimensions for the chart.                             *
//*                                                                            *
//* Input  : none (references the 'chType' and 'gStyle' members)               *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Notes on calculating the number of active cells within the chart grid.     *
//*                                                                            *
//* Vertical dimension (vCells):                                               *
//* ----------------------------                                               *
//* The preliminary height of the chart (_active_ character cells) is the      *
//* height of the display area MINUS the specified 'Y-offset' and 'Y-footer'   *
//* reserved areas, MINUS the row occupied by the horizontal-axis.             *
//*  -- The minumum for 'Y-offset' and 'Y-footer' is one(1). This reserves     *
//*     one row above and below the chart area used for axis labels.           *
//* This preliminary height is adjusted for three(3) conditions:               *
//*  1) The X-axis column is included as an active cell for charts with a      *
//*     centered X-axis.                                                       *
//*  2) For some chart types, when the data bars are displayed parallel to the *
//*     Y-axis, the Y-axis cell furthest from the origin is reserved as a      *
//*     "cap" cell which may be used to display annotations.                   *
//*  3) If a border around the chart area is defined, then the height of the   *
//*     chart is reduced to compensate for the top and bottom borders.         *
//*                                                                            *
//* Horizontal dimension (hCells):                                             *
//* ------------------------------                                             *
//* The preliminary width of the chart (_active_ character cells) is the       *
//* width of the display area MINUS the specified X-offset, MINUS the column   *
//* occupied by the vertical-axis, MINUS one open column between the end of    *
//* the X-axis and the edge of the display area.                               *
//* This preliminary width is adjusted for three(3) conditions:                *
//*  1) The Y-axis column is included as an active cell for charts with a      *
//*     centered Y-axis.                                                       *
//*  2) For some chart types, when the data bars are displayed parallel to the *
//*     X-axis, the last cell of the X-axis (furthest from origin) is reserved.*
//*     This is the "cap" cell which may be used to display annotations.       *
//*  3) If a border around the chart area is defined, then the width of the    *
//*     chart is reduced to compensate for the left and right borders.         *
//*                                                                            *
//*                                                                            *
//* Special Case:                                                              *
//* For Cartesian charts, all cells of both horizontal and vertical axes are   *
//* active and are included in hCells and vCells, respectively. No "cap" cell  *
//* is defined for either axis of Cartesian charts.                            *
//******************************************************************************

void Chart::SetDimensions ( void )
{
   //*****************************************
   //* Set the base height of the chart grid *
   //*****************************************
   this->gridRows = this->dispRows - this->yFooter - this->yOffset ;
   this->vCells = this->gridRows - 1 ;

   //* For bars that run parallel to the Y-axis, reserve *
   //* the "cap" cell for specific chart types.          *
   //* The remaining chart types have no "cap" cell.     *
   if ( ! this->hBars &&
        ((this->chType == ctLowerLeft)   ||
         (this->chType == ctLowerRight)  ||
         (this->chType == ctUpperLeft)   ||
         (this->chType == ctUpperRight)) )
   { --this->vCells ; }

   //* Include the H-axis column as an active cell for these chart types.*
   if ( this->chType == ctCartesian )
   { ++this->vCells ; }

   //****************************************
   //* Set the base width of the chart grid *
   //****************************************
   this->gridCols = this->dispCols - this->xOffset - 1 ;
   this->hCells = this->gridCols - 1 ;

   //* Include the V-axis column as an active cell for these chart types.*
   if ( (this->chType == ctLowerCenter) ||
        (this->chType == ctUpperCenter) ||
        (this->chType == ctCartesian) )
   { ++this->hCells ; }

   //* For bars that run parallel to the X-axis, reserve the "cap" *
   //* cell for specific chart types. The remaining chart types    *
   //* have no "cap" cell.                                         *
   if ( this->hBars &&
        ((this->chType == ctLowerLeft)  ||
         (this->chType == ctLowerRight) ||
         (this->chType == ctUpperLeft)  ||
         (this->chType == ctUpperRight)) )
   { --this->hCells ; }

   //* If caller has specified a border around the display area, *
   //* reduce the dimensions of the chart accordingly.           *
   if ( this->bdrFlag )
   {
      //* Reduce the horizontal dimension of the chart *
      //* to compensate for left and right borders.    *
      this->gridCols -= 2 ;
      this->hCells -= 2 ;

      //* Reduce the vertical dimension of the chart *
      //* to compensate for top and bottom borders,  *
      //* then advance Y offset to maintain spacing. *
      this->gridRows -= 2 ;
      this->vCells -= 2 ;
      ++this->yOffset ;
   }

}  //* End SetDimensions() *

//*************************
//*       SetOrigin       *
//*************************
//******************************************************************************
//* For the specified chart type, set the origin (the point at which the       *
//* X and Y axes meet.                                                         *
//*                                                                            *
//* Input  : none (references the 'chType' member)                             *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Special Case: If an axis is centered, then we must be aware of the number  *
//* of cells on either size of the centered axis. If there is an odd number    *
//* cells in the crossing axis, then there will be an equal number of cells on *
//* either side. However, if there are an even number of cells for the crossing*
//* axis we must be consistent about which side the extra cell inhabits.       *
//* Logically (at least in the author's head), the POSITIVE side of the axis   *
//* should receive the extra cell. Arithematically, this is natural because    *
//* the divide-by-two operation used to calculate the halfway point will       *
//* truncate the fractional cell.                                              *
//*                                                                            *
//* Example: If the vertical axis is centered:                                 *
//*  a) If the horizontal axis has 61 cells, there will be 30 cells on either  *
//*     side of the vertical axis and the origin cell: 30 + 1 + 30 == 61       *
//*  b) If the horizontal axis has 80 cells, there will be 39 cells on the     *
//*     negative (lower) side, 40 cells on the positive (upper) side and the   *
//*     origin cell: 39 + 1 + 40 == 80                                         *
//*                                                                            *
//* Example: If the horizontal axis is centered:                               *
//*  a) If the vertical axis has 25 cells, there will be 12 cells on either    *
//*     side of the horizontal axis and the origin cell: 12 + 1 + 12 == 25     *
//*  b) If the vertical axis has 22 cells, there will be 10 cells on the       *
//*     negative (left) side, 11 cells on the positive (right) side and the    *
//*     origin cell: 10 + 1 + 11 == 22                                         *
//*                                                                            *
//******************************************************************************

void Chart::SetOrigin ( void )
{
   switch ( this->chType )
   {
      case ctLowerLeft:    // origin at lower-left (default type)
         this->wpOrig.ypos = this->wpBase.ypos + this->yOffset + this->vCells 
                             + 1 - (this->hBars ? 1 : 0) ;
         this->wpOrig.xpos = this->wpBase.xpos + this->xOffset 
                             + (this->bdrFlag ? 1 : 0) ;
         break ;
      case ctLowerRight:   // origin at lower-right
         this->wpOrig.ypos = this->wpBase.ypos + this->yOffset + this->vCells 
                             + 1 - (this->hBars ? 1 : 0) ;
         this->wpOrig.xpos = this->wpBase.xpos + this->xOffset + this->hCells 
                             + (this->hBars ? 1 : 0) + (this->bdrFlag ? 1 : 0) ;
         break ;
      case ctUpperLeft:    // origin at upper-left
         this->wpOrig.ypos = this->wpBase.ypos + this->yOffset ;
         this->wpOrig.xpos = this->wpBase.xpos + this->xOffset 
                             + (this->bdrFlag ? 1 : 0) ;
         break ;
      case ctUpperRight:   // origin at upper-right
         this->wpOrig.ypos = this->wpBase.ypos + this->yOffset ;
         this->wpOrig.xpos = this->wpBase.xpos + this->xOffset + this->hCells 
                             + (this->hBars ? 1 : 0) + (this->bdrFlag ? 1 : 0) ;
         break ;
      case ctLowerCenter:  // origin at lower-center
         this->wpOrig.ypos = this->wpBase.ypos + this->yOffset + this->vCells 
                             + (this->hBars ? 0 : 1) ;
         this->wpOrig.xpos = this->wpBase.xpos + this->xOffset + (this->hCells / 2)
                             + (this->bdrFlag ? 1 : 0) ;
         break ;
      case ctUpperCenter:  // origin at upper-center
         this->wpOrig.ypos = this->wpBase.ypos + this->yOffset ;
         this->wpOrig.xpos = this->wpBase.xpos + this->xOffset + (this->hCells / 2)
                             + (this->bdrFlag ? 1 : 0) ;
         break ;
      case ctCenterLeft:   // origin at center-left
         this->wpOrig.ypos = this->wpBase.ypos + this->yOffset + this->vCells
                             - (this->vCells / 2) ;
         this->wpOrig.xpos = this->wpBase.xpos + this->xOffset
                             + (this->bdrFlag ? 1 : 0) ;
         break ;
      case ctCenterRight:  // origin at center-right
         this->wpOrig.ypos = this->wpBase.ypos + this->yOffset + this->vCells
                             - (this->vCells / 2) ;
         this->wpOrig.xpos = this->wpBase.xpos + this->xOffset + this->hCells 
                             + (this->hBars ? 1 : 0) + (this->bdrFlag ? 1 : 0) ;
         break ;
      case ctCartesian:    // Cartesian coordinates
#if 1    // UNDER CONSTRUCTION - CART ORIGIN
         this->wpOrig.ypos = this->wpBase.ypos + this->yOffset + (this->vCells / 2) ;
         //* - For an even number of vertical cells, the number of cells below *
         //*   the axis is one greater than the number of cells above the axis.*
         //*   Because the horizontal axis is considered positive, there will  *
         //*   be an equal number of positive and negative cells.              *
         //* - For an odd number of vertical cells, the horizontal axis is     *
         //*   centered with an equal number cells above and below the axis.   *
         //*   Because the horizontal axis is considered positive, there is    *
         //*   effectively one more positive cell than negative cells.         *
         if ( (this->vCells % 2) == ZERO )   // if even number of vertical cells
            --this->wpOrig.ypos ;
#else    //Y OFFSET IS WRONG
         this->wpOrig.ypos = this->wpBase.ypos + this->yOffset 
                             + (this->vCells / 2) ;
//                             + (this->vCells / 2) + 1 ;
#endif   // U/C
         this->wpOrig.xpos = this->wpBase.xpos + this->xOffset + (this->hCells / 2)
                             + (this->bdrFlag ? 1 : 0) ;
         break ;
      default:             // (unlikely to happen, bug origin at lower-left)
         this->wpOrig.ypos = this->wpBase.ypos + this->yOffset + this->vCells + 1 ;
         this->wpOrig.xpos = this->wpBase.xpos + this->xOffset ;
         break ;
   }

}  //* End SetOrigin() *

//*************************
//*       DrawGrid        *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* Draw the vertical and horizontal axes of the grid.                         *
//*                                                                            *
//* Input  : labels : (optional, 'false' by default)                           *
//*                   if 'true' draw axis labels if provided                   *
//*          refresh: (optional, 'false' by default)                           *
//*                   if 'true' refresh the display (make axes visible)        *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
// UNDER CONSTRUCTION: VERIFY LENGTH OF AXES, AXIS CAP, WITH AND WITHOUT BORDER.
// v  == vertical axis no border
// vb == vertical axis with border
// h  == horizontal axis no border
// hb == horizontal axis with border
//Parent Dialog
//                      VERTICAL              HORIZONTAL
//    CHART TYPE          BARS                  BARS
//    -------------- -Rows-Cols-v-vb-h-hb  -Rows-Cols-v-vb-h-hb
//    ctLowerLeft     14   60   x x  x x    15   59   x x  x x
//    ctLowerRight    14   60   x x  x x    15   59   x x  x x
//    ctUpperLeft     14   60   x x  x x    15   59   x x  x x
//    ctUpperRight    14   60   x x  x x    15   59   x x  x x
//    ctLowerCenter   --   --   - -  - -    15   61   x x  x x
//    ctUpperCenter   --   --   - -  - -    15   61   x x  x x
//    ctCenterLeft    15   60   x x  x x    --   --   - -  - -
//    ctCenterRight   15   60   x x  x x    --   --   - -  - -
//    ctCartesian                                             

//Child Dialog
//    ctLowerLeft     15   68   x -  x -    17   67   x -  x -
//    ctLowerRight    15   68   x -  x -    17   67   x -  x -
//    ctUpperLeft     16   68   x -  x -    17   67   x -  x -
//    ctUpperRight    16   68   x -  x -    17   67   x -  x -
//    ctLowerCenter   17   69   - -  - -    17   69   x -  x -
//    ctUpperCenter   17   69   - -  - -    17   69   x -  x -
//    ctCenterLeft    17   68   x -  - -    --   --   - -  - -
//    ctCenterRight   17   68   x -  - -    --   --   - -  - -
//    ctCartesian                                             

void Chart::DrawGrid ( bool labels, bool refresh )
{
   winPos  wp ;            // cursor position
   gString gs ;            // text formatting
   short   trgpos ;        // target position

   if ( this->chType == ctLowerLeft )
   {
      wp = { short(this->wpOrig.ypos - this->vCells - (this->hBars ? 0 : 1)), 
             this->wpOrig.xpos } ;
      this->jYlab = tjLeft ;        // position the Y label 
      this->wpYlab = { short(wp.ypos - 1), wp.xpos } ;
      if ( ! this->hBars )
      {
         gs.compose( "%C\n", &this->axisCap ) ;
         wp = this->dp->WriteParagraph ( wp, gs, this->gColor ) ;
      }
      gs.compose( "%C\n", &this->vaxisChar ) ;
      while ( wp.ypos < this->wpOrig.ypos )
         wp = this->dp->WriteParagraph ( wp, gs, this->gColor ) ;
      wp = this->dp->WriteChar ( wp, this->axisCross, this->gColor ) ;

      //* Draw the horiontal axis *
      this->jXlab = tjLeft ;        // position the X label
      this->wpXlab = { short(wp.ypos + 1), wp.xpos } ;
      trgpos = this->hCells ;
      for ( short i = ZERO ; i < trgpos ; ++i )
         wp = this->dp->WriteChar ( wp, this->haxisChar, this->gColor ) ;
      if ( this->hBars )
         this->dp->WriteChar ( wp, this->axisCap, this->gColor ) ;
   }
   else if ( this->chType == ctLowerRight )
   {
      wp = { short(this->wpOrig.ypos - this->vCells - (this->hBars ? 0 : 1)),
             this->wpOrig.xpos } ;
      this->jYlab = tjRight ;       // position the Y label
      this->wpYlab = { short(wp.ypos - 1), wp.xpos } ;
      if ( ! this->hBars )
      {
         gs.compose( "%C\n", &this->axisCap ) ;
         wp = this->dp->WriteParagraph ( wp, gs, this->gColor ) ;
      }
      gs.compose( "%C\n", &this->vaxisChar ) ;
      while ( wp.ypos < this->wpOrig.ypos )
         wp = this->dp->WriteParagraph ( wp, gs, this->gColor ) ;
      wp = this->dp->WriteChar ( wp, this->axisCross, this->gColor ) ;

      //* Draw the horiontal axis *
      this->jXlab = tjRight ;       // position the X label
      this->wpXlab = { short(this->wpOrig.ypos + 1), short(this->wpOrig.xpos - 1) } ;
      wp.xpos -= this->hCells + (this->hBars ? 2 : 1) ;
      if ( this->hBars )
         wp = this->dp->WriteChar ( wp, this->axisCap, this->gColor ) ;
      while ( wp.xpos <  this->wpOrig.xpos )
         wp = this->dp->WriteChar ( wp, this->haxisChar, this->gColor ) ;
   }
   else if ( this->chType == ctUpperLeft )
   {
      //* Draw the vertical axis *
      wp = this->wpOrig ;
      trgpos = wp.ypos + this->vCells + 1 ;
      gs.compose( "%C\n", &this->axisCross ) ;
      wp = this->dp->WriteParagraph ( wp, gs, this->gColor ) ;
      gs.compose( "%C\n", &this->vaxisChar ) ;
      while ( wp.ypos < trgpos )
         wp = this->dp->WriteParagraph ( wp, gs, this->gColor ) ;
      if ( ! this->hBars )
         this->dp->WriteChar ( wp.ypos++, wp.xpos, this->axisCap, this->gColor ) ;
      this->jYlab = tjLeft ;        // position the Y label
      this->wpYlab = wp ;

      //* Draw the horiontal axis *
      this->jXlab = tjLeft ;        // position the X label
      this->wpXlab = { short(this->wpOrig.ypos - 1), short(this->wpOrig.xpos + 1) } ;
      wp = { this->wpOrig.ypos, short(this->wpOrig.xpos + 1) } ;
      trgpos = wp.xpos + this->hCells ;
      while ( wp.xpos < trgpos )
         wp = this->dp->WriteChar ( wp, this->haxisChar, this->gColor ) ;
      if ( this->hBars )
         this->dp->WriteChar ( wp, this->axisCap, this->gColor ) ;
   }
   else if ( this->chType == ctUpperRight )
   {
      //* Draw the vertical axis *
      wp = this->wpOrig ;
      trgpos = wp.ypos + this->vCells + 1 ;
      gs.compose( "%C\n", &this->axisCross ) ;
      wp = this->dp->WriteParagraph ( wp, gs, this->gColor ) ;
      gs.compose( "%C\n", &this->vaxisChar ) ;
      while ( wp.ypos < trgpos )
         wp = this->dp->WriteParagraph ( wp, gs, this->gColor ) ;
      if ( ! this->hBars )
         this->dp->WriteChar ( wp.ypos++, wp.xpos, this->axisCap, this->gColor ) ;
      this->jYlab = tjRight ;       // position the Y label
      this->wpYlab = wp ;

      //* Draw the horiontal axis *
      this->jXlab = tjRight ;       // position the X label
      this->wpXlab = { short(this->wpOrig.ypos - 1), short(this->wpOrig.xpos - 1) } ;
      wp = { this->wpOrig.ypos, 
             short(this->wpOrig.xpos - this->hCells - (this->hBars ? 1 : 0)) } ;
      if ( this->hBars )
         wp = this->dp->WriteChar ( wp, this->axisCap, this->gColor ) ;
      while ( wp.xpos < this->wpOrig.xpos )
         wp = this->dp->WriteChar ( wp, this->haxisChar, this->gColor ) ;
   }
   else if ( this->chType == ctLowerCenter ) //** ctLowerCenter **
   {
      //* Draw the vertical axis *
      // Programmer's Note: This chart type has no axis caps and supports 
      // only horizontal bars.
      wp = { short(this->wpOrig.ypos - this->vCells), this->wpOrig.xpos } ;
      this->jYlab = tjCenter ;      // position the Y label
      this->wpYlab = { short(wp.ypos - 1), wp.xpos } ;
      gs.compose( "%C\n", &this->vaxisChar ) ;
      while ( wp.ypos < this->wpOrig.ypos  )
         wp = this->dp->WriteParagraph ( wp, gs, this->gColor ) ;

      //* Draw the horiontal axis *
      this->jXlab = tjCenter ;      // position the X label
      this->wpXlab = { short(this->wpOrig.ypos + 1), this->wpOrig.xpos } ;
      wp.xpos = this->wpOrig.xpos - (this->hCells / 2) ;
      trgpos = wp.xpos + this->hCells ;
      while ( wp.xpos < trgpos )
         wp = this->dp->WriteChar ( wp, this->haxisChar, this->gColor ) ;
      this->dp->WriteChar( this->wpOrig, this->axisCross, this->gColor ) ;
   }
   else if ( this->chType == ctUpperCenter ) //** ctUpperCenter **
   {
      //* Draw the vertical axis *
      // Programmer's Note: This chart type has no axis caps and supports 
      // only horizontal bars.
      wp = { short(this->wpOrig.ypos + 1), this->wpOrig.xpos } ;
      trgpos = wp.ypos + this->vCells ;
      this->jYlab = tjCenter ;      // position the Y label
      this->wpYlab = { short(trgpos + (this->hBars ? 0 : 1)), wp.xpos } ;
      gs.compose( "%C\n", &this->vaxisChar ) ;
      while ( wp.ypos < trgpos  )
         wp = this->dp->WriteParagraph ( wp, gs, this->gColor ) ;

      //* Draw the horiontal axis *
      wp = { this->wpOrig.ypos, short(this->wpOrig.xpos - (this->hCells / 2)) } ;
      trgpos = wp.xpos + this->hCells ;
      this->jXlab = tjCenter ;      // position the X label
      this->wpXlab = { short(this->wpOrig.ypos - 1), this->wpOrig.xpos } ;
      while ( wp.xpos < trgpos )
         wp = this->dp->WriteChar ( wp, this->haxisChar, this->gColor ) ;
      this->dp->WriteChar( this->wpOrig, this->axisCross, this->gColor ) ;
   }
   else if ( this->chType == ctCenterLeft )
   {
      //* Draw the vertical axis *
      wp = { short(this->wpBase.ypos + this->yOffset), this->wpOrig.xpos } ;
      trgpos = wp.ypos + this->vCells + (this->hBars ? 0 : 1) ;
      this->jYlab = tjLeft ;        // position the Y label
      this->wpYlab = { short(wp.ypos - 1), wp.xpos } ;
      if ( ! this->hBars )
      {
         gs.compose( "%C\n", &this->axisCap ) ;
         wp = this->dp->WriteParagraph ( wp, gs, this->gColor ) ;
      }
      gs.compose( "%C\n", &this->vaxisChar ) ;
      while ( wp.ypos < trgpos  )
         wp = this->dp->WriteParagraph ( wp, gs, this->gColor ) ;

      //* Draw the horiontal axis *
      this->jXlab = tjRight ;       // position the Y label
      this->wpXlab = { wp.ypos, short(wp.xpos + this->hCells) } ;
      wp = this->dp->WriteChar ( this->wpOrig, this->axisCross, this->gColor ) ;
      trgpos = wp.xpos + this->hCells ;
      while ( wp.xpos < trgpos )
         wp = this->dp->WriteChar ( wp, this->haxisChar, this->gColor ) ;
   }
   else if ( this->chType == ctCenterRight )
   {
      //* Draw the vertical axis *
      wp = { short(this->wpBase.ypos + this->yOffset), this->wpOrig.xpos } ;
      trgpos = wp.ypos + this->vCells + (this->hBars ? 0 : 1) ;
      this->jYlab = tjRight ;       // position the Y label
      this->wpYlab = { short(wp.ypos - 1), wp.xpos } ;
      if ( ! this->hBars )
      {
         gs.compose( "%C\n", &this->axisCap ) ;
         wp = this->dp->WriteParagraph ( wp, gs, this->gColor ) ;
      }
      gs.compose( "%C\n", &this->vaxisChar ) ;
      while ( wp.ypos < trgpos )
         wp = this->dp->WriteParagraph ( wp, gs, this->gColor ) ;

      //* Draw the horiontal axis *
      this->jXlab = tjLeft ;        // position the Y label
      this->wpXlab = { wp.ypos, short(this->wpOrig.xpos - this->hCells) } ;
      wp = { this->wpOrig.ypos, short(this->wpOrig.xpos - this->hCells) } ;
      while ( wp.xpos < this->wpOrig.xpos )
         wp = this->dp->WriteChar ( wp, this->haxisChar, this->gColor ) ;
      wp = this->dp->WriteChar ( this->wpOrig, this->axisCross, this->gColor ) ;
   }
   else if ( this->chType == ctCartesian )
   {
#if 1    // UNDER CONSTRUCTION - DrawGrid(Cartesian)
      //* Draw the vertical axis *
      wp = { short(this->wpBase.ypos + this->yOffset), this->wpOrig.xpos } ;
      trgpos = wp.ypos + this->vCells - 1 ;
      this->jYlab = tjCenter ;      // position the Y label
      this->wpYlab = { short(wp.ypos - 1), wp.xpos } ;
      gs.compose( "%C\n", &this->vaxisChar ) ;
      while ( wp.ypos < trgpos  )
         wp = this->dp->WriteParagraph ( wp, gs, this->gColor ) ;
      this->dp->WriteChar ( wp, this->axisCap, this->gColor ) ;
      this->wpXlab.ypos = wp.ypos + 1 ;

      //* Draw the horiontal axis *
      wp = { this->wpOrig.ypos, short(this->wpOrig.xpos - (this->hCells / 2)) } ;
      trgpos = wp.xpos + this->hCells ;
      this->jXlab = tjLeft ;        // position the X label
      this->wpXlab.xpos = wp.xpos ;
      while ( wp.xpos < trgpos )
         wp = this->dp->WriteChar ( wp, this->haxisChar, this->gColor ) ;
      wp = this->dp->WriteChar ( this->wpOrig, this->axisCross, this->gColor ) ;
#endif   // U/C
   }

   //* Draw label(s) if specified *
   if ( labels && this->vLabel != NULL )
      this->DrawAxisLabel ( this->vLabel, this->wpYlab, this->jYlab ) ;
   if ( labels && this->hLabel != NULL )
      this->DrawAxisLabel ( this->hLabel, this->wpXlab, this->jXlab ) ;

}  //* End DrawGrid() *

//*************************
//*     MapData2Grid      *
//*************************
//******************************************************************************
//* Map the data onto the grid.                                                *
//* This method selects the appropriate lower-level method based on the chart  *
//* type and the direction of the bars.                                        *
//*                                                                            *
//* There are two basic chart types:                                           *
//*   1) those with both axes on the edges of the field, and                   *
//*   2) those with one or both axes centered in the field.                    *
//* There are two basic bar types:                                             *
//*   1) Vertical, and                                                         *
//*   2) Horizontal                                                            *
//* Cartesian charts support both bar types _plus_ X/Y pairs (scatter chart).  *
//*                                                                            *
//* For data bars which start at one edge of the field and moves toward the    *
//* opposite edge WITHOUT crossing an axis, data mapping is a simple operation.*
//*                                                                            *
//*    Chart Type      Bar Direction                                           *
//*   --------------- ------------------                                       *
//*    ctLowerLeft     B-to-T (vert) or L-to-R (horiz)                         *
//*    ctLowerRight    B-to-T (vert) or R-to-L (horiz)                         *
//*    ctUpperLeft     T-to-B (vert) or L-to-R (horiz)                         *
//*    ctUpperRight    T-to-B (vert) or R-to-L (horiz)                         *
//*    ctLowerCenter   B-to-T (vert)     (ONLY HORIZ CENTERED BARS MAKE SENSE) *
//*    ctUpperCenter   T-to-B (vert)     (ONLY HORIZ CENTERED BARS MAKE SENSE) *
//*    ctCenterLeft    L-to-R (horiz)    (ONLY VERT CENTERED BARS MAKE SENSE)  *
//*    ctCenterRight   R-to-L (horiz)    (ONLY VERT CENTERED BARS MAKE SENSE)  *
//*                                                                            *
//* For data bars which start on a centered axis and grow in both directions   *
//* at right angles to that axis, the calculation is a special case.           *
//*                                                                            *
//*    Chart Type      Bar Direction                                           *
//*   --------------- ------------------                                       *
//*    ctLowerCenter   L-and-R (horiz)                                         *
//*    ctUpperCenter   L-and-R (horiz)                                         *
//*    ctCenterLeft    U-and-D (vert)                                          *
//*    ctCenterRight   U-and-D (vert)                                          *
//*    ctCartesian     U-and-D (vert) or   (not fully implemented)             *
//*                    L-and-R (horiz)     (not fully implemented)             *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Chart::MapData2Grid ( void )
{
   if ( this->hBars )      // draw horizontal bars
   {
      if ( (this->chType == ctLowerLeft)   ||
           (this->chType == ctLowerRight)  ||
           (this->chType == ctUpperLeft)   ||
           (this->chType == ctUpperRight) )
      {
         this->MapHorizontalBars () ;
      }
      else if ( (this->chType == ctLowerCenter) ||
                (this->chType == ctUpperCenter) )
      {
         this->MapHorizontalCentered () ;
      }
      // Programmer's Note: Cartesian charts produce X/Y data points, 
      // not bars, but the 'hBars' member is used to indicate pair order.
      else if ( this->chType == ctCartesian )
         this->MapCartesianPairs () ;
      // Programmer's Note: Chart types ctCenterLeft and ctCenterRight
      // do not support horizontal bars. See Configuration().
   }
   else                    // draw vertical bars
   {
      if ( (this->chType == ctLowerLeft)   ||
           (this->chType == ctLowerRight)  ||
           (this->chType == ctUpperLeft)   ||
           (this->chType == ctUpperRight) )
      {
         this->MapVerticalBars () ;
      }
      else if ( (this->chType == ctCenterLeft)  ||
                (this->chType == ctCenterRight) )
      {
         this->MapVerticalCentered () ;
      }
      // Programmer's Note: Cartesian charts produce X/Y data points, 
      // not bars, but the 'hBars' member is used to indicate pair order.
      else if ( this->chType == ctCartesian )
         this->MapCartesianPairs () ;
      // Programmer's Note: Chart types ctLowerCenter and ctUpperCenter
      // do not support vertical bars. See Configuration().
   }

}  //* End MapData2Grid() *

//*************************
//*    MapVerticalBars    *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* Draw the dataset as vertical bars in sequential order.                     *
//* For chart styles: ctLowerLeft, ctLowerRight, ctUpperLeft, ctUpperRight.    *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Chart::MapVerticalBars ( void )
{
   double curVal,                      // data value to be mapped
          curDiv,                      // display divisions representing 'curVal'
          fWhole,                      // number of divisions (whole cells)
          fFract ;                     // number of divisions (fractional cell)
   short  iWhole,                      // number of filled cells (integer)
          iFract,                      // divisions in the fractional cell (integer)
          vStep,                       // step direction in which bar grows
          hStep,                       // step direction/distance between bars
          lastCol ;                    // last writable cell (loop control)
   winPos wp, wpo ;                    // cursor positioning
   wchar_t fcChar = hBlock[this->barWidth - 1], // full-cell bar character
           fracChar ;                  // fractional cell bar character (bar tip)
   attr_t  fcAttr = this->barColor,    // full-cell-character color attribute
           fracAttr = fcAttr ;         // color attribute for fractional bar cell

   //* Establish the origin-point cell, direction in which bar grows, step *
   //* direction/distance between bars, color attribute and endpoint.      *
   switch ( this->chType )
   {
      case ctLowerLeft:
         wpo = { short(this->wpOrig.ypos - 1), short(this->wpOrig.xpos + 1) } ;
         vStep = -1 ;
         hStep = this->barSpace + 1 ;
         lastCol = this->wpOrig.xpos + this->hCells ;
         break ;
      case ctLowerRight:
         wpo =  { short(this->wpOrig.ypos - 1), short(this->wpOrig.xpos - 1) } ;
         vStep = -1 ;
         hStep = -(this->barSpace + 1) ;
         lastCol = this->wpOrig.xpos - this->hCells ;
         if ( this->barWidth != cellDIV )
            fcAttr = InvertColorAttribute ( fcAttr ) ;
         break ;
      case ctUpperLeft:
         wpo = { short(this->wpOrig.ypos + 1), short(this->wpOrig.xpos + 1) } ;
         vStep = 1 ;
         hStep = this->barSpace + 1 ;
         lastCol = this->wpOrig.xpos + this->hCells ;
         fracAttr = InvertColorAttribute ( fcAttr ) ;
         break ;
      case ctUpperRight:
         wpo = { short(this->wpOrig.ypos + 1), short(this->wpOrig.xpos - 1) } ;
         vStep = 1 ;
         hStep = -(this->barSpace + 1) ;
         lastCol = this->wpOrig.xpos - this->hCells ;
         if ( this->barWidth != cellDIV )
            fcAttr = InvertColorAttribute ( fcAttr ) ;
         fracAttr = InvertColorAttribute ( fracAttr ) ;
         break ;
      case ctLowerCenter:     // silence the compiler warning
      case ctUpperCenter:     // silence the compiler warning
      case ctCenterLeft:      // silence the compiler warning
      case ctCenterRight:     // silence the compiler warning
      case ctCartesian:       // silence the compiler warning
         break ;
   } ;

   //* Display as much of the data as will fit on the horizonal axis *
   for ( int32_t indx = this->dataOffset ; indx < this->dataCount ; ++indx )
   {
      wp = wpo ;                       // position of bar-graph character
      if ( this->attrPtr != NULL )     // sequenced color attribute
      {
         fcAttr = fracAttr = this->attrPtr[indx] ;
         // If bar grows downward, invert fractional-cell color attribute
         if ( vStep > ZERO )     
            fracAttr = InvertColorAttribute ( fracAttr ) ;
         //* If narrow bar AND moving leftward, invert full-cell attribute *
         if ( (this->barWidth != cellDIV) && hStep < ZERO )
            fcAttr = InvertColorAttribute ( fcAttr ) ;
      }
      curVal = this->dataPtr[indx] ;   // get a value

      //* Number of display divisions needed to represent the value *
      curDiv = this->rngDiv * ((curVal - this->minVal) / this->rngVal) ;
      if ( curDiv < this->minDiv ) curDiv = this->minDiv ;  // range limiting
      if ( curDiv > this->maxDiv ) curDiv = this->maxDiv ;

      fFract = modf ( curDiv, &fWhole ) ; // separate whole and fractional divisions
      iWhole = short(fWhole) ;            // whole divisions
      iFract = iWhole % cellDIV ;         // divisions in fractional cell
      iWhole /= cellDIV ;                 // whole cells
      if ( fFract > 0.5 )                 // round up to next division
      {  //* If this yields a whole cell *
         if ( ++iFract >= cellDIV )
         { ++iWhole ; iFract = ZERO ; }
      }
      iFract -= 1 ;                       // convert fractional cell to index
      //* If bar grows downward, mirror the fractional cell. *
      if ( (vStep > ZERO) && (iFract >= ZERO && iFract < cellDIV) )
         iFract = InvertFractionalCell ( iFract ) ;

      //*************************
      //* Draw the vertical bar *
      //*************************
      for ( short i = ZERO ; i < iWhole ; ++i )
      {
         if ( ! this->bTips )
            this->dp->WriteChar ( wp, fcChar, fcAttr ) ;
         wp.ypos += vStep ;
      }
      if ( (iFract >= ZERO) || this->bTips )
      {
         fracChar = this->bTips ? vTipChar : vBlock[iFract] ;
         this->dp->WriteChar ( wp, fracChar, fracAttr ) ;
      }

      wpo.xpos += hStep ;                 // advance to next bar position
      //* If last column of chart has been filled, we're done *
      if ( ((hStep > ZERO) && (wpo.xpos > lastCol)) ||
           ((hStep < ZERO) && (wpo.xpos < lastCol)) )
         break ;
   }

}  //* MapVerticalBars() *

//*************************
//*   MapHorizontalBars   *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* Draw the dataset as horizontal bars in sequential order.                   *
//* For chart styles: ctLowerLeft, ctLowerRight, ctUpperLeft, ctUpperRight.    *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Chart::MapHorizontalBars ( void )
{
   double curVal,                      // data value to be mapped
          curDiv,                      // display divisions representing 'curVal'
          fWhole,                      // number of divisions (whole cells)
          fFract ;                     // number of divisions (fractional cell)
   short  iWhole,                      // number of filled cells (integer)
          iFract,                      // divisions in the fractional cell (integer)
          vStep,                       // step direction in which bar grows
          hStep,                       // step direction/distance between bars
          lastRow ;                    // last writable cell
   winPos wp, wpo ;                    // cursor positioning
   wchar_t fcChar = vBlock[this->barWidth - 1], // full-cell bar character
           fracChar ;                  // fractional cell bar character (bar tip)
   attr_t  fcAttr = this->barColor,    // full-cell-character color attribute
           fracAttr = fcAttr ;         // color attribute for fractional bar cell

   //* Establish the origin-point cell, direction in which bar grows, step *
   //* direction/distance between bars, color attribute and endpoint.      *
   switch ( this->chType )
   {
      case ctLowerLeft:
         wpo = { short(this->wpOrig.ypos - 1), short(this->wpOrig.xpos + 1) } ;
         vStep = -(this->barSpace + 1) ;
         hStep = 1 ;
         lastRow = this->wpOrig.ypos - this->vCells ;
         break ;
      case ctLowerRight:
         wpo=  { short(this->wpOrig.ypos - 1), short(this->wpOrig.xpos - 1) } ;
         vStep = -(this->barSpace + 1) ;
         hStep = -1 ;
         lastRow = this->wpOrig.ypos - this->vCells ;
         fracAttr = InvertColorAttribute ( fracAttr ) ;
         break ;
      case ctUpperLeft:
         wpo = { short(this->wpOrig.ypos + 1), short(this->wpOrig.xpos + 1) } ;
         vStep = this->barSpace + 1 ;
         hStep = 1 ;
         lastRow = this->wpOrig.ypos + this->vCells ;
         if ( this->barWidth != cellDIV )
            fcAttr = InvertColorAttribute ( fcAttr ) ;
         break ;
      case ctUpperRight:
         wpo = { short(this->wpOrig.ypos + 1), short(this->wpOrig.xpos - 1) } ;
         vStep = this->barSpace + 1 ;
         hStep = -1 ;
         lastRow = this->wpOrig.ypos + this->vCells ;
         if ( this->barWidth != cellDIV )
            fcAttr = InvertColorAttribute ( fcAttr ) ;
         fracAttr = InvertColorAttribute ( fracAttr ) ;
         break ;
      case ctLowerCenter:     // silence the compiler warning
      case ctUpperCenter:     // silence the compiler warning
      case ctCenterLeft:      // silence the compiler warning
      case ctCenterRight:     // silence the compiler warning
      case ctCartesian:       // silence the compiler warning
         break ;
   } ;

   //* Display as much of the data as will fit on the vertical axis *
   for ( int32_t indx = this->dataOffset ; indx < this->dataCount ; ++indx )
   {
      wp = wpo ;                       // position of bar-graph character
      if ( this->attrPtr != NULL )     // sequenced color attribute
      {
         fcAttr = fracAttr = this->attrPtr[indx] ;
         //* If bar grows leftward, invert fractional-cell color attribute.*
         if ( hStep < ZERO )     
            fracAttr = InvertColorAttribute ( fracAttr ) ;
         //* If narrow bar AND moving downward, invert full-cell attribute *
         if ( (this->barWidth != cellDIV) && vStep > ZERO )
            fcAttr = InvertColorAttribute ( fcAttr ) ;
      }
      curVal = this->dataPtr[indx] ;   // get a value

      //* Number of display divisions needed to represent the value *
      curDiv = this->rngDiv * ((curVal - this->minVal) / this->rngVal) ;
      if ( curDiv < this->minDiv ) curDiv = this->minDiv ;  // range limiting
      if ( curDiv > this->maxDiv ) curDiv = this->maxDiv ;

      fFract = modf ( curDiv, &fWhole ) ; // separate whole and fractional divisions
      iWhole = short(fWhole) ;            // whole divisions
      iFract = iWhole % cellDIV ;         // divisions in fractional cell
      iWhole /= cellDIV ;                 // whole cells
      if ( fFract > 0.5 )                 // round up to next division
      {  //* If this yields a whole cell *
         if ( ++iFract >= cellDIV )
         { ++iWhole ; iFract = ZERO ; }
      }
      iFract -= 1 ;                       // convert fractional cell to index
      //* If bar grows rightward, mirror the fractional cell. *
      if ( (hStep < ZERO) && (iFract >= ZERO && iFract < cellDIV) )
         iFract = InvertFractionalCell ( iFract ) ;

      //* Draw the horizontal bar *
      for ( short i = ZERO ; i < iWhole ; ++i )
      {
         if ( ! this->bTips )
            this->dp->WriteChar ( wp, fcChar, fcAttr ) ;
         wp.xpos += hStep ;
      }
      if ( (iFract >= ZERO) || this->bTips )
      {
         fracChar = this->bTips ? hTipChar : hBlock[iFract] ;
         this->dp->WriteChar ( wp, fracChar, fracAttr ) ;
      }

      wpo.ypos += vStep ;                 // advance to next bar position
      //* If last row of chart has been filled, we're done *
      if ( ((vStep > ZERO) && (wpo.ypos > lastRow)) ||
           ((vStep < ZERO) && (wpo.ypos < lastRow)) )
         break ;
   }

}  //* MapHorizontalBars() *

//*************************
//*  MapVerticalCentered  *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* Draw the dataset as vertical bars extending above and below the horizontal *
//* axis.                                                                      *
//* For chart styles: ctCenterLeft, ctCenterRight.                             *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Chart::MapVerticalCentered ( void )
{
   double curVal,                      // data value to be mapped
          curDiv,                      // display divisions representing 'curVal'
          fWhole,                      // number of divisions (whole cells)
          fFract ;                     // number of divisions (fractional cell)
   short  iWhole,                      // number of filled cells (integer)
          iFract,                      // divisions in the fractional cell (integer)
          vStep,                       // step direction in which bar grows
          hStep,                       // step direction/distance between bars
          lastCol ;                    // last writable cell (opposite vertical axis)
   winPos wp, wpo ;                    // cursor positioning
   wchar_t fcChar = hBlock[this->barWidth - 1], // full-cell bar character
           bChar ;                     // working bar character
   attr_t  fcAttr = this->barColor,    // full-cell-character color attribute
           fracAttr = fcAttr,          // color attribute for fractional bar cell
           bAttr ;                     // working bar attribute

   //* Establish the origin-point cell, direction in which bar grows, step *
   //* direction/distance between bars, color attribute and endpoint.      *
   switch ( this->chType )
   {
      case ctCenterLeft:
         wpo = { this->wpOrig.ypos, short(this->wpOrig.xpos + 1) } ;
         hStep = this->barSpace + 1 ;
         lastCol = this->wpOrig.xpos + this->hCells ;
         break ;
      case ctCenterRight:
         wpo = { this->wpOrig.ypos, short(this->wpOrig.xpos - 1) } ;
         hStep = -(this->barSpace + 1) ;
         lastCol = this->wpOrig.xpos - this->hCells ;
         fcAttr = InvertColorAttribute ( fcAttr ) ;
         break ;
      case ctLowerLeft:       // silence the compiler warning
      case ctLowerRight:      // silence the compiler warning
      case ctUpperLeft:       // silence the compiler warning
      case ctUpperRight:      // silence the compiler warning
      case ctLowerCenter:     // silence the compiler warning
      case ctUpperCenter:     // silence the compiler warning
      case ctCartesian:       // silence the compiler warning
         break ;
   }

   //* Display as much of the data as will fit on the horizontal axis *
   for ( int32_t indx = this->dataOffset ; indx < this->dataCount ; ++indx )
   {
      wp = wpo ;                       // position of bar-graph character
      curVal = this->dataPtr[indx] ;   // get a value

      //* Number of display divisions needed to represent the value *
      if ( curVal > 0.0 )
         curDiv = this->rngDiv * (curVal / this->maxVal) ;
      else if ( curVal < 0.0 )
         curDiv = this->rngDiv * (std::abs(curVal) / std::abs(this->minVal)) ;
      else
         curDiv = 0.0 ;
      //* Range limiting: must be <= 'maxDiv'.      *
      //* No minimum is specified for centered axes.*
      if ( curDiv > this->maxDiv ) curDiv = this->maxDiv ;

      //* Replace axis character with first block of bar.*
      // Programmer's Note: There is no character to accurately represent 
      // less than half a cell for for this position. Therefore, for all 
      // values in this group are represented by a half-cell character.
      if ( curVal > 0.0 )
      {
         vStep = -1 ;                     // direction of bar growth
         fracAttr = this->attrPtr != NULL ? this->attrPtr[indx] : this->barColor ;
         fracAttr = InvertColorAttribute ( fracAttr ) ; // color attribute for cell
         bChar = vBlock[3] ;              // half-block character
         if ( curDiv < (cellDIV / 2) )    // subtract divisions for half-cell
            curDiv = 0.0 ;
         else
            curDiv -= 4.0 ;
      }
      else if ( curVal < 0.0 )
      {
         vStep = 1 ;                      // direction of bar growth
         fracAttr = this->attrPtr != NULL ? this->attrPtr[indx] : this->negColor ;
         bChar = vBlock[3] ;              // half-block character
         if ( curDiv < (cellDIV / 2) )    // subtract divisions for half-cell
            curDiv = 0.0 ;
         else
            curDiv -= 4.0 ;
      }
      else  // curVal==0.0
      {
         vStep = ZERO ;                   // direction of bar growth
         fracAttr = this->barColor ;      // color attribute for cell
         bChar = this->gStyle == ncltDUAL ? wcsHORIZd : wcsHORIZb ;
      }
      //* Special test: If 'bTips', replace axis character ONLY *
      //* if no whole cells in bar.                             *
      if ( (curVal == 0.0) || !this->bTips || (curDiv < cellDIV) )
         this->dp->WriteChar ( wp, bChar, fracAttr ) ; // replace the axis character
      wp.ypos += vStep ;                  // advance to next vertical cell

      //* Set color attributes for the cell *
      if ( this->attrPtr != NULL )     // sequenced color attribute
         bAttr = fracAttr = this->attrPtr[indx] ;
      else
      {
         if ( curVal > 0.0 )
            fracAttr = bAttr = this->barColor ;
         else
            fracAttr = bAttr = this->negColor ;
      }
      //* If narrow bar AND moving leftward, invert 'bAttr'.*
      if ( (this->barWidth != cellDIV) && (hStep < ZERO) )
         bAttr = InvertColorAttribute ( bAttr ) ;

      fFract = modf ( curDiv, &fWhole ) ; // separate whole and fractional divisions
      iWhole = short(fWhole) ;            // whole divisions
      iFract = iWhole % cellDIV ;         // divisions in fractional cell
      iWhole /= cellDIV ;                 // whole cells
      if ( fFract > 0.5 )                 // round up to next division
      {  //* If this yields a whole cell *
         if ( ++iFract >= cellDIV )
         { ++iWhole ; iFract = ZERO ; }
      }
      iFract -= 1 ;                       // convert fractional cell to index
      //* If entire bar was displayed on axis line, *
      //* then display of tip would be redundant.   *
      if ( this->bTips && (iWhole == ZERO) )
         iFract = -1 ;

      //* If bar grows downward, mirror the fractional cell. *
      if ( (vStep > ZERO) && (iFract >= ZERO && iFract < cellDIV) )
      {
         iFract = InvertFractionalCell ( iFract ) ;
         if ( ! this->bTips )
            fracAttr = InvertColorAttribute ( fracAttr ) ;
      }

      //*************************
      //* Draw the vertical bar *
      //*************************
      if ( curVal != 0.0 )
      {
         for ( short i = ZERO ; i < iWhole ; ++i )
         {
            if ( ! this->bTips )
               this->dp->WriteChar ( wp, fcChar, bAttr ) ;
            wp.ypos += vStep ;
         }
         if ( (iFract >= ZERO) || (this->bTips && iWhole > ZERO) )
         {
            bChar = this->bTips ? vTipChar : vBlock[iFract] ;
            this->dp->WriteChar ( wp, bChar, fracAttr ) ;
         }
      }

      wpo.xpos += hStep ;                 // advance to next bar position
      //* If last column of chart has been filled, we're done *
      if ( ((hStep > ZERO) && (wpo.xpos > lastCol)) ||
           ((hStep < ZERO) && (wpo.xpos < lastCol)) )
         break ;
   }

}  //* End MapVerticalCentered() *

//*************************
//* MapHorizontalCentered *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* Draw the dataset as horizontal bars extending left and right from the      *
//* the vertical axis.                                                         *
//* For chart styles: ctLowerCenter, ctUpperCenter.                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Chart::MapHorizontalCentered ( void )
{
   double curVal,                      // data value to be mapped
          curDiv,                      // display divisions representing 'curVal'
          fWhole,                      // number of divisions (whole cells)
          fFract ;                     // number of divisions (fractional cell)
   short  iWhole,                      // number of filled cells (integer)
          iFract,                      // divisions in the fractional cell (integer)
          vStep,                       // step direction/distance between bars
          hStep,                       // step direction in which bar grows
          lastRow ;                    // last writable cell (opposite horizontal axis)
   winPos wp, wpo ;                    // cursor positioning
   wchar_t fcChar = vBlock[this->barWidth - 1], // full-cell bar character
           bChar ;                     // working bar character
   attr_t  fcAttr = this->barColor,    // full-cell-character color attribute
           fracAttr = fcAttr,          // color attribute for fractional bar cell
           bAttr ;                     // working bar attribute

   switch ( this->chType )
   {
      case ctLowerCenter:
         wpo = { short(this->wpOrig.ypos - 1), this->wpOrig.xpos } ;
         vStep = -(this->barSpace + 1) ;
         lastRow = this->wpOrig.ypos - this->vCells ;
         break ;
      case ctUpperCenter:
         wpo = { short(this->wpOrig.ypos + 1), this->wpOrig.xpos } ;
         vStep = this->barSpace + 1 ;
         fcAttr = InvertColorAttribute ( fcAttr ) ;
         lastRow = this->wpOrig.ypos + this->vCells ;
         break ;
      case ctLowerLeft:       // silence the compiler warning
      case ctLowerRight:      // silence the compiler warning
      case ctUpperLeft:       // silence the compiler warning
      case ctUpperRight:      // silence the compiler warning
      case ctCenterLeft:      // silence the compiler warning
      case ctCenterRight:     // silence the compiler warning
      case ctCartesian:       // silence the compiler warning
         break ;
   }

   //* Display as much of the data as will fit on the horizontal axis *
   for ( int32_t indx = this->dataOffset ; indx < this->dataCount ; ++indx )
   {
      wp = wpo ;                       // position of bar-graph character
      curVal = this->dataPtr[indx] ;   // get a value

      //* Number of display divisions needed to represent the value *
      if ( curVal > 0.0 )
         curDiv = this->rngDiv * (curVal / this->maxVal) ;
      else if ( curVal < 0.0 )
         curDiv = this->rngDiv * (std::abs(curVal) / std::abs(this->minVal)) ;
      else
         curDiv = 0.0 ;
      //* Range limiting: must be <= 'maxDiv'.      *
      //* No minimum is specified for centered axes.*
      if ( curDiv > this->maxDiv ) curDiv = this->maxDiv ;

      //* Replace axis character with first block of bar.*
      // Programmer's Note: There is no character to accurately represent 
      // less than half a cell for for this position. Therefore, for all 
      // values in this group are represented by a half-cell character.
      if ( curVal > 0.0 )
      {
         hStep = 1 ;                      // direction of bar growth
         fracAttr = this->attrPtr != NULL ? this->attrPtr[indx] : this->barColor ;
         fracAttr = InvertColorAttribute ( fracAttr ) ; // color attribute for cell
         bChar = hBlock[3] ;              // half-block character
         if ( curDiv < (cellDIV / 2) )    // subtract divisions for half-cell
            curDiv = 0.0 ;
         else
            curDiv -= 4.0 ;
      }
      else if ( curVal < 0.0 )
      {
         hStep = -1 ;                     // direction of bar growth
         fracAttr = this->attrPtr != NULL ? this->attrPtr[indx] : this->negColor ;
         bChar = hBlock[3] ;              // half-block character
         if ( curDiv < (cellDIV / 2) )    // subtract divisions for half-cell
            curDiv = 0.0 ;
         else
            curDiv -= 4.0 ;
      }
      else  // curVal==0.0
      {
         hStep = ZERO ;                   // direction of bar growth
         fracAttr = this->barColor ;      // color attribute for cell
         bChar = this->gStyle == ncltDUAL ? wcsVERTd : wcsVERTb ;
      }
      //* Special test: If 'bTips', replace axis character ONLY *
      //* if no whole cells in bar.                             *
      if ( (curVal == 0.0) || !this->bTips || (curDiv < cellDIV) )
         this->dp->WriteChar ( wp, bChar, fracAttr ) ; // replace the axis character
      wp.xpos += hStep ;                  // advance to next horizontal cell

      //* Set color attributes for the cell *
      if ( this->attrPtr != NULL )     // sequenced color attribute
         bAttr = fracAttr = this->attrPtr[indx] ;
      else
      {
         if ( curVal > 0.0 )
            fracAttr = bAttr = this->barColor ;
         else
            fracAttr = bAttr = this->negColor ;
      }
      //* If narrow bar AND moving downward, invert 'bAttr'.*
      if ( (this->barWidth != cellDIV) && (vStep < ZERO) )
         bAttr = InvertColorAttribute ( bAttr ) ;

      fFract = modf ( curDiv, &fWhole ) ; // separate whole and fractional divisions
      iWhole = short(fWhole) ;            // whole divisions
      iFract = iWhole % cellDIV ;         // divisions in fractional cell
      iWhole /= cellDIV ;                 // whole cells
      if ( fFract > 0.5 )                 // round up to next division
      {  //* If this yields a whole cell *
         if ( ++iFract >= cellDIV )
         { ++iWhole ; iFract = ZERO ; }
      }
      iFract -= 1 ;                       // convert fractional cell to index
      //* If entire bar was displayed on axis line, *
      //* then display of tip would be redundant.   *
      if ( this->bTips && (iWhole == ZERO) )
         iFract = -1 ;

      //* If bar grows leftward, mirror the fractional cell. *
      if ( (hStep < ZERO) && (iFract >= ZERO && iFract < cellDIV) )
      {
         iFract = InvertFractionalCell ( iFract ) ;
         if ( ! this->bTips )
            fracAttr = InvertColorAttribute ( fracAttr ) ;
      }

      //*************************
      //* Draw the vertical bar *
      //*************************
      if ( curVal != 0.0 )
      {
         for ( short i = ZERO ; i < iWhole ; ++i )
         {
            if ( ! this->bTips )
               this->dp->WriteChar ( wp, fcChar, bAttr ) ;
            wp.xpos += hStep ;
         }
         if ( (iFract >= ZERO) || (this->bTips && iWhole > ZERO) )
         {
            bChar = this->bTips ? vTipChar : hBlock[iFract] ;
            this->dp->WriteChar ( wp, bChar, fracAttr ) ;
         }
      }

      wpo.ypos += vStep ;                 // advance to next bar position
      //* If last column of chart has been filled, we're done *
      if ( ((vStep > ZERO) && (wpo.ypos > lastRow)) ||
           ((vStep < ZERO) && (wpo.ypos < lastRow)) )
         break ;
   }

}  //* End MapHorizontalCentered() *

//*************************
//*   MapCartesianPairs   *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* Draw the dataset as Y/X pairs extending from the axis origin (center).     *
//* For chart style: ctCartesian.                                              *
//*                                                                            *
//* The 'horizBars' flag is re-purposed for Cartesian charts.                  *
//*    'false' : source data are X/Y pairs                                     *
//*    'true;  : source data are Y/X pairs                                     *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Notes:                                                                     *
//* ------                                                                     *
//* 1) Ideally, the units in X should be equal to the units in Y,              *
//*    i.e. an aspect ratio of 1:1. Unfortunately, the character cells in a    *
//*    terminal window are not square. Nor are the pixels in either dimension  *
//*    an even multiple of the other dimension. This leads to a dilema:        *
//*    How do we make the display appear "square-ish"?                         *
//* 2) To make the above problem even more difficult, there is no _portable_   *
//*    way to query the terminal for the aspect ratio. All we can do is guess  *
//*    at an aspect ratio for the grid that approximates a square and map the  *
//*    data according to that ratio. Not a very satisfying solution.           *
//*    Typical font aspect ratio is between 1.5 and 1.7 vertical-to-horizontal.*
//*    This translates to approximately 2 vertical cells to 3 horizontal cells.*
//* 3) Note that any attempt to subdivide character cells to contain multiple  *
//*    datapoints would yield an unsightly mess (as well as being              *
//*    mathematically suspect).                                                *
//*                                                                            *
//* Mapping the Data                                                           *
//* ----------------                                                           *
//* To create a "square-ish" layout, we use the X/Y character-cell aspect ratio*
//* described above:                                                           *
//*  -- For each coordinate pair, the X coordinate is multiplied by 'xCONST'   *
//*     to achieve this squarish aspect ratio.                                 *
//*  -- The resulting coordinate pair is scaled to fit on the physical grid.   *
//* This is a lot of (slow) floating-point math, but we must assume that the   *
//* dataset is fairly small. The larger the dataset, the more likely that the  *
//* chart will just be a big, useless blob.                                    *
//*                                                                            *
//*                       ---  ---  ---  ---  ---                              *
//* Programmer's Note: The minimum- and maximum-value variables are initialized*
//* to "safe" values. It is possible (though unlikely) that all source values  *
//* will be above the minimum or all below the maximum. If this happened, the  *
//* data range would be artificially large, causing the displayed dataset      *
//* to be unnecessarily compacted. Again, this is unlikely.                    *
//******************************************************************************

void Chart::MapCartesianPairs ( void )
{
   winPos  wp ;                     // cell positioning
   bool    yxPair  = this->hBars ;  // source value-pair order

   //* Establish data range *
   for ( int32_t i = ZERO ; i < this->dataCount ; i += 2 )
   {
      if ( yxPair )     // Y/X pairs
      {
         if ( this->dataPtr[i] < this->cartRng.minValY )
            this->cartRng.minValY = this->dataPtr[i] ;
         if ( this->dataPtr[i] > this->cartRng.maxValY )
            this->cartRng.maxValY = this->dataPtr[i] ;
         if ( this->dataPtr[i+1] < this->cartRng.minValX )
            this->cartRng.minValX = this->dataPtr[i+1] ;
         if ( this->dataPtr[i+1] > this->cartRng.maxValX )
            this->cartRng.maxValX = this->dataPtr[i+1] ;
      }
      else              // X/Y pairs
      {
         if ( this->dataPtr[i] < this->cartRng.minValX )
            this->cartRng.minValX = this->dataPtr[i] ;
         if ( this->dataPtr[i] > this->cartRng.maxValX )
            this->cartRng.maxValX = this->dataPtr[i] ;
         if ( this->dataPtr[i+1] < this->cartRng.minValY )
            this->cartRng.minValY = this->dataPtr[i+1] ;
         if ( this->dataPtr[i+1] > this->cartRng.maxValY )
            this->cartRng.maxValY = this->dataPtr[i+1] ;
      }
   }
   //* Apply the aspect-ratio correction to 'minValX' and 'maxValX'.*
   this->cartRng.minValX *= xCONST ;
   this->cartRng.maxValX *= xCONST ;

   //**********************
   //* Plot the X/Y pairs *
   //**********************
   this->PlotCartPairs ( this->dataPtr, this->dataCount, 
                         this->cartChar, this->barColor, false ) ;

}  //* End MapCartesianPairs() *

//***************************
//* OverlayCartesianDataset *
//***************************
//******************************************************************************
//* Superimpose an additional dataset onto an existing Cartesian chart.        *
//*                                                                            *
//* 1) 'cartData' pairs MUST be in the same order as the original dataset,     *
//*    either X/Y pairs or Y/X pairs.                                          *
//* 2) The absolute range of the data in X and Y must be less than or equal    *
//*    to the range of the original data. Datapoints which fall outside the    *
//*    established range i.e. off-the-chart, will be silently discarded.       *
//* 3) 'cartCount' is the number of VALUES in the array, NOT the number of     *
//*    data pairs.                                                             *
//* 4) 'cartAttr' will typically be a color which contrasts with the color     *
//*    used for the original plot so that the new data may be easily           *
//*    identified.                                                             *
//* 5) 'dType' is optional, and indicates the type of data referenced by the   *
//*    'cartData' parameter. 'dType' indicates double-precision floating-point *
//*    data by default. If 'cartData' points to data of another type, indicate *
//*    the type as a member of enum hartType.                                  *
//* 6) 'cartChar' is optional. By default, the character used for the original *
//*    data will also be used for the new data; however, if desired a different*
//*    character may be used. (See SetCartesianChar() method for details.)     *
//*                                                                            *
//* Input  : cartData : pointer to array of data values to be plotted,         *
//*                     arranged as X/Y pairs (or as Y/X pairs)                *
//*          cartCount: number of elements in 'cartData' array                 *
//*          cartAttr : color attribute for datapoint display                  *
//*          dType    : (optional, idtDouble by default) type of data          *
//*                     referenced by 'cartData'                               *
//*          cartChar : (optional, "dblDiamond" (0x25C8) by default) character *
//*                     to be displayed at each datapoint                      *
//*          refresh: (optional, 'false' by default)                           *
//*                   if 'false', setup is performed, but display not updated  *
//*                   if 'true',  after setup, update display immediately      *
//*                                                                            *
//* Returns: 'true'  if success,                                               *
//*          'false' if invalid parameter(s)                                   *
//******************************************************************************

bool Chart::OverlayCartesianDataset ( const void* cartData, int32_t cartCount,
                                      attr_t cartAttr, idataType dType, 
                                      wchar_t cartChar, bool refresh )
{
   const double *dblPtr = NULL ; // pointer to converted dataset
   bool status = false ;         // return value
//* TEMP */ this->dp->WriteChar ( this->wpOrig, cartChar, nc.gr, true ) ;

#if 1    // UNDER CONSTRUCTION - OverlayCartesianDataset()
   //* Validate the input *
   if ( (cartData != NULL) && (cartCount > ZERO) )
   {
      status = true ;         // declare success

      //* Initialize the source data as an array of *
      //* double-precision floating point values.   *
      this->ConvertData ( cartData, cartCount, dType, dblPtr ) ;

      //**********************
      //* Plot the X/Y pairs *
      //**********************
      this->PlotCartPairs ( dblPtr, cartCount, cartChar, cartAttr, true ) ;

      if ( refresh )
         this->dp->RefreshWin () ;
   }
#endif

   return status ;

}  //* End OverlayCartesianDataset() *

//*************************
//*     PlotCartPairs     *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* Plot the Cartesian datapoints according to caller's parameters.            *
//*                                                                            *
//* Called by MapCartesianPair() to plot the original dataset.                 *
//* Called by OverlayCartesianDataset() to plot secondary dataset(s).          *
//*                                                                            *
//* Input  : dPtr    : pointer to an array of double-precision values,         *
//*                    arranged as X/Y pairs (or as Y/X pairs)                 *
//*          dCnt    : number of values in 'dPtr' array                        *
//*          pntChar : character to display at each datapoint                  *
//*          pntAttr : color attribute for 'pntChar'                           *
//*          truncate: 'false' plot all data points                            *
//*                    'true'  silently discard data points which fall outside *
//*                            the specified data range                        *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Chart::PlotCartPairs ( const double* dPtr, int32_t dCnt, 
                            wchar_t pntChar, attr_t pntAttr, bool truncate )
{
   #define DEBUG_CART_PLOT (0)         // For debugging only

   double  xVal, yVal,                 // current coordinate pair
           fWhole, fFract,             // whole and fractional cells
           xpCells = double(this->hCells / 2),  // available cells in X
           xnCells = double(this->hCells / 2),
           ypCells = double(this->vCells / 2),  // available cells in Y
           ynCells = double(this->vCells / 2) ;
           if ( (this->vCells % 2) == ZERO )    // adjust cell count in positive Y
              ypCells -= 1.0 ;
           if ( (this->hCells % 2) == ZERO )    // adjust cell count in positive X
              xpCells -= 1.0 ;
   winPos  wp ;                        // cell positioning
   short   yOrig = this->wpOrig.ypos,  // local copy of chart origin Y
           xOrig = this->wpOrig.xpos,  // local copy of chart origin X
           tCell = yOrig - (this->vCells / 2),  // topmost available display cell
           bCell = yOrig + (this->vCells / 2),  // bottom available display cell
           lCell = xOrig - (this->hCells / 2),  // leftmost available display cell
           rCell = xOrig + (this->hCells / 2) ; // rightmost available display cell
           if ( (this->vCells % 2) == ZERO )
              ++tCell ;
           if ( (this->hCells % 2) == ZERO )
              --rCell ;
   bool    yxPair  = this->hBars ;     // source value-pair order

   #if DEBUG_CART_PLOT != 0
   gString gx ;
   ofstream ofs( "./A_mcp.txt", ofstream::out | ofstream::trunc ) ;
   if ( ofs.is_open() )
   {
      gx.compose( "MapCartesianPairs\n-----------------"
                  "\nwpOrig: %hd/%hd"
                  "\nthis->vCells: %hd this->hCells: %hd"
                  "\ntCell: %2hd"
                  "\nbCell: %2hd"
                  "\nlCell: %2hd"
                  "\nrCell: %2hd"
                  "\nminValX:%5.1lf" 
                  "\nmaxValX:%5.1lf" 
                  "\nminValY:%5.1lf" 
                  "\nmaxValY:%5.1lf"
                  "\nxpCells:% 2.1lf xnCells:% 2.1lf"
                  "\nypCells:% 5.1lf ynCells:% 5.1lf",
                  &this->wpOrig.ypos, &this->wpOrig.xpos, &this->vCells, &this->hCells,
                  &tCell, &bCell, &lCell, &rCell,
                  &this->cartRng.minValX, &this->cartRng.maxValX,
                  &this->cartRng.minValY, &this->cartRng.maxValY,
                  &xpCells, &xnCells, &ypCells, &ynCells ) ;
      ofs << gx.ustr() << endl ;
   }
   #endif   // DEBUG_CART_PLOT

   for ( int32_t i = ZERO ; i < dCnt ; i += 2 )
   {
      if ( yxPair )     // Y/X pairs
      {
         yVal = dPtr[i] ;
         xVal = dPtr[i+1] * xCONST ;
      }
      else              // X/Y pairs
      {
         xVal = dPtr[i] * xCONST ;
         yVal = dPtr[i+1] ;
      }

      if ( xVal > 0.0 )
      {
         fFract = modf ( (xpCells * xVal / this->cartRng.maxValX), &fWhole ) ;
         wp.xpos = short(xOrig + fWhole - 1) ;
         if ( (fFract > 0.5) && (wp.xpos < rCell) )
            ++wp.xpos ;
      }
      else if ( xVal < 0.0 )
      {
         fFract = modf ( (xnCells * xVal / this->cartRng.minValX), &fWhole ) ;
         wp.xpos = short(xOrig - (fWhole - (fFract < -0.5 ? 1.0 : 0.0))) ;
      }
      else     // (xVal == 0.0)
         wp.xpos = xOrig ;

      #if DEBUG_CART_PLOT != 0
      if ( ofs.is_open() )
      {
         gx.compose( "\ni:% 3d y/x:%.1lf/%.1lf  yVal/xVal:%.1lf/%.1lf"
                     "\n   fWhole:%.1lf fFract:%.1lf = xpos:%hd",
                     &i, &dPtr[i+1], &dPtr[i], &yVal, &xVal,
                     &fWhole, &fFract, &wp.xpos ) ;
         if ( (wp.xpos < lCell) || (wp.xpos > rCell) )
            gx.append( " (xpos out-of-range)" ) ;
         ofs << gx.ustr() ;
      }
      #endif   // DEBUG_CART_PLOT

      if ( yVal > 0.0 )
      {
         fFract = modf ( (ypCells * yVal / this->cartRng.maxValY), &fWhole ) ;
         wp.ypos = short(yOrig - (fWhole + (fFract > 0.5 ? 1.0 : 0.0))) ;
      }
      else if ( yVal < 0.0 )
      {
         fFract = modf ( (ynCells * yVal / this->cartRng.minValY), &fWhole ) ;
         wp.ypos = short(yOrig + (fWhole - (fFract < -0.5 ? 1.0 : 0.0))) ;
      }
      else     // (yVal == 0.0)
         wp.ypos = yOrig ;

      #if DEBUG_CART_PLOT != 0
      if ( ofs.is_open() )
      {
         gx.compose( "\n   fWhole:%.1lf fFract:%.1lf = ypos:%hd",
                     &fWhole, &fFract, &wp.ypos ) ;
         if ( (wp.ypos < tCell) || (wp.ypos > bCell) )
            gx.append( " (ypos out-of-range)" ) ;
         ofs << gx.ustr() ;
      }
      #endif   // DEBUG_CART_PLOT

      #if DEBUG_CART_PLOT == 0   // PRODUCTION
      if ( ! truncate ||
           (truncate && 
            ((wp.ypos <= bCell) && (wp.ypos >= tCell) &&
             (wp.xpos <= rCell) && (wp.xpos >= lCell))) )
      this->dp->WriteChar ( wp, pntChar, pntAttr ) ;
      #else    // DEBUG_CARTPLOT - Plot out-of-range datapoints in a constrasting color.
      if ( ((wp.ypos > bCell) || (wp.ypos < tCell) ||
           (wp.xpos > rCell) || (wp.xpos < lCell)) )
         this->dp->WriteChar ( wp, pntChar, nc.gr ) ;
      else
         this->dp->WriteChar ( wp, pntChar, pntAttr ) ;
      #endif   // U/C
   }

   #if DEBUG_CART_PLOT != 0
   if ( ofs.is_open() )
   {
      ofs << endl ;
      ofs.close() ;
   }
   #endif   // DEBUG_CART_PLOT

   #undef DEBUG_CART_PLOT
}  //* End PlotCartPairs() *

//*************************
//*     AudibleShift      *
//*************************
//******************************************************************************
//* Enable or disable audible alert in ShiftData() method.                     *
//*                                                                            *
//* Input  : enable: 'true'  to enable alert                                   *
//*                  'false' to disable alert                                  *
//*                                                                            *
//* Returns: 'true' if alert enabled, 'false' if disabled                      *
//******************************************************************************

bool Chart::AudibleShift ( bool enable )
{
   this->audible = enable ;

   return ( this->audible ) ;

}  //* End AudibleShift() *

//*************************
//*   SetCartesianChar    *
//*************************
//******************************************************************************
//* Set the display character for the datapoints of a Cartesian chart.         *
//* Any single-column, printable, non-whitespace character may be specified,   *
//* but be aware that for best results, the character should relatively small  *
//* compared to the size of the character cell, and should be centered in the  *
//* cell. By default, the '◈' character, codepoint U+25C8 is used.             *
//*                                                                            *
//* The standard bullet characters and similar would be good choices:          *
//* U+25CF (●) black-circle                U+25CB (○) white-circle             *
//* U+26AB (⚫) medium-black-circle         U+26AA (⚪) medium-white-circle      *
//* U+23FA (⏺) black-circle-for-record     U+26AC (⚬) medium-small-white-circle*
//* U+2219 (∙) bullet operator (math)      U+25E6 (◦) white bullet             *
//* U+2022 (•) medium-small-black-circle   U+2660 (♠) spade                    *
//* U+25A0 (■) black-square                U+2665 (♥) heart                    *
//* U+25FC (◼) black-medium-square         U+2666 (♦) diamond                  *
//* U+25FE (◾) black-medium-small-square   U+2663 (♣) club                     *
//* U+25AA (▪) black-small-square          U+263A (☺) smiley face              *
//* U+2699 (⚙) gear symbol                 U+263B (☻) smiley face              *
//*                                                                            *
//* Input  : cartchar : character to be displayed at each datapoint            *
//*                                                                            *
//* Returns: 'true'  if success                                                *
//*          'false' if not a single-column printing character                 *
//******************************************************************************

bool Chart::SetCartesianChar ( wchar_t cartchar )
{
   bool status = false ;      // return value

   //* Verify that the specified character is both a single-column *
   //* character AND that it is a printing character.              *
   gString gs( &cartchar ) ;
   if ( (gs.gscols() == 1) && isgraph(cartchar) )
   {
      if ( cartchar != this->cartChar )
      {
         this->cartChar = cartchar ;         // set the new chararacter
         if ( this->chType == ctCartesian )  // redraw the chart
            this->MapCartesianPairs () ;
      }
      status = true ;
   }
   return status ;

}  //* End SetCartesianChar() *

//*************************
//*       ShiftData       *
//*************************
//******************************************************************************
//* Interact with user to shift the data visible in the chart window forward   *
//* and backward through the data array.                                       *
//* An audible alert (beep) is generated if unable to perform user-specified)  *
//* specified shift. The audible alert can be disabled via AudibleShift().     *
//*                                                                            *
//* If all data are currently displayed, returns immediately.                  *
//* Otherwise, returns when an un-handled keycode is received.                 *
//*                                                                            *
//* This method translates the defined keycodes (defaults listed below) into   *
//* parameters used by the private ShiftData() method, below.                  *
//*                                                                            *
//* If the 'sdPtr' argument is not provided, then the following default        *
//* shift-operation/keycode combinations are used.                             *
//*    Shift Operation           Keycode              Key type                 *
//*    -----------------------   -------------------------------               *
//*    top of data             : Home key (sbFirstPage)                        *
//*    end of data             : End key  (sbLastPage)                         *
//*    next page of data       : PageDown (sbNextPage)                         *
//*    previous page of data   : PageUp   (sbPrevPage)                         *
//*     1 step toward end      : RightArrow           wktFUNKEY                *
//*     1 step toward top      : LeftArrow            wktFUNKEY                *
//*     5 steps toward end     : Shift + RightArrow   wktFUNKEY                *
//*     5 steps toward top     : Shift + LeftArrow    wktFUNKEY                *
//*    10 steps toward end     : Ctrl + RightArrow    wktFUNKEY                *
//*    10 steps toward top     : Ctrl + LeftArrow     wktFUNKEY                *
//*                                                                            *
//* Input  : wkey : (by reference) receives the unhandled                      *
//*                 keycode that triggered return                              *
//*          sdCnt: (optional, ZERO by default)                                *
//*                 If specified, this is the number of objects in the         *
//*                 'sdPtr' array.                                             *
//*          sdPtr: (optional, NULL pointer by default)                        *
//*                 If specifed, this is a pointer to an array of ShiftDef     *
//*                 objects which define the valid shift options and           *
//*                 associated keycodes. See documentation for examples.       *
//*                                                                            *
//* Returns: index of first currently-displayed data item                      *
//*          'wkey' contains the unhandled key input. Note:                    *
//*          if all data currently displayed, wkey.key==null                   *
//******************************************************************************

int32_t Chart::ShiftData ( wkeyCode& wkey, short sdCnt, ShiftDef* sdPtr )
{
   int32_t    shiftCnt = ZERO ;        // parameters for call to private shiftData()
   ShiftBlock shiftBlk = sbNoShift ;
   short indx ;                        // index into sdPtr array
   bool dfltKeys = false,              // 'true' if default key definitions used
        done = false ;                 // loop control

   //* If all data are currently displayed, return immediately.*
   //* For these chart types, all data are currently displayed *
   //* because data have been scaled to fit the grid.          *
   if ( (this->hBars && this->dataCount <= this->vCells)  ||
        (!this->hBars && this->dataCount <= this->hCells) ||
        (this->chType == ctCartesian) ) 
   {
      wkey = { nckNULLCHAR, wktERR } ;
      done = true ;
   }

   //* Create the keycode-to-shift-operation map *
   if ( ! done && (sdPtr == NULL) )
   {
      sdCnt = 10 ;
      sdPtr = new ShiftDef[sdCnt]
      { //  wk.key     wk.type     sb           sc
         { {nckHOME,   wktFUNKEY}, sbFirstPage, ZERO },
         { {nckEND,    wktFUNKEY}, sbLastPage,  ZERO },
         { {nckPGDOWN, wktFUNKEY}, sbNextPage,  ZERO },
         { {nckPGUP,   wktFUNKEY}, sbPrevPage,  ZERO },
         { {nckRIGHT,  wktFUNKEY}, sbNoShift,     1  },
         { {nckLEFT,   wktFUNKEY}, sbNoShift,    -1  },
         { {nckSRIGHT, wktFUNKEY}, sbNoShift,     5  },
         { {nckSLEFT,  wktFUNKEY}, sbNoShift,    -5  },
         { {nckCRIGHT, wktFUNKEY}, sbNoShift,    10  },
         { {nckCLEFT,  wktFUNKEY}, sbNoShift,   -10  },
      } ;
      dfltKeys = true ;
   }

   while ( ! done )
   {
      shiftCnt = ZERO ;          // reset the call parameters
      shiftBlk = sbNoShift ;

      //* Get user input *
      this->dp->GetKeyInput ( wkey ) ;

      //* If valid keycodes are defined locally, we silently remap     *
      //* certain keycodes that the user might confuse with valid keys.*
      //* If caller defined the key list, remapping does not apply.    *
      if ( dfltKeys )
      {
         if      ( wkey.key == nckDOWN )     wkey.key = nckRIGHT ;   // toward bottom
         else if ( wkey.key == nckSDOWN )    wkey.key = nckSRIGHT ;
         else if ( wkey.key == nckADOWN )    wkey.key = nckSRIGHT ;
         else if ( wkey.key == nckCDOWN )    wkey.key = nckCRIGHT ;
         else if ( wkey.key == nckARIGHT )   wkey.key = nckCRIGHT ;
         else if ( wkey.key == nckASDOWN )   wkey.key = nckPGDOWN ;
         else if ( wkey.key == nckACDOWN )   wkey.key = nckPGDOWN ;
         else if ( wkey.key == nckCSDOWN )   wkey.key = nckPGDOWN ;
         else if ( wkey.key == nckCSRIGHT )  wkey.key = nckPGDOWN ;
         else if ( wkey.key == nckASRIGHT )  wkey.key = nckPGDOWN ;
         else if ( wkey.key == nckACRIGHT )  wkey.key = nckPGDOWN ;
         else if ( wkey.key == nckUP )       wkey.key = nckLEFT ;    // toward top
         else if ( wkey.key == nckSUP )      wkey.key = nckSLEFT ;
         else if ( wkey.key == nckAUP )      wkey.key = nckSLEFT ;
         else if ( wkey.key == nckCUP )      wkey.key = nckCLEFT ;
         else if ( wkey.key == nckALEFT )    wkey.key = nckCLEFT ;
         else if ( wkey.key == nckASUP )     wkey.key = nckPGUP ;
         else if ( wkey.key == nckACUP )     wkey.key = nckPGUP ;
         else if ( wkey.key == nckCSUP )     wkey.key = nckPGUP ;
         else if ( wkey.key == nckCSLEFT )   wkey.key = nckPGUP ;
         else if ( wkey.key == nckASLEFT )   wkey.key = nckPGUP ;
         else if ( wkey.key == nckACLEFT )   wkey.key = nckPGUP ;
      }

      //* Scan for matching keycode *
      for ( indx = ZERO ; indx < sdCnt ; ++indx )
      {
         if ( (wkey.type == sdPtr[indx].wk.type) &&
              (wkey.key  == sdPtr[indx].wk.key) )
         {
            if ( (sdPtr[indx].sb == sbFirstPage) ||
                 (sdPtr[indx].sb == sbLastPage)  ||
                 (sdPtr[indx].sb == sbNextPage)  ||
                 (sdPtr[indx].sb == sbPrevPage) )
            {
               shiftBlk = sdPtr[indx].sb ;
               break ;
            }
            //* Set specific shift count *
            else if ( (sdPtr[indx].sb == sbNoShift) && 
                      (sdPtr[indx].sc != ZERO) )
            {
               shiftCnt = sdPtr[indx].sc ;
               break ;
            }
            //* Else, invalid ShiftDef definition, return to caller.*
            else
               done = true ;
         }
      }

      //* If no keycode match, return to caller.*
      if ( indx == sdCnt )
         done = true ;

      //* Execute the shift, beep if error returned.*
      if ( ! done )
      {
         if ( (this->shiftData ( shiftCnt, shiftBlk )) && this->audible )
            this->dp->UserAlert () ; // call user a dumbguy
      }
   }  // while()

   if ( dfltKeys && sdPtr != NULL )    // release local dynamic allocation
      delete [] sdPtr ;

   return ( this->dataOffset ) ;       // return current data offset (index)

}  //* End ShiftData() *

//*************************
//*       shiftData       *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* Shift the data display forward or backward by the specified amount.        *
//*                                                                            *
//* IMPORTANT NOTE: This method assumes that some of the data are not currently*
//* displayed. DO NOT call this method if all data are currently displayed.    *
//*                                                                            *
//* Members of enum ShiftBlock:                                                *
//* ---------------------------                                                *
//* sbFirstPage: Start display with the first element of the data array and    *
//*              continue until chart is filled or end of data. If already on  *
//*              first page or if all data are currently displayed, this       *
//*              command will have no effect.                                  *
//* sbLastPage : Fill the chart starting at the offset which will allow the    *
//*              last element of the data array to be displayed. If already on *
//*              last page, or if all data are currently displayed, this       *
//*              command will have no effect.                                  *
//* sbNextPage : Shift the currently-displayed data out of the frame and       *
//*              display the next full page of data. If end of data is already *
//*              displayed, this command has no effect.                        *
//* sbPrevPage : Shift the currently-displayed data out of the frame and       *
//*              display the previous full page of data. If start of data is   *
//*              already displayed, this command has no effect.                *
//* sbNoShift  : Ignore this argument and use the shift value specified by     *
//*              'shiftCnt'. (This is the default value for 'shiftBlk'.)       *
//*                                                                            *
//*                                                                            *
//* Input  : shiftCnt : if a positive value, shift toward tail of data         *
//*                     if a negative value, shift toward head of data         *
//*          shiftBlk : (optional, sbNoShift by default)                       *
//*                     If specified, 'shiftCnt' is ignored and data shift is  *
//*                     calculated based on specified member of enum ShiftBlock*
//*                                                                            *
//* Returns: 'false' if shift successful                                       *
//*          'true'  if shift failed                                           *
//******************************************************************************

bool Chart::shiftData ( int32_t shiftCnt, ShiftBlock shiftBlk )
{
   int32_t oldOffset = this->dataOffset ; // remember current offset
   int32_t avail = ZERO,      // number of elements available in specified direction
           bCells = ZERO ;    // number of cells in bar direction
   if ( this->hBars )         // horizontal bars
      bCells = this->vCells ;
   else                       // vertical bars
      bCells = this->hCells ;

   bool shiftError = false ;  // return value

   //* If block-shift not specified, shift index by 'shiftCnt' elements *
   if ( (shiftBlk == sbNoShift) && (shiftCnt != ZERO) )
   {
      //* Un-displayed elements in selected direction *
      if ( shiftCnt > ZERO )
         avail = this->dataCount - this->dataOffset - bCells ;
      else
         avail = this->dataOffset ;

      //* If un-displayed elements >= requested shift,   *
      //* shift by specified number of elements.         *
      //* Otherwise, set shift to display remaining data.*
      if ( avail > ZERO )
      {
         if ( avail >= abs ( shiftCnt ) )
            this->dataOffset = this->dataOffset + shiftCnt ;
         else
         {
            if ( shiftCnt > ZERO )
               this->dataOffset = this->dataCount - bCells ;
            else
               this->dataOffset = ZERO ;
         }
      }
   }

   //* If block shift was specified. Calculate new start index *
   else if ( shiftBlk != sbNoShift )
   {
      if ( shiftBlk == sbFirstPage )
         this->dataOffset = ZERO ;
      else if ( shiftBlk == sbLastPage )
         this->dataOffset = this->dataCount - bCells ;
      else if ( shiftBlk == sbNextPage )
      {
         avail = this->dataCount - this->dataOffset - bCells ;
         if ( avail > ZERO )
         {
            if ( avail >= bCells )
               this->dataOffset += bCells ;
            else
               this->dataOffset = this->dataCount - bCells ;
         }
      }
      else if ( shiftBlk == sbPrevPage )
      {
         avail = this->dataOffset ;
         if ( avail > ZERO )
         {
            if ( avail >= bCells )
               this->dataOffset -= bCells ;
            else
               this->dataOffset = ZERO ;
         }
      }
   }

   //* If data are to be shifted *
   if ( this->dataOffset != oldOffset )
   {
      this->ClearGrid () ;             // clear and re-draw the chart grid
      this->MapData2Grid () ;          // map the data into the grid

      #if DEBUG_CONFIG != 0 && DEBUG_SHIFT != 0 // Report shift positioning
      bCells = this->dataOffset + bCells - 1 ;
      gString gs( "Offsets %03d -> %03d of %03d items.",
                  &this->dataOffset, &bCells, &this->dataCount ) ;
      this->dp->WriteString ( wpShift, gs, nc.grB ) ;
      #endif   // DEBUG_CONFIG && DEBUG_SHIFT

      this->dp->RefreshWin () ;        // make changes visible
   }
   else
      shiftError = true ;

   return shiftError ;

}  //* End shiftData() *

//*************************
//*     DrawAxisLabel     *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* Draw the specified axis label (or other text) at the specified position.   *
//*                                                                            *
//* Input  : label  : text to be written                                       *
//*          wpl    : cursor position for text                                 *
//*          justify: (optional, tjLeft by default) member of enum txtJust:    *
//*                   tjLeft   : left-justified, write starting at 'wpl'       *
//*                   tjCenter : center-justified, center text on 'wpl'        *
//*                   tjRight  : right-justified, end text at 'wpl'            *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Chart::DrawAxisLabel ( const char* label, const winPos& wpl, txtJust justify )
{
   gString gs( label ) ;      // text formatting
   winPos  wp = wpl ;         // cursor position

   if ( justify == tjCenter )
      wp.xpos -= (gs.gscols() / 2) ;
   else if ( justify == tjRight )
      wp.xpos -= (gs.gscols() - 1) ;
   this->dp->WriteParagraph ( wp, gs, this->dColor ) ;

}  //* DrawAxisLabel() *

//*************************
//*   GetHeaderPosition   *
//*************************
//******************************************************************************
//* Get position and size of header area.                                      *
//* The vertical dimension EXCLUDES the row reserved for the axis label.       *
//*                                                                            *
//* Input  : hdrPos : (by reference) receives Y/X position for upper-left      *
//*                   corner of target area                                    *
//*          rows   : (by reference) receives number of display rows           *
//*          cols   : (by reference) receives number of display columns        *
//*                                                                            *
//* Returns: 'true' if successful                                              *
//*          'false' if target area not defined (rows == ZERO)                 *
//******************************************************************************

bool Chart::GetHeaderPosition ( winPos& hdrPos, short& rows, short& cols )
{
   bool status = false ;      // return value

   hdrPos = { ZERO, ZERO } ;  // initialize caller's parameters
   rows = cols  = ZERO ;

   if ( (rows = this->yOffset - 1) > ZERO )
   {
      //* Y/X position *
      hdrPos = this->wpBase ;
      if ( this->bdrFlag )    // if border defined, inset by one
      { ++hdrPos.ypos ; ++hdrPos.xpos ; }

      cols = this->dispCols ;
      if ( this->bdrFlag )    // if border defined, reduce width by two
         cols -= 2 ; 

      status = true ;
   }
   return status ;

}  //* End GetHeaderPosition() *

//*************************
//*   GetFooterPosition   *
//*************************
//******************************************************************************
//* Get position and size of footer area.                                      *
//* This area EXCLUDES the row reserved for the axis label.                    *
//*                                                                            *
//* Input  : ftrPos : (by reference) receives Y/X position for upper-left      *
//*                   corner of target area                                    *
//*          rows   : (by reference) receives number of display rows           *
//*          cols   : (by reference) receives number of display columns        *
//*                                                                            *
//* Returns: 'true' if successful                                              *
//*          'false' if target area not defined (rows == ZERO)                 *
//******************************************************************************

bool Chart::GetFooterPosition ( winPos& ftrPos, short& rows, short& cols )
{
   bool status = false ;      // return value

   ftrPos = { ZERO, ZERO } ;  // initialize caller's parameters
   rows = cols  = ZERO ;

   if ( (rows = this->yFooter - 1) > ZERO )
   {
      //* Y/X position *
      ftrPos = { short(this->wpBase.ypos + this->dispRows - this->yFooter 
                       + (this->bdrFlag ? 0 : 1)),
                 short(this->wpBase.xpos + (this->bdrFlag ? 1 : 0)) } ;

      cols = this->dispCols ;
      if ( this->bdrFlag )    // if border defined, reduce width by two
         cols -= 2 ; 

      status = true ;
   }
   return status ;

}  //* End GetFooterPosition() *

//*************************
//*    DrawHeaderText     *
//*************************
//******************************************************************************
//* Clear the header-text area and write the specified text to the header area.*
//*                                                                            *
//* Note: To clear the area without writing new text, 'txt' may be a NULL      *
//*       pointer or a pointer to an empty string ("").                        *
//*                                                                            *
//* Input  : txt    : text to be written                                       *
//*          attr   : color attribute for text                                 *
//*          refresh: (optional, 'false' by default)                           *
//*                   if 'false', do not refresh the display after writing     *
//*                   if 'true',  refresh the display (make changes visible)   *
//*                                                                            *
//* Returns: 'true' if successful                                              *
//*          'false' if target area not defined                                *
//******************************************************************************

bool Chart::DrawHeaderText ( const char* txt, attr_t txtAttr, bool refresh )
{
   winPos wpHead ;                  // position of target area
   short  trgRows, trgCols ;        // dimensions of target area
   bool  status = false ;           // return value
   this->GetHeaderPosition ( wpHead, trgRows, trgCols ) ;

   if ( trgRows )
   {
      //* Clear the target area. *
      gString gs( " " ) ;
      gs.padCols( trgCols ) ;
      gs.append( L'\n' ) ;
      winPos wp = wpHead ;
      for ( short i = ZERO ; i < trgRows ; ++i )
         wp = this->dp->WriteParagraph ( wp, gs, txtAttr ) ;

      //* Write the new text *
      if ( txt != NULL && txt[0] != NULLCHAR )
         this->dp->WriteParagraph ( wpHead, txt, txtAttr ) ;

      if ( refresh )
         this->dp->RefreshWin () ;
      status = true ;
   }
   return status ;

}  //* End DrawHeaderText() *

//*************************
//*    DrawFooterText     *
//*************************
//******************************************************************************
//* Clear the footer-text area and write the specified text to the footer area.*
//*                                                                            *
//* Note: To clear the area without writing new text, 'txt' may be a NULL      *
//*       pointer or a pointer to an empty string ("").                        *
//*                                                                            *
//* Input  : txt    : text to be written                                       *
//*          attr   : color attribute for text                                 *
//*          refresh: (optional, 'false' by default)                           *
//*                   if 'false', do not refresh the display after writing     *
//*                   if 'true',  refresh the display (make changes visible)   *
//*                                                                            *
//* Returns: 'true' if successful                                              *
//*          'false' if target area not defined                                *
//******************************************************************************

bool Chart::DrawFooterText ( const char* txt, attr_t txtAttr, bool refresh )
{
   winPos wpFoot ;                  // position of target area
   short  trgRows, trgCols ;        // dimensions of target area
   bool  status = false ;           // return value
   this->GetFooterPosition ( wpFoot, trgRows, trgCols ) ;

   if ( trgRows > ZERO )
   {
      //* Clear the target area. *
      gString gs( " " ) ;
      gs.padCols( trgCols ) ;
      gs.append( L'\n' ) ;
      winPos wp = wpFoot ;
      for ( short i = ZERO ; i < trgRows ; ++i )
         wp = this->dp->WriteParagraph ( wp, gs, txtAttr ) ;

      //* Write the new text *
      if ( txt != NULL && txt[0] != NULLCHAR )
         this->dp->WriteParagraph ( wpFoot, txt, txtAttr ) ;

      if ( refresh )
         this->dp->RefreshWin () ;
      status = true ;
   }
   return status ;

}  //* End DrawFooterText() *

//*************************
//*  DrawExtendedFooter   *
//*************************
//******************************************************************************
//* Add text to the footer area at the specified offset.                       *
//* See GetFooterPosition().                                                   *
//* This method is provided as a convenience for creating custom footers.      *
//* For example, display of instructions or explanations.                      *
//* Only the start position (pos) is verified, so please use extra care when   *
//* positioning and sizing the data to be displayed.                           *
//*                                                                            *
//* Input  : pos    : (by reference) start position                            *
//*                   Y/X offset from footer base position                     *
//*                   Y offset must be >= ZERO and within                      *
//*                            the target window                               *
//*                   X offset Must be >= ZERO and within                      *
//*                            the target window                               *
//*          txt    : text to be written. Line breaks are                      *
//*                   indicated by newline '\n'.                               *
//*          attr   : color attribute for text                                 *
//*          refresh: (optional, 'false' by default)                           *
//*                   if 'false', do not refresh the display                   *
//*                               after writing                                *
//*                   if 'true',  refresh the display                          *
//*                               (make changes visible)                       *
//* Returns: Y/X offset at end of text written.                                *
//*          Returns Y/X==0/0 if 'pos' out-of-range                            *
//******************************************************************************

winPos Chart::DrawExtendedFooter ( const winPos& pos, const char* txt,
                                   attr_t txtAttr, bool refresh )
{
   winPos wp( ZERO, ZERO ),         // return value
          wpFoot ;                  // position of footer area
   short  trgRows, trgCols,         // dimensions of footer area
          dlgRows, dlgCols ;        // dimensions of dialog window

   //* Get dimensions of dialog window *
   this->dp->GetDialogDimensions ( dlgRows, dlgCols ) ;

   //* Get footer position within the dialog *
   this->GetFooterPosition ( wpFoot, trgRows, trgCols ) ;

   //* Verify that starting position is at or below the *
   //* footer area, and inside the dialog borders.      *
   if ( (pos.ypos >= wpFoot.ypos) && (pos.ypos <= (dlgRows - 2)) &&
        (pos.xpos >= 1) && (pos.xpos <= (dlgCols - 2)) )
   {
      wp = this->dp->WriteParagraph ( pos, txt, txtAttr, refresh ) ;
   }
   return ( wp ) ;

}  //* End DrawExtendedFooter() *

//*************************
//*  DrawHorizontalLine   *
//*************************
//******************************************************************************
//* Draw a horizontal line across the display area. If the line intersects a   *
//* border, a visual connection will be made.                                  *
//* The line must be at or below the footer position. See GetFooterPosition(). *
//*                                                                            *
//* Input  : yOffset: offset in Y from footer position                         *
//*                   must be >= ZERO and must be within the chart area        *
//*          lType  : line type  (member of enum ncLineType, excl. ncltVERT)   *
//*          lAttr  : color attribute for text                                 *
//*          refresh: (optional, 'false' by default)                           *
//*                   if 'false', do not refresh the display                   *
//*                               after writing                                *
//*                   if 'true',  refresh the display                          *
//*                               (make changes visible)                       *
//*                                                                            *
//* Returns: 'true'  if successful                                             *
//*          'false' if 'pos' out-of-range                                     *
//******************************************************************************

bool Chart::DrawHorizontalLine ( short yOffset, ncLineType lType,
                                 attr_t lAttr, bool refresh )
{
   winPos wpFoot ;                  // position of footer area
   short  trgRows, trgCols ;        // dimensions of footer area
   bool  status = false ;           // return value
   this->GetFooterPosition ( wpFoot, trgRows, trgCols ) ;

   //* Validate the Y offset *
   if ( (trgRows > ZERO) && (yOffset >= ZERO) && (yOffset < trgRows) )
   {
      //* Validate the line type and silently correct errors.*
      if ( (lType == ncltHORIZ) || (lType == ncltVERT) )
         lType = ncltSINGLE ;

      //* Line is the full width of the defined chart area, and       *
      //* "connects" with the right and left borders (if encountered).*
      LineDef ld { ncltHORIZ, lType, 
                   short(wpFoot.ypos + yOffset), 
                   short(this->localDlg ? ZERO : this->wpBase.xpos), 
                   short(this->localDlg ? this->ldCols : this->dispCols), 
                   lAttr } ;
      this->dp->DrawLine ( ld ) ;
      if ( refresh )
         this->dp->RefreshWin () ;

      status = true ;
   }
   return status ;

}  //* End DrawHorizontalLine() *

//*************************
//*       ClearArea       *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* Clear the display area, and if specified draw a border around the area.    *
//*                                                                            *
//* Note that for a locally-defined dialog window, the actual clearing         *
//* operation is unnecessary because it is a new window.                       *
//*                                                                            *
//* Programmer's Note: The NcDialog 'ClearArea()' and 'ClearLine()' methods    *
//* both refresh the display, but this method does not refresh.                *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Chart::ClearArea ( void )
{
   if ( ! this->localDlg )
   {
      gString gs( " " ) ;
      gs.padCols( this->dispCols ) ;
      gs.append( L'\n' ) ;
      winPos wp = this->wpBase ;
      for ( short r = ZERO ; r < this->dispRows ; ++r )
         wp = this->dp->WriteParagraph ( wp, gs, this->dColor ) ;
   }

   //* If specified, draw a border around the chart area *
   if ( this->bdrFlag )
   {
      this->dp->DrawBox ( this->wpBase.ypos, this->wpBase.xpos,
                          this->dispRows, this->dispCols, this->bdrColor,
                          this->titText, this->bStyle ) ;
   }

}  //* End ClearArea() *

//*************************
//*       ClearGrid       *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* Clear the area occupied by the chart grid and re-draw the grid.            *
//*                                                                            *
//* Programmer's Note: The NcDialog 'ClearArea()' and 'ClearLine()' methods    *
//* both refresh the display, but this method does not refresh.                *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Chart::ClearGrid ( void )
{
   //* Test size and position of cleared area. For debugging only.*
   #define DEBUG_CLEAR (0)

   //* Erase data from chart area including axes *
   winPos wp( short(this->wpBase.ypos + this->yOffset), 
              short(this->wpBase.xpos + this->xOffset + (this->bdrFlag ? 1 : 0)) ) ;
   short eog = wp.ypos + this->gridRows ;

   gString gs( " " ) ;
   gs.padCols( this->gridCols ) ;
   gs.append( L'\n' ) ;

   #if DEBUG_CLEAR != 0    // Position test only
   gs.replace( L' ', L'/', ZERO, false, true ) ;
   #endif   // DEBUG_CLEAR

   while ( wp.ypos < eog )
      wp = this->dp->WriteParagraph ( wp, gs, this->gColor ) ;

   #if DEBUG_CLEAR != 0
   this->dp->RefreshWin () ;
   chrono::duration<short>aWhile( 2 ) ;
   this_thread::sleep_for( aWhile ) ;
   #endif   // DEBUG_CLEAR

   this->DrawGrid () ;     // re-draw the axes

}  //* End ClearGrid() *

//*************************
//*        DumpCfg        *
//*************************
//******************************************************************************
//* Debugging Only: display data configuration                                 *
//* ---------------                                                            *
//*                                                                            *
//*                                                                            *
//* Input  : wpos   : start position for data display                          *
//*          refresh: (optional, 'false' by default)                           *
//*                   if 'true', refresh the display (make data visible)       *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Chart::DumpCfg ( const winPos& wpos, bool refresh )
{
#if DEBUG_CONFIG != 0
   winPos wp = wpos ;
   gString gsOut( 
      "wpBase(%02hd,%02hd)\n"
      "wpOrig(%02hd,%02hd)\n"
      "dispRows:%02hd\n"
      "dispCols:%02hd\n"
      "yOffset :%02hd\n"
      "xOffset :%02hd\n"
      "yFooter :%02hd\n"
      "barWidth:%02hd\n"
      "barSpace:%02hd\n"
      "chType  :%s\n",
      &this->wpBase.ypos, &this->wpBase.xpos,
      &this->wpOrig.ypos, &this->wpOrig.xpos,
      &this->dispRows, &this->dispCols,
      &this->yOffset, &this->xOffset, &this->yFooter, 
      &this->barWidth, &this->barSpace,
      (this->chType == ctLowerLeft   ? "LL" :
       this->chType == ctUpperLeft   ? "UL" :
       this->chType == ctLowerRight  ? "LR" :
       this->chType == ctUpperRight  ? "UR" :
       this->chType == ctLowerCenter ? "LC" :
       this->chType == ctUpperCenter ? "UC" :
       this->chType == ctCenterLeft  ? "CL" :
       this->chType == ctCenterRight ? "CR" : "CT") ) ;
   wp = this->dp->WriteParagraph ( wp, gsOut, nc.gy ) ;

   wp.ypos = wpos.ypos ;
   wp.xpos += 15 ;
   gsOut.compose( "vCells:%02hd\n"
                  "hCells:%02hd\n"
                  "minVal:%.3lf\n"
                  "maxVal:%.3lf\n"
                  "rngVal:%.3lf\n"
                  "meaVal:%.3lf\n"
                  "medVal:%.3lf\n"
                  "verDiv:%.3lf\n"
                  "horDiv:%.3lf\n"
                  "perDiv:%.3lf\n",
                  &this->vCells, &this->hCells,
                  &this->minVal, &this->maxVal, &this->rngVal, 
                  &this->meaVal, &this->medVal, &this->verDiv, 
                  &this->horDiv, &this->perDiv ) ;
   wp = this->dp->WriteParagraph ( wp, gsOut, nc.gy ) ;

   wp.ypos = wpos.ypos ;
   wp.xpos += 15 ;
   gsOut.compose( "minDiv:%.3lf\n"
                  "maxDiv:%.3lf\n"
                  "rngDiv:%.3lf\n",
                  &this->minDiv, &this->maxDiv, &this->rngDiv ) ;
//   wp = this->dp->WriteParagraph ( wp, gsOut, nc.gy ) ;
   wpShift = this->dp->WriteParagraph ( wp, gsOut, nc.gy ) ;

   if ( refresh )
      this->dp->RefreshWin () ;

#endif   // DEBUG_CONFIG
}  //* End DumpCfg() *

//*************************
//*      GetVersion       *
//*************************
//******************************************************************************
//* Returns the version number of the Chart class.                             *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: pointer to const string                                           *
//******************************************************************************

const char* Chart::GetVersion ( void )
{

   return ( ChartVersion ) ;

}  //* End GetVersion


//*** ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ***
//*** ---  ---  ---  ---  Non-member Methods  ---  ---  ---  ---  ---  ---  ***
//*** ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ***

//*************************
//* InvertFractionalCell  *
//*************************
//******************************************************************************
//* Non-member method:                                                         *
//* ------------------                                                         *
//* Invert the index into the block-character arrays, hBlock[] and vBlock[].   *
//* This allows a visual inversion of the filled vs. empty pixes withing the   *
//* character cell.                                                            *
//*                                                                            *
//* When used in conjunction with the mirrored color attribute, it allows for  *
//* fractionally-filled cells used to terminate a bar of the graph.            *
//*                                                                            *
//* Input  : blkIndx : index into character array referencing character to     *
//*                    be inverted. NOTE: blkIndx should be strictly controlled*
//*                    in the range of 0-6 (inclusive). The arrays are exactly *
//*                    eight(8) elements AND offset 7 is the full-cell block   *
//*                    character.                                              *
//*                                                                            *
//* Returns: index of block character which mirrors the character refrenced    *
//*          by blkIndx                                                        *
//******************************************************************************

static short InvertFractionalCell ( short blkIndx )
{
   short invIndx ;

   switch ( blkIndx )
   {
      case 0: invIndx = 6 ; break ;
      case 1: invIndx = 5 ; break ;
      case 2: invIndx = 4 ; break ;
      case 3: invIndx = 3 ; break ;
      case 4: invIndx = 2 ; break ;
      case 5: invIndx = 1 ; break ;
      case 6: invIndx = 0 ; break ;
      default: invIndx = blkIndx ; break ;   // should never happen
   } ;

   return invIndx ;
}  //* End InvertFractionalCell() *

//*************************
//* InvertColorAttribute  *
//*************************
//******************************************************************************
//* Non-member method:                                                         *
//* ------------------                                                         *
//* For the specified color attribute return a color attribute with the        *
//* foreground and background colors swapped.                                  *
//*                                                                            *
//* When used in conjunction with the mirrored block character, it allows for  *
//* fractionally-filled cells used to terminate a bar of the graph.            *
//*                                                                            *
//* IMPORTANT NOTE: This methods uses intimate knowledge of the way color      *
//* attributes are encoded within the ncurses world, specifically as used by   *
//* the NcDialog API. All inversions have been manually tested and visually    *
//* verified. In the unlikely event that the ncurses library changes the way   *
//* it handles color attributes, this method may break.                        *
//*                                                                            *
//* Bit Definitions for a 32-bit character/attribute value (7-bit ASCII).      *
//* This is known as 'attr_t'. (see curses.h)                                  *
//* Note that 'wide' characters are combined with their color attribute        *
//* internally by the ncurses library; however, the color attribute bits       *
//* are the same for both narrow and wide characters.                          *
//*  BIT  31    = unused    -----------+                                       *
//*  BIT  30    = vertical             |                                       *
//*  BIT  29    = top                  |                                       *
//*  BIT  28    = right                +-- we make no use of these             *
//*  BIT  27    = low                  |   definitions in our implementation   *
//*  BIT  26    = left                 |                                       *
//*  BIT  25    = horizontal           |                                       *
//*  BIT  24    = "protected" mode ----+                                       *
//*  BIT  23    = invisible (background color is also used for foreground)     *
//*  BIT  22    = alternate character set (with line draw characters, etc.)    *
//*  BIT  21    = bold foreground against default background                   *
//*  BIT  20    = dim                                                          *
//*  BIT  19    = blinking (not functional under most terminal implementations)*
//*  BIT  18    = reverse (foreground/background swap)                         *
//*  BIT  17    = underline                                                    *
//*  BIT  16    = standout (foreground-to-background, with fg & bg bolded)     *
//*  BITS 15-8  = color pair number  (8, 16, 64, 256.  see init_pair() )       *
//*  BITS 7-0   = character code                                               *
//*                                                                            *
//* Refer to ISO 6429 for a description of color support for terminals.        *
//*                                                                            *
//* In this method, we simply toggle the 'Reverse' bit.                        *
//* In some cases we may swap the 8-bit color-pair sequence, and/or toggle     *
//* the 'Bold' bit.                                                            *
//* IMPORTANT NOTE: Inverting BOLD colors will likely fail due to inconsistent *
//*                 combinations of BOLD and REVERSE.                          *
//*                                                                            *
//* Input  : normColor : non-inverted color attribute                          *
//*                                                                            *
//* Returns: color attribute with foreground and background swapped.           *
//******************************************************************************

static attr_t InvertColorAttribute ( attr_t normColor )
{
   const attr_t mask = ~ncrATTR ;
   attr_t invColor ;

   if ( normColor & ncrATTR )
   {
      invColor = normColor & mask ;
   }
   else
   {
      invColor = normColor | ncrATTR ;
   }
   return invColor ;

}  //* End InvertColorAttribute() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

