//******************************************************************************
//* File       : Chart.hpp                                                     *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2020-2021 Mahlon R. Smith, The Software Samurai *
//*                 GNU GPL copyright notice below                             *
//* Date       : 01-Jul-2021                                                   *
//* Version    : (see AppVersion string)                                       *
//*                                                                            *
//* Description: Class definition and for the "Chart" class.                   *
//*              This is a widget built on the NcDialog API by the same author.*
//*              Draw a bar chart representing the specified data.             *
//*                                                                            *
//******************************************************************************
//* Copyright Notice:                                                          *
//* This program is free software: you can redistribute it and/or modify it    *
//* under the terms of the GNU General Public License as published by the Free *
//* Software Foundation, either version 3 of the License, or (at your option)  *
//* any later version.                                                         *
//*                                                                            *
//* This program is distributed in the hope that it will be useful, but        *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
//* for more details.                                                          *
//*                                                                            *
//* You should have received a copy of the GNU General Public License along    *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.            *
//*                                                                            *
//*         Full text of the GPL License may be found in the Texinfo           *
//*         documentation for this program under 'Copyright Notice'.           *
//******************************************************************************

//****************
//* Header Files *
//****************
#include "GlobalDef.hpp"   //* NcDialog API family of classes

//***********************************************
//* Constant Data:                              *
//***********************************************

//***************
//* Definitions *
//***************

const short cellDIV = 8 ;     // divisions per character cell (vertical OR horizontal)

//* Default character used by Cartesian chart for mapping datapoints.*
const wchar_t dblDiamond = 0x25C8 ;   // '◈'

//* Specify text justification *
enum txtJust : short
{
   tjLeft,                    // Left-justified
   tjCenter,                  // Centered
   tjRight                    // Right-justified
} ;

//* Specifies the type of chart grid used to map the data *
enum chartType : short
{
   ctLowerLeft,               // Horizontal axis at the bottom  (default type)
                              // Vertical axis on the left
   ctLowerRight,              // Horizontal axis at the bottom
                              // Vertical axis on the right
   ctUpperLeft,               // Horizontal axis at the top
                              // Vertical axis on the left
   ctUpperRight,              // Horizontal axis at the top
                              // Vertical axis on the right
   ctLowerCenter,             // Horizontal axis at the bottom
                              // Vertical axis in the center (horizontal bars only)
   ctUpperCenter,             // Horizontal axis at the top
                              // Vertical axis in the center (horizontal bars only)
   ctCenterLeft,              // Horizontal axis in the center
                              // Vertical axis on the left (vertical bars only)
   ctCenterRight,             // Horizontal axis in the center
                              // Vertical axis on the right (vertical bars only)
   ctCartesian,               // Vertical and Horizontal cross in the center
                              // (not yet implemented)
} ;

//* Supported data types. The default value is "double". Other data types *
//* may be dynamically converted to doubles for processing and display.   *
//* Note that the actual range of any data type is hardware dependent;    *
//* however, most 32-bit and 64-bit systems support the precision shown.  *
enum idataType : short
{
   idtDouble = ZERO,          // Double-precision (64 bits) floating point (default)
   idtFloat,                  // Single-precision (32 bits) floating point
   idtByte_s,                 // 8-bit, signed
   idtByte_u,                 // 8-bit, unsigned
   idtShort_s,                // 16-bit, signed
   idtShort_u,                // 16-bit, unsigned
   idtInt_s,                  // 32-bit, signed
   idtInt_u,                  // 32-bit, unsigned
   idtLong_s,                 // 64-bit, signed (64-bit systems)
   idtLong_u,                 // 64-bit, signed (64-bit systems)
   idtLongLong_s,             // 64-bit, signed
   idtLongLong_u,             // 64-bit, signed
} ;

//* Definitions for shifting data on the chart forward/backward.*
//* See ShiftData() method for detailed description.            *
enum ShiftBlock : short 
{
   sbNoShift,           // no shift, parameter ignored
   sbFirstPage,         // display data from beginning of array
   sbLastPage,          // display last page (tail) of data
   sbNextPage,          // display next page of data (shift current data out)
   sbPrevPage,          // display previous page of data (shift current data out)
} ;

//* Initialization data for call to ShiftData() method.   *
//* ShiftData() optionally accepts a pointer to an array  *
//* of ShiftDef objects indicating the types of data      *
//* shift operations the user may access and the keypress *
//* associated with each type of shift.                   *
//* See ShiftData() prototype below for more information. *
class ShiftDef
{
   public:
   //* Keycode to activate the shift. This is keycode AND key type. *
   //* See NCursesKeyDef.hpp for the complete list.                 *
   wkeyCode   wk ;
   //* Type of shift (member of enum ShiftBlock). Set to sbNoShift  *
   //* if specific shift count is specified by 'sc' member.         *
   ShiftBlock sb ;
   //* If the 'sb' member == sbNoShift, specify the number of       *
   //* elements (character cells) to shift.                         *
   //* A positive value shifts data forward (toward end of data).   *
   //* A negative value shifts data backward (toward beginning).    *
   //* If 'sb' != sbNoShift, then this value is ignored.            *
   short      sc ;
} ;

//* Calculated range for each quadrant of Cartesian *
//* charts. Used to scale the data points.          *
class CartRange
{
   public:
   double minValX ;
   double maxValX ;
   double minValY ;
   double maxValY ;
} ;

//*****************************************
//** Initialization data for Chart-class **
//**     initialization constructor.     **
//*****************************************
class chartDef
{
   public:
   NcDialog *dPtr;            // pointer to dialog window in which to draw chart
                              // if null pointer, a new dialog will
   short  ulY ;               // upper left of chart area, Y
   short  ulX ;               // upper left of chart area, X
   short  rows ;              // number of rows in chart area
   short  cols ;              // number of columns in chart area
   short  yOff ;              // Y offset of grid within the display area
   short  xOff ;              // X offset of grid within the display area
   short  yFoot ;             // lines of freespace below grid
   short  barWidth ;          // bar width in "divisions" (1-8 divisions)
                              // (Note: fractional cell is always full cell width.)
   short  barSpace ;          // bar spacing 0 or 1 empty character cells between bars
   chartType chType ;         // X/Y grid type (see above)
   attr_t borderColor ;       // border color attribute
   attr_t titleColor ;        // title color attribute
   attr_t textColor ;         // basic text/background color
   attr_t boldColor ;         // bold text color
   attr_t gridColor ;         // color attribute for X/Y grid
   attr_t barColor ;          // default color attribute for bars of bar chart
                              // (see also 'attrPtr' member)
   attr_t negColor ;          // bar color for negative values (mixed positive and negative)

   int32_t dataCount ;        // number of elements in 'dataPtr' (and 'attrPtr' if specified)
   int32_t dataOffset ;       // index into 'dataPtr' array at which to begin display
   const void* dataPtr ;      // pointer to array of data values to be charted
   idataType dataType ;       // type of data pointed to by 'dataPtr' member
   attr_t *attrPtr ;          // (optional) array of attributes, one for each bar

   ncLineType borderStyle ;   // border style
   ncLineType gridStyle ;     // grid-line style
   const char* vaxisLabel ;   // vertical-axis label
   const char* haxisLabel ;   // horizontal-axis label
   const char* title ;        // (optional) title of the chart (displayed in top line of area)
   const char* headText ;     // (optional) text to write above chart
   const char* footText ;     // (optional) text to write below chart (legend, footnotes, etc.)
   bool horizBars ;           // 'true' if horizontal bars, 'false' if vertical bars
                              // (for Cartesian charts: 'false'==X/Y pairs, 'true'==Y/X pairs)
   bool barTips ;             // 'false'==display entire bar, 'true'==display tip of bar only
   bool border ;              // if 'true' draw border around chart area
} ;   // chartDef


//****************************
//** Chart-class Definition **
//****************************
class Chart
{
   public:

   //* Destructor. Return all resources to the system.           *
   //*                                                           *
   //* Input  : none                                             *
   //* Returns: nothing                                          *
   virtual ~Chart ( void ) ;

   //* Full-initialization constructor.                          *
   //* All data members set according to provided parameters.    *
   //*                                                           *
   //* Input  : cdef   : (by reference) a fully-initialized      *
   //*                   instance of the 'chartDef' class.       *
   //*          refresh: (optional, 'false' by default)          *
   //*                   if 'false', setup is performed, but     *
   //*                               display not updated         *
   //*                   if 'true',  after setup, update display *
   //*                               immediately                 *
   //* Returns: implicitly returns pointer to instantiated object*
   Chart ( const chartDef& cdef, bool refresh = false ) ;

   //* Refresh the display. This makes any changes visible.      *
   //*                                                           *
   //* Input  : none                                             *
   //* Returns: nothing                                          *
   void refresh ( void )
   {
      this->dp->RefreshWin () ;
   }

   //* Specify a new data set to be displayed *
   //bool SetChartData ( const double *dataPtr, short dataCount, bool refresh = false ) ;

   //* Specify a new array of color attributes for chart bars *
   //bool SetChartColors ( const attr_t *attrPtr, short attrCount, bool refresh = false ) ;

   //* Set the horizontal-axis or vertical-axis text *
   //bool SetAxisLabel ( bool vertical, const char* label ) ;

   //* Display the horizontal-axis or vertical-axis label *
   //bool DrawAxisLabel ( bool vertical, txtJust justify = tjLeft, const char* newLabel = NULL ) ;

   //* Clear the header-text area and write the specified text.  *
   //*                                                           *
   //* Input  : txt    : text to be written                      *
   //*          attr   : color attribute for text                *
   //*          refresh: (optional, 'false' by default)          *
   //*                   if 'false', do not refresh the display  *
   //*                               after writing               *
   //*                   if 'true',  refresh the display         *
   //*                               (make changes visible)      *
   //* Returns: 'true' if successful                             *
   //*          'false' if target area not defined (rows == ZERO)*
   bool DrawHeaderText ( const char* txt, attr_t txtAttr, bool refresh = false ) ;

   //* Clear the footer-text area and write the specified text.  *
   //*                                                           *
   //* Input  : txt    : text to be written                      *
   //*          attr   : color attribute for text                *
   //*          refresh: (optional, 'false' by default)          *
   //*                   if 'false', do not refresh the display  *
   //*                               after writing               *
   //*                   if 'true',  refresh the display         *
   //*                               (make changes visible)      *
   //* Returns: 'true' if successful                             *
   //*          'false' if target area not defined (rows == ZERO)*
   bool DrawFooterText ( const char* txt, attr_t txtAttr, bool refresh = false ) ;

   //* Add text to the footer area at the specified offset.      *
   //* See GetFooterPosition().                                  *
   //* This method is provided as a convenience for creating     *
   //* custom footers. For example, display of instructions or   *
   //* explanations. Only the start position (pos) is verified,  *
   //* so please use extra care when positioning and sizing the  *
   //* data to be displayed.                                     *
   //*                                                           *
   //* Input  : pos    : (by reference) start position           *
   //*                   Y/X offset from footer base position    *
   //*                   Y offset must be >= ZERO and within     *
   //*                            the target window              *
   //*                   X offset Must be >= ZERO and within     *
   //*                            the target window              *
   //*          txt    : text to be written. Line breaks are     *
   //*                   indicated by newline '\n'.              *
   //*          attr   : color attribute for text                *
   //*          refresh: (optional, 'false' by default)          *
   //*                   if 'false', do not refresh the display  *
   //*                               after writing               *
   //*                   if 'true',  refresh the display         *
   //*                               (make changes visible)      *
   //* Returns: Y/X offset at end of text written.               *
   //*          Returns Y/X==0/0 if 'pos' out-of-range           *
   winPos DrawExtendedFooter ( const winPos& pos, const char* txt,
                               attr_t txtAttr, bool refresh = false ) ;

   //* Draw a horizontal line across the display area. If the    *
   //* line intersects a border, a visual connection will be     *
   //* made. The line must be at or below the footer position.   *
   //* (See GetFooterPosition().)                                *
   //*                                                           *
   //* Input  : yOffset: offset in Y from footer position        *
   //*                   must be >= ZERO and must be within the  *
   //*                   chart area                              *
   //*          lType  : line type  (member of enum ncLineType,  *
   //*                    excluding. ncltVERT)                   *
   //*          lType  : line type (member of enum ncLineType)   *
   //*          lAttr  : color attribute for text                *
   //*          refresh: (optional, 'false' by default)          *
   //*                   if 'false', do not refresh the display  *
   //*                               after writing               *
   //*                   if 'true',  refresh the display         *
   //*                               (make changes visible)      *
   //* Returns: 'true'  if successful                            *
   //*          'false' if 'pos' out-of-range                    *
   bool DrawHorizontalLine ( short yOffset, ncLineType lType,
                             attr_t lAttr, bool refresh = false ) ;

   //* Get position and size of header area.                     *
   //*                                                           *
   //* Input  : hdrPos : (by reference) receives Y/X position    *
   //*                   for upper-left corner of target area    *
   //*          rows   : (by reference) receives number of       *
   //*                   display rows                            *
   //*          cols   : (by reference) receives number of       *
   //*                                  display columns          *
   //* Returns: 'true' if successful                             *
   //*          'false' if target area not defined (rows == ZERO)*
   bool GetHeaderPosition ( winPos& hdrPos, short& rows, short& cols ) ;

   //* Get position and size of footer area.                     *
   //*                                                           *
   //* Input  : ftrPos : (by reference) receives Y/X position    *
   //*                   for upper-left corner of target area    *
   //*          rows   : (by reference) receives number of       *
   //*                   display rows                            *
   //*          cols   : (by reference) receives number of       *
   //*                   display columns                         *
   //* Returns: 'true' if successful                             *
   //*          'false' if target area not defined (rows == ZERO)*
   bool GetFooterPosition ( winPos& ftrPos, short& rows, short& cols ) ;

   //* Enable or disable audible alert in ShiftData() method.    *
   //*                                                           *
   //* Input  : enable: 'true'  to enable alert                  *
   //*                  'false' to disable alert                 *
   //* Returns: 'true' if alert enabled, 'false' if disabled     *
   bool  AudibleShift ( bool enable ) ;

   //* Set the display character for the datapoints of a         *
   //* Cartesian chart.Any single-column, printable,             *
   //* non-whitespace character may be specified, but be aware   *
   //* that for best results, the character should relatively    *
   //* small compared to the size of the character cell, and     *
   //* should be centered in the cell.                           *
   //* By default, the '◈' character, codepoint U+25C8 is used.  *
   //* The standard bullet characters would be good alternate    *
   //* choices.                                                  *
   //*                                                           *
   //* Input  : cartChar : character to be displayed at each     *
   //*                     datapoint                             *
   //* Returns: 'true'  if success,                              *
   //*          'false' if not a 1-column, printing character    *
   bool  SetCartesianChar ( wchar_t cartChar ) ;

   //* Superimpose an additional dataset onto an existing        *
   //* Cartesian chart.                                          *
   //*                                                           *
   //* 1) 'cartData' pairs MUST be in the same order as the      *
   //*    original dataset, either X/Y pairs or Y/X pairs.       *
   //* 2) The absolute range of the data in X and Y must be less *
   //*    than or equal to the range of the original data.       *
   //*    Datapoints which fall outside the established range    *
   //*    i.e. off-the-chart, will be silently discarded.        *
   //* 3) 'cartCount' is the number of VALUES in the array, NOT  *
   //*    the number of data pairs.                              *
   //* 4) 'cartAttr' will typically be a color which contrasts   *
   //*    with the color used for the original plot so that the  *
   //*    new data may be easily identified.                     *
   //* 5) 'dType' is optional, and indicates the type of data    *
   //*    referenced by the 'cartData' parameter.                *
   //*    'dType' indicates double-precision floating-point data *
   //*    by default. If 'cartData' points to data of another    *
   //*    type, indicate the type as a member of enum hartType.  *
   //* 6) 'cartChar' is optional. By default, the character used *
   //*    for the original data will also be used for the new    *
   //*    data; however, if desired a different character may    *
   //*    be used.                                               *
   //*    (See SetCartesianChar() method above for details.)     *
   //*                                                           *
   //* Input  : cartData : pointer to array of data values to be *
   //*                     plotted, arranged as X/Y pairs        *
   //*                     (or as Y/X pairs)                     *
   //*          cartCount: number of elements in 'cartData' array*
   //*          cartAttr : color attribute for datapoint display *
   //*          dType    : (optional, idtDouble by default)      *
   //*                     type of data referenced by 'cartData' *
   //*          cartChar : (optional, "dblDiamond" (0x25C8) by   *
   //*                     default) character to be displayed    *
   //*                     at each datapoint                     *
   //*          refresh  : (optional, 'false' by default)        *
   //*                     if 'false', setup is performed, but   *
   //*                                 display not updated       *
   //*                     if 'true',  after setup, update       *
   //*                                 display immediately       *
   //* Returns: 'true'  if success,                              *
   //*          'false' if invalid parameter(s)                  *
   bool  OverlayCartesianDataset ( const void* cartData, int32_t cartCount,
                                   attr_t cartAttr, 
                                   idataType dType = idtDouble, 
                                   wchar_t cartChar = dblDiamond, bool refresh = false ) ;

   //* Interact with user to shift the data visible in the chart *
   //* window forward and backward through the data array.       *
   //*                                                           *
   //* If all data are currently displayed, returns immediately. *
   //* Otherwise, returns when an un-handled keycode is received.*
   //*                                                           *
   //* If the 'sdPtr' argument is not provided, then the         *
   //* following default shift/keycode combinations are used.    *
   //* Shift Operation           Keycode              Key type   *
   //* -----------------------   ------------------------------- *
   //* top of data             : Home key (sbFirstPage)          *
   //* end of data             : End key  (sbLastPage)           *
   //* next page of data       : PageDown (sbNextPage)           *
   //* previous page of data   : PageUp   (sbPrevPage)           *
   //*  1 step toward end      : RightArrow           wktFUNKEY  *
   //*  1 step toward top      : LeftArrow            wktFUNKEY  *
   //*  5 steps toward end     : Shift + RightArrow   wktFUNKEY  *
   //*  5 steps toward top     : Shift + LeftArrow    wktFUNKEY  *
   //* 10 steps toward end     : Ctrl + RightArrow    wktFUNKEY  *
   //* 10 steps toward top     : Ctrl + LeftArrow     wktFUNKEY  *
   //*                                                           *
   //* Input  : wkey : (by reference) receives the unhandled     *
   //*                 keycode that triggered return             *
   //*          sdCnt: (optional, ZERO by default)               *
   //*                 If specified, this is the number of       *
   //*                 objects in the 'sdPtr' array.             *
   //*          sdPtr: (optional, NULL pointer by default)       *
   //*                 If specifed, this is a pointer to an array*
   //*                 of ShiftDef objects which define the valid*
   //*                 shift options and associated keycodes.    *
   //*                 See documentation for examples.           *
   //*                                                           *
   //* Returns: index of first currently-displayed data item     *
   //*          'wkey' contains the unhandled key input. Note:   *
   //*          if all data currently displayed, wkey.key==null  *
   int32_t ShiftData ( wkeyCode& wkey, short sdCnt = ZERO, ShiftDef *sdPtr = NULL ) ;

   //* Returns a pointer to target dialog (either caller's       *
   //* dialog (redundant), or locally instantiated dialog).      *
   //*                                                           *
   //* Input  : none                                             *
   //* Returns: pointer to NcDialog object                       *
   NcDialog * GetDialog ( void )
   {
      return this->dp ;
   }

   //* Returns the version number of the Chart class.            *
   //*                                                           *
   //* Input  : none                                             *
   //* Returns: pointer to const string                          *
   const char* GetVersion ( void ) ;

   private:    // all other members are private

   //*******************
   //* Private Methods *
   //*******************
   Chart ( void ) ;                        //* default constructor (private)
   void reset ( void ) ;                   //* reset all data members to default values
   void ClearArea ( void ) ;               //* clear the display area (border if specified)
   void ClearGrid ( void ) ;               //* clear chart grid area, re-draw the grid.
   void Configure ( const chartDef& cdef );//* initialize all data members
   void ConvertData ( const void* vdPtr,   //* convert data to double-precision floating-point
                      int32_t dCount, idataType dType, const double*& trgPtr ) ;
   void SetStyle ( void ) ;                //* set the axis-line style for the chart
   void SetDimensions ( void ) ;           //* set length of X and Y axes
   void SetOrigin ( void ) ;               //* set the origin cell for the chart
   void DrawGrid ( bool labels = false,    //* draw the vertical and horizontal axes (and labels)
                   bool refresh = false ) ;
   void DrawAxisLabel ( const char* label, //* Draw axis label text
                        const winPos& wpl, txtJust justify = tjLeft ) ;
   void MapData2Grid ( void ) ;            //* Draw the data set onto the grid
   void MapVerticalBars ( void ) ;         //* Draw the dataset as vertical bars
   void MapVerticalCentered ( void ) ;     //* Draw the dataset extending above and below horiz. axis
   void MapHorizontalBars ( void ) ;       //* Draw the dataset as horizontal bars
   void MapHorizontalCentered ( void ) ;   //* Draw the dataset extending left and right from vert. axis
   void MapCartesianPairs ( void ) ;       //* Draw the dataset as X/Y pairs
   void PlotCartPairs ( const double* dPtr, int32_t dCnt, 
                        wchar_t pntChar, attr_t pntAttr, bool truncate ) ;
   bool shiftData ( int32_t shiftCnt, ShiftBlock shiftBlk = sbNoShift ) ;
   void DumpCfg ( const winPos& wpos,     //* Debugging Only: display data-member values
                  bool refresh = false ) ;

   //****************
   //* Data Members *
   //****************
   NcDialog *dp ;             //* pointer to target dialog window
   winPos   wpTerm ;          //* if local dialog defined, terminal-window offset
   winPos   wpBase ;          //* display-area base (upper-left corner)
   winPos   wpOrig ;          //* origin of chart grid (where Y and X axes connect)
   winPos   wpYlab ;          //* Y-axis label position
   winPos   wpXlab ;          //* X-axis label position
   txtJust  jYlab ;           //* text justification for Y-axis label
   txtJust  jXlab ;           //* text justification for X-axis label
   short    ldRows ;          //* if local dialog defined, dialog height
   short    ldCols ;          //* if local dialog defined, dialog width
   short    dispRows ;        //* number of rows in display area
   short    dispCols ;        //* number of columns in display area
   short    gridRows ;        //* number of rows occupied by grid (incl. axis)
   short    gridCols ;        //* number of columns occupied by grid (incl. axis)
   short    yOffset ;         //* vertical offset from top of display area to top of grid
   short    xOffset ;         //* horizontal offset from left of display area to left of grid
   short    yFooter ;         //* freespace below grid (including line for axis label)
   short    barWidth ;        //* bar width in "divisions" (8 divisions per character cell)
   short    barSpace ;        //* bar spacing 0 or 1 empty character cells between bars
   chartType chType ;         //* X/Y grid type (see above)
   short    vCells ;          //* number of vertical character cells in grid
   short    hCells ;          //* number of horizontal character cells in grid
   attr_t   bdrColor ;        //* border color attribute
   attr_t   titColor ;        //* title text color
   attr_t   barColor ;        //* default color for chart bars (but see 'attrPtr')
   attr_t   negColor ;        //* bar color for negative values (mixed positive and negative)
   attr_t   dColor ;          //* default text color (interior foreground/background)
   attr_t   bColor ;          //* bold text color
   attr_t   gColor ;          //* Y/X grid-line color
   int32_t  dataCount ;       //* number of elements in 'dataPtr' and 'attrPtr'
   int32_t  dataOffset ;      //* index of first element of 'dataPtr' to be displayed
   const double *dataPtr ;    //* pointer to array of data points (data pairs for Cartesian)
   const attr_t *attrPtr ;    //* pointer to array of color-attributes (optional)
   const char *vLabel ;       //* vertical-axis label
   const char *hLabel ;       //* horizontal-axis label
   const char *titText ;      //* title for chart area
   const char *hdrText ;      //* text written above the grid (if any)
   const char *ftrText ;      //* text written below the grid (if any)
   ncLineType bStyle ;        //* border line type
   ncLineType gStyle ;        //* grid line type

   double minVal ;            //* minimum data value
   double maxVal ;            //* maximum data value
   double rngVal ;            //* data range (max - min)
   double meaVal ;            //* arithmetic mean data value
   double medVal ;            //* median data value (calculated on demand)
   double verDiv ;            //* total vertical-axis divisions
   double horDiv ;            //* total horizontal-axis divisions
   double minDiv ;            //* number of divisions displayed for "minimum value"
   double maxDiv ;            //* number of divisions displayed for "maximum value"
   double rngDiv ;            //* range of available divisions (maxDiv - minBAR)
   double perDiv ;            //* data units represented by one division
   CartRange cartRng ;        //* Cartesian charts only: X and Y range for dataset
   wchar_t vaxisChar ;        //* vertical-axis character
   wchar_t haxisChar ;        //* horizontal-axis character
   wchar_t axisCross ;        //* character at axis origin (connect X and Y axis)
   wchar_t axisCap ;          //* axis-cap character (denotes reserved cell)
   wchar_t cartChar ;         //* character used to map datapoints on Cartesian chart 
   bool    dynoData ;         //* 'true' if dynamic allocation for converted data
   bool    localDlg ;         //* 'true' if dialog instantiated locally (false if caller provided)
   bool    bdrFlag ;          //* 'true' == draw border, 'false' == no border
   bool    hBars ;            //* 'true' == horizontal bars, 'false' == vertical bars
   bool    bTips ;            //* 'false'==display entire bar, 'true'==display tip of bar only
   bool    audible ;          //* 'true'==beep when data-shift fails, 'false'==silent
} ;   // Chart
