//******************************************************************************
//* File       : EcLog.cpp                                                     *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2020-2021 Mahlon R. Smith, The Software Samurai *
//*                 GNU GPL copyright notice below                             *
//* Date       : 11-Jun-2021                                                   *
//* Version    : (see AppVersion string)                                       *
//*                                                                            *
//* Description: Application log-file methods and general file management.     *
//*                                                                            *
//*                                                                            *
//******************************************************************************
//* Copyright Notice:                                                          *
//* This program is free software: you can redistribute it and/or modify it    *
//* under the terms of the GNU General Public License as published by the Free *
//* Software Foundation, either version 3 of the License, or (at your option)  *
//* any later version.                                                         *
//*                                                                            *
//* This program is distributed in the hope that it will be useful, but        *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
//* for more details.                                                          *
//*                                                                            *
//* You should have received a copy of the GNU General Public License along    *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.            *
//*                                                                            *
//*         Full text of the GPL License may be found in the Texinfo           *
//*         documentation for this program under 'Copyright Notice'.           *
//******************************************************************************

//****************
//* Header Files *
//****************
#include "Exercalc.hpp"    //* Class definition


//***************
//* Definitions *
//***************

//***************
//* Prototypes  *
//***************

//**********
//*  Data  *
//**********
extern InitNcDialog tdInit ;     // defined locally in Exercalc.cpp
extern InitNcDialog sdInit ;
extern InitNcDialog udInit ;

//* Container information for HTML logfile *
const char* const htmlContainerA =     // top of file (before LogID)
   "<!DOCTYPE HTML>\n"
   "<html>\n"
   "<head>\n"
   "<!-- " ;
const char* const htmlContainerB =     // timestamp template (after LogID)
   " -->\n"
   "<!-- Created:  %s  Exercalc, written by The Software Samurai -->\n" ;
const char* const htmlContainerC =     // top of file (after timestamp)
   "<meta charset=\"utf-8\" />  <!-- Import the global stylesheet -->\n"
   "<link rel=\"stylesheet\" href=\"infodoc-styles.css\" lang=\"en\" type=\"text/css\"/>\n"
   "</head>\n"
   "\n"
   "<body>\n" ;
const short HCD_LEN = 30 ;             // number of characters to compare
const char* const htmlContainerD =     // user data begins after this tag
   "<div class=\"curses_container\">\n" ;
const char* const htmlContainerE =     // terminate file
   "<br></div><!-- (end curses_container class) -->\n"
   "</body>\n"
   "</html>\n" ;
const char* const htmlEndUserData =    // marker for end of user data
   "<!-- End User Data -->\n" ;
const char* const htmlInterspace =     // space between records
   "<br>\n" ;

//* Template for encoding and decoding user stats.*
//* Used when writing and reading log file.       *
const wchar_t* const recMark = L"###: " ; // record identification token
const wchar_t* const lfstatsTemplateW = 
   L"###: %.3f:%.3f:%.3f:%.2f:%.2f:%.2f:%.2f:%.2f:%.2f:%.2f:"
    "%.3f:%.3f:%.3f:%.0f:%.2f:%.2f:%.3f:%.3f:%.3f:%02hd:%02hd:%llX" ;
const wchar_t* const lfstatsTemplateR = 
   L"###: %f:%f:%f:%f:%f:%f:%f:%f:%f:%f:%f:%f:%f:%f:%f:%f:%f:%f:%f:%hd:%hd:%llX" ;
const wchar_t* const htmlComment_Begin = L"<!-- " ;
const wchar_t* const htmlComment_End   = L" -->" ;
//* Number of encoding/decoding conversions that SHOULD occur.*
//* This is the number of data members in the UserData class. *
const short convCOUNT = 22 ;

//* Character definitions for drawing graphs *
//const short cellDIV = 8 ;     // character cell divided into eighths
const wchar_t hBlock[cellDIV] =
{
   0x0258F,    // ▏
   0x0258E,    // ▎
   0x0258D,    // ▍
   0x0258C,    // ▌
   0x0258B,    // ▋
   0x0258A,    // ▊
   0x02589,    // ▉
   0x02588,    // █
} ;
const wchar_t vBlock[cellDIV] =
{
   0x02581,    // ▁
   0x02582,    // ▂
   0x02583,    // ▃
   0x02584,    // ▄
   0x02585,    // ▅
   0x02586,    // ▆
   0x02587,    // ▇
   0x02588,    // █
} ;


//*************************
//*        ViewLog        *
//*************************
//******************************************************************************
//* Use 'less' utility to view log file.                                       *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true'  if view is successful                                     *
//*          'false' if file not found or inaccessible                         *
//******************************************************************************

bool Exercalc::ViewLog ( void )
{
   const char* const viewTemplate = "less -R -c \"%s\"" ;
   gString gs( this->cfg.logPath ) ;
   fmFType fType ;
   chrono::duration<short, std::milli>aMoment( 1000 ) ;  // pause
   bool status = false ;

   //* Verify that target file exists *
   if ( (this->TargetExists ( gs, fType )) && (fType == fmREG_TYPE) )
   {
      //* Create the shell-out command *
      gs.compose( viewTemplate, this->cfg.logPath ) ;

      //* Save display data for each dialog window *
      this->titleDlg->SetDialogObscured () ;
      this->statsDlg->SetDialogObscured () ;

      //* Call the system (dialog display data are saved automatically) *
      this->userDlg->ShellOut ( soX, gs.ustr() ) ;

      //* Restore the display *
      this->titleDlg->RefreshWin () ;
      this->statsDlg->RefreshWin () ;
   }
   else
   {
      this->UserMessage ( LogMsgs[this->cfg.lang][lmNoAccess], this->cfg.em, true ) ;
      this_thread::sleep_for( aMoment ) ;
      this->UserMessage ( "", this->cfg.em, true ) ;
   }
   return status ;

}  //* End ViewLog() *

//*************************
//*    LogfileCompose     *
//*************************
//******************************************************************************
//* Compose the proposed filespec for the log file.                            *
//*                                                                            *
//* 1) If 'logPath' is an empty string, create the default target filespec.    *
//* 2) If 'logPath' specifies a directory target append the default filename   *
//*    to the specified path.                                                  *
//* 3) If 'logPath' is a filename only (no '/' characters, then prepend the    *
//*    CWD path.                                                               *
//* 4) If 'logPath' is a full filespec an existing 'regular' file, OR if the   *
//*    target file does not exist, return the filespec unmodified.             *
//*                                                                            *
//* Input  : src  : (by reference) source filespec (if any)                    *
//*          trg  : (by reference) receives constructed filespec               *
//*                                                                            *
//* Returns: 'true' if successful  ('trg' contains the constructed filespec)   *
//*          'false' if error      ('trg' will contain the NULL string)        *
//******************************************************************************

bool Exercalc::LogfileCompose ( const gString&src, gString& trg )
{
   gString cwd ;              // path of current working directory
   fmFType fType ;            // filetype
   bool exists,               // 'true' if target exists
        status = true ;       // return value
   this->GetCWD ( cwd ) ;     // get the CWD

   //* If a non-empty path string *
   if ( src.gschars() > 1 )
   {
      if ( (src.find( L'/' )) >= ZERO )   // if a pathspec
         trg = src ;
      else                                // if a filename only
         trg.compose( "%S/%S", cwd.gstr(), src.gstr() ) ;
   }
   //* Default logfile name in current working directory (CWD) *
   else
      trg.compose( "%S/%s%s", cwd.gstr(), ecLog, txtExt ) ;

   //* Expand environment variables and resolve links.         *
   //* Note that the file must exist for the expansion to work.*
   this->Realpath ( trg ) ;

   //* Stat the file and if it exists, get filetype. *
   //* If a directory, append default filename.      *
   if ( (exists = this->TargetExists ( trg, fType )) && (fType == fmDIR_TYPE) )
      trg.append( "/%s%s", ecLog, txtExt ) ;
   else if ( exists && !(fType == fmREG_TYPE) )
   {
      trg.clear() ;        // clear caller's buffer
      status = false ;     // target is not a 'regular' file
   }

   return status ;

}  //* End LogfileCompose() *

//*************************
//*     LogfileCreate     *
//*************************
//******************************************************************************
//* Initialize the log file specified by the 'cfg.logPath' member.             *
//* 1) If the target file  does not exist, create it.                          *
//* 2) If the file DOES exist, delete its contents.                            *
//*    This is a dangerous operation, so ask user to verify.                   *
//*                                                                            *
//* Input  : logSpec : full filespec of file to be created                     *
//*                                                                            *
//* Returns: 'true' if target file initialized                                 *
//*          'false' if no file specified, or if unable to create the file     *
//*                  or if unable to write to the file                         *
//*                  or if user did not confirm file truncation                *
//******************************************************************************

bool Exercalc::LogfileCreate ( const gString& logSpec )
{
   gString gs ;                  // text formatting
   localTime timeStamp ;         // current local timestamp
   fmFType fType ;               // file type
   bool trunc  = true,           // 'true' if user confirmed file truncation OR
                                 //        if target does not exist
        status = false ;         // return value

   //* If the file exists, truncate its contents *
   if ( (this->TargetExists ( logSpec, fType )) )
   {
      //* Ask user to confirm that file is to be truncated.*
      if ( fType == fmREG_TYPE )
         trunc = this->Confirm ( confirmTruncate ) ;
      else  // do not allow deletion of 'special' files
         this->UserMessage ( LogMsgs[this->cfg.lang][lmNoRegular], this->cfg.em, true ) ;
   }

   //* If target does not exist OR if we have *
   //* user's permission, truncate the file.  *
   if ( trunc )
   {
      ofstream ofs( logSpec.ustr(), ofstream::out | ofstream::trunc ) ;
      if ( ofs.is_open() )
      {
         this->GetLocalTime ( timeStamp ) ;     // get system-local time
         //* If target is an HTML file *
         if ( (logSpec.find( htmlExt )) > ZERO )
         {
            ofs << htmlContainerA << LogID[this->cfg.lang] ;
            gs.compose( htmlContainerB, timeStamp.iso ) ;
            ofs << gs.ustr() << htmlContainerC << htmlContainerD
                << htmlEndUserData << htmlContainerE << endl ;
         }
         //* Else, target is a plain-text file *
         else
         {
            ofs << LogID[this->cfg.lang] 
                << "\n# Created: " << timeStamp.iso << "\n" << endl ;
         }
         ofs.close() ;
         status = true ;
      }
   }
   return status ;

}  //* End LogfileCreate() *

//*************************
//*     LogfileAppend     *
//*************************
//******************************************************************************
//* Append the currently displayed data record ('ud' member) to the log file   *
//* specified by 'cfg.logPath' member.                                         *
//*                                                                            *
//* User is asked to confirm the write and has the option of appending a       *
//* comment to the record.                                                     *
//*                                                                            *
//* If the displayed record is the same as the base record ('udb' member),     *
//* then the 'logPend' flag is reset to prevent the auto-save-on-exit from     *
//* performing a redundant save. Note however that we allow the user to        *
//* _explicitly_ save any record, even if the save is redundant.               *
//*                                                                            *
//* If the target file does not exist, create it.                              *
//*                                                                            *
//* Input  : confirm : (optional, 'true' by default)                           *
//*                    if 'true',  ask user for confirmation before saving     *
//*                    if 'false', save without user confirmation              *
//*                                                                            *
//* Returns: 'true' if data successfully written                               *
//*          'false' if: user aborts the save,                                 *
//*                      attempt to save already-saved base record             *
//*                      user does not have write access                       *
//******************************************************************************

bool Exercalc::LogfileAppend ( bool confirm )
{
   char  lineData[gsMAXBYTES] ;  // raw UTF-8 input
   gString gs,                   // text formatting
           gsSrc, gsTrg,         // source and target filespecs
           gsComment ;           // user comment
   tnFName fStats ;              // target file stats
   chrono::duration<short, std::milli>aMoment( 1500 ) ;  // pause
   bool    goodFile = false,     // 'true' if target exists and user has full access
           userConfirm = true,   // 'true' if user confirms (or confirmation not required)
           htmlTarget = false,   // 'true' if target file is HTML format
           isBaseRecord = false, // 'true' if record to be saved is base record
           success = false ;     // return value

   isBaseRecord = bool(this->ud == this->udb) ;

   //* Test whether target exists *
   gsTrg = this->cfg.logPath ;
   if ( (this->GetFileStats ( gsTrg, fStats )) == OK )
   {
      //* If user has full access *
      if ( (fStats.fType == fmREG_TYPE) && fStats.readAcc && fStats.writeAcc )
         goodFile = true ;
   }
   //* Target does not exist, create it now *
   else
      goodFile = this->LogfileCreate ( gsTrg ) ;

   //* If target exists AND if user has full access, *
   //* open the file and verify the file format.     *
   if ( goodFile )
   {
      ifstream ifs ( this->cfg.logPath, ifstream::in ) ;
      if ( ifs.is_open() )             // if input file open
      {
         ifs.getline ( lineData, gsMAXBYTES, NEWLINE ) ;
         if ( (ifs.good()) || (ifs.gcount()) > ZERO )
         {
            gs = lineData ;

            //* Test for HTML-formatted document *
            if ( (gs.find( "<!DOCTYPE HTML>" )) == ZERO )
            {
               //* Read and discard the "<html>" and "<head>" lines *
               ifs.getline ( lineData, gsMAXBYTES, NEWLINE ) ;
               ifs.getline ( lineData, gsMAXBYTES, NEWLINE ) ;
               //* The next line S/B a comment containing the LogID *
               ifs.getline ( lineData, gsMAXBYTES, NEWLINE ) ;
               gs = lineData ;
               gs.shiftChars( -5 ) ;   // discard: "<!-- "
               htmlTarget = true ;     // remember that target is HTML format
            }
            if ( (gs.compare( LogID[this->cfg.lang], true, LogID_CompChars )) != ZERO )
            {
               this->UserMessage ( LogMsgs[this->cfg.lang][lmBadFormat], 
                                   this->cfg.em, true ) ;
               this_thread::sleep_for( aMoment ) ;
               this->UserMessage ( "", this->cfg.bb ) ;
               goodFile = false ;
            }
         }
         ifs.close() ;
      }
   }

   //* If target file format validated AND confirmation    *
   //* required get user confirmation and optional comment.*
   if ( goodFile && confirm)
      userConfirm = this->lfaConfirm ( gsComment ) ;

   //*****************************
   //* Append record to log file *
   //*****************************
   if ( goodFile && userConfirm )
   {
      //* Capture and format the data record. Formatted record is stored *
      //* in dynamic memory allocation attached to 'udAlloc' member.     *
      // Programmer's Note: If capture fails, only the timestamp and a
      // warning message will be returned.
      this->LogfileFormatRecord ( this->ud, &gsComment, htmlTarget ) ;

      //* If target file is HTML, rename the target file as a backup file,*
      //* then copy contents to target until end of division ("</div>").  *
      //* If a plain-text file, we can skip this step and simply append   *
      //* the data to the target file.                                    *
      if ( htmlTarget )
      {
         //* Rename the target file *
         gsSrc.compose( "%S~", gsTrg.gstr() ) ;
         this->RenameFile ( gsTrg, gsSrc ) ;

         //* Open source and _new_ target *
         ifstream ifs ( gsSrc.ustr(), ifstream::in ) ;
         ofstream ofs ( gsTrg.ustr(), ofstream::out | ofstream::trunc ) ;
         if ( (ifs.is_open()) && (ofs.is_open()) )
         {
            //* Copy the header and user data *
            while ( true )
            {
               ifs.getline ( lineData, gsMAXBYTES, NEWLINE ) ;
               if ( (ifs.good()) || (ifs.gcount() > ZERO) )
               {
                  gs.compose( "%s\n", lineData ) ;
                  if ( (gs.find( htmlEndUserData, ZERO, true )) >= ZERO )
                     break ;
                  ofs.write( gs.ustr(), (gs.utfbytes() - 1) ) ;
               }
               else        // unexpected EOF
                  break ;
            }

            //* Close the target file *
            ofs.close() ;

            //* Close and Delete the backup file *
            ifs.close() ;
            this->DeleteFile ( gsSrc ) ;
         }
      }

      //* Open the target for append *
      ofstream ofs( gsTrg.ustr(), ofstream::out | ofstream::app ) ;
      if ( ofs.is_open() )
      {
         ofs << this->udAlloc ;

         //* For an HTML target, write the closing sequence *
         if ( htmlTarget )
            ofs << htmlInterspace << htmlEndUserData << htmlContainerE ;

         ofs.flush() ;
         ofs.close() ;
         success = true ;
         if ( isBaseRecord )     // reset the flag to disable auto-save-on-exit
            this->logPend = false ;
      }
   }

   //* Report the results *
   this->UserMessage ( LogMsgs[this->cfg.lang]
            [success ? lmGoodWrite : !userConfirm ? lmAbortWrite : lmNoCreate],
            this->cfg.em, true ) ;
   this_thread::sleep_for( aMoment ) ;
   this->UserMessage ( "", this->cfg.em, true ) ;

   return success ;

}  //* End LogfileAppend() *

//*************************
//*      lfaConfirm       *
//*************************
//******************************************************************************
//* Get user confirmation for writing a record to the logfile.                 *
//* User may optionally specify a comment to be written with the record data.  *
//*                                                                            *
//* Input  : gsComment : (by reference) receives user's comment text.          *
//*                                                                            *
//* Returns: 'true' if confirm, 'false' if abort                               *
//******************************************************************************

bool Exercalc::lfaConfirm ( gString& gsComment )
{
   bool conf = false ;           // return value

   gsComment.clear() ;           // initialize caller's comment buffer

   attr_t bColor = this->cfg.em,             // color attributes
          dColor = this->cfg.mf ;

   const short dlgROWS = udInit.dLines - this->userRows - this->pictRows - 1,
               dlgCOLS = udInit.dColumns,
               dlgYPOS = udInit.dYoffset + this->userMsg - 1,
               dlgXPOS = udInit.dXoffset ;

   //* Add the control objects *
   enum lfaControls : short { lfaSavePB = ZERO, lfaCancelPB, lfaCommTB, lfaItems } ;

   InitCtrl ic[lfaItems] = 
   {
   {  //* 'SAVE' pushbutton   - - - - - - - - - - - - - - - - - - -  lfaSavePB *
      dctPUSHBUTTON,                // type:      control type
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dlgROWS - 2),           // ulY:       upper left corner in Y
      8,                            // ulX:       upper left corner in X
      1,                            // lines:     control lines
      23,                           // cols:      control columns
      SaveDlgLabels[this->cfg.lang][sdlSave],   // dispText:  display text
      this->cfg.pn,                 // nColor:    non-focus color
      this->cfg.pf,                 // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[lfaCancelPB],             // nextCtrl:  link in next structure
   },

   {  //* 'CANCEL' pushbutton   - - - - - - - - - - - - - - - - -  lfaCancelPB *
      dctPUSHBUTTON,                // type:      control type
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[lfaSavePB].ulY,            // ulY:       upper left corner in Y
      short(ic[lfaSavePB].ulX + ic[lfaSavePB].cols + 4),// ulX: upper left corner in X
      1,                            // lines:     control lines
      23,                           // cols:      control columns
      SaveDlgLabels[this->cfg.lang][sdlCancel], // dispText:  display text
      this->cfg.pn,                 // nColor:    non-focus color
      this->cfg.pf,                 // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[lfaCommTB],               // nextCtrl:  link in next structure
   },

   {  //* 'COMMENT' Textbox  - - - - - - - - - - - - - - - - - - - - lfaCommTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[lfaSavePB].ulY - 2), // ulY:       upper left corner in Y
      2,                            // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      62,                           // cols:      control columns
      NULL,                         // dispText:  
      this->cfg.tn,                 // nColor:    non-focus color
      this->cfg.tf,                 // fColor:    focus color
      tbPrint,                      // filter:    all printing claracters
      SaveDlgLabels[this->cfg.lang][sdlComm], // label:
      1,                            // labY:      
      1,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   }, 
   } ;

   InitNcDialog dInit
   (
    dlgROWS,            // number of display lines
    dlgCOLS,            // number of display columns
    dlgYPOS,            // Y offset from upper-left of terminal
    dlgXPOS,            // X offset from upper-left of terminal
    NULL,               // dialog title
    ncltSINGLE,         // border line-style
    bColor,             // border color attribute
    dColor,             // interior color attribute
    ic                  // pointer to list of control definitions
   ) ;

   //* Adjust textbox label position *
   if ( this->cfg.rtl )
      ic[lfaCommTB].labX = ic[lfaCommTB].cols - 2 ;

   //* Save the underlying dialog *
   this->userDlg->SetDialogObscured () ;

   //* Instantiate the dialog *
   NcDialog *dp = new NcDialog ( dInit ) ;

   //* If user interface language is an RTL language, *
   //* set the internal NcDialog flag for RTL output. *
   if ( this->cfg.rtl )
   {
      dp->DrawLabelsAsRTL () ;
      dp->DrawContentsAsRTL ( lfaCommTB, true ) ;
   }

   if ( (dp->OpenWindow()) == OK )  // open the stats dialog
   {
      dp->SetDialogTitle ( SaveDlgLabels[this->cfg.lang][sdlTitle], bColor | ncuATTR ) ;

      //* Write the static text *
      winPos wp( 1, (this->cfg.rtl ? dlgCOLS - 3 : 2) ) ;
      wp = dp->WriteParagraph ( wp, SaveDlgLabels[this->cfg.lang][sdlInstr], 
                                dColor, false, this->cfg.rtl ) ;

      dp->RefreshWin () ;           // make everything visible

      uiInfo Info ;                 // user interface data returned here
      short  icIndex = ZERO ;       // index of control with input focus
      bool   done = false ;         // loop control

      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( icIndex == lfaSavePB )
               {
                  //* Get user's comment (if any) *
                  dp->GetTextboxText ( lfaCommTB, gsComment ) ;
                  //* For data captured as RTL, reverse it because *
                  //* it will be written to the log as LTR text.   *
                  if ( this->cfg.rtl )
                     gsComment.textReverse() ;
                  conf = true ;     // we have user's confirmation
#if 0    // TEMP - CAPTURE
dp->CaptureDialog ( "capturedlg.txt" ) ;
#if 0    // CAPTURE - TWO-COLUMN CHARACTERS
dp->CaptureDialog ( "capturedlg.html", true, false, 
                    "infodoc-styles.css", 4, false, nc.blR, true ) ;
#else    // CAPTURE - SINGLE-COLUMN CHARACTERS
dp->CaptureDialog ( "capturedlg.html", true, false, 
                    "infodoc-styles.css", 4, false, nc.blR ) ;
#endif   // CAPTURE
#endif   // CAPTURE
               }
               done = true ;
            }
         }

         if ( ic[icIndex].type == dctTEXTBOX )
         {
            Info.viaHotkey = false ;
            icIndex = dp->EditTextbox ( Info ) ;
         }

         if ( done == false && Info.viaHotkey == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }  // while()
   }

   if ( dp != NULL )                // delete (close) the dialog
      delete dp ;

   //* Restore the underlying dialog *
   this->userDlg->RefreshWin () ;

   return conf ;

}  //* End lfaConfirm() *

//*************************
//*  LogfileFormatRecord  *
//*************************
//******************************************************************************
//* Capture and format user data as text. Caller may write these data to the   *
//* statsDlg window (text format only), or may write the data (either text or  *
//* HTML format) to the log file.                                              *
//*                                                                            *
//* IMPORTANT NOTE: This method allocates dynamic memory and attaches it to    *
//*                 the 'udAlloc' member. It is the caller's responsibility    *
//*                 to release the allocation.                                 *
//*                 Note also that any existing allocation attached to         *
//*                 'udAlloc' will be silently released before the capture     *
//*                 proceeds.                                                  *
//*                                                                            *
//* Input  : uData : (by reference) record to be formatted                     *
//*          uComm : (optional, NULL pointer by default)                       *
//*                  user comment to be appended to record                     *
//*          html  : (optional, 'false' by default)                            *
//*                  if 'false', capture data and format as plain text         *
//*                  if 'true',  capture data and format as HTML source code   *
//*                                                                            *
//* Returns: number of bytes written to target buffer (incl. nullchar)         *
//*          Note: If capture fails, only the current timestamp and a          *
//*                warning message will be written to the dynamic              *
//*                memory-allocation area.                                     *
//******************************************************************************
//* Notes: The data are captured by the NcDialog-class CaptureDialog() method. *
//* The interesting information is extracted from the capture for writing to   *
//* the logfile.                                                               *
//*                                                                            *
//* Parsing the plain-text capture:                                            *
//* -------------------------------                                            *
//* 1) Open the file and discard the first line of the capture.                *
//* 2) Write the timestamp as the first line of output.                        *
//* 3) Copy the column headings line.                                          *
//* 4) Copy each user-data line, removing the borders and discarding trailing  *
//*    whitespace.                                                             *
//* 5) Discard the remainder of the capture file.                              *
//* 6) Close and delete the capture file.                                      *
//*                                                                            *
//*                                                                            *
//* Parsing the HTML capture:                                                  *
//* -------------------------                                                  *
//* Open the file and discard lines to but not including                       *
//* the line that begins with: "<div class="tscap" ..."                        *
//* 1) Split this line at the "<span ..."                                      *
//* 2) Append the timestamp after the "<div>" tag.                             *
//*    a) If specified, append the user's comment. Note that comment is        *
//*       truncated to the width of the dialog window.                         *
//* 3) Write the "<span ..." portion of the input line as a line of its own.   *
//* 4) Copy each user-data line. Note that column alignment may be slightly    *
//*    off for CJK or other multi-column character text, but the capture       *
//*    method makes adjustments if the 'microSpacing' flag (see below) is set. *
//* 5) Scan and discard the remainder of the capture up to the bottom border   *
//*    of the dialog. Save the bottom border line.                             *
//* 6) Copy the closing "</div>" tag line.                                     *
//* 7) Close and delete the capture file.                                      *
//*                                                                            *
//******************************************************************************

short Exercalc::LogfileFormatRecord ( const UserData& uData, 
                                      const gString* uComm, bool html )
{
   //* Size (in bytes) of the dynamic allocation. *
   // Programmer's Note: This is a generous size, and it is unlikely that the 
   // data written will approach filling it. Experimentally, the plain text 
   // data are approximately 700 bytes, while the HTML data are approximately
   // 3,300 bytes for English and 5,500 bytes for Chinese.
   const short DYN_ALLOC = 8192 ;

   if ( this->udAlloc != NULL )           // release old allocation
      delete [] this->udAlloc ;
   this->udAlloc = new char[DYN_ALLOC] ;  // allocate output buffer

   char  lineData[gsMAXBYTES] ;           // raw UTF-8 input
   gString gsOut, gs,                     // text formatting
           gsEncode,                      // encoded statistics
           gsCap ;                        // filespec of capture file
   localTime lt ;                         // system local timestamp
   short /*padWidth = udInit.dColumns,      // width of dialog window (columns)*/
         bindx = ZERO,                    // text insertion index for 'udAlloc'
         cindx ;                          // character index

   //* Create a temp-file name. In the unlikely event that the  *
   //* call fails, create the file in the application directory.*
   if ( !(this->CreateTempname ( gsCap )) )
      gsCap.compose( "%s/capturedUD.txt", this->cfg.appPath ) ;

   //* Get the system-local time *
   this->GetLocalTime ( lt ) ;

   //* Create the encoded statistics which are used when *
   //* calculating file-summary data.                    *
   this->lfEncodeStats ( uData, gsEncode, html ) ;

   //***********************
   //*** For HTML format ***
   //***********************
   if ( html )
   {
      const short PADW1 = udInit.dColumns - 1 ;
      bool microSpacing = bool(this->cfg.lang == zhLang) ;
      //* Capture the dialog in "simple" HTML format *
      this->userDlg->CaptureDialog ( gsCap.ustr(), true, false, 
                                     "infodoc-styles.css", 4, false, 
                                     this->cfg.bb, microSpacing ) ;

      //* Open the capture and extract the user-data lines. See notes above.*
      ifstream ifs ( gsCap.ustr(), ifstream::in ) ;
      if ( ifs.is_open() )             // if input file open
      {
         //* Read and discard to and including the *
         //* <div class="curses_container"> tag.   *
         do
         {
            ifs.getline ( lineData, gsMAXBYTES, NEWLINE ) ;
            if ( (ifs.good()) || (ifs.gcount()) )
            {
               gsOut = lineData ;
            }
            else        // unexpected EOF
               break ;
         }
         while ( (gsOut.compare( htmlContainerD, true, HCD_LEN )) != ZERO ) ;

         //* Read the first line: "<div class="tscap"...>" tag.        *
         //* Separate the tag from the following "<span ...>" sequence.*
         //* Append the timestamp and optional comment to the tag      *
         //* (in contrasting color).                                   *
         //* Write the "<span ...>" sequence on the following line.    *
         ifs.getline ( lineData, gsMAXBYTES, NEWLINE ) ;
         if ( (ifs.good()) || (ifs.gcount()) )
         {
            gsOut = lineData ;
            cindx = gsOut.after( L'>' ) ;
            gs.compose( "%S\n", &gsOut.gstr()[cindx] ) ;
            gString gsSpan( "<span class=\"%s\"> ",   // select color-attribute class
                            (((gs.find( "bw_r" )) >= ZERO) ? "bl_r" : "bw_r") ) ;
            gsOut.limitChars( cindx ) ;
            gsOut.copy( &this->udAlloc[bindx], gsMAXBYTES ) ;
            bindx += (gsOut.utfbytes() - 1) ;   // index the null terminator

            //* Write the timestamp *
            gsOut = lt.iso ;
            gsOut.padCols( PADW1 ) ;
            gsOut.insert( gsSpan.gstr() ) ;
            gsOut.append( "</span>\n" ) ;
            gsOut.copy( &this->udAlloc[bindx], gsMAXBYTES ) ;
            bindx += (gsOut.utfbytes() - 1) ;

            //* Write the user-provided a comment, if provided *
            if ( (uComm != NULL) && (uComm->gschars() > 1) )
            {
               gsOut = uComm->gstr() ;             // get user comment
               if ( (gsOut.gscols()) > PADW1 )     // limit to width of dialog
                  gsOut.limitCols( PADW1 ) ;
               gsOut.padCols( PADW1 ) ;            // pad to width of dialog
               gsOut.insert( gsSpan.gstr() ) ;     // define within a span
               gsOut.append( "</span>\n" ) ;
               gsOut.copy( &this->udAlloc[bindx], gsMAXBYTES ) ;
               bindx += (gsOut.utfbytes() - 1) ;
            }
            gs.copy( &this->udAlloc[bindx], gsMAXBYTES ) ;
            bindx += (gs.utfbytes() - 1) ;
         }

         //* Read and format lines until we reach the dialog divider line.*
         while ( true )
         {
            ifs.getline ( lineData, gsMAXBYTES, NEWLINE ) ;
            if ( (ifs.good()) || (ifs.gcount()) )
            {
               gsOut = lineData ;
               if ( (gsOut.find( wcsLTEEdh )) < ZERO )
               {
                  gsOut.append( L'\n' ) ;
                  gsOut.copy( &this->udAlloc[bindx], gsMAXBYTES ) ;
                  bindx += (gsOut.utfbytes() - 1) ;   // index the null terminator
               }
               else     // all user data written
                  break ;
            }
            else        // unexpected EOF
               break ;
         }

         //* Discard lines from divider line to bottom border of dialog, *
         //* then save the bottom border line and append the "</div> tag.*
         short cornerCount = 2 ; // number of lines with lower-left corner characters
         do
         {
            ifs.getline ( lineData, gsMAXBYTES, NEWLINE ) ;
            if ( (ifs.good()) || (ifs.gcount()) )
            {
               gsOut = lineData ;
               if ( (gsOut.find( wcsLLs )) >= ZERO )
                  --cornerCount ;
            }
            else        // unexpected EOF
               break ;
         }
         while ( cornerCount > ZERO ) ;
         gsOut.append( L'\n' ) ;
         gsOut.copy( &this->udAlloc[bindx], gsMAXBYTES ) ;
         bindx += (gsOut.utfbytes() - 1) ;   // index the null terminator

         while ( true )
         {
            ifs.getline ( lineData, gsMAXBYTES, NEWLINE ) ;
            if ( (ifs.good()) || (ifs.gcount()) )
            {
               gsOut.compose( "%s\n", lineData ) ;
               if ( (gsOut.find( L"</div>" )) == ZERO )
               {
                  gsOut.copy( &this->udAlloc[bindx], gsMAXBYTES ) ;
                  bindx += (gsOut.utfbytes() - 1) ;   // index the null terminator
                  break ;
               }
            }
            else        // unexpected EOF
               break ;
         }

         //* Insert the encoded values. *
         gsEncode.copy( &this->udAlloc[bindx], gsMAXBYTES ) ;
         bindx += (gsEncode.utfbytes()) ; // (includes the null terminator)

         ifs.close() ;  // close the source file
      }
   }

   //*****************************
   //*** For plain text format ***
   //*****************************
   else
   {
      //* Capture the dialog in plain-text format *
      this->userDlg->CaptureDialog ( gsCap.ustr(), false, false ) ;

      //* Open the capture and extract the user-data lines. See notes above.*
      ifstream ifs ( gsCap.ustr(), ifstream::in ) ;
      if ( ifs.is_open() )             // if input file open
      {
         //* Write the system-local timestamp and (optional) comment *
         if ( (uComm != NULL) && (uComm->gschars() > 1) )
            gsOut.compose( "%s - %S\n", lt.iso, uComm->gstr() ) ;
         else
            gsOut.compose( "%s\n", lt.iso ) ;
         gsOut.copy( &this->udAlloc[bindx], gsMAXBYTES ) ;
         bindx += (gsOut.utfbytes() - 1) ;   // index the null terminator

         //* Read and discard the first line of the file.*
         //* (This is the top border.)                   *
         ifs.getline ( lineData, gsMAXBYTES, NEWLINE ) ;

         //* Read and format lines until we reach the dialog divider line.*
         while ( true )
         {
            ifs.getline ( lineData, gsMAXBYTES, NEWLINE ) ;
            if ( (ifs.good()) || (ifs.gcount()) )
            {
               //* Strip the left and right border characters *
               gsOut = lineData ;
               if ( (gsOut.find( wcsLTEEdh )) != ZERO )
               {
                  cindx = ZERO ;
                  cindx = gsOut.erase( wcsVERTs, cindx ) ;
                  gsOut.erase( wcsVERTs, cindx ) ;
                  gsOut.strip ( false, true ) ;       // remove trailing whitespace
                  gsOut.insert( "   " ) ;             // indent the line
                  gsOut.append( L'\n' ) ;             // append a newline
                  gsOut.copy( &this->udAlloc[bindx], gsMAXBYTES ) ;
                  bindx += (gsOut.utfbytes() - 1) ;   // index the null terminator
               }
               else     // all user data written
                  break ;
            }
            else        // unexpected EOF
               break ;
         }

         //* Insert the encoded values. *
         gsEncode.copy( &this->udAlloc[bindx], gsMAXBYTES ) ;
         bindx += (gsEncode.utfbytes()) ; // (includes the null terminator)

         ifs.close() ;  // close the source file
      }
   }

   this->DeleteTempname ( gsCap ) ;    // delete the temporary file

   return bindx ;

}  //* End LogfileFormatRecord()

//*************************
//*     lfEncodeStats     *
//*************************
//******************************************************************************
//* Convert the numeric source data to a delimited string of ASCII-numeric     *
//* values. Values are delimited by colon (':') characters.                    *
//* The current system local time (ASCII-hex) is included in the output.       *
//*                                                                            *
//* Input  : uData  : (by reference) numerical user data to be encoded         *
//*          gsStat : (by reference, initial contents ignored)                 *
//*                   receives the encoded data in the form of a UTF-8 string  *
//*          html   : indicates whether formatting s/b HTML or plain text      *
//*                   if 'false', return the encoded data with a header        *
//*                               followed by a series of encoded values       *
//*                               "EncodedStats: DATA"                         *
//*                   if 'true',  return the encoded data string formatted as  *
//*                               an HTML comment:                             *
//*                               "<!-- EncodedStats: DATA -->"                *
//*                                                                            *
//* Returns: 'true' if data successfully encoded                               *
//*          'false' corrupted source data (unlikely)                          *
//******************************************************************************

bool Exercalc::lfEncodeStats ( const UserData& uData, gString& gsStat, bool html )
{
   int64_t epoch_time ;          // system local time
   short convCount = ZERO ;      // number of conversions performed
   short cindx = ZERO ;          // character index
   bool  status = false ;        // return value

   //* Get the local time *
   this->GetLocalTime ( epoch_time ) ;

   gsStat.compose( lfstatsTemplateW,
                   &uData.heightM,
                   &uData.massKg,
                   &uData.bmi,
                   &uData.kphWalk,
                   &uData.kphBike,
                   &uData.kphRun,
                   &uData.metsWalk,
                   &uData.metsBike,
                   &uData.metsRun,
                   &uData.metsGen,
                   &uData.inpValue,
                   &uData.kmeters,
                   &uData.miles,
                   &uData.mins,
                   &uData.metTable,
                   &uData.metRatio,
                   &uData.kph,
                   &uData.metmins,
                   &uData.kcal,
                   &uData.exerType,
                   &uData.inpUnits,
                   &epoch_time ) ;

   //* Test actual conversions against expected conversions *
   // Programmer's Note: The header includes a ':' and there is no trailing
   // ':', so the count should be stable. The only way there would be an
   // undercount is if the conversion was aborted due to syntax errors.
   while ( (cindx = gsStat.after( wCOLON, cindx )) > ZERO )
      ++convCount ;
   if ( convCount == convCOUNT )
      status = true ;

   //* If HTML format specified, enclose the data in comment tokens *
   if ( html )
   {
      gsStat.append( htmlComment_End ) ;
      gsStat.insert( htmlComment_Begin ) ;
   }
   //* For text format append an extra newline *
   else
      gsStat.append( L'\n' ) ;
   gsStat.append( L'\n' ) ;      // append a newline

   return status ;

}  //* End lfEncodeStats() *

//*************************
//*     lfDecodeStats     *
//*************************
//******************************************************************************
//* Convert a colon-delimited string of ASCII-numeric values to numeric        *
//* data values.                                                               *
//*                                                                            *
//* Input  : gsStat : (by reference) contains the source data in string format *
//*                   data may be either a string of ASCII-numeric values:     *
//*                             "###: EncodedStats: DATA"                      *
//*                   OR the data may be enclosed in an HTML comment:          *
//*                             "<!--###: EncodedStats: DATA -->"              *
//*          uData  : (by reference, initial contents ignored)                 *
//*                   receives the decoded numerical data                      *
//*          ltPtr  : (optional, NULL pointer by default)                      *
//*                   if specified, the target object receives the decoded     *
//*                   system-local timestamp                                   *
//*                                                                            *
//* Returns: 'true' if data successfully decoded                               *
//*          'false' (corrupted source data)                                   *
//******************************************************************************

bool Exercalc::lfDecodeStats ( const gString& gsStat, UserData& uData, localTime* ltPtr )
{
   short convCount = ZERO ;      // number of conversions performed
   bool  status = false ;        // return value

   //* Initialize caller's data objects *
   uData.reset() ;
   if ( ltPtr != NULL )
      ltPtr->reset() ;

   convCount = gsStat.gscanf( lfstatsTemplateR,
                              &uData.heightM,
                              &uData.massKg,
                              &uData.bmi,
                              &uData.kphWalk,
                              &uData.kphBike,
                              &uData.kphRun,
                              &uData.metsWalk,
                              &uData.metsBike,
                              &uData.metsRun,
                              &uData.metsGen,
                              &uData.inpValue,
                              &uData.kmeters,
                              &uData.miles,
                              &uData.mins,
                              &uData.metTable,
                              &uData.metRatio,
                              &uData.kph,
                              &uData.metmins,
                              &uData.kcal,
                              &uData.exerType,
                              &uData.inpUnits,
                              &uData.timeStamp ) ;
   if ( convCount == convCOUNT )
   {
      if ( (uData.timeStamp > ZERO) && (ltPtr != NULL) )
         this->DecodeEpochTime ( uData.timeStamp, *ltPtr ) ;
      status = true ;
   }

   return status ;

}  //* End lfDecodeStats() *

//*************************
//*    LogfileSummary     *
//*************************
//******************************************************************************
//* Create and optionally display a summary of log data.                       *
//* 1) Read the encoded data for each record.                                  *
//* 2) Calculate totals for specific fields: massKg, bmi, kmeters, miles,      *
//*    mins, kph, metmins, kcal.                                               *
//* 3) Calculate averages for specific values: kmeters, miles, mins, metmins,  *
//*    kcal, mass, kph, bmi.                                                   *
//* 4) Capture bracketing timestamps to define the time period covered by the  *
//*    logfile records.                                                        *
//* 5) Format and display the captured data.                                   *
//* 6) Graph the exercise data over the time period. The vertical axis of the  *
//*    graph is determined by the user's "goal" (elapsed time, distance        *
//*    travelled or energy consumed) as specified in the configuration file.   *
//*                                                                            *
//* Input  : display : if 'true'  display the summary results                  *
//*                    if 'false' do not display the summary                   *
//*                                                                            *
//* Returns: number of valid records captured                                  *
//*          ZERO if log file not found, invalid format or no data available   *
//******************************************************************************

short Exercalc::LogfileSummary ( bool display )
{
   //* Formatting template for filename, record count and timestamps *
   const char *timeTemplate = "%S\n"
                              "%hd\n"
                              "%04hd-%02hd-%02hdT%02hd:%02hd\n"
                              "%04hd-%02hd-%02hdT%02hd:%02hd" ;
   const short timeOffset[LangCount] = 
   {
       9,   //** English **
      10,   //** Español **
      10,   //** Zhōngwén (中文) **
       9,   //** TiếngViệt **
   } ;
   //* Formatting template for summary totals *
   const char *totalsTemplate = "%-.3f\n"
                                "%-.3f\n"
                                "%02hd:%02hd\n"
                                "%-.0f\n"
                                "%-.0f\n" ;
   const short totalsOffset[LangCount] =
   {
      16,   //** English **
      16,   //** Español **
      15,   //** Zhōngwén (中文) **
      20,   //** TiếngViệt **
   } ;
   const char *avgTemplate = "%-.3f\n"          // 'averages' base data
                             "%-.3f\n"
                             "%-.3f\n"
                             "%-.3f\n"
                             "%-.3f\n" ;
   const short GRAPH_Y = 14,           // upper-left corner of graph area
               GRAPH_X = 1 ;

   char  lineData[gsMAXBYTES] ;        // raw UTF-8 input
   gString gsOut, gs,                  // text formatting
           gsTrg( this->cfg.logPath ); // target filespec
   winPos wpl( 1, 2 ),                 // cursor position, left column
          wpr( 2, (sdInit.dColumns / 2) ), // cursor position, right column
          wptmp ;
   localTime lta, ltb ;                // timestamp formatting
   tnFName fStats ;                    // target file stats
   logMsgs errMsg ;                    // error message code
   short   recCount = ZERO,            // number of records captured
           totIndex = ZERO,            // index of the 'totals' record
           avgIndex = ZERO ;           // index of the 'averages' record

   //* If RTL, adjust column positions *
   if ( this->cfg.rtl )
   {
      wpl.xpos = sdInit.dColumns - 3 ;
   }

   //* Discard any previous dynamic allocation *
   if ( this->udCap != NULL )
      delete[] this->udCap ;
   this->udcCount = ZERO ;

   //* Test whether target exists *
   if ( (gsTrg.gschars() > 1) && (this->GetFileStats ( gsTrg, fStats )) == OK )
   {
      bool formatVerified = true ;     // file validation

      //* Open the file and verify the file format.    *
      //* If valid format, count the number of records.*
      ifstream ifs ( this->cfg.logPath, ifstream::in ) ;
      if ( ifs.is_open() )             // if input file open
      {
         ifs.getline ( lineData, gsMAXBYTES, NEWLINE ) ;
         if ( (ifs.good()) || (ifs.gcount()) > ZERO )
         {
            gs = lineData ;

            //* Test for HTML-formatted document *
            if ( (gs.find( "<!DOCTYPE HTML>" )) == ZERO )
            {
               //* Read and discard the "<html>" and "<head>" lines *
               ifs.getline ( lineData, gsMAXBYTES, NEWLINE ) ;
               ifs.getline ( lineData, gsMAXBYTES, NEWLINE ) ;
               //* The next line S/B a comment containing the LogID *
               ifs.getline ( lineData, gsMAXBYTES, NEWLINE ) ;
               gs = lineData ;
               gs.shiftChars( -5 ) ;   // discard: "<!-- "
            }
            if ( (gs.compare( LogID[this->cfg.lang], true, LogID_CompChars )) != ZERO )
            {
               formatVerified = false ;
               errMsg = lmBadFormat ;
            }
         }

         if ( formatVerified )
         {
            //* Count the records *
            while ( true )
            {
               ifs.getline ( lineData, gsMAXBYTES, NEWLINE ) ;
               if ( (ifs.good()) || (ifs.gcount()) > ZERO )
               {
                  //* Scan for marker indicating an encoded record *
                  gs = lineData ;
                  if ( (gs.find( recMark )) >= ZERO )
                     ++recCount ;
               }
               else
                  break ;
            }
         }

         //* If one or more valid records *
         if ( recCount > ZERO )
         {
            //* Allocate dynamic storage and index the summary records *
            this->udcCount = recCount ;
            totIndex = recCount ;
            avgIndex = totIndex + 1 ;
            this->udCap = new UserData[this->udcCount + 2] ;

            //* Rewind to top of file *
            ifs.clear() ;                 // reset the EOF flag
            ifs.seekg( ZERO ) ;           // return to top of input file

            for ( short indx = ZERO ; indx < recCount ; )
            {
               ifs.getline ( lineData, gsMAXBYTES, NEWLINE ) ;
               if ( (ifs.good()) || (ifs.gcount()) > ZERO )
               {
                  //* Scan for marker indicating an encoded record *
                  gs = lineData ;
                  if ( (gs.find( recMark )) >= ZERO )
                  {
                     this->lfDecodeStats ( gs, this->udCap[indx] ) ;
                     //* Add relevant members to the accumulator *
                     this->udCap[totIndex].massKg  += this->udCap[indx].massKg ;
                     this->udCap[totIndex].bmi     += this->udCap[indx].bmi ;
                     this->udCap[totIndex].kmeters += this->udCap[indx].kmeters ;
                     this->udCap[totIndex].miles   += this->udCap[indx].miles ;
                     this->udCap[totIndex].mins    += this->udCap[indx].mins ;
                     this->udCap[totIndex].kph     += this->udCap[indx].kph ;
                     this->udCap[totIndex].metmins += this->udCap[indx].metmins ;
                     this->udCap[totIndex].kcal    += this->udCap[indx].kcal ;
                     //* 'totals' record gets earliest timestamp and  *
                     //* 'averages' record gets most recent timestamp.*
                     if ( (this->udCap[totIndex].timeStamp == ZERO) ||
                          (this->udCap[indx].timeStamp < this->udCap[totIndex].timeStamp) )
                        this->udCap[totIndex].timeStamp = this->udCap[indx].timeStamp ;
                     if ( (this->udCap[indx].timeStamp > this->udCap[avgIndex].timeStamp) )
                        this->udCap[avgIndex].timeStamp = this->udCap[indx].timeStamp ;
                     ++indx ;
                  }
               }
               else        // file error (unlikely)
                  break ;
            }
         }

         ifs.close() ;        // close the file
      }
      else
         errMsg = lmNoAccess ;
   }
   else
      errMsg = lmNoTarget ;

   //* If records have been captured. Format and display the results *
   if ( (display != false) && (recCount > ZERO) )
   {
      //* Clear the stats dialog window *
      this->statsDlg->ClearWin () ;

      //* Set the dialog title *
      this->statsDlg->SetDialogTitle ( summRecHdr[this->cfg.lang][srhTitle], 
                                       this->cfg.bb | ncuATTR ) ;

      //* Decode the timestamps *
      this->DecodeEpochTime ( this->udCap[totIndex].timeStamp, lta ) ;
      this->DecodeEpochTime ( this->udCap[avgIndex].timeStamp, ltb ) ;

      //* Draw the labels *
      wptmp = { wpl.ypos, short(wpl.xpos + timeOffset[this->cfg.lang]) } ;
      #if DEBUG_RTL != 0
      if ( this->cfg.rtl )
         wptmp.xpos -= (timeOffset[this->cfg.lang] * 2) ;
      #endif   // DEBUG_RTL
      gsOut = summRecHdr[this->cfg.lang][srhTimes] ;
      wpl = this->statsDlg->WriteParagraph ( wpl, gsOut, this->cfg.sd, 
                                             false, this->cfg.rtl ) ;

      //* Format the timestamp data, etc. *
      gs = &gsTrg.gstr()[(gsTrg.findlast( L'/' )) + 1] ;
      gsOut.compose( timeTemplate, gs.gstr(), &this->udcCount, 
                     &lta.year, &lta.month, &lta.date, &lta.hours, &lta.minutes,
                     &ltb.year, &ltb.month, &ltb.date, &ltb.hours, &ltb.minutes ) ;
      if ( this->cfg.rtl )    // reverse text for all values
         gsOut.textReverse( false, true ) ;
      this->statsDlg->WriteParagraph ( wptmp, gsOut, this->cfg.sd, 
                                       false, this->cfg.rtl ) ;

      //* Write the 'totals' column heading and labels *
      this->statsDlg->WriteParagraph ( wpl.ypos++, wpl.xpos, 
                                       summRecHdr[this->cfg.lang][srhTotHdr], 
                                       (this->cfg.bb | ncuATTR), false, this->cfg.rtl ) ;
      this->statsDlg->WriteParagraph ( wpl, summRecHdr[this->cfg.lang][srhTotData],
                                       this->cfg.bb, false, this->cfg.rtl ) ;

      //* Format the totals data *
      wptmp = { wpl.ypos, short(wpl.xpos + totalsOffset[this->cfg.lang]) } ;
      short iHr, iMn ;
      this->Minutes2Time ( this->udCap[totIndex].mins, iHr, iMn ) ;
      gsOut.compose( totalsTemplate,
                     &this->udCap[totIndex].kmeters,
                     &this->udCap[totIndex].miles,
                     &iHr, &iMn,
                     &this->udCap[totIndex].metmins,
                     &this->udCap[totIndex].kcal ) ;

      if ( this->cfg.rtl )    // reverse text for all values
      {
         gsOut.textReverse( false, true ) ;
         #if DEBUG_RTL != 0
         wptmp.xpos -= (totalsOffset[this->cfg.lang] * 2) ;
         #endif   // DEBUG_RTL
      }
      this->statsDlg->WriteParagraph ( wptmp, gsOut, this->cfg.sd, 
                                       false, this->cfg.rtl ) ;

      //* Write the 'averages' column heading and labels *
      this->statsDlg->WriteString ( wpr.ypos++, wpr.xpos, 
                                    summRecHdr[this->cfg.lang][srhAvgHdr],
                                    (this->cfg.bb | ncuATTR), false, this->cfg.rtl) ;
      wptmp = { wpr.ypos, short(wpr.xpos + totalsOffset[this->cfg.lang]) } ;
      wpr = this->statsDlg->WriteParagraph ( wpr, summRecHdr[this->cfg.lang][srhAvgData],
                                             this->cfg.bb, false, this->cfg.rtl) ;

      //* Calculate the 'averages' data *
      float pounds = 0.0, mph = 0.0 ;
      this->udCap[avgIndex].kmeters = this->udCap[totIndex].kmeters / this->udcCount ;
      this->udCap[avgIndex].miles   = this->udCap[totIndex].miles / this->udcCount ;
      this->udCap[avgIndex].mins    = this->udCap[totIndex].mins / this->udcCount ;
      this->Minutes2Time ( this->udCap[avgIndex].mins, iHr, iMn ) ;
      this->udCap[avgIndex].metmins = this->udCap[totIndex].metmins / this->udcCount ;
      this->udCap[avgIndex].kcal    = this->udCap[totIndex].kcal / this->udcCount ;
      this->udCap[avgIndex].massKg  = this->udCap[totIndex].massKg / this->udcCount ;
      pounds = this->Kilograms2Pounds ( this->udCap[avgIndex].massKg ) ;
      this->udCap[avgIndex].kph     = this->udCap[totIndex].kph / this->udcCount ;
      mph    = this->Kilometers2Miles ( this->udCap[avgIndex].kph ) ;
      this->udCap[avgIndex].bmi     = this->udCap[totIndex].bmi / this->udcCount ;

      //* Format the 'averages' data *
      gsOut.compose( avgTemplate,
                     &this->udCap[avgIndex].massKg,
                     &pounds,
                     &this->udCap[avgIndex].kph,
                     &mph,
                     &this->udCap[avgIndex].bmi ) ;
      if ( this->cfg.rtl )    // reverse text for all values
      {
         gsOut.textReverse( false, true ) ;
         #if DEBUG_RTL != 0
         wptmp.xpos -= (totalsOffset[this->cfg.lang] * 2) ;
         #endif   // DEBUG_RTL
      }
      wptmp = this->statsDlg->WriteParagraph ( wptmp, gsOut, this->cfg.sd, 
                                               false, this->cfg.rtl ) ;

      this->statsDlg->WriteParagraph ( wpr, summRecHdr[this->cfg.lang][srhTotData],
                                       this->cfg.bb, false, this->cfg.rtl ) ;
      gsOut.compose( totalsTemplate,
                     &this->udCap[avgIndex].kmeters,
                     &this->udCap[avgIndex].miles,
                     &iHr, &iMn,
                     &this->udCap[avgIndex].metmins,
                     &this->udCap[avgIndex].kcal ) ;
      this->statsDlg->WriteParagraph ( wptmp, gsOut, this->cfg.bb, 
                                       false, this->cfg.rtl ) ;

      //* Create the graph of "goal" vs. time. *
      wpl = { GRAPH_Y, GRAPH_X } ;
      this->lfsGraph ( wpl ) ;

      this->statsDlg->RefreshWin () ;  // make everything visible
   }
   else
      errMsg = lmNoRecords ;

   if ( recCount == ZERO )
   {
      chrono::duration<short, std::milli>aMoment( 1500 ) ;  // pause
      this->UserMessage ( LogMsgs[this->cfg.lang][errMsg], this->cfg.em, true ) ;
      this->userDlg->UserAlert ( 2 ) ;
      this_thread::sleep_for( aMoment ) ;
      this->UserMessage ( "", this->cfg.bb ) ;
   }
   return recCount ;

}  //* End LogfileSummary() *

//*************************
//*      LoadRecord       *
//*************************
//******************************************************************************
//* Scan the log file and ask user to select a record.                         *
//* Logfile records will be attached to the 'udCap' member.                    *
//* User-selected record (if any) will be copied to 'ud' member.               *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note: For RTL language rendering in the Scrollext control:    *
//* The contents of the Scrollext control are primarily numeric data.          *
//* Therefore the contents ARE NOT drawn as RTL data even though the control's *
//* label (column headings) IS drawn as RTL.                                   *
//* When adding an RTL language, this must be handled as a special case by     *
//* aligning the column headings to match the displayed data.                  *
//******************************************************************************

void Exercalc::LoadRecord ( void )
{
   //* Define a dialog window the same size as the stats dialog *
   InitNcDialog dInit = sdInit ;
   const short seCOLS = dInit.dColumns - 2 ; // display columns for Scrollext control

   //* Add the control objects *
   enum lrControls : short { lrLoadPB = ZERO, lrCancelPB, lrListSE, lrItems } ;

   InitCtrl ic[lrItems] = 
   {
   {  //* 'SELECT' pushbutton   - - - - - - - - - - - - - - - - - -   lrLoadPB *
      dctPUSHBUTTON,                // type:      control type
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dInit.dLines - 3),      // ulY:       upper left corner in Y
      short(dInit.dColumns / 2 - 25),// ulX:       upper left corner in X
      1,                            // lines:     control lines
      23,                           // cols:      control columns
      LoadDlgLabels[this->cfg.lang][ldlLoad],   // dispText:  display text
      this->cfg.pn,                 // nColor:    non-focus color
      this->cfg.pf,                 // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[lrCancelPB],              // nextCtrl:  link in next structure
   },

   {  //* 'CANCEL' pushbutton   - - - - - - - - - - - - - - - - - - lrCancelPB *
      dctPUSHBUTTON,                // type:      control type
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[lrLoadPB].ulY,             // ulY:       upper left corner in Y
      short(ic[lrLoadPB].ulX + ic[lrLoadPB].cols + 2),// ulX: upper left corner in X
      1,                            // lines:     control lines
      23,                           // cols:      control columns
      LoadDlgLabels[this->cfg.lang][ldlCancel], // dispText:  display text
      this->cfg.pn,                 // nColor:    non-focus color
      this->cfg.pf,                 // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[lrListSE],                // nextCtrl:  link in next structure
   },

   { //* 'LIST' Scrollext  - - - - - - - - - - - - - - - - - - - - -  lrListSE *
      dctSCROLLEXT,                 // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      2,                            // ulY:       upper left corner in Y
      ZERO,                         // ulX:       upper left corner in X
      short(dInit.dLines - 8),      // lines:     control lines
      dInit.dColumns,               // cols:      control columns
      NULL,                         // dispText:  (n/a - set below)
      dInit.borderColor,            // nColor:    non-focus border color
      this->cfg.pf,                 // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      LoadDlgLabels[this->cfg.lang][ldlList],   // dispText:  display text
      -1,                           // labY:      offset from control's ulY
      1,                            // labX       offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      ZERO,                         // scrItems:  (n/a - set below)
      ZERO,                         // scrSel:    (n/a - set below)
      NULL,                         // scrColor:  (n/a - set below)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
   } ;
   dInit.ctrlPtr = ic ;             // attach control objects to initializer

   gString gs( this->cfg.logPath ), // target filespec
           gsOut,                   // text formatting
           gsrev ;                  // for reversing RTL text
   localTime lt ;                   // decode epoch time
   fmFType fType ;                  // file type of target file
   short iHr, iMn,                  // formatted time values
         rCnt = ZERO ;              // number of records captured

   //* Verify that the log file exists and capture valid data *
   bool trgExists = this->TargetExists ( gs, fType ) ;
   if ( trgExists && (fType == fmREG_TYPE) )
      rCnt = this->LogfileSummary ( false ) ;

   if ( rCnt > ZERO )
   {
      //* Save the stats dialog display *
      this->statsDlg->SetDialogObscured () ;

      //* Format the records for display *
      const char* lrTemplate = 
            "%04hd-%02hd-%02hd   %02hd:%02hd     %-14S %05.2fkm   %6.1f %7.2f " ;
      const short MAXRECBYTES = 256 ;
      char *seRecords = new char[rCnt * MAXRECBYTES] ;
      const char** serecPtr = new const char*[rCnt] ;
      attr_t *serecColors = new attr_t[rCnt] ;
      short serIndx = ZERO ;
      for ( short i = ZERO ; i < rCnt ; ++i )
      {
         this->DecodeEpochTime ( this->udCap[i].timeStamp, lt ) ;
         this->Minutes2Time ( this->udCap[i].mins, iHr, iMn ) ;
         gsrev = UserExerType[this->cfg.lang][this->udCap[i].exerType] ;
         if ( this->cfg.rtl )
            gsrev.textReverse() ;
         gsOut.compose( lrTemplate, &lt.year, &lt.month, &lt.date, &iHr, &iMn, 
                        gsrev.gstr(),
                        &this->udCap[i].kmeters, &this->udCap[i].metmins, 
                        &this->udCap[i].kcal ) ;
         while ( gsOut.gscols() < seCOLS )   // fit text width to display area
            gsOut.append( L' ' ) ;
         gsOut.limitCols( seCOLS ) ;
         serecPtr[i] = &seRecords[serIndx] ;
         serIndx += gsOut.utfbytes() ;
         gsOut.copy( (char*)serecPtr[i], MAXRECBYTES ) ;
         serecColors[i] = this->cfg.sb ;     // set color attribute
      }

      //* Adjust offset for column headings (see note in method header) *
      if ( this->cfg.rtl )
         ic[lrListSE].labX = ic[lrListSE].cols - 2 ;

      //* Instantiate the dialog window *
      NcDialog *dp = new NcDialog ( dInit ) ;

      //* If user interface language is an RTL language, *
      //* set the internal NcDialog flag for RTL output. *
      if ( this->cfg.rtl )
         dp->DrawLabelsAsRTL () ;

      //* Open the dialog window *
      if ( (dp->OpenWindow()) == OK )
      {
         //* Set dialog title *
         dp->SetDialogTitle ( LoadDlgLabels[this->cfg.lang][ldlTitle], 
                              this->cfg.ub | ncuATTR ) ;

         //* Write the static text *
         winPos wp( short(ic[lrListSE].ulY + ic[lrListSE].lines),
                    short(ic[lrListSE].ulX + 
                          this->cfg.rtl ? (ic[lrListSE].cols - 3) : 2 ) ) ;
         wp = dp->WriteParagraph ( wp, LoadDlgLabels[this->cfg.lang][ldlLabel], 
                                   dInit.interiorColor, false, this->cfg.rtl ) ;
         gsOut.compose( "%hd\n", &rCnt ) ;
         if ( this->cfg.rtl ) // numeric value: adjust X offset and write as LTR
            wp.xpos -= gsOut.gscols() - 1 ;
         wp = dp->WriteParagraph ( wp, gsOut, this->cfg.sb ) ;
         wp.xpos = ic[lrListSE].ulX + this->cfg.rtl ? (ic[lrListSE].cols - 3) : 2 ;
         wp = dp->WriteParagraph ( wp, LoadDlgLabels[this->cfg.lang][ldlInstr], 
                                   dInit.interiorColor, false, this->cfg.rtl ) ;

         //* Integrate dctSCROLLEXT control with dialog border *
         cdConnect cdc ;
         cdc.ul2Left = cdc.ur2Right = 
         cdc.ll2Left = cdc.lr2Right = cdc.connection = true ;
         dp->ConnectControl2Border ( lrListSE, cdc ) ;

         //* Initialize display data for dctSCROLLEXT control *
         ssetData sData( serecPtr, serecColors, rCnt, ZERO, true ) ;
         dp->SetScrollextText ( lrListSE, sData ) ;

         dp->RefreshWin () ;           // make everything visible

         uiInfo Info ;                 // user interface data returned here
         short  icIndex = ZERO ;       // index of control with input focus
         bool   done = false ;         // loop control

         while ( ! done )
         {
            //*************************************************
            //* If focus is currently on a Pushbutton control *
            //*************************************************
            if ( ic[icIndex].type == dctPUSHBUTTON )
            {
               if ( Info.viaHotkey )
                  Info.HotData2Primary () ;
               else
                  icIndex = dp->EditPushbutton ( Info ) ;
               if ( Info.dataMod != false )
               {
                  if ( icIndex == lrLoadPB )
                  {
                     short seIndx = dp->GetScrollextSelect ( lrListSE ) ;
                     this->ud = this->udCap[seIndx] ;
                     this->Recalculate () ;
                  }
                  done = true ;
               }
            }

            //************************************************
            //* If focus is currently on a Scrollext control *
            //************************************************
            else if ( ic[icIndex].type == dctSCROLLEXT )
            {
               Info.viaHotkey = false ;   // ignore hotkey
               icIndex = dp->EditScrollext ( Info ) ;
#if 0    // TEMP - CAPTURE
dp->CaptureDialog ( "capturedlg.txt" ) ;
dp->CaptureDialog ( "capturedlg.html", true, false, 
                    "infodoc-styles.css", 4, false, nc.blR ) ;
#endif   // CAPTURE
            }

            //* If user exited the control edit method via a hotkey,*
            //* then the new control already has focus. Otherwise,  *
            //* move input focus to next/previous control.          *
            if ( done == false && Info.viaHotkey == false )
            {
               if ( Info.keyIn == nckSTAB )
                  icIndex = dp->PrevControl () ; 
               else
                  icIndex = dp->NextControl () ;
            }
         }     // while()
      }        // OpenWindow()

      if ( dp != NULL )                // delete (close) the dialog
         delete dp ;
      if ( seRecords != NULL )         // release the dynamic allocation
         delete [] seRecords ;
      if ( serecPtr != NULL )
         delete [] serecPtr ;
      if ( serecColors != NULL )
         delete [] serecColors ;

      //* Restore the obscured dialog *
      this->statsDlg->RefreshWin () ;
   }

   //* If no records captured, give user a clue. *
   else
   {
      chrono::duration<short, std::milli>aMoment( 1500 ) ;  // pause
      this->UserMessage ( LogMsgs[this->cfg.lang][lmNoAccess], this->cfg.em, true ) ;
      this_thread::sleep_for( aMoment ) ;
      this->UserMessage ( "", this->cfg.em, true ) ;
   }

}  //* End LoadRecord() *

//*************************
//*       lfsGraph        *
//*************************
//******************************************************************************
//* Create a graph of user's exercise goal vs. time.                           *
//* Called by LigfileSummar() to define and display a graph in the stats       *
//* dialog window.                                                             *
//*                                                                            *
//* Input  : wpOrig : upper-left corner of display area                        *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Notes:                                                                     *
//* ------                                                                     *
//* 1) The horizontal axis is one column for each record in the log.           *
//*    For example, if the log is a record of daily exercise, then each cell   *
//*    would represent one day. If there is no record for a given day, a blank *
//*    column represents the lazy day.                                         *
//*    A) Alternate layout would be to spread the range of records to fit a    *
//*       daily sequence, leaving a blank column for a missed day(s). To do    *
//*       this would require user intervention and may not be practical.       *
//*    B) A weekly record is an obvious possibility. In this case, each        *
//*       column would represent a week.                                       *
//*    C) If the number of records is less than half the number of columns,    *
//*       we could use two columns for each record. That might be fun.         *
//*    D) If the logfile covers more days than we have divisions, a design     *
//*       decision is needed. We could implement a sliding horizontal graph    *
//*       which would provide a window of records equivalent to the available  *
//*       display columns. This would require user interaction (yuck!), but    *
//*       would be extensible.                                                 *
//* 2) The units of the vertical axis are determined by the user's "goal"      *
//*    as specified in the configuration file and recorded in the members      *
//*    'gValue' and 'gUnits'.                                                  *
//* 3) The divisions of the vertical axis are set at 16 rows * 8 divisions     *
//*    per row == 128 divisions. This makes for easy math, but if the          *
//*    vertical space available changes, this may need to be adjusted.         *
//* 4) The vertical grid is sized to the data range to be displayed, with a    *
//*    small amount at the top of the chart left unused for aesthetic reasons. *
//*    The data are mapped over 93.0% of the vertical grid size.               *
//*    a) the minimum value displays as 5.0% of the vertical height which      *
//*       corresponds to approximately 1/2 of one vertical cell.               *
//*       It would be a program error to have an entry with a value < 0.0 and  *
//*       it is illogical to have a value of 0.0; however, we test for these.  *
//*       If 'curVal' <= 0.0 then no bar is drawn for that value.              *
//*    b) the maximum value displays as 93.0% of the vertical height of the    *
//*       grid. This fills approximately 15 of the 16 vertical cells.          *
//* 5) The 'modff' function breaks the floating-point value into its integer   *
//*    and fractional components. (Use 'modf' for doubles.)                    *
//*       float modff ( float fVal, float* intPart ) ;                         *
//*    Note that the integer part is always truncated, not rounded.            *
//*       float intPart, fracPart ;                                            *
//*       fracPart = modff ( fVal, &intPart ) ;                                *
//*    Programmer's Note: The internal format of floating-point numbers may    *
//*    be hardware specific, so if problems arise, please yell at your         *
//*    compiler vendor. (As an historical note, your author wrote the          *
//*    floating-point parsing routines for several early 8- and 16-bit         *
//*    microprocessors.)                                                       *
//*                                                                            *
//*    See also: float rintf ( float f ) ;                                     *
//*               float fWhole = rintf ( f ) ;                                 *
//*    This can be used if the fractional division is used internally to       *
//*    round according to the current "rounding mode."                         *
//*    See also: float truncf ( float f ) ;                                    *
//*               float fWhole = truncf ( f ) ;                                *
//*    This can be used if the fractional division is to be discarded.         *
//*    (rounding is always toward 0.0: 1.5->1.0  and  -1.5->-1.0)              *
//*                                                                            *
//* -------------------------------------------------------------------------- *
//* TEMP - To Make a Widget: (move this functionality to the NcDialog API)     *
//* ------------------------                                                   *
//* a) Define area and position.                                               *
//* b) Specify verticle or horizontal bars                                     *
//* c) Specify colors for background, bars, labels, grid, trend line, etc.     *
//* d) Define vertical and horizontal units                                    *
//* e) Define size of a horizontal cell (1-5 character cells)                  *
//* f) Specify character for drawing the bars                                  *
//* g) Spacing between bars.                                                   *
//* h) Provide pointer to data array or a callback method to retrieve data     *
//*    values from the main program.                                           *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//******************************************************************************

void Exercalc::lfsGraph ( const winPos& wpOrig )
{
#if 0    // UNDER CONSTRUCTION - LogfileSummary(GRAPH) - REPLACE WITH CHART WIDGET
   Chart *cp = NULL ;            // pointer to Chart instance

   //* Define the chart parameters *
   chartDef cdef = 
   {
      this->statsDlg,               // dialog pointer
      wpOrig.ypos,                  // Y origin
      wpOrig.xpos,                  // X origin
      short(sdInit.dLines - wpOrig.ypos - 2),     // rows
      short(sdInit.dColumns - wpOrig.xpos - 4),   // columns
      1,                            // Y offset
      1,                            // X offset
      1,                            // footer rows
      cellDIV,                      // bar width (divisions)
      ZERO,                         // bar spacing
      ctLowerLeft,                  // chart type
      this->cfg.bb,                 // border color
      this->cfg.bb,                 // title color
      this->cfg.bb,                 // text color
      this->cfg.sb,                 // bold color
      this->cfg.bb,                 // grid color
      this->cfg.bb,                 // bar color
      this->udcCount,               // data element count
      ZERO,                         // offset into data array
      this->udCap,                  // data array
      NULL,                         // bar-color array
      ncltSINGLE,                   // border style
      ncltSINGLE,                   // grid style (see below)
      vertLabel,                    // vertical-axis label
      horizLabel,                   // horizontal-axis label
      NULL,                         // title text
      NULL,                         // header text
      NULL,                         // footer text
      false,                        // 'true' if horizontal bars, 'false' if vertical bars
      false,                        // 'false' if full bar, 'true' if pseudo-line
      false,                        // border flag
   } ;


#else    // HARD-CODED GRAPH
   const float 
      vDIV  = 16.0 * (float)cellDIV,   // total vertical-axis divisions
      minDIV = vDIV * 0.05,            // number of divisions for a "minimum value" (5%)
      maxDIV = vDIV * 0.93,            // number of divisions for a "maximum value" (93%)
      rngDIV = maxDIV - minDIV ;       // range of available divisions
   const short wcINDX = 7 ;            // index of whole-cell character

   gString gsOut, gsGoal ;             // text formatting
   winPos wp = wpOrig,                 // cursor positioning
          wpt,                         // cursor positioning (temp)
          wpBase ;                     // base position in chart
   float  minVal = 0.0,                // minimum unit value
          maxVal = 0.0,                // maximum unit value
          rngVal = 0.0,                // range of unit values
          curVal = 0.0,                // value under analysis for bar
          //minDiv = 0.0,                // divisions representing minimum value
          //maxDivs = 0.0,               // divisions representing maximum value
          curDiv = 0.0,                // divisions representing current value
          //uprDiv = 0.0,              // units per vertical division (interesting, but unused)
          fWhole,                      // integer portion of floating-point value
          fFract,                      // fractional portion of floating-point value
          fDays ;                      // number of exercise days
   short  gCols = sdInit.dColumns - wp.xpos - 4,   // graph-area width
          //gRows = sdInit.dLines - wp.ypos - 2,     // graph-area height (currently unused)
          //hDiv  = gCols,               // horizontal-axis divisions (currently unused)
          iWhole,                      // number of whole cells in bar
          iFract,                      // index of partial-cell character
          totIndex = this->udcCount,   // index of the 'totals' record
          avgIndex = totIndex + 1,     // index of the 'averages' record
          axindx = ZERO ;              // index into 'axisLabelV[]'

   //* Range the value set for the specified units *
   switch ( this->gUnits )
   {
      case vuKmeters:
         minVal = maxVal = this->udCap[ZERO].kmeters ;
         axindx = alhDist ;      // axis label index
         gsGoal.compose( axisLabelG[this->cfg.lang][algDist], &this->gValue ) ;
         for ( short i = 1 ; i < this->udcCount ; ++i )
         {
            if ( this->udCap[i].kmeters < minVal )
               minVal = this->udCap[i].kmeters ;
            if ( this->udCap[i].kmeters > maxVal )
               maxVal = this->udCap[i].kmeters ;
         }
         break ;
      case vuKcalories:
         minVal = this->udCap[ZERO].kcal ;
         axindx = alhEner ;      // vertical-axis label index
         gsGoal.compose( axisLabelG[this->cfg.lang][algEner], &this->gValue ) ;
         for ( short i = 1 ; i < this->udcCount ; ++i )
         {
            if ( this->udCap[i].kcal < minVal )
               minVal = this->udCap[i].kcal ;
            if ( this->udCap[i].kcal > maxVal )
               maxVal = this->udCap[i].kcal ;
         }
         break ;
      case vuMinutes:
      default:
         minVal = this->udCap[ZERO].mins ;
         axindx = alhTime ;      // vertical-axis label index
         short hr  = short(this->gValue / 60.0),
               min = short(this->gValue - (hr * 60)) ;
         gsGoal.compose( axisLabelG[this->cfg.lang][algTime], &hr, &min ) ;
         for ( short i = 1 ; i < this->udcCount ; ++i )
         {
            if ( this->udCap[i].mins < minVal )
               minVal = this->udCap[i].mins ;
            if ( this->udCap[i].mins > maxVal )
               maxVal = this->udCap[i].mins ;
         }
         break ;
   } ;
   // Programmer's Note: If 'rngVal' == ZERO (all records have the same value), 
   // no data will be displayed, so we ensure that data will be represented 
   // by forcing a positive value range. (Data will display as a straight line.)
   rngVal = maxVal - minVal ;       // range of unit values
   if ( rngVal <= 0.0 )             // prevent divide-by-zero
      rngVal = 1.0 ;
   //uprDiv = rngVal / rngDIV ;       // units per vertical division
   //minDivs = minDIV ;               // divisions displayed for minimum value
   //maxDivs = (rngVal / vDIV) ;      // upper limit of used divisions

   //* Calculate number of days covered by the records *
   fDays = this->udCap[avgIndex].timeStamp - this->udCap[totIndex].timeStamp ;
   fDays = (fDays / 60.0 / 60.0 / 24.0) + 1 ;
   if ( fDays < 1.0 )   fDays = 1.0 ;

   //* Position the goal line and draw the goal-line label.*
   //* The goal line itself will be drawn after the base   *
   //* position of the grid is known.                      *
   curDiv = rngDIV * ((this->gValue - minVal) / rngVal) ;
   if ( curDiv < minDIV ) curDiv = minDIV ;
   if ( curDiv > maxDIV ) curDiv = maxDIV ;
   iWhole = short(trunc ( curDiv )) ;
   iWhole /= cellDIV ;
   winPos gLine( -(iWhole), wp.xpos + 2 ) ;
   //* Goal-line label and goal line stub *
   wpt = this->statsDlg->WriteParagraph ( wp.ypos, 
                                          (wp.xpos + (gCols - gsGoal.gscols() + 2)), 
                                          gsGoal, this->cfg.em ) ;
   wpt = this->statsDlg->WriteParagraph ( wpt, L"┐\n", this->cfg.sb ) ;

   //*****************
   //* Draw the grid *
   //*****************
   //* Vertical axis *
   wp = this->statsDlg->WriteParagraph ( wp.ypos, ++wp.xpos, L"│\n", this->cfg.bb ) ;
   for ( short i = ZERO ; i < (vDIV / cellDIV - 1) ; ++i )
      wp = this->statsDlg->WriteParagraph ( wp, L"├\n", this->cfg.bb ) ;
   wp = this->statsDlg->WriteChar ( wp, L'└', this->cfg.bb ) ;
   wpBase = { short(wp.ypos - 1), wp.xpos } ;    // chart-data base position

   //* Horizontal axis *
   for ( short i = ZERO ; i < gCols ; ++i )
      wp = this->statsDlg->WriteString ( wp, L"┴", this->cfg.bb ) ;
   ++wp.ypos ; wp.xpos = wpOrig.xpos ;

   //* Axis labels *
   wp.xpos = wpOrig.xpos ;
   gsOut.compose( axisLabelH[this->cfg.lang][axindx], &minVal, &maxVal, &fDays ) ;
   short offset = gsOut.find( "--" ) ;
   while ( (gsOut.gscols()) < gCols )
      gsOut.insert( L'-', offset ) ;
   this->statsDlg->WriteParagraph ( wp, gsOut, this->cfg.bb ) ;

   //* Draw the 'goal' line *
   gLine.ypos += wpBase.ypos ;
   if ( gLine.ypos <= wpt.ypos )
      gLine.ypos = wpt.ypos ;
   if ( gLine.ypos >= wpBase.ypos )
      gLine.ypos = wpBase.ypos ;
   while ( gLine.xpos < wpt.xpos )
      gLine = this->statsDlg->WriteChar ( gLine, L'─', this->cfg.sb ) ;
   this->statsDlg->WriteChar ( gLine.ypos--, gLine.xpos, L'┘', this->cfg.sb ) ;
   while ( gLine.ypos >= wpt.ypos )
      this->statsDlg->WriteChar ( gLine.ypos--, gLine.xpos, L'│', this->cfg.sb ) ;

   #if 0    // FOR DEBUGGING ONLY
   this->statsDlg->ClearArea ( 1, 1, 12, sdInit.dColumns - 2 ) ;
   gsOut.compose ( "vDIV      : %.3f\n"
                   "minDIV    : %.3f\n"
                   "maxDIV    : %.3f\n"
                   "rngDIV    : %.3f\n"
                   "wcINDX    : %hd\n"
                   "minVal    : %.3f\n"
                   "maxVal    : %.3f\n"
                   "rngVal    : %.3f\n",
                   //"uprDiv    : %.3f\n",
                   &vDIV, &minDIV, &maxDIV, &rngDIV, &wcINDX,
                   &minVal, &maxVal, &rngVal/*, &uprDiv*/ ) ;
   this->statsDlg->WriteParagraph ( 1, 16, gsOut, this->cfg.bb, true ) ;
   #endif   // FOR DEBUGGING ONLY


   //* Display as much of the data as will fit on the horizonal axis *
   for ( short indx = ZERO ; indx < this->udcCount ; ++indx )
   {
      wp = wpBase ;                             // position of bar-graph character
      //* Get the value from the record *
      if ( this->gUnits == vuKmeters)
         curVal = this->udCap[indx].kmeters ;
      else if ( this->gUnits == vuKcalories )
         curVal = this->udCap[indx].kcal ;
      else  // ( this->gUnits == vuMinutes )
         curVal = this->udCap[indx].mins ;
      if ( curVal > 0.0 )
      {
         curDiv = rngDIV * ((curVal - minVal) / rngVal) ; // convert to vertical divisions
         if ( curDiv < minDIV ) curDiv = minDIV ;  // range limiting
         if ( curDiv > maxDIV ) curDiv = maxDIV ;
         fFract = modff ( curDiv, &fWhole ) ;      // separate whole and fractional divisions
         iWhole = short(fWhole) ;                  // whole divisions
         iFract = iWhole % cellDIV ;               // divisions in fractional cell
         iWhole /= cellDIV ;                       // whole cells
         if ( fFract > 0.5 ) ++iFract ;            // round up to next division
         iFract -= 1 ;                             // convert fractional cell to index

         //* Draw the vertical bar *
         for ( short i = ZERO ; i < iWhole ; ++i )
            this->statsDlg->WriteChar ( wp.ypos--, wp.xpos, vBlock[wcINDX], this->cfg.bb ) ;
         if ( iFract >= ZERO )
            this->statsDlg->WriteChar ( wp, vBlock[iFract], this->cfg.bb ) ;
      }
      if ( ++wpBase.xpos > (gCols + 2) )  // advance to next cell
         break ;                          // last column of chart has been filled
   }

   this->statsDlg->RefreshWin () ;
#endif   // U/C

}  //* End lfsGraph() *

//*************************
//*      RenameFile       *
//*************************
//******************************************************************************
//* Rename the specified source file to specified destination file.            *
//*                                                                            *
//* For the rename() primitive: If srcPath refers to a symbolic link the       *
//* link is renamed; if trgPath refers to a symbolic link the link will be     *
//* overwritten.                                                               *
//*                                                                            *
//* Input  : srcPath : full path/filename specification of source              *
//*          trgPath : full path/filename specification of target              *
//*                                                                            *
//* Returns: 'true' if rename is successful, else 'false'                      *
//******************************************************************************
//* Programmer's Note: The C-language version of 'rename' INSISTS on doing its *
//* own escaping of special characters. If it receives escaped characters, it  *
//* escapes the '\' characters, negating the previous escape sequence.         *
//******************************************************************************

bool Exercalc::RenameFile ( const gString& srcPath, const gString& trgPath )
{
   bool status = true ;

   if ( (rename ( srcPath.ustr(), trgPath.ustr() )) != OK )
   {
      status = false ;
   }
   return status ;

}  //* End RenameFile() *

//*************************
//*     DeleteFile        *
//*************************
//******************************************************************************
//* Delete (unlink) the specified source file.                                 *
//*                                                                            *
//* Input  : trgPath : full path/filename specification of target              *
//*                                                                            *
//* Returns: 'true' if delete is successful, else 'false'                      *
//******************************************************************************
//* Programmer's Note: The C-language version of 'unlink' INSISTS on doing its *
//* own escaping of special characters. If it receives escaped characters, it  *
//* escapes the '\' characters, negating the previous escape sequence.         *
//******************************************************************************

bool Exercalc::DeleteFile ( const gString& trgPath )
{
   bool status = true ;

   if ( (status = unlink ( trgPath.ustr() )) != OK )
   {
      status = false ;
   }
   return status ;

}  //* End DeleteFile() *

//*************************
//*       Realpath        *
//*************************
//******************************************************************************
//* Normalize a file specification string.                                     *
//*                                                                            *
//* Given a filename, relative path/filename or full path/filename, with or    *
//* without environmental variables or symbolic links, resolve to an absolute  *
//* path/filename string.                                                      *
//*                                                                            *
//* Programmer's Note: If user specifies a path or filename that contains      *
//* spaces, then he/she/it must enclose the string with quotations marks       *
//* (either single or double). Otherwise part of the string will be            *
//* interpreted as a separate, probably invalid argument.                      *
//*                                                                            *
//* Input  : rpath   : receives full filespec string                           *
//*                                                                            *
//* Returns: 'true'  if successful, rpath contains converted filespec          *
//*          'false' if parsing error, rpath is unchanged                      *
//******************************************************************************
//* NOTES:                                                                     *
//* ------                                                                     *
//* 'realpath' will follow symbolic links; however, it WILL NOT do tilde ('~') *
//*            or environment-variable expansion, so do these substitutions    *
//*            before calling.                                                 *
//*                                                                            *
//* 'wordexp'                                                                  *
//* The 'wordexp' function is a pretty cool, but watch out:                    *
//*  a) wordexp returns ZERO on success or WRDE_BADCHAR (2) if an invalid      *
//*     character is detected in the stream.                                   *
//*     - Note that an empty string will pass the scan, but then               *
//*       'wexp.we_wordc' will be ZERO.                                        *
//*  b) Dynamic memory allocation happens, so remember to free it.             *
//*     - If a bad character in the stream, then freeing the dynamic           *
//*       allocation will cause a segmentation fault. This is a Standard       *
//*       Library bug, so the work-around is to call 'wordfree' ONLY if        *
//*      'wordexp' returns success.                                            *
//*  c) SPACE characters delimit the parsing, so if the path contains spaces,  *
//*     then we must concatenate the resulting substrings, reinserting the     *
//*     space characters. Leading and trailing spaces are ignored.             *
//*     (We assume that a path will never contain a TAB or NEWLINE character.) *
//*  d) wordexp will choke on the following characters in the stream:          *
//*             & | ; < >  \n     (unless they are quoted)                     *
//*     - Parentheses and braces should appear ONLY as part of a token to be   *
//*       expanded (or if they are quoted).                                    *
//*     - single-quotes ''' are seen as delimiters unless they are quoted.     *
//*     - double-quotes '"' are also seen as delimiters unless they are        *
//*       quoted; however, the use of double-quotes is further complicated by  *
//*       the fact that string literals are always delimited by double-quotes, *
//*       so a double-quote within a set of double-quotes must be SENT to us   *
//*       as quoted in order for it to arrive as the double-quote character.   *
//*  e) Wildcard characters '*' and '?' are tricky, but '?' especially is a    *
//*     common filename character.                                             *
//*  f) The tokens we are most likely to see are '${HOME}' and '~'.            *
//*     These are both expanded as '/home/sam' or the equivalent.              *
//*  g) The most likely special characters (especially for song titles, book   *
//*     title and similar) that represent themselves are:                      *
//*          single-quote '''  question mark '?' and parentheses '(' ')'       *
//*  h) Note that the characters: '$'  '{'  '}'  '`'  are reserved by the      *
//*     shell for expansion of environment variables, and therefore should     *
//*     not be used as filename characters; therefore, we do not quote them    *
//*     here.                                                                  *
//******************************************************************************

bool Exercalc::Realpath ( gString& rpath )
{
   gString gstmp = rpath,        // working copy of source
           relpath ;
   short wchars, index = ZERO ;
   const wchar_t* wptr = gstmp.gstr( wchars ) ;
   bool status = true ;          // return value

   //* Quote (escape) the set of special characters so     *
   //* wordexp() will process them without special meaning.*
   while ( index < wchars )
   {
      if ( (wptr[index] == L'(') || (wptr[index] == L')') ||
           (wptr[index] == L'"') || (wptr[index] == L'\'') ||
           (wptr[index] == L'?') || (wptr[index] == L'&') ||
           (wptr[index] == L'<') || (wptr[index] == L'>') ||
           (wptr[index] == L';') || (wptr[index] == L'|')
         )
      {
         if ( (index == ZERO) || (wptr[index - 1] != L'\\') )
         {
            gstmp.insert( L'\\', index ) ;
            ++wchars ;
            index += 2 ;
         }
         else
            ++index ;
      }
      else
         ++index ;
   }

   //* Perform environment-variable and tilde substitutions. (see notes above) *
   wordexp_t wexp ;              // target structure
   if ( (wordexp ( gstmp.ustr(), &wexp, ZERO )) == ZERO )
   {
      if ( wexp.we_wordc > ZERO )   // if we have at least one element
      {
         relpath.clear() ;
         for ( UINT i = ZERO ; i < wexp.we_wordc ; )
         {
            relpath.append( wexp.we_wordv[i++] ) ;
            if ( i < wexp.we_wordc )      // re-insert stripped spaces (see note)
               relpath.append( L' ' ) ;
         }
      }
      wordfree ( &wexp ) ;
   }
   else
      status = false ;

   //* Expand relative paths and paths containing symbolic links.*
   if ( status != false )
   {
      char rtmp[MAX_PATH] ;      // temp buffer

      if ( (realpath ( relpath.ustr(), rtmp )) != NULL )
         rpath = rtmp ;
      else
         status = false ;
   }

   return status ;

}  //* End Realpath() *

//*************************
//*     TargetExists      *
//*************************
//******************************************************************************
//* Determine whether specified file exists and return its filetype.           *
//*                                                                            *
//* Input  : trgPath : filespec of target file                                 *
//*          fType   : (by reference)                                          *
//*                  : if target exists, initialized to target fileType        *
//*                    else fmUNKNOWN_TYPE                                     *
//*                                                                            *
//* Returns: 'true' if target exists, else 'false'                             *
//******************************************************************************
//* Programmer's Note: This method is grossly inefficient, but this            *
//* application requires simplicity of design over speed.                      *
//******************************************************************************

bool Exercalc::TargetExists ( const gString& trgPath, fmFType& fType )
{
   tnFName fs ;               // file stats
   bool exists = false ;      // return value

   if ( (this->GetFileStats ( trgPath, fs )) == OK )
   {
      fType = fs.fType ;
      exists = true ;
   }
   return exists ;

}  //* End TargetExists() *

//*************************
//*     GetFileStats      *
//*************************
//******************************************************************************
//* Perform a 'stat' ('lstat') on the file specified by the full path string.  *
//*                                                                            *
//* Input  : trgPath: full path/filename specification                         *
//*          fStats : (by reference) receives the stat data for the file       *
//*                                                                            *
//* Returns: OK if successful, all data members of fStats will be initialized  *
//*          Else returns system 'errno' value                                 *
//******************************************************************************

short Exercalc::GetFileStats ( const gString& trgPath, tnFName& fStats )
{
   short success ;         // return value

   //* Provides an orderly crash in case of failure *
   fStats.modTime.reset() ;
   fStats.fBytes = ZERO ;
   fStats.fType = fmUNKNOWN_TYPE ;
   fStats.readAcc = fStats.writeAcc = false ;

   if ( (success = lstat64 ( trgPath.ustr(), &fStats.rawStats )) == OK )
   {
      //* Extract filename from caller's string *
      short indx = trgPath.findlast( L'/' ) ;
      if ( indx < ZERO )   indx = ZERO ;
      else                 ++indx ;
      gString fName( &trgPath.gstr()[indx] ) ;
      fName.copy( fStats.fName, MAX_FNAME ) ;

      fStats.fBytes = fStats.rawStats.st_size ;    // number of bytes in file

      //* Decode the filetype *
      UINT mode = fStats.rawStats.st_mode ;
      if ( (S_ISREG(mode)) != false )        fStats.fType = fmREG_TYPE ;
      else if ( (S_ISDIR(mode)) != false )   fStats.fType = fmDIR_TYPE ;
      else if ( (S_ISLNK(mode)) != false )   fStats.fType = fmLINK_TYPE ;
      else if ( (S_ISCHR(mode)) != false )   fStats.fType = fmCHDEV_TYPE ;
      else if ( (S_ISBLK(mode)) != false )   fStats.fType = fmBKDEV_TYPE ;
      else if ( (S_ISFIFO(mode)) != false )  fStats.fType = fmFIFO_TYPE ;
      else if ( (S_ISSOCK(mode)) != false )  fStats.fType = fmSOCK_TYPE ;

      //* Decode the mod time *
      fStats.modTime.epoch = (int64_t)fStats.rawStats.st_mtime ;
      fStats.modTime.sysepoch = (time_t)fStats.rawStats.st_mtime ; // (possible narrowing)
      Tm bdt ;       // receives broken-down time
      if ( (localtime_r ( &fStats.modTime.sysepoch, &bdt )) != NULL )
      {
         //* Translate to localTime format (timezone data not decoded) *
         fStats.modTime.date    = bdt.tm_mday ;         // today's date
         fStats.modTime.month   = bdt.tm_mon + 1 ;      // month
         fStats.modTime.year    = bdt.tm_year + 1900 ;  // year
         fStats.modTime.hours   = bdt.tm_hour ;         // hour
         fStats.modTime.minutes = bdt.tm_min ;          // minutes
         fStats.modTime.seconds = bdt.tm_sec ;          // seconds
         fStats.modTime.day     = bdt.tm_wday ;         // day-of-week (0 == Sunday)
         fStats.modTime.julian  = bdt.tm_yday ;         // Julian date (0 == Jan.01)
      }

      //* Initialize the read-access and write-access flags *
      fStats.readAcc = bool((access ( trgPath.ustr(), R_OK )) == ZERO) ;
      if ( fStats.readAcc != false && fStats.fType == fmDIR_TYPE )
         fStats.readAcc = bool((access ( trgPath.ustr(), X_OK )) == ZERO) ;
      fStats.writeAcc = bool((access ( trgPath.ustr(), W_OK )) == ZERO) ;
   }
   else
      success = errno ;

   return success ;

}  //* End GetFileStats() *

//*************************
//*       GetCWD          *
//*************************
//******************************************************************************
//* Returns the path of user's current working directory.                      *
//*                                                                            *
//* Input  : cwd : (by reference) receives target path string                  *
//*                                                                            *
//* Returns: 'true' if successful                                              *
//*          'false' if target directory does not exist or is inaccessible     *
//******************************************************************************

bool Exercalc::GetCWD ( gString& cwd )
{
   char  buff[MAX_PATH] ;
   bool  status = false ;

   if ( (getcwd ( buff, MAX_PATH )) != NULL )
   {
      cwd = buff ;
      status = true ;
   }
   return status ;

}  //* End GetCWD() *

//************************
//*    GetLocalTime      *
//************************
//******************************************************************************
//* Get the system timecode and convert it to localTime format.                *
//*                                                                            *
//* Input  : lt  : (by reference, initial values ignored)                      *
//*                on return, contains decoded date/time                       *
//*                                                                            *
//* Returns: true if successful, false if system call fails                    *
//******************************************************************************
//* Programmer's Note: Currently we do not decode the timezone and DST fields. *
//*                                                                            *
//* -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  *
//* System conversion from time_t code to local time is returned in this       *
//* structure.                                                                 *
//*                                                                            *
//*  struct tm                                                                 *
//*  {                                                                         *
//*     int  tm_sec ;          // 0-59  (leap seconds to 61)                   *
//*     int  tm_min ;          // 0-59                                         *
//*     int  tm_hour ;         // 0-23                                         *
//*     int  tm_mday ;         // 1-31                                         *
//*     int  tm_mon ;          // 0-11                                         *
//*     int  tm_year ;         // since 1900                                   *
//*     int  tm_wday ;         // 0-6                                          *
//*     int  tm_yday ;         // 0-365                                        *
//*     int  tm_isdst ;        // >0 == is DST, 0 == not DST, <0 == unknown    *
//*  } ;                                                                       *
//*                                                                            *
//* -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  *
//* Note that we do a bit of defensive programming in anticipation of the      *
//* dreaded year 2038 overflow of the 32-bit time_t type.                      *
//******************************************************************************

bool Exercalc::GetLocalTime ( localTime& lt )
{
   bool success = false ;                       // return value
  
   lt.reset() ;                                 // set to default values
   if ( (time ( &lt.sysepoch )) != (-1) )
   {
      if ( lt.sysepoch >= ZERO )                // reported time is AFTER the epoch
      {
         lt.epoch = (int64_t)lt.sysepoch ;      // promote to 64-bit
         //* Decode the system time *
         Tm    tm ;                             // Linux time structure
         if ( (localtime_r ( &lt.sysepoch, &tm )) != NULL )
         {
            //* Translate to localTime format *
            lt.julian   = tm.tm_yday ;          // Julian date
            lt.day      = tm.tm_wday ;          // 0 == Sunday ... 6 == Saturday
            lt.date     = tm.tm_mday ;          // today's date
            lt.month    = tm.tm_mon + 1 ;       // month
            lt.year     = tm.tm_year + 1900 ;   // year
            lt.hours    = tm.tm_hour ;          // hour
            lt.minutes  = tm.tm_min ;           // minutes
            lt.seconds  = tm.tm_sec ;           // seconds
            snprintf ( lt.iso, ltISO_LEN, "%04hu-%02hu-%02huT%02hu:%02hu:%02hu",
                       lt.year, lt.month, lt.date, lt.hours, lt.minutes, lt.seconds ) ;
            success = true ;
         }
      }
      else
      {  /* SYSTEM ERROR - time_t OVERFLOW */ }
   }
   return success ;

}  //* End GetLocalTime() *

//************************
//*    GetLocalTime      *
//************************
//******************************************************************************
//* Return the current system-local epoch time.                                *
//*                                                                            *
//* Input  : epoch_time : (by reference, initial value ignored)                *
//*                       on return, contains epoch timecode                   *
//*                       (or ZERO if system call fails)                       *
//*                                                                            *
//* Returns: true if successful, false if system call fails                    *
//******************************************************************************

bool Exercalc::GetLocalTime ( int64_t& epoch_time )
{
   time_t calTime ;                             // simple calendar time
   bool   success = false ;                     // return value

   epoch_time = ZERO ;                          // initialize caller's value

   if ( (calTime = time ( NULL )) != (-1) )
   {
      epoch_time = (int64_t)calTime ;
      success = true ;
   }

   return success ;

}  //* End GetLocalTime() *

//*************************
//*    DecodeEpochTime    *
//*************************
//******************************************************************************
//* Convert GNU/UNIX epoch time code (seconds since the epoch: Jan. 01, 1970)  *
//* to local time in a structured format (class localTime).                    *
//*                                                                            *
//* See the info page for the 'stat' command for more information.             *
//* See also the notes in DirEntryDisplayFormat() above.                       *
//*                                                                            *
//* Input  : eTime  : epoch time (this is a signed, 64-bit value)              *
//*                   (See note in definition of localTime class.)             *
//*          ftTime : (by reference, initial values ignored)                   *
//*                                                                            *
//* Returns: nothing, but all members of ftTime will be initialized            *
//******************************************************************************
//* System conversion from time_t code to local time is returned in this       *
//* structure.                                                                 *
//*                                                                            *
//*  struct tm                                                                 *
//*  {                                                                         *
//*     int  tm_sec ;          // 0-59  (leap seconds to 61)                   *
//*     int  tm_min ;          // 0-59                                         *
//*     int  tm_hour ;         // 0-23                                         *
//*     int  tm_mday ;         // 1-31                                         *
//*     int  tm_mon ;          // 0-11                                         *
//*     int  tm_year ;         // since 1900                                   *
//*     int  tm_wday ;         // 0-6                                          *
//*     int  tm_yday ;         // 0-365                                        *
//*     int  tm_isdst ;        // >0 == is DST, 0 == not DST, <0 == unknown    *
//*  } ;                                                                       *
//*                                                                            *
//* -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  *
//******************************************************************************

void Exercalc::DecodeEpochTime ( int64_t eTime, localTime& ftTime )
{
   ftTime.reset() ;           // initialize caller's data
   if ( eTime >= ZERO )       // defend against stupidity
   {  //* Width of time_t is system dependent *
      ftTime.epoch = eTime ;
      ftTime.sysepoch = (time_t)eTime ;   // (possible narrowing)

      //* Receives the broken-down time.              *
      Tm bdt ;       // receives broken-down time
      if ( (localtime_r ( &ftTime.sysepoch, &bdt )) != NULL )
      {
         //* Translate to localTime format *
         ftTime.date    = bdt.tm_mday ;         // today's date
         ftTime.month   = bdt.tm_mon + 1 ;      // month
         ftTime.year    = bdt.tm_year + 1900 ;  // year
         ftTime.hours   = bdt.tm_hour ;         // hour
         ftTime.minutes = bdt.tm_min ;          // minutes
         ftTime.seconds = bdt.tm_sec ;          // seconds
         ftTime.day     = bdt.tm_wday ;         // day-of-week (0 == Sunday)
         ftTime.julian  = bdt.tm_yday ;         // Julian date (0 == Jan.01)

         //* Create the ISO time string *
         snprintf ( ftTime.iso, ltISO_LEN, "%04hu-%02hu-%02huT%02hu:%02hu:%02hu",
                    ftTime.year, ftTime.month, ftTime.date, 
                    ftTime.hours, ftTime.minutes, ftTime.seconds ) ;

         #if DECODE_TZ != 0   // time-zone data
         ftTime.gmtoffset = bdt.tm_gmtoff ;     // offset as number of seconds
         ftTime.dst = (bdt.tm_isdst > ZERO ) ? true : false ; // DST in effect

         #if 1    // additional time zone data (not very useful) *
         gString gs( bdt.tm_zone ) ;
         gs.copy( ftTime.timezone, ltFMT_LEN ) ;
         int absOffset = abs(ftTime.gmtoffset) ;
         int hOffset = absOffset / (60*60),     // convert offset seconds to 
             mOffset = absOffset % (60*60) ;    // hours and minutes
         if ( ftTime.gmtoffset < ZERO )
            hOffset = -(hOffset) ;
         gs.compose( L"UTC%+d:%02d", &hOffset, &mOffset ) ;
         gs.copy( ftTime.utc_zone, ltFMT_LEN ) ;
         #endif   // additional time zone data
         #endif   // DECODE_TZ
      }
   }

}  //* End DecodeEpochTime() *

//*************************
//*    GetTempdirPath     *
//*************************
//******************************************************************************
//* This is a stub function which provides access to the C++17 function:       *
//*                       temp_directory_path()                                *
//*                                                                            *
//* Because most programmer's will not reach C++17 for at least several        *
//* months, we isolate the call here and implement what we expect is the way   *
//* the library function will be implemented.                                  *
//*         (This WILL NOT work under Windoze--and we don't care.)             *
//*         (Screw Windoze, and the horse it rode in on.         )             *
//* -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- *
//* From <http://en.cppreference.com>                                          *
//* ---------------------------------                                          *
//* These functions are available in C++17 (released Jan. 2018):               *
//* const char* fs::temp_directory_path ( void ) ;                             *
//* const char* fs::temp_directory_path ( std::error_code& ec ) ;              *
//*                                                                            *
//* Returns A directory suitable for temporary files. The path is guaranteed   *
//* to exist and to be a directory. The overload that takes error_code&        *
//* argument returns an empty path on error.                                   *
//*                                                                            *
//* Exceptions: The overload that does not take a std::error_code& parameter   *
//* throws filesystem_error on underlying OS API errors, constructed with path *
//* to be returned as the first argument and the OS error code as the error    *
//* code argument. std::bad_alloc may be thrown if memory allocation fails.    *
//* The overload taking a std::error_code& parameter sets it to the OS API     *
//* error code if an OS API call fails, and executes ec.clear() if no errors   *
//* occur.                                                                     *
//*                                                                            *
//* On POSIX systems, the path may be the one specified in the environment     *
//* variables TMPDIR, TMP, TEMP, TEMPDIR, and, if none of them are specified,  *
//* the path "/tmp" is returned.                                               *
//*                                                                            *
//* Input  : tdPath : (by reference) receives the path string                  *
//*                                                                            *
//* Returns: 'true'  if successful, 'false' if system error                    *
//******************************************************************************

bool Exercalc::GetTempdirPath ( gString& tdPath )
{
   //* Default path of temp directory on GNU/Linux systems. *
   //* (Used only if environment variable not set.)         *
   const char* const dfltPath = "/tmp" ;

   const char* envPath ;            // returned by getenv()
   FileStats   fStats ;             // target filestats
   bool        status = false ;     // return value

   if ( (envPath = std::getenv ( "TMPDIR" )) == NULL )
      if ( (envPath = std::getenv ( "TMP" )) == NULL )
         if ( (envPath = std::getenv ( "TEMP" )) == NULL )
            if ( (envPath = std::getenv ( "TEMPDIR" )) == NULL )
               envPath = dfltPath ;
   tdPath = envPath ;
   if ( (stat64 ( tdPath.ustr(), &fStats )) == ZERO )
   {
      if ( ((S_ISDIR(fStats.st_mode)) != false) &&
           ((access ( tdPath.ustr(), R_OK )) == ZERO) && 
           ((access ( tdPath.ustr(), W_OK )) == ZERO) )
         status = true ;
   }
   return status ;

}  //* End GetTempdirPath()

//*************************
//*    CreateTempname     *
//*************************
//******************************************************************************
//* Create a unique path/filename for a temporary file.                        *
//*                                                                            *
//* Input  : tmpSpec: (by reference) receives the path/filename                *
//*                                                                            *
//* Returns: 'true'  if path/filename created successfully                     *
//*          'false' if library call failed                                    *
//******************************************************************************

bool Exercalc::CreateTempname ( gString& tmpSpec )
{
   bool  status = false ;

   char tn[gsMAXBYTES] ;
   gString gstmp( "%s/EC_XXXXXX", this->cfg.tmpPath ) ;
   gstmp.copy( tn, gsMAXBYTES ) ;
   int descriptor ;
   if ( (descriptor = mkstemp ( tn )) != (-1) )
   {
      close ( descriptor ) ;     // close the file
      tmpSpec = tn ;             // copy target path to caller's buffer
      status = true ;            // declare success
   }
   else
      tmpSpec.clear() ;
   return status ;

}  //* End CreateTempname() *

//*************************
//*    DeleteTempname     *
//*************************
//******************************************************************************
//* Delete the specified temporary file.                                       *
//*                                                                            *
//* Input  : tmpSpec: (by reference) filespec of temp file to be deleted       *
//*                                                                            *
//* Returns: 'true'  if file deleted                                           *
//*          'false' if library call failed (file not found, or system error)  *
//******************************************************************************

bool Exercalc::DeleteTempname ( const gString& tmpSpec )
{
   bool  status = false ;

   if ( (unlink ( tmpSpec.ustr())) == ZERO )
      status = true ;

   return status ;

}  //* End DeleteTempname() *

//*************************
//*       Cry4Help        *
//*************************
//******************************************************************************
//* Open the application Help file.                                            *
//* a) Shell out and invoke the 'info' documentation reader.                   *
//* b) Invoke the default browser with help in HTML format.                    *
//*                                                                            *
//* The 'exercalc.info' or 'exercalc.html' document _or_ a symlink to that     *
//* document must be in the same directory as the application binary.          *
//*                                                                            *
//* Input  : htmlFormat : if 'false', invoke help through info reader          *
//*                       if 'true',  load help into default browser           *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Exercalc::Cry4Help ( bool htmlFormat )
{
   fmFType ft ;
   gString hPath( "%s/exercalc.%s", 
                  this->cfg.appPath, (htmlFormat ? "html" : "info") ) ;
   if ( this->TargetExists ( hPath, ft ) )
   {
      if ( htmlFormat )    // help (HTML format)
      {
         this->LaunchDefaultApplication ( hPath.ustr() ) ;
      }
      else                 // help (info format)
      {
         //* Save the auxilliary dialogs *
         this->titleDlg->SetDialogObscured () ;
         this->statsDlg->SetDialogObscured () ;

         //* Invoke the infor reader *
         gString gs( "info -f \"%S\"", hPath.gstr() ) ;
         this->userDlg->ShellOut ( soX, gs.ustr() ) ;

         //* Restore the auxilliary dialogs *
         this->titleDlg->RefreshWin () ;
         this->statsDlg->RefreshWin () ;
      }
   }
   else                    // Error: help file not found
   {
      if ( htmlFormat )
         this->UserMessage ( NoHelp_html[this->cfg.lang], this->cfg.em, true ) ;
      else
         this->UserMessage ( NoHelp_info[this->cfg.lang], this->cfg.em, true ) ;
      this->userDlg->UserAlert ( 2 ) ;
      chrono::duration<short>aWhile( 3 ) ;
      this_thread::sleep_for( aWhile ) ;
      this->UserMessage ( "", this->cfg.bb ) ;
   }

}  //* End Cry4Help() *

//*************************
//*       HelpAbout       *
//*************************
//******************************************************************************
//* Display the application's Help-About window.                               *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note: This dialog is very simple for LTR languages, but       *
//* raises multiple issues for RTL languages because a large percentage of the *
//* data are both LTR and English.                                             *
//*                                                                            *
//******************************************************************************

void Exercalc::HelpAbout ( void )
{
   const short
         dlgROWS = 17,                 // dialog rows
         dlgCOLS = 70,                 // dialog columns
         dciROWS = dlgROWS - 4,        // sub-dialog rows
         dciCOLS = dlgCOLS - 2 ;       // sub-dialog columns
   short ctrY = udInit.dLines / 2 - 1, // center of application in Y
         ctrX = udInit.dColumns,       // center of application in X
         ulY = ctrY - dlgROWS / 2,     // dialog position in Y
         ulX = ctrX - dlgCOLS / 2,     // dialog position in X
         dciulY   = ulY + 3,           // sub-dialog position in Y
         dciulX   = ulX + 1 ;          // sub-dialog position in X
   attr_t dColor = this->cfg.ub,       // dialog interior color
          vColor = this->cfg.em ;      // color for version strings
   AppLang lang = this->cfg.lang ;     // language index

   enum ctrls : short { closePB = ZERO, suppPB, haCTRLS } ;

   //* Index the text for this dialog *
   enum haText : short { hatDesc = ZERO , hatCaption, hatAuthor, hatUni, 
                         hatDev1, hatDev2, hatLic1, hatLic2, hatContact, 
                         hatUrl, hatCOUNT } ;

   static const wchar_t* const DialogText[LangCount][hatCOUNT] = 
   {
      //** English **
      {
      L"Exercise-analysis application and demo program for\n"   // hatDesc
       "development of a multilingual user interface.",
      L" Exercalc: \n"                                          // hatCaption
       "           \n"
       "  Version: \n"
      L"Copyright: (c) %S %S\n",
      L"Mahlon R. Smith, The Software Samurai",                 // hatAuthor
      L"               Beijing University of Technology - Beijing, PRC\n" // hatUni
       "               马伦教授 北京工业大学 - 北京，中华人民共和国\n"
       "\n",
      L"Development: ",                                         // hatDev1
      L"Fedora Linux 32 -- GNU G++ (Gcc v: 10.2.1)",            // hatDev2
      L"Licensing  : ",                                         // hatLic1
      L"GNU General Public License, version 3,\n"               // hatLic2
       "GNU Free Documentation License, version 1.3",
      L"  For bugs or enhancement requests, please contact the author at:\n", // hatContact
      L"http://www.SoftwareSam.us/",                            // hatUrl
      },

      //** Español **
      {
      L"Aplicación de análisis de ejercicio físico y ejemplo de \n" // hatDesc
       "desarrollo para la interfaz de usuario multilingüe.\n",
      L" Exercalc: \n"                                          // hatCaption
       "           \n"
       "  Versión: \n"
       "Propiedad Artística: (c) %S %S\n",
      L"Mahlon R. Smith,\n"
       "                     The Software Samurai",             // hatAuthor
      L"                     Universidad Tecnológica de Beijing\n" // hatUni
       "\n",
      L"Desarrollo: ",                                          // hatDev1
      L"Fedora Linux 32 -- GNU G++ (Gcc v: 10.2.1)",            // hatDev2
      L"Licencia  : ",                                          // hatLic1
      L"GNU Licencia Publica General (GPL), versión 3,\n"       // hatLic2
       "GNU Licencia de documentación gratuita (FDL), versión1.3",
      L"Para errores o solicitudes de mejora, comuníquese con el autor en:\n",// hatContact
      L"http://www.SoftwareSam.us/",                            // hatUrl
      },

      //** Zhōngwén (中文) **
      {
      // Programmer's Note: There is a compiler error here.
      // For the first two lines, the first character of each line is lost, so
      // we insert a space (which is lost) to avoid losing the first actual 
      // display character. This is annoying, but we don't have time to chase it 
      // at the moment. This is true for both char* data and wchar_t* data.
      // Note however, that a second and subsequent rendering of the same data 
      // within a single session will recover the missing two-column character. 
      // Other lines and languages with single-column characters do not seem 
      // to be affected, i.e. pointer initialization trouble in compiler land. Hmmm...
      L" 体育锻炼分析应用程序和演示程序，\n"                           // hatDesc
       " 用于开发多语言用户界面。\n",
      L" Exercalc: \n"                                          // hatCaption
       "           \n"
       "     版本: \n"
       "     版权: (c) %S %S\n",
      L"Mahlon R. Smith, The Software Samurai",                  // hatAuthor
      L"               马伦教授 北京工业大学 - 北京，中华人民共和国\n"  // hatUni
       "\n",
      L"     发展：",                                             // hatDev1
      L"Fedora Linux 32 -- GNU G++ (Gcc v: 10.2.1)",             // hatDev2
      L"     许可: ",                                             // hatLic1
      L"根据GNU通用公共许可证发布的软件，版本3，和\n"                   // hatLic2
       "根据GNU自由文档许可证，版本1.3发布的文档",
      L"     对于软件错误，建议或增强要求,请联系作者：\n",              // hatContact
      L"http://www.SoftwareSam.us/",                            // hatUrl
      },

      //** TiếngViệt **
      {
      L"Ứng dụng phân tích bài tập và chương trình demo để phát\n" // hatDesc
       "triển giao diện người dùng đa ngôn ngữ.\n",
      L" Exercalc: \n"                                          // hatCaption
       "           \n"
       "Phiên Bản: \n"
       "Bản Quyền: (c) %S %S\n",
      L"Mahlon R. Smith, The Software Samurai",                  // hatAuthor
      L"               Đại học Công nghệ Bắc Kinh - Bắc Kinh, Trung Quốc\n" // hatUni
       "\n",
      L"Phát triển: ",                                          // hatDev1
      L"Fedora Linux 32 -- GNU G++ (Gcc v: 10.2.1)",            // hatDev2
      L"Cấp phép  : ",                                          // hatLic1
      L"GNU Giấy phép Công cộng Chung (GPL), phiên bản 3,\n"    // hatLic2
       "GNU Giấy phép Tài liệu Miễn phí (FDL), phiên bản 1.3",
      L"Đối với lỗi hoặc yêu cầu nâng cao, vui lòng liên hệ với tác giả tại:\n", // hatContact
      L"http://www.SoftwareSam.us/",                            // hatUrl
      },
   } ;

   //* X offset for display of application version number *
   short verOffset[LangCount] =
   {
      11,   //** English **
      11,   //** Español **
      10,   //** Zhōngwén (中文) **
      11,   //** TiếngViệt **
   } ;
   #if DEBUG_RTL != 0
   if ( this->cfg.rtl )
      verOffset[this->cfg.lang] = -(verOffset[this->cfg.lang]) ;
   #endif   // DEBUG_RTL

   //* Calculate text position *
   const short xBase = this->cfg.rtl ? (dlgCOLS - 3) : 1 ;
   winPos wpText( 1, xBase ),
          wpVers( 3, xBase + verOffset[this->cfg.lang] ) ;

   static const char* Labels[][3] = 
   {
      {  //** English **
         "   About Exercalc   ",       // dialog title
         "    CLOSE    ",              // 'close' pushbutton
         "   SUPPORT INFORMATION   ",  // 'support' pushbutton
      },

      {  //** Español **
         "    Sobre Exercalc   ",      // dialog title
         "   CIERRE    ",              // 'close' pushbutton
         " Información de Soporte  ",  // 'support' pushbutton
      },

      {  //** Zhōngwén (中文) **
         "    关于 Exercalc   ",        // dialog title
         "    关闭     ",               // 'close' pushbutton
         "      技术支持信息       ",    // 'support' pushbutton
         
      },

      {  //** TiếngViệt **
         "   Mọi Nơi Exercalc  ",       // dialog title
         " Đóng Cửa Sổ ",              // 'close' pushbutton
         "    Thông Tin Hỗ Trợ     ",  // 'support' pushbutton
      },

   } ;
   this->titleDlg->SetDialogObscured () ; // save parent dialogs
   this->userDlg->SetDialogObscured () ;
   this->statsDlg->SetDialogObscured () ;

   InitCtrl ic[haCTRLS] =        // array of dialog control info
   {
      {  //* 'CLOSE' pushbutton  - - - - - - - - - - - - - - - - - -   closePB *
         dctPUSHBUTTON,                // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         short(dlgROWS - 2),           // ulY:       upper left corner in Y
         short(dlgCOLS / 2 - 7),       // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         13,                           // cols:      control columns
         Labels[lang][1],              // dispText:  
         this->cfg.pn,                 // nColor:    non-focus color
         this->cfg.pf,                 // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "",                           // label:     (n/a)
         ZERO,                         // labY:      (n/a)
         ZERO,                         // labX       (n/a)
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[suppPB],                  // nextCtrl:  link in next structure
      },
      {  //* 'SUPPORT INFO' pushbutton   - - - - - - - - - - - - - - -  suppPB *
         dctPUSHBUTTON,                // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         ic[closePB].ulY,              // ulY:       upper left corner in Y
         short(ic[closePB].ulX + ic[closePB].cols + 2), // ulX: upper left corner in X
         1,                            // lines:     (n/a)
         25,                           // cols:      control columns
         Labels[lang][2],              // dispText:  
         this->cfg.pn,                 // nColor:    non-focus color
                                       // fColor:    focus color
         attr_t(this->cfg.scheme == ncbcGR ? nc.brR : nc.grR), 
         tbPrint,                      // filter:    (n/a)
         "",                           // label:     (n/a)
         ZERO,                         // labY:      (n/a)
         ZERO,                         // labX       (n/a)
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         NULL                          // nextCtrl:  link in next structure
      },
   } ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dlgROWS,        // number of display lines
                       dlgCOLS,        // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltDUAL,       // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* If user interface language is an RTL language, *
   //* set the internal NcDialog flag for RTL output. *
   if ( this->cfg.rtl )
      dp->DrawLabelsAsRTL () ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Set the dialog title *
      dp->SetDialogTitle ( Labels[lang][0], this->cfg.em ) ;

      //* Print dialog window's static text *
      // Programmer's Note: Copyright years, author and version number 
      // are always displayed as LTR.
      gString gsOut,             // text formatting
              gsYear( crYears ),
              gsAuth( DialogText[this->cfg.lang][hatAuthor] ),
              gsVers( " %s ", AppVersion ) ;
      if ( this->cfg.rtl )
      {
         gsYear.textReverse() ;
         gsAuth.textReverse() ;
         gsVers.textReverse() ;
      }
      dp->WriteString ( wpVers, gsVers, vColor, false, this->cfg.rtl ) ;
      dp->WriteParagraph ( wpText.ypos, wpText.xpos + verOffset[this->cfg.lang], 
                           DialogText[this->cfg.lang][hatDesc], 
                           dColor, false, this->cfg.rtl ) ;
      gsOut.compose( DialogText[this->cfg.lang][hatCaption],
                     gsYear.gstr(), gsAuth.gstr() ) ;
      wpText = dp->WriteParagraph ( wpText, gsOut, dColor, false, this->cfg.rtl ) ;
      gsOut = DialogText[this->cfg.lang][hatUni] ;
      wpText = dp->WriteParagraph ( wpText, DialogText[this->cfg.lang][hatUni], 
                                    dColor, false, this->cfg.rtl ) ;
      wpText.xpos = xBase ;
      wpText = dp->WriteParagraph ( wpText, DialogText[this->cfg.lang][hatDev1], 
                                    dColor, false, this->cfg.rtl ) ;
      gsOut = DialogText[this->cfg.lang][hatDev2] ;
      if ( this->cfg.rtl )
         gsOut.textReverse() ;
      gsOut.append( L'\n' ) ;
      wpText = dp->WriteParagraph ( wpText, gsOut, dColor, false, this->cfg.rtl ) ;
      wpText.xpos = xBase ;
      wpText = dp->WriteParagraph ( wpText, DialogText[this->cfg.lang][hatLic1], 
                                    dColor, false, this->cfg.rtl ) ;
      gsOut = DialogText[this->cfg.lang][hatLic2] ;
      if ( this->cfg.rtl )
         gsOut.textReverse( false, true, true ) ;
      gsOut.append( L"\n\n" ) ;
      wpText = dp->WriteParagraph ( wpText, gsOut, dColor, false, this->cfg.rtl ) ;
      wpText.xpos = xBase ;
      dp->WriteParagraph ( wpText, DialogText[this->cfg.lang][hatContact], 
                           dColor, false, this->cfg.rtl ) ;
      ++wpText.ypos ; wpText.xpos = 1 ;
      gsOut = DialogText[this->cfg.lang][hatUrl] ;
      gsOut.padCols( dlgCOLS - 2, L' ', true ) ;
      wpText = dp->WriteParagraph ( wpText, gsOut, dColor ) ;

      dp->RefreshWin () ;                    // make the text visible

      uiInfo Info ;                 // user interface data returned here
      bool   done = false ;         // loop control
      while ( ! done )
      {
         if ( Info.viaHotkey )
            Info.HotData2Primary () ;
         else
            dp->EditPushbutton ( Info ) ;
         if ( Info.dataMod != false )
         {
            if ( Info.ctrlIndex == closePB )
            {
#if 0    // TEMP - CAPTURE
dp->CaptureDialog ( "capturedlg.txt" ) ;
dp->CaptureDialog ( "capturedlg.html", true, false, 
                    "infodoc-styles.css", 4, false, nc.blR ) ;
#endif   // CAPTURE
               done = true ;
            }
            else if ( Info.ctrlIndex == suppPB )
            {  //* Open a dialog to display interesting data needed to submit *
               //* a tech support request.                                    *
               //* Current dialog will be obscured, so save its display data. *
               dp->SetDialogObscured () ;
               this->haSupportInfo ( dciulY, dciulX, dciROWS, dciCOLS ) ;
               dp->NextControl () ;
               dp->RefreshWin () ;
            }
         }
         if ( !done && !Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               dp->PrevControl () ; 
            else if ( Info.keyIn != ZERO )
               dp->NextControl () ;
         }
      }     // while()
   }        // OpenWindow()
   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

   this->titleDlg->RefreshWin () ;        // restore parent dialogs
   this->userDlg->RefreshWin () ;
   this->statsDlg->RefreshWin () ;

}  //* End HelpAbout() *

//*************************
//*     haSupportInfo     *
//*************************
//******************************************************************************
//* Display version numbers for NcDialog API, ncursesw library.                *
//* Also reports current 'locale' setting and dynamic load (shared) libraries. *
//*                                                                            *
//* Input  : ulY   : upper-left corner of dialog in Y                          *
//*          ulX   : upper-left corner of dialog in X                          *
//*          dRows : dialog rows                                               *
//*          dCols : dialog columns                                            *
//*                                                                            *
//* Returns: 'true' if support request saved, else 'false'                     *
//******************************************************************************
//* Notes: For technical support requests, ask user to save the information    *
//* to a file:                                                                 *
//*                                                                            *
//* 1) Header                                                                  *
//* 1) Exercalc version                                                        *
//* 2) NcDialog API version                                                    *
//* 3) ncursesw version                                                        *
//* 4) locale                                                                  *
//* 5) compiler version                                                        *
//* 6) DLLs                                                                    *
//* 7) system info: uname -a                                                   *
//* 8) Environment variables: printenv SHELL TERM COLORTERM LANG               *
//* 9) UDC time                                                                *
//*10) Prompt for description of issue                                         *
//*                                                                            *
//******************************************************************************

bool Exercalc::haSupportInfo ( short ulY, short ulX, short dRows, short dCols )
{
   static const char* Labels[][8] = 
   {
      {  //** English **
         "  Technical Support Information  ",      // dialog title
         "    CLOSE    ",                          // 'close' pushbutton
         "     SAVE TO FILE      ",                // 'save' pushbutton

         "    Please include this information\n"
         "    with all tech support requests.\n",

         "Compiler:\n",

         "Dynamic (shared) Libs:\n",

         "\nPlease describe the reason for this request."
         "\n (Please write in English, if possible.)"
         "\nInclude: a) the operation you were attempting to complete,"
         "\n            or the feature you would like to have added."
         "\n         b) what you believe should happen"
         "\n         c) what actually happened"
         "\n         d) any other information you think would be helpful",

         "Saved to: ",
      },

      {  //** Español **
         "  Información de Soporte Técnico  ",     // dialog title
         "   CIERRE    ",                          // 'close' pushbutton
         " Guardar En Un Archivo ",                // 'save' pushbutton

         "Por favor, incluya esta información con\n"
         "todas las solicitudes de soporte técnico.\n",

         "Compilador:\n",                          // compiler

         "Bibliotecas Dinámicas:\n",               // shared libraries

         "\nDescriba el motivo de esta solicitud."
         "\n (Por favor escriba en inglés, si es posible.)"
         "\nIncluya: a) la operación que estaba intentando completar,"
         "\n            o la característica que usted quisiera haber agregado."
         "\n         b) ¿Qué crees que debería suceder?"
         "\n         c) ¿Lo que realmente pasó?"
         "\n         d) Cualquier otra información que usted piensa que sería útil.",

         "Guardado en: ",                          // saved to:
      },

      {  //** Zhōngwén (中文) **
         "          技术支持信息          ",         // dialog title
         "    关闭     ",                           // 'close' pushbutton
         "       保存到文件      ",                  // 'save' pushbutton

         "    请将此信息与您的技术支\n"
         "    持请求一起提供。\n",

         "编译器：\n",                              // compiler

         "动态（共享）库:\n",                         // shared libraries

         "\n请说明此请求的原因。"
         "\n （如果可能，请用英文写。）"
         "\n包括： a) 您尝试完成的操作，或您要添加的功能。"
         "\n       b) 你认为应该发生什么？"
         "\n       c) 实际发生了什么？"
         "\n       d) 任何其他您认为有用的信息。",

         "存到： ",                                 // saved to:
      },

      {  //** TiếngViệt **
         "   Thông Tin Hỗ Trợ Kỹ Thuật   ",        // dialog title
         " Đóng Cửa Sổ ",                          // 'close' pushbutton
         "  Lưu Vào Một Tập Tin  ",                // 'save' pushbutton

         "Xin bao gồm thông tin này với tất\n"
         "cả sự hỗ trợ kỹ thuật được yêu cầu.\n",

         "Trình biên dịch:\n",                     // compiler

         "Năng Động Thư Viện:\n",                  // shared libraries

         "\nVui lòng mô tả lý do cho yêu cầu này."
         "\n (Vui lòng viết bằng tiếng Anh, nếu có thể.)"
         "\nBao gồm: a) hoạt động bạn đang cố gắng để hoàn thành,"
         "\n            hoặc các tính năng mới, bạn mong muốn."
         "\n         b) Bạn nghĩ gì nên xảy ra?"
         "\n         c) Điều gì thực sự đã xảy ra?"
         "\n         d) Thêm bất kỳ thông tin hữu ích khác.",

         "Đã lưu vào: ",                              // saved to:
      },

   } ;

   //* Technical Support form. See DisplaySupportInfo(). *
   const char* const tsFName = "Exercalc_SupportRequest.txt" ;

   AppLang lang = this->cfg.lang ;  // language index
   attr_t dColor = this->cfg.ub,    // dialog interior color
          vColor = this->cfg.sb,    // color for version strings
          hColor = this->cfg.er,    // highlight color
          tColor = this->cfg.em ;   // title color
   bool tsRequestSaved = false ;    // return value

   enum ctrls : short { closePB = ZERO, savePB, controlsDEFINED } ;

   InitCtrl ic[controlsDEFINED] =      // array of dialog control info
   {
   {  //* 'CLOSE' pushbutton  - - - - - - - - - - - - - - - - - - - -  closePB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dRows - 2),             // ulY:       upper left corner in Y
      short(dCols / 2 - 7),         // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      13,                           // cols:      control columns
      Labels[lang][1],              // dispText:  
      this->cfg.pn,                 // nColor:    non-focus color
      this->cfg.pf,                 // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[savePB]                   // nextCtrl:  link in next structure
   },
   {  //* 'SAVE' pushbutton   - - - - - - - - - - - - - - - - - - - -   savePB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[closePB].ulY,              // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      23,                           // cols:      control columns
      Labels[lang][2],              // dispText:  
      this->cfg.pn,                 // nColor:    non-focus color
      attr_t(this->cfg.scheme == ncbcGR ? nc.brR : nc.grR),
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
   } ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dRows,          // number of display lines
                       dCols,          // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltDUAL,       // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* If user interface language is an RTL language, *
   //* set the internal NcDialog flag for RTL output. *
   if ( this->cfg.rtl )
      dp->DrawLabelsAsRTL () ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Set the dialog title *
      dp->SetDialogTitle ( Labels[lang][0], tColor ) ;

      localTime lt ;                   // formatted timestamp
      this->GetLocalTime ( lt ) ;      // get the local time
      gString gsOut( "  Exercalc Tech Support Request\n"
                     "=================================\n"
                     "Exercalc v:%s\n"
                     "Requested: %04hd-%02hd-%02hdT%02hd:%02hd:%02hd\n", 
                     AppVersion, &lt.year, &lt.month, &lt.date,
                     &lt.hours, &lt.minutes, &lt.seconds ) ;
      gString gstmp ;

      //* Create a temporary log *
      gString gsLog ;
      this->CreateTempname ( gsLog ) ;
      ofstream ofs( gsLog.ustr(), ofstream::out | ofstream::trunc ) ;
      ofs << gsOut.ustr() ;

      winPos wp( 1, 2 ) ;
      gsOut = Labels[lang][3] ;
      if ( this->cfg.rtl )
      {
         winPos wptmp( wp.ypos, short(dCols / 2 + 4) ) ;
         #if DEBUG_RTL != 0
         gstmp = gsOut ;
         short offset = gsOut.scan() ;
         if ( offset > ZERO )
            gsOut.shiftChars( -(offset) ) ;
         if ( (offset = gsOut.after( L'\n' )) > ZERO )
         {
            while ( gsOut.gstr()[offset] == L' ' )
               offset = gsOut.erase( offset, 1 ) ;
         }
         #endif   // DEBUG_RTL
         dp->WriteParagraph ( wptmp, gsOut, hColor, false, this->cfg.rtl ) ;
         #if DEBUG_RTL != 0
         gsOut = gstmp ;
         #endif   // DEBUG_RTL
         wp.ypos += 2 ;
      }
      else
         wp = dp->WriteParagraph ( wp, gsOut, hColor ) ;
      ofs << '\n' << gsOut.ustr() << '\n' ;

      gsOut = "NcDialog API library : " ;
      wp = dp->WriteString ( wp, gsOut, dColor ) ;
      ofs << gsOut.ustr() ;
      gsOut.compose( " %s \n\n", dp->Get_NcDialog_Version () ) ;
      wp = dp->WriteParagraph ( wp, gsOut, vColor ) ;
      ofs << gsOut.ustr() ;

      gsOut = "    ncursesw library : " ;
      wp = dp->WriteString ( wp.ypos, 2, gsOut, dColor ) ;
      ofs << gsOut.ustr() ;
      gsOut.compose( L" %s \n\n", nc.Get_nclibrary_Version () ) ;
      wp = dp->WriteParagraph ( wp, gsOut, vColor ) ;
      ofs << gsOut.ustr() ;

      gsOut = "      Locale setting : " ;
      wp = dp->WriteString ( wp.ypos, 2, gsOut, dColor ) ;
      ofs << gsOut.ustr() ;
      gsOut.compose( L" %s \n", nc.GetLocale () ) ;
      wp = dp->WriteParagraph ( wp, gsOut, vColor ) ;
      ofs << gsOut.ustr() ;
      gsOut.compose( L" (%S)\n", alStrings[lang] ) ;
      wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
      ofs << "                       " << gsOut.ustr() << '\n' ;

      //* Display compiler version and a list of   *
      //* shared libraries used in the application.*
      gString gsTmp, gx ;
      this->CreateTempname ( gsTmp ) ;
      short libCount = ZERO ;
      wp = { 1, 43 } ;


      //* Capture the compiler version *
      // Programmer's Note: If user calls a different compiler 
      // i.e. "clang", this will be misleading.
      gsOut.compose( "g++ --version 1>\"%S\" 2>/dev/null", gsTmp.gstr() ) ;
      system ( gsOut.ustr() ) ;

      //* Append the list of libraries *
      gsOut.compose( "ldd \"%s/ecalc\" 1>>\"%S\" 2>/dev/null", 
                     this->cfg.appPath, gsTmp.gstr() ) ;
      system ( gsOut.ustr() ) ;
      gsOut.clear() ;
      ifstream ifs( gsTmp.ustr(), ifstream::in ) ;
      if ( ifs.is_open() )
      {
         char  lineData[gsMAXBYTES] ;
         short sIndex ;
         bool  done = false ;

         //* Read the line containing the compiler version *
         //* and discard the copyright message.            *
         wp = dp->WriteParagraph ( wp, Labels[lang][4], hColor ) ;
         ofs << Labels[lang][4] ;

         ifs.getline( lineData, gsMAXBYTES, NEWLINE ) ;
         if ( ifs.good() || (ifs.gcount() > ZERO) )
         {
            gsOut = lineData ;
            ofs << gsOut.ustr() << '\n' ;    // write full line to temp file
            if ( (gsOut.gscols()) > 24 )     // avoid writing into dialog border
               gsOut.limitChars( 24 ) ;
            gsOut.append( L'\n' ) ;
            wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

            for ( short i = 4 ; i > ZERO ; --i ) // copyright notice
               ifs.getline( lineData, gsMAXBYTES, NEWLINE ) ;
         }
         else        // unexpected end-of-file
         {
            ofs << "(unknown)\n" ;
            done = true ;
         }

         //* Report the list of load libraries *
         wp = dp->WriteParagraph ( wp, Labels[lang][5], hColor ) ;
         ofs << Labels[lang][5] ;
         gsOut.clear() ;
         while ( ! done )
         {
            ifs.getline( lineData, gsMAXBYTES, NEWLINE ) ;
            if ( ifs.good() || (ifs.gcount() > ZERO) )
            {
               ofs << &lineData[1] << '\n' ; // write all entries to log

               if ( libCount < 7 )
               {
                  gx = lineData ;
                  if ( ((sIndex = gx.find( L' ' )) > ZERO) &&
                       ((gx.find( "lib" )) == 1) ) // (line begins with "\tlib")
                  {
                     gx.limitChars( sIndex ) ;
                     gsOut.append( "%S\n", gx.gstr() ) ;
                     ++libCount ;
                  }
               }
            }
            else
               done = true ;
         }
         ifs.close() ;
      }
      if ( libCount == ZERO )
         gsOut = "(Unable to determine.)\n" ;
      wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
      this->DeleteTempname ( gsTmp ) ;

      //* Supplementary information for the log file *
      ofs << "\nSystem Information:" << endl ;
      ofs.close() ;     // temporarily close the log file

      gsOut.compose( "printenv SHELL TERM COLORTERM LANG 1>>\"%S\" 2>/dev/null",
                     gsLog.gstr() ) ;
      system ( gsOut.ustr() ) ;
      gsOut.compose( "uname -srvmpio 1>>\"%S\" 2>/dev/null", gsLog.gstr() ) ;
      system ( gsOut.ustr() ) ;

      ofs.open( gsLog.ustr(), ofstream::out | ofstream::app ) ;
      if ( ofs.is_open() )
      {
         ofs << "\n------------------------------------------------------------"
             << Labels[lang][6] << "\n\n\n" << endl ;
         ofs.close() ;              // close the log file
      }

      dp->RefreshWin () ;        // make everything visible

      uiInfo   Info ;            // user interface data returned here
      short    icIndex = ZERO ;  // index of control with input focus
      bool     done = false ;    // loop control
      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;

            if ( Info.dataMod != false )
            {
               if ( icIndex == closePB )
                  done = true ;
               else if ( icIndex == savePB )
               {
                  //* Do a simple 'move-with-rename' of the temp file *
                  //* to the specified source directory.              *
                  // Programmer's Note: The "mv" utility is able to cross 
                  // filesystem boundaries by doing a 
                  // copy-to-target/verify-copy/delete-source sequence.
                  char cmd[gsMAXBYTES * 2] ;
                  snprintf ( cmd, (gsMAXBYTES * 2), 
                             "mv \"%s\" \"./%s\" 1>/dev/null 2>/dev/null", 
                             gsLog.ustr(), tsFName ) ;
                  system ( cmd ) ;
                  tsRequestSaved = true ;

                  wp = { short(ic[savePB].ulY - 1), short(this->cfg.rtl ? 
                  (ic[closePB].ulX + ic[closePB].cols - 1) : ic[savePB].ulX) } ;
                  gstmp = tsFName ;
                  if ( this->cfg.rtl )
                     gstmp.textReverse() ;
                  gsOut.compose( "%s%s", Labels[lang][7], gstmp.ustr() ) ;
                  dp->WriteString ( wp, gsOut, hColor, true, this->cfg.rtl ) ;
               }
            }
         }

         //* Move focus to appropriate control *
         if ( ! done && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }        // while(!done)

      this->DeleteTempname ( gsLog ) ;       // delete temp log (if any)
   }     // window opened
   if ( dp != NULL )
      delete ( dp ) ;                        // close the window

   return tsRequestSaved ;

}  //* End haSupportInfo() *

//****************************
//* LaunchDefaultApplication *
//****************************
//******************************************************************************
//* Launch the system default application for handling the specified file type.*
//*                                                                            *
//* Input  : targFile : filespec of file to be processed by the external       *
//*                     application                                            *
//*          redirect : (optional, 'true' by default)                          *
//*                     if 'true',  redirect stdout and stderr to temp files   *
//*                     if 'false', allow stdout and stderr to write to the    *
//*                                 terminal window                            *
//*                                                                            *
//* Returns: a) the child process ID                                           *
//*          b) -1 if child process not created                                *
//******************************************************************************
//* Notes:                                                                     *
//* ------                                                                     *
//* 1) fork a new process: vfork()                                             *
//* 2) Redirection:                                                            *
//*    a) It is possible that the called application will write to the terminal*
//*       window, either during startup or with some later status message.     *
//*    b) If the called application is a console application, this is almost   *
//*       certain; while for external GUI applications, it is still likely to  *
//*       some extent.                                                         *
//*    c) If the parent application has relinquished control of stdout and     *
//*       stderr. i.e. the NCurses engine has been put into hibernation mode,  *
//*       then the called application can display messages without conflict.   *
//*    d) If, however, the parent application is still in active control of    *
//*       stdout and stderr, then resource conflicts are likely to occur.      *
//*       In this case, it is necessary to redirect the called application's   *
//*       output. Our solution is to redirect stdout and stderr to temporary   *
//*       files by default. These files will persist until the parent          *
//*       application closes. (To prevent this, call the method with           *
//*       'redirect' set to 'false'.) The Taggit-class destructor will delete  *
//*       these temporary files during application shutdown.                   *
//* 3) Use the new process to launch the application:                          *
//*    a) This method uses the 'xdg-open' system utility (shell script) which  *
//*       launches the application associated with the file type of 'targFile'.*
//*    b) execlp searches the path looking for 'xdg-open'                      *
//*       Because we are fairly safe in assuming that 'xdg-open' actually IS   *
//*       on the path, the first argument is the name of the script (no path). *
//*    c) The second argument is argv[0] i.e. the script name again.           *
//*    d) The third argument is argv[1] i.e. the filespec passed in by caller. *
//*    e) The fourth argument is argv[2] i.e. a null pointer                   *
//*    f) Note that xdg-open passes only the specified filespec to the target  *
//*       application, and cannot pass additional configuration arguments.     *
//*       To invoke the target application will additional setup/configuration *
//*       parameters, please see the LaunchExternalApplication() method, below.*
//*    g) The child process is not directly associated with the terminal       *
//*       window from which it was launched, so the window may be closed       *
//*       asynchronously.                                                      *
//*    h) The exec functions do not return. The child process terminates when  *
//*       the external application exits.                                      *
//* 4) primary process returns to caller                                       *
//*                                                                            *
//* Programmer's Note: This sequence assumes that multi-threading is not       *
//* active. For multi-threaded applications, use pthread_fork() instead.       *
//******************************************************************************

int Exercalc::LaunchDefaultApplication ( const char* targFile, bool redirect )
{
   gString stdoutTemp, stderrTemp ;       // stdout target file
   this->CreateTempname ( stdoutTemp ) ;  // stderr target file
   this->CreateTempname ( stderrTemp ) ;  // create the temp filenames
   //int soDesc = ZERO,                     // stdout target descriptor
   //    seDesc ;                           // stderr target descriptor

   pid_t fpid = vfork () ;                // create the child process

   if ( fpid == ZERO)
   {
      // Programmer's Note: This is a generalized launch method, and can do 
      // considerably more than this application requires. Because this 
      //* application calls this method ONLY for opening the HTML help file 
      // in a browser, there is no need to use redirect.
      #if 0    // DISABLE REDIRECT
      //* The child process (if created) executes here.                  *
      //* Create temp files as targets for redirecting stdout and stderr.*
      if ( redirect )
      {
         soDesc = open ( stdoutTemp.ustr(), O_WRONLY | O_CREAT | O_TRUNC, 
                           S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH ) ;
         seDesc = open ( stderrTemp.ustr(), O_WRONLY | O_CREAT | O_TRUNC, 
                           S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH ) ;
         dup2 ( soDesc, STDOUT_FILENO) ;
         dup2 ( seDesc, STDERR_FILENO) ;
         close ( soDesc ) ;
         close ( seDesc ) ;
      }
      #endif   // DISABLE REDIRECT

      execlp ( "xdg-open", "xdg-open", targFile, NULL ) ;

      //* In case the exec call fails: child process MUST NOT return.*
      _exit ( 0 ) ;
   }
   else
   {  //* The parent process continues execution here *
      if ( fpid > ZERO )      // child successfully launched
      {
         /* Currently, nothing to be done. */
      }
   }
   return fpid ;

}  //* End LaunchDefaultApplication() *

//*************************
//*        Confirm        *
//*************************
//******************************************************************************
//* Generalized confirmation dialog.                                           *
//* ================================                                           *
//* -- The supported dialog descriptions are controlled by the "confType"      *
//*    enumerated type.                                                        *
//* -- For simplicity, the functionality of this method, INCLUDING the text    *
//*    translations is as much as possible encapsulated here.                  *
//* -- The number of text lines (including title) for the dialog is initially  *
//*    limited to six(6). This may be expanded as necessary by adding more     *
//*    elements to 'msgList' and 'msgAttr'.                                    *
//* -- The width of the dialog is dynamically set based on the widest display  *
//*    text line defined for the case.                                         *
//*                                                                            *
//* Input  : cfType : member of enum confType indicating the confirmation type *
//*                                                                            *
//* Returns: 'true' if user confirms the operation, else 'false'               *
//*            For information-only dialogs (no decision), the return value    *
//*            should be ignored.                                              *
//******************************************************************************

bool Exercalc::Confirm ( confType cfType )
{
   gString gs1, gs2, gs3, gs4, gs5, gs6, gstmp ;
   const char* msgList[6] = 
   {
      gs1.ustr(),
      gs2.ustr(),
      gs3.ustr(),
      gs4.ustr(),
      gs5.ustr(),
      gs6.ustr(),
   } ;
   attr_t msgAttr[6] = 
   {
      this->cfg.ub,     // title
      this->cfg.ub,
      this->cfg.ub,
      this->cfg.ub,
      this->cfg.ub,
      this->cfg.ub,
   } ;
   bool confirm = false ;


   //********************************************
   //** Confirm truncation of logfile contents **
   //********************************************
   if ( cfType == confirmTruncate )
   {
      const wchar_t* const cftText[LangCount][4] = 
      {
         //** English **
         {
          L"  Confirm Truncate of Log File  ",
          L"Are you sure you want to delete the",
          L"contents of the log file:",
          L"This operation cannot be reversed.",
         },
         //** Español **
         {
          L"  Confirmar Truncar el Archivo de Registro  ",
          L"¿Estás seguro de que quieres eliminar?",
          L"el contenido del archivo de registro?",
          L"Esta operación no se puede revertir.",
         },
         //** Zhōngwén (中文) **
         {
          L"  确认截断日志文件  ",
          L"你确定你要删除",
          L"日志文件的内容:",
          L"此操作不能撤消。        ",
         },
         //** TiếngViệt **
         {
          L"  Xác nhận cắt ngắn tệp nhật ký  ",
          L"Bạn có chắc chắn muốn xóa",
          L"nội dung của tệp nhật ký:",
          L"Không thể hoàn tác thao tác này.",
         },
      } ;
      gs1 = cftText[this->cfg.lang][0] ;  // set the constant strings
      gs2 = cftText[this->cfg.lang][1] ;
      gs3 = cftText[this->cfg.lang][2] ;
      gs5 = cftText[this->cfg.lang][3] ;
      gstmp = this->cfg.logPath ;         // extract and format the filename
      short fn = gstmp.findlast( L'/' ) ;
      if ( fn >= ZERO )
         gstmp.shiftChars( -(fn + 1) ) ;
      gs4.compose( "%S", gstmp.gstr() ) ;
      msgAttr[3] = this->cfg.em ;         // color attribute for filename

      short cols = gs1.gscols() ;         // set width of dialog
      if ( gs2.gscols() > cols )   cols = gs2.gscols() ;
      if ( gs3.gscols() > cols )   cols = gs3.gscols() ;
      if ( gs4.gscols() > cols )   cols = gs4.gscols() ;
      if ( gs5.gscols() > cols )   cols = gs5.gscols() ;
      if ( gs6.gscols() > cols )   cols = gs6.gscols() ;
      gs4.shiftCols( (cols - (gs4.gscols())) / 2 ) ;  // center the filename
      gs4.padCols( cols ) ;
      //* For RTL languages, filename is still displayed as LTR text *
      if ( this->cfg.rtl )    
         gs4.textReverse() ;
      cols += 4 ;

      genDialog gd( msgList, msgAttr[1], 8, cols, -1, -1, 
                    msgAttr, this->cfg.rtl, this->cfg.pn, this->cfg.pf,
                    Yes[this->cfg.lang], No[this->cfg.lang] ) ;

      //* Ask the user to confirm *
      confirm = this->userDlg->DecisionDialog ( gd ) ;
   }

   return confirm ;

}  //* End Confirm() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

