//******************************************************************************
//* File       : Exercalc.cpp                                                  *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2020-2021 Mahlon R. Smith, The Software Samurai *
//*                  GNU GPL copyright notice located below.                   *
//* Date       : 01-Jul-2021                                                   *
//* Version    : (see AppVersion string below)                                 *
//*                                                                            *
//* Description:                                                               *
//* This is a simple calculator for weekly exercise. It provides calculations  *
//* for various data related to physical exercise.                             *
//*                                                                            *
//* This application is structured as three(3) dialog windows, although only   *
//* one of the dialogs accepts user input. The same functionality could have   *
//* been achieved with a single dialog window, but we experimentally determined*
//* complexity of the application is reduced by having single-purpose dialogs. *
//*                                                                            *
//* The application is built specifically for a multi-language user interface, *
//* so adding a new user-interface language (or correcting the text of our     *
//* sub-standard command of languages) should be a simple task.                *
//*                                                                            *
//*     This application was written out of boredom caused by the              *
//*     multiple corona-virus quarentines we have endured both here            *
//*     in China and in the U.S. The author doesn't much enjoy being           *
//*     around people, but THIS level of isolation is just ridiculous!         *
//*   ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---     *
//*                                                                            *
//* Disclaimer:                                                                *
//* -----------                                                                *
//*    *** This program is provided for informational purposes only. ***       *
//* The author of this application is a software designer, teacher, musician   *
//* and lover. As such, he has no expertise in physical training, fitness or   *
//* any medical discipline. He does however regularly track workout activity   *
//* (i.e. nerd) and has found this application to be useful in daily life.     *
//*                                                                            *
//*                                                                            *
//* Configuration and command-line processing adapted from the author's        *
//* "Taggit" application.                                                      *
//*                                                                            *
//* The data tables used in the configuration file are derived from the        *
//* University of South Carolina's Arnold School of Public Health,             *
//* "Compendium Of Physical Activities" (2011) downloaded from                 *
//*     http://prevention.sph.sc.edu/tools/docs/documents_compendium.pdf       *
//* Translations of this document and additional background information,       *
//* including notes for applying corrections to the standard METs values may   *
//* be found at:                                                               *
//*     https://sites.google.com/site/compendiumofphysicalactivities/home      *
//*                                                                            *
//* Wayland clipboard access is through the "wl-clipboard" utilities by        *
//* Sergey Bugaev, et al (if installed.) See NcDialog API documentation        *
//* for details of clipboard access.                                           *
//*                                                                            *
//*                                                                            *
//* Developed Using:                                                           *
//*   Fedora Linux 33, GNOME terminal 3.38.1.1                                 *
//*   Wintel (Lenovo) hardware.                                                *
//*   GNU G++ (Gcc v:10.2.1-9)                                                 *
//*   ncurses v:6.2.20200222                                                   *
//*                                                                            *
//******************************************************************************
//* Copyright Notice:                                                          *
//* This program is free software: you can redistribute it and/or modify it    *
//* under the terms of the GNU General Public License as published by the Free *
//* Software Foundation, either version 3 of the License, or (at your option)  *
//* any later version.                                                         *
//*                                                                            *
//* This program is distributed in the hope that it will be useful, but        *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
//* for more details.                                                          *
//*                                                                            *
//* You should have received a copy of the GNU General Public License along    *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.            *
//*                                                                            *
//*         Full text of the GPL License may be found in the Texinfo           *
//*         documentation for this program under 'Copyright Notice'.           *
//******************************************************************************
//* Version History (most recent first):                                       *
//*                                                                            *
//* v: 0.0.01 29-Dec-2020                                                      *
//*   -- First effort.                                                         *
//*      Track daily exercise and analyze progress over time.                  *
//*      -- First draft of documentation.                                      *
//*      -- Experimental bar-chart widget.                                     *
//*   -- First Release 01-Jul-2021                                             *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//******************************************************************************
//* NOTES:                                                                     *
//* ======                                                                     *
//*                                                                            *
//* Definitions:                                                               *
//* -- A MET is a ratio of your working metabolic rate relative to your        *
//*    resting metabolic rate.                                                 *
//*    One MET is approximately 3.5 milliliters of oxygen consumed per         *
//*    kilogram (kg) of body mass per minute while the person is sitting       *
//*    quietly.                                                                *
//*    The body combines the oxygen with energy-producing material (usually    *
//*    a form of sugar). This combination of sugar with oxygen produces energy *
//*    which is used to power the various bodily functions.                    *
//*                                                                            *
//* -- 1 MET = 1.0 minute x 1.0 kilograms x 3.5 milliliters of oxygen.         *
//*    Another way to calculate a MET is:                                      *
//*                        kilocalorie                                         *
//*                1.0 x -----------------                                     *
//*                       kilogram x hour                                      *
//*    These calculations must be considered as an approximation because the   *
//*    person's BMI (Body Mass Index), the ratio of body fat to lean           *
//*    body mass, has a significant influence on the actual value for an       *
//*    individual.                                                             *
//*                                                                            *
//* -- The METs value specified for a given activity is the ratio of energy    *
//*    expended for that activity as compared with the energy expended while   *
//*    the body is at rest, i.e. one MET, for the same period of time.         *
//*                                                                            *
//* -- One MET minute is energy expended per kilogram of mass in one minute    *
//*    while the body is at rest.                                              *
//*                                                                            *
//* -- MET-minutes, then, is the energy expended during the exercise period.   *
//*    Example: Walking briskly at 5.6 km/hour (3.5 mph) requires approximately*
//*    3.3 times as much energy as sitting quietly. Therefore one minute of    *
//*    walking equals 3.3 MET-minutes. Walking for 60 minutes would then be    *
//*    3.3 x 60.0 = 198.0 MET-minutes.                                         *
//*                                                                            *
//* -- Kilocalories burned ('Calories') is a common way to express the         *
//*    efficacy of exercise. Though this is mostly a bogus measurement for     *
//*    weight loss, we do calculate and report that value here.                *
//*                                                                            *
//*    The reason that kCal burned is an unreliable and unrealistic measure is *
//*    that although this program reports the the mass equivalent to the       *
//*    number of fat grams burned, there is no guarantee that the mass burned  *
//*    was actually fat. In fact, until the blood-glucose level drops to a     *
//*    certain minimum threshold, it is likely that little or no fat will      *
//*    actually be converted to energy.                                        *
//*    After exercise, it would be wise to ignore the body's request for food, *
//*    and satisfy the craving by allowing the body to convert fat to glucose  *
//*    which will replentish what was lost during exercise.                    *
//*                                                                            *
//*                                                                            *
//******************************************************************************

//****************
//* Header Files *
//****************
#include "Exercalc.hpp"    //* Class definition


//***************
//* Definitions *
//***************

//***************
//* Prototypes  *
//***************
static bool fltComp ( float fA, float fB, const char *fmt ) ;

//**********
//*  Data  *
//**********

//************************************************
//** Definitions for application dialog windows **
//************************************************
InitNcDialog tdInit     // Title Dialog
      (
       3,               // number of display lines
       MIN_COLS,        // number of display columns
       ZERO,            // Y offset from upper-left of terminal 
       ZERO,            // X offset from upper-left of terminal 
       NULL,            // dialog title
       ncltSINGLE,      // border line-style
       nc.bw,           // border color attribute
       nc.bw,           // interior color attribute
       NULL             // pointer to list of control definitions
      ) ;
InitNcDialog sdInit     // Statistics Dialog
      (
       MIN_ROWS - 3,    // number of display lines
       MIN_COLS / 2 - 1,// number of display columns
       3,               // Y offset from upper-left of terminal 
       MIN_COLS / 2 + 1,// X offset from upper-left of terminal 
       NULL,            // dialog title
       ncltSINGLE,      // border line-style
       nc.bw,           // border color attribute
       nc.bw,           // interior color attribute
       NULL             // pointer to list of control definitions
      ) ;
InitNcDialog udInit     // User-interface Dialog
      (
       MIN_ROWS - 3,    // number of display lines
       MIN_COLS / 2,    // number of display columns
       3,               // Y offset from upper-left of terminal 
       ZERO,            // X offset from upper-left of terminal 
       NULL,            // dialog title
       ncltSINGLE,      // border line-style
       nc.bw,           // border color attribute
       nc.bw,           // interior color attribute
       NULL             // pointer to list of control definitions
      ) ;

//*********************************************************
//** Dialog control definitions for User-interface dialog *
//*********************************************************
InitCtrl ic[CtrlCount] = 
{
   {  //* 'CLOSE' pushbutton  - - - - - - - - - - - - - - - - - - - - -  cloPB *
      dctPUSHBUTTON,                // type:      control type
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(USER_ROWS + 3),         // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      1,                            // lines:     control lines
      23,                           // cols:      control columns
      UserDlgLabels[ZERO][cloPB],   // dispText:  display text
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[valTB],                   // nextCtrl:  link in next structure
   },

   {  //* 'VALUE' Textbox  - - - - - - - - - - - - - - - - - - - - - -   valTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[cloPB].ulY,                // ulY:       upper left corner in Y
      short(ic[cloPB].ulX + ic[cloPB].cols + 2), // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      16,                           // cols:      control columns
      NULL,                         // dispText:  
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    all printing characters
      NULL,                         // label:     
      ZERO,                         // labY:      
      16,                           // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    allow control to gain focus
      &ic[selDD]                    // nextCtrl:  link in next structure
   }, 

   { //* 'SELECT' Dropdown  - - - - - - - - - - - - - - - - - - - - - -  selDD *
      dctDROPDOWN,                  // type:      define a drop-down control
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[valTB].ulY + 1),     // ulY:       upper left corner in Y
      ic[valTB].ulX,                // ulX:       upper left corner in X
      short(spITEMS + 2),           // lines:     control lines
      short(spWIDTH + 1),           // cols:      control columns
      NULL,                         // dispText:  text-data array
      nc.bw,                        // nColor:    non-focus border color
      nc.bw,                        // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     control label
      3,                            // labY:      label offset
      1,                            // labX       
      ddBoxUP,                      // exType:    expansion type
      spITEMS,                      // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    index of initial highlighted element
      NULL,                         // scrColor:  color-attribute list
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[savPB],                   // nextCtrl:  link in next structure
   },

   {  //* 'SAVE' pushbutton   - - - - - - - - - - - - - - - - - - - -    savPB *
      dctPUSHBUTTON,                // type:      control type
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[cloPB].ulY + 2),     // ulY:       upper left corner in Y
      ic[cloPB].ulX,                // ulX:       upper left corner in X
      1,                            // lines:     control lines
      ic[cloPB].cols,               // cols:      control columns
      UserDlgLabels[ZERO][savPB],   // dispText:  display text
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[optMW]                    // nextCtrl:  link in next structure
   },

   {  //* 'OPERATIONS' Menuwin   - - - - - - - - - - - - - - - - - - -   optMW *
      dctMENUWIN,                   // type:      define a menu-data control
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[savPB].ulY + 2),     // ulY:       upper left corner in Y
      ic[savPB].ulX,                // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      short(optCOLS + 2),           // cols:      control columns
      NULL,                         // dispText:  text-data array
      nc.bw,                        // nColor:    non-focus border color
      nc.bw,                        // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     label text
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      optITEMS,                     // scrItems:  elements in text/color arrays
      ZERO,                         // scrSel:    (n/a)
      NULL,                         // scrColor:  color-attribute list
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[hlpMW]                    // nextCtrl:  link in next structure
   },

   {  //* 'HELP' sub-Menuwin    - - - - - - - - - - - - - - - - - - - -  hlpMW *
      dctMENUWIN,             // type:      
      rbtTYPES,               // rbSubtype: (n/a)
      false,                  // rbSelect:  (n/a)
      short(ic[optMW].ulY + 10), // ulY: upper left corner in Y
      short(ic[optMW].ulX + ic[optMW].cols - 1), // ulX: upper left corner in X
      1,                      // lines:     (n/a)
      short(hlpCOLS + 2),     // cols:      control columns, 'expanded' state
      NULL,                   // dispText:  array of menu items
      nc.bw,                  // nColor: (n/a) context menus are invisible when collapsed
      nc.bw,                  // fColor: focus border color / non-focus 'collapsed' color
      tbPrint,                // filter:    (n/a)
      NULL,                   // label:     hidden menus have no label
      ZERO,                   // labY:      (n/a)
      ZERO,                   // labX       (n/a)
      ddBoxTYPES,             // exType:    (n/a)
      hlpITEMS,               // scrItems:  number of menu items
      ZERO,                   // scrSel:    (n/a)
      NULL,                   // scrColor:  array of color attributes for menu items
      NULL,                   // spinData:  (n/a)
      false,                  // active:    initially inactive
      #if DEBUG_MENU != 0
      &ic[dbgMW]              // nextCtrl:  link in next structure
      #else    // PRODUCTION
      NULL                    // nextCtrl:  link in next structure
      #endif   // DEBUG_MENU
   },

   #if DEBUG_MENU != 0
   {  //* 'Debug' sub-Menuwin    - - - - - - - - - - - - - - - - - - - - dbgMW *
      dctMENUWIN,             // type:      
      rbtTYPES,               // rbSubtype: (n/a)
      false,                  // rbSelect:  (n/a)
      short(ic[optMW].ulY + 5), // ulY: upper left corner in Y
      short(ic[optMW].ulX + ic[optMW].cols - 4), // ulX: upper left corner in X
      1,                      // lines:     (n/a)
      short(dbgCOLS + 2),     // cols:      control columns, 'expanded' state
      NULL,                   // dispText:  array of menu items
      nc.bw,                  // nColor: (n/a) context menus are invisible when collapsed
      nc.bw,                  // fColor: focus border color / non-focus 'collapsed' color
      tbPrint,                // filter:    (n/a)
      NULL,                   // label:     hidden menus have no label
      ZERO,                   // labY:      (n/a)
      ZERO,                   // labX       (n/a)
      ddBoxTYPES,             // exType:    (n/a)
      dbgITEMS,               // scrItems:  number of menu items
      ZERO,                   // scrSel:    (n/a)
      NULL,                   // scrColor:  array of color attributes for menu items
      NULL,                   // spinData:  (n/a)
      false,                  // active:    initially inactive
      NULL                    // nextCtrl:  link in next structure
   },
   #endif   // DEBUG_MENU
} ;


//*************************
//*         main          *
//*************************
//******************************************************************************
//* Program entry point.                                                       *
//*                                                                            *
//* Command-line Usage:  GetCommandLineArgs()                                  *
//*                                                                            *
//*                                                                            *
//******************************************************************************

int main ( int argc, char* argv[], char* argenv[] )
{
   //* User may have specified one or more command-line arguments.           *
   commArgs clArgs( argc, argv, argenv ) ;

   //* Create the application class object and interact with the user.       *
   Exercalc* ecptr = new Exercalc( clArgs ) ;

   //* Before returning to the system, delete the Exercalc object.   *
   //* While the class object and all its resources will be released *
   //* by the system on exit, we want to force execution of the      *
   //* destructor for an orderly termination.                        *
   delete ecptr ;

   exit ( ZERO ) ;

}  //* End main() *

//*************************
//*       ~Exercalc       *
//*************************
//******************************************************************************
//* Destructor. Return all resources to the system.                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

Exercalc::~Exercalc ( void )
{
   //* If the dynamic memory allocation has not been released. *
   if ( this->udAlloc != NULL )
   {
      delete[] this->udAlloc ;
      this->udAlloc = NULL ;
   }
   if ( this->udCap != NULL )
   {
      delete[] this->udCap ;
      this->udCap = NULL ;
   }

}  //* End ~Exercalc() *

//*************************
//*       Exercalc        *
//*************************
//******************************************************************************
//* Constructor.                                                               *
//*                                                                            *
//*                                                                            *
//* Input  : commArgs class object (by reference)                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

Exercalc::Exercalc ( commArgs& clArgs )
{
   //* Initialize our data members (ConfigOptions class is self-initializing) *
   this->titleDlg = this->statsDlg = this->userDlg = NULL ;
   this->userArea = { 0, 0 } ; // (initialized when dialog is instantiated)
   this->userRows = ZERO ;     // (initialized when dialog is instantiated)
   this->userCols = ZERO ;     // (initialized when dialog is instantiated)
   this->userMsg  = ZERO ;     // information message written here
   this->pictArea = { 0, 0 } ; // (initialized when dialog is instantiated)
   this->pictRows = ZERO ;     // (initialized when dialog is instantiated)
   this->pictCols = ZERO ;     // (initialized when dialog is instantiated)
   this->udAlloc  = NULL ;     // (dynamic memory allocation attached here)
   this->udCap    = NULL ;     // (dynamic memory allocation attached here)
   this->udcCount = ZERO ;     // no records captured
   this->gValue   = 30 ;       // default goal: walk 30 min/day at 3.3 METs intensity
   this->gUnits   = vuMinutes ;
   this->gXType   = xtWalk ;
   this->gMets    = 3.3 ;
   this->gWeek    = false ;
   this->gDisp    = false ;    // 'goal' not currently displayed
   this->wcbActive = false ;   // (initialized when dialog is instantiated)
   this->logPend   = false ;   // (no write-to-logfile pending)
   this->suPos = { 0, 0 } ;    // start-up message position
   //* Start-up message color attributes will be  *
   //* initialized after NCurses Engine is active.*
   this->suNorm = this->suBold = this->suGood = this->suError = attr_t(ZERO) ;

   //*********************************
   //* See what the user has to say. *
   //* Save the application path,    *
   //* and config-file path (if any).*
   //*********************************
   this->GetCommandLineArgs ( clArgs ) ;
   gString gsOut ;
   gsOut = clArgs.appPath ;
   gsOut.copy( this->cfg.appPath, MAX_PATH ) ;
   gsOut = clArgs.cfgPath ;
   gsOut.copy( this->cfg.cfgPath, MAX_PATH ) ;

   //***************************************
   //***  Initialize the NCurses Engine  ***
   //***************************************
   if( (!(clArgs.helpFlag || clArgs.verFlag)) && 
       ((nc.StartNCursesEngine ()) == OK) )
   {
      //* Get the size of our playground *
      nc.ScreenDimensions ( this->cfg.termRows, this->cfg.termCols ) ;

      //* Locate the system's temp-file directory *
      bool goodTempdir = false ;
      if ( (goodTempdir = this->GetTempdirPath ( gsOut )) )
         gsOut.copy( this->cfg.tmpPath, gsMAXBYTES ) ;

      //* Read the configuration file *
      CfgMsg *cfgMsg = new CfgMsg[CFGMSG_COUNT] ;
      short cfgMsgs = ZERO,
            syntaxErr = this->Configure ( cfgMsg, cfgMsgs, bool(clArgs.diagPause > 1) ) ;

      //* Verify that the current locale supports UTF-8 character encoding.    *
      gString envLocale( nc.GetLocale() ) ;  // name of locale from environment
      short localeSetOk = nc.VerifyLocale ();// env locale supports UTF-8?
      //* If alternate locale specified *
      bool altLocale = false ;
      if ( clArgs.altLocale[0] != NULLCHAR )
      {
         if ( (nc.SetLocale ( clArgs.altLocale )) == OK )
         { envLocale = clArgs.altLocale ; localeSetOk = OK ; altLocale = true ; }
      }
      else if ( this->cfg.altLocale[0] != NULLCHAR )
      {
         if ( (nc.SetLocale ( this->cfg.altLocale )) == OK )
         { envLocale = this->cfg.altLocale ; localeSetOk = OK ; altLocale = true ; }
      }

      //* If language specified as a command-line option *
      if ( clArgs.appLanguage != locLang )
      {
         this->cfg.lang = clArgs.appLanguage ;
         if ( AppDirection[this->cfg.lang] != false ) // test for RTL language
            this->cfg.rtl = true ;
      }

      //* If language derived from locale, initialize language index.*
      if ( this->cfg.lang == locLang )
      {
         if ( (envLocale.find( "es_" )) == ZERO )      this->cfg.lang = esLang ;
         else if ( (envLocale.find( "zh_" )) == ZERO ) this->cfg.lang = zhLang ;
         else if ( (envLocale.find( "vi_" )) == ZERO ) this->cfg.lang = viLang ;
         else                                          this->cfg.lang = enLang ;
      }

      nc.SetCursorState ( nccvINVISIBLE ) ;    // hide the cursor
      nc.SetKeyProcState ( nckpRAW ) ;         // allow CTRL keys through unprocessed
      gsOut.compose( titleTemplate, AppTitle, AppVersion, crYears ) ;
      this->DiagMsg ( gsOut.ustr(), nc.bwU ) ; // draw app title line to console

      this->DiagMsg ( "NCurses engine initialized.", nc.bw ) ;

      //* Determine whether the hardware supports display of color text. *
      if ( nc.ColorText_Available () )
      {
         //* Color Engine successfully initialized? *
         if ( nc.ColorText_Initialized () )
         {
            //* Set the color scheme according to: *
            //* 1) Command-line option             *
            //* 2) Config file option              *
            //* 3) Default setting                 *
            this->SetColorScheme ( 
                  ((clArgs.scheme != dfltCS) ? clArgs.scheme :
                  (this->cfg.scheme != dfltCS) ? this->cfg.scheme : dfltCS),
                  (clArgs.rscheme ? true : this->cfg.rscheme ? true : false ) ) ;

            termColorInfo tci ;
            nc.TerminalColorSupport ( tci ) ;
            gsOut.compose( L"Color Engine started. (%hd RGB Registers and "
                            "%hd fgnd/bkgnd Color Pairs)", 
                           &tci.rgb_regs, &tci.fgbg_pairs ) ;
            this->DiagMsg ( gsOut.ustr(), this->suGood ) ;
         }
         else
         {
            this->DiagMsg ( "Unable to start Color Engine.", nc.bw ) ;
         }
      }
      else
      {
         this->DiagMsg ( "Terminal does not support color output.", nc.bw ) ;
         this->DiagMsg ( " Starting in monochrome mode.", nc.bw ) ;
      }

      //* Report the results of locale settings for character-encoding *
      if ( altLocale )
         this->DiagMsg ( "Specified alternate locale: ", this->suGood, false ) ;
      else
         this->DiagMsg ( "Locale from environment: ", this->suGood, false ) ;
      gsOut.compose( L"\"%S\"", envLocale.gstr() ) ;
      this->DiagMsg ( gsOut.ustr(), this->suNorm, false ) ;
      if ( localeSetOk == OK )
         this->DiagMsg ( "  UTF-8 encoding support verified.", this->suGood ) ;
      else
         this->DiagMsg ( "  may not support UTF-8 encoding.", this->suError ) ;

      //* Report terminal-window dimensions *
      gsOut.compose( "Terminal window dimensions: (%hd x %hd)",
                     &this->cfg.termRows, &this->cfg.termCols ) ;
      this->DiagMsg ( gsOut.ustr(), this->suGood ) ;

      //* Report results of configuration.             *
      //* If error(s), give user time to read messages.*
      for ( short mi = ZERO ; mi < cfgMsgs ; ++mi )
         this->DiagMsg ( cfgMsg[mi].msg, *cfgMsg[mi].attr, cfgMsg[mi].nl ) ;
      if ( (syntaxErr != ZERO) && (clArgs.diagPause == ZERO) )
         clArgs.diagPause = 1 ;
      delete [] cfgMsg ;      // release the dynamic allocation
      cfgMsg = NULL ;

      //* Complain if terminal window is too small
      if ( (this->cfg.termRows < MIN_ROWS) || (this->cfg.termCols < MIN_COLS) )
      {
         this->TermsizeError () ;
         clArgs.diagPause = 3 ;  // force a pause for keystroke
      }

      //* Complain if temp-file directory not found (unlikely) *
      if ( ! goodTempdir )
      {
         this->TempdirError () ;
         clArgs.diagPause = 3 ;  // force a pause for keystroke
      }

      //* If no input record was specified on the command *
      //* line, signal display of 'Goal' record instead.  *
      if ( (clArgs.inpValue == 0.0) && (clArgs.showGoal == sgNoshow) )
         clArgs.showGoal = this->gWeek ? sgWeekly : sgDaily ;

      //* If specified, pause so user can read diagnostics.*
      if ( clArgs.diagPause > ZERO )
      {
         if ( clArgs.diagPause == 1 )
         {
            chrono::duration<short>aWhile( 4 ) ;
            this_thread::sleep_for( aWhile ) ;
         }
         else
         {
            if ( clArgs.diagPause == 2 )
            {
               this->DiagMsg ( " * Press Any Key to Continue *", this->suBold ) ;
               nckPause();
            }
            else
            {
               this->DiagMsg ( " * Press Any Key to Exit *", this->suBold ) ;
               //* Wait for a keypress.                      *
               //* Special Case: For terminal-resize events, *
               //* report new terminal dimensions.           *
               this->suPos.ypos -= 2 ;          // reference the message line
               this->suPos.xpos = 2 ;
               wkeyCode wk ;                    // key input
               while ( true )
               {
                  nc.GetKeyInput ( wk ) ;       // get the key input

                  //* If a terminal-resize event *
                  if ( (wk.type == wktFUNKEY) && (wk.key == nckRESIZE) )
                  {
                     nc.ScreenDimensions ( this->cfg.termRows, this->cfg.termCols ) ;
                     gsOut.compose( "Terminal window dimensions: (%hd x %hd)",
                     &this->cfg.termRows, &this->cfg.termCols ) ;
                     nc.ClearLine ( this->suPos.ypos ) ;
                     nc.WriteString ( this->suPos, gsOut.ustr(), 
                                      (((this->cfg.termRows < MIN_ROWS) ||
                                       (this->cfg.termCols < MIN_COLS)) ?
                                       this->suError : this->suGood) ) ;
                  }
                  else
                     break ;
               }
            }
         }
      }

      if ( clArgs.diagPause <= 2 )
      {
         //***************************
         //* Interact with the user. *
         //* (Watch out! They bite!) *
         //***************************
         this->UserInterface ( clArgs ) ;
      }

      nc.RestoreCursorState () ;             // make cursor visible
      nc.StopNCursesEngine () ;              // Deactivate the NCurses engine

   }
   else
   {
      if ( clArgs.verFlag )
         this->DisplayVersion () ;
      else if ( clArgs.helpFlag )
         this->DisplayHelp () ;
      else
         wcout << "\n ERROR! Unable to initialize NCurses engine.\n" << endl ;
   }

}  //* End Exercalc() *

//*************************
//*     UserInterface     *
//*************************
//******************************************************************************
//* Interact with the user.                                                    *
//*                                                                            *
//* Input  : ca   : (by reference) command-line arguments                      *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Exercalc::UserInterface ( const commArgs& ca )
{
   gString gsOut ;               // output formatting
   wkeyCode wk ;                 // user input
   winPos wp( 1, 2 ) ;           // cursor position

   this->ClearWindow () ;        // clear the terminal window

   //* Attach the dialog controls to the dialog definition.*
   udInit.ctrlPtr = ic ;

   //* Reinitialize color attributes for the dialog definitions.*
   tdInit.borderColor = tdInit.interiorColor = this->cfg.tt ;
   sdInit.borderColor = sdInit.interiorColor = this->cfg.bb ;
   udInit.borderColor = this->cfg.ub ; udInit.interiorColor = this->cfg.bb ;
   udInit.ctrlPtr[cloPB].nColor   = this->cfg.pn ;
   udInit.ctrlPtr[cloPB].fColor   = this->cfg.pf ;
   udInit.ctrlPtr[selDD].nColor   = this->cfg.dn ;
   udInit.ctrlPtr[selDD].fColor   = this->cfg.df ;
   udInit.ctrlPtr[selDD].scrColor = this->cfg.dattr ;
   udInit.ctrlPtr[valTB].nColor   = this->cfg.bb ;
   udInit.ctrlPtr[valTB].fColor   = this->cfg.tf ;
   udInit.ctrlPtr[optMW].nColor   = this->cfg.mn ;
   udInit.ctrlPtr[optMW].fColor   = this->cfg.mf ;
   udInit.ctrlPtr[optMW].scrColor = this->cfg.mattr ;
   udInit.ctrlPtr[hlpMW].nColor   = this->cfg.mn ;
   udInit.ctrlPtr[hlpMW].fColor   = this->cfg.mf ;
   udInit.ctrlPtr[hlpMW].scrColor = this->cfg.mono ;
   udInit.ctrlPtr[savPB].nColor   = this->cfg.pn ;
   udInit.ctrlPtr[savPB].fColor   = this->cfg.pf ;

   //* Adjust additional control parameters *
   ic[cloPB].dispText = UserDlgLabels[this->cfg.lang][cloPB] ;
   ic[savPB].dispText = UserDlgLabels[this->cfg.lang][savPB] ;
   ic[selDD].dispText = (const char*)&selparmData[this->cfg.lang] ;
   ic[optMW].label    = UserDlgLabels[this->cfg.lang][optMW] ;
   ic[optMW].dispText = (const char*)&optparmData[this->cfg.lang] ;
   ic[hlpMW].dispText = (const char*)&hlpparmData[this->cfg.lang] ;
   ic[valTB].label    = UserDlgLabels[this->cfg.lang][valTB] ;

   #if DEBUG_MENU != 0
   udInit.ctrlPtr[dbgMW].nColor   = this->cfg.mn ;
   udInit.ctrlPtr[dbgMW].fColor   = this->cfg.mf ;
   udInit.ctrlPtr[dbgMW].scrColor = this->cfg.mono ;
   ic[dbgMW].dispText = (const char*)&dbgparmData[this->cfg.lang] ;
   #endif   // DEBUG_MENU

   //** Command-line options (if specified) override **
   //** configuration-file parameters.               **
   this->CommandLineOverrides ( ca ) ;

   //* Instantiate the title dialog window *
   this->titleDlg = new NcDialog ( tdInit ) ;
   //* Instantiate the static data dialog window *
   this->statsDlg = new NcDialog ( sdInit ) ;
   //* Instantiate the user-interface dialog window *
   this->userDlg  = new NcDialog ( udInit ) ;

   //* If user interface language is an RTL language,   *
   //* set the internal NcDialog flags for RTL output.  *
   //* Note that the Textbox control remains configured *
   //* for LTR input. (Input is primarily numeric data.)*
   if ( this->cfg.rtl )
   {
      this->titleDlg->DrawLabelsAsRTL () ;
      this->statsDlg->DrawLabelsAsRTL () ;
      this->userDlg->DrawLabelsAsRTL () ;
      this->userDlg->DrawContentsAsRTL ( optMW, true ) ;
      this->userDlg->DrawContentsAsRTL ( hlpMW, true ) ;
      this->userDlg->DrawContentsAsRTL ( selDD, true ) ;
      #if DEBUG_MENU != 0
      this->userDlg->DrawContentsAsRTL ( dbgMW, true ) ;
      #endif   // DEBUG_MENU
   }


   if ( (this->titleDlg->OpenWindow()) == OK )  // open the title dialog
   {
      gsOut.compose( titleTemplate, AppTitle, AppVersion, crYears ) ;
      gsOut.append( L"\x2014  " ) ;
      gString gstmp( TitleDesc[this->cfg.lang] ) ;
      if ( this->cfg.rtl )
         gstmp.textReverse() ;
      gsOut.append( gstmp.gstr() ) ;
      gsOut.padCols( tdInit.dColumns - 4 ) ; // pad data (for dark backgrounds)
      this->titleDlg->WriteString ( wp, gsOut, this->cfg.tt, true ) ;
#if 0    // TEMP - CAPTURE
this->titleDlg->CaptureDialog ( "capturedlg.txt" ) ;
#if 1    // CAPTURE - TWO-COLUMN CHARACTERS
this->titleDlg->CaptureDialog ( "capturedlg.html", true, false, 
                                "infodoc-styles.css", 4, false, nc.blR, true ) ;
#else    // CAPTURE - SINGLE-COLUMN CHARACTERS
this->titleDlg->CaptureDialog ( "capturedlg.html", true, false, 
                                "infodoc-styles.css", 4, false, nc.blR ) ;
#endif   // CAPTURE
#endif   // CAPTURE
   }

   if ( (this->statsDlg->OpenWindow()) == OK )  // open the stats dialog
   {
      this->statsDlg->SetDialogTitle ( StatInfoTitle[this->cfg.lang], 
                                       this->cfg.bb | ncuATTR ) ;

      #if DEBUG_SETUP_PARMS == 0
      this->DisplayMETsTable ()  ;  // Display the METs table
      #else                         // Display Debugging Data
      this->DebugInfo2StatDlg ( ca ) ;
      #endif   // DEBUG_SETUP_PARMS
   }
   if ( (this->userDlg->OpenWindow()) == OK )   // open the user-interface dialog
   {
      //* Set dialog title *
      this->userDlg->SetDialogTitle ( UserDlgTitle[this->cfg.lang], 
                                      this->cfg.ub | ncuATTR ) ;

      //* Erase Textbox label (control is initially invisible) *
      this->userDlg->ClearLine ( (ic[valTB].ulY + ic[valTB].labY), false, 
                                 (ic[valTB].ulX + ic[valTB].labX) ) ;

      //* Attach sub-menus to main menu controls and set border colors *
      short mwList[] = { MAX_DIALOG_CONTROLS, MAX_DIALOG_CONTROLS,
                         MAX_DIALOG_CONTROLS, MAX_DIALOG_CONTROLS,
                         MAX_DIALOG_CONTROLS, MAX_DIALOG_CONTROLS,
                         MAX_DIALOG_CONTROLS, MAX_DIALOG_CONTROLS,
                         hlpMW, 
                         #if DEBUG_MENU != 0
                         dbgMW, 
                         #endif   // DEBUG_MENU
                         -1 } ;
      this->userDlg->AttachMenuwinSubmenus ( optMW, mwList ) ;
      this->userDlg->SetMenuwinBorderAttr ( optMW, true ) ;
      this->userDlg->SetMenuwinBorderAttr ( hlpMW, true ) ;
      #if DEBUG_MENU != 0
      this->userDlg->SetMenuwinBorderAttr ( dbgMW, true ) ;
      #endif   // DEBUG_MENU

      //* Clear the picture area and adjust dialog border.*
      this->pictArea = { short(ic[optMW].ulY + 2), ZERO } ;
      this->pictRows = udInit.dLines - this->pictArea.ypos ;
      this->pictCols = udInit.dColumns ;
      this->userDlg->ClearArea ( this->pictArea.ypos, this->pictArea.xpos,
                                 this->pictRows, this->pictCols, this->cfg.ub ) ;
      LineDef bdef( ncltHORIZ, ncltSINGLE,
                    this->pictArea.ypos - 1, ZERO, 
                    udInit.dColumns, this->cfg.ub ) ;
      bdef.cLeft = bdef.cRight = false ;
      this->userDlg->DrawLine ( bdef ) ;
      this->userDlg->WriteChar ( this->pictArea.ypos - 1, ZERO, wcsLLs, this->cfg.ub ) ;
      this->userDlg->WriteChar ( this->pictArea.ypos - 1, pictCols - 1, wcsLRs, this->cfg.ub ) ;
      //* Draw the picture in the picture area *
      this->DrawPicture () ;

      //* Establish user-data display area.*
      this->userArea = { 1, 1 } ;
      this->userRows = USER_ROWS ;
      this->userCols = udInit.dColumns - 2 ;
      this->userMsg = this->userRows + 2 ;
      LineDef ldef( ncltHORIZ, ncltDUAL, 
                    short(this->userArea.ypos + this->userRows), ZERO, 
                    udInit.dColumns, this->cfg.ub ) ;
      this->userDlg->DrawLine ( ldef ) ;

      //* Calculate the dynamic member values and display the   *
      //* user data and make a backup copy of the original data.*
      //* If valid data specified, indicate that data record    *
      //* not yet saved to log file.                            *
      this->Recalculate ( bool(ca.showGoal == sgNoshow) ) ;
      this->udb = this->ud ;
      if ( this->udb.inpValue > 0.0 )
         this->logPend = true ;

      //* If specified, display the "Exercise Goal" record *
      if ( ca.showGoal != sgNoshow )
         this->GoalRecord ( bool(ca.showGoal == sgWeekly) ) ;

      //* Establish communications with the Wayland system clipboard.    *
      //*     (This connection is good across all dialog windows.)       *
      //* If unable to establish connection, disable clipboard menu item.*
      if ( ! (this->wcbActive = this->titleDlg->wcbEnable ()) )
      {
         this->cfg.mattr[optCopy] = this->cfg.pn ;
         this->cfg.mflag[optCopy] = false ;
         this->userDlg->SetActiveMenuItems ( optMW, this->cfg.mflag, this->cfg.mattr ) ;
      }
#if 0    // TEMP - CAPTURE
this->userDlg->CaptureDialog ( "capturedlg.txt" ) ;
#if 0    // CAPTURE - TWO-COLUMN CHARACTERS
this->userDlg->CaptureDialog ( "capturedlg.html", true, false, 
                               "infodoc-styles.css", 4, false, nc.blR, true ) ;
#else    // CAPTURE - SINGLE-COLUMN CHARACTERS
this->userDlg->CaptureDialog ( "capturedlg.html", true, false, 
                               "infodoc-styles.css", 4, false, nc.blR ) ;
#endif   // CAPTURE
#endif   // CAPTURE
   }

   uiInfo Info ;                 // user interface data returned here
   short  icIndex = ZERO ;       // index of control with input focus
   bool   done = false ;         // loop control

//* TEMP */ this->DebugChartWidget () ;

   while ( ! done )
   {
      //*******************************************
      //* If focus is currently on a Pushbutton   *
      //*******************************************
      if ( ic[icIndex].type == dctPUSHBUTTON )
      {
         if ( Info.viaHotkey )
            Info.HotData2Primary () ;
         else
            icIndex = this->userDlg->EditPushbutton ( Info ) ;
         if ( Info.dataMod != false )
         {
            if ( Info.ctrlIndex == savPB )
            {  //* Add the currently-displayed data to the log file.*
               this->LogfileAppend () ;
            }
            else  // (InfoctrlIndex == cloPB)
            {
               //* If automatic log-file save is enabled AND   *
               //* if the base record has not been previously  *
               //* saved, save the base record to the log file.*
               if ( this->cfg.logAuto && this->logPend )
               {
                  this->ud = this->udb ;  // make base record the current record
                  this->LogfileAppend ( false ) ;
               }

               done = true ;
            }
         }
      }

      //***********************************************
      //* If focus is currently on a Dropdown control *
      //***********************************************
      else if ( ic[icIndex].type == dctDROPDOWN )
      {
         //* Note that the selected member is always offset ZERO on entry.*
         icIndex = this->userDlg->EditDropdown ( Info ) ;
         //* Only an actively-selected item will trigger action. *
         //* A passive selection (TAB/SHIFT_TAB)==no change.     *
         if ( (Info.dataMod != false) && (Info.selMember != spNone) &&
              ((Info.wk.type == wktFUNKEY && Info.wk.key == nckENTER) || 
               (Info.wk.type == wktPRINT && Info.wk.key == nckSPACE)) )
         {
            this->NewParameter ( (selParm)(Info.selMember) ) ;

            //* Focus will move to 'close' pushbutton *
            while ( (icIndex = this->userDlg->NextControl ()) != savPB ) ;
            Info.keyIn = nckTAB ;

            //* Set Dropdown selection back to 'none' *
            this->userDlg->SetDropdownSelect ( selDD, spNone ) ;
         }
         else     // no DD selection
         {
            icIndex = this->userDlg->PrevControl () ;
            this->userDlg->SetDropdownSelect ( selDD, spNone ) ;
            icIndex = this->userDlg->NextControl () ;
         }

      }     // dctDROPDOWN

      //*******************************************
      //* If focus is currently on a Menuwin      *
      //*******************************************
      else if ( ic[icIndex].type == dctMENUWIN )
      {
         icIndex = this->userDlg->EditMenuwin ( Info ) ;

         if ( Info.dataMod )
         {
            //* Options Menu *
            if ( Info.ctrlIndex == optMW )
            {
               optParm op = optParm(Info.selMember) ;
               switch ( op )
               {
                  case optCopy:              // Copy data to clipboard
                     this->CopyRecord2Clipboard () ;
                     break ;

                  case optLogV:              // Log File View
                     this->ViewLog () ;
                     break ;

                  case optLogY:              // Log File Summary
                     this->LogfileSummary ( true ) ;
                     break ;

                  case optLogW:              // Write the base record to the log file
                     this->LogfileAppend () ;
                     break ;

                  case optLogR:              // Read (load) a record from the log file
                     this->LoadRecord () ;
                     break ;

                  case optLogE:              // Erase contents of log file
                     {
                     gsOut = this->cfg.logPath ;
                     if ( (this->LogfileCreate ( gsOut )) )
                        this->UserMessage ( LogMsgs[this->cfg.lang][lmReinit], 
                                            this->cfg.em, true ) ;
                     else
                        this->UserMessage ( LogMsgs[this->cfg.lang][lmNoTarget], 
                                            this->cfg.em, true ) ;
                     chrono::duration<short>aWhile( 2 ) ;
                     this_thread::sleep_for( aWhile ) ;
                     this->UserMessage ( "", this->cfg.bb ) ;
                     }
                     break ;

                  case optComp:              // Compare current data set with prevous set
                     this->CompareRecords ( this->udb, this->ud ) ;
                     break ;

                  case optRest:              // Restore previous data set
                     this->NewParameter ( spDiscard ) ;
                     break ;


                  case optHelp:  break ;  // silence the compiler warning
                  case optITEMS: break ;  // silence the compiler warning
                  #if DEBUG_MENU != 0
                  case optDebug: break ;  // silence the compiler warning
                  #endif   // DEBUG_MENU
               } ;
            }

            //* Help sub-Menu *
            else if ( Info.ctrlIndex == hlpMW )
            {
               switch ( Info.selMember )
               {
                  case hlpInfo:           // info-reader Help
                  case hlpHtml:           // HTML-format Help
                     this->Cry4Help ( Info.selMember == hlpHtml ? true : false ) ;
                     break ;
                  case hlpAbout:          // display Help-about dialog
                     this->HelpAbout () ;
                     break ;
               } ;
            }

            #if DEBUG_MENU != 0
            //* Debugging Options *
            else if ( Info.ctrlIndex == dbgMW )
            {
               if ( Info.selMember == dbgColor )
                  this->ColorScheme2StatDlg () ;
               else if ( Info.selMember == dbgCmap )
                  this->ColorMap2StatDlg () ;
               else if ( Info.selMember == dbgSample )
                  this->LogfileDebugSample () ;
               else if ( Info.selMember == dbgConfig )
                  this->DebugInfo2StatDlg ( ca ) ;
               else if ( Info.selMember == dbgRtl )
                  this->DebugRtlExample () ;
               else if ( Info.selMember == dbgWidget )
                  this->DebugChartWidget () ;
            }
            #endif   // DEBUG_MENU
         }
      }

      //*******************************************
      //* If focus is currently on a Textbox      *
      //*******************************************
      else if ( ic[icIndex].type == dctTEXTBOX )
      { /* Textbox input is handled in NewParameter() method. */ }

      //*******************************************
      //* If focus is currently on a Radiobutton  *
      //*******************************************
      else if ( ic[icIndex].type == dctRADIOBUTTON )
         { /* there are no Radiobutton controls defined for this dialog */ }
      //*******************************************
      //* If focus is currently on a Scrollbox    *
      //*******************************************
      else if ( ic[icIndex].type == dctSCROLLBOX )
         { /* there are no Scrollbox controls defined for this dialog */ }
      //*******************************************
      //* If focus is currently on a Scrollext    *
      //*******************************************
      else if ( ic[icIndex].type == dctSCROLLEXT )
         { /* there are no Scrollext controls defined for this dialog */ }
      //*******************************************
      //* If focus is currently on a Spinner      *
      //*******************************************
      else if ( ic[icIndex].type == dctSPINNER )
         { /* there are no Spinner controls defined for this dialog */ }

      //* If user exited the control edit method via a hotkey,*
      //* then the new control already has focus. Otherwise,  *
      //* move input focus to next/previous control.          *
      if ( done == false && Info.viaHotkey == false )
      {
         if ( Info.keyIn == nckSTAB )
            icIndex = this->userDlg->PrevControl () ; 
         else
            icIndex = this->userDlg->NextControl () ;
      }
   }     // while()

   if ( this->titleDlg != NULL )
   { delete ( this->titleDlg ) ; this->titleDlg = NULL ; }
   if ( this->statsDlg != NULL )
   { delete ( this->statsDlg ) ; this->statsDlg = NULL ; }
   if ( this->userDlg != NULL )
   { delete ( this->userDlg ) ; this->userDlg = NULL ; }

}  //* End UserInterface() *

//*************************
//*     NewParameter      *
//*************************
//******************************************************************************
//* User has selected a parameter to be modified. Get user input, recalculate  *
//* the data members and update the displayed data.                            *
//*                                                                            *
//* Input  : pType : parameter type (member of enum selParm)                   *
//*                                                                            *
//* Returns: 'true' if operation successful                                    *
//*          'false' if user aborts selection                                  *
//******************************************************************************

bool Exercalc::NewParameter ( selParm pType )
{
   gString gs,                // receives user input
           gsPrompt ;         // prompt for exercise type
   chrono::duration<short>aWhile( 3 ) ;
   bool    status = true ;    // return value

   //* If user command is to discard parameter modifications, *
   //* restore the original parameters and refresh display.   *
   if ( pType == spDiscard )
   {
      this->ud = this->udb ;        // restore from backup
      // Programmer's Note: The just-restored record may have been previously 
      // saved, but we do not track that here.
      this->Recalculate ()  ;       // recalculate and refresh display
      this->DisplayMETsTable () ;   // be sure correct METS table is displayed
      this->UserMessage ( parmsRestored[this->cfg.lang], this->cfg.em, true ) ;
      this_thread::sleep_for( aWhile ) ;
      this->UserMessage ( "", this->cfg.bb ) ;
      this->gDisp = false ;         // be sure "Goal" flag is reset
      return status ;
   }

   //* Display the selection prompt (right-justified on the line) *
   gsPrompt = selectPrompt[this->cfg.lang][pType] ;
   if ( ! this->cfg.rtl )
      gsPrompt.shiftCols( udInit.dColumns - gsPrompt.gscols() - 4 ) ;
   this->UserMessage ( gsPrompt, this->cfg.sb ) ;

   //* Enable the input textbox, set focus and display textbox label *
   // Programmer's Note: All user input to the Textbox control should be 
   // numeric data. This avoids difficulties with RTL text input.
   this->userDlg->ControlActive ( valTB, true ) ;
   while ( (this->userDlg->PrevControl ()) != valTB ) ;
   gs = UserDlgLabels[this->cfg.lang][valTB] ;
   short yOffset = ic[valTB].ulY + ic[valTB].labY,
         xOffset = ic[valTB].ulX + ic[valTB].labX ;
   if ( this->cfg.rtl )
      xOffset += gs.gscols() ;
   this->userDlg->WriteString ( yOffset, xOffset, gs, 
                                this->cfg.sb, true, this->cfg.rtl ) ;
#if 0    // TEMP - CAPTURE
this->userDlg->CaptureDialog ( "capturedlg.txt" ) ;
this->userDlg->CaptureDialog ( "capturedlg.html", true, false, 
                               "infodoc-styles.css", 4, false, nc.blR ) ;
#endif   // CAPTURE

   //***********************
   //* Get user's response *
   //***********************
   uiInfo Info ;              // user interface data returned here
   bool update = false,       // 'true' if display updated
        done = false ;        // loop control

   while ( ! done )
   {
      //* TAB key returned from EditTextbox() aborts the operation.*
      this->userDlg->EditTextbox ( Info ) ;
      if ( Info.dataMod != false )
      {
         //* Convert to numeric value *
         this->userDlg->GetTextboxText ( valTB, gs ) ;
         float userval = this->DecodeInputValues ( gs, vuNone ) ;
         bool  altUnits = false ;   // 'true' if English units
         //* If user entered distance in miles *
         if ( (pType == spKm) && ((gs.find( L'm' )) > ZERO) )
            altUnits = true ;
         //* If user entered mass in pounds *
         if ( (pType == spKg) && 
              (((gs.find( L'l' )) > ZERO) || ((gs.find( L'p' )) > ZERO)) )
            altUnits = true ;
         done = true ;     // assume that user has entered a valid argument

         //* If user is selecting a new exercise type.                         *
         //* -- When moving among walk/bike/run, the 'inpValue' and 'inpUnits' *
         //*    members will not be modified, which may render the data in the *
         //*    'Workout Results' column nearly useless (but harmless).        *
         //* -- When moving from walk/bike/run to general exercise, OR         *
         //*    when moving from general exercise to walk/bike/run, only the   *
         //*    elapsed time has value, so copy 'mins' to 'inpValue' and set   *
         //*    'inpUnits' to vuMinutes.                                       *
         if ( pType == spXType )
         {
            short type = short(userval) ;
            update = false ;
            switch ( type )
            {
               case 1:
                  if ( this->ud.exerType != xtWalk )
                  {
                     //if ( this->exerType == xtGeneral )
                     {
                        this->ud.inpValue = this->ud.mins ;
                        this->ud.inpUnits = vuMinutes ;
                     }
                     this->ud.exerType = xtWalk ;
                     update = true ;
                  }
                  break ;
               case 2:
                  if ( this->ud.exerType != xtBike )
                  {
                     //if ( this->exerType == xtGeneral )
                     {
                        this->ud.inpValue = this->ud.mins ; 
                        this->ud.inpUnits = vuMinutes ;
                     }
                     this->ud.exerType = xtBike ;
                     update = true ;
                  }
                  break ;
               case 3:
                  if ( this->ud.exerType != xtRun )
                  {
                     //if ( this->exerType == xtGeneral )
                     {
                        this->ud.inpValue = this->ud.mins ;
                        this->ud.inpUnits = vuMinutes ;
                     }
                     this->ud.exerType = xtRun ;
                     update = true ;
                  }
                  break ;
               case 4:
                  if ( this->ud.exerType != xtGeneral )
                  {
                     this->ud.inpValue = this->ud.mins ;
                     this->ud.inpUnits = vuMinutes ;
                     this->ud.exerType = xtGeneral ;
                     update = true ;
                  }
                  break ;
               default:             // value out-of-range
                  this->UserMessage ( BadUserMsg[this->cfg.lang][bumXType], nc.reB, true ) ;
                  this->userDlg->UserAlert ( 2 ) ;
                  this_thread::sleep_for( aWhile ) ;
                  this->UserMessage ( gsPrompt, this->cfg.sb ) ;
                  done = false ;    // continue
                  break ;
            }

            //* If valid input, recalculate and display user data.   *
            //* Display the lookup table for specified exercise type.*
            if ( update )
            {
               this->Recalculate ()  ;
               this->DisplayMETsTable () ;
            }
         }

         else if ( pType == spGoal )
         {
            short eperiod = short(userval) ; // user selection (daily or weekly)
            if ( eperiod == 1 || eperiod == 2 )
            {
               update = this->GoalRecord ( bool(eperiod == 2) ) ;
            }
            else     // invalid user input
            {
               this->UserMessage ( BadUserMsg[this->cfg.lang][bumGoal], nc.reB, true ) ;
               this->userDlg->UserAlert ( 2 ) ;
               this_thread::sleep_for( aWhile ) ;
               this->UserMessage ( gsPrompt, this->cfg.sb ) ;
               done = false ;    // continue
            }
         }

         //* Else, assign units to the numeric argument *
         else if ( userval > 0.0 )
         {
            update = true ;         // assume a valid value

            //* If moving from General exercise to walk/bike/run, *
            //* recalculations will be based on elapsed time.     *
            if ( this->ud.exerType == xtGeneral )
            {
               this->ud.inpValue = this->ud.mins ;
               this->ud.inpUnits = vuMinutes ;
            }

            //* Else, moving among the walk/bike/run group *
            else
            {
               this->ud.inpValue = userval ; // user's new value is input value

               switch ( pType )
               {
                  case spTime:      // elapsed time
                     this->ud.inpUnits = vuMinutes ;
                     break ;
                  case spKm:        // kilometers (or miles)
                     this->ud.inpUnits = altUnits ? vuMiles : vuKmeters ;
                     break ;
                  case spKg:        // kilograms (or pounds)
                     this->ud.inpUnits = altUnits ? vuPounds : vuKilos ;
                     break ;
                  case spKc:        // kilocalories
                     this->ud.inpUnits = vuKcalories;
                     break ;
                  case spMets:               // METs
                     //* Changing the METs value leaves the exercise *
                     //* type, distance and units unchanged.         *
                     //* Recalculate using specified METs value and  *
                     //* the elapsed time.                           *
                     this->ud.inpValue = this->ud.mins ;
                     this->ud.inpUnits = vuMinutes ;
                     if ( this->ud.exerType == xtWalk )
                        this->ud.metsWalk = userval ;
                     else if ( this->ud.exerType == xtBike )
                        this->ud.metsBike = userval ;
                     else if ( this->ud.exerType == xtRun )
                        this->ud.metsRun = userval ;
                     break ;
                  case spNone:               // no selection
                  case spXType:              // (silence compiler warning)
                  default:
                     status = update = false ;
                     break ;
               } ;
            }

            //* If valid input, recalculate and display user data.*
            //* Exercise type has not changed, therefore lookup   *
            //* table is also unchanged.                          *
            if ( update )
            {
               this->Recalculate ()  ;
            }
         }
         else     // value <= 0.0 (out-of-range)
         {
            this->UserMessage ( BadUserMsg[this->cfg.lang][bumRange], nc.reB, true ) ;
            this->userDlg->UserAlert ( 2 ) ;
            this_thread::sleep_for( aWhile ) ;
            this->UserMessage ( "", this->cfg.bb ) ;
            done = false ;    // continue
         }
      }
      else     // user did not enter a value, (abort)
      {
         status = false ;
         done = true ;
      }

      //* Clear Textbox contents *
      this->userDlg->NextControl () ;     // move focus
      this->userDlg->SetTextboxText ( valTB, "" ) ;
      this->userDlg->PrevControl () ;     // restore focus
   }     // while()

   //* Move focus away from Textbox control, disable *
   //* the Textbox and erase the prompt (if any).    *
   while ( (this->userDlg->NextControl ()) != savPB ) ; // last control
   this->userDlg->ControlActive ( valTB, false ) ;
   this->userDlg->ClearLine ( (ic[valTB].ulY + ic[valTB].labY), false, 
                              (ic[valTB].ulX + ic[valTB].labX) ) ;
   this->UserMessage ( "", this->cfg.bb ) ;

   //* If "Goal" is not displayed, reset the flag.*
   if ( update && (pType != spGoal) )
      this->gDisp = false ;

   return status ;

}  //* End NewParameter() *

//*************************
//*      GoalRecord       *
//*************************
//******************************************************************************
//* Generate and display an "ideal" record that reflects user's goal for       *
//* exercise as specified in the configuration file.                           *
//* The record is generated based on the 'gUnits', 'gValue' and 'gXType'       *
//* members, as well as the METs value (intensity) associated with the         *
//* specified exercise type (metsWalk/metsBike/metsRun/metsGen).               *
//*                                                                            *
//* Input  : weekly : 'true'  if weekly goal                                   *
//*                   'false' if daily goal                                    *
//*                                                                            *
//* Returns: 'true'                                                            *
//******************************************************************************

bool Exercalc::GoalRecord ( bool weekly )
{
   gString gsOut ;                     // text formatting
   float   saveMets ;                  // temp storage
   bool    updated = true ;            // return value (always 'true')

   this->gWeek = weekly ;              // initialize the daily/weekly flag
   this->ud = this->udb ;              // retrieve the static values
   this->ud.inpUnits = this->gUnits ;  // set the goal values
   this->ud.inpValue = this->gValue * (this->gWeek ? 7.0 : 1.0) ;
   this->ud.exerType = this->gXType ;
   switch ( this->ud.exerType )        // copy METs goal to associated METs value
   {
      case xtBike:
         saveMets = this->ud.metsBike ;
         this->ud.metsBike = this->gMets ;
         break ;
      case xtRun:
         saveMets = this->ud.metsRun ;
         this->ud.metsRun = this->gMets ;
         break ;
      case xtGeneral:
         saveMets = this->ud.metsGen ;
         this->ud.metsGen = this->gMets ;
         break ;
      case xtWalk:
      default:
         saveMets = this->ud.metsWalk ;
         this->ud.metsWalk = this->gMets ;
         break ;
   }

   this->Recalculate ()  ;       // calculate and display the record
   switch ( this->gXType )       // restore obscured static value
   {
      case xtBike:    this->ud.metsBike = saveMets ;       break ;
      case xtRun:     this->ud.metsRun  = saveMets ;       break ;
      case xtGeneral: this->ud.metsGen  = saveMets ;       break ;
      case xtWalk: default: this->ud.metsWalk = saveMets ; break ;
   }
   this->DisplayMETsTable () ;   // display the appropriate METs table
   this->gDisp = true ;          // set 'goal displayed' flag

   //* Indicate that the "Goal" record is displayed.       *
   //*             (includes age and gender)               *
   //* The target display area (last row of the user area) *
   //* is assumed to be free.                              *
   const wchar_t* const gmTemplate = L"  %S - %S: %S %S: %S  " ;
   gString gsAge( "%.1f", &this->ud.ageY ) ;
   if ( this->cfg.rtl ) // age value reversed for RTL output
      gsAge.textReverse() ;
   gsOut.compose( gmTemplate,
         GoalMsgs[this->cfg.lang][this->gWeek ? gmWeekly : gmDaily],
         GoalMsgs[this->cfg.lang][gmAge], gsAge.gstr(),
         GoalMsgs[this->cfg.lang][gmGender], 
         GoalMsgs[this->cfg.lang][this->ud.gender == goMale ? gmMale :
                                  this->ud.gender == goFemale ? gmFemale : gmOther] ) ;
   winPos wp( short(this->userArea.ypos + this->userRows - 1),
              short(this->userCols / 2 - gsOut.gscols() / 2) ) ;
   if ( this->cfg.rtl ) // adjust X offset for RTL text
      wp.xpos += gsOut.gscols() ;
   this->userDlg->WriteString ( wp, gsOut, this->cfg.em, true, this->cfg.rtl ) ;

   return updated ;

}  //* End GoalRecord() *

//*************************
//*      Recalculate      *
//*************************
//******************************************************************************
//* Given: 1) 'inpValue' (user's input value)                                  *
//*        2) 'inpUnits' (units of inpValue)                                   *
//*        3) 'exerType' (type of exercise performed)                          *
//* Calculate values for dynamic member variables:                             *
//*        1) BMI (Body Mass Index)                                            *
//*        2) 'kmeters' and 'miles' (distance travelled)                       *
//*                 (n/a for 'General' activity)                               *
//*        3) 'kph' (kilometers-per-hour for activity)                         *
//*                 (n/a for 'General' activity)                               *
//*        4) 'mins' (duration of activity in minutes)                         *
//*        5) 'mets' (energy burned during activity)                           *
//*        6) 'kcal' (kilocalories burned during activity)                     *
//*        7) various transient data points which are calculated and           *
//*           displayed, but not saved                                         *
//*                                                                            *
//* Write the data into the user-interface window.                             *
//*                                                                            *
//* Input  : display : (optional, 'true' by default)                           *
//*                    if 'true',  display the results in user-interface window*
//*                    if 'false', do not update the display                   *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Exercalc::Recalculate ( bool display )
{
   //* Calculate the BMI *
   if ( (this->ud.heightM > 0.0) && (this->ud.massKg > 0.0) )
      this->ud.bmi = this->ud.massKg / (this->ud.heightM * this->ud.heightM) ;

   //* Get the user's standard velocity for this activity *
   //* (Not applicable for 'General' exercise.)           *
   //* Get METs ratio for this activity.
   if ( this->ud.exerType == xtWalk )
   { this->ud.kph = this->ud.kphWalk ; this->ud.metTable = this->ud.metsWalk ; }
   else if ( this->ud.exerType == xtBike )
   { this->ud.kph = this->ud.kphBike ; this->ud.metTable = this->ud.metsBike ; }
   else if ( this->ud.exerType == xtRun )
   { this->ud.kph = this->ud.kphRun ;  this->ud.metTable = this->ud.metsRun ;  }
   else
   { this->ud.kph = 0.0 ; this->ud.metTable = this->ud.metsGen ; }

   //* Apply correction factors for BMI etc. to    *
   //* the METs value taken from the lookup table. *
   this->ud.metRatio = this->ApplyMETsFactors ( this->ud.metTable ) ;

   if ( this->ud.inpUnits == vuMiles )
   {
      this->ud.miles   = this->ud.inpValue ;
      this->ud.kmeters = this->Miles2Kilometers ( this->ud.miles ) ;
      this->ud.mins    = this->Kilometers2Minutes ( this->ud.kmeters, this->ud.kph ) ;
      this->ud.metmins = this->MetMinutes ( this->ud.metRatio, this->ud.mins ) ;
      this->ud.kcal    = this->Mets2kCal ( this->ud.metRatio, this->ud.massKg, 
                                           this->ud.mins ) ;
   }
   else if ( this->ud.inpUnits == vuKmeters )
   {
      this->ud.kmeters = this->ud.inpValue ;
      this->ud.miles = this->Kilometers2Miles ( this->ud.kmeters ) ;
      this->ud.mins    = this->Kilometers2Minutes ( this->ud.kmeters, this->ud.kph ) ;
      this->ud.metmins = this->MetMinutes ( this->ud.metRatio, this->ud.mins ) ;
      this->ud.kcal    = this->Mets2kCal ( this->ud.metRatio, this->ud.massKg, 
                                           this->ud.mins ) ;
   }
   else if ( this->ud.inpUnits == vuMinutes )
   {
      this->ud.mins = this->ud.inpValue ;
      //* If applicable, calculate distance travelled *
      if ( this->ud.exerType == xtWalk || 
           this->ud.exerType == xtBike || this->ud.exerType == xtRun )
      {
         this->ud.kmeters = this->Minutes2Kilometers ( this->ud.mins, this->ud.kph ) ;
         this->ud.miles   = this->Kilometers2Miles ( this->ud.kmeters ) ;
      }
      else
      { this->ud.kmeters = this->ud.miles = ZERO ; }

      this->ud.metmins = this->MetMinutes ( this->ud.metRatio, this->ud.mins ) ;
      this->ud.kcal = this->Mets2kCal ( this->ud.metRatio, this->ud.massKg, 
                                        this->ud.mins ) ;
   }
   else if ( this->ud.inpUnits == vuKcalories )
   {
      this->ud.kcal = this->ud.inpValue ;

      //* Convert kCalories to elapsed time in minutes *
      this->ud.mins = 
         this->kCal2Minutes ( this->ud.kcal, this->ud.metRatio, this->ud.massKg ) ;

      //* If applicable, calculate distance travelled *
      if ( this->ud.exerType == xtWalk || 
           this->ud.exerType == xtBike || this->ud.exerType == xtRun )
      {
         //* For distance-based exercises, we must assume that kph *
         //* remains unchanged. Therefore the change in elapsed    *
         //* time will cause an update to kmeters/miles.           *
         this->ud.kmeters = this->Minutes2Kilometers ( this->ud.mins, this->ud.kph ) ;
         this->ud.miles   = this->Kilometers2Miles ( this->ud.kmeters ) ;
      }
      else
      { this->ud.kmeters = this->ud.miles = ZERO ; }

      this->ud.metmins = this->MetMinutes ( this->ud.metRatio, this->ud.mins ) ;
   }
   else if ( (this->ud.inpUnits == vuKilos) || (this->ud.inpUnits == vuPounds) )
   {
      if ( this->ud.inpUnits == vuKilos )
         this->ud.massKg = this->ud.inpValue ;
      else     // vuPounds
         this->ud.massKg = this->Pounds2Kilograms ( this->ud.inpValue ) ;

      // Programmer's Note:
      // Distance, velocity and elapsed time remain unchanged.
      // BMI has been recalculated above.
      // Because mass has changed, kCal burned must be updated.
      this->ud.kcal = this->Mets2kCal ( this->ud.metRatio, this->ud.massKg, 
                                        this->ud.mins ) ;
   }

   //* If caller provided no input value, display a warning. *
   if ( this->ud.inpValue == 0.0 )
      this->UserMessage ( UserNoData[this->cfg.lang], this->cfg.bb, true ) ;


   //*******************************
   //* Display the calculated data *
   //*******************************
   if ( display )
   {
      gString gsOut, gsrev ;
      winPos wp ( this->userArea.ypos, short(this->userArea.xpos + 1) ) ;
      if ( this->cfg.rtl )    // adjust cursor position for RTL languages
         wp.xpos = this->userArea.xpos + udInit.dColumns - 4 ;

      //* Clear the display area *
      this->userDlg->ClearArea ( this->userArea.ypos, this->userArea.xpos, 
                                 this->userRows, this->userCols, this->cfg.bb ) ;

      //* Format and display user input values and calculated data.*
      //* Column title *
      wp = this->userDlg->WriteParagraph ( wp, UserGenlTitle[this->cfg.lang],
                                           this->cfg.bb | ncuATTR, false, this->cfg.rtl ) ;
      //* Basic body data *
      wp = this->userDlg->WriteParagraph ( wp, UserGenlData[this->cfg.lang][xtNone], 
                                           this->cfg.bb, false, this->cfg.rtl ) ;
      //* Data specific to type of activity *
      wp = this->userDlg->WriteParagraph ( wp, UserGenlData[this->cfg.lang][this->ud.exerType], 
                                           this->cfg.bb, false, this->cfg.rtl ) ;
      wp.ypos = this->userArea.ypos + 1 ;
      wp.xpos = UserGenlDataOffset[this->cfg.lang] ;
      #if DEBUG_RTL != 0
      if ( this->cfg.rtl )
         wp.xpos = this->userArea.xpos + udInit.dColumns - UserGenlDataOffset[this->cfg.lang] - 8 ;
      #endif   // DEBUG_RTL

      //* Template for data which is the same for "All" exercise types.*
      const wchar_t* UserGenlTemplateA = L"%6.2f\n%6.2f\n%6.2f\n%6.1f\n%6.2f\n" ;
      //* Template for "Distance" data.*
      const wchar_t* UserGenlTemplateD = L"%6.2f\n%6.2f\n%6.2f\n%6.2f\n" ;
      //* Template for "General" exercise.*
      const wchar_t* UserGenlTemplateG = L"%6.2f\n%6.2f\n" ;

      float miperhr, kcalperhr ;
      float cm = this->ud.heightM * 100.0,
            in = this->Centimeters2Inches ( cm ),
            lb = Kilograms2Pounds ( this->ud.massKg ) ;
      gsOut.compose( UserGenlTemplateA,
                     &cm, &in, &this->ud.massKg, &lb, &this->ud.bmi ) ;
      wp = this->userDlg->WriteParagraph ( wp, gsOut, this->cfg.sb ) ;

      switch ( this->ud.exerType )
      {
         case xtWalk:
            miperhr   = this->Kilometers2Miles ( this->ud.kphWalk ) ;
            kcalperhr = this->Mets2kCal ( this->ud.metsWalk, this->ud.massKg, 60.0 ) ;
            gsOut.compose( UserGenlTemplateD, &this->ud.metsWalk, 
                           &this->ud.kphWalk, &miperhr, &kcalperhr ) ;
            break ;
         case xtBike:
            miperhr   = this->Kilometers2Miles ( this->ud.kphBike ) ;
            kcalperhr = this->Mets2kCal ( this->ud.metsBike, this->ud.massKg, 60.0 ) ;
            gsOut.compose( UserGenlTemplateD, &this->ud.metsBike, &this->ud.kphBike,
                           &miperhr, &kcalperhr ) ;
            break ;
         case xtRun:
            miperhr   = this->Kilometers2Miles ( this->ud.kphRun ) ;
            kcalperhr = this->Mets2kCal ( this->ud.metsRun, this->ud.massKg, 60.0 ) ;
            gsOut.compose( UserGenlTemplateD, &this->ud.metsRun, &this->ud.kphRun,
                           &miperhr, &kcalperhr ) ;
            break ;
         case xtGeneral:
            kcalperhr = this->Mets2kCal ( this->ud.metsGen, this->ud.massKg, 60.0 ) ;
            gsOut.compose( UserGenlTemplateG, 
                           &this->ud.metsGen, &kcalperhr ) ;
            break ;
         case xtNone: default: break ;   // (silence the compiler warning)
      } ;
      wp = this->userDlg->WriteParagraph ( wp, gsOut, this->cfg.sb ) ;

      //* Report Current Statistics *
      wp.ypos = this->userArea.ypos ;
      wp.xpos = UserCurrHdrOffset[this->cfg.lang] ;
      #if DEBUG_RTL
      if ( this->cfg.rtl )
         wp.xpos = this->userArea.xpos + udInit.dColumns - UserCurrHdrOffset[this->cfg.lang] - 2 ;
      #endif   // DEBUG_RTL

      wp = this->userDlg->WriteParagraph ( wp, UserCurrTitle[this->cfg.lang], 
                                           this->cfg.bb | ncuATTR, false, this->cfg.rtl ) ;
      wp.xpos += (this->cfg.rtl ? (-1) : 1) ;
   
      this->userDlg->WriteParagraph ( wp, UserCurrData[this->cfg.lang], 
                                           this->cfg.bb, false, this->cfg.rtl ) ;

      wp.xpos += UserCurrDataOffset[this->cfg.lang] ;
      #if DEBUG_RTL != 0
      if ( this->cfg.rtl )
         wp.xpos -= UserCurrDataOffset[this->cfg.lang] + 17 ;
      #endif   // DEBUG_RTL

      float kgBurned = this->kCal2Kilograms ( this->ud.kcal ),
            lbBurned = this->Kilograms2Pounds ( kgBurned ) ;
      int64_t kCal = int64_t(this->ud.kcal),
              metm = int64_t(this->ud.metmins) ;
      gString gskcal( kCal, FI_MAX_FIELDWIDTH, true ),   // formatted integer strings
              gsmetm( metm, FI_MAX_FIELDWIDTH, true ) ;
      short hour = ZERO,
            minute = ZERO,
            minutes = short(this->ud.mins) ;
      this->Minutes2Time ( this->ud.mins, hour, minute ) ; // break down the time

      //* Exercise Type (text only) *
      gsOut.compose( "%S\n", UserExerType[this->cfg.lang][this->ud.exerType] ) ;
      wp = this->userDlg->WriteParagraph ( wp, gsOut, this->cfg.sb, false, this->cfg.rtl ) ;

      //* Format the time parameters *
      gsOut.compose( "%02hd:%02hd\n"
                     "%hd\n", &hour, &minute, &minutes ) ;
      //* Distance values do not apply to xtGeneral *
      if ( this->ud.exerType == xtGeneral )
         gsOut.append( "--\n--\n" ) ;

      if ( this->cfg.rtl )                // RTL output
      {
         gsOut.textReverse( false, true ) ;

         if ( this->ud.exerType != xtGeneral )
         {
            gsrev.compose( "%6.2f\n"
                           "%6.2f\n",
                           &this->ud.kmeters, &this->ud.miles ) ;
         }
         gsrev.append( "%S\n"
                       "%S\n"
                       "%9.5f\n"
                       "%9.5f\n",
                       gsmetm.gstr(), gskcal.gstr(),
                       &kgBurned, &lbBurned ) ;
         gsrev.textReverse( false, true ) ;
         gsOut.append( "%S\n", gsrev.gstr() ) ;
      }
      else                                // LTR output
      {
         if ( this->ud.exerType != xtGeneral )
         {
            gsOut.append( "%-6.2f\n"
                          "%-6.2f\n",
                          &this->ud.kmeters, &this->ud.miles ) ;
         }
         gsOut.append( "%S\n"
                       "%S\n"
                       "%-9.5f\n"
                       "%-9.5f\n",
                       gsmetm.gstr(), gskcal.gstr(),
                       &kgBurned, &lbBurned ) ;
      }
      wp = this->userDlg->WriteParagraph ( wp, gsOut, this->cfg.sb, 
                                           false, this->cfg.rtl ) ;

      this->userDlg->RefreshWin () ;

      //* Reset the "goal displayed" flag. (see GoalRecord()) *
      this->gDisp = false ;

   }     // if(display)

}  //* End Recalculate() *

//*************************
//*   DisplayMETsTable    *
//*************************
//******************************************************************************
//* Display METs table in the Stats dialog window.                             *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Exercalc::DisplayMETsTable ( void )
{
   //* Redraw the window and clear the display area *
   this->statsDlg->DrawBox ( ZERO, ZERO, sdInit.dLines, sdInit.dColumns, this->cfg.bb ) ;
   this->statsDlg->SetDialogTitle ( StatInfoTitle[this->cfg.lang], 
                                    this->cfg.bb | ncuATTR ) ;
   this->statsDlg->ClearWin () ;

   winPos wp( 1, (this->cfg.rtl ? (sdInit.dColumns - 2) : 1) ) ;
   gString gsOut( StatInfoHdr[this->cfg.lang][this->ud.exerType] ) ;
   gsOut.padCols( sdInit.dColumns - 2 ) ;
   gsOut.append( "\n" ) ;

   wp = this->statsDlg->WriteParagraph ( wp, gsOut, this->cfg.bb | ncuATTR,
                                         false, this->cfg.rtl ) ;
      wp.xpos += (this->cfg.rtl ? -1 : 1) ;

   wp = this->statsDlg->WriteParagraph ( wp, StatInfo[this->cfg.lang][this->ud.exerType], 
                                         this->cfg.bb, false, this->cfg.rtl ) ;
   this->statsDlg->RefreshWin () ;

}  //* End DisplayMETsTable() *

//*************************
//*    CompareRecords     *
//*************************
//******************************************************************************
//* Compare the specified user-stats groups.                                   *
//* -- Usually the 'udA' group will be the original user stats ('udb' member), *
//*    but could be any stats group.                                           *
//* -- The 'udB' group could be a record from the log file, a manually-modified*
//*    group ('ud' member), the "Goal" record or any stats group.              *
//*                                                                            *
//* Data are displayed in the Stats dialog window:                             *
//* 'udA' data are displayed on the left using the base dialog text color.     *
//* 'udB' data are displayed on the right.                                     *
//*       - Values that are identical to (or within the rounding margin) the   *
//*         matching 'udA' value are displayed in the base dialog text color.  *
//*       - Values that differ from the 'udA' value are displayed in a         *
//*         contrasting color.                                                 *
//* Programmer's Note: When floating-point data are encoded for writing to     *
//* the logfile, some precision may be lost which potentially will cause the   *
//* compared values to be reported as different even though they are based on  *
//* a single, shared floating-point value. (example: 'bmi')                    *
//*                                                                            *
//* Input  : udA  : (by reference) first group to be displayed                 *
//*          udB  : (by reference) group to be compared with 'udA'             *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note:                                                         *
//* For RTL languages, the title should have been set as RTL during            *
//* instantiation of the dialog. The contents, however, must be explicitly     *
//* positioned as either LTR or RTL.                                           *
//******************************************************************************

void Exercalc::CompareRecords ( const UserData& udA, const UserData& udB )
{
   //* Format specification for comparison data.*
   const char* f0 = "%-.f" ;
   const char* f2 = "%-.2f" ;
   const char* f3 = "%-.3f" ;
   const short dItems = 10 ;  // number of items in record

   gString gsOut( compRecHdr[this->cfg.lang][crhLeft] ), // text formatting
           gsEt( UserExerType[this->cfg.lang][udA.exerType] ) ;
   winPos wpl( 2, 2 ),        // cursor position, original (left) column
          wpr( wpl.ypos,      // cursor position, retrieved (right) column
               wpl.xpos + gsOut.gscols() + 2 ),
          wptmp ;
   attr_t dAttr[dItems] ;
   for ( short i = ZERO ; i < dItems ; ++i )
      dAttr[i] = this->cfg.bb ;

   //* Adjust for RTL language text *
   if ( this->cfg.rtl )
   {
      wpl.xpos = sdInit.dColumns - 3 ;
      wpr.xpos = wpl.xpos - (gsOut.gscols() + 2) ;
      gsEt.textReverse() ;  // exercise type will be written as LTR
   }

   //* Clear the stats dialog window *
   this->statsDlg->ClearWin () ;

   //* Set the dialog title *
   this->statsDlg->SetDialogTitle ( compRecHdr[this->cfg.lang][crhTitle], 
                                    this->cfg.bb | ncuATTR ) ;

   //* Write left column header *
   this->statsDlg->WriteString ( wpl, compRecHdr[this->cfg.lang][crhLeft], 
                                 this->cfg.bb | ncuATTR, false, this->cfg.rtl ) ;
   ++wpl.ypos ;

   //* Write the variable names *
   wptmp = this->statsDlg->WriteParagraph ( wpl, compRecHdr[this->cfg.lang][crhVnames], 
                                            this->cfg.bb, false, this->cfg.rtl ) ;
   wpl.xpos = wptmp.xpos ;    // offset for data column *

   //* Format and display the base data *
   this->crFormatRecord ( udA, wpl, dAttr ) ;

   //* Write the right column header *
   gsOut = compRecHdr[this->cfg.lang][crhRight] ;
   if ( this->gDisp )
   {
      short colWidth = gsOut.gscols() ;   // remember column width
      gsOut = GoalMsgs[this->cfg.lang][this->gWeek ? gmWeekly : gmDaily] ;
      gsOut.insert( L' ' ) ;              // offset by one space
      gsOut.padCols( colWidth ) ;         // pad to full column width
   }
   this->statsDlg->WriteString ( wpr, gsOut, this->cfg.bb | ncuATTR,
                                 false, this->cfg.rtl ) ;
   ++wpr.ypos ;

   //* Reinitialize the color-attribute array *
   short ai = ZERO ;
   if ( !(fltComp ( udA.heightM, udB.heightM, f3 )) ) { dAttr[ai] = this->cfg.sb ; } ++ai ;
   if ( !(fltComp ( udA.massKg,  udB.massKg,  f2 )) ) { dAttr[ai] = this->cfg.sb ; } ++ai ;
   if ( !(fltComp ( udA.bmi,     udB.bmi,     f2 )) ) { dAttr[ai] = this->cfg.sb ; } ++ai ;
   if ( (udA.exerType != udB.exerType) )              { dAttr[ai] = this->cfg.sb ; } ++ai ;
   if ( !(fltComp ( udA.mins,    udB.mins,    f0 )) ) { dAttr[ai] = this->cfg.sb ; } ++ai ;
   if ( !(fltComp ( udA.kmeters, udB.kmeters, f3 )) ) { dAttr[ai] = this->cfg.sb ; } ++ai ;
   if ( !(fltComp ( udA.kph,     udB.kph,     f3 )) ) { dAttr[ai] = this->cfg.sb ; } ++ai ;
   if ( !(fltComp ( udA.metRatio,udB.metRatio,f3 )) ) { dAttr[ai] = this->cfg.sb ; } ++ai ;
   if ( !(fltComp ( udA.metmins, udB.metmins, f3 )) ) { dAttr[ai] = this->cfg.sb ; } ++ai ;
   if ( !(fltComp ( udA.kcal,    udB.kcal,    f3 )) ) { dAttr[ai] = this->cfg.sb ; } ++ai ;

   //* Format and display the comparison data *
   this->crFormatRecord ( udB, wpr, dAttr ) ;

   this->statsDlg->RefreshWin () ;  // make everything visible

}  //* End CompareRecords() *

//*************************
//*    crFormatRecord     *
//*************************
//******************************************************************************
//* Called by CompareRecords() method to format and display the specified      *
//* UserData record. Display is not refreshed.                                 *
//*                                                                            *
//* Input  : ud    : (by reference) data to be formatted                       *
//*          wpBase: (by reference) base cursor position                       *
//*          dAttr : array of color attributes, one for each value             *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Exercalc::crFormatRecord ( const UserData& ud, const winPos& wpBase,
                                const attr_t* dAttr )
{
   const char* f2 = "%-.2f" ;    // format specifications
   const char* f3 = "%-.3f" ;

   gString gsOut ;               // text formatting
   winPos wp = wpBase ;          // cursor position
   short ai = ZERO,              // index into dAttr[] array
         hours, minutes ;        // formatted time

   gsOut.compose( f3, &ud.heightM ) ;                    // heightM
   if ( this->cfg.rtl ) gsOut.textReverse() ;
   gsOut.append( L'\n' ) ;
   wp = this->statsDlg->WriteParagraph ( wp, gsOut, dAttr[ai++], false, this->cfg.rtl ) ;

   gsOut.compose( f2, &ud.massKg ) ;                     // massKg
   if ( this->cfg.rtl ) gsOut.textReverse() ;
   gsOut.append( L'\n' ) ;
   wp = this->statsDlg->WriteParagraph ( wp, gsOut, dAttr[ai++], false, this->cfg.rtl ) ;

   gsOut.compose( f2, &ud.bmi ) ;                        // bmi (see note above)
   if ( this->cfg.rtl ) gsOut.textReverse() ;
   gsOut.append( L'\n' ) ;
   wp = this->statsDlg->WriteParagraph ( wp, gsOut, dAttr[ai++], false, this->cfg.rtl ) ;

   gsOut.compose( L"%S\n", UserExerType[this->cfg.lang][ud.exerType] ) ; // exercise type
   wp = this->statsDlg->WriteParagraph ( wp, gsOut, dAttr[ai++], false, this->cfg.rtl ) ;

   this->Minutes2Time ( ud.mins, hours, minutes ) ;      // convert to HH:MM
   gsOut.compose( "%02hd:%02hd", &hours, &minutes ) ;    // elapsed time
   if ( this->cfg.rtl ) gsOut.textReverse() ;
   gsOut.append( L'\n' ) ;
   wp = this->statsDlg->WriteParagraph ( wp, gsOut, dAttr[ai++], false, this->cfg.rtl ) ;

   gsOut.compose( f3, &ud.kmeters ) ;                    // kilometers
   if ( this->cfg.rtl ) gsOut.textReverse() ;
   gsOut.append( L'\n' ) ;
   wp = this->statsDlg->WriteParagraph ( wp, gsOut, dAttr[ai++], false, this->cfg.rtl ) ;

   gsOut.compose( f3, &ud.kph ) ;                        // kilometers-per-hour
   if ( this->cfg.rtl ) gsOut.textReverse() ;
   gsOut.append( L'\n' ) ;
   wp = this->statsDlg->WriteParagraph ( wp, gsOut, dAttr[ai++], false, this->cfg.rtl ) ;

   gsOut.compose( f3, &ud.metRatio ) ;                   // METs Ratio
   if ( this->cfg.rtl ) gsOut.textReverse() ;
   gsOut.append( L'\n' ) ;
   wp = this->statsDlg->WriteParagraph ( wp, gsOut, dAttr[ai++], false, this->cfg.rtl ) ;

   gsOut.compose( f3, &ud.metmins ) ;                    // MET minutes
   if ( this->cfg.rtl ) gsOut.textReverse() ;
   gsOut.append( L'\n' ) ;
   wp = this->statsDlg->WriteParagraph ( wp, gsOut, dAttr[ai++], false, this->cfg.rtl ) ;

   gsOut.compose( f3, &ud.kcal ) ;                       // kilocalories
   if ( this->cfg.rtl ) gsOut.textReverse() ;
   gsOut.append( L'\n' ) ;
   wp = this->statsDlg->WriteParagraph ( wp, gsOut, dAttr[ai++], false, this->cfg.rtl ) ;

}  //* End crFormatRecord() *

//*************************
//*        fltComp        *
//*************************
//******************************************************************************
//* Non-member Method - called only by CompareRecords().                       *
//* ----------------------------------------------------                       *
//* Compare two floating-point values as formatted text.                       *
//* We do this because direct comparision of floating-point values is          *
//* unreliable. The floorf(), roundf() and ceilf() C-library functions are all *
//* functional, but not accurate for our application.                          *
//*                                                                            *
//* Input  : fA    : floating-point value to compare A                         *
//*          fB    : floating-point value to compare B                         *
//*          fmt   : formatting template                                       *
//*                                                                            *
//* Returns: 'true'  values compare as equal                                   *
//*          'false' values compare as different                               *
//******************************************************************************

static bool fltComp ( float fA, float fB, const char *fmt )
{

   gString gsA( fmt, &fA ),
           gsB( fmt, &fB ) ;
   return ( bool((gsB.compare( gsA, true )) == ZERO) ) ;

}  //* fltComp() *

//*************************
//* CopyRecord2Clipboard  *
//*************************
//******************************************************************************
//* Copy the currently-displayed record to the system clipboard.               *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if data successfuly placed on clipboard                    *
//*          'false' if clipboard interface is not active                      *
//******************************************************************************

bool Exercalc::CopyRecord2Clipboard ( void )
{
   chrono::duration<short>aWhile( 3 ) ;
   gString gsOut, gsbw ;
   short bytesCaptured = ZERO,
         bytesWritten  = ZERO ;
   bool status = false ;

   if ( (this->userDlg->wcbIsConnected ()) )
   {
      //* Capture the record to a temp buffer (this->udAlloc) *
      bytesCaptured = this->LogfileFormatRecord ( this->ud ) ;

      //* Copy record to clipboard *
      bytesWritten = this->userDlg->wcbSet ( this->udAlloc ) ;

      //* Release the dynamic allocation *
      if ( this->udAlloc != NULL )
      { delete [] this->udAlloc ; this->udAlloc = NULL ; }

      //* Report status *
      if ( bytesCaptured == bytesWritten )
      {
         gsbw.formatInt( bytesWritten, 6, true ) ;
         gsOut.compose( Copy2CbMsgs[this->cfg.lang][c2cbTemplate], gsbw.gstr() ) ;
         status = true ;
      }
      else
         gsOut = Copy2CbMsgs[this->cfg.lang][c2cbError] ;
      this->UserMessage ( gsOut, this->cfg.em, true ) ;
   }
   else     // no clipboard connection
      this->UserMessage ( Copy2CbMsgs[this->cfg.lang][c2cbAvail], this->cfg.em, true ) ;

   this_thread::sleep_for( aWhile ) ;
   this->UserMessage ( "", this->cfg.bb ) ;
   return status ;

}  //* End CopyRecord2Clipboard() *

//*************************
//*   DecodeInputValues   *
//*************************
//******************************************************************************
//* Decode input argument sequence. Add the sequence of values specified.      *
//* a) Called by GetCommandLineArgs() only for decoding arguments of the       *
//*    '-i=VALUE' option.                                                      *
//* b) Called by UserInterface() to decode user input to the textbox control.  *
//*                                                                            *
//*  Examples of valid syntax:                                                 *
//*      Miles or Kilometers: 3.55                                             *
//*                           2.30+12.75+17.236                                *
//*                     Time: 3:50                                             *
//*                           3:50+2:30+7:10+45+75                             *
//*                           256+35+20+45                                     *
//*                                                                            *
//* Input  : gsIn  : (by reference) argument string, i.e. one or more values   *
//*                  separated by the plus sign ( '+' )                        *
//*          units : input-value units (member of enum valUnits)               *
//*                  (currently ignored)                                       *
//*                                                                            *
//* Returns: sum of argument elements                                          *
//******************************************************************************

float Exercalc::DecodeInputValues ( const gString& gsIn, valUnits units )
{
   gString gs = gsIn ;
   float totVal = 0.0 ;
   short indx   = ZERO ;

   do
   {
      totVal += this->Time2Minutes ( gs.ustr() ) ;
      if ( (indx = gs.after( wPLUS )) > ZERO )
         gs.shiftChars( -(indx) ) ;
   }
   while ( indx > ZERO ) ;

   return totVal ;

}  //* End DecodeInputValues() *

//*************************
//*   Miles2Kilometers    *
//*************************
//******************************************************************************
//* Convert Miles to Kilometers.                                               *
//*                                                                            *
//* Input  : fMi : miles                                                       *
//*                                                                            *
//* Returns: kilometers                                                        *
//******************************************************************************

float Exercalc::Miles2Kilometers ( float fMi )
{

   return ( fMi * Mi_per_Km ) ;

}  //* End Miles2Kilometers() *

//*************************
//*   Kilometers2Miles    *
//*************************
//******************************************************************************
//* Convert Kilometers to Miles.                                               *
//*                                                                            *
//* Input  : fKm : kilometers                                                  *
//*                                                                            *
//* Returns: miles                                                             *
//******************************************************************************

float Exercalc::Kilometers2Miles ( float fKm )
{

   return ( fKm * Km_per_Mi ) ;

}  //* End Kilometers2Miles() *

//*************************
//*  Kilometers2Minutes   *
//*************************
//******************************************************************************
//* Given kilometers and kilometers-per-hour, calculate elapsed time.          *
//*                                                                            *
//* Input  : fKm : kilometers                                                  *
//*          fKph: kilometers-per-hour                                         *
//*                                                                            *
//* Returns: minutes                                                           *
//******************************************************************************

float Exercalc::Kilometers2Minutes ( float fKm, float fKph )
{
   float fTotal = fKm / fKph ;               // floating-point hours-and-minutes
   short iHours = short(fTotal) ;            // integer hours
   float fMin   = float(iHours) * 60.0 ;     // minutes in integer hours
   float fFrac  = (fTotal - float(iHours)) * 60.0 ; // minutes in fractional hour
   short iMin   = short(fFrac) ;             // truncate to even minute
   fMin += float(iMin) ;

   return fMin ;

}  //* End Kilometers2Minutes() *

//*************************
//*  Kilometers2Minutes   *
//*************************
//******************************************************************************
//* Given the velocity and the elapsed time, calculate the distance travelled. *
//*                                                                            *
//* Input  : fMin: elapsed time in minutes                                     *
//*          fKph: kilometers-per-hour                                         *
//*                                                                            *
//* Returns: kilometers                                                        *
//******************************************************************************

float Exercalc::Minutes2Kilometers ( float fMin, float fKph )
{

   return ( (fKph / 60.0) * fMin ) ;

}  //* End Minutes2Kilometers() *

//*************************
//*  Inches2Centimeters   *
//*************************
//******************************************************************************
//* Convert Convert Inches to Centimeters.                                     *
//*                                                                            *
//* Input  : fIn : length in inches                                            *
//*                                                                            *
//* Returns: length in centimeters                                             *
//******************************************************************************

float Exercalc::Inches2Centimeters ( float fIn )
{

   return ( fIn * Cm_per_In ) ;

}  //* End Inches2Centimeters() *

//*************************
//*  Centimeters2Inches   *
//*************************
//******************************************************************************
//* Convert Convert Inches to Centimeters.                                     *
//*                                                                            *
//* Input  : fIn : length in centimeters                                       *
//*                                                                            *
//* Returns: length in inches                                                  *
//******************************************************************************

float Exercalc::Centimeters2Inches ( float fCm )
{

   return ( fCm * In_per_Cm ) ;

}  //* End Centimeters2Inches() *

//*************************
//*   Pounds2Kilograms    *
//*************************
//******************************************************************************
//* Convert Pounds to Kilograms.                                               *
//*                                                                            *
//* Input  : fLb : mass in avoirdupois pounds                                  *
//*                                                                            *
//* Returns: mass in kilograms                                                 *
//******************************************************************************

float Exercalc::Pounds2Kilograms ( float fLb )
{

   return ( fLb * Kg_per_Lb ) ;

}  //* End Pounds2Kilograms() *

//*************************
//*   Kilograms2Pounds    *
//*************************
//******************************************************************************
//* Convert Kilograms to Pounds.                                               *
//*                                                                            *
//* Input  : fKg : mass in kilograms                                           *
//*                                                                            *
//* Returns: mass in avoirdupois pounds                                        *
//******************************************************************************

float Exercalc::Kilograms2Pounds ( float fKg )
{

   return ( fKg * Lb_per_Kg ) ;

}  //* End Kilograms2Pounds() *

//*************************
//*   ApplyMETsFactors    *
//*************************
//******************************************************************************
//* Adjust a standardized METs value for various individualized factors.       *
//*                                                                            *
//*  1) BMI (body mass index)                                                  *
//*     Research has shown that the ratio of body fat to lean body mass is     *
//*     the most significat factor when errors between the standardized METs   *
//*     value and impirical data for an individual are observed.               *
//*     The values in the METs table tend to be 20% to 30% too high on         *
//*     average (especially for people who are overweight).                    *
//*                                                                            *
//*  2) Age                                                                    *
//*     As a person ages, his or her basal metabolic rate changes in subtle    *
//*     and not-so-subtle ways. For this reason, age is a factor in determining*
//*     the accuracy of a METs value from the standard tables.                 *
//*                                                                            *
//*  3) Gender                                                                 *
//*     Due to statistical differences in muscle mass and body fat between     *
//*     the male and female body, gender is a factor; however, it seems        *
//*     likely that these differences are adequately addressed by application  *
//*     of adjustments for BMI.                                                *
//*                                                                            *
//* Input  : fMets : a METs value from the standard lookup table               *
//*                                                                            *
//* Returns: adjusted METs                                                     *
//******************************************************************************

float Exercalc::ApplyMETsFactors ( float fMets )
{
// UNDER CONSTRUCTION - TWEAK THE METs VALUE FROM THE TABLE TO TAKE THE 'BMI' INTO CONSIDERATION.
// THIS IS PROBABLY A LOG FUNCTION OF SOME KIND.
// NEEDS RESEARCH. https://www.nature.com/articles/ijo201422.pdf?origin=ppub
// THIS ARTICLE DESCRIBES CORRECTION FACTORS FOR OVERWEIGHT SUBJECTS. (DOWNLOADED)

   float adjMets = fMets ;

   return adjMets ;

}  //* End ApplyMETsFactors() *

//*************************
//*      MetMinutes       *
//*************************
//******************************************************************************
//* Calculate METs burned during activity.                                     *
//* This is the number of METs per minute for the activity time the duration   *
//* of the activity in minutes.                                                *
//*                                                                            *
//* Input  : fRatio : ratio of METs expended per unit time compared to sitting *
//*          fMins : (optional, 1.0 by default) duration of exercise in minutes*
//*                                                                            *
//* Returns: METs                                                              *
//******************************************************************************
//* Calculating METs:                                                          *
//*   1 MET is                                                                 *
//*   One min  == 1.0 / 60.0                                                   *
//*   One kCal == 1.0 * 3.5 * kg / 200.0                                       *
//*   One MET  == kCal / (kg * hr)                                             *
//*   One MET  == (1.0 * 3.5 * kg / 200.0) / (kg * 1 / 60.0)                   *
//*     The kg cancel, so:                                                     *
//*   One MET  == (1.0 * 3.5 / 200.0) / (1 / 60.0)                             *
//*     or                                                                     *
//*   One MET  == (1.0 * 3.5 / 200.0) * 60.0                                   *
//*     This results in: 1.05   What is that?  A more accurate MET?            *
//******************************************************************************

float Exercalc::MetMinutes ( float fRatio, float fMins )
{

   return ( fRatio * fMins ) ;

}  //* End MetMinutes() *

//*************************
//*       Mets2kCal       *
//*************************
//******************************************************************************
//* Convert METs to kilocalories.                                              *
//*                                                                            *
//* Input  : fRatio : ratio of METs expended per unit time compared to sitting *
//*          fMass  : body mass in kilograms                                   *
//*          fMins : (optional, 1.0 by default) duration of exercise in minutes*
//*                                                                            *
//* Returns: kilocalories                                                      *
//******************************************************************************

float Exercalc::Mets2kCal ( float fRatio, float fMass, float fMins )
{

   return ( (fRatio * 3.5 * fMass / 200.0) * fMins ) ;

}  //* End Mets2kCal ()

//*************************
//*    kCal2Kilograms     *
//*************************
//******************************************************************************
//* Convert kilocalories burned to kilograms of fat burned.                    *
//*                                                                            *
//* Input  : fKcal : number of kilocalories                                    *
//*                                                                            *
//* Returns: kilograms                                                         *
//******************************************************************************

float Exercalc::kCal2Kilograms ( float fKcal )
{

   return ( (fKcal / kCal_per_Gm) / 1000.0 ) ;

}  //* End kCal2Kilograms() *

//*************************
//*     kCal2Minutes      *
//*************************
//******************************************************************************
//* Convert kCal burned to minutes of activity.                                *
//*                                                                            *
//*                                                                            *
//* Input  : fKcal  : number of kilocalories burned                            *
//*          fRatio : METs ratio for specific activity type                    *
//*          fMass  : body mass                                                *
//*                                                                            *
//* Returns: number of minutes                                                 *
//******************************************************************************

float Exercalc::kCal2Minutes ( float fKcal, float fRatio, float fMass )
{
   float fMins = 0.0 ;     // return value

   fMins = fKcal / ((fRatio * 3.5 * fMass) / 200.0) ;

   return fMins ;

}  //* End kCal2Minutes() *

//*************************
//*     Time2Minutes      *
//*************************
//******************************************************************************
//* Convert clock time in hours and minutes to minutes.                        *
//*                                                                            *
//* Input  : iHr : hours                                                       *
//*          iMn : minutes                                                     *
//*                                                                            *
//* Returns: minutes                                                           *
//******************************************************************************

float Exercalc::Time2Minutes ( short iHr, short iMn )
{

   return ( float(iHr) * 60.0 + float(iMn) ) ;

}  //* End Time2Minutes() *

//*************************
//*     Time2Minutes      *
//*************************
//******************************************************************************
//* Convert a clock time string in the format: "hh:mm" OR any integer-value    *
//* string to minutes.                                                         *
//*                                                                            *
//* If source contains a group of concatenated values (separated by '+'),      *
//* then process only the first value of the list.                             *
//*                                                                            *
//* NOTE: Decimal values must not be used to specify time.                     *
//*       If user specifies time as a decimal value e.g. 3.5, this WILL NOT    *
//*       be interpreted as 3.5 hours. It will instead be interpreted as       *
//*       3.5 minutes, which caller will truncate to 3 minutes.                *
//*                                                                            *
//* Input  : cTime : time string                                               *
//*                                                                            *
//* Returns: minutes  (if invalid numeric string, returns 0.0)                 *
//******************************************************************************

float Exercalc::Time2Minutes ( const char* cTime )
{
   gString gs( cTime ) ;
   short indx = gs.find( wPLUS ) ;
   float fMin = 0.0, fHtmp, fMtmp ;

   //* If concatenated values, isolate the first value *
   if ( indx > ZERO )
      gs.limitChars( indx ) ;

   //* Is value in HH:MM format? *
   indx = gs.find( wCOLON ) ;

   if ( indx > ZERO )
   {
      if ( (gs.gscanf( "%f:%f", &fHtmp, &fMtmp )) == 2 )
         fMin = fHtmp * 60.0 + fMtmp ;
   }
   else if ( indx == ZERO )
   {
      gs.shiftChars ( -1 ) ;     // discard leading colon
   }
   if ( indx <= ZERO )
   {  // (see note above about time format)
      if ( (gs.gscanf( "%f", &fMtmp )) == 1 )
         fMin = fMtmp ;
   }
   return fMin ;

}  //* End Time2Minutes() *

//*************************
//*     Minutes2Time      *
//*************************
//******************************************************************************
//* Convert minutes to clock time in hours and minutes.                        *
//*                                                                            *
//* Input  : fMn   : total number of minutes                                   *
//*          iHr   : (by reference) receives number of full hours              *
//*          iMn   : (by reference) receives number of minutes (modulo 60)     *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Exercalc::Minutes2Time ( float fMn, short& iHr, short& iMn )
{

   iHr = iMn = ZERO ;               // discard current values
   iHr = short(fMn / 60.0) ;        // full hours
   iMn = short(fMn - iHr * 60.0) ;  // minutes remaining

}  //* End Minutes2Time() *

//*************************
//*  DecodeLength2Meters  *
//*************************
//******************************************************************************
//* Extract a length value from input string and return the value expressed    *
//* in meters.                                                                 *
//* Units may be in:                                                           *
//*    meters     : 1.80m                                                      *
//*    centimeters: 180.0cm                                                    *
//*    inches     : 73.5in                                                     *
//*    If no units specified, centimeters is assumed.                          *
//*                                                                            *
//* Input  : gsInput : (by reference) contains string data to be converted     *
//*          meterVal: (by reference) receives the converted value             *
//*                    (0.0 returned if conversion fails)                      *
//*                                                                            *
//* Returns: 'true' if value converted, else 'false'                           *
//*                                                                            *
//******************************************************************************

bool Exercalc::DecodeLength2Meters ( const gString& gsInput, float& meterVal )
{
   float fIn ;
   bool status = false ;

   meterVal = 0.0 ;     // initialize caller's variable

   if ( (gsInput.gscanf( " %f", &fIn )) == 1 )
   {
      //* Test for units *
      if ( (gsInput.find( "cm" )) > ZERO )      // centimeters
         meterVal = fIn / 100.0 ;
      else if ( (gsInput.find( "m" )) > ZERO )  // meters
         meterVal = fIn ;
      else if ( (gsInput.find( "in" )) > ZERO ) // inches
         meterVal = this->Inches2Centimeters ( fIn ) / 100.0 ;
      else                                      // units not specified (assume cm)
         meterVal = fIn / 100.0 ;
      status = true ;
   }
   return status ;

}  //* End DecodeLength2Meters() *

//*************************
//* DecodeMass2Kilograms  *
//*************************
//******************************************************************************
//* Extract a mass value from input string and return the value expressed      *
//* in kilograms.                                                              *
//* Units may be in:                                                           *
//*    kilograms  : 64.85kg                                                    *
//*    pounds     : 138.45lb                                                   *
//*    If no units specified, kilograms is assumed.                            *
//*                                                                            *
//* Input  : gsInput : (by reference) contains string data to be converted     *
//*          kiloVal : (by reference) receives the converted value             *
//*                    (0.0 returned if conversion fails)                      *
//*                                                                            *
//* Returns: 'true' if value converted, else 'false'                           *
//*                                                                            *
//******************************************************************************

bool Exercalc::DecodeMass2Kilograms ( const gString& gsInput, float& kiloVal )
{
   float fIn ;
   bool status = false ;

   kiloVal = 0.0 ;      // initialize caller's variable

   if ( (gsInput.gscanf( " %f", &fIn )) == 1 )
   {
      //* Test for units *
      if ( (gsInput.find( "lb" )) > ZERO )
         kiloVal = this->Pounds2Kilograms ( fIn ) ;
      else
         kiloVal = fIn ;   // mass in kilograms
      status = true ;
   }
   return status ;

}  //* End DecodeMass2Kilograms() *

//*************************
//*  DecodeVelocity2Kph   *
//*************************
//******************************************************************************
//* Extract velocity value from input string and return the value expressed    *
//* in kilometers-per-hour.                                                    *
//* Units may be in:                                                           *
//*    kph    : 12.53kph                                                       *
//*    mph    : 7.790mph                                                       *
//*    If no units specified, KPH is assumed.                                  *
//*                                                                            *
//* Input  : gsInput : (by reference) contains string data to be converted     *
//*          kphVal  : (by reference) receives the converted value             *
//*                    (0.0 returned if conversion fails)                      *
//*                                                                            *
//* Returns: 'true' if value converted, else 'false'                           *
//*                                                                            *
//******************************************************************************

bool Exercalc::DecodeVelocity2Kph ( const gString& gsInput, float& kphVal )
{
   float fIn ;
   bool status = false ;

   kphVal = 0.0 ;       // initialize caller's variable

   if ( (gsInput.gscanf( " %f", &fIn )) == 1 )
   {
      if ( (gsInput.find( "mph" )) > ZERO )
         kphVal = this->Miles2Kilometers ( fIn ) ;
      else
         kphVal = fIn ;    // velocity in KPH
      status = true ;
   }
   return status ;

}  //* End DecodeVelocity2Kph() *

//*************************
//*      UserMessage      *
//*************************
//******************************************************************************
//* Write an informational message to the user dialog.                         *
//*                                                                            *
//* Input  : msgText : text data (gString, wchar_t* or char*)                  *
//*          msgColor: text color attribute                                    *
//*          centered: (optional, 'false' by default)                          *
//*                    if specified, center the message on the display line    *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Exercalc::UserMessage ( const gString& msgText, attr_t msgColor, bool centered )
{
   short xoffset = this->userArea.xpos + 
                   (this->cfg.rtl ? (this->userCols - 2) : 1) ;

   if ( centered )
   {
      if ( this->cfg.rtl )
         xoffset = this->userCols / 2 + (msgText.gscols() / 2 + 1);
      else
         xoffset = this->userCols / 2 - (msgText.gscols() / 2 + 1) ;
   }
   this->userDlg->ClearLine ( this->userMsg ) ;
   this->userDlg->WriteString ( this->userMsg, xoffset, msgText, msgColor, 
                                true, this->cfg.rtl ) ;

}  //* End UserMessage() *

void Exercalc::UserMessage ( const wchar_t* msgText, attr_t msgColor, bool centered )
{

   gString gs( msgText ) ;
   this->UserMessage ( gs, msgColor, centered ) ;

}  //* End UserMessage() *

void Exercalc::UserMessage ( const char* msgText, attr_t msgColor, bool centered )
{

   gString gs( msgText ) ;
   this->UserMessage ( gs, msgColor, centered ) ;

}  //* End UserMessage() *

//*************************
//*    DisplayVersion     *
//*************************
//******************************************************************************
//* Print application version number and copyright notice to tty (bypassing    *
//* NCurses output).                                                           *
//* The NCurses engine has been shut down (or didn't start) so we use simple   *
//* console I/O to display the version/copyright text.                         *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Exercalc::DisplayVersion ( void )
{
   const short fsWidth = 68 ;
   static const char* const freeSoftware = 
    "License GPLv3+: GNU GPL version 3 <http://gnu.org/licenses/gpl.html>\n"
    "This is free software: you are free to modify and/or redistribute it\n"
    "under the terms set out in the license.\n"
    "There is NO WARRANTY, to the extent permitted by law.\n" ;

   gString gsOut( titleTemplate, AppTitle, AppVersion, crYears ) ;
   gsOut.append( wNEWLINE ) ;
   gsOut.padCols( ((gsOut.gscols()) + fsWidth), L'=' ) ;
   wcout << wNEWLINE << gsOut.gstr() << wNEWLINE << freeSoftware << endl ;

}  //* End DisplayVersion() *

//*************************
//*    DisplayHelp        *
//*************************
//******************************************************************************
//* Print command-line help to tty (bypassing NCurses output).                 *
//* The NCurses engine has been shut down (or didn't start) so we use simple   *
//* console I/O to display the help text.                                      *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//*                                                                            *
//* Programmer's Note: Multilingual Help cannot be written to stdout unless we *
//* replace the 'C' language (ASCII) locale with a true UTF-8 locale.          *
//* -- Since this application is an exercise in constructing a multilingual    *
//*    user interface, a first step is to dynamically translate the quick help *
//*    based on information from the terminal environment.                     *
//* -- Unfortunately, this doesn't work correctly _unless_ the text is first   *
//*    converted to 'wide' characters. We _could have_ simply defined all the  *
//*    text as wchar_t data, but that would bloat the binary with little       *
//*    benefit. Instead, we define the text as char data and convert it before *
//*    writing to stdout. This should not be necessary, but is a quirk of the  *
//*    console design.                                                         *
//* -- Once we are inside the ncurses environment, the entire character set    *
//*    becomes available because the NcDialog API is specifically designed for *
//*    a multilingual user interface.                                          *
//*                                                                            *
//* Adding a new UI language to this method:                                   *
//* ========================================                                   *
//* 1) Enable the two debugging definitions.                                   *
//* 2) Explicitly set the system locale if it is not already the one you want. *
//* 3) Add a test to match the language to the locale name.                    *
//* 4) Translate the descriptions for the command-line options.                *
//* 5) Divide the text among the four Help[] arrays, verifying that the total  *
//*    character count for each array fits comfortably within a gString object *
//*    i.e. 1024 (wchar_t) characters.                                         *
//* 6) Disable the debugging definitions.                                      *
//******************************************************************************

void Exercalc::DisplayHelp ( void )
{
   #define DEBUG_QHELP (0)
   #define DEBUG_QHELP_COUNT (0)

const short sections = 8 ;
const char* const Help[LangCount][sections] = 
{
//** English **
 {
 // Section 1
  //12345678901234567890123456789012345678901234567890123456789012345678901234567890
 "\nUSAGE: ecalc [OPTIONS]"
 "\n       All options are optional, and may be specified in any order."
 "\n       All options are case sensitive."
 "\n       Options with single-character arguments may be concatenated."
 "\n         Examples: ecalc -im=38.5 -gw    ecalc -tgih=2:25"
 "\n       For options which are not specified, the value will be taken from the"
 "\n       configuration file or will be calculated from available data."
 "\n ",

 // Section 2
 "\n --height=VALUE  Specify your height in meters (or centimeters or inches)."
 "\n       If no units specified, centimeters are assumed."
 "\n       Ex: --height=178.5cm  --height=178.5  --height 1.785m  --height=70.3in"
 "\n --mass=VALUE  Specify your mass (weight) in kilograms (or pounds)."
 "\n       If no units specified, kilograms are assumed."
 "\n       Ex: --mass=74.0  --mass 63.5kg  --mass=163.2lb",

 // Section 3
 "\n --age=VALUE  Your age in years. Used to apply METs value corrections."
 "\n              Ex: --age=22    (currently ignored)"
 "\n --gender=[m | f | o]  Your gender, (male / female / other)."
 "\n              Used to apply METs value corrections. (currently ignored)"
 "\n ",

 // Section 4
 "\n -i   [m | k | h | c]=VALUE  Input value"
 "\n       Where: m = Miles, k = Kilometers, h = Hours, c = kiloCalories"
 "\n       Miles, Kilometers and Calories may be specified as a decimal value:"
 "\n          \"12.50\" or \"4.5\" or \"300\""
 "\n       Time may be specified as:"
 "\n          hours and minutes: \"hh:mm\" or minutes only: \"mmm\""
 "\n       Multiple values may be concatenated, separated by a plus sign ('+')."
 "\n       These values will be summed, yielding the total exercise period"
 "\n       or distance.   Ex: -im=3.5+7+4.2+3.4+5.0",

 // Section 5
 "\n -t   [w | r | b | g]  Type of exercise"
 "\n       Where: w = walking (default), r = running, b = bicycling,"
 "\n              g = general (weights, yoga, stretching, etc.)"
 "\n -v   [w | r | b]=VALUE Specify your average velocity for the type of activity:"
 "\n       Where: w = walking, r = running, b = bicycling"
 "\n       Specify the parameters in either KPH or MPH. If no units specified, KPH"
 "\n       is assumed. Ex: -vw=4  -vr 5.70kph  -vb=12.25mph"
  //12345678901234567890123456789012345678901234567890123456789012345678901234567890
 "\n -g   [d | w]  Display exercise Goal, where: d = daily (default), w = weekly"
 "\n      On startup, display your \"Exercise Goal\" record as specified in the"
 "\n      configuration file. The input record (-i option), if specified, is"
 "\n      initially obscured. NOTE: If the '-i' option is not specified, then"
 "\n      display of the goal record is the default action."
 "\n ",

 // Section 6
 "\n --cfg=PATH  Path/filename of alternate configuration file."
 "\n             Default: \"ecalc.cfg\" in same directory as application file."
 "\n --log=PATH  Path/filename of target log file. Default: \"./ecalc_log.txt\""
 "\n             Ex: $HOME/Documents/Exercise/DailyExercise.txt"
 "\n             Ex: ~/Documents/Exercise/DailyExercise.html"
 "\n -a=[e | d] Automatic log file save on exit: e = enable, d = disable (default).",

 // Section 7
 "\n -C=COLOR  Color scheme for application. (default: brown)"
 "\n       Colors: [black|red|green|brown|blue|magenta|cyan|gray][,reverse]"
 "\n -l=LANGUAGE User interface language. (set from locale by default)"
 "\n       Languages: Español, Zhōngwén (中文), TiếngViệt, English"
 "\n          Ex:  -l=Zh   (Note: two characters to uniquely identify language.)"
 "\n -L=LOCALE  Specify an alternate locale (must support UTF-8 encoding)"
 "\n            (taken from environment by default)  Ex: -L=zh_CN.utf8",

 // Section 8
 "\n -m=[e | d] Mouse support: e = enable. d = disable (default)."
 "\n -p[v] Pause to display startup diagnostics before entering main program."
 "\n --version        Display application version number, then exit."
 "\n --help, -h, -?   Help for command line options."
 "\n",
 },

//** Espanol **
 {
 // Section 1
  //12345678901234567890123456789012345678901234567890123456789012345678901234567890
 "\nUSO: ecalc [OPCIONES]"
 "\n       Todas las opciones son opcionales y pueden especificarse en cualquier"
 "\n       orden. Todas las opciones distinguen entre mayúsculas y minúsculas."
 "\n       Las opciones con argumentos de un solo carácter se pueden concatenar."
 "\n         Ejemplos: ecalc -im=38.5 -gw    ecalc -tgih=2:25"
 "\n       Para las opciones que no están especificadas, el valor se tomará del"
 "\n       archivo de configuración o se calculará a partir de los datos"
 "\n       disponibles."
 "\n ",

 // Section 2
 "\n --height=VALOR  Especifique su altura en metros (o centímetros o pulgadas)."
 "\n       Si no se especifican unidades, se asumen centimetros."
 "\n       Ejemplo: --height=178.5cm --height=178.5 --height 1.785m --height=70.3in"
 "\n --mass=VALOR  Especifique su masa (peso) en kilogramos (o libras)"
 "\n       Si no se especifican unidades, se asumen kilogramos."
 "\n       Ejemplo: --mass=74.0  --mass 63.5kg  --mass=163.2lb",

 // Section 3
 "\n --age=VALOR  Tu edad en años. Se utiliza para aplicar correcciones de"
 "\n              valores MET.  Ex: --age=22    (actualmente ignorado)"
 "\n --gender=[m | f | o]  Tu género (masculino / femenino / otro)."
 "\n              no en el estereotipo.) Se utiliza para aplicar correcciones"
 "\n              de valores MET. (actualmente ignorado)"
 "\n ",

 // Section 4
 "\n -i   [m | k | h | c]=VALOR  Valor de entrada"
 "\n       Donde: m = Millas, k = Kilómetros, h = Horas, c = kiloCalorías"
 "\n       Las millas, los kilómetros y las calorías se pueden especificar como"
 "\n       un valor decimal:    \"12.50\" o \"4.5\" or \"300\""
 "\n       Los valores de tiempo se pueden especificar como:"
 "\n          horas y minutos: \"hh:mm\" o solo minutos: \"mmm\""
 "\n       Se pueden concatenar varios valores, separados por un signo más ('+')."
 "\n       Estos valores se sumarán, dando como resultado el período total"
 "\n       de ejercicio o la distancia.   Ejemplo: -im=3.5+7+4.2+3.4+5.0"
 "\n -t   [w | r | b | g]  Tipo de ejercicio"
 "\n       Donde: w = caminar (valor por defecto), r = correr, b = bicicleta,"
 "\n              g = general (pesas, yoga, estiramientos, etc.)",

 // Section 5
 "\n -v   [w | r | b]=VALOR  Especifique su velocidad promedio para el tipo de"
 "\n              actividad:   Donde: w = caminar, r = correr, b = bicicleta"
 "\n       Especifique los parámetros en KPH o MPH. Si no se especifican unidades,"
 "\n       se asume KPH.  Ejemplo: -vw=4  -vr 5.70kph  -vb=12.25mph"
 "\n -g   [d | w]  Mostrar objetivo del ejercicio."
 "\n      Donde: d = diario (valor por defecto), w = semanal"
 "\n      Al inicio, muestre su registro \"Objetivo de ejercicio\" como se"
 "\n      especifica en el archivo de configuración. El registro de datos de entrada"
 "\n      (opción -i), si se especifica, se oculta inicialmente. NOTA: Si no se"
 "\n      especifica la opción '-i', la visualización del registro de goles es"
 "\n      la acción predeterminada."
 "\n ",

 // Section 6
 "\n --cfg=RUTA  Ruta/nombre de archivo del archivo de configuración alternativo."
 "\n               Valor por defecto: \"ecalc.cfg\" en el mismo directorio que"
 "\n               el archivo de la aplicación."
 "\n --log=RUTA  Ruta/nombre de archivo del archivo de registro de destino."
 "\n               Valor por defecto: \"./exalc_log.txt\""
 "\n             Ejemplo: $HOME/Documentos/Ejercicio/Ejercicio_diario.txt"
 "\n             Ejemplo: ~/Documentos/Ejercicio/Ejercicio_diario.html"
 "\n -a=[e | d] Guardado automático del archivo de registro al salir:"
 "\n             e = habilitar, d = deshabilitar (valor por defecto).",

 // Section 7
 "\n -C=COLOR  Esquema de colores para la aplicación. (predeterminado: marrón)"
 "\n       Colores: [black|red |green|brown |blue|magenta|cyan|gray][,reverse]"
 "\n                (negro|rojo|verde|marrón|azul|magenta|cian|gris)(,inverso)"
 "\n -l=IDIOMA Idioma de la interfaz de usuario. (establecido desde la configuración"
 "\n           regional del sistema de forma predeterminada)"
 "\n           Idiomas: Español, Zhōngwén (中文), TiếngViệt, English  Ejemplo: -l=Es"
 "\n           (Nota: dos caracteres para identificar el idioma de forma única)."
 "\n -L=LUGAR  Especifique una configuración regional alternativa. (debe admitir "
 "\n           la codificación UTF-8) (tomado del entorno del sistema de forma"
 "\n           predeterminada)  Ejemplo: -L=es_MX.utf8",

 // Section 8
 "\n -m=[e | d] Soporte de mouse: e = habilitar, d = deshabilitar (defecto)."
 "\n -p[v] Haga una pausa para mostrar los diagnósticos de inicio antes"
 "\n       de ingresar al programa principal."
 "\n --version        Muestra el número de versión de la aplicación y luego sal."
 "\n --help, -h, -?   Ayuda para las opciones de la línea de comandos"
 "\n",
 },

 //** Zhongwen **
 {
 // Section 1
  //12345678901234567890123456789012345678901234567890123456789012345678901234567890
 "\n用法: ecalc [选择]"
 "\n       所有选项都是可选的，并且可以以任何顺序指定。"
 "\n       所有选项均区分大小写。 具有单字符参数的选项可以串联在一起。"
 "\n         例子： ecalc -im=38.5 -gw    ecalc -tgih=2:25"
 "\n       对于未指定的选项，该值将从配置文件中获取或从可用数据中计算得出。"
 "\n ",

 // Section 2
 "\n --height=值   以米（或厘米或英寸）为单位指定身高。"
 "\n       如果未指定单位，则假定为厘米。"
 "\n       例子： --height=178.5cm  --height=178.5  --height 1.785m  --height=70.3in"
 "\n --mass=值   以千克（或磅）为单位指定您的质量（重量）。"
 "\n       如果未指定单位，则以千克为单位。"
 "\n       例子： --mass=74.0  --mass 63.5kg  --mass=163.2lb",

 // Section 3
 "\n --age=值  您的年龄（以年为单位）。 用于应用MET值校正。"
 "\n           例子： --age=22    （此时将忽略值）"
 "\n --gender=[m | f | o]  您的性别，（男性 / 女性 / 其他）。"
 "\n                   用于应用MET值校正。 （此时将忽略值）"
 "\n ",

 // Section 4
 "\n -i   [m | k | h | c]=值   输入值"
 "\n       指标：m =英里，k = 公里，h = 小时，c = 千卡"
 "\n       英里，公里和卡路里可以指定为十进制值：  \"12.50\" 或者 \"4.5\" 或者 \"300\""
 "\n       时间可以指定为小时和分钟： \"hh:mm\" 或仅按分钟： \"mmm\""
 "\n       可以串联多个值，并用加号（'+'）分隔。"
 "\n       将这些值相加，得出总运动时间或距离。   例子： -im=3.5+7+4.2+3.4+5.0",

 // Section 5
 "\n -t   [w | r | b | g]  锻炼类型"
 "\n       指标： w = 步行（默认），r = 跑步，b = 骑自行车，"
 "\n              g = 常规（重量，瑜伽，伸展运动等）"
 "\n -v   [w | r | b]=值   指定活动类型的平均速度："
 "\n       指标： w = 步行，r = 跑步，b = 骑自行车"
 "\n       在KPH或MPH中指定参数。 如果未指定单位，则假定为KPH。"
 "\n       例子： -vw=4  -vr 5.70kph  -vb=12.25mph"
 "\n -g   [d | w]  显示你的锻炼目标, 指标： d = 每日（默认）, w = 每周"
 "\n      启动时，显示配置文件中指定的“锻炼目标”。 输入记录（“ -i”选项）"
 "\n     （如果已指定）最初被遮盖。"
 "\n      注意：如果未指定'-i'选项，则显示目标记录是默认操作。"
 "\n ",

 // Section 6
 "\n --cfg=路径  备用配置文件的路径/文件名。"
 "\n             默认值： \"ecalc.cfg\" 与应用程序文件位于同一目录中。"
 "\n --log=路径  目标日志文件的路径/文件名。 默认： \"./ecalc_log.txt\""
 "\n             Ex: $HOME/文件资料/锻炼/日常运动.txt"
 "\n             Ex: ~/文件资料/锻炼/日常运动.html"
 "\n -a=[e | d] 退出时自动保存日志文件：e =启用，d =禁用（默认）。",

 // Section 7
 "\n -C=颜色  应用的配色方案。 （默认值：棕色）"
 "\n       颜色: [black|red |green|brown|blue|magenta|cyan|gray][,reverse]"
 "\n            （黑色 |红色|绿色 |棕色 |蓝色|洋红色 |青色|灰色）（,撤销）"
 "\n -l=语言   用户界面语言。 （默认情况下从系统区域设置设置）"
 "\n       语言: Español, Zhōngwén (中文), TiếngViệt, English"
 "\n          例子：  -l=Zh   （注意：两个字符唯一地标识语言。）"
 "\n -L=地区     指定备用语言环境（必须支持UTF-8编码）"
 "\n            （默认情况下取自环境的值）  例子： -L=zh_CN.utf8",

 // Section 8
 "\n -m=[e | d] 鼠标支持：“ e” =启用。 'd'=禁用（默认）。"
 "\n -p[v] 进入主程序之前暂停显示启动诊断信息。"
 "\n --version        显示应用程序版本号，然后退出。"
 "\n --help, -h, -?   命令行选项的帮助。"
 "\n",
 },

//** TiengViet **
 {
  // Section 1
  //12345678901234567890123456789012345678901234567890123456789012345678901234567890
 "\nSỬ DỤNG: ecalc [LỰA CHỌN]"
 "\n       Tất cả các tùy chọn là tùy chọn và có thể được chỉ định theo bất kỳ "
 "\n       thứ tự nào. Tất cả các tùy chọn đều phân biệt chữ hoa chữ thường."
 "\n       Các tùy chọn với các đối số một ký tự có thể được nối với nhau."
 "\n         Thí dụ: ecalc -im=38.5 -gw    ecalc -tgih=2:25"
 "\n       Đối với các tùy chọn không được chỉ định, giá trị sẽ được lấy từ tệp"
 "\n       cấu hình hoặc sẽ được tính toán từ dữ liệu có sẵn."
 "\n ",

 // Section 2
 "\n --height=GIÁ_TRỊ  Chỉ định chiều cao của bạn bằng mét (hoặc cm hoặc inch)."
 "\n       Nếu không có đơn vị nào được chỉ định, thì centimet được giả định."
 "\n       Thí dụ: --height=178.5cm  --height=178.5  --height 1.785m  --height=70.3in"
 "\n --mass=GIÁ_TRỊ  Chỉ định khối lượng (trọng lượng) của bạn theo kg (hoặc pound)."
 "\n       Nếu không có đơn vị nào được chỉ định, thì kilôgam được giả định."
 "\n       Thí dụ: --mass=74.0  --mass 63.5kg  --mass=163.2lb",

 // Section 3
 "\n --age=GIÁ_TRỊ  Tuổi của bạn tính bằng năm. Được sử dụng để áp dụng hiệu chỉnh"
 "\n                giá trị METs.   Thí dụ: --age=22    (hiện đang bị bỏ qua)"
 "\n --gender=[m | f | o]  Giới tính của bạn, (nam / nữ / khác)."
 "\n                Được sử dụng để áp dụng hiệu chỉnh giá trị METs."
 "\n                (hiện đang bị bỏ qua)"
 "\n ",

 // Section 4
 "\n -i   [m | k | h | c]=GIÁ_TRỊ  Giá trị đầu vào"
 "\n       Chỉ báo: m = Miles, k = Kilômét, h = Giờ, c = kcal"
 "\n       Miles, Ki lô mét và calo có thể được quy định như một giá trị thập phân:"
 "\n          \"12.50\" hoặc là \"4.5\" hoặc là \"300\""
 "\n       Thời gian có thể được chỉ định dưới dạng giờ và phút hoặc chỉ là phút:"
 "\n          \"hh:mm\" hoặc là \"mmm\""
 "\n       Nhiều giá trị có thể được nối với nhau, phân tách bằng dấu cộng ('+')."
 "\n       Các giá trị này sẽ được tính tổng, cho ra tổng thời gian hoặc khoảng"
 "\n       cách tập luyện. Thí dụ: -im=3.5+7+4.2+3.4+5.0"
 "\n -t   [w | r | b | g]  Loại bài tập"
 "\n       Chỉ báo: w = đi bộ (mặc định), r = chạy, b = đi xe đạp,"
 "\n                g = tập thể dục chung (tạ, yoga, kéo căng, v.v.)",

 // Section 5
 "\n -v   [w | r | b]=GIÁ_TRỊ  Chỉ định vận tốc trung bình của bạn cho loại hoạt"
 "\n                động. Chỉ báo: w = đi bộ, r = chạy, b = đi xe đạp"
 "\n       Chỉ định các thông số trong KPH hoặc MPH. Nếu không có đơn vị nào được"
 "\n       chỉ định, KPH được giả định.  Thí dụ: -vw=4  -vr 5.70kph  -vb=12.25mph"
 "\n -g   [d | w]  Hiển thị Mục tiêu bài tập."
 "\n       Chỉ báo: d = hằng ngày (mặc định), w = hàng tuần"
 "\n      Khi khởi động, hiển thị bản ghi \"Mục tiêu bài tập\" của bạn như được chỉ"
 "\n      định trong tệp cấu hình. Bản ghi đầu vào (tùy chọn '-i'), nếu được chỉ"
 "\n      định, ban đầu bị che khuất."
 "\n      LƯU Ý: Nếu tùy chọn '-i' không được chỉ định, thì việc hiển thị bản"
 "\n      ghi bàn thắng là hành động mặc định."
 "\n ",

 // Section 6
 "\n --cfg=CON_ĐƯỜNG  Đường dẫn/tên tệp của tệp cấu hình thay thế."
 "\n             Mặc định: \"ecalc.cfg\" trong cùng một thư mục với tệp ứng dụng."
 "\n --log=CON_ĐƯỜNG  Đường dẫn / tên tệp của tệp nhật ký đích."
 "\n             Mặc định: \"./ecalc_log.txt\""
 "\n             Thí dụ: $HOME/Tài_liệu/Bài_tập/Bài_tập_hàng_ngày.txt"
 "\n             Thí dụ: ~/Tài_liệu/Bài_tập/Bài_tập_hàng_ngày.html"
 "\n -a=[e | d]  Tự động lưu tệp nhật ký khi thoát:"
 "\n             e = kích hoạt, d = vô hiệu hóa (mặc định).",

 // Section 7
 "\n -C=MÀU_SẮC  Phối màu cho ứng dụng. (mặc định: nâu)"
 "\n       Màu sắc:"
 "\n         [black|red|green   |brown|blue    |magenta|cyan   |gray][,reverse]"
 "\n         (đen  |đỏ |xanh lục|nâu  |xanh lam|đỏ tươi|lục lam|xám )(,đảo ngược)"
 "\n -l=NGÔN_NGỮ  Ngôn ngữ giao diện người dùng. (được đặt từ khu vực theo mặc định)"
 "\n        Ngôn ngữ:  Español, Zhōngwén (中文), TiếngViệt, English"
 "\n          Thí dụ:  -l=Vi   (Lưu ý: hai ký tự để xác định ngôn ngữ duy nhất.)"
 "\n -L=KHU_VỰC  Chỉ định một vùng thay thế (phải hỗ trợ mã hóa UTF-8)"
 "\n             (được lấy từ môi trường theo mặc định)  Ex: -L=vi_VN.utf8",

 // Section 8
 "\n -m=[e | d] Hỗ trợ chuột: e = cho phép. d = vô hiệu hóa (mặc định)."
 "\n -p[v]      Tạm dừng để hiển thị chẩn đoán khởi động trước khi"
 "\n            vào chương trình chính."
 "\n --version  Hiển thị số phiên bản ứng dụng, sau đó thoát."
 "\n --help, -h, -?   Trợ giúp cho các tùy chọn dòng lệnh."
 "\n",
 }
} ;

   //* Set locale from 'C' to (it is hoped) the system's UTF-8 locale.*
   #if DEBUG_QHELP == 0    // PRODUCTION
   locale loco( "" ) ;
   #else    // DEBUG_QHELP - SET LANGUAGE IN ENVIRONMENT FOR TESTING TRANSLATIONS
   locale loco( "en_US.utf8" ) ;
   //locale loco( "es_MX.utf8" ) ;
   //locale loco( "zh_CN.utf8" ) ;
   //locale loco( "vi_VN.utf8" ) ;
   #if DEBUG_QHELP_COUNT != 0
   gString gsDebug ;
   short cCnt ;      // character count
   #endif   // DEBUG_QHELP_COUNT
   #endif   // DEBUG_QHELP
   wcout.imbue ( loco.global( loco ) ) ;

   //* 'this->cfg.lang' is not initialized at this point. *
   //* Try to determine language from the locale.         *
   //* If we fail, then no harm done.                     *
   AppLang lang = enLang ;
   std::string st = loco.name() ;
   gString gs( st.c_str() ) ;
   if ( (gs.find( "es" )) == ZERO )
      lang = esLang ;
   else if ( (gs.find( "zh" )) == ZERO )
      lang = zhLang ;
   else if ( (gs.find( "vi" )) == ZERO )
      lang = viLang ;

   //* Display the application title, version and copyright *
   gString gsOut( titleTemplate, AppTitle, AppVersion, crYears ) ;
   gsOut.append( wNEWLINE ) ;
   gsOut.padCols( ((gsOut.gscols()) * 2), L'=' ) ;
   wcout << wNEWLINE << gsOut.gstr() << endl ;

   //* Display quick help in designated language *
   for ( short indx = ZERO ; indx < sections ; ++indx )
   {
      gsOut = Help[lang][indx] ;

      #if DEBUG_QHELP != 0 && DEBUG_QHELP_COUNT != 0
      cCnt = gsOut.gschars() ;
      gsDebug.compose( "\n** indx:%hd cCnt:%hd", &indx, &cCnt ) ;
      wcout << gsDebug.gstr() ;
      #endif   // DEBUG_QHELP && DEBUG_QHELP_COUNT

      wcout << gsOut.gstr() ;
   }
   wcout << endl ;      // flush the output buffer

   #undef DEBUG_QHELP
   #undef DEBUG_QHELP_COUNT

}  //* End DisplayHelp() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

