//******************************************************************************
//* File       : Exercalc.hpp                                                  *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2020-2021 Mahlon R. Smith, The Software Samurai *
//*                 GNU GPL copyright notice below                             *
//* Date       : 07-Jun-2021                                                   *
//* Version    : (see AppVersion string)                                       *
//*                                                                            *
//* Description: Class definition and miscellaneous constant definitions       *
//*              for the Exercalc application.                                 *
//*                                                                            *
//*                                                                            *
//******************************************************************************
//* Copyright Notice:                                                          *
//* This program is free software: you can redistribute it and/or modify it    *
//* under the terms of the GNU General Public License as published by the Free *
//* Software Foundation, either version 3 of the License, or (at your option)  *
//* any later version.                                                         *
//*                                                                            *
//* This program is distributed in the hope that it will be useful, but        *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
//* for more details.                                                          *
//*                                                                            *
//* You should have received a copy of the GNU General Public License along    *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.            *
//*                                                                            *
//*         Full text of the GPL License may be found in the Texinfo           *
//*         documentation for this program under 'Copyright Notice'.           *
//******************************************************************************

//* For debugging only: On startup, display command-line  *
//* and config-file parameters in the Stats dialog window.*
#define DEBUG_SETUP_PARMS (0)

//* For debugging only: Enable the Debug-Options menu.*
#define DEBUG_MENU (1)

//* Use simulated RTL data to debug RTL-language formatting and display.*
#define DEBUG_RTL (1)


//****************
//* Header Files *
//****************
#include "GlobalDef.hpp"   //* NcDialog API family of classes
#include "EcConfig.hpp"    //* Configuration data and dialog definitions, and
                           //* Definitions and data for log-file access
#include "EcLang.hpp"      //* Text data for supported UI languages
#include "Chart.hpp"       //* Definitions for the "Chart" class (summary graph)

//***************
//* Definitions *
//***************

//***********************************************
//* Constant Data:                              *
//***********************************************

//* Application version string. Keep it current! *
const char* const AppVersion = "0.0.01" ;
const char* const crYears = "2020-2021" ;
const char* const AppTitle = "Exercalc - v:" ;
const char* const titleTemplate = "%s%s (c)%s The Software Samurai  " ;

//* Base name for default log file.                     *
//* Filename extension will be either ".txt" or ".html" *
const char* const ecLog = "ecalc_log" ;
const char* const txtExt  = ".txt" ;
const char* const htmlExt = ".html" ;
const NcBaseColors dfltCS = ncbcBR ;   // default color scheme

const wchar_t wSPACE   = L' ' ;
const wchar_t wNEWLINE = L'\n' ;
const wchar_t wCOLON   = L':' ;
const wchar_t wEQUAL   = L'=' ;
const wchar_t wPLUS    = L'+' ;
const wchar_t wHASH    = L'#' ;

const float Km_per_Mi = 0.621371192 ;  // kilometers per international mile
const float Mi_per_Km = 1.609344 ;     // international miles per kilometer
const float Kg_per_Lb = 0.45359237 ;   // kilograms per avoirdupois pound
const float Lb_per_Kg = 2.204622622 ;  // avoirdupois pounds per kilogram
const float In_per_Cm = 0.3937007874 ; // inches per centimeter
const float Cm_per_In = 2.54 ;         // centimeters per inch
const float kCal_per_Gm = 9.0 ;        // kilocalories in one gram of fat
const int64_t oneDay = (60 * 60 * 24) ; // seconds * minutes * hours == one day

//* Unit of measure for input and output values *
enum valUnits : short
{
   vuNone,        // no units specified
   vuMiles,       // miles
   vuKmeters,     // kilometers
   vuMinutes,     // minutes
   vuKcalories,   // kilocalories
   vuPounds,      // pound avoirdupois
   vuKilos,       // kilograms
} ;

//* Type of exercise (for calculating kilocalories burned) *
enum XType : short
{
   xtWalk = ZERO, // walking
   xtBike,        // cycling
   xtRun,         // running
   xtGeneral,     // other type (calculated for free weights)
   xtNone,        // none specified
} ;

//* Interface language options. *
enum AppLang : short
{ 
  enLang = ZERO,  // English
  esLang,         // Español (Spanish)
  zhLang,         // Zhōngwén (中文) (Chinese, simplified)
  viLang,         // Tiếng Việt (Vietnamese)
  locLang         // language taken from locale if supported, else default language
} ;

//* Gender specification *
enum GenOpt : short
{
   goNone,        // not specified, do not apply gender adjustments
   goOther,       // user is not comfortable with binary gender, do not apply gender adjustments
   goMale,        // male gender
   goFemale,      // female gender
} ;

//* Command-line option for display of "Goal" record.*
enum ShowGoal : short
{
   sgNoshow,      // do not display goal record
   sgDaily,       // display daily goal record
   sgWeekly,      // display weekly goal record
} ;

//* Arguments to 'Confirm()' method for confirmation type.*
//* Add elements as needed.                               *
enum confType : short
{
   confirmTruncate,     // confirm truncation of log file
} ;

//* Gather and process command-line arguments *
class commArgs
{
   public:
   commArgs ( int argc, char** argv, char** argenv ) :
               argCount(argc), argList(argv), envList(argenv)
   {
      *this->appPath = NULLCHAR ;
      *this->cfgPath = NULLCHAR ;
      *this->logPath = NULLCHAR ;
      this->heightM  = this->massKg  = this->ageY   = 
      this->kphWalk  = this->kphBike = this->kphRun = this->inpValue = 0.0 ;
      this->gender   = goNone ;        // unknown
      this->inpUnits = vuNone ;        // unknown
      this->exerType = xtNone ;        // none specified
      this->showGoal = sgNoshow ;      // default
      this->altLocale[0] = NULLCHAR ;  // none specified
      this->appLanguage = locLang ;    // default
      this->scheme = dfltCS ;          // default
      this->rscheme = false ;          // default
      this->mouse = false ;            // default
      this->logAuto = false ;          // default
      this->diagPause = ZERO ;         // no pause
      this->verFlag = this->helpFlag = false ;
   }
   short    argCount ;           // application's 'argc'
   char**   argList ;            // application's 'argv'
   char**   envList ;            // application's 'argenv'
   char     appPath[MAX_PATH] ;  // directory of application executable
   char     cfgPath[MAX_PATH] ;  // filespec for configuration file
   char     logPath[MAX_PATH] ;  // log-file path
   float    heightM ;            // user's height in meters
   float    massKg ;             // user's body mass in kilograms
   float    ageY ;               // user's age in years
   GenOpt   gender ;             // user's gender (member of GenOpt)
   float    kphWalk ;            // user's average KPH for walking
   float    kphBike ;            // user's average KPH for bicycling
   float    kphRun ;             // user's average KPH for running
   float    inpValue ;           // value to be converted
   valUnits inpUnits ;           // units for input value
   XType    exerType ;           // type of exercise (for calculating kCal values)
   ShowGoal showGoal ;           // display option for user's goal record

   char altLocale[MAX_FNAME] ;   // Locale name
   AppLang appLanguage ;         // Language used for application interface
   NcBaseColors scheme ;         // color scheme indicator
   bool     rscheme ;            // 'true' if color scheme fg/bg reversed
   bool     logAuto ;            // Auto-save log file: 'true'== enable, 'false'==disable
   bool     mouse ;              // Mouse support: 'true'==enable, 'false'==disable
   short    diagPause ;          // -p option: >0 == pause after start-up diagnostics
                                 //    >1 == display verbose start-up diagnostics
   bool     verFlag ;            // --version: 'true' if display command-line version
   bool     helpFlag ;           // -H option: 'true' if display command-line help
} ;


//* Configuration Options structure *
class ConfigOptions
{
   public:

   //* Default constructor *
   ConfigOptions ( void )
   {
      this->reset () ;
   }

   void reset ( void )
   {
      *this->appPath    = NULLCHAR ;   // will be initialized during startup
      *this->tmpPath    = NULLCHAR ;   // will be initialized during startup
      *this->cfgPath    = NULLCHAR ;   // will be initialized during startup
      *this->logPath    = NULLCHAR ;   // will be initialized during startup
      *this->altLocale  = NULLCHAR ;   // receives value (if any) from config file
      this->lang        = locLang ;    // default language from environment (OR English)
      this->logAuto     = false ;      // (disabled by default)
      this->enableMouse = false ;      // (disabled by default)
      this->rtl         = false ;      // (assume an LTR language)
      this->termRows = this->termCols = ZERO ;
      this->scheme = dfltCS ;          // default color scheme
      this->rscheme = false ;          // not reversed
      //* Individual color attributes are unknown at this time *
      this->ub = this->bb = this->sd = this->sb = this->em = this->er = this->mn = 
      this->mf = this->dn = this->df = this->pn = this->pf = this->tn = this->tf = 
      this->tt = attr_t(ZERO) ;
      this->mono[0] = this->mono[1] = attr_t(ZERO) ;
      //* Dropdown-item color attributes must be initialized AFTER   *
      //* NCurses Engine starts. All menu items are initially active.*
      for ( short i = ZERO ; i < spITEMS ; ++i )
         this->dattr[i] = attr_t(ZERO) ;

      //* Menu-item color attributes must be initialized AFTER       *
      //* NCurses Engine starts. All menu items are initially active.*
      for ( short i = ZERO ; i < optITEMS ; ++i )
      {
         this->mattr[i] = attr_t(ZERO) ;
         this->mflag[i] = true ;
      }
   }

   //* DATA MEMBERS (public) *
   char appPath[MAX_PATH] ;   //* Directory of application executable
                              //* default config file, and Help live here
   char tmpPath[MAX_PATH] ;   //* System's temp-file directory
   char cfgPath[MAX_PATH] ;   //* Filespec for configuration file
   char logPath[MAX_PATH] ;   //* Logfile path
   char altLocale[MAX_FNAME] ;//* Locale user-supplied locale (not from environment)
   AppLang lang ;             //* Language used for application interface
   NcBaseColors scheme ;      //* color scheme indicator
   attr_t ub ;                //* user-interface dialog background and border
   attr_t bb ;                //* other dialog backgrounds and borders
   attr_t sd ;                //* sub-dialog color
   attr_t sb ;                //* sub-dialog Bold color
   attr_t em ;                //* sub-dialog Emphasis color
   attr_t er ;                //* emphasis color for 'reverse' bkgnd
   attr_t mn ;                //* menu color (non-focus)
   attr_t mf ;                //* menu color (with focus)
   attr_t dn ;                //* dropdown (non-focus)
   attr_t df ;                //* dropdown (with focus)
   attr_t pn ;                //* pushbutton color (non-focus)
   attr_t pf ;                //* pushbutton color (with focus)
   attr_t tn ;                //* text box color (non-focus)
   attr_t tf ;                //* text box color (with focus)
   attr_t tt ;                //* application's title color
   attr_t dattr[spITEMS] ;    //* dropdown interior color (one attribute for each item)
   attr_t mono[2] ;           //* mono-color for sub-menus
   attr_t mattr[optITEMS] ;   //* menu interior color (one attribute for each item)
   bool   mflag[optITEMS] ;   //* menu-item-active-flags

   short termRows ;           //* Terminal rows
   short termCols ;           //* Terminal columns
   bool  rscheme ;            //* 'true' if color scheme fg/bg reversed
   bool  logAuto ;            //* Logfile auto-save
   bool  enableMouse ;        //* If true, mouse support enabled
   bool  rtl ;                //* Set (true) if user-interface language is RTL
} ;   // ConfigOptions

//* Temp storage for diagnostic messages generated during configuration. *
//* Messages are saved temporarily because configuration occurs before   *
//* the NCurses Engine (specifically, application locale) is initialized.*
const short CFGMSG_COUNT = 32 ;  // records count for dynamic allocation
class CfgMsg
{
   public:
   ~CfgMsg ( void ) {}
   CfgMsg ( void )
   {
      this->reset () ;
   }
   void reset ( void )
   {
      this->msg[0] = '\0' ;
      this->attr = NULL ;
      this->nl = true ;
   }
   //** Data members **
   char   msg[MAX_FNAME] ;    // message text
   attr_t *attr ;             // color attribute
   bool   nl ;                // 'true' if append newline
} ;   // CfgMsg


//* Application's user-provided data and values *
//* calculated from the user-provided data.     *
class UserData
{
   public:
   UserData ( void )          //* default constructor
   {
      this->reset () ;
   }  // constructor

   void reset ( void )
   {
      //* Initialized by data from configuration file *
      //* (or overridden by command-line argument)    *
      this->heightM = this->massKg = this->bmi = this->ageY = 0.0 ;
      this->gender   = goNone ;
      this->kphWalk  = DEFAULT_Kph_Walk ;
      this->kphBike  = DEFAULT_Kph_Bike ;
      this->kphRun   = DEFAULT_Kph_Run ;
      this->metsWalk = DEFAULT_Mets_Walk ;
      this->metsBike = DEFAULT_Mets_Bike ;
      this->metsRun  = DEFAULT_Mets_Run ;
      this->metsGen  = DEFAULT_Mets_Gen ;
      this->metTable = this->metRatio = 0.0 ;

      //* Initialized via command-line options, or        *
      //* calculated from user input. (See Recalculate()) *
      this->inpValue = this->kmeters = this->miles = this->mins = 
      this->kph      = this->metmins = this->kcal = 0.0 ;
      this->exerType = xtNone ;
      this->inpUnits = vuNone ;
      this->timeStamp = ZERO ;
   }

   void operator = ( const UserData& udata )
   {
      this->heightM   = udata.heightM ;
      this->massKg    = udata.massKg ;
      this->bmi       = udata.bmi ;
      this->ageY      = udata.ageY ;
      this->gender    = udata.gender ;
      this->kphWalk   = udata.kphWalk ;
      this->kphBike   = udata.kphBike ;
      this->kphRun    = udata.kphRun ;
      this->metsWalk  = udata.metsWalk ;
      this->metsBike  = udata.metsBike ;
      this->metsRun   = udata.metsRun ;
      this->metsGen   = udata.metsGen ;

      this->inpValue  = udata.inpValue ;
      this->kmeters   = udata.kmeters ;
      this->miles     = udata.miles ;
      this->mins      = udata.mins ;
      this->metTable  = udata.metTable ;
      this->metRatio  = udata.metRatio ;
      this->kph       = udata.kph ;
      this->metmins   = udata.metmins ;
      this->kcal      = udata.kcal ;
      this->exerType  = udata.exerType ;
      this->inpUnits  = udata.inpUnits ;
      this->timeStamp = udata.timeStamp ;
   }  // operator=

   //* Compare data members with members of another object *
   bool operator == ( const UserData& udata )
   {
      bool match = false ;

      if ( (this->heightM   == udata.heightM)  &&
           (this->massKg    == udata.massKg)   &&
           (this->bmi       == udata.bmi)      &&
           (this->ageY      == udata.ageY)     &&
           (this->gender    == udata.gender)   &&
           (this->kphWalk   == udata.kphWalk)  &&
           (this->kphBike   == udata.kphBike)  &&
           (this->kphRun    == udata.kphRun)   &&
           (this->metsWalk  == udata.metsWalk) &&
           (this->metsBike  == udata.metsBike) &&
           (this->metsRun   == udata.metsRun)  &&
           (this->metsGen   == udata.metsGen)  &&

           (this->inpValue  = udata.inpValue)  &&
           (this->kmeters   = udata.kmeters)   &&
           (this->miles     = udata.miles)     &&
           (this->mins      = udata.mins)      &&
           (this->metTable  = udata.metTable)  &&
           (this->metRatio  = udata.metRatio)  &&
           (this->kph       = udata.kph)       &&
           (this->metmins   = udata.metmins)   &&
           (this->kcal      = udata.kcal)      &&
           (this->exerType  = udata.exerType)  &&
           (this->inpUnits  = udata.inpUnits)  &&
           (this->timeStamp = udata.timeStamp) )
      { match = true ; }

      return match ;
   }  // operator==

   //* Configuration data used for calculations *
   float    heightM ;         //* User's height in Meters
   float    massKg ;          //* User's mass in kilograms
   float    bmi ;             //* User's BMI - Body Mass Index
   float    ageY ;            //* User's age in years
   GenOpt   gender ;          //* User's gender (member of GenOpt)
   float    kphWalk ;         //* User's average KPH for walking
   float    kphBike ;         //* User's average KPH for bicycling
   float    kphRun ;          //* User's average KPH for running
   float    metsWalk ;        //* User's intensity while walking (in METs)
   float    metsBike ;        //* User's intensity while bicycling (in METs)
   float    metsRun ;         //* User's intensity while running (in METs)
   float    metsGen ;         //* User's intensity for general exercise (in METs)

   //* User's input parameters and values calculated from that input *
   float    inpValue ;        //* User's input value (see also 'inpUnits')
   float    kmeters ;         //* Distance covered in kilometers
   float    miles ;           //* Distance covered in miles
   float    mins ;            //* Elapsed time in minutes
   float    metTable ;        //* Intensity (std, from table) for current exercise type
   float    metRatio ;        //* Intensity (adjusted for BMI/age/gender etc.)
   float    kph ;             //* Velocity during activity
   float    metmins ;         //* Met-minutes
   float    kcal ;            //* Kilocalories burned
   XType    exerType ;        //* Indicates type of exercise provided by user
   valUnits inpUnits ;        //* Indicates which parameter value is provided by user
   int64_t  timeStamp ;       //* Timestamp for encoded data
} ;   // UserData

//************************************
//*** Application class definition ***
//************************************

class Exercalc
{
   public:

   virtual ~Exercalc ( void ) ;                 //* destructor
   Exercalc ( commArgs& clArgs ) ;              //* constructor

   //* User-interface methods *
   void  UserInterface ( const commArgs& ca ) ; //* Interact with the user.
   bool  NewParameter ( selParm pType ) ;       //* Allow user to modify setup parameters
   bool  GoalRecord ( bool weekly ) ;           //* Generate an "ideal" record that reflects goal
   void  Recalculate ( bool display = true ) ;  //* Calculate member variable values
   void  DisplayMETsTable ( void ) ;            //* Display METs table in the Stats dialog window
   void  CompareRecords ( const UserData& udA, const UserData& udB ) ; //* Compare specified stats groups
   void  crFormatRecord ( const UserData& ud, const winPos& wpBase, const attr_t* dAttr ) ; //* Format the record
   bool  CopyRecord2Clipboard ( void ) ;        //* Copy currently-displayed record to clipboard
   float DecodeInputValues ( const gString& gsIn, valUnits units ) ; //* Decode input argument sequence
   float Miles2Kilometers ( float fMi ) ;       //* Convert Miles to Kilometers
   float Kilometers2Miles ( float fKm ) ;       //* Convert Kilometers to Miles
   float Kilometers2Minutes ( float fKm, float fKph ) ;  //* Calculate elapsed time
   float Minutes2Kilometers ( float fMin, float fKph ) ; //* Calculate distance travelled
   float Inches2Centimeters ( float fIn ) ;     //* Convert Inches to Centimeters
   float Centimeters2Inches ( float fCm ) ;     //* Convert Centimeters to Inches
   float Pounds2Kilograms ( float fLb ) ;       //* Convert Pounds to Kilograms
   float Kilograms2Pounds ( float fKg ) ;       //* Convert Kilograms to Pounds
   float ApplyMETsFactors ( float fMets ) ;     //* Adjust METs value for various factors
   float MetMinutes ( float fRatio, float fMins = 1 ) ; //* Calculate total METs burned
   float Mets2kCal ( float fRatio, float fMass, float fMins = 1 ) ;  //* Convert METs to kilocalories
   float kCal2Kilograms ( float fKcal ) ;       //* Convert kCal burned to kg of fat burned
   float kCal2Minutes ( float fKcal, float fRatio, float fMass ) ; //* Convert kCal burned to minutes of activity
   float Time2Minutes ( short hr, short mn ) ;  //* Convert clock time to minutes
   float Time2Minutes ( const char* cTime ) ;   //* Convert clock time string to minutes
   void  Minutes2Time ( float fMn, short& iHr, short& iMn ) ; //* Convert minutes to clock time
   bool  DecodeLength2Meters ( const gString& gsInput, float& meterVal ) ; //* Extract length value from input string
   bool  DecodeMass2Kilograms ( const gString& gsInput, float& kiloVal ) ; //* Extract mass value from input string
   bool  DecodeVelocity2Kph ( const gString& gsInput, float& kphVal ) ; //* Extract velocity value from input string
   void  UserMessage ( const gString& msgText, attr_t msgColor, bool centered = false ) ;
   void  UserMessage ( const wchar_t* msgText, attr_t msgColor, bool centered = false ) ;
   void  UserMessage ( const char* msgText, attr_t msgColor, bool centered = false ) ;
   void  DisplayVersion ( void ) ;
   void  DisplayHelp ( void ) ;

   //* Setup and configuration methods *
   short Configure ( CfgMsg* cfgMsg, short& cfgMsgs, bool verbose = false ) ;
   void  CommandLineOverrides ( const commArgs& ca ) ; //* Override conf parms with comm-line args
   void  SetColorScheme ( NcBaseColors scheme, bool reverse ) ;
   void  DiagMsg ( const char* msg, attr_t color, bool newLine = true ) ;
   void  ClearWindow ( void ) ;
   void  DrawPicture ( void ) ;
   bool  GetCommandLineArgs ( commArgs& ca ) ;
   void  TermsizeError ( void ) ;
   void  TempdirError ( void ) ;

   //* File-access methods *
   bool  ViewLog ( void ) ;               //* Use 'less' utility to view log file
   bool  LogfileCompose ( const gString&src, gString& trg ) ; //* Normalize log file path
   bool  LogfileCreate ( const gString& logSpec ) ; //* Create/reinitialize log file
   bool  LogfileAppend ( bool confirm = true ) ; //* Write a record to log file
   bool  lfaConfirm ( gString& gsComment ) ; //* User confirmation of append to logfile
   short LogfileFormatRecord ( const UserData& uData, //* Capture and format user data as text
                               const gString* uComm = NULL, bool html = false ) ;
   bool  lfEncodeStats ( const UserData& uData, gString& gsStat, bool html ) ; //* Encode user data
   bool  lfDecodeStats ( const gString& gsStat, UserData& uData, localTime* ltPtr = NULL ) ; //* Decode user data
   short LogfileSummary ( bool display ) ;//* Create and display a summary of log data
   void  LoadRecord ( void ) ;            //* Load a record from the logfile
   void  lfsGraph ( const winPos& wpOrig ) ; //* Display a graph
   bool  RenameFile ( const gString& srcPath, const gString& trgPath ) ; //* Rename source to target
   bool  DeleteFile ( const gString& trgPath ) ;   //* Delete the target file
   bool  Realpath ( gString& rpath ) ;    //* Normalize a filespec
   bool  TargetExists ( const gString& trgPath, fmFType& fType ) ;  //* Test whether target exists
   short GetFileStats ( const gString& trgPath, tnFName& fStats ) ; //* Get target file stats
   bool  GetCWD ( gString& cwd ) ;        //* Get current-working-directory
   bool  GetLocalTime ( localTime& lt ) ; //* Get formatted local time
   bool  GetLocalTime ( int64_t& epoch_time ) ; //* Get local epoch time
   void  DecodeEpochTime ( int64_t eTime, localTime& ftTime ) ; //* Convert epoch time to human time
   bool  GetTempdirPath ( gString& tdPath ) ;   //* Locate temp-file directory
   bool  CreateTempname ( gString& tmpSpec ) ;  //* Create a temporary file
   bool  DeleteTempname ( const gString& tmpSpec ) ; //* Delete a temporary file
   int   LaunchDefaultApplication ( const char* targFile, bool redirect = true ) ;
   void  Cry4Help ( bool htmlFormat ) ;   //* Info or HTML help
   void  HelpAbout ( void ) ;             //* Help-about dialog
   bool  haSupportInfo ( short ulY, short ulX, short dRows, short dCols ) ; //* Tech support info
   bool  Confirm ( confType cfType ) ;    //* Get user confirmation for operation

   //* Utility and debugging methods *
   void  DebugInfo2StatDlg ( const commArgs& ca ) ;
   void  ColorScheme2StatDlg ( void ) ;
   void  ColorMap2StatDlg ( void ) ;
   void  LogfileDebugSample ( void ) ;
   void  DebugRtlExample ( void ) ;
   void  DebugChartWidget ( void ) ;

   //****************
   //* Data Members *
   //****************
   private:

   //* Application setup and housekeeping *
   NcDialog *titleDlg ;       //* Application title dialog
   NcDialog *statsDlg ;       //* Static data dialog
   NcDialog *userDlg ;        //* Dialog for user interaction
   ConfigOptions cfg ;        //* Application configuration options
   winPos  userArea ;         //* Upper-left corner of user data display area
   short   userRows ;         //* Number of display rows in user display area
   short   userCols ;         //* Number of display columns in user display area
   short   userMsg ;          //* Line of user dialog for display of info messages
   winPos  pictArea ;         //* Upper-left corner of picture area
   short   pictRows ;         //* Number of display rows in picture area
   short   pictCols ;         //* Number of display columns in picture area
   winPos  suPos ;            //* Cursor position for start-up messages
   attr_t  suNorm ;           //* Color attributes for start-up messages
   attr_t  suBold ;           //*  (bold color)
   attr_t  suGood ;           //*  (good results)
   attr_t  suError ;          //*  (bad results)
   bool    wcbActive ;        //* 'true' if Wayland clipboard communication established
   bool    logPend ;          //* 'true' if write-to-logfile pending

   //* User Data: Original data collected and calculated during startup.*
   UserData ud ;              //* Active data set
   UserData udb ;             //* Backup copy of original user data set
   char     *udAlloc ;        //* Dynamic memory allocation for formatted output (.txt/.html)
   UserData *udCap ;          //* Data sets captured from log file
   short    udcCount ;        //* Number of records captured to 'udCap' array
                              //* Note: Two(2) additional records, totals/average appended
   float    gValue ;          //* User's goal: value (stored as a daily value)
   float    gMets ;           //* User's goal: intensity (in METs)
   valUnits gUnits ;          //* User's goal: units
   XType    gXType ;          //* User's goal: type of exercise
   bool     gWeek ;           //* User's goal: 'true' if user specified a weekly goal
   bool     gDisp ;           //* User's goal: 'true' if goal data is displayed ('ud' member)

} ;   // Exercalc