//********************************************************************************
//* File       : FMgr.cpp                                                        *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2005-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in FileMangler.hpp         *
//* Date       : 09-Jul-2025                                                     *
//* Version    : (see FMgrVersion string, below)                                 *
//*                                                                              *
//* Description: This class implements a wrapper for Linux system file           *
//* management.  Although C++ has an extensive library of file management        *
//* routines, this is more fun AND more educational, as well as isolating the    *
//* main application from all direct knowledge of file-manipulation routines.    *
//*                                                                              *
//* Developed using GNU G++ (Gcc v: 4.4.2)                                       *
//*  under Fedora Release 12, Kernel Linux 2.6.31.5-127.fc12.i686.PAE            *
//* Compiler and library update 25 March, 2013 (full C++11 support)              *
//*  GNU G++ (Gcc v: 4.8.0 and libstdC++.so.6.0.18)                              *
//*  Fedora Release 16, Kernel Linux 3.6.11-4.fc16.i686.PAE                      *
//********************************************************************************
//* Version History (most recent first):                                         *
//*                                                                              *
//* v: 0.00.26 02-Apr-2025                                                       *
//*   -- Update references to character/byte counts based upon updates to the    *
//*      gString-class definitions.                                              *
//*   -- When allocating arrays of tnFName and TreeNode objects, allocate a      *
//*      few extra records to the array. This was done to increase robustness    *
//*      in the case where a new file or directory may be created _during_       *
//*      the scan of the directory tree.                                         *
//*      -- Handle the scan of the "/proc" directory as a special case.          *
//*         System processes create temporary files in this directory, so the    *
//*         contents of the directory change rapidly.                            *
//*   -- Update Tree-view scan to make it more robust (and faster).              *
//*                                                                              *
//* v: 0.00.25 23-Nov-2020                                                       *
//*   -- Bug Fix: When displaying filesystem information, if the filesystem      *
//*      label contained a space character, the label and mountpoint path were   *
//*      being parsed incorrectly.                                               *
//*   -- Bug Fix: The "ExFat" filesystem, which was until recently proprietary   *
//*      to Microsoft is now being used in an array of new, high-capacity        *
//*      (>32GiB) memory devices such as SD cards and USB flash memory devices.  *
//*      Microsoft, in it's infinite wisdom (i.e. arrogance) has set the         *
//*      cluster size for these devices to 32KiB (131,072 bytes).                *
//*         (Most Microsoft engineers are not qualified to work as)              *
//*         (Walmart greeters, much less in a technical capacity. )              *
//*      The 32KiB size is too large for the 16-bit 'blockSize' and              *
//*      'fblockSize' members of the fileSystemStats class. For this reason,     *
//*      we have increased the size of these members to 32 bits.                 *
//*      Because this class definition is used in a number of applications,      *
//*      the increased size of these data members may affect the storage,        *
//*      retrieval and formatting of these fields. Check your code!              *
//*   -- Enhanced support for optical drives (DVD/CD/Blu-Ray) and for optical    *
//*      media. This includes gathering additional filesystem information on     *
//*      optical-media targets as well as the ability to open and close the      *
//*      optical-drive tray for drives that support software control.            *
//*                                                                              *
//* v: 0.00.24 01-Aug-2020                                                       *
//*   -- Update the hack in EncodeEpochTime() which compensates for a bug in     *
//*      the standard C library timelocal() function. This library bug causes    *
//*      the resulting timestamp to be off by exactly one hour if the function   *
//*      chooses the wrong timezone. This affects the methods which allow the    *
//*      user to directly set the target file's timestamp.                       *
//*                                                                              *
//* v: 0.00.23 05-Jul-2020                                                       *
//*   -- Implement special-purpose code for operations on virtual filesystems.   *
//*      MTP/GVfs filesystems do not respond cleanly to the basic command-       *
//*      utilities (cp, mov, lstat, etc.); therefore, for these virtual          *
//*      filesystems, we use the 'gio' utility and related software to handle    *
//*      view, copy/cut/paste, rename and other common operations.               *
//*      -- Create a new module, FMgrGvfs.cpp to contain the majority of the     *
//*         new code.                                                            *
//*      -- All new methods appear to function correctly. Be aware that these    *
//*         methods are slow by Linux standards because they rely on capturing   *
//*         and interpreting the output of the 'gio' command-line utility.       *
//*         Every effort has been made to move as much of the actual processing  *
//*         as possible onto the Linux system, thus minimizing the time spent    *
//*         waiting for unbelievably-slow "smart" devices to respond.            *
//*      -- Add 'irVirtual' flag as a member of the FMgr class. This flag is     *
//*         initialized by CaptureDirTree() and is referenced by other file      *
//*         operation methods to determine whether special processing ('gio'     *
//*         utility rather than kernel tools) is required for the target         *
//*         filesystem.                                                          *
//*   -- Bug Fix: Integer size mismatch for three members of the ExpStats        *
//*      class. This sometimes caused an incorrect narrowing of data reported.   *
//*   -- Add debugging method UserAlert() which can beep at the interesting      *
//*      part of the code under development.                                     *
//*   -- Disabled the previous hack in EncodeEpochTime() which compensated       *
//*      for an old compiler bug in the timelocal() function when calculating    *
//*      the timezone. The compiler bug seems to have been corrected.            *
//*                                                                              *
//* v: 0.00.22 04-Nov-2018                                                       *
//*   -- Expand the number of threads used to scan the directory tree.           *
//*      -- Generalize the parameters for DirectoryCount() method                *
//*         No change in functionality.                                          *
//*      -- Implement manager-thread target method, sdtMgrthread() to launch     *
//*         the individual sub-threads. This allows the primary thread to        *
//*         immediately return to the user-interface loop (although it           *
//*         voluntarily blocks until all sub-threads signal scan completion.     *
//*      -- Redefine and simplify the thread-target method sdtSubthread().       *
//*         This gives the manager thread more flexibility in assigning          *
//*         sub-threads to the tree scan.                                        *
//*      -- System exceptions for thread allocation and launch are now           *
//*         explicitly "caught" and reported to the user. See DispData class     *
//*         and sdtMgrthread() method.                                           *
//*      -- New data members to manage "intelligent recursion" through the       *
//*         directory tree, which avoids recursion into external filesystems     *
//*         during the scan. See CaptureDirTree() method for details.            *
//*      -- Because scanning the entire directory tree from the root directory   *
//*         ( '/' ) on down is very slow, resource intensive and has the         *
//*         potential to overload the system, we have added the member           *
//*         variable 'irRootScan'.                                               *
//*         -- If 'irRootScan' is reset, then when the CWD is the root           *
//*            directory, the scan will capture ONLY the top level directories.  *
//*         -- When 'irRootScan' is set, then when the CWD is the root           *
//*            directory, the scan will capture the entire directory tree        *
//*            EXCEPT for external filesystems (block devices) mounted as        *
//*            nodes on the tree.                                                *
//*         Public method 'ScanFromRoot()' controls this flag.                   *
//*   -- Consolidate allocation and release of TreeNode-class and tnFName-class  *
//*      objects and other dynamic memory allocations to more closely track      *
//*      dynamic-memory management.                                              *
//*      See TNA_LOG definition in FMgr.hpp.                                     *
//*   -- Make FMgr-class callback method for debugging messages always           *
//*      available. Formerly, it was under a conditional-compile flag.           *
//*      See DebugMsg() method.                                                  *
//*      -- Note that the debugging log is now separate from debugging           *
//*         messages. See DebugLog() and 'ENABLE_FMgrDebugLog' conditional-      *
//*         compile flag.                                                        *
//*   -- Update compiler to gcc v:9.2.1: Warning: -Wdeprecated-declarations      *
//*      readdir64_r() is deprecated. Switch to readdir64() which is             *
//*      now thread-safe under the GNU compiler.                                 *
//*   -- In EscapeSpecialChars() method, test for previously-escaped             *
//*      characters using the full special-character list. This avoids           *
//*      potential double-escaping errors.                                       *
//*   -- In RemoveCharEscapes() method, add optional argument to retain the      *
//*      double-quote character ( " ). The double quote is the filename          *
//*      character most likely to break a call to the shell program.             *
//*   -- Implement a single-threaded, non-recursive scan algorith for            *
//*      smartphone/tablet devices. This scan uses the 'gio' utility (if         *
//*      installed) to scan the top level of the CWD. See GvfsFlatScan()         *
//*      method for details.                                                     *
//*   -- Implement USB_DeviceStats() method. Scans for MTP/GVfs (virtual)        *
//*      filesystems attached to the USB bus. This scan is primarily for         *
//*      locating attached smartphones and tablets, but can optionally scan      *
//*      all USB devices (mouse, touchpad, camera, fingerprint scanner, etc.).   *
//*      This allows reporting of filesystem stats for MTP/GVfs devices.         *
//*                                                                              *
//* v: 0.00.21 15-Sep-2018                                                       *
//*   - Update FileSystemStats() method to include UUID, label, device name      *
//*     and mountpoint.                                                          *
//*   - Add two new methods to the TreeNode class for incrementing the private   *
//*     node and file counters. Used only for clipboard data initialization.     *
//*                                                                              *
//* v: 0.00.20 29-Feb-2016                                                       *
//*   - Implement the SetCWD() method to consolidate the number of different     *
//*     places where the current working directory can change.                   *
//*   - Update the GetCWD() method to make it more robust and to work better     *
//*     with the new SetCWD().                                                   *
//*   - Overload the GetCurrDir() method to return path in a gString object.     *
//*   - Implement GetBaseNode() method to enable scan of directory tree for      *
//*     filenames that match caller's criteria.                                  *
//*   - For the public DirName2TreeNode() method, make the returned value a      *
//*     const TreeNode*. Formerly, was not 'const'.                              *
//*   - Simplify the ExtractPathname(), ExtractFilename() and                    *
//*     ExtractExtension() methods to take advantage of gString::findlast().     *
//*   - Implement the DQuoteEscape() method to enable handling of embedded       *
//*     special characters into filenames. The NcDialog API still won't allow    *
//*     user to enter these characters into a Textbox, but files imported        *
//*     from elsewhere will be handled correctly.                                *
//*   - Implement the Syscall() method to gather all calls to the system()       *
//*     family of C library functions. This reduces clutter, and will            *
//*     optionally return the exit code of the called external program.          *
//*   - Implement: strnCpy ( char* target, const char* source ) ;                *
//*           and: strnCpy ( wchar_t* target, const wchar_t* source ) ;          *
//*     as more efficient replacements for strncpy() and wcsncpy().              *
//*   - Bug Fix: in ExtractPathname() if path was root directory, was returning  *
//*     an empty string.                                                         *
//*   - In EncodeEpochTime() compensate for quirk in the timelocal() function.   *
//*     See note in that method.                                                 *
//*                                                                              *
//* v: 0.00.19 01-Dec-2015                                                       *
//*   - Update all instances of the gString 'formatInt' call to match changes    *
//*     in the gString class.                                                    *
//*   - Add the CreateTempname() method for creating temporary files in a        *
//*     more orderly way. Update methods that use temporary files.               *
//*   - Clean up some prototypes (no change in functionality).                   *
//*   - Update copyright notices.                                                *
//*                                                                              *
//* v: 0.00.18 22-Sep-2013                                                       *
//*   - Update method of calculating the 'readAcc' member of the tnFName class.  *
//*     Previously, we assumed that if we could 'stat' the file, then user had   *
//*     read access, but this is not actually the case. See ReadPermission().    *
//*   - Re-write the BrokenLinkTargetFile() method for a cleaner test.           *
//*   - Initialize 'writeAcc' member of tnFName class as part of the             *
//*     GetFileStats() and GetLinkTargetStats() methods. This simplifies the     *
//*     higher-level code with some loss of efficiency.                          *
//*   - Simplify the ExecutePermission() method.                                 *
//*   - Added a read-access test for all subdirectory names scanned before       *
//*     trying to read directory contents. (This is important because certain    *
//*     system directories are not available even to superuser.)                 *
//*     - This includes the case where user does not have read access to the     *
//*       base directory (CWD), then CaptureDirTree() sets error code to         *
//*       EACCES. This avoids a system-call error caused if we try to read       *
//*       contents of the restricted directory.                                  *
//*   - Fix bug in CdChild() for descending from root directory.                 *
//*   - Replace the clunky FileSizeDisplayFormat() method with the               *
//*     gString-class formatInteger() method.                                    *
//*                                                                              *
//* v: 0.00.17 25-Mar-2013                                                       *
//*   - Begin design of multi-threaded version of reading directory tree.        *
//*     -Multi-threaded capture requires that the compiler and the C++ library   *
//*      fully support the std::thread, std::mutex and std::chrono classes.      *
//*     -Redefine DispData class to protect data members with a mutex, so        *
//*      multiple threads can use the same data. This affects CdChild(),         *
//*      CdParent(), CdPath(), RefreshCurrDir(), and their children.             *
//*      CaptureDirTree() / ScanDirTree() group of methods.                      *
//*     -Update DispData class to use number of display columns, rather than     *
//*      display-string length as input.                                         *
//*     -The CaptureTreeSummary() / TreeSummaryScan() group of methods MAY be    *
//*      modified for multi-threaded scans in a future version.                  *
//*   - Update SetFileTimestamp to use the utime() function instead of the       *
//*     'touch' console command.                                                 *
//*   - Convert MonthString[] array to wide characters.                          *
//*   - Fix nasty bug in AllocateDynamicStorage() method.                        *
//*                                                                              *
//* v: 0.00.16 04-Aug-2012                                                       *
//*   - Enhanced definition of the FMgrConfigOptions class with 'sortOption',    *
//*     'caseSensitive', and 'showHidden' for a smoother instantiation.          *
//*   - Fix bug in DebugMsg().                                                   *
//*                                                                              *
//* v: 0.00.15 23-May-2012                                                       *
//*   - Replace the use of the 32-bit 'struct stat' with the 64-bit              *
//*     'struct stat64' which accomodates file sizes greater than 2*31 bytes     *
//*     on 32-bit file systems. The 'struct dirent' is also replaced by          *
//*     'struct dirent64'. All instances of stat(), lstat() and readdir_r()      *
//*     become stat64() and lstat64() and readdir64_r(), respecively. All use    *
//*     of the inode, filesize, and block count fields of these structures       *
//*     were also updated. These changes were necessary because the size of      *
//*     files, especially video files, DVD image files, etc. are getting         *
//*     larger all the time.                                                     *
//*   - Eliminate redundant fileCount and dirSize.                               *
//*     This info is now in deHead[BNODE].                                       *
//*   - Fixed masked bug (the Lone Disarranger?) in file-sorting methods         *
//*     related to variable overflow.                                            *
//*                                                                              *
//* v: 0.00.14 11-Jan-2012                                                       *
//*   - Minimize use of C-language, ASCII-oriented text manipulation functions.  *
//*   - Modernize the DebugMsg() method to use a callback to the caller who      *
//*     instantiates the FMgr class. See FMgrConfigOptions class definition.     *
//*   - Add method ExecutePermission(). Report, and optionally enable user's     *
//*     exec permissions on a file.                                              *
//*                                                                              *
//* v: 0.00.13 25-Jan-2011                                                       *
//*   - Replace all references to struct FileTime with references to             *
//*     class localTime.                                                         *
//*                                                                              *
//* v: 0.00.12 17-Aug-2010                                                       *
//*   - The fully-integrated TreeNode/tnFName capture of file data as well as    *
//*     clipboard storage is now implemented (except for memory leaks.) See      *
//*     below.                                                                   *
//*   - Create a default constructor and destructor for the tnFName class.       *
//*     There is no actual need for this except the need to to funnel            *
//*     construction and deletion through points which can be monitored to       *
//*     avoid memory leaks. This necessitated changes in the way the memory      *
//*     allocation log file is written. See also the DynaMo utility for          *
//*     parsing the log.                                                         *
//*   - TreeNode and tnFName memory leaks have been eliminated, thanks to the    *
//*     simplified allocation and release design. Of course, they could pop up   *
//*     again in a galaxy far, far away, or maybe even here in Beijing.          *
//*     Run the diagnostic occasionally.                                         *
//*   - DirEntry structure (left over from college days) has now been obsoleted  *
//*     and replace by the TreeNode and tnFName classes.                         *
//*   - All system call errors are now stored in this->recentErrno.              *
//*                                                                              *
//* v: 0.00.11 28-Jun-2010                                                       *
//*   - Rework data members, memory allocation, and methods for sorting files.   *
//*     (major design revision). Memory allocation and release is now through    *
//*     the 'new' and 'delete' keywords rather than through 'calloc()'. This     *
//*     may be slower, but it's more reliable and C++ cool.                      *
//*     Many memory-management methods obsoleted and removed.                    *
//*   - Add new module, FMgrTree.cpp for sub-directory tree traversal and use    *
//*     it to replace the DirEntry linked list format for data storage.          *
//*                   See TreeNode class and tnFName class.                      *
//*   - Replaced DirEntry versions of GetFileStats() and ExpandStats() with      *
//*     tnFName versions.                                                        *
//*                                                                              *
//* v: 0.00.10 27-Apr-2010                                                       *
//*   - Begin integration with FileDlg class. Update CdParent() and CdChild()    *
//*     to be more robust in error recovery.                                     *
//*   - Add method GetErrorCode(), to return 'errno' value.                      *
//*   - Make ExtractFilename() method public.                                    *
//*   - Convert CreateDirectory() to library call mkdir() from system 'mkdir'    *
//*     call.                                                                    *
//*   - Create CopyFifo() method to create a copy of a file of type FIFO.        *
//*     Note that CopyFile() hangs when trying to copy a FIFO file.              *
//*                                                                              *
//* v: 0.00.09 14-Feb-2010                                                       *
//*   - Change development platform and compiler version.                        *
//*     Fedora 12 and GNU G++ (Gcc v: 4.4.2)                                     *
//*     This neccessitated several syntax changes in class and method            *
//*     definitions to accomodate changes in the C++ definition and the GCC      *
//*     compiler's warning and error messages.                                   *
//*   - Add errno.h to list of included headers                                  *
//*                                                                              *
//* v: 0.00.08 23-Mar-2007                                                       *
//*   - Modify CopyFile(). Added optional parameter to control whether source    *
//*     file symbolic links are followed.                                        *
//*   - Fixed bug in FileStats() so if getTarg!=false, it follows all symbolic   *
//*     links in the path to the real filename.                                  *
//*                                                                              *
//* v: 0.00.07 10-Feb-2007 Add methods: CreateSymbolicLink()                     *
//*                        CopyFile(), DeleteFile(), DeleteEmptyDirectory().     *
//* v: 0.00.06 04-Jan-2007 Add methods: SetFilePermissions(),                    *
//*                        SetFileOwnerGroup() and SetFileTimestamp().           *
//* v: 0.00.05 23-Sep-2006 Move MAX_PATH and MAX_FNAME to GlobalDef.hpp.         *
//* v: 0.00.04 14-Jun-2006 Add ExpandStats() method group                        *
//* v: 0.00.03 12-Mar-2006 Add CdPath() method                                   *
//* v: 0.00.02 21-Dec-2005 Add error messages (see DebugMsg())                   *
//* v: 0.00.01 02-Sep-2005 Original release                                      *
//*                        Created using GNU G++ (Gcc v: 3.2.2)                  *
//*                        under RedHat Linux, kernel 2.4.20-31.9                *
//********************************************************************************
//* Programmer's Notes:                                                          *
//*                                                                              *
//* Please note the difference between the two methods GetCWD(() and             *
//* GetCurrDir(). The first makes a system call to retrieve the application's    *
//* current working directory path. This is similar to the 'pwd' command-line    *
//* utility.  The second method simply returns a copy of the path string         *
//* associated with the directory data currently stored in the class data        *
//* structures. The distinction between these two methods is important if more   *
//* than one execution thread is changing directories through the CdParent()     *
//* and CdChild() methods.                                                       *
//*                                                                              *
//*                                                                              *
//********************************************************************************

#include "GlobalDef.hpp"
#ifndef FMGR_INCLUDED
#include "FMgr.hpp"
#endif

//** Local Definitions **

//** Local Data **
static const char FMgrVersion[] = "0.0.26" ; //* Class version number
static const char ParentPath[] = ".." ;      //* Path string for parent directory
static const char ThisPath[] = "." ;         //* Path string for current directory



//*************************
//*        ~FMgr          *
//*************************
//********************************************************************************
//* Destructor.                                                                  *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

FMgr::~FMgr ( void )
{

   //* Release all dynamically allocated memory *
   this->ReleaseDynamicStorage () ;

   //* If dynamic memory not fully released, report it. *
   if ( (this->treenodeObjects > ZERO) || (this->tnfnameObjects > ZERO) )
   {
      gString gs( "GetTreeNode_Allocation( tnObj:%u tnfObj:%u )", 
                  &this->treenodeObjects, &this->tnfnameObjects ) ;
      this->DebugLog ( gs.ustr() ) ;
   }

   #if TNA_LOG != 0     // For debugging only
   //* Used only to test for memory leaks *
   if ( this->tnalogofs.is_open() )
   {
      this->tnalogofs << "\n** End TNA Log **" << endl ;
      this->tnalogofs.close() ;
      this->DebugMsg ( "TNA Log Written", 10 ) ;   // this call waits for a user keypress
      this->DeleteTempname ( this->tnalogFile ) ;  // delete the temp file
   }
   #endif   // TNA_LOG

   //* If debug log file is open, close it now *
   //* and wait for a user keypress, then      *
   //* delete the log file.                    *
   //* If log file NOT open, call does nothing.*
   this->DebugLog ( "" ) ;

}  //* End ~FMgr() *

//*************************
//*         FMgr          *
//*************************
//********************************************************************************
//* Default constructor.                                                         *
//*                                                                              *
//* Input  : cfg    : configuration parameters from caller contains              *
//*                   a) path of temp-file directory                             *
//*                   b) display colors                                          *
//*                   c) startup directory (optional)                            *
//*                   d) various other parameters for reading and displaying     *
//*                      data from the directory tree                            *
//*                                                                              *
//* Returns: nothing (but constructor returns a pointer to class object)         *
//********************************************************************************

FMgr::FMgr ( FMgrConfigOptions& cfg )
{
   //* Initialize data variables *
   this->fileSizeFieldWidth = DFLT_fsFIELD ;
   this->sortOption = fmNAME_SORT ;       // sort displayed files by filename
   this->caseSensitive = true ;           // filename sort is case sensitive
   this->groupDirectories = true ;        // group directory names during sort
   this->recentErrno = ZERO ;             // no error to report
   this->fileSizeCommaFormat = true ;     // comma-format for display of file size
   this->showHidden = false ;             // hidden files are not displayed
   this->deHead      = NULL ;
   this->textData    = NULL ;
   this->colorData   = NULL ;
   this->deList      = NULL ;
   this->rawText     = NULL ;
   this->dlBlockPtr  = NULL ;
   this->tfPath      = cfg.tfPath ;

   this->irID[0]     = NULLCHAR ;         // clear filesystem identifier
   this->irMntpath   = false ;            // assume NOT on a mount path
   this->irOnesys    = true ;             // assume all data for scan in on same filesystem
   this->irVirtual   = false ;            // assume base not not on a virtual filesystem
   this->irRootscan  = false ;            // disable scan from root directory

   for ( USHORT i = ZERO ; i < fmTYPES ; i++ )
      this->colorLookup[i] = cfg.ftColors[i] ;
   this->sortOption = cfg.sortOption ;
   this->caseSensitive = cfg.caseSensitive ;
   this->showHidden = cfg.showHidden ;
   this->irRootscan = cfg.rootScan ;
   if ( cfg.stPath != NULL )     // optional, alternate start-up directory
   {  //* If specified path exists, make it the working directory.*
      // Programmer's Note: This path has already been normalized.
      char tmp[MAX_PATH];
      if ( (realpath ( cfg.stPath, tmp )) != NULL )
         this->SetCWD ( tmp ) ;
   }
   this->GetCWD ( this->currDir ) ;

   //* Get information on application user *
   this->IdentifyUser () ;

   //* Track TreeNode and tnFName objects instantiated/released.*
   this->treenodeObjects = ZERO ;
   this->tnfnameObjects  = ZERO ;

   #if TNA_LOG != 0     //* For debugging only: test for memory leaks.
   this->CreateTempname ( this->tnalogFile ) ;
   this->tnalogofs.open( this->tnalogFile.ustr(), ofstream::out | ofstream::trunc ) ;
   if ( this->tnalogofs.is_open() )
   {
      this->tnalogofs << "TreeNode and tnFName allocation and release log.\n"
                         "------------------------------------------------\n"
                         "     INC\tTN   TFN\n"
                         "     ---\t---  ---"
                         << endl ;
   }
   #endif   // TNA_LOG

   //* Establish callback for display of debugging messages.*
   this->DebugMsgCB = cfg.dmCallback ;

}  //* End FMgr() *

//*************************
//*        SetCWD         *
//*************************
//********************************************************************************
//* Set the current-working directory (CWD). This method does not update the     *
//* data stored in the class members. It simply changes the CWD.                 *
//*                                                                              *
//* Input  : newCwd : path of new directory                                      *
//*          oldCwd : (optional, NULL pointer by default)                        *
//*                   if specified, receives the old (previous) CWD              *
//*                                                                              *
//* Returns: OK  if successful                                                   *
//*          ERR if target directory does not exist or is inaccessible           *
//********************************************************************************

short FMgr::SetCWD ( const gString& newCwd, gString* oldCwd )
{
   //* If specified get the current CWD *
   if ( oldCwd != NULL )
      this->GetCWD ( *oldCwd ) ;

   //* Set the new CWD *
   int rval = chdir ( newCwd.ustr() ) ;
   return ( rval == ZERO ? OK : ERR ) ;

}  //* End SetCWD() *

short FMgr::SetCWD ( const char* newCwd, gString* oldCwd )
{

   gString gs( newCwd ) ;
   return ( (this->SetCWD ( gs, oldCwd )) ) ;

}  //* End SetCWD() *

//*************************
//*       GetCWD          *
//*************************
//********************************************************************************
//* Returns the path of user's current working directory.                        *
//* This is not necessarily the directory associated with the data stored in     *
//* our class members.                                                           *
//*                                                                              *
//* Input  : cdir : pointer to buffer to hold path string                        *
//*                 (buffer must be >= MAX_PATH bytes)                           *
//*                                                                              *
//* Returns: OK if successful                                                    *
//*          ERR if target directory does not exist or is inaccessible           *
//********************************************************************************

short FMgr::GetCWD ( gString& cdir )
{
char     buff[MAX_PATH] ;
short    status = OK ;

   if ( (getcwd ( buff, MAX_PATH )) != NULL )
      cdir = buff ;
   else
   {
      status = ERR ;                // return an error condition
      cdir.clear() ;                // caller's buffer set to empty string
      this->recentErrno = errno ;
      this->DebugLog ( "In GetCWD, getcwd() failed", this->recentErrno ) ;
   }
   return status ;

}  //* End GetCWD() *

short FMgr::GetCWD ( char* cdir )
{
   gString gs ;

   short status = this->GetCWD ( gs ) ;
   gs.copy( cdir, gsDFLTBYTES ) ;

   return status ;

}  //* End GetCWD() *

//*************************
//*      GetCurrDir       *
//*************************
//********************************************************************************
//* Returns a copy of the path string associated with the directory data         *
//* stored in our class data members.                                            *
//*                                                                              *
//* Input  : cdir : pointer to buffer to hold path string                        *
//*                 (must be >= MAX_PATH bytes)                                  *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FMgr::GetCurrDir ( char* cdir )
{

   this->strnCpy ( cdir, this->currDir, MAX_PATH ) ;

}  //* End GetCurrDir() *

void FMgr::GetCurrDir ( gString& cdir )
{

   cdir = this->currDir ;

}  //* End GetCurrDir() *

//*************************
//*      CopyFile         *
//*************************
//********************************************************************************
//* Create a copy of the specified source file.                                  *
//* Preserves all permission bits and date/timestamps.                           *
//*                                                                              *
//* WARNING: Do not call this method to copy a 'special' file. (see below)       *
//*                                                                              *
//* Input  : srcPath : full path/filename specification of source                *
//*          trgPath : full path/filename specification of target                *
//*          followLink: (optional, default==false)                              *
//*                      If source is a symbolic link the following              *
//*                      applies; else the 'followLink' has no effect.           *
//*                      if false, copy the link itself                          *
//*                      if true, copy the file link points to                   *
//*                                                                              *
//* Returns: OK if successful, else returns errno value                          *
//********************************************************************************
//* Programmer's Note: WARNING: Do not call this method to copy a 'special'      *
//*                    file. The cp command opens the pipe and waits forever     *
//*                    for data to be sent through the pipe, OR will refuse to   *
//*                    operate on the file at all - both bad results.            *
//*                                                                              *
//* Programmer's Note: If we are copying a symbolic link, all three file         *
//* dates are set to the CURRENT date/timestamp, ignoring our command            *
//* option to retain all file settings. Using the individual parameters          *
//* instead of the '--preserve=all' option doesn't work since 'links' is not     *
//* recognized in the list, nor is 'links' included in 'all' -- hence:           *
//* '--preserve=all -d'.                                                         *
//* This is not a major problem, but it offends my sense of order.               *
//*                               MRS 25 MAR 2007                                *
//*                                                                              *
//* From the 'cp' man page:                                                      *
//* -----------------------                                                      *
//*    When copying from a symbolic link, `cp' normally follows the link         *
//* only when not copying recursively.  This default can be overridden with      *
//* the `--archive' (`-a'), `-d', `--dereference' (`-L'),                        *
//* `--no-dereference' (`-P'), and `-H' options.  If more than one of these      *
//* options is specified, the last one silently overrides the others.            *
//*    When copying to a symbolic link, `cp' follows the link only when it       *
//* refers to an existing regular file.  However, when copying to a              *
//* dangling symbolic link, `cp' refuses by default, and fails with a            *
//* diagnostic, since the operation is inherently dangerous.  This behavior      *
//* is contrary to historical practice and to POSIX.  Set `POSIXLY_CORRECT'      *
//* to make `cp' attempt to create the target of a dangling destination          *
//* symlink, in spite of the possible risk.  Also, when an option like           *
//* `--backup' or `--link' acts to rename or remove the destination before       *
//* copying, `cp' renames or removes the symbolic link rather than the file      *
//* it points to.                                                                *
//*    By default, `cp' copies the contents of special files only when not       *
//* copying recursively.  This default can be overridden with the                *
//* `--copy-contents' option. [BUT THIS WORKS _ONLY_ WHEN RECURSIVE COPY]        *
//*                                                                              *
//    cp -H src trg    - copy link target                                        *
//                       (cp does this automagically unless in recursive mode )  *
//*                      (and we use cp only for single files, not recursively)  *
//*                                                                              *
//* Valid filename characters:                                                   *
//* --------------------------                                                   *
//* 1) The list of valid filename characters is OS or filesystem specific.       *
//*    Linux/UNIX allow anything except forward slash (/) and NULLCHAR (\0)      *
//*    Ambiguous characters may be used as filename characters _IF_ the are      *
//*    escaped. This application ensures that calls to external programs and     *
//*    and anything else sent to the shell are escaped by enclosing the          *
//*    filespecs in double quotes, but the double-quote delimiters will not      *
//*    escape the following: ( '$' '\' '`' '"' ).                                *
//*    See the EscapeSpecialChars() method for more information.                 *
//*                                                                              *
//* 2) Some filesystems e.g. vfat will not accept all special characters, in     *
//*    which case the copy will fail.                                            *
//*                                                                              *
//********************************************************************************

short FMgr::CopyFile ( const char* srcPath, const char* trgPath, bool followLink )
{
   gString gsSrc( srcPath ), gsTrg( trgPath ) ; // adjusted filespecs
   const short bSIZE = MAX_PATH * 3 ;  // size of command buffer
   char     cmd[bSIZE] ;               // holds command string
   short    status ;                   // return value

   //* Escape any special characters not handled by double-quote delimiters.*
   this->EscapeSpecialChars ( gsSrc, escMIN ) ;
   this->EscapeSpecialChars ( gsTrg, escMIN ) ;

   if ( followLink == false )
      snprintf ( cmd, bSIZE, "cp --preserve=all -d \"%s\" \"%s\" 1>/dev/null 2>/dev/null", 
                             gsSrc.ustr(), gsTrg.ustr() );
   else
      snprintf ( cmd, bSIZE, "cp --preserve=all \"%s\" \"%s\" 1>/dev/null 2>/dev/null", 
                             gsSrc.ustr(), gsTrg.ustr() );

   if ( (status = this->Systemcall ( cmd )) != OK )
   {
      status = this->recentErrno = errno ;         // save 'errno' value
      this->DebugLog ( "In CopyFile: cp failed", status, srcPath ) ;
   }
   return status ;

}  //* End CopyFile() *

//*************************
//*     CopyDirName       *
//*************************
//********************************************************************************
//* Create a copy of the specified Directory-type source file.                   *
//*                                                                              *
//* Directories are a special case in that:                                      *
//*  - if target directory does not exist, create the directory, preserving      *
//*    all the source file's permission bits and both 'modified' and 'access'    *
//*    date/timestamps are set to source file's mod date/time.                   *
//*    - Exception: If a directory is created, then owner (application's user)   *
//*      is given write access on the assumption that the source directory's     *
//*      contents are about to be copied into the new target directory.          *
//*  - if a target file of the same name already exists, do not overwrite it.    *
//*    - if the existing target file is of Directory-type AND user has write     *
//*      access to it, do nothing and report that file was copied successfully.  *
//*    - if the existing target file is of Directory-type BUT user does not      *
//*      have write access to it,                                                *
//*      return 'EACCES - File write protected'                                  *
//*    - if existing target file is not Directory-type,                          *
//*      return 'ENOTDIR - Not a directory'                                      *
//*  - IMPORTANT NOTE:  If source OR target is an MTP/GVfs virtual filesystem,   *
//*    then call CopyDirName_gvfs() instead.                                     *
//*                                                                              *
//* Input  : srcPath : full path/filename specification of source                *
//*          trgPath : full path/filename specification of target                *
//*                                                                              *
//* Returns: OK if successful, else returns errno value                          *
//********************************************************************************
//* Note: Caller must test for MTP/GVfs target filesystem, and call the          *
//*       gvfs version of this method if necessary.                              *
//*                                                                              *
//********************************************************************************

short FMgr::CopyDirName ( const char* srcPath, const char* trgPath )
{
   short status ;                   // return value
   tnFName  srcStats, trgStats ;
   if ( (status = this->GetFileStats ( srcStats, srcPath )) == OK )
   {
      if ( srcStats.fType == fmDIR_TYPE )
      {
         if ( (this->GetFileStats ( trgStats, trgPath )) == OK )
         {  //* Target file (of some type) already exists *
            if ( trgStats.fType == fmDIR_TYPE )
            {
               if ( (this->WritePermission ( trgStats, trgPath )) != false )
                  status = OK ;        // existing target directory with write access
               else
                  status = EACCES ;    // existing target directory write-protected
            }
            else
               status = ENOTDIR ;      // existing target IS NOT a directory
         }
         else
         {  //* Target does not exist - create it *
            if ( (status = CreateDirectory ( trgPath )) == OK )
            {
               //* Adjust permissions and date/times (see note above) *
               mode_t trgMode = (srcStats.rawStats.st_mode | S_IWUSR) ;
               this->SetFilePermissions ( trgPath, trgMode ) ;
               this->SetFileTimestamp ( trgPath, srcStats.modTime, true, true ) ;
            }
         }
      }
      else  // source file is not a directory - caller screwed up
         status = ENOTDIR ;
   }
   return status ;
   
}  //* End CopyDirName() *

//*************************
//*      CopyFifo         *
//*************************
//********************************************************************************
//* Create a copy of the specified FIFO-type source file.                        *
//* Preserves all permission bits and date/timestamps.                           *
//*                                                                              *
//* Input  : srcPath : full path/filename specification of source                *
//*          trgPath : full path/filename specification of target                *
//*                                                                              *
//* Returns: OK if successful, else returns errno value                          *
//********************************************************************************
//* Creating a FIFO is straightforward because it has no special properties.     *
//* Note: FIFO copy not supported for GVfs filesystems, (should not be needed).  *
//********************************************************************************

short FMgr::CopyFifo ( const char* srcPath, const char* trgPath )
{
short    status ;                            // return value

   //* Get file stat informaton, date/timestamp and 'mode'bits for source file *
   tnFName  tnf ;
   if ( (status = this->GetFileStats ( tnf, srcPath )) == OK )
   {
      //* Create a new FIFO with the same permissions as the source *
      if ( (status = mkfifo ( trgPath, tnf.rawStats.st_mode )) == ZERO )
      {
         //* Adjust the mod data and access date to match the source *
         // (an error status could be masked here, but it's extremely unlikely)
         status = this->SetFileTimestamp ( trgPath, tnf.modTime ) ;
         if ( tnf.rawStats.st_atime != tnf.rawStats.st_mtime )
         {
            ExpStats es ;
            es.symbTarg = NULL ;
            this->ExpandStats ( &tnf.rawStats, es ) ;
            status = this->SetFileTimestamp ( trgPath, es.accTime, false ) ;
         }
      }
      else     // error creating file
      {
         status = this->recentErrno = errno ;         // save 'errno' value
         this->DebugLog ( "In CopyFifo: mkfifo failed", status, srcPath ) ;
      }
   }
   return status ;

}  //* End CopyFifo() *

//*************************
//*    CopyCharDevice     *
//*************************
//********************************************************************************
//* Create a copy of the specified CHDEV-type source file.                       *
//* Preserves all permission bits and date/timestamps.                           *
//*                                                                              *
//* Input  : srcPath : full path/filename specification of source                *
//*          trgPath : full path/filename specification of target                *
//*                                                                              *
//* Returns: OK if successful, else returns errno value                          *
//********************************************************************************
//* Creating a CHAR device or BLOCK device requires knowledge of the source      *
//* file's associated device driver AND the unique identifier by which the       *
//* device is known by the device driver.                                        *
//* Note also that creating a CHAR or BLOCK device requires 'root'               *
//* privilege and that 'root' owns the file unless this is explicitly            *
//* changed (see ModifyStats()).                                                 *
//*                                                                              *
//* Note: Char-device copy not supported for GVfs filesystems,                   *
//*       (should not be needed).                                                *
//********************************************************************************

short FMgr::CopyCharDevice ( const char* srcPath, const char* trgPath )
{
short    status ;                            // return value

   //* Get file stat informaton, date/timestamp and 'mode'bits for source file *
   tnFName  tnf ;
   status = this->GetFileStats ( tnf, srcPath ) ;
   if ( status == OK && tnf.fType == fmCHDEV_TYPE )
   {
   }
   else if ( status == OK && tnf.fType != fmCHDEV_TYPE )
   {
      status = ENOTTY ;    // incorect file type
   }
   /* CHAR-DEVICE COPY NOT PERMITTED */ status = EPERM ;
   return status ;

}  //* End CopyCharDevice() *

//*************************
//*   CopyBlockDevice     *
//*************************
//********************************************************************************
//* Create a copy of the specified BKDEV-type source file.                       *
//* Preserves all permission bits and date/timestamps.                           *
//*                                                                              *
//* Input  : srcPath : full path/filename specification of source                *
//*          trgPath: full path/filename specification of target                 *
//*                                                                              *
//* Return : OK if successful, else returns errno value                          *
//********************************************************************************
//* Note: Block-device copy not supported for GVfs filesystems,                  *
//*       (should not be needed).                                                *
//*                                                                              *
//********************************************************************************

short FMgr::CopyBlockDevice ( const char* srcPath, const char* trgPath )
{
short    status ;                            // return value

   //* Get file stat informaton, date/timestamp and 'mode'bits for source file *
   tnFName  tnf ;
   status = this->GetFileStats ( tnf, srcPath ) ;
   if ( status == OK && tnf.fType == fmBKDEV_TYPE )
   {
   }
   else if ( status == OK && tnf.fType != fmBKDEV_TYPE )
   {
      status = ENOTBLK ;    // incorect file type
   }
   /* BLOCK-DEVICE COPY NOT PERMITTED  */ status = EPERM ;
   return status ;

}  //* End CopyBlockDevice() *

//*************************
//*      CopySocket       *
//*************************
//********************************************************************************
//* Create a copy of the specified SOCK-type source file.                        *
//* Preserves all permission bits and date/timestamps.                           *
//*                                                                              *
//* Input  : srcPath : full path/filename specification of source                *
//*          trgPath : full path/filename specification of target                *
//*                                                                              *
//* Returns: OK if successful, else returns errno value                          *
//********************************************************************************
//* Creating a SOCKET, however, is more difficult, because socket creation       *
//* requires knowledge of the source file's 'address domain' or 'NAMESPACE'      *
//* and the 'socket type' or 'STYLE', and for internet sockets, other things     *
//* as well. Determining these attributes for the source file requires opening   *
//* the socket, which is both restricted by the system, and dangerous if the     *
//* device is in use.                                                            *
//* 'address domain' options:                                                    *
//*    AF_LOCAL, also known as the UNIX domain (file-system domain), which       *
//*              means that the address is a file name. (1)                      *
//*    AF_INET,  which is the internet domain (IP address will be assigned       *
//*              later).  (2)                                                    *
//*    There are several more possible address domains, but we don't care.       *
//*    (see <bits/socket.h>)                                                     *
//*                                                                              *
//* 'socket type' options:                                                       *
//*    SOCK_STREAM, character-oriented communication     (1)                     *
//*    SOCK_DGRAM,  message-oriented communication       (2)                     *
//*    SOCK_RAW,    low-level network communication      (3)                     *
//*    (there are other types defined in the header files, but the man page      *
//*     lists only these three for use with the socket(int,int,int) method.)     *
//*                                                                              *
//* ('protocol' always set to ZERO, meaning that protocol will be assigned)      *
//* (when the file is opened for use.                                     )      *
//*                                                                              *
//* Note: Socket copy not supported for GVfs filesystems,                        *
//*       (should not be needed).                                                *
//*                                                                              *
//********************************************************************************

short FMgr::CopySocket ( const char* srcPath, const char* trgPath )
{
short    status ;                            // return value

   //* Get file stat informaton, date/timestamp and 'mode'bits for source file *
   tnFName  tnf ;
   status = this->GetFileStats ( tnf, srcPath ) ;
   if ( status == OK && tnf.fType == fmSOCK_TYPE )
   {
// SEE CreateSocketFile() in FMgrStats.cpp
   }
   else if ( status == OK && tnf.fType != fmSOCK_TYPE )
   {
      status = ENOTSOCK ;    // incorect file type
   }
   /* SOCKET COPY NOT PERMITTED  */ status = EPERM ;
   return status ;

}  //* End CopySocket() *

//*************************
//* CopyUnsupportedType   *
//*************************
//********************************************************************************
//* Create a copy of the specified Unsupported-type file.                        *
//* Preserves all permission bits and date/timestamps.                           *
//*                                                                              *
//* Input  : srcPath : full path/filename specification of source                *
//*          trgPath : full path/filename specification of target                *
//*                                                                              *
//* Returns: OK if successful, else returns errno value                          *
//********************************************************************************
//* For unsupported file types, we do a regular file copy and hope that the      *
//* system on which that type is implemented, has fully-integrated support       *
//* for that type. It is assumed that user has been warned about the             *
//* 'unsupported type' at a higher level, so if we crash and burn it is all      *
//* his/her/its fault.                                                           *
//********************************************************************************
short FMgr::CopyUnsupportedType ( const char* srcPath, const char* trgPath )
{

   return ( this->CopyFile ( srcPath, trgPath ) ) ;

}  //* End CopyUnsupportedType() *

//*************************
//*  CreateSymbolicLink   *
//*************************
//********************************************************************************
//* Create a symbolic link (shortcut file) referencing the source file.          *
//* A symbolic link can reference a file of any type.                            *
//*                                                                              *
//* Input  : srcPath : full path/filename specification of source                *
//*          trgPath : full path/filename specification of target                *
//*                                                                              *
//* Returns: OK if successful, else returns errno value                          *
//*******************************************************************************
//* A symbolic link can even reference another symbolic link (although that      *
//* would be stupid). It makes more sense to create a new symbolic link which    *
//* references the first symlink's target.                                       *
//*                                                                              *
//* A symbolic link can also be created which references a non-existing file.    *
//* This is also generally stupid, though it might sometimes make sense to       *
//* create a symbolic link to a file that will be created later.                 *
//*                                                                              *
//*                                                                              *
//********************************************************************************

short FMgr::CreateSymbolicLink ( const char* srcPath, const char* trgPath )
{
short    status ;

   if ( (status = symlink ( srcPath, trgPath )) != OK )
   {
      status = this->recentErrno = errno ;         // save 'errno' value
      this->DebugLog ( "In CreateSymbolicLink: symlink() failed", status, srcPath ) ;
   }

   return status ;

}  //* End CreateSymbolicLink() *

//*************************
//*    CreateHardLink     *
//*************************
//********************************************************************************
//* Create a 'hard link' to the source file.                                     *
//* A hard link can be created for a file of any type EXCEPT a Directory.        *
//* A hard link to a symbolic-link source makes no sense - don't do it.          *
//* A hard link cannot generally be made across filesystems.                     *
//*                                                                              *
//* Input  : srcPath : full path/filename specification of source                *
//*          trgPath : full path/filename specification of target                *
//*                                                                              *
//* Returns: OK if successful, else returns errno value                          *
//********************************************************************************
//* A hard link is another path to the same file. The target is just as real     *
//* as the souce file, and is in fact the same file. It is not possible to       *
//* determine later which was the 'original' file.                               *
//*                                                                              *
//********************************************************************************

short FMgr::CreateHardLink ( const char* srcPath, const char* trgPath )
{
short    status ;                            // return value

   if ( (status = link ( srcPath, trgPath )) != OK )
   {
      status = this->recentErrno = errno ;         // save 'errno' value
      this->DebugLog ( "In CreateHardLink: link() failed", status, srcPath ) ;
   }
   return status ;

}  //* End CreateSymbolicLink() *

//*************************
//*     DeleteFile        *
//*************************
//********************************************************************************
//* Delete (unlink) the specified source file.                                   *
//*                                                                              *
//* Input  : srcPath : full path/filename specification of source                *
//*                                                                              *
//* Returns: OK if successful, else returns errno value                          *
//********************************************************************************
//* Programmer's Note: The C-language version of 'unlink' INSISTS on doing its   *
//* own escaping of special characters. If it receives escaped characters, it    *
//* escapes the '\' characters, negating the previous escape sequence.           *
//********************************************************************************

short FMgr::DeleteFile ( const char* srcPath )
{
   short    status ;

   if ( (status = unlink ( srcPath )) != OK )
   {
      status = this->recentErrno = errno ;         // save 'errno' value
      this->DebugLog ( "In DeleteFile: unlink() failed", status, srcPath ) ;
   }
   return status ;

}  //* End DeleteFile() *

//************************
//*    DeleteEmptyDir    *
//************************
//********************************************************************************
//* Delete the specified directory. (directory must be empty)                    *
//*                                                                              *
//*                                                                              *
//* Input  : dirPath : full path specification of source                         *
//*                                                                              *
//* Returns: OK if successful, else returns errno value                          *
//********************************************************************************
//* Programmer's Note: The C-language version of 'rmdir' INSISTS on doing its    *
//* own escaping of special characters. If it receives escaped characters, it    *
//* escapes the '\' characters, negating the previous escape sequence.           *
//********************************************************************************

short FMgr::DeleteEmptyDir ( const char* dirPath )
{
   short    status ;

   if ( (status = rmdir ( dirPath )) != OK )
   {
      status = this->recentErrno = errno ;         // save 'errno' value
      this->DebugLog ( "In DeleteEmptyDir: rmdir() failed", status, dirPath ) ;
   }
   return status ;

}  //* End DeleteEmptyDir() *

//*************************
//*      RenameFile       *
//*************************
//********************************************************************************
//* Rename the specified source file to specified destination file.              *
//*                                                                              *
//* For the rename() primitive: If srcPath refers to a symbolic link the         *
//* link is renamed; if trgPath refers to a symbolic link the link will be       *
//* overwritten.                                                                 *
//*                                                                              *
//* Input  : srcPath : full path/filename specification of source                *
//*          trgPath : full path/filename specification of target                *
//*                                                                              *
//* Returns: OK if successful, else returns errno value                          *
//********************************************************************************
//* Programmer's Note: The C-language version of 'rename' INSISTS on doing its   *
//* own escaping of special characters. If it receives escaped characters, it    *
//* escapes the '\' characters, negating the previous escape sequence.           *
//********************************************************************************

short FMgr::RenameFile ( const char* srcPath, const char* trgPath )
{
   short    status ;

   if ( (status = rename ( srcPath, trgPath )) != OK )
   {
      status = this->recentErrno = errno ;         // save 'errno' value
      this->DebugLog ( "In RenameFile: rename() failed", status, srcPath ) ;
   }
   return status ;

}  //* End RenameFile() *

//*************************
//*    RefreshCurrDir     *
//*************************
//********************************************************************************
//* Re-read the current directory, format and sort the display data.             *
//*                                                                              *
//* Input  : dData: pointer to a partially initialized instance of the           *
//*                  DispData class:                                             *
//*                  a) dData.strWidth must contain the number of display        *
//*                     columns for the display strings that will be created.    *
//*                  b) dData.monitorMethod must be initialized in order to      *
//*                     enable the scan progress monitor.                        *
//*                  c) dData.autoRecurse may have been reset to indicate that   *
//*                     target is the mountpoint of a large external disc.       *
//*                                                                              *
//* Returns: If successful, returns 'true' and on return, the remaining          *
//*            members of the DispData class object will have been               *
//*            initialized.                                                      *
//*                                                                              *
//*            Call dData.getActiveThreads() to obtain the number of threads     *
//*             still reading data from the directory tree. When the value       *
//*             returned == ZERO, then entire tree has been read, and the        *
//*             secondary threads rejoined with the main application thread.     *
//*                                                                              *
//*          If operation failed, returns 'false' dData.fileCount == ZERO        *
//*            and remaining members of DispData class object will be            *
//*            NULL pointers and ZERO values.                                    *
//********************************************************************************
//* NOTE: The CaptureDirTree() method captures all files (except "." and "..").  *
//*       If hidden files are not to be reported, we filter them out in this     *
//*       method. In this case, the hidden files are still in the TreeNode       *
//*       list controlled by this->deHead, but they are not referenced or        *
//*       counted in the caller's data arrays.                                   *
//*                                                                              *
//* NOTE: If there are inaccessible files (no read access), those files or       *
//*       directories and contents will not be included in the totals.           *
//*       For example, the contents of some directories may be read only by      *
//*       the superuser, or may be opaque to all users.                          *
//*                                                                              *
//* NOTE: A scan of the directory tree will potentially require several seconds  *
//*       to complete, so the top-level file- and directory names are returned   *
//*       for display while a secondary execution thread continues the scan.     *
//*                                                                              *
//*                                                                              *
//********************************************************************************

bool FMgr::RefreshCurrDir ( DispData& dData )
{
   bool  status = false ;     // return value
   //* Get current state of caller's recursion flag  *
   //* (Recursion is re-enabled by the reset below.) *
   bool  recurse = dData.autoRecursion( true ) ;

   //* Initialize the un-initialized members of caller's DispData class *
   dData.reset() ;

   //* Be sure we are in the directory we think we are. We must do this        *
   //* because external processes (forces of evil?) may have changed the CWD   *
   //* since we last read the directory.                                       *
   gString cwDir( this->currDir ) ;
   this->SetCWD ( cwDir ) ;

   //* If memory resources have been previously allocated for the directory    *
   //* data, release the memory before continuing. Pointers are set to NULL.   *
   this->ReleaseDynamicStorage () ;

   #if 0    // CZONE - FORBID RECURSION INTO A REMOTE FILESYSTEM
   // this->GetFilesystemStats_x ( );
   // if ( fsStats.fsRemote )
   //    recurse = false ;
   /* NOT YET IMPLEMENTED */
   #endif   // U/C

   //* Special Case: If base directory is the root directory OR if CWD is the  *
   //* mountpoint of a large external storage device, a full tree scan will be *
   //* unbearably slow. Therefore, we test for these conditions to determine   *
   //* whether to recurse.                                                     *
   recurse = ( (recurse && 
                (this->irRootscan || ((cwDir.compare( ROOT_PATH )) != ZERO))) ?
                    true :    // 'true' if tree scan is to recurse into all branches
                    false ) ; // 'false' if tree scan is to read only the base node

   //* Read the contents of the directory tree beginning *
   //* at the current working directory (CWD).           *
   if ( (this->deHead = this->CaptureDirTree ( cwDir, dData, recurse )) != NULL )
   {
      //* If the directory is not empty, allocate space for new *
      //* display data. (This includes any "hidden" files.)     *
      if (   this->deHead[BNODE].totFiles > ZERO 
          && (AllocateDynamicStorage ( this->deHead[BNODE].totFiles )) != false )
      {
         //* Create an array of tnFName pointers to reference the data array   *
         bool  hiddenFile ;   // for filtering 'hidden' files
         UINT i = ZERO ;      // index the deList array

         //* For each subdirectory in the 'nextLevel' array of TreeNode objects*
         for ( UINT k = ZERO ; k < this->deHead[BNODE].nlnodes ; k++ )
         {
            hiddenFile = 
               (bool)((this->deHead[BNODE].nextLevel[k].dirStat.fName[0] == PERIOD) 
                                                            ? true : false) ;
            if ( this->showHidden != false || ! hiddenFile )
            {
               this->deList[i++] = &this->deHead[BNODE].nextLevel[k].dirStat ;
            }
            else
            {
               //* This file is not reported to caller, so decrement counters. *
               //* (Prevent going below zero (don't even ask).)                *
               if ( this->deHead[BNODE].totFiles > ZERO )
                  --this->deHead[BNODE].totFiles ;
               if ( this->deHead[BNODE].totBytes >= 
                    this->deHead[BNODE].nextLevel[k].dirStat.fBytes )
                  this->deHead[BNODE].totBytes -= 
                              this->deHead[BNODE].nextLevel[k].dirStat.fBytes ;
               if ( this->deHead[BNODE].dirFiles > ZERO )
                  --this->deHead[BNODE].dirFiles ;
               if ( this->deHead[BNODE].dirBytes >= 
                    this->deHead[BNODE].nextLevel[k].dirStat.fBytes )
                  this->deHead[BNODE].dirBytes -= 
                              this->deHead[BNODE].nextLevel[k].dirStat.fBytes ;
            }
         }
         //* For each file in the 'tnFiles' list *
         for ( UINT k = ZERO ; k < this->deHead[BNODE].tnfcount ; k++ )
         {
            hiddenFile = 
               (bool)((this->deHead[BNODE].tnFiles[k].fName[0] == PERIOD) ? true : false) ;
            if ( this->showHidden != false || ! hiddenFile )
               this->deList[i++] = &this->deHead[BNODE].tnFiles[k] ;
            else
            {
               //* This file is not reported to caller, so decrement counters  *
               if ( this->deHead[BNODE].totFiles > ZERO )
                  --this->deHead[BNODE].totFiles ;
               if ( this->deHead[BNODE].totBytes >= this->deHead[BNODE].tnFiles[k].fBytes )
                  this->deHead[BNODE].totBytes -= 
                                       this->deHead[BNODE].tnFiles[k].fBytes ;
               switch ( this->deHead[BNODE].tnFiles[k].fType )
               {
                  case fmREG_TYPE:
                     --this->deHead[BNODE].regFiles ;
                     this->deHead[BNODE].regBytes -= 
                                    this->deHead[BNODE].tnFiles[k].fBytes ;
                     break ;
                  case fmLINK_TYPE:
                     --this->deHead[BNODE].linkFiles ;
                     this->deHead[BNODE].linkBytes -= 
                                    this->deHead[BNODE].tnFiles[k].fBytes ;
                     break ;
                  case fmCHDEV_TYPE:
                     --this->deHead[BNODE].cdevFiles ;
                     this->deHead[BNODE].cdevBytes -= 
                                    this->deHead[BNODE].tnFiles[k].fBytes ;
                     break ;
                  case fmBKDEV_TYPE:
                     --this->deHead[BNODE].bdevFiles ;
                     this->deHead[BNODE].bdevBytes -= 
                                    this->deHead[BNODE].tnFiles[k].fBytes ;
                     break ;
                  case fmFIFO_TYPE:
                     --this->deHead[BNODE].fifoFiles ;
                     this->deHead[BNODE].fifoBytes -= 
                                    this->deHead[BNODE].tnFiles[k].fBytes ;
                     break ;
                  case fmSOCK_TYPE:
                     --this->deHead[BNODE].sockFiles ;
                     this->deHead[BNODE].sockBytes -= 
                                    this->deHead[BNODE].tnFiles[k].fBytes ;
                     break ;
                  case fmUNKNOWN_TYPE:
                     --this->deHead[BNODE].unkFiles ;
                     this->deHead[BNODE].unkBytes -= 
                                    this->deHead[BNODE].tnFiles[k].fBytes ;
                     break ;
                  default:
                     // unlikely
                     break ;
               }
               --this->deHead[BNODE].tnFCount ; // reduce count of files to be displayed
            }
         }

         //* Create the display text and initialize the array   *
         //* of color attributes, one for each entry            *
         gString  fmtText ;
         char*    tptr = this->rawText ;
         for ( UINT index = ZERO ; index < this->deHead[BNODE].totFiles ; index++ )
         {
            this->DirEntryDisplayFormat ( this->deList[index], fmtText, 
                                          dData.strWidth, dfDEFAULT ) ;
            fmtText.copy( tptr, fmtText.utfbytes() ) ;// store the raw string data
            this->textData[index] = tptr ;   // store pointer to string
            tptr += fmtText.utfbytes() ;     // advance the raw-text pointer
            this->colorData[index] = this->colorLookup[this->deList[index]->fType] ;
         }

         //* Sort the data according to the current sort option *
         if ( this->sortOption != fmNO_SORT )
            this->ReSortDirData ( this->sortOption ) ;
         
         //* Let caller know what we've been up to *
         dData.setData( this->deList, this->textData, this->colorData,
                        this->deHead[BNODE].totFiles,
                        this->deHead[BNODE].totBytes ) ;
         status = true ;
      }
      else if ( this->deHead[BNODE].totFiles == ZERO )
      {
         //* If directory is empty, that is not an error. Base node has been   *
         //* allocated, and caller's data indicate the facts.                  *
         status = true ;
      }
   }
   return status ;

}  //* End RefreshCurrDir() *

//*************************
//*      CdChild          *
//*************************
//********************************************************************************
//* Change directory to a child directory below current directory.               *
//*                                                                              *
//*                                                                              *
//* Input  : dPath: (by reference)                                               *
//*                 Initially contains name of target child directory.           *
//*                 On return, will contain current-working-directory path.      *
//*          dData: (by reference) partially initialized instance of the         *
//*                 DispData class:                                              *
//*                  dData.strWidth must contain the number of display columns   *
//*                  for the display strings that will be created.               *
//*                                                                              *
//* Returns: If successful, returns 'true' and the remaining members of the      *
//*            DispData class object will have been initialized.                 *
//*          If operation fails, returns 'false', dData.fileCount == ZERO        *
//*            and remaining members of DispData class object will be            *
//*            NULL pointers and ZERO values.                                    *
//********************************************************************************
//* Side effects:  Dynamic memory allocated for information on old directory     *
//*                is released before processing data for new directory.         *
//*                                                                              *
//* Note: If specified filename is not a directory name, or other error,         *
//*       we re-read current directory, but still report the original error.     *
//*                                                                              *
//********************************************************************************

bool FMgr::CdChild ( gString& dPath, DispData& dData )
{
   char  tPath[MAX_PATH] ;
   bool  status = false ;

   //* Save relative path (subdirectory name), then        *
   //* create a full path specificaton for child directory *
   dPath.copy( tPath, MAX_PATH ) ;
   if ( this->currDir[1] == NULLCHAR )    // if CWD is root directory
      dPath.compose( L"%s%s", this->currDir, tPath ) ;
   else
      this->CatPathFname ( dPath, this->currDir, tPath ) ;

   //* Set the new current-working-directory (CWD) *
   if ( (this->SetCWD ( dPath )) == OK )
   {
      dPath.copy( this->currDir, MAX_PATH ) ;
      //* Read the contents of the new directory *
      status = this->RefreshCurrDir ( dData ) ;
   }
   else
   {
      this->recentErrno = errno ;         // save 'errno' value
      //* Read contents of wherever we are: probably original directory *
      this->RefreshCurrDir ( dData ) ;
      dPath = this->currDir ;    // keep caller in the know
   }
   return status ;

}  //* End CdChild() *

//*************************
//*      CdParent         *
//*************************
//********************************************************************************
//* Change directory to parent directory above current directory.                *
//* If already at 'root' directory, simply returns a copy of current data.       *
//*                                                                              *
//* Input  : dPath: (by reference, initial value ignored)                        *
//*                 On return, will contain current-working-directory path.      *
//*          dData: (by reference) partially initialized instance of the         *
//*                 DispData class:                                              *
//*                  dData.strWidth must contain the number of display columns   *
//*                  for the display strings that will be created.               *
//*                                                                              *
//* Returns: If successful, returns 'true' and the remaining members of the      *
//*            DispData class object will have been initialized.                 *
//*          If operation fails, returns 'false', dData.fileCount == ZERO        *
//*            and remaining members of DispData class object will be            *
//*            NULL pointers and ZERO values.                                    *
//********************************************************************************
//* Side effects:  Dynamic memory allocated for information on old directory     *
//*                is released before processing data for new directory.         *
//*                                                                              *
//* Note: If already at root directory, we just return a copy of the current     *
//*       path specification and other file data and do not report an error.     *
//*                                                                              *
//* Note on 'hidden' files. We must display the current working directory, even  *
//* if it is hidden.  However, we do not display any 'hidden' contents of that   *
//* directory unless the 'showHidden' flag is SET.                               *
//* On the other hand, if the 'showHidden' flag is RESET, user could not         *
//* navigate to a hidden child directory from within the application because     *
//* they are not visible; however it is entirely possible to move to a parent    *
//* directory, hidden or not, FROM a hidden directory, regardless of the         *
//* 'showHidden' flag's state. This would be like popping out of a rabbit hole,  *
//* Alice, and then not being able to find it again because it has disappeared.  *
//* This is not a critical problem, just be aware of it.                         *
//********************************************************************************

bool FMgr::CdParent ( gString& dPath, DispData& dData )
{
   bool status = false ;

   //* Get a copy of current path string; then if not already at        *
   //* root directory, create a path specification for parent directory.*
   dPath = this->currDir ;
   if ( (dPath.compare( ROOT_PATH )) != ZERO )
   {
      gString dParent ;
      this->ExtractPathname ( dParent, dPath ) ;
      //* Set the new current-working-directory *
      //* and read its contents.                *
      if ( (this->SetCWD ( dParent )) == OK )
      {
         dParent.copy( this->currDir, MAX_PATH ) ;
         status = this->RefreshCurrDir ( dData ) ;
      }
      dPath = this->currDir ;    // keep caller in the know
   }
   else
   {
      //* We didn't go anywhere, so return a copy of the current data *
      dData.deList      = this->deList ;
      dData.textData    = this->textData ;
      dData.colorData   = this->colorData ;
      dData.listCount   = this->deHead != NULL ? this->deHead[BNODE].totFiles : ZERO ;
      dData.listBytes   = this->deHead != NULL ? this->deHead[BNODE].totBytes : ZERO ;
      status = true ;   // report 'no error'
   }
   return status ;

}  //* End CdParent() *

//*************************
//*        CdPath         *
//*************************
//********************************************************************************
//* Change directory to directory specified by full path string.                 *
//*                                                                              *
//* Input  : dPath: full path specification for new target directory             *
//*          dData: (by reference) partially initialized instance of the         *
//*                 DispData class:                                              *
//*                  dData.strWidth must contain the number of display columns   *
//*                  for the display strings that will be created.               *
//*                                                                              *
//* Returns: If successful, returns 'true' and the remaining members of the      *
//*            DispData class object will have been initialized.                 *
//*           If operation fails, returns 'false', dData.fileCount == ZERO       *
//*            and remaining members of DispData class object will be            *
//*            NULL pointers and ZERO values.                                    *
//*******************************************************************************

bool FMgr::CdPath ( gString& dPath, DispData& dData )
{
   bool  status = false ;

   if ( (this->SetCWD ( dPath )) == OK )
   {
      dPath.copy( this->currDir, MAX_PATH ) ;
      //* Read the contents of the new directory *
      status = this->RefreshCurrDir ( dData ) ;
   }
   else
   {
      this->recentErrno = errno ;         // save 'errno' value
      this->DebugLog ( "In CdPath: chdir() failed", this->recentErrno, dPath.ustr() ) ;

      //* Read contents of wherever we are: probably original directory *
      this->RefreshCurrDir ( dData ) ;
   }
   return status ;

}  //* End CdPath() *

//**************************
//* AllocateDynamicStorage *
//**************************
//********************************************************************************
//* Allocate space to store the data associated with the files in the target     *
//* directory, as well as space to store the corresponding display data.         *
//*                                                                              *
//* Any memory previously allocated for display data will be released before     *
//* allocating the new memory space.                                             *
//* Data for the scanned directory tree WILL NOT be deleted.                     *
//*                                                                              *
//* Input  : fCount == number of files to plan for                               *
//*                                                                              *
//* Returns: 'true' if successful, else 'false'                                  *
//********************************************************************************
//* Programmer's Note: We allocate one more instance of each object that we      *
//* actually need. Better safe, than get a midnight phone call...                *
//*                                                                              *
//* Note: The TreeNode list attached to this->deHead is allocated elsewhere.     *
//*                                                                              *
//* Programmer's Note: The calculated total size of the memory block allocated   *
//* for 'rawText' is based on a maximum-average-width of MAX_DIALOG_WIDTH.       *
//* This is considered to be a "comfortable" amount. The actual average width    *
//* of a display record is less than 80 characters (78 text columns), most of    *
//* which are single-byte numeric characters or spaces (0x20).                   *
//* The average record, then, is about _half_ of the space allocated for it.     *
//* "    18,349  09-Aug-2019  13:11:52 FMgrDispData.hpp"                         *
//* "     4,096  25-Jan-2016  11:42:06 荒谬的名字"                                 *
//* " 4,776,683  23-Apr-2019  21:30:23 Chuyao_2018_interest_statement-01.jpg"    *
//********************************************************************************

bool FMgr::AllocateDynamicStorage ( UINT fCount )
{
   //* If any previous allocations have not yet *
   //* been released, do so now.                *
   this->ReleaseDynamicStorage ( false ) ;
   bool  status = false ;        // return value
   
   if ( (fCount > ZERO) && (this->deHead != NULL) )
   {
      //* Establish access lock *
      lock_guard<mutex> guard ( heapLock ) ;

      UINT aCount = fCount + 1 ;    // allocate one more than requested

      for ( short i = 3 ; i > ZERO ; --i )
      {
         if ( (this->deList = new (nothrow) tnFName*[aCount]) != NULL )
         {
            #if TNA_LOG != 0     // For debugging only
            if ( this->tnalogofs.is_open() )
            {
               this->tnalogOut.compose( "---:%+4d\tdeList (tnFName*[])", &aCount ) ;
               this->tnalogofs << tnalogOut.ustr() << endl ;
            }
            #endif   // TNA_LOG

            break ;
         }
      }
      for ( short i = 3 ; i > ZERO ; --i )
      {
         if ( (this->textData = new (nothrow) char*[aCount]) != NULL )
         {
            #if TNA_LOG != 0     // For debugging only
            if ( this->tnalogofs.is_open() )
            {
               this->tnalogOut.compose( "---:%+4d\ttextData (char*[])", &aCount ) ;
               this->tnalogofs << tnalogOut.ustr() << endl ;
            }
            #endif   // TNA_LOG

            break ;
         }
      }
      for ( short i = 3 ; i > ZERO ; --i )
      {
         if ( (this->colorData = new (nothrow) attr_t[aCount]) != NULL )
         {
            #if TNA_LOG != 0     // For debugging only
            if ( this->tnalogofs.is_open() )
            {
               this->tnalogOut.compose( "---:%+4d\tcolorData (attr_t[])", &aCount ) ;
               this->tnalogofs << tnalogOut.ustr() << endl ;
            }
            #endif   // TNA_LOG

            break ;
         }
      }
      for ( short i = 3 ; i > ZERO ; --i )
      {
         if ( (this->rawText = new (nothrow) char[MAX_DIALOG_WIDTH * aCount]) != NULL )
         {
            #if TNA_LOG != 0     // For debugging only
            if ( this->tnalogofs.is_open() )
            {
               int  x = MAX_DIALOG_WIDTH * aCount ;
               gString gs( x, 8, true, true ) ;
               this->tnalogOut.compose( "---:%S\trawText (char[])", gs.gstr() ) ;
               this->tnalogofs << tnalogOut.ustr() << endl ;
            }
            #endif   // TNA_LOG

            break ;
         }
      }
      if ( (this->deList != NULL)    &&
           (this->textData != NULL)  &&
           (this->colorData != NULL) &&
           (this->rawText != NULL) )
      {
         status = true ;
      }
      else
      {
         this->recentErrno = errno ;
         this->DebugLog ( "In AllocateDynamicStorage: memory allocation error", 
                          this->recentErrno ) ;
      }
   }
   return status ;
   // As we leave the scope of the lock, the lock is freed.

}  //* End AllocateDynamicStorage() *

//****************************
//*   ReleaseDynamicStorage  *
//****************************
//********************************************************************************
//* Release all spaces previously reserved file information by a call to         *
//* AllocateDynamicStorage(), AND optionally release the TreeNode list           *
//* attached to this->deHead.                                                    *
//*                                                                              *
//* For the display data, each array contains the number of elements equal to    *
//* the sum of all directory files + non-directory files (+ 1) in the directory  *
//* tree. This includes entries for any "hidden" files. Hidden files may or may  *
//* not be displayed, but they are included in the count of array elements.      *
//*                                                                              *
//* Input  : releaseALL : (optional, 'true' by default)                          *
//*                       if 'true',  release the directory tree data attached   *
//*                                   to 'deHead' member.                        *
//*                       if 'false', release the display data only, but retain  *
//*                                   the directory tree data                    *
//*                                                                              *
//* Returns: nothing                                                             *
//*          pointers to released data are set to NULL pointer                   *
//********************************************************************************

void FMgr::ReleaseDynamicStorage ( bool releaseAll )
{
   #if TNA_LOG != 0
   //* Number of objects to be released from each array.  *
   //* This value is written to the log so amount of data *
   //* released may be compared with amount allocated.    *
   int  rCount = ZERO ;
   if ( this->deHead != NULL )
   {
      //* Hidden files may not be displayed, but *
      //* there is still an allocation for them. *
      //*          (see note above)              *
      UINT nln, tnf ;
      this->deHead[BNODE].GetActual ( nln, tnf ) ;
      rCount = -(nln + tnf + 1) ;
   }
   #endif   // TNA_LOG

   if ( this->deList != NULL )
   {
      delete [] this->deList ;
      this->deList = NULL ;

      #if TNA_LOG != 0     // For debugging only
      if ( this->tnalogofs.is_open() )
      {
         this->tnalogOut.compose( "---:%+4d\tdeList (tnFName*[])", &rCount ) ;
         this->tnalogofs << tnalogOut.ustr() << endl ;
      }
      #endif   // TNA_LOG
   }
   if ( this->textData != NULL )
   {
      delete [] this->textData ;
      this->textData = NULL ;

      #if TNA_LOG != 0     // For debugging only
      if ( this->tnalogofs.is_open() )
      {
         this->tnalogOut.compose( "---:%+4d\ttextData (char*[])", &rCount ) ;
         this->tnalogofs << tnalogOut.ustr() << endl ;
      }
      #endif   // TNA_LOG
   }
   if ( this->colorData != NULL )
   {
      delete [] this->colorData ;
      this->colorData = NULL ;

      #if TNA_LOG != 0     // For debugging only
      if ( this->tnalogofs.is_open() )
      {
         this->tnalogOut.compose( "---:%+4d\tcolorData (attr_t[])", &rCount ) ;
         this->tnalogofs << tnalogOut.ustr() << endl ;
      }
      #endif   // TNA_LOG
   }
   if ( this->rawText != NULL )
   {
      delete [] this->rawText ;
      this->rawText = NULL ;

      #if TNA_LOG != 0     // For debugging only
      if ( this->tnalogofs.is_open() )
      {
         int  x = MAX_DIALOG_WIDTH * rCount ;
         gString gs( x, 8, true, true ) ;
         this->tnalogOut.compose( "---:%S\trawText (char[])", gs.gstr() ) ;
         this->tnalogofs << tnalogOut.ustr() << endl ;
      }
      #endif   // TNA_LOG
   }
   if ( (releaseAll != false) && (this->deHead != NULL) )
   {
      this->ReleaseDirTree () ;
   }

}  //* End ReleaseDynamicStorage() *

//*************************
//*  Allocate_TreeNodes   *
//*************************
//********************************************************************************
//* Instantiate an array of TreeNode-class objects.                              *
//* The array is allocated from the memory heap.                                 *
//* Do not call this method for automatic-variable instantiation.                *
//*                                                                              *
//* Access is controlled by a mutex to avoid having multiple execution threads   *
//* overwhelm the system with memory-allocation requests.                        *
//*                                                                              *
//* Note: When allocating an array of objects, the requested allocation is padded*
//*       by a fixed number of records. This is done because there is a lag time *
//*       between calculating the number of records needed and the actual        *
//*       initialization of those records. During this interval the actual number*
//*       of files or directories to be scanned may have changed. This padding   *
//*       reduces the chance of array overrun if a new item is created in the    *
//*       delay interval.                                                        *
//*                                                                              *
//* Input  : tnCount : number of objects to allocate for the array               *
//*                                                                              *
//* Returns: pointer to array of TreeNode objects                                *
//********************************************************************************
//* Programmer's Note: The 'new' and 'delete' keywords in C++ are just a way     *
//* of masking the underlying C code (probably 'calloc'). This must be kept in   *
//* mind when creating and releasing class instances. For a single instance,     *
//* there is a single pointer, so the process is straightforward.                *
//*         TreeNode* tnPtr = new TreeNode ;     delete tnPtr ;                  *
//*                                                                              *
//* However, if an array of class instances is instantiated with one             *
//* controlling pointer, the C++ code calls the constructor once for each        *
//* member (low-to-high) of the array, which makes perfect sense.                *
//*         TreeNode* tnPtr = new TreeNode[10] ;                                 *
//*                                                                              *
//* The tricky part comes when it is time to release the array. The following    *
//* code will cause the C++ code to delete each instance separately,             *
//* IN REVERSE ORDER:                                                            *
//*         delete tnPtr ;                                                       *
//*                                                                              *
//* It is NOT possible, however, to delete just one member of the array:         *
//*         delete &tnPtr[5] ;  <-- BAD CODE!!                                   *
//*                                                                              *
//* If instances are allocated as an array, they must be deleted as an array,    *
//* using the pointer to the head of the list because although C++ dutifully     *
//* tries to release the memory for tnPtr[5] in the case shown, that pointer     *
//* is not really a separate block of memory, even though the C++ code called    *
//* the constructor to create the instances separately.  Apparently, the C++     *
//* code allocates a SINGLE BLOCK OF MEMORY for all instances, and THEN calls    *
//* the constructor to do whatever magic it wants with each instance.            *
//*                                                                              *
//* The following allocation for an array with one element works as expected:    *
//*         TreeNode* tnPtr = new TreeNode[1] ;  delete [] tnPtr ;               *
//*                                                                              *
//* BUT THIS IS AN ERROR:                                                        *
//*         TreeNode* tnPtr = new TreeNode ;     delete [] tnPtr ;               *
//*                                                                              *
//* A single allocation, must be released as a single allocation:                *
//*         TreeNode* tnPtr = new TreeNode ;     delete tnPtr ;                  *
//*                                                                              *
//* There is nothing wrong with the processes described; just be aware that an   *
//* array of class instances lives in a SINGLE BLOCK OF MEMORY - and act         *
//* accordingly.                                                                 *
//*                                                                              *
//* Note that for automatic variables, the instances go in and out of scope      *
//* tranparently:                                                                *
//*         top of method . . .                                                  *
//*         TreeNode  tnArray[5] ;                                               *
//*         TreeNode  tmpNode ;                                                  *
//*         do stuff . . .                                                       *
//*         bottom of method . . . (bye-bye instances)                           *
//*                                                                              *
//*                   ---  ---  ---  ---  ---  ---  ---                          *
//*                                                                              *
//* Access to the system's memory-allocation functionality is controlled by a    *
//* mutex which allows multiple threads to allocate (arrays of) TreeNode         *
//* instances in an orderly way. This means that a thread requesting             *
//* instantiation of these objets will block while waiting for system access.    *
//*                                                                              *
//********************************************************************************

TreeNode* FMgr::Allocate_TreeNodes ( UINT tnCount )
{
   //* Establish access lock *
   lock_guard<mutex> guard ( heapLock ) ;

   //* If memory allocation fails wait a moment before retry.*
   chrono::duration<short, std::milli>aMoment( 70 ) ;

   const UINT  allocPAD = 2 ;    // just in case (see note above)
   TreeNode* tnPtr = NULL ;
   tnCount += allocPAD ;         // (see note above)

   for ( short i = 3 ; i > ZERO ; --i )
   {
      if ( (tnPtr = new (nothrow) TreeNode[tnCount]) != NULL )
      {
         this->treenodeObjects += tnCount ;

         #if TNA_LOG != 0     // For debugging only
         if ( this->tnalogofs.is_open() )
         {
            this->tnalogOut.compose( "tn :%+4d\t%3u  %3u", &tnCount, 
                                     &this->treenodeObjects, &this->tnfnameObjects ) ;
            this->tnalogofs << tnalogOut.ustr() << endl ;
         }
         #endif   // TNA_LOG

         break ;
      }
      else
      {
         this_thread::sleep_for( aMoment ) ;  // give the system some time
         #if ENABLE_FMgrDebugLog != 0     //* For debugging only:
         this->dbgFile.compose( "Memory Allocation Error in "
                                "Allocate_TreeNodes(Retry Count:%hd tnPtr:%p)", &i, tnPtr ) ;
         this->DebugLog ( this->dbgFile.ustr() ) ;
         #endif   // ENABLE_FMgrDebugLog
      }
   }
   if ( tnPtr == NULL )
   {
      aMoment = chrono::duration<short, std::milli>(minUAINTERVAL * 100) ; ;
      for ( short i = 3 ; i > ZERO ; --i )
      {
         nc.UserAlert () ;
         this_thread::sleep_for( aMoment ) ;  // give the system some time
      }
      #if ENABLE_FMgrDebugLog != 0     //* For debugging only:
      this->dbgFile.compose( "Memory Allocation Error in "
                             "Allocate_TreeNodes(tnPtr:%p)", tnPtr ) ;
      this->DebugLog ( this->dbgFile.ustr() ) ;
      #endif   // ENABLE_FMgrDebugLog
   }
   return tnPtr ;
   // As we leave the scope of the lock, the lock is freed.

}  //* End Allocate_TreeNodes() *

//*************************
//*   Release_TreeNodes   *
//*************************
//********************************************************************************
//* Release (delete) an array of TreeNode objects which was allocated by a       *
//* call to Allocate_TreeNodes().                                                *
//*                                                                              *
//* If an array of tnFName objects is attached to the 'tnFiles' member of any    *
//* element in the array, the tnFName array is also deleted.                     *
//*                                                                              *
//* Input  : tnPtr  : pointer to an array of TreeNode objects                    *
//*                   on return, tnPtr == NULL                                   *
//*          tnCount: number of elements in the array                            *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* Programmer's Note: The TreeNode destructor will automatically delete any     *
//* tnFName[] array attached to the object; however, we delete the attached      *
//* array here at a higher level so we can track memory allocation and release   *
//* as user traverses the tree.                                                  *
//********************************************************************************

void FMgr::Release_TreeNodes ( TreeNode*& tnPtr, UINT tnCount )
{

   if ( tnPtr != NULL )
   {
      for ( UINT i = ZERO ; i < tnCount ; ++i )
      {
         if ( tnPtr[i].tnFiles != NULL )
            this->Release_tnFNames ( tnPtr[i].tnFiles, tnPtr[i].tnfcount ) ;
      }
      delete [] tnPtr ;
      tnPtr = NULL ;

      this->treenodeObjects -= tnCount ;

      #if TNA_LOG != 0     // For debugging only
      if ( this->tnalogofs.is_open() )
      {
         int neg = -(tnCount) ;
         this->tnalogOut.compose( "tn :%+4d\t%3u  %3u", &neg, 
                                  &this->treenodeObjects, &this->tnfnameObjects ) ;
         this->tnalogofs << tnalogOut.ustr() << endl ;
      }
      #endif   // TNA_LOG
   }

}  //* End Release_TreeNodes() *

//*************************
//*   Allocate_tnFNames   *
//*************************
//********************************************************************************
//* Instantiate an array of tnFName-class objects.                               *
//* The array is allocated from the memory heap.                                 *
//* Do not call this method for automatic-variable instantiation.                *
//*                                                                              *
//* Access is controlled by a mutex to avoid having multiple execution threads   *
//* overwhelm the system with memory-allocation requests.                        *
//*                                                                              *
//* Note: When allocating an array of objects, the requested allocation is padded*
//*       by a fixed number of records. This is done because there is a lag time *
//*       between calculating the number of records needed and the actual        *
//*       initialization of those records. During this interval the actual number*
//*       of files or directories to be scanned may have changed. This padding   *
//*       reduces the chance of array overrun if a new item is created in the    *
//*       delay interval.                                                        *
//*                                                                              *
//* Input  : tnfCount: number of objects to allocate for the array               *
//*                                                                              *
//* Returns: pointer to array of tnFName objects                                 *
//********************************************************************************
//* Access to the system's memory-allocation functionality is controlled by a    *
//* mutex which allows multiple threads to allocate (arrays of) TreeNode         *
//* instances in an orderly way. This means that a thread requesting             *
//* instantiation of these objets will block while waiting for system access.    *
//*                                                                              *
//********************************************************************************

tnFName* FMgr::Allocate_tnFNames ( UINT tnfCount )
{
   //* Establish access lock *
   lock_guard<mutex> guard ( heapLock ) ;

   //* If memory allocation fails wait a moment before retry.*
   chrono::duration<short, std::milli>aMoment( 70 ) ;

   const UINT  allocPAD = 2 ;    // just in case (see note above)
   tnFName* tnfPtr = NULL ;
   tnfCount += allocPAD ;        // (see note above)

   for ( short i = 3 ; i > ZERO ; --i )
   {
      if ( (tnfPtr = new (nothrow) tnFName[tnfCount]) != NULL )
      {
         this->tnfnameObjects += tnfCount ;

         #if TNA_LOG != 0     // For debugging only
         if ( this->tnalogofs.is_open() )
         {
            this->tnalogOut.compose( "tnf:%+4d\t%3u  %3u", &tnfCount, 
                                     &this->treenodeObjects, &this->tnfnameObjects ) ;
            this->tnalogofs << tnalogOut.ustr() << endl ;
         }
         #endif   // TNA_LOG

         break ;
      }
      else
      {
         this_thread::sleep_for( aMoment ) ;  // give the system some time
         #if ENABLE_FMgrDebugLog != 0     //* For debugging only:
         this->dbgFile.compose( "Memory Allocation Error in "
                                "Allocate_tnFNames(Retry Count:%hd tnfPtr:%p)", &i, tnfPtr ) ;
         this->DebugLog ( this->dbgFile.ustr() ) ;
         #endif   // ENABLE_FMgrDebugLog
      }
   }
   if ( tnfPtr == NULL )
   {
      aMoment = chrono::duration<short, std::milli>(minUAINTERVAL * 100) ; ;
      for ( short i = 3 ; i > ZERO ; --i )
      {
         nc.UserAlert () ;
         this_thread::sleep_for( aMoment ) ;  // give the system some time
      }
      #if ENABLE_FMgrDebugLog != 0     //* For debugging only:
      this->dbgFile.compose( "Memory Allocation Error in Allocate_tnFNames(tnfPtr:%p)", tnfPtr ) ;
      this->DebugLog ( this->dbgFile.ustr() ) ;
      #endif   // ENABLE_FMgrDebugLog
   }
   return tnfPtr ;
   // As we leave the scope of the lock, the lock is freed.

}  //* End Allocate_tnFNames() *

//*************************
//*   Release_tnFNames    *
//*************************
//********************************************************************************
//* Release (delete) an array of tnFName objects which was allocated by a        *
//* call to Allocate_tnFNames().                                                 *
//*                                                                              *
//* Input  : tnfPtr  : pointer to an array of tnFName objects                    *
//*                    on return, tnfPtr == NULL                                 *
//*          tnfCount: number of elements in the array                           *
//*                     (primarily for debugging memory leaks)                   *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FMgr::Release_tnFNames ( tnFName*& tnfPtr, UINT tnfCount )
{

   if ( tnfPtr != NULL )
   {
      delete [] tnfPtr ;
      tnfPtr = NULL ;

      this->tnfnameObjects -= tnfCount ;

      #if TNA_LOG != 0     // For debugging only
      if ( this->tnalogofs.is_open() )
      {
         int neg = -(tnfCount) ;
         this->tnalogOut.compose( "tnf:%+4d\t%3u  %3u", &neg, 
                                  &this->treenodeObjects, &this->tnfnameObjects ) ;
         this->tnalogofs << tnalogOut.ustr() << endl ;
      }
      #endif   // TNA_LOG
   }

}  //* End Release_tnFNames() *

//*************************
//*    CatPathFname       *
//*************************
//********************************************************************************
//* Concatenate a path string with a filename string to create a path/filename   *
//* specification.                                                               *
//*                                                                              *
//* Input  : pgs  : gString object (by reference) to hold the path/filename      *
//*                 On return, pgs contains the path/filename string in both     *
//*                 UTF-8 and wchar_t formats.                                   *
//*          uPath: pointer to UTF-8 path string                                 *
//*          uFile: pointer to UTF-8 filename string                             *
//*               OR                                                             *
//*          wPath: pointer to wchar_t path string                               *
//*          wFile: pointer to wchar_t filename string                           *
//*                                                                              *
//* Returns: OK if success                                                       *
//*          ERR if string truncated                                             *
//********************************************************************************

short FMgr::CatPathFname ( gString& pgs, const char* uPath, const char* uFile )
{
short    success = OK ;

   pgs.compose( L"%s/%s", uPath, uFile ) ;
   if ( pgs.gschars() >= (gsALLOCDFLT-1) )
      success = ERR ;
   return success ;

}  //* End CatPathFname() *

short FMgr::CatPathFname ( gString& pgs, const wchar_t* wPath, const wchar_t* wFile )
{
short    success = OK ;

   pgs.compose( L"%S/%S", wPath, wFile ) ;
   if ( pgs.gschars() >= (gsALLOCDFLT-1) )
      success = ERR ;
   return success ;

}  //* End CatPathFname() *

//*************************
//*    ExtractFilename    *
//*************************
//********************************************************************************
//* Given a string that may contain one of the following:                        *
//*  - a filename                                                                *
//*  - a relative path/filename                                                  *
//*  - an absolute path/filename                                                 *
//* extract the filename string from the input string and return it to caller.   *
//*                                                                              *
//* Input  : nameBuff: buffer to hold filename                                   *
//*          pfSource: source filename or path/filename                          *
//*                                                                              *
//* Returns: 'true' if filename successfully isolated                            *
//*          'false' in special case of pfSource=="/" (at root directory)        *
//*                  OR if invalid input                                         *
//********************************************************************************
//* Programmer's Note: It is assumed that caller has sent us a valid filespec    *
//* in pfSource. If not, the source will be returned in the target.              *
//********************************************************************************

bool FMgr::ExtractFilename ( gString& nameBuff, const gString& pfSource )
{
   bool status = false ;      // return value

   short fIndx = (pfSource.findlast( fSLASH )) + 1 ;
   nameBuff = &pfSource.gstr()[fIndx] ;
   if ( ((nameBuff.gschars()) > 2) ||
        (((nameBuff.gschars()) == 2) && (*nameBuff.gstr() != fSLASH)) )
   {
      status = true ;
   }
   return status ;

}  //* End ExtractFilename() *

bool FMgr::ExtractFilename ( char nameBuff[MAX_FNAME], const gString& pfSource )
{
   bool status = false ;      // return value

   gString gs ;
   status = this->ExtractFilename ( gs, pfSource ) ;
   gs.copy( nameBuff, MAX_FNAME ) ;

   return status ;

}  //* End ExtractFilename() *

bool FMgr::ExtractFilename ( gString& nameBuff, const char* pfSource )
{
   bool status = false ;      // return value

   gString gs( pfSource ) ;
   status = this->ExtractFilename ( nameBuff, gs ) ;
   
   return status ;

}  //* End ExtractFilename() *

bool FMgr::ExtractFilename ( char nameBuff[MAX_FNAME], const char* pfSource )
{
   bool status = false ;      // return value

   gString gss( pfSource ), gsn ;
   status = this->ExtractFilename ( gsn, gss ) ;
   gsn.copy( nameBuff, MAX_FNAME ) ;

   return status ;

}  //* End ExtractFilename() *

//*************************
//*   ExtractExtension    *
//*************************
//********************************************************************************
//* Given a string that may contain one of the following:                        *
//*  - a filename                                                                *
//*  - a path/filename                                                           *
//* extract the filename-extension including the leading '.' from the input      *
//* string and return it to caller.                                              *
//*                                                                              *
//* Input  : extBuff : buffer to hold extension                                  *
//*                    If source filename does not contain an extension, then    *
//*                    a null string is returned (extBuff.gschars() == 1)        *
//*          pfSource: source filename or path/filename                          *
//* Returns: nothing                                                             *
//********************************************************************************
//* Programmer's Note: It is assumed that caller has sent us a valid filespec    *
//* in pfSource. If not, the source will be returned in the target.              *
//********************************************************************************

void FMgr::ExtractExtension ( gString& extBuff, const gString& pfSource )
{

   short pIndx = (pfSource.findlast( L'.' )) + 1 ;
   extBuff = &pfSource.gstr()[pIndx] ;

}  //* End ExtractExtension() *

void FMgr::ExtractExtension ( gString& extBuff, const char* pfSource )
{

   gString gs( pfSource ) ;
   this->ExtractExtension ( extBuff, gs ) ;

}  //* End ExtractExtension() *

void FMgr::ExtractExtension ( gString& extBuff, const wchar_t* pfSource )
{

   gString gs( pfSource ) ;
   this->ExtractExtension ( extBuff, gs ) ;

}  //* End ExtractExtension() *

//*************************
//*    ExtractPathname    *
//*************************
//********************************************************************************
//* Given a string that contains a full path/filename specification, extract     *
//* the pathname, i.e. truncate the string to remove the filename.               *
//*                                                                              *
//*                                                                              *
//* Input  : pathBuff: buffer to contain path string                             *
//*          pfSource: full path/filename specification                          *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* Programmer's Note: It is assumed that caller has sent us a valid filespec    *
//* in pfSource. If not, the source will be returned in the target.              *
//********************************************************************************

void FMgr::ExtractPathname ( gString& pathBuff, const gString& pfSource )
{
   pathBuff = pfSource ;
   short indx = pathBuff.findlast( fSLASH ) ;
   if ( indx >= ZERO )
   {
      if ( indx == ZERO )     // if pathname == root directory
         ++indx ;
      pathBuff.limitChars( indx ) ;
   }

}  //* End of ExtractPathname() *

void FMgr::ExtractPathname ( char pathBuff[MAX_PATH], const gString& pfSource )
{
   gString gsp ;
   this->ExtractPathname ( gsp, pfSource ) ;
   gsp.copy( pathBuff, MAX_PATH ) ;

}  //* End of ExtractPathname() *

void FMgr::ExtractPathname ( gString& pathBuff, const char* pfSource )
{
   gString gss( pfSource ) ;
   this->ExtractPathname ( pathBuff, gss ) ;

}  //* End of ExtractPathname() *

void FMgr::ExtractPathname ( char pathBuff[MAX_PATH], const char* pfSource )
{
   gString gss( pfSource ), gsp ;
   this->ExtractPathname ( gsp, gss ) ;
   gsp.copy( pathBuff, MAX_PATH ) ;

}  //* End of ExtractPathname() *

//*************************
//* DirEntryDisplayFormat *
//*************************
//********************************************************************************
//* Format a file-display text string based upon file information in its         *
//* tnFName class object.                                                        *
//*                                                                              *
//* Note that maxWidth parameter is assumed to be at least large enough for      *
//* the filesize, date, time, spaces and an 8.3 filename format.                 *
//* For the default formula this would be 11 + 8 + 5 + 13 = 37 columns;          *
//* however, it will typically be at least 57 columns.                           *
//*                                                                              *
//* Input  : fnptr    : pointer to a tnFName object containing                   *
//*                     information to be displayed                              *
//*          fmtText  : receives formatted display data                          *
//*          maxWidth : maximum display columns                                  *
//*          fmt      : format option                                            *
//*          fullWidth: (optional, 'true' by default)                            *
//*                     if true, pad with trailing spaces until maxWidth         *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* The point here is that even though the raw data are compact, the driving     *
//* application should not have to know anything about system-dependent or       *
//* encrypted data associated with a table entry.  We handle all of that in      *
//* this class, and this method is the first, most obvious step.                 *
//* See also FormatExpandedStats().                                              *
//*                                                                              *
//* Currently, there is only the 'default' format, but we may have a use for     *
//* alternate formatting in a future revision.                                   *
//*                                                                              *
//********************************************************************************

void FMgr::DirEntryDisplayFormat ( tnFName* fnptr, gString& fmtText, 
                                   short maxWidth, DispFormat fmt, bool fullWidth )
{
   const char* const pad = 
      "                                                  "
      "                                                  " ;
   //* Format the file-size field substring * 
   const short MAX_FSIZE = 16 ;
   wchar_t fsBuff[MAX_FSIZE] ;
   fmtText.formatInt( fnptr->fBytes, this->fileSizeFieldWidth ) ;
   fmtText.copy( fsBuff, MAX_FSIZE ) ;

   if ( fmt == dfDEFAULT )
   {
      fmtText.compose( L"%S  %02hu-%S-%04hu  %02hu:%02hu:%02hu %s",
         fsBuff, 
         &fnptr->modTime.date, wMonthString[fnptr->modTime.month], 
         &fnptr->modTime.year, &fnptr->modTime.hours, 
         &fnptr->modTime.minutes, &fnptr->modTime.seconds,
         fnptr->fName ) ;
   }

   //* If entry is a directory name, AND if it is tagged as an    *
   //* unscanned external filesystem, append the visual indicator.*
   if ( (fnptr->fType == fmDIR_TYPE) && (fnptr->fsMatch == false) )
      fmtText.append( extFileSys ) ;

   if ( fullWidth != false )
      fmtText.append( pad ) ;
   fmtText.limitCols( maxWidth ) ;

}  //* End DirEntryDisplayFormat() *

//*************************
//* GetFileSizeFieldWidth *
//*************************
//********************************************************************************
//* Returns the width setting for the filesize display field.                    *
//* Power-up default == DEF_fsFIELD                                              *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: returns width of display field                                      *
//********************************************************************************

short FMgr::GetFileSizeFieldWidth ( void )
{

   return this->fileSizeFieldWidth ;

}  //* End GetFileSizeFieldWidth() *

//*************************
//* SetFileSizeFieldWidth *
//*************************
//********************************************************************************
//* Set the width of the filesize display field. If invalid input value,         *
//* value is unchanged and ERR is returned.                                      *
//*                                                                              *
//* Power-up default == DEF_fsFIELD                                              *
//*                                                                              *
//* Input  : field width (range: 8 through 13)                                   *
//*                                                                              *
//* Returns: returns OK if successful, else ERR                                  *
//********************************************************************************

short FMgr::SetFileSizeFieldWidth ( short width )
{
short    success = false ;

   if ( width >= MIN_fsFIELD && width <= MAX_fsFIELD )
   {
      this->fileSizeFieldWidth = width ;
      success = OK ;
   }

   return success ;

}  //* End SetFileSizeFieldWidth() *

//****************************
//* EnableFileSizeFormatting *
//****************************
//********************************************************************************
//* Enable comma-separation formatting for the filesize display field.           *
//*              formatting off:  1234567890                                     *
//*              formatting on :  1,234,567,890                                  *
//*                                                                              *
//* Power-up default == enabled                                                  *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: OK                                                                  *
//********************************************************************************

short FMgr::EnableFileSizeFormatting ( void )
{

   fileSizeCommaFormat = true ;

   return OK ;

}  //* End EnableFileSizeFormatting() *

//*****************************
//* DisableFileSizeFormatting *
//*****************************
//********************************************************************************
//* Disable comma-separation formatting for the filesize display field.          *
//*              formatting off:  1234567890                                     *
//*              formatting on :  1,234,567,890                                  *
//*                                                                              *
//* Power-up default == enabled                                                  *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: OK                                                                  *
//********************************************************************************

short FMgr::DisableFileSizeFormatting ( void )
{

   fileSizeCommaFormat = false ;

   return OK ;

}  //* End DisableFileSizeFormatting() *

//*************************
//*    DecodeEpochTime    *
//*************************
//********************************************************************************
//* Convert GNU/UNIX epoch time code (seconds since the epoch: Jan. 01, 1970)    *
//* to local time in a structured format (class localTime).                      *
//*                                                                              *
//* See the info page for the 'stat' command for more information.               *
//* See also the notes in DirEntryDisplayFormat() above.                         *
//*                                                                              *
//* Input  : eTime  : epoch time (this is a signed, 64-bit value)                *
//*                   (See note in definition of localTime class.)               *
//*          ftTime : (by reference, initial values ignored)                     *
//*                                                                              *
//* Returns: nothing, but all members of ftTime will be initialized              *
//********************************************************************************
//* System conversion from time_t code to local time is returned in this         *
//* structure.                                                                   *
//*                                                                              *
//*  struct tm                                                                   *
//*  {                                                                           *
//*     int  tm_sec ;          // 0-59  (leap seconds to 61)                     *
//*     int  tm_min ;          // 0-59                                           *
//*     int  tm_hour ;         // 0-23                                           *
//*     int  tm_mday ;         // 1-31                                           *
//*     int  tm_mon ;          // 0-11                                           *
//*     int  tm_year ;         // since 1900                                     *
//*     int  tm_wday ;         // 0-6                                            *
//*     int  tm_yday ;         // 0-365                                          *
//*     int  tm_isdst ;        // >0 == is DST, 0 == not DST, <0 == unknown      *
//*  } ;                                                                         *
//*                                                                              *
//* -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -    *
//* Note that we do a bit of defensive programming in anticipation of the        *
//* dreaded year 2038 overflow of the 32-bit time_t type.                        *
//* -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -    *
//* Daylight Savings Time:                                                       *
//* ----------------------                                                       *
//* Because we use this method to convert file timestamps, the DST calculation   *
//* done by 'localtime_r' throws the time off by an hour during the DST          *
//* part of the year. Unfortunately it does not do this uniformly. Files on      *
//* Linux filesystems drop back an hour (3,600 seconds), while files on          *
//* non-Linux filesystem such as VFAT do not drop back. What's up with that?     *
//* Presumably because VFAT does not include TZ data?                            *
//*                                                                              *
//* It's actually much worse than that. The system dynamically MODIFIES the      *
//* reported mod and stat timestamps for all files based on DST.                 *
//* This means that localtime_r is doing its job properly, but is working with   *
//* bad data sent from the kernel for filesystems that have DST offsets.         *
//* We're pretty sure this isn't the whole story but we don't have enough        *
//* information at this time to solve it. See the 'zdump' utility which lists    *
//* the offsets in a zoneinfo file:                                              *
//*         zdump -v /etc/localtime                                              *
//*         zdump -v /usr/share/zoneinfo/America/New_York | grep '2017 E.T'      *
//* This yields:                                                                 *
//* Sun Mar 12 06:59:59 2017 UTC =                                               *
//* Sun Mar 12 01:59:59 2017 EST isdst=0 gmtoff=-18000                           *
//* Sun Mar 12 07:00:00 2017 UTC =                                               *
//* Sun Mar 12 03:00:00 2017 EDT isdst=1 gmtoff=-14400                           *
//* Sun Nov  5 05:59:59 2017 UTC =                                               *
//* Sun Nov  5 01:59:59 2017 EDT isdst=1 gmtoff=-14400                           *
//* Sun Nov  5 06:00:00 2017 UTC =                                               *
//* Sun Nov  5 01:00:00 2017 EST isdst=0 gmtoff=-18000                           *
//*                                                                              *
//* It seems that file timestamps are in UTC, and the system applies the offset  *
//* listed in the specified zoneinfo file (/etc/localtime). If this is true,     *
//* then why is one offset applied to ext4, and a different offset applied       *
//* to VFAT filesystem (SD card)?                                                *
//* What EXACTLY is the difference that causes the DST offset to be applied in   *
//* one case and the standard time offset to be applied in the other?            *
//*  -- the 'tm_isdst' value seems to be > ZERO for both                         *
//*  -- 'tm_gmtoff' value ?                                                      *
//*                                                                              *
//* "To disable DST changes, link your /etc/localtime file to one of zoneinfo    *
//* files placed under the folder /usr/share/zoneinfo/Etc/"                      *
//* "Zones like Etc/GMT+6 are intentionally reversed for backwards               *
//* compatibility with POSIX standards"                                          *
//*   Example command:                                                           *
//*         ln -s /usr/share/zoneinfo/Etc/GMT+3 /etc/localtime                   *
//* Note that the 'GMT' files do not contain DST offsets. Instead, they contain  *
//* two epoch values with the second value exactly 24 hours (24*60*60 seconds)   *
//* ahead of the first. What's up with that?                                     *
//*                                                                              *
//* Because this development machine is on New York time (-5:00 i.e. 5 hours     *
//* west of UTC, -4:00DST), and because the GMT files are "intentionally         *
//* reversed," our setting would be:                                             *
//*         ln -s /usr/share/zoneinfo/Etc/GMT+5 /etc/localtime                   *
//* Be aware, however, that this doesn't actually resolve the issue. It simply   *
//* sets BOTH vfat and ext4 timestamps back an _additional_ hour.                *
//*                                                                              *
//* If the kernel would tell us the truth, we would consider using ctime_r       *
//* which converts the time_t value to a string, which we would then scan for    *
//* the values, but since the kernel is a lying SOS, this won't help us.         *
//*                                                                              *
//********************************************************************************

void FMgr::DecodeEpochTime ( int64_t eTime, localTime& ftTime )
{
   ftTime.reset() ;           // initialize caller's data
   if ( eTime >= ZERO )       // defend against stupidity
   {  //* Width of time_t is system dependent *
      ftTime.epoch = eTime ;
      ftTime.sysepoch = (time_t)eTime ;   // (possible narrowing)

      //* Receives the broken-down time.              *
      Tm bdt ;       // receives broken-down time
      if ( (localtime_r ( &ftTime.sysepoch, &bdt )) != NULL )
      {
         //* Translate to localTime format *
         ftTime.date    = bdt.tm_mday ;         // today's date
         ftTime.month   = bdt.tm_mon + 1 ;      // month
         ftTime.year    = bdt.tm_year + 1900 ;  // year
         ftTime.hours   = bdt.tm_hour ;         // hour
         ftTime.minutes = bdt.tm_min ;          // minutes
         ftTime.seconds = bdt.tm_sec ;          // seconds
         ftTime.day     = bdt.tm_wday ;         // day-of-week (0 == Sunday)
         ftTime.julian  = bdt.tm_yday ;         // Julian date (0 == Jan.01)

         #if DECODE_TZ != 0   // time-zone data
         ftTime.gmtoffset = bdt.tm_gmtoff ;     // offset as number of seconds
         ftTime.dst = (bdt.tm_isdst > ZERO ) ? true : false ; // DST in effect

         #if 1    // additional time zone data (not very useful) *
         gString gs( bdt.tm_zone ) ;
         gs.copy( ftTime.timezone, ltFMT_LEN ) ;
         int absOffset = abs(ftTime.gmtoffset) ;
         int hOffset = absOffset / (60*60),     // convert offset seconds to 
             mOffset = absOffset % (60*60) ;    // hours and minutes
         if ( ftTime.gmtoffset < ZERO )
            hOffset = -(hOffset) ;
         gs.compose( L"UTC%+d:%02d", &hOffset, &mOffset ) ;
         gs.copy( ftTime.utc_zone, ltFMT_LEN ) ;
         #endif   // additional time zone data
         #endif   // DECODE_TZ
      }
   }

}  //* End DecodeEpochTime() *

//*************************
//*    EncodeEpochTime    *
//*************************
//********************************************************************************
//* Convert human-readable time to GNU/UNIX epoch time code:                     *
//* (seconds since the epoch: Jan. 01, 1970)                                     *
//*                                                                              *
//* Input  : ftTime : (by reference)                                             *
//*                   members 'sysepoch' and 'epoch' are initialized             *
//*                                                                              *
//* Returns: 64-bit epoch time                                                   *
//********************************************************************************
//* Programmer's Note: If we are beyond the dreaded year 2038 overflow point,    *
//* the assignment to member time_t sysepoch may be truncated here.              *
//*                                                                              *
//* Note also that caller could have bad data in one or more fields which the    *
//* 'timelocal' function may not catch OR may obligingly use to create a false   *
//* time. Note: 'timelocal' returns (-1) on error, but accepts and accurately    *
//* encodes data like: 2016-02-21T19:95:72                                       *
//* At this time, it's not certain that error checking the input fields is       *
//* actually worth the additional CPU cycles.                                    *
//*                                                                              *
//* Important Note:                                                              *
//* The timelocal() function does not simply perform the conversion from         *
//* human-readable time to epoch time. It apparently consults (often unreliable) *
//* timezone data and automagically (and inappropriately) adjusts the hour.      *
//* -- In Indiana for instance, which geographically is in the Central Time      *
//*    zone (UTC-6), a majority of counties are in the Eastern Time zone         *
//*    (UTC-5) by political decree. This seems to cause timelocal() to go        *
//*    bananas. This bug is not evident in California (UTC-8) or in Bejing       *
//*    (UTC+8).                                                                  *
//* -- We have instituted a hack to correct this problem. Although the odds      *
//*    are good that this fix will work regardless of the local timezone, it     *
//*    ASSUMES that 'tm_hour', 'tm_min' and 'tm_sec' fields are all within the   *
//*    expected range. If they are not, then timelocal() could have              *
//*    legitimately changed the hour to reflect an overflow in some other        *
//*    field. This should be revisited at intervals, especially when updating    *
//*    the compiler version.                                                     *
//********************************************************************************

int64_t FMgr::EncodeEpochTime ( localTime& ftTime )
{
   #define EET_TZ_HACK (1)    // EXPERIMENTAL - SEE NOTE ABOVE

   //* Convert formatted time to simple epoch time *
   Tm tm ;
   //tm.tm_yday = ftTime.julian ;  // 0 == Jan 01 (this field updated by timelocal())
   //tm.tm_wday = ftTime.day ;     // 0 == Sunday (this field updated by timelocal())
   tm.tm_mday = ftTime.date ;
   tm.tm_mon  = ftTime.month - 1 ;
   tm.tm_year = ftTime.year - 1900 ;
   tm.tm_hour = ftTime.hours ;
   tm.tm_min  = ftTime.minutes ;
   tm.tm_sec  = ftTime.seconds ;
   #if EET_TZ_HACK != 0
   int tmpHour = tm.tm_hour,
       tmpMin  = tm.tm_min,
       tmpSec  = tm.tm_sec ;
   #endif   // EET_TZ_HACK

   ftTime.sysepoch = timelocal ( &tm ) ;     // (see mktime())

   #if EET_TZ_HACK != 0
   //* See important note, above... *
   if ( (tm.tm_hour != tmpHour) )
   {
      if ( (tmpMin == tm.tm_min) && (tmpSec == tm.tm_sec) )
      {
         int hrdiff = tmpHour - tm.tm_hour ; // difference between expected and actual
         tm.tm_hour = tmpHour + (hrdiff * 60 * 60) ;
         ftTime.sysepoch = timelocal ( &tm ) ;
      }
   }
   #endif   // EET_TZ_HACK

   if ( ftTime.sysepoch >= ZERO )
   {
      ftTime.epoch = (int64_t)ftTime.sysepoch ; // (see note above)
      ftTime.day = tm.tm_wday ;
      ftTime.julian = tm.tm_yday ;
   }
   else     // error condition returned, reset caller's time data
      ftTime.reset () ;

   return ftTime.epoch ;

}  //* End EncodeEpochTime() *

//*************************
//*   ShowHiddenFiles     *
//*************************
//********************************************************************************
//* Set the flag indicating whether 'hidden' files will be reported in the       *
//* file-display list.                                                           *
//* Will take effect the next time a directory is read.                          *
//*                                                                              *
//* Input  : true  = show hidden files                                           *
//*          false = omit hidden files from the list (default)                   *
//*                                                                              *
//* Returns: OK                                                                  *
//********************************************************************************
//* Programmer's Note: Under Linux/UNIX, 'hidden' files are those whose          *
//* filenames begin with a PERIOD ('.').                                         *
//*                                                                              *
//* Data on hidden files are collected by our scan of directory contents,        *
//* but are reported outside the FMgr class only if the showHidden flag          *
//* is set.                                                                      *
//********************************************************************************

short FMgr::ShowHiddenFiles ( bool show )
{

   this->showHidden = show ;
   return OK ;

}  //* End ShowHiddenFiles() *

//*************************
//*  CreateSubDirectory   *
//*************************
//********************************************************************************
//* Create a new sub-directory in the current directory.                         *
//*                                                                              *
//* Input  : buffer containing name of new directory                             *
//*                                                                              *
//* Returns: OK if successful, else errno value                                  *
//********************************************************************************
//* Note that if target filesystem is an MTP/GVfs virtual filesystem, the        *
//* the special gvfs method is called to create the directory. The 'gio'         *
//* utility is rudimentary in design, in that it has little control over         *
//* timestamp and permission bits, but it is _much_ faster for this operation.   *
//********************************************************************************

short FMgr::CreateSubDirectory ( const char* dirName )
{
   //* Construct the path/filename string *
   gString  pgs ;
   this->CatPathFname ( pgs, this->currDir, dirName ) ;
   short status ;

   //* Create the new subdirectory *
   if ( this->irVirtual )
      status = this->CreateDirectory_gvfs ( pgs.ustr() ) ;
   else
      status = this->CreateDirectory ( pgs.ustr() ) ;

   return status ;

}  //* End CreateSubDirectory() *

//*************************
//*   CreateDirectory     *
//*************************
//********************************************************************************
//* Create a new sub-directory corresponding to the specified path/filename.     *
//*                                                                              *
//* IMPORTANT NOTE: If the target filesystem is an MTP/GVfs virtual filesystem,  *
//* then call CreateDirectory_gvfs() instead.                                    *
//*                                                                              *
//* Input  : buffer containing path/filename of new directory                    *
//*                                                                              *
//* Returns: OK if successful, else errno value                                  *
//********************************************************************************
//* Programmer's Note: Note that the shell's 'umask' setting screws with our     *
//* permission bits during the mkdir() call, so we try to overcome it here.      *
//*                                                                              *
//* Note that 'umask' is generally set by the user's login shell script.         *
//* It is dangerous to change the umask, but we can compensate by following      *
//* the directory creation with an explicit setting of the file's permission     *
//* bits.                                                                        *
//*                                                                              *
//* If a user creates a directory using the core utils, 'mkdir' command, the     *
//* permissions will (usually) be: rwxrwxr-x meaning full permissions for        *
//* owner and group,  with read/execute for others; so, that is what we          *
//* strive for here.                                                             *
//*                                                                              *
//* Note on escaping special characters: The 'mkdir()' function treats all       *
//* filename characters literally, so manually escaping characters will result   *
//* in extra '\' characters in the resulting directory name.                     *
//********************************************************************************

short FMgr::CreateDirectory ( const char* dirPath )
{
   const mode_t   dirMode = (S_IRWXU | S_IRWXG | S_IROTH | S_IXOTH) ;
   short    status ;

// CZONE - MKDIR FOR LARGE EXTERNAL FILESYSTEMS (SD CARDS, USB FLASH DRIVES)
// THIS MAY BE A VERY SLOW PROCESS. EVEN THOUGH 'mkdir()' RETURNS SUCCESS, THE 
// ACTUAL WRITE MAY BE SOMEWHERE IN THE SYSTEM CACHE. WHEN THIS HAPPENS, WE MUST 
// NOT ALLOW DATA TO BE WRITTEN TO THE DIRECTORY BEFORE IT IS ACTUALLY CREATED.
// WE NEED TO FLUSH THE CACHE TO MAKE SURE THE DIRECTORY WAS CREATED. THIS COULD BE 
// SLOW. MAYBE MAKE THIS A CONDITIONAL COMPILE AND TEST THE PERFORMANCE.
   //* Create the directory and adjust the permission bits. *
   //* See notes above.                                     *
   status = mkdir ( dirPath, dirMode ) ;
   if ( status == ZERO )
      this->SetFilePermissions ( dirPath, dirMode ) ;
   else
      this->recentErrno = status = errno ;

   return status ;

}  //* End CreateDirectory() *

//*************************
//*     isEmptyDir        *
//*************************
//********************************************************************************
//* Scans the contents of the specified directory and returns 'true' if          *
//* directory contains no files (other than "." and "..").                       *
//*                                                                              *
//* Input  : dirPath : full path of directory to be scanned                      *
//*                                                                              *
//* Returns: 'true' if empty directory, file is not a directory, OR              *
//*            file not found                                                    *
//*          else 'false'                                                        *
//********************************************************************************
//* Programmer's Note: This method could be considerably simplified if it is     *
//* for a directory in the current display list, but it's good as-is for         *
//* general use.                                                                 *
//********************************************************************************

bool FMgr::isEmptyDir ( const char* dirPath )
{
   DIR*     dirPtr ;
   bool     empty = true ;

   //* Open the directory *
   if ( (dirPtr = this->Safe_opendir ( dirPath )) != NULL )
   {
      short    count = ZERO ;    // number of entries read
      while ( (readdir64 ( dirPtr )) != NULL )
      {
         if ( ++count >= 3 )     // ( '.' is 1, '..' is 2, but 3 is a crowd )
         {
            empty = false ;
            break ;
         }
      }  // while()

      if ( (closedir ( dirPtr )) != OK )     // Close the directory
      {
         this->recentErrno = errno ;
         this->DebugLog ( "IsEmptyDir: closedir() failed", 
                          this->recentErrno, dirPath ) ;
      }
   }
   return empty ;

}  //* End isEmptyDir() *

//*************************
//*     Safe_opendir      *
//*************************
//********************************************************************************
//* ** Private Method **                                                         *
//* Open a directory file for reading.                                           *
//* Before opening the file, verify that we have read access to it.              *
//*                                                                              *
//* NOTE: It is assumed that caller is smart enough to have verified that the    *
//*       path actually points to a directory file. If not, then fail.           *
//*                                                                              *
//* NOTE: If source is a symbolic link to the directory, the read will fail.     *
//*       We report only what is ACTUALLY there, not some distant data.          *
//*                                                                              *
//* Input  : dirPath : full path of directory to be scanned                      *
//*                                                                              *
//* Returns: pointer to open directory file --OR--                               *
//*          returns NULL* if no read access or otherwise unable to open file    *
//********************************************************************************

DIR* FMgr::Safe_opendir ( const char* dirPath )
{
   DIR* dirPtr = NULL ;
   this->recentErrno = EACCES ;     // assume no access

   if ( (access ( dirPath, (R_OK | X_OK) )) == ZERO )
   {
      if ( (dirPtr = opendir ( dirPath )) != NULL )
         this->recentErrno = ZERO ;
      else
      {
         this->recentErrno = errno ;
         this->DebugLog ( "Safe_opendir: failed", this->recentErrno, dirPath ) ;
      }
   }
   else  // report no-access directory names
      this->DebugLog ( "No read access.", (-1), dirPath ) ;

   return dirPtr ;

}  //* End Safe_opendir() *

//*************************
//*     IsFNameChar       *
//*************************
//********************************************************************************
//* Test the specified character to determine whether it may be used as a        *
//* character in a filename or directory name.                                   *
//*                                                                              *
//* Input  : character to test (wchar_t)                                         *
//*                                                                              *
//* Returns: true if acceptable, else false                                      *
//********************************************************************************
//* Programmer's Note: All alphanumeric characters (ASCII and supra-ASCII) are   *
//* passed according to the 'wide' library function, iswalnum().                 *
//* All non-alphanumeric characters that are passed by the this method have      *
//* been manually tested by creating a directory name using the character.       *
//********************************************************************************

bool FMgr::IsFNameChar ( wchar_t wch )
{
bool  goodChar = false ;

   //* First, test for 'wide' alphanumeric characters.*
   if ( (iswalnum ( wch )) != ZERO )
      goodChar = true ;
   //* Test for valid, non-AlNum characters within the ASCII range. *
   else if (   wch == '.' || wch == ' ' || wch == '-' || wch == '_' || wch == '#' 
            || wch == '%' || wch == '+' || wch == ',' || wch == ':' || wch == '=' 
            || wch == '@' || wch == '(' || wch == ')' || wch == '[' || wch == ']' 
            || wch == '{' || wch == '}' || wch == '~' )
      goodChar = true ;
   return goodChar ;

}  //* End IsFNameChar() *

//*************************
//*    IdentifyUser       *
//*************************
//********************************************************************************
//* Get a copy of application user's information.                                *
//*                                                                              *
//* Input  : (by reference, initial data ignored) receives user information      *
//*                                                                              *
//* Returns: true if successful, else false                                      *
//********************************************************************************

bool FMgr::IdentifyUser ( UserInfo& ui )
{

   ui = this->userInfo ;
   return this->userInfo.goodData ;

}  //* End IdentifyUser() *

//*************************
//*    IdentifyUser       *
//*************************
//********************************************************************************
//* Private Method.                                                              *
//* Get application user's information.                                          *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: true if successful, else false                                      *
//********************************************************************************
//* Programmer's Note:                                                           *
//*                                                                              *
//* The passwd structure - from pwd.h                                            *
//* struct passwd                                                                *
//* {                                                                            *
//*   char *pw_name;                /* Username.  */                             *
//*   char *pw_passwd;              /* Password.  */                             *
//*   __uid_t pw_uid;               /* User ID.  */                              *
//*   __gid_t pw_gid;               /* Group ID.  */                             *
//*   char *pw_gecos;               /* Real name.  */                            *
//*   char *pw_dir;                 /* Home directory.  */                       *
//*   char *pw_shell;               /* Shell program.  */                        *
//* };                                                                           *
//*                                                                              *
//* The group structure - from grp.h                                             *
//* struct group                                                                 *
//*   {                                                                          *
//*     char *gr_name;              /* Group name.  */                           *
//*     char *gr_passwd;            /* Password.    */                           *
//*     __gid_t gr_gid;             /* Group ID.    */                           *
//*     char **gr_mem;              /* Member list. */                           *
//*   };                                                                         *
//*                                                                              *
//* See also the getgroups ( int COUNT, gid_t *GROUPS ) function.                *
//*  Note that user's effective group ID may, or may not be returned in          *
//*  this list. If it is the ONLY item in the supplimentary list, we don't       *
//*  display it a second time.                                                   *
//*                                                                              *
//* Note also that the actual password strings will probably not be returned     *
//*  by the system calls, but that is system dependent.                          *
//*                                                                              *
//* In an earlier version of this method, we used a dynamically-allocated        *
//* block of memory to store all this data, but it was just too awkward.         *
//* We have made a decision that no password, home path or user name string is   *
//* likely to be more than 128 bytes, so we reserve an area of the following     *
//* size to hold the strings: 128 bytes per string * 7 strings == 896 bytes,     *
//* MAX_STRING bytes to be "safe".                                               *
//* The user could belong to an unlimited number of supplimentary groups, but    *
//* we store only the first uiMAX_sGID supplimentary-group IDs.                  *
//********************************************************************************

bool FMgr::IdentifyUser ( void )
{
   PwdInfo*    pw ;                 // pointer to kernel structure for user info
   GrpInfo*    gp ;                 // pointer to kernel structure for group info
   uid_t       userID = getuid () ; // get user's ID
   gid_t       grpID  = getgid () ; // get user's group ID

   //* Get the user's name and other user information *
   this->userInfo.goodData = false ;
   if ( (pw = getpwuid ( userID )) != NULL )
   {
      this->userInfo.userID = pw->pw_uid ;
      this->userInfo.grpID  = pw->pw_gid ;

      //* Get the user's group information *
      if ( (gp = getgrgid ( grpID )) != NULL )
      {
         //* Get number of supplimentary groups user belongs to.    *
         //* Save the second copy of group ID and declare good data.*
         gid_t tg ;
         this->userInfo.suppGroups = getgroups ( ZERO, &tg ) ;
         this->userInfo.ggrpID = (ULONG)gp->gr_gid ;
         this->userInfo.goodData = true ;
      }
   }

   if ( this->userInfo.goodData != false )
   {  //* Save the string data an initialize pointers to each string *
      char *cptr = (char*)this->userInfo.stringAlloc,
           *eptr = cptr + uiMAX_STRING - 1 ;
      short i = ZERO ;
      this->userInfo.userName = cptr ;
      while ( (pw->pw_name[i] != NULLCHAR) && (cptr < eptr) ) 
         { *(cptr++) = pw->pw_name[i++] ; } *(cptr++) = NULLCHAR ;
      i = ZERO ;
      this->userInfo.realName = cptr ;
      while ( (pw->pw_gecos[i] != NULLCHAR) && (cptr < eptr) ) 
         { *(cptr++) = pw->pw_gecos[i++] ; } *(cptr++) = NULLCHAR ;
      i = ZERO ;
      this->userInfo.shellProg = cptr ;
      while ( (pw->pw_shell[i] != NULLCHAR) && (cptr < eptr) ) 
         { *(cptr++) = pw->pw_shell[i++] ; } *(cptr++) = NULLCHAR ;
      i = ZERO ;
      this->userInfo.homeDir = cptr ;
      while ( (pw->pw_dir[i] != NULLCHAR) && (cptr < eptr) ) 
         { *(cptr++) = pw->pw_dir[i++] ; } *(cptr++) = NULLCHAR ;
      i = ZERO ;
      this->userInfo.grpName = cptr ;
      while ( (gp->gr_name[i] != NULLCHAR) && (cptr < eptr) ) 
         { *(cptr++) = gp->gr_name[i++] ; } *(cptr++) = NULLCHAR ;
      i = ZERO ;
      this->userInfo.passWord = cptr ;
      while ( (pw->pw_passwd[i] != NULLCHAR) && (cptr < eptr) ) 
         { *(cptr++) = pw->pw_passwd[i++] ; } *(cptr++) = NULLCHAR ;
      i = ZERO ;
      this->userInfo.grpPWord = cptr ;
      while ( (gp->gr_passwd[i] != NULLCHAR) && (cptr < eptr) ) 
         { *(cptr++) = gp->gr_passwd[i++] ; } *(cptr++) = NULLCHAR ;
      *eptr = NULLCHAR ;   // be sure string data are terminated

      //* If user belongs to supplimentary groups, store the first uiMAX_sGIDs *
      if ( this->userInfo.suppGroups > ZERO )
      {
         getgroups ( uiMAX_sGID, this->userInfo.suppGID ) ;
         //* If the only supplimentary group ID is a duplicate *
         //* of the primary GID, don't report it again         *
         if ( this->userInfo.suppGroups == 1 
              && (this->userInfo.suppGID[0] == this->userInfo.grpID) )
         {
            this->userInfo.suppGroups = ZERO ;
         }
         //* We only report the first uiMAX_SGID groups *
         else if ( this->userInfo.suppGroups > uiMAX_sGID )
            this->userInfo.suppGroups = uiMAX_sGID ;
      }
   }

   return this->userInfo.goodData ;

}  //* End IdentifyUser() *


   //************************************************************
   //*                     DEBUGGING METHODS                    *
   //************************************************************
   
//*************************
//*   Get_FMgr_Version    *
//*************************
//********************************************************************************
//* Returns a pointer to FMgrVersion, the FMgr class version number.             *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: (const char*) pointer to version string                             *
//********************************************************************************

const char* FMgr::Get_FMgr_Version ( void )
{

   return FMgrVersion ;

}  //* End Get_FMgr_Version() *

//*************************
//*    GetErrorCode       *
//*************************
//********************************************************************************
//* Returns error code for the most recent file management error. The value      *
//* returned is from the system 'errno' variable indicating the file access      *
//* or memory access error encountered.                                          *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: most recently captured value from global 'errno' variable           *
//********************************************************************************

short FMgr::GetErrorCode ( void )
{

   return this->recentErrno ;

}  //* End GetErrorCode() *

//*************************
//*    GetErrnoMessage    *
//*************************
//********************************************************************************
//* Returns a short error message string for the specified errno code.           *
//*                                                                              *
//* Input  : code value returned to caller from this->recentErrno.               *
//*                                                                              *
//* Returns: (const char*) pointer to message string                             *
//********************************************************************************
//* Definitions for values 'errno' were taken from                               *
//*         /usr/include/asm-generic/errno-base.h  August 18, 2010               *
//*         /usr/include/asm-generic/errno.h       August 18, 2010               *
//* from:  GCC v:4.4.4.20100630  Copyright 2010 Free Software Foundation, Inc.   *
//* (Longest message length 59 bytes incl. NULLCHAR; however, the longer)        *
//* (messages are network related and unlikely to be needed here.       )        *
//* (50 columns should be more than enough to display most messages.    )        *
//*                                                                              *
//*   Some definitions for errno for special packages (e.g. math package)        *
//*   can be found elsewhere, but this class is interested only in               *
//*   file-related and memory-related values.                                    *
//*                                                                              *
//********************************************************************************

const char* FMgr::GetErrnoMessage ( short errCode )
{
const short maxMSGS = 133 ;
static const char* gemMsg[maxMSGS] = 
{
   "No Error",                                                    // 00
   "EPERM - Operation not permitted",                             // 01
   "ENOENT - No such file or directory",                          // 02
   "ESRCH - No such process",                                     // 03
   "EINTR - Interrupted system call",                             // 04
   "EIO - I/O error",                                             // 05
   "ENXIO - No such device or address",                           // 06
   "E2BIG - Arg list too long",                                   // 07
   "ENOEXEC - Exec format error",                                 // 08
   "EBADF - Bad file number",                                     // 09
   "ECHILD - No child processes",                                 // 10
   "EAGAIN - Try again",                                          // 11
   "ENOMEM - Out of memory",                                      // 12
   "EACCES - Permission denied",                                  // 13
   "EFAULT - Bad address",                                        // 14
   "ENOTBLK - Block device required",                             // 15
   "EBUSY - Device or resource busy",                             // 16
   "EEXIST - File exists",                                        // 17
   "EXDEV - Cross-device link",                                   // 18
   "ENODEV - No such device",                                     // 19
   "ENOTDIR - Not a directory",                                   // 20
   "EISDIR - Is a directory",                                     // 21
   "EINVAL - Invalid argument",                                   // 22
   "ENFILE - File table overflow",                                // 23
   "EMFILE - Too many open files",                                // 24
   "ENOTTY - Not a typewriter",                                   // 25
   "ETXTBSY - Text file busy",                                    // 26
   "EFBIG - File too large",                                      // 27
   "ENOSPC - No space left on device",                            // 28
   "ESPIPE - Illegal seek",                                       // 29
   "EROFS - Read-only file system",                               // 30
   "EMLINK - Too many links",                                     // 31
   "EPIPE - Broken pipe",                                         // 32
   "EDOM - Math argument out of domain of func",                  // 33
   "ERANGE - Math result not representable",                      // 34
   "EDEADLK - Resource deadlock would occur",                     // 35
   "ENAMETOOLONG - File name too long",                           // 36
   "ENOLCK - No record locks available",                          // 37
   "ENOSYS - Function not implemented",                           // 38
   "ENOTEMPTY - Directory not empty",                             // 39
   "ELOOP - Too many symbolic links encountered",                 // 40
   "EWOULDBLOCK - EAGAIN Operation would block",                  // 41
   "ENOMSG - No message of desired type",                         // 42
   "EIDRM - Identifier removed",                                  // 43
   "ECHRNG - Channel number out of range",                        // 44
   "EL2NSYNC - Level 2 not synchronized",                         // 45
   "EL3HLT - Level 3 halted",                                     // 46
   "EL3RST - Level 3 reset",                                      // 47
   "ELNRNG - Link number out of range",                           // 48
   "EUNATCH - Protocol driver not attached",                      // 49
   "ENOCSI - No CSI structure available",                         // 50
   "EL2HLT - Level 2 halted",                                     // 51
   "EBADE -  Invalid exchange",                                   // 52
   "EBADR -  Invalid request descriptor",                         // 53
   "EXFULL - Exchange full",                                      // 54
   "ENOANO - No anode",                                           // 55
   "EBADRQC - Invalid request code",                              // 56
   "EBADSLT - Invalid slot",                                      // 57
   "EDEADLOCK - Resource deadlock would occur",                   // 58
   "EBFONT - Bad font file format",                               // 59
   "ENOSTR - Device not a stream",                                // 60
   "ENODATA - No data available",                                 // 61
   "ETIME - Timer expired",                                       // 62
   "ENOSR - Out of streams resources",                            // 63
   "ENONET - Machine is not on the network",                      // 64
   "ENOPKG - Package not installed",                              // 65
   "EREMOTE - Object is remote",                                  // 66
   "ENOLINK - Link has been severed",                             // 67
   "EADV - Advertise error",                                      // 68
   "ESRMNT - Srmount error",                                      // 69
   "ECOMM - Communication error on send",                         // 70
   "EPROTO - Protocol error",                                     // 71
   "EMULTIHOP - Multihop attempted",                              // 72
   "EDOTDOT - RFS specific error",                                // 73
   "EBADMSG - Not a data message",                                // 74
   "EOVERFLOW - Value too large for defined data type",           // 75
   "ENOTUNIQ - Name not unique on network",                       // 76
   "EBADFD - File descriptor in bad state",                       // 77
   "EREMCHG - Remote address changed",                            // 78
   "ELIBACC - Can not access a needed shared library",            // 79
   "ELIBBAD - Accessing a corrupted shared library",              // 80
   "ELIBSCN - .lib section in a.out corrupted",                   // 81
   "ELIBMAX - Attempting to link in too many shared libraries",   // 82
   "ELIBEXEC - Cannot exec a shared library directly",            // 83
   "EILSEQ - Illegal byte sequence",                              // 84
   "ERESTART - Interrupted system call should be restarted",      // 85
   "ESTRPIPE - Streams pipe error",                               // 86
   "EUSERS - Too many users",                                     // 87
   "ENOTSOCK - Socket operation on non-socket",                   // 88
   "EDESTADDRREQ - Destination address required",                 // 89
   "EMSGSIZE - Message too long",                                 // 90
   "EPROTOTYPE - Protocol wrong type for socket",                 // 91
   "ENOPROTOOPT - Protocol not available",                        // 92
   "EPROTONOSUPPORT - Protocol not supported",                    // 93
   "ESOCKTNOSUPPORT - Socket type not supported",                 // 94
   "EOPNOTSUPP - Operation not supported on transport endpoint",  // 95
   "EPFNOSUPPORT - Protocol family not supported",                // 96
   "EAFNOSUPPORT - Address family not supported by protocol",     // 97
   "EADDRINUSE - Address already in use",                         // 98
   "EADDRNOTAVAIL - Cannot assign requested address",             // 99
   "ENETDOWN - Network is down",                                  // 100
   "ENETUNREACH - Network is unreachable",                        // 101
   "ENETRESET - Network dropped connection because of reset",     // 102
   "ECONNABORTED - Software caused connection abort",             // 103
   "ECONNRESET - Connection reset by peer",                       // 104
   "ENOBUFS - No buffer space available",                         // 105
   "EISCONN - Transport endpoint is already connected",           // 106
   "ENOTCONN - Transport endpoint is not connected",              // 107
   "ESHUTDOWN - Cannot send after transport endpoint shutdown",   // 108
   "ETOOMANYREFS - Too many references: cannot splice",           // 109
   "ETIMEDOUT - Connection timed out",                            // 110
   "ECONNREFUSED - Connection refused",                           // 111
   "EHOSTDOWN - Host is down",                                    // 112
   "EHOSTUNREACH - No route to host",                             // 113
   "EALREADY - Operation already in progress",                    // 114
   "EINPROGRESS - Operation now in progress",                     // 115
   "ESTALE - Stale NFS file handle",                              // 116
   "EUCLEAN - Structure needs cleaning",                          // 117
   "ENOTNAM - Not a XENIX named type file",                       // 118
   "ENAVAIL - No XENIX semaphores available",                     // 119
   "EISNAM -  Is a named type file",                              // 120
   "EREMOTEIO - Remote I/O error",                                // 121
   "EDQUOT - Quota exceeded",                                     // 122
   "ENOMEDIUM - No medium found",                                 // 123
   "EMEDIUMTYPE - Wrong medium type",                             // 124
   "ECANCELED - Operation Canceled",                              // 125
   "ENOKEY - Required key not available",                         // 126
   "EKEYEXPIRED - Key has expired",                               // 127
   "EKEYREVOKED - Key has been revoked",                          // 128
   "EKEYREJECTED - Key was rejected by service",                  // 129
   "EOWNERDEAD - Owner died",                                     // 130
   "ENOTRECOVERABLE - State not recoverable",                     // 131
   "ERFKILL - Operation not possible due to RF-kill",             // 132
} ;
static char gemUnk[32] ;

   if ( errCode >= ZERO && errCode < maxMSGS )
      return gemMsg[errCode] ;
   else
   {
      errCode &= 0x1FFF ;  // limit to 4 decimal digits (it's bogus anyway)
      snprintf ( gemUnk, 32, "%04d - unknown system error", errCode ) ;
      return gemUnk ;
   }

}  //* End GetErrnoMessage() *

//**************************
//* GetTreeNode_Allocation *
//**************************
//********************************************************************************
//* DEBUGGING ONLY - Used only to test for memory leaks for TreeNode and         *
//* tnFName allocation and release.                                              *
//*                                                                              *
//* Input  : tnObjects : (by reference)                                          *
//*                      receives count of TreeNode objects currently allocated  *
//*        : tnfObjects: (by reference)                                          *
//*                      receives count of tnFName objects currently allocated   *
//*                                                                              *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FMgr::GetTreeNode_Allocation ( UINT& tnObjects, UINT& tnfObjects )
{

   tnObjects = this->treenodeObjects ;
   tnfObjects = this->tnfnameObjects ;

}  //* End GetTreeNode_Allocation() *

//*************************
//*       DebugMsg        *
//*************************
//********************************************************************************
//* Display a debugging message in the bottom line of the application dialog.    *
//* Because the FMgr class has no direct access to the application layer, this   *
//* method uses a callback method provided by the application layer.             *
//*                                                                              *
//* Input  : msg  : pointer to caller's message string                           *
//*          pause: (optional, ZERO by default)                                  *
//*                 if: zero < pause < 10                                        *
//*                     pause for the specified number of seconds                *
//*                 if >= 10, wait for a keypress                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FMgr::DebugMsg ( const char* msg, short pause )
{

   this->DebugMsgCB ( msg ) ;

   if ( pause > ZERO )
   {
      if ( pause < 10 )
      {
         chrono::duration<short>aWhile( pause ) ;
         this_thread::sleep_for( aWhile ) ;
      }
      else
         nckPause();
      this->DebugMsgCB ( "%" ) ;
   }

}  //* End DebugMsg() *

//*************************
//*       DebugLog        *
//*************************
//********************************************************************************
//* Write an entry to the debugging log.                                         *
//* If log file output is disabled, then call has no effect.                     *
//*                                                                              *
//* Input  : msg: pointer to caller's message string                             *
//*          err: (optional, -1 by default)                                      *
//*               value of global variable, errno, or other error code           *
//*          aux: (optional, NULL pointer by default)                            *
//*               if specified, include as an auxilliary message in the log,     *
//*               (usually the filespec for a file that generated an error)      *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FMgr::DebugLog ( const char* msg, short err, const char* aux )
{
   #if ENABLE_FMgrDebugLog != 0
   //* If the file is not yet open, open it now.*
   if ( ! (dbgofs.is_open()) && (msg[ZERO] != NULLCHAR) )
   {
      this->CreateTempname ( this->dbgFile ) ;
      dbgofs.open( this->dbgFile.ustr(), ofstream::out | ofstream::trunc ) ;
   }

   if ( dbgofs.is_open() )
   {
      gString gs ;
      if ( msg[ZERO] != NULLCHAR )
      {
         if ( err <= ZERO )
            gs = msg ;
         else
         {
            if ( err < 133 )  // (magic number is size of message array in GetErrnoMessage())
               gs.compose( L"%s - errno:%04hd %s", msg, &err, this->GetErrnoMessage(err) ) ;
            else
               gs.compose( L"%s - %hd", msg, &err ) ;
         }

         if ( aux != NULL )               // append any auxilliary message
            gs.append( "\n %s", aux ) ;

         dbgofs << gs.ustr() << endl ;    // write entry to file
      }

      //* An empty message string is the signal to close the file.*
      else
      {
         dbgofs << "\n** End Log **" << endl ;
         dbgofs.close() ;
         gs.compose( "DebugLog Closed: '%S'", this->dbgFile.gstr() ) ;
         this->DebugMsg ( gs.ustr(), 10 ) ;     // this call waits for user to press a key
         this->DeleteTempname ( this->dbgFile ) ;
      }
   }
   #endif   // ENABLE_FMgrDebugLog

}  //* End DebugLog() *

//*************************
//*       UserAlert       *
//*************************
//********************************************************************************
//* Make some noise. Used only for development and debugging.                    *
//* Calls the NCurses-class UserAlert() method.                                  *
//*                                                                              *
//* Input  : beeps : (optional, 1 by default) number of beeps to produce.        *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FMgr::UserAlert ( short beeps )
{
   chrono::duration<short, std::milli>aMoment( 200 ) ;

   for ( short b = ZERO ; b < beeps ; ++b )
   {
      nc.UserAlert () ;
      this_thread::sleep_for( aMoment ) ;
   }

}  //* End UserAlert() *

//*************************
//*                       *
//*************************
//********************************************************************************
//*                                                                              *
//*                                                                              *
//*                                                                              *
//* Input  :                                                                     *
//*                                                                              *
//* Returns:                                                                     *
//********************************************************************************

