//********************************************************************************
//* File       : FMgrGvfs.cpp                                                    *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2005-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in FileMangler.hpp         *
//* Date       : 20-May-2025                                                     *
//* Version    : (see FMgrVersion string)                                        *
//*                                                                              *
//* Description: This is a support module for the FMgr class.                    *
//* This module contains the methods specific to MTP/GVfs filesystem access,     *
//* specifically the use of the external 'gio' utility to manage files on        *
//* GNOME Virtual File System (GVfs) devices. Access to these devices is         *
//* through the Media Transfer Protocol (MTP) as implemented in the 'gio'        *
//* utility.                                                                     *
//*                                                                              *
//* Note that the 'gio' utility is not particularly capable or robust.           *
//* MTP/GVfs is still evolving, and we hope that 'gio' will evolve with it.      *
//*                                                                              *
//* If the 'gio' utility is not installed on the target system, the directory    *
//* scan (GvfsFlatScan()) will automatically call the system built-in            *
//* utility to complete the scan. Other methods in this module rely upon the     *
//* presence of the 'gio' utility to perform their tasks, but see the            *
//* GVFS_FAILSAFE conditional-compile flag below.                                *
//*                                                                              *
//* Developed using 'gio' v: 2.64.3                                              *
//********************************************************************************
//* Notes on using 'gio' to access MTP/GVfs virtual filesystems.                 *
//* ------------------------------------------------------------                 *
//* 1) The 'gio' utility provides basic access to the underlying lib-gio         *
//*    functionality. 'gio' is neither robust nor elegant. It is an inelegant    *
//*    hack in its current form (July 2020). We hope sincerely that it improves  *
//*    over time.                                                                *
//*                                                                              *
//* 2) A subset of the file operations available on "real" filesystems           *
//*    (ext3/ext4/vfat/ntfs, etc.) are available for gvfs filesystems.           *
//*    a) Recursive operations on the directory tree are not supported by the    *
//*       available tools. We _could_ synthesize recursive operations;           *
//*       however, smartphones and similar devices are so unbearably slow to     *
//*       provide filesystem information, that we gather only information that   *
//*       is immediately necessary for the available operations.                 *
//*    b) It is expected that the user's primary goal is to transfer files       *
//*       from the gvfs device to the Linux system to which the device is        *
//*       attached. This could be a straightforward copy-and-paste operation,    *
//*       but could also take the form of cut-and-paste or of a data backup      *
//*       operation.                                                             *
//*    c) Renaming files on the device, especially photos, is also a common      *
//*       operation that is often inconvenient to perform using the device's     *
//*       built-in tools, so both single-file rename and batch rename are        *
//*       supported here.                                                        *
//*    d) The user may occasionally wish to transfer data from the Linux         *
//*       system to the gvfs device. Music files would be a primary example.     *
//*    e) Removing stale data from the device would usually be done using the    *
//*       device's internal tools, but the operation might be more convenient    *
//*       on a full-sized computer screen, so deleting files is supported.       *
//*    f) Special Files: We restrict the user to operations on three types of    *
//*       files only: "regular" (ordinary) files, directory files (not           *
//*       directory trees), and symbolic links. It is likely that symbolic       *
//*       links will seldom be encountered in gvfs user space, so the focus is   *
//*       on regular files and directory files.                                  *
//*                                                                              *
//* 3) The most important design question is whether to use the standard kernel  *
//*    tools or 'gio' for a given operation. The philosophy here it that if      *
//*    either source or target is a gvfs filesystem, then the gvfs tools should  *
//*    be used. This is because even though the kernel tools can do the job,     *
//*    the 'gio' utility is MUCH FASTER. The kernel tools are used only if       *
//*    neither source nor target are gvfs filesystems. Some factors involved     *
//*    in the decision of which tools to use are:                                *
//*    a) Access to necessary data.                                              *
//*       In general, within the FMgr class, we know about only the source       *
//*       data OR the target, but not both. Only the caller (FileDlg class)      *
//*       has reliable information on both source and target. For this reason,   *
//*       for most operations, the caller selects the appropriate tool for the   *
//*       operation.                                                             *
//*    b) Maintaining clean (uncluttered) code style.                            *
//*       An analysis our daily file-management tasks and the reported habits    *
//*       of our beta-testers reveals that less than five(5) percent of daily    *
//*       file management involves smartphones, tablets and other gvfs devices,  *
//*       and that this is primarily for backup of photos. Therefore, it seems   *
//*       inefficient to place the selection of tools within the main flow of    *
//*       execution.                                                             *
//*    c) Speed of execution.                                                    *
//*       For low-level looped operations and other processor-intensive          *
//*       operations, we want to avoid redundant tests by keeping the test for   *
//*       gvfs filesystems and tests for restricted file operations outside      *
//*       the loop as much as possible. Therefore, the naturally less-efficient  *
//*       application-layer code is the logical place for such tests.            *
//*       Only three exceptions are made:                                        *
//*       -- The public GetFileStats() method tests the filespec to determine    *
//*          whether the target file is on a gvfs filesystem and calls the       *
//*          appropriate private method. Because application-layer requests      *
//*          for stat information are made in a wide variety of contexts,        *
//*          it was determined that the decision for stat information requests   *
//*          is best made here.                                                  *
//*       -- The public GetFilesystemStats() method tests the filespec to        *
//*          determine whether the target file is on a gvfs filesystem and       *
//*          calls the appropriate private method.                               *
//*       -- In the CopyFile_gvfs() method, we test for an intra-directory       *
//*          copy operation to compensate for a bug in the 'gio' utility which   *
//*          keeps the source directory locked while writing to the target       *
//*          directory, causing intra-directory copy operations to fail.         *
//*          -- A related bug in 'gio' is that it allows the user to write a     *
//*             file onto itself which of course fails, resulting in the source  *
//*             file being deleted. This is a major bug; however the             *
//*             application layer explicity forbids writing a file onto itself   *
//*             so we need not worry about it here.                              *
//*                                                                              *
//* 4) FileMangler-class (top layer) deals directly with temporary files only,   *
//*    and the temporary directory is assumed to be on a non-gvfs filesystem     *
//*    because the FileMangler application is running under GNU/Linux, and not   *
//*    under Android, iOS, or the Huawei OS (Android ripoff).                    *
//*                                                                              *
//* 5) FileDlg-class methods that perform file management fall into two(2)       *
//*    categories:                                                               *
//*    a) Creating, modifying and deleting temporary files. Again, all temp      *
//*       files are assumed to be on a non-gvfs filesystem.                      *
//*    b) Methods that manage files placed on the application clipboard.         *
//*       For these method, the decision of whether to use kernel tools or       *
//*       'gio' is based on the two clipboard flags indicating whether the       *
//*       source or target respectively is a gvfs filesystem.                    *
//*                                                                              *
//* 6) FMgr-class methods that perform file management are specific to either    *
//*    kernel tools or to 'gio', so caller must decide which set of methods      *
//*    to call. The two exceptions to this are listed in item 3c above.          *
//*                                                                              *
//********************************************************************************

#include "GlobalDef.hpp"
#ifndef FMGR_INCLUDED
#include "FMgr.hpp"
#endif

//***********************
//** Local Definitions **
//***********************

//* Because the 'gio' utility is not integrated into the Linux kernel, *
//* it is possible that it is not installed by default on the target   *
//* system, or that the target system does not use GVfs tools to access*
//* MPT devices. In that case, the methods below will call the         *
//* equivalent kernel-based method. When we are confident that 'gio'   *
//* will be present on all systems, this failsafe may be removed.      *
#define GVFS_FAILSAFE (1)

//* System environment variable (if present) indicating *
//* the base mountpoint path for GVfs filesystems.      *
const char* const GVFS_MNT_DIR = "XDG_RUNTIME_DIR" ;


//****************
//** Local Data **
//****************

//* Initialized on the first call to GvfsFlatScan() *
//* or by a call to USB_DeviceStats().              *
//* 'true' if 'gio' command-line utility installed. *
static bool gioInstalled = false ;

//* If 'gio' utility installed, version string.     *
static const short gvLEN = 16 ;
static char gioVersion[gvLEN] = "" ;

//* Base mountpoint directory for GVfs filesystems. *
//* Initialized by the first call to gioAvailable().*
static char gioMountBase[MAX_PATH] = "" ;

//********************
//* Local prototypes *
//********************


//*************************
//*     GvfsFlatScan      *
//*************************
//********************************************************************************
//* PRIVATE METHOD:  Called by CaptureDirTree()                                  *
//* ---------------                                                              *
//* Non-recursively scan a directory below the gvfs base path.                   *
//*             See: 'gioMountBase' at top of this module.                       *
//* This is a single-pass scan which captures the data for all files and         *
//* subdirectory names at the current level of the tree which bypasses the       *
//* need for a preliminary scan to size the TreeNode/tnFName arrays.             *
//* The single-pass scan compensates for the unbearably slow response time       *
//* when accessing smartphones and tablets. The scan can still be unbearably     *
//* slow, but requires approximately half the processing of the standard,        *
//* dual-pass algorithm.                                                         *
//*                                                                              *
//*                                                                              *
//* Input  : nodecnt : (by reference - initial counts ignored)                   *
//*                    'trgPath' contains full path specification of directory   *
//*                              to be scanned                                   *
//*                    'dirCnt' receives count of directory names                *
//*                    'regCnt' receives count of non-directory names            *
//*                    'level' member is not referenced                          *
//*                                                                              *
//* Returns: total number of filenames scanned                                   *
//* (Totals do not include current "." or parent ".." directory shortcut names)  *
//********************************************************************************
//* Notes on GVfs filesystems:                                                   *
//* --------------------------                                                   *
//* The standard scan uses a two-pass algorithm:                                 *
//* 1) Call DirectoryCount() to get the size of the arrays needed to store       *
//*    the top-level filename data.                                              *
//* 2) Call ScanDirTree() to initialize the allocated data arrays including      *
//*    an 'lstat' on each source file.                                           *
//*                                                                              *
//* Because a standard scan of the base directory file is so unbearably slow     *
//* for smartphone/tablet devices, we perform a single pass, storing the         *
//* intermediate results in a temporary file. This would be inefficient for a    *
//* real filesystem because the file I/O and the parsing would take longer       *
//* than manually reading the directory file. However, for the GVfs (virtual)    *
//* filesystem, the overhead of a temp file is not significant.                  *
//*                                                                              *
//* For directories on the target device that contain only a few (approx.20)     *
//* items, the standard scan would be fast enough. However, for directories      *
//* which contain potentially hundreds of files (e.g. /Phone/DCIM/Camera),       *
//* it can easily take a full minute to scan the top level directory, even       *
//* without recursion.                                                           *
//*                                                                              *
//* This experimental method compares the performance of 'ls' and 'gio' in       *
//* reporting the contents of the target directory.                              *
//*   1) Using the 'gio' utility seems to be both more consistent and faster     *
//*      than using the kernel's 'ls' command. Presumably this is because the    *
//*      'gio' utility operates outside the kernel, with more detailed           *
//*      knowledge of the target filesystem and therefore requires less          *
//*      processing and storage overhead.                                        *
//*   2) Preliminary testing of the 'gio' utility shows that the break point     *
//*      between acceptably-fast and unbelievably-slow is between 30 and 50      *
//*      files.                                                                  *
//*      a) It seems that directory names scan much faster that regular files,   *
//*         but this is a subjective evaluation based on only three MTP/GVfs     *
//*         devices of varying ages and firmware versions. This performance      *
//*         differential may be due to the fact that malware scanners see        *
//*         directory files as passive, that is, unable to copy or execute       *
//*         malware; and therefore require less screening.                       *
//*   3) Even though 'gio' seems clearly superior to 'ls' for virtual            *
//*      filesystems, we will retain the 'ls' scan code in case 'gio' is not     *
//*      installed on the target system. When we have confidence that 'gio'      *
//*      is reliably present on all systems, we will remove the 'ls' code.       *
//*   4) Because there is no recursion with either of these methods, the user    *
//*      will not be able to perform copy/cut/delete/find/etc. operations on     *
//*      directory trees within a virtual filesystem. In fact, the 'gio'         *
//*      utility reports an error if an attempt is made to recursively copy      *
//*      a directory tree.                                                       *
//*                                                                              *
//*  ---  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ---  *
//* The 'gio' command-line utility is designed specifically for GNOME virtual    *
//* filesystems. The equivalent library is: libgio-2.0 (bundled with glib).      *
//* See: https://developer.gnome.org/gio/stable/ for additional information.     *
//*                                                                              *
//* The following 'gio' command yields most of what we need to create the        *
//* data array.                                                                  *
//*                                                                              *
//*   gio list -nh                                                               *
//*        --attributes=standard::display-name,standard::content-type,           *
//*                     access::can-read,access::can-write,                      *
//*                     access::can-execute,time::modified                       *
//*                                                                              *
//* Yields:       +-------+--[tab]----+                        [space]------+    *
//* Somefile-04.a   14612   (regular)   standard::display-name=somefile.03.a     *
//*                                     standard::content-type=aaaaa/xxx         *
//*                                     access::can-read=TRUE                    *
//*                                     access::can-write=TRUE                   *
//*                                     access::can-execute=FALSE                *
//*                                     time::modified=1367469540                *
//*                                                                              *
//* standard::content-type is essentially the MIME type                          *
//* Where "xxx" is: directory, symlink, socket, chardevice, blockdevice, fifo.   *
//* We reference this field only when the filetype == "(special)".               *
//* Regular files have a content-type that is too specific, while directories    *
//* and symbolic links are adequately identified by the value in parentheses.    *
//* If the 'gio' utility encounters a filetype it does not know, it simply       *
//* omits the entire token. This is incredibly poor programming practice, and    *
//* we must be aware of this during parsing.                                     *
//*                                                                              *
//* Note that there appears to be a bug (undocumented annoyance) with the        *
//*    gio -d                                                                    *
//* option. It works as expected for real filesystems (edxt3/ext4), but for      *
//* GVfs filesystems it does not display the filename as one of the attributes.  *
//* For this reason, we explicity use the 'standard::display-name' attribute.    *
//*                                                                              *
//* Additional Available Fields:                                                 *
//* ----------------------------                                                 *
//* Note that not all of these tokens generate output in all cases.              *
//* It may be that the target device does not support that data. It also may     *
//* be true that the 'gio' utility is not particularly robust or that it is      *
//* incompletely implemented.                                                    *
//*                                                                              *
//*    standard::edit-name             time::changed                             *
//*    standard::copy-name             time::changed-usec                        *
//*    standard::icon                  unix::device                              *
//*    standard::fast-content-type     unix::inode                               *
//*    standard::allocated-size        unix::mode                                *
//*    standard::symbolic-icon         unix::nlink                               *
//*    etag::value                     unix::uid                                 *
//*    id::file                        unix::gid                                 *
//*    id::filesystem                  unix::rdev                                *
//*    access::can-execute             unix::block-size                          *
//*    access::can-delete              unix::blocks                              *
//*    access::can-trash               owner::user                               *
//*    access::can-rename              owner::user-real                          *
//*    time::modified-usec             owner::group                              *
//*    time::access                    selinux::context                          *
//*    time::access-usec               xattr-sys::security.selinux               *
//*                                                                              *
//*                  ----  ----  ----  ----  ----  ----  ----                    *
//* Because we do not have full stats for each file, some operations, for        *
//* instance 'display stats' will not be fully supported; however, we            *
//* synthesize as much of the stat information as possible:                      *
//*  a) file type                                                                *
//*  b) file size                                                                *
//*  c) user read/write/execute permissions                                      *
//*  d) modification timestamp                                                   *
//*  e) access timestamp                                                         *
//*  f) change (stat) timestamp                                                  *
//*  g) assume that UID and GID are the current user                             *
//* Other fields are set to default (zero) values.                               *
//*                                                                              *
//*                                                                              *
//*  ---  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ---  *
//* The following 'ls' command yields most of what we need to create the         *
//* data array.                                                                  *
//*                                                                              *
//* ls --time-style="+%Y-%m-%d %H:%M:%S" --quoting-style=c -lgG                  *
//* yields:                                                                      *
//* drwx------. 1        0 2018-08-13 13:07:28 "Rockferry"                       *
//* -rw-------. 1 10143808 2018-08-13 13:07:27 "01-Rockferry - Duffy.mp3"        *
//* -rw-------. 1  9082060 2018-08-13 13:07:26 "02-Warwick Avenue - Duffy.mp3"   *
//*    This includes the mode bits, hard-link count, full (non-ISO) timestamp    *
//*    and double-quotes enclosing the filename.                                 *
//*                                                                              *
//* It is important to note that the 'ls' utility is no faster than our manual   *
//* directory scan, so the scan can still be extremely slow, but the single      *
//* read of the directory theoretically cuts processing time in half.            *
//* The down-side is twofold:                                                    *
//* 1) The overhead of creating and parsing the temp file, and                   *
//* 2) The loss of some data due to not performing an 'lstat' on each file.      *
//*    The basic information is available by decoding the entries in the         *
//*    temp file, and is used to initialize the relevant fields of the arrays.   *
//*    -- filename           (in double quotes at end of record)                 *
//*    -- filetype           (first character of record)                         *
//*    -- read permission    (second character of record)                        *
//*    -- write permission   (third character of record)                         *
//*    -- execute permission (fourth character of record)                        *
//*    -- file size in bytes (3rd field of record)                               *
//*    -- modification timestamp (according to the format shown above)           *
//*       Note that the idiotic use of two timestamp formats for "recent' vs.    *
//*       "non-recent" is avoided here,                                          *
//*    -- It is assumed that all entries are on the same filesystem.             *
//*    -- Accumulators:                                                          *
//*       tnPtr->totFiles                                                        *
//*       tnPtr->totBytes                                                        *
//*       tnPtr->wprotFiles                                                      *
//*       tnPtr: file-type totals (dirFiles, regFiles, etc.)                     *
//*              file-size totals (dirBytes, regBytes, etc.)                     *
//*                                                                              *
//********************************************************************************

UINT FMgr::GvfsFlatScan ( nodeCount& nodecnt, TreeNode* tnPtr )
{
   const char* lsCmd = 
   "ls --time-style=\"+%Y-%m-%dT%H:%M:%S\" --quoting-style=c -lgG" ;
   const char* gioCmd = 
   "gio list -nh --attributes=standard::display-name,standard::content-type,"
                      "access::can-read,access::can-write,access::can-execute,"
                      "time::modified ./." ;
   const char* gioNameTag = "standard::display-name=" ;
   const char* gioTypeTag = "standard::content-type=" ;
   const char* gioRaccTag = "access::can-read=" ;
   const char* gioWaccTag = "access::can-write=" ;
   const char* gioXaccTag = "access::can-execute=" ;
   const char* gioTimeTag = "time::modified=" ;
   const char* gioDtype   = "(directory)" ;
   const char* gioRtype   = "(regular)" ;
   const char* gioLtype   = "(symlink)" ;
   //const char* gioStype   = "(special)" ; // chDev, bkDev, FIFO, etc.
   const char* gioTcName  = "chardevice" ;
   const char* gioTbName  = "blockdevice" ;
   const char* gioTsName  = "socket" ;
   const char* gioTfName  = "fifo" ;

   gString tmpSpec,                    // filespec of temp file
           gsRec,                      // for parsing the record
           gstmp ;
   char    inbuff[gsDFLTBYTES] ;       // input buffer
   UINT    fnIndex = ZERO,             // index into tnFName array
           tnIndex = ZERO ;            // index into TreeNode array

   nodecnt.dirCnt = nodecnt.regCnt = ZERO ;  // initialize accumulators

   //* Create a temporary file to hold the results of the scan.*
   if ( (this->CreateTempname ( tmpSpec )) )
   {
      ifstream ifs ;       // temp-file input stream

      //* Verify whether the 'gio' utility is available. *
      if ( ! gioInstalled )
         this->gioAvailable ( tmpSpec ) ;

      //* Create the command.*
      if ( gioInstalled )
      {
         gsRec.compose( "%s 1>\"%s\" 2>/dev/null", gioCmd, tmpSpec.ustr() ) ;
      }
      else
         gsRec.compose( "%s 1>\"%s\" 2>/dev/null", lsCmd, tmpSpec.ustr() ) ;

      //* Call the utility to do the scan for us.*
      if ( (this->Systemcall ( gsRec.ustr() )) == OK )
      {
         //* Open the temp file.*
         ifs.open ( tmpSpec.ustr(), ifstream::in ) ;
         if ( ifs.is_open() )
         {
            //* Count the number of records.*
            ifs.getline ( inbuff, gsDFLTBYTES ) ;
            while ( ifs.gcount() > ZERO )
            {
               //* Test record format *
               if ( gioInstalled )           // 'gio' command
               {
                  gsRec = inbuff ;
                  if ( (gsRec.find( gioNameTag )) > ZERO )
                  {
                     if ( (gsRec.find( gioDtype )) > ZERO )
                        ++nodecnt.dirCnt ;
                     else
                        ++nodecnt.regCnt ;
                  }
               }
               else                          // 'ls' command
               {
                  // Programmer's Note: We do not test for special or 
                  // system-specific filetypes: 'C' 'D' 'M' 'n' 'P'.
                  if ( ((inbuff[1] == 'r') || (inbuff[1] == '-')) &&
                       ((inbuff[0] == '-') || (inbuff[0] == 'd') ||   // reg/dir
                        (inbuff[0] == 'l') || (inbuff[0] == 'p') ||   // lnk/pipe(FIFO)
                        (inbuff[0] == 'b') || (inbuff[0] == 'c') ||   // block/char
                        (inbuff[0] == 's') || (inbuff[0] == '?')) )   // socket/unknown
                  {
                     if ( inbuff[0] == 'd' )
                        ++nodecnt.dirCnt ;
                     else
                        ++nodecnt.regCnt ;
                  }
               }
               ifs.getline ( inbuff, (gsDFLTBYTES) ) ;
            }     // while()

            //* Allocate the array of TreeNode objects (if needed).*
            if ( (tnPtr->nlnodes = nodecnt.dirCnt) > ZERO )
            {
               tnPtr->nextLevel = this->Allocate_TreeNodes ( tnPtr->nlnodes ) ;
               for ( UINT i = ZERO ; i < tnPtr->nlnodes ; ++i )
               {
                  tnPtr->nextLevel[i].level = tnPtr->level + 1 ;  // next lower level
                  tnPtr->nextLevel[i].prevLevel = tnPtr ;         // point to parent
               }
               // Programmer's Note: tnPtr->dirFiles is initialized inside the loop below.
            }

            //* Allocate the array of tnFName objects (if needed).*
            if ( (tnPtr->tnfcount = nodecnt.regCnt) > ZERO )
            {
               tnPtr->tnFiles = this->Allocate_tnFNames ( tnPtr->tnfcount ) ;
               tnPtr->tnFCount = tnPtr->tnfcount ; // user-visible file count
               tnPtr->allocBytes += sizeof(tnFName) * tnPtr->tnfcount ;
            }

            //* Parse the temporary file to extract the individual filename entries.*
            char     fname[MAX_FNAME],    // filename
                     mbits[16] ;          // discarded mode bits
            localTime lt ;                // modification timestamp
            uint64_t st_size,             // file size
                     st_nlink ;           // number of hard links
            char     ftype,               // character representing filetype
                     racc,                // character representing read-access flag
                     wacc,                // character representing write-access flag
                     xacc ;               // character representing execute-access flag
            const wchar_t *wPtr ;         // pointer to record text
            short    indx,                // general index
                     nindx,               // name index
                     tindx,               // content-type index
                     sindx,               // file-size index
                     aindx ;              // r/w access index
            bool     goodRecord ;         // 'true' if valid record format

            ifs.clear() ;                 // reset the EOF flag
            ifs.seekg( ZERO ) ;           // return to top of input file
            ifs.getline ( inbuff, gsDFLTBYTES ) ;
            while ( ifs.gcount() > ZERO )
            {
               gstmp.clear() ;            // initialize temp variables
               lt.reset() ;
               st_size  = ZERO ;
               st_nlink = ZERO ;
               ftype    = '?' ;
               racc = wacc = xacc = '-' ;
               goodRecord = false ;       // initialize the flag
               gsRec = inbuff ;           // get a copy of the record

               //* Test for 'gio' command format *
               if ( gioInstalled && (nindx = gsRec.find( gioNameTag )) > ZERO )
               {
                  //* Parse the line *
                  wPtr = gsRec.gstr() ;   // wide-char pointer

                  //* Index the filetype *
                  for ( indx = (nindx - 1) ; indx > ZERO ; --indx )
                  {
                     if ( wPtr[indx] == L'(' )
                     {
                        //* Index the file-size field (delimited by TAB chars) *
                        for ( sindx = (indx - 2) ; sindx > ZERO ; --sindx )
                        {
                           if ( wPtr[sindx] == nckTAB )
                           {
                              gsRec.gscanf( sindx, L" %llu", &st_size ) ;
                              break ;
                           }
                        }

                        if ( (gsRec.find( gioRtype, indx )) == indx )
                           ftype = '-' ;
                        else if ( (gsRec.find( gioDtype, indx )) == indx )
                           ftype = 'd' ;
                        else if ( (gsRec.find( gioLtype, indx )) == indx )
                           ftype = 'l' ;
                        //* "(special)" or unknown type:      *
                        //* Scan the value for "content-type" *
                        else
                        {
                           // If the 'gio' utility can supply the filetype, *
                           if ( (tindx = gsRec.after( gioTypeTag )) >= ZERO )
                           {
                              if ( (gsRec.find( gioTcName, false, tindx )) > tindx )
                                 ftype = 'c' ;     // character device
                              else if ( (gsRec.find( gioTbName, tindx )) > tindx )
                                 ftype = 'b' ;     // block device
                              else if ( (gsRec.find( gioTsName, tindx )) > tindx )
                                 ftype = 's' ;     // socket device
                              else if ( (gsRec.find( gioTfName, tindx )) > tindx )
                                 ftype = 'p'  ;    // fifo (pipe) device
                              else
                                 ftype = '?' ;     // unknown type
                           }
                           else
                              ftype = '?' ;        // unknown type
                        }
                        break ;
                     }
                  }        // for(;;)

                  //* Extract the filename *
                  nindx = gsRec.after( gioNameTag, nindx ) ;   // 1st char of filename
                  tindx = gsRec.find( gioTypeTag, nindx ) ;
                  //* If gioTypeTag not found, it means that the file *
                  //* is not affiliated with a known MIME type. When  *
                  //* this happens, set 'tindx' on the next token.    *
                  if ( tindx < ZERO )
                     tindx = gsRec.find( gioRaccTag, nindx ) ;
                  while ( nindx < (tindx - 1) )                // last char of filename
                     gstmp.append( wPtr[nindx++] ) ;

                  //* Extract the read/write/execute access flags *
                  aindx = gsRec.after( gioRaccTag, tindx ) ;
                  if ( wPtr[aindx] == L'T' )
                     racc = 'r' ;
                  aindx = gsRec.after( gioWaccTag, aindx ) ;
                  if ( wPtr[aindx] == L'T' )
                     wacc = 'w' ;
                  aindx = gsRec.after( gioXaccTag, aindx ) ;
                  if ( wPtr[aindx] == L'T' )
                     xacc = 'x' ;

                  //* Decode the (epoch) timestamp *
                  indx = gsRec.after( gioTimeTag, aindx ) ;
                  gsRec.gscanf( indx, L" %llu", &lt.epoch ) ;
                  this->DecodeEpochTime ( lt.epoch, lt ) ;

                  goodRecord = true ;
               }

               //* Test for 'ls' command record format *
               else if ( ((inbuff[1] == 'r') || (inbuff[1] == '-')) &&
                         ((inbuff[0] == '-') || (inbuff[0] == 'd') ||   // reg/dir
                          (inbuff[0] == 'l') || (inbuff[0] == 'p') ||   // lnk/pipe(FIFO)
                          (inbuff[0] == 'b') || (inbuff[0] == 'c') ||   // block/char
                          (inbuff[0] == 's') || (inbuff[0] == '?')) )   // socket/unknown
               {
                  //* Parse the line *
                  // Programmer's Note: An embedded double-quotation mark in the 
                  // filename will cause truncation of the filename.
                  // This is unlikely on an Android system, but we aren't sure 
                  // about iOS and the Huawei Android clone.
                  gsRec.gscanf( "%c%c%c%c%7s %llu %llu %hu-%hu-%huT%hu:%hu:%hu \"%255[^\"]\"", 
                                &ftype, &racc, &wacc, &xacc, mbits, &st_nlink, &st_size,
                                &lt.year, &lt.month, &lt.date, 
                                &lt.hours, &lt.minutes, &lt.seconds, fname ) ;
                  gstmp = fname ;                  // get a copy of the filename
                  this->EncodeEpochTime ( lt ) ;   // calculate epoch time
                  goodRecord = true ;              // valid record
               }

               //* If a valid record captured, copy the data *
               //* to the appropriate record array.          *
               if ( goodRecord )
               {
                  ++tnPtr->totFiles ;           // update global accumulators
                  tnPtr->totBytes += st_size ;

                  if ( ftype == 'd' )
                  {
                     gstmp.copy( tnPtr->nextLevel[tnIndex].dirStat.fName, MAX_FNAME ) ;
                     tnPtr->nextLevel[tnIndex].dirStat.fType = fmDIR_TYPE ;
                     tnPtr->nextLevel[tnIndex].dirStat.fBytes = st_size ;
                     tnPtr->nextLevel[tnIndex].dirStat.modTime = lt ;
                     tnPtr->nextLevel[tnIndex].dirStat.rawStats.st_mode = S_IFDIR ;
                     if ( racc == 'r' )   // user read permission
                     {
                        tnPtr->nextLevel[tnIndex].dirStat.readAcc = true ;
                        tnPtr->nextLevel[tnIndex].dirStat.rawStats.st_mode |= S_IRUSR ;
                     }
                     else                 // user write permission
                        tnPtr->nextLevel[tnIndex].dirStat.readAcc = false ;
                     if ( wacc == 'w' )
                     {
                        tnPtr->nextLevel[tnIndex].dirStat.writeAcc = true ;
                        tnPtr->nextLevel[tnIndex].dirStat.rawStats.st_mode |= S_IWUSR ;
                     }
                     else
                     {
                        tnPtr->nextLevel[tnIndex].dirStat.writeAcc = false ;
                        ++tnPtr->wprotFiles ;
                     }
                     if ( xacc == 'x' )
                     {
                        tnPtr->nextLevel[tnIndex].dirStat.rawStats.st_mode |= S_IXUSR ;
                     }
                     tnPtr->nextLevel[tnIndex].dirStat.fsMatch = true ;
                     ++tnPtr->dirFiles ;
                     tnPtr->dirBytes += st_size ;
                     //* Partially initialize raw-stats structure.*
                     tnPtr->nextLevel[tnIndex].dirStat.rawStats.st_nlink = st_nlink ;
                     tnPtr->nextLevel[tnIndex].dirStat.rawStats.st_mtime = 
                     tnPtr->nextLevel[tnIndex].dirStat.rawStats.st_atime = 
                     tnPtr->nextLevel[tnIndex].dirStat.rawStats.st_ctime = 
                              tnPtr->nextLevel[tnIndex].dirStat.modTime.epoch ;
                     tnPtr->nextLevel[tnIndex].dirStat.rawStats.st_size = 
                              tnPtr->nextLevel[tnIndex].dirStat.fBytes ;
                     tnPtr->nextLevel[tnIndex].dirStat.rawStats.st_uid = 
                              this->userInfo.userID ;
                     tnPtr->nextLevel[tnIndex].dirStat.rawStats.st_gid = 
                              this->userInfo.grpID ;
                     tnPtr->nextLevel[tnIndex].dirStat.rawStats.st_dev = ZERO ; // values unknown
                     tnPtr->nextLevel[tnIndex].dirStat.rawStats.st_ino = ZERO ;
                     tnPtr->nextLevel[tnIndex].dirStat.rawStats.st_rdev = ZERO ;
                     tnPtr->nextLevel[tnIndex].dirStat.rawStats.st_blksize = ZERO ;
                     tnPtr->nextLevel[tnIndex].dirStat.rawStats.st_blocks = ZERO ;

                     ++tnIndex ;
                  }
                  else
                  {
                     gstmp.copy( tnPtr->tnFiles[fnIndex].fName, MAX_FNAME ) ;
                     tnPtr->tnFiles[fnIndex].fBytes = st_size ;
                     tnPtr->tnFiles[fnIndex].modTime = lt ;
                     if ( racc == 'r' )   tnPtr->tnFiles[fnIndex].readAcc = true ;
                     else                 tnPtr->tnFiles[fnIndex].readAcc = false ;
                     if ( wacc == 'w' )   tnPtr->tnFiles[fnIndex].writeAcc = true ;
                     else                 tnPtr->tnFiles[fnIndex].writeAcc = false ;

                     switch ( ftype )
                     {
                        case '-':      // regular file
                           tnPtr->tnFiles[fnIndex].fType = fmREG_TYPE ;
                           tnPtr->tnFiles[fnIndex].rawStats.st_mode = S_IFREG ;
                           ++tnPtr->regFiles ;
                           tnPtr->regBytes += st_size ;
                           break ;
                        case 'l':      // symbolic link file
                           tnPtr->tnFiles[fnIndex].fType = fmLINK_TYPE ;
                           tnPtr->tnFiles[fnIndex].rawStats.st_mode = S_IFLNK ;
                           ++tnPtr->linkFiles ;
                           tnPtr->linkBytes += st_size ;
                           break ;
                        case 'p':      // FIFO file
                           tnPtr->tnFiles[fnIndex].fType = fmFIFO_TYPE ;
                           tnPtr->tnFiles[fnIndex].rawStats.st_mode = S_IFIFO ;
                           ++tnPtr->fifoFiles ;
                           tnPtr->fifoBytes += st_size ;
                           break ;
                        case 'b':      // block-device file
                           tnPtr->tnFiles[fnIndex].fType = fmBKDEV_TYPE ;
                           tnPtr->tnFiles[fnIndex].rawStats.st_mode = S_IFBLK ;
                           ++tnPtr->bdevFiles ;
                           tnPtr->bdevBytes += st_size ;
                           break ;
                        case 'c':      // character-device file
                           tnPtr->tnFiles[fnIndex].fType = fmCHDEV_TYPE ;
                           tnPtr->tnFiles[fnIndex].rawStats.st_mode = S_IFCHR ;
                           ++tnPtr->cdevFiles ;
                           tnPtr->cdevBytes += st_size ;
                           break ;
                        case 's':      // socket file
                           tnPtr->tnFiles[fnIndex].fType = fmSOCK_TYPE ;
                           tnPtr->tnFiles[fnIndex].rawStats.st_mode = S_IFSOCK ;
                           ++tnPtr->sockFiles ;
                           tnPtr->sockBytes += st_size ;
                           break ;
                        default:
                           tnPtr->tnFiles[fnIndex].fType = fmUNKNOWN_TYPE ;
                           tnPtr->tnFiles[fnIndex].rawStats.st_mode = ZERO ;
                           ++tnPtr->unkFiles ;
                           tnPtr->unkBytes += st_size ;
                           break ;
                     } ;
                     if ( tnPtr->tnFiles[fnIndex].readAcc )   // user read permission
                        tnPtr->tnFiles[fnIndex].rawStats.st_mode |= S_IRUSR ;
                     if ( tnPtr->tnFiles[fnIndex].writeAcc )  // user write permission
                        tnPtr->tnFiles[fnIndex].rawStats.st_mode |= S_IWUSR ;
                     if ( xacc == 'x' )
                        tnPtr->tnFiles[fnIndex].rawStats.st_mode |= S_IXUSR ;
                     //* Partially initialize raw-stats structure.*
                     tnPtr->tnFiles[fnIndex].rawStats.st_nlink = st_nlink ;
                     tnPtr->tnFiles[fnIndex].rawStats.st_mtime = 
                     tnPtr->tnFiles[fnIndex].rawStats.st_atime = 
                     tnPtr->tnFiles[fnIndex].rawStats.st_ctime = 
                              tnPtr->tnFiles[fnIndex].modTime.epoch ;
                     tnPtr->tnFiles[fnIndex].rawStats.st_size = 
                              tnPtr->tnFiles[fnIndex].fBytes ;
                     tnPtr->tnFiles[fnIndex].rawStats.st_uid = this->userInfo.userID ;
                     tnPtr->tnFiles[fnIndex].rawStats.st_gid = this->userInfo.grpID ;
                     tnPtr->tnFiles[fnIndex].rawStats.st_dev = ZERO ; // values unknown
                     tnPtr->tnFiles[fnIndex].rawStats.st_ino = ZERO ;
                     tnPtr->tnFiles[fnIndex].rawStats.st_rdev = ZERO ;
                     tnPtr->tnFiles[fnIndex].rawStats.st_blksize = ZERO ;
                     tnPtr->tnFiles[fnIndex].rawStats.st_blocks = ZERO ;

                     ++fnIndex ;
                  }
               }
               //* Get the next record *
               ifs.getline ( inbuff, gsDFLTBYTES ) ;
            }     // while()
            ifs.close () ;     // close the temp file
         }        // is_open()
      }           // SystemCall()

      //* Delete the temporary file.*
      this->DeleteTempname ( tmpSpec ) ;
   }

   return ( nodecnt.dirCnt + nodecnt.regCnt ) ;

}  // End GvfsFlatScan()

//*************************
//*     CopyFile_gvfs     *
//*************************
//********************************************************************************
//* Create a copy of the specified source file.                                  *
//* Preserve all permission bits and date/timestamps that are supported by the   *
//* target filesystem.                                                           *
//*                                                                              *
//* WARNING: Do not call this method to copy a 'special' file.                   *
//*          ('regular', 'symlink' and 'FIFO' files only)                        *
//*                                                                              *
//* Input  : srcPath : full path/filename specification of source                *
//*          trgPath : full path/filename specification of target                *
//*          followLink: (optional, default==false)                              *
//*                      If source is a symbolic link the following              *
//*                      applies; else the 'followLink' has no effect.           *
//*                      if false, copy the link itself                          *
//*                      if true, copy the file link points to                   *
//*                                                                              *
//* Returns: OK if successful                                                    *
//*          returns errno value if available, else EASSES                       *
//********************************************************************************
//* Notes:                                                                       *
//* ------                                                                       *
//* a) Copy from external filesystem to target gvfs filesystem works correctly.  *
//* b) Copy from gvfs filesystem to external filesystem works correctly.         *
//* c) Copy from one directory to another on gvfs filesystem works correctly.    *
//* d) Within a gvfs filesystem, an attempt to write a file onto itself will     *
//*    delete the file. This is a bug in the 'gio' utility, so watch out!        *
//*    Caller must verify that a file is not being written onto itself.          *
//*    On a "real" filesystem, copying a file onto itself silently succeeds,     *
//*    which is indicated by the file being given a new inode number.            *
//* e) Copy-with-rename within current directory of a gvfs filesystem fails;     *
//*    therefore, a two-stage copy is necessary:                                 *
//*    1) Copy source file to temp directory (with rename).                      *
//*    2) Copy temp file to target directory (then delete temp copy).            *
//* f) Note that the "--preserve" option seems to preserve the owner's           *
//*    permission bits correctly, but is not reliable for group/other            *
//*    permissions or for the modification timestamp.                            *
//*    -- Permissions for group/other may not be supported on a given gvfs       *
//*       filesystem, so it is not surprising that they are not preserved.       *
//*    -- Modification timestamp is preserved on a "real" filesystem (ext4);     *
//*       however, access timestamp may be garbage. On gvfs filesystems, mod     *
//*       timestamp to the current system time. It is unclear whether the        *
//*       access/change timestamps are supported by gvfs filesystems.            *
//*    -- See "gio info --query-writable FILENAME                                *
//*       This will indicate what attributes can be set independently for the    *
//*       file on the host filesystem.                                           *
//*       (Hint: On gvfs, only metadata is writable.)                            *
//*       If mod timestamp is "writable" the the following is allowed:           *
//*          gio set --type=uint64 test1.txt time::modified 1546567890           *
//*                                                                              *
//********************************************************************************

short FMgr::CopyFile_gvfs ( const char* srcPath, const char* trgPath, bool followLink )
{
   const char* const gioTemplate = "gio copy --preserve %s \"%s\" \"%s\" 1>%s 2>%s" ;
   const char* const dref = " " ;
   const char* const noderef = "--no-dereference" ;
   const char* lnkcmd = followLink ? dref : noderef ; // following symbolic link?
   gString spath( srcPath ), tpath( trgPath ) ; // working copy of source and target
   const short bSIZE = MAX_PATH * 3 ;           // size of command buffer
   char cmd[bSIZE] ;                            // command buffer
   short status = OK ;                          // return value
   bool  tmpsource = false ;                    // 'true' if temporary source file

   #if GVFS_FAILSAFE != 0     // Note the early return
   if ( ! gioInstalled )
      return ( (this->CopyFile ( srcPath, trgPath, followLink )) ) ;
   #endif   // GVFS_FAILSAFE

   //* Create a temporary file *
   gString tmpPath ;
   this->CreateTempname ( tmpPath ) ;

   //* Test for intra-directory copy (see notes above).*
   if ( (this->isGvfsPath ( spath )) )
   {
      short spindx = spath.findlast( L'/' ),
            tpindx = tpath.findlast( L'/' ) ;
      
      if ( (spindx > ZERO) && (tpindx > ZERO) && (spindx == tpindx) )
      {
         gString sp = spath, tp = tpath ;
         sp.limitChars( spindx ) ;
         tp.limitChars( tpindx ) ;
         //* If source directory == target directory, *
         //* copy source file to temporary location.  *
         if ( sp == tp )      // if source dir == target dir
         {
            tp.compose( "%s%S", this->tfPath, &tpath.gstr()[tpindx] ) ;
            snprintf ( cmd, bSIZE, gioTemplate, lnkcmd, spath.ustr(), 
                          tp.ustr(), tmpPath.ustr(), tmpPath.ustr() ) ;
            if ( (status = this->Systemcall ( cmd )) == OK )
            {
               spath = tp ;      // source is temp file
               tmpsource = true ;
            }
         }
      }
   }

   //* Create the command *
   snprintf ( cmd, bSIZE, gioTemplate, lnkcmd, spath.ustr(), tpath.ustr(), 
                          tmpPath.ustr(), tmpPath.ustr() ) ;

   //* Execute the command *
   if ( (status = this->Systemcall ( cmd )) == OK )
   {
      //* If successful, the call will produce *
      //* no output, so output == error.       *
      //* If error: "gio: ...."                *
      tnFName tnf ;
      this->GetFileStats ( tnf, tmpPath.ustr() ) ;
      if ( tnf.fBytes > ZERO )
         status = EACCES ;    // assume that access was denied
   }
   if ( status != OK )
   {
      status = this->recentErrno = EACCES ;  // assume access violation
      this->DebugLog ( "In CopyFile_gvfs: copy failed", status, trgPath ) ;
   }

   //* If two-stage copy, delete temp source file.*
   if ( tmpsource )
      this->DeleteFile ( spath.ustr() ) ;

   //* Delete the temp file
   this->DeleteTempname ( tmpPath ) ;

   return status ;

}  //* End CopyFile_gvfs() *

//*************************
//*    DeleteFile_gvfs    *
//*************************
//********************************************************************************
//* Delete the specified source file.                                            *
//* The 'gio' utility silently overrides write protection, so caller must test   *
//* for write access.                                                            *
//*                                                                              *
//* Input  : srcPath : full path/filename specification of source                *
//*                                                                              *
//* Returns: OK if successful, else returns errno value                          *
//********************************************************************************

short FMgr::DeleteFile_gvfs ( const char* srcPath )
{
   const char* const gioTemplate = "gio remove %s 1>%s 2>%s" ;
   short status = ERR ;

   #if GVFS_FAILSAFE != 0     // Note the early return
   if ( ! gioInstalled )
      return ( (this->DeleteFile ( srcPath )) ) ;
   #endif   // GVFS_FAILSAFE

   //* Create a temporary file *
   gString tmpPath ;
   this->CreateTempname ( tmpPath ) ;

   //* Create and execute the command *
   const short bSIZE = MAX_PATH * 3 ;  // size of command buffer
   char cmd[bSIZE] ;
   snprintf ( cmd, bSIZE, gioTemplate, srcPath, tmpPath.ustr(), tmpPath.ustr() ) ;

   if ( (status = this->Systemcall ( cmd )) == OK )
   {
      //* If successful, the call will produce *
      //* no output, so output == error.       *
      //* If error: "gio: ...."                *
      tnFName tnf ;
      this->GetFileStats ( tnf, tmpPath.ustr() ) ;
      if ( tnf.fBytes > ZERO )
         status = EACCES ;    // assume that access was denied
   }

   //* Delete the temp file
   this->DeleteTempname ( tmpPath ) ;

   return status ;

}  //* End DeleteFile_gvfs() *

//*************************
//* CreateDirectory_gvfs  *
//*************************
//********************************************************************************
//* Create a new subdirectory.                                                   *
//* The mode bits will be set to the default for the filesystem.                 *
//*    Example: all-access-for-user: drwx---------                               *
//*                                  drwxr-x-r-x--                               *
//* The timestamp will be the current local time.                                *
//*                                                                              *
//* Input  : dirPath : full path/filename specification for new directory        *
//*                                                                              *
//* Returns: OK if successful, else returns errno value                          *
//********************************************************************************
//* Notes:                                                                       *
//* ------                                                                       *
//* a) When the directory is created, the timestamp is initially reported        *
//*    as 0 (zero), and it takes considerable time for the timestamp to be       *
//*    reported correctly.                                                       *
//*                                                                              *
//********************************************************************************

short FMgr::CreateDirectory_gvfs ( const char* dirPath )
{
   const char* const gioTemplate = "gio mkdir '%s' 1>%s 2>%s" ;
   short status = ERR ;

   #if GVFS_FAILSAFE != 0     // Note the early return
   if ( ! gioInstalled )
      return ( (this->CreateDirectory ( dirPath )) ) ;
   #endif   // GVFS_FAILSAFE

   //* Create a temporary file *
   gString tmpPath ;
   this->CreateTempname ( tmpPath ) ;

   //* Create and execute the command *
   const short bSIZE = MAX_PATH * 3 ;  // size of command buffer
   char cmd[bSIZE] ;
   snprintf ( cmd, bSIZE, gioTemplate, dirPath, tmpPath.ustr(), tmpPath.ustr() ) ;

   if ( (status = this->Systemcall ( cmd )) == OK )
   {
      //* If successful, the call will produce *
      //* no output, so output == error.       *
      //* If error: "gio: ...."                *
      tnFName tnf ;
      this->GetFileStats ( tnf, tmpPath.ustr() ) ;
      if ( tnf.fBytes > ZERO )
         status = EACCES ;    // assume that access was denied
   }
   if ( status != OK )
   {
      status = this->recentErrno = errno ;         // save 'errno' value
      this->DebugLog ( "In CreateDirectory_gvfs: mkdir failed", status, dirPath ) ;
   }

   //* Delete the temp file
   this->DeleteTempname ( tmpPath ) ;

   return status ;

}  //* End CreateDirectory_gvfs() *

//*************************
//*    isEmptyDir_gvfs    *
//*************************
//********************************************************************************
//* Scans the contents of the specified directory and returns 'true' if          *
//* directory contains no files (other than "." and "..").                       *
//*                                                                              *
//* Input  : dirPath : full path specification of source                         *
//*                                                                              *
//* Returns: 'true' if directory is empty                                        *
//*          'false' if directory not empty OR if target is not a directory      *
//********************************************************************************

bool FMgr::isEmptyDir_gvfs ( const char* dirPath )
{
   const char* const gioTemplate = "gio list -nh \"%s\" 1>%s 2>%s" ;
   tnFName tnf ;              // target stats
   bool isEmpty = false ;     // return value

   #if GVFS_FAILSAFE != 0     // Note the early return
   if ( ! gioInstalled )
      return ( (this->isEmptyDir ( dirPath )) ) ;
   #endif   // GVFS_FAILSAFE

   //* Verify that target is a directory.*
   if ( ((this->GetFileStats_gvfs ( tnf, dirPath )) == OK) &&
        (tnf.fType == fmDIR_TYPE) )
   {
      //* Create a temporary file *
      gString tmpPath ;
      this->CreateTempname ( tmpPath ) ;

      //* Create and execute the command *
      const short bSIZE = MAX_PATH * 3 ;  // size of command buffer
      char cmd[bSIZE] ;
      snprintf ( cmd, bSIZE, gioTemplate, dirPath, tmpPath.ustr(), tmpPath.ustr() ) ;

      if ( (this->Systemcall ( cmd )) == OK )
      {
         //* If directory is empty, the call will  *
         //* produce no output, so output == error.*
         tnFName tnf ;
         this->GetFileStats ( tnf, tmpPath.ustr() ) ;
         if ( tnf.fBytes == ZERO )
            isEmpty = true ;
      }

      //* Delete the temp file
      this->DeleteTempname ( tmpPath ) ;
   }
   return isEmpty ;

}  //* End isEmptyDir_gvfs() *

//*************************
//*  DeleteEmptyDir_gvfs  *
//*************************
//********************************************************************************
//* Delete the specified directory.                                              *
//* The 'gio' utility does not perform recursive operations such as deletion     *
//* of a directory tree, so caller must verify that target directory is empty.   *
//*                                                                              *
//* Input  : dirPath : full path specification of source                         *
//*                                                                              *
//* Returns: OK if successful, else returns errno value                          *
//********************************************************************************

short FMgr::DeleteEmptyDir_gvfs ( const char* dirPath )
{

   #if GVFS_FAILSAFE != 0     // Note the early return
   if ( ! gioInstalled )
      return ( (this->DeleteEmptyDir ( dirPath )) ) ;
   #endif   // GVFS_FAILSAFE

   return ( (this->DeleteFile_gvfs ( dirPath )) ) ;

}  //* End DeleteEmptyDir_gvfs() *

//*************************
//*   CopyDirName_gvfs    *
//*************************
//********************************************************************************
//* Create a copy of the specified Directory-type source file.                   *
//* This method, insofar as possible, mirrors the corresponding method for       *
//* "real" filesystems.                                                          *
//*                                                                              *
//* Directories are a special case in that:                                      *
//*  - if target directory does not exist, create the directory, preserving      *
//*    all the source file's permission bits and both 'modified' and 'access'    *
//*    date/timestamps are set to source file's mod date/time.                   *
//*    - Exception: If a directory is created, then owner (application's user)   *
//*      is given write access on the assumption that the source directory's     *
//*      contents are about to be copied into the new target directory.          *
//*  - if a target file of the same name already exists, do not overwrite it.    *
//*    - if the existing target file is of Directory-type AND user has write     *
//*      access to it, do nothing and report that file was copied successfully.  *
//*    - if the existing target file is of Directory-type BUT user does not      *
//*      have write access to it,                                                *
//*      return 'EACCES - File write protected'                                  *
//*    - if existing target file is not Directory-type,                          *
//*      return 'ENOTDIR - Not a directory'                                      *
//*                                                                              *
//* Input  : srcPath : full path/filename specification of source                *
//*          trgPath : full path/filename specification of target                *
//*                                                                              *
//* Returns: OK if successful, else returns errno value                          *
//********************************************************************************

short FMgr::CopyDirName_gvfs ( const char* srcPath, const char* trgPath )
{
   tnFName  srcStats, trgStats ;
   short status ;                   // return value

   #if GVFS_FAILSAFE != 0     // Note the early return
   if ( ! gioInstalled )
      return ( (this->CopyDirName ( srcPath, trgPath )) ) ;
   #endif   // GVFS_FAILSAFE

   //* Get stats of source file. *
   if ( (status = this->GetFileStats_gvfs ( srcStats, srcPath )) == OK )
   {
      //* Verify that source is a directory *
      if ( srcStats.fType == fmDIR_TYPE )
      {
         //* Get stats of target file (if it exists) *
         if ( (status = this->GetFileStats_gvfs ( trgStats, trgPath )) == OK )
         {
            //* Target exists, verify that it is a directory *
            //* and that user has write permission.          *
            if ( trgStats.fType == fmDIR_TYPE )
            {
               if ( trgStats.writeAcc )
                  status = OK ;        // existing target directory with write access
               else
                  status = EACCES ;    // existing target directory write-protected
            }
            else
               status = ENOTDIR ;      // existing target IS NOT a directory
         }
         //* Else, target does not exist - create it.*
         else
         {
            status = this->CreateDirectory_gvfs ( trgPath ) ;
         }
      }
      else  // source file is not a directory - caller screwed up
         status = ENOTDIR ;
   }
   return status ;

}  //* End CopyDirName_gvfs()

//*************************
//*    RenameFile_gvfs    *
//*************************
//********************************************************************************
//* Rename the specified source file to specified destination file.              *
//*                                                                              *
//* Input  : srcPath : full path/filename specification of source                *
//*          trgPath : full path/filename specification of target                *
//*                                                                              *
//* Returns: OK if successful, else returns errno value                          *
//********************************************************************************
//* Programmer's Note: The 'gio' utility insists that the target filespec be     *
//* the filename ONLY. The target path may not be specified. Therefore, we       *
//* extract the filename from 'trgPath' to create the command.                   *
//********************************************************************************

short FMgr::RenameFile_gvfs ( const char* srcPath, const char* trgPath )
{
   const char* const gioTemplate = "gio rename \"%s\" \"%s\" 1>%s 2>%s" ;
   const wchar_t* const errorToken  = L"gio: " ;
   char trgName[MAX_FNAME] ;
   short status = ERR ;

   #if GVFS_FAILSAFE != 0     // Note the early return
   if ( ! gioInstalled )
      return ( (this->RenameFile ( srcPath, trgPath )) ) ;
   #endif   // GVFS_FAILSAFE

   //* Extract the target filename. (see note above) *
   this->ExtractFilename ( trgName, trgPath ) ;

   //* Create a temporary file *
   gString tmpPath ;
   this->CreateTempname ( tmpPath ) ;

   //* Create the command *
   const short bSIZE = MAX_PATH * 3 ;  // size of command buffer
   char cmd[bSIZE] ;
   snprintf ( cmd, bSIZE, gioTemplate, srcPath, trgName, 
                          tmpPath.ustr(), tmpPath.ustr() ) ;

   //* Execute the command *
   if ( (status = this->Systemcall ( cmd )) == OK )
   {
      //* If successful, the file will contain a single line *
      //* of the form: "Rename successful. New uri: ...."    *
      //* If error: "gio: Error renaming ...."  or similar.  *
      ifstream ifs( tmpPath.ustr(), ifstream::in ) ;
      if ( ifs.is_open() )
      {
         ifs.getline ( cmd, bSIZE ) ;
         gString gs( cmd ) ;
         if ( (gs.find( errorToken )) == ZERO )
            status = EACCES ; // assume no target access
         ifs.close() ;
      }
   }

   //* Delete the temp file *
   this->DeleteTempname ( tmpPath ) ;

   return status ;

}  //* End RenameFile_gvfs() *

//*************************
//*   GetFileStats_gvfs   *
//*************************
//********************************************************************************
//* Private Method                                                               *
//* --------------                                                               *
//* Gather statistics on the target file.                                        *
//*                                                                              *
//* Initialize the tnFName-class object using data returned by the 'gio'         *
//* utility. If sufficient data are reported, then all data members will be      *
//* initialized. If data for a member is not available, that member will         *
//* receive a default value.                                                     *
//*                                                                              *
//* Input  : trgStat: tnFName class object (by reference) to hold 'stat' data    *
//*          trgPath: full path/filename specification                           *
//*          linkTrg: (optional, false by default) if true, symbolic links are   *
//*                   followed. Else, do not follow links                        *
//*                                                                              *
//* Returns: OK if successful, all data members of tnFile will be initialized    *
//*          Else returns system 'errno' value                                   *
//*           If error during system call, all fields of 'trgStat' will be       *
//*           set to a default value.                                            *
//********************************************************************************
//* Notes:                                                                       *
//* ------                                                                       *
//* 'gio info FILESPEC' displays the available information about the target      *
//* file. The reported fields will vary depending upon the target filesystem.    *
//* and other factors. The following is for a typical image file on a generic    *
//* smartphone.                                                                  *
//*                                                                              *
//* gio info /run/user/$UID/gvfs/                                                *
//*                mtp:host=SAMSUNG_SAMSUNG_Android_ab012345c012345d01/          *
//*                Phone/DCIM/Camera/20200702_215040.jpg                         *
//* Yields:                                                                      *
//* -------                                                                      *
//* display name: 20200702_215040.jpg                                            *
//* name: 20200702_215040.jpg                                                    *
//* type: regular                                                                *
//* size:  3990828                                                               *
//* uri: mtp://SAMSUNG_SAMSUNG_Android_ab012345c012345d01/Phone/DCIM/Camera/     *
//*            20200702_215040.jpg                                               *
//* local path: /run/user/1000/gvfs/                                             *
//*                mtp:host=SAMSUNG_SAMSUNG_Android_ab012345c012345d01/          *
//*                Phone/DCIM/Camera/20200702_215040.jpg                         *
//* unix mount: gvfsd-fuse /run/user/1000/gvfs fuse.gvfsd-fuse rw,nosuid,        *
//*                                nodev,relatime,user_id=1000,group_id=1000     *
//* attributes:                                                                  *
//*   standard::type: 1                                                          *
//*   standard::name: 20200702_215040.jpg                                        *
//*   standard::display-name: 20200702_215040.jpg                                *
//*   standard::icon: image-jpeg, image-x-generic, image-jpeg-symbolic,          *
//*                   image-x-generic-symbolic                                   *
//*   standard::content-type: image/jpeg                                         *
//*   standard::fast-content-type: image/jpeg                                    *
//*   standard::size: 3990828                                                    *
//*   standard::symbolic-icon: image-jpeg-symbolic, image-x-generic-symbolic,    *
//*                            image-jpeg, image-x-generic                       *
//*   id::file: mtp:host=SAMSUNG_SAMSUNG_Android_ab012345c012345d01:10601        *
//*   id::filesystem: mtp:host=SAMSUNG_SAMSUNG_Android_ab012345c012345d01        *
//*   access::can-read: TRUE                                                     *
//*   access::can-write: TRUE                                                    *
//*   access::can-execute: FALSE                                                 *
//*   access::can-delete: TRUE                                                   *
//*   access::can-trash: FALSE                                                   *
//*   access::can-rename: TRUE                                                   *
//*   time::modified: 1593741040                                                 *
//*   time::modified-usec: 0                                                     *
//*   preview::icon: GVfsIcon:0x7f3ce800f040                                     *
//*                                                                              *
//* The tnFName object is initialized:                                           *
//* fName      : (from 'trgPath')                                                *
//* fType      : standard::content-type                                          *
//*              standard::type (1==reg,2==dir,3==symlink,4==chdev/bkdev)        *
//*              (Note that when '--nofollow-symlinks' is not specified,)        *
//*              (standard::type is the type of the link target         )        *
//* fBytes     : standard::size                                                  *
//* modTime    : time::modified                                                  *
//* readAcc    : access::can-read                                                *
//* writeAcc   : access::can-write                                               *
//* fsMatch    : n/a                                                             *
//* rawStats   : -----                                                           *
//*  st_dev    : id::file: device:inode   (or 0)                                 *
//*  st_ino    : id::file: device:inode   (or 0)                                 *
//*  st_mode   : filetype | can-read | can-write | can-execute                   *
//*  st_nlink  : unix::nlink      (or 0)                                         *
//*  st_uid    : unix::uid        (or this->userInfo.userID)                     *
//*  st_gid    : unix::gid        (or this->userInfo.userID)                     *
//*  st_rdev   : unix::rdev       (or 0)                                         *
//*  st_size   : standard::size                                                  *
//*  st_blksize: unix::block-size (or 0)                                         *
//*  st_blocks : unix::blocks     (or 0)                                         *
//*  st_mtime  : time::modified                                                  *
//*  st_atime  : time::access     (or time::modified)                            *
//*  st_ctime  : time::changed    (or time::modified)                            *
//*                                                                              *
//********************************************************************************

short FMgr::GetFileStats_gvfs ( tnFName& trgStat, const char* trgPath, bool linkTrg )
{
   const char* const gioTemplate = "gio info %s \"%s\" 1>%s 2>%s" ;
   const wchar_t* const errorToken  = L"gio: " ;
   const char* const dref = " " ;
   const char* const noderef = "--nofollow-symlinks" ;
   const char* lnkcmd = linkTrg ? dref : noderef ;
   gString gs ;                  // text formatting
   short indxA, indxB,           // search indices
         rawType = ZERO,         // basic "type" code (see above)
         status ;                // return value

   #if GVFS_FAILSAFE != 0     // Note the early return
   if ( ! gioInstalled )
      return ( (this->GetFileStats ( trgStat, trgPath, linkTrg )) ) ;
   #endif   // GVFS_FAILSAFE

   trgStat.ReInit() ;            // initialize caller's data structure
   trgStat.rawStats.st_dev     = // manually initialize rawStats fields
   trgStat.rawStats.st_ino     = // (in case no value available)
   trgStat.rawStats.st_nlink   = 
   trgStat.rawStats.st_rdev    = 
   trgStat.rawStats.st_size    = 
   trgStat.rawStats.st_blksize = 
   trgStat.rawStats.st_blocks  = 
   trgStat.rawStats.st_atime   = 
   trgStat.rawStats.st_mtime   = 
   trgStat.rawStats.st_ctime   = ZERO ;
   trgStat.rawStats.st_mode = ZERO ;
   trgStat.rawStats.st_uid  = this->userInfo.userID ; // assume current user
   trgStat.rawStats.st_gid  = this->userInfo.grpID ;

   this->ExtractFilename ( gs, trgPath ) ;   // initialize filename field
   gs.copy( trgStat.fName, MAX_FNAME ) ;

   //* Create a temporary file *
   gString tmpPath ;
   this->CreateTempname ( tmpPath ) ;

   //* Create the command *
   const short bSIZE = MAX_PATH * 3 ;  // size of command buffer
   char cmd[bSIZE] ;
   snprintf ( cmd, bSIZE, gioTemplate, lnkcmd, trgPath, 
                          tmpPath.ustr(), tmpPath.ustr() ) ;

   //* Execute the command *
   if ( (status = this->Systemcall ( cmd )) == OK )
   {
      ifstream ifs( tmpPath.ustr(), ifstream::in ) ;
      if ( ifs.is_open() )
      {
         ifs.getline( cmd, gsDFLTBYTES ) ;
         while ( ifs.gcount() > ZERO )
         {
            gs = cmd ;
            if ( (indxA = gs.after( "  standard::" )) > ZERO )
            {
               if ( (indxB = gs.after( "size:", indxA )) == (indxA + 5) )
               {
                  gs.gscanf( indxB, " %llu", &trgStat.fBytes ) ;
                  trgStat.rawStats.st_size = trgStat.fBytes ;
               }
               else if ( (indxB = gs.after( "type:", indxA )) == (indxA + 5) )
               {
                  gs.gscanf( indxB, " %hd", &rawType ) ;
                  switch ( rawType )
                  {
                     // Note: case 1 regular file
                     //       case 2 is directory
                     //       case 3 symbolic link (if --nofollow-symlinks)
                     //       case 4 may be: block/char/socket/fifo/unknown
                     case 1:
                        trgStat.fType = fmREG_TYPE ;
                        trgStat.rawStats.st_mode |= __S_IFREG ;
                        break ;
                     case 2:
                        trgStat.fType = fmDIR_TYPE ;
                        trgStat.rawStats.st_mode |= __S_IFDIR ;
                        break ;
                     case 3:
                        trgStat.fType = fmLINK_TYPE ;
                        trgStat.rawStats.st_mode |= __S_IFLNK ;
                        break ;
                     case 4:     trgStat.fType = fmCHDEV_TYPE ;   break ;
                     default:    trgStat.fType = fmUNKNOWN_TYPE ; break ;
                  }
               }
               else if ( (indxB = gs.after( "content-type:", indxA )) > ZERO )
               {
                  if ( rawType == 4 )
                  {
                     if ( (gs.find( "blockdevice", indxB )) > ZERO )
                     {
                        trgStat.fType = fmBKDEV_TYPE ;
                        trgStat.rawStats.st_mode |= __S_IFBLK ;
                     }
                     else if ( (gs.find( "chardevice", indxB )) > ZERO )
                     {
                        trgStat.fType = fmCHDEV_TYPE ;
                        trgStat.rawStats.st_mode |= __S_IFCHR ;
                     }
                     else if ( (gs.find( "socket", indxB )) > ZERO )
                     {
                        trgStat.fType = fmSOCK_TYPE ;
                        trgStat.rawStats.st_mode |= __S_IFSOCK ;
                     }
                     else if ( (gs.find( "fifo", indxB )) > ZERO )
                     {
                        trgStat.fType = fmFIFO_TYPE ;
                        trgStat.rawStats.st_mode |= __S_IFIFO ;
                     }
                     else
                        trgStat.fType = fmUNKNOWN_TYPE ;
                  }
               }
            }
            else if ( (indxA = gs.after( "  access::" )) > ZERO )
            {
               if ( (indxB = gs.after( "can-read: ", indxA )) > ZERO )
               {
                  if ( gs.gstr()[indxB] == L'T' )
                  {
                     trgStat.readAcc = true ;
                     trgStat.rawStats.st_mode |= __S_IREAD ;
                  }
               }
               else if ( (indxB = gs.after( "can-write: ", indxA )) > ZERO )
               {
                  if ( gs.gstr()[indxB] == L'T' )
                  {
                     trgStat.writeAcc = true ;
                     trgStat.rawStats.st_mode |= __S_IWRITE ;
                  }
               }
               else if ( (indxB = gs.after( "can-execute: ", indxA )) > ZERO )
               {
                  if ( gs.gstr()[indxB] == L'T' )
                     trgStat.rawStats.st_mode |= __S_IEXEC ;
               }
            }
            else if ( (indxA = gs.after( "  time::" )) > ZERO )
            {
               if ( (indxB = gs.after( "modified:", indxA )) > ZERO )
                  gs.gscanf( indxB, " %llu", &trgStat.rawStats.st_mtime ) ;
               else if ( (indxB = gs.after( "access:", indxA )) > ZERO )
                  gs.gscanf( indxB, " %llu", &trgStat.rawStats.st_atime ) ;
               else if ( (indxB = gs.after( "changed:", indxA )) > ZERO )
                  gs.gscanf( indxB, " %llu", &trgStat.rawStats.st_ctime ) ;
            }
            else if ( (indxA = gs.after( "  id::" )) > ZERO )
            {
               if ( (indxB = gs.after( "file:", indxA )) > ZERO )
               {
                  //* For GVfs files, this is: id::file: mtp:host=....:12345  *
                  //* For Linux inode/block files, this is:                   *
                  //* id::file: l64770:19398734                               *
                  //*   filesystem: 1st half: "l64770"                        *
                  //*   inode     : 2nd half: "19398734"                      *
                  if ( (gs.gscanf( indxB, " l%llu", &trgStat.rawStats.st_dev )) != 1 )
                     trgStat.rawStats.st_dev = ZERO ;
                  if ( (indxB = (gs.findlast( L':' )) + 1) > ZERO )
                     gs.gscanf( indxB, " %llu", &trgStat.rawStats.st_ino ) ;
               }
            }
            else if ( (indxA = gs.after( "  unix::" )) > ZERO )
            {  // Programmer's Note: Files on GVfs filesystems will probably not 
               // report the "unix" group of data items. For "real" block devices, 
               // some of these items will be reported.
               if ( (indxB = gs.after( "uid:", indxA )) > ZERO )
                  gs.gscanf( indxB, " %u", &trgStat.rawStats.st_uid ) ;
               else if ( (indxB = gs.after( "gid:", indxA )) > ZERO )
                  gs.gscanf( indxB, " %u", &trgStat.rawStats.st_gid ) ;
               else if ( (indxB = gs.after( "block-size:", indxA )) > ZERO )
                  gs.gscanf( indxB, " %llu", &trgStat.rawStats.st_blksize ) ;
               else if ( (indxB = gs.after( "blocks:", indxA )) > ZERO )
                  gs.gscanf( indxB, " %llu", &trgStat.rawStats.st_blocks ) ;
               else if ( (indxB = gs.after( "nlink:", indxA )) > ZERO )
                  gs.gscanf( indxB, " %llu", &trgStat.rawStats.st_nlink ) ;
               // unix::device ? ignore, taken from ID
               // unix::inode: ? ignore, taken from ID
               // unix::mode:  ? ignore, do not overwrite existing bits
               // unix::rdev:  ? device type (inode devices only)
            }

            else if ( (gs.find( errorToken )) == ZERO )
            {
               status = ENOENT ; // assume file-not-found
               break ;
            }

            ifs.getline ( cmd, gsDFLTBYTES ) ;
         }     // while()
         ifs.close() ;

         //* If 'mtime' captured, format timestamp *
         if ( trgStat.rawStats.st_mtime > ZERO )
            this->DecodeEpochTime( (int64_t)trgStat.rawStats.st_mtime, 
                                   trgStat.modTime ) ;

         //* If 'atime' or 'ctime' not captured, substitute 'mtime'.*
         if ( trgStat.rawStats.st_atime == ZERO )
            trgStat.rawStats.st_atime = trgStat.rawStats.st_mtime ;
         if ( trgStat.rawStats.st_ctime == ZERO )
            trgStat.rawStats.st_ctime = trgStat.rawStats.st_mtime ;
      }
   }
   else
   {
      //* This method is often called to determine whether a file exists,   *
      //* and if it doesn't, it's not an error so don't display the message.*
      this->recentErrno = ENOENT ;
   }

   //* Delete the temp file *
   this->DeleteTempname ( tmpPath ) ;

   return status ;

}  //* End GetFileStats_gvfs() *

//***************************
//* GetFilesystemStats_gvfs *
//***************************
//********************************************************************************
//* Called primarily by GetFilesystemStats( const char*, fileSystemStats&);      *
//*       (May be called independently for diagnostic purposes.)                 *
//*           ----  ----  ----  ----  ----  ----  ----  ----                     *
//* Read file system stats for the specified virtual (MTP/GVfs) filesystem.      *
//* Caller has verified that the filesystem device is "gvfsd-fuse".              *
//*                                                                              *
//* The data gathered here are supplimentary to the filesystem data obtained     *
//* by the calling method.                                                       *
//*                                                                              *
//* Input  : trgPath: full path/filename for any file on the target file system  *
//*          usbDev : (by reference) unitialized instance of                     *
//*                   usbDevice class: receives stat data                        *
//*                                                                              *
//* Returns: OK if successful, else ERR                                          *
//*           if 'OK', all fields of usbDev will be initialized                  *
//********************************************************************************
//* Notes:                                                                       *
//* ======                                                                       *
//* Many of the fields reported for block devices by "stat -f" are not           *
//* reported by the "gio info" call, so the data from both sources may be        *
//* combined for a more complete description of the target device.               *
//*                                                                              *
//********************************************************************************

short FMgr::GetFilesystemStats_gvfs ( const char* trgPath, usbDevice& usbDev )
{
   usbDevice *usbdPtr = NULL ;         // pointer to array of usbDevice objects
   gString gs( trgPath ) ;             // text analysis
   short   usbCount = ZERO,            // number of MTP/GVfs devices found
           status = ERR ;              // return value
   usbDev.reset() ;                    // initialize caller's data object

   //* Scan for MTB/GVfs devices *
   if ( (usbCount = this->USB_DeviceStats ( usbdPtr, true )) != ZERO )
   {
      //* Select the matching record (if any) *
      for ( short usbIndex = ZERO ; usbIndex < usbCount ; ++usbIndex )
      {
         if ( (gs.find( usbdPtr[usbIndex].mntdir )) >= ZERO )
         {
            usbDev = usbdPtr[usbIndex] ;
            status = OK ;
            break ;
         }
      }
      delete [] usbdPtr ; usbdPtr = NULL ;   // release the dynamic allocation

      //* Get additional information about the target device.*
      if ( status == OK )
      {
         //* Initialize 'usbDev' members:             *
         //*  - totalbytes    - readonly  - backend   *
         //*  - usedbytes     - remote                *
         //*  - freebytes     - fstype                *
         if ( ! (this->GetFilesystemStats_aux ( trgPath, usbDev )) )
            status = ERR ; // target not found or inaccessible (unlikely)
      }
   }
   return status ;

}  //* End GetFilesystemStats_gvfs() *

//***************************
//* GetFilesystemStats_opti *
//***************************
//********************************************************************************
//* Called primarily by GetFilesystemStats( const char*, fileSystemStats&);      *
//*       (May be called independently for diagnostic purposes.)                 *
//*           ----  ----  ----  ----  ----  ----  ----  ----                     *
//* Read file system stats for the specified optical-drive (DVD,CD,Blu-Ray)      *
//* filesystem.                                                                  *
//*                                                                              *
//* The data gathered here are supplimentary to the filesystem data obtained     *
//* by the calling method.                                                       *
//*                                                                              *
//* Input  : trgPath: full path/filename for any file on the target file system  *
//*          usbDev : (by reference) unitialized instance of                     *
//*                   usbDevice class: receives stat data                        *
//*                                                                              *
//* Returns: OK if successful, else ERR                                          *
//*           if 'OK', all fields of usbDev will be initialized                  *
//********************************************************************************
//* Notes:                                                                       *
//* ======                                                                       *
//* Many of the fields reported for block devices by "stat -f" are not           *
//* reported by the "gio info" call, so the data from both sources may be        *
//* combined for a more complete description of the target device.               *
//*                                                                              *
//********************************************************************************

short FMgr::GetFilesystemStats_opti ( const char* trgPath, usbDevice& usbDev )
{
   usbDevice *usbdPtr = NULL ;         // pointer to array of usbDevice objects
   gString gs( trgPath ) ;             // text analysis
   short   dvdCount = ZERO,            // number of optical-drive devices found
           status = ERR ;              // return value
   usbDev.reset() ;                    // initialize caller's data object

   //* Scan for optical drives *
   if ( (dvdCount = this->DVD_DeviceStats ( usbdPtr, true )) != ZERO )
   {
      //* Select the matching record (if any) *
      for ( short dvdIndex = ZERO ; dvdIndex < dvdCount ; ++dvdIndex )
      {
         if ( (gs.find( usbdPtr[dvdIndex].mntdir )) >= ZERO )
         {
            usbDev = usbdPtr[dvdIndex] ;
            status = OK ;
            break ;
         }
      }
      delete [] usbdPtr ; usbdPtr = NULL ;   // release the dynamic allocation
   }
   return status ;

}  //* End GetFilesystemStats_opti() *

//***************************
//* GetFilesystemStats_aux  *
//***************************
//********************************************************************************
//* Private Method                                                               *
//* ==============                                                               *
//* Read supplementary file system stats for the specified filesystem:           *
//*  1) MTP/GVfs (smartphones/tablets), or                                       *
//*  2) optical-drive filesystems (DVD/CD/Blu-Ray).                              *
//*                                                                              *
//* Caller has verified that the 'gio' utility is available.                     *
//*                                                                              *
//* The data gathered here are supplimentary to the filesystem data obtained     *
//* by the calling method.                                                       *
//*                                                                              *
//* Input  : trgPath: full path/filename for any file on the target file system  *
//*          usbDev : (by reference) unitialized (or partially initialized       *
//*                   instance of usbDevice class: receives stat data            *
//*                                                                              *
//* Returns: 'true' if mounted and accessible                                    *
//*                 fields of usbDev initialized (if reported by 'gio):          *
//*                  usbDev.totalbytes                                           *
//*                  usbDev.freebytes                                            *
//*                  usbDev.usedbytes                                            *
//*                  usbDev.readonly                                             *
//*                  usbDev.remote                                               *
//*                  usbDev.fstype                                               *
//*                  usbDev.backend                                              *
//*          'false' if target not found or is inaccessible                      *
//*                  (above fields set to default values)                        *
//********************************************************************************
//* Notes:                                                                       *
//* ======                                                                       *
//* The "gio info" command reports information on the target file by default;    *
//* however, the '--filesystem' ('-f') option provides additional information    *
//* on the filesystem which contains the target file. Because we do not need     *
//* the file-specific data here, we specify the attributes to be reported.       *
//* Example Report:                                                              *
//*  filesystem::size: 57938345984                                               *
//*  filesystem::used: 22358387008    (may not be reported)                      *
//*  filesystem::free: 35576958976                                               *
//*  filesystem::type: mtpfs                                                     *
//*  filesystem::readonly: FALSE                                                 *
//*  filesystem::remote: FALSE                                                   *
//*  gvfs::backend: mtp                                                          *
//*                                                                              *
//********************************************************************************

bool FMgr::GetFilesystemStats_aux ( const char* trgPath, usbDevice& usbDev )
{
   const char* const gioTemplate = 
      "gio info --filesystem --nofollow-symlinks "
         "--attributes='filesystem::size,filesystem::used,filesystem::free,"
         "filesystem::type,filesystem::readonly,filesystem::remote,gvfs::backend' " 
         "\"%s\" 1>'%s' 2>'%s'" ;
   const char* const accessError = "Error getting filesystem info" ;
   // Programmer's Note: This error message comes through 'stderr'.

   bool status = false ;     // return value

   //* Initialize target fields of caller's object *
   usbDev.totalbytes = usbDev.freebytes = usbDev.usedbytes = ZERO ;
   usbDev.readonly = usbDev.remote = false ;
   usbDev.fstype[0] = usbDev.backend[0] = NULLCHAR ;

   //* Create a temporary file to capture the raw data.*
   gString tmpPath ;
   this->CreateTempname ( tmpPath ) ;

   //* Create and execute the command *
   gString gscmd( gioTemplate, trgPath, tmpPath.ustr(), tmpPath.ustr() ) ;
   this->Systemcall ( gscmd.ustr() ) ;

   //* Open the temp file and extract the interesting data.*
   ifstream ifs( tmpPath.ustr(), ifstream::in ) ; // open the file
   if ( ifs.is_open() )                           // if file opened
   {
      char lineBuff[gsDFLTBYTES] ;
      gString gs ;
      short indxA, indxB ;          // search indices
      status = true ;               // hope for success

      ifs.getline( lineBuff, gsDFLTBYTES ) ;
      while ( ifs.gcount() > ZERO )
      {
         gs = lineBuff ;
         if ( (indxA = gs.after( "  filesystem::" )) > ZERO )
         {
            if ( (indxB = gs.after( "size:", indxA )) == (indxA + 5) )
               gs.gscanf( indxB, " %llu", &usbDev.totalbytes ) ;
            else if ( (indxB = gs.after( "free:", indxA )) == (indxA + 5) )
               gs.gscanf( indxB, " %llu", &usbDev.freebytes ) ;
            else if ( (indxB = gs.after( "used:", indxA )) == (indxA + 5) )
               gs.gscanf( indxB, " %llu", &usbDev.usedbytes ) ;
            else if ( (indxB = gs.after( "readonly: ", indxA )) == (indxA + 10) )
            { if ( gs.gstr()[indxB] == L'T' ) usbDev.readonly = true ; }
            else if ( (indxB = gs.after( "remote: ", indxA )) == (indxA + 8) )
            { if ( gs.gstr()[indxB] == L'T' ) usbDev.remote = true ; }
            else if ( (indxB = gs.after( "type:", indxA )) == (indxA + 5) )
            {
               //* Compare this 'type' with previously-captured type.*
               if ( (gs.compare( usbDev.fstype, true, fssLEN, indxB )) != ZERO )
                  gs.gscanf( indxB, " %64s", usbDev.fstype ) ;
            }
         }
         else if ( (indxA = gs.after( "  gvfs::backend:" )) > ZERO )
         { gs.gscanf( indxA, " %64s", usbDev.backend ) ; }

         //* If error accessing target filesystem *
         else if ( (gs.find( accessError )) >= ZERO )
         { status = false ; break ; }

         ifs.getline ( lineBuff, gsDFLTBYTES ) ;
      }     // while()

      //* For read-only drives, set "usedbytes" == "totalbytes".*
      //* ("freebytes" should already be zero)                  *
      if ( usbDev.readonly && (usbDev.usedbytes == ZERO) && (usbDev.totalbytes > ZERO) )
         usbDev.usedbytes = usbDev.totalbytes ;

      ifs.close() ;                    // close the file
   }

   this->DeleteFile ( tmpPath.ustr() ) ;  // delete the temporary file

   return status ;

}  //* End GetFilesystemStats_aux() *

//*************************
//*      isGvfsPath       *
//*************************
//********************************************************************************
//* Report whether the current base directory lies within a GVfs filesystems     *
//* i.e. whether the operation requires the "MTP" protocol.                      *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: 'true'  if target lies within (i.e. BELOW) the gvfs anchor point    *
//*          'false' otherwise                                                   *
//********************************************************************************

bool FMgr::isGvfsPath ( void )
{

   gString basedir( this->currDir ) ;
   return ( (this->isGvfsPath ( basedir )) ) ;

}  //* End isGvfsPath() *

//*************************
//*      isGvfsPath       *
//*************************
//********************************************************************************
//* Determine whether the specified filespec lies on the mount path for          *
//* GVfs filesystems i.e. whether the operation requires the "MTP" protocol.     *
//*                                                                              *
//* Input  : trgPath : target filespec                                           *
//*                                                                              *
//* Returns: 'true'  if target lies within (i.e. BELOW) the gvfs anchor point    *
//*          'false' otherwise                                                   *
//********************************************************************************

bool FMgr::isGvfsPath ( const gString& trgPath )
{
   if ( *gioMountBase == NULLCHAR )    // if static data not yet initialized
   {
      gString tmpSpec ;
      this->CreateTempname ( tmpSpec ) ;
      this->gioAvailable ( tmpSpec ) ;
      this->DeleteTempname ( tmpSpec ) ;
   }

   gString gvfsPath( gioMountBase ) ;
   bool gvfs = false ;

   if ( ((trgPath.find( gvfsPath.gstr() )) == ZERO) &&
        ((gvfsPath.compare( trgPath )) < ZERO) )
      gvfs = true ;
   return gvfs ;

}  //* End isGvfsPath() *

//*************************
//*   isMountpoint_gvfs   *
//*************************
//********************************************************************************
//* PRIVATE METHOD                                                               *
//* --------------                                                               *
//* Determine whether the specified filespec refers to a mountpoint directory    *
//* for an MTP/GVfs filesystem.                                                  *
//* We trust caller has verified that 'trgPath' references a gvfs filesystem.    *
//*                                                                              *
//* Input  : trgPath : target filespec                                           *
//*                                                                              *
//* Returns: 'true'  if target is a mountpoint directory                         *
//*          'false' if target is not a mountpoint                               *
//*                  (or is not a directory, or does not exist)                  *
//********************************************************************************

bool FMgr::isMountpoint_gvfs ( const gString& trgPath )
{
   usbDevice usbd ;
   bool isMp = false ;           // return value

   //* Get filesystem info for the target *
   if ( (this->GetFilesystemStats_gvfs ( trgPath.ustr(), usbd )) == OK )
   {
      if ( (trgPath.compare( usbd.mntpath )) == ZERO )
         isMp = true ;
   }

   return isMp ;

}  //* End isMountpoint_gvfs() *

//*************************
//*     gioAvailable      *
//*************************
//********************************************************************************
//* PRIVATE METHOD                                                               *
//* --------------                                                               *
//* Verify whether the 'gio' utility is available.                               *
//* If installed, a single line should be written to the temp file containing    *
//* the version number in the format: 2.64.3                                     *
//* If installed, set the static 'gioInstalled' flag at the top of this module.  *
//*                                                                              *
//* Note that the 'gioMountBase' non-member variable (top of this module) is     *
//* also initialized. Initialization is from the system environment variable     *
//* (if defined), otherwise the path is synthesized from the "most likely"       *
//* path definition.                                                             *
//*                                                                              *
//* Input  : tmpSpec : filespec of a temporary file to capture the response      *
//*                    from the call to the shell                                *
//*                    Note that on return, temp file will contain stale data    *
//*                                                                              *
//* Returns: state of 'gioInstalled' flag                                        *
//********************************************************************************

bool FMgr::gioAvailable ( const gString& tmpSpec )
{
   if ( ! gioInstalled )
   {
      gString gsCmd( "gio version 1>\"%s\" 2>/dev/null", tmpSpec.ustr() ) ;
      char inbuff[gsDFLTBYTES] ;

      if ( (this->Systemcall ( gsCmd.ustr() )) == OK )
      {
         ifstream ifs( tmpSpec.ustr(), ifstream::in ) ;
         if ( ifs.is_open() )
         {
            ifs.getline ( inbuff, gsDFLTBYTES ) ;
            if ( (inbuff[1] == '.') && 
                 ((inbuff[0] >= '1') && (inbuff[0] <= '9')) )
            {
               gsCmd = inbuff ;     // store the version number
               gsCmd.copy( gioVersion, gvLEN ) ;
               gioInstalled = true ;
            }
            ifs.close() ;
         }
      }

      //* Initialize static 'gioMountBase' variable.*
      gString gsEnv( getenv ( GVFS_MNT_DIR ) ) ;
      if ( (gsEnv.gschars()) == 1 )    // if environment variable not defined
         gsEnv.compose( "/run/user/%u", &this->userInfo.userID ) ;
      gsEnv.append( "/gvfs" ) ;
      gsEnv.copy( gioMountBase, MAX_PATH ) ;
   }
   return gioInstalled ;

}  //* End gioAvailable() *

//*************************
//*     gioGetVersion     *
//*************************
//********************************************************************************
//* Returns the version number of the 'gio' utility (if installed).              *
//*                                                                              *
//* If the static variable 'gioInstalled' is not set, call the 'gioAvailable()'  *
//* method to determine whether the 'gio' utility is installed and to            *
//* initialize the version string in the static 'gioVersion' variable at the     *
//* top of this module.                                                          *
//*                                                                              *
//* Input  : gvfsVersion : (by reference) receives the 'gio' utility version     *
//*                        or empty string if 'gio' not installed                *
//*                                                                              *
//* Returns: 'true' if GVfs tools are accessible, else 'false'                   *
//********************************************************************************

bool FMgr::gioGetVersion ( gString& gvfsVersion )
{
   //* Verify whether the 'gio' utility is available *
   //* (if not previously verified).                 *
   if ( ! gioInstalled )
   {
      gString tmpSpec ;       // filespec of temp file
      this->CreateTempname ( tmpSpec ) ;
      this->gioAvailable ( tmpSpec ) ;
      this->DeleteTempname ( tmpSpec ) ;
   }

   if ( gioInstalled  )
      gvfsVersion = gioVersion ;
   else
      gvfsVersion.clear() ;

   return gioInstalled ;

}  //* End gioGetVersion() *

//*************************
//*     Uri2Filespec      *
//*************************
//********************************************************************************
//* Construct the full filespec for the specified filesystem based on the        *
//* Uniform Resource Identifier (URI).                                           *
//* 1) MTP/GVfs URI:                                                             *
//*    Example URI: mtp://Android_ce012345c012345d01/                            *
//*    Filespec   : /run/user/1000/gvfs/mtp:host=Android_ce012345c012345d01      *
//* 2) Audio Disc URI:                                                           *
//*    Example URI: cdda://sr0/                                                  *
//*    Filespec   : /run/user/1000/gvfs/cdda:host=sr0                            *
//*                                                                              *
//* Input  : pathSpec : (by reference) receives the target filespec              *
//*          uri      : URI for an MTP/GVfs filesystem                           *
//*                                                                              *
//* Returns: 'true' if valid filespec,       (pathSpec initialized)              *
//*          'false' if invalid or unrecognized 'uri' format (pathSpec cleared)  *
//********************************************************************************
//* Example:                                                                     *
//* gio mount --list --detail                                                    *
//* Yields:                                                                      *
//*  . . .                                                                       *
//*  activation_root=mtp://SAMSUNG_SAMSUNG_Android_ce012345c012345d01/           *
//*  . . .                                                                       *
//*                                                                              *
//* All modern systems mount GVfs filesystems at:  /run/user/$UID/gvfs           *
//* The "activation_root" (URI) must be modified to indicate the mtp host:       *
//*       "mtp://"  is modified to  "mtp:host="                                  *
//* Therefore, for this example, the filespec would be:                          *
//* /run/user/$UID/gvfs/mtp:host=SAMSUNG_SAMSUNG_Android_ce012345c012345d01      *
//*                                                                              *
//* This conversion algorithm may be verified using the "gio info" command       *
//* described in the notes for the GetFileStats_gvfs() method. Briefly, the      *
//* command: "gio info cdda://sr0/" will yield information containing the        *
//* translated specification in at least one of two line items:                  *
//*   "local path: /run/user/1000/gvfs/cdda:host=sr0"                            *
//*   "id::filesystem: cdda:host=sr0"                                            *
//* Note that the filesystem must be mounted in order to retrieve this info,     *
//* while our shortcut conversion algorithm can create the target filespec       *
//* regardless of whether the device is mounted.                                 *
//*                                                                              *
//* Programmer's Note: We would prefer to get the filespec information           *
//* directly, but the operation is both inconvenient and slow. For this reason,  *
//* we perform this conversion manually.                                         *
//********************************************************************************

bool FMgr::Uri2Filespec ( gString& pathSpec, const char* uri )
{
   bool  status = false ;     // return value

   if ( *gioMountBase == NULLCHAR )    // if static data not yet initialized
   {
      gString tmpSpec ;
      this->CreateTempname ( tmpSpec ) ;
      this->gioAvailable ( tmpSpec ) ;
      this->DeleteTempname ( tmpSpec ) ;
   }

   //* Construct the basic filespec *
   pathSpec.compose( "%s/%s", gioMountBase, uri ) ;

    //* Remove any trailing forward-slash ('/') character *
   short trunc = pathSpec.gschars() - 2 ;
   if ( (pathSpec.findlast( L'/' )) == trunc )
      pathSpec.limitChars( (pathSpec.gschars() - 2) ) ;

   if ( (pathSpec.find( mtpURI )) >= ZERO )
      status = pathSpec.replace( mtpURI, mtpHost ) ;
   else if ( (pathSpec.find( optiURI )) >= ZERO )
      status = pathSpec.replace( optiURI, optiHost ) ;

   if ( ! status )
      pathSpec.clear() ;

   return status ;

}  //* End Uri2Filespec() *

//*************************
//*      EjectMedia       *
//*************************
//********************************************************************************
//* Eject the removable media from the specified device, or if no device         *
//* specified, then eject the media from the first DVD/CD drive.                 *
//*    Example: EjectMedia( "/dev/sr0" ) ;                                       *
//* Typical devices with ejectable media: CD-ROM, DVD-ROM, floppy disk, tape     *
//* drive, ZIP disk or USB disk.                                                 *
//*                                                                              *
//* 1) Not all optical drives can be closed under software control.              *
//* 2) If target device is mounted and writable, be sure to close the output     *
//*    stream or unmount the device before ejecting the tray to avoid potential  *
//*    loss of data.                                                             *
//* 3) If target device does not support removable media, the operation will     *
//*    silently fail.                                                            *
//* 4) The 'close' and 'toggle' flags are mutually exclusive. If 'toggle' is     *
//*    set, then 'close' is ignored.                                             *
//*                                                                              *
//* Input  : device : (optional, NULL pointer by default)                        *
//*                   If not specified, media will be ejected from the           *
//*                    system's default DVD/CD drive: "/dev/cdrom"               *
//*                   If specified, device whose media are to be ejected.        *
//*          close  : (optional, 'false' by default)                             *
//*                   If 'false', open the tray.                                 *
//*                   If 'true',  close it tray.                                 *
//*          toggle : (optional, 'false' by default)                             *
//*                   If 'false', ignore.                                        *
//*                   If 'true', toggle the state of the tray.                   *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FMgr::EjectMedia ( const char* device, bool close, bool toggle )
{
   const char* const dfltTarget = "/dev/cdrom" ;

   gString gsCmd( "eject '%s'", (device != NULL ? device : dfltTarget) ) ;
   short indx = gsCmd.after( "eject" ) ;

   if ( toggle )
      gsCmd.insert( " --traytoggle ", indx ) ;
   else if ( close )
      gsCmd.insert( " --trayclose ", indx ) ;

   this->Systemcall ( gsCmd.ustr() ) ;

}  //* End EjectMedia() *

//*************************
//*    USB_DeviceStats    *
//*************************
//********************************************************************************
//* Scan for devices attached to the Universal Serial Bus (USB).                 *
//* Capture filesystem and support information for the devices.                  *
//* This method is designed primarily for smartphones, tablets and other         *
//* virtual filesystems attached to the USB bus. Devices may, or may not be      *
//* mounted and visible to the user.                                             *
//*                                                                              *
//* The "all" parameter:                                                         *
//* --------------------                                                         *
//* Optionally, this method can report the basic bus, device, ID and             *
//* descriptive text information for each device attached to the USB bus.        *
//* This option is not used by the FileMangler application, but may be helpful   *
//* in other applications.                                                       *
//*                                                                              *
//* Important Note: This method allocates dynamic memory. It is the caller's     *
//*                 responsibility to release this memory when it is no longer   *
//*                 needed. Example: delete [] usbdPtr ;                         *
//*                                                                              *
//* Input  : usbdPtr  : (by reference, initial value ignored)                    *
//*                     receives pointer to a dynamically-allocated array of     *
//*                     usbDevice objects. Set to NULL pointer if no USB         *
//*                     devices matching the criteria are found.                 *
//*          testmnt  : (optional, 'false' by default)                           *
//*                     if 'false', do not test for mounted status               *
//*                     if 'true',  for each MTP/GVFS device determine whether   *
//*                                 it is currently mounted                      *
//*          all      : (optional, 'false' by default)                           *
//*                     if 'false', report only MTP/GVFS devices                 *
//*                     if 'true',  summary report of all devices attached       *
//*                                 to USB bus                                   *
//*                                                                              *
//* Returns: count of USB devices captured (elements in usbDevice array)         *
//********************************************************************************
//* Notes:                                                                       *
//* MTP devices are treated like remote servers, not mounted devices.            *
//* Also, these devices _are not_ block devices and are not handled within the   *
//* system kernel. Therefore these devices will not be seen by utilities such    *
//* as 'df' and 'lsblk' which scan kernel databases.                             *
//*                                                                              *
//* For this reason, the scan is performed using stand-alone (non-kernel)        *
//* utilities such 'lsusb' and 'gio'. These utilities are still evolving as      *
//* the technology changes, but the features used in this method should be       *
//* relatively stable.                                                           *
//* ---  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ---   *
//*                                                                              *
//* The 'lsusb' utility is used to obtain the basic information for each         *
//* device attached to the USB bus.                                              *
//*                                                                              *
//* 1) Capture the output from the 'lsusb' command.                              *
//* 2) Scan for a record containing reference to an MTP device. Example:         *
//*    Bus 003 Device 006: ID 04e8:6860 Samsung Electronics Co. ... (MTP mode)   *
//* 3) Extract the bus number and device number and use them to create a         *
//*    formatted string which identified a portion of the device-driver path.    *
//*    For the above example, the search string would be: "/003/006"             *
//* 4) Extract the "ID" which consists of two 4-digit hex values separated by    *
//*    a colon. These values represent the vendor (manufacturer) and the         *
//*    product, respectively. For the above example, this is: "04e8:6860"        *
//*    where "04e8" indicates Samsung Electronics, and "6860" represents a       *
//*    Galaxy S8 smartphone. This identifier has some value for                  *
//*    cross-referencing with data obtained from other sources, but is not       *
//*    critical to this particular algorithm.                                    *
//* 5) Extract the descriptive text string identifying this device.              *
//*    This is primarily informational only, and is not used.                    *
//* ---  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ---   *
//*                                                                              *
//* The 'gio' utility is used to obtain additional information related to        *
//* MTP/GVfs virtual-filesystem devices such as smartphones and tablets.         *
//*                                                                              *
//* 1) Capture the output from the command: "gio mount --list --detail"          *
//*                                                                              *
//* 2) Scan for a line that contains the "unix-device: '" entry.                 *
//*    The entry matching the example record above would be:                     *
//*    "unix-device: '/dev/bus/usb/003/006'"                                     *
//*                                                                              *
//* 3) Using the formatted bus/device string described in item 3 above, scan     *
//*    this entry to determine if the record matches the bus/device number of    *
//*    that device.                                                              *
//*    If bus number and device number are a match, then extract the full        *
//*    device-driver filespec.                                                   *
//*                                                                              *
//* 4) If a matching record has been found (item 3), then the next line of the   *
//*    output will contain the "activation_root" entry. This entry is the        *
//*    MTP/GVfs entry which define the device and consists of concatenated       *
//*    elements of the filesystem information in the format:                     *
//*       "mtp://VENDORNAME_PRODUCTNAME_DEVICESERIALNUMBER/"                     *
//*    For the above example, this would be:                                     *
//*       "mtp://SAMSUNG_SAMSUNG_Android_ce012345c012345d01/"                    *
//*       (Note both the URL construct "mtp://", and the trailing '/')           *
//*                                                                              *
//* 5) Derive the actual name of the mountpoint directory from the               *
//*    activation_root entry described in item 4.                                *
//*    a) Strip the trailing '/' character.                                      *
//*    b) The mountpoint directory name is constructed by replacing              *
//*       the URI construct "mtp://" with the mtp-protocol identifier            *
//*       constant: "mtp:host=" which will yield:                                *
//*       "mtp:host=SAMSUNG_SAMSUNG_Android_ce012345c012345d01"                  *
//*                                                                              *
//* 6) Construct the mountpoint path.                                            *
//*    (The device may or may not be mounted at this time.)                      *
//*    a) All modern systems mount MTP/GVfs devices under the following path:    *
//*       "/run/user/$UID/gvfs"                                                  *
//*       The base directory is identified by the environment variable:          *
//*       $XDG_RUNTIME_DIR (if available).                                       *
//*          (Programmer's Note: Some older systems mounted GVfs devices)        *
//*          (under "$HOME/.gvfs"; however, that location is deprecated.)        *
//*       Each user's path is identified by the userID. For a single-user        *
//*       system operating at user-privilege, $UID will be "1000".               *
//*       For the superuser, the userID will be "0".                             *
//*    b) Concatenate the path described above with the mountpoint directory     *
//*       name in item 5. For this example, the result wiould be:                *
//*  "/run/user/1000/gvfs/mtp:host=SAMSUNG_SAMSUNG_Android_ce012345c012345d01"   *
//*  ---  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ---  *
//*                                                                              *
//* Additional Information:                                                      *
//* A) To mount this device, use the full device-driver filespec and             *
//*    (optionally) the mtp-protocol identifier:                                 *
//*       gio mount --device='/dev/bus/usb/003/006'                              *
//*                 "mtp://SAMSUNG_SAMSUNG_Android_ce012345c012345d01/"          *
//* B) To unmount this device, use the mtp-protocol identifier:                  *
//*    gio mount --unmount "mtp://SAMSUNG_SAMSUNG_Android_ce012345c012345d01/"   *
//*                                                                              *
//* C) Use "gio info" command to gather additional info on _mounted_ device:     *
//*    gio info --attributes='filesystem::type,filesystem::size,                 *
//*                                        filesystem::free' ACTIVATION_ROOT     *
//*    This yields:                                                              *
//*       uri       : [same as activation_root]                                  *
//*       local path: [mountpoint filespec]                                      *
//*       unix mount: gvfsd-fuse /run/user/1000 gvfs . . .                       *
//*       id:filesystem: [mountpoint directory]                                  *
//*       filesystem::size: [decimal value, user-accessible (not incl. system)]  *
//*       filesystem::free: [decimal value, free user-accessible space]          *
//*       filesystem::type: mtpfs                                                *
//*     Note that 'size' and 'free' values track with data gathered by caller.   *
//*                                                                              *
//* D) sudo dnf install mtpfs                                                    *
//*    sudo dnf install jmtpfs mtp-tools                                         *
//*                                                                              *
//*    mtp-detect                                                                *
//*      Listing raw device(s)                                                   *
//*      Device 0 (VID=04e8 and PID=6860) is a Samsung Galaxy models (MTP).      *
//*         Found 1 device(s):                                                   *
//*         Samsung: Galaxy models (MTP) (04e8:6860) @ bus 3, dev 7              *
//*                                                                              *
//*    FROM: libmtp-examples-1.1.16-2.fc30.x86_64 //Example programs for libmtp  *
//* E) See also 'usb-devices'.                                                   *
//* F) See also 'udisks'.                                                        *
//*                                                                              *
//*                                                                              *
//********************************************************************************

short FMgr::USB_DeviceStats ( usbDevice*& usbdPtr, bool testmnt, bool all )
{
   #define DEBUG_USB_SCAN (0)    // for debugging only

   const char* const lsusbTemplate =      // capture only MTP devices
         "lsusb | grep -i 'MTP mode' 1>\"%s\" 2>/dev/null" ;
   const char* const lsallTemplate =      // capture all USB devices
         "lsusb 1>\"%s\" 2>/dev/null" ;
   const char* const giomountTemplate = 
         "gio mount --list --detail 1>\"%s\" 2>/dev/null" ;
   const char* const devfmtTemplate = "/%03hd/%03hd" ;

   char  lineBuff[gsDFLTBYTES] ; // input buffer
   gString tmpPath,              // filespec of temporary file
           cmdBuff,              // command string buffer
           gs ;                  // text formatting
   FileStats fs ;                // stats for target filesystem
   short usbIndex = ZERO,        // index into 'usbdPtr' array
         usbCount = ZERO ;       // return value
   bool  done = false ;          // loop control

   usbdPtr = NULL ;              // initialize caller's pointer

   //* Create a temporary file to capture the raw data.*
   this->CreateTempname ( tmpPath ) ;

   //* Verify whether the 'gio' utility is available. *
   if ( ! gioInstalled )
      this->gioAvailable ( tmpPath ) ;

   //* Call the 'lsusb' utility and capture the output *
   // Programmer's Note: If 'lsusb' not installed, usbCount will be ZERO.
   if ( all != false )
      cmdBuff.compose( lsallTemplate, tmpPath.ustr() ) ;
   else
      cmdBuff.compose( lsusbTemplate, tmpPath.ustr() ) ;
   this->Systemcall ( cmdBuff.ustr() ) ;

   //* Open the temp file.*
   ifstream ifs( tmpPath.ustr(), ifstream::in ) ; // open the file
   if ( ifs.is_open() )                           // if file opened
   {
      done = false ;

      //* Count the number of devices attached to the USB.*
      while ( ! done )
      {
         ifs.getline ( lineBuff, gsDFLTBYTES, NEWLINE ) ;// read a source line
         if ( ifs.good() || ifs.gcount() > ZERO )        // if a good read
            ++usbCount ;
         else           // end-of-file found
            done = true ;
      }

      //* If records to be processed *
      if ( usbCount > ZERO )
      {
         //* Instantiate the required storage *
         usbdPtr = new usbDevice[usbCount] ;

         //* Rewind to top of file *
         ifs.clear() ;                 // reset the EOF flag
         ifs.seekg( ZERO ) ;           // return to top of input file

         //* Capture the data for each record *
         for ( usbIndex = ZERO ; usbIndex < usbCount ; ++usbIndex )
         {
            ifs.getline ( lineBuff, gsDFLTBYTES, NEWLINE ) ;// read a source line
            if ( ifs.good() || ifs.gcount() > ZERO )        // if a good read
            {
               gs = lineBuff ;            // copy of input line
               if ( (gs.gscanf( L"Bus %hd Device %hd: ID %hx:%hx %64[^\n]",
                          &usbdPtr[usbIndex].bus,    &usbdPtr[usbIndex].dev, 
                          &usbdPtr[usbIndex].vendor, &usbdPtr[usbIndex].product, 
                           usbdPtr[usbIndex].desc )) == 5 )
               {
                  //* Format the device information *
                  gs.compose( devfmtTemplate, 
                              &usbdPtr[usbIndex].bus, &usbdPtr[usbIndex].dev ) ;
                  gs.copy( usbdPtr[usbIndex].devfmt, fssLEN ) ;
               }
            }
            else        // file error (unlikely)
               break ;
         }
      }
      ifs.close() ;           // close the file
   }

   //* If one or more records found, find the device  *
   //* filespec and other information for each record.*
   // Programmer's Note: If 'gio' not installed, data returned will be incomplete.
   if ( gioInstalled && (usbCount > ZERO) && !all )
   {
      cmdBuff.compose( giomountTemplate, tmpPath.ustr() ) ; // create command
      this->Systemcall ( cmdBuff.ustr() ) ;                 // execute command

      //* Open the temp file and extract the interesting data.*
      ifs.open( tmpPath.ustr(), ifstream::in ) ; // open the file
      if ( ifs.is_open() )                           // if file opened
      {
         short di ;                       // input buffer index
         done = false ;                   // loop control

         while ( ! done )
         {
            ifs.getline ( lineBuff, gsDFLTBYTES, NEWLINE ) ;// read a source line
            if ( ifs.good() || ifs.gcount() > ZERO )        // if a good read
            {
               gs = lineBuff ;

               for ( usbIndex = ZERO ; usbIndex < usbCount ; ++usbIndex )
               {
                  if ( ! usbdPtr[usbIndex].init &&
                       ((gs.find( usbdPtr[usbIndex].devfmt )) >= ZERO) &&
                       ((di = gs.after( "unix-device: '" )) > ZERO) )
                  {
                     //* Save device-driver filespec *
                     gs.gscanf( di, L"%64[^']", usbdPtr[usbIndex].devpath ) ;
                     ifs.getline ( lineBuff, gsDFLTBYTES, NEWLINE ) ;// read a source line
                     if ( ifs.good() || ifs.gcount() > ZERO )        // if a good read
                     {
                        gs = lineBuff ;
                        if ( (di = gs.after( "activation_root=" )) > ZERO )
                        {
                           gs.gscanf( di, L"%64[^\n]", usbdPtr[usbIndex].uri ) ;

                           //* Assume filesystem type == "mtpfs".*
                           gs = "mtpfs" ;
                           gs.copy( usbdPtr[usbIndex].fstype, fssLEN ) ;

                           //* Construct the mountpoint filespec and    *
                           //* determine whether filesystem is mounted. *
                           this->Uri2Filespec ( gs, usbdPtr[usbIndex].uri ) ;
                           gs.copy( usbdPtr[usbIndex].mntpath, MAX_PATH ) ;
                           short mdi = (gs.findlast( L'/' )) + 1 ;
                           gs.substr( usbdPtr[usbIndex].mntdir, mdi, MAX_FNAME ) ;
                           if ( testmnt != false )
                           {
                              short ismnt = lstat64 ( usbdPtr[usbIndex].mntpath, &fs ) ;
                              if ( ismnt == ZERO )
                                 usbdPtr[usbIndex].mounted = true ;
                           }
                        }
                     }
                     usbdPtr[usbIndex].init = true ; // device record processed
                     break ;
                  }     // (matching 'devfmt')
               }        // for(;;)
            }           // (good read of source line)
            else        // end-of-file found
               done = true ;
         }
         ifs.close() ;           // close the file
      }
   }

   #if DEBUG_USB_SCAN != 0    // DEBUG ONLY
   //* Dump the contents of the usbDevice array to the *
   //* temp file and pause so the data may be reviewed.*
   if ( usbCount > ZERO )
   {
      ofstream ofs( tmpPath.ustr(), ofstream::out | ofstream::trunc ) ;
      if ( ofs.is_open() )
      {
         ofs << "Dump Contents of usbDevice Array\n"
                "================================\n" << endl ;
         for ( usbIndex = ZERO ; usbIndex < usbCount ; ++usbIndex )
         {
            ofs << "desc    : " << usbdPtr[usbIndex].desc << "\n"
                << "devfmt  : " << usbdPtr[usbIndex].devfmt << "\n"
                << "devpath : " << usbdPtr[usbIndex].devpath << "\n"
                << "uri     : " << usbdPtr[usbIndex].uri << "\n"
                << "fstype  : " << usbdPtr[usbIndex].fstype << "\n"
                << "mntdir  : " << usbdPtr[usbIndex].mntdir << "\n"
                << "mntpath : " << usbdPtr[usbIndex].mntpath << "\n" ;
             gs.compose( "bus     : %03hd\n"
                         "dev     : %03hd\n"
                         "vendor  : %04hx\n"
                         "product : %04hx\n"
                         "mounted : %hhd\n"
                         "init    : %hhd\n",
                         &usbdPtr[usbIndex].bus,     &usbdPtr[usbIndex].dev, 
                         &usbdPtr[usbIndex].vendor,  &usbdPtr[usbIndex].product,
                         &usbdPtr[usbIndex].mounted, &usbdPtr[usbIndex].init ) ;
             ofs << gs.ustr() << endl ;
         }
         ofs.close() ;
         this->UserAlert ( 2 ) ;
         nckPause();
      }
   }
   #endif   // DEBUG_USB_SCAN

   this->DeleteFile ( tmpPath.ustr() ) ;  // delete the temporary file

   return usbCount ;

}  //* End USB_DeviceStats()

//*************************
//*    DVD_DeviceStats    *
//*************************
//********************************************************************************
//* Scan for drives attached to the local system.                                *
//* Capture filesystem and support information for the devices.                  *
//*                                                                              *
//* This method is designed primarily for locating optical drives                *
//* (DVD,CD,BLU-RAY, etc). Data discs contained in the drive may, or may not     *
//* be mounted and visible to the user.                                          *
//*                                                                              *
//* Optionally, this method can report information for all attached drives.      *
//* This option is not used by the FileMangler application, but may be helpful   *
//* in other applications.                                                       *
//*                                                                              *
//* Important Note: This method allocates dynamic memory. It is the caller's     *
//*                 responsibility to release this memory when it is no longer   *
//*                 needed. Example: delete [] usbDev ;                          *
//*                                                                              *
//* Input  : usbdPtr  : (by reference, initial value ignored)                    *
//*                     receives pointer to a dynamically-allocated array of     *
//*                     usbDevice objects. Set to NULL pointer if no optical     *
//*                     devices matching the criteria are found.                 *
//*          testmnt  : (optional, 'false' by default)                           *
//*                     if 'false', do not test for mounted status               *
//*                     if 'true',  for each optical drive identified,           *
//*                                 determine whether it is currently mounted    *
//*          all      : (optional, 'false' by default)                           *
//*                     if 'false', report only optical drive devices            *
//*                     if 'true',  summary report of all drives identified      *
//*                                                                              *
//* Returns: count of optical drives captured (elements in usbDevice array)      *
//********************************************************************************
//* Notes:                                                                       *
//* 1) Audio discs will have a URI and therefore have a virtual device driver.   *
//* 2) Commercial and cloned video discs will have NO URI, and therefore will    *
//*    have a kernel "real" device driver.                                       *
//* 3) Data discs will have NO URI, and therefore will have a kernel "real"      *
//*    device driver.                                                            *
//*                                                                              *
//********************************************************************************

short FMgr::DVD_DeviceStats ( usbDevice*& usbdPtr, bool testmnt, bool all )
{
   #define DEBUG_DVD_SCAN (0)    // for debugging only

   const char* const giomountTemplate = 
         "gio mount --list --detail 1>\"%s\" 2>/dev/null" ;
   const char* const driveTag  = "Drive(" ;
   const char* const deviceTag = "unix-device: '" ;   // contains device-driver filespec
   const char* const optiDev   = "/sr" ;              // indicates an optical-device driver
   const char* const themeIcon = "themed icons: " ;   // list of tag strings
   const char* const optiIcon  = "optical" ;          // indicates an optical drive
   const char* const sortKey   = "sort_key=" ;        // last element of base record
   const char* const volumeTag = "  Volume(0): " ;    // begins media record
   const char* const freeVolume= "Volume" ;           // free-range "Volume" record
   const char* const mountTag  = "Mount(0): " ;       // begins mounted-media record

   gString tmpPath ;             // filespec of temporary file
   usbDevice usbDev ;            // filesystem information
   short dvdCount = ZERO ;       // return value

   usbdPtr = NULL ;              // initialize caller's pointer

   //* Verify whether the 'gio' utility is available. *
   //* If not, return immediately with null pointer.  *
   // Programmer's Note: If 'gio' not installed, data returned will be incomplete.
   if ( ! gioInstalled )
      this->gioAvailable ( tmpPath ) ;
   if ( ! gioInstalled )
      return dvdCount ;


   char  lineBuff[gsDFLTBYTES] ; // input buffer
   gString cmdBuff,              // command string buffer
           gs ;                  // text formatting
   short dvdIndex = ZERO ;       // index into 'dbddPtr' array
   bool  pushback = false ;      // 'true' if lineBuff contains unprocessed data

   //* Create a temporary file to capture the raw data.*
   this->CreateTempname ( tmpPath ) ;

   cmdBuff.compose( giomountTemplate, tmpPath.ustr() ) ; // create command
   this->Systemcall ( cmdBuff.ustr() ) ;                 // execute command

   //* Open the temp file and extract the interesting data.*
   ifstream ifs( tmpPath.ustr(), ifstream::in ) ; // open the file
   if ( ifs.is_open() )                           // if file opened
   {
      short di ;                       // input buffer index
      bool keep,                       // 'true' if record to be retained
           done = false ;              // loop control

      //* Count the number of drives located *
      while ( ! done )
      {
         ifs.getline ( lineBuff, gsDFLTBYTES, NEWLINE ) ;// read a source line
         if ( ifs.good() || ifs.gcount() > ZERO )        // if a good read
         {
            gs = lineBuff ;            // working copy of input line
            if ( (gs.find( driveTag )) == ZERO )
               ++dvdCount ;
         }
         else           // end-of-file found
            done = true ;
      }

      //* If one or more records found, find the device  *
      //* filespec and other information for each record.*
      if ( dvdCount > ZERO )
      {
         //* Instantiate storage for each record *
         usbdPtr = new usbDevice[dvdCount] ;

         //* Rewind to top of file *
         ifs.clear() ;                 // reset the EOF flag
         ifs.seekg( ZERO ) ;           // return to top of input file

         //* Capture and format the data                      *
         //* 'all' == false : capture optical-drive data only *
         //* 'all' != false : capture data for all drives     *
         while ( dvdIndex < dvdCount )
         {
            //* If input buffer contains unprocessed data, use it.*
            //* Else, read a line from the source file.           *
            if ( ! pushback )
               ifs.getline ( lineBuff, gsDFLTBYTES, NEWLINE ) ;
            if ( ifs.good() || ifs.gcount() > ZERO || pushback )
            {
               gs = lineBuff ;            // copy of input line
               keep = false ;             // reset the keep-record flag
               pushback = false ;         // reset the pushback flag

               //* Beginning of drive record.         *
               //* Provisionally save the descriptor, *
               //* and parse the record.              *
               if ( (gs.find( driveTag )) == ZERO )
               {
                  gs.copy( usbdPtr[dvdIndex].desc, fssLEN ) ;

                  //* Locate the filespec of the device driver *
                  do
                  {
                     ifs.getline ( lineBuff, gsDFLTBYTES, NEWLINE ) ;
                     if ( ifs.good() || ifs.gcount() > ZERO )
                        gs = lineBuff ;
                     else     // unexpected EOF (unlikely)
                     { di = ERR ; break ; }
                  }
                  while ( (di = gs.after( deviceTag )) <= ZERO ) ;
                  if ( di > ZERO )
                  {
                     if ( all || (gs.find( optiDev )) >= ZERO )
                     {
                        gs.shiftChars( -(di) ) ;
                        gs.erase( L"'" ) ;
                        gs.copy( usbdPtr[dvdIndex].devpath, fssLEN ) ;
                        keep = true ;
                     }
                  }

                  //* If record is to be retained *
                  if ( keep )
                  {
                     do
                     {
                        ifs.getline ( lineBuff, gsDFLTBYTES, NEWLINE ) ;
                        gs = lineBuff ;
                        if ( ifs.good() || ifs.gcount() > ZERO )
                        {
                           if ( (di = gs.after( themeIcon )) > ZERO )
                           {
                              #if DVD_CAPTURE_ALL !=(0) // capture non-essential data
                              gs.shiftChars( -(di) ) ;
                              gs.strip() ;
                              gs.copy( usbdPtr[dvdIndex].icons, fssLEN ) ;
                              #endif   // DVD_CAPTURE_ALL

                              //* If both "srX" and "optical" tests are *
                              //* successful, set optical-drive flag.   *
                              if ( (gs.find( optiIcon )) >= ZERO )
                                 usbdPtr[dvdIndex].optiDrive = true ;
                           }
                           else if ( (di = gs.after( "has_media=" )) > ZERO )
                           {
                              if ( (gs.gstr()[di]) == L'1' )
                                 usbdPtr[dvdIndex].has_media = true ;
                           }
                           else if ( (di = gs.after( "can_eject=" )) > ZERO )
                           {
                              if ( (gs.gstr()[di]) == L'1' )
                                 usbdPtr[dvdIndex].can_eject = true ;
                           }

                           #if DVD_CAPTURE_ALL !=(0) // capture non-essential data
                           else if ( (di = gs.after( "is_removable=" )) > ZERO )
                           {
                              if ( (gs.gstr()[di]) == L'1' )
                                 usbdPtr[dvdIndex].rem = true ;
                           }
                           else if ( (di = gs.after( "is_media_removable=" )) > ZERO )
                           {
                              if ( (gs.gstr()[di]) == L'1' )
                                 usbdPtr[dvdIndex].mrem = true ;
                           }
                           else if ( (di = gs.after( "is_media_check_automatic=" )) > ZERO )
                           {
                              if ( (gs.gstr()[di]) == L'1' )
                                 usbdPtr[dvdIndex].mcheck = true ;
                           }
                           else if ( (di = gs.after( "can_poll_for_media=" )) > ZERO )
                           {
                              if ( (gs.gstr()[di]) == L'1' )
                                 usbdPtr[dvdIndex].mpoll = true ;
                           }
                           else if ( (di = gs.after( "can_start=" )) > ZERO )
                           {
                              if ( (gs.gstr()[di]) == L'1' )
                                 usbdPtr[dvdIndex].can_start = true ;
                           }
                           else if ( (di = gs.after( "can_stop=" )) > ZERO )
                           {
                              if ( (gs.gstr()[di]) == L'1' )
                                 usbdPtr[dvdIndex].can_stop = true ;
                           }
                           else if ( (di = gs.after( "start_stop_type=" )) > ZERO )
                           {
                              gs.substr( usbdPtr[dvdIndex].sstype, di, fssLEN ) ;
                           }
                           #endif   // DVD_CAPTURE_ALL
                        }
                        else     // unexpected EOF (unlikely)
                           break ;
                     }
                     while ( (di = gs.after( sortKey )) < ZERO ) ;
                     #if DVD_CAPTURE_ALL !=(0) // capture non-essential data
                     if ( di >= ZERO )
                     {
                        gs.shiftChars( -(di) ) ;
                        gs.copy( usbdPtr[dvdIndex].sortkey, fssLEN ) ;
                     }
                     #endif   // DVD_CAPTURE_ALL

                     //* For optical drives only, read the next line.  *
                     //* 1) If line describes media contained in the   *
                     //*    drive, capture the media sub-record.       *
                     //*    Example: "Volume(0) Audio Disc"            *
                     //* 2) If line begins a new drive record, push it *
                     //*    back into the stream.                      *
                     if ( usbdPtr[dvdIndex].optiDrive )
                     {
                        bool volFound = false,
                             mntFound = false ;

                        while ( true )
                        {
                           ifs.getline ( lineBuff, gsDFLTBYTES, NEWLINE ) ;
                           if ( ifs.good() || ifs.gcount() > ZERO )
                           {
                              gs = lineBuff ;

                              //* If this data begins a new drive record, *
                              //* it will be processed at the top of loop.*
                              if ( (gs.find( driveTag )) >= ZERO )
                              { pushback = true ; break ; }
                              //* If a free-standing "Volume" record, *
                              //* we're finished with this record.    *
                              if ( (gs.find( freeVolume )) == ZERO )
                                 break ;

                              if ( (di = gs.after( volumeTag )) >= ZERO )
                              {
                                 gs.substr( usbdPtr[dvdIndex].label, di, fssLEN ) ;
                                 volFound = true ;
                              }
                              else if ( volFound && !mntFound && 
                                        ((di = gs.after( "activation_root=" )) > ZERO) )
                              {
                                 //* Save the Universal Resource Identifier *
                                 gs.substr( usbdPtr[dvdIndex].uri, di, fssLEN ) ;
                                 //* Convert the URI to full filespec and  *
                                 //* extract the directory name. Example:  *
                                 //* "cdda://sr0/" becomes "cdda:host=sr0" *
                                 this->Uri2Filespec ( gs, usbdPtr[dvdIndex].uri ) ;
                                 gs.copy( usbdPtr[dvdIndex].mntpath, MAX_PATH ) ;
                                 short trunc = gs.findlast( L'/' ) ;
                                 gs.shiftChars( -(trunc + 1) ) ;
                                 gs.copy( usbdPtr[dvdIndex].mntdir, MAX_FNAME ) ;
                              }
                              else if ( volFound && !mntFound &&
                                        ((di = gs.after( "can_mount=" )) > ZERO) )
                              {
                                 if ( (gs.gstr()[di]) == L'1' )
                                    usbdPtr[dvdIndex].can_mount = true ;
                              }
                              else if ( volFound && !mntFound &&
                                        ((di = gs.after( "label: '" )) > ZERO) )
                              {
                                 gs.shiftChars( -(di) ) ;
                                 if ( (di = gs.findlast( L"'" )) == ((gs.gschars()) - 2) )
                                    gs.limitChars( di ) ;
                                 gs.copy( usbdPtr[dvdIndex].label, fssLEN ) ;
                              }
                              else if ( volFound && !mntFound &&
                                        ((di = gs.after( "uuid: '" )) > ZERO) )
                              {
                                 gs.shiftChars( -(di) ) ;
                                 if ( (di = gs.findlast( L"'" )) == ((gs.gschars()) - 2) )
                                    gs.limitChars( di ) ;
                                 gs.copy( usbdPtr[dvdIndex].uuid, fssLEN ) ;
                              }
                              else if ( volFound && !mntFound &&
                                        ((di = gs.after( "uuid=" )) > ZERO) )
                              {
                                 gs.substr( usbdPtr[dvdIndex].uuid, di, fssLEN ) ;
                              }

                              //* If media are currently mounted *
                              else if ( volFound && 
                                        ((di = gs.after( mountTag )) >= ZERO) )
                              {
                                 #if DVD_CAPTURE_ALL !=(0) // capture non-essential data
                                 gs.shiftChars( -(di) ) ;
                                 if ( (di = gs.after( "-> " )) >= ZERO )
                                    gs.shiftChars( -(di) ) ;
                                 gs.append( " (mountTag)" ) ;
                                 gs.copy( usbdPtr[dvdIndex].dfltloc, MAX_PATH ) ;
                                 #endif   // DVD_CAPTURE_ALL

                                 mntFound = true ;
                              }
                              else if ( volFound && mntFound &&
                                        ((di = gs.after( "can_unmount=" )) > ZERO) )
                              {
                                 if ( (gs.gstr()[di]) == L'1' )
                                    usbdPtr[dvdIndex].can_unmount = true ;
                              }
                              else if ( volFound && mntFound &&
                                        ((di = gs.after( "default_location=" )) > ZERO) )
                              {
                                 #if DVD_CAPTURE_ALL !=(0) // capture non-essential data
                                 gs.substr( usbdPtr[dvdIndex].dfltloc, di, MAX_PATH ) ;
                                 #endif   // DVD_CAPTURE_ALL

                                 //* If no URI was found, parse this entry for *
                                 //* a valid mountpath.                        *
                                 if ( (usbdPtr[dvdIndex].uri[0] == NULLCHAR) &&
                                      ((di = gs.after( "file://" )) > ZERO) )
                                 {
                                    gs.replace( L"%20", L" ", ZERO, false, true ) ;
                                    gs.substr( usbdPtr[dvdIndex].mntpath, di, MAX_PATH ) ;
                                 }
                              }

                              #if DVD_CAPTURE_ALL !=(0) // capture non-essential data
                              else if ( volFound && mntFound &&
                                        ((di = gs.after( "x_content_types: " )) > ZERO) )
                              {
                                 gs.substr( usbdPtr[dvdIndex].mimetype, di, fssLEN ) ;
                              }
                              #endif   // DVD_CAPTURE_ALL
                           }
                           else        // end-of-file
                              break ;
                        }
                     }

                     //* If specified, test whether filesystem is mounted,*
                     //* and if so, gather info about the media.          *
                     // Programmer's Note" When the 'mntFound' flag has been 
                     // set we can be reasonably certain that the media are 
                     // mounted, but we double-check here.
                     if ( testmnt && (usbdPtr[dvdIndex].mntpath[0] != NULLCHAR) )
                     {
                        if ( (this->GetFilesystemStats_aux ( usbdPtr[dvdIndex].mntpath,
                                                             usbDev )) )
                        {
                           usbdPtr[dvdIndex].mounted = true ;
                           usbdPtr[dvdIndex].totalbytes = usbDev.totalbytes ;
                           usbdPtr[dvdIndex].usedbytes  = usbDev.usedbytes ;
                           usbdPtr[dvdIndex].freebytes  = usbDev.freebytes ;
                           usbdPtr[dvdIndex].readonly   = usbDev.readonly ;
                           usbdPtr[dvdIndex].remote     = usbDev.remote ;
                           gs = usbDev.fstype ;
                           gs.copy( usbdPtr[dvdIndex].fstype, fssLEN ) ;
                           gs = usbDev.backend ;
                           gs.copy( usbdPtr[dvdIndex].backend, fssLEN ) ;
                        }
                     }
                     usbdPtr[dvdIndex++].init = true ;
                  }  // keep
               }     // driveTag
            }
            else     // end-of-file
               break ;
         }           // while()
      }              // dvdCount

      ifs.close() ;                    // close the file
   }

   #if DEBUG_DVD_SCAN != 0    // DEBUG ONLY
   //* Dump the contents of the usbDevice array to the *
   //* temp file and pause so the data may be reviewed.*
   ofstream ofs( tmpPath.ustr(), ofstream::out | ofstream::trunc ) ;
   if ( ofs.is_open() )
   {
      gs.compose( "Dump Contents of usbDevice Array - dvdCount:%2hd\n"
                  "==============================================",
                  &dvdCount ) ;
      ofs << gs.ustr() << endl ;

      for ( short i = ZERO ; i < dvdCount ; ++i )
      {
         if ( all || usbdPtr[i].optiDrive )
         {
            ofs << usbdPtr[i].desc 
                << "\n  devpath   : " << usbdPtr[i].devpath 
                << "\n  uri       : " << usbdPtr[i].uri
                << "\n  fstype    : " << usbdPtr[i].fstype
                << "\n  mntdir    : " << usbdPtr[i].mntdir
                << "\n  mntpath   : " << usbdPtr[i].mntpath
                << "\n  uuid      : " << usbdPtr[i].uuid
                << "\n  label     : " << usbdPtr[i].label
                << "\n  backend   : " << usbdPtr[i].backend
                << "\n  optiDrive : " << (usbdPtr[i].optiDrive ? "yes" : "no") 
                << "\n  has_media : " << (usbdPtr[i].has_media ? "yes" : "no") 
                << "\n  readonly  : " << (usbdPtr[i].readonly ? "yes" : "no")
                << "\n  remote    : " << (usbdPtr[i].remote ? "yes" : "no")
                << "\n  can_eject : " << (usbdPtr[i].can_eject ? "yes" : "no")
                << "\n  can_mount : " << (usbdPtr[i].can_mount ? "yes" : "no")
                << "\n  can_unmout: " << (usbdPtr[i].can_unmount ? "yes" : "no")
                << "\n  mounted   : " << (usbdPtr[i].mounted ? "yes" : "no") ;
            gs.formatInt( usbdPtr[i].totalbytes, 13, true ) ;
            ofs << "\n  totalbytes: " << gs.ustr() ;
            gs.formatInt( usbdPtr[i].usedbytes, 13, true ) ;
            ofs << "\n  usedbytes : " << gs.ustr() ;
            gs.formatInt( usbdPtr[i].freebytes, 13, true ) ;
            ofs << "\n  freebytes : " << gs.ustr() 
                << "\n  init      : " << (usbdPtr[i].init ? "yes" : "no") ;

            #if DVD_CAPTURE_ALL !=(0) // REPORT CAPTURED-BUT-NON-ESSENTIAL DATA
            ofs << "\n  ---------------"
                << "\n  icons     : " << usbdPtr[i].icons
                << "\n  sstype    : " << usbdPtr[i].sstype
                << "\n  sortkey   : " << usbdPtr[i].sortkey 
                << "\n  mimetype  : " << usbdPtr[i].mimetype
                << "\n  dfltloc   : " << usbdPtr[i].dfltloc
                << "\n  rem       : " << (usbdPtr[i].rem ? "yes" : "no") 
                << "\n  mrem      : " << (usbdPtr[i].mrem ? "yes" : "no") 
                << "\n  mcheck    : " << (usbdPtr[i].mcheck ? "yes" : "no") 
                << "\n  mpoll     : " << (usbdPtr[i].mpoll ? "yes" : "no")
                << "\n  can_start : " << (usbdPtr[i].can_start ? "yes" : "no")
                << "\n  can_stop  : " << (usbdPtr[i].can_stop ? "yes" : "no")
                << "\n  automnt   : " << (usbdPtr[i].automnt ? "yes" : "no") ;
            #endif   // DVD_CAPTURE_ALL

            ofs << endl ;
         }
      }
      ofs << endl ;
      ofs.close() ;
      this->UserAlert ( 2 ) ;
      nckPause();
   }
   #endif   // DEBUG_DVD_SCAN

   this->DeleteFile ( tmpPath.ustr() ) ;  // delete the temporary file

   return dvdCount ;

   #undef DEBUG_DVD_SCAN
}  //* DVD_DeviceStats() *




#if 0    // OBSOLETE - NOT CURRENTLY USED
//*************************
//*  sdtMgrthread_LevelG  *
//*************************
//********************************************************************************
//* PRIVATE METHOD - called only by stdMgrthread() above.                        *
//*                                                                              *
//* When the base directory of the scan is within a GVfs filesystem, we limit    *
//* the scan to a single (1) sub-thread. This is done to accomodate the fact     *
//* that filesystem storage access on smartphones, tablets and other GVfs        *
//* devices is outrageously, unbearably slow, especially if being simultaneouly  *
//* accessed by multiple threads. This is due primarily to the slow response     *
//* of silicon-based storage devices, but may also be affected by other factors  *
//* such as lack of cache memory on the target device, access-security           *
//* protocols, lack of Linux-like control structures in the filesystem tree,     *
//* and the general "user-friendliness" (barriers) built into such consumer-     *
//* oriented devices.                                                            *
//*                                                                              *
//* Input  : baseCnt  : number of subdirectories and non-dir filenames in CWD    *
//*                     (basecnt.dirCnt > ZERO)                                  *
//*          baseNode : pointer to partially-initialized TreeNode class object   *
//*                     ( baseNode.nextLevel[] array is already initialized )    *
//*          dData    : pointer to DispData-class object for controlling         *
//*                     access to shared data: updateData() method, and for      *
//*                     thread management.                                       *
//*                                                                              *
//* Returns: number of sub-threads allocated (and launched)                      *
//********************************************************************************

UINT FMgr::sdtMgrthread_LevelG ( const nodeCount& baseCnt, TreeNode* baseNode, 
                                 DispData* dData )
{
   USHORT tAlloc = ZERO ;  // number of sub-threads allocated (return value)

   //* Instantiate the array of top-level thread objects.*
   if ( (baseCnt.dirCnt > ZERO) &&
        (tAlloc = dData->createSubthreads( 1 )) == 1 )
   {
      #if DEBUG_MULTITHREAD != 0
      if ( dmtofs.is_open() )
      {
         dmtOut.compose( "** Level G Threads Allocated : %3hu of   1 **", &tAlloc ) ;
         dmtofs << dmtOut.ustr() << endl ;
      }
      #endif   // DEBUG_MULTITHREAD

      //* Signal that the user interface (progress monitor) *
      //* thread may continue.                              *
      dData->unblockUI() ;

      //* Pause timer for launched thread to respond *
      chrono::duration<short, std::milli>aMoment( 80 ) ;

      //* Put the thread to work. *
      gString basePath( baseCnt.trgPath ) ;  // filespec of base directory
      UINT   tIndx = ZERO ;                  // index of thread definition
      for ( short i = 3 ; i > ZERO ; --i ) // three strikes, and you're out!
      {
         try
         {
            dData->scanThread[tIndx] = 
               thread( &FMgr::sdtGvfsthread, this, basePath, baseNode, dData ) ;

            //* Wait for the launched thread to respond.*
            for ( short j = 3 ; j > ZERO ; --j )
            {
               this_thread::sleep_for( aMoment ) ;
               if ( (dData->goodSubthreads()) > ZERO )
                  break ;

               #if DEBUG_MULTITHREAD != 0
               if ( dmtofs.is_open() )
               {
                  dmtOut.compose( "Scan sub-thread not responding. (j==%hd)", &j ) ;
                  dmtofs << dmtOut.ustr() << endl ;
               }
               #endif   // DEBUG_MULTITHREAD

               if ( j <= 1 )
                  dData->setException( "Scan sub-thread not responding." ) ;
            }
            break ;
         }
         catch ( ... )     // system exception
         {
            #if DEBUG_MULTITHREAD != 0
            if ( dmtofs.is_open() )
            {
               dmtofs << dData->scanThreadPath[tIndx].ustr() 
                      << "  (" << tIndx << ")\n"
                         " == System Exception sdtSubthread_LevelG() launch. =="
                      << endl ;
            }
            #endif   // DEBUG_MULTITHREAD

            //* If terminal retry failed, increment exception counter.*
            if ( i <= 1 )
               dData->setException( "Scan sub-thread launch failed." ) ;
            else
               this_thread::sleep_for( aMoment ) ;  // give the system some time
         }
      }  // for(;;)
   }        // sub-threads allocated

   //* Sub-threads unneeded, OR system error in allocation of sub-threads. *
   else
   {
      //* Signal that the user interface (progress monitor) *
      //* thread may continue.                              *
      dData->unblockUI() ;

      #if DEBUG_MULTITHREAD != 0
      if ( baseCnt.dirCnt > ZERO )
      {
         if ( dmtofs.is_open() )
         {
            dmtofs << " ==========================================\n"
                      " = System Exception in CreateSubthreads() =\n"
                      " ==========================================" << endl ;
         }
      }
      #endif   // DEBUG_MULTITHREAD
   }

   return tAlloc ;

}  //* End sdtMgrthread_LevelG() *

//*************************
//*     sdtGvfsthread     *
//*************************
//********************************************************************************
//* PRIVATE METHOD                                                               *
//* --------------                                                               *
//* Called only by sdtMgrthread_G(), above. Scans the contents of a GVfs         *
//* filesystem using a single thread for the entire directory tree.              *
//*                                                                              *
//* -- The top-level subdirectories and top-level non-directory files have       *
//*    already been scanned.                                                     *
//*                                                                              *
//* -- The CONTENTS of each top-level subdirectory will be scanned in sequence,  *
//*    and the data will be appended to the specified TreeNode object.           *
//*                                                                              *
//* -- Note: It is assumed that all data scanned will be on a single             *
//*    filesystem, i.e. that no filesystem will be mounted WITHIN the target     *
//*    filesystem.                                                               *
//*                                                                              *
//*                                                                              *
//*                                                                              *
//* Input  : dPath  : full path spec of directory in which to begin scan         *
//*          tnPtr  : pointer to TreeNode class object with only the top-level   *
//*                   stats initialized. All second-level data "nextLevel"       *
//*                   set to initialization values.                              *
//*          dData  : pointer to DispData-class object for controlling           *
//*                   access to shared data. This method uses only:              *
//*                   goodSubthread(),updateData() and completedSubthread()      *
//*                   methods of the DispData class.                             *
//*                                                                              *
//* Returns: nothing (does not "return" in the conventional sense)               *
//********************************************************************************

void FMgr::sdtGvfsthread ( const gString& dPath, TreeNode* tnPtr, DispData* dData )
{
   //* Register that thread was properly launched.*
   dData->goodSubthread() ;

   gString gsPath ;           // formatting of filespecs
   nodeCount nodecnt ;        // count contents of lower-level nodes

   #if DEBUG_MULTITHREAD != 0 && DMT_VERBOSE != 0 
   if ( dmtofs.is_open() )
   {
      dmtOut.compose( dmtFmt, dPath.ustr(), &tnPtr->nlnodes, 
                      &tnPtr->tnfcount, tnPtr->dirStat.fName, 
                      &tnPtr->dirStat.fsMatch ) ;
     dmtofs << "\n" << dmtOut.ustr() << endl ;
   }
   #endif   // DEBUG_MULTITHREAD && DMT_VERBOSE

   //* Scan each node of the tree *
   for ( UINT d = ZERO ; d < tnPtr->nlnodes ; ++d )
   {
      #if 0    // DISABLE RECURSIVE SCAN FOR SMARTPHONES
      //* Summarize the node contents.*
      nodecnt.reset() ;
      nodecnt.level = 1 ;     // target is a second-level directory
      gsPath.compose( "%S/%s", dPath.gstr(), tnPtr->nextLevel[d].dirStat.fName ) ;
      gsPath.copy( nodecnt.trgPath, MAX_PATH ) ;
      this->DirectoryCount ( nodecnt ) ;

      #if DEBUG_MULTITHREAD != 0 && DMT_VERBOSE != 0 
      if ( dmtofs.is_open() )
      {
         dmtOut.compose( dmtFmt, nodecnt.trgPath, &nodecnt.dirCnt, &nodecnt.regCnt, 
                         tnPtr->nextLevel[d].dirStat.fName, 
                         &tnPtr->nextLevel[d].dirStat.fsMatch ) ;
         dmtofs << dmtOut.ustr() << endl ;
      }
      #endif   // DEBUG_MULTITHREAD && DMT_VERBOSE

      //* Scan the specified branch of the directory tree,   *
      //* creating storage nodes at each level. Scan-summary *
      //* data are appended to caller's TreeNode object.     *
      this->ScanDirTree ( nodecnt, &tnPtr->nextLevel[d] ) ;
      #endif   // DISABLE RECURSIVE SCAN FOR SMARTPHONES
   }     // for(;;)

   //* Count the number of files processed.*
   UINT   fc = ZERO ;         // files scanned
   UINT64 bc = ZERO ;         // file bytes
   this->TreeNodeSummary ( tnPtr, fc, bc ) ;
   dData->updateData ( fc, bc ) ;

   #if DEBUG_MULTITHREAD != 0 && DMT_VERBOSE != 0 
   if ( dmtofs.is_open() )
   {
      --fc ; bc -= tnPtr->dirStat.fBytes ; // do not report data for base directory
      dmtOut.compose( "Summary: %u files, %llu bytes\n"
                      "** End sdtGvfsthread()\n", &fc, &bc ) ;
      dmtofs << dmtOut.ustr() << endl ;
   }
   #endif   // DEBUG_MULTITHREAD && DMT_VERBOSE

   //* Decrement the active-subthread counter *
   dData->completedSubthread () ;

   //* IMPORTANT NOTE: When the thread leaves the scope of this method, 
   //*                 it exits (stops executing); however, its definition
   //*                 and data still exist, and must be released separately.

}  //* End sdtGvfsthread() *
#endif   // OBSOLETE - NOT CURRENTLY USED

#undef GVFS_FAILSAFE    // this definition is specific to this module

//*************************
//*                       *
//*************************
//********************************************************************************
//*                                                                              *
//*                                                                              *
//*                                                                              *
//* Input  :                                                                     *
//*                                                                              *
//* Returns:                                                                     *
//********************************************************************************

