//********************************************************************************
//* File       : FMgrSort.cpp                                                    *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2005-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in FileMangler.hpp         *
//* Date       : 19-Apr-2025                                                     *
//* Version    : see FMgrVersion[] string in FMgr.cpp                            *
//*                                                                              *
//* Description: This source file contains the file-sorting methods for the      *
//* FMgr class.                                                                  *
//*                                                                              *
//* Developed using GNU G++ (Gcc v: 4.4.2)                                       *
//*  under Fedora Release 12, Kernel Linux 2.6.31.5-127.fc12.i686.PAE and above  *
//********************************************************************************
//* Programmer's Notes:                                                          *
//* - We have implemented alphabetical sorting in a way that isolates the        *
//*   the actual library call used. This is done because we have a design        *
//*   decision about accuracy versus CPU efficiency versus speed.                *
//*                                                                              *
//* - The comparisons for alphabetical data are isolated in the  rsddCompare()   *
//*   and rsddCompareI() methods for easy future updates to the type of          *
//*   comparison used. These sorting methods are used for alphabetical sorting   *
//*   (Name, Reverse Name, Extension, Reverse Extension).                        *
//*                                                                              *
//* - The actual type of character sort to be used has not yet been finalized.   *
//*   We have four basic choices, three of which are fully implemented, but      *
//*   only the the first two options are practical.                              *
//*                See SORT_TYPE conditional compile, below.                     *
//*   a) Numerical sort using wcsncmp() and wcsncasecmp(). This is               *
//*      fairly efficient, but is not sensitive to localized alphabets.          *
//*      Although wchar_t characters generally follow an alphabetical ordering,  *
//*      mixed ASCII and supra-ASCII cannot be guaranteed as both numerically    *
//*      ordered and alphabetically ordered.                                     *
//*   b) True alpha sorting using the inefficient wcscoll() which                *
//*      requires multiple string conversions for each element of the            *
//*      array. (uses the LC_COLLATE category of current locale AND takes        *
//*      out of our hands the decision whether to use case-sensitive sorting,    *
//*      AND does not have a character limit i.e. no wcsncoll())                 *
//*   c) True alpha sorting using the more efficient, but expensive and          *
//*      awkward wcsxfrm()/wcsncmp() combination which requires only one         *
//*      transformation for each string but needs a huge temporary memory        *
//*      allocation and a different sorting algorithm. [NOT IMPLEMENTED]         *
//*   d) A UTF-8 string comparison (strncmp) is inexpensive but is actually      *
//*      accurate only for ASCII filenames. This is because UTF-8 encoding is    *
//*      designed neither for alphabetical order nor for numerical linearity     *
//*      and is therefore unsuitable for internationalized application code.     *
//********************************************************************************

#include "GlobalDef.hpp"
#ifndef FMGR_INCLUDED
#include "FMgr.hpp"
#endif

//** Local Definitions **
// NOTE: Debug routines work properly ONLY is Single-Window mode
#define DEBUGrsdd (0)      // SET TO 1 FOR ReSortDirData() DEBUG ONLY
#define rsddRANDOMIZE (0)  // SET TO 1 TO RANDOMIZE DATA BEFORE SORT - DEBUG ONLY

//* Type of sort to perform for alphabetical data (see notes above). *
//* 0 == use wcsncmp() and wcscasecmp()
//* 1 == use wcscoll()
//* 2 == use strncmp() and strncasecmp()
//* 3 == use wcsxfrm() with wcsncmp() and wcsncasecmp() [NOT IMPLEMENTED]
#define SORT_TYPE 0

//** Local Prototypes **
static int  rsddCompare ( const gString& sa, const gString&sb ) ;
static int  rsddCompareI ( const gString& sa, const gString&sb ) ;


//*************************
//*     SetSortOption     *
//*************************
//******************************************************************************
//* Set the directory-sort option, and re-sort the existing display data.      *
//*                                                                            *
//* If there is currently file-display data, sort it according to the new      *
//* sort option. (Does not update the display.)                                *
//*                                                                            *
//* Input  : sOption: member of enum fmSort                                    *
//*                                                                            *
//* Returns: OK if valid input, else ERR                                       *
//******************************************************************************

short FMgr::SetSortOption ( fmSort sOption )
{
short    success = ERR ;

   if ( sOption >= fmNO_SORT && sOption < fmSORT_OPTIONS )
   {
      this->sortOption = sOption ;
      this->ReSortDirData ( this->sortOption ) ;
      success = OK ;
   }

   return success ;
   
}  //* End SetSortOption() *

//*************************
//*     GetSortOption     *
//*************************
//******************************************************************************
//* Returns the current sort option.                                           *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: member of enum fmSort                                             *
//******************************************************************************

fmSort FMgr::GetSortOption ( void )
{

   return this->sortOption ;

}  //* End GetSortOption() *

//**************************
//* CaseSensitiveAlphaSort *
//**************************
//******************************************************************************
//* Set or reset the case-sensitive flag for sorting display data              *
//* alphabetically, and re-sort the existing display data (if any).            *
//* THIS METHOD DOES NOT RE-DRAW THE DISPLAY.                                  *
//*                                                                            *
//* This flag affects the following sort options: fmNAME_SORT, fmNAMEr_SORT,   *
//* fmEXT_SORT, fmEXTr_SORT.                                                   *
//*                                                                            *
//* Input  : 'true' for case-sensitive sort (default on instantiation)         *
//*          'false' for case-insensitive sort                                 *
//*                                                                            *
//* Returns: OK                                                                *
//******************************************************************************

short FMgr::CaseSensitiveAlphaSort ( bool caseSensitive )
{
   this->caseSensitive = caseSensitive ;
   this->ReSortDirData ( this->sortOption ) ;

   return OK ;
   
}  //* End CaseSensitiveAlphaSort() *

//*************************
//*    ReSortDirData      *
//*************************
//******************************************************************************
//* Sort the data which is currently displayed in the file-display control.    *
//* Retain 'selection' on any selected items and do not discard any data that  *
//* are currently on the clipboard.                                            *
//*                                                                            *
//* We must sort all three display-data sets:                                  *
//*     1) the array of DirEntry pointers referenced by this->deList           *
//*     2) the array of display strings referenced by this->textData           *
//*     3) the array of color attributes referenced by this->colorData         *
//* according to the specified sort option (see enum fmSort).                  *
//*                                                                            *
//* The FMgr class uses a sophisticated, two-level sort, known as a cocktail   *
//* sort or bi-directional bubble sort; and although the algorithm is not fast,*
//* the data set is usually less than 100 items and and seldom more than       *
//* 200 items.                                                                 *
//*                                                                            *
//* Because moving that much data is rather expensive, we pretest the data to  *
//* determine where it will go in the new order before moving it. The exception*
//* is subdirectory names which are always grouped at either the top or bottom *
//* of the array except when the NO_SORT option is selected.                   *
//*                                                                            *
//*                                                                            *
//* SORT ORDER:                                                                *
//* -----------                                                                *
//* fmNAME_SORT, sort by filename - ascending order                            *
//*  - Directories are grouped at the top of the list are are sub-sorted by    *
//*    filename (ascending).                                                   *
//*                                                                            *
//* fmNAME_SORTr, sort by filename - descending order                          *
//*  - Directories are grouped at the top of the list are are sub-sorted by    *
//*    filename (descending).                                                  *
//*                                                                            *
//* fmDATE_SORT, sort by mod date - ascending order                            *
//*  - Directories are grouped at the top of the list are are sub-sorted by    *
//*    filename (ascending).                                                   *
//*  - Note that the mod date of a directory is changed every time something   *
//*    within that directory is updated. This could lead to all kinds of       *
//*    headaches, so we ignore it.                                             *
//*  - Within a group of files with the same mod date, files are sorted by     *
//*    filename (ascending).                                                   *
//*                                                                            *
//* fmDATE_SORTr, sort by mod date - descending order                          *
//*  - Directories are grouped at the top of the list are are sub-sorted by    *
//*    filename (ascending).                                                   *
//*  - Within a group of files with the same mod date, files are sorted by     *
//*    filename (ascending).                                                   *
//*                                                                            *
//* fmSIZE_SORT, sort by file size - ascending order                           *
//*  - Directories are grouped at the top of the list are are sub-sorted by    *
//*    filename (ascending).                                                   *
//*  - Within a group of files with the same file size, files are sorted by    *
//*    filename (ascending).                                                   *
//*                                                                            *
//* fmSIZE_SORTr, sort by file size - descending order                         *
//*  - Directories are grouped at the top of the list are are sub-sorted by    *
//*    filename (ascending).                                                   *
//*  - Within a group of files with the same file size, files are sorted by    *
//*    filename (ascending).                                                   *
//*                                                                            *
//* fmTYPE_SORT, sort by file type - ascending order                           *
//*  - Directories are grouped at the top of the list are are sub-sorted by    *
//*    filename (ascending).                                                   *
//*  - Regular files are grouped next and are sub-sorted by filename           *
//*    (ascending)                                                             *
//*  - The remaining types are in arbitrary order based on the order of        *
//*    elements in enum fmFType. All are sub-sorted by filename (ascending).   *
//*                                                                            *
//* fmTYPE_SORTr, sort by file type - descending order                         *
//*  - The order of this sort is the reverse of fmTYPE_SORT, with the directory*
//*    type at the bottom of the list and 'unknown' types at the top.          *
//*  - All types are sub-sorted by filename (ascending)                        *
//*                                                                            *
//* fmEXT_SORT, sort by filename extension - ascending order                   *
//*    (filename extension is the part of the filename after the rightmost     *
//*     period, if any)                                                        *
//*  - Directories are grouped at the top of the list are are sub-sorted by    *
//*    filename (ascending).                                                   *
//*    (Directory names may have extensions, but we treat them as merely part  *
//*     part of the filename.)                                                 *
//*  - All other files are sorted alphabetically by extension.                 *
//*  - Within a group of files with the same filename extension, files are     *
//*    sorted by filename (ascending).                                         *
//*                                                                            *
//* fmEXT_SORTr, sort by filename extension - descending order                 *
//*  - Directories are grouped at the top of the list are are sub-sorted by    *
//*    filename (ascending).                                                   *
//*  - All other files are sorted reverse-alphabetically by extension.         *
//*  - Within a group of files with the same filename extension, files are     *
//*    sorted by filename (ascending).                                         *
//*                                                                            *
//* fmNO_SORT, sort the display data according to the order of elements in the *
//*            original, deHead linked list of DirEntry structures because     *
//*            this is the 'unsorted' order in which the directory data were   *
//*            originally read.                                                *
//*                                                                            *
//* Input Values : member of enum fmSort                                       *
//*                                                                            *
//* Return Value : none                                                        *
//*                CALLER IS RESPONSIBLE FOR RE-DRAWING THE DISPLAY            *
//******************************************************************************

void FMgr::ReSortDirData ( fmSort sOption )
{
   #if DEBUGrsdd != 0
   // NOTE: Debug routines work properly ONLY is Single-Window mode
   int      dYpos = 1,
            dXpos = 82 ;
   char     dBuff[64] ;
      //* Clear the area for display of debug data *
      for ( int i = ZERO ; i < 40 ; i++ )
         nc.ClearLine ( i, true, 80 ) ;
   #endif   // DEBUGrsdd

   //* If data-space has been allocated for the file list *
   if ( this->deHead != NULL )
   {
      UINT totalFiles = this->deHead[BNODE].totFiles ; // number of files in list

      //* Create an array to hold pointers to the data *
      ReSort   iList[totalFiles] ;
      for ( UINT i = ZERO ; i < totalFiles ; i++ )
      {
         iList[i].tPtr  = this->textData[i] ;
         iList[i].cAttr = this->colorData[i] ;
         iList[i].dPtr  = this->deList[i] ;
      }

      #if rsddRANDOMIZE != 0     // DEBUG ONLY
      rsddRandomize ( iList, totalFiles ) ;
      #endif   // rsddRANDOMISE

      //* Move the subdirectory names to the top/bottom of the list and sort them *
      //* in the appropriate order. Return value is number of directories in list.*
      //* (Note that for TYPE and TYPEr, directories are naturally positioned     *
      //* because fmDIR_TYPE == ZERO. Keep it that way!)                          *
      UINT  dirCount = ZERO ;
      if ( sOption != fmNO_SORT && sOption != fmTYPE_SORT && sOption != fmTYPEr_SORT )
         dirCount = rsddIsolateDirNames ( iList, totalFiles, sOption, this->caseSensitive ) ;
      UINT  itemCount = totalFiles - dirCount ;

      //* Do the primary sort *
      if ( itemCount > ZERO )       // if 1 or more non-directory items were found
      {
         switch ( sOption )
         {
            case fmNAME_SORT:
               rsddName_Sort ( &iList[dirCount], itemCount, this->caseSensitive ) ;
               #if DEBUGrsdd != 0
               nc.WriteString ( ZERO, 80, "primary", nc.bl ) ;
               for ( UINT i = ZERO ; i < totalFiles ; i++ )
               {
                  snprintf ( dBuff, 64, "%2u: %s %c", i, iList[i].dPtr->fName,
                              iList[i].dPtr->fType == fmDIR_TYPE ? 'D' : SPACE ) ;
                  nc.WriteString ( dYpos++, dXpos, dBuff, nc.bl ) ;
               }
               #endif   // DEBUGrsdd
               break ;
            case fmNAMEr_SORT:
               rsddNameR_Sort ( &iList[dirCount], itemCount, this->caseSensitive ) ;
               break ;
            case fmDATE_SORT:
               rsddDate_Sort  ( &iList[dirCount], itemCount, this->caseSensitive ) ;
               break ;
            case fmDATEr_SORT:
               rsddDateR_Sort ( &iList[dirCount], itemCount, this->caseSensitive ) ;
               break ;
            case fmSIZE_SORT:
               rsddSize_Sort  ( &iList[dirCount], itemCount, this->caseSensitive ) ;
               break ;
            case fmSIZEr_SORT:
               rsddSizeR_Sort  ( &iList[dirCount], itemCount, this->caseSensitive ) ;
               break ;
            case fmTYPE_SORT:
               rsddType_Sort  ( &iList[dirCount], itemCount, this->caseSensitive ) ;
               break ;
            case fmTYPEr_SORT:
               rsddTypeR_Sort ( iList, itemCount, this->caseSensitive ) ;
               break ;
            case fmEXT_SORT:
               rsddExt_Sort  ( &iList[dirCount], itemCount, this->caseSensitive ) ;
               break ;
            case fmEXTr_SORT:
               rsddExtR_Sort ( &iList[dirCount], itemCount, this->caseSensitive ) ;
               break ;
            case fmNO_SORT:
               rsddUn_Sort ( iList, totalFiles ) ;
            default:
               break ;     // this is very unlikely
         }

      }  // if(itemCount>ZERO)

      //* Move the data to the newly assigned positions in the array *
      if ( totalFiles > ZERO && (itemCount > ZERO || dirCount > ZERO ) )
      {
         for ( UINT i = ZERO ; i < totalFiles ; i++ )
         {
            this->textData[i]  = iList[i].tPtr ;
            this->colorData[i] = iList[i].cAttr ;
            this->deList[i]    = iList[i].dPtr ;
         }
      }
   }
}  //* End ReSortDirData() *

//******************************************************************************
//**               PRIVATE. METHODS USED IN SORTING FILENAMES                  *
//******************************************************************************

//*************************
//* rsddIsolateDirNames   *
//*************************
//******************************************************************************
//* Find the entries in the list that are of type fmDIR_TYPE and group them    *
//* at the top of the file list.                                               *
//* Then, sort the directory names in the group in the specified order         *
//* (ascending or descending).                                                 *
//*                                                                            *
//* Note that there is a high probability that the directory entries are       *
//* ALREADY in the correct position AND in the correct order, so do inexpensive*
//* tests first.                                                               *
//*                                                                            *
//* Input Values : iList     : pointer to a list of ReSort class objects       *
//*                fCount    : number of items in iList                        *
//*                sOption   : type of sort for directory names                *
//*                sensi     : 'true' if case-sensitive alpha sort,            *
//*                            else 'false'                                    *
//*                                                                            *
//* Return Value : number of directory names in the file list                  *
//******************************************************************************
//*                                                                            *
//******************************************************************************

UINT FMgr::rsddIsolateDirNames ( ReSort iList[], UINT fCount, 
                                 fmSort sOption, bool sensi )
{
   UINT     dirCount = ZERO,
            firstDir, lastDir, i ;
   #define DEBUGidn (0)
   #if DEBUGrsdd != 0 && DEBUGidn != 0
   int   dYpos = 1,
         dXpos = 82,
         loopCounter = ZERO,
         sMoves = ZERO ;
   char  dBuff[64] ;
   #endif   // DEBUGrsdd && DEBUGidn

   //* Count the directory entries in the list and       *
   //* note the index of first and last directory entry  *
   for ( i = ZERO ; i < fCount ; i++ )
   {
      //* If entry is of directory type *
      if ( iList[i].dPtr->fType == fmDIR_TYPE )
      {
         if ( dirCount == ZERO )
            firstDir = lastDir = i ;
         else
            lastDir = i ;
         ++dirCount ;
      }
   }
   //* If at least one directory item was found AND there is also at least one *
   //* non-directory item, continue - else isolation is complete.              *
   if ( dirCount > ZERO && dirCount < fCount )
   {
      //* If the directory entries are not already grouped *
      //* at top of list, do it now.                       *
      if ( ((lastDir - firstDir + 1) != dirCount) || firstDir > ZERO )
      {
         ReSort   saveIt ;
         UINT     src, trg ;
         //* Find the first non-directory position *
         i = ZERO ;
         while ( (i < fCount) && (iList[i].dPtr->fType == fmDIR_TYPE) )
            ++i ;
         trg = i ;
         for ( ++i ; i < fCount ; i++ )
         {
            #if DEBUGrsdd != 0 && DEBUGidn != 0
            ++loopCounter ;
            #endif   // DEBUGrsdd && DEBUGidn
            //* Find next directory position *
            while ( (i < fCount) && (iList[i].dPtr->fType != fmDIR_TYPE) )
               ++i ;
            if ( i >= fCount )   // are we done?
               break ;
            src = i ;

            #if DEBUGrsdd != 0 && DEBUGidn != 0
            ++sMoves ;
            #endif   // DEBUGrsdd && DEBUGidn
            //* Do the swap *
            saveIt = iList[trg] ;
            iList[trg] = iList[src] ;
            iList[src] = saveIt ;

            //* Find the next non-directory position *
            while ( (trg < fCount) && (iList[trg].dPtr->fType == fmDIR_TYPE) )
               ++trg ;
            if ( trg >= fCount ) // are we done?
               break ;
         }
         firstDir = ZERO ;
      }
   }
   #if DEBUGrsdd != 0 && DEBUGidn != 0
   snprintf ( dBuff, 64, "primary loops:%3u - moves:%3u", loopCounter, sMoves ) ; 
   nc.WriteString ( ZERO, 80, dBuff, nc.bl ) ;
   for ( UINT i = ZERO ; i < fCount ; i++ )
   {
      snprintf ( dBuff, 64, "%2u: %4u %c", i, iList[i].dPtr->fType,
                  iList[i].dPtr->fType == fmDIR_TYPE ? 'D' : SPACE ) ;
      nc.WriteString ( dYpos++, dXpos, dBuff, nc.bl ) ;
   }
   chrono::duration<short>aWhile( 5 ) ;
   this_thread::sleep_for( aWhile ) ;
   #endif   // DEBUGrsdd && DEBUGidn

   if ( dirCount > 1 )
   {
      switch ( sOption )
      {
         #if 0    // DISABLE DATE SORT FOR DIRECTORIES
         // Programmer's Note: Sort-by-date is not such a good idea for 
         // directories because entries will not align them with their 
         // couterparts in the alternate window. On the other hand, it would 
         // put the most-recently-modified directory at the bottom/top of 
         // the list indicating that it contains the most-recently-modified data.
         case fmDATEr_SORT:
            rsddDateR_Sort ( &iList[firstDir], dirCount, sensi ) ;
            break ;
         case fmDATE_SORT:
            rsddDate_Sort ( &iList[firstDir], dirCount, sensi ) ;
            break ;
         #endif   // DISABLE DATE SORT FOR DIRECTORIES

         case fmNAMEr_SORT:
            rsddNameR_Sort ( &iList[firstDir], dirCount, sensi ) ;
            break ;
         default:       // sort by name, ascending
            rsddName_Sort ( &iList[firstDir], dirCount, sensi ) ;
            break ;
      }
   }
   return dirCount ;

   #undef DEBUGidn
}  //* End rsddIsolateDirNames() *

//**************************
//** Primary-sort methods **
//**************************
//*************************
//*   rsddName_Sort       *
//*************************
void FMgr::rsddName_Sort ( ReSort iList[], UINT fCount, bool sensi )
{
// Programmer's Note: For wide character comparisons, see wcscmp() and wcscasecmp()
ReSort   saveLow, saveHigh ;
gString  gsi, gsl, gsg, gss ;
int      lIndex, gIndex,               // low and high limits search indices
         result, i ;
#if DEBUGrsdd != 0
int   loopCounter = ZERO,
      sMoves = ZERO ;
#endif   // DEBUGrsdd

   for ( lIndex = 0, gIndex = fCount-1 ; (gIndex-lIndex) > ZERO ; lIndex++, gIndex-- )
   {
      #if DEBUGrsdd != 0
      ++loopCounter ;
      #endif   // DEBUGrsdd

      for ( i = lIndex+1 ; i <= gIndex ; i++ )
      {
         gsi = iList[i].dPtr->fName ;
         gsl = iList[lIndex].dPtr->fName ;
         if ( sensi )
            result = rsddCompare ( gsi, gsl ) ;
         else
            result = rsddCompareI ( gsi, gsl ) ;
         if ( result < ZERO )
         {
            #if DEBUGrsdd != 0
            ++sMoves ;
            #endif   // DEBUGrsdd
            saveLow = iList[lIndex] ;
            iList[lIndex] = iList[i] ;
            iList[i] = saveLow ;
            gss = gsl ;
            gsl = gsi ;
            gsi = gss ;
         }
            
         if ( i < gIndex )
         {
            gsg = iList[gIndex].dPtr->fName ;
            if ( sensi )
               result = rsddCompare ( gsg, gsi ) ;
            else
               result = rsddCompareI ( gsg, gsi ) ;
            if ( result < ZERO )
            {
               #if DEBUGrsdd != 0
               ++sMoves ;
               #endif   // DEBUGrsdd
               saveHigh = iList[gIndex] ;
               iList[gIndex] = iList[i] ;
               iList[i] = saveHigh ;
               gsi = gsg ;

               if ( sensi )
                  result = rsddCompare ( gsi, gsl ) ;
               else
                  result = rsddCompareI ( gsi, gsl ) ;
               if ( result < ZERO )
               {
                  #if DEBUGrsdd != 0
                  ++sMoves ;
                  #endif   // DEBUGrsdd
                  saveLow = iList[lIndex] ;
                  iList[lIndex] = iList[i] ;
                  iList[i] = saveLow ;
               }
            }
         }
      }
   }

}  //* End rsddName_Sort() *

//*************************
//*   rsddNameR_Sort      *
//*************************
void FMgr::rsddNameR_Sort ( ReSort iList[], UINT fCount, bool sensi )
{
ReSort   saveLow, saveHigh ;
gString  gsi, gsl, gsg, gss ;
int      lIndex, gIndex,               // low and high limits search indices
         result, i ;
//bool     sensi = this->caseSensitive ;
#if DEBUGrsdd != 0
int   loopCounter = ZERO,
      sMoves = ZERO ;
#endif   // DEBUGrsdd

   for ( lIndex = 0, gIndex = fCount-1 ; (gIndex-lIndex) > ZERO ; lIndex++, gIndex-- )
   {
      #if DEBUGrsdd != 0
      ++loopCounter ;
      #endif   // DEBUGrsdd

      for ( i = lIndex+1 ; i <= gIndex ; i++ )
      {
         gsi = iList[i].dPtr->fName ;
         gsl = iList[lIndex].dPtr->fName ;
         if ( sensi )
            result = rsddCompare ( gsi, gsl ) ;
         else
            result = rsddCompareI ( gsi, gsl ) ;
         if ( result > ZERO )
         {
            #if DEBUGrsdd != 0
            ++sMoves ;
            #endif   // DEBUGrsdd
            saveLow = iList[lIndex] ;
            iList[lIndex] = iList[i] ;
            iList[i] = saveLow ;
            gss = gsl ;
            gsl = gsi ;
            gsi = gss ;
         }
            
         if ( i < gIndex )
         {
            gsg = iList[gIndex].dPtr->fName ;
            if ( sensi )
               result = rsddCompare ( gsg, gsi ) ;
            else
               result = rsddCompareI ( gsg, gsi ) ;
            if ( result > ZERO )
            {
               #if DEBUGrsdd != 0
               ++sMoves ;
               #endif   // DEBUGrsdd
               saveHigh = iList[gIndex] ;
               iList[gIndex] = iList[i] ;
               iList[i] = saveHigh ;
               gsi = gsg ;

               if ( sensi )
                  result = rsddCompare ( gsi, gsl ) ;
               else
                  result = rsddCompareI ( gsi, gsl ) ;
               if ( result > ZERO )
               {
                  #if DEBUGrsdd != 0
                  ++sMoves ;
                  #endif   // DEBUGrsdd
                  saveLow = iList[lIndex] ;
                  iList[lIndex] = iList[i] ;
                  iList[i] = saveLow ;
               }
            }
         }
      }
   }

}  //* End rsddNameR_Sort() *

//*************************
//*   rsddDate_Sort       *
//*************************
void FMgr::rsddDate_Sort ( ReSort iList[], UINT fCount, bool sensi )
{
ReSort   saveLow, saveHigh ;
int      lIndex, gIndex,               // low and high limits search indices
         i ;
#if DEBUGrsdd != 0
int   dYpos = 1,
      dXpos = 82,
      loopCounter = ZERO,
      sMoves = ZERO ;
char  dBuff[64] ;
#endif   // DEBUGrsdd

   for ( lIndex = 0, gIndex = fCount-1 ; (gIndex-lIndex) > ZERO ; lIndex++, gIndex-- )
   {
      #if DEBUGrsdd != 0
      ++loopCounter ;
      #endif   // DEBUGrsdd

      for ( i = lIndex+1 ; i <= gIndex ; i++ )
      {
         if ( iList[i].dPtr->rawStats.st_mtime < iList[lIndex].dPtr->rawStats.st_mtime )
         {
            #if DEBUGrsdd != 0
            ++sMoves ;
            #endif   // DEBUGrsdd
            saveLow = iList[lIndex] ;
            iList[lIndex] = iList[i] ;
            iList[i] = saveLow ;
         }
            
         if ( i < gIndex )
         {
            if ( iList[gIndex].dPtr->rawStats.st_mtime < iList[i].dPtr->rawStats.st_mtime )
            {
               #if DEBUGrsdd != 0
               ++sMoves ;
               #endif   // DEBUGrsdd
               saveHigh = iList[gIndex] ;
               iList[gIndex] = iList[i] ;
               iList[i] = saveHigh ;
               
               if ( iList[i].dPtr->rawStats.st_mtime < iList[lIndex].dPtr->rawStats.st_mtime )
               {
                  #if DEBUGrsdd != 0
                  ++sMoves ;
                  #endif   // DEBUGrsdd
                  saveLow = iList[lIndex] ;
                  iList[lIndex] = iList[i] ;
                  iList[i] = saveLow ;
               }
            }
         }
      }
   }
   #if DEBUGrsdd != 0
   snprintf ( dBuff, 64, "primary loops:%3u - moves:%3u", loopCounter, sMoves ) ; 
   nc.WriteString ( ZERO, 80, dBuff, nc.bl ) ;
   for ( UINT i = ZERO ; i < fCount ; i++ )
   {
      snprintf ( dBuff, 64, "%2u: %8lu %c", i, iList[i].dPtr->rawStats.st_mtime,
                  iList[i].dPtr->fType == fmDIR_TYPE ? 'D' : SPACE ) ;
      nc.WriteString ( dYpos++, dXpos, dBuff, nc.bl ) ;
   }
   #endif   // DEBUGrsdd

   //* Search for files with the same file date. Sub-sort by filename *
   UINT top, bot ;
   for ( top = ZERO, bot = 1 ; bot < fCount ; )
   {
      while (   (bot < fCount) 
             && iList[bot].dPtr->rawStats.st_mtime == iList[top].dPtr->rawStats.st_mtime )
         ++bot ;
      //* If we found at least two files of the same date *
      if ( (bot - top) > 1 ) 
      {
         --bot ;     // range it
         UINT cnt = bot - top + 1 ;
         rsddName_Sort ( &iList[top], cnt, sensi ) ;
         top = bot + 1 ;
      }
      else
         top = bot ;
      bot = top + 1 ;
      if ( top >= fCount )
         break ;
   }

}  //* End rsddDate_Sort() *

//*************************
//*   rsddDateR_Sort      *
//*************************
void FMgr::rsddDateR_Sort ( ReSort iList[], UINT fCount, bool sensi )
{
ReSort   saveLow, saveHigh ;
int      lIndex, gIndex,               // low and high limits search indices
         i ;
#if DEBUGrsdd != 0
int   dYpos = 1,
      dXpos = 82,
      loopCounter = ZERO,
      sMoves = ZERO ;
char  dBuff[64] ;
#endif   // DEBUGrsdd

   for ( lIndex = 0, gIndex = fCount-1 ; (gIndex-lIndex) > ZERO ; lIndex++, gIndex-- )
   {
      #if DEBUGrsdd != 0
      ++loopCounter ;
      #endif   // DEBUGrsdd

      for ( i = lIndex+1 ; i <= gIndex ; i++ )
      {
         if ( iList[i].dPtr->rawStats.st_mtime > iList[lIndex].dPtr->rawStats.st_mtime )
         {
            #if DEBUGrsdd != 0
            ++sMoves ;
            #endif   // DEBUGrsdd
            saveLow = iList[lIndex] ;
            iList[lIndex] = iList[i] ;
            iList[i] = saveLow ;
         }

         if ( i < gIndex )
         {
            if ( iList[gIndex].dPtr->rawStats.st_mtime > iList[i].dPtr->rawStats.st_mtime )
            {
               #if DEBUGrsdd != 0
               ++sMoves ;
               #endif   // DEBUGrsdd
               saveHigh = iList[gIndex] ;
               iList[gIndex] = iList[i] ;
               iList[i] = saveHigh ;

               if ( iList[i].dPtr->rawStats.st_mtime > iList[lIndex].dPtr->rawStats.st_mtime )
               {
                  #if DEBUGrsdd != 0
                  ++sMoves ;
                  #endif   // DEBUGrsdd
                  saveLow = iList[lIndex] ;
                  iList[lIndex] = iList[i] ;
                  iList[i] = saveLow ;
               }
            }
         }
      }
   }
   #if DEBUGrsdd != 0
   snprintf ( dBuff, 64, "primary loops:%3u - moves:%3u", loopCounter, sMoves ) ; 
   nc.WriteString ( ZERO, 80, dBuff, nc.bl ) ;
   for ( UINT i = ZERO ; i < fCount ; i++ )
   {
      snprintf ( dBuff, 64, "%2u: %8lu %c", i, iList[i].dPtr->rawStats.st_mtime,
                  iList[i].dPtr->fType == fmDIR_TYPE ? 'D' : SPACE ) ;
      nc.WriteString ( dYpos++, dXpos, dBuff, nc.bl ) ;
   }
   #endif   // DEBUGrsdd

   //* Search for files with the same file date. Sub-sort by filename *
   UINT top, bot ;
   for ( top = ZERO, bot = 1 ; bot < fCount ; )
   {
      while (   (bot < fCount) 
             && iList[bot].dPtr->rawStats.st_mtime == iList[top].dPtr->rawStats.st_mtime )
         ++bot ;
      //* If we found at least two files of the same date *
      if ( (bot - top) > 1 ) 
      {
         --bot ;     // range it
         UINT cnt = bot - top + 1 ;
         rsddName_Sort ( &iList[top], cnt, sensi ) ;
         top = bot + 1 ;
      }
      else
         top = bot ;
      bot = top + 1 ;
      if ( top >= fCount )
         break ;
   }

}  //* End rsddDateR_Sort() *

//*************************
//*   rsddSize_Sort       *
//*************************
void FMgr::rsddSize_Sort ( ReSort iList[], UINT fCount, bool sensi )
{
ReSort   saveLow, saveHigh ;
int      lIndex, gIndex,               // low and high limits search indices
         i ;
#if DEBUGrsdd != 0
int   dYpos = 1,
      dXpos = 82,
      loopCounter = ZERO,
      sMoves = ZERO ;
char  dBuff[64] ;
#endif   // DEBUGrsdd

   for ( lIndex = 0, gIndex = fCount-1 ; (gIndex-lIndex) > ZERO ; lIndex++, gIndex-- )
   {
      #if DEBUGrsdd != 0
      ++loopCounter ;
      #endif   // DEBUGrsdd

      for ( i = lIndex+1 ; i <= gIndex ; i++ )
      {
         if ( iList[i].dPtr->fBytes < iList[lIndex].dPtr->fBytes )
         {
            #if DEBUGrsdd != 0
            ++sMoves ;
            #endif   // DEBUGrsdd
            saveLow = iList[lIndex] ;
            iList[lIndex] = iList[i] ;
            iList[i] = saveLow ;
         }
            
         if ( i < gIndex )
         {
            if ( iList[gIndex].dPtr->fBytes < iList[i].dPtr->fBytes )
            {
               #if DEBUGrsdd != 0
               ++sMoves ;
               #endif   // DEBUGrsdd
               saveHigh = iList[gIndex] ;
               iList[gIndex] = iList[i] ;
               iList[i] = saveHigh ;
               
               if ( iList[i].dPtr->fBytes < iList[lIndex].dPtr->fBytes )
               {
                  #if DEBUGrsdd != 0
                  ++sMoves ;
                  #endif   // DEBUGrsdd
                  saveLow = iList[lIndex] ;
                  iList[lIndex] = iList[i] ;
                  iList[i] = saveLow ;
               }
            }
         }
      }
   }
   #if DEBUGrsdd != 0
   snprintf ( dBuff, 64, "primary loops:%3u - moves:%3u", loopCounter, sMoves ) ; 
   nc.WriteString ( ZERO, 80, dBuff, nc.bl ) ;
   for ( UINT i = ZERO ; i < fCount ; i++ )
   {
      snprintf ( dBuff, 64, "%2u: %8llu %c", i, iList[i].dPtr->fBytes,
                  iList[i].dPtr->fType == fmDIR_TYPE ? 'D' : SPACE ) ;
      nc.WriteString ( dYpos++, dXpos, dBuff, nc.bl ) ;
   }
   #endif   // DEBUGrsdd

   //* Search for files with the same file size. Sub-sort by filename *
   UINT top, bot ;
   for ( top = ZERO, bot = 1 ; bot < fCount ; )
   {
      while ( (bot < fCount) && iList[bot].dPtr->fBytes == iList[top].dPtr->fBytes )
         ++bot ;
      //* If we found at least two files of the same size *
      if ( (bot - top) > 1 ) 
      {
         --bot ;     // range it
         rsddName_Sort ( &iList[top], (bot - top + 1), sensi ) ;
         top = bot + 1 ;
      }
      else
         top = bot ;
      bot = top + 1 ;
      if ( top >= fCount )
         break ;
   }

}  //* End rsddSize_Sort() *

//*************************
//*   rsddSizeR_Sort      *
//*************************
void FMgr::rsddSizeR_Sort ( ReSort iList[], UINT fCount, bool sensi )
{
ReSort   saveLow, saveHigh ;
int      lIndex, gIndex,               // low and high limits search indices
         i ;
#if DEBUGrsdd != 0
int   dYpos = 1,
      dXpos = 82,
      loopCounter = ZERO,
      sMoves = ZERO ;
char  dBuff[64] ;
#endif   // DEBUGrsdd

   for ( lIndex = 0, gIndex = fCount-1 ; (gIndex-lIndex) > ZERO ; lIndex++, gIndex-- )
   {
      #if DEBUGrsdd != 0
      ++loopCounter ;
      #endif   // DEBUGrsdd

      for ( i = lIndex+1 ; i <= gIndex ; i++ )
      {
         if ( iList[i].dPtr->fBytes > iList[lIndex].dPtr->fBytes )
         {
            #if DEBUGrsdd != 0
            ++sMoves ;
            #endif   // DEBUGrsdd
            saveLow = iList[lIndex] ;
            iList[lIndex] = iList[i] ;
            iList[i] = saveLow ;
         }

         if ( i < gIndex )
         {
            if ( iList[gIndex].dPtr->fBytes > iList[i].dPtr->fBytes )
            {
               #if DEBUGrsdd != 0
               ++sMoves ;
               #endif   // DEBUGrsdd
               saveHigh = iList[gIndex] ;
               iList[gIndex] = iList[i] ;
               iList[i] = saveHigh ;

               if ( iList[i].dPtr->fBytes > iList[lIndex].dPtr->fBytes )
               {
                  #if DEBUGrsdd != 0
                  ++sMoves ;
                  #endif   // DEBUGrsdd
                  saveLow = iList[lIndex] ;
                  iList[lIndex] = iList[i] ;
                  iList[i] = saveLow ;
               }
            }
         }
      }
   }
   #if DEBUGrsdd != 0
   snprintf ( dBuff, 64, "primary loops:%3u - moves:%3u", loopCounter, sMoves ) ; 
   nc.WriteString ( ZERO, 80, dBuff, nc.bl ) ;
   for ( UINT i = ZERO ; i < fCount ; i++ )
   {
      snprintf ( dBuff, 64, "%2u: %8llu %c", i, iList[i].dPtr->fBytes,
                  iList[i].dPtr->fType == fmDIR_TYPE ? 'D' : SPACE ) ;
      nc.WriteString ( dYpos++, dXpos, dBuff, nc.bl ) ;
   }
   #endif   // DEBUGrsdd

   //* Search for files with the same file size. Sub-sort by filename *
   UINT top, bot ;
   for ( top = ZERO, bot = 1 ; bot < fCount ; )
   {
      while ( (bot < fCount) && iList[bot].dPtr->fBytes == iList[top].dPtr->fBytes )
         ++bot ;
      //* If we found at least two files of the same size *
      if ( (bot - top) > 1 ) 
      {
         --bot ;     // range it
         rsddName_Sort ( &iList[top], (bot - top + 1), sensi ) ;
         top = bot + 1 ;
      }
      else
         top = bot ;
      bot = top + 1 ;
      if ( top >= fCount )
         break ;
   }
}  //* End rsddSizeR_Sort() *

//*************************
//*   rsddType_Sort       *
//*************************
void FMgr::rsddType_Sort ( ReSort iList[], UINT fCount, bool sensi )
{
ReSort   saveLow, saveHigh ;
int      lIndex, gIndex,               // low and high limits search indices
         i ;
#if DEBUGrsdd != 0
int   dYpos = 1,
      dXpos = 82,
      loopCounter = ZERO,
      sMoves = ZERO ;
char  dBuff[64] ;
#endif   // DEBUGrsdd

   for ( lIndex = 0, gIndex = fCount-1 ; (gIndex-lIndex) > ZERO ; lIndex++, gIndex-- )
   {
      #if DEBUGrsdd != 0
      ++loopCounter ;
      #endif   // DEBUGrsdd

      for ( i = lIndex+1 ; i <= gIndex ; i++ )
      {
         if ( iList[i].dPtr->fType < iList[lIndex].dPtr->fType )
         {
            #if DEBUGrsdd != 0
            ++sMoves ;
            #endif   // DEBUGrsdd
            saveLow = iList[lIndex] ;
            iList[lIndex] = iList[i] ;
            iList[i] = saveLow ;
         }
            
         if ( i < gIndex )
         {
            if ( iList[gIndex].dPtr->fType < iList[i].dPtr->fType )
            {
               #if DEBUGrsdd != 0
               ++sMoves ;
               #endif   // DEBUGrsdd
               saveHigh = iList[gIndex] ;
               iList[gIndex] = iList[i] ;
               iList[i] = saveHigh ;
               
               if ( iList[i].dPtr->fType < iList[lIndex].dPtr->fType )
               {
                  #if DEBUGrsdd != 0
                  ++sMoves ;
                  #endif   // DEBUGrsdd
                  saveLow = iList[lIndex] ;
                  iList[lIndex] = iList[i] ;
                  iList[i] = saveLow ;
               }
            }
         }
      }
   }

   //* Search for files with the same file type. Sub-sort by filename *
   UINT top, bot ;
   for ( top = ZERO, bot = 1 ; bot < fCount ; )
   {
      while ( (bot < fCount) && iList[bot].dPtr->fType == iList[top].dPtr->fType )
         ++bot ;
      //* If we found at least two files of the same type *
      if ( (bot - top) > 1 ) 
      {
         --bot ;     // range it
         rsddName_Sort ( &iList[top], (bot - top + 1), sensi ) ;
         top = bot + 1 ;
      }
      else
         top = bot ;
      bot = top + 1 ;
   }

   #if DEBUGrsdd != 0
   snprintf ( dBuff, 64, "primary loops:%3u - moves:%3u", loopCounter, sMoves ) ; 
   nc.WriteString ( ZERO, 80, dBuff, nc.bl ) ;
   for ( UINT i = ZERO ; i < fCount ; i++ )
   {
      snprintf ( dBuff, 64, "%2u: %02d %s %c", i, iList[i].dPtr->fType,
                  iList[i].dPtr->fName, 
                  iList[i].dPtr->fType == fmDIR_TYPE ? 'D' : SPACE ) ;
      nc.WriteString ( dYpos++, dXpos, dBuff, nc.bl ) ;
   }
   #endif   // DEBUGrsdd
}  //* End rsddType_Sort() *

//*************************
//*   rsddTypeR_Sort      *
//*************************
void FMgr::rsddTypeR_Sort ( ReSort iList[], UINT fCount, bool sensi )
{
ReSort   saveLow, saveHigh ;
int      lIndex, gIndex,               // low and high limits search indices
         i ;
#if DEBUGrsdd != 0
int   dYpos = 1,
      dXpos = 82,
      loopCounter = ZERO,
      sMoves = ZERO ;
char  dBuff[64] ;
#endif   // DEBUGrsdd

   for ( lIndex = 0, gIndex = fCount-1 ; (gIndex-lIndex) > ZERO ; lIndex++, gIndex-- )
   {
      #if DEBUGrsdd != 0
      ++loopCounter ;
      #endif   // DEBUGrsdd

      for ( i = lIndex+1 ; i <= gIndex ; i++ )
      {
         if ( iList[i].dPtr->fType > iList[lIndex].dPtr->fType )
         {
            #if DEBUGrsdd != 0
            ++sMoves ;
            #endif   // DEBUGrsdd
            saveLow = iList[lIndex] ;
            iList[lIndex] = iList[i] ;
            iList[i] = saveLow ;
         }
            
         if ( i < gIndex )
         {
            if ( iList[gIndex].dPtr->fType > iList[i].dPtr->fType )
            {
               #if DEBUGrsdd != 0
               ++sMoves ;
               #endif   // DEBUGrsdd
               saveHigh = iList[gIndex] ;
               iList[gIndex] = iList[i] ;
               iList[i] = saveHigh ;
               
               if ( iList[i].dPtr->fType > iList[lIndex].dPtr->fType )
               {
                  #if DEBUGrsdd != 0
                  ++sMoves ;
                  #endif   // DEBUGrsdd
                  saveLow = iList[lIndex] ;
                  iList[lIndex] = iList[i] ;
                  iList[i] = saveLow ;
               }
            }
         }
      }
   }

   //* Search for files with the same file type. Sub-sort by filename *
   UINT top, bot ;
   for ( top = ZERO, bot = 1 ; bot < fCount ; )
   {
      while ( (bot < fCount) && iList[bot].dPtr->fType == iList[top].dPtr->fType )
         ++bot ;
      //* If we found at least two files of the same type *
      if ( (bot - top) > 1 ) 
      {
         --bot ;     // range it
         rsddName_Sort ( &iList[top], (bot - top + 1), sensi ) ;
         top = bot + 1 ;
      }
      else
         top = bot ;
      bot = top + 1 ;
   }

   #if DEBUGrsdd != 0
   snprintf ( dBuff, 64, "primary loops:%3u - moves:%3u", loopCounter, sMoves ) ; 
   nc.WriteString ( ZERO, 80, dBuff, nc.bl ) ;
   for ( UINT i = ZERO ; i < fCount ; i++ )
   {
      snprintf ( dBuff, 64, "%2u: %02d %s %c", i, iList[i].dPtr->fType,
                  iList[i].dPtr->fName, 
                  iList[i].dPtr->fType == fmDIR_TYPE ? 'D' : SPACE ) ;
      nc.WriteString ( dYpos++, dXpos, dBuff, nc.bl ) ;
   }
   #endif   // DEBUGrsdd
}  //* End rsddTypeR_Sort() *

//*************************
//*    rsddExt_Sort       *
//*************************
void FMgr::rsddExt_Sort ( ReSort iList[], UINT fCount, bool sensi )
{
ReSort   rsSwap ;
gString  gsi, gsl, gsg, gss ;
int      lIndex, gIndex,               // low and high limits search indices
         result, i ;
bool     gsiExt, gslExt, gsgExt ;
#if DEBUGrsdd != 0
int   dYpos = 1,
      dXpos = 82,
      loopCounter = ZERO,
      sMoves = ZERO ;
gString dBuff ;
#endif   // DEBUGrsdd

   for ( lIndex = 0, gIndex = fCount-1 ; (gIndex-lIndex) > ZERO ; lIndex++, gIndex-- )
   {
      #if DEBUGrsdd != 0
      ++loopCounter ;
      #endif   // DEBUGrsdd

      for ( i = lIndex+1 ; i <= gIndex ; i++ )
      {
         //* Isolate filename extensions for current index and low limit *
         gsi = iList[i].dPtr->fName ;
         gsiExt = IsolateFileExtension ( gsi ) ;
         gsl = iList[lIndex].dPtr->fName ;
         gslExt = IsolateFileExtension ( gsl ) ;

         //* If both names have extensions, compare them *
         if ( gsiExt && gslExt )
         {
            if ( sensi )
               result = rsddCompare ( gsi, gsl ) ;
            else
               result = rsddCompareI ( gsi, gsl ) ;
         }
         //* Else if neither name has an extension, they are equal *
         else if ( !gsiExt && !gslExt )
            result = ZERO ;
         //* Else only one name has an extension, so                *
         //* 'no extension' is numerically less than any extension. *
         else if ( !gsiExt )
            result = -1 ;
         else
            result = 1 ;

         //* gsi extension < gsl extension *
         if ( result < ZERO )
         {
            #if DEBUGrsdd != 0
            ++sMoves ;
            #endif   // DEBUGrsdd
            //* Do the swap *
            rsSwap = iList[lIndex] ;
            iList[lIndex] = iList[i] ;
            iList[i] = rsSwap ;
            if ( gsiExt && gslExt )
            { gss = gsl ; gsl = gsi ; gsi = gss ; }
            else if ( gsiExt )
            { gsl = gsi ; gslExt = true ; gsiExt = false ; }
            else if ( gslExt )
            { gsi = gsl ; gsiExt = true ; gslExt = false ; }
         }

         //* If current index has not reached the upper limit *
         if ( i < gIndex )
         {
            //* Compare filename extensions for current index and high limit *
            gsg = iList[gIndex].dPtr->fName ;
            gsgExt = IsolateFileExtension ( gsg ) ;
            if ( gsgExt && gsiExt )
            {
               if ( sensi )
                  result = rsddCompare ( gsg, gsi ) ;
               else
                  result = rsddCompareI ( gsg, gsi ) ;
            }
            else if ( !gsgExt && !gsiExt )
               result = ZERO ;
            else if ( !gsgExt )
               result = -1 ;
            else
               result = 1 ;

            //* If gsg extension < gsi extension *
            if ( result < ZERO )
            {
               #if DEBUGrsdd != 0
               ++sMoves ;
               #endif   // DEBUGrsdd
               rsSwap = iList[gIndex] ;
               iList[gIndex] = iList[i] ;
               iList[i] = rsSwap ;
               if ( gsgExt && gsiExt )
               { gss = gsg ; gsg = gsi ; gsi = gss ; }
               else if ( gsgExt )
               { gsi = gsg ; gsiExt = true ; gsgExt = false ; }
               else if ( gsiExt )
               { gsg = gsi ; gsgExt = true ; gsiExt = false ; }
                  
               //* Compare extensions for current index and low limit *
               if ( gsiExt && gslExt )
               {
                  if ( sensi )
                     result = rsddCompare ( gsi, gsl ) ;
                  else
                     result = rsddCompareI ( gsi, gsl ) ;
               }
               else if ( !gsiExt && !gslExt )
                  result = ZERO ;
               else if ( !gsiExt )
                  result = -1 ;
               else
                  result = 1 ;
               if ( result < ZERO )
               {
                  #if DEBUGrsdd != 0
                  ++sMoves ;
                  #endif   // DEBUGrsdd
                  rsSwap = iList[lIndex] ;
                  iList[lIndex] = iList[i] ;
                  iList[i] = rsSwap ;
               }
            }     // if(result<ZERO)
         }        // if(i<gIndex)
      }           // inner for(;;)
   }              // outer for(;;)

   #if DEBUGrsdd != 0
   dBuff.compose( L"primary loops:%3d - moves:%3d", &loopCounter, &sMoves ) ;
   nc.WriteString ( ZERO, 80, dBuff.gstr(), nc.bl ) ;
   char type ;
   for ( UINT i = ZERO ; i < fCount ; i++ )
   {
      type = iList[i].dPtr->fType == fmDIR_TYPE ? 'D' : SPACE ;
      gsi = iList[i].dPtr->fName ;
      if ( (IsolateFileExtension ( gsi )) )
         dBuff.compose( L"%2u: %-8S %c", &i, gsi.gstr(), &type ) ;
      else
         dBuff.compose( L"%2u:          %c", &i, &type ) ;
      nc.WriteString ( dYpos++, dXpos, dBuff.gstr(), nc.bl ) ;
      if ( i == 38 )
      {
         nckPause();
         dYpos = 1 ;
      }
   }
   #endif   // DEBUGrsdd

   //* Search for files with the same file extension. Sub-sort by filename *
   if ( fCount > 1 )
   {
      UINT top, bot ;
      int  difExt ;
      for ( bot = ZERO, top = ZERO ; bot < fCount ; )
      {
         //* First entry in POSSIBLE group of names with the same extension *
         gsl = iList[bot].dPtr->fName ;
         gslExt = IsolateFileExtension ( gsl ) ;
         do
         {
            ++top ;
            if ( top < fCount )
            {
               gsi = iList[top].dPtr->fName ;
               gsiExt = IsolateFileExtension ( gsi ) ;
               if ( gslExt && gsiExt )
               {
                  if ( sensi )
                     difExt = rsddCompare ( gsl, gsi ) ;
                  else
                     difExt = rsddCompareI ( gsl, gsi ) ;
               }
               //* Else if neither name has an extension, they are equal *
               else if ( !gslExt && !gsiExt )
                  difExt = ZERO ;
               //* Else only one name has an extension, so                *
               //* 'no extension' is numerically less than any extension. *
               else if ( !gslExt )
                  difExt = -1 ;
               else
                  difExt = 1 ;
            }
            else
               break ;
         }
         while ( ! difExt ) ;

         //* If we found at least two files with the same extension *
         --top ;
         if ( (top - bot) > ZERO ) 
         {
            rsddName_Sort ( &iList[bot], (top - bot + 1), sensi ) ;
         }
         bot = top + 1 ;
         top = bot ;
      }
   }

}  //* End rsddExt_Sort() *

//*************************
//*   rsddExtR_Sort       *
//*************************
void FMgr::rsddExtR_Sort ( ReSort iList[], UINT fCount, bool sensi )
{
ReSort   rsSwap ;
gString  gsi, gsl, gsg, gss ;
int      lIndex, gIndex,               // low and high limits search indices
         result, i ;
bool     gsiExt, gslExt, gsgExt ;
#if DEBUGrsdd != 0
int   dYpos = 1,
      dXpos = 82,
      loopCounter = ZERO,
      sMoves = ZERO ;
gString dBuff ;
#endif   // DEBUGrsdd

   for ( lIndex = 0, gIndex = fCount-1 ; (gIndex-lIndex) > ZERO ; lIndex++, gIndex-- )
   {
      #if DEBUGrsdd != 0
      ++loopCounter ;
      #endif   // DEBUGrsdd

      for ( i = lIndex+1 ; i <= gIndex ; i++ )
      {
         //* Isolate filename extensions for current index and low limit *
         gsi = iList[i].dPtr->fName ;
         gsiExt = IsolateFileExtension ( gsi ) ;
         gsl = iList[lIndex].dPtr->fName ;
         gslExt = IsolateFileExtension ( gsl ) ;

         //* If both names have extensions, compare them *
         if ( gsiExt && gslExt )
         {
            if ( sensi )
               result = rsddCompare ( gsi, gsl ) ;
            else
               result = rsddCompareI ( gsi, gsl ) ;
         }
         //* Else if neither name has an extension, they are equal *
         else if ( !gsiExt && !gslExt )
            result = ZERO ;
         //* Else only one name has an extension, so                *
         //* 'no extension' is numerically less than any extension. *
         else if ( !gsiExt )
            result = -1 ;
         else
            result = 1 ;

         //* gsi extension > gsl extension *
         if ( result > ZERO )
         {
            #if DEBUGrsdd != 0
            ++sMoves ;
            #endif   // DEBUGrsdd
            //* Do the swap *
            rsSwap = iList[lIndex] ;
            iList[lIndex] = iList[i] ;
            iList[i] = rsSwap ;
            if ( gsiExt && gslExt )
            { gss = gsl ; gsl = gsi ; gsi = gss ; }
            else if ( gsiExt )
            { gsl = gsi ; gslExt = true ; gsiExt = false ; }
            else if ( gslExt )
            { gsi = gsl ; gsiExt = true ; gslExt = false ; }
         }

         //* If current index has not reached the upper limit *
         if ( i < gIndex )
         {
            //* Compare filename extensions for current index and high limit *
            gsg = iList[gIndex].dPtr->fName ;
            gsgExt = IsolateFileExtension ( gsg ) ;
            if ( gsgExt && gsiExt )
            {
               if ( sensi )
                  result = rsddCompare ( gsg, gsi ) ;
               else
                  result = rsddCompareI ( gsg, gsi ) ;
            }
            else if ( !gsgExt && !gsiExt )
               result = ZERO ;
            else if ( !gsgExt )
               result = -1 ;
            else
               result = 1 ;

            //* If gsg extension > gsi extension *
            if ( result > ZERO )
            {
               #if DEBUGrsdd != 0
               ++sMoves ;
               #endif   // DEBUGrsdd
               rsSwap = iList[gIndex] ;
               iList[gIndex] = iList[i] ;
               iList[i] = rsSwap ;
               if ( gsgExt && gsiExt )
               { gss = gsg ; gsg = gsi ; gsi = gss ; }
               else if ( gsgExt )
               { gsi = gsg ; gsiExt = true ; gsgExt = false ; }
               else if ( gsiExt )
               { gsg = gsi ; gsgExt = true ; gsiExt = false ; }

               //* Compare extensions for current index and low limit *
               if ( gsiExt && gslExt )
               {
                  if ( sensi )
                     result = rsddCompare ( gsi, gsl ) ;
                  else
                     result = rsddCompareI ( gsi, gsl ) ;
               }
               else if ( !gsiExt && !gslExt )
                  result = ZERO ;
               else if ( !gsiExt )
                  result = -1 ;
               else
                  result = 1 ;
               if ( result > ZERO )
               {
                  #if DEBUGrsdd != 0
                  ++sMoves ;
                  #endif   // DEBUGrsdd
                  rsSwap = iList[lIndex] ;
                  iList[lIndex] = iList[i] ;
                  iList[i] = rsSwap ;
               }
            }     // if(result>ZERO)
         }        // if(i<gIndex)
      }           // inner for(;;)
   }              // outer for(;;)

   #if DEBUGrsdd != 0
   dBuff.compose( L"primary loops:%3d - moves:%3d", &loopCounter, &sMoves ) ;
   nc.WriteString ( ZERO, 80, dBuff.gstr(), nc.bl ) ;
   char type ;
   for ( UINT i = ZERO ; i < fCount ; i++ )
   {
      type = iList[i].dPtr->fType == fmDIR_TYPE ? 'D' : SPACE ;
      gsi = iList[i].dPtr->fName ;
      if ( (IsolateFileExtension ( gsi )) )
         dBuff.compose( L"%2u: %-8S %c", &i, gsi.gstr(), &type ) ;
      else
         dBuff.compose( L"%2u:          %c", &i, &type ) ;
      nc.WriteString ( dYpos++, dXpos, dBuff.gstr(), nc.bl ) ;
      if ( i == 38 )
      {
         nckPause();
         dYpos = 1 ;
      }
   }
   #endif   // DEBUGrsdd

   //* Search for files with the same file extension. Sub-sort by filename *
   if ( fCount > 1 )
   {
      UINT top, bot ;
      int  difExt ;
      for ( bot = ZERO, top = ZERO ; bot < fCount ; )
      {
         //* First entry in POSSIBLE group of names with the same extension *
         gsl = iList[bot].dPtr->fName ;
         gslExt = IsolateFileExtension ( gsl ) ;
         do
         {
            ++top ;
            if ( top < fCount )
            {
               gsi = iList[top].dPtr->fName ;
               gsiExt = IsolateFileExtension ( gsi ) ;
               if ( gslExt && gsiExt )
               {
                  if ( sensi )
                     difExt = rsddCompare ( gsl, gsi ) ;
                  else
                     difExt = rsddCompareI ( gsl, gsi ) ;
               }
               //* Else if neither name has an extension, they are equal *
               else if ( !gslExt && !gsiExt )
                  difExt = ZERO ;
               //* Else only one name has an extension, so                *
               //* 'no extension' is numerically less than any extension. *
               else if ( !gslExt )
                  difExt = -1 ;
               else
                  difExt = 1 ;
            }
            else
               break ;
         }
         while ( ! difExt ) ;

         //* If we found at least two files with the same extension *
         --top ;
         if ( (top - bot) > ZERO ) 
         {
            rsddName_Sort ( &iList[bot], (top - bot + 1), sensi ) ;
         }
         bot = top + 1 ;
         top = bot ;
      }
   }

}  //* End rsddExtR_Sort() *

//************************
//* IsolateFileExtension *
//************************
//******************************************************************************
//* Determine whether a filename string includes a filename extension.         *
//*                                                                            *
//* A filename extension is defined as whatever characters follow the          *
//* rightmost PERIOD ('.') of the filename.                                    *
//*                                                                            *
//* Input  : fname: (by reference) gString object containing the filename      *
//*                                                                            *
//* Returns: 'true' if extension located                                       *
//*            fname contains the extension string                             *
//*          'false' if no filename extension                                  *
//*            fname contents unchanged                                        *
//******************************************************************************

bool FMgr::IsolateFileExtension ( gString& fname )
{
const wchar_t *nPtr = fname.gstr() ; 
wchar_t  tmp[MAX_FNAME] ;
short    nLen = fname.gschars() ;
bool     ext = false ;

   for ( short i = (nLen-1) ; i > ZERO ; i-- )
   {
      if ( nPtr[i] == PERIOD )
      {
         //* If not a 'hidden' file with no extension AND   *
         //* period is not the last character in the string,*
         //* then filename does have an extension.          *
         if ( i > ZERO && i < (nLen-2) )
         {  //* Return the filename extension string to caller *
            fname.copy( tmp, MAX_FNAME ) ;
            ++i ;
            fname = &tmp[i] ;
            ext = true ;
         }
         break ;
      }
   }
   return ext ;

}  //* End IsolateFileExtension() *

//*************************
//*    rsddUn_Sort        *
//*************************
void FMgr::rsddUn_Sort ( ReSort iList[], UINT fCount )
{
ReSort   saveLow, saveHigh ;
int      lIndex, gIndex,               // low and high limits search indices
         i ;
#if DEBUGrsdd != 0
int   dYpos = 1,
      dXpos = 82,
      loopCounter = ZERO,
      sMoves = ZERO ;
char  dBuff[64] ;
#endif   // DEBUGrsdd

   for ( lIndex = 0, gIndex = fCount-1 ; (gIndex-lIndex) > ZERO ; lIndex++, gIndex-- )
   {
      #if DEBUGrsdd != 0
      ++loopCounter ;
      #endif   // DEBUGrsdd

      for ( i = lIndex+1 ; i <= gIndex ; i++ )
      {
         if ( iList[i].dPtr < iList[lIndex].dPtr )
         {
            #if DEBUGrsdd != 0
            ++sMoves ;
            #endif   // DEBUGrsdd
            saveLow = iList[lIndex] ;
            iList[lIndex] = iList[i] ;
            iList[i] = saveLow ;
         }
            
         if ( i < gIndex )
         {
            if ( iList[gIndex].dPtr < iList[i].dPtr )
            {
               #if DEBUGrsdd != 0
               ++sMoves ;
               #endif   // DEBUGrsdd
               saveHigh = iList[gIndex] ;
               iList[gIndex] = iList[i] ;
               iList[i] = saveHigh ;
               
               if ( iList[i].dPtr < iList[lIndex].dPtr )
               {
                  #if DEBUGrsdd != 0
                  ++sMoves ;
                  #endif   // DEBUGrsdd
                  saveLow = iList[lIndex] ;
                  iList[lIndex] = iList[i] ;
                  iList[i] = saveLow ;
               }
            }
         }
      }
   }
   #if DEBUGrsdd != 0
   snprintf ( dBuff, 64, "primary loops:%3u - moves:%3u", loopCounter, sMoves ) ; 
   nc.WriteString ( ZERO, 80, dBuff, nc.bl ) ;
   for ( UINT i = ZERO ; i < fCount ; i++ )
   {
      snprintf ( dBuff, 64, "%2u: %8lu %c", i, (ULONG)iList[i].dPtr,
                  iList[i].dPtr->fType == fmDIR_TYPE ? 'D' : SPACE ) ;
      nc.WriteString ( dYpos++, dXpos, dBuff, nc.bl ) ;
   }
   #endif   // DEBUGrsdd
}  //* End rsddUn_Sort() *

//*************************
//*     rsddCompare       *
//*************************
//******************************************************************************
//* NON-MEMBER METHOD:                                                         *
//* Compare two strings (case sensitive).                                      *
//*                                                                            *
//* Input Values : sa: string A                                                *
//*                sb: string B                                                *
//*                                                                            *
//* Return Value : > ZERO if sa > sb                                           *
//*                  ZERO if sa == sb                                          *
//*                < ZERO if sa < sb                                           *
//******************************************************************************

static int rsddCompare ( const gString& sa, const gString& sb )
{
#if SORT_TYPE == 0 || SORT_TYPE == 3
   return ( wcsncmp ( sa.gstr(), sb.gstr(), MAX_FNAME ) ) ;
#elif SORT_TYPE == 1
   return ( wcscoll ( sa.gstr(), sb.gstr() ) ) ;
#else    // SORT_TYPE == 2
   return ( strncmp ( sa.ustr(), sb.ustr(), MAX_FNAME ) ) ;
#endif   // SORT_TYPE

}  //* End rsddCompare() *

//*************************
//*     rsddCompareI      *
//*************************
//******************************************************************************
//* NON-MEMBER METHOD:                                                         *
//* Compare two strings (case insensitive).                                    *
//*                                                                            *
//* Input Values : sa: string A                                                *
//*                sb: string B                                                *
//*                                                                            *
//* Return Value : > ZERO if sa > sb                                           *
//*                  ZERO if sa == sb                                          *
//*                < ZERO if sa < sb                                           *
//******************************************************************************

static int rsddCompareI ( const gString& sa, const gString& sb )
{
#if SORT_TYPE == 0 || SORT_TYPE == 3
   return ( wcsncasecmp ( sa.gstr(), sb.gstr(), MAX_FNAME ) ) ;
#elif SORT_TYPE == 1
   return ( wcscoll ( sa.gstr(), sb.gstr() ) ) ;
#else    // SORT_TYPE == 2
   return ( strncasecmp ( sa.ustr(), sb.ustr(), MAX_FNAME ) ) ;
#endif   // SORT_TYPE

}  //* End rsddCompareI() *

#if DEBUGrsdd != 0 || rsddRANDOMIZE != 0
//*************************
//*   rsddRandomize       *
//*************************
//******************************************************************************
//* NON-MEMBER METHOD:                                                         *
//* FOR DEBUG ONLY!                                                            *
//* Randomize the order of data in the list before performing the sort.        *
//* This acts as a stress test for the sorting routines.                       *
//*                                                                            *
//* Input Values : iList     : pointer to a list of ReSort class objects       *
//*                fCount    : number of items in iList                        *
//*                                                                            *
//* Return Value : none                                                        *
//******************************************************************************

void FMgr::rsddRandomize ( ReSort iList[], UINT fCount )
{
      
   if ( fCount >= 3 )
   {
      ReSort   saveIt ;
      int      src = ZERO, trg = fCount - 2 ;
      while ( src < trg )
      {
         //* Do the swap *
         saveIt = iList[trg] ;
         iList[trg] = iList[src] ;
         iList[src] = saveIt ;
         ++src ;
         trg -= 2 ;
      }
      src = fCount - 1 ;
      trg = 1 ;
      while ( src > trg )
      {
         //* Do the swap *
         saveIt = iList[trg] ;
         iList[trg] = iList[src] ;
         iList[src] = saveIt ;
         src -= 2 ;
         ++trg ;
      }
   }
   else     // 3 or fewer elements in list
   {
   }

}  //* End rsddRandomize() *
#endif   // DEBUGrsdd || rsddRANDOMIZE
#undef SORT_TYPE
#undef DEBUGrsdd
#undef rsddRANDOMIZE

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************


