//********************************************************************************
//* File       : FileDlg.cpp                                                     *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2005-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in FileMangler.hpp         *
//* Date       : 18-Jul-2025                                                     *
//* Version    : (see FileDlgVersion string, below)                              *
//*                                                                              *
//* Description: This class implements a file-browsing window. This class        *
//* encapsulates the file manipulation and display functionality so that even    *
//* though FileDlg works closely with the mainline application code, the         *
//* main application code simply enjoys the benefits of our labor, without the   *
//* need to actually know very much about the file system.                       *
//*                                                                              *
//* The FileDlg class is built on the functionality of the NcDialog class.       *
//* An NcDialog object with the appropriate control sub-objects has been         *
//* instantiated at a higher (application) level before the FileDlg constructor  *
//* is called.                                                                   *
//*                                                                              *
//* Four controls are required by the FileDlg class:                             *
//*                                                                              *
//*   1) The File data are displayed through a dctSCROLLEXT control.             *
//*   2) The Path of the current working directory (CWD) is displayed in a       *
//*      dctTEXTBOX control.                                                     *
//*   3) The File Statistics are displayed in another dctTEXTBOX control.        *
//*   4) The Messages to the user are displayed in another dctTEXTBOX control    *
//*                                                                              *
//* The constructor receives a pointer to the caller's NcDialog class, as well   *
//* as data describing the configuration of the controls that are needed.        *
//*                                                                              *
//* Some of the NCurses class methods and data members are also used -           *
//* primarily color attribute variables. These are accessed through the global   *
//* class handle 'nc', which resides within the NCurses class code and is        *
//* declared extern in GlobalDef.hpp for use by all application code.            *
//*                                                                              *
//* Note that the FileDlg class instantiates the FMgr class and controls all     *
//* access to its methods and data members. The application has no direct        *
//* access to the methods and data of the FMgr class.                            *
//*                                                                              *
//*                                                                              *
//* Development tools: See FileMangler.cpp                                       *
//********************************************************************************
//* Version History (most recent first):                                         *
//*                                                                              *
//* v: 0.00.55 02-Apr-2025                                                       *
//*   -- Update references to character/byte counts based upon updates to the    *
//*      gString-class definitions.                                              *
//*   -- Update the sort algorithm for the find-files dialog. When display of    *
//*      timestamp is active, the list is sorted by timestamp. See ffSortRecords.*
//*   -- For display of '.eml' files, speed up the decoding.                     *
//*   -- Increase threshold value for enabling the operation progress monitor    *
//*      thread. In 2005, we set this to 5MB because the hardware of the time    *
//*      started to seem unbearably slow at that size. Hardware is somewhat      *
//*      faster in 2025, so the threshold has been increased to 20MB.            *
//*   -- Update the Tree-view Mode dialog and enhance scan of the directory tree *
//*      for both speed and more robust operation.                               *
//*      -- The "/proc" directory is now handled as a special case. This is      *
//*         because the files and directories it contains change rapidly and     *
//*         previously would cause an occasional system exception.               *
//*      -- A recursive root scan would often hang or throw an exception, or     *
//*         just take half a millennium to complete. For application stability,  *
//*         recursive scans from the root directory have been disabled.          *
//*                                                                              *
//* v: 0.00.54 26-Nov-2023                                                       *
//*   -- Bug Fix: In CreateDirectory_gvfs(), target filename was not enclosed    *
//*      in quotes, so spaces in filename were not recognized.                   *
//*                                                                              *
//* v: 0.00.53 22-Oct-2021                                                       *
//*   -- Add functionality to the BackupClipboardList() method group to list     *
//*      the files needing backup in the logfile, but do not update any files.   *
//*      Note that while the log file for Backup/Synch operations reports the    *
//*      _source_ files backed up, the Scan-Only operation reports the targets   *
//*      that would have been updated.                                           *
//*   -- Add ".webp" file format to the list of files known to require GUI       *
//*      viewing applications. (see FileDlg;;oepGooeyType().                     *
//*      Note that some Linux distros do not include a default viewer for        *
//*      ".webp" file, OR that they default to the web browser. To ensure that   *
//*      the Gnome Image Viewer supports this format, install the extension:     *
//*              sudo dnf install webp-pixbuf-loader                             *
//*   -- Bug Fix: In ExtractMetadata_JPG(), test for empty string was incorrect  *
//*      but was masked and never executed.                                      *
//*                                                                              *
//* v: 0.00.52 27-Nov-2020                                                       *
//*   -- Add new pass-through method: DVD_DeviceStats() to gather info on        *
//*      optical drives.                                                         *
//*   -- Enhance "File System Information" dialog to support information         *
//*      specific to USB devices and optical-media.                              *
//*                                                                              *
//* v: 0.00.51 01-Aug-2020                                                       *
//*   -- When viewing contents of files, include "backup" copies of HTML files   *
//*      to be passed through the ANSI color filter. Example: Ratburger.html~    *
//*   -- Bug Fix: When viewing the contents of HTML markup, the ANSI color       *
//*      filter was not always handling HTML comments correctly. Valid HTML      *
//*      markup within the comment was being processed as command tokens rather  *
//*      than as comment data. Example: <!-- <h2>Howdy!</h2> -->"                *
//*   -- Add Windows(tm) bitmap files (".bmp") to the list of files handled      *
//*      automatically by the 'open file' context menu.                          *
//*   -- Add support for ".eml" email files in the ViewFileContents() method.    *
//*      This is not a sophisticated algorithm. It handles modern                *
//*      MIME-compliant formats, but support for older email formats is          *
//*      simplistic at best.                                                     *
//*   -- Add support for reading and display of ".wma" audio and ".wmv" video    *
//*      metadata.                                                               *
//*      -- WMA/WMV streams are encapsulated within an "Advanced System          *
//*         Format" (ASF) container object.                                      *
//*      -- Please note that WMA/WMV and ASF are proprietary Windows(tm)         *
//*         standards (which are publicly documented).                           *
//*         The current implementation violates neither their f-ing patents      *
//*         nor their f-ing copyright.                                           *
//*   -- Bug Fix: When converting UTF-16 to UTF-8, if source value was two(2)    *
//*      16-bit words, the upper-range-check value was incorrect. This error     *
//*      (potentially) affected decoding of metadata from MP3 audio files.       *
//*      See id3v2_framehdr class, utf16Conv() method.                           *
//*   -- In the View-File-Contents dialog, enable 'strict' test for Perl source  *
//*      files. In earlier releases, we identified Perl source only by filename  *
//*      extension; however, Perl mavens often omit the filename extension, so   *
//*      we test for the shebang and Perl invocation. Ex: "#!/usr/bin/perl"      *
//*   -- In View File Contents dialog, when user attempts to view the contents   *
//*      of a binary file as if it were plain text, automatically set view to    *
//*      'ASCII-hex-byte' format. See the vfcIsTextfile() method for details.    *
//*   -- In ViewFileContents(), add support for viewing Microsoft Word(tm)       *
//*      1997, 2000, 2002, 2003, 2007 documents. These are the obsolete Word     *
//*      binary file formats used before OpenXML(tm) was implemented.            *
//*      Decoding the text of these documents is experimental, and shortcuts     *
//*      were taken rather than fully implementing the binary decoding           *
//*      algorithm; however, this simple implementation works well for all       *
//*      sample documents submitted for testing. Full decoding of the binary     *
//*      structures may be implemented in a future release.                      *
//*   -- In GrepFiles() add support for scan of Microsoft Word(tm) 1997, 2000,   *
//*      2002, 2003, 2007 documents.                                             *
//*                                                                              *
//* v: 0.00.50 10-Jul-2020                                                       *
//*   -- Add 'clipBoard_gvfsSrc' and 'clipBoard_gvfsTrg' flags to the clipboard  *
//*      group of variables to facilitate handling of copy/cut/paste/rename      *
//*      operations when source or target data reside on an MTP/GVfs virtual     *
//*      filesystem. These flags are used to determine which low-level           *
//*      (FMgr-class) method to call for the operation.                          *
//*      -- These flags are initialized by ClearClipboard() and FillClipboard()  *
//*      -- clipBoard_gvfsSrc is set by FillClipboard() and MarkSelectionNR().   *
//*      -- clipBoard_gvfsTrg is set by PasteClipboardList() and                 *
//*         BackupClipboardList().                                               *
//*   -- Add strategic tests to determine whether to call kernel methods or      *
//*      gvfs methods.                                                           *
//*                                                                              *
//* v: 0.00.49 07-Dec-2019                                                       *
//*   -- Bug Fix: In WorkInProgress class, protect against accessing ProgBar     *
//*      if pBar has not been instantiated. Was causing occasional access        *
//*      violation after display refresh in cowPrompt().                         *
//*   -- Enhance the GrepFiles() dialog:                                         *
//*      -- Optionally report the summary count of matching items rather than    *
//*         the detailed report.                                                 *
//*      -- Implement a secondary search filter which can either exclude items   *
//*         which match the secondary criterion, OR report only items which      *
//*         match both the primary and secondary criteria.                       *
//*      -- Automatically escape spaces, single/double quotations and other      *
//*         regexp "special" characters when used as ordinary filename           *
//*         characters.                                                          *
//*     -- Change definition of manually-entered filename separator character    *
//*        from space (' ') to comma (',').                                      *
//*     -- In GrepFiles() and gfScan() methods, adjust algorithm for escaping    *
//*        Linux "special characters" in filenames. This avoids potential        *
//*        problems in extracting XML text from OpenDocument/OpenXML files.      *
//*   -- Implement support for reading MPEG-4 (m4a) audio file metadata.         *
//*      As is usual with the authors of tag editors, they have failed to even   *
//*      read the ISO/IEC 14496-12 specification for MPEG-4 files. This has      *
//*      made our implementation more difficult and not as clean as one would    *
//*      wish. (Some people should not be allowed anywhere near a computer.)     *
//*      Still, it is functional, if not elegant.                                *
//*   -- Implement ANSI color coding when viewing Perl source code files.        *
//*      Color scheme is similar to that in the jEdit text editor. The color     *
//*      coding makes the code much easier to read.                              *
//*   -- Documentation update.                                                   *
//*                                                                              *
//* v: 0.00.48 15-Nov-2018                                                       *
//*   -- Restructure scan-thread termination loop to accomodate changes in the   *
//*      FMgr-class scan of the directory-tree. This includes a report of        *
//*      "caught" system exceptions. See rcdScan() method.                       *
//*   -- Reformat display for debugging method Display_tnFName().                *
//*   -- Reformat display for debugging method Display_TreeNode().               *
//*   -- Add a test for backup-target filesystems to determine whether the       *
//*      filesystem supports storage of LinuxUNIX symlink files.                 *
//*      See BackupClipboardList() method.                                       *
//*   -- Add support for MS-Office (OOXML) documents which are structured as     *
//*      .ZIP archive files.                                                     *
//*      -- view document text                                                   *
//*      -- view ZIP archive                                                     *
//*      -- expand ZIP archive                                                   *
//*      -- 'grep' the document's text data                                      *
//*   -- For setup of an "Archive" operation enhance and simplify testing for    *
//*      automatic determination of the Create/Update/Expand sub-option.         *
//*      -- Included in this enhancement is support for expanding                *
//*         Office Open XML (MS-Office) document files.                          *
//*      -- Redesign the Archive_UpdateQuery() dialog.                           *
//*      -- Implement the ScanClipboard() method to ensure that user does not    *
//*         try to add an archive to itself.                                     *
//*      See Archive() method for more information.                              *
//*   -- Split FileDlgUtils.cpp into FileDlgUtil1.cpp and FileDlgUtil2.cpp.      *
//*      This module was growing beyond a manageable size.                       *
//*   -- Completed implementation of user interruption of data backup            *
//*      operation. User can now pause, and optionally abort a backup operation  *
//*      in progress. This includes adding the pbarRefresh() method to the       *
//*      WorkInProgress class.                                                   *
//*   -- Implement ANSI color formatting for HTML source code. Tested with       *
//*      pure HTML-5 source only. NOTE: Older, ad-hoc HTML command syntax may    *
//*      not be colorized accurately.                                            *
//*   -- Implement testing for recursion into different filesystems while        *
//*      scanning the directory tree. See FMgr class for additional information  *
//*   -- Simplify Tree-View mode. Use the directory-tree data from the normal    *
//*      file-view mode to create the Tree-View display data. Previously, a      *
//*      specialized tree-view scan was performed.                               *
//*      -- Eliminated two data members: 'tRoot' and 'tRootPath'.                *
//*      -- Add context help (calls the 'info' reader).                          *
//*      -- Overall, smaller, faster, and more robust.                           *
//*   -- Consolidate allocation and release of TreeNode-class and tnFName-class  *
//*      objects to better track potential memory leaks.                         *
//*      See FMgr-class methods Allocate_TreeNodes(), Allocate_tnFNames().       *
//*   -- Make several instantiations of objects using the "new" keyword          *
//*      (class Ptr = new class[n]) into "std::nothrow" instantiantions with     *
//*      automatic retry in the case of instantiation failure.                   *
//*      This is part of the effort to minimize the chance of the application    *
//*      crashing due to a system exception.                                     *
//*   -- Bug Fix: In Paste-Special dialog, if source and target were different   *
//*      filesystems, 'hard link' Radiobutton was not being disabled as it       *
//*      should have been. (Hard links cannot be created across filesystems.)    *
//*   -- Update the single-file and batch-mode rename methods to allow           *
//*      filenames with Linux "special characters": ( \ ' " ? : ; & > < | * ).   *
//*      -- Test all calls to the shell to verify that these characters are      *
//*         seen as text rather than as commands to the shell.                   *
//*      -- Add a warning dialog, gnfSpecialCharsQuery(), asking user to         *
//*         verify use of these special characters when renaming files.          *
//*      -- Note that these charaters may interact with shell programs, so       *
//*         it is also necessary to "escape" these characters before sending     *
//*         them to the shell. See EscapeLinuxSpecialChars() method.             *
//*         -- Note that delimiting a filename with double quotes removes the    *
//*            "specialness" of all characters _except_: '$' '\' '`' and '"'.    *
//*   -- Implement multi-file comparison CompareFiles().                         *
//*      Previously, comparison could only be between two files. This            *
//*      enhancement allows each file in one group to be compared with the       *
//*      corresponding file in another group.                                    *
//*                                                                              *
//* v: 0.00.47 13-Oct-2018                                                       *
//*   -- Update formatting of ViewFile text extracted from OpenDocument files.   *
//*      As a side-effect, the grep of OpenDocuments is now also more robust.    *
//*      See odExtractOD_Text().                                                 *
//*   -- Implement support for viewing .ZIP archives. This already worked by     *
//*      magic (external 'less' preprocessor), but now it works intentionally.   *
//*      See ArchiveTarget(), vfcDecodeArchive().                                *
//*   -- Implement create/update/expand for ZIP archives.                        *
//*      -- Backup to archive target (ArchiveClipboardList()) implemented        *
//*         (with significant optimization).                                     *
//*      -- Create Archive: Archive_Create(). Most functionality shifted to      *
//*         three(3) sub-methods for optimized access across all archive code.   *
//*      -- Append To Archive: Archive_Append() and shared sub-methods.          *
//*      -- Expand Archive: Archive_Expand() and its sub-methods.                *
//*                                                                              *
//* v: 0.00.46 11-Aug-2018                                                       *
//*   - First pass at implementing 'grep' functionality is complete. Simple      *
//*     (only two user-accessible parameters), but functional, and includes      *
//*     direct scan of text in OpenDocument files.                               *
//*     In FileDlgUtils.cpp, see GrepFiles() and the gfxxx() method group.       *
//*   - Add support for viewing the (unformatted) _text_ of an OpenDocument      *
//*     file from the ViewFile context menu.                                     *
//*   - ProcessCWD() method now has a optional parameter to clear previous       *
//*     message. Implemented specifically for archive operations, but may        *
//*     be applied elsewhere.                                                    *
//*                                                                              *
//* v: 0.00.45 24-Sep-2017                                                       *
//*   - Complete transfer from Taggit of enhanced audio metadata reporting.      *
//*   - Implement the OpenExternalProgram() group of methods.                    *
//*     See FileDlgContext.cpp.                                                  *
//*   - Place hooks for constructing a multi-language user interface.            *
//*   - First pass at implementing 'grep' functionality is complete.             *
//*     - Filename selection is rather good, but search pattern entry leaves     *
//*       user with too many opportunities to screw up.                          *
//*     - Only the case-sensitivity and line-number switches are implemented.    *
//*     - For future enhancement: add additional switches, second pass with      *
//*       option to exclude ( -v ), option to save a copy of results WITHOUT     *
//*       ANSI color data.                                                       *
//*     See GrepFiles() and the gfXXX() methods.                                 *
//*                                                                              *
//* v: 0.00.44 05-Mar-2016                                                       *
//*   - Implement ArchiveOnClipboard() methods to support creation and           *
//*     expansion of archive files by the parent dialog.                         *
//*   - Add a new source module FileDlgBackup.cpp containing the new code for    *
//*     performing Backup, Synchronization and Archive-creation operations.      *
//*     This functionality is new, so there will likely be some bugs.            *
//*   - Enhance and overload the 'Scroll2MatchingFile' method. This supports     *
//*     user search of displayed filenames.                                      *
//*   - Implement the 'FindFiles' group of methods which scan the directory      *
//*     tree for filenames that contains the specified substring.                *
//*   - Removed upper-limit restriction on user-supplied timestamps for 64-bit   *
//*     systems. 32-bit systems are still limited to 31 Dec 2037 23:59:59.       *
//*     (The test is based on sizeof(time_t).)                                   *
//*   - Significant restructuring of the restore-from-trash method group:        *
//*     - Update 'tcSummaryScan' based on the equivalent method in the cTrash    *
//*       utility. Eliminates parameter clutter with no functionality change.    *
//*     - Implement the trashcan undo, to restore the most recent move-to-trash  *
//*       operation.                                                             *
//*     - Added sort-by-filename to the trashcan display sort options.           *
//*   - Implement Progress Bar for operations that will likely take more than    *
//*     a few seconds. Currently the Progbar is implemented only for Backup      *
//*     and Synch operations.                                                    *
//*   - Bug Fix: In dfsDisplayFilename() if path was root, '/' was duplicated.   *
//*   - Merge the contents of FileDlgBrowseCB.cpp into FileDlgClipbrd.cpp        *
//*     eliminating one source module.                                           *
//*                                                                              *
//* v: 0.00.43 01-Dec-2015                                                       *
//*   - Update all instances of the gString 'formatInt' call to match changes    *
//*     in the gString class.                                                    *
//*   - Update the names of 'limitChars' and 'limitCols' calls to gString.       *
//*   - Bug fix in pslSinglePaste_Options(): labels of inactive controls were    *
//*     being written at incorrect offset.                                       *
//*   - In Tree-view Mode, beautify the corners of the border.                   *
//*   - Implement the 'Find Link Target' item in the ViewFile context menu.      *
//*   - Removed 'backupDir' data member. It had never been used for anything.    *
//*   - Target for temporary files moved from installation directory to          *
//*     system's defined temp-file directory.                                    *
//*     - Some debugging code that relied on the old path had to be modified.    *
//*     - The VerifyTempStorage() method no longer makes sense because the       *
//*       base application code now does the setup for temporary files; so       *
//*       verification of temp storage is no longer done in this class.          *
//*       However, verification of trashcan location is still done, so the       *
//*       verification method was renamed to VerifyTrashcanConfig().             *
//*   - In ViewFileContents() method, replace call to 'xxd' utility with direct  *
//*     formatting for binary output. This was necessary because many Linux      *
//*     distros no longer include 'xxd'.                                         *
//*                                                                              *
//* v: 0.00.42 06-Dec-2014                                                       *
//*   - Bug fix in tsfTopFile(). If name of file to be trashed had no extension  *
//*     AND if it already existed in Trashcan, target path was being corrupted.  *
//*                                                                              *
//* v: 0.00.41 26-Nov-2013                                                       *
//*   - Convert the FormatInteger calls to gString-class formatInteger().        *
//*     This eliminates the grossly inefficient FormatInteger() methods AND      *
//*     eliminates the need for FormatInteger.hpp.                               *
//*   - Limit directory-name length and filename length changes to MAX_FNAME.    *
//*     Silently truncates names to avoid system error generated if name is      *
//*     too long.                                                                *
//*   - Enable TextboxAlert() for selected text-input prompts.                   *
//*   - Empty-Trashcan functionality is now functional for both files and        *
//*     directory trees.                                                         *
//*   - Restore-from-Trashcan for individual files is functional, and            *
//*     restore-from-Trashcan for directory trees is UNDER DEVELOPMENT.          *
//*                                                                              *
//* v: 0.00.40 05-Sep-2013                                                       *
//*   - Enhance SelectFile(), DeselectFile(), IsSelected(), NextDisplayItem()    *
//*     and add PrevDisplayItem() to support group filename selection.           *
//*   - Add an option to view contents of symbolic-link file.                    *
//*     See ViewFileContents() method.                                           *
//*   - Update tests for write-enable which simplifies the execution flow.       *
//*   - Replace SourceWEnable() methods, swpDecision() method and contents of    *
//*     SourceWriteEnable() with SourcePermission(), spEnable() methods, and     *
//*     spDecision() method. All of this allows for test of both access flags,   *
//*     readAcc and writeAcc for each file's tnFName-class information.          *
//*     These are fairly extensive changes. Look out for regression insects.     *
//*   - If caller is starting in TreeView Mode, skip read/display of CWD         *
//*     contents. See constructor.                                               *
//*                                                                              *
//* v: 0.00.39 23-Mar-2013                                                       *
//*   - Convert the embedded 'FormatInteger' group to use the FormatInteger.hpp  *
//*     'included'd in FileDlgBrowseCB.cpp. Still looking for a more efficient   *
//*     algorithm.                                                               *
//*   - The DispData class in FMgr has been redefined for multi-threaded data    *
//*     capture. This affects four (4) methods: ChildDirectory(),                *
//*     ParentDirectory(), SetDirectory(), and RefreshCurrDir().                 *
//*   - Trashcan implementation now uses the desktop's trash directory rather    *
//*     than the previously-specified custom trashcan implementation.            *
//*   - Modified 'NextSelectedItem' method to leave highlight unchanged if       *
//*     already referencing the last 'selected' item. Previously highlight       *
//*     went to bottom of list if called too many times. Not sure which is the   *
//*     user-friendly choice.                                                    *
//*   - Combine the calls to SetErrorCode and DisplayStatus methods and          *
//*     enhance error-reporting mechanism.                                       *
//*   - Begin design of the 'Backup' and 'Synchronize' functionality.            *
//*                                                                              *
//* v: 0.00.38 18-May-2012                                                       *
//*   - Implement application's user-selectable color scheme based on            *
//*     enhancements in NcDialog/NcWindow/NCurses classes. This is handled       *
//*     through the new 'cs' data member which is a 'class ColorScheme' object.  *
//*   - Create Scroll2MatchingFile() method. This method scans the list of       *
//*     displayed files for a filename match. Allows user to highlight a         *
//*     filename by typing part of the name.                                     *
//*   - Update various scrolling methods to eliminate redundant display          *
//*     refreshes by using the NcDialog MoveScrollextHighlight() method.         *
//*   - Rewrite of DisplayFileTree() method group to reflect TreeNode scanning   *
//*     algorithm.                                                               *
//*                                                                              *
//* v: 0.00.37 22-Jan-2012                                                       *
//*   - Move methods and data related to the file paste-from-clipboard           *
//*     operation into a new module, FileDlgPaste.cpp.                           *
//*   - Create new module FileDlgBrowseCB.cpp.                                   *
//*   - Eliminate the FileDlgClipbrd2.cpp module.                                *
//*   - Implement 'Paste Special' functionality, in two flavors for a robust     *
//*     algorithm.                                                               *
//*     1. A single file (or a single directory tree):                           *
//*        Intended primarily for 'special', non-directory files: FIFO,          *
//*        Char/Block device, Socket, Symlink.                                   *
//*     2. Multi-directory and/or multiple non-directory files at top level.     *
//*        All top-level files must be either directories or 'regular' files.    *
//*                                                                              *
//* v: 0.00.36 25-Jan-2011                                                       *
//*   - Replace all references to struct FileTime with references to class       *
//*     localTime.                                                               *
//*   - Move methods and data related to the file rename operation into a new    *
//*     module, FileDlgRename.cpp.                                               *
//*   - Move methods and data related to the file deletion operation into a      *
//*     new module, FileDlgDelete.cpp.                                           *
//*   - Minimize use of C-language, ASCII-oriented text manipulation functions.  *
//*                                                                              *
//* v: 0.00.35 19-Aug-2010                                                       *
//*   - Both file tree capture and copy/cut to clipboard are now converted to    *
//*     use the TreeNode and tnFName classes. Begin rewrite of                   *
//*     PasteClipboardList() to take advantage of the new functionality.         *
//*   - Change display color for Block Device, Char Device, and Socket to        *
//*     nc.ma (magenta). Because we will now allow user to see and modify (but   *
//*     not move or delete) files of these types, they now stand out as          *
//*     dangerous to touch.                                                      *
//*   - Enhance VerifyTempStorage() method to attempt repairs on application     *
//*     directory tree.                                                          *
//*   - Add a new source module, FileDlgTrash.cpp to contain methods related     *
//*     to the FileMangler Trashcan. Complete rewrite of DeleteSelectedFiles()   *
//*     to streamline the process and reduce chance of errors.                   *
//*                                                                              *
//* v: 0.00.34 22-Jul-2010                                                       *
//*   - Begin conversion of clipboard from ClipBoard class to TreeNode class.    *
//*     As it turns out, the original clipbloard data structure worked well      *
//*     for single-level cut/copy/paste, but was rather clunky for recursive     *
//*     operations.                                                              *
//*   - Because the TreeNode class is so flexible, we have tentatively decided   *
//*     to convert the linked list of DirEntry structures used to capture        *
//*     display data into an array of TreeNode/tnFName class instances.          *
//*     This is a huge structural upheaval, but worth it in terms of             *
//*     simplicity, relying as it does on recursion to do most of the work.      *
//*   - Add config option for path specification where temporary files are to    *
//*     be written.                                                              *
//*   - Replaced DirEntry versions of GetFileStats() and ExpandStats() with      *
//*     tnFName versions.                                                        *
//*                                                                              *
//* v: 0.00.33 07-Jul-2010                                                       *
//*   - Convert clipboard dynamic allocation from 'calloc' and 'free' to 'new'   *
//*     and 'delete'. This is a bit less efficient, but more C++ cool.           *
//*   - Implement BrowseDirTreeList() to view directory tree info created in     *
//*     FMgr class ScanDirTree().                                                *
//*                                                                              *
//* v: 0.00.32 15-Jun-2010                                                       *
//*   - Rewrite FormatInteger() method.                                          *
//*   - Add optional parameter 'enableWrite' to                                  *
//*     FileWritePermission(short pdIndex) method.                               *
//*   - Eliminate enum cowReturn and replace with corReturn.                     *
//*   - Begin implementation of recursive subdirectory scanning for copy, cut,   *
//*     delete, etc. operations. This requires modification of certain           *
//*     variables which are not wide enough to hold the data; specifically the   *
//*     clipBoard structure's members, as well as some FileDlg class members     *
//*     for tracking file count and file size. There is potential for overflow   *
//*     and unintended narrowing here, so regression testing is needed.          *
//*                                                                              *
//* v: 0.00.31 26-Apr-2010                                                       *
//*   - FileDlg class begins life as an exact copy of the FileWin class, but     *
//*     quickly eclipses its sibling. While the FileWin class relied on          *
//*     NcWindow objects to display the file data, the FileDlg class uses an     *
//*     NcDialog class object for file data display.                             *
//*                                                                              *
//* v: 0.00.30 14-Feb-2010                                                       *
//*   - Change development platform and compiler version.                        *
//*     Fedora 12 and GNU G++ (Gcc v: 4.4.2)                                     *
//*     This neccessitated several syntax changes in class and method            *
//*     definitions to accomodate changes in the C++ definition and the GCC      *
//*     compiler's warning and error messages.                                   *
//*   - Begin conversion from NcWindow class data display to NcDialog class      *
//*     data display.                                                            *
//********************************************************************************
//* Programmer's Notes:                                                          *
//*                                                                              *
//*                                                                              *
//********************************************************************************

#include "GlobalDef.hpp"
#include "FileDlg.hpp"
#include "FMgr.hpp"

//***********************
//** Local Definitions **
//***********************
//* Prototype for non-member, static callback method used by the *
//* FMgr class to display debugging messages, and a kludgy       *
//* pointer to NcDialog class used only by fmgrDebugMsg() method.*
static void fmgrDebugMsg ( const char* msg ) ;
static NcDialog* fmgrDbPtr = NULL ;


//****************
//** Local Data **
//****************
static const char FileDlgVersion[] = "0.0.55" ;    //* Class version number

static char  bLine[MAX_FNAME] ;
static char* blankLine = bLine ; // for display of empty directory

//* Pointer to dynamically-allocated memory used as a "clipboard" *
//* i.e. a method of passing file information from one FileDlg    *
//* object to another. See note in FileDlg.hpp and the following  *
//* methods: ClearClipboard() and FillClipboard().                *
TreeNode*   clipBoard = NULL ;            // pointer to base node of clipboard tree
char        clipBoard_srcPath[MAX_PATH] ; // pathspec for clipboard source data
UINT        clipBoard_dirTrees ;          // number of top-level directory files
UINT        clipBoard_topFiles ;          // number of top-level non-directory files
OpPend      clipBoard_opPending ;         // code for pending clipboard operation
bool        clipBoard_gvfsSrc ;           // 'true' if src data on virtual filesystem
bool        clipBoard_gvfsTrg ;           // 'true' if target is virtual filesystem


               //**********************************************
               //* Public Access Methods of the FileDlg class *
               //**********************************************
//*************************
//*      ~FileDlg         *
//*************************
//******************************************************************************
//* Destructor.                                                                *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

FileDlg::~FileDlg ( void )
{

   this->ClearClipboard ( true ) ;        // release any clipboard allocation
   delete this->fmPtr ;                   // release FMgr class instance

}  //* End ~FileDlg() *

//*************************
//*      FileDlg          *
//*************************
//******************************************************************************
//* Constructor for FileDlg object.                                            *
//*                                                                            *
//* Instantiate the FileDlg class, and the FMgr class to access the file       *
//* system and to display the results.                                         *
//* Caller provides information for NcDialog controls to be used.              *
//* On return, the caller's specified Path, File, and Statistics controls have *
//* been updated.                                                              *
//*                                                                            *
//* Input : init : InitFileDlg structure (by reference)                        *
//*                See discription of the fields of the InitFileDlg            *
//*                structure.                                                  *
//*                                                                            *
//* Returns: if successful, returns a reference to FileDlg object.             *
//*           (else will probably throw a system exception)                    *
//******************************************************************************
//* If an error occurs, the error number and a short description will be       *
//* displayed, but it is not possible to refuse to instantiate the class       *
//* without serious exception handling code which is hardly necessary since,   *
//* if something that awful happens, the calling application will probably     *
//* crash on return, alerting the user that something may be wrong.            *
//*                                                                            *
//* There are varied and complex things happening in this class, so we do      *
//* our best to validate the data that are gathered and used here.             *
//*                                                                            *
//* See also the notes in the header of this module.                           *
//*                                                                            *
//******************************************************************************

FileDlg::FileDlg ( InitFileDlg& init )
{
   //* Initalize object and data pointers *
   this->fmPtr       = NULL ; // pointer to instance of FMgr class
   this->dPtr        = NULL ; // pointer to caller's NcDialog object
   this->deList      = NULL ; // array of pointers to tnFName pointers (corresponds to textData order)
   this->textData    = NULL ; // display data for file information
   this->colorData   = NULL ; // array of color attributes for the display data
   this->fIndex      = ZERO ; // index of file-display control
   this->mIndex      = ZERO ; // index of status-message control
   this->pIndex      = ZERO ; // index of path-display control
   this->sIndex      = ZERO ; // index of statistics-display control
   
   //* Initialize misc member variables *
   this->fileCount   = ZERO ;
   this->dirSize     = ZERO ;
   this->selInfo.Count    = ZERO ;
   this->selInfo.Size     = ZERO ;
   this->selInfo.subCount = ZERO ;
   this->selInfo.subSize  = ZERO ;
   this->hIndex      = -1 ;
   this->selectAttr  = ncuATTR ;
   this->copyColor   = nc.grU ;
   this->cutColor    = nc.gyU ;
   this->deleteColor = nc.reU ;
   this->renameColor = nc.blU ;
   this->touchColor  = nc.blU ;
   *this->currDir    = NULLCHAR ;
   this->recentError = ecNOERROR ;
   this->recentErrno = ZERO ;
   this->userInfo.goodData = false ;
   this->autoRecurse = true ;
   this->trashcanOK  = false ;

   //* Initialize color attributes for the different file types *   
   this->ftColor[fmDIR_TYPE]     = nc.cy ;
   this->ftColor[fmREG_TYPE]     = nc.bw ;
   this->ftColor[fmLINK_TYPE]    = nc.bl ;
   this->ftColor[fmFIFO_TYPE]    = nc.br ;
   this->ftColor[fmCHDEV_TYPE]   = nc.ma ;
   this->ftColor[fmBKDEV_TYPE]   = nc.ma ;
   this->ftColor[fmSOCK_TYPE]    = nc.ma ;
   this->ftColor[fmUNKNOWN_TYPE] = nc.bw ;

   //* Transfer initialization data from caller's data structure *
   this->dPtr     = init.dPtr ;              // pointer to parent's dialog object
   this->fIndex   = init.fdispIndex ;        // index of file display control
   this->fulY     = init.fdispInit.ulY ;     // upper left Y of file display control
   this->fulX     = init.fdispInit.ulX ;     // upper left X of file display control
   this->fMaxY    = init.fdispInit.lines ;   // display rows in file display control
   this->fMaxX    = init.fdispInit.cols ;    // display columns in file display control
   this->mIndex   = init.statusIndex ;       // index of status-message control
   this->mulY     = init.statusInit.ulY ;    // upper left Y of status-message control
   this->mulX     = init.statusInit.ulX ;    // upper left X of status-message control
   this->mMax     = init.statusInit.cols ;   // max width of status message
   this->mNcolor  = init.statusInit.nColor ; // status message normal color
   this->mFcolor  = init.statusInit.fColor ; // status message special color
   this->pIndex   = init.pathIndex ;         // index of path-display control
   this->pMax     = init.pathInit.cols ;     // max width of path display
   this->pNcolor  = init.pathInit.nColor ;   // path string normal color
   this->pFcolor  = init.pathInit.fColor ;   // path string special color
   this->sIndex   = init.statsIndex ;        // index of file-statistics control
   this->sMax     = init.statsInit.cols ;    // max length of statistics message
   this->sNcolor  = init.statsInit.nColor ;  // stats message normal color
   this->sFcolor  = init.statsInit.fColor ;  // stats message special color
   this->cs       = init.cfgInit.cScheme ;   // application 'color scheme'
         // initialization of remaining configuration options is handled below

   //* Get the parent dialog's size and position.                       *
   //* Because we are only fully in control of the file-display control,*
   //* we try to only open other dialogs within its borders. Convert the*
   //* dialog-relative position to absolute terminal window position.   *
   this->dPtr->GetDialogDimensions ( this->dRows, this->dCols ) ;
   winPos wpos = this->dPtr->GetDialogPosition () ;
   this->dulY   = wpos.ypos ;
   this->dulX   = wpos.xpos ;
   this->fulY  += this->dulY ;
   this->fulX  += this->dulX ;

   //* Create a display string for empty file window *
   this->EmptyStr  = &blankLine ;
   this->EmptyAttr = &this->ftColor[ZERO] ;
   short i ;
   for ( i = ZERO ; i < (this->fMaxX-2) ; i++ )
      bLine[i] = SPACE ;
   bLine[i] = NULLCHAR ;

   //* Set the paths to directories used for the *
   //* Trashcan and for storing temporary files. *
   gString pgs ;
   pgs = init.cfgInit.trPath ;
   pgs.copy( this->trashDir, MAX_PATH ) ;
   pgs = init.cfgInit.tfPath ;
   pgs.copy( this->tempDir, MAX_PATH ) ;

   //* Give static, non-member method, fmgrDebugMsg() access to NcDialog class.*
   //* Used only to display debugging message sent from the FMgr class.        *
   if ( fmgrDbPtr == NULL )
      fmgrDbPtr = this->dPtr ;

   //* Instantiate the FMgr (file manager) class                               *
   //* The parameters initialize the identifying color attributes for each     *
   //* file-type and provide a place to create temporary files.                *
   //* If a start-up directory was specified by caller, change to new CWD.     *
   FMgrConfigOptions cfg = { 
                             this->ftColor, 
                             this->tempDir,
                             init.startPath,
                             init.cfgInit.sortOption,
                             init.cfgInit.caseSensitive,
                             init.cfgInit.showHidden,
                             init.rootScan,
                             (DMSG*)(&fmgrDebugMsg),
                           } ;
   this->fmPtr = new FMgr( cfg ) ;

   //* Caller's sort option may be invalid, so be safe *
   init.cfgInit.sortOption = this->fmPtr->GetSortOption () ;

   //* Remaining flags and options of the ConfigOptions class are internal to  *
   //* the FileDlg class. Save a copy of the configuration options.            *
   this->cfgOptions = init.cfgInit ;

   //* Get a copy of the user's info: User ID,   *
   //* User Name, User GID, User Group Name, etc.*
   this->fmPtr->IdentifyUser ( this->userInfo ) ;

   //* Verify existence of Trashcan directory tree. *
   this->trashcanOK = this->VerifyTrashcanConfig () ;

   //* If specified, read and display contents of current directory.           *
   //* If clipboard has not yet been instantiated, do so now. Note that if     *
   //* clipboard has already been instantiated, its data are not disturbed     *
   //* unless the CWD is also the source of the clipboard data, in which case, *
   //* the clipboard is cleared.                                               *
   //* Path display control, and statistics display control are also updated.  *
   bool createClipboard = (bool)(clipBoard == NULL ? true : false) ;
   if ( init.refresh )
      this->RefreshCurrDir ( true, createClipboard ) ;
   //* Otherwise, caller wants to start in TreeView Mode, so skip directory    *
   //* read. Just get CWD and instantiate/clear the clipboard.                 *
   else
   {
      this->ClearClipboard () ;
      this->fmPtr->GetCurrDir ( this->currDir ) ;
   }

}  //* End FileDlg() *

//*************************
//*  ScrollThruFileList   *
//*************************
//******************************************************************************
//* Scroll through the list of files in the dctSCROLLEXT control which         *
//* currently has the input focus.                                             *
//* Returns when a non-scrolling key is detected.                              *
//* Scrolling keys are: nckUP, nckDOWN, nckPGUP, nckPGDOWN, nckHOME, nckEND.   *
//* Note that nckLEFT and nckRIGHT ARE NOT handled by this method; they are    *
//* returned to caller for processing.                                         *
//*                                                                            *
//* Input  : info : uiInfo class object (by reference, initial values ignored) *
//*                 on return, contains data on user selection                 *
//*                                                                            *
//* Returns: index of dialog control which currently has input focus           *
//******************************************************************************
//* This method is a pass-through to the NcDialog::EditScrollext() method.     *
//*                                                                            *
//******************************************************************************

short FileDlg::ScrollThruFileList ( uiInfo& info )
{
   short cIndex = this->dPtr->EditScrollext ( info ) ;
   this->hIndex = info.selMember ;        // track the highlight position
   return cIndex ;

}  //* End ScrollThruFileList() *

//*************************
//* Scroll2MatchingFile   *
//*************************
//******************************************************************************
//* Scan from currently-highlighted filename forward, wrapping around to top   *
//* of list if necessary, until a filename which begins with the matching      *
//* sub-string is found. Highlight the matching file.                          *
//* If no match found, highlight is not moved.                                 *
//*                                                                            *
//* Input  : substr : (by reference) contains at least one character to be     *
//*                   compared with the beginning of the displayed filenames   *
//*          indx   : (by reference) receives index of currently-highlighted   *
//*                   item                                                     *
//*          sensi  : (optional, false by default)                             *
//*                   if 'true' do a case-sensitive comparison,                *
//*                   if 'false' do a case-INsensitive comparison              *
//*                                                                            *
//* Returns: 'true' if substring match found                                   *
//******************************************************************************

bool FileDlg::Scroll2MatchingFile ( const gString& substr, short& indx, bool sensi )
{
   gString fName ;
   UINT  tIndex = this->hIndex ;          // target index
   bool  found = false ;

   while ( tIndex < this->fileCount )
   {
      fName = this->deList[tIndex]->fName ;
      if ( (fName.find( substr, ZERO, sensi )) == ZERO )
      {
         found = true ;
         break ;
      }
      ++tIndex ;
   }

   //* If not found below current index, wrap to top of list *
   if ( ! found )
   {
      tIndex = ZERO ;
      while ( tIndex < this->hIndex )
      {
         fName = this->deList[tIndex]->fName ;
         if ( (fName.find( substr, ZERO, sensi )) == ZERO )
         {
            found = true ;
            break ;
         }
         ++tIndex ;
      }
   }

   //* If a matching filename was found, move highlight to it *
   if ( found != false )
      this->hIndex = this->dPtr->MoveScrollextHighlight ( this->fIndex, ZERO, tIndex ) ;

   indx = this->hIndex ;         // set caller's index
   return found ;                // return search results

}  //* End Scroll2MatchingFile() *

//*************************
//* Scroll2MatchingFile   *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* Scan from currently-highlighted filename forward, wrapping around to top   *
//* of list if necessary, until a filename which begins with the matching      *
//* sub-string is found. Highlight the matching file.                          *
//* If no match found, highlight is not moved.                                 *
//*                                                                            *
//* Input  : substr: (by reference) contains at least one character to be      *
//*                  compared with the beginning of the displayed filenames    *
//*          sensi : (optional, false by default)                              *
//*                  if 'true' do a case-sensitive comparison,                 *
//*                  if 'false' do a case-INsensitive comparison               *
//*                                                                            *
//* Returns: index of currently-highlighted file                               *
//******************************************************************************

short FileDlg::Scroll2MatchingFile ( const char* substr, bool sensi )
{
   gString gs( substr ) ;
   short dummyIndex ;

   this->Scroll2MatchingFile ( gs, dummyIndex, sensi ) ;
   return this->hIndex ;

}  //* End Scroll2MatchingFile() *

//*************************
//*     DisplayStatus     *
//*************************
//******************************************************************************
//* Update the status information in the path control and status control.      *
//* Also report any errors sent to us.                                         *
//*                                                                            *
//* For non-error status (errNum == ecNOERROR):                                *
//* ---------------------                                                      *
//* Update status information for this directory.                              *
//*  1) path specification                                                     *
//*  2) file count and total size for displayed file data                      *
//*     and info for any 'selected' files,                                     *
//*                                                                            *
//* For error conditions:                                                      *
//* ---------------------                                                      *
//* 1) For incidental errors i.e. those caught before calling the system with  *
//*    a request that would likely have failed, display a momentary warning    *
//*    in the status control before updating the status info as above.         *
//* 2) For more serious errors i.e. errors returned from system calls or       *
//*    situations were user must be told that operation cannot proceed,        *
//*    display the status info as above and then open an informational dialog  *
//*    to display the bad news.                                                *
//*                                                                            *
//* Input  : fdErrnum : (optional, ecNOERROR by default)                       *
//*                     used to set 'recentError' member variable              *
//*          sysErrnum: (optional, ZERO by default)                            *
//*                     used to set 'recentErrno' member variable              *
//*                      if  0, then reset 'recentErrno'                       *
//*                      if fdErrnum != ecNOERROR:                             *
//*                        if -1, then get the system's 'errno' value          *
//*                               and store it in 'recentErrno'                *
//*                        else, interpret the value as a valid 'errno' value  *
//*                        (see also SetErrorCode method below)                *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void FileDlg::DisplayStatus ( fdErrCode fdErrnum, short sysErrnum )
{
   const short incidentalINTERVAL = 2800 ;   // message display interval (mS)
   gString gs ;                     // message formatting
   errorCode ec ;                   // error-reporting info

   //* Track the operational status *
   this->SetErrorCode ( fdErrnum, sysErrnum ) ;
   if ( fdErrnum != ecNOERROR )     // get error codes and messages
      this->GetErrorCode ( ec ) ;

   //* Update path-display control *
   this->DisplayDirPath () ;

   //* Incidental-error status information *
   if ( fdErrnum != ecNOERROR && sysErrnum == ZERO )
   {
      gs.compose( L" Error %hd: %s", &ec.fdErrorCode, ec.fdErrorDesc ) ;
      attr_t dColor = dtbmFcolor ;
      dtbmData tbMsg( gs.gstr(), &dColor, true ) ;
      this->dPtr->DisplayTextboxMessage ( this->sIndex, tbMsg ) ;
      chrono::duration<short, std::milli>aMoment( incidentalINTERVAL ) ;
      this_thread::sleep_for( aMoment ) ;
   }

   //* Normal status information plus 'selected file' information *
   char  dirFiles[16], dirBytes[16] ;
   gs.formatInt( (ULONG)this->fileCount, 5 ) ;
   gs.copy( dirFiles, 6 ) ;
   gs.formatInt( this->dirSize, 7 ) ;
   gs.copy( dirBytes, 8 ) ;
   gs.compose( L"%s files %s bytes", dirFiles, dirBytes ) ;
   if ( this->selInfo.Count > ZERO )
   {
      char selFiles[16], selBytes[16] ;
      gString gss ;
      gss.formatInt( (ULONG)(this->selInfo.Count + this->selInfo.subCount), 5 ) ;
      gss.copy( selFiles, 6 ) ;
      gss.formatInt( (UINT64)(this->selInfo.Size + this->selInfo.subSize), 7 ) ;
      gss.copy( selBytes, 8 ) ;
      gss.compose( L"  Select:%s: %s", selFiles, selBytes ) ;
      gs.append( gss.gstr() ) ;
   }
   dPtr->SetTextboxText ( this->sIndex, gs ) ;  // Display the status message

   //* Serious error status information - display it in a dialog.*
   if ( fdErrnum != ecNOERROR && sysErrnum != ZERO )
   {
      //* Message string color attributes *
      attr_t msgColor[] = 
      {
       this->cs.sd | ncbATTR, // title color attribute 
       this->cs.sd,           // blank line
       this->cs.sd,           // FileDlg message heading
       this->cs.sd,           // FileDlg error
       this->cs.sd,           // blank line
       this->cs.sd,           // Errno message heading
       this->cs.sd,           // Errno message
       this->cs.sd            // null string (to be safe)
      } ;
      gs.compose( L" Error %hd: %s", &ec.fdErrorCode, ec.fdErrorDesc ) ;
      gString gs2, gs3 ;
      if ( ec.sysErrorCode > ZERO )
      {
         gs2.compose( L" Sys Error:: %hd", &ec.sysErrorCode ) ;
         gs3.compose( L" %s", ec.sysErrorDesc ) ;
         gs3.limitCols( 51 ) ;
      }
      const char* dlgMsg[] = 
      {
         "  ERROR STATUS  ",
         " ",
         "Unable to complete specified operation!",
         gs.ustr(),
         " ",
         gs2.ustr(),
         gs3.ustr(),
         NULL
      } ;
      this->InfoDialog ( (const char**)dlgMsg, msgColor ) ;
   }

}  //* End DisplayStatus() *

//*************************
//*     SetErrorCode      *
//*************************
//******************************************************************************
//* A file system error has just occurred. Get the system's 'errno' value,     *
//* store it, and set our local error code appropriately                       *
//*                                                                            *
//* Initializes recentError and recentErrno values.                            *
//*                                                                            *
//* NOTE: Don't call this method for application errors or user errors         *
//*       because the recentErrno value will be incorrect.                     *
//*                                                                            *
//* Input  : eCode : member of enum fdErrCode                                  *
//*                  Note: if called with ecNOERROR, previous error is erased. *
//*          erCode: (optional, -1 by default)                                 *
//*                  if specified, this replaces the 'errno' code retrieved    *
//*                  from the FMgr class.                                      *
//*                                                                            *
//* Returns: error code                                                        *
//******************************************************************************

fdErrCode FileDlg::SetErrorCode ( fdErrCode eCode, short erCode )
{

   this->recentError = eCode ;            // set local error code
   if ( eCode == ecNOERROR )
      this->recentErrno = ZERO ;          // discard any previous error code
   else
   {  //* Force an errno code, (indicates that application *
      //* found the error before system call was made).    *
      if ( erCode >= ZERO )
         this->recentErrno = erCode ;
      //* Get the most recently-reported system error code *
      else
         this->recentErrno = this->fmPtr->GetErrorCode () ;
   }
   return this->recentError ;

}  //* End SetErrorCode() *

//*************************
//*    DisplayDirPath     *
//*************************
//******************************************************************************
//* Update the directory-display window with the full path of the current      *
//* current directory.                                                         *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void FileDlg::DisplayDirPath ( void )
{
   //* Get a working copy of the path string *
   gString  pgs ;
   pgs.compose( L" %s", this->currDir ) ;

   //* If path string is too wide, compress it *
   short maxCols = this->pMax - 1 ;
   if ( pgs.gscols() > maxCols )
      this->TrimPathString ( pgs, maxCols ) ;

   //* Update the path-display control *
   dPtr->SetTextboxText ( this->pIndex, pgs ) ;

}  //* End DisplayDirPath() *

//*************************
//*    RefreshCurrDir     *
//*************************
//********************************************************************************
//* Read contents of the current directory and display them in the dialog's      *
//* file-display control.                                                        *
//*                                                                              *
//* Any data on the clipboard MAY BE discarded, if it is from the currently-     *
//* displayed directory.                                                         *
//*                                                                              *
//* Input  : withHilight:    (optional, true by default)                         *
//*                          if true, display data with highlight on             *
//*                            current item                                      *
//*                          if false, display data with no highlight            *
//*        : forceClipClear: (optional, false by default)                        *
//*                          if 'true' force clipboard to be cleared             *
//*                            regardless of its source directory                *
//*                                                                              *
//* Returns: OK if successful, else ERR                                          *
//*          (if ERR, access error number through GetErrorCode())                *
//********************************************************************************
//* Programmer's Note: We may occasionally discard valid clipboard data here,    *
//* but it is better than allowing invalid data to remain on the clipboard.      *
//* See notes in ClearClipboard() method.                                        *
//*                                                                              *
//* If there is data on the clipboard, AND the data came from this directory,    *
//* clear the clipboard on the assumption that something in the directory has    *
//* changed - potentially invalidating the clipboard data. (see note)            *
//*                                                                              *
//*                                                                              *
//* Notes for enhancing read speed:                                              *
//*  1) Read top-level directories and files.                                    *
//*  2) Read second-level directories and files.                                 *
//*  3) Mountpoints:                                                             *
//*     a) If top-level directory is a mountpoint, stop the read after           *
//*        second level                                                          *
//*     b) If top-level directory IS NOT a mountpoint (is on the same            *
//*        filesystem as CWD), continue recursion.                               *
//*     c) Note that several directories just below '/' are reported as          *
//*        mountpoints, even though they seem to be on the _same_ filesystem.    *
//*        /dev, /proc, /run, /sys, /tmp                                         *
//*                                                                              *
//*                                                                              *
//*                                                                              *
//*                                                                              *
//*                                                                              *
//*                                                                              *
//********************************************************************************

short FileDlg::RefreshCurrDir ( bool withHilight, bool forceClipClear )
{
short    status = OK ;

   //* Un-select any file selections. Also, if the displayed directory is      *
   //* anywhere on the clipboard data path, clear clipboard.                   *
   if ( (forceClipClear != false) || ((this->TargDirEqualsClipDir ()) != false) )
      this->ClearClipboard () ;
   this->DeselectFile ( true ) ;

   this->ProcessCWD () ;   // let user know that we are working
   status = this->rcdScan ( rcdsRefresh, NULL, withHilight ) ;

   return status ;

}  //* End RefreshCurrDir() *

//*************************
//*    RedrawCurrDir      *
//*************************
//******************************************************************************
//* Redraw the current data set in the file-display control.                   *
//* Statistics control is also updated.                                        *
//*                                                                            *
//* Any data on the clipboard are not affected.                                *
//*                                                                            *
//* Input  : withHilight:    (optional, true by default)                       *
//*                          if true, display data with highlight on           *
//*                            current item                                    *
//*                          if false, display data with no highlight          *
//*        : clearSelection: (optional, false by default)                      *
//*                          if 'true' clear file selections.                  *
//*                                                                            *
//* Return : OK if successful, else ERR                                        *
//*          (if ERR, access error number through GetErrorCode())              *
//******************************************************************************

short FileDlg::RedrawCurrDir ( bool withHilight, bool clearSelection )
{
short    status = OK ;

   if ( clearSelection != false )
   {
      //* If there are 'selected' files in the file-display control *
      //* de-select all files in the list.                          *
      if ( (this->IsSelected ( true )) != false )
         this->DeselectFile ( true ) ;
   }
   //* Update the statistics control and re-draw file data *
   this->DisplayStatus () ;
   this->dPtr->RefreshScrollextText ( this->fIndex, withHilight ) ;

   return   status ;

}  //* End RedrawCurrDir() *

//*************************
//*    ChildDirectory     *
//*************************
//******************************************************************************
//* If the currently-highlighted filename is the name of a directory,          *
//* set it as the current directory and display its contents.                  *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
//*          (if ERR, access error number through GetErrorCode())              *
//******************************************************************************

short FileDlg::ChildDirectory ( void )
{
   short    status = OK ;     // return value

   //* If current directory is not empty *
   if ( this->fileCount > ZERO )
   {
      //* If file is a _link_ to a directory, follow the link *
      // NOT YET IMPLEMENTED!

      //* If the highlighted file is a directory type *
      if ( (this->deList[this->hIndex])->fType == fmDIR_TYPE )
      {
         this->ProcessCWD () ;   // let user know that we are working
         status = this->rcdScan ( rcdsChild ) ;
      }
      else
      {  //* This is an application error. Caller should have verified the target.*
         status = ERR ;
         this->recentError = ecTARGDIR ;
         this->recentErrno = ZERO ;
      }
   }
   else
   { /* request to select filename from empty directory was discarded */ }

   return status ;

}  //* End ChildDirectory() *

//*************************
//*    ParentDirectory    *
//*************************
//******************************************************************************
//* If the current directory is not the root directory, set the parent         *
//* directory as the current directory and display its contents.               *
//* Else, redisplay the root directory.                                        *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
//*          (if ERR, access error number through GetErrorCode())              *
//******************************************************************************

short FileDlg::ParentDirectory ( void )
{
   short    status = OK ;     // return value

   //* If user has selected files in the      *
   //* current directory, discard selections  *
   //* (any data on the clipboard stays there)*
   if ( this->selInfo.Count > ZERO )
   {
      this->selInfo.Count = this->selInfo.subCount = ZERO ;
      this->selInfo.Size  = this->selInfo.subSize  = ZERO ;
   }

   this->ProcessCWD () ;   // let user know that we are working
   status = this->rcdScan ( rcdsParent ) ;

   return status ;

}  //* End ParentDirectory() *

//*************************
//*     SetDirectory      *
//*************************
//******************************************************************************
//* Set the specified directory path as the new directory to be displayed      *
//* in the window.                                                             *
//*                                                                            *
//* Note: Always change directories using the automatic built-in               *
//* ChildDir() and ParentDir() methods when possible, and call this method     *
//* only when it is necessary to 'jump' to a different directory.              *
//*                                                                            *
//* Input  : dirPath : full path specification for new target directory        *
//*          recurse : (optional, 'true' by default)                           *
//*                    'true'  : scan the full target-directory tree           *
//*                    'false' : scan only the top-level of the target tree    *
//*                              Resets the 'autoRecurse' member flag.         *
//*                              a) mountpoint for a large external drive      *
//*                              b) target is the root directory and full scan *
//*                                 is not desired.                            *
//*                              c) target is one of the "special" directories *
//*                                 i.e. '/proc' and full scan not is desired. *
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
//*          (if ERR, access error number through GetErrorCode())              *
//******************************************************************************

short FileDlg::SetDirectory ( const gString& dirPath, bool recurse )
{
   short    status = ERR ;    // return value

   //* Stat the path to see if it exists AND to           *
   //* verify that it specifies a directory name          *
   //* If file is a link to a directory, follow the link. *
   tnFName  tnStat ;
   if ( ((this->fmPtr->GetFileStats ( tnStat, dirPath, true )) == OK) 
        && (tnStat.fType == fmDIR_TYPE) )
   {
      if ( ! recurse )        // disable recursion
         this->autoRecurse = false ;
      this->ProcessCWD () ;   // let user know that we are working
      status = this->rcdScan ( rcdsSet, &dirPath ) ;
   }
   else
   {  //* This is an application error. Caller should have verified the target.*
      this->recentError = ecTARGDIR ;
      this->recentErrno = ZERO ;
   }
   return status ;

}  //* End SetDirectory() *

//*************************
//* VerifyTrashcanConfig  *
//*************************
//******************************************************************************
//* PRIVATE METHOD:                                                            *
//*                                                                            *
//* Verify that the directories specified for desktop's Trashcan (or user's    *
//* alternate trashcan) actually exist and that user has read/write access.    *
//*                                                                            *
//* Note that the Trashcan specification says that if the 'files' and/or 'info'*
//* subdirectories do not exist, that they should immediately be created;      *
//* however, locating, validating and/or repairing the Trashcan is performed   *
//* at a higher level (see FmConfig.cpp). Here, we simply determine whether    *
//* we have access to the specified Trashcan, without attempting repair.       *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if temporary storage verified, else 'false'                *
//*                 Note that if the Trashcan is not found, it WILL NOT trigger*
//*                 an error return, but WILL disable move-to-trash operations.*
//******************************************************************************

bool FileDlg::VerifyTrashcanConfig ( void )
{
   bool configured = false ;

   //* Test for accessibility of Trashcan. *
   if ( ! this->trashcanOK )
   {
      //* Test for existence of base Trashcan directory.*
      fmFType  baseType ;     // file type
      if ( (this->TargetExists ( this->trashDir, baseType )) != false )
      {
         //* Verify that it is a directory *
         if ( baseType == fmDIR_TYPE )
         {
            //* Test for existence, type and read/write accessibility *
            //* of the two necessary subdirectories.                  *
            gString fileSub( "%s/%s", this->trashDir, trashFiles ),
                    infoSub( "%s/%s", this->trashDir, trashInfo ) ;
            tnFName  fileStats, infoStats ;
            bool fileExists = this->TargetExists ( fileSub.ustr(), fileStats ), 
                 infoExists = this->TargetExists ( infoSub.ustr(), infoStats ) ;
            if ( fileExists && infoExists &&
                 (fileStats.fType == fmDIR_TYPE) && fileStats.readAcc && fileStats.writeAcc &&
                 (infoStats.fType == fmDIR_TYPE) && infoStats.readAcc && infoStats.writeAcc )
               configured = true ;
         }
      }
   }
   return configured ;

}  //* End VerifyTrashcanConfig() *

//*************************
//*        rcdScan        *
//*************************
//********************************************************************************
//* PRIVATE METHOD:                                                              *
//*                                                                              *
//* Display top-level of captured directory-tree data and wait for the           *
//* directory-tree scan to complete. This method isolates the invocation of      *
//* the multi-threaded directory-tree scan i.e. creation and destruction of      *
//* DispData-class objects. While it is not strictly necessary to isolate this   *
//* functionality, it allows easy debugging and exception handling.              *
//*                                                                              *
//* Input  : callID : indicates which of the public methods called us            *
//*                   Called by ChildDirectry(), ParentDirectory(),              *
//*                   SetDirectory(), RefreshCurrDir().                          *
//*          dirPath: (optional, NULL pointer by default)                        *
//*                   for call from SetDirectory() only: new directory path      *
//*          hilight: (optional, 'true' by default)                              *
//*                   for call from RefreshCurrDir() only: display data          *
//*                   with/without a visible highlight                           *
//*                                                                              *
//* Returns: OK if successful, else ERR                                          *
//*          (if ERR, access error number through GetErrorCode())                *
//********************************************************************************
//* Note: This method launches a child thread as a progress monitor, and then    *
//*       deletes it when the scan is complete.                                  *
//*                                                                              *
//********************************************************************************

short FileDlg::rcdScan ( rcdsCaller callID, const gString* dirPath, bool hilight )
{
   const short  MAX_RETRIES = 2 ;
   static short retryCount = MAX_RETRIES ; // directory-scan retry count

   short status = OK ;           // return value

   //* If user has selected files in the      *
   //* current directory, discard selections  *
   //* (any data on the clipboard stays there)*
   if ( this->selInfo.Count > ZERO )
   {
      this->selInfo.Count = this->selInfo.subCount = ZERO ;
      this->selInfo.Size  = this->selInfo.subSize  = ZERO ;
   }

   //* Width of the display string is width of the control - 1. *
   DispData dData( (this->fMaxX - 2) ) ;  // raw display data
   //* If we are about to scan a large external drive, *
   //* (see CmdMount()), scan only the top level.      *
   if ( ! this->autoRecurse )
   {
      dData.autoRecursion( false ) ;      // signal the low-level scan routine
      this->autoRecurse = true ;          // reset the member flag
   }

   gString newPath ;                      // target directory filespec
   bool cdStatus ;                        // change-directory status

   //* Launch a thread to handle reporting of scan progress *
   thread* monitorThread = NULL ;
   bool monitorActive = this->rcdLaunchMonitor ( dData, &monitorThread ) ;

   //* If monitor thread is now active *
   if ( monitorActive )
   {
      switch ( callID )
      {
         //* Move to the child directory. Name of target  *
         //* is name of directory under the highlight.    *
         case rcdsChild:
            newPath = this->deList[hIndex]->fName ;
            cdStatus = this->fmPtr->CdChild ( newPath, dData ) ;
            break ;
         //* Move to the parent directory.                *
         case rcdsParent:
            cdStatus = this->fmPtr->CdParent ( newPath, dData ) ;
            break ;
         //* Move to the specified directory path.        *
         case rcdsSet:
            newPath = *dirPath ;
            cdStatus = this->fmPtr->CdPath ( newPath, dData ) ;
            this->fmPtr->GetCurrDir ( this->currDir ) ;
            newPath = this->currDir ;
            break ;
         //* Re-read contents of current directory.       *
         case rcdsRefresh:
            cdStatus = this->fmPtr->RefreshCurrDir ( dData ) ;
            this->fmPtr->GetCurrDir ( this->currDir ) ;
            newPath = this->currDir ;
            break ;
      }

      //* Save a copy of returned data (will be valid even if there is *
      //* an error), then write the new data to the dialog controls.   *
      dData.getData( this->deList, this->textData, this->colorData, 
                     this->fileCount, this->dirSize ) ;  
      this->hIndex = ZERO ;      // index first item in display list
      newPath.copy( this->currDir, MAX_PATH ) ; // refresh our path variable
      this->DisplayFiles ( hilight ) ;          // update file display
      if ( cdStatus != false )
      {  //* Update status info
         fdErrCode fde = ecNOERROR ;
         if ( this->fileCount == ZERO && ((this->fmPtr->GetErrorCode ()) == EACCES) )
            fde = ecDIRNOREAD ;
         this->DisplayStatus ( fde ) ;
      }
      else
      {  //* Error reading target directory *
         this->DisplayStatus ( ecTARGDIR, EACCES ) ;
         status = ERR ;
      }

      //* Wait for monitor thread to finish.       *
      //* This indicates that the scan is complete.*
      if ( monitorThread != NULL )
      {
         monitorThread->join() ;
         delete monitorThread ;     // delete the monitor thread
      }
   }     // progress monitor active
   else
      ;  // progress monitor error

   //* If system exception(s) occurred during the scan,      *
   //* alert the user and force application rescan (or exit).*
   const char* errMsg = NULL ;
   if ( (dData.getExceptions ( errMsg )) > ZERO )
   {
      const char *ExitMsg6 = "however; the application must exit in order to",
                 *ExitMsg7 = "clear any potentially corrupted control information.",
                 *msgText[9] = 
      { //1234567890123456789012345678901234567890123456789012 - (max line length)
         "  WARNING  -  WARNING  ",
         "A system error has occurred.",
         errMsg,
         "The application has requested a dynamic memory",
         "allocation or other system resources, and the system",
         "was unable to comply. No data have been lost;",
         "however, to be safe, the data in current directory",
         "will be rescanned when this dialog is closed.",
         NULL
      } ;
      attr_t msgAttr[] = 
      { this->cs.em, this->cs.sd, this->cs.wr, this->cs.sd, this->cs.sd, 
        this->cs.sd, this->cs.sd, this->cs.sd, this->cs.sd, this->cs.sd } ;

      //** Decrement the retry counter. If ZERO, display "Exit" message. **
      if ( --retryCount <= ZERO )
      {
         msgText[6] = ExitMsg6 ;
         msgText[7] = ExitMsg7 ;
      }

      //* Alert the user *
      this->InfoDialog ( msgText, msgAttr ) ;

      //* Push the "Update" (or "Exit") keycode onto the keystroke stack.*
      wkeyCode wkPush( (retryCount > ZERO ? nckC_U : nckC_Q), wktFUNKEY ) ;
      this->dPtr->UngetKeyInput ( wkPush ) ;
   }
   else
   {
      //* No system exceptions during scan. Reset the retry counter.*
      retryCount = MAX_RETRIES ;
   }
   return status ;

}  //* End rcdScan() *

//*************************
//*   rcdLaunchMonitor    *
//*************************
//********************************************************************************
//* PRIVATE METHOD - Launch a thread to monitor the progress of a directory      *
//* tree scan.                                                                   *
//*                                                                              *
//* Input  : dData     : (by reference) Container for display data               *
//*                      if exception thrown, receives error message             *
//*          monThread : pointer to caller's pointer to thread                   *
//*                                                                              *
//* Returns: 'true' if monitor thread created and launched successfully          *
//*          'false' if error (exception message written to 'dData')             *
//*                  and caller's thread pointer will be null pointer.           *
//********************************************************************************

bool FileDlg::rcdLaunchMonitor ( DispData& dData, thread** monThread )
{
   bool  monAllocated = false,   // 'true' if monitor thread allocated
         monLaunched  = false ;  // 'true' if monitor thread launched

   chrono::duration<short, std::milli>aMoment( 10 ) ;
   for ( short i = 3 ; i > ZERO ; --i ) // three strikes, and you're out!
   {
      if ( (*monThread = new (std::nothrow) std::thread) != NULL )
      {
         monAllocated = true ;
         try
         {
            **monThread = thread( &FileDlg::rcdProgmon, this, &dData, &monLaunched ) ;
            this_thread::sleep_for( aMoment ) ;  // give the new thread time to respond
            if ( ! monLaunched )    // second chance
            {
               **monThread = thread( &FileDlg::rcdProgmon, this, &dData, &monLaunched ) ;
               this_thread::sleep_for( aMoment ) ;  // give the new thread time to respond
            }
         }
         catch ( ... )        // monitor thread created but not launched
         {
            dtbmData tbMsg( " Progress-monitor thread not launched! " ) ;
            this->dPtr->DisplayTextboxMessage ( this->sIndex, tbMsg ) ;
            // Programmer's Note: We cannot set an exception in 'dData' here 
            // because 'dData' is reset before scanning the directory tree.
         }
         break ;
      }
      this_thread::sleep_for( aMoment ) ;  // give the system some time
   }

   if ( ! monAllocated )
      dData.setException( "Progress-monitor thread not allocated." ) ;
   else if ( ! monLaunched )
   {
      dData.setException( "Progress-monitor thread not launched." ) ;

      if ( *monThread != NULL )
      { delete *monThread ; *monThread = NULL ; }
   }

   return ( monAllocated && monLaunched ) ;

}  //* End rcdLaunchMonitor() *

//*************************
//*      rcdProgmon       *
//*************************
//********************************************************************************
//* PRIVATE METHOD - monitor the progress of directory-tree scan and display     *
//* the number of files scanned so far.                                          *
//*                                                                              *
//* The timing is set up so that if the scan completes in less than              *
//* 'aWhile', then no display updates will occur. Updates will occur only if     *
//* the scan is taking a "long" time e.g. from slow external devices or large    *
//* amounts of data.                                                             *
//*                                                                              *
//* Input  : ddPtr  : pointer to DispData object containing status of            *
//*                   directory-tree scan                                        *
//*          active : pointer to flag is set on entry to signal that             *
//*                   the monitor thread is active                               *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* Although there is a small performance hit involved in redundantly counting   *
//* the number of files processed, it gives the user something shiny to look     *
//* at. Simple creatures, users.                                                 *
//*                                                                              *
//* Note that only the contents of the subdirectories below the base directory   *
//* will be reported here. The top-level non-directory files will not be         *
//* reported because the directory-scanning threads all will have completed      *
//* (and this thread will exit) BEFORE the top-level files are added to list.    *
//* This should not be a problem because the true contents of the path textbox   *
//* will be restored immediately before return.                                  *
//*                                                                              *
//* All of the following scenarios have been observed during testing:            *
//*  a) If the scan is finished before the first call to getActiveThreads(),     *
//*     then we do not report interim progress and just return immediately.      *
//*  b) If the scan encountered an error, then we wait until caller has          *
//*     cleaned up the mess before returning.                                    *
//*  c) If on entry the scan has not yet started, it means that the allocation   *
//*     and launch of the scan threads is taking considerable time. We must      *
//*     ensure the user does not regain control of the user interface before     *
//*     all data are captured. This condition is caught by the uiReport() loop.  *
//********************************************************************************

void FileDlg::rcdProgmon ( DispData* ddPtr, bool* active )
{
   *active = true ;     // signal to caller that we are active

   chrono::duration<short, std::milli>aWhile( 100 ) ; // pause before reporting
   chrono::duration<short, std::milli>aMoment( 50 ) ; // sleep interval
   dtbmData tbMsg ;              // message output
   gString  gsOut, gsi ;         // message formatting
   UINT     fCount = ZERO, fc ;  // number of files scanned
   UINT64   fBytes = ZERO, fb ;  // total file size

   //* Wait for either the scan to be initiated OR a signal *
   //* that the scan is unnecessary (or system error).      *
   //* Note that in many cases, a single iteration is enough*
   //* time for the scan to be completed, so there will be  *
   //* no need to display progress.                         *
   do
   { this_thread::sleep_for( aWhile ) ; }
   while ( ! (ddPtr->uiReport()) ) ;

   //* Report on scan progress *
   while ( (ddPtr->getActiveThreads ()) > ZERO )
   {
      ddPtr->getData ( fc, fb ) ;            // get progress
      if ( fc != fCount || fb != fBytes )    // if progress made, report it
      {
         fCount = fc ;
         fBytes = fb ;
         gsi.formatInt ( fCount, 13, true ) ;
         gsOut.compose( "    files scanned: %S", gsi.gstr() ) ;
         gsOut.copy( tbMsg.textData, gsALLOCDFLT ) ;
         this->dPtr->DisplayTextboxMessage ( this->pIndex, tbMsg ) ;
         this_thread::sleep_for( aMoment ) ; // wait a moment
      }
   }

   //* Clear the progress message. *
   tbMsg.textData[ZERO] = NULLCHAR ;
   this->dPtr->DisplayTextboxMessage ( this->pIndex, tbMsg ) ;

}  //* End rcdProgmon() *

//*************************
//*     GetUserInfo       *
//*************************
//********************************************************************************
//* Returns a copy of user information: uid, gid, user name, etc. stored in      *
//* userInfo structure.                                                          *
//*                                                                              *
//* Input  : (by reference) UserInfo-class object to receive data                *
//*                                                                              *
//* Returns: true if successful, false if data unavailable                       *
//********************************************************************************

bool FileDlg::GetUserInfo ( UserInfo& uInfo )
{

   uInfo = this->userInfo ;
   return userInfo.goodData ;

}  //* End GetUserInfo() *

//*************************
//*     SelectFile        *
//*************************
//********************************************************************************
//* 'Select' i.e. set the the color attribute for the highlighted file to        *
//* indicate that the file has been selected for processing.                     *
//* 'Selected' files will be operated on by the next Copy/Cut etc. operation     *
//*                                                                              *
//* Input  : all : (optional, false by default)                                  *
//*                if true, select all files in directory                        *
//*                   (highlight position unchanged)                             *
//*                if false, select highlighted file and move highlight          *
//*                   in specified direction                                     *
//*          dir : (optional, 1 by default - referenced only if all==false)      *
//*                after highlighted file is selected:                           *
//*                if  1, then move highlight down by one (default)              *
//*                if  0, then do not move highlight                             *
//*                if -1, then move highlight up by one                          *
//*                                                                              *
//* Returns: true if successful, else false                                      *
//********************************************************************************
//* Programmer's Note: The most difficult part of moving from displayed-files    *
//* to selected-files to clipboard-files is handling the 'hidden' files.         *
//* 'Hidden' files may or may not be displayed; however, if a subdirectory tree  *
//* containing hidden files is 'selected', it means selecting ALL files in that  *
//* tree, INCLUDING hidden files. When 'selected' files and subdirectory trees   *
//* are placed onto the clipboard, then obviously the hidden files go with       *
//* them.                                                                        *
//*                                                                              *
//* Note that if hidden files are not displayed, it is not possible for the      *
//* user to 'select' a top-level hidden file because it can't be seen.           *
//* It is only hidden files in lower-level subdirectories that are selected      *
//* with all other files in that subdirectory. This could cause an anomalous     *
//* situation - but what's a mutha' to do?                                       *
//* For a recursive cut-and-paste situation, clearly the hidden files must be    *
//* moved; however, for a recursive copy-and-paste situation the choices         *
//* aren't so clear. These scenarios must be handled during the paste-clipboard  *
//* operation because FMgr class methods don't know whether it is a copy or      *
//* cut operation.                                                               *
//********************************************************************************

bool FileDlg::SelectFile ( bool all, short dir )
{
   bool  result = false ;

   //* There must be something to 'select' in the directory *
   if ( this->fileCount > ZERO )
   {
      //* If files from this directory are in the base node of the clipboard   *
      //* tree AND user is selecting a file, user is probably confused. Clear  *
      //* the existing selections and clear the clipboard before selecting the *
      //* new file. (this is not a perfect test)                               *
      //* see notes in ClearClipboard() method)                                *
      if ( (this->TargDirEqualsClipDir ( false )) != false )
      {
         this->ClearClipboard () ;     // release files from clipboard
         this->DeselectFile ( true ) ; // de-select all selections
      }

      //* If selecting only the highlighted file, *
      //* start there, else start at ZERO.        *
      UINT  i = (all == false ? this->hIndex : ZERO ) ;
      for ( ; i < this->fileCount ; i++ )
      {
         //* If file is not already 'selected' *
         if ( !(this->colorData[i] & this->selectAttr) )
         {
            //* If file selected is a directory name, count * 
            //* all bytes in child files and directories.   *
            if ( (this->deList[i])->fType == fmDIR_TYPE )
            {
               this->colorData[i] |= this->selectAttr ;   // set the attribute
               //* Scan the contents of the specified directory to the bottom  *
               //* of the tree node. Add the file count and file size for all  *
               //* files in this node to the selection totals.                 *
               //* An empty directory has only its own size.                   *
               UINT     subdirFiles ;
               UINT64   subdirBytes ;
               this->fmPtr->TreeNodeSummary ( this->deList[i], subdirFiles, subdirBytes ) ;
               ++this->selInfo.Count ;
               this->selInfo.Size += this->deList[i]->fBytes ;
               this->selInfo.subCount += subdirFiles - 1 ;
               this->selInfo.subSize += subdirBytes - this->deList[i]->fBytes ;
               result = true ;
            }
            //* Else, is a non-directory file name *
            else
            {
               this->colorData[i] |= this->selectAttr ;   // set the attribute
               ++this->selInfo.Count ;
               this->selInfo.Size += (this->deList[i])->fBytes ;
               result = true ;
            }
            //* If selecting only one file, move        *
            //* highlight to next display item (if any) *
            if ( all == false )
            {
               if ( dir == 1 )
                  this->NextDisplayItem () ;
               else if ( dir == (-1) )
                  this->PrevDisplayItem () ;
               else  // ( dir == ZERO )
                  ;  // do nothing
            }
         }
         else
         {
            //* User is re-selecting an already-selected file                  *
            //* (this is actually an application error since this method     ) *
            //* (should not have been called for a 'selected' file.          ) *
            if ( all == false )
            {
               //* Highlight is on indicated file., De-select it.              *
               //* If it is on the clipboard, clear the clipboard also.        *
               this->DeselectFile () ;
            }
         }
         if ( all == false )                 // hilighted file processed
            break ;
      }
      //* Update the status control and re-draw *
      //* file data with new color attributes   *
      this->DisplayStatus () ;
      this->dPtr->RefreshScrollextText ( this->fIndex ) ;
   }
   return result ;

}  //* End SelectFile() *

//*************************
//*    DeselectFile       *
//*************************
//******************************************************************************
//* Reset the 'selected' state (color attribute) for the file under the        *
//* cursor indicating that it IS NOT "selected"                                *
//*                                                                            *
//* Clipboard contents are not affected.                                       *
//*                                                                            *
//* Input  : all : (optional, false by default)                                *
//*                if true, de-select all files in directory                   *
//*                   (highlight position unchanged)                           *
//*                if false, de-select highlighted file and move highlight     *
//*                   in specified direction                                   *
//*          dir : (optional, 1 by default - referenced only if all==false)    *
//*                after highlighted file is de-selected:                      *
//*                if  1, then move highlight down by one (default)            *
//*                if  0, then do not move highlight                           *
//*                if -1, then move highlight up by one                        *
//*                                                                            *
//* Returns: true if successful, else false                                    *
//******************************************************************************

bool FileDlg::DeselectFile ( bool all, short dir )
{
bool  result = false ;

   //* If there is at least one 'selected' file in the list *
   if ( IsSelected ( true ) )
   {
      if ( all != false )
      {
         this->ResetFileDisplayColors () ;   // reset display attributes
         this->selInfo.Count = this->selInfo.subCount = ZERO ;
         this->selInfo.Size  = this->selInfo.subSize  = ZERO ;
         //* Update the status control and re-draw *
         //* file data with new color attributes   *
         this->DisplayStatus () ;
         this->dPtr->RefreshScrollextText ( this->fIndex ) ;
         result = true ;                     // return success
      }
      else                                   // de-select the highlighted file only
      {
         if ( this->colorData[this->hIndex] & this->selectAttr )  // if file is 'selected'
         {
            //* If file being de-selected is on the clipboard, clear ALL       *
            //* selections because they will no longer reflect reality.        *
            //* Note that the clipboard data are unchanged, only the visual    *
            //* indicators will be removed.                                    *
            if (    (this->colorData[this->hIndex] == this->copyColor) 
                 || (this->colorData[this->hIndex] == this->cutColor) )
            {
               this->ResetFileDisplayColors () ;
               this->selInfo.Count = this->selInfo.subCount = ZERO ;
               this->selInfo.Size  = this->selInfo.subSize  = ZERO ;
            }
            else
            {
               //* If file being de-selected is a directory name, * 
               //* count all bytes in child files and directories.*
               if ( (this->deList[this->hIndex])->fType == fmDIR_TYPE )
               {
                  this->colorData[this->hIndex] &= ~this->selectAttr ; // reset the attribute
                  //* Scan the contents of the specified directory to the      *
                  //* bottom of the tree node. Subtract the file count and     *
                  //* file size for all files in this node from the selection  *
                  //* totals                                                   *
                  UINT     subdirFiles ;
                  UINT64   subdirBytes ;
                  this->fmPtr->TreeNodeSummary ( this->deList[this->hIndex], 
                                                 subdirFiles, subdirBytes ) ;
                  --this->selInfo.Count ;
                  this->selInfo.Size -= (this->deList[this->hIndex])->fBytes ;
                  this->selInfo.subCount -= subdirFiles - 1 ;
                  this->selInfo.subSize -= 
                              subdirBytes - this->deList[this->hIndex]->fBytes ;
               }
               //* Else, is a non-directory file name *
               else
               {
                  //* Reset the color attribut to un-selected,       *
                  //* decrement the number of files selected, and    *
                  //* reduce the total bytes of selected files       *
                  this->colorData[this->hIndex] &= ~this->selectAttr ;
                  --this->selInfo.Count ;
                  this->selInfo.Size -= (this->deList[this->hIndex])->fBytes ;
               }
            }
            //* Move highlight to next/previous display item (if any) *
            if ( dir == 1 )
               this->NextDisplayItem () ;
            else if ( dir == (-1) )
               this->PrevDisplayItem () ;
            else  // ( dir == ZERO )
               ;  // do nothing
         }
         else
         {
            //* User is 'deselecting' an un-selected file.                     *
            //* (this is actually an application error since this method     ) *
            //* (should not have been called for a 'un-selected' file.       ) *
            if ( all == false )
            {
               //* Highlight is on indicated file., Select it.                 *
               //* Standard clipboard rules apply.                             *
               this->SelectFile () ;
            }
         }
         this->DisplayStatus () ;            // update status display
         this->dPtr->RefreshScrollextText ( this->fIndex ) ;
         result = true ;                     // return success
      }
   }

   return result ;

}  //* End DeselectFile() *

//*************************
//*      IsSelected       *
//*************************
//******************************************************************************
//* Returns 'true' if highlighted file has been 'selected'.                    *
//* Alternatively, if 'all' parameter is true, returns 'true' if ANY file in   *
//* the list is selected.                                                      *
//*                                                                            *
//* Input  : all (optional, false by default): if true, test whether           *
//*            ANY files in directory have been 'selected'.                    *
//*                                                                            *
//* Returns: true if currently-highlighted file is 'selected', else false      *
//******************************************************************************

bool FileDlg::IsSelected ( bool all )
{
bool  result = false ;

   if ( this->selInfo.Count > ZERO )
   {
      if ( all == false )
         result = (bool)(this->colorData[this->hIndex] & this->selectAttr) ; 
      else
         result = true ;
   }

   return result ;

}  //* End IsSelected() *

//*************************
//*      IsSelected       *
//*************************
//******************************************************************************
//* Returns 'true' if highlighted file has been 'selected'.                    *
//* This method provides the data necessary for caller to perform group file   *
//* selections. For all other needs, call the previous IsSelected() method.    *
//*                                                                            *
//* Input  : currItem  : (by reference) receives index of highlighted item     *
//*          totalItems: (by reference) receives number of items in list       *
//*          prevSel   : (by reference)                                        *
//*                      set to 'true' if currItem > ZERO AND item at          *
//*                      (currItem-1) is selected                              *
//*                      else set to 'false'                                   *
//*          nextSel   : (by reference)                                        *
//*                      set to 'true' if currItem < (titalItems-1) AND item   *
//*                      at (currItem+1) is selected                           *
//*                                                                            *
//* Returns: true if currently-highlighted file is 'selected', else false      *
//******************************************************************************

bool FileDlg::IsSelected ( short& currItem, short& totalItems, 
                           bool& prevSel, bool& nextSel )
{
   currItem = this->hIndex ;
   totalItems = this->fileCount ;
   prevSel = nextSel = false ;
   if ( currItem > ZERO )
      prevSel = (bool)(this->colorData[currItem - 1] & this->selectAttr) ;
   if ( currItem < (totalItems - 1) )
      nextSel = (bool)(this->colorData[currItem + 1] & this->selectAttr) ;

   return ( this->IsSelected () ) ;

}  //* End IsSelected() *

//*************************
//*   UpdateSortOptions   *
//*************************
//******************************************************************************
//* Set file-sorting options according to the following data-member values.    *
//*         copt.sortOption: file sort order                                   *
//*      copt.caseSensitive: whether sort is case sensitive                    *
//*         copt.showHidden: whether 'hidden' files will be displayed          *
//*                                                                            *
//* Input  : copt  : (by reference)                                            *
//*                                                                            *
//* Returns: 'true'  if options changed, else 'false'                          *
//*          This method DOES NOT redraw the display.                          *
//******************************************************************************

bool FileDlg::UpdateSortOptions ( ConfigOptions& copt )
{
   bool omods = false ;

   if ( copt.showHidden != this->cfgOptions.showHidden )
   {
      if ( (this->fmPtr->ShowHiddenFiles ( copt.showHidden )) == OK )
      {
         this->cfgOptions.showHidden = copt.showHidden ;
         omods = true ;
      }
   }
   if ( copt.caseSensitive != this->cfgOptions.caseSensitive )
   {
      if ( (this->fmPtr->CaseSensitiveAlphaSort ( copt.caseSensitive )) == OK )
      {
         this->cfgOptions.caseSensitive = copt.caseSensitive ;
         omods = true ;
      }
   }
   if ( copt.sortOption != this->cfgOptions.sortOption )
   {
      if ( (this->fmPtr->SetSortOption ( copt.sortOption )) == OK )
      {
         this->cfgOptions.sortOption = this->fmPtr->GetSortOption () ;
         omods = true ;
      }
      copt.sortOption = this->cfgOptions.sortOption ;
   }
   return omods ;

}  //* End UpdateSortOptions() *

//*************************
//*     ScanFromRoot      *
//*************************
//********************************************************************************
//* Allow/disallow scan of full directory tree when CWD is the root directory.   *
//* This is a pass-through method to the FMgr-class method of the same name.     *
//*                                                                              *
//* Input  : enable     : if 'true',  recursive tree scan from root is enabled   *
//*                       if 'false', recursive tree scan from root is disabled  *
//*          reportOnly : (optional, 'false' by default)                         *
//*                       if 'false', internal flag is set/reset as indicated    *
//*                                   by 'enable' parameter                      *
//*                       if 'true',  state of internal flag is is not           *
//*                                   modified, only reported                    *
//*                                                                              *
//* Returns: current state of recursive-scan flag                                *
//********************************************************************************

bool FileDlg::ScanFromRoot ( bool enable, bool report )
{

   return ( (this->fmPtr->ScanFromRoot ( enable, report )) ) ;

}  //* End ScanFromRoot() *

//*************************
//*      UpdateMouse      *
//*************************
//******************************************************************************
//* Update the 'configOpt.mouseEnabled' flag indicating whether mouse is active*
//* NOTE: This flag is not currently referenced within the FileDlg class, but  *
//*       keep it synched with parent class, just in case.                     *
//*                                                                            *
//* Input  : mEnabled : 'true' if mouse currently enabled, else 'false'        *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void FileDlg::UpdateMouse ( bool mEnabled )
{

   this->cfgOptions.enableMouse = mEnabled ;

}  //* End UpdateMouse() *

//*************************
//*    CreateDirectory    *
//*************************
//******************************************************************************
//* Create a new sub-directory in the currently-displayed directory.           *
//*                                                                            *
//* Input  : dirName: name of directory to be created (not filespec)           *
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
//******************************************************************************
//* Programmer's Note: Called method knows whether to use kernel tools or      *
//* 'gio' utility to create a directory on the target filesystem.              *
//******************************************************************************

short FileDlg::CreateDirectory ( const gString& dirName )
{
   short status = this->fmPtr->CreateSubDirectory ( dirName.ustr() ) ;
   if ( status != OK )  // (the low-level method returns 'errno' on error)
      status = ERR ;
   return status ;

}  //* End CreateDirectory() *

//*************************
//*      DeleteFile       *
//*************************
//******************************************************************************
//* Delete the specified file.                                                 *
//* NOTE: This method is for removing unneeded application temp files and      *
//*       other stuff the user doesn't need to know about.                     *
//*       File deletions requested by user must go through a more formal       *
//*       sequence.                                                            *
//*                                                                            *
//* Input  : full path/filename of target to be deleted                        *
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
//******************************************************************************

short FileDlg::DeleteFile ( const gString& trgPath )
{
   short    status = ERR ;          // return value
   fmFType  fType ;                 // file-type of target
   if ( (this->TargetExists ( trgPath, fType )) && (fType == fmREG_TYPE) )
   {
      //* If filespec identifies a file on a virtual (MTP/GVfs)       *
      //* filesystem, use the 'gio remove' command to delete the file.*
      //* Note that 'gio remove' silently overrides write protection, *
      //* so don't call for delete if you don't mean it.              *
      if ( (this->fmPtr->isGvfsPath ( trgPath )) )
         status = this->fmPtr->DeleteFile_gvfs ( trgPath.ustr() ) ;
      //* Else use the standard 'unlink' function   *
      //* which calls the kernel to delete the file.*
      else
         status = this->fmPtr->DeleteFile ( trgPath.ustr() ) ;

      if ( status != OK )  // (the low-level methods return 'errno' on error)
         status = ERR ;
   }
   return status ;

}  //* End DeleteFile() *

//*************************
//*       GetPath         *
//*************************
//******************************************************************************
//* Return a copy of the path string for the currently-displayed directory.    *
//*                                                                            *
//* Input  : currPath: (by reference) gString object to receive path string    *
//*                                                                            *
//* Returns: OK                                                                *
//******************************************************************************

short FileDlg::GetPath ( gString& currPath )
{
   currPath = this->currDir ;
   return OK ;

}  //* End GetPath() *

//*************************
//*       GetPath         *
//*************************
//******************************************************************************
//* Return a copy of the path string for the currently-displayed directory.    *
//* Also returns read- and write-access flags.                                 *
//*                                                                            *
//* Input  : currPath: (by reference) gString object to receive path string    *
//*          rPerm   : (by reference) receives read-access flag                *
//*          wPerm   : (by reference) receives write-access flag               *
//*                                                                            *
//* Returns: OK                                                                *
//******************************************************************************

short FileDlg::GetPath ( gString& currPath, bool& rPerm, bool& wPerm )
{
   currPath = this->currDir ;
   tnFName tnf ;
   this->fmPtr->GetFileStats ( tnf, currPath ) ;
   rPerm = tnf.readAcc ;
   wPerm = tnf.writeAcc ;
   return OK ;

}  //* End GetPath() *

//*************************
//*     TargetExists      *
//*************************
//******************************************************************************
//* Determine whether the specified target file exists.                        *
//* (public-access method,  see also private methods)                          *
//*                                                                            *
//* Input  : fPath   : full path/filename specification of file to be tested   *
//*          fType   : if target exists, initialized to target fileType        *
//*                    else fmUNKNOWN_TYPE                                     *
//*                                                                            *
//* Returns: true if target file exists, else false                            *
//******************************************************************************

bool FileDlg::TargetExists ( const gString& fPath, fmFType& fType )
{
   tnFName fStats ;           // file stats
   bool fileExists = false ;  // return value

   fType = fmUNKNOWN_TYPE ;   // initialize caller's parameter

   if ( (fileExists = this->TargetExists ( fPath.ustr(), fStats )) )
   {
      fType = fStats.fType ;
   }
   return fileExists ;

}  //* End TargetExists() *

//*************************
//*      GetStats         *
//*************************
//******************************************************************************
//* Returns a copy of the tnFName class object containing data for the         *
//* currently highlighted file.                                                *
//*                                                                            *
//* Input  : fn   : (by reference, initial values ignored)                     *
//*                 receives a copy of file stats for highlighted file         *
//*                                                                            *
//* Returns: index of highlighted filename, or ERR if empty directory          *
//******************************************************************************
//* Programmer's Note: If the main thread is inside the ScrollThruFileList()   *
//* method, then the 'hIndex' data member will be stale. For this reason, we   *
//* refresh 'hIndex' here.                                                     *
//*                                                                            *
//******************************************************************************

short FileDlg::GetStats ( tnFName& fn )
{
   short status = ERR ;       // return value

   if ( this->fileCount > ZERO  )
   {
      status = this->hIndex = this->dPtr->GetScrollextSelect ( this->fIndex ) ;
      fn = *this->deList[this->hIndex] ;
   }
   return status ;

}  //* End GetStats() *

//*************************
//*   GetExpandedStats    *
//*************************
//******************************************************************************
//* Initializes caller's ??????? class object with data for the currently      *
//* highlighted file.                                                          *
//*                                                                            *
//* Input  : es   : caller's tnFName object (by reference)                     *
//*                                                                            *
//* Returns: OK if there is a highlighted filename, else ERR                   *
//******************************************************************************

#if 0    // NOT YET IMPLEMENTED
short FileDlg::GetExpandedStats ( ???????& es )
{
short    success = ERR ;

   if ( this->fileCount > ZERO  )
   {
      fn = *this->deList[this->hIndex] ;
      success = OK ;
   }
   return success ;

}  //* End GetExpandedStats() *
#endif   // NOT YET IMPLEMENTED

               //**********************************************
               //*   Private  Methods of the FileDlg class    *
               //**********************************************
//*************************
//*     DisplayFiles      *
//*************************
//******************************************************************************
//* Display the file list in the file display control window.                  *
//*                                                                            *
//*                                                                            *
//* Input  : withHilight: (optional, true by default)                          *
//*                       if true, draw data with highlight on current         *
//*                       item, else draw data without highlight               *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void FileDlg::DisplayFiles ( bool withHilight )
{
   ssetData    sData ;                    // container for display data

   //* If there are files to display *
   if ( this->fileCount > ZERO )
   {
      this->hIndex = ZERO ;               // highlight first file in list

      //* Send display data to our file-display control *
      sData = 
      {
         (const char**)this->textData,
         this->colorData,
         int(this->fileCount),   // (narrowing, UINT to int is acceptable here)
         ZERO,
         withHilight
      } ;
      dPtr->SetScrollextText ( fIndex, sData ) ;
   }  
   else  //* Else, no files in directory - display our empty line *
   {
      sData = 
      {
         (const char**)this->EmptyStr,
         this->EmptyAttr,
         1,
         ZERO,
         withHilight
      } ;
      dPtr->SetScrollextText ( fIndex, sData ) ;
   }

}  //* End DisplayFiles() *

//*************************
//*      ProcessCWD       *
//*************************
//******************************************************************************
//* Display a message that we are in the process of reading and processing     *
//* the information contained in a directory. This is done to give the user    *
//* something to look at while processing large directories which may require  *
//* several seconds to process, and while processing directories from very     *
//* slow devices such as CD-ROM, DVD, and other externally mounted devices.    *
//*                                                                            *
//* Input  : clear : (optional, 'false' by default)                            *
//*                  if 'false', display the message                           *
//*                  if 'true',  clear the message                             *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void FileDlg::ProcessCWD ( bool clear )
{
   dtbmData  data( clear ? "" : " " ) ;
   if ( ! clear )
   {
      //* Clear the file-statistics control's display *
      this->dPtr->DisplayTextboxMessage ( this->sIndex, data ) ;

      //* Display a friendly message in the Path-display control *
      data = "    Processing..." ;
      this->dPtr->DisplayTextboxMessage ( this->pIndex, data ) ;
   }
   else
   {  //* Erase previous messages *
      this->dPtr->DisplayTextboxMessage ( this->sIndex, data ) ;
      this->dPtr->DisplayTextboxMessage ( this->pIndex, data ) ;
   }

}  //* End ProcessCWD() *

//*************************
//*     IsFNameChar       *
//*************************
//******************************************************************************
//* Test the specified character to determine whether it may be used as a      *
//* character in a filename or directory name.                                 *
//*                                                                            *
//* Input  : character to test (wchar_t)                                       *
//*                                                                            *
//* Returns: true if acceptable, else false                                    *
//******************************************************************************

bool FileDlg::IsFNameChar ( wchar_t ch )
{

   return this->fmPtr->IsFNameChar ( ch ) ;

}  //* End IsFNameChar() *

//**************************
//* ResetFileDisplayColors *
//**************************
//******************************************************************************
//* Restore the colors for all display strings to the color associated with    *
//* the file type. This is useful when the pending operation has finished,     *
//* or when the pending operation has been cancelled.                          *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
//******************************************************************************

short FileDlg::ResetFileDisplayColors ( void )
{
UINT        tIndex ;
short result = OK ;

   for ( tIndex = ZERO ; tIndex < this->fileCount ; tIndex++ )
      this->colorData[tIndex] = this->ftColor[this->deList[tIndex]->fType] ;

   return result ;

}  //* End ResetFileDisplayColors() *

//*************************
//*    TrimPathString     *
//*************************
//********************************************************************************
//* Shorten the path string so it will fit into the available display space.     *
//* This is done by replacing one or more directory names with the ellipsis.     *
//*     Example: /home/mongoose/Applications/CoffeeCups   --becomes--            *
//*              /home/mongoose/.../CoffeeCups                                   *
//*                                                                              *
//* Input  : gsPath  : gString object (by reference) containing path string      *
//*          maxWidth: maximum number of columns in display string               *
//*                                                                              *
//* Returns: OK if successful, else ERR                                          *
//********************************************************************************
//* Programmer's Note: Hidden-directory names (names beginning with a PERIOD)    *
//* may exist in the path, but are correctly handled.                            *
//*                                                                              *
//* Limits:                                                                      *
//* a) If maxWidth < filenameWidth, we fail.                                     *
//* b) If maxWidth < (filenameWidth + baseDirWidth), we fail.                    *
//* c) If maxWidth < (filenameWidth + eWidth + baseDirWidth), we fail.           *
//* d) If we remove 'failsafe' directories, and the result is _still_ too wide,  *
//     we fail. (this is very unlikely)                                          *
//********************************************************************************

short FileDlg::TrimPathString ( gString& gsPath, short maxWidth )
{
   short status = OK ;     // return value

#define DEBUG_TEMP_TPS (0)
#if DEBUG_TEMP_TPS != 0  // TEMP TEMP TEMP
// CZONE - Note Displayed path in std mode may have initial double slash: "//usr/"
//       - possibly an artifact from ascending through a very long directory name. 
// Check path when repeatedly moving to parent directory.
// Ex: /var/lib/flatpak/app/org.videolan.VLC/...
// May also be related to exit from Tree-view to file view.
// Cured by switching between single and dual-win modes.
// See DisplayDirPath() and 'this->currDir'
// See TrimPathString()
// a) if trim causes the "//"
// b) if 'gsPath' input contains the "//"
if ( (gsPath.find( L"//" )) >= ZERO ) { this->dPtr->UserAlert ( 3 ) ; sleep( 1 ) ; }
#endif   // TEMP TEMP TEMP
   if ( (gsPath.gscols()) > maxWidth )
   {
      gString eString = "/...",
              pString, fString ;
      const wchar_t* wPtr ;
      int   eWidth = eString.gscols(),    // column count for ellipsis string
            nWidth = ZERO,                // column count for filename string
            pWidth = ZERO,                // column count for path in progress
            xWidth = ZERO,                // column count in progress
            bLen = gsPath.find( fSLASH, 1), // length of base directory string
            pi,                           // path length in progress
            failsafe = 15 ;               // when zero, break out of loop

      this->fmPtr->ExtractPathname ( pString, gsPath ) ;
      pWidth = pString.gscols() ;
      this->fmPtr->ExtractFilename ( fString, gsPath ) ;
      fString.insert( fSLASH ) ;
      nWidth = fString.gscols() ;
      xWidth = pWidth + nWidth ;

      while ( xWidth > maxWidth )
      {
         wPtr = pString.gstr( pi ) ;      // pointer to wide string
         --pi ;                           // reference the NULLCHAR
         while ( (wPtr[pi] != fSLASH) && (pi > bLen) )
            --pi ;
         pString.limitChars( pi ) ;
         pWidth = pString.gscols() ;

         xWidth = pWidth + eWidth + nWidth ;
         if ( (pi == bLen) || (--failsafe <= ZERO) )
            break ;
      }

      //* If loop not aborted, create the trimmed filespec.*
      if ( xWidth <= maxWidth )
      {
         gsPath.compose( L"%S%S%S", pString.gstr(), eString.gstr(), fString.gstr() ) ;
      }
      else
      {  // NOTE: This simply truncates the string, so even if caller ignores
         //       the return value, the string will still fit in the field.
         //       See above for notes on trim limits.
         gsPath.limitCols( maxWidth ) ;
         status = ERR ;
      }
#if DEBUG_TEMP_TPS != 0  // TEMP TEMP TEMP
if ( (gsPath.find( L"//" )) >= ZERO ) { this->dPtr->UserAlert ( 5 ) ; }
#endif   // TEMP TEMP TEMP
   }
   return status ;

}  //* End of TrimPathString() *

//***********************
//* Get_FileDlg_Version *
//***********************
//******************************************************************************
//* Returns a pointer to FileDlgVersion, the FileDlg class version number.     *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: (const char*) pointer to version string                           *
//******************************************************************************

const char* FileDlg::Get_FileDlg_Version ( void )
{

   return FileDlgVersion ;

}  //* End Get_FileDlg_Version() *

//*************************
//*   Get_FMgr_Version    *
//*************************
//********************************************************************************
//* Returns a pointer to FMgr-class version string.  This is a pass-through      *
//* to the FMgr::GetFMgr_Version() method, to which the mainline code has        *
//* no access.                                                                   *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: (const char*) pointer to version string                             *
//********************************************************************************

const char* FileDlg::Get_FMgr_Version ( void )
{

   return this->fmPtr->Get_FMgr_Version () ;

}  //* End Get_FMgr_Version() *

//*************************
//*     InvokeDialog      *
//*************************
//********************************************************************************
//* FOR DEBUGGING ONLY                                                           *
//* Used to invoke private, FileDlg dialog windows which are under development.  *
//* Unless a particular dialog is currently under construction, this method      *
//* should always do nothing but return OK.                                      *
//*                                                                              *
//* May also be used for development of FMgr-class methods.                      *
//* (FMgr-class methods are not directly available to the application layer.)    *
//*                                                                              *
//* Input  : tstnum: number of test to be invoked                                *
//*                                                                              *
//* Returns: whatever is returned from the dialog itself                         *
//********************************************************************************

short FileDlg::InvokeDialog ( short tstnum )
{
   short status = OK ;

   #if 0    // DIALOG WINDOW TESTING ONLY
   this->dPtr->SetDialogObscured () ;

   if ( tstnum == 1 )
   {
   }
   else if ( tstnum == 2 )
   {
   }
   else if ( tstnum == 3 )
   {
   }
   else if ( tstnum == 4 )
   {
   }
   else if ( tstnum == 5 )
   {
   }
   else if ( tstnum == 6 )
   {
   }

   this->dPtr->RefreshWin () ;
   #endif   // -------------------

   return status ;

}  //* End InvokeDialog() *

//*************************
//*     fmgrDebugMsg      *
//*************************
//********************************************************************************
//* Callback method used by FMgr class to display debugging messages             *
//*                                                                              *
//* Input  : msg  : UTF-8 message to be displayed                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

static void fmgrDebugMsg ( const char* msg )
{
   if ( fmgrDbPtr != NULL )
   {
      fmgrDbPtr->DebugMsg ( msg ) ;
   }
}  //* End fmgrDebugMsg() *

//*************************
//*                       *
//*************************
//********************************************************************************
//*                                                                              *
//*                                                                              *
//*                                                                              *
//* Input  :                                                                     *
//*                                                                              *
//* Returns:                                                                     *
//********************************************************************************


