//********************************************************************************
//* File       : FileDlgBackup.cpp                                               *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2005-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in FileMangler.hpp         *
//* Date       : 11-May-2025                                                     *
//* Version    : (see FileDlgVersion string in FileDlg.cpp)                      *
//*                                                                              *
//* Description: This module of the FileDlg class contains the sub-dialog and    *
//* processing for performing a file backup operation.                           *
//*                                                                              *
//* This module also contains the methods for two utility classes:               *
//*   1) WorkInProgress class   (file operation tracking)                        *
//*   2) Progbar class          (file operation reporting)                       *
//*                                                                              *
//*                                                                              *
//* Development tools: See FileMangler.cpp header.                               *
//********************************************************************************
//* Version History (most recent first):                                         *
//*   See version history in FileDlg.cpp.                                        *
//********************************************************************************
//* General Notes On Backup/Archive/Synch Operations:                            *
//* -------------------------------------------------                            *
//*                                                                              *
//* Backup/Synch operations are essentially the same operation as                *
//* PasteClipboardList(), EXCEPT that WE make all decisions about which files    *
//* will be updated without prompting the user about write-protected targets,    *
//* newer targets, rename or other exception processing. Instead, we just        *
//* count the file updates attempted and the file updates completed.             *
//*                                                                              *
//* a) Caller has verified that we have read access to the base source           *
//*    directory/directories.                                                    *
//* b) Caller has verified that we have read/write access to the base target     *
//*    directory/directories.                                                    *
//* c) If there are write-protected existing targets, we DO NOT overwrite them.  *
//* d) If we run out of space on the target filesystem, TOO BAD; all remaining   *
//*    updates will fail. We must do it this way because we don't know ahead     *
//*    of time how much free space we will need for the updates, and             *
//*    calculating the needed free space ahead of time would be complex and      *
//*    would take longer than the actual backup operation.                       *
//* e) If user does not have read permission on a source file, then the target   *
//*    will not be updated.                                                      *
//*                                                                              *
//* Logic When source and target overlap.                                        *
//* a) Source base directory MAY NOT EQUAL target base directory UNLESS the      *
//*    target is an archive file.                                                *
//* b) If the target directory tree CONTAINS the source directory tree, there    *
//*    is no conflict because the source will be copied to a different base      *
//*    position.                                                                 *
//* c) If the source directory tree CONTAINS the target directory tree, then     *
//*    again there is no conflict because the source data is a SNAPSHOT of the   *
//*    data which is placed on the clipboard before the operation begins.        *
//*    The copy operation actually occurs between the clipboard and the target,  *
//*    NOT directly between source and target.                                   *
//*                                                                              *
//********************************************************************************


//* Header Files *
#include "GlobalDef.hpp"
#include "FileDlg.hpp"

//*******************************
//* Local Definitions and Data. *
//*******************************

#if ENABLE_DEBUGGING_CODE != 0
#define enableDBBAK (0)    // set to non-zero to enable debugging output to log file
#define enableDBNEED (0)   // set to non-zero to write 'needs-update' test to log
#define enableDBBAR (0)    // set to non-zero to enable debug of progress bar
#define enableDBSIM (0)    // set to non-zero to simulate file updates and simplify log file
#if enableDBBAK != 0
static const char* debugHDR = "DEBUG:" ;// logfile record header for debug
static gString dbgs ;                   // for formatting debugging messages
#endif   // enableDBBAK
#endif   // ENABLE_DEBUGGING_CODE

//* Indicates the type of record written to the Backup/Synch log file.*
//* See bkWriteLogfile() for more information.                        *
enum logRec : short
{
   lrecHDR = ZERO,// Header record (or debugging entry) i.e. mandatory write
   lrecSUM,       // Summary record
   lrecDIR,       // Directory target record
   lrecUPD,       // File update attempted (updated or update error)
   lrecERR,       // File errors, access, unsupported type, type mismatch, etc.
   lrecSKP,       // File skipped (no update needed)
} ;

//* Log file record headers *
static const short     fileNOINDENT = 5 ;           // for reporting directory errors
static const char* const fileINDENT = "     ▬▬▬▶" ; // indent non-directory file records
static const char* const fileDIRECT = "DIR:" ;      // directory record
static const char* const fileSKIPIT = "     SKP:" ; // non-error file-not-processed
static const char* const fileACCESS = "     ACC:" ; // access-error file-not-processed
static const char* const fileCOPYER = "     ERR:" ; // copy error
static const char* const fileTYPEER = "     TYP:" ; // source/target filetype mismatch

static const char* const scanTemplate1 = 
                     "     (Target does not exist. %S files need update.)" ;
static const char* const userAbortMsg = 
                     "Operation Aborted: Not all files were processed." ;

//* Verbose 'tar' output presents each file processed in a format similar to   *
//* the output of 'ls' built-in command. To parse the captured output of the   *
//* operation, this list of the letters representing each of the Linux/UNIX    *
//* filetypes is used to identify a record in the log as a file record and not *
//* an error message.                                                          *
static const UINT64 minBLOCK = 4096 ;  // assumed block size for target filesystem
static const short ftypeCOUNT = 13 ;   // number of unique file types
static const char  fileTypes[ftypeCOUNT] =
{ '-', 'd', 'l', 'p', 's', 'b', 'c', 'C', 'D', 'M', 'n', 'P', '?' } ;

//*************************
//* Non-member Prototypes *
//*************************
static void ParseArchiveName ( const gString& gsSrc, gString& gsFName, gString& gsFExt ) ;
static bool scanCb ( const gString& srcSpec, const gString& baseDir, TreeNode* tnPtr ) ;


//*************************
//*  BackupClipboardList  *
//*************************
//********************************************************************************
//* Perform a backup of all files and directory trees on the clipboard.          *
//* 1) Source file must be of a supported filetype:                              *
//*    'Directory', 'Regular', 'SymLink', 'FIFO'                                 *
//*    -- Symbolic links, NOT LINK TARGETS, will be processed.                   *
//*    -- Unsupported filetypes will not be included in the backup.              *
//* 2) If target file does not already exist, copy source to target.             *
//* 3) If target file already exists, then compare source and target             *
//*    modification dates.                                                       *
//*    a) If source is newer, overwrite old target file.                         *
//*    b) If target is newer or the same date, then                              *
//*       do not copy source to target.                                          *
//*    c) User must have permission to overwrite the existing target.            *
//*    d) source and target must also be of the same filetype                    *
//*                                                                              *
//* Input  : bkSet : operational parameters                                      *
//*                  'bex' is an array of exclusion records                      *
//*                  'bexCount' is the number of records in 'bex' array          *
//*                  'blog' is filespec for operational log                      *
//*          pbForce: (optional, false by default)                               *
//*                  'false' programatically determine whether the progress      *
//*                          bar will be instantiated.                           *
//*                   'true' force the progress bar to be instantiated           *
//*                    NOTE: At this time, this flag is used _only_ during       *
//*                          directory synchronizaton.                           *
//*          passTwo: (optional, false by default, for Synch only)               *
//*                   If 'pbForce' != false, then:                               *
//*                      false == first pass                                     *
//*                      true  == second pass                                    *
//*                   If 'pbForce' == false, then 'passTwo' is ignored           *
//*                                                                              *
//* Returns: number of files successfully copied or updated                      *
//*          'ftotal'   receives total number of source files                    *
//*          'fupdated' receives number of target files updated                  *
//*          'fskipped' receives number of target files requiring                *
//*                     no update (this includes 'excluded' files)               *
//*          'ferrors'  receives the number of files which were                  *
//*                     not processed due to access errors or other errors       *
//*                     OR not processed because user aborted the operation      *
//********************************************************************************
//* Programmer's Note:                                                           *
//* Conditional Compilation -- SKIP_SYMLINK_BACKUP:                              *
//* Set to non-zero to enable backup logic to skip backup of symbolic-link       *
//* files when the backup target filesystem does not support storage of          *
//* symbolic links. This prevents errors being generated due solely to           *
//* deficiencies in the target (VFAT/EXFAT) filesystem design.                   *
//* If the source file is a symbolic link, AND if the target filesystem does     *
//* not support symbolic links, then bkNeedsUpdate() will report the file as     *
//* not requiring an update i.e. "fileSKIPIT".                                   *
//*                                                                              *
//*   ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----     *
//* Notes on identification and correction of the unhandled exception bug:       *
//* ----------------------------------------------------------------------       *
//* In the loop which creates the Progbar thread, there has been an              *
//* intermittent memory-allocation error. After much experimentation, we have    *
//* set the delay in the loop to 80ms, which prevents most, if not all errors.   *
//* Keep an eye on this for a few more weeks to ensure that the error does       *
//* not return. 01-Oct-2-21                                                      *
//*                                                                              *
//*                                                                              *
//********************************************************************************

UINT FileDlg::BackupClipboardList ( BSet& bkSet, bool pbForce, bool passTwo )
{
   #define SKIP_SYMLINK_BACKUP (1)     // See note in method header

   this->SetErrorCode ( ecNOERROR ) ;  // clear any previous error code

   //* Create a temp-file log to record the operation *
   if ( bkSet.blfmt != blNONE )
      this->bkOpenLogfile ( bkSet ) ;

   //* Data passed among lower-level methods *
   pclParm  pcl( clipBoard_srcPath, this->currDir, &clipBoard[BNODE], 
                 psstSimpleCopy, false ) ;

   if ( pbForce == false )             // 'passTwo' only valid during synch
      passTwo = false ;

   //* Prepare a pbarInit-class object in case a progress bar is needed.*
   //* If Backup operation, use full width of target area.              *
   //* If Synch operation, use first or second half of target area.     *
   //* See pbManager() for more information.                            *
   int   cellCountA = (pbForce ? this->mMax / 2 + 1 : this->mMax),
         cellCountB = this->mMax - cellCountA,
         cellCount  = ((passTwo == false) ? cellCountA : cellCountB) ;
   short xOffset = this->mulX + (passTwo ? cellCountA : ZERO) ;
   pbarInit pbi = this->ProgbarInit ( cellCount, this->mulY, xOffset, true ) ;

   //* Initialize tracking for work-in-progress *
   this->wip.initialize( clipBoard[BNODE].totFiles, clipBoard[BNODE].totBytes, 
                         (pbForce == false ? PB_THRESHOLD : ZERO) ) ;

   #if ENABLE_DEBUGGING_CODE != 0 && enableDBBAK != 0
   dbgs.compose( "%s Tree Nodes:%u", debugHDR, &clipBoard_dirTrees ) ;
   this->bkWriteLogfile ( dbgs ) ;

   #if enableDBBAR != 0    // set pbar threshold to zero
   this->wip.initialize( clipBoard[BNODE].totFiles, clipBoard[BNODE].totBytes, ZERO ) ;
   #endif   // enableDBBAR
   #endif   // ENABLE_DEBUGGING_CODE && enableDBBAK

   gString gsLog, gsVal ;           // text formatting
   UINT   nodeFCount,               // total files in TreeNode
          updCount,                 // files updated within a node
          errCount ;                // errors within a node
   UINT64 nodeTBytes = ZERO ;       // total size of TreeNode (not used)
   thread *progbarThread = NULL ;   // Progbar thread pointer
   bool   trgCreated = false,       // 'true' if base dir of node created
          userAbort = false,        // 'true' if user aborts before completion
          pbtAllocated = false,     // 'true' if Progbar thread is allocated
          pbtLaunched = false ;     // 'true' if Progbar thread is launched

   //* If source data >= threshold, launch a thread *
   //* to manage the visual progress indicator.     *
   if ( clipBoard[BNODE].totBytes >= (this->wip.getThreshold()) )
   {
      //* See notes above about the size of this delay.*
      chrono::duration<short, std::milli>aMoment( 80 ) ;
      for ( short i = 3 ; i > ZERO ; --i ) // three strikes, and you're out!
      {
         if ( (progbarThread = new (std::nothrow) std::thread) != NULL )
         {
            pbtAllocated = true ;
            this_thread::sleep_for( aMoment ) ;// THIS DELAY IS PROBABLY UNNECESSARY

            //* launch the thread at target method. "Catch" any exception(s).*
            try
            {
               *progbarThread = thread( &FileDlg::pbManager, this, &pbi, &pbtLaunched ) ;
               this_thread::sleep_for( aMoment ) ;  // give the new thread time to respond
               if ( pbtLaunched != false )
               {
                  break ;
               }
               else              // thread did not set the response flag
               {
                  delete progbarThread ;              // delete the thread object
                  progbarThread = NULL ;              // reinitialize the pointer
                  pbtAllocated = false ;              // reset the flag
                  this_thread::sleep_for( aMoment ) ; // wait a moment and try again
               }
            }
            catch ( ... )        // monitor thread was created but not launched
            {
               this_thread::sleep_for( aMoment ) ;
               delete progbarThread ;              // delete the thread object
               progbarThread = NULL ;              // reinitialize the pointer
               pbtAllocated = false ;              // reset the flag
               this_thread::sleep_for( aMoment ) ; // wait a moment and try again
            }
         }     // allocated
      }        // Progbar launch loop
   }
   //* If source data < threshold, display a simple message.*
   else
      this->ProcessCWD () ;

   #if SKIP_SYMLINK_BACKUP != 0     // See note in method header
   //* Stat the target filesystem and compare filesystem type to list of       *
   //* filesystems which do not support symbolic links. (currently VFAT/EXFAT).*
   static const UINT VFAT_FSTYPE = 0x4D44 ;  // filesystem type for VFAT (MSDOS)
   static const UINT EXFAT_FSTYPE = 0x2011BAB0 ; // filesystem type for EXFAT
   UINT64 fsID ;           // target filesystem ID (ignored)
   UINT   fsType ;         // target filesystem type
   this->fmPtr->FilesystemID ( pcl.trgDir.ustr(), fsID, fsType ) ;
   if ( (fsType == VFAT_FSTYPE) || (fsType == EXFAT_FSTYPE) )
      bkSet.slSupport = false ;
   #endif   // SKIP_SYMLINK_BACKUP

   //* Is target an MTP/GVfs filesystem? *
   clipBoard_gvfsTrg = this->fmPtr->isGvfsPath () ;


   //* Copy the directory sub-trees *
   pclParm pclr( pcl.srcDir.ustr(), NULL, NULL, psstSimpleCopy, false ) ;
   for ( UINT nodeIndex = ZERO ; nodeIndex < clipBoard_dirTrees && !userAbort ; ++nodeIndex )
   {
      //* Set parameters for copying target's contents.*
      pclr.srcTree = &clipBoard[BNODE].nextLevel[nodeIndex] ;
      this->fmPtr->CatPathFname ( pclr.srcDir, pcl.srcDir.ustr(), pclr.srcTree->dirStat.fName ) ;
      this->fmPtr->CatPathFname ( pclr.trgDir, pcl.trgDir.ustr(), pclr.srcTree->dirStat.fName ) ;
      pclr.attCount = ZERO ;        // reset the attempt counter

      //* Get number of files in node *
      nodeFCount = ZERO ;
      nodeTBytes = ZERO ;
      this->fmPtr->TreeNodeSummary ( pclr.srcTree, nodeFCount, nodeTBytes ) ;

      //* Verify read access to the source node data  *
      if ( pclr.srcTree->dirStat.readAcc )
      {
         //* Verify that base directory of target node exists and if it        *
         //* doesn't, create it. Else, verify read/write access to target node.*
         //* If "scan-only" (no target updates), target directory will not be  *
         //* created.                                                          *
         if ( (this->bkVerifyTargetDir ( pclr.trgDir, trgCreated, bkSet.scanOnly )) )
         {  //* Update work-in-progress stats *
            this->wip.update( 1, pclr.srcTree->dirStat.fBytes ) ;

            //* Process contents (if any) of this node.*
            if ( bkSet.scanOnly )
               updCount = this->bkScanTnList ( pclr, bkSet ) ;
            else
               updCount = this->bkCopyTnList ( pclr, bkSet, trgCreated ) ;
            errCount = pclr.attCount - updCount ;
            bkSet.fupdated += updCount ;
            bkSet.ferrors  += errCount ;
            bkSet.fskipped += nodeFCount - (updCount + errCount + 1) ;

            //* If the target directory was created, *
            //* count it as an updated file.         *
            if ( trgCreated )
               ++bkSet.fupdated ;
            else
               ++bkSet.fskipped ;

            //* Test for user abort of the operation and if yes, add *
            //* contents of any unprocessed nodes to bkSet.fskipped. *
            if ( (userAbort = (pclr.psOption == psstABORT)) )
            {
               //* Nothing to do at this time. *
               // Programmer's Note: If user aborts the operation then the summary 
               // stats will likely be inaccurate; specifically, update count may 
               // not match the number of line items reported. This is due to the 
               // decisision-making process used in determining whether an existing 
               // directory or an empty directory is counted as an update. However, 
               // because operation was aborted, accuracy of the summary is secondary.
            }
         }
         else
         {
            //* If scan-only, count missing target directory *
            //* and all its contents as needing update.      *
            if ( bkSet.scanOnly )
            {
               bkSet.fupdated += nodeFCount ;
               this->bkWriteLogfile ( fileDIRECT, blVERBOSE, lrecDIR, false ) ;
               this->bkWriteLogfile ( pclr.trgDir, blVERBOSE, lrecDIR ) ;
               gsVal.formatInt( nodeFCount, FI_MAX_FIELDWIDTH, true ) ;
               gsLog.compose( scanTemplate1, gsVal.gstr() ) ;
               this->bkWriteLogfile ( gsLog, blVERBOSE, lrecDIR ) ;
            }
            //* Else, target access error.             *
            //* Count all files in the node as errors. *
            else
            {
               bkSet.ferrors += nodeFCount ;
               this->bkWriteLogfile ( &fileCOPYER[fileNOINDENT], bkSet.blfmt, lrecERR, false ) ;
               this->bkWriteLogfile ( pclr.trgDir, bkSet.blfmt, lrecERR ) ;
            }
         }
      }
      else
      {  //* Source access error.                   *
         //* Count all files in the node as errors. *
         // Programmer's Note: Because we don't have read access to the 
         // directory, nodeFCount is probably wrong, but that can't be helped.
         bkSet.ferrors += nodeFCount ;

         this->bkWriteLogfile ( &fileACCESS[fileNOINDENT], bkSet.blfmt, lrecERR, false ) ;
         this->fmPtr->CatPathFname ( gsLog, pclr.srcDir.ustr(), 
                                     pclr.srcTree->dirStat.fName ) ;
         this->bkWriteLogfile ( gsLog, bkSet.blfmt, lrecERR ) ;
      }
   }     // for(;;)

   if ( (this->wip.getAbort()) == false )
   {
      //* If non-directory files at this (top) level, process them *
      if ( !userAbort && (clipBoard[BNODE].tnFCount > ZERO) 
           && (clipBoard[BNODE].tnFiles != NULL) )
      {
         if ( bkSet.scanOnly )
            updCount = this->bkScanFileList ( pcl, bkSet ) ;
         else
            updCount = this->bkCopyFileList ( pcl, bkSet ) ;
         errCount = pcl.attCount - updCount ;
         bkSet.fupdated += updCount ;
         bkSet.ferrors  += errCount ;
         bkSet.fskipped += clipBoard[BNODE].tnFCount - (updCount + errCount) ;

         //* Test for user abort of the operation.*
         userAbort = (pcl.psOption == psstABORT) ;
      }
   }

   //* Refresh the file list for the target directory and display *
   //* the results of the operation (without highlight).          *
   this->RefreshCurrDir ( false ) ;

   #if ENABLE_DEBUGGING_CODE != 0 && enableDBBAK != 0
   UINT tot = this->ClipboardFiles () ;
   dbgs.compose( "%s total:%u updated:%u skipped:%u errors:%u", debugHDR, 
                 &tot, &bkSet.fupdated, &bkSet.fskipped, &bkSet.ferrors ) ;
   this->bkWriteLogfile ( dbgs ) ;
   if ( userAbort )
      this->bkWriteLogfile ( "USER ABORTED THE OPERATION." ) ;
   #endif   // ENABLE_DEBUGGING_CODE && enableDBBAK

   //* Close the log file *
   this->bkCloseLogfile () ;

   //* If a highly-caffinated user has tried to abort the operation, there    *
   //* may be multiple keystrokes waiting in the queue. Clear the input queue.*
   this->dPtr->FlushKeyInputStream () ;

   if ( pbtAllocated != false )  // if Progbar thread was allocated
   {
      if ( pbtLaunched )         // if Progbar thread was launched
      {
         this->wip.setAbort() ;  // signal thread to return (if it hasn't already)
         progbarThread->join() ; // wait for thread to terminate
      }
      delete progbarThread ;     // delete the secondary thread
   }

   return bkSet.fupdated ;

}  //* End BackupClipboardList() *

//*************************
//*     bkCopyTnList      *
//*************************
//******************************************************************************
//* Perform backup on the specified TreeNode and all its contents.             *
//* If:                                                                        *
//*   a) target directory exists AND user has read/write access,               *
//*      OR                                                                    *
//*   b) we successfully create the target directory with read/write access,   *
//* Then:                                                                      *
//*   a) recursively backup source directory's contents                        *
//*   b) backup any non-directory files at this level                          *
//*                                                                            *
//* Input  : pclParm : source and target access                                *
//*                    'srcDir'  : source directory path                       *
//*                    'trgDir'  : target directory path                       *
//*                    'srcTree' : pointer to TreeNode object describing       *
//*                                subdir tree                                 *
//*                    'attCount : number of backup operations attempted       *
//*                                is added to the total                       *
//*          bkSet   : operational parameters                                  *
//*                    'bex'      : array of exclusion records                 *
//*                    'bexCount' : number of exclusion records                *
//*          parentCreated : (optional, 'false' by default)                    *
//*                    This flag is used for log-file output control:          *
//*                    'false' == report parent directory as a directory       *
//*                    'true'  == report parent directory as a updated file    *
//*                    The output filter for the log will then report the      *
//*                    record according to the filter rules.                   *
//*                                                                            *
//* Returns: number of files updated                                           *
//*           (if operation was aborted, pcl.psOption set to psstABORT)        *
//******************************************************************************

UINT FileDlg::bkCopyTnList ( pclParm& pcl, const BSet& bkSet, bool parentCreated )
{
   gString gsLog ;               // format data for writing to log file
   UINT updCount = ZERO ;        // return value, number of targets updated
   tnFName tdStats ;             // target directory stats
   int64_t epoch_start,          // epoch time at start of operation
           epoch_interval ;      // interval for testing for user abort
   wkeyCode userKey ;            // capture user key/mouse input
   bool userAbort = false,       // 'true' if user aborts before completion
        logged = false ;         // 'true' if record written to log file

   this->GetLocalTime ( epoch_start ) ;  // get start time

   #if ENABLE_DEBUGGING_CODE != 0 && enableDBBAK != 0
   dbgs.compose( "%s Node:%s readAcc:%hhd subddirs:%02u", debugHDR,
                 pcl.srcTree->dirStat.fName,
                 &pcl.srcTree->dirStat.readAcc, &pcl.srcTree->dirFiles ) ;
   this->bkWriteLogfile ( dbgs ) ;
   #endif   // ENABLE_DEBUGGING_CODE && enableDBBAK

   //* If this node contains subdirectories *
   if ( pcl.srcTree->dirFiles > ZERO && pcl.srcTree->nextLevel != NULL )
   {
      bool   trgCreated = false ;      // 'true' if base dir of node created

      //* Step through the list of source subdirectories.*
      pclParm pclr ;          // for making recursive calls
      for ( UINT i = ZERO ; (i < pcl.srcTree->dirFiles) && !userAbort ; i++ )
      {
         //* Verify read access for source *
         if ( pcl.srcTree->nextLevel[i].dirStat.readAcc )
         {
            //* Set parameters for copying target's contents.*
            pclr.srcTree = &pcl.srcTree->nextLevel[i] ;
            this->fmPtr->CatPathFname ( pclr.srcDir, pcl.srcDir.ustr(), 
                                        pclr.srcTree->dirStat.fName ) ;
            this->fmPtr->CatPathFname ( pclr.trgDir, pcl.trgDir.ustr(), 
                                        pclr.srcTree->dirStat.fName ) ;
            pclr.attCount = ZERO ;

            //* Validate (or create) base target directory for this node.*
            if ( (this->bkVerifyTargetDir ( pclr.trgDir, trgCreated )) )
            {  //* Update work-in-progress stats *
               this->wip.update( 1, pclr.srcTree->dirStat.fBytes ) ;

               #if ENABLE_DEBUGGING_CODE != 0 && enableDBBAK != 0 && enableDBBAR != 0
               //* Insert 150 mS delay to allow debug of Progress Bar *
               chrono::duration<short, std::milli>aMoment( 150 ) ;
               this_thread::sleep_for( aMoment ) ;
               #endif   // ENABLE_DEBUGGING_CODE && enableDBBAK && enableDBBAR

               UINT cpyCount = ZERO ;
               //* If this node contains subdirectories *
               if ( pclr.srcTree->dirFiles > ZERO && 
                    pclr.srcTree->nextLevel != NULL )
               {
                  cpyCount = this->bkCopyTnList ( pclr, bkSet, trgCreated ) ;

                  //* If user aborted operation at lower level, *
                  //* echo the abort to the higher level.       *
                  if ( pclr.psOption == psstABORT )
                  {
                     pcl.psOption = pclr.psOption ;
                     userAbort = true ;
                  }
               }
               //* If this node contains non-directory files *
               else if ( pclr.srcTree->tnFCount > ZERO )
               {
                  cpyCount += this->bkCopyFileList ( pclr, bkSet, trgCreated ) ;

                  //* If user aborted operation at lower level, *
                  //* echo the abort to the higher level.       *
                  if ( pclr.psOption == psstABORT )
                  {
                     pcl.psOption = pclr.psOption ;
                     userAbort = true ;
                  }
               }
               //* Empty source subdirectory *
               else
               {
                  logRec lrec = ((bkSet.blfmt == blUPDATE) && trgCreated)
                                 ? lrecUPD : lrecDIR ;
                  this->bkWriteLogfile ( fileDIRECT, bkSet.blfmt, lrec, false ) ;
                  this->bkWriteLogfile ( pclr.trgDir, bkSet.blfmt, lrec ) ;
               }
               logged = true ;      // log entry created

               //* Accumulate file updates attempted *
               pcl.attCount += pclr.attCount ;
               //* Accumulate files updated *
               updCount += cpyCount ;

               //* If base directory of node was created, then   *
               //* count it as an updated file.                  *
               if ( trgCreated )
               {
                  ++pcl.attCount ;
                  ++updCount ;
               }
            }  // target verified
            else
            {  //* Target access error *
               this->bkWriteLogfile ( fileACCESS, bkSet.blfmt, lrecERR, false ) ;
               this->bkWriteLogfile ( pclr.trgDir, bkSet.blfmt, lrecERR ) ;
               logged = true ;      // log entry created
            }
         }     // source access
         else
         {  //* Source access error *
            this->bkWriteLogfile ( fileACCESS, bkSet.blfmt, lrecERR, false ) ;
            this->fmPtr->CatPathFname ( gsLog, pcl.srcDir.ustr(), 
                                        pcl.srcTree->nextLevel[i].dirStat.fName ) ;
            this->bkWriteLogfile ( gsLog, bkSet.blfmt, lrecERR ) ;
            logged = true ;      // log entry created
         }

         //* Test for user abort of the operation.        *
         //* If abort, number of unprocessed files will   *
         //* be added to the 'skipped' count.             *
         if ( epoch_start > ZERO )     // if initial system call successful
         {
            this->GetLocalTime ( epoch_interval ) ;      // get timecode
            if ( (epoch_interval - epoch_start) > 3 )
            {
               if ( (this->dPtr->KeyPeek ( userKey )) != wktERR )
               {
                  this->dPtr->GetKeyInput ( userKey ) ;  // discard input

                  //* Ask user if we should abort *
                  if ( (userAbort = this->AbortQuery ()) )
                  {
                     pcl.psOption = psstABORT ;

                     //* Make a note in the log file *
                     this->bkWriteLogfile ( fileCOPYER, blVERBOSE, lrecHDR, false ) ;
                     this->bkWriteLogfile ( userAbortMsg, blVERBOSE, lrecHDR ) ;
                  }
                  else     // restart the interval
                     this->GetLocalTime ( epoch_start ) ;
               }
            }
         }     // if(epoch_start>ZERO)
      }        // for(;;)
   }           // if source subdirectories

   //* If this node contains non-directory files, process them.*
   if ( !userAbort && (pcl.srcTree->tnFCount > ZERO) )
   {
      updCount += this->bkCopyFileList ( pcl, bkSet, parentCreated ) ;
      logged = true ;      // log entry created
   }

   //* Special case: If the base-node directory contains neither *
   //* subdirectories nor files, BUT it was validated by caller, *
   //* then it has not yet been reported in the log.             *
   if ( ! logged )
   {
      logRec lrec = ((bkSet.blfmt == blUPDATE) && parentCreated)
                     ? lrecUPD : lrecDIR ;
      this->bkWriteLogfile ( fileDIRECT, bkSet.blfmt, lrec, false ) ;
      this->bkWriteLogfile ( pcl.trgDir, bkSet.blfmt, lrec ) ;
   }
   return updCount ;

}  //* End bkCopyTnList() *

//***********************
//*   bkCopyFileList    *
//***********************
//******************************************************************************
//* Perform backup for the list of non-directory files in target directory.    *
//*                                                                            *
//* Caller has verified that user has read permission for source directory     *
//* and read/write permission on target directory.                             *
//*                                                                            *
//*                                                                            *
//* Input  : pclParm : source and target access                                *
//*                    'srcDir'  : source directory path                       *
//*                    'trgDir   : target directory path                       *
//*                    'srcTree  : pointer to TreeNode object describing       *
//*                                subdir tree                                 *
//*                    'srcTree->tnFiles':array of tnFName objects for source  *
//*                                files                                       *
//*                    'srcTree->tnFCount':number of items in tnFiles array    *
//*                    'attCount : number of backup operations attempted       *
//*                                is added to the total                       *
//*          bkSet   : operational parameters                                  *
//*                    'bex'     : array of exclusion records                  *
//*                    'bexCount': number of exclusion records                 *
//*          parentCreated : (optional, 'false' by default)                    *
//*                    This flag is used for log-file output control:          *
//*                    'false' == report parent directory as a directory       *
//*                    'true'  == report parent directory as a updated file    *
//*                    The output filter for the log will then report the      *
//*                    record according to the filter rules.                   *
//*                                                                            *
//* Returns: number of files copied                                            *
//*           (if operation was aborted, pcl.psOption set to psstABORT)        *
//******************************************************************************

UINT FileDlg::bkCopyFileList ( pclParm& pcl, const BSet& bkSet, bool parentCreated )
{
   UINT updCount = ZERO ;        // return value, number of targets updated

   #if ENABLE_DEBUGGING_CODE != 0 && enableDBBAK != 0
   dbgs.compose( L"%s Non-directory files in: %s (%02u)", debugHDR, 
                 pcl.srcTree->dirStat.fName, &pcl.srcTree->tnFCount ) ;
   this->bkWriteLogfile ( dbgs ) ;
   #endif   // ENABLE_DEBUGGING_CODE && enableDBBAK

   gString srcPath, trgPath ;    // for creating path/filename strings
   tnFName* srcStats ;           // pointer to source-file stats
   int64_t epoch_start,          // epoch time at start of operation
           epoch_interval ;      // interval for testing for user abort
   wkeyCode userKey ;            // capture user key/mouse input
   bool     updateTarget,        // 'true' if target to be updated
            userAbort = false ;  // 'true' if user aborts before completion

   this->GetLocalTime ( epoch_start ) ;  // get start time

   logRec lrec = ((bkSet.blfmt == blUPDATE) && parentCreated)
                  ? lrecUPD : lrecDIR ;
   this->bkWriteLogfile ( fileDIRECT, bkSet.blfmt, lrec, false ) ;
   this->bkWriteLogfile ( pcl.srcDir, bkSet.blfmt, lrec ) ;

   for ( UINT i = ZERO ; i < pcl.srcTree->tnFCount && !userAbort ; ++i )
   {
      srcStats = &pcl.srcTree->tnFiles[i] ;     // reference source-file stats

      //* Determine whether target exists, and if so, get its stats *
      //* and compare the modification dates.                       *
      updateTarget = true ;
      this->fmPtr->CatPathFname ( srcPath, pcl.srcDir.ustr(), srcStats->fName ) ;
      this->fmPtr->CatPathFname ( trgPath, pcl.trgDir.ustr(), srcStats->fName ) ;
      updateTarget = this->bkNeedsUpdate ( srcPath, srcStats, trgPath, bkSet, 
                                           &pcl.attCount ) ;
      if ( updateTarget )
      {
         #if ENABLE_DEBUGGING_CODE != 0 && enableDBSIM != 0
         // To verify that the backup operation actually is working without 
         // actually moving any files around, we want to write to the log 
         // that the file was successfully copied, but NOT actually copy it. 
         // In addition, we want the log to report ONLY "updated" files, not 
         // skipped files or errors. Directory-file updates will still be 
         // performed and reported
         if ( true )
         #else    // Production
         if ( (this->CopyFile ( *srcStats, srcPath, trgPath )) == OK )
         #endif   // Production
         {
            ++updCount ;         // update successful
            this->bkWriteLogfile ( fileINDENT, bkSet.blfmt, lrecUPD, false ) ;
            this->bkWriteLogfile ( srcPath, bkSet.blfmt, lrecUPD ) ;
         }
         else
         {
            this->bkWriteLogfile ( fileCOPYER, bkSet.blfmt, lrecERR, false ) ;
            this->bkWriteLogfile ( srcPath, bkSet.blfmt, lrecERR ) ;
         }

         #if ENABLE_DEBUGGING_CODE != 0 && enableDBBAK != 0 && enableDBBAR != 0
         //* Insert 150 mS delay to allow debug of Progress Bar *
         chrono::duration<short, std::milli>aMoment( 150 ) ;
         this_thread::sleep_for( aMoment ) ;
         #endif   // ENABLE_DEBUGGING_CODE && enableDBBAK && enableDBBAR
      }
      //* Update work-in-progress stats *
      this->wip.update( 1, srcStats->fBytes ) ;

      //* Test for user abort of the operation.        *
      //* If abort, number of unprocessed files will   *
      //* be added to the 'skipped' count.             *
      if ( epoch_start > ZERO )     // if initial system call successful
      {
         // Programmer's Note: For efficiency, we do not perform this test for 
         // every iteration of the loop. Instead we arbitrarily test after each 
         // eight(8) iterations. Note also that the operation can be interrupted 
         // only on three-second intervals. It would be an extraordinary list of 
         // files (or a very slow target device) that would require as much as 
         // three seconds to complete -- 
         if ( (i > ZERO) && (i % 8) == ZERO )
         {
            this->GetLocalTime ( epoch_interval ) ;      // get timecode
            if ( (epoch_interval - epoch_start) > 3 )
            {
               if ( (this->dPtr->KeyPeek ( userKey )) != wktERR )
               {
                  this->dPtr->GetKeyInput ( userKey ) ;  // discard input

                  //* Ask user if we should abort *
                  if ( (userAbort = this->AbortQuery ()) )
                  {
                     pcl.psOption = psstABORT ;

                     //* Make a note in the log file *
                     this->bkWriteLogfile ( fileCOPYER, blVERBOSE, lrecHDR, false ) ;
                     this->bkWriteLogfile ( userAbortMsg, blVERBOSE, lrecHDR ) ;
                  }
                  else     // restart the interval
                     this->GetLocalTime ( epoch_start ) ;
               }
            }
         }
      }     // if(epoch_start>ZERO)
   }        // for(;;)
   return updCount ;

}  //* End bkCopyFileList() *

//*************************
//*     bkScanTnList      *
//*************************
//******************************************************************************
//* Scan the specified TreeNode and all its contents, comparing them with the  *
//* contents of the source TreeNode. Record the results of the compaison in    *
//* the log file, but DO NOT perform the backup operation.                     *
//*                                                                            *
//* 1) If a target sub-directory exists AND user has read access, then scan    *
//*    the directory's contents.                                               *
//*    a) recursively scan source and target directory's contents.             *
//* 2) If a target sub-directory does not exist OR if user does not have read  *
//*    access then we cannot scan it, and as a consequence, all source files   *
//*    for that directory will be listed as needing to be updated.             *
//*                                                                            *
//* Input  : pclParm : source and target access                                *
//*                    'srcDir'  : source directory path                       *
//*                    'trgDir'  : target directory path                       *
//*                    'srcTree' : pointer to TreeNode object describing       *
//*                                subdir tree                                 *
//*                    'attCount : number of backup operations attempted       *
//*                                is added to the total                       *
//*          bkSet   : operational parameters                                  *
//*                    'blfmt'   : log format                                  *
//*                                                                            *
//* Returns: number of target files that require update                        *
//*           (if operation was aborted, pcl.psOption set to psstABORT)        *
//******************************************************************************

UINT FileDlg::bkScanTnList ( pclParm& pcl, const BSet& bkSet )
{
   gString gsLog, gsVal ;        // format data for writing to log file
   UINT   nodeFCount,            // total files in TreeNode
          cpyCount,              // recursive return value or current-node non-dir files
          updCount = ZERO ;      // return value, number of targets requiring update
   tnFName tdStats ;             // target directory stats
   UINT64 nodeTBytes = ZERO ;    // total size of TreeNode (not used)
   int64_t epoch_start,          // epoch time at start of operation
           epoch_interval ;      // interval for testing for user abort
   wkeyCode userKey ;            // capture user key/mouse input
   bool trgExists = false,       // 'true' if target exists
        userAbort = false ;      // 'true' if user aborts before completion

   this->GetLocalTime ( epoch_start ) ;  // get start time

   //* If this node contains subdirectories *
   if ( pcl.srcTree->dirFiles > ZERO && pcl.srcTree->nextLevel != NULL )
   {
      //* Step through the list of source subdirectories.*
      pclParm pclr ;          // for making recursive calls
      for ( UINT i = ZERO ; (i < pcl.srcTree->dirFiles) && !userAbort ; i++ )
      {
         //* Verify read access for source *
         if ( pcl.srcTree->nextLevel[i].dirStat.readAcc )
         {
            //* Set parameters for copying source contents to target.*
            pclr.srcTree = &pcl.srcTree->nextLevel[i] ;
            this->fmPtr->CatPathFname ( pclr.srcDir, pcl.srcDir.ustr(), 
                                        pclr.srcTree->dirStat.fName ) ;
            this->fmPtr->CatPathFname ( pclr.trgDir, pcl.trgDir.ustr(), 
                                        pclr.srcTree->dirStat.fName ) ;
            pclr.attCount = ZERO ;
            cpyCount = ZERO ;

            //* Validate base target directory for this node.*
            if ( (this->bkVerifyTargetDir ( pclr.trgDir, trgExists, true )) )
            {  //* Update work-in-progress stats *
               this->wip.update( 1, pclr.srcTree->dirStat.fBytes ) ;

               //* If this node contains subdirectories, *
               //* scan the lower level directories.     *
               if ( pclr.srcTree->dirFiles > ZERO && 
                    pclr.srcTree->nextLevel != NULL )
               {
                  cpyCount = this->bkScanTnList ( pclr, bkSet ) ;

                  //* If user aborted operation at lower level, *
                  //* echo the abort to the higher level.       *
                  if ( pclr.psOption == psstABORT )
                  {
                     pcl.psOption = pclr.psOption ;
                     userAbort = true ;
                  }
               }

               //* If this node contains non-directory files *
               //* but NOT lower-level subdirectories.       *
               else if ( pclr.srcTree->tnFCount > ZERO )
               {
                  cpyCount += this->bkScanFileList ( pclr, bkSet ) ;

                  //* If user aborted operation at lower level, *
                  //* echo the abort to the higher level.       *
                  if ( pclr.psOption == psstABORT )
                  {
                     pcl.psOption = pclr.psOption ;
                     userAbort = true ;
                  }
               }

               //* Empty source subdirectory AND target exists *
               else
               {
                  //* If target directory does not exist, *
                  //* count it as needing an update.      *
                  logRec lrec = ((bkSet.blfmt == blUPDATE) && trgExists)
                                 ? lrecUPD : lrecDIR ;
                  this->bkWriteLogfile ( fileDIRECT, bkSet.blfmt, lrec, false ) ;
                  this->bkWriteLogfile ( pclr.trgDir, bkSet.blfmt, lrec ) ;
               }

               //* Accumulate file updates that would have been attempted *
               pcl.attCount += pclr.attCount ;
               //* Accumulate files requiring update *
               updCount += cpyCount ;

            }  // target verified

            //* Target directory access error *
            else
            {
               //* 1) Target does not exist and was not created. Report    *
               //*    the directory and all its contents as needing update.*
               //* 2) Actual target access error. Count directory and all  *
               //*    its contents as errors. (errors calculated by caller)*
               if ( ! (this->TargetExists ( pclr.trgDir, tdStats.fType)) )
               {
                  //* Count the files in the source node *
                  nodeFCount = ZERO ;
                  nodeTBytes = ZERO ;
                  this->fmPtr->TreeNodeSummary ( pclr.srcTree, nodeFCount, nodeTBytes ) ;
                  pclr.attCount += nodeFCount ;

                  //* Accumulate file updates that would have been attempted *
                  pcl.attCount += pclr.attCount ;
                  //* Accumulate files requiring update *
                  updCount += nodeFCount ;

                  //* Report results in the log *
                  this->bkWriteLogfile ( fileDIRECT, blVERBOSE, lrecDIR, false ) ;
                  this->bkWriteLogfile ( pclr.trgDir, blVERBOSE, lrecDIR ) ;
                  gsVal.formatInt( nodeFCount, FI_MAX_FIELDWIDTH, true ) ;
                  gsLog.compose( scanTemplate1, gsVal.gstr() ) ;
                  this->bkWriteLogfile ( gsLog, blVERBOSE, lrecDIR ) ;
               }
               else
               {
                  this->bkWriteLogfile ( fileACCESS, bkSet.blfmt, lrecERR, false ) ;
                  this->bkWriteLogfile ( pclr.trgDir, bkSet.blfmt, lrecERR ) ;
               }
            }
         }     // source accessible

         else
         {  //* Source access error *
            this->bkWriteLogfile ( fileACCESS, bkSet.blfmt, lrecERR, false ) ;
            this->fmPtr->CatPathFname ( gsLog, pcl.srcDir.ustr(), 
                                        pcl.srcTree->nextLevel[i].dirStat.fName ) ;
            this->bkWriteLogfile ( gsLog, bkSet.blfmt, lrecERR ) ;
         }

         //* Test for user abort of the operation.        *
         //* If abort, number of unprocessed files will   *
         //* be added to the 'skipped' count.             *
         if ( epoch_start > ZERO )     // if initial system call successful
         {
            this->GetLocalTime ( epoch_interval ) ;      // get timecode
            if ( (epoch_interval - epoch_start) > 3 )
            {
               if ( (this->dPtr->KeyPeek ( userKey )) != wktERR )
               {
                  this->dPtr->GetKeyInput ( userKey ) ;  // discard input

                  //* Ask user if we should abort *
                  if ( (userAbort = this->AbortQuery ()) )
                  {
                     pcl.psOption = psstABORT ;

                     //* Make a note in the log file *
                     this->bkWriteLogfile ( fileCOPYER, blVERBOSE, lrecHDR, false ) ;
                     this->bkWriteLogfile ( userAbortMsg, blVERBOSE, lrecHDR ) ;
                  }
                  else     // restart the interval
                     this->GetLocalTime ( epoch_start ) ;
               }
            }
         }     // if(epoch_start>ZERO)
      }        // for(;;)
   }           // if source subdirectories

   if ( ! userAbort )
   {
      //* Report the top-level directory according to filter rules   *
      //* If the base-node directory contains neither subdirectories *
      //* nor files, BUT it was validated by caller, then it has not *
      //* yet been reported in the log.                              *
      //* If this node contains non-directory files, process them.   *
      updCount += this->bkScanFileList ( pcl, bkSet ) ;
   }

   return updCount ;

}  //* End bkScanTnList() *

//***********************
//*   bkScanFileList    *
//***********************
//******************************************************************************
//* Scan the list of non-directory files in target directory, comparing them   *
//* with the files in the source directory. Record the results of the          *
//* comparison in the log file, but DO NOT perform the backup operation.       *
//*                                                                            *
//* If blfmt == "blUPDATE", report the parent directory in the log only if     *
//* directory contents require update(s).                                      *
//*                                                                            *
//* Caller has verified that user has read permission for source directory     *
//* and read permission on target directory.                                   *
//*                                                                            *
//* Input  : pclParm : source and target access                                *
//*                    'srcDir'  : source directory path                       *
//*                    'trgDir   : target directory path                       *
//*                    'srcTree  : pointer to TreeNode object describing       *
//*                                subdir tree                                 *
//*                    'srcTree->tnFiles':array of tnFName objects for source  *
//*                                files                                       *
//*                    'srcTree->tnFCount':number of items in tnFiles array    *
//*                    'attCount : number of backup operations attempted       *
//*                                is added to the total                       *
//*          bkSet   : operational parameters                                  *
//*                    'blfmt'   : log format                                  *
//*                                                                            *
//* Returns: number of target files that require update                        *
//*           (if operation was aborted, pcl.psOption set to psstABORT)        *
//******************************************************************************

UINT FileDlg::bkScanFileList ( pclParm& pcl, const BSet& bkSet )
{
   UINT updCount = ZERO ;        // return value, number of targets requiring update

   gString srcPath, trgPath ;    // for creating path/filename strings
   tnFName* srcStats ;           // pointer to source-file stats
   int64_t epoch_start,          // epoch time at start of operation
           epoch_interval ;      // interval for testing for user abort
   wkeyCode userKey ;            // capture user key/mouse input
   bool     updateTarget,        // 'true' if target to be updated
            doScan = true,       // 'true' if secondary scan required
            userAbort = false ;  // 'true' if user aborts before completion

   this->GetLocalTime ( epoch_start ) ;  // get start time

   //* If log format is for updates only, and one or more source files   *
   //* exist, then perform preliminary scan of target files to determine *
   //* whether to add the container directory to the log file.           *
   if ( (bkSet.blfmt == blUPDATE) && (pcl.srcTree->tnFCount) )
   {
      BSet tmpBSet ;
      UINT tmpAttCnt ;
      tmpBSet.blfmt = blUPDATE ;
      tmpBSet.scanOnly = true ;
      for ( UINT i = ZERO ; i < pcl.srcTree->tnFCount && !userAbort ; ++i )
      {
         //* Determine whether target exists, and if so, *
         //* determine whether it requires an update.    *
         srcStats = &pcl.srcTree->tnFiles[i] ;     // reference source-file stats
         this->fmPtr->CatPathFname ( srcPath, pcl.srcDir.ustr(), srcStats->fName ) ;
         this->fmPtr->CatPathFname ( trgPath, pcl.trgDir.ustr(), srcStats->fName ) ;
         if ( (doScan = this->bkNeedsUpdate ( srcPath, srcStats, trgPath, tmpBSet, &tmpAttCnt )) )
            break ;
      }
   }

   //* If prescan identified targets needing update *
   if ( doScan )
   {
      //* Report container directory to the log file *
      logRec lrec = ((bkSet.blfmt == blUPDATE) && (pcl.srcTree->tnFCount))
                     ? lrecUPD : lrecDIR ;
      this->bkWriteLogfile ( fileDIRECT, bkSet.blfmt, lrec, false ) ;
      this->bkWriteLogfile ( pcl.trgDir, bkSet.blfmt, lrec ) ;

      //* Recursively scan subdirectories in the tree *
      for ( UINT i = ZERO ; i < pcl.srcTree->tnFCount && !userAbort ; ++i )
      {
         srcStats = &pcl.srcTree->tnFiles[i] ;     // reference source-file stats

         //* Determine whether target exists, and if so, get its stats *
         //* and compare the modification dates.                       *
         //updateTarget = true ;
         this->fmPtr->CatPathFname ( srcPath, pcl.srcDir.ustr(), srcStats->fName ) ;
         this->fmPtr->CatPathFname ( trgPath, pcl.trgDir.ustr(), srcStats->fName ) ;
         updateTarget = this->bkNeedsUpdate ( srcPath, srcStats, trgPath, bkSet, 
                                              &pcl.attCount ) ;

         if ( updateTarget )        // if update needed
         {
            ++updCount ;
            this->bkWriteLogfile ( fileINDENT, bkSet.blfmt, lrecUPD, false ) ;
            this->bkWriteLogfile ( trgPath, bkSet.blfmt, lrecUPD ) ;
         }
         //* Update work-in-progress stats *
         this->wip.update( 1, srcStats->fBytes ) ;

         //* Test for user abort of the operation.        *
         //* If abort, number of unprocessed files will   *
         //* be added to the 'skipped' count.             *
         if ( epoch_start > ZERO )     // if initial system call successful
         {
            // Programmer's Note: For efficiency, we do not perform this test for 
            // every iteration of the loop. Instead we arbitrarily test after each 
            // eight(8) iterations. Note also that the operation can be interrupted 
            // only on three-second intervals. It would be an extraordinary list of 
            // files (or a very slow target device) that would require as much as 
            // three seconds to complete -- 
            if ( (i > ZERO) && (i % 8) == ZERO )
            {
               this->GetLocalTime ( epoch_interval ) ;      // get timecode
               if ( (epoch_interval - epoch_start) > 3 )
               {
                  if ( (this->dPtr->KeyPeek ( userKey )) != wktERR )
                  {
                     this->dPtr->GetKeyInput ( userKey ) ;  // discard input

                     //* Ask user if we should abort *
                     if ( (userAbort = this->AbortQuery ()) )
                     {
                        pcl.psOption = psstABORT ;

                        //* Make a note in the log file *
                        this->bkWriteLogfile ( fileCOPYER, blVERBOSE, lrecHDR, false ) ;
                        this->bkWriteLogfile ( userAbortMsg, blVERBOSE, lrecHDR ) ;
                     }
                     else     // restart the interval
                        this->GetLocalTime ( epoch_start ) ;
                  }
               }
            }
         }     // if(epoch_start>ZERO)
      }
   }
   return updCount ;

}  //* End bkScanFileList() *

//*************************
//*   bkVerifyTargetDir   *
//*************************
//******************************************************************************
//* Determine whether specified directory exists.                              *
//*  a) If it exists, verify whether user has read/write access.               *
//*  b) If it does not exist, create it _with_ read/write access.              *
//*                                                                            *
//* Input  : trgPath    : full path/filename of directory to be tested         *
//*          trgCreated : (by reference)                                       *
//*                       receives 'true' if target direct directory created   *
//*                       receives 'false' if target already exists            *
//*          testOnly   : (optional, 'false' by default)                       *
//*                       if 'false', create the target if if it does not exist*
//*                       if 'true',  test existing target only, do not create *
//*                                   a new target directory                   *
//*                                                                            *
//* Returns: 'true'  if target exists and is accessible, else 'false'          *
//******************************************************************************
//* Programmer's Note: We make an assumption that if we successfully create a  *
//* subdirectory in user space that user DOES have read/write/exec access.     *
//******************************************************************************

bool FileDlg::bkVerifyTargetDir ( const gString& trgPath, bool& trgCreated, bool testOnly )
{
   tnFName tdStats ;             // target directory stats
   bool trgOk = false ;          // return value
   trgCreated = false ;          // initialize caller's flag

   if ( (this->fmPtr->GetFileStats ( tdStats, trgPath )) != OK )
   {
      if ( ! testOnly )
      {
         if ( clipBoard_gvfsTrg )
         {
            if ( (this->fmPtr->CreateDirectory_gvfs ( trgPath.ustr() )) == OK )
               trgCreated = trgOk = true ;
         }
         else
         {
            if ( (this->fmPtr->CreateDirectory ( trgPath.ustr() )) == OK )
               trgCreated = trgOk = true ;
         }
      }
   }
   //* Target directory must have read/execute access, *
   //* and if updates will occur, write access also.   *
   else if ( (tdStats.fType == fmDIR_TYPE) && tdStats.readAcc && 
             (tdStats.rawStats.st_mode & S_IXUSR) )
   {
      if ( testOnly || (!testOnly && tdStats.writeAcc) )
         trgOk = true ;    // target exists and is accessible
   }

   return trgOk ;

}  //* End bkVerifyTargetDir() *

//*************************
//*     bkNeedsUpdate     *
//*************************
//********************************************************************************
//* Determine whether the target file needs to be updated.                       *
//*                                                                              *
//* 1) Verify that user has read access to source                                *
//*    a) If no read access, report an error.                                    *
//* 2) Verify that source's file type is one that we can process:                *
//*       'Regular', 'SymLink', 'FIFO',                                          *
//*       NOTE: This method SHOULD NOT be called for directory names.            *
//*    a) If not a supported type, report it as a file to be skipped,            *
//*       (not an error).                                                        *
//*    b) Special note on symlink files: Symbolic link files cannot be written   *
//*       to VFAT and other primitive filesystems. Therefore, if caller          *
//*       indicates (bkSet.slSupport==false) that the target filesystem does     *
//*       not support symlinks, we report the update as skipped knowing that     *
//*       the write would fail anyway.                                           *
//* 3) Test source file against exclusion list:                                  *
//*    Note that the comparison is against the full source path/filename,        *
//*    in case user is excluding a specific file.                                *
//*    a) If no match, then continue to step 4.                                  *
//*    b) If a match, then no update needed, report it as a file to be skipped   *
//* 4) Determine whether target file exists, and if it does:                     *
//*    a) Compare source and target modification dates.                          *
//*       If target date >= source date, then no update needed (skipped)         *
//*    b) Test target read/write access flags.                                   *
//*       If no access, then no update is possible, report access error.         *
//*    c) Verify that source and target are of the same file type.               *
//*       If different file types, then deny the update (type mismatch error)    *
//*                                                                              *
//* Input  : srcPath  : path/filename of source file to be tested                *
//*          srcStats : pointer to source-file stats                             *
//*          trgPath  : path/filename of target to be tested                     *
//*          bkSet    : operational parameters                                   *
//*                     'bex' is an array of exclusion records                   *
//*                     'bexCount' is the number of exclusion records            *
//*                     'scanOnly' indicates the type of record written to log.  *
//*          attPtr   : pointer to 'attCount' which is incremented IF target     *
//*                     is older than source or target does not exist            *
//*                                                                              *
//* Returns: 'true'  update the target                                           *
//*                  'attempt' counter is incremented                            *
//*          'false' do not update the target                                    *
//*                  if error condition, 'attempt' counter is incremented        *
//*                  if no error, then 'attempt' counter not modified            *
//********************************************************************************
//* Programmer's Note: We preserve all timestamps during copy operations;        *
//* therefore, timestamp comparison should be straightforward.                   *
//* - Unfortunately, not all programs do this, so identical files may have a     *
//*   small timestamp differential, especially if a copy was to a different      *
//*   system or crossed filesystem boundries.                                    *
//* - For this reason, we may occasionally do an unnecessary file update, but    *
//*   this is better than skipping a needed update.                              *
//* - When copying to a FAT/VFAT drive such as a USB drive, it is important to   *
//*   note that the timestamp is truncated in some way compared to the native    *
//*   Linux filesystem (ext2/3/4). This can cause two identical files to be      *
//*   different by one second, with the VFAT copy being one second behind the    *
//*   native drive. This causes an annoyingly large number of unnecessary        *
//*   backup operations, and in fact causes the same file to be backed up day    *
//*   after day even though it may not have been modified for months.            *
//*   - An additional wrinkle in this scenario is that after a backup operation  *
//*     where the unnecessary backups took place, the timestamp comparison       *
//*     shows they are equal, HOWEVER, the next day source and target are again  *
//*     off by one second. WTF?                                                  *
//*   - Experimentally, we compare the target timestamp to the source            *
//*     timestamp - 1. This stops the unnecessary target updates, but the        *
//*     better answer would be to advance the target timestamp by one second.    *
//*                                                                              *
//* Note on exclusion test:                                                      *
//* -----------------------                                                      *
//* a) The comparison must be with the LAST instance of the substring in the     *
//*    source.                                                                   *
//* b) The index of the match must be EXACTLY 'bex[i].len' characters less       *
//*    than the number of characters in the source.                              *
//********************************************************************************

bool FileDlg::bkNeedsUpdate ( const gString& srcPath, const tnFName* srcStats, 
                              const gString& trgPath, const BSet& bkSet, UINT* attPtr )
{
   tnFName trgStats ;               // target-file stats
   bool tExists,                    // 'true' if target exists
        updateTarget = true ;       // return value

   if ( srcStats->readAcc )
   {
      if ( (srcStats->fType == fmREG_TYPE)  ||
           ((srcStats->fType == fmLINK_TYPE) && bkSet.slSupport) ||
           (srcStats->fType == fmFIFO_TYPE) )
      {
         //* Compare source file's filename extension/tail to the exclusion list.*
         short eIndex ;
         for ( UINT i = ZERO ; i < bkSet.bexCount ; ++i )
         {
            if ( ((eIndex = srcPath.findlast( bkSet.bex[i].excl )) >= ZERO) &&
                 ((srcPath.gschars() - bkSet.bex[i].len) == eIndex) )
            {  //* Exclusion match *
               updateTarget = false ;  // do not process the file (not an error)
               #if ENABLE_DEBUGGING_CODE == 0 || enableDBSIM == 0
               if ( ! bkSet.scanOnly )
               {
                  this->bkWriteLogfile ( fileSKIPIT, bkSet.blfmt, lrecSKP, false ) ;
                  this->bkWriteLogfile ( srcPath, bkSet.blfmt, lrecSKP ) ;
               }
               #endif    // ENABLE_DEBUGGING_CODE==0 || enableDBSIM==0
               break ;
            }
         }
         if ( updateTarget )
         {
            if ( (tExists = this->TargetExists ( trgPath.ustr(), trgStats )) )
            {
               // Programmer's Note: As detailed elsewhere, copying to a FAT32 
               // or NTFS filesystem often truncates the timestamp which can 
               // cause the target mod time to be one second older than the 
               // source. We compensate here.
               if ( trgStats.modTime.epoch < (srcStats->modTime.epoch - 1) )
               {
                  ++*attPtr ; // target is older, update needed

                  if ( ! trgStats.readAcc || ! trgStats.writeAcc )
                  {  //* Target access error *
                     updateTarget = false ;
                     this->bkWriteLogfile ( fileACCESS, bkSet.blfmt, lrecERR, false ) ;
                     this->bkWriteLogfile ( trgPath, bkSet.blfmt, lrecERR ) ;
                  }
                  else if ( trgStats.fType != srcStats->fType )
                  {  //* Source/target mismatch error *
                     updateTarget = false ;
                     this->bkWriteLogfile ( fileTYPEER, bkSet.blfmt, lrecERR, false ) ;
                     this->bkWriteLogfile ( trgPath, bkSet.blfmt, lrecERR ) ;
                  }
               }
               else
               {  //* Target is up-to-date, skip it *
                  updateTarget = false ;
                  #if ENABLE_DEBUGGING_CODE == 0 || enableDBSIM == 0
                  this->bkWriteLogfile ( fileSKIPIT, bkSet.blfmt, lrecSKP, false ) ;
                  this->bkWriteLogfile ( bkSet.scanOnly ? trgPath : srcPath, 
                                         bkSet.blfmt, lrecSKP ) ;
                  #endif    // ENABLE_DEBUGGING_CODE==0 || enableDBSIM==0
               }
            }
            else              // target does not exist, attempt update
               ++*attPtr ;
         }
      }
      else
      {  //* Source is of an unsupported filetype (this is not an error) *
         updateTarget = false ;
         #if ENABLE_DEBUGGING_CODE == 0 || enableDBSIM == 0
         if ( ! bkSet.scanOnly )
         {
            this->bkWriteLogfile ( fileSKIPIT, bkSet.blfmt, lrecSKP, false ) ;
            this->bkWriteLogfile ( srcPath, bkSet.blfmt, lrecSKP ) ;
         }
         #endif    // ENABLE_DEBUGGING_CODE==0 || enableDBSIM==0
      }
   }
   else
   {  //* No read access to source *
      updateTarget = false ;
      ++*attPtr ;
      this->bkWriteLogfile ( fileACCESS, bkSet.blfmt, lrecERR, false ) ;
      this->bkWriteLogfile ( srcPath, bkSet.blfmt, lrecERR ) ;
   }

   //* This sequence is used only for debugging; however, it remains active as *
   //* an option in the setup dialog.                                          *
   if ( updateTarget && bkSet.blfmt == blTSTAMP )
   {
      gString gs( "          s: %hhd %hhd %04hd-%02hd-%02hdT%02hd:%02hd:%02hd %LX",
                  &srcStats->readAcc, &srcStats->writeAcc, &srcStats->modTime.year,
                  &srcStats->modTime.month, &srcStats->modTime.date,
                  &srcStats->modTime.hours, &srcStats->modTime.minutes, 
                  &srcStats->modTime.seconds, &srcStats->modTime.epoch ) ;
      if ( tExists )
      {
         gs.append( "\n          t: %hhd %hhd %04hd-%02hd-%02hdT%02hd:%02hd:%02hd %LX",
                    &trgStats.readAcc, &trgStats.writeAcc, &trgStats.modTime.year,
                    &trgStats.modTime.month, &trgStats.modTime.date,
                    &trgStats.modTime.hours, &trgStats.modTime.minutes, 
                    &trgStats.modTime.seconds, &trgStats.modTime.epoch ) ;
      }
      this->bkWriteLogfile ( gs, bkSet.blfmt, lrecUPD ) ;
   }

   #if ENABLE_DEBUGGING_CODE != 0 && enableDBBAK != 0 && enableDBNEED != 0
   dbgs.compose( "%s %s  readAcc:%hhd updateTarget:%hhd tExists:%hhd", debugHDR, 
                 srcStats->fName, &srcStats->readAcc, &updateTarget, &tExists ) ;
   if ( tExists )
      dbgs.append( " target%Ssource fType:%02hd readACC:%hhd writeAcc:%hhd", 
                   ((trgStats.modTime.epoch < srcStats->modTime.epoch) ? L"<" : L">="),
                   &trgStats.fType, &trgStats.readAcc, &trgStats.writeAcc ) ;
   this->bkWriteLogfile ( dbgs ) ;
   #endif   // ENABLE_DEBUGGING_CODE && enableDBBAK && enableDBNEED

   return updateTarget ;

}  //* End bkNeedsUpdate() *

//*************************
//*     bkOpenLogfile     *
//*************************
//********************************************************************************
//* Open the Backup/Archive/Synch logfile for writing, (truncate), and write     *
//* the header record.                                                           *
//*                                                                              *
//* Input  : bkSet   : operational parameters (by reference)                     *
//*                    'blog' contains the path/filename spec                    *
//*                    'btime' contains the operation timestamp                  *
//*                                                                              *
//* Returns: 'true' if file opened successfully (this->ofsLog gets handle)       *
//*          'false' if file error                                               *
//********************************************************************************

bool FileDlg::bkOpenLogfile ( const BSet& bkSet )
{
   bool status = false ;

   if ( !(this->ofsLog.is_open()) )    // if file is not already open
   {
      this->ofsLog.open( bkSet.blog.ustr(), ofstream::out | ofstream::app ) ;
      if ( (this->ofsLog.is_open()) )
      {
         gString gsOut ;
         if ( bkSet.btype == bsetBackup || 
              bkSet.btype == bsetSynch  || bkSet.btype == bsetArchive )
         {
            gsOut.compose( 
               "FileMangler Backup Logfile: %04hd-%02hd-%02hdT%02hd:%02hd:%02hd\n"
               "===============================================",
               &bkSet.btime.year, &bkSet.btime.month, &bkSet.btime.date,
               &bkSet.btime.hours, &bkSet.btime.minutes, &bkSet.btime.seconds ) ;
         }
         else
         {
            gsOut.compose(
               "FileMangler Archive Logfile: %04hd-%02hd-%02hdT%02hd:%02hd:%02hd\n"
               "================================================",
               &bkSet.btime.year, &bkSet.btime.month, &bkSet.btime.date,
               &bkSet.btime.hours, &bkSet.btime.minutes, &bkSet.btime.seconds ) ;
         }
         this->bkWriteLogfile ( gsOut, bkSet.blfmt ) ;
         status = true ;
      }
   }
   return status ;

}  //* End bkOpenLogfile() *

//*************************
//*    bkWriteLogfile     *
//*************************
//********************************************************************************
//* Write a data record to the Backup/Archive/Synch logfile.                     *
//*                                                                              *
//* Input  : logData : data to be written                                        *
//*          filter  : (optional, blVERBOSE by default)                          *
//*                    output filter (member of enum bsetLog)                    *
//*          rectype : (optional, lrecHDR by default)                            *
//*                    indicates the type of record sent                         *
//*                    (this is actually a member of enum logRec.)               *
//*                    (see top of this module                   )               *
//*          lf      : (optional, 'true' by default)                             *
//*                    if 'true',  terminate the output with a linefeed          *
//*                    if 'false', terminate the output with a space             *
//*                                                                              *
//* Returns: 'true' if data written successfully                                 *
//*          'false' if file error                                               *
//********************************************************************************
//* Notes on what is included in the log file. See enum bsetLog.                 *
//* We isolate the decision-making process here to keep the calling methods      *
//* clean. Each calling method tells us what kind of record they are sending     *
//* to us, and we write the record only if it meets the formatting criteria.     *
//*                                                                              *
//* 1) blVERBOSE (default):                                                      *
//*    Write all Backup/Synch data to the log file:                              *
//*    header, summary, directory records, record for each source file scanned.  *
//* 2) blUPDATE:                                                                 *
//*    header, summary, record for each update attempted.(incl. files and dirs)  *
//* 3) blERROR:                                                                  *
//*    header, summary, directory records, record for each _failed_ update.      *
//* 4) blSUMMARY:                                                                *
//*    header, summary                                                           *
//* 5) blNONE:                                                                   *
//*    Log file was not opened, so we write nothing.                             *
//* 6) blTSTAMP: Used during development only.                                   *
//*    header, summary, directory records, record for each update attempted,     *
//*      _plus_ source and target timestamps and r/w access flags                *
//********************************************************************************

bool FileDlg::bkWriteLogfile ( const gString& logData, bsetLog filter, 
                               short rectype, bool lf )
{
   bool status = false ;

   if ( (this->ofsLog.is_open()) )
   {
      //* See above for explanation of output filter *
      if ( (filter == blVERBOSE) || (rectype == lrecHDR) || (rectype == lrecSUM) ||
           ((filter == blUPDATE) && (rectype != lrecSKP) && (rectype != lrecDIR)) ||
           (((filter != blSUMMARY) && (filter != blERROR) && (filter != blUPDATE)) 
               && ((rectype == lrecDIR) || (rectype == lrecUPD) || (rectype == lrecERR))) ||
           ((filter == blERROR) && ((rectype == lrecDIR) || (rectype == lrecERR))) ||
           ((filter == blTSTAMP) && (rectype != lrecSKP))
         )
      {
         this->ofsLog << logData.ustr() ;
         if ( lf )
            this->ofsLog << endl ;
         else
         {
            this->ofsLog << " " ;
            this->ofsLog.flush() ;
         }
      }
      status = true ;
   }
   return status ;

}  //* End bkWriteLogfile() *
bool FileDlg::bkWriteLogfile ( const char* logData, bsetLog filter,
                               short rectype, bool lf )
{
   gString gs( logData ) ;
   return ( this->bkWriteLogfile ( gs, filter, rectype, lf ) ) ;

}  //* End bkWriteLogfile() *

//*************************
//*    bkCloseLogfile     *
//*************************
//******************************************************************************
//* Close the Backup/Archive/Synch logfile.                                    *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//*          (this->ofsLog set to NULL pointer)                                *
//******************************************************************************

void FileDlg::bkCloseLogfile ( void )
{

   if ( (this->ofsLog.is_open()) )
   {
      this->bkWriteLogfile ( "\n" ) ;  // write an empty line
      this->ofsLog.close() ;
   }

}  //* End bkCloseLogfile() *

//*************************
//*      AbortQuery       *
//*************************
//******************************************************************************
//* User has interrupted an operation in progress:                             *
//*  - Backup                                                                  *
//*  - Synch                                                                   *
//*  -                                                                         *
//*  -                                                                         *
//* Ask whether user wants to terminate the operation and report to caller.    *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if user aborts the operation, else 'false'                 *
//******************************************************************************

bool FileDlg::AbortQuery ( void )
{
   const char* aqMessage[] = 
   { //1234567890123456789012345678901234567890123456789012 - (max line length)
      "  OPERATION INTERRUPTED  ",
      " ",
      "  The operation in progress has been interrupted.",
      " ",
      "      Do you want to terminate the operation?      ",
      " ",
      " ",
      NULL
   } ;
   attr_t dColor = this->cfgOptions.cScheme.sd,
          eColor = this->cfgOptions.cScheme.em ;
   attr_t aqColors[] = { eColor, dColor, dColor, dColor, dColor, dColor, dColor } ;

   genDialog gd
   (
   aqMessage,                    // message text
   dColor,                       // dialog background color
   9,                            // dialog rows
   INFODLG_WIDTH,                // dialog columns
   -1,                           // default Y position
   -1,                           // default X position
   aqColors,                     // color-attribute array
   false,                        // LTR text
   attrDFLT,                     // 'Yes' button default color attribute
   attrDFLT,                     // 'No' button default color attribute
   "  TERMINATE  ",
   "  CONTINUE   "
   ) ;

   //* Launch a decision dialog to query the user.*
   bool abort = this->dPtr->DecisionDialog ( gd ) ;

   this->wip.pbarRefresh() ;     // refresh display of progress bar

   return abort ;

}  //* End AbortQuery() *

//*************************
//*        Archive        *
//*************************
//******************************************************************************
//* Initialize parameters for creation, update or expansion of an archive file.*
//* User may specify:                                                          *
//*  1) source data to create a new archive.                                   *
//*  2) the name of an existing archive to be expanded                         *
//*  3) source data to be appended to a specified existing archive.            *
//*                                                                            *
//* Because a single command is used for 'create' and 'expand' AND 'update',   *
//* the logic for determining the type of operation is a bit complicated, so   *
//* study it carefully.                                                        *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Operation Logic                                                            *
//* ===============                                                            *
//* If the user's intended operation is unambiguous, go directly to the dialog *
//* for that operation:                                                        *
//* 1) Expand an existing archive:                                             *
//*    There are two ways to specify that an archive is to be expanded:        *
//*    a) Copy the archive file to the clipboard as the ONLY FILE on the       *
//*       clipboard. Then invoke the archive dialog: ALT+SHIFT+A.              *
//*    b) With the clipboard empty AND with no files 'selected', highlight     *
//*       the archive file and invoke the archive dialog: ALT+SHIFT+A.         *
//*    c) Other scenarios cannot be automatically identified as an 'Expand"    *
//*       operation, and it will be necessary to query the user for the        *
//*       desired action.                                                      *
//*                                                                            *
//* 2) Update an existing archive:                                             *
//*    The archive to be updated must be the highlighted file, AND it          *
//*    must not be 'selected'.                                                 *
//*    a) If highlight does reference an archive file AND that file is not     *
//*       'selected',then:                                                     *
//*       -- If 'selected' files in the active window OR if data on clipboard, *
//*          then the operation is to update the specified archive using the   *
//*          specified data.                                                   *
//*          -- Note that OpenDoc/OpenXML files are not allowed as update      *
//*             targets.                                                       *
//*       -- If 'selected' files in the active window AND data on the          *
//*          clipboard, then we test whether the 'selections' are the same     *
//*          data as on the clipboard.                                         *
//*          -- If yes, then the clipboard data will be used to update the     *
//*             archive.                                                       *
//*          -- If no, then the 'selected' data replace the existing clipboard *
//*             data. The archive will then be updated using the new clipboard *
//*             data.                                                          *
//*    b) Other scenarios cannot be automatically identified as an 'Update"    *
//*       operation, and it will be necessary to query the user for the        *
//*       desired action.                                                      *
//*                                                                            *
//* 3) Create an archive:                                                      *
//*    a) If no 'selected' files AND no data on the clipboard, then we cannot  *
//*       continue. Give user a clue.                                          *
//*    b) If highlight does not reference an archive file, then:               *
//*       -- If 'selected' files in active window OR if data on the clipboard, *
//*          then the operation is to create an archive using the specified    *
//*          data. Offer a target archive filename and archive-type.           *
//*       -- If 'selected' files in the active window AND data on the          *
//*          clipboard, then we test whether the 'selections' are the same     *
//*          data as on the clipboard.                                         *
//*          -- If yes, then the clipboard data will be used to create the     *
//*             archive. Offer a target archive filename and archive-type.     *
//*          -- If no, then the 'selected' data replace the existing clipboard *
//*             data. The archive will then be created using the new clipboard *
//*             data. Offer a target archive filename and archive-type.        *
//*    c) Other scenarios cannot be automatically identified as a 'Create"     *
//*       operation, and it will be necessary to query the user for the        *
//*       desired action.                                                      *
//*                                                                            *
//* 4) If unable to automatically determine the archive operation sub-type,    *
//*    open a query dialog to ask the user how to proceed.                     *
//*                                                                            *
//* 5) If insufficient information for any archive operation, then complain    *
//*    and abort the operation.                                                *
//*                                                                            *
//*                                                                            *
//* Logical difficulties and processing exceptions:                            *
//* -----------------------------------------------                            *
//* A) Archives will always be created/updated/expanded in the active window.  *
//* B) 'Selected' files in the inactive window (Dual-Window Mode):             *
//*    The inactive window is not involved in the operation except that        *
//*    'selections' in the inactive window which are not also on the clipboard *
//*    will have been cleared by the caller to avoid ambiguous visual cues.    *
//* C) OpenDocument files, Office Open XML files and ePub documents are        *
//*    structurally ZIP archive files; however, we support only _expansion_ of *
//*    these archives, NOT creation or update. For Create and Update, these    *
//*    files are handled as ordinary source data files.                        *
//*    -- The user can certainly create or update OD/OX/EPUB archives, but     *
//*       he/she/it must first rename the existing file as a .ZIP archive or   *
//*       must rename the created archive as an OpenDoc/OpenXML document.      *
//*    -- As with all supported archive types, we also implement viewing the   *
//*       contents of OpenDoc/OpenXML archives (see ViewFileContents()).       *
//* D) It is stupid (but not forbidden) to add an archive file to an existing  *
//*    archive because due to overhead the resulting archive will always be    *
//*    larger than if the raw data were archived.                              *
//*    -- 'tar' will include an existing zip archive into the target archive,  *
//*        but the overhead is significant.                                    *
//*    -- In our tests, 'tar' will intelligently refuse to add an archive to   *
//*       itself. If this happens, the error message will appear in the log.   *
//*    -- 'zip' will "store" (no additional deflation) an existing zip archive *
//*       into the target archive, causing minimal overhead.                   *
//*    -- In our tests, 'zip' will semi-intelligently refuse to add an archive *
//*       to itself. (Actually, it goes ape-shit.)                             *
//*    -- In any case, it is more user-friendly to alert user before           *
//*       performing an update which we know will fail. (see ScanClipboard())  *
//*                                                                            *
//******************************************************************************

void FileDlg::Archive ( void )
{
   arcType cbaType = atNONE,     // if archive (only archive) on clipboard, its type
           hlaType = atNONE ;    // if highlighted file is an archive, its type
   bool hlSelect   = false,      // 'true' if highlighted file is 'selected'
        awSelect   = false,      // 'true' if any 'selections' in active window
        freshCbData  = false,    // 'true' if this method (not user) places data on the clipboard
        complexQuery = false,    // 'true' if user's setup is ambiguous
        opAbort = false ;        // 'true' if forced or user-specified abort

   //* Operational parameters.                                     *
   //* -- Primary operation is archive, but the operation sub-type *
   //*    is not yet known.                                        *
   //* -- Target directory for creation/update/expansion is our    *
   //*    current directory.                                       *
   BSet    bkSet ;
   bkSet.btype = bsetArchive ;
   bkSet.btrg = this->currDir ;

   UINT   cbFiles = this->ClipboardFiles () ; // number of files on clipboard

   //* Get stats and pathspec for highlighted item to determine     *
   //* whether it is an archive file. Note that if highlighted file *
   //* is also 'selected', it will be interpreted only as source    *
   //* data, NOT as an archive, regardless of filename extension.   *
   tnFName tnf ;
   this->GetStats ( tnf ) ;
   gString hlPath( "%S/%s", bkSet.btrg.gstr(), tnf.fName ) ;
   if ( !(hlSelect = this->IsSelected ()) )
      hlaType = this->ArchiveTarget ( hlPath ) ;

   //* If there are 'selected' data, copy to clipboard.*
   //* (Note: These data may already be on clipboard.) *
   if ( (awSelect = this->IsSelected ( true )) )
   {
      this->Copy2Clipboard () ;
      cbFiles = this->ClipboardFiles () ;
      hlSelect = awSelect = false ; // ignore 'selected' data from this point onward
      freshCbData = true ;          // indicate that CB data are fresh
   }

   //* Else, there are no 'selections'. If highlighted *
   //* file is an archive AND if clipboard is empty,   *
   //* copy the archive to the clipboard.              *
   else if ( (hlaType != atNONE) && (cbFiles == ZERO) )
   {
      this->Copy2Clipboard () ;
      cbFiles = this->ClipboardFiles () ;
      hlaType = atNONE ;            // reset highlighted-file type
      freshCbData = true ;          // indicate that CB data are fresh
   }

   //* If user has previously placed data on the clipboard, *
   //* the data may be intended for this operation, or it   *
   //* may be stale data from a previous operation.         *
   //* If no target archive highlighted, then we can assume *
   //* clipboard data is for archive creation (see below).  *
   //* Otherwise, operation setup is ambiguous, query user. *
   else if ( (cbFiles > ZERO) && (hlaType != atNONE) )
   {
      bkSet.barc = hlPath ;      // path of highlighted file
      complexQuery = true ;
   }

   //* If clipboard contains ONLY a single archive file, *
   //* get its path and compression type.                *
   cbaType = this->ArchiveOnClipboard ( bkSet.barc ) ;

   if ( !complexQuery )
   {
      //* Determine whether the operation can be identified as an 'Expand' *
      //* operation i.e. the only file on the clipboard is an archive file *
      //* AND the highlighted file IS NOT an archive.                      *
      if ( (cbaType != atNONE) && (hlaType == atNONE) )
      {
         bkSet.btype = bsetAExpand ;   // operation is Expand
      }

      //* If operation sub-type not yet set, determine whether the operation  *
      //* can be identified as a 'Create' operation.                          *
      //* If there are data on clipboard, but no source or target archive     *
      //* specified, then data are source for a Create operation.             *
      else if (   (bkSet.btype == bsetArchive) && (cbFiles > ZERO)
               && (cbaType == atNONE) && (hlaType == atNONE) )
      {
         bkSet.btype = bsetACreate ;         // operation is Create
      }

      //* If operation sub-type not yet set, determine whether the operation  *
      //* can be identified as an 'Update' operation.                         *
      //* If highlighted file is the target archive and clipboard contains    *
      //* _fresh_ source data.                                                *
      //* Note: Office and ePub documents cannot be targets of archive update.*
      //* Note: For Update operation, target archive must be either an        *
      //*       uncompressed TAR archive or a ZIP archive. If target archive  *
      //*       is a _compressed_ Tar archive, then an error message will be  *
      //*       displayed by Archive_Prompt().                                *
      //* Note: An archive cannot be added to itself. If the highlighted file *
      //*       is ALSO on the clipboard, then an error message will be       *
      //*       displayed by Archive_Prompt().                                *
      else if ( (bkSet.btype == bsetArchive) && 
                ((hlaType != atNONE) && 
                 (hlaType != atODOC && hlaType != atOXML && hlaType != atEPUB)) &&
                (cbFiles > ZERO) && (freshCbData != false) )
      {
         bkSet.barc = hlPath ;            // path of highlighted archive
         bkSet.btype = bsetAUpdate ;      // operation is Update
      }

      //* Operation sub-type cannot be programatically determined. *
      //* User will be asked how to proceed.                       *
      else
         complexQuery = true ;
   }

   //* If operation sub-type has not been automatically determined. *
   if ( bkSet.btype == bsetArchive )
   {
      //* An archive file, source data, or both have been specified; *
      //* however, we are unable to automatically determine the      *
      //* operation to be performed.                                 *
      //* Invoke a dialog which asks user how to proceed.            *
      bsetType auqStatus = 
            this->Archive_UpdateQuery ( bkSet, hlPath, hlaType, cbaType, cbFiles ) ;

      switch ( auqStatus )
      {
         case bsetACreate:
            bkSet.barc.clear() ;          // clear archive name
         case bsetAExpand:
         case bsetAUpdate:
            bkSet.btype = auqStatus ;     // set operation sub-type
            break ;
         default:
            opAbort = true ;              // abort the operation
            break ;
      } ;
   }

   //* If all parameters verified, launch the operation dialog.*
   if ( !opAbort )
   {
      this->Archive_Prompt ( bkSet ) ;
   }

}  //* End Archive() *

//*************************
//*    Archive_Prompt     *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* Caller is unable to programatically determine what operation to perform.   *
//*                                                                            *
//* Query the user to determine whether to:                                    *
//*  1) create an archive using the clipboard data                             *
//*  2) update existing (highlighted) archive using the clipboard data         *
//*  3) expand an existing archive, either:                                    *
//*     a) expand highlighted archive, ignoring any clipboard data             *
//*     b) expand the clipboard archive (if any), ignoring highlighted archive *
//*        (Currently, we do not offer this option.)
//*  4) cancel the operation                                                   *
//*                                                                            *
//*                                                                            *
//* Input  : bkSet : operational parameters                                    *
//*                  'barc' is the filespec for an existing archive (if any)   *
//*                                                                            *
//*                                                                            *
//* Returns: OK  if successful                                                 *
//*          ERR if error or if user abort                                     *
//******************************************************************************
//* Notes:                                                                     *
//* ------                                                                     *
//* -- For all operations, the source data are not modified.                   *
//*    -- Clipboard source data are not modified.                              *
//*    -- Data are extracted from an archive without modifying the archive.    *
//* -- If the Update target archive is not an uncompressed Tar archive or a    *
//*    Zip archive, it is an error because compressed Tar archives cannot be   *
//*    updated.                                                                 *
//* -- If the Update target archive is ALSO on the clipboard. it is an error   *
//*    because an archive may not be added to itself.                          *
//*    However, if the source archive is among multiple items on the clipboard,*
//*    we don't check for that, and 'tar' will likely complain and skip it.    *
//* -- Reporting errors in either creation or expansion is a difficult task    *
//*    because the warning/error messages will be mostly lost to us.           *
//* -- If an archive has been created (updated), then we highlight it before   *
//*    returning to caller.                                                    *
//* -- There is currently no plan for deleting files from an existing archive. *
//* -- There is currently no plan for doing a partial expansion of an existing *
//*    archive. The user interface for such an operation would be a major      *
//*    effort.                                                                 *
//*                                                                            *
//******************************************************************************

short FileDlg::Archive_Prompt ( BSet& bkSet )
{
   const char* arcTitle[] = 
   {
      "  Create an Archive  ",
      "  Update an Archive  ",
      "  Expand an Archive  "
   } ;
   const char* arcInstructions[] = 
   {
      "Create an archive file from the selected data.",
      " Append the selected data to specified archive.",
      " Expand the specified archive to current directory."
   } ;
   const char* pbText[] = 
   {
      " ^CREATE ",
      " ^UPDATE ",
      " ^EXPAND "
   } ;
   const char* badName[] = 
   {
      " Archive Already Exists! ",
      "                         "
   } ;
   const char* noSourceMsg = "  To create an archive, first 'select' data to be\n"
                             "  archived, or place source data on the clipboard." ;
   const short ddITEMS = 12,        // items in ddText
               ddWIDTH = 26,        // width of Dropdown control text
               firstZIP = 9,        // first Zip utility option
               lastZIP  = 11 ;      // last Zip utility option
   const char  ddText[ddITEMS][ddWIDTH + 1] = // Text data for logDD
   {
      ".tar          None        ",
      ".tar.bz2      bzip2       ", // (default)
      ".tbz          bzip2       ",
      ".tar.gz       gzip        ",
      ".tgz          gzip        ",
      ".tar.lzma     lzma        ",
      ".tlz          lzma        ",
      ".tar.lzo      lzop        ",
      ".tar.xz       xz          ",
      ".zip          zip(0,store)",
      ".zip          zip(6)      ",
      ".zip          zip(9)      ",
   } ;

   UINT   cbFiles = this->ClipboardFiles () ;   // clipboard files
   UINT64 cbBytes = this->ClipboardSize () ;    // clipboard total size
   gString gsOut,                               // formatted output
           arcName,                             // name of existing archive (if any)
           dispName,                            // archive name displayed in textbox
           extName,                             // archive filename extension (complex)
           gsError ;                            // error message
   short tbWidth = (bkSet.btype == bsetACreate) ? 32 : 40, // textbox width
         status = OK ;                          // return value
   bool abort = false ;                         // 'true' if user abort or serious error

   this->SetErrorCode ( ecNOERROR ) ;  // clear any previous error code

   //* For update operations, verify that the target       *
   //* archive is one of the types that can be updated and *
   //* that source data does not include target archive.   *
   if ( bkSet.btype == bsetAUpdate )
   {
      arcType aType = this->ArchiveTarget ( bkSet.barc ) ;
      if ( !((aType == atTAR) || (aType == atZIP)) )
      {
         gsError = "    Only uncompressed \".tar\" archives and \".zip\"\n"
                   "    archives may be updated.  Unable to proceed." ;
         abort = true ;    // cannot complete the operation
      }
      //* Scan the clipboard data to be sure that user is not trying to  *
      //* add the target archive to itself.                              *
      else if ( (this->ScanClipboard ( bkSet.barc )) )
      {
         gsError = "   You are attempting to add an archive to itself.\n"
                   "                 Unable to proceed." ;
         abort = true ;
      }
   }


   //* Index the appropriate title and instruction text *
   short titleIndex = (bkSet.btype == bsetACreate) ? ZERO : 
                      ((bkSet.btype == bsetAUpdate) ? 1 : 2) ;

   //* Define the dialog window *
   const short dlgROWS = minfitROWS + 1,
               dlgCOLS = minfitCOLS,
               dlgYPOS = this->fulY + 1,
               dlgXPOS = this->fulX + (this->fMaxX / 2) - (dlgCOLS / 2) ;
   const attr_t dColor = this->cs.sd,                 // dialog text
                hColor = this->cs.em,                 // highlighted text
                xColor = this->cs.sb & ~ncbATTR ;     // greyed-out text
   attr_t monoColor[2] = { attrDFLT, this->cs.mn } ;  // dropdown item color

   //* List of controls in this dialog *
   enum arControls : short
   {
      okPB = ZERO, arTB, arDD, canPB, hlpPB, arCONTROLS
   } ;

   //** Define the dialog controls **
   InitCtrl ic[arCONTROLS] = 
   {
   {  //* 'OK' pushbutton - - - - - - - - - - - - - - - - - - - - - -     okPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dlgROWS - 2),           // ulY:       upper left corner in Y
      6,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      8,                            // cols:      control columns
      pbText[titleIndex],           // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[arTB]                     // nextCtrl:  link in next structure
   },
   {  //* 'ARCHIVE NAME' Textbox - - - - - - - - - - - - - - - - - - - -  arTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      2,                            // ulY:       upper left corner in Y
      15,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      tbWidth,                      // cols:      control columns
      dispName.ustr(),              // dispText:  
      this->cs.tn,                  // nColor:    non-focus color
      this->cs.tf,                  // fColor:    focus color
      #if LINUX_SPECIAL_CHARS != 0
      tbFileLinux,                  // filter: valid filename chars (incl. Linux "special")
      #else    // BASIC FILENAME FILTER
      tbFileName,                   // filter:    valid filename characters
      #endif   // BASIC FILENAME FILTER
      "Archive ^Name",              // label:     
      ZERO,                         // labY:      
      -13,                          // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[arDD]                     // nextCtrl:  link in next structure
   },
   {  //* 'COMPRESSION TYPE' Dropdown  - - - - - - - - - - - - - - - - -  arDD *
      dctDROPDOWN,                  // type:      define a dropdown control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      short(ic[arTB].ulY + 4),      // ulY:       upper left corner in Y
      short(dlgCOLS / 2 ),          // ulX:       upper left corner in X
      8,                            // lines:     control lines
      short(ddWIDTH + 2),           // cols:      control columns
      (const char*)&ddText,         // dispText
      this->cs.sd,                  // nColor:    non-focus border color
      this->cs.wr,                  // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     
      ZERO,                         // labY:      offset from control's ulY
      ZERO,                         // labX       offset from control's ulX
      ddBoxDOWN,                    // exType:    expansion direction
      ddITEMS,                      // scrItems:  number of elements in text/color arrays
      1,                            // scrSel:    index of initial highlighted element
      monoColor,                    // scrColor:  single-color data display
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[canPB]                    // nextCtrl:  link in next structure
   },
   {  //* 'CANCEL' pushbutton - - - - - - - - - - - - - - - - - - - - -  canPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[okPB].ulY,                 // ulY:       upper left corner in Y
      short(ic[okPB].ulX + ic[okPB].cols + 3), // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      8,                            // cols:      control columns
      " C^ANCEL ",                  // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[hlpPB]                    // nextCtrl:  link in next structure
   },
   {  //* 'HELP' pushbutton - - - - - - - - - - - - - - - - - - - - -    hlpPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[canPB].ulY,                // ulY:       upper left corner in Y
      short(ic[canPB].ulX + ic[canPB].cols + 5), // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      8,                            // cols:      control columns
      "  ^HELP  ",                  // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      attr_t((this->cs.scheme == ncbcGR) ? nc.brR : nc.grR), // fColor: focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
   } ;

   //* If creating a new archive, offer a default archive name.*
   //* Else, parse the existing archive name.                  *
   this->GetLocalTime ( bkSet.btime ) ;      // get the local time
   this->DecodeLocalTime ( bkSet.btime ) ;   // decode timestamp
   if ( bkSet.btype == bsetACreate )
   {
      arcName.compose( "%04hd_%02hd_%02hd.tar.bz2",
                       &bkSet.btime.year, &bkSet.btime.month, &bkSet.btime.date ) ;
      ParseArchiveName ( arcName, dispName, extName ) ;
   }
   else
   {
      gString baseName ;
      this->fmPtr->ExtractFilename ( gsOut, bkSet.barc ) ;
      ParseArchiveName ( gsOut, baseName, extName ) ;
   }

   //* If expanding or updating an archive OR abort, do some housekeeping. *
   if ( bkSet.btype == bsetAExpand || bkSet.btype == bsetAUpdate || abort )
   {
      //* Display archive name and set Textbox as read-only *
      if ( bkSet.barc.gschars() > 1 )
      {
         short nindex = (bkSet.barc.findlast( fSLASH )) + 1 ;
         arcName = &bkSet.barc.gstr()[nindex] ;
         dispName = arcName ;
      }
      ic[arTB].active = false ;

      //* Set the selected item of the Dropdown to match the archive type.*
      // Programmer's Note: Because Zip archives may contain mixed compression 
      // types, we assume the default (deflate 6).
      bool ddIndexSet = false ;
      for ( short i = ZERO ; i < ddITEMS ; ++i )
      {
         if ( (extName.compare( ddText[i], false, (extName.gschars() - 1) )) == ZERO )
         {
            if ( (extName.find( L".zip" )) == ZERO )
               ++i ;
            ic[arDD].scrSel = i ;
            ddIndexSet = true ;
            break ;
         }
      }
      // Programmer's Note: We may see an OpenDocument/OpenXML/ePub file here. 
      // If so, it is not in the list of options, so set index to ".zip(6)".
      if ( ! ddIndexSet )
         ic[arDD].scrSel = (firstZIP + 1) ;

      //* Disable the compression-type Dropdown.*
      ic[arDD].nColor = xColor ;
      monoColor[1]    = xColor ;
      ic[arDD].active = false ;

      if ( abort )
      {  //* In case user tries again, next time the clipboard will be clear *
         this->ClearClipboard () ;
         if ( gsError.gschars() == 1 )
            gsError = noSourceMsg ;    // we have no source data to work with
      }
   }

   //* Save the parent dialog's display data *
   this->dPtr->SetDialogObscured () ;

   //* Open the interface dialog *
   //* Initial parameters for dialog window *
   InitNcDialog dInit( dlgROWS,        // number of display lines
                       dlgCOLS,        // number of display columns
                       dlgYPOS,        // Y offset from upper-left of terminal 
                       dlgXPOS,        // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Set the dialog title *
      dp->SetDialogTitle ( arcTitle[titleIndex], this->cs.em ) ;

      winPos wp( 1, 2 ) ;
      winPos wpe( short(ic[okPB].ulY - 1), short(ic[okPB].ulX - 3) ) ;
      winPos wpExt( ic[arTB].ulY, short(ic[arTB].ulX + ic[arTB].cols) ) ;

      dp->WriteString ( wp, arcInstructions[titleIndex], hColor ) ;
      wp = { short(ic[arTB].ulY + 1), short(ic[arTB].ulX) } ;
      if ( bkSet.btype == bsetACreate )
      {
         dp->SetTextboxCursor ( arTB, tbcpRIGHTJUST ) ;
         dp->WriteString ( wp, "(base name only, no extension)", dColor ) ;
      }
      attr_t txtColor = ic[arDD].nColor ;    // info text color attribute

      wp = { short(ic[arDD].ulY - 1), short(ic[arDD].ulX + 1) } ;
      wp = dp->WriteString ( wp, "EXTENSION", txtColor | ncuATTR ) ;
      wp = dp->WriteString ( wp, "     ", dColor ) ;
      wp = dp->WriteString ( wp, "COMPRESSION\n", txtColor | ncuATTR ) ;
      wp = { short(ic[arDD].ulY + 3), ic[arDD].ulX } ;
      wp = dp->WriteParagraph ( wp,
               "Filename extension indicates\n"
               "    type of compression.", txtColor ) ;

      //* Position of stats messages *
      wp.ypos = ic[arTB].ulY + ((bkSet.btype == bsetAUpdate) ? 3 : 4) ;
      wp.xpos = 2 ;

      //* For create, display filename extension *
      if ( bkSet.btype == bsetACreate )
      {
         dp->ClearLine ( wpExt.ypos, false, wpExt.xpos ) ;
         dp->WriteString ( wpExt, extName, hColor ) ;
      }

      if ( ! abort )
      {
         //* Get a copy of the path to the displayed data *
         this->fmPtr->GetCurrDir ( bkSet.btrg ) ;

         //* Get a copy of the path to the clipboard source data (if any) *
         this->ClipboardPath ( bkSet.bsrc ) ;

         //* Create a filespec for the log file *
         this->fmPtr->CreateTempname ( bkSet.blog ) ;

         if ( bkSet.btype == bsetAUpdate || bkSet.btype == bsetAExpand )
         {
            tnFName tnf ;              // if expanding an archive, archive stats
            UINT   aFiles ;            // number of files in archive
            UINT64 aBytes ;            // size of expanded archive data (approximate)
            this->ArchiveSummary ( bkSet.barc, tnf, aFiles, aBytes ) ;
            gsOut.formatInt ( aFiles, 11, true ) ;
            wp = dp->WriteString ( wp, "Archive Files: ", hColor ) ;
            dp->WriteString ( wp, gsOut, dColor ) ;
            ++wp.ypos ; wp.xpos = 2 ;
            wp = dp->WriteString ( wp, "Archive Size : ", hColor ) ;
            gsOut.formatInt( tnf.fBytes, 5, true, false, false, fiKb ) ;
            dp->WriteString ( wp, gsOut, dColor ) ;
            ++wp.ypos ; wp.xpos = 2 ;
            gsOut.formatInt ( aBytes, 5, true, false, false, fiKb ) ;
            wp = dp->WriteString ( wp, "Expanded Size: ", hColor ) ;
            dp->WriteString ( wp, gsOut, dColor ) ;
            wp.ypos += 2 ; wp.xpos = 2 ;
         }

         if ( bkSet.btype == bsetACreate || bkSet.btype == bsetAUpdate )
         {
            gsOut.formatInt( cbFiles, 11, true ) ;
            wp = dp->WriteString ( wp, "Source Files: ", hColor ) ;
            dp->WriteString ( wp.ypos++, wp.xpos, gsOut, dColor ) ;
            wp.xpos = 2 ;
            gsOut.formatInt( cbBytes, 5, true, false, false, fiKb ) ;
            wp = dp->WriteString ( wp, "Source Bytes: ", hColor ) ;
            dp->WriteString ( wp, gsOut, dColor ) ;
         }
         else if ( (bkSet.btype == bsetAExpand) && this->fileCount > ZERO )
         {  //* If the target directory contains data other than the archive *
            //* itself, caution the user about potential data loss.          *
            bool warn = bool(this->fileCount > 1) ;
            if ( ! warn )
            {
               //* If there is a file in the target directory, *
               //* but it is not the archive file, then warn.  *
               gString aPath ;
               this->fmPtr->ExtractPathname ( aPath, bkSet.barc ) ;
               if ( (aPath.compare( this->currDir, true )) != ZERO )
                  warn = true ;
            }
            if ( warn )
            {
               dp->ClearLine ( wp.ypos - 1 ) ;
               dp->ClearLine ( wp.ypos ) ;
               wp = dp->WriteString ( wp.ypos, wp.xpos - 1, "Caution: ", hColor ) ;
               wp = dp->WriteParagraph ( wp,
                  "Any existing target items will be overwritten.\n", dColor ) ;
               wp.xpos = 2 ;
               dp->WriteParagraph ( wp,
                  "To avoid potential data loss, it is recommended that\n"
                  "archives always be expanded into an empty directory.", dColor ) ;
            }
         }
      }
      else     // operation cannot proceed
      {
         dp->WriteParagraph ( dlgROWS - 4, 2, gsError, hColor ) ;
         dp->NextControl () ;          // move focus to 'Cancel' button
         for ( short i = ZERO ; i < arCONTROLS ; ++i ) // deactive remaining controls
            if ( i != canPB )
               dp->ControlActive ( i, false ) ;
      }

      dp->RefreshWin () ;                 // make the dialog visible

      uiInfo Info ;                       // user interface data returned here
      short  icIndex = ZERO ;             // index of control with input focus
      bool   done = false ;               // loop control
      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;

            //* If a Pushbutton was pressed *
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == okPB ) // 'OK' Pushbutton
               {  //* If operation is to create an archive, retrieve *
                  //* and validate the target archive name.          *
                  //* Otherwise, we already have the target path.    *
                  if ( bkSet.btype == bsetACreate )
                  {
                     //* Combine user-provided base name with selected *
                     //* filename extension. Create the target filespec*
                     //* and verify that target does not already exist.*
                     dp->GetTextboxText ( arTB, arcName ) ;
                     arcName.append( extName.gstr() ) ;

                     #if LINUX_SPECIAL_CHARS != 0
                     if ( (this->gnfSpecialCharsQuery ( arcName, dp, arTB, tbFileName )) )
                     {
                        bkSet.barc.compose( "%S/%S", bkSet.btrg.gstr(), arcName.gstr() ) ;
                        fmFType ft ;
                        if ( (this->TargetExists ( bkSet.barc, ft )) )
                           dp->WriteString ( wpe, badName[0], dColor | ncbATTR, true ) ;
                        else
                        {  //* Clear the message space.*
                           dp->WriteString ( wpe, badName[1], dColor, true ) ;
                           done = true ;
                        }
                     }
                     else     // continue archive setup
                        ;     // focus returns to archive name Textbox
                     #endif   // LINUX_SPECIAL_CHARS
                  }
                  else
                     done = true ;
               }
               else if ( Info.ctrlIndex == hlpPB )
               {
                  dp->SetDialogObscured () ;
                  this->CallContextHelp ( (bkSet.btype == bsetACreate) ? cxtARCHIVE_C :
                                          (bkSet.btype == bsetAUpdate) ? cxtARCHIVE_U : 
                                          cxtARCHIVE_E ) ;
                  this->dPtr->RefreshWin () ;
                  this->dPtr->SetDialogObscured () ;
                  dp->RefreshWin () ;
               }
               else if ( Info.ctrlIndex == canPB )
                  done = abort = true ;
            }
         }
         else if ( ic[icIndex].type == dctTEXTBOX )
         {
            Info.viaHotkey = false ;   // discard hotkey data
            icIndex = dp->EditTextbox ( Info ) ;

            if ( Info.dataMod != false )
            {
               //* Combine user-provided base name with selected *
               //* filename extension. Create the target filespec*
               //* and verify that target does not already exist.*
               dp->GetTextboxText ( arTB, gsOut ) ;
               gsOut.append( extName.gstr() ) ;
               gString gstmp( "%S/%S", bkSet.btrg.gstr(), gsOut.gstr() ) ;
               fmFType ft ;
               if ( (this->TargetExists ( gstmp, ft )) )
                  dp->WriteString ( wpe, badName[0], dColor | ncbATTR, true ) ;
               else  //* Clear the message space.*
                  dp->WriteString ( wpe, badName[1], dColor, true ) ;
            }
         }
         //* If focus is currently on a Dropdown   *
         // Programmer's Note: Dropdown control is active ONLY for 'create' operation.
         if ( ic[icIndex].type == dctDROPDOWN )
         {
            Info.viaHotkey = false ;
            icIndex = dp->EditDropdown ( Info ) ;

            //* If user has selected a new archive type, *
            //* update the displayed filename extension. *
            if ( Info.dataMod != false )
            {
               extName = ddText[Info.selMember] ;              // get display text
               extName.limitChars( (extName.find( L' ' )) ) ;  // truncate to extension only
               bkSet.zipLevel = (Info.selMember == firstZIP) ? ZERO :   // set Zip level
                                (Info.selMember == lastZIP) ? 9 : zipDEFAULT ;
               dp->ClearLine ( wpExt.ypos, false, wpExt.xpos ) ;// update displayed extension
               dp->WriteString ( wpExt, extName, hColor, true ) ;

               //* For user-friendliness, position *
               //* focus for "CREATE" Pushbutton.  *
               if ( Info.keyIn != nckSTAB )
                  while ( (icIndex = dp->PrevControl ()) != hlpPB ) ;
            }
         }
         //* Move input focus to next/previous control *
         if ( ! done && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }        // while()
   }
   if ( dp != NULL )                      // close the window
      delete ( dp ) ;

   this->dPtr->RefreshWin () ;   // Restore parent dialog display

   //* If no setup errors, create and execute the command *
   if ( ! abort )
   {
      status = this->Archive_Setup ( bkSet ) ;

      //* Re-read the target directory, contents may have been changed *
      this->RefreshCurrDir () ;

      //* Summary report *
      this->Archive_Summary ( bkSet, status ) ;

      //* If an archive was created/updated, highlight it *
      if ( bkSet.btype == bsetACreate || bkSet.btype == bsetAUpdate )
         this->HilightItemByName ( arcName ) ;

      //* Delete the temporary logfile *
      this->DeleteFile ( bkSet.blog ) ;

   }
   return status ;

}  //* End Archive_Prompt() *

//*************************
//*   ParseArchiveName    *
//*************************
//******************************************************************************
//* Non-member method:                                                         *
//* ------------------                                                         *
//*                                                                            *
//* Input  : gsSrc   : (by reference) full archive filename (not filespec)     *
//*          gsFName : (by reference) receives base filename                   *
//*          gsFExt  : (by reference) receives filename extension              *
//*                                                                            *
//* Returns: nothing  (We "trust" the caller to send us valid data.)           *
//******************************************************************************

static void ParseArchiveName ( const gString& gsSrc, gString& gsFName, gString& gsFExt )
{
   short indx ;
   if ( (indx = gsSrc.findlast( L'.' )) > ZERO )
   {
      //* If a Zip archive filename *
      if ( (gsSrc.compare( L".zip", false, 4, indx)) == ZERO )
      {
         gsFExt = &gsSrc.gstr()[indx] ;
      }
      //* If a complex Tar archize filename *
      else if ( (indx >= 4) && 
                ((gsSrc.compare( L".tar", true, 4, (indx - 4) )) == ZERO) )
      {
         indx -= 4 ;
         gsFExt = &gsSrc.gstr()[indx] ;
      }
      //* Short-form Tar archive filename   *
      //* (or unrelated filename extension) *
      else
      {
         gsFExt = &gsSrc.gstr()[indx] ;
      }
      gsFName = gsSrc ;
      gsFName.limitChars( indx ) ;
   }

}  //* End ParseArchiveName() *

//*************************
//*     Archive_Setup     *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* Called by Archive_Prompt() when user launches the operation.               *
//*                                                                            *
//* Input  : bkSet    : (by reference)                                         *
//*                     'bsrc' contains filespec for source data directory     *
//*                            (if any)                                        *
//*                     'btrg' contains filespec of target directory           *
//*                     'btype' indicates the Archive operation sub-type       *
//*                     'barc' contains filespec of archive file               *
//*                            (source or target)                              *
//*                     'btime' contains system local timestamp                *
//*                                                                            *
//* Returns: OK  if successful                                                 *
//*              bkSet members receive accumulated data:                       *
//*               'ftotal'   receives number of source files on clipboard      *
//*               'fupdated' receives number of files added to archive         *
//*               'ferrors'  receives number of errors during processing       *
//*          ERR if errors encountered                                         *
//******************************************************************************
//* NOTES:                                                                     *
//* ==============                                                             *
//*                                                                            *
//******************************************************************************

short FileDlg::Archive_Setup ( BSet& bkSet )
{
      short status = OK ;

      //* Create a temp-file log to record the operation *
      this->bkOpenLogfile ( bkSet ) ;

      #if ENABLE_DEBUGGING_CODE != 0 && enableDBBAK != 0
      dbgs.compose( "\n%s Archive_Prompt\n---------------------", debugHDR ) ;
      this->bkWriteLogfile ( dbgs ) ;
      #endif   // ENABLE_DEBUGGING_CODE && enableDBBAK

      //* Because we do not know what the current-working-directory (CWD) is,  *
      //* we must explicity set the CWD before the operation begins. Then, we  *
      //* set it back after the operation.                                     *
      //* a) For create and append operations, the CWD must be the directory   *
      //*    where the source data live: 'bkSet.bsrc'                          *
      //* b) For expand operations, the CWD must be the directory where the    *
      //*    extracted data will be written: 'bkSet.btrg'                      *
      gString currCwd ;

      //* Create an archive *
      if ( bkSet.btype == bsetACreate )
      {
         this->fmPtr->SetCWD ( bkSet.bsrc, &currCwd ) ;  // set source directory
         this->Archive_Create ( bkSet ) ;
      }

      //* If target archive exists, update it.                 *
      //* For 'tar', append new data (existing data unchanged).*
      //* For 'zip', append new data and update existing data. *
      else if ( bkSet.btype == bsetAUpdate )
      {
         this->fmPtr->SetCWD ( bkSet.bsrc, &currCwd ) ;  // set source directory
         this->Archive_Append ( bkSet ) ;
      }

      //* Extract archive data to current-working-directory *
      else if ( bkSet.btype == bsetAExpand )
      {
         this->fmPtr->SetCWD ( bkSet.btrg, &currCwd ) ;  // set target directory
         this->Archive_Expand ( bkSet ) ;
      }

      //* Return to the previous CWD *
      this->fmPtr->SetCWD ( currCwd ) ;

      //* Close the log file *
      this->bkCloseLogfile () ;

   return status ;

}  //* End Archive_Setup() *

//*************************
//*    Archive_Create     *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* Create an archive using the data on the clipboard as the source.           *
//*                                                                            *
//*                                                                            *
//* Input  : bkSet   : (by reference)                                          *
//*                    'barc' contains filespec of target archive              *
//*                                                                            *
//* Returns: OK if successful, or ERR if one or more errors detected           *
//*          bkSet members receive accumulated data:                           *
//*           'ftotal'   receives number of source files on clipboard          *
//*           'fupdated' receives number of files added to archive             *
//*           'ferrors'  receives number of errors during processing           *
//******************************************************************************
//* NOTES:                                                                     *
//* ==============                                                             *
//*                                                                            *
//* 1) Create an archive:                                                      *
//*    Tar Archives:                                                           *
//*    a) Archives are created using the default formatting specified at 'tar' *
//*       compile time (probably posix). See 'tar' documentation for details.  *
//*    b) The '-a' option is used to enable automatic compression based on     *
//*       the target filename's extension.                                     *
//*    c) See acCreateTarArchive() for an explanation of all invocation        *
//*       options used.                                                        *
//*    d) We record the operation to the log file using the double-v ('-vv')   *
//*       verbose option.                                                      *
//*    e) Verification is done by comparing the post-creation archive summary  *
//*       with the contents of the clipboard.                                  *
//*       -- Note that verification using the '--verify' ('-W') option is not  *
//*          used at this time due to difficulties in parsing the output AND   *
//*          for symmetry with the archive update operations which cannot use  *
//*          the '-W' option.                                                  *
//*                                                                            *
//*    Zip Archives:                                                           *
//*    a) Archive is created using the 'deflate' compression level option      *
//*       specified by bkSet.zipLevel.                                         *
//*    b) ZIP archives may be updated under some circumstances so we allow     *
//*       the user to update an existing ZIP target. Note that internally, zip *
//*       creates the updated file as a temp file, and replaces the specified  *
//*       target only if there are no errors.                                  *
//*    c) See acCreateZipArchive() for an explanation of all invocation        *
//*       options used.                                                        *
//*                                                                            *
//* 2) Note on visual progress indicator:                                      *
//*    Because archive operations seldom take more than 2-3 seconds (unless    *
//*    the target filesystem is a flash memory device or CD/DVD), we do not    *
//*    invoke the Progress Bar for archive operations. Instead, we display the *
//*    simple default 'Processing' message. The reason for this is that the    *
//*    archive operation, unlike Backup and Synch operations is monolythic,    *
//*    so it is inconvenient to identify the percentage of the operation which *
//*    is complete at any given moment.                                        *
//*                                                                            *
//******************************************************************************

short FileDlg::Archive_Create ( BSet& bkSet )
{
   arcType aType = this->ArchiveTarget ( bkSet.barc ) ;  // archive type
   short status = OK ;              // return value

   //* Verify that the target archive type is supported.*
   if ( (aType != atNONE) && (aType != atODOC) && (aType != atOXML) )
   {
      this->ProcessCWD () ;            // display the default processing message

      //********************************
      //* If target is a 'zip' archive.*
      //********************************
      if ( aType == atZIP )
      {
         status = this->acCreateZipArchive ( bkSet ) ;
      }     // Zip Archive

      //********************************
      //* If target is a 'tar' archive.*
      //********************************
      else
      {
         status = this->acCreateTarArchive ( bkSet ) ;
      }     // Tar Archive

      this->ProcessCWD ( true ) ;   // clear the processing message
   }
   else // (this should never happen because caller has control of user input)
      status = ERR ;

   return status ;

}  //* End Archive_Create() *

//*************************
//*  acCreateTarArchive   *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* Create a 'tar' archive using the data on the clipboard.                    *
//*                                                                            *
//* Input  : bkSet   : (by reference)                                          *
//*                    'barc' contains filespec of target archive              *
//*                    Operational log file ('blog') has been initialized for  *
//*                    appending log data.                                     *
//*          update  : (optional, 'false' by default)                          *
//*                    if 'false', create a new archive                        *
//*                    if 'true',  update an existing archive                  *
//*                                                                            *
//* Returns: OK if successful, or ERR if one or more errors detected           *
//*          bkSet members receive accumulated data:                           *
//*           'ftotal'   receives number of source files (clipboard contents)  *
//*           'fupdated' receives number of files added to archive             *
//*           'ferrors'  receives number of errors during processing           *
//*           'fskipped' always ZERO for archive operations (unless operation  *
//*                      aborted, in which case this method won't be called)   *
//*           'blog'     formatted log data are inserted into this file        *
//******************************************************************************
//* Notes:                                                                     *
//*                                                                            *
//* Tar Archives:                                                              *
//* -- By default, archive files are created using the 'tar' utility with      *
//*    optional data compression. Source data are not modified.                *
//*                                                                            *
//* -- There are MANY options for the 'tar' utility, most of which are for     *
//*    very specific circumstances. We use only a basic set of options,        *
//*    related only to creating the archive, with no embedded expansion        *
//*    instructions.                                                           *
//*    -- Note that we don't give a rat's ass about actually writing to        *
//*       physical tape drives. Tape operations, if used at all, are done      *
//*       only on very large corporate systems which have their own IT         *
//*       departments; and while we respect the expertise of corporate admin   *
//*       weenies, they have their own set of archive tools, and don't need    *
//*       our support.                                                         *
//* -- Invocation options:                                                     *
//*    '-c' (--create) operation is performed. Data are never appended to an   *
//*         existing archive. (Caller has verified that the archive name is    *
//*         unique to the target directory.                                    *
//*    '-a' (--auto-compress) option is used to optionally compress the        *
//*       archive based on the target's filename.                              *
//*    '-v' (--verbose) option reports details of the operation.               *
//*         Note that we use '-vv' to give greater detail on errors, etc.      *
//*    '--numeric-owner' option uses UID and GID numbers rather than the       *
//*         name of the file owner.                                            *
//*    '--file=ARCHIVE' option specifies the path/filename of the archive.     *
//*    '--files-from=FILE' ('-T') option is the temp file containing the list  *
//*         of top-level files and directory names to be archived.             *
//*    '--index-file=FILE' write verbose output to specified file rather than  *
//*         to standard output.                                                *
//*    '--totals' option reports the number of bytes transtered from the       *
//*         source data. Note that this is written to stderr, NOT stdout.      *
//* -- Other info:                                                             *
//*    -- The default archive format is used. The default format is determined *
//*       at the time the 'tar' utility is compiled. This is usually either    *
//*      'gnu' or 'posix'. This format is reported by invoking: 'tar --help'   *
//*    -- symbolic links are not dereferenced, that is, the symlink file       *
//*       itself is saved, NOT the file it points to.                          *
//*    -- '-W' (--verify) option NOT CERTAIN                                   *
//*    -- '--ignore-failed-read' NOT CERTAIN                                   *
//*    -- '--one-filesystem' option IS NOT used:  NOT CERTAIN                  *
//*       It is logical that 'tar' be allowed to access all data referenced on *
//*       the clipboard, and the data on the clipboard may cross filesystems.  *
//*                                                                            *
//******************************************************************************

short FileDlg::acCreateTarArchive ( BSet& bkSet, bool update )
{
   //* Template for creating the tar command *
   const char* tarCreateTemplate = 
      "tar -cavv --numeric-owner --file=\"%S\" --files-from=\"%S\" "
         "--index-file=\"%S\" --totals 2>\"%S\"" ;
   const char* tarUpdateTemplate = 
      "tar -rvv --numeric-owner --file=\"%S\" --files-from=\"%S\" "
          "--index-file=\"%S\" --totals 2>\"%S\"" ;

   arcType arctype = this->ArchiveTarget ( bkSet.barc ) ;
   short status = OK ;                 // return value

   if ( ((arctype != atNONE) && (arctype != atZIP) 
         && (arctype != atODOC) && (arctype != atOXML)) && 
        ((! update) || (update && (arctype == atTAR))) )
   {
      char cmdBuff[gsDFLTBYTES * 3] ;     // buffer for creating the command string
      char inBuff[gsDFLTBYTES] ;          // input buffer for reading temp files
      bool done = false ;                 // loop control

      //* Get filespecs for the temporary files *
      gString gsOut, fileList, stdOut, stdErr ;
      this->fmPtr->CreateTempname ( fileList ) ;
      this->fmPtr->CreateTempname ( stdOut ) ;
      this->fmPtr->CreateTempname ( stdErr ) ;

      //* Create the list of items to be archived *
      this->Archive_FileList ( fileList ) ;

      //* Escape any "special characters" which *
      //* might be misinterpreted by the shell. *
      gString escPath = bkSet.barc ;
      this->EscapeSpecialChars ( escPath, escMIN ) ;

      //* Construct the command string *
      if ( update )
      {
         snprintf ( cmdBuff, (gsDFLTBYTES * 3), tarUpdateTemplate, 
                    escPath.gstr(), fileList.gstr(), stdOut.gstr(), stdErr.gstr() ) ;
      }
      else
      {
         snprintf ( cmdBuff, (gsDFLTBYTES * 3), tarCreateTemplate, 
                    escPath.gstr(), fileList.gstr(), stdOut.gstr(), stdErr.gstr() ) ;
      }

      //* Copy the command to the log file.*
      gsOut = cmdBuff ;
      this->bkWriteLogfile ( "TAR command:", blVERBOSE, lrecHDR, false ) ;
      gsOut.insert( L"\n             ", (gsOut.find( L"--files-from" )) ) ;
      gsOut.append( L'\n' ) ;
      this->bkWriteLogfile ( gsOut ) ;

      //* Execute the external command *
      this->fmPtr->Systemcall ( cmdBuff ) ;

      //* Copy the redirected log data from the temp files to the log file *
      //* Data lines in stdOut temp file == 'fupdated'.                    *
      //* 'ferrors' == ('ftotal' - 'fupdated').                            *
      //* 'fskipped' will always be ZERO (unless operation aborted)        *
      ifstream ifs( stdErr.ustr(), ifstream::in ) ;
      if ( ifs.is_open() )
      {
         while ( ! done )
         {
            ifs.getline( inBuff, gsDFLTBYTES, NEWLINE ) ;
            if ( ifs.good() || ifs.gcount() > ZERO )
               this->bkWriteLogfile ( inBuff ) ;
            else
               done = true ;
         }
         ifs.close() ;
         this->bkWriteLogfile ( "" ) ;    // separate error messages from verbose data
      }
      ifs.open( stdOut.ustr(), ifstream::in ) ;
      if ( ifs.is_open() )
      {
         done = false ;
         while ( ! done )
         {
            ifs.getline( inBuff, gsDFLTBYTES, NEWLINE ) ;
            if ( ifs.good() || ifs.gcount() > ZERO )
            {
               this->bkWriteLogfile ( inBuff ) ;
               ++bkSet.fupdated ;      // count files added to archive
            }
            else
               done = true ;
         }
         ifs.close() ;
      }

      //* Verification *
      this->acVerifyArchive ( bkSet, update ) ;

      //* Delete the temporary files *
      this->DeleteFile ( fileList ) ;
      this->DeleteFile ( stdOut ) ;
      this->DeleteFile ( stdErr ) ;
   }
   else     // (this should never happen)
      status = ERR ;

   return status ;

}  //* End acCreateTarArchive() *

//*************************
//*  acCreateZipArchive   *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* Create a 'zip' archive using the data on the clipboard.                    *
//*                                                                            *
//* Input  : bkSet   : (by reference)                                          *
//*                    'barc' contains filespec of target archive              *
//*                    Operational log file ('blog') has been initialized for  *
//*                    appending log data.                                     *
//*          update  : (optional, 'false' by default)                          *
//*                    if 'false', create a new archive                        *
//*                    if 'true',  update an existing archive                  *
//*                                                                            *
//* Returns: OK if successful, or ERR if one or more errors detected           *
//*          bkSet members receive accumulated data:                           *
//*           'ftotal'   receives number of source files (clipboard contents)  *
//*           'fupdated' receives number of files added to archive             *
//*           'ferrors'  receives number of errors during processing           *
//*           'fskipped' always ZERO for archive operations (unless operation  *
//*                      aborted, in which case this method won't be called)   *
//*           'blog'     formatted log data are inserted into this file        *
//******************************************************************************
//* Notes:                                                                     *
//*                                                                            *
//* ZIP Archives:                                                              *
//* -- Archive is created using the 'deflate' level specified by 'zipLevel'.   *
//* -- ZIP archives may be "updated" by adding new files and replacing items   *
//*    with newer (or different size) source items. The invocation options     *
//*    are the same for archive creation and update.                           *
//*    -- Note: For all existing items with a matching source filename, Zip    *
//*       records it as "updating" whether or not it is actually updated.      *
//*       If the source item is identical to the stored item, it should be     *
//*       recorded as "skipped" or not recorded at all. However, we don't have *
//*       the time to actually compare source to target to get an accurate     *
//*       update count, so we accept the reported count.                       *
//* -- Invocation options:                                                     *
//*    '.'   scan current directory and lower subdirectories                   *
//*    -i@   "include" -  file containing list of files to archive             *
//*          Example: zip -r foo . -i@include.lst                              *
//*    -r    recurse directories below CWD                                     *
//*    -lf   log-filespec (warnings and errors, overwrite existing)            *
//           When a filename is passed with this option, ZIP automagically     *
//*          adds a ".log" to the specified filename.                          *
//*          This orphans the specified temp file. If, however, we rename the  *
//*          temp file (/tmp/FMG_xxx/FMG_xxx.log), then ZIP will not modify    *
//*          the filename. This outrageous arrogance on the part of the ZIP    *
//*          developers is unacceptable. They assume that we don't know what   *
//*          we're doing (which is sometimes true, but none of their business).*
//*    -la   log-filespec (warnings and errors, append to existing) [not used] *
//*    -li   include additional info in log incl. filenames                    *
//*    -MM   "must-match" If pattern in source list is not matched, generate   *
//*          a warning message to the log, then immediately exit without       *
//*          creating an archive. If this option is not specified, then a      *
//*          non-matched pattern will generate a warning message, but process  *
//*          will continue.                                                    *
//*    -v    verbose output (unless it's the only argument)                    *
//*    -q    quiet, no informational messages to stdout                        *
//*          Note: In addition to any errors sent to the log, errors may be    *
//*          reported either to stdout or stderr, but in either case they are  *
//*          sent to the bit bucket.                                           *
//*    -u    Update Mode: Update existing items if source is newer and adding  *
//*          new files to the archive.                                         *
//*    -f    Freshen Mode: Update existing items if source is newer. Does not  *
//*          add new files to the archive.                                     *
//*    -d    Delete Mode: Delete specified items in an existing archive.       *
//*    -U    Copy Mode: Copy specified items in an existing archive to a new   *
//*          archive.                                                          *
//*    -Zd   compression option (deflate)                                      *
//*    -0 (store), -1, -2, -3, -4, -5, -6, -7, -8, -9 (deflate level)          *
//*          -- Note that deflation level reported by 'unzip -lv arc.zip'      *
//*             does not report the actual deflation option used.              *
//*             Rather, deflation is reported in ranges: 0 == "Stored",        *
//*             1 thru 2 "Defl:F", 3 thru 7 "Defl:N", and 8 thru 9 "Defl:X".   *
//*             What's up with that?                                           *
//*          -- Also, Zip allows for files of different deflation levels in a  *
//*             single archive. An update could write files with a different   *
//*             user-selected deflation than the existing files. HOWEVER,      *
//*             the user can select the deflation level only for 'create'      *
//*             operations, and the user interface forces the default          *
//*             deflation for all zip-archive updates.                         *
//*             (The interface would be too complex otherwise.)                *
//*          -- Zip allows BZIP2 and possibly other compressions in addition   *
//*             to 'Store' and 'Deflate'.                                      *
//*                                                                            *
//* -- Sample Log:                                                             *
//*    ---------
//*    Zip log opened Sat Oct 20 18:52:28 2018
//*    command line arguments:
//*     -r -i@foo.lst -lf foo.log -li -6 -q foo.zip . 
//*    
//*      adding: Somefile-01.a (deflated 68%)
//*      adding: Somefile-02.a (deflated 61%)
//*      adding: Somefile-03.a (deflated 65%)
//*    total bytes=51636, compressed=18209 -> 65% savings
//*    
//*    Total 3 entries (50K bytes)
//*    Done Sat Oct 20 18:52:28 2018
//******************************************************************************

short FileDlg::acCreateZipArchive ( BSet& bkSet, bool update )
{
   //* Template for creating the zip command *
   const char* zipCreateTemplate = 
      "zip -r -li -%hd -q \"%S\" . -i@\"%S\" -lf \"%S\" 1>/dev/null 2>/dev/null" ;
   const char* zipUpdateTemplate = 
      "zip -u -r -li -%hd -q \"%S\" . -i@\"%S\" -lf \"%S\" 1>/dev/null 2>/dev/null" ;

   short status = OK ;                 // return value

   if ( (this->ArchiveTarget ( bkSet.barc )) == atZIP )
   {
      char cmdBuff[gsDFLTBYTES * 3] ;     // buffer for creating the command string
      char inBuff[gsDFLTBYTES] ;          // input buffer for reading temp files
      bool done = false ;                 // loop control

      //* Get filespecs for the temporary files *
      gString gsOut, fileList, stdErr ;
      this->fmPtr->CreateTempname ( fileList ) ;
      this->fmPtr->CreateTempname ( stdErr ) ;

      //* Proactively rename the log file as: *
      //*    "/tmp/FMG_xxx/FMG_yyy.log"       *
      //* (See note in method header.)        *
      gsOut.compose( "%S.log", stdErr.gstr() ) ;
      this->fmPtr->RenameFile ( stdErr.ustr(), gsOut.ustr() ) ;
      stdErr = gsOut ;

      //* Create the list of items to be archived. *
      //*  (special processing for dir names)      *
      this->Archive_FileList ( fileList, true ) ;

      //* Escape any "special characters" which *
      //* might be misinterpreted by the shell. *
      gString escPath = bkSet.barc ;
      this->EscapeSpecialChars ( escPath, escMIN ) ;

      //* Construct the command string *
      if ( update != false )
      {
         snprintf ( cmdBuff, (gsDFLTBYTES * 3), zipUpdateTemplate,
                    bkSet.zipLevel, escPath.gstr(), fileList.gstr(), stdErr.gstr() ) ;
      }
      else
      {
         snprintf ( cmdBuff, (gsDFLTBYTES * 3), zipCreateTemplate,
                    bkSet.zipLevel, escPath.gstr(), fileList.gstr(), stdErr.gstr() ) ;
      }

      //* Copy the command to the log file.*
      gsOut = cmdBuff ;
      this->bkWriteLogfile ( "ZIP command:", blVERBOSE, lrecHDR, false ) ;
      gsOut.insert( L"\n             ", (gsOut.find( L" . " )) ) ;
      gsOut.append( L'\n' ) ;
      this->bkWriteLogfile ( gsOut ) ;

      //* Execute the external command *
      this->fmPtr->Systemcall ( cmdBuff ) ;

      //* Copy the redirected log data from the temp file to the log file.  *
      //* Data lines beginning with "added: " || "updating: " are counted   *
      //* as updated ('fupdated' member). See acVerifyArchive() for details.*
      ifstream ifs( stdErr.ustr(), ifstream::in ) ;
      if ( ifs.is_open() )
      {
         this->bkWriteLogfile ( "Archived Files\n==============" ) ;
         short indx ;
         bool  fileUpdate = false ;

         while ( ! done )
         {
            ifs.getline( inBuff, gsDFLTBYTES, NEWLINE ) ;
            if ( ifs.good() || ifs.gcount() > ZERO )
            {
               gsOut = inBuff ;
               gsOut.strip() ;
               if ( (fileUpdate = (gsOut.find( "updating: " )) == ZERO) || 
                    ((gsOut.find( "adding: " )) == ZERO) )
               {
                  ++bkSet.fupdated ;      // count files added to archive
                  if ( (indx = gsOut.find( " (def" )) > ZERO )
                     gsOut.limitChars( indx ) ;
                  else if ( (indx = gsOut.find( " (sto" )) > ZERO )
                     gsOut.limitChars( indx ) ;
                  if ( fileUpdate )
                  {
                     gsOut.erase( "updating:" ) ;
                     gsOut.append( " (updated)" ) ;
                  }
                  else
                  {
                     gsOut.erase( "adding:" ) ;
                  }
                  this->bkWriteLogfile ( gsOut ) ;
               }
               else if ( (gsOut.find( "total", ZERO, true )) == ZERO )
               {
                  gsOut.insert( L'\n' ) ;
                  this->bkWriteLogfile ( gsOut ) ;
               }
               else
                  ;  // discard the input line
            }
            else
               done = true ;
         }
         ifs.close() ;
      }

      //* Verification *
      this->acVerifyArchive ( bkSet, update ) ;

      //* Delete the temporary files *
      this->DeleteFile ( fileList ) ;
      this->DeleteFile ( stdErr ) ;
   }
   else     // (this should never happen)
      status = ERR ;

   return status ;

}  //* End acCreateZipArchive() *

//*************************
//*    acVerifyArchive    *
//*************************
//******************************************************************************
//* Private method                                                             *
//* --------------                                                             *
//* Validate the results of archive creation and append the validation table   *
//* to the log file.                                                           *
//*                                                                            *
//* Input  : bkSet   : (by reference) statistics on archive creation           *
//*                     'fupdated' == number of archive files added(or updated)*
//*                        - For create, this is the total number of files     *
//*                          in the archive.                                   *
//*                        - For update, this is the number of files added     *
//*                          PLUS number of files updated.                     *
//*                          (Note that 'tar' does not update existing files.) *
//*                    If update != false:                                     *
//*                     'uaTarget' previous size of archive (in bytes)         *
//*                     'uaFiles'  previous archive file count                 *
//*                     'uaBytes'  previous archive (approx) expanded bytes    *
//*                                                                            *
//*          update  : (optional, 'false' by default)                          *
//*                    if 'false', report totals for created archive           *
//*                    if 'true',  report both original AND updated totals     *
//*                                for the archive                             *
//*                                                                            *
//* Returns: OK if successful, or ERR if one or more errors detected           *
//*          bkSet members receive accumulated data:                           *
//*           'ftotal'   receives number of source files (clipboard contents)  *
//*           'ferrors'  receives number of errors during processing           *
//*           'fskipped' always ZERO for archive operations (unless operation  *
//*                      aborted, in which case this method won't be called)   *
//*           'blog'     formatted log data are inserted into this file        *
//******************************************************************************

short FileDlg::acVerifyArchive ( BSet& bkSet, bool update )
{
   UINT   cbFiles = this->ClipboardFiles () ; // source files (items attempted)
   UINT64 cbBytes = this->ClipboardSize () ;  // source files total size
   tnFName tnf ;                              // target archive stats
   UINT   arcFiles ;                          // files in archive
   UINT64 arcBytes ;                          // approx. size of expanded archive data
   short  status = OK ;                       // return value

   this->ArchiveSummary ( bkSet.barc, tnf, arcFiles, arcBytes ) ;

   bkSet.ftotal   = cbFiles ;       // source files on clipboard
   bkSet.ferrors  = bkSet.ftotal - bkSet.fupdated ; // attempted minus updated
   if ( bkSet.ferrors > ZERO )      // if errors, return the bad news
      status = ERR ;

   //* Format the data *
   gString af( arcFiles, 7 ),
           ab( arcBytes, 6, false, false, false, fiKb ),
           cf( cbFiles,  7 ),
           cb( cbBytes,  6, false, false, false, fiKb ),
           as( tnf.fBytes, 6, false, false, false, fiKb ),
           ae( bkSet.ferrors, 7 ),
           of( bkSet.uaFiles, 7, true ),
           ob( bkSet.uaBytes, 6, true, false, false, fiKb ),
           os( bkSet.uaTarget, 6, true, false, false, fiKb ),
           gsOut ;

   if ( update )     // update archive
   {
      gsOut.compose( 
         "\n      Verification"
         "\n----------------------------------"
         "\n         Archive  Source  Previous"
         "\nFiles :  %S %S  (%S)"
         "\nBytes :   %S  %S  (%S)"
         "\nTarget:   %S          (%S)"
         "\nErrors:  %S",
         af.gstr(), cf.gstr(), of.gstr(), 
         ab.gstr(), cb.gstr(), ob.gstr(), 
         as.gstr(), os.gstr(), ae.gstr() ) ;
   }
   else              // create archive
   {
      gsOut.compose( 
         "\n      Verification"
         "\n------------------------"
         "\n         Archive  Source"
         "\nFiles :  %S %S"
         "\nBytes :   %S  %S"
         "\nTarget:   %S"
         "\nErrors:  %S",
         af.gstr(), cf.gstr(), ab.gstr(), cb.gstr(), as.gstr(), ae.gstr() ) ;
   }
   this->bkWriteLogfile ( gsOut ) ;

   #if ENABLE_DEBUGGING_CODE != 0 && enableDBBAK != 0
   //* Details not reported for archive update *
   if ( update != false )
   {
      UINT  itemsAdded   = arcFiles - bkSet.uaFiles,
            itemsUpdated = bkSet.fupdated - itemsAdded ;
      cf.formatInt( itemsAdded, 4 ) ;
      cb.formatInt( itemsUpdated, 4 ) ;
      gsOut.compose( "** Added  : %S\n"
                     "** Updated: %S",
                     cf.gstr(), cb.gstr() ) ;
      this->bkWriteLogfile ( gsOut ) ;
   }
   #endif   // ENABLE_DEBUGGING_CODE && enableDBBAK

   return status ;

}  //* End acVerifyArchive() *

//*************************
//*    Archive_Append     *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* Update an existing archive by append the data on the clipboard.            *
//* Existing items will not be modified.                                       *
//*                                                                            *
//* Input  : bkSet   : (by reference)                                          *
//*                    'barc' contains filespec of target archive              *
//*                                                                            *
//* Returns: OK if successful, or ERR if one or more errors detected           *
//*          bkSet members receive accumulated data:                           *
//*           'ftotal'   receives number of source files on clipboard          *
//*           'fupdated' receives number of files added to archive             *
//*           'ferrors'  receives number of errors during processing           *
//*           'blog'     formatted log data are inserted into this file        *
//******************************************************************************
//* NOTES:                                                                     *
//* ==============                                                             *
//* Appending data to an existing archive:                                     *
//*                                                                            *
//* Tar Archives:                                                              *
//* a) The archive may be updated ONLY if the target archive is not compressed.*
//* b) We record the operation to the log file using the double-v ('-vv')      *
//*    verbose option.                                                         *
//* c) Verification is done by comparing the original archive, the updated     *
//*    archive and the contents of the clipboard.                              *
//*    -- The total files in the updated archive should increase by the number *
//*       of files on the clipboard. Otherwise the difference is the number    *
//*       of errors reported.                                                  *
//* d) The '--verify' ('-W') option is incompatible with append/update         *
//*    operations; therefore, the '--compare' ('-d') option must be used       *
//*    instead as a separate operation. This is slower and more tedious,       *
//*    AND cannot verify the pre-update contents of the archive, and is        *
//*    therefore not an attractive option.                                     *
//*    Verification is done by comparing the original archive, the updated     *
//*    archive and the contents of the clipboard. The total files in the       *
//*    updated archive should increase by the number of files on the clipboard.*
//*    Otherwise the difference is the number of errors reported.              *
//*    (See acVerifyArchive() for details.)                                    *
//*                                                                            *
//* Zip Archives:                                                              *
//* a) Zip allows updates to existing archives with few restrictions.          *
//* b) The compression level specified for the update must match the           *
//*    compression level used in the existing archive. If not, the update      *
//*    will fail.                                                              *
//* c) No special invocation option is needed to distinguish between 'create'  *
//*    and 'update'. Therefore, except for verification, the same invocation   *
//*    may be used for both (see next item), the differences lie in whether    *
//*    the action to be taken is:                                              *
//*      add (default) : Update existing entries and add new files. No         *
//*                      difference between 'create' and 'update' of archive.  *
//*      '-u' (update) : Same as 'add' except that if the archive does not     *
//*                      exist, a warning will be generated.                   *
//*      '-f' (freshen): Update existing entries only. Do not add new files.   *
//*    Unfortunately, there is no '-r' (append new files) mode comparable to   *
//*    the mode used with Tar. We see no advantage in specifying '-u' over the *
//*    default (add) since caller has verified that the target archive exists. *
//*    The '-f' (freshen), '-d' (delete), '-U' (copy) and '-FS' modes are      *
//*    beyond the scope of this iteration of our project.                      *
//* d) Verification is done by comparing the original archive, the updated     *
//*    archive and the contents of the clipboard. The total files in the       *
//*    updated archive should increase by the number of files on the clipboard *
//*    MINUS the number of updated files. Otherwise the difference is the      *
//*    number of errors reported. (See acVerifyArchive() for details.)         *
//*                                                                            *
//******************************************************************************

short FileDlg::Archive_Append ( BSet& bkSet )
{
   arcType aType = this->ArchiveTarget ( bkSet.barc ) ;  // archive type
   short status = OK ;           // return value

   //* Verify that the target archive type is supported.*
   if ( (aType == atTAR) || (aType == atZIP) )
   {
      this->ProcessCWD () ;         // display the default processing message

      //* Capture the current stats for the target archive *
      tnFName origTnf ;             // stats of existing archive
      this->ArchiveSummary ( bkSet.barc, origTnf, bkSet.uaFiles, bkSet.uaBytes ) ;
      bkSet.uaTarget = origTnf.fBytes ;

      //********************************
      //* If target is a 'zip' archive.*
      //********************************
      if ( aType == atZIP )
      {
         this->acCreateZipArchive ( bkSet, true ) ;
      }

      //**********************************************
      //* If target is an uncompressed 'tar' archive.*
      //**********************************************
      else if ( aType == atTAR )
      {
         this->acCreateTarArchive ( bkSet, true ) ;
      }

      this->ProcessCWD ( true ) ;   // clear the processing message
   }
   else // (this should never happen because caller has control of user input)
      status = ERR ;

   return status ;

}  //* End Archive_Append() *

//*************************
//*    Archive_Expand     *
//*************************
//********************************************************************************
//* Private Method:                                                              *
//* ---------------                                                              *
//* Expand the specified archive to the current directory.                       *
//* Any existing targets will be overwritten.                                    *
//*                                                                              *
//* Input  : bkSet   : (by reference)                                            *
//*                    'barc' contains filespec of source archive                *
//*                                                                              *
//* Returns: OK if successful (or ERR if unsupported archive type (unlikely))    *
//*          remaining 'bkSet' members are initialized                           *
//*          'fupdated' receives number of files successfully written to target  *
//*          'uaTarget' receives approximate size of all data written to target  *
//*          'ferrors'  receives number of errors during processing              *
//*                     i.e. number of archive items not successfully extracted  *
//*          'ftotal'   receives number of file in source archive                *
//*          'uaFiles'  receives number of file in source archive                *
//*          'uaBytes'  receives approximate expanded size of archive data       *
//*           'blog'     formatted log data are inserted into this file          *
//********************************************************************************
//* Expand an archive (extract data from the archive):                           *
//* a) All files are extracted from the archive and are written to the CWD.      *
//*    -- In a future release, we may implement an option to extract only        *
//*       the user-specified items.                                              *
//* b) For 'tar' archives, we record the operation to the log file using the     *
//*    double-v ('-vv') verbose option. For 'zip' archives ('-v') is used.       *
//* c) Owner and Group will be those of the user extracting the data,            *
//*    UNLESS the user is the Superuser, in which case ownership will be         *
//*    preserved if possible.                                                    *
//* d) Modification timestamp will be preserved during extraction.               *
//* e) Access timestamp will be updated during extraction.                       *
//* f) Permissions will be set according to user's 'umask'.                      *
//* g) Order of extraction will be the order of data in the archive.             *
//*    This shouldn't matter when extracting all data, but it may enhance        *
//*    user-friendliness when extracting only specified items to report in the   *
//*    order items were specified.                                               *
//* h) Note that user may request expansion of an OpenDocument file.             *
//*    This is fully supported.                                                  *
//*   ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----     *
//* Programmer's Note:                                                           *
//* The file count returned in bkSet.uaFiles and/or bkSet.fupdated is sometimes  *
//* incorrect, causing 'bkSet.ferrors' to be incorrect (i.e. -1 or similar)      *
//* This is most likely caused by (ferrors = uaFiles - fupdated) < ZERO)         *
//*  - 'uaFiles' is initialized by ArchiveSummary()                              *
//*  - Under what circumstances would 'fupdated' be larger?                      *
//*    a) 'uaFiles' calculation too small                                        *
//*       - sometimes returned as ZERO                                           *
//*       - sometimes returns undercount (base directory not counted)            *
//*    b) 'fupdated' calculation too large                                       *
//*       - Created-directories count may be the issue.                          *
//* This method compensates for the potential file undercount/overcount during   *
//* archive expansion.                                                           *
//*                                                                              *
//********************************************************************************

short FileDlg::Archive_Expand ( BSet& bkSet )
{
   arcType arctype = this->ArchiveTarget ( bkSet.barc ) ; // target-archive type
   short status = OK ;                 // return value

   #if ENABLE_DEBUGGING_CODE != 0 && enableDBBAK != 0
   this->bkWriteLogfile ( "Archive_Expand:" ) ;
   #endif   // ENABLE_DEBUGGING_CODE && enableDBBAK

   //* Retrieve archive stats.*
   tnFName tnf ;
   this->ArchiveSummary ( bkSet.barc, tnf, bkSet.uaFiles, bkSet.uaBytes ) ;
   bkSet.ftotal = bkSet.uaFiles ;
   bkSet.fupdated = ZERO ;    // initialize updated-files counter
   bkSet.uaTarget = ZERO ;    // initialize updated-files byte count

   //***************************************************************************
   //* If source is a ZIP archive (including OpenDocuments and ePub documents).*
   //***************************************************************************
   if ( (arctype == atZIP)  || (arctype == atODOC) || 
        (arctype == atOXML) || (arctype == atEPUB) )
   {
      this->aeExpandZipArchive ( bkSet ) ;
   }

   //******************************
   //* If source is a TAR archive.*
   //******************************
   else if ( arctype != atNONE )
   {
      this->aeExpandTarArchive ( bkSet ) ;
   }
   else     // (this should never happen)
      status = ERR ;

   //* Potential file undercount/overcount (see note above). *
   if ( bkSet.uaFiles < bkSet.fupdated )
   {
      // Programmer's Note: If all items in the tarball are contained within 
      // a base directory (as most distribution archives are), then that base 
      // directory will not be reported by ArchiveSummary(). If that base 
      // directory is created during archive expansion, 'fupdated' will be one 
      // item greater than 'uaFiles'.
      // This is not really an error and should not be reported as such.
      if ( bkSet.uaFiles != (bkSet.fupdated - 1) )
      {
         gString gstmp( "\n*** Error! - uaFiles < fupdated (%u vs %u) ***\n", 
                        &bkSet.uaFiles, &bkSet.fupdated ) ;
         this->bkWriteLogfile ( gstmp ) ;
      }
      bkSet.uaFiles = bkSet.fupdated ;
   }

   //* Verification *
   bkSet.ferrors = bkSet.uaFiles - bkSet.fupdated ;
   if ( bkSet.ferrors > ZERO )      // if errors, return the bad news
      status = ERR ;
   gString af( bkSet.uaFiles, 7 ),
           ab( bkSet.uaBytes, 6, false, false, false, fiKb ),
           ef( bkSet.fupdated, 7 ),
           eb( bkSet.uaTarget, 6, false, false, false, fiKb ),
           ae( bkSet.ferrors, 7 ) ;
   gString gsOut( "\n      Verification"
                  "\n----------------------------------"
                  "\n         Archive  Extracted"
                  "\nFiles :  %S    %S"
                  "\nBytes :   %S     %S"
                  "\nErrors:  %S",
                  af.gstr(), ef.gstr(), ab.gstr(), eb.gstr(), ae.gstr() ) ;
   this->bkWriteLogfile ( gsOut ) ;

   return status ;

}  //* End Archive_Expand() *

//*************************
//*  aeExpandTarArchive   *
//*************************
//******************************************************************************
//* Private Method                                                             *
//* --------------                                                             *
//* Expand a Zip archive, writing its contents to target directory (CWD).      *
//* (See notes below for details.)                                             *
//*                                                                            *
//* Input  : bkSet   : (by reference)                                          *
//*                    'barc' contains filespec of source archive              *
//*                    'ftotal'   contains total archive item to be expanded   *
//*                    'uaFiles'  contains number of file in source archive    *
//*                    'uaBytes'  contains approximate expanded size of archive*
//*                                                                            *
//* Returns: OK (currently, no major error conditions to report)               *
//*          'fupdated' receives number of files successfully written to target*
//*          'uaTarget' receives approximate size of all data written to target*
//*          'ferrors'  receives number of errors during processing            *
//*                     i.e. number of archive items not successfully extracted*
//*           'blog'     formatted log data are inserted into this file        *
//******************************************************************************
//* Tar Notes:                                                                 *
//* ----------                                                                 *
//* 1) Extracting all data from the archive is straightforward except for      *
//*    dealing with existing targets (see below).                              *
//* 2) Extracting individual items/files from the archive:                     *
//*    a) Extracting a single item/file would be possible with only a ScrollExt*
//*       control containing a list of the archive's contents.                 *
//*       - Embedded paths may be wider than the display window, so we would   *
//*         need a way to show the full path (horizontal scrolling?).          *
//*    b) Extracting multiple items in a single operation is a significant     *
//*       user-interface challenge, which we would like to avoid at this time. *
//*       It would require that we present the user with a list of the archive *
//*       contents, and provide a method for the user to select multiple items.*
//* 3) Ownership of an extracted item is given to the user by default. This    *
//*    can be overridden, but that would only be useful if the user were the   *
//*    superuser, extracting data into a regular user's data space.            *
//*    See '--same-owner' option for details.                                  *
//* 4) Permissions are controlled by the user's 'umask' by default.            *
//*    To preserve the archive's permissions, use '--preserve-permissions'.    *
//* 5) By default, the mod timestamp of the archive item is the mod timestamp  *
//*    of the target. It is possible to override this ('--touch') but that     *
//*    seems a bit ridiculous.                                                 *
//* 6) By default, symbolic links are not followed when removing existing      *
//*    targets or writing items from the archive to the filesystem. This is    *
//*    good since extracting data in one directory should not affect data in   *
//*    other directories. Only a moron would try to override this behavior.    *
//* 7) If the archive contains multiple copies of the same file, only the      *
//*    most-recently-archived copy will survive the extraction unless the      *
//*    '--occurrence' option is used.                                          *
//* 8) Verbose output may go to stdout and/or stderr, so we must capture both  *
//*    to our log file.                                                        *
//* 9) Existing targets:                                                       *
//*    a) When expanding an archive, there may be existing target items/files. *
//*       To deal with these existing targets, we have options.                *
//*       '--unlink-first' will unconditionally remove the existing target     *
//*         before extracting the file. This is the most efficient, but        *
//*         potentially destroys data the user may want to keep.               *
//*       '--keep-old-files' causes detection of an existing target to be      *
//*         reported as a error. This is the safest because it will not destroy*
//*         any data. On the other hand, it may be difficult for the user to   *
//*         determine if the resulting target was extracted from the archive,  *
//*         or is stale data. This determination would require the user to     *
//*         study the log file, which is not very user friendly.               *
//*       '--keep-newer-files' compares the existing target with the           *
//*         corresponding archive file and refuses to overwrite a target that  *
//*         is the same date or newer.                                         *
//*         This is the logical choice, but again, the user would have to read *
//*         the log file to know what happened. It is possible to scan the     *
//*         log for existing-file warnings and alert the user to the number of *
//*         such warnings.                                                     *
//*       '--warning=existing-file' will report an existing target before      *
//*         other options cause retention or overwrite of the target.          *
//*         Note that this option works only if the '--skip-old-files' option  *
//*         or '--keep-newer-files' option is also used.                       *
//*         This means that the log will contain no messages about overwritten *
//*         files UNLESS we specify that we want to keep them. This is a       *
//*         logical error in tar itself.                                       *
//*    b) Existing target directories are even more difficult to deal with,    *
//*       due to potential overwrite of some or all of the directory's         *
//*       contents, as well as how to handle the directory's metadata.         *
//*    c) It is possible, though non-trivial that we could move all existing   *
//*       targets to the trashcan before performing the extraction. This would *
//*       prevent data loss, but is rather non-intuitive for the user.         *
//*    d) Our solution to existing files is to use the default behavior of     *
//*       'tar' which silently overrides write protection on existing targets, *
//*       and replaces them with the archive version of the file. This ensures *
//*       that what is in the target directory matches the contents of the     *
//*       archive. This approach is the logical one for all but a few unlikely *
//*       cases.                                                               *
//*       Handling existing targets and reporting on exceptions is tedious and *
//*       error prone. For this reason, we caution the user that archives      *
//*       should always be expanded into a clean subdirectory.                 *
//*                                                                            *
//******************************************************************************

short FileDlg::aeExpandTarArchive ( BSet& bkSet )
{
   const char* tarTemplate = 
      "tar -xvv --file=\"%S\" --index-file=\"%S\" --totals 2>\"%S\"" ;

   char cmdBuff[gsDFLTBYTES * 3] ;     // buffer for creating the command string
   char inBuff[gsDFLTBYTES] ;          // input buffer for reading temp files
   short status = OK ;                 // return value
   bool done = false ;                 // loop control

   //* Get filespecs for the temporary files *
   gString gsOut, stdOut, stdErr ;
   this->fmPtr->CreateTempname ( stdOut ) ;
   this->fmPtr->CreateTempname ( stdErr ) ;

   //* Escape any "special characters" which *
   //* might be misinterpreted by the shell. *
   gString escPath = bkSet.barc ;
   this->EscapeSpecialChars ( escPath, escMIN ) ;

   //* Construct the command *
   snprintf ( cmdBuff, (gsDFLTBYTES * 3), tarTemplate, 
              escPath.gstr(), stdOut.gstr(), stdErr.gstr() ) ;

   //* Copy the command to the log file.*
   gsOut = cmdBuff ;
   this->bkWriteLogfile ( "TAR command:", blVERBOSE, lrecHDR, false ) ;
   gsOut.insert( L"\n             ", (gsOut.find( L"--index-file" )) ) ;
   gsOut.append( L'\n' ) ;
   this->bkWriteLogfile ( gsOut ) ;

   //* Execute the external command *
   this->fmPtr->Systemcall ( cmdBuff ) ;

   //* If errors written to stderr, copy them to the log file.*
   ifstream ifs( stdErr.ustr(), ifstream::in ) ;
   if ( ifs.is_open() )
   {
      UINT stderrCount = ZERO ;     // lines captured from stderr
      done = false ;
      while ( ! done )
      {
         ifs.getline( inBuff, gsDFLTBYTES, NEWLINE ) ;
         if ( ifs.good() || ifs.gcount() > ZERO )
         {
            this->bkWriteLogfile ( inBuff ) ;
            ++stderrCount ;
         }
         else
            done = true ;
      }
      if ( stderrCount > ZERO )     // separate errors from item list
         this->bkWriteLogfile ( "" ) ;

      ifs.close() ;
   }

   this->bkWriteLogfile ( "Extracted Files\n===============" ) ;

   //* Report each item written to target directory.*
   ifs.open( stdOut.ustr(), ifstream::in ) ;
   if ( ifs.is_open() )
   {
      UINT64 tmpSize ;              // for capture of file size
      short fsOffset ;              // index of file-size data to be captured
      done = false ;
      while ( ! done )
      {
         ifs.getline( inBuff, gsDFLTBYTES, NEWLINE ) ;
         if ( ifs.good() || ifs.gcount() > ZERO )
         {
            this->bkWriteLogfile ( inBuff ) ;

            //* Determine whether this is a file record *
            for ( short i = ZERO ; i < ftypeCOUNT ; ++i )
            {
               if ( *inBuff == fileTypes[i] )
               {
                  if ( (inBuff[1] == 'r') || (inBuff[1] == '-') )
                  {
                     ++bkSet.fupdated ;
                     gsOut = inBuff ;
                     if ( (fsOffset = gsOut.find( L' ' )) >= ZERO )
                     {
                        if ( (fsOffset = gsOut.find( L' ', ++fsOffset )) >= ZERO )
                        {
                           if ( (swscanf ( &gsOut.gstr()[fsOffset], L" %llu", &tmpSize )) == 1 )
                              bkSet.uaTarget += (tmpSize > ZERO) ? tmpSize : minBLOCK ;
                        }
                     }
                  }
                  break ;
               }
            }
         }
         else
            done = true ;
      }
      ifs.close() ;
   }

   //* Delete the temporary files *
   this->DeleteFile ( stdOut ) ;
   this->DeleteFile ( stdErr ) ;

   return status ;

}  //* End aeExpandTarArchive() *

//*************************
//*  aeExpandZipArchive   *
//*************************
//******************************************************************************
//* Private Method                                                             *
//* --------------                                                             *
//* Expand a Zip archive, writing its contents to target directory (CWD).      *
//* (See notes below for details.)                                             *
//*                                                                            *
//* Input  : bkSet   : (by reference)                                          *
//*                    'barc' contains filespec of source archive              *
//*                    'ftotal'   contains total archive item to be expanded   *
//*                    'uaFiles'  contains number of file in source archive    *
//*                    'uaBytes'  contains approximate expanded size of archive*
//*                                                                            *
//* Returns: OK (currently, no major error conditions to report)               *
//*          'fupdated' receives number of files successfully written to target*
//*          'uaTarget' receives approximate size of all data written to target*
//*          'ferrors'  receives number of errors during processing            *
//*                     i.e. number of archive items not successfully extracted*
//*           'blog'     formatted log data are inserted into this file        *
//******************************************************************************
//* Unzip Notes:                                                               *
//* ------------                                                               *
//* 1) Options:                                                                *
//*    -l    list archive contents (short format) - files are not extracted    *
//*    -v    list archive contents (verbose format) - files are not extracted  *
//*    -o    overwrite existing targets without prompting                      *
//*    -n    never overwrite existing targets                                  *
//*    -q    perform operations quietly (-qq suppress all messages)            *
//*    -f    freshen: If archive file is newer than corresponding target file, *
//*          extract the archive file. This option will query for overwrite    *
//*          unless '-o' is also specified.                                    *
//*    -u    update: If archive file is newer than corresponding target file,  *
//*          extract the archive file. Also, extract the archive file if there *
//*          is no corresponding target file. This option will query for       *
//*          overwrite unless '-o' is also specified.                          *
//*                                                                            *
//* 2) Unfortunately, there is no option to unconditionally extract all archive*
//*    files without regard to potential existing targets.                     *
//*    a) In addition, extraction using '-f' or '-u' relies on the timestamp   *
//*       comparisons being properly set up on the system. Unfortunately,      *
//*       systems don't always agree on timestamps. Linux/UNIX uses UTC (GMT)  *
//*       time, dynamically adjusted for timezone and daylight-savings-time.   *
//*       If the system which created the archive uses a different scheme,     *
//*       then the "TZ" environment variable must be present to fascilitate    *
//*       timestamp comparisons, but of course we can't rely on that.          *
//*    b) We will need serious manual intervention to force overwrite of newer *
//*       existing targets; however, for this iteration, we simply report the  *
//*       results.                                                             *
//*                                                                            *
//* 3) Even more unfortunately, we cannot get verbose output during archive    *
//*    expansion. If '-uv' is used, then the verbose data are reported, BUT    *
//*    the archive is not expanded. On its own, '-u' reports only:             *
//*                  "inflating: Somefile-03.xx"                               *
//*    for each expanded item, which is useless for obtaining the file size.   *
//*    And of course there will be no report at all if there is an existing    *
//*    target whose timestamp is the same or newer than the archived file.     *
//*                                                                            *
//* 4) Our Implementation:                                                     *
//*    To obtain meaningful data for each expanded item, we execute a          *
//*    verbose-data request for each individual item extracted.                *
//*    HOWEVER, if there are existing targets with same or newer timestamp,    *
//*    then the corresponding archive item will not be expanded. We count      *
//*    this as an error to alert the dumbass user that he/she/it should have   *
//*    expanded the archive into a clean directory.                            *
//*    a) Note that we treat supported OpenDocument files as ordinary zip      *
//*       archive files. By 'supported' we mean OD files constructed according *
//*       to the XML document format. See odIsOpenDocFile() and                *
//*       odIsOpenXmlFile() for details.                                       *
//*                                                                            *
//* -  -  -  -  -                                                              *
//* Programmer's Note: Once again Microsloth proves itself to be more trouble  *
//* than it's worth: Newer MS-Office (OOXML) files include a special file,     *
//*                       [Content_Types].xml                                  *
//* Unfortunately, the command-line interpreter sees the square brackets       *
//* '[' and ']' as special characters and strips them. Therefore, filenames    *
//* that contain these special characters are not found by the 'unzip -v'      *
//* command, which causes an undercount while compiling statistics on the      *
//* extracted files. To compensate, we must escape the '[' and ']' characters  *
//* in the filenames sent to the command line.  Stupid, fucking Microsloth!    *
//******************************************************************************

short FileDlg::aeExpandZipArchive ( BSet& bkSet )
{
   const char* zipTemplate =
      "unzip -u -o \"%S\" 1>\"%S\" 2>\"%S\"" ;
   const char* statTemplate = 
      "unzip -v \"%S\" \"%S\" 1>>\"%S\" 2>/dev/null" ; 

   char cmdBuff[gsDFLTBYTES * 3] ;     // buffer for creating the command string
   char inBuff[gsDFLTBYTES] ;          // input buffer for reading temp files
   short escIndex = ZERO,              // for scanning filenames
         status = OK ;                 // return value
   bool done = false ;                 // loop control

   //* Get filespecs for the temporary files *
   gString gsOut, fileList, stdOut, stdErr ;
   this->fmPtr->CreateTempname ( fileList ) ;// captures stats on each expanded item
   this->fmPtr->CreateTempname ( stdOut ) ;  // captures 'stdout'
   this->fmPtr->CreateTempname ( stdErr ) ;  // captures 'stderr'

   //* Retrieve archive stats.*
   bkSet.fupdated = ZERO ;    // initialize updated-files counter
   bkSet.uaTarget = ZERO ;    // initialize updated-files byte count


   //* Escape any "special characters" which *
   //* might be misinterpreted by the shell. *
   gString escPath = bkSet.barc ;
   this->EscapeSpecialChars ( escPath, escMIN ) ;

   //* Construct the command *
   snprintf ( cmdBuff, (gsDFLTBYTES * 3), zipTemplate, 
              escPath.gstr(), stdOut.gstr(), stdErr.gstr() ) ;

   //* Copy the command to the log file.*
   gsOut = cmdBuff ;
   this->bkWriteLogfile ( "ZIP command:", blVERBOSE, lrecHDR, false ) ;
   gsOut.insert( L"\n             ", (gsOut.find( L"1>" )) ) ;
   gsOut.append( L'\n' ) ;
   this->bkWriteLogfile ( gsOut ) ;

   //* Execute the external command *
   this->fmPtr->Systemcall ( cmdBuff ) ;

   //* If errors written to stderr, copy them to the log file.*
   ifstream ifs( stdErr.ustr(), ifstream::in ) ;
   if ( ifs.is_open() )
   {
      UINT stderrCount = ZERO ;
      done = false ;
      while ( ! done )
      {
         ifs.getline( inBuff, gsDFLTBYTES, NEWLINE ) ;
         if ( ifs.good() || ifs.gcount() > ZERO )
         {
            this->bkWriteLogfile ( inBuff ) ;
            ++stderrCount ;
         }
         else
            done = true ;
      }
      if ( stderrCount > ZERO )     // separate errors from item list
         this->bkWriteLogfile ( "" ) ;

      ifs.close() ;
   }

   //* Scan the list of extracted items and       *
   //* generate statistics on each extracted file.*
   ifs.open( stdOut.ustr(), ifstream::in ) ;
   if ( ifs.is_open() )
   {
      short fnIndex = ZERO ;  // index into record for name of expanded file
      done = false ;          // loop control
      while ( ! done )
      {
         ifs.getline( inBuff, gsDFLTBYTES, NEWLINE ) ;
         if ( ifs.good() || ifs.gcount() > ZERO )
         {
            gsOut = inBuff ;
            gsOut.strip() ;      // strip leading and trailing whitespace
            if ( ((gsOut.find( "inflating: " )) >= ZERO) ||
                 ((gsOut.find( "creating: " )) >= ZERO) ||
                 ((gsOut.find( "extracting: " )) >= ZERO) )
            {
               fnIndex = gsOut.after( L':' ) ;
               fnIndex = gsOut.scan( fnIndex ) ;

               //* Escape any square brackets in the filename. (see note above)*
               if ( (escIndex = gsOut.find( L'[', fnIndex )) >= ZERO )
               {
                  gsOut.insert( L'\\', escIndex ) ;
                  if ( (escIndex = gsOut.find( L']', escIndex )) >= ZERO )
                     gsOut.insert( L'\\', escIndex ) ;
               }

               snprintf ( cmdBuff, (gsDFLTBYTES * 3), statTemplate, 
                          escPath.gstr(), &gsOut.gstr()[fnIndex], fileList.gstr() ) ;
               this->fmPtr->Systemcall ( cmdBuff ) ;
            }
         }
         else
            done = true ;
      }
      ifs.close() ;
   }

   this->bkWriteLogfile ( "Extracted Files\n===============" ) ;

   //* Get cumulative stats on extracted files.*
   ifs.open( fileList.ustr(), ifstream::in ) ;
   if ( ifs.is_open() )
   {
      const wchar_t *Defl = L"Defl:",
                    *Stor = L"Stored",
                    *Bzip = L"BZip2",
                    *Pcnt = L"% " ;
      UINT64 tmpSize ;                 // for capture of file size
      short idxBeg, idxEnd ;           // for scanning line data
      done = false ;                   // loop control
      while ( ! done )
      {
         ifs.getline( inBuff, gsDFLTBYTES, NEWLINE ) ;
         if ( ifs.good() || ifs.gcount() > ZERO )
         {
            gsOut = inBuff ;
            if ( ((gsOut.find( Defl )) > ZERO) ||
                 ((gsOut.find( Stor )) > ZERO) ||
                 ((gsOut.find( Bzip )) > ZERO) )
            {
               ++bkSet.fupdated ;
               if ( (swscanf ( gsOut.gstr(), L" %llu", &tmpSize )) == 1 )
                  bkSet.uaTarget += (tmpSize > ZERO) ? tmpSize : minBLOCK ;

               //* Reformat the data for beauty.*
               if ( (idxBeg = gsOut.find( Defl )) < ZERO )
                  idxBeg = gsOut.find( Stor ) ;
               if (    (idxBeg > ZERO) 
                    && ((idxEnd = gsOut.after( Pcnt )) > ZERO)
                    && (idxEnd > idxBeg) )
               {
                  --idxBeg ;
                  gsOut.erase( idxBeg, (idxEnd - idxBeg + 1) ) ;
                  idxBeg = gsOut.after( L' ', idxBeg ) ;
                  idxBeg = gsOut.after( L' ', idxBeg ) ;
                  idxEnd = gsOut.after( L' ', idxBeg ) ;
                  gsOut.erase( idxBeg, (idxEnd - idxBeg + 1) ) ;
               }
               this->bkWriteLogfile ( gsOut ) ;
            }
         }
         else
            done = true ;
      }
      ifs.close() ;
   }

   //* Compare archive item count with count of files extracted *
   //* and complain if source count != expanded count.          *
   if ( bkSet.fupdated != bkSet.uaFiles )
   {
      gsOut.compose( 
         "\nWARNING! - One or more archived files not expanded to target directory!\n"
           "           %u of %u files expanded.", 
                       &bkSet.fupdated, &bkSet.uaFiles ) ;
      this->bkWriteLogfile ( gsOut ) ;
   }

   //* Delete the temporary files *
   this->DeleteFile ( fileList ) ;
   this->DeleteFile ( stdOut ) ;
   this->DeleteFile ( stdErr ) ;

   return status ;

}  //* End aeExpandZipArchive() *

//*************************
//*    Archive_Summary    *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* Display a summary of the completed Archive operation.                      *
//*                                                                            *
//* Input  : bkSet    : (by reference) contains summary data                   *
//*          opStatus : operational status: OK==success, ERR==user abort       *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Notes:                                                                     *
//*                                                                            *
//******************************************************************************

void FileDlg::Archive_Summary ( const BSet& bkSet, short opStatus )
{
   const char* aopName[] = 
   {
      "Create Archive",             // create
      "Update Archive",             // append
      "Expand Archive"              // extract
   } ;
   const char *aHdr = "Archive",
              *sHdr = "Source Data",
              *oHdr = "Previous",
              *fHdr = " files: ",
              *bHdr = " bytes: ",
              *tHdr = "Target: ",
              *eHdr = "Errors: ",
              *xsHdr = "Source",
              *xxHdr = "Extracted" ;
   const char* summaryTemplate[] =
   {
      "Summary Results: %s\n"                // create
      "-------------------------------\n"
      "       %s  %s\n"
      "       -------  -----------\n"
      "%s%S",
      "Summary Results: %s\n"                // update
      "-------------------------------\n"
      "       %s  %s  %s\n"
      "       -------  -----------  --------\n"
      "%s%S",
      "Summary Results: %s\n"                // expand
      "-------------------------------\n"
      "       %s\n"
      "       %s  %s\n"
      "       -------  ---------\n"
      "%s%S\n",
   } ;
      
   //* Define the dialog window *
   const short dlgROWS = minfitROWS - 2,
               dlgCOLS = minfitCOLS,
               dlgYPOS = this->fulY + 1,
               dlgXPOS = this->fulX + (this->fMaxX / 2) - (dlgCOLS / 2) ;
   const attr_t dColor = this->cs.sd,
                hColor = this->cs.em ;

   enum asControls : short { clPB, vlPB, slPB, asCONTROLS } ;

   //** Define the dialog controls **
   InitCtrl ic[asCONTROLS] = 
   {
   {  //* 'CLOSE' pushbutton - - - - - - - - - - - - - - - - - - - - -    clPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dlgROWS - 2),           // ulY:       upper left corner in Y
      11,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      9,                            // cols:      control columns
      "  ^CLOSE  ",                 // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[vlPB]                     // nextCtrl:  link in next structure
   },
   {  //* 'VIEW LOG' pushbutton - - - - - - - - - - - - - - - - - - - -   vlPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[clPB].ulY,                 // ulY:       upper left corner in Y
      short(ic[clPB].ulX + ic[clPB].cols + 3), // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      10,                           // cols:      control columns
      " ^VIEW LOG ",                // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[slPB]                     // nextCtrl:  link in next structure
   },
   {  //* 'SAVE LOG' pushbutton - - - - - - - - - - - - - - - - - - - -   slPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[vlPB].ulY,                 // ulY:       upper left corner in Y
      short(ic[vlPB].ulX + ic[vlPB].cols + 3), // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      10,                           // cols:      control columns
      " ^SAVE LOG ",                // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
   } ;

   this->dPtr->SetDialogObscured () ;   // Save parent dialog display data

   //* Open the interface dialog *
   //* Initial parameters for dialog window *
   InitNcDialog dInit( dlgROWS,        // number of display lines
                       dlgCOLS,        // number of display columns
                       dlgYPOS,        // Y offset from upper-left of terminal 
                       dlgXPOS,        // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Set the dialog title *
      dp->SetDialogTitle ( "  Operation Summary  ", hColor ) ;

      gString localLog,                // for viewing log file
              publicLog,               // filespec for public copy of log file
              gsSummary ;              // contains summary stats for log file

      //* 1) Type of operation   *
      gString gsOut, gsa, gsb, gsc ;
      winPos wp( 1, 2 ), wpd ;
      short opIndex = (bkSet.btype == bsetACreate) ? 0 :
                      (bkSet.btype == bsetAUpdate) ? 1 : 2 ;
      wpd = dp->WriteString ( wp.ypos++, wp.xpos, "Operation    : ", hColor ) ;
      dp->WriteString ( wpd, aopName[opIndex], dColor ) ;
      //* 1a) Name of target archive *
      short arcnameIndex = (bkSet.barc.findlast( fSLASH )) + 1 ;
      gsOut.compose( " %S ", &bkSet.barc.gstr()[arcnameIndex] ) ;
      dp->WriteString ( ++wpd.ypos, wpd.xpos, gsOut, this->cs.tf ) ;
      wp.ypos += 2 ;

      if ( (bkSet.btype == bsetACreate) || (bkSet.btype == bsetAUpdate) )
      {
         tnFName tnf ;
         UINT    arcFiles,
                 cbFiles = this->ClipboardFiles () ;
         UINT64  arcBytes, 
                 cbBytes = this->ClipboardSize () ;
         this->ArchiveSummary ( bkSet.barc, tnf, arcFiles, arcBytes ) ;
         wpd = dp->WriteString ( wp.ypos++, (wp.xpos + 7), aHdr, hColor | ncuATTR ) ;
         wpd.xpos += 2 ;
         wpd = dp->WriteString ( wpd, sHdr, hColor | ncuATTR ) ;
         if ( bkSet.btype == bsetAUpdate )
         {
            wpd.xpos += 2 ;
            dp->WriteString ( wpd, oHdr, hColor | ncuATTR ) ;
         }
         gsOut.compose( "%s\n%s\n%s\n%s", fHdr, bHdr, tHdr, eHdr ) ;
         dp->WriteParagraph ( wp, gsOut, hColor ) ;
         wp.xpos += 8 ;
         gsa.formatInt ( arcFiles, 6 ) ;
         gsb.formatInt ( cbFiles, 6 ) ;
         gsc.formatInt ( bkSet.uaFiles, 6, true ) ;
         gsOut.compose ( "%S       %S", gsa.gstr(), gsb.gstr() ) ;
         if ( bkSet.btype == bsetAUpdate )
            gsOut.append( "  (%S)", gsc.gstr() ) ;
         gsOut.append( L'\n' ) ;
         wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
         if ( bkSet.btype == bsetACreate )
            gsSummary.compose( summaryTemplate[0], aopName[opIndex], 
                               aHdr, sHdr, fHdr, gsOut.gstr() ) ;
         else
            gsSummary.compose( summaryTemplate[1], aopName[opIndex], 
                               aHdr, sHdr, oHdr, fHdr, gsOut.gstr() ) ;

         gsa.formatInt ( arcBytes, 6, false, false, false, fiKb ) ;
         gsb.formatInt ( cbBytes, 6, false, false, false, fiKb ) ;
         gsc.formatInt ( bkSet.uaBytes, 6, true, false, false, fiKb ) ;
         gsOut.compose ( "%S       %S", gsa.gstr(), gsb.gstr() ) ;
         if ( bkSet.btype == bsetAUpdate )
            gsOut.append( "  (%S)", gsc.gstr() ) ;
         gsOut.append( L'\n' ) ;
         wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
         gsSummary.append( "%s%S", bHdr, gsOut.gstr() ) ;

         gsa.formatInt ( tnf.fBytes, 6, false, false, false, fiKb ) ;
         gsc.formatInt ( bkSet.uaTarget, 6, true, false, false, fiKb ) ;
         gsOut = gsa ;
         if ( bkSet.btype == bsetAUpdate )
            gsOut.append( "               (%S)", gsc.gstr() ) ;
         gsOut.append( L'\n' ) ;
         wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
         gsSummary.append( "%s%S", tHdr, gsOut.gstr() ) ;

         gsOut.formatInt ( bkSet.ferrors, 6 ) ;
         gsOut.append( L'\n' ) ;
         dp->WriteString ( wp, gsOut, dColor ) ;
         gsSummary.append( "%s%S", eHdr, gsOut.gstr() ) ;
      }
      else     // (bkSet.btype == bsetAExpand)
      {
         tnFName tnf ;
         UINT    arcFiles ;
         UINT64  arcBytes ;
         this->ArchiveSummary ( bkSet.barc, tnf, arcFiles, arcBytes ) ;
         dp->WriteString ( wp.ypos++, (wp.xpos + 7), xsHdr, hColor ) ;
         wpd = dp->WriteString ( wp.ypos++, (wp.xpos + 7), aHdr, hColor | ncuATTR ) ;
         wpd.xpos += 2 ;
         dp->WriteString ( wpd, xxHdr, hColor | ncuATTR ) ;
         gsa.formatInt ( arcFiles, 6 ) ;
         gsb.formatInt ( bkSet.uaFiles, 6 ) ;
         wpd = dp->WriteString ( wp.ypos++, wp.xpos, fHdr, hColor ) ;
         gsOut.compose ( "%S     %S", gsa.gstr(), gsb.gstr() ) ;
         dp->WriteString ( wpd, gsOut, dColor ) ;
         gsSummary.compose( summaryTemplate[2], aopName[opIndex], 
                            xsHdr, aHdr, xxHdr, fHdr, gsOut.gstr() ) ;

         gsa.formatInt ( arcBytes, 6, false, false, false, fiKb ) ;
         gsb.formatInt ( bkSet.uaTarget, 6, false, false, false, fiKb ) ;
         wpd = dp->WriteString ( wp.ypos++, wp.xpos, bHdr, hColor ) ;
         gsOut.compose ( "%S     %S", gsa.gstr(), gsb.gstr() ) ;
         dp->WriteString ( wpd, gsOut, dColor ) ;
         gsSummary.append( "%s%S\n", bHdr, gsOut.gstr() ) ;

         wpd = dp->WriteString ( wp.ypos++, wp.xpos, eHdr, hColor ) ;
         gsOut.formatInt ( bkSet.ferrors, 6 ) ;
         dp->WriteString ( wpd, gsOut, dColor ) ;
         gsSummary.append( "%s%S\n", eHdr, gsOut.gstr() ) ;
      }

      //* Log message position *
      wp = { short(ic[clPB].ulY - 1), ic[clPB].ulX } ;

      //* Format the log file:        *
      //* a) get a temp-file filespec *
      //* b) write formatted data     *
      this->fmPtr->CreateTempname ( localLog ) ;
      this->Archive_FormatLog ( bkSet, localLog, gsSummary ) ;

      dp->RefreshWin () ;                 // make the dialog visible

      uiInfo   Info ;                     // user interface data returned here
      short    icIndex = ZERO ;           // index of control with input focus
      bool     done = false ;             // loop control
      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;

            //* If a Pushbutton was pressed *
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == clPB )
               {
                  done = true ;
               }
               else if ( Info.ctrlIndex == vlPB )
               {
                  //* Create the command to view the log file *
                  gString gsCmd( "less -c %S", localLog.gstr() ) ;

                  //* Shell out to view the file *
                  dp->ShellOut ( soX, gsCmd.ustr() ) ;

                  //* Restore the display *
                  this->dPtr->RefreshWin () ;
                  this->dPtr->SetDialogObscured () ;
                  dp->RefreshWin () ;
               }
               else if ( Info.ctrlIndex == slPB )
               {
                  //* Save the temporary log file to the public log file *
                  publicLog.compose( "%S/%S.log", bkSet.btrg.gstr(), 
                                     &bkSet.barc.gstr()[arcnameIndex] ) ;
                  short nIndex = (publicLog.findlast( fSLASH )) + 1 ;
                  if ( (this->Archive_SaveLog ( localLog, publicLog )) == OK )
                     wpd = dp->WriteString ( wp, "Log Written: ", hColor ) ;
                  else
                     wpd = dp->WriteString ( wp, "Error Writing Log: ", hColor ) ;
                  dp->WriteString ( wpd, &publicLog.gstr()[nIndex], dColor, true ) ;

                  //* Update the parent dialog's display *
                  dp->SetDialogObscured () ;
                  this->dPtr->RefreshWin () ;
                  this->RefreshCurrDir () ;
                  this->dPtr->SetDialogObscured () ;
                  dp->RefreshWin () ;

                  //* Disable the 'SAVE LOG' Pushbutton *
                  //* to avoid duplicate saves.         *
                  dp->PrevControl () ;    // remove focus from target control
                  dp->ControlActive ( slPB, false ) ;
               }
            }

            if ( ! done && ! Info.viaHotkey )
            {
               if ( Info.keyIn == nckSTAB )
                  icIndex = dp->PrevControl () ; 
               else
                  icIndex = dp->NextControl () ;
            }
         }
      }     // while()

      this->fmPtr->DeleteFile ( localLog.ustr() ) ; // delete the temporary log file
   }
   if ( dp != NULL )                      // close the window
      delete ( dp ) ;


   this->dPtr->RefreshWin () ;   // Restore parent dialog display

}  //* End Archive_Summary() *

//*************************
//*   Archive_FormatLog   *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* Concatenate the temporary-file data to create a formatted log file.        *
//*                                                                            *
//* Input  : bkSet   : operational parameters                                  *
//*                    'blog' is the filespec for the temporary file           *
//*          trgLog  : filespec for target log file                            *
//*          summary : operation summary data record to be inserted into log   *
//*                                                                            *
//* Returns: OK  if successful                                                 *
//*          ERR if error(s) writing log file                                  *
//******************************************************************************
//* Notes:                                                                     *
//* 1) Open the source file for reading.                                       *
//* 2) Open the target file for writing (truncate).                            *
//* 3) Read the header (2 lines) from the source file and write them to target.*
//* 4) Write source and target paths into the target file.                     *
//* 5) Insert the summary statistics into the target file.                     *
//* 6) Copy the remaining source file to target.                               *
//* 7) Close both source and target files.                                     *
//******************************************************************************

short FileDlg::Archive_FormatLog ( const BSet& bkSet, const gString& trgLog, 
                                   const gString& summary )
{
   gString gsOut ;               // output formatting
   bool readError = false,       // 'true' if read error
        writeError = false ;     // 'true' if write error

   //* Open the source file *
   ifstream ifs( bkSet.blog.ustr(), ifstream::in ) ;

   //* Open the target file *
   ofstream ofs( trgLog.ustr(), ofstream::out | ofstream::trunc ) ;
   if ( ifs.is_open() && ofs.is_open() )
   {
      char  lineData[gsDFLTBYTES] ; // raw UTF-8 input

      //* Copy source file header *
      ifs.getline ( lineData, gsDFLTBYTES, NEWLINE ) ;
      if ( ifs.good() || ifs.gcount() > ZERO )
         ofs << lineData << endl ;
      else  readError = true ;
      ifs.getline ( lineData, gsDFLTBYTES, NEWLINE ) ;
      if ( ifs.good() || ifs.gcount() > ZERO )
         ofs << lineData << endl ;
      else  readError = true ;

      //* Source and Target paths   *
      ofs << "\nSource:\n   " << ((bkSet.btype == bsetAExpand) ? 
                                 bkSet.barc.ustr() : bkSet.bsrc.ustr()) 
          << "\nTarget:\n   " << ((bkSet.btype == bsetAExpand) ? 
                                 bkSet.btrg.ustr() : bkSet.barc.ustr())
          << "\n" << endl ;

      //* Insert summary statistics *
      ofs << summary.ustr() << endl ;

      //* Copy remainder of source to target *
      bool done = readError ;
      while ( ! done )
      {
         ifs.getline ( lineData, gsDFLTBYTES, NEWLINE ) ;
         if ( ifs.good() || ifs.gcount() > ZERO )
            ofs << lineData << endl ;
         else
         {
            ofs << "END-OF-LOG\n\n" << endl ;
            done = true ;
         }
      }
   }

   //* Close the files *
   if ( ifs.is_open() )
      ifs.close() ;
   else
      readError = true ;
   if ( ofs.is_open() )
      ofs.close() ;
   else
      writeError = true ;

   return ( (!readError && !writeError) ? OK : ERR ) ;

}  //* End Archive_FormatLog() *

//*************************
//*    Archive_SaveLog    *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* Copy the temporary log file to the specified public-access log file.       *
//*                                                                            *
//* Input  : srcLog  : temp file containing the formatted log data             *
//*          trgLog  : public-access target log file                           *
//*                    If target exists, the new data will be appended to it.  *
//*                                                                            *
//* Returns: OK  if successful                                                 *
//*          ERR if error(s) writing log file                                  *
//******************************************************************************

short FileDlg::Archive_SaveLog ( const gString& srcLog, const gString& trgLog )
{
   bool readError = false,       // 'true' if read error
        writeError = false ;     // 'true' if write error

   //* Open the source file *
   ifstream ifs( srcLog.ustr(), ifstream::in ) ;

   //* Open the target file *
   ofstream ofs( trgLog.ustr(), ofstream::out | ofstream::app ) ;
   if ( ifs.is_open() && ofs.is_open() )
   {
      char  lineData[gsDFLTBYTES] ; // raw UTF-8 input

      //* Copy remainder of source to target *
      bool done = false ;
      while ( ! done )
      {
         ifs.getline ( lineData, gsDFLTBYTES, NEWLINE ) ;
         if ( ifs.good() || ifs.gcount() > ZERO )
            ofs << lineData << endl ;
         else
            done = true ;
      }
   }
   //* Close the files *
   if ( ifs.is_open() )
      ifs.close() ;
   else
      readError = true ;
   if ( ofs.is_open() )
      ofs.close() ;
   else
      writeError = true ;

   return ( (!readError && !writeError) ? OK : ERR ) ;

}  //* End Archive_SaveLog() *

//*************************
//*  Archive_UpdateQuery  *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* Caller has identified BOTH a target archive AND data on the clipboard.     *
//* Ask the user how to proceed. (see return values)                           *
//*                                                                            *
//* Input  : bkSet  : operational parameters                                   *
//*                   'barc' is the filespec for an existing archive (if any)  *
//*          hlPath : if highlighted file is an archive, pathspec of archive   *
//*                   (else, empty string)                                     *
//*          hlaType: member of enum arcType indicating archive type of        *
//*                   highlighted file (atNONE if file is not an archive)      *
//*          cbaType: member of enum arcType indicating archive type of        *
//*                   the only file on the clipboard                           *
//*                   (atNONE if file on clipboard is not an archive)          *
//*                   (OR if multiple clipboard files               )          *
//*          cbFiles: number of files on the clipboard                         *
//*                                                                            *
//* Returns: member of enum bsetType corresponding to the type of archive      *
//*          operation selected:                                               *
//*             bsetACreate - create an archive using clipboard data           *
//*             bsetAUpdate - update specified archive using clipboard data    *
//*             bsetAExpand - expand (unpack) specified archive                *
//*             bsetArchive - indicates that user aborted the operation        *
//******************************************************************************
//* Logic:                                                                     *
//* ------                                                                     *
//* There are (effectively) no 'selected' files at this point because if user  *
//* had initially 'selected' files, caller has moved them to the clipboard.    *
//*                                                                            *
//* The parameter 'hlaType' should never be 'atNONE'. To put it another way,   *
//* the highlighted file will always be an archive; otherwise caller would     *
//* have been able to programatically determine the user's desired operation.  *
//*                                                                            *
//* The clipboard will always contain at least one file (cbFiles >= 1);        *
//* otherwise caller would have been able to programically determine the       *
//* user's desired operation.                                                  *
//*                                                                            *
//* 1) If highlighted file is an archive file (hlaType != atNONE):             *
//*    a) It could be the target of a potential Update operation (if clipboard *
//*       contains source data).                                               *
//*    b) It could be the source for an Expand operation (any data on the      *
//*       clipboard will be ignored).                                          *
//* 2) If clipboard contains a single archive file (cbaType != atNONE):        *
//*    a) It could be the source for an Expand operation.                      *
//*    b) It could be the source for an Update operation (if the highlighted   *
//*       file is an archive AND if they are not the same archive).            *
//* 3) If the clipboard contains data (but cbaType==atNONE):                   *
//*    a) Data could be the source for a Create operation.                     *
//*    b) Data could be the source for an Update (if the highlighted file is   *
//*       an archive AND if the highlighted file is not also on the clipboard).*
//*                                                                            *
//* Special Note:                                                              *
//* -------------                                                              *
//* FileMangler does not allow an archive to be added to itself; however, we   *
//* do not test for that here. Instead, we leave that test for when the        *
//* Archive_Prompt() method has been called for the 'bsetAUpdate' operation.   *
//******************************************************************************

bsetType FileDlg::Archive_UpdateQuery ( const BSet& bkSet, const gString& hlPath,
                                        arcType hlaType, arcType cbaType, UINT cbFiles )
{
   //* If user has specified NEITHER source data, NOR an archive, *
   //* then we cannot proceed.                                    *
   if ( (cbaType == atNONE) && (hlaType == atNONE) && (cbFiles == ZERO) )
   {
      const char* alertMsg[] =
      {
         " ALERT ALERT ",
         " ",
         "For archive operations, please place source data",
         "on the clipboard, highlight the name of an existing",
         "archive file, or both.",
         "Please see documentation: \"Working With Archives\"",
         NULL
      } ;
      this->InfoDialog ( alertMsg ) ;
      return ( bsetArchive ) ;   //*** Note the Early Return! ***
   }


   //* Target of a potential Update/Append operation must be the highlighted  *
   //* file. Therefore, the 'Update' radiobutton is active ONLY if target     *
   //* exists and is an uncompressed Tar archive _OR_ a Zip archive.          *
   bool upRB_Active = (bool)((hlaType == atTAR) || (hlaType == atZIP)) ;

   //* Source data for a potential Create operation must exist.               *
   //* (Note: There should always be clipboard data present.)                 *
   bool crRB_Active =  (bool)( cbFiles > ZERO ) ;

   //* Source of a potential Expand operation must exist. Either the          *
   //* highlighted file is an archive _OR_ an archive is the only file on     *
   //* the clipboard. (Note: There should always be a specified archive.)     *
   bool exRB_Active = (bool)((hlaType != atNONE) || (cbaType != atNONE)) ;


   //*************************
   //**  Create the Dialog  **
   //*************************
   const char* rbLabels[] =      // Radiobutton labels
   {
      "^Append source data to the target archive\n"
      "(note: must be either Zip or uncompressed Tar)",
      "^Create a new archive file using source data\n"
      "from clipboard (ignore specified archive)",
      "^Extract files from archive (ignore clipboard)",
   } ;
   const short dialogROWS = 13,
               dialogCOLS = INFODLG_WIDTH,
               dialogYPOS = this->fulY + 2,
               dialogXPOS = this->fulX + (this->fMaxX / 2) - (dialogCOLS / 2) ;
   const attr_t dColor  = this->cs.sd,
                hColor  = this->cs.em,
                gyColor = this->cs.sb & ~ncbATTR ;
   bsetType status = bsetArchive ;   // return value

   //* List of controls in this dialog *
   enum qControls : short
   {
      okPB = ZERO, canPB, upRB, crRB, exRB, qCONTROLS
   } ;

   //** Define the dialog controls **
   InitCtrl ic[qCONTROLS] = 
   {
   {  //* 'OK' pushbutton - - - - - - - - - - - - - - - - - - - - - -     okPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 2),        // ulY:       upper left corner in Y
      short(dialogCOLS / 2 - 9),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      8,                            // cols:      control columns
      "   ^OK   ",                  // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[canPB]                    // nextCtrl:  link in next structure
   },
   {  //* 'CANCEL' pushbutton - - - - - - - - - - - - - - - - - - - -    canPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 2),        // ulY:       upper left corner in Y
      short(ic[okPB].ulX + ic[okPB].cols + 2), // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      8,                            // cols:      control columns
      " CA^NCEL ",                  // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[upRB]                     // nextCtrl:  link in next structure
   },
   {  //* 'APPEND' radio button     - - - - - - - - - - - - - - - - - -   upRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      upRB_Active,                  // rbSelect:  initial value
      5,                            // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      (upRB_Active ? dColor : gyColor), // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      rbLabels[0],                  // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      upRB_Active,                  // active:    allow control to gain focus
      &ic[crRB]                     // nextCtrl:  link in next structure
   },
   {  //* 'CREATE' radio button     - - - - - - - - - - - - - - - - - -   crRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      (!upRB_Active),               // rbSelect:  initial value
      short(ic[upRB].ulY + 2),      // ulY:       upper left corner in Y
      ic[upRB].ulX,                 // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      dColor,                       // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      rbLabels[1],                  // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      crRB_Active,                  // active:    allow control to gain focus
      &ic[exRB]                     // nextCtrl:  link in next structure
   },
   {  //* 'EXPAND' radio button     - - - - - - - - - - - - - - - - - -   exRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initial value
      short(ic[crRB].ulY + 2),      // ulY:       upper left corner in Y
      ic[crRB].ulX,                 // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      dColor,                       // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      rbLabels[2],                  // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      exRB_Active,                  // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
   } ;

   //* Save the parent dialog's display data *
   this->dPtr->SetDialogObscured () ;

   //* Open the interface dialog *
   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       dialogYPOS,     // Y offset from upper-left of terminal 
                       dialogXPOS,     // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltDUAL,       // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Set the dialog title *
      dp->SetDialogTitle ( "  Select an Operation  ", this->cs.em ) ;

      //* Set radio buttons as an exclusive-OR group *
      //* (only one can be selected at any moment).  *
      short XorGroup[] = { upRB, crRB, exRB, -1 } ;
      dp->GroupRadiobuttons ( XorGroup ) ;

      UINT cbFiles = this->ClipboardFiles () ;
      gString gsFmt( cbFiles, 7, true ),
              gsOut( "You have specified an existing archive file,\n\n"
                     "and there %s %S source file%son the clipboard.\n", 
                     (cbFiles == 1 ? "is" : "are"), gsFmt.gstr(),
                     (cbFiles == 1 ? " " : "s ")) ;
      winPos wp = dp->WriteParagraph ( 1, 3, gsOut, hColor ) ;
      short nIndex = (hlPath.findlast( fSLASH )) + 1 ;  // name of highlighted file
      if ( nIndex < ZERO )    // no clipboard archive specified
         nIndex = (bkSet.barc.findlast( fSLASH )) + 1 ;
      if ( nIndex < ZERO ) nIndex = ZERO ;   // stay within the array
      gsOut.compose( " %S ", ((hlaType != atNONE) ? 
                               &hlPath.gstr()[nIndex] :
                               &bkSet.barc.gstr()[nIndex]) ) ;
      //* Offset in X for archive name display *
      short nxoffset = (dialogCOLS / 2) - ((gsOut.gschars() + 1) / 2) ;
      dp->WriteString ( 2, nxoffset, gsOut, this->cs.tf ) ;
      //* If there is a single archive file on the clipboard *
      //* display its name.                                  *
      if ( cbaType != atNONE )
      {
         nIndex = (bkSet.barc.findlast( fSLASH )) + 1 ;
         gsOut.compose( " %S ", &bkSet.barc.gstr()[nIndex] ) ;
         nxoffset = (dialogCOLS / 2) - ((gsOut.gschars() + 1) / 2) ;
         dp->WriteString ( 4, nxoffset, gsOut, this->cs.tf ) ;
      }
      else
         dp->WriteString ( wp.ypos, wp.xpos + 11, 
                           "Please select an operation.", hColor | ncuATTR ) ;

      //* Do not offer 'Update' if target is a compressed Tar archive.*
      if ( ! upRB_Active )
      {
         dp->WriteParagraph ( ic[upRB].ulY + ic[upRB].labY,
                              ic[upRB].ulX + ic[upRB].labX,
                              &rbLabels[0][1], gyColor ) ;
      }

      //* Do not offer 'Expand' unless a source archive is identified.*
      if ( ! exRB_Active )
      {
         dp->WriteParagraph ( ic[exRB].ulY + ic[exRB].labY,
                              ic[exRB].ulX + ic[exRB].labX,
                              &rbLabels[0][1], gyColor ) ;
      }

      dp->RefreshWin () ;                 // make the dialog visible

      uiInfo Info ;                       // user interface data returned here
      short  icIndex = ZERO ;             // index of control with input focus
      bool   done = false ;               // loop control
      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;

            //* If a Pushbutton was pressed *
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == okPB )
               {
                  //* Initialize the return value *
                  short selection = dp->GetRbGroupSelection ( upRB ) ;
                  switch ( selection )
                  {
                     case upRB:     status = bsetAUpdate ; break ;
                     case exRB:     status = bsetAExpand ; break ;
                     case crRB:
                     default:       status = bsetACreate ; break ;
                  } ;
               }
               else if ( Info.ctrlIndex == canPB )
               {
                  status = bsetArchive ;
               }
               done = true ;
            }
         }
         else if ( ic[icIndex].type == dctRADIOBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditRadiobutton ( Info ) ;
         }

         if ( ! done && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }        // while()
   }
   if ( dp != NULL )                      // close the window
      delete ( dp ) ;

   this->dPtr->RefreshWin () ;   // Restore parent dialog display

   return status ;

}  //* End Archive_UpdateQuery() *

//*************************
//*   Archive_FileList    *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* Write a list of all top-level items on the clipboard to a temporary file.  *
//* The list is used as an argument to the '--files-from' tar option and       *
//* as an argument to the '-i@' zip option.                                    *
//*                                                                            *
//* Input  : fileList  : filespec of target temp file                          *
//*          zipWild   : (optional, 'false' by default)                        *
//*                      if 'false', write subdirectory names as listed        *
//*                      if 'true',  For ZIP Archive Input Lists ONLY!         *
//*                                  append a wildcard to the end of each      *
//*                                  subdirectory name, Example: "SubDir/*"    *
//* Returns: OK                                                                *
//******************************************************************************

short FileDlg::Archive_FileList ( const gString& fileList, bool zipWild )
{
   #if ENABLE_DEBUGGING_CODE != 0 && enableDBBAK != 0
   dbgs.compose( "\n%s Clipboard Data\n---------------------", debugHDR ) ;
   this->bkWriteLogfile ( dbgs ) ;
   #endif   // ENABLE_DEBUGGING_CODE && enableDBBAK

   ofstream ofs( fileList.ustr(), ofstream::out | ofstream::trunc ) ;
   for ( UINT nodeIndex = ZERO ; nodeIndex < clipBoard_dirTrees ; ++nodeIndex )
   {
      ofs << clipBoard[BNODE].nextLevel[nodeIndex].dirStat.fName ;
      if ( zipWild )
         ofs << "/*" ;
      ofs << endl ;

      #if ENABLE_DEBUGGING_CODE != 0 && enableDBBAK != 0
      if ( zipWild )
      {
         dbgs.compose("%s/*", clipBoard[BNODE].nextLevel[nodeIndex].dirStat.fName ) ;
         this->bkWriteLogfile ( dbgs ) ;
      }
      else
         this->bkWriteLogfile ( clipBoard[BNODE].nextLevel[nodeIndex].dirStat.fName ) ;
      #endif   // ENABLE_DEBUGGING_CODE && enableDBBAK
   }
   for ( UINT fileIndex = ZERO ; fileIndex < clipBoard[BNODE].tnFCount ; ++fileIndex )
   {
      ofs << clipBoard[BNODE].tnFiles[fileIndex].fName << endl ;

      #if ENABLE_DEBUGGING_CODE != 0 && enableDBBAK != 0
      this->bkWriteLogfile ( clipBoard[BNODE].tnFiles[fileIndex].fName ) ;
      #endif   // ENABLE_DEBUGGING_CODE && enableDBBAK
   }
   ofs.close() ;     // close fileList

   #if ENABLE_DEBUGGING_CODE != 0 && enableDBBAK != 0
   dbgs.compose( "%s ---------------------\n", debugHDR ) ;
   this->bkWriteLogfile ( dbgs ) ;
   #endif   // ENABLE_DEBUGGING_CODE && enableDBBAK

   return OK ;

}  //* End Archive_FileList() *

//*************************
//*  ArchiveOnClipboard   *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* If there is exactly one file on the clipboard, then test whether it is     *
//* an archive file.                                                           *
//*                                                                            *
//* Note: OpenDocument/OpenXML files ARE reported as archive files by          *
//*       this method.                                                         *
//*                                                                            *
//* Input  : arcPath : (by reference) receives path/filename of archive if     *
//*                    found; else unchanged                                   *
//*                                                                            *
//* Returns: member of enum arcType                                            *
//******************************************************************************

arcType FileDlg::ArchiveOnClipboard ( gString& arcPath )
{
   arcType arcFound = atNONE ;         // return value

   if ( ((this->ClipboardFiles ()) == 1) &&
        (clipBoard[BNODE].tnFCount == 1) )
   {
      arcPath.compose( "%s/%s", clipBoard_srcPath, clipBoard[BNODE].tnFiles[ZERO].fName ) ;
      arcFound = this->ArchiveTarget ( arcPath ) ;
   }

   return arcFound ;

}  //* End ArchiveOnClipboard() *

//*************************
//*     ScanClipboard     *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* Scan the files on the clipboard for a file matching the specified filespec.*
//*                                                                            *
//* A comparison of filenames alone is not a guarantee that a matching filename*
//* is actually the same file; it could be a different file with the same name.*
//* Therefore the full source filespec is required for each comparison.        *
//*                                                                            *
//* Input  : srcSpec : (by reference) full filespec of file for which to scan  *
//*                                                                            *
//* Returns: 'true' if matching filename found, else 'false'                   *
//******************************************************************************

bool FileDlg::ScanClipboard ( const gString& srcSpec )
{
   gString cbPath( clipBoard_srcPath ),   // base path of data on clipboard
           trgPath ;                      // path of clipboard file to be tested
   UINT    dCount,                        // actual top-level directory count
           fCount ;                       // actual top-level file count
   bool matchFound = false ;              // return value

   //* Be sure that clipboard contains data.*
   clipBoard[BNODE].GetActual ( dCount, fCount ) ;
   if ( (dCount > ZERO) || (fCount > ZERO) )
   {
      //* Scan top-level file list first. This is the inexpensive test *
      //* because for Archive_Prompt() at least, 'srcSpec' is almost   *
      //* always in caller's  current directory.                       *
      for ( UINT fIndex = ZERO ; fIndex < fCount ; ++fIndex )
      {
         trgPath.compose( "%S/%s", cbPath.gstr(), clipBoard[BNODE].tnFiles[fIndex].fName ) ;
         if ( srcSpec == trgPath )
         { matchFound = true ; break ; }
      }

      //* If match not yet found, do recursive scan.*
      if ( !matchFound )
      {
         //* Scan each node (subdirectory tree).*
         for ( UINT dIndex = ZERO ; dIndex < dCount ; ++dIndex )
         {
            trgPath.compose( "%S/%s", cbPath.gstr(), 
                             clipBoard[BNODE].nextLevel[dIndex].dirStat.fName ) ;
            if ( (matchFound = scanCb ( srcSpec, trgPath, &clipBoard[BNODE].nextLevel[dIndex] )) )
               break ;
         }
      }
   }  // ((dCount > ZERO) || (fCount > ZERO))
   return matchFound ;

}  //* End ScanClipboard() *

//*************************
//*        scanCb         *
//*************************
//******************************************************************************
//* Non-member method:                                                         *
//* ------------------                                                         *
//* Recursively scan the contents of the FileMangler clipboard searching for   *
//* the specified file.                                                        *
//* Called only by ScanClipboard().                                            *
//*                                                                            *
//* Input  : srcSpec : (by reference) full filespec of file for which to scan  *
//*          basePath: (by reference) path of base directory for this node     *
//*          tnPtr   : pointer to TreeNode object as this level of the node    *
//*                                                                            *
//* Returns: 'true' if matching filename found, else 'false'                   *
//******************************************************************************

static bool scanCb ( const gString& srcSpec, const gString& basePath, TreeNode* tnPtr )
{
   gString trgPath ;                      // path of clipboard file to be tested
   UINT    dCount,                        // actual top-level directory count
           fCount ;                       // actual top-level file count
   bool matchFound = false ;              // return value

   tnPtr->GetActual ( dCount, fCount ) ;
   //* If there are non-directory files at this level, scan them now.*
   for ( UINT fIndex = ZERO ; fIndex < fCount ; ++fIndex )
   {
      trgPath.compose( "%S/%s", basePath.gstr(), tnPtr->tnFiles[fIndex].fName ) ;
      if ( srcSpec == trgPath )
      { matchFound = true ; break ; }
   }

   if ( !matchFound )
   {
      //* Scan each node of the subdirectory tree.*
      for ( UINT dIndex = ZERO ; dIndex < dCount ; ++dIndex )
      {
         trgPath.compose( "%S/%s", basePath.gstr(), 
                          tnPtr->nextLevel[dIndex].dirStat.fName ) ;
         if ( (matchFound = scanCb ( srcSpec, trgPath, &tnPtr->nextLevel[dIndex] )) )
            break ;
      }
   }
   return matchFound ;

}  //* end scanCb() *

//*************************
//* ArchiveClipboardList  *
//*************************
//******************************************************************************
//* Create an archive file containing all files and directory trees on the     *
//* clipoard. Called when target of a backup operation is an archive i.e.      *
//* (i.e. from Backup_DirTree()).                                              *
//*                                                                            *
//* Input  : bkSet : operational parameters                                    *
//*                  'barc' is the filespec for the target archive             *
//*                  'blog' is filespec for operational log                    *
//*                                                                            *
//* Returns: OK if successful, or ERR if one or more errors detected           *
//*          bkSet members receive accumulated data:                           *
//*           'ftotal'   receives number of source files on clipboard          *
//*           'fupdated' receives number of files added to archive             *
//*           'ferrors'  receives number of errors during processing           *
//******************************************************************************

short FileDlg::ArchiveClipboardList ( BSet& bkSet )
{
   this->SetErrorCode ( ecNOERROR ) ;  // clear any previous error code

   //* We do not know what the current-working-directory (CWD) is; however *
   //* for the system call to work, the CWD must be the directory where    *
   //* the source data live. If the directory change fails (unlikely),     *
   //* the archive-creation error messages will be written to the log.     *
   gString currCwd ;
   this->fmPtr->SetCWD ( bkSet.bsrc, &currCwd ) ;

   //* Create a temp-file log to record the operation *
   this->bkOpenLogfile ( bkSet ) ;

   short status = this->Archive_Create ( bkSet ) ;

   //* Return to the previous CWD *
   this->fmPtr->SetCWD ( currCwd ) ;

   //* Refresh the file list for the target directory *
   //* to display the results of the operation.       *
   this->RefreshCurrDir ( false ) ;

   #if ENABLE_DEBUGGING_CODE != 0 && enableDBBAK != 0
   UINT tot = this->ClipboardFiles () ;
   dbgs.compose( "%s total:%u updated:%u skipped:%u errors:%u", debugHDR, 
                 &tot, &bkSet.fupdated, &bkSet.fskipped, &bkSet.ferrors ) ;
   this->bkWriteLogfile ( dbgs ) ;
   #endif   // ENABLE_DEBUGGING_CODE && enableDBBAK

   //* Close the log file *
   this->bkCloseLogfile () ;

   return status ;

}  //* End ArchiveClipboardList() *

//*************************
//*     ArchiveTarget     *
//*************************
//******************************************************************************
//* Determine whether the specified file is one of the supported archive-file  *
//* types.                                                                     *
//*                                                                            *
//* Scan the filename for a recognized archive filename extension.             *
//* If a match found, return the archive type. Note that this does not         *
//* guarantee that the target is actually of the specified type, only that     *
//* the user named it that way.                                                *
//*                                                                            *
//* Supported archive types: tar (with or without compression), zip, or        *
//* or an OpenDocument/OpenXML file (zip archive).                             *
//*                                                                            *
//* IMPORTANT NOTE: To perform operations on the target file, the corresponding*
//*                 utility or utilities must be correctly installed on the    *
//*                 system.                                                    *
//*                                                                            *
//* Input  : trgPath: path/filename of target file                             *
//*          trgExt : (optional, NULL pointer by default)                      *
//*                   pointer to gString object to receive filename extension  *
//*                                                                            *
//* Returns: member of enum arcType                                            *
//******************************************************************************
//* NOTES:                                                                     *
//* -- Regardless of filename, only 'regular' files may be archives; however,  *
//*    we don't check for that here because the file may not yet exist.        *
//* -- The tar utility parses using a case-sensitive comparison, so we must    *
//*    also be case-sensitive.                                                 *
//* -- Zip archive filenames (including OpenDocument/OpenXML files) ARE NOT    *
//*    case-sensitive, so we test for these separately.                        *
//* -- We do not test for obsolete 'compress' tar extensions '.Z' and '.taZ'.  *
//* -- RAR archives: NOT YET SUPPORTED                                         *
//* -- RPM archives: NOT YET SUPPORTED                                         *
//******************************************************************************

arcType FileDlg::ArchiveTarget ( const gString& trgPath, gString* trgExt )
{
   const short extCnt = 13 ;
   const wchar_t* extName[extCnt] = 
   {
      L".tar",                // uncompressed tar archive
      L".bz2",                // bzip2 compression
      L".tz2",
      L".tbz2",
      L".tbz",
      L".gz",                 // gzip compression
      L".tgz",
      L".taz",
      L".lzma",               // lzma compression
      L".tlz",
      L".lz",                 // lzip compression
      L".lzo",                // lzop compression
      L".xz",                 // xz compression
   } ;
   const short extOff[extCnt] = 
   {
      -5, -5, -5, -6, -5, -4, -5, -5, -6, -5, -4, -5, -4//, -3, 5
   } ;
   const arcType extType[extCnt] =
   {
      atTAR,      // uncompressed tar archive
      atBZIP2,    // tar archive with bz2 compression
      atBZIP2,
      atBZIP2,
      atBZIP2,
      atGZIP,     // tar archive with gzip compression
      atGZIP,
      atGZIP,
      atLZMA,     // tar archive with lzma compression
      atLZMA,
      atLZIP,     // tar archive with lzip compression
      atLZOP,     // tar archive with lzop compression
      atXZ,       // tar archive with xz archive
   } ;

   arcType status = atNONE ;        // return value

   //* Test for valid tar archive filename extension *
   //* and various related compression formats.      *
   short ei = trgPath.findlast( L"." ) ;
   if ( ei >= ZERO )
   {
      for ( short i = ZERO ; i < extCnt ; ++i )
      {
         if ( (trgPath.compare( extName[i], true, extOff[i], ei )) == ZERO )
         {
            status = extType[i] ;
            if ( trgExt != NULL )
               *trgExt = extName[i] ;
            break ;
         }
      }
   }

   //* Because zip archive filenames ARE NOT case sensitive, *
   //* we do a non-sensitive test for them here.             *
   if ( status == atNONE )
   {
      if ( (((ei = trgPath.findlast( L".zip" )) >= ZERO) 
            && (ei == ((trgPath.gschars()) - 5 )))
           ||
           (((ei = trgPath.findlast( L".zipx" )) >= ZERO) 
            && (ei == ((trgPath.gschars()) - 6 ))) )
      {
         status = atZIP ;
      }
      else if ( (this->odIsOpenDocFile ( trgPath, true )) != false )
      {
         status = atODOC ;
         ei = trgPath.findlast( L'.' ) ;  // (we 'know' there is a filename extension)
      }
      else if ( (this->odIsOpenXmlFile ( trgPath, true )) != false )
      {
         status = atOXML ;
         ei = trgPath.findlast( L'.' ) ;  // (we 'know' there is a filename extension)
      }
      else if ( ((ei = trgPath.findlast( L".epub" )) >= ZERO) &&
                (ei == ((trgPath.gschars()) - 6 )) )
      {
         status = atEPUB ;
         ei = trgPath.findlast( L'.' ) ;  // (we 'know' there is a filename extension)
      }
      if ( trgExt != NULL )
         *trgExt = &trgPath.gstr()[ei] ;
   }
   return status ;

}  //* End ArchiveTarget() *

//*************************
//*    ArchiveSummary     *
//*************************
//******************************************************************************
//* Scan the specified archive, counting the number of files it contains and   *
//* the approximate total size of the data when expanded.                      *
//*                                                                            *
//* Input  : trgPath: path/filename of target file                             *
//*          tfFile : (by reference) receives the target-file stats            *
//*          aFiles : (by reference) receives number of files in archive       *
//*          aBytes : (by reference) receives (approximate) expanded total size*
//* Returns: OK  if success                                                    *
//*          ERR if file not found or is not a recognized archive type         *
//******************************************************************************
//* Notes:                                                                     *
//* ======                                                                     *
//* Tar:                                                                       *
//* -- Unfortunately, tar does not provide an option to get this information   *
//*    directly, so we have to print the archive data to a temp file and then  *
//*    scan it for the data we're looking for. This is not foolproof, but      *
//*    works for archives we created ourselves, AND for most test archives     *
//*    (downloaded application source, game patches, etc.).                    *
//* -- Archived directories have a zero(0) size, but they occupy space when    *
//*    expanded. The amount of space required is usually the target            *
//*    filesystem's block size (4096 bytes on ext4 filesystems).               *
//*    - We could query the filesystem for the actual block size, but at this  *
//*      time, we don't consider it worth the CPU cycles.                      *
//*    - A directory file may be larger than a single block if it contains an  *
//*      extremely large number of files. We don't check for this.             *
//*    - In addition, most filesystems will not allocate less than a physical  *
//*      cluster to an object, so the actual size on disk may be larger than   *
//*      reported.                                                             *
//* -- If an archive contains error messages or other non-item data records,   *
//*    they may or may not affect the count reported.                          *
//*    - We do a simple parsing of each entry. The entry should begin with     *
//*      the filetype, which is one of:                                        *
//*      '-'  'b'  'c'  'C'  'd'  'D'  'l'  'M'  'n'  'p'  'P'  's'  '?'       *
//*      We scan this group for the most likely matches first.                 *
//*      The filetype is followed by the mode bits, and the first one should   *
//*      be either 'r' for read access or '-' for no read access.              *
//*                                                                            *
//* Zip / Unzip / Zcat:                                                        *
//* -- Zip/Unzip is compatible with the PKZip archive utility originally       *
//*    implemented in MS-DOS(tm). Although DOS is dead (thankfully), PKZIP     *
//*    lives on as WinZip. ISO/IEC 21320-1:2015 specifies the format of zip    *
//*    archives.                                                               *
//* -- 'unzip -l archive.zip[x]' yield the basic info, but it requires parsing.*
//*    The 'Length' column contains the uncompressed file sizes (+ total size).*
//*    The 'Name' column contains the names of the individual files (+total).  *
//*    Example:                                                                *
//*      Archive:  1_TestData/Zip/At_the_Great_Wall.zip                        *
//*        Length      Date    Time    Name                                    *
//*      ---------  ---------- -----   ----                                    *
//*        5115872  04-24-2015 07:21   Great_Wall-01.JPG                       *
//*        6823345  06-12-2014 10:05   Great_Wall-02.JPG                       *
//*        5295339  04-24-2015 07:23   Great_Wall-03.JPG                       *
//*        1076862  11-04-2015 21:55   Great_Wall-04.JPG                       *
//*         131093  11-04-2015 21:56   Great_Wall-05.JPG                       *
//*      ---------                     -------                                 *
//*       18442511                     5 files                                 *
//*  -- Note that OpenDocument files are Zip files, so we process them as such.*
//*     Note, however, that OpenDocument files have a special "meta" entry     *
//*     which is actually a subdirectory containing one file. However, the     *
//*     base directory is not listed as a separate item.                       *
//*                "META-INF/manifest.xml"                                     *
//*     This is actually two(2) items when expanded. Unzip reports this as a   *
//*     single item extraction, so summary and extraction counts match,        *
//*     so why quibble?                                                        *
//*  -- ePub document files are Zip files, consisting of '.xhtml' text data,   *
//*     plus fonts, images, XML metadata including a table-of-contents, and    *
//*     CSS stylesheets. Most of these files are in the OEBPS subdirectory.    *
//*                                                                            *
//*                                                                            *
//* RAR:                                                                       *
//*   NOT YET IMPLEMENTED                                                      *
//*                                                                            *
//* RPM:                                                                       *
//*   NOT YET IMPLEMENTED                                                      *
//*                                                                            *
//******************************************************************************

short FileDlg::ArchiveSummary ( const gString& trgPath, tnFName& tnFile, 
                                UINT& aFiles, UINT64& aBytes )
{
   aFiles = ZERO ;                  // initialize caller's accumulators
   aBytes = ZERO ;
   tnFile.ReInit() ;
   arcType aType = atNONE ;         // archive type
   short status = ERR ;             // return value


   //* Verify that the source file exists and test for a valid filename *
   if ( (this->TargetExists ( trgPath.ustr(), tnFile ))
        && (tnFile.fType == fmREG_TYPE)
        && ((aType = this->ArchiveTarget ( trgPath )) != atNONE) )
   {
      char cmdBuff[gsDFLTBYTES * 3] ;// command-construction buffer
      gString tfPath ;              // get a unique temporary path/filename *
      this->fmPtr->CreateTempname ( tfPath ) ;

      //* Escape any "special characters" which *
      //* might be misinterpreted by the shell. *
      gString escPath = trgPath ;
      this->EscapeSpecialChars ( escPath, escMIN ) ;

      //* If target is a ZIP archive, OpenDocument/OpenXML *
      //* file or ePub document                            *
      if ( (aType == atZIP)  || (aType == atODOC) || 
           (aType == atOXML) || (aType == atEPUB) )
      {
         //* Create the command *
         const char* zipTemplate = "unzip -lv \"%S\" 1>\"%S\" 2>/dev/null" ;
         snprintf ( cmdBuff, (gsDFLTBYTES * 3), zipTemplate, 
                    escPath.gstr(), tfPath.gstr() ) ;
      }  // (aType==atZIP || aType==atODOC || aType==atOXML || aType==atEPUB)

      //* If target is a Tar archive *
      else
      {
         //* Create the command *
         const char* tarTemplate = "tar -tv --file=\"%S\" --index-file=\"%S\" 2>/dev/null" ;
         snprintf ( cmdBuff, (gsDFLTBYTES * 3), tarTemplate, 
                    escPath.gstr(), tfPath.gstr() ) ;

      }  // (aType == Tar Archive)

      //* Write the archive report to the temp file *
      this->fmPtr->Systemcall ( cmdBuff ) ;

      //* Open the temp file and scan for interesting data *
      ifstream ifs( tfPath.ustr(), ifstream::in ) ;
      if ( ifs.is_open() )
      {
         gString gs ;
         char inBuff[gsDFLTBYTES] ;
         UINT64 tmpSize ;
         short fsOffset ;
         bool done = false ;

         if ( (aType == atZIP)  || (aType == atODOC) || 
              (aType == atOXML) || (aType == atEPUB) )
         {
            short lineCount = ZERO ;

            while ( ! done )
            {
               ifs.getline( inBuff, gsDFLTBYTES, NEWLINE ) ;
               gs = inBuff ;
               if ( ifs.good() || ifs.gcount() > ZERO )
               {
                  //* Read and discard:           *
                  //* 1) filespec line            *
                  //* 2) column heading line      *
                  //* 3) column heading underline *
                  //* Count the lines containing filename records.*
                  if ( lineCount > 2 )
                  {
                     if ( (gs.find( "---" ) == ZERO) )   // totals line
                     {  //* Read the total uncompressed bytes.*
                        ifs.getline( inBuff, gsDFLTBYTES, NEWLINE ) ;
                        if ( ifs.good() || ifs.gcount() > ZERO )
                        {
                           gs = inBuff ;
                           if ( (swscanf ( gs.gstr(), L" %llu", &tmpSize )) == 1 )
                              aBytes = tmpSize ;
                        }
                        status = OK ;
                        break ;
                     }
                     ++aFiles ;
                  }
                  ++lineCount ;
               }
               else
                  done = true ;
            }   // while()
         }     // (atZIP || atODOC || atOXML || atEPUB)

         else
         {
            while ( ! done )
            {
               ifs.getline( inBuff, gsDFLTBYTES, NEWLINE ) ;
               if ( ifs.good() || ifs.gcount() > ZERO )
               {
                  for ( short i = ZERO ; i < ftypeCOUNT ; ++i )
                  {
                     if ( *inBuff == fileTypes[i] )
                     {
                        if ( (inBuff[1] == 'r') || (inBuff[1] == '-') )
                        {
                           ++aFiles ;
                           gs = inBuff ;
                           if ( (fsOffset = gs.find( L' ' )) >= ZERO )
                           {
                              if ( (fsOffset = gs.find( L' ', ++fsOffset )) >= ZERO )
                              {
                                 if ( (swscanf ( &gs.gstr()[fsOffset], L" %llu", &tmpSize )) == 1 )
                                    aBytes += (tmpSize > ZERO) ? tmpSize : minBLOCK ;
                              }
                           }
                        }
                        break ;
                     }
                  }
               }
               else
                  done = true ;
            }     // while()
         }        // (Tar archive)
         ifs.close() ;  // close the file
      }
      this->DeleteFile ( tfPath ) ;   // delete the temp file
   }
   return status ;

}  //* End ArchiveSummary() *

//*************************
//*      pbManager        *
//*************************
//******************************************************************************
//* Manage the visual Progress Bar during file operations.                     *
//* This method should be driven by a secondary thread launched specifically   *
//* to instantiate and manage a Progbar object.                                *
//*                                                                            *
//* The thread will return when:                                               *
//*   a) the operation completes i.e. wip.progress() == 1.0                    *
//*   b) the primary thread sets the wipAbort flag (wip.getAbort() != 'false') *
//*                                                                            *
//* Input  : pbi   : pointer to progress bar definition (partially initialized)*
//*                  'dlgColor' and 'barColor' == color attributes             *
//*                  'cells' == number of character cells (width of progbar)   *
//*                  'horiz' == true (horizontal bar)                          *
//*                  'steps' == number of incremental steps in progbar         *
//*                  'border' == false (no border)                             *
//*                  'yOffset' and 'xOffset' == position of progbar            *
//*          active : pointer to flag is set on entry to signal that           *
//*                   the monitor thread is active                             *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Progress Indicator:                                                        *
//* ===================                                                        *
//* 1) Caller starts a secondary thread just before launching the operation.   *
//*    A) This method monitors the progress of the operation as reported by    *
//*       the 'wip' (work-in-progress) member.                                 *
//*       a) Takes no action for approximately 1.5 seconds.                    *
//*          If operation finishes during this period, return immediately.     *
//*          Exception: If the 'threshold' value == ZERO, then skip the        *
//*          initial waiting period and open the progress bar immediately.     *
//*       b) If operation has not completed during the initial period, then    *
//*          define and instantiate a Progbar object to visually track progress*
//*          of the operation.                                                 *
//*          -- Set initial completion percentage.                             *
//*          -- Enter a loop which sleeps for approximately 150mS and then     *
//*             retrieves the current bytes-completed for calculating the      *
//*             increment.                                                     *
//*          -- Return when operation is complete OR if user aborts the        *
//*             operation.                                                     *
//*                                                                            *
//******************************************************************************
//* Notes:                                                                     *
//* ------                                                                     *
//* The progress bar is instantiated then deleted here, but visually, it       *
//* should persist at least until the target window has been updated.          *
//* The main application thread is responsible for this visual continuity.     *
//******************************************************************************

void FileDlg::pbManager ( pbarInit* pbi, bool* active )
{
   const int pbINTERVAL = 150 ;  // 150 milliseconds between polls of output progress
   UINT   targetFiles = ZERO ;   // target files written so far
   UINT64 targetBytes = ZERO ;   // target bytes written so far
   double currPct,               // current percentage of operation complete
          prevPct ;              // previous percentage of operation complete
   short  progSteps,             // number of Progbar steps completed in an iteration
          remSteps = pbi->steps ;// steps not yet taken

   *active = true ;     // signal to caller that we are active

   //* Wait for a short time to see whether the operation will finish quickly.*
   chrono::duration<short, std::milli>aMoment( pbINTERVAL ) ;
   if ( (this->wip.getThreshold()) > ZERO )  // (see note above)
   {
      for ( short i = ZERO ; i < 10 ; ++i )
      {
         this_thread::sleep_for( aMoment ) ;
         if ( (prevPct = this->wip.progress( targetFiles, targetBytes )) == 1.0 )
            break ;
      }
   }

   //* If initial interval expired and operation is not yet complete,    *
   //* create a visual progress indicator and monitor operation progress.*
   if ( prevPct < 1.0 )
   {
      if ( (this->wip.createPBar( *pbi )) )
      {
         //* Set ProgBar to indicate progress so far *
         remSteps -= progSteps = (double)pbi->steps * prevPct ;
         this->wip.pbarUpdate( progSteps ) ;

         //* Monitor and visually report progress of operation *
         do
         {
            this_thread::sleep_for( aMoment ) ;
            currPct = this->wip.progress( targetFiles, targetBytes ) ;

            if ( currPct > prevPct )
            {
               remSteps -= progSteps = (short)((double)pbi->steps * (currPct - prevPct)) ;

               //* Truncation may have occurred in integer division; *
               //* be sure progbar is filled on final step.          *
               if ( currPct == 1.0 )
                  progSteps += remSteps ;

               //* If progress has been made, report it *
               if ( progSteps > ZERO )
               {
                  this->wip.pbarUpdate( progSteps ) ;
                  prevPct = currPct ;
               }
            }
         }
         while ( (currPct < 1.0) && !(this->wip.getAbort()) ) ;

         //* Wait until parent thread signals for us to return. *
         //* This allows the progress bar to remain visible     *
         //* until after the target file list has been updated. *
         while ( !(this->wip.getAbort()) )
            this_thread::sleep_for( aMoment ) ;
         this->wip.deletePBar() ;      // delete the progress bar object
      }
      else
         this->ProcessCWD () ;      // default processing message (unlikely to happen)
   }  // if(prevPct<1.0)

}  //* End pbManager() *

//*************************
//*      ProgbarInit      *
//*************************
//******************************************************************************
//* Initialize a progress bar to keep user amused. Called by Backup, Synch,    *
//* Copy, Move and others.                                                     *
//* 1) 'dlgColor', 'barColor', 'cells' and 'horiz' are set during instantiation*
//* 2) 'steps', 'border', 'yOffset' and 'xOffset' are explicitly set within    *
//*    this method.                                                            *
//* 3) All other pbarInit members receive default values.                      *
//*                                                                            *
//* The progInit object is created in anticipation of the need for a progress  *
//* bar. The progress bar itself will be instantiated only if data to be       *
//* processed >= monitor threshold AND if the operation extends beyond the     *
//* grace period.                                                              *
//*                                                                            *
//* Input  : cells   : size of progress bar in cells (rows or columns)         *
//*          yOff    : offset in Y from upper-left of application dialog       *
//*          xOff    : offset in X from upper-left of application dialog       *
//*          horiz   : (optional, 'true' by default)                           *
//*                    if 'true'  initialize as a horizontal ProgBar           *
//*                    if 'false' initialize as a vertical ProgBar             *
//*                                                                            *
//* Returns: partially-initialized pbarInit object                             *
//******************************************************************************

pbarInit FileDlg::ProgbarInit ( short cells, short yOff, short xOff, bool horiz )
{
   attr_t dlgColor = (attr_t)(this->cs.scheme == ncbcMA ? nc.br : nc.ma), 
          barColor = dlgColor ;
   pbarInit pbi( dlgColor, barColor, cells, horiz ) ;
   pbi.steps    = cells * 4 ;
   pbi.border   = false ;
   pbi.yOffset  = yOff ;
   pbi.xOffset  = xOff ;
//   // Programmer's Note: The Y/X position of the status control are magic 
//   // numbers based its the "known" position. The FileDlg class does not have 
//   // this information. It is located in the FileMangler-class: ic[dwMsgsTB].
//   pbi.yOffset  = 1 ;
//   pbi.xOffset  = 8 ;

   return ( pbi ) ;

}  //* End ProgbarInit *

//******************************************************************************
//***    This group implements the methods of the 'WorkInProgress' class.    ***
//******************************************************************************

//******************************
//*        Destructor          *
//******************************
WorkInProgress::~WorkInProgress ( void )
{
   //* The progress bar pointer controls dynamically-allocated *
   //* memory, so we must explicitly release it.               *
   this->deletePBar () ;
}

//******************************
//*    Default constructor     *
//******************************
WorkInProgress::WorkInProgress ( void )
{
   this->pBar = NULL ;
   this->reset () ;
}

//******************************
//* Initialization constructor *
//******************************
//******************************************************************************
//* Initializes the source values and optionally sets the progress-bar         *
//* threshold.                                                                 *
//*                                                                            *
//* Input  : sFiles: total number of source files                              *
//*          sBytes: total number of source bytes                              *
//*          thresh: (optional, PB_THRESHOLD by default)                       *
//*                  indicates the threshold (minimum byte count) for          *
//*                  launching a secondary thread to monitor progress of       *
//*                  the operation                                             *
//*                                                                            *
//* Returns: implicitly returns a pointer to the object                        *
//******************************************************************************

WorkInProgress::WorkInProgress ( UINT sFiles, UINT64 sBytes, UINT64 thresh )
{
   this->pBar = NULL ;
   this->reset () ;
   this->srcFiles = sFiles ;
   this->srcBytes = sBytes ;
   this->pbThresh = thresh ;
}

//*************************
//*      initialize       *
//*************************
//******************************************************************************
//* Initializes the source values and optionally sets the progress-bar         *
//* threshold. (All other data members are reset.)                             *
//*                                                                            *
//* Input  : sFiles: total number of source files                              *
//*          sBytes: total number of source bytes                              *
//*          thresh: (optional, PB_THRESHOLD by default)                       *
//*                  indicates the threshold (minimum byte count) for          *
//*                  launching a secondary thread to monitor progress of       *
//*                  the operation                                             *
//*                                                                            *
//* Returns: implicitly returns a pointer to the object                        *
//******************************************************************************

void WorkInProgress::initialize (  UINT sFiles, UINT64 sBytes, UINT64 thresh )
{
   //* Obtain ownership lock on data members *
   lock_guard<mutex> guard ( this->data_Lock ) ;

   this->reset () ;
   this->srcFiles = sFiles ;
   this->srcBytes = sBytes ;
   this->pbThresh = thresh ;
   // As we leave the scope of the lock, the lock is freed.
}

//*************************
//*       progress        *
//*************************
//******************************************************************************
//* Reports the percentage of the source data which have been processed so far.*
//*                                                                            *
//* Input  : fProc : (by reference) receives number of source files processed  *
//*          bProc : (by reference) receives number of source bytes processed  *
//*                                                                            *
//* Returns: percentage of operation completed                                 *
//*          This is a floating point value: >= 0.00 && <= 1.00                *
//*          - Caller's parameters are also receive a copy of completion       *
//*            accumulators.                                                   *
//******************************************************************************

double WorkInProgress::progress ( UINT& fProc, UINT64& bProc )
{
   //* Obtain ownership lock on data members *
   lock_guard<mutex> guard ( this->data_Lock ) ;

   fProc = this->trgFiles ;
   bProc = this->trgBytes ;
   double pct  = (double)this->trgBytes / (double)this->srcBytes ;
   if ( pct >= 0.995 )     // compensate for rounding errors
      pct = 1.0 ;
   return pct ;
   // As we leave the scope of the lock, the lock is freed.
}

//*************************
//*        update         *
//*************************
//******************************************************************************
//* Update the data completion accumulators.                                   *
//* Targets have been updated, or in the case of backup/synch, updated if      *
//* required (else validated only).                                            *
//*                                                                            *
//* Input  : tFiles : number of files processed since last update              *
//*          tBytes : number of bytes processed since last update              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void WorkInProgress::update ( UINT tFiles, UINT64 tBytes )
{
   //* Obtain ownership lock on data members *
   lock_guard<mutex> guard ( this->data_Lock ) ;

   this->trgFiles += tFiles ;
   if ( this->trgFiles > this->srcFiles ) // prevent overflow
      this->trgFiles = this->srcFiles ;
   this->trgBytes += tBytes ;
   if ( this->trgBytes > this->srcBytes ) // prevent overflow
      this->trgBytes = this->srcBytes ;
   // As we leave the scope of the lock, the lock is freed.
}

//*************************
//*      createPBar       *
//*************************
//******************************************************************************
//* Create and open (display) a Progbar object.                                *
//*                                                                            *
//* Input  : pbi  : definition for initializing the progress bar               *
//*                                                                            *
//* Returns: 'true' if successful, else 'false'                                *
//******************************************************************************

bool WorkInProgress::createPBar ( const pbarInit& pbi )
{
   //* Obtain ownership lock on data members *
   lock_guard<mutex> guard ( this->data_Lock ) ;

   this->reset_pbar() ;       // delete previous instance (if any)
   bool status = false ;      // return value
   this->pBar = new Progbar ( pbi ) ;
   if ( this->pBar != NULL )
      status = this->pBar->open() ;

   return status ;
   // As we leave the scope of the lock, the lock is freed.
}

//*************************
//*      deletePBar       *
//*************************
//******************************************************************************
//* Delete any attached Progbar objects.                                       *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true'  if there was an existing progress bar                     *
//*          'false' if no progress bar had been instantiated                  *
//******************************************************************************

bool WorkInProgress::deletePBar ( void )
{
   //* Obtain ownership lock on data members *
   lock_guard<mutex> guard ( this->data_Lock ) ;

   return ( (this->reset_pbar()) ) ; // call the private method to do the deed
   // As we leave the scope of the lock, the lock is freed.
}

//*************************
//*      pbarUpdate       *
//*************************
//******************************************************************************
//* Advance the progress bar by the specified number of steps.                 *
//*                                                                            *
//* Note that the pBar data members are not under the access lock because only *
//* the secondary thread may call this method.                                 *
//*                                                                            *
//* Input  : inc  : number of step to advance the progress bar                 *
//*                                                                            *
//* Returns: percentage of steps completed (integer value: 0 - 100)            *
//******************************************************************************

short WorkInProgress::pbarUpdate ( short inc )
{
   short pct = ZERO ;
   if ( this->pBar != NULL )
      pct = this->pBar->update ( inc ) ;
   return ( pct ) ;
}

//*************************
//*      pbarRefresh      *
//*************************
//******************************************************************************
//* Refresh displayed progress bar. Because the progress bar is a dialog which *
//* is typically overlays part of another dialog, if the underlying dialog     *
//* is updated, then the progress bar will become invisible. Call this method  *
//* to redraw the progress bar after the underlying dialog display has changed.*
//*                                                                            *
//* Note that the pBar data members are not under the access lock because only *
//* the secondary thread may call this method.                                 *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void WorkInProgress::pbarRefresh ( void )
{
   //* Obtain ownership lock on data members *
   lock_guard<mutex> guard ( this->data_Lock ) ;

   if ( this->pBar != NULL )
      this->pBar->refresh () ;
   // As we leave the scope of the lock, the lock is freed.
}

//*************************
//*     getThreshold      *
//*************************
//******************************************************************************
//* Returns progress bar threshold for creating secondary execution thread.    *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: percentage of steps completed (integer value: 0 - 100)            *
//******************************************************************************

UINT64 WorkInProgress::getThreshold ( void )
{
   //* Obtain ownership lock on data members *
   lock_guard<mutex> guard ( this->data_Lock ) ;

   return this->pbThresh ;
   // As we leave the scope of the lock, the lock is freed.
}

//*************************
//*       setAbort        *
//*************************
//******************************************************************************
//* Set the 'wipAbort' flag to indicate that secondary thread should return    *
//* immediately (if it hasn't already).                                        *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: new state of the 'wipAbort' flag                                  *
//******************************************************************************

bool WorkInProgress::setAbort ( void )
{
   //* Obtain ownership lock on data members *
   lock_guard<mutex> guard ( this->data_Lock ) ;

   this->wipAbort = true ;
   return this->wipAbort ;
   // As we leave the scope of the lock, the lock is freed.
}

//*************************
//*       getAbort        *
//*************************
//******************************************************************************
//* Return the current state of the 'wipAbort' flag indicating to the secondary*
//* thread that user has aborted the operation OR that the main thread has     *
//* completed the operation before the secondary thread realizes it.           *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: current state of the 'wipAbort' flag                              *
//******************************************************************************

bool WorkInProgress::getAbort ( void )
{
   //* Obtain ownership lock on data members *
   lock_guard<mutex> guard ( this->data_Lock ) ;

   return this->wipAbort ;
   // As we leave the scope of the lock, the lock is freed.
}

//*************************
//*         reset         *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* Reset all data members.                                                    *
//* Called either by the constructor OR after a data lock has been obtained.   *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void WorkInProgress::reset ( void )
{
   this->srcFiles = this->trgFiles = ZERO ;
   this->srcBytes = this->trgBytes = ZERO ;
   this->wipAbort = this->wipPause = false ;
   this->pbThresh = PB_THRESHOLD ;
   this->reset_pbar() ;
}

//*************************
//*      reset_pbar       *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* Delete attached Progbar object (if any).                                   *
//* Called either by the constructor OR after a data lock has been obtained.   *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true'  if there was an existing progress bar                     *
//*          'false' if no progress bar had been instantiated                  *
//******************************************************************************

bool WorkInProgress::reset_pbar ( void )
{
   bool status = false ;
   if ( this->pBar != NULL )
   {
      delete this->pBar ;
      this->pBar = NULL ;
      status = true ;
   }
   return status ;
}

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

