//********************************************************************************
//* File       : FileDlgClipbrd.cpp                                              *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2005-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in FileMangler.hpp         *
//* Date       : 02-Apr-2025                                                     *
//* Version    : (see FileDlgVersion string in FileDlg.cpp)                      *
//*                                                                              *
//* Description:                                                                 *
//* This module contains clipboard-related methods for the FileDlg class and     *
//* miscellaneous utility methods.                                               *
//*                                                                              *
//* Developed using GNU G++ (Gcc v: 4.4.2)                                       *
//*  under Fedora Release 12, Kernel Linux 2.6.31.5-127.fc12.i686.PAE and above  *
//********************************************************************************
//* Version History (most recent first):                                         *
//*   See version history in FileDlg.cpp.                                        *
//********************************************************************************
//* Programmer's Notes:                                                          *
//*                                                                              *
//*                                                                              *
//********************************************************************************

#include "GlobalDef.hpp"
#include "FileDlg.hpp"

//******************************
//* Local definitions and data *
//******************************

#define DISPLAY_BASENODE (0)     // For Debugging Only!!
#if DISPLAY_BASENODE != 0
// Works only in single-window mode and 132-column terminal window
winPos dbnUL( 4, 79 ) ;
#endif   // DISPLAY_BASENODE
#define DISPLAY_BN_CLEARCB (0)   // For Debugging Only!!
#define DISPLAY_BN_FILLCB (0)    // For Debugging Only!!

//* Filename formatting field *
const short MAX_FNWIDTH = 19 ;// max filename length (at level zero)
const char fPad[] = { "                   " } ;

//* Message text defined in FileDlgPrompt.cpp *
extern const char* fmtypeString[] ;
extern const char* fmtypeName[] ;
extern const char* readonlyFileSys[] ;
extern const char* notOntoSelf[] ;
extern const char* cannotOverrideWP[] ;
extern const char* noLinkOverwrite[] ;
extern const char* differentFileTypes[] ;
extern const char* cannotDeleteSpecial[] ;
extern const char* cannotOverwriteFifo[] ;
extern const char* cannotOverwriteDirectory[] ;
extern const char* warnSocketMod[] ;
extern const char* warnBdevCdevMod[] ;
extern const char* warnUnsuppMod[] ;
extern const char* fdNotImplemented[] ;



//*************************
//*    ClearClipboard     *
//*************************
//******************************************************************************
//* Release memory allocated to clipboard storage and reset the tracking       *
//* variables.                                                                 *
//*   (See also FillClipboard().)                                              *
//*                                                                            *
//* On first call, the base node TreeNode class object is instantiated and     *
//* initialized to indicate that the clipboard is empty.                       *
//*                                                                            *
//* The approved way to check whether data is on the clipboard is the following*
//*           if ( clipBoard[BNODE].totFiles > ZERO ) { do stuff }             *
//*                                                                            *
//* Only the FileDlg class destructor should call with the optional parameter, *
//* 'freeAll' set to 'true'. This will indicate that the base node TreeNode    *
//* object should also be deleted.                                             *
//*                                                                            *
//* Input  : freeAll: (optional, false by default)                             *
//*                   if 'true', release all memory associated with the        *
//*                   clipboard, including the base node object.               *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note:                                                         *
//*                                                                            *
//* The concept of clearing the clipboard is not only a practical issue, but   *
//* a design decision as well. The clipboard should be cleared under the       *
//* following circumstances.                                                   *
//*                                                                            *
//* 1. If any file in the CB source directory has been changed, the CB must be *
//*    cleared, unless we want to go to the trouble of checking whether the    *
//*    change made in the directory affected one of the files on the clipboard.*
//*    Actually, we DO NOT want to open that can of lumbricus terrestris, so   *
//*    any change to the CB source directory causes the CB to be cleared.      *
//*    Some examples would be a file rename, a file deletion, a file addition  *
//*    (it may overwrite an existing file).                                    *
//*    Any such change to the CB source directory would necessitate a call to  *
//*    this->RefreshCurrDir(), because no change can happen to a directory     *
//*    unless it is in view (except that the undisplayed source directory      *
//*    changes in a single-window cut-and-paste), so that is the logical method*
//*    to capture this condition. A test is made in RefreshCurrDir() to see if *
//*    the directory being re-read is also the CB source, and if it is, CB is  *
//*    cleared before the re-read.                                             *
//*                                                                            *
//*    Note that such changes may be caused by forces external to our          *
//*    application, in which case a subsequent 'paste' command in our          *
//*    application may fail: embarrassing, but not fatal.                      *
//*                                                                            *
//*    (Note that this->RefreshCurrDir() includes a parameter to FORCE the CB) *
//*    (to be cleared. Use it wisely and sparingly. For instance, it is used ) *
//*    (during FileDlg instantiation to initialize the clipboard tracking    ) *
//*    (variables and on return from a user shell-out.                       ) *
//*                                                                            *
//* 2. If the CB source directory's name has changed, OR the name of any parent*
//*    directory in the path of the CB source has changed, then the CB files   *
//*    will become inaccessible. Thus, this->RenameSelectedFiles() clears      *
//*    the clipboard.                                                          *
//*                                                                            *
//*    Note that such changes may be caused by forces external to our          *
//*    application, in which case we may not be aware of it.                   *
//*                                                                            *
//* 3. It may seem obvious, but if the user puts something on the CB through   *
//*    a 'copy' or 'cut' operation, it will displace any previous CB contents. *
//*    This is handled at the top of this->FillClipboard().                    *
//*                                                                            *
//*  4. If user 'selects' a file in the CB source directory, (whether or not   *
//*     selections/copy/cut are visually indicated) it creates an ambiguous    *
//*     condition.                                                             *
//*     The user might believe that he/she is removing one or more files from  *
//*     the clipboard by de-selecting them, but this is not the case. Except   *
//*     for this special case, 'selection' and 'deselection' have no effect    *
//*     on the contents of the clipboard.                                      *
//*     It would be confusing if other files in that view were already set to  *
//*     the 'copy' color or 'cut' color (indicating that they are on the       *
//*     clipboard), and then to add a 'selected' file or subtract a copy/cut   *
//*     file. Although, technically, an additional 'selection' does not affect *
//*     the CB contents, it is in logical conflict with any other 'selected'   *
//*     files in that window. Thus, if an additional 'selection'               *
//*     (or 'deselection') is made in the CB source directory, we clear the    *
//*     clipboard to avoid any ambiguity.                                      *
//*                                                                            *
//* 5. If user 'deselects' any or all files in the CB source, then we assume   *
//*    that it indicates an abandonment of the clipboard contents and we       *
//*    clear the clipboard. (see also item 4 above)                            *
//*                                                                            *
//*                                                                            *
//* Programmer's Note: Be sure clipboard data are deleted BEFORE deleting the  *
//* FMgr-class instance. (See FilDlg destructor.)                              *
//******************************************************************************

void FileDlg::ClearClipboard ( bool freeAll )
{
   //* If clipboard control class has been instantiated,    *
   //* release dynamic memory allocation for files and      *
   //* directories, and reset the tracking variables.       *
   if ( clipBoard != NULL )
   {
      for ( UINT dt = ZERO ; dt < clipBoard_dirTrees ; ++dt )
         this->fmPtr->ReleaseDirTree ( &clipBoard[BNODE].nextLevel[dt] ) ;

      //* Release the TreeNode array.*
      if ( clipBoard[BNODE].nextLevel != NULL )
         this->fmPtr->Release_TreeNodes ( clipBoard[BNODE].nextLevel, clipBoard_dirTrees ) ;
      clipBoard_dirTrees = ZERO ;

      //* Release the non-directory file array.*
      if ( clipBoard[BNODE].tnFiles != NULL )
         this->fmPtr->Release_tnFNames ( clipBoard[BNODE].tnFiles, clipBoard_topFiles ) ;
      clipBoard_topFiles = ZERO ;

      //* Reinitialize the anchor node.*
      clipBoard[BNODE].ReInit () ;
      this->fmPtr->strnCpy ( clipBoard[BNODE].dirStat.fName, "BASENODE", MAX_FNAME ) ;
      clipBoard_srcPath[0] = NULLCHAR ;
      clipBoard_opPending = opNONE ;
      clipBoard_gvfsSrc = false ;
      clipBoard_gvfsTrg = false ;

      //* If FileDlg destructor called us, release the base node *
      if ( freeAll != false )
      {
         this->fmPtr->Release_TreeNodes ( clipBoard, 1 ) ;
      }
      #if DISPLAY_BASENODE != 0 && DISPLAY_BN_CLEARCB != 0
      else
      {
         this->fmPtr->strnCpy ( clipBoard[BNODE].dirStat.fName, 
                   "BASENODE - ClearClipboard", MAX_DIALOG_WIDTH ) ;
         this->Display_TreeNode ( dbnUL, clipBoard ) ;
      }
      #endif   // DISPLAY_BASENODE && DISPLAY_BN_CLEARCB
   }
   //* Else, instantiate the clipBoard base node and initialize its members *
   else
   {
      clipBoard = this->fmPtr->Allocate_TreeNodes ( 1 ) ;
      this->fmPtr->strnCpy ( clipBoard[BNODE].dirStat.fName, "BASENODE", MAX_FNAME ) ;
      clipBoard_srcPath[0] = NULLCHAR ;
      clipBoard_dirTrees = ZERO ;
      clipBoard_topFiles = ZERO ;
      clipBoard_opPending = opNONE ;
      clipBoard_gvfsSrc = false ;
      clipBoard_gvfsTrg = false ;
   }
}  //* End ClearClipboard() *

//***********************
//*   FillClipboard     *
//***********************
//******************************************************************************
//* Copy a list of the selected files to our clipboard.                        *
//*   Note: Caller has verified that there is at least one file in the         *
//*         current directory, and therefore at least one file selected,       *
//*         i.e. the hilighted file if no other.                               *
//*                                                                            *
//* The dynamic memory allocation for the clipboard data must be carefully     *
//* tracked because it is a prime opportunity for memory leaks.                *
//* Allocations are:                                                           *
//* 1. Base TreeNode class object which anchors all other data.                *
//* 2. Array of tnFName class objects, one object for each non-directory file  *
//*    in the top-level directory of the 'selected' directory tree.            *
//* 3. A TreeNode class object for each subdirectory name (not including the   *
//*    directory which CONTAINS the tree data but which will not be copied).   *
//*                                                                            *
//*                                                                            *
//* Input  : member of enum OpPend                                             *
//*            (either opCOPY or opCUT if to be followed by a Paste operation) *
//* Returns: nothing                                                           *
//******************************************************************************

void FileDlg::FillClipboard ( OpPend operation )
{
   //* Discard existing clipboard data, if any *
   this->ClearClipboard () ;

   //* Copy the path string to clipboard *
   this->fmPtr->strnCpy ( clipBoard_srcPath, this->currDir, MAX_PATH ) ;
   clipBoard_opPending  = operation ;             // pending operation

   //* Count the 'selected' directory and non-directory items at the top level *
   UINT  topLevelDirs  = ZERO,
         topLevelFiles = ZERO ;
   for ( UINT i = ZERO ; i < this->fileCount ; i++ )
   {
      if ( this->colorData[i] & this->selectAttr )
      {
         if ( this->deList[i]->fType == fmDIR_TYPE )
            ++topLevelDirs ;
         else
            ++topLevelFiles ;
      }
   }

   //* If source data reside on a virtual (MTP/GVfs) filesystem, indicate that *
   //* the pending operation should use GVfs tools rather than kernel tools.   *
   if ( (topLevelDirs > ZERO) || (topLevelFiles > ZERO) )
      clipBoard_gvfsSrc = this->fmPtr->isGvfsPath () ;

   //* Allocate data space for the data to be placed on the clipboard *
   if ( topLevelDirs > ZERO )       // top-level directory names
   {
      TreeNode cnt ;                // gets sub-tree summary data
      clipBoard[BNODE].nextLevel = this->fmPtr->Allocate_TreeNodes ( topLevelDirs ) ;

      //* Copy the file info to the clipboard. Attach a copy of each sub-tree  *
      //* list to a member of our top-level node array.                        *
      for ( UINT i = ZERO ; i < this->fileCount ; i++ )
      {
         if ( this->colorData[i] & this->selectAttr 
              && this->deList[i]->fType == fmDIR_TYPE )
         {
            this->fmPtr->CopyDirTree ( this->deList[i], 
                          clipBoard[BNODE].nextLevel[clipBoard_dirTrees] ) ;
            //* Update the accumulators *
            this->fmPtr->TreeNodeSummary ( 
                     &clipBoard[BNODE].nextLevel[clipBoard_dirTrees], cnt ) ;
            clipBoard[BNODE] += cnt ;

            // #if DISPLAY_BASENODE != 0 && DISPLAY_BN_FILLCB != 0
            // //* Display top node of this sub-tree *
            // this->Display_TreeNode ( dbnUL, &clipBoard[BNODE].nextLevel[clipBoard_dirTrees] ) ;
            // #endif   // DISPLAY_BASENODE && DISPLAY_BN_FILLCB

            ++clipBoard_dirTrees ;  // top-level nodes (nextLevel[])

            //* Update the _private_ 'nlnodes' member.*
            clipBoard[BNODE].Increment_nlnodes() ;
         }
      }
   }
   if ( topLevelFiles > ZERO )      // top-level non-directory file names
   {
      //* Allocate an array of tnFName class objects to hold the file info *
      clipBoard[BNODE].tnFiles = this->fmPtr->Allocate_tnFNames ( topLevelFiles ) ;
      clipBoard[BNODE].allocBytes += sizeof(tnFName) * topLevelFiles ;

      //* Copy the file info to the clipboard *
      for ( UINT i = ZERO, j = ZERO ; i < this->fileCount ; i++ )
      {
         if ( this->colorData[i] & this->selectAttr 
              && this->deList[i]->fType != fmDIR_TYPE )
         {
            //* Copy the file data *
            clipBoard[BNODE].tnFiles[j] = *this->deList[i] ;
            //* Update the accumulators *
            switch ( clipBoard[BNODE].tnFiles[j].fType )
            {
               case fmREG_TYPE:
                  ++clipBoard[BNODE].regFiles ;
                  clipBoard[BNODE].regBytes += clipBoard[BNODE].tnFiles[j].fBytes ;
                  break ;
               case fmLINK_TYPE:
                  ++clipBoard[BNODE].linkFiles ;
                  clipBoard[BNODE].linkBytes += clipBoard[BNODE].tnFiles[j].fBytes ;
                  break ;
               case fmCHDEV_TYPE:
                  ++clipBoard[BNODE].cdevFiles ;
                  clipBoard[BNODE].cdevBytes += clipBoard[BNODE].tnFiles[j].fBytes ;
                  break ;
               case fmBKDEV_TYPE:
                  ++clipBoard[BNODE].bdevFiles ;
                  clipBoard[BNODE].bdevBytes += clipBoard[BNODE].tnFiles[j].fBytes ;
                  break ;
               case fmFIFO_TYPE:
                  ++clipBoard[BNODE].fifoFiles ;
                  clipBoard[BNODE].fifoBytes += clipBoard[BNODE].tnFiles[j].fBytes ;
                  break ;
               case fmSOCK_TYPE:
                  ++clipBoard[BNODE].sockFiles ;
                  clipBoard[BNODE].sockBytes += clipBoard[BNODE].tnFiles[j].fBytes ;
                  break ;
               case fmUNKNOWN_TYPE:
                  ++clipBoard[BNODE].unkFiles ;
                  clipBoard[BNODE].unkBytes += clipBoard[BNODE].tnFiles[j].fBytes ;
                  break ;
               case fmDIR_TYPE:  // already handled
               case fmTYPES:     // unlikely
               default:          // even less likely
                  break ;
            }
            clipBoard[BNODE].totBytes += clipBoard[BNODE].tnFiles[j].fBytes ;
            ++clipBoard[BNODE].totFiles ;
            ++clipBoard[BNODE].tnFCount ;
            ++j ;

            ++clipBoard_topFiles ;  // top-level non-dir files (tnFiles[])

            //* Update the _private_ 'tnfcount' member.*
            clipBoard[BNODE].Increment_tnfcount() ;
         }
      }
   }

   #if DISPLAY_BASENODE != 0 && DISPLAY_BN_FILLCB != 0
   this->fmPtr->strnCpy ( clipBoard[BNODE].dirStat.fName, 
                          "BASENODE - FillClipboard", MAX_DIALOG_WIDTH ) ;
   this->Display_TreeNode ( dbnUL, clipBoard ) ;
   #endif   // DISPLAY_BASENODE && DISPLAY_BN_FILLCB

}  //* End FillClipboard() *

//***********************
//*    MarkSelection    *
//***********************
//******************************************************************************
//* Determine whether user has selected any files in the current view for      *
//* processing, and visually mark them as selected using the specified color   *
//* attribute. If no files have been selected, the currently-highlighted file  *
//* is the default selection.                                                  *
//*                                                                            *
//* After selection, re-display the data and set highlight on first 'selected' *
//* item.                                                                      *
//*                                                                            *
//* Input  : selColor  : color attribute with which to mark 'selected' files   *
//*          srcdirPerm: (by reference, initial value ignored)                 *
//*            'true' if user has read/write permission on source directory    *
//*            else, 'false'                                                   *
//*                                                                            *
//* Returns: number of files 'selected' for processing                         *
//******************************************************************************
//* Programmer's Note: MTP/GVfs filesystems are scanned non-recursively, so    *
//* either the recursive or non-recursive version of this method will be       *
//* equivalent in the number of selections reported. However, file operations  *
//* which do not use the clipboard (rename, touch) will call the non-recursive *
//* version which also initializes the 'clipBoard_gvfsSrc' flag.               *
//******************************************************************************

UINT FileDlg::MarkSelection ( attr_t selColor, bool& srcdirPerm )
{
   //* Test whether user has read/write permission on parent(current) directory*
   srcdirPerm = this->CurrDirPermission () ;

   //* Be sure there is something in this view that CAN BE selected *
   if ( this->fileCount > ZERO )
   {
      //* Discard current contents of clipboard (if any).                      *
      //* If 'selected' files in this view are associated with the clipboard   *
      //* contents, they will be cleared before making new selections.         *
      if ( (this->TossSelection ()) != false )
         this->DeselectFile ( true ) ;
      this->ClearClipboard () ;

      //* If one or more files previously set as 'selected' *
      if ( this->selInfo.Count > ZERO )
      {
         //* Visually identify files as marked for processing *
         for ( UINT i = ZERO ; i < this->fileCount ; i++ )
         {
            if ( this->colorData[i] & this->selectAttr )
               this->colorData[i] = selColor ;
         }
      }
      //* Else, the hilighted file is selected by implication *
      else
      {
         if ( this->SelectFile ( false, ZERO ) )
            this->colorData[this->hIndex] = selColor ;
      }

      //* Re-display the data *
      this->DisplayStatus () ;
      this->dPtr->RefreshScrollextText ( this->fIndex ) ;
      this->FirstSelectedItem () ;        // track to first 'selected' item
   }
   return (this->selInfo.Count + this->selInfo.subCount) ;

}  //* End MarkSelection() *

//***********************
//*   MarkSelectionNR   *
//***********************
//******************************************************************************
//* Determine whether user has selected any files in the current view for      *
//* processing, and visually mark them as selected using the specified color   *
//* attribute. If no files have been selected, the currently-highlighted file  *
//* is the default selection.                                                  *
//*                                                                            *
//* After selection, re-display the data and set highlight on first 'selected' *
//* item.                                                                      *
//*                                                                            *
//* IMPORTANT NOTE: This is the non-recursive version of MarkSelection().      *
//*    The file count returned is for top-level files and subdirectory names   *
//*    only! The DisplayStatus() method will be forced to report only the      *
//*    top-level count and size. This method should be called ONLY by          *
//*    operations that do not recursively travel the TreeNode sub-trees, but   *
//*    instead operate only on the files which are actually displayed in the   *
//*    file-display control. See RenameSelectedFiles() for an example.         *
//*                                                                            *
//* IMPORTANT NOTE: Because MTP/GVfs virtual filesystems do not support        *
//*    recursive file operations, this method should be called to mark         *
//*    selections for those filesystems so selection totals will be reported   *
//*    accurately. (See also above method.)                                    *
//*                                                                            *
//*                                                                            *
//* Input  : selColor  : color attribute with which to mark 'selected' files   *
//*          srcdirPerm: (by reference, initial value ignored)                 *
//*            'true' if user has read/write permission on source directory    *
//*            else, 'false'                                                   *
//*                                                                            *
//* Returns: number of files 'selected' for processing                         *
//******************************************************************************

UINT FileDlg::MarkSelectionNR ( attr_t selColor, bool& srcdirPerm )
{
   //* Test whether user has read/write permission on parent(current) directory*
   srcdirPerm = this->CurrDirPermission () ;

   //* Be sure there is something in this view that CAN BE selected *
   if ( this->fileCount > ZERO )
   {
      //* Discard current contents of clipboard (if any) *
      this->ClearClipboard () ;

      //* If source data reside on a virtual (MTP/GVfs) filesystem, *
      //* indicate that the pending operation should use GVfs tools *
      //* rather than kernel tools.                                 *
      clipBoard_gvfsSrc = this->fmPtr->isGvfsPath () ;

      //* Remember the recursive file count and total size, *
      //* (will be restored before returning to caller)     *
      UINT     oldSubCount = this->selInfo.subCount ;
      UINT64   oldSubSize  = this->selInfo.subSize ;

      //* If one or more files previously set as 'selected' *
      if ( this->selInfo.Count > ZERO )
      {
         //* Visually identify files as marked for processing *
         for ( UINT i = ZERO ; i < this->fileCount ; i++ )
         {
            if ( this->colorData[i] & this->selectAttr )
               this->colorData[i] = selColor ;
         }
      }
      //* Else, the hilighted file is selected by implication *
      else
      {
         if ( this->SelectFile ( false, ZERO ) )
            this->colorData[this->hIndex] = selColor ;
         oldSubCount = this->selInfo.subCount ;
         oldSubSize  = this->selInfo.subSize ;
      }

      //* Re-display the data *
      this->selInfo.subCount = ZERO ;
      this->selInfo.subSize = ZERO ;
      this->DisplayStatus () ;
      this->selInfo.subCount = oldSubCount ; // restore true status values
      this->selInfo.subSize = oldSubSize ;
      this->dPtr->RefreshScrollextText ( this->fIndex ) ;
      this->FirstSelectedItem () ;        // track to first 'selected' item
   }
   return this->selInfo.Count ;

}  //* End MarkSelectionNR() *

//*************************
//*    Copy2Clipboard     *
//*************************
//******************************************************************************
//* Set color attribute of selected file(s) to indicate that files are in      *
//* the "copy" state.                                                          *
//*                                                                            *
//* Copy the list of selected files to our clipboard using FillClipboard().    *
//*                                                                            *
//* The "copy" state means that selected file(s) have been identified as       *
//* candidates for copying from one directory to another using the copy-and-   *
//* paste sequence.                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: true if successful, false if no files tagged.                     *
//******************************************************************************

bool FileDlg::Copy2Clipboard ( void )
{
bool  result = false ;

   //* Set 'copy' indication either for previously-selected file(s) if any or  *
   //* for currently-highlighted file.                                         *
   bool wPerm ;
   if ( (MarkSelection ( this->copyColor, wPerm )) > ZERO )
   {
      //* Copy filenames of selected files to the clipboard *
      this->FillClipboard ( opCOPY ) ;

      //* Update the status control to indicate captured clipboard contents *
      this->DisplayStatus () ;
      result = true ;               // return success
   }
   return result ;

}  //* End Copy2Clipboard() *

//*************************
//*    Cut2Clipboard      *
//*************************
//******************************************************************************
//* Set color attribute of selected file(s) to indicate that files are in      *
//* the "cut" state.                                                           *
//*                                                                            *
//* Copy the list of selected files to our clipboard using FillClipboard().    *
//*                                                                            *
//* The "cut" state means that selected file(s) have been identified as        *
//* candidates for moving from one directory to another using the cut-and-     *
//* paste sequence.                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: true if successful, false if no files tagged.                     *
//******************************************************************************

bool FileDlg::Cut2Clipboard ( void )
{
bool  result = false ;

   //* Set 'cut' indication either for previously-selected file(s) if any or   *
   //* for currently-highlighted file.                                         *
   bool wPerm ;
   if ( (MarkSelection ( this->cutColor, wPerm )) > ZERO )
   {
      //* Copy filenames of selected files to the clipboard *
      this->FillClipboard ( opCUT ) ;

      //* Update the status control to indicate captured clipboard contents *
      this->DisplayStatus () ;
      result = true ;               // return success
   }
   return result ;

}  //* End Cut2Clipboard() *

//***********************
//*    CutPending       *
//***********************
//******************************************************************************
//* Returns 'true' if files on clipboard were placed there by a                *
//* cut-to-clipboard operation. See enum OpPend.                               *
//*                                                                            *
//* Input  :none                                                               *
//*                                                                            *
//* Returns: true if pending operation is opCUT                                *
//*           else 'false', operation is not a 'cut', or clipboard empty       *
//******************************************************************************

bool FileDlg::CutPending ( void )
{
   return clipBoard_opPending == opCUT ? true : false ;

}  //* End CutPending() *

//****************************
//*     SourcePermission     *
//****************************
//******************************************************************************
//* ** Private Method **                                                       *
//* Test for read and write access for clipboard source files and directories. *
//* Indicate the results by setting/resetting the 'readAcc' and 'writeAcc'     *
//* flags for each file's tnFName object.                                      *
//*                                                                            *
//* Optionally, attempt to enable read-access and/or write-access on all       *
//* protected files before returning to caller.                                *
//*                                                                            *
//* Read Access is required for:                                               *
//*  a) copy   b) move   c) adjusting permission bits                          *
//*                                                                            *
//* Write Access is required for:                                              *
//*  a) move   b) trash    c) delete   d) touch    e) rename                   *
//*                                                                            *
//* NOTE: We do not attempt to override write or read protection UNLESS caller *
//*       explicitly asks us to do it. If caller has instructed us to override *
//*       protection AND if there are protected files in the list, then ask    *
//*       user whether to override the write/read protection. If user grants   *
//*       permission, then enable read and/or write protection if possible.    *
//*       (but see also 'silent' option)                                       *
//*                                                                            *
//* Input  : wpCount    : (by reference, initial value ignored)                *
//*                       receives count of write-protected files              *
//*          rpCount    : (by reference, initial value ignored)                *
//*                       receives count of read-protected files               *
//*          writeEnable: (optional, false by default)                         *
//*                       if 'true' attempt to override write protection on    *
//*                       each write-protected file in the list.               *
//*          readEnable : (optional, false by default)                         *
//*                       if 'true' attempt to override read protection on     *
//*                       each read-protected file in the list.                *
//*          silent     : (optional, false by default)                         *
//*                       if 'true' then do not prompt user for a decision     *
//*                                                                            *
//* Returns: 'true'  if all caller-requested read and write permissions        *
//*                  obtained OR if user has indicated that we should ignore   *
//*                  any remaining protection that could not be removed.       *
//*          'false' if user has aborted the operation                         *
//******************************************************************************
//* Programmer's Note: Yes, it causes us pain to duplicate 15 lines of code    *
//* here, but it is the most straightforward approach.                         *
//******************************************************************************

bool FileDlg::SourcePermission ( UINT& wpCount, UINT& rpCount,
                                 bool writeEnable, bool readEnable, bool silent )
{
   bool permissionGranted = true ;

   UINT weCount, reCount, we, re ;        // counters
   gString basePath( clipBoard_srcPath ), // clipboard base path/filename
           dirPath ;                      // subdirectory path/filename

   //* Test non-directory top-level files *
   this->spEnable ( weCount, reCount, clipBoard[BNODE].tnFiles, 
                    clipBoard[BNODE].tnFCount, basePath ) ;

   //* Test directory tree nodes *
   for ( UINT i = ZERO ; i < clipBoard_dirTrees ; i++ )
   {
      //* Test the node-directory name *
      this->spEnable ( we, re, &clipBoard[BNODE].nextLevel[i].dirStat, 1, basePath ) ;
      weCount += we ;
      reCount += re ;

      //* Test contents of subdirectory tree *
      this->fmPtr->CatPathFname ( dirPath, basePath.ustr(), 
                                  clipBoard[BNODE].nextLevel[i].dirStat.fName ) ;
      this->spEnable ( we, re, &clipBoard[BNODE].nextLevel[i], dirPath ) ;
      weCount += we ;
      reCount += re ;
   }
   //* Calculate number of read/write protected files *
   wpCount = clipBoard[BNODE].wprotFiles = clipBoard[BNODE].totFiles - weCount ;
   rpCount = clipBoard[BNODE].totFiles - reCount ;

   //* If there are protected files, ask user if we *
   //* should read/write enable the protected files *
   if ( ((wpCount > ZERO && writeEnable != false) || 
        (rpCount > ZERO && readEnable != false)) )
   {
      if ( ! silent )
         permissionGranted = 
                  this->spDecision ( wpCount, rpCount, clipBoard_opPending ) ;

      if ( permissionGranted )
      {
         //* Enable non-directory top-level files *
         this->spEnable ( weCount, reCount, clipBoard[BNODE].tnFiles, 
                          clipBoard[BNODE].tnFCount, 
                          basePath, writeEnable, readEnable ) ;

         //* Enable directory tree nodes *
         for ( UINT i = ZERO ; i < clipBoard_dirTrees ; i++ )
         {
            //* Enable the node-directory name *
            this->spEnable ( we, re, &clipBoard[BNODE].nextLevel[i].dirStat, 1, 
                             basePath, writeEnable, readEnable ) ;
            weCount += we ;
            reCount += re ;

            //* Enable contents of subdirectory tree *
            this->fmPtr->CatPathFname ( dirPath, basePath.ustr(), 
                                        clipBoard[BNODE].nextLevel[i].dirStat.fName ) ;
            this->spEnable ( we, re, &clipBoard[BNODE].nextLevel[i], dirPath, true ) ;
            weCount += we ;
            reCount += re ;
         }
         //* Calculate number of read/write protected files *
         wpCount = clipBoard[BNODE].wprotFiles = clipBoard[BNODE].totFiles - weCount ;
         rpCount = clipBoard[BNODE].totFiles - reCount ;
      }
   }
   return permissionGranted ;

}  //* End SourcePermission *

//***********************
//*      spEnable       *
//***********************
//******************************************************************************
//* ** Private Method **                                                       *
//* Count the number of write-enabled and read-enabled files in the specified  *
//* array of non-directory files.                                              *
//* Optionally, attempt write-enable and/or read-enable on protected files.    *
//*                                                                            *
//* Input  : weCount : (by reference, initial value ignored)                   *
//*                    receives count of write-enabled files                   *
//*          reCount : (by reference, initial value ignored)                   *
//*                    receives count of read-enabled files                    *
//*          tnfPtr  : pointer to an array of tnFName objects containing file  *
//*                    data.                                                   *
//*          tnfCount: number of files in the tnFName array                    *
//*          dirPath : base directory path for the files in this list.         *
//*          wEnable : (optional, false by default)                            *
//*                    if 'true' attempt write-enable for each protected file. *
//*          rEnable : (optional, false by default)                            *
//*                    if 'true' attempt read-enable for each protected file.  *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note: For symbolic link source files:                         *
//*   a) if !enable, then return statistics for the link itself                *
//*   b) if enable,  then return statistics for link target                    *
//*                                                                            *
//******************************************************************************

void FileDlg::spEnable ( UINT& weCount, UINT& reCount, tnFName* tnfPtr, UINT tnfCount, 
                         const gString& dirPath, bool wEnable, bool rEnable )
{
   weCount = reCount = ZERO ;       // initialize caller's data
   gString gsPath ;                 // path/filename specification string
   bool    wModify, rModify ;       // true if mods needed

   //* Test write and read permission for each file in the list. *
   for ( UINT i = ZERO ; i < tnfCount ; i++ )
   {
      //* If we are counting only (no modifications) *
      if ( ! wEnable && ! rEnable )
      {
         if ( tnfPtr[i].writeAcc != false )
            ++weCount ;
         if ( tnfPtr[i].readAcc != false )
            ++reCount ;
      }
      else        // write and/or read enable was requested
      {
         wModify = rModify = false ;

         if ( tnfPtr[i].fType != fmLINK_TYPE )
         {
            if ( tnfPtr[i].writeAcc != false )
               ++weCount ;
            else if ( wEnable )
               wModify = true ;
            if ( tnfPtr[i].readAcc != false )
               ++reCount ;
            else if ( rEnable )
               rModify = true ;
         }
         else     // get sym-link-target permissions
         {
            this->fmPtr->CatPathFname ( gsPath, dirPath.ustr(), tnfPtr[i].fName ) ;
            if ( (this->fmPtr->WritePermission ( tnfPtr[i], gsPath )) != false )
               ++weCount ;
            else
               wModify = true ;
            if ( (this->fmPtr->ReadPermission ( tnfPtr[i], gsPath )) != false )
               ++reCount ;
            else
               rModify = true ;
         }

         //* If we need to modify permission bits *
         if ( wModify || rModify )
         {
            this->fmPtr->CatPathFname ( gsPath, dirPath.ustr(), tnfPtr[i].fName ) ;
            if ( wModify != false &&
               ((this->fmPtr->WritePermission ( tnfPtr[i], gsPath, true )) != false) )
               ++weCount ;
            if ( rModify != false &&
               ((this->fmPtr->ReadPermission ( tnfPtr[i], gsPath, true )) != false) )
               ++reCount ;
         }
      }
   }

}  //* End spEnable() *

//***********************
//*      spEnable       *
//***********************
//******************************************************************************
//* ** Private Method **                                                       *
//* Recursively travel the specified TreeNode tree and count the number of     *
//* write-enabled and read-enabled files.                                      *
//* Optionally, attempt write-enable and/or read-enable on protected files.    *
//*                                                                            *
//* Input  : weCount : (by reference, initial value ignored)                   *
//*                    receives count of write-enabled files                   *
//*          reCount : (by reference, initial value ignored)                   *
//*                    receives count of read-enabled files                    *
//*          tnPtr: pointer to top of TreeNode list to be tested               *
//*          dirPath: directory path in which the specified directory data live*
//*          wEnable : (optional, false by default)                            *
//*                    if 'true' attempt write-enable for each protected file. *
//*          rEnable : (optional, false by default)                            *
//*                    if 'true' attempt read-enable for each protected file.  *
//*                                                                            *
//* Returns: number of write-enabled files in the list.                        *
//*           on return tnPtr->wprotFile contains the number of                *
//*           write-PROTECTED files in the node                                *
//******************************************************************************

void FileDlg::spEnable ( UINT& weCount, UINT& reCount, TreeNode* tnPtr,  
                         const gString& dirPath, bool wEnable, bool rEnable )
{
   weCount = reCount = ZERO ;       // initialize caller's data
   tnPtr->wprotFiles = ZERO ;       // write-protected files in this node

   //* Process non-directory files on this level *
   if ( tnPtr->tnFCount > ZERO && tnPtr->tnFiles != NULL )
   {
      this->spEnable ( weCount, reCount, tnPtr->tnFiles, tnPtr->tnFCount, 
                       dirPath, wEnable, rEnable ) ;
      //* Number of write-protected files for this node *
      tnPtr->wprotFiles = tnPtr->tnFCount - weCount ;
   }

   //* If there are subdirectories below this one, process them *
   if ( tnPtr->dirFiles > ZERO && tnPtr->nextLevel != NULL )
   {
      gString sPath ;         // subdirectory path/filename string
      for ( UINT i = ZERO ; i < tnPtr->dirFiles ; i++ )
      {
         this->fmPtr->CatPathFname ( sPath, dirPath.ustr(), 
                                     tnPtr->nextLevel[i].dirStat.fName ) ;
         if ( tnPtr->nextLevel[i].dirStat.writeAcc )
            ++weCount ;    // subdirectory name is write-enabled
         //* Subdirectory name is protected. Attempt write-enable *
         else if ( wEnable != false && 
                   ((this->fmPtr->WritePermission ( tnPtr->nextLevel[i].dirStat, 
                   sPath, true )) != false) )
            ++weCount ;
         else  // increment count of write-protected files in this node
            ++tnPtr->wprotFiles ;

         if ( tnPtr->nextLevel[i].dirStat.readAcc )
            ++reCount ;    // subdirectory name is read-enabled
         //* Subdirectory name is protected. Attempt read-enable *
         else if ( rEnable != false && 
                   ((this->fmPtr->ReadPermission ( tnPtr->nextLevel[i].dirStat, 
                   sPath, true )) != false) )
            ++reCount ;

         //* Test lower-level subdirectory tree contents *
         UINT we, re ;                    // temp counters
         this->spEnable ( we, re, &tnPtr->nextLevel[i], sPath, wEnable, rEnable ) ;
         weCount += we ;
         reCount += re ;
      }
   }

}  //* End spEnable() *

//****************************
//*        spDecision        *
//****************************
//******************************************************************************
//* One or more source files which are to be copied, modified or deleted are   *
//* currently write protected and/or read protected.                           *
//* Ask user whether write/read access should be enabled for the protected     *
//* files, so operation can proceed.                                           *
//*                                                                            *
//* Input  : wpCount  : number of write-protected files in caller's file list  *
//*          rpCount  : number of read-protected files in caller's file list   *
//*          opPending: member of enum OpPend which determines the message to  *
//*                     be displayed                                           *
//*                                                                            *
//* Returns: 'true' if permission to override protection is granted            *
//*                 else, 'false'                                              *
//******************************************************************************

bool FileDlg::spDecision ( UINT wpCount, UINT rpCount, enum OpPend opPending )
{
   const char* const msgs[] = 
   {
      "moved",
      "deleted",
      "renamed",
      "updated",
   } ;
   gString gsMsg[6] ;
   const char* dlgMsg[] = 
   {
      "  ALERT  ",
      " ",
      gsMsg[0].ustr(),
      gsMsg[1].ustr(),
      gsMsg[2].ustr(),
      gsMsg[3].ustr(),
      gsMsg[4].ustr(),
      gsMsg[5].ustr(),
      NULL,
   } ;
   attr_t msgColor[INFODLG_MAX_MSGS] = 
   {
      this->cs.sd | ncbATTR, this->cs.sb, this->cs.sb, this->cs.sb, 
      this->cs.sb, this->cs.sb, this->cs.sb, this->cs.sb
   } ;

   short msgIndex ;
   switch ( opPending )
   {
      case opCUT:    msgIndex = 0 ;    break ;     // moving files
      case opDELETE: msgIndex = 1 ;    break ;     // delete or trash files
      case opRENAME: msgIndex = 2 ;    break ;     // rename files
      case opTOUCH:  msgIndex = 3 ;    break ;     // touch files

      case opNONE:   // this method is not called for these options
      case opPROTECT:
      default:
         msgIndex = 3 ; // blank line
         break ;
   }

   short gsMsgIndex = ZERO ;
   gsMsg[gsMsgIndex++].compose( L"Of the files to be %s,", msgs[msgIndex] ) ;
   if ( wpCount > ZERO )
      gsMsg[gsMsgIndex++].compose( L"  %4u %s write protected.", 
                                   &wpCount, 
                                   (const char*)(wpCount > 1 ? "are" : "is") ) ;
   if ( rpCount > ZERO )
   {
      if ( wpCount > ZERO )
         gsMsg[gsMsgIndex++] = "  and" ;
      gsMsg[gsMsgIndex++].compose( L"  %4u %s read protected.", 
                                   &rpCount,
                                   (const char*)(rpCount > 1 ? "are" : "is") ) ;
   }
   gsMsg[gsMsgIndex++] = " " ;
   gsMsg[gsMsgIndex++] = "  Do you want to remove protections and continue?" ;

   return (this->DecisionDialog ( (const char**)dlgMsg, (const attr_t*)&msgColor )) ;

}  //* End spDecision() *

//****************************
//*   SourceSetPermission    *
//****************************
//******************************************************************************
//* ** Private Method **                                                       *
//* Apply the specified permision mask to all clipboard source files and       *
//* directories.                                                               *
//* Modified: User RWX, Group RWX, Other RWX. Other permission bits unmodified.*
//*                                                                            *
//* NOTE: The order in which subdirectories (and their contents) below the     *
//*       base directory are processed is an important consideration.          *
//*       1) If read access for the subdirectory name is enabled, AND if the   *
//*          operation will not affect read access, then the order of          *
//*          operation is not critical.                                        *
//*       2) If, however, read access for the subdirectory name is initially   *
//*          reset, OR if the operation will affect read access, then we must  *
//*          process the subdirectory name and the subdirectory contents in a  *
//*          specific order.                                                   *
//*          a) If initial read access for a subdirectory name is enabled, BUT *
//*             operation will disable read access,then we must process the    *
//*             subdirectory contents BEFORE processing the subdirectory name. *
//*          b) If initial read access for a subdirectory name is disabled,    *
//*             BUT operation will enable it, then we must process the         *
//*             subdirectory name BEFORE processing the subdirectory contents. *
//*       3) Our implementation avoids tortured code by enabling read access   *
//*          for each subdirectory name in the tree (if necessary AND          *
//*          possible), and THEN performing the requested operation.           *
//*          a) We begin at the bottom of the subdirectory tree and work our   *
//*             way back to the subdirectory name, processing the subdirectory *
//*             name LAST.                                                     *
//*          b) This process is not as straightforward as it sounds because    *
//*             if user does not initially have read access to the subdirectory*
//*             name, then we will not have been able to read that             *
//*             subdirectory's contents AND it is not certain whether the user *
//*             will actually be able to set/reset the permission bits         *
//*             he/she/it has chosen.                                          *
//*             Therefore, we must give careful attention to error checking.   *
//*                                                                            *
//* Input  : newPerms: (by reference)                                          *
//*                    Only usrProt, grpProt and othProt members initialized   *
//*                    These members are used to create the bitmask            *
//*          xwd     : for directory names only: if 'true', then if 'read' bit *
//*                    is set in a group, then also set group's 'execute' bit  *
//*                    NOTE: Because read access for a directory name is       *
//*                          actually Read+Execute, this option allows for     *
//*                          special processing of directory names             *
//*                                                                            *
//* Returns: number of files modified                                          *
//*            If return value==number of files in clipboard list, then ALL    *
//*            files successfully modified.                                    *
//******************************************************************************

UINT FileDlg::SourceSetPermission ( const ExpStats& newPerms, bool xwd )
{
   UINT  modCount = ZERO ;          // return value - number of files updated

   //* Initialize base path and permission bits *
   gString gs( clipBoard_srcPath ) ;
   ExpStats eStats ;
   gs.copy( eStats.filePath, MAX_PATH ) ;
   eStats.usrProt = newPerms.usrProt ;
   eStats.grpProt = newPerms.grpProt ;
   eStats.othProt = newPerms.othProt ;

   //* Process each directory tree node *
   for ( UINT i = ZERO ; i < clipBoard_dirTrees ; i++ )
   {
      //* Set the new permisisons for the subdirectory tree *
      modCount += this->sspSet ( eStats, &clipBoard[BNODE].nextLevel[i], xwd ) ;
   }
   //* Process non-directory top-level files *
   modCount += this->sspSet ( eStats, clipBoard[BNODE].tnFiles, 
                              clipBoard[BNODE].tnFCount, xwd ) ;

   return modCount ;

}  //* End SourceSetPermission *

//****************************
//*          sspSet          *
//****************************
//******************************************************************************
//* ** Private Method **                                                       *
//* Apply the specified permision mask to all clipboard source files and       *
//* directories.                                                               *
//*                                                                            *
//* Input  : eStats  : (by reference)                                          *
//*                    .filePath == name of base directory                     *
//*                    .usrProt  == owner read/write/execute flags             *
//*                    .grpProt  == group read/write/execute flags             *
//*                    .othProt  == other read/write/execute flags             *
//*          tnfPtr  : pointer to array of tnFName objects containing file data*
//*                     (NOTE THAT ON RETURN, SOURCE DATA WILL BE STALE.)      *
//*          tnfCount: number of files in the tnFName array                    *
//*          xwd     : for directory names only: if 'true', then if 'read' bit *
//*                    is set in a group, then also set group's 'execute' bit  *
//*                                                                            *
//* Returns: number of files modified                                          *
//******************************************************************************

UINT FileDlg::sspSet ( const ExpStats& eStats, const tnFName* tnfPtr, 
                       UINT tnfCount, bool xwd )
{
   UINT modCount = ZERO ;
   if ( tnfPtr != NULL && tnfCount > ZERO )
   {
      ExpStats fStats ;
      this->fmPtr->strnCpy ( fStats.filePath, eStats.filePath, MAX_PATH ) ;
      for ( UINT i = ZERO ; i < tnfCount ; i++ )
      {  //* Expand the file stats and overlay the new permisison bits *
         this->fmPtr->ExpandStats ( &tnfPtr[i], fStats ) ;
         fStats.usrProt = eStats.usrProt ;
         fStats.grpProt = eStats.grpProt ;
         fStats.othProt = eStats.othProt ;
         if ( xwd != false && tnfPtr[i].fType == fmDIR_TYPE )
         {
            if ( fStats.usrProt.read == 'r' )   fStats.usrProt.exec = 'x' ;
            if ( fStats.grpProt.read == 'r' )   fStats.grpProt.exec = 'x' ;
            if ( fStats.othProt.read == 'r' )   fStats.othProt.exec = 'x' ;
         }

         //* Update the file stats *
         if ( (this->SetFilePermissions ( fStats )) == OK )
            ++modCount ;
      }
   }
   return modCount ;

}  //* End sspSet *

//****************************
//*          sspSet          *
//****************************
//******************************************************************************
//* ** Private Method **                                                       *
//* Recursively travel the specified TreeNode tree and apply the specified     *
//* permision mask to all subdirectories and files in the tree.                *
//*                                                                            *
//* Input  : eStats  : (by reference)                                          *
//*                    .filePath == name of base directory                     *
//*                    .usrProt  == owner read/write/execute flags             *
//*                    .grpProt  == group read/write/execute flags             *
//*                    .othProt  == other read/write/execute flags             *
//*          tnPtr   : pointer to top of TreeNode list to be processed         *
//*                     (NOTE THAT ON RETURN, SOURCE DATA WILL BE STALE.)      *
//*          xwd     : for directory names only: if 'true', then if 'read' bit *
//*                    is set in a group, then also set group's 'execute' bit  *
//*                                                                            *
//* Returns: number of files modified                                          *
//******************************************************************************

UINT FileDlg::sspSet ( const ExpStats& eStats, TreeNode* tnPtr, bool xwd )
{
   UINT modCount = ZERO ;     // number of files successfully modified
   gString  gsPath ;          // source subdirectory
   this->fmPtr->CatPathFname ( gsPath, eStats.filePath, tnPtr->dirStat.fName ) ;
   ExpStats dStats ;
   dStats.usrProt = eStats.usrProt ;
   dStats.grpProt = eStats.grpProt ;
   dStats.othProt = eStats.othProt ;
   gsPath.copy( dStats.filePath, MAX_PATH ) ;

   //* Obtain read access to base directory of subdirectory tree.*
   if ( tnPtr->dirStat.readAcc == false )
      this->fmPtr->ReadPermission ( tnPtr->dirStat, gsPath, true ) ;

   //* If there are subdirectories below, AND if we have access to them *
   if ( tnPtr->dirStat.readAcc != false && 
        tnPtr->dirFiles > ZERO && tnPtr->nextLevel != NULL )
   {
      //* Process each directory tree node *
      for ( UINT i = ZERO ; i < tnPtr->dirFiles ; i++ )
      {  //* Set permissions for subdirectory and its contents *
         modCount += this->sspSet ( dStats, &tnPtr->nextLevel[i], xwd ) ;
      }
   }

   //* Process non-directory top-level files *
   modCount += this->sspSet ( dStats, tnPtr->tnFiles, 
                              tnPtr->tnFCount, xwd ) ;

   //* Process the base directory name *
   modCount += this->sspSet ( eStats, &tnPtr->dirStat, 1, xwd ) ;

   return modCount ;

}  //* End sspSet *

//*************************
//*   ClipboardFiles      *
//*************************
//******************************************************************************
//* Returns number of files on clipboard.                                      *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: file count                                                        *
//******************************************************************************

UINT FileDlg::ClipboardFiles ( void )
{
   UINT status = ZERO ;

   if ( clipBoard != NULL )
      status = clipBoard[BNODE].totFiles ;

   return status ;

}  //* End ClipboardFiles() *

//*************************
//*    ClipboardSize      *
//*************************
//******************************************************************************
//* Returns total number of bytes represented by files on clipboard.           *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: total size of all files on the clipboard (in bytes)               *
//*          NOTE: value returned is an 'unsigned long long int' (64-bit value)*
//******************************************************************************

UINT64 FileDlg::ClipboardSize ( void )
{
   UINT64 status = ZERO ;

   if ( clipBoard != NULL )
      status = clipBoard[BNODE].totBytes ;

   return status ;

}  //* End ClipboardSize() *

//*************************
//*    ClipboardPath      *
//*************************
//******************************************************************************
//* Returns a copy of the path to base directory where clipboard data live.    *
//*                                                                            *
//* Input  : cbPath : (by reference) receives path to clipboard data           *
//*                                                                            *
//* Returns: OK  if clipboard contains data                                    *
//*          ERR if clipboard is empty (cbPath == empty string)                *
//******************************************************************************

short FileDlg::ClipboardPath ( gString& cbPath )
{
   short status = ERR ;

   cbPath = clipBoard_srcPath ;
   if ( cbPath.gschars() > 1 )
      status = OK ;

   return status ;

}  //* End ClipboardPath() *

//**************************
//*  TargDirEqualsClipDir  *
//**************************
//******************************************************************************
//* Compare the path of the target (currently displayed) directory with the    *
//* directory from which the clipboard was filled. This helps the mainline     *
//* code decide whether the clipboard data are stale (or are about to become   *
//* so).                                                                       *
//*                                                                            *
//* For example, if the name/date/etc. of any file on the clipboard list has   *
//* been (or is about to be) modified, the clipboard data become invalid.      *
//*                                                                            *
//* Input  : treeMember: (optional, true by default)                           *
//*                  if 'true' test whether specified directory is             *
//*                    a node on the clipboard tree list                       *
//*                  if 'false' test only whether specified directory          *
//*                    is the top node in the clipboard tree list              *
//*                                                                            *
//* Returns: 'true' if target directory == base clipboard directory            *
//*            OR target directory is a member node on the clipboard tree      *
//*          'false' if target is not a node on clipboard list or if           *
//*            clipboard is empty.                                             *
//******************************************************************************
//* Programmer's Note: This test is a little complicated because we must test  *
//* the path strings rather than scan the individual TreeNode-class objects    *
//* in the clipboard list.                                                     *
//* 1) If target directory is actually the same as the clipboard data's base   *
//*    directory, then obviously we return 'true' regardless of the state of   *
//*    the treeMember flag.                                                    *
//* 2) If target directory is not a subdirectory or sub-subdirectory below the *
//*    clipboard data's base directory, then obviously we return 'false'       *
//*    regardless of the state of the treeMember flag.                         *
//* 3) If treeMember is 'true', AND if the target directory lives in or below  *
//*    the clipboard data's base directory, then we need to investigate        *
//*    further.                                                                *
//*    a. If the target is a node or sub-node on the clipboard list, then      *
//*       we return 'true'.                                                    *
//*    b. If target IS NOT a node or sub-node on the clipboard list, then      *
//*       we return 'false'.                                                   *
//******************************************************************************

bool FileDlg::TargDirEqualsClipDir ( bool treeMember )
{
bool  equal = false ;

   if ( clipBoard[BNODE].totFiles > ZERO )
   {
      equal = ((bool)((strncmp ( this->currDir, clipBoard_srcPath, MAX_PATH )) == ZERO )) ;

      //* If target directory != clipboard base directory AND if caller wants  *
      //* us to check whether target is a node on the clipboard list           *
      if ( ! equal && treeMember != false )
      {  //* Do the simple test first: determine whether target directory      *
         //* lives below the clipboard base directory. If not, we're done.     *
         short cbLen ;  // length of clipboard pathname string
         for ( cbLen = ZERO ; clipBoard_srcPath[cbLen] != NULLCHAR ; cbLen++ ) ;
         if ( (strncmp ( this->currDir, clipBoard_srcPath, cbLen )) == ZERO )
         {  //* Target directory is a child/grandchild directory of the base   *
            //* clipboard directory. Determine whether target directory is     *
            //* actually a node on the clipboard list.                         *
            //* First, isolate the target directory's parent directory name,   *
            //* i.e. the directory which MAY be a node just below the clipboard*
            //* base directory. Then compare that name with the list of node   *
            //* names just below the CB base directory name.                   *
            gString trgNode ( &this->currDir[cbLen+1] ) ;
            const wchar_t* wp = trgNode.gstr() ;
            for ( short tlen = ZERO ; wp[tlen] != NULLCHAR ; tlen++ )
            {
               if ( wp[tlen] == fSLASH )
               {
                  trgNode.limitChars( tlen ) ;
                  break ;
               }
            }
            const char* up = trgNode.ustr() ;
            for ( UINT i = ZERO ; i < clipBoard_dirTrees ; i++ )
            {
               TreeNode* tnp = &clipBoard[BNODE].nextLevel[i] ;
               if ( !(strncmp ( tnp->dirStat.fName, up, MAX_FNAME )) )
               {
                  equal = true ;
                  break ;
               }
            }
         }
      }
   }
   return equal ;

}  //* End TargDirEqualsClipDir() *

//*************************
//*   FirstDisplayItem    *
//*************************
//******************************************************************************
//* Move the scrolling highlight to the first item in the list.                *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void FileDlg::FirstDisplayItem ( void )
{

   this->hIndex = this->dPtr->MoveScrollextHighlight ( this->fIndex, nckHOME ) ;

}  // End FirstDisplayItem() *

//*************************
//*   NextDisplayItem     *
//*************************
//******************************************************************************
//* Move the scrolling highlight to the next item in the list.                 *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void FileDlg::NextDisplayItem ( void )
{

   if ( this->hIndex < (this->fileCount - 1) )
      this->hIndex = this->dPtr->MoveScrollextHighlight ( this->fIndex, nckDOWN ) ;

}  // End NextDisplayItem() *

//*************************
//*   PrevDisplayItem     *
//*************************
//******************************************************************************
//* Move the scrolling highlight to the previous item in the list.             *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void FileDlg::PrevDisplayItem ( void )
{

   if ( this->hIndex > ZERO )
      this->hIndex = this->dPtr->MoveScrollextHighlight ( this->fIndex, nckUP ) ;

}  // End PrevDisplayItem() *

//*************************
//*   FirstSelectedItem   *
//*************************
//******************************************************************************
//* Locate the first 'selected' item in the display-data. This is determined   *
//* by scanning from the top of the colorData array, looking for the first     *
//* item whose selectAttr bit in the color attribute is set.                   *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: index of highlighted item or ERR if no selected items             *
//******************************************************************************

short FileDlg::FirstSelectedItem ( void )
{
   short trgIndex = ERR ;

   if ( this->selInfo.Count > ZERO )      // if at least one item is 'selected'
   {
      UINT  tIndex = ZERO, 
            maxIndex = this->fileCount - 1 ;
      while ( tIndex < maxIndex )
      {
         if ( this->colorData[tIndex] & this->selectAttr )
            break ;
         ++tIndex ;
      }
      trgIndex = this->hIndex = 
         this->dPtr->MoveScrollextHighlight ( this->fIndex, ZERO, tIndex ) ;
   }

   return trgIndex ;

}  //* End FirstSelectedItem() *

//*************************
//*   NextSelectedItem    *
//*************************
//******************************************************************************
//* Locate the next 'selected' item in the display-data. This is determined    *
//* by scanning from the current+1 position of the colorData array, looking    *
//* for the next item whose selectAttr bit in the color attribute is set.      *
//*       If last 'selected' item is already highlighted, do nothing.          *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: index of highlighted item                                         *
//*          returns ERR if: a) no selected items or                           *
//*                          b) last selected item already highlighted         *
//******************************************************************************

short FileDlg::NextSelectedItem ( void )
{
   short trgIndex = ERR ;

   //* If at least one item is 'selected' AND we are not yet at bottom of list *
   UINT maxIndex = this->fileCount - 1 ;
   if ( (this->selInfo.Count > ZERO) && (this->hIndex < maxIndex) )
   {
      UINT  tIndex = this->hIndex + 1 ;   // index item-after-current-item
      while ( tIndex <= maxIndex )
      {
         if ( (this->colorData[tIndex] & this->selectAttr) != ZERO )
         {
            trgIndex = this->hIndex = 
               this->dPtr->MoveScrollextHighlight ( this->fIndex, ZERO, tIndex ) ;
            break ;
         }
         ++tIndex ;
      }
   }

   return trgIndex ;

}  //* End NextSelectedItem() *

//*************************
//*   HilightItemByName   *
//*************************
//******************************************************************************
//* Locate the display line for the specified file and highlight that line.    *
//* This is done by comparing filenames.                                       *
//*                                                                            *
//* Note that the display data are not necessarily in filename order, so we    *
//* need to check the entire list until a match is found.                      *
//*                                                                            *
//* Input  : fName: filename for which to search                               *
//*                                                                            *
//* Returns: index of highlighted item                                         *
//******************************************************************************

short FileDlg::HilightItemByName ( const char* fName )
{
   short trgIndex = ERR, indx ;

   if ( (this->Scroll2MatchingFile ( fName, indx, true )) != false )
      trgIndex = indx ;

   return trgIndex ;

}  //* End HilightItemByName() *

short FileDlg::HilightItemByName ( const gString& fName )
{

   return ( (this->HilightItemByName ( fName.ustr() )) ) ;

}  //* End HilightItemByName() *

//**************************
//*   CurrDirPermission    *
//**************************
//******************************************************************************
//* Test read/write permission on the current directory.                       *
//* Note: for directory files, the 'execute' flag must also be set for full    *
//* read access.                                                               *
//*                                                                            *
//* Call before writing files to target directory.                             *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: true if user has read AND write permission on current directory   *
//*          else, false                                                       *
//******************************************************************************

bool FileDlg::CurrDirPermission ( void )
{
   gString gscd( this->currDir ) ;  // filespec of current directory
   tnFName fStats ;                 // file stats
   bool    hasPerm = false ;        // return value

   if ( (this->fmPtr->GetFileStats ( fStats, gscd )) == OK )
   {
      if ( fStats.writeAcc && fStats.readAcc && (fStats.rawStats.st_mode & S_IXUSR) )
         hasPerm = true ;
   }
   return hasPerm ;

}  //* End CurrDirPermission() *

//***********************
//*    TargetExists     *
//***********************
//******************************************************************************
//* Determine whether specified target file exists, and if it does, return its *
//* vital statistics.                                                          *
//*                                                                            *
//*                                                                            *
//* Input  : fPath : full path/filename specification of target file           *
//*          fStats: (by reference, initial values ignored)                    *
//*                  if target exists, its stat data is returned               *
//*                    INCLUDING fStats.writeAcc indicating whether user has   *
//*                    write access to the file.                               *
//*                  if target does not exist, all members undefined           *
//*                                                                            *
//* Returns: 'true' if target file exists, else 'false'                        *
//******************************************************************************

bool FileDlg::TargetExists ( const char* fPath, tnFName& fStats )
{
   gString gsPath( fPath ) ;        // filespec of current directory
   bool  targExists = false ;

   //* If 'stat' is successful, then file exists *
   if ( (this->fmPtr->GetFileStats ( fStats, gsPath )) == OK )
      targExists = true ;
   return targExists ;

}  //* End TargetExists() *

//*************************
//*     TargetExists      *
//*************************
//******************************************************************************
//* Determine whether specified target file exists.                            *
//* For operations that need info on the file type of the target.              *
//*                                                                            *
//* Input  : fPath   : full path/filename specification                        *
//*          fType   : if target exists, initialized to target fileType        *
//*                    else fmUNKNOWN_TYPE                                     *
//*          wPerm   : initialized to 'true' if user has write                 *
//*                    permission on file or if file doesn't exist             *
//*                    else, false                                             *
//*                                                                            *
//* Returns: true if target file exists, else false                            *
//******************************************************************************

bool FileDlg::TargetExists ( const char* fPath, fmFType& fType, bool& wPerm )
{
   tnFName fStats ;           // file stats
   bool fileExists = false ;  // return value

   wPerm = true ;       // initialize caller's parms (assume file does not exist)
   fType = fmUNKNOWN_TYPE ;

   if ( (fileExists = this->TargetExists ( fPath, fStats )) )
   {
      fType = fStats.fType ;        // target file type
      wPerm = fStats.writeAcc ;     // user write access
   }
   return fileExists ;

}  //* End TargetExists() *

//*************************
//*     TargetExists      *
//*************************
//******************************************************************************
//* Determine whether specified target file exists.                            *
//* If target exists, determine whether user has write permission for file.    *
//*                                                                            *
//*                                                                            *
//* Input  : fPath   : full path/filename specification                        *
//*          wPerm   : initialized to 'true' if user has write                 *
//*                    permission on file or if file doesn't exist             *
//*                                                                            *
//* Returns: true if target file exists, else false                            *
//******************************************************************************
bool FileDlg::TargetExists ( const char* fPath, bool& wPerm )
{
   tnFName fStats ;           // file stats
   bool fileExists = false ;  // return value

   wPerm = true ;       // initialize caller's flags (assume file does not exist)

   if ( (fileExists = this->TargetExists ( fPath, fStats )) )
   {
      wPerm = fStats.writeAcc ;     // user write access
   }
   return fileExists ;

}  //* End TargetExists() *

//***********************
//*   TossSelection     *
//***********************
//******************************************************************************
//* This method solves anomalous situation where user wants to copy or cut to  *
//* clipboard, but files in the current directory are both 'selected' AND      *
//* already on the clipboard.  This method tests for that condition.           *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if condition described exists, else 'false'                *
//******************************************************************************

bool FileDlg::TossSelection ( void )
{
   bool tossIt = false ;

   if ( (this->TargDirEqualsClipDir ( false )) != false )
   {
      for ( UINT i = ZERO ; i < this->fileCount ; i++ )
      {
         if (   (this->colorData[i] == this->copyColor) 
             || (this->colorData[i] == this->cutColor) )
         {
            tossIt = true ;
            break ;
         }
      }
   }
   return tossIt ;

}  //* End TossSelection() *

//******************************************************************************
//*** This group of methods implements the 'browse-clipboard functionality.  ***
//******************************************************************************
//* Programmer's Note: Because the dispText and dispColor members of the       *
//* ssetData class are are pointers to 'const', we play a trick to initialize  *
//* the data arrays. The 'const' keyword is used for these data members to     *
//* indicate that the data are not changed within the NcDialog library, but    *
//* it is our data. In any case, sorry about that.                             *
//******************************************************************************

//*************************
//*    BrowseClipboard    *
//*************************
//******************************************************************************
//* Useful mostly for debugging, this function opens a dialog window and       *
//* displays the contents of the clipboard.                                    *
//*                                                                            *
//* Note: If possible, we fit the browsing window within the dctSCROLLEXT      *
//*       control's borders for a pleasant view; however, we must have a       *
//*       certain mininimum dialog width. If necessary, we expand outside the  *
//*       file-display control.                                                *
//*                                                                            *
//* Input Values : none                                                        *
//*                                                                            *
//* Return Value : 'true' if clipboard cleared and we need to redisplay        *
//*                   else 'false'                                             *
//******************************************************************************

bool FileDlg::BrowseClipboard ( void )
{
const char*  Header01 = " TYPE  FILENAME " ;
const char*  Instructions01 = "(Tab to list, then scroll keys to view)" ;
const short  MIN_CLIPVIEW_WIDTH = 76 ;
enum bcControls : short { closePB = ZERO, clearPB, scrollSE, browsePB, controlsDEFINED } ;
const short  msgLine = 9 ;
short        ulY = this->fulY,
             ulX = this->fulX,
             dialogLines = this->dRows - ulY,
             dialogCols  = (this->fMaxX >= MIN_CLIPVIEW_WIDTH ? 
                            this->fMaxX : MIN_CLIPVIEW_WIDTH) ;
attr_t       tColor = this->cs.bb & ~ncrATTR ;  // status data text color
bool         redisplay = false ;

   //* Re-position the dialog if necessary *
   while ( (ulX > this->dulX) && ((ulX + dialogCols) > (this->dulX + this->dCols)) )
      --ulX ;

InitCtrl    ic[controlsDEFINED] = 
{
   {  //* 'CLOSE' pushbutton  - - - - - - - - - - - - - - - - - - - -  closePB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      8,                            // ulY:       upper left corner in Y
      short(dialogCols / 2 - 4),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      7,                            // cols:      control columns
      " Close ",                    // dispText:
      this->cs.pn & ~ncrATTR,       // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[clearPB],                 // nextCtrl:  link in next structure
   },
   {  //* 'CLEAR' pushbutton  - - - - - - - - - - - - - - - - - - - -  clearPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      8,                            // ulY:       upper left corner in Y
      short(dialogCols -20),        // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      17,                           // cols:      control columns
      " Clear Clipboard ",          // dispText:  
      this->cs.pn & ~ncrATTR,       // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[scrollSE],                // nextCtrl:  link in next structure
   },
   {  //* 'SCROLL' Scroll Ext control - - - - - - - - - - - - - - -   scrollSE *
      dctSCROLLEXT,                 // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      10,                           // ulY:       upper left corner in Y
      ZERO,                         // ulX:       upper left corner in X
      short(dialogLines - 10),      // lines:     control lines
      dialogCols,                   // cols:      control columns
      NULL,                         // dispText:  (n/a)
      this->cs.bb,                  // nColor:    non-focus border color
      this->cs.bb,                  // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "  Files On Clipboard  ",     // label:     
      ZERO,                         // labY:      offset from control's ulY
      ZERO,                         // labX       offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a) 
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[browsePB],                // nextCtrl:  link in next structure
   },
   {  //* 'BrowseSDT' pushbutton  - - - - - - - - - - - - - - - - - - browsePB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      8,                            // ulY:       upper left corner in Y
      3,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      19,                           // cols:      control columns
      "   Detailed View   ",        // dispText:  
      this->cs.pn & ~ncrATTR,       // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   },
} ;

   //* If clipboard is empty, do not instantiate the scrolling control *
   if ( clipBoard[BNODE].totFiles == ZERO )
   {
      ic[clearPB].active = false ;  // nothing to clear, so deactivate control
      ic[clearPB].nextCtrl = NULL ; // last control
   }

   //* Save the display for the parent dialog window *
   this->dPtr->SetDialogObscured () ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogLines,    // number of display lines
                       dialogCols,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       this->cs.bb,    // border color attribute
                       this->cs.sd,    // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   if ( (dp->OpenWindow()) == OK )
   {
      //* Give dialog a title *
      dp->SetDialogTitle ( "  Browse Clipboard  ", this->cs.sd ) ;

      //* Make a visual connection between the *
      //* dialog and the parent application.   *
      dp->WriteChar ( 0, 0, wcsLTEE, this->cs.bb ) ;
      dp->WriteChar ( 0, dialogCols - 1, wcsRTEE, this->cs.bb ) ;

      //* Make a visual connection for the dctSCROLLEXT control *
      //* that overlaps the dialog window's border.             *
      //* (if control not instantiated, this call does nothing) *
      cdConnect cdConn ;
      cdConn.ul2Left = true ;
      cdConn.ur2Right = true ;
      cdConn.connection = true ;
      dp->ConnectControl2Border ( scrollSE, cdConn ) ;

      //* Local copies of key values for storage allocation and data display   *
      //* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*
      //* Number of display elements needed for detailed data display          *
      UINT     dAlloc  = clipBoard[BNODE].totFiles ;
      // Items in top-level (source) dir (count for top-level data display)    *
      UINT     tlFiles = clipBoard[BNODE].tnFCount + clipBoard_dirTrees ;
      //* File bytes represented on clipboard (total)                          *
      UINT64   fBytes  = clipBoard[BNODE].totBytes ;
      //* Bytes allocated to hold clipboard data                               *
      ULONG    aBytes  = clipBoard[BNODE].allocBytes ;
      //* Type of clipboard operation in progress                              *
      OpPend   oPend   = clipBoard_opPending ;
      winPos   wPos ;            // Y/X display positioning
      gString  pBuff,            // for creating display strings
               iBuff ;           // for integer formatting

      //* Is there something on the clipboard ? *
      if ( clipBoard[BNODE].totFiles == ZERO )
      {
         wPos = dp->WriteString ( 2, dialogCols/2-16, " Clipboard Is Currently Empty ", tColor ) ;
      }
      else
      {
         //* Trim (if necessary) and display the source path for data *
         wPos = dp->WriteString ( 2, 2, "Source Dir: ", this->cs.sd ) ;
         gString  pgs( clipBoard_srcPath ) ;
         this->TrimPathString ( pgs, (dialogCols - 16) ) ;
         wPos = dp->WriteString ( wPos.ypos, wPos.xpos, pgs, tColor ) ;
      }

      //* Display the static data *
      wPos.ypos += 2 ;
      wPos.xpos = 2 ;
      wPos = dp->WriteString ( wPos.ypos, wPos.xpos, "Files on Clipboard: ", this->cs.sd ) ;
      iBuff.formatInt( ((ULONG)dAlloc), 7 ) ;
      pBuff.compose( "%S ", iBuff.gstr() ) ;
      wPos = dp->WriteString ( wPos.ypos, wPos.xpos, pBuff, tColor ) ;
      wPos = dp->WriteString ( wPos.ypos, wPos.xpos, "  Byte Total, All Files: ", this->cs.sd ) ;
      iBuff.formatInt( fBytes, 9 ) ;
      pBuff.compose( " %S ", iBuff.gstr() ) ;
      wPos = dp->WriteString ( wPos.ypos, wPos.xpos, pBuff, tColor ) ;
      wPos.ypos += 2 ;
      wPos.xpos = 2 ;
      wPos = dp->WriteString ( wPos.ypos, wPos.xpos, 
                  "Clipboard Storage Bytes Allocated: ", this->cs.sd ) ;
      iBuff.formatInt( aBytes, 7 ) ;
      pBuff.compose( " %S ", iBuff.gstr() ) ;
      wPos = dp->WriteString ( wPos.ypos, wPos.xpos, pBuff, tColor ) ;
      if ( tlFiles > ZERO )
      {
         wPos = dp->WriteString ( wPos.ypos, wPos.xpos, "  Pending Operation: ", this->cs.sd ) ;
         pBuff.compose( " %s ", oPend == opCUT ? "Move" : "Copy" ) ;
         wPos = dp->WriteString ( wPos.ypos, wPos.xpos, pBuff, tColor ) ;
         wPos = { msgLine, 1 } ;
         wPos = dp->WriteString ( wPos.ypos, wPos.xpos, Header01, this->cs.sd | ncbATTR ) ;
         wPos.xpos = dialogCols / 2 - 20 ;
         wPos = dp->WriteString ( wPos.ypos, wPos.xpos, Instructions01, this->cs.sd ) ;
      }

      //* If there is file data to display 'Just Do It' (tm Nike) *
      char*       blkPtr = NULL ;
      ssetData    sData( NULL, NULL, tlFiles, ZERO, false ) ;
      if ( tlFiles > ZERO )
      {
         //* Space for display strings (there is extra space for each string) *
         for ( short i = 3 ; i > ZERO ; --i )
         {
            if ( (blkPtr = new (std::nothrow) char[dAlloc * dialogCols]) != NULL )
               break ;
         }
         //* Space for pointers to strings *
         for ( short i = 3 ; i > ZERO ; --i )
         {
            if ( (sData.dispText  = new (std::nothrow) const char*[dAlloc]) != NULL )
               break ;
         }
         //* Space for color attributes *
         for ( short i = 3 ; i > ZERO ; --i )
         {
            if ( (sData.dispColor = new (std::nothrow) attr_t[dAlloc]) != NULL )
               break ;
         }
         //* Create display strings *
         if ( (blkPtr != NULL) && (sData.dispText != NULL) && (sData.dispColor != NULL) )
            this->bcbStandardView ( blkPtr, sData ) ;
         else  //* Memory-allocation error.                                   *
         {     //* This is unlikely, but if it happens, we need to alert user.*
            if ( blkPtr == NULL )
               blkPtr = new char[gsDFLTBYTES] ;
            if ( sData.dispText == NULL )
               sData.dispText = new const char*[1] ;
            if ( sData.dispColor == NULL )
               sData.dispColor = new attr_t[1] ;
            gString gs( "** Memory Allocation Error! **" ) ;
            gs.copy( blkPtr, gsDFLTBYTES ) ;
            sData.dispText[ZERO] = blkPtr ;
            ((attr_t*)sData.dispColor)[ZERO] = nc.reR ;
            sData.dispItems = 1 ;
         }

         //* Send the data to the scrolling control *
         dp->SetScrollextText ( scrollSE, sData ) ;
      }  // if(tlFiles>ZERO)

      #if DISPLAY_BASENODE != 0     // DEBUG ONLY
      this->fmPtr->strnCpy ( clipBoard[BNODE].dirStat.fName, "BASENODE", MAX_FNAME ) ;
      NcDialog* dnPtr = this->Display_TreeNode ( dbnUL, clipBoard, true ) ;
      #endif   // DISPLAY_BASENODE

      //* Make everything visible *
      dp->RefreshWin () ;

      uiInfo   Info ;               // user interface data returned here
      short    icIndex = ZERO ;     // index of control with input focus
      bool     done = false ;       // loop control
      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( icIndex == closePB )
                  done = true ;
               else if ( icIndex == clearPB )
               {
                  //* Clear the clipboard.                                     *
                  //* Important Note: If files are 'selected', caller must     *
                  //* de-select them and redraw the controls' display data to  *
                  //* show that there are no longer selected files in any view.*
                  this->ClearClipboard () ;
                  done = redisplay = true ;
               }
               else if ( icIndex == browsePB )
               {
                  //* Display detailed view of clipboard contents.             *
                  //* Call and return with focus on closePB.                   *
                  //* 'redisplay' == true if clipboard was cleared and caller  *
                  //* needs to do stuff and call BrowseClipboard() again.      *
                  icIndex = dp->NextControl () ;
                  redisplay = this->BrowseCbDetail ( dp, blkPtr, sData ) ;
                  icIndex = dp->PrevControl () ;
                  if ( redisplay == false )
                  {  //* Restore our own data *
                     wPos = { msgLine, 1 } ;
                     wPos = dp->WriteString ( wPos.ypos, wPos.xpos, Header01, 
                                              this->cs.sd | ncbATTR ) ;
                     wPos.xpos = dialogCols / 2 - 20 ;
                     wPos = dp->WriteString ( wPos.ypos, wPos.xpos, Instructions01, 
                                              this->cs.sd, true ) ;
                     this->bcbStandardView ( blkPtr, sData ) ;
                     dp->SetScrollextText ( scrollSE, sData ) ;
                  }
                  else
                     done = true ;
               }
            }
         }
         else if ( ic[icIndex].type == dctSCROLLEXT )
         {
            dp->RefreshScrollextText ( icIndex ) ;  // make highlight visible 
            icIndex = dp->EditScrollext ( Info ) ;
            dp->RefreshScrollextText ( icIndex, false ) ;  // make highlight invisible 
         }
         //* Move focus to appropriate control *
         if ( Info.viaHotkey == false && done == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }     // while(!done)

      //* Release our dynamic memory allocation *
      if ( tlFiles > ZERO )
      {
         if ( blkPtr != NULL )          { delete [] blkPtr ; blkPtr = NULL ; }
         if ( sData.dispText != NULL )  { delete [] sData.dispText ; sData.dispText = NULL ; }
         if ( sData.dispColor != NULL ) { delete [] sData.dispColor ; sData.dispColor = NULL ; }
      }
      #if DISPLAY_BASENODE != 0     // DEBUG ONLY
      if ( dnPtr != NULL )
         delete dnPtr ;
      #endif   // DISPLAY_BASENODE
   }
   if ( dp != NULL )
      delete ( dp ) ;                        // close the window

   //* Restore display of parent dialog window *
   this->dPtr->RefreshWin () ;

   return redisplay ;

}  //* End BrowseClipboard() *

//*************************
//*    BrowseCbDetail     *
//*************************
//******************************************************************************
//* Display a detailed tree view of clipboard contents.                        *
//* This method is called from inside the BrowseClipboard() method and uses    *
//* that method's resources to display the data. For convenience, certain      *
//* assumptions are made about the way the BrowseClipboard method is written.  *
//* For instance, it is assumed that this method cannot be called if the       *
//* clipboard is empty, the width of the display window is a certain minimum,  *
//* and controls and static data are in predictable positions.                 *
//* If changes happen in BrowseClipboard(), then be sure to update this method.*
//*                                                                            *
//* Input  : dp    : NcDialog pointer to open dialog window                    *
//*          blkPtr    : area for storing display strings                      *
//*          sData     : arrays of text pointers and color attributes          *
//*                      (by reference)                                        *
//*                                                                            *
//* Return Value : 'true' if clipboard cleared and we need to redisplay        *
//*                   else 'false'                                             *
//******************************************************************************

bool FileDlg::BrowseCbDetail ( NcDialog* dp, char* blkPtr, ssetData& sData )
{
enum bcControls : short { closePB = ZERO, clearPB, scrollSE, browsePB, controlsDEFINED } ;
const short    msgLine = 9 ;
short          icIndex = ZERO ;
bool           redisplay = false ;

   //* Draw a header line for our data *
   dp->ClearLine ( msgLine, false ) ;
   gString hdrBuff ;
   hdrBuff.compose( " FILENAME     Files: %s %s%s %s %s %s %s %s  Wprot AccViol",
      fmtypeString[0], fmtypeString[1], fmtypeString[2], fmtypeString[3], 
      fmtypeString[4], fmtypeString[5], fmtypeString[6], fmtypeString[7] ) ; 
   dp->WriteString ( msgLine, 1, hdrBuff, (this->cs.sd | ncbATTR), true ) ;

   //* Create and display the data *
   this->bcbDetailedView ( blkPtr, sData ) ;
   dp->SetScrollextText ( scrollSE, sData ) ;

   uiInfo   Info ;                     // user interface data returned here
   bool     done = false ;             // loop control
   while ( ! done )
   {
      if ( icIndex == closePB || icIndex == clearPB || icIndex == browsePB )
      {
         if ( Info.viaHotkey )
            Info.HotData2Primary () ;
         else
            icIndex = dp->EditPushbutton ( Info ) ;
         if ( Info.dataMod != false )
         {
            if ( icIndex == closePB )
            {
               done = true ;
            }
            else if ( icIndex == clearPB )
            {
               //* Clear the clipboard *
               this->ClearClipboard () ;
               done = redisplay = true ;
            }
            else if ( icIndex == browsePB )
               { /* do nothing, we're already here! */ }
         }
      }
      else if ( icIndex == scrollSE )
      {
         dp->RefreshScrollextText ( icIndex ) ;  // make highlight visible 
         icIndex = dp->EditScrollext ( Info ) ;
         dp->RefreshScrollextText ( icIndex, false ) ;  // make highlight invisible 
      }
      //* Move focus to appropriate control *
      if ( Info.viaHotkey == false && done == false )
      {
         if ( Info.keyIn == nckSTAB )
            icIndex = dp->PrevControl () ; 
         else
            icIndex = dp->NextControl () ;
      }
   }     // while(!done)
   dp->ClearLine ( msgLine, false ) ;  // remove our header line
   return redisplay ;

}  //* End BrowseCbDetail() *

//***********************
//*   bcbStandardView   *
//***********************
//******************************************************************************
//* Build display data for BrowseClipboard() - standard view.                  *
//*                                                                            *
//*                                                                            *
//* Input  : blkPtr    : area for storing display strings                      *
//*          sData     : arrays of text pointers and color attributes          *
//*                      (by reference)                                        *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void FileDlg::bcbStandardView ( char* blkPtr, ssetData& sData )
{
   char*    strPtr = blkPtr ;
   UINT     subdirFiles ;
   UINT64   subdirBytes ;
   gString  sBuff, iBuff ;
   attr_t*  dcPtr = (attr_t*)sData.dispColor ;  // see note in module header

   sData.dispItems = ZERO ;      // reset display-item counter

   //* Initialize the data - top-level directory names first *
   for ( UINT i = ZERO ; i < clipBoard_dirTrees ; i++ )
   {
      if ( clipBoard[BNODE].nextLevel[i].totFiles > ZERO )
      {
         //* Count the files in this subdirectory tree                   *
         //* Note: The count includes directory name being displayed, so *
         //*       we subtract one.                                      *
         subdirFiles = ZERO ;
         subdirBytes = ZERO ;
         this->fmPtr->TreeNodeSummary ( &clipBoard[BNODE].nextLevel[i], 
                                         subdirFiles, subdirBytes ) ;
         --subdirFiles ;
         iBuff.formatInt( (ULONG)subdirFiles, 7, true ) ;
         sBuff.compose( L" %s  %-47s (%S files)", 
                        fmtypeString[fmDIR_TYPE], 
                        clipBoard[BNODE].nextLevel[i].dirStat.fName, 
                        iBuff.gstr() ) ;
      }
      else  // subdirectory is empty
      {
         sBuff.compose( L" %s  %-47s (empty)", 
                        fmtypeString[fmDIR_TYPE], 
                        clipBoard[BNODE].nextLevel[i].dirStat.fName ) ;
      }
      sBuff.copy( strPtr, gsDFLTBYTES ) ;
      sData.dispText[sData.dispItems] = strPtr ;
      strPtr += sBuff.utfbytes() ;
      dcPtr[sData.dispItems] = ftColor[fmDIR_TYPE] ;
      ++sData.dispItems ;
   }
   //* Now, build display data for non-directory files *
   for ( UINT i = ZERO ; i < clipBoard[BNODE].tnFCount ; i++ )
   {
      //* Build the display string, save pointer to display string, and  *
      //* advance to next free space. Initialize color attribute.        *
      sBuff.compose( L" %s  %-47s ", 
                     fmtypeString[clipBoard[BNODE].tnFiles[i].fType], 
                     clipBoard[BNODE].tnFiles[i].fName ) ;
      sBuff.copy( strPtr, gsDFLTBYTES ) ;
      sData.dispText[sData.dispItems] = strPtr ;
      strPtr += sBuff.utfbytes() ;
      dcPtr[sData.dispItems] = ftColor[clipBoard[BNODE].tnFiles[i].fType] ;
      ++sData.dispItems ;
   }

}  //* End bcbStandardView() *

//***********************
//*   bcbDetailedView   *
//***********************
//******************************************************************************
//* Build display data for BrowseClipboard() - detailed view.                  *
//*                                                                            *
//* Note: This calls bsdtl_List() which recursively scans each node list to    *
//*       build display data.                                                  *
//*                                                                            *
//* Input  : blkPtr    : area for storing display strings                      *
//*          sData     : arrays of text pointers and color attributes          *
//*                      (by reference)                                        *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void FileDlg::bcbDetailedView ( char* blkPtr, ssetData& sData )
{
   char*    strPtr = blkPtr ;    // pointer to string storage
   gString  fName ;              // for sizing the filename sub-string
   tnFName* fptr ;               // pointer to file stats
   attr_t*  dcPtr = (attr_t*)sData.dispColor ;  // see note in module header

   sData.dispItems = ZERO ;      // reset display-item counter

   //* Create display data for each subdirectory tree on the clipboard         *
   for ( UINT i = ZERO ; i < clipBoard_dirTrees ; i++ )
   {
      this->bcbDetailedLevel ( clipBoard[BNODE].nextLevel[i], strPtr, sData ) ;
   }

   //* Now, build display data for non-directory files at the top level *
   for ( UINT f = ZERO ; f < clipBoard[BNODE].tnFCount ; f++ )
   {
      //* Build the display string, save pointer to display string, and  *
      //* advance to next free space. Initialize color attribute.        *
      fptr = &clipBoard[BNODE].tnFiles[f] ;
      fName.compose( L" %s%s", fptr->fName, fPad ) ;
      fName.limitCols( MAX_FNWIDTH ) ;
      fName.copy( strPtr, gsDFLTBYTES ) ;
      sData.dispText[sData.dispItems] = strPtr ;
      strPtr += fName.utfbytes() ;
      dcPtr[sData.dispItems] = ftColor[fptr->fType] ;
      ++sData.dispItems ;
   }

}  //* End bcbDetailedView() *

//***********************
//*  bcbDetailedLevel   *
//***********************
//******************************************************************************
//* Scan the specified level of a TreeNode list, and recursively scan all      *
//* lower levels. Create display strings for BrowseCbDetail() method.          *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  : tnPtr : TreeNode object (by reference)                            *
//*          strPtr: pointer to next free space in a character array           *
//*                  used for storing the display strings                      *
//*          sData : ssetData object (by reference)                            *
//*                   sData.dispText is an array of pointers to display strings*
//*                   sData.dispColor is an array of color attributes for the  *
//*                    display strings                                         *
//*                   sData.dispItems is the number of display items already   *
//*                    initialized AND the index of the next free space in the *
//*                    data arrays                                             *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void FileDlg::bcbDetailedLevel ( TreeNode& tnB, char*& strPtr, ssetData& sData )
{
   const short iLEN = 5 ;        // integer string length
   gString  fName,               // for sizing the filename sub-string
            nOffs ;
   char  df[5], rf[5], lf[5], cf[5], bf[5], ff[5], sf[5], uf[5], wf[5], vf[5] ;
   attr_t*  dcPtr = (attr_t*)sData.dispColor ;  // see note in module header

   //* Build display string for this directory name   *
   //* Align the filename for the corresponding level *
   nOffs = fPad ;
   nOffs.limitCols( tnB.level ) ;
   fName.compose( L"%S%s%s", nOffs.gstr(), tnB.dirStat.fName, fPad ) ;
   fName.limitCols( MAX_FNWIDTH ) ;
   if ( fName.gscols() < MAX_FNWIDTH ) // if we split a two-column character
      fName.append( L" " ) ;

   gString iBuff ;
   iBuff.formatInt( (ULONG)tnB.dirFiles,   (iLEN-1) ) ; iBuff.copy( df, iLEN ) ;
   iBuff.formatInt( (ULONG)tnB.regFiles,   (iLEN-1) ) ; iBuff.copy( rf, iLEN ) ;
   iBuff.formatInt( (ULONG)tnB.linkFiles,  (iLEN-1) ) ; iBuff.copy( lf, iLEN ) ;
   iBuff.formatInt( (ULONG)tnB.cdevFiles,  (iLEN-1) ) ; iBuff.copy( cf, iLEN ) ;
   iBuff.formatInt( (ULONG)tnB.bdevFiles,  (iLEN-1) ) ; iBuff.copy( bf, iLEN ) ;
   iBuff.formatInt( (ULONG)tnB.fifoFiles,  (iLEN-1) ) ; iBuff.copy( ff, iLEN ) ;
   iBuff.formatInt( (ULONG)tnB.sockFiles,  (iLEN-1) ) ; iBuff.copy( sf, iLEN ) ;
   iBuff.formatInt( (ULONG)tnB.unkFiles,   (iLEN-1) ) ; iBuff.copy( uf, iLEN ) ;
   iBuff.formatInt( (ULONG)tnB.wprotFiles, (iLEN-1) ) ; iBuff.copy( wf, iLEN ) ;
   iBuff.formatInt( (ULONG)tnB.violations, (iLEN-1) ) ; iBuff.copy( vf, iLEN ) ;
   iBuff.compose( L"%S %s %s %s %s %s %s %s %s  %s  %s", 
                     fName.gstr(), df, rf, lf, cf, bf, ff, sf, uf, wf, vf ) ;
   iBuff.copy( strPtr, gsDFLTBYTES ) ;
   sData.dispText[sData.dispItems] = strPtr ;
   strPtr += iBuff.utfbytes() ;  // point to first free position
   dcPtr[sData.dispItems] = ftColor[fmDIR_TYPE] ;
   ++sData.dispItems ;

   //* Build display strings for each lower subdir and the files it contains   *
   for ( UINT i = ZERO ; i < tnB.dirFiles ; i++ )
      this->bcbDetailedLevel ( tnB.nextLevel[i], strPtr, sData ) ;

   //* Build display strings for non-directory files at this level             *
   tnFName* fptr ;
   for ( UINT f = ZERO ; f < tnB.tnFCount ; f++ )
   {
      //* Build the display string, save pointer to display string, and  *
      //* advance to next free space. Initialize color attribute.        *
      fptr = &tnB.tnFiles[f] ;
      nOffs = fPad ;
      nOffs.limitCols( tnB.level + 1 ) ;
      fName.compose( L"%S%s%s", nOffs.gstr(), tnB.tnFiles[f].fName, fPad ) ;
      fName.limitCols( MAX_FNWIDTH ) ;
      fName.copy( strPtr, gsDFLTBYTES ) ;
      sData.dispText[sData.dispItems] = strPtr ;
      strPtr += iBuff.utfbytes() ;  // point to first free position
      dcPtr[sData.dispItems] = ftColor[fptr->fType] ;
      ++sData.dispItems ;
   }

}  //* End bcbDetailedLevel() *

//***********************
//*                     *
//***********************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************


