//********************************************************************************
//* File       : FileDlgTree.cpp                                                 *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2005-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in FileMangler.hpp         *
//* Date       : 17-Jul-2025                                                     *
//* Version    : (see FileDlg Version string)                                    *
//*                                                                              *
//* Description:                                                                 *
//* 1) This module contains methods related a graphical representation of        *
//*    the file-system tree structure.                                           *
//*                                                                              *
//* 2) This module also contains the implementation of the 'FIND' command        *
//*    which scans the directory tree for filenames that match a specified       *
//*    pattern.                                                                  *
//*                                                                              *
//********************************************************************************
//* Version History (most recent first):                                         *
//*   See version history in FileDlg.cpp.                                        *
//********************************************************************************
//* Programmer's Notes:                                                          *
//* -------------------                                                          *
//* 1) Scanning From the Root Directory:                                         *
//*    a) Recent enhancements to the algorithm (2025-05) have made it techincally*
//*       possible to scan and display the entire local filesystem; however,     *
//*       the accumulated records (directory names) could easily reach           *
//*       50 thousand or more, so the amount of resources and the time needed    *
//*       for the scan make that operation impractical.                          *
//*    b) We _could_ allow a full report of the entire unholy mess that is the   *
//*       local filesystem; however, 50,000 directory names is a very large pile *
//*       of garbage to be generated just for the user's momentary amusement.    *
//*    c) The most attractive route for handling the root directory in Tree-view *
//*       mode is to simply display the data at the top level. Although the      *
//*       resulting display is wimpy and unsatisfactory, it is fast, accurate    *
//*       and not resource intensive.                                            *
//*    d) Note that user can toggle the recursive-scan flag while in standard    *
//*       view mode, but this is not available in Tree-view mode.                *
//*                                                                              *
//* 2) The Special-case "/proc" Directory:                                       *
//*    This directory is used by every process in the system to create the       *
//*    virtual equivalent of temporary files. Experimentation shows that this    *
//*    directory contains more files and subdirectories than any other directory *
//*    (17,000+), AND that most of it has no user-level value.                   *
//*    See notes in the 'ProcFlatScan' method of FMgrTree.cpp for additional     *
//*    information.                                                              *
//*                                                                              *
//* 3) The tree-view functionality could be significantly enhanced.              *
//*    - expand/collapse levels (stub commands in dftDisplayTree())              *
//*    - display file statistics for each directory                              *
//*    - read/write permissions                                                  *
//*    At this time, these enhancements are not worth the extra effort.          *
//*                                                                              *
//* --- --- --- --- --- --- --- --- --- --- --- --- --- --- --- --- --- --- ---  *
//* In a typical system, there are approximately 20 levels in the _local_        *
//* directory tree. This can include anywhere from 10,000 nodes (directory names)*
//* to 50,000 and upward. It is impractical to create a scrolling list with that *
//* many items, so if the scan begins in the root directory, a recursive scan    *
//* is not performed. With the possible exception of the procfs filesystem       *
//* ("/proc" directory), no single node is likely to overwhelm the system's      *
//* resources.                                                                   *
//*   -- As always under Wayland, if the system is not fully awake when          *
//*      Tree-view mode is invoked, Wayland may not be able to handle a          *
//*      large memory allocation request or may fill that request only           *
//*      after a significant delay. Be aware.                                    *
//*   -- To compensate for the system's weak memory allocation support,          *
//*      Tree-view memory allocations are performed in a "nothrow" loop          *
//*      to minimize system exceptions.                                          *
//********************************************************************************

#include "GlobalDef.hpp"
#include "FileDlg.hpp"

//******************************
//* Local definitions and data *
//******************************
//* Message text defined in FileDlgPrompt.cpp *
extern const char* fmtypeString[] ;
extern const char* fmtypeName[] ;
extern const char* readonlyFileSys[] ;
extern const char* notOntoSelf[] ;
extern const char* cannotOverrideWP[] ;
extern const char* noLinkOverwrite[] ;
extern const char* differentFileTypes[] ;
extern const char* cannotDeleteSpecial[] ;
extern const char* cannotOverwriteFifo[] ;
extern const char* cannotOverwriteDirectory[] ;
extern const char* warnSocketMod[] ;
extern const char* warnBdevCdevMod[] ;
extern const char* warnUnsuppMod[] ;
extern const char* fdNotImplemented[] ;

static const char* const noMatchMsg[] = 
{
   "   No matching directory name found.   ",
   "No more matching directory names found.",
} ;

static const char *parentDir   = ".. (parent dir)" ;
static const char *procMsgText = "    Processing..." ;
static const char *formMsgText = "    Formatting..." ;

//* Setup data for the 'stats' Textbox in the Tree-view dialog.*
static short statWidth = 41 ;
static const char *statInvisible = "─────────────────────────────────────────" ;
// (emulates a horizontal line)     123456789-123456789-123456789-123456789-1
static const char *statTemplate  = "┤ %-38s├" ; // (statWidth - 3)

//* Dialog control definitions for Tree-view Mode *
enum ctrlNames : short { dummyTB = ZERO, statTB, pathTB, dftcontrolsDEFINED } ;

//* User Action Codes *
enum UserChoice : short
{
   ucNO_SEL = 0,           // no selection - exit tree view to initial directory
   ucNEW_CWD,              // exit tree view with specified directory as CWD
   ucNEW_BASE,             // set specified directory as base for tree-view scan
   ucDEEP_SCAN,            // enable recursive scan from root directory
   ucEXIT_APP              // no selection - exit application [not implemented]
} ;

//* For debugging only. Write scan progress messages to a temp file.*
#define DEBUG_DFT (0)
#if DEBUG_DFT != 0
const char* dftTemplate = "%s/Temp/dftDebug.txt" ;
gString gsdb ;
ofstream dftofs ;
#endif   // DEBUG_DFT

//********************
//* Local prototypes *
//********************
static void dftSortNode ( TreeNode* tnPtr, UINT count ) ;
static void dftMarkRestricted ( gString& dName ) ;
static void dftMarkExternal ( gString& dName ) ;
static bool dftNameSearch ( const dftDisplay& dftd, 
                            const gString ptrn, UINT& fIndex ) ;
static void dftNoMatch ( const winPos& spPos, const char* msg, attr_t dColor ) ;
static void dtfFormatStatMsg ( dtbmData& statMsg, UINT dirFiles, UINT totFiles, bool isRoot ) ;


//*************************
//*   DisplayFileTree     *
//*************************
//********************************************************************************
//* Display the file-system tree structure beginning with the 'current'          *
//* directory and traversing all subdirectories below current directory.         *
//*                                                                              *
//* User has the option of selecting a new current-working-directory' (CWD)      *
//* from the displayed data.                                                     *
//*                                                                              *
//* Input  : ul        : (by reference) upper left corner dialog position        *
//*          rows      : number of rows for dialog                               *
//*          cols      : number of columns for dialog                            *
//*                                                                              *
//* Returns: true if current working directory (CWD) has changed                 *
//*          else false                                                          *
//********************************************************************************

bool FileDlg::DisplayFileTree ( winPos ul, short rows, short cols )
{
   gString origcwd ;                // CWD on entry
   this->GetPath ( origcwd ) ;      // where we are now
   gString newcwd = origcwd ;       // where we will be on return
   dtbmData procMsg( procMsgText ) ;// 'processing' messages
   dtbmData formMsg( formMsgText ) ;
   dtbmData msgData ;               // textbox messages to user
   gString  gsStat ;                // format text for display in stats textbox
   dtbmData statMsg ;               // stats textbox messages
   TreeNode* treeBase = NULL ;      // pointer to captured directory tree
   TreeNode tnAcc ;                 // directory-tree accumulator
   DispData dData( cols - 2 ) ;     // manage multi-threaded tree capture
   bool orig_irRoot = this->fmPtr->ScanFromRoot ( false, true ),  // original flag state
        curr_irRoot = false,        // current flag state
        isRoot      = false,        // set if current tree base == root directory
        newCWD      = false ;       // return value
   short status ;                   // value returned from user interaction

   dData.setTreeViewFlag () ;       // Set the tree-view-scan flag

   #if DEBUG_DFT != 0
   // Debug the Tree-mode sequence
   const char* fmghome = getenv ( "FMG_HOME" ) ;
   if ( fmghome != NULL )
   {
      gsdb.compose( dftTemplate, fmghome ) ; dftofs << gsdb << endl ;
      dftofs.open( gsdb.ustr(), ofstream::out | ofstream::trunc ) ;
      if ( dftofs.is_open() )
      {
         dftofs << "DisplayFileTree: debugging data\n"
                << "===============================" << endl ;
      }
   }
   #endif   // DEBUG_DFT

   //* Save the display for the parent dialog window *
   this->dPtr->SetDialogObscured () ;

   //* Open the Tree-view dialog *
   attr_t iColor = this->cs.tn,     // item color
          hColor = this->cs.sd ;    // highlight color
   NcDialog* dp = this->dftOpenDialog ( ul, rows, cols, iColor ) ;

   #if DEBUG_DFT != 0
   if ( dftofs.is_open() ) { dftofs << "dft dialog opened" << endl ; }
   else { dp->DebugMsg ( "                    ** Debug File Did Not Open. **  " ) ; }
   #endif   // DEBUG_DFT

   //* Disable the root-scan flag *
   this->fmPtr->ScanFromRoot ( false ) ;

   //* Initialize the display parameters *
   dftDisplay dftd( ZERO, (rows - 8), (cols - 2), iColor, hColor ) ;

   #if DEBUG_DFT != 0
   if ( dftofs.is_open() )
   {
      gsdb.compose( "dftd parameters: rows:%u cols:%hd lshift:%hd width:%hd levels:%hd\n"
                    "                 first:%u last:%u curr:%u total:%u",
                    &dftd.rows, &dftd.cols, &dftd.lshift, &dftd.width, &dftd.levels, 
                    &dftd.first, &dftd.last, &dftd.curr, &dftd.total ) ;
      dftofs << gsdb << endl ;
   }
   #endif   // DEBUG_DFT

   //* Loop until either user makes a selection or *
   //* gets too tired to climb the tree.           *
   bool  done = (dp == NULL) ? true : false ;  // loop control
   while ( ! done )
   {
      //* Initial processing message. If the scan requires a long time, *
      //* then this message will be overwritten by the progress monitor.*
      dp->DisplayTextboxMessage ( pathTB, procMsg ) ;

      //* Capture the data for the directory tree.                   *
      //* -- For root directory, capture only the top-level          *
      //* subdirectories _unless_ user has requested a full scan.    *
      //* If below root directory, capture the full directory tree.  *
      if ( (isRoot = ((newcwd.compare( ROOT_PATH )) == ZERO)) )
      {
         #if DEBUG_DFT != 0
         if ( dftofs.is_open() )
         { gsdb.compose( "isRoot" ) ; dftofs << gsdb << endl ; }
         #endif   // DEBUG_DFT

         if ( !curr_irRoot )
            treeBase = this->fmPtr->CaptureDirTree ( newcwd, dData, false, true ) ;
         else
         {
            // Programmer's Note: Currently, this is never executed.
            //this->fmPtr->ScanFromRoot ( true ) ;   // enable deep scan
            //treeBase = this->dftCaptureTree ( newcwd, dp, &dData ) ;
            //this->fmPtr->ScanFromRoot ( false ) ;  // disable deep scan
            //curr_irRoot = false ;
         }
      }
      else
      {
         #if DEBUG_DFT != 0
         if ( dftofs.is_open() )
         { gsdb.compose( "dftCaptureTree(call)" ) ; dftofs << gsdb << endl ; }
         #endif   // DEBUG_DFT

         treeBase = this->dftCaptureTree ( newcwd, dp, &dData ) ;

         #if DEBUG_DFT != 0
         if ( dftofs.is_open() )
         {
            gsdb.compose( "dftCaptureTree(return: treeBase:%p)", treeBase ) ;
            dftofs << gsdb << endl ;
            if ( treeBase != NULL )
            {
               gsdb.compose( "   fName:'%s'   fType:%hd fBytes:%llu\n"
                             "   totFiles:%u dirFiles:%u regFiles:%u linkFiles:%u\n"
                             "   nextLevel:%p, prevLevel:%p", 
                             treeBase->dirStat.fName, 
                             &treeBase->dirStat.fType, &treeBase->dirStat.fBytes,
                             &treeBase->totFiles, &treeBase->dirFiles, &treeBase->regFiles,
                             &treeBase->linkFiles,
                             treeBase->nextLevel, treeBase->prevLevel ) ;
               dftofs << gsdb << endl ;
            }
         }
         #endif   // DEBUG_DFT
      }

      //* If scan successfully completed,display the results.*
      if ( ! (dData.getAbortFlag()) )
      {
         //* Allocate space for the display data (incl. "parent" record) *
         this->fmPtr->TreeNodeSummary ( treeBase, tnAcc ) ;
         dftd.realloc( tnAcc.dirFiles + 1 ) ;

         #if DEBUG_DFT != 0
         if ( dftofs.is_open() )
         {
            gsdb.compose( "dftd reallocate: tnAcc.dirFiles:%u  display-item total:%u",
                          &tnAcc.dirFiles, &dftd.total ) ;
            dftofs << gsdb << endl ;
         }
         #endif   // DEBUG_DFT

         //* Construct tree-view data from the directory tree data.*
         dp->DisplayTextboxMessage ( pathTB, formMsg ) ;
         this->dftFormatTree ( dftd, treeBase ) ;

         #if DEBUG_DFT != 0
         if ( dftofs.is_open() )
         {
            gsdb.compose( 
               "dftFormatTree  : rows:%u cols:%hd lshift:%hd width:%hd levels:%hd\n"
               "                 first:%u last:%u curr:%u total:%u",
               &dftd.rows, &dftd.cols, &dftd.lshift, &dftd.width, &dftd.levels, 
               &dftd.first, &dftd.last, &dftd.curr, &dftd.total ) ;
            dftofs << gsdb << endl ;
         }
         #endif   // DEBUG_DFT

         //* Report scan statistics *
         dtfFormatStatMsg ( statMsg, tnAcc.dirFiles, tnAcc.totFiles, isRoot ) ;
         dp->DisplayTextboxMessage ( statTB, statMsg ) ;

         //* Display the current directory path.*
         dp->SetTextboxText ( pathTB, newcwd ) ;

         #if DEBUG_DFT != 0
         if ( dftofs.is_open() )
         { gsdb.compose( "dftDisplayTree(call)" ) ; dftofs << gsdb << endl ; }
         #endif   // DEBUG_DFT

         //* Display the data and interact with the user *
         //*     (as distasteful as that may be...)      *
         status = this->dftDisplayTree ( dp, dftd, newcwd ) ;

         #if DEBUG_DFT != 0
         if ( dftofs.is_open() )
         { gsdb.compose( "dftDisplayTree(status:%hd)", &status ) ; dftofs << gsdb << endl ; }
         #endif   // DEBUG_DFT
      }

      //* User aborted scan: Display warning message, pause for response, *
      //* then set status to code for exit to original working directory. *
      else
      {
         msgData = "Scan aborted, Press any key to exit Tree-view Mode." ;
         msgData.centered = true ;
         for ( short c = ZERO ; c < gsALLOCDFLT ; ++c )
            msgData.colorData[c] = nc.reG ;
         dp->DisplayTextboxMessage ( pathTB, msgData ) ;
         nckPause();
         status = ucNO_SEL ;
      }

      //* Release local copy of directory tree *
      if ( treeBase != NULL )
      { this->fmPtr->ReleaseDirTree ( treeBase ) ; treeBase = NULL ; }

      //* Decode user's instructions *
      switch ( status )
      {
         //* Exit tree-view mode and signal that main *
         //* dialog should exit the application.      *
         //* CTRL+X option not currently implemented. *
         //* Same functionality as CTRL+Q.            *
         case ucEXIT_APP:
         //* Exit tree view to original directory *
         case ucNO_SEL:
            //* Close the Tree-view dialog *
            {
            if ( dp != NULL )
            { delete ( dp ) ; dp = NULL ; this->dPtr->RefreshWin () ; }
            bool not_root = bool((newcwd.compare( L"/" )) != ZERO) ;
            this->SetDirectory ( origcwd, not_root ) ;
            }
            done = true ;
            break ;

         //* If user selected a new target directory, close the *
         //* tree-view dialog, set the the target directory, `  *
         //* then scan and display contents of new target.      *
         case ucNEW_CWD:
            {
               if ( dp != NULL )             // close the Tree-view dialog
                  { delete ( dp ) ; dp = NULL ; this->dPtr->RefreshWin () ; }
               //* If target is root directory, signal *
               //* that root-scan is to be disabled.   *
               bool not_root = bool((newcwd.compare( L"/" )) != ZERO) ;
               if ( (this->SetDirectory ( newcwd, not_root )) == OK )
               {
                  msgData = " New working directory selected. " ;
                  this->dPtr->DisplayTextboxMessage ( this->mIndex, msgData ) ;
                  newCWD = true ;      // alert caller that directory has changed
               }
               else
               {  //* Error reading new directory. (Should not happen unless *
                  //* user selected a directory with no read access.)        *
                  msgData = " Access Error! Unable to read specified directory." ;
                  this->dPtr->DisplayTextboxMessage ( this->mIndex, msgData ) ;
                  chrono::duration<short>aWhile( 2 ) ;
                  this_thread::sleep_for( aWhile ) ;
               }
            }
            done = true ;
            break ;

         //* A new tree-view base directory has been selected.*
         //* New base directory is in 'newcwd'. This could    *
         //* be parent, child or root target.                 *
         case ucNEW_BASE:
            /* nothing to be done at this time */
            break ;

         //* For root directory, enable recursive scan *
         //* for next iteration of the loop.           *
         case ucDEEP_SCAN:
            //curr_irRoot = true ;
            break ;

         default:                         // (this should never happen)
            break ;
      }
   }     // while(!done)

   if ( treeBase != NULL )       // be sure that dynamic memory is released
   { this->fmPtr->ReleaseDirTree ( treeBase ) ; treeBase = NULL ; }

   //* Close the Tree-view dialog (in case it didn't happen earlier) *
   if ( dp != NULL )
   {
      delete ( dp ) ;
      dp = NULL ; 
      this->dPtr->RefreshWin () ;      // restore the parent dialog
   }

   //* Return root-scan flag to its original state. *
   if ( orig_irRoot )
      this->fmPtr->ScanFromRoot ( orig_irRoot ) ;

   #if DEBUG_DFT != 0
   if ( dftofs.is_open() )
   {
      dftofs << "\n--------\nBye-bye!" << endl ;
      dftofs.close() ;
   }
   #endif   // DEBUG_DFT

   return newCWD ;

}  //* End DisplayFileTree() *

//*************************
//*     dftOpenDialog     *
//*************************
//********************************************************************************
//* Open the Tree-view dialog window and return a pointer to it.                 *
//*                                                                              *
//*                                                                              *
//* Input  : ul    : (by reference) upper left corner dialog position            *
//*          rows  : number of rows for dialog                                   *
//*          cols  : number of columns for dialog                                *
//*          iColor: dialog interior color                                       *
//*                                                                              *
//* Returns: pointer to open dialog window                                       *
//*          OR NULL pointer if dialog did not open                              *
//********************************************************************************

NcDialog* FileDlg::dftOpenDialog ( const winPos& ul, short rows, 
                                   short cols, attr_t iColor )
{
/*
UP_ARROW   == PgUp == Home == Scroll Upward
DOWN_ARROW == PgDn == End  == Scroll Downward
   (also mouse wheel and top/bot border clicks)
ENTER == Select new CWD
CTRL+F/CTRL+G == Find/Find_Again
CTRL+Y == Display Filesystem Info                                                
CTRL+H == Tree-mode Help
CTRL+U == Tree-mode up one level
CTRL+D == Tree-mode down one level
CTRL+R == Tree-mode root base
CTRL+Q == Exit to original directory
CTRL+X == Exit the application.
*/
const char* const userInstructions =
//  ---------|---------|---------|---------|---------|---------|---------|-----|
   "Scrolling: Up/Down/Right/Left, PgUp/PgDn, Home/End. Exit Tree to Cwd:ENTER. \n"
   "Level Up:CTRL+U, Down:CTRL+D, Go to Root Dir:CTRL+R. Quit Tree Mode:CTRL+Q. \n"
   "Find:CTRL+F. Filesystem Info:CTRL+Y. Help:CTRL+H. " ;

attr_t bColor = this->cs.bb ;    // border color

InitCtrl ic[dftcontrolsDEFINED] =                                  
{
   {  //* 'dummy' textbox  - - - - - - - - - - - - - - - - - - - - -   dummyTB *
      dctTEXTBOX,                   // type:
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      3,                            // ulY:       upper left corner in Y
      1,                            // ulX:       upper left corner in X
      1,                            // lines:     control rows
      1,                            // cols:      control columns
      "",                           // dispText:  initially empty
      bColor,                       // nColor:    non-focus color
      bColor,                       // fColor:    focus color
      tbPathLinux,                  // filter: valid filespec chars (incl. Linux "special")
      NULL,                         // label:     
      ZERO,                         // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[statTB],                  // nextCtrl:  link in next structure
   },
   {  //* 'stat' textbox  - - - - - - - - - - - - - - - - - - - - - -  statTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      2,                            // ulY:       upper left corner in Y
      short(cols / 2 - statWidth / 2), // ulX:       upper left corner in X
      1,                            // lines:     control rows
      statWidth,                    // cols:      control columns
      statInvisible,                // dispText:  initially invisible
      bColor,                       // nColor:    non-focus color
      bColor,                       // fColor:    focus color
      tbPathLinux,                  // filter: valid filespec chars (incl. Linux "special")
      NULL,                         // label:     
      ZERO,                         // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[pathTB],                  // nextCtrl:  link in next structure
   },
   {  //* 'new target path' textbox  - - - - - - - - - - - - - - - -    pathTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      1,                            // ulY:       upper left corner in Y
      1,                            // ulX:       upper left corner in X
      1,                            // lines:     control rows
      short(cols - 2),              // cols:      control columns
      "",                           // dispText:  initially empty
      this->cs.tf,                  // nColor:    non-focus color
      (this->cs.pf & ~ncrATTR),     // fColor:    focus color
      #if LINUX_SPECIAL_CHARS != 0
      tbPathLinux,                  // filter: valid filespec chars (incl. Linux "special")
      #else    // BASIC FILESPEC FILTER
      tbPathName,                   // filter:    valid filename characters
      #endif   // BASIC FILESPEC FILTER
      NULL,                         // label:     
      ZERO,                         // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    view only
      NULL                          // nextCtrl:  link in next structure
   },
} ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( rows,           // number of display lines
                       cols,           // number of display columns
                       ul.ypos,        // Y offset from upper-left of terminal 
                       ul.xpos,        // X offset from upper-left of terminal
                       " FileMangler - Tree View  (Close: CTRL+Q) ", // dialog title
                       ncltSINGLE,     // border line-style
                       bColor,         // border color attribute
                       iColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;
   if ( (dp->OpenWindow()) == OK )
   {
      //* Draw the static data *
      dp->ClearLine ( 3, bColor ) ;
      dp->ClearLine ( 4, bColor ) ;
      dp->ClearLine ( 5, bColor ) ;
      dp->WriteParagraph ( 3, 2, userInstructions, bColor ) ;
      LineDef  lDefH(ncltHORIZ, ncltSINGLE, 2, ZERO, cols, bColor) ;
      dp->DrawLine ( lDefH ) ;
      lDefH.startY = 6 ;
      dp->DrawLine ( lDefH ) ;
      dp->RefreshWin () ;
   }
   return dp ;

}  //* End dftOpenDialog() *

//*************************
//*    dftCaptureTree     *
//*************************
//********************************************************************************
//* Private Method:                                                              *
//* ---------------                                                              *
//* Called by DisplayDirTree() to perform the multi-threaded capture of the      *
//* directory tree.                                                              *
//*                                                                              *
//* Input  : gsBase : (by reference) filespec of base directory for scan         *
//*        : dp     : pointer to dialog in which data are to be displayed        *
//*                   This is used to display progress update messages.          *
//*          ddPtr  : for controlling access to shared data during scan          *
//*                   Initialized members:                                       *
//*                   'strWidth' : width of display area                         *
//*                   'treeView' : flag indicating capture of directories only   *
//*                                                                              *
//* Returns: pointer to captured tree data                                       *
//*          In the unlikely event of a scan failure, a null pointer is returned *
//********************************************************************************
//* Note: This method launches a child thread as a progress monitor, and then    *
//*       deletes it when the scan is complete.                                  *
//*                                                                              *
//********************************************************************************

TreeNode* FileDlg::dftCaptureTree ( const gString& gsBase, NcDialog* dp, DispData* ddPtr )
{
   dtbmData msgData ;               // textbox messages to user
   TreeNode* treeBase = NULL ;      // pointer to captured directory tree (return value)

   //********************************************************
   //* Launch a thread to handle reporting of scan progress *
   //********************************************************
   thread* monitorThread ;       // thread which reports progress
   bool  monAllocated = false,   // 'true' if monitor thread allocated
         monActive    = false ;  // 'true' if monitor thread launched
   chrono::duration<short, std::milli>aMoment( 10 ) ;
   for ( short i = 3 ; i > ZERO ; --i ) // three strikes, and you're out!
   {
      if ( (monitorThread = new (std::nothrow) std::thread) != NULL )
      {
         monAllocated = true ;
         try
         {
            *monitorThread = thread( &FileDlg::dftProgmon, this, ddPtr, &monActive, dp ) ;
            this_thread::sleep_for( aMoment ) ;  // give the new thread time to respond
            if ( ! monActive )      // second chance
            {
               *monitorThread = thread( &FileDlg::dftProgmon, this, ddPtr, &monActive, dp ) ;
               this_thread::sleep_for( aMoment ) ;  // give the new thread time to respond
            }
         }
         catch ( ... )        // monitor thread created but not launched
         {
            dtbmData tbMsg( " Progress-monitor thread not launched! " ) ;
            dp->DisplayTextboxMessage ( pathTB, tbMsg ) ;
            // Programmer's Note: We cannot set an exception in 'ddPtr' here 
            // because '*ddPtr' reset before scanning the directory tree.
         }
         break ;
      }
      this_thread::sleep_for( aMoment ) ;  // give the system some time
   }
   if ( ! monAllocated )
      ddPtr->setException( "Progress-monitor thread not allocated." ) ;
   else if ( ! monActive )
   {
      ddPtr->setException( "Progress-monitor thread not launched." ) ;

      if ( monitorThread != NULL )  // failure to launch, delete the thread structure
      { delete monitorThread ; monitorThread = NULL ; }
   }

   //* If thread allocation failed, return with the bad news.*
   //* Note the early return.                                *
   if ( ! (monAllocated && monActive) )
      return treeBase ;

   //***************************************************
   //* Scan the directory tree from current directory, *
   //* downward. (only directory files are captured)   *
   //* Then wait for all scan threads to report.       *
   //***************************************************
   treeBase = this->fmPtr->CaptureDirTree ( gsBase, *ddPtr, true, true ) ;
   aMoment *= 5 ;                         // extend sleep interval to 50 mSec.
   do { this_thread::sleep_for( aMoment ) ; }
   while ( (ddPtr->getActiveThreads ()) > ZERO ) ;

   return treeBase ;

}  //* End dftCaptureTree() *

//*************************
//*      dftProgmon       *
//*************************
//********************************************************************************
//* PRIVATE METHOD:                                                              *
//* ---------------                                                              *
//* Monitor the progress of directory-tree scan and display the number of files  *
//* scanned so far.                                                              *
//*                                                                              *
//* The timing is set up so that if the scan completes in less than              *
//* 'aWhile', then no display updates will occur. Updates will occur only if     *
//* the scan is taking a "long" time e.g. from slow external devices or large    *
//* amounts of data.                                                             *
//*                                                                              *
//* Key input from user is monitored, and if any keycode is received, the        *
//* scan-abort flag will be set to signal that all scan threads should terminate *
//* immediately. Note that for each active thread there will be a finite delay   *
//* before responding to the abort signal because the flag will be tested only   *
//* at certain points in the scan,                                               *
//*                                                                              *
//*                                                                              *
//* Input  : ddPtr  : pointer to DispData object containing status of            *
//*                   directory-tree scan                                        *
//*          active : pointer to flag: flag is set on entry to signal that       *
//*                   the monitor thread is active                               *
//*        : dp     : pointer to dialog in which data are to be displayed        *
//*                   This is used to display progress update messages.          *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FileDlg::dftProgmon ( DispData* ddPtr, bool* active, NcDialog* dp )
{
   *active = true ;     // signal to caller that we are active

   chrono::duration<short, std::milli>aWhile( 100 ) ; // pause before reporting
   chrono::duration<short, std::milli>aMoment( 50 ) ; // sleep interval
   dtbmData tbMsg ;              // message output
   gString  gsOut, gsi ;         // message formatting
   wkeyCode wKey ;               // test for usetr abort
   UINT     fCount = ZERO, fc ;  // number of files scanned
   UINT64   fBytes = ZERO, fb ;  // total file size
   bool     aborted = false ;    // set if user aborts scan

   //* Wait for either the scan to be initiated OR a signal *
   //* that the scan is unnecessary (or system error).      *
   //* Note that in many cases, a single iteration is enough*
   //* time for the scan to be completed, so there will be  *
   //* no need to display progress.                         *
   do
   { this_thread::sleep_for( aWhile ) ; }
   while ( ! (ddPtr->uiReport()) ) ;

   //* Report on scan progress *
   while ( (ddPtr->getActiveThreads ()) > ZERO )
   {
      ddPtr->getData ( fc, fb ) ;         // get progress
      if ( fc != fCount || fb != fBytes ) // if progress made, report it
      {
         fCount = fc ;
         fBytes = fb ;
         gsi.formatInt ( fCount, 13, true ) ;
         gsOut.compose( "    files scanned: %S", gsi.gstr() ) ;

         //* Test key input stream for possible user abort.*
         if ( ! aborted &&
              (aborted = bool((this->dPtr->KeyPeek ( wKey )) != wktERR)) )
         {
            ddPtr->setAbortFlag () ;             // signal all sub-threads to abort
            this->dPtr->FlushKeyInputStream () ; // discard the keycode
            #if DEBUG_DFT != 0
            if ( dftofs.is_open() )
            { gsdb.compose( "dftProgMon(***Abort***)" ) ; dftofs << gsdb << endl ; }
            #endif   // DEBUG_DFT
         }
         if ( aborted ) { gsOut.append( L" -- Scan Aborted..." ) ; }

         gsOut.copy( tbMsg.textData, gsALLOCDFLT ) ;
         dp->DisplayTextboxMessage ( pathTB, tbMsg ) ;

         this_thread::sleep_for( aMoment ) ; // wait a moment
      }
   }

   //* Clear the progress message. *
   tbMsg.textData[ZERO] = NULLCHAR ;
   dp->DisplayTextboxMessage ( pathTB, tbMsg ) ;

}  //* End dftProgmon() *

//*************************
//*    dftDisplayTree     *
//*************************
//********************************************************************************
//* Display a graphic representation of the tree structure in the open area      *
//* of the specified dialog.                                                     *
//* User may optionally select a new CWD.                                        *
//*                                                                              *
//* Input  : dp    : pointer to open Tree-view dialog window                     *
//*          dftd  : (by reference) display-data control info                    *
//*          newcwd: (by reference)                                              *
//*                  - on entry contains full path of current-working directory  *
//*                  - if user selects a new current-working-directory path,     *
//*                    then path is returned here, else, contents undefined      *
//*                                                                              *
//* Returns: member of enum UserChoice:                                          *
//*          ucNO_SEL   : user made no selection - exit tree view and return     *
//*                       control to main application dialog in directory from   *
//*                       which tree view was invoked                            *
//*          ucNEW_CWD  : exit tree view mode with filespec for new CWD          *
//*          ucNEW_BASE : set specified directory (child/parent/root) as the     *
//*                       new base directory for tree view mode                  *
//*          ucDEEP_SCAN: enable recursive scan from root directory              *
//*                       (note that root is already base node of tree view)     *
//*                       - Programmer's Note: The Linux/Wayland allocation of   *
//*                         dynamic memory, coupled with accessing files reserved*
//*                         for the system can cause various system exceptions   *
//*                         during a recursive scan from root. For this reason,  *
//*                         we have disabled the recursive root scan. See below. *
//*          ucEXIT_APP : exit tree view mode and signal main dialog that it     *
//*                       should exit the application                            *
//********************************************************************************
//* Programmer's Note:                                                           *
//* For left-shift and right-shift: The sense of the arrows is intended to be    *
//* user-oriented. That is, if user presses right arrow, the _highlight_         *
//* appears to move to the right (the data move left). This is the opposite of   *
//* programmer thinking where 'go left' means that the data should go left.      *
//*                                                                              *
//* Also note that because characters and columns are not the same thing,        *
//* when shifted left, the first character displayed on a line may occasionally  *
//* be different from the one you might expect. In general, characters require   *
//* either one or two columns for display. In the case where the first           *
//* character to be displayed in a string is a two-column character beginning    *
//* on an ODD column offset, it may be skipped. No harm done though.             *
//*                                                                              *
//********************************************************************************

short FileDlg::dftDisplayTree ( NcDialog* dp, dftDisplay& dftd, gString& newcwd )
{
   gString oldcwd = newcwd,      // CWD on entry
           gsName,               // filename formatting
           gsPath,               // target-path construction
           gstmp,                // general text formatting
           sPattern ;            // user's search criteria
   UINT  sIndex = ZERO,          // search index
         sMatch = ZERO,          // index of search match
         prevCurr = ZERO ;       // previous position of highlight
   short firstDLine, y,          // tracking for data redraw
         fnIndx ;                // filename formatting
   attr_t tColor ;               // line-item color attribute
   wkeyCode wk ;                 // key input data
   UserChoice status = ucNO_SEL ;// return value
   bool  redraw   = true,        // if set, redraw display data
         himove   = false,       // if set, move highlight to new position
         done     = false ;      // loop control

   //* Establish position of active window (for mouse input) *
   short rows, cols ;
   dp->GetDialogDimensions ( rows, cols ) ;
   short btop = ZERO + 6,        // top border (of tree-view area)
         bbot = rows - 1,        // bottom border
         blef = ZERO,            // left border
         brgt = cols - 1 ;       // right border
   firstDLine = btop + 1 ;       // top display row

   #if DEBUG_DFT != 0
   if ( dftofs.is_open() )
   { gsdb.compose( "dftDisplayTree(enter '%S')", newcwd.gstr() ) ; dftofs << gsdb << endl ; }
   #endif   // DEBUG_DFT

   while ( ! done )
   {
      //* Redraw the display area, *
      if ( redraw )
      {
         #if DEBUG_DFT != 0
         if ( dftofs.is_open() )
         { gsdb.compose( "   ( redraw #1 )" ) ; dftofs << gsdb << endl ; }
         #endif   // DEBUG_DFT

         //* Clear the display space *
         y = firstDLine ;
         for ( UINT i = ZERO ; i < dftd.rows ; i++ )
            dp->ClearLine ( y++ ) ;
         y = firstDLine ;

         //* Display the data *
         for ( UINT di = dftd.first ; di <= dftd.last ; di++ )
         {
            //* Set text color attribute for this line.*
            tColor = (di == dftd.curr) ? dftd.hColor : dftd.iColor ;
            gstmp = dftd.data[di].n ;
            if ( dftd.lshift > ZERO )     // if data are left-shifted
            {
               //* Shift left by specified amount *
               gstmp.shiftCols( -(dftd.lshift) ) ;

               //* If currently-highlighted item has been shifted out of    *
               //* display area, mark the spot so user doesn't get confused.*
               //* If entire (non-highlighted line) has been shifted out of *
               //* display area, nothing to display (gstmp is empty string).*
               if ( di == dftd.curr ) 
                  gstmp = L"<<" ;
            }
            gstmp.limitCols( dftd.cols ) ;
            dp->WriteString ( y++, 1, gstmp, tColor ) ;
         }

         #if DEBUG_DFT != 0
         if ( dftofs.is_open() )
         { gsdb.compose( "   ( redraw #2 )" ) ; dftofs << gsdb << endl ; }
         #endif   // DEBUG_DFT
      }  // (redraw)

      //* Move the highlight within the display area. *
      else if ( himove )
      {
         #if DEBUG_DFT != 0
         if ( dftofs.is_open() )
         { gsdb.compose( "   ( himove #1 )" ) ; dftofs << gsdb << endl ; }
         #endif   // DEBUG_DFT

         y = firstDLine + prevCurr - dftd.first ;
         gstmp = dftd.data[prevCurr].n ;
         if ( dftd.lshift > ZERO )     // if data are left-shifted
            gstmp.shiftCols( -(dftd.lshift) ) ;
         gstmp.limitCols( dftd.cols ) ;
         dp->WriteString ( y++, 1, gstmp, dftd.iColor ) ;

         y = firstDLine + dftd.curr - dftd.first ;
         gstmp = dftd.data[dftd.curr].n ;
         if ( dftd.lshift > ZERO )     // if data are left-shifted
         { gstmp.shiftCols( -(dftd.lshift) ) ; gstmp = L"<<" ; }
         gstmp.limitCols( dftd.cols ) ;
         dp->WriteString ( y++, 1, gstmp, dftd.hColor ) ;

         #if DEBUG_DFT != 0
         if ( dftofs.is_open() )
         { gsdb.compose( "   ( himove #2 )" ) ; dftofs << gsdb << endl ; }
         #endif   // DEBUG_DFT
      }

      //* Update the filespec display.                        *
      //* Display the full path of highlighted directory name *
      if ( redraw || himove )
      {
         #if DEBUG_DFT != 0
         if ( dftofs.is_open() )
         { gsdb.compose( "   ( fspec  #1 )" ) ; dftofs << gsdb << endl ; }
         #endif   // DEBUG_DFT

         gsName = dftd.data[dftd.curr].n ;
         if ( (gsName.compare( parentDir )) == ZERO )
            this->fmPtr->ExtractPathname ( gsPath, oldcwd ) ;
         else
         {
            gsName.strip() ;           // strip leading and trailing whitespace

            //* If tagged with one of the special indicators, *
            //* strip the indicator.                          *
            if ( (fnIndx = gsName.find( extFileSys )) > ZERO )
               gsName.limitChars( fnIndx ) ;
            else if ( (fnIndx = gsName.find( noAccess )) > ZERO )
               gsName.limitChars( fnIndx ) ;

            //* If highlighted name is base directory     *
            //* a) dftd.curr == ZERO, then no parent item *
            //* b) dftd.curr == 1, and parent item exists *
            gstmp = dftd.data[ZERO].n ;
            if ( (dftd.curr == ZERO) || 
                 ((dftd.curr == 1) && ((gstmp.compare( parentDir )) == ZERO)) )
            { gsPath = oldcwd ; }
            //* Construct the path of highlighted directory name *
            else
            {
               gsPath = oldcwd ;
               if ( (oldcwd.compare( ROOT_PATH )) != ZERO )
                  gsPath.append( fSLASH ) ;
               gsPath.append( gsName.gstr() ) ;
               fnIndx = gsPath.findlast( fSLASH ) ;
               UINT di = dftd.curr,
                    prevlev ;
               short tmpIndx ;
               while ( (prevlev = dftd.data[di].l) > 1 )
               {
                  while ( dftd.data[--di].l >= prevlev ) ;
                  gstmp = dftd.data[di].n ;
                  gstmp.strip() ;
                  if ( (tmpIndx = gstmp.find( extFileSys )) > ZERO )
                     gstmp.limitChars( tmpIndx ) ;
                  else if ( (tmpIndx = gstmp.find( noAccess )) > ZERO )
                     gstmp.limitChars( tmpIndx ) ;
                  gstmp.insert( fSLASH ) ;
                  gsPath.insert( gstmp.gstr(), fnIndx ) ;
                  prevlev = dftd.data[di].l ;
               }
            }
         }

         //* Shift text to the left so the rightmost char will be visible *
         gstmp = gsPath ;
         if ( gstmp.gscols() > (cols - 2) )
            gstmp.shiftCols( (cols - 2) - gstmp.gscols() ) ;
         dp->SetTextboxText ( pathTB, gstmp ) ;

         #if DEBUG_DFT != 0
         if ( dftofs.is_open() )
         { gsdb.compose( "   ( fspec  #2 )" ) ; dftofs << gsdb << endl ; }
         #endif   // DEBUG_DFT
      }  // (redraw || himove)
      redraw = himove = false ;  // reset the flags

      dp->RefreshWin () ;        // make changes visible

      #if DEBUG_DFT != 0
      if ( dftofs.is_open() ) { dftofs << "   ( refreshed )" << endl ; }
      #endif   // DEBUG_DFT

      //*************************
      //** Get input from user **
      //*************************
      dp->GetKeyInput ( wk ) ;
      #if DEBUG_DFT != 0
      if ( dftofs.is_open() )
      { gsdb.compose( "   ( keycode: type:%hX key:%04X )", &wk.type, &wk.key ) ; dftofs << gsdb << endl ; }
      #endif   // DEBUG_DFT

      //* Convert any unconverted mouse input within the scrolling window *
      //* to keycodes. Emulates conversions within the NcDialog API.      *
      if ( (wk.type == wktMOUSE) && !wk.mevent.conv && (this->dPtr->meMouseEnabled ()) )
      {
         //* Test whether the event lies within our scrolling window *
         if ( wk.mevent.ypos >= btop && wk.mevent.xpos >= blef &&
              wk.mevent.ypos <= bbot && wk.mevent.xpos <= brgt )
         {
            //* ScrollWheel upward or click in top border *
            if ( (wk.mevent.meType == metSW_D) ||
                 ((wk.mevent.meType == metB1_S || wk.mevent.meType == metB1_D)
                  && (wk.mevent.ypos == btop)) )
            { wk.type = wktFUNKEY ; wk.key  = nckUP ; }

            //* ScrollWheel downward or click in bottom border *
            else if ( (wk.mevent.meType == metSW_U) ||
                      ((wk.mevent.meType == metB1_S || wk.mevent.meType == metB1_D)
                       && (wk.mevent.ypos == bbot)) )
            { wk.type = wktFUNKEY ; wk.key = nckDOWN ; }

            //* Single-click and double-click events on the left *
            //* and right borders shift the data left and right. *
            else if ( ((wk.mevent.meType == metB1_S || wk.mevent.meType == metB1_D))
                      && (wk.mevent.xpos == blef) )
            { wk.type = wktFUNKEY ; wk.key  = nckLEFT ; }
            else if ( ((wk.mevent.meType == metB1_S || wk.mevent.meType == metB1_D))
                      && (wk.mevent.xpos == brgt) )
            { wk.type = wktFUNKEY ; wk.key = nckRIGHT ; }

            //* Single-click and double-click events inside the *
            //* scrolling area yield selection of a new target  *
            //* directory (or a move to parent directory).      *
            else if ( ((wk.mevent.ypos > btop) && (wk.mevent.ypos < bbot) &&
                       (wk.mevent.xpos > blef) && (wk.mevent.xpos < brgt))
                      &&
                      ((wk.mevent.meType == metB1_S) || (wk.mevent.meType == metB1_D)) )
            {
               //* Calculate the new position for the highlight *
               //* and push the ENTER key into the input stream.*
               dftd.curr = dftd.first + UINT(wk.mevent.ypos - btop - 1) ;
               wk.type = wktFUNKEY ; wk.key = nckENTER ;
               dp->UngetKeyInput ( wk ) ;
               redraw = true ;
               continue ;
            }
         }
      }     // mouse input

      if ( dp->IsScrollKey ( wk ) )    // scroll through the data
      {
         prevCurr = dftd.curr ;
         short prevLshift = dftd.lshift ;
         switch ( wk.key )
         {
            case nckDOWN:
               if ( dftd.curr < dftd.last )
               {
                  ++dftd.curr ;
                  himove = true ;
               }
               else if ( dftd.last < (dftd.total - 1) )
               {
                  ++dftd.first ;
                  ++dftd.last ;
                  ++dftd.curr ;
               }
               else
                  dp->UserAlert () ;
               break ;
            case nckUP:
               if ( dftd.curr > dftd.first )
               {
                  --dftd.curr ;
                  himove = true ;
               }
               else if ( dftd.first > ZERO )
               {
                  --dftd.first ;
                  --dftd.last ;
                  --dftd.curr ;
               }
               else
                  dp->UserAlert () ;
               break ;
            case nckPGDOWN:
               if ( dftd.curr < dftd.last )
               {
                  dftd.curr = dftd.last ;
                  himove = true ;
               }
               else if ( dftd.last < (dftd.total - 1) )
               {
                  dftd.last = dftd.curr = 
                     ((dftd.last + dftd.rows) < dftd.total) ? 
                      (dftd.last + dftd.rows) : (dftd.total - 1) ;
                  dftd.first = dftd.last - dftd.rows + 1 ;
               }
               else
                  dp->UserAlert () ;
               break ;
            case nckPGUP:
               if ( dftd.curr > dftd.first )
               {
                  dftd.curr = dftd.first ;
                  himove = true ;
               }
               else if ( dftd.first > ZERO )
               {
                  dftd.first = dftd.curr = 
                   (dftd.first <= dftd.rows) ? ZERO : (dftd.first - dftd.rows) ;
                  dftd.last = dftd.first + dftd.rows - 1 ;
               }
               else
                  dp->UserAlert () ;
               break ;
            case nckHOME:
               if ( dftd.curr > ZERO || dftd.lshift > ZERO )
               {
                  if ( (dftd.first == ZERO) && (dftd.lshift == ZERO) )
                     himove = true ;
                  dftd.first = dftd.curr = ZERO ;
                  dftd.last  = ((dftd.total < dftd.rows) ?
                                 dftd.total : dftd.rows) - 1 ;
                  dftd.lshift = ZERO ;
               }
               else
                  dp->UserAlert () ;
               break ;
            case nckEND:
               if ( dftd.curr < (dftd.total - 1) || dftd.lshift > ZERO )
               {
                  if ( (dftd.last == (dftd.total - 1)) && (dftd.lshift == ZERO) )
                     himove = true ;
                  dftd.last = dftd.curr = dftd.total - 1 ;
                  dftd.first = (dftd.total <= dftd.rows) ? 
                                ZERO : (dftd.last - dftd.rows + 1) ;
                  dftd.lshift = ZERO ;
               }
               else
                  dp->UserAlert () ;
               break ;
            case nckRIGHT:
               if ( ((dftd.width - dftd.lshift) > dftd.cols) &&
                    (dftd.lshift < (dftd.levels * 2)) )
               { dftd.lshift += 2 ; }
               break ;
            case nckLEFT:
               if ( dftd.lshift > ZERO )
               {
                  dftd.lshift -= 2 ;
                  if ( dftd.lshift < ZERO )  dftd.lshift = ZERO ; // (unlikely)
               }
               break ;
         }
         if ( !himove && ((dftd.curr != prevCurr) || (dftd.lshift != prevLshift)) )
            redraw = true ;
      }

      //* Request to display tree for highlighted child directory. *
      else if ( (wk.type == wktFUNKEY) && ((wk.key == nckC_D) || (wk.key == nckADOWN)) )
      {
         if ( gsPath != oldcwd )
         {
            newcwd = gsPath ; // return path for currently highlighted subdirectory
            status = ucNEW_BASE ;
            done = true ;
         }
         else                 // attempt to re-read the currently-displayed data
            dp->UserAlert () ;
      }

      //* Request to display tree for parent directory. *
      else if ( (wk.type == wktFUNKEY) && ((wk.key == nckC_U) || (wk.key == nckAUP)) )
      {
         if ( (oldcwd.compare( ROOT_PATH )) != ZERO )
         {
            this->fmPtr->ExtractPathname ( newcwd, oldcwd ) ;
            status = ucNEW_BASE ;
            done = true ;
         }
         else  // already at root of directory tree
            dp->UserAlert () ;
      }

      //* Request to navigate to parent directory *
      //* (if not already at root) and continue.  *
      else if ( ((oldcwd.compare( ROOT_PATH )) != ZERO) && 
                 (wk.type == wktFUNKEY) && (((dftd.curr == ZERO) &&
                 ((wk.key == nckENTER) || (wk.key == nckpENTER)))) )
      {
         this->fmPtr->ExtractPathname ( newcwd, oldcwd ) ;
         status = ucNEW_BASE ;
         done = true ;
      }

      //* Navigation to root directory requested.   *
      //* If already at root, enable recursive scan.*
      //* (This is disabled as too dangerous.)      *
      else if ( (wk.type == wktFUNKEY) && (wk.key == nckC_R) )
      {
         newcwd = ROOT_PATH ;             // base target
         #if 0    // Enable/Disable recursive scan from root.
         if ( ((oldcwd.compare( ROOT_PATH )) == ZERO) )
            status = ucDEEP_SCAN ;
         else
         #endif   // Enable/Disable recursive scan from root.
            status = ucNEW_BASE ;
         done = true ;
      }

      //* Request to navigate to specified (lower-level) directory *
      //* This selection will cause exit from Tree-View mode.      *
      else if ( (wk.type == wktFUNKEY && 
                 (wk.key == nckENTER || wk.key == nckpENTER)) || 
                (wk.type == wktEXTEND && wk.key == nckA_Q) )
      {
         newcwd = gsPath ;
         status = ucNEW_CWD ;
         done = true ;
      }

      //* Exit Tree-View mode without making a selection *
      //*   (returned path will be ignored by caller)    *
      else if ( (wk.type == wktFUNKEY) && 
              ((wk.key == nckC_Q) || (wk.key == nckESC) || (wk.key == nckC_X)) )
      {
         if ( wk.key == nckC_X )
            status = ucEXIT_APP ;
         else
            status = ucNO_SEL ;
         done = true ;
      }

      //* Display filesystem info for highlighted directory name.*
      else if ( wk.type == wktFUNKEY && wk.key == nckC_Y )
      {
         this->DisplayFilesystemInfo ( gsPath.ustr() ) ;
      }

      //* Search the list for a filename matching user's criteria.*
      else if ( ((wk.type == wktFUNKEY) && ((wk.key == nckC_F) || (wk.key == nckC_G)))
              ||
              ((wk.type == wktPRINT) && ((wk.key == fSLASH) || (wk.key == L'n'))) )
      {  //* Duplicate 'less' search key style *
         if ( wk.key == fSLASH )  wk.key = nckC_F ;
         else if ( wk.key == L'n' )  wk.key = nckC_G ;

         winPos spPos( (firstDLine + 1), (cols / 2) ) ;
         bool newSearch = false ;
         if ( wk.key == nckC_F || sPattern.gschars() == 1 )
         {  //* Ask user for search pattern, and search for a match *
            dp->SetDialogObscured () ;
            if ( (newSearch = this->dftNamePrompt ( spPos, sPattern )) != false )
            {
               sIndex = ZERO ;
               sMatch = ZERO ;
               newSearch = true ;
            }
            dp->RefreshWin () ;
         }
         if ( newSearch || wk.key == nckC_G ) // 'find' OR 'find again'
         {
            if ( dftd.curr < sIndex )
               sIndex = ZERO ;
            else if ( !newSearch && (sIndex < (dftd.total-2)) )
               ++sIndex ;
            if ( (sIndex < (dftd.total-1)) &&
                 (dftNameSearch ( dftd, sPattern, sIndex )) != false )
            {
               dftd.curr = sIndex ;    // new highlighted item
               //* Scroll the item into view *
               while ( dftd.curr < dftd.first && dftd.first > ZERO )
               {
                  --dftd.first ;
                  --dftd.last ;
               }
               while ( dftd.curr > dftd.last && dftd.last < (dftd.total-1) )
               {
                  ++dftd.first ;
                  ++dftd.last ;
               }
               sMatch = 1 ;
               redraw = true ;
            }
            else
            {  //* Report no-match *
               dp->SetDialogObscured () ;
               dftNoMatch ( spPos, noMatchMsg[sMatch], this->cs.sb ) ;
               dp->RefreshWin () ;
               sIndex = ZERO ;   // force next search to top of list
            }
         }
      }

      //* Invoke context help for Tree View *
      else if ( (wk.type == wktFUNKEY) && 
                ((wk.key == nckSF01) || (wk.key == nckC_H)) )
      {
         dp->SetDialogObscured () ;    // save the dialog display data
         this->CallContextHelp ( cxtTREE_VIEW ) ;
         dp->RefreshWin () ;           // restore dialog display data
      }

      //* Call user a dumb-guy *
      else
         dp->UserAlert () ;

      #if 0    // Potential future enhancements
      else if ( (wk.type == wktFUNKEY && wk.key == nckC_C) )
      {  //* Collapse current node of the tree view - hide lower branches *
      }
      else if ( (wk.type == wktFUNKEY && wk.key == nckC_E) )
      {  //* Expand current node of the tree view - show lower branches *
      }
      else if ( (wk.type == wktFUNKEY && wk.key == nckAC_C) )
      {  //* Collapse the entire tree view - show only top two levels *
      }
      else if ( (wk.type == wktFUNKEY && wk.key == nckAC_E) )
      {  //* Expand the entire tree view - show all branches *
      }
      #endif   // U/C
   }

   return status ;

}  //* End dftDisplayTree() *

//*************************
//*    dftFormatTree      *
//*************************
//********************************************************************************
//* Create a two-dimensional character array and format the directory            *
//* information for display.                                                     *
//*                                                                              *
//* Input  : dftd    : (by reference) initialized display control data           *
//*                    dftd.data[] receives the formatted display/index data     *
//*          baseNode: base node of source directory tree                        *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* Programmer's Note: Formatting and display of the data uses a two-column      *
//* indentation for each successive directory-tree level AND note that           *
//* formatting of the data will fail (low-level branches may not be fully        *
//* visible) if any string is longer than dftITEM_LENGTH.                        *
//*                                                                              *
//********************************************************************************

void FileDlg::dftFormatTree ( dftDisplay& dftd, TreeNode* baseNode )
{
   gString gt ;         // for formatting target node's name for display
   UINT di = ZERO ;     // index into top-level subdirectories

   // If there is a parent directory (basePath != root) *
   gt = baseNode->dirStat.fName ;
   if ( (gt.compare( ROOT_PATH )) != ZERO )
   {
      gt = parentDir ;
      gt.copy( dftd.data[di++].n, dftITEM_LEN ) ;
      dftd.width = gt.gscols() ; // set baseline column count
   }

   dftd.curr = di ;              // initially-highlighted item

   //* Base Directory Name: If directory name is tagged as no-read-access *
   //* or external-filesystem. Note that there will be no more than one   *
   //* indicator per line item with 'noAccess' having precedence.         *
   gt = baseNode->dirStat.fName ;
   if ( ! baseNode->dirStat.readAcc )
      dftMarkRestricted ( gt ) ;
   else if ( ! baseNode->dirStat.fsMatch )
      dftMarkExternal ( gt ) ;
   gt.copy( dftd.data[di++].n, dftITEM_LEN ) ;
   if ( gt.gscols() > dftd.width )  dftd.width = gt.gscols() ;

   //* Recursively initialize the display data *
   di += this->dftFormatNode ( dftd, baseNode, di ) ;

   //* After the records have been sorted, there may be a number of empty  *
   //* records at the end of the display list. This is especially (only?)  *
   //* true when the scan started at the root directory. Truncate the list *
   //* to remove these empty records. Where do these records come from?    *
   //* It would be better if these records were not created at all.        *
   UINT i = dftd.total - 1 ;
   while ( dftd.data[i].n[0] == NULLCHAR )
   { --dftd.total ; --i ; }

   //* Define initial block of items to display.*
   dftd.first = ZERO ;
   dftd.last  = (dftd.total < dftd.rows ? dftd.total : dftd.rows) - 1 ;
   dftd.lshift = ZERO ;

}  //* End dftFormatTree() *

//*************************
//*     dftFormatNode     *
//*************************
//********************************************************************************
//* Format for display the specified node on the tree.                           *
//* Caller has verified that there is at least one subdirectory attached         *
//* to this node.                                                                *
//*                  NOTE: This is a recursive method.                           *
//*                                                                              *
//* Input  : dftd  : (by reference) initialized display control data             *
//*                  dftd.data[] receives the formatted display and index data   *
//*          tnPtr : pointer to array of TreeNode objects to be processed        *
//*          di    : index of next free item in dftd.data[] array                *
//*                                                                              *
//* Returns: next free item in dftd.data[] array                                 *
//********************************************************************************

UINT FileDlg::dftFormatNode ( dftDisplay& dftd, TreeNode* tnPtr, UINT di )
{
   gString gt ;
   UINT parentIndex = di - 1, 
        actualNodes, actualFiles ;

   tnPtr->GetActual ( actualNodes, actualFiles ) ;

   for ( UINT ni = ZERO ; (ni < actualNodes) && (di < dftd.total) ; ++ni )
   {
      //* Sort the TreeNode array alphabetically *
      dftSortNode ( tnPtr->nextLevel, actualNodes ) ;

      //* Format the directory name *
      gt.compose( "%s", tnPtr->nextLevel[ni].dirStat.fName ) ;
      gt.shiftCols( tnPtr->nextLevel[ni].level * 2 ) ;

      //* If restricted-access directory *
      if ( ! tnPtr->nextLevel[ni].dirStat.readAcc )
         dftMarkRestricted ( gt ) ;    // mark directory as read-only

      //* If mountpoint for external filesystem *
      else if ( ! tnPtr->nextLevel[ni].dirStat.fsMatch )
         dftMarkExternal ( gt ) ;      // mark directory as external filesystem

      //* Save the formatted entry *
      gt.copy( dftd.data[di].n, dftITEM_LEN ) ;
      dftd.data[di].p = parentIndex ;
      dftd.data[di++].l = tnPtr->nextLevel[ni].level ;

      //* Track the widest entry *
      if ( gt.gscols() > dftd.width )  dftd.width = gt.gscols() ;

      //* Process lower levels of the tree *
      if ( tnPtr->nextLevel[ni].nextLevel != NULL )
         di = this->dftFormatNode ( dftd, &tnPtr->nextLevel[ni], di ) ;

      //* Track the depth of the tree *
      if ( (ni == ZERO) && (tnPtr->nextLevel[ni].level > dftd.levels) )
         dftd.levels = tnPtr->nextLevel[ni].level ;
   }

   return di ;       // index of next free display item

}  //* End dftFormatNode() *

//*************************
//*     dftNamePrompt     *
//*************************
//********************************************************************************
//* Ask user for all or part of the subdirectory name for which to search.       *
//*                                                                              *
//* Input  : spPos : (by reference) position of dialog window                    *
//*          ptrn  : (by reference) receives the user's search-pattern text      *
//*                                                                              *
//* Returns: 'true'  if search pattern entered, else 'false'                     *
//********************************************************************************

bool FileDlg::dftNamePrompt ( const winPos& spPos, gString& ptrn )
{
   enum dCtrls : short { patTB, okPB, canPB, controlsDEFINED } ;
   const short dialogROWS = 8,
               dialogCOLS = 48,
               dialogULY  = spPos.ypos,
               dialogULX  = spPos.xpos - dialogCOLS / 2 ;
   const attr_t dColor = this->cs.sb ;
   InitCtrl ic[controlsDEFINED] =      // array of dialog control info
   {
      {  //* 'Pattern" Text Box  - - - - - - - - - - - - - - - - - - -   patTB *
         dctTEXTBOX,                   // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         3,                            // ulY:       upper left corner in Y
         2,                            // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         short(dialogCOLS - 4),        // cols:      control columns
         ptrn.ustr(),                  // dispText:  
         this->cs.tn,                  // nColor:    non-focus color
         this->cs.tf,                  // fColor:    focus color
         #if LINUX_SPECIAL_CHARS != 0
         tbFileLinux,                  // filter: valid filename chars (incl. Linux "special")
         #else    // BASIC FILENAME FILTER
         tbFileName,                   // filter:    valid filename characters
         #endif   // BASIC FILENAME FILTER
         "Enter all or part of subdirectory name:", // label:     
         -1,                           // labY:      
         ZERO,                         // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[okPB],                    // nextCtrl:  link in next structure
      },
      {  //* 'OK' pushbutton - - - - - - - - - - - - - - - - - - - - -    okPB *
         dctPUSHBUTTON,                // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         short(dialogROWS - 2),        // ulY:       upper left corner in Y
         short(dialogCOLS / 2 - 11),   // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         8,                            // cols:      control columns
         "   OK   ",                   // dispText:  
         this->cs.pn,                  // nColor:    non-focus color
         this->cs.pf,                  // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "",                           // label:     (n/a)
         ZERO,                         // labY:      (n/a)
         ZERO,                         // labX       (n/a)
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[canPB],                   // nextCtrl:  link in next structure
      },
      {  //* 'Cancel' pushbutton - - - - - - - - - - - - - - - - - -   cancelPush *
         dctPUSHBUTTON,                // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         ic[okPB].ulY,                 // ulY:       upper left corner in Y
         short(ic[okPB].ulX + 11),     // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         8,                            // cols:      control columns
         " CANCEL ",                   // dispText:  
         this->cs.pn,                  // nColor:    non-focus color
         this->cs.pf,                  // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "",                           // label:     (n/a)
         ZERO,                         // labY:      (n/a)
         ZERO,                         // labX       (n/a)
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         NULL,                         // nextCtrl:  link in next structure
      },
   } ;

   bool search = false ;            // return value

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       dialogULY,      // Y offset from upper-left of terminal 
                       dialogULX,      // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      dp->SetDialogTitle ( "  NAME SEARCH  ", this->cs.em ) ;
      dp->WriteString ( ic[patTB].ulY + 1, ic[patTB].ulX + 7,
                        "(search IS NOT case sensitive)", dColor ) ;
      dp->RefreshWin () ;                    // make the text visible

      uiInfo   Info ;                     // user interface data returned here
      short    icIndex = ZERO ;           // index of control with input focus
      bool     done = false ;             // loop control
      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == okPB )
               {
                  if ( (dp->GetTextboxText ( patTB, ptrn )) > 1 )
                  {
                     search = true ;
                     done = true ;
                  }
                  else
                  {
                     dp->ClearLine ( 5, false ) ;
                     dp->WriteString ( 5, (dialogCOLS / 2 - 21), 
                        " No directory name specified! - Try again. ", 
                        dColor | ncbATTR, true ) ;
                     dp->NextControl () ;
                  }
               }
               else     // 'Cancel'
               {
// Do not erase previous pattern text.
//                  ptrn.clear() ;
                  search = false ;
                  done = true ;
               }
            }
         }
         else if ( ic[icIndex].type == dctTEXTBOX )
         {
            Info.viaHotkey  = false ;     // ignore hotkey data
            icIndex = dp->EditTextbox ( Info ) ;
            dp->ClearLine ( 5, false ) ;
         }
         //* Move focus to appropriate control *
         if ( !done && !Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }     // while(!done)
   }
   if ( dp != NULL )
      delete ( dp ) ;                        // close the window
   return search ;

}  //* End dftNamePrompt() *


//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *
//* - - - - - - - - -  Context-help Invocation Group  - - - - - - - - - - - -  *
//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *

//*************************
//*    CallContextHelp    *
//*************************
//******************************************************************************
//* Invoke the 'Info' system to display contextual help information.           *
//*                                                                            *
//* -- This method uses the NcDialog-class 'ShellOut' method                   *
//* -- The ShellOut() method saves the application dialog (if caller hasn't    *
//*    already saved it).                                                      *
//* -- Note that after returning from the shell the application dialog is      *
//*    refreshed, which releases the saved display data; so if caller has set  *
//*    the application dialog as 'obscured', then caller must save it AGAIN    *
//*    on return.                                                              *
//* -- Since we don't know about any sub-dialogs that may be open, it is       *
//*    assumed that caller has saved the display data for those dialogs and    *
//*    will restore them on return.                                            *
//*                                                                            *
//* Example sequence:                                                          *
//*   this->dPtr->SetDialogObscured () ;                                       *
//*     .  .  . (do stuff) .  .  .                                             *
//*   dp->SetDialogObscured () ;                                               *
//*   this->CallContextHelp ( cxtMOD_STATS ) ;                                 *
//*   this->dPtr->SetDialogObscured () ;                                       *
//*   dp->RefreshWin () ;                                                      *
//*                                                                            *
//*                                                                            *
//* Input  : ccode  : member of contextCode indicating the context of call     *
//*                                                                            *
//* Returns:  OK if successful, ERR if error                                   *
//******************************************************************************
//* Programmer's Notes:                                                        *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//******************************************************************************

short FileDlg::CallContextHelp ( contextCode ccode )
{
   //* For development only. The development version of the application *
   //* is not in the installation directory, so compensate.             *
   //*         REMEMBER TO DISABLE THIS BEFORE POSTING A RELEASE.       *
   #define DEBUG_CallContextHelp (0)

   short status = OK ;        // return value

   gString  helpPath( "%s/filemangler.info", this->cfgOptions.appPath ),
            shellCmd ;         // construct the command

   #if DEBUG_CallContextHelp != 0
   helpPath.insert( "/Texinfo", helpPath.findlast( fSLASH ) ) ;
   #endif   // DEBUG_CallContextHelp

   //* Be sure the help file exists *
   fmFType ft ;
   if ( (this->TargetExists ( helpPath, ft )) != false )
   {
      if ( ccode == cxtMAIN )
         shellCmd.compose( "info -f %S", helpPath.gstr() ) ;
      else if ( ccode == cxtMOD_STATS )
         shellCmd.compose( "info -f %S -n 'modify stats'", helpPath.gstr() ) ;
      else if ( ccode == cxtARCHIVE_C )
         shellCmd.compose( "info -f %S -n 'Creating an Archive'", helpPath.gstr() ) ;
      else if ( ccode == cxtARCHIVE_U )
         shellCmd.compose( "info -f %S -n 'Updating an Archive'", helpPath.gstr() ) ;
      else if ( ccode == cxtARCHIVE_E )
         shellCmd.compose( "info -f %S -n 'Expanding an Archive'", helpPath.gstr() ) ;
      else if ( ccode == cxtTREE_VIEW )
         shellCmd.compose( "info -f %S -n 'Tree-view Mode'", helpPath.gstr() ) ;
      else                       // invalid option
         status = ERR ;
   }
   else
      status = ERR ;

   if ( status == OK )
      this->dPtr->ShellOut ( soX, shellCmd.ustr() ) ;

   if ( status != OK )
   {  //* Write a warning to the status window *
      dtbmData msgData = "  Unable to access the help file.  " ;
      this->dPtr->DisplayTextboxMessage ( this->mIndex, msgData ) ;
   }
   return status ;

}  //* End CallContextHelp() *

//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *
//* - - - - - - - - -   Non-member Methods  - - - - - - - - - - - - - - - - -  *
//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *

//*************************
//*      dftSortNode      *
//*************************
//******************************************************************************
//* Sort the items in specified TreeNode array.                                *
//*                                                                            *
//* Input  : tnPtr  : pointer to array of TreeNode objects                     *
//*          count  : number of items in the array                             *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

int snq ( const TreeNode* itemA, const TreeNode* itemB )
{  //* qsort callback function *
   gString A( itemA->dirStat.fName ),
           B( itemB->dirStat.fName ) ;
   int compared = A.compare( B.gstr() ) ;
   return compared ;
}

static void dftSortNode ( TreeNode* tnPtr, UINT count )
{
   if ( count > 1 )
   {
      //* Sort the array items by directory name *
      qsort ( tnPtr, count, sizeof(TreeNode), (comparison_fn_t)snq ) ;
   }
}  //* End dftSortNode() *

//*  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  - *
//*  -  -  -  -  Non-member methods for DisplayFileTree group. -  -  -  -  -   *
//*  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  - *

//*************************
//*   dftMarkRestricted   *
//*************************
//******************************************************************************
//* If read access to named directory is restricted, then append an indicator. *
//*                                                                            *
//* Input  : dName : (by reference) display data including directory name      *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void dftMarkRestricted ( gString& dName )
{

   dName.append( noAccess ) ;

}  //* End dftMarkRestricted)() *

//*************************
//*    dftMarkExternal    *
//*************************
//******************************************************************************
//* If named directory is on a different filesystem from its parent, then      *
//* append an indicator.                                                       *
//*                                                                            *
//* Input  : dName : (by reference) display data including directory name      *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void dftMarkExternal ( gString& dName )
{

   dName.append( L"%S", extFileSys ) ;

}  //* End dftMarkExternal() *

//*************************
//*     dftNameSearch     *
//*************************
//******************************************************************************
//* Search the list of subdirectory names for a match to caller's string.      *
//* Search is CASE INSENSITIVE.                                                *
//*                                                                            *
//* Input  : dftd  : (by reference) initialized display control data           *
//*          ptrn  : text pattern for which to search                          *
//*          fIndex: index of first record to search (search proceeds downward)*
//*                                                                            *
//* Returns: 'true' if a matching entry found                                  *
//*            fIndex will reference matched record                            *
//*          else, 'false' (fIndex unchanged)                                  *
//******************************************************************************

static bool dftNameSearch ( const dftDisplay& dftd, const gString ptrn, UINT& fIndex )
{
   gString gsItem ;
   bool matchFound = false ;     // return value

   for ( UINT di = fIndex ; !matchFound && di < dftd.total ; ++di )
   {
      gsItem = dftd.data[di].n ;
      if ( (gsItem.find( ptrn.gstr() )) >= ZERO )
      {
         fIndex = di ;
         matchFound = true ;
      }
   }
   return matchFound ;

}  //* End dftNameSearch() *

//*************************
//*      dftNoMatch       *
//*************************
//******************************************************************************
//* Reports that no matching sufdirectory name (or no additional names) found. *
//*                                                                            *
//* Input  : spPos : (by reference) position of dialog window                  *
//*          msg   : message to be displayed                                   *
//*          dColor: background color of dialog window                         *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void dftNoMatch ( const winPos& spPos, const char* msg, attr_t dColor )
{
   const short dialogROWS = 3,
               dialogCOLS = 44,
               dialogULY  = spPos.ypos,
               dialogULX  = spPos.xpos - dialogCOLS / 2 ;
   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       dialogULY,      // Y offset from upper-left of terminal 
                       dialogULX,      // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       NULL            // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      dp->WriteString ( 1, 2, msg, dColor, true ) ;
      chrono::duration<short, std::milli>aMoment( 1600 ) ;
      this_thread::sleep_for( aMoment ) ;
   }
   if ( dp != NULL )
      delete ( dp ) ;                        // close the window

}  //* End dftNoMatch

//*************************
//*   dtfFormatStatMsg    *
//*************************
//********************************************************************************
//* Non-member Method:                                                           *
//* ------------------                                                           *
//* Format text for display in the statTB control.                               *
//*                                                                              *
//* When base directory is not root directory, display the total numuber of      *
//* records (directory names) displayed and the number of non-directory files    *
//* contained in those directories.                                              *
//*                                                                              *
//* For root directory only, there is no recursive scan, so report the number of *
//* directories and non-directory files at the top level.                        *
//*                                                                              *
//* Input  : dtbmd   : (by reference) receives formatted text data               *
//*          dirFiles: total number of records (directory names) captured        *
//*          totFiles: total number of files scanned (dir + non-dir files)       *
//*          isRoot  : set if tree base is root directory, else reset            *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

static void dtfFormatStatMsg ( dtbmData& statMsg, UINT dirFiles, UINT totFiles, bool isRoot )
{
   gString gsDir( dirFiles, 6, true ) ;
   gString gs, gstmp ;
   if ( isRoot )
   {
      UINT ndirFiles = totFiles - dirFiles ;
      gstmp.compose( "Root: %S Directories + %u Files", gsDir.gstr(), &ndirFiles ) ;
   }
   else
   {
      gs.formatInt( totFiles, 6, true ) ;
      gstmp.compose( "Directories:%S Total Files:%S", gsDir.gstr(), gs.gstr() ) ;
   }
   gs.compose( statTemplate, gstmp.ustr() ) ;
   statMsg = gs.ustr() ;
}  //* End dtfFormatStatMsg() *

//*************************
//*                       *
//*************************
//********************************************************************************
//*                                                                              *
//*                                                                              *
//*                                                                              *
//* Input  :                                                                     *
//*                                                                              *
//* Returns:                                                                     *
//********************************************************************************

