//********************************************************************************
//* File       : FmBackup.cpp                                                    *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2005-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in FileMangler.hpp         *
//* Date       : 12-May-2025                                                     *
//* Version    : (see AppVersion string)                                         *
//*                                                                              *
//* Description: This module of the FileMangler class contains the user          *
//* interface and control code for performing file 'backup', 'Archive' and       *
//* 'synchronize' operations.                                                    *
//*                                                                              *
//* Development tools: See FileMangler.cpp header.                               *
//********************************************************************************
//* Version History (most recent first):                                         *
//*   See version history in FileDlg.cpp.                                        *
//********************************************************************************
//* Programmer's Notes:                                                          *
//*                                                                              *
//* Operations Performed:                                                        *
//* ---------------------                                                        *
//* Three types of operation may be performed:                                   *
//*  1) Backup:                                                                  *
//*     incremental backup from source to target for all 'selected' files        *
//*     and subdirectory trees.                                                  *
//*  2) Archive:                                                                 *
//*     create a 'tar' archive (optionally compressed) from source to target     *
//*     for all 'selected' files and subdirectory trees.                         *
//*  3) Synchronize:                                                             *
//*     synchronize all data in both 'source' and 'target' directories.          *
//*     This is essentially a bidirectional incremental backup, except that      *
//*     ALL data are automatically 'selected'.                                   *
//*                                                                              *
//* File Types:                                                                  *
//* -----------                                                                  *
//* Supported filetypes are 'Regular', 'Directory', 'FIFO' and 'Symlink'.        *
//*   - Note that symlinks ARE NOT followed.                                     *
//*   - Note also that copying a Linux symlink to a FAT filesystem will fail.    *
//* Other filetypes in source data will be silently ignored; however, if         *
//* 'special' files ('Character Device', 'Block Device', 'Socket' or others)     *
//* are encountered, it will be noted in the logfile so if desired, user can     *
//* manually backup files of these types.                                        *
//*                                                                              *
//* Backup Set:                                                                  *
//* -----------                                                                  *
//* A Backup Set is a definition for the operation to be performed.              *
//* User can specify a file containing the definition, or the definition can     *
//* be interactively defined in the setup dialog.                                *
//*                                                                              *
//* 1) Internally the backup set is defined in a BSet-class object.              *
//* 2) A definition file is laid out as shown in the example.                    *
//*    a) All record lines are optional; however, if specified, the record       *
//*       must be valid.                                                         *
//*    b) Comment lines begin with the '#' (hash) character.                     *
//*    c) Blank lines are ignored.                                               *
//*                                                                              *
//*    # FileMangler Backup, Archive, Synch Definition File                      *
//*    SOURCEDIR: ~/Documents                                                    *
//*    TARGETDIR: ${MOUNT_DIR}/TravelDrive/Documents                             *
//*    OPBEGIN  : 18:30                                                          *
//*    OPTYPE   : Backup                                                         *
//*    ARCHIVE  :                                                                *
//*    LOGFILE  :                                                                *
//*    EXCLUDE  : .o                                                             *
//*    EXCLUDE  : ~                                                              *
//*                                                                              *
//* See the Backup_Definition() method for details.                              *
//*                                                                              *
//********************************************************************************


//* Header Files *
#include "FileMangler.hpp"    // FileMangler definitions and data

//*******************************
//* Local Definitions and Data. *
//*******************************

//* Prompt dialog *
enum bkControls : short
{
   okPB = ZERO, canPB, hhSP, mmSP,
   arcRB, arcTB, excRB, excTB, scnRB,
   logDD, bkCONTROLS
} ;

//* Callback method access to necessary data *
static NcDialog*  bkDlg = NULL ;
static localTime const * bkDlgLt = NULL ;
static attr_t     bkDlgColor ;
static winPos     bkDlgWp ;

//* Range for Spinner controls *
const int minHRSP = 0,
          maxHRSP = 744,      // number of hours in 31 days
          minMNSP = 0,
          maxMNSP = 59 ;      // minutes per hour (minus 1)

static const char* opName[] = 
{
   "Backup",
   "Synch",
   "Archive",
   "Scan-Only",
} ;

static const char* dowStr[] =    // for construction of timestamps
{
   " Sunday",
   " Monday",
   " Tuesday",
   " Wednesday",
   " Thursday",
   " Friday",
   " Saturday"
} ;


//*************************
//* Non-member Prototypes *
//*************************

static short Backup_Callback ( short cIndex, wkeyCode wkey, bool firstTime ) ;
static void  bcEncodeLocalTime ( localTime& ltime, gString& tstamp ) ;

//* Debugging only: Display contents of BSet object. *
#if ENABLE_DEBUGGING_CODE != 0
#define DEBUG_BDP (0)
#if DEBUG_BDP != 0
void Display_BSet ( short dialogROWS, short dialogCOLS,
                    short dialogYPOS, short dialogXPOS,
                    const BSet& bkSet, attr_t dColor, short status = ZERO ) ;
#define BDP_PRELIM (0)
#define BDP_FINAL  (0)
#endif   // DEBUG_BDP
#endif   // ENABLE_DEBUGGING_CODE


//*************************
//*        Backup         *
//*************************
//******************************************************************************
//* Initialize and validate backup/archive/synchronize options.                *
//*                                                                            *
//* Important Note: This method assumes that we are in Dual-window Mode.       *
//*                 Do not call this method while in Single-window Mode.       *
//*                                                                            *
//* Input  : synch  : 'false' if 'Backup' operation (but see below)            *
//*                   'true'  if 'Synch' operation                             *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note: Setting up a backup requires access to BOTH of the      *
//* application's FileDlg objects. For this reason, we do a portion of the     *
//* heavy lifting in the FileMangler class although we would prefer that       *
//* the FileMangler class remain as stupid as possible.                        *
//*                                                                            *
//* Notes:                                                                     *
//* ======                                                                     *
//*                                                                            *
//* -- The basic mechanism:                                                    *
//*    - The source data are the data 'selected' by the user before invoking   *
//*      the operation, OR if user has selected no data, then ALL data in the  *
//*      active window will be automatically selected.                         *
//*    - If the target doesn't exist OR                                        *
//*      if target is older than source, then source will be copied to target. *
//*    - For synch operations, this test goes both directions.                 *
//*    - All 'selected' top-level files and directories (and the contents of   *
//*      of the directory trees will be scanned and processed recursively.     *
//*                                                                            *
//* -- If user specified a definition file on the command line, take initial   *
//*    settings from that file. Else:                                          *
//*    - Use the currently-displayed directories as source and target          *
//*      directories.                                                          *
//*    - If the default definition file exists ("Backup_Def.txt" in the        *
//*      application directory) AND if that file contains exclusion records,   *
//*      then take the exclusion records from that file. Otherwise, no         *
//*      exclusions specified.                                                 *
//*    - Set all other parameters to default values.                           *
//*                                                                            *
//* -- User must have read access to both source and target directories.       *
//*                                                                            *
//* -- User must have write access to the target directory, and if a synch     *
//*    operation, must also have write access to the 'source' directory.       *
//*    Note: We display "read only" for either read- or write-access errors    *
//*    because even users aren't dumb enough to try backing up a directory to  *
//*    which he/she/it does not have read access.                              *
//*    -- Exception: If the "Scan Only" radiobutton is set, then write access  *
//*       to target is not required.                                           *
//*                                                                            *
//* -- If the target data are to be written as an archive file, then the name  *
//*    of the archive file must follow the rules of the 'tar' utility, or the  *
//*    'zip' utility, AND must not indicate an existing file. The only         *
//*    exception is for adding data to an existing _uncompressed_ archive.     *
//*    a) If user specified an archive filename, use it.                       *
//*    b) Else archive filename will be created according to 'arcTemplate'     *
//*       (see below).                                                         *
//*    See Backup_Validate() method for details.                               *
//*                                                                            *
//* -- An error log will be compiled and optionally displayed/saved at the     *
//*    end of the operation. The log is created as a temporary file; however,  *
//*    if user wants to save the log after the operation, it will be written   *
//*    to the base target directory using the filename in 'bkSet.blog'.        *
//*    a) If user specified a log filename, use it.                            *
//*    b) Else log filename will be created according to 'logTemplate'         *
//*       (see below).                                                         *
//*                                                                            *
//* -- When the operation terminates, any reference to a definition file       *
//*    specified by the user will be erased (this->cfgOpt.bkPath).             *
//*    This is done because the user may explicity invoke another operation    *
//*    during the same session, and the existing parameters would conflict     *
//*    with user's manual setup.                                               *
//*                                                                            *
//* -- The operation timestamp is created when we enter this method; however   *
//*    if this is a delayed operation, then all data previously initialized    *
//*    using the stale timestamp will also be stale.                           *
//*    1) For Archive operations, bkSet.barc (the archive filename) will be    *
//*       stale, but it already will have been accepted, and possibly modified *
//*       by the user, so we can't change it arbitrarily.                      *
//*    2) bkSet.btime (the actual launch time) will be created with a stale    *
//*       value that is off by the amount of time user needed to set up the    *
//*       operational parameters.                                              *
//*       a) bkSet.btime is updated just before the countdown begins.          *
//*       b) bkSet.blog (the user-space logfile name) will be stale and will   *
//*          need to be updated using bkSet.btime to match the actual launch   *
//*          time.                                                             *
//*                                                                            *
//******************************************************************************

void FileMangler::Backup ( bool synch )
{
   const char *bkTitle = "  Perform Data Backup  ",
              *syTitle = "  Perform Data Synch  ",
              *rdOnly  = " (read only)",
              *arcTemplate = "Backup_%04hd_%02hd_%02hd.tar.bz2",
              *logTemplate = "Backup_%04hd_%02hd_%02hd.log",
              *dfltExList = "Backup_Def.txt" ;
   const short ddITEMS = 5, ddWIDTH = 17 ;
   const char  ddText[ddITEMS][ddWIDTH + 1] = // Text data for logDD
   {  // Programmer's Note: This array must remain synchronized with enum bsetLog.
      " Verbose (all)   ",
      " Updates Only    ",
      " Errors Only     ",
      " Summary Only    ",
      " Debug Selection "
   } ;
   const short dialogROWS = 14,
               dialogCOLS = 90,
               dialogYPOS = this->ic[dwFileSE].ulY + 1,
               dialogXPOS = this->ic[dwFileSE].ulX + this->ic[dwFileSE].cols
                             - (dialogCOLS / 2) - 1 ;

   const attr_t dColor  = this->cfgOpt.cScheme.sd,
                hColor  = this->cfgOpt.cScheme.em,
                gyColor = this->cfgOpt.cScheme.sb & ~ncbATTR ;

   dspinData hspData( minHRSP, maxHRSP, 0, dspinINTEGER, dColor, true, L':' ) ;
   dspinData mspData( minMNSP, maxMNSP, 0,  dspinINTEGER, dColor, true, nckNULLCHAR ) ;

   //* Parameters for operation. Either specified by user, or defaults.*
   BSet bkSet ;
   short bkSetStatus = ERR ;
   bool bkdefWarn = false,       // 'true' if warn of definition-file error
        srAcc, swAcc,            // source permissions
        trAcc, twAcc ;           // target permissions
   localTime lt ;                // formatted timestamp
   this->fdPtr->GetLocalTime ( lt ) ; // get the local time
   lt.epoch -= (lt.epoch % 60) ; // round to even minute
   this->fdPtr->DecodeLocalTime ( lt ) ;

   //* Certain controls may be active or inactive depending *
   //* on the operation being performed and parameters      *
   //* specified in the definition file (if any).           *
   //* These values represent active controls, and may be   *
   //* adjusted before creating the list of control objects.*
   attr_t arcRB_nColor = dColor,
          excRB_nColor = dColor,
          scnRB_nColor = dColor,
          arcTB_nColor = this->cfgOpt.cScheme.tn,
          excTB_nColor = this->cfgOpt.cScheme.tn,
          monoColor[2] = { attrDFLT, this->cfgOpt.cScheme.mn } ; // dropdown data colors
   bool arcRB_Active = true,     // 'Archive' radiobutton active
        arcRB_Select = false,    // 'Archive' radiobutton selected
        excRB_Active = true,     // 'Exclusion' radiobutton active
        excRB_Select = false,    // 'Exclusion' radiobutton selected
        excTB_Active = false,    // 'Exclusion' textbox active
        scnRB_Active = true ;    // 'Scan-only' radiobutton active


   //* If definition file specified, get initial settings.*
   if ( *this->cfgOpt.bkPath != NULLCHAR )
   {
      bkSet.btime = lt ;         // pass a copy of the current time
      if ( (bkSetStatus = this->Backup_Definition ( this->cfgOpt.bkPath, bkSet )) == OK )
      {
         //* Set Source and Target as new window contents. *
         if ( (bkSet.bsrc.gschars()) > 1 )
            bkSetStatus = this->fdPtr->SetDirectory ( bkSet.bsrc ) ;
         else
            this->fdPtr->GetPath ( bkSet.bsrc ) ;
         if ( (bkSetStatus == OK) && ((bkSet.btrg.gschars()) > 1) )
         {
            if ( (bkSetStatus = this->altPtr->SetDirectory ( bkSet.btrg )) == OK )
               this->altPtr->RefreshCurrDir ( false ) ;
         }
         else
            this->altPtr->GetPath ( bkSet.btrg ) ;

         if ( bkSetStatus == OK )
         {  //* Get source and target permissions (tested below). *
            gString gs ;
            this->fdPtr->GetPath ( gs, srAcc, swAcc ) ;
            this->altPtr->GetPath ( gs, trAcc, twAcc ) ;

            //* Setting in definition file overrides our 'synch' parameter.*
            synch = bkSet.btype == bsetSynch ? true : false ;

            //* If Synch operation OR no archive *
            //* filename specified, use default. *
            if ( synch || bkSet.barc.gschars() == 1 )
            {
               bkSet.barc.compose( arcTemplate, &lt.year, &lt.month, &lt.date ) ;
               if ( bkSet.btype == bsetArchive )
                  arcRB_Select = true ;
            }
            else  // !synch && archive target specified
               arcRB_Select = true ;

            //* If no log filespec specified, use default. *
            if ( ! bkSet.blflag )
               bkSet.blog.compose( logTemplate, &lt.year, &lt.month, &lt.date ) ;

            //* If a delayed backup time specified, update initial spinner values.*
            if ( bkSet.bhour > ZERO )
               hspData.iniValue = bkSet.bhour ;
            if ( bkSet.bmin > ZERO )
               mspData.iniValue = bkSet.bmin ;

            //* Definition file verified. If it contains valid *
            //* Exclusion records, set Exclusion Radiobutton   *
            //* to 'selected' and deactivate the button.       *
            //* (Note that for Synch operation, this will be   *
            //*  overridden below.)                            *
            if ( bkSet.bexCount > ZERO )
            {
               excRB_Select = true ;
               excRB_Active = false ;
               excRB_nColor = gyColor ;
               excTB_nColor = nc.bkgy ;
            }
         }
      }
      if ( bkSetStatus != OK )
         bkdefWarn = true ;
   }

   //* Either no definition file specified, or    *
   //* definition-file error. Set default values. *
   if ( bkSetStatus != OK )
   {
      //* a) Operation type determined by caller's 'synch' parameter.       *
      //* b) Source and Target directories (if not previously specified     *
      //*    and validated), will be directories displayed in focus and     *
      //*    non-focus windows, respectively. (user access is tested below) *
      //* c) For Backup only, if the default definition file exists, then   *
      //*    extract the exclusion list from it. If exclusions found, then  *
      //*    set the exclusion radio button, 'excRB' via 'excRB_Select' and *
      //*    enable 'excTB' via 'excTB_Active'.                             *
      //* d) All other members set to default values.                       *
      bkSet.reset( false ) ;     // clear all data _except_ 'bsrc' and 'btrg'
      bkSet.btype = synch ? bsetSynch : bsetBackup ;

      if ( (bkSet.bsrc.gschars()) > 1 )   // set source directory
         this->fdPtr->SetDirectory ( bkSet.bsrc ) ;
      this->fdPtr->GetPath ( bkSet.bsrc, srAcc, swAcc ) ;

      if ( (bkSet.btrg.gschars()) > 1 )   // set target directory
         this->altPtr->SetDirectory ( bkSet.btrg ) ;
      this->altPtr->GetPath ( bkSet.btrg, trAcc, twAcc ) ;

      if ( bkSet.btype == bsetBackup )
      {
         //* Construct filespec for default definition file,   *
         //* and extract the exclusion records (if any).       *
         //* If exclusion records, then set 'excRB' radio      *
         //* button and display definition filename in 'excTB'.*
         gString gsdef( "%s/%s", this->cfgOpt.appPath, dfltExList ) ;
         if (   ((this->Backup_Definition ( gsdef.ustr(), bkSet, true )) == OK)
             && (bkSet.bexCount > ZERO) )
         {
            gsdef.copy( this->cfgOpt.bkPath, gsDFLTBYTES ) ;
            excRB_Select = excTB_Active = true ;
         }
         else     // no definition file found
         {
            *this->cfgOpt.bkPath = NULLCHAR ;
            excRB_Select = false ;
         }
      }

      bkSet.barc.compose( arcTemplate, &lt.year, &lt.month, &lt.date ) ;
      bkSet.blog.compose( logTemplate, &lt.year, &lt.month, &lt.date ) ;
      bkSetStatus = OK ;            // declare success
   }

   #if ENABLE_DEBUGGING_CODE != 0 && DEBUG_BDP != 0 && BDP_PRELIM != 0
   //* View the initialized BSet data *
   this->dWin->SetDialogObscured () ;  // save parent window
   Display_BSet ( dialogROWS, dialogCOLS, dialogYPOS, dialogXPOS, bkSet, dColor ) ;
   this->dWin->RefreshWin () ;         // restore parent window
   #endif   // ENABLE_DEBUGGING_CODE && DEBUG_BDP && BDP_PRELIM

   //* Archive and Exclusion controls do not apply to synch operations.*
   if ( synch )
   {
      bkSet.bex_reset() ;              // discard any exclusion records
      arcRB_Active = excRB_Active = excTB_Active = scnRB_Active =
      arcRB_Select = excRB_Select = false ;
      arcRB_nColor = excRB_nColor = scnRB_nColor = gyColor ;
      arcTB_nColor = excTB_nColor = nc.bkgy ;
   }

   //** Define the dialog controls **
   InitCtrl bic[bkCONTROLS] = 
   {
   {  //* 'OK' pushbutton - - - - - - - - - - - - - - - - - - - - - -     okPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 2),        // ulY:       upper left corner in Y
      short(dialogCOLS / 2 - 9),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      8,                            // cols:      control columns
      "   ^OK   ",                  // dispText:  
      this->cfgOpt.cScheme.pn,      // nColor:    non-focus color
      this->cfgOpt.cScheme.pf,      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &bic[canPB]                   // nextCtrl:  link in next structure
   },
   {  //* 'CANCEL' pushbutton - - - - - - - - - - - - - - - - - - - - -  canPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      bic[okPB].ulY,                // ulY:       upper left corner in Y
      short(bic[okPB].ulX + bic[okPB].cols + 2), // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      8,                            // cols:      control columns
      " ^CANCEL ",                  // dispText:  
      this->cfgOpt.cScheme.pn,      // nColor:    non-focus color
      this->cfgOpt.cScheme.pf,      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &bic[hhSP]                    // nextCtrl:  link in next structure
   },
   { //* 'Hour' Spinner    - - - - - - - - - - - - - - - - - - - - - -    hhSP *
      dctSPINNER,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      7,                            // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      4,                            // cols:      control columns
      NULL,                         // dispText:  (n/a)
      this->cfgOpt.cScheme.tn,      // nColor:    non-focus color
      this->cfgOpt.cScheme.tf,      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "hrs",                        // label:     
      1,                            // labY:      
      0,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      &hspData,                     // spinData:  spinner init
      true,                         // active:    allow control to gain focus
      &bic[mmSP]                    // nextCtrl:  link in next structure
   },
   { //* 'Minute' Spinner  - - - - - - - - - - - - - - - - - - - - - -    mmSP *
      dctSPINNER,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      bic[hhSP].ulY,                // ulY:       upper left corner in Y
      short(bic[hhSP].ulX + bic[hhSP].cols), // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      2,                            // cols:      control columns
      NULL,                         // dispText:  (n/a)
      this->cfgOpt.cScheme.tn,      // nColor:    non-focus color
      this->cfgOpt.cScheme.tf,      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "mn",                         // label:     
      1,                            // labY:      
      0,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      &mspData,                     // spinData:  spinner init
      true,                         // active:    allow control to gain focus
      &bic[arcRB]                   // nextCtrl:  link in next structure
   },
   {  //* 'ARCHIVE' radio button    - - - - - - - - - - - - - - - - - -  arcRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      arcRB_Select,                 // rbSelect:  initial value
      short(bic[hhSP].ulY - 1),     // ulY:       upper left corner in Y
      short(dialogCOLS / 2),        // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      arcRB_nColor,                 // nColor:    non-focus color
      this->cfgOpt.cScheme.pf,      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Create Backup ^Archive",     // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      arcRB_Active,                 // active:    allow control to gain focus
      &bic[arcTB]                   // nextCtrl:  link in next structure
   },
   {  //* 'ARCHIVE NAME' Textbox - - - - - - - - - - - - - - - - - - -   arcTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(bic[arcRB].ulY + 1),    // ulY:       upper left corner in Y
      short(bic[arcRB].ulX + 6),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      32,                           // cols:      control columns
      bkSet.barc.ustr(),            // dispText:  
      arcTB_nColor,                 // nColor:    non-focus color
      this->cfgOpt.cScheme.tf,      // fColor:    focus color
      #if LINUX_SPECIAL_CHARS != 0
      tbFileLinux,                  // filter: valid filename chars (incl. Linux "special")
      #else    // BASIC FILENAME FILTER
      tbFileName,                   // filter:    valid filename characters
      #endif   // BASIC FILENAME FILTER
      NULL,                         // label:     
      ZERO,                         // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      bic[arcRB].rbSelect,          // active:    allow control to gain focus
      &bic[excRB]                   // nextCtrl:  link in next structure
   },
   {  //* 'EXCLUSION LIST' radio button   - - - - - - - - - - - - - - -  excRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      excRB_Select,                 // rbSelect:  initial value
      short(bic[arcRB].ulY + 3),    // ulY:       upper left corner in Y
      bic[arcRB].ulX,               // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      excRB_nColor,                 // nColor: non-focus color
      this->cfgOpt.cScheme.pf,      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "^Exclusion List",            // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      excRB_Active,                 // active:    allow control to gain focus
      &bic[excTB]                   // nextCtrl:  link in next structure
   },
   {  //* 'EXCLUSION LIST' Textbox - - - - - - - - - - - - - - - - - -   excTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(bic[excRB].ulY + 1),    // ulY:       upper left corner in Y
      short(bic[excRB].ulX + 6),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      32,                           // cols:      control columns
      NULL,                         // dispText:  initially empty
      excTB_nColor,                 // nColor:    non-focus color
      this->cfgOpt.cScheme.tf,      // fColor:    focus color
      #if LINUX_SPECIAL_CHARS != 0
      tbPathLinux,                  // filter: valid filespec chars (incl. Linux "special")
      #else    // BASIC FILESPEC FILTER
      tbPathName,                   // filter:    valid filename characters
      #endif   // BASIC FILESPEC FILTER
      NULL,                         // label:     
      ZERO,                         // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      excTB_Active,                 // active:    allow control to gain focus
      &bic[scnRB]                   // nextCtrl:  link in next structure
   },
   {  //* 'SCAN-ONLY' radio button    - - - - - - - - - - - - - - - - -  scnRB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 3-wide
      false,                        // rbSelect:  initial value
      bic[canPB].ulY,               // ulY:       upper left corner in Y
      short(bic[canPB].ulX + bic[canPB].cols + 3), // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      scnRB_nColor,                 // nColor: non-focus color
      this->cfgOpt.cScheme.pf,      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "^Scan Only (no updates)",    // label:     
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      scnRB_Active,                 // active:    allow control to gain focus
      &bic[logDD]                   // nextCtrl:  link in next structure
   },
   {  //* 'LOG FORMAT' Dropdown  - - - - - - - - - - - - - - - - - - -   logDD *
      dctDROPDOWN,                  // type:      define a dropdown control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 4),        // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      short(ddITEMS + 2),           // lines:     control lines
      short(ddWIDTH + 2),           // cols:      control columns
      (const char*)&ddText,         // dispText
      this->cfgOpt.cScheme.sd,      // nColor:    non-focus border color
      this->cfgOpt.cScheme.pf,      // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "Log ^Format",                // label:     
      ZERO,                         // labY:      offset from control's ulY
      short(ddWIDTH + 3),           // labX       offset from control's ulX
      ddBoxUP,                      // exType:    expansion direction
      ddITEMS,                      // scrItems:  number of elements in text/color arrays
      1,                            // scrSel:    index of initial highlighted element
      monoColor,                    // scrColor:  single-color data display
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
   } ;

   this->dWin->SetDialogObscured () ;   // Save parent dialog display data

   //* Open the interface dialog *
   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       dialogYPOS,     // Y offset from upper-left of terminal 
                       dialogXPOS,     // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       bic             // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;
   bkDlg = dp ;                     // make dialog visible within this module
   bkDlgColor = dColor | ncbATTR ;  // dialog text color attribute
   bkDlgLt = &lt ;                  // pointer to current local time
   short setupStatus = ERR ;        // 'OK' if all setup parameters are valid

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Set the dialog title *
      dp->SetDialogTitle ( synch ? syTitle : bkTitle, this->cfgOpt.cScheme.em ) ;

      //* Enable audible boink for invalid characters in Textbox controls. *
      dp->TextboxAlert ( arcTB, true ) ;
      dp->TextboxAlert ( excTB, true ) ;

      //* If Archive Radiobutton is inactive, grey-out its label.*
      if ( !arcRB_Active )
         dp->WriteString ( bic[arcRB].ulY, bic[arcRB].ulX + bic[arcRB].labX,
                           "Create Backup Archive", gyColor ) ;

      //* If Exclusion Radiobutton is inactive, grey-out its label.*
      if ( !excRB_Active )
         dp->WriteString ( bic[excRB].ulY, bic[excRB].ulX + bic[excRB].labX,
                           "Exclusion List", gyColor ) ;

      //* If Scan-only Radiobutton is inactive, grey-out its label.*
      if ( !scnRB_Active )
         dp->WriteString ( bic[scnRB].ulY, bic[scnRB].ulX + bic[scnRB].labX,
                           "Scan Only (no updates)", gyColor ) ;


      //* Display source and target paths *
      winPos wp( 1, 2 ), wppath( 2, 2 ) ;
      gString gsOut ;            // formatted text
      bool accessError = false ; // data-access error
      gsOut = bkSet.bsrc ;
      this->fdPtr->TrimPathString ( gsOut, dialogCOLS - 4 ) ;
      dp->WriteString ( wppath, gsOut, hColor ) ;
      gsOut = bkSet.btrg ;
      this->fdPtr->TrimPathString ( gsOut, dialogCOLS - 4 ) ;
      wppath.ypos += 2 ;
      dp->WriteString ( wppath, gsOut, hColor ) ;

      //* Display static text *
      gsOut =  "Source Path:\n\n"
               "Target Path:\n\n" ;
      //* For synch, must have read/write access *
      //* to both 'source' and 'target'.         *
      if ( synch && !(srAcc && swAcc && trAcc && twAcc) )
      {
         short ins = gsOut.find( L':' ) ;
         if ( !srAcc || !swAcc )
            gsOut.insert( rdOnly, ins ) ;
         if ( !trAcc || !twAcc )
         {
            ins = gsOut.find( L':' ) ;
            gsOut.insert( rdOnly, gsOut.find( L':', ++ins ) ) ;
         }
         accessError = true ;
      }
      //* For backup, must have read access to source *
      //* and read/write access to target.            *
      else if ( !synch && !(srAcc && trAcc && twAcc) )
      {
         short ins = gsOut.find( L':' ) ;
         if ( !srAcc )
            gsOut.insert( rdOnly, ins );
         if ( !trAcc || !twAcc )
         {
            ins = gsOut.find( L':' ) ;
            gsOut.insert( rdOnly, gsOut.find( L':', ++ins ) ) ;
         }
         accessError = true ;
      }
      wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
      dp->WriteString ( ++wp.ypos, wp.xpos, 
                        "Delay the operation by:           ",
                        dColor | ncuATTR ) ;
      wp.ypos += 5 ; wp.xpos = bic[logDD].ulX + bic[logDD].cols + 1 ;

      //* Set up the callback method to update the *
      //* delayed-operation timestamp.             *
      bkDlgWp = dp->WriteParagraph ( bic[mmSP].ulY, 
                                     (bic[mmSP].ulX + bic[mmSP].cols + 1), 
                                     "Operation will begin:\n", dColor ) ;
      dp->EstablishCallback ( &Backup_Callback ) ;

      //* If a definition file specified, display *
      //* its tail in the Exclusion Textbox, else *
      //* leave the field empty.                  *
      if ( excRB_Select )
      {
         dp->SetTextboxText ( excTB, this->cfgOpt.bkPath ) ;
         dp->DisplayTextboxTail ( excTB ) ;
      }

      //* If error in specified definition file *
      if ( bkdefWarn )
      {
         dp->ClearLine ( wp.ypos, false, wp.xpos ) ;
         dp->WriteString ( wp, 
            "Warning: Error in definition file. Default values used.", hColor ) ;
      }

      dp->RefreshWin () ;                 // make the dialog visible

      //* Enter the user-interface loop.*
      if ( ! accessError )
      {
         uiInfo   Info ;                     // user interface data returned here
         short    icIndex = ZERO ;           // index of control with input focus
         bool     done = false ;             // loop control
         while ( ! done )
         {
            //* If focus is currently on a Pushbutton   *
            if ( bic[icIndex].type == dctPUSHBUTTON )
            {
               if ( Info.viaHotkey )
                  Info.HotData2Primary () ;
               else
                  icIndex = dp->EditPushbutton ( Info ) ;

               //* If a Pushbutton was pressed *
               if ( Info.dataMod != false )
               {
                  if ( Info.ctrlIndex == okPB ) // 'OK' Pushbutton
                  {  //***********************************************
                     //* User has launched the operation.            *
                     //* Gather all the parameters and validate them.*
                     //***********************************************
                     //* Get the delay (hours and minutes) from the *
                     //* spinner controls. If a non-zero offset, set*
                     //* the delay time, else reset the timestamp.  *
                     dp->GetSpinnerValue ( hhSP, bkSet.bhour ),
                     dp->GetSpinnerValue ( mmSP, bkSet.bmin ) ;
                     dp->GetRadiobuttonState ( scnRB, bkSet.scanOnly ) ;
                     bkSet.blfmt = (bsetLog)dp->GetDropdownSelect ( logDD ) ;
                     if ( bkSet.bhour > ZERO || bkSet.bmin > ZERO )
                     {
                        //* Update the timestamp to actual launch time *
                        //*           (see notes above)                *
                        this->fdPtr->GetLocalTime ( lt ) ; // get the local time
                        short roundup = (lt.epoch % 60) ; // round up to even minute
                        bkSet.btime.epoch = lt.epoch + (60 - roundup) + 
                                            (bkSet.bhour * 60 * 60) + 
                                            (bkSet.bmin * 60) ;
                        this->fdPtr->DecodeLocalTime ( bkSet.btime ) ;
                        bkSet.blog.compose( logTemplate, &bkSet.btime.year, 
                                            &bkSet.btime.month, &bkSet.btime.date ) ;
                     }
                     else     // no delay
                        bkSet.btime.reset() ;

                     //* If target is an archive file, get archive name. *
                     if ( bkSet.btype != bsetSynch )
                     {
                        bool arcTarget ;
                        dp->GetRadiobuttonState ( arcRB, arcTarget ) ;
                        if ( arcTarget )
                        {
                           dp->GetTextboxText ( arcTB, bkSet.barc ) ;
                           bkSet.btype = bsetArchive ;
                           //* Scan-only does not apply to archive target *
                           bkSet.scanOnly = false ;
                        }
                        else
                           bkSet.btype = bsetBackup ;
                     }

                     //* Exclusion records apply ONLY to Backup operations *
                     if ( bkSet.btype == bsetBackup )
                     {  //* If an exclusion list was specified, AND if an   *
                        //* exclusion list has not already been created,    *
                        //* then get the filespec and create the list.      *
                        if ( bkSet.bexCount == ZERO && bkSet.bex == NULL )
                        {
                           bool exl ;
                           dp->GetRadiobuttonState ( excRB, exl ) ;
                           if ( exl )
                           {  //* Name (or path/filename) of definition file *
                              gString exList ;
                              dp->GetTextboxText ( excTB, exList ) ;
                              this->Backup_Exclusions ( exList, bkSet ) ;
                           }
                        }
                     }
                     else  // discard any exclusion records for Archive and Synch
                        bkSet.bex_reset() ;

                     //* Validate the parameters *
                     setupStatus = this->Backup_Validate ( bkSet, gsOut ) ;

                     #if ENABLE_DEBUGGING_CODE != 0 && DEBUG_BDP != 0 && BDP_FINAL != 0
                     dp->SetDialogObscured () ;
                     Display_BSet ( dialogROWS, dialogCOLS, dialogYPOS, 
                                    dialogXPOS, bkSet, dColor,
                                    setupStatus == OK ? 1 : ERR ) ;
                     dp->RefreshWin () ;
                     #endif   // ENABLE_DEBUGGING_CODE && DEBUG_BDP && BDP_FINAL

                     //* If a validation error occurred *
                     if ( setupStatus != OK )
                     {  //* Operation has been aborted. Display the  *
                        //* error message, then wait for a keypress. *
                        dp->ClearLine ( wp.ypos, false, wp.xpos ) ;
                        dp->WriteString ( wp, gsOut, hColor, true ) ;
                        dp->NextControl () ;
                        nckPause();
                     }

                     //* If the validation method updated the archive *
                     //* filename, then update the displayed name and *
                     //* wait a moment for user to read it.           *
                     else if ( bkSet.btype == bsetArchive )
                     {  //* Isolate the filename in archive-target's path. *
                        short fnIndex = bkSet.barc.findlast( L'/' ) ;
                        if ( fnIndex >= ZERO )     // if found
                           ++fnIndex ;
                        else                       // if not found
                           fnIndex = ZERO ;
                        //* Compare with contents of archive Textbox. *
                        dp->GetTextboxText ( arcTB, gsOut ) ;
                        if ( (gsOut.compare( &bkSet.barc.gstr()[fnIndex] )) != ZERO )
                        {
                           dp->SetTextboxText ( arcTB, &bkSet.barc.gstr()[fnIndex] ) ;
                           gsOut = "Specified archive target already exists. "
                                   "Target filename updated." ;
                           dp->ClearLine ( wp.ypos, false, wp.xpos ) ;
                           dp->WriteString ( wp, gsOut, hColor, true ) ;
                           chrono::duration<short, std::milli>aMoment( 3000 ) ;
                           this_thread::sleep_for( aMoment ) ;
                        }
                     }
                  }
                  else if ( Info.ctrlIndex == canPB ) // 'CANCEL' Pushbutton
                  {
                     dp->ClearLine ( wp.ypos, false, wp.xpos ) ;
                     dp->WriteString ( wp, "             Operation Cancelled.",
                                       hColor, true ) ;
                     chrono::duration<short, std::milli>aMoment( 800 ) ;
                     this_thread::sleep_for( aMoment ) ;
                  }
                  done = true ;
               }
            }

            //* If focus is currently on a Radiobutton  *
            else if ( bic[icIndex].type == dctRADIOBUTTON )
            {
               if ( Info.viaHotkey )
                  Info.HotData2Primary () ;
               else
                  icIndex = dp->EditRadiobutton ( Info ) ;
   
               //* If a Radiobutton selected *
               if ( Info.dataMod != false )
               {
                  //* If user has selected/deselected the archive option. *
                  //* (does not apply to synch operations)                *
                  if ( !synch && (Info.ctrlIndex == arcRB) )
                  {
                     //* Activate the archive filename Textbox *
                     if ( Info.isSel )
                        dp->ControlActive ( arcTB, true ) ;
                     //* De-activate the archive filename Textbox *
                     else
                        dp->ControlActive ( arcTB, false ) ;
                  }

                  //* If user has selected/deselected the exclusion option *
                  //* (does not apply to synch operations)                 *
                  else if ( !synch && (Info.ctrlIndex == excRB) )
                  {  //* Enable/disable exclusions Textbox *
                     if ( Info.isSel )    // exclusions enabled
                        dp->ControlActive ( excTB, true ) ;
                     else                 // exclusions disabled
                        dp->ControlActive ( excTB, false ) ;

                     // Programmer's Note: Exclusion records apply ONLY to 
                     // Backup operations. In an earlier version, we disabled 
                     // the exclusion radiobutton AND the exclusion textbox 
                     // if the Archive radiobutton was selected; however, 
                     // this made the user interface unnecessarily difficult 
                     // to use. Now, we simply ignore Exclusion records for 
                     // Archive and Synch operations.
                  }
               }
            }

            //* If focus is currently on a Textbox      *
            else if ( bic[icIndex].type == dctTEXTBOX )
            {
               Info.viaHotkey = false ;   // discard hotkey data
               icIndex = dp->EditTextbox ( Info ) ;

               if ( Info.dataMod != false )
               {
                  if ( Info.ctrlIndex == excTB )
                  {
                     //* De-reference previous definition file,     *
                     //* and discard any previous exclusion records.*
                     *this->cfgOpt.bkPath = NULLCHAR ;
                     bkSet.bex_reset() ;

                     //* If the specified file exists, extract *
                     //* the exclusion records, if any.        *
                     bool newExclusions = false ;
                     gString gs,
                             gsdef ;
                     dp->GetTextboxText ( excTB, gs ) ;
                     if (   (this->gcaRealpath ( gsdef, gs.ustr() ))
                         && ((this->Backup_Definition ( gsdef.ustr(), bkSet, true )) == OK) )
                     {
                        //* Write full filespec to the Textbox control.*
                        dp->PrevControl () ;
                        dp->SetTextboxText ( excTB, gsdef ) ;
                        //dp->DisplayTextboxTail ( excTB ) ; // (doesn't stick)
                        dp->NextControl () ;

                        if ( bkSet.bexCount > ZERO )
                        {
                           gsdef.copy( this->cfgOpt.bkPath, gsDFLTBYTES ) ;
                           newExclusions = true ;
                        }
                        else
                           gs = "Specified definition file "
                                "contains no exclusion records!" ;
                     }
                     else
                        gs = "Specified definition file not "
                             "found or contains syntax errors!" ;

                     //* If error, alert user *
                     if ( ! newExclusions )
                     {
                        dp->ClearLine ( wp.ypos, false, wp.xpos ) ;
                        dp->WriteString ( wp, gs, hColor, true ) ;
                     }
                  }
               }
            }

            //* If focus is currently on a Spinner      *
            else if ( bic[icIndex].type == dctSPINNER )
            {
               Info.viaHotkey = false ;   // discard hotkey data
               icIndex = dp->EditSpinner ( Info ) ;
            }

            //* If focus is currently on a Dropdown   *
            if ( bic[icIndex].type == dctDROPDOWN )
            {
               //* If Info.viaHotkey is set, control will expand immediately.*
               icIndex = dp->EditDropdown ( Info ) ;
            }

            if ( ! done && ! Info.viaHotkey )
            {
               if ( Info.keyIn == nckSTAB )
                  icIndex = dp->PrevControl () ; 
               else
                  icIndex = dp->NextControl () ;
            }
         }  // while()
      }
      //* If user does not have write access to target, *
      //* then just wait for a keypress.                *
      else
      {
         dp->ClearLine ( wp.ypos, false, wp.xpos ) ;
         dp->WriteString ( wp, "Restricted access - unable to continue!", 
                           hColor, true ) ;
         dp->NextControl () ;
         nckPause();
      }
   }
   if ( dp != NULL )                      // close the window
      delete ( dp ) ;
   bkDlg = NULL ;                         // reset the local dialog handle
   bkDlgLt = NULL ;                       // and time reference

   this->dWin->RefreshWin () ;   // Restore parent dialog display

   //* If setup valid and we have a go code.*
   if ( setupStatus == OK )
   {
      //* If delayed operation, do the countdown.      *
      //* Else get current time for logfile timestamp. *
      short launch = OK ;
      if ( bkSet.btime.epoch > ZERO )
         launch = this->Backup_Countdown ( bkSet ) ;
      else
         this->fdPtr->GetLocalTime ( bkSet.btime ) ;

      if ( launch == OK )
      {
         gString pubLog = bkSet.blog ;    // copy public logfile name
         //* Get filespec for temporary log file *
         if ( !(this->CreateTempname ( bkSet.blog )) )
         {  // NOTE: The chance of the system call failing is remote, but if
            // it does, the temp file will be created in the target directory.
            // (We append a '~' to avoid name conflict during save.)
            bkSet.blog.compose( "%S/%S~", bkSet.btrg.gstr(), pubLog.gstr() ) ;
         }

         short opStatus ;
         if ( bkSet.btype == bsetSynch )  // Synch operation
            opStatus = this->Synch_DirTree ( bkSet ) ;
         else                             // Backup operation
            opStatus = this->Backup_DirTree ( bkSet ) ;

         //* If operation was successful, update the status window. *
         //* Else, leave status window blank.                       *
         if ( opStatus == OK )
         {
            attr_t barColor = (attr_t)((this->cfgOpt.cScheme.scheme == ncbcMA 
                                                   ? nc.br : nc.ma) | ncrATTR) ;
            gString msgText( "%s Operation Complete!", (synch ? "Synch" : "Backup") ) ;
            attr_t msgColor[this->ic[dwMsgsTB].cols] ;
            for ( short i = ZERO ; i < this->ic[dwMsgsTB].cols ; ++i )
               msgColor[i] = barColor ;
            dtbmData msg( msgText.gstr(), msgColor, true, this->ic[dwMsgsTB].cols ) ;
            this->dWin->DisplayTextboxMessage ( dwMsgsTB, msg ) ;
         }

         //* Display results of operation *
         this->Backup_Summary ( bkSet, pubLog, opStatus ) ;

         //* If active (source) window equals inactive (target) *
         //* window, source window data may be stale.           *
         if ( bkSet.bsrc == bkSet.btrg )
            this->fdPtr->RefreshCurrDir () ;

         //* Delete the temporary log file *
         this->fdPtr->DeleteFile ( bkSet.blog ) ;
      }
      this->RestoreQuickHelpMsg () ;
   }

   //* Any definitions from the specified file  *
   //* are now obsolete. (see note above)       *
   *this->cfgOpt.bkPath = NULLCHAR ;

}  //* End Backup() *

//*************************
//*    Backup_Validate    *
//*************************
//******************************************************************************
//* Validate the parameters for the pending backup/synchronize operation.      *
//*                                                                            *
//* Input  : bkSet    : parameters to be validated (see notes below)           *
//*          errMsg   : if error, receives the error message                   *
//*                                                                            *
//* Returns: OK      if all parameters verified                                *
//*          ERR     if parameter error (errMsg receives the error message)    *
//******************************************************************************
//* Notes:                                                                     *
//* -- Operation type, (bkSet.btype) is controlled and needs no validation.    *
//*                                                                            *
//* -- If bkSet.bsrc==bkSet.btrg AND if target is not an archive file, then    *
//*    we signal abort. Otherwise, 'bsrc' and 'btrg' members are accepted      *
//*    as valid.                                                               *
//*                                                                            *
//* -- For archive filename, if specified, verify that filename ends with:     *
//*    a) '.tar'  (uncompressed tar archive), OR                               *
//*    b) one of the compression options supported by the tar utility:         *
//*       - '.bz2' , '.tz2' , 'tbz2' , '.tbz'      (bzip2)                     *
//*       - '.gz'  , '.tgz' , '.taz'               (gzip)                      *
//*       - '.lzma' , '.tlz'                       (lzma)                      *
//*       - '.lz'                                  (lzip)                      *
//*       - '.lzo'                                 (lzop)                      *
//*       - '.xz'                                  (xz)                        *
//*       - '.Z' , '.taZ'                          (compress)                  *
//*       Note that the call to tar specifies the '-auto-compress' option, so  *
//*       recognized suffixes will invoke the specified type of compression.   *
//*                                                                            *
//* -- For archive filename, if specified and if filename is valid, verify     *
//*    that target does not already exist. If target DOES exist, then modify   *
//*    the name by adding a sequence letter and test again until a             *
//*    non-existing name is found.                                             *
//*    Examples:                                                               *
//*      Backup_2016_03_31.tar      >  Backup_2016_03_31a.tar                  *
//*      Backup_2016_03_31.tar.bz2  >  Backup_2016_03_31a.tar.bz2              *
//*      Backup_2016_03_31a.gz      >  Backup_2016_03_31b.gz                   *
//*    -- 'bkSet.barc' returns with the full path/filename of the              *
//*        archive target.                                                     *
//*                                                                            *
//* -- If user-accessible logfile specified, (bkSet.blflag != false):          *
//*    This has been validated by caller, and is not referenced here.          *
//*                                                                            *
//* -- If exclusion records were specified, (bkSet.bexCount > ZERO) they are   *
//*    free-form and so cannot be validated in a meaningful way.               *
//*    NOTE: In a future revision, we may want to scan the exclusion records   *
//*          for environment variable substitutions, but currently the         *
//*          documentation specifies that this WILL NOT be done.               *
//*          ("Can open; worms everywhere!" -- Chandler Bing)                  *
//*                                                                            *
//* -- If delayed operation:                                                   *
//*    bkSet.btime contains the timestamp for start of operation.              *
//*    This has been validated by caller, and is not referenced here.          *
//*                                                                            *
//******************************************************************************

short FileMangler::Backup_Validate ( BSet& bkSet, gString& errMsg )
{
   short status = OK ;        // return value

   //* If target is an archive file:                                 *
   //* a) Verify that filename extension is either '.tar' or         *
   //*    one of the supported compression extensions (see above).   *
   //* b) Create the filespec and verify that target does not        *
   //*    already exist, and update the target filename as necessary.*
   if ( (bkSet.btype == bsetArchive) && ((bkSet.barc.gschars()) > 1) )
   {
      gString tExt ;
      short archType = this->fdPtr->ArchiveTarget ( bkSet.barc, &tExt ) ;

      //* Tar and Zip targets only *
      if ( (archType != atNONE) && (archType != atODOC) && (archType != atOXML) )
      {  //* tExt contains the valid filename extension substring. *
         //* Create the filespec and test whether it exists.       *
         // Programmer's Note: This algorithm is a bit tricky, so study it carefully.
         fmFType ft ;
         gString arcPath( "%S/%S", bkSet.btrg.gstr(), bkSet.barc.gstr() ) ;
         if ( (this->fdPtr->TargetExists ( arcPath, ft )) )
         {  //* Add a sequencing character and scan for a filename that *
            //* isn't already in use.                                   *
            wchar_t seqChar = L'a',       // sequence character
                    seqOld  ;
            short   seqIndx = arcPath.findlast( tExt ) ;
            arcPath.insert( seqChar, seqIndx-- ) ; // first sequenced name
            while ( (this->fdPtr->TargetExists ( arcPath, ft )) && (seqChar <= L'z') )
            {
               seqOld = seqChar++ ;
               arcPath.replace( seqOld, seqChar, seqIndx ) ;
            }
            //* Update caller's archive filename *
            if ( seqChar <= L'z' )
               bkSet.barc = arcPath ;
            else
            {  // (this is unlikely)
               errMsg = "Target archive already exists. - unable to continue." ;
               status = ERR ;
            }
         }
         else           // target does not exist
            bkSet.barc = arcPath ;
      }
      else                 // unsupported filename extension
      {
         errMsg = "Unsupported archive filename extension - unable to continue." ;
         status = ERR ;
      }
   }
   else     // target is not an archive file
   {  //* Test whether source directory == target directory.*
      if ( bkSet.bsrc == bkSet.btrg )
      {
         errMsg = "Source directory equals Target directory - unable to continue." ;
         status = ERR ;
      }
   }
   return status ;

}  //* End Backup_Validate() *

//*************************
//*   Backup_Countdown    *
//*************************
//******************************************************************************
//* Open a dialog to visually count down to launch time.                       *
//*                                                                            *
//* Input  : bkSet  : (by reference) contains type of operation and the        *
//*                   timestamp for beginning the backup/synch                 *
//*                                                                            *
//* Returns: OK  if launch is a go, ERR if user aborts                         *
//******************************************************************************
//* Programmer's Note: This dialog does not have a user interface control      *
//* because we are sleeping most of the time until terminal count is reached,  *
//* so we must not call one of the NcDialog EditXXX() methods.                 *
//* The dialog _appears_ to have a Pushbutton to give the user something       *
//* shiny to play with, but in reality, we monitor the input stream directly.  *
//******************************************************************************

short FileMangler::Backup_Countdown ( const BSet& bkSet )
{
   const char *bkTitle = "  Countdown to Data Backup  ",
              *syTitle = "  Countdown to Data Synch   " ;
   const char* Dow[]   = { "Sunday", "Monday", "Tuesday", "Wednesday",
                           "Thursday", "Friday", "Saturday", "" } ;
   const short dialogROWS = 10,
               dialogCOLS = 34,
               dialogYPOS = this->ic[dwFileSE].ulY + 1,
               dialogXPOS = this->ic[dwFileSE].ulX + this->ic[dwFileSE].cols
                             - (dialogCOLS / 2) - 1 ;

   const attr_t dColor  = this->cfgOpt.cScheme.sd,
                hColor  = this->cfgOpt.cScheme.em ;
   short status = OK ;           // return value

   this->dWin->SetDialogObscured () ;   // Save parent dialog display data

   //* Open the interface dialog *
   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       dialogYPOS,     // Y offset from upper-left of terminal 
                       dialogXPOS,     // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       NULL            // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Set the dialog title *
      dp->SetDialogTitle ( (bkSet.btype == bsetSynch) ? 
                           syTitle : bkTitle, this->cfgOpt.cScheme.em ) ;
      //* Fake a Pushbutton control *
      dp->WriteString ( dialogROWS - 2, dialogCOLS / 2 - 4, 
                        " CANCEL ", this->cfgOpt.cScheme.pf ) ;

      winPos wp( 2, 3 ) ;
      gString gsOut( "Operation is scheduled for:\n" ) ;
      wp = dp->WriteParagraph ( wp, gsOut, hColor ) ;
      gsOut.compose( "%s, %04hu-%02hu-%02huT%02hu:%02hu\n\n       ", 
                     Dow[bkSet.btime.day], &bkSet.btime.year, &bkSet.btime.month,
                     &bkSet.btime.date, &bkSet.btime.hours, &bkSet.btime.minutes ) ;
      wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
      wp = dp->WriteParagraph ( wp, "Time Remaining\n  ", hColor ) ;

      dp->RefreshWin () ;                 // make the dialog visible

      //* Sleep interval is chosen to be about the limit of *
      //* a caffinated user's patience i.e 3 seconds.       *
      //* During the final minute of the countdown, we      *
      //* reduce the sleep interval to 1 second.            *
      chrono::duration<short, std::milli>aMoment( 3000 ) ;
      chrono::duration<short, std::milli>aSecond( 1000 ) ;
      wkeyCode wKey ;            // user input
      short mY = dialogROWS - 2, // mouse-click target
            mlX = 13, mrX = 20 ;
      localTime lt ;       // system local time
      int64_t rem ;        // time remaining
      short hh, mm, ss ;   // formatted time remaining
      do
      {
         this->fdPtr->GetLocalTime ( lt ) ;     // get current time
         rem = bkSet.btime.epoch - lt.epoch ;   // calculate time remaining
         if ( rem >= ZERO )                     // if time remaining, display it 
         {
            hh = (short)(rem / 3600) ;          // format time for display
            rem -= hh * 3600 ;
            mm = (short)(rem / 60) ;
            rem -= mm * 60 ;
            ss = (short)rem ;
            gsOut.compose( " %02hd:%02hd:%02hd ", &hh, &mm, &ss ) ;
            dp->WriteString ( wp, gsOut, this->cfgOpt.cScheme.tf, true ) ;
         }
         if ( (dp->KeyPeek ( wKey )) != wktERR )
         {
            //* Get the input, with modified mouse event (if any), *
            //* then discard any remaining key/mouse input events  *
            dp->GetKeyInput ( wKey ) ;
            dp->FlushKeyInputStream () ;  // guard against over-caffinated user
            //* If a meaningful keystroke OR a mouse *
            //* click on the 'CANCEL' Pushbutton *
            if (    (wKey.type == wktPRINT && wKey.key == nckSPACE)
                 || (wKey.type == wktFUNKEY &&
                     (wKey.key == nckENTER || wKey.key == nckESC || wKey.key == nckC_C))
                 || (wKey.type == wktMOUSE && wKey.mevent.ypos == mY &&
                      (wKey.mevent.xpos >= mlX && wKey.mevent.xpos <= mrX) && 
                      (wKey.mevent.meType == metB1_S || wKey.mevent.meType == metB1_D)) )
            {
               dp->WriteString ( wp.ypos + 1, wp.xpos - 5, 
                                 "Operation Cancelled.", hColor, true ) ;
               status = ERR ;
            }
         }
         //*Go to sleep  (see note above) *
         if ( (mm > ZERO) || (hh > ZERO) )
            this_thread::sleep_for( aMoment ) ;
         else if ( ss > ZERO )
            this_thread::sleep_for( aSecond ) ;
      } while ( lt.epoch < bkSet.btime.epoch && status == OK ) ;
   }
   if ( dp != NULL )                      // close the window
      delete ( dp ) ;

   this->dWin->RefreshWin () ;   // Restore parent dialog display

   return status ;

}  //* End Backup_Countdown() *

//*************************
//*    Backup_Summary     *
//*************************
//******************************************************************************
//* Display a summary of the completed backup/archive/synch operation.         *
//*                                                                            *
//* Input  : bkSet    : (by reference) contains summary data                   *
//*          logName  : filename for public logfile                            *
//*                     will be written to target directory ('btrg')           *
//*          opStatus : operational status: OK==success, ERR==user abort       *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Notes:                                                                     *
//*                                                                            *
//******************************************************************************

void FileMangler::Backup_Summary ( const BSet& bkSet, const gString& logName, short opStatus )
{
   const char* opHdr  = "Operation    : " ;
   const char* sfHdr  = "Source files : " ;
   const char* fuHdr  = "Files updated: " ;
   const char* fsHdr  = "Files skipped: " ;
   const char* soHdr  = "Needs Update : " ;  // scan-only - files that need update
   const char* feHdr  = "Errors       : " ;
   const char* summaryTemplate = 
   "Summary Results:\n"
   "----------------\n"
   "%s%S" ;

   const short dialogROWS = 12,
               dialogCOLS = 90,
               dialogYPOS = this->ic[dwFileSE].ulY + 1,
               dialogXPOS = this->ic[dwFileSE].ulX + this->ic[dwFileSE].cols
                             - (dialogCOLS / 2) - 1 ;

   const attr_t dColor  = this->cfgOpt.cScheme.sd,
                hColor  = this->cfgOpt.cScheme.em ;

   enum bsControls : short { clPB, vlPB, slPB, bsCONTROLS } ;

   //** Define the dialog controls **
   InitCtrl bic[bsCONTROLS] = 
   {
   {  //* 'CLOSE' pushbutton - - - - - - - - - - - - - - - - - - - - -    clPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 2),        // ulY:       upper left corner in Y
      28,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      9,                            // cols:      control columns
      "  ^CLOSE  ",                 // dispText:  
      this->cfgOpt.cScheme.pn,      // nColor:    non-focus color
      this->cfgOpt.cScheme.pf,      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &bic[vlPB]                    // nextCtrl:  link in next structure
   },
   {  //* 'VIEW LOG' pushbutton - - - - - - - - - - - - - - - - - - - -   vlPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      bic[clPB].ulY,                // ulY:       upper left corner in Y
      short(bic[clPB].ulX + bic[clPB].cols + 3), // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      10,                           // cols:      control columns
      " ^VIEW LOG ",                // dispText:  
      this->cfgOpt.cScheme.pn,      // nColor:    non-focus color
      this->cfgOpt.cScheme.pf,      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &bic[slPB]                    // nextCtrl:  link in next structure
   },
   {  //* 'VIEW LOG' pushbutton - - - - - - - - - - - - - - - - - - - -   vlPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      bic[vlPB].ulY,                // ulY:       upper left corner in Y
      short(bic[vlPB].ulX + bic[vlPB].cols + 3), // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      10,                           // cols:      control columns
      " ^SAVE LOG ",                // dispText:  
      this->cfgOpt.cScheme.pn,      // nColor:    non-focus color
      this->cfgOpt.cScheme.pf,      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
   } ;

   this->dWin->SetDialogObscured () ;  // Save parent dialog display data

   //* Open the interface dialog *
   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       dialogYPOS,     // Y offset from upper-left of terminal 
                       dialogXPOS,     // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       bic             // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Set the dialog title *
      dp->SetDialogTitle ( "  Operation Summary  ", this->cfgOpt.cScheme.em ) ;

      gString localLog,                // for viewing log file
              gsSummary,               // contains summary stats for log file
              publicLog( "%S/%S",      // filespec for public log file
                         bkSet.btrg.gstr(), logName.gstr() ) ;

      //* Display source and target paths *
      winPos wp( 1, 2 ), wpd( 2, 2 ) ;
      gString gsOut =  "Source Path:\n\n"
                       "Target Path:\n\n\n" ;
      wp = dp->WriteParagraph ( wp, gsOut, hColor ) ;
      gsOut = bkSet.bsrc ;
      this->fdPtr->TrimPathString ( gsOut, dialogCOLS - 4 ) ;
      gsOut.append( L"\n\n" ) ;
      wpd = dp->WriteParagraph ( wpd, gsOut, dColor ) ;
      gsOut = bkSet.btrg ;
      this->fdPtr->TrimPathString ( gsOut, dialogCOLS - 4 ) ;
      gsOut.append( L"\n\n" ) ;
      wpd = dp->WriteParagraph ( wpd, gsOut, dColor ) ;

      //* 1) Type of operation   *
      short opIndex = bkSet.scanOnly ? 3 : (bkSet.btype == bsetBackup) ? 0 :
                                           (bkSet.btype == bsetSynch) ? 1 : 2 ;
      wpd = dp->WriteString ( wp.ypos++, wp.xpos, opHdr, hColor ) ;
      this->Backup_Timestamp ( gsOut, bkSet.btime, false ) ;
      gsOut.insert( "  " ) ;
      gsOut.insert( opName[opIndex] ) ;
      gsOut.append( "  " ) ;
      if ( opStatus != OK )
         gsOut.append( "(operation aborted)" ) ;
      winPos wpArch = dp->WriteString ( wpd.ypos++, wpd.xpos, gsOut, dColor ) ;
      gsSummary.compose( summaryTemplate, opHdr, gsOut.gstr() ) ;

      if ( (opStatus == OK) && (bkSet.btype == bsetArchive) )
      {  //* 1a) Name of target archive *
         short arcnameIndex = (bkSet.barc.findlast( L'/' )) + 1 ;
         gsOut.compose( " %S ", &bkSet.barc.gstr()[arcnameIndex] ) ;
         dp->WriteString ( wpArch, gsOut, this->cfgOpt.cScheme.tf ) ;
         gsSummary.append( gsOut.gstr() ) ;
      }
      gsSummary.append( L'\n' ) ;

      //* 2) total files         *
      dp->WriteString ( wp.ypos++, wp.xpos, sfHdr, hColor ) ;
      gsOut.formatInt ( bkSet.ftotal, FI_MAX_FIELDWIDTH, true ) ;
      dp->WriteString ( wpd.ypos++, wpd.xpos, gsOut, dColor ) ;
      gsSummary.append( "%s%S\n", sfHdr, gsOut.gstr() ) ;

      // 3) Files updated (or needing update)
      dp->WriteString ( wp.ypos++, wp.xpos, (bkSet.scanOnly ? soHdr : fuHdr), hColor ) ;
      gsOut.formatInt ( bkSet.fupdated, FI_MAX_FIELDWIDTH, true ) ;
      dp->WriteString ( wpd.ypos++, wpd.xpos, gsOut, dColor ) ;
      gsSummary.append( "%s%S\n", (bkSet.scanOnly ? soHdr : fuHdr), gsOut.gstr() ) ;

      // 4) Files not updated (targets up-to-date or source files on exclusion list)
      dp->WriteString ( wp.ypos++, wp.xpos, fsHdr, hColor ) ;
      gsOut.formatInt ( bkSet.fskipped, FI_MAX_FIELDWIDTH, true ) ;
      dp->WriteString ( wpd.ypos++, wpd.xpos, gsOut, dColor ) ;
      gsSummary.append( "%s%S\n", fsHdr, gsOut.gstr() ) ;

      //* 5) number of errors    *
      dp->WriteString ( wp, feHdr, hColor ) ;
      gsOut.formatInt ( bkSet.ferrors, FI_MAX_FIELDWIDTH, true ) ;
      dp->WriteString ( wpd, gsOut, dColor ) ;
      gsSummary.append( "%s%S\n", feHdr, gsOut.gstr() ) ;

      wp = { short(bic[clPB].ulY - 1), bic[clPB].ulX } ; // log message position

      //* Format the log file:               *
      //* a) get a temp-file filespec        *
      //* b) write formatted data to the log *
      this->CreateTempname ( localLog ) ;
      this->Backup_FormatLog ( bkSet, localLog, gsSummary ) ;

      //* If automatic logfile save specified, save the temporary log *
      //* file to the public log file, delete the temporary log file, *
      //* and disable the 'SAVE LOG' Pushbutton.                      *
      if ( bkSet.blflag )
      {
         this->Backup_SaveLog ( localLog, publicLog, dp, wp, hColor, dColor ) ;
         dp->ControlActive ( slPB, false ) ;
         this->fdPtr->DeleteFile ( localLog ) ; // delete the temporary logfile
         localLog = publicLog ;  // target the saved log for additional views
      }

      dp->RefreshWin () ;                 // make the dialog visible

      uiInfo   Info ;                     // user interface data returned here
      short    icIndex = ZERO ;           // index of control with input focus
      bool     done = false ;             // loop control
      while ( ! done )
      {
         if ( bic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;

            //* If a Pushbutton was pressed *
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == clPB )
               {
                  done = true ;
               }
               else if ( Info.ctrlIndex == vlPB )
               {  //* Create the command to view the log file *
                  gString gsCmd( "less -c %S", localLog.gstr() ) ;

                  //* Shell out to view the file *
                  dp->ShellOut ( soX, gsCmd.ustr() ) ;

                  //* Restore the display *
                  this->dWin->RefreshWin () ;
                  this->dWin->SetDialogObscured () ;
                  icIndex = dp->NextControl () ;
                  dp->RefreshWin () ;
               }
               else if ( Info.ctrlIndex == slPB )
               {  // Programmer's Note: If log file already saved to user's
                  // directory, this Pushbutton has been disabled.
                  //* Save the temporary log file to the public log file *
                  this->Backup_SaveLog ( localLog, publicLog, dp, wp, hColor, dColor ) ;

                  //* Disable the 'SAVE LOG' Pushbutton *
                  //* to avoid duplicate saves.         *
                  dp->PrevControl () ;    // remove focus from target control
                  dp->ControlActive ( slPB, false ) ;
                  this->fdPtr->DeleteFile ( localLog ) ; // delete the temporary logfile
                  localLog = publicLog ;  // target the saved log for additional views
               }
            }

            if ( ! done && ! Info.viaHotkey )
            {
               if ( Info.keyIn == nckSTAB )
                  icIndex = dp->PrevControl () ; 
               else
                  icIndex = dp->NextControl () ;
            }
         }
      }     // while()

      //* If the temp log filespec references a temp file *
      //* (NOT the user's copy), delete the file.         *
      if ( localLog != publicLog )
         this->fdPtr->DeleteFile ( localLog ) ;
   }
   if ( dp != NULL )                      // close the window
      delete ( dp ) ;

   this->dWin->RefreshWin () ;   // Restore parent dialog display

}  //* End Backup_Summary() *

//*************************
//*   Backup_FormatLog    *
//*************************
//******************************************************************************
//* Concatenate the temporary-file data to create a formatted log file.        *
//*                                                                            *
//* Input  : bkSet   : operational parameters                                  *
//*                    'blog' is the filespec for the temporary file           *
//*          trgLog  : filespec for target log file                            *
//*          summary : operation summary data record to be inserted into log   *
//*                                                                            *
//* Returns: OK  if successful                                                 *
//*          ERR if error(s) writing log file                                  *
//******************************************************************************
//* 3) Read the header (2 lines) from the source file and write them to target.*
//* 4) Write source and target paths into the target file.                     *
//* 5) Insert the summary statistics into the target file.                     *
//* 6) Copy the remaining source file to target.                               *
//* 7) Close both source and target files.                                     *
//******************************************************************************

short FileMangler::Backup_FormatLog ( const BSet& bkSet, const gString& trgLog, 
                                      const gString& summary )
{
   gString gsOut ;               // output formatting
   bool readError = false,       // 'true' if read error
        writeError = false ;     // 'true' if write error

   //* Open the source file *
   ifstream ifs( bkSet.blog.ustr(), ifstream::in ) ;

   //* Open the target file *
   ofstream ofs( trgLog.ustr(), ofstream::out | ofstream::trunc ) ;
   if ( ifs.is_open() && ofs.is_open() )
   {
      char  lineData[gsDFLTBYTES] ; // raw UTF-8 input

      //* Copy source file header *
      ifs.getline ( lineData, gsDFLTBYTES, NEWLINE ) ;
      if ( ifs.good() || ifs.gcount() > ZERO )
         ofs << lineData << endl ;
      else  readError = true ;
      ifs.getline ( lineData, gsDFLTBYTES, NEWLINE ) ;
      if ( ifs.good() || ifs.gcount() > ZERO )
         ofs << lineData << endl ;
      else  readError = true ;

      //* Source and Target paths   *
      ofs << "\nSource:\n   " << bkSet.bsrc 
          << "\nTarget:\n   " << bkSet.btrg << "\n" << endl ;

      //* Insert summary statistics *
      ofs << summary.ustr() << endl ;

      //* Copy remainder of source to target *
      bool done = readError ;
      while ( ! done )
      {
         ifs.getline ( lineData, gsDFLTBYTES, NEWLINE ) ;
         if ( ifs.good() || ifs.gcount() > ZERO )
         {  //* Do a simple test for a second header (Synch operation) *
            //* and if found, discard it.                              *
            if ( (bkSet.btype == bsetSynch) && (lineData[ZERO] == 'F') )
            {
               ifs.getline ( lineData, gsDFLTBYTES, NEWLINE ) ;
               gsOut = "Second Pass:\n------------" ;
               gsOut.copy( lineData, gsDFLTBYTES ) ;
            }
            ofs << lineData << endl ;
         }
         else
         {
            ofs << "END-OF-LOG\n\n" << endl ;
            done = true ;
         }
      }
   }

   //* Close the files *
   if ( ifs.is_open() )
      ifs.close() ;
   else
      readError = true ;
   if ( ofs.is_open() )
      ofs.close() ;
   else
      writeError = true ;

   return ( (!readError && !writeError) ? OK : ERR ) ;

}  //* End Backup_FormatLog() *

//*************************
//*    Backup_SaveLog     *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* Copy the temporary log file to the specified public-access log file.       *
//*                                                                            *
//* Input  : srcLog  : temp file containing the formatted log data             *
//*          trgLog  : public-access target log file                           *
//*                    If target exists, the new data will be appended to it.  *
//*          dp      : pointer to caller's dialog window                       *
//*          wp      : position for message output                             *
//*          hColor  : header color                                            *
//*          dColor  : data color                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void FileMangler::Backup_SaveLog ( const gString& srcLog, const gString& trgLog, 
                                   NcDialog* dp, winPos wp, attr_t hColor, attr_t dColor )
{
   winPos wpd ;
   short nIndex = (trgLog.findlast( L'/' )) + 1 ;

   if ( (this->Backup_SaveLog ( srcLog, trgLog )) == OK )
      wpd = dp->WriteString ( wp, "Log Written: ", hColor ) ;
   else
      wpd = dp->WriteString ( wp, "Error Writing Log: ", hColor ) ;
   dp->WriteString ( wpd, &trgLog.gstr()[nIndex], dColor, true ) ;

   //* Update the parent dialog's display *
   dp->SetDialogObscured () ;
   this->dWin->RefreshWin () ;
   this->altPtr->RefreshCurrDir ( false ) ;
   this->dWin->SetDialogObscured () ;
   dp->RefreshWin () ;

}  //* End Backup_SaveLog() *

//*************************
//*    Backup_SaveLog     *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* Copy the temporary log file to the specified public-access log file.       *
//*                                                                            *
//* Input  : srcLog  : temp file containing the formatted log data             *
//*          trgLog  : public-access target log file                           *
//*                    If target exists, the new data will be appended to it.  *
//*                                                                            *
//* Returns: OK  if successful                                                 *
//*          ERR if error(s) writing log file                                  *
//******************************************************************************

short FileMangler::Backup_SaveLog ( const gString& srcLog, const gString& trgLog )
{
   bool readError = false,       // 'true' if read error
        writeError = false ;     // 'true' if write error

   //* Open the source file *
   ifstream ifs( srcLog.ustr(), ifstream::in ) ;

   //* Open the target file *
   ofstream ofs( trgLog.ustr(), ofstream::out | ofstream::app ) ;
   if ( ifs.is_open() && ofs.is_open() )
   {
      char  lineData[gsDFLTBYTES] ; // raw UTF-8 input

      //* Copy remainder of source to target *
      bool done = false ;
      while ( ! done )
      {
         ifs.getline ( lineData, gsDFLTBYTES, NEWLINE ) ;
         if ( ifs.good() || ifs.gcount() > ZERO )
            ofs << lineData << endl ;
         else
            done = true ;
      }
   }
   //* Close the files *
   if ( ifs.is_open() )
      ifs.close() ;
   else
      readError = true ;
   if ( ofs.is_open() )
      ofs.close() ;
   else
      writeError = true ;

   return ( (!readError && !writeError) ? OK : ERR ) ;

}  //* End Backup_SaveLog() *

//*************************
//*   Backup_Definition   *
//*************************
//******************************************************************************
//* Read the specified Backup/Archive/Synch definition file and initialize     *
//* caller's data.                                                             *
//*                                                                            *
//* NOTE: If 'exclOnly' == false, then 'btime' member initially contains the   *
//*       current system local time.                                           *
//*                                                                            *
//* Input  : defPath  : path/filename of definition file                       *
//*          bkSet    : receives formatted contents of definition file         *
//*          exclOnly : (optional, false by default)                           *
//*                     'false' : process all records                          *
//*                     'true'  : process only exclusion records               *
//*                                                                            *
//* Returns: OK  if successful                                                 *
//*          ERR if:                                                           *
//*              a) file not found                                             *
//*              b) file contains syntax errors                                *
//******************************************************************************
//* Notes:                                                                     *
//* 1) If reading all records from file, we reset caller's data before reading.*
//* 2) If reading only exclusion records, we modify only 'bex' and 'bexCount'. *
//* 3) We read the entire file, even if an error is encountered, i.e. we do    *
//*    not terminate on error.                                                 *
//* 4) If reading all records, operation type if specified, must be one of:    *
//*               'Backup' 'Archive' 'Synch' (case insensitive)                *
//*    - If record is not present, btype==bsetBackup with no error reported.   *
//*    - If record is invalid, btype==bsetBackup with error reported.          *
//*    - If reading only exclusion records, this record, if present is ignored.*
//* 5) For 'bsrc' and 'btrg', if the record is present and valid, the field    *
//*    is initialized, else cleared.                                           *
//*    - If cleared, then caller must initialize the field (probably CWD).     *
//*    - Note that if neither is specified, then in effect, source==target,    *
//*      which could cause an error at a higher level.                         *
//* 6) 'barc' and 'blog' must be filenames ONLY, not path specifications       *
//*    because they will be written to the directory specified by 'btrg'.      *
//*    If the record is present and valid, the field is initialized, else      *
//*    cleared.                                                                *
//* 7) If a delayed operation is specified, then 'btime' object is initialized,*
//*    else cleared ('epoch' member == ZERO).                                  *
//*                                                                            *
//* n) For exclusion records. 'bex' and 'bexCount' are initialized only if one *
//*    or more valid exclusion records are identified. If no valid exclusion   *
//*    records, then bexCount==ZERO and bex==NULL.                             *
//*                                                                            *
//*                                                                            *
//* Programmer's Note: This algorithm is inefficient in terms of both time and *
//* code space, and we may be able to do a better job. Think about this....    *
//******************************************************************************

short FileMangler::Backup_Definition ( const char* defPath, BSet& bkSet, bool exclOnly )
{
   const wchar_t *bSOURCE  = L"SOURCEDIR",
                 *bTARGET  = L"TARGETDIR",
                 *bBEGIN   = L"OPBEGIN",
                 *bTYPE    = L"OPTYPE",
                 *bARCH    = L"ARCHIVE",
                 *bLOG     = L"LOGFILE",
                 *bEXCL    = L"EXCLUDE",
                 *bBACKUP  = L"Backup",
                 *bARCHIVE = L"Archive",
                 *bSYNCH   = L"Synch",
                 bCOLON    = L':',
                 bSPACE    = L' ',
                 bSLASH    = L'/',
                 bHASH     = L'#' ;

   localTime lt = bkSet.btime ; // copy of system local time from caller
   UINT  xRecords = ZERO ; // count of exclusion records identified in first pass
   short status = OK ;     // return value

   //* If we are reading only exclusion records, do not disturb the other *
   //* data, but if we are processing all records, reset caller's data.   *
   if ( ! exclOnly )
      bkSet.reset() ;

   ifstream ifs ( defPath, ifstream::in ) ;
   if ( ifs.is_open() )
   {
      gString gs ;                     // data-string formatting
      short   gsi ;                    // string index
      char    lineData[gsDFLTBYTES] ;  // raw UTF-8 input
      bool    done = false ;           // loop control

      do
      {
         ifs.getline ( lineData, gsDFLTBYTES, NEWLINE ) ;
         if ( ifs.good() || ifs.gcount() > ZERO )
         {
            gs = lineData ;
            if ( (gs.gstr()[ZERO] == bHASH) ||  // ignore comments
                 (gs.gschars() == 1) )          // ignore empty lines
               continue ;

            if (   ((gsi = gs.find( bEXCL)) >= ZERO)
                     && ((gsi = gs.find( bCOLON, ++gsi )) > ZERO) )
            {  //* If valid exclusion header *
               gs.shiftChars( -(gsi + 1) ) ;    // discard the header
               while ( *gs.gstr() == bSPACE )   // step over whitespace
                  gs.shiftChars( -1 ) ;
               if ( gs.gschars() > 1 )          // if not an empty string
                  ++xRecords ;
            }
            else if ( ! exclOnly )
            {
               if (   ((gsi = gs.find( bSOURCE )) >= ZERO)
                   && ((gsi = gs.find( bCOLON, ++gsi )) > ZERO) )
               {  //* If valid source header.                   *
                  //* Discard the header and leading whitespace *
                  gs.shiftChars( -(gs.scan( ++gsi )) ) ;
                  if ( gs.gschars() > 1 )          // if field contains text data
                  {  //* Expand any environment variable substitutions,    *
                     //* then expand to a full path/filename specification.*
                     //* If gcaRealpath() succeeds, then the directory     *
                     //* _probably_ exists, but lstat to be sure.          *
                     if ( (this->gcaRealpath ( gs )) )
                     {
                        fmFType ft ;
                        if ( (this->fdPtr->TargetExists ( gs, ft )) &&
                             (ft == fmDIR_TYPE) )
                           bkSet.bsrc = gs ;
                        else
                           status = ERR ;
                     }
                     else
                        status = ERR ;
                  }
               }
               else if (   ((gsi = gs.find( bTARGET )) >= ZERO)
                        && ((gsi = gs.find( bCOLON, ++gsi )) > ZERO) )
               {  //* If valid source header.                   *
                  //* Discard the header and leading whitespace *
                  gs.shiftChars( -(gs.scan( ++gsi )) ) ;
                  if ( gs.gschars() > 1 )          // if field contains text data
                  {  //* Expand any environment variable substitutions,    *
                     //* then expand to a full path/filename specification.*
                     //* If gcaRealpath() succeeds, then the directory     *
                     //* _probably_ exists, but lstat to be sure.          *
                     if ( (this->gcaRealpath ( gs )) )
                     {
                        fmFType ft ;
                        if ( (this->fdPtr->TargetExists ( gs, ft )) &&
                             (ft == fmDIR_TYPE) )
                           bkSet.btrg = gs ;
                        else
                           status = ERR ;
                     }
                     else
                        status = ERR ;
                  }
               }
               else if (   ((gsi = gs.find( bBEGIN)) >= ZERO)
                        && ((gsi = gs.find( bCOLON, ++gsi )) > ZERO) )
               {  //* If valid begin header *
                  gs.shiftChars( -(gsi + 1) ) ;    // discard the header
                  while ( *gs.gstr() == bSPACE )   // step over whitespace
                     gs.shiftChars( -1 ) ;
                  if ( gs.gschars() > 1 )          // if field contains text data
                  {
                     int64_t delaySec = ZERO,      // calculate delay
                             delayHrs = ZERO,
                             delayMin = ZERO ;

                     //* Short-form timestamp - add to current time *
                     UINT hh, mm ;
                     if (   ((swscanf ( gs.gstr(), L"%u:%u", &hh, &mm )) == 2)
                         && ((hh >= ZERO) && (hh < 24) && (mm >= ZERO) && (mm < 60)) )
                     {  //* Specified time is a clock time no more than  *
                        //* 24 hours into the future.                    *
                        UINT startSecs = (hh * 60 * 60) + (mm * 60),
                             currentSecs = (lt.hours * 60 * 60) + 
                                           (lt.minutes * 60) + lt.seconds ;
                        //* If start time is later today *
                        if ( startSecs >= currentSecs )
                           bkSet.btime.epoch = lt.epoch + (startSecs - currentSecs ) ;
                        //* Else, start time is tomorrow *
                        else
                        {
                           UINT fullDay = (24 * 60 * 60) ;
                           bkSet.btime.epoch = lt.epoch + fullDay - 
                                               (currentSecs - startSecs ) ;
                        }
                        this->fdPtr->DecodeLocalTime ( bkSet.btime ) ;
                        delaySec = bkSet.btime.epoch - lt.epoch ;
                        delayMin = delaySec / 60 ;
                        delayHrs = delayMin / 60 ;
                        delayMin -= delayHrs * 60 ;
                        bkSet.bhour = int(delayHrs) ;
                        bkSet.bmin  = int(delayMin) ;
                     }
                     //* Long-form timestamp - set absolute time *
                     else
                     {
                        int convCount = 
                           swscanf ( gs.gstr(), L"%hu-%hu-%huT%hu:%hu:%hu",
                                     &bkSet.btime.year, &bkSet.btime.month,
                                     &bkSet.btime.date, &bkSet.btime.hours,
                                     &bkSet.btime.minutes, &bkSet.btime.seconds ) ;
                        if ( convCount == 6 )
                        {  //* Specified time is a specific timestamp. *
                           //* Verify that timestamp refers to a       *
                           //* date/time >= current system local time  *
                           //* AND <= 31 days in the future.           *
                           // Programmer's Note: It is possible that the user 
                           // could specify invalid values e.g. Feb. 30th. 
                           // The underlying system call would accept and 
                           // correctly encode such a value, but it would not be 
                           // what the user wanted. We do only a simple 
                           // range-check here.
                           // See the EncodeEpochTime() method.
                           this->fdPtr->EncodeLocalTime ( bkSet.btime ) ;
                           if ( bkSet.btime.epoch >= lt.epoch )
                           {
                              delaySec = bkSet.btime.epoch - lt.epoch ;
                              delayMin = delaySec / 60 ;
                              delayHrs = delayMin / 60 ;
                              delayMin -= delayHrs * 60 ;
                           }

                           if ( (bkSet.btime.epoch > ZERO) &&      // encoding successful
                                (bkSet.btime.epoch >= lt.epoch) && // target is in the future
                                (delayHrs <= maxHRSP) &&           // <= 31 days
                                (delayMin <= maxMNSP) &&           // <= 59 minutes
                                (bkSet.btime.month <= 12) &&       // valid month
                                (bkSet.btime.date <= 31) &&        // valid date (approx)
                                (bkSet.btime.hours <= 23) &&       // valid hours
                                (bkSet.btime.minutes <= 59) &&     // valid minutes
                                (bkSet.btime.seconds <= 59) )      // valid seconds
                           {
                              bkSet.bhour = int(delayHrs) ;
                              bkSet.bmin  = int(delayMin) ;
                           }
                           else
                           {
                              bkSet.btime.reset() ;
                              bkSet.bhour = bkSet.bmin = ZERO ;
                              status = ERR ;
                           }
                        }
                        else        // invalid timestamp string format
                        {
                           bkSet.btime.reset() ;
                           bkSet.bhour = bkSet.bmin = ZERO ;
                           status = ERR ;
                        }
                     }
                  }
               }
               else if (   ((gsi = gs.find( bTYPE)) >= ZERO)
                        && ((gsi = gs.find( bCOLON, ++gsi )) > ZERO) )
               {  //* If valid operation-type header *
                  gs.shiftChars( -(gsi + 1) ) ;    // discard the header
                  while ( *gs.gstr() == bSPACE )   // step over whitespace
                     gs.shiftChars( -1 ) ;
                  if ( (gs.find( bBACKUP, ZERO, false )) >= ZERO )
                     bkSet.btype = bsetBackup ;
                  else if ( (gs.find( bARCHIVE, ZERO, false )) >= ZERO )
                     bkSet.btype = bsetArchive ;
                  else if ( (gs.find( bSYNCH, ZERO, false )) >= ZERO )
                     bkSet.btype = bsetSynch ;
                  else
                     status = ERR ;
               }
               else if (   ((gsi = gs.find( bARCH)) >= ZERO)
                        && ((gsi = gs.find( bCOLON, ++gsi )) > ZERO) )
               {  //* If valid archive filename header *
                  gs.shiftChars( -(gsi + 1) ) ;    // discard the header
                  while ( *gs.gstr() == bSPACE )   // step over whitespace
                     gs.shiftChars( -1 ) ;
                  if ( gs.gschars() > 1 )          // if field contains text data
                  {
                     //* This must be a filename only, not a path. *
                     //* It may be an invalid filename for a tar   *
                     //* archive. but that will be tested later.   *
                     if ( (gs.find( bSLASH )) < ZERO )
                        bkSet.barc = gs ;
                     else
                        status = ERR ;
                  }
               }
               else if (   ((gsi = gs.find( bLOG)) >= ZERO)
                        && ((gsi = gs.find( bCOLON, ++gsi )) > ZERO) )
               {  //* If valid logfile header *
                  gs.shiftChars( -(gsi + 1) ) ;    // discard the header
                  while ( *gs.gstr() == bSPACE )   // step over whitespace
                     gs.shiftChars( -1 ) ;
                  if ( gs.gschars() > 1 )          // if field contains text data
                  {
                     //* This must be a filename only, not a path. *
                     //* This may be an existing or non-existent   *
                     //* filename, but that will be tested later.. *
                     if ( (gs.find( bSLASH )) < ZERO )
                     {
                        bkSet.blog = gs ;
                        bkSet.blflag = true ;
                     }
                     else
                        status = ERR ;
                  }
               }
               else
               {  //* Not a valid header. If only whitespace, ignore, else Error.*
                  short i ;
                  const wchar_t* wPtr = gs.gstr() ;
                  for ( i = ZERO ; wPtr[i] != nckNULLCHAR ; ++i )
                  {
                     if ( wPtr[i] != bSPACE )
                        status = ERR ;
                  }
               }
            }
         }
         else                       // read failed, we're done
            done = true ;
      }
      while ( ! done ) ;

      //* If there are exclusion records, allocate space for them.*
      if ( (status == OK) && (xRecords > ZERO) )
      {
         //* Delete any existing records (this should never happen).*
         bkSet.bex_reset() ;

         //* Allocate exclusion-record array.*
         bkSet.bex = new bsetExcl[xRecords] ;
         bkSet.bexCount = ZERO ;
         ifs.clear() ;           // reset the EOF flag
         ifs.seekg( ZERO ) ;     // return to top of input file
         done = false ;

         do
         {
            ifs.getline ( lineData, gsDFLTBYTES, NEWLINE ) ;
            if ( ifs.good() || ifs.gcount() > ZERO )
            {
               gs = lineData ;
               if ( (gs.gstr()[ZERO] == bHASH) ||  // ignore comments
                    (gs.gschars() == 1) )          // ignore empty lines
                  continue ;

               if (   ((gsi = gs.find( bEXCL)) >= ZERO)
                   && ((gsi = gs.find( bCOLON, ++gsi )) > ZERO) )
               {  //* If valid exclusion header *
                  gs.shiftChars( -(gsi + 1) ) ;    // discard the header
                  while ( *gs.gstr() == bSPACE )   // step over whitespace
                     gs.shiftChars( -1 ) ;
                  if ( gs.gschars() > 1 )          // if not an empty string
                  {  //* Save the substring and its length.*
                     gs.copy( bkSet.bex[bkSet.bexCount].excl, MAX_PATH ) ;
                     bkSet.bex[bkSet.bexCount].len = gs.gschars() ;
                     if ( ++bkSet.bexCount >= xRecords )
                        done = true ;
                  }
               }
            }
            else                       // read failed, we're done
               done = true ;
         }
         while ( ! done ) ;
      }

      ifs.close() ;                 // close the input file
   }
   else                 // file not found
      status = ERR ;

   return status ;

}  //* End Backup_Definition() *

//*************************
//*   Backup_Exclusions   *
//*************************
//******************************************************************************
//* Read the specified definition file, extracting ONLY the exclusion records. *
//*                                                                            *
//* Input  : exList : name of definition file                                  *
//*                   may be either a full path/filename specification or a    *
//*                   filename only.                                           *
//*                   -- If this is a full filespec, expand any environment or *
//*                      other substitutions and use it.                       *
//*                   -- If a filename only, look for it in the following      *
//*                      sequence:                                             *
//*                      a) the directory specified by the 'bsrc' member,      *
//*                      b) the application's installation directory           *
//*          bkSet  : receives formatted contents of definition file           *
//*                   ('bex' and 'bexCount' members only)                      *
//*                                                                            *
//* Returns: OK  if successful                                                 *
//*          ERR if:                                                           *
//*              a) file not found                                             *
//*              b) file contains syntax errors (in exclusion records)         *
//******************************************************************************
//* Programmer's Note: We ignore errors in path validation because if file     *
//* does not exist, then call to Backup_Definition() will fail.                *
//******************************************************************************

short FileMangler::Backup_Exclusions ( const gString& exList, BSet& bkSet )
{
   const wchar_t bSLASH = L'/' ;
   gString defPath ;
   short status = OK ;              // return value

   //* Determine whether caller has sent us a filename or a filespec.*
   //* If a filename only, create the filespec (see above).          *
   if ( (exList.find( bSLASH )) < ZERO )
   {
      fmFType ft ;
      defPath.compose( "%S/%S", bkSet.bsrc.gstr(), exList.gstr() ) ;
      if ( !(this->fdPtr->TargetExists ( defPath, ft )) )
         defPath.compose( "%s/%S", this->cfgOpt.appPath, exList.gstr() ) ;
   }
   else
      this->gcaRealpath ( defPath, exList.ustr() ) ;

   status = this->Backup_Definition ( defPath.ustr(), bkSet, true ) ;

   return status ;

}  //* End Backup_Exclusions() *

//*************************
//*   Backup_Timestamp    *
//*************************
//******************************************************************************
//* Format the specified date/time into a canonical string of the form:        *
//*                        yyyy-mm-ddThh:mm:ss                                 *
//*                                                                            *
//* Input  : tStamp  : (by reference) receives the formatted timestamp string  *
//*          lt      : an initialized localTime object (by reference)          *
//*                    containing the raw date/time                            *
//*          secs    : (optional, 'true' by default)                           *
//*                    'true'  include the seconds in the output               *
//*                    'false  do not include seconds in the output            *
//*          dow     : (optional, 'false' by default)                          *
//*                    'true'  insert the day-of-the-week into the output      *
//*                    'false' do not insert the day-of-the-week into output   *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void FileMangler::Backup_Timestamp ( gString& tStamp, const localTime& lt, 
                                     bool secs, bool dow )
{
   tStamp.compose( "%04hd-%02hd-%02hdT%02hd:%02hd:%02hd",
                   &lt.year, &lt.month, &lt.date,
                   &lt.hours, &lt.minutes, &lt.seconds ) ;
   if ( ! secs )
      tStamp.limitChars( (tStamp.gschars() - 4) ) ;
   if ( dow )
   {
      tStamp.insert( L' ' ) ;
      tStamp.insert( dowStr[lt.day] ) ;
   }
   
}  //* End Backup_Timestamp() *

//*************************
//*    Backup_DirTree     *
//*************************
//******************************************************************************
//* Perform a backup of all data (or 'selected' data) in source directory to   *
//* target directory.                                                          *
//*  OR                                                                        *
//* Create an archive file in the target directory containing all data (or     *
//* 'selected' data) in the source directory.                                  *
//*                                                                            *
//* Input  : bkSet   : operational parameters                                  *
//*                    if 'btype' == bsetBackup, then update the data in       *
//*                       target directory                                     *
//*                    if 'btype' == bsetArchive, then create an archive of    *
//*                       the source data located in target directory          *
//*                    'bex' contains the list of exclusions                   *
//*                    'bexCount' is the number of exclusion records           *
//*                    'blog' is filespec for operational log                  *
//*                                                                            *
//* Returns: OK  if operation successful                                       *
//*          ERR if user aborted before operation complete                     *
//*              'fupdated' receives number of target files updated            *
//*              'fskipped' receives number of target files requiring          *
//*                         no update (this includes 'excluded' files)         *
//*              'ferrors'  receives the number of files which were            *
//*                         not processed due to access errors or other        *
//*                         errors                                             *
//******************************************************************************
//* Notes:                                                                     *
//* For a backup operation:                                                    *
//*  1) Caller has verified that user has read access to the source directory. *
//*  2) Caller has verified that user has read/write access to target directory*
//*  3) Perform the backup. (If any source data are inaccessible, they will    *
//*     not be included in the backup.)                                        *
//*                                                                            *
//* For an archive operation:                                                  *
//*  1) Caller has verified that user has read access to the source directory. *
//*  2) Caller has verified that target archive does not exist, and that user  *
//*     has read/write access to the target directory.                         *
//*  3) Create the archive (optionally compressed).                            *
//*     All 'selected' data in the source directory will be included in the    *
//*     archive.                                                               *
//*     (If any source data are inaccessible, they will not be archived.)      *
//*                                                                            *
//* For a _scheduled_ backup/synch/archive operation:                          *
//*  1) If user has selected one or more source files, then the operation will *
//*     process ONLY those selected files, even though the files may have been *
//*     modified or deleted during the countdown period.                       *
//*  2) If _all_ data in the source directory is to be processed, then there   *
//*     will be no stale data because the data are dynamically selected and    *
//*     moved to the clipboard just before the operation begins.               *
//*                                                                            *
//*                                                                            *
//******************************************************************************

short FileMangler::Backup_DirTree ( BSet& bkSet )
{
   short status = OK ;        // return value

   //* Clear all selections in the alternate window.*
   this->ClearAltFileListSelections () ;

   //* If user has selected items in the current window, perform backup on *
   //* those items only. Otherwise, select all items in the current window.*
   if ( !(this->fdPtr->IsSelected ( true )) )
      this->fdPtr->SelectFile ( true ) ;

   //* Copy all selected items to the clipboard.*
   if ( (this->fdPtr->Copy2Clipboard ()) != false )
   {
      //* Get total number of files on clipboard *
      bkSet.ftotal = this->fdPtr->ClipboardFiles () ;

      //* Perform the backup/archive operation.*
      if ( bkSet.btype == bsetBackup ) // backup
         this->altPtr->BackupClipboardList ( bkSet ) ;
      else                             // archive
         this->altPtr->ArchiveClipboardList ( bkSet ) ;
   }

   //* Determine whether user aborted the operation.*
   if ( bkSet.ftotal > (bkSet.fupdated + bkSet.fskipped + bkSet.ferrors) )
   {  //* If abort, add number of unprocessed files to 'skipped' count. *
      bkSet.fskipped += bkSet.ftotal - (bkSet.fupdated + bkSet.fskipped + bkSet.ferrors) ;
      status = ERR ;
   }

   //* Clear selections in active window *
   this->fdPtr->DeselectFile ( true ) ;

   return status ;

}  //* End Backup_DirTree() *

//*************************
//*     Synch_DirTree     *
//*************************
//******************************************************************************
//* Perform a synchronization of all data between the two specified            *
//* directories.                                                               *
//*                                                                            *
//* Input  : bkSet   : operational parameters                                  *
//*                    'btype' == bsetSynch                                    *
//*                    'bsrc'  == nominal source directory                     *
//*                    'btrg'  == nominal target directory                     *
//*                    'bex' and 'bexCount' have been reset by caller and      *
//*                               are ignored                                  *
//*                    'blog' is filespec for operational log                  *
//*                                                                            *
//* Returns: OK  if operation successful                                       *
//*          ERR if user aborted before operation complete                     *
//*              'fupdated' receives number of target files updated            *
//*              'fskipped' receives number of target files requiring          *
//*                         no update                                          *
//*              'ferrors'  receives the number of files which were            *
//*                         not processed due to access errors or other        *
//*                         errors                                             *
//******************************************************************************
//* Notes:                                                                     *
//*  1) Caller has verified that user has read AND write access to both source *
//*     and target directories.                                                *
//*  2) There are no 'excluded' files for a sync operation; all files in both  *
//*     source and target directories (except for unsupported filetypes) will  *
//*     be processed.                                                          *
//*  3) Accumulated counts:                                                    *
//*     a) 'ftotal' is the sum of the source files in the first pass PLUS the  *
//*         the source files in the second pass.                               *
//*     b) 'fupdated' is the combined file updates for both passes.            *
//*     c) 'ferrors' is the combined file errors for both passes.              *
//*     d) 'fskipped' is the combined total of files skipped in both passes..  *
//*        - this allows for a skipped source/target pair to be counted only   *
//*          once. This is the major source of 'skipped' files, so it gives    *
//*          the best chance for a correct count.                              *
//*        - If an unsupported filetype is found in the first pass, it will    *
//*          be skipped, but the skip WILL NOT BE reported, This should be     *
//*          an infrequent occurance.                                          *
//*        - If an unsupported filetype is found in the second pass, it will   *
//*          skipped, and the skip WILL BE reported.                           *
//*        - If a pair was 'updated' on the first pass, then the same pair     *
//*          will be counted as 'skipped' on the second pass. To correct this, *
//*          we subtract the number of updates on the first pass from the      *
//*          number of skips on the second pass.                               *
//*        For these reasons, we document that for Synch operations the        *
//*        'skipped' count may be inaccurate.                                  *
//*  4) For technical reasons, we force instantiation of the progress bar      *
//*     during synch operations. This is because the operation is in two       *
//*     separate passes, (source-to-target AND target-to-source), and we need  *
//*     to be consistent between the passes about how the progress bar is used.*
//******************************************************************************

short FileMangler::Synch_DirTree ( BSet& bkSet )
{
   short status = OK ;        // return value

   //* The progress bar will always be instantiated for synch *
   //* operations. (see note 4 above).                        *
   //* Because the progress bar is in two sections, we prepare*
   //* by clearing the area occupied by both sections.        *
   dtbmData    msg = " " ;
   this->dWin->DisplayTextboxMessage ( dwMsgsTB, msg ) ;
   attr_t dlgColor = (attr_t)(this->cfgOpt.cScheme.scheme == ncbcMA ? nc.br : nc.ma), 
          barColor = dlgColor ;
   int cellCountA = this->ic[dwMsgsTB].cols / 2 + 1 ;

   //* Clear all selections in the alternate window *
   this->ClearAltFileListSelections () ;

   //* Select all items in the current window *
   this->fdPtr->SelectFile ( true ) ;

   //* Copy all selected items to the clipboard.    *
   //* (If no files in source dir, skip this step.) *
   if ( (this->fdPtr->Copy2Clipboard ()) != false )
   {
      //* Get total number of files on clipboard *
      bkSet.ftotal = this->fdPtr->ClipboardFiles () ;

      //* Backup source files to target (first pass) *
      this->altPtr->BackupClipboardList ( bkSet, true, false ) ;
   }

   //* Clear selections and remove highlight from active window *
   this->fdPtr->RedrawCurrDir ( false, true ) ;

   //* Determine whether user aborted the operation.*
   if ( bkSet.ftotal > (bkSet.fupdated + bkSet.fskipped + bkSet.ferrors) )
   {  //* If abort, add number of unprocessed files to 'skipped' count. *
      bkSet.fskipped += bkSet.ftotal - (bkSet.fupdated + bkSet.fskipped + bkSet.ferrors) ;
      status = ERR ;
   }

   //* If operation is still in progress *
   if ( status == OK )
   {
      //* The progress bar was instantiated in the first pass, so *
      //* continue it here. This simulates the completed progbar  *
      //* from the first pass.                                    *
      for ( short i = ZERO ; i < cellCountA ; ++i )
      {
         msg.textData[i] = hblock[cellDIV] ;
         msg.colorData[i] = barColor ;
      }
      msg.textData[cellCountA] = NULLCHAR ;
      this->dWin->DisplayTextboxMessage ( dwMsgsTB, msg ) ;

      //* Save the results from first half of operation *
      //* and reset the accumulators.                   *
      UINT  ftotalA   = bkSet.ftotal,
            fupdatedA = bkSet.fupdated,
            fskippedA = bkSet.fskipped,
            ferrorsA  = bkSet.ferrors ;
      bkSet.ftotal = bkSet.fupdated = bkSet.fskipped = bkSet.ferrors = ZERO ;

      //* Exchange 'source' and 'target' paths *
      gString gstmp = bkSet.bsrc ;
      bkSet.bsrc = bkSet.btrg ;
      bkSet.btrg = gstmp ;

      //* Select all items in the inactive window and *
      //* copy all selected items to the clipboard.   *
      this->altPtr->SelectFile ( true ) ;
      if ( (this->altPtr->Copy2Clipboard ()) != false )
      {
         //* Get total number of files on clipboard *
         bkSet.ftotal = this->fdPtr->ClipboardFiles () ;

         //* Perform backup from 'target' to 'source' directory (second pass) *
         this->fdPtr->BackupClipboardList ( bkSet, true, true ) ;
      }

      //* Determine whether user aborted the operation.*
      if ( bkSet.ftotal > (bkSet.fupdated + bkSet.fskipped + bkSet.ferrors) )
      {  //* If abort, add number of unprocessed files to 'skipped' count. *
         bkSet.fskipped += bkSet.ftotal - (bkSet.fupdated + bkSet.fskipped + bkSet.ferrors) ;
         status = ERR ;
      }

      //* Combine the interim results (see notes above) *
      bkSet.ftotal   += ftotalA ;
      bkSet.fupdated += fupdatedA ;
      bkSet.fskipped += fskippedA ;
      bkSet.ferrors  += ferrorsA ;

      //* Remove highlight from inactive window and clear selections *
      this->altPtr->RedrawCurrDir ( false, true ) ;

      //* Show highlight in active window *
      this->fdPtr->RedrawCurrDir () ;

      //* Re-exchange 'source' and 'target' paths so log *
      //* file will be saved to inactive (target) window.*
      gstmp = bkSet.bsrc ;
      bkSet.bsrc = bkSet.btrg ;
      bkSet.btrg = gstmp ;
   }
   else     //* If abort or error, update the active window *
      this->fdPtr->RedrawCurrDir () ;

   return status ;

}  //* End Synch_DirTree() *



//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *
//* - - - - - - - - -   Non-member Methods  - - - - - - - - - - - - - - - - -  *
//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *

//*************************
//*    Backup_Callback    *
//*************************
//******************************************************************************
//* NON-MEMBER METHOD.                                                         *
//* Callback method for the Backup() method input loop.                        *
//* Calculates and displays timestamp for delayed operation based on contents  *
//* of the time-spinner controls.                                              *
//*                                                                            *
//* Input  : cIndex    : index of control with input focus                     *
//*          wkey      : user's key input (not used)                           *
//*          firstTime : 'true' first time called, 'false' otherwise           *
//*                                                                            *
//* Returns: OK                                                                *
//******************************************************************************

short Backup_Callback ( short cIndex, wkeyCode wkey, bool firstTime )
{
   static const char* bkClr = "                           " ;
   static const char* bkNow = " immediately " ;
   static int hh, mm ;           // 'hours' and 'minutes' spinner values

   int nhh, nmm ;
   bkDlg->GetSpinnerValue ( hhSP, nhh ) ;
   bkDlg->GetSpinnerValue ( mmSP, nmm ) ;

   //* If user is updating one of the time spinners *
   //*        (or if this is the first call)        *
   if ( cIndex == hhSP || cIndex == mmSP || firstTime != false )
   {
      gString ts ;               // timestamp display formatting
      if ( nhh != hh || nmm != mm || firstTime != false )
      {  //* Clear the message area *
         bkDlg->WriteString ( bkDlgWp, bkClr, bkDlgColor ) ;

         //* If offset is zero *
         if ( nhh == ZERO && nmm == ZERO )
            bkDlg->WriteString ( bkDlgWp, bkNow, bkDlgColor, true ) ;

         //* Non-zero offset *
         else
         {
            //* Get the current local time and add the offset *
            localTime nlt = *bkDlgLt ;
            nlt.epoch += (nhh * 60 * 60) + (nmm * 60) ;
            gString tstamp ;
            bcEncodeLocalTime ( nlt, tstamp ) ;
            bkDlg->WriteString ( bkDlgWp, tstamp, bkDlgColor, true ) ;
         }

         hh = nhh ;     // save the new values
         mm = nmm ;
      }
   }

   return OK ;

}  //* End Backup_Callback() *

//*************************
//*   bcEncodeLocalTime   *
//*************************
//******************************************************************************
//* NON-MEMBER METHOD.
//* Called by the callback method to format the timestamp for output.          *
//*                                                                            *
//*                                                                            *
//* Input  : ltime  : adjusted local time ('epoch' member)                     *
//*          tstamp : receives formatted timestamp string                      *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void bcEncodeLocalTime ( localTime& ltime, gString& tstamp )
{
   int64_t eTime = ltime.epoch ;
   ltime.reset () ;

   ltime.epoch = eTime ;
   ltime.sysepoch = (time_t)eTime ;   // (possible narrowing)

   Tm bdt ;                // receives broken-down time
   if ( (localtime_r ( &ltime.sysepoch, &bdt )) != NULL )
   {
      //* Translate to localTime format *
      ltime.date    = bdt.tm_mday ;          // today's date
      ltime.month   = bdt.tm_mon + 1 ;       // month
      ltime.year    = bdt.tm_year + 1900 ;   // year
      ltime.hours   = bdt.tm_hour ;          // hour
      ltime.minutes = bdt.tm_min ;           // minutes
      ltime.seconds = bdt.tm_sec ;           // seconds
      ltime.day     = bdt.tm_wday ;          // day-of-week (0 == Sunday)
      ltime.julian  = bdt.tm_yday ;          // Julian date (0 == Jan.01)
   }

   tstamp.compose ( "%s %04hd-%02hd-%02hdT%02hd:%02hd",
                    dowStr[ltime.day], 
                    &ltime.year, &ltime.month, &ltime.date,
                    &ltime.hours, &ltime.minutes ) ;

}  //* End bcEncodeLocalTime() *

#if ENABLE_DEBUGGING_CODE != 0 && DEBUG_BDP != 0
//*************************
//*     Display_BSet      *
//*************************
//******************************************************************************
//* DEBUGGING ONLY - NON-MEMBER METHOD.                                        *
//* Display contents of BSet object.                                           *
//*                                                                            *
//* Input  : dialogROWS : dialog rows                                          *
//*          dialogCOLS : dialog columns                                       *
//*          dialogYPOS : Y position for dialog                                *
//*          dialogXPOS : X position for dialog                                *
//*          bkSet      : data to be displayed                                 *
//*          dColor     : dialog color attribute                               *
//*          status     : (optional, ZERO by default)                          *
//*                       if specified, report status of parameter validation: *
//*                       ERR == one or more invalid parameters                *
//*                       >0  == all parameters verified                       *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Display_BSet ( short dialogROWS, short dialogCOLS,
                    short dialogYPOS, short dialogXPOS,
                    const BSet& bkSet, attr_t dColor, short status )
{
   InitNcDialog dInid( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       dialogYPOS,     // Y offset from upper-left of terminal 
                       dialogXPOS,     // X offset from upper-left of terminal 
                       " Debug bkSet ",// dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       NULL            // pointer to list of control definitions
                     ) ;
   NcDialog* dd = new NcDialog ( dInid ) ;
   if ( (dd->OpenWindow()) == OK )
   {
      winPos wp( 1, 2 ) ;
      if ( status != ZERO )
      {
         gString gx( "Validation %s\n", ((status == ERR) ? "failed" : "successful") ) ;
         wp = dd->WriteParagraph ( wp, gx, dColor ) ;
      }
      dd->WriteString ( wp.ypos++, wp.xpos, bkSet.bsrc, dColor ) ;
      dd->WriteString ( wp.ypos++, wp.xpos, bkSet.btrg, dColor ) ;
      short lindex = bkSet.barc.gschars() > 60 ? 12 : 0 ;
      dd->WriteString ( wp.ypos++, wp.xpos, &bkSet.barc.gstr()[lindex], dColor ) ;
      lindex = bkSet.blog.gschars() > 60 ? 12 : 0 ;
      dd->WriteString ( wp.ypos++, wp.xpos, &bkSet.blog.gstr()[lindex], dColor ) ;

      gString gs( "bex(%u):", &bkSet.bexCount ) ;
      for ( UINT i = ZERO ; i < bkSet.bexCount ; ++i )
         gs.append( " '%S'", bkSet.bex[i].excl ) ;
      dd->WriteString ( wp.ypos++, wp.xpos, gs, dColor ) ;
      gs.compose( "btype: %s\n"
                  "btime: %04hd-%02hd-%02hdT%02hd:%02hd:%02hd (epoch: %lld)\n"
                  "bhour: %d  bmin: %d\n"
                  "blflag:%hhd\n\n", 
                  opName[bkSet.btype], 
                  &bkSet.btime.year, &bkSet.btime.month, &bkSet.btime.date, 
                  &bkSet.btime.hours, &bkSet.btime.minutes, &bkSet.btime.seconds,
                  &bkSet.btime.epoch, 
                  &bkSet.bhour, &bkSet.bmin,
                  &bkSet.blflag ) ;
      wp = dd->WriteParagraph ( wp.ypos, wp.xpos, gs, dColor ) ;

      dd->WriteString ( wp.ypos, wp.xpos, "  Press a Key  ", nc.reR ) ;
      dd->RefreshWin () ;
      nckPause();
   }
   if ( dd != NULL )                      // close the window
      delete ( dd ) ;

}  //* End Display_BSet() *
#endif   // ENABLE_DEBUGGING_CODE && DEBUG_BDP

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

