//********************************************************************************
//* File       : FmConfigi.cpp                                                   *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2005-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in FileMangler.hpp         *
//* Date       : 06-Apr-2025                                                     *
//* Version    : (see FileMangler AppVersion string)                             *
//*                                                                              *
//* Description: This module contains the interactive configuration methods      *
//* for the FileMangler application's configuration sub-program, FmConfig.       *
//*                                                                              *
//* Development Tools See FileMangler.cpp                                        *
//*  Originally developed under: Fedora Release 12,                              *
//*                              Kernel Linux 2.6.31.5-127.fc12.i686.PAE         *
//********************************************************************************

//****************
//* Header Files *
//****************
#include "FmConfig.hpp"             //* FmConfig class definition and constant data


//******************************
//* Local Definitions and Data *
//******************************

//* Indices for controls in main dialog *
//* (see also enum cciArgs)             *
enum ecoControls : short
{
   modPB = ZERO,  // 'Modify' pushbutton
   savPB,         // 'Save' pushbutton
   canPB,         // 'Cancel' pushbutton
   defPB,         // 'Default All' pushbutton
   hlpPB,         // 'Help' pushbutton
   wmRB,          // 'WindowMode' radio button
   soRB,          // 'SortOrder' radio button
   csRB,          // 'CaseSensitive' radio button
   hfRB,          // 'HiddenFiles' radio button
   cdRB,          // 'ConfirmDelete' radio button
   coRB,          // 'ConfirmOverwrite' radio button
   lmRB,          // 'LockMenuBar' radio button
   scRB,          // 'ColorScheme' radio button
   emRB,          // 'Temporary Storage' radio button
   epRB,          // 'ExternalPrograms' radio button
   fdRB,          // 'Favorite Directories' radio button
   kmRB,          // 'KeyMap' radio button
   mcRB,          // 'Mount Commands' radio button
   atRB,          // 'AltTrashcan' radio button
   cfTB,          // Configuration path/filename display
   ecoCONTROLS    // Number of controls defined
} ;

//* Context help messages *
static const char* const ContextHelp[] = 
{ //------------------------------------------------------------------------|
   "Modify parameters for indicated configuration options.",
   "Save edited configuration options to specified configuration file.",
   "Discard all changes. Configuration file will not be modified.",
   "Set all configuration options to their default values.",
   "Display Texinfo page (node) for FileMangler configuration.",
   "Specify the way file data are displayed: Single or Dual window mode.",
   "Specify the option to be used in sorting the display list.",
   "Specify whether filename sorting will be case sensitive.",
   "Specify whether to include 'hidden' files in the file display list.",
   "Specify the rule about asking for confirmation before file deletion.",
   "Specify the rule about asking for confirmation before file overwrite.",
   "Specify whether to lock the application's Menu Bar in the visible state.",
   "Specify the 'color scheme' to be used for FileMangler dialog windows.",
   "Specify whether to enable mouse support.",
   "Specify how external programs are to be launched.",
   "Edit the list of shortcuts to your most-often-visited directories.",
   "Edit command-key shortcuts assigned to actions in the FilMangler menus.",
   "Edit the list of mount commands for non-automount filesystems.",
   "Specify an alternate or non-standard location for desktop's trashcan.",
   "Specify the configuration file where options are to be saved.",
} ;

//* Used as display data in dctSCROLLEXT controls if no data provided *
static const char* seNoFiles = "  none specified  " ;

//*************************
//*   EditConfigOptions   *
//*************************
//******************************************************************************
//* Display and interactively modify all configuration parameters.             *
//* 1. Set the data members of the shared-memory object                        *
//*    a. If specified configuration file exists, it has been read by caller,  *
//*       setting the data members accordingly.                                *
//*    b. If specified configuration file does not exist, the data members     *
//*       have been set to default values by caller.                           *
//* 2. Open a dialog and allow user to modify the settings.                    *
//* 3. Save the data to the target file.                                       *
//*    a. If a valid target file already exists, save the data to it.          *
//*    b. If target does not exist, create it and then save the data to it.    *
//*                                                                            *
//* Input  : cfgCmd : member of enum ConfigCommand: ccINTERACT -or- ccKEYMAP.  *
//*                                                                            *
//* Returns: member of enum ConfigStatus                                       *
//*          NOTE: Data in this->shm is now stale and should not be used by    *
//*                calling application, specifically the un-expanded paths.    *
//******************************************************************************

ConfigStatus FmConfig::EditConfigOptions ( ConfigCommand cfgCmd )
{
   gString  gsStat ;             // context-help messages
   winPos   wpStat( (this->dialogRows - 3), 2 ) ; // context-help position
   winPos   wpVal ;
   short lenStat = this->dialogCols - 4 ; // length of context-help messages
   ConfigStatus status = csOK ;  // return value

   //* Define the controls for the dialog *
   this->icCount = ecoCONTROLS ;
   this->ic = new InitCtrl[this->icCount] ;
   this->ic[modPB] = 
   {  //* 'Modify' pushbutton - - - - - - - - - - - - - - - - - - - - -  modPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(this->dialogRows - 5),  // ulY:       upper left corner in Y
      short(this->dialogCols / 2 - 24),// ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      10,                           // cols:      control columns
      "  M^ODIFY  ",                // dispText:  
      this->pbnColor,               // nColor:    non-focus color
      this->pbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &this->ic[savPB],             // nextCtrl:  link in next structure
   } ;
   this->ic[savPB] = 
   {  //* 'Save' pushbutton - - - - - - - - - - - - - - - - - - - - - -  savPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      this->ic[modPB].ulY,          // ulY:       upper left corner in Y
      short(this->ic[modPB].ulX + this->ic[modPB].cols + 2),// ulX:
      1,                            // lines:     (n/a)
      10,                           // cols:      control columns
      "   ^SAVE   ",                // dispText:  
      this->pbnColor,               // nColor:    non-focus color
      this->pbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &this->ic[canPB],             // nextCtrl:  link in next structure
   } ;
   this->ic[canPB] = 
   {  //* 'Cancel' pushbutton - - - - - - - - - - - - - - - - - - - - -  canPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      this->ic[savPB].ulY,          // ulY:       upper left corner in Y
      short(this->ic[savPB].ulX + this->ic[savPB].cols + 2),// ulX:
      1,                            // lines:     (n/a)
      10,                           // cols:      control columns
      "  ^CANCEL  ",                // dispText:  
      this->pbnColor,               // nColor:    non-focus color
      this->pbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &this->ic[defPB],             // nextCtrl:  link in next structure
   } ;
   this->ic[defPB] = 
   {  //* 'DefaultAll' pushbutton - - - - - - - - - - - - - - - - - - -  defPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      this->ic[canPB].ulY,          // ulY:       upper left corner in Y
      short(this->ic[canPB].ulX + this->ic[canPB].cols + 2),// ulX:
      1,                            // lines:     (n/a)
      13,                           // cols:      control columns
      " ^DEFAULT ALL ",             // dispText:  
      this->pbnColor,               // nColor:    non-focus color
      this->pbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &this->ic[hlpPB],             // nextCtrl:  link in next structure
   } ;
   this->ic[hlpPB] = 
   {  //* 'Help' pushbutton   - - - - - - - - - - - - - - - - - - - - -  hlpPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      this->ic[defPB].ulY,          // ulY:       upper left corner in Y
      short(this->ic[defPB].ulX + this->ic[defPB].cols + 2),// ulX:
      1,                            // lines:     (n/a)
      10,                           // cols:      control columns
      "   ^HELP   ",                // dispText:  
      this->pbnColor,               // nColor:    non-focus color
      this->pbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &this->ic[wmRB],              // nextCtrl:  link in next structure
   } ;
   this->ic[wmRB] = 
   {  //* 'WindowMode' radio button - - - - - - - - - - - - - - - - - -   wmRB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 3-wide
      false,                        // rbSelect:  initially reset
      3,                            // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Window Mode  - - - - - -",   // label:     
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &this->ic[soRB],              // nextCtrl:  link in next structure
   } ;
   this->ic[soRB] = 
   {  //* 'SortOrder' radio button    - - - - - - - - - - - - - - - - -   soRB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 3-wide
      false,                        // rbSelect:  initially reset
      short(this->ic[wmRB].ulY + 1),// ulY:       upper left corner in Y
      short(this->ic[wmRB].ulX),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Sort Order   - - - - - -",   // label:     
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &this->ic[csRB],              // nextCtrl:  link in next structure
   } ;
   this->ic[csRB] = 
   {  //* 'CaseSensitive' radio button  - - - - - - - - - - - - - - - -   csRB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 3-wide
      false,                        // rbSelect:  initially reset
      short(this->ic[soRB].ulY + 1),// ulY:       upper left corner in Y
      short(this->ic[soRB].ulX),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Case Sensitive   - - - -",   // label:     
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &this->ic[hfRB],              // nextCtrl:  link in next structure
   } ;
   this->ic[hfRB] = 
   {  //* 'HiddenFiles' radio button  - - - - - - - - - - - - - - - - -   hfRB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 3-wide
      false,                        // rbSelect:  initially reset
      short(this->ic[csRB].ulY + 1),// ulY:       upper left corner in Y
      short(this->ic[csRB].ulX),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Hidden Files   - - - - -",   // label:     
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &this->ic[cdRB],              // nextCtrl:  link in next structure
   } ;
   this->ic[cdRB] = 
   {  //* 'ConfirmDelete' radio button  - - - - - - - - - - - - - - - -   cdRB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 3-wide
      false,                        // rbSelect:  initially reset
      short(this->ic[hfRB].ulY + 1),// ulY:       upper left corner in Y
      short(this->ic[hfRB].ulX),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Confirm Delete   - - - -",   // label:     
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &this->ic[coRB],              // nextCtrl:  link in next structure
   } ;
   this->ic[coRB] = 
   {  //* 'ConfirmOverwrite' radio button - - - - - - - - - - - - - - -   coRB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 3-wide
      false,                        // rbSelect:  initially reset
      short(this->ic[cdRB].ulY + 1),// ulY:       upper left corner in Y
      short(this->ic[cdRB].ulX),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Confirm Overwrite  - - -",   // label:     
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &this->ic[lmRB],              // nextCtrl:  link in next structure
   } ;
   this->ic[lmRB] = 
   {  //* 'LockMenuBar' radio button  - - - - - - - - - - - - - - - - -   lmRB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 3-wide
      false,                        // rbSelect:  initially reset
      short(this->ic[coRB].ulY + 1),// ulY:       upper left corner in Y
      short(this->ic[coRB].ulX),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Lock Menu Bar  - - - - -",   // label:     
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &this->ic[scRB],              // nextCtrl:  link in next structure
   } ;
   this->ic[scRB] = 
   {  //* 'ColorScheme' radio button    - - - - - - - - - - - - - - - -   scRB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 3-wide
      false,                        // rbSelect:  initially reset
      short(this->ic[lmRB].ulY + 1),// ulY:       upper left corner in Y
      short(this->ic[lmRB].ulX),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Color Scheme   - - - - -",   // label:     
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &this->ic[emRB],              // nextCtrl:  link in next structure
   } ;
   this->ic[emRB] = 
   {  //* 'EnableMouse' radio button  - - - - - - - - - - - - - - - - -   emRB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 3-wide
      false,                        // rbSelect:  initially reset
      short(this->ic[scRB].ulY + 1),// ulY:       upper left corner in Y
      short(this->ic[scRB].ulX),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Enable Mouse Support - -",   // label:     
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &this->ic[epRB],              // nextCtrl:  link in next structure
   } ;
   this->ic[epRB] = 
   {  //* 'ExternalPrograms' radion button  - - - - - - - - - - - - - -   epRB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 3-wide
      false,                        // rbSelect:  initially reset
      short(this->ic[emRB].ulY + 1),// ulY:       upper left corner in Y
      short(this->ic[emRB].ulX),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "External Programs  - - -",   // label:     
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &this->ic[fdRB],              // nextCtrl:  link in next structure
   } ;
   this->ic[fdRB] = 
   {  //* 'FavoriteDirectories' radio button  - - - - - - - - - - - - -   fdRB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 3-wide
      false,                        // rbSelect:  initially reset
      short(this->ic[epRB].ulY + 1),// ulY:       upper left corner in Y
      short(this->ic[epRB].ulX),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Favorite Directories - -",   // label:     
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &this->ic[kmRB],              // nextCtrl:  link in next structure
   } ;
   this->ic[kmRB] = 
   {  //* 'KeyMap' radio button         - - - - - - - - - - - - - - - -   kmRB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 3-wide
      false,                        // rbSelect:  initially reset
      short(this->ic[fdRB].ulY + 1),// ulY:       upper left corner in Y
      short(this->ic[fdRB].ulX),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Command-key Map  - - - -",   // label:     
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &this->ic[mcRB],              // nextCtrl:  link in next structure
   } ;
   this->ic[mcRB] = 
   {  //* 'MountCommands' radio button  - - - - - - - - - - - - - - - -   mcRB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 3-wide
      false,                        // rbSelect:  initially reset
      short(this->ic[kmRB].ulY + 1),// ulY:       upper left corner in Y
      short(this->ic[kmRB].ulX),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Mount Commands   - - - -",   // label:     
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &this->ic[atRB],              // nextCtrl:  link in next structure
   } ;
   this->ic[atRB] = 
   {  //* 'TrashcanPath' radio button - - - - - - - - - - - - - - - -     atRB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 3-wide
      false,                        // rbSelect:  initially reset
      short(this->ic[mcRB].ulY + 1),// ulY:       upper left corner in Y
      short(this->ic[mcRB].ulX),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Alternate Trashcan - - -",   // label:     
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &this->ic[cfTB],              // nextCtrl:  link in next structure
   } ;
   this->ic[cfTB] = 
   {  //* 'Configuration path/filename text box - - - - - - - - - - - -   cfTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(this->dialogRows - 2),  // ulY:       upper left corner in Y
      short(wpStat.xpos + 13),      // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      short(lenStat - 13),          // cols:      control columns
      NULL,                         // dispText:  initially empty
      this->dColor,                 // nColor:    non-focus color
      this->tbfColor,               // fColor:    focus color
      #if LINUX_SPECIAL_CHARS != 0
      tbPathLinux,                  // filter: valid filespec chars (incl. Linux "special")
      #else    // BASIC FILESPEC FILTER
      tbPathName,                   // filter:    valid filename characters
      #endif   // BASIC FILESPEC FILTER
      "Target File:",               // label:     
      ZERO,                         // labY:      
      -13,                          // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    view only
      NULL,                         // nextCtrl:  link in next structure
   } ;


   //* Initial parameters for dialog window *
   InitNcDialog dInit( this->dialogRows,     // number of display lines
                       this->dialogCols,     // number of display columns
                       this->dialogPosY,     // Y offset from upper-left of terminal 
                       this->dialogPosX,     // X offset from upper-left of terminal 
                       NULL,                 // dialog title
                       ncltSINGLE,           // border line-style
                       this->dColor,         // border color attribute
                       this->dColor,         // interior color attribute
                       this->ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   this->dPtr = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (this->dPtr->OpenWindow()) == OK )
   {
      //* Give dialog a title (with different color than border color) *
      gString gsOut ;
      gsOut.compose( L"  %sEdit Configuration Parameters    ", CfgTitle ) ;
      this->dPtr->SetDialogTitle ( gsOut.gstr(), nc.brR ) ;

      LineDef  lDefH(ncltHORIZ, ncltSINGLE, (this->dialogRows - 4), 
                     ZERO, dialogCols, dColor) ;
      this->dPtr->DrawLine ( lDefH ) ;

      //* Draw static text *
      this->dPtr->WriteString ( 2,  2, "   Configuration Options   ", this->dColor | ncuATTR ) ;
      this->dPtr->WriteString ( 2, 30, "        Current Value                         ", 
                                this->dColor | ncuATTR ) ;
      //* Be sure that the config file path/filename will fit into the field.*
      gString gsCfg( this->shm->cfgPath ) ;
      this->ecoPathFit ( gsCfg, this->ic[cfTB].cols ) ;
      this->dPtr->SetTextboxText ( cfTB, gsCfg ) ;

      //* If viewing the keymap, set the keymap radiobutton *
      //* and push the hotkey for the "Modify" pushbutton.  *
      if ( cfgCmd == ccKEYMAP )
      {
         this->dPtr->SetRadiobuttonState ( kmRB, true ) ;
         wkeyCode wktmp( nckC_O, wktFUNKEY ) ;
         this->dPtr->UngetKeyInput ( wktmp ) ;
      }

      this->dPtr->RefreshWin () ;   // make everything visible

      winPos   wpVal( this->ic[wmRB].ulY, 31 ) ; // current-value display position
      uiInfo   Info ;               // user interface data returned here
      short    icIndex = ZERO ;     // index of control with input focus
      bool     settingMod = true,   // 'true' if one or more parameters modified
               done = false ;       // loop control
      while ( ! done )
      {
         //* Display context help for control with focus *
         this->dPtr->ClearLine ( wpStat.ypos, false ) ;
         gsStat.compose( 
            L" %s                                                                        ~",
            ContextHelp[icIndex] ) ;
         gsStat.limitCols( lenStat ) ;
         this->dPtr->WriteString ( wpStat.ypos, wpStat.xpos, gsStat, this->tbnColor, true ) ;

         //* Update the list of current settings for each config option *
         if ( settingMod )
         {
            this->ecoDisplayCurrent ( wpVal ) ;
            settingMod = false ;
         }

         //* If focus is currently on a Pushbutton *
         if ( this->ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey == false )
               icIndex = this->dPtr->EditPushbutton ( Info ) ;
            else
               Info.HotData2Primary () ;
            if ( Info.dataMod != false )
            {
               //* Modify the configuration parameters *
               if ( Info.ctrlIndex == modPB )
               {
                  this->dPtr->ClearLine ( wpStat.ypos, false ) ;
                  settingMod = this->ecoEditParms () ;
               }
               //* Save edited data to target configuration file.       *
               else if ( Info.ctrlIndex == savPB )
               {
                  //* Validate the temp storage path before allowing write.*
                  status = this->WriteConfigFile () ;
                  done = true ;
               }
               //* Cancel the operation. Configuration file not updated.*
               else if ( Info.ctrlIndex == canPB )
               {
                  done = true ;
               }
               //* Set all options to default values *
               else if ( Info.ctrlIndex == defPB )
               {  //* Reset all config options EXCEPT:         *
                  //* appPath: path of application executable, *
                  //* tfPath : temp-files directory            *
                  //* trPath : trashcan filespec               *
                  //* bkPath : filespec of backup definition   *
                  this->shm->cfgOpt.Reset() ;

                  this->shm->favCount = ZERO ;  // discard favorite dirs (except 'home')
                  gsOut = ConfigDfltFavorite ;
                  gsOut.copy ( this->shm->favPath[this->shm->favCount++], MAX_PATH ) ;
                  this->shm->mntCount = ZERO ;  // discard mount commands

                  this->shm->lockMb = false ;   // MenuBar not locked by default

                  //* Keymap is inactive by default with no keymap file specified.*
                  this->shm->keyMap.reset() ;

                  //* Remaining config data is housekeeping information. *
                  //* this->shm->msg[]       process messages
                  //* this->shm->msgCount    process message count
                  //* this->shm->ccCmd       config command to perform (already processes)
                  //* this->shm->fsStatus    returned status
                  //* this->shm->verbose     'extended diagnostics' flag
                  settingMod = true ;
               }
               //* Give user a clue *
               else if ( Info.ctrlIndex == hlpPB )
               {
                  this->ConfigHelp ( wpStat ) ;
               }
            }
         }
         //* If focus is currently on a Radio Button *
         else if ( this->ic[icIndex].type == dctRADIOBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = this->dPtr->EditRadiobutton ( Info ) ;
         }
         //* If focus is currently on a Text Box *
         else if ( this->ic[icIndex].type == dctTEXTBOX )
         {
            //Info.viaHotkey = false ;      // ignore hotkey data
            //icIndex = this->dPtr->EditTextbox ( Info ) ;
            /* Currently, view only */
         }
         //* Move focus to appropriate control *
         if ( ! done && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = this->dPtr->PrevControl () ; 
            else
               icIndex = this->dPtr->NextControl () ;
         }
      }     // while(!done)
   }        // if((this->dPtr->OpenWindow())==OK)

   if ( this->dPtr != NULL )        // close the window
   {
      delete ( this->dPtr ) ;
      this->dPtr = NULL ;
   }
   return status ;

}  //* End EditConfigOptions() *

//*************************
//*     ecoEditParms      *
//*************************
//******************************************************************************
//* For each 'selected' radio button in this-ic[] array, allow user to         *
//* edit the associated parameter(s).                                          *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if one or more parameters modified, else 'false'           *
//******************************************************************************

bool FmConfig::ecoEditParms ( void )
{
   //* Basic control definitions. Note that we allocate an array with the *
   //* maximum number of controls, but we populate ONLY the first two.    *
   //* Called method is responsible for populating any additional controls*
   //* needed beyond the 'OK' and 'CANCEL' buttons.                       *
   const short 
         icdROWS  = 19,
         icdCOLS  = 74,
         icdULY   = this->dialogPosY + this->dialogRows / 2 - icdROWS / 2 - 1,
         icdULX   = this->dialogPosX + this->dialogCols / 2 - icdCOLS / 2,
         icdCTRLS = MAX_DIALOG_CONTROLS ;
   InitCtrl icd[icdCTRLS] =            // control definitions
   {
   {  //* 'OK' pushbutton   - - - - - - - - - - - - - - - - - - - - - -   okPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(icdROWS - 3),           // ulY:       upper left corner in Y
      short(icdCOLS / 2 - 10),      // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      8,                            // cols:      control columns
      "   OK   ",                   // dispText:  
      this->pbnColor,               // nColor:    non-focus color
      this->pbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &icd[1]                       // nextCtrl:  link in next structure
   },
   icd[1] =
   {  //* 'CANCEL' pushbutton - - - - - - - - - - - - - - - - - - - - -  canPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      icd[0].ulY,                   // ulY:       upper left corner in Y
      short(icd[0].ulX + 12),       // ulX:
      1,                            // lines:     (n/a)
      8,                            // cols:      control columns
      " CANCEL ",                   // dispText:  
      this->pbnColor,               // nColor:    non-focus color
      this->pbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   },
   } ;

   //* Initial parameters for dialog window, passed to sub-methods.*
   InitNcDialog dInit
   ( 
      icdROWS,              // number of display lines
      icdCOLS,              // number of display columns
      icdULY,               // Y offset from upper-left of terminal 
      icdULX,               // X offset from upper-left of terminal 
      NULL,                 // dialog title (see below)
      ncltSINGLE,           // border line-style
      this->dColor,         // border color attribute
      this->dColor,         // interior color attribute
      icd                   // pointer to list of control definitions
   ) ;

   bool  pmods = false ;            // return value

   for ( short icIndex = ZERO ; icIndex < this->icCount ; icIndex++ )
   {
      if ( this->ic[icIndex].type == dctRADIOBUTTON )
      {
         //* Has user requested to edit this parameter? *
         bool rbSet ;
         this->dPtr->GetRadiobuttonState ( icIndex, rbSet ) ;
         if ( rbSet )
         {
            //* Save the parent dialog's display data *
            this->dPtr->SetDialogObscured () ;
   
            switch ( icIndex )
            {
               case wmRB:
                  dInit.dTitle = "  Edit 'Window Mode'  " ;
                  if ( (this->ecoEditWM ( dInit )) != false )
                     pmods = true ;
                  break ;
               case soRB:
               case hfRB:
               case csRB:
                  dInit.dTitle = "  Edit Sort Options  " ;
                  if ( (this->ecoEditSO ( dInit )) != false )
                     pmods = true ;
                  break ;
               case cdRB:
                  dInit.dTitle = "  Edit 'Confirm Delete'  " ;
                  if ( (this->ecoEditCD ( dInit )) != false )
                     pmods = true ;
                  break ;
               case coRB:
                  dInit.dTitle = "  Edit 'Confirm Overwrite'  " ;
                  if ( (this->ecoEditCO ( dInit )) != false )
                     pmods = true ;
                  break ;
               case lmRB:
                  dInit.dTitle = "  Edit 'Lock Menu Bar'  " ;
                  if ( (this->ecoEditLM ( dInit )) != false )
                     pmods = true ;
                  break ;
               case atRB:
                  dInit.dTitle = "  Edit Trashcan Options  " ;
                  if ( (this->ecoEditTS ( dInit )) != false )
                     pmods = true ;
                  break ;
               case scRB:
                  dInit.dTitle = "  Edit Color Scheme  " ;
                  if ( (this->ecoEditCS ( dInit )) != false )
                     pmods = true ;
                  break ;
               case emRB:
                  dInit.dTitle = "  Edit 'Enable Mouse Support'  " ;
                  if ( (this->ecoEditEM ( dInit )) != false )
                     pmods = true ;
                  break ;
               case epRB:
                  dInit.dTitle = "  Edit 'External Programs'  " ;
                  if ( (this->ecoEditEP ( dInit )) != false )
                     pmods = true ;
                  break ;
               case fdRB:
                  dInit.dTitle = "  Edit Favorite Directories  " ;
                  if ( (this->ecoEditFD ( dInit )) != false )
                     pmods = true ;
                  break ;
               case kmRB:
                  dInit.dTitle = "  Edit Command-key Map  " ;
                  if ( (this->ecoEditKM ( dInit )) != false )
                     pmods = true ;
                  break ;
               case mcRB:
                  dInit.dTitle = "  Edit Mount Commands  " ;
                  if ( (this->ecoEditMC ( dInit )) != false )
                     pmods = true ;
                  break ;
               default:    // (not likely)
                  break ;
            }
            //* Restore parent dialog's display data *
            this->dPtr->RefreshWin () ;
            //* Reset the state of the button(s) to indicate modifications complete.*
            this->dPtr->SetRadiobuttonState ( icIndex, false ) ;
            if ( icIndex == soRB || icIndex == csRB || icIndex == hfRB )
            {
               this->dPtr->SetRadiobuttonState ( soRB, false ) ;
               this->dPtr->SetRadiobuttonState ( csRB, false ) ;
               this->dPtr->SetRadiobuttonState ( hfRB, false ) ;
            }
         }
      }
   }
   return pmods ;

}  //* End ecoEditParms() *

//*************************
//*       ecoEditWM       *
//*************************
//******************************************************************************
//* Interactively set the Window Mode option.                                  *
//*                                                                            *
//* Input  : dInit  : dialog description (first 2 controls defined)            *
//*                                                                            *
//* Returns: 'true' if user modifies settings, else 'false'                    *
//******************************************************************************

bool FmConfig::ecoEditWM ( InitNcDialog& dInit )
{
   //* Create a list of needed controls *
   enum dpCtrls { okPB, canPB, swinRB, dwinRB, twinRB, dpcCOUNT } ;
   char Labels[3][MAX_LABEL_CHARS] ; 
   dInit.ctrlPtr[canPB].nextCtrl = &dInit.ctrlPtr[swinRB] ;
   dInit.ctrlPtr[swinRB] = 
   {  //* 'SingleWin' radio button  - - - - - - - - - - - - - - - - - - swinRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially reset
      4,                            // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[0],                    // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &dInit.ctrlPtr[dwinRB],       // nextCtrl:  link in next structure
   } ;
   dInit.ctrlPtr[dwinRB] = 
   {  //* 'DualWin' radio button    - - - - - - - - - - - - - - - - - - dwinRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially reset
      short(dInit.ctrlPtr[swinRB].ulY + 3),// ulY:       upper left corner in Y
      short(dInit.ctrlPtr[swinRB].ulX),// ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[1],                    // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &dInit.ctrlPtr[twinRB],       // nextCtrl:  link in next structure
   } ;
   dInit.ctrlPtr[twinRB] = 
   {  //* 'TermWin' radio button    - - - - - - - - - - - - - - - - - - twinRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially reset
      short(dInit.ctrlPtr[dwinRB].ulY + 3),// ulY:       upper left corner in Y
      short(dInit.ctrlPtr[dwinRB].ulX),// ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[2],                    // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   } ;

   //* Construct control labels *
   gString gsOut ;
   gsOut.compose( L"'%S' ", 
                  wmArgs[wmSINGLE_WIN].str ) ;
   gsOut.copy( Labels[0], MAX_LABEL_CHARS ) ;
   gsOut.compose( L"'%S' ", 
                  wmArgs[wmDUAL_WIN].str ) ;
   gsOut.copy( Labels[1], MAX_LABEL_CHARS ) ;
   gsOut.compose( L"'%S'  (default)", 
                  wmArgs[wmTERM_WIN].str ) ;
   gsOut.copy( Labels[2], MAX_LABEL_CHARS ) ;

   //* 'Select' the radio button for the current selection *
   WinMode  wmOrig = this->shm->cfgOpt.winMode ;
   short origIndex = wmOrig == wmSINGLE_WIN ? swinRB : 
                     (wmOrig == wmDUAL_WIN ? dwinRB : twinRB) ;
   dInit.ctrlPtr[origIndex].rbSelect = true ;

   bool pmods = false ;             // return value

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;
   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Group all radio buttons in the dialog *
      short XorGroup[] = { swinRB, dwinRB, twinRB, -1 } ;
      dp->GroupRadiobuttons ( XorGroup ) ;

      //* Display static text *
      winPos wp ;
      wp.ypos = dInit.ctrlPtr[swinRB].ulY + dInit.ctrlPtr[swinRB].labY + 1 ;
      wp.xpos = dInit.ctrlPtr[swinRB].ulX + dInit.ctrlPtr[swinRB].labX ;
      dp->WriteString ( wp, 
                        "- Open application with one file-display window.", 
                        this->expColor ) ;
      wp.ypos = dInit.ctrlPtr[dwinRB].ulY + dInit.ctrlPtr[dwinRB].labY + 1 ;
      wp.xpos = dInit.ctrlPtr[dwinRB].ulX + dInit.ctrlPtr[dwinRB].labX ;
      dp->WriteString ( wp, 
                        "- Open application with dual file-display windows.", 
                        this->expColor ) ;
      wp.ypos = dInit.ctrlPtr[twinRB].ulY + dInit.ctrlPtr[twinRB].labY + 1 ;
      wp.xpos = dInit.ctrlPtr[twinRB].ulX + dInit.ctrlPtr[twinRB].labX ;
      dp->WriteString ( wp, 
                        "- Determine display mode according to size of terminal window.", 
                        this->expColor ) ;

      dp->RefreshWin () ;           // make everything visible

      uiInfo   Info ;               // user interface data returned here
      short    icIndex = ZERO ;     // index of control with input focus
      bool     done = false ;       // loop control
      while ( ! done )
      {
         //* If focus is currently on a Pushbutton *
         if ( dInit.ctrlPtr[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == okPB )
               {  //* Get user's selection and store it in our data member *
                  short newIndex = dp->GetRbGroupSelection ( swinRB ) ;
                  if ( newIndex != origIndex )
                  {
                     this->shm->cfgOpt.winMode = newIndex == swinRB ? wmSINGLE_WIN :
                                       newIndex == dwinRB ? wmDUAL_WIN : wmTERM_WIN ;
                     pmods = true ;
                  }
               }
               done = true ;
            }
         }
         //* If focus is currently on a Radio Button *
         else if ( dInit.ctrlPtr[icIndex].type == dctRADIOBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditRadiobutton ( Info ) ;
         }
         //* Move focus to appropriate control *
         if ( ! done  && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }     // while(!done)
   }        // if((this->dPtr->OpenWindow())==OK)
   if ( dp != NULL )
      delete dp ;
   return pmods ;

}  //* End of ecoEditWM() *

//*************************
//*       ecoEditSO       *
//*************************
//******************************************************************************
//* Interactively set sort options for display of file lists.                  *
//*                                                                            *
//* Input  : dInit  : dialog description (first 2 controls defined)            *
//*                                                                            *
//* Returns: 'true' if user modifies settings, else 'false'                    *
//******************************************************************************
//* Programmer's Note:                                                         *
//*  - Display string initialization may seem more complex than necessary, but *
//*    the soArgs[] array order is different from display order for technical  *
//*    reasons, and it is designed for easy internationalization.              *
//*  - Sort-option display items are in the same order as the application menu.*
//******************************************************************************

bool FmConfig::ecoEditSO ( InitNcDialog& dInit )
{
   //* Create a list of needed controls *
   enum dpCtrls { okPB, canPB, sortSB, caseRB, hideRB, dpcCOUNT } ;
   char Labels[3][MAX_LABEL_CHARS] ;
   attr_t  singleColor[2] = { attrDFLT, this->tbfColor } ;

   //* Construct control labels *
   gString gsOut ;
   gsOut.compose( L" '%S' ", 
                  CfgParm[cciSO].str ) ;
   gsOut.copy( Labels[0], MAX_LABEL_CHARS ) ;
   gsOut.compose( L" '%S' ", 
                  CfgParm[cciCS].str ) ;
   gsOut.copy( Labels[1], MAX_LABEL_CHARS ) ;
   gsOut.compose( L" '%S' ", 
                  CfgParm[cciHF].str ) ;
   gsOut.copy( Labels[2], MAX_LABEL_CHARS ) ;

   //* Construct display text for sortSB *
   const char* sortDesc[] = 
   {
      " - by filename        (ascending) x",
      "- by filename       (descending) x",
      " - by modify date     (ascending) x",
      "- by modify date    (descending) x",
      " - by file size       (ascending) x",
      "- by file size      (descending) x",
      "  - by file extension  (ascending) x",
      " - by file extension (descending) x",
      " - by file type (Dir,Reg,Link...) x",
      "- by file type   (reverse order) x",
      " - no sort,  (as stored on media) x",
   } ;
   //* This array maps the order of items in the Scroll Box to the order of *
   //* items in the soArgs[] array, so initialization can be done in a loop.*
   const short sortOrder[] = { 2, 1, 4, 3, 6, 5, 10, 9, 8, 7, 0 } ;
   //* And this array maps the members of enum fmSort to the items in the   *
   //* Scroll Box.                                                          *
   fmSort sortOpt[] = { fmNAME_SORT, fmNAMEr_SORT, fmDATE_SORT, fmDATEr_SORT,
                        fmSIZE_SORT, fmSIZEr_SORT, fmEXT_SORT,  fmEXTr_SORT,
                        fmTYPE_SORT, fmTYPEr_SORT, fmNO_SORT } ;

   const short sortWidth = 42 ;  // display columns inside sortSB control
   char soStrings[CfgParm[cciSO].argcnt][sortWidth+1] ;
   for ( short sosIndex = ZERO ; sosIndex < CfgParm[cciSO].argcnt ; sosIndex++ )
   {
      gsOut.compose( L" '%S' %s", soArgs[sortOrder[sosIndex]].str, sortDesc[sosIndex] ) ;
      gsOut.limitCols( sortWidth ) ;
      gsOut.copy( soStrings[sosIndex], MAX_LABEL_CHARS ) ;
   }

   //* Determine the current settings *
   short origSort = ZERO ;
   while ( (this->shm->cfgOpt.sortOption != sortOpt[origSort]) && 
           (sortOpt[origSort] != fmNO_SORT) )
      ++origSort ;
   bool  origCase = this->shm->cfgOpt.caseSensitive,
         origHide = this->shm->cfgOpt.showHidden ;

   dInit.ctrlPtr[canPB].nextCtrl = &dInit.ctrlPtr[sortSB] ;
   dInit.ctrlPtr[sortSB] = 
   { //* 'SortOption' Scroll Box - - - - - - - - - - - - - - - - - - - -sortSB *
      dctSCROLLBOX,                 // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      2,                            // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      short(CfgParm[cciSO].argcnt + 2),// lines:     control lines
      short(sortWidth + 2),         // cols:      control columns
      (const char*)&soStrings,      // dispText:  n/a
      this->dColor,                 // nColor:    non-focus border color
      (this->dColor & ~ncrATTR),    // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      Labels[0],                    // label:     
      ZERO,                         // labY:      offset from control's ulY
      ZERO,                         // labX       offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      CfgParm[cciSO].argcnt,        // scrItems:  number of elements in text/color arrays
      origSort,                     // scrSel:    index of initial highlighted element
      singleColor,                  // scrColor:  single-color data display
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &dInit.ctrlPtr[caseRB],       // nextCtrl:  link in next structure
   } ;
   dInit.ctrlPtr[caseRB] = 
   {  //* 'CaseSensitive' radio button  - - - - - - - - - - - - - - - - caseRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      origCase,                     // rbSelect:  initial value
      short(dInit.ctrlPtr[sortSB].ulY + 2), // ulY:       upper left corner in Y
      short(dInit.ctrlPtr[sortSB].ulX + dInit.ctrlPtr[sortSB].cols + 2), // ulX:
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[1],                    // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &dInit.ctrlPtr[hideRB],       // nextCtrl:  link in next structure
   } ;
   dInit.ctrlPtr[hideRB] = 
   {  //* 'HiddenFiles' radio button    - - - - - - - - - - - - - - - - hideRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      origHide,                     // rbSelect:  initial value
      short(dInit.ctrlPtr[caseRB].ulY + 4 ), // ulY:       upper left corner in Y
      dInit.ctrlPtr[caseRB].ulX,    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[2],                    // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   } ;

   bool pmods = false ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;
   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Display static text *
      winPos wp ;
      wp.ypos = dInit.ctrlPtr[caseRB].ulY + dInit.ctrlPtr[caseRB].labY + 1 ;
      wp.xpos = dInit.ctrlPtr[caseRB].ulX + dInit.ctrlPtr[caseRB].labX ;
      dp->WriteParagraph ( wp, 
                        " if selected, sort\n is case sensitive", this->expColor ) ;
      wp.ypos = dInit.ctrlPtr[hideRB].ulY + dInit.ctrlPtr[hideRB].labY + 1 ;
      wp.xpos = dInit.ctrlPtr[hideRB].ulX + dInit.ctrlPtr[hideRB].labX ;
      dp->WriteParagraph ( wp, 
                        " if selected,\n show hidden files", this->expColor ) ;

      dp->RefreshWin () ;           // make everything visible

      uiInfo   Info ;               // user interface data returned here
      short    icIndex = ZERO ;     // index of control with input focus
      bool     done = false ;       // loop control
      while ( ! done )
      {
         //* If focus is currently on a Pushbutton *
         if ( dInit.ctrlPtr[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == okPB )
               {  //* Get user's selections and store them in our data members *
                  short newSort = dp->GetScrollboxSelect ( sortSB ) ;
                  bool  newCase, newHide ;
                  dp->GetRadiobuttonState ( caseRB, newCase ) ;
                  dp->GetRadiobuttonState ( hideRB, newHide ) ;
                  if ( newSort != origSort )
                  { this->shm->cfgOpt.sortOption = sortOpt[newSort] ; pmods = true ; }
                  if ( newCase != origCase )
                  { this->shm->cfgOpt.caseSensitive = newCase ; pmods = true ; }
                  if ( newHide != origHide )
                  { this->shm->cfgOpt.showHidden = newHide ; pmods = true ; }
               }
               done = true ;
            }
         }
         //* If focus is currently on a Scroll Box control *
         else if ( dInit.ctrlPtr[icIndex].type == dctSCROLLBOX )
         {
            Info.viaHotkey = false ;      // ignore hotkey data
            icIndex = dp->EditScrollbox ( Info ) ;
         }
         //* If focus is currently on a Radio Button *
         else if ( dInit.ctrlPtr[icIndex].type == dctRADIOBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditRadiobutton ( Info ) ;
         }
         //* Move focus to appropriate control *
         if ( ! done  && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }     // while(!done)
   }        // if((this->dPtr->OpenWindow())==OK)
   if ( dp != NULL )
      delete dp ;
   return pmods ;
}  //* End of ecoEditSO() *

//*************************
//*       ecoEditCD       *
//*************************
//******************************************************************************
//* Interactively set the 'confirm file deletion' option.                      *
//*                                                                            *
//* Input  : dInit  : dialog description (first 2 controls defined)            *
//*                                                                            *
//* Returns: 'true' if user modifies settings, else 'false'                    *
//******************************************************************************

bool FmConfig::ecoEditCD ( InitNcDialog& dInit )
{
   //* Construct control labels *
   const char* Labels[2] = 
   {
      "- Always ask for confirmation when moving files to trashcan.",
      "- Never ask for confirmation when moving files to trashcan.\n"
      "\n"
      "\n"
      "Note that confirmation is always required for non-reversible\n"
      "file deletions and operations involving entire directory trees."
   } ;

   bool pmods = false,
        dfltVal = CfgParm[cciCD].dflt == ZERO ? true : false,
        origVal = this->shm->cfgOpt.confirmDelete,
        newVal  = this->ecoBinaryDecision ( dInit, Labels[0], Labels[1], origVal, dfltVal ) ;

   if ( origVal != newVal )
   {
      this->shm->cfgOpt.confirmDelete = newVal ;
      pmods = true ;
   }
   return pmods ;

}  //* End of ecoEditCD() *

//*************************
//*       ecoEditCO       *
//*************************
//******************************************************************************
//* Interactively set the 'confirm file overwrite' option.                     *
//*                                                                            *
//* Input  : dInit  : dialog description (first 2 controls defined)            *
//*                                                                            *
//* Returns: 'true' if user modifies settings, else 'false'                    *
//******************************************************************************

bool FmConfig::ecoEditCO ( InitNcDialog& dInit )
{
   //* Create a list of needed controls *
   enum dpCtrls { okPB, canPB, neverRB, newerRB, alwaysRB, dpcCOUNT } ;
   dInit.ctrlPtr[canPB].nextCtrl = &dInit.ctrlPtr[neverRB] ;
   char Labels[3][MAX_LABEL_CHARS] ;
   dInit.ctrlPtr[neverRB] = 
   {  //* 'Never' radio button      - - - - - - - - - - - - - - - - - -neverRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially reset
      4,                            // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[0],                    // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &dInit.ctrlPtr[newerRB],      // nextCtrl:  link in next structure
   } ;
   dInit.ctrlPtr[newerRB] = 
   {  //* 'ConfirmNewer' radio button - - - - - - - - - - - - - - - -  newerRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially reset
      short(dInit.ctrlPtr[neverRB].ulY + 3),// ulY:       upper left corner in Y
      short(dInit.ctrlPtr[neverRB].ulX),// ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[1],                    // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &dInit.ctrlPtr[alwaysRB],     // nextCtrl:  link in next structure
   } ;
   dInit.ctrlPtr[alwaysRB] = 
   {  //* 'TermWin' radio button    - - - - - - - - - - - - - - - - - - twinRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially reset
      short(dInit.ctrlPtr[newerRB].ulY + 3),// ulY:       upper left corner in Y
      short(dInit.ctrlPtr[newerRB].ulX),// ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[2],                    // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   } ;

   //* Construct control labels *
   gString gsOut ;
   gsOut.compose( L"'%S' ", 
                  coArgs[0].str ) ;
   gsOut.copy( Labels[0], MAX_LABEL_CHARS ) ;
   gsOut.compose( L"'%S'  (default)", 
                  coArgs[1].str ) ;
   gsOut.copy( Labels[1], MAX_LABEL_CHARS ) ;
   gsOut.compose( L"'%S' ", 
                  coArgs[2].str ) ;
   gsOut.copy( Labels[2], MAX_LABEL_CHARS ) ;

   //* 'Select' the radio button for the current selection *
   owTypes  coOrig = this->shm->cfgOpt.overWrite ;
   short origIndex = coOrig == owNO_CONFIRM ? neverRB : 
                     (coOrig == owCONFIRM_NEWER ? newerRB : alwaysRB) ;
   dInit.ctrlPtr[origIndex].rbSelect = true ;

   bool pmods = false ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;
   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Group all radio buttons in the dialog *
      short XorGroup[] = { neverRB, newerRB, alwaysRB, -1 } ;
      dp->GroupRadiobuttons ( XorGroup ) ;

      //* Display static text *
      winPos wp ;
      wp.ypos = dInit.ctrlPtr[neverRB].ulY + dInit.ctrlPtr[neverRB].labY + 1 ;
      wp.xpos = dInit.ctrlPtr[neverRB].ulX + dInit.ctrlPtr[neverRB].labX ;
      dp->WriteString ( wp, 
                        "- Do not ask before overwriting existing target files.", 
                        this->expColor ) ;
      wp.ypos = dInit.ctrlPtr[newerRB].ulY + dInit.ctrlPtr[newerRB].labY + 1 ;
      wp.xpos = dInit.ctrlPtr[newerRB].ulX + dInit.ctrlPtr[newerRB].labX ;
      dp->WriteString ( wp, 
                        "- Ask before overwriting a newer file with an older file.", 
                        this->expColor ) ;
      wp.ypos = dInit.ctrlPtr[alwaysRB].ulY + dInit.ctrlPtr[alwaysRB].labY + 1 ;
      wp.xpos = dInit.ctrlPtr[alwaysRB].ulX + dInit.ctrlPtr[alwaysRB].labX ;
      dp->WriteString ( wp, 
                        "- Always ask before overwriting existing target files.", 
                        this->expColor ) ;

      dp->RefreshWin () ;           // make everything visible

      uiInfo   Info ;               // user interface data returned here
      short    icIndex = ZERO ;     // index of control with input focus
      bool     done = false ;       // loop control
      while ( ! done )
      {
         //* If focus is currently on a Pushbutton *
         if ( dInit.ctrlPtr[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == okPB )
               {  //* Get user's selection and store it in our data member *
                  short newIndex = dp->GetRbGroupSelection ( neverRB ) ;
                  if ( newIndex != origIndex )
                  {
                     this->shm->cfgOpt.overWrite = newIndex == neverRB ? owNO_CONFIRM :
                                       newIndex == newerRB ? owCONFIRM_NEWER : owCONFIRM_ALL ;
                     pmods = true ;
                  }
               }
               done = true ;
            }
         }
         //* If focus is currently on a Radio Button *
         else if ( dInit.ctrlPtr[icIndex].type == dctRADIOBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditRadiobutton ( Info ) ;
         }
         //* Move focus to appropriate control *
         if ( ! done  && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }     // while(!done)
   }        // if((this->dPtr->OpenWindow())==OK)
   if ( dp != NULL )
      delete dp ;
   return pmods ;
}  //* End of ecoEditCO() *

//*************************
//*       ecoEditLM       *
//*************************
//******************************************************************************
//* Interactively set the 'lock menu bar' option.                              *
//*                                                                            *
//* Input  : dInit  : dialog description (first 2 controls defined)            *
//*                                                                            *
//* Returns: 'true' if user modifies settings, else 'false'                    *
//******************************************************************************

bool FmConfig::ecoEditLM ( InitNcDialog& dInit )
{
   //* Construct control labels *
   const char* Labels[2] = 
   {
      "- Lock the Menu Bar in the visible state.",
      "- Menu Bar will be visible only when making a menu selection.\n",
   } ;

   bool pmods = false,
        dfltVal = CfgParm[cciLM].dflt == ZERO ? true : false,
        origVal = this->shm->lockMb,
        newVal  = this->ecoBinaryDecision ( dInit, Labels[0], Labels[1], origVal, dfltVal ) ;

   if ( origVal != newVal )
   {
      this->shm->lockMb = newVal ;
      pmods = true ;
   }
   return pmods ;

}  //* End of ecoEditLM() *

//*************************
//*       ecoEditTS       *
//*************************
//******************************************************************************
//* Interactively set the Trashcan-related options.                            *
//*                                                                            *
//* Input  : dInit  : dialog description (first 2 controls defined)            *
//*                                                                            *
//* Returns: 'true' if user modifies settings, else 'false'                    *
//******************************************************************************

bool FmConfig::ecoEditTS ( InitNcDialog& dInit )
{
   //* Create a list of needed controls *
   enum dpCtrls { okPB, canPB, altTB, dpcCOUNT } ;
   const char* const warnMsg[] = 
   {
      "  Warning - Warning  ",
      " ",
      "Specified location is not accessible.",
      "           Save it anyway?",
      NULL
   } ;
   attr_t warnColor[] = 
   {
      this->dColor | ncbATTR, this->dColor, this->expColor, 
      this->expColor, this->dColor, this->dColor
   } ;
   genDialog gd( (const char**)warnMsg, this->dColor, 
                 7, 41, dInit.dLines - 8, dInit.dColumns / 2 - 21, 
                 warnColor, false, this->pbnColor, this->pbfColor ) ;

   const char* Labels[4] = {
                             "Alternate Trashcan Path",
                             "(Alt Path inaccessible)",
                             "(Default Trashcan not found)",
                             "Expanded Alt Path\n\n\n\n"
                             "Default Desktop Trashcan\n"
                           } ;
   dInit.ctrlPtr[canPB].nextCtrl = &dInit.ctrlPtr[altTB] ;
   dInit.ctrlPtr[altTB] = 
   {  //* 'Alternate Trashcan Path text box   - - - - - - - - - - - - -  altTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      3,                            // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      short(dInit.dColumns - 4),    // cols:      control columns
      NULL,                         // dispText:  initially empty
      this->tbnColor,               // nColor:    non-focus color
      this->tbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    all printing characters allowed
      Labels[0],                    // label:     
      -1,                           // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    can receive input focus
      NULL,                         // nextCtrl:  link in next structure
   } ;
   gString gsFmt ;
   attr_t errorColor = nc.rebl ;
   bool pmods = false ;             // return value

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;
   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {  //* Display the static text *
      winPos expPos( dInit.ctrlPtr[altTB].ulY + 1, dInit.ctrlPtr[altTB].ulX + 1 ) ;
      winPos dfltPos = dp->WriteParagraph ( expPos, Labels[3], dInit.interiorColor ) ;
      ++expPos.ypos ;
      if ( *this->shm->cfgOpt.trPath != NULLCHAR )
      {
         gsFmt = this->shm->cfgOpt.trPath ;
         this->ecoPathFit ( gsFmt, dInit.dColumns - dfltPos.xpos - 2 ) ;
         dp->WriteString ( dfltPos, gsFmt, this->expColor ) ;
      }
      else
         dp->WriteString ( dfltPos, Labels[2], errorColor ) ;

      //* Set the text-box text *
      dp->SetTextboxText ( altTB, this->shm->trashPath ) ;

      dp->RefreshWin () ;           // make everything visible
      uiInfo   Info ;               // user interface data returned here
      short    icIndex = ZERO ;     // index of control with input focus
      bool     refreshExp = true,   // if 'true' re-calculate alt path
               validExp = false,    // 'true' if alt path is valid
               done = false ;       // loop control
      while ( ! done )
      {  //* Update display of expanded Alternate Trashcan path.*
         if ( refreshExp )
         {
            dp->ClearLine ( expPos.ypos ) ;
            dp->ClearLine ( expPos.ypos + 1 ) ;
            dp->GetTextboxText ( altTB, gsFmt ) ;
            if ( gsFmt.gschars() > 1 )
            {
               validExp = this->rcfValidateTrashPath ( gsFmt ) ;
               this->ecoPathFit ( gsFmt, dInit.dColumns - expPos.xpos - 2 ) ;
            }
            else
            {
               validExp = true ;    // no path specified
               gsFmt = "(none specified)" ;
            }
            dp->WriteString ( expPos, gsFmt, this->expColor ) ;
            if ( ! validExp )
               dp->WriteString ( expPos.ypos+1, expPos.xpos, Labels[1], errorColor ) ;
            dp->RefreshWin () ;
            refreshExp = false ;
         }
         //* If focus is currently on a Pushbutton *
         if ( dInit.ctrlPtr[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == okPB )
               {  //* Get user's input. If path is invalid, warn user *
                  //* that he/she/it is saving an invalid path.       *
                  dp->GetTextboxText ( altTB, gsFmt ) ;
                  validExp = true ;    // assume a valid path (or empty string)
                  gString gt = gsFmt ;
                  if ( (gt.gschars() > 1) &&
                       ! (validExp = this->rcfValidateTrashPath ( gsFmt )) )
                     validExp = dp->DecisionDialog ( gd ) ;
                  //* If path is valid, empty OR if user has verified *
                  //* save, then copy the path to our data member.    *
                  if ( validExp )
                  {
                     gt.copy( this->shm->trashPath, MAX_PATH ) ;
                     pmods = true ;
                  }
               }
               done = true ;
            }
         }
         //* If focus is currently on a Textbox *
         if ( dInit.ctrlPtr[icIndex].type == dctTEXTBOX )
         {
            Info.viaHotkey = false ;      // ignore hotkey data
            icIndex = dp->EditTextbox ( Info ) ;
            if ( Info.dataMod != false )  // if data modified
               refreshExp = true ;        // update real-path display
         }
         //* Move focus to appropriate control *
         if ( ! done  && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }     // while(!done)
   }        // if((this->dPtr->OpenWindow())==OK)
   if ( dp != NULL )
      delete dp ;
   return pmods ;

}  //* End of ecoEditTS() *

//*************************
//*       ecoEditCS       *
//*************************
//******************************************************************************
//* Interactively set the application's Color Scheme, the combination of       *
//* display colors used in creating the main dialog, sub-dialogs and control   *
//* objects.                                                                   *
//*                                                                            *
//* Input  : dInit  : dialog description (first 2 controls defined)            *
//*                                                                            *
//* Returns: 'true' if user modifies settings, else 'false'                    *
//******************************************************************************

short ecoEditCS_ControlUpdate ( const short currIndex, const wkeyCode wkey, 
                                bool firstTime = false ) ;
static NcDialog* ecoesc = NULL ;    // used ONLY by call-back for this method

bool FmConfig::ecoEditCS ( InitNcDialog& dInit )
{
   //* Create a list of needed controls *
   enum dpCtrls { okPB, canPB, defRB, blaRB, redRB, greRB, broRB, bluRB, magRB, 
                  cyaRB, gryRB, terRB, dpcCOUNT } ;
   char Labels[10][MAX_LABEL_CHARS] ; 
   dInit.ctrlPtr[canPB].nextCtrl = &dInit.ctrlPtr[defRB] ;
   dInit.ctrlPtr[defRB] = 
   {  //* 'Default' radio button  - - - - - - - - - - - - - - - - - - -  defRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially reset
      3,                            // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[0],                    // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &dInit.ctrlPtr[blaRB],        // nextCtrl:  link in next structure
   } ;
   dInit.ctrlPtr[blaRB] = 
   {  //* 'Black' radio button    - - - - - - - - - - - - - - - - - - -  blaRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially reset
      short(dInit.ctrlPtr[defRB].ulY + 1),// ulY:       upper left corner in Y
      short(dInit.ctrlPtr[defRB].ulX),// ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[1],                    // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &dInit.ctrlPtr[redRB],        // nextCtrl:  link in next structure
   } ;
   dInit.ctrlPtr[redRB] = 
   {  //* 'Red' radio button    - - - - - - - - - - - - - - - - - - - -  redRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially reset
      short(dInit.ctrlPtr[blaRB].ulY + 1),// ulY:       upper left corner in Y
      short(dInit.ctrlPtr[blaRB].ulX),// ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[2],                    // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &dInit.ctrlPtr[greRB],        // nextCtrl:  link in next structure
   } ;
   dInit.ctrlPtr[greRB] = 
   {  //* 'Green' radio button  - - - - - - - - - - - - - - - - - - - -  greRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially reset
      short(dInit.ctrlPtr[redRB].ulY + 1),// ulY:       upper left corner in Y
      short(dInit.ctrlPtr[redRB].ulX),// ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[3],                    // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &dInit.ctrlPtr[broRB],        // nextCtrl:  link in next structure
   } ;
   dInit.ctrlPtr[broRB] = 
   {  //* 'Brown' radio button  - - - - - - - - - - - - - - - - - - - -  broRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially reset
      short(dInit.ctrlPtr[greRB].ulY + 1),// ulY:       upper left corner in Y
      short(dInit.ctrlPtr[greRB].ulX),// ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[4],                    // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &dInit.ctrlPtr[bluRB],        // nextCtrl:  link in next structure
   } ;
   dInit.ctrlPtr[bluRB] = 
   {  //* 'Blue' radion button  - - - - - - - - - - - - - - - - - - - -  bluRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially reset
      short(dInit.ctrlPtr[broRB].ulY + 1),// ulY:       upper left corner in Y
      short(dInit.ctrlPtr[broRB].ulX),// ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[5],                    // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &dInit.ctrlPtr[magRB],        // nextCtrl:  link in next structure
   } ;
   dInit.ctrlPtr[magRB] = 
   {  //* 'Magenta' radio button  - - - - - - - - - - - - - - - - - - -  magRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially reset
      short(dInit.ctrlPtr[bluRB].ulY + 1),// ulY:       upper left corner in Y
      short(dInit.ctrlPtr[bluRB].ulX),// ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[6],                    // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &dInit.ctrlPtr[cyaRB],        // nextCtrl:  link in next structure
   } ;
   dInit.ctrlPtr[cyaRB] = 
   {  //* 'Cyan' radio button     - - - - - - - - - - - - - - - - - - -  cyaRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially reset
      short(dInit.ctrlPtr[magRB].ulY + 1),// ulY:       upper left corner in Y
      short(dInit.ctrlPtr[magRB].ulX),// ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[7],                    // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &dInit.ctrlPtr[gryRB],        // nextCtrl:  link in next structure
   } ;
   dInit.ctrlPtr[gryRB] = 
   {  //* 'Grey' radio button   - - - - - - - - - - - - - - - - - - - -  gryRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially reset
      short(dInit.ctrlPtr[cyaRB].ulY + 1),// ulY:       upper left corner in Y
      short(dInit.ctrlPtr[cyaRB].ulX),// ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[8],                    // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &dInit.ctrlPtr[terRB],        // nextCtrl:  link in next structure
   } ;
   dInit.ctrlPtr[terRB] = 
   {  //* 'Terminal' radio button - - - - - - - - - - - - - - - - - - -  terRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially reset
      short(dInit.ctrlPtr[gryRB].ulY + 1),// ulY:       upper left corner in Y
      short(dInit.ctrlPtr[gryRB].ulX),// ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[9],                    // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   } ;

   //* This array maps the members of enum NcBaseColors to the order of  *
   //* items in the scArgs[] array. Note the leading place-holder values.*
   NcBaseColors schemeOrder[] = 
   { ncbcDEFAULT, ncbcDEFAULT, ncbcCOLORS, ncbcBK, ncbcRE, 
     ncbcGR, ncbcBR, ncbcBL, ncbcMA, ncbcCY, ncbcGY, ncbcBW } ;
//   //* This array maps the control indices to the items in the scArgs[]  *
//   //* array. Note the leading place-holder values.                      *
//   short argOrder[] = { 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 } ;
   //* Construct control labels *
   gString gsOut ;
   for ( short i = ZERO ; i <  CfgParm[cciSC].argcnt ; i++ )
   {
      gsOut.compose( L"'%S'", scArgs[i].str ) ;
      gsOut.copy( Labels[i], MAX_LABEL_CHARS ) ;
   }

   //* 'Select' the radio button for the current selection *
   NcBaseColors origScheme = this->shm->cfgOpt.cScheme.scheme ;
   short origIndex = defRB ;
   while ( schemeOrder[origIndex] != origScheme && origIndex < dpcCOUNT )
      ++origIndex ;
   dInit.ctrlPtr[origIndex].rbSelect = true ;

   bool pmods = false ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;
   ecoesc = dp ;              // give call-bac method access

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Group all radio buttons in the dialog *
      short XorGroup[] = { defRB, blaRB, redRB, greRB, broRB, bluRB, magRB,
                           cyaRB, gryRB, terRB, -1 } ;
      dp->GroupRadiobuttons ( XorGroup ) ;

      //* Establish a call-back method for updating the color-scheme sample.*
      dp->EstablishCallback ( &ecoEditCS_ControlUpdate ) ;

      dp->RefreshWin () ;           // make everything visible

      uiInfo   Info ;               // user interface data returned here
      short    icIndex = ZERO ;     // index of control with input focus
      bool     done = false ;       // loop control
      while ( ! done )
      {
         //* If focus is currently on a Pushbutton *
         if ( dInit.ctrlPtr[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == okPB )
               {  //* Get user's selection and store it in our data member *
                  short newIndex = dp->GetRbGroupSelection ( defRB ) ;
                  if ( newIndex != origIndex )
                  {
                     this->shm->cfgOpt.cScheme.scheme = schemeOrder[newIndex] ;
                     pmods = true ;
                  }
               }
               done = true ;
            }
         }
         //* If focus is currently on a Radio Button *
         else if ( dInit.ctrlPtr[icIndex].type == dctRADIOBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditRadiobutton ( Info ) ;
         }
         //* Move focus to appropriate control *
         if ( ! done  && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }     // while(!done)
   }        // if((this->dPtr->OpenWindow())==OK)
   if ( dp != NULL )
   {
      delete dp ;
      ecoesc = NULL ;   // pointer's target is out-of-scope
   }
   return pmods ;
}  //* End of ecoEditCS() *

//***************************
//* ecoEditCS_ControlUpdate *
//***************************
//******************************************************************************
//* NON-MEMBER METHOD.                                                         *
//* This is a call-back method for FmConfig::ecoEditCS().                      *
//* It is called from within the NcDialog class during the user-input loop.    *
//*                                                                            *
//* This is an exercise in user friendliness. We draw a sample (pseudo)dialog  *
//* window using the color scheme appropriate to the control which currently   *
//* has the input focus.                                                       *
//*                                                                            *
//* Input  : currIndex: index of control that currently has focus              *
//*          wkey     : user's key input data                                  *
//*          firstTime: the EstablishCallback() method calls this method once  *
//*                     with firstTime==true, to perform any required          *
//*                     initialization. Subsequently, the NcDialog class       *
//*                     always calls with firstTime==false.                    *
//* Returns: OK                                                                *
//******************************************************************************
//* Programmer's Note: Call-backs are fragile by nature; they make intimate    *
//* assumptions about what's happening in the higher-level code. Be careful.   *
//*                                                                            *
//* NOTE: Determining the correct color scheme to be displayed depends not     *
//* only on the control which currently has focus, but also on which control   *
//* control previously had focus. This makes for a tortured test.              *
//* Note that 'currIndex' is the index of control that currently has focus.    *
//* The EditGroupRadio() method calls the callback _after_ the focus has       *
//* moved to the next/previous member control. The EditPushbutton() method     *
//* calls the callback after the keystroke but _before_ focus actually moves   *
//* (except for hotkeys, of which there are none in this dialog).              *
//*                                                                            *
//* Programmer's Note: We don't have access to the actual mapping used by the  *
//* application's color scheme. For this reason, always keep this method       *
//* synchronized with with the color-scheme definitions established in the     *
//* InitFmMenuData() method located in FmMenu.cpp.                             *
//*                                                                            *
//******************************************************************************

short ecoEditCS_ControlUpdate ( const short currIndex, const wkeyCode wkey, 
                                bool firstTime )
{
   //* Programmer's Note: we don't want to spew module-scope variables all over*
   //* the place, so we duplicate the salient values. Keep them synchronized.  *
   enum dpCtrls : short { okPB, canPB, defRB, blaRB, redRB, greRB, broRB, 
                          bluRB, magRB, cyaRB, gryRB, terRB, dpcCOUNT } ;
   NcBaseColors schemeOrder[] = 
   { ncbcDEFAULT, ncbcDEFAULT, ncbcCOLORS, ncbcBK, ncbcRE, 
     ncbcGR, ncbcBR, ncbcBL, ncbcMA, ncbcCY, ncbcGY, ncbcBW } ;
   short argOrder[] = { 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 } ;
   //*- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*

   static short previousControl = okPB ;  // focus tracking (see note)

   //* Initialize the color-scheme index to the *
   //* selected member of the Radiobutton group.*
   short selMem = ecoesc->GetRbGroupSelection ( defRB ) ;   // 'selected' radio button
   NcBaseColors scs = schemeOrder[selMem] ;                 // index of matching color scheme
   const wchar_t* title = scArgs[argOrder[selMem]].str ;    // index of matching title
   //* If focus is currently on a Pushbutton, BUT is      *
   //* moving to a Radiobutton, then set the color-scheme *
   //* index for the Radiobutton which will receive focus.*
   if ( currIndex == okPB && wkey.type == wktFUNKEY && 
        (wkey.key == nckSTAB || wkey.key == nckLEFT ) )
   {
      scs = schemeOrder[terRB] ;
      title = scArgs[argOrder[terRB]].str ;
   }
   else if ( currIndex == canPB && wkey.type == wktFUNKEY && 
             (wkey.key == nckTAB || wkey.key == nckRIGHT ) )
   {
      scs = schemeOrder[defRB] ;
      title = scArgs[argOrder[defRB]].str ;
   }
   //* Else if focus is on a Radiobutton which IS NOT the selected *
   //* member, AND if a selection has not just been made.          *
   else if ( (currIndex >= defRB && currIndex <= terRB) && 
             (currIndex != selMem) && 
             !((wkey.type == wktFUNKEY && (wkey.key == nckENTER || wkey.key == nckpENTER)) ||
               (wkey.type == wktPRINT  && wkey.key == nckSPACE)) )
   {
      //* If Radiobutton group IS NOT about to lose focus *
      // Programmer's Note: This is a trick because if the group is about to 
      // lose focus, then the first or last member also _previously_ had focus.
      if ( currIndex != previousControl )
      {
         scs = schemeOrder[currIndex] ;
         title = scArgs[argOrder[currIndex]].str ;
      }
   }
   previousControl = currIndex ;

   //* Color mapping according to Color Scheme *
   ColorScheme cs ;
   cs.scheme = scs ;
   switch ( scs )
   {
      case ncbcBK:         // 'Black'
         cs.bb = nc.bk ;
         cs.sd = nc.bk ;
         cs.sb = nc.bk & ~ncbATTR ;
         cs.em = nc.brbk ;
         cs.wr = nc.rebk ;
         cs.mn = nc.cyR ;
         cs.mf = nc.bk ;
         cs.pn = nc.gyR ;
         cs.pf = nc.reG ;
         cs.tn = nc.bw ;
         cs.tf = nc.bl ;
         break ;
      case ncbcRE:         // 'Red'
         cs.bb = nc.reR ;
         cs.sd = nc.reR ;
         cs.sb = nc.gyre | ncbATTR ;
         cs.em = nc.brre ;
         cs.wr = nc.mare ;
         cs.mn = nc.re ;
         cs.mf = nc.reR ;
         cs.pn = nc.gyR ;
         cs.pf = nc.maG ;
         cs.tn = nc.bw ;
         cs.tf = nc.bl ;
         break ;
      case ncbcGR:         // 'Green'
         cs.bb = nc.grR ;
         cs.sd = nc.grR ;
         cs.sb = nc.gygr | ncbATTR ;
         cs.em = nc.brgr ;
         cs.wr = nc.regr ;
         cs.mn = nc.gr ;
         cs.mf = nc.grR ;
         cs.pn = nc.gyR ;
         cs.pf = nc.reG ;
         cs.tn = nc.bw ;
         cs.tf = nc.bl ;
         break ;
      case ncbcBR:         // 'Brown'
         cs.bb = nc.brR ;
         cs.sd = nc.brR ;
         cs.sb = nc.gybr | ncbATTR ;
         cs.em = nc.cybr ;
         cs.wr = nc.mabr ;
         cs.mn = nc.br ;
         cs.mf = nc.brR ;
         cs.pn = nc.gyR ;
         cs.pf = nc.maG ;
         cs.tn = nc.bw ;
         cs.tf = nc.bl ;
         break ;
      case ncbcBL:         // 'Blue'
         cs.bb = nc.blR ;
         cs.sd = nc.blR ;
         cs.sb = nc.gybl | ncbATTR ;
         cs.em = nc.brbl ;
         cs.wr = nc.rebl ;
         cs.mn = nc.bl ;
         cs.mf = nc.blR ;
         cs.pn = nc.gyR ;
         cs.pf = nc.reG ;
         cs.tn = nc.bw ;
         cs.tf = nc.gr ;
         break ;
      case ncbcMA:         // 'Magenta'
         cs.bb = nc.maR ;
         cs.sd = nc.maR ;
         cs.sb = nc.gyma | ncbATTR ;
         cs.em = nc.brma ;
         cs.wr = nc.rema ;
         cs.mn = nc.ma ;
         cs.mf = nc.maR ;
         cs.pn = nc.gyR ;
         cs.pf = nc.reG ;
         cs.tn = nc.bw ;
         cs.tf = nc.bl ;
         break ;
      case ncbcCY:         // 'Cyan'
         cs.bb = nc.cyR ;
         cs.sd = nc.cyR ;
         cs.sb = nc.gycy | ncbATTR ;
         cs.em = nc.brcy ;
         cs.wr = nc.recy ;
         cs.mn = nc.cy ;
         cs.mf = nc.cyR ;
         cs.pn = nc.gyR ;
         cs.pf = nc.reG ;
         cs.tn = nc.bw ;
         cs.tf = nc.bl ;
         break ;
      case ncbcGY:         // 'Grey'
         cs.bb = nc.gyR ;
         cs.sd = nc.gyR ;
         cs.sb = nc.bkgy ;
         cs.em = nc.blgy ;
         cs.wr = nc.regy ;
         cs.mn = nc.cy ;
         cs.mf = nc.cyR ;
         cs.pn = nc.bw ;
         cs.pf = nc.reG ;
         cs.tn = nc.bw ;
         cs.tf = nc.bl ;
         break ;
      case ncbcBW:         // 'Terminal'
         cs.bb = nc.bw ;
         cs.sd = nc.cyR ;
         cs.sb = nc.gycy | ncbATTR ;
         cs.em = nc.brcy ;
         cs.wr = nc.recy ;
         cs.mn = nc.gyR ;
         cs.mf = nc.bw ;
         cs.pn = nc.gyR ;
         cs.pf = nc.reG ;
         cs.tn = nc.bw ;
         cs.tf = nc.blR ;
         break ;
      case ncbcCOLORS:     // 'Default'
      default:
         cs.bb = nc.cyR ;
         cs.sd = nc.blR ;
         cs.sb = nc.gybl | ncbATTR ;
         cs.em = nc.brbl ;
         cs.wr = nc.rebl ;
         cs.mn = nc.blR ;
         cs.mf = nc.cyR ;
         cs.pn = nc.gyR ;
         cs.pf = nc.reG ;
         cs.tn = nc.bw ;
         cs.tf = nc.gr ;
         break ;
   }

   winPos ul( 1, 26 ) ;
   winPos wp( ul.ypos + 1, ul.xpos + 1 ) ;
   const short boxROWS = 12,
               boxCOLS = 44 ;
   //* Clear an area using the terminal background color *
   gString gsOut( "                                              " ) ;
   for ( short y = ul.ypos ; y <= (ul.ypos + boxROWS + 1) ; y++ )
      ecoesc->WriteString ( y, ul.xpos, gsOut, nc.bw ) ;

   //* Draw and fill a box that resembles a dialog *
   gsOut.limitCols( boxCOLS ) ;
   for ( short y = wp.ypos + 1 ; y < (wp.ypos + boxROWS - 1) ; y++ )
      ecoesc->WriteString ( y, wp.xpos, gsOut, cs.bb ) ;
   gsOut.compose( L"  %S  ", title ) ;
   ecoesc->DrawBox ( wp.ypos, wp.xpos, boxROWS, boxCOLS, cs.bb, gsOut.ustr() ) ;
   //* Draw pretty pictures using the appropriate control colors *
   ++wp.ypos ;
   ++wp.xpos ;
   ul = wp ;

   //* dctTEXTBOX *
   wp = ecoesc->WriteString ( wp.ypos, wp.xpos + 1, "  Textbox (focus)  ", cs.tf ) ;
   ecoesc->WriteString ( wp.ypos, wp.xpos + 1, " Textbox(non-focus) ", cs.tn ) ;
   wp.ypos += 1 ;
   wp.xpos = ul.xpos ;

   //* dctPUSHBUTTON *
   wp = ecoesc->WriteString ( wp.ypos, wp.xpos + 1, " Pushbutton(focus) ", cs.pf ) ;
   ecoesc->WriteString ( wp.ypos, wp.xpos + 1, "  Pushbutton (n-f)  ", cs.pn ) ;
   wp.ypos += 1 ;
   wp.xpos = ul.xpos ;

   //* dctMENUWIN *
   attr_t menuInterior = cs.mf ;
   if ( cs.scheme == ncbcCOLORS ) // default color scheme does it a bit differently
      menuInterior = cs.bb ;
   ecoesc->WriteString ( wp.ypos++, wp.xpos, "  Menu  ", cs.mn ) ;
   ecoesc->DrawBox ( wp.ypos, wp.xpos, 5, 10, cs.mn ) ;
   ecoesc->WriteParagraph ( wp.ypos + 1, wp.xpos + 1, 
                            " without\n"
                            " input  \n"
                            " focus  ", menuInterior ) ;
   ecoesc->DrawBox ( wp.ypos + 2, wp.xpos + 9, 5, 10, cs.mf ) ;
   ecoesc->WriteParagraph ( wp.ypos + 3, wp.xpos + 10, 
                            " with   \n"
                            " input  \n"
                            " focus  ", menuInterior ) ;

   //* Sub-dialogs *
   ecoesc->DrawBox ( wp.ypos, wp.xpos + 20, 6, 21, cs.sd, " sub-dialogs " ) ;
   ecoesc->WriteParagraph ( wp.ypos + 1, wp.xpos + 21, 
                            "   border and      \n"
                            "        interior   \n"
                            "                   \n"
                            " < x >control label", cs.sd ) ;
   gsOut.compose( L"[ %C ]", &wcsDIAMOND ) ;
   ecoesc->WriteString ( wp.ypos + 4, wp.xpos + 22, gsOut, cs.pf ) ;

   ecoesc->RefreshWin () ;

   return OK ;

}  //* End ecoEditCS_ControlUpdate() *

//*************************
//*       ecoEditEM       *
//*************************
//******************************************************************************
//* Interactively set the parameters for mouse support.                        *
//*                                                                            *
//* Input  : dInit  : dialog description (first 2 controls defined)            *
//*                                                                            *
//* Returns: 'true' if user modifies settings, else 'false'                    *
//******************************************************************************

bool FmConfig::ecoEditEM ( InitNcDialog& dInit )
{
   bool pmods = false ;       // return value

   //* Create a list of needed controls *
   enum dpCtrls { okPB, canPB, enaRB, disRB, dpcCOUNT } ;
   dInit.ctrlPtr[canPB].nextCtrl = &dInit.ctrlPtr[enaRB] ;
   dInit.ctrlPtr[enaRB] = 
   {  //* 'enable' radio button  - - - - - - - - - - - - - - - - - - - - enaRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      this->shm->cfgOpt.enableMouse,// rbSelect:  initial value
      2,                            // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "'enable'",                   // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &dInit.ctrlPtr[disRB],        // nextCtrl:  link in next structure
   } ;
   dInit.ctrlPtr[disRB] = 
   {  //* 'disable' radio button    - - - - - - - - - - - - - - - - -    disRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      (this->shm->cfgOpt.enableMouse ? false : true), // rbSelect:  initial value
      short(dInit.ctrlPtr[enaRB].ulY + 4),// ulY:       upper left corner in Y
      short(dInit.ctrlPtr[enaRB].ulX),// ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "'disable' (default)",        // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   } ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;
   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Group all radio buttons in the dialog *
      short XorGroup[] = { enaRB, disRB, -1 } ;
      dp->GroupRadiobuttons ( XorGroup ) ;

      //* Display static text *
      winPos wp ;
      wp.ypos = dInit.ctrlPtr[enaRB].ulY + dInit.ctrlPtr[enaRB].labY + 1 ;
      wp.xpos = dInit.ctrlPtr[enaRB].ulX + dInit.ctrlPtr[enaRB].labX ;
      dp->WriteParagraph ( wp, 
                           "- Enable FileMangler mouse support for selection of\n"
                           "  filenames, control activation and data scrolling.",
                           this->expColor ) ;
      wp.ypos = dInit.ctrlPtr[disRB].ulY + dInit.ctrlPtr[disRB].labY + 1 ;
      wp.xpos = dInit.ctrlPtr[disRB].ulX + dInit.ctrlPtr[disRB].labX ;
      dp->WriteParagraph ( wp, 
                           "- Disable FileMangler mouse support.",
                           this->expColor ) ;
      wp.ypos = dInit.ctrlPtr[disRB].ulY + 4 ;
      wp.xpos = dInit.ctrlPtr[disRB].ulX ;
      dp->WriteParagraph ( wp, 
         "Text-mode mouse support is necessarily very basic; however, you may\n"
         "find it to be useful. Please refer to the FileMangler online\n"
         "documentation, or the NcDialog API documentation for more information.",
         this->expColor ) ;

      dp->RefreshWin () ;           // make everything visible

      uiInfo   Info ;               // user interface data returned here
      short    icIndex = ZERO ;     // index of control with input focus
      bool     done = false ;       // loop control
      while ( ! done )
      {
         //* If focus is currently on a Pushbutton *
         if ( dInit.ctrlPtr[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == okPB )
               {  //* Get user's selection and return it to caller *
                  short newIndex = dp->GetRbGroupSelection ( enaRB ) ;
                  bool decision = newIndex == enaRB ? true : false ;
                  if ( decision != this->shm->cfgOpt.enableMouse )
                  {
                     this->shm->cfgOpt.enableMouse = decision ;
                     pmods = true ;
                  }
               }
               done = true ;
            }
         }

         //* If focus is currently on a Radiobutton *
         else if ( dInit.ctrlPtr[icIndex].type == dctRADIOBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditRadiobutton ( Info ) ;
         }

         //* Move focus to appropriate control *
         if ( ! done  && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }     // while(!done)
   }        // if((this->dPtr->OpenWindow())==OK)
   if ( dp != NULL )
      delete dp ;
   return pmods ;

}  //* ecoEditEM() *

//*************************
//*       ecoEditEP       *
//*************************
//******************************************************************************
//* Interactively set the parameters for launching external programs           *
//*                                                                            *
//* Input  : dInit  : dialog description (first 2 controls defined)            *
//*                                                                            *
//* Returns: 'true' if user modifies settings, else 'false'                    *
//******************************************************************************

bool FmConfig::ecoEditEP ( InitNcDialog& dInit )
{
   //* Create a list of needed controls *
   enum dpCtrls { okPB, canPB, autoRB, autoaRB, safeRB, manRB, dpcCOUNT } ;
   char Labels[4][MAX_LABEL_CHARS] ; 
   dInit.ctrlPtr[canPB].nextCtrl = &dInit.ctrlPtr[autoRB] ;
   dInit.ctrlPtr[autoRB] = 
   {  //* 'AutoLaunch' radio button - - - - - - - - - - - - - - - - - - autoRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially reset
      1,                            // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[0],                    // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &dInit.ctrlPtr[autoaRB]       // nextCtrl:  link in next structure
   } ;
   dInit.ctrlPtr[autoaRB] = 
   {  //* 'AutoAudio' radio button  - - - - - - - - - - - - - - - - -  autoaRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially reset
      short(dInit.ctrlPtr[autoRB].ulY + 4),// ulY:       upper left corner in Y
      short(dInit.ctrlPtr[autoRB].ulX),// ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[1],                    // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &dInit.ctrlPtr[safeRB],       // nextCtrl:  link in next structure
   } ;
   dInit.ctrlPtr[safeRB] = 
   {  //* 'SafeLaunch' radio button   - - - - - - - - - - - - - - - -   safeRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially reset
      short(dInit.ctrlPtr[autoaRB].ulY + 4),// ulY:  upper left corner in Y
      short(dInit.ctrlPtr[autoaRB].ulX),    // ulX:  upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[2],                    // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &dInit.ctrlPtr[manRB],        // nextCtrl:  link in next structure
   } ;
   dInit.ctrlPtr[manRB] = 
   {  //* 'ManualLaunch' radio button   - - - - - - - - - - - - - - - -  manRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially reset
      short(dInit.ctrlPtr[safeRB].ulY + 4),// ulY:  upper left corner in Y
      short(dInit.ctrlPtr[safeRB].ulX),    // ulX:  upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[3],                    // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   } ;

   //* Construct control labels *
   gString gsOut ;
   gsOut.compose( L"'%S'  (default)", 
                  epArgs[lcAutoLaunch].str ) ;
   gsOut.copy( Labels[0], MAX_LABEL_CHARS ) ;
   gsOut.compose( L"'%S' ", 
                  epArgs[lcAutoAudio].str ) ;
   gsOut.copy( Labels[1], MAX_LABEL_CHARS ) ;
   gsOut.compose( L"'%S' ", 
                  epArgs[lcSafeLaunch].str ) ;
   gsOut.copy( Labels[2], MAX_LABEL_CHARS ) ;
   gsOut.compose( L"'%S' ", 
                  epArgs[lcManualLaunch].str ) ;
   gsOut.copy( Labels[3], MAX_LABEL_CHARS ) ;

   //* 'Select' the radio button for the current selection *
   LaunchCodes lcOrig = this->shm->cfgOpt.lCode ;
   short origIndex ;
   switch ( lcOrig )
   {
      case lcAutoAudio:    origIndex = autoaRB ;   break ;
      case lcSafeLaunch:   origIndex = safeRB ;    break ;
      case lcManualLaunch: origIndex = manRB ;     break ;
      case lcAutoLaunch:
      default:             origIndex = autoRB ;    break ;
   } ;
   dInit.ctrlPtr[origIndex].rbSelect = true ;

   bool pmods = false ;             // return value

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;
   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Group all radio buttons in the dialog *
      short XorGroup[] = { autoRB, autoaRB, safeRB, manRB, -1 } ;
      dp->GroupRadiobuttons ( XorGroup ) ;

      //* Display static text *
      winPos wp ;
      wp.ypos = dInit.ctrlPtr[autoRB].ulY + dInit.ctrlPtr[autoRB].labY + 1 ;
      wp.xpos = dInit.ctrlPtr[autoRB].ulX + dInit.ctrlPtr[autoRB].labX ;
      dp->WriteParagraph ( wp, 
                        "- Application determines launch parameters if possible,\n"
                        "  else open a dialog to get user's preference.",
                        this->expColor ) ;
      wp.ypos = dInit.ctrlPtr[autoaRB].ulY + dInit.ctrlPtr[autoaRB].labY + 1 ;
      wp.xpos = dInit.ctrlPtr[autoaRB].ulX + dInit.ctrlPtr[autoaRB].labX ;
      dp->WriteParagraph ( wp, 
                        "- Same as 'AutoLaunch' but assumes that audio files\n"
                        "  will be opened by a GUI media player.",
                        this->expColor ) ;
      wp.ypos = dInit.ctrlPtr[safeRB].ulY + dInit.ctrlPtr[safeRB].labY + 1 ;
      wp.xpos = dInit.ctrlPtr[safeRB].ulX + dInit.ctrlPtr[safeRB].labX ;
      dp->WriteParagraph ( wp, 
                        "- Launch from current window if it can be done without resource\n"
                        "  conflict, otherwise open program from a new terminal window.",
                        this->expColor ) ;
      wp.ypos = dInit.ctrlPtr[manRB].ulY + dInit.ctrlPtr[manRB].labY + 1 ;
      wp.xpos = dInit.ctrlPtr[manRB].ulX + dInit.ctrlPtr[manRB].labX ;
      dp->WriteParagraph ( wp, 
                        "- Always open an interactive dialog to configure launch\n"
                        "  parameters.",
                        this->expColor ) ;

      dp->RefreshWin () ;           // make everything visible

      uiInfo   Info ;               // user interface data returned here
      short    icIndex = ZERO ;     // index of control with input focus
      bool     done = false ;       // loop control
      while ( ! done )
      {
         //* If focus is currently on a Pushbutton *
         if ( dInit.ctrlPtr[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == okPB )
               {  //* Get user's selection and store it in our data member *
                  short newIndex = dp->GetRbGroupSelection ( autoRB ) ;
                  if ( newIndex != origIndex )
                  {
                     switch ( newIndex )
                     {
                        case autoaRB:
                           this->shm->cfgOpt.lCode = lcAutoAudio ;    break ;
                        case safeRB:
                           this->shm->cfgOpt.lCode = lcSafeLaunch ;   break ;
                        case manRB:
                           this->shm->cfgOpt.lCode = lcManualLaunch ; break ;
                        case autoRB:
                        default:
                           this->shm->cfgOpt.lCode = lcAutoLaunch ; break ;
                     }
                     pmods = true ;
                  }
               }
               done = true ;
            }
         }
         //* If focus is currently on a Radio Button *
         else if ( dInit.ctrlPtr[icIndex].type == dctRADIOBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditRadiobutton ( Info ) ;
         }
         //* Move focus to appropriate control *
         if ( ! done  && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }     // while(!done)
   }        // if((this->dPtr->OpenWindow())==OK)

   if ( dp != NULL )
      delete dp ;

   return pmods ;

}  //* End of ecoEditEP() *

//*************************
//*       ecoEditFD       *
//*************************
//******************************************************************************
//* Interactively set the list of 'Favorite Directories'.                      *
//*                                                                            *
//* Input  : dInit  : dialog description (first 2 controls defined)            *
//*                                                                            *
//* Returns: 'true' if user modifies settings, else 'false'                    *
//******************************************************************************

bool FmConfig::ecoEditFD ( InitNcDialog& dInit )
{
   //* Create a list of needed controls *
   enum dpCtrls { okPB, canPB, favSE, entTB, dpcCOUNT } ;
   dInit.ctrlPtr[canPB].nextCtrl = &dInit.ctrlPtr[favSE] ;
   attr_t  singleColor[2] = { attrDFLT, this->tbnColor } ;
   short cdPathWidth = dInit.dColumns - 16 ;  // width of text-entry control
   dInit.ctrlPtr[favSE] = 
   { //* 'Favorite Directories' scroll ext - - - - - - - - - - - - - -   favSE *
      dctSCROLLEXT,                 // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      2,                            // ulY:       upper left corner in Y
      ZERO,                         // ulX:       upper left corner in X
      10,                           // lines:     control lines
      dInit.dColumns,               // cols:      control columns
      NULL,                         // dispText:  n/a
      this->dColor,                 // nColor:    non-focus border color
      this->pbfColor,               // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      " Favorite Directories ",     // label:     
      ZERO,                         // labY:      offset from control's ulY
      ZERO,                         // labX       offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      MAX_FAVDIRS,                  // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    (n/a)
      singleColor,                  // scrColor:  single-color data display
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &dInit.ctrlPtr[entTB],        // nextCtrl:  link in next structure
   } ;
   dInit.ctrlPtr[entTB] = 
   {  //* 'Enter/edit path' Text Box   - - - - - - - - - - - - - -       entTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      1,                            // ulY:       upper left corner in Y
      short(dInit.dColumns - cdPathWidth - 1),// ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      cdPathWidth,                  // cols:      control columns
      NULL,                         // dispText:  
      dInit.interiorColor,          // nColor:    non-focus color
      this->tbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      NULL,                         // label:     
      ZERO,                         // labY:      
      -11,                          // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    initially inaccessable to user
      NULL,                         // nextCtrl:  link in next structure
   } ;

   //* Formatting for the favorites list display data *
   const char* seboxData[MAX_FAVDIRS] ;
   gString fgs[MAX_FAVDIRS], gx ;
   attr_t  seboxColors[MAX_FAVDIRS] ;
   for ( short i = ZERO ; i < MAX_FAVDIRS ; ++i )
   {
      seboxColors[i] = singleColor[1] ;
      fgs[i] = this->shm->favPath[i] ;

      //* If the pathspec width fits within the display area *
      if ( (fgs[i].gscols()) <= (dInit.dColumns - 2) )
         seboxData[i] = fgs[i].ustr() ;
      //* Else, display only the tail of the pathspec string *
      else
      {
         // Programmer's Note: This is a bit fancy because we must 
         // anticipate multi-column characters in the pathspec strings, 
         // i.e. one character does not necessarily equal one column.
         gx = fgs[i] ;
         gx.shiftCols( -((gx.gscols()) - (dInit.dColumns - 1)) ) ;
         seboxData[i] = &fgs[i].ustr()[(fgs[i].utfbytes()) - (gx.utfbytes())] ;
      }
   }
   ssetData sData( seboxData, seboxColors, this->shm->favCount, ZERO, true ) ;

   bool pmods = false ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Make visual connections to dialog border *
      cdConnect cdConn ;
      cdConn.connection = true ;
      cdConn.ul2Left = cdConn.ll2Left = cdConn.ur2Right = cdConn.lr2Right = true ;
      dp->ConnectControl2Border ( favSE, cdConn ) ;

      //* Draw static text *
      dp->WriteParagraph ( dInit.dLines - 7, 1,
         "Highlight the desired item using arrow keys, then:\n"
         "  Ctrl+U  : move item Up            Ctrl+R  : Remove item from list\n"
         "  Ctrl+D  : move item Down          Ctrl+A  : Add new item to list\n"
         "  Ctrl+E  : Edit item\n", dInit.interiorColor ) ;

      dp->RefreshWin () ;           // make everything visible

      uiInfo   Info ;               // user interface data returned here
      short    icIndex = dp->PrevControl () ; // index of control with input focus
      bool     refreshSE = true,    // refresh contents of dctSCROLLEXT control
               done = false ;       // loop control
      while ( ! done )
      {
         if ( refreshSE )
         {
            if ( sData.dispItems == ZERO )
               fgs[sData.dispItems++] = seNoFiles ; ;
            dp->SetScrollextText ( favSE, sData ) ;
            refreshSE = false ;
         }

         //* If focus is currently on a Pushbutton *
         if ( dInit.ctrlPtr[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == okPB )
               {  //* Copy edited strings to our data member *
                  for ( short i = ZERO ; i < MAX_FAVDIRS ; i++ )
                     fgs[i].copy( this->shm->favPath[i], MAX_PATH ) ;
                  this->shm->favCount = sData.dispItems ;
                  pmods = true ;    // force parent to do a status update
               }
               done = true ;
            }
         }

         //* If focus is currently on the Scroll Box *
         else if ( dInit.ctrlPtr[icIndex].type == dctSCROLLEXT )
         {
            Info.viaHotkey = false ;      // ignore hotkey data
            icIndex = dp->EditScrollext ( Info ) ;
            if ( Info.keyIn != nckTAB && Info.keyIn != nckSTAB && Info.keyIn != nckESC )
            {
               winPos pptPos( dInit.ctrlPtr[entTB].ulY, dInit.ctrlPtr[entTB].ulX - 14 ) ;
               winPos msgPos( dInit.dLines - 2, 2 ) ;
               refreshSE = 
                  this->ecoEditScrollextData ( dp, Info, sData, fgs, favSE, entTB, 
                                               MAX_FAVDIRS, pptPos, msgPos ) ;
               if ( refreshSE )
               {
                  for ( short i = ZERO ; i < sData.dispItems ; ++i )
                  {
                     //* If the pathspec width fits within the display area *
                     if ( (fgs[i].gscols()) <= (dInit.dColumns - 2) )
                        sData.dispText[i] = fgs[i].ustr() ;
                     //* Else, display only the tail of the pathspec string *
                     else
                     {
                        // Programmer's Note: This is a bit fancy because we must 
                        // anticipate multi-column characters in the pathspec strings, 
                        // i.e. one character does not necessarily equal one column.
                        gx = fgs[i] ;
                        gx.shiftCols( -((gx.gscols()) - (dInit.dColumns - 1)) ) ;
                        sData.dispText[i] = &fgs[i].ustr()[(fgs[i].utfbytes()) - (gx.utfbytes())] ;
                     }
                  }
               }

               //* Keep focus on scroll box until user explicitly moves on.*
               while ( (icIndex = dp->NextControl ()) != canPB ) ;
            }
         }

         //* Move focus to appropriate control *
         if ( ! done  && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }     // while(!done)
   }        // if((this->dPtr->OpenWindow())==OK)
   if ( dp != NULL )
      delete dp ;
   return pmods ;

}  //* End of ecoEditFD() *

//*************************
//*       ecoEditKM       *
//*************************
//******************************************************************************
//* Interactively display and modify the command-key map.                      *
//*                                                                            *
//* Input  : dInit  : dialog description (first 2 controls defined)            *
//*                                                                            *
//* Returns: 'true' if user modifies settings, else 'false'                    *
//******************************************************************************

static short EditKM_Callback ( const short currIndex, const wkeyCode wkey, bool firstTime ) ;
static NcDialog* kmdp = NULL ;      // pointer to dialog window

bool FmConfig::ecoEditKM ( InitNcDialog& dInit )
{
   const char* quickHelp = 
         "To remap a keycode, first highlight the keycode you\n"
         "wish to use and press CTRL+C (copy), then highlight\n"
         "the desired operation and press CTRL+V (paste). To\n"
         "remove a keycode, highlight the item and press CTRL+X." ; 
   const char* clearMsgArea = "                      \n"
                              "                      \n"
                              "                      " ;
   const char* in_useTemplate = "Warning! The keycode:\n"
                                "%s\n"
                                "is already in use." ;

   //* Create a list of needed controls *
   enum dpCtrls { okPB, canPB, keySE, defSE, dpcCOUNT } ;

   //* If the keymap has not yet been initialized *
   if ( ! this->shm->keyMap.kmInit )
      this->shm->keyMap.initialize() ;
   //* If keymap configuration file not specified, *
   //* initialize to default filespec.             *
   if ( *this->shm->keyMap.keyFile == NULLCHAR )
      strncpy ( this->shm->keyMap.keyFile, kmcfgName, MAX_FNAME ) ;

   gString gs( this->shm->keyMap.keydef[0].ktext ), // text formatting
           gsBlank ;
   short dOffset = (gs.find( L'|' )) + 2, // index of description text
         tbytes  = dOffset - 1,           // storage bytes for user-def text
         tcols   = tbytes - 1,            // column count for user-def text
         userKeys = ZERO,                 // number of user-defined keycodes
         dup1, dup2 ;                     // indices of duplicated records
   bool  blankFound = false,              // 'true' if gsBlank initialized
         pmods = false ;                  // return value

   //* Initialize the display text *
   const attr_t dColor = dInit.interiorColor,
                hColor = nc.brbl,
                mColor = nc.rebl,
                nColor = nc.gy,
                fColor = nc.gr ;
   char  keyString[KEYMAP_KEYS][tbytes] ; // raw text for 'keySE' control
   const char* remapText[KEYMAP_KEYS] ;   // text pointers for 'keySE' control
   attr_t      remapAttr[KEYMAP_KEYS] ;   // color attributes for 'keySE' control
   const char* descText[KEYMAP_KEYS] ;    // text pointers for 'defSE' control
   attr_t      descAttr[KEYMAP_KEYS] ;    // color attributes for 'defSE' control
   wkeyCode    userCodes[KEYMAP_KEYS] ;   // backup copy of user-defined keycodes
   for ( short i = ZERO ; i < KEYMAP_KEYS ; ++i )
   {
      gs = &this->shm->keyMap.keydef[i].ktext[1] ; // format and save user-def text
      gs.limitCols( tcols - 1 ) ;
      gs.padCols( tcols ) ;
      gs.copy( keyString[i], tbytes ) ;
      if ( ! blankFound )                          // save a copy of blank user-def
      {
         if ( (gs.find( L"--" )) >= ZERO )
         {
            gsBlank = gs ;
            blankFound = true ;
         }
      }
      remapText[i] = keyString[i] ;                // pointer to user-def text
      remapAttr[i] = (this->shm->keyMap.keydef[i].cmd) ? fColor : nColor ; // user-def color
      descText[i] = &this->shm->keyMap.keydef[i].ktext[dOffset] ; // point to description text
      descAttr[i]  = dColor ;                      // description-text color
      userCodes[i] = this->shm->keyMap.keydef[i].kuser ;
      if ( userCodes[i] != nonCode )               // count user-keycode definitions
         ++userKeys ;
   } ;

   //* Re-size and re-position the sub-dialog *
   dInit.dColumns += 6 ;
   dInit.dXoffset -= 3 ;
   dInit.dLines += 5 ;
   dInit.dYoffset -= 1 ;
   dInit.ctrlPtr[okPB].ulY += 3 ;
   dInit.ctrlPtr[canPB].ulY = dInit.ctrlPtr[okPB].ulY ;
   dInit.ctrlPtr[okPB].ulX = 3 ;
   dInit.ctrlPtr[canPB].ulX = dInit.ctrlPtr[okPB].ulX + dInit.ctrlPtr[okPB].cols + 3 ;

   //* Attach additional controls *
   dInit.ctrlPtr[canPB].nextCtrl = &dInit.ctrlPtr[keySE] ;
   dInit.ctrlPtr[keySE] = 
   {  //* User-defined-key Scrollext  - - - - - - - - - - - - - - -      keySE *
      dctSCROLLEXT,                 // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      2,                            // ulY:       upper left corner in Y
      ZERO,                         // ulX:       upper left corner in X
      short(dInit.dLines - 7),      // lines:     control lines
      19,                           // cols:      control columns
      NULL,                         // dispText:  (n/a)
      dInit.interiorColor,          // nColor:    non-focus border color
      dInit.interiorColor,          // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "REMAPPED KEY",               // label:     
      -1,                           // labY:      offset from control's ulY
      1,                            // labX       offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      ZERO,                         // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &dInit.ctrlPtr[defSE]         // nextCtrl:  link in next structure
   } ;
   dInit.ctrlPtr[defSE] = 
   { //* Default-key-definition Scrollbox  - - - - - - - - - - - - - - - defSE *
      dctSCROLLEXT,                 // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      dInit.ctrlPtr[keySE].ulY,     // ulY:       upper left corner in Y
      short(dInit.ctrlPtr[keySE].ulX + dInit.ctrlPtr[keySE].cols), // ulX: upper left corner in X
      dInit.ctrlPtr[keySE].lines,   // lines:     control lines
      short(dInit.dColumns - dInit.ctrlPtr[keySE].cols), // cols: control columns
      NULL,                         // dispText:  display items
      dInit.interiorColor,          // nColor:    non-focus border color
      dInit.interiorColor,          // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "DEFAULT        DESCRIPTION", // label:     
      -1,                           // labY:      offset from control's ulY
      2,                            // labX       offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    index of initial highlighted element
      NULL,                         // scrColor:  single-color data display
      NULL,                         // spinData:  (n/a)
      false,                        // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   } ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Make visual connections with borders *
      cdConnect ccb ;
      ccb.connection = ccb.ul2Left = ccb.ll2Left = true ;
      dp->ConnectControl2Border ( keySE, ccb ) ;
      ccb.ul2Left  = ccb.ll2Left = false ;
      ccb.ur2Right = ccb.lr2Right = true ;
      dp->ConnectControl2Border ( defSE, ccb ) ;

      //* Initialize the dynamic display data *
      ssetData ssd( remapText, remapAttr, KEYMAP_KEYS, ZERO, false ) ;
      dp->SetScrollextText ( keySE, ssd ) ;
      ssd.dispText  = descText ;
      ssd.dispColor = descAttr ;
      ssd.hlShow    = true ;
      dp->SetScrollextText ( defSE, ssd ) ;

      //* Static help message *
      winPos msgPos( dInit.ctrlPtr[canPB].ulY, dInit.ctrlPtr[canPB].ulX + 11 ) ;
      dp->WriteParagraph ( msgPos, quickHelp, hColor ) ;
      msgPos = { short(dInit.ctrlPtr[okPB].ulY + 1), 1 } ; // position for error messages
      dp->RefreshWin () ;           // make everything visible

      //* Establish a callback method for highlight tracking *
      dp->EstablishCallback ( &EditKM_Callback ) ;
      kmdp = dp ;

      uiInfo   Info ;               // user interface data returned here
      short    icIndex = ZERO ;     // index of control with input focus
      bool     done = false ;       // loop control

      //* Disable the OK pushbutton. It will be active only when *
      //* there are (verified) changes to the keymap.            *
      icIndex = dp->PrevControl () ;
      dp->ControlActive ( okPB, false ) ;

      while ( ! done )
      {
         //* If focus is currently on a Pushbutton *
         if ( dInit.ctrlPtr[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == okPB )
               {
                  //* Reformat the keymap descriptive *
                  //* text to include user's changes. *
                  this->shm->keyMap.initialize( false, bool(userKeys > ZERO) ) ;

                  #if 0    // DEBUG ONLY - OUTPUT THE RESULTS
                  const char* ecoOut = "ecoEditKM.cfg" ;
                  ofstream ofs( ecoOut, ofstream::out | ofstream::trunc ) ;
                  if ( ofs.is_open() )
                  {
                     ofs << "Formatted output returned from ecoEditKM method.\n"
                            "================================================\n\n" ;
                  
                     gs.compose( "File    : %s\n"
                                 "kmInit  : %hhd   kmActive: %hhd\n\n"
                                 "USER         DEFAUT       C M    DESCRIPTION\n"
                                 "---- ----    ---- ----    - -    ------------------------------------\n",
                                 this->shm->keyMap.keyFile, &this->shm->keyMap.kmInit,
                                 &this->shm->keyMap.kmActive ) ;
                     ofs << gs.ustr() ;
                  
                     for ( short i = ZERO ; i < KEYMAP_KEYS ; ++i )
                     {
                        gs.compose( "%04hX %04X    %04hX %04X    %hhd %hhd    '%s'\n",
                                    &this->shm->keyMap.keydef[i].kuser.type,
                                    &this->shm->keyMap.keydef[i].kuser.key,
                                    &this->shm->keyMap.keydef[i].kdflt.type,
                                    &this->shm->keyMap.keydef[i].kdflt.key,
                                    &this->shm->keyMap.keydef[i].cmd,
                                    &this->shm->keyMap.keydef[i].map,
                                    this->shm->keyMap.keydef[i].ktext ) ;
                        ofs << gs.ustr() ;
                     }
                     ofs << endl ;
                     ofs.close() ;
                  }
                  #endif   // DEBUG ONLY

                  done = true ;
               }
               else if ( Info.ctrlIndex == canPB )
               {  //* Restore previous set of user-defined keycodes *
                  for ( short i = ZERO ; i < KEYMAP_KEYS ; ++i )
                     this->shm->keyMap.keydef[i].kuser = userCodes[i] ;
                  done = true ;
               }
            }
         }

         //* If focus is currently on a ScrollExt control *
         else if ( dInit.ctrlPtr[icIndex].type == dctSCROLLEXT )
         {
            gString  srcText ;            // source display text
            wkeyCode srcKeycode ;         // source keycode
            short    srcIndex,            // index of source record
                     trgIndex ;           // index of target record
            bool  seClip = false,         // 'true' if source record selected
                  seDone = false ;        // loop control

            while ( ! seDone )
            {
               icIndex = dp->EditScrollext ( Info ) ;
               if ( (Info.wk.type == wktFUNKEY) )
               {
                  if ( (Info.wk.key == nckTAB) || (Info.wk.key == nckRIGHT) ||
                       (Info.wk.key == nckSTAB) || (Info.wk.key == nckLEFT) ||
                       (Info.wk.key == nckESC) )
                  {
                     seDone = true ;   // user is finished with this control
                  }

                  //* Select a source record *
                  else if ( Info.wk.key == nckC_C )
                  {
                     srcIndex = Info.selMember ;
                     gs = clearMsgArea ;
                     dp->WriteParagraph ( msgPos, gs, mColor ) ;
                     srcText = descText[srcIndex] ;
                     srcText.limitChars( srcText.find( L' ' ) ) ;
                     srcText.padCols( dInit.ctrlPtr[keySE].cols - 2 ) ;
                     srcText.append( "\n selected" ) ;
                     dp->WriteParagraph ( msgPos, srcText, mColor, true ) ;
                     srcKeycode = this->shm->keyMap.keydef[srcIndex].kdflt ;
                     seClip = true ;
                  }

                  //* Select a target record for update *
                  else if ( Info.wk.key == nckC_V )
                  {
                     trgIndex = Info.selMember ;
                     if ( this->shm->keyMap.keydef[trgIndex].cmd && seClip )
                     {
                        gs = clearMsgArea ;
                        dp->WriteParagraph ( msgPos, gs, mColor, true ) ;
                        srcText.copy( keyString[trgIndex], tbytes ) ;
                        remapAttr[trgIndex] = fColor ;
                        this->shm->keyMap.keydef[trgIndex].kuser = srcKeycode ;
                        if ( ! this->shm->keyMap.keydef[trgIndex].map )
                           ++userKeys ;
                        this->shm->keyMap.keydef[trgIndex].map = true ;
                        seClip = false ;
                        dp->RefreshScrollextText ( keySE ) ;

                        //* Scan the keymap for duplications. *
                        //*    (see note in method header)    *
                        if ( (this->rcfValidateKeymap ( dup1, dup2 )) == csOK )
                           dp->ControlActive ( okPB, true ) ; // enable okPB
                        else
                        {
                           gs = clearMsgArea ;
                           dp->WriteParagraph ( msgPos, gs, mColor, true ) ;
                           gs.compose( in_useTemplate, keyString[trgIndex] ) ;
                           dp->WriteParagraph ( msgPos, gs, mColor, true ) ;
                           dp->ControlActive ( okPB, false ) ; // disable okPB
                        }
                     }
                     else
                        dp->UserAlert () ;
                  }

                  //* Remove user-defined keycode (if any) for this record *
                  else if ( Info.wk.key == nckC_X )
                  {
                     trgIndex = Info.selMember ;
                     if ( this->shm->keyMap.keydef[trgIndex].map )
                     {
                        gs = clearMsgArea ;
                        dp->WriteParagraph ( msgPos, gs, mColor, true ) ;
                        gsBlank.copy( keyString[trgIndex], tbytes ) ;
                        this->shm->keyMap.keydef[trgIndex].kuser = nonCode ;
                        this->shm->keyMap.keydef[trgIndex].map = false ;
                        dp->RefreshScrollextText ( keySE ) ;
                        --userKeys ;

                        //* Scan the keymap for duplications. *
                        //*    (see note in method header)    *
                        if ( (this->rcfValidateKeymap ( dup1, dup2 )) == csOK )
                           dp->ControlActive ( okPB, true ) ; // enable okPB
                        else
                        {
                           srcText = &this->shm->keyMap.keydef[trgIndex].ktext[dOffset] ;
                           srcText.limitChars( srcText.find( L' ' ) ) ;
                           gs.compose( in_useTemplate, srcText.ustr() ) ;
                           dp->WriteParagraph ( msgPos, gs, mColor, true ) ;
                           dp->ControlActive ( okPB, false ) ; // disable okPB
                        }
                     }
                     else
                        dp->UserAlert () ;   // attempt to remove nonexistant keycode
                  }
               }
            }
            dp->RefreshScrollextText ( keySE, false ) ;  // hide the highlight
         }

         //* Move focus to appropriate control *
         if ( ! done  && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }     // while(!done)
   }        // if((this->dPtr->OpenWindow())==OK)
   if ( dp != NULL )
   {
      delete dp ;
   }

   return pmods ;

}  //* End of ecoEditKM() *

//*************************
//*    EditKM_Callback    *
//*************************
//******************************************************************************
//* NON-MEMBER METHOD. Callback method for ecoEditKM().                        *
//* Tracks the position of the highlight in the user-definition Scrollext      *
//* control and synchronizes the highlight in the description Scrollext.       *
//*                                                                            *
//* Input  : currIndex: index of control that currently has focus              *
//*          wkey     : user's key input data                                  *
//*          firstTime: the EstablishCallback() method calls this method once  *
//*                     with firstTime==true, to perform any required          *
//*                     initialization. Subsequently, the NcDialog class       *
//*                     always calls with firstTime==false.                    *
//* Returns: OK                                                                *
//******************************************************************************
//* Important Note: This method makes some intimate assumptions about what     *
//* is happening in the method that established the callback. If changes are   *
//* made in the establishing method that affect the callback functionality,    *
//* be sure to update this method to address those changes.                    *
//******************************************************************************

static short EditKM_Callback ( const short currIndex, const wkeyCode wkey, bool firstTime )
{
   const short keySE = 2,          // control object indices
               defSE = 3 ;

   if ( currIndex == keySE )
   {  //* Synchronize the highlighted items *
      short kseIndex = kmdp->GetScrollextSelect ( keySE ) ;
      kmdp->SetScrollextSelect ( defSE, kseIndex ) ;
   }
   return OK ;

}  //* End EditKM_Callback() *

//*************************
//*       ecoEditMC       *
//*************************
//******************************************************************************
//* Interactively set the list of 'Mount Commands'.                            *
//*                                                                            *
//* Input  : dInit  : dialog description (first 2 controls defined)            *
//*                                                                            *
//* Returns: 'true' if user modifies settings, else 'false'                    *
//******************************************************************************

bool FmConfig::ecoEditMC ( InitNcDialog& dInit )
{
   //* Create a list of needed controls *
   enum dpCtrls { okPB, canPB, attPB, mntSE, entTB, dpcCOUNT } ;
   dInit.ctrlPtr[canPB].nextCtrl = &dInit.ctrlPtr[attPB] ;
   attr_t  singleColor[2] = { attrDFLT, this->tbnColor } ;
   short cdPathWidth = dInit.dColumns - 16 ;  // width of text-entry control
   dInit.ctrlPtr[attPB] = 
   {  //* 'ATTACHED' pushbutton - - - - - - - - - - - - - - - - - - - -  attPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      dInit.ctrlPtr[canPB].ulY,     // ulY:       upper left corner in Y
      short(dInit.dColumns - 23),   // ulX:
      1,                            // lines:     (n/a)
      22,                           // cols:      control columns
      " Attached Filesystems ",     // dispText:  
      nc.gr,                        // nColor:    non-focus color
      this->pbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &dInit.ctrlPtr[mntSE],        // nextCtrl:  link in next structure
   } ;
   dInit.ctrlPtr[mntSE] = 
   { //* 'Mount Commands' scroll ext - - - - - - - - - - - - - - - - -   mntSE *
      dctSCROLLEXT,                 // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      2,                            // ulY:       upper left corner in Y
      ZERO,                         // ulX:       upper left corner in X
      10,                           // lines:     control lines
      dInit.dColumns,               // cols:      control columns
      NULL,                         // dispText:  n/a
      this->dColor,                 // nColor:    non-focus border color
      this->pbfColor,               // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      " Mount-filesystem Commands ",// label:     
      ZERO,                         // labY:      offset from control's ulY
      ZERO,                         // labX       offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      MAX_FAVDIRS,                  // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    (n/a)
      singleColor,                  // scrColor:  single-color data display
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &dInit.ctrlPtr[entTB],        // nextCtrl:  link in next structure
   } ;
   dInit.ctrlPtr[entTB] = 
   {  //* 'Enter/edit path' Text Box   - - - - - - - - - - - - - -       entTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      1,                            // ulY:       upper left corner in Y
      short(dInit.dColumns - cdPathWidth - 1),// ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      cdPathWidth,                  // cols:      control columns
      NULL,                         // dispText:  
      dInit.interiorColor,          // nColor:    non-focus color
      this->tbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      NULL,                         // label:     
      ZERO,                         // labY:      
      -11,                          // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    initially inaccessable to user
      NULL,                         // nextCtrl:  link in next structure
   } ;

   //* Formatting for the mount-command list display data *
   const char* seboxData[MAX_MNTCMDS] ;
   gString fgs[MAX_MNTCMDS], gx ;
   for ( short i = ZERO ; i < MAX_MNTCMDS ; i++ )
   {
      fgs[i] = this->shm->mntCmd[i] ;

      //* If the pathspec width fits within the display area *
      if ( (fgs[i].gscols()) <= (dInit.dColumns - 2) )
         seboxData[i] = fgs[i].ustr() ;
      //* Else, display only the tail of the pathspec string *
      else
      {
         // Programmer's Note: This is a bit fancy because we must 
         // anticipate multi-column characters in the pathspec strings, 
         // i.e. one character does not necessarily equal one column.
         gx = fgs[i] ;
         gx.shiftCols( -((gx.gscols()) - (dInit.dColumns - 1)) ) ;
         seboxData[i] = &fgs[i].ustr()[(fgs[i].utfbytes()) - (gx.utfbytes())] ;
      }
   }
   attr_t seboxColors[MAX_MNTCMDS] = 
   { 
      singleColor[1], singleColor[1], singleColor[1], singleColor[1], 
      singleColor[1], singleColor[1], singleColor[1], singleColor[1], 
      singleColor[1], singleColor[1], singleColor[1], singleColor[1], 
      singleColor[1], singleColor[1], singleColor[1], singleColor[1], 
      singleColor[1], singleColor[1], singleColor[1], singleColor[1], 
      singleColor[1], singleColor[1], singleColor[1], singleColor[1], 
   } ;
   ssetData sData( seboxData, seboxColors, this->shm->mntCount, ZERO, true ) ;

   chrono::duration<short>aWhile( 3 ) ;
   bool pmods = false ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Make visual connections to dialog border *
      cdConnect cdConn ;
      cdConn.connection = true ;
      cdConn.ul2Left = cdConn.ll2Left = cdConn.ur2Right = cdConn.lr2Right = true ;
      dp->ConnectControl2Border ( mntSE, cdConn ) ;

      //* Draw static text *
      winPos msgPos = dp->WriteParagraph ( dInit.dLines - 7, 1,
         "Highlight the desired item using arrow keys, then:\n"
         "  Ctrl+U  : move item Up            Ctrl+R  : Remove item from list\n"
         "  Ctrl+D  : move item Down          Ctrl+A  : Add new item to list\n"
         "  Ctrl+E  : Edit item\n\n ", dInit.interiorColor ) ;

      dp->RefreshWin () ;           // make everything visible

      uiInfo   Info ;               // user interface data returned here
      short    icIndex = dp->PrevControl () ; // index of control with input focus
      bool     refreshSE = true,    // refresh contents of dctSCROLLEXT control
               done = false ;       // loop control
      while ( ! done )
      {
         if ( refreshSE )
         {
            if ( sData.dispItems == ZERO )
               fgs[sData.dispItems++] = seNoFiles ; ;
            dp->SetScrollextText ( mntSE, sData ) ;
            refreshSE = false ;
         }

         //* If focus is currently on a Pushbutton *
         if ( dInit.ctrlPtr[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == okPB )
               {  //* Copy edited strings to our data member *
                  for ( short i = ZERO ; i < MAX_MNTCMDS ; i++ )
                     fgs[i].copy( this->shm->mntCmd[i], MAX_PATH ) ;
                  this->shm->mntCount = sData.dispItems ;
                  pmods = true ;    // force parent to do a status update
                  done = true ;
               }
               else if ( Info.ctrlIndex == canPB )
                  done = true ;
               else if ( Info.ctrlIndex == attPB )
               {
                  //* Create a dialog description and call *
                  //* the attached-filesystems method.     *
                  InitCtrl ice[MAX_DIALOG_CONTROLS] ;
                  ice[okPB] = dInit.ctrlPtr[okPB] ; 
                  ice[canPB] = dInit.ctrlPtr[canPB] ; ice[canPB].nextCtrl = NULL ;
                  InitNcDialog de = dInit ;
                  de.ctrlPtr = ice ;
                  gString gsMnt ;
                  dp->SetDialogObscured () ;
                  bool newItem = this->ecoSelectFilesys ( de, gsMnt ) ;
                  dp->RefreshWin () ;

                  if ( newItem )
                  {
                     //* Save the mountpoint or URI, then format for display.*
                     if ( this->shm->mntCount < MAX_MNTCMDS )
                     {
                        //* Test for duplicates *
                        bool dupe = false ;
                        for ( short i = ZERO ; i < this->shm->mntCount ; ++i )
                        {
                           if ( (gsMnt.compare( fgs[i] )) == ZERO )
                           {
                              dupe = true ;
                              dp->UserAlert ( 2 ) ;
                              dp->ClearLine ( msgPos.ypos ) ;
                              dp->WriteString ( msgPos, 
                                 "   Selected entry duplicates an "
                                 "existing entry. Not added to list.   ",
                                 this->pbfColor, true ) ;
                              this_thread::sleep_for( aWhile ) ;
                              dp->ClearLine ( msgPos.ypos ) ;
                              break ;
                           }
                        }
                        if ( ! dupe )     // if not a duplicate
                        {
                        }
                        short i = this->shm->mntCount ;
                        fgs[i] = gsMnt ;
                        if ( (fgs[i].gscols()) <= (dInit.dColumns - 2) )
                           sData.dispText[i] = fgs[i].ustr() ;
                        else
                        {
                           gx = fgs[i] ;
                           gx.shiftCols( -((gx.gscols()) - (dInit.dColumns - 1)) ) ;
                           sData.dispText[i] = 
                              &fgs[i].ustr()[(fgs[i].utfbytes()) - (gx.utfbytes())] ;
                        }
                        ++this->shm->mntCount ;
                        ++sData.dispItems ;
                        refreshSE = true ;
                     }
                     else
                     {
                        dp->UserAlert ( 2 ) ;
                        dp->ClearLine ( msgPos.ypos ) ;
                        dp->WriteString ( msgPos, 
                           "   Sorry, maximum entries. "
                           "Remove an entry before adding a new one.   ",
                           this->pbfColor, true ) ;
                        this_thread::sleep_for( aWhile ) ;
                        dp->ClearLine ( msgPos.ypos ) ;
                     }
                  }
               }
            }
         }

         //* If focus is currently on the Scroll Box *
         if ( dInit.ctrlPtr[icIndex].type == dctSCROLLEXT )
         {
            Info.viaHotkey = false ;      // ignore hotkey data
            icIndex = dp->EditScrollext ( Info ) ;
            if ( Info.keyIn != nckTAB && Info.keyIn != nckSTAB && Info.keyIn != nckESC )
            {
               winPos pptPos( dInit.ctrlPtr[entTB].ulY, dInit.ctrlPtr[entTB].ulX - 14 ) ;
               winPos msgPos( dInit.dLines - 2, 2 ) ;
               refreshSE = 
                  this->ecoEditScrollextData ( dp, Info, sData, fgs, mntSE, entTB, 
                                               MAX_MNTCMDS, pptPos, msgPos ) ;
               if ( refreshSE )
               {
                  for ( short i = ZERO ; i < sData.dispItems ; ++i )
                  {
                     //* If the pathspec width fits within the display area *
                     if ( (fgs[i].gscols()) <= (dInit.dColumns - 2) )
                        sData.dispText[i] = fgs[i].ustr() ;
                     //* Else, display only the tail of the pathspec string *
                     else
                     {
                        // Programmer's Note: This is a bit fancy because we must 
                        // anticipate multi-column characters in the pathspec strings, 
                        // i.e. one character does not necessarily equal one column.
                        gx = fgs[i] ;
                        gx.shiftCols( -((gx.gscols()) - (dInit.dColumns - 1)) ) ;
                        sData.dispText[i] = &fgs[i].ustr()[(fgs[i].utfbytes()) - (gx.utfbytes())] ;
                     }
                  }
               }

               //* Keep focus on scroll box until user explicitly moves on.*
               while ( (icIndex = dp->NextControl ()) != attPB ) ;
            }
         }

         //* Move focus to appropriate control *
         if ( ! done  && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }     // while(!done)
   }        // if((this->dPtr->OpenWindow())==OK)
   if ( dp != NULL )
      delete dp ;
   return pmods ;

}  //* End of ecoEditMC() *

//*************************
//*   ecoSelectFilesys    *
//*************************
//******************************************************************************
//* Called by ecoEditMC(), this method scans the system for filesystems which  *
//* are currently attached. User may select a displayed item which caller can  *
//* then add to the list of mountpoints.                                       *
//*                                                                            *
//* Input  : dInit  : dialog description (2 controls semi-defined)             *
//*          gsMnt  : (by reference, initial contents ignored)                 *
//*                   on return, MAY contain a new mount command               *
//*                                                                            *
//* Returns: 'true'  if gsMnt contains a mountpoint or URI to be added to list *
//*          'false' (gsMnt cleared)                                           *
//******************************************************************************

bool FmConfig::ecoSelectFilesys ( InitNcDialog& dInit, gString& gsMnt )
{
   const char* Labels[] = 
   {
      " SELECT ",
      " CANCEL ",
      "  Filesystems  ",
      "  Select Attached Filesystem  ",
   } ;
   attr_t  singleColor[2] = { attrDFLT, this->tbnColor } ;
   bool noItems = false,         // 'true' if no attached storage devices
        newItem = false ;        // return value

   gsMnt.clear() ;               // initialize caller's buffer

   //* Create a list of needed controls *
   enum dpCtrls : short { selPB, canPB, attSE, dpcCOUNT } ;
   dInit.dTitle = NULL ;

   //* Clean up the control definitions sent by caller *
   dInit.ctrlPtr[selPB].ulY = dInit.dLines - 2 ;
   dInit.ctrlPtr[selPB].cols = 8 ;
   dInit.ctrlPtr[selPB].dispText = Labels[selPB] ;
   dInit.ctrlPtr[selPB].nextCtrl = &dInit.ctrlPtr[canPB] ;
   dInit.ctrlPtr[canPB].ulY = dInit.ctrlPtr[selPB].ulY ;
   dInit.ctrlPtr[canPB].cols = 8 ;
   dInit.ctrlPtr[canPB].dispText = Labels[canPB] ;
   dInit.ctrlPtr[canPB].nextCtrl = &dInit.ctrlPtr[attSE] ;

   dInit.ctrlPtr[attSE] = 
   { //* 'ATTACHED' scroll ext - - - - - - - - - - - - - - - - - - - -   attSE *
      dctSCROLLEXT,                 // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      1,                            // ulY:       upper left corner in Y
      ZERO,                         // ulX:       upper left corner in X
      10,                           // lines:     control lines
      dInit.dColumns,               // cols:      control columns
      NULL,                         // dispText:  n/a
      this->pbnColor,               // nColor:    non-focus border color
      this->pbfColor,               // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      Labels[attSE],                // label:     
      ZERO,                         // labY:      offset from control's ulY
      ZERO,                         // labX       offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      MAX_FAVDIRS,                  // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    (n/a)
      singleColor,                  // scrColor:  single-color data display
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   } ;


   //* Display data for Scrollext control *
   const char* seboxData[MAX_MNTCMDS] ;
   attr_t seboxColors[MAX_MNTCMDS] = 
   { 
      singleColor[1], singleColor[1], singleColor[1], singleColor[1], 
      singleColor[1], singleColor[1], singleColor[1], singleColor[1], 
      singleColor[1], singleColor[1], singleColor[1], singleColor[1], 
      singleColor[1], singleColor[1], singleColor[1], singleColor[1], 
      singleColor[1], singleColor[1], singleColor[1], singleColor[1], 
      singleColor[1], singleColor[1], singleColor[1], singleColor[1], 
   } ;

   //* Scan for attached filesystems and initialize the display data.*
   fsInfo* fsiPtr = NULL ;
   short fsCount = this->ecoScan4Filesystems ( fsiPtr ) ;
   if ( fsCount > ZERO )
   {
      for ( short i = ZERO ; i < fsCount ; ++i )
         seboxData[i] = fsiPtr[i].fsMountpt ;
   }
   else     // no storage devices (unlikely)
   {
      seboxData[ZERO] = "No mass storage devices found on this system." ;
      fsCount = 1 ;
      noItems = true ;
   }
   ssetData sData( seboxData, seboxColors, fsCount, ZERO, true ) ;


   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Set dialog title *
      dp->SetDialogTitle ( Labels[dpcCOUNT], this->expColor ) ;
      //* Make visual connections to dialog border *
      cdConnect cdConn ;
      cdConn.connection = true ;
      cdConn.ul2Left = cdConn.ll2Left = cdConn.ur2Right = cdConn.lr2Right = true ;
      dp->ConnectControl2Border ( attSE, cdConn ) ;

      //* Draw static text *
      winPos wpi( short(dInit.ctrlPtr[attSE].ulY + dInit.ctrlPtr[attSE].lines), 4 ) ;
      wpi = dp->WriteParagraph ( wpi, 
        "This is a list of storage devices currently visible to the system.\n"
        "For block devices, the mountpoint path is listed, and for virtual\n"
        "filesystems, the Uniform Resource Identifier (URI) is listed.\n"
        "To add a listed item to the configuration-file mount list, \n"
        "highlight the desired item, and press the \"Select\" pushbutton.\n",
        this->dColor ) ;

      //* Set contents of Scrollext control *
      dp->SetScrollextText ( attSE, sData ) ;

      dp->RefreshWin () ;           // make everything visible


      uiInfo   Info ;               // user interface data returned here
      short    icIndex = dp->PrevControl () ; // index of control with input focus
      bool     done = false ;       // loop control

      if ( noItems )                // no display items
      {
         while ( icIndex != canPB )
            icIndex = dp->NextControl () ;
         dp->ControlActive ( selPB, false ) ;
         dp->ControlActive ( attSE, false ) ;
      }

      while ( ! done )
      {
         //* If focus is currently on a Pushbutton *
         if ( dInit.ctrlPtr[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               //* Copy the selected item to caller's buffer.*
               if ( Info.ctrlIndex == selPB )
               {
                  short itemIndex = dp->GetScrollextSelect ( attSE ) ;
                  gsMnt = fsiPtr[itemIndex].fsMountpt ;
                  newItem = true ;
               }
               done = true ;
            }
         }

         //* If focus is currently on the Scroll Box *
         else if ( dInit.ctrlPtr[icIndex].type == dctSCROLLEXT )
         {
            Info.viaHotkey = false ;      // ignore hotkey data
            icIndex = dp->EditScrollext ( Info ) ;
         }

         //* Move focus to appropriate control *
         if ( ! done  && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }     // while(!done)
   }        // if((this->dPtr->OpenWindow())==OK)
   if ( dp != NULL )
      delete dp ;

   return newItem ;

}  //* End ecoSelectFilesys() *

//*************************
//*  ecoScan4Filesystems  *
//*************************
//******************************************************************************
//* Called only by ecoSelectFilesys().                                         *
//*                                                                            *
//* Scan the system for "real" (physical storage) filesystems.                 *
//* 1) Scan the system for mounted filesystems. (/proc/self/mountinfo)         *
//* 2) Scan 'fstab' for defined mountpoints.    (/etc/fstab)                   *
//* 3) Add the mountpoints from config file, then eliminate duplicate entries. *
//*                                                                            *
//* Programmer's Note: This duplicates the mount scan code in the main         *
//* application except that we do care whether they are mounted, so we do not  *
//* stat the filesystems for the captured items.                               *
//*                                                                            *
//* Input  : fsCount : (by reference, initial value ignored)                   *
//*                                                                            *
//* Returns: pointer to dynamically-allocated array of fsInfo objects          *
//*          It is caller's responsibility to free the dynamic allocation.     *
//******************************************************************************

short FmConfig::ecoScan4Filesystems ( fsInfo*& fsiPtr )
{
   #define MAX_RECORDS (64)      // maximum number of records to capture
   #define VERIFY_MOUNTS (0)     // optionally verify mounts (see note below)

   fsiPtr = new fsInfo[MAX_RECORDS] ; // array of mountpoint records
   fileSystemStats fsStats ;     // determine whether the target is mounted
   gString tmpPath,              // filespec of temporary file
           cmdBuff,              // command string buffer
           gs ;                  // text formatting
   char  lineBuff[gsDFLTBYTES] ; // input buffer
   short srcIndx,                // source text index
         //trgIndx,                // target text index
         fsiIndex = ZERO ;       // index into fsiPtr array (return value)
   bool  done = false ;          // loop control

   //* Create a temporary file to capture the raw data.*
   this->CreateTempname ( tmpPath ) ;

   //**************************************************************
   //* Scan the /proc/self/mountinfo file for currently-mounted   *
   //* physical storage devices, (SATA, SSD, USB, SD cards, etc.).*
   //**************************************************************
   const char* fsTemplate = 
   "findmnt --df --noheadings -o SOURCE,FSTYPE,TARGET "
   "| grep -v '^tmpfs' "            // (exclude "tmpfs" filesystems)
   "| grep -v '[[:space:]]/sys' "   // (exculde filesystems mounted under '/sys')
   "| grep -v '[[:space:]]/dev' "   // (exculde filesystems mounted under '/dev')
   "| grep -v '[[:space:]]/var/' "  // (exculde filesystems mounted under '/var') 
   "| grep -v '[[:space:]]/$' "     // (exclude root directory)                   
   "1>>\"%s\" 2>>\"%s\"" ;             
   cmdBuff.compose( fsTemplate, tmpPath.ustr(), tmpPath.ustr() ) ;
   system ( cmdBuff.ustr() ) ;

   //* Open the temp file and extract the interesting data.*
   ifstream ifs( tmpPath.ustr(), ifstream::in ) ; // open the file
   if ( ifs.is_open() )                           // if file opened
   {
      while ( ! done && (fsiIndex < MAX_RECORDS) )
      {
         ifs.getline ( lineBuff, gsDFLTBYTES, NEWLINE ) ;// read a source line
         if ( ifs.good() || ifs.gcount() > ZERO )        // if a good read
         {
            gs = lineBuff ;
            gs.strip() ;            // strip any leading/trailing whitespace
            //* Extract device filespec.*
            srcIndx = gs.find( SPACE ) ;
            if ( srcIndx < ZERO )  srcIndx = gsDFLTBYTES ;
            gs.substr( fsiPtr[fsiIndex].fsDevice, ZERO, srcIndx ) ;
            gs.shiftChars( -(srcIndx) ) ;
            gs.strip( true ) ;
            //* Extract device type.*
            srcIndx = gs.find( SPACE ) ;
            if ( srcIndx < ZERO )  srcIndx = gsDFLTBYTES ;
            gs.substr( fsiPtr[fsiIndex].fsType, ZERO, srcIndx ) ;
            gs.shiftChars( -(srcIndx) ) ;
            gs.strip( true ) ;
            //* Extract mountpoint filespec.*
            srcIndx = gs.find( SPACE ) ;
            if ( srcIndx < ZERO )  srcIndx = gsDFLTBYTES ;
            gs.copy( fsiPtr[fsiIndex].fsMountpt, gsDFLTBYTES ) ;

            fsiPtr[fsiIndex].fsMounted = true ;

            ++fsiIndex ;   // increment the record count
         }
         else           // end-of-file found
            done = true ;
      }
      ifs.close() ;           // close the file
   }

   //********************************************************
   //* Scan the /etc/fstab file for mountpoint definitions, *
   //* (may, or may not be currently mounted).              *
   //********************************************************
   //* Discard current contents of temp file.*
   ofstream ofs( tmpPath.ustr(), ofstream::out | ofstream::trunc ) ;
   if ( ofs.is_open() )
      ofs.close() ;

   const char* const fstabTemplate = 
         "findmnt --fstab --noheadings -o SOURCE,FSTYPE,TARGET "
         "| grep -v 'none$' "             // (exclude swap partition)
         "| grep -v '[[:space:]]/$' "     // (exclude root directory)                   
         "1>>\"%s\" 2>>\"%s\"" ;
   cmdBuff.compose( fstabTemplate, tmpPath.ustr(), tmpPath.ustr() ) ;
   system ( cmdBuff.ustr() ) ;
   //* Open the temp file and extract the interesting data.*
   ifs.open( tmpPath.ustr(), ifstream::in ) ; // open the file
   if ( ifs.is_open() )                           // if file opened
   {
      done = false ;

      while ( ! done && (fsiIndex < MAX_RECORDS) )
      {
         ifs.getline ( lineBuff, gsDFLTBYTES, NEWLINE ) ;// read a source line
         if ( ifs.good() || ifs.gcount() > ZERO )        // if a good read
         {
            gs = lineBuff ;
            gs.strip() ;            // strip any leading/trailing whitespace
            //* Extract device filespec.*
            srcIndx = gs.find( SPACE ) ;
            if ( srcIndx < ZERO )  srcIndx = gsDFLTBYTES ;
            gs.substr( fsiPtr[fsiIndex].fsDevice, ZERO, srcIndx ) ;
            gs.shiftChars( -(srcIndx) ) ;
            gs.strip( true ) ;
            //* Extract device type.*
            srcIndx = gs.find( SPACE ) ;
            if ( srcIndx < ZERO )  srcIndx = gsDFLTBYTES ;
            gs.substr( fsiPtr[fsiIndex].fsType, ZERO, srcIndx ) ;
            gs.shiftChars( -(srcIndx) ) ;
            gs.strip( true ) ;
            //* Extract mountpoint filespec.*
            srcIndx = gs.find( SPACE ) ;
            if ( srcIndx < ZERO )  srcIndx = gsDFLTBYTES ;
            gs.copy( fsiPtr[fsiIndex].fsMountpt, gsDFLTBYTES ) ;

            ++fsiIndex ;   // increment the record count
         }
         else           // end-of-file found
            done = true ;
      }
      ifs.close() ;           // close the file
   }

   //********************************************************
   //* Scan for optical-drive filesystems attached system.  *
   //* (may, or may not be currently mounted).              *
   //********************************************************
   usbDevice *usbdPtr = NULL ;
   short dvdCount = this->ecoDVD_DeviceStats ( usbdPtr ) ;
   for ( short dvdIndex = ZERO ; dvdIndex < dvdCount ; ++dvdIndex )
   {
      if ( usbdPtr[dvdIndex].optiDrive && (usbdPtr[dvdIndex].uri[0] != NULLCHAR) )
      {
         gs = usbdPtr[dvdIndex].devpath ;
         gs.copy( fsiPtr[fsiIndex].fsDevice, FS_BYTES ) ;
         gs = usbdPtr[dvdIndex].uri ;
         gs.copy( fsiPtr[fsiIndex].fsMountpt, FS_BYTES ) ;
         fsiPtr[fsiIndex].fsMounted = usbdPtr[dvdIndex].mounted ;
         gs = usbdPtr[dvdIndex].fstype ;
         gs.copy( fsiPtr[fsiIndex].fsType, FS_BYTES ) ;
         fsiPtr[fsiIndex].fsDvdMedia = usbdPtr[dvdIndex].has_media ;
         gs = usbdPtr[dvdIndex].uri ;
         gs.copy( fsiPtr[fsiIndex].fsUri, FS_BYTES ) ;
         fsiPtr[fsiIndex].fsDvd = true ;     // indicates optical-drive filesystem

         ++fsiIndex ;    // filesystem record initialized
      }
   }
   if ( usbdPtr != NULL )     // release the dynamic memory allocation
   { delete [] usbdPtr ; usbdPtr = NULL ; }

   //********************************************************
   //* Scan for mtb/gvfs filesystems attached via USB.      *
   //* (may, or may not be currently mounted).              *
   //********************************************************
   usbdPtr = NULL ;
   short usbCount = this->ecoUSB_DeviceStats ( usbdPtr ) ;
   for ( short usbIndex = ZERO ; usbIndex < usbCount ; ++usbIndex )
   {
      gs = usbdPtr[usbIndex].devpath ;
      gs.copy( fsiPtr[fsiIndex].fsDevice, FS_BYTES ) ;
      gs = usbdPtr[usbIndex].uri ;
      gs.copy( fsiPtr[fsiIndex].fsMountpt, FS_BYTES ) ;
      fsiPtr[fsiIndex].fsMounted = usbdPtr[usbIndex].mounted ;
      gs = usbdPtr[usbIndex].fstype ;
      gs.copy( fsiPtr[fsiIndex].fsType, FS_BYTES ) ;
      gs = usbdPtr[usbIndex].uri ;
      gs.copy( fsiPtr[fsiIndex].fsUri, FS_BYTES ) ;
      fsiPtr[fsiIndex].fsMtp = true ;     // indicates MTP/GVfs filesystem

      ++fsiIndex ;    // filesystem record initialized
   }
   if ( usbdPtr != NULL )     // release the dynamic memory allocation
   { delete [] usbdPtr ; usbdPtr = NULL ; }

   unlink ( tmpPath.ustr() ) ;  // delete the temporary file

   return fsiIndex ;

}  //* End ecoScan4Filesystems() *

//*************************
//*  ecoUSB_DeviceStats   *
//*************************
//******************************************************************************
//* Scan for devices attached to the Universal Serial Bus (USB).               *
//* Capture filesystem and support information for the devices.                *
//* This method is designed primarily for smartphones, tablets and other       *
//* virtual filesystems attached to the USB bus. Devices may, or may not be    *
//* mounted and visible to the user.                                           *
//*                                                                            *
//* Important Note: This method allocates dynamic memory. It is the caller's   *
//*                 responsibility to release this memory when it is no longer *
//*                 needed. Example: delete [] usbDev ;                        *
//*                                                                            *
//* Important Note: If either 'lsusb' or 'gio' is not installed on the system, *
//*                 results will be unreliable.                                *
//*                                                                            *
//* Programmer's Note: This is a simplified version of the USB device scan in  *
//*                    the main application.                                   *
//*                                                                            *
//* Input  : usbdPtr: (by reference) receives a pointer to an array of         *
//*                   usbDevice objects (NULL if unable to perfor scan)        *
//*                                                                            *
//* Returns: number of USB devices identified                                  *
//******************************************************************************

short FmConfig::ecoUSB_DeviceStats ( usbDevice*& usbdPtr )
{
   const char* const lsusbTemplate =      // capture only MTP devices
         "lsusb | grep -i 'MTP mode' 1>\"%s\" 2>/dev/null" ;
   const char* const giomountTemplate = 
         "gio mount --list --detail 1>\"%s\" 2>/dev/null" ;
   const char* const devfmtTemplate = "/%03hd/%03hd" ;

   char  lineBuff[gsDFLTBYTES] ; // input buffer
   gString tmpPath,              // filespec of temporary file
           cmdBuff,              // command string buffer
           gs ;                  // text formatting
   short usbIndex = ZERO,        // index into 'usbRec'
         usbCount = ZERO ;       // return value
   bool  done = false ;          // loop control

   //* Create a temporary file to capture the raw data.*
   this->CreateTempname ( tmpPath ) ;

   cmdBuff.compose( lsusbTemplate, tmpPath.ustr() ) ;
   system ( cmdBuff.ustr() ) ;

   //* Open the temp file.*
   ifstream ifs( tmpPath.ustr(), ifstream::in ) ; // open the file
   if ( ifs.is_open() )                           // if file opened
   {
      done = false ;

      //* Count the number of devices attached to the USB.*
      while ( ! done )
      {
         ifs.getline ( lineBuff, gsDFLTBYTES, NEWLINE ) ;// read a source line
         if ( ifs.good() || ifs.gcount() > ZERO )        // if a good read
            ++usbCount ;
         else           // end-of-file found
            done = true ;
      }

      //* If records to be processed *
      if ( usbCount > ZERO )
      {
         //* Instantiate the required storage *
         usbdPtr = new usbDevice[usbCount] ;

         //* Rewind to top of file *
         ifs.clear() ;                 // reset the EOF flag
         ifs.seekg( ZERO ) ;           // return to top of input file

         //* Capture the data for each record *
         for ( usbIndex = ZERO ; usbIndex < usbCount ; ++usbIndex )
         {
            ifs.getline ( lineBuff, gsDFLTBYTES, NEWLINE ) ;// read a source line
            if ( ifs.good() || ifs.gcount() > ZERO )        // if a good read
            {
               gs = lineBuff ;            // copy of input line
               if ( (gs.gscanf( L"Bus %hd Device %hd: ID %hx:%hx %256[^\n]",
                          &usbdPtr[usbIndex].bus,    &usbdPtr[usbIndex].dev, 
                          &usbdPtr[usbIndex].vendor, &usbdPtr[usbIndex].product, 
                           usbdPtr[usbIndex].desc )) == 5 )
               {
                  //* Format the device information *
                  gs.compose( devfmtTemplate, 
                              &usbdPtr[usbIndex].bus, &usbdPtr[usbIndex].dev ) ;
                  gs.copy( usbdPtr[usbIndex].devfmt, fssLEN ) ;

               }
            }
            else        // file error (unlikely)
               break ;
         }
      }
      ifs.close() ;           // close the file
   }

   //* If one or more records found, find the device  *
   //* filespec and other information for each record.*
   if ( usbCount > ZERO )
   {
      cmdBuff.compose( giomountTemplate, tmpPath.ustr() ) ; // create command
      system ( cmdBuff.ustr() ) ;                           // execute command

      //* Open the temp file and extract the interesting data.*
      ifs.open( tmpPath.ustr(), ifstream::in ) ; // open the file
      if ( ifs.is_open() )                           // if file opened
      {
         short di ;                       // input buffer index
         done = false ;                   // loop control

         while ( ! done )
         {
            ifs.getline ( lineBuff, gsDFLTBYTES, NEWLINE ) ;// read a source line
            if ( ifs.good() || ifs.gcount() > ZERO )        // if a good read
            {
               gs = lineBuff ;

               for ( usbIndex = ZERO ; usbIndex < usbCount ; ++usbIndex )
               {
                  if ( ! usbdPtr[usbIndex].init &&
                       ((gs.find( usbdPtr[usbIndex].devfmt )) >= ZERO) &&
                       ((di = gs.after( "unix-device: '" )) > ZERO) )
                  {
                     //* Save device-driver filespec *
                     gs.gscanf( di, L"%255[^']", usbdPtr[usbIndex].devpath ) ;
                     ifs.getline ( lineBuff, gsDFLTBYTES, NEWLINE ) ;// read a source line
                     if ( ifs.good() || ifs.gcount() > ZERO )        // if a good read
                     {
                        gs = lineBuff ;
                        if ( (di = gs.after( "activation_root=" )) > ZERO )
                        {
                           gs.gscanf( di, L"%255[^\n]", usbdPtr[usbIndex].uri ) ;

                           //* Assume filesystem type == "mtpfs".*
                           gs = "mtpfs" ;
                           gs.copy( usbdPtr[usbIndex].fstype, fssLEN ) ;
                        }
                     }
                     usbdPtr[usbIndex].init = true ; // device record processed
                     break ;
                  }     // (matching 'devfmt')
               }        // for(;;)
            }           // (good read of source line)
            else        // end-of-file found
               done = true ;
         }
         ifs.close() ;           // close the file
      }
   }

   unlink ( tmpPath.ustr() ) ;  // delete the temporary file

   return usbCount ;

}  //* End ecoUSB_DeviceStats()

//*************************
//*    DVD_DeviceStats    *
//*************************
//******************************************************************************
//* Scan for drives visible to the local system.                               *
//* Capture filesystem and support information for the devices.                *
//*                                                                            *
//* This method is designed primarily for locating optical drives              *
//* (DVD,CD,BLU-RAY, etc). Data discs contained in the drive may, or may not   *
//* be mounted and visible to the user.                                        *
//*                                                                            *
//* Optionally, this method can report information for all attached drives.    *
//* This option is not used by the FileMangler application, but may be helpful *
//* in other applications.                                                     *
//*                                                                            *
//* Important Note: This method allocates dynamic memory. It is the caller's   *
//*                 responsibility to release this memory when it is no longer *
//*                 needed. Example: delete [] usbDev ;                        *
//*                                                                            *
//* Important Note: If 'gio' is not installed on the system, results will      *
//*                 be unreliable.                                             *
//*                                                                            *
//* Programmer's Note: This is a simplified version of the DVD device scan in  *
//*                    the main application.                                   *
//*                                                                            *
//* Input  : usbdPtr: (by reference) receives a pointer to an array of         *
//*                   usbDevice objects (NULL if unable to perfor scan)        *
//*                                                                            *
//* Returns: number of optical-drive devices identified                        *
//******************************************************************************
//* Notes:                                                                     *
//* 1) Audio discs will have a URI and therefore have a virtual device driver. *
//* 2) Commercial and cloned video discs will have NO URI, and therefore will  *
//*    have a kernel "real" device driver.
//* 3) Data discs will have NO URI, and therefore will have a kernel "real"    *
//*    device driver.                                                          *
//*                                                                            *
//******************************************************************************

short FmConfig::ecoDVD_DeviceStats ( usbDevice*& usbdPtr )
{
   const char* const giomountTemplate = 
         "gio mount --list --detail 1>\"%s\" 2>/dev/null" ;
   const char* const driveTag  = "Drive(" ;
   const char* const deviceTag = "unix-device: '" ;   // contains device-driver filespec
   const char* const optiDev   = "/sr" ;              // indicates an optical-device driver
   const char* const themeIcon = "themed icons: " ;   // list of tag strings
   const char* const optiIcon  = "optical" ;          // indicates an optical drive
   const char* const sortKey   = "sort_key=" ;        // last element of base record
   const char* const volumeTag = "Volume(0): " ;      // begins media record
   const char* const mountTag  = "Mount(0): " ;       // begins mounted-media record

   gString tmpPath ;             // filespec of temporary file
   usbDevice usbDev ;            // filesystem information
   short dvdCount = ZERO ;       // return value

   char  lineBuff[gsDFLTBYTES] ; // input buffer
   gString cmdBuff,              // command string buffer
           gs ;                  // text formatting
   short dvdIndex = ZERO ;       // index into 'usbdPtr' array
   bool  pushback = false ;      // 'true' if lineBuff contains unprocessed data

   //* Create a temporary file to capture the raw data.*
   this->CreateTempname ( tmpPath ) ;

   cmdBuff.compose( giomountTemplate, tmpPath.ustr() ) ; // create command
   system ( cmdBuff.ustr() ) ;                           // execute command

   //* Open the temp file and extract the interesting data.*
   ifstream ifs( tmpPath.ustr(), ifstream::in ) ; // open the file
   if ( ifs.is_open() )                           // if file opened
   {
      short di ;                       // input buffer index
      bool keep,                       // 'true' if record to be retained
           done = false ;              // loop control

      //* Count the number of drives located *
      while ( ! done )
      {
         ifs.getline ( lineBuff, gsDFLTBYTES, NEWLINE ) ;// read a source line
         if ( ifs.good() || ifs.gcount() > ZERO )        // if a good read
         {
            gs = lineBuff ;            // working copy of input line
            if ( (gs.find( driveTag )) == ZERO )
               ++dvdCount ;
         }
         else           // end-of-file found
            done = true ;
      }

      //* If one or more records found, find the device  *
      //* filespec and other information for each record.*
      if ( dvdCount > ZERO )
      {
         //* Instantiate storage for each record *
         usbdPtr = new usbDevice[dvdCount] ;

         //* Rewind to top of file *
         ifs.clear() ;                 // reset the EOF flag
         ifs.seekg( ZERO ) ;           // return to top of input file

         //* Capture and format the data                      *
         //* 'all' == false : capture optical-drive data only *
         //* 'all' != false : capture data for all drives     *
         while ( dvdIndex < dvdCount )
         {
            //* If input buffer contains unprocessed data, use it.*
            //* Else, read a line from the source file.           *
            if ( ! pushback )
               ifs.getline ( lineBuff, gsDFLTBYTES, NEWLINE ) ;
            if ( ifs.good() || ifs.gcount() > ZERO || pushback )
            {
               gs = lineBuff ;            // copy of input line
               keep = false ;             // reset the keep-record flag
               pushback = false ;         // reset the pushback flag

               //* Beginning of drive record.         *
               //* Provisionally save the descriptor, *
               //* and parse the record.              *
               if ( (gs.find( driveTag )) == ZERO )
               {
                  gs.copy( usbdPtr[dvdIndex].desc, fssLEN ) ;

                  //* Locate the filespec of the device driver *
                  do
                  {
                     ifs.getline ( lineBuff, gsDFLTBYTES, NEWLINE ) ;
                     if ( ifs.good() || ifs.gcount() > ZERO )
                        gs = lineBuff ;
                     else     // unexpected EOF (unlikely)
                     { di = ERR ; break ; }
                  }
                  while ( (di = gs.after( deviceTag )) <= ZERO ) ;
                  if ( di > ZERO )
                  {
                     if ( (gs.find( optiDev )) >= ZERO )
                     {
                        gs.shiftChars( -(di) ) ;
                        gs.erase( L"'" ) ;
                        gs.copy( usbdPtr[dvdIndex].devpath, fssLEN ) ;
                        keep = true ;
                     }
                  }

                  //* If record is to be retained *
                  if ( keep )
                  {
                     do
                     {
                        ifs.getline ( lineBuff, gsDFLTBYTES, NEWLINE ) ;
                        gs = lineBuff ;
                        if ( ifs.good() || ifs.gcount() > ZERO )
                        {
                           if ( (di = gs.after( themeIcon )) > ZERO )
                           {
                              //* If both "srX" and "optical" tests are *
                              //* successful, set optical-drive flag.   *
                              if ( (gs.find( optiIcon )) >= ZERO )
                                 usbdPtr[dvdIndex].optiDrive = true ;
                           }
                           else if ( (di = gs.after( "has_media=" )) > ZERO )
                           {
                              if ( (gs.gstr()[di]) == L'1' )
                                 usbdPtr[dvdIndex].has_media = true ;
                           }
                           else if ( (di = gs.after( "can_eject=" )) > ZERO )
                           {
                              if ( (gs.gstr()[di]) == L'1' )
                                 usbdPtr[dvdIndex].can_eject = true ;
                           }
                        }
                        else     // unexpected EOF (unlikely)
                           break ;
                     }
                     while ( (di = gs.after( sortKey )) < ZERO ) ;

                     //* For optical drives only, read the next line.  *
                     //* 1) If line describes media contained in the   *
                     //*    drive, capture the media sub-record.       *
                     //*    Example: "Volume(0) Audio Disc"            *
                     //* 2) If line begins a new drive record, push it *
                     //*    back into the stream.                      *
                     if ( usbdPtr[dvdIndex].optiDrive )
                     {
                        bool volFound = false,
                             mntFound = false ;

                        while ( true )
                        {
                           ifs.getline ( lineBuff, gsDFLTBYTES, NEWLINE ) ;
                           if ( ifs.good() || ifs.gcount() > ZERO )
                           {
                              gs = lineBuff ;

                              //* If this data begins a new drive record, *
                              //* it will be processed at the top of loop.*
                              if ( (gs.find( driveTag )) >= ZERO )
                              { pushback = true ; break ; }

                              if ( (di = gs.after( volumeTag )) >= ZERO )
                              {
                                 gs.substr( usbdPtr[dvdIndex].label, di, fssLEN ) ;
                                 volFound = true ;
                              }
                              else if ( volFound && !mntFound &&
                                        ((di = gs.after( "activation_root=" )) > ZERO) )
                              {
                                 gs.substr( usbdPtr[dvdIndex].uri, di, fssLEN ) ;

                                 //* Filesystem type is unknown unless we stat *
                                 //* the filesystem, so we use generic type.   *
                                 gs = gvfsDriver ;
                                 gs.copy( usbdPtr[dvdIndex].fstype, fssLEN ) ;
                              }
                              else if ( volFound && !mntFound &&
                                        ((di = gs.after( "can_mount=" )) > ZERO) )
                              {
                                 if ( (gs.gstr()[di]) == L'1' )
                                    usbdPtr[dvdIndex].can_mount = true ;
                              }
                              else if ( volFound && !mntFound &&
                                        ((di = gs.after( "label: '" )) > ZERO) )
                              {
                                 gs.shiftChars( -(di) ) ;
                                 if ( (di = gs.findlast( L"'" )) == ((gs.gschars()) - 2) )
                                    gs.limitChars( di ) ;
                                 gs.copy( usbdPtr[dvdIndex].label, fssLEN ) ;
                              }
                              else if ( volFound && !mntFound &&
                                        ((di = gs.after( "uuid: '" )) > ZERO) )
                              {
                                 gs.shiftChars( -(di) ) ;
                                 if ( (di = gs.findlast( L"'" )) == ((gs.gschars()) - 2) )
                                    gs.limitChars( di ) ;
                                 gs.copy( usbdPtr[dvdIndex].uuid, fssLEN ) ;
                              }
                              else if ( volFound && !mntFound &&
                                        ((di = gs.after( "uuid=" )) > ZERO) )
                              {
                                 gs.substr( usbdPtr[dvdIndex].uuid, di, fssLEN ) ;
                              }

                              //* If media are currently mounted *
                              else if ( volFound && 
                                        ((di = gs.after( mountTag )) >= ZERO) )
                              {
                                 mntFound = true ;
                              }
                              else if ( volFound && mntFound &&
                                        ((di = gs.after( "can_unmount=" )) > ZERO) )
                              {
                                 if ( (gs.gstr()[di]) == L'1' )
                                    usbdPtr[dvdIndex].can_unmount = true ;
                              }
                              else if ( volFound && mntFound &&
                                        ((di = gs.after( "default_location=" )) > ZERO) )
                              {
                                 //* If no URI was found, parse this entry for *
                                 //* a valid mountpath.                        *
                                 if ( (usbdPtr[dvdIndex].uri[0] == NULLCHAR) &&
                                      ((di = gs.after( "file://" )) > ZERO) )
                                 {
                                    gs.replace( L"%20", L" ", ZERO, false, true ) ;
                                    gs.substr( usbdPtr[dvdIndex].mntpath, di, MAX_PATH ) ;
                                 }
                              }
                           }
                           else        // end-of-file
                              break ;
                        }
                     }
                     usbdPtr[dvdIndex++].init = true ;
                  }  // keep
               }     // driveTag
            }
            else     // end-of-file
               break ;
         }           // while()
      }              // dvdCount

      ifs.close() ;                    // close the file
   }

   unlink ( tmpPath.ustr() ) ;  // delete the temporary file

   return dvdCount ;

}  //* DVD_DeviceStats() *

//*************************
//* ecoEditScrollextData  *
//*************************
//******************************************************************************
//* Allows user to manipulate the strings in a dctSCROLLEXT control.           *
//* a) The raw pathspec data reside in an array of gString objects 'fgs'.      *
//* b) The display data reside in an ssetData-class object 'sData' used to     *
//*    load the data into the scrollext control.                               *
//*                                                                            *
//* This method is called by both ecoEditFD() and ecoEditMC().                 *
//*                                                                            *
//*                                                                            *
//* Input  : dp    : pointer to the active dialog window                       *
//*          Info  : (by reference) data returned from user's edit of control  *
//*          sData : (by reference) interface class for setting the display    *
//*                  data in a dctSCROLLEXT control                            *
//*          fgs   : raw pathspec strings                                      *
//*          dspSE : index of dctSCROLLEXT control for displaying data         *
//*          entTB : index of dctTEXTBOX control for editing data              *
//*          maxEnt: maximum entries allowed                                   *
//*          pptPos: Y/X position of edit prompt                               *
//*          msgPos: Y/X position for warning messages                         *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

bool FmConfig::ecoEditScrollextData ( NcDialog* dp, uiInfo& Info, ssetData& sData, 
                                      gString* fgs, short dspSE, short entTB, 
                                      short maxEnt, winPos&pptPos, winPos& msgPos )
{
   bool refreshSE = false ;

   //* Shift highlighted entry up *
   if ( (Info.wk.type == wktFUNKEY && Info.wk.key == nckC_U) ||
        (Info.wk.type == wktPRINT && 
         (Info.wk.key == 'u' || Info.wk.key == 'U')))
   {
      if ( Info.selMember > ZERO )
      {
         gString gst( fgs[Info.selMember-1] ) ;
         fgs[Info.selMember - 1] = fgs[Info.selMember] ;
         fgs[Info.selMember] = gst ;
         sData.hlIndex = Info.selMember - 1 ;
         refreshSE = true ;
      }
   }
   //* Shift highlighted entry down *
   else if ( (Info.wk.type == wktFUNKEY && Info.wk.key == nckC_D) ||
             (Info.wk.type == wktPRINT && 
              (Info.wk.key == 'd' || Info.wk.key == 'D')))
   {
      if ( Info.selMember < (sData.dispItems - 1) )
      {
         gString gst( fgs[Info.selMember+1] ) ;
         fgs[Info.selMember + 1] = fgs[Info.selMember] ;
         fgs[Info.selMember] = gst ;
         sData.hlIndex = Info.selMember + 1 ;
         refreshSE = true ;
      }
   }
   //* Edit currently-highlighted entry *
   else if ( (Info.wk.type == wktFUNKEY && Info.wk.key == nckC_E) ||
             (Info.wk.type == wktPRINT && 
              (Info.wk.key == 'e' || Info.wk.key == 'E')))
   {
      if ( (fgs[Info.selMember].compare( seNoFiles )) != ZERO )
      {  //* 1) Display a prompt.                                  *
         //* 2) Set the text to be edited.                         *
         //* 3) Set 'insert' mode and allow text box to gain focus *
         //* 4) Move input focus to text box and allow user to edit*
         //* 5) Retrieve the edited text.                          *
         //* 6) Move input focus and disable user access to control*
         //* 7) Clear the text box (so it will disappear).         *
         //* 8) Erase prompt.                                      *
         dp->WriteString ( pptPos, " Enter Path: ", this->pbfColor, true ) ;
         short seMember = Info.selMember ;
         dp->SetTextboxText ( entTB, fgs[seMember] ) ;
         dp->SetTextboxInputMode ( false ) ;
         dp->ControlActive ( entTB, true ) ;
         while ( (dp->NextControl ()) != entTB ) ;
         dp->EditTextbox ( Info ) ;
         dp->GetTextboxText ( entTB, fgs[seMember] ) ;
         while ( (dp->NextControl ()) != canPB ) ;
         dp->ControlActive ( entTB, false ) ;
         dp->SetTextboxText ( entTB, L"" ) ;
         dp->WriteString ( pptPos, "             ", this->dColor, true ) ;
         refreshSE = true ;
      }
   }
   //* Add a new entry to the list *
   else if ( (Info.wk.type == wktFUNKEY && Info.wk.key == nckC_A) ||
             (Info.wk.type == wktPRINT && 
              (Info.wk.key == 'a' || Info.wk.key == 'A')))
   {
      if ( sData.dispItems < maxEnt )
      {  //* 1) Display a prompt.                                  *
         //* 2) Set 'insert' mode and allow text box to gain focus *
         //* 3) Move input focus to text box and allow user to edit*
         //* 4) Retrieve the edited text to first free position.   *
         //* 5) Move input focus and disable user access to control*
         //* 6) Clear the text box (so it will disappear).         *
         //* 7) Erase prompt.                                      *
         dp->WriteString ( pptPos, " Enter Path: ", this->pbfColor, true ) ;
         dp->SetTextboxInputMode ( false ) ;
         dp->ControlActive ( entTB, true ) ;
         while ( (dp->NextControl ()) != entTB ) ;
         dp->EditTextbox ( Info ) ;
         short seMember = ZERO ;    // position of new entry
         if ( !((sData.dispItems == 1) && 
               (fgs[ZERO].compare( seNoFiles )) == ZERO) )
            seMember = sData.dispItems++ ;
         dp->GetTextboxText ( entTB, fgs[seMember] ) ;
         sData.hlIndex = sData.dispItems - 1 ;  // highlight new item
         while ( (dp->NextControl ()) != canPB ) ;
         dp->ControlActive ( entTB, false ) ;
         dp->SetTextboxText ( entTB, L"" ) ;
         dp->WriteString ( pptPos, "             ", this->dColor, true ) ;
         refreshSE = true ;
      }
      else
      {
         dp->WriteString ( msgPos, 
            "   Sorry, maximum entries. Remove an entry before adding a new one.   ",
            this->pbfColor, true ) ;
         chrono::duration<short>aWhile( 3 ) ;
         this_thread::sleep_for( aWhile ) ;
         dp->ClearLine ( msgPos.ypos ) ;
      }
   }
   //* Remove highlighted entry from the list *
   else if ( (Info.wk.type == wktFUNKEY && Info.wk.key == nckC_R) ||
             (Info.wk.type == wktPRINT && 
              (Info.wk.key == 'r' || Info.wk.key == 'R')))
   {
      if ( sData.dispItems > ZERO ) // (should always be true)
      {
         short i = Info.selMember, j = i + 1 ;
         while ( j < maxEnt )                // shift subsequent items upward
            fgs[i++] = fgs[j++] ;
         fgs[--sData.dispItems].clear() ;    // erase the last item
         sData.hlIndex = ((Info.selMember < sData.dispItems) || 
                          (Info.selMember == ZERO)) ? 
                          Info.selMember : (Info.selMember - 1) ;
         refreshSE = true ;
      }
   }

   return refreshSE ;

}  //* End ecoEditScrollextData() *

//*************************
//*   ecoBinaryDecision   *
//*************************
//******************************************************************************
//* Called by the parameter-editing routines which require the user to select  *
//* either 'true' or 'false'.                                                  *
//*                                                                            *
//* Input  : dInit   : description of dialog window                            *
//*          tmsg    : label for the 'true' radio button                       *
//*          fmsg    : label for the 'false' radio button                      *
//*          currSel : indicates which radio button is initially active        *
//*          dfltSel : indicates whether 'true' or 'false' is default setting  *
//*                                                                            *
//* Returns: user's selection: 'true' or 'false'                               *
//******************************************************************************

bool FmConfig::ecoBinaryDecision ( InitNcDialog& dInit, const char* tmsg, 
                                   const char* fmsg, bool currSel, bool dfltSel )
{
   //* Create a list of needed controls *
   enum dpCtrls { okPB, canPB, trueRB, falseRB, dpcCOUNT } ;
   char Labels[2][MAX_LABEL_CHARS] ; 
   dInit.ctrlPtr[canPB].nextCtrl = &dInit.ctrlPtr[trueRB] ;
   dInit.ctrlPtr[trueRB] = 
   {  //* 'true' radio button   - - - - - - - - - - - - - - - - - - - - trueRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      currSel,                      // rbSelect:  initial value
      5,                            // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[0],                    // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &dInit.ctrlPtr[falseRB],      // nextCtrl:  link in next structure
   } ;
   dInit.ctrlPtr[falseRB] = 
   {  //* 'false' radio button      - - - - - - - - - - - - - - - - -  falseRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      (currSel ? false : true),     // rbSelect:  initial value
      short(dInit.ctrlPtr[trueRB].ulY + 3),// ulY:       upper left corner in Y
      short(dInit.ctrlPtr[trueRB].ulX),// ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[1],                    // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   } ;

   //* Construct control labels *
   gString gsOut ;
   gsOut.compose( L"'true'  %s", (const char*)(dfltSel ? "(default)" : " ") ) ;
   gsOut.copy( Labels[0], MAX_LABEL_CHARS ) ;
   gsOut.compose( L"'false'  %s", (const char*)(!dfltSel ? "(default)" : " ") ) ;
   gsOut.copy( Labels[1], MAX_LABEL_CHARS ) ;

   bool decision = currSel ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;
   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Group all radio buttons in the dialog *
      short XorGroup[] = { trueRB, falseRB, -1 } ;
      dp->GroupRadiobuttons ( XorGroup ) ;

      //* Display static text *
      winPos wp ;
      wp.ypos = dInit.ctrlPtr[trueRB].ulY + dInit.ctrlPtr[trueRB].labY + 1 ;
      wp.xpos = dInit.ctrlPtr[trueRB].ulX + dInit.ctrlPtr[trueRB].labX ;
      dp->WriteParagraph ( wp, tmsg, this->expColor ) ;
      wp.ypos = dInit.ctrlPtr[falseRB].ulY + dInit.ctrlPtr[falseRB].labY + 1 ;
      wp.xpos = dInit.ctrlPtr[falseRB].ulX + dInit.ctrlPtr[falseRB].labX ;
      dp->WriteParagraph ( wp, fmsg, this->expColor ) ;

      dp->RefreshWin () ;           // make everything visible

      uiInfo   Info ;               // user interface data returned here
      short    icIndex = ZERO ;     // index of control with input focus
      bool     done = false ;       // loop control
      while ( ! done )
      {
         //* If focus is currently on a Pushbutton *
         if ( dInit.ctrlPtr[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == okPB )
               {  //* Get user's selection and return it to caller *
                  short newIndex = dp->GetRbGroupSelection ( trueRB ) ;
                  decision = newIndex == trueRB ? true : false ;
               }
               done = true ;
            }
         }

         //* If focus is currently on a Radio Button *
         else if ( dInit.ctrlPtr[icIndex].type == dctRADIOBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditRadiobutton ( Info ) ;
         }

         //* Move focus to appropriate control *
         if ( ! done && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }     // while(!done)
   }        // if((this->dPtr->OpenWindow())==OK)
   if ( dp != NULL )
      delete dp ;
   return decision ;

}  //* End of ecoBinaryDecision() *

//*************************
//*   ecoDisplayCurrent   *
//*************************
//******************************************************************************
//* Display a list of current values for the configuration parameters.         *
//*                                                                            *
//* Input  : wp   : top of list, position at which to begin writing            *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* IMPORTANT NOTE: This method relies upon the CfgComp-class objects being    *
//* synchronized with the associated enum definitions. KEEP THEM SYNCHRONIZED! *
//*       wmArgs   and   enum WinMode                                          *
//*       coArgs   and   enum owTypes                                          *
//*       scArgs   and   enum NcBaseColors                                     *
//*       epArgs   and   enum LaunchCodes                                      *
//*       soArgs   and   enum fmSort                                           *
//******************************************************************************

void FmConfig::ecoDisplayCurrent ( const winPos& wp )
{
   gString  gsOut ;
   const wchar_t* argPtr ;
   attr_t   errorColor = nc.rebl ;
   short    y = wp.ypos ;

   //* Parameterized commands *
   for ( short index = ZERO ; index < cciEC ; index++ )
   {
      this->dPtr->ClearLine ( y, false, wp.xpos ) ;
      switch ( index )
      {
         case cciWM:       // WindowMode
            gsOut.compose( L"'%S'", wmArgs[this->shm->cfgOpt.winMode].str ) ;
            break ;
         case cciSO:       // SortOption
            gsOut.compose( L"'%S'", soArgs[this->shm->cfgOpt.sortOption].str ) ;
            break ;
         case cciCS:       // CaseSensitive
            argPtr = this->shm->cfgOpt.caseSensitive ? csArgs[0].str : csArgs[1].str ;
            gsOut.compose( L"'%S'", argPtr ) ;
            break ;
         case cciHF:       // HiddenFiles
            argPtr = this->shm->cfgOpt.showHidden ? hfArgs[0].str : hfArgs[1].str ;
            gsOut.compose( L"'%S'", argPtr ) ;
            break ;
         case cciCD:       // ConfirmDelete
            argPtr = this->shm->cfgOpt.confirmDelete ? cdArgs[0].str : cdArgs[1].str ;
            gsOut.compose( L"'%S'", argPtr ) ;
            break ;
         case cciCO:       // ConfirmOverwrite
            gsOut.compose( L"'%S'", coArgs[this->shm->cfgOpt.overWrite].str ) ;
            break ;
         case cciLM:       // LockMenuBar
            argPtr = this->shm->lockMb ? lmArgs[0].str : lmArgs[1].str ;
            gsOut.compose( L"'%S'", argPtr ) ;
            break ;
         case cciSC:       // ColorScheme
            switch ( this->shm->cfgOpt.cScheme.scheme )
            {
               case ncbcBK:      argPtr = scArgs[1].str ;   break ;
               case ncbcRE:      argPtr = scArgs[2].str ;   break ;
               case ncbcGR:      argPtr = scArgs[3].str ;   break ;
               case ncbcBR:      argPtr = scArgs[4].str ;   break ;
               case ncbcBL:      argPtr = scArgs[5].str ;   break ;
               case ncbcMA:      argPtr = scArgs[6].str ;   break ;
               case ncbcCY:      argPtr = scArgs[7].str ;   break ;
               case ncbcGY:      argPtr = scArgs[8].str ;   break ;
               case ncbcDEFAULT: argPtr = scArgs[9].str ;   break ;  // terminal default
               case ncbcCOLORS:                                      // application default
               default:          argPtr = scArgs[0].str ;   break ;
            }
            gsOut.compose( L"'%S'", argPtr ) ;
            break ;
         case cciEM:       // EnableMouse
            argPtr = this->shm->cfgOpt.enableMouse ? emArgs[0].str : emArgs[1].str ;
            gsOut.compose( L"'%S'", argPtr ) ;
            break ;
         case cciEP:       // ExternalPrograms
            argPtr = epArgs[this->shm->cfgOpt.lCode].str ;
            gsOut.compose( L"'%S'", argPtr ) ;
            break ;
         case cciEC:       // EndCfgCommands (not displayed)
         default:          // (unlikely)
            gsOut = "Program error while parsing data!" ;
            break ;
      }
      this->dPtr->WriteString ( y++, wp.xpos, gsOut, this->dColor ) ;
   }

   //* 'Favorite' dirs  *
   this->dPtr->ClearLine ( y, false, wp.xpos ) ;
   gsOut.compose( L"%hd directory paths specified", &this->shm->favCount ) ;
   this->dPtr->WriteString ( y++, wp.xpos, gsOut, this->dColor ) ;

   //* Key Map *
   this->dPtr->ClearLine ( y, false, wp.xpos ) ;
   const char* kmNamePtr = *this->shm->keyMap.keyFile == NULLCHAR ?
            "default key map" : this->shm->keyMap.keyFile ;
   gsOut.compose( L"Using: %s", kmNamePtr ) ;
   this->dPtr->WriteString ( y++, wp.xpos, gsOut, this->dColor ) ;

   //* 'Mount' commands *
   this->dPtr->ClearLine ( y, false, wp.xpos ) ;
   gsOut.compose( L"%hd mount commands specified", &this->shm->mntCount ) ;
   this->dPtr->WriteString ( y++, wp.xpos, gsOut, this->dColor ) ;

   //* Alternate Trashcan location *
   this->dPtr->ClearLine ( y, false, wp.xpos ) ;
   if ( *this->shm->trashPath != NULLCHAR )
   {
      //* Verify existence of Alternate Trashcan location *
      gsOut = this->shm->trashPath ;
      bool goodAltTrash = this->rcfValidateTrashPath ( gsOut ) ;
      gsOut = this->shm->trashPath ;
      this->ecoPathFit ( gsOut, (this->dialogCols - wp.xpos - 2) ) ;
      this->dPtr->WriteString ( y++, wp.xpos, gsOut, 
         (goodAltTrash ? this->dColor : errorColor) ) ;
      if ( ! goodAltTrash )
         this->dPtr->SetRadiobuttonState ( atRB, true ) ;
   }
   else
      this->dPtr->WriteString ( y++, wp.xpos, "(none specified)", this->dColor ) ;

   this->dPtr->RefreshWin () ;

}  //* End ecoDisplayCurrent() *

//*************************
//*      ecoPathFit       *
//*************************
//******************************************************************************
//* Trim the head of the path string (if necessary) so that it will fit in     *
//* the specified display field.                                               *
//*                                                                            *
//* Input  : gsPath   : path string                                            *
//*          colsAvail: width of display field (columns)                       *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void FmConfig::ecoPathFit ( gString& gsPath, short colsAvail )
{
   if ( (gsPath.gscols()) > colsAvail )
   {
      gString gst = gsPath ;
      int width = gst.gscols(),
          offset = ZERO,
          charCount ;
      const short* colArray = gst.gscols( charCount ) ;

      while ( width > (colsAvail - 3) )
         width -= colArray[offset++] ;
      gsPath.compose( L"...%S", &gst.gstr()[offset] ) ;
   }

}  //* End ecoPathFit() *

//*************************
//*      ConfigHelp       *
//*************************
//******************************************************************************
//* Invoke the 'Info' system to display configuration help.                    *
//*                                                                            *
//* -- This method uses the NcDialog-class 'ShellOut' method                   *
//* -- The ShellOut() method saves the application dialog (if caller hasn't    *
//*    already saved it).                                                      *
//* -- Note that after returning from the shell the application dialog is      *
//*    refreshed, which releases the saved display data; so if caller has set  *
//*    the application dialog as 'obscured', then caller must save it AGAIN    *
//*    on return.                                                              *
//* -- Since we don't know about any sub-dialogs that may be open, it is       *
//*    assumed that caller has saved the display data for those dialogs and    *
//*    will restore them on return.                                            *
//*                                                                            *
//* Input  : wpStat : position for status messages in caller's dialog window   *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void FmConfig::ConfigHelp ( const winPos& wpStat )
{
   gString  helpPath( "%s/%s", this->shm->appPath, HelpFilename ),
            shellCmd ;         // construct the command

   //* Be sure the help file exists *
   bool rdAcc, wrAcc ;
   if ( (this->vtpTargetExists ( helpPath, rdAcc, wrAcc )) && rdAcc )
   {  //* The node referenced in the call is an anchor within the *
      //* 'Configuration Overview' chapter of the documentation.  *
      shellCmd.compose( "info -f '%S' -n 'Main Configuration Dialog'", 
                        helpPath.gstr() ) ;
      this->dPtr->ShellOut ( soX, shellCmd.ustr() ) ;
   }
   else
   {
      this->dPtr->ClearLine ( wpStat.ypos, false ) ;
      this->dPtr->WriteString ( wpStat.ypos, (wpStat.xpos + 21), 
                                "  Unable to access the help file.  ", 
                                this->pbfColor &(~ncrATTR), true ) ;
      chrono::duration<short>aWhile( 3 ) ;
      this_thread::sleep_for( aWhile ) ;
   }

}  //* End ConfigHelp() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

