//******************************************************************************
//* File       :  renamePattern.hpp                                            *
//* Author     :  Mahlon R. Smith                                              *
//* Date       :  01-Dec-2015                                                  *
//* Version    :  (see FileDlgVersion string in FileDlg.cpp)                   *
//*                                                                            *
//* Description: This class is used to establish a regular-expression (regexp) *
//* pattern used during batch-rename of files.                                 *
//*                                                                            *
//* The renamePattern class is used by:                                        *
//*   1. RenameSelectedFiles()                                                 *
//*      - see rsfConstructPattern()                                           *
//*      - see rsfCP_Help()                                                    *
//*   2. pclCopyWithRename()                                                   *
//*      - see pclCopyFileList()                                               *
//*                                                                            *
//* Developed using GNU G++ (Gcc v: 4.4.2)                                     *
//*  under Fedora Release 12, Kernel Linux 2.6.31.5-127.fc12.i686.PAE          *
//******************************************************************************
//* Version History (most recent first):                                       *
//*   See version history in FileDlg.cpp                                       *
//******************************************************************************
//* Programmer's Notes:                                                        *
//*                                                                            *
//******************************************************************************

//******************************************************************************
//* Class definition for object used to pass the batch-rename pattern among    *
//* the methods used for renaming files.                                       *
//******************************************************************************
class renamePattern
{
   public:
   //* Constructor *
   renamePattern ()
   {
      this->reinit () ;
   }

   //* Create a formatted filename string from the current class data values   *
   //*                                                                         *
   //* Input  :   trgPattern: buffer to which the new string will be written   *
   //*          existingName: pointer to existing filename string              *
   //*          resetCounter: (optional, false by default)                     *
   //*                        if true, restart the sequence counter at the     *
   //*                        value specified by seqStart                      *
   //*                                                                         *
   //* Returns: number of columns needed to display the string                 *
   //*                                                                         *
   short formatFName ( gString& trgPattern, const char* existingName, bool resetCounter = false )
   {
      wchar_t  eName[MAX_FNAME+1],        // base filename
               eExt[MAX_FNAME+1],         // filename extension
               sBuff[MAX_FNAME+1] ;       // prepend/append text

      //* If caller wants to restart the sequence counter *
      if ( resetCounter != false )
      {
         this->seqNum = this->seqStart ;
      }

      //* Parse the existing filename.                                         *
      //* If existing filename has an extension, extIndex will index it.       *
      //* Else, extIndex == -1 indicating that there is no extension           *
      //* Note that for 'hidden' files, the first character is a PERIOD, so in *
      //* that case, if the file does not have an extension, extIndex == ZERO, *
      //* but it does not count as an extension delimiter.                     *
      wchar_t  exName[MAX_FNAME] ;
      gString  gsName( existingName ) ;
      gsName.copy( exName, MAX_FNAME ) ;
      short extIndex = -1, i ;
      for ( i = ZERO ; exName[i] != NULLCHAR ; i++ )
      {
         eName[i] = exName[i] ;
         if ( exName[i] == PERIOD )
            extIndex = i ;
      }
      if ( extIndex > ZERO )  // filename extension found
      {
         eName[extIndex] = NULLCHAR ;  // terminate base-name string
         for ( i = ZERO ; exName[extIndex] != NULLCHAR ; i++, extIndex++ )
            eExt[i] = exName[extIndex] ;
          eExt[i] = NULLCHAR ;
      }
      else                    // no filename extension
      {
         eName[i] = NULLCHAR ;         // terminate base-name string
         *eExt = NULLCHAR ;            // empty string for extension
      }

      //* If new filename includes a sequence number, create the string *
      if ( this->base && this->seqDigits > ZERO )
      {  //* Create the sequence string specification, *
         //* then use it to create the sequence string *
         const char seqLen = 32 ;
         wchar_t seqSpec[seqLen] ;
         swprintf ( seqSpec, seqLen, L"%%0%dd", this->seqDigits ) ;
         swprintf ( sBuff, (seqLen/2+1), seqSpec, this->seqNum ) ;
         ++this->seqNum ;           // increment the sequence counter
      }
      else
         *sBuff = NULLCHAR ;

      const wchar_t *prs   = (this->pr != false) ? this->prStr : L"",
                    *res   = (this->base != false) ? this->reStr : eName,
                    *ins   = (this->in != false) ? this->inStr : L"",
                    *exs   = (this->ext != false) ? this->reStr : eExt,
                    *aps   = (this->ap != false) ? this->apStr : L"" ;
      trgPattern.compose( L"%S%S%S%S%S%S", prs, res, ins, sBuff, exs, aps ) ;
      trgPattern.limitChars( MAX_FNAME ) ;   // be safe

      //* Return display columns required to display full string *
      return trgPattern.gscols() ;
   }

   //* Reinitialize all data members *
   void reinit ( void )
   {
      *prStr = *apStr = *inStr = *reStr = wchar_t(NULLCHAR) ;
      seqStart = seqNum = ZERO ;
      seqDigits = 1 ;
      pr = ap = in = base = ext = valid = false ;
   }

   //* Set the initial value for the sequence counter *
   void setSequence ( short firstSeqNumber )
   {
      this->seqStart = firstSeqNumber ;
      this->restartSequence () ;
   }

   //* Restart sequence counter with current seqStart value *
   void restartSequence ( void )
   {
      this->seqNum = this->seqStart ;
   }

   //* Public data for this class *
   wchar_t prStr[MAX_FNAME+1] ;  // string literal to prepend
   wchar_t apStr[MAX_FNAME+1] ;  // string literal to append
   wchar_t inStr[MAX_FNAME+1] ;  // string literal to insert
   wchar_t reStr[MAX_FNAME+1] ;  // string literal to replace base name or extension
   short seqStart ;  // if base != false, first number in sequence
   short seqDigits ; // if base != false, number of display digits in sequence
   bool  pr ;        // if 'true' prepend literal in prStr to filename
   bool  ap ;        // if 'true' append literal in aPendStr to filename
   bool  in ;        // if 'true' insert literal in iPendStr before filename extension (if any)
   bool  base ;      // if 'true' replace base filename with literal in rBaseStr
   bool  ext ;       // if 'true' replace filename extension with literal in rExtStr
   bool  valid ;     // if 'true' instance contains valid rename data

   private:
   short seqNum ;    // next sequence number to be incorporated into filename
                     // number advances whenever a sequence number is included 
                     // in a formatted filename.
} ;


