//********************************************************************************
//* File       : gsmeter.cpp                                                     *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2024-2025 Mahlon R. Smith, The Software Samurai   *
//*                 GNU GPL copyright notice located in gString.hpp              *
//* Date       : 11-May-2025                                                     *
//* Version    : (see appVersion string below)                                   *
//*                                                                              *
//* This is a simple test program to verify that the gString module builds       *
//* properly, and that all public methods are functional.                        *
//*                                                                              *
//* ============================================================================ *
//* Important Note: The conditional-compile declaration: DEBUG_GSTRING must be   *
//* set to non-zero to enable reporting of gString class internal configuration. *
//* ============================================================================ *
//*                                                                              *
//* Build using:  G++ / Gcc 13.2 or higher (GNU++11 support), for Linux          *
//*                                                                              *
//* Run using  : ./gsmeter OPTION (see help menu below)                          *
//********************************************************************************
//* Version History (most recent first):                                         *
//*                                                                              *
//* v: 0.0.06 11-Feb-2025                                                        *
//*   - Include 'gsmeter' with the gString distribution package.                 *
//*     - Located at: NcDialog/install/gString_install                           *
//*     - Update the Perl script for creating the distribution archive.          *
//*     - Update the Makefile included with the stand-alone distribution to      *
//*       optionally build the 'gsmeter' app.                                    *
//*                                                                              *
//* v: 0.0.05 04-Feb-2025                                                        *
//*   - After three months of exhaustive testing, not only with this application,*
//*     but also with the author's AnsiCmd library and BubbleSort demonstration  *
//*     app, the bString class appears to be stable. (but keep an eye on it).    *
//*     Begin to re-integrate the bString functionality into the mainline        *
//*     gString class.                                                           *
//*     - The first step is to remove all instances of the static gString class  *
//*       from this application. Approximately 30 static gString instances were  *
//*       used for the initial testing.                                          *
//*       - Archive the functional 'bsmeter' application.                        *
//*       - Change name of 'bsmeter.cpp' to 'gsmeter.cpp'.                       *
//*       - Update the makefile to remove the combined bString/gString library   *
//*         and replace it with the bString-only library.                        *
//*       - Remove the backward-compatibility tests. These tests use an          *
//*         instance of the gString class as a parameter for the bString test.   *
//*       - Replace each instance of the static gString with a bString object    *
//*         and test the update for compatibility.                               *
//*   - Add a sequence of tests for capture of potential buffer overflow.        *
//*   - Add a sequence of tests for the new gString 'compcoll' method.           *
//*   - Report the active locale along with version number.                      *
//*   - Modify all 'bString' instances to 'gString' to match the name change     *
//*     of the class.                                                            *
//*                                                                              *
//* v: 0.0.04 18-Nov-2024                                                        *
//*   - Transfer gString definitions to bString. Create conditional compile      *
//*     declaration: BACKWARD_COMPATIBILITY in bString.hpp which allows this     *
//*     application to perform all tests.                                        *
//*     This declaration should be set to zero(0) when using bString in any      *
//*     application which does not ALREADY include the static gString class.     *
//*                                                                              *
//* v: 0.0.03 30-Oct-2024                                                        *
//*   - All tested bString functionality has been verified to perform as         *
//*     expected. This _does not_ mean that the bString class is bug free, but   *
//*     it's a step in that direction.                                           *
//*                                                                              *
//* v: 0.0.02 15-Oct-2024                                                        *
//*   - Clean up the mess created by the early effort.                           *
//*   - Add a simple menu structure for ease in accessing test groups.           *
//*   - Completed statistics tests.                                              *
//*   - Completed analysis tests.                                                *
//*   - Completed output tests.                                                  *
//*   - Make bString aware of external gString objects for assignment,           *
//*     comparison, etc.                                                         *
//*   - Design and implement dynamic re-allocation of storage space to accomodate*
//*     increase in data size. (decreases in data size are ignored)              *
//*   - Completed reallocation tests.                                            *
//*     - Note that allocations that attempt to exceed the maximum allocation,   *
//*       and the data truncation which may occur when at the upper limit of     *
//*       buffer size have not be exhaustively tested. The code IS WRITTEN to    *
//*       work properly, but the size of the data needed for definitive testing  *
//*       is too difficult to manage efficiently. We may revisit this later.     *
//*   - Clean up temporary code and include additional explanation of tests.     *
//*                                                                              *
//* v: 0.0.01 04-Oct-2024                                                        *
//*   - Adapt framework from gs_test application which is used to test the       *
//*     build for the gString class.                                             *
//*   - Completed constructor tests.                                             *
//*   - Completed initializer tests.                                             *
//********************************************************************************
//* To Do:                                                                       *
//* ------                                                                       *
//* --                                                                           *
//*                                                                              *
//*                                                                              *
//********************************************************************************

#include <iostream>        //* Standard I/O definitions
#include <fstream>         //* Standard file I/O definitions
#include "gString.hpp"     //* gString class definition

using namespace std ;                  //* Scope quailfier

const char* const appVersion = "0.0.06" ; // application version
const char* const crYears = "2024-2025" ; // copyright years
const char* const dbFile = "gsmdbg.txt" ; // output filename
static int32_t ZERO = 0 ;                       // useful constant

const wchar_t* Chuck = 
L"Despite his manly name, Chuck was seldom asked to join the games of the "
 "other boys. First, he was rather small for his age and had the upper body "
 "strength of a 5-year-old, even though he was nearly nine. The continual "
 "asthma attacks and the prepubescent OCD contributed something as well.——"
 "Yes, Chuck was a nerd, but he had a burning desire to express "
 "himself and an overwhelming need to create something that would "
 "be of lasting value to someone, though his peers would probably "
 "still steal his lunch, and he was sure that his family would "
 "just continue to stare at him in slack-jawed incomprehension. " ;
//* Boolean specify: Note: This assumes that 'true' and 'false' *
//* have canonical values.                                      *
const char* bSpec[] =
{
   "false",
   "true"
} ;
const char*    enableDbgMsg  =  "Error! - Please enable DEBUG_GSTRING in gString.hpp" ;
const wchar_t* enableDbgMsgW = L"Error! - Please enable DEBUG_GSTRING in gString.hpp" ;

locale *ioLocale = NULL ;                       // application locale
static bool getLocale ( gString& gsloc ) ;      // report current locale

// Test methods for the gString object
bool gsmTest ( wchar_t maintest, wchar_t subtest, wchar_t auxtest ) ;
void gsmTestConstructorsA ( wchar_t wchar, std::ofstream& ofs ) ;
void gsmTestConstructorsB ( wchar_t wchar, std::ofstream& ofs ) ;
void gsmTestInitializersA ( wchar_t wchar, std::ofstream& ofs ) ;
void gsmTestInitializersB ( wchar_t wchar, std::ofstream& ofs ) ;
void gsmTestStatistics    ( wchar_t wchar, std::ofstream& ofs ) ;
void gsmTestModifiersA    ( wchar_t wchar, std::ofstream& ofs ) ;
void gsmTestModifiersB    ( wchar_t wchar, std::ofstream& ofs ) ;
void gsmTestModifiersC    ( wchar_t wchar, std::ofstream& ofs ) ;
void gsmTestModifiersD    ( wchar_t wchar, std::ofstream& ofs ) ;
void gsmTestAnalysisA     ( wchar_t wchar, std::ofstream& ofs ) ;
void gsmTestAnalysisB     ( wchar_t wchar, std::ofstream& ofs ) ;
void gsmTestAnalysisC     ( wchar_t wchar, std::ofstream& ofs ) ;
void gsmTestAnalysisD     ( wchar_t wchar, std::ofstream& ofs ) ;
void gsmTestAnalysisE     ( wchar_t wchar, std::ofstream& ofs ) ;
void gsmTestAnalysisF     ( wchar_t wchar, std::ofstream& ofs ) ;
void gsmTestOutputs       ( wchar_t wchar, std::ofstream& ofs ) ;
void dumpGString ( const char* title, const gString& bs, 
                   ofstream& ofs, bool tostdout = false ) ;
void find_report ( gString& gs, const gString& bs2, int32_t coff, int32_t csen, int32_t max, bool lng ) ;
void find_report ( gString& gs, const gString& bs1, const gString& bs2, int32_t fi, int32_t max, std::ofstream& ofs ) ;
void findlast_report ( const gString& bs1, const gString& bs2, int32_t fi, bool csen, bool lng, ofstream& ofs, bool ischar = false ) ; 
void after_report ( gString& gs, const gString& bs2, int32_t coff, bool csen, bool lng = false, bool ischar = false ) ;
void after_report ( gString& gs, const gString& bs1, const gString& bs2, int32_t fi, std::ofstream& ofs ) ;
void findr_report ( gString& gs, const gString& bs2, int32_t coff, bool csen, bool lng = false, bool ischar = false ) ;
void findr_report ( gString& gs, const gString& bs1, const gString& bs2, int32_t fi, std::ofstream& ofs ) ;
void replace_report ( const gString& bssrc, const gString& bsnew, const gString& bs, 
                      int32_t coff, bool csen, bool rall, ofstream& ofs, 
                      bool s_lng = false, bool n_lng = false, bool s_chr = false, bool n_chr = false ) ;
void gsmBufferOverflowA ( wchar_t wchar, std::ofstream& ofs ) ;
void gsmBufferOverflowB ( wchar_t wchar, std::ofstream& ofs ) ;
void gsmTestReallocationA ( wchar_t verbose, std::ofstream& ofs ) ;
void gsmTestReallocationB ( wchar_t verbose, std::ofstream& ofs ) ;
void Sandbox ( wchar_t testchar, std::ofstream& ofs ) ;
int  Title ( bool clear ) ;
void setLocale ( const char * localeName = NULL ) ; // set locale from environment


//*************************
//*         main          *
//*************************
//********************************************************************************
//* Test instantiation and basic functionality of gString class.                 *
//********************************************************************************
int main ( int argc, char* argv[], char* argenv[] )
{
   wchar_t maintest = L'\0',
           subtest = L'\0',
           auxtest = L'\0' ;
   bool    validTest = false,
           version   = false ;

   if ( argc > 1 )
   {
      short indx = ZERO ;
      if ( (maintest = argv[1][indx]) == L'-' )
         maintest = argv[1][++indx] ;
      maintest = tolower( maintest ) ;
      if ( argv[1][++indx] != L'\0' )
      { subtest = argv[1][indx] ; subtest = tolower( subtest ) ; }
      if ( argv[1][++indx] != L'\0' )
         auxtest = argv[1][indx] ;

      switch ( maintest )
      {
         case L'c':     // constructors
         case L'i':     // initializers
         case L's':     // statistics
         case L'm':     // modifiers
         case L'a':     // analysis
         case L'o':     // output
         case L'b':     // buffer-overflow tests
         case L'r':     // re-allocation tests
         case L'p':     // playground (sandbox)
            validTest = true ;
            break ;
         case L'-':     // version (or help) request
            if ( subtest == L'v' )
               version = true ;
            break ;
         case L'h':     // display quick help
         default:
            break ;
      }
   }

   setLocale () ;                         // set locale from environment

   if ( validTest )
   {
      Title ( true ) ;

      //** Execute the specified test(s) **
      validTest = gsmTest ( maintest, subtest, auxtest ) ;

      wcout << "\nbye-bye!\n" << endl ;
   }

   else if ( version )
   {
      static const char* const freeSoftware = 
      "License GPLv3+: GNU GPL version 3 <http://gnu.org/licenses/gpl.html>\n"
      "This is free software: you are free to modify and/or redistribute it\n"
      "under the terms set out in the license.\n"
      "There is NO WARRANTY, to the extent permitted by law.\n" ;
      gString gs, gsloc ;
      int titWidth = Title ( false ) ;
#if 1    // CZONE - If multiple entries in locale string. See AnsiCmd::acGetLocale().
         //       - Create a non-member method to parse the string.
      getLocale ( gsloc ) ;
      gs.compose( "current locale: %S\n", gsloc.gstr() ) ;
#else    // FUNCTIONAL FOR SINGLE ENTRY
      const char* locPtr = (ioLocale != NULL ? ioLocale->name().c_str() : "unknown") ;
      gs.compose( "current locale: %s\n", locPtr ) ;
#endif   // FUNCTIONAL FOR SINGLE ENTRY
      gs.padCols( (gs.gscols() + titWidth), L'-' ) ;
      wcout << gs << L"\n" << freeSoftware << endl ;

      validTest = true ;                  // disable help display
   }

   if ( ! validTest )
   {
      Title ( false ) ;
      wcout << L"Test Suite for gString Class  Usage: gsmeter TEST [SUBTEST]\n"
                "   C[a|b] --------- Constructors  :  S ------- Statistics   \n"
                "   I[a|b] --------- Initializers  :  O ------- Output group \n"
                "   M[a|b|c|d]------ Modifiers     :  B[a|b]--- Buffer size  \n"
                "   A[a|b|c|d|e|f]-- Analyzers     :  R[a|b] -- Reallocation \n"
                "   --help  --version              :  P ------- Playground \n" << endl ;
   }

   if ( ioLocale != NULL )             // release the dynamic allocation
      delete ioLocale ;

   return (0) ;

}  //* End main() *

//********************************************************************************
//*                                                                              *
//*                                                                              *
//* Input : gsloc : (by reference) receives adjusted locale string               *
//*                                                                              *
//* Returns: 'true' if successful                                                *
//********************************************************************************
static bool getLocale ( gString& gsloc )
{
   int i ;                                // character index
   bool status = false ;                  // return value

   gsloc = ioLocale->name().c_str() ;     // get the raw locale string

   if ( (i = gsloc.after( L'=' )) > ZERO )// remove (initial) tag name
   { gsloc.shiftChars( -(i) ) ; status = true ; }
   if ( (i = gsloc.find( L';' )) > ZERO ) // if multiple locale tags
      gsloc.limitChars( i ) ;             // remove all except the first value

   return status ;

}  //* End getLocale()

//********************************************************************************
//* Test the gString class functionality.                                        *
//*                                                                              *
//* Input : mainchar == primary test group,                                      *
//*         subtest == specific test                                             *
//*         auxtest == argument for some tests                                   *
//********************************************************************************
bool gsmTest ( wchar_t maintest, wchar_t subtest, wchar_t auxtest )
{
   gString gsOut ;                           // messages
   bool status = true ;                      // return value

   ofstream ofs ( dbFile, ofstream::out | ofstream::trunc ) ;
   if ( ofs.is_open() )                      // if output file open
   {
      ofs << "gsmeter test results\n--------------------" << endl ;

      //* gString constructor group *
      if ( maintest == L'c' )
      {
         //* Basic set of constructors *
         if ( subtest == L'a' )
         {
            gsOut = "gString Constructors, Group A:" ;
            wcout << L" " << gsOut << endl ;
            ofs << gsOut << endl ;
            gsmTestConstructorsA ( L'a', ofs ) ;
         }
         //* Integer format-specification constructors *
         else if ( subtest == L'b' )
         {
            wcout << L" gString Constructors, Group B:" << endl ;
            ofs   <<  "gString Constructors, Group B:" << endl ;
            gsmTestConstructorsB ( L'f', ofs ) ;
         }
         else
         {
            wcout 
               << L"A: Constructors A, default, specify allocation, char*, \n"
                   "   wchar_t*, format specification\n"
                   "B: Constructors B: integer-formatting constructors"
               << endl ;
         }
      }  // constructors

      //* Initializers and general reporting *
      else if ( maintest == L'i' )
      {
         if ( subtest == L'a' )
         {
            wcout << L" Initializers, Group A, and Basic Stats:" << endl ;
            ofs   <<  "Initializers, Group A, and Basic Stats:" << endl ;
            gsmTestInitializersA ( L'a', ofs ) ;
         }
         else if ( subtest == L'b' )
         {
            wcout << L" Initializers, Group B:" << endl ;
            ofs   <<  "Initializers, Group B:" << endl ;
            gsmTestInitializersB ( L'm', ofs ) ;
         }
         else
         {
            wcout 
               << L"A: Initializers A, and basic reporting methods: \n"
                   "   wAlloc(), uAlloc(), gstr(), ustr(), gschars(), uftbytes(), gscols(), \n"
                   "   isASCII(), operator=, loadChars(), formatInt()\n"
                   "B: Initializers B: compose(format, ...), compose(binary extension)"
               << endl ;
         }
      }  // initializers

      //* Statistical Reporting Methods *
      else if ( maintest == L's' )
      {
         wcout << L" Statistical Reporting Methods:" << endl ;
         ofs   <<  "Statistical Reporting Methods:" << endl ;
         gsmTestStatistics ( L'a', ofs ) ;
      }  // statistics methods

      //* Text Modification Methods *
      else if ( maintest == L'm' )
      {
         gsOut = L"Text Modification Methods, Group " ;
         switch ( subtest )
         {
            case 'a':
               gsOut.append( L"A:" ) ;
               wcout << L" " << gsOut << endl ;
               ofs   << gsOut << endl ;
               gsmTestModifiersA ( L'a', ofs ) ;
               break ;
            case 'b':
               gsOut.append( L"B:" ) ;
               wcout << L" " << gsOut << endl ;
               ofs   << gsOut << endl ;
               gsmTestModifiersB ( L'f', ofs ) ;
               break ;
            case 'c':
               gsOut.append( L"C:" ) ;
               wcout << L" " << gsOut << endl ;
               ofs   << gsOut << endl ;
               gsmTestModifiersC ( L'i', ofs ) ;
               break ;
            case 'd':
               gsOut.append( L"D:" ) ;
               wcout << L" " << gsOut << endl ;
               ofs   << gsOut << endl ;
               gsmTestModifiersD ( L'a', ofs ) ;
               break ;
            default:
               wcout << L"A: append(), limitChars(), limitCols(), insert(), erase()\n"
                         "B: replace(), shiftChars(), shiftCols()\n"
                         "C: rightJustify(), padCols(), strip(), textReverse()\n"
                         "D: formatParagraph(), clear()"
                     << endl ;
               break ;
         }
      }  // 'm' for modifications

      //* Analysis Methods *
      else if ( maintest == L'a' )
      {
         gsOut = L"Data Analysis Methods, Group " ;
         switch ( subtest )
         {
            case 'a':
               gsOut.append( L"A:" ) ;
               wcout << L" " << gsOut << endl ;
               ofs   << gsOut << endl ;
               gsmTestAnalysisA ( L'a', ofs ) ;
               break ;
            case 'b':
               gsOut.append( L"B:" ) ;
               wcout << L" " << gsOut << endl ;
               ofs   << gsOut << endl ;
               gsmTestAnalysisB ( L'i', ofs ) ;
               break ;
            case 'c':
               gsOut.append( L"C:" ) ;
               wcout << L" " << gsOut << endl ;
               ofs   << gsOut << endl ;
               gsmTestAnalysisC ( L'p', ofs ) ;
               break ;
            case 'd':
               gsOut.append( L"D:" ) ;
               wcout << L" " << gsOut << endl ;
               ofs   << gsOut << endl ;
               gsmTestAnalysisD ( L'a', ofs ) ;
               break ;
            case 'e':
               gsOut.append( L"E:" ) ;
               wcout << L" " << gsOut << endl ;
               ofs   << gsOut << endl ;
               gsmTestAnalysisE ( L'f', ofs ) ;
               break ;
            case 'f':
               gsOut.append( L"F:" ) ;
               wcout << L" " << gsOut << endl ;
               ofs   << gsOut << endl ;
               gsmTestAnalysisF ( L'l', ofs ) ;
               break ;
            default:
               wcout << L"A: operator==, operator!=, compare(gString&), compare()\n"
                         "B: compcoll(), find()\n"
                         "C: findlast(), findx(), scan()\n"
                         "D: after()\n"
                         "E: findr()\n"
                         "F: gscanf()"
                     << endl ;
               break ;
         }
      }  // analysis methods

      //* Output Methods *
      else if ( maintest == L'o' )
      {
         gsOut = "gString Output Methods:" ;
         wcout << L" " << gsOut << endl ;
         ofs << gsOut << endl ;
         gsmTestOutputs ( L'a', ofs ) ;
      }  // output methods

      //* Buffer-overflow tests - identify and reallocate *
      else if ( maintest == L'b' )
      {
         gsOut = L"Buffer Overflow-prevention, Group " ;
         if ( subtest == L'a' )
         {
            gsOut.append( L"A:" ) ;
            wcout << L" " << gsOut << endl ;
            ofs   << gsOut << endl ;
            gsmBufferOverflowA ( L'a', ofs ) ;
         }
         else if ( subtest == L'b' )
         {
            gsOut.append( L"B:" ) ;
            wcout << L" " << gsOut << endl ;
            ofs   << gsOut << endl ;
            gsmBufferOverflowB ( L'k', ofs ) ;
         }
         else
         {
            wcout << L"A: Constructors and Assignment Overflow tests\n"
                      "B: Insert and Append Overflow tests"
                  << endl ;
         }
      }

      //* Automatic reallocation of storage space *
      else if ( maintest == L'r' )
      {
         gsOut = L"Reallocation Methods, Group " ;
         if ( subtest == L'a' )
         {
            gsOut.append( L"A:" ) ;
            wcout << L" " << gsOut << endl ;
            ofs   << gsOut << endl ;
            gsmTestReallocationA ( auxtest, ofs ) ;
         }
         else if ( subtest == L'b' )
         {
            gsOut.append( L"B:" ) ;
            wcout << L" " << gsOut << endl ;
            ofs   << gsOut << endl ;
            gsmTestReallocationB ( auxtest, ofs ) ;
         }
         else
         {
            wcout << L"A: Reallocation tests, Group A\n"
                      "B: Reallocation tests, Group B"
                  << endl ;
         }
      }  // reallocation

      //* Sandbox *
      else if ( maintest == L'p' )
      {
         Sandbox ( subtest, ofs ) ;
      }  // sandbox

      ofs.close() ;                       // close the debugging file
   }
   return status ;

}  //* End gsmTest() *

//********************************************************************************
//* Test the gString basic constructor group.                                    *
//*                                                                              *
//********************************************************************************
void gsmTestConstructorsA ( wchar_t wchar, std::ofstream& ofs )
{
   gString gs, gsdbg ;
   int32_t gsw ;

   //* Default constructor *
   gString gscon ;
   gsw = gscon.wAlloc() ;
   gs.compose( "  %C) default constructor, alloc:%04d", &wchar, &gsw ) ;
   wcout << gs << endl ;

   dumpGString ( "dflt", gscon, ofs, false ) ;

   //* Allocation constructor (no text data specified) *
   ++wchar ;
   gString gscap( gsALLOCMED * 2 ) ;
   gsw = gscap.wAlloc() ;
   gs.compose( "  %C) allocation constructor, alloc:%04d", &wchar, &gsw ) ;
   wcout << gs << endl ;

   dumpGString ( "gscap", gscap, ofs, false ) ;

   //* UTF-8 data && default allocation *
   ++wchar ;
   gs = L"你好世界！ Hello World!" ;
   gString bsutf8( gs.ustr() ) ;
   gsw = bsutf8.wAlloc() ;
   gs.compose( "  %C) UTF-8  source data constructor, default alloc:%04d\n" 
               "     '%S'", &wchar, &gsw, bsutf8.gstr() ) ;
   wcout << gs << endl ;

   dumpGString ( "cchars", bsutf8, ofs, false ) ;

   //* UTF-32 data && default allocation *
   ++wchar ;
   gString bsutf32( L"你好世界！ Hello World!" ) ;
   gsw = bsutf32.wAlloc() ;
   gs.compose( "  %C) UTF-32 source data constructor, default alloc:%04d\n"
               "     '%S'", &wchar, &gsw, bsutf32.gstr() ) ;
   wcout << gs << endl ;
   dumpGString ( "wchars", bsutf32, ofs, false ) ;

   //* Format-specification Constructor *
   ++wchar ;
   short sval1 = 14, sval2 = 16, sval3 = 18, sval4 = 20 ;
   int   ival = 22 ;
   long long int llval = 2468 ;
   double dval = 3.14159 ;
   const char* const str1  = "What?" ;
   const char* const str1lj = "(l-j)" ;
   const char* const str1rj = "(r-j)" ;
   const wchar_t* const strL   = L"L What?" ;
   const wchar_t* const strLlj = L"(L l-j)" ;
   const wchar_t* const strLrj = L"(L r-j)" ;
   wchar_t wch1 = L'X' ;
   char ch1 = 'x' ;
   gString bsfmt( "%hd %% %hd %hd %hd %Xh %lld %lf %C %c \"%s\",\"%-7s\",\"%7s\",\"%S\",\"%-9S\",\"%9S\"", 
                  &sval1, &sval2, &sval3, &sval4, &ival, &llval, 
                  &dval, &wch1, &ch1, str1, str1lj, str1rj, strL, strLlj, strLrj ) ;
   gsw = bsfmt.wAlloc() ;
   gs.compose( "  %C) format-specification constructor, default alloc:%04d\n"
               "     '%S'", &wchar, &gsw, bsfmt.gstr() ) ;
   wcout << gs << endl ;
   dumpGString ( "format", bsfmt, ofs, false ) ;

}  //* End gsmTestConstructorsA() *

//********************************************************************************
//* Test the gString formatted-integer constructor group.                        *
//*                                                                              *
//********************************************************************************
void gsmTestConstructorsB ( wchar_t wchar, std::ofstream& ofs )
{
   gString gs, gsdbg ;

   short fshort = 2450 ;            // signed short
   gString bsshort( fshort, 5 ) ;
   gs.compose( "  %C) signed short format 5 cols: '%S'", &wchar, bsshort.gstr() ) ;
   wcout << gs << endl ;
   dumpGString ( "sshort", bsshort, ofs, false ) ;

   ++wchar ;
   unsigned short ufshort = 2455 ;  // unsigned short
   gString bsushort( ufshort, 6 ) ;
   gs.compose( "  %C) unsigned short format 6 cols: '%S'", &wchar, bsushort.gstr() ) ;
   wcout << gs << endl ;
   dumpGString ( "ushort", bsushort, ofs, false ) ;

   ++wchar ;
   int sival = 12456 ;              // signed int
   gString bssi( sival, 4 ) ;
   gs.compose( "  %C) signed int format 4 cols: '%S'", &wchar, bssi.gstr() ) ;
   wcout << gs << endl ;
   dumpGString ( "sint", bssi, ofs, false ) ;

   ++wchar ;
   unsigned int uival = 12456789 ;  // unsigned int
   gString bsui( uival, 8, true, true, true, fiKiB ) ;
   gs.compose( "  %C) unsigned int format 8 cols, signed, left-just, Kibi: '%S'", &wchar, bsui.gstr() ) ;
   wcout << gs << endl ;
   dumpGString ( "uint", bsui, ofs, false ) ;

   ++wchar ;
   long int slval = 28467932 ;      // signed long int
   gString bssli( slval, 11, true, true, false, fiKb ) ;
   gs.compose( "  %C) signed long int format 11 cols, Kilo: '%S'", &wchar, bssli.gstr() ) ;
   wcout << gs << endl ;
   dumpGString ( "lint", bssli, ofs, false ) ;

   ++wchar ;
   unsigned long int ulval = 28467932 ;   // unsigned long int
   gString bsuli( ulval, 8, false, false, false, fiKb ) ;
   gs.compose( "  %C) unsigned long int format 8 cols, Kilo: '%S'", &wchar, bsuli.gstr() ) ;
   wcout << gs << endl ;
   dumpGString ( "ulint", bsuli, ofs, false ) ;

   ++wchar ;
   long long int sllval = -9999999999994 ; // signed long long int
   gString bslli( sllval, (FI_MAX_8BYTE_WIDTH-1), false, false, false, fiKiB ) ;
   gs.compose( "  %C) signed long long int format max cols-1 Kibi: '%S'", &wchar, bslli.gstr() ) ;
   wcout << gs << endl ;
   dumpGString ( "sllint", bslli, ofs, false ) ;

   ++wchar ;
   unsigned long long int ullval = 9999999999994 ;
   gString bulli( ullval, (FI_MAX_8BYTE_WIDTH-2), false, false, false, fiKiB ) ;
   gs.compose( "  %C) unsigned long long int format max cols-2 Kibi: '%S'", &wchar, bulli.gstr() ) ;
   wcout << gs << endl ;
   dumpGString ( "ullint", bulli, ofs, false ) ;

}  //* End gsmTestConstructorsB() *

//********************************************************************************
//* Test the gString data initializers, group A (plus basic reporting).          *
//* wAlloc(), uAlloc(), gstr(), ustr(), gschars(), uftbytes(), gscols(),         *
//* isASCII(), operator=, loadChars(), compose(format), formatInt()              *
//********************************************************************************

void gsmTestInitializersA ( wchar_t wchar, std::ofstream& ofs )
{
   gString gs, gsdbg ;

   gString bs( "const-pointer to stored data." ) ;
   int32_t w = bs.wAlloc(), u = bs.uAlloc() ;
   const wchar_t* ws = bs.gstr() ;     // operator= (wchar_t*)
   const char*    us = bs.ustr() ;     // operator= (char*)
   gs.compose( "  %C) wAlloc: %hd(%04X) gstr:'%S'\n"
               "     uAlloc:%hd(%04X) ustr:'%s'",
               &wchar, &w, &w, ws, &u, &u, us ) ;
   wcout << gs << endl ;
   dumpGString ( "dflt", bs, ofs, false ) ;

   ++wchar ;
   bs = L"gstr(int32_t&) reports wchar_t* and char count." ;
   ws = bs.gstr( w ) ;
   gs.compose( "  %C) '%S' (%02d chars)", &wchar, ws, &w ) ;
   wcout << gs << endl ;
   ofs << "report: " << &gs.ustr()[5] << endl ;

   ++wchar ;
   bs = L"ustr(int32_t&, int32_t&) reports char*, char count and byte count." ;
   us = bs.ustr( w, u ) ;
   gs.compose( "  %C) '%s' (%02d chars, %02d bytes)", &wchar, us, &w, &u ) ;
   wcout << gs << endl ;
   ofs << "report: " << &gs.ustr()[5] << "\n" << endl ;

   ++wchar ;
   bs = L"A sentence with some multi-column text:一些多列字符。 Reported by:\n"
         "      gschars() utfbytes() gscols() gscols(int32_t&)  isASCII()." ;
   w = bs.gschars() ;
   u = bs.utfbytes() ;
   int32_t colsA = bs.gscols(),
           chrs ;
   const short* colsPtr = bs.gscols( chrs ) ;
   bool  ascii = bs.isASCII() ;
   gs.compose( "  %C) '%S' \n"
               "        %03d        %03d       %03d    p:%p(%03d)    %hhd", 
               &wchar, bs.gstr(), &w, &u, &colsA, colsPtr, &chrs, &ascii ) ;
   wcout << gs << endl ;

   ++wchar ;
   bs = L"operator=  assign wchar_t string data." ;
   gs.compose( "  %C) '%S'", &wchar, bs.gstr() ) ;
   wcout << gs << endl ;
   dumpGString ( "wassgn", bs, ofs, false ) ;

   ++wchar ;
   bs = L"operator=  report wchar_t* and char count." ;
   ws = bs.gstr( w ) ;
   gs.compose( "  %C) '%S' (%02d)", &wchar, ws, &w ) ;
   wcout << gs << endl ;
   dumpGString ( "wassgn", bs, ofs, false ) ;

   ++wchar ;
   bs = (uint8_t*)("operator=  assign unsigned char data.") ;
   gs.compose( "  %C) '%S'", &wchar, bs.gstr() ) ;
   wcout << gs << endl ;
   dumpGString ( "uassgn", bs, ofs, false ) ;

   ++wchar ;
   bs = "operator=  assign char string data." ;
   gs.compose( "  %C) '%S'", &wchar, bs.gstr() ) ;
   wcout << gs << endl ;
   dumpGString ( "cassgn", bs, ofs, false ) ;

   ++wchar ;
   bs = "operator=  report char*, char count and byte count（字数和字节数）." ;
   us = bs.ustr( w, u ) ;
   gs.compose( "  %C) '%s' (%02d && %02d)", &wchar, us, &w, &u ) ;
   wcout << gs << endl ;
   dumpGString ( "cassgn", bs, ofs, false ) ;

   ++wchar ;
   gString bssrc( gsALLOCMAX ) ;
   w = bs.wAlloc() ;    // old wchar_t storage
   bssrc = "operator=  assign gString to gString (allocation adj.)" ;
   bs = bssrc ;
   u = bs.wAlloc() ;    // new wchar_storage
   gs.compose( "  %C) '%S' wAlloc:%Xh->%Xh", &wchar, bs.gstr(), &w, &u ) ;
   wcout << gs << endl ;
   dumpGString ( "bssrc", bssrc, ofs, false ) ;
   dumpGString ( "bstrg", bs, ofs, false ) ;

   //* loadChars() group *
   ++wchar ;
   const wchar_t* wCapture = L"-->captured wchar_t substring<--" ;
   const char*    cCapture =  "-->captured char substring<--" ;
   const uint8_t* uCapture =  (uint8_t*)("-->_captured unsigned char substring<--") ;
   w = 26 ;
   u = 23 ;
   gs.compose( "  %C) loadChars: wcharlimit:%02d ccharlimit:%02d ucharlimit:33", &wchar, &w, &u ) ;
   wcout << gs << endl ;
   ofs << &gs.ustr()[5] << endl ;
   bs.loadChars( &wCapture[3], w ) ;
   ofs << "        '" << bs.ustr() << "'" << endl ;
   wcout << L"     '" << bs << L"'" << endl ;
   bs.loadChars( &wCapture[3], w, true ) ;
   ofs << "        '" << bs.ustr() << "'" << endl ;
   wcout << L"     '" << bs << L"'" << endl ;
   bs.loadChars( &cCapture[3], u ) ;
   ofs << "        '" << bs.ustr() << "'" << endl ;
   wcout << L"     '" << bs << L"'" << endl ;
   bs.loadChars( &cCapture[3], u, true ) ;
   ofs << "        '" << bs.ustr() << "'" << endl ;
   wcout << L"     '" << bs << L"'" << endl ;
   bs.loadChars( &uCapture[3], 33, true ) ;
   ofs << "        '" << bs.ustr() << "'\n" << endl ;
   wcout << L"     '" << bs << L"'" << endl ;

   //** formatInt **
   const wchar_t* fSpec[] =   // synchronized with enum fiUnits
   {
      L"fiK",
      L"fik",
      L"fiKb",
      L"fikB",
      L"fiKiB"
   } ;

   ++wchar ;
   gs.compose( "  %C) formatInt ( intVal, width, left-justify, signed, kibi(base2), units )", &wchar ) ;
   wcout << gs << endl ;

   short fshort = 2450 ;                  // signed short int
   int32_t fwid = 5 ;
   fiUnits units = fiK ;
   bool lJust = false, sign = false, kibi = false ;
   bs.formatInt( fshort, fwid, lJust, sign, kibi ) ;
   gs.compose( "               short int, formatInt ( %hd, %d, %s, %s, %s, '%S' )      --> '%S'", 
               &fshort, &fwid, bSpec[lJust], bSpec[sign], bSpec[kibi], fSpec[units], 
               bs.gstr() ) ;
   wcout << gs << endl ;
   dumpGString ( "sshort", bs, ofs, false ) ;


   unsigned short ufshort = 2455 ;        // unsigned short int
   fwid = 6 ; lJust = false ; sign = false ; kibi = false ; units = fiK ;
   bs.formatInt( ufshort, fwid, lJust, sign, kibi ) ;
   gs.compose( "      unsigned short int, formatInt ( %hu, %d, %s, %s, %s, '%S' )      --> '%S'", 
               &ufshort, &fwid, bSpec[lJust], bSpec[sign], bSpec[kibi], fSpec[units], 
               bs.gstr() ) ;
   wcout << gs << endl ;
   dumpGString ( "ushort", bs, ofs, false ) ;

   int sival = 12456 ;                    // signed int
   fwid = 4 ; lJust = false ; sign = false ; kibi = false ; units = fiK ;
   bs.formatInt( sival, fwid, lJust, sign, kibi ) ;
   gs.compose( "                     int, formatInt ( %d, %d, %s, %s, %s, '%S' )     --> '%S'", 
               &sival, &fwid, bSpec[lJust], bSpec[sign], bSpec[kibi], fSpec[units], 
               bs.gstr() ) ;
   wcout << gs << endl ;
   dumpGString ( "sint", bs, ofs, false ) ;

   unsigned int uival = 12456789 ;                 // unsigned int
   fwid = 8 ; lJust = true ; sign = true ; kibi = true ; units = fiKiB ;
   bs.formatInt( uival, fwid, lJust, sign, kibi, units ) ;
   gs.compose( "            unsigned int, formatInt ( %u, %d, %s, %s, %s, '%S' )   --> '%S'", 
               &uival, &fwid, bSpec[lJust], bSpec[sign], bSpec[kibi], fSpec[units], bs.gstr() ) ;
   wcout << gs << endl ;
   dumpGString ( "uint", bs, ofs, false ) ;

   long int slval = 28467932 ;                     // signed long int
   fwid = 11 ; lJust = true ; sign = true ; kibi = false ; units = fiKb ;
   bs.formatInt( slval, fwid, lJust, sign, kibi, units ) ;
   gs.compose( "                long int, formatInt ( %ld, %d, %s, %s, %s, '%S' )  --> '%S'", 
               &slval, &fwid, bSpec[lJust], bSpec[sign], bSpec[kibi], fSpec[units], bs.gstr() ) ;
   wcout << gs << endl ;
   dumpGString ( "lint", bs, ofs, false ) ;

   unsigned long int ulval = 28467932 ;            // unsigned long int
   fwid = 8 ; lJust = false ; sign = false ; kibi = false ; units = fiKb ;
   bs.formatInt( ulval, fwid, lJust, sign, kibi, units ) ;
   gs.compose( "       unsigned long int, formatInt ( %lu, %d, %s, %s, %s, '%S' ) --> '%S'", 
               &ulval, &fwid, bSpec[lJust], bSpec[sign], bSpec[kibi], fSpec[units], bs.gstr() ) ;
   wcout << gs << endl ;
   dumpGString ( "ulint", bs, ofs, false ) ;

   long long int sllval = -9999999999994 ;         // signed long long int
   fwid = (FI_MAX_8BYTE_WIDTH-1) ; lJust = false ; sign = false ; kibi = false ; units = fiKiB ;
   bs.formatInt( sllval, fwid, lJust, sign, kibi, units ) ;
   gs.compose( "           long long int, formatInt ( %lld, %d, %s, %s, %s, '%S' ) --> '%S'", 
               &sllval, &fwid, bSpec[lJust], bSpec[sign], bSpec[kibi], fSpec[units], bs.gstr() ) ;
   wcout << gs << endl ;
   dumpGString ( "sllint", bs, ofs, false ) ;

   unsigned long long int ullval = 9999999999994 ; // unsigned long long int
   fwid = (FI_MAX_8BYTE_WIDTH-2) ; lJust = false ; sign = false ; kibi = false ; units = fiKiB ;
   bs.formatInt( ullval, (FI_MAX_8BYTE_WIDTH-2), false, false, false, fiKiB ) ;
   gs.compose( "       uns.long long int, formatInt (  %llu, %d, %s, %s, %s, '%S' ) --> '%S'", 
               &ullval, &fwid, bSpec[lJust], bSpec[sign], bSpec[kibi], fSpec[units], bs.gstr() ) ;
   wcout << gs << endl ;
   dumpGString ( "ullint", bs, ofs, false ) ;

   //* ANSI escape sequence scrolls upward one row *
   wcout << L"\x1B[1F" ; wcout.flush() ;

}  //* End gsmTestInitializersA() *

//********************************************************************************
//* Test the gString data initializers, group B.                                 *
//* compose ( numeric data formatted for binary display )                        *
//********************************************************************************
void gsmTestInitializersB ( wchar_t wchar, std::ofstream& ofs )
{
   gString gs ;
   gString bs( gsALLOCMIN ) ;
   uint32_t gschars, gscols ;

   //* Compose source data *
   short sv1 = 14, sv2 = 16, sv3 = 18, sv4 = 20 ;
   int   iv1 = 22 ;
   long long int llval = 2468 ;
   double dval = 3.14159 ;
   const char* const str1  = "What?" ;
   const char* const str1lj = "(l-just)" ;
   const char* const str1rj = "(r-just)" ;
   const wchar_t* const strL   = L"L What?" ;
   const wchar_t* const strLlj = L"(L l-just)" ;
   const wchar_t* const strLrj = L"(L r-just)" ;
   wchar_t wch1 = L'X' ;
   char ch1 = 'x' ;

   //** compose( const char* ) **
   gs.compose( "  %C) compose: const char* formatting spec.", &wchar ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   bs.compose( "%hd %% %hd %hd %hd %Xh %lld %lf %C %c \"%s\",\"%-10s\",\"%10s\",\"%S\",\"%-12S\",\"%12S\"", 
               &sv1, &sv2, &sv3, &sv4, &iv1, &llval, 
               &dval, &wch1, &ch1, str1, str1lj, str1rj, strL, strLlj, strLrj ) ;
   dumpGString ( "chfmt", bs, ofs, false ) ;

   bs.clear() ;         // reset all data members

   //** compose( const wchar_t* ) **
   ++wchar ;
   gs.compose( "  %C) compose: const wchar_t* formatting spec.", &wchar ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   bs.compose( L"%hd %% %hd %hd %hd %Xh %lld %lf %C %c \"%s\",\"%-10s\",\"%10s\",\"%S\",\"%-12S\",\"%12S\"", 
               &sv1, &sv2, &sv3, &sv4, &iv1, &llval, 
               &dval, &wch1, &ch1, str1, str1lj, str1rj, strL, strLlj, strLrj ) ;
   dumpGString ( "wchfmt", bs, ofs, false ) ;

   //** compose( composeFieldwidthBugFix ) **
   ++wchar ;
   const char* insectTemplate = "%02d chars with %02d columns: |%S|" ;
   const wchar_t* insect = L"踩到一只昆虫" ;
   bs = insect ;
   int32_t insectChars = bs.gschars() - 1, insectCols = bs.gscols() ;
   gs.compose( "  %C) compose: field-width bug fix. The width of a text field s/b columns, not characters. (naughty swprintf!)\n"
               "              source data : ", &wchar ) ;
   gs.append( insectTemplate, &insectChars, &insectCols, insect ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   int32_t specCols = 20 ;
   bs.compose( "%20S", insect ) ;
   gschars = bs.gschars() - 1 ;
   gscols  = bs.gscols() ;
   gs.compose( "          Specify width of %02d cols (right justified): |%S| formatted to %02d columns using %02d characters", 
               &specCols, bs.gstr(), &gscols, &gschars ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   bs.compose( "%-20S", insect ) ;
   gschars = bs.gschars() - 1 ;
   gscols  = bs.gscols() ;
   gs.compose( "          Specify width of %02d cols (left justified) : |%S| formatted to %02d columns using %02d characters", 
               &specCols, bs.gstr(), &gscols, &gschars ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   //** compose ( binary display ) **
   ++wchar ;
   int8_t   bval1 = 0x5C,   bval2   = 0xA5 ;
   int16_t  sval1 = 0x7C7C, sval2   = 0x3A3A ;
   int32_t  ival  = 0x01020304 ;
   int64_t  qval  = 0xFEDCBA9876543210 ;
   gs.compose( "  %C) compose method: binary formatting of integer data.\n"
               "     ___________4-bit grouping____________       ___________8-bit grouping___________", &wchar ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   bs.compose( L"      int8_t (%%hhb)   %hhb                   int8_t (%%.8hhb)  %.8hhb", 
               &bval1, &bval2 ) ;
   wcout << bs << endl ;
   ofs   << bs << endl ;

   bs.compose( L"     int16_t (%%hb)    %hb        int16_t (%%.8hb)   %.8hb", 
               &sval1, &sval2 ) ;
   wcout << bs << endl ;
   ofs   << bs << endl ;

   bs.compose( L"     int16_t (%% hb)   % hb        int16_t (%%-.8hb)  %-.8hb", 
               &sval1, &sval2 ) ;
   wcout << bs << endl ;
   ofs   << bs << endl ;

   bs.compose( L"     int16_t (%%#-hb) %#-hb        int16_t (%%-#.8hb) %-#.8hb", 
               &sval1, &sval2 ) ;
   wcout << bs << endl ;
   ofs   << bs << endl ;

   sval1 = 0xA35A ; sval2 = 0xA031 ;
   bs.compose( L"     int16_t (%%#-hB) %#-hB        int16_t (%%-#.8hB) %-#.8hB\n", 
               &sval1, &sval2 ) ;
   wcout << bs << endl ;
   ofs   << bs << endl ;

   bs.compose( L"     int32_t (%%b)       %b\n"
                "     int32_t (%%.8b)     %.8b", &ival, &ival ) ;
   wcout << bs << endl ;
   ofs   << bs << endl ;

   bs.compose( L"     int32_t (%%# b)    %# b\n"
                "     int32_t (%%-#_.8b)  %-#_.8b", &ival, &ival ) ;
   wcout << bs << endl ;
   ofs   << bs << endl ;

   bs.compose( L"     int64_t (%%#.4qB)  %#.4qB\n"
                "     int64_t (%%-# .8qb) %-# .8qb", &qval, &qval ) ;
   wcout << bs << endl ;
   ofs   << bs << endl ;

}  //* End gsmTestInitializersB() *

//********************************************************************************
//* Test the gString statistics methods.                                         *
//* gstr(), ustr(), wAlloc(), uAlloc(), freeSpace(), gschars(), utfbytes(),      *
//* gscols(), isASCII(), Get_gString_Version() and dumpGstring()                 *
//********************************************************************************
void gsmTestStatistics ( wchar_t wchar, std::ofstream& ofs )
{
   gString gs, gsdbg ;
   const wchar_t* gstr ;
   const char*    ustr ;
   const short*   cwid ;
   int32_t gsw, gsu, gsfree, gscols, gschars, utfbytes ;
   bool isascii ;

   gString bs( "Are the Ursidae likely to deposit their 污物 in the primordial weald?" ) ;
   gs.compose( "  %C) Source text: '%S'", &wchar, bs.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   #if DEBUG_GSTRING != 0
   bs.dumpGstring ( gstr, ustr, cwid, gsw, gsu, gschars, gscols, utfbytes, isascii ) ;
   #else    // dummy report
   gstr = enableDbgMsgW ;
   ustr = enableDbgMsg ;
   cwid = NULL ;
   gsw = gsu = gschars = gscols = utfbytes = ZERO ;
   isascii = false ;
   #endif   // dummy report
   gsdbg.compose( "basic : gstr:%p gsw:%04Xh(%6d) '%S' \n"
                  "        ustr:%p gsu:%04Xh(%6d) '%s'\n"
                  "        cwid:%p gschars:%02d gscols:%02d utfbytes:%02d isascii:%hhd\n",
                  gstr, &gsw, &gsw, gstr, ustr, &gsu, &gsu, ustr, cwid,
                  &gschars, &gscols, &utfbytes, &isascii
                ) ;
   gsdbg.replace( L"basic :", L"     " ) ;
   gsdbg.replace( L"  ustr:", L"ustr:" ) ;
   gsdbg.replace( L"  cwid:", L"cwid:" ) ;
   gsdbg.insert( L"     Info Dump (for debugging only):\n" ) ;
   wcout << gsdbg ; wcout.flush() ;
   ofs   << gsdbg ; ofs.flush() ;

   ++wchar ;
   gstr     = bs.gstr() ;        // pointer to wchar_t data
   ustr     = bs.ustr() ;        // pointer to char data
   gsw      = bs.wAlloc() ;      // get wchar_t buffer allocation
   gsu      = bs.uAlloc() ;      // get char buffer allocation
   gsfree   = bs.freeSpace() ;   // get free space
   gschars  = bs.gschars() ;     // get count of stored characters
   utfbytes = bs.utfbytes() ;    // get count of UTF-8 bytes
   gscols   = bs.gscols() ;      // get count of display columns
   isascii  = bs.isASCII() ;     // get ASCII flag
   gs.compose( "  %C) Public Statistical Information\n"
               "      UTF-32 allocation : %-5d   UTF-32 data: '%S'\n"
               "      UTF-8  allocation : %-5d   UTF-8  data: '%s'\n"
               "      UTF-32 characters : %d\n"
               "      utf-8 bytes       : %d\n"
               "      display columns   : %d\n"
               "      ASCII data flag   : %hhd\n"
               "      free space (chars): %d\n", &wchar, 
               &gsw, gstr, &gsu, ustr, &gschars, &utfbytes, &gscols, &isascii, &gsfree ) ;
   wcout << gs ; wcout.flush() ;
   ofs   << gs ; ofs.flush() ;

   ++wchar ;
   gstr = bs.gstr( gschars ) ;   // pointer to wchar_t data
   gs.compose( "  %C) wchar_t data with char count:%d\n"
               "     '%S'", &wchar, &gschars, gstr ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   ++wchar ;
   ustr = bs.ustr( gschars, utfbytes ) ;   // pointer to char data
   gs.compose( "  %C) utf-8 data with char count:%d and byte count:%d\n"
               "      '%s'", &wchar, &gschars, &utfbytes, ustr ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   ++wchar ;
   cwid = bs.gscols( gscols ) ;  // array of column widths
   gs.compose( "  %C) array of character widths:%d\n"
               "      ", &wchar, &gscols ) ;
   for ( int32_t i = ZERO ; i < gschars ; ++i )
   {
      gs.append( "%hd ", &cwid[i] ) ;
      if ( i == (gschars / 2) )
         gs.append( L"\n      " ) ;
   }
   wcout << gs << endl ;
   ofs   << gs << endl ;

   //** Version number **
   ++wchar ;
   gs.compose( "  %C) gString version: '%s'", &wchar, bs.Get_gString_Version() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

}  //* End gsmTestStatistics() *

//********************************************************************************
//* Test the gString text modification methods, group A.                         *
//* append() limitChars(), limitCols(), insert(), erase()                        *
//********************************************************************************
void gsmTestModifiersA ( wchar_t wchar, std::ofstream& ofs )
{
   gString gs, gsdbg ;
   int32_t indx ;

   //** append **
   gString bs( "Does a bear poop in the woods? " ) ;
   gs.compose( "  %C) append: initial data: '%S'", &wchar, bs.gstr() ) ;
   wcout << gs << endl ;
   dumpGString ( "plain", bs, ofs, false ) ;

   bs.append( L"Yes, " ) ;
   gs.compose( "            '%S'", bs.gstr() ) ;
   wcout << gs << endl ;
   dumpGString ( "addwch", bs, ofs, false ) ;

   bs.append( "Occasionally. " ) ;
   gs.compose( "            '%S'", bs.gstr() ) ;
   wcout << gs << endl ;
   dumpGString ( "addch", bs, ofs, false ) ;

   short daynum = 13 ;
   bs.append( L'(' ) ;
   bs.append( "%S %s %hdth.", L"But", "never on the", &daynum ) ;
   bs.append( ')' ) ;
   gs.compose( "            '%S'", bs.gstr() ) ;
   wcout << gs << endl ;
   dumpGString ( "addfmt", bs, ofs, false ) ;

   //** limitChars **
   ++wchar ;
   bs.limitChars( 30 ) ;
   gs.compose( "  %C) limitChars: '%S'", &wchar, bs.gstr() ) ;
   wcout << gs << endl ;
   dumpGString ( "lim ch", bs, ofs, false ) ;

   //** limitCols **
   ++wchar ;
   bs = "Mixed single-column 单列数据and multi-column 多列数据data." ;
   dumpGString ( "limcol", bs, ofs, false ) ;

   int32_t c = bs.gscols() ;
   int32_t w = bs.gschars() ;
   int32_t csub = 27 ;
   gs.compose( "  %C) limitCols: '%S'  (%02d columns)", &wchar, bs.gstr(), &c ) ;
   wcout << gs << endl ;
   bs.limitCols( bs.gscols() - csub ) ;
   c = bs.gscols() ;
   w -= bs.gschars() ;
   gs.compose( "      minus %02d: '%S'  (%02d columns,%2d full chars)", &csub, bs.gstr(), &c, &w ) ;
   wcout << gs << endl ;
   dumpGString ( "limcol", bs, ofs, false ) ;

   csub = 5 ;
   w = bs.gschars() ;
   bs.limitCols( bs.gscols() - csub ) ;
   c = bs.gscols() ;
   w -= bs.gschars() ;
   gs.compose( "      minus %02d: '%S'       (%02d columns,%2d full chars)", &csub, bs.gstr(), &c, &w ) ;
   wcout << gs << endl ;
   dumpGString ( "limcol", bs, ofs, false ) ;

   csub = 1 ;
   bs.limitCols( bs.gscols() - csub ) ;
   c = bs.gscols() ;
   gs.compose( "      minus %02d: '%S'         (%02d columns, partial char)", &csub, bs.gstr(), &c ) ;
   wcout << gs << endl ;
   dumpGString ( "limcol", bs, ofs, false ) ;

   //** insert **
   ++wchar ;
   bs = "BASE_DATA" ;
   dumpGString ( "insert", bs, ofs, false ) ;

   gs.compose( "  %C) insert: '%S'", &wchar, bs.gstr() ) ;
   wcout << gs << endl ;
   bs.insert( "cCharData " ) ;                  // at head-of-text (char data)
   gs.compose( "             '%S'", bs.gstr() ) ;
   wcout << gs << endl ;
   ofs << &gs.ustr()[5] << endl ;
 
   bs.insert( " cCharData", bs.gschars() ) ;    // append to tail (char data)
   gs.compose( "             '%S'", bs.gstr() ) ;
   wcout << gs << endl ;
   ofs << &gs.ustr()[5] << endl ;

   indx = bs.find( L'_' ) ;                     // insert within existing text (UTF-32)
   bs.insert( L" wCharData", indx ) ;
   gs.compose( "             '%S'", bs.gstr() ) ;
   wcout << gs << endl ;
   ofs << &gs.ustr()[5] << endl ;
   indx = bs.find( L'w' ) ;
   bs.insert( (uint8_t*)("uCharData "), indx ) ; // insert within existing text (UTF-8)
   gs.compose( "             '%S'", bs.gstr() ) ;
   wcout << gs << endl ;
   ofs << &gs.ustr()[5] << endl ;
   indx = ZERO ;
   bs.insert ( L'*', indx ) ;
   while ( (indx = bs.find( L' ', indx )) >= ZERO )
      bs.insert( L'*', ++indx ) ;
   bs.insert( L'*', 99 ) ; // (beyond text area)
   gs.compose( "             '%S'", bs.gstr() ) ;
   wcout << gs << endl ;
   ofs << &gs.ustr()[5] << endl ;
   dumpGString ( "insert", bs, ofs, false ) ;

   //** erase **
   ++wchar ;
   bs = "Try to ExtraErase oRange Xerase the orAnge extra smelly rotten oraNge data.oranGe " ;
   dumpGString ( "eraseA", bs, ofs, false ) ;

   gs.compose( "  %C) erase : '%S'", &wchar, bs.gstr() ) ;
   wcout << gs << endl ;
   bs.erase( L"orange ", 0, false, true ) ;
   gs.compose( "       rept: '%S'", bs.gstr() ) ;
   wcout << gs << endl ;
   ofs << &gs.ustr()[5] << endl ;
   bs.erase( L'X', 0, true ) ;
   gs.compose( "       sing: '%S'", bs.gstr() ) ;
   wcout << gs << endl ;
   ofs << &gs.ustr()[5] << endl ;
   bs.erase( "Extra", 2, true ) ;
   gs.compose( "       cstr: '%S'", bs.gstr() ) ;
   wcout << gs << endl ;
   ofs << &gs.ustr()[5] << endl ;
   bs.erase( (uint8_t*)("smelly ") ) ;
   gs.compose( "       ustr: '%S'", bs.gstr() ) ;
   wcout << gs << endl ;
   ofs << &gs.ustr()[5] << endl ;
   gString bse( "rotten " ) ;
   bs.erase( bse ) ;
   gs.compose( "       bstr: '%S'", bs.gstr() ) ;
   wcout << gs << endl ;
   ofs << &gs.ustr()[5] << endl ;
   gs = "erase " ;
   bs.erase( gs, 0, true ) ;
   gs.compose( "       gstr: '%S'", bs.gstr() ) ;
   wcout << gs << endl ;
   ofs << &gs.ustr()[5] << endl ;
   bs.erase( 0, 7 ) ;
   gs.compose( "     substr: '%S'", bs.gstr() ) ;
   wcout << gs << endl ;
   //ofs << &gs.ustr()[5] << endl ;
   dumpGString ( "eraseB", bs, ofs, false ) ;

}  //* End gsmTestModifiersA() *

//********************************************************************************
//* Test the gString text modification methods, group B.                         *
//* replace(), shiftChars(), shiftCols()                                         *
//********************************************************************************
void gsmTestModifiersB ( wchar_t wchar, std::ofstream& ofs )
{
   gString gs, gsdbg ;
   gString bssrc, bsnew ;
   int32_t coff ;
   bool csen, rall ;

   gString bs( L"Ply_thot_frAnkly_awful_Muzak_while_blitzed._\x2014_Wilt_Cherry_1976" ) ;

   gs.compose( "  %C) replace ( srctxt, newtext, offset, casesen, all )\n"
               "     '%S'", &wchar, bs.gstr() ) ;
   wcout << gs << endl ;

   bssrc = "ply" ;
   bsnew = "Play" ;
   coff = ZERO ; csen = false ; rall = false ;
   bs.replace( bssrc.gstr(), bsnew.gstr(), coff, csen, rall ) ;
   replace_report ( bssrc, bsnew, bs, coff, csen, rall, ofs, true, true ) ;

   bssrc = "thot" ;
   bsnew = "that" ;
   coff = ZERO ; csen = true ; rall = false ;
   bs.replace( bssrc.gstr(), bsnew.ustr(), coff, csen, rall ) ;
   replace_report ( bssrc, bsnew, bs, coff, csen, rall, ofs, true, false ) ;

   bssrc = "A" ;
   bsnew = "a" ;
   coff = ZERO ; csen = true ; rall = false ;
   bs.replace( bssrc.gstr(), *bsnew.gstr(), coff, csen, rall ) ;
   replace_report ( bssrc, bsnew, bs, coff, csen, rall, ofs, true, true, false, true ) ;

   bssrc = "frankly_awful" ;
   bsnew = "funky" ;
   coff = 5 ; csen = true ; rall = false ;
   bs.replace( bssrc.ustr(), bsnew.gstr(), coff, csen, rall ) ;
   replace_report ( bssrc, bsnew, bs, coff, csen, rall, ofs, false, true ) ;

   bssrc = "muzak" ;
   bsnew = "music" ;
   coff = ZERO ; csen = false ; rall = false ;
   bs.replace( bssrc.ustr(), bsnew.ustr(), coff, csen, rall ) ;
   replace_report ( bssrc, bsnew, bs, coff, csen, rall, ofs ) ;

   bssrc = "l" ;
   bsnew = "t" ;
   coff = 5 ; csen = true ; rall = false ;
   bs.replace( bssrc.ustr(), *bsnew.gstr(), coff, csen, rall ) ;
   replace_report ( bssrc, bsnew, bs, coff, csen, rall, ofs, false, true, false, true ) ;

   bssrc = "t" ;
   bsnew = "d" ;
   coff = bs.find( L'.' ) ; csen = true ; rall = false ;
   bs.replace( *bssrc.gstr(), bsnew.ustr(), coff, csen, rall ) ;
   replace_report ( bssrc, bsnew, bs, coff, csen, rall, ofs, true, false, true, false ) ;

   bssrc = "b" ;
   bsnew = "g" ;
   coff = ZERO ; csen = true ; rall = false ;
   bs.replace( *bssrc.gstr(), bsnew.gstr(), coff, csen, rall ) ;
   replace_report ( bssrc, bsnew, bs, coff, csen, rall, ofs, true, true, true, false ) ;

   bssrc = "_" ;
   bsnew = " " ;
   coff = ZERO ; csen = true ; rall = true ;
   bs.replace( *bssrc.gstr(), *bsnew.gstr(), coff, csen, rall ) ;
   replace_report ( bssrc, bsnew, bs, coff, csen, rall, ofs, true, true, true, true ) ;

   bssrc = "glitzed" ;
   bsnew = "boy" ;
   coff = ZERO ; csen = true ; rall = false ;
   bs.replace( bssrc.gstr(), bsnew.gstr(), coff, csen, rall ) ;
   replace_report ( bssrc, bsnew, bs, coff, csen, rall, ofs, true, true ) ;

   //** shiftChars **
   ++wchar ;
   bs = "Shift text characters left and right using default padding character." ;
   gs.compose( "  %C) shiftChars ( shiftCount, padChar )\n"
               "     \"%S\"",
               &wchar, bs.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   coff = 12 ;
   bs.shiftChars( coff ) ;
   gs.compose( "     shiftChars(%3d ) : '%S'", &coff, bs.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   coff = -8 ;
   bs.shiftChars( coff ) ;
   gs.compose( "     shiftChars(%3d ) : '%S'", &coff, bs.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   bs = "Shift text left and right using specified padding character." ;
   coff = 12 ;
   bs.shiftChars( coff, L'-' ) ;
   gs.compose( "     shiftChars(%3d ) : '%S'", &coff, bs.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   coff = -8 ;
   bs.shiftChars( coff, L'-' ) ;
   gs.compose( "     shiftChars(%3d ) : '%S'", &coff, bs.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   coff = -15 ;
   bs.shiftChars( coff, L'-' ) ;
   gs.compose( "     shiftChars(%3d ) : '%S'", &coff, bs.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   coff = 15 ;
   bs.shiftChars( coff, L'.' ) ;
   gs.compose( "     shiftChars(%3d ) : '%S'", &coff, bs.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   //** shiftCols **
   ++wchar ;
   bs = "Shift 这些是多列字符 (text columns) left and right using default padding character." ;
   gs.compose( "  %C) shiftCols ( shiftCount, padChar )\n"
               "     \"%S\"",
               &wchar, bs.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   coff = 12 ;
   bs.shiftCols( coff ) ;
   gs.compose( "     shiftCols(%3d ) : '%S'", &coff, bs.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   coff = -8 ;
   bs.shiftCols( coff ) ;
   gs.compose( "     shiftCols(%3d ) : '%S'", &coff, bs.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   bs = "Shift 这些是多列字符 (text columns) left and right using specified padding character." ;
   coff = 12 ;
   bs.shiftCols( coff, L'*' ) ;
   gs.compose( "     shiftCols(%3d ) : '%S'", &coff, bs.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   coff = -8 ;
   bs.shiftCols( coff, L'*' ) ;
   gs.compose( "     shiftCols(%3d ) : '%S'", &coff, bs.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   coff = -12 ;
   bs.shiftCols( coff, L'*' ) ;
   gs.compose( "     shiftCols(%3d ) : '%S'", &coff, bs.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   coff = 4 ;
   bs.shiftCols( coff, L'.' ) ;
   gs.compose( "     shiftCols(%3d ) : '%S'", &coff, bs.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   coff = -5 ;
   bs.shiftCols( coff, L'.' ) ;
   gs.compose( "     shiftCols(%3d ) : '%S'", &coff, bs.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

}  //* End gsmTestModifierB() *

//********************************************************************************
//* Test the gString text modification methods, group C.                         *
//* rightJustify(), padCols(), strip(), textReverse(), formatParagraph(), clear()*
//********************************************************************************
void gsmTestModifiersC ( wchar_t wchar, std::ofstream& ofs )
{
   gString gs, gsdbg ;
   int32_t fldWid = 45,
           txtWid,
           dispOff,
           indx ;
   wchar_t padChar = L' ' ;
   bool    ctr = false ;

   //** rightJustify **
   gString bs( "Right-justify text within a field." ) ;
   txtWid = bs.gscols() ;
   gs.compose( "  %C) rightJustify ( fieldWidth, padChar ) : ", &wchar ) ;
   dispOff = gs.gscols() ;
   gs.append( "'%S' (%d columns)", bs.gstr(), &txtWid ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   bs.rightJustify( fldWid, padChar ) ;
   gs.compose( "     rightJustify( %2d, L'%C' )", &fldWid, &padChar ) ;
   gs.padCols( dispOff ) ;
   gs.append( "|%S|", bs.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   bs.shiftCols( txtWid - fldWid ) ;
   padChar = L'>' ;
   bs.rightJustify( fldWid, padChar ) ;
   gs.compose( "     rightJustify( %2d, L'%C' )", &fldWid, &padChar ) ;
   gs.padCols( dispOff ) ;
   gs.append( "|%S|", bs.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   //** padCols **
   ++wchar ;
   const char* padText = "Pad text columns to specified width.(填充)" ;
   bs = padText ;
   txtWid = bs.gscols() ;
   fldWid = 52 ;
   padChar = L' ' ;
   gs.compose( "  %C) padCols ( fieldWidth, padChar, centered ) : ", &wchar ) ;
   dispOff = gs.gscols() ;
   gs.append( "'%S' (%d columns)", bs.gstr(), &txtWid ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   bs.padCols( fldWid, padChar, ctr ) ;
   gs.compose( "     padCols ( %2d, L'%C', %s )", &fldWid, &padChar, bSpec[ctr] ) ;
   gs.padCols( dispOff ) ;
   gs.append( "|%S|", bs.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   bs = padText ;
   padChar = L'>' ;
   bs.padCols( fldWid, padChar, ctr ) ;
   gs.compose( "     padCols ( %2d, L'%C', %s )", &fldWid, &padChar, bSpec[ctr] ) ;
   gs.padCols( dispOff ) ;
   gs.append( "|%S|", bs.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   bs = padText ;
   padChar = L'-' ;
   ctr = true ;
   bs.padCols( fldWid, padChar, ctr ) ;
   gs.compose( "     padCols ( %2d, L'%C', %s )", &fldWid, &padChar, bSpec[ctr] ) ;
   gs.padCols( dispOff ) ;
   gs.append( "|%S|", bs.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   //** strip **
   // Programmer's Note: The unmodified text cannot be displayed directly 
   // because it would trash the display, so we do a visual trick to 
   // represent the "unmodified" text
   ++wchar ;
   const wchar_t* stripText = L" \t \n \r \x0B \x3000 \x0C Leading and trailing "
                               "whitespace including: space, CJK space, \\n, \\r, "
                               "\\t, v-tab, formfeed. \t \n \r \x0B \x3000 \x0C " ;
   bs = stripText ;
   gsdbg = stripText ;
   indx = gsdbg.find( "Leading" ) ;
   gsdbg.shiftChars( -(indx) ) ;
   indx = gsdbg.after( L'.' ) ;
   gsdbg.limitChars ( indx ) ;
   gsdbg.shiftCols( 8 ) ;
   gsdbg.padCols( gsdbg.gscols() + 8 ) ;
   gs.compose( "  %C) strip ( leading, trailing ) : ", &wchar ) ;
   dispOff = gs.gscols() ;
   gs.append( "'%S'", gsdbg.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   bs.strip() ;
   gs.compose( "     strip ( true, true )" ) ;
   gs.padCols( dispOff ) ;
   gs.append( "|%S|", bs.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   //** textReverse **
  ++wchar ;
   const char* revText = "RTL Language text is often displayed incorrectly so reverse the text." ;
   const char* rtBrk   = "displayed " ;
   bs = revText ;
   bool punct = false, para = false, rjust = false ;
   gs.compose( "  %C) textReverse ( punctuation, paragraph, right-justify )", &wchar ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   gs.compose( "     textReverse ( %s, %s, %s )\n"
               "     '%S'\n", bSpec[punct], bSpec[para], bSpec[rjust], bs.gstr() ) ;
   bs.textReverse( punct, para, rjust ) ;
   gs.append( "     '%S'", bs.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   bs = revText ;
   punct = true, para = false, rjust = false ;
   gs.compose( "     textReverse ( %s, %s, %s )\n"
               "     '%S'\n", bSpec[punct], bSpec[para], bSpec[rjust], bs.gstr() ) ;
   bs.textReverse( punct, para, rjust ) ;
   gs.append( "     '%S'", bs.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   bs = revText ;
   indx = bs.after( rtBrk ) ;
   bs.insert( L'\n', indx ) ;
   punct = true, para = true, rjust = false ;
   gs.compose( "     textReverse ( %s, %s, %s )\n"
               "%S\n   ------\n", bSpec[punct], bSpec[para], bSpec[rjust], bs.gstr() ) ;
   bs.textReverse( punct, para, rjust ) ;
   gs.append( "%S", bs.gstr() ) ;
   indx = ZERO ;
   while ( (indx = gs.after( L'\n', indx )) >= ZERO )
      gs.insert( L"      |", indx ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   bs = revText ;
   indx = bs.after( rtBrk ) ;
   bs.insert( L'\n', indx ) ;
   punct = false, para = true, rjust = true ;
   gs.compose( "     textReverse ( %s, %s, %s )\n"
               "%S\n   ------\n", bSpec[punct], bSpec[para], bSpec[rjust], bs.gstr() ) ;
   bs.textReverse( punct, para, rjust ) ;
   gs.append( "%S", bs.gstr() ) ;
   indx = ZERO ;
   while ( (indx = gs.after( L'\n', indx )) >= ZERO )
      gs.insert( L"      |", indx ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

}  //* End gsmTestModifiersC() *

//********************************************************************************
//* Test the gString text modification methods, group D.                         *
//********************************************************************************
void gsmTestModifiersD ( wchar_t wchar, std::ofstream& ofs )
{
   gString gs, gsdbg ;
   int32_t rmax = 36, cmax = 96 ;
   bool    trnc = false, hyph = true ;

   //** formatParagraph **
   gString bs( Chuck ) ;
   gs.compose( "  %C) formatParagraph ( maxRows, maxCols, trunc, hyphenbrk, truncIndex )", &wchar ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   gs.compose( "formatParagraph ( %d, %d, %s, %s ):", 
               &rmax, &cmax, bSpec[trnc], bSpec[hyph] ) ;
   gs.padCols( (cmax - 1), L'>' ) ;
   gs.append( L'|' ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   bs.formatParagraph( rmax, cmax, trnc, hyph ) ;
   wcout << bs << L"\n   ------" << endl ;

   bs = Chuck ;
   rmax = 36, cmax = 78 ;
   trnc = false, hyph = true ;
   gs.compose( "formatParagraph ( %d, %d, %s, %s ):", 
               &rmax, &cmax, bSpec[trnc], bSpec[hyph] ) ;
   gs.padCols( (cmax - 1), L'>' ) ;
   gs.append( L'|' ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   bs.formatParagraph( rmax, cmax, trnc, hyph ) ;
   wcout << bs << L"\n" << endl ;

   //** clear **
   ++wchar ;
   bs = "Contents before calling clear()." ;
   gs.compose( "  %C) Clear gString contents to empty string:", &wchar ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   dumpGString ( "before", bs, ofs, true ) ;
   bs.clear() ;
   dumpGString ( "after", bs, ofs, true ) ;

}  //* End gsmTestModifiersD() *

//********************************************************************************
//* Test the gString Analysis methods, group A.                                  *
//********************************************************************************
void gsmTestAnalysisA ( wchar_t wchar, std::ofstream& ofs )
{
   gString gs, gssub ;

   const wchar_t* compEqual = L"'operator==' compares strings." ;
   const wchar_t* compNotEqual = L"'operator!=' compares strings." ;
   const wchar_t* same = L"identical" ;
   const wchar_t* diff = L"different" ;
   const wchar_t* Greater = L"is greater than" ;
   const wchar_t* EqualTo = L"is identical to" ;
   const wchar_t* Smaller = L" is less than  " ;

   gString bs1( compEqual ),
           bs2( compEqual ),
           bs3( "'operator==' compares insurance policies." ) ;

   wchar_t wsub[gsALLOCMIN] ;
   int32_t compVal, clen, coff ;
   bool    csen, result ;

   //** operator== (gString) **
   gs.compose( "  %C) gString1 == gString2 ?", &wchar ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   result = bool(bs1 == bs2) ;
   gs.compose( "      \"%S\" VS \"%S\" (%S)", 
               bs1.gstr(), bs2.gstr(), (result ? same : diff) ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   ++wchar ;
   gs.compose( "  %C) gString1 == gString3 ?", &wchar ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   result = bool(bs1 == bs3) ;
   gs.compose( "      \"%S\" VS \"%S\" (%S)", 
               bs1.gstr(), bs3.gstr(), (result ? same : diff) ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   //** operator!= (gString) **
   ++wchar ;
   bs1 = compNotEqual ;
   gs.compose( "  %C) gString1 != gString3 ?", &wchar ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   bs2 = compNotEqual ;
   bs3 = "'operator!=' compares Premier League strikers." ;
   result = bool(bs1 != bs3) ;
   gs.compose( "      \"%S\" VS \"%S\" (%S)", 
               bs1.gstr(), bs3.gstr(), (result ? diff : same) ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   ++wchar ;
   result = bool(bs1 != bs2) ;
   gs.compose( "  %C) gString1 != gString2 ?", &wchar ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   gs.compose( "      \"%S\" VS \"%S\" (%S)", 
               bs1.gstr(), bs2.gstr(), (result ? diff : same) ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   //** compare ( gString& ) **
   ++wchar ;
   bs1 = "There are 30 days in a month." ;
   bs2 = "There are 28 days in a month." ;
   bs3 = "There are 31 days in a month." ;
   gs.compose( "  %C) compare ( gString& )", &wchar ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   compVal = bs1.compare( bs2 ) ;
   gs.compose( "      \"%S\"  (%S)  \"%S\"", bs1.gstr(), 
               (compVal > ZERO ? Greater : compVal < ZERO ? Smaller : EqualTo), bs2.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   compVal = bs1.compare( bs1 ) ;
   gs.compose( "      \"%S\"  (%S)  \"%S\"", bs1.gstr(), 
               (compVal > ZERO ? Greater : compVal < ZERO ? Smaller : EqualTo), bs1.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   compVal = bs1.compare( bs3 ) ;
   gs.compose( "      \"%S\"  (%S)  \"%S\"", bs1.gstr(), 
               (compVal > ZERO ? Greater : compVal < ZERO ? Smaller : EqualTo), bs3.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   //** compare ( char* ) **
   ++wchar ;
   bs1 = "abcde fghij AbCdE" ;
   bs2 = "abcde" ;
   gssub = bs1.gstr() ;
   gs.compose( "  %C) compare ( char*, casesen, length, offset )  gString data:'%s'", 
               &wchar, bs1.ustr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   clen = 5 ; coff = ZERO ; csen = true ; 
   compVal = bs1.compare( bs2.ustr(), csen, clen, coff ) ;
   gssub.substr( wsub, coff, clen ) ; 
   gs.compose( "      compare( \"%s\", %s, %d,%2d ) gString \"%S\" %s \"%s\"", 
               bs2.ustr(), (csen ? " true" : "false"), &clen, &coff, wsub, 
               (compVal < ZERO ? " <" : compVal > ZERO ? "> " : "=="),
               bs2.ustr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   clen = 5 ; coff = 6 ; csen = true ; 
   compVal = bs1.compare( bs2.ustr(), csen, clen, coff ) ;
   gssub.substr( wsub, coff, clen ) ; 
   gs.compose( "      compare( \"%s\", %s, %d,%2d ) gString \"%S\" %s \"%s\"", 
               bs2.ustr(), (csen ? " true" : "false"), &clen, &coff, wsub, 
               (compVal < ZERO ? " <" : compVal > ZERO ? "> " : "=="),
               bs2.ustr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   clen = 5 ; coff = 12 ; csen = true ; 
   compVal = bs1.compare( bs2.ustr(), csen, clen, coff ) ;
   gssub.substr( wsub, coff, clen ) ; 
   gs.compose( "      compare( \"%s\", %s, %d,%2d ) gString \"%S\" %s \"%s\"", 
               bs2.ustr(), (csen ? " true" : "false"), &clen, &coff, wsub, 
               (compVal < ZERO ? " <" : compVal > ZERO ? "> " : "=="),
               bs2.ustr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   clen = 5 ; coff = 12 ; csen = false ; 
   compVal = bs1.compare( bs2.ustr(), csen, clen, coff ) ;
   gssub.substr( wsub, coff, clen ) ; 
   gs.compose( "      compare( \"%s\", %s, %d,%2d ) gString \"%S\" %s \"%s\"", 
               bs2.ustr(), (csen ? " true" : "false"), &clen, &coff, wsub, 
               (compVal < ZERO ? " <" : compVal > ZERO ? "> " : "=="),
               bs2.ustr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   //** compare ( wchar_t* ) **
   ++wchar ;
   gs.compose( "  %C) compare ( wchar_t*, casesen, length, offset )  gString data:'%S'", 
               &wchar, bs1.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   bs2 = "AbCdE" ;
   clen = 5 ; coff = ZERO ; csen = false ; 
   compVal = bs1.compare( bs2.gstr(), csen, clen, coff ) ;
   gssub.substr( wsub, coff, clen ) ; 
   gs.compose( "      compare( L\"%S\", %s, %d,%2d ) gString L\"%S\" %s L\"%S\"", 
               bs2.gstr(), (csen ? " true" : "false"), &clen, &coff, wsub, 
               (compVal < ZERO ? " <" : compVal > ZERO ? "> " : "=="),
               bs2.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   clen = 5 ; coff = 6 ; csen = true ; 
   compVal = bs1.compare( bs2.gstr(), csen, clen, coff ) ;
   gssub.substr( wsub, coff, clen ) ; 
   gs.compose( "      compare( L\"%S\", %s, %d,%2d ) gString L\"%S\" %s L\"%S\"", 
               bs2.gstr(), (csen ? " true" : "false"), &clen, &coff, wsub, 
               (compVal < ZERO ? " <" : compVal > ZERO ? "> " : "=="),
               bs2.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   clen = 5 ; coff = 12 ; csen = true ; 
   compVal = bs1.compare( bs2.gstr(), csen, clen, coff ) ;
   gssub.substr( wsub, coff, clen ) ; 
   gs.compose( "      compare( L\"%S\", %s, %d,%2d ) gString L\"%S\" %s L\"%S\"", 
               bs2.gstr(), (csen ? " true" : "false"), &clen, &coff, wsub, 
               (compVal < ZERO ? " <" : compVal > ZERO ? "> " : "=="),
               bs2.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   clen = 5 ; coff = 12 ; csen = true ;
   bs2 = "abcde" ;
   compVal = bs1.compare( bs2.gstr(), csen, clen, coff ) ;
   gssub.substr( wsub, coff, clen ) ; 
   gs.compose( "      compare( L\"%S\", %s, %d,%2d ) gString L\"%S\" %s L\"%S\"", 
               bs2.gstr(), (csen ? " true" : "false"), &clen, &coff, wsub, 
               (compVal < ZERO ? " <" : compVal > ZERO ? "> " : "=="),
               bs2.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   //** compare ( gString& ) **
   ++wchar ;
   bs1 = "I want 100 doughnuts." ;
   gs.compose( "  %C) compare ( gString&, casesen )  gString data:'%S'", 
               &wchar, bs1.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   bs2 = "I want 200 doughnuts." ;
   csen = true ; 
   compVal = bs1.compare( bs2, csen ) ;
   gs.compose( "      \"%S\"  %s  \"%S\"", bs1.gstr(), 
               (compVal < ZERO ? "<" : compVal > ZERO ? ">" : "=="), bs2.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

}  //* End gsmTestAnalysisA() *

//********************************************************************************
//* Test the gString Analysis methods, group B.                                  *
//********************************************************************************
void gsmTestAnalysisB ( wchar_t wchar, std::ofstream& ofs )
{
   const wchar_t* Greater = L"is greater than" ;
   const wchar_t* EqualTo = L"is identical to" ;
   const wchar_t* Smaller = L" is less than  " ;
   gString gs, gssub ;
   int32_t compVal, coff, fi, max ;
   bool    csen ;

   //** compcoll ( gString& ) **
   gString bs1( "Bears poop in the forest." ) ;
   gString bs2( bs1 ) ;
   gs.compose( "  %C) compcoll ( gString& )  gString data:'%s'", 
               &wchar, bs1.ustr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   compVal = bs1.compcoll( bs2 ) ;
   gs.compose( "      \"%S\"  (%S)  \"%S\"", bs1.gstr(), 
               (compVal > ZERO ? Greater : compVal < ZERO ? Smaller : EqualTo), bs2.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   bs2 = "Bears Poop in the forest." ;
   compVal = bs1.compcoll( bs2 ) ;
   gs.compose( "      \"%S\"  (%S)  \"%S\"", bs1.gstr(), 
               (compVal > ZERO ? Greater : compVal < ZERO ? Smaller : EqualTo), bs2.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   bs2.replace( "Poop", "crap" ) ;
   compVal = bs1.compcoll( bs2 ) ;
   gs.compose( "      \"%S\"  (%S)  \"%S\"", bs1.gstr(), 
               (compVal > ZERO ? Greater : compVal < ZERO ? Smaller : EqualTo), bs2.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   //** compcoll ( char* ) **
   bs1 = "abcde" ;
   bs2 = bs1 ;
   gs.compose( "  %C) compcoll ( char* )  gString data:'%s'", 
               &wchar, bs1.ustr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   compVal = bs1.compcoll( bs2.ustr() ) ;
   gs.compose( "      compcoll( \"%s\" ) gString \"%s\" %s \"%s\"", 
               bs2.ustr(), bs1.ustr(), 
               (compVal < ZERO ? " <" : compVal > ZERO ? "> " : "=="),
               bs2.ustr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   bs2 = "bcdef" ;
   compVal = bs1.compcoll( bs2.ustr() ) ;
   gs.compose( "      compcoll( \"%s\" ) gString \"%s\" %s \"%s\"", 
               bs2.ustr(), bs1.ustr(), 
               (compVal < ZERO ? " <" : compVal > ZERO ? "> " : "=="),
               bs2.ustr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   //** compcoll ( wchar_t* ) **
   ++wchar ;
   bs2 = "Abcde" ;
   gs.compose( "  %C) compcoll ( wchar_t* )  gString data:'%S'", 
               &wchar, bs1.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   compVal = bs1.compcoll( bs2.gstr() ) ;
   gs.compose( "      compcoll( L\"%S\" ) gString \"%S\" %s \"%S\"", 
               bs2.gstr(), bs1.gstr(), 
               (compVal < ZERO ? " <" : compVal > ZERO ? "> " : "=="),
               bs2.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   bs1 = bs2 ;
   bs2 = "aBcde" ;
   compVal = bs1.compcoll( bs2.gstr() ) ;
   gs.compose( "      compcoll( L\"%S\" ) gString \"%S\" %s \"%S\"", 
               bs2.gstr(), bs1.gstr(), 
               (compVal < ZERO ? " <" : compVal > ZERO ? "> " : "=="),
               bs2.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   //** find ( char* ) **
   ++wchar ;
   bs1 = "They say that the best things in life are free. "
         "That is true; they are purchased with sweat and anguish." ;
   gs.compose( "  %C) find ( char*, offset, casesen, maxcmp )  gString data:\n"
               "      '%S'", 
               &wchar, bs1.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   coff = ZERO ; csen = true ; max = -1 ;
   bs2 = "things" ;
   find_report ( gs, bs2, coff, csen, max, false ) ;
   fi = bs1.find( bs2.ustr(), coff, csen, max ) ;
   find_report ( gs, bs1, bs2, fi, max, ofs ) ;
//   #if 0    // This is a redundant test.
//   find_report ( gs, bs2, coff, csen, max, false ) ;
//   fi = bs1.find( (const uint8_t*)(bs2.ustr()), coff, csen, max ) ;
//   find_report ( gs, bs1, bs2, fi, max, ofs ) ;
//   #endif   // This is a redundant test.

   coff = ZERO ; csen = false ; max = -1 ;
   bs2 = "purchased" ;
   find_report ( gs, bs2, coff, csen, max, false ) ;
   fi = bs1.find( bs2.ustr(), coff, csen, max ) ;
   find_report ( gs, bs1, bs2, fi, max, ofs ) ;

   coff = fi ; csen = false ; max = 3 ;
   bs2 = "purfect!" ;
   find_report ( gs, bs2, coff, csen, max, false ) ;
   fi = bs1.find( bs2.ustr(), coff, csen, max ) ;
   find_report ( gs, bs1, bs2, fi, max, ofs ) ;

   coff = fi ; csen = false ; max = 4 ;
   find_report ( gs, bs2, coff, csen, max, false ) ;
   fi = bs1.find( bs2.ustr(), coff, csen, max ) ;
   find_report ( gs, bs1, bs2, fi, max, ofs ) ;

   //** find( wchar_t* ) **
   ++wchar ;
   gs.compose( "  %C) find ( wchar_t*, offset, casesen, maxcmp )", &wchar ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   coff = ZERO ; csen = false ; max = -1 ;
   bs2 = "That" ;
   find_report ( gs, bs2, coff, csen, max, true ) ;
   fi = bs1.find( bs2.gstr(), coff, csen, max ) ;
   find_report ( gs, bs1, bs2, fi, max, ofs ) ;

   coff = ZERO ; csen = true ; max = -1 ;
   find_report ( gs, bs2, coff, csen, max, true ) ;
   fi = bs1.find( bs2.gstr(), coff, csen, max ) ;
   find_report ( gs, bs1, bs2, fi, max, ofs ) ;

   coff = fi ; csen = true ; max = -1 ;
   bs2 = "free" ;
   find_report ( gs, bs2, coff, csen, max, true ) ;
   fi = bs1.find( bs2.gstr(), coff, csen, max ) ;
   find_report ( gs, bs1, bs2, fi, max, ofs ) ;

   /*coff = fi ;*/ csen = true ; max = -1 ;
   bs2 = "Eat" ;
   find_report ( gs, bs2, coff, csen, max, true ) ;
   fi = bs1.find( bs2.gstr(), coff, csen, max ) ;
   find_report ( gs, bs1, bs2, fi, max, ofs ) ;

   /*coff = fi ;*/ csen = false ; max = -1 ;
   find_report ( gs, bs2, coff, csen, max, true ) ;
   fi = bs1.find( bs2.gstr(), coff, csen, max ) ;
   find_report ( gs, bs1, bs2, fi, max, ofs ) ;

   //** find( wchar_t ) **
   ++wchar ;
   gs.compose( "  %C) find ( wchar_t, offset, casesen, maxcmp )", &wchar ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   coff = 15 ; csen = false ; max = -1 ;
   bs2 = "T" ;
   find_report ( gs, bs2, coff, csen, max, true ) ;
   gs.replace( L'"', L"'", 0, false, true ) ;
   fi = bs1.find( L'T', coff, csen, max ) ;
   find_report ( gs, bs1, bs2, fi, max, ofs ) ;

   coff = 15 ; csen = true ; max = -1 ;
   find_report ( gs, bs2, coff, csen, max, true ) ;
   gs.replace( L'"', L"'", 0, false, true ) ;
   fi = bs1.find( L'T', coff, csen, max ) ;
   find_report ( gs, bs1, bs2, fi, max, ofs ) ;

   //** find( gString& ) **
   ++wchar ;
   gs.compose( "  %C) find ( gString&, offset, casesen, maxcmp )", &wchar ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   coff = ZERO ; csen = true ; max = 5 ;
   bs2 = "chase" ;
   find_report ( gs, bs2, coff, csen, max, true ) ;
   fi = bs1.find( bs2, coff, csen, max ) ;
   find_report ( gs, bs1, bs2, fi, max, ofs ) ;

   coff = fi ; csen = false ; max = 5 ;
   bs2 = "Chasex" ;
   find_report ( gs, bs2, coff, csen, max, true ) ;
   fi = bs1.find( bs2, coff, csen, max ) ;
   find_report ( gs, bs1, bs2, fi, max, ofs ) ;

   coff = ZERO ; csen = true ; max = -1 ;
   gssub = "Anguish" ; bs2 = gssub ;
   find_report ( gs, bs2, coff, csen, max, true ) ;
   fi = bs1.find( gssub, coff, csen, max ) ;
   find_report ( gs, bs1, bs2, fi, max, ofs ) ;

   coff = 51 ; csen = false ; max = -1 ;
   gssub = "Anguish" ; bs2 = gssub ;
   find_report ( gs, bs2, coff, csen, max, true ) ;
   fi = bs1.find( gssub, coff, csen, max ) ;
   find_report ( gs, bs1, bs2, fi, max, ofs ) ;

//   //* ANSI escape sequence scrolls upward one row *
//   wcout << L"\x1B[1F" ; wcout.flush() ;

}  //* End gsmTestAnalysisB() *

//********************************************************************************
//* Test the gString Analysis methods, group C.                                  *
//********************************************************************************
void gsmTestAnalysisC ( wchar_t wchar, std::ofstream& ofs )
{
   gString gs, gssub ;
   int32_t off, fi ;
   bool    csen ;

   //** findlast ( char* ) **
   gString bs1( "Tornado slams into a barn. A tornado slams into a barn. Tornado picks up a Barnacle." ) ;
   gString bs2( "Tornado" ) ;
   gs.compose( "  %C) findlast ( char*, casesen )  gString data:\n"
               "     '%S'", &wchar, bs1.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   csen = true ;
   fi = bs1.findlast( bs2.ustr(), csen ) ;
   findlast_report ( bs1, bs2, fi, csen, false, ofs ) ;

   csen = true ;
   bs2 = "tornado" ;
   fi = bs1.findlast( bs2.ustr(), csen ) ;
   findlast_report ( bs1, bs2, fi, csen, false, ofs ) ;

   csen = false ;
   fi = bs1.findlast( bs2.ustr(), csen ) ;
   findlast_report ( bs1, bs2, fi, csen, false, ofs ) ;

   //** findlast ( uint8_t* ) **
   ++wchar ;
   gs.compose( "  %C) findlast ( uint8_t*, casesen )", &wchar ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   csen = false ;
   bs2 = "parks" ;
   fi = bs1.findlast( bs2.ustr(), csen ) ;
   findlast_report ( bs1, bs2, fi, csen, false, ofs ) ;

   csen = true ;
   bs2 = "picks" ;
   fi = bs1.findlast( bs2.ustr(), csen ) ;
   findlast_report ( bs1, bs2, fi, csen, false, ofs ) ;

   //** findlast ( wchar_t* ) **
   ++wchar ;
   gs.compose( "  %C) findlast ( wchar_t*, casesen )", &wchar ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   csen = true ;
   bs2 = "barn" ;
   fi = bs1.findlast( bs2.gstr(), csen ) ;
   findlast_report ( bs1, bs2, fi, csen, true, ofs ) ;

   csen = false ;
   fi = bs1.findlast( bs2.gstr(), csen ) ;
   findlast_report ( bs1, bs2, fi, csen, true, ofs ) ;

   //** findlast ( wchar_t ) **
   ++wchar ;
   gs.compose( "  %C) findlast ( wchar_t, casesen )", &wchar ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   csen = true ;
   bs2 = "A" ;
   fi = bs1.findlast( L'A', csen ) ;
   findlast_report ( bs1, bs2, fi, csen, true, ofs, true ) ;

   csen = false ;
   fi = bs1.findlast( L'A', csen ) ;
   findlast_report ( bs1, bs2, fi, csen, true, ofs, true ) ;

   //** findlast ( gString& ) **
   ++wchar ;
   gs.compose( "  %C) findlast ( gString&, casesen )", &wchar ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   csen = true ;
   bs2 = "slams" ;
   fi = bs1.findlast( bs2, csen ) ;
   findlast_report ( bs1, bs2, fi, csen, true, ofs ) ;

   csen = false ;
   bs2 = "a barn" ;
   fi = bs1.findlast( bs2, csen ) ;
   findlast_report ( bs1, bs2, fi, csen, true, ofs ) ;

   csen = false ;
   bs2 = "barnacle." ;
   fi = bs1.findlast( bs2, csen ) ;
   findlast_report ( bs1, bs2, fi, csen, true, ofs ) ;

   //** findx ( wchar_t, offset ) **
   ++wchar ;
   wchar_t searchChar = L'$' ;
   bs1 = "$$$$$$$$250,000.00" ;
   gs.compose( "  %C) findx ( wchar_t, offset )  gString data:\"%S\"", &wchar, bs1.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   off = ZERO ;
   fi = bs1.findx( searchChar, off ) ;
   gssub.loadChars( &bs1.gstr()[fi], 1 ) ;
   gs.compose( "      findx( '%C',%2d ) --> \"%S\" at:%2d", 
               &searchChar, &off, gssub.gstr(), &fi ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   off = fi ;
   searchChar = bs1.gstr()[off] ;
   fi = bs1.findx( searchChar, off ) ;
   gssub.loadChars( &bs1.gstr()[fi], 1 ) ;
   gs.compose( "      findx( '%C',%2d ) --> \"%S\" at:%2d", 
               &searchChar, &off, gssub.gstr(), &fi ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   //** scan ( offset ) **
   ++wchar ;
   bs1 = L" \t \n \r \x0B \x3000 \x0C Leading whitespace includes:    "
          "space, CJK space, \\n, \\r, \\t, v-tab, formfeed." ;
   gs.compose( "  %C) scan ( offset )  gString data:\"            %S\"", 
               &wchar, &bs1.gstr()[12] ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   off = ZERO ;
   fi = bs1.scan( off ) ;
   gssub = &bs1.gstr()[fi] ;
   gs.compose( "      scan( %d ) --> \"%S\" at:%2d", 
               &off, gssub.gstr(), &fi ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   off = bs1.after( L':', off ) ;
   fi = bs1.scan( off ) ;
   gssub = &bs1.gstr()[fi] ;
   gs.compose( "      scan( %d ) --> \"%S\" at:%2d", 
               &off, gssub.gstr(), &fi ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

}  //* End gsmTestAnalysisC() *

//********************************************************************************
//* Test the gString Analysis methods, group D.                                  *
//********************************************************************************
void gsmTestAnalysisD ( wchar_t wchar, std::ofstream& ofs )
{
   gString gs, gssub ;
   int32_t coff, fi ;
   bool    csen ;

   //** after ( char* ) **
   gString bs1( "Hundreds of flightless birds waddled along the beach of Bruny Island." ) ;
   gString bs2( gsALLOCMIN ) ;
   gs.compose( "  %C) after ( char*, offset, casesen )  gString data:\n"
               "     '%S'", &wchar, bs1.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   coff = ZERO ; csen = true ;
   bs2 = "of " ;
   after_report ( gs, bs2, coff, csen ) ;
   fi = bs1.after( bs2.ustr(), coff, csen ) ;
   after_report ( gs, bs1, bs2, fi, ofs ) ;
   
   coff = fi ; csen = true ;
   after_report ( gs, bs2, coff, csen ) ;
   fi = bs1.after( bs2.ustr(), coff, csen ) ;
   after_report ( gs, bs1, bs2, fi, ofs ) ;

   coff = ZERO ; csen = false ;
   bs2 = "FliGht" ;
   after_report ( gs, bs2, coff, csen ) ;
   fi = bs1.after( bs2.ustr(), coff, csen ) ;
   after_report ( gs, bs1, bs2, fi, ofs ) ;

   //** after ( uint8_t* ) **
   ++wchar ;
   gs.compose( "  %C) after ( uint8_t*, offset, casesen )", &wchar ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   coff = 4 ; csen = false ;
   bs2 = "add" ;
   after_report ( gs, bs2, coff, csen ) ;
   fi = bs1.after( (uint8_t*)(bs2.ustr()), coff, csen ) ;
   after_report ( gs, bs1, bs2, fi, ofs ) ;

   //** after ( wchar_t* ) **
   ++wchar ;
   gs.compose( "  %C) after ( wchar_t*, offset, casesen )", &wchar ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   coff = ZERO ; csen = false ;
   bs2 = "hund" ;
   after_report ( gs, bs2, coff, csen, true ) ;
   fi = bs1.after( bs2.gstr(), coff, csen ) ;
   after_report ( gs, bs1, bs2, fi, ofs ) ;

   //** after ( wchar_t ) **
   ++wchar ;
   gs.compose( "  %C) after ( wchar_t, offset, casesen )", &wchar ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   coff = ZERO ; csen = false ;
   bs2 = "B" ;
   after_report ( gs, bs2, coff, csen, true, true ) ;
   fi = bs1.after( bs2.gstr()[ZERO], coff, csen ) ;
   after_report ( gs, bs1, bs2, fi, ofs ) ;

   coff = ZERO ; csen = true ;
   after_report ( gs, bs2, coff, csen, true, true ) ;
   fi = bs1.after( bs2.gstr()[ZERO], coff, csen ) ;
   after_report ( gs, bs1, bs2, fi, ofs ) ;

   //** after ( gString& ) **
   ++wchar ;
   gs.compose( "  %C) after ( gString&, offset, casesen )", &wchar ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   coff = ZERO ; csen = false ;
   bs2 = "birds w" ;
   after_report ( gs, bs2, coff, csen, true, true ) ;
   fi = bs1.after( bs2, coff, csen ) ;
   after_report ( gs, bs1, bs2, fi, ofs ) ;

   coff = ZERO ; csen = false ;
   bs2 = "the " ;
   after_report ( gs, bs2, coff, csen, true, true ) ;
   fi = bs1.after( bs2, coff, csen ) ;
   after_report ( gs, bs1, bs2, fi, ofs ) ;

}  //* End gsmTestAnalysisD() *

//********************************************************************************
//* Test the gString Analysis methods, group E.                                  *
//********************************************************************************
void gsmTestAnalysisE ( wchar_t wchar, std::ofstream& ofs )
{
   gString gs, gssub ;
   int32_t coff, fi ;
   bool    csen ;

   //** findr ( char* )
   gString bs1( "You’re braver than you believe, and Stronger than you seem, "
                "And smarter than you think. - Winnie the Pooh" ) ;
   gString bs2( gsALLOCMIN ) ;
   gs.compose( "  %C) findr ( char*, offset, casesen )  gString data:\n"
               "     '%S'", &wchar, bs1.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   coff = -1 ; csen = false ;
   bs2 = "you’re" ;
   findr_report ( gs, bs2, coff, csen ) ;
   fi = bs1.findr( bs2.ustr(), coff, csen ) ;
   findr_report ( gs, bs1, bs2, fi, ofs ) ;

   coff = -1 ; csen = true ;
   bs2 = "Win" ;
   findr_report ( gs, bs2, coff, csen ) ;
   fi = bs1.findr( bs2.ustr(), coff, csen ) ;
   findr_report ( gs, bs1, bs2, fi, ofs ) ;

   coff = fi ; csen = true ;
   bs2 = "th" ;
   findr_report ( gs, bs2, coff, csen ) ;
   fi = bs1.findr( bs2.ustr(), coff, csen ) ;
   findr_report ( gs, bs1, bs2, fi, ofs ) ;

   //** findr ( uint8_t* ) **
   ++wchar ;
   gs.compose( "  %C) findr ( uint8_t*, offset, casesen )", &wchar ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   bs2 = "And" ;
   coff = -1 ; csen = true ;
   findr_report ( gs, bs2, coff, csen ) ;
   fi = bs1.findr( (uint8_t*)(bs2.ustr()), coff, csen ) ;
   findr_report ( gs, bs1, bs2, fi, ofs ) ;

   //** findr ( wchar_t* ) **
   ++wchar ;
   gs.compose( "  %C) findr ( wchar_t*, offset, casesen )", &wchar ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   coff = fi - 1 ; csen = false ;
   findr_report ( gs, bs2, coff, csen, true ) ;
   fi = bs1.findr( bs2.gstr(), coff, csen ) ;
   findr_report ( gs, bs1, bs2, fi, ofs ) ;

   bs2 = "Winnie" ;
   coff = -1 ; csen = true ;
   findr_report ( gs, bs2, coff, csen, true ) ;
   fi = bs1.findr( bs2.gstr(), coff, csen ) ;
   findr_report ( gs, bs1, bs2, fi, ofs ) ;

   bs2 = "Pooh" ;
   coff = fi ; csen = false ;
   findr_report ( gs, bs2, coff, csen, true ) ;
   fi = bs1.findr( bs2.gstr(), coff, csen ) ;
   findr_report ( gs, bs1, bs2, fi, ofs ) ;

   //** findr ( wchar_t ) **
   ++wchar ;
   gs.compose( "  %C) findr ( wchar_t, offset, casesen )", &wchar ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   bs2 = "w" ;
   coff = -1 ; csen = false ;
   findr_report ( gs, bs2, coff, csen, true, true ) ;
   fi = bs1.findr( bs2.gstr()[0], coff, csen ) ;
   findr_report ( gs, bs1, bs2, fi, ofs ) ;

   bs2 = "A" ;
   coff = fi ; csen = true ;
   findr_report ( gs, bs2, coff, csen, true, true ) ;
   fi = bs1.findr( bs2.gstr()[0], coff, csen ) ;
   findr_report ( gs, bs1, bs2, fi, ofs ) ;

   bs2 = "S" ;
   coff = fi ; csen = true ;
   findr_report ( gs, bs2, coff, csen, true, true ) ;
   fi = bs1.findr( bs2.gstr()[0], coff, csen ) ;
   findr_report ( gs, bs1, bs2, fi, ofs ) ;

   //** findr ( gString& ) **
   ++wchar ;
   gs.compose( "  %C) findr ( gString&, offset, casesen )", &wchar ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   bs2 = "Brave" ;
   coff = -1 ; csen = true ;
   findr_report ( gs, bs2, coff, csen, true ) ;
   fi = bs1.findr( bs2, coff, csen ) ;
   findr_report ( gs, bs1, bs2, fi, ofs ) ;

   coff = -1 ; csen = false ;
   findr_report ( gs, bs2, coff, csen, true ) ;
   fi = bs1.findr( bs2, coff, csen ) ;
   findr_report ( gs, bs1, bs2, fi, ofs ) ;

}  //* End gsmTestAnalysisE() *

//********************************************************************************
//* Test the gString Analysis methods, group F.                                  *
//********************************************************************************
void gsmTestAnalysisF ( wchar_t wchar, std::ofstream& ofs )
{
   gString gs,                            // display
           gst ;                          // template
   wchar_t wbuff1[256], wbuff2[256] ;     // target buffers for gscanf()
   char    cbuff1[256], cbuff2[256], cbuff3[256] ;
   double   flt_64 ;                      // target variables for gscanf()
   float    flt_32 ;
   int64_t  int_64 ;
   int32_t  int_32 ;
   short    int_16 ;
   int8_t   int_08 ;
   wchar_t  scanChar ;
   gString bs1 ;
   int32_t coff, reqItems, recItems ;

   const char* srcA = "Grapes: $4.99/lb. 12 rolls of moon tape: $9.59. "
                      "AA batteries: 8-pack for 2 dollars or each: 75 cents." ;
   const char* srcB = "Cabbage: 14.50元 per kilogram or 每片6元。 "
                      "包子(pork buns):每件 2.5元。 Unlimited 饺子:55元。" ;
   const char* srcC = "If a blue train with 18 cars and 6 conductors is "
                      "travelling at 254.72kph, when will it get there?" ;
   const char* srcD = "Fred has 27 apples, 14 oranges and 32 tomatoes "
                      "for which he paid $42.61. What kind of car does he drive?" ;
   const char* errTemplate = "       Error: %d items expected, %d items scanned." ;


   //** gscanf ( char* spec, args, ... ) **
   bs1 = srcA ;
   gs.compose( "  %C) gscanf ( char* spec, args, ... )\n"
               "     '%S'", &wchar, bs1.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   reqItems = 10 ;
   gst = "%s $%f/lb. %hd rolls of moon %s $%lf. AA batteries: %d-pack for %lld %S or each: %hhd %S" ;
   gs.compose( "      gscanf ( \"%S\", ... ) [%d items]", gst.gstr(), &reqItems ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   recItems = bs1.gscanf( gst.ustr(), cbuff1, &flt_32, &int_16, cbuff2, &flt_64, 
                          &int_32, &int_64, wbuff1, &int_08, wbuff2 ) ;
   if ( recItems == reqItems )
   {
      gs.compose( "       [%d Items] %s%1.02f %hd %s%1.02lf %d-pack %lld %S %hhd%S", 
                  &recItems, cbuff1, &flt_32, &int_16, cbuff2, &flt_64,
                  &int_32, &int_64, wbuff1, &int_08, wbuff2 ) ;
   }
   else
      gs.compose( errTemplate, &reqItems, &recItems ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   //** gscanf ( wchar_t* spec, args, ... ) **
   ++wchar ;
   bs1 = srcB ;
   gs.compose( "  %C) gscanf ( wchar_t* spec, args, ... )\n"
               "     '%S'", &wchar, bs1.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   reqItems = 9 ;
   gst = "%s %f%S per kilogram or 每片%hhd%S 包子%s %s %lf元。 Unlimited 饺子:%hd" ;
   gs.compose( "      gscanf ( \"%S\", ... ) [%d items]", gst.gstr(), &reqItems ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   recItems = bs1.gscanf( gst.gstr(), cbuff1, &flt_32, wbuff1, &int_08, wbuff2, 
                          cbuff2, cbuff3, &flt_64, &int_16 ) ;
   if ( recItems == reqItems )
   {
      gs.compose( "       [%d Items] %s %2.2f%S %hhd%S %s %s%2.1lf dumplings:%hd元。", 
                  &recItems, cbuff1, &flt_32, wbuff1, &int_08, wbuff2, 
                  cbuff2, &cbuff3, &flt_64, &int_16 ) ;
   }
   else
      gs.compose( errTemplate, &reqItems, &recItems ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   //** gscanf ( offset, char* spec, args, ... ) **
   ++wchar ;
   bs1 = srcC ;
   gs.compose( "  %C) gscanf ( offset, char* spec, args, ... )\n"
               "     '%S'", &wchar, bs1.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   reqItems = 6 ;
   coff = bs1.find( L" 1" ) ;
   gst = " %hd %s and %hhd conductors is travelling at %lf%3S, %24[^.]" ;
   gs.compose( "      gscanf ( %d, \"%S\", ... ) [%d items]", &coff, gst.gstr(), &reqItems ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   recItems = bs1.gscanf( coff, gst.ustr(), &int_16, cbuff1, &int_08, 
                          &flt_64, wbuff1, cbuff2 ) ;
   if ( recItems == reqItems )
   {
      gs.compose( "       [%d Items] %hd %s %hhd %.2lf %S '%s'", 
                  &recItems, &int_16, cbuff1, &int_08, &flt_64, wbuff1, cbuff2 ) ;
   }
   else
      gs.compose( errTemplate, &reqItems, &recItems ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

   //** gscanf ( offset, wchar_t* spec, args, ... ) **
   ++wchar ;
   bs1 = srcD ;
   reqItems = 9 ;
   coff = bs1.find( L"2" ) ;
   gs.compose( "  %C) gscanf ( offset, wchar_t* spec, args, ... )\n"
               "     '%S'", &wchar, bs1.gstr() ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   gst = "%hhd %6s, %hd %s and %d %S %[^$]%C%lf" ;
   gs.compose( "      gscanf ( %d, \"%S\", ... ) [%d items]", &coff, gst.gstr(), &reqItems ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;
   recItems = bs1.gscanf( coff, gst.gstr(), &int_08, cbuff1, &int_16, cbuff2, 
                          &int_32, wbuff1, cbuff3, &scanChar, &flt_64 ) ;
   if ( recItems == reqItems )
   {
      gs.compose( "       [%d Items] %hhd %s %hd %s %d %S '%s' %C%.2lf  (a can't-a-Ford, duh! :)", 
                  &recItems, &int_08, cbuff1, &int_16, cbuff2, 
                  &int_32, wbuff1, cbuff3, &scanChar, &flt_64 ) ;
   }
   else
      gs.compose( errTemplate, &reqItems, &recItems ) ;
   wcout << gs << endl ;
   ofs   << gs << endl ;

}  //* End gsmTestAnalysisF() *

//********************************************************************************
//* Retrieve gString debugging data and write to debugging file.                 *
//* Optionally, also display debugging data to display.                          *
//* Note: 'tostdout' is optional, 'false' by default.                            *
//********************************************************************************
void dumpGString ( const char* title, const gString& bs, ofstream& ofs, bool tostdout )
{
   const wchar_t* gstr ;
   const char*    ustr ;
   const short*   cwid ;
   int32_t gsw, gsu, gscols, gschars, utfbytes ;
   bool isascii ;

   gString gs( "%-6s: ", title ), gspad ;
   gspad.padCols( gs.gscols() ) ;

   #if DEBUG_GSTRING != 0
   bs.dumpGstring ( gstr, ustr, cwid, gsw, gsu, gschars, gscols, utfbytes, isascii ) ;
   #else    // dummy report
   gstr = enableDbgMsgW ;
   ustr = enableDbgMsg ;
   cwid = NULL ;
   gsw = gsu = gschars = gscols = utfbytes = ZERO ;
   isascii = false ;
   #endif   // dummy report
   gs.append( "gstr:%p gsw:%04Xh(%6d) '%S' \n"
              "%Sustr:%p gsu:%04Xh(%6d) '%s'\n"
              "%Scwid:%p gschars:%02d gscols:%02d utfbytes:%02d isascii:%hhd\n",
              gstr, &gsw, &gsw, gstr,
              gspad.gstr(), ustr, &gsu, &gsu, ustr,
              gspad.gstr(), cwid, &gschars, &gscols, &utfbytes, &isascii ) ;
   ofs << gs << endl ;

   if ( tostdout )
   {
      gspad.insert( L"\n          " ) ;
      int32_t indx = ZERO ;
      gs.limitChars( gs.gschars() - 2 ) ;    // remove trailing newline
      do                                     // indent each the row
      {
         gs.insert( "     ", indx ) ;        // inset base of row
         //* For the long rows, break the rows to fit the display. *
         if ( (gs.after( L"?\",", indx )) > ZERO )
         {
            indx = gs.after( L"?\",", indx ) ;
            gs.insert( gspad.gstr(), indx ) ;   // insert the line break
            indx = gs.find( L"?\",", indx ) ;   // step over the insertion
         }
      }
      while ( (indx = gs.after( L'\n', indx )) >= ZERO ) ;
      wcout << gs << endl ;
   }

}  //* End dumpGString() *

//********************************************************************************
//* Construct the display string for reporting the find() test.                  *
//********************************************************************************
void find_report ( gString& gs, const gString& bs2, int32_t coff, 
                   int32_t csen, int32_t max, bool lng )
{
   gs.compose( "      find ( \"%S\", %2d, %s, %2d ) --> ", 
               bs2.gstr(), &coff, (csen ? " true" : "false"), &max ) ;
   if ( lng )
   {
      int32_t i = gs.after( "find ( " ) ;
      gs.insert( L'L', i ) ;
   }

}  //* End find_report() *

//********************************************************************************
//* Complete the find() test report and output to display and file.              *
//********************************************************************************
void find_report ( gString& gs, const gString& bs1, const gString& bs2, int32_t fi, int32_t max, std::ofstream& ofs )
{
   int32_t clen ;
   gString gssub ;

   if ( fi >= ZERO )
   {
      clen = max >= 1 ? max : bs2.gschars() - 1 ;
      gssub.loadChars( &bs1.gstr()[fi], clen ) ;
      gs.append( "\"%S\" at:%2d", gssub.gstr(), &fi ) ;
   }
   else { gs.append( "(not found)" ) ; }
   wcout << gs << endl ;
   ofs   << gs << endl ;

}  //* End find_report() *

//********************************************************************************
//* Report on the findlast() test. Output to display and file.                   *
//********************************************************************************
void findlast_report ( const gString& bs1, const gString& bs2, int32_t fi, 
                       bool csen, bool lng, ofstream& ofs, bool ischar )
{
   gString gs, gssub ;

   if ( fi >= ZERO )
      gssub.loadChars( &bs1.gstr()[fi], bs2.gschars() - 1 ) ;
   else
      gssub = "(not found)" ;
   gs.compose( "      findlast(%s\"%S\", %s ) --> \"%S\" at:%2d",
               (lng ? " L" : " "),
               bs2.gstr(), (csen ? " true" : "false"), gssub.gstr(), &fi ) ;
   if ( ischar )
   { gs.replace( L'"', L"'" ) ; gs.replace( L'"', L"'" ) ; }
   wcout << gs << endl ;
   ofs   << gs << endl ;

}  //* End findlast_report() *

//********************************************************************************
//* Construct the display string for reporting the after() test.                 *
//********************************************************************************
void after_report ( gString&gs, const gString& bs2, int32_t coff, 
                    bool csen, bool lng, bool ischar )
{
   gs.compose( "      after ( \"%S\", %2d, %s ) --> ", 
               bs2.gstr(), &coff, (csen ? " true" : "false") ) ;
   if ( lng )
   {
      int32_t i = gs.after( "after ( " ) ;
      gs.insert( L'L', i ) ;
   }
   if ( ischar )
   { gs.replace( L'"', L"'" ) ; gs.replace( L'"', L"'" ) ; }

}  //* End after_report() *

//********************************************************************************
//* Complete the after() test report and output to display and file.             *
//********************************************************************************
void after_report ( gString&gs, const gString& bs1, const gString& bs2, int32_t fi, std::ofstream& ofs )
{
   gString gssub ;
   int32_t indx ;

   if ( fi >= ZERO )
   {
      gssub = &bs1.gstr()[fi] ;
      if ( (indx = gssub.after( L' ' )) > ZERO )
      {
         gssub.limitChars( indx ) ;
         gssub.append( L"..." ) ;
      }
      gs.append( "\"%S\" at:%2d", gssub.gstr(), &fi ) ;
   }
   else { gs.append( "(not found)" ) ; }
   wcout << gs << endl ;
   ofs   << gs << endl ;

}  //* End after_report() *

//********************************************************************************
//* Construct the display string for reporting the findr() test.                 *
//********************************************************************************
void findr_report ( gString& gs, const gString& bs2, int32_t coff, 
                    bool csen, bool lng, bool ischar )
{
   gs.compose( "      findr ( \"%S\", %2d, %s ) --> ", 
               bs2.gstr(), &coff, (csen ? " true" : "false") ) ;

   if ( lng )
   {
      int32_t i = gs.after( "findr ( " ) ;
      gs.insert( L'L', i ) ;
   }
   if ( ischar )
   { gs.replace( L'"', L"'" ) ; gs.replace( L'"', L"'" ) ; }

}  //* End findr_report() *

//********************************************************************************
//* Complete the findr() test report and output to display and file.             *
//********************************************************************************
void findr_report ( gString& gs, const gString& bs1, const gString& bs2, int32_t fi, std::ofstream& ofs )
{

   after_report ( gs, bs1, bs2, fi, ofs ) ;

}  //* End findr_report() *

//********************************************************************************
//* Format and display the results of a replace() operation.                     *
//********************************************************************************
void replace_report ( const gString& bssrc, const gString& bsnew, const gString& bs, 
                      int32_t coff, bool csen, bool rall, ofstream& ofs, 
                      bool s_lng, bool n_lng, bool s_chr, bool n_chr )
{
   short padWid = 58, indx ;

   gString gs( "     replace( \"%S\", \"%S\", %d, %s, %s ) ",
               bssrc.gstr(), bsnew.gstr(), &coff, 
               (csen ? "true" : "false"), (rall ? "true" : "false")
             ) ;
   if ( s_lng )
   { indx = gs.find( L'"' ) ; gs.insert( L'L', indx ) ; }
   if ( n_lng )
   { indx = gs.after( L", " ) ; gs.insert( L'L', indx ) ; }
   if ( s_chr )
   { gs.replace( L'"', L"'" ) ; gs.replace( L'"', L"'" ) ; }
   if ( n_chr )
   {
      indx = gs.after( L", " ) ;
      gs.replace( L'"', L"'", indx ) ;
      gs.replace( L'"', L"'", indx ) ;
   }
   gs.padCols( padWid ) ;
   gs.append( "--> \"%S\"", bs.gstr() ) ;

   wcout << gs << endl ;
   ofs   << gs << endl ;

}  //* End replace_report() *

//********************************************************************************
//* Test the gString Output method group.                                        *
//********************************************************************************
void gsmTestOutputs ( wchar_t wchar, std::ofstream& ofs )
{
   gString gs, gsdbg ;
   wchar_t wc[gsALLOCMIN] ;               // target buffers for copy()
   char    cc[gsALLOCMIN] ;
   uint8_t uc[gsALLOCMIN] ;
   int32_t limit ;                        // output limit

   //** copy **
   gString bs( "Copy text from gString to target buffer.（完整副本）" ) ;
   dumpGString ( "  copy", bs, ofs, false ) ;

   //* Copy to wchar_t target. *
   gs.compose( "  %C) '%S'", &wchar, bs.gstr() ) ;
   wcout << gs << endl ;
   bs.copy( wc, gsALLOCMIN ) ;
   gs.compose( "     wchar_t target:\n"
               "       '%S' (full copy)", wc ) ;
   wcout << gs << endl ;
   ofs << gs << endl ;
   limit = 41 ;
   bs.copy( wc, limit ) ;
   gs.compose( "       '%S'             (limit to %d characters)", wc, &limit ) ;
   wcout << gs << endl ;
   ofs << gs << endl ;
   limit = 47 ;
   bs.copy( wc, gsALLOCMIN, limit ) ;
   gs.compose( "       '%S'       (limit to %d columns)", wc, &limit ) ;
   wcout << gs << endl ;
   ofs << gs << endl ;

   //* Copy to char target. *
   wc[ZERO] = L'\0' ; cc[ZERO] = '\0' ; uc[ZERO] = '\0' ;   // clear the buffers
   bs.copy( cc, gsALLOCMIN ) ;
   gs.compose( "     char target:\n"
               "       '%s' (full copy)", cc ) ;
   wcout << gs << endl ;
   ofs << gs << endl ;
   limit = 41 ;
   bs.copy( cc, limit ) ;
   gs.compose( "       '%s'             (limit to %d bytes)", cc, &limit ) ;
   wcout << gs << endl ;
   ofs << gs << endl ;
   limit = 47 ;
   bs.copy( cc, gsALLOCMIN, limit ) ;
   gs.compose( "       '%s'       (limit to %d columns)", cc, &limit ) ;
   wcout << gs << endl ;
   ofs << gs << endl ;

   //* Copy to uint8_t target. *
   wc[ZERO] = L'\0' ; cc[ZERO] = '\0' ; uc[ZERO] = '\0' ;   // clear the buffers
   bs.copy( uc, gsALLOCMIN ) ;
   gs.compose( "     uint8_t target:\n"
               "       '%s' (full copy)", uc ) ;
   wcout << gs << endl ;
   ofs << gs << endl ;
   limit = 41 ;
   bs.copy( uc, limit ) ;
   gs.compose( "       '%s'             (limit to %d bytes)", uc, &limit ) ;
   wcout << gs << endl ;
   ofs << gs << endl ;
   limit = 47 ;
   bs.copy( uc, gsALLOCMIN, limit ) ;
   gs.compose( "       '%s'       (limit to %d columns)", uc, &limit ) ;
   wcout << gs << endl ;
   ofs << gs << endl ;

   //** substr **
   ++wchar ;
   bs = "Extract a substring -->target（子字符串）sub-string<-- and copy it to target." ;
   dumpGString ( "substr", bs, ofs, false ) ;
   gs.compose( "  %C) '%S'", &wchar, bs.gstr() ) ;
   wcout << gs << endl ;

   int32_t suboff = 23, sublen = 22, retlen ;
   retlen = bs.substr( wc, suboff, sublen ) ;   // wchar_t target
   gs.compose( "      wchar_t target: '%S' offset:%02d length:%02d(%02d)", wc, &suboff, &sublen, &retlen ) ;
   wcout << gs << endl ;
   ofs << gs << endl ;
   //* char target *
   retlen = bs.substr( cc, suboff, sublen ) ;
   gs.compose( "         char target: '%s' offset:%02d length:%02d(%02d)", cc, &suboff, &sublen, &retlen ) ;
   wcout << gs << endl ;
   ofs << gs << endl ;
   //* uint8_t target *
   retlen = bs.substr( uc, suboff, sublen ) ;
   gs.compose( "      uint8_t target: '%s' offset:%02d length:%02d(%02d)", uc, &suboff, &sublen, &retlen ) ;
   wcout << gs << endl ;
   ofs << gs << endl ;
   //* gString target *
   gString bstrg( gsALLOCMIN ) ;
   retlen = bs.substr( bstrg, suboff, sublen ) ;
   gs.compose( "      gString target: '%S' offset:%02d length:%02d(%02d)", bstrg.gstr(), &suboff, &sublen, &retlen ) ;
   wcout << gs << endl ;
   ofs << gs << endl ;

   //** Output Stream **
   ++wchar ;
   bs.compose( "  %C) operator<< is used to write wchar_t text to \"wcout\" stream. 我们正顺着溪流漂流。\n"
               "     operator<< for narrow-stream text is written to UTF-8 file.", &wchar ) ;
   wcout << bs << endl ;
   bs = "operator<< is used to write narrow_stream text to UTF-8 file. 我们正顺着溪流漂流。\n"
        "operator<< for wchar_t text is written to \"wcout\" (stdout)." ;
   ofs << bs << endl ;

   //** Version number **
   ++wchar ;
   gs.compose( "  %C) gString version: '%s'", &wchar, bs.Get_gString_Version() ) ;
   wcout << gs << endl ;
   ofs << "gString version: '" << bs.Get_gString_Version() << "'" << endl ;

   //** dbMsg - debugging only **
   ++wchar ;
#if DEBUG_GSTRING != 0
   bs.dbMsg( bstrg ) ;
#else    // DEBUGGING NOT ENABLED
   bstrg = enableDbgMsg ;
#endif   // DEBUG_GSTRING
   gs.compose( "  %C) dbMsg reports: '%S'", &wchar, bstrg.gstr() ) ;
   wcout << gs << endl ;

}  //* End gsmTestOutputs() *

//********************************************************************************
//* Attempt to overflow the data storage buffers by specifying source data       *
//* larger than default allocation.                                              *
//*                                                                              *
//* Constructors:                                                                *
//*    source == char*                                                           *
//*    source == uint8_t*                                                        *
//*    source == wchar_t*                                                        *
//*    source == formatting template ( char and wchar_t )                        *
//*                                                                              *
//* Assignments:                                                                 *
//*    source == char*                                                           *
//*    source == uint8_t*                                                        *
//*    source == wchar_t*                                                        *
//*    source == formatting template ( char and wchar_t )                        *
//*                                                                              *
//* -- The data used is a small Lorem Ipsum sentence, duplicated as needed to    *
//*    reach the required data size.                                             *
//* -- Note that because Lorem Ipsum is ASCII data the word count would equal    *
//*    the byte count by default. For this reason, we insert the Chinese word    *
//*    "Ma" (horse) into the test data at various points. This exercises the     *
//*    various byte vs. character calculations used during encoding operations.  *
//********************************************************************************
void gsmBufferOverflowA ( wchar_t wchar, std::ofstream& ofs )
{
   const char* const LorenWho = "Lorem ipsum dolor sit amet, consectetur "
                                "adipiscing elit, sed do eiusmod tempor "
                                "incididunt ut labore et dolore magna 马aliqua." ;
   gString gsSrc( LorenWho ),             // source data
           gsOut, gsVal1, gsVal2,         // output formatting
           gsVal3, gsVal4, gsVal5 ;
   int charCnt = ZERO,                    // number of characters written to buffer
       trgCnt,                            // target count
       gsscnt  = gsSrc.utfbytes() - 1,    // characters/bytes block size (not incl. nullchar)
       trgPad  = gsscnt * 2,              // work buffer padding
       freeSp,                            // free buffer space after initialization
      capBytes, capChars, capCbuff, capBbuff ; // final values

   //********************
   //* char Source Data *
   //********************
   trgCnt = gsALLOCMAX * 4 + gsscnt ;     // larger than byte buffer
   char* cbuff = new char[trgCnt + trgPad] ;
   do
   {
      gsSrc.copy( &cbuff[charCnt], gsSrc.utfbytes() ) ;
      charCnt += gsscnt ;
   }
   while ( charCnt < trgCnt ) ;
   cbuff[charCnt++] = '!' ;                // mark the end of source text
   cbuff[charCnt++] = '\0' ;               // terminate the string
   gsVal1.formatInt( charCnt, 9, true ) ;  // report the results
   gsOut.compose( "  %C) gString(const char*    : bytes:%S source)", &wchar, gsVal1.gstr() ) ;
   wcout << gsOut << endl ;
   ofs   << gsOut << endl ;
   gString gsCharIn( cbuff ) ;
   capBytes = gsCharIn.utfbytes() ;
   capChars = gsCharIn.gschars() ;
   capCbuff = gsCharIn.wAlloc() ;
   capBbuff = gsCharIn.uAlloc() ;
   freeSp   = gsCharIn.freeSpace() ;
   gsVal1.formatInt( capBytes, 9, true ) ;
   gsVal2.formatInt( capChars, 9, true ) ;
   gsVal3.formatInt( capCbuff, 9, true ) ;
   gsVal4.formatInt( capBbuff, 9, true ) ;
   gsOut.compose( "            (initialized    : bytes:%S chars:%S storage:%S words and %S bytes freeSp:%d)",
                  gsVal1.gstr(), gsVal2.gstr(), gsVal3.gstr(), gsVal4.gstr(), &freeSp ) ;
   wcout << gsOut << endl ;
   ofs   << gsOut << endl ;
   delete [] cbuff ;                      // release the dynamic allocation

   //***********************
   //* uint8_t Source Data *
   //***********************
   ++wchar ;
   trgCnt = gsALLOCMAX * 4 + gsscnt ;     // larger than byte buffer
   uint8_t* ubuff = new uint8_t[trgCnt + trgPad] ;
   charCnt = ZERO ;
   do
   {
      gsSrc.copy( &ubuff[charCnt], gsSrc.utfbytes() ) ;
      charCnt += gsscnt ;
   }
   while ( charCnt < trgCnt ) ;
   cbuff[charCnt++] = '!' ;                // mark the end of source text
   cbuff[charCnt++] = '\0' ;               // terminate the string
   gsVal1.formatInt( charCnt, 9, true ) ;  // report the results
   gsOut.compose( "  %C) gString(const uint8_t* : bytes:%S source)", &wchar, gsVal1.gstr() ) ;
   wcout << gsOut << endl ;
   ofs   << gsOut << endl ;
   gString gsByteIn( ubuff ) ;
   capBytes = gsByteIn.utfbytes() ;
   capChars = gsByteIn.gschars() ;
   capCbuff = gsByteIn.wAlloc() ;
   capBbuff = gsByteIn.uAlloc() ;
   freeSp   = gsByteIn.freeSpace() ;
   gsVal1.formatInt( capBytes, 9, true ) ;
   gsVal2.formatInt( capChars, 9, true ) ;
   gsVal3.formatInt( capCbuff, 9, true ) ;
   gsVal4.formatInt( capBbuff, 9, true ) ;
   gsOut.compose( "            (initialized    : bytes:%S chars:%S storage:%S words and %S bytes freeSp:%d)",
                  gsVal1.gstr(), gsVal2.gstr(), gsVal3.gstr(), gsVal4.gstr(), &freeSp ) ;
   wcout << gsOut << endl ;
   ofs   << gsOut << endl ;
   delete [] ubuff ;                      // release the dynamic allocation

   //***********************
   //* wchar_t Source Data *
   //***********************
   ++wchar ;
   gsscnt  = gsSrc.gschars() - 1 ;        // source characters
   trgCnt = gsALLOCMAX + gsscnt ;         // larger than word buffer
   wchar_t* wbuff = new wchar_t[trgCnt + trgPad] ;
   charCnt = ZERO ;
   do
   {
      gsSrc.copy( &wbuff[charCnt], gsSrc.gschars() ) ;
      charCnt += gsscnt ;
   }
   while ( charCnt < trgCnt ) ;
   wbuff[charCnt++] = L'!' ;               // mark the end of source text
   wbuff[charCnt++] = L'\0' ;              // terminate the string
   gsVal1.formatInt( charCnt, 9, true ) ;  // report the results
   gsOut.compose( "  %C) gString(const wchar_t* : chars:%S source)", &wchar, gsVal1.gstr() ) ;
   wcout << gsOut << endl ;
   ofs   << gsOut << endl ;
   gString gsWordIn( wbuff ) ;
   capChars = gsWordIn.gschars() ;
   capBytes = gsWordIn.utfbytes() ;
   capCbuff = gsWordIn.wAlloc() ;
   capBbuff = gsWordIn.uAlloc() ;
   freeSp   = gsWordIn.freeSpace() ;
   gsVal1.formatInt( capChars, 9, true ) ;
   gsVal2.formatInt( capBytes, 9, true ) ;
   gsVal3.formatInt( capCbuff, 9, true ) ;
   gsVal4.formatInt( capBbuff, 9, true ) ;
   gsOut.compose( "            (initialized    : chars:%S bytes:%S storage:%S words and %S bytes freeSp:%d)",
                  gsVal1.gstr(), gsVal2.gstr(), gsVal3.gstr(), gsVal4.gstr(), &freeSp ) ;
   wcout << gsOut << endl ;
   ofs   << gsOut << endl ;
   delete [] wbuff ;

   //***********************
   //* gString Source Data *
   //***********************
   ++wchar ;
   charCnt = gsWordIn.gschars()  ;
   gsVal1.formatInt( charCnt, 9, true ) ;  // report the results
   gsOut.compose( "  %C) gString(const gString& : chars:%S source)", &wchar, gsVal1.gstr() ) ;
   wcout << gsOut << endl ;
   ofs   << gsOut << endl ;
   gString gsGsIn( gsWordIn ) ;
   capChars = gsGsIn.gschars() ;
   capBytes = gsGsIn.utfbytes() ;
   capCbuff = gsGsIn.wAlloc() ;
   capBbuff = gsGsIn.uAlloc() ;
   freeSp   = gsGsIn.freeSpace() ;
   gsVal1.formatInt( capChars, 9, true ) ;
   gsVal2.formatInt( capBytes, 9, true ) ;
   gsVal3.formatInt( capCbuff, 9, true ) ;
   gsVal4.formatInt( capBbuff, 9, true ) ;
   gsOut.compose( "            (initialized    : chars:%S bytes:%S storage:%S words and %S bytes freeSp:%d)",
                  gsVal1.gstr(), gsVal2.gstr(), gsVal3.gstr(), gsVal4.gstr(), &freeSp ) ;
   wcout << gsOut << endl ;
   ofs   << gsOut << endl ;

   //***********************************
   //* Formatting template Source Data *
   //***********************************
   ++wchar ;
   const char* cTemplate = 
      "This is a template including gsWordIn characters: \"%S\" to over-fill buffer." ;
   // Programmer's Note: In the EXTREMELY unlikely case where the free space 
   // available for a string formatting token (%s or %S) is insufficient, then 
   // swprintf will not process that string at all. For this test, we limit the 
   // length of text loaded via the "%S" token to avoid this. The combined length 
   // of the formatting string 'cTemplate' and the contents of 'gsGsIn' is 
   // greater than the maximum gString allocation, so it triggers the overflow 
   // test and truncates the constructed string by approximately 22 characters.
   // (The actual failure would occur at approximately 1023949 characters.)
   // Please see overflow tests, group B, below for examples of using the 'append' 
   // and 'insert' methods to intentionally cause the swprintf() function to 
   // report an "insufficient buffer space" error condition.
   gsGsIn.limitChars( 1023945 ) ;
   gsOut = cTemplate ;
   charCnt = gsOut.gschars() + gsGsIn.gschars() ;
   gsVal1.formatInt( charCnt, 9, true ) ;  // report the results
   gsOut.compose( "  %C) gString(format template: chars:%S source)", &wchar, gsVal1.gstr() ) ;
   wcout << gsOut << endl ;
   ofs   << gsOut << endl ;
   gString gsCtIn( cTemplate, gsGsIn.gstr() ) ;
   capChars = gsCtIn.gschars() ;
   capBytes = gsCtIn.utfbytes() ;
   capCbuff = gsCtIn.wAlloc() ;
   capBbuff = gsCtIn.uAlloc() ;
   freeSp   = gsCtIn.freeSpace() ;
   gsVal1.formatInt( capChars, 9, true ) ;
   gsVal2.formatInt( capBytes, 9, true ) ;
   gsVal3.formatInt( capCbuff, 9, true ) ;
   gsVal4.formatInt( capBbuff, 9, true ) ;
   gsOut.compose( "            (initialized    : chars:%S bytes:%S storage:%S words and %S bytes freeSp:%d)",
                  gsVal1.gstr(), gsVal2.gstr(), gsVal3.gstr(), gsVal4.gstr(), &freeSp ) ;
   wcout << gsOut << endl ;
   ofs   << gsOut << endl ;

   //*************************
   //* Assignment: Char Data *
   //*************************
   ++wchar ;
   gsscnt  = gsSrc.utfbytes() - 1 ;       // source bytes
   trgCnt = gsALLOCMAX * 4 + gsscnt ;     // larger than byte buffer
   cbuff = new char[trgCnt + trgPad] ;    // allocate the source buffer
   charCnt = ZERO ;
   do
   {
      gsSrc.copy( &cbuff[charCnt], gsSrc.utfbytes() ) ;
      charCnt += gsscnt ;
   }
   while ( charCnt < trgCnt ) ;
   cbuff[charCnt++] = '!' ;                // mark the end of source text
   cbuff[charCnt++] = '\0' ;               // terminate the string
   gsVal1.formatInt( charCnt, 9, true ) ;  // report the results
   gsOut.compose( "  %C) Assign (const char*    : bytes:%S source)", &wchar, gsVal1.gstr() ) ;
   wcout << gsOut << endl ;
   ofs   << gsOut << endl ;
   gString gsAc ;                          // (default allocation)
   gsAc = cbuff ;
   capChars = gsAc.gschars() ;
   capBytes = gsAc.utfbytes() ;
   capCbuff = gsAc.wAlloc() ;
   capBbuff = gsAc.uAlloc() ;
   freeSp   = gsAc.freeSpace() ;
   gsVal1.formatInt( capChars, 9, true ) ;
   gsVal2.formatInt( capBytes, 9, true ) ;
   gsVal3.formatInt( capCbuff, 9, true ) ;
   gsVal4.formatInt( capBbuff, 9, true ) ;
   gsOut.compose( "            (initialized    : chars:%S bytes:%S storage:%S words and %S bytes freeSp:%d)",
                  gsVal1.gstr(), gsVal2.gstr(), gsVal3.gstr(), gsVal4.gstr(), &freeSp ) ;
   wcout << gsOut << endl ;
   ofs   << gsOut << endl ;

   //*************************
   //* Assignment: Byte Data *
   //*************************
   ++wchar ;
   gsVal1.formatInt( charCnt, 9, true ) ;  // report the results
   gsOut.compose( "  %C) Assign (const uint8_t* : bytes:%S source)", &wchar, gsVal1.gstr() ) ;
   wcout << gsOut << endl ;
   ofs   << gsOut << endl ;
   gString gsAb ;
   gsAb = (uint8_t*)(cbuff) ;             // (lie about the data type)
   capChars = gsAb.gschars() ;
   capBytes = gsAb.utfbytes() ;
   capCbuff = gsAb.wAlloc() ;
   capBbuff = gsAb.uAlloc() ;
   freeSp   = gsAb.freeSpace() ;
   gsVal1.formatInt( capChars, 9, true ) ;
   gsVal2.formatInt( capBytes, 9, true ) ;
   gsVal3.formatInt( capCbuff, 9, true ) ;
   gsVal4.formatInt( capBbuff, 9, true ) ;
   gsOut.compose( "            (initialized    : chars:%S bytes:%S storage:%S words and %S bytes freeSp:%d)",
                  gsVal1.gstr(), gsVal2.gstr(), gsVal3.gstr(), gsVal4.gstr(), &freeSp ) ;
   wcout << gsOut << endl ;
   ofs   << gsOut << endl ;
   delete [] cbuff ;                      // release the dynamic allocation

   //****************************
   //* Assignment: wchar_t Data *
   //****************************
   ++wchar ;
   gsscnt  = gsSrc.gschars() - 1 ;        // source characters
   trgCnt = gsALLOCMAX + gsscnt ;         // larger than word buffer
   wbuff = new wchar_t[trgCnt + trgPad] ;
   charCnt = ZERO ;
   do
   {
      gsSrc.copy( &wbuff[charCnt], gsSrc.gschars() ) ;
      charCnt += gsscnt ;
   }
   while ( charCnt < trgCnt ) ;
   wbuff[charCnt++] = L'!' ;               // mark the end of source text
   wbuff[charCnt++] = L'\0' ;              // terminate the string
   gsVal1.formatInt( charCnt, 9, true ) ;  // report the results
   gsOut.compose( "  %C) Assign (const wchar_t* : chars:%S source)", &wchar, gsVal1.gstr() ) ;
   wcout << gsOut << endl ;
   ofs   << gsOut << endl ;
   gString gsAw ;
   gsAw = wbuff ;
   capChars = gsAw.gschars() ;
   capBytes = gsAw.utfbytes() ;
   capCbuff = gsAw.wAlloc() ;
   capBbuff = gsAw.uAlloc() ;
   freeSp   = gsAw.freeSpace() ;
   gsVal1.formatInt( capChars, 9, true ) ;
   gsVal2.formatInt( capBytes, 9, true ) ;
   gsVal3.formatInt( capCbuff, 9, true ) ;
   gsVal4.formatInt( capBbuff, 9, true ) ;
   gsOut.compose( "            (initialized    : chars:%S bytes:%S storage:%S words and %S bytes freeSp:%d)",
                  gsVal1.gstr(), gsVal2.gstr(), gsVal3.gstr(), gsVal4.gstr(), &freeSp ) ;
   wcout << gsOut << endl ;
   ofs   << gsOut << endl ;
   delete [] wbuff ;                      // release the dynamic allocation

   //*******************************************
   //* Assignment: Formatting Template (char*) *
   //*******************************************
   ++wchar ;
   gsOut = cTemplate ;
   charCnt = gsOut.gschars() + gsGsIn.gschars() ;
   gsVal1.formatInt( charCnt, 9, true ) ;  // report the results
   gsOut.compose( "  %C) Assign (char template  : chars:%S source)", &wchar, gsVal1.gstr() ) ;
   wcout << gsOut << endl ;
   ofs   << gsOut << endl ;
   gString gsAfc ;
   gsAfc.compose( cTemplate, gsGsIn.gstr() ) ;
   capChars = gsAfc.gschars() ;
   capBytes = gsAfc.utfbytes() ;
   capCbuff = gsAfc.wAlloc() ;
   capBbuff = gsAfc.uAlloc() ;
   freeSp   = gsAfc.freeSpace() ;
   gsVal1.formatInt( capChars, 9, true ) ;
   gsVal2.formatInt( capBytes, 9, true ) ;
   gsVal3.formatInt( capCbuff, 9, true ) ;
   gsVal4.formatInt( capBbuff, 9, true ) ;
   gsOut.compose( "            (initialized    : chars:%S bytes:%S storage:%S words and %S bytes freeSp:%d)",
                  gsVal1.gstr(), gsVal2.gstr(), gsVal3.gstr(), gsVal4.gstr(), &freeSp ) ;
   wcout << gsOut << endl ;
   ofs   << gsOut << endl ;

   //*******************************************
   //* Assignment: Formatting Template (char*) *
   //*******************************************
   ++wchar ;
   // See note above about swprintf() limitations.
   const wchar_t* wTemplate = 
      L"This is a template including gsWordIn characters: \"%S\" to over-fill buffer." ;
   gsOut = wTemplate ;
   charCnt = gsOut.gschars() + gsGsIn.gschars() ;
   gsVal1.formatInt( charCnt, 9, true ) ;  // report the results
   gsOut.compose( "  %C) Assign (wchart template: chars:%S source)", &wchar, gsVal1.gstr() ) ;
   wcout << gsOut << endl ;
   ofs   << gsOut << endl ;
   gString gsAfw ;
   gsAfw.compose( wTemplate, gsGsIn.gstr() ) ;
   capChars = gsAfw.gschars() ;
   capBytes = gsAfw.utfbytes() ;
   capCbuff = gsAfw.wAlloc() ;
   capBbuff = gsAfw.uAlloc() ;
   freeSp   = gsAfw.freeSpace() ;
   gsVal1.formatInt( capChars, 9, true ) ;
   gsVal2.formatInt( capBytes, 9, true ) ;
   gsVal3.formatInt( capCbuff, 9, true ) ;
   gsVal4.formatInt( capBbuff, 9, true ) ;
   gsOut.compose( "            (initialized    : chars:%S bytes:%S storage:%S words and %S bytes freeSp:%d)",
                  gsVal1.gstr(), gsVal2.gstr(), gsVal3.gstr(), gsVal4.gstr(), &freeSp ) ;
   wcout << gsOut << endl ;
   ofs   << gsOut << endl ;

}  //* End gsmBufferOverflowA() *

//********************************************************************************
//* Attempt to overflow the data storage buffers by specifying source data       *
//* larger than default allocation.                                              *
//*                                                                              *
//* Insert:                                                                      *
//*    source == char*                                                           *
//*    source == uint8_t*                                                        *
//*    source == wchar_t*                                                        *
//*    source == formatting template ( char and wchar_t )                        *
//*                                                                              *
//* Append:                                                                      *
//*    source == char*                                                           *
//*    source == uint8_t*                                                        *
//*    source == wchar_t*                                                        *
//*    source == formatting template ( char and wchar_t )                        *
//*                                                                              *
//* -- The data used is a small Lorem Ipsum sentence, duplicated as needed to    *
//*    reach the required data size.                                             *
//* -- Note that because Lorem Ipsum is ASCII data the word count would equal    *
//*    the byte count by default. For this reason, we insert the Chinse word     *
//*    "Ma" (horse) into the test data at various points. This exercises the     *
//*    various byte vs. character calculations used during encoding operations.  *
//*                                                                              *
//* Programmer's Note: The 'insert' method takes an optional offset for intertion*
//* point. The offset is zero by default, but a non-zero offset can be specified *
//* to verify that it does not affect the re-allocation. (see 'insOff')          *
//********************************************************************************
void gsmBufferOverflowB ( wchar_t wchar, std::ofstream& ofs )
{
   const char* const LorenWho = "Lorem ipsum dolor sit amet, consectetur "
                                "adipiscing elit, sed do eiusmod tempor "
                                "incididunt ut labore et dolore magna 马aliqua." ;
   gString gsSrc( LorenWho ),             // source data
           gsOut, gsVal1, gsVal2,         // output formatting
           gsVal3, gsVal4, gsVal5 ;
   int charCnt = ZERO,                    // number of characters written to buffer
       byteCnt = ZERO,                    // number of bytes written to byte buffer
       trgCnt,                            // target count
       gsscnt  = gsSrc.utfbytes() - 1,    // characters/bytes block size (not incl. nullchar)
       trgPad  = gsscnt * 2,              // work buffer padding
       insOff  = ZERO,                    // insertion offset
       freeSp,                            // free buffer space after initialization
      capBytes, capChars, capCbuff, capBbuff ; // final values

   //************************
   //* Insert: wchar_t Data *
   //************************
   gsscnt = gsSrc.gschars() - 1 ;         // length of source paragraph (not incl. nullchar)
   trgCnt = gsALLOCMAX + gsscnt ;         // larger than word buffer
   wchar_t* wbuff = new wchar_t[trgCnt + trgPad] ;
   charCnt = ZERO ;
   do
   {
      gsSrc.copy( &wbuff[charCnt], gsSrc.gschars() ) ;
      charCnt += gsscnt ;
   }
   while ( charCnt < trgCnt ) ;
   wbuff[charCnt++] = '!' ;                // mark the end of source text                    
   wbuff[charCnt++] = '\0' ;               // terminate the string
   gsVal1.formatInt( charCnt, 9, true ) ;  // report the results
   gsOut.compose( "  %C) Insert (const wchar_t* : chars:%S source)", &wchar, gsVal1.gstr() ) ;
   wcout << gsOut << endl ;
   ofs   << gsOut << endl ;
   gString gsIw( LorenWho ) ;
   gsIw.insert( wbuff, insOff ) ;
   capChars = gsIw.gschars() ;
   capBytes = gsIw.utfbytes() ;
   capCbuff = gsIw.wAlloc() ;
   capBbuff = gsIw.uAlloc() ;
   freeSp   = gsIw.freeSpace() ;
   gsVal1.formatInt( capChars, 9, true ) ;                                               
   gsVal2.formatInt( capBytes, 9, true ) ;
   gsVal3.formatInt( capCbuff, 9, true ) ;
   gsVal4.formatInt( capBbuff, 9, true ) ;
   gsOut.compose( "            (initialized    : chars:%S bytes:%S storage:%S words and %S bytes freeSp:%d)",
                  gsVal1.gstr(), gsVal2.gstr(), gsVal3.gstr(), gsVal4.gstr(), &freeSp ) ;
   wcout << gsOut << endl ;
   ofs   << gsOut << endl ;

   //*********************
   //* Insert: Char Data *
   //*********************
   ++wchar ;
   gsscnt = gsSrc.utfbytes() - 1 ;        // length of source paragraph (not incl. nullchar)
   trgCnt = gsALLOCMAX * 4 + gsscnt ;     // larger than byte buffer
   char* cbuff = new char[trgCnt + trgPad] ;
   do
   {
      gsSrc.copy( &cbuff[byteCnt], gsSrc.utfbytes() ) ;
      byteCnt += gsscnt ;
   }
   while ( byteCnt < trgCnt ) ;
   cbuff[byteCnt++] = '!' ;                // mark the end of source text
   cbuff[byteCnt++] = '\0' ;               // terminate the string
   gsVal1.formatInt( byteCnt, 9, true ) ;  // report the results
   gsOut.compose( "  %C) Insert (const char*    : bytes:%S source)", &wchar, gsVal1.gstr() ) ;
   wcout << gsOut << endl ;
   ofs   << gsOut << endl ;
   gString gsIc( LorenWho ) ;
   gsIc.insert( cbuff, insOff ) ;
   capBytes = gsIc.utfbytes() ;
   capChars = gsIc.gschars() ;
   capCbuff = gsIc.wAlloc() ;
   capBbuff = gsIc.uAlloc() ;
   freeSp   = gsIc.freeSpace() ;
   gsVal1.formatInt( capBytes, 9, true ) ;
   gsVal2.formatInt( capChars, 9, true ) ;
   gsVal3.formatInt( capCbuff, 9, true ) ;
   gsVal4.formatInt( capBbuff, 9, true ) ;
   gsOut.compose( "            (initialized    : bytes:%S chars:%S storage:%S words and %S bytes freeSp:%d)",
                  gsVal1.gstr(), gsVal2.gstr(), gsVal3.gstr(), gsVal4.gstr(), &freeSp ) ;
   wcout << gsOut << endl ;
   ofs   << gsOut << endl ;

   //*********************
   //* Insert: Byte Data *
   //*********************
   ++wchar ;
   gsVal1.formatInt( byteCnt, 9, true ) ;  // report the source data size
   gsOut.compose( "  %C) Insert (const uint8_t* : bytes:%S source)", &wchar, gsVal1.gstr() ) ;
   wcout << gsOut << endl ;
   ofs   << gsOut << endl ;
   gString gsIb( LorenWho ) ;
   gsIb.insert( (uint8_t*)(cbuff), insOff ) ; // (lie about data type)
   capBytes = gsIb.utfbytes() ;
   capChars = gsIb.gschars() ;                                                       
   capCbuff = gsIb.wAlloc() ;                                                        
   capBbuff = gsIb.uAlloc() ;
   freeSp   = gsIb.freeSpace() ;
   gsVal1.formatInt( capBytes, 9, true ) ;
   gsVal2.formatInt( capChars, 9, true ) ;
   gsVal3.formatInt( capCbuff, 9, true ) ;
   gsVal4.formatInt( capBbuff, 9, true ) ;
   gsOut.compose( "            (initialized    : bytes:%S chars:%S storage:%S words and %S bytes freeSp:%d)",
                  gsVal1.gstr(), gsVal2.gstr(), gsVal3.gstr(), gsVal4.gstr(), &freeSp ) ;
   wcout << gsOut << endl ;
   ofs   << gsOut << endl ;

   //**********************************************
   //* Insert: Formatting Specification (wchar_t) *
   //**********************************************
   ++wchar ;
   gsVal1.formatInt( charCnt, 9, true ) ;  // report the source data size
   gsOut.compose( "  %C) Insert (format template: chars:%S source)", &wchar, gsVal1.gstr() ) ;
   wcout << gsOut << endl ;
   ofs   << gsOut << endl ;
   gString gsIfw( LorenWho ) ;
   gsIfw.insert( insOff, L"Inserted data here: >>> %S <<<", wbuff ) ;
   capChars = gsIfw.gschars() ;
   capBytes = gsIfw.utfbytes() ;
   capCbuff = gsIfw.wAlloc() ;
   capBbuff = gsIfw.uAlloc() ;
   freeSp   = gsIfw.freeSpace() ;
   gsVal1.formatInt( capChars, 9, true ) ;                                               
   gsVal2.formatInt( capBytes, 9, true ) ;
   gsVal3.formatInt( capCbuff, 9, true ) ;
   gsVal4.formatInt( capBbuff, 9, true ) ;
   gsOut.compose( "            (initialized    : chars:%S bytes:%S storage:%S words and %S bytes freeSp:%d)",
                  gsVal1.gstr(), gsVal2.gstr(), gsVal3.gstr(), gsVal4.gstr(), &freeSp ) ;
   wcout << gsOut << endl ;
   ofs   << gsOut << endl ;

   //*******************************************
   //* Insert: Formatting Specification (char) *
   //*******************************************
   ++wchar ;
   gsVal1.formatInt( byteCnt, 9, true ) ;  // report the source data size
   gsOut.compose( "  %C) Insert (format template: bytes:%S source)", &wchar, gsVal1.gstr() ) ;
   wcout << gsOut << endl ;
   ofs   << gsOut << endl ;
   gString gsIfc( LorenWho ) ;
   gsIfc.insert( insOff, "Inserted data here: >>> %S <<<", wbuff ) ;
   capBytes = gsIfc.utfbytes() ;
   capChars = gsIfc.gschars() ;
   capCbuff = gsIfc.wAlloc() ;
   capBbuff = gsIfc.uAlloc() ;
   freeSp   = gsIfc.freeSpace() ;
   gsVal1.formatInt( capBytes, 9, true ) ;
   gsVal2.formatInt( capChars, 9, true ) ;                                               
   gsVal3.formatInt( capCbuff, 9, true ) ;
   gsVal4.formatInt( capBbuff, 9, true ) ;
   gsOut.compose( "            (initialized    : bytes:%S chars:%S storage:%S words and %S bytes freeSp:%d)",
                  gsVal1.gstr(), gsVal2.gstr(), gsVal3.gstr(), gsVal4.gstr(), &freeSp ) ;
   wcout << gsOut << endl ;
   ofs   << gsOut << endl ;

   //************************
   //* Append: wchar_t Data *
   //************************
   ++wchar ;
   gsVal1.formatInt( charCnt, 9, true ) ;  // report the source data size
   gsOut.compose( "  %C) Append (const wchar_t* : chars:%S source)", &wchar, gsVal1.gstr() ) ;
   wcout << gsOut << endl ;
   ofs   << gsOut << endl ;
   gString gsAw( LorenWho ) ;
   gsAw.append( wbuff ) ;
   capChars = gsAw.gschars() ;
   capBytes = gsAw.utfbytes() ;
   capCbuff = gsAw.wAlloc() ;
   capBbuff = gsAw.uAlloc() ;
   freeSp   = gsAw.freeSpace() ;
   gsVal1.formatInt( capChars, 9, true ) ;                                               
   gsVal2.formatInt( capBytes, 9, true ) ;
   gsVal3.formatInt( capCbuff, 9, true ) ;
   gsVal4.formatInt( capBbuff, 9, true ) ;
   gsOut.compose( "            (initialized    : chars:%S bytes:%S storage:%S words and %S bytes freeSp:%d)",
                  gsVal1.gstr(), gsVal2.gstr(), gsVal3.gstr(), gsVal4.gstr(), &freeSp ) ;
   wcout << gsOut << endl ;
   ofs   << gsOut << endl ;

   //*********************
   //* Append: Char Data *
   //*********************
   ++wchar ;
   gsVal1.formatInt( byteCnt, 9, true ) ;  // report number of source bytes
   gsOut.compose( "  %C) Append (const char*    : bytes:%S source)", &wchar, gsVal1.gstr() ) ;
   wcout << gsOut << endl ;
   ofs   << gsOut << endl ;
   gString gsAc( LorenWho ) ;
   gsAc.append( cbuff ) ;
   capBytes = gsAc.utfbytes() ;
   capChars = gsAc.gschars() ;
   capCbuff = gsAc.wAlloc() ;
   capBbuff = gsAc.uAlloc() ;
   freeSp   = gsAc.freeSpace() ;
   gsVal1.formatInt( capBytes, 9, true ) ;
   gsVal2.formatInt( capChars, 9, true ) ;
   gsVal3.formatInt( capCbuff, 9, true ) ;
   gsVal4.formatInt( capBbuff, 9, true ) ;
   gsOut.compose( "            (initialized    : bytes:%S chars:%S storage:%S words and %S bytes freeSp:%d)",
                  gsVal1.gstr(), gsVal2.gstr(), gsVal3.gstr(), gsVal4.gstr(), &freeSp ) ;
   wcout << gsOut << endl ;
   ofs   << gsOut << endl ;

   //*********************
   //* Append: Byte Data *
   //*********************
   ++wchar ;
   gsVal1.formatInt( byteCnt, 9, true ) ;  // report number of source bytes
   gsOut.compose( "  %C) Append (const uint_t*  : bytes:%S source)", &wchar, gsVal1.gstr() ) ;
   wcout << gsOut << endl ;
   ofs   << gsOut << endl ;
   gString gsAb( LorenWho ) ;
   gsAb.append( (uint8_t*)cbuff ) ;       // (lie about data type)
   capBytes = gsAc.utfbytes() ;
   capChars = gsAc.gschars() ;
   capCbuff = gsAc.wAlloc() ;
   capBbuff = gsAc.uAlloc() ;
   freeSp   = gsAc.freeSpace() ;
   gsVal1.formatInt( capBytes, 9, true ) ;
   gsVal3.formatInt( capCbuff, 9, true ) ;
   gsVal4.formatInt( capBbuff, 9, true ) ;
   gsOut.compose( "            (initialized    : bytes:%S chars:%S storage:%S words and %S bytes freeSp:%d)",
                  gsVal1.gstr(), gsVal2.gstr(), gsVal3.gstr(), gsVal4.gstr(), &freeSp ) ;
   wcout << gsOut << endl ;
   ofs   << gsOut << endl ;

   //*****************************************
   //* Append: Formatting Template (wchar_t) *
   //*****************************************
   ++wchar ;
   gsVal1.formatInt( charCnt, 9, true ) ;  // report number of source bytes
   gsOut.compose( "  %C) Append (format template: chars:%S source)", &wchar, gsVal1.gstr() ) ;
   wcout << gsOut << endl ;
   ofs   << gsOut << endl ;
   gString gsAfw( LorenWho ) ;
   gsAfw.append( L"Appended data here: >>> %S <<<", wbuff ) ;
   capChars = gsAfw.gschars() ;
   capBytes = gsAfw.utfbytes() ;
   capCbuff = gsAfw.wAlloc() ;
   capBbuff = gsAfw.uAlloc() ;
   freeSp   = gsAfw.freeSpace() ;
   gsVal1.formatInt( capChars, 9, true ) ;                                               
   gsVal2.formatInt( capBytes, 9, true ) ;
   gsVal3.formatInt( capCbuff, 9, true ) ;
   gsVal4.formatInt( capBbuff, 9, true ) ;
   gsOut.compose( "            (initialized    : chars:%S bytes:%S storage:%S words and %S bytes freeSp:%d)",
                  gsVal1.gstr(), gsVal2.gstr(), gsVal3.gstr(), gsVal4.gstr(), &freeSp ) ;
   wcout << gsOut << endl ;
   ofs   << gsOut << endl ;

   //**************************************
   //* Append: Formatting Template (char) *
   //**************************************
   ++wchar ;
   gsVal1.formatInt( byteCnt, 9, true ) ;  // report number of source bytes
   gsOut.compose( "  %C) Append (format template: bytes:%S source)", &wchar, gsVal1.gstr() ) ;
   wcout << gsOut << endl ;
   ofs   << gsOut << endl ;
   gString gsAfc( LorenWho ) ;
   gsAfc.append( "Appended data here: >>> %s <<<", cbuff ) ;
   capBytes = gsAfw.utfbytes() ;
   capChars = gsAfw.gschars() ;
   capCbuff = gsAfw.wAlloc() ;
   capBbuff = gsAfw.uAlloc() ;
   freeSp   = gsAfw.freeSpace() ;
   gsVal1.formatInt( capBytes, 9, true ) ;
   gsVal2.formatInt( capChars, 9, true ) ;                                               
   gsVal3.formatInt( capCbuff, 9, true ) ;
   gsVal4.formatInt( capBbuff, 9, true ) ;
   gsOut.compose( "            (initialized    : bytes:%S chars:%S storage:%S words and %S bytes freeSp:%d)",
                  gsVal1.gstr(), gsVal2.gstr(), gsVal3.gstr(), gsVal4.gstr(), &freeSp ) ;
   wcout << gsOut << endl ;
   ofs   << gsOut << endl ;

   //***********************************
   //* loadChars(append) wchar_t data  *
   //***********************************
   ++wchar ;
   gsVal1.formatInt( charCnt, 9, true ) ;  // report number of source bytes
   gsOut.compose( "  %C) loadChars (append set  : chars:%S source)", &wchar, gsVal1.gstr() ) ;
   wcout << gsOut << endl ;
   ofs   << gsOut << endl ;
   gString gslch( L"Initial Data. " ) ;
   gslch.loadChars( wbuff, gsALLOCMAX, true ) ;
   capChars = gslch.gschars() ;
   capBytes = gslch.utfbytes() ;
   capCbuff = gslch.wAlloc() ;
   capBbuff = gslch.uAlloc() ;
   freeSp   = gslch.freeSpace() ;
   gsVal1.formatInt( capChars, 9, true ) ;                                               
   gsVal2.formatInt( capBytes, 9, true ) ;
   gsVal3.formatInt( capCbuff, 9, true ) ;
   gsVal4.formatInt( capBbuff, 9, true ) ;
   gsOut.compose( "               (initialized : chars:%S bytes:%S storage:%S words and %S bytes freeSp:%d)",
                  gsVal1.gstr(), gsVal2.gstr(), gsVal3.gstr(), gsVal4.gstr(), &freeSp ) ;
   wcout << gsOut << endl ;
   ofs   << gsOut << endl ;

   delete [] wbuff ;                      // release the dynamic allocation
   delete [] cbuff ;                      // release the dynamic allocation

}  //* End gsmBufferOverflowB() *
                                     
//********************************************************************************
//* Automatic storage re-allocation tests, group A.                              *
//* - Data for each test is reported in pairs i.e. before and after reallocation.*
//* - To verify that automatic reallocation has occurred, compare the buffer     *
//*   sizes reported in the first and second report of each pair.                *
//* - Note that "verbose" output verifies that the data were written as expected,*
//*   (whether data truncation has occurred). Verbose output is writte only to   *
//*   the display, and is not copied to the target file.                         *
//********************************************************************************
void gsmTestReallocationA ( wchar_t verbose, std::ofstream& ofs )
{
   const wchar_t* data = L"These data are used to incrementally increase "
                          "stored data to trigger automatic re-sizing of gString." ;

   gString bss( gsALLOCMED ), bsOut, 
           bs( gsALLOCMIN ), bsw( gsALLOCMIN ), bsTrunc( gsALLOCMIN ) ;
   int32_t wchars, bsch = bs.wAlloc() ;

   //* constructor( const char* ) *
   bss.clear() ;
   do
   {
      bss.append( data ) ;
      wchars = bss.gschars() + 6 ;
      bsOut.compose( "%-4d ", &wchars ) ;
      bss.append( "%S\n", bsOut.gstr() ) ;
   }
   while ( wchars <= gsALLOCMIN ) ;
   bsch   = gsALLOCMIN ;
   wchars = 1 ;
   bsOut.compose( "bs capacity:%d   initialize with %4d chars. (constructor(const char*))", &bsch, &wchars ) ;
   wcout << bsOut ; wcout.flush() ;
   ofs   << bsOut ; ofs.flush() ;
   gString bscc( bss.ustr() ) ;
   bsch   = bscc.wAlloc() ;
   wchars = bscc.gschars() ;
   if ( (bsch - wchars) < 4 ) { bsTrunc = " truncated!" ; } else { bsTrunc.clear() ; }
   bsOut.compose( "\nbs capacity:%d  initialized with %4d chars. (constructor(const char*))", &bsch, &wchars ) ;
   wcout << bsOut << bsTrunc ; wcout.flush() ;
   ofs   << bsOut << bsTrunc ; ofs.flush() ;
   if ( verbose == L'v' )  // verbose output
   { wcout << L"Stored Data:\n" << bscc ; wcout.flush() ; }

   //* constructor( const uint8_t* ) *
   bsch   = gsALLOCMIN ;
   wchars = 1 ;
   bsOut.compose( "\nbs capacity:%d   initialize with %4d chars. (constructor(const uint8_t*))", &bsch, &wchars ) ;
   wcout << bsOut ; wcout.flush() ;
   ofs   << bsOut ; ofs.flush() ;
   gString bsuu( (const uint8_t*)(bss.ustr()) ) ;
   bsch   = bsuu.wAlloc() ;
   wchars = bsuu.gschars() ;
   if ( (bsch - wchars) < 4 ) { bsTrunc = " truncated!" ; } else { bsTrunc.clear() ; }
   bsOut.compose( "\nbs capacity:%d  initialized with %4d chars. (constructor(const uint8_t*))", &bsch, &wchars ) ;
   wcout << bsOut << bsTrunc ; wcout.flush() ;
   ofs   << bsOut << bsTrunc ; ofs.flush() ;
   if ( verbose == L'v' )  // verbose output
   { wcout << L"Stored Data:\n" << bsuu ; wcout.flush() ; }

   //* constructor( const wchar_t* ) *
   bsch   = gsALLOCMIN ;
   wchars = 1 ;
   bsOut.compose( "\nbs capacity:%d   initialize with %4d chars. (constructor(const wchar_t*))", &bsch, &wchars ) ;
   wcout << bsOut ; wcout.flush() ;
   ofs   << bsOut ; ofs.flush() ;
   gString bsww( bss.gstr() ) ;
   bsch   = bsww.wAlloc() ;
   wchars = bsww.gschars() ;
   if ( (bsch - wchars) < 4 ) { bsTrunc = " truncated!" ; } else { bsTrunc.clear() ; }
   bsOut.compose( "\nbs capacity:%d  initialized with %4d chars. (constructor(const wchar_t*))", &bsch, &wchars ) ;
   wcout << bsOut << bsTrunc << endl ;
   ofs   << bsOut << bsTrunc << endl ;
   if ( verbose == L'v' )  // verbose output
   { wcout << L"Stored Data:\n" << bsww ; wcout.flush() ; }

   //* operator= UTF-8 data *
   do
   {
      bss.append( data ) ;
      wchars = bss.gschars() + 6 ;
      bsOut.compose( "%-4d ", &wchars ) ;
      bss.append( "%S\n", bsOut.gstr() ) ;
   }
   while ( wchars <= gsALLOCMIN ) ;
   bsch   = bs.wAlloc() ;
   wchars = bss.gschars() ;
   bsOut.compose( "\nbs capacity:%d   initialize with %d chars. (operator= utf-8)", &bsch, &wchars ) ;
   wcout << bsOut ; wcout.flush() ;
   ofs   << bsOut ; ofs.flush() ;
   bs     = bss.ustr() ;   // assignment triggers re-allocation
   bsch   = bs.wAlloc() ;
   wchars = bs.gschars() ;
   if ( (bsch - wchars) < 4 ) { bsTrunc = " truncated!" ; } else { bsTrunc.clear() ; }
   bsOut.compose( "\nbs capacity:%d  initialized with %d chars. (operator= utf-8)", &bsch, &wchars ) ;
   wcout << bsOut << bsTrunc ; wcout.flush() ;
   ofs   << bsOut << bsTrunc ; ofs.flush() ;
   if ( verbose == L'v' )  // verbose output
   { wcout << L"Stored Data:\n" << bs ; wcout.flush() ; }

   //* operator= UTF-32 data *
   bsch   = bsw.wAlloc() ;
   wchars = bss.gschars() ;
   bsOut.compose( "\nbs capacity:%d   initialize with %d chars. (operator= utf-32)", &bsch, &wchars ) ;
   wcout << bsOut ; wcout.flush() ;
   ofs   << bsOut ; ofs.flush() ;
   bsw    = bss.gstr() ;   // assignment triggers re-allocation
   bsch   = bsw.wAlloc() ;
   wchars = bsw.gschars() ;
   if ( (bsch - wchars) < 4 ) { bsTrunc = " truncated!" ; } else { bsTrunc.clear() ; }
   bsOut.compose( "\nbs capacity:%d  initialized with %d chars. (operator= utf-32)", &bsch, &wchars ) ;
   wcout << bsOut << bsTrunc ; wcout.flush() ;
   ofs   << bsOut << bsTrunc ; ofs.flush() ;
   if ( verbose == L'v' )  // verbose output
   { wcout << L"Stored Data:\n" << bsw ; wcout.flush() ; }

   //* operator= gString& *
   gString bsmin( gsALLOCMIN ) ;
   bsch   = bsmin.wAlloc() ;
   wchars = bsmin.gschars() ;
   bsOut.compose( "\nbs capacity:%d   initialize with %4d chars. (operator= gString)", &bsch, &wchars ) ;
   wcout << bsOut ; wcout.flush() ;
   ofs   << bsOut ; ofs.flush() ;
   bsmin  = bss ;          // assignment triggers re-allocation
   bsch   = bsmin.wAlloc() ;
   wchars = bsmin.gschars() ;
   if ( (bsch - wchars) < 4 ) { bsTrunc = " truncated!" ; } else { bsTrunc.clear() ; }
   bsOut.compose( "\nbs capacity:%d  initialized with %d chars. (operator= gString)", &bsch, &wchars ) ;
   wcout << bsOut << bsTrunc << endl ;
   ofs   << bsOut << bsTrunc << endl ;
   if ( verbose == L'v' )  // verbose output
   { wcout << L"Stored Data:\n" << bsmin ; wcout.flush() ; }

   //* loadChars( append UTF-8 ) *
   bss = "This sentence will very likely cause the target object to overflow the character buffer." ;
   gString bsLoadAu( gsALLOCMIN ) ;
   do
   {
      bsLoadAu.append( data ) ;
      wchars = bsLoadAu.gschars() + 6 ;
      bsOut.compose( "%-4d ", &wchars ) ;
      bsLoadAu.append( "%S\n", bsOut.gstr() ) ;
   }
   while ( wchars <= (gsALLOCMIN - 96) ) ;
   bsch   = bsLoadAu.wAlloc() ;
   wchars = bsLoadAu.gschars() ;
   bsOut.compose( "\nbs capacity:%d   initialize with %4d chars. (loadChars(append  utf-8))", &bsch, &wchars ) ;
   wcout << bsOut ; wcout.flush() ;
   ofs   << bsOut ; ofs.flush() ;
   wchars = bss.gschars() - 1 ;
   bsLoadAu.loadChars( bss.ustr(), wchars, true ) ;
   bsch   = bsLoadAu.wAlloc() ;
   wchars = bsLoadAu.gschars() ;
   if ( (bsch - wchars) < 4 ) { bsTrunc = " truncated!" ; } else { bsTrunc.clear() ; }
   bsOut.compose( "\nbs capacity:%d  initialized with %d chars. (loadChars(append  utf-8))", &bsch, &wchars ) ;
   wcout << bsOut << bsTrunc ; wcout.flush() ;
   ofs   << bsOut << bsTrunc ; ofs.flush() ;
   if ( verbose == L'v' )  // verbose output
   { wcout << L"Stored Data:\n" << bsLoadAu << endl ; }

   //* loadChars( replace UTF-8 ) *
   gString bsLoadRu( data ) ;
   bsch   = bsLoadRu.wAlloc() ;
   wchars = bsLoadRu.gschars() ;
   bsOut.compose( "\nbs capacity:%d   initialize with %4d chars. (loadChars(replace utf-8))", &bsch, &wchars ) ;
   wcout << bsOut ; wcout.flush() ;
   ofs   << bsOut ; ofs.flush() ;
   bss.clear() ;
   do
   {
      bss.append( data ) ;
      wchars = bss.gschars() + 6 ;
      bsOut.compose( "%-4d ", &wchars ) ;
      bss.append( "%S\n", bsOut.gstr() ) ;
   }
   while ( wchars < (gsALLOCMIN + 120) ) ;
   wchars = bss.gschars() - 1 ;
   bsLoadRu.loadChars( bss.ustr(), wchars, false ) ;
   bsch   = bsLoadRu.wAlloc() ;
   wchars = bsLoadRu.gschars() ;
   if ( (bsch - wchars) < 4 ) { bsTrunc = " truncated!" ; } else { bsTrunc.clear() ; }
   bsOut.compose( "\nbs capacity:%d  initialized with %d chars. (loadChars(replace utf-8))", &bsch, &wchars ) ;
   wcout << bsOut << bsTrunc ; wcout.flush() ;
   ofs   << bsOut << bsTrunc ; ofs.flush() ;
   if ( verbose == L'v' )  // verbose output
   { wcout << L"Stored Data:\n" << bsLoadRu << endl ; }

   //* loadChars( append UTF3-32 ) *
   bss = "This sentence will very likely cause the target object to overflow the character buffer." ;
   gString bsLoadAw( gsALLOCMIN ) ;
   do
   {
      bsLoadAw.append( data ) ;
      wchars = bsLoadAw.gschars() + 6 ;
      bsOut.compose( "%-4d ", &wchars ) ;
      bsLoadAw.append( "%S\n", bsOut.gstr() ) ;
   }
   while ( wchars <= (gsALLOCMIN - 96) ) ;
   bsch   = bsLoadAw.wAlloc() ;
   wchars = bsLoadAw.gschars() ;
   bsOut.compose( "\nbs capacity:%d   initialize with %4d chars. (loadChars(append  utf-32))", &bsch, &wchars ) ;
   wcout << bsOut ; wcout.flush() ;
   ofs   << bsOut ; ofs.flush() ;
   wchars = bss.gschars() - 1 ;
   bsLoadAw.loadChars( bss.gstr(), wchars, true ) ;
   bsch   = bsLoadAw.wAlloc() ;
   wchars = bsLoadAw.gschars() ;
   if ( (bsch - wchars) < 4 ) { bsTrunc = " truncated!" ; } else { bsTrunc.clear() ; }
   bsOut.compose( "\nbs capacity:%d  initialized with %d chars. (loadChars(append  utf-32))", &bsch, &wchars ) ;
   wcout << bsOut << bsTrunc ; wcout.flush() ;
   ofs   << bsOut << bsTrunc ; ofs.flush() ;
   if ( verbose == L'v' )  // verbose output
   { wcout << L"Stored Data:\n" << bsLoadAw << endl ; }

   //* loadChars( replace UTF3-32 ) *
   gString bsLoadRw( data ) ;
   bsch   = bsLoadRw.wAlloc() ;
   wchars = bsLoadRw.gschars() ;
   bsOut.compose( "\nbs capacity:%d   initialize with %4d chars. (loadChars(replace utf-32))", &bsch, &wchars ) ;
   wcout << bsOut ; wcout.flush() ;
   ofs   << bsOut ; ofs.flush() ;
   bss.clear() ;
   do
   {
      bss.append( data ) ;
      wchars = bss.gschars() + 6 ;
      bsOut.compose( "%-4d ", &wchars ) ;
      bss.append( "%S\n", bsOut.gstr() ) ;
   }
   while ( wchars < (gsALLOCMIN + 120) ) ;
   wchars = bss.gschars() - 1 ;
   bsLoadRw.loadChars( bss.gstr(), wchars, false ) ;
   bsch   = bsLoadRw.wAlloc() ;
   wchars = bsLoadRw.gschars() ;
   if ( (bsch - wchars) < 4 ) { bsTrunc = " truncated!" ; } else { bsTrunc.clear() ; }
   bsOut.compose( "\nbs capacity:%d  initialized with %d chars. (loadChars(replace utf-32))", &bsch, &wchars ) ;
   wcout << bsOut << bsTrunc << endl ;
   ofs   << bsOut << bsTrunc << endl ;
   if ( verbose == L'v' )  // verbose output
   { wcout << L"Stored Data:\n" << bsLoadRw << endl ; }

   //** append( wchar_t* ) **
   bss.clear() ;
   do
   {
      bss.append( data ) ;
      wchars = bss.gschars() + 6 ;
      bsOut.compose( "%-4d ", &wchars ) ;
      bss.append( "%S\n", bsOut.gstr() ) ;
   }
   while ( wchars <= (gsALLOCMIN - 96) ) ;
   gString bsapp( bss.gstr() ) ;
   bsch   = bsapp.wAlloc() ;
   wchars = bsapp.gschars() ;
   bsOut.compose( "\nbs capacity:%d   initialize with %4d chars. (append(wchar_t*))", &bsch, &wchars ) ;
   wcout << bsOut ; wcout.flush() ;
   ofs   << bsOut ; ofs.flush() ;

   bss = "Append approximately 100 characters which should trigger "
         "an incremental increase in storage capacity.\n" ;
   bsapp.append( bss.gstr() ) ;     //*** APPEND ***
   bsch   = bsapp.wAlloc() ;
   wchars = bsapp.gschars() ;
   if ( (bsch - wchars) < 4 ) { bsTrunc = " truncated!" ; } else { bsTrunc.clear() ; }
   bsOut.compose( "\nbs capacity:%d  initialized with %4d chars. (append(wchar_t*))", &bsch, &wchars ) ;
   wcout << bsOut << bsTrunc ; wcout.flush() ;
   ofs   << bsOut << bsTrunc ; ofs.flush() ;
   if ( verbose == L'v' )  // verbose output
   { wcout << L"Stored Data:\n" << bsapp ; wcout.flush() ; }

   //** append( char* ) **
   bss.clear() ;
   do
   {
      bss.append( data ) ;
      wchars = bss.gschars() + 6 ;
      bsOut.compose( "%-4d ", &wchars ) ;
      bss.append( "%S\n", bsOut.gstr() ) ;
   }
   while ( wchars <= (gsALLOCMIN - 96) ) ;
   gString bsapc( bss.gstr() ) ;
   bsch   = bsapc.wAlloc() ;
   wchars = bsapc.gschars() ;
   bsOut.compose( "\nbs capacity:%d   initialize with %4d chars. (append(char*))", &bsch, &wchars ) ;
   wcout << bsOut ; wcout.flush() ;
   ofs   << bsOut ; ofs.flush() ;

   bss = "Append approximately 100 characters which should trigger "
         "an incremental increase in storage capacity.\n" ;
   bsapc.append( bss.ustr() ) ;     //*** APPEND ***
   bsch   = bsapc.wAlloc() ;
   wchars = bsapc.gschars() ;
   if ( (bsch - wchars) < 4 ) { bsTrunc = " truncated!" ; } else { bsTrunc.clear() ; }
   bsOut.compose( "\nbs capacity:%d  initialized with %4d chars. (append(char*))", &bsch, &wchars ) ;
   wcout << bsOut << bsTrunc ; wcout.flush() ;
   ofs   << bsOut << bsTrunc ; ofs.flush() ;
   if ( verbose == L'v' )  // verbose output
   { wcout << L"Stored Data:\n" << bsapc << endl ; }

   //** append( const char* fmt, ... ) **
   bss.clear() ;
   do
   {
      bss.append( data ) ;
      wchars = bss.gschars() + 6 ;
      bsOut.compose( "%-4d ", &wchars ) ;
      bss.append( "%S\n", bsOut.gstr() ) ;
   }
   while ( wchars <= (gsALLOCMIN - 96) ) ;
   gString bsfc( bss.gstr() ) ;
   bsch   = bsfc.wAlloc() ;
   wchars = bsfc.gschars() ;
   bsOut.compose( "\nbs capacity:%d   initialize with %4d chars. (append(char* fmt, ...))", &bsch, &wchars ) ;
   wcout << bsOut ; wcout.flush() ;
   ofs   << bsOut ; ofs.flush() ;
   bsfc.append( "There are %d elegant green frogs croaking in the pond until after midnight.", &wchars ) ;
   bsch   = bsfc.wAlloc() ;
   wchars = bsfc.gschars() ;
   if ( (bsch - wchars) < 4 ) { bsTrunc = " truncated!" ; } else { bsTrunc.clear() ; }
   bsOut.compose( "\nbs capacity:%d   initialize with %4d chars. (append(char* fmt, ...))", &bsch, &wchars ) ;
   wcout << bsOut << bsTrunc ; wcout.flush() ;
   ofs   << bsOut << bsTrunc ; ofs.flush() ;
   if ( verbose == L'v' )  // verbose output
   { wcout << L"Stored Data:\n" << bsfc ; wcout.flush() ; }

   //** append( const wchar_t* fmt, ... ) **
   gString bsfw( bss.gstr() ) ;
   bsch   = bsfw.wAlloc() ;
   wchars = bsfw.gschars() ;
   bsOut.compose( "\nbs capacity:%d   initialize with %4d chars. (append(wchar_t* fmt, ...))", &bsch, &wchars ) ;
   wcout << bsOut ; wcout.flush() ;
   ofs   << bsOut ; ofs.flush() ;
   bsfw.append( L"There are %d elegant green frogs croaking in the pond until after midnight.", &wchars ) ;
   bsch   = bsfw.wAlloc() ;
   wchars = bsfw.gschars() ;
   if ( (bsch - wchars) < 4 ) { bsTrunc = " truncated!" ; } else { bsTrunc.clear() ; }
   bsOut.compose( "\nbs capacity:%d   initialize with %4d chars. (append(wchar_t* fmt, ...))", &bsch, &wchars ) ;
   wcout << bsOut << bsTrunc << endl ;
   ofs   << bsOut << bsTrunc << endl ;
   if ( verbose == L'v' )  // verbose output
   { wcout << L"Stored Data:\n" << bsfw << endl ; }

}  //* End gsmTestReallocationA() *

//********************************************************************************
//* Automatic storage re-allocation tests, group B.                              *
//* - Data for each test is reported in pairs i.e. before and after reallocation.*
//* - To verify that automatic reallocation has occurred, compare the buffer     *
//*   sizes reported in the first and second report of each pair.                *
//* - Note that "verbose" output verifies that the data were written as expected,*
//*   (whether data truncation has occurred). Verbose output is writte only to   *
//*   the display, and is not copied to the target file.                         *
//********************************************************************************
void gsmTestReallocationB ( wchar_t verbose, std::ofstream& ofs )
{
   const wchar_t* data = L"These data are used to incrementally increase "
                          "stored data to trigger automatic re-sizing of gString." ;
   int32_t wchars, bsch, coff ;
   gString bss( gsALLOCMED ), bsTrunc( gsALLOCDFLT ) ;
   gString bsOut ;

   //** insert( wchar_t*, offset ) **
   do
   {
      bss.append( data ) ;
      wchars = bss.gschars() + 6 ;
      bsOut.compose( "%-4d ", &wchars ) ;
      bss.append( "%S\n", bsOut.gstr() ) ;
   }
   while ( wchars <= (gsALLOCDFLT - 96) ) ;
   gString bsi( bss.gstr() ) ;
   bsch   = bsi.wAlloc() ;
   wchars = bsi.gschars() ;
   bsOut.compose( "bs capacity:%d   initialize with %4d chars. (insert(wchar_t*, offset))", &bsch, &wchars ) ;
   wcout << bsOut ; wcout.flush() ;
   ofs   << bsOut ; ofs.flush() ;
   coff = bsi.after( L'\n' ) ;
   bss.compose( "Inserted approx. 100 characters at offset %d, which should "
                "trigger an incremental increase in capacity.\n", &coff ) ;
   bsi.insert( bss.gstr(), coff ) ;
   bsch   = bsi.wAlloc() ;
   wchars = bsi.gschars() ;
   if ( (bsch - wchars) < 4 ) { bsTrunc = " truncated!" ; } else { bsTrunc.clear() ; }
   bsOut.compose( "\nbs capacity:%d  initialized with %4d chars. (insert(wchar_t*, offset))", &bsch, &wchars ) ;
   wcout << bsOut << bsTrunc << endl ;
   ofs   << bsOut << bsTrunc << endl ;
   if ( verbose == L'v' )  // verbose output
   { wcout << L"Stored Data:\n" << bsi ; wcout.flush() ; }

   //** replace **
   gString bsrep ;
   do
   {
      bsrep.append( data ) ;
      wchars = bsrep.gschars() + 6 ;
      bsOut.compose( "%-4d ", &wchars ) ;
      bsrep.append( "%S\n", bsOut.gstr() ) ;
   }
   while ( wchars <= (gsALLOCDFLT - 96) ) ;
   bsch   = bsrep.wAlloc() ;
   wchars = bsrep.gschars() ;
   bsOut.compose( "\nbs capacity:%d   initialize with %4d chars. "
                  "(replace(wchar_t*, wchar_t*, offset, casesen, all))", &bsch, &wchars ) ;
   wcout << bsOut ; wcout.flush() ;
   ofs   << bsOut ; ofs.flush() ;

   bsrep.replace( L"data", L"sandwiches", ZERO, true, true ) ;
   bsch   = bsrep.wAlloc() ;
   wchars = bsrep.gschars() ;
   bsOut.compose( "\nbs capacity:%d   initialize with %4d chars. "
                  "(replace( L\"data\", L\"sandwiches\", 0, true, true))", &bsch, &wchars ) ;
   wcout << bsOut << endl ;
   ofs   << bsOut << endl ;
   if ( verbose == L'v' )  // verbose output
   { wcout << L"Stored Data:\n" << bsrep ; wcout.flush() ; }

   //** reAlloc( steps, clear ) **
   int32_t steps = 2, chfree ; bool trnc = false ;
   gString bsr( gsALLOCMED ) ;
   bsr = Chuck ;
   bsr.append( Chuck ) ;
   bsr.append( Chuck ) ;
   bsch   = bsr.wAlloc() ;
   wchars = bsr.gschars() ;
   chfree = bsr.freeSpace() ;
   bsOut.compose( "\nreAlloc ( steps, clear )     capacity:%7d wchars:%4d free:%4d",
                  &bsch, &wchars, &chfree ) ;
   wcout << bsOut ; wcout.flush() ;
   ofs   << bsOut ; ofs.flush() ;
   steps = 2 ; trnc = false ;
   bsOut.compose( "\nreAlloc ( %4d, %6s ) --> ", &steps, bSpec[trnc] ) ;
   bsr.reAlloc( steps, trnc ) ;
   bsch   = bsr.wAlloc() ;
   wchars = bsr.gschars() ;
   chfree = bsr.freeSpace() ;
   bsOut.append( "capacity:%7d wchars:%4d free:%4d", &bsch, &wchars, &chfree ) ;
   wcout << bsOut ; wcout.flush() ;
   ofs   << bsOut ; ofs.flush() ;

   steps = -2 ; trnc = false ;
   bsOut.compose( "\nreAlloc ( %4d, %6s ) --> ", &steps, bSpec[trnc] ) ;
   bsr.reAlloc( steps, trnc ) ;
   bsch   = bsr.wAlloc() ;
   wchars = bsr.gschars() ;
   chfree = bsr.freeSpace() ;
   bsOut.append( "capacity:%7d wchars:%4d free:%4d", &bsch, &wchars, &chfree ) ;
   wcout << bsOut ; wcout.flush() ;
   ofs   << bsOut ; ofs.flush() ;

   steps = 999 ; trnc = false ;
   bsOut.compose( "\nreAlloc ( %4d, %6s ) --> ", &steps, bSpec[trnc] ) ;
   bsr.reAlloc( steps, trnc ) ;
   bsch   = bsr.wAlloc() ;
   wchars = bsr.gschars() ;
   chfree = bsr.freeSpace() ;
   bsOut.append( "capacity:%7d wchars:%4d free:%4d", &bsch, &wchars, &chfree ) ;
   wcout << bsOut ; wcout.flush() ;
   ofs   << bsOut ; ofs.flush() ;

   steps = -999 ; trnc = false ;
   bsOut.compose( "\nreAlloc ( %4d, %6s ) --> ", &steps, bSpec[trnc] ) ;
   bsr.reAlloc( steps, trnc ) ;
   bsch   = bsr.wAlloc() ;
   wchars = bsr.gschars() ;
   chfree = bsr.freeSpace() ;
   bsOut.append( "capacity:%7d wchars:%4d free:%4d", &bsch, &wchars, &chfree ) ;
   wcout << bsOut ; wcout.flush() ;
   ofs   << bsOut ; ofs.flush() ;

   steps = 1 ; trnc = true ;
   bsOut.compose( "\nreAlloc ( %4d, %6s ) --> ", &steps, bSpec[trnc] ) ;
   bsr.reAlloc( steps, trnc ) ;
   bsch   = bsr.wAlloc() ;
   wchars = bsr.gschars() ;
   chfree = bsr.freeSpace() ;
   bsOut.append( "capacity:%7d wchars:%4d free:%4d", &bsch, &wchars, &chfree ) ;
   wcout << bsOut << endl ;
   ofs   << bsOut << endl ;

}  //* End gsmTestReallocationB() *

//********************************************************************************
//* Construct and display the application title.                                 *
//********************************************************************************
int Title ( bool clear )
{
   const char* const TitleTemplate = 
       "gsmeter: version:%s  gString v:%s - (c) %s The Software Samurai" ;

   gString gs ;
   gs.compose( TitleTemplate, appVersion, gs.Get_gString_Version(), crYears ) ;
   if ( clear )
      gs.append( "(test results in: \"%s\")", dbFile ) ;
   gs.append( L'\n' ) ;
   int titWidth = gs.gscols() ;            // return value
   gs.padCols( (titWidth * 2), L'-' ) ;
   if ( clear )
      system ( "clear" ) ;
   wcout << gs << endl ;

   #if DEBUG_GSTRING == 0
   int32_t pad = gs.gscols() / 2 ;
   gs.compose( "*** %S ***", &enableDbgMsgW[9] ) ;
   gs.padCols( pad, L'-', true ) ;
   wcout << gs << endl ;
   #endif   // DEBUG_GSTRING

   return titWidth ;

}  //* End Title() *

//********************************************************************************
//* Set the "locale" for the application from the terminal environment.          *
//********************************************************************************
void setLocale ( const char * localeName )
{
   //* Create a locale object if it does not exist *
   if ( ioLocale == NULL )
      ioLocale = new locale("") ;// get default locale from environment

   //* If a locale name (filename) is specified, overwrite *
   //* the default locale with the specified locale.       *
   if ( (localeName != NULL) && (localeName[ZERO] != '\0') )
   {
      locale tmp(localeName) ;   // create a locale object using specified file
      *ioLocale = tmp ;          // copy it to our global object
   }

   //* Give the locale "global" (application/process) scope *
   ioLocale->global( *ioLocale ) ;

}  //* End setLocale() *

//********************************************************************************
//* Sandbox: Create ad-hoc tests.                                                *
//********************************************************************************
void Sandbox ( wchar_t testchar, std::ofstream& ofs )
{
   #if 0    // Test





   #else    // Default message
   wcout << L"Sandbox - Create your ad-hoc tests here." << endl ;
   #endif   // Default message

}  //* End Sandbox() *

