//******************************************************************************
//* File       : idpp.cpp                                                      *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2014-2015 Mahlon R. Smith, The Software Samurai *
//*                 GNU GPL copyright notice below                             *
//* Date       : 22-Feb-2015                                                   *
//* Version    : (see AppVersion string)                                       *
//*                                                                            *
//* Description: Definitions and data for Infodoc Post-processor (idpp),       *
//* an HTML post-processing utility for use with HTML documents generated from *
//* Texinfo source.                                                            *
//*                                                                            *
//* For full documentation for idpp is located in 'infodoc.info' :             *
//*             info -f infodoc.info -n 'Infodoc Post-processor'               *
//*                                                                            *
//*                                                                            *
//* Copyright Notice:                                                          *
//* -----------------                                                          *
//* This program is free software: you can redistribute it and/or modify it    *
//* under the terms of the GNU General Public License as published by the Free *
//* Software Foundation, either version 3 of the License, or (at your option)  *
//* any later version, PROVIDED THAT the copyright notices for both code and   *
//* documentation are included and unmodified.                                 *
//*                                                                            *
//* This program is distributed in the hope that it will be useful, but        *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
//* for more details.                                                          *
//*                                                                            *
//* You should have received a copy of the GNU General Public License along    *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.            *
//*                                                                            *
//*         Full text of the GPL License may be found in the TexInfo           *
//*         documentation for this program under 'Copyright Notice'.           *
//******************************************************************************
//* Version History (most recent first):                                       *
//*                                                                            *
//* v: 0.0.04 20-Feb-2015                                                      *
//*   - Create a more robust test for end-of-block markers. This reduces the   *
//*     chance that really strange constructs will hide the end-of-block tag,  *
//*     causing a processing error. See ppfEndBlock().                         *
//*   - Create a more robust test for the beginning of <ul> and <ol> lists.    *
//*     Occasional strange constructs were sometimes causing the opening tag   *
//*     to be missed, and therefore the list to be unprocessed.                *
//*     See ppfTestUlistBegin() and ppfTestOlistBegin().                       *
//*   - Bug fix: In ppfProcFormattedBlock(), if source contained one           *
//*     preformatted block nested inside another preformatted block, then      *
//*     processing of the block terminated too soon.                           *
//*   - Lists inside pre-formatted blocks. In the previous release we did not  *
//*     process anything inside a preformatted block. In fact, lists           *
//*     SHOULD NOT be placed inside preformatted blocks, but if they are, we   *
//*     now identify them. Note, however, that we DO NOT process the lists     *
//*     themselves (that would require a second pass through the source HTML), *
//*     but we do compensate for the unfortunate spacing in the generated HTML.*
//*     See ppfProcFormattedBlock() and ppfPFB_List().                         *
//*                                                                            *
//* v: 0.0.03 02-Feb-2015                                                      *
//*   - Bug fix: If <ul> or <ol> list ends with a block construct, then the    *
//*     </ul> or </ol> tag was sometimes being missed.                         *
//*   - Bug fix: If first chapter header is not for 'index-main-menu', then    *
//*     end of TOC was not being found.                                        *
//*   - Bug fix: In ppfProcUL(), if user specified that <ul> lists should not  *
//*     be processed, the disable-processing flag was not being set.           *
//*   - Remove the methods ppfCopyUL() and ppfCopyOL(). These made sense when  *
//*     the algorithm for list processing was unstable, but are no longer      *
//*     necessary.                                                             *
//*   - Allow for a '0' (zero) value in response to the <ol> list processing   *
//*     start value prompt IF the specified enumeration type == 'd' (decimal). *
//*     This is to accomodate the fact that the text of both the GPL and FDL   *
//*     licenses begin with item '0'. Note that the prompt still calls for a   *
//*     value >= 1.                                                            *
//*   - Add a special test for GPL and FDL license text. Process the enumerated*
//*     lists associated with these licenses (unless the '--no_mods' option    *
//*     has been specified). See 'ppfProcGNU' method for details.              *
//*   - Identify and process tables that are nested inside 'indentedblock'     *
//*     blocks.                                                                *
//*                                                                            *
//* v: 0.0.02 01-Feb-2015                                                      *
//*   - First public release.                                                  *
//*   - All documented options are now fully functional except '--css_mods'.   *
//*   - All issues reported by beta-testers through 28 Jan 2015 have been      *
//*     addressed.                                                             *
//*   - All basic functionality seems to be working; but there are a number of *
//*     issues related to Texinfo configuration options which have not yet     *
//*     been investigated.                                                     *
//*   - Channel all user input and display output through a single point for   *
//*     ease in future modifications.                                          *
//*   - Add the '--scan' and '--book' options for debugging only. These provide*
//*     super-verbose output with line numbers to more easily find mis-handled *
//*     HTML source constructs. (tip-of-the-hat to Xiaoxiao on this one :)     *
//*   - Texinfo documentation is relatively complete.                          *
//*   - Building with static libraries has not yet been tested.                *
//*                                                                            *
//* v: 0.0.01 06-Dec-2014                                                      *
//*   - Experimental only. Application structure based on SourceProfiler.      *
//*   - All NcDialog/ncurses code has been temporarily disabled. This will     *
//*     be reinstated when (if) we implement Interactive Mode.                 *
//*                                                                            *
//******************************************************************************
//* Programmer's Notes:                                                        *
//* - The raw HTML mark-up generated by the 'makeinfo' utility is actually     *
//*   pretty good; however, a few things are clunky and backward-looking,      *
//*   while a few things in the displayed output really do look second-rate.   *
//*                                                                            *
//* - There are a number of Texinfo build options that affect the HTML output. *
//*   The use of some of them may cause the early releases of idpp to report   *
//*   parsing errors or to incorrectly parse and process the data.             *
//*   These build options will be investigated as time permits.                *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *
//* =================                                                          *
//* == To-do List: ==                                                          *
//* =================                                                          *
//* - command-line: --css_mods (modify the CSS definition file)                *
//*                   Overrides all other options except --help and --version  *
//*                   - Add a method to modify the whole-document definitions  *
//*                     in the CSS file. Uses the '-f' option for the target   *
//*                     filename if specified, else the default definition     *
//*                     file 'infodoc-styles.css'.                             *
//*                     - background color                                     *
//*                     - foreground color                                     *
//*                     - base font size                                       *
//*                     - resize width of infodoc_container class              *
//*                     - enable/disable container border                      *
//* - command-line: -L Insert a Link (Back-to-Doc-Page) at top and bottom of   *
//*    container. This might or might not be generalized for inserting any     *
//*    object (image?) at top and/or bottom of visible document.               *
//*                                                                            *
//* - Decide what to do about the texi-to-HTML converter's incorrect formatting*
//*   of the lists in the GPL and FDL license text.                            *
//*                                                                            *
//* - Research other Texinfo customization variables.                          *
//*   - At least SOME of the HTML-only build options in makeinfo cause invalid *
//*     parsing of the document. (BEFORE_TOC, etc.)                            *
//* - Research how images (or other objects) are embedded in the HTML output.  *
//*   There is a good chance that it's not handled very well.                  *
//* - Research how footnotes are generated in the HTML output.                 *
//* - Verify that user is not prompted for a response if the file has already  *
//*   been processed for response-worthy objects. I.E. If a table already has  *
//*   a border, don't ask again. If an <ol> list already has a class, don't    *
//*   ask again.                                                               *
//* - Add a chapter for instructions on installing info document into the      *
//*   info system?                                                             *
//* - If TOC is removed, then we should also remove the '[Contents]' links.    *
//*                                                                            *
//* - test second-pass scenarios for robustness                                *
//* - The @smallformat followed by a @verbatim block suffers from the same     *
//*   extra whitespace as the preformatted block sequences. Not sure if it is  *
//*   worth the code and performance hit to check for it. Think about this.    *
//*                                                                            *
//* - Remember to insert the following at the top of infodoc_css.html          *
//*   <img style="margin:0px 0px 0px 1082px; border:0;width:88px;height:31px;" 
//*   src="vcss-new.gif" title="W3C Validated CSS3!" alt="Valid CSS3!"/>
//*                                                                            *
//* - Interactive Mode implementation                                          *
//*   Is this worth the effort?                                                *
//*                                                                            *
//* - infodoc-styles.css                                                       *
//*   - Note that the @shortcontents command generates:                        *
//*      <h2 class="shortcontents-heading">Short Table of Contents</h2>        *
//*     and a:                                                                 *
//*      <div class="shortcontents">                                           *
//*     Neither of these classes are defined in our CSS file. This is not      *
//*     critical because it is just an unnumbered list, but we may want to add *
//*     these definitions.                                                     *
//*   - Note that the SIMPLE_MENU customization variable generates:            *
//*     <div class="menu"><pre class="menu-preformatted">                      *
//*                                                                            *
//*                                                                            *
//******************************************************************************

//****************
//* Header Files *
//****************
#include "idpp.hpp"

//****************
//* Local Data   *
//****************
extern const wchar_t* dToken ;
const wchar_t* dfltTargPath = L"#" ;
const short    dfltTargPath_len = 2 ;
const wchar_t* dfltTargText = L"(top)" ;
const short    dfltTargText_len = 6 ;


//*************************
//*         main          *
//*************************
//******************************************************************************
//* Program entry point.                                                       *
//*                                                                            *
//* Command-line Usage:  See GetCommandLineArgs() below for command-line       *
//*                      argument processing.                                  *
//*                                                                            *
//* Returns: OK  (0)  if all processing completed successfully                 *
//*          ERR (-1) if                                                       *
//*                   a) one or more command-line arguments invalid            *
//*                   b) one or more specified source files not found,         *
//*                      inaccessible, or not valid HTML                       *
//*                   c) CSS definition file not found or corrupted            *
//*                   d) processing error                                      *
//*                   e) user aborted the operation                            *
//******************************************************************************

int main ( int argc, char* argv[], char* argenv[] )
{
   //* Gather our entry-sequence data *
   commArgs clArgs( argc, argv, argenv ) ;

   //* Create the application class object *
   Idpp idpp( clArgs ) ;

   return ( int(idpp.ProcStatus()) ) ;

}  //* End main() *

//*************************
//*         ~Idpp         *
//*************************
//******************************************************************************
//* Destructor. Return all resources to the system.                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

Idpp::~Idpp ( void )
{

   // NOTHING TO DO AT THIS TIME

}  //* End ~Idpp() *

//*************************
//*         Idpp          *
//*************************
//******************************************************************************
//* Default constructor.                                                       *
//*                                                                            *
//* Input  : commArgs class object (by reference)                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

Idpp::Idpp ( commArgs& ca )
{
   //* Set the locale to the environment's choice i.e. UTF-8 encoding.*
   //* This affects all input and output streams.                     *
   std::locale::global(std::locale(""));

   //* Initialize our data members *
   this->sfCount = this->emCount = ZERO ;
   this->slCount = this->tlCount = ZERO ; // (unsigned values)
   this->textMode = true ; // Text-only is the default operational mode
   this->allFiles  = this->oLists  = this->oOffst  = 
   this->tocMod    = this->tocDel  = this->upTarg  = this->tabBorder = 
   this->tabPrompt = this->verbose = this->css_mod = this->my_meta = 
   this->no_mods   = this->no_doct = this->no_utrg = this->no_body = 
   this->no_bull   = this->no_bloc = this->no_auth = this->no_meta = 
   this->no_link   = this->no_cont = this->no_cart = false ;
   this->cssFile = L"infodoc-styles.css" ;

   //* Initialize default "Up" target link path and display text *
   wcsncpy ( this->upTargPath, dfltTargPath, dfltTargPath_len ) ;
   wcsncpy ( this->upTargText, dfltTargText, dfltTargText_len ) ;

   this->ppfGetCWD ( this->cwDir ) ;   // get path of current-working directory

   //* Initialize debugging option parameters *
   this->scan_beg = ZERO ; this->scan_end = 0xFFFF ; this->scan = false ;
   this->book = false ;

   //* User may have specified one or more command-line arguments *
   bool validArgs = this->GetCommandLineArgs ( ca ) ;

   //* If '--up_target' option specified, parse the input string *
   //* into target-path and display-text segments.               *
   if ( this->upTarg != false )
   {
      for ( short i = ZERO ; this->upTargPath[i] != NULLCHAR ; i++ )
      {
         if ( this->upTargPath[i] == COMMA )
         {
            this->upTargPath[i++] = NULLCHAR ;
            if ( this->upTargPath[i] != NULLCHAR )
            {
               gString gsx( &this->upTargPath[i], TARG_TEXT_LEN ) ;
               gsx.copy( this->upTargText, TARG_TEXT_LEN ) ;
            }
            break ;
         }
      }
   }

   //* If valid user input, and not a cry for help *
   if ( (validArgs != false) && (ca.helpFlag == false) && (ca.verFlag == false) ) 
   {
      //* Process data in text mode. *
      if ( ca.textFlag != false )
      {
         this->procStatus = this->TextMode () ;
      }
   
      //* Process data in interactive mode *
      else
      {
         this->procStatus = this->InteractiveMode () ;
      }
   }           // valid user input

   else           // explicit or implied cry for help
   {
      if ( ca.verFlag != false )
         this->DisplayAppVersion () ;
      else if ( (ca.helpFlag != false) || (this->emCount == ZERO) )
         this->DisplayCommandLineHelp () ;
      else     // display accumulated command-line errors
      {
         gString gsOut ;
         gsOut.compose( L"\n%S%S%S\n%S", AppTitle1, AppVersion, AppTitle2, AppTitle3 ) ;
         this->textOut ( gsOut ) ;
         for ( short i = ZERO ; i < this->emCount ; i++ )
            this->textOut ( this->ErrorMsg[i] ) ;
         this->textOut ( "\nFor more information: 'idpp --help'\n" ) ;
      }
   }

}  //* End Idpp() *

//*************************
//*       TextMode        *
//*************************
//******************************************************************************
//* Process the specified HTML documents in text-only mode.                    *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: OK  if all files processed successfully                           *
//*          ERR if processing error(s) or user aborted operation              *
//******************************************************************************

short Idpp::TextMode ( void )
{
   gString gsSrc,          // source path/filename
           gsTrg,          // target path/filename
           gsName,         // misc. filename
           gsOut ;         // message output
   short   status = OK ;   // return value

   //* Display the application title, CWD, and CSS filename/version *
   gsOut.compose( L"\n%S%S%S\n%S", AppTitle1, AppVersion, AppTitle2, AppTitle3 ) ;
   this->textOut ( gsOut ) ;
   gsOut.compose( L"CWD: %S", this->cwDir.gstr() ) ;
   this->textOut ( gsOut ) ;
   gsOut.compose( L"CSS: %S", this->cssFile.gstr() ) ;
   this->textOut ( gsOut ) ;
   gsOut.compose( L"CSS: %S", this->cssVersion ) ;
   this->textOut ( gsOut ) ;

   //* Process each file in list *
   for ( short i = ZERO ; (i < this->sfCount) && (status == OK) ; i++ )
   {  //* Declare the file being processed *
      gsOut.compose( L">>> %S", this->srcFiles[i] ) ;
      this->textOut ( gsOut ) ;

      //* If processing is enabled *
      if ( this->no_mods == false )
      {
         //* Create full path/filename for source(backup) file *
         this->ppfCatPathFilename ( gsTrg, this->cwDir, this->srcFiles[i] ) ;
         gsSrc.compose( L"%S~", gsTrg.gstr() ) ;
         this->ppfExtractFilename ( gsName, gsSrc ) ;
         gsOut.compose( L"    src: %S", gsName.gstr() ) ;
   
         //* If existing backup file, delete it *
         if ( this->ppfTargetExists ( gsSrc ) )
         {
            status = this->ppfDeleteFile ( gsSrc ) ;
            if ( this->verbose || status != OK )
               gsOut.append( L"  (delete existing backup)" ) ;
            if ( status != OK )
               gsOut.append( L" FAILED!" ) ;
            this->textOut ( gsOut ) ;
         }

         //* Rename original file as backup file *
         if ( status == OK )
         {
            this->ppfExtractFilename ( gsName, gsTrg ) ;
            gsOut.compose( L"    trg: %S", gsName.gstr() ) ;
            status = this->ppfRenameFile ( gsTrg, gsSrc ) ;
            if ( this->verbose || status != OK )
               gsOut.append( L"   (backup source file)" ) ;
            if ( status != OK )
               gsOut.append( L" FAILED!" ) ;
            this->textOut ( gsOut ) ;
         }
         gsOut.clear() ; this->textOut( gsOut ) ;
      }
      //* Else, we are performing a scan-only pass *
      else
      {
         //* Create full path/filename for source file *
         this->ppfCatPathFilename ( gsSrc, this->cwDir, this->srcFiles[i] ) ;
         this->ppfExtractFilename ( gsName, gsSrc ) ;
         gsOut.compose( L"    src: %S\n"
                         "    trg: (none)", gsName.gstr() ) ;
         this->textOut ( gsOut ) ;
         gsTrg.clear() ;      // target set to null string
      }

      //* Process source file, creating new target file *
      if ( status == OK )
      {
         if ( (status = ppfProcessSrcHTML ( gsSrc, gsTrg )) == OK )
         {
            this->textOut ( L"    Conversion successful!" ) ;
            if ( this->verbose )
            {
               gsOut.compose( L"    (%4hu source lines processed and )", 
                              &this->slCount ) ;
               if ( this->no_mods != false )
               {
                  gsOut.limitCharacters( gsOut.gschars() - 6 ) ;
                  gsOut.append( L')' ) ;
               }
               this->textOut ( gsOut ) ;
               if ( this->no_mods == false )
               {
                  gsOut.compose( L"    (%4hu lines written to target    )", 
                                 &this->tlCount ) ;
                  this->textOut ( gsOut ) ;
               }
            }
         }
         else
         {
            gsOut.compose( L"    Processing error on source line: %hu\n"
                            "    Conversion failed: STOP!", &this->slCount ) ;
            this->textOut ( gsOut ) ;
         }
         gsOut.clear() ; this->textOut( gsOut ) ;
      }
      else  // preparing to exit
         { gsOut.clear() ; this->textOut( gsOut ) ; }
   }
   return status ;

}  //* End TextMode() *

//*************************
//*    InteractiveMode    *
//*************************
//******************************************************************************
//* Process the specified HTML documents in interactive mode.                  *
//* NOTE: This is just a stub. When interactive mode is implemented, it will   *
//*       live in its own source module.                                       *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: OK  if all files processed successfully                           *
//*          ERR if processing error(s) or user aborted operation              *
//******************************************************************************

short Idpp::InteractiveMode ( void )
{
   this->textMode = false ;      // indicate operational mode
   short status = OK ;

/* TEMP - NOT YET IMPLEMENTED */ status = ERR ;

   return status ;

}  //* End InteractiveMode() *

//*************************
//*       textOut         *
//*************************
//******************************************************************************
//* All, or nearly all text written to the display goes through this method,   *
//* so we can control whether it goes through stdout (wide stream) or through  *
//* the NcDialog output stream.                                                *
//*                                                                            *
//* Input  : tOut   : text data to be displayed                                *
//*          newln  : (optional, true by default)                              *
//*                   if 'true', then terminate the line with an 'endl'        *
//*                   if 'false', do not termnate the line                     *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note: We channel as much of the stdout stream as possible     *
//* through this method to avoid future effort if we later add the NcDialog    *
//* (ncurses) interface which is incompatible with stdin/stdout.               *
//******************************************************************************

void Idpp::textOut ( const gString& tOut, bool newln )
{
   if ( this->textMode != false )
   {
      wcout << tOut.gstr() ;
      if ( newln != false )
         wcout << endl ;
   }
   else
   {
      /* Interactive Mode not yet implemented. */
   }

}  //* End textOut() *
void Idpp::textOut ( const char* tOut, bool newln )
{
   gString gs( tOut ) ;
   this->textOut ( gs, newln ) ;

}  //* End textOut() *
void Idpp::textOut ( const wchar_t* tOut, bool newln )
{
   gString gs( tOut ) ;
   this->textOut ( gs, newln ) ;

}  //* End textOut() *

//*************************
//*     userResponse      *
//*************************
//******************************************************************************
//* Get user response to text-mode prompts.                                    *
//*                                                                            *
//* Input  : gsIn   : (by reference) receives user input                       *
//*                                                                            *
//* Returns: OK  if data successfully received                                 *
//*          ERR if end of input script encountered, indicating that caller    *
//*              should re-prompt the user. (gsIn will contain an empty string)*
//******************************************************************************
//* Programmer's Notes:                                                        *
//* - User is prompted to enter a response for selecting a class type and      *
//*   optionally, a start index for <ol> (enumerated list) tags.               *
//* - User is prompted to enter a response for selecting whether to place a    *
//*   border around a <table> object.                                          *
//* - Direct user input is straightforward: the interface expects character or *
//*   string input, terminated by an ENTER key.                                *
//* - Input may be redirected from a response file, for which we define a      *
//*   placeholder string: 'default_token'.                                     *
//* - We wait indefinitely for user input.                                     *
//* - If the first character of the response is the :'#' (hash) character,     *
//*   then assume that the token is a comment and ignore it.                   *
//* - First character of input must not be an ASCII control character.         *
//*   (later, we may want to enforce a _printing_ character as the first)      *
//*                                                                            *
//* - If the token comes from a response file, then the shell does not echo it *
//*   to the display. This throws off the formatting of the prompts.           *
//*   Unfortunately, we can't tell whether the token came from the keyboard or *
//*   from a response file--they look the same after buffering. So, we output  *
//*   an extra linefeed after a valid token.                                   *
//******************************************************************************

short Idpp::userResponse ( gString& gsIn )
{
   wchar_t userInput[gsMAXCHARS] = L"" ;
   bool  done = false ;             // loop control
   short status = OK ;              // return value

   gsIn.clear() ;             // clear caller's old data

   //* If we are running in scan-only mode, don't prompt for responses. *
   //* Instead, just return the 'default_token' string.                 *
   if ( this->no_mods != false )
   {
      gsIn = dToken ;
      this->textOut ( gsIn ) ;
      done = true ;
   }

   while ( ! done )
   {
      wcin >> userInput ;     // get user response

      //* Verify that input is not a control character and not a comment *
      if ( *userInput >= SPACE && *userInput != L'#' )
      {
         gsIn = userInput ;   // copy response to caller's buffer
         this->textOut ( L"" ) ;
         break ;
      }
   }
   return status ;

}  //* End userResponse() *

//*************************
//*  GetCommandLineArgs   *
//*************************
//******************************************************************************
//* Capture user's command-line arguments.                                     *
//*                                                                            *
//* Valid Arguments: (see DisplayCommandLineHelp())                            *
//*                                                                            *
//* Input  : commArgs class object (by reference)                              *
//*                                                                            *
//* Returns: 'true' if all arguments are valid                                 *
//*          'false' if invalid argument(s)                                    *
//******************************************************************************
//* Text Mode:                                                                 *
//*  a) must be at least one filename specified (16 names max)                 *
//*  b) all command switches are optional                                      *
//*  c) switches with string arguments may be specified as:                    *
//*     - string immediately follows:  idpp -dpublic_html                      *
//*     - string is next argument   :  idpp -d public_html                     *
//*     - string follows '=' sign   :  idpp -d=public_html                     *
//*  d) switches with sub-switch arguments                                     *
//*     - sub-switches follow immediately and in any order                     *
//*     - Note: no sub-switches defined at this time.                          *
//*  e) -h, -H, -? or arg error overrides all switches except --version        *
//*  f) --version overrides all other switches                                 *
//*                                                                            *
//* Interactive Mode:                                                          *
//*  a) all arguments except 'i' are optional                                  *
//*                                                                            *
//* -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  *
//* Undocumented options:                                                      *
//* =====================                                                      *
//* --scan=start,end                                                           *
//*   As lines are read from source print them to the display.                 *
//*   Optional arguments: starting source line number and ending source line   *
//*   number, separated by a comma ','. (See ppfReadSrcLine method)            *
//*   a) If no arguments, then display all lines read.                         *
//*      --scan                                                                *
//*   b) If only first argument, then it is start. Continue scan to the end.   *
//*      --scan=400                                                            *
//*   c) If both arguments, then display all lines in the range, inclusive.    *
//*      --scan=400,525                                                        *
//*                                                                            *
//* --book                                                                     *
//*   For the main processing loop (ppfProcessSrcHTML method), display a       *
//*   message when a sub-process method is called and another message when it  *
//*   returns. This helps to locate the place in the source document where the *
//*   processing logic has gone off into the weeds, i.e. if the end of the     *
//*   processing block is not properly identified by the sub-process method.   *
//*                                                                            *
//******************************************************************************

bool Idpp::GetCommandLineArgs ( commArgs& ca )
{
#define DEBUG_GCLA (0)        // for debugging only
#define ENABLE_IMODE (0)      // enable/disable interactive-mode switch

   gString gs,                   // data formatting
           gsOut ;               // user messages
   const short argTest_len = 9 ; // number of chararacters to compare
   bool status = true ;          // return value

   if ( ca.argCount > 1 )
   {
      short j = ZERO ;
      bool multiarg = false ;
      for ( short i = 1 ; (i < ca.argCount) || (multiarg != false) ; i++ )
      {
         if ( multiarg != false )   // if additional switches in same argument
            --i ;
         else
            j = ZERO ;

         //* If a command-line switch OR continuing previous switch argument *
         if ( ca.argList[i][j] == DASH || multiarg != false )
         {
            multiarg = false ;
            ++j ;
            if ( ca.argList[i][j] == DASH ) // (double dash)
            {  //* Long-form command switches *
               bool goodarg = false ;

               //* Test for "up target" specification *
               if ( ca.argList[i][++j] == 'u' )
               {
                  gs = ca.argList[i] ;
                  if ( (gs.compare( L"--up_target", argTest_len ) == ZERO) )
                  {
                     for ( short idx = argTest_len ; gs.gstr()[idx] != NULLCHAR ; idx++ )
                     {
                        if ( (gs.gstr()[idx]) == L'=' )
                        {
                           gsOut = &gs.gstr()[idx + 1] ;
                           gsOut.copy( this->upTargPath, gsMAXCHARS ) ;
                           goodarg = this->upTarg = true ;
                           break ;
                        }
                     }
                  }
               }

               //* Request to place borders around table objects *
               else if ( ca.argList[i][j] == 't' )
               {
                  gs = ca.argList[i] ;
                  if ( (gs.compare( L"--table_border", argTest_len ) == ZERO) )
                  {
                     goodarg = this->tabBorder = true ;
                     for ( short idx = argTest_len ; gs.gstr()[idx] != NULLCHAR ; idx++ )
                     {
                        if ( (gs.gstr()[idx] == L'=') && (gs.gstr()[idx + 1] != NULLCHAR) )
                        {
                           ++idx ;
                           if ( (gs.compare( L"specify", 4, idx )) == ZERO )
                              this->tabPrompt = true ;
                           else if ( (gs.compare( L"all", 3, idx )) != ZERO )
                              goodarg = false ;
                           break ;
                        }
                     }
                  }
               }

               //* Request to modify the whole-document CSS options *
               else if ( ca.argList[i][j] == 'c' )
               {  /* NOT YET IMPLEMENTED */
                  gs = ca.argList[i] ;
                  if ( (gs.compare( L"--css_mods", argTest_len ) == ZERO) )
                  {
                     goodarg = this->css_mod = true ;

                     #if 1    // TEMP TEMP TEMP
                     if ( this->emCount < emMAX )
                     {
                        gsOut = "Sorry, interactive modification of CSS definitions "
                                "not yet implemented." ;
                        gsOut.copy( this->ErrorMsg[this->emCount++], gsMAXCHARS ) ;
                     }
                     status = false ;
                     break ;
                     #endif   // TEMP TEMP TEMP
                  }

               }

               //* Request to insert custom data into <head> section *
               else if ( ca.argList[i][j] == 'm' )
               {
                  gs = ca.argList[i] ;
                  if ( (gs.compare( L"--my_metadata", argTest_len ) == ZERO) )
                  {
                     goodarg = this->my_meta = true ; // hope for the best
                     status = false ;                 // prepare for the worst
                     short idx = argTest_len ;
                     for ( ; gs.gstr()[idx] != NULLCHAR ; idx++ )
                     {
                        if ( (gs.gstr()[idx] == L'=') && (gs.gstr()[idx + 1] != NULLCHAR) )
                        {
                           ++idx ;
                           this->userMeta = &gs.gstr()[idx] ;
                           if ( ((this->ppfRealpath ( gsOut, this->userMeta )) == OK)
                                && (this->ppfTargetExists ( gsOut )) )
                           {  //* Save the real path/filename spec *
                              this->userMeta = gsOut ;
                              status = true ;
                           }
                           else
                              this->userMeta.clear() ;
                           break ;
                        }
                     }
                     if ( (this->userMeta.gschars() == 1) &&
                          (this->emCount < emMAX) )
                     {
                        gsOut.compose( "Error! Metadata file '%S'"
                                       " not found!", 
                                       &gs.gstr()[idx] ) ;
                        gsOut.copy( this->ErrorMsg[this->emCount++], gsMAXCHARS ) ;
                     }
                  }
               }

               //* Request for application version number *
               else if ( ca.argList[i][j] == 'v' )
               {  //* Most Linux console apps respond to a       *
                  //* '--version' request in a cannonical manner.*
                  //* Overrides everything else on comand line.  *
                  gs = ca.argList[i] ;
                  if ( (gs.compare( L"--version", argTest_len ) == ZERO) )
                  {
                     ca.reset() ;
                     ca.helpFlag = ca.verFlag = true ;
                     #if DEBUG_GCLA != 0
                     this->textOut ( L"--version: specified" ) ;
                     #endif   // DEBUG_GCLA
                     break ;
                  }
               }

               //* Long-form request for Help *
               else if ( ca.argList[i][j] == 'h' || ca.argList[i][j] == 'H' )
               {
                  ca.helpFlag = true ;
                  #if DEBUG_GCLA != 0
                  this->textOut ( L"--help: specified" ) ;
                  #endif   // DEBUG_GCLA
               }

               //* Exceptions to default processing *
               else if ( ca.argList[i][j] == 'n' )
               {
                  gs = ca.argList[i] ;
                  if ( (gs.compare( L"--no_mods", argTest_len ) == ZERO) )
                     goodarg = this->no_mods = this->verbose = true ;
                  else if ( (gs.compare( L"--no_doct", argTest_len ) == ZERO) )
                     goodarg = this->no_doct = true ;
                  else if ( (gs.compare( L"--no_uplink", argTest_len ) == ZERO) )
                     goodarg = this->no_utrg = true ;
                  else if ( (gs.compare( L"--no_body", argTest_len ) == ZERO) )
                     goodarg = this->no_body = true ;
                  else if ( (gs.compare( L"--no_bull", argTest_len ) == ZERO) )
                     goodarg = this->no_bull = true ;
                  else if ( (gs.compare( L"--no_meta", argTest_len ) == ZERO) )
                     goodarg = this->no_meta = true ;
                  else if ( (gs.compare( L"--no_link", argTest_len ) == ZERO) )
                     goodarg = this->no_link = true ;
                  else if ( (gs.compare( L"--no_bloc", argTest_len ) == ZERO) )
                     goodarg = this->no_bloc = true ;
                  else if ( (gs.compare( L"--no_auth", argTest_len ) == ZERO) )
                     goodarg = this->no_auth = true ;
                  else if ( (gs.compare( L"--no_cont", argTest_len ) == ZERO) )
                     goodarg = this->no_cont = true ;
                  else if ( (gs.compare( L"--no_cart", argTest_len ) == ZERO) )
                     goodarg = this->no_cart = true ;
               }

               //* Undocumented '--scan' option *
               else if ( ca.argList[i][j] == 's' )
               {
                  gs = ca.argList[i] ;
                  if ( (gs.compare( L"--scan", 6 )) == ZERO )
                  {
                     goodarg = this->scan = true ;
                     USHORT b, e ;
                     int args = swscanf ( &gs.gstr()[6], L"=%hu,%hu", &b, &e ) ;
                     switch ( args )
                     {
                        case 1:     // scan from
                           this->scan_beg = b ;
                           break ;
                        case 2:     // scan range
                           this->scan_beg = b ;
                           this->scan_end = e ;
                           break ;
                        case 0:     // scan all
                        case WEOF:
                           this->scan_beg = ZERO ;
                           this->scan_end = 0xFFFF ;
                           break ;
                        default:
                           goodarg = this->scan = false ;
                           break ;
                     }
                  }
               }

               //* Undocumented '--book' option (book-end the blocks) *
               else if ( ca.argList[i][j] == 'b' )
               {
                  gs = ca.argList[i] ;
                  if ( (gs.compare( L"--book", 6 )) == ZERO )
                  {
                     goodarg = this->book = true ;
                  }
               }

               if ( goodarg != false )
               {
                  #if DEBUG_GCLA != 0
                  this->textOut ( gs, false ) ;
                  this->textOut ( L": specified" ) ;
                  #endif   // DEBUG_GCLA
               }
               else           // invalid switch
               {  //* Generate an error mesage *
                  if ( this->emCount < emMAX )
                  {
                     gs.compose( "Error! Unrecognized command: '%s' ", ca.argList[i] ) ;
                     gs.copy( this->ErrorMsg[this->emCount++], gsMAXCHARS ) ;
                  }
                  status = false ;
               }
               continue ;     // finished with this argument
            }  // (double dash)
            char argLetter = ca.argList[i][j] ;

            //* Text Mode processing *
            if ( argLetter == 't' || argLetter == 'T' )
            {
               ca.textFlag = true ;
               if ( ca.argList[i][j+1] != NULLCHAR ) multiarg = true ;
               #if DEBUG_GCLA != 0
               this->textOut ( L"-t: specified" ) ;
               #endif   // DEBUG_GCLA
            }

            //* Interactive Mode processing *
            else if ( argLetter == 'i' || argLetter == 'I' )
            {
               #if ENABLE_IMODE != ZERO
               ca.textFlag = false ;
               #else
               if ( this->emCount < emMAX )
               {
                  gs = "Sorry, Interactive Mode not yet implemented." ;
                  gs.copy( this->ErrorMsg[this->emCount++], gsMAXCHARS ) ;
               }
               status = false ;
               #endif
               if ( ca.argList[i][j+1] != NULLCHAR ) multiarg = true ;
               #if DEBUG_GCLA != 0
               this->textOut ( L"-i: specified" ) ;
               #endif   // DEBUG_GCLA
            }

            //* Verbose output (applies only to text mode) *
            else if ( (argLetter == 'v') && 
                      (strncmp ( ca.argList[i], "--v", 3 )) != ZERO)
            {
               this->verbose = true ;
               if ( ca.argList[i][j+1] != NULLCHAR ) multiarg = true ;
               #if DEBUG_GCLA != 0
               this->textOut ( L"-v: specified" ) ;
               #endif   // DEBUG_GCLA
            }

            //* Specify alternate CSS definition file *
            else if ( argLetter == 'f' )
            {
               ++j ;    // index next character after switch character
               if ( ca.argList[i][j] == NULLCHAR ) // if path is in next arg
               { ++i ; j = ZERO ; }
               else if ( ca.argList[i][j] == '=' )
                  ++j ;
               if ( ca.argList[i][j] != DASH && ca.argList[i][j] != NULLCHAR )
               {
                  this->cssFile = &(ca.argList[i][j]) ;
                  #if DEBUG_GCLA != 0
                  this->textOut ( L"-f: ", false ) ;
                  this->textOut ( this->cssFile ) ;
                  #endif   // DEBUG_GCLA
               }
               else
               {
                  if ( this->emCount < emMAX )
                  {
                     gs = "Error! '-f' switch specified without filename" ;
                     gs.copy( this->ErrorMsg[this->emCount++], gsMAXCHARS ) ;
                  }
                  status = false ;
               }
            }

            //* Process ALL files in directory *
            else if ( argLetter == 'a' )
            {
               this->allFiles = true ;
               if ( ca.argList[i][j+1] != NULLCHAR ) multiarg = true ;
               #if DEBUG_GCLA != 0
               this->textOut ( L"-a: specified" ) ;
               #endif   // DEBUG_GCLA
            }

            //* Alternate target Directory *
            else if ( argLetter == 'd' )
            {  //* Change the current working directory *
               ++j ;    // index next character after switch character
               if ( ca.argList[i][j] == NULLCHAR ) // if path is in next arg
               { ++i ; j = ZERO ; }
               else if ( ca.argList[i][j] == '=' )
                  ++j ;
               gs = &(ca.argList[i][j]) ;
               #if DEBUG_GCLA != 0
               this->textOut ( L"-d: ", false ) ;
               this->textOut ( gs ) ;
               #endif   // DEBUG_GCLA
               if ( (this->ppfCdTarget ( gs.ustr() )) != OK )
               {
                  if ( this->emCount < emMAX )
                  {
                     gs.compose( "Error! Alternate target directory '%S' not found.", gs.gstr() ) ;
                     gs.copy( this->ErrorMsg[this->emCount++], gsMAXCHARS ) ;
                  }
                  status = false ;
               }
            }

            //* Process Ordered lists *
            else if ( argLetter == 'o' )
            {
               this->oLists = true ;
               if ( ca.argList[i][j+1] != NULLCHAR ) multiarg = true ;
               #if DEBUG_GCLA != 0
               this->textOut ( L"-o: specified" ) ;
               #endif   // DEBUG_GCLA
            }

            //* For Ordered-list processing, prompt ALSO for sequence offset *
            //* (Also sets the '-o' switch.)                                 *
            else if ( argLetter == 'p' )
            {
               this->oLists = this->oOffst = true ;
               if ( ca.argList[i][j+1] != NULLCHAR ) multiarg = true ;
               #if DEBUG_GCLA != 0
               this->textOut ( L"-p: specified" ) ;
               #endif   // DEBUG_GCLA
            }

            //* Process table of Contents *
            else if ( argLetter == 'c' )
            {
               this->tocMod = true ;
               if ( ca.argList[i][j+1] != NULLCHAR ) multiarg = true ;
               #if DEBUG_GCLA != 0
               this->textOut ( L"-c: specified" ) ;
               #endif   // DEBUG_GCLA
            }

            //* Remove table of contents *
            else if ( argLetter == 'r' )
            {
               this->tocDel = true ;
               if ( ca.argList[i][j+1] != NULLCHAR ) multiarg = true ;
               #if DEBUG_GCLA != 0
               this->textOut ( L"-r: specified" ) ;
               #endif   // DEBUG_GCLA
            }

            //* Request for Help *
            else if ( argLetter == 'h' || argLetter == 'H' || argLetter == '?' )
            {  //* A cry for help overrides everything else   *
               //* on the command line except Version.        *
               //* (see below)
               ca.helpFlag = true ;
               #if DEBUG_GCLA != 0
               this->textOut ( L"-h: specified" ) ;
               #endif   // DEBUG_GCLA
            }

            else           // invalid switch
            {  //* Generate an error mesage *
               if ( this->emCount < emMAX )
               {
                  gs.compose( "Error! Unrecognized command: '%s' ", ca.argList[i] ) ;
                  gs.copy( this->ErrorMsg[this->emCount++], gsMAXCHARS ) ;
               }
               status = false ;
            }
         }
         else
         {  //* Interpret argument as a filename. (validation occurs below) *
            if ( this->sfCount < sfMAX )
            {
               gs = ca.argList[i] ;
               gs.copy( this->srcFiles[this->sfCount++], gsMAXCHARS ) ;
            }
            else
            {  //* Generate an error message *
               if ( this->emCount < emMAX )
               {
                  gs.compose( "Error! Too many source files specified. '%s' ignored.", ca.argList[i] ) ;
                  gs.copy( this->ErrorMsg[this->emCount++], gsMAXCHARS ) ;
               }
            }
         }
      }     // for(;;)

      //* Verify that CSS definition file exists and is our file.*
      if ( status != false )
      {
         if ( ((this->ppfRealpath ( gs, this->cssFile )) == OK)
              && ((this->ppfTargetExists ( gs )) != false) )
         {  //* Verify copyright notice and version number *
            gString gsVer ;
            if ( this->ppfTargetIsCSS ( gs, gsVer ) )
            {
               gsVer.copy( this->cssVersion, CSS_VER_LEN ) ;
            }
            else
            {
               if ( this->emCount < emMAX )
               {
                  gs.compose( "Error! CSS definition file '%S'"
                              " is of unknown configuration!", 
                              this->cssFile.gstr() ) ;
                  gs.copy( this->ErrorMsg[this->emCount++], gsMAXCHARS ) ;
               }
               status = false ;
            }
         }
         else
         {
            if ( this->emCount < emMAX )
            {
               gs.compose( "Error! CSS definition file '%S' was not found!", 
                           this->cssFile.gstr() ) ;
               gs.copy( this->ErrorMsg[this->emCount++], gsMAXCHARS ) ;
            }
            status = false ;
         }
      }

      if ( ca.helpFlag && !ca.verFlag )
      {  //* Help overrides everything except Version *
         ca.reset() ;
         ca.helpFlag = true ;
      }
   }        // if(ca.argCount>1)

   //* If the '-a' switch specified, scan all files in the target directory  *
   //* for HTML documents, and create a list of files to be processed.       *
   if ( this->allFiles != false && status != false )
      status = this->ppfScan4Src () ;


   //* Validate the list of source files to be processed.*
   if ( status != false )
   {
      for ( short sfIndex = ZERO ; sfIndex < this->sfCount ; sfIndex++ )
      {
         this->ppfCatPathFilename ( gs, this->cwDir, this->srcFiles[sfIndex] ) ;
         if ( this->ppfTargetExists ( gs ) )
         {
            if ( (this->ppfTargetIsHTML ( gs )) == false )
            {  //* Generate an error message *
               if ( this->emCount < emMAX )
               {
                  gs.compose( "Error! Source file '%S' is not valid HTML!", this->srcFiles[sfIndex] ) ;
                  gs.copy( this->ErrorMsg[this->emCount++], gsMAXCHARS ) ;
               }
               status = false ;
            }
         }
         else
         {  //* Generate an error mesage *
            if ( this->emCount < emMAX )
            {
               gs.compose( "Error! Source file '%S' was not found!", this->srcFiles[sfIndex] ) ;
               gs.copy( this->ErrorMsg[this->emCount++], gsMAXCHARS ) ;
            }
            status = false ;
         }
      }
   }

   //* No command line arguments OR text mode and no filenames specified. *
   //* If --css_mods specified then no HTML filenames are necessary.      *
   #if ENABLE_IMODE != 0   // interactive mode enabled
   if ( (ca.argCount == 1) || 
        ((ca.textFlag != false) && (this->sfCount == ZERO && this->css_mod == false)) )
   #else    // Interactive mode disabled
   // NOTE: Interactive Mode not yet implemented, so filenames must be specified.
   if ( (ca.argCount == 1) || (this->sfCount == ZERO && this->css_mod == false) )
   #endif   // Interactive mode
   {
      gs = "--no source filenames specified--" ;
      gs.copy( this->ErrorMsg[this->emCount++], gsMAXCHARS ) ;
      status = false ;
   }
   return status ;

#undef ENABLE_IMODE
#undef DEBUG_GCLA
}  //* End GetCommandLineArgs() *

//*************************
//*  DisplayAppVersion    *
//*************************
//******************************************************************************
//* Display the application's title, version and copyright info.               *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note: We play tricks with our title strings to make this      *
//* message look canonical. If the strings are changed, this message may get   *
//* ugly. That's why Trix are for kids....                                     *
//******************************************************************************

void Idpp::DisplayAppVersion ( void )
{
   const wchar_t* freeSoftware = 
   L"License GPLv3+: GNU GPL version 3 or later <http://gnu.org/licenses/gpl.html>\n"
    "This is free software: you are free to change and redistribute it.\n"
    "There is NO WARRANTY, to the extent permitted by law.\n" ;

   gString gsOut, gstmp( AppTitle1 ) ;
   gstmp.limitCharacters( 23 ) ;
   gsOut.compose( L"%S(idpp) version: %S\nCopyright (C) %S\n", 
                  gstmp.gstr(), AppVersion, &AppTitle2[4] ) ;
   this->textOut ( gsOut ) ;
   this->textOut ( freeSoftware ) ;
   
}  //* End DisplayAppVersion() *

//**************************
//* DisplayCommandLineHelp *
//**************************
//******************************************************************************
//* Display the brief version of command-line help.                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Idpp::DisplayCommandLineHelp ( void )
{
   gString gsOut ;
   gsOut.compose( L"\n%S%S%S\n%S", AppTitle1, AppVersion, AppTitle2, AppTitle3 ) ;
   this->textOut ( gsOut ) ;
   wcout << 
   //123456789-123456789-123456789-123456789-123456789-123456789-123456789-123456789-
   L"  Apply external CSS style definitions to HTML documents generated by\n"
    "  'makeinfo' utility. Source documents renamed as backup files, e.g.\n"
    "  'home.htm' becomes 'home.htm~'. Modifications written to NEW 'home.htm'.\n\n"
    "Usage  : idpp [OPTIONS][HTML_FILENAMES]   EXAMPLE: idpp -to --no_meta home.htm\n"
    "Options:\n"
    " -t     Start the application in text-only mode (default).\n"
    " -i     Start the application in interactive mode. [not currently implemented]\n"
    " -v     Verbose output, report details of each operation\n"
    " -a     Process all files in the target directory that look like HTML.\n"
    "        recognized filename extensions: html, htm, shtml, shtm, xhtml\n"
    " -o     Ordered-list processing: specification of enumeration type\n"
    "        Application prompts for a decision on each <ol> list.\n"
    " -p     Prompt for ordered-list sequence offset.\n"
    "        (default is to start sequence at beginning without prompting)\n"
    " -d     Specify an alternate target directory.\n"
    "                  (current working directory is the default)\n"
    "        Examples: idpp -d public_html\n"
    "                  idpp -dpublic_html\n"
    " -f     Specify an alternate filename for the CSS definition file.\n"
    "        ('infodoc-styles.css' in the target directory is the default)\n"
    "        Examples: idpp -f my-styles.css home.html\n"
    "                  idpp -fmy-styles.css home.html\n"
    " -c     Contents: process Table of Contents as a multi-level unordered list\n"
    " -r     Remove Table of Contents from the document\n"
    "        (Table of Contents is left unmodified by default)\n"
    " --up_target\n"
    "        If your document is a node on a document tree, specify path to\n"
    "        parent node. Example: idpp --up_target='../parent_node.htm'\n"
    "        (Up Target is set to top of current page by default.)\n"
    " --table_border [=[all | specify]]\n"
    "        Add a border/grid around the elements of a table.\n"
    "        Sub-options:\n"
    "           'all' (default): automatically draw borders around all tables\n"
    "           'specify': application prompts for a decision on each table\n"
    " --my_metadata\n"
    "        Insert custom metadata elements.\n"
    "        Example: idpp --my_metadata=metadata_filename\n"
    "        Contents of the specified file will be copied into the\n"
    "         document's <head> block, just above the </head> tag.\n"
    " --css_mods\n"
    "        Interactively modify CSS definition file.\n"
    "          [THIS OPTION IS NOT YET IMPLEMENTED]\n"
    " --no_mods\n"
    "        Do not perform modifications. Instead, list the operations that\n"
    "        WOULD BE processed for the options specified. (assumes '-v' option)\n"
    " --no_doctype\n"
    "        Do not update the <!DOCTYPE> tag. (updated to HTML5 spec by default)\n"
    " --no_meta\n"
    "        Do not delete the _valid_ <meta> elements in the <head> section.\n"
    "        section. These are: \"application-name\", \"author\", \"description\",\n"
    "        \"generator\", \"keywords\". (all metadata is discarded by default)\n"
    " --no_links\n"
    "        Do not delete the auto-generated <link> elements in the <head>\n"
    "        section (all links are discarded by default)\n"
    " --no_body\n"
    "        Do not update the <body> tag. (extra definitions discarded by default)\n"
    " --no_bullet\n"
    "        Do not update the <ul> lists. (bulleted lists are processed by default)\n"
    " --no_uplink\n"
    "        Do not modify the target specified in the \"Up\" hyperlink at the top\n"
    "        of the document. (Default is to point the link to top of the document.)\n"
    " --no_block\n"
    "        Do not remove the unnecessary blank line before the block classes\n"
    "        'format', 'display', 'example' and 'lisp'.  (deleted by default)\n"
    " --no_author\n"
    "        For a <blockquote> block followed by an author's name, do not adjust\n"
    "        the horizontal offset of the name. (adjusted by default)\n"
    " --no_cartouche\n"
    "        Do not remove the forced border styling from the header of a\n"
    "        'cartouch' (bordered paragraph).\n"
    "        (Default is to let the 'cartouche' class control the object.)\n"
    " --no_contain\n"
    "        Do not insert the 'infodoc_container' class into the document.\n"
    "        This will cause text to wrap at the edge of the browser window.\n"
    "        (Default is to wrap text at the edge of the container.)\n"
    " --version   Display version number and copyright notice.\n"
    " --help, -h  Command-line Help\n" 
    "        For detailed information, see Texinfo documentation:\n"
    "              info -f infodoc.info -n 'Infodoc Post-processor'\n"
    << endl ;

}  //* End DisplayCommandLineHelp() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

