//******************************************************************************
//* File       : idpp_file.cpp                                                 *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2014-2015 Mahlon R. Smith, The Software Samurai *
//*                 GNU GPL copyright notice below                             *
//* Date       : 22-Feb-2015                                                   *
//* Version    : (see AppVersion string)                                       *
//*                                                                            *
//* Description: This module contains methods for accessing files to be        *
//* processed, extracting information about each item and creating the         *
//* modified output.                                                           *
//*                                                                            *
//* These methods are adapted from a small subset of the FMgr class written    *
//* for the FileMangler file-management project by the same author.            *
//*                                                                            *
//*                                                                            *
//* Copyright Notice:                                                          *
//* -----------------                                                          *
//* This program is free software: you can redistribute it and/or modify it    *
//* under the terms of the GNU General Public License as published by the Free *
//* Software Foundation, either version 3 of the License, or (at your option)  *
//* any later version, PROVIDED THAT the copyright notices for both code and   *
//* documentation are included and unmodified.                                 *
//*                                                                            *
//* This program is distributed in the hope that it will be useful, but        *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
//* for more details.                                                          *
//*                                                                            *
//* You should have received a copy of the GNU General Public License along    *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.            *
//*                                                                            *
//*         Full text of the GPL License may be found in the TexInfo           *
//*         documentation for this program under 'Copyright Notice'.           *
//******************************************************************************
//* Version History (see idpp.cpp)                                             *
//*                                                                            *
//******************************************************************************

//****************
//* Header Files *
//****************
#include "idpp.hpp"

//****************
//* Local Data   *
//****************
//* Const strings for scanning files *
const wchar_t* topDOCT = L"<!DOCTYPE HTML" ;
const short    topDOCT_len = 14 ;
const wchar_t* topHTML = L"<HTML" ;
const short    topHTML_len = 5 ;
const wchar_t* cssCOPY = L"Copyright" ;
const short    cssCOPY_len = 9 ;
const wchar_t* cssSAMU = L"The Software Samurai" ;
const short    cssSAMU_len = 20 ;
const wchar_t* cssVERS = L"Version: " ;
const short    cssVERS_len = 9 ;
const wchar_t* headTag = L"<head" ;
const short    headTag_len = 5 ;
const wchar_t* headendTag = L"</head" ;
const short    headendTag_len = 6 ;
const wchar_t* titleTag = L"<title" ;
const short    titleTag_len = 6 ;
const wchar_t* bodyTag = L"<body" ;
const short    bodyTag_len = 5 ;
const wchar_t* bodyendTag = L"</body" ;
const short    bodyendTag_len = 6 ;
const wchar_t* indexTarget = L"<a name=\"Index-1\"></a>" ;
const short    indexTarget_len = 22 ;
const wchar_t* indexJumpto = L"<table><tr><th valign=\"top\">Jump to:" ;
const short    indexJumpto_len = 36 ;

const wchar_t* tocHead = L"<h2 class=\"contents-heading\">Table of Contents</h2>" ;
const short    tocHead_len = 51 ;
const wchar_t* tocTarget1 = L"<a name=\"SEC_Contents\"></a>" ;
const short    tocTarget1_len = 27 ;
const wchar_t* tocTarget2 = L"<a name=\"Top\"></a>" ;
const short    tocTarget2_len = 18 ;
const wchar_t* tocTarget3 = L"<a name=\"index-" ;
const short    tocTarget3_len = 15 ;
const wchar_t* tocMenuTitle = L"<h1 class=\"top\">" ;
const short    tocMenuTitle_len = 16 ;

const wchar_t* headerClass = L"<div class=\"header\">" ;
const short    headerClass_len = 20 ;
const wchar_t* headerLinks = L"Next: <a href=\"" ;
const short    headerLinks_len = 15 ;

const wchar_t* ulBegin = L"<ul" ;
const short    ulBegin_len = 3 ;
const wchar_t* ulNobull = L"<ul class=\"no-bullet\">" ;
const short    ulNobull_len = 22 ;
const wchar_t* ulEnd = L"</ul>" ;
const short    ulEnd_len = 5 ;
const wchar_t* olBegin = L"<ol>" ;
const short    olBegin_len = 4 ;
const wchar_t* olEnd = L"</ol>" ;
const short    olEnd_len = 5 ;
const wchar_t* liBegin = L"<li>" ;
const short    liBegin_len = 4 ;
const wchar_t* liLongBegin = L"</li><li>" ;
const short    liLongBegin_len = 9 ;
const wchar_t* paraEnd = L"</p>" ;
const short    paraEnd_len = 4 ;
const wchar_t* tabBegin = L"<table" ;
const short    tabBegin_len = 6 ;

const wchar_t* idbDIV = L"<div class=\"indentedblock\">" ;
const short    idbDIV_len = 27 ;
const wchar_t* smidbDIV = L"<div class=\"smallindentedblock\">" ;
const short    smidbDIV_len = 32 ;
const wchar_t* qtBEGIN = L"<blockquote" ;
const short    qtBEGIN_len = 11 ;
const wchar_t* qtLongBEGIN = L"</p><blockquote" ;
const short    qtLongBEGIN_len = 15 ;

//* Const output strings *
const char* stdDOCT = "<!DOCTYPE HTML>" ;
const char* stdHTML = "<html>" ;
const char* cssLINK0 = 
   "<!-- Post-processing of this document performed using %S%S -->" ;
const char* cssLINK1 = 
   "<meta charset=\"utf-8\" />  <!-- Import the global stylesheet -->\n"
   "<link rel=\"stylesheet\" href=\"" ;
const char* cssLINK2 = 
   "\" lang=\"en\" type=\"text/css\"/>\n" ;
const char* stdBODY = "<body>" ;
const wchar_t* begCONTAINER = L"<div class=\"infodoc_container\">" ;
const short begCONTAINER_len = 31 ;
const wchar_t* endCONTAINER = L"</div>    <!-- infodoc_container -->" ;
const short endCONTAINER_len = 36 ;
const char* tocDISC   = "<ul class=\"toc-level1\">" ;
const char* tocCIRCLE = "<ul class=\"toc-level2\">" ;
const char* tocSQUARE = "<ul class=\"toc-level3\">" ;
const char* ulDISC   = "<ul class=\"disc-bullet\">" ;
const char* ulCIRCLE = "<ul class=\"circle-bullet\">" ;
const char* ulSQUARE = "<ul class=\"square-bullet\">" ;

//* Placeholder user response. (see 'userResponse' method for details) *
const wchar_t* dToken = L"default_token" ;
const short    dToken_len = 13 ;



//*************************
//*   ppfProcessSrcHTML   *
//*************************
//******************************************************************************
//* Convert raw HTML to CSS-styled HTML.                                       *
//*                                                                            *
//* Input  : src : (by reference) path/filename of source data                 *
//*          trg : (by reference) path/filename of target                      *
//*                                                                            *
//* Returns: OK if successful, ERR processing error or user aborted process    *
//******************************************************************************
//* Programmer's Note: The TexInfo HTML converter always outputs a DOCTYPE     *
//* statement; however, in HTML3, it was allowed to omit it. For this reason,  *
//* we test for both with and without (just to be safe).                       *
//* If the DOCTYPE statement is missing, then the <html> tag probably contains *
//* some definitions similar to those in the converter's DOCTYPE line. Thus,   *
//* if 'no_doct' != false we keep them, else we output a standard <html> tag.  *
//******************************************************************************

short Idpp::ppfProcessSrcHTML ( const gString& src, const gString& trg )
{
   const wchar_t* metaApp = L"<meta name=\"application-name" ;
   const short    metaApp_len = 28 ;
   const wchar_t* metaAuth = L"<meta name=\"author" ;
   const short    metaAuth_len = 18 ;
   const wchar_t* metaDesc = L"<meta name=\"description" ;
   const short    metaDesc_len = 23 ;
   const wchar_t* metaGen = L"<meta name=\"generator" ;
   const short    metaGen_len = 21 ;
   const wchar_t* metaKeys = L"<meta name=\"keywords" ;
   const short    metaKeys_len = 20 ;
   const wchar_t* metaDate = L"<meta name=\"date\"" ;
   const short    metaDate_len = 17 ;
   const wchar_t* linkBegin = L"<link" ;
   const short    linkBegin_len = 5 ;
   const wchar_t* gplBegin = L"<a name=\"GNU-General-Public-License\"></a>" ;
   const short    gplBegin_len = 41 ;
   const wchar_t* fdlBegin = L"<a name=\"GNU-Free-Documentation-License\"></a>" ;
   const short    fdlBegin_len = 45 ;

   this->slCount = ZERO ;     // reset source-line counter (zero-based counter)
   this->tlCount = ZERO ;     // reset target-line counter (zero-based counter)
   short status = OK ;        // return value

   //* Open the source file, and                      *
   //* if processing is enabled, open the target file.*
   this->ifs.open ( src.ustr(), ifstream::in ) ;
   if ( this->no_mods == false )
      this->ofs.open ( trg.ustr(), ofstream::trunc ) ;

   //* Access to source and target? *
   if ( this->ifs.is_open() && ((this->ofs.is_open()) || this->no_mods != false) )
   {
      gString gs,             // conversion to wide text
              gsOut,          // output data for target file
              gsBook ;        // debugging output
      const wchar_t* wptr ;   // pointer and index to wide text
      short wi ;

      //* Read the first source line (validates document as HTML) *
      if ( (status = this->ppfReadSrcLine ( gs, wptr, wi )) == OK )
      {  //* 'true' if document does not have a doctype statement.*
         bool dtMissing = ((wcsncasecmp ( &wptr[wi], topHTML, topHTML_len )) == ZERO ) ;
         //* 'true' if document has a valid doctype statement.*
         bool dtValid = ((wcsncasecmp ( &wptr[wi], topDOCT, topDOCT_len )) == ZERO) ;
         if ( dtMissing || (dtValid && this->no_doct == false) )
         {
            gsOut = stdDOCT ;
            this->ppfWriteLine ( gsOut ) ;

            //* If first line was a valid <html> tag (see note above) *
            if ( dtMissing )
            {
               if ( this->no_doct == false ) // output a standard tag
                  gsOut = stdHTML ;
               else                          // output the original line
                  gsOut = gs ;
               this->ppfWriteLine ( gsOut ) ;
            }
         }
         //* Else if user wants to keep original DOCTYPE line *
         else if ( dtValid && this->no_doct )
            this->ppfWriteLine ( gs ) ;
         else
            status = ERR ;
      }

      //* Copy source lines to target until '<head>' tag copied *
      while ( status == OK )
      {
         if ( (status = this->ppfReadSrcLine ( gs, wptr, wi )) == OK )
         {
            this->ppfWriteLine ( gs.ustr() ) ;
            if ( (wcsncasecmp ( &wptr[wi], headTag, headTag_len)) == ZERO )
            {
               if ( this->book != false )
               {
                  gsBook.compose( L"ML(%hu) <HEAD>", &this->slCount ) ;
                  this->textOut ( gsBook ) ;
               }
               break ;     // <head> found and copied
            }
         }
      }

      //* Discard everything inside <head></head> EXCEPT: retain <title> tag, *
      //* and IF specified, retain the special metadata and/or links.         *
      while ( status == OK )
      {
         if ( (status = this->ppfReadSrcLine ( gs, wptr, wi )) == OK )
         {
            if ( (wcsncasecmp ( &wptr[wi], titleTag, titleTag_len)) == ZERO )
            {  //* <title> found, insert link to CSS definition file. *
               gsOut.compose( cssLINK0, AppTitle1, AppVersion ) ;
               this->ppfWriteLine ( gsOut ) ;
               gString gscf ;
               gsOut.compose( L"%s%S%s", cssLINK1, this->cssFile.gstr(), cssLINK2 ) ;
               this->ppfWriteLine ( gsOut ) ;   // write file link to target
               this->tlCount += 2 ; // this is a 3-line sequence
               this->ppfWriteLine ( gs ) ;      // copy the title to target
               // NOTE: We assume that the entire title is on one line.
            }

            //* If end of <head></head> block found *
            else if ( (wcsncasecmp ( &wptr[wi], headendTag, headTag_len)) == ZERO )
            {

               //* If specified, insert user's custom metadata into the    *
               //* <head>...</head> block. We DO NOT validate these data.*
               if ( this->my_meta != false )
               {
                  if ( this->verbose != false )
                  {
                     gString gsVerb( "(%4hu) Inserting custom data at line %hu.", 
                                     &this->slCount, &this->tlCount ) ;
                     this->textOut ( gsVerb ) ;
                  }

                  //* Copy data from user file to target *
                  status = this->ppfInsertCustomData ( this->userMeta ) ;
               }

               //* Write the end-of-head tag *
               this->ppfWriteLine ( gs.ustr() ) ;
               if ( this->book != false )
               {
                  gsBook.compose( L"ML(%hu) </HEAD>", &this->slCount ) ;
                  this->textOut ( gsBook ) ;
               }
               break ;        // our <head> is now (nearly :) empty
            }

            //* If user wants to retain the VALID metadata *
            else if ( (this->no_meta != false) &&
                      (((wcsncasecmp ( &wptr[wi], metaDesc, metaDesc_len)) == ZERO)
                      ||
                      ((wcsncasecmp ( &wptr[wi], metaKeys, metaKeys_len)) == ZERO)
                      ||
                      ((wcsncasecmp ( &wptr[wi], metaAuth, metaAuth_len)) == ZERO)
                      ||
                      ((wcsncasecmp ( &wptr[wi], metaApp, metaApp_len)) == ZERO)
                      ||
                      ((wcsncasecmp ( &wptr[wi], metaGen, metaGen_len)) == ZERO)
                      ||
                      ((wcsncasecmp ( &wptr[wi], metaDate, metaDate_len)) == ZERO)) )
            {
               status = this->ppfProcHEAD ( gs ) ;
            }

            //* If user wants to retain the <link> entries *
            else if ( (this->no_link != false) &&
                      ((wcsncasecmp ( &wptr[wi], linkBegin, linkBegin_len)) == ZERO) )
            {
               status = this->ppfProcHEAD ( gs ) ;
            }

#if 0    // NOT CURRENTLY NEEDED
            //* If start of <style></style> block found,*
            //* discard the <style> block.              *
            if ( false )
            {
            }

            //* If user wants to retain all non-style data *
            else if ( this->no_head )
            {
            }
#endif   // NOT CURRENTLY NEEDED

            else
            { /* Discard the line */ }
         }
      }

      //* Copy lines until we find the <body> tag *
      while ( status == OK )
      {
         if ( (status = this->ppfReadSrcLine ( gs, wptr, wi )) == OK )
         {
            if ( (wcsncasecmp ( &wptr[wi], bodyTag, bodyTag_len)) == ZERO )
            {  //* <body> found, replace it with a lean <body>. *
               if ( this->no_body == false )
                  gsOut = stdBODY ;
               else
                  gsOut = gs ;
               this->ppfWriteLine ( gsOut ) ;

               if ( this->book != false )
               {
                  gsBook.compose( L"ML(%hu) <BODY>", &this->slCount ) ;
                  this->textOut ( gsBook ) ;
               }

               //* Establish the container just inside the <body> tag. *
               if ( this->no_cont == false )
               {
                  gsOut = begCONTAINER ;
                  this->ppfWriteLine ( gsOut ) ;
               }
               break ;
            }
            this->ppfWriteLine ( gs ) ;
         }
      }

      //* Copy lines until we reach the Table of Contents (if any) *
      //*    a) tocTarget1 is present if TOC is present.           *
      //*    b) tocTarget2 is always present.                      *
      while ( status == OK )
      {
         if ( (status = this->ppfReadSrcLine ( gs, wptr, wi )) == OK )
         {
            if ( (wcsncasecmp ( &wptr[wi], tocTarget1, tocTarget1_len)) == ZERO )
            {  //* Table of Contents intra-page target found. *
               //* Copy it, then process Table of Contents.   *
               this->ppfWriteLine ( gs ) ;
               status = this->ppfProcTOC ( true ) ;
               break ;
            }
            else if ( (wcsncasecmp ( &wptr[wi], tocTarget2, tocTarget2_len)) == ZERO )
            {
               this->ppfWriteLine ( gs ) ;
               status = this->ppfProcTOC ( false ) ;
               break ;
            }

            //* If container class was established on a previous pass, *
            //* delete the old copy to avoid duplication.              *
            if ( (wcsncasecmp ( &wptr[wi], begCONTAINER, begCONTAINER_len)) == ZERO )
               continue ;
            this->ppfWriteLine ( gs ) ;
         }
      }

      //*****************************************************************
      //* Copy lines until we reach the end of the <body></body> block. *
      //*****************************************************************
      wchar_t bType = ZERO ;     // type of preformatted block found
      while ( status == OK )
      {
         if ( (status = this->ppfReadSrcLine ( gs, wptr, wi )) == OK )
         {  //* If </body> (end-of-body) tag found *
            if ( (wcsncasecmp ( &wptr[wi], bodyendTag, bodyendTag_len)) == ZERO )
            {
               if ( this->book != false )
               {
                  gsBook.compose( L"ML(%hu) </BODY>", &this->slCount ) ;
                  this->textOut ( gsBook ) ;
               }

               //* Insert close of container just above it. *
               if ( this->no_cont == false )
               {
                  gsOut = endCONTAINER ;
                  this->ppfWriteLine ( gsOut ) ;
               }
               this->ppfWriteLine ( gs ) ;
               break ;
            }

            //* If beginning an indentedblock or smallindentedblock block *
            else if ( ((wcsncasecmp ( &wptr[wi], idbDIV, idbDIV_len)) == ZERO)
                      ||
                      ((wcsncasecmp ( &wptr[wi], smidbDIV, smidbDIV_len)) == ZERO) )
            {
               if ( this->book != false )
               {
                  gsBook.compose( L"ML(%hu) IBLOCK START", &this->slCount ) ;
                  this->textOut ( gsBook, false ) ;
               }

               status = this->ppfProcIDB ( gs ) ;

               if ( this->book != false )
               {
                  gsBook.compose( L"--(%hu) IBLOCK END", &this->slCount ) ;
                  this->textOut ( gsBook ) ;
               }
            }

            //* If beginning of a preformatted block is detected ('display, *
            //* 'format', 'example', 'lisp' and their 'small' versions.     *
            else if ( (this->ppfTestFormattedBlock ( bType, &wptr[wi] )) != false )
            {
               if ( this->book != false )
               {
                  gsBook.compose( L"ML(%hu) PBLOCK START", &this->slCount ) ;
                  this->textOut ( gsBook, false ) ;
               }

               //* Copy the block, optionally eliminating           *
               //* the unnecessary leading blank line.              *
               //* NOTE: Data inside these blocks ARE NOT modified. *
               status = this->ppfProcFormattedBlock ( bType, gs ) ;

               if ( this->book != false )
               {
                  gsBook.compose( L"--(%hu) PBLOCK END", &this->slCount ) ;
                  this->textOut ( gsBook ) ;
               }
            }

            //* If a <blockquote> (@quotation command) or                     *
            //* <blockquote class="smallquotation'> (@smallquotation command) *
            else if ( ((wcsncasecmp ( &wptr[wi], qtBEGIN, qtBEGIN_len)) == ZERO)
                      ||
                      ((wcsncasecmp ( &wptr[wi], qtLongBEGIN, qtLongBEGIN_len)) == ZERO) )
            {
               if ( this->book != false )
               {
                  gsBook.compose( L"ML(%hu) QBLOCK START", &this->slCount ) ;
                  this->textOut ( gsBook, false ) ;
               }

               status = this->ppfProcQUOTE ( gs ) ;

               if ( this->book != false )
               {
                  gsBook.compose( L"--(%hu) QBLOCK END", &this->slCount ) ;
                  this->textOut ( gsBook ) ;
               }
            }

            //* For <ul> lists *
            else if ( (this->ppfTestUlistBegin ( &wptr[wi] )) != false )
            {
               if ( this->book != false )
               {
                  gsBook.compose( L"ML(%hu) UBLOCK START", &this->slCount ) ;
                  this->textOut ( gsBook, false ) ;
               }

               //* Process the (possibly nested) <ul> / <ol> lists.*
               status = this->ppfProcUL ( gs ) ;

               if ( this->book != false )
               {
                  gsBook.compose( L"--(%hu) UBLOCK END", &this->slCount ) ;
                  this->textOut ( gsBook ) ;
               }
            }

            //* For <ol> lists *
            else if ( (this->ppfTestOlistBegin ( &wptr[wi] )) != false )
            {
               if ( this->book != false )
               {
                  gsBook.compose( L"ML(%hu) OBLOCK START", &this->slCount ) ;
                  this->textOut ( gsBook, false ) ;
               }

               //* Process the (possibly nested) <ol> / <ul> lists.*
               status = this->ppfProcOL ( gs ) ;

               if ( this->book != false )
               {
                  gsBook.compose( L"--(%hu) OBLOCK END", &this->slCount ) ;
                  this->textOut ( gsBook ) ;
               }
            }

            //* If a <table> object found *
            else if ( (wcsncasecmp ( &wptr[wi], tabBegin, tabBegin_len)) == ZERO )
            {
               status = this->ppfProcTAB ( gs ) ;
            }

            //* If the GNU General Public License or GNU Free  *
            //* Documentation License is detected, process it. *
            else if (   ((wcsncasecmp ( &wptr[wi], gplBegin, gplBegin_len)) == ZERO)
                     || ((wcsncasecmp ( &wptr[wi], fdlBegin, fdlBegin_len)) == ZERO) )
            {
               this->ppfWriteLine ( gs ) ;      // write the unmodified line
               bool gpl = true ;
               if ( (wcsncasecmp ( &wptr[wi], fdlBegin, fdlBegin_len)) == ZERO )
                  gpl = false ;
               status = this->ppfProcGNU ( gpl ) ;
            }

            //* If we have reached the beginning if the Index node. *
            //*          (this is below the node header)            *
            else if ( (wcsncasecmp ( &wptr[wi], indexTarget, indexTarget_len)) == ZERO )
            {
               if ( this->book != false )
               {
                  gsBook.compose( L"ML(%hu) XBLOCK START", &this->slCount ) ;
                  this->textOut ( gsBook, false ) ;
               }
               
               this->ppfWriteLine ( gs ) ;      // write the target line
               status = this->ppfProcINDEX () ;

               if ( this->book != false )
               {
                  gsBook.compose( L"--(%hu) XBLOCK END", &this->slCount ) ;
                  this->textOut ( gsBook ) ;
               }
            }

            //* If container class was established on a previous pass, *
            //* delete the old copy to avoid duplication.              *
            else if ( (wcsncasecmp ( &wptr[wi], endCONTAINER, endCONTAINER_len)) == ZERO )
            {
               /* do nothing */
            }

            else        // this line gets no processing
               this->ppfWriteLine ( gs ) ;
         }
      }

      //* Finish copying the file *
      while ( status == OK )
      {
         if ( (this->ppfReadSrcLine ( gs, wptr, wi )) == OK )
            this->ppfWriteLine ( gs ) ;
         else        // last source line copied to target
            break ;
      }
   }
   if ( this->ifs.is_open() )
      this->ifs.close() ;           // close the source file
   if ( this->ofs.is_open() )
      this->ofs.close() ;           // close the target file
   return status ;

}  //* End ppfProcessSrcHTML()

//*************************
//*      ppfProcTOC       *
//*************************
//******************************************************************************
//* Called by ppfProcessSrcHTML() to process the Table Of Contents list.       *
//* 1) If 'tocDel' == false :                                                  *
//*    a) If 'tocMod' == false :                                               *
//*       Copy Table of Contents unmodified.                                   *
//*    b) If 'tocMod' != false :                                               *
//*       Convert the "no-bullet" classes within the Table of Contents to      *
//*       the appropriate level of bullet class.                               *
//*       - 1st level "no-bullet" class becomes toc-level1" class              *
//*       - 2nd level "no-bullet" class becomes toc-level2" class              *
//*       - 3rd and lower levels "no-bullet" class becomes toc-level3" class   *
//* 2) If 'tocDel' != false :                                                  *
//*    Delete the entire Table of Contents.                                    *
//* 3) If user has specified an alternate "up" target, insert it into the      *
//*    "up" link. Else, just insert a '#' character, so "up" just goes to the  *
//*    Table Of Contents on the same page. Note that if user has specified     *
//*    that the TOC be removed, the link will still work.                      *
//* 4) If TOC is not present at the top of the document, don't process it,     *
//*    BUT, do other housekeeping chores.                                      *
//*                                                                            *
//* Input  : tocFound : 'true' if caller found the TOC, else 'false'           *
//*                                                                            *
//* Returns: OK if successful, ERR processing error                            *
//******************************************************************************
//* Programmer's Note: We would also like to adjust the header's               *
//*             'Contents' LINK: href="#" and rel="up"                         *
//* to point to the top of the page instead of to the Table Of Contents.       *
//* However, it is a minor problem requiring a complex parsing solution, so    *
//* we have elected to ignore it.                                              *
//******************************************************************************

short Idpp::ppfProcTOC ( bool tocFound )
{
   const wchar_t* upLink = L"Up: <a href=\"" ;
   const short    upLink_len = 13 ;
//   const wchar_t* dfltUplink = L"dir.html#Top" ;
//   const short    dfltUplink_len = 12 ;

   gString gs,             // conversion to wide text
           gsOut ;         // output data for target file
   const wchar_t* wptr ;   // pointer and index to wide text
   short wi,
         tocLevel = ZERO,  // nesting level for TOC line items
         trgs = tocFound ? 2 : 1 ; // when this reaches zero, TOC is processed
   short status = OK ;     // return value

   while ( status == OK )
   {
      if ( (status = this->ppfReadSrcLine ( gs, wptr, wi )) == OK )
      {
         //* The intra-page targets must be retained for navigation *
         if ( ((wcsncasecmp ( &wptr[wi], tocTarget2, tocTarget2_len)) == ZERO)
              ||
              ((wcsncasecmp ( &wptr[wi], tocTarget3, tocTarget3_len)) == ZERO) )
         {
            this->ppfWriteLine ( gs ) ;
            if ( --trgs == ZERO )   // count intra-page targets found
               break ;
         }
         //* The source document may, or may not have node headers. If it   *
         //* does, then the "UP" link of the first header points into space.*
         //* Unless user has specified that we should leave it alone, adjust*
         //* link to point to the top of the current page, or to the        *
         //* user-specified target.                                         *
         else if ( (wcsncasecmp ( &wptr[wi], headerLinks, headerLinks_len)) == ZERO )
         {
            //* If not prohibited OR if user specified an alternate link target*
            if ( this->no_utrg == false || this->upTarg != false )
            {
               //* Scan for "Up" link specification *
               short lineLen = gs.gschars() ;
               wi += headerLinks_len ;
               while ( (++wi + upLink_len) < lineLen ) // prevent buffer overrun
               {
                  if ( (wcsncasecmp ( &wptr[wi], upLink, upLink_len )) == ZERO )
                  {
                     wi += upLink_len ;
                     gString gstmp = gs ;
                     gstmp.limitCharacters( wi ) ;
                     gstmp.append( this->upTargPath ) ;
                     while ( wptr[wi] != L'"' && wptr[wi] != NULLCHAR )
                        ++wi ;   // step over current target string
                     //* Append remainder of <a> tag, replace displayed text *
                     //* and append remainder of line.                       *
                     wchar_t aChar ;
                     while ( wptr[wi] != NULLCHAR )
                     {
                        aChar = wptr[wi++] ;
                        gstmp.append( aChar ) ;
                        if ( aChar == L'>' )
                           break ;
                     }
                     while ( wptr[wi] != L'<' && wptr[wi] != NULLCHAR )
                        ++wi ;
                     gstmp.append( this->upTargText ) ;
                     gstmp.append( &wptr[wi] ) ;
                     gs = gstmp ;
                     break ;
                  }
               }
            }
            this->ppfWriteLine ( gs ) ;
         }
         else if ( this->tocDel == false )
         {
            //* If user specified that TOC should be treated as an    *
            //* unordered lists, process it. Else copy TOC unmodified.*
            if ( this->tocMod != false )
            {
               if ( (wcsncasecmp ( &wptr[wi], ulNobull, ulNobull_len )) == ZERO )
               {
                  gs.limitCharacters( wi ) ;
                  if ( ++tocLevel == 1 )
                     gs.append( tocDISC ) ;
                  else if ( tocLevel == 2 )
                     gs.append( tocCIRCLE ) ;
                  else
                     gs.append( tocSQUARE ) ;
                  this->ppfWriteLine ( gs ) ;
               }
               else
               {
                  this->ppfWriteLine ( gs ) ;
                  if ( ((wcsncasecmp ( &wptr[wi], ulEnd, ulEnd_len )) == ZERO)
                       && (tocLevel > ZERO) )
                  {
                     --tocLevel ;
                  }
               }
            }
            else
               this->ppfWriteLine ( gs ) ;
         }
         else if ( this->tocDel != false && 
                   ((wcsncasecmp ( &wptr[wi], tocMenuTitle, tocMenuTitle_len )) == ZERO) )
         {  //* Retain the title of the Main Menu node. *
            this->ppfWriteLine ( gs ) ;
         }
      }
   }
   return status ;

}  //* End ppfProcTOC() *

//*************************
//*     ppfProcINDEX      *
//*************************
//******************************************************************************
//* Called by ppfProcessSrcHTML() to process the Index table.                  *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: OK if successful, ERR processing error                            *
//******************************************************************************

short Idpp::ppfProcINDEX ( void )
{
   const char* indexJUMPTO = " class=\"jumpto\"" ;
   const short jtOFFSET = 6 ;    // insertion point for indexJUMPTO
   //* This is what 'Jump to:' will look like if it's already been processed.*
   const wchar_t* fixedTable = L"<table class=\"jumpto\">" ;
   const short fixedTable_len = 22 ;

   gString gs ;            // conversion to wide text
   const wchar_t* wptr ;   // pointer and index to wide text
   short wi,
         trgs = 2 ;        // when this reaches zero, Index is processed
   short status = OK ;     // return value

   while ( status == OK )
   {
      if ( (status = this->ppfReadSrcLine ( gs, wptr, wi )) == OK )
      {
         if ( (wcsncasecmp ( &wptr[wi], indexJumpto, indexJumpto_len)) == ZERO )
         {
            gs.insert( indexJUMPTO, (wi + jtOFFSET) ) ;
            this->ppfWriteLine ( gs ) ;
            if ( --trgs == ZERO )
               break ;
         }
         else
         {
            this->ppfWriteLine ( gs ) ;   // just copy the line

            //* If index already repaired, move on.*
            if ( (wcsncasecmp ( &wptr[wi], fixedTable, fixedTable_len)) == ZERO )
            {
               if ( --trgs == ZERO )
                  break ;
            }
         }
      }
   }
   return status ;

}  //* End ppfProcINDEX() *

//*************************
//*      ppfProcIDB       *
//*************************
//******************************************************************************
//* Called by ppfProcessSrcHTML() to process 'indentedblock' and               *
//* 'smallindentedblock' sequences.                                            *
//*                                                                            *
//* Input  : gsbq  : (by reference) contains the first line of the block.      *
//*                                                                            *
//* Returns: OK if successful, ERR processing error                            *
//******************************************************************************
//* Programmer's Note: It is possible that this method will be called          *
//* recursively, so be aware.                                                  *
//*                                                                            *
//* Programmer's Note: Processing this block separately is not actually        *
//* necessary because everthing is processed smoothly from the main loop;      *
//* however, it is potentially more robust to process the contents here.       *
//* Currently (v:0.0.03), the HTML target is identical, whether the block is   *
//* processed from the main loop or processed here.                            *
//*                                                                            *
//******************************************************************************

short Idpp::ppfProcIDB ( const gString& gsib )
{
   const wchar_t* idbEND = L"</div>" ;
   const short    idbEND_len = 6 ;

   gString gs ;            // conversion to wide text
   wchar_t bType = ZERO ;  // type of preformatted block found
   const wchar_t* wptr ;   // pointer and index to wide text
   short wi = ZERO,
         status = OK ;     // return value

   //* Process the block header *
   this->ppfWriteLine ( gsib ) ;

   //* Process the body of the block *
   bool done = false ;
   while ( ! done && status == OK )
   {  //* If at the end of the block, write the line and jump out
      if ( (status = this->ppfReadSrcLine ( gs, wptr, wi )) == OK )
      {
         if ( (this->ppfEndBlock ( gs, idbEND, idbEND_len )) != false )
         {
            this->ppfWriteLine ( gs ) ;
            done = true ;
         }

         //* If beginning an indentedblock or smallindentedblock block *
         else if ( ((wcsncasecmp ( &wptr[wi], idbDIV, idbDIV_len)) == ZERO)
                   ||
                   ((wcsncasecmp ( &wptr[wi], smidbDIV, smidbDIV_len)) == ZERO) )
         {
            status = this->ppfProcIDB ( gs ) ;
         }

         //* If beginning of a preformatted block is detected ('display, *
         //* 'format', 'example', 'lisp' and their 'small' versions.     *
         else if ( (this->ppfTestFormattedBlock ( bType, &wptr[wi] )) != false )
         {
            //* Copy the block, optionally eliminating           *
            //* the unnecessary leading blank line.              *
            //* NOTE: Data inside these blocks ARE NOT modified. *
            status = this->ppfProcFormattedBlock ( bType, gs ) ;
         }

         //* If a <blockquote> (@quotation command) or                     *
         //* <blockquote class="smallquotation'> (@smallquotation command) *
         else if ( ((wcsncasecmp ( &wptr[wi], qtBEGIN, qtBEGIN_len)) == ZERO)
                   ||
                   ((wcsncasecmp ( &wptr[wi], qtLongBEGIN, qtLongBEGIN_len)) == ZERO) )
         {
            status = this->ppfProcQUOTE ( gs ) ;
         }

         //* If a <table> object found *
         else if ( (wcsncasecmp ( &wptr[wi], tabBegin, tabBegin_len)) == ZERO )
         {
            status = this->ppfProcTAB ( gs ) ;
         }

         //* For <ul> lists *
         else if ( (this->ppfTestUlistBegin ( &wptr[wi] )) != false )
         {
            //* Process the (possibly nested) <ul> / <ol> lists.*
            status = this->ppfProcUL ( gs ) ;
         }

         //* For <ol> lists *
         else if ( (this->ppfTestOlistBegin ( &wptr[wi] )) != false )
         {
            //* Process the (possibly nested) <ol> / <ul> lists.*
            status = this->ppfProcOL ( gs ) ;
         }

         else     // copy the line without processing
            this->ppfWriteLine ( gs ) ;
      }
   }
   return status ;

}  //* End ppfProcIDB() *

//*************************
//*     ppfProcQUOTE      *
//*************************
//******************************************************************************
//* Called by ppfProcessSrcHTML() to process <blockquote> sequences.           *
//* This includes both <blockquote> (standard font size) and                   *
//*                    <blockquote class="smallquotation"> sequences.          *
//* Sequence is modified ONLY if we find an author's name sequence following   *
//* the block. The author sequence will have the form:                         *
//*         <div align=\"center\">&mdash; <em>Author Name>/em>                 *
//*         </div>                                                             *
//*                                                                            *
//* Input  : gsbq  : (by reference) contains the first line of the blockquote  *
//*                  sequence                                                  *
//*                                                                            *
//* Returns: OK if successful, ERR processing error                            *
//******************************************************************************
//* Programmer's Note: The author name sequence is moved INSIDE the block,     *
//* where we believe it should have been all along. In addition, we replace    *
//* the centered style with a simple offset style which looks much better.     *
//*                                                                            *
//* Programmer's Note: The procedure for this modification is not particularly *
//* robust, but it handles all _reasonable_ tests well.                        *
//*                                                                            *
//******************************************************************************

short Idpp::ppfProcQUOTE ( const gString& gsbq )
{
   const wchar_t* blkQUOTE = L"<blockquote>" ;
   const short    blkQUOTE_len = 12 ;
//   const wchar_t* smblkQUOTE = L"<blockquote class=\"smallquotation\">" ;
//   const short    smblkQUOTE_len = 35 ;
   const wchar_t* bqCLASS = L"<blockquote class=\"quotation\">" ;
//   const short    bqCLASS_len = 30 ;
   const wchar_t* blkEndQUOTE = L"</blockquote>" ;
   const short    blkEndQUOTE_len = 13 ;
   const wchar_t* endDIV = L"</div>" ;
   const short    endDIV_len = 6 ;
   const wchar_t* authorCMD = L"<div align=\"center\">&mdash; <em>" ;
   const short    authorCMD_len = 32 ;
   const wchar_t* modTEMPLATE = L"<br><span style=\"margin-left:3.2em;\">%S" ;

   gString gs ;            // conversion to wide text
   const wchar_t* wptr ;   // pointer and index to wide text
   short wi = ZERO ;
   short status = OK ;     // return value

   //* Process the block header *
   wptr = gsbq.gstr() ;
   for ( ; wptr[wi] == SPACE ; wi++ ) ;   // ignore leading whitespace
   if ( wptr[wi + 1] == SLASH )           // if leading </p> tag
      wi += 4 ;
   if ( ((wcsncasecmp ( &wptr[wi], blkQUOTE, blkQUOTE_len)) == ZERO)
        &&
        (wptr[wi + blkQUOTE_len] == NULLCHAR) )
   {
      gs = gsbq ;
      gs.limitCharacters( wi ) ;
      gs.append( bqCLASS ) ;
      this->ppfWriteLine ( gs ) ;
   }
   else     // write the header line as-is
      this->ppfWriteLine ( gsbq ) ;

   while ( status == OK )
   {
      if ( (status = this->ppfReadSrcLine ( gs, wptr, wi )) == OK )
      {
         if ( (this->ppfEndBlock ( gs, blkEndQUOTE, blkEndQUOTE_len )) != false )
         {
            gString gsauth, gsnew, gsdiv  ;
            const wchar_t *ptr ;
            short pi ;
            if ( ((status = this->ppfReadSrcLine ( gsauth, ptr, pi )) == OK)
                 &&
                 (this->no_auth == false) &&
                 ((wcsncasecmp ( &ptr[pi], authorCMD, authorCMD_len)) == ZERO) )
            {
               //* Reformat and write the 'author' line *
               gsauth.shiftChars( ZERO - (pi + authorCMD_len - 12) ) ;
               gsnew.compose( modTEMPLATE, gsauth.gstr() ) ;
               this->ppfWriteLine ( gsnew ) ;

               //* Read the end-of-author (</div> tag) line. If ONLY *
               //* the tag, discard it. Else, retain trailing data.  *
               status = this->ppfReadSrcLine ( gsdiv, ptr, pi ) ;
               if ( (wcsncasecmp ( &ptr[pi], endDIV, endDIV_len )) == ZERO )
               {
                  gsdiv.shiftChars( ZERO - (pi + endDIV_len) ) ;
                  if ( gsdiv.gschars() > 1 )
                     this->ppfWriteLine ( gsdiv ) ;
               }
               else  // This is unlikely and would be a source error.
               {
                  this->ppfWriteLine ( gsdiv ) ;
                  status = ERR ;
               }
               this->ppfWriteLine ( gs ) ;   // output the end-of-block line

               if ( this->verbose != false )
               {
                  USHORT authLine = this->tlCount - 1 ;
                  gString gsVerb( "(%4hu) Quotation Author repositioned.", &authLine ) ;
                  this->textOut ( gsVerb ) ;
               }
            }
            else
            {
               this->ppfWriteLine ( gs ) ;      // output the end-of-block line
               this->ppfWriteLine ( gsauth ) ;  // output the following line
            }
            break ;     // block processing complete
         }
         this->ppfWriteLine ( gs ) ;
      }
   }
   return status ;

}  //* End ppfProcQUOTE() *

//*************************
//*      ppfProcTAB       *
//*************************
//******************************************************************************
//* The beginning of a <table...> ... </table> sequence has been found.        *
//* Determine the TYPE of table object, and optionally modify it if it meets   *
//* our criteria.                                                              *
//*                                                                            *
//* Note that we process ONLY the table header here. Caller must copy the      *
//* remainder of the table.                                                    *
//*                                                                            *
//* Input  : gstab : (by reference) contains the first line of the <table>     *
//*                  sequence                                                  *
//*                                                                            *
//* Returns: OK if successful, ERR processing error                            *
//******************************************************************************
//* Notes:                                                                     *
//* 1) Only tables within the main sequence (after Table of Contents and before*
//*    the Index) are seen here.                                               *
//* 2) The tables representing the auto-generated menus are not modified.      *
//*          <table class="menu" border="0" cellspacing="0">                   *
//*    We may offer an option to modify menus in a future release.             *
//* 3) Plain <table> tags are generated by the texi @multitable command.       *
//*    a) If no border is to be added (default), then we write the <table>     *
//*       tag as: <table class="borderless">                                   *
//*       [NOTE: THIS IS NOT YET IMPLEMENTED. AT THIS TIME, WE WRITE <table>]  *
//*    b) If user has specified that borders are to be added to the table,     *
//*       then we write the <table> tag as: <table class="bordered">           *
//*       -- The '--table_border' command-line option sets the 'tabBorder'     *
//*          flag. There are two (2) sub-options for table-border processing:  *
//*          1) this->tabPrompt == false  (--table_border[=[all]])             *
//*             Automatically convert all identified tables of this type to    *
//*             include borders.                                               *
//*          2) this->tabPrompt != false  (table_border=specify)               *
//*             For each table of this type identified, prompt the user for a  *
//*             decision whether to include a border.                          *
//* 4) <table class="cartouche" ...> tags are generated by the texi @cartouche *
//*    command.                                                                *
//*    - By default, the table header is modified to remove the 'border="1"'   *
//*      style. This allows the 'cartouche' class to control the object.       *
//*    - If user has specified that this modification IS NOT to be performed,  *
//*      (--no_cartouche command-line option), then this->no_cart is set, and  *
//*      we write the header without modification.                             *
//*  5) All other '<table...> sequence headers are written unmodified to the   *
//*     target file.                                                           *
//*                                                                            *
//*  6) The user prompt for table border inclusion is documented as Yes/No;    *
//*     however, we have found it convenient to have a hidden respons, 'a'     *
//*     which resets this->tabPrompt, allowing the remaing tables to receive   *
//*     borders without further prompting.                                     *
//******************************************************************************

short Idpp::ppfProcTAB ( const gString& gstab )
{
   const wchar_t* tabPlain = L"<table>" ;
   const short    tabPlain_len = 7 ;
   const wchar_t* tabCartouche = L"<table class=\"cartouche\" border=\"1\">" ;
   const short tabCartouche_len = 36 ;
   const short tabCartMin_len = 24 ;
   const short tabCartTail_len = tabCartMin_len + 11 ;
   const wchar_t* bdrCLASS = L" class=\"bordered\">" ;

   gString gst ;           // for modifying the header text
   short status = OK ;     // return value

   //* Step over the whitespace *
   const wchar_t* wptr = gstab.gstr() ;
   short wi = ZERO ;
   while ( wptr[wi] == SPACE )
      ++wi ;

   //* If borders are to be put around tables *
   if ( (this->tabBorder != false) &&
        ((wcsncasecmp ( &wptr[wi], tabPlain, tabPlain_len)) == ZERO) )
   {
      gString gsNext ;
      wchar_t resp = L'y' ;
      bool modTable = true ;

      //* If we must prompt user for a decision *
      if ( this->tabPrompt != false )
      {
         //* Read the next line of the table to give user some context *
         const wchar_t* nPtr ;
         short ni ;
         if ( (status = this->ppfReadSrcLine ( gsNext, nPtr, ni )) == OK )
         {
            gString gsn = gsNext ;
            const wchar_t* nptr = gsn.gstr() ;
            for ( short ni = ZERO ; nptr[ni] != NULLCHAR ; ni++ )
            {
               if ( ((gsn.compare( L"</th>", 5, ni )) == ZERO) ||
                    ((gsn.compare( L"</td>", 5, ni )) == ZERO) )
               {
                  gsn.limitCharacters( ni + 5 ) ;
                  gsn.append( L". . ." ) ;
               }
            }
            gString gsIn,
                    gsOut( "____________________________\n"
                            "Table found on Line:%4hu\n"
                            "First Row: %S\n"
                            "Add border to this table?",
                            &this->slCount, gsn.gstr()
                         ) ;
            this->textOut ( gsOut ) ;
            this->textOut ( L"your choice (y/n): ", false ) ;
            while ( true )
            {
               if ( (this->userResponse ( gsIn )) == OK )
               {
                  if ( (gsIn.compare( dToken, dToken_len )) == ZERO )
                     resp = L'n' ;     // default is 'No'
                  else
                     resp = *gsIn.gstr() ;

                  if ( resp == L'y' || resp == L'Y' )
                  { modTable = true ; break ; }
                  else if ( resp == L'n' || resp == L'N' )
                  { modTable = false ; break ; }
                  else
                  { this->textOut (  L"'y' or 'n' only, your choice?: ", false ) ; }
               }
            }
         }
      }

      if ( modTable != false && status == OK )
      {
         gst = gstab ;
         gst.limitCharacters( wi + tabPlain_len - 1 ) ;
         gst.append( bdrCLASS ) ;
         if ( gstab.gschars() > (tabPlain_len + 1) )  // if data follows the tag
         {
            gString gs( gstab ) ;
            gs.shiftChars( ZERO - tabPlain_len ) ;
            gst.append( gs.gstr() ) ;
         }
         this->ppfWriteLine ( gst ) ;        // write the modified header line
         if ( this->tabPrompt != false || resp == L'a' ) // write the 'next' line
            this->ppfWriteLine ( gsNext ) ;
      }
      else
      {
         this->ppfWriteLine ( gstab ) ;      // write the unmodified header line
         if ( status == OK )                 // write the 'next' line
            this->ppfWriteLine ( gsNext ) ;
      }
   }

   //* If cartouche objects are to be processed *
   else if ( (this->no_cart == false ) &&
             ((wcsncasecmp ( &wptr[wi], tabCartouche, tabCartouche_len)) == ZERO) )
   {
         gst = gstab ;
         gst.limitCharacters( wi + tabCartMin_len ) ;
         gst.append( &gstab.gstr()[tabCartTail_len] ) ;
         this->ppfWriteLine ( gst ) ;
   }
   else     // write the unmodified line
   {
      this->ppfWriteLine ( gstab ) ;
   }
   return status ;

}  //* End ppfProcTAB() *

//*************************
//*      ppfProcHEAD      *
//*************************
//******************************************************************************
//* If the '--no_meta' option OR the '--no_links' option OR the '--no_head'    *
//* option has been invoked, then this method is called to retain the entry.   *
//* See ppfProcessSrcHTML method for element selection.                        *
//*                                                                            *
//* Although the default entries are single-line entries, IF the source        *
//* document was generated with custom data, then the entry can span multiple  *
//* lines.                                                                     *
//*                                                                            *
//* Input  : (by reference) contains the first line of the entry to be retained*
//*                                                                            *
//* Returns: OK if successful, ERR processing error                            *
//******************************************************************************

short Idpp::ppfProcHEAD ( const gString& gsmeta )
{
   const wchar_t endMeta = L'>' ;

   gString gs ;            // additional source data
   bool done = false ;     // loop control
   short status = OK ;     // return value

   //* Write the first (and possibly only) line of the entry.*
   this->ppfWriteLine ( gsmeta.ustr() ) ;

   //* Scan the first line for end-of-entry character.*
   const wchar_t* wptr = gsmeta.gstr() ;
   short wi ;
   for ( wi = ZERO ; wptr[wi] != NULLCHAR ; wi++ )
   {
      if ( wptr[wi] == endMeta )
      {
         done = true ;
         break ;
      }
   }

   //* If metadata entry not yet terminated, scan and copy additional lines.*
   while ( ! done )
   {
      status = this->ppfReadSrcLine ( gs, wptr, wi ) ;   // read from source
      this->ppfWriteLine ( gs.ustr() ) ;             // write to target
      if ( status == OK )
      {  //* Scan the line for end-of-entry character.*
         for ( ; wptr[wi] != NULLCHAR ; wi++ )
         {
            if ( wptr[wi] == endMeta )
            {
               done = true ;
               break ;
            }
         }
      }
      else     // read error
      {
         status = ERR ;
         break ;
      }
   }
   return status ;

}  //* End ppfProcHEAD() *

//*************************
//*       ppfProcUL       *
//*************************
//******************************************************************************
//* Called by ppfProcessSrcHTML() and by ppfProcOL() to process <ul> lists.    *
//* Also called recursively.                                                   *
//*                                                                            *
//* Note that only <ul> blocks declared with the "no-bullet" class are actively*
//* processed. Other <ul> blocks will be processed as plain text.              *
//*                                                                            *
//* Input  : gsul   : (by reference) contains the first line of the list:      *
//*                   '<ul class="no-bullet">'                                 *
//*                                                                            *
//* Returns: OK if successful, ERR processing error                            *
//******************************************************************************
//* Programmer's Notes:                                                        *
//* 1) Because lists may be embedded within other lists, this method may be    *
//*    called recursively, so be sure all data are safely on the stack.        *
//*                                                                            *
//* 2) The HTML source may substitute a defined character for the actual       *
//*    character specified by the texi source. For instance, the HTML converter*
//*    substitutes '&gt;' for the '>' character when it is used as a bullet in *
//*    the texi source. This will cause our post-processing to insert the      *
//*    'square-bullet class as planned, but without additional care, there will*
//*    be artifacts left behind in the line item.                              *
//*    - If we find a '&' character where the embedded bullet character should *
//*      be, we try to determine if we have a '&nnn;' sequence, and if we do,  *
//*      we remove it and substitute a dummy bullet which will get converted   *
//*      to the square-bullet class.                                           *
//*    - If we cannot verify that it is a '&nnn;' sequence, we can't modify    *
//*      the list.                                                             *
//*                                                                            *
//* 3) Also, the HTML converter (stupidly) inserts an HTML comment if:         *
//*     texi source:  @itemize @w{}                                            *
//*     HTML output:  <li><!-- /@w --> ...                                     *
//*    We strip out the unnecessary comment to create a true 'no-bullet' list. *
//*    While it is not technically necessary to do this, this kind of garbage  *
//*    in our output offends us.                                               *
//*                                                                            *
//* 4) In rare cases, a list can end with the 'ulVlongEnd' sequence or a       *
//*    'ulVplongEnd' sequence. This happens when there is a block construct    *
//*    at the end of an itemized list.                                         *
//*                                                                            *
//*                                                                            *
//******************************************************************************

short Idpp::ppfProcUL ( gString& gsul )
{
   const wchar_t* ulparaNobull = L"</p><ul class=\"no-bullet\">" ;
   const short    ulparaNobull_len = 26 ;
   const wchar_t* ulpreNobull = L"</pre><ul class=\"no-bullet\">" ;
   const short    ulpreNobull_len = 28 ;
   const wchar_t* sComment = L"<!-- /@w -->" ;
   const short sComment_len = 12 ;
   const wchar_t* bullDEF = L"&bull;" ;
   const short    bullDEF_len = 6 ;
   const wchar_t* degDEF = L"&deg;" ;
   const short    degDEF_len = 5 ;

   const wchar_t bigDISC = (wchar_t)0x25CF ;    // (●) black-circle
   const wchar_t medDISC = (wchar_t)0x26AB ;    // (⚫) medium-black-circle (texi macro @BDISC)
   const wchar_t smaDISC = (wchar_t)0x2219 ;    // (∙) bullet operator (math)
   const wchar_t dscBULLET = (wchar_t)0x2022 ;  // (texi @bullet command)
   const wchar_t bigCIRCLE = (wchar_t)0x25CB ;  // (○) white-circle
   const wchar_t medCIRCLE = (wchar_t)0x26AA ;  // (⚪) medium-white-circle (texi macro @BCIRCLE)
   const wchar_t msmCIRCLE = (wchar_t)0x26AC ;  // (⚬) medium-small-white-circle
   const wchar_t smaCIRCLE = (wchar_t)0x25E6 ;  // (◦) white bullet (small white circle)
   const wchar_t degBULLET = (wchar_t)0x0176 ;  // (texi @textdegree command)
//   const wchar_t bigSQUARE = (wchar_t)0x25A0 ;  // (■) black-square
//   const wchar_t medSQUARE = (wchar_t)0x25FC ;  // (◼) black-medium-square
//   const wchar_t msmSQUARE = (wchar_t)0x25FE ;  // (◾) black-medium-small-square
   const wchar_t squBULLET = (wchar_t)0x25AA ;  // (▪) black-small-square (texi macro @BSQUARE)

   gString gs, gst ;          // conversion to wide text
   gString gsVerb ;           // for 'verbose' output
   wchar_t bullChar = ZERO ;  // embedded bullet character
   const wchar_t* wptr ;      // pointer and index to wide text
   short wi ;
   short lineItems = ZERO ;   // count the <li> tags
   wchar_t bType ;            // used for removing extra white space
   bool  procList = !this->no_bull ; // process control flag
   bool  pul = false ;        // 'true' if long <ul> line
   short status = OK ;        // return value

   if ( this->verbose )    // verbose diagnostics
   {
      short i = ZERO, bLine = this->tlCount + 1 ;
      while ( gsul.gstr()[i] == SPACE )
         ++i ;
      gsVerb.compose( L"(%4hu) '%S' ==>> ", &bLine, &gsul.gstr()[i] ) ;
   }

   //* If <ul> list processing is active *
   if ( procList != false )
   {  //* Test whether this particular list should be processed. *
      const wchar_t* gsulPtr = gsul.gstr() ;
      short ui = ZERO ;
      while ( gsulPtr[ui] == SPACE )
         ++ui ;
      if ( !(((wcsncasecmp ( &gsulPtr[ui], ulNobull, ulNobull_len)) == ZERO)
             && (gsulPtr[ui + ulNobull_len] == NULLCHAR))
           &&
           !(((wcsncasecmp ( &gsulPtr[ui], ulparaNobull, ulparaNobull_len)) == ZERO)
             && (gsulPtr[ui + ulparaNobull_len] == NULLCHAR))
           &&
           !(((wcsncasecmp ( &gsulPtr[ui], ulpreNobull, ulpreNobull_len)) == ZERO)
             && (gsulPtr[ui + ulpreNobull_len] == NULLCHAR)) )
      {
         procList = false ;
      }

      //* If <ul> list processing is STILL active, *
      //* determine where the <ul> tag begins.     *
      else if ( (wcsncasecmp ( &gsulPtr[ui], paraEnd, paraEnd_len)) == ZERO )
         pul = true ;
   }
   //* If <ul> list processing is not active, OR if this particular *
   //* list is not to be processed, then output line unmodified.    *
   if ( procList == false )
      this->ppfWriteLine ( gsul ) ;

   bool done = false ;
   do
   {
      //*************************************************
      //*     Read lines from the source document.      *
      //*************************************************
      if ( (status = this->ppfReadSrcLine ( gs, wptr, wi )) == OK )
      {
         //* If at the end of the list, return to caller *
         if ( (this->ppfEndBlock ( gs, ulEnd, ulEnd_len )) != false )
         {
            this->ppfWriteLine ( gs ) ;   // pass line through unmodified
            if ( this->verbose )          // verbose diagnostics
            {
               gString gsv( " (%hd items)", &lineItems ) ;
               gsVerb.append( gsv.gstr() ) ;
               if ( procList )
                  this->textOut ( gsVerb ) ;
            }
            done = true ;     // return to caller
            break ;
         }

         //* Test for a <ul> list nested within current list.*
         else if ( (this->ppfTestUlistBegin ( &wptr[wi] )) != false )
         {
            status = this->ppfProcUL ( gs ) ;
         }

         //* Test for an <ol> list nested within the current list *
         else if ( (this->ppfTestOlistBegin ( &wptr[wi] )) != false )
         {
            status = this->ppfProcOL ( gs ) ;
         }

         //* If there is a preformatted block inside the list *
         else if ( (this->ppfTestFormattedBlock ( bType, &wptr[wi] )) != false )
         {
            //* Copy the block, optionally eliminating the unnecessary *
            //* leading blank line. NOTE: Data inside the block        *
            //* IS NOT modified.                                       *
            status = this->ppfProcFormattedBlock ( bType, gs ) ;
         }

         //* If <ul> list processing is active AND if we have a line item *
         else if ( (procList != false) &&
                   (((wcsncasecmp ( &wptr[wi], liLongBegin, liLongBegin_len)) == ZERO)
                   ||
                    ((wcsncasecmp ( &wptr[wi], liBegin, liBegin_len)) == ZERO)) )
         {  //* Push the index past the <li> tag *
            if ( (wcsncasecmp ( &wptr[wi], liBegin, liBegin_len)) == ZERO )
               wi += liBegin_len ;
            else
               wi += liLongBegin_len ;
            //*************************************************
            //* If first line item, extract the embedded      *
            //* bullet character and modify the class called  *
            //* out by the <ul> tag.                          *
            //*************************************************
            if ( ++lineItems == 1 )
            {
               bullChar = wptr[wi] ;      // copy of embedded bullet character
               if ( bullChar == L'<' || bullChar == L'&' )
               {
                  //* For some reason, if source specifies a @itemize command  *
                  //* with a true 'no-bullet' bullet, then the texi-to-HTML    *
                  //* converter inserts an HTML comment. (see note above)      *
                  if ( (wcsncasecmp ( &wptr[wi], sComment, sComment_len)) == ZERO )
                  {  //* Delete the comment + one SPACE character *
                     gst = gs ;                       // working copy of line item
                     gst.limitCharacters( wi ) ;      // save the <li> tag
                     gst.append( &wptr[wi + sComment_len] ) ;
                     gs = gst ;
                     bullChar = SPACE ;   // indicate a true 'no-bullet' list
                  }

                  //* Embedded bullet in the source has been written to HTML   *
                  //* as a defined (or literal hex) character. We can't test   *
                  //* for all possible defined characters, but we handle what  *
                  //* we know, and for the rest, we insert a dummy character   *
                  //* and allow it to default to the 'square-bullet' class.    *
                  //* Note that if source uses a '&' character as a bullet,    *
                  //* then it will appear in the HTML as '&amp;'               *
                  else if ( bullChar == L'&' )
                  {
                     //* Defined character '&deg;' is generated by a *
                     //* @textdegree in source.                      *
                     if ( (wcsncasecmp ( &wptr[wi], degDEF, degDEF_len)) == ZERO )
                     {
                        gst = gs ;                 // working copy of line item
                        gst.limitCharacters( wi ) ;// save the <li> tag
                        gst.append( L'-' ) ;       // insert dummy bullet
                        gst.append( &wptr[wi + degDEF_len] ) ;
                        gs = gst ;
                        bullChar = degBULLET ;
                     }
                     //* Defined character '&bull;' is generated by an  *
                     //* actual bullet character (U+2022) in the source.*
                     else if ( (wcsncasecmp ( &wptr[wi], bullDEF, bullDEF_len)) == ZERO )
                     {
                        gst = gs ;                 // working copy of line item
                        gst.limitCharacters( wi ) ;// save the <li> tag
                        gst.append( L'-' ) ;       // insert dummy bullet
                        gst.append( &wptr[wi + bullDEF_len] ) ;
                        gs = gst ;
                        bullChar = dscBULLET ;
                     }

                     //* Else, is likely an unknown defined character of the   *
                     //* form '&xxxx;'. Delete the sequence and substitute a   *
                     //* dummy character which will be converted to            *
                     //* 'square-bullet' class below.                          *
                     else
                     {
                        procList = false ;// in case we can't parse the item
                        for ( short i = wi ; wptr[i] != NULLCHAR ; i++ )
                        {
                           if ( wptr[i] == L';' )        // defined character identified
                           {
                              gst = gs ;                 // working copy of line item
                              gst.limitCharacters( wi ) ;// save the <li> tag
                              gst.append( L'-' ) ;       // insert dummy bullet
                              gst.append( &wptr[i + 1] ) ;
                              gs = gst ;
                              bullChar = squBULLET ; // replaces the defined-char bullet
                              procList = true ;
                              break ;
                           }
                        }
                     }
                  }
                  else     // cannot parse line items for this list
                  { bullChar = SPACE ; procList = false ; }
               }

               //* True 'no-bullet' bullet *
               if ( bullChar == SPACE )
               { /* leave the <ul class="no-bullet"> tag unchanged.*/ }

               //* This includes the character generated by @bullet command *
               //* as well as various disc-bullet characters.               *
               else if ( bullChar == dscBULLET || bullChar == bigDISC ||
                         bullChar == medDISC   || bullChar == smaDISC )
               {
                  bullChar = medDISC ; // indicator for subsequent line items
                  gsul = ulDISC ;      // specify the bullet class
                  if ( pul != false )  // if leading paragraph close needed
                     gsul.insert( paraEnd ) ;
               }

               //* This includes the character generated by @textdegree *
               //* command as well as various circle-bullet characters. *
               else if ( bullChar == degBULLET || bullChar == bigCIRCLE ||
                         bullChar == medCIRCLE || bullChar == msmCIRCLE ||
                         bullChar == smaCIRCLE )
               {
                  bullChar = medCIRCLE ; // indicator for subsequent line items
                  gsul = ulCIRCLE ;    // specify the bullet class
                  if ( pul != false )  // if leading paragraph close needed
                     gsul.insert( paraEnd ) ;
               }

               else  //* For all other bullet types, use 'square-bullet' class.*
               {
                  bullChar = squBULLET ; // indicator for subsequent line items
                  gsul = ulSQUARE ;    // specify the bullet class
                  if ( pul != false )  // if leading paragraph close needed
                     gsul.insert( paraEnd ) ;
               }

               //* Remove the embedded bullet character *
               if ( bullChar != SPACE )
               {
                  gst = gs ;                    // working copy of line item
                  gst.limitCharacters( wi ) ;   // save the <li> tag
                  gst.append( &wptr[wi + 2] ) ; // discard the embedded bullet
                  gs = gst ;
               }
               this->ppfWriteLine ( gsul ) ;    // output the <ul> tag

               if ( this->verbose )    // verbose diagnostics
               {
                  short i = ZERO ;
                  while ( gsul.gstr()[i] == SPACE )
                     ++i ;
                  gString gsv( "type:%C (U+%04X) '%S'", &bullChar, &bullChar, &gsul.gstr()[i] ) ;
                  gsVerb.append( gsv.gstr() ) ;
               }
            }

            //*************************************************
            //* Second and subsequent line items,             *
            //* delete the embedded bullet character.         *
            //*************************************************
            else
            {
               gst = gs ;                    // working copy of line item
               gst.limitCharacters( wi ) ;   // save the <li> tag

               //* If 'no-bullet' bullet, remove the unnecessary comment *
               if ( bullChar == SPACE )
               {
                  if ( (wcsncasecmp ( &wptr[wi], sComment, sComment_len)) == ZERO )
                  {  //* Delete the comment + one SPACE character *
                     gst.append( &wptr[wi + sComment_len + 1] ) ;
                  }
                  else
                  {  //* We should never arrive here, parsing error *
                     gst = gs ;
                     procList = false ;
                  }
               }

               //* If a defined-character bullet, delete it *
               else if ( (bullChar == squBULLET 
                          || bullChar == medCIRCLE 
                          || bullChar == medDISC)
                         && (wptr[wi] == L'&') )
               {
                  bool id = false ; // in the unlikely event that data are corrupt
                  for ( short i = wi ; wptr[i] != NULLCHAR ; i++ )
                  {
                     if ( wptr[i] == L';' )        // defined character identified
                     {
                        gst.append( &wptr[i + 1] ) ;
                        id = true ;
                        break ;
                     }
                  }
                  if ( id == false )   // data corrupt, pass it unmodified
                     gst = gs ;
               }

               //* Single, embedded character, discard it *
               else
                  gst.append( &wptr[wi + 2] ) ;
               gs = gst ;     // copy results back to primary
            }

            this->ppfWriteLine ( gs ) ;   // output the <li> line-item tag
         }

         //* <ul> processing not active OR is not an operator line.*
         //* Output the data line without processing.              *
         else
         {
            this->ppfWriteLine ( gs ) ;
         }
      }
   }
   while ( ! done && status == OK ) ;

   return status ;

}  //* End ppfProcUL() *

//*************************
//*       ppfProcOL       *
//*************************
//******************************************************************************
//* Called by ppfProcessSrcHTML() to process <ol> lists.                       *
//* Also called recursively, and by ppfProcUL().                               *
//*                                                                            *
//* Input  : gsol   : (by reference) contains the first line of the list:      *
//*                   '<ol>' || '</pre><ol>' || '</p><ol>' or similar          *
//*                                                                            *
//* Returns: OK if successful, ERR processing error                            *
//******************************************************************************
//* Programmer's Note:                                                         *
//* 1) Because lists may be embedded within other lists, this method may be    *
//*    called recursively, so be sure all data are safely on the stack.        *
//*                                                                            *
//* 2) Two switches control processing of <ol> lists:                          *
//*    - this->oLists: prompt user for enumeration type and modify the         *
//*      <ol> tag accordingly.                                                 *
//*         Example: <ol class="enum-lower-alpha">                             *
//*    - this->oOffst: prompt user for both enumeration type AND arbitrary     *
//*      point at which to begin the sequence (greater than ZERO), and modify  *
//*      the <ol> tag accordingly.                                             *
//*         Example: <ol class="enum-lower-alpha" start="14">                  *
//*                                                                            *
//* 3) Note that HTML will accept alphabetical offsets of any reasonable size  *
//*    and display the sequence as: ... y, z, aa, ab, ac, ...                  *
//*                                                                            *
//* 4) GPL and FDL license text:                                               *
//*    Please see ppfProgGNU() method for a discussion of processing these     *
//*    special enumerated lists.                                               *
//*                                                                            *
//******************************************************************************

short Idpp::ppfProcOL ( gString& gsol )
{
//* Temporarily set this to (1) to disable interactive scan.*
#define DISABLE_INTERACTION (0)

   gString gs, gst,           // conversion to wide text
           gsVerb,            // for 'verbose' output
           gsIn ;             // receives user input
   const wchar_t* wptr ;      // pointer and index to wide text
   short wi ;
   short lineItems = ZERO ;   // count the <li> tags
   wchar_t bType ;            // used for removing extra white space
   short status = OK ;        // return value

   if ( this->verbose )    // verbose diagnostics
   {
      short bLine = this->tlCount + 1 ;
      gsVerb.compose( L"(%4hu) '%S' ==>> ", &bLine, gsol.gstr() ) ;
   }

   //* If list is not to be processed, output the original <ol> tag.*
   if ( this->oLists == false )
   {
      this->ppfWriteLine ( gsol ) ;

      if ( this->verbose )
         gsVerb.append( gsol.gstr() ) ;
   }

   bool done = false ;
   do
   {  //* Read lines from source file.*
      if ( (status = this->ppfReadSrcLine ( gs, wptr, wi )) == OK )
      {
         //* If at the end of the list, return to caller *
         if ( (this->ppfEndBlock ( gs, olEnd, olEnd_len )) != false )
         {
            this->ppfWriteLine ( gs ) ;   // pass line through unmodified
            if ( this->verbose )       // verbose diagnostics
            {
               gString gsv( " (%hd items)", &lineItems ) ;
               gsVerb.append( gsv.gstr() ) ;
               if ( this->oLists != false )
                  this->textOut ( gsVerb ) ;
            }
            done = true ;
            break ;
         }

         //* Scan for line items *
         if ( (((wcsncasecmp ( &wptr[wi], liLongBegin, liLongBegin_len)) == ZERO)
               ||
               ((wcsncasecmp ( &wptr[wi], liBegin, liBegin_len)) == ZERO)) )
         {
            ++lineItems ;              // count the line items

            //* If we are processing the list, ask user what to do *
            if ( this->oLists && lineItems == 1 )
            {
               wchar_t eType = L'd' ;
               short   eStart = 1 ;

               if ( this->textMode )
               {
                  #if DISABLE_INTERACTION == 0
                  gst.compose( L"____________________________\n"
                                "Enumerated List: Line: %hu\n"
                                "First Line Item: %S\n"
                                "Choose Enumeration type (decimal is the default)\n"
                                "d:decimal          D:leading-zero decimal\n"
                                "a:lower case alpha A:upper case alpha\n"
                                "r:lower case Roman R:upper case Roman\n"
                                "g:lower case Greek c:custom enumeration",
                               &this->slCount, &wptr[wi] ) ;
                  this->textOut ( gst ) ;
                  this->textOut ( L"your choice: ", false ) ;

                  while ( true )
                  {
                     if ( (this->userResponse ( gsIn )) == OK )
                     {
                        if ( (gsIn.compare( dToken, dToken_len )) == ZERO )
                           eType = L'd' ;     // default is 'd'ecimal
                        else
                           eType = *gsIn.gstr() ;

                        if ( eType == L'd' || eType == L'D' || 
                             eType == L'a' || eType == L'A' || 
                             eType == L'r' || eType == L'R' || 
                             eType == L'g' || eType == L'c' )
                           break ;
                        else
                           this->textOut ( L"Invalid selection, your choice?: ", false ) ;
                     }
                  }
                  if ( this->oOffst )
                  {
                     this->textOut ( L"Sequence start(1-n): ", false ) ;
                     while ( true )
                     {
                        if ( (this->userResponse ( gsIn )) == OK )
                        {
                           if ( (gsIn.compare( dToken, dToken_len )) == ZERO )
                              gsIn = L"1" ;       // default is '1'
                           eStart = (-1) ;
                           swscanf ( gsIn.gstr(), L"%hd", &eStart ) ;

                           //* Note that user is prompted to enter a         *
                           //* value >= 1, but we allow a zero value IF      *
                           //* enumeration type is decimal. This is done     *
                           //* primarily to accomodate the fact that the GPL *
                           //* and FDL license documents start with '0'.     *
                           if ( (eStart >= 1)
                                || (eStart == ZERO && eType == L'd') )
                           {
                              break ;
                           }
                           this->textOut ( L"Positive integer only: ", false ) ;
                        }
                     }
                  }
                  #endif   // DISABLE_INTERACTION
               }

               gsol.limitCharacters( gsol.gschars() - 2 ) ;
               switch ( eType )
               {
                  case 'D': gsol.append( L" class=\"enum-decimal-zero\">" ) ; break ;
                  case 'a': gsol.append( L" class=\"enum-lower-alpha\">" ) ;  break ;
                  case 'A': gsol.append( L" class=\"enum-upper-alpha\">" ) ;  break ;
                  case 'r': gsol.append( L" class=\"enum-lower-roman\">" ) ;  break ;
                  case 'R': gsol.append( L" class=\"enum-upper-roman\">" ) ;  break ;
                  case 'g': gsol.append( L" class=\"enum-lower-greek\">" ) ;  break ;
                  case 'c': gsol.append( L" class=\"enum-custom\">" ) ;       break ;
                  case 'd': default: gsol.append( L">" ) ;                    break ;
               }

               //* Insert the starting offset if specified *
               if ( this->oOffst )
               {
                  gst.compose( L" start=\"%hd\"", &eStart ) ;
                  gsol.insert( gst.gstr(), (gsol.gschars() - 2) ) ;
               }

               //* Write the modified <ol> tag *
               this->ppfWriteLine ( gsol ) ;

               if ( this->verbose )
                  gsVerb.append( gsol.gstr() ) ;
            }
            this->ppfWriteLine ( gs ) ;   // pass line through unmodified
         }

         //* Test for list embedded within current list *
         else if ( (this->ppfTestUlistBegin ( &wptr[wi] )) != false )
         {
            status = this->ppfProcUL ( gs ) ;
         }

         else if ( (this->ppfTestOlistBegin ( &wptr[wi] )) != false )
         {
            status = this->ppfProcOL ( gs ) ;
         }

         //* If there is a preformatted block inside the list *
         else if ( (this->ppfTestFormattedBlock ( bType, &wptr[wi] )) != false )
         {
            //* Copy the block, optionally eliminating the unnecessary leading *
            //* blank line. NOTE: Data inside the block IS NOT modified.       *
            status = this->ppfProcFormattedBlock ( bType, gs ) ;
         }

         //* Process line as text data *
         else
         {
            this->ppfWriteLine ( gs ) ;   // pass line through unmodified
         }
      }
      else  done = true ;     // read error, abort the loop
   }
   while ( ! done ) ;

   return status ;

#undef DISABLE_INTERACTION
}  //* End ppfProcOL() *

//*************************
//*   ppfTestUlistBegin   *
//*************************
//******************************************************************************
//* Test the provided text for the beginning of a <ul> block.                  *
//*    See notes below.                                                        *
//*                                                                            *
//* Input  : wptr  : pointer to beginning of non-whitespace data               *
//*                  for current source line                                   *
//*                                                                            *
//* Returns: 'true'  if line contains a <ul class="no-bullet'> tag             *
//*          'false' otherwise                                                 *
//******************************************************************************
//* Programmer's Notes:                                                        *
//* 1) Normally, a <ul> tag is alone on a line, but in some cases it could be  *
//*    preceeded by a </p> (close paragraph) tag or a </pre> (close            *
//*    preformatted) or other data. This happens if the <ul> is immediately    *
//*    preceeded by one of the block commands.                                 *
//*                                                                            *
//* 2) We must distinguish between <ul> lists generated by the texi-to-HTML    *
//*    converter and HTML code embedded within the texi source document.       *
//*    There is no way to be 100% certain about this, but we "know" that the   *
//*    converter always ends the first line of the list immediately after      *
//*    the opening tag. Thus if there is data following the opening tag on the *
//*    same line, we assume that the list WAS NOT generated by the converter   *
//*    (or that it has already been post-processed).                           *
//*                                                                            *
//* 3) It is also remotely possible (though stupid) that the source document   *
//*    will have a list INSIDE a preformatted block. In this case, we could    *
//*    have some other end tag(s) leading the <ul>.  For example: </pre></div> *
//*    Here is the unbearable HTML for an @itemize inside a @display block:    *
//*                                                                            *
//*    <div class="display">                                                   *
//*    <pre class="display">Here we are inside an @display block.              *
//*    </pre><ul>                                                              *
//*    <li> <pre class="display">Bathe.                                        *
//*    </pre></li><li> <pre class="display">Brush your teeth.                  *
//*    </pre></li><li> <pre class="display">Shave unwanted hair.               *
//*    </pre></li></ul>                                                        *
//*    <pre class="display">Here we are inside an @display block.              *
//*    </pre></div>                                                            *
//*                                                                            *
//*    Note that we WILL identify the <ul> tag in the above mess; however, the *
//*    caller may choose not to process it.                                    *
//******************************************************************************

bool Idpp::ppfTestUlistBegin ( const wchar_t* wptr )
{
   bool ulListFound = false ;
   for ( short wi = ZERO ; wptr[wi] != NULLCHAR ; wi++ )
   {
      if ( wptr[wi] == L'<' )
      {
         if ( (wcsncasecmp ( &wptr[wi], ulBegin, ulBegin_len)) == ZERO )
         {
            while ( wptr[++wi] != L'>' && wptr[wi] != NULLCHAR ) ; 
            if ( wptr[wi] == L'>' && wptr[wi + 1] == NULLCHAR)
            {
               ulListFound = true ;
               break ;
            }
         }
      }
   }
   return ulListFound ;

}  //* End ppfTestUlistBegin() *

//*************************
//*   ppfTestOlistBegin   *
//*************************
//******************************************************************************
//* Test the provided text for the beginning of an UNPROCESSED <ol> block.     *
//*                                                                            *
//* Input  : wptr  : pointer to beginning of non-whitespace data               *
//*                  for current source line                                   *
//*                                                                            *
//* Returns: 'true'  if line contains a plain (unprocessed) <ol> tag           *
//*          'false' otherwise                                                 *
//******************************************************************************
//* Programmer's Notes:                                                        *
//* 1) Normally, an <ol> tag is alone on a line, but in rare cases it could be *
//*    preceeded by a </p> (close paragraph) tag or a </pre> (close            *
//*    preformatted). This happens if the <ol> is immediately preceeded by     *
//*    one of the block commands.                                              *
//* 2) Currently, (makeinfo 5.2), all <ol> tags are declared without a class,  *
//*    so if it has a class, then it has already been processed.               *
//* 3) Also, unprocessed <ol> tags are the last thing on the source line, so   *
//*    if there is additional data AFTER the <ol> tag, then the list has       *
//*    already been processed.                                                 *
//* 4) We COULD BE fooled by HTML embedded directly into the texi source, but  *
//*    if we are fooled, processing the list will not change its appearance    *
//*    unless user selects a non-default formatting option.                    *
//******************************************************************************

bool Idpp::ppfTestOlistBegin ( const wchar_t* wptr )
{
   bool olListFound = false ;

   for ( short wi = ZERO ; wptr[wi] != NULLCHAR ; wi++ )
   {
      if ( wptr[wi] == L'<' )
      {
         if ( ((wcsncasecmp ( &wptr[wi], olBegin, olBegin_len)) == ZERO)
              && (wptr[wi + olBegin_len] == NULLCHAR) )
         {
            olListFound = true ;
            break ;
         }
      }
   }
   return olListFound ;

}  //* End ppfTestOlistBegin() *

//**************************
//* ppfTestFormattedBlock  *
//**************************
//******************************************************************************
//* Test whether the provided data indicates the beginning of a class block    *
//* for 'format', 'display' 'example' or 'lisp', or their 'small' counterparts.*.                                                                        *
//*                                                                            *
//* Input  : bType  : (by reference) recieves indicator of block type:         *
//*                   'F' == 'format' class                                    *
//*                   'f' == 'smallformat' class                               *
//*                   'D' == 'display' class                                   *
//*                   'd' == 'smalldisplay' class                              *
//*                   'E' == 'example' class                                   *
//*                   'e' == 'smallexample' class                              *
//*                   'L' == 'lisp' class                                      *
//*                   'l' == 'smalllisp' class                                 *
//*          wptr   : const pointer to data for first line of the block:       *
//*                      <div class="xxxx">                                    *
//*                      or: </p><div class="xxxx">                            *
//*                      or: unrelated data                                    *
//*                                                                            *
//* Returns: 'true' if this is a preformatted block header, else 'false'       *
//******************************************************************************
//* Programmer's Note:                                                         *
//* A block declaration MAY BE preceeded by a '</p>' tag, so we test for both  *
//* with and without this extra tag.                                           *
//*                                                                            *
//?   //* If the block declaration has already been processed, it will probably be   *
      //* followed on the same line by the <pre class="xxxx"> declaration.           *
      //* We must capture this to avoid having the caller try to process the contents*
      //* of the block, which could easily lead to parsing problems.                 *
//******************************************************************************

bool Idpp::ppfTestFormattedBlock ( wchar_t& bType, const wchar_t* wptr )
{
   const wchar_t* divclassBegin = L"<div class=" ;
   const short    divclassBegin_len = 11 ;
   const wchar_t* fClass = L"\"format\">" ;
   const short    fClass_len = 9 ;
   const wchar_t* sfClass = L"\"smallformat\">" ;
   const short    sfClass_len = 14 ;
   const wchar_t* dClass = L"\"display\">" ;
   const short    dClass_len = 10 ;
   const wchar_t* sdClass = L"\"smalldisplay\">" ;
   const short    sdClass_len = 15 ;
   const wchar_t* eClass = L"\"example\">" ;
   const short    eClass_len = 10 ;
   const wchar_t* seClass = L"\"smallexample\">" ;
   const short    seClass_len = 15 ;
   const wchar_t* lClass = L"\"lisp\">" ;
   const short    lClass_len = 7 ;
   const wchar_t* slClass = L"\"smalllisp\">" ;
   const short    slClass_len = 12 ;

   bool fmtBlock = false ;    // return value
   bType = L'?' ;             // initialize caller's variable

   for ( short wi = ZERO ; wptr[wi] != NULLCHAR ; wi++ )
   {
      if ( wptr[wi] == L'<' )
      {
         if ( (wcsncasecmp ( &wptr[wi], divclassBegin, divclassBegin_len)) == ZERO )
         {
            fmtBlock = true ;    // tentatively identified
            wi += divclassBegin_len ;

            //* Now check for type of block *
            if ( (wcsncasecmp ( &wptr[wi], fClass, fClass_len)) == ZERO )
               bType = L'F' ;    // 'format' class block
            else if ( (wcsncasecmp ( &wptr[wi], sfClass, sfClass_len)) == ZERO )
               bType = L'f' ;    // 'smallformat' class block
            else if ( (wcsncasecmp ( &wptr[wi], dClass, dClass_len)) == ZERO )
               bType = L'D' ;    // 'display' class block
            else if ( (wcsncasecmp ( &wptr[wi], sdClass, sdClass_len)) == ZERO )
               bType = L'd' ;    // 'smalldisplay' class block
            else if ( (wcsncasecmp ( &wptr[wi], eClass, eClass_len)) == ZERO )
               bType = L'E' ;    // 'example' class block
            else if ( (wcsncasecmp ( &wptr[wi], seClass, seClass_len)) == ZERO )
               bType = L'e' ;    // 'smallexample' class block
            else if ( (wcsncasecmp ( &wptr[wi], lClass, lClass_len)) == ZERO )
               bType = L'L' ;    // 'lisp' class block
            else if ( (wcsncasecmp ( &wptr[wi], slClass, slClass_len)) == ZERO )
               bType = L'l' ;    // 'smalllisp' class block
// SHOULD WE ALSO TEST FOR 'verbatim' BLOCKS?
            else           // line does not delimit a formatted data block
               fmtBlock = false ;
         }
      }
   }
   return fmtBlock ;

}  //* End ppfTestFormattedBlock() *

//**************************
//* ppfProcFormattedBlock  *
//**************************
//******************************************************************************
//* Copy the contents of a pre-formatted block from source to target.          *
//* Optionally, eliminate the extra blank line before the block by combining   *
//* the <div> and <pre> tags onto a single line.                               *
//*                                                                            *
//*      CONTENTS OF PREFORMATTED BLOCKS ARE COPIED AS PLAIN TEXT.             *
//*                                                                            *
//* Input  : bType  : indicates block type:                                    *
//*                   (see 'ppfTestFormattedBlock' method for list)            *
//*          gsb    : (by reference) contains the first line of the block.     *
//*                                                                            *
//* Returns: OK if successful, ERR processing error                            *
//******************************************************************************
//* Programmer's Note:                                                         *
//* The reason we copy the contents of preformatted blocks as a separate task  *
//* is to avoid processing <ul> and <ol> lists that may be inside these blocks,*
//* as lists, AND to count nested preformatted blocks.                         *
//*                                                                            *
//* In our view, putting <ul> and <ol> lists or other formatted constructs     *
//* inside a preformatted block is illogical and unreasonable; however, if it  *
//* CAN be done, then some fool WILL do it.                                    *
//*                                                                            *
//* Thus, if we don't explicity state that we are refusing to process such     *
//* such foolishness, then we need to capture it and minimize the ugliness.    *
//* Note that WE DO NOT PASS THE LISTS to list-processing methods because the  *
//* tortured formatting would require much more detailed parsing than we       *
//* currently perform. So, whatever the type of list, it remains as written.   *
//*                --   --   --   --   --   --   --   --                       *
//*                                                                            *
//*                                                                            *
//******************************************************************************

short Idpp::ppfProcFormattedBlock ( wchar_t bType, gString& gsb )
{
#define DEBUG_PFB (0)      // for debugging only
#if DEBUG_PFB != 0
short fLine = this->slCount ; // remember first line
#endif   // DEBUG_PFB

   const wchar_t* blockEND = L"</div>" ;
   const short    blockEND_len = 6 ;

   gString gs ;               // source line data
   const wchar_t* wptr ;      // wide-string pointer
   wchar_t eType ;            // if nested block, its type
   short wi,                  // string index
         nestCount = 1,       // depth of nesting
         listCount = ZERO,    // embedded <ul> and <ol> lists
         status = OK ;        // return value

   //* If specified, eliminate the leading blank line *
   if ( this->no_bloc == false )
      status = ppfProcBlockWhitespace ( bType, gsb ) ;

   //* Else, copy the header line as-is *
   else
      this->ppfWriteLine ( gsb ) ;

   bool done = false ;
   while ( ! done && status == OK )
   {
      //*************************************************
      //*     Read lines from the source document.      *
      //*************************************************
      if ( (status = this->ppfReadSrcLine ( gs, wptr, wi )) == OK )
      {
         //* If end of block, then return to caller *
         if ( (this->ppfEndBlock ( gs, blockEND, blockEND_len )) != false )
         {
            this->ppfWriteLine ( gs ) ;      // write line to target
            if ( --nestCount <= ZERO )
            {
               #if DEBUG_PFB != 0      // for debugging only
               if ( this->verbose != false )
               {
                  const wchar_t* bClass ;
                  switch ( bType )
                  {
                     case L'F':  bClass = L"format" ;       break ;
                     case L'D':  bClass = L"display" ;      break ;
                     case L'E':  bClass = L"example" ;      break ;
                     case L'L':  bClass = L"lisp" ;         break ;
                     case L'f':  bClass = L"smallformat" ;  break ;
                     case L'd':  bClass = L"smalldisplay" ; break ;
                     case L'e':  bClass = L"smallexample" ; break ;
                     case L'l':  bClass = L"smalllisp" ;    break ;
                  }
                  gString gsVerb( "(%4hd -%4hd) Formatted Block('%S')", 
                                  &fLine, &this->slCount, bClass ) ;
                     this->textOut ( gsVerb ) ;
               }
               #endif   // DEBUG_PFB

               done = true ;
               break ;
            }
            else  // end of nested block found, keep processing
               continue ;
         }

         //* Test for <ol>, <ul> lists and other preformatted blocks INSIDE *
         //* this preformatted block. No intelligent user would do this,    *
         //* but if it happens, we need to identify it.                     *
         if ( ((ppfTestOlistBegin ( gs.gstr() )) != false)
              ||
              ((ppfTestUlistBegin ( gs.gstr() )) != false) )
         {
            this->ppfWriteLine ( gs ) ;      // write line to target
            ++listCount ;                    // count embedded lists
         }
         else if ( ((this->ppfEndBlock ( gs, ulEnd, ulEnd_len )) != false)
                   ||
                   ((this->ppfEndBlock ( gs, olEnd, olEnd_len )) != false) )
         {
            this->ppfPFB_List ( gs ) ;       // strip the garbage
            this->ppfWriteLine ( gs ) ;      // write line to target
            --listCount ;                    // count embedded lists
         }
         else if ( (this->ppfTestFormattedBlock ( eType, gs.gstr() )) != false )
         {
            this->ppfWriteLine ( gs ) ;      // write line to target
            ++nestCount ;                    // nesting level
         }
         else
         {
            if ( listCount > ZERO )
               this->ppfPFB_List ( gs ) ;    // strip the garbage
            this->ppfWriteLine ( gs ) ;
         }
      }
   }
   return status ;

#undef DEBUG_PFB
}  //* End ppfProcFormattedBlock() *

//*************************
//*      ppfPFB_List      *
//*************************
//******************************************************************************
//* Called only by ppfProcFormattedBlock(), and ONLY if embedded lists have    *
//* been identified. Removes extra HTML garbage from the lists to make them    *
//* readable.                                                                  *
//*                                                                            *
//* Input  : gsLine : (by reference) contains the line data to be scanned      *
//*                   On return, contains the reformatted data.                *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Idpp::ppfPFB_List ( gString& gsLine )
{
   const wchar_t* pcBEGIN = L" <pre class=" ;
   const short    pcBEGIN_len = 12 ;
   const wchar_t* plBEGIN = L"</pre></li>" ;
   const short    plBEGIN_len = 11 ;
   const short    plSHIFT_len = 6 ;

   const wchar_t* p = gsLine.gstr() ;
   for ( short i = ZERO ; p[i] != NULLCHAR ; i++ )
   {
      //* If there is not enough of the line remaining   *
      //* to hold anything interesting, then we're done. *
      if ( (gsLine.gschars() - i) <= plBEGIN_len )
         break ;

      if ( p[i] == L'<' )
      {
         if ( ((wcsncasecmp ( &p[i], plBEGIN, plBEGIN_len)) == ZERO) )
         {  //* Remove '</pre>' tag *
            gsLine.shiftChars( -(plSHIFT_len) ) ;
            p = gsLine.gstr() ;
            i = -1 ;
            continue ;
         }
         else if ( ((wcsncasecmp ( &p[i], liBegin, liBegin_len)) == ZERO) )
         {
            i += liBegin_len ;
            if ( (wcsncasecmp ( &p[i], pcBEGIN, pcBEGIN_len)) == ZERO )
            {
               gString gt( gsLine.gstr(), i ) ;
               i += pcBEGIN_len ;
               while ( p[i] != L'>' && p[i] != NULLCHAR )
                  ++i ;
               if ( p[i] == L'>' )
                  ++i ;
               gt.append( &p[i] ) ;
               gsLine = gt ;
               break ;
            }
         }
      }
   }

}  //* End ppfPFB_List() *

//**************************
//* ppfProcBlockWhitespace *
//**************************
//******************************************************************************
//* Eliminate the extra blank line before the block by combining               *
//* the <div> and <pre> tags onto a single line:                               *
//* class blocks.                                                              *
//*         '<div class="format">' and '<pre class="format">'                  *
//*         '<div class="smallformat">' and '<pre class="smallformat">'        *
//*         OR                                                                 *
//*         '<div class="display">' and '<pre class="display">'                *
//*         '<div class="smalldisplay">' and '<pre class="smalldisplay">'      *
//*         OR                                                                 *
//*         '<div class="example">' and '<pre class="example">'                *
//*         '<div class="smallexample">' and '<pre class="smallexample">'      *
//*         OR                                                                 *
//*         '<div class="lisp">' and '<pre class="lisp">'                      *
//*         '<div class="smalllisp">' and '<pre class="smalllisp">'            *
//*                                                                            *
// NOTE: If the text of the first line in the block is getting some special    *
//*      formatting, then the '<pre class...' line may not follow immediately, *
//*      so the leading blank line will not be removed in that case.           *
//*                                                                            *
//* Input  : bType  : indicates block type:                                    *
//*                   (see 'ppfTestFormattedBlock' method for list)            *
//*          gsb    : (by reference) contains the first line of the block.     *
//*                                                                            *
//* Returns: OK if successful, ERR processing error                            *
//******************************************************************************

short Idpp::ppfProcBlockWhitespace ( wchar_t bType, gString& gsb )
{
   const wchar_t* formatPRE = L"<pre class=\"format\">" ;
   const short    formatPRE_len = 20 ;
   const wchar_t* smformatPRE = L"<pre class=\"smallformat\">" ;
   const short    smformatPRE_len = 25 ;
   const wchar_t* displayPRE = L"<pre class=\"display\">" ;
   const short    displayPRE_len = 21 ;
   const wchar_t* smdisplayPRE = L"<pre class=\"smalldisplay\">" ;
   const short    smdisplayPRE_len = 26 ;
   const wchar_t* examplePRE = L"<pre class=\"example\">" ;
   const short    examplePRE_len = 21 ;
   const wchar_t* smexamplePRE = L"<pre class=\"smallexample\">" ;
   const short    smexamplePRE_len = 26 ;
   const wchar_t* lispPRE = L"<pre class=\"lisp\">" ;
   const short    lispPRE_len = 18 ;
   const wchar_t* smlispPRE = L"<pre class=\"smalllisp\">" ;
   const short    smlispPRE_len = 23 ;

   gString gs,                // source line data
           gsOut ;            // output data for target file
   const wchar_t* wptr ;      // wide-string pointer
   short wi ;                 // string index
   short status = OK ;        // return value

   if ( (status = this->ppfReadSrcLine ( gs, wptr, wi )) == OK )
   {
      bool repo = false ;     // 'true' if block repositioned

      if ( bType == L'F' )       //* 'format' class block *
      {
         if ( ((wcsncasecmp ( &wptr[wi], formatPRE, formatPRE_len)) == ZERO) )
            { gsOut.compose( L"%S%S", gsb.gstr(), gs.gstr() ) ; repo = true ; }
         else
            gsOut.compose( L"%S\n%S", gsb.gstr(), gs.gstr() ) ;
      }
      else if ( bType == L'f' )  //* 'smallformat' class block *
      {
         if ( ((wcsncasecmp ( &wptr[wi], smformatPRE, smformatPRE_len)) == ZERO) )
            { gsOut.compose( L"%S%S", gsb.gstr(), gs.gstr() ) ; repo = true ; }
         else
            gsOut.compose( L"%S\n%S", gsb.gstr(), gs.gstr() ) ;
      }
      else if ( bType == L'D' )  //* 'display' class block *
      {
         if ( ((wcsncasecmp ( &wptr[wi], displayPRE, displayPRE_len)) == ZERO) )
            { gsOut.compose( L"%S%S", gsb.gstr(), gs.gstr() ) ; repo = true ; }
         else
            gsOut.compose( L"%S\n%S", gsb.gstr(), gs.gstr() ) ;
      }
      else if ( bType == L'd' )  //* 'smalldisplay' class block *
      {
         if ( ((wcsncasecmp ( &wptr[wi], smdisplayPRE, smdisplayPRE_len)) == ZERO) )
            { gsOut.compose( L"%S%S", gsb.gstr(), gs.gstr() ) ; repo = true ; }
         else
            gsOut.compose( L"%S\n%S", gsb.gstr(), gs.gstr() ) ;
      }
      else if ( bType == L'E' )  //* 'example' class block *
      {
         if ( ((wcsncasecmp ( &wptr[wi], examplePRE, examplePRE_len)) == ZERO) )
            { gsOut.compose( L"%S%S", gsb.gstr(), gs.gstr() ) ; repo = true ; }
         else
            gsOut.compose( L"%S\n%S", gsb.gstr(), gs.gstr() ) ;
      }
      else if ( bType == L'e' )  //* 'smallexample' class block *
      {
         if ( ((wcsncasecmp ( &wptr[wi], smexamplePRE, smexamplePRE_len)) == ZERO) )
            { gsOut.compose( L"%S%S", gsb.gstr(), gs.gstr() ) ; repo = true ; }
         else
            gsOut.compose( L"%S\n%S", gsb.gstr(), gs.gstr() ) ;
      }
      else if ( bType == L'L' )  //* 'lisp' class block *
      {
         if ( ((wcsncasecmp ( &wptr[wi], lispPRE, lispPRE_len)) == ZERO) )
            { gsOut.compose( L"%S%S", gsb.gstr(), gs.gstr() ) ; repo = true ; }
         else
            gsOut.compose( L"%S\n%S", gsb.gstr(), gs.gstr() ) ;
      }
      else if ( bType == L'l' )  //* 'smalllisp' class block *
      {
         if ( ((wcsncasecmp ( &wptr[wi], smlispPRE, smlispPRE_len)) == ZERO) )
            { gsOut.compose( L"%S%S", gsb.gstr(), gs.gstr() ) ; repo = true ; }
         else
            gsOut.compose( L"%S\n%S", gsb.gstr(), gs.gstr() ) ;
      }
      else     // this is unlikely and would indicate programmer error
         status = ERR ;

      //* Write the re-formatted data to target *
      if ( status == OK )
      {
         this->ppfWriteLine ( gsOut ) ;
         if ( repo == false )
            ++this->tlCount ; // if we have just output 2 lines in one write

         if ( this->verbose != false && repo != false )
         {
            gString gsVerb( "(%4hd) Formatted Block repositioned.", &this->tlCount ) ;
            this->textOut ( gsVerb ) ;
         }
      }
   }
   else     // read error - save current line
      this->ppfWriteLine ( gsb ) ;
   return status ;

}  //* End ppfProcBlockWhitespace() *

//*************************
//*      ppfProcGNU       *
//*************************
//******************************************************************************
//* Process the GNU General Public License sub-document.                       *
//*  OR                                                                        *
//* Process the GNU Free Documentation License sub-document.                   *
//*                                                                            *
//* 1) Start the top-level enumerated list at item '0' (zero).                 *
//* 2) Convert the three (3) nested lists in the GPL to 'a.' 'b.' 'c.' lists.  *
//* 3) Convert the one (1) nested list in the FDL to an 'A.' 'B.' 'C.' list.   *
//*                                                                            *
//* Input  : gpl    : 'true' if we are processing the GPL text                 *
//*                   'false' if we are processing the FDL text                *
//*                                                                            *
//* Returns: OK if successful, ERR processing error                            *
//******************************************************************************
//* NOTES:                                                                     *
//* -- GPL and FDL license text:                                               *
//*    We silently correct the enumerated lists in these documents if detected.*
//*    Because it is boilerplate AND because by law, we as licensees are not   *
//*    allowed to modify it, we can be reasonably confident that it can be     *
//*    correctly identified. For this reason, our tests for the tags are quite *
//;    simple. However, past or future versions of the licenses MAY cause      *
//*    problems. Also, if the texi-to-HTML converter changes the way it formats*
//*    <ol> lists, we COULD see problems.                                      *
//* -- This code is based on GPL v:3 and FDL v:1.3 and Texinfo v:5.2.          *
//* -- We aren't sure whether to offer the user a flag to disable this         *
//*    operation; so at this time, unless the 'no_mods' flag is set, we        *
//*    correct the lists whether the user wants it or not.                     *
//*                                                                            *
//******************************************************************************

short Idpp::ppfProcGNU ( bool gpl )
{
   const wchar_t* gnuZstart = L"<ol class=\"enum-decimal\" start=\"0\">" ;
   const wchar_t* gnuAstart = L"<ol class=\"enum-upper-alpha\">" ;
   const wchar_t* gnuaStart = L"<ol class=\"enum-lower-alpha\">" ;
   const wchar_t* olLongEnd = L"</li></ol>" ;
   const short    olLongEnd_len = 10 ;

   gString gs ;                     // source data
   const wchar_t* wptr ;            // pointer and index to wide text
   short wi ;
   short subLists = gpl ? 3 : 1 ;   // number of sub-lists to process
   short status = OK ;              // return value

   //* Scan until we reach the first <ol> tag.              *
   //* Set the first <ol> tag as 'enum-decimal' and start=0 *
   bool done = false ;
   do
   {
      if ( (status = this->ppfReadSrcLine ( gs, wptr, wi )) == OK )
      {
         if ( (wcsncasecmp ( wptr, olBegin, olBegin_len)) == ZERO )
         {
            gs = gnuZstart ;
            done = true ;
         }
         this->ppfWriteLine ( gs ) ;   // write the line to target
      }
   }
   while ( ! done && status == OK ) ;

   //* Set the sub-list <ol> tags as 'enum-lower-alpha' *
   while ( status == OK && subLists > ZERO )
   {
      if ( (status = this->ppfReadSrcLine ( gs, wptr, wi )) == OK )
      {
         if ( (wcsncasecmp ( wptr, olBegin, olBegin_len)) == ZERO )
         {
            if ( gpl != false )
               gs = gnuaStart ;
            else
               gs = gnuAstart ;
         }

         else if ( (wcsncasecmp ( wptr, olLongEnd, olLongEnd_len)) == ZERO )
         {
            --subLists ;
         }

         this->ppfWriteLine ( gs ) ;   // write the line to target
      }
   }

   //* Scan until the top-level list has been closed *
   while ( status == OK )
   {
      if ( (status = this->ppfReadSrcLine ( gs, wptr, wi )) == OK )
      {
         this->ppfWriteLine ( gs ) ;   // write the line to target
         if ( (wcsncasecmp ( wptr, olLongEnd, olLongEnd_len)) == ZERO )
         {
            break ;
         }
      }
   }

   return status ;

}  //* End ppfProcGNU() *

//*************************
//*     ppfEndBlock       *
//*************************
//******************************************************************************
//* Test the data for a matching substring.                                    *
//*                                                                            *
//*                                                                            *
//* Input  : gsLine :                                                          *
//*          etag   : substring for which to search                            *
//*          elen   : characters in substring                                  *
//*                                                                            *
//* Returns: 'true' if a match found, else 'false'                             *
//******************************************************************************
bool Idpp::ppfEndBlock ( const gString& gsLine, const wchar_t* etag, short elen )
{
   short wi ;
   const wchar_t* wptr = gsLine.gstr( wi ) ;
   wi -= (elen + 1) ;
   bool match = false ;

   if ( (wcsncasecmp ( &wptr[wi], etag, elen)) == ZERO )
      match = true ;

   return match ;

}  //* End ppfEndBlock()) *

//*************************
//*  ppfInsertCustomData  *
//*************************
//******************************************************************************
//* Insert user-supplied data at the current position of the target document.  *
//* (existence of the data file has been previously verified)                  *
//*                                                                            *
//* Input  : fileSpec: path/filename specification of source data              *
//*                                                                            *
//* Returns: OK if successful, ERR processing error                            *
//******************************************************************************

bool Idpp::ppfInsertCustomData ( const gString& fileSpec )
{
   short status = ERR ;
   ifstream dataIn( fileSpec.ustr(), ifstream::in ) ;
   if ( dataIn.is_open() )
   {
      status = OK ;
      gString gsin ;
      while ( (this->ppfReadLine ( dataIn, gsin )) == OK )
         this->ppfWriteLine ( gsin ) ;

      dataIn.close() ;           // close the source file
   }
   return status ;

}  //* End ppfInsertCustomData() *

//*************************
//*    ppfReadSrcLine     *
//*************************
//******************************************************************************
//* Read one line of text from the specified source file,                      *
//* Idpp::slCount is incremented.                                              *
//*                                                                            *
//* Input  : gs     : (by reference) receives text data                        *
//*          wptr   : (by reference) receives a pointer to wide data in 'gs'   *
//*          windex : (by reference) returned with index of first (wide data)  *
//*                   non-whitespace character on line                         *
//*                                                                            *
//* Returns: OK if successful, ERR processing error                            *
//******************************************************************************

short Idpp::ppfReadSrcLine ( gString& gs, const wchar_t*& wptr, short& windex )
{
   char  tmp[gsMAXBYTES] ; // UTF-8 line input from file
   short status = ERR ;    // return value
   windex = ZERO ;         // initialize caller's index

   this->ifs.getline ( tmp, gsMAXBYTES, NEWLINE ) ; // read a source line
   if ( this->ifs.good() || this->ifs.gcount() > ZERO ) // if a good read
   {
      ++this->slCount ;                         // increment line counter
      gs = tmp ;                                // convert to wide data
      wptr = gs.gstr() ;                        // pointer to wide data
      for ( ; wptr[windex] == SPACE ; windex++ ) ; // ignore leading whitespace
      status = OK ;                             // return with good news

      //* For debugging only *
      if ( this->scan != false 
           && (this->slCount >= this->scan_beg && 
               this->slCount <= this->scan_end) )
      {
         gString gss( "SCAN (s:%4hu t:%4hu) '%S'", 
                      &this->slCount, &this->tlCount, gs.gstr() ) ;
         this->textOut ( gss ) ;
      }
   }
   return status ;

}  //* End ppfReadSrcLine() *

//*************************
//*      ppfReadLine      *
//*************************
//******************************************************************************
//* Read one line of text from the specified input stream.                     *
//* This is a generic ReadLine. See ppfReadSrcLine() for specialized HTML      *
//* source file read.                                                          *
//*                                                                            *
//* Input  : ifs    : open input stream (by reference)                         *
//*          gs     : (by reference) receives text data                        *
//*                                                                            *
//* Returns: OK if successful, ERR processing error                            *
//******************************************************************************

short Idpp::ppfReadLine ( ifstream& ifs, gString& gs )
{
   char  tmp[gsMAXBYTES] ; // UTF-8 line input from file
   short status = ERR ;    // return value

   ifs.getline ( tmp, gsMAXBYTES, NEWLINE ) ;   // read a source line
   if ( ifs.good() || ifs.gcount() > ZERO )     // if a good read
   {
      gs = tmp ;                                // convert to wide data
      status = OK ;                             // return with good news
   }
   return status ;

}  //* End ppfReadLine() *

//*************************
//*     ppfWriteLine      *
//*************************
//******************************************************************************
//* Write a line of data to the target file. the line will be terminated with  *
//* a newline character ('\n'), and the output buffer is flushed.              *
//* Idpp::tlCount is incremented.                                              *
//*                                                                            *
//* Input  : gsOut : (by reference) text to be written.                        *
//*                                                                            *
//* Returns: OK if successful, or ERR if write error.                          *
//*            Note: currently, we don't check for output stream errors.       *
//******************************************************************************

short Idpp::ppfWriteLine ( const gString& gsOut )
{
   short status = OK ;

   if ( this->no_mods == false )
   {
      this->ofs << gsOut.ustr() << endl ;
   }
   ++this->tlCount ;    // increment the output-line counter
   return status ;

}  //* End ppfWriteLine() *

//*************************
//*      ppfScan4Src      *
//*************************
//******************************************************************************
//* Scan the target directory for HTML source documents.                       *
//* Files are selected by filename extension.                                  *
//* Valid extensions:  .html  .htm                                             *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if at least one valid file found                           *
//**         'false' if: a) no HTML files found                                *
//*                      b) directory read error                               *
//******************************************************************************

bool Idpp::ppfScan4Src ( void )
{
   const wchar_t* extA = L".html" ;    const short extA_len = 6 ;
   const wchar_t* extB = L".htm" ;     const short extB_len = 5 ;
   const wchar_t* extC = L".shtml" ;   const short extC_len = 7 ;
   const wchar_t* extD = L".shtm" ;    const short extD_len = 6 ;
   const wchar_t* extE = L".xhtml" ;   const short extE_len = 7 ;

   DIR*      dirPtr ;            // pointer to open directory file
   deStats   destat, *sptr ;     // entry read from directory file
   gString   fnPath,             // source file's path/filename
             fnName,             // source file's filename
             fnExt ;             // source file's filename extension
   FileStats rawStats ;          // source file's stat info
   bool      status = false ;    // return value

   //* Discard any filenames specified directly on *
   //* command line. This avoids duplicates.       *
   this->sfCount = ZERO ;

   if ( ((this->ppfTargetExists ( this->cwDir, true )) != false) &&
        ((dirPtr = opendir ( this->cwDir.ustr() )) != NULL) )
   {
      while ( (readdir64_r ( dirPtr, &destat, &sptr )) == ZERO )
      {
         // NOTE: This compensates for a bug in the return value of readdir64_r.
         //       If sptr == NULL, it means end-of-list.
         if ( sptr == NULL )
            break ;

         //* '.' and '..' are not included, but hidden files are *
         if ( (destat.d_name[ZERO] == PERIOD)
              &&
              ((destat.d_name[1] == NULLCHAR) || 
               (destat.d_name[1] == PERIOD && destat.d_name[2] == NULLCHAR)) )
         {
            continue ;
         }

         //* 'lstat' the file to determine its type 'regular' files only *
         fnName = destat.d_name ;
         this->ppfCatPathFilename ( fnPath, this->cwDir, fnName.ustr() ) ;
         if ( (lstat64 ( fnPath.ustr(), &rawStats )) == OK )
         {
            if ( (S_ISREG(rawStats.st_mode)) != false )
            {
               this->ppfExtractFileExtension ( fnExt, fnPath.ustr() ) ;

               if (    (fnExt.compare( extA, extA_len ) == ZERO)
                    || (fnExt.compare( extB, extB_len ) == ZERO)
                    || (fnExt.compare( extC, extC_len ) == ZERO)
                    || (fnExt.compare( extD, extD_len ) == ZERO)
                    || (fnExt.compare( extE, extE_len ) == ZERO)
                  )
               {
                  fnName.copy( this->srcFiles[this->sfCount++], gsMAXCHARS ) ;
                  status = true ;   // at least one valid file found
               }
            }
            else { /* Can't access the file, so ignore it */ }
         }
      }  // while()
      closedir ( dirPtr ) ;               // close the directory
   }     // opendir()
   return status ;

}  //* End ppfScan4Src()

//*************************
//*       ppfGetCWD       *
//*************************
//******************************************************************************
//* Returns the path of user's current working directory.                      *
//*                                                                            *
//* Input  : dPath:(by reference, initial value ignored)                       *
//*                receives path string                                        *
//*                                                                            *
//* Returns: OK if successful, ERR if path too long to fit the buffer          *
//******************************************************************************

short Idpp::ppfGetCWD ( gString& dPath )
{
char     buff[MAX_PATH] ;
short    status = OK ;

   if ( (getcwd ( buff, MAX_PATH )) != NULL )
      dPath = buff ;
   else
      status = ERR ;    // (this is very unlikely)

   return status ;

}  //* End ppfGetCWD() *

//*************************
//*    ppfTargetExists    *
//*************************
//******************************************************************************
//* 'stat' the specified path/filename to see if it exists.                    *
//* Target must be a 'regular' file or optionally, a 'directory' file.         *
//*                                                                            *
//* We also do a simple test for read access: if owner, group, or other has    *
//* read access, we assume that user has read access although this may not be  *
//* true in all cases.                                                         *
//*                                                                            *
//* Input  : fPath: target path/filename                                       *
//*          isDir: (optional, false by default)                               *
//*                 if 'true', test whether file type is 'regular'             *
//*                 if 'false', test whether file type is 'directory'          *
//*                                                                            *
//* Returns: 'true' if file exists AND is a Regular file, or optionally, a     *
//*                 'directory' file                                           *
//*          'false' otherwise                                                 *
//******************************************************************************

bool Idpp::ppfTargetExists ( const gString& fPath, bool isDir )
{
   bool exists = false ;
   FileStats rawStats ;

   //* If the file exists *
   if ( (lstat64 ( fPath.ustr(), &rawStats )) == OK )
   {  //* Simple, read-access test (see note above) *
      if ( (rawStats.st_mode & S_IRUSR) || (rawStats.st_mode & S_IRGRP) || 
           (rawStats.st_mode & S_IROTH) )
      {
         //* If testing for the existence of a directory *
         if ( isDir != false )
         {
            if ( (S_ISDIR(rawStats.st_mode)) != false )
               exists = true ;
         }
         //* We operate only on 'regular' files *
         else
         {
            if ( (S_ISREG(rawStats.st_mode)) != false )
               exists = true ;
         }
      }
   }
   return exists ;

}  //* End ppfTargetExists() *

//*************************
//*    ppfTargetIsHTML    *
//*************************
//******************************************************************************
//* Test the first line of specified file.                                     *
//* A valid HTML (v4 or v5) document must begin with "<!DOCTYPE HTML".         *
//* A valid HTML (v3x) MAY begin with "<HTML".                                 *
//* If neither, then we assume that the document is not valid HTML.            *
//*                                                                            *
//* Input  : fPath: target path/filename                                       *
//*                                                                            *
//* Returns: 'true' if file begins with <!DOCTYPE HTML" OR "<HTML"             *
//*                 (comparison IS NOT case sensitive)                         *
//*          'false' otherwise                                                 *
//******************************************************************************

bool Idpp::ppfTargetIsHTML ( const gString& fPath )
{
   bool status = false ;
   ifstream ifs ( fPath.ustr(), ifstream::in ) ;
   if ( ifs.is_open() )             // if input file open
   {
      //* Read the first source line *
      gString gs ;
      if ( (status = this->ppfReadLine ( ifs, gs )) == OK )
      {
         if ( ((wcsncasecmp ( gs.gstr(), topDOCT, topDOCT_len )) == ZERO)
              ||
              ((wcsncasecmp ( gs.gstr(), topHTML, topHTML_len )) == ZERO) )
            status = true ;
      }
      ifs.close() ;                 // close the file
   }
   return status ;

}  //* End ppfTargetIsHTML() *

//*************************
//*    ppfTargetIsCSS     *
//*************************
//******************************************************************************
//* Test whether the specified CSS definition file is valid.                   *
//* a) Verify that the copyright notice is intact.                             *
//* b) Extract the version number string.                                      *
//* If version string not found, then we assume that the document is           *
//* incorrectly formatted      *
//*                                                                            *
//* Input  : fPath: target path/filename                                       *
//*          gsVer: receives version number string                             *
//*                                                                            *
//* Returns: 'true' if file format verified                                    *
//*          'false' otherwise                                                 *
//******************************************************************************

bool Idpp::ppfTargetIsCSS ( const gString& fPath, gString& gsVer )
{
   bool status = false ;
   ifstream ifs ( fPath.ustr(), ifstream::in ) ;
   if ( ifs.is_open() )             // if input file open
   {
      gString gs ;
      short i ;
      bool  cright_found = false, done = false ;

      while ( !done )
      {
         //* Read a source line *
         if ( (status = this->ppfReadLine ( ifs, gs )) == OK )
         {
            if ( !cright_found )
            {
               for ( i = ZERO ; gs.gstr()[i] != *cssCOPY && gs.gstr()[i] != NULLCHAR ; i++ ) ;
               if ( (gs.gstr()[i] == *cssCOPY) &&
                       ((gs.compare( cssCOPY, cssCOPY_len, i )) == ZERO) )
               {
                  gs.shiftChars( (ZERO - i) ) ;
                  for ( i = ZERO ; gs.gstr()[i] != *cssSAMU && gs.gstr()[i] != NULLCHAR ; i++ ) ;
                  if ( (gs.gstr()[i] == *cssSAMU) && 
                       ((gs.compare( cssSAMU, cssSAMU_len, i )) == ZERO) )
                  {
                     cright_found = true ;
                  }
               }
            }
            else
            {
               for ( i = ZERO ; gs.gstr()[i] != *cssVERS && gs.gstr()[i] != NULLCHAR ; i++ ) ;
               if ( (gs.gstr()[i] == *cssVERS) &&
                       ((gs.compare( cssVERS, cssVERS_len, i )) == ZERO) )
               {
                  gs.shiftChars( (ZERO - i) ) ;
                  gs.limitCharacters( CSS_VER_LEN ) ;
                  gsVer = gs ;
                  done = status = true ;
               }
            }
         }
         else
            done = true ;
      }
      ifs.close() ;                 // close the file
   }
   return status ;

}  //* End ppfTargetIsCSS() *

//*************************
//*  ppfCatPathFilename   *
//*************************
//******************************************************************************
//* Concatenate a path string with a filename string to create a path/filename *
//* specification.                                                             *
//*                                                                            *
//* Input  : pgs  : gString object (by reference) to hold the path/filename    *
//*                 On return, pgs contains the path/filename string in both   *
//*                 UTF-8 and wchar_t formats.                                 *
//*          wPath: gString object (by reference) path string                  *
//*          uFile: pointer to UTF-8 filename string                           *
//*               OR                                                           *
//*          wFile: pointer to wchar_t filename string                         *
//*                                                                            *
//* Returns: OK if success                                                     *
//*          ERR if string truncated                                           *
//******************************************************************************

short Idpp::ppfCatPathFilename ( gString& pgs, const gString& wPath, const char* uFile )
{
short    success = OK ;

   pgs.compose( L"%S/%s", wPath.gstr(), uFile ) ;
   if ( pgs.gschars() >= (gsMAXCHARS-1) )
      success = ERR ;   // (this is very unlikely)
   return success ;

}  //* End ppfCatPathFilename() *

short Idpp::ppfCatPathFilename ( gString& pgs, const gString& wPath, const wchar_t* wFile )
{
short    success = OK ;

   pgs.compose( L"%S/%S", wPath.gstr(), wFile ) ;
   if ( pgs.gschars() >= (gsMAXCHARS-1) )
      success = ERR ;   // (this is very unlikely)
   return success ;

}  //* End ppfCatPathFilename() *

//*************************
//*     ppfCdTarget      *
//*************************
//******************************************************************************
//* Change the current working directory to specified target path.             *
//*                                                                            *
//* Input  : dirPath: path of new CWD (relative or absolute)                   *
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
//******************************************************************************

short Idpp::ppfCdTarget ( const char* dirPath )
{
   short status ;       // return value

   //* If a relative path, convert to absolute path *
   char tmpPath[gsMAXBYTES] ;
   realpath ( dirPath, tmpPath ) ;
   gString targPath( tmpPath ) ;

   //* Change our current working directory and update CWD data member *
   if ( (status = chdir ( targPath.ustr() )) == OK )
      this->ppfGetCWD ( this->cwDir ) ;
   return status ;

}  //* End ppfCdTarget() *

//*************************
//*    ppfDeleteFile      *
//*************************
//******************************************************************************
//* Delete the specified file.                                                 *
//*                                                                            *
//* Input  : trgPath : full path/filename specification of target              *
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
//******************************************************************************

short Idpp::ppfDeleteFile ( const gString& trgPath )
{

   return ( unlink ( trgPath.ustr() ) ) ;

}  //* End ppfDeleteFile() *

//*************************
//*    ppfRenameFile      *
//*************************
//******************************************************************************
//* Rename the specified file.                                                 *
//*                                                                            *
//* Input  : srcPath : full path/filename specification of source              *
//*          trgPath : full path/filename specification of target              *
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
//******************************************************************************
//* For the rename() primitive: If srcPath refers to a symbolic link the       *
//* link is renamed; if trgPath refers to a symbolic link the link will be     *
//* overwritten.                                                               *
//******************************************************************************

short Idpp::ppfRenameFile ( const gString& srcPath, const gString& trgPath )
{

   return ( rename ( srcPath.ustr(), trgPath.ustr() ) ) ;

}  //* End ppfRenameFile() *

//*************************
//*     ppfRealpath       *
//*************************
//******************************************************************************
//* Decode the relative or aliased path for the specified target.              *
//*                                                                            *
//*                                                                            *
//* Input  : realPath : (by reference) receives decoded path specification     *
//*          rawPath  : (by reference) caller's path specification             *
//*                                                                            *
//* Returns: OK  if successful                                                 *
//*          ERR if invalid path or if system call fails ('realPath' unchanged)*
//******************************************************************************
//* Programmer's Note: The 'realpath' library function cannot handle a         *
//* symboloc substitution such as: ${HOME}/Docs/filename.txt                   *
//* It returns: /home/Sam/Docs/(CWD)/${HOME}                                   *
//* This is a little bit nuts, but we don't have time to investigate it.       *
//*                                                                            *
//******************************************************************************

short Idpp::ppfRealpath ( gString& realPath, const gString& rawPath )
{
   short status = ERR ;

   const char* rpath ;
   if ( (rpath = realpath ( rawPath.ustr(), NULL )) != NULL )
   {
      realPath = rpath ;
      free ( (void*)rpath ) ;    // release the temp storage
      status = OK ;
   }
   return status ;

}  //* End ppfRealpath() *

//*************************
//*  ppfExtractFilename   *
//*************************
//******************************************************************************
//* Extract the filename from the path/filename provided.                      *
//*                                                                            *
//* Input  : gsName: (by reference, initial contents ignored)                  *
//*                  receives the extracted filename                           *
//*          fPath : path/filename string                                      *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Idpp::ppfExtractFilename ( gString& gsName, const gString& fPath )
{
   const wchar_t* fptr = fPath.gstr() ;
   short findex = fPath.gschars() - 1 ;
   while ( findex > ZERO && fptr[findex] != SLASH )
      --findex ;
   if ( fptr[findex] == SLASH )
      ++findex ;
   gsName = &fptr[findex] ;

}  //* End ppfExtractFilename() *

//***************************
//* ppfExtractFileExtension *
//***************************
//******************************************************************************
//* Extract the filename extension (including the '.') from the path/filename  *
//* provided.                                                                  *
//*                                                                            *
//* Input  : gsName: (by reference, initial contents ignored)                  *
//*                  receives the extracted filename                           *
//*          fName : filename or path/filename string                          *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Idpp::ppfExtractFileExtension ( gString& gsExt, const char* fName )
{
   wchar_t tmpName[gsMAXCHARS] ;    // convert to wide character string
   gsExt = fName ;
   gsExt.copy( tmpName, gsMAXCHARS ) ;

   short   tpIndex = gsExt.gschars() - 1 ; // index the NULLCHAR
   while ( tpIndex > ZERO && tmpName[tpIndex] != PERIOD ) // isolate the extension
      --tpIndex ;
   if ( tmpName[tpIndex] == PERIOD )
      gsExt = &tmpName[tpIndex] ;
   else
      gsExt = L"" ;

}  //* End ppfExtractFileExtension() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

