//********************************************************************************
//* File       : Dialog1.cpp                                                     *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2006-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in NcDialog.hpp            *
//* Date       : 21-Mar-2025                                                     *
//* Version    : (see AppVersion string below)                                   *
//*                                                                              *
//* Description:                                                                 *
//* Dialog1 is a test and demonstration application for development of the       *
//* NcDialog class and associated classes. It began life as part of a thought    *
//* process regarding what would be needed to build a C++ API around the         *
//* ncurses C library primitives, and has quickly become a second-level          *
//* testing platform for the most useful of the ncurses C library primitives,    *
//* as well as a design-and-test application for the NcWindow and NcDialog       *
//* classes.                                                                     *
//*                                                                              *
//*        See also the Dialog2, Dialog3, Dialog4, Dialogw and Dialogx           *
//*        test applications for additional examples of using the                *
//*        NCurses class, NcWindow class, NcDialog class and gString class       *
//*        methods for building your own NcDialog-based applications.            *
//*                                                                              *
//* Development Tools: See NcDialog.cpp.                                         *
//********************************************************************************
//* Version History (most recent first):                                         *
//*                                                                              *
//* v: 0.1.11 20-Mar-2025                                                        *
//*   - The gString 'gsForm&' constructor is obsolete. Replace gString           *
//*     constructors which take a 'gsForm&' argument with standard               *
//*     formatting-template constructors.                                        *
//*   - Certain gString method parameters required promotion from 'short' to     *
//*     'int32_t'.                                                               *
//*   - Add a debugging-only test to HelpAbout() which reports the state of      *
//*     ncurses definitions which control access to certain structural internals.*
//*                                                                              *
//* v: 0.1.10 23-Jun-2024                                                        *
//*   - Update Help dialog to reference G++13 and C++11.                         *
//*                                                                              *
//* v: 0.1.09 08-Aug-2021                                                        *
//*   - Modify Test#8 to include an instance of the dctSLIDER control.           *
//*     The Test#8 dialog contains at least one instance of every control type.  *
//*   - Enable mouse support in Test#8.                                          *
//*                                                                              *
//* v: 0.1.08 18-Apr-2020                                                        *
//*   - Because the WaylandCB class has now been integrated into the             *
//*     NcDialogAPI, activate the WaylandCB class for Test01() to exercise       *
//*     the interface between the system clipboard and the local Textbox         *
//*     clipboard.                                                               *
//*                                                                              *
//* v: 0.1.07 29-Jan-2018                                                        *
//*   - Allow equals sign ('=') to delimit '-a' option and its argument.         *
//*                                                                              *
//* v: 0.1.06 15-Jan-2017                                                        *
//*   - Update Test #10 to exercise the new overload of SetActiveMenuItems().    *
//*                                                                              *
//* v: 0.1.05 30-Apr-2014                                                        *
//*   - Add command-line argument to support special options in Test01()         *
//*     related to the NcDialog-class DisplayTextboxMessage() method and the     *
//*     newly-enhanced support for multi-line dctTEXTBOX controls and editing    *
//*     of RTL language (right-to-left) data.                                    *
//*   - Enhance GetCommandLineArgs() to scan multiple command-line arguments.    *
//*   - Update Test08 callback for enhanced CaptureDialog() functionality.       *
//*   - Enhance Dropdown and Menuwin constructors' parsing test.                 *
//*                                                                              *
//* v: 0.1.04 23-Apr-2012                                                        *
//*   - Update the startup routine to match the changes made in the              *
//*     StartNCursesEngine() method. These changes enable remapping of color     *
//*     pairs and RGB registers on startup.                                      *
//*                                                                              *
//* v: 0.1.03 30-Aug-2011                                                        *
//*   - Updates to support enhanced definition of the InitNcDialog class.        *
//*   - Add consistent message for dialog windows that fail to open.             *
//*   - Move certain shared definitions, prototypes and data common to all the   *
//*     test and demonstration applications to a header, DialogAppShared.hpp.    *
//*                                                                              *
//* v: 0.1.02 05-Mar-2011                                                        *
//*   - Restructure control flow in main() to be more like the more polished     *
//*     Dialogw test application.                                                *
//*   - Converted from member assignments to class-initialization lists for      *
//*     all control definitions. This not only is a better demonstration for     *
//*     initialization of the InitCtrl class, but also makes it more difficult   *
//*     to 'forget' to assign a value to a necessary data member.                *
//*   - Overloaded the StatWin() method so both UTF-8 and wchar_t strings        *
//*     can be printed.                                                          *
//*                                                                              *
//* v: 0.1.01 04-Feb-2011                                                        *
//*   - Link against the ncursesw library ('wide" character I/O rather than      *
//*     the 8-bit ncurses library. Preliminary tests show that ncursesw is       *
//*     code-compatible with the ncurses library.                                *
//*   - Focus of development is on the keyboard input and screen output          *
//*     routines, to determine the changes needed in the NCurses, NcWindow       *
//*     and NcDialog  classes to support UTF-8 input/output. Fortunately, these  *
//*     I/O routines are all funneled through a very few methods which directly  *
//*     call ncurses functions.                                                  *
//*   - Add command-line argument for specifying an alternate 'locale'.          *
//*                                                                              *
//* v: 0.1.00 13-Mar-2010 Set version number for distribution.                   *
//*                                                                              *
//* v: 0.0.16 07-Feb-2010                                                        *
//*   - Change development platform and compiler version. Fedora 12 and          *
//*     GNU G++ (Gcc v: 4.4.2)                                                   *
//*     This neccessitated several syntax changes in class and method            *
//*     definitions to accomodate changes in the C++ definition and the GCC      *
//*     compiler's warning and error messages.                                   *
//*   - Add #include <cstring> for strncpy() etc.                                *
//*   - Statwin() now takes a const char* parameter.                             *
//*   - Enhance Test6(): additional hotkey testing                               *
//*   - Add Test8(), the move-window test, which also provides an opportunite    *
//*     to test for hotkeys detected while inside NcDialog control editing       *
//*     routines.                                                                *
//*   - Add Test9(), for development and testing of dctDROPDOWN control type.    *
//*   - Add Test10(), for development and testing of the dctMENUWIN control      *
//*     type, including the MenuBar.                                             *
//*                                                                              *
//* v: 0.0.15 12-Apr-2007                                                        *
//*   - Significant changes in many modules, reintegrate.                        *
//*   - Move functionality for exclusive-OR radiobutton group from application   *
//*     to NcDialog class.                                                       *
//*   - Add a dialog for more extensive testing of radio button groups.          *
//*                                                                              *
//* v: 0.0.14 16-Mar-2007 Integrate NcScroll.cpp and NcDialog class updates      *
//* v: 0.0.13 03-Feb-2007 Add calls to SetDialogTitle() method                   *
//* v: 0.0.12 05-Jan-2007 Enhance dctTEXTBOX control                             *
//* v: 0.0.11 18-Oct-2006 Add dctSCROLLBOX control                               *
//* v: 0.0.10 17-Jun-2006 Base code is from the FileMangler application.         *
//********************************************************************************
//* Programmer's Notes:                                                          *
//* - Because internationalization of I/O is always a possibility in any         *
//*   application, we use as few ASCII-oriented C/C++ library functions as       *
//*   possible. e.g. sprintf(), strlen(), toupper(), etc. Where such             *
//*   functionality is needed for character manipulation, we strongly advise     *
//*   using the equivalent 'wide chararacter' functions OR the gString class.    *
//*                                                                              *
//********************************************************************************

//****************
//* Header Files *
//****************
//* All necessary information for:                  *
//* NCurses, NcWindow, NcDialog and gString classes.*
#include "GlobalDef.hpp"

//* Definitions, prototypes and data shared among the test applications *
#include "DialogAppShared.hpp"

//***************
//* Definitions *
//***************

//***************
//* Prototypes  *
//***************
short    Test8_GetNewPosition ( winPos oPos, winPos& wPos ) ;
void     Test8_CircusCircus ( short dLines, short dCols ) ;

//**********
//*  Data  *
//**********
//* Application version string. Keep it current! *
const char* AppVersion = "0.1.11" ;
const char* crYears = "2006-2025" ;

//* Programmer's Note: For any dialogs that use a callback method, a set of 
//* constants should be defined at file scope for the dialog's control indices, 
//* so both the instantiating method and its callback method have access to the 
//* same definitions. These is done as an enum with initial value == ZERO.

//* Test01() dialog control constants. *
enum T01Controls {
                   T1okPush = 0, 
                   T1cancelPush,  T1applyPush, T1plainText, T1numberText, 
                   T1numericText, T1ucaseText, T1lcaseText, 
                   T1rtlText,     T1multiText, 
                   T1rbGroupA,    T1rbGroupB,  T1rbGroupC,  T1rbStd5, 
                   T1rbCust3,     T1rbCust5,   T1trackr1,   T1trackr2, 
                   T1insOver,     T1SysCb,
                   T1controlsDEFINED
                 } ;
static bool  t1dtmExercise = false,  // Set according to 'd' command-line switch
             t1dtmCentered = false,  // (see Test01 header)
             t1dtmColorful = false,
             t1dtmRgt2Left = false ;
static short t1MtextOption = ZERO ;

//* Test02() dialog control constants. *
enum T02Controls {
                   T2okPush = 0, 
                   T2cancelPush, T2applyPush, T2singleSB, 
                   T2multiSB,    T2tbONE,     T2tbTWO
                 } ;

//* Test06() dialog control constants. *
enum T06Controls {
                   T6donePush = 0, 
                   T6groupA_1,    T6groupA_2, T6groupA_3, T6groupA_4, 
                   T6groupB_1,    T6groupB_2, T6groupB_3, T6standAlone8, 
                   T6standAlone9, T6tbText,   T6utBOX1,   T6utBOX2, 
                   T6utBOX3,      T6utBOX4
                 } ;

//* Test07() dialog control constants. *
enum T07Controls {
                   T7okBUTTON = 0, 
                   T7dealBOX, T7worldBUTTON, T7laundryBUTTON, T7pizzaBUTTON, 
                   T7flavorBOX, T7toppingsTBOX, T7strawberryBUTTON, 
                   T7chocolateBUTTON, T7vanillaBUTTON, T7toppingsSBOX
                 } ;

//* Test08() dialog control constants. *
enum T08Controls {
                   T8donePush = 0, T8trackrBox, T8positionControl, 
                   T8planetBox, T8icecreamBox, T8tuscaloosaButton, 
                   T8sandiegoButton, T8cucamongaButton, T8valparisoButton, 
                   T8dumpPush, T8scottyButton, T8spinnerControl, T8dropdownBox, 
                   T8scrollextControl, T8sliderControl, 
                   T8menuwin01, T8menuwin02, T8menuwin03, T8menuwin04
                 } ;

//* Test09() dialog control constants. *
enum T09Controls {
                   T9donePush = 0, 
                   T9ddA, T9ddB, T9ddC, T9ddD, T9ddE, T9ddF, T9ddG, T9dumpPush
                 } ;

//* Test10() dialog control constants. *
enum T10Controls {
                   T10donePush = 0, 
                   T10mwA, T10mwB, T10mwC, T10mwI, T10cmA, T10cmB, T10cmC, 
                   T10cmD, T10cmE, 
                   T10mwG, T10cmG2, T10cmG3, T10cmG4, T10cmG5, T10cmG6, 
                   T10mwM, T10rb1, T10rb2, T10rb3, T10rb4, T10rb5, T10tbA
                 } ;


//*************************
//*         main          *
//*************************
//******************************************************************************
//* Program entry point.                                                       *
//*                                                                            *
//* Command-line Usage:  SEE BELOW FOR COMMAND-LINE ARGUMENT PROCESSING        *
//*                                                                            *
//*                                                                            *
//******************************************************************************

int main ( int argc, char* argv[], char* argenv[] )
{
short    testNum = - 1 ;         // number of test to execute (default TestMenu)
bool     winTooSmall = false ;   // set 'true' if terminal window is too small

   //* User may have specified one or more command-line arguments *
   commArgs clArgs( argc, argv, argenv ) ;
   testNum = GetCommandLineArgs ( clArgs ) ;

   #if 0    // PRETEST THE LOCALE SPECIFIED BY THE TERMINAL ENVIRONMENT
   //* Note that on start-up, the NCurses Engine sets the 'locale' (character  *
   //* encoding for internationalization) according to the terminal window's   *
   //* environment variables; however, if the default locale does not support  *
   //* UTF-8 encoding, you should adjust the terminal environment variables    *
   //* LANG=  ( || LC_ALL=  ||  LC_CTYPE= ).                                   *
   //* If your application is not ASCII only, please set the terminal          *
   //* environment variable to 'en_US.UTF-8' or another locale that supports   *
   //* UTF-8 character encoding, before running the application, or at least   *
   //* before starting the NCurses Engine.                                     *
   //* -At the command prompt    :  LANG=en_US.UTF-8                           *
   //* -or within the application:  setenv("LANG", "en_US.UTF-8", 1);          *
   //*                                                                         *
   //* Note that UTF-8 is the native language of Linux.                        *
   //*                                                                         *
   //* If user has not specified a locale on the command line, test the default*
   if ( clArgs.altLocale == false )
   {
      locale loco("") ;
      if( (nc.StartNCursesEngine ()) == OK )
      {
         short dfltLocaleOk = nc.VerifyLocale ( loco.name().c_str() ) ;

         if ( dfltLocaleOk != OK )
         {
            nc.ClearScreen () ;
            gString gs( "  BAD DEFAULT LOCALE: '%s'  ", loco.name().c_str() ) ;
            nc.WriteString ( 0, 0, gs.gstr(), nc.reR ) ;
            sleep ( 3 ) ;

            //* Default locale does not support UTF-8 character encoding *
            setenv ( "LANG", "en_US.UTF-8", 1 ) ;
         }
         nc.StopNCursesEngine () ;
      }
   }
   #endif   // PRETEST

   //*******************************
   //** Initialize NCurses Engine **
   //*******************************
   //* NCurses class has been previously instantiated at the time the          *
   //* application was loaded, and is available through the name: 'nc'.        *
   //* (see note in GlobalDef.hpp)                                             *
   if( (nc.StartNCursesEngine ()) == OK )
   {
      //* Verify that the current locale supports UTF-8 character encoding.    *
      short localeSetOk = nc.VerifyLocale () ;  // verify default locale setting

      //* If user has specified an alternate locale for interpretation of      *
      //* character output and input, it must be passed to the NCurses class   *
      //* before any I/O occurs.                                               *
      //* (NOTE: If clArgs.localeName does not specify a locale supported by ) *
      //* (      the terminal program, OR the locale specified is not a      ) *
      //* (      UTF-8-encoded locale, OR the default encoding specified in  ) *
      //* (      the environment was not UTF-8, then the results will be     ) *
      //* (      ambiguous at best.                                          ) *
      short altLocaleOk = OK ;
      if ( clArgs.altLocale != false )
         altLocaleOk = nc.SetLocale ( clArgs.localeName ) ;

      //* Even though no windows have been opened yet, you can think of the    *
      //* entire screen as a big window which can be written to and which      *
      //* is the parent/grandparent of all windows and NcDialog objects that   *
      //* will be opened. (cursor state: nccvINVISIBLE and key state: nckpRAW) *
      ClearTerminalWindow () ;               // display application title

      //* Initialize and open the application's status window *
      sWin = new NcWindow ( swLINES, swCOLS, swOFFy, swOFFx ) ;
      if ( (sWin->OpenWindow ()) != ERR )
      {
         bool  done = false ;    // loop control

         //* Report any error in initializing color output*
         if ( !(nc.ColorText_Initialized ()) )
         {
            if ( ! (nc.ColorText_Available ()) )
               StatWin ( "This terminal does not support multi-color output..." ) ;
            else
               StatWin ( "Initialization of multi-color output failed..." ) ;
            sleep ( 5 ) ;
         }
         //* Report any error in setting default or alternate locale.  *
         //* (Even if error, ASCII-only I/O SHOULD cause no problems.) *
         if ( localeSetOk != OK || altLocaleOk != OK )
         {
            if ( localeSetOk != OK )
               StatWin ( "Current locale setting does not support UTF-8 character encoding..." ) ;
            else
               StatWin ( "Locale setting specified on command line is not valid..." ) ;
            sleep ( 5 ) ;
         }

         //* Refresh main window *
         nc.RefreshScreen () ;
         nc.ScreenDimensions ( termRows, termCols ) ;   // get terminal-window size
         if ( termRows < minTERMROWS || termCols < minTERMCOLS )
         {
            //* We need room to play *
            winTooSmall = done = true ;
         }

         //* Find out what the user wants *
         while ( ! done )
         {
            //* If command-line agrument for test number provided, use it.     *
            //* Else, open a dialog so user can choose which test(s) to run.   *
            if ( testNum < ZERO )
               testNum = TestMenu ( termRows / 2, termCols / 2 ) ;

            //* Check whether user has re-sized the terminal window *
            nc.ScreenDimensions ( termRows, termCols ) ;
            if ( termRows < minTERMROWS || termCols < minTERMCOLS )
            {
               winTooSmall = done = true ;
               continue ;
            }

            //* Execute selected test *            
            switch ( testNum )
            {
               case indexT01:          // text boxes, pushbuttons and radio buttons
                  Test01 () ;
                  break ;
               case indexT02:          // scroll boxes
                  Test02 () ; 
                  break ;
               case indexT03:          // scrolling with NcWindow primitives
                  Test03 () ;
                  break ;
               case indexT04:          // test display of base window background color
                  Test04 () ;
                  break ;
               case indexT05:          // display options for radio buttons
                  Test05 () ;
                  break ;
               case indexT06:          // Radio-button mutual-exclusion test
                  Test06 () ;
                  break ;
               case indexT07:          // Hotkey test for Radio Buttons and Pusbuttons
                  Test07 () ;
                  break ;
               case indexT08:          // Move-dialog window test (hide and show dialog)
                  Test08 () ;    
                  break ;
               case indexT09:          // Drop-Down control testing
                  Test09 () ;
                  break ;
               case indexT10:          // Menu control testing
                  Test10 () ;
                  break ;
               case indexEXIT:         // exit the application
                  StatWin ( " Exiting Application . . . ", nc.blG ) ;
                  done = true ;
                  break ;
               case indexHELP:         // help-about dialog
               default:
                  HelpAbout ( termRows / 2 + 1, termCols / 2 ) ;
                  break ;
            }
            testNum = -1 ;    // force use of the menu
         }        // while()
      }
      else
      {
         winTooSmall = true ;
      }

      //****************************
      //* Shut down ncurses engine *
      //****************************
      delete sWin ;                          // release status line resources       
      nc.RefreshScreen () ;                  // refresh the main window
      nc.StopNCursesEngine () ;              // Deactivate the NCurses engine

      //* Most likely cause of failure in opening the background *
      //* window is that the terminal window is too small.       *
      //* Memory allocation error is also a possibility.         *
      if ( winTooSmall != false )      
         cout << "\n ******************************************"
                 "\n * ERROR! Unwilling to open main window.  *"
                 "\n * Terminal window must be >= 80x25 chars.*"
                 "\n ******************************************\n\n" ;
   }
   else
   {
      cout << "\n **********************************************"
              "\n * ERROR! Unable to initialize NCurses engine.*"
              "\n **********************************************\n\n" ;
   }

   return ZERO ;

}  //* End main() *

//*************************
//*       TestMenu        *
//*************************
//******************************************************************************
//* Open a dialog and display a list of test choices.                          *
//*                                                                            *
//*                                                                            *
//* Input  : ctrY: Y center position for dialog window                         *
//*          ctrX: X center position for dialog window                         *
//*                                                                            *
//* Returns: member of cIndices                                                *
//******************************************************************************

short    TestMenu ( short ctrY, short ctrX )
{
static const short dialogROWS = 19 ;         // display lines
static const short dialogCOLS = 57 ;         // display columns
static const char* testNames[] = 
{// 1234567890123456789012345678901234567890123456

   "Test #1: ^Text Box and Radio Button controls  ",
   "Test #2: ^Scroll Box controls                 ",
   "Test #3: Scrolling with NcWindow ^primitives  ",
   "Test #4: Nc^Window draw with background color ",
   "Test #5: Display ^Radio Button types          ",
   "Test #6: Radio Button ^group selection        ",
   "Test #7: Hot^key Selection test               ",
   "Test #8: Mo^ve Dialog Window                  ",
   "Test #9: ^Dropdown Control Test               ",
   "Test#10: Me^nu Window Test                    ",
} ;
static NcDialog*  dp ;                       // dialog window pointer
static bool    dialogOpen = false ;          // set to true when dialog opened
short    ulY     = ctrY - dialogROWS / 2,    // upper left corner in Y
         ulX     = ctrX - dialogCOLS / 2 ;   // upper left corner in X
attr_t   dColor  = nc.blR,                   // dialog's base color
         rnColor = nc.bw,                    // radio button non-focus color
         rfColor = nc.cyG ;                  // radio button focus color
short    testNum ;                           // return value

   StatWin ( "Opening Test-Selection Menu" ) ; // keep user up-to-date

   //* Instantiate the dialog window and install its control objects *
   if ( dialogOpen == false )
   {
      //****************************************
      //* Initial parameters for dialog window *
      //****************************************
      InitCtrl ic[indexCOUNT] =                    // control initialization structures
      {
         {  //* TEST01 radiobutton  - - - - - - - - - - - - - - - - -    indexT01 *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            2,                            // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            testNames[0],                 // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[1]                        // nextCtrl:  link in next structure
         },
         {  //* TEST02 radiobutton  - - - - - - - - - - - - - - - - -    indexT02 *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            3,                            // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            testNames[1],                 // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[2]                        // nextCtrl:  link in next structure
         },
         {  //* TEST03 radiobutton  - - - - - - - - - - - - - - - - -    indexT03 *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            4,                            // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            testNames[2],                 // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[3]                        // nextCtrl:  link in next structure
         },
         {  //* TEST04 radiobutton  - - - - - - - - - - - - - - - - -    indexT04 *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            5,                            // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            testNames[3],                 // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[4]                        // nextCtrl:  link in next structure
         },
         {  //* TEST05 radiobutton  - - - - - - - - - - - - - - - - -    indexT05 *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            6,                            // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            testNames[4],                 // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[5]                        // nextCtrl:  link in next structure
         },
         {  //* TEST06 radiobutton  - - - - - - - - - - - - - - - - -    indexT06 *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            7,                            // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            testNames[5],                 // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[6]                        // nextCtrl:  link in next structure
         },
         {  //* TEST07 radiobutton  - - - - - - - - - - - - - - - - -    indexT07 *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            8,                            // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            testNames[6],                 // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[7]                        // nextCtrl:  link in next structure
         },
         {  //* TEST08 radiobutton  - - - - - - - - - - - - - - - - -    indexT08 *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            9,                            // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            testNames[7],                 // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[8]                        // nextCtrl:  link in next structure
         },
         {  //* TEST09 radiobutton  - - - - - - - - - - - - - - - - -    indexT09 *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            10,                           // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            testNames[8],                 // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[9]                        // nextCtrl:  link in next structure
         },
         {  //* TEST10 radiobutton  - - - - - - - - - - - - - - - - -    indexT10 *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            11,                           // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            testNames[9],                 // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[10]                       // nextCtrl:  link in next structure
         },
         {  //* HELP radiobutton    - - - - - - - - - - - - - - - - -   indexHELP *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            true,                         // rbSelect:  default selection
            12,                           // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            "^Help About dialog",         // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[11]                       // nextCtrl:  link in next structure
         },
         {  //* EXIT radiobutton    - - - - - - - - - - - - - - - - -   indexEXIT *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            13,                           // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            "E^xit",                      // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            NULL                          // nextCtrl:  link in next structure
         },
      } ;
   
   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal
                       "  Select Test To Be Executed  ", // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   dp = new NcDialog ( dInit ) ;

      //* Open the dialog window *
      if ( (dp->OpenWindow()) == OK )
      {
         dialogOpen = true ;              // remember that dialog is open
         
         dp->DrawBox ( 1,  2, dialogROWS-5, 53, dColor ) ;  // enclose radio-button group
         //* Display instructions *
         gString gsOut( "Use Tab and Shift+Tab or arrow keys to scroll through\n"
                        "options, then Space or Enter key to select;\n"
                        "or select a test via 'hotkey' (underlined character)" ) ;
         dp->WriteParagraph ( dialogROWS-4, 2, gsOut, dColor ) ;
         dp->RefreshWin () ;
   
         //* Group all radio buttons in the dialog *
         short XorGroup[] = { indexT01, indexT02, indexT03, indexT04, indexT05, 
                              indexT06, indexT07, indexT08, indexT09, indexT10, 
                              indexHELP, indexEXIT, -1 } ;
         dp->GroupRadiobuttons ( XorGroup ) ;
      }
   }           // If(dialogOpen==false)

   //* Un-hide the existing dialog window *
   else
   {
      dp->RefreshWin () ;
   }

   if ( dialogOpen != false )
   {
      //* Give the focus to the currently 'selected' button *
      short sIndex = dp->GetRbGroupSelection ( indexEXIT ) ;
      short cIndex = dp->GetCurrControl () ;
      while ( cIndex != sIndex )
         cIndex = dp->PrevControl () ;

      uiInfo Info ;                       // user interface info
      bool  testSelected = false ;        // loop control
      while ( !testSelected )
      {
         //* Move among the members of this Radio Button group *
         //* allowing user to select a test to be executed.    *
         // NOTE: This interface loop is fairly simple because *
         //       all controls in this dialog are Radio Buttons*
         //       and they all belong to the same button group.*
         dp->EditRadiobutton ( Info ) ;
         
         //* If a new selection has been made, OR  *
         //* if previous selection was re-selected.*
         if ( Info.dataMod || Info.keyIn == nckENTER )
            testSelected = true ;
      }
      testNum = Info.selMember ; // return value
   }

   //* If we are about to exit application, close the dialog. *
   if ( testNum == indexEXIT )
   {
      if ( dp != NULL )                      // close the window
         delete ( dp ) ;
      dp = NULL ;
   }
   //* Else, hide the window for future use *
   else
   {
      dp->HideWin () ;
   }

   return testNum ;

}  //* End TestMenu() *

//*************************
//*      Test01           *
//*************************
//******************************************************************************
//* Open a dialog with various elements: Pushbuttons, Text Boxes, and          *
//* Radio Buttons (both grouped and independent).                              *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note: For purposes of demonstration in this test application, *
//* the list of control-initialization objects is coded inline within each     *
//* test method. This allows a clear view of how to initialize the InitCtrl    *
//* class for each control type. However, in a more professional application,  *
//* it would be better to populate the InitCtrl ic[] array in a separate       *
//* header file to make the dialog's execution code more readable.             *
//*         Initialization of color attribute fields must, of course,          *
//*         be done AFTER the NCurses Engine has been started.                 *
//*                                                                            *
//* Example:                                                                   *
//* bool GetUserInput ( parameters )                                           *
//* {                                                                          *
//*    dialog definitions . . .                                                *
//*                                                                            *
//*    #include "ControlDefinitions_GetUserInput.hpp"                          *
//*                                                                            *
//*    instantiation of dialog                                                 *
//*                                                                            *
//*    initialization of color attributes                                      *
//*                                                                            *
//*    user input loop . . .                                                   *
//*                                                                            *
//*    destruction of dialog                                                   *
//*                                                                            *
//*    return validData ;                                                      *
//* }                                                                          *
//*                                                                            *
//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *
//* Note about exercising the multi-line dctTEXTBOX control (T1multiText)      *
//* and the 'NcDialog-class DisplayTextboxMessage method:                      *
//* For these special test options, this test method references four flags     *
//* and an indexing variable defined at the top of this module.                *
//*     t1dtmExercise, t1dtmCentered, t1dtmColorful and t1MtextOption          *
//* Two controls, T1plainText and T1multiText are affected by these options.   *
//* By default, the flags are all 'false' and the index == ZERO. However, if   *
//* the '-d' switch is specified on the command line, then the flags have been *
//* set according to the following command-line arguments:                     *
//*    -d        t1dtmExercise set 'true'                                      *
//*    -dc       t1dtmExercise and t1dtmCentered set 'true'                    *
//*    -dm       t1dtmExercise and t1dtmColorful set 'true'                    *
//*    -dcm      t1dtmExercise , t1dtmCentered and t1dtmColorful set 'true'    *
//*    -dr       t1dtmExercise and t1dtmRgt2Left set 'true'                    *
//*    -drc      t1dtmExercise , t1dtmRgt2Left and t1dtmCentered set 'true'    *
//*    -drm      t1dtmExercise , t1dtmRgt2Left and t1dtmColorful set 'true'    *
//*    -drcm     (this 'works' but make no sense for this test because there)  *
//*              (are several zero-width characters in the Arabic test data )  *
//*              (which makes character-to-attribute matching difficult.    )  *
//*                                                                            *
//* The text initially contained in the T1multiText control (before editing)   *
//* is specified by the value of t1MtextOption:                                *
//* 0 == Mixed Chinese and Pinyin text (default)                               *
//*      Note that Chinese text uses multi-column characters, and Pinyin       *
//*      uses single-column characters. This provids a more robust test.       *
//* 1 == Chinese text                                                          *
//* 2 == Pinyin text                                                           *
//* 3 == ASCII text (simplest test)                                            *
//******************************************************************************

void Test01 ( void )
{
char Head1[] = "Test dialog for text boxes, radio buttons and pushbuttons" ;
const short dialogROWS = 23 ;         // display lines
const short dialogCOLS = 75 ;         // display columns
const short buttonWidth = 8 ;
const char OkButtonText[buttonWidth+2]     = "   ^OK   " ;
const char CancelButtonText[buttonWidth+1] = " CANCEL " ;
const char ApplyButtonText[buttonWidth+1]  = " APPLY  " ;

//* Test data for RTL textbox. Note that display of this string here depends   *
//* on the sophistication of your text editor. jEdit displays it properly.     *
const char* rtlTextData = "الفصحى العربية الحديثة" ; // ("Modern Standard Arabic")

//* "This sentence is written in Modern Standard Arabic."
//const char* rtlArabic = "تتم كتابة هذه الجملة في اللغة العربية الفصحى الحديثة." ;

//* "Ready for lunch?" (Yiddish locale: yi_US.utf8) *
//const char* rtlYiddishA = "גרייט פֿאַר לאָנטש?" ;        // displayed backward in source
//const char* rtlYiddishB = "?שטנאָל ראַפֿ טיירג" ;        // manually reversed
//* "Ready for lunch?" (Hebrew locale: he_IL.utf8) *
//const char* rtlHebrewA = "מוכנים לארוחת צהריים?" ;    // displayed backward in source
//const char* rtlHebrewB = "?םיירהצ תחוראל םינכומ" ;    // manually reversed

//* Test data for multi-column textbox                                         *
//* Chinese: 光棍儿 贱人 寻求 兼容 女人.                                          *
//* Pinyin : Guānggùn er jiàn rén xúnqiú jiānróng nǚrén.                       *
//* English: Slutty unmarried man seeking compatible woman.                    *
//*    The translation may not be perfect, but the sentiment fits  :-)         *
const char* multiTextData[] = 
{
   "光棍儿 Guānggùn er 贱人 jiàn rén 寻求 xúnqiú 兼容 jiānróng 女人 nǚrén。",
   "光棍儿 贱人 寻求 兼容 女人。",
   "Guānggùn er jiàn rén xúnqiú jiānróng nǚrén.",
   "Slutty unmarried man seeking compatible woman.",
} ;

//* Text data and color attributes for exercise of the DisplayTextboxMessage() *
//* method. See note in method header.                                         *
const char* dtmSingleLineText = "*Going On Strike*" ;
const char* dtmMultiLineText  = "Hello There!\n"
                                "What's happenin' with\n"
                                "you these days,\n"
                                "mate?" ;
const char* dtmSingleLineRtlText =  // "what is your name?"
"ما اسمك؟" ;
const char* dtmMultiLineRtlText  =  // "hello" "how are you?" "I'm fine, and you?"
"السلام عليكم"
"\n"
"كيف الحال؟" 
"\n"
"بخير، الحمد لله. "
"\n"
"وكيف أنت؟" ;

attr_t dtmMonoColor[] = { dtbmFcolor } ; // monochrome output
attr_t dtmMultiColor[gsALLOCDFLT] =     // multi-colored output
     { nc.bwR, nc.reR, nc.grR, nc.brR, nc.blR, nc.maR, nc.cyR, 
       nc.bwR, nc.reR, nc.grR, nc.brR, nc.blR, nc.maR, // end 1st multi line
       nc.cyR, nc.bwR, nc.reR, nc.grR, nc.brR, nc.blR, nc.maR, 
       nc.cyR, nc.bwR, nc.reR, nc.grR, nc.brR, nc.blR, nc.maR, 
       nc.cyR, nc.bwR, nc.reR, nc.grR, nc.brR, nc.blR, nc.maR, 
       nc.cyR, // end 2nd multi line
       nc.bwR, nc.reR, nc.grR, nc.brR, nc.blR, nc.maR, nc.cyR, 
       nc.bwR, nc.reR, nc.grR, nc.brR, nc.blR, nc.maR, nc.cyR, 
       nc.bwR, nc.reR, // end 3rd multi line
       nc.grR, nc.brR, nc.blR, nc.maR, nc.cyR, nc.bwR, // end 4th multi line
       // If T1multiText is centered, these are used for expanded display text:
       // Beyond the technicolor area, characters change to white-on-red.
       // '?' character should be yellow-on-blue, and
       // lower right corner of control should be green
       nc.reR, nc.reR, nc.reR, nc.reR, nc.reR, nc.reR, nc.reR, nc.reR, 
       nc.reR, nc.reR, nc.reG, nc.reG, nc.reG, nc.reG, nc.reG, nc.reG, 
       nc.reG, nc.reG, nc.reG, nc.reG, nc.reG, nc.reG, nc.reG, nc.reG, 
       nc.reR, nc.reR, nc.reR, nc.reR, nc.brbl, 
       nc.reG, nc.reG, nc.reG, nc.reG, nc.reG, nc.reG, nc.reG, nc.reG, 
       nc.brgr,               // lower right of centered data
       nc.re, nc.re, nc.re    // (spares)
     } ;

short    ctrY   = termRows/2,                // terminal window center in Y
         ctrX   = termCols/2,                // terminal window center in X
         ulY    = ctrY - dialogROWS / 2,     // upper left corner in Y
         ulX    = ctrX - dialogCOLS / 2 ;    // upper left corner in X
attr_t   dColor = nc.blR ;                   // dialog window background color

   StatWin ( "Opening Test Dialog #1 - Text Box and Radio Button controls" ) ;

InitCtrl ic[T1controlsDEFINED] =   // array of dialog control initialization objects
{
   {  //* 'OK' pushbutton - - - - - - - - - - - - - - - - - - - - -   T1okPush *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 2),        // ulY:       upper left corner in Y
      14,                           // ulX:       upper left corner in X
      1,                            // lines:     control lines
      buttonWidth,                  // cols:      control columns
      OkButtonText,                 // dispText:  
      nc.re,                        // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T1cancelPush]             // nextCtrl:  link in next structure
   },
   {  //* 'CANCEL' pushbutton - - - - - - - - - - - - - - - - -   T1cancelPush *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[T1okPush].ulY,             // ulY:       upper left corner in Y
      33,                           // ulX:       upper left corner in X
      1,                            // lines:     control lines
      buttonWidth,                  // cols:      control columns
      CancelButtonText,             // dispText:  
      nc.re,                        // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T1applyPush]              // nextCtrl:  link in next structure
   },
   {  //* 'APPLY' pushbutton - - - - - - - - - - - - - - - - - -   T1applyPush *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[T1okPush].ulY,             // ulY:       upper left corner in Y
      52,                           // ulX:       upper left corner in X
      1,                            // lines:     control lines
      buttonWidth,                  // cols:      control columns
      ApplyButtonText,              // dispText:  
      nc.re,                        // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T1plainText]              // nextCtrl:  link in next structure
   },
   {  //* 'Plain Text' Textbox - - - - - - - - - - - - - - - - - - T1plainText *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      3,                            // ulY:       upper left corner in Y
      16,                           // ulX:       upper left corner in X
      1,                            // lines:     control lines
      short((dialogCOLS / 3) - 2),  // cols:      control columns
      "Ginsu knives: $19.95",       // dispText:  
      nc.brR,                       // nColor:    non-focus color
      nc.grR,                       // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "All Printing ",              // label:     
      ZERO,                         // labY:      
      -14,                          // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T1numberText]             // nextCtrl:  link in next structure
   },
   {  //* 'Number' Textbox - - - - - - - - - - - - - - - - - - -  T1numberText *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[T1plainText].ulY + 2), // ulY:       upper left corner in Y
      ic[T1plainText].ulX,          // ulX:       upper left corner in X
      1,                            // lines:     control lines
      ic[T1plainText].cols,         // cols:      control columns
      "2801",                       // dispText:
      nc.bwR,                       // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbNumber,                     // filter:    accepts only numbers
      "Number(rtJust)",             // label:     
      ZERO,                         // labY:      
      -14,                          // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T1numericText]            // nextCtrl:  link in next structure
   },
   {  //* 'Numeric' Textbox - - - - - - - - - - - - - - - - -    T1numericText *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[T1numberText].ulY + 2), // ulY:       upper left corner in Y
      ic[T1plainText].ulX,          // ulX:       upper left corner in X
      1,                            // lines:     control lines
      ic[T1plainText].cols,         // cols:      control columns
      NULL,                         // dispText:
      nc.re,                        // nColor:    non-focus color
      nc.reR,                       // fColor:    focus color
      tbNumeric,                    // filter:    accepts signed numbers and PERIOD
      "Numeric Chars",              // label:     
      ZERO,                         // labY:      
      -14,                          // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T1ucaseText]              // nextCtrl:  link in next structure
   },
   {  //* 'Uppercase' Textbox - - - - - - - - - - - - - - - - -    T1ucaseText *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      9,                            // ulY:       upper left corner in Y
      16,                           // ulX:       upper left corner in X
      1,                            // lines:     control lines
      ic[T1plainText].cols,         // cols:      control columns
      NULL,                         // dispText:  
      nc.cyR,                       // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbUpperSp,     // filter: accepts upper/lower/SPACE, converts to uppercase
      "ForceUpper Sp",              // label:     
      ZERO,                         // labY:      
      -14,                          // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T1lcaseText]              // nextCtrl:  link in next structure
   },
   {  //* 'Lowercase' Textbox - - - - - - - - - - - - - - - - -    T1lcaseText *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      11,                           // ulY:       upper left corner in Y
      16,                           // ulX:       upper left corner in X
      1,                            // lines:     control lines
      ic[T1plainText].cols,         // cols:      control columns
      "cUrSoR sTaRtS aT lEfT",      // dispText:  
      nc.gyR,                       // nColor:    non-focus color
      nc.brgr,                      // fColor:    focus color
      tbLowerSp,     // filter: accepts upper/lower/SPACE, converts to lowercase
      "ForceLower Sp",              // label:     
      ZERO,                         // labY:      
      -14,                          // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T1rtlText]                // nextCtrl:  link in next structure
   },
   {  //* 'Right-to-left' Textbox - - - - - - - - - - - - - - - -    T1rtlText *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[T1lcaseText].ulY + 2), // ulY:       upper left corner in Y
      short(ic[T1lcaseText].ulX),   // ulX:       upper left corner in X
      1,                            // lines:     control lines
      ic[T1plainText].cols,         // cols:      control columns
      rtlTextData,                  // dispText:
      nc.bw,                        // nColor:    non-focus color
      nc.maG,                       // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "All Printing\n (RTL)",       // label:     
      ZERO,                         // labY:      
      -14,                          // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T1multiText]              // nextCtrl:  link in next structure
   },
   {  //* 'Multi-line' Textbox  - - - - - - - - - - - - - - - -    T1multiText *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[T1rtlText].ulY + 2), // ulY:       upper left corner in Y
      short(ic[T1rtlText].ulX),     // ulX:       upper left corner in X
      4,                            // lines:     control lines
      ic[T1plainText].cols,         // cols:      control columns
      multiTextData[t1MtextOption], // dispText:  initial display text
      nc.brR,                       // nColor:    non-focus color
      nc.grR,                       // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Multi-Line TB\n (all print)",// label:     
      ZERO,                         // labY:      
      -14,                          // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T1rbGroupA]               // nextCtrl:  link in next structure
   },
   {  //* 'Radiobutton Group' button A- - - - - - - - - - - - - -   T1rbGroupA *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      true,                         // rbSelect:  default selection
      short(ic[T1plainText].ulY + 1), // ulY:       upper left corner in Y
      43,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "rbtS3s Button Group A",      // label:     label text
      ZERO,                         // labY:      label offset
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T1rbGroupB]               // nextCtrl:  link in next structure
   },
   {  //* 'Radiobutton Group' button B- - - - - - - - - - - - - -   T1rbGroupB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[T1rbGroupA].ulY + 1),// ulY:       upper left corner in Y
      ic[T1rbGroupA].ulX,           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "rbtS3s Button Group B",      // label:     label text
      ZERO,                         // labY:      label offset
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T1rbGroupC]               // nextCtrl:  link in next structure
   },
   {  //* 'Radiobutton Group' button C- - - - - - - - - - - - - -   T1rbGroupC *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[T1rbGroupB].ulY + 1),// ulY:       upper left corner in Y
      ic[T1rbGroupB].ulX,           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "rbtS3s Button Group C",      // label:     label text
      ZERO,                         // labY:      labell offset
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T1rbStd5]                 // nextCtrl:  link in next structure
   },
   {  //* 'Standard, 5-wide, angle brackets' radio button - - - - -   T1rbStd5 *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5 chars wide
      false,                        // rbSelect:  default selection
      short(ic[T1rbGroupC].ulY + 3),// ulY:       upper left corner in Y
      ic[T1rbGroupC].ulX,           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.re,                        // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "rbtS5a Standard 5 wide",     // label:     label text
      ZERO,                         // labY:      label offset
      7,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T1rbCust3]                // nextCtrl:  link in next structure
   },
   {  //* 'Custom Radiobutton' - 3-wide - - - - - - - - - - - - - -  T1rbCust3 *
      dctRADIOBUTTON,               // type:      
      rbtC3,                        // rbSubtype: custom, 3 chars wide
      true,                         // rbSelect:  default selection
      short(ic[T1rbStd5].ulY + 2),  // ulY:       upper left corner in Y
      ic[T1rbStd5].ulX,             // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      "{#}",                        // dispText:  custom text definition
      nc.re,                        // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "rbtC3 Custom, 3 wide",       // label:
      ZERO,                         // labY:      
      7,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T1rbCust5]                // nextCtrl:  link in next structure
   },
   {  //* 'Custom Radiobutton' - 5-wide - - - - - - - - - - - - - -  T1rbCust5 *
      dctRADIOBUTTON,               // type:      
      rbtC5,                        // rbSubtype: custom, 5 chars wide
      false,                        // rbSelect:  default selection
      short(ic[T1rbCust3].ulY + 2), // ulY:       upper left corner in Y
      ic[T1rbCust3].ulX,            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      "{ % }",                      // dispText:  custom text definition
      nc.re,                        // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "rbtC5 Custom, 5 wide",       // label:     label text
      ZERO,                         // labY:      label offset
      7,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T1trackr1]                // nextCtrl:  link in next structure
   },
   {  //* Non-active text box tracking T1rbCust5 radio button - - -  T1trackr1 *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[T1rbCust5].ulY + 1), // ulY:       upper left corner in Y
      ic[T1rbCust5].ulX,            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      15,                           // cols:      control columns
      NULL,                         // dispText:  initially blank
      nc.bkgy,                      // nColor:    non-focus color
      nc.bkgy,                      // fColor:    focus color
      tbPrint,                      // filter:    accepts all printing chars
      "Track rbtC5",                // label:     label text
      ZERO,                         // labY:      label offset
      17,                           // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    allow control to gain focus
      &ic[T1trackr2]                // nextCtrl:  link in next structure
   },
   {  //* Non-active radio button tracking T1rbCust5 radio button -  T1trackr2 *
      dctRADIOBUTTON,               // type:      
      rbtC1,                        // rbSubtype: custom, 1 wide
      false,                        // rbSelect:  default selection
      ic[T1trackr1].ulY,            // ulY:       upper left corner in Y
      short(ic[T1trackr1].ulX - 2), // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      "S",                          // dispText:  custom text definition
      nc.bkgy,                      // nColor:    non-focus color
      nc.bkgy,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     no label
      ZERO,                         // labY:      label offset
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    allow control to gain focus
      &ic[T1insOver]                // nextCtrl:  link in next structure
   },
   {  //* 'Insert/Overstrike' indicator radiobutton (inactive) - -   T1insOver *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[T1trackr2].ulY + 2), // ulY:       upper left corner in Y
      ic[T1trackr2].ulX,            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a) for standard button types
      nc.grR,                       // nColor:    non-focus color
      nc.grR,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Overstrike Mode",            // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    view only
      &ic[T1SysCb]                  // nextCtrl:  link in next structure
   },
   {  //* 'System-clipboard' indicator radiobutton (inactive)  - - - - T1SysCb *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[T1insOver].ulY + 1), // ulY:       upper left corner in Y
      ic[T1insOver].ulX,            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a) for standard button types
      nc.grR,                       // nColor:    non-focus color
      nc.grR,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "System Clipboard Connected", // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    view only
      NULL                          // nextCtrl:  link in next structure
   },
} ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;
   Test01Dialog = dp ;              // give callback method access to the dialog


   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Print dialog window's static text *
      LineDef  lDef(ncltHORIZ, ncltSINGLE, 2, ZERO, dialogCOLS, dColor) ;
      dp->DrawLine ( lDef ) ;                   // divide window horizontally
      dp->WriteString ( 1, 9, Head1, dColor ) ; // dialog description
      dp->DrawBox ( 3, 41, 5, 31, dColor ) ;    // group box
      dp->WriteString ( ic[T1multiText].ulY + ic[T1multiText].lines, 
                        ic[T1multiText].ulX, 
                        "See command-line help for additional test options.", 
                        nc.brbl ) ;

      //* Some of the textboxes have existing text.         *
      //* - Leave cursor position of T1ucaseText and        *
      //*   T1multiText at the default (top left) position. *
      //* - Set cursor for T1plainText to the 'append'      *
      //*   (end-of-text) position.                         *
      dp->SetTextboxCursor ( T1plainText, tbcpAPPEND ) ;

      //* Configure T1numberText for right-justified editing. *
      dp->SetTextboxCursor ( T1numberText, tbcpRIGHTJUST ) ;

      //* Configure T1rtlText for right-to-left text editing. *
      dp->DrawContentsAsRTL ( T1rtlText ) ;

      //* Set three radio buttons as an exclusive-OR group         *
      //* (only one can be selected at any moment).                *
      short XorGroup[4] = { T1rbGroupA, T1rbGroupB, T1rbGroupC, -1 } ;
      dp->GroupRadiobuttons ( XorGroup ) ;

      //* Enable audible alert for invalid textbox input.*
      dp->TextboxAlert ( T1plainText, true ) ;
      dp->TextboxAlert ( T1numberText, true ) ;
      dp->TextboxAlert ( T1numericText, true ) ;
      dp->TextboxAlert ( T1ucaseText, true ) ;
      dp->TextboxAlert ( T1lcaseText, true ) ;
      dp->TextboxAlert ( T1rtlText, true ) ;
      dp->TextboxAlert ( T1multiText, true ) ;

      //* Make everything visible *
      dp->RefreshWin () ;

      //* If the "wl-clipboard" utilities, "wl-copy" and "wl-paste" are        *
      //* installed on the local system, then establish communications         *
      //* between the Wayland system clipboard and the NcDialog's              *
      //* Textbox Local Clipboard.                                             *
      //* If unable to establish a connection with the system clipboard,       *
      //* copy/cut/paste operations will use the local clipboard only.         *
      //* CTRL+C == Copy, CTRL+X == Cut, CTRL+V == Paste, CTRL+A == Select all,*
      //* SHIFT+LeftArrow and SHIFT+RightArrow == Select by character.         *
      if ( (dp->wcbEnable ()) != false )
         dp->SetRadiobuttonState ( T1SysCb, true ) ;

      //* Establish a call-back method that can be called from within the *
      //* NcDialog code. Called each time through a control's user-input  *
      //* loop so we can manually modify the controls when necessary.     *
      dp->EstablishCallback ( &T01_ControlUpdate ) ;

      //* If user has specified that we should exercise the                *
      //* 'DisplayTextboxMessage' method. Please give particular attention *
      //* to the initialization constructors of the dtbmData class.        *
      //* See note in method header for more information.                  *
      if ( t1dtmExercise )
      {
         const char* singleText = t1dtmRgt2Left ? dtmSingleLineRtlText 
                                                : dtmSingleLineText ;
         const char* multiText  = t1dtmRgt2Left ? dtmMultiLineRtlText 
                                                : dtmMultiLineText ;
         const attr_t* colorText = t1dtmColorful ? dtmMultiColor : dtmMonoColor ;
         short ctrSingle = t1dtmCentered ? 
                           (t1dtmColorful ? ic[T1plainText].cols : 1) : 0 ;
         short ctrMulti  = t1dtmCentered ? 
                           (ic[T1multiText].lines * ic[T1multiText].cols 
                            + ic[T1multiText].lines) : 0 ;

         dtbmData sMsg( singleText, colorText, true, ctrSingle, t1dtmRgt2Left ) ;
         dp->DisplayTextboxMessage ( T1plainText, sMsg ) ;
         dtbmData uMsg( multiText, colorText, true, ctrMulti, t1dtmRgt2Left ) ;
         dp->DisplayTextboxMessage ( T1multiText, uMsg ) ;
      }  // end t1dtmExercise

#if 0    // MULTI-LINE, RTL STRESS TESTING
#if 0
/*
'אונדזער יונגערמאַן מענטש ביינגז וועט אָפֿט - אָפֿט - אַנטוישן אונדז. אָבער יעדער איצט און דעריבער זיי וועט יבערראַשן אונדז אין וועגן אַז
מאַכן אַלע די אנדערע מיסט ווערטיק. זיין אַז יבערראַשן. '
*/

//* “Our fellow human beings will often – often – disappoint us. 
//*  But every now and then they will surprise us in ways that 
//*  make all the other garbage worthwhile. Be that surprise.” 
gString g1        // Test data for RTL multi-line controls (not yet implemented)
(
"'אונדזער יונגערמאַן "
"מענטש ביינגז וועט אָפֿט - "
"אָפֿט - אַנטוישן אונדז. "
"אָבער יעדער איצט און "
"דעריבער זיי וועט יבערראַשן אונדז אין וועגן אַז"
"מאַכן אַלע די אנדערע "
"מיסט ווערטיק. זיין אַז יבערראַשן. '"
) ;

#elif 0
gString g1        // Test data for RTL multi-line controls (not yet implemented)
(
"أغشم من السيل أغن من وليته عن السرقة أغنى الأغنياء من لم يكن للبخل أسيراً أغيرة وجبنًا آفة الحديث الكذب"
) ;
#elif 0
gString g1( "abcdefghijklmnopqrstuvw xyz" // rowboat
          ) ;
#elif 1
gString g1( "abcdef ghijklm nopqrs tuvwxy " // full boat
            "zabcdef ghijkl mnopqr stuvwxy "
            "zabcde fghijk lmnopqr stuvwx# "
            "(overflow text is stored in 'mlTail')" ) ;
#else
gString g1 (            // full Chinese boat (refugees?)
"光棍儿贱人寻求兼容女人"
"光棍儿贱人寻求兼容女人"
"光棍儿贱人寻求兼容女人"
"光棍儿贱人寻求兼容女人"
"光棍儿贱人寻求兼容女人"
"光棍儿贱人寻求兼容女人"
"光棍儿贱人寻求兼容女人"
"光棍儿贱人寻求兼容女人"
"光棍儿贱人寻求兼容女人"
"光棍儿贱人寻求兼容女人"
"光棍儿贱人寻求兼容女人"
) ;
#endif

dp->SetTextboxText ( T1multiText, g1 ) ;
dp->DrawContentsAsRTL ( T1multiText ) ;
dp->RefreshWin () ;
//dp->SetTextboxCursor ( T1multiText, tbcpAPPEND ) ;
//dp->DisplayTextboxTail ( T1multiText ) ;
#endif   // MULTI-LINE STRESS TEST

      //* Let user play with the controls *
      //* Note that there are no hotkeys defined for the controls 
      //* in this dialog; therefore, the user interface is quite 
      //* simple even though there are many different controls 
      //* and control types used. Compare with Test08() in which ALL
      //* active controls have been assigned hotkeys.
      uiInfo Info ;                 // user interface data returned here
      short  icIndex = ZERO ;       // index of control with input focus
      bool   done = false ;         // loop control
      while ( ! done )
      {
         //*******************************************
         //* If focus is currently on a Pushbutton   *
         //*******************************************
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            //* Get user input for this control.                           *
            //* Returns when the Pushbutton is pressed OR when the user    *
            //* requests that the input focus moves to another control.    *
            if ( !Info.viaHotkey )
               icIndex = dp->EditPushbutton ( Info ) ;
            //* However, If we arrived here via 'Hotkey', then calling the *
            //* edit method is unnecessary for binary controls like        *
            //* Pushbuttons. Just retrieve the edit data.                  *
            else
               Info.HotData2Primary () ;

            //* If the Pushbutton was pressed *
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == T1okPush )         // finish and exit
               {
                  // DO STUFF
                  done = true ;
               }
               else if ( Info.ctrlIndex == T1cancelPush ) // cancel and exit
               {
                  // DO STUFF
                  done = true ;
               }
               else if ( Info.ctrlIndex == T1applyPush ) // apply specified changes
               {
                  // DO STUFF

                  //* Politely move to OK button, so user can close the dialog *
                  while ( icIndex != T1okPush )
                     icIndex = dp->PrevControl () ;
                  icIndex = dp->PrevControl () ;
                  Info.keyIn = nckTAB ;
               }
               // (no other pushbuttons defined in this dialog)
            }
            else
            { /* No button press, so nothing to do */ }
         }

         //*******************************************
         //* If focus is currently on a Text Box     *
         //*******************************************
         else if ( ic[icIndex].type == dctTEXTBOX )
         {
            //* Allow user to modify the text data in the text box.    *
            //* Returns when edit is complete.                         *
            icIndex = dp->EditTextbox ( Info ) ;

            //* If a control's data have changed *
            if ( Info.dataMod != false )
            {
               //* Get a copy of the text and display it *
               //* in the application's status window.   *
               gString tBuff, oBuff ;
               short len = dp->GetTextboxText ( icIndex, tBuff ) ;
               oBuff.compose( L"control index:%02hd len:%02hd \"%S\"",
                              &icIndex, &len, tBuff.gstr() ) ;
               StatWin ( oBuff.gstr() ) ;

               switch ( Info.ctrlIndex )  // which text box has the input focus
               {
                  case T1plainText:
                     // DO STUFF
                     break ;
                  case T1numberText:
                     // DO STUFF
                     break ;
                  case T1numericText:
                     // DO STUFF
                     break ;
                  case T1ucaseText:
                     // DO STUFF
                     break ;
                  case T1lcaseText:
                     // DO STUFF
                     break ;
                  case T1multiText:
                     // DO STUFF
                     break ;
                  default:
                     // no other ACTIVE text boxes defined for this control
                     // (inactive controls cannot receive the input focus)
                     break ;
               }
            }
         }

         //*******************************************
         //* If focus is currently on a Scroll Box   *
         //*******************************************
         else if ( ic[icIndex].type == dctSCROLLBOX )
         {
            //* Allow user to modify the selected member of the scroll box *
            //* Returns when edit is complete.                             *
            icIndex = dp->EditScrollbox ( Info ) ;
            
            // THERE ARE NO SCROLL BOXES DEFINED IN THIS DIALOG
         }

         //*******************************************
         //* If focus is currently on a Radio Button *
         //*******************************************
         else if ( ic[icIndex].type == dctRADIOBUTTON )
         {
            //* Get user input for this control                            *
            //* Returns when radio-button control or radio-button group is *
            //* about to lose the input focus.                             *
            //* A selection may or may not have been made.                 *
            icIndex = dp->EditRadiobutton ( Info ) ;

            //* If 'selected' button is a member of a button group *
            if ( Info.selMember < MAX_DIALOG_CONTROLS )
            {
               //* If a new 'selected' member of the group *
               //* OR previous selection was re-selected   *
               if ( Info.dataMod != false || Info.keyIn == nckENTER )
               {
                  char groupName = (char)(Info.selMember==T1rbGroupA ? 'A' : 
                                         (Info.selMember==T1rbGroupB ? 'B' : 'C')) ;
                  gString  oBuff ;
                  oBuff.compose ( L"In Radio Button group, button %c is now selected", 
                                  &groupName ) ;
                  StatWin ( oBuff.gstr() ) ;

                  switch ( Info.selMember )  // which member of group was 'selected'
                  {
                     case T1rbGroupA:
                        // DO STUFF
                        break ;
                     case T1rbGroupB:
                        // DO STUFF
                        break ;
                     case T1rbGroupC:
                        // DO STUFF
                        break ;
                     default:
                        // there are no other Radio Buttons which are members 
                        // of a button group defined in this dialog
                        break ;
                  }
               }
            }
            //* Else, button is an independent button *
            //* (Info.selMember==MAX_DIALOG_CONTROLS) *
            else
            {
               //* If a radio button's state has changed *
               if ( Info.dataMod != false )
               {
                  gString  oBuff ;
                  oBuff.compose ( L"Independent Radio Button #%hd is now %s", 
                                  &Info.ctrlIndex, 
                                  (Info.isSel ? "Selected" : "Un-selected") ) ;
                  StatWin ( oBuff.gstr() ) ;

                  switch ( Info.ctrlIndex )  // which button has the input focus
                  {
                     case T1rbStd5:
                        // DO STUFF
                        break ;
                     case T1rbCust3:
                        // DO STUFF
                        break ;
                     case T1rbCust5:
                        // DO STUFF
                        break ;
                     default:
                        // there are no other ACTIVE independent Radio Buttons
                        // defined in this dialog
                        break ;
                  }
               }
            }
         }

         //* If user exited the control edit method via a hotkey,*
         //* then the new control already has focus. Otherwise,  *
         //* move input focus to next/previous control.          *
         if ( done == false && Info.viaHotkey == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }     // while(!done)

      //* If currently connected to the system clipboard, *
      //* terminate the connection, delete temp files and *
      //* return resources to the system.                 *
      dp->wcbDisable () ;

   }        // if((dp->OpenWindow())==OK)
   else
   {  //* Most likely cause of dialog not opening is that the terminal window  *
      //* is too small. Give user a clue...                                    *
      short neededLines = (dialogROWS + minTestULY) >= minTERMROWS ?
                          (dialogROWS + minTestULY) : minTERMROWS, 
            neededCols  = dialogCOLS >= minTERMCOLS ? dialogCOLS : minTERMCOLS ;
      gString gs ;
      gs.compose ( L" Sorry, size of terminal window must be at least %hd x %hd for this test. ", 
                   &neededCols, &neededLines ) ;
      StatWin ( gs.gstr(), nc.cyG ) ;
      sleep ( 5 ) ;
   }

   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

}  //* End Test01() *

//*************************
//*  T01_ControlUpdate    *
//*************************
//******************************************************************************
//* This is a callback method for manually updating the controls in the        *
//* Test01() dialog.                                                           *
//*                                                                            *
//*  For this test, the following are updated by the callback method:          *
//*   1. The Insert/Overstrike indicator is not selectable by the user;        *
//*      however, it tracks the state of the Insert key through this           *
//*      callback  method. If the NCurses engine is waiting for key input      *
//*      from the user, (and this is how the time is spent), it will monitor   *
//*      the state of this toggle because it is needed for text field input.   *
//*      Since we have the information available, why not display it!          *
//*   2. Radio Button 14 status is tracked by two other non-selectable controls*
//*      which are updated by this method.                                     *
//*   3.                                                                       *
//*                                                                            *
//* Input  : currIndex: index of control that currently has focus              *
//*          wkey     : user's key input data                                  *
//*          firstTime: the EstablishCallback() method calls this method once  *
//*                     with firstTime==true, to perform any required          *
//*                     initialization. Subsequently, the NcDialog class       *
//*                     always calls with firstTime==false.                    *
//* Returns: OK                                                                *
//******************************************************************************
//* Important Note: This method makes some intimate assumptions about what     *
//* is happening in the method that established the callback. If changes are   *
//* made in the establishing method that affect the callback functionality,    *
//* be sure to update this method to address those changes.                    *
//*                                                                            *
//* Programmer's Note: See dialog control index constants at file scope above. *
//******************************************************************************

short T01_ControlUpdate ( const short currIndex, const wkeyCode wkey, bool firstTime )
{
static bool    oldOver, oldRBT ;
static char    tbText[2][12] = { { "rbtC5 RESET"}, { "rbtC5 SET"} } ;

   //*************************
   //* First-time processing *
   //*************************
   if ( firstTime != false )
   {
      //* Initialize  the static variables *
      oldOver = Test01Dialog->IsOverstrike () ; // initial state of insert/overstrike flag
      bool currOver ;
      Test01Dialog->GetRadiobuttonState ( T1insOver, currOver ) ;
      if ( currOver != oldOver )
         Test01Dialog->SetRadiobuttonState ( T1insOver, oldOver ) ;

      //* Text box (index == T1trackr1) and non-active radio button                   *
      //* (index == T1trackr2) track state of rbtC5 radio button (index == T1rbCust5) *
      Test01Dialog->GetRadiobuttonState ( T1rbCust5, oldRBT ) ;
      Test01Dialog->SetTextboxText ( T1trackr1, (oldRBT ? tbText[1] : tbText[0]) ) ;
      Test01Dialog->SetRadiobuttonState ( T1trackr2, oldRBT ) ;
   }

   //***********************
   //* Standard processing *
   //***********************
   //* If necessary, update the non-active radio button *
   //* that indicates the insert/overstrike state       *
   bool newOver = Test01Dialog->IsOverstrike () ; // current state of insert/overstrike
   if ( newOver != oldOver )
   {
      Test01Dialog->SetRadiobuttonState ( T1insOver, oldOver = newOver ) ;
   }

   //* Text Box T1trackr1 and Radio Button T1trackr2 track *
   //* the state of radio button T1rbCust5                 *
   bool newRBT ;   
   Test01Dialog->GetRadiobuttonState ( T1rbCust5, newRBT ) ;
   if ( newRBT != oldRBT )
   {
      Test01Dialog->SetTextboxText ( T1trackr1, (newRBT ? tbText[1] : tbText[0]) ) ;
      Test01Dialog->SetRadiobuttonState ( T1trackr2, newRBT ) ;
      oldRBT = newRBT ;
   }

   return OK ;
   
}  //* End T01_ControlUpdate() *

//*************************
//*       Test02          *
//*************************
//******************************************************************************
//* Test dctSCROLLBOX dialog control.                                          *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Test02 ( void )
{
static const short dialogROWS = 18 ;         // display lines
static const short dialogCOLS = 75 ;         // display columns
static const short controlsDEFINED = 7 ;     // # of controls in dialog
static const short DATA_ITEMS1 = 11 ;        // elements in string and color arrays
static const short DATA_ITEMS2 = 16 ;        // elements in string and color arrays
static const short ITEM_LEN = 27 ;           // chars per item ( incl. NULLCHAR)
static char Help1[] = "Use TAB to move among controls, and scroll keys in scrolling controls. " ;

//* Text data are passed to the Scrollbox constructor in a    *
//* variety of formats, to stress-test the parsing algorithm. *
#if 1
static char dispData1[DATA_ITEMS2][ITEM_LEN] = 
{  // Full-width items - exact width for target control
   "0xxxxxxxxxxxxxxxxxxxxxxxx0",
   "1xxxxxxxxxxxxxxxxxxxxxxxx1",
   "2xxxxxxxxxxxxxxxxxxxxxxxx2",
   "3xxxxxxxxxxxxxxxxxxxxxxxx3",
   "4xxxxxxxxxxxxxxxxxxxxxxxx4",
   "5xxxxxxxxxxxxxxxxxxxxxxxx5",
   "6xxxxxxxxxxxxxxxxxxxxxxxx6",
   "7xxxxxxxxxxxxxxxxxxxxxxxx7",
   "8xxxxxxxxxxxxxxxxxxxxxxxx8",
   "9xxxxxxxxxxxxxxxxxxxxxxxx9",
   "AxxxxxxxxxxxxxxxxxxxxxxxxA",
   "BxxxxxxxxxxxxxxxxxxxxxxxxB",
   "CxxxxxxxxxxxxxxxxxxxxxxxxC",
   "DxxxxxxxxxxxxxxxxxxxxxxxxD",
   "ExxxxxxxxxxxxxxxxxxxxxxxxE",
   "FxxxxxxxxxxxxxxxxxxxxxxxxF"
} ;
#elif 0
static char dispData1[DATA_ITEMS2][ITEM_LEN] =  
{  // Mixed-width items <= to required length (will be padded on right)
   "0xxxxxxxxxxxxxxxxxxxxx0",
   "1xxxxxxxxxxxxxxxxxxxxxxx1",
   "2xxxxxxxxxxxxxxxxxxxxxx2",
   "3xxxxxxxxxxxxxxxxxxxxxxxx3",
   "4xxxxxxxxxxxxxxxxxx4",
   "5xxxxxxxxxxxxxxxxx5",
   "6xxxxxxxxxxxxxxxx6",
   "7xxxxxxxxxxxxxxx7",
   "8xxxxxxxxxxxxxx8",
   "9xxxxxxxxxxxxxxxxxxxxx9",
   "AxxxxxxxxxxxxxxxxxxxxxxA",
   "BxxxxB",
   "CxxxxxxxxC",
   "DxxxxxxxxxxxD",
   "ExxxxxxxxxxxxxxxxE",
   "FxxxxxxxxxxF"
} ;
#else
static char dispData1[DATA_ITEMS2*ITEM_LEN] = 
{  // Single data string - will be divided into correct lengths
   "0xxxxxxxxxxx"
   "xxxxxxxxxxxxx01xxxxxxxxxxxxxxxxxxxxxxxx1"
   "2xxxxxxxxxxxxx"
   "xxxxxxxxxxx2"
   "3xxxxxxxxxxxxxxxxxxxxxxxx34xxxxxxxxxxxxxxxxxxxxxxxx4"
   "5xxxxxxxxxxxx"
   "xxxxxxxxxxxx56xxxxxxxxxxx"
   "xxxxxxxxxxxxx6"
   "7xxxxxxxxxxxxxxxxxxxxxxxx7"
   "8xxxxxxxxxxxxxxxxxxxxxxxx8"
   "9xxxxxxxxxxxxxxxxxxxxxxxx9"
   "AxxxxxxxxxxxxxxxxxxxxxxxxA"
   "BxxxxxxxxxxxxxxxxxxxxxxxxB"
   "CxxxxxxxxxxxxxxxxxxxxxxxxC"
   "DxxxxxxxxxxxxxxxxxxxxxxxxD"
   "ExxxxxxxxxxxxxxxxxxxxxxxxE"
   "FxxxxxxxxxxxxxxxxxxxxxxxxF"
} ;
#endif
attr_t monoColor[2] = { attrDFLT, nc.gr } ;
attr_t multiColor[DATA_ITEMS2] = 
{ nc.bw, nc.re, nc.gr, nc.br, nc.bl, nc.ma, nc.cy, nc.gy, 
  nc.bwR, nc.reR, nc.grR, nc.brR, nc.blR, nc.maR, nc.cyR, nc.gyR } ;
const short buttonWidth = 8 ;
static const char OkButtonText[buttonWidth+1]     = "   OK   " ;
static const char CancelButtonText[buttonWidth+1] = " CANCEL " ;
static const char ApplyButtonText[buttonWidth+1]  = " APPLY  " ;
short ctrY = termRows/2,               // terminal window center in Y
      ctrX = termCols/2,               // terminal window center in X
      ulY  = ctrY - dialogROWS / 2,    // upper left corner in Y
      ulX  = ctrX - dialogCOLS / 2 ;   // upper left corner in X
attr_t dColor = nc.blR ;

   StatWin ( "Opening Test Dialog #2 - Scroll Box Controls" ) ;

InitCtrl ic[controlsDEFINED] =   // array of dialog control initialization objects
{
   {  //* 'OK' pushbutton - - - - - - - - - - - - - - - - - - - - -   T2okPush *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 2),        // ulY:       upper left corner in Y
      14,                           // ulX:       upper left corner in X
      1,                            // lines:     control lines
      buttonWidth,                  // cols:      control columns
      OkButtonText,                 // dispText:  
      nc.re,                        // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[1],                       // nextCtrl:  link in next structure
   },
   {  //* 'CANCEL' pushbutton - - - - - - - - - - - - - - - - -   T2cancelPush *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 2),        // ulY:       upper left corner in Y
      33,                           // ulX:       upper left corner in X
      1,                            // lines:     control lines
      buttonWidth,                  // cols:      control columns
      CancelButtonText,             // dispText:  
      nc.re,                        // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[2],                       // nextCtrl:  link in next structure
   },
   {  //* 'APPLY' pushbutton - - - - - - - - - - - - - - - - - -   T2applyPush *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 2),        // ulY:       upper left corner in Y
      52,                           // ulX:       upper left corner in X
      1,                            // lines:     control lines
      buttonWidth,                  // cols:      control columns
      ApplyButtonText,              // dispText:  
      nc.re,                        // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[3],                       // nextCtrl:  link in next structure
   },
   { //* Scrollbox #1 (single-color data) - - - - - - - - - - - -   T2singleSB *
      dctSCROLLBOX,                 // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      4,                            // ulY:       upper left corner in Y
      7,                            // ulX:       upper left corner in X
      8,                            // lines:     control lines
      28,                           // cols:      control columns
      (const char*)&dispData1,      // dispText:  text-data array
      nc.gyR,                       // nColor:    non-focus border color
      nc.gr,                        // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "Single-Color Data",          // label:     label text
      -1,                           // labY:      label offset
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      DATA_ITEMS1,                  // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    index of initial highlighted element
      monoColor,                    // scrColor:  color-attribute list
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[4],                       // nextCtrl:  link in next structure
   },
   { //* Scrollbox #2 (multi-color data ) - - - - - - - - - - - - -  T2multiSB *
      dctSCROLLBOX,                 // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      4,                            // ulY:       upper left corner in Y
      40,                           // ulX:       upper left corner in X
      10,                           // lines:     control lines
      28,                           // cols:      control columns
      (const char*)&dispData1,      // dispText:  text-data array
      nc.gyR,                       // nColor:    non-focus border color
      nc.gr,                        // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "Multi-Color Data",           // label:     label text
      -1,                           // labY:      label offset
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      DATA_ITEMS2,                  // scrItems:  number of elements in text/color arrays
      2,                            // scrSel:    index of initial highlighted element
      multiColor,                   // scrColor:  color-attribute list
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[5],                       // nextCtrl:  link in next structure
   },
   { //* Textbox #1 (inactive) track SB selections  - - - - - - - -    T2tbONE *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      12,                           // ulY:       upper left corner in Y
      23,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      4,                            // cols:      control columns
      NULL,                         // dispText:  initial display text
      nc.cyR,                       // nColor:    non-focus color
      nc.cyR,                       // fColor:    focus color
      tbPrint,                      // filter:    accepts all printing text
      "Item Selected: ",            // label:     label text
      ZERO,                         // labY:      label offset
      -15,                          // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    allow control to gain focus
      &ic[6],                       // nextCtrl:  link in next structure
   },
   { //* Textbox #2 (inactive) track SB selections  - - - - - - - -    T2tbTWO *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      14,                           // ulY:       upper left corner in Y
      56,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      4,                            // cols:      control columns
      NULL,                         // dispText:  initial display text
      nc.cyR,                       // nColor:    non-focus color
      nc.cyR,                       // fColor:    focus color
      tbPrint,                      // filter:    accepts all printing text
      "Item Selected: ",            // label:     label text
      ZERO,                         // labY:      label offset
      -15,                          // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   },
} ;


   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       NULL,           // no dialog title (but see below)
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;
   Test02Dialog = dp ;              // give callback method access to the dialog

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Give the dialog a title.                                             *
      //* Note that the dialog title could have been specified during          *
      //* instantiation; however, we do it here to demonstrate the use of the  *
      //* SetDialogTitle() method, and as a means of testing its functionality.*
      //* Text data may be provided in any of the following formats:           *
      //* UTF-8 encoded string, wchar_t array or gString class.                *
      dp->SetDialogTitle ( " Test dialog for scroll box controls " ) ;

      //* Print dialog window's static text *
      LineDef  lDef(ncltHORIZ, ncltSINGLE, 2, ZERO, dialogCOLS, dColor) ;
      dp->DrawLine ( lDef ) ;
      dp->WriteString ( 1, 2, Help1, dColor ) ;

      dp->RefreshWin () ;

      //* Establish a call-back method that can be called from within the *
      //* NcDialog code. Called each time through a control's user-input  *
      //* loop so we can manually modify the controls when necessary.     *
      dp->EstablishCallback ( &T02_ControlUpdate ) ;

      //* Allow user to manipulate the controls.     *
      //* (Please see the notes in NcDialog.hpp on   *
      //* values returned (from the editing routines)*
      uiInfo Info ;                 // user interface data returned here
      short  icIndex = ZERO ;       // index of control with input focus
      bool   done = false ;         // loop control
      while ( ! done )
      {
         //*******************************************
         //* If focus is currently on a Pushbutton   *
         //*******************************************
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            //* Get user input for this control                            *
            //* Returns when the Pushbutton is pressed OR when the user    *
            //* requests that the input focus moves to another control.    *
            icIndex = dp->EditPushbutton ( Info ) ;

            //* If the Pushbutton was pressed *
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == T2okPush )         // finish and exit
               {
                  // DO STUFF
                  done = true ;
               }
               else if ( Info.ctrlIndex == T2cancelPush ) // cancel and exit
               {
                  // DO STUFF
                  done = true ;
               }
               else if ( Info.ctrlIndex == T2applyPush ) // apply specified changes
               {
                  // DO STUFF
                  
                  //* Politely move to OK button, so user can close the dialog *
                  while ( icIndex > ZERO )
                     icIndex = dp->PrevControl () ;
               }
               // (no other pushbuttons defined in this dialog)
            }
            else
            { /* No button press, so nothing to do */ }
         }

         //*******************************************
         //* If focus is currently on a Scroll Box   *
         //*******************************************
         else if ( ic[icIndex].type == dctSCROLLBOX )
         {
            //* Allow user to modify the selected member of the scroll box *
            //* Returns when edit is complete.                             *
            icIndex = dp->EditScrollbox ( Info ) ;
            
            //* If selection in either Scroll Box control has changed *
            //* OR if previous selection has been re-selected.        *
            if ( Info.dataMod != false || Info.keyIn == nckENTER )
            {
               switch ( Info.selMember )  // which control has been modified
               {
                  case T2singleSB:
                     // DO STUFF
                     break ;
                  case T2multiSB:
                     // DO STUFF
                     break ;
                  default:
                     // no other Scroll Boxes defined in this dialog
                     break ;
               }
            }
         }
         else
            { /* No other ACTIVE control types defined for this dialog */ }

         //* If user exited the control edit method via a hotkey,*
         //* then the new control already has focus. Otherwise,  *
         //* move input focus to next/previous control.          *
         if ( done == false && Info.viaHotkey == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }  // while()
   }        // if((dp->OpenWindow())==OK)
   else
   {  //* Most likely cause of dialog not opening is that the terminal window  *
      //* is too small. Give user a clue...                                    *
      short neededLines = (dialogROWS + minTestULY) >= minTERMROWS ?
                          (dialogROWS + minTestULY) : minTERMROWS, 
            neededCols  = dialogCOLS >= minTERMCOLS ? dialogCOLS : minTERMCOLS ;
      gString gs ;
      gs.compose ( L" Sorry, size of terminal window must be at least %hd x %hd for this test. ", 
                   &neededCols, &neededLines ) ;
      StatWin ( gs.gstr(), nc.cyG ) ;
      sleep ( 5 ) ;
   }

   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

}  //* End Test02() *

//*************************
//*  T02_ControlUpdate    *
//*************************
//******************************************************************************
//* This is a callback method for manually updating the controls in the        *
//* Test02() dialog.                                                           *
//*                                                                            *
//*  For this test, the following are updated by the callback method:          *
//*   1. Each scroll box in the dialog opened in Test02() has an associated    *
//*      external text box that displays the index of the currently-           *
//*      highlighted item in that scroll box. These indicators are updated     *
//*      dynamically as the user scrolls through the data by means of the      *
//*      following code through the remote callback in the main loop of        *
//*      NcDialog::EditScrollBox(). This is a trivial example of tracking      *
//*      the information in the dialog's individual controls, but perhaps      *
//*      more interesting uses for dynamically-updated information could be    *
//*      applied in an application that actually does something more than      *
//*      move electrons around the display.                                    *
//*                                                                            *
//* Input  : currIndex: index of control that currently has focus              *
//*          wkey     : user's key input data                                  *
//*          firstTime: the EstablishCallback() method calls this method once  *
//*                     with firstTime==true, to perform any required          *
//*                     initialization. Subsequently, the NcDialog class       *
//*                     always calls with firstTime==false.                    *
//* Returns: OK                                                                *
//******************************************************************************
//* Important Note: This method makes some intimate assumptions about what     *
//* is happening in the method that established the callback. If changes are   *
//* made in the establishing method that affect the callback functionality,    *
//* be sure to update this method to address those changes.                    *
//*                                                                            *
//* Programmer's Note: See dialog control index constants at file scope above. *
//******************************************************************************

short T02_ControlUpdate ( const short currIndex, const wkeyCode wkey, bool firstTime )
{
static short   sbOneIndex, sbTwoIndex ;
gString  gs ;

   //*************************
   //* First-time processing *
   //*************************
   if ( firstTime != false )
   {
      //* Initialize the index tracking text boxes *
      sbOneIndex = Test02Dialog->GetScrollboxSelect ( T2singleSB ) ;
      gs.compose( L" %02hd ", &sbOneIndex ) ;
      Test02Dialog->SetTextboxText ( T2tbONE, gs ) ;
      sbTwoIndex = Test02Dialog->GetScrollboxSelect ( T2multiSB ) ;
      gs.compose( L" %02hd ", &sbTwoIndex ) ;
      Test02Dialog->SetTextboxText ( T2tbTWO, gs ) ;
   }

   //***********************
   //* Standard processing *
   //***********************
   if ( currIndex == T2singleSB )
   {
      short tmp = Test02Dialog->GetScrollboxSelect ( currIndex ) ;
      if ( tmp != sbOneIndex )
      {
         sbOneIndex = tmp ;
         gs.compose( L" %02hd ", &sbOneIndex ) ;
         Test02Dialog->SetTextboxText ( T2tbONE, gs ) ;
      }
   }
   else if ( currIndex == T2multiSB )
   {
      short tmp = Test02Dialog->GetScrollboxSelect ( currIndex ) ;
      if ( tmp != sbTwoIndex )
      {
         sbTwoIndex = tmp ;
         gs.compose( L" %02hd ", &sbTwoIndex ) ;
         Test02Dialog->SetTextboxText ( T2tbTWO, gs ) ;
      }
   }

   return OK ;
   
}  //* End T02_ControlUpdate() *

//*************************
//*       Test03          *
//*************************
//******************************************************************************
//* This method uses the NcWindow primitives to open a window with             *
//* scrollable data. This is in constrast to opening a scrolling-data dialog   *
//* control inside a dialog window (see Test #2) where the border is           *
//* automatic and data strings are normalized.                                 *
//*                                                                            *
//* Note that the data strings are passed to the primitives as an array of     *
//* pointers instead of a two-dimensional character array as in the dialog,    *
//* and that, unlike with the dialog controls, the data strings passed to the  *
//* primitives must be exactly the right width.                                *
//*                                                                            *
//* Note also that this test uses monochrome scrolling data, while,            *
//* INTERNALLY, dialog scroll boxes use multi-color data exclusively,          *
//* regardless of whether multi-color or monochrome data are specified.        *
//*                                                                            *
//* In short, you would have to be self-destructive to use NcWindow-class      *
//* primitives directly, but this is where we started the development of       *
//* dctSCROLLBOX, dctSCROLLEXT and dctDROPOWN dialog controls, and it remains  *
//* as a demonstration of the internal workings of NcDialog-class scrolling.   *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Test03 ( void )
{
static const short WIN_LINES = 8 ;           // display lines
static const short WIN_COLS = 28 ;           // display columns
static const short DATA_ITEMS = 16 ;         // elements in string array
static const char* dispData[DATA_ITEMS] =    // scrolling-data strings 
{
   "0xxxxxxxxxxxxxxxxxxxxxxxx0",
   "1xxxxxxxxxxxxxxxxxxxxxxxx1",
   "2xxxxxxxxxxxxxxxxxxxxxxxx2",
   "3xxxxxxxxxxxxxxxxxxxxxxxx3",
   "4xxxxxxxxxxxxxxxxxxxxxxxx4",
   "5xxxxxxxxxxxxxxxxxxxxxxxx5",
   "6xxxxxxxxxxxxxxxxxxxxxxxx6",
   "7xxxxxxxxxxxxxxxxxxxxxxxx7",
   "8xxxxxxxxxxxxxxxxxxxxxxxx8",
   "9xxxxxxxxxxxxxxxxxxxxxxxx9",
   "AxxxxxxxxxxxxxxxxxxxxxxxxA",
   "BxxxxxxxxxxxxxxxxxxxxxxxxB",
   "CxxxxxxxxxxxxxxxxxxxxxxxxC",
   "DxxxxxxxxxxxxxxxxxxxxxxxxD",
   "ExxxxxxxxxxxxxxxxxxxxxxxxE",
   "FxxxxxxxxxxxxxxxxxxxxxxxxF"
} ;
static char Header1[] = { "Window with scrolling data" } ;
static char Header2[] = { "constructed from" } ;
static char Header3[] = { "NcWindow-class primitives" } ;
static char Instr1[]  = { "Scroll with Arrow keys, Home, End, PgUp, PgDn" } ;
static char Instr2[]  = { "Press Enter to exit." } ;
short    ctrY    = termRows/2,               // terminal window center in Y
         ctrX    = termCols/2,               // terminal window center in X
         ulY     = ctrY - WIN_LINES / 2,     // upper left corner in Y
         ulX     = ctrX - WIN_COLS / 2 ;     // upper left corner in X
NcWindow* bPtr ;
NcWindow* wPtr ;
attr_t   monoColor = nc.gr ;
short    initIndex = 7 ;
short    hIndex ;                   // index of highlighted item

   StatWin ( "Opening Test Window #3 - Scrolling with NcWindow primitives" ) ;

   nc.WriteString ( ulY-3, ulX+1, Header1, nc.br ) ;
   nc.WriteString ( ulY-2, ulX+6, Header2, nc.br ) ;
   nc.WriteString ( ulY-1, ulX+1, Header3, nc.br ) ;
   nc.WriteString ( ulY+WIN_LINES, ulX-8, Instr1, nc.br ) ;
   nc.WriteString ( ulY+WIN_LINES+1, ulX+4, Instr2, nc.br ) ;
   bPtr = new NcWindow ( WIN_LINES, WIN_COLS, ulY, ulX ) ;
   if ( (bPtr->OpenWindow ()) != ERR )
   {
      bPtr->DrawBorder ( nc.br ) ;
      bPtr->RefreshWin () ;
      wPtr = new NcWindow ( WIN_LINES-2, WIN_COLS-2, ulY+1, ulX+1 ) ;
      if ( (wPtr->OpenWindow ()) != ERR )
      {  //* Paint the data and get the initial item index *
         hIndex = wPtr->PaintData ( dispData, DATA_ITEMS, initIndex, monoColor ) ;
         wPtr->RefreshWin () ;
         // Note: We have to update the Status Window AFTER NcWindow class has
         //       updated its objects
         StatWin ( "Opening Test Window #3 - Scrolling with NcWindow primitives" ) ;

         wkeyCode wk ;
         bool     done = false ;          // loop control
         while ( ! done )                 // allow user to scroll through the data
         {  // Programmer's Note: Because we are receiving 'wide' keycodes from
            // the user, we need to check the key 'type' to determine whether 
            // the key data is a character key or a 'function' key. Note that 
            // all scroll keys (cursor keys) are function keys. This test 
            // prevents valid Polish characters (for instance) from being 
            // interpreted as arrow keys.
            // Please refer to enum wkeyType and the wkeyCode class definition 
            // in NCurses.hpp for additional details.
            wPtr->GetKeyInput ( wk ) ;
            if ( (wPtr->IsScrollKey ( wk )) != false )
            {
               hIndex = wPtr->ScrollData ( wk.key ) ;

               //* Take some action based on user's selection *
               switch ( hIndex )
               {
                  default:
                     break ;
               }
            }
            #if 0    // TESTING OF Track2x METHODS ONLY
            else if (   (wk.key >= '0' && wk.key <= '9') || (wk.key >= 'a' && wk.key <= 'g')
                     || (wk.key == 'T') || (wk.key == 'B') || (wk.key == 'U') 
                     || (wk.key == 'D')  )
            {
               short iIndex = -1 ;
               switch ( wk.key )
               {
                  case '0':   iIndex = ZERO ;   break ;
                  case '1':   iIndex = 1 ;      break ;
                  case '2':   iIndex = 2 ;      break ;
                  case '3':   iIndex = 3 ;      break ;
                  case '4':   iIndex = 4 ;      break ;
                  case '5':   iIndex = 5 ;      break ;
                  case '6':   iIndex = 6 ;      break ;
                  case '7':   iIndex = 7 ;      break ;
                  case '8':   iIndex = 8 ;      break ;
                  case '9':   iIndex = 9 ;      break ;
                  case 'a':   iIndex = 10 ;     break ;
                  case 'b':   iIndex = 11 ;     break ;
                  case 'c':   iIndex = 12 ;     break ;
                  case 'd':   iIndex = 13 ;     break ;
                  case 'e':   iIndex = 14 ;     break ;
                  case 'f':   iIndex = 15 ;     break ;
                  case 'g':   iIndex = 20 ;     break ;  // force invalid index
                  case 'T':   wPtr->Track2Top () ;    break ;
                  case 'B':   wPtr->Track2Bottom () ; break ;
                  case 'D':   wPtr->Track2Next () ;   break ;
                  case 'U':   wPtr->Track2Previous ();break ;
               }
               if ( iIndex >= ZERO )
                  wPtr->Track2Item ( iIndex ) ;
               sleep ( 3 ) ;
               wPtr->RepaintData () ;
            }
            #endif   // TESTING OF Track2x METHODS ONLY
            else
               done = true ;
         }
      }
      //* Release scrolling-data resources & reset tracking variables. Close window. *
      delete wPtr ;
   }
   delete bPtr ;                             // release outer window's resources       

}  //* End Test03() *

//*************************
//*       Test04          *
//*************************
//******************************************************************************
//* Open a basic window (not dialog) and test background colors.               *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Test04 ( void )
{
   NcWindow*  cWin = new NcWindow ( 5, 30, 5, 5 ) ;

   StatWin ( "Opening Test Dialog #4 - NcWindow draw with background color" ) ;
   usleep ( 250000 ) ;

   cWin->OpenWindow () ;
   cWin->DrawBorder ( nc.re ) ;
   cWin->WriteString ( 1, 1, "Line #1...................", nc.re, true ) ;

   StatWin ( "Press A Key to clear interior" ) ;
   nckPause() ;
   cWin->ClearWin (false) ;
   StatWin ( "Press A Key to clear entire window" ) ;
   nckPause() ;
   cWin->ClearWin () ;
   
   StatWin ( "Press A Key to redraw with interior color" ) ;
   nckPause() ;
   cWin->DrawBorder ( nc.reR ) ;
   cWin->SetInteriorColor ( nc.reR ) ;
   cWin->ClearWin ( false ) ;
   cWin->WriteString ( 1, 1, "Line #1...................", nc.reR, true ) ;
   
   StatWin ( "Press A Key to clear interior" ) ;
   nckPause() ;
   cWin->SetInteriorColor ( nc.gyR ) ;
   cWin->ClearWin ( false ) ;

   StatWin ( "Done . . .(Press A Key)" ) ;
   nckPause() ;
   delete cWin ;
   nc.RefreshScreen () ;   // clear residual display data
      
}  //* End Test04() *

//*************************
//*       Test05          *
//*************************
//******************************************************************************
//* Open a dialog to anchor the output from the DisplayRadiobuttonTypes()      *
//* method.                                                                    *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Test05 ( void )
{
static const short dialogROWS = 20 ;   // display lines
static const short dialogCOLS = 60 ;   // display columns
attr_t   dColor = nc.cyR ;
short    ctrY    = termRows/2,         // terminal window center in Y
         ctrX    = termCols/2,         // terminal window center in X
         //* Upper left corner in Y (cannot obscure status window) *
         ulY     = (ctrY - dialogROWS/2-2) >= minTestULY ?
                   (ctrY - dialogROWS/2-2) : minTestULY,
         ulX     = (ctrX - dialogCOLS/2-16) >= ZERO ? // upper left corner in X
                   (ctrX - dialogCOLS/2-16) : ZERO ;

   StatWin ( "Opening Test Dialog #5 - Display Radio Button types" ) ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       "  Some Dialog Window Under Development  ", // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       NULL            // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Note: to the user, this looks like a pushbutton control, *
      //* but for such a simple dialog, it is all we need.         *
      dp->WriteString ( dialogROWS-2, dialogCOLS-18, " Press Any Key ", nc.reG, true ) ;
      dp->RefreshWin () ;
      nckPause() ;
      
      //* Display examples of all Radio Button types *
      //* Note that the coordinates for display are absolute 
      //* screen coordinates, not dialog offsets
      winPos   wPos(ctrY-6,ctrX-22) ;
      dp->DisplayRadiobuttonTypes ( wPos, dColor, nc.blcy ) ;
   }
   else
   {  //* Most likely cause of dialog not opening is that the terminal window  *
      //* is too small. Give user a clue...                                    *
      short neededLines = (dialogROWS + minTestULY) >= minTERMROWS ?
                          (dialogROWS + minTestULY) : minTERMROWS, 
            neededCols  = dialogCOLS >= minTERMCOLS ? dialogCOLS : minTERMCOLS ;
      gString gs ;
      gs.compose ( L" Sorry, size of terminal window must be at least %hd x %hd for this test. ", 
                   &neededCols, &neededLines ) ;
      StatWin ( gs.gstr(), nc.cyG ) ;
      sleep ( 5 ) ;
   }

   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

}  //* End Test05() *

//*************************
//*       Test06          *
//*************************
//******************************************************************************
//* Test the functionality of the radio-button grouping code.                  *
//* Independent radio buttons and a text box are also defined to provide       *
//* additional testing for moving the input focus among controls via hotkeys.  *
//*                                                                            *
//* Note: Please also see Dialogw application, Test07() which implements the   *
//*       same controls, but using multiple languages to demonstrate full      *
//*       support for UTF-8 character encoding.                                *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Test06 ( void )
{
static const short dialogROWS = 12 ;         // display lines
static const short dialogCOLS = 70 ;         // display columns
static const short controlsDEFINED = 15 ;    // # of controls in dialog
short    ctrY    = termRows/2,               // terminal window center in Y
         ctrX    = termCols/2,               // terminal window center in X
         ulY     = ctrY - dialogROWS / 2,    // upper left corner in Y
         ulX     = ctrX - dialogCOLS / 2 ;   // upper left corner in X
attr_t   dColor = nc.blR ;                   // dialog color attribute

   StatWin ( "Opening Test Dialog #6 - Radio Button Group Selection" ) ;

InitCtrl ic[controlsDEFINED] =   // array of dialog control initialization objects
{
   {  //* 'DONE' pushbuttion  - - - - - - - - - - - - - - - - - -   T6donePush *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 2),        // ulY:       upper left corner in Y
      short(dialogCOLS / 2 - 4),    // ulX:       upper left corner in X
      1,                            // lines:     control lines
      8,                            // cols:      control columns
      "  DON^E  ",                  // dispText:  
      nc.gyR,                       // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[1],                       // nextCtrl:  link in next structure
   },
   {  //* Radio Button Group A1  - - - - - - - - - - - - - - - -    T6groupA_1 *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      true,                         // rbSelect:  initial selection state
      2,                            // ulY:       upper left corner in Y
      4,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "^rbtS3s Radio Button #1",    // label:     label text
      ZERO,                         // labY:      label offset
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[2]                        // nextCtrl:  link in next structure
   },
   {  //* Radio Button Group A2  - - - - - - - - - - - - - - - -    T6groupA_2 *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  initial selection state
      3,                            // ulY:       upper left corner in Y
      4,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "r^btS3s Radio Button #2",    // label:     label text
      ZERO,                         // labY:      label offset
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[3]                        // nextCtrl:  link in next structure
   },
   {  //* Radio Button Group A3  - - - - - - - - - - - - - - - -    T6groupA_3 *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  initial selection state
      4,                            // ulY:       upper left corner in Y
      4,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "rbt^S3s Radio Button #3",    // label:     label text
      ZERO,                         // labY:      label offset
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[4]                        // nextCtrl:  link in next structure
   },
   {  //* Radio Button Group A4  - - - - - - - - - - - - - - - -    T6groupA_4 *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  initial selection state
      5,                            // ulY:       upper left corner in Y
      4,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "rbtS3s Radio B^utton #4",    // label:     label text
      ZERO,                         // labY:      label offset
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[5]                        // nextCtrl:  link in next structure
   },
   {  //* Radio Button Group B1  - - - - - - - - - - - - - - - -    T6groupB_1 *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      true,                         // rbSelect:  initial selection state
      2,                            // ulY:       upper left corner in Y
      39,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "rbtS3s Radio But^ton #5",    // label:     label text
      ZERO,                         // labY:      label offset
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[6]                        // nextCtrl:  link in next structure
   },
   {  //* Radio Button Group B2  - - - - - - - - - - - - - - - -    T6groupB_2 *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  initial selection state
      3,                            // ulY:       upper left corner in Y
      39,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "rbtS3s Radio Butt^on #6",    // label:     label text
      ZERO,                         // labY:      label offset
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[7]                        // nextCtrl:  link in next structure
   },
   {  //* Radio Button Group B3  - - - - - - - - - - - - - - - -    T6groupB_3 *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  initial selection state
      4,                            // ulY:       upper left corner in Y
      39,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "rbtS3s Radio Butto^n #7",    // label:     label text
      ZERO,                         // labY:      label offset
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[8]                        // nextCtrl:  link in next structure
   },
   {  //* Stand-alone Radio Button 8 - - - - - - - - - - - - -   T6standAlone8 *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  initial selection state
      6,                            // ulY:       upper left corner in Y
      39,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "St^and-alone Button #8",     // label:     label text
      ZERO,                         // labY:      label offset
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[9]                        // nextCtrl:  link in next structure
   },
   {  //* Stand-alone Radio Button 9 - - - - - - - - - - - - -   T6standAlone9 *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      true,                         // rbSelect:  initial selection state
      8,                            // ulY:       upper left corner in Y
      39,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Stan^d-alone Button #9",     // label:     label text
      ZERO,                         // labY:      label offset
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[10]                       // nextCtrl:  link in next structure
   },
   {  //* Active Text Box  - - - - - - - - - - - - - - - - - - - -    T6tbText *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      9,                            // ulY:       upper left corner in Y
      11,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      8,                            // cols:      control columns
      NULL,                         // dispText:  (initially blank)
      nc.bw,                        // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Text Bo^x",                  // label:     label text
      ZERO,                         // labY:      label ofset
      -9,                           // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[11],                      // nextCtrl:  link in next structure
   },
   {  //* Non-active Text Box holds Group 'A' selection  - - - - -    T6utBOX1 *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      7,                            // ulY:       upper left corner in Y
      21,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      4,                            // cols:      control columns
      NULL,                         // dispText:  (initialized by callback method)
      nc.grG,                       // nColor:    non-focus color
      nc.grG,                       // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Group A Selection",          // label:     label text
      ZERO,                         // labY:      label ofset
      -19,                          // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    allow control to gain focus
      &ic[12],                      // nextCtrl:  link in next structure
   },
   {  //* Non-active Text Box holds Group 'B' selection  - - - - -    T6utBOX2 *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      8,                            // ulY:       upper left corner in Y
      21,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      4,                            // cols:      control columns
      NULL,                         // dispText:  (initialized by callback method)
      nc.grB,                       // nColor:    non-focus color
      nc.grB,                       // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Group B Selection",          // label:     label text
      ZERO,                         // labY:      label ofset
      -19,                          // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    allow control to gain focus
      &ic[13],                      // nextCtrl:  link in next structure
   },
   {  //* Non-active Text Box holds Button '8' state   - - - - - -    T6utBOX3 *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      7,                            // ulY:       upper left corner in Y
      58,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      7,                            // cols:      control columns
      NULL,                         // dispText:  (initialized by callback method)
      nc.grG,                       // nColor:    non-focus color
      nc.grG,                       // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Button 8 State",             // label:     label text
      ZERO,                         // labY:      label ofset
      -15,                          // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    allow control to gain focus
      &ic[14],                      // nextCtrl:  link in next structure
   },
   {  //* Non-active Text Box holds Button '9' state   - - - - - -    T6utBOX4 *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      9,                            // ulY:       upper left corner in Y
      58,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      7,                            // cols:      control columns
      NULL,                         // dispText:  (initialized by callback method)
      nc.grG,                       // nColor:    non-focus color
      nc.grG,                       // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Button 9 State",             // label:     label text
      ZERO,                         // labY:      label ofset
      -15,                          // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   },
} ;


   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       "  Radio-Button Grouping Test  ", // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;
   Test06Dialog = dp ;              // give callback method access to the dialog

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      dp->DrawBox ( 1,  2, 6, 31, dColor, " Group A " ) ; // group 1 box
      dp->DrawBox ( 1, 37, 5, 31, dColor, " Group B " ) ; // group 2 box
      dp->RefreshWin () ;

      //* Set radio buttons 1, 2, 3, and 4 as an exclusive-OR group.*
      //* Set radio buttons 5, 6, and 7 as an exclusive-OR group.   *
      //* (only one from each group can be selected at any moment). *
      short XorGroup1[5] = { T6groupA_1, T6groupA_2, T6groupA_3, T6groupA_4, -1 } ;
      dp->GroupRadiobuttons ( XorGroup1 ) ;
      short XorGroup2[4] = { T6groupB_1, T6groupB_2, T6groupB_3, -1 } ;
      dp->GroupRadiobuttons ( XorGroup2 ) ;

      //* Establish a call-back method that can be called from within the *
      //* NcDialog code. Called each time through a control's user-input  *
      //* loop so we can manually modify the controls when necessary.     *
      dp->EstablishCallback ( &T06_ControlUpdate ) ;

      //* Allow user to play with the controls.      *
      //* (Please see the notes in NcDialog.hpp on   *
      //* values returned (from the editing routines)*
      uiInfo Info ;                 // user interface data returned here
      short  icIndex = ZERO ;       // index of control with input focus
      bool   done = false ;         // loop control
      while ( ! done )
      {
         //*******************************************
         //* If focus is currently on a Pushbutton   *
         //*******************************************
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            //* Get user input for this control                            *
            //* Returns when the Pushbutton is pressed OR when the user    *
            //* requests that the input focus moves to another control.    *
            if ( Info.viaHotkey == false )
            {
               icIndex = dp->EditPushbutton ( Info ) ;
            }
            //* If user selected the current control object via hotkey, we *
            //* don't need to call the input routine; but simply process   *
            //* the existing data.                                         *
            else
            {
               //* Information arrived in the Info.h_xxx fields.*
               //* Transfer it to the primary positions.        *
               //* Previous data in these fields has already    *
               //* been handled in prior loop iteration.        *
               Info.HotData2Primary () ;
            }

            //* If a Pushbutton was pressed *
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == T6donePush ) // request to exit the dialog
                  done = true ;
               else
               { /* no other Pushbuttons defined for this dialog */ }
            }
            else
            {
               // No button press, so nothing to do
            }
         }

         //*******************************************
         //* If focus is currently on a Text Box     *
         //*******************************************
         else if ( ic[icIndex].type == dctTEXTBOX )
         {
            //* It doesn't matter how this control gained the input    *
            //* focus, by Tab/ShiftTab or via hotkey. If by hotkey, no *
            //* processing was done anyway, so discard any hotkey data *
            Info.viaHotkey = false ;

            //* Allow user to modify the text data in the text box.    *
            //* Returns when edit is complete.                         *
            icIndex = dp->EditTextbox ( Info ) ;

            //* If the data in the text box being edited has changed *
            if ( Info.dataMod != false )
            {
               //* Do any processing related to a change in the text box data *
               if ( Info.ctrlIndex == T6tbText )
               {
                  // DO STUFF
               }
               else
                  { /* no other text boxes defined in this dialog */ }
            }
         }

         //*******************************************
         //* If focus is currently on a Radio Button *
         //*******************************************
         else if ( ic[icIndex].type == dctRADIOBUTTON )
         {
            //* Get user input for this control                            *
            //* Returns when radio-button control or radio-button group is *
            //* about to lose the input focus.                             *
            //* A selection may or may not have been made.                 *
            if ( Info.viaHotkey == false )
            {
               icIndex = dp->EditRadiobutton ( Info ) ;
            }
            //* If user selected the current control object via hotkey, we *
            //* don't need to call the input routine; but simply process   *
            //* the existing data.                                         *
            //* For radio buttons, display update will already have been   *
            //* done, so we can just move the focus to the next control.   *
            else
            {
               //* Information arrived in the Info.h_xxx fields.*
               //* Transfer it to the primary positions.        *
               //* Previous data in these fields has already    *
               //* been handled in prior loop iteration.        *
               Info.HotData2Primary () ;
            }
            
            //* If 'selected' button is a member of a button group *
            if ( Info.selMember < MAX_DIALOG_CONTROLS )
            {
               //* If a new 'selected' member of the group *
               //* OR previous selection was re-selected   *
               if ( Info.dataMod != false || Info.keyIn == nckENTER )
               {
                  //* Process the user's selection.             *
                  //* There are two Radio Button groups defined *
                  switch ( Info.selMember )
                  {
                     case T6groupA_1:
                     case T6groupA_2:
                     case T6groupA_3:
                     case T6groupA_4:
                        // DO STUFF
                        break ;
                     case T6groupB_1:
                     case T6groupB_2:
                     case T6groupB_3:
                        // DO STUFF
                        break ;
                     default:
                        // no other radio buttons associated with a 
                        // button group have been defined for this dialog
                        break ;
                  }
               }
            }
            //* Else, button is an independent button *
            //* (Info.selMember==MAX_DIALOG_CONTROLS) *
            else
            {
               //* If the selected/deselected state of an * 
               //* independent radio button has changed,  *
               if ( Info.dataMod != false )
               {
                  // NOTE: all the interesting action for this test dialog 
                  //       is happening in the callback method.
                  switch ( Info.ctrlIndex )
                  {
                     case T6standAlone8:
                        // DO STUFF
                        break ;
                     case T6standAlone9:
                        // DO STUFF
                        break ;
                     default:
                        // no other independent radio buttons defined for this dialog
                        break ;
                  }
               }
            }
         }

         //* If user exited the control edit method via a hotkey,*
         //* then the new control already has focus. Otherwise,  *
         //* move input focus to next/previous control.          *
         if ( done == false && Info.viaHotkey == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }  // while(!done)
   }        // if((dp->OpenWindow())==OK)
   else
   {  //* Most likely cause of dialog not opening is that the terminal window  *
      //* is too small. Give user a clue...                                    *
      short neededLines = (dialogROWS + minTestULY) >= minTERMROWS ?
                          (dialogROWS + minTestULY) : minTERMROWS, 
            neededCols  = dialogCOLS >= minTERMCOLS ? dialogCOLS : minTERMCOLS ;
      gString gs ;
      gs.compose ( L" Sorry, size of terminal window must be at least %hd x %hd for this test. ", 
                   &neededCols, &neededLines ) ;
      StatWin ( gs.gstr(), nc.cyG ) ;
      sleep ( 5 ) ;
   }

   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

}  //* End Test06() *

//*************************
//*  T06_ControlUpdate    *
//*************************
//******************************************************************************
//* This is a callback method for manually updating the controls in the        *
//* Test06() dialog.                                                           *
//*                                                                            *
//*  For this test, the following are updated by the callback method:          *
//*   1. indicate the 'selected' member of radio button group A                *
//*   2. indicate the 'selected' member of radio button group B                *
//*   3. indicate the state of radion buttons 8 and 9                          *
//*                                                                            *
//* Input  : currIndex: index of control that currently has focus              *
//*          wkey     : user's key input data                                  *
//*          firstTime: the EstablishCallback() method calls this method once  *
//*                     with firstTime==true, to perform any required          *
//*                     initialization. Subsequently, the NcDialog class       *
//*                     always calls with firstTime==false.                    *
//* Returns: OK                                                                *
//******************************************************************************
//* Important Note: This method makes some intimate assumptions about what     *
//* is happening in the method that established the callback. If changes are   *
//* made in the establishing method that affect the callback functionality,    *
//* be sure to update this method to address those changes.                    *
//*                                                                            *
//* Programmer's Note: See dialog control index constants at file scope above. *
//******************************************************************************

short T06_ControlUpdate ( const short currIndex, const wkeyCode wkey, bool firstTime )
{
const char     tbText[][8] = { { " reset " }, { "  set  " } } ;

               // initial values will force a display update on first call
static short   oldRBG_A = -1, oldRBG_B = -1 ;
static bool    oldRB8, oldRB9 ;
short          gIndex ;
bool           bState ;

   //*************************
   //* First-time processing *
   //*************************
   if ( firstTime != false )
   {
      //** Initialize the tracking data **
      //* Get state of Radio Button 8 and indicate the state in Text Box 3 *
      Test06Dialog->GetRadiobuttonState ( T6standAlone8, oldRB8 ) ;
      Test06Dialog->SetTextboxText ( T6utBOX3, (oldRB8 ? 
                                        (char*)tbText[1] : (char*)tbText[0]) ) ;

      //* Get state of Radio Button 9 and indicate the state in Text Box 4 *
      Test06Dialog->GetRadiobuttonState ( T6standAlone9, oldRB9 ) ;
      Test06Dialog->SetTextboxText ( T6utBOX4, (oldRB9 ? 
                                        (char*)tbText[1] : (char*)tbText[0]) ) ;
   }

   //***********************
   //* Standard processing *
   //***********************
   //* Get number (index) of 'selected' Radio Button in Group A and *
   //* indicate it in Text Box 1.                                   *
   gString  oBuff ;
   gIndex = Test06Dialog->GetRbGroupSelection ( T6groupA_1 ) ;
   if ( gIndex != oldRBG_A )
   {
      oldRBG_A = gIndex ;
      oBuff.compose( L" %02hd ", &gIndex ) ;
      Test06Dialog->SetTextboxText ( T6utBOX1, oBuff ) ;
   }

   //* Get number (index) of 'selected' Radio Button in Group B and *
   //* indicate it in Text Box 2.                                   *
   gIndex = Test06Dialog->GetRbGroupSelection ( T6groupB_1 ) ;
   if ( gIndex != oldRBG_B )
   {
      oBuff.compose( L" %02hd ", &gIndex ) ;
      Test06Dialog->SetTextboxText ( T6utBOX2, oBuff ) ;
   }

   //* Get state of Radio Button 8 and indicate the state in Text Box 3 *
   Test06Dialog->GetRadiobuttonState ( T6standAlone8, bState ) ;
   if ( bState != oldRB8 )
   {
      Test06Dialog->SetTextboxText ( T6utBOX3, (bState ? 
                                        (char*)tbText[1] : (char*)tbText[0]) ) ;
      oldRB8 = bState ;
   }

   //* Get state of Radio Button 9 and indicate the state in Text Box 4 *
   Test06Dialog->GetRadiobuttonState ( T6standAlone9, bState ) ;
   if ( bState != oldRB9 )
   {
      Test06Dialog->SetTextboxText ( T6utBOX4, (bState ? 
                                        (char*)tbText[1] : (char*)tbText[0]) ) ;
      oldRB9 = bState ;
   }
   return OK ;

}  //* End T06_ControlUpdate() *

//*************************
//*       Test07          *
//*************************
//******************************************************************************
//* Test the hotkey selection of radio-buttons (group and independent) and     *
//* pushbuttons.                                                               *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note: See dialog control index constants at file scope above. *
//******************************************************************************

void Test07 ( void )
{
static const short dialogROWS = 21 ;         // display lines
static const short dialogCOLS = 74 ;         // display columns
static const short controlsDEFINED = 11 ;    // # of controls in dialog
const short scrITEMS = 3, scrITEM_LEN = 17 ;
static const char scrData[scrITEMS][scrITEM_LEN] = 
{
   " Sprinkles      ",
   " Pistachios     ",
   " Jelly Beans    "
} ;
const short INFO_LINES = 10 ;
gString Info(
   "Hotkeys, also known as shortcut keys, are used to reduce the time and\n" 
   "the number of keystrokes required to access often-used operations.\n" 
   "           The NcDialog class supports hotkey access for all\n" 
   "                  ACTIVE (user-accessible) controls.\n" 
   "\n" 
   "When initializing the control, the (optional) hotkey is specified\n" 
   "by placing the caret character ('^') in the control's label string\n"
   "before the desired hotkey character ['A'-'Z', 'a'-'z']\n"
   "            Examples:  \"   ^OK   \"  or  \"Protect the ^World\"\n" 
   "Note that 'O', 'o' and 'CTRL+o' are all recognized as the same hotkey." 
) ;
// 'OK' button display text. Note that this is a test of 
// multi-row pushbuttons and multi-line text with hotkey.
#define BORDERED_OK (1)
#if BORDERED_OK != 0    // Text for Bordered control
gString OkText = "  ^OK  " ;
#else                   // Text for un-Bordered control
gString OkText( "%------%\n%  ^OK  %\n%------%" ) ;
#endif   // BORDERED_OK
short    ctrY = termRows/2,               // terminal window center in Y
         ctrX = termCols/2,               // terminal window center in X
         ulY  = ctrY - dialogROWS / 2,    // upper left corner in Y
         ulX  = ctrX - dialogCOLS / 2 ;   // upper left corner in X
char     tboxBuff[30] ;                   // for constructing messages
attr_t   dColor = nc.blR ;                // dialog window color attribute
attr_t   monoColor[2] = { attrDFLT, nc.cy } ; // color for interior of scroll box

   StatWin ( "Opening Test Dialog #7 - Hotkey Selection Test" ) ;

InitCtrl ic[controlsDEFINED] =   // array of dialog control initialization objects
{
   {  //* Pushbutton  - - - - - - - - - - - - - - - - - - - - - - - T7okBUTTON *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 4),        // ulY:       upper left corner in Y
      short(dialogCOLS / 2 - 7),    // ulX:       upper left corner in X
      3,                            // lines:     control lines
      8,                            // cols:      control columns
      OkText.ustr(),                // dispText:  display text
      nc.gyR,                       // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T7dealBOX]                // nextCtrl:  link in next structure
   },
   {  //* Text Box - - - - - - - - - - - - - - - - - - - - - - - -   T7dealBOX *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(INFO_LINES + 2),        // ulY:       upper left corner in Y
      14,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      25,                           // cols:      control columns
      "Total Cost: $39.95 + tax",   // dispText:  initial display text
      nc.bw,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Such a ^deal ",              // label:     label text
      ZERO,                         // labY:      label offset
      -12,                          // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T7worldBUTTON]            // nextCtrl:  link in next structure
   },
   {  //* Radio Button Group A,   - - - - - - - - - - - - - - -  T7worldBUTTON *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      true,                         // rbSelect:  default selection
      short(INFO_LINES + 5),        // ulY:       upper left corner in Y
      4,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a for standard types)
      nc.bw,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Protect the ^World",         // label:     label text
      ZERO,                         // labY:      label offset
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T7laundryBUTTON]          // nextCtrl:  link in next structure
   },
   {  //* Radio Button Group A,   - - - - - - - - - - - - - -  T7laundryBUTTON *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(INFO_LINES + 6),        // ulY:       upper left corner in Y
      4,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a for standard types)
      nc.bw,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Do smelly ^laundry",         // label:     label text
      ZERO,                         // labY:      label offset
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T7pizzaBUTTON]            // nextCtrl:  link in next structure
   },
   {  //* Radio Button Group A,   - - - - - - - - - - - - - - -  T7pizzaBUTTON *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(INFO_LINES + 7),        // ulY:       upper left corner in Y
      4,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a for standard types)
      nc.bw,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Reheat the ^pizza",          // label:     label text
      ZERO,                         // labY:      label offset
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T7flavorBOX]              // nextCtrl:  link in next structure
   },
   {  //* Text Box (inactive)  - - - - - - - - - - - - - - - - -   T7flavorBOX *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(INFO_LINES + 3),        // ulY:       upper left corner in Y
      41,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      12,                           // cols:      control columns
      NULL,                         // dispText:  initial display text
      nc.grG,                       // nColor:    non-focus color
      nc.grG,                       // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Flavour",                    // label:     label text
      -1,                           // labY:      label offset
      2,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    allow control to gain focus
      &ic[T7toppingsTBOX]           // nextCtrl:  link in next structure
   },
   {  //* Text Box (inactive)  - - - - - - - - - - - - - - - -  T7toppingsTBOX *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(INFO_LINES + 3),        // ulY:       upper left corner in Y
      58,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      14,                           // cols:      control columns
      NULL,                         // dispText:  initial display text
      nc.grG,                       // nColor:    non-focus color
      nc.grG,                       // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Topping",                    // label:     label text
      -1,                           // labY:      label offset
      3,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    allow control to gain focus
      &ic[T7strawberryBUTTON]       // nextCtrl:  link in next structure
   },
   {  //* Pushbutton  - - - - - - - - - - - - - - - - - - - T7strawberryBUTTON *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(INFO_LINES + 5),        // ulY:       upper left corner in Y
      41,                           // ulX:       upper left corner in X
      1,                            // lines:     control lines
      12,                           // cols:      control columns
      " ^Strawberry ",              // dispText:  display text
      nc.gyR,                       // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T7chocolateBUTTON]        // nextCtrl:  link in next structure
   },
   {  //* Pushbutton  - - - - - - - - - - - - - - - - - - -  T7chocolateBUTTON *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(INFO_LINES + 7),        // ulY:       upper left corner in Y
      41,                           // ulX:       upper left corner in X
      1,                            // lines:     control lines
      11,                           // cols:      control columns
      " ^Chocolate ",               // dispText:  display text
      nc.gyR,                       // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T7vanillaBUTTON]          // nextCtrl:  link in next structure
   },
   {  //* Pushbutton  - - - - - - - - - - - - - - - - - - - -  T7vanillaBUTTON *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(INFO_LINES + 9),        // ulY:       upper left corner in Y
      41,                           // ulX:       upper left corner in X
      1,                            // lines:     control lines
      9,                            // cols:      control columns
      " ^Vanilla ",                 // dispText:  display text
      nc.gyR,                       // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T7toppingsSBOX]           // nextCtrl:  link in next structure
   },
   { //* Scrollbox  - - - - - - - - - - - - - - - - - - - - -   T7toppingsSBOX *
      dctSCROLLBOX,                 // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(INFO_LINES + 5),        // ulY:       upper left corner in Y
      54,                           // ulX:       upper left corner in X
      5,                            // lines:     control lines
      18,                           // cols:      control columns
      (const char*)&scrData,        // dispText:  text-data array
      nc.gyR,                       // nColor:    non-focus border color
      dColor,                       // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "^Topping Select",            // label:     label text
      #if 1       // set to 1 to draw label in 'title' position
      ZERO,                         // labY:      label offset
      ZERO,                         // labX
      #else       // else draw as a 'normal' label
      -1,                           // labY:      label offset
      2,                            // labX
      #endif      // label position
      ddBoxTYPES,                   // exType:    (n/a)
      scrITEMS,                     // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    index of initial highlighted element
      monoColor,                    // scrColor:  color-attribute list
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   },
} ;


   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       "  Selecting Controls via Hotkey  ", // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;
   Test07Dialog = dp ;              // give callback method access to the dialog

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      LineDef  lDef(ncltHORIZ, ncltSINGLE, INFO_LINES+1, ZERO, dialogCOLS, dColor) ;
      dp->DrawLine ( lDef ) ;
      dp->WriteParagraph ( 1, 2, Info, dColor ) ;
      dp->DrawBox ( INFO_LINES+4, 2, 5, 26, dColor, " Radio Button Group " ) ;

      #if BORDERED_OK != 0    // Optionally enable Bordered 'OK' button
      dp->EnablePushbuttonBorder ( T7okBUTTON ) ;
      #endif   // BORDERED_OK

      dp->RefreshWin () ;     // make everything visible

      //* Create a Radio Button group using indices of specified buttons *
      short XorGroup1[4] = { T7worldBUTTON, T7laundryBUTTON, T7pizzaBUTTON, -1 } ;
      dp->GroupRadiobuttons ( XorGroup1 ) ;

      //* Establish a call-back method that can be called from within the *
      //* NcDialog code. Called each time through a control's user-input  *
      //* loop so we can manually modify the controls when necessary.     *
      dp->EstablishCallback ( &T07_ControlUpdate ) ;

      //* Allow user to play with the controls.      *
      //* (Please see the notes in NcDialog.hpp on   *
      //* values returned (from the editing routines)*
      uiInfo Info ;                 // user interface data returned here
      short  icIndex = ZERO ;       // index of control with input focus
      bool   done = false ;         // loop control
      while ( ! done )
      {
         //*******************************************
         //* If focus is currently on a Pushbutton   *
         //*******************************************
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            //* Get user input for this control                            *
            //* Returns when the Pushbutton is pressed OR when the user    *
            //* requests that the input focus moves to another control.    *
            if ( Info.viaHotkey == false )
            {
               icIndex = dp->EditPushbutton ( Info ) ;
            }
            //* If user selected the current control object via hotkey, we *
            //* don't need to call the input routine; but simply process   *
            //* the existing data.                                         *
            else
            {
               //* Information arrived in the Info.h_xxx fields.*
               //* Transfer it to the primary positions.        *
               //* Previous data in these fields has already    *
               //* been handled in prior loop iteration.        *
               Info.HotData2Primary () ;
            }

            //* If a Pushbutton was pressed *
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == T7okBUTTON )
               {
                  done = true ;           // 'OK' - request to exit the dialog
               }
               else
               {
                  switch ( Info.ctrlIndex )
                  {
                     case T7strawberryBUTTON:   // strawberry flavour selector
                        strncpy ( tboxBuff, " Strawberry ", 30 ) ;
                        break ;
                     case T7chocolateBUTTON:    // chocolate flavour selector
                        strncpy ( tboxBuff, " Chocolate  ", 30 ) ;
                        break ;
                     case T7vanillaBUTTON:      // vanilla flavour selector
                        strncpy ( tboxBuff, "  Vanilla   ", 30 ) ;
                        break ;
                     default:
                        // no other Pushbuttons defined for this dialog
                        break ;
                  }
                  //* Update the inactive (non-selectable) text box *
                  //* that displays the selected ice cream flavor.  *
                  dp->SetTextboxText ( T7flavorBOX, tboxBuff ) ;
               }
            }
            else
            {
               // No button press, so nothing to do
            }
         }

         //*******************************************
         //* If focus is currently on a Text Box     *
         //*******************************************
         else if ( ic[icIndex].type == dctTEXTBOX )
         {
            //* It doesn't matter how this control gained the input    *
            //* focus, by Tab/ShiftTab or via hotkey. If by hotkey, no *
            //* processing was done anyway, so discard any hotkey data *
            Info.viaHotkey = false ;

            //* Allow user to modify the text data in the text box.    *
            //* Returns when edit is complete.                         *
            icIndex = dp->EditTextbox ( Info ) ;

            //* If the data in the text box being edited has changed *
            if ( Info.dataMod != false )
            {
               //* Do any processing related to a change in the text box data *
               if ( Info.ctrlIndex == T7dealBOX )
               {
                  // DO STUFF
               }
               else
               { /* no other ACTIVE text boxes defined in this dialog */ }
            }
         }

         //*******************************************
         //* If focus is currently on a Scroll Box   *
         //*******************************************
         else if ( ic[icIndex].type == dctSCROLLBOX )
         {
            //* It doesn't matter how this control gained the input    *
            //* focus, by Tab/ShiftTab or via hotkey. If by hotkey, no *
            //* processing was done anyway, so discard any hotkey data *
            Info.viaHotkey = false ;

            //* Allow user to 'select' a member of the scroll box *
            //* Returns when edit is complete.                    *
            icIndex = dp->EditScrollbox ( Info ) ;

            //* If 'selected' member of Scroll Box has changed *
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == T7toppingsSBOX )
               {
                  //* NOTE: The necessary processing for this Scroll Box 
                  //*       selection could have been done here; however, in
                  //*       this case, it is done in the callback method i.e.
                  //*       the Scroll Box selection has been reflected in 
                  //*       the INACTIVE 'Toppings' text box.
               }
               else
               { /* no other scroll boxes defined for this dialog */ }
            }
            else if ( Info.keyIn == nckENTER )
            {
               //* NOTE: If previous member was re-selected, you could 
               //*       also respond to that.
            }
         }

         //*******************************************
         //* If focus is currently on a Radio Button *
         //*******************************************
         else if ( ic[icIndex].type == dctRADIOBUTTON )
         {
            //* Get user input for this control                            *
            //* Returns when radio-button control or radio-button group is *
            //* about to lose the input focus.                             *
            //* A selection may or may not have been made.                 *
            if ( Info.viaHotkey == false )
            {
               icIndex = dp->EditRadiobutton ( Info ) ;
            }
            //* If user selected the current control object via hotkey, we *
            //* don't need to call the input routine; but simply process   *
            //* the existing data.                                         *
            //* For radio buttons, display update will already have been   *
            //* done, so we can just move the focus to the next control.   *
            else
            {
               //* Information arrived in the Info.h_xxx fields.*
               //* Transfer it to the primary positions.        *
               //* Previous data in these fields has already    *
               //* been handled in prior loop iteration.        *
               Info.HotData2Primary () ;
            }
            
            //* If 'selected' button is a member of a button group *
            if ( Info.selMember < MAX_DIALOG_CONTROLS )
            {
               //* If a new 'selected' member of the group *
               //* OR previous selection was re-selected   *
               if ( Info.dataMod != false || Info.keyIn == nckENTER )
               {
                  //* Process the user's selection *
                  switch ( Info.selMember )
                  {
                     case T7worldBUTTON:
                        // DO STUFF
                        break ;
                     case T7laundryBUTTON:
                        // DO STUFF
                        break ;
                     case T7pizzaBUTTON:
                        // DO STUFF
                        break ;
                     default:
                        // No other Radio Buttons associated with a button 
                        // group have been defined for this dialog
                        break ;
                  }
               }
            }
            //* Else, button is an independent button *
            //* (Info.selMember==MAX_DIALOG_CONTROLS) *
            else
            {
               // There are no independent Radio Buttons defined for this dialog
            }
         }  // radio button

         //* If user exited the control edit method via a hotkey,*
         //* then the new control already has focus. Otherwise,  *
         //* move input focus to next/previous control.          *
         if ( done == false && Info.viaHotkey == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }     // while()
   }        // if((dp->OpenWindow())==OK)
   else
   {  //* Most likely cause of dialog not opening is that the terminal window  *
      //* is too small. Give user a clue...                                    *
      short neededLines = (dialogROWS + minTestULY) >= minTERMROWS ?
                          (dialogROWS + minTestULY) : minTERMROWS, 
            neededCols  = dialogCOLS >= minTERMCOLS ? dialogCOLS : minTERMCOLS ;
      gString gs ;
      gs.compose ( L" Sorry, size of terminal window must be at least %hd x %hd for this test. ", 
                   &neededCols, &neededLines ) ;
      StatWin ( gs.gstr(), nc.cyG ) ;
      sleep ( 5 ) ;
   }

   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

}  //* End Test07() *

//*************************
//*  T07_ControlUpdate    *
//*************************
//******************************************************************************
//* This is a callback method for manually updating the controls in the        *
//* Test07() dialog.                                                           *
//*                                                                            *
//*  For this test, the following are updated by the callback method:          *
//*   1. indicate the 'selected' member of icecream-toppings scroll box        *
//*   2.                                                                       *
//*   3.                                                                       *
//*                                                                            *
//* Input  : currIndex: index of control that currently has focus              *
//*          wkey     : user's key input data                                  *
//*          firstTime: the EstablishCallback() method calls this method once  *
//*                     with firstTime==true, to perform any required          *
//*                     initialization. Subsequently, the NcDialog class       *
//*                     always calls with firstTime==false.                    *
//* Returns: OK                                                                *
//******************************************************************************
//* Important Note: This method makes some intimate assumptions about what     *
//* is happening in the method that established the callback. If changes are   *
//* made in the establishing method that affect the callback functionality,    *
//* be sure to update this method to address those changes.                    *
//*                                                                            *
//* Programmer's Note: See dialog control index constants at file scope above. *
//******************************************************************************

short T07_ControlUpdate ( const short currIndex, const wkeyCode wkey, bool firstTime )
{
static short   oldIndex ;
short          newIndex ;
char           tboxBuff[30] ;                // for constructing messages

   //*************************
   //* First-time processing *
   //*************************
   if ( firstTime != false )
   {
      //* Initialize the tracking data *
      oldIndex = Test07Dialog->GetScrollboxSelect ( T7toppingsSBOX ) ;
   }

   //***********************
   //* Standard processing *
   //***********************
   //* Get index of currently-highlighted item *
   newIndex = Test07Dialog->GetScrollboxSelect ( T7toppingsSBOX ) ;

   //* Update non-active text box indicating the current setting of the scroll box *
   if ( newIndex != oldIndex || firstTime != false )
   {
      switch ( newIndex )
      {
         case 0:                    // 'sprinkles' selected
            strncpy ( tboxBuff, " Sprinkles  ", 30 ) ;
            break ;
         case 1:                    // 'pistachios' selected
            strncpy ( tboxBuff, " Pistachios ", 30 ) ;
            break ;
         case 2:                    // 'Jelly Beans' selected
            strncpy ( tboxBuff, " Jelly Beans ", 30 ) ;
            break ;
      }
      oldIndex = newIndex ;
      Test07Dialog->SetTextboxText ( T7toppingsTBOX, tboxBuff ) ;
   }
   return OK ;

}  //* End T07_ControlUpdate() *

//*************************
//*       Test08          *
//*************************
//******************************************************************************
//* Test the NcDialog methods related to capture and restore of dialog         *
//* window display data.                                                       *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note:                                                         *
//* Hotkeys are a great benefit, but can cause havoc. If you do not define     *
//* any controls with hotkeys, the application's input loop is quite simple:   *
//* you just call the edit routine for the type of control that currently has  *
//* the input focus.                                                           *
//*                                                                            *
//* If you DO define hotkeys for accessing controls, your input loop becomes   *
//* more complex. Look carefully at the input loop in this example.            *
//*                                                                            *
//* For further clarification, please see the notes in NcDialog.hpp, or study  *
//* the NcDialog source code, specifically, the private method,                *
//* NcDialog::ChangedFocusViaHotkey().                                         *
//*                                                                            *
//*                                                                            *
//******************************************************************************

void Test08 ( void )
{
const short dialogROWS = 30 ;             // display lines
const short dialogCOLS = 62 ;             // display columns
const short controlsDEFINED = 19 ;        // # of controls in dialog
const short sDATA_ITEMS = 9 ;             // elements in SB string and color arrays
const short sCOL_WIDTH = 21 ;             // width of control in display columns
attr_t monoColor[2] = { attrDFLT, nc.gr } ; // scroll box data colors
const short minNeededLines = 36,          // min hight for ACS window display
            minNeededCols  = 90 ;         // min width for ACS window display
attr_t dColor = nc.blR,                   // text color for writing to dialog
       bColor = nc.brbl | ncbATTR ;       // bold text color
short ctrY = termRows/2,                  // terminal window center in Y
      ctrX = termCols/2,                  // terminal window center in X
      ulY  = ctrY - dialogROWS / 2,       // upper left corner in Y
      ulX  = ctrX - dialogCOLS / 2 ;      // upper left corner in X

//* Display strings for the T8dumpPush control *
const char dumpitStrings[][20] = { "  Enable UI D^ump  ", " Disable UI D^ump  " } ;

//* Setup data for the T8spinnerControl *
const dspinData dsData( -999, 999, 0, dspinINTEGER, nc.brR ) ;

//* Setup data for the T8sliderControl *
sliderData slData( -30.0, 5.0, -10.0, true, false ) ;

const short sboxItemCircus = 7, sboxItemCustom = 8 ;
//(keep these two array synchronized)
static char sboxData[sDATA_ITEMS][sCOL_WIDTH] =   
{  // mixed-length data strings will be normalized during instantiation
   "Upper Left",
   "Lower Left",
   "Upper Right",
   "Lower Right",
   "Centered",
   "Off-center Left",
   "Off-center Right",
   "Traveling Circus",
   "Custom YX Position"
} ;
//* Data to populate the dctDROPDOWN and dctSCROLLEXT controls *
const short scDATA_ITEMS = 32, scITEM_LEN = 15 ; // elements in Dropdown and Scrollext
static const char ddboxData[scDATA_ITEMS][scITEM_LEN] =   
{
   "00AAAAAAAAAA00",  "01BBBBBBBBBB01",  "02CCCCCCCCCC02",  "03DDDDDDDDDD03",
   "04EEEEEEEEEE04",  "05FFFFFFFFFF05",  "06GGGGGGGGGG06",  "07HHHHHHHHHH07",
   "08IIIIIIIIII08",  "09JJJJJJJJJJ09",  "10KKKKKKKKKK10",  "11LLLLLLLLLL11",
   "12MMMMMMMMMM12",  "13NNNNNNNNNN13",  "14OOOOOOOOOO14",  "15PPPPPPPPPP15",
   "16QQQQQQQQQQ16",  "17RRRRRRRRRR17",  "18SSSSSSSSSS18",  "19TTTTTTTTTT19",
   "20UUUUUUUUUU20",  "21VVVVVVVVVV21",  "22WWWWWWWWWW22",  "23XXXXXXXXXX23",
   "24YYYYYYYYYY24",  "25ZZZZZZZZZZ25",  "26   THIS   26",  "27    IS    27", 
   "28   THE    28",  "29UPPER-CASE29",  "30  ASCII   30",  "31 ALPHABET 31", 
} ;
static const char* seboxData[scDATA_ITEMS] =   
{
   "00AAAAAAAAAA00",  "01BBBBBBBBBB01",  "02CCCCCCCCCC02",  "03DDDDDDDDDD03",
   "04EEEEEEEEEE04",  "05FFFFFFFFFF05",  "06GGGGGGGGGG06",  "07HHHHHHHHHH07",
   "08IIIIIIIIII08",  "09JJJJJJJJJJ09",  "10KKKKKKKKKK10",  "11LLLLLLLLLL11",
   "12MMMMMMMMMM12",  "13NNNNNNNNNN13",  "14OOOOOOOOOO14",  "15PPPPPPPPPP15",
   "16QQQQQQQQQQ16",  "17RRRRRRRRRR17",  "18SSSSSSSSSS18",  "19TTTTTTTTTT19",
   "20UUUUUUUUUU20",  "21VVVVVVVVVV21",  "22WWWWWWWWWW22",  "23XXXXXXXXXX23",
   "24YYYYYYYYYY24",  "25ZZZZZZZZZZ25",  "26   THIS   26",  "27    IS    27", 
   "28   THE    28",  "29UPPER-CASE29",  "30  ASCII   30",  "31 ALPHABET 31", 
} ;
static const attr_t seboxColors[scDATA_ITEMS] = 
{
   nc.gr, nc.gr, nc.gr, nc.gr, nc.gr, nc.gr, nc.gr, nc.gr, nc.gr, nc.gr, 
   nc.gr, nc.gr, nc.gr, nc.gr, nc.gr, nc.gr, nc.gr, nc.gr, nc.gr, nc.gr, 
   nc.gr, nc.gr, nc.gr, nc.gr, nc.gr, nc.gr, nc.gr, nc.gr, nc.gr, nc.gr,
   nc.gr, nc.gr, 
} ;

//* Data to populate Menu Win 'Y' *
const short mw01DATA_ITEMS = 8,  // menu items in array
            mw01ITEM_LEN = 17 ;  // display columns required for menu item
            // note compensation for hotkey indicators
static const char mw01Data[mw01DATA_ITEMS][mw01ITEM_LEN+2] =   
{  // data with hotkeys
   "^First Item       ",
   "^Second Item     >",
   "^Third Item       ",
   "^Fourth Item      ",
   "F^ifth Item       ",
   "Si^xth Item       ",
   "Se^venth Item     ",
   "^Eighth Item      ",
} ;
const short mw02DATA_ITEMS = 8,  // menu items in array
            mw02ITEM_LEN = 17 ;  // display columns required for menu item
            // note compensation for hotkey indicators
static const char mw02Data[mw02DATA_ITEMS][mw02ITEM_LEN+2] =   
{  // data with hotkeys
   "^Ninth Item       ",
   "^Tenth Item       ",
   "^Eleventh Item    ",
   "T^welfth Item    >",
   "T^hirteenth Item  ",
   "^Fourteenth Item  ",
   "F^ifteenth Item   ",
   "^Sixteenth Item   ",
} ;

const short mw03DATA_ITEMS = 4,  // menu items in array
            mw03ITEM_LEN = 12 ;  // display columns required for menu item
            // note compensation for multi-byte characters and hotkey indicator
static const char mw03Data[mw03DATA_ITEMS][mw03ITEM_LEN+2] =   
{  // data with hotkeys
   " ^ILMT       ",
   " I^LHE       ",
   " IL^HM       ",
   " IHS^WMMOD   ",
} ;

const short mw04DATA_ITEMS = 4,  // menu items in array
            mw04ITEM_LEN = 12 ;  // display columns required for menu item
            // note compensation for multi-byte characters (no hotkey indicator)
static const char mw04Data[mw04DATA_ITEMS][mw04ITEM_LEN+1] =   
{  // data with hotkeys
   " ILMT       ",
   " ILHE       ",
   " ILHM       ",
   " IHSWMMOD   ",
} ;

//* Specifies a list of (y,x) positions for *
//* the dialog within the terminal window   *
winPos winPosList[sDATA_ITEMS] 
{
   winPos( 2,  2 ),                          // Upper Left
   winPos( termRows-dialogROWS-1,  2 ),      // Lower Left
   winPos( 2, termCols-dialogCOLS-2 ),       // Upper Right
   winPos( termRows-dialogROWS-1, termCols-dialogCOLS-2 ), // Lower Right
   winPos( termRows/2-dialogROWS/2, termCols/2-dialogCOLS/2 ), // Centered
   winPos( termRows/2-dialogROWS/2-4, termCols/2-dialogCOLS/2-10 ), // Off-center Left
   winPos( termRows/2-dialogROWS/2+4, termCols/2-dialogCOLS/2+10 ), // Off-center Right
   winPos( 0, 0 ),                           // Traveling Circus (dummy values)
   winPos( 0, 0 )                            // Custom YX Position (dummy values)
} ;           

   StatWin ( "Opening Test Dialog #8 - Move Dialog Window" ) ;

InitCtrl ic[controlsDEFINED] =   // array of dialog control initialization objects
{
   {  //* 'DONE' pushbuttion  - - - - - - - - - - - - - - - - - -   T8donePush *
      dctPUSHBUTTON,                // type:      control type
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 2),        // ulY:       upper left corner in Y
      short(dialogCOLS / 2 - 12),   // ulX:       upper left corner in X
      1,                            // lines:     control lines
      8,                            // cols:      control columns
      "  ^DONE  ",                  // dispText:  
      nc.gyR,                       // nColor:    non-focus color
      nc.reR,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T8trackrBox]              // nextCtrl:  link in next structure
   },
   {  //* Textbox (non-active)  - - - - - - - - - - - - - - - - -  T8trackrBox *
      dctTEXTBOX,                   // type:      control type
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[T8donePush].ulY - 2),// ulY:       upper left corner in Y
      short(ic[T8donePush].ulX - 3),// ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      14,                           // cols:      control columns
      NULL,                         // dispText:  initially blank
      nc.brR,                       // nColor:    non-focus color
      nc.grR,                       // fColor:    focus color
      tbPrint,                      // filter:    accepts all printing chars
      "Current Dialog Position",    // label:     label text
      1,                            // labY:      label offset
      -5,                           // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    allow control to gain focus
      &ic[T8positionControl]        // nextCtrl:  link in next structure
   },
   { //* Scrollbox  - - - - - - - - - - - - - - - - - - - -  T8positionControl *
      dctSCROLLBOX,                 // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      2,                            // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      short(sDATA_ITEMS - 2),       // lines:     control lines
      short(sCOL_WIDTH + 1),        // cols:      control columns
      (const char*)&sboxData,       // dispText:  text-data array
      nc.gyR,                       // nColor:    non-focus border color
      nc.gr,                        // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "^Where To?",                 // label:     label text
      -1,                           // labY:      label offset
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      sDATA_ITEMS,                  // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    index of initial highlighted element
                                    //            (but see first-time processing in callback)
      monoColor,                    // scrColor:  color-attribute list
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T8planetBox]              // nextCtrl:  link in next structure
   },
   { //* 'Planet' Textbox  - - - - - - - - - - - - - - - - - - - - T8planetBox *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      2,                            // ulY:       upper left corner in Y
      41,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      20,                           // cols:      control columns
      NULL,                         // dispText:  (initially blank)
      nc.bw,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Favorite ^Planet",           // label:     label text
      ZERO,                         // labY:      label offset
      -16,                          // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T8icecreamBox]            // nextCtrl:  link in next structure
   },
   { //* 'Ice Cream' Textbox - - - - - - - - - - - - - - - - - - T8icecreamBox *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[T8planetBox].ulY + 2),// ulY:       upper left corner in Y
      short(ic[T8planetBox].ulX),   // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      20,                           // cols:      control columns
      NULL,                         // dispText:  (initially blank)
      nc.bw,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "^Fave Ice Cream",            // label:     label text
      ZERO,                         // labY:      label offset
      -16,                          // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T8tuscaloosaButton]       // nextCtrl:  link in next structure
   },
   { //* Radio Button Group  - - - - - - - - - - - - - - -  T8tuscaloosaButton *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      true,                         // rbSelect:  default selection
      short(ic[T8icecreamBox].ulY + 2),// ulY:       upper left corner in Y
      short(ic[T8icecreamBox].ulX), // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a for standard types)
      nc.bw,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Tuscal^oosa",                // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T8sandiegoButton]         // nextCtrl:  link in next structure
   },
   { //* Radio Button Group  - - - - - - - - - - - - - - - -  T8sandiegoButton *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[T8tuscaloosaButton].ulY + 1),// ulY:       upper left corner in Y
      short(ic[T8tuscaloosaButton].ulX),// ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a for standard types)
      nc.bw,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "San Die^go",                 // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T8cucamongaButton]        // nextCtrl:  link in next structure
   },
   { //* Radio Button Group  - - - - - - - - - - - - - - - - T8cucamongaButton *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[T8sandiegoButton].ulY + 1),// ulY:       upper left corner in Y
      short(ic[T8sandiegoButton].ulX),// ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a for standard types)
      nc.bw,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Cu^camonga",                 // label:     
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T8valparisoButton]        // nextCtrl:  link in next structure
   },
   { //* Radio Button Group  - - - - - - - - - - - - - - - - T8valparisoButton *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[T8cucamongaButton].ulY + 1),// ulY:       upper left corner in Y
      short(ic[T8cucamongaButton].ulX),// ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a for standard types)
      nc.bw,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Valpa^riso",                 // label:     
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T8dumpPush]               // nextCtrl:  link in next structure
   },
   { //* 'DUMP UI' pushbutton  - - - - - - - - - - - - - - - - - -  T8dumpPush *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[T8valparisoButton].ulY + 2),// ulY:       upper left corner in Y
      short(ic[T8valparisoButton].ulX + 1),// ulX:       upper left corner in X
      1,                            // lines:     control lines
      18,                           // cols:      control columns
      dumpitStrings[0],             // dispText:  
      nc.cyR,                       // nColor:    non-focus color
      nc.grG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T8scottyButton]           // nextCtrl:  link in next structure
   },
   { //* Stand-alone Radio Button -- - - - - - - - - - - - - -  T8scottyButton *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      12,                           // ulY:       upper left corner in Y
      26,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "\"Be^am me up, Scotty!\"",   // label:     
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T8spinnerControl]         // nextCtrl:  link in next structure
   },
   { //* Spinner Control   - - - - - - - - - - - - - - - - -  T8spinnerControl *
      dctSPINNER,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      10,                           // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      5,                            // cols:      control columns
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.grbk,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "^Spin Control",              // label:     
      ZERO,                         // labY:      
      7,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      &dsData,                      // spinData:  spinner init
      true,                         // active:    allow control to gain focus
      &ic[T8dropdownBox]            // nextCtrl:  link in next structure
   },
   { //* Dropdown Control  - - - - - - - - - - - - - - - - - - - T8dropdownBox *
      dctDROPDOWN,                  // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      13,                           // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      scDATA_ITEMS / 2,             // lines:     control lines
      scITEM_LEN + 1,               // cols:      control columns
      (char*)&ddboxData,            // dispText:  n/a
      nc.gyR,                       // nColor:    non-focus border color
      nc.gr,                        // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "Dropdown a We^ll!",          // label:     
      -1,                           // labY:      offset from control's ulY
      ZERO,                         // labX       offset from control's ulX
      ddBoxDOWN,                    // exType:    (n/a)
      scDATA_ITEMS,                 // scrItems:  number of elements in text/color arrays
      26,                           // scrSel:    index of initial highlighted element
      monoColor,                    // scrColor:  single-color data display
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T8scrollextControl]       // nextCtrl:  link in next structure
   },
   { //* Scrollext Control - - - - - - - - - - - - - - - -  T8scrollextControl *
      dctSCROLLEXT,                 // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      17,                           // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      5,                            // lines:     control lines
      scITEM_LEN + 1,               // cols:      control columns
      NULL,                         // dispText:  (n/a - set below)
      nc.gyR,                       // nColor:    non-focus border color
      nc.gr,                        // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "Scroll O^n Down!",           // label:     
      -1,                           // labY:      offset from control's ulY
      ZERO,                         // labX       offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      scDATA_ITEMS,                 // scrItems:  (n/a - set below)
      ZERO,                         // scrSel:    (n/a - set below)
      NULL,                         // scrColor:  (n/a - set below)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T8sliderControl]          // nextCtrl:  link in next structure
   },
   { //* 'Temperature' Slider   - - - - - - - - - - - - - - -  T8sliderControl *
      dctSLIDER,                    // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[T8dropdownBox].ulY,        // ulY:       upper left corner in Y
      short(ic[T8dropdownBox].ulX + ic[T8dropdownBox].cols + 1), // ulX: upper left corner in X
      9,                            // lines:     control rows
      2,                            // cols:      control columns
      NULL,                         // dispText:  (n/a)
      nc.bkcy,                      // nColor:    non-focus color
      nc.brgr,                      // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Freezer ^Temperature\n",     // label:     label text
      ZERO,                         // labY:      label offset
      3,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T8menuwin01]              // nextCtrl:  link in next structure
   },
   { //* Menuwin Control #1  - - - - - - - - - - - - - - - - - -   T8menuwin01 *
      dctMENUWIN,                   // type:      define a menu-window control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      18,                           // ulY:       upper left corner in Y
      22,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      mw01ITEM_LEN + 2,             // cols:      control columns (incl. borders)
      (const char*)mw01Data,        // dispText:  n/a
      nc.gr,                        // nColor:    non-focus border color
      nc.grR,                       // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      " Menu ^Y Items 1-8  ",       // label:     'primary menu'
      -1,                           // labY:      offset from control's ulY
      ZERO,                         // labX       offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      mw01DATA_ITEMS,               // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    (n/a)
      monoColor,                    // scrColor:  single-color data display
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T8menuwin02]              // nextCtrl:  link in next structure
   },
   { //* Menuwin Control #2  - - - - - - - - - - - - - - - - - -   T8menuwin02 *
      dctMENUWIN,                   // type:      define a menu-window control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      ic[T8menuwin01].ulY,          // ulY:       upper left corner in Y
      short(ic[T8menuwin01].ulX + ic[T8menuwin01].cols + 1), // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      mw02ITEM_LEN + 2,             // cols:      control columns (incl. borders)
      (const char*)mw02Data,        // dispText:  n/a
      nc.gr,                        // nColor:    non-focus border color
      nc.grR,                       // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      " Menu ^Z Items 9-16 ",       // label:     'primary menu'
      -1,                           // labY:      offset from control's ulY
      ZERO,                         // labX       offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      mw01DATA_ITEMS,               // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    (n/a)
      monoColor,                    // scrColor:  single-color data display
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T8menuwin03]              // nextCtrl:  link in next structure
   },
   { //* Menuwin Sub-menu (for menu 01)  - - - - - - - - - - - -   T8menuwin03 *
      dctMENUWIN,                   // type:      define a menu-window control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      19,                           // ulY:       upper left corner in Y
      40,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      mw03ITEM_LEN + 2,             // cols:      control columns (incl. borders)
      (const char*)mw03Data,        // dispText:  n/a
      nc.gr,                        // nColor:    non-focus border color
      nc.grR,                       // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     'sub-menu'
      ZERO,                         // labY:      offset from control's ulY
      ZERO,                         // labX       offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      mw03DATA_ITEMS,               // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    (n/a)
      monoColor,                    // scrColor:  single-color data display
      NULL,                         // spinData:  (n/a)
      false,                        // active:    initially unavailable to user
      &ic[T8menuwin04]              // nextCtrl:  link in next structure
   },
   { //* Menuwin Sub-menu (for menu 02)  - - - - - - - - - - - -   T8menuwin04 *
      dctMENUWIN,                   // type:      define a menu-window control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      21,                           // ulY:       upper left corner in Y
      29,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      mw04ITEM_LEN + 2,             // cols:      control columns (incl. borders)
      (const char*)mw04Data,        // dispText:  n/a
      nc.gr,                        // nColor:    non-focus border color
      nc.grR,                       // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     'sub-menu'
      ZERO,                         // labY:      offset from control's ulY
      ZERO,                         // labX       offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      mw04DATA_ITEMS,               // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    (n/a)
      monoColor,                    // scrColor:  single-color data display
      NULL,                         // spinData:  (n/a)
      false,                        // active:    initially unavailable to user
      NULL,                         // nextCtrl:  link in next structure
   },
} ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       "  Moving the Dialog Window  ", // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;
   Test08Dialog = dp ;              // give callback method access to the dialog

   //* Open the dialog window *
   if (   (termRows >= minNeededLines && termCols >= minNeededCols) 
       && ((dp->OpenWindow()) == OK) )
   {
      //* Enable Mouse Support *
      if ( (dp->meEnableStableMouse ()) != OK )
         ;  // warn user if mouse support was not enabled

      //* Button-group label *
      dp->WriteString ( ic[T8tuscaloosaButton].ulY, ic[T8tuscaloosaButton].ulX - 16, 
                                                   "4-syllable City", dColor ) ;

      //* Initialize display data for dctSCROLLEXT control *
      ssetData sData( seboxData, seboxColors, scDATA_ITEMS, 5, true ) ;
      dp->SetScrollextText ( T8scrollextControl, sData ) ;

      //* Use Right-justify format for editing text in icecreamBox *
      dp->SetTextboxCursor ( T8icecreamBox, tbcpRIGHTJUST ) ;

      //* Give user a clue *
      dp->WriteParagraph ( dialogROWS-6, 2, 
               "Use Tab or hotkeys to navigate among controls, and then\n"
               "select a new dialog position from the ScrollBox control.", dColor ) ;
      dp->WriteParagraph ( dialogROWS - 4, dialogCOLS - 24,
                           "Capture Dialog To File\n"
                           "  ALT+z == plain text\n"
                           "  ALT+SHIFT+z == html", bColor ) ;


      //* Some color text to test synchronization of character/color *
      //* on window capture-hide-restore.                            *
      winPos wp( 8, 25 ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "=Magenta Stout=", nc.maG ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "==Green Stout==", nc.grG ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "===Bold Blue===", nc.blB ) ;
      wp.ypos += 5 ;
      gString gsOut ( "  Menu Bar Is Initially Hidden  \n"
                      " Press CTRL+V To View/Hide Menu " ) ;
      dp->WriteParagraph ( wp, gsOut, dColor | ncbATTR ) ;

      //* Create a MenuBar (a group of DialogMenuwin controls), attach         *
      //* associated sub-menus, and make MenuBar initially invisible.          *
      short mwList[] = { T8menuwin01, T8menuwin02, -1 } ;
      if ( (dp->GroupMenuwinControls ( mwList, nckC_V )) == OK )
      {
         //* Attach sub-menus to main menu controls *
         //* One sub-menu is attached to T8menuwin01 at item 2 (index 1).      *
         //* One sub-menu is attached to T8menuwin02 at item 4 (index 3).      *
         short submenuList01[] = { MAX_DIALOG_CONTROLS, T8menuwin03, -1 } ;
         if ( (dp->AttachMenuwinSubmenus ( T8menuwin01, submenuList01 )) != OK )
            dp->DebugMsg ( "Attach Sub-menu Error!" ) ;
         short submenuList02[] = { MAX_DIALOG_CONTROLS, MAX_DIALOG_CONTROLS, 
                                   MAX_DIALOG_CONTROLS, T8menuwin04, -1 } ;
         if ( (dp->AttachMenuwinSubmenus ( T8menuwin02, submenuList02 )) != OK )
            dp->DebugMsg ( "Attach Sub-menu Error!" ) ;
   
         if ( (dp->HideMenuBar ( T8menuwin01 )) != OK )
            dp->DebugMsg ( "Hide-Menu-Bar Error!" ) ;
      }
      else
         dp->DebugMsg ( "Group-Menuwin Error!" ) ;

      //* Set radio buttons as an exclusive-OR group.                *
      //* (only one button in a group can be selected at any moment).*
      short XorGroup1[5] = { 
                             T8tuscaloosaButton, T8sandiegoButton, 
                             T8cucamongaButton, T8valparisoButton, -1 
                           } ;
      dp->GroupRadiobuttons ( XorGroup1 ) ;

      //* Initialize the dctSLIDER control.*
      dp->SetSliderConfig ( T8sliderControl, slData ) ;
      dp->SliderAlert ( T8sliderControl, true ) ;
      dp->SliderAutoReport ( T8sliderControl, true, 
                             (ic[T8sliderControl].ulY + ic[T8sliderControl].labY + 1),
                             (ic[T8sliderControl].ulX + ic[T8sliderControl].labX),
                             "%5.1lf°C", bColor, false ) ;

      //* Enable audible alert for invalid input into Textbox controls.*
      dp->TextboxAlert ( T8planetBox, true ) ;
      dp->TextboxAlert ( T8icecreamBox, true ) ;

      //* Make our data visible *
      dp->RefreshWin () ;

      //* Establish a call-back method that can be called from within the *
      //* NcDialog code. Called each time through a control's user-input  *
      //* loop so we can manually modify the controls when necessary.     *
      dp->EstablishCallback ( &T08_ControlUpdate ) ;

      //* Allow user to play with the controls.      *
      //* (Please see the notes in NcDialog.hpp on   *
      //* values returned (from the editing routines)*
      uiInfo Info ;                 // user interface data returned here
      short  icIndex = ZERO ;       // index of control with input focus
      bool   uidumpEnabled = false, // enable user-interface info dump
             done = false ;         // loop control
      while ( ! done )
      {
         //*******************************************
         //* If focus is currently on a Pushbutton   *
         //*******************************************
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            //* Get user input for this control                            *
            //* Returns when the Pushbutton is pressed OR when the user    *
            //* requests that the input focus moves to another control.    *
            if ( Info.viaHotkey == false )
            {
               icIndex = dp->EditPushbutton ( Info ) ;
               if ( uidumpEnabled )
               {  // To view contents of Info
                  //* Note that the coordinates for display are absolute 
                  //* screen coordinates, not dialog offsets
                  winPos wPos(2,1) ;
                  dp->Dump_uiInfo ( wPos, Info ) ;
               }
            }
            //* If user selected the current control object via hotkey, we *
            //* don't need to call the input routine; but simply process   *
            //* the existing data.                                         *
            else
            {
               if ( uidumpEnabled )
               {  // To view contents of Info
                  winPos wPos(2,1) ;
                  dp->Dump_uiInfo ( wPos, Info ) ;
               }
               //* Information arrived in the Info.h_xxx fields.*
               //* Transfer it to the primary positions.        *
               //* Previous data in these fields has already    *
               //* been handled in prior loop iteration.        *
               Info.HotData2Primary () ;
               if ( uidumpEnabled )
               {  // To view contents of Info
                  winPos wPos(2,1) ;
                  dp->Dump_uiInfo ( wPos, Info ) ;
               }
            }

            //* If a Pushbutton was pressed *
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == T8donePush ) // request to exit the dialog
                  done = true ;
               else if ( Info.ctrlIndex == T8dumpPush )
               {
                  uidumpEnabled = (uidumpEnabled ? false : true) ;
                  dp->SetPushbuttonText ( T8dumpPush, dumpitStrings[uidumpEnabled] ) ;
               }
               // (no other pushbuttons defined in this dialog)
            }
            else
            {
               // No button press, so nothing to do
            }
         }

         //*******************************************
         //* If focus is currently on a Text Box     *
         //*******************************************
         else if ( ic[icIndex].type == dctTEXTBOX )
         {
            //* It doesn't matter how this control gained the input    *
            //* focus, by Tab/ShiftTab or via hotkey. If by hotkey, no *
            //* processing was done anyway, so discard any hotkey data *
            Info.viaHotkey = false ;

            //* Allow user to modify the text data in the text box.    *
            //* Returns when edit is complete.                         *
            icIndex = dp->EditTextbox ( Info ) ;
            if ( uidumpEnabled )
            {  // To view contents of Info
               winPos wPos(2,1) ;
               dp->Dump_uiInfo ( wPos, Info ) ;
            }

            //* If the data in the text box being edited has changed *
            if ( Info.dataMod != false )
            {
               //* Do any processing related to a change in the text box data *
               if ( Info.ctrlIndex == T8planetBox )
               {
                  // DO STUFF
                  dp->DebugMsg ( "planetBox changed", 2 ) ;
               }
               else if ( Info.ctrlIndex == T8icecreamBox )
               {
                  // DO STUFF
                  dp->DebugMsg ( "icecreamBox changed", 2 ) ;
               }
               // no other text boxes defined in this dialog
               dp->DebugMsg ( "%" ) ;     // clear the message
            }
         }

         //*******************************************
         //* If focus is currently on a Scroll Box   *
         //*******************************************
         else if ( ic[icIndex].type == dctSCROLLBOX )
         {
            //* It doesn't matter how this control gained the input    *
            //* focus, by Tab/ShiftTab or via hotkey. If by hotkey, no *
            //* processing was done anyway, so discard any hotkey data *
            Info.viaHotkey = false ;

            //* Allow user to modify the selected member of the scroll box *
            //* Returns when edit is complete.                             *
            icIndex = dp->EditScrollbox ( Info ) ;
            if ( uidumpEnabled )
            {  // To view contents of Info
               winPos wPos(2,1) ;
               dp->Dump_uiInfo ( wPos, Info ) ;
            }

            //* If 'selected' member of Scroll Box has changed *
            if ( Info.dataMod != false || Info.keyIn == nckENTER )
            {
               // NOTE: We have only one dctSCROLLBOX defined in this dialog.
               //       If we had more, we would test for that here.
               if ( Info.ctrlIndex == T8positionControl )
               {
                  //* We are going to make the dialog dance *
                  if ( Info.selMember == sboxItemCircus )
                  {
                     Test8_CircusCircus ( dialogROWS, dialogCOLS ) ;
                     //* Focus will be on T8donePush at top of loop
                     while ( icIndex < T8scrollextControl )
                        icIndex = dp->NextControl () ;
                  }
                  else
                  {
                     //* Use the specified standard position *
                     winPos wPos (winPosList[Info.selMember].ypos, 
                                          winPosList[Info.selMember].xpos) ;
                     //* UNLESS custom dialog position, in which case,*
                     //* open a dialog to get coordinates from user.  *
                     if ( Info.selMember == sboxItemCustom )
                     {
                        wPos = dp->GetDialogPosition () ; // current dialog position
                        //* position for opening sub-dialog (center of dialog) *
                        winPos oPos ( wPos.ypos+(dialogROWS/2), wPos.xpos+(dialogCOLS/2) ) ;
                        dp->SetDialogObscured () ; // protect our display data
                        Test8_GetNewPosition ( oPos, wPos ) ;
                        dp->RefreshWin () ;        // restore display
                     }
                     //* Move the window to the specified position *
                     //* and update the displayed coordinates.     *
                     if ( (dp->MoveWin ( wPos )) == OK )
                     {
                        wkeyCode wktmp ;
                        T08_ControlUpdate ( icIndex, wktmp ) ;
                     }
                     else
                     {
                        dp->DebugMsg ( "   Invalid dialog position specified!!   ", 
                                       4, true ) ;
                     }
                  }
               }
            }
         }

         //*******************************************
         //* If focus is currently on a Radio Button *
         //*******************************************
         else if ( ic[icIndex].type == dctRADIOBUTTON )
         {
            //* Get user input for this control                            *
            //* Returns when radio-button control or radio-button group is *
            //* about to lose the input focus.                             *
            //* A selection may or may not have been made.                 *
            if ( Info.viaHotkey == false )
            {
               icIndex = dp->EditRadiobutton ( Info ) ;
               if ( uidumpEnabled )
               {  // To view contents of Info
                  winPos wPos(2,1) ;
                  dp->Dump_uiInfo ( wPos, Info ) ;
               }
            }
            //* If user selected the current control object via hotkey, we *
            //* don't need to call the input routine; but simply process   *
            //* the existing data.                                         *
            //* For radio buttons, display update will already have been   *
            //* done, so we can just move the focus to the next control.   *
            else
            {
               if ( uidumpEnabled )
               {  // To view contents of Info
                  winPos wPos(2,1) ;
                  dp->Dump_uiInfo ( wPos, Info ) ;
               }
               //* Information arrived in the Info.h_xxx fields.*
               //* Transfer it to the primary positions.        *
               //* Previous data in these fields has already    *
               //* been handled in prior loop iteration.        *
               Info.HotData2Primary () ;
               if ( uidumpEnabled )
               {  // To view contents of Info
                  winPos wPos(2,1) ;
                  dp->Dump_uiInfo ( wPos, Info ) ;
               }
            }
            
            //* Note that these buttons are only for testing, and don't 
            //* actually do anything, but any processing based on state 
            //* changed in these controls would happen here...

            //* If 'selected' button is a member of a button group *
            if ( Info.selMember < MAX_DIALOG_CONTROLS )
            {
               //* If a new 'selected' member of the group *
               //* OR previous selection was re-selected   *
               if ( Info.dataMod != false || Info.keyIn == nckENTER )
               {
                  //* Process the user's selection *
                  switch ( Info.selMember )
                  {
                     case T8tuscaloosaButton:
                        // DO STUFF
                        dp->DebugMsg ( "  Tuscaloosa!  ", 2 ) ;
                        break ;
                     case T8sandiegoButton:
                        // DO STUFF
                        dp->DebugMsg ( "  San Diego!  ", 2 ) ;
                        break ;
                     case T8cucamongaButton:
                        // DO STUFF
                        dp->DebugMsg ( "  Cucamonga!  ", 2 ) ;
                        break ;
                     case T8valparisoButton:
                        // DO STUFF
                        dp->DebugMsg ( "  Valpariso!  ", 2 ) ;
                        break ;
                  }
                  dp->DebugMsg ( "%" ) ;
               }
            }
            //* Else, button is an independent button *
            //* (Info.selMember==MAX_DIALOG_CONTROLS) *
            else
            {
               //* If it is the scottyButton AND its state has changed *
               if ( Info.ctrlIndex == T8scottyButton && Info.dataMod != false )
               {
                  if ( Info.isSel != false )
                  {
                     // Beam Kirk Up to the ship - saving his ass once again
                     dp->DebugMsg ( "  ~~~ Beam Up ~~~  ", 2 ) ;
                     dp->DebugMsg ( "%" ) ;
                  }
                  else
                  {
                     // Don't Beam Kirk Up - let him spend the night 
                     //                      with the sexy alien wench
                     dp->DebugMsg ( "  ~~~ No Beam Up ~~~  ", 2 ) ;
                     dp->DebugMsg ( "%" ) ;
                  }
               }
               // (no other independent buttons in this dialog)

            }
         }  // radio button

         //**********************************************
         //* If focus is currently on a Spinner Control *
         //**********************************************
         else if ( ic[icIndex].type == dctSPINNER )
         {
            //* It doesn't matter how this control gained the input    *
            //* focus, by Tab/ShiftTab or via hotkey. If by hotkey, no *
            //* processing was done anyway, so discard any hotkey data *
            Info.viaHotkey = false ;

            //* Allow user to modify the spinner control's value.      *
            //* Returns when edit is complete.                         *
            icIndex = dp->EditSpinner ( Info ) ;
            if ( uidumpEnabled )
            {  // To view contents of Info
               winPos wPos(2,1) ;
               dp->Dump_uiInfo ( wPos, Info ) ;
            }

            //* If displayed value has been modified *
            if ( Info.dataMod != false )
            {
               //* Take any necessary actions here.  *
               //* For now, we just report the value.*
               int spval ;
               dp->GetSpinnerValue ( T8spinnerControl, spval ) ;
               std::ostringstream ostr(ostringstream::out) ;
               ostr << "Spinner Value " << spval ;
               std::string str = ostr.str() ;
               dp->DebugMsg ( str.c_str(), 2 ) ;
               dp->DebugMsg ( "%" ) ;
            }
            else
            { /* Spinner data not modified, so nothing to do */ }
         }

         //*********************************************
         //* If focus is currently on a Slider Control *
         //*********************************************
         else if ( ic[icIndex].type == dctSLIDER )
         {
            Info.viaHotkey = false ;   // ignore arrival via hotkey

            //* Allow user to modify the slider control's value.*
            //* Returns when edit is complete.                  *
            icIndex = dp->EditSlider ( Info ) ;
            if ( uidumpEnabled )
            {  // To view contents of Info
               winPos wPos(2,1) ;
               dp->Dump_uiInfo ( wPos, Info ) ;
            }

            //* If displayed value has been modified *
            if ( Info.dataMod != false )
            {
            }
         }

         //***********************************************
         //* If focus is currently on a Dropdown control *
         //***********************************************
         else if ( ic[icIndex].type == dctDROPDOWN )
         {
            //* It doesn't matter how this control gained the input    *
            //* focus, by Tab/ShiftTab or via hotkey. If by hotkey, no *
            //* processing was done anyway, so discard any hotkey data *
            //*                                                        *
            //* However, if you want the control to expand immediately *
            //* when EditDropdown() is called, then set                *
            //* Info.viaHotkey==true. Otherwise, the control will      *
            //* expand when the user presses Enter or one of the       *
            //* vertical scrolling keys. This is useful if the user    *
            //* arrived at the current control via hotkey because by   *
            //* selecting the control via hotkey, he/she has already   *
            //* demonstrated the intention to edit the control.        *
            //* (all other data in the Info class is ignored on entry) *
            //Info.viaHotkey = false ;

            //* Allow user to modify which member of the Dropdown      *
            //* is selected.  Returns when edit is complete.           *
            icIndex = dp->EditDropdown ( Info ) ;
            if ( uidumpEnabled )
            {  // To view contents of Info
               //* Note that the coordinates for display are absolute 
               //* screen coordinates, not dialog offsets
               winPos wPos(2,1) ;
               dp->Dump_uiInfo ( wPos, Info ) ;
            }

            //* If 'selected' member of Dropdown has changed *
            if ( Info.dataMod != false || Info.keyIn == nckENTER )
            {
               // selection has been modified - do any processing here
            }
         }

         //************************************************
         //* If focus is currently on a Scrollext Control *
         //************************************************
         else if ( ic[icIndex].type == dctSCROLLEXT )
         {
            //* It doesn't matter how this control gained the input    *
            //* focus, by Tab/ShiftTab or via hotkey. If by hotkey, no *
            //* processing was done anyway, so discard any hotkey data *
            Info.viaHotkey = false ;

            //* Allow user to modify the selected member of the scroll ext.*
            //* Returns when edit is complete.                             *
            icIndex = dp->EditScrollext ( Info ) ;
            if ( uidumpEnabled )
            {  // To view contents of Info
               winPos wPos(2,1) ;
               dp->Dump_uiInfo ( wPos, Info ) ;
            }

            //* If 'selected' member of Scroll Box has changed *
            if ( Info.dataMod != false || Info.keyIn == nckENTER )
            {
               // NOTE: We have only one dctSCROLLEXT defined in this dialog.
               //       If we had more, we would test for that here.
               if ( Info.ctrlIndex == T8scrollextControl )
               {
               }
            }
         }

         //**********************************************
         //* If focus is currently on a Menuwin Control *
         //**********************************************
         else if ( ic[icIndex].type == dctMENUWIN )
         {
            //* Note: If we arrived via hotkey, menu is expanded immediately *
            icIndex = dp->EditMenuwin ( Info ) ;
            if ( uidumpEnabled )
            {  // To view contents of Info
               winPos wPos(2,1) ;
               dp->Dump_uiInfo ( wPos, Info ) ;
            }

            //* If a selection has been made *
            if ( Info.dataMod != false )
            {
               #if 0    // TEST ONLY
               //* Most menu items in these menu controls have no functionality*
               //* attached; however, we informally use the following menu     *
               //* items to test the following NcDialog-class methods.         *
               //* SetScrollboxSelect and GetScrollboxSelect methods           *
               //*   T8menuwin01, index 2 and 3: get/set dctSCROLLBOX item     *
               //* SetScrollextSelect and GetScrollextSelect methods           *
               //*   T8menuwin01, index 4 and 5: get/set dctSCROLLEXT item     *
               //* SetDropdownSelect and GetDropdownSelect methods             *
               //*   T8menuwin01, index 6 and 7: get/set dctDROPDOWN item      *
               if ( Info.ctrlIndex == T8menuwin01 )
               {
                  short oldItem, newItem ;
                  //* Set scrollbox index *
                  if ( Info.selMember == 2 || Info.selMember == 3 )
                  {
                     oldItem = dp->GetScrollboxSelect ( T8positionControl ) ;
                     newItem = Info.selMember == 2 ? 2 : 3 ;
                     if ( newItem != oldItem )
                     {
                        if ( (dp->SetScrollboxSelect ( T8positionControl, 
                                                       newItem )) == newItem )
                        {
                           dp->DebugMsg ( "dctScrollBox selection modified", 2 ) ;
                           //* Move the window to the specified position *
                           winPos wPos (winPosList[newItem].ypos, 
                                                winPosList[newItem].xpos) ;
                           dp->MoveWin ( wPos ) ;
                        }
                        else     // error condition
                        {
                           dp->DebugMsg ( "dctScrollBox selection error", 2 ) ;
                        }
                        dp->DebugMsg ( "%" ) ;
                     }
                  }
                  else if ( Info.selMember == 4 || Info.selMember == 5 )
                  {
                     oldItem = dp->GetScrollextSelect ( T8scrollextControl ) ;
                     newItem = Info.selMember == 4 ? 10 : 23 ;
                     if ( newItem != oldItem )
                     {
                        if ( (dp->SetScrollextSelect ( T8scrollextControl, 
                                                       newItem )) == newItem )
                           dp->DebugMsg ( "dctScrollext selection modified", 2 ) ;
                        else     // error condition
                           dp->DebugMsg ( "dctScrollext selection error", 2 ) ;
                        dp->DebugMsg ( "%" ) ;
                     }
                  }
                  else if ( Info.selMember == 6 || Info.selMember == 7 )
                  {
                     oldItem = dp->GetDropdownSelect ( T8dropdownBox ) ;
                     newItem = Info.selMember == 6 ? 15 : 31 ;
                     if ( newItem != oldItem )
                     {
                        if ( (dp->SetDropdownSelect ( T8dropdownBox, 
                                                      newItem )) == newItem )
                           dp->DebugMsg ( "dctDROPDOWN selection modified", 2 ) ;
                        else     // error condition
                           dp->DebugMsg ( "dctDROPDOWN selection error", 2 ) ;
                        dp->DebugMsg ( "%" ) ;
                     }
                  }
               }
               #endif   // TEST ONLY - SetSbSelect and GetSbSelect methods
            }
         }

         else
            {  /* no other active controls defined for this dialog */ }

         //* If user exited the control edit method via a hotkey,*
         //* then the new control already has focus. Otherwise,  *
         //* move input focus to next/previous control.          *
         if ( done == false && Info.viaHotkey == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }  // while()
   }  // OpenWindow()
   else
   {  //* Most likely cause of dialog not opening is that the terminal window  *
      //* is too small. Give user a clue...                                    *
      short neededLines = minNeededLines, 
            neededCols  = minNeededCols ;
      gString gs ;
      gs.compose ( L" Sorry, size of terminal window must be at least %hd x %hd for this test. ", 
                   &neededCols, &neededLines ) ;
      StatWin ( gs.gstr(), nc.cyG ) ;
      sleep ( 5 ) ;
   }

   if ( (dp->meMouseEnabled ()) )            // disable mouse interface
      dp->meDisableMouse ();

   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

}  //* End Test08() *

//*************************
//*  T08_ControlUpdate    *
//*************************
//******************************************************************************
//* This is a callback method for manually updating the controls in the        *
//* Test08() dialog.                                                           *
//*                                                                            *
//*  For this test, the following are updated by the callback method:          *
//*   1. manual update of non-active (display-only) text box that reports      *
//*      the current position of the dialog window                             *
//*   2.                                                                       *
//*   3.                                                                       *
//*                                                                            *
//* Input  : currIndex: index of control that currently has focus              *
//*          wkey     : user's key input data                                  *
//*          firstTime: the EstablishCallback() method calls this method once  *
//*                     with firstTime==true, to perform any required          *
//*                     initialization. Subsequently, the NcDialog class       *
//*                     always calls with firstTime==false.                    *
//* Returns: OK                                                                *
//******************************************************************************
//* Important Note: This method makes some intimate assumptions about what     *
//* is happening in the method that established the callback. If changes are   *
//* made in the establishing method that affect the callback functionality,    *
//* be sure to update this method to address those changes.                    *
//*                                                                            *
//* Programmer's Note: See dialog control index constants at file scope above. *
//******************************************************************************

short T08_ControlUpdate ( const short currIndex, const wkeyCode wkey, bool firstTime )
{
static winPos  oldPos(0,0) ;     // previous position of the dialog window
winPos         newPos(0,0) ;     // current position of the dialog window

   //*************************
   //* First-time processing *
   //*************************
   if ( firstTime != false )
   {
      //* Scroll through T8positionControl scrollbox *
      //* until 'Centered' item is centered.         *
      Test08Dialog->SetScrollboxSelect ( T8positionControl, 6 ) ;
      Test08Dialog->SetScrollboxSelect ( T8positionControl, 4 ) ;
   }

   //***********************
   //* Standard processing *
   //***********************
   //* If dialog window has moved, update its coordinates 
   //* displayed in the T8BOX Text Box.
   newPos = Test08Dialog->GetDialogPosition () ;
   if ( newPos.ypos != oldPos.ypos || newPos.xpos != oldPos.xpos || firstTime != false )
   {
      gString  gs( " Y:%03hd  X:%03hd ", &newPos.ypos, &newPos.xpos ) ;
      Test08Dialog->SetTextboxText ( T8trackrBox, gs ) ;
      oldPos = newPos ;
      //* Test for stale display data (see note above) *
      if ( currIndex == T8menuwin01 )
         Test08Dialog->RefreshWin () ;
   }

   //* If command to capture dialog's text/attribute data to a file.*
   //* This tests the CaptureDialog() method.                       *
   if ( (wkey.type == wktEXTEND) && 
        ((wkey.key == nckA_Z) || (wkey.key == nckAS_Z)) )
   {
      const char* const capText = "./capturedlg.txt" ;
      const char* const capHtml = "./capturedlg.html" ;
      bool fmt = bool(wkey.key == nckAS_Z) ;
      short capStatus =
         Test08Dialog->CaptureDialog ( 
            (fmt ? capHtml : capText), // output filename
            fmt,                       // format text/html
            #if 1    // Table-based HTML output for maximum number of colors
            true,                      // timestamp enable
            NULL,                      // default 'screenshot-styles.css'
            4,                         // table cells per output line
            true                       // enable comments
            #else    // Simple HTML output for Texinfo documentation
            false,                     // timestamp disable
            "../Texinfo/infodoc-styles.css", // simple CSS formatting
            4,                         // (ignored for simple format)
            false,                     // disable comments
            nc.blR                     // dialog background color
            #endif
            ) ;
      gString gs ;
      gs.compose( L"Screen Capture to: '%s'", 
                  (fmt ? capHtml : capText) ) ;
      if ( capStatus != OK )
         gs.append( L" FAILED!" ) ;
      Test08Dialog->DebugMsg ( gs.ustr(), 2, true ) ;
   }

   return OK ;

}  //* End T08_ControlUpdate() *

//**************************
//*  Test8_GetNewPosition  *
//**************************
//******************************************************************************
//* Prompt user for a new set of coordinates for positioning the Test08()      *
//* dialog window.                                                             *
//*                                                                            *
//* Input  : oPos: center Y/X of parent dialog                                 *
//*          wPos: current Y/X of upper left corner of dialog                  *
//* Returns: OK if successful, else ERR                                        *
//******************************************************************************

short Test8_GetNewPosition ( winPos oPos, winPos& wPos )
{
static const short dialogROWS = 8 ;       // display lines
static const short dialogCOLS = 34 ;      // display columns
static const short controlsDEFINED = 3 ;  // number of controls defined
const short    T08pbA = 0, T08tbY = 1, T08tbX = 2 ;
attr_t   dColor = nc.cyR ;                // text color for writing to dialog
short    result = OK ;                    // return value

   //* Initial display values for Y/X Textboxes *
   //* == current position of parent dialog     *
   gString  yVal( "%02hd", &wPos.ypos ) ;
   gString  xVal( "%02hd", &wPos.xpos ) ;

InitCtrl ic[controlsDEFINED] =   // array of dialog control initialization objects
{
   {  //* 'DONE' pushbutton - - - - - - - - - - - - - - - - - - - - -   T08PBA *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 2),        // ulY:       upper left corner in Y
      short(dialogCOLS/2 - 5),      // ulX:       upper left corner in X
      1,                            // lines:     control lines
      8,                            // cols:      control columns
      "  ^DONE  ",                  // dispText:  button text
      nc.gyR,                       // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[1],                       // nextCtrl:  link in next structure
   },
   {  //* Text Box for Y position  - - - - - - - - - - - - - - - - - -  T08tbY *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      2,                            // ulY:       upper left corner in Y
      13,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      5,                            // cols:      control columns
      yVal.ustr(),                  // dispText:  initial display text
      nc.bw,                        // nColor:    non-focus color
      nc.blR,                       // fColor:    focus color
      tbNumber,                     // filter:    accepts only numerals (0-9)
      "New Y:",                     // label:     label text
      ZERO,                         // labY:      label offset
      -7,                           // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[2],                       // nextCtrl:  link in next structure
   },
   {  //* Text Box for Y position  - - - - - - - - - - - - - - - - - -  T08tbY *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      4,                            // ulY:       upper left corner in Y
      13,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      5,                            // cols:      control columns
      xVal.ustr(),                  // dispText:  initial display text
      nc.bw,                        // nColor:    non-focus color
      nc.blR,                       // fColor:    focus color
      tbNumber,                     // filter:    accepts only numerals (0-9)
      "New X:",                     // label:     label text
      ZERO,                         // labY:      label offset
      -7,                           // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   },
} ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       short(oPos.ypos-(dialogROWS/2)), 
                       short(oPos.xpos-(dialogCOLS/2)),
                       "  Y/X Coordinates for Dialog  ", // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      dp->RefreshWin () ;
      
      //* Allow user to play with the controls.      *
      //* (Please see the notes in NcDialog.hpp on   *
      //* values returned (from the editing routines)*
      uiInfo Info ;                 // user interface data returned here
      short  icIndex = ZERO ;       // index of control with input focus
      bool   done = false ;         // loop control
      while ( ! done )
      {
         //*******************************************
         //* If focus is currently on a Pushbutton   *
         //*******************************************
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            icIndex = dp->EditPushbutton ( Info ) ;
            if ( icIndex == T08pbA && Info.isSel != false )
               done = true ;
         }
         //*******************************************
         //* If focus is currently on a Text Box     *
         //*******************************************
         else if ( ic[icIndex].type == dctTEXTBOX )
         {
            icIndex = dp->EditTextbox ( Info ) ;
            if ( Info.dataMod != false )
            {
               char vbTmp[32] ;
               switch ( icIndex )
               {
                  case T08tbY:
                     dp->GetTextboxText ( T08tbY, vbTmp ) ;
                     yVal = vbTmp ;
                     swscanf ( yVal.gstr(), L"%hd", &wPos.ypos ) ;
                     break ;
                  case T08tbX:
                     dp->GetTextboxText ( T08tbX, vbTmp ) ;
                     xVal = vbTmp ;
                     swscanf ( xVal.gstr(),  L"%hd", &wPos.xpos ) ;
                     break ;
               }
            }
         }
         if ( done == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }  // while()
   }  // OpenWindow()

   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

   return result ;

}  //* End Test8_GetNewPosition() *

//*************************
//*  Test8_CircusCircus   *
//*************************
//******************************************************************************
//* Dancing Dialog Diva!                                                       *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Test8_CircusCircus ( short dLines, short dCols )
{
static attr_t multiColor[16] = { nc.bwB, nc.reB, nc.grB, nc.brB, nc.blB, nc.maB, 
                                 nc.cyB, nc.gyB, nc.bwB, nc.reB, nc.grB, nc.brB, 
                                 nc.blB, nc.maB, nc.cyB, nc.gyB } ;
const char Msg[] = "Press a key, Dammit!" ;
winPos   wPos(-1,-2), 
         hitPos(0,0),
         mPos1(0,0) ;
wkeyCode wKey ;
struct timespec nanosec = { 0, 5000000 } ;
short    moveCount = 2000,
         ymax = termRows - dLines,
         xmax = termCols - dCols,
         yinc = -1, xinc = -2,
         mI = ZERO,
         origDelay = Test08Dialog->GetKeyDelay () ;
         
   Test08Dialog->SetKeyDelay ( ZERO ) ;  // non-blocking keyboard read

   do
   {
      if ( (Test08Dialog->MoveWin ( wPos, true )) != OK )
      {
         Test08Dialog->DebugMsg ( "     Hit a Wall!!     " ) ;
         nanosec.tv_nsec = 5000000 ;
         nanosleep ( &nanosec, NULL ) ;
         Test08Dialog->DebugMsg ( "%" ) ;
         hitPos = Test08Dialog->GetDialogPosition () ; // current dialog position
         if ( (yinc < ZERO) && ((hitPos.ypos + yinc) < ZERO) )
            yinc = 1 ;
         else if ( (yinc > ZERO) && ((hitPos.ypos + yinc) > ymax) )
            yinc = -1 ;
         if ( (xinc < ZERO) && ((hitPos.xpos + xinc) < ZERO) )
            xinc = 1 ;
         else if ( (xinc > ZERO) && ((hitPos.xpos + xinc) > xmax) )
            xinc = -1 ;
      }
      Test08Dialog->SetDialogObscured () ;
      do
      {
         nc.WriteString ( mPos1.ypos, mPos1.xpos, Msg, multiColor[mI] ) ;
         if ( (mPos1.ypos += 1) >= termRows )
            mPos1.ypos = ZERO ;
         if ( (mPos1.xpos += 1) >= termCols )
            mPos1.xpos = ZERO ;
         nanosec.tv_nsec = 2000000 ;
         nanosleep ( &nanosec, NULL ) ;
         ++mI ;
      }
      while ( mI % 2 ) ;
      if ( mI >= 16 )
         mI = ZERO ;
      Test08Dialog->RefreshWin () ;
      wPos.ypos = yinc ;
      wPos.xpos = xinc ;
      nanosec.tv_nsec = 10000000 ;
      nanosleep ( &nanosec, NULL ) ;
      if ( (Test08Dialog->GetKeyInput ( wKey )) != wktERR )  // if a keypress detected
         break ;
   }
   while ( --moveCount > ZERO ) ;
   Test08Dialog->SetKeyDelay ( origDelay ) ;
   Test08Dialog->SetDialogObscured () ;
   nc.ClearScreen () ;
   Test08Dialog->RefreshWin () ;

}  //* End Test8_CircusCircus() *

//*************************
//*       Test09          *
//*************************
//******************************************************************************
//* This dialog tests the functionality of the dctDROPDOWN control.            *
//*                                                                            *
//* See notes below about the dynamic re-sizing of the control to keep the     *
//* control's display entirely within the borders of the dialog window.        *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* NOTE: A Dropdown control is initialized very much like a Scrollbox.        *
//*       All the fields are the same as with a Scrollbox control, with the    *
//*       addition of the exType field that specifies the control's            *
//*       expansion type. Internal calculations are performed by the           *
//*       control's constructor for size and position of both the              *
//*       'collapsed' and 'expanded' control.                                  *
//******************************************************************************

void Test09 ( void )
{
static const short dialogROWS = 20 ;         // display lines
static const short dialogCOLS = 60 ;         // display columns
static const short controlsDEFINED = 9 ;     // # of controls in dialog
static attr_t monoColor[2] = { attrDFLT, nc.bw } ; // scroll box data colors

const short saDATA_ITEMS = 4, saITEM_LEN = 14 ;       // Data for T9ddA

//* Text data are passed to the Dropdown constructor in a     *
//* variety of formats, to stress-test the parsing algorithm. *
#if 1
static char saboxData[saDATA_ITEMS][saITEM_LEN] =   
{  // Full-width items - exact width for target control
   "00aaaaaaaaa00",  "01bbbbbbbbb01",  "02ccccccccc02",  "03ddddddddd03"
} ;
#elif 0
static char saboxData[saDATA_ITEMS][saITEM_LEN] =   
{  // Mixed-width items <= to required length (will be padded on right)
   "00aaaaaaa00",  "01bbbbbb01",  "02ccccc02",  "03dddd03"
} ;
#else
static char saboxData[] =   
{  // Single data string - will be divided into correct lengths
   "00aaaaaaaaa00""01bbbbbbbbb01""02ccccccccc02""03ddddddddd03""discarded-crap"
} ;
#endif

const short sbDATA_ITEMS = 12, sbITEM_LEN = 20 ;      // data for T9ddB
static char sbboxData[sbDATA_ITEMS][sbITEM_LEN] =   
{
   "00bbbbbbbbbbbbbbb00", "01ccccccccccccccc01", "02ddddddddddddddd02",
   "03eeeeeeeeeeeeeee03", "04fffffffffffffff04", "05ggggggggggggggg05",
   "06hhhhhhhhhhhhhhh06", "07iiiiiiiiiiiiiii07", "08-MOVE WIN LEFT-08",
   "09-MOVE WIN BACK-09", "10lllllllllllllll10", "11mmmmmmmmmmmmmmm11",
} ;

const short scDATA_ITEMS = 25, scITEM_LEN = 14 ;      // data for T9ddC
static char scboxData[scDATA_ITEMS][scITEM_LEN] =   
{
   "00CCCCCCCCC00",  "01DDDDDDDDD01",  "02EEEEEEEEE02",  "03FFFFFFFFF03",
   "04GGGGGGGGG04",  "05HHHHHHHHH05",  "06IIIIIIIII06",  "07JJJJJJJJJ07",
   "08KKKKKKKKK08",  "09LLLLLLLLL09",  "10MMMMMMMMM10",  "11NNNNNNNNN11",
   "12OOOOOOOOO12",  "13PPPPPPPPP13",  "14QQQQQQQQQ14",  "15RRRRRRRRR15",
   "16SSSSSSSSS16",  "17TTTTTTTTT17",  "18UUUUUUUUU18",  "19VVVVVVVVV19",
   "20WWWWWWWWW20",  "21XXXXXXXXX21",  "22YYYYYYYYY22",  "23ZZZZZZZZZ23",
   "24%@%@%@%@%24",
} ;

const short sdDATA_ITEMS = 12, sdITEM_LEN = 14 ;      // data for T9ddD
static char sdboxData[sdDATA_ITEMS][sdITEM_LEN] =   
{
   "00ddddddddd00", "01eeeeeeeee01", "02fffffffff02", "03ggggggggg03",
   "04hhhhhhhhh04", "05hhhhhhhhh05", "06hhhhhhhhh06", "07hhhhhhhhh07",
   "08hhhhhhhhh08", "09hhhhhhhhh09", "10hhhhhhhhh10", "11hhhhhhhhh11",
} ;

const short seDATA_ITEMS = 14, seITEM_LEN = 20 ;      // data for T9ddE
static char seboxData[seDATA_ITEMS][seITEM_LEN] =   
{
   "00eeeeeeeeeeeeeee00", "01fffffffffffffff01", "02ggggggggggggggg02",
   "03hhhhhhhhhhhhhhh03", "04iiiiiiiiiiiiiii04", "05jjjjjjjjjjjjjjj05",
   "06kkkkkkkkkkkkkkk06", "07lllllllllllllll07", "08mmmmmmmmmmmmmmm08",
   "09nnnnnnnnnnnnnnn09", "10ooooooooooooooo10", "11ppppppppppppppp11",
   "12ppppppppppppppp12", "13ppppppppppppppp13",
} ;

const short sfDATA_ITEMS = 11, sfITEM_LEN = 14 ;      // data for T9ddF
static char sfboxData[sfDATA_ITEMS][sfITEM_LEN] =   
{
   "00fffffffff00",  "01ggggggggg01",  "02hhhhhhhhh02",  "03iiiiiiiii03",
   "04jjjjjjjjj04",  "05kkkkkkkkk05",  "06lllllllll06",  "07mmmmmmmmm07",
   "08nnnnnnnnn08",  "09ooooooooo09",  "10ppppppppp10",  
} ;

const short sgDATA_ITEMS = 17, sgITEM_LEN = 14 ;      // data for T9ddG
static char sgboxData[sgDATA_ITEMS][sgITEM_LEN] =   
{
   "00ggggggggg00",  "01hhhhhhhhh01",  "02iiiiiiiii02",  "03jjjjjjjjj03",
   "04kkkkkkkkk04",  "05lllllllll05",  "06mmmmmmmmm06",  "07nnnnnnnnn07",
   "08ooooooooo08",  "09ppppppppp09",  "10qqqqqqqqq10",  "11rrrrrrrrr11",
   "12sssssssss12",  "13ttttttttt13",  "14uuuuuuuuu14",  "15vvvvvvvvv15",
   "16wwwwwwwww16",  
} ;

//* Data for T9dumpPush *
const char dumpitStrings[][19] = { " Enable UI D^ump  ", " Disable UI D^ump " } ;


attr_t dColor = nc.cyR,                   // dialog's interior color
       bColor = nc.blR ;                  // dialog's border color
short ctrY    = termRows/2,               // terminal window center in Y
      ctrX    = termCols/2,               // terminal window center in X
      ulY     = ctrY - dialogROWS/2,      // upper left corner in Y
      ulX     = ctrX - dialogCOLS/2 ;     // upper left corner in X

   StatWin ( "Opening Test Dialog #9 - Dropdown Control Test" ) ;

InitCtrl ic[controlsDEFINED] =   // array of dialog control initialization objects
{
   {  //* 'DONE' pushbuttion  - - - - - - - - - - - - - - - - - -   T9donePush *
      dctPUSHBUTTON,                // type:      control type
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 3),        // ulY:       upper left corner in Y
      short(dialogCOLS / 2 - 5),    // ulX:       upper left corner in X
      1,                            // lines:     control lines
      8,                            // cols:      control columns
      "  DO^NE  ",                  // dispText:  display text
      nc.gyR,                       // nColor:    non-focus color
      nc.reR,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[1],                       // nextCtrl:  link in next structure
   },
   //* This control 'expands' downward and fits within the dialog window       *
   //* without the need for the constructor to re-size it.                     *
   //* When expanded, all data items are visible simultaneously.               *
   { //* Dropdown control A   - - - - - - - - - - - - - - - - - - - -    T9ddA *
      dctDROPDOWN,                  // type:      define a drop-down control
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      4,                            // ulY:       upper left corner in Y
      3,                            // ulX:       upper left corner in X
      short(saDATA_ITEMS + 2),      // lines:     control lines
      short(saITEM_LEN + 1),        // cols:      control columns
      (const char*)&saboxData,      // dispText:  text-data array
      nc.cyR,                       // nColor:    non-focus border color
      nc.gr,                        // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "Box ^A",                     // label:     label text
      // NOTE: Style would indicate that the control's label be positioned 
      //       so that it will NOT BE obscured when the control expands
      -1,                           // labY:      label offset
      1,                            // labX       
      ddBoxDOWN,                    // exType:    expansion type
      saDATA_ITEMS,                 // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    index of initial highlighted element
      monoColor,                    // scrColor:  color-attribute list
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[2],                       // nextCtrl:  link in next structure
   },
   //* This control 'expands' upward but must be dynamically resized by the    *
   //* constructor to avoid extending to (or beyond) the border of the dialog  *
   //* window. Some data items will be out of view until user scrolls to them. *
   { //* Dropdown control B   - - - - - - - - - - - - - - - - - - - -    T9ddB *
      dctDROPDOWN,                  // type:      define a drop-down control
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      5,                            // ulY:       upper left corner in Y
      19,                           // ulX:       upper left corner in X
      short(sbDATA_ITEMS + 2),      // lines:     control lines
      short(sbITEM_LEN + 1),        // cols:      control columns
      (const char*)&sbboxData,      // dispText:  text-data array
      nc.cyR,                       // nColor:    non-focus border color
      nc.gr,                        // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "Box ^B",                     // label:     label text
      3,                            // labY:      label offset
      1,                            // labX       
      ddBoxUP,                      // exType:    expansion type
      sbDATA_ITEMS,                 // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    index of initial highlighted element
      monoColor,                    // scrColor:  color-attribute list
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[3],                       // nextCtrl:  link in next structure
   },
   //* Expansion of this control is centered on the base line specified (ulY). *
   //* This control is dynamically resized by the constructor to avoid         *
   //* extending to (or beyond) the border of the dialog window.               *
   { //* Dropdown control C   - - - - - - - - - - - - - - - - - - - -    T9ddC *
      dctDROPDOWN,                  // type:      define a drop-down control
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      3,                            // ulY:       upper left corner in Y
      42,                           // ulX:       upper left corner in X
      short(scDATA_ITEMS / 2),      // lines:     control lines
      short(scITEM_LEN + 1),        // cols:      control columns
      (const char*)&scboxData,      // dispText:  text-data array
      nc.cyR,                       // nColor:    non-focus border color
      nc.gr,                        // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "Box ^C",                     // label:     label text
      1,                            // labY:      label offset
      -6,                           // labX       
      ddBoxCENTER,                  // exType:    expansion type
      scDATA_ITEMS,                 // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    index of initial highlighted element
      monoColor,                    // scrColor:  color-attribute list
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[4],                       // nextCtrl:  link in next structure
   },
   //* This control 'expands' downward but must be dynamically resized by the  *
   //* constructor to avoid extending to (or beyond) the border of the dialog  *
   //* window.                                                                 *
   { //* Dropdown control D   - - - - - - - - - - - - - - - - - - - -    T9ddD *
      dctDROPDOWN,                  // type:      define a drop-down control
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      10,                           // ulY:       upper left corner in Y
      3,                            // ulX:       upper left corner in X
      short(sdDATA_ITEMS + 2),      // lines:     control lines
      short(sdITEM_LEN + 1),        // cols:      control columns
      (const char*)&sdboxData,      // dispText:  text-data array
      nc.cyR,                       // nColor:    non-focus border color
      nc.gr,                        // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "Box ^D",                     // label:     label text
      -1,                           // labY:      label offset
      1,                            // labX       
      ddBoxDOWN,                    // exType:    expansion type
      sdDATA_ITEMS,                 // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    index of initial highlighted element
      monoColor,                    // scrColor:  color-attribute list
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[5],                       // nextCtrl:  link in next structure
   },
   //* This control 'expands' upward and fits within the dialog window         *
   //* without the need for the constructor to re-size it, but note that the   *
   //* number of lines specified for the control is less than the number of    *
   //* data items, so not all items are visible at the same time.              *
   { //* Dropdown control E   - - - - - - - - - - - - - - - - - - - -    T9ddE *
      dctDROPDOWN,                  // type:      define a drop-down control
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      13,                           // ulY:       upper left corner in Y
      19,                           // ulX:       upper left corner in X
      short(seDATA_ITEMS - 2),      // lines:     control lines
      short(seITEM_LEN + 1),        // cols:      control columns
      (const char*)&seboxData,      // dispText:  text-data array
      nc.cyR,                       // nColor:    non-focus border color
      nc.gr,                        // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "Box ^E",                     // label:     label text
      3,                            // labY:      label offset
      1,                            // labX       
      ddBoxUP,                      // exType:    expansion type
      seDATA_ITEMS,                 // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    index of initial highlighted element
      monoColor,                    // scrColor:  color-attribute list
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[6],                       // nextCtrl:  link in next structure
   },
   //* Expansion of this control is centered on the base line specified (ulY)  *
   //* and fits within the dialog window without the need for the constructor  *
   //* to re-size it.                                                          *
   { //* Dropdown control F   - - - - - - - - - - - - - - - - - - - -    T9ddF *
      dctDROPDOWN,                  // type:      define a drop-down control
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      8,                            // ulY:       upper left corner in Y
      42,                           // ulX:       upper left corner in X
      short(sfDATA_ITEMS + 2),      // lines:     control lines
      short(sfITEM_LEN + 1),        // cols:      control columns
      (const char*)&sfboxData,      // dispText:  text-data array
      nc.cyR,                       // nColor:    non-focus border color
      nc.gr,                        // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "Box ^F",                     // label:     label text
      1,                            // labY:      label offset
      -6,                           // labX       
      ddBoxCENTER,                  // exType:    expansion type
      sfDATA_ITEMS,                 // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    index of initial highlighted element
      monoColor,                    // scrColor:  color-attribute list
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[7],                       // nextCtrl:  link in next structure
   },
   //* Expansion of this control is centered on the base line specified (ulY). *
   //* This control is dynamically resized by the constructor to avoid         *
   //* extending to (or beyond) the border of the dialog window.               *
   { //* Dropdown control G   - - - - - - - - - - - - - - - - - - - -    T9ddG *
      dctDROPDOWN,                  // type:      define a drop-down control
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      11,                           // ulY:       upper left corner in Y
      42,                           // ulX:       upper left corner in X
      short(sgDATA_ITEMS + 2),      // lines:     control lines
      short(sgITEM_LEN + 1),        // cols:      control columns
      (const char*)&sgboxData,      // dispText:  text-data array
      nc.cyR,                       // nColor:    non-focus border color
      nc.gr,                        // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "Box ^G",                     // label:     label text
      1,                            // labY:      label offset
      -6,                           // labX       
      ddBoxCENTER,                  // exType:    expansion type
      sgDATA_ITEMS,                 // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    index of initial highlighted element
      monoColor,                    // scrColor:  color-attribute list
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[8],                       // nextCtrl:  link in next structure
   },
   { //* 'DUMP UI' pushbutton  - - - - - - - - - - - - - - - - - -  T9dumpPush *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 4),        // ulY:       upper left corner in Y
      40,                           // ulX:       upper left corner in X
      1,                            // lines:     control lines
      17,                           // cols:      control columns
      dumpitStrings[0],             // dispText:  
      nc.brR,                       // nColor:    non-focus color
      nc.grR,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   },
} ;


   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       "  Drop-Down Controls  ", // dialog title
                       ncltSINGLE,     // border line-style
                       bColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;
   //Test09Dialog = dp ;              // give callback method access to the dialog

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Mark our territory (no, not with cat urine) *
      dp->WriteString ( 2, 0, "----5----0----5----0----5----"
                  "0----5----0----5----0----5----0", bColor ) ;
      gString  gsMark ;
      for ( short i = ZERO ; i < dialogROWS ; i++ )
      {
         gsMark.compose( L"%02hd", &i ) ;
         dp->WriteString ( i, 0, gsMark, bColor ) ; 
         dp->WriteString ( i, 58, gsMark, bColor ) ; 
      }

      dp->RefreshWin () ;                 // Display the results

      //* Establish a call-back method that can be called from within the *
      //* NcDialog code. Called each time through a control's user-input  *
      //* loop so we can manually modify the controls when necessary.     *
      // NOTE: We don't actually need a callback method for this test, but the 
      //       callback exists - we just don't register it with the dialog 
      //       because we don't use it at this time.
      //dp->EstablishCallback ( &T09_ControlUpdate ) ;

      //* Allow user to play with the controls.      *
      //* (Please see the notes in NcDialog.hpp on   *
      //* values returned (from the editing routines)*
      uiInfo Info ;                 // user interface data returned here
      short  icIndex = ZERO ;       // index of control with input focus
      bool   uidumpEnabled = false, // enable user-interface info dump
             done = false ;         // loop control
      while ( ! done )
      {
         //*******************************************
         //* If focus is currently on a Pushbutton   *
         //*******************************************
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            //* Get user input for this control                            *
            //* Returns when the Pushbutton is pressed OR when the user    *
            //* requests that the input focus moves to another control.    *
            if ( Info.viaHotkey == false )
            {
               icIndex = dp->EditPushbutton ( Info ) ;
               if ( uidumpEnabled )
               {  // To view contents of Info
                  //* Note that the coordinates for display are absolute 
                  //* screen coordinates, not dialog offsets
                  winPos wPos(2,1) ;
                  dp->Dump_uiInfo ( wPos, Info ) ;
               }
            }
            //* If user selected the current control object via hotkey, we *
            //* don't need to call the input routine; but simply process   *
            //* the existing data.                                         *
            else
            {
               if ( uidumpEnabled )
               {  // To view contents of Info
                  winPos wPos(2,1) ;
                  dp->Dump_uiInfo ( wPos, Info ) ;
               }
               //* Information arrived in the Info.h_xxx fields.*
               //* Transfer it to the primary positions.        *
               //* Previous data in these fields has already    *
               //* been handled in prior loop iteration.        *
               Info.HotData2Primary () ;
               if ( uidumpEnabled )
               {  // To view contents of Info
                  winPos wPos(2,1) ;
                  dp->Dump_uiInfo ( wPos, Info ) ;
               }
            }

            //* If a Pushbutton was pressed *
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == T9donePush ) // request to exit the dialog
                  done = true ;
               else if ( Info.ctrlIndex == T9dumpPush )
               {
                  uidumpEnabled = (uidumpEnabled ? false : true) ;
                  dp->SetPushbuttonText ( T9dumpPush, dumpitStrings[uidumpEnabled] ) ;
               }
               // (no other pushbuttons defined in this dialog)
            }
            else
            {
               // No button press, so nothing to do
            }
         }

         //***********************************************
         //* If focus is currently on a Dropdown control *
         //***********************************************
         else if ( ic[icIndex].type == dctDROPDOWN )
         {
            //* It doesn't matter how this control gained the input    *
            //* focus, by Tab/ShiftTab or via hotkey. If by hotkey, no *
            //* processing was done anyway, so discard any hotkey data *
            //*                                                        *
            //* However, if you want the control to expand immediately *
            //* when EditDropdown() is called, then set                *
            //* Info.viaHotkey==true. Otherwise, the control will      *
            //* expand when the user presses Enter or one of the       *
            //* vertical scrolling keys. This is useful if the user    *
            //* arrived at the current control via hotkey because by   *
            //* selecting the control via hotkey, he/she has already   *
            //* demonstrated the intention to edit the control.        *
            //* (all other data in the Info class is ignored on entry) *
            //Info.viaHotkey = false ;

            //* Allow user to modify which member of the Dropdown      *
            //* is selected.  Returns when edit is complete.           *
            icIndex = dp->EditDropdown ( Info ) ;
            if ( uidumpEnabled )
            {  // To view contents of Info
               //* Note that the coordinates for display are absolute 
               //* screen coordinates, not dialog offsets
               winPos wPos(2,1) ;
               dp->Dump_uiInfo ( wPos, Info ) ;
            }

            //* If 'selected' member of Dropdown has changed *
            if ( Info.dataMod != false || Info.keyIn == nckENTER )
            {
               switch ( Info.ctrlIndex )
               {
                  case T9ddA:
                     // DO STUFF
                     break ;
                  case T9ddB:
                     //* Test whether the controls move smoothly *
                     //* with the rest of the dialog window.     *
                     if ( Info.dataMod != false )
                     {
                        if ( Info.selMember == 8 )
                        {
                           winPos newPos(3, 3) ;
                           dp->MoveWin ( newPos ) ;
                        }
                        else if ( Info.selMember == 9 )
                        {
                           winPos newPos(ulY, ulX) ;
                           dp->MoveWin ( newPos ) ;
                        }
                     }
                     else
                        { /* do nothing */ }
                     break ;
                  case T9ddC:
                     // DO STUFF
                     break ;
                  case T9ddD:
                     // DO STUFF
                     break ;
                  case T9ddE:
                     // DO STUFF
                     break ;
                  case T9ddF:
                     // DO STUFF
                     break ;
                  case T9ddG:
                     // DO STUFF
                     break ;
                  default:
                     // no other Dropdown controls defined for this dialog
                     break ;
               }
            }
         }     // dctDROPDOWN

         //*******************************************
         //* If focus is currently on a Textbox      *
         //*******************************************
         else if ( ic[icIndex].type == dctTEXTBOX )
            { /* there are no Textbox controls defined for this dialog */ }
         //*******************************************
         //* If focus is currently on a Radiobutton  *
         //*******************************************
         else if ( ic[icIndex].type == dctRADIOBUTTON )
            { /* there are no Radiobutton controls defined for this dialog */ }
         //*******************************************
         //* If focus is currently on a Scrollbox    *
         //*******************************************
         else if ( ic[icIndex].type == dctSCROLLBOX )
            { /* there are no Scrollbox controls defined for this dialog */ }
         //*******************************************
         //* If focus is currently on a Scrollext    *
         //*******************************************
         else if ( ic[icIndex].type == dctSCROLLEXT )
            { /* there are no Scrollext controls defined for this dialog */ }
         //*******************************************
         //* If focus is currently on a Menuwin      *
         //*******************************************
         else if ( ic[icIndex].type == dctMENUWIN )
            { /* there are no Menuwin controls defined for this dialog */ }
         //*******************************************
         //* If focus is currently on a Spinner      *
         //*******************************************
         else if ( ic[icIndex].type == dctSPINNER )
            { /* there are no Spinner controls defined for this dialog */ }

         //* If user exited the control edit method via a hotkey,*
         //* then the new control already has focus. Otherwise,  *
         //* move input focus to next/previous control.          *
         if ( done == false && Info.viaHotkey == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }  // while()
   }  // OpenWindow()
   else
   {  //* Most likely cause of dialog not opening is that the terminal window  *
      //* is too small. Give user a clue...                                    *
      short neededLines = (dialogROWS + minTestULY) >= minTERMROWS ?
                          (dialogROWS + minTestULY) : minTERMROWS, 
            neededCols  = dialogCOLS >= minTERMCOLS ? dialogCOLS : minTERMCOLS ;
      gString gs ;
      gs.compose ( L" Sorry, size of terminal window must be at least %hd x %hd for this test. ", 
                   &neededCols, &neededLines ) ;
      StatWin ( gs.gstr(), nc.cyG ) ;
      sleep ( 5 ) ;
   }

   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

}  //* End Test09() *

//*************************
//*  T09_ControlUpdate    *
//*************************
//******************************************************************************
//* This is a callback method for manually updating the controls in the        *
//* Test09() dialog.                                                           *
//*                                                                            *
//*  For this test, the following are updated by the callback method:          *
//*   1. Nothing at this time....                                              *
//*   2.                                                                       *
//*   3.                                                                       *
//*                                                                            *
//* Input  : currIndex: index of control that currently has focus              *
//*          wkey     : user's key input data                                  *
//*          firstTime: the EstablishCallback() method calls this method once  *
//*                     with firstTime==true, to perform any required          *
//*                     initialization. Subsequently, the NcDialog class       *
//*                     always calls with firstTime==false.                    *
//* Returns: OK                                                                *
//******************************************************************************
//* Important Note: This method makes some intimate assumptions about what     *
//* is happening in the method that established the callback. If changes are   *
//* made in the establishing method that affect the callback functionality,    *
//* be sure to update this method to address those changes.                    *
//*                                                                            *
//* Programmer's Note: See dialog control index constants at file scope above. *
//******************************************************************************

short T09_ControlUpdate ( const short currIndex, const wkeyCode wkey, bool firstTime )
{

   //*************************
   //* First-time processing *
   //*************************
   if ( firstTime != false )
   {
      //* Initialize the tracking data *
      // nothing to do at this time...
   }

   //***********************
   //* Standard processing *
   //***********************
   // nothing to do at this time...

   return OK ;

}  //* End T09_ControlUpdate() *

//*************************
//*       Test10          *
//*************************
//******************************************************************************
//* This dialog tests the functionality of the dctMENUWIN control.             *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Accessing the menu controls:                                               *
//* The Menu titles are visible initially, with the menus themselves closed.   *
//* When the user accesses a Menu via hotkey or by tabbing to it, the          *
//* application will then call the EditMenuwin() method which will open the    *
//* menu (just below the title) and will allow the user to make a selection    *
//* from the menu.                                                             *
//*                                                                            *
//* Within the menu, selection may occur in one of two ways:                   *
//*  1. User may select a menu item by scrolling through the items and         *
//*     selecting an item via the Enter key, OR                                *
//*  2. User may select a menu item via the hotkeys specified (if any) in      *
//*     the menu items.                                                        *
//*     Note that the hotkeys within the menu are not the same as the hotkeys  *
//*     defined for accessing individual dialog controls. The menu hotkeys     *
//*     are visible and accessible ONLY when the menu is open i.e. inside      *
//*     the EditMenuwin() method.                                              *
//*                                                                            *
//* If no label is specified for the control, there will be no visible         *
//* control for the dctMENUWIN object in the dialog. This means that the       *
//* application loop will have to define the method for making the dctMENUWIN  *
//* control visible. This implements a 'context menu'.                         *
//*                                                                            *
//******************************************************************************

void Test10 ( void )
{
static const short dialogROWS = 22 ;      // display lines
static const short dialogCOLS = 78 ;      // display columns
static const short controlsDEFINED = 23 ; // # of controls in dialog
const short minNeededLines = 34,          // min hight for ACS window display
            minNeededCols  = 90 ;         // min width for ACS window display
static attr_t monoColorG[2]  =  { attrDFLT, nc.grR } ; // scroll box data colors
static attr_t monoColorB[2]  =  { attrDFLT, nc.blR } ;
static attr_t monostColor[2] =  { attrDFLT, nc.grR|ncbATTR } ;
static attr_t multiColor[16] =  { nc.bw, nc.re, nc.gr, nc.br, 
                                  nc.bl, nc.ma, nc.cy, nc.gy, 
                                  nc.bw, nc.re, nc.gr, nc.br, 
                                  nc.bl, nc.ma, nc.cy, nc.gy } ;
static attr_t multiRColor[16] = { nc.bwR, nc.reR, nc.grR, nc.brR, 
                                  nc.blR, nc.maR, nc.cyR, nc.gyR, 
                                  nc.bwR, nc.reR, nc.grR, nc.brR, 
                                  nc.blR, nc.maR, nc.cyR, nc.gyR } ;
//* Data for Menuwin A, T10mwA *
const short saDATA_ITEMS = 10, saITEM_LEN = 18 ;
const char saboxData[saDATA_ITEMS][saITEM_LEN] =   
{
   "^File Commands...",
   "^Sort by...   C+O",
   "F^avorites... C+F",
   "^View File... ENT",
   "^Mount...     C+Y",
   "Shell Ou^t    A+Z",
   "^Configure...    ",
   "Misc^ellaneous...",
   "FileMgr ^Help F01",
   "E^xit FileMgr C+Q"
} ;
const char saaltData[saDATA_ITEMS][saITEM_LEN] =   
{
   "^Fly Commands ...",
   "^Slop by...   C+O",
   "F^anout Boys  C+F",
   "^View Flop... ENT",
   "^Mould...     C+Y",
   "Shut Ou^t     A+Z",
   "^Confligrate  ...",
   "Misc^ing You  ...",
   "FileMgr ^Hope F01",
   "E^xit Gladly  C+Q"
} ;

//* Data for Menuwin B, T10mwB *
const short sbDATA_ITEMS = 10, sbITEM_LEN = 18 ;
const char sbboxData[sbDATA_ITEMS][sbITEM_LEN] =   
{
   "^Select File    >",
   "Select ^All      ",
   "C^opy Files      ",
   "C^ut Files      >",
   "^Paste Files     ",
   "PasteSpecia^l    ",
   "^Delete Files    ",
   "^Rename Files   >",
   "^Touch Files     ",
   "^NewDirectory   >"
} ;

//* Data for Menuwin C, T10mwC *
const short scDATA_ITEMS = 5, scITEM_LEN = 18 ;
const char scboxData[scDATA_ITEMS][scITEM_LEN] =   
{
   " Charming ^Mom   ",
   " ^Housecleaning  ",
   " Missing ^Link...",
   " Trust Invectiv^e",
   "Slant ^Options..."
} ;

//* Data for invisible (context) Menuwin, T10mwI *
const short siDATA_ITEMS = 3, siITEM_LEN = 18 ;
const char siboxData[siDATA_ITEMS][siITEM_LEN] =   
{
   "  File ^Stats    ",
   "  ^View as Text  ",
   "  ^Execute       "
} ;

//* Data for sub-menu A, T10cmA *
const short caDATA_ITEMS = 4, caITEM_LEN = 18 ;
const char caboxData[caDATA_ITEMS][caITEM_LEN] =   
{
   " Charlie Fil^e   ",
   " Gertrude Fi^le  ",
   " Bascombe F^ile  ",
   " Dingbats ^File  "
} ;

//* Data for sub-menu B, T10cmB *
const short cbDATA_ITEMS = 4, cbITEM_LEN = 18 ;
const char cbboxData[cbDATA_ITEMS][cbITEM_LEN] =   
{
   " Cut ^squares    ",
   " Cut ^triangles  ",
   " Cut ^rhomboids  ",
   " Cut ^cylinders  "
} ;

//* Data for sub-menu C, T10cmC *
const short ccDATA_ITEMS = 4, ccITEM_LEN = 18 ;
const char ccboxData[ccDATA_ITEMS][ccITEM_LEN] =   
{
   " ^Name as Fred   ",
   " N^ame as Gracie ",
   " Name as ^Red   >",
   " Nam^e as Phil   "
} ;

//* Data for sub-menu D, T10cmD *
const short cdDATA_ITEMS = 4, cdITEM_LEN = 18 ;
//* Text data are passed to the Menuwin constructor in a      *
//* variety of formats, to stress-test the parsing algorithm. *
#if 1
const char cdboxData[cdDATA_ITEMS][cdITEM_LEN] =   
{  // Full-width items - exact width for target control
   " Direct 2 ^Paris ",
   " Direct 2 ^Tahoe ",
   " Direct 2 ^Ocala ",
   " Direct 2 ^Dyson "
} ;
#elif 0
const char cdboxData[cdDATA_ITEMS][cdITEM_LEN] =   
{  // Mixed-width items <= to required length (will be padded on right)
   " Off 2 ^Paris ",
   " On 2 ^Tahoe ",
   " Out 2 ^Ocala ",
   " Into ^Dyson "
} ;
#else
const char cdboxData[] =   
{  // Single data string - will be divided into correct lengths
   " Direct 2 ^Paris "" Direct 2 ^Tahoe "" Direct 2 ^Ocala "
   " Direct 2 ^Dyson ""discarded-crap"
} ;
#endif

//* Data for sub-menu E, T10cmE *
const short ceDATA_ITEMS = 4, ceITEM_LEN = 18 ;
const char ceboxData[ceDATA_ITEMS][ceITEM_LEN] =   
{
   " Red ^Skelton    ",
   " Red ^Buttons    ",
   " Red ^Ryder      ",
   " Red ^Adair      "
} ;

//* Data for stand-alone (move-win) menu, T10mwM *
const short smDATA_ITEMS = 5, smITEM_LEN = 18 ;
const char smboxData[smDATA_ITEMS][smITEM_LEN] =   
{
   "  Up^per Left    ",    // item 0
   "  Upp^er Right   ",    // item 1
   "  Lowe^r Left    ",    // item 2
   "  Lo^wer Right   ",    // item 3
   "  Centere^d      "     // item 4
} ;
//* Data for stand-alone (Grandchildren) menu, T10mwG *
// (Note that we declare our array element length as one less than)
// (the defined item width because unlike other menu-item arrays  )
// (in this test, these strings do not include hotkey indicators. )
const short grDATA_ITEMS = 3, grITEM_LEN = 18 ;
//* Text data are passed to the Menuwin constructor in a      *
//* variety of formats, to stress-test the parsing algorithm. *
#if 1
const char grboxData[grDATA_ITEMS][grITEM_LEN-1] = // 'Grand-children'
{  // Full-width items - exact width for target control
    " on your knee  >",
    " in your hair   ",
    " up your nose   ",
} ;
#elif 0
const char grboxData[grDATA_ITEMS][grITEM_LEN-1] = // 'Grand-children'
{  // Mixed-width items <= to required length (will be padded on right)
    " on your knee  >",
    " in your hair ",
    " in a nostril",
} ;
#else
const char grboxData[] = // 'Grand-children'
{  // Single data string - will be divided into correct lengths
    " on your knee  >"" in your hair   "" up your nose   ""discarded-crap"
} ;
#endif
//* Data for sub-menus of Grandchildren menu, T10cmG2 - T10cmG6 *
const short gcDATA_ITEMS = 3, gcITEM_LEN = 11 ;
const char gcboxData[gcDATA_ITEMS][gcITEM_LEN-1] = 
{
   "  Vera  >",
   "  Chuck  ",
   "  Dave   ",
} ;
const char jpgrData[gcDATA_ITEMS+1][gcITEM_LEN-1] = 
{
   "  John   ",
   "  Paul   ",
   "  George ",
   "  Ringo  ",
} ;

//* Data arrays for the active/inactive menu item test *
bool mwA_allActive[saDATA_ITEMS] = 
{ true, true, true, true, true, true, true, true, true, true } ;
attr_t mwA_allColor[saDATA_ITEMS] = 
{ monoColorG[1], monoColorG[1], monoColorG[1], monoColorG[1], monoColorG[1], 
  monoColorG[1], monoColorG[1], monoColorG[1], monoColorG[1], monoColorG[1] } ;
bool mwA_someInactive[saDATA_ITEMS] = 
{ true, true, true, false, true, true, true, false, true, true } ;
attr_t mwA_someColor[saDATA_ITEMS] = 
{ monoColorG[1], monoColorG[1], monoColorG[1], nc.gyR, monoColorG[1], 
  monoColorG[1], monoColorG[1], nc.gyR, monoColorG[1], monoColorG[1] } ;

//* Specifies a list of (y,x) positions for *
//* the dialog within the terminal window   *
winPos winPosList[smDATA_ITEMS] 
{
   winPos( 2,  2 ),                          // Upper Left
   winPos( 2, termCols-dialogCOLS-2 ),       // Upper Right
   winPos( termRows-dialogROWS-1,  2 ),     // Lower Left
   winPos( termRows-dialogROWS-1, termCols-dialogCOLS-2 ), // Lower Right
   winPos( termRows/2-dialogROWS/2, termCols/2-dialogCOLS/2 ), // Centered
} ;           
short dialogPos = 4 ;    // initial dialog position (Centered)


attr_t   dColor = nc.cyR,                   // dialog's interior color
         bColor = nc.blR ;                  // dialog's border color
short    ctrY   = termRows/2,               // terminal window center in Y
         ctrX   = termCols/2,               // terminal window center in X
         //* Upper left corner in Y (cannot obscure status window) *
         ulY    = (ctrY - dialogROWS/2 - 4) >= minTestULY ?
                  (ctrY - dialogROWS/2 - 4) : minTestULY, 
         ulX     = ctrX - dialogCOLS/2 ;     // upper left corner in X

   StatWin ( "Opening Test Dialog #10 - Menu Window Test" ) ;

InitCtrl ic[controlsDEFINED] =   // array of dialog control initialization objects
{
   {  //* 'DONE' pushbutton - - - - - - - - - - - - - - - - - - -  T10donePush *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 3),        // ulY:       upper left corner in Y
      short(dialogCOLS / 2 - 5),    // ulX:       upper left corner in X
      1,                            // lines:     control lines
      8,                            // cols:      control columns
      "  DO^NE  ",                  // dispText:  
      nc.gyR,                       // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[1],                       // nextCtrl:  link in next structure
   },
   {  //* Menuwin A (group member)- - - - - - - - - - - - - - - - - -   T10mwA *
      //* Notes for definining a dctMENUWIN control:
      //* - Internal calculations are performed by the control's constructor   *
      //*   for size and position of both the 'collapsed' and 'expanded'       *
      //*   control. The 'collapsed' control is positioned at the specified    *
      //*   ulY/ulX position. The 'expanded' control is positioned one display *
      //*   line below the specified ulY/ulX position.                         *
      //* - The label text specified is written as the display text for the    *
      //*   'collapsed' control. The display width of the label string (not    *
      //*   including '^' caret if any) determines the width of the 'collapsed'*
      //*   control. Note that it is generally prudent to have one space at    *
      //*   each end of the label string for visual effect.                    *
      //* - The 'cols' member determines the width of the 'expanded' control,  * 
      //*   i.e.the menu data display. 
      //*   Specify column width for the expanded control ('cols' member) as   *
      //*   number of visible character-columns PLUS left and right borders    *
      //*   but NOT including '^' hotkey indicator (if any), which is not a    *
      //*   visible character in display of the menu item.                     *
      //* - If no label is specified ('label' member == NULL), then there is   *
      //*   no visible 'collapsed' control (it is handled as a sub-menu or     *
      //*   context menu), and the position of the 'expanded' control is the   *
      //*   specified ulY/ulX.                                                 *
      dctMENUWIN,             // type:      
      rbtTYPES,               // rbSubtype: (n/a)
      false,                  // rbSelect:  (n/a)
      1,                      // ulY:       upper left corner in Y
      2,                      // ulX:       upper left corner in X
      1,                      // lines:     (n/a)
      saITEM_LEN,             // cols:      control columns, 'expanded' state
      (const char*)&saboxData,// dispText:  array of menu items
      nc.gr,         // nColor: non-focus border color / focus 'collapsed' color
      nc.grR,        // fColor: focus border color / non-focus 'collapsed' color
      tbPrint,                // filter:    (n/a)
      " Menuwin ^A ",         // label:     label text == 'collapsed state' text
      ZERO,                   // labY:      (n/a)
      ZERO,                   // labX       (n/a)
      ddBoxTYPES,             // exType:    (n/a)
      saDATA_ITEMS,           // scrItems:  number of menu items
      ZERO,                   // scrSel:    (n/a)
      monoColorG,             // scrColor:  array of color attributes for menu items
      NULL,                   // spinData:  (n/a)
      true,                   // active:    allow control to gain focus
      &ic[T10mwB],            // nextCtrl:  link in next structure
   },
   {  //* Menuwin B (group member)- - - - - - - - - - - - - - - - - -   T10mwB *
      dctMENUWIN,             // type:      
      rbtTYPES,               // rbSubtype: (n/a)
      false,                  // rbSelect:  (n/a)
      1,                      // ulY:       upper left corner in Y
      13,                     // ulX:       upper left corner in X
      1,                      // lines:     (n/a)
      sbITEM_LEN,             // cols:      control columns, 'expanded' state
      (const char*)&sbboxData,// dispText:  array of menu items
      nc.gr,         // nColor: non-focus border color / focus 'collapsed' color
      nc.grR,        // fColor: focus border color / non-focus 'collapsed' color
      tbPrint,                // filter:    (n/a)
      " Menuwin ^B ",         // label:     label text == 'collapsed state' text
      ZERO,                   // labY:      (n/a)
      ZERO,                   // labX       (n/a)
      ddBoxTYPES,             // exType:    (n/a)
      sbDATA_ITEMS,           // scrItems:  number of menu items
      ZERO,                   // scrSel:    (n/a)
      multiRColor,            // scrColor:  array of color attributes for menu items
      NULL,                   // spinData:  (n/a)
      true,                   // active:    allow control to gain focus
      &ic[T10mwC],            // nextCtrl:  link in next structure
   },
   {  //* Menuwin C (group member)- - - - - - - - - - - - - - - - - -   T10mwC *
      dctMENUWIN,             // type:      
      rbtTYPES,               // rbSubtype: (n/a)
      false,                  // rbSelect:  (n/a)
      1,                      // ulY:       upper left corner in Y
      24,                     // ulX:       upper left corner in X
      1,                      // lines:     (n/a)
      scITEM_LEN,             // cols:      control columns, 'expanded' state
      (const char*)&scboxData,// dispText:  array of menu items
      nc.gr,         // nColor: non-focus border color / focus 'collapsed' color
      nc.grR,        // fColor: focus border color / non-focus 'collapsed' color
      tbPrint,                // filter:    (n/a)
      " Menuwin ^C ",         // label:     label text == 'collapsed state' text
      ZERO,                   // labY:      (n/a)
      ZERO,                   // labX       (n/a)
      ddBoxTYPES,             // exType:    (n/a)
      scDATA_ITEMS,           // scrItems:  number of menu items
      ZERO,                   // scrSel:    (n/a)
      multiColor,             // scrColor:  array of color attributes for menu items
      NULL,                   // spinData:  (n/a)
      true,                   // active:    allow control to gain focus
      &ic[T10mwI],            // nextCtrl:  link in next structure
   },
   {  //* Menuwin (context menu, invisible when collapsed)  - - - - -   T10mwI *
      dctMENUWIN,             // type:      
      rbtTYPES,               // rbSubtype: (n/a)
      false,                  // rbSelect:  (n/a)
      5,                      // ulY:       upper left corner in Y
      50,                     // ulX:       upper left corner in X
      1,                      // lines:     (n/a)
      siITEM_LEN,             // cols:      control columns, 'expanded' state
      (const char*)&siboxData,// dispText:  array of menu items
      nc.gr,         // nColor: (n/a) context menus are invisible when collapsed
      nc.grR,        // fColor: focus border color / non-focus 'collapsed' color
      tbPrint,                // filter:    (n/a)
      NULL,                   // label:     hidden menus have no label
      ZERO,                   // labY:      (n/a)
      ZERO,                   // labX       (n/a)
      ddBoxTYPES,             // exType:    (n/a)
      siDATA_ITEMS,           // scrItems:  number of menu items
      ZERO,                   // scrSel:    (n/a)
      monoColorG,             // scrColor:  array of color attributes for menu items
      NULL,                   // spinData:  (n/a)
      false,                  // active:    initially inactive
      &ic[T10cmA],            // nextCtrl:  link in next structure
   },
   {  //* Menuwin (sub-menu A, invisible when collapsed)  - - - - - -   T10cmA *
      dctMENUWIN,             // type:      
      rbtTYPES,               // rbSubtype: (n/a)
      false,                  // rbSelect:  (n/a)
      short(ic[T10mwB].ulY + 1),                // ulY: upper left corner in Y
      short(ic[T10mwB].ulX + (sbITEM_LEN - 1)), // ulX: upper left corner in X
      1,                      // lines:     (n/a)
      caITEM_LEN,             // cols:      control columns, 'expanded' state
      (const char*)&caboxData,// dispText:  array of menu items
      nc.gr,         // nColor: (n/a) sub-menus are invisible when collapsed
      nc.grR,        // fColor: focus border color / non-focus 'collapsed' color
      tbPrint,                // filter:    (n/a)
      NULL,                   // label:     hidden menus have no label
      ZERO,                   // labY:      (n/a)
      ZERO,                   // labX       (n/a)
      ddBoxTYPES,             // exType:    (n/a)
      caDATA_ITEMS,           // scrItems:  number of menu items
      ZERO,                   // scrSel:    (n/a)
      monoColorG,             // scrColor:  array of color attributes for menu items
      NULL,                   // spinData:  (n/a)
      false,                  // active:    initially inactive
      &ic[T10cmB],            // nextCtrl:  link in next structure
   },
   {  //* Menuwin (sub-menu B, invisible when collapsed)  - - - - - -   T10cmB *
      dctMENUWIN,             // type:      
      rbtTYPES,               // rbSubtype: (n/a)
      false,                  // rbSelect:  (n/a)
      short(ic[T10mwB].ulY + 4),                // ulY: upper left corner in Y
      short(ic[T10mwB].ulX + (sbITEM_LEN - 1)), // ulX: upper left corner in X
      1,                      // lines:     (n/a)
      cbITEM_LEN,             // cols:      control columns, 'expanded' state
      (const char*)&cbboxData,// dispText:  array of menu items
      nc.gr,         // nColor: (n/a) sub-menus are invisible when collapsed
      nc.grR,        // fColor: focus border color / non-focus 'collapsed' color
      tbPrint,                // filter:    (n/a)
      NULL,                   // label:     hidden menus have no label
      ZERO,                   // labY:      (n/a)
      ZERO,                   // labX       (n/a)
      ddBoxTYPES,             // exType:    (n/a)
      cbDATA_ITEMS,           // scrItems:  number of menu items
      ZERO,                   // scrSel:    (n/a)
      monoColorG,             // scrColor:  array of color attributes for menu items
      NULL,                   // spinData:  (n/a)
      false,                  // active:    initially inactive
      &ic[T10cmC],            // nextCtrl:  link in next structure
   },
   {  //* Menuwin (sub-menu C, invisible when collapsed)  - - - - - -   T10cmC *
      dctMENUWIN,             // type:      
      rbtTYPES,               // rbSubtype: (n/a)
      false,                  // rbSelect:  (n/a)
      short(ic[T10mwB].ulY + 8),                // ulY: upper left corner in Y
      short(ic[T10mwB].ulX + (sbITEM_LEN - 1)), // ulX: upper left corner in X
      1,                      // lines:     (n/a)
      ccITEM_LEN,             // cols:      control columns, 'expanded' state
      (const char*)&ccboxData,// dispText:  array of menu items
      nc.gr,         // nColor: (n/a) sub-menus are invisible when collapsed
      nc.grR,        // fColor: focus border color / non-focus 'collapsed' color
      tbPrint,                // filter:    (n/a)
      NULL,                   // label:     hidden menus have no label
      ZERO,                   // labY:      (n/a)
      ZERO,                   // labX       (n/a)
      ddBoxTYPES,             // exType:    (n/a)
      ccDATA_ITEMS,           // scrItems:  number of menu items
      ZERO,                   // scrSel:    (n/a)
      monostColor,            // scrColor:  array of color attributes for menu items
      NULL,                   // spinData:  (n/a)
      false,                  // active:    initially inactive
      &ic[T10cmD],            // nextCtrl:  link in next structure
   },
   {  //* Menuwin (sub-menu D, invisible when collapsed)  - - - - - -   T10cmD *
      dctMENUWIN,             // type:      
      rbtTYPES,               // rbSubtype: (n/a)
      false,                  // rbSelect:  (n/a)
      short(ic[T10mwB].ulY + 10),               // ulY: upper left corner in Y
      short(ic[T10mwB].ulX + (sbITEM_LEN - 1)), // ulX: upper left corner in X
      1,                      // lines:     (n/a)
      cdITEM_LEN,             // cols:      control columns, 'expanded' state
      (const char*)&cdboxData,// dispText:  array of menu items
      nc.gr,         // nColor: (n/a) sub-menus are invisible when collapsed
      nc.grR,        // fColor: focus border color / non-focus 'collapsed' color
      tbPrint,                // filter:    (n/a)
      NULL,                   // label:     hidden menus have no label
      ZERO,                   // labY:      (n/a)
      ZERO,                   // labX       (n/a)
      ddBoxTYPES,             // exType:    (n/a)
      cdDATA_ITEMS,           // scrItems:  number of menu items
      ZERO,                   // scrSel:    (n/a)
      multiRColor,            // scrColor:  array of color attributes for menu items
      NULL,                   // spinData:  (n/a)
      false,                  // active:    initially inactive
      &ic[T10cmE],            // nextCtrl:  link in next structure
   },
   {  //* Menuwin (sub-sub-menu, invisible when collapsed)  - - - - -   T10cmE *
      dctMENUWIN,             // type:      
      rbtTYPES,               // rbSubtype: (n/a)
      false,                  // rbSelect:  (n/a)
      short(ic[T10cmC].ulY + 2),                // ulY: upper left corner in Y
      short(ic[T10cmC].ulX + (ccITEM_LEN - 1)), // ulX: upper left corner in X
      1,                      // lines:     (n/a)
      ceITEM_LEN,             // cols:      control columns, 'expanded' state
      (const char*)&ceboxData,// dispText:  array of menu items
      nc.gr,         // nColor: (n/a) sub-menus are invisible when collapsed
      nc.grR,        // fColor: focus border color / non-focus 'collapsed' color
      tbPrint,                // filter:    (n/a)
      NULL,                   // label:     hidden menus have no label
      ZERO,                   // labY:      (n/a)
      ZERO,                   // labX       (n/a)
      ddBoxTYPES,             // exType:    (n/a)
      ceDATA_ITEMS,           // scrItems:  number of menu items
      ZERO,                   // scrSel:    (n/a)
      monostColor,            // scrColor:  array of color attributes for menu items
      NULL,                   // spinData:  (n/a)
      false,                  // active:    initially inactive
      &ic[T10mwG],            // nextCtrl:  link in next structure
   },
   {  //* Menuwin (stand-alone, initially collapsed)  - - - - - - - -   T10mwG *
      dctMENUWIN,             // type:      
      rbtTYPES,               // rbSubtype: (n/a)
      false,                  // rbSelect:  (n/a)
      7,                      // ulY: upper left corner in Y
      2,                      // ulX: upper left corner in X
      1,                      // lines:     (n/a)
      grITEM_LEN,             // cols:      control columns, 'expanded' state
      (const char*)&grboxData,// dispText:  array of menu items
      nc.bl,         // nColor: non-focus border color / focus 'collapsed' color
      nc.blR,        // fColor: focus border color / non-focus 'collapsed' color
      tbPrint,                // filter:    (n/a)
      "  ^Grand-children  ",  // label:     label text == 'collapsed state' text
      ZERO,                   // labY:      (n/a)
      ZERO,                   // labX       (n/a)
      ddBoxTYPES,             // exType:    (n/a)
      grDATA_ITEMS,           // scrItems:  number of menu items
      ZERO,                   // scrSel:    (n/a)
      monoColorB,             // scrColor:  array of color attributes for menu items
      NULL,                   // spinData:  (n/a)
      true,                   // active:    initially active
      &ic[T10cmG2],           // nextCtrl:  link in next structure
   },
   {  //* Menuwin (sub-menu G2) - - - - - - - - - - - - - - - - - -    T10cmG2 *
      dctMENUWIN,             // type:      
      rbtTYPES,               // rbSubtype: (n/a)
      false,                  // rbSelect:  (n/a)
      short(ic[T10mwG].ulY - 1),// ulY: upper left corner in Y
      short(ic[T10mwG].ulX + (grITEM_LEN - 1)),// ulX: upper left corner in X
      1,                      // lines:     (n/a)
      gcITEM_LEN,             // cols:      control columns, 'expanded' state
      (const char*)&gcboxData,// dispText:  array of menu items
      nc.bl,         // nColor: (n/a) sub-menus are invisible when collapsed
      nc.blR,        // fColor: focus border color / non-focus 'collapsed' color
      tbPrint,                // filter:    (n/a)
      NULL,                   // label:     hidden menus have no label
      ZERO,                   // labY:      (n/a)
      ZERO,                   // labX       (n/a)
      ddBoxTYPES,             // exType:    (n/a)
      gcDATA_ITEMS,           // scrItems:  number of menu items
      ZERO,                   // scrSel:    (n/a)
      monoColorB,             // scrColor:  array of color attributes for menu items
      NULL,                   // spinData:  (n/a)
      false,                  // active:    initially inactive
      &ic[T10cmG3],           // nextCtrl:  link in next structure
   },
   {  //* Menuwin (sub-menu G3) - - - - - - - - - - - - - - - - - -    T10cmG3 *
      dctMENUWIN,             // type:      
      rbtTYPES,               // rbSubtype: (n/a)
      false,                  // rbSelect:  (n/a)
      short(ic[T10cmG2].ulY + 1),// ulY: upper left corner in Y
      short(ic[T10cmG2].ulX + (gcITEM_LEN - 1)),// ulX: upper left corner in X
      1,                      // lines:     (n/a)
      gcITEM_LEN,             // cols:      control columns, 'expanded' state
      (const char*)&gcboxData,// dispText:  array of menu items
      nc.bl,         // nColor: (n/a) sub-menus are invisible when collapsed
      nc.blR,        // fColor: focus border color / non-focus 'collapsed' color
      tbPrint,                // filter:    (n/a)
      NULL,                   // label:     hidden menus have no label
      ZERO,                   // labY:      (n/a)
      ZERO,                   // labX       (n/a)
      ddBoxTYPES,             // exType:    (n/a)
      gcDATA_ITEMS,           // scrItems:  number of menu items
      ZERO,                   // scrSel:    (n/a)
      monoColorB,             // scrColor:  array of color attributes for menu items
      NULL,                   // spinData:  (n/a)
      false,                  // active:    initially inactive
      &ic[T10cmG4],           // nextCtrl:  link in next structure
   },
   {  //* Menuwin (sub-menu G4) - - - - - - - - - - - - - - - - - -    T10cmG4 *
      dctMENUWIN,             // type:      
      rbtTYPES,               // rbSubtype: (n/a)
      false,                  // rbSelect:  (n/a)
      short(ic[T10cmG3].ulY + 1),// ulY: upper left corner in Y
      short(ic[T10cmG3].ulX + (gcITEM_LEN - 1)),// ulX: upper left corner in X
      1,                      // lines:     (n/a)
      gcITEM_LEN,             // cols:      control columns, 'expanded' state
      (const char*)&gcboxData,// dispText:  array of menu items
      nc.bl,         // nColor: (n/a) sub-menus are invisible when collapsed
      nc.blR,        // fColor: focus border color / non-focus 'collapsed' color
      tbPrint,                // filter:    (n/a)
      NULL,                   // label:     hidden menus have no label
      ZERO,                   // labY:      (n/a)
      ZERO,                   // labX       (n/a)
      ddBoxTYPES,             // exType:    (n/a)
      gcDATA_ITEMS,           // scrItems:  number of menu items
      ZERO,                   // scrSel:    (n/a)
      monoColorB,             // scrColor:  array of color attributes for menu items
      NULL,                   // spinData:  (n/a)
      false,                  // active:    initially inactive
      &ic[T10cmG5],           // nextCtrl:  link in next structure
   },
   {  //* Menuwin (sub-menu G5) - - - - - - - - - - - - - - - - - -    T10cmG5 *
      dctMENUWIN,             // type:      
      rbtTYPES,               // rbSubtype: (n/a)
      false,                  // rbSelect:  (n/a)
      short(ic[T10cmG4].ulY + 1),// ulY: upper left corner in Y
      short(ic[T10cmG4].ulX + (gcITEM_LEN - 1)),// ulX: upper left corner in X
      1,                      // lines:     (n/a)
      gcITEM_LEN,             // cols:      control columns, 'expanded' state
      (const char*)&gcboxData,// dispText:  array of menu items
      nc.bl,         // nColor: (n/a) sub-menus are invisible when collapsed
      nc.blR,        // fColor: focus border color / non-focus 'collapsed' color
      tbPrint,                // filter:    (n/a)
      NULL,                   // label:     hidden menus have no label
      ZERO,                   // labY:      (n/a)
      ZERO,                   // labX       (n/a)
      ddBoxTYPES,             // exType:    (n/a)
      gcDATA_ITEMS,           // scrItems:  number of menu items
      ZERO,                   // scrSel:    (n/a)
      monoColorB,             // scrColor:  array of color attributes for menu items
      NULL,                   // spinData:  (n/a)
      false,                  // active:    initially inactive
      &ic[T10cmG6],           // nextCtrl:  link in next structure
   },
   {  //* Menuwin (sub-menu G6) - - - - - - - - - - - - - - - - - -    T10cmG6 *
      dctMENUWIN,             // type:      
      rbtTYPES,               // rbSubtype: (n/a)
      false,                  // rbSelect:  (n/a)
      short(ic[T10cmG5].ulY + 1),// ulY: upper left corner in Y
      short(ic[T10cmG5].ulX + (gcITEM_LEN - 1)),// ulX: upper left corner in X
      1,                      // lines:     (n/a)
      gcITEM_LEN,             // cols:      control columns, 'expanded' state
      (const char*)&jpgrData, // dispText:  array of menu items
      nc.re,         // nColor: (n/a) sub-menus are invisible when collapsed
      nc.reR,        // fColor: focus border color / non-focus 'collapsed' color
      tbPrint,                // filter:    (n/a)
      NULL,                   // label:     hidden menus have no label
      ZERO,                   // labY:      (n/a)
      ZERO,                   // labX       (n/a)
      ddBoxTYPES,             // exType:    (n/a)
      short(gcDATA_ITEMS + 1),// scrItems:  number of menu items
      ZERO,                   // scrSel:    (n/a)
      monoColorB,             // scrColor:  array of color attributes for menu items
      NULL,                   // spinData:  (n/a)
      false,                  // active:    initially inactive
      &ic[T10mwM],            // nextCtrl:  link in next structure
   },
   {  //* Menuwin (stand-alone, initially collapsed)  - - - - - - - -   T10mwM *
      dctMENUWIN,             // type:      
      rbtTYPES,               // rbSubtype: (n/a)
      false,                  // rbSelect:  (n/a)
      7,                      // ulY: upper left corner in Y
      30,                     // ulX: upper left corner in X
      1,                      // lines:     (n/a)
      smITEM_LEN,             // cols:      control columns, 'expanded' state
      (const char*)&smboxData,// dispText:  array of menu items
      nc.gr,         // nColor: non-focus border color / focus 'collapsed' color
      nc.grR,        // fColor: focus border color / non-focus 'collapsed' color
      tbPrint,                // filter:    (n/a)
      " Move-^Dialog Menu ",  // label:     label text == 'collapsed state' text
      ZERO,                   // labY:      (n/a)
      ZERO,                   // labX       (n/a)
      ddBoxTYPES,             // exType:    (n/a)
      smDATA_ITEMS,           // scrItems:  number of menu items
      ZERO,                   // scrSel:    (n/a)
      multiColor,             // scrColor:  array of color attributes for menu items
      NULL,                   // spinData:  (n/a)
      true,                   // active:    initially inactive
      &ic[T10rb1],            // nextCtrl:  link in next structure
   },
   {  //* Stand-alone Radio Button, - - - - - - - - - - - - - - - -     T10rb1 *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[T10mwM].ulY + 2),    // ulY:       upper left corner in Y
      25,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a) for standard button types
      nc.grG,                       // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Ac^tivate Context Menu",     // label:     label text
      ZERO,                         // labY:      label offset
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T10rb2],                  // nextCtrl:  link in next structure
   },
   {  //* Stand-alone Radio Button, - - - - - - - - - - - - - - - -     T10rb2 *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[T10rb1].ulY + 2),    // ulY:       upper left corner in Y
      short(ic[T10rb1].ulX),        // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a) for standard button types
      nc.grG,                       // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Mo^ve Context Menu",         // label:     label text
      ZERO,                         // labY:      label offset
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T10rb3],                  // nextCtrl:  link in next structure
   },
   {  //* Stand-alone Radio Button, - - - - - - - - - - - - - - - -     T10rb3 *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[T10rb2].ulY + 2),    // ulY:       upper left corner in Y
      short(ic[T10rb2].ulX),        // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a) for standard button types
      nc.grG,                       // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Hide Men^u Bar",             // label:     label text
      ZERO,                         // labY:      label offset
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T10rb4],                  // nextCtrl:  link in next structure
   },
   {  //* Stand-alone Radio Button, - - - - - - - - - - - - - - - -     T10rb4 *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[T10rb3].ulY + 2),    // ulY:       upper left corner in Y
      short(ic[T10rb3].ulX),        // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a) for standard button types
      nc.grG,                       // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Di^sable Two Items in Menu A",// label:    label text
      ZERO,                         // labY:      label offset
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T10rb5],                  // nextCtrl:  link in next structure
   },
   {  //* Stand-alone Radio Button, - - - - - - - - - - - - - - - -     T10rb5 *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[T10rb4].ulY + 2),    // ulY:       upper left corner in Y
      short(ic[T10rb4].ulX),        // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a) for standard button types
      nc.grG,                       // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "^Replace MenuA Text Data",   // label:     label text
      ZERO,                         // labY:      label offset
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T10tbA],                  // nextCtrl:  link in next structure
   },
   {  //* Text Box (inactive) for status messages  - - - - - - - - -    T10tbA *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      3,                            // ulY:       upper left corner in Y
      45,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      30,                           // cols:      control columns
      " Welcome to Menu World-Enjoy! ", // dispText:  initial display text
      nc.cy,                        // nColor:    non-focus color
      nc.cy,                        // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Status Messages",            // label:     label text
      -1,                           // labY:      label offset
      8,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    view only
      NULL,                         // nextCtrl:  link in next structure
   },
} ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       "  Menu-Win Controls  ", // dialog title
                       ncltSINGLE,     // border line-style
                       bColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;
   Test10Dialog = dp ;              // give callback method access to the dialog

   //* Open the dialog window *
   if (   (termRows >= minNeededLines && termCols >= minNeededCols) 
       && ((dp->OpenWindow()) == OK) )
   {
      //* Write the static text *
      dp->WriteString ( 3, 6, "Visibility of the MenuBar", dColor ) ;
      dp->WriteString ( 4, 6, "may be toggled via Ctrl+Z           ", dColor ) ;
      dp->RefreshWin () ;                 // Display the results

      //* Create a MenuBar (a group of DialogMenuwin controls), and attach     *
      //* associated sub-menus                                                 *
      short mwList[] = { T10mwA, T10mwB, T10mwC, -1 } ;
      if ( (dp->GroupMenuwinControls ( mwList, nckC_Z )) == OK )
      {
         //* Attach sub-menus to main menu controls *
         //* Three sub-menus are attached to T10mwB at items 0, 3, 7 and 9.    *
         //* One sub-sub-menu is attached to sub-menu T10cmC at item 2.        *
         const char* asmMsg = "Attach Sub-menu Error!" ;
         short smList[] = { T10cmA, MAX_DIALOG_CONTROLS, MAX_DIALOG_CONTROLS,
                            T10cmB, MAX_DIALOG_CONTROLS, MAX_DIALOG_CONTROLS, 
                            MAX_DIALOG_CONTROLS, T10cmC, MAX_DIALOG_CONTROLS, 
                            T10cmD, -1 
                          } ;
         short ssList[] = { MAX_DIALOG_CONTROLS, MAX_DIALOG_CONTROLS, T10cmE, -1 } ;
         if ( (dp->AttachMenuwinSubmenus ( T10mwB, smList )) != OK )
            dp->DebugMsg ( asmMsg ) ;
         if ( (dp->AttachMenuwinSubmenus ( T10cmC, ssList )) != OK )
            dp->DebugMsg ( asmMsg ) ;
         //* Children and grandchildren of T10mwG to a ridiculous level *
         short gsList[] = { T10cmG2, -1 } ;
         if ( (dp->AttachMenuwinSubmenus ( T10mwG, gsList )) != OK ) dp->DebugMsg ( asmMsg ) ;
         gsList[ZERO] = T10cmG3 ;
         if ( (dp->AttachMenuwinSubmenus ( T10cmG2, gsList )) != OK ) dp->DebugMsg ( asmMsg ) ;
         gsList[ZERO] = T10cmG4 ;
         if ( (dp->AttachMenuwinSubmenus ( T10cmG3, gsList )) != OK ) dp->DebugMsg ( asmMsg ) ;
         gsList[ZERO] = T10cmG5 ;
         if ( (dp->AttachMenuwinSubmenus ( T10cmG4, gsList )) != OK ) dp->DebugMsg ( asmMsg ) ;
         gsList[ZERO] = T10cmG6 ;
         if ( (dp->AttachMenuwinSubmenus ( T10cmG5, gsList )) != OK ) dp->DebugMsg ( asmMsg ) ;
         #if 0    // Set to 1 for testing of a sub-menu attached to a context menu
         gsList[ZERO] = T10cmG5 ;
         if ( (dp->AttachMenuwinSubmenus ( T10mwI, gsList )) != OK ) dp->DebugMsg ( asmMsg ) ;
         #endif   // for testing of a sub-menu attached to a context menu
      }
      else
         dp->DebugMsg ( "Group Menuwin Error!" ) ;
      
      //* Establish a call-back method that can be called from within the *
      //* NcDialog code. Called each time through a control's user-input  *
      //* loop so we can manually modify the controls when necessary.     *
      dp->EstablishCallback ( &T10_ControlUpdate ) ;

      //* Allow user to play with the controls.      *
      //* (Please see the notes in NcDialog.hpp on   *
      //* values returned (from the editing routines)*
      gString mBuff ;               // status messages
      uiInfo Info ;                 // user interface data returned here
      short  icIndex = ZERO ;       // index of control with input focus
      bool   uidumpEnabled = false, // enable user-interface info dump
             done = false ;         // loop control
      while ( ! done )
      {
         //*******************************************
         //* If focus is currently on a Pushbutton   *
         //*******************************************
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            //* Get user input for this control                            *
            //* Returns when the Pushbutton is pressed OR when the user    *
            //* requests that the input focus moves to another control.    *
            if ( Info.viaHotkey == false )
            {
               icIndex = dp->EditPushbutton ( Info ) ;
            }
            //* If user selected the current control object via hotkey, we *
            //* don't need to call the input routine; but simply process   *
            //* the existing data.                                         *
            else
            {
               //* Information arrived in the Info.h_xxx fields.*
               //* Transfer it to the primary positions.        *
               //* Previous data in these fields has already    *
               //* been handled in prior loop iteration.        *
               Info.HotData2Primary () ;
            }

            //* If a Pushbutton was pressed *
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == T10donePush ) // request to exit the dialog
               {
                  dp->SetTextboxText ( T10tbA, "   T10donePush - Bye Bye !!   " ) ;
                  done = true ;
               }
               // (no other pushbuttons defined in this dialog)
            }
            else
            {
               // No button press, so nothing to do
            }
         }

         //***********************************************
         //* If focus is currently on a Menu-win control *
         //***********************************************
         else if ( ic[icIndex].type == dctMENUWIN )
         {
            //* It doesn't matter how this control gained the input    *
            //* focus, by Tab/ShiftTab or via hotkey. If by hotkey, no *
            //* processing was done anyway, so discard any hotkey data *
            //*                                                        *
            //* However, if you want the control to expand immediately *
            //* when EditMenuwin() is called, then set                 *
            //* Info.viaHotkey==true. Otherwise, the control will      *
            //* expand when the user presses Enter or one of the       *
            //* vertical scrolling keys. This is useful if the user    *
            //* arrived at the current control via hotkey because by   *
            //* selecting the control via hotkey, he/she has already   *
            //* demonstrated the intention to edit the control.        *
            //* (all other data in the Info class is ignored on entry) *

            //* Allow user to select a member from the menu data.      *
            //* Returns when edit is complete.                         *
            icIndex = dp->EditMenuwin ( Info ) ;
            if ( uidumpEnabled )
            {  // To view contents of Info
               winPos wPos(2,1) ;
               dp->Dump_uiInfo ( wPos, Info ) ;
            }

            //* If a selection has been made *
            if ( Info.dataMod != false )
            {
               char  tMenu ;
               short tItem ;
               switch ( Info.ctrlIndex )
               {
                  case T10mwA:            // Menu A-G, and I : do nothing
                     if ( Info.selMember == 5 ) // EXCEPT: Shell Out
                     {
                        dp->ShellOut ( ) ;
                     }  /* Fall Through... */
                  case T10mwB:
                  case T10mwC:
                  case T10mwG:
                  case T10mwI:
                     switch ( Info.ctrlIndex )
                     {
                        case T10mwC:  tMenu = 'C' ;  break ;
                        case T10mwA:  tMenu = 'A' ;  break ;
                        case T10mwB:  tMenu = 'B' ;  break ;
                        case T10mwG:  tMenu = 'G' ;  break ;
                        case T10mwI:  tMenu = 'I' ;  break ;
                     }
                     mBuff.compose( L"  T10mw%c - Item %02d selected   ", 
                                    &tMenu, &Info.selMember ) ;
                     dp->SetTextboxText ( T10tbA, mBuff ) ;
                     break ;
                  case T10cmG2:           // Menu G2 through g6 do nothing
                  case T10cmG3:
                  case T10cmG4:
                  case T10cmG5:
                  case T10cmG6:
                     switch ( Info.ctrlIndex )
                     {
                        case T10cmG2:  tItem = 2 ;  break ;
                        case T10cmG3:  tItem = 3 ;  break ;
                        case T10cmG4:  tItem = 4 ;  break ;
                        case T10cmG5:  tItem = 5 ;  break ;
                        case T10cmG6:  tItem = 6 ;  break ;
                     }
                     mBuff.compose( L"  T10cmG%hd - Item %02hd selected   ", 
                                    &tItem, &Info.selMember ) ;
                     dp->SetTextboxText ( T10tbA, mBuff ) ;
                     break ;
                  case T10mwM:
                     //* Request to reposition the dialog *
                     //* within the terminal window.      *
                     if ( Info.selMember != dialogPos )  // check curent position
                     {
                        //* Move the window to the specified position *
                        winPos wPos (winPosList[Info.selMember].ypos, 
                                             winPosList[Info.selMember].xpos) ;
                        dp->MoveWin ( wPos ) ;
                        dialogPos = Info.selMember ;  // record the new posiiton
                        mBuff.compose( L" T10mwM - %s  ", 
                                       &smboxData[Info.selMember][2] ) ;
                        dp->SetTextboxText ( T10tbA, mBuff ) ;
                     }
                     break ;
                  case T10cmA:            // Sub-menu A-E do nothing
                  case T10cmB:
                  case T10cmC:
                  case T10cmD:
                  case T10cmE:
                     switch ( Info.ctrlIndex )
                     {
                        case T10cmA:  tMenu = 'A' ;  break ;
                        case T10cmB:  tMenu = 'B' ;  break ;
                        case T10cmC:  tMenu = 'C' ;  break ;
                        case T10cmD:  tMenu = 'D' ;  break ;
                        case T10cmE:  tMenu = 'E' ;  break ;
                     }
                     mBuff.compose( L"  T10cm%c - Item %02d selected   ", 
                                    &tMenu, &Info.selMember ) ;
                     dp->SetTextboxText ( T10tbA, mBuff ) ;
                     break ;
                  default:
                     // no other Menu-win controls defined for this dialog
                     break ;
               }
            }
         }     // dctMENUWIN

         //*******************************************
         //* If focus is currently on a Radio Button *
         //*******************************************
         else if ( ic[icIndex].type == dctRADIOBUTTON )
         {
            //* Get user input for this control                            *
            //* Returns when radio-button control or radio-button group is *
            //* about to lose the input focus.                             *
            //* A selection may or may not have been made.                 *
            if ( Info.viaHotkey == false )
            {
               icIndex = dp->EditRadiobutton ( Info ) ;
            }
            //* If user selected the current control object via hotkey, we *
            //* don't need to call the input routine; but simply process   *
            //* the existing data.                                         *
            //* For radio buttons, display update will already have been   *
            //* done, so we can just move the focus to the next control.   *
            else
            {
               //* Information arrived in the Info.h_xxx fields.*
               //* Transfer it to the primary positions.        *
               //* Previous data in these fields has already    *
               //* been handled in prior loop iteration.        *
               Info.HotData2Primary () ;
            }
            
            //* If 'selected' button is a member of a button group *
            if ( Info.selMember < MAX_DIALOG_CONTROLS )
            {
               //* If a new 'selected' member of the group *
               //* OR previous selection was re-selected   *
               if ( Info.dataMod != false || Info.keyIn == nckENTER )
               {
                  //* Process the user's selection *
                  switch ( Info.selMember )
                  {
                     default:
                        // (there are no radio-button groups defined for this dialog)
                        break ;
                  }
               }
            }
            //* Else, button is an independent button *
            //* (Info.selMember==MAX_DIALOG_CONTROLS) *
            else
            {
               //* If a button has changed state *
               if ( Info.dataMod != false )
               {
                  switch ( Info.ctrlIndex )
                  {
                     case T10rb1:         // context menu visibility
                        if ( Info.isSel != false )
                        {
                           dp->SetTextboxText ( T10tbA, " T10rb1 - Edit context menu " ) ;
                           //* Make context menu visible and edit it *
                           dp->EditMenuwin ( T10mwI, Info ) ;
                           //* Reset the menu activation radio button *
                           dp->SetRadiobuttonState ( T10rb1, false ) ;
                           if ( Info.dataMod )
                           {
                              mBuff.compose( L"  T10mwI - Item %02d selected   ", 
                                                            &Info.selMember ) ;
                           }
                           else
                           {
                              mBuff.compose( L"%s", " T10mwI - No selection      ", 
                                                            &Info.selMember ) ;
                           }
                           dp->SetTextboxText ( T10tbA, mBuff ) ;
                        }
                        break ;
                     case T10rb2:         // context menu position
                        dp->SetTextboxText ( T10tbA, " T10rb2 - Move context menu." ) ;
                        if ( Info.isSel != false )
                        {
                           //* Move context menu to alternate position *
                           if ( (dp->PositionContextMenu ( T10mwI, 4, 9 )) != OK )
                           {
                              dp->DebugMsg ( "Error Moving Context Menu!", 2 ) ;
                              dp->DebugMsg ( "%" ) ;
                           }
                        }
                        else
                        {
                           //* Return context menu to original position *
                           if ( (dp->PositionContextMenu ( T10mwI, 5, 50 )) != OK )
                           {
                              dp->DebugMsg ( "Error Moving Context Menu!", 2 ) ;
                              dp->DebugMsg ( "%" ) ;
                           }
                        }
                        break ;
                     case T10rb3:         // MenuBar visibility
                        if ( Info.isSel != false )
                        {
                           dp->SetTextboxText ( T10tbA, " T10rb3 - Make MBar invisible " ) ;
                           dp->HideMenuBar ( T10mwA ) ;
                        }
                        else
                        {
                           dp->SetTextboxText ( T10tbA, " T10rb3 - Make MBar visible   " ) ;
                           icIndex = dp->ShowMenuBar ( T10mwA, true ) ;
                           //* We want menu with focus to open immediately     *
                           //* when edit method is called, so set viaHotkey.   *
                           Info.viaHotkey = true ;
                        }
                        break ;
                     case T10rb4:         // Disable/Enable some menu items in Menu 'A'
                        bool rb4set ;
                        dp->GetRadiobuttonState ( T10rb4, rb4set ) ;
                        if ( ! rb4set )
                           dp->SetActiveMenuItems ( T10mwA, mwA_allActive, mwA_allColor ) ;
                        else
                           dp->SetActiveMenuItems ( T10mwA, mwA_someInactive, mwA_someColor ) ;
                        break ;
                     case T10rb5:         // Replace the Menu 'A' text data
                        bool rb5set ;
                        dp->GetRadiobuttonState ( T10rb5, rb5set ) ;
                        if ( ! rb5set )
                           dp->SetActiveMenuItems ( T10mwA, (const char*)saboxData ) ;
                        else
                           dp->SetActiveMenuItems ( T10mwA, (const char*)saaltData ) ;
                        break ;
                     default:
                        // (no other independent radio buttons in this dialog)
                        break ;
                  }
               }
            }
         }

         //* If user exited the control edit method via a hotkey,*
         //* then the new control already has focus. Otherwise,  *
         //* move input focus to next/previous control.          *
         if ( done == false && Info.viaHotkey == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }  // while()
   }  // OpenWindow()
   else
   {  //* Most likely cause of dialog not opening is that the terminal window  *
      //* is too small. Give user a clue...                                    *
      short neededLines = minNeededLines, 
            neededCols  = minNeededCols ;
      gString gs ;
      gs.compose ( L" Sorry, size of terminal window must be at least %hd x %hd for this test. ", 
                   &neededCols, &neededLines ) ;
      StatWin ( gs.gstr(), nc.cyG ) ;
      sleep ( 5 ) ;
   }

   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

}  //* End Test10() *

//*************************
//*  T10_ControlUpdate    *
//*************************
//******************************************************************************
//* This is a callback method for manually updating the controls in the        *
//* Test10() dialog.                                                           *
//*                                                                            *
//*  For this test, the following are updated by the callback method:          *
//*   1. If the MenuBar is hidden (radio button T10rb3 set), and the MenuBar   *
//*      is re-activated via hotkey, T10rb3 is reset to indicate that the      *
//*      MenuBar is now visible.                                               *
//*   2.                                                                       *
//*   3.                                                                       *
//*                                                                            *
//* Input  : currIndex: index of control that currently has focus              *
//*          wkey     : user's key input data                                  *
//*          firstTime: the EstablishCallback() method calls this method once  *
//*                     with firstTime==true, to perform any required          *
//*                     initialization. Subsequently, the NcDialog class       *
//*                     always calls with firstTime==false.                    *
//* Returns: OK                                                                *
//******************************************************************************
//* Important Note: This method makes some intimate assumptions about what     *
//* is happening in the method that established the callback. If changes are   *
//* made in the establishing method that affect the callback functionality,    *
//* be sure to update this method to address those changes.                    *
//*                                                                            *
//* Programmer's Note: See dialog control index constants at file scope above. *
//******************************************************************************

short T10_ControlUpdate ( const short currIndex, const wkeyCode wkey, bool firstTime )
{
   static bool mbVisible = false ;

   //*************************
   //* First-time processing *
   //*************************
   if ( firstTime != false )
   {
      //* Initialize the tracking data *
      mbVisible = Test10Dialog->MenuBarVisible ( T10mwA ) ;
   }

   //***********************
   //* Standard processing *
   //***********************

   //* Match 'Hide MenuBar' radio button state to actual MenuBar visibility *
   bool nowVisible = Test10Dialog->MenuBarVisible ( T10mwA ) ;
   if ( (nowVisible && !mbVisible) || (!nowVisible && mbVisible) )
   {
      Test10Dialog->SetRadiobuttonState ( T10rb3, (bool)(!nowVisible) ) ;
      mbVisible = nowVisible ;
   }

   return OK ;

}  //* End T10_ControlUpdate() *

//*************************
//*  GetCommandLineArgs   *
//*************************
//******************************************************************************
//* Capture user's command-line arguments.                                     *
//* If conflicting arguments are specified, the LAST one will be accepted.     *
//*                                                                            *
//* Valid Arguments:                                                           *
//*   Test number: -[1 | 2 | 3 | 4 | 5 | 6 | 7 | 8 | 9 | 10]                   *
//*   Alt Locale : -A, -a     Specify an alternate 'locale' for interpretation *
//*                           of character output and input conversions.       *
//*   Exercise special options for Test01: [c , m , [0|1|2|3]]                 *
//*   Help       : -H, -h     Start application with display of Help Dialog.   *
//*                                                                            *
//* Input  : commArgs class object (by reference)                              *
//*                                                                            *
//* Returns: test number to be executed OR -1 if no test specified             *
//******************************************************************************

short GetCommandLineArgs ( commArgs& ca )
{
   if ( ca.argCount > 1 )
   {
      short v = 1,         // argv index
            i ;            // current arg index

      while ( v < ca.argCount )
      {
         i = ZERO ;
         if ( ca.argList[v][i] == '-' )  ++i ;     // step over dashes
         if ( ca.argList[v][i] == '-' )  ++i ;

         //* Test number specified? *
         if ( ca.argList[v][i] >= '1' && ca.argList[v][i] <= '9' )
         {
            sscanf ( &ca.argList[v][i], "%hd", &ca.testNum ) ;
            --ca.testNum ; // make it a zero-based index
         }

         //* Alternate locale specified? *
         else if ( ca.argList[v][i] == 'A' || ca.argList[v][i] == 'a' )
         {  //* If user specified an alternate locale string, use it instead *
            //* of the default locale found in the terminal environment.     *
            ++i ;
            if ( ca.argList[v][i] != NULLCHAR )
            {
               if ( ca.argList[v][i] == '=' )
                  ++i ;
               strncpy ( ca.localeName, &ca.argList[v][i], 30 ) ;
               ca.altLocale = true ;
            }
            else if ( ca.argCount > v )
            {
               strncpy ( ca.localeName, ca.argList[v+1], 30 ) ;
               ca.altLocale = true ;
               ++v ;    // step over the locale name
            }
         }

         //* Special options for Test01() *
         else if ( ca.argList[v][i] == 'd' )
         {
            char arg1 = ca.argList[v][i+1], arg2 = nckNULLCHAR, arg3 = nckNULLCHAR ;
            if ( arg1 != nckNULLCHAR )
            {
               arg2 = ca.argList[v][i+2] ;
               if ( arg2 != nckNULLCHAR )
                  arg3 = ca.argList[v][i+3] ;
               if ( arg1 == 'c' || arg2 == 'c' || arg3 == 'c' )
               { t1dtmExercise = true ; t1dtmCentered = true ; }
               if ( arg1 == 'm' || arg2 == 'm' || arg3 == 'm' )
               { t1dtmExercise = true ; t1dtmColorful = true ; }
               if ( arg1 == 'r' || arg2 == 'r' || arg3 == 'r' )
               { t1dtmExercise = true ; t1dtmRgt2Left = true ; }
               if ( t1MtextOption == ZERO )
               {
                  if ( arg1 == '1' || arg2 == '1' || arg3 == '1' )
                     t1MtextOption = 1 ;
                  else if ( arg1 == '2' || arg2 == '2' || arg3 == '2' )
                     t1MtextOption = 2 ;
                  else if ( arg1 == '3' || arg2 == '3' || arg3 == '3' )
                     t1MtextOption = 3 ;
               }
            }
            else     // no sub-arguments, so just set the basic flag
               t1dtmExercise = true ;
         }

         //* Else, request for help OR invalid command-line argument *
         else
         {
            ca.testNum = indexHELP ;
            break ;
         }

         ++v ;          // next command-line argument
      }
   }
   return ca.testNum ;
   
}  //* End GetCommandLineArgs() *

//*************************
//*      HelpAbout        *
//*************************
//******************************************************************************
//* Display the application's Help-About window.                               *
//* This is a simple informational dialog with a 'CLOSE' button.               *
//*                                                                            *
//*                                                                            *
//* Input  : ctrY : position at which to center dialog in Y                    *
//*          ctrX : position at which to center dialog in X                    *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void HelpAbout ( short ctrY, short ctrX )
{
const gString DialogText( 
   "Dialog1, \n"
   "Copyright (c) %s Mahlon R. Smith, The Software Samurai\n"
   "                        Beijing University of Technology\n"
   " \n"
   "A simple utility for development and testing of the NcDialog family\n"
   "of classes based on the Linux/UNIX ncurses library.\n"
   " \n"
   "Command Line arguments (optional):\n"
   "    -n      where 'n' is a test number\n"
   "    -A, -a  specify alternate 'locale' Ex: -Awo_SN.utf8 \n"
   "    -d[c , m , r , [0|1|2|3]] (See header of Test01 for details.)\n"
   "    -H, -h  Help (this dialog)\n"
   " \n"
   "The API behind this utility provides a means of easily designing\n"
   "applications for curses, while in a modern, C++ environment.\n"
   "For bugs, suggestions, possible praise: http://www.SoftwareSam.us\n"
   " \n"
   "Tools: GNU G++ v:13 (C++11) under Fedora and Ubuntu Linux.\n"
   "       Requires ncursesw library v:6.4+ ", crYears
);//xxxxxxxxx0xxxxxxxxx0xxxxxxxxx0xxxxxxxxx0xxxxxxxxx0xxxxxxxxx0xxxxxxxx|
static const short dialogROWS = 23 ;      // # of display lines
static const short dialogCOLS = 70 ;      // # of display columns
static const short controlsDEFINED = 3 ;  // # of controls in dialog
enum ctrls { closePUSH = ZERO, classPUSH, appverTBOX } ;
short ulY = ctrY - dialogROWS / 2,
      ulX = ctrX - dialogCOLS / 2 ;
attr_t dColor = nc.blR ;

   //* Format application version string for display *
   gString verString( " %s ", AppVersion ) ;


InitCtrl ic[controlsDEFINED] =      // array of dialog control info
{
   {  //* 'CLOSE' pushbutton  - - - - - - - - - - - - -  closePUSH *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 2),        // ulY:       upper left corner in Y
      short(dialogCOLS / 2 - 4),    // ulX:       upper left corner in X
      1,                            // lines:     control lines
      7,                            // cols:      control columns
      " CLOSE ",                    // dispText:  
      nc.gyR,                       // nColor:    non-focus color
      nc.reR,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[1],                       // nextCtrl:  link in next structure
   },
   {  //* 'CLASS VERSION' pushbutton  - - - - - - - - -  classPUSH *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 2),        // ulY:       upper left corner in Y
      short(dialogCOLS - 25),       // ulX:       upper left corner in X
      1,                            // lines:     control lines
      24,                           // cols:      control columns
      " DISPLAY CLASS VERSIONS ",   // dispText:  
      nc.gyR,                       // nColor:    non-focus color
      nc.reR,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[2],                       // nextCtrl:  link in next structure
   },
   {  //* Non-active Textbox for app. version  - - - -  appverTBOX *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      1,                            // ulY:       upper left corner in Y
      19,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      short(verString.gscols()),    // cols:      control columns
      verString.ustr(),             // dispText:  (value of t3indexO3)
      nc.bl,                        // nColor:    non-focus color
      nc.bl,                        // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Version:",                   // label:     
      ZERO,                         // labY:      
      -9,                           // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    cannot gain focus
      NULL,                         // nextCtrl:  link in next structure
   },
} ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      dp->WriteParagraph ( 1, 1, DialogText, dColor ) ;

      #if 0    // For Debugging Only
      // Ncurses directives determine whether internal elements of certain structures
      // are available at the application level. The documentation says that these 
      // directives are set as "opaque" by default, and should be defined at the 
      // application level if access is desired. (It also says that this is dangerous.)
      // The NcDialog API relies upon access to the members of "WINDOW" 
      // (aka "struct _win_st"), so opacity must be disabled for that structure.
      // See NCurses.hpp for these definitions.
      wchar_t ni = L'x', oo = L'x', of = L'x', om = L'x', op = L'x' ;
      #ifdef NCURSES_INTERNALS
      ni = L'y' ;
      #endif   // NCURSES_INTERNALS
      #ifdef NCURSES_OPAQUE
      oo = NCURSES_OPAQUE != 0 ? L'1' : L'0' ;
      #endif   // NCURSES_OPAQUE
      #ifdef NCURSES_OPAQUE_FORM
      of = NCURSES_OPAQUE_FORM != 0 ? L'1' : L'0' ;
      #endif   // NCURSES_OPAQUE_FORM
      #ifdef NCURSES_OPAQUE_MENU
      om = NCURSES_OPAQUE_MENU != 0 ? L'1' : L'0' ;
      #endif   // NCURSES_OPAQUE_MENU
      #ifdef NCURSES_OPAQUE_PANEL
      op = NCURSES_OPAQUE_PANEL != 0 ? L'1' : L'0' ;
      #endif   // NCURSES_OPAQUE_PANEL
      gString gs( "Internals:%C Opaque:%C Form:%C Menu:%C Panel:%C", &ni, &oo, &of, &om, &op ) ;
      dp->WriteParagraph ( dialogROWS - 3, 1, gs, nc.bbrbl ) ;
      #endif   // For Debugging Only

      dp->RefreshWin () ;

      uiInfo Info ;                 // user interface data returned here
      bool   done = false ;         // loop control
      while ( ! done )
      {
         dp->EditPushbutton ( Info ) ;
         if ( Info.dataMod != false )
         {
            if ( Info.ctrlIndex == closePUSH )
               done = true ;
            else if ( Info.ctrlIndex == classPUSH )
            {  //* Open a dialog to display class version numbers and other    *
               //* interesting data. Current dialog will be obscured, so save  *
               //* its display data.                                           *
               dp->SetDialogObscured () ;
               DisplayClassInfo ( (ulY + 4), (ctrX - 20), 15, 40 ) ;
               dp->NextControl () ;
               dp->RefreshWin () ;
            }
         }
         if ( done == false )
         {
            if ( Info.keyIn == nckSTAB )
               dp->PrevControl () ; 
            else if ( Info.keyIn != ZERO )
               dp->NextControl () ;
         }
      }
   }
   if ( dp != NULL )
      delete ( dp ) ;                        // close the window

}  //* End HelpAbout() *

//***********************
//* ClearTerminalWindow *
//***********************
//****************************************************************************
//* Clear the terminal window and display the application title.             *
//*                                                                          *
//*                                                                          *
//* Input Values : none                                                      *
//*                                                                          *
//* Return Value : none                                                      *
//****************************************************************************

void ClearTerminalWindow ( void )
{
   winPos   wpos( 0, 0 ) ;
   nc.ClearScreen () ;
   gString gsTitle( "  Dialog1 - v:%s (c) %s Mahlon R. Smith    : Locale: %s      ",
                    AppVersion, crYears, nc.GetLocale() ) ;
   nc.WriteString ( wpos, gsTitle.gstr(), nc.bw | ncuATTR ) ;
   nc.RefreshScreen () ;

}  //* End ClearTerminalWindow() *

//*************************
//*                       *
//*************************
//********************************************************************************
//*                                                                              *
//*                                                                              *
//*                                                                              *
//* Input  :                                                                     *
//*                                                                              *
//* Returns:                                                                     *
//********************************************************************************

