//********************************************************************************
//* File       : DialogAppShared.hpp                                             *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2006-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in NcDialog.hpp            *
//* Date       : 20-Mar-2025                                                     *
//*                                                                              *
//* Description: This header file contains the shared definitions, prototypes,   *
//*              and data shared by the demonstration and test applications      *
//*              for the NCurses class, NcWindow class and NcDialog class.       *
//*                                                                              *
//********************************************************************************

//***************
//* Definitions *
//***************

//* Data control definitions *
enum cIndices : short { indexT01 = ZERO, indexT02, indexT03, indexT04, indexT05, 
                        indexT06, indexT07, indexT08, indexT09, indexT10, 
                        indexHELP, indexEXIT, indexCOUNT } ;

//* Minimum terminal window size for this application *
const short minTERMROWS = 25 ;
const short minTERMCOLS = 80 ;

//* Status window definitions *
const short swLINES = 1 ;                    //* Line count (height)
const short swCOLS = 77 ;                    //* Column count (width)
const short swOFFy = 1 ;                     //* Y window offset
const short swOFFx = 2 ;                     //* X window offset
const short swLEFTj = 0 ;                    //* Left-justify
const short swCENTERj = -1 ;                 //* Center-justify
const short swRIGHTj = -2 ;                  //* Right-justify
const short minTestULY = swOFFy + 1 ;  //* minimum start line for any test window

//* Holds captured command-line arguments *
class commArgs
{
   public:
   commArgs ( int argc, char** argv, char** argenv ) :
               argCount(argc), argList(argv), envList(argenv)
   {
      testNum  = -1 ;
      *localeName = NULLCHAR ;
      altLocale = altColors = altColorMap = unassignedFlag = false ;
   }
   short    argCount ;
   char**   argList ;
   char**   envList ;
   short    testNum ;
   char     localeName[32] ;
   bool     altLocale ;
   bool     altColors ;
   bool     altColorMap ;
   bool     unassignedFlag ;
} ;

//***************
//* Prototypes  *
//***************
short    TestMenu ( short ctrY, short ctrX ) ;
short    GetCommandLineArgs ( commArgs& clArgs ) ;
void     DisplayClassInfo ( short ulY, short ulX, short dLines, short dCols ) ;
winPos   StatWin ( const wchar_t* wStr, attr_t color = nc.bl, 
                   bool clear = true, short xoff = swLEFTj ) ;
winPos   StatWin ( const char* uStr, attr_t color = nc.bl, 
                   bool clear = true, short xoff = swLEFTj ) ;
void     ClearTerminalWindow ( void ) ;
void     HelpAbout ( short ctrY, short ctrX ) ;

void     Test01 ( void ) ;
short    T01_ControlUpdate ( const short currIndex, 
                             const wkeyCode wkey, bool firstTime = false ) ;
void     Test02 ( void ) ;
short    T02_ControlUpdate ( const short currIndex, 
                             const wkeyCode wkey, bool firstTime = false ) ;
void     Test03 ( void ) ;
short    T03_ControlUpdate ( const short currIndex, 
                             const wkeyCode wkey, bool firstTime = false ) ;
void     Test04 ( void ) ;
short    T04_ControlUpdate ( const short currIndex, 
                             const wkeyCode wkey, bool firstTime = false ) ;
void     Test05 ( void ) ;
short    T05_ControlUpdate ( const short currIndex, 
                             const wkeyCode wkey, bool firstTime = false ) ;
void     Test06 ( void ) ;
short    T06_ControlUpdate ( const short currIndex, 
                             const wkeyCode wkey, bool firstTime = false ) ;
void     Test07 ( void ) ;
short    T07_ControlUpdate ( const short currIndex, 
                             const wkeyCode wkey, bool firstTime = false ) ;
void     Test08 ( void ) ;
short    T08_ControlUpdate ( const short currIndex, 
                             const wkeyCode wkey, bool firstTime = false ) ;
void     Test09 ( void ) ;
short    T09_ControlUpdate ( const short currIndex, 
                             const wkeyCode wkey, bool firstTime = false ) ;
void     Test10 ( void ) ;
short    T10_ControlUpdate ( const short currIndex, 
                             const wkeyCode wkey, bool firstTime = false ) ;

//**********
//*  Data  *
//**********
short    termRows,               // size of terminal window, rows
         termCols ;              // size of terminal window, columns

NcWindow* sWin ;                 // Status line window access
NcDialog* Test01Dialog ;         // file-scope access to dialog window pointer
NcDialog* Test02Dialog ;         // file-scope access to dialog window pointer
NcDialog* Test03Dialog ;         // file-scope access to dialog window pointer
NcDialog* Test04Dialog ;         // file-scope access to dialog window pointer
NcDialog* Test05Dialog ;         // file-scope access to dialog window pointer
NcDialog* Test06Dialog ;         // file-scope access to dialog window pointer
NcDialog* Test07Dialog ;         // file-scope access to dialog window pointer
NcDialog* Test08Dialog ;         // file-scope access to dialog window pointer
NcDialog* Test09Dialog ;         // file-scope access to dialog window pointer
NcDialog* Test10Dialog ;         // file-scope access to dialog window pointer


//*************************
//*  DisplayClassInfo     *
//*************************
//******************************************************************************
//* Display version numbers for NCurses, NcWindow, NcDialog classes.           *
//* Also reports current 'locale' setting. Other info may be added later.      *
//*                                                                            *
//* Input  : ulY   : upper-left corner of dialog in Y                          *
//*          ulX   : upper-left corner of dialog in X                          *
//*          dLines: dialog lines                                              *
//*          dCols : dialog columns                                            *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void DisplayClassInfo ( short ulY, short ulX, short dLines, short dCols )
{
attr_t dColor = nc.maR | ncbATTR ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dLines,         // number of display lines
                       dCols,          // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       "  Class Info  ", // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       NULL            // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      gString gs ;
      winPos wp(1, 1) ;
      wp = dp->WriteString ( wp.ypos, wp.xpos, "NcDialog class : ", dColor ) ;
      gs.compose( L" %s ", dp->Get_NcDialog_Version () ) ;
      wp = dp->WriteString ( wp, gs, nc.blB ) ;

      wp = { short(wp.ypos + 2), 1 } ;
      wp = dp->WriteString ( wp.ypos, wp.xpos, "NcWindow class : ", dColor ) ;
      gs.compose( L" %s ", dp->Get_NcWindow_Version () ) ;
      wp = dp->WriteString ( wp, gs, nc.blB ) ;

      wp = { short(wp.ypos + 2), 1 } ;
      wp = dp->WriteString ( wp, "NCurses class  : ", dColor ) ;
      gs.compose( L" %s ", dp->Get_NCurses_Version () ) ;
      wp = dp->WriteString ( wp, gs, nc.blB ) ;

      wp = { short(wp.ypos + 2), 1 } ;
      wp = dp->WriteString ( wp.ypos, wp.xpos, "gString class  : ", dColor ) ;
      gs.compose( L" %s ", gs.Get_gString_Version () ) ;
      wp = dp->WriteString ( wp, gs, nc.blB ) ;

      wp = { short(wp.ypos + 2), 1 } ;
      wp = dp->WriteString ( wp.ypos, wp.xpos, "ncurses library: ", dColor ) ;
      gs.compose( L" %s ", dp->Get_nclibrary_Version () ) ;
      wp = dp->WriteString ( wp, gs, nc.blB ) ;

      wp = { short(wp.ypos + 2), 1 } ;
      wp = dp->WriteString ( wp.ypos, wp.xpos, "Locale setting : ", dColor ) ;
      gs.compose( L" %s ", nc.GetLocale () ) ;
      wp = dp->WriteString ( wp, gs, nc.blB ) ;

      dp->WriteString ( (dLines - 2), (dCols / 2 - 3), " CLOSE ", nc.reR ) ;
      dp->RefreshWin () ;
      nckPause() ;
   }
   if ( dp != NULL )
      delete ( dp ) ;                        // close the window

}  //* End DisplayClassInfo() *

//*************************
//*       StatWin         *
//*************************
//******************************************************************************
//* Manage the status window which handles application messages.               *
//* Both UTF-8 and wchar_t input accepted.                                     *
//*                                                                            *
//* Input  : uStr OR wStr: UTF-8 or wchar_t string to be displayed             *
//*          color  : (optional, default==nc.bl)                               *
//*                   color attribute for string                               *
//*          clear  : (optional, default==true)                                *
//*                   if true, clear window before writing                     *
//*          xoff   : (optional, default==swLEFTj)                             *
//*                   begin writing at specified offset (column)               *
//*                                                                            *
//* Returns: current cursor position within the status window                  *
//******************************************************************************

winPos StatWin ( const wchar_t* wStr, attr_t color, bool clear, short xoff )
{

   if ( clear )
      sWin->ClearWin () ;
   return ( sWin->WriteString ( ZERO, xoff, wStr, color, true ) ) ;
   
}  //* End StatWin() *

winPos StatWin ( const char* uStr, attr_t color, bool clear, short xoff )
{
   gString  gs( uStr ) ;
   return ( StatWin ( gs.gstr(), color, clear, xoff ) ) ;
   
}  //* End StatWin() *


