//********************************************************************************
//* File       : NCurses.cpp                                                     *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2005-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in NCurses.hpp             *
//* Date       : 22-Jul-2025                                                     *
//* Version    : (see NCursesVersion string, below)                              *
//*                                                                              *
//* Description: This class implements a wrapper for the ncurses environment,    *
//* allowing a more easily understood interface and simplifying access to        *
//* many of the more obscure elements of the ncurses engine.                     *
//*                                                                              *
//* Note that there is NO attempt made to wrap the full functionality of the     *
//* ncurses or ncursesw libraries. We use the minimum number of ncurses          *
//* primitives necessary to implement basic configuration and input/output.      *
//* In our estimation, most of the ncurses library is unnecessary, and many      *
//* functions are buggy; thus, we use only the most reliable and necessary of    *
//* the available library functions.                                             *
//*                                                                              *
//* Development Tools: see NcDialog.cpp                                          *
//********************************************************************************
//* Version History (most recent first):                                         *
//*                                                                              *
//* v: 0.0.25 20-Mar-2025                                                        *
//*   -- gString class update (v:0.0.37) uses int32_t rather than int16_t type,  *
//*      so some integer variables were promoted to silence compiler warnings.   *
//*   -- ncurses defines (or fails to define) certain directives in curses.h     *
//*      related to accessing the internals of certain structures.               *
//*      At some point, these definitions were modified to make the structures   *
//*      "opaque" by default. This was a good move for reentrant code and        *
//*      multi-threaded code which allows multiple threads to simultaneously     *
//*      modify the structures; HOWEVER, the NcDialog API relies upon direct     *
//*      access to window position, color attributes and other structure         *
//*      members, specifically for: "WINDOW" aka "struct _win_st".               *
//*      For this reason, these directives are now explicitly defined in         *
//*      NCurses.hpp to allow access to the needed members.                      *
//*      See #ifndef NCURSES_INTERNALS in curses.h.                              *
//*         https://invisible-island.net/ncurses/man/curs_opaque.3x.html         *
//*         Opacity config provides SOME get/set-option functions:               *
//*         See: int mvwin(WINDOW * win, int y, int x);                          *
//*         https://invisible-island.net/ncurses/man/curs_window.3x.html         *
//*      - Enabling access to window internals may be a configuration option,    *
//*        but if so, we haven't yet been able to identify it; and therefore     *
//*        we can't expect other developers to independently locate and properly *
//*        initialize them.                                                      *
//*   -- Update the GetLocale() method to capture the locale name string rather  *
//*      simply returning the 'name().cstr()' from the locale object.            *
//*      This was done to avoid potetial timing issues with the persistence of   *
//*      of the pointer returned by the locale object. The captured data are     *
//*      parsed to determine whether the string contains multiple entries.       *
//*                                                                              *
//* v: 0.0.24 25-Jun-2023                                                        *
//*   -- The update from ncurses v:6.2 to v:6.4 (2023-06-06) broke the           *
//*      keycodes for cursor key combinations (AGAIN)! Used 'keypap' to          *
//*      generate a new set of default keycode definitions and integrated them   *
//*      into NCursesKeyDef.hpp.                                                 *
//*                                                                              *
//* v: 0.0.23 09-May-2020                                                        *
//*   -- The transition from Fedora30 to Fedora32 has broken a large number of   *
//*      keycodes in the range of 0x0102 - 0x0246. The broken key combinations   *
//*      generate the same escape sequences as previously; however, the          *
//*      keycodes assigned to them have been arbitrarily changed.                *
//*      The dynamic table in NCursesKeyDef.hpp and the TransTable2 lookup       *
//*      table have been updated to reflect the new keycode assignments.         *
//*   -- Bug Fix: Recent enhancements (2019 and 2020) to Gnometerm and Konsole   *
//*      interfere with our algorithm for display of RTL (Right-To-Left) text.   *
//*      To compensate, we explicitly disable BiDi (bidirectional text) support  *
//*      during the NCurses Engine start-up sequence, and then re-enable BiDi    *
//*      during the shut-down sequence. See notes in the BiDiEnable() method.    *
//*      -- BiDi support is also re-enabled by a call to Hibernate() in          *
//*         anticipation of temporarily returning to the shell program. BiDi     *
//*         support is then disabled again by a call to Wake().                  *
//*                                                                              *
//* v: 0.0.22 09-Sep-2019                                                        *
//*   -- Certain keycode definitions reported through the ncurses library are    *
//*      dependent on release version, patch date, OS distribution, terminal     *
//*      emulator definition, and other factors. These keycodes have been        *
//*      causing headaches since Thomas Dickey began work on ncurses v:6.0.      *
//*      Since that time, we have issued various official and unofficial         *
//*      patches to compensate, but it was clear that we needed a more           *
//*      generalized solution to the random shifts in keycode definitions from   *
//*      one installation to the next.                                           *
//*      The keycode definitions for the NcDialog API are located in:            *
//*      NCursesKeyDef.hpp.                                                      *
//*      -- Many keycodes such as the ASCII control codes, basic                 *
//*         (unconditioned) navigation keys and our Alt+printing_key extensions  *
//*         are stable and unlikey to change.                                    *
//*      -- The variable keycodes fall into three(3) categories:                 *
//*           1) Conditioned navigation keys (Alt/Ctrl/Shift + key)              *
//*           2) Function keys, both basic and conditioned                       *
//*           3) Numeric keypad keys                                             *
//*         Keycodes for these keys may require system-specific definitions      *
//*         rather than hard-coded definitions.                                  *
//*      -- Ideally, these system-specific definitions can be generated with a   *
//*         minimum of pain for the software designer using the NcDialog API.    *
//*         For this reason, we have isolated these keycode definitions under    *
//*         a conditional-compile flag which can reference either the embedded   *
//*         "default" values, OR an external header file:                        *
//*                            "NCursesKeyDef_auto.hpp"                          *
//*         This is an programatically-generated file which can be regenerated   *
//*         by the developer as-needed on his/her target system. For details,    *
//*         please see the "Keymap" utility which is distributed as part of      *
//*         the NcDialog API.                                                    *
//*      -- Remove the temporary (hack) definition: "SPECIAL_KEYCODES" defined   *
//*         in the previous release.                                             *
//*      -- The possibility of custom keycodes also means that custom versions   *
//*         of TransTable2[] must be generated to remain synchronized with the   *
//*         key definitions. This is also done by the "Keymap" utility.          *
//*   -- Define the argument to the UngetKeyInput( const wkeyCode wk ) method    *
//*      as "const" as it should have been from the beginning.                   *
//*   -- Bug Fix: In the WriteString() method, if there were ASCII control       *
//*      codes in the source text, the method would hang. (stupid programmers!)  *
//*   -- Update mouse interface to match changes in event reporting. These       *
//*      changes may be related to the ncursesw library OR may be related to     *
//*      the differences between the X11 mouse driver and the GNOME/Wayland      *
//*      mouse driver. Our update is primarily about the way scroll-wheel        *
//*      events are handled.                                                     *
//*      -- The mask bit for scroll-up has changed, and                          *
//*      -- Modifier keys are no longer reported in conjunction with             *
//*         scroll-wheel events. This has little effect on the NCurses mouse     *
//*         interface, but it does affect the NcDialog mouse interface.          *
//*                                                                              *
//* v: 0.0.21 11-Dec-2017                                                        *
//*   -- In NCursesKeydef.hpp, define nckACPGUP and nckACPGDN. These are         *
//*      translated by the ncurses library, but we forgot to include them        *
//*      in the keycode list. Instead, we had (inappropriately) defined these    *
//*      keycodes as belonging to the keypad nckpACPGUP and nckpACPGDN which     *
//*      have been removed.                                                      *
//*   -- Because both 5.9 and 6.0 of the ncurses library are currently being     *
//*      distributed, the ncurses version number is now stored in an             *
//*      NCurses-class data member for easy access by any future patches which   *
//*      are necessary for version-specific code. See Get_nclibrary_Version().   *
//*   -- Due to a bug in the ncurses library version 6.0, we have placed a       *
//*      temporary workaround in the public GetKeyInput() method. When we        *
//*      figure out the nature of this bug and which versions are affected,      *
//*      we will modify or remove the patch.                                     *
//*   -- Due to some subtle change in the way the compiler puts string pointers  *
//*      on the stack the VerifyLocale() method had stopped working. We now      *
//*      read the local string directly from the static ioLocal object.          *
//*   -- Add new conditional compile definition: "SPECIAL_KEYCODES" to           *
//*      re-define the subset of keycodes which changed between ncurses 5.9      *
//*      and ncurses 6.0. Support for ncurses 6.1 is still under development.    *
//*                                                                              *
//* v: 0.0.20 14-Apr-2016                                                        *
//*   - Bug Fix: In DisableMouseEvents(), even though mouse events were          *
//*     being correctly disabled, failure was always reported.                   *
//*   - Bug Fix: In TransTable2[], the human-readable output for the following   *
//*     keycodes was incorrect: nckAPGDN, nckAPGUP, nckACPGDN, nckACPGUP.        *
//*     This does not affect functionality, only the output of keycode test.     *
//*                                                                              *
//* v: 0.0.19 05-Feb-2015                                                        *
//*   - Adjust key definitions for 'Delete' and 'Insert' key groups.             *
//*     Adjust key definitions for Ctrl+arrow, Alt+arrow, Alt+Shift+arrow keys.  *
//*     Some ncurses keycodes have been changed for the current ncursed version  *
//*     (5.9.20130511). They probably changed with the patch in August, 2012,    *
//*     but we aren't sure about this. See notes in NCursesKeyDef.hpp.           *
//*   - Remove definitions for Alt+Shift+Fxx key group (never used).             *
//*     This makes room for more useful definitions.                             *
//*   - Add key definitions for the built-in numeric keypad.                     *
//*   - Split 'maxColors' into 'maxRgb' and 'maxColors' to reflect the way       *
//*     the values are actually used. Rename 'modColors' to 'modRgb'.            *
//*   - For terminals that support more than the basic eight (0 through 7)       *
//*     ncurses library colors, we now support terminal colors 8 through 15.     *
//*     This uses 8 additional color pairs (0x40-0x47) AND provides 8 new        *
//*     'bright' color variables: bbk, bre, bgr, bbr, bbl, bma, bcy, bgy, AND    *
//*     if color pairs 0x48-0x7E are available, these are initialized to the     *
//*     'bright' equivalents of pairs 0x08-0x3F, and assigned to additional      *
//*     color variables. Note that if the terminal supports only the 8 basic     *
//*     colors, then these new variables are duplicates of the basic eight.      *
//*   - Update test in VerifyLocale() method.                                    *
//*   - Finish implementing output stream automatic scrolling. See the           *
//*     OutputStreamScrolling() and WriteStream() methods. The stubs,            *
//*     EnableIOStreamScrolling() and DisableIOStreamScrolling() are now         *
//*     obsolete. This is a much cleaner and more reliable implementation.       *
//*   - nckPause() i.e. GetKeyInput(void) no longer returns on a                 *
//*     terminal-resize event.                                                   *
//*   - Visibility of cursor is now explicitly set to nccvINVISIBLE during the   *
//*     startup sequence, and is set th nccvNORMAL during the shutdown           *
//*     sequence. See StartNCursesEngine() and StopNCursesEngine().              *
//*   - Combine the EnableKeyEcho() and DisableKeyEcho() methods as              *
//*     KeyEcho(bool). This method is never used anyway, and this takes up       *
//*     less space.                                                              *
//*   - Add an optional color-attribute parameter to the ClearLine() method.     *
//*                                                                              *
//* v: 0.0.18 14-Jan-2014                                                        *
//*   - Experimentation with capture of display data. Unfortunately NCurses      *
//*     class cannot see data written to the terminal by other windows, so the   *
//*     project was abandoned.                                                   *
//*   - Update the WriteString() and WriteChar() methods to accept either        *
//*     discrete Y and X start position OR a winPos object. Concatenate the      *
//*     header info, one header for each group.                                  *
//*   - Add an optional parameter to all WriteString() and WriteChar() methods   *
//*     to support RTL (right-to-left) written languages.                        *
//*                                                                              *
//* v: 0.0.17 28-Nov-2013                                                        *
//*   - Add a SetCursor(winPos yxpos) method for convenience.                    *
//*                                                                              *
//* v: 0.0.16 19-May-2013                                                        *
//*   - Implement audible alarm method, UserAlert().                             *
//*   - Note that some ncurses library keycodes have been redefined somewhere    *
//*     between v:5.7 and v:5.9. Because we rely on v:5.9 of the library, we     *
//*     have updated the corresponding values in NCursesKeyDef.hpp, and have     *
//*     updated the necessary local escape-sequence translations. Keep an eye    *
//*     on this in case the keycode definitions change again.                    *
//*   - Implement three new debugging methods for keycode translation. See       *
//*     NcKey.cpp for details.                                                   *
//*   - The GetKeyInput( wchar_t&, bool ) method goes private. It shouldn't      *
//*     have been public to begin with.                                          *
//*   - Update 'UngetKeyInput' method to use a local keycode pushback queue      *
//*     rather than the wholly-inadequate 'unget_wch' ncurses primitive.         *
//*     This is necessary to support both our wktEXTEND keycodes and wktMOUSE    *
//*     event notifications. This also preserves the key type for wktFUNKEY      *
//*     keycodes which was previously being lost.                                *
//*   - Begin implementation of limited mouse support.                           *
//*     This entails an update to the wkeyCode class and enhancements to the     *
//*     GetKeyInput() methods, as well as a new group of mouse-event methods.    *
//*     The expectation is that the NCurses-class mouse-support functionality    *
//*     will be used _indirectly_ through a dialog object, and not through the   *
//*     NCurses primitives only. For this reason, the mouse methods provided     *
//*     in this class perform only the basic I/O rather than any sophisticated   *
//*     mapping of mouse events to screen objects.                               *
//*                                                                              *
//* v: 0.0.15 18-Apr-2012                                                        *
//*   - Integrate startup of the Color Engine with startup of the NCurses        *
//*     Engine. Except under unusual circumstances, it is desirable to start     *
//*     the Color Engine for all applications, so it is more convenient to       *
//*     integrate the two functions.                                             *
//*     Note that automatic startup of the Color Engine may be skipped by        *
//*     overriding the default parameters when calling StartNCursesEngine().     *
//*   - Add data members maxColors (from ncurses COLORS global variable),        *
//*     maxPairs (from ncurses COLOR_PAIRS global variable), modColors           *
//*     (returned by has_colors() library primitive), and modPairs (returned     *
//*     by can_change_color() library primitive).                                *
//*   - Fix bug in cursor position returned from the WriteString() and           *
//*     WriteChar() method group.                                                *
//*   - Filter out ASCII control characters in WriteString() and WriteChar()     *
//*     group.                                                                   *
//*                                                                              *
//* v: 0.0.14 08-Mar-2012                                                        *
//*   - Change development platform and compiler version.                        *
//*      Fedora 15 and GNU G++ (Gcc v: 4.6.1)                                    *
//*      ncurses library v: 5.9.20110404                                         *
//*      GNOME terminal 3.0.1                                                    *
//*   - Change the way screen output methods pass data to the ncurses library    *
//*     primitives. Due to changes in the ncurses library between v:5.7 to       *
//*     v:5.9, it has become necessary to pass only one wide character at a      *
//*     time to wadd_wch() via the cchar_t structure. Although this is less      *
//*     efficient than passing five characters per call, it is a relatively      *
//*     straight-forward update.                                                 *
//*                                                                              *
//* v: 0.0.13 04-Feb-2012                                                        *
//*   - Convert all color attributes from 'int' to 'attr_t'. Should have done    *
//*     this from the beginning, but type aliases can be a major pain.           *
//*   - Add the wcsXXXX group of constants to NCurses.hpp to give access to      *
//*     the wchar_t equivalents of the Alternate Character Set (ACS).            *
//*     NOTE: There is a bug in the definition of WACS_BOARD in ncurses.h,       *
//*           v:5.7, 5.8, 5.9 (defined with same code as WACS_PATTERN)           *
//*           Need to find wide tic-tac-toe board character.                     *
//*                                                                              *
//* v: 0.0.12 13-Aug-2011                                                        *
//*   - Add WriteChar() methods for both UTF-8 and wchar_t characters.           *
//*   - Update DebugBorder() method to use WriteChar() rather than ncurses       *
//*     primitives.                                                              *
//*   - Renamed GetKeyInput32() and GetKeyInput16() to GetKeyInput()             *
//*     (overloaded by parameter type).                                          *
//*   - Support for UTF-8-encoded text output and keyboard input implemented     *
//*     and verified. (Bugs, of course, may surface...)                          *
//*                                                                              *
//* v: 0.0.11 30-Jun-2011                                                        *
//*   - NCurses key input methods: internally to the class, all key input now    *
//*     comes through the ncursesw, 32-bit key input functions, get_wch(),       *
//*     wget_wch(), etc.                                                         *
//*   - For the NCurses class, internally, all key input previously went         *
//*     through the GetKeyInput16() method. While this works fine for ASCII      *
//*     input and tokenized escape sequences, it is likely that some 'wide'      *
//*     character input for international characters requires more than 16       *
//*     bits. The ncurses primitives, getch(), wgetch(), etc. all return int     *
//*     (32-bit) values, but apparently only 16 bits of the int are actually     *
//*     used.                                                                    *
//*   - The application programmer may still request 16-bit input through        *
//*     NCurses::GetKeyInput16(), but this is not recommended in                 *
//*     internationalized applications.                                          *
//*   - The WriteString() and GetCursor() return values have been updated from   *
//*     a WinPos structure to a winPos class object, and the WinPos structure    *
//*     is being phased out.                                                     *
//*                                                                              *
//* v: 0.0.10 19-Mar-2011                                                        *
//*   - Replace manual conversion from UTF-8 to wchar_t and from wchar_t to      *
//*     cchar_t with the gString class functionality which implicitly does       *
//*     these conversions. The technical details of UTF-8 conversion, wchar_t    *
//*     conversion, and calculation of needed display columns is encapsulated    *
//*     in the gString class. Please refer to gString.hpp and gString.cpp for    *
//*     more information.                                                        *
//*   - For performance reasons we call the mvwaddch() primitive for line        *
//*     drawing and clearing the window. However, except for the alternate       *
//*     character set, for any non-space characters, we must use the wadd_wch()  *
//*     primitive to accomodate supra-ASCII output.                              *
//*   - All NCurses class methods verified for UTF-8 output support as of        *
//*     26 March, 2011.                                                          *
//*                                                                              *
//* v: 0.0.09 19-Feb-2011                                                        *
//*   - Move version history to NCurses.cpp.                                     *
//*   - Add support for 'wide' character output and multi-byte key input.        *
//*     This is a necessary step for internationalization because the 'narrow'   *
//*     library, ncurses, has only minimal support for wide character data.      *
//*   - New method: WriteString () overloads the ASCII char output method and    *
//*     takes pointer to an array of wchar_t characters.                         *
//*   - Note that we have provided only minimal support for 'wide'               *
//*     (supra-ASCII) characters in the NCurses class because no one would use   *
//*     this class as a base for serious code without also including the         *
//*     NcWindow or NcDialog class as well.                                      *
//*   - Add methods SetLocale() and GetLocale() to support the use of locale     *
//*     settings in interpretation of multi-byte character I/O. To support       *
//*     this, two includes were added to GlobalDef.hpp: <locale> and             *
//*     <langinfo.h>.                                                            *
//*   - IMPORTANT NOTE: The NCurses class knows nothing about UTF-8 character    *
//*     encoding (or any other encoding scheme). Please see the NcWindow class   *
//*      and NcDialog class as well as the gString class for more information    *
//*     on UTF-8 awareness.                                                      *
//*   - Removed the following temporary pragma from NCurses.hpp. All relevant    *
//*     code has been brought into conformance with the changes to the C++       *
//*     specification, so it is no longer necessary.                             *
//*               #pragma GCC diagnostic ignored "-Wwrite-strings"               *
//*   - All our classes now compile cleanly using the '-Wall' argument to g++.   *
//*     Hurray for us!                                                           *
//*                                                                              *
//* v: 0.0.08 07-Feb-2010                                                        *
//*   - Change development platform and compiler version.                        *
//*     Fedora 12 and GNU G++ (Gcc v: 4.4.2)                                     *
//*     This neccessitated several syntax changes in class and method            *
//*     definitions to accomodate changes in the C++ definition and the GCC      *
//*     compiler's warning and error messages.                                   *
//*     - Conversion of a string constant to a char* has been deprecated for     *
//*       the new version of GCC. Thus, all the variations on WriteString()      *
//*       and other methods that take a char* parameter yielded a warning:       *
//*       "deprecated conversion from string constant to ‘char*’"  To overcome   *
//*       this we could have added the line:                                     *
//*                 #pragma GCC diagnostic ignored "-Wwrite-strings"             *
//*       or we could have declared all those methods with const char*           *
//*       parameters. NOTE: This issue is not fully resolved.                    *
//*   - Repair the SetKeyDelay() and GetKeyInput16() methods.                    *
//*   - Implement escape sequence capture and translation.                       *
//*   - Add ColorText_Available() method so application does not need to call    *
//*     the primitives.                                                          *
//*   - Changed startup default ncKeyProc state to nckpRAW.                      *
//*   - For GetKeyInput16() and UngetKeyInput16(): parameter changed from        *
//*     USHORT to short.                                                         *
//*   - Add compiler option for using the ISO C++ extensions currently under     *
//*     development. Specifically, ISO document N2672=08-0182 to allow           *
//*     initializer lists for instantiation of arrays of classes or structs.     *
//*     G++ supports this as of version 4.3.                                     *
//*   - Add winPos class to take advantage of the ISO C++ extensions described   *
//*     above.                                                                   *
//*   - Add Get_nclibrary_Version() to return the version number of the ncurses  *
//*     (curses) library upon which the NCurses class is built.                  *
//*   - Convert #define'd values to const values for better compiler             *
//*     diagnostics.                                                             *
//*   - Define constants for Alternate Character Set character code values       *
//*     (acsXXXXXX).                                                             *
//*                                                                              *
//* v: 0.0.06 07-Apr-2007                                                        *
//*   - Change name of class version string. Add method Get_NCurses_Version().   *
//* v: 0.0.05 17-Sep-2006                                                        *
//*   - Delete DrawBox() method for borders.                                     *
//*   - Repair development-only methods.                                         *
//* v: 0.0.04 29-Dec-2005                                                        *
//*   - Add ClearLine method                                                     *
//* v: 0.0.03 18-Dec-2005                                                        *
//*   - Add const definitions for color attribute bitmasks                       *
//* v: 0.0.02 12-Oct-2005                                                        *
//*   - Add color variables, replacing hard-coded definitions.                   *
//* v: 0.0.01 02-Sep-2005 Original release                                       *
//*   - Created using GNU G++ (Gcc v: 3.2.2)                                     *
//*     under RedHat Linux, kernel 2.4.20-31.9                                   *
//********************************************************************************
//* Programmer's Notes:                                                          *
//* - In NCurses world, OK == 0 and ERR == (-1)                                  *
//*                     'TRUE' == 1 and 'FALSE' == 0                             *
//*   In C/C++ world,   'false' == 0 and 'true' == !false (in practice == 1)     *
//*                                                                              *
//* - IMPORTANT NOTE: The ncurses C-language library is not thread safe;         *
//*   thus, the NCurses class is not thread safe.                                *
//*   Because applications will seldom, if ever, use NCurses-class I/O routines  *
//*   directly, this is not a major problem. Just be aware.                      *
//*                                                                              *
//********************************************************************************

//*****************
//* Include Files *
//*****************
#define DEFINE_NC_GLOBALS_HERE 1
#include "GlobalDef.hpp"               //* General definitions
#undef DEFINE_NC_GLOBALS_HERE

#ifndef NCURSES_INCLUDED
#include "NCurses.hpp"
#endif

//*********************
//* Local Definitions *
//*********************

//** Local Data **
static const char NCursesVersion[] = "0.0.25" ; //* Class version number
static bool NCurses_Initialized = false ;       //* Set true by call to StartNCursesEngine()
static bool NColors_Initialized = false ;       //* Set true by call to StartColorEngine()

//* Contains input/output locale setting from terminal environment OR          *
//* from explicit setting by SetLocale().                                      *
static locale* ioLocale = NULL ;



//*************************
//*       NCurses         *
//*************************
//******************************************************************************
//* Default constructor.                                                       *
//*                                                                            *
//* There may be only one instance of this class in an application and it      *
//* should persist until just before the application terminates.               *
//* More than one instance may terminally confuse the terminal.                *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

NCurses::NCurses ( void )
{
   //* These variables are initialized in StartNCursesEngine() *
   this->stdscrAddress = NULL ;                 // (NCurses engine not yet active)
   this->sRows = this->sCols = ZERO ;           // size of initial screen
   this->maxColors = this->maxRgb =             // terminal's color capabilities
   this->maxPairs = ZERO ;
   this->modRgb = this->modPairs = false ;
   this->cursorState = this->cursorRestore = nccvNORMAL ; // (inherited from terminal)
   this->keyProcState = nckpCOOKED ;            // (inherited from terminal)
   this->inputEchoEnabled = false ;             // (inherited from terminal)
   this->outputScroll = false ;                 // (inherited from terminal)
   this->keyInputDelay = nckdFOREVER ;          // (inherited from terminal)
   this->mouseEnabled = false ; // ncurses mouse driver is disabled by default
   this->mouseFilter = true ;   // filtering of spurious events is enabled by default
   this->mouseMask = this->oldMouseMask = ZERO ;
   *this->nclibVersion = NULLCHAR ;             // initialized after NCurses engine started

   //* Initialize all color variables for monochrome display.*
   //* black-on-white, or white-on-black (system dependent)  *
   this->InitColorVariables ( false ) ;

}  //* End NCurses() *

//*************************
//*      ~NCurses         *
//*************************
//******************************************************************************
//* Destructor.                                                                *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

NCurses::~NCurses ( void )
{
   
   if ( NCurses_Initialized != false )
   {
      NCurses_Initialized = false ;
      endwin () ;       // Shut down the NCurses engine

      if ( ioLocale != NULL ) // release our local copy of the locale setting
         delete ( ioLocale ) ;
   }
}  //* End ~NCurses() *

//*************************
//*  StartNCursesEngine   *
//*************************
//******************************************************************************
//* Initialize the NCurses engine for text-based application support.          *
//*  Must be called after instantiation of the NCurses class (which happens    *
//*  automagically when the application is executed) but before any other      *
//*  method of this class. See examples in test applications.                  *
//* Default Settings On Startup:                                               *
//*  Locale: set according to terminal window's environment                    *
//*  Cursor shape: nccvINVISIBLE                                               *
//*  Key processing: nckpRAW                                                   *
//*  Key echo: disabled                                                        *
//*  Key delay: nckdFOREVER                                                    *
//*  Keycode width: nckbEIGHT (note that UTF-8 encoding requires 8-bit width)  *
//*  Escape sequence translation: enabled                                      *
//*  Delay after ESC key detected: none                                        *
//*  Display colors: If a color-map pointer is not specified, the NCurses      *
//*                  default color mapping is used.                            *
//*                                                                            *
//* Input  : autostartColorEngine: (optional, true by default)                 *
//*             - if 'true' (AND if terminal supports color output), call the  *
//*               StartColorEngine() method before return to caller            *
//*             - if 'false', do not start the Color Engine, only 2-color      *
//*               output will be available unless StartColorEngine() is called *
//*               before any color output is performed.                        *
//*          colorMap: (optional, NULL* pointer by default)                    *
//*             - if autostartColorEngine != false AND this pointer != NULL* , *
//*               then this is a pointer to a fully-initialized NcColorMap     *
//*               object containing the color-pair and RGB values for the      *
//*               available color attributes.                                  *
//*               - If colorMap.rgbCount == actual number of RGB register sets *
//*                 supported by the terminal AND the terminal allows          *
//*                 modification of the RGB registers, then colorMap.rgbMap[]  *
//*                 data is applied to the registers.                          *
//*               - If colorMap.pairCount == actual number of color pairs      *
//*                 supported by the terminal AND the terminal allows          *
//*                 modification of the color pairs, then colorMap.pairMap[]   *
//*                 data is applied to the terminal's color pairs.             *
//*             - if autostartColorEngine != false AND this pointer == NULL* , *
//*               then the NCurses default color mapping will be used to       *
//*               used to initialize the Color Engine.                         *
//*                                                                            *
//* Returns: OK if successful                                                  *
//*          ERR if unable to gain access to terminal firmware (or emulator)   *
//*              OR if provided color data do not match terminal definition    *
//******************************************************************************
//* Note that when setting the default locale, we do not test for UTF-8        *
//* support. Our caller should call nc.VerifyLocale() upon return.             *
//* Without a UTF-8 locale, only ASCII I/O will be reliably supported.         *
//*                                                                            *
//* NOTE: If, in a future version, we want to support applications that output *
//*       to more than one terminal, we would need to call newterm()           *
//*       before-or-instead of initscr(), and close with delscreen().          *
//*       This needs some research...                                          *
//******************************************************************************

short NCurses::StartNCursesEngine ( bool autostartColorEngine, 
                                    NcColorMap* colorMap )
{
short    status = ERR ;

   if ( NCurses_Initialized == false )
   {
      //* Set global locale for character encoding from terminal environment.  *
      //* This must happen before we call ncurses initialization functions.    *
      ioLocale = new locale("") ;      // get locale from environment
      ioLocale->global( *ioLocale ) ;  // make it global

      //* Disable the terminal emulator program's automatic BiDi support.*
      this->BiDiEnable ( false ) ;

      if ( (this->stdscrAddress = initscr ()) != NULL ) // initialize ncurses engine
      {
         NCurses_Initialized = true ;              // remember that we're running
         this->ScreenDimensions ( this->sRows, this->sCols ) ;// get dimensions of stdscr
         this->SetCursorState ( nccvINVISIBLE ) ;  // default cursor visibility==invisible
         this->SetKeyProcState ( nckpRAW ) ;       // allow max key input through without delay
         this->KeyEcho ( false ) ;                 // disable key-input echo
         this->SetKeyDelay ( nckdFOREVER ) ;       // wait indefinitely for key data
         this->SetKeycodeWidth ( nckbEIGHT ) ;     // 8-bit key input
         keypad ( stdscr, true ) ;                 // enable escape-sequence translation
         notimeout ( stdscr, true ) ;              // no delay after ESC code
         this->Get_nclibrary_Version () ;          // be sure version gets captured

         //* Note that color parms may not be valid at this point. *
         //* These will be initialized in StartColorEngine().      *
         this->maxColors = 1 ;                     // maximum terminal colors (at least one)
         this->maxPairs = 1 ;                      // maximum terminal color-pairs (at least one)
         this->modRgb = this->modPairs = false ;   // terminal support for color modifications

         status = OK ;                             // return success
         
         //* If the terminal supports display of color text *
         if ( (this->ColorText_Available ()) )
         {
            //* If auto-start of Color Engine specified, initialize color      *
            //* mapping and color text output.                                 *
            if ( autostartColorEngine != false )
            {  //* Initialize the color registers for color output *
               status = this->StartColorEngine ( colorMap ) ;
            }
            else
            {
               //* Caller has specified that the color engine should not be    *
               //* started and that the application should start in monochrome *
               //* mode. However, true monochrome mode is unusable because only*
               //* the base eight(8) color pairs are initialized. What we do   *
               //* here is to start the color engine anyway, but with          *
               //* fully-initialized color pairs -- all set for the default    *
               //* foreground/background.                                      *
               //* Note that using monochrome output for display data written  *
               //* to use multi-color text may yield something semi-ugly, but  *
               //* less ugly than running it in actual monochrome mode.        *

               //***** DISABLE THIS SECTION TO START A COLOR-CAPABLE *****
               //*****        SYSTEM IN TRUE MONOCHROME MODE.        *****
               #if 1    // SIMULATED MONOCHROME MODE
               //* Start the Color Engine in its default mode, *
               //* and get a copy of the current color map.    *
               this->StartColorEngine () ;
               NcColorMap monoMap ;
               this->GetColorMap ( monoMap ) ;

               //* Initialize the color pairs for monochrome output.*
               for ( short i = ZERO ; i < monoMap.pairCount ; i++ )
               {
                  monoMap.pairMap[i].fgnd = ncbcDEFAULT ;
                  monoMap.pairMap[i].bkgnd = ncbcDEFAULT ;
               }
               this->StartColorEngine ( &monoMap ) ;
               #endif   // SIMULATED MONOCHROME MODE
            }
         }
         else
         { /* Color text output not supported */ }
      }
   }
   
   return status ;
   
}  // End StartNCursesEngine() *

//*************************
//*   StopNCursesEngine   *
//*************************
//******************************************************************************
//* Shut down the NCurses Engine and release all associated resources.         *
//* (to be called just before driving application exits.)                      *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note: What, if anything, does the ncurses library do with the *
//* RGB regester settings and color pairs upon exit? If we leave them in       *
//* whatever state on exit, does this have any lasting effect on applications  *
//* that subsequently run in the terminal? Should we return the color data to  *
//* the start-up state?                                                        *
//******************************************************************************

void NCurses::StopNCursesEngine ( void )
{

   this->RestoreCursorState () ;       // make cursor visible (redundant)
   endwin () ;                         // disable ncurses
   this->BiDiEnable ( true ) ;         // re-enable terminal's BiDi support
   NCurses_Initialized = false ;       // be tidy
   NColors_Initialized = false ;


}  //* End StopNCursesEngine() *

//*************************
//*   StartColorEngine    *
//*************************
//******************************************************************************
//* Initialize the NCurses color engine for color text output.                 *
//* This method is called automatically as part of the StartNCursesEngine()    *
//*                               UNLESS                                       *
//* the application specifically overrides the default start-up sequence:      *
//* (autostartColorEngine parameter == false) to start the NCurses Engine in   *
//* monochrome mode.                                                           *
//*                                                                            *
//* If multi-color output is desired, but automatic Color Engine startup was   *
//* disabled in the call to StartNCursesEngine(), then call this method        *
//* immediately after a successful call to StartNCursesEngine() OR at the      *
//* latest, before any color-text output is performed.                         *
//*                                                                            *
//*                                                                            *
//* Input  : colorMap: (optional, NULL* pointer by default)                    *
//*             - if specified, then this is a pointer to a fully-initialized  *
//*               NcColorMap object containing the RGB register set values     *
//*               and/or the color-pair values.                                *
//*               - to set RGB registers: set colorMap.rgbCount == number of   *
//*                 register sets supported by the terminal                    *
//*                 (see the TerminalColorSupport() method)                    *
//*               - to set Color Pairs: set colorMap.pairCount == number of    *
//*                 color pairs supported by the terminal                      *
//*             - if colorMap not specified, then the NCurses default color    *
//*               mapping will be used to initialize the Color Engine.         *
//*                                                                            *
//* Returns: OK if successful                                                  *
//*          ERR if unable to gain access to terminal firmware (or emulator)   *
//*              OR if provided color data do not match terminal definition    *
//*              OR if NCurses engine previously initialized                   *
//*              OR if color mapping data do not match the terminal definition *
//******************************************************************************
//* Programmer's Notes:                                                        *
//*  - The 'COLORS' definition in curses.h reports the actual number of        *
//*    colors available, NOT the number of RGB register sets. The square root  *
//*    of 'COLORS' should yield the number of RGB register sets.               *
//*       256 colors == 16 RGB, 64 colors == 8 RGB, 16 colors == 4 RGB         *
//*    For other values (e.g. 32, 128) square root is not an integer.          *
//*                                                                            *
//*  - The 'COLOR_PAIRS' definition in curses.h reports the actual number of   *
//*    foreground/background color pairs available for mapping to specific     *
//*    colors.                                                                 *
//*                                                                            *
//*  - Color pair zero (normally white-on-black) cannot be set using the       *
//*    init_pair() function.  However, an ncurses extension uses the number    *
//*    minus one (-1) to indicate the terminal definition's default foreground *
//*    and background colors.                                                  *
//*    Color pair zero can be set to the terminal default colors by calling    *
//*    either use_default_colors() OR by calling assume_default_colors(-1,-1). *
//*    The assume_default_colors() function may also be used to set the color  *
//*    pair zero foreground and background to any of the available colors.     *
//*      Example: assume_default_colors( ncbcBL, ncbcGY );                     *
//*                                                                            *
//*  - The Linux system terminal swaps the color pair zero foreground and      *
//*    background by default i.e. the terminal displays text as black on       *
//*    slightly off-white, or what the documentation calls light yellow.       *
//*    This presents positive and negative aspects:                            *
//*     a. We need to keep the default almost-white background, since the      *
//*        ncurses engine does not allow us to specify a BOLD color as the     *
//*        background and may not allow us to modify the (virtual) RGB         *
//*        registers, so this is the only way I know of to get a bright white  *
//*        background (see note above).                                        *
//*     b. On the other hand, since we have one swapped pair, we need to       *
//*        balance it by swapping another pair to retain access to the         *
//*        maximum number of color combinations.  This can cause significant   *
//*        confusion when creating aliases for naming the various color        *
//*        combinations.                                                       *
//*    By default, we keep the system-provided near-white background on color  *
//*    pairs 0-7 although the application can override this, either by passing *
//*    us an NcColorMap object OR by calling SetDefaultBackground().           *
//*                                                                            *
//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *
//* Bit Definitions for a 32-bit character/attribute value (7-bit ASCII).      *
//* This are the internal bit definitions for 'chtype' (unsigned long), also   *
//* known as 'attr_t'. (see curses.h)                                          *
//* (Note that 'wide' character output is handled internally by the ncurses  ) *
//* (library, so we don't know how the bits are defined for wchar_characters;) *
//* (however, the color attribute bits are the same for both narrow and wide.) *
//*  BIT  31    = unused    -----------+                                       *
//*  BIT  30    = vertical             |                                       *
//*  BIT  29    = top                  |                                       *
//*  BIT  28    = right                +-- we make no use of these             *
//*  BIT  27    = low                  |   definitions in our implementation   *
//*  BIT  26    = left                 |                                       *
//*  BIT  25    = horizontal           |                                       *
//*  BIT  24    = "protected" mode ----+                                       *
//*  BIT  23    = invisible (background color is also used for foreground)     *
//*  BIT  22    = alternate character set (with line draw characters, etc.)    *
//*  BIT  21    = bold foreground against default background                   *
//*  BIT  20    = dim                                                          *
//*  BIT  19    = blinking (not functional under most terminal implementations)*
//*  BIT  18    = reverse (foreground/background swap)                         *
//*  BIT  17    = underline                                                    *
//*  BIT  16    = standout (foreground-to-background, with fg & bg bolded)     *
//*  BITS 15-8  = color pair number  (8, 16, 64, 256.  see init_pair() )       *
//*  BITS 7-0   = character code                                               *
//*                                                                            *
//* Refer to ISO 6429 for a description of color support for terminals.        *
//*                                                                            *
//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *
//* The NcColorMap class data members are of a static size, so regardless of   *
//* the actual number of colors and color pairs supported by the terminal, we  *
//* report no more colors and color pairs than will fit into the NcColorMap    *
//* arrays. Actually, the ncurses library seems to have some internal limits   *
//* of its own; specifically, that only 8 base colors are supported, (see      *
//* enum NcBaseColors and that 8 color pairs are initialized. The upper limit  *
//* is established by the eight (8) attr_t bits reserved for color-pair use.   *
//*                                                                            *
//* IMPORTANT NOTE:                                                            *
//*  The number of color pairs available is system dependent.                  *
//*  - Lenovo T500, 32-bit Wintel, Fedora 12, Gnome terminal v:2.28.2),        *
//*    there are 64 color pairs reported by the ncurses library via the global *
//*    variable, COLOR_PAIRS.                                                  *
//*  - Lenovo W540, 64-bit Wintel, Fedora 20, Gnome terminal v:3.10.2),        *
//*    there are 256 color pairs reported by the ncurses library via the       *
//*    global variable, COLOR_PAIRS.                                           *
//*                                                                            *
//*  If the color pair code is correctly positioned in the attr_t value passed *
//*  to WriteChar() / WriteString(), then the color will be accurately         *
//*  displayed using the foreground/background specified for that color pair.  *
//*  This uses bits 15-8 of the attr_t value. The consequence is that we can   *
//*  directly display up to 256 different attribute combinations, but of       *
//*  course, not all 256 pairs would be visually acceptable. We directly define*
//*  colors for ncSUPPORTED_PAIRS (currently 128) color pairs only, and set    *
//*  the remainder (if any) to the default foreground/background color.        *
//*                                                                            *
//* IMPORTANT NOTE: Setting values other than (-1, -1) for color pair ZERO     *
//*                 yields anomalous results.                                  *
//*  1) The background color declared in pair ZERO becomes the default         *
//*     background for ALL color pairs i.e. it it referenced by a (-1)         *
//*     background set in any other pair. While this makes sense, it needs to  *
//*     be kept in mind.                                                       *
//   2) However, if background of pair ZERO is set to (-1), then the terminal  *
//      default becomes the default background for all pairs.                  *
//   3) If foreground of color-pair ZERO is set to something other than (-1),  *
//      BUT color-pair ZERO's background is (-1), then a BUG condition exists: *
//      - Text color remains the terminal default, although when ncrATTR is    *
//        applied, the background shows as the specified foreground.           *
//   4) Only if NEITHER foreground NOR background of color-pair ZERO is set    *
//      to (-1), will the colors map as expected.                              *
//   5) When foreground of pair ZERO != (-1), then init_pair() library function*
//*     sometimes fails on other pairs because setting the foreground on       *
//*     another pair seems to be dependent in some way on the foreground of    *
//*     pair ZERO. If the other pair has either FG or BG set as (-1), then     *
//*     init_pair() will fail if pair ZERO is not also (-1, -1).               *
//*                                                                            *
//*        Further experimentation will determine the whole truth.             *
//******************************************************************************

short NCurses::StartColorEngine ( NcColorMap* colorMap )
{
   short    status = OK ; 

   if ( NColors_Initialized == false )
   {
      if ( (status = start_color ()) != ERR )
      {
         this->maxRgb    = (COLORS >= 256) ? 16 : (COLORS >= 64) ? 8 : 4 ;
         this->maxColors = COLORS ;
         this->maxPairs = COLOR_PAIRS <= ncPAIR_MAX ? COLOR_PAIRS : ncPAIR_MAX ;
         this->modRgb = can_change_color () ;
         this->modPairs = has_colors () ; // if there is color support, pairs can be modified
         this->InitColorVariables () ;
         NColors_Initialized = true ;
      }
   }

   if ( NColors_Initialized != false )
   {
      if ( colorMap != NULL )
      {
         //* Use caller's data to initialize either or both: *
         //* RGB register sets and Color Pairs.              *
         bool setPairs = this->modPairs && (colorMap->pairCount == this->maxPairs),
              setRGB   = this->modRgb && (colorMap->rgbCount == this->maxRgb) ;
         if ( setPairs || setRGB )
            this->SetColorMap ( colorMap, setPairs, setRGB ) ;
         else              // caller's data are invalid (nothing to be set)
            status = ERR ;

         //* If caller's data have not set up the color pairs, use defaults *
         if ( ! setPairs )
            this->InitColorPairs () ;
      }
      else
      {  //* No color map provided, use default color-pair settings *
         //* RGB register sets remain unmodified.                   *
         this->InitColorPairs () ;
      }
   }
   return status ;

}  //* End StartColorEngine() *

//*************************
//*    InitColorPairs     *
//*************************
//******************************************************************************
//* PRIVATE METHOD.                                                            *
//* Set default values for all color pairs.                                    *
//*                                                                            *
//* Input  : bkGround: (optional, ncbcDEFAULT by default)                      *
//*                    background color for the basic color pairs directly     *
//*                    supported by the ncurses library (usually pairs 0-7)    *
//*          initExt : (optional, 'true' by default)                           *
//*                    if 'true' initialize the 'extended' color pairs (8-63)  *
//*                     if these are supported by the terminal                 *
//*                    if 'false' do not initialize the 'extended' color pairs *
//*                     Note that this is 'false' only when called by the      *
//*                     public method, SetDefaultBackground().                 *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void NCurses::InitColorPairs ( NcBaseColors bkGround, bool initExt )
{
   assume_default_colors ( ncbcDEFAULT, bkGround ) ;  // color-pair ZERO

   //* If using the default background color, simply map *
   //* all foreground colors to their default positions. *
   if ( bkGround == ncbcDEFAULT )
   {
      init_pair ( ncbcRE, ncbcRE, bkGround ) ;  // Red color pair
      init_pair ( ncbcGR, ncbcGR, bkGround ) ;  // Green color pair
      init_pair ( ncbcBR, ncbcBR, bkGround ) ;  // Brown color pair
      init_pair ( ncbcBL, ncbcBL, bkGround ) ;  // Blue color pair
      init_pair ( ncbcMA, ncbcMA, bkGround ) ;  // Magenta color pair
      init_pair ( ncbcCY, ncbcCY, bkGround ) ;  // Cyan color pair
      init_pair ( ncbcGY, ncbcGY, bkGround ) ;  // Grey color pair
   }

   //* Non-default background color specified:                                 *
   //* If caller specifies a background color other than the default, then for *
   //* the color used, we must change the foreground of that color pair to     *
   //* avoid having a foreground and background of the same color,             *
   //* Red-on-Red, Blue-on-Blue, etc., i.e. invisible text.                    *
   else
   {  //* Note that if caller has specified a black background, then color     *
      //* pair zero may not look very good due to the terminal definition      *
      //* swapping the foreground/background on color pair zero.               *
      if ( bkGround == ncbcBK )
         assume_default_colors ( ncbcGY, bkGround ) ;
      if ( bkGround == ncbcRE )  init_pair ( ncbcRE, ncbcBK, bkGround ) ;
      else                       init_pair ( ncbcRE, ncbcRE, bkGround ) ;
      if ( bkGround == ncbcGR )  init_pair ( ncbcGR, ncbcBK, bkGround ) ;
      else                       init_pair ( ncbcGR, ncbcGR, bkGround ) ;
      if ( bkGround == ncbcBR )  init_pair ( ncbcBR, ncbcBK, bkGround ) ;
      else                       init_pair ( ncbcBR, ncbcBR, bkGround ) ;
      if ( bkGround == ncbcBL )  init_pair ( ncbcBL, ncbcBK, bkGround ) ;
      else                       init_pair ( ncbcBL, ncbcBL, bkGround ) ;
      if ( bkGround == ncbcMA )  init_pair ( ncbcMA, ncbcBK, bkGround ) ;
      else                       init_pair ( ncbcMA, ncbcMA, bkGround ) ;
      if ( bkGround == ncbcCY )  init_pair ( ncbcCY, ncbcBK, bkGround ) ;
      else                       init_pair ( ncbcCY, ncbcCY, bkGround ) ;
      if ( bkGround == ncbcGY )  init_pair ( ncbcGY, ncbcBK, bkGround ) ;
      else                       init_pair ( ncbcGY, ncbcGY, bkGround ) ;
   }                                                      

   //* Initialize additional pairs up to 64 total pairs (and up to 128 pairs   *
   //* if the hardware supports them. This provides the application with all   *
   //* the basic foreground/background color combinations.                     *
   if ( initExt != false )
   {
      short pair = ncbcCOLORS ;
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcBK, ncbcRE ) ; // black-on-red
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcBK, ncbcGR ) ; // black-on-green
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcBK, ncbcBR ) ; // black-on-brown
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcBK, ncbcBL ) ; // black-on-blue
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcBK, ncbcMA ) ; // black-on-magenta
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcBK, ncbcCY ) ; // black-on-cyan
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcBK, ncbcGY ) ; // black-on-grey
   
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcRE, ncbcBK ) ; // red-on-black
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcRE, ncbcGR ) ; // red-on-green
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcRE, ncbcBR ) ; // red-on-brown
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcRE, ncbcBL ) ; // red-on-blue
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcRE, ncbcMA ) ; // red-on-magenta
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcRE, ncbcCY ) ; // red-on-cyan
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcRE, ncbcGY ) ; // red-on-grey
   
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcGR, ncbcBK ) ; // green-on-black
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcGR, ncbcRE ) ; // green-on-red
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcGR, ncbcBR ) ; // green-on-brown
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcGR, ncbcBL ) ; // green-on-blue
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcGR, ncbcMA ) ; // green-on-magenta
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcGR, ncbcCY ) ; // green-on-cyan
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcGR, ncbcGY ) ; // green-on-grey
   
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcBR, ncbcBK ) ; // brown-on-black
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcBR, ncbcRE ) ; // brown-on-red
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcBR, ncbcGR ) ; // brown-on-green
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcBR, ncbcBL ) ; // brown-on-blue
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcBR, ncbcMA ) ; // brown-on-magenta
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcBR, ncbcCY ) ; // brown-on-cyan
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcBR, ncbcGY ) ; // brown-on-grey
   
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcBL, ncbcBK ) ; // blue-on-black
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcBL, ncbcRE ) ; // blue-on-red
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcBL, ncbcGR ) ; // blue-on-green
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcBL, ncbcBR ) ; // blue-on-brown
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcBL, ncbcMA ) ; // blue-on-magenta
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcBL, ncbcCY ) ; // blue-on-cyan
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcBL, ncbcGY ) ; // blue-on-grey
   
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcMA, ncbcBK ) ; // magenta-on-black
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcMA, ncbcRE ) ; // magenta-on-red
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcMA, ncbcGR ) ; // magenta-on-green
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcMA, ncbcBR ) ; // magenta-on-brown
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcMA, ncbcBL ) ; // magenta-on-blue
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcMA, ncbcCY ) ; // magenta-on-cyan
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcMA, ncbcGY ) ; // magenta-on-grey
   
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcCY, ncbcBK ) ; // cyan-on-black
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcCY, ncbcRE ) ; // cyan-on-red
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcCY, ncbcGR ) ; // cyan-on-green
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcCY, ncbcBR ) ; // cyan-on-brown
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcCY, ncbcBL ) ; // cyan-on-blue
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcCY, ncbcMA ) ; // cyan-on-magenta
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcCY, ncbcGY ) ; // cyan-on-grey
   
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcGY, ncbcBK ) ; // grey-on-black
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcGY, ncbcRE ) ; // grey-on-red
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcGY, ncbcGR ) ; // grey-on-green
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcGY, ncbcBR ) ; // grey-on-brown
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcGY, ncbcBL ) ; // grey-on-blue
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcGY, ncbcMA ) ; // grey-on-magenta
      if ( pair < this->maxPairs )  init_pair ( pair++, ncbcGY, ncbcCY ) ; // grey-on-cyan

      //* If terminal supports 16 or more colors, AND if we have space, *
      //* define terminal 'bright' colors 8 - 15 and their combinations.*
      if ( this->maxRgb >= ncCOLOR_MAX )
      {
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecBK, bkGround ) ; // bright black
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecRE, bkGround ) ; // bright red
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecGR, bkGround ) ; // bright green
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecBR, bkGround ) ; // bright brown
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecBL, bkGround ) ; // bright blue
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecMA, bkGround ) ; // bright magenta
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecCY, bkGround ) ; // bright cyan
#if 0    // DEFAULT BLACK ON LIGHT GREY
         if ( pair < this->maxPairs )  init_pair ( pair++, bkGround, ncecGY ) ; // bright grey (reversed)
#else    // STANDARD-BUT-INVISIBLE
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecGY, bkGround ) ; // bright grey
#endif   // STANDARD-BUT-INVISIBLE

         if ( pair < this->maxPairs )  init_pair ( pair++, ncecBK, ncecRE ) ; // black-on-red
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecBK, ncecGR ) ; // black-on-green
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecBK, ncecBR ) ; // black-on-brown
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecBK, ncecBL ) ; // black-on-blue
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecBK, ncecMA ) ; // black-on-magenta
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecBK, ncecCY ) ; // black-on-cyan
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecBK, ncecGY ) ; // black-on-grey
   
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecRE, ncecBK ) ; // red-on-black
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecRE, ncecGR ) ; // red-on-green
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecRE, ncecBR ) ; // red-on-brown
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecRE, ncecBL ) ; // red-on-blue
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecRE, ncecMA ) ; // red-on-magenta
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecRE, ncecCY ) ; // red-on-cyan
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecRE, ncecGY ) ; // red-on-grey
      
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecGR, ncecBK ) ; // green-on-black
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecGR, ncecRE ) ; // green-on-red
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecGR, ncecBR ) ; // green-on-brown
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecGR, ncecBL ) ; // green-on-blue
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecGR, ncecMA ) ; // green-on-magenta
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecGR, ncecCY ) ; // green-on-cyan
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecGR, ncecGY ) ; // green-on-grey
      
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecBR, ncecBK ) ; // brown-on-black
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecBR, ncecRE ) ; // brown-on-red
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecBR, ncecGR ) ; // brown-on-green
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecBR, ncecBL ) ; // brown-on-blue
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecBR, ncecMA ) ; // brown-on-magenta
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecBR, ncecCY ) ; // brown-on-cyan
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecBR, ncecGY ) ; // brown-on-grey
      
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecBL, ncecBK ) ; // blue-on-black
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecBL, ncecRE ) ; // blue-on-red
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecBL, ncecGR ) ; // blue-on-green
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecBL, ncecBR ) ; // blue-on-brown
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecBL, ncecMA ) ; // blue-on-magenta
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecBL, ncecCY ) ; // blue-on-cyan
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecBL, ncecGY ) ; // blue-on-grey
      
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecMA, ncecBK ) ; // magenta-on-black
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecMA, ncecRE ) ; // magenta-on-red
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecMA, ncecGR ) ; // magenta-on-green
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecMA, ncecBR ) ; // magenta-on-brown
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecMA, ncecBL ) ; // magenta-on-blue
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecMA, ncecCY ) ; // magenta-on-cyan
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecMA, ncecGY ) ; // magenta-on-grey
      
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecCY, ncecBK ) ; // cyan-on-black
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecCY, ncecRE ) ; // cyan-on-red
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecCY, ncecGR ) ; // cyan-on-green
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecCY, ncecBR ) ; // cyan-on-brown
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecCY, ncecBL ) ; // cyan-on-blue
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecCY, ncecMA ) ; // cyan-on-magenta
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecCY, ncecGY ) ; // cyan-on-grey

         if ( pair < this->maxPairs )  init_pair ( pair++, ncecGY, ncecBK ) ; // grey-on-black
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecGY, ncecRE ) ; // grey-on-red
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecGY, ncecGR ) ; // grey-on-green
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecGY, ncecBR ) ; // grey-on-brown
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecGY, ncecBL ) ; // grey-on-blue
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecGY, ncecMA ) ; // grey-on-magenta
         if ( pair < this->maxPairs )  init_pair ( pair++, ncecGY, ncecCY ) ; // grey-on-cyan
      }

      //* For all unused foreground/background color pairs. set to defaults.*
      short reportedPairs = COLOR_PAIRS ;
      while ( pair < reportedPairs )
         init_pair ( pair++, ncbcBK, bkGround ) ;
   }  // if(initExt)

}  //* End InitColorPairs() *

//*************************
//*  InitColorVariables   *
//*************************
//******************************************************************************
//* PRIVATE METHOD.                                                            *
//* Initialize all color-variable data members.                                *
//*                                                                            *
//* Input  : enableColor: (optional, 'true' by default)                        *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void NCurses::InitColorVariables ( bool enableColor )
{
   if ( enableColor != false )
   {
      this->bw = (ncbcBK << 8) ;                // Normal colors
      this->bk = this->bw | ncrATTR | ncbATTR ; //(synthesized swap of bw fg/bg)
      this->re = (ncbcRE << 8) ; this->gr = (ncbcGR << 8) ;
      this->br = (ncbcBR << 8) ; this->bl = (ncbcBL << 8) ; this->ma = (ncbcMA << 8) ;
      this->cy = (ncbcCY << 8) ; this->gy = (ncbcGY << 8) ;

      this->bwB = this->bw | ncbATTR ;    // Bold attribute
      this->bkB = this->bk | ncbATTR ; this->reB = this->re | ncbATTR ;
      this->grB = this->gr | ncbATTR ; this->brB = this->br | ncbATTR ;
      this->blB = this->bl | ncbATTR ; this->maB = this->ma | ncbATTR ;
      this->cyB = this->cy | ncbATTR ; this->gyB = this->gy | ncbATTR ;
   
      this->bwR = this->bw | ncrATTR ;    // Reverse attribute
      this->bkR = this->bk & ~(ncrATTR | ncbATTR) ; 
      this->reR = this->re | ncrATTR ; 
      this->grR = this->gr | ncrATTR ; this->brR = this->br | ncrATTR ;
      this->blR = this->bl | ncrATTR ; this->maR = this->ma | ncrATTR ;
      this->cyR = this->cy | ncrATTR ; this->gyR = this->gy | ncrATTR ;
   
      this->bwG = this->bwB |  ncrATTR ;  // Both Bold AND Reverse attributes
      this->bkG = this->bkB & ~ncrATTR ;
      this->reG = this->reB | ncrATTR ; 
      this->grG = this->grB | ncrATTR ; this->brG = this->brB | ncrATTR ;
      this->blG = this->blB | ncrATTR ; this->maG = this->maB | ncrATTR ;
      this->cyG = this->cyB | ncrATTR ; this->gyG = this->gyB | ncrATTR ;
   

      this->bwS = this->bw | ncsATTR ;    // Standout attribute
      this->bkS = (this->bk & ~ncbATTR) | ncsATTR ;
      this->reS = this->re | ncsATTR ;
      this->grS = this->gr | ncsATTR ; this->brS = this->br | ncsATTR ;
      this->blS = this->bl | ncsATTR ; this->maS = this->ma | ncsATTR ;
      this->cyS = this->cy | ncsATTR ; this->gyS = this->gy | ncsATTR ;
   
      this->bwD = this->bw | ncdATTR ;    // Dim attribute
      this->bkD = (this->bk & ~ncbATTR) | ncdATTR ;
      this->reD = this->re | ncdATTR ;
      this->grD = this->gr | ncdATTR ; this->brD = this->br | ncdATTR ; 
      this->blD = this->bl | ncdATTR ; this->maD = this->ma | ncdATTR ;
      this->cyD = this->cy | ncdATTR ; this->gyD = this->gy | ncdATTR ;
   
      this->bwU = this->bw | ncuATTR ;    // Underline attribute
      this->bkU = (this->bk & ~ncbATTR) | ncuATTR ; 
      this->reU = this->re | ncuATTR ; 
      this->grU = this->gr | ncuATTR ; this->brU = this->br | ncuATTR ; 
      this->blU = this->bl | ncuATTR ; this->maU = this->ma | ncuATTR ;
      this->cyU = this->cy | ncuATTR ; this->gyU = this->gy | ncuATTR ;
   }

   //* Initialize for a monochrome system *
   else
   {
      this->bw  = ncbcBK ;                // Standard colors
      this->bk  = this->re  = this->gr  = this->br = nc.bw ; 
      this->bl  = this->ma  = this->cy  = this->gy = nc.bw ;
   
      this->bwB = this->bw | ncbATTR ;    // Bold attribute
      this->bkB = this->reB = this->grB = this->brB = this->bwB ; 
      this->blB = this->maB = this->cyB = this->gyB = this->bwB ;
   
      this->bwR = this->bw | ncrATTR ;    // Reverse attribute
      this->bkR = this->reR = this->grR = this->brR = this->bwR ;
      this->blR = this->maR = this->cyR = this->gyR = this->bwR ;
   
      this->bwG = nc.bwB | ncrATTR ;      // Both Bold AND Reverse attributes
      this->bkG = this->reB = this->grB = this->brB = this->bwG ; 
      this->blG = this->maB = this->cyB = this->gyB = this->bwG ;
   
      this->bwS = nc.bw | ncsATTR ;       // Standout attribute
      this->bkS = this->reR = this->grS = this->brS = nc.bwS ;
      this->blS = this->maS = this->cyS = this->gyS = nc.bwS ;
   
      this->bwD = nc.bw | ncdATTR ;       // Dim attribute
      this->bkD = this->reD = this->grD = this->brD = this->bwD ;
      this->blD = this->maD = this->cyD = this->gyD = this->bwD ;
   
      this->bwU = nc.bw | ncuATTR ;       // Underline attribute
      this->bkU = this->reU = this->grU = this->brU = this->bwU ;
      this->blU = this->maU = this->cyU = this->gyU = this->bwU ;
   }

   //* Additional foreground/background combinations.             *
   //* For optimal contrast, values are initialized as follows:   *
   //* - The bkxx group and the gyxx group are initialized using  *
   //*   standard foreground colors on standard background colors.*
   //* - The other color groups are initialized using 'bold'      *
   //*   foreground colors on standard background colors.         *
   //* The naming scheme is based on 64 or more available color   *
   //* pairs. If the terminal supports fewer than 64 pairs, the   *
   //* extra values will be initialized to the terminal default   *
   //* color. For example output, please see the test application,*
   //* Dialog2, tests 4 and 6.                                    *
   short pairNum = ncbcCOLORS ;
   if ( pairNum < this->maxPairs ) { this->bkre = pairNum << 8 ; } ++pairNum ; // black-on-red
   if ( pairNum < this->maxPairs ) { this->bkgr = pairNum << 8 ; } ++pairNum ; // black-on-green
   if ( pairNum < this->maxPairs ) { this->bkbr = pairNum << 8 ; } ++pairNum ; // black-on-brown
   if ( pairNum < this->maxPairs ) { this->bkbl = pairNum << 8 ; } ++pairNum ; // black-on-blue
   if ( pairNum < this->maxPairs ) { this->bkma = pairNum << 8 ; } ++pairNum ; // black-on-magenta
   if ( pairNum < this->maxPairs ) { this->bkcy = pairNum << 8 ; } ++pairNum ; // black-on-cyan
   if ( pairNum < this->maxPairs ) { this->bkgy = pairNum << 8 ; } ++pairNum ; // black-on-grey
                                                              
   if ( pairNum < this->maxPairs ) { this->rebk = pairNum << 8 ; } ++pairNum ; // red-on-black
   if ( pairNum < this->maxPairs ) { this->regr = pairNum << 8 ; } ++pairNum ; // red-on-green
   if ( pairNum < this->maxPairs ) { this->rebr = pairNum << 8 ; } ++pairNum ; // red-on-brown
   if ( pairNum < this->maxPairs ) { this->rebl = pairNum << 8 ; } ++pairNum ; // red-on-blue
   if ( pairNum < this->maxPairs ) { this->rema = pairNum << 8 ; } ++pairNum ; // red-on-magenta
   if ( pairNum < this->maxPairs ) { this->recy = pairNum << 8 ; } ++pairNum ; // red-on-cyan
   if ( pairNum < this->maxPairs ) { this->regy = pairNum << 8 ; } ++pairNum ; // red-on-grey
                                                              
   if ( pairNum < this->maxPairs ) { this->grbk = pairNum << 8 ; } ++pairNum ; // green-on-black
   if ( pairNum < this->maxPairs ) { this->grre = pairNum << 8 ; } ++pairNum ; // green-on-red
   if ( pairNum < this->maxPairs ) { this->grbr = pairNum << 8 ; } ++pairNum ; // green-on-brown
   if ( pairNum < this->maxPairs ) { this->grbl = pairNum << 8 ; } ++pairNum ; // green-on-blue
   if ( pairNum < this->maxPairs ) { this->grma = pairNum << 8 ; } ++pairNum ; // green-on-magenta
   if ( pairNum < this->maxPairs ) { this->grcy = pairNum << 8 ; } ++pairNum ; // green-on-cyan
   if ( pairNum < this->maxPairs ) { this->grgy = pairNum << 8 ; } ++pairNum ; // green-on-grey
                                                              
   if ( pairNum < this->maxPairs ) { this->brbk = pairNum << 8 ; } ++pairNum ; // brown-on-black
   if ( pairNum < this->maxPairs ) { this->brre = pairNum << 8 ; } ++pairNum ; // brown-on-red
   if ( pairNum < this->maxPairs ) { this->brgr = pairNum << 8 ; } ++pairNum ; // brown-on-green
   if ( pairNum < this->maxPairs ) { this->brbl = pairNum << 8 ; } ++pairNum ; // brown-on-blue
   if ( pairNum < this->maxPairs ) { this->brma = pairNum << 8 ; } ++pairNum ; // brown-on-magenta
   if ( pairNum < this->maxPairs ) { this->brcy = pairNum << 8 ; } ++pairNum ; // brown-on-cyan
   if ( pairNum < this->maxPairs ) { this->brgy = pairNum << 8 ; } ++pairNum ; // brown-on-grey
                                                              
   if ( pairNum < this->maxPairs ) { this->blbk = pairNum << 8 ; } ++pairNum ; // blue-on-black
   if ( pairNum < this->maxPairs ) { this->blre = pairNum << 8 ; } ++pairNum ; // blue-on-red
   if ( pairNum < this->maxPairs ) { this->blgr = pairNum << 8 ; } ++pairNum ; // blue-on-green
   if ( pairNum < this->maxPairs ) { this->blbr = pairNum << 8 ; } ++pairNum ; // blue-on-brown
   if ( pairNum < this->maxPairs ) { this->blma = pairNum << 8 ; } ++pairNum ; // blue-on-magenta
   if ( pairNum < this->maxPairs ) { this->blcy = pairNum << 8 ; } ++pairNum ; // blue-on-cyan
   if ( pairNum < this->maxPairs ) { this->blgy = pairNum << 8 ; } ++pairNum ; // blue-on-grey
                                                              
   if ( pairNum < this->maxPairs ) { this->mabk = pairNum << 8 ; } ++pairNum ; // magenta-on-black
   if ( pairNum < this->maxPairs ) { this->mare = pairNum << 8 ; } ++pairNum ; // magenta-on-red
   if ( pairNum < this->maxPairs ) { this->magr = pairNum << 8 ; } ++pairNum ; // magenta-on-green
   if ( pairNum < this->maxPairs ) { this->mabr = pairNum << 8 ; } ++pairNum ; // magenta-on-brown
   if ( pairNum < this->maxPairs ) { this->mabl = pairNum << 8 ; } ++pairNum ; // magenta-on-blue
   if ( pairNum < this->maxPairs ) { this->macy = pairNum << 8 ; } ++pairNum ; // magenta-on-cyan
   if ( pairNum < this->maxPairs ) { this->magy = pairNum << 8 ; } ++pairNum ; // magenta-on-grey
                                                              
   if ( pairNum < this->maxPairs ) { this->cybk = pairNum << 8 ; } ++pairNum ; // cyan-on-black
   if ( pairNum < this->maxPairs ) { this->cyre = pairNum << 8 ; } ++pairNum ; // cyan-on-red
   if ( pairNum < this->maxPairs ) { this->cygr = pairNum << 8 ; } ++pairNum ; // cyan-on-green
   if ( pairNum < this->maxPairs ) { this->cybr = pairNum << 8 ; } ++pairNum ; // cyan-on-brown
   if ( pairNum < this->maxPairs ) { this->cybl = pairNum << 8 ; } ++pairNum ; // cyan-on-blue
   if ( pairNum < this->maxPairs ) { this->cyma = pairNum << 8 ; } ++pairNum ; // cyan-on-magenta
   if ( pairNum < this->maxPairs ) { this->cygy = pairNum << 8 ; } ++pairNum ; // cyan-on-grey
                                                              
   if ( pairNum < this->maxPairs ) { this->gybk = pairNum << 8 ; } ++pairNum ; // grey-on-black
   if ( pairNum < this->maxPairs ) { this->gyre = pairNum << 8 ; } ++pairNum ; // grey-on-red
   if ( pairNum < this->maxPairs ) { this->gygr = pairNum << 8 ; } ++pairNum ; // grey-on-green
   if ( pairNum < this->maxPairs ) { this->gybr = pairNum << 8 ; } ++pairNum ; // grey-on-brown
   if ( pairNum < this->maxPairs ) { this->gybl = pairNum << 8 ; } ++pairNum ; // grey-on-blue
   if ( pairNum < this->maxPairs ) { this->gyma = pairNum << 8 ; } ++pairNum ; // grey-on-magenta
   if ( pairNum < this->maxPairs ) { this->gycy = pairNum << 8 ; } ++pairNum ; // grey-on-cyan
                                                              
   // Programmer's Note: The output of a dark foreground on a dark background 
   // produces what is, at best, a murky display color. For this reason, we 
   // add the 'bold' attribute to the 'extended' values.
   // Exceptions:
   // - values with black foreground are clearer without the 'bold' attribute 
   // - values with grey foreground which would only be duplicates of the 
   //   basic set if 'bold' were added.
   this->rebk |= ncbATTR ; this->regr |= ncbATTR ; this->rebr |= ncbATTR ;
   this->rebl |= ncbATTR ; this->rema |= ncbATTR ; this->recy |= ncbATTR ;
   this->regy |= ncbATTR ;

   this->grbk |= ncbATTR ; this->grre |= ncbATTR ; this->grbr |= ncbATTR ;
   this->grbl |= ncbATTR ; this->grma |= ncbATTR ; this->grcy |= ncbATTR ;
   this->grgy |= ncbATTR ;

   this->brbk |= ncbATTR ; this->brre |= ncbATTR ; this->brgr |= ncbATTR ;
   this->brbl |= ncbATTR ; this->brma |= ncbATTR ; this->brcy |= ncbATTR ;
   this->brgy |= ncbATTR ;

   this->blbk |= ncbATTR ; this->blre |= ncbATTR ; this->blgr |= ncbATTR ;
   this->blbr |= ncbATTR ; this->blma |= ncbATTR ; this->blcy |= ncbATTR ;
   this->blgy |= ncbATTR ;

   this->mabk |= ncbATTR ; this->mare |= ncbATTR ; this->magr |= ncbATTR ;
   this->mabr |= ncbATTR ; this->mabl |= ncbATTR ; this->macy |= ncbATTR ;
   this->magy |= ncbATTR ;

   this->cybk |= ncbATTR ; this->cyre |= ncbATTR ; this->cygr |= ncbATTR ;
   this->cybr |= ncbATTR ; this->cybl |= ncbATTR ; this->cyma |= ncbATTR ;
   this->cygy |= ncbATTR ;

   //* For terminals that directly support 16 or more colors, *
   //* initialize the extended colors to the next eight color *
   //* pairs and their extended color-pair combinations.      *
   if ( (this->maxRgb >= ncCOLOR_MAX) && ((pairNum + ncbcCOLORS) <= this->maxPairs) )
   {
      this->bbk = pairNum << 8 ; ++pairNum ; // bright black   on default background
      this->bre = pairNum << 8 ; ++pairNum ; // bright red     on default background
      this->bgr = pairNum << 8 ; ++pairNum ; // bright green   on default background
      this->bbr = pairNum << 8 ; ++pairNum ; // bright brown   on default background
      this->bbl = pairNum << 8 ; ++pairNum ; // bright blue    on default background
      this->bma = pairNum << 8 ; ++pairNum ; // bright magenta on default background
      this->bcy = pairNum << 8 ; ++pairNum ; // bright cyan    on default background
      this->bgy = pairNum << 8 ; ++pairNum ; // bright grey    on default background
      
      if ( pairNum < this->maxPairs ) { this->bbkre = pairNum << 8 ; } ++pairNum ; // bright black-on-red
      if ( pairNum < this->maxPairs ) { this->bbkgr = pairNum << 8 ; } ++pairNum ; // bright black-on-green
      if ( pairNum < this->maxPairs ) { this->bbkbr = pairNum << 8 ; } ++pairNum ; // bright black-on-brown
      if ( pairNum < this->maxPairs ) { this->bbkbl = pairNum << 8 ; } ++pairNum ; // bright black-on-blue
      if ( pairNum < this->maxPairs ) { this->bbkma = pairNum << 8 ; } ++pairNum ; // bright black-on-magenta
      if ( pairNum < this->maxPairs ) { this->bbkcy = pairNum << 8 ; } ++pairNum ; // bright black-on-cyan
      if ( pairNum < this->maxPairs ) { this->bbkgy = pairNum << 8 ; } ++pairNum ; // bright black-on-grey
                                                                    
      if ( pairNum < this->maxPairs ) { this->brebk = pairNum << 8 ; } ++pairNum ; // bright red-on-black
      if ( pairNum < this->maxPairs ) { this->bregr = pairNum << 8 ; } ++pairNum ; // bright red-on-green
      if ( pairNum < this->maxPairs ) { this->brebr = pairNum << 8 ; } ++pairNum ; // bright red-on-brown
      if ( pairNum < this->maxPairs ) { this->brebl = pairNum << 8 ; } ++pairNum ; // bright red-on-blue
      if ( pairNum < this->maxPairs ) { this->brema = pairNum << 8 ; } ++pairNum ; // bright red-on-magenta
      if ( pairNum < this->maxPairs ) { this->brecy = pairNum << 8 ; } ++pairNum ; // bright red-on-cyan
      if ( pairNum < this->maxPairs ) { this->bregy = pairNum << 8 ; } ++pairNum ; // bright red-on-grey
                                                                    
      if ( pairNum < this->maxPairs ) { this->bgrbk = pairNum << 8 ; } ++pairNum ; // bright green-on-black
      if ( pairNum < this->maxPairs ) { this->bgrre = pairNum << 8 ; } ++pairNum ; // bright green-on-red
      if ( pairNum < this->maxPairs ) { this->bgrbr = pairNum << 8 ; } ++pairNum ; // bright green-on-brown
      if ( pairNum < this->maxPairs ) { this->bgrbl = pairNum << 8 ; } ++pairNum ; // bright green-on-blue
      if ( pairNum < this->maxPairs ) { this->bgrma = pairNum << 8 ; } ++pairNum ; // bright green-on-magenta
      if ( pairNum < this->maxPairs ) { this->bgrcy = pairNum << 8 ; } ++pairNum ; // bright green-on-cyan
      if ( pairNum < this->maxPairs ) { this->bgrgy = pairNum << 8 ; } ++pairNum ; // bright green-on-grey
                                                                    
      if ( pairNum < this->maxPairs ) { this->bbrbk = pairNum << 8 ; } ++pairNum ; // bright brown-on-black
      if ( pairNum < this->maxPairs ) { this->bbrre = pairNum << 8 ; } ++pairNum ; // bright brown-on-red
      if ( pairNum < this->maxPairs ) { this->bbrgr = pairNum << 8 ; } ++pairNum ; // bright brown-on-green
      if ( pairNum < this->maxPairs ) { this->bbrbl = pairNum << 8 ; } ++pairNum ; // bright brown-on-blue
      if ( pairNum < this->maxPairs ) { this->bbrma = pairNum << 8 ; } ++pairNum ; // bright brown-on-magenta
      if ( pairNum < this->maxPairs ) { this->bbrcy = pairNum << 8 ; } ++pairNum ; // bright brown-on-cyan
      if ( pairNum < this->maxPairs ) { this->bbrgy = pairNum << 8 ; } ++pairNum ; // bright brown-on-grey
                                                                    
      if ( pairNum < this->maxPairs ) { this->bblbk = pairNum << 8 ; } ++pairNum ; // bright blue-on-black
      if ( pairNum < this->maxPairs ) { this->bblre = pairNum << 8 ; } ++pairNum ; // bright blue-on-red
      if ( pairNum < this->maxPairs ) { this->bblgr = pairNum << 8 ; } ++pairNum ; // bright blue-on-green
      if ( pairNum < this->maxPairs ) { this->bblbr = pairNum << 8 ; } ++pairNum ; // bright blue-on-brown
      if ( pairNum < this->maxPairs ) { this->bblma = pairNum << 8 ; } ++pairNum ; // bright blue-on-magenta
      if ( pairNum < this->maxPairs ) { this->bblcy = pairNum << 8 ; } ++pairNum ; // bright blue-on-cyan
      if ( pairNum < this->maxPairs ) { this->bblgy = pairNum << 8 ; } ++pairNum ; // bright blue-on-grey
                                                                    
      if ( pairNum < this->maxPairs ) { this->bmabk = pairNum << 8 ; } ++pairNum ; // bright magenta-on-black
      if ( pairNum < this->maxPairs ) { this->bmare = pairNum << 8 ; } ++pairNum ; // bright magenta-on-red
      if ( pairNum < this->maxPairs ) { this->bmagr = pairNum << 8 ; } ++pairNum ; // bright magenta-on-green
      if ( pairNum < this->maxPairs ) { this->bmabr = pairNum << 8 ; } ++pairNum ; // bright magenta-on-brown
      if ( pairNum < this->maxPairs ) { this->bmabl = pairNum << 8 ; } ++pairNum ; // bright magenta-on-blue
      if ( pairNum < this->maxPairs ) { this->bmacy = pairNum << 8 ; } ++pairNum ; // bright magenta-on-cyan
      if ( pairNum < this->maxPairs ) { this->bmagy = pairNum << 8 ; } ++pairNum ; // bright magenta-on-grey

      if ( pairNum < this->maxPairs ) { this->bcybk = pairNum << 8 ; } ++pairNum ; // bright cyan-on-black
      if ( pairNum < this->maxPairs ) { this->bcyre = pairNum << 8 ; } ++pairNum ; // bright cyan-on-red
      if ( pairNum < this->maxPairs ) { this->bcygr = pairNum << 8 ; } ++pairNum ; // bright cyan-on-green
      if ( pairNum < this->maxPairs ) { this->bcybr = pairNum << 8 ; } ++pairNum ; // bright cyan-on-brown
      if ( pairNum < this->maxPairs ) { this->bcybl = pairNum << 8 ; } ++pairNum ; // bright cyan-on-blue
      if ( pairNum < this->maxPairs ) { this->bcyma = pairNum << 8 ; } ++pairNum ; // bright cyan-on-magenta
      if ( pairNum < this->maxPairs ) { this->bcygy = pairNum << 8 ; } ++pairNum ; // bright cyan-on-grey

      if ( pairNum < this->maxPairs ) { this->bgybk = pairNum << 8 ; } ++pairNum ; // bright grey-on-black
      if ( pairNum < this->maxPairs ) { this->bgyre = pairNum << 8 ; } ++pairNum ; // bright grey-on-red
      if ( pairNum < this->maxPairs ) { this->bgygr = pairNum << 8 ; } ++pairNum ; // bright grey-on-green
      if ( pairNum < this->maxPairs ) { this->bgybr = pairNum << 8 ; } ++pairNum ; // bright grey-on-brown
      if ( pairNum < this->maxPairs ) { this->bgybl = pairNum << 8 ; } ++pairNum ; // bright grey-on-blue
      if ( pairNum < this->maxPairs ) { this->bgyma = pairNum << 8 ; } ++pairNum ; // bright grey-on-magenta
      if ( pairNum < this->maxPairs ) { this->bgycy = pairNum << 8 ; } ++pairNum ; // bright grey-on-cyan

      #if 1    // It's not certain how helpful this is.
      // Programmer's Note: To produce optimal contrast, the 'bright color' 
      // combinations are given a Bold foreground color.
      // Exceptions: the bright-grey-background group.
      // Some combinations STILL have weak contrast. Can we do anything?
      //  bblbk, bmare, bcygr, bgybr, bbkbl, brema, bbrcy
      this->brebk |= ncbATTR ; this->bgrbk |= ncbATTR ; this->bbrbk |= ncbATTR ; 
      this->bblbk |= ncbATTR ; this->bmabk |= ncbATTR ; this->bcybk |= ncbATTR ; 
      this->bgybk |= ncbATTR ;

      this->bbkre |= ncbATTR ; this->bgrre |= ncbATTR ; this->bbrre |= ncbATTR ; 
      this->bblre |= ncbATTR ; this->bmare |= ncbATTR ; this->bcygr |= ncbATTR ; 
      this->bgyre |= ncbATTR ;

      this->bbkgr |= ncbATTR ; this->bregr |= ncbATTR ; this->bbrgr |= ncbATTR ; 
      this->bblgr |= ncbATTR ; this->bmagr |= ncbATTR ; this->bcygr |= ncbATTR ; 
      this->bgygr |= ncbATTR ;

      this->bbkbr |= ncbATTR ; this->brebr |= ncbATTR ; this->bgrbr |= ncbATTR ; 
      this->bblbr |= ncbATTR ; this->bmabr |= ncbATTR ; this->bcybr |= ncbATTR ; 
      this->bgybr |= ncbATTR ;

      this->bbkbl |= ncbATTR ; this->brebl |= ncbATTR ; this->bgrbl |= ncbATTR ; 
      this->bbrbl |= ncbATTR ; this->bmabl |= ncbATTR ; this->bcybl |= ncbATTR ; 
      this->bgybl |= ncbATTR ;

      this->bbkma |= ncbATTR ; this->brema |= ncbATTR ; this->bgrma |= ncbATTR ; 
      this->bbrma |= ncbATTR ; this->bblma |= ncbATTR ; this->bcyma |= ncbATTR ; 
      this->bgyma |= ncbATTR ;

      this->bbkcy |= ncbATTR ; this->brecy |= ncbATTR ; this->bgrcy |= ncbATTR ; 
      this->bbrcy |= ncbATTR ; this->bblcy |= ncbATTR ; this->bmacy |= ncbATTR ; 
      this->bgycy |= ncbATTR ;
      #endif   // It's not certain how helpful this is.
   }                                                                              
   //* If the terminal supports only 8 colors, then extended *
   //* colors are copies of the eight basic colors.          *
   else
   {
      this->bbk = this->bk ;
      this->bre = this->re ;
      this->bgr = this->gr ;
      this->bbr = this->br ;
      this->bbl = this->bl ;
      this->bma = this->ma ;
      this->bcy = this->cy ;
      this->bgy = this->gy ;

      this->bbkre = this->bkre ;
      this->bbkgr = this->bkgr ;
      this->bbkbr = this->bkbr ;
      this->bbkbl = this->bkbl ;
      this->bbkma = this->bkma ;
      this->bbkcy = this->bkcy ;
      this->bbkgy = this->bkgy ;

      this->brebk = this->rebk ;
      this->bregr = this->regr ;
      this->brebr = this->rebr ;
      this->brebl = this->rebl ;
      this->brema = this->rema ;
      this->brecy = this->recy ;
      this->bregy = this->regy ;

      this->bgrbk = this->grbk ;
      this->bgrre = this->grre ;
      this->bgrbr = this->grbr ;
      this->bgrbl = this->grbl ;
      this->bgrma = this->grma ;
      this->bgrcy = this->grcy ;
      this->bgrgy = this->grgy;

      this->bbrbk = this->brbk ;
      this->bbrre = this->brre ;
      this->bbrgr = this->brgr ;
      this->bbrbl = this->brbl ;
      this->bbrma = this->brma ;
      this->bbrcy = this->brcy ;
      this->bbrgy = this->brgy ;

      this->bblbk = this->blbk ;
      this->bblre = this->blre ;
      this->bblgr = this->blgr ;
      this->bblbr = this->blbr ;
      this->bblma = this->blma ;
      this->bblcy = this->blcy ;
      this->bblgy = this->blgy ;

      this->bmabk = this->mabk ;
      this->bmare = this->mare ;
      this->bmagr = this->magr ;
      this->bmabr = this->mabr ;
      this->bmabl = this->mabl ;
      this->bmacy = this->macy ;
      this->bmagy = this->magy ;

      this->bcybk = this->cybk ;
      this->bcyre = this->cyre ;
      this->bcygr = this->cygr ;
      this->bcybr = this->cybr ;
      this->bcybl = this->cybl ;
      this->bcyma = this->cyma ;
      this->bcygy = this->cygy ;

      this->bgybk = this->gybk ;
      this->bgyre = this->gyre ;
      this->bgygr = this->gygr ;
      this->bgybr = this->gybr ;
      this->bgybl = this->gybl ;
      this->bgyma = this->gyma ;
      this->bgycy = this->gycy ;
   }

}  //* End InitColorVariables() *

//*************************
//*     GetColorMap       *
//*************************
//******************************************************************************
//* Get a copy of the terminal's color-pair and RGB-register settings.         *
//* Each element of a color-pair is indicated as a member of:                  *
//*            enum NcBaseColors  OR  enum NcExtendedColors                    *
//* RGB register value range: minRGBvalue to maxRGBvalue (virtual values).     *
//*                                                                            *
//* Input  : cmapPtr: (by reference, initial values ignored)                   *
//*                   NcColorMap-class object to receive the color-pair and    *
//*                   RGB register data                                        *
//*                                                                            *
//* Returns: OK if successful,                                                 *
//*          ERR if Color Engine not started                                   *
//******************************************************************************

short NCurses::GetColorMap ( NcColorMap& cMap )
{
   short status = ERR ;

   //* If Color Engine not started, these are monochrome default values *
   cMap.pairCount = this->maxPairs ;
   cMap.rgbCount  = this->maxRgb ;
   cMap.supColors = this->maxColors ;     // (this is informational only)
   cMap.canModifyPairs = this->modPairs ;
   cMap.canModifyRGB   = this->modRgb ;

   if ( NColors_Initialized )
   {
      status = OK ;
      for ( short i = ZERO ; i < this->maxRgb ; i++ )
      {
         if ( (color_content ( i, &cMap.rgbMap[i].r, &cMap.rgbMap[i].g, 
                               &cMap.rgbMap[i].b )) == ERR )
         {
            while ( i < this->maxRgb )
            {  //* (valid RGB color range == 0 to 1000 according to ncurses docs) *
               cMap.rgbMap[i].r = cMap.rgbMap[i].g = 
               cMap.rgbMap[i].b = ZERO ;
               ++i ;
            }
            status = ERR ;
            break ;
         }
      }
      for ( short i = ZERO ; i < this->maxPairs ; i++ )
      {
         if ( (pair_content ( i, (short*)(&cMap.pairMap[i].fgnd), 
                              (short*)(&cMap.pairMap[i].bkgnd) ) ) == ERR )
         {
            while ( i < this->maxPairs )
            {
               cMap.pairMap[i].fgnd = cMap.pairMap[i].bkgnd = ncbcDEFAULT ;
               ++i ;
            }
            status = ERR ;
            break ;
         }
      }
   }
   return status ;

}  //* End GetColorMap() *

//*************************
//*     SetColorMap       *
//*************************
//******************************************************************************
//* Modify the terminal's foreground/background color pairs AND/OR modify      *
//* the RGB (Red-Green-Blue) register values (color hue).                      *
//*                                                                            *
//* The safe way to modify the color data is to first get a copy of the        *
//* existing color data and then modify it as needed.                          *
//*   NcColorMap colorMap ;                                                    *
//*   nc.GetColorMap ( colorMap ) ;                                            *
//*     [ perform desired modifications to the data ]                          *
//*   nc.SetColorMap ( colorMap, true, true ) ;                                *
//*                                                                            *
//* Input  : cmapPtr : pointer to an initialized NcColorMap-class object       *
//*                    containing the new color-pair AND/OR RGB register data  *
//*          setPairs: if 'true', use data provided in cmapPtr.pairMap[] array *
//*                    to modify ALL color pairs supported by the terminal     *
//*                     - Each value is a member of enum NcBaseColors or       *
//*                       enum NcExtendedColors.                               *
//*                     - Each terminal implementation has a 'default'         *
//*                       foreground and background color, usually             *
//*                       white-on-black OR black-on-white. Use of defaults    *
//*                       may be specified for either or both fgnd/bkgnd by    *
//*                       entering ncbcDEFAULT in the appropriate field(s).    *
//*          setRGB  : if 'true', use data provided in cmapPtr.rgbMap[] array  *
//*                    to modify ALL RGB registers supported by the terminal   *
//*                     - RGB register value range: minRGBvalue to maxRGBvalue.*
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
//******************************************************************************
//* Programmer's Note:                                                         *
//* RGB Registers:                                                             *
//* ==============                                                             *
//* - For the first eight (8) register sets, the default R, G, or B value is   *
//*   dfltRGBvalue (680 decimal, 2A8 hex).                                     *
//* - For register sets 8-15 (if supported by the terminal), the default       *
//*   R, G, or B value is maxRGBvalue (1000 decimal, 3E8 hex).                 *
//* - These values are of course subject to terminal setup.                    *
//* - Provided R, G and B values are range checked, and if out-of-range, is    *
//*   set to its default value.  Range: minRGBvalue <= value <= maxRGBvalue.   *
//*                                                                            *
//*                 BLUE  GREEN  RED          BLUE  GREEN  RED                 *
//*  COLOR    REG   VALUE VALUE VALUE   REG   VALUE VALUE VALUE                *
//*  Black     0     0000  0000  0000     8    0000  0000  0000                *
//*  Red       1     0000  0000  0680     9    0000  0000  1000                *
//*  Green     2     0000  0680  0000    10    0000  1000  0000                *
//*  Brown     3     0000  0680  0680    11    0000  1000  1000                *
//*  Blue      4     0680  0000  0000    12    1000  0000  0000                *
//*  Magenta   5     0680  0000  0680    13    1000  0000  1000                *
//*  Cyan      6     0680  0680  0000    14    1000  1000  0000                *
//*  Grey      7     0680  0680  0680    15    1000  1000  1000                *
//*                                                                            *
//* Color Pairs:                                                               *
//* ============                                                               *
//* - For color terminals there should be either eight-squared (64) register   *
//*   pairs, or 16-squared (256) register pairs.                               *
//*     Note: We trashed the last of our Cromemco and Wyse terminals in 1998,  *
//*           before this project started, but we must assume that the old     *
//*           dumb terminals have only one color register.                     *
//* - The ncurses library initializes ONLY the first eight (8) register pairs. *
//* - The NCurses class initializes all available color pairs by default.      *
//*   See InitColorPairs() method for more information.                        *
//* - The ncurses library uses a special value ncbcDEFAULT (-1) to indicate a  *
//*   default foreground or background.                                        *
//* - Provided foreground/background color pairs are range checked and if      *
//*   out-of-range, is set to it default value, ncbcDEFAULT.                   *
//* - The NCurses-class color variables ARE NOT modified by SetColorMap().     *
//*                                                                            *
//******************************************************************************

short NCurses::SetColorMap ( NcColorMap* cmapPtr, bool setPairs, bool setRGB )
{
   short status = ERR ;
   if ( NColors_Initialized && (setPairs || setRGB) )
   {
      status = OK ;
      //* Be sure caller isn't fooling us about the array size *
      cmapPtr->pairCount = this->maxPairs ;
      cmapPtr->rgbCount = this->maxRgb ;

      //* If caller wants to respecify the color pairs *
      if ( setPairs )
      {  //* Range check caller's data *
         short tf = cmapPtr->pairMap[ZERO].fgnd, tb = cmapPtr->pairMap[ZERO].bkgnd ;
         if ( tf < ncbcDEFAULT || tf >= this->maxRgb )
            tf = ncbcDEFAULT ;
         if ( tb < ncbcDEFAULT || tb >= this->maxRgb )
            tb = ncbcDEFAULT ;
         assume_default_colors ( tf, tb ) ;
         for ( short i = 1 ; i < cmapPtr->pairCount ; i++ )
         {
            tf = cmapPtr->pairMap[i].fgnd, tb = cmapPtr->pairMap[i].bkgnd ;
            if ( tf < ncbcDEFAULT || tf >= this->maxRgb )
               tf = ncbcDEFAULT ;
            if ( tb < ncbcDEFAULT || tb >= this->maxRgb )
               tb = ncbcDEFAULT ;
            if ( (init_pair ( i, tf, tb )) == ERR )
               status = ERR ;    // set the error flag but don't stop
         }
      }

      //* If caller wants to adjust the RGB registers *
      if ( setRGB )
      {
         short tr, tg, tb ;
         for ( short i = ZERO ; i < cmapPtr->rgbCount ; i++ )
         {
            //* Range check caller's data *
            tr = cmapPtr->rgbMap[i].r ;
            if ( tr < minRGBvalue || tr > maxRGBvalue )
               tr = dfltRGBvalue ;
            tg = cmapPtr->rgbMap[i].g ;
            if ( tg < minRGBvalue || tg > maxRGBvalue )
               tg = dfltRGBvalue ;
            tb = cmapPtr->rgbMap[i].b ;
            if ( tb < minRGBvalue || tb > maxRGBvalue )
               tb = dfltRGBvalue ;
            if ( (init_color ( i, tr, tg, tb )) == ERR )
               status = ERR ;    // set the error flag but don't stop
         }
      }
      this->RefreshScreen () ;   // be sure that the changes become visible
   }
   return status ;

}  //* End SetColorMap() *

//*************************
//*     SetColorMap       *
//*************************
//******************************************************************************
//* Modify a single foreground/background color-pair.                          *
//*                                                                            *
//* Input  : pairIndex: index of color pair to be modified                     *
//*          pairData : new foreground and background color numbers            *
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
//******************************************************************************

short NCurses::SetColorMap ( short pairIndex, const NcColorPair& pairData )
{
   short status = ERR ;
   if ( NColors_Initialized && (pairIndex < this->maxPairs) &&
        (pairData.fgnd >= ncbcDEFAULT && pairData.fgnd < this->maxRgb) &&
        (pairData.bkgnd >= ncbcDEFAULT && pairData.bkgnd < this->maxRgb) )
   {
      if ( pairIndex == ncbcBK )
         status = assume_default_colors ( pairData.fgnd, pairData.bkgnd ) ;
      else
         status = init_pair ( pairIndex, pairData.fgnd, pairData.bkgnd ) ;
      this->RefreshScreen () ;   // be sure that the changes become visible
   }
   return status ;

}  //* End SetColorMap() *

//*************************
//*     SetColorMap       *
//*************************
//******************************************************************************
//* Modify a single RGB register set.                                          *
//*                                                                            *
//* Input  : rgbIndex : index of RGB register set to be modified               *
//*          rgbData  : new red/green/blue register-set values                 *
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
//******************************************************************************

short NCurses::SetColorMap ( short rgbIndex, const NcRgbSet& rgbData )
{
   short status = ERR ;
   if ( NColors_Initialized && (rgbIndex < this->maxRgb) &&
        (rgbData.r >= minRGBvalue && rgbData.r <= maxRGBvalue) &&
        (rgbData.g >= minRGBvalue && rgbData.g <= maxRGBvalue) &&
        (rgbData.b >= minRGBvalue && rgbData.b <= maxRGBvalue) )
   {
      status = init_color ( rgbIndex, rgbData.r, rgbData.g, rgbData.b ) ;
      this->RefreshScreen () ;   // be sure that the changes become visible
   }
   return status ;

}  //* End SetColorMap() *

//**************************
//*  SetDefaultBackground  *
//**************************
//******************************************************************************
//* Establish a default background color to be used with the basic             *
//* text (foreground) colors.                                                  *
//*                                                                            *
//* Note that the results of this operation will likely be disappointing,      *
//* especially if a black background (ncbcBK) is specified.                    *
//* See also: SetColorMap().                                                   *
//*                                                                            *
//* Input  : bkGround : background color for the basic color pairs directly    *
//*                     supported by the ncurses library (usually pairs 0-7)   *
//*          pzfgnd   : (optional, ncbcDEFAULT by default)                     *
//*                     foreground text color for color pair zero (ncbcBK)     *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void NCurses::SetDefaultBackground ( NcBaseColors bkGround, NcBaseColors pzfgnd )
{
   //* Initialize the basic ncbcCOLORS (8) color pairs *
   this->InitColorPairs ( bkGround, false ) ;
   if ( pzfgnd != ncbcDEFAULT )
      assume_default_colors ( pzfgnd, bkGround ) ;  // color-pair ZERO

}  //* End SetDefaultBackground() *

//*************************
//*      Hibernate        *
//*************************
//******************************************************************************
//* Temporarily disable the NCurses engine so external terminal applications   *
//* can gain full access to the terminal's functionality.                      *
//*                                                                            *
//* Call Wake() or RefreshScreen() method to re-activate the NCurses engine.   *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void NCurses::Hibernate ( void )
{

   endwin () ;
   this->BiDiEnable ( true ) ;      // re-enable terminal's BiDi support


}  //* End Hibernate() *

//*************************
//*         Wake          *
//*************************
//******************************************************************************
//* Re-enable the NCurses engine after a previous call to Hibernate().         *
//* Also flushes any remaining data from the keyboard input stream.            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void NCurses::Wake ( void )
{

   this->BiDiEnable ( false ) ;     // disable terminal's BiDi support
   this->RefreshScreen () ;
   this->FlushKeyInputStream () ;

}  //* End Wake() *

//*************************
//*   Get_stdscr_Address  *
//*************************
//******************************************************************************
//* Returns a pointer to the WINDOW structure associated with the initial      *
//* screen for the application.                                                *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: address of WINDOW structure for stdscr, the ncurses default window*
//******************************************************************************

WINDOW* NCurses::Get_stdscr_Address ( void )
{

   return this->stdscrAddress ;

}  //* End Get_stdscr_Address() *

//***************************
//*  OutputStreamScrolling  *
//***************************
//******************************************************************************
//* Enable or Disable "standard output" style scrolling in the stdscr window.  *
//*                                                                            *
//* By default, the underlying 'ncurses' library disables scrolling.           *
//* The NCurses library also leaves scrolling disabled by default because it   *
//* produces ugly and undisciplined output.                                    *
//*                                                                            *
//* If, however, you have a need to emulate a common console application which *
//* is unaware of screen edges and data formatting then use this method to     *
//* enable/disable scrolling.                                                  *
//*                                                                            *
//* a) If the cursor reaches the right edge of the terminal window, the cursor *
//*    is automatically repositioned to the leftmost column of the following   *
//*    display line. (This is dumb-terminal character wrapping, NOT word wrap.)*
//* b) If the cursor reaches the right edge of the last display line in the    *
//*    terminal window, then all data in the window will be scrolled up by one *
//*    line, and the cursor will be repositioned to the leftmost column of the *
//*    last display line.                                                      *
//*                                                                            *
//* IMPORTANT INFORMATION ABOUT AUTOMATIC SCROLLING:                           *
//*  - The NCurses-class output methods 'WriteString' and 'WriteChar' are fully*
//*    aware of the terminal-window edges and therefore truncate output at the *
//*    edge of the window REGARDLESS of the state of Output-stream scrolling.  *
//*  - Only the 'WriteStream' method ignores the terminal window size and      *
//*    allows for automatic scrolling.                                         *
//*  - Scrolling output applies ONLY to the main terminal window and NOT to    *
//*    any sub-window objects within the terminal window. Therefore, do not    *
//*    define any sub-windows when output-stream scrolling is enabled. In      *
//*    practical terms, this means that you must not call the methods of the   *
//*    higher-level NcWindow and NcDialog classes while scrolling is enabled.  *
//*  - The cursor remains where is was placed after the last character in the  *
//*    string is written, so subsequent output will begin where the previous   *
//*    output left off unless the 'SetCursor' method is explicitly called.     *
//*    Note that any attempt to set the cursor at a position outside the       *
//*    terminal window will be ignored.                                        *
//*  - This functionality IS NOT THREAD SAFE. It is the application's          *
//*    responsibility to ensure that only one thread writes data to the output *
//*    stream.                                                                 *
//*  - Because output-stream scrolling emulates output to a dumb terminal,     *
//*    this method does not support RTL languages.                             *
//*                                                                            *
//*                                                                            *
//* Input  : enable : if 'true', enable automatic line wrap and scrolling      *
//*                   if 'false', disable automatic line wrap and scrolling    *
//*                                                                            *
//* Returns: OK                                                                *
//******************************************************************************

short NCurses::OutputStreamScrolling ( bool enable )
{
   this->outputScroll = enable ;
   idlok ( this->stdscrAddress, this->outputScroll ) ;
   scrollok ( this->stdscrAddress, this->outputScroll ) ;
   return OK ;

}  //* End OutputStreamScrolling() *

//*************************
//*      SetLocale        *
//*************************
//******************************************************************************
//* Specify a new locale setting for multi-byte character interpretation.      *
//*                                                                            *
//* Note that when the NCurses class is instantiated, the locale used is the   *
//* one specified by the terminal window's environment variables. If the       *
//* terminal window's locale is not a UTF-8 locale, then this method must be   *
//* called to specify a UTF-8-compliant locale before any I/O takes place.     *
//* If the locale is not UTF-8 compliant, then NCurses methods will not be able*
//* to handle characters beyond simple ASCII (7-bit) characters.               *
//*                                                                            *
//* Input  : string containing name of new locale                              *
//*           NOTE: For a complete list of locales supported by your terminal  *
//*                 window, type:  locale -a    at the command prompt.         *
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
//******************************************************************************

short NCurses::SetLocale ( const char* newLocaleName )
{
short status = ERR ;

   if ( newLocaleName != NULL && (this->VerifyLocale ( newLocaleName )) == OK )
   {
      locale tmp(newLocaleName) ;
      *ioLocale = tmp ;
      ioLocale->global( *ioLocale ) ;  // set global locale
      status = OK ;
   }
   return status ;

}  //* SetLocale() *

//*************************
//*      GetLocale        *
//*************************
//********************************************************************************
//* Returns a pointer to a const string containing the name of the currently-    *
//* active locale setting.                                                       *
//*                                                                              *
//* Input  : all  : (optional, default: false)                                   *
//*                 if false, report only the first locale in the string         *
//*                 if true,  report the full locale string                      *
//*                                                                              *
//* Returns: pointer to locale string                                            *
//********************************************************************************
//* Programmer's Note: We use a static buffer to capture the string whose        *
//* pointer is returned by the called library function. This is because the      *
//* pointer may be deleted before the calling method can copy the string.        *
//* We sincerely dislike std::string.                                            *
//*                                                                              *
//* Technical Note: In a default system configuration the locale specified is    *
//* the same for all locale sections (characters, numeric, date, etc).           *
//* However, it is possible that the locale is split among the sections, for     *
//* instance US English for currency and UK English for timestamps.              *
//* In this case, the data returned will be a concatenated sequence of entries,  *
//* one for each section of the locale specification.                            *
//* The entries of the list will be seperated by6 semicolons (';').              *
//* For simplicity, only the first (only) item in the list is reported by        *
//* default.                                                                     *
//********************************************************************************

const char* NCurses::GetLocale ( bool all )
{
   const short lcnBUFF = 128 ;   // static buffer size
   static char lcName[lcnBUFF] ; // query the system for current locale setting

   #if 0    // For Debugging Only
   strncpy ( lcName, setlocale (LC_ALL, NULL), lcnBUFF ) ;
   if ( (strncmp ( lcName, (ioLocale->name().c_str()), lcnBUFF )) )
   {
      strncpy ( lcName, "LOCALE ERROR!", lcnBUFF ) ;
      return lcName ;
   }
   #endif   // For Debugging Only

   int i ;                                // character index
   gString gsloc( ioLocale->name().c_str() ) ;
   if ( (i = gsloc.after( L'=' )) > ZERO )// remove (initial) tag name
      gsloc.shiftChars( -(i) ) ;

   //* Unless otherwise specified, retain only the first entry. *
   if ( ! all )
   {
      if ( (i = gsloc.find( L';' )) > ZERO ) // if multiple locale tags
         gsloc.limitChars( i ) ;             // remove all except the first value
   }
   gsloc.copy( lcName, lcnBUFF ) ;        // save the adjusted string

   return lcName ;

}  //* GetLocale() *

//*************************
//*     VerifyLocale      *
//*************************
//********************************************************************************
//* Attempts to verify whether the locale name provided specifies a locale       *
//* that supports UTF-8 encoding.                                                *
//*                                                                              *
//* We perform a very simple test: If the locale name contains one of the        *
//* following substrings (case insensitive) then UTF-8 support is assumed.       *
//*                       'UTF-8'    'utf8'                                      *
//*                                                                              *
//* Input  : (optional, NULL by default)                                         *
//*            NULL indicates that current local setting should be tested        *
//*            If non-NULL, test the locale name provided                        *
//*                                                                              *
//* Returns: OK if specified locale known to support UTF-8 encoding, else ERR    *
//********************************************************************************

short NCurses::VerifyLocale ( const char* lcName )
{
   short status = ERR ;    // return value

// CZONE - Use GetLocale() here to avoid possible loss of string data.
//   gString gs( ((lcName != NULL) ? lcName : ioLocale->name().c_str()), 30 ) ;
/* EXPERIMENTAL */ gString gs( ((lcName != NULL) ? lcName : this->GetLocale ()), 30 ) ;
   if ( ((gs.find( L"UTF-8" ) >= ZERO)) || ((gs.find( L"UTF8" )) >= ZERO) )
      status = OK ;

   return status ;

}  //* VerifyLocale() *

//*************************
//*     ClearScreen       *
//*************************
//********************************************************************************
//* Erase the entire contents of the terminal window.                            *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* Programmer's Note: The options for clearing the entire terminal window       *
//* are erase(), werase(), clear(), wclear(). They all do the same thing if      *
//* the window versions are called with 'curscr' or 'stdscrAddress'. The         *
//* only difference is that the clear() and wclear() also call clearok() so      *
//* that ANY call to refresh ANY window will cause the entire screen to be       *
//* cleared and redrawn with blank characters.                                   *
//********************************************************************************

void NCurses::ClearScreen ( void )
{

   werase ( this->stdscrAddress ) ;
   wrefresh ( this->stdscrAddress ) ;  // redraw the window

}  //* End ClearScreen() *

//*************************
//*     RefreshScreen     *
//*************************
//******************************************************************************
//* Refresh the entire contents of the terminal window.                        *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void NCurses::RefreshScreen ( void )
{
   
   touchwin ( this->stdscrAddress ) ;  // mark entire window as needing refreshed
   wrefresh ( this->stdscrAddress ) ;  // redraw the window

}  //* End RefreshScreen() *

//*************************
//*   ScreenDimensions    *
//*************************
//******************************************************************************
//* Returns the terminal window dimensions in lines and columns.               *
//*                                                                            *
//*                                                                            *
//* Input  : scrLines (by reference) receives screen line-count                *
//*          scrColumns (by reference) receives screen column-count            *
//*                                                                            *
//* Returns: nothing                                                           *
//*******************************************************************************

void NCurses::ScreenDimensions ( short& scrLines, short& scrColumns )
{
   
   getmaxyx ( this->stdscrAddress, this->sRows, this->sCols ) ;
   scrLines   = this->sRows ;
   scrColumns = this->sCols ;

}  //* End ScreenDimensions() *

//*************************
//* TerminalColorSupport  *
//*************************
//******************************************************************************
//* Report terminal (or terminal emulator) color output capabilities.          *
//* See 'termColorInfo' class for mor information.                             *
//*                                                                            *
//* Input  : cinfo : (by reference) receives terminal color information        *
//*                                                                            *
//* Returns: 'true' if the terminal supports multi-color text output           *
//*                 parameters will be initialized                             *
//*          'false' if the terminal does not support multi-color text output  *
//*                  OR if Color Engine has not yet been started i.e. info is  *
//*                     not yet available                                      *
//*                  (term_colors, rgb_regs and fgbg_pairs set to ZERO)        *
//*                  (mod_pairs and mod_rgb set to 'false')                    *
//******************************************************************************
//* Programmer's Note: See note in StartColorEngine() about limits on the      *
//* number of colors and color pairs reported to the application.              *
//******************************************************************************

bool NCurses::TerminalColorSupport ( termColorInfo& cinfo )
{
   bool statusFlag = false ; // returns 'true' if terminal supports color output

   //* If both NCurses Engine AND Color Engine have been *
   //* initialized, then requested data are available.   *
   if ( NCurses_Initialized && NColors_Initialized )
   {
      cinfo.term_colors = this->maxColors ;
      cinfo.rgb_regs    = this->maxRgb ;
      cinfo.fgbg_pairs  = this->maxPairs ;
      cinfo.mod_rgb     = this->modRgb ;
      statusFlag = cinfo.mod_pairs   = this->modPairs ;
   }
   else
   {  //* Data not available. Set to defaults *
      cinfo.term_colors =
      cinfo.rgb_regs = 
      cinfo.fgbg_pairs = ZERO ;
      cinfo.mod_pairs = cinfo.mod_rgb = false ;
   }
   return statusFlag ;
   
}  //* End TerminalColorSupport() *

//*************************
//*  ColorText_Available  *
//*************************
//******************************************************************************
//* Determine whether the terminal or terminal emulator is capable of          *
//* displaying color text. This will almost always be true on a computer, but  *
//* may return false if running on a monochrome dumb terminal.                 *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if the terminal supports multi-color text output           *
//*          'false' if the terminal does not support multi-color text output  *
//*                  OR if NCurses Engine has not yet been started i.e. info   *
//*                     is not yet available                                   *
//******************************************************************************

bool NCurses::ColorText_Available ( void )
{
bool statusFlag = false ;

   if ( NCurses_Initialized )
   {
      if ( (has_colors ()) )
         statusFlag = true ;
   }
   return statusFlag ;

}  //* End ColorText_Available() *

//**************************
//*  ColorText_Initialized *
//**************************
//******************************************************************************
//* Determine whether the Color Engine has been successfully initialized.      *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: true if Color Engine has been successfully initialized            *
//*          else false                                                        *
//******************************************************************************

bool NCurses::ColorText_Initialized ( void )
{
   return NColors_Initialized ;

}  //* End ColorText_Initialized() *

//*************************
//*      ClearLine        *
//*************************
//******************************************************************************
//* Clear the line. Sets all characters of specified line to SPACE character.  *
//* Terminal window is refreshed.                                              *
//*                                                                            *
//* Input  : ypos : line number (zero-based)                                   *
//*          cAttr: (optional, default==terminal background color: nc.bw       *
//*                                                                            *
//* Returns: OK if successful, ERR if invalid line index                       *
//******************************************************************************

short NCurses::ClearLine ( short ypos, attr_t cAttr )
{
   short status = ERR ;
   if ( ypos >= ZERO && ypos < this->sRows )
   {
      gString gs( L" " ) ;
      if ( cAttr == ZERO )
         cAttr = this->bw ;
      for ( short colIndex = ZERO ; colIndex < this->sCols ; colIndex++ )
         this->WriteChar ( ypos, colIndex, gs, cAttr ) ;
      this->RefreshScreen () ;
      status = OK ;
   }
   return status ;

}  //* End ClearLine() *

//***********************
//*     GetCursor       *
//***********************
//******************************************************************************
//* Get the NCurses main screen (terminal window) cursor position.             *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: current cursor position in a winPos class object                  *
//******************************************************************************
//* Programmer's Note: because getyx() is a macro, there is no return value.   *
//******************************************************************************

winPos NCurses::GetCursor ( void )
{
winPos wp ;

   getyx ( this->stdscrAddress, wp.ypos, wp.xpos ) ;

   return wp ;
   
}  //* End GetCursor() *

//***********************
//*      SetCursor      *
//***********************
//******************************************************************************
//* Set the main screen's cursor position.                                     *
//*                                                                            *
//*                                                                            *
//* Input  : ypos : Y offset from upper-left corner (zero-based)               *
//*          xpos : X offset from upper-left corner (zero-based)               *
//*            OR                                                              *
//*          yxpos: Y/X offset from upper-left corner (zero-based)             *
//*                                                                            *
//* Returns: OK if successful, ERR if parameter(s) out of range                *
//******************************************************************************

short NCurses::SetCursor ( short ypos, short xpos )
{
short result = OK ;

   if ( ypos >= ZERO && ypos < this->sRows && xpos >= ZERO && xpos < this->sCols )
   {
      wmove ( this->stdscrAddress, ypos, xpos ) ;
   }
   else
      result = ERR ;
   
   return result ;
   
}  //* End SetCursor() *

short NCurses::SetCursor ( winPos yxpos )
{

   return ( this->SetCursor ( yxpos.ypos, yxpos.xpos ) ) ;

}  //* End SetCursor() *

//*************************
//*     WriteString       *
//*************************
//******************************************************************************
//* Write a string at the specified position in the stdscr window i.e. the     *
//* window in which the NCurses engine was started.  Window is refreshed.      *
//*  NOTE: String output will be truncated at the edge of the window.          *
//*  NOTE: ASCII control characters, 0x01 through 0x1F are ignored.            *
//*                                                                            *
//* Input  : startY : Y cursor start position                                  *
//*          startX : X cursor start position                                  *
//*                OR                                                          *
//*          startYX: Y/X cursor start position                                *
//*                                                                            *
//*          -- FOR UTF-8-ENCODED CHARACTERS (INCLUDING ASCII)                 *
//*            uStr : pointer to null-terminated UTF-8-encoded string          *
//*                    (maximum number of bytes incl. NULLCHAR == gsALLOCMED ) *
//*          -- FOR wchar_t 'WIDE' CHARACTERS                                  *
//*            wStr : pointer to null-terminated 'wide' wchar_t string         *
//*                    (maximum number of chars incl. NULLCHAR == gsALLOCDFLT) *
//*                                                                            *
//*          cAttr  : color attribute                                          *
//*          rtl    : (optional, 'false' by default) for RTL written languages *
//*                   if 'true', cursor moves from right-to-left               *
//*                                                                            *
//* Returns: returns final position of cursor                                  *
//*          If specified starting Y/X position is valid, then cursor position *
//*           returned will be the the column following the last character of  *
//*           displayed data (or the last column of the target line).          *
//*          If specified starting position is invalid, then cursor position   *
//*           is set to window origin before writing to window.                *
//******************************************************************************
//* Programmer's Note: 'origin' is 0/0 for LTR and 0/cols-1 forRTL.            *
//******************************************************************************

winPos NCurses::WriteString ( short startY, short startX, const char* uStr, 
                              attr_t cAttr, bool rtl )
{
   //* Convert source data to gString format (wide characters + statistics) *
   gString ogs( uStr ) ;

   //* Output the character * 
   return ( this->WriteString ( startY, startX, ogs, cAttr, rtl ) ) ;

}  //* End WriteString() *

winPos NCurses::WriteString ( const winPos& startYX, const char* uStr, 
                              attr_t cAttr, bool rtl )
{
   //* Convert source data to gString format (wide characters + statistics) *
   gString ogs( uStr ) ;

   //* Output the character * 
   return ( this->WriteString ( startYX.ypos, startYX.xpos, ogs, cAttr, rtl ) ) ;

}  //* End WriteString() *

winPos NCurses::WriteString ( short startY, short startX, const wchar_t* wStr, 
                              attr_t cAttr, bool rtl )
{
   //* Convert source data to gString format (wide characters + statistics) *
   gString ogs( wStr ) ;

   //* Output the character * 
   return ( this->WriteString ( startY, startX, ogs, cAttr, rtl ) ) ;

}  //* End WriteString() *

winPos NCurses::WriteString ( const winPos& startYX, const wchar_t* wStr, 
                              attr_t cAttr, bool rtl )
{
   //* Convert source data to gString format (wide characters + statistics) *
   gString ogs( wStr ) ;

   //* Output the character * 
   return ( this->WriteString ( startYX.ypos, startYX.xpos, ogs, cAttr, rtl ) ) ;

}  //* End WriteString() *

//*************************
//*     WriteString       *
//*************************
//******************************************************************************
//* PRIVATE METHOD, called only by the public WriteString() methods.           *
//* Write a string at the specified position in the stdscr window.             *
//* Window is refreshed.                                                       *
//*                                                                            *
//* Input  : startY = Y cursor start position                                  *
//*          startX = X cursor start position                                  *
//*          ogs    = character string to display in gString format            *
//*                   (ASCII control characters, 0x01 through 0x1F are ignored)*
//*          cAttr  = color attribute                                          *
//*          rtl    : (optional, 'false' by default) for RTL written languages *
//*                   if 'true', cursor moves from right-to-left               *
//*                                                                            *
//* Returns: final position of cursor                                          *
//*          If specified starting Y/X position is valid, then cursor position *
//*           returned will be the the column following the last character of  *
//*           displayed data (or the last column of the target line).          *
//*          If specified starting position is invalid, then cursor position   *
//*           is set to window origin before writing to window.                *
//******************************************************************************
//* Note that the formatting algorithm used in this method ASSUMES that a      *
//* printing (non-zero-width) character may be modified by either leading      *
//* zero-width characters OR by trailing zero-width characters, BUT NOT BOTH!  *
//* This assumption holds true for all our test data, but the author is neither*
//* a linguist nor an orthographer, so it's possible that this assumption will *
//* come back to bite us on the ass.                                           *
//******************************************************************************

winPos NCurses::WriteString ( short startY, short startX, gString& ogs, 
                              attr_t cAttr, bool rtl )
{
   //* Test range and set initial cursor positon *
   if ( startY < ZERO || startY >= this->sRows || 
        startX < ZERO || startX >= this->sCols )
   {
      startY = ZERO ;
      startX = rtl ? (this->sCols - 1) : ZERO ;
   }
   this->SetCursor ( startY, startX ) ;

   //* Test for possible wrap-around and truncate the output if necessary. *
   short colsAvail = rtl == false ?          // avaliable display columns
                     (this->sCols - startX) : (startX + 1),
         colsNeeded = ogs.gscols() ;         // display columns required
   if ( colsAvail < colsNeeded )
      colsNeeded = ogs.limitCols(colsAvail) ;

   //* Output the supra-ASCII string *
   if ( colsAvail >= colsNeeded && colsNeeded > ZERO )
   {
      //* Output structure with color attribute initialized.*
      cchar_t oChunk = { cAttr } ;
      const wchar_t* sPtr = ogs.gstr() ;     // pointer to source text
      int   loopCount,                       // number of source characters
            cci,                             // index into oChunk.chars[]
            xoffset = startX ;               // X offset (for RTL cursor position)
      const short* cPtr = ogs.gscols( loopCount ) ; // array of column widths
      --loopCount ;                          // do not count null terminator
      bool  leadzero ;                       // 'true' if leading zero-length chars

      for ( int si = ZERO ; si < loopCount ; )
      {
         if ( sPtr[si] >= SPACE )            // ignore control codes
         {
            leadzero = false ;
            for ( cci = ZERO ; cci < CCHARW_MAX && sPtr[si] != nckNULLCHAR && 
                               cPtr[si] == ZERO ; )
            {  //* Scan for and store leading, zero-width characters *
               oChunk.chars[cci++] = sPtr[si++] ;
               leadzero = true ;
            }
            if ( (cci < CCHARW_MAX) && (sPtr[si] != nckNULLCHAR) )
            {  //* Printing character (non-zero width) *
               oChunk.chars[cci++] = sPtr[si++] ;
            }
            if ( !leadzero )        // trailing, zero-width characters
            {
               while ( cci < CCHARW_MAX && 
                       sPtr[si] != nckNULLCHAR && cPtr[si] == ZERO )
                  oChunk.chars[cci++] = sPtr[si++] ;
            }
            while ( cci < CCHARW_MAX ) // fill remainder of array with nulls
               oChunk.chars[cci++] = nckNULLCHAR ;

            wadd_wch ( this->stdscrAddress, &oChunk ) ;  // write the data

            //* If direction of text flow is left-to-right (LTR), *
            //* then cursor is automagically positioned correctly.*
            //* If direction of text flow is right-to-left (RTL), *
            //* then set cursor for the next character.           *
            if ( rtl != false )     // flow is right-to-left
            {
               short i = si ;
               while ( cPtr[i] == ZERO && sPtr[i] != nckNULLCHAR )
                  ++i ;
               if ( sPtr[i] == nckNULLCHAR )    // end of string
                  --xoffset ;
               else                             // width of next char to process
                  xoffset -= cPtr[si] ;
               if ( xoffset < ZERO ) xoffset = ZERO ; // reached edge of window
               this->SetCursor ( startY, xoffset ) ;
            }
         }
         else  // step over the ASCII control code
            ++si ;
      }
   }

   this->RefreshScreen () ;         // refresh screen

   //* Return current cursor position *
   winPos wpos = this->GetCursor () ;
   bool badX = bool((rtl == false && wpos.xpos < startX) ||
                    (rtl != false && wpos.xpos > startX) ||
                    (wpos.xpos < ZERO) || (wpos.xpos >= this->sCols)) ;
   if ( wpos.ypos != startY || badX )
   {
      wpos.ypos = startY ;
      if ( badX )
         wpos.xpos = rtl == false ? this->sCols - 1 : ZERO ;
      this->SetCursor ( wpos.ypos, wpos.xpos ) ;
   }
   return wpos ;

}  //* End WriteString() *

//**************************
//*      WriteChar         *
//**************************
//******************************************************************************
//* Write a single character at the specified position.                        *
//* Optionally, refresh the display before return to caller.                   *
//*  NOTE: ASCII control characters, 0x01 through 0x1F are ignored.            *
//*                                                                            *
//* Input  : startY : Y cursor position                                        *
//*          startX : X cursor position                                        *
//*                OR                                                          *
//*          startYX: Y/X cursor start position                                *
//*                                                                            *
//*          -- FOR UTF-8-ENCODED CHARACTERS (INCLUDING ASCII)                 *
//*            uChar: pointer to character to display                          *
//*                    (single-byte or multi-byte character)                   *
//*          -- FOR wchar_t 'WIDE' CHARACTERS                                  *
//*            wChar: character to display in wchar_t format                   *
//*                                                                            *
//*          cAttr  : color attribute                                          *
//*          refresh: (optional, default==false) refresh display window        *
//*          rtl    : (optional, 'false' by default) for RTL written languages *
//*                   if 'true', cursor moves from right-to-left               *
//*                                                                            *
//* Returns: returns final position of cursor                                  *
//*          If specified starting Y/X position is valid, AND the specified    *
//*           character is a printing character, then cursor position returned *
//*           will be the the column following the character (or the last      *
//*           column on the target line).                                      *
//*          If invalid character specified, then position returned will be    *
//*           start position.                                                  *
//*          If specified starting position is invalid, then cursor position   *
//*           is set to window origin before writing to window.                *
//******************************************************************************
//* Programmer's Note: 'origin' is 0/0 for LTR and 0/cols-1 forRTL.            *
//******************************************************************************

winPos NCurses::WriteChar ( short startY, short startX, const char* uChar, 
                            attr_t cAttr, bool refresh, bool rtl )
{
   //* Convert source data to gString format (wide characters + statistics) *
   gString ogs(uChar, 1) ;

   //* Output the character * 
   return ( this->WriteChar ( startY, startX, ogs, cAttr, refresh, rtl ) ) ;

}  //* End WriteChar() *

winPos NCurses::WriteChar ( const winPos& startYX, const char* uChar, 
                            attr_t cAttr, bool refresh, bool rtl )
{
   //* Convert source data to gString format (wide characters + statistics) *
   gString ogs(uChar, 1) ;

   //* Output the character * 
   return ( this->WriteChar ( startYX.ypos, startYX.xpos, ogs, 
                              cAttr, refresh, rtl ) ) ;

}  //* End WriteChar() *

winPos NCurses::WriteChar ( short startY, short startX, wchar_t wChar, 
                            attr_t cAttr, bool refresh, bool rtl )
{
   //* Convert source data to gString format (wide characters + statistics) *
   gString ogs(&wChar, 1) ;

   //* Output the character * 
   return ( this->WriteChar ( startY, startX, ogs, cAttr, refresh, rtl ) ) ;

}  //* End WriteChar() *

winPos NCurses::WriteChar ( const winPos& startYX, wchar_t wChar, 
                            attr_t cAttr, bool refresh, bool rtl )
{
   //* Convert source data to gString format (wide characters + statistics) *
   gString ogs(&wChar, 1) ;

   //* Output the character * 
   return ( this->WriteChar ( startYX.ypos, startYX.xpos, ogs, 
                              cAttr, refresh, rtl ) ) ;

}  //* End WriteChar() *

//**************************
//*      WriteChar         *
//**************************
//******************************************************************************
//* PRIVATE METHOD, called only by the public WriteChar() methods.             *
//* Write a single character at the specified position.                        *
//*                                                                            *
//* Input  : startY = Y cursor position                                        *
//*          startX = X cursor position                                        *
//*          ogs    = character to display in gString format                   *
//*                   (ASCII control characters, 0x01 through 0x1F are ignored)*
//*          cAttr  = color attribute                                          *
//*          refresh= (optional, default==false) refresh display window        *
//*          rtl    : (optional, 'false' by default) for RTL written languages *
//*                   if 'true', cursor moves from right-to-left               *
//*                                                                            *
//* Returns: returns final position of cursor                                  *
//******************************************************************************
//* Note on cursor position returned: Y value will be the same as startY,      *
//* and X value will never be greater than last column in the window, nor less *
//* than ZERO.                                                                 *
//*                                                                            *
//* Note that without counting columns, a two-column character positioned at   *
//* sCols-2 will display properly, while a two-column character positioned at  *
//* sCols-1 will wrap to start of next row to print. While this is reasonable, *
//* it is ugly, and we wish to minimize ugliness in the world.                 *
//******************************************************************************

winPos NCurses::WriteChar ( short startY, short startX, gString& ogs, 
                            attr_t cAttr, bool refresh, bool rtl )
{
   //* Test range and set initial cursor positon *
   if ( startY < ZERO || startY >= this->sRows || 
        startX < ZERO || startX >= this->sCols )
   {
      startY = ZERO ;
      startX = rtl ? (this->sCols - 1) : ZERO ;
   }
   this->SetCursor ( startY, startX ) ;

   //* If not a NULLCHAR and not an ASCII control character *
   if ( ogs.gschars() > 1 && *ogs.gstr() >= SPACE )
   {
      //* Test for possible wrap-around and truncate the output if necessary. *
      short colsAvail = rtl == false ?          // avaliable display columns
                        (this->sCols - startX) : (startX + 1),
            colsNeeded = ogs.gscols() ;         // display columns required
      if ( colsAvail < colsNeeded )
         colsNeeded = ogs.limitCols(colsAvail) ;
   
      //* Output the supra-ASCII character *
      if ( colsAvail >= colsNeeded && colsNeeded > ZERO )
      {
         cchar_t oChunk = { cAttr, { wchar_t(*ogs.gstr()), wchar_t(NULLCHAR), 
                            wchar_t(NULLCHAR), wchar_t(NULLCHAR), wchar_t(NULLCHAR)} } ;
         wadd_wch ( this->stdscrAddress, &oChunk ) ;

         //* If direction of text flow is left-to-right (LTR), *
         //* then cursor is automagically positioned correctly.*
         //* If direction of text flow is right-to-left (RTL), *
         //* then set cursor for the next character.           *
         // NOTE: If next character written is a multi-column character, then 
         //       ugliness will occur, but we can't anticipate the future.
         if ( rtl != false )     // flow is right-to-left
            this->SetCursor ( startY, (startX - 1) ) ;
      }
   }

   //* Refresh display if requested *
   if ( refresh != false )
      this->RefreshScreen () ;

   //* Return current cursor position *
   winPos wpos = this->GetCursor () ;
   bool badX = bool((rtl == false && wpos.xpos < startX) ||
                    (rtl != false && wpos.xpos > startX) ||
                    (wpos.xpos < ZERO) || (wpos.xpos >= this->sCols)) ;
   if ( wpos.ypos != startY || badX )
   {
      wpos.ypos = startY ;
      if ( badX )
         wpos.xpos = rtl == false ? this->sCols - 1 : ZERO ;
      this->SetCursor ( wpos.ypos, wpos.xpos ) ;
   }
   return wpos ;

}  //* End WriteChar() *

//*************************
//*      WriteStream      *
//*************************
//******************************************************************************
//* Write data to the main terminal window ('stdscr'). This method is to be    *
//* called ONLY to output data to the terminal window when output-stream       *
//* scrolling has been enabled. (see the OutputStreamScrolling() method)       *
//* If output-stream scrolling has not been enabled, then this method calls    *
//* the standard NCurses-class 'WriteString' method to produce formatted,      *
//* non-scrolling data.                                                        *
//*                                                                            *
//* This method does not filter the output in any way. While the standard      *
//* output methods ignore ASCII control characters to ensure consistent        *
//* formatting, WriteStream() outputs all characters, regardless of type.      *
//* Note however, that the C language output primitives will either ignore     *
//* most ASCII control characters, or will choke and output garbage.           *
//* Only newline '\n' and tab '\t' will be reliably recognized on all systems. *
//*                                                                            *
//* IMPORTANT NOTE:                                                            *
//* This method gives the underlying 'ncurses' C-language library direct       *
//* control of the output, which is generally not a good idea. That is why     *
//* this method is classified as a debugging method. Use with caution.         *
//*                                                                            *
//* IMPORTANT NOTE:                                                            *
//* This method is not to be used when the higher-level NcWindow and NcDialog  *
//* classes have been instantiated in the terminal window. Doing so will cause *
//* all kinds of ugliness. You have been warned.                               *
//*                                                                            *
//*                                                                            *
//* Input  : wStr : pointer to null-terminated wchar_t 'wide'-character string *
//*          cAttr: color attribute                                            *
//*                                                                            *
//* Returns: returns final position of cursor                                  *
//******************************************************************************

winPos NCurses::WriteStream ( const wchar_t* wStr, attr_t cAttr )
{
   if ( this->outputScroll != false )
   {
      cchar_t oChunk = { cAttr, { wchar_t(NULLCHAR), wchar_t(NULLCHAR), 
                   wchar_t(NULLCHAR), wchar_t(NULLCHAR), wchar_t(NULLCHAR)} } ;
      for ( short i = ZERO ; wStr[i] != NULLCHAR ; i++ )
      {
         oChunk.chars[ZERO] = wStr[i] ;
         wadd_wch ( this->stdscrAddress, &oChunk ) ;
      }
      this->RefreshScreen () ;      // refresh screen
   }

   //* Caller is a moron. Call the standard output method. *
   else
   {
      winPos wp = this->GetCursor () ;
      gString gs( wStr ) ;
      this->WriteString ( wp.ypos, wp.xpos, gs, cAttr ) ;
   }

   //* Return current cursor position *
   return ( this->GetCursor () ) ;

}  //* End WriteStream() *

//*************************
//*       UserAlert       *
//*************************
//******************************************************************************
//* Sound an audible alert. The actual sound produced is system dependent.     *
//* Note that if the system does not support audible alerts, then then the     *
//* screen will visibly flash instead.                                         *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: OK if audible alert supported, else ERR                           *
//******************************************************************************

short NCurses::UserAlert ( void )
{

   return ( beep () ) ;

}  //* End UserAlert() *

//*************************
//*  DisplayColorOptions  *
//*************************
//******************************************************************************
//* Display samples of the standard color schemes using the current            *
//* background color.                                                          *
//*                                                                            *
//* Input  : ypos  : (optional, default==3) Y screen offset                    *
//*          xpos  : (optional, default==1) X screen offset                    *
//*          pause : (optional, default==true)                                 *
//*                  if 'true' wait for key input before returning             *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void NCurses::DisplayColorOptions ( short ypos, short xpos, bool pause )
{
int      cursor_state = curs_set ( 0 ) ;     // make cursor invisible
winPos   wp(ypos, xpos) ;

   //* Draw a title *
   WriteString (  wp.ypos,  wp.xpos, 
              "          Standard Color Scheme with Current Background Color ", nc.bw ) ;
   ++wp.ypos ;
   WriteString (  wp.ypos,  wp.xpos, 
    "  (Note: some terminals do not support all color-attribute modifiers.)", nc.bw ) ;

   //* Draw a pseudo-border *
   ++wp.ypos ;
   int wLines = 19, wCols = 72 ;
   this->DebugBorder ( wLines, wCols, wp.ypos, wp.xpos, nc.bw ) ;

   //* Display the data within the border *
   ++wp.xpos ;
   WriteString ( (wp.ypos +  1), wp.xpos, "  0 Defaults    ", (ncbcBK << 8) ) ;
   WriteString ( (wp.ypos +  2), wp.xpos, "  1 Defaults    ", (ncbcRE << 8) ) ;
   WriteString ( (wp.ypos +  3), wp.xpos, "  2 Defaults    ", (ncbcGR << 8) ) ;
   WriteString ( (wp.ypos +  4), wp.xpos, "  3 Defaults    ", (ncbcBR << 8) ) ;
   WriteString ( (wp.ypos +  5), wp.xpos, "  4 Defaults    ", (ncbcBL << 8) ) ;
   WriteString ( (wp.ypos +  6), wp.xpos, "  5 Defaults    ", (ncbcMA << 8) ) ;
   WriteString ( (wp.ypos +  7), wp.xpos, "  6 Defaults    ", (ncbcCY << 8) ) ;
   WriteString ( (wp.ypos +  8), wp.xpos, "  7 Defaults    ", (ncbcGY << 8) ) ;

   WriteString ( (wp.ypos + 10), wp.xpos, "  0 Standout    ", ((ncbcBK << 8) | ncsATTR) ) ;
   WriteString ( (wp.ypos + 11), wp.xpos, "  1 Standout    ", ((ncbcRE << 8) | ncsATTR) ) ;
   WriteString ( (wp.ypos + 12), wp.xpos, "  2 Standout    ", ((ncbcGR << 8) | ncsATTR) ) ;
   WriteString ( (wp.ypos + 13), wp.xpos, "  3 Standout    ", ((ncbcBR << 8) | ncsATTR) ) ;
   WriteString ( (wp.ypos + 14), wp.xpos, "  4 Standout    ", ((ncbcBL << 8) | ncsATTR) ) ;
   WriteString ( (wp.ypos + 15), wp.xpos, "  5 Standout    ", ((ncbcMA << 8) | ncsATTR) ) ;
   WriteString ( (wp.ypos + 16), wp.xpos, "  6 Standout    ", ((ncbcCY << 8) | ncsATTR) ) ;
   WriteString ( (wp.ypos + 17), wp.xpos, "  7 Standout    ", ((ncbcGY << 8) | ncsATTR) ) ;

   wp.xpos += 18 ;
   WriteString ( (wp.ypos +  1), wp.xpos, "  0 Dimmed      ", ((ncbcBK << 8) | ncdATTR) ) ;
   WriteString ( (wp.ypos +  2), wp.xpos, "  1 Dimmed      ", ((ncbcRE << 8) | ncdATTR) ) ;
   WriteString ( (wp.ypos +  3), wp.xpos, "  2 Dimmed      ", ((ncbcGR << 8) | ncdATTR) ) ;
   WriteString ( (wp.ypos +  4), wp.xpos, "  3 Dimmed      ", ((ncbcBR << 8) | ncdATTR) ) ;
   WriteString ( (wp.ypos +  5), wp.xpos, "  4 Dimmed      ", ((ncbcBL << 8) | ncdATTR) ) ;
   WriteString ( (wp.ypos +  6), wp.xpos, "  5 Dimmed      ", ((ncbcMA << 8) | ncdATTR) ) ;
   WriteString ( (wp.ypos +  7), wp.xpos, "  6 Dimmed      ", ((ncbcCY << 8) | ncdATTR) ) ;
   WriteString ( (wp.ypos +  8), wp.xpos, "  7 Dimmed      ", ((ncbcGY << 8) | ncdATTR) ) ;

   WriteString ( (wp.ypos + 10), wp.xpos, "  0 Underlined  ", ((ncbcBK << 8) | ncuATTR) ) ;
   WriteString ( (wp.ypos + 11), wp.xpos, "  1 Underlined  ", ((ncbcRE << 8) | ncuATTR) ) ;
   WriteString ( (wp.ypos + 12), wp.xpos, "  2 Underlined  ", ((ncbcGR << 8) | ncuATTR) ) ;
   WriteString ( (wp.ypos + 13), wp.xpos, "  3 Underlined  ", ((ncbcBR << 8) | ncuATTR) ) ;
   WriteString ( (wp.ypos + 14), wp.xpos, "  4 Underlined  ", ((ncbcBL << 8) | ncuATTR) ) ;
   WriteString ( (wp.ypos + 15), wp.xpos, "  5 Underlined  ", ((ncbcMA << 8) | ncuATTR) ) ;
   WriteString ( (wp.ypos + 16), wp.xpos, "  6 Underlined  ", ((ncbcCY << 8) | ncuATTR) ) ;
   WriteString ( (wp.ypos + 17), wp.xpos, "  7 Underlined  ", ((ncbcGY << 8) | ncuATTR) ) ;

   wp.xpos += 18 ;
   WriteString ( (wp.ypos +  1), wp.xpos, "  0 Bold Fgnd   ", ((ncbcBK << 8) | ncbATTR) ) ;
   WriteString ( (wp.ypos +  2), wp.xpos, "  1 Bold Fgnd   ", ((ncbcRE << 8) | ncbATTR) ) ;
   WriteString ( (wp.ypos +  3), wp.xpos, "  2 Bold Fgnd   ", ((ncbcGR << 8) | ncbATTR) ) ;
   WriteString ( (wp.ypos +  4), wp.xpos, "  3 Bold Fgnd   ", ((ncbcBR << 8) | ncbATTR) ) ;
   WriteString ( (wp.ypos +  5), wp.xpos, "  4 Bold Fgnd   ", ((ncbcBL << 8) | ncbATTR) ) ;
   WriteString ( (wp.ypos +  6), wp.xpos, "  5 Bold Fgnd   ", ((ncbcMA << 8) | ncbATTR) ) ;
   WriteString ( (wp.ypos +  7), wp.xpos, "  6 Bold Fgnd   ", ((ncbcCY << 8) | ncbATTR) ) ;
   WriteString ( (wp.ypos +  8), wp.xpos, "  7 Bold Fgnd   ", ((ncbcGY << 8) | ncbATTR) ) ;

   //* Alternate Fonts - characters 0x28 - 0z6E
   WriteString ( (wp.ypos + 10), wp.xpos, "\x2B\x2C\x2D\x2E\x30\x60\x61\x66"
                                          "\x67\x68\x69\x6A\x6B\x6C\x6D\x6E", 
                                          ((ncbcBK << 8) | ncaATTR) ) ;
   WriteString ( (wp.ypos + 11), wp.xpos, "\x2B\x2C\x2D\x2E\x30\x60\x61\x66"
                                          "\x67\x68\x69\x6A\x6B\x6C\x6D\x6E", 
                                          ((ncbcRE << 8) | ncaATTR) ) ;
   WriteString ( (wp.ypos + 12), wp.xpos, "\x2B\x2C\x2D\x2E\x30\x60\x61\x66"
                                          "\x67\x68\x69\x6A\x6B\x6C\x6D\x6E", 
                                          ((ncbcGR << 8) | ncaATTR) ) ; 
   WriteString ( (wp.ypos + 13), wp.xpos, "\x2B\x2C\x2D\x2E\x30\x60\x61\x66"
                                          "\x67\x68\x69\x6A\x6B\x6C\x6D\x6E", 
                                          ((ncbcBR << 8) | ncaATTR) ) ; 
   WriteString ( (wp.ypos + 14), wp.xpos, "\x2B\x2C\x2D\x2E\x30\x60\x61\x66"
                                          "\x67\x68\x69\x6A\x6B\x6C\x6D\x6E", 
                                          ((ncbcBL << 8) | ncaATTR) ) ; 
   WriteString ( (wp.ypos + 15), wp.xpos, "\x2B\x2C\x2D\x2E\x30\x60\x61\x66"
                                          "\x67\x68\x69\x6A\x6B\x6C\x6D\x6E", 
                                          ((ncbcMA << 8) | ncaATTR) ) ; 
   WriteString ( (wp.ypos + 16), wp.xpos, "\x2B\x2C\x2D\x2E\x30\x60\x61\x66"
                                          "\x67\x68\x69\x6A\x6B\x6C\x6D\x6E", 
                                          ((ncbcCY << 8) | ncaATTR) ) ; 
   WriteString ( (wp.ypos + 17), wp.xpos, "\x2B\x2C\x2D\x2E\x30\x60\x61\x66"
                                          "\x67\x68\x69\x6A\x6B\x6C\x6D\x6E", 
                                          ((ncbcGY << 8) | ncaATTR) ) ;

   wp.xpos += 18 ;
   WriteString ( (wp.ypos +  1), wp.xpos, "  0 Reverse Vid ", ((ncbcBK << 8) | ncrATTR) ) ;
   WriteString ( (wp.ypos +  2), wp.xpos, "  1 Reverse Vid ", ((ncbcRE << 8) | ncrATTR) ) ;
   WriteString ( (wp.ypos +  3), wp.xpos, "  2 Reverse Vid ", ((ncbcGR << 8) | ncrATTR) ) ;
   WriteString ( (wp.ypos +  4), wp.xpos, "  3 Reverse Vid ", ((ncbcBR << 8) | ncrATTR) ) ;
   WriteString ( (wp.ypos +  5), wp.xpos, "  4 Reverse Vid ", ((ncbcBL << 8) | ncrATTR) ) ;
   WriteString ( (wp.ypos +  6), wp.xpos, "  5 Reverse Vid ", ((ncbcMA << 8) | ncrATTR) ) ;
   WriteString ( (wp.ypos +  7), wp.xpos, "  6 Reverse Vid ", ((ncbcCY << 8) | ncrATTR) ) ;
   WriteString ( (wp.ypos +  8), wp.xpos, "  7 Reverse Vid ", ((ncbcGY << 8) | ncrATTR) ) ;

   WriteString ( (wp.ypos + 10), wp.xpos, "\x6F\x70\x71\x72\x73\x74\x75\x76"
                                          "\x77\x78\x79\x7A\x7B\x7C\x7D\x7E", 
                                          ((ncbcBK << 8) | ncaATTR) ) ;
   WriteString ( (wp.ypos + 11), wp.xpos, "\x6F\x70\x71\x72\x73\x74\x75\x76"
                                          "\x77\x78\x79\x7A\x7B\x7C\x7D\x7E", 
                                          ((ncbcRE << 8) | ncaATTR) ) ;
   WriteString ( (wp.ypos + 12), wp.xpos, "\x6F\x70\x71\x72\x73\x74\x75\x76"
                                          "\x77\x78\x79\x7A\x7B\x7C\x7D\x7E", 
                                          ((ncbcGR << 8) | ncaATTR) ) ;
   WriteString ( (wp.ypos + 13), wp.xpos, "\x6F\x70\x71\x72\x73\x74\x75\x76"
                                          "\x77\x78\x79\x7A\x7B\x7C\x7D\x7E", 
                                          ((ncbcBR << 8) | ncaATTR) ) ;
   WriteString ( (wp.ypos + 14), wp.xpos, "\x6F\x70\x71\x72\x73\x74\x75\x76"
                                          "\x77\x78\x79\x7A\x7B\x7C\x7D\x7E", 
                                          ((ncbcBL << 8) | ncaATTR) ) ;
   WriteString ( (wp.ypos + 15), wp.xpos, "\x6F\x70\x71\x72\x73\x74\x75\x76"
                                          "\x77\x78\x79\x7A\x7B\x7C\x7D\x7E", 
                                          ((ncbcMA << 8) | ncaATTR) ) ;
   WriteString ( (wp.ypos + 16), wp.xpos, "\x6F\x70\x71\x72\x73\x74\x75\x76"
                                          "\x77\x78\x79\x7A\x7B\x7C\x7D\x7E", 
                                          ((ncbcCY << 8) | ncaATTR) ) ;
   WriteString ( (wp.ypos + 17), wp.xpos, "\x6F\x70\x71\x72\x73\x74\x75\x76"
                                          "\x77\x78\x79\x7A\x7B\x7C\x7D\x7E", 
                                          ((ncbcGY << 8) | ncaATTR) ) ;
   touchwin ( this->stdscrAddress ) ;
   nc.RefreshScreen () ;
   if ( pause )
      nckPause() ;

   curs_set ( cursor_state ) ;         // restore cursor attribute

}  //* End DisplayColorOptions() *

//**************************
//*  DisplayAlternateFont  *
//**************************
//******************************************************************************
//* Display the characters of the alternate font.                              *
//* Alternate font consists mainly of line-drawing characters.                 *
//*                                                                            *
//* Important Note:  Always keep in mind that the characters that make up      *
//*                  the alternate font are 1/2 the width of the standard      *
//*                  characters.                                               *
//*                                                                            *
//* Input  : ypos  : (optional, default==3) Y screen offset                    *
//*          xpos  : (optional, default==1) X screen offset                    *
//*          pause : (optional, default==true)                                 *
//*                  if 'true' wait for key input before returning             *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note: The alternate font (G1) under Fedora 12 is defined      *
//* somewhat differently than under Red Hat 9.0. 0x7C used to be a blank       *
//* character, but now, it is a not-equal-to character. This may have          *
//* something to do with the terminal emulation program, VT100 vs xterm vs     *
//* whatever. We haven't had time to experiment thoroughly with the            *
//* different terminal emulation choices available.                            *
//*                                                                            *
//* We are not sure where the SPACE character lives in the alternate font      *
//* because there are several characters between 0x00 and 0x2B that do not     *
//* display.                                                                   *
//*                                                                            *
//* For now, we have changed our table, replacing the 0x7C separator code      *
//* with a 0x7E code. Logically, the 0x20 code would be best since it is the   *
//* ASCII SPACE, but this needs more investigation.                            *
//* Presumably, understanding terminfo would yield the answers.                *
//******************************************************************************

void NCurses::DisplayAlternateFont ( short ypos, short xpos, bool pause )
{
   // ALTERNATE FONT CHARS: \x2B\x2C\x2D\x2E\x30\x60\x61\x66 through \x7E
   const char char2B6E[] = 
   {
      "\x2B\x7E\x7E\x7E\x2C\x7E\x7E\x7E\x2D\x7E\x7E\x7E\x2E\x7E\x7E\x7E"
      "\x30\x7E\x7E\x7E\x60\x7E\x7E\x7E\x61\x7E\x7E\x7E\x66\x7E\x7E\x7E"
      "\x67\x7E\x7E\x7E\x68\x7E\x7E\x7E\x69\x7E\x7E\x7E\x6A\x7E\x7E\x7E"
      "\x6B\x7E\x7E\x7E\x6C\x7E\x7E\x7E\x6D\x7E\x7E\x7E\x6E"
   } ;
   const char char6F7E[] = 
   {
      "\x6F\x7E\x7E\x7E\x70\x7E\x7E\x7E\x71\x7E\x7E\x7E\x72\x7E\x7E\x7E"
      "\x73\x7E\x7E\x7E\x74\x7E\x7E\x7E\x75\x7E\x7E\x7E\x76\x7E\x7E\x7E"
      "\x77\x7E\x7E\x7E\x78\x7E\x7E\x7E\x79\x7E\x7E\x7E\x7A\x7E\x7E\x7E"
      "\x7B\x7E\x7E\x7E\x7C\x7E\x7E\x7E\x7D\x7E\x7E\x7E\x7E"
   } ;
   int      cursor_state = curs_set ( 0 ) ;     // make cursor invisible

   //* Draw a title *
   this->WriteString (  ypos++,  (xpos + 9), 
         " Display the Characters of the Alternate Character Set ", nc.bw ) ;

   //* Draw a pseudo-border *
   int wLines = 21, wCols = 73, wulY = ypos, wulX = xpos ;
   this->DebugBorder ( wLines, wCols, wulY, wulX, nc.bw ) ;
   ++ypos ;
   xpos += 6 ;

   WriteString ( ypos++, (xpos - 4), "Alternate-Font Characters:", nc.bw ) ; 
   WriteString ( ypos++, (xpos - 5), "    2Bh 2Ch 2Dh 2Eh 30h 60h 61h 66h 67h "
                                     "68h 69h 6Ah 6Bh 6Ch 6Dh 6Eh    ", nc.bw ) ;

#if 0    // SET TO 1 FOR TESTING ALTERNATE FONT 0x01 - 0x2A and 0x7F - 0xFF
// Note: there is likely nothing printable associated with these codes.
   WriteString ( ypos++, xpos, 
//      "\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0A\x0B\x0C\x0D\x0E\x0F\x10"
//      "\x11\x12\x13\x14\x15\x16\x17\x18\x19\x1A\x1B\x1C\x1D\x1E\x1F"
      "\x20"
      "\x21\x22\x23\x24\x25\x26\x27\x28\x29\x2A\x7F\x80\x81\x82\x83\x84"
      "\x85\x86\x87\x88\x89\x8A\x8B\x8C\x8D\x8E\x8F\x90\x91", (nc.bw | ncaATTR) ) ;
   WriteString ( ypos++, xpos, 
      "\x92\x93\x94\x95\x96\x97\x98\x99\x9A\x9B\x9C\x9D\x9E\x9F\xA0\xA1"
      "\xA2\xA3\xA4\xA5\xA6\xA7\xA8\xA9\xAA\xAB\xAC\xAD\xAE\xAF\xB0\xB1"
      "\xB2\xB3\xB4\xB5\xB6\xB7\xB8\xB9\xBA\xBB\xBC\xBD\xBE\xBF\xC0\xC1"
      "\xC2\xC3\xC4\xC5\xC6\xC7\xC8\xC9\xCA\xCB\xCC\xCD\xCE", (nc.bw | ncaATTR) ) ;
   WriteString ( ypos++, xpos, 
      "\xCF\xD0\xD1\xD2\xD3\xD4\xD5\xD6\xD7\xD8\xD9\xDA\xDB\xDC\xDD\xDE"
      "\xDF\xE0\xE1\xE2\xE3\xE4\xE5\xE6\xE7\xE8\xE9\xEA\xEB\xEC\xED\xEE"
      "\xEF\xF0\xF1\xF2\xF3\xF4\xF5\xF6\xF7\xF8\xF9\xFA\xFB\xFC\xFD\xFE\xFF", 
      (nc.bw | ncaATTR) ) ;
#else    // DISPLAY ALTERNATE FONT 0x2B - 0x7E IN LIVING COLOR
   WriteString ( ypos++, xpos, char2B6E, (nc.bw | ncaATTR) ) ;
   WriteString ( ypos++, xpos, char2B6E, (nc.re | ncrATTR | ncaATTR) ) ;
   WriteString ( ypos++, xpos, char2B6E, (nc.gr | ncrATTR | ncaATTR) ) ;
#endif   // 0x2B - 0x7E IN LIVING COLOR

   WriteString ( ypos++, xpos, char2B6E, (nc.br | ncrATTR | ncaATTR) ) ;
   WriteString ( ypos++, xpos, char2B6E, (nc.bl | ncrATTR | ncaATTR) ) ;
   WriteString ( ypos++, xpos, char2B6E, (nc.ma | ncrATTR | ncaATTR) ) ;
   WriteString ( ypos++, xpos, char2B6E, (nc.cy | ncrATTR | ncaATTR) ) ;
   WriteString ( ypos++, xpos, char2B6E, (nc.gy | ncrATTR | ncaATTR) ) ;

   WriteString ( ypos++, (xpos - 5), "    6Fh 70h 71h 72h 73h 74h 75h 76h 77h "
                                     "78h 79h 7Ah 7Bh 7Ch 7Dh 7Eh    ", nc.bw ) ;

   WriteString ( ypos++, xpos, char6F7E, (nc.bw | ncaATTR) ) ;
   WriteString ( ypos++, xpos, char6F7E, (nc.re | ncrATTR | ncaATTR) ) ;
   WriteString ( ypos++, xpos, char6F7E, (nc.gr | ncrATTR | ncaATTR) ) ;
   WriteString ( ypos++, xpos, char6F7E, (nc.br | ncrATTR | ncaATTR) ) ;
   WriteString ( ypos++, xpos, char6F7E, (nc.bl | ncrATTR | ncaATTR) ) ;
   WriteString ( ypos++, xpos, char6F7E, (nc.ma | ncrATTR | ncaATTR) ) ;
   WriteString ( ypos++, xpos, char6F7E, (nc.cy | ncrATTR | ncaATTR) ) ;
   WriteString ( ypos++, xpos, char6F7E, (nc.gy | ncrATTR | ncaATTR) ) ;

   this->RefreshScreen () ;
   if ( pause )
      nckPause() ;
            
   curs_set ( cursor_state ) ;         // restore cursor attribute

}  //* End DisplayAlternateFont() *

//*************************
//* DisplayDefinedColors  *
//*************************
//******************************************************************************
//* Display examples for each of the public color variables defined in         *
//* NCurses.hpp. These variables represent the foreground/background color     *
//* pairs defined by InitColorPairs() and initialized by InitColorVariables(). *
//*                                                                            *
//* Terminal and ncurses Library Support                                       *
//* ====================================                                       *
//* - The number of display colors and color pairs supported by the terminal   *
//*   determine the number of unique colors available.                         *
//*   - Nearly all color terminals and terminal emulators directly support     *
//*     eight (8) colors, and most modern distributions support sixteen (16)   *
//*     terminal colors. This is determined by the number of (virtual) RGB     *
//*     registers, one register per terminal color. The actual color produced  *
//*     by each RGB register may be selected through RGB presets or through    *
//*     custom color settings during terminal setup.                           *
//*                                                                            *
//* - The ncurses C-language library directly supports only the eight (8)      *
//*   basic colors; however, the ncurses library PASSIVELY supports up to      *
//*   sixteen (16) colors through the 8 bits of the 'attr_t' type which are    *
//*   reserved for output of color-pairs.                                      *
//*                                                                            *
//* NCurses Class Color Setup                                                  *
//* =========================                                                  *
//* - The NCurses class extends the functionality of the ncurses library by    *
//*   defining 128 color variables representing unique foreground/background   *
//*   color combinations. These variables are initialized to unique color-pair *
//*   values IF the terminal hardware/software support them. If the terminal   *
//*   supports fewer colors and/or fewer color-pairs, then the variables will  *
//*   contain some color-pair duplication.                                     *
//* - The NCurses class sets all color pairs above the first 128 (if any) to   *
//*   the default value (usually black-on-white or white-on-black).            *
//*                                                                            *
//* Display of the Color Variables                                             *
//* ==============================                                             *
//* - The terminal window size determines the formatting of the displayed data *
//*   for this method.                                                         *
//*   The following dimensions assume the default Y/X offsets (y==3, x==1).    *
//*   a) The termnal window must be at least 27 rows by 74 columns in order to *
//*      display the data. If the terminal window is smaller, formatting will  *
//*      be corrupted.                                                         *
//*   b) If the terminal window is at least 27 rows by 144 columns, then all   *
//*      color variables will be displayed simultaneously.                     *
//*   c) If the terminal window is less tha 144 columns in width, then the     *
//*      color variables will be displayed in two sequential groups, the basic *
//*      group and the 16-color group (if 16 colors supported by the terminal).*
//* - For each color variable, an example string indicating the color-variable *
//*   name will be displayed using the color contained in that variable.       *
//*   Please refer to NCurses.hpp for the color-variable definitions.          *
//*                                                                            *
//* Input  : ypos  : (optional, default==3) Y screen offset                    *
//*          xpos  : (optional, default==1) X screen offset                    *
//*          pause : (optional, default==true)                                 *
//*                  if 'true' wait for key input before returning             *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void  NCurses::DisplayDefinedColors ( short ypos, short xpos, bool pause )
{
   const char* header = 
      "NORMAL   BOLD      REVERSE   BOLD/REV  STANDOUT   DIM      UNDERLINE" ;
   int      cursor_state = curs_set ( 0 ) ;     // make cursor invisible
   winPos   wpTitle( ypos, xpos ), wp( ypos + 2, xpos ), wpSec ;

   //* Refresh Terminal-window Dimensions */
   this->ScreenDimensions ( this->sRows, this->sCols ) ;

   //* Draw a pseudo-border based on the size of the terminal *
   //* window and the amount of data to be displayed.         *
   short colBase = 72, wLines = 22, wCols = colBase ;
   if ( (this->maxRgb >= ncCOLOR_MAX) && (this->sCols >= 144) )
   {
      wCols += colBase - 1 ;
      wpSec.ypos = wp.ypos - 1 ;
      wpSec.xpos = wp.xpos + colBase - 1 ;
      while ( ++wpSec.ypos < (ypos + wLines + 1) )
         this->WriteChar ( wpSec, wcsVERT, this->bw ) ;
   }
   this->DebugBorder ( wLines, wCols, wp.ypos, wp.xpos, nc.bw ) ;
   ++wp.ypos ; ++wp.xpos ;

   //* Divide the sections *
   wpSec.ypos = wp.ypos + wLines / 2 - 1 ; wpSec.xpos = wp.xpos ;
   while ( wpSec.xpos < (xpos + wCols - 1) )
      wpSec = this->WriteChar ( wpSec, wcsHORIZ, this->bw ) ;

   //* Draw a title *
   this->WriteString (  wpTitle, "            NCurses Defined Colors with "
                                 "Attribute Modifiers ", this->bw ) ;
   ++wpTitle.ypos ;
   this->WriteString (  wpTitle, "  (Note: some terminals do not support all "
                                 "color-attribute modifiers.) ", this->bw ) ;

   //* Display the basic color attributes and their modified forms. *
   //* The ncurses library supports only the first 8 color pairs.   *
   this->WriteString ( wp.ypos++, (wp.xpos + 1), header, this->bwU ) ;

   const short baseCOUNT = 63 ;
   const char* textBase[baseCOUNT] = 
   {
      "  nc.bw  ", "  nc.bk  ", "  nc.re  ", "  nc.gr  ", "  nc.br  ", 
      "  nc.bl  ", "  nc.ma  ", "  nc.cy  ", "  nc.gy  ", 

      " nc.bwB  ", " nc.bkB  ", " nc.reB  ", " nc.grB  ", " nc.brB  ", 
      " nc.blB  ", " nc.maB  ", " nc.cyB  ", " nc.gyB  ", 

      " nc.bwR  ", " nc.bkR  ", " nc.reR  ", " nc.grR  ", " nc.brR  ", 
      " nc.blR  ", " nc.maR  ", " nc.cyR  ", " nc.gyR  ", 

      " nc.bwG  ", " nc.bkG  ", " nc.reG  ", " nc.grG  ", " nc.brG  ", 
      " nc.blG  ", " nc.maG  ", " nc.cyG  ", " nc.gyG  ", 

      " nc.bwS  ", " nc.bkS  ", " nc.reS  ", " nc.grS  ", " nc.brS  ", 
      " nc.blS  ", " nc.maS  ", " nc.cyS  ", " nc.gyS  ", 

      " nc.bwD  ", " nc.bkD  ", " nc.reD  ", " nc.grD  ", " nc.brD  ", 
      " nc.blD  ", " nc.maD  ", " nc.cyD  ", " nc.gyD  ", 

      " nc.bwU  ", " nc.bkU  ", " nc.reU  ", " nc.grU  ", " nc.brU  ", 
      " nc.blU  ", " nc.maU  ", " nc.cyU  ", " nc.gyU  ", 
   } ;
   const attr_t attrBase[baseCOUNT] = 
   {
   this->bw,  this->bk,  this->re,  this->gr,  this->br,  this->bl,  this->ma,  this->cy,  this->gy,
   this->bwB, this->bkB, this->reB, this->grB, this->brB, this->blB, this->maB, this->cyB, this->gyB,
   this->bwR, this->bkR, this->reR, this->grR, this->brR, this->blR, this->maR, this->cyR, this->gyR,
   this->bwG, this->bkG, this->reG, this->grG, this->brG, this->blG, this->maG, this->cyG, this->gyG,
   this->bwS, this->bkS, this->reS, this->grS, this->brS, this->blS, this->maS, this->cyS, this->gyS,
   this->bwD, this->bkD, this->reD, this->grD, this->brD, this->blD, this->maD, this->cyD, this->gyD,
   this->bwU, this->bkU, this->reU, this->grU, this->brU, this->blU, this->maU, this->cyU, this->gyU,
   } ;

   wpSec.xpos = wp.xpos - 10 ;
   for ( short i = ZERO ; i < baseCOUNT ; i++ )
   {
      if ( i % 9 == ZERO )
      { wpSec.ypos = wp.ypos  ; wpSec.xpos += 10 ; }

      this->WriteString ( wpSec, textBase[i], attrBase[i] ) ;
      ++wpSec.ypos ;
   }

   wp.ypos = ypos + 14 ;
   wp.xpos = xpos + 1 ;

   //* Display the 'extended' color attributes for color pairs 8 through 63 *
   //* If the terminal does not support the extended color pairs, these are *
   //* simply duplicates of the supported colors.                           *
   this->WriteString ( wp.ypos, (wp.xpos + 1), " Additional foreground/background "
                       "color combinations (if supported) ", this->bwU ) ;

   const short VAR64 = 56 ;
   const char* text64[VAR64] = 
   {
   " nc.rebk ", " nc.grre ", " nc.brgr ", " nc.blbr ", " nc.mabl ", " nc.cyma ", " nc.gycy ", 
   " nc.grbk ", " nc.brre ", " nc.blgr ", " nc.mabr ", " nc.cybl ", " nc.gyma ", " nc.bkgy ", 
   " nc.brbk ", " nc.blre ", " nc.magr ", " nc.cybr ", " nc.gybl ", " nc.bkcy ", " nc.regy ", 
   " nc.blbk ", " nc.mare ", " nc.cygr ", " nc.gybr ", " nc.bkma ", " nc.recy ", " nc.grgy ", 
   " nc.mabk ", " nc.cyre ", " nc.gygr ", " nc.bkbl ", " nc.rema ", " nc.grcy ", " nc.brgy ", 
   " nc.cybk ", " nc.gyre ", " nc.bkbr ", " nc.rebl ", " nc.grma ", " nc.brcy ", " nc.blgy ", 
   " nc.gybk ", " nc.bkgr ", " nc.rebr ", " nc.grbl ", " nc.brma ", " nc.blcy ", " nc.magy ", 
   " nc.bkre ", " nc.regr ", " nc.grbr ", " nc.brbl ", " nc.blma ", " nc.macy ", " nc.cygy ", 
   } ;
   const attr_t attr64[VAR64] = 
   {
   this->rebk, this->grre, this->brgr, this->blbr, this->mabl, this->cyma, this->gycy, 
   this->grbk, this->brre, this->blgr, this->mabr, this->cybl, this->gyma, this->bkgy, 
   this->brbk, this->blre, this->magr, this->cybr, this->gybl, this->bkcy, this->regy, 
   this->blbk, this->mare, this->cygr, this->gybr, this->bkma, this->recy, this->grgy, 
   this->mabk, this->cyre, this->gygr, this->bkbl, this->rema, this->grcy, this->brgy, 
   this->cybk, this->gyre, this->bkbr, this->rebl, this->grma, this->brcy, this->blgy, 
   this->gybk, this->bkgr, this->rebr, this->grbl, this->brma, this->blcy, this->magy, 
   this->bkre, this->regr, this->grbr, this->brbl, this->blma, this->macy, this->cygy, 
   } ;

   for ( short i = ZERO ; i < VAR64 ; i++ )
   {
      if ( i % 7 == ZERO )
      { ++wp.ypos ; wp.xpos = xpos + 1 ; }

      wp = this->WriteString ( wp, text64[i], attr64[i] ) ;
      ++wp.xpos ;
   }

   //* If terminal directly supports sixteen (16) colors, display the    *
   //* other half of the table (pairs 40h through 47h + 48h through 7Fh) *
   if ( this->maxRgb >= ncCOLOR_MAX )
   {
      //* Clear old data if necessary, and establish the base cursor position.*
      if ( wCols == colBase )
      {
         nckPause();    // wait for user
         nc.ClearLine ( wpTitle.ypos ) ;
         wp.ypos = wpTitle.ypos + 2 ; wp.xpos = wpTitle.xpos + 1 ;
         short yLimit = wp.ypos + 20, xLimit = wp.xpos + wCols - 2 ;
         for ( wpSec.ypos = wp.ypos ; (wpSec.ypos < yLimit) ; wpSec.ypos++ )
         {
            wpSec.xpos = wp.xpos ;
            if ( wpSec.ypos == yLimit - 10 )
               ++wpSec.ypos ;
            while ( wpSec.xpos < xLimit )
               wpSec = this->WriteChar ( wpSec, L' ', nc.bw ) ;
         }
         wpSec = wp ;
      }
      else
      {
         wpTitle.xpos += colBase ;
         wp.ypos = wpTitle.ypos + 2 ;
         wp.xpos = wpTitle.xpos ;
      }

      //* Draw a title *
      this->WriteString (  wpTitle, "             For Terminals That Directly "
                           "Support 16 Colors ", this->bw ) ;
      //* Draw the section header *
      WriteString ( wp.ypos++, (wp.xpos + 1), header, this->bwU ) ;

      const char* text16Base[baseCOUNT] = 
      {
         " nc.bbk  ", "  ---    ", " nc.bre  ", " nc.bgr  ", " nc.bbr  ", 
         " nc.bbl  ", " nc.bma  ", " nc.bcy  ", " nc.bgy  ", 
   
         " nc.bbkB ", "   ---   ", " nc.breB ", " nc.bgrB ", " nc.bbrB ", 
         " nc.bblB ", " nc.bmaB ", " nc.bcyB ", " nc.bgyB ", 
   
         " nc.bbkR ", "   ---   ", " nc.breR ", " nc.bgrR ", " nc.bbrR ", 
         " nc.bblR ", " nc.bmaR ", " nc.bcyR ", " nc.bgyR ", 
   
         " nc.bbkG ", "   ---   ", " nc.breG ", " nc.bgrG ", " nc.bbrG ", 
         " nc.bblG ", " nc.bmaG ", " nc.bcyG ", " nc.bgyG ", 
   
         " nc.bbkS ", "   ---   ", " nc.breS ", " nc.bgrS ", " nc.bbrS ", 
         " nc.bblS ", " nc.bmaS ", " nc.bcyS ", " nc.bgyS ", 
   
         " nc.bbkD ", "   ---   ", " nc.breD ", " nc.bgrD ", " nc.bbrD ", 
         " nc.bblD ", " nc.bmaD ", " nc.bcyD ", " nc.bgyD ", 
   
         " nc.bbkU ", "   ---   ", " nc.breU ", " nc.bgrU ", " nc.bbrU ", 
         " nc.bblU ", " nc.bmaU ", " nc.bcyU ", " nc.bgyU " 
      } ;
      const attr_t attr16Base[baseCOUNT] = 
      {
         this->bbk, this->bw,  this->bre, this->bgr, this->bbr, 
         this->bbl, this->bma, this->bcy, this->bgy,

         this->bbk | ncbATTR, this->bw,            this->bre | ncbATTR, 
         this->bgr | ncbATTR, this->bbr | ncbATTR, this->bbl | ncbATTR, 
         this->bma | ncbATTR, this->bcy | ncbATTR, this->bgy | ncbATTR,

         this->bbk | ncrATTR, this->bw,            this->bre | ncrATTR, 
         this->bgr | ncrATTR, this->bbr | ncrATTR, this->bbl | ncrATTR, 
         this->bma | ncrATTR, this->bcy | ncrATTR, this->bgy | ncrATTR,

         this->bbk | ncgATTR, this->bw,            this->bre | ncgATTR, 
         this->bgr | ncgATTR, this->bbr | ncgATTR, this->bbl | ncgATTR, 
         this->bma | ncgATTR, this->bcy | ncgATTR, this->bgy | ncgATTR,

         this->bbk | ncsATTR,           this->bw,  this->bre | ncsATTR, 
         this->bgr | ncsATTR, this->bbr | ncsATTR, this->bbl | ncsATTR, 
         this->bma | ncsATTR, this->bcy | ncsATTR, this->bgy | ncsATTR,

         this->bbk | ncdATTR, this->bw,            this->bre | ncdATTR, 
         this->bgr | ncdATTR, this->bbr | ncdATTR, this->bbl | ncdATTR, 
         this->bma | ncdATTR, this->bcy | ncdATTR, this->bgy | ncdATTR,

         this->bbk | ncuATTR,           this->bw,  this->bre | ncuATTR, 
         this->bgr | ncuATTR, this->bbr | ncuATTR, this->bbl | ncuATTR, 
         this->bma | ncuATTR, this->bcy | ncuATTR, this->bgy | ncuATTR
      } ;

      wpSec.xpos = wp.xpos - 10 ;
      for ( short i = ZERO ; i < baseCOUNT ; i++ )
      {
         if ( i % 9 == ZERO )
         {
            wpSec.ypos = wp.ypos  ; wpSec.xpos += 10 ;
            if ( i == 9 ) --wpSec.xpos ;
            else if ( i == 18 ) ++wpSec.xpos ;
         }
   
         WriteString ( wpSec, text16Base[i], attr16Base[i] ) ;
         ++wpSec.ypos ;
      }

      //* Draw the section header *
      wp.ypos += 10 ;
      WriteString ( wp.ypos, (wp.xpos + 1), 
                    "         Additional foreground/background "
                    "color combinations        ", this->bwU ) ;

      //* Display the 'extended' color attributes for color pairs 48 through 127.*
      const short VAR7F = 56 ;
      const char* text7F[VAR7F] = 
      {
      "nc.brebk ", "nc.bgrre ", "nc.bbrgr ", "nc.bblbr ", "nc.bmabl ", "nc.bcyma ", "nc.bgycy ", 
      "nc.bgrbk ", "nc.bbrre ", "nc.bblgr ", "nc.bmabr ", "nc.bcybl ", "nc.bgyma ", "nc.bbkgy ", 
      "nc.bbrbk ", "nc.bblre ", "nc.bmagr ", "nc.bcybr ", "nc.bgybl ", "nc.bbkcy ", "nc.bregy ", 
      "nc.bblbk ", "nc.bmare ", "nc.bcygr ", "nc.bgybr ", "nc.bbkma ", "nc.brecy ", "nc.bgrgy ", 
      "nc.bmabk ", "nc.bcyre ", "nc.bgygr ", "nc.bbkbl ", "nc.brema ", "nc.bgrcy ", "nc.bbrgy ", 
      "nc.bcybk ", "nc.bgyre ", "nc.bbkbr ", "nc.brebl ", "nc.bgrma ", "nc.bbrcy ", "nc.bblgy ", 
      "nc.bgybk ", "nc.bbkgr ", "nc.brebr ", "nc.bgrbl ", "nc.bbrma ", "nc.bblcy ", "nc.bmagy ", 
      "nc.bbkre ", "nc.bregr ", "nc.bgrbr ", "nc.bbrbl ", "nc.bblma ", "nc.bmacy ", "nc.bcygy ", 
      } ;
      const attr_t attr7F[VAR7F] = 
      {
      this->brebk, this->bgrre, this->bbrgr, this->bblbr, this->bmabl, this->bcyma, this->bgycy, 
      this->bgrbk, this->bbrre, this->bblgr, this->bmabr, this->bcybl, this->bgyma, this->bbkgy, 
      this->bbrbk, this->bblre, this->bmagr, this->bcybr, this->bgybl, this->bbkcy, this->bregy, 
      this->bblbk, this->bmare, this->bcygr, this->bgybr, this->bbkma, this->brecy, this->bgrgy, 
      this->bmabk, this->bcyre, this->bgygr, this->bbkbl, this->brema, this->bgrcy, this->bbrgy, 
      this->bcybk, this->bgyre, this->bbkbr, this->brebl, this->bgrma, this->bbrcy, this->bblgy, 
      this->bgybk, this->bbkgr, this->brebr, this->bgrbl, this->bbrma, this->bblcy, this->bmagy, 
      this->bbkre, this->bregr, this->bgrbr, this->bbrbl, this->bblma, this->bmacy, this->bcygy, 
      } ;

      wpSec = wp ;
      for ( short i = ZERO ; i < VAR7F ; i++ )
      {
         if ( i % 7 == ZERO )
         { ++wpSec.ypos ; wpSec.xpos = wp.xpos ; }
   
         wpSec = WriteString ( wpSec, text7F[i], attr7F[i] ) ;
         ++wpSec.xpos ;
      }
   }

   this->RefreshScreen () ;
   if ( pause )
      nckPause() ;

   curs_set ( cursor_state ) ;         // restore cursor attribute

}  //* End DisplayDefinedColors() *

//*************************
//*     DebugBorder       *
//*************************
//******************************************************************************
//* Draw a fake border around an imaginary window.                             *
//*                                                                            *
//* Input  : wLines: verical size of area in lines                             *
//*          wCols : horizontal size of area in columns                        *
//*          wulY  : Y offset from upper left of terminal window               *
//*          wulX  : X offset from upper left of terminal window               *
//*          attr  : color attribute for drawing                               *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* NOTE: THIS METHOD IS FOR USE ONLY WITH THE DEVELOPMENT METHODS:            *
//*       DisplayDefinedColors(), DisplayColorOptions(), and                   *
//*       DisplayAlternateFont().                                              *
//*       For application-window borders use:                                  *
//*           void NcWindow::DrawBorder ( attr_t cAttr, const char* title ) ;  *
//*                                                                            *
//* Programmer's Note:                                                         *
//* - The wborder() function of the ncurses library and all its associated     *
//*   macros have a bug that shows up when painting a border on a sub-         *
//*   window: the draw trashes the left side of the parent window because      *
//*   the routine draws spaces from the left edge of the parent window         *
//*   before drawing the border characters.  This bug may be masked when       *
//*   using the system default background, but try using a non-default         *
//*   background to prove it to yourself.                                      *
//*                                                                            *
//******************************************************************************

void  NCurses::DebugBorder ( short wLines, short wCols, short wulY, short wulX, attr_t attr )
{
   //* Set the line-draw characters *
   wchar_t  ulCorner = wcsUL, urCorner = wcsUR, llCorner = wcsLL, 
            lrCorner = wcsLR, vLine = wcsVERT, hLine = wcsHORIZ ;
   winPos wp( wulY, wulX ) ;

   this->WriteChar ( wp.ypos, wp.xpos++, ulCorner, attr ) ;
   while ( wp.xpos < (wulX+wCols-1) )     // top border
      this->WriteChar ( wp.ypos, wp.xpos++, hLine, attr ) ;
   this->WriteChar ( wp.ypos, wp.xpos, urCorner, attr ) ;

   while ( ++wp.ypos < (wulY+wLines-1) )  // left and right borders
   {
      this->WriteChar ( wp.ypos, wulX, vLine, attr ) ;
      this->WriteChar ( wp.ypos, wp.xpos, vLine, attr ) ;
   }

   wp.xpos = wulX ;
   this->WriteChar ( wp.ypos, wp.xpos++, llCorner, attr ) ;
   while ( wp.xpos < (wulX+wCols-1) )     // bottom border
      this->WriteChar ( wp.ypos, wp.xpos++, hLine, attr ) ;
   this->WriteChar ( wp.ypos, wp.xpos, lrCorner, attr ) ;
   this->RefreshScreen () ;

}  //* End DebugBorder

//*************************
//*  Get_NCurses_Version  *
//*************************
//******************************************************************************
//* Returns a pointer to NCursesVersion, the NCurses class version number.     *
//*                                                                            *
//* Input  :none                                                               *
//*                                                                            *
//* Returns: (const char*) pointer to version string                           *
//******************************************************************************

const char* NCurses::Get_NCurses_Version ( void )
{

   return NCursesVersion ;

}  //* End Get_NCurses_Version() *

//*************************
//* Get_nclibrary_Version *
//*************************
//******************************************************************************
//* Returns a pointer to the ncurses (curses) library version number.          *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: (const char*) pointer to version string                           *
//******************************************************************************

const char* NCurses::Get_nclibrary_Version ( void )
{
   //* If we have not already captured    *
   //* the version string, capture it now.*
   if ( *this->nclibVersion == NULLCHAR )
   {
      const char* cptr = curses_version () ;
      while ( *cptr < '0' || *cptr > '9' )
         ++cptr ;
      short i ;
      for ( i = ZERO ; i < 15 ; ++i )  // (target buffer is 16 bytes)
      {
         if ( (this->nclibVersion[i] = cptr[i]) == NULLCHAR )
            break ;
      }
      this->nclibVersion[i] = NULLCHAR ;  // be sure string is terminated
   }
   return this->nclibVersion ;

}  //* End Get_nclibrary_Version() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

