//******************************************************************************
//* File       : NcWindow.hpp                                                  *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2005-2025 Mahlon R. Smith, The Software Samurai *
//*                 GNU GPL copyright notice below                             *
//* Date       : 20-Mar-2025                                                   *
//* Version    : (see NcWindowVersion string in NcWindow.cpp)                  *
//*                                                                            *
//* Description: This class implements a companion class to the NCurses class. *
//* This class instantiates ncurses-based windows and allows access to all     *
//* attributes of those windows while encapsulating many of the messy and      *
//* frankly primitive details of the ncurses function library.                 *
//*                                                                            *
//* Note: While it is possible to write a simple console utility using the     *
//* NCurses and NcWindow classes directly, this is not recommended.            *
//* Please see the derived class, NcDialog for a fully-realized API.           *
//*                                                                            *
//*                                                                            *
//* Development Tools: See NcWindow.cpp.                                       *
//******************************************************************************
//* Version History (most recent first):                                       *
//*   See version history in NcWindow.cpp                                      *
//*                                                                            *
//******************************************************************************
//* Copyright Notice:                                                          *
//* This program is free software: you can redistribute it and/or modify it    *
//* under the terms of the GNU General Public License as published by the Free *
//* Software Foundation, either version 3 of the License, or (at your option)  *
//* any later version.                                                         *
//*                                                                            *
//* This program is distributed in the hope that it will be useful, but        *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
//* for more details.                                                          *
//*                                                                            *
//* You should have received a copy of the GNU General Public License along    *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.            *
//*                                                                            *
//*         Full text of the GPL License may be found in the Texinfo           *
//*         documentation for this program under 'Copyright Notice'.           *
//******************************************************************************

#ifndef NCWINDOW_INCLUDED
#define NCWINDOW_INCLUDED

//* - * - * - * - * - * - * - * - * - * - * - * - * - * - * - * - * - * - * - **
//* Thread-safe operation:                                                     *
//* -- Thread safety for the NcWindow class (and all its descendants) is       *
//*    controlled by the NCD_THREAD_SAFETY flag, which allows for temporarily  *
//*    disabling the thread safety mutexes during testing.                     *
//*    However, this flag should be enabled for all production builds.         *
//*          (See additional notes in the NcwKey.cpp source module.)           *
//* - * - * - * - * - * - * - * - * - * - * - * - * - * - * - * - * - * - * - **
#define NCD_THREAD_SAFETY (1) // Enable thread-safe operation

//** Definitiions **
const char HOTCHAR = CARET ;     //* Indicator for menu-item shortcut key

//* Line-drawing codes for drawing lines and window borders *
enum ncLineType : short 
{ 
   ncltHORIZ,  ncltVERT, 
   ncltSINGLE, ncltSINGLEBOLD, ncltDUAL,
   ncltDASH2,  ncltDASH2BOLD, 
   ncltDASH3,  ncltDASH3BOLD,
   ncltDASH4,  ncltDASH4BOLD
} ;

//* Supported mouse-event types. Reported in wkeyCode.mevent.meType            *
//* The number of mouse buttons supported by your hardware and mouse driver    *
//* will determine which of these event types will be available on your system.*
//* See also wkeyCode.mevent members 'cKey', 'sKey' and 'altKey' for           *
//* conditioned mouse events.                                                  *
enum meTypes : short
{
   metNONE = ZERO,   // No event
   metB1_P,          // Button 1 - button press
   metB1_R,          // Button 1 - button release
   metB1_S,          // Button 1 - single click
   metB1_D,          // Button 1 - double click
   metB1_T,          // Button 1 - triple click
   metB2_P,          // Button 2 - button press
   metB2_R,          // Button 2 - button release
   metB2_S,          // Button 2 - single click
   metB2_D,          // Button 2 - double click
   metB2_T,          // Button 2 - triple click
   metB3_P,          // Button 3 - button press
   metB3_R,          // Button 3 - button release
   metB3_S,          // Button 3 - single click
   metB3_D,          // Button 3 - double click
   metB3_T,          // Button 3 - triple click
   metB4_P,          // Button 4 - button press
   metB4_R,          // Button 4 - button release
   metB4_S,          // Button 4 - single click
   metB4_D,          // Button 4 - double click
   metB4_T,          // Button 4 - triple click
   metB5_P,          // Button 5 - button press
   metB5_R,          // Button 5 - button release
   metB5_S,          // Button 5 - single click
   metB5_D,          // Button 5 - double click
   metB5_T,          // Button 5 - triple click
   metPOS,           // Mouse pointer position report
   metSW_D,          // Scroll-wheel - scroll down
   metSW_U,          // Scroll-wheel - scroll up
   metTYPES
} ;

//* Line drawing definition as parameter for DrawLine().                       *
//* Note that lines are drawn left-to-right and top-to-bottom.                 *
//* By default, as new lines are drawn, they will form a visual connection     *
//* with existing lines. To prevent this behavior, set the appropriate flag to *
//* 'false'.
class LineDef
{
   public:
   LineDef( ncLineType t, ncLineType s, short y, short x, short l, attr_t c ) :
         type(t), style(s), startY(y), startX(x), length(l), color(c)
         {
            cInsect  = true ;
            cLeft    = true ;
            cRight   = true ;
            cTop     = true ;
            cBot     = true ;
         }
   ncLineType  type ;      // line type (ncltHORIZ or ncltVERT)
   ncLineType  style ;     // line style: member of enum ncLineType
   short       startY ;    // start position Y
   short       startX ;    // start position X
   short       length ;    // line length in columns (horiz) or rows (vert)
   attr_t      color ;     // color attribute
   bool        cInsect ;   // 'true' create an intersection with an existing line 
                           //       (other than at new line's endpoints)
   bool        cLeft ;     // 'true' connect on left with an existing line
   bool        cRight ;    // 'true' connect on right with an existing line
   bool        cTop ;      // 'true' connect on top with an existing line
   bool        cBot ;      // 'true' connect on bottom with an existing line
} ;

//* Define an object to hold captured display data for the window object.      *
//* NOTE: 'cData' and 'aData' are dynamically-allocated memory within this     *
//* class. For safety, allocation is slightly larger than the data they hold.  *
class SaveWin
{
   public:
   //* Initialization constructor *
   SaveWin ( short r, short c, short y = ZERO, short x = ZERO )
   {
      this->wRows = r ;
      this->wCols = c ;
      this->wulY  = y ;
      this->wulX  = x ;
      this->cData = new wchar_t[(this->wRows * this->wCols + 31) & 0xFFF0] ;
      this->wText = this->cData ;
      this->aData = new attr_t[(this->wRows * this->wCols + 31) & 0xFFF0] ;
      this->wAttr = this->aData ;
   }
   //* Destructor *
   // Do not call directly; use 'delete' 
   // or let object go out of scope.
   ~SaveWin ( void )
   {
      if ( this->cData != NULL )    delete[] this->cData ;
      if ( this->aData != NULL )    delete[] this->aData ;
   }
   short wRows ;           // number of display lines of text saved
   short wCols ;           // number of text characters per line
   short wulY ;            // Y offset from upper-left
   short wulX ;            // X offset from upper-left
   const wchar_t* wText ;  // pointer to text data
   const attr_t*  wAttr ;  // pointer to attribute data

   protected:
   wchar_t *cData ;  // pointer to memory block allocated to hold character data
   attr_t  *aData ;  // pointer to memory block allocated to hold color attribute data
   friend class NcWindow ;

   private:
   //* Default constructor - included for safety. If a parameterless instance  *
   //* is declared, then the compiler will complain. (you're welcome :-)       *
   SaveWin ( void ) {}
} ;


class NcWindow
{
public:
   virtual ~NcWindow () ;

//* Constructor for a NcWindow object.                                         *
//* Input  : lines   : number of lines in the window                           *
//*          columns : number of columns in the window                         *
//*          begin_y : vertical offset from upper left corner of screen        *
//*          begin_x : horizontal offset from upper left of screen             *
//* Returns: nothing                                                           *
NcWindow ( short lines, short columns, short startY, short startX ) ;

//* Allocate resources for the window based upon the parameters passed to      *
//* the constructor.                                                           *
//* Input  : none                                                              *
//* Returns: OK if successfully opened, else ERR                               *
//*          Note that most likely causes of error are:                        *
//*           1. terminal screen too small to hold the window plus             *
//*              offsets from upper-left corner                                *
//*           2. insufficient heap memory                                      *
short OpenWindow ( void ) ;

//* Redraw the entire contents of the window *
void RefreshWin ( void ) ;

//* Erase the contents of the window and refresh the redisplay.                *
//*                                                                            *
//* Input  : clearAll: (optional, true by default)                             *
//*                    if false do not clear window border.                    *
//*                    else, clear entire window.                              *
//*          refresh : (optional, true by default)                             *
//*                    if true, refresh the display                            *
//*                                                                            *
//* Returns: OK                                                                *
short ClearWin ( bool clearAll = true, bool refresh = true ) ;

//* Clear from specified cursor position to end of line. Display is refreshed. *
//*                                                                            *
//* Input  : ypos : line number (zero-based)                                   *
//*          clearAll: (optional, false by default) if 'true', clear entire    *
//*                    line, if 'false' do not clear border area               *
//*          xpos : (optional, by default, for LTR: one(1), for RTL: columns-2)*
//*                 starting column (zero-based)                               *
//*                 (ignored if clearAll != false)                             *
//*          rtl  : (optional, 'false' by default)                             *
//*                 if 'false', then clear from 'xpos' toward right            *
//*                 if 'true',  then clear from 'xpos' toward left             *
//*                 (ignored if clearAll != false)                             *
//* Returns: OK if successful, ERR if invalid line index                       *
short ClearLine ( short ypos, bool clearAll = false, short xpos = 1, bool rtl = false ) ;

//* Get the window's cursor position.                                          *
//* Input  : none                                                              *
//* Returns: a winPos class object containing the Y/X cursor position          *
winPos GetCursor ( void ) ;

//* Set the window's cursor position.                                          *
//* Input  : ypos : Y offset from upper-left corner (zero-based)               *
//*          xpos : X offset from upper-left corner (zero-based)               *
//*            OR                                                              *
//*          yxpos: Y/X offset from upper-left corner (zero-based)             *
//* Returns: OK if successful, ERR if parameter(s) out of range                *
short  SetCursor ( short ypos, short xpos ) ;
short  SetCursor ( winPos yxpos ) ;

//* Draw an inside border around the window.                                   *
//*                                                                            *
//* Input  : cAttr  : color attribute                                          *
//*          title  : (optional, default==NULL)                                *
//*                   if non-NULL, points to window title string               *
//*          style  : (optional, ncltSINGLE by default)                        *
//*                   member of enum ncLineType                                *
//*          rtlText: (optional, false by default)                             *
//*                   if 'title' != NULL, then:                                *
//*                   if false, draw as LTR text, if true, draw as RTL text    *
//* Returns: If no title string is specified, returns OK on success,           *
//*           else ERR (window too small, must be at least 2 lines by 2 cols)  *
//*          If a title string IS specified, returns the X offset at           *
//*           at which the title is displayed.                                 *
short DrawBorder ( attr_t cAttr, const char* title=NULL, 
                   ncLineType style = ncltSINGLE, bool rtlText = false ) ;

//* Draw a rectangle within the window.                                        *
//* Note: For drawing a window border, use the Border() method instead.        *
//*                                                                            *
//* Input  : startY : y offset for start of line                               *
//*          startX : x offset for start of line                               *
//*          rows   : number of rows                                           *
//*          cols   : number of columns                                        *
//*          cAttr  : color attribute                                          *
//*          title  : (optional, default==NULL)                                *
//*                   if non-NULL, points to box title string                  *
//*          style  : (optional, ncltSINGLE by default)                        *
//*          rtlText: (optional, false by default)                             *
//*                   if 'title' != NULL, then:                                *
//*                   if false, draw as LTR text, if true, draw as RTL text    *
//* Returns: If no title string is specified, returns OK on success,           *
//*           else ERR (box bigger than parent window)                         *
//*          If a title string IS specified, returns the X offset at           *
//*           at which the title is displayed.                                 *
short DrawBox ( short startY, short startX, short rows, short cols, attr_t cAttr, 
                const char* title=NULL, 
                ncLineType style = ncltSINGLE, bool rtlText = false ) ;

//* Set a background color for the interior of the window.                     *
//* Does not redraw the window in the new color.                               *
//* Input  : cAttr : color attribute                                           *
//* Returns: nothing                                                           *
void SetInteriorColor ( attr_t cAttr ) ;

//* Draw a line of the specified type and style (see enum ncLineType).         *
//* Note that by default any contact between the new line and any existing     *
//* lines will generate a visual connection.                                   *
//* To prevent automagical line connections, please set the appropriate flag(s)*
//* (cLeft, cRight, cTop, cBot, cInsect) to 'false'.                           *
//*                                                                            *
//* Input  : lDef (by reference)                                               *
//*             fields : type  : ncltHORIZ or ncltVERT                         *
//*                      style : normal, bold, dual-line, or one of the dashed *
//*                              line types                                    *
//*                      startY: y offset for start of line                    *
//*                      startX: x offset for start of line                    *
//*                      length: number of lines (vert) or columns (horiz)     *
//*                      attr  : color attribute                               *
//* Returns: OK if success, else ERR                                           *
short DrawLine ( const LineDef& ldef ) ;

//* Returns the dimensions of the window expressed in lines and columns *
void WinDimensions ( short& lines, short& columns ) ;

//* Write a character string at the specified position in the window.          *
//* Optionally, refresh the display before return to caller.                   *
//*                                                                            *
//* Input  : startY : Y cursor start position                                  *
//*          startX : X cursor start position                                  *
//*                OR                                                          *
//*          startYX: Y/X cursor start position                                *
//*                                                                            *
//*          FOR UTF-8-ENCODED CHARACTERS (INCLUDING ASCII)                    *
//*            uStr : pointer to null-terminated UTF-8-encoded string          *
//*          OR FOR wchar_t 'WIDE' CHARACTERS                                  *
//*            wStr : pointer to null-terminated 'wide' wchar_t string         *
//*          FOR DIRECT DISPLAY OF DATA STRING IN A gString CLASS OBJECT       *
//*            gStr : a gString-class object (by reference)                    *
//*          NOTE: String length will be truncated at right edge of window     *
//*          NOTE: ASCII control characters, 0x01 through 0x1F are ignored.    *
//*                                                                            *
//*          cAttr  : color attribute                                          *
//*          refresh: (optional, default==false) refresh display window        *
//*          rtl    : (optional, 'false' by default) for RTL written languages *
//*                   if 'true', cursor moves from right-to-left               *
//*                                                                            *
//* Returns: returns final position of cursor                                  *
//*          If specified starting Y/X position is valid, then cursor position *
//*           returned will be the the column following the last character of  *
//*           displayed data (or the last column of the target line).          *
//*          If specified starting position is invalid, then cursor position   *
//*           is set to window origin (ZERO/ZERO) and no data will be written. *
winPos WriteString ( short startY, short startX, const char* uStr, 
                     attr_t cAttr, bool refresh = false, bool rtl = false ) ;
winPos WriteString ( const winPos& startYX, const char* uStr, 
                     attr_t cAttr, bool refresh = false, bool rtl = false ) ;
winPos WriteString ( short startY, short startX, const wchar_t* wStr, 
                     attr_t cAttr, bool refresh = false, bool rtl = false ) ;
winPos WriteString ( const winPos& startYX, const wchar_t* wStr, 
                     attr_t cAttr, bool refresh = false, bool rtl = false ) ;
winPos WriteString ( short startY, short startX, const gString& gStr, 
                     attr_t cAttr, bool refresh = false, bool rtl = false ) ;
winPos WriteString ( const winPos& startYX, const gString& gStr, 
                     attr_t cAttr, bool refresh = false, bool rtl = false ) ;

//* Write a multi-line string, starting at the specified position.             *
//* A newline character '\n' signals the end of each line.                     *
//* Automagic line-wrapping is not supported. It is the caller's responsibility*
//* to correctly size each line to fit within the number of columns available  *
//* in the target window.                                                      *
//*                                                                            *
//* NOTE: Text will not be written outside window boundaries.                  *
//* NOTE: Except for the newline character '\n', ASCII control characters,     *
//*       0x01 through 0x1F are ignored.                                       *
//*                                                                            *
//* Input  : startY : Y cursor start position                                  *
//*          startX : X cursor start position                                  *
//*                OR                                                          *
//*          startYX: Y/X cursor start position                                *
//*                                                                            *
//*          gStr   : a gString-class object (by reference)                    *
//*                OR                                                          *
//*          wStr   : pointer to a wchar_t string                              *
//*                OR                                                          *
//*          uStr   : pointer to a UTF-8 (char*) string                        *
//*                                                                            *
//*          cAttr  : color attribute                                          *
//*          refresh: (optional, 'false' by default) refresh display window    *
//*          rtl    : (optional, 'false' by default) for RTL written languages *
//*                   if 'true', cursor moves from right-to-left               *
//*                                                                            *
//* Returns: final position of cursor                                          *
winPos WriteParagraph ( short startY, short startX, const gString& gStr, 
                        attr_t cAttr, bool refresh = false, bool rtl = false ) ;
winPos WriteParagraph ( const winPos& startYX, const gString& gStr, 
                        attr_t cAttr, bool refresh = false, bool rtl = false ) ;
winPos WriteParagraph ( short startY, short startX, const wchar_t* wStr, 
                        attr_t cAttr, bool refresh = false, bool rtl = false ) ;
winPos WriteParagraph ( const winPos& startYX, const wchar_t* wStr, 
                        attr_t cAttr, bool refresh = false, bool rtl = false ) ;
winPos WriteParagraph ( short startY, short startX, const char* uStr, 
                        attr_t cAttr, bool refresh = false, bool rtl = false ) ;
winPos WriteParagraph ( const winPos& startYX, const char* uStr, 
                        attr_t cAttr, bool refresh = false, bool rtl = false ) ;


//* Write a single character at the specified position.                        *
//* Optionally, refresh the display before return to caller.                   *
//*                                                                            *
//* Input  : startY : Y cursor position                                        *
//*          startX : X cursor position                                        *
//*                OR                                                          *
//*          startYX: Y/X cursor start position                                *
//*                                                                            *
//*          FOR UTF-8-ENCODED CHARACTERS (INCLUDING ASCII)                    *
//*            uChar: pointer to character to display                          *
//*                   (single-byte or multi-byte character)                    *
//*          OR FOR wchar_t 'WIDE' CHARACTERS                                  *
//*            wChar: character to display in wchar_t format                   *
//*          NOTE: ASCII control characters, 0x01 through 0x1F are ignored.    *
//*                                                                            *
//*          cAttr  : color attribute                                          *
//*          refresh: (optional, default==false) refresh display window        *
//*          rtl    : (optional, 'false' by default) for RTL written languages *
//*                   if 'true', cursor moves from right-to-left               *
//*                                                                            *
//* Returns: returns final position of cursor                                  *
//*          If specified starting Y/X position is valid, AND the specified    *
//*           character is a printing character, then cursor position returned *
//*           will be the the column following the character (or the last      *
//*           column on the target line).                                      *
//*          If specified starting position is invalid, OR if non-printing     *
//*           character specified, then cursor position is set to window       *
//*           origin (ZERO/ZERO) and no data will be written.                  *
winPos WriteChar ( short startY, short startX, const char* uChar, 
                   attr_t cAttr, bool refresh = false, bool rtl = false ) ;
winPos WriteChar ( const winPos& startYX, const char* uChar, 
                   attr_t cAttr, bool refresh = false, bool rtl = false ) ;
winPos WriteChar ( short startY, short startX, wchar_t wChar, 
                   attr_t cAttr, bool refresh = false, bool rtl = false ) ;
winPos WriteChar ( const winPos& startYX, wchar_t wChar, 
                   attr_t cAttr, bool refresh = false, bool rtl = false ) ;

//* Get a keycode and keytype from the ncursesw input stream.                  *
//* This includes all supported international character codes as well as       *
//* captured-and-translated escape sequences representing function keys,       *
//* cursor keys and other special-purpose key combinations.                    *
//*                                                                            *
//* If mouse input has been enabled, mouse events will also be returned in     *
//* caller's wkeyCode-class object's 'mevent' data member.                     *
//*                                                                            *
//* - This method is the thread-safe version of the lower-level keyboard       *
//*   interface in the NCurses class.                                          *
//* - In addition to the simple filtering of invalid input done by the         *
//*   NCurses-class version of this method, significant filtering of           *
//*   unrequested and/or broken mouse input is performed.                      *
//*                                                                            *
//* Input  : wKey (by reference, initial values ignored)                       *
//*          receives the key input data                                       *
//* Returns: member of enum wkeyType                                           *
wkeyType GetKeyInput ( wkeyCode& wKey ) ;


//*****************************
//* Mouse Configuration group *
//*****************************
//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *
//* Notes on the mouse interface:                                              *
//* a) The xterm mouse driver, the ncurses(w) mouse interface, and therefore   *
//*    the NCurses-class mouse interface are completely useless as distributed,*
//*    so if you value your sanity, please don't access them directly.         *
//*    Instead, use the application-layer mouse interface presented here.      *
//* b) The NcWindow-class mouse interface does its best to overcome its        *
//*    primitive roots, and we've been moderately successful, so please have   *
//*    a go; to determine whether adding a mouse interface is right for your   *
//*    application.                                                            *
//* c) Refer to the online documentation (Texinfo 'info' or 'HTML' format)     *
//*    for a detailed discussion of shovelling mouse droppings.                *
//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *

//* Enable reporting of mouse events.                                          *
//* Mouse events are reported within the 'GetKeyInput' keyboard input stream.  *
//*                                                                            *
//* Input  : eventTypes : bit mask of event types to be enabled                *
//*                       Bit definitions are the BUTTONxxxx group of          *
//*                       macros defined in ncurses.h (yes, it's clunky,       *
//*                       but you only have to define the mask once).          *
//*                       NOTE:                                                *
//*                       If no mouse-event types are specified,               *
//*                       (eventTypes==ZERO), then the following mask is set   *
//*                       as the default:                                      *
//*                          eventTypes = BUTTON1_CLICKED |                    *
//*                                       BUTTON1_DOUBLE_CLICKED |             *
//*                                       REPORT_SCROLL_WHEEL;                 *
//*          interval  : (optional, ncmiDFLT by default)                       *
//*                      the maximum interval to wait between the start of an  *
//*                      event and the end of the event in thousandths (0.001) *
//*                      of a second. Specify an interval OR a member of       *
//*                      enum ncmInterval.                                     *
//* Returns: OK  if mouse-event reporting enabled AND the system supports all  *
//*              requested event types                                         *
//*          ERR if mouse could not be enabled                                 *
//*           OR if optionally-specified 'interval' value is out of range      *
//*           OR one or more of the requested event types is not supported by  *
//*              the system (remaing event types will still be reported)       *
//*              In this case, the meGetEventTypes() method should be called   *
//*              to determine which events will be reported.                   *
short meEnableMouse ( mmask_t eventTypes, short interval = ncmiDFLT ) ;

//* Enable enhanced reporting of mouse events.                                 *
//* Mouse events are reported within the 'GetKeyInput' keyboard input stream.  *
//*                                                                            *
//* This form of enabling the mouse interface is different from the more       *
//* general meEnableMouse() method above. Whereas meEnableMouse() will continue*
//* even if the system does not fully support the requested configuration,     *
//* this method requires that all necessary parameters be fully configured.    *
//* If the system cannot be configured as required, then the mouse interface   *
//* is disabled and ERR is returned.                                           *
//*                                                                            *
//* The purpose of this configuration method is to reduce the unfortunate      *
//* timing artifacts inherited from the xterm mouse driver and from the        *
//* ncurses(w) C-language library. It is hoped that this will reduce the loss  *
//* of valid mouse-event data and improve overall mouse performance.           *
//*                                                                            *
//* Enable synthesized timing for mouse events, AND restrict the mouse         *
//* interface to event types:                                                  *
//*      metB1_S : Button 1 - single click                                     *
//*      metB1_D : Button 1 - double click                                     *
//*      metSW_U : Scroll-wheel - scroll up   (see 'swheel' parameter)         *
//*      metSW_D : Scroll-wheel - scroll down                                  *
//*      with and without modifier keys (CTRL, SHIFT, ALT).                    *
//*                                                                            *
//* Input  : dclick : (optional, ncmiSTABLE by default)                        *
//*                   This parameter indicates the time to wait after the      *
//*                   first click event is received to determine whether       *
//*                   the event is a double-click.                             *
//*                   Specified in thousandths (0.001) of a second OR a        *
//*                   member of enum ncmInterval.                              *
//*          swheel : (optional, 'true' by default)                            *
//*                   'true'  : ScrollWheel events are reported                *
//*                   'false' : ScrollWheel events ARE NOT reported            *
//* Returns: OK  if mouse-event reporting enabled and the system supports all  *
//*              required mouse event types and specified timing parameters.   *
//*          ERR if:                                                           *
//*            a) mouse could not be enabled                                   *
//*            b) if optionally-specified 'interval' value is out of range     *
//*            c) one or more required event types not supported by the system *
//*            d) the required timing interval could not be established        *
//*            If ERR, then mouse interface will be disabled.                  *
short meEnableStableMouse ( short dclick = ncmiSTABLE, bool swheel = true ) ;

//* Disable reporting of mouse events.                                         *
//*                                                                            *
//* Input  : none                                                              *
//* Returns: OK  if mouse events successfully disabled, else ERR               *
short meDisableMouse ( void ) ;

//* Reports whether mouse input is currently enabled.                          *
//*                                                                            *
//* Input  : none                                                              *
//* Returns: 'true'  if mouse-event reporting enabled, else, 'false'           *
bool  meMouseEnabled ( void ) ;

//* Modify the type(s) of mouse events which will be reported.                 *
//*                                                                            *
//* Input  : eventTypes : bit mask of event types to be enabled                *
//* Returns: OK  if ALL requested event-types are supported by the system      *
//*          ERR if mouse-event reporting not enabled                          *
//*           OR no event types specified                                      *
//*           OR one or more of the requested event types is not supported by  *
//*              the system (remaing event types will still be reported)       *
short meSetEventTypes ( mmask_t eventTypes ) ;

//* Returns the mouse-event-type mask for currently-enabled event types.       *
//* Generally useful only if meEnableMouse() method returns an error.          *
//*                                                                            *
//* Input  : eventTypes : (by reference) receives the event mask               *
//* Returns: OK  if successful                                                 *
//*          ERR if mouse-event reporting not enabled                          *
short meGetEventTypes ( mmask_t& eventTypes ) ;

//* Get the current maximum mouse-click interval in thousandths of a second.   *
//*                                                                            *
//* Input  : waitMax   : (by reference, initial value ignored)                 *
//*                      receives current interval                             *
//* Returns: OK  if successtul                                                 *
//*          ERR if mouse-event input disabled                                 *
short meGetClickInterval ( short& waitMax ) ;

//* Specify the maximum interval between a button-press event and a following  *
//* button-release event that will result in a mouse 'click' being reported.   *
//* Note: this interval applies to single-, double- and triple-click events.   *
//* If the interval between press and release is greater than this value, then *
//* a 'click' event will not be reported.                                      *
//* Note that this inverval is optionally set when mouse-event reporting is    *
//* is enabled through the meEnableMouse method.                               *
//*                                                                            *
//* Input  : waitMax   : interval in thousandths (0.001) of a second           *
//*                      specify an interval OR a member of enum ncmInterval   *
//*          oldWait   : (optional, NULL pointer by default)                   *
//*                      pointer to variable to receive previous interval value*
//* Returns: OK  if successtul                                                 *
//*          ERR if value out-of-range OR mouse-event input disabled           *
short meSetClickInterval ( short waitMax, short* oldMax = NULL ) ;

//* Remove any mouse events waiting in the mouse-event queue.                  *
//*                                                                            *
//* Input  : none                                                              *
//* Returns: OK  if successful                                                 *
//*          ERR if mouse-event reporting not enabled                          *
short meFlushEventQueue ( void ) ;

//* Within an NcDialog window, mouse event are, by default, automatically and  *
//* intelligently converted to keycodes (when possible). This eliminates the   *
//* need for the application to handle mouse events manually. In general, the  *
//* less the application needs to know about mouse events, the better.         *
//* There may be circumstances, however, when you need to temporarily disable  *
//* this automatic conversion, but use caution: It means that the edit routines*
//* for the dialog controls will return from the edit with the unprocessed     *
//* mouse event, and your user-input loop needs to know what to do with it.    *
//*                                                                            *
//* Input  : enable : 'true'  to enable auto-conversion (default setting)      *
//*                   'false' to disable auto-conversion                       *
//* Returns: OK  if successful                                                 *
//*          ERR if mouse-event reporting not enabled                          *
short meAutoConvert ( bool enable ) ;

//* Temporarily disable filtering of spurious (unrequested) mouse events.      *
//* Filtering is enabled by default. Should REMAIN enabled except for testing. *
//*                                                                            *
//* Input  : filter : 'true' enables filtering                                 *
//*                   'false disables filtering                                *
//* Returns: OK if successful, or ERR if mouse-event input disabled            *
short meAutoFilter ( bool filter ) ;


//****************************
//* Scroll-Data method group *
//****************************
//* Display data in window and initialize scroll-tracking variables *
//* When finished, call EraseData() to clear the window and reset   *
//* the internal tracking data.                                     *
short PaintData ( const char* dataPtr[], short count, short hiliteIndex, 
                  attr_t color, bool hilite = true, bool rtlFmt = false ) ;
short PaintData ( const char* dataPtr[], const attr_t colorData[], short count, 
                  short hiliteIndex, bool hilite = true, bool rtlFmt = false ) ;
short PaintMenuData ( const char* textData[], short count, short hiliteIndex, 
                      attr_t color, bool hilite = true, bool rtlFmt = false ) ;
short PaintMenuData ( const char* textData[], attr_t colorData[], short count, 
                      short hiliteIndex, bool hilite = true, bool rtlFmt = false ) ;
short EraseData ( void ) ;

//* Redraw scrolling data according to the parameters of scrollData class *
void  RepaintData ( bool hilite = true ) ;
short ScrollData ( wchar_t scrollKey ) ; //* Move the hilight through the data lines
short ScrollMenuData ( wchar_t scrollKey ) ; //* Move the hilight through menu data
short ScrollView ( wchar_t scrollKey ) ; //* Scroll without hilight
short GetHilightIndex ( void ) ;         //* Returns index of highlighted item
bool  IsScrollKey ( wkeyCode key ) ;     //* Returns true if key == a scrolling key
void  HilightItem ( bool show,           //* Hilight or un-hilight the current line
                    bool refresh = false ) ;

//* Move the logical highlight to the first item in window, and optionally     *
//* update the display.                                                        *
//* If display refresh not specified, then call RepaintData() to refresh.      *
//*                                                                            *
//* Input  : refresh: (optional, false by default)                             *
//*                   if 'true', repaint the data and refresh the display      *
//* Returns: index of the logically-highlighted item i.e. ZERO                 *
short Track2Top ( bool refresh = false ) ;

//* Move the logical highlight to the last item in window, and optionally      *
//* update the display.                                                        *
//* If display refresh not specified, then call RepaintData() to refresh.      *
//*                                                                            *
//* Input  : refresh: (optional, false by default)                             *
//*                   if 'true', repaint the data and refresh the display      *
//*                              (highlight will be visible)                   *
//* Returns: index of the logically-highlighted item                           *
short Track2Bottom ( bool refresh = false ) ;

//* Move the logical highlight to the next (lower) item in window, and         *
//* optionally update the display.                                             *
//* If display refresh not specified, then call RepaintData() to refresh.      *
//*                                                                            *
//* Input  : refresh: (optional, false by default)                             *
//*                   if 'true', repaint the data and refresh the display      *
//*                              (highlight will be visible)                   *
//* Returns: index of the logically-highlighted item                           *
short Track2Next ( bool refresh = false ) ;

//* Move the logical highlight to the previous (higher) item in window, and    *
//* optionally update the display.                                             *
//* If display refresh not specified, then call RepaintData() to refresh.      *
//*                                                                            *
//* Input  : refresh: (optional, false by default)                             *
//*                   if 'true', repaint the data and refresh the display      *
//*                              (highlight will be visible)                   *
//* Returns: index of the logically-highlighted item                           *
short Track2Previous ( bool refresh = false ) ;

//* Move the logical highlight to the specified item item in window, and       *
//* optionally update the display.                                             *
//* If display refresh not specified, then call RepaintData() to refresh.      *
//*                                                                            *
//* Input  : iIndex : index of target item                                     *
//*          refresh: (optional, false by default)                             *
//*                   if 'true', repaint the data and refresh the display      *
//*                              (highlight will be visible)                   *
//* Returns: index of the logically-highlighted item                           *
//*            If iIndex out-of-range, then index of currently-highlighted     *
//*            item is returned.                                               *
short Track2Item ( short iIndex, bool refresh = false ) ;


               //*****************************************
               //** Passthrough Methods to NCurses class**
               //*****************************************
//* These NcWindow pass-through methods are provided for convenience in        *
//* accessing often-used NCurses-class methods of the same name using an       *
//* NcWindow pointer. Please refer to NCurses.hpp for a more complete          *
//* description of each of these methods.                                      *
//* NOTE: All NCurses public methods and public data members are available     *
//*       through the global pointer 'nc' defined in GlobalDef.hpp.            *
//*             ** HOWEVER, They are not thread-safe **                        *
//*       Call these methods through the follow stubs whenever possible        *
//*       to ensure thread-safe operation.                                     *
//*                                                                            *
//* NOTE: The method prototypes for the NcWindow stubs are the same as the     *
//*       matching NCurses-class prototypes.                                   *
//*  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -   *
wkeyType KeyPeek ( wkeyCode& wKey ) ;
short UngetKeyInput ( wkeyCode& wKey ) ;
short FlushKeyInputStream ( void ) ;
ncCurVis GetCursorState ( void ) ;
short SetCursorState ( ncCurVis state ) ;
short RestoreCursorState ( void ) ;
ncKeyProc GetKeyProcState ( void ) ;
short SetKeyProcState ( ncKeyProc procState ) ;
short GetKeyDelay ( void ) ;
short SetKeyDelay ( short delay ) ;
void  ScreenDimensions ( short& scrLines, short& scrColumns ) ;
void  Hibernate ( void ) ;
void  Wake ( void ) ;
const char* Get_NCurses_Version ( void ) ;
const char* Get_nclibrary_Version ( void ) ;


   //* Methods for development and debugging *
   const char* Get_NcWindow_Version ( void ) ;
   void  DebugMsg ( const char* msg, short startY = -1, short startX = 1, 
                                                        bool pause = false ) ; 


//*----------------------------------------------*
//* Derived classes need access to the following *
//* methods and data, but applications do not.   *
//*----------------------------------------------*
protected:

//********
//* DATA *
//********
WINDOW*  wp ;        //* Pointer returned by newwin() or dupwin()
short    wLines ;    //* Number of lines in the window
short    wCols ;     //* Number of columns in the window
short    wulY,       //* Coordinates for upper-lefthand corner of window
         wulX ;
attr_t   wColor ;    //* Background color for interior of window 
                     //* (default==win default, nc.bw)
short    meDClick ;  //* If 'meStable', then double-click interval, else ignored
bool     meStable ;  //* If true, mouse events are restricted to the
                     //* conversion list and no user access to event timing
bool     meAuto ;    //* If true, mouse events are automagically 
                     //* converted to keycodes

class ScrollData
{
   private:
   const char** dptr ;  //* Pointer to an array of data pointers 
                        //* (data lives in the application space)
   const attr_t* cptr ; //* Pointer to an array of color attributes 
                        //* (data lives in the application space)
   attr_t   monocolor ; //* Color attribute for monochromatic data
   short    count ;     //* Number of data items in array
   short    hRow ;      //* Display row for hilighted item
   short    hIndex ;    //* Index of hilighted item
   short    tIndex ;    //* Index of item displayed at top of window
   short    bIndex ;    //* Index of item displayed at bottom of window (or end-of-data)
   bool     rtl ;       //* if 'true', output as RTL data

   friend class NcWindow ;
} scrollData ;

//*********************
//* Protected methods *
//*********************
winPos WriteHotString ( short startY, short startX, const char* ustr, 
                        attr_t attr ) ;
short Hotkey2Index ( wchar_t& hkey ) ;   //* Process 'hotkey' menu selection
winPos WriteChar ( short startY, short startX, gString& ogs, attr_t 
                   cAttr, bool refresh = false, bool rtl = false ) ;
bool dlNextChar ( const LineDef& ldef, short ypos, short xpos, 
                  wchar_t& dChar, attr_t& cAttr ) ;
short meIdentifyClickItem ( NcWindow* wPtr, short yOffset ) ;
short meClickSynthesis ( wkeyCode& wk ) ;
short CaptureWindow ( SaveWin& sw ) ;
void CloseWin ( void ) ;   // release the ncurses window resources

//*******************
//* Private methods *
//*******************
private:
void  AcquireInputLock ( void ) ;
void  ReleaseInputLock ( void ) ;
void  AcquireOutputLock ( void ) ;
void  ReleaseOutputLock ( void ) ;

//* Declare the NcDialog class as a domestic partner.  *
//* (as a derived class, this should not be necessary) *
friend class NcDialog ;

} ;

#endif   // NCWINDOW_INCLUDED

