//********************************************************************************
//* File       : NcdControl.cpp                                                  *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2006-2025 Mahlon R. Smith, The Software Samurai   *
//*                 GNU GPL copyright notice located in NcDialog.hpp             *
//* Date       : 21-Mar-2025                                                     *
//* Version    : (see NcDialogVersion string in NcDialog.cpp)                    *
//*                                                                              *
//* Description: NcDialog is derived from the NcWindow class and implements      *
//* a more specialized window, the dialog window. NcdControl.cpp is a            *
//* support file containing methods for creating and manipulating control        *
//* objects within the dialog window.                                            *
//* This module also contains implementation of the default methods of the       *
//* abstract class, 'class DialogControl', which is inherited by all dialog      *
//* control classes.                                                             *
//*                                                                              *
//* Development Tools: See NcDialog.cpp.                                         *
//********************************************************************************
//* Version History (most recent first):                                         *
//*   See version history in NcDialog.cpp.                                       *
//********************************************************************************
//* To add a new dialog control type:                                            *
//* 1. Add a member to enum DCType just before dctTYPES.                         *
//* 2. Create a new C++ source file NcdControlXX.cpp (note file-naming           *
//*    convention), and add it to the list in Makefile, of the files to be       *
//*    built and linked to create the NcDialog link library, NcDialog.a.         *
//* 3. Create a new class DialogXXX which is derived from the base class,        *
//*    DialogControl (or one of its descendants). The definition of this class   *
//*    lives in NcDialog.hpp.                                                    *
//*    a. create a destructor, DialogXXX::~DialogXXX that releases all           *
//*       resources associated with the control object.                          *
//*    b. create a constructor DialogXXX::DialogXXX which takes as its           *
//*       parameter a pointer to an InitCtrl structure which the application     *
//*       programmer will use to pass initialization data to the dialog window   *
//*       constructor (which in turn, calls the control's constructor).          *
//*    c. AT MINIMUM, create the following methods for the control class in      *
//*       addition to the destructor and constructor:                            *
//*        short DialogXXX::OpenControl ( bool hasFocus = false ) ;              *
//*        void  DialogXXX::RedrawControl ( bool hasFocus ) ;                    *
//*        short DialogXXX::SetOutputFormat ( bool rtlFormat ) ;                 *
//*    d. Create a method, short NcDialog::EditXXX ( uiInfo& info ) ;            *                  
//*        This method allows user to interact with the control.                 *
//*    e. Some or all of the following may also be needed if the default         *
//*        method does not satisfy the requirements for the new control:         *
//*        void DialogXXX::DrawLabel ( NcDialog* dlgPtr, attr_t tColor ) ;       *
//*        void DialogXXX::InitHotkey ( char* src, bool rtl, bool nlStrip ) ;    *
//*        void DialogXXX::RefreshControl ( void ) ;                             *
//*    f. Create any additional methods needed by the control.                   *
//*       IMPORTANT NOTE: ALL METHODS AND DATA BELONGING TO                      *
//*                       THE CONTROL MUST BE 'PROTECED' !!                      *
//*             The application does not have direct access to any control       *
//*             object. Public access to the functionality of any control must   *
//*             be in the form of an NcDialog class method. Thus, the control's  *
//*             methods and data are declared as 'Protected' and the NcDialog    *
//*             class is declared as:   friend class NcDialog;                   *
//* 4. Define how the fields in the InitCtrl structure will be used to pass the  *
//*    necessary information to the constructor. If you do not need a            *
//*    particular field, list it as 'don't care'.                                *
//*    NOTE: Only if absolutely necessary, should you define additional          *
//*          fields in the InitCtrl structure since doing so would break all     *
//*          existing code that defines controls. Instead, if additional         *
//*          information is needed to make the control functional, you           *
//*          should use a default value for it inside the constructor, and then  *
//*          create an additional NcDialog::yyy method that can adjust the       *
//*          control's data after it has been instantiated. For an example of    *
//*          this, see NcDialog::GroupRadiobuttons().                            *
//* 5. The following NcDialog class methods must be made aware of your new       *
//*    control type; this will generally be in the form of this type of          *
//*    statement: else if ( dCtrl[currIndex]->type == dctXXX ) { /*DO STUFF*/}   *
//*     NcDialog::~NcDialog()                                                    *
//*     NcDialog::NcDialog()                                                     *
//*     NcDialog::OpenWindow()                                                   *
//*     NcDialog::RefreshWin()                                                   *
//*     NcDialog::MoveWin()                                                      *
//*     NcDialog::IsHotkey()                                                     *
//*     NcDialog::ControlActive()                                                *
//*     NcDialog::CaptureDialog()                                                *
//*     NcDialog::Dump_uiInfo()                                                  *
//* 6. The following NcDialog class methods must also be made aware of           *
//*    your new control type:                                                    *
//*     NcDialog::ChangedFocusViaHotkey()                                        *
//*     NcDialog::RedrawCurrControl()                                            *
//*     NcDialog::CaptureDialog() (private method)                               *
//* 7. If the new control can change size depending on it state, OR if the       *
//*    control's position or size are not determined in a straightforward        *
//*    manner, then the following method needs to be made aware of it so mouse   *
//*    events can be properly associated with the control.                       *
//*       NcDialog::meTransformEvent()                                           *
//*    The EditXXX() method for the control may also need to be made aware of    *
//*    mouse activity if it is to do more than receive the input focus when      *
//*    selected via mouse event.                                                 *
//* 8. It is strongly recommended that you add one or more additional tests to   *
//*    the Dialog1, Dialog2, Dialog3, Dialog4 or Dialogw sample applications     *
//*    to exercise all functionality related to your new control type, and to    *
//*    ensure that your new control plays nicely with its brothers and sisters.  *
//*                                                                              *
//* Important Note: DO NOT hack the NcWindow class or NCurses class to make      *
//*                 your new control type work. This would not only violate      *
//*                 the basic C++ principles of abstraction, inheritance, etc.,  *
//*                 but would mightily offend me. Additional functionality can   *
//*                 of course be added to these classes, but should not be done  *
//*                 with a specific NcDialog class control type in mind, since   *
//*                 NcDialog controls are Grandchildren of the NcWindow class.   *
//********************************************************************************

//*****************
//* Include Files *
//*****************

#include "GlobalDef.hpp"               //* General definitions

#ifndef NCURSES_INCLUDED
#include "NCurses.hpp"
#endif

#ifndef NCDIALOG_INCLUDED
#include "NcDialog.hpp"
#endif

//*********************
//* Local Definitions *
//*********************


//***************************
//*  ChangedFocusViaHotkey  *
//***************************
//******************************************************************************
//* While in an editing method for one of the control types, the user has      *
//* pressed a valid hotkey signalling that he/she wants to change the input    *
//* focus to the control associated with that hotkey.                          *
//*                                                                            *
//* This method IS NOT called when the hotkey pressed is associated with the   *
//* control currently being edited. The control that gains focus here, must    *
//* be a DIFFERENT control than the one previously being edited                *
//*                                                                            *
//* This method exists to ensure that what happens after a hotkey detection    *
//* event, is handled in a consistent manner, regardless of the type of        *
//* control being edited when the event occurred.                              *
//*                                                                            *
//* First action performed:                                                    *
//*   Give input focus to the control associated with the hotkey.              *
//*                                                                            *
//* Processing by Control Type:                                                *
//*  Pushbutton:   Pushbuttons are 'activated' (receive input focus) and       *
//*                are 'selected' (pressed) by hotkey as if an Enter key had   *
//*                been pressed while the control had input focus.             *
//*                                                                            *
//*  Textbox:      Text boxes are 'activated' (receive input focus) via        *
//*                hotkey, but allowing the user to edit the textbox data      *
//*                must be done by the application on return from the          *
//*                current editing call.                                       *
//*                                                                            *
//*  Billboard:    Billboard controls, when _inappropriately_ 'activated',     *
//*                can receive input focus via hotkey. The EditBillboard()     *
//*                stub (if called) can handle this error, but it shouldn't    *
//*                have happened.                                              *
//*                                                                            *
//*  Scrollbox:    Scrollbox controls can be 'activated' (receive input focus) *
//*                via hotkey, but allowing the user to edit the Scrollbox     *
//*                control's data must be done by the application on return    *
//*                from the current editing call.                              *
//*                                                                            *
//*  Radiobutton:  Radio Buttons are 'activated' (receive input focus) and     *
//*                are 'selected' by hotkey as if an Enter key had been        *
//*                pressed while the control had input focus.                  *
//*                - Style dictates that Radio Buttons or radiobutton groups   *
//*                  should lose the input focus immediately after             * 
//*                  selection/deselection, so the application should handle   *
//*                  this on return from current editing call.                 *
//*                                                                            *
//*  Dropdown:     Dropdown controls can be 'activated' (receive input focus)  *
//*                via hotkey, but allowing the user to edit the drop-down     *
//*                control's data must be done by the application on return    *
//*                from the current editing call.                              *
//*                                                                            *
//*  Menuwin:      Menu-win controls can be 'activated' (receive input focus)  *
//*                via hotkey, but allowing the user to edit the control's     *
//*                data must be done by the application on return from the     *
//*                current editing call.                                       *
//*                                                                            *
//*  Scrollext:    Scroll Ext controls can be 'activated' (receive input       *
//*                focus) via hotkey, but allowing the user to edit the        *
//*                scrollext control's data must be done by the application    *
//*                on return from the current editing call.                    *
//*                                                                            *
//*  Spinner:      Spinner controls can be 'activated' (receive input focus)   *
//*                via hotkey, but allowing the user to edit the control's     *
//*                data must be done by the application on return from the     *
//*                current editing call.                                       *
//*                                                                            *
//*  Slider:       Slider controls can be 'activated' (receive input focus)    *
//*                via hotkey, but allowing the user to edit the control's     *
//*                data must be done by the application on return from the     *
//*                current editing call.                                       *
//*                                                                            *
//* Input : keyIn:    keycode of the hotkey pressed                            *
//*         hotIndex: index of control associated with the hotkey              *
//*         info:     user input results returned to application               *
//*                    (by reference)                                          *
//*                   Fields updated by this method:                           *
//*                    Data belonging to the calling method:                   *
//*                     viaHotkey                                              *
//*                     hasFocus                                               *
//*                    Data initialized only by this method:                   *
//*                     h_ctrlType                                             *
//*                     h_ctrlIndex                                            *
//*                     h_hasFocus                                             *
//*                     h_dataMod                                              *
//*                     h_keyIn                                                *
//*                     h_selMember                                            *
//*                     h_isSel                                                *
//*                                                                            *
//* Returns: index of control that now has focus                               *
//******************************************************************************

short NcDialog::ChangedFocusViaHotkey ( wchar_t keyIn, short hotIndex, uiInfo& info )
{
   //* Give focus to the control indicated by the hotkey *
   while ( this->currCtrl > hotIndex )
      this->PrevControl () ;
   while ( this->currCtrl < hotIndex )
      this->NextControl () ;

   //* If the control that now has focus is a Pushbutton,    *
   //* (see notes above)                                     *
   if ( this->dCtrl[this->currCtrl]->type == dctPUSHBUTTON )
   {
      info.h_ctrlType = dctPUSHBUTTON ;      // control type
      info.h_ctrlIndex = this->currCtrl ;    // index of this control
      info.hasFocus = false ;                // former control has lost focus
      info.h_hasFocus = true ;               // this control has input focus
      info.h_isSel = true ;                  // button IS pressed
      info.h_dataMod = true ;                // state change (not previously pressed)
      info.h_keyIn = ZERO ;                  // discard the key input
      info.h_selMember = MAX_DIALOG_CONTROLS ; // don't care
      info.viaHotkey = true ;                // control DID receive focus via hotkey
   }

   //* If the control that now has focus is a Textbox        *
   //* (see notes above)                                     *
   else if ( this->dCtrl[this->currCtrl]->type == dctTEXTBOX )
   {
      info.h_ctrlType = dctTEXTBOX ;         // control type
      info.h_ctrlIndex = this->currCtrl ;    // index of this control
      info.hasFocus = false ;                // former control has lost focus
      info.h_hasFocus = true ;               // this control has input focus
      info.h_dataMod = false ;               // data not modified
      info.h_keyIn = ZERO ;                  // discard the key input
      info.h_isSel = false ;                 // don't care
      info.h_selMember = MAX_DIALOG_CONTROLS ; // don't care
      info.viaHotkey = true ;                // control DID receive focus via hotkey
   }

   //* If the control that now has focus is a Billboard      *
   //* (see notes above)                                     *
   else if ( this->dCtrl[this->currCtrl]->type == dctBILLBOARD )
   {
      info.h_ctrlType = dctBILLBOARD ;       // control type
      info.h_ctrlIndex = this->currCtrl ;    // index of this control
      info.hasFocus = false ;                // former control has lost focus
      info.h_hasFocus = true ;               // this control has input focus
      info.h_dataMod = false ;               // data not modified
      info.h_keyIn = ZERO ;                  // discard the key input
      info.h_isSel = false ;                 // don't care
      info.h_selMember = MAX_DIALOG_CONTROLS ; // don't care
      info.viaHotkey = true ;                // control DID receive focus via hotkey
   }

   //* If the control that now has focus is a Scrollbox      *
   //* (see notes above)                                     *
   else if ( this->dCtrl[this->currCtrl]->type == dctSCROLLBOX )
   {
      info.h_ctrlType = dctSCROLLBOX ;       // control type
      info.h_ctrlIndex = this->currCtrl ;    // index of this control
      info.hasFocus = false ;                // former control has lost focus
      info.h_hasFocus = true ;               // this control has input focus
      info.h_dataMod = false ;               // data not modified
      info.h_keyIn = ZERO ;                  // discard the key input
      info.h_isSel = false ;                 // don't care
      DialogScrollbox* cpt = (DialogScrollbox*)dCtrl[currCtrl] ;
      info.h_selMember = cpt->bIndex ;       // currently 'selected' member
      info.viaHotkey = true ;                // control DID receive focus via hotkey
   }

   //* If the control that now has focus is a Radiobutton    *
   //* (see notes above)                                     *
   else if ( this->dCtrl[this->currCtrl]->type == dctRADIOBUTTON )
   {
      //* If the radio button is a member of a radio button group *
      if ( this->dCtrl[this->currCtrl]->groupCode != ZERO )
      {
         //* Remember originally 'selected' member of the group *
         short origSelMember = GetRbGroupSelection ( this->currCtrl ) ;

         //* Update the group tracking variables for new 'selected' *
         //* member and re-display the Radio Button                 *
         this->RadioXorUpdate ( hotIndex ) ;

         info.h_ctrlType = dctRADIOBUTTON ;  // control type
         info.h_ctrlIndex = this->currCtrl ; // index of last button in group
         info.hasFocus = false ;             // former control has lost focus
         info.h_hasFocus = true ;            // this control group has input focus
         info.h_selMember = hotIndex ;       // index of 'selected' member of group
         info.h_keyIn = nckTAB ;             // focus to move forward out of group
         info.h_isSel = false ;              // don't care
         //* If the new 'selected' member is different *
         //* from previous, indicate the change        *
         info.h_dataMod = ((hotIndex==origSelMember) ? false : true) ;
         info.viaHotkey = true ;             // control DID receive focus via hotkey
      }
      //* Else, is an independent Radio button *
      else
      {
         info.h_ctrlType = dctRADIOBUTTON ;  // control type
         info.h_ctrlIndex = this->currCtrl ; // index of this control
         info.hasFocus = false ;             // former control has lost focus
         info.h_hasFocus = true ;            // this control has input focus
         info.h_selMember = MAX_DIALOG_CONTROLS ; // don't care
         info.h_keyIn = nckTAB ;             // focus to move forward to next control

         DialogRadiobutton* cpt = (DialogRadiobutton*)this->dCtrl[this->currCtrl] ;
         if ( (this->meMouseEnabled ()) && (info.wk.mevent.conv) )
         {  //* Toggle the button state *
            if ( cpt->selected == false )
               info.h_isSel = cpt->selected = true ;
            else
               info.h_isSel = cpt->selected  = false ;
            info.h_dataMod = true ;             // indicate state change
         }
         else
         {
            if ( cpt->selected == false )
               cpt->selected = info.h_dataMod = true ;
            else
               info.h_dataMod = false ;         // no state change
            info.h_isSel = true ;               // button state on return
         }
         info.viaHotkey = true ;             // control DID receive focus via hotkey
         cpt->RedrawControl ( true ) ;       // refresh the control's display
      }
   }

   //* If the control that now has focus is a Dropdown *
   //* (see notes above)                               *
   else if ( this->dCtrl[this->currCtrl]->type == dctDROPDOWN )
   {
      info.h_ctrlType = dctDROPDOWN ;        // control type
      info.h_ctrlIndex = this->currCtrl ;    // index of this control
      info.hasFocus = false ;                // former control has lost focus
      info.h_hasFocus = true ;               // this control has input focus
      info.h_dataMod = false ;               // data not modified
      info.h_keyIn = ZERO ;                  // discard the key input
      info.h_isSel = false ;                 // don't care
      DialogDropdown* cpt = (DialogDropdown*)dCtrl[currCtrl] ;
      info.h_selMember = cpt->bIndex ;       // currently 'selected' member
      info.viaHotkey = true ;                // control DID receive focus via hotkey
   }

   //* If the control that now has focus is a Menu-Win control*
   //* (see notes above)                                      *
   else if ( this->dCtrl[this->currCtrl]->type == dctMENUWIN )
   {
      info.h_ctrlType = dctMENUWIN ;         // control type
      info.h_ctrlIndex = this->currCtrl ;    // index of this control
      info.hasFocus = false ;                // former control has lost focus
      info.h_hasFocus = true ;               // this control has input focus
      info.h_dataMod = false ;               // data not modified
      info.h_keyIn = ZERO ;                  // discard the key input
      info.h_isSel = false ;                 // don't care
      DialogMenuwin* cpt = (DialogMenuwin*)this->dCtrl[this->currCtrl] ;
      info.h_selMember = cpt->bIndex ;       // currently 'selected' member
      info.viaHotkey = true ;                // control DID receive focus via hotkey
   }

   //* If the control that now has focus is a Scroll-Ext     *
   //* (see notes above)                                     *
   else if ( this->dCtrl[this->currCtrl]->type == dctSCROLLEXT )
   {
      info.h_ctrlType = dctSCROLLEXT ;       // control type
      info.h_ctrlIndex = this->currCtrl ;    // index of this control
      info.hasFocus = false ;                // former control has lost focus
      info.h_hasFocus = true ;               // this control has input focus
      info.h_dataMod = false ;               // data not modified
      info.h_keyIn = ZERO ;                  // discard the key input
      info.h_isSel = false ;                 // don't care
      DialogScrollext* cpt = (DialogScrollext*)this->dCtrl[this->currCtrl] ;
      info.h_selMember = cpt->bIndex ;       // currently 'selected' member
      info.viaHotkey = true ;                // control DID receive focus via hotkey
   }

   //* If the control that now has focus is a Spinner        *
   //* (see notes above)                                     *
   else if ( this->dCtrl[this->currCtrl]->type == dctSPINNER )
   {
      info.h_ctrlType = dctSPINNER ;         // control type
      info.h_ctrlIndex = currCtrl ;          // index of this control
      info.hasFocus = false ;                // former control has lost focus
      info.h_hasFocus = true ;               // this control has input focus
      info.h_dataMod = false ;               // data not modified
      info.h_keyIn = ZERO ;                  // discard the key input
      info.h_isSel = false ;                 // don't care
      info.h_selMember = MAX_DIALOG_CONTROLS ; // don't care
      info.viaHotkey = true ;                // control DID receive focus via hotkey
   }

   //* If the control that now has focus is a Slider         *
   //* (see notes above)                                     *
   else if ( this->dCtrl[this->currCtrl]->type == dctSLIDER )
   {
      info.h_ctrlType = dctSLIDER ;          // control type
      info.h_ctrlIndex = currCtrl ;          // index of this control
      info.hasFocus = false ;                // former control has lost focus
      info.h_hasFocus = true ;               // this control has input focus
      info.h_dataMod = false ;               // data not modified
      info.h_keyIn = ZERO ;                  // discard the key input
      info.h_isSel = false ;                 // don't care
      info.h_selMember = MAX_DIALOG_CONTROLS ; // don't care
      info.viaHotkey = true ;                // control DID receive focus via hotkey
   }

   //* Else, unknown control type (this is unlikely) *
   else
   {
      info.keyIn = ZERO ;        // discard the key input that brought us here
      info.viaHotkey = false ;   // no control selected via hotkey (invalidates all h_xx fields)
   }

   return this->currCtrl ;

}  //* End ChangedFocusViaHotkey() *

//*************************
//*     NextControl       *
//*************************
//******************************************************************************
//* Shift focus to next control in list.                                       *
//* Note that only controls that are marked as Active may receive the input    *
//* focus.                                                                     *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: index of control which currently has the input focus              *
//******************************************************************************

short NcDialog::NextControl ( void )
{
   //* If more than one control *
   if ( this->lastCtrl > ZERO )
   {
      //* Remove focus indicator from current control *
      this->RedrawCurrControl ( false ) ;
      
      //* Advance the index to next control in list *
      do
      {
         if ( this->currCtrl < this->lastCtrl )
            ++this->currCtrl ;
         //* Or return to first item in list *
         else
            currCtrl = ZERO ;
      }
      while ( this->dCtrl[this->currCtrl]->active == false ) ;
   
      //* Redraw new control with focus indicator * 
      this->RedrawCurrControl ( true ) ;
   }
   return currCtrl ;

}  //* End NextControl() *
   
//**************************
//*     PrevControl        *
//**************************
//******************************************************************************
//* Shift focus to previous control in list.                                   *
//* Note that only controls that are marked as Active may receive the input    *
//* focus.                                                                     *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: index of control which currently has the input focus              *
//******************************************************************************

short NcDialog::PrevControl ( void )
{
   //* If more than one control *
   if ( this->lastCtrl > ZERO )
   {
      //* Remove focus indicator from current control *
      this->RedrawCurrControl ( false ) ;
      
      //* Decrement the index to previous control in list *
      do
      {
         if ( this->currCtrl > ZERO )
            --this->currCtrl ;
         //* Or index the last item in list *
         else
            this->currCtrl = this->lastCtrl ;
      }
      while ( this->dCtrl[this->currCtrl]->active == false ) ;
   
      //* Redraw new control with focus indicator * 
      this->RedrawCurrControl ( true ) ;
   }
   return this->currCtrl ;

}  //* End PrevControl() *
   
//*************************
//*     GetCurrControl    *
//*************************
//******************************************************************************
//* Return the index of the control that currently has the input focus.        *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: index of control which currently has the input focus              *
//******************************************************************************

short NcDialog::GetCurrControl ( void )
{

   return this->currCtrl ;

}  //* End PrevControl() *
   
//*************************
//*   RedrawCurrControl   *
//*************************
//******************************************************************************
//* Redraw the control indexed by currCtrl.                                    *
//*                                                                            *
//*                                                                            *
//* Input  : focus: if true use control's fColor, else use nColor              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note: It should not be necessary to cast the control pointer  *
//* to the specific control type since all controls are descended from the     *
//* DialogControl base type (although the base type's method is 'pure virtual'.*
//* The fact that it seems necessary is disturbing.                            *
//*   Mahlon Smith 07-July-2006                                                *
//******************************************************************************

void  NcDialog::RedrawCurrControl ( bool focus )
{
   if ( dCtrl[currCtrl]->type == dctPUSHBUTTON )
   {
      DialogPushbutton* cp = (DialogPushbutton*)(dCtrl[currCtrl]) ; 
      cp->RedrawControl ( focus ) ;
   }
   else if ( dCtrl[currCtrl]->type == dctTEXTBOX )
   {
      DialogTextbox* cp = (DialogTextbox*)(dCtrl[currCtrl]) ; 
      cp->RedrawControl ( focus ) ;
   }
   else if ( dCtrl[currCtrl]->type == dctBILLBOARD )
   {
      DialogBillboard* cp = (DialogBillboard*)(dCtrl[currCtrl]) ; 
      cp->RedrawControl ( focus ) ;
   }
   else if ( dCtrl[currCtrl]->type == dctRADIOBUTTON )
   {
      DialogRadiobutton* cp = (DialogRadiobutton*)(dCtrl[currCtrl]) ;
      cp->RedrawControl ( focus ) ;
   }
   else if ( dCtrl[currCtrl]->type == dctSCROLLBOX )
   {
      DialogScrollbox* cp = (DialogScrollbox*)(dCtrl[currCtrl]) ;
      cp->RedrawControl ( focus ) ;
   }
   else if ( dCtrl[currCtrl]->type == dctDROPDOWN )
   {
      DialogDropdown* cp = (DialogDropdown*)(dCtrl[currCtrl]) ;
      cp->RedrawControl ( focus ) ;
   }
   else if ( dCtrl[currCtrl]->type == dctMENUWIN )
   {
      DialogMenuwin* cp = (DialogMenuwin*)(dCtrl[currCtrl]) ;
      cp->RedrawControl ( focus ) ;
   }
   else if ( dCtrl[currCtrl]->type == dctSCROLLEXT )
   {
      DialogScrollext* cp = (DialogScrollext*)(dCtrl[currCtrl]) ;
      cp->RedrawControl ( focus ) ;
   }
   else if ( dCtrl[currCtrl]->type == dctSPINNER )
   {
      DialogSpinner* cp = (DialogSpinner*)(dCtrl[currCtrl]) ;
      cp->RedrawControl ( focus ) ;
   }
   else if ( dCtrl[currCtrl]->type == dctSLIDER )
   {
      DialogSlider* cp = (DialogSlider*)(dCtrl[currCtrl]) ;
      cp->RedrawControl ( focus ) ;
   }

}  //* End RedrawCurrControl() *

//*************************
//* ConnectControl2Border *
//*************************
//******************************************************************************
//* Connect the border of a dialog control object to the dialog window border. *
//*                                                                            *
//* Visual connection does not alter the behavior of the control in any way;   *
//* it is just visually pleasing at times to integrate display of a control    *
//* and the dialog border.                                                     *
//*                                                                            *
//* Note that not all controls have borders AND that not all controls are      *
//* allowed to extend into the parent window's border. Currently, only         *
//* dctSCROLLBOX and dctSCROLLEXT controls may be connected to the parent      *
//* dialog's border.                                                           *
//*                                                                            *
//* Input  : cIndex: index of target control                                   *
//*          connPoint: flags indicating which connections to make             *
//*                                                                            *
//* Returns: OK if successful, or                                              *
//*          ERR if:                                                           *
//*           1. invalid control index                                         *
//*           2. connection is not supported for target control type           *
//******************************************************************************
//* NOTE: The actual connections take place in the control's RedrawControl()   *
//*       method. All we do here is set the appropriate flags.                 *
//*                                                                            *
//* NOTE: we do not validate the control's position within the dialog or the   *
//*       combination of connection points on the assumption that the          *
//*       application knows what it wants. Vain hope, indeed!                  *
//******************************************************************************

short NcDialog::ConnectControl2Border ( short cIndex, cdConnect& connPoint )
{
short    result = ERR ;

   if ( cIndex >= ZERO && cIndex <= this->lastCtrl )
   {
      if ( dCtrl[cIndex]->type == dctSCROLLBOX )
      {
         DialogScrollbox* cp = (DialogScrollbox*)dCtrl[cIndex] ;
         cp->bConnect = connPoint ;
         cp->RedrawControl ( (bool)(cIndex == this->currCtrl) ) ;
         result = OK ;
      }
      else if ( dCtrl[cIndex]->type == dctSCROLLEXT )
      {
         DialogScrollext* cp = (DialogScrollext*)dCtrl[cIndex] ;
         cp->bConnect = connPoint ;
         cp->RedrawControl ( (bool)(cIndex == this->currCtrl) ) ;
         result = OK ;
      }
   }

   return result ;

}  //* End ConnectControl2Border() *

//*************************
//*     ControlActive     *
//*************************
//******************************************************************************
//* Activate or deactivate the specified control.                              *
//*   a. If active, control can receive input focus.                           *
//*   b. If inactive, control cannot receive input focus.                      *
//*                                                                            *
//* On instantiation, each control may be declared Active or Inactive.         *
//* During application execution, however, circumstances may arise where it    *
//* does not make logical sense to allow the user to access a given control.   *
//* Under these circumstances, the control could be deactivated, that is,      *
//* it would still be visible in the dialog, but would not be able to          *
//* obtain the input focus.                                                    *
//*                                                                            *
//* Input  : cIndex   : index of control to activate/deactivate                *
//*          activate : if true, activate the control                          *
//*                     if false, de-activate the control                      *
//*                                                                            *
//* Returns: OK if successful                                                  *
//*          ERR if invalid index OR target control currently has input focus  *
//******************************************************************************

short NcDialog::ControlActive ( short cIndex, bool activate )
{
short    result = OK ;

   if ( cIndex >= ZERO && cIndex <= this->lastCtrl && cIndex != this->currCtrl )
   {
      switch ( dCtrl[cIndex]->type )
      {
         case dctPUSHBUTTON:
            dCtrl[cIndex]->active = activate ;
            break ;
         case dctTEXTBOX:
            dCtrl[cIndex]->active = activate ;
            break ;
         case dctBILLBOARD:
            dCtrl[cIndex]->active = activate ;
            break ;
         case dctRADIOBUTTON:
            dCtrl[cIndex]->active = activate ;
            break ;
         case dctSCROLLBOX:
            dCtrl[cIndex]->active = activate ;
            break ;
         case dctDROPDOWN:
            dCtrl[cIndex]->active = activate ;
            break ;
         case dctMENUWIN:
            dCtrl[cIndex]->active = activate ;
            break ;
         case dctSCROLLEXT:
            dCtrl[cIndex]->active = activate ;
            break ;
         case dctSPINNER:
            dCtrl[cIndex]->active = activate ;
            break ;
         case dctSLIDER:
            dCtrl[cIndex]->active = activate ;
            break ;
         default:                         // invalid type - unlikely
            result = ERR ;
            break ;
      }
   }
   else
      result = ERR ;

   return result ;

}  //* End ControlActive() *

//*************************
//*     GetActiveCtrl    *
//************************
//******************************************************************************
//* Return the next/previous ACTIVE control in the control list.               *
//*                                                                            *
//* Input  : startIndex: index at which to begin search                        *
//*          fwd       : if true, search forward, else backward                *
//*                                                                            *
//* Returns: index of next/previous active control in the control list         *
//******************************************************************************

short NcDialog::GetActiveCtrl ( short startIndex, bool fwd )
{
short    newCtrl ;
   if ( fwd == false )     // search backward through control list
   {
      newCtrl = (startIndex == ZERO ? this->lastCtrl : startIndex - 1 ) ;
      while ( dCtrl[newCtrl]->active == false && newCtrl != startIndex )
      {
         newCtrl = (newCtrl == ZERO ? this->lastCtrl : newCtrl - 1 ) ;
      }
   }
   else                    // search foreward through control list
   {
      newCtrl = (startIndex == this->lastCtrl ? ZERO : startIndex + 1 ) ;
      while ( dCtrl[newCtrl]->active == false && newCtrl != startIndex )
      {
         newCtrl = (newCtrl == this->lastCtrl ? ZERO : newCtrl + 1 ) ;
      }
   }
   return newCtrl ;

}  //* End GetActiveCtrl()

//*************************
//*     Dump_uiInfo       *
//*************************
//******************************************************************************
//* Displays the contents of a uiInfo-class object returned from one of the    *
//* control editing routines.                                                  *
//*                                                                            *
//* Input  : wPos : screen coordinates for display                             *
//*          info : uiInfo-class object                                        *
//*          moose: (optional, 'false' by default)                             *
//*                 if 'true', then display mouse-event data (if any)          *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void  NcDialog::Dump_uiInfo ( winPos wPos, uiInfo info, bool moose )
{
#if ENABLE_DEVELOPMENT_METHODS != 0
const short dROWS = 20, dCOLS = 30 ;
static const char typeStrings[][16] = 
{
   "dctPUSHBUTTON",
   "dctTEXTBOX",
   "dctBILLBOARD",
   "dctRADIOBUTTON",
   "dctSCROLLBOX",
   "dctDROPDOWN",
   "dctMENUWIN",
   "dctSCROLLEXT",
   "dctSPINNER",
   "dctSLIDER",
   "unknown type"
} ;
static const char* metStrings[] =
{  // KEEP SYNCHRONIZED WITH enum meTypes
   "metNONE",
   "metB1_P",
   "metB1_R",
   "metB1_S",
   "metB1_D",
   "metB1_T",
   "metB2_P",
   "metB2_R",
   "metB2_S",
   "metB2_D",
   "metB2_T",
   "metB3_P",
   "metB3_R",
   "metB3_S",
   "metB3_D",
   "metB3_T",
   "metB4_P",
   "metB4_R",
   "metB4_S",
   "metB4_D",
   "metB4_T",
   "metB5_P",
   "metB5_R",
   "metB5_S",
   "metB5_D",
   "metB5_T",
   "metPOS ", 
   "metSW_D",
   "metSW_U",
   "unknown"
} ;
static const char tfStrings[][6] = 
{
   "false",
   "true",
} ;
attr_t dColor = nc.blR ;

   //* Save parent dialog, just in case *
   this->CaptureDialogDisplayData () ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dROWS,          // number of display lines
                       dCOLS,          // number of display columns
                       wPos.ypos,      // Y offset from upper-left of terminal 
                       wPos.xpos,      // X offset from upper-left of terminal
                       " Dump uiInfo ",// dialog title
                       ncltSINGLE,     // border line-style
                       nc.blR,         // border color attribute
                       nc.blR,         // interior color attribute
                       NULL            // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow ()) == OK )
   {
      //* Prevent embarrassing indexing errors *
      if ( info.ctrlType > dctTYPES )
         info.ctrlType = dctTYPES ;
      if ( info.wk.mevent.meType < ZERO || info.wk.mevent.meType >= metTYPES )
         info.wk.mevent.meType = metTYPES ;

      //* Display the data and then wait for a keypress *
      winPos   wp(2, 2) ;
      gString  gs ;

      gs.compose( L"ctrlType  : %s\n", typeStrings[info.ctrlType] ) ;
      wp = dp->WriteParagraph ( wp, gs, dColor ) ;
      gs.compose( L"ctrlIndex : %hd\n", &info.ctrlIndex ) ;
      wp = dp->WriteParagraph ( wp, gs, dColor ) ;

      if ( ! moose )
      {
         gs.compose( L"hasFocus  : %s\n"
                      "dataMod   : %s\n"
                      "keyIn     : 0x%04X\n"
                      "selMember : %hd\n"
                      "isSel     : %s\n"
                      "viaHotkey : %s\n",
                     tfStrings[info.hasFocus],
                     tfStrings[info.dataMod],
                     &info.keyIn,
                     &info.selMember,
                     tfStrings[info.isSel],
                     tfStrings[info.viaHotkey] ) ;
         wp = dp->WriteParagraph ( wp, gs, dColor ) ;
         ++wp.ypos ;
         if ( info.viaHotkey != false )
         {
            gs.compose( L"h_ctrlType : %s\n"
                         "h_ctrlIndex: %hd\n"
                         "h_hasFocus : %s\n"
                         "h_dataMod  : %s\n"
                         "h_keyIn    : 0x%04X\n"
                         "h_selMember: %hd\n"
                         "h_isSel    : %s",
                        typeStrings[info.h_ctrlType],
                        &info.h_ctrlIndex,
                        tfStrings[info.h_hasFocus],
                        tfStrings[info.h_dataMod],
                        &info.h_keyIn,
                        &info.h_selMember,
                        tfStrings[info.h_isSel] ) ;
            wp = dp->WriteParagraph ( wp, gs, dColor ) ;
         }
         else
            dp->WriteString ( wp.ypos++, wp.xpos-1, "(remaining fields undefined)", dColor ) ;
      }
      //* Moose droppings requested *
      else
      {
         gs = L"     Moose Droppings\n" ;
         wp = dp->WriteParagraph ( wp, gs, dColor ) ;

         if ( info.wk.mevent.eventType != ZERO )
         {
            gString gset ;
            gset.compose( L"   %-#b", &info.wk.mevent.eventType ) ;
            gset.insert( L"\n   ", 23 ) ;
            gs.compose( L"eventType (32-bit mask):\n%S\n"
                         "deviceID  : %hd\n"
                         "screen   y:%-3hd x:%-3hd z:%hd\n"
                         "dialog   y:%-3hd x:%-3hd z:%hd\n"
                         "cIndex    : %hd\n"
                         "meType    : %s\n"
                         "cKey      : %s\n"
                         "sKey      : %s\n"
                         "aKey      : %s\n"
                         "mPos      : %s\n"
                         "conv      : %s\n",
                         gset.gstr(),
                         &info.wk.mevent.deviceID,
                         &info.wk.mevent.sypos,
                         &info.wk.mevent.sxpos,
                         &info.wk.mevent.szpos,
                         &info.wk.mevent.ypos,
                         &info.wk.mevent.xpos,
                         &info.wk.mevent.zpos,
                         &info.wk.mevent.cIndex,
                         metStrings[info.wk.mevent.meType],
                         tfStrings[info.wk.mevent.cKey],
                         tfStrings[info.wk.mevent.sKey],
                         tfStrings[info.wk.mevent.aKey],
                         tfStrings[info.wk.mevent.mPos],
                         tfStrings[info.wk.mevent.conv] ) ;
            wp = dp->WriteParagraph ( wp, gs, dColor ) ;
         }
         else
            dp->WriteString ( wp.ypos++, wp.xpos-1, "      (none available)", dColor ) ;
      }

      dp->WriteString ( dROWS-2, dCOLS/2-8, " Press Any Key ", nc.reS, true ) ;

      nckPause() ;
   }
   if ( dp != NULL )                      // close the window
      delete ( dp ) ;
   
   this->RefreshWin () ;                  // restore parent dialog
   
#endif   // ENABLE_DEVELOPMENT_METHODS
}  //* End Dump_uiInfo()

      //*************************************************
      //** THIS SECTION IMPLEMENTS THE DEFAULT METHODS **
      //** FOR THE ABSTRACT CLASS DialogControl.       **
      //*************************************************
//*************************
//*    RefreshControl     *
//*************************
//******************************************************************************
//* Refresh the display of the specified dialog control object.                *
//*                                                                            *
//* DEFAULT METHOD:                                                            *
//* This is the default method for refreshing the display for the control.     *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void DialogControl::RefreshControl ( void )
{

   this->wPtr->RefreshWin () ;   // refresh the underlying NcWindow object

}  //* End RefreshControl() *

//*************************
//*      DrawLabel        *
//*************************
//******************************************************************************
//* Draw the label text for the control.                                       *
//*                                                                            *
//* DEFAULT METHOD:                                                            *
//* This is the default method for drawing the control's label in the parent   *
//* dialog's space.                                                            *
//*                                                                            *
//* Input  : dlgPtr : pointer to parent window                                 *
//*          tColor : text color attribute                                     *
//*          rtlText: if true, draw RTL text, else draw LTR text (see note)    *
//*                                                                            *
//* Returns: none                                                              *
//******************************************************************************
//* Programmer's Note:                                                         *
//* At this point, the DialogControl::bHotkey member has already been          *
//* initialized.                                                               *
//*  a) Label (if any) for each control has been stored.                       *
//*  b) Character for Hotkey (if any) within label has been determined.        *
//*  c) Y/X offset of Hotkey character has been calculated for LTR text.       *
//*     Y offset is ok as-is, but X offset must be adjusted if 'rtlText'       *
//*     is specified.                                                          *
//*                                                                            *
//******************************************************************************

void DialogControl::DrawLabel ( NcDialog* dlgPtr, attr_t tColor, bool rtlText )
{
   //* Display the control's label in parent dialog window *
   if ( *this->wLabel != NULLCHAR )
   {
      dlgPtr->WriteParagraph ( this->labY, this->labX, this->wLabel, 
                               tColor, false, rtlText ) ;
      if ( this->bHotkey.hotkey != false )
      {
         if ( rtlText != false )
         {  //* Adjust 'hotchar' position (see note above) *
            this->bHotkey.xoffset = -(this->bHotkey.xoffset) ;
         }
         dlgPtr->WriteChar ( this->labY + this->bHotkey.yoffset, 
                             this->labX + this->bHotkey.xoffset, 
                             this->bHotkey.hotchar, tColor | ncuATTR ) ;
      }
   }

}  //* End DrawLabel() *

//*************************
//*      IsHotkey         *
//*************************
//******************************************************************************
//* Scan the control array. If a control (or control group) has a hotkey       *
//* defined for it AND the input parameter matches the specified hotkey        *
//* character, returns the index of the associated control.                    *
//*                                                                            *
//* Note: If the hotkey is associated with an inactive control or control      *
//*       group, that control or group must be activated before it can         *
//*       receive the input focus. Thus, a hotkey associated with a control    *
//*       that remains inactive on return will cause IsHotkey() to return a    *
//*       value indicating 'no-match'.                                         *
//*                                                                            *
//* Control Activation:                                                        *
//* -------------------                                                        *
//* If the hotkey is associated with an INACTIVE control, the control may be   *
//* activated, so that the input focus can move to it. The decision whether    *
//* to activate an inactive control is not made here, but rather through a     *
//* call to the appropriate method for that control or control type.           *
//*                                                                            *
//* If the control associated with the hotkey is an inactive dctMENUWIN        *
//* control, call ShowMenuBar() method, so that if the Menuwin control         *
//* or control group associated with that hotkey is currently hidden, it       *
//* will be made visible and active.                                           *
//*                                                                            *
//*                                                                            *
//* Input  : wKey: key input from user                                         *
//*                                                                            *
//* Returns: index of control referenced by hotkey (if any)                    *
//*          else, if no match found, returns -1                               *
//******************************************************************************
//* Programmer's Note:                                                         *
//* We normalize the potential hotkey for easier comparison with hotkeys       *
//* embedded within the controls' label strings. For hotkeys, the lower case   *
//* character, upper case character and Ctrl+character are interpreted as the  *
//* same key. For instance 'a', 'A', and nckC_A (Ctrl key + 'A') are normalized*
//* to uppercase 'A' for comparison test.                                      *
//*                                                                            *
//* Note that the codes for nckTAB (Ctrl+I), nckENTER (Ctrl+J), and Ctrl+M     *
//* (carriage return) ARE NOT normalized for this test because they should not *
//* be used as hotkeys.                                                        *
//*                                                                            *
//* Note also that auxilliary hotkeys are specific keycodes, NOT one of the    *
//* 3-code groups, and must be tested directly against the caller's keycode.   *
//******************************************************************************

short NcDialog::IsHotkey ( wkeyCode wKey )
{
   short cIndex = -1 ;        // return value, initially == 'no match found'

   //* Normalize the keycode for comparison testing (see note above). *
   wchar_t  testKey = nckNULLCHAR ;
   if ( wKey.type == wktPRINT )
   {  //* Uppercase or lowercase alpha *
      if ( wKey.key >= 'A' && wKey.key <= 'Z' )
         testKey = wKey.key ;
      else if ( wKey.key >= 'a' && wKey.key <= 'z' )
         testKey = wKey.key & 0x005F ;
   }
   else if ( wKey.type == wktFUNKEY )
   {  //* Convert ASCII control code to uppercase alpha *
      if ( (wKey.key >= nckC_A && wKey.key <= nckC_H) || 
           (wKey.key == nckC_K || wKey.key == nckC_L) || 
           (wKey.key >= nckC_N && wKey.key <= nckC_Z) )
      {
         testKey = wKey.key + 0x0040 ;
      }
   }
   //* If we have a keycode that was converted from a mouse input *
   else if ( wKey.mevent.conv != false )
      testKey = wKey.key ;
   bool  validHotkeyCode = (testKey >= 'A' && testKey <= 'Z') ? true : false,
         done = (testKey == nckNULLCHAR) ? true : false ;

   //* Scan each control in the control list *
   for ( short tIndex = ZERO ; tIndex <= this->lastCtrl && (! done) ; tIndex++ )
   {
      //* If control is a member of a group, test for auxilliary hotkey *
      if ( (cIndex < ZERO) && (this->dCtrl[tIndex]->groupCode != ZERO) )
      {
         //* If an auxilliary hotkey match found, OR is a standard hotkey      *
         //* associated with a member of a group, then perform any needed      *
         //* processing.                                                       *
         //* If Menubar (group of controls of dctMENUWIN type) *
         if ( this->dCtrl[tIndex]->type == dctMENUWIN )
         {
            DialogMenuwin* cpt = (DialogMenuwin*)this->dCtrl[tIndex] ;
            bool  wasVisible = cpt->grpVisible ;
            if (   (wKey.key == cpt->grpHotkey) 
                || (wasVisible == false && cpt->bHotkey.hotkey && 
                    (testKey == (cpt->bHotkey.hotchar & 0x005F))) )
            {
               if ( (this->ToggleMenuBar ( tIndex )) != false )
               {
                  //* If MenuBar successfully hidden, return to user with      *
                  //* cIndex==(-1) i.e. no additional processing necessary.    *
                  if ( wasVisible )
                     done = true ;
                  //* If MenuBar successfully made visible:                    *
                  //* If selection was through group hotkey, we're done, so    *
                  //* return with cIndex==index of first control in group.     *
                  //* Else let next section select the target index via        *
                  //* normal hotkey.                                           *
                  else
                  {
                     if ( wKey.key == cpt->grpHotkey )
                     {
                        cIndex = tIndex ;
                        done = true ;
                     }
                  }
               }
            }
         }  // dctMENUWIN
         else
         {
            //* At this time, there are no other control *
            //* types whose groups need processing here. *
         }
      }  // if(group member)

      //* If wKey did not correspond to a group auxilliary hotkey *
      //* and control is 'active', test for match with control.   *
      if ( ! done && (this->dCtrl[tIndex]->active != false) )
      {
         //* If a 'standard' hotkey has been specified for control *
         //* AND if testKey is a valid hotkey code                 *
         if ( (this->dCtrl[tIndex]->bHotkey.hotkey != false) && 
              (validHotkeyCode != false) )
         {
            //* Compare the keycodes *
            if ( testKey == (this->dCtrl[tIndex]->bHotkey.hotchar & 0x005F) )
            {
               //* Hotkey specified for this control matches caller's key.*
               //* Return index of the matching control.                  *
               cIndex = tIndex ;
               done = true ;
            }
         }
         //* If mouse support enabled AND a conversion from mouse event *
         //* to keycode occurred AND it matches this control's hotkey   *
         else if ( (this->meMouseEnabled () != false) && 
                   (wKey.mevent.conv != false) && 
                   (wKey.key == this->dCtrl[tIndex]->bHotkey.hotchar) )
         {
            cIndex = tIndex ;
            done = true ;
         }
      }  // (!done && active)
   }     // for(;;)
   return cIndex ;

}  //* End IsHotkey() *

//*************************
//*      InitHotkey       *
//*************************
//******************************************************************************
//* Copy the source string to the target buffer, scanning for a indicator of   *
//* a hotkey.  If a hotkey is found, the control's dcHotkey data member is     *
//* initialized with the specified keycode.                                    *
//*                                                                            *
//* IMPORTANT NOTE: Valid hotkeys are 'A' <= hotkey <= 'Z'  OR                 *
//*                 'a' <= hotkey <= 'z'                                       *
//*                 This restriction is necessary to allow a single, control   *
//*                 key sequence to invoke the hotkey functionality:           *
//*                 Ctrl+A, Ctrl+b, etc.                                       *
//*                                                                            *
//* DEFAULT METHOD:                                                            *
//* This is the default method for Hotkey initialization.                      *
//*                                                                            *
//* Input : src    : pointer to UTF-8-encoded source text                      *
//*         stripNL: (optional, 'true' by default)                             *
//*                  if 'true', ignore newline characters in the label string  *
//*                  if 'false', then interpret the newline character as a     *
//*                              signal to move to the next display line       *
//*                                                                            *
//*                                                                            *
//* Returns: none                                                              *
//******************************************************************************
//* Programmer's Note:                                                         *
//* Hotkey positioning:                                                        *
//* 1) Hotkey characters are always one column wide 'a'-'z' and 'A'-'Z'.       *
//* 2) Hotkey can be located on any line of the label-text output: 'yoffset'   *
//* 3) Offset in X, 'xoffset', must take into accout that characters preceeding*
//*    the hotkey character may be either single- or multi-column characters.  *
//*    a) If hotkey is on first line of text, the calculation is easy:         *
//*       all columns preceeding the hotkey character.                         *
//*    b) If hotkey is on second or subquent line of text, then offset is all  *
//*       columns preceeding the hotkey ON THAT LINE.                          *
//*                                                                            *
//*  (Default (invisible) hotkeys are assigned to controls for which         ) *
//*  (application did not specify a hotkey. This is done at the dialog level.) *
//*                                                                            *
//******************************************************************************

void  DialogControl::InitHotkey ( const char* src, bool stripNL )
{
   this->bHotkey.reset() ;          // assume no hotkey specified

   if ( src != NULL && *src != NULLCHAR )
   {
      //* Convert from UTF-8 characters to gString data *
      wchar_t wdst[gsALLOCDFLT] ;   // temp buffer
      gString  gssrc(src), gsdst ;  // formatted source data and processed text
      gssrc.limitChars( MAX_LABEL_CHARS ) ; // limit text to max label length
      const wchar_t* wsrc = gssrc.gstr() ;
      short i = -1, j = ZERO ;
      do
      {
         ++i ;
         //* If this is a hot-key specifier *
         if ( wsrc[j] == HOTCHAR )
         {
            ++j ;
            if (   (wsrc[j] >= 'A' && wsrc[j] <= 'Z')    // if a valid hotkey
                || (wsrc[j] >= 'a' && wsrc[j] <= 'z') )
            {  //* Record info for hotkey and its x-offset (column) in the string*
               this->bHotkey.hotchar = wsrc[j] ;
               gsdst = gssrc ; // # of columns up-to-and-including hotkey character
               gsdst.limitChars(j) ;      // convert count to offset
               if ( this->bHotkey.yoffset == ZERO )// if hotkey is on first line
                  this->bHotkey.xoffset = gsdst.gscols() - 1 ;
               else
               {  //* Calculate column offset - see note above *
                  int charCount ;
                  const wchar_t* wcharPtr = gsdst.gstr() ;
                  const short* colPtr = gsdst.gscols( charCount ) ;
                  for ( int i = charCount - 3 ; 
                        i >= ZERO && wcharPtr[i] != NEWLINE ; --i )
                  {
                     this->bHotkey.xoffset += colPtr[i] ;
                  }
               }
               this->bHotkey.hotkey = true ;       // hotkey specified
            }
         }
         //* If control does not support multi-line labels, delete newlines.   *
         //* If control does support multi-line labels, move on to next line.  *
         else if ( wsrc[j] == NEWLINE )
         {
            if ( stripNL )
            {
               ++j ;
               --i ;
               continue ;
            }
            else if ( ! this->bHotkey.hotkey )
               ++this->bHotkey.yoffset ;
         }
         wdst[i] = wsrc[j++] ;
      }
      while ( wdst[i] != NULLCHAR ) ;

      //* Store the processed text data *
      wcsncpy ( this->wLabel, wdst, MAX_LABEL_CHARS ) ;
   }
   else
   {
      *this->wLabel = NULLCHAR ;             // no label specified
   }

}  //* InitHotkey() *

//************************
//*                      *
//************************
//********************************************************************************
//*                                                                              *
//*                                                                              *
//*                                                                              *
//* Input  :                                                                     *
//*                                                                              *
//* Returns:                                                                     *
//********************************************************************************


