//********************************************************************************
//* File       : NcdControlMW.cpp                                                *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2006-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in NcDialog.hpp            *
//* Date       : 21-Mar-2025                                                     *
//* Version    : (see NcDialogVersion string in NcDialog.cpp)                    *
//*                                                                              *
//* Description: Contains the methods of the DialogMenuwin class                 *
//*              and the public NcDialog-class methods for accessing             *
//*              the DialogMenuwin object's functionality.                       *
//*              See NcDialog.hpp for the definition of this class.              *
//*                                                                              *
//*                                                                              *
//* Development Tools: See NcDialog.cpp.                                         *
//********************************************************************************
//* Version History (most recent first):                                         *
//*   See version history in NcDialog.cpp.                                       *
//********************************************************************************
//* Programmer's Notes:                                                          *
//*                                                                              *
//*                                                                              *
//*                                                                              *
//*******************************************************************************

//*****************
//* Include Files *
//*****************

#include "GlobalDef.hpp"               //* General definitions

#ifndef NCURSES_INCLUDED
#include "NCurses.hpp"
#endif

#ifndef NCDIALOG_INCLUDED
#include "NcDialog.hpp"
#endif

//**************
//* Local Data *
//**************


      //*************************************************
      //** THIS SECTION IMPLEMENTS THE METHODS OF THE  **
      //**            DialogMenuwin CLASS.             **
      //** - - - - - - - - - - - - - - - - - - - - - - **
      //** See below for public NcDialog-class methods **
      //**   related to DialogMenuwin-class access.    **
      //*************************************************

//*************************
//*    DialogMenuwin      *
//*************************
//******************************************************************************
//* Constructor for DialogMenuwin class object.                                *
//*                                                                            *
//*                                                                            *
//* Input  : iPtr: pointer to initialization structure                         *
//*          dUY : screen offset in Y for first line of parent dialog          *
//*          dLI : number of lines in parent dialog                            *
//*                                                                            *
//* Returns: constructors implicitly return a pointer to the object            *
//******************************************************************************
//* Programmer's Note: The display data is inherently multi-color data, even   *
//* if every line is the same color. In other words, there is a color          *
//* attribute for each menu item.                                              *
//*                                                                            *
//* Note that we engage in some defensive programming when formatting the      *
//* display data for local storage. We believe that there is some danger of    *
//* difficult-to-find pointer errors if we trust the application to pass good  *
//* data; therefore we carefully scan the display strings - verifying each     *
//* menu item.                                                                 *
//*                                                                            *
//* The source text is a two-dimentional array of const char strings; however, *
//* there is no guarantee that all the strings contain the same number of      *
//* bytes, nor the same number of characters, nor that they all require the    *
//* same number of display columns. We must normalize these things locally.    *
//*                                                                            *
//* Note that stored text data will include the caret '^' hotkey indicator     *
//* (if any). Short strings will be lengthened to exactly fit the menu width,  *
//* and long strings will produce an ugly display, BUT will not give us a      *
//* segmentation fault, no matter how inept the application programmer is --   *
//* unless we are tricked into looking beyond the data space, in which case    *
//* may a large rodent crawl into his/her/its smallest orifice....             *
//******************************************************************************

DialogMenuwin::DialogMenuwin ( InitCtrl* iPtr, short dUY, short dLI )
{
   //* Transfer caller's initialization data *
   this->type   = dctMENUWIN ;      // set control type
   this->ulY    = iPtr->ulY ;       // control's position in dialog
   this->ulX    = iPtr->ulX ;
   this->bItems = iPtr->scrItems ;  // number of display strings in menu
   this->lines  = this->bItems + 2 ;// all menu items + top and bottom border
   this->cols   = iPtr->cols ;      // width of menu including left and right border
   this->nColor = iPtr->nColor ;    // ? color of border when control does not have focus
   this->fColor = iPtr->fColor ;    // ? color of border when control has focus
   this->nBorder = iPtr->nColor ;   // color of border when control does not have focus
   this->fBorder = iPtr->fColor ;   // color of border when control has focus
   this->bIndex = ZERO ;            // always begin with highlight on first item in menu
   this->active = iPtr->active ;    // indicates whether control can be selected (but see below)
   this->groupCode = ZERO ;         // initially not a member of a group

   //* Default values for all other data members of the class *
   this->expanded = false ;         // control is initially collapsed
   this->grpVisible = true ;        // initially all group members are visible
   this->grpHotkey = ZERO ;         // initially no group hotkey defined
   this->tLines = 1 ;               // only 1 display line in 'collapsed' form
   this->hideItems = false ;        // all menu items initially visible
   //* The number of columns required to display the label determines the width*
   //* of the 'collapsed' menu. Therefore, we count the display columns        *
   //* required. The count DOES NOT include a column for the caret '^' which is*
   //* the hotkey indicator, but is not displayed as part of the string.       *
   //* Test for presence of a hotkey is done by InitHotkey(), below.           *
   if ( iPtr->label != NULL )
   {
      gString gs(iPtr->label) ;
      this->tCols = gs.gscols() ;
   }
   else
      this->tCols = ZERO ;
   this->tulY   = this->ulY ;       // position of 'collapsed' control
   this->tulX   = this->ulX ;
   if ( this->tCols > ZERO )        // if control has a label, adjust menu position
      ++this->ulY ;
   this->tcAttr = this->fColor ;    // default non-focus color for 'collapsed' control
   this->typos  = ZERO ;            // Y offset for 'collapsed' control string display
   this->txpos  = ZERO ;            // X offset for 'collapsed' control string display
   this->labY = this->labX = ZERO ; // label offsets not used for this control
   this->rtlContent = false ;       // left-to-right language content by default

   //* Copy the label text (display string for collapsed control) *
   //* to data member, wLabel and initialize bHotkey.             *
   this->InitHotkey ( iPtr->label ) ;
   if ( this->bHotkey.hotkey != false )   // if hotkey specified, shrink 'collapsed' window
      --this->tCols ;                     // don't include hotkey indicator in column count

   //* For a control that is initially invisible, i.e. the width of the        *
   //* 'collapsed' control is ZERO, it must also initially be non-selectable   *
   //* via user's key input. Context menus and sub-menus are in this category. *
   if ( this->tCols == ZERO )
      this->active = false ;

   //* Allocate space for strings, pointers, color attributes, and sub-menu    *
   //* handles. This is a single memory allocation, divided into sections.     *
   //* Note that our initial allocation is likely larger than we actually need,*
   //* so we give back the excess, below.                                      *
   void* blkptr ;                   // pointer to memory-allocation block
   char* dString ;                  // pointer to display text target buffers
   int bytesNeeded = this->bItems * (this->cols) * 4 ; // enough for 4 bytes per character
   bytesNeeded    += this->bItems * sizeof(char*) ;   // space for string pointers
   bytesNeeded    += this->bItems * sizeof(attr_t) ;  // space for color attributes
   bytesNeeded    += this->bItems * sizeof(bool) ;    // space for active-item flags
   bytesNeeded    += this->bItems * sizeof(short) ;   // space for sub-menu handles
   if ( (blkptr = calloc ( bytesNeeded, 1 )) != NULL )// allocate a memory block
   {  //* Attach the display data memory to our class data members *
      //* Pointer to array of menu-item pointers *
      this->bText   = (char**)blkptr ;
      //* Point to array of color attributes *
      this->bColor  = (attr_t*)((char*)this->bText + (this->bItems*sizeof(char*))) ;
      //* Point to array of active-item flags *
      this->bActive = (bool*)((char*)this->bColor + (this->bItems*sizeof(attr_t))) ;
      //* Point to array of sub-menu handles (no sub-menus yet defined) *
      this->subMenu = (short*)((char*)this->bActive + (this->bItems*sizeof(bool))) ;
      //* Point to array of display-item data strings *
      dString       = (char*)((char*)this->subMenu + (this->bItems*sizeof(short))) ;

      //* Copy display data and color attributes to local storage. Normalizing *
      //* the number of display columns required by each menu item.            *
      const char* cPtr = iPtr->dispText ;    // pointer to source text
      short reqCols  = this->cols - 2,       // display columns available for menu item
            ubytes ;                         // for item-width calculations
      gString gs ;                           // text formatting

      for ( short iCount = ZERO ; iCount < this->bItems ; iCount++ ) // for each menu item
      {
         this->bText[iCount] = dString ; // initialize pointer to formatted display string

         //* Scan for a hotkey indicator embedded in the string *
         short isHK = ZERO ;        // == 1 if string includes embedded hotkey
         for ( short j = ZERO ; cPtr[j] != NULLCHAR ; j++ )
         {
            if ( cPtr[j] == HOTCHAR 
                 && ((cPtr[j+1] >= 'A' && cPtr[j+1] <= 'Z') ||
                     (cPtr[j+1] >= 'a' && cPtr[j+1] <= 'z')) )
            {
               isHK = 1 ;
               break ;
            }
         }
         //* Copy the display item to a formatting buffer.                     *
         //* Note that string length is limited to number of display columns.  *
         gs = cPtr ;

         //* If caller has correctly formatted the display item, just copy it. *
         if ( gs.gscols() == (reqCols + isHK) )
         {
            ubytes = gs.utfbytes () ;
            gs.copy( this->bText[iCount], ubytes ) ;
            cPtr += ubytes ;                 // point to next source string
            if ( iCount < (this->bItems -1) )// step over NULLCHAR filler
            { while ( *cPtr == NULLCHAR ) ++cPtr ; }
            dString += ubytes ;              // point to next target buffer
         }
         //* If a display item is too short, copy what we have and pad on right*
         else if ( gs.gscols() < (reqCols + isHK) )
         {
            ubytes = gs.utfbytes () ;
            while ( gs.gscols() < (reqCols + isHK) )
               gs.append( L' ' ) ;
            gs.copy( this->bText[iCount], gs.utfbytes() ) ;
            cPtr += ubytes ;                 // point to next source string
            if ( iCount < (this->bItems -1) )// step over NULLCHAR filler
            { while ( *cPtr == NULLCHAR ) ++cPtr ; }
            dString += gs.utfbytes() ;       // point to next target buffer
         }
         //* Source string is too long - truncate it to fit control window.    *
         else  // gs.gscols() > reqCols
         {
            gs.limitCols( reqCols + isHK ) ; // limit string to available width
            ubytes = gs.utfbytes () ;
            gs.copy( this->bText[iCount], ubytes ) ;
            cPtr += ubytes - 1 ;             // point to next source string
            if ( iCount < (this->bItems -1) )// step over NULLCHAR filler
            { while ( *cPtr == NULLCHAR ) ++cPtr ; }
            dString += ubytes ;              // point to next target buffer
         }

         //* Establish color of display string *
         if ( iPtr->scrColor[0] != attrDFLT )// each item has its own color
            this->bColor[iCount] = iPtr->scrColor[iCount] ;
         else                                // all items are the same color
            this->bColor[iCount] = iPtr->scrColor[1] ;

         //* Initialize the sub-menu handle (no-sub-menus yet established) *
         this->subMenu[iCount] = MAX_DIALOG_CONTROLS ;

         //* Initialize the active-menu-items flags. Initially, all are active.*
         this->bActive[iCount] = true ;

      }  // for(;;)

      //* Return unneeded portion of memory allocation to the heap.            *
      // Programmer's Note: realloc() MAY copy the block to a new location,    *
      // even when shrinking block, necessitating adjustement of our pointers. *
      void* newblkPtr = realloc ( blkptr, (dString - (char*)blkptr) ) ;
      if ( newblkPtr != blkptr )
      {
         this->bText   = (char**)newblkPtr ;
         this->bColor  = (attr_t*)((char*)this->bText + (this->bItems*sizeof(char*))) ;
         this->bActive = (bool*)((char*)this->bColor + (this->bItems*sizeof(attr_t))) ;
         this->subMenu = (short*)((char*)this->bActive + (this->bItems*sizeof(bool))) ;
         dString       = (char*)((char*)this->subMenu + (this->bItems*sizeof(short))) ;
         for ( short iCount = ZERO ; iCount < this->bItems ; iCount++ ) // for each menu item
         {
            this->bText[iCount] = dString ;
            while ( *dString != NULLCHAR )
               ++dString ;
            ++dString ;
         }
      }
   }  // calloc()

   //* Memory allocaton error will be reported in control window *
   else
   {
      //* Set all variables to 'safe' values *
      // (of course, nothing is 'safe' after a memory allocation error)
      bPtr     = NULL ;
      wPtr     = NULL ;
      tPtr     = NULL ;
      bText    = NULL ;
      bColor   = NULL ;
      bIndex   = -1 ;
      bItems   = -1 ;
      *wLabel  = NULLCHAR ;
   }
   
   //*********************************************
   //* Instantiate the underlying window objects.*
   //* No display data are drawn to screen here; *
   //* see DialogMenuwin::OpenControl()          *
   //*********************************************
   // 'Collapsed' control's window contains only a copy of the 'selected' item
   this->tPtr = new NcWindow ( this->tLines, this->tCols, this->tulY, this->tulX ) ;
   // 'Expanded' control's outer window contains border
   this->bPtr = new NcWindow ( this->lines, this->cols, this->ulY, this->ulX ) ;
   // 'Expanded' control's inner window contains scrolling data
   this->wPtr = new NcWindow ( this->lines-2, this->cols-2, this->ulY+1, this->ulX+1 ) ;

}  //* End DialogMenuwin() *

//*************************
//*    ~DialogMenuwin     *
//*************************
//******************************************************************************
//* Destructor for DialogDropdown class object.                                *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

DialogMenuwin::~DialogMenuwin ( void )
{
   //* Release dynamic allocation for display data *
   free ( (void*)this->bText ) ;

   //* Close the underlying windows *
   delete this->bPtr ;
   delete this->wPtr ;
   delete this->tPtr ;

}  //* End ~DialogMenuwin() *

//*************************
//*     OpenControl       *
//*************************
//******************************************************************************
//* Draw a dialog menu-win control i.e. open the window previously             *
//* instantiated by a call to DialogMenuwin().                                 *
//*                                                                            *
//* Input : hasFocus : optional boolean, false by default, determines          *
//*                    whether to draw the object in nColor (does              *
//*                    not have focus) or fColor (has focus)                   *
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
//******************************************************************************

short DialogMenuwin::OpenControl ( bool hasFocus )
{
   attr_t   color = hasFocus ? this->fColor : this->nColor ;
   short    result = ERR ;

   //* Open the control in 'collapsed' form, that  *
   //* is,with the scrolling window invisible.     *
   if (   (tPtr->OpenWindow () == OK)
       && (bPtr->OpenWindow () == OK) 
       && (result = wPtr->OpenWindow ()) == OK )
   {
      //* Hurray! all component windows of the control are open!
      
      //* Set interior color of the scrolling-data window.   *
      //* This is done in case the scrolling data do not fill*
      //* all the rows of the scroll window                  *
      wPtr->SetInteriorColor ( color ) ;

      //* The call to RedrawControl makes the control visible.*
      this->RedrawControl ( hasFocus ) ;
   }
   return result ;

}  //* End OpenControl() *

//*************************
//*     RedrawControl     *
//*************************
//******************************************************************************
//* Redraw the data in a Menuwin window.                                       *
//* (refreshes the control's window)                                           *
//*                                                                            *
//* There are two modes: 'collapsed' mode and 'expanded mode. The mode in      *
//* which the control is drawn depends on the state of the 'expanded' flag.    *
//* Also to be considered are whether the menu is currently 'hidden'.          *
//*                                                                            *
//* Input  : hasFocus:  if true, control has input focus                       *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void DialogMenuwin::RedrawControl ( bool hasFocus )
{

   this->RedrawControl ( hasFocus, true, true ) ;

}  //* End RedrawControl()

//*************************
//*     RedrawControl     *
//*************************
//******************************************************************************
//* Redraw the data in a Menuwin window.                                       *
//* This is the private method, called only from within the Menuwin edit       *
//* routines. It allows us to control whether the highlight is initialized     *
//* and/or visible. This is important when the parent menu is being redrawn    *
//* in preparation for opening a sub-menu from within the parent.              *
//*                                                                            *
//* Input  : hasFocus   : if true, control has input focus                     *
//*          initHilite : if 'true', then set highlight on first item          *
//*                       if 'false', then do not move highlight               *
//*                         (see note below)                                   *
//*          refresh : (optional, 'true' by default) refresh display           *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Notes:                                                                     *
//* 1. There are different kinds of menus:                                     *
//*    a. Primary menus are visible in both the 'collapsed' state and in the   *
//*       'expanded' state. When collapsed, only the 'collapsed' portion of    *
//*       the menu is visible i.e. its title. When in the expanded state, both *
//*       the 'collapsed' and 'expanded' portions of the menu are visible with *
//*       the color of each portion depending on input focus.                  *
//*                                                                            *
//*    b. context menus, including sub-menus which are visible under only two  *
//*       sets of circumstances: when they have input focus, or when a         *
//*       sub-menu attached to it has input focus. Context menus are invisible *
//*       and inactive at all other times.                                     *
//*                                                                            *
//* 2. Parameter interpretation:                                               *
//*     hasFocus &&  initHlite   initialize and display highlight              *
//*    !hasFocus &&  initHlite   (unnecessary, never happens)                  *
//*     hasFocus && !initHlite   display highlight at current position         *
//*    !hasFocus && !initHlite   do not initialize and do not display highlight*
//*                                                                            *
//******************************************************************************

void DialogMenuwin::RedrawControl ( bool hasFocus, bool initHilite, bool refresh )
{
   //* If control is currently expanded for editing *
   if ( this->expanded != false )
   {
      //* NOTE: We arrive here ONLY via a call from inside the EditMenuwin() 
      //* group of methods.
      //* Caller has marked the parent dialog window as 'obscured' i.e. saved 
      //* the display data for the entire dialog; thus, we can overwrite 
      //* whatever we need to when displaying the expanded control window.
      attr_t borderColor = hasFocus ? this->fBorder : this->nBorder ;
      this->bPtr->DrawBorder ( borderColor ) ;// draw the outer border in appropriate color
      this->bPtr->RefreshWin () ;         // display the outer window
      if ( this->bText != NULL )          // are we pointing at valid scroll data?
      {
         if ( initHilite != false )
         {
            this->bIndex = ZERO ;
            while ( this->bActive[this->bIndex] == false && 
                    this->bIndex < (this->bItems - 1) )
               ++this->bIndex ;
         }
         else if ( hasFocus != false )
            initHilite = true ;  // display highlight at current position
         this->wPtr->PaintMenuData ( (const char**)bText, this->bColor, this->bItems, 
                                     this->bIndex, initHilite, this->rtlContent ) ;
      }
      this->wPtr->RefreshWin () ;         // refresh inner window display

      //* If control is also visible when in the collapsed state, redraw the   *
      //* title (collapsed portion) in complementary color.                    *
      if ( this->tCols > ZERO )
      {
         short x = this->rtlContent ? (this->tCols - 1) : this->txpos ;
         attr_t collColor = this->nColor ;
         this->tPtr->WriteString ( this->typos, x,
                         this->wLabel, collColor, false, this->rtlContent ) ;
         if ( this->bHotkey.hotkey != false )
         {
            short xpos = this->rtlContent ? 
                         (this->tCols - this->bHotkey.xoffset - 1) : 
                         this->bHotkey.xoffset ;
            this->tPtr->WriteChar ( this->typos, xpos, 
                         this->bHotkey.hotchar, collColor | ncuATTR ) ;
         }
         this->tPtr->RefreshWin () ;
      }
   }

   //* Else control is in the 'collapsed' state *
   else
   {
      //* If control is visible when in collapsed state *
      if ( this->tCols > ZERO )
      {
         //* If control is not a group member OR is a group member and group   *
         //* is currently visible                                              *
         if (   (this->groupCode == ZERO) 
             || (this->groupCode > ZERO && this->grpVisible != false) )
         {
            //* Determine start position LTR or RTL *
            short x = this->rtlContent ? (this->tCols - 1) : this->txpos ;
            //* The has-focus color is opposite that of the expanded control   *
            attr_t collColor = hasFocus ? this->nColor : this->fColor ;

            this->tPtr->WriteString ( this->typos, x,
                            this->wLabel, collColor, false, this->rtlContent ) ;
            if ( this->bHotkey.hotkey != false )
            {
               short xpos = this->rtlContent ? 
                            (this->tCols - this->bHotkey.xoffset - 1) : 
                            this->bHotkey.xoffset ;
               this->tPtr->WriteChar ( this->typos, xpos, 
                            this->bHotkey.hotchar, collColor | ncuATTR ) ;
            }
            this->tPtr->RefreshWin () ;
         }
         else
         {
            //* Control is member of a currently-invisible group, and as we    *
            //* know, drawing pictures of invisible objects is of limited use. *
         }
      }
   }

}  //* End RedrawControl() *

//*************************
//*    RefreshControl     *
//*************************
//******************************************************************************
//* Refresh the display of the specified dialog control object.                *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note: See discussion in RedrawControl() above.                *
//******************************************************************************

void DialogMenuwin::RefreshControl ( void )
{
   //* Refresh an 'expanded' (open) menu *
   if ( this->expanded != false )
   {
      this->bPtr->RefreshWin () ;         // refresh expanded window
      this->wPtr->RefreshWin () ;         // refresh the inner (data) window
      if ( this->tCols > ZERO )           // if also visible when collapsed
         this->tPtr->RefreshWin () ;      // then refresh the title
   }

   //* Refresh a 'collapsed' (closed) menu UNLESS:                    *
   //*  1. control is invisible when collapsed                        *
   //*  2. control is a member of a MenuBar which is currently hidden *
   else if ( (this->tCols > ZERO) && 
             ((this->groupCode == ZERO)
             ||
             (this->groupCode != ZERO && this->grpVisible != false)) )
   {
      this->tPtr->RefreshWin () ;      // refresh collapsed window
   }
   
}  //* End RefreshControl() *

//*************************
//*    SetOutputFormat    *
//*************************
//******************************************************************************
//* Set the text data output format: RTL (right-to-left), or                   *
//* LTR (left-to-right). Does not refresh display.                             *
//* Note that the default (unmodified) output format is LTR.                   *
//*                                                                            *
//* Input  : rtlFormat : if 'true',  then set as RTL format                    *
//*                      if 'false', then set as LTR format                    *
//*                                                                            *
//* Returns: OK  if successful                                                 *
//*          ERR if data format cannot be changed                              *
//*              a) if control is currently in 'expanded' state                *
//******************************************************************************
//* Programmer's Note: When redrawing the control's contents, we assume that   *
//* control does not have focus. This could be incorrect, but since we don't   *
//* update the display, the user will never know it.                           *
//******************************************************************************

short DialogMenuwin::SetOutputFormat ( bool rtlFormat )
{
   short status = ERR ;
   if ( this->expanded == false )
   {
      this->rtlContent = rtlFormat ;
      this->RedrawControl ( false, false, false ) ;
      status = OK ;
   }
   return status ;

}  //* End SetOutputFormat() *

//************************
//*   ScrollMenuData     *
//************************
//******************************************************************************
//* Scroll through the menu data. Highlight the menu item indicated by the     *
//* scroll key OR the nearest 'active' menu item                               *
//*                                                                            *
//* Input  : key input                                                         *
//*                                                                            *
//* Returns: index of currently highlighted data item                          *
//*          (if highlight moved, window is refreshed before return)           *
//******************************************************************************

short DialogMenuwin::ScrollMenuData ( wkeyCode& wk )
{
   //* Scroll to the item indicated by the scroll key *
   this->bIndex = this->wPtr->ScrollMenuData ( wk.key ) ;

   //* If the highlighted item is not 'active', find a nearby one that is *
   if ( ! this->bActive[this->bIndex] )
   {  //* If highlight is on first item or last item, then wrap around *
      if ( wk.key == nckUP && this->bIndex == ZERO )
         this->bIndex = this->wPtr->ScrollMenuData ( nckEND ) ;
      else if ( wk.key == nckDOWN && this->bIndex == (this->bItems - 1) )
         this->bIndex = this->wPtr->ScrollMenuData ( nckHOME ) ;

      //* Highlight the nearest ACTIVE item *
      if ( wk.key == nckDOWN || wk.key == nckPGDOWN || wk.key == nckHOME )
      {
         while ( (this->bIndex < (this->bItems-1)) && (this->bActive[this->bIndex] == false) )
            this->bIndex = this->wPtr->ScrollMenuData ( nckDOWN ) ;
         if ( ! this->bActive[this->bIndex] )
            while ( (this->bIndex > ZERO) && (this->bActive[this->bIndex] == false) )
               this->bIndex = this->wPtr->ScrollMenuData ( nckUP ) ;
      }
      else if ( wk.key == nckUP || wk.key == nckPGUP || wk.key == nckEND )
      {
         while ( (this->bIndex > ZERO) && (this->bActive[this->bIndex] == false) )
            this->bIndex = this->wPtr->ScrollMenuData ( nckUP ) ;
         if ( ! this->bActive[this->bIndex] )
            while ( (this->bIndex < (this->bItems-1)) && (this->bActive[this->bIndex] == false) )
               this->bIndex = this->wPtr->ScrollMenuData ( nckDOWN ) ;
      }
   }
   return this->bIndex ;

}  //* End ScrollMenuData() *



      //*************************************************
      //**     THIS SECTION IMPLEMENTS THE PUBLIC,     **
      //**    NcDialog-class METHODS FOR ACCESSING     **
      //**    DialogMenuwin-class FUNCTIONALITY.       **
      //*************************************************

//************************
//*    EditMenuwin       *
//************************
//******************************************************************************
//* If control with input focus is of type dctMENUWIN, call this method to get *
//* user's key input.                                                          *
//*                                                                            *
//* - For a menu which is a member of a MenuBar (group of DialogMenuwin        *
//*   controls): returns when the user has selected an item from any member    *
//*   control in the MenuBar OR when user has exited the MenuBar without       *
//*   making a selection.                                                      *
//*                                                                            *
//* - For a menu which is not a member of a MenuBar:                           *
//*   returns when the user has selected an item from the menu OR when user    *
//*   has exited the menu without making a selection.                          *
//*                                                                            *
//* Input  : info: uiInfo class (by reference) - initial values ignored        *
//*                EXCEPT: If info.viaHotkey != false, expand the control      *
//*                        immediately on entry.                               *
//*                                                                            *
//* Returns: index of control that currently has the input focus               *
//*           The uiInfo-class data members have been initialized.             *
//*           (See note in NcDialog.hpp about interpretation )                 *
//*           (of values returned in the uiInfo-class object.)                 *
//******************************************************************************
//* IMPORTANT NOTE: Unlike other controls, a Menuwin control may extend beyond *
//*   the borders of the dialog when in the 'expanded' state; however, when    *
//*   the menu returns to the 'collapsed' state, it is the application's       *
//*   responsibility to refresh the display of any data outside the dialog     *
//*   window that may have been obscured by the expanded menu.                 *
//******************************************************************************

short NcDialog::EditMenuwin ( uiInfo& info )
{
   //* Be sure the control with focus is actually a DialogMenuwin control *
   if ( this->dCtrl[this->currCtrl]->type == dctMENUWIN )
   {
      DialogMenuwin* cp ;              // pointer to control with input focus
      bool  done = false ;             // loop control

      //* Menu is initially in 'collapsed' state (only menu title is visible). *
      //* If caller has not requested that the menu with focus be opened       *
      //* immediately, determine whether user wants to 'expand' (open) the     *
      //* menu for editing.                                                    *
      if ( info.viaHotkey == false )
      {
         done = this->emwEditCollapsed ( info ) ;
      }

      while ( ! done )
      {
         //* Get access to the control's methods and data members *
         cp = (DialogMenuwin*)this->dCtrl[this->currCtrl] ;

         //* Save dialog window's display data because *
         //* we are about to overwrite some of it.     *
         this->CaptureDialogDisplayData () ;

         //* Expand the menu for editing *
         cp->bIndex = ZERO ;        // Start with highlight on first menu item
         cp->expanded = true ;
         cp->RedrawControl ( true ) ;

         //* Edit the open menu and any of its sub-menus *
         // Note: Cascaded open menus must be refreshed in ascending order 
         //       according to when each was opened, thus we send an array 
         //       of pointers to the list of open menu controls.
         short refList[MAX_DIALOG_CONTROLS + 2] = { 1 } ;
         this->emwEditMultilevel ( refList, info ) ;

         //* Collapse the menu and restore parent dialog's display *
         cp->expanded = false ;
         this->RefreshWin () ;

         //* If user has not made a selection from the menu *
         if ( info.dataMod == false )
         {
            //* If menu is a member of a MenuBar, determine *
            //* whether we're finished with MenuBar edit.   *
            if ( cp->groupCode > ZERO )
            {
               //* nckESC indicates a move out of the group *
               if ( info.keyIn == nckESC )
               {  //* Move focus to last group member *
                  this->emwFocus2GroupEnd ( true ) ;
                  done = true ;
               }
               else
               {
                  short targetControl ;
                  bool  forward = true ;
                  if ( info.wk.key == nckSTAB || info.wk.key == nckLEFT )
                  {
                     targetControl = this->currCtrl == ZERO ? 
                                     this->lastCtrl : (this->currCtrl - 1) ;
                     forward = false ;
                  }
                  else  // (nckTAB || nckRIGHT)
                  {
                     targetControl = this->currCtrl == this->lastCtrl ? 
                                     ZERO : (this->currCtrl + 1) ;
                  }
                  if ( this->dCtrl[targetControl]->groupCode == cp->groupCode )
                  {
                     if ( forward )
                        this->NextControl () ;
                     else
                        this->PrevControl () ;
                  }
                  else     // moving out of the group, so we're done
                     done = true ;
               }
            }
            //* This is an independent menu, and focus is shifting to another  *
            //* control, so we're done.                                        *
            else if ( info.wk.key == nckTAB  || info.wk.key == nckSTAB  || 
                      info.wk.key == nckLEFT || info.wk.key == nckRIGHT || 
                      info.wk.key == nckESC )
            {
               done = true ;
            }
            else
            { /* This would be a logical error, but we don't make misteaks */ }
         }
         else              // menu selection made, return to application
         {  //* If a group member, move to last control in the group *
            if ( cp->groupCode > ZERO )
               this->emwFocus2GroupEnd ( true ) ;
            done = true ;
         }
      }
   }
   else            // control is not a menu-win control i.e. application error
   {
      //* Do what we can to minimize the damage *
      info.ctrlType = this->dCtrl[this->currCtrl]->type ;
      info.ctrlIndex = this->currCtrl ;
      info.keyIn = nckTAB ;               // move on to next control
      info.hasFocus = true ;              // control has focus
      info.dataMod = false ;              // no data modified
      info.selMember = MAX_DIALOG_CONTROLS ; // don't care
      info.isSel = false ;                // don't care
      info.viaHotkey = false ;            // invalidate any hotkey data
   }
   return this->currCtrl ;

}  //* End EditMenuwin() *

//************************
//*     EditMenuwin      *
//************************
//******************************************************************************
//* For a context menu (control of type dctMENUWIN which is inactive and       *
//* invisible), call this method to:                                           *
//*  1. activate the menu (make it accessible to user input)                   *
//*  2. make the menu visible                                                  *
//*  3. get user's key input                                                   *
//*  4. hide the menu                                                          *
//*  5. deactivate the menu (make it not accessible to user input)             *
//*                                                                            *
//* Input  : cIndex : index of a context (normally invisible) Menuwin control  *
//*          info   : uiInfo class (by reference) - initial values ignored     *
//*                                                                            *
//* Returns: index of control that currently has the input focus               *
//*           The uiInfo-class data members have been initialized.             *
//*           (See note in NcDialog.hpp about interpretation )                 *
//*           (of values returned in the uiInfo-class object.)                 *
//******************************************************************************

short NcDialog::EditMenuwin ( short cIndex, uiInfo& info )
{
   short exitCtrl = this->currCtrl ; // we return to this control before exit to caller

   //* Validate caller's input *
   if ( cIndex >= ZERO && cIndex <= this->lastCtrl && this->dCtrl[cIndex]->type == dctMENUWIN )
   {
      //* Access to the control's methods and data members *
      DialogMenuwin* cp = (DialogMenuwin*)this->dCtrl[cIndex] ;
      if ( cp->tCols == ZERO && cp->active == false )
      {  //* Redraw control with focus in non-focus colors *
         this->RedrawCurrControl ( false ) ;

         //* Temporarily make this control active, so it can receive *
         //* input focus, then give it the focus.                    *
         cp->active = true ;
         this->currCtrl = cIndex ;

         //* Instruct edit method to immediately expand control on entry *
         info.viaHotkey = true ;
         this->EditMenuwin ( info ) ;

         //* Return focus to original control and deactivate the context menu  *
         this->currCtrl = exitCtrl ;
         this->RedrawCurrControl ( true ) ;
         cp->active = false ;
      }
      else     // caller's index does not reference a context menu
      {  //* Minimize the effects of caller's error *
         //this->DebugMsg ( "Error! Not a context menu", 3 ) ;
         info.ctrlType = this->dCtrl[this->currCtrl]->type ; // control type
         info.ctrlIndex = this->currCtrl ;   // control with input focus
         info.hasFocus = true ;              // focus was not changed
         info.keyIn = ZERO ;                 // no need for focus change
         info.dataMod = false ;              // data unchanged
         info.selMember = ZERO ;             // don't care
         info.isSel = false ;                // don't care
         info.viaHotkey = false ;            // control did not get focus via hotkey
      }
   }
   return this->currCtrl ;

}  //* End EditMenuwin() *

//************************
//*   emwEditCollapsed   *
//************************
//******************************************************************************
//* Private method: A 'collapsed' menu or menu group has the input focus.      *
//* Interact with user until:                                                  *
//*   1. user request to open 'expand' the menu or a member of menu group      *
//*   2. the original menu or menu group loses focus either via cursor keys    *
//*      or via kotkey                                                         *
//*                                                                            *
//* Input  : info: uiInfo class (by reference) - initial values ignored        *
//*                  (info is initialized with the result of the edit)         *
//*                                                                            *
//* Returns: 'false' if user requested that current menu or a member of the    *
//*                  current menu group should be opened                       *
//*          'true'  if original menu or menu group has lost focus             *
//******************************************************************************

bool NcDialog::emwEditCollapsed ( uiInfo& info )
{
   bool lostFocus = true ;          // return code
   DialogMenuwin* cp = (DialogMenuwin*)this->dCtrl[this->currCtrl] ;

   //**************************
   //* Interact with the user *
   //**************************
   bool  done = false ;          // loop control
   while ( ! done )
   {
      //* 1) Get user input.                                  *
      //* 2) Check for toggle of Insert/Overstrike mode.      *
      //* 3) Check for terminal-resize event.                 *
      //* 4) Access callback method, if specified.            *
      //* 5) If input has been handled, return to top of loop.*
      this->GetKeyInput ( info.wk ) ;
      
      if ( (info.wk.type == wktFUNKEY)
            && (info.wk.key == nckINSERT || info.wk.key == nckRESIZE) )
      {
         if ( info.wk.key == nckINSERT )
            this->ToggleIns () ;
         else
            this->TermResized () ;

         if ( ExternalControlUpdate != NULL )
            ExternalControlUpdate ( this->currCtrl, info.wk, false ) ;
         continue ;
      }

      //* Expand the current Menuwin for edit.*
      if ( ((info.wk.type == wktPRINT) && (info.wk.key == nckSPACE))
           ||
           ((info.wk.type == wktFUNKEY) && (info.wk.key == nckENTER || 
             info.wk.key == nckpENTER || info.wk.key == nckPGDOWN )) )
      {
         info.wk = { nckENTER, wktFUNKEY };// be consistent
         info.ctrlType = dctMENUWIN ;     // control type
         info.ctrlIndex = this->currCtrl ;// control with input focus
         info.hasFocus = true ;           // focus was not lost
         info.keyIn = info.wk.key ;       // indicates menu to be expanded
         info.dataMod = false ;           // no menu selection made
         info.selMember = cp->bIndex ;    // current member is the 'selected' member
         info.isSel = false ;             // don't care
         info.viaHotkey = false ;         // control did not get focus via hotkey
         done = true ;                    // done with input loop
         lostFocus = false ;              // returning to caller
      }

      //* If user is moving to next control or previous control *
      else if ( (info.wk.type == wktFUNKEY) &&
                (info.wk.key == nckTAB  || info.wk.key == nckRIGHT || 
                 info.wk.key == nckDOWN || info.wk.key == nckSTAB  || 
                 info.wk.key == nckLEFT || info.wk.key == nckUP    || 
                 info.wk.key == nckPGUP) )
      {
         if (info.wk.key == nckTAB  || info.wk.key == nckRIGHT || 
             info.wk.key == nckDOWN )
            info.wk.key = nckTAB ;
         else
            info.wk.key = nckSTAB ;

         //* If current menu control is a member of a MenuBar *
         if ( cp->groupCode != ZERO )
         {
            short newTarget = this->GetActiveCtrl ( this->currCtrl, 
                                 (bool)(info.wk.key==nckTAB ? true : false) ) ;
            //* If new target control is a member of the same group *
            if (this->dCtrl[newTarget]->groupCode == cp->groupCode )
            {
               if ( info.wk.key == nckTAB )
                  this->NextControl () ;
               else
                  this->PrevControl () ;
               cp = (DialogMenuwin*)this->dCtrl[this->currCtrl] ;
            }
            else  // new target control is not a member of this group, return to caller
               done = true ;
         }
         else  // current control is not a member of any group, return to caller
            done = true ;

         //* Menuwin edit is complete, return to caller, then to application.*
         if ( done != false )
         {
            info.ctrlType = dctMENUWIN ;  // control type
            info.ctrlIndex = this->currCtrl ;// control with input focus
            info.hasFocus = true ;        // focus was not lost
            info.keyIn = info.wk.key ;    // direction of focus shift
            info.dataMod = false ;        // indicate no selection made
            info.selMember = cp->bIndex ; // current member is the 'selected' member
            info.isSel = false ;          // don't care
            info.viaHotkey = false ;      // control did not get focus via hotkey
         }
      }

      //* If user is trying to leave a MenuBar (or is simply confused) *
      else if ( (info.wk.type == wktFUNKEY) && (info.wk.key == nckESC) )
      {
         //* If this control is a member of a Menuwin control group, *
         //* move the input focus to the last member of the group.   *
         this->emwFocus2GroupEnd ( true ) ;

         //* Signal a move out of the control or group (no selection made)  *
         info.ctrlType = dctMENUWIN ;     // control type
         info.ctrlIndex = this->currCtrl ;// control with input focus
         info.hasFocus = true ;           // focus on current control
         info.keyIn = info.wk.key ;       // focus will move forward
         info.dataMod = false ;           // data unchanged
         info.selMember = cp->bIndex ;    // current member is the 'selected' member
         info.isSel = false ;             // don't care
         info.viaHotkey = false ;         // control did not get focus via hotkey

         done = true ;                    // done with input loop
         lostFocus = true ;               // returning to caller
      }

      //* Else, check for control selection via hotkey *
      else
      {
         //* Scan the controls for one whose hotkey matches the input.*
         //* If match found, returns control's index, else -1.        *
         short hotIndex = this->IsHotkey ( info.wk ) ;
         if ( hotIndex >= ZERO )
         {
            //* We have a valid hotkey. If hotkey references current control   *
            //* or a member of current control's group, alert caller to expand *
            //* referenced menu.                                               *
            if ( hotIndex == this->currCtrl ||
                 (cp->groupCode != ZERO && 
                  (cp->groupCode == this->dCtrl[hotIndex]->groupCode)) )
            {
               if ( hotIndex != this->currCtrl )
               {
                  while ( this->currCtrl > hotIndex )
                     this->PrevControl () ;
                  while ( this->currCtrl < hotIndex )
                     this->NextControl () ;
               }
               info.wk = { nckENTER, wktFUNKEY };// be consistent
               info.ctrlType = dctMENUWIN ;     // control type
               info.ctrlIndex = this->currCtrl ;// control with input focus
               info.hasFocus = true ;           // focus was not lost
               info.keyIn = info.wk.key ;       // indicates menu to be expanded
               info.dataMod = false ;           // no menu selection made
               info.selMember = cp->bIndex ;    // current member is the 'selected' member
               info.isSel = false ;             // don't care
               info.viaHotkey = false ;         // control did not get focus via hotkey
               done = true ;                    // done with input loop
               lostFocus = false ;              // returning to caller
            }
            //* Else hotIndex references a control that is neither the current *
            //* control nor a member of its group. Return news to caller.      *
            else
            {
               info.ctrlType = dctMENUWIN ;     // control type
               info.ctrlIndex = this->currCtrl ;// control with input focus
               info.hasFocus = true ;           // this may be reset by ChangedFocusViaHotkey()
               info.keyIn = ZERO ;              // indicate no shift in focus necessary
               info.dataMod = false ;           // indicate no selection change
               info.selMember = cp->bIndex ;    // current member is the 'selected' member
               info.isSel = false ;             // don't care
               info.viaHotkey = false ;         // this may be set by ChangedFocusViaHotkey()
               done = true ;

               //* Make the indicated control the current/active control *
               this->ChangedFocusViaHotkey ( info.wk.key, hotIndex, info ) ;
               // on return, new control has the focus, and the info.h_XX
               // variables adjusted
            }
         }
         //* Is NOT a standard hotkey. Check for auxilliary hotkey indicating  *
         //* an attempt to close the MenuBar currently under edit. Aux hotkey  *
         //* for a MenuBar NOT under edit was handled silently by IsHotkey().  *
         else
         {
            short ahkIndex ;     // receives index of first menu in group (or -1)
            bool  isAuxHotkey = emwIsAuxHotkey ( info.wk, ahkIndex ) ;

            if ( isAuxHotkey && (this->dCtrl[ahkIndex]->groupCode == cp->groupCode) )
            {
               //* Move the input focus away from the MenuBar and then hide it.*
               //* If successful, focus is now on the control BEFORE MenuBar;  *
               //* else the Hide failed, so focus is on LAST member of         *
               //* MenuBar. Either way, return to caller.                      *
               this->HideMenuBar ( this->currCtrl ) ;

               //* Set up the returned data to indicate a move out of the      *
               //* MenuBar (no selection made). The data are somewhat          *
               //* fictional, but application should not use it anyway if      *
               //* dataMod==false.                                             *
               info.ctrlType = this->dCtrl[this->currCtrl]->type ; // control type
               info.ctrlIndex = this->currCtrl ;// control with input focus
               info.hasFocus = false ;          // control under edit has lost focus
               info.keyIn = ZERO ;              // indicate no shift in focus necessary
               info.dataMod = false ;           // indicate no selection change
               info.selMember = ZERO ;          // don't care
               info.isSel = false ;             // don't care
               info.viaHotkey = false ;// focus NOT received via hotkey (even though it was)
               done = true ;                    // done with input loop
               lostFocus = true ;               // returning to caller
            }
            else
               { /* all other key input is discarded */ }
         }
      }
      //* If caller has specified a callback method, do it now.*
      if ( ExternalControlUpdate != NULL )
         ExternalControlUpdate ( this->currCtrl, info.wk, false ) ;
   }     // while()
   return lostFocus ;

   
}  //* End emwEditCollapsed() *

//************************
//*  emwEditMultilevel   *
//************************
//******************************************************************************
//* Private method: Interact with user until he/she/it makes a selection from  *
//* the specified menu or from any of its sub-menus, or exits without making   *
//* a selection.                                                               *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  : refList : array of indices to open parent menu controls that must *
//*                    be refreshed when a sub-menu closes                     *
//*                    First element is index of next free element             *
//*          info    : uiInfo class (by reference) - initial values ignored    *
//*                                                                            *
//* Returns: key data for last key user pressed is returned in info.wk         *
//*           The uiInfo-class data members have been initialized.             *
//******************************************************************************

void NcDialog::emwEditMultilevel ( short refList[], uiInfo& info )
{
   //* Get a pointer to control with input focus, *
   //* and save its index in refresh list.        *
   DialogMenuwin* cp = (DialogMenuwin*)this->dCtrl[this->currCtrl] ;
   refList[(*refList)++] = this->currCtrl ;

   //**************************
   //* Interact with the user *
   //**************************
   bool     done = false ;          // loop control
   while ( ! done )
   {
      //* Let user make a selection *
      this->emwEditSingle ( cp, info ) ;

      //* If an item was selected AND the selected item does not have an       *
      //* attached sub-menu, return the selection to caller.                   *
      if ( (info.dataMod != false) && (cp->subMenu[info.selMember] == MAX_DIALOG_CONTROLS) )
         done = true ;

      //* If the currently-highlighted menu item DOES HAVE an attached         *
      //* sub-menu, AND either user selected the item OR user has pressed      *
      //* nckRIGHT key to enter that sub-menu, open the sub-menu and get       *
      //* user's selection.                                                    *
      else if (   (cp->subMenu[info.selMember] < MAX_DIALOG_CONTROLS) 
               && ((info.dataMod != false) ||
                   (info.wk.type == wktFUNKEY && info.wk.key == nckRIGHT)) )
      {
         //* Save dialog window's display data *
         this->CaptureDialogDisplayData () ;

         //* Enable user access to the submenu *
         short ptIndex = this->currCtrl,              // index of this parent menu
               smIndex = cp->subMenu[info.selMember] ;// index of sub-menu
         DialogMenuwin* cpt = (DialogMenuwin*)this->dCtrl[smIndex] ;
         cpt->active = true ;

         //* Count the attached sub-sub-menus *
         short submenuCount = this->emwSubmenuCount ( cpt ) ;

         //* Give input focus to the target menu *
         cp->RedrawControl ( false, false ) ;
         this->currCtrl = smIndex ;
         cpt->RedrawControl ( true, true ) ;

         //* Expand the sub-menu for editing *
         cpt->bIndex = ZERO ;       // Start with highlight on first menu item
         cpt->expanded = true ;
         cpt->RedrawControl ( true ) ;

         //* If there are no sub-sub menus, call edit method directly *
         if ( submenuCount == ZERO )
            this->emwEditSingle ( cpt, info ) ;
         //* Else, there are additional menu levels, so recurse *
         else
            this->emwEditMultilevel ( refList, info ) ;

         cpt->expanded = false ;             // collapse the sub-menu
         this->RefreshWin () ;               // refresh parent dialog's display
         this->currCtrl = ptIndex ;          // parent menu becomes control with focus
         cp->RedrawControl ( true, false ) ;
         cpt->active = false ;               // de-activate the sub-menu

         //* Refresh the list of open menus, oldest-to-newest *
         if ( info.dataMod == false && *refList > 2 )
         {
            if ( submenuCount > ZERO )
               --(*refList) ;
            for ( short i = 1 ; i < *refList ; i++ )
            {
               DialogMenuwin* cx = (DialogMenuwin*)this->dCtrl[refList[i]] ;
               cx->RefreshControl () ;
            }
         }

         //* Unless user has returned to continue edit of parent menu, *
         //* we're done here, so return to caller.                     *
         if ( !((info.wk.type == wktFUNKEY) && (info.wk.key == nckLEFT)) )
            done = true ;
      }

      //* Else, no selection was made. Determine what to do next based on      *
      //* last key pressed.                                                    *
      else
      {
         if ( (info.wk.type == wktFUNKEY) && 
              (info.wk.key == nckTAB   || info.wk.key == nckSTAB || 
               info.wk.key == nckRIGHT || info.wk.key == nckLEFT || 
               info.wk.key == nckESC) )
         {
            //* Tab and right arrow mean the same thing here *
            if ( info.wk.key == nckRIGHT )
               info.wk.key = nckTAB ;

            //* A left arrow key when in a submenu means a return to parent *
            //* menu, but when in a primary menu, it's the same as nckSTAB. *
            // NOTE: IF THE SUB-MENU IS POSITIONED ON THE LEFT OF THE PARENT, 
            //       THEN THE RIGHT ARROW SHOULD RETURN TO PARENT, BUT AT THIS 
            //       LEVEL, ALTHOUGH WE CAN DETECT WHETHER WE HAVE A PARENT, 
            //       IT IS NOT EASY TO DETECT WHERE WE ARE POSITIONED 
            //       RELATIVE TO THAT PARENT -IT IS _POSSIBLE_ HOWEVER. 
            //       THINK ABOUT THIS...
            if (   (info.wk.type == wktFUNKEY) && (info.wk.key == nckLEFT) 
                && (cp->tCols > ZERO) )
               info.wk.key = nckSTAB ;

            info.ctrlType = dctMENUWIN ;     // control type
            info.ctrlIndex = this->currCtrl ;// control with input focus
            info.hasFocus = true ;           // focus was not lost
            info.keyIn = info.wk.key ;       // direction focus will move
            info.dataMod = false ;           // indicate no item selected
            info.selMember = cp->bIndex ;    // don't care, no selection made
            info.isSel = false ;             // don't care
            info.viaHotkey = false ;         // control did not get focus via hotkey
            done = true ;                    // returning to caller
         }
         else
            { /* all other key input is discarded */ }
      }
   }     // while()

}  //* End emwEditMultilevel() *

//************************
//*    emwEditSingle     *
//************************
//******************************************************************************
//* Private method: Interact with user until he/she/it makes a selection from  *
//* the menu or exits without making a selection.                              *
//*                                                                            *
//* On entry, the menu has been 'expanded' AND has received the input focus.   *
//* On exit, menu remains 'expanded' and caller must decide whether to         *
//*          to 'collapse' the menu.                                           *
//*                                                                            *
//* Input  : cp     : pointer to DialogMenuwin object to be edited             *
//*          info   : uiInfo class (by reference) - initial values ignored     *
//*                                                                            *
//* Returns: key data for last key user pressed is returned in info.wk         *
//*           The uiInfo-class data members have been initialized.             *
//******************************************************************************
//* Programmer's Note:                                                         *
//* A menu's internal hotkeys are 'A'-'Z' and 'a'-'z' only. Alt and Ctrl       *
//* combinations are not processed. Still, there is a good chance that any     *
//* hotkeys defined within the menu items will shadow hotkeys defined for      *
//* other controls. Tough rocks, pal; the menu is open and thus gets first     *
//* choice of juicy keystrokes. Hotkeys attached to other controls are         *
//* ignored while we are editing within an expanded menu control.              *
//******************************************************************************

void NcDialog::emwEditSingle ( DialogMenuwin* cp, uiInfo& info )
{
   #define CAPTURE_MENUWIN (0)      // For debugging only, capture screenshot

   //**************************
   //* Interact with the user *
   //**************************
   bool  done = false ;       // loop control
   while ( ! done )
   {
      //* 1) Get user input.                                  *
      //* 2) Check for toggle of Insert/Overstrike mode.      *
      //* 3) Check for terminal-resize event.                 *
      //* 4) Access callback method, if specified.            *
      //* 5) If input has been handled, return to top of loop.*
      this->GetKeyInput ( info.wk ) ;

      if ( (info.wk.type == wktFUNKEY)
            && (info.wk.key == nckINSERT || info.wk.key == nckRESIZE) )
      {
         if ( info.wk.key == nckINSERT )
            this->ToggleIns () ;
         else
            this->TermResized () ;

         if ( ExternalControlUpdate != NULL )
            ExternalControlUpdate ( this->currCtrl, info.wk, false ) ;
         continue ;
      }

      //* If input is a hotkey internal to the menu, convert it to an index    *
      //* into the menu-item array, scroll to that item, and set key input     *
      //* as nckENTER indicating 'item selected'.                              *
      //* If  not a valid menu hotkey, do nothing.                             *
      if ( !(info.wk.type == wktFUNKEY && 
             (info.wk.key == nckENTER || info.wk.key == nckpENTER)) )
         cp->bIndex = cp->wPtr->Hotkey2Index ( info.wk.key ) ;
      //* Fool the function-key test below *
      if ( info.wk.key == nckENTER || info.wk.key == nckpENTER )
         info.wk.type = wktFUNKEY ;

      //* User has selected the currently-highlighted menu item *
      if (   (info.wk.type == wktPRINT && info.wk.key == nckSPACE)
          || (info.wk.type == wktFUNKEY && 
              (info.wk.key == nckENTER || info.wk.key == nckpENTER)) )
      {
         info.ctrlType = dctMENUWIN ;     // control type
         info.ctrlIndex = this->currCtrl ;// control with input focus
         info.hasFocus = true ;           // focus was not lost
         info.keyIn = nckENTER ;          // selection made, not just leaving the control
         info.dataMod = true ;            // indicate selection has been made
         info.selMember = cp->bIndex ;    // current member is the 'selected' member
         info.isSel = false ;             // don't care
         info.viaHotkey = false ;         // control did not get focus via hotkey
         done = true ;                    // returning to caller
      }

      //* Terminated edit without making a selection *
      else if ( (info.wk.type == wktFUNKEY) && 
                (info.wk.key == nckTAB  || info.wk.key == nckRIGHT || 
                 info.wk.key == nckSTAB || info.wk.key == nckLEFT  || 
                 info.wk.key == nckESC) )
      {
         info.ctrlType = dctMENUWIN ;     // control type
         info.ctrlIndex = this->currCtrl ;// control with input focus
         info.hasFocus = true ;           // focus was not lost
         if ( info.wk.key == nckRIGHT )   // key input 
            info.keyIn = nckTAB ;         //   (note: application does not see)
         else if ( info.wk.key == nckLEFT )//  (left and right arrow, that's  ) 
            info.keyIn = nckSTAB ;        //   (only for us...                )
         else
            info.keyIn = info.wk.key ;
         info.dataMod = false ;           // indicate no item selected
         info.selMember = cp->bIndex ;    // don't care, no selection made
         info.isSel = false ;             // don't care
         info.viaHotkey = false ;         // control did not get focus via hotkey
         done = true ;                    // returning to caller

         #if CAPTURE_MENUWIN != 0         // Debugging Only: capture screenshot
         //* Used only to capture screenshots of open Menuwin *
         //* objects for documentation purposes.              *
         if ( info.wk.key == nckESC )
         {  //* Capture the entire dialog *
            this->CaptureDialog ( "./captureMW.txt" ) ;
            this->CaptureDialog ( "./captureMW.html", true, false, 
                                  "infodoc-styles.css", 4, false, nc.blR ) ;
         }
         #endif // CAPTURE_MENUWIN != 0
      }

      //* Scroll up or down through list *
      // Note: nckLEFT and nckRIGHT are scroll keys also, but we eliminated 
      //       them in above test because they don't have a function in 
      //       a vertical scrolling control.
      else if ( (cp->wPtr->IsScrollKey ( info.wk )) != false )
      {
         //* On return, cp->bIndex holds currently-highlighted item *
         cp->ScrollMenuData ( info.wk ) ;
      }

      else
      {
         //* All other key input is discarded.                                 *
         //* Note that hotkeys for controls are not recognized during          *
         //* the menu edit. This is due to the fact that many menus have       *
         //* internal hotkeys which would mask hotkeys associated with some    *
         //* external controls; thus, it is more consistent to ignore all      *
         //* external hotkey requests while the menu is open for editing.      *
      }

      //* If caller has specified a callback method, do it now.*
      if ( ExternalControlUpdate != NULL )
         ExternalControlUpdate ( this->currCtrl, info.wk, false ) ;
   }     // while()

   #undef CAPTURE_MENUWIN     // For debugging only, capture screenshot
}  //* End emwEditSingle() *

//************************
//*   emwSubmenuCount    *
//************************
//******************************************************************************
//* Private method: Count the number of sub-menus attached to the specified    *
//* Menuwin control.                                                           *
//*                                                                            *
//* A sub-menu is a context menu which has been assigned to an individual item *
//* of a primary menu and opens for user edit when the primary menu item is    *
//* selected by the user. Sub-menus may also have attached sub-menus, to an    *
//* arbitrary depth limited only by the total number of controls allowed for   *
//* the parent dialog.                                                         *
//*                                                                            *
//*               First-level    Second-level                                  *
//* Primary Menu  Submenu        Submenu                                       *
//* +----------+                                                               *
//* | Item 1   |                                                               *
//* | Item 2   +--------------+                                                *
//* | Item 3  >| Subitem 1    +--------------+                                 *
//* +----------+ Subitem 2   >| Ssubitem 1   |                                 *
//*            | Subitem 3    + Ssubitem 2   |  and so on...                   *
//*            | Subitem 4    | Ssubitem 3   |                                 *
//*            +--------------| Ssubitem 4   |                                 *
//*                           +--------------+                                 *
//*                                                                            *
//* Input  : cp     : pointer to DialogMenuwin object to be edited             *
//*                                                                            *
//* Returns: number of attached sub-menus                                      *
//******************************************************************************

short NcDialog::emwSubmenuCount ( DialogMenuwin* cp )
{
   short submenuCount = ZERO ;
   for ( short i = ZERO ; i < cp->bItems ; i++ )
   {
      if ( cp->subMenu[i] < MAX_DIALOG_CONTROLS )
      {
         ++submenuCount ;

         //* Count sub-sub menus *
         DialogMenuwin* scp = (DialogMenuwin*)this->dCtrl[cp->subMenu[i]] ;
         submenuCount += this->emwSubmenuCount ( scp ) ;
      }
   }
   return submenuCount ;

}  //* End emwSubmenuCount() *

//************************
//*   emwIsAuxHotkey     *
//************************
//******************************************************************************
//* Scan all controls for matching auxilliary hotkey.                          *
//*                                                                            *
//*                                                                            *
//* Input  : wk      : key data to compare against controls' aux hotkeys       *
//*          ahkIndex: (by reference)                                          *
//*                    if a match is found, receives index of first control    *
//*                    in corresponding MenuBar group, else (-1)               *
//*                                                                            *
//* Returns: 'true' if match found, else 'false'                               *
//******************************************************************************

bool NcDialog::emwIsAuxHotkey ( wkeyCode wk, short& ahkIndex )
{
   bool  matchFound = false ;

   for ( short i = ZERO ; i <= this->lastCtrl && matchFound == false ; i++ )
   {  //* If the control is a member control of a MenuBar *
      if ( (this->dCtrl[i]->type == dctMENUWIN) && (this->dCtrl[i]->groupCode != ZERO) )
      {
         if ( (matchFound = (wk.key == ((DialogMenuwin*)this->dCtrl[i])->grpHotkey)) )
            ahkIndex = i ;
      }
   }
   return matchFound ;

}  //* End emwIsAuxHotkey() *

//************************
//*    ToggleMenuBar     *
//************************
//******************************************************************************
//* Private Method:  Toggle visibility of specified MenuBar.                   *
//*                                                                            *
//*                                                                            *
//* Input  : mbIndex: index of any member of the MenuBar control group that is *
//*                   to be made visible/invisible                             *
//*                                                                            *
//* Returns: 'true' if MenuBar state changed                                   *
//*          'false' if:                                                       *
//*            1. mbIndex is invalid                                           *
//*            2. mbIndex does not reference a control of type dctMENUWIN      *
//*            3. control referenced by mbIndex is not a member of a MenuBar   *
//*            4. the control that has input focus is a member of the target   *
//*               MenuBar                                                      *
//*            5. control with focus is a context menu or sub-menu             *
//*            6. control with focus is an expanded dctMENUWIN control         *
//*            7. control with focus is an expanded dctDROPDOWN control        *
//******************************************************************************
//* If focus is currently on an 'expanded' control (dctMENUWIN or dctDROPDOWN),*
//* we must disallow the show/hide operation because it would prematurely      *
//* refresh the dialog window, restoring the display data saved before control *
//* was expanded --  AND/OR the MenuBar and the 'expanded' control may share   *
//* display space causing visual ugliness.                                     *
//******************************************************************************

bool NcDialog::ToggleMenuBar ( short mbIndex )
{
   short fIndex = this->currCtrl ;  // index of control with input focus
   bool  success = false ;          // return value

   //* Test for control under edit in 'expanded' state. See note above.*
   bool  focusCtrlIsExpanded = false ;
   if ( this->dCtrl[fIndex]->type == dctMENUWIN )
   {
      DialogMenuwin* cpt = (DialogMenuwin*)this->dCtrl[fIndex] ;
      focusCtrlIsExpanded = cpt->expanded ;
   }
   if ( focusCtrlIsExpanded == false && this->dCtrl[fIndex]->type == dctDROPDOWN )
   {
      DialogDropdown* cpt = (DialogDropdown*)this->dCtrl[fIndex] ;
      focusCtrlIsExpanded = cpt->expanded ;
   }

   //* Perform preliminary tests of target control and focus control.*
   if ( (mbIndex >= ZERO && mbIndex <= this->lastCtrl) &&
        (this->dCtrl[mbIndex]->type == dctMENUWIN) && 
        (this->dCtrl[mbIndex]->groupCode > ZERO) &&
        (! focusCtrlIsExpanded) )
   {
      //* Determine whether control with input focus is *
      //* a member control of the target MenuBar.       *
      if ( this->dCtrl[fIndex]->type == dctMENUWIN )
      {
         //* If control with focus is not a primary member of any MenuBar      *
         //* group, but MAY be an independent menu.                            *
         char  mbGroup = this->dCtrl[mbIndex]->groupCode,
               fGroup  = this->dCtrl[fIndex]->groupCode ;
         if ( fGroup == ZERO )
         {
            DialogMenuwin* cpt = (DialogMenuwin*)this->dCtrl[fIndex] ;
            //* If control with focus is an un-grouped primary menu *
            if ( cpt->tCols > ZERO )
               success = true ;
         }
         else if ( fGroup > ZERO && fGroup != mbGroup )
            success = true ;        // control with focus is member of different MenuBar
         else
         {  /* mbGroup == fGroup so toggle is not possible.  */ }
      }
      else                          // control with focus not a Menuwin control
         success = true ;

      //* If there are no conflicts, toggle MenuBar visibility. *
      if ( success )
      {
         if ( ((DialogMenuwin*)this->dCtrl[mbIndex])->grpVisible == false )
            this->ShowMenuBar ( mbIndex ) ;
         else
            this->HideMenuBar ( mbIndex ) ;
      }
   }
   return success ;

}  //* End ToggleMenuBar() *

//**************************
//*   emwFocus2GroupEnd    *
//**************************
//******************************************************************************
//* If the control which currently has focus is a member of a MenuBar group,   *
//* move the focus to the last member (highest index) of that group or first   *
//* member (lowest index) of that group.                                       *
//* This method is called when the MenuBar is about to lose focus i.e.         *
//* when the focus is about to be moved to a control which IS NOT a member.    *
//*                                                                            *
//* If the current control IS NOT a member of a MenuBar, we do nothing.        *
//*                                                                            *
//*                                                                            *
//* Input  : last: 'true'  == move to last member (highest index) of group     *
//*                'false' == move to first member (lowest index) of group     *
//*                                                                            *
//* Returns: index of control which initially has focus (for comparison)       *
//******************************************************************************

short NcDialog::emwFocus2GroupEnd ( bool last )
{
   short initialIndex = this->currCtrl ;     // return value
   char  gCode = this->dCtrl[this->currCtrl]->groupCode ;
   if ( gCode != ZERO )    // if current control is member of a group
   {
      short targetIndex = this->currCtrl ;   // where we want focus to be
      if ( last != false )       // move to last member of the group
      {
         for ( short i = this->currCtrl+1 ; i <= this->lastCtrl ; i++ )
         {
            if ( this->dCtrl[i]->groupCode == gCode )
               targetIndex = i ;
         }
         //* Advance the focus to the last control in the the group.*
         while ( this->currCtrl < targetIndex )
            this->NextControl () ;
      }
      else                       // move to first member of the group
      {
         for ( short i = this->currCtrl-1 ; i >= ZERO ; i-- )
         {
            if ( this->dCtrl[i]->groupCode == gCode )
               targetIndex = i ;
         }
         //* Advance the focus to the first control in the the group.*
         while ( this->currCtrl > targetIndex )
            this->PrevControl () ;
      }
   }
   return initialIndex ;

}  //* End Focus2GroupEnd() *

//*************************
//*  PositionContextMenu  *
//*************************
//******************************************************************************
//* Set the position of a dctMENUWIN control which acts as a context menu.     *
//*                                                                            *
//* Context menus are DialogMenuwin class objects that are invisible when in   *
//* the 'collapsed' state, which is any time they are not being edited by the  *
//* user. In a GUI application, this would be the menu that opens when the     *
//* right mouse button is pressed. A context menu may be opened anywhere       *
//* within the dialog window. A context menu is created by instantiating a     *
//* dctMENUWIN object with a zero-length label.                                *
//*                                                                            *
//* Input  : cIndex: index of control to be positioned                         *
//*          offY  : Y offset from upper left of parent dialog                 *
//*          offX  : X offset from upper left of parent dialog                 *
//*                                                                            *
//* Returns: OK if successful                                                  *
//*          ERR if error:                                                     *
//*           1. specified control is not a context menu,                      *
//*           2. the menu is already expanded (visible),                       *
//*           3. specified offset puts menu on or beyond dialog border.        *
//******************************************************************************

short NcDialog::PositionContextMenu ( short cIndex, short offY, short offX )
{
short    result = ERR ;

   //* Check control type and for caller stupidity,   *
   //* i.e. offsets must be inside the dialog borders.*
   if ( this->dCtrl[cIndex]->type == dctMENUWIN && offY > ZERO && offX > ZERO )
   {
      DialogMenuwin* cp = (DialogMenuwin*)this->dCtrl[cIndex] ;
      //* If it is a context menu AND it is not currently visible *
      if ( cp->tCols == ZERO && cp->expanded == false )
      {
         short limitRight = this->dCols - cp->cols - 1,
               limitBottom = this->dLines - cp->lines - 1 ;
         //* If control will fit entirely within dialog borders, *
         //* I like to Move-It Move-It!!                         *
         if ( offY <= limitBottom && offX <= limitRight )
         {
            cp->tPtr->wp->_begy = cp->tPtr->wulY = cp->tulY = this->dulY + offY ;
            cp->tPtr->wp->_begx = cp->tPtr->wulX = cp->tulX = this->dulX + offX ;
            cp->bPtr->wp->_begy = cp->bPtr->wulY = 
                  this->dCtrl[cIndex]->ulY = this->dulY + offY ;
            cp->bPtr->wp->_begx = cp->bPtr->wulX = 
                  this->dCtrl[cIndex]->ulX = this->dulX + offX ;
            this->dCtrl[cIndex]->wPtr->wp->_begy = 
                  this->dCtrl[cIndex]->wPtr->wulY = this->dCtrl[cIndex]->ulY + 1 ;
            this->dCtrl[cIndex]->wPtr->wp->_begx = 
                  this->dCtrl[cIndex]->wPtr->wulX = this->dCtrl[cIndex]->ulX + 1 ;
            result = OK ;
         }
      }
   }

   return result ;

}  //* End PositionContextMenu() *

//************************
//* GroupMenuwinControls *
//************************
//******************************************************************************
//* Establish a MenuBar (a group of DialogMenuwin class objects).              *
//* The controls of a MenuBar may be edited as a group when EditMenuwin() is   *
//* called, returning when a selection is made in ANY member control of the    *
//* MenuBar's group.                                                           *
//*                                                                            *
//* The list of group members is passed as an array of index numbers for the   *
//* DialogMenuwin objects. List is terminated by a negative 1 (-1).            *
//*   Example: short List[] = { 4, 5, 6, -1 } ;                                *
//* NOTE: Sub-menus and other context menus MAY NOT be members of a MenuBar.   *
//*       Only visible, top-level menus, may be members of a MenuBar.          *
//*                                                                            *
//* NOTE: If specified indices are not consecutive, anomolous display behavior *
//*       may result, so members of a group should have consecutive indices.   *
//*                                                                            *
//* Input  : indexList: array of control index numbers (-1 ends list)          *
//*          auxHotkey: (optional, default==ZERO): if a value is specified,    *
//*                     it is an auxilliary hotkey that toggles the MenuBar's  *
//*                     visibility/invisibility                                *
//*                     Note that this key will not be recognized while in the *
//*                     EditXxx() method for an 'expanded' control because the *
//*                     expanded control may obscure the MenuBar's position.   *
//*                                                                            *
//* Returns: OK if successful                                                  *
//*          or ERR if:                                                        *
//*          1. if index list contains an invalid index number                 *
//*          2. an index in the list references a control which is not a       *
//*             DialogMenuwin object                                           *
//*          3. an index in the list references a sub-menu or context menu     *
//******************************************************************************
short NcDialog::GroupMenuwinControls ( const short indexList[], wchar_t auxHotkey )
{
DialogMenuwin* cp ;
short    lIndex = ZERO,
         status = OK ;

   //* Mark the specified DialogMenuwin controls as members of the group.      *
   while ( indexList[lIndex] > -1 && lIndex <= this->lastCtrl && status == OK )
   {
      if ( this->dCtrl[indexList[lIndex]]->type == dctMENUWIN )
      {
         cp = (DialogMenuwin*)this->dCtrl[indexList[lIndex]] ;
         if ( cp->tCols > ZERO )    // if menu IS NOT an (invisible) context menu
         {
            cp->groupCode = this->cgGroupCode ; // assign the group code
            cp->grpVisible = true ;             // MenuBars are initially visible
            cp->grpHotkey = auxHotkey ;         // (optional) auxilliary hotkey
            ++lIndex ;
         }
         else
            status = ERR ;
      }
      else
         status = ERR ;
   }

   if ( status == OK )
      ++this->cgGroupCode ;
   return status ;

}  //* End GroupMenuwinControls() *

//***************************
//*  AttachMenuwinSubmenus  *
//***************************
//******************************************************************************
//* Establish a connection between display elements in a dctMENUWIN control    *
//* and context menus that will act as submenus for those elements.            *
//* Note: A context menu is a dctMENUWIN control that has been defined with    *
//*       a NULL string label, and is therefore invisible when in its          *
//*       collapsed state i.e. does not have input focus.                      *
//*                                                                            *
//*                                                                            *
//* Input  : pMenu   : index of parent menu of type dctMENUWIN to              *
//*                      which sub-menus will be attached.                     *
//*          smList[]: array of index numbers for the DialogMenuwin            *
//*                      context menu objects to be associated with            *
//*                      each data item in the parent menu.                    *
//*                                                                            *
//* Returns: OK if successful OR                                               *
//*          ERR if:                                                           *
//*           1. invalid index number                                          *
//*           2. any referenced control is not of dctMENUWIN type              *
//*           3. any specified sub-menu not defined as a context menu          *
//******************************************************************************
//* User can then access the sub-menus by first opening the parent menu,       *
//* then scrolling to the desired data item and pressing the nckRIGHT arrow    *
//* key or nckENTER key. The parent menu will remain open and the sub-menu     *
//* will be opened at its defined position. Note that style indicates the      *
//* sub-menu shoud be adjacent to or overlapping the border of the right       *
//* edge (or left edge if necessary) of the parent menu with an appropriate    *
//* vertical position.                                                         *
//* parentMenu == index of parent menu of type dctMENUWIN to which sub-menus   *
//*               will be attached.                                            *
//* submenuList[] is an array of index numbers for the DialogMenuwin context   *
//* menu objects to be associated with each data item in the parent menu.      *
//* For data items with no associated sub-menu, use the default value of       *
//* MAX_DIALOG_CONTROLS. The list is terminated by a -1.                       *
//* Example: short submenuList[] = { 4, 5, MAX_DIALOG_CONTROLS, 9, -1 } ;      *
//*  This will yield the following associations:                               *
//*   Parent menu first data item  -> dctMENUWIN context menu (index 4)        *
//*   Parent menu second data item -> dctMENUWIN context menu (index 5)        *
//*   Parent menu third data item  -> not associated with a sub-menu           *
//*   Parent menu fourth data item -> dctMENUWIN context menu (index 9)        *
//*   (all other parent menu data items will remain unassociated)              *
//******************************************************************************

short NcDialog::AttachMenuwinSubmenus ( short pMenu, const short smList[] )
{
#define DEBUG_AMWS (0)
short    result = ERR ;

   //* Verify that the parent control is actually a Menuwin object *
   if ( dCtrl[pMenu]->type == dctMENUWIN )
   {
      //* Verify that parent menu is a visible (non-context) menu *
      //* and that each member of the list IS a context menu.     *
      DialogMenuwin* cp = (DialogMenuwin*)dCtrl[pMenu] ;

      #if DEBUG_AMWS != 0
      wchar_t b[32];
      nc.WriteString ( 2,1, "ORIG  LIST  NEWS", nc.bl | ncuATTR ) ;
      for ( short j = ZERO ; j < cp->bItems ; j++ )
      {
         swprintf ( b, 32, L"%02d", cp->subMenu[j] ) ;
         nc.WriteString ( 3+j, 2, b, nc.bl ) ;
         if ( smList[j] == -1 )
            break ;
      }
      for ( short j = ZERO ; smList[j] >= (-1) ; j++ )
      {
         swprintf ( b, 32, L"%02d",smList[j] ) ;
         nc.WriteString ( 3+j, 8, b, nc.bl ) ;
         if ( smList[j] == -1 )
            break ;
      }
      #endif   // DEBUG_AMWS

      //* Attach the sub-menus *
      result = OK ;                    // we can be hopeful
      for ( short i = ZERO ; smList[i] >= ZERO && i < cp->bItems && result==OK ; i++ )
      {
         short smIndex = smList[i] ;
         if ( smIndex <= this->lastCtrl && this->dCtrl[smIndex]->type == dctMENUWIN )
         {
            DialogMenuwin* cpt = (DialogMenuwin*)dCtrl[smIndex] ;
            if ( cpt->tCols == ZERO )
            {
               //* Store the context-menu index in the sub-menu array *
               cp->subMenu[i] = smIndex ;
            }
            else     // non-context menu specified
               result = ERR ;
         }
         else if ( smIndex != MAX_DIALOG_CONTROLS )
            result = ERR ;      // non-dctMENUWIN control or invalid index
      }
      #if DEBUG_AMWS != 0
      for ( short j = ZERO ; j < cp->bItems ; j++ )
      {
         swprintf ( b, 32, L"%02d", cp->subMenu[j] ) ;
         nc.WriteString ( 3+j, 14, b, nc.bl ) ;
         if ( smList[j] == -1 )
            break ;
      }
      #endif   // DEBUG_AMWS
   }
   return result ;

#undef DEBUG_AMWS
}  //* End AttachMenuwinSubmenus() *

//**************************
//*      HideMenuBar       *
//**************************
//******************************************************************************
//* Make a MenuBar (DialogMenuwin control group) invisible.                    *
//* Grouping must have been previously established through a call to           *
//* GroupMenuwinControls().                                                    *
//*                                                                            *
//* An application may wish to hide a MenuBar to provide additional screen     *
//* area, prevent user input, etc.                                             *
//*                                                                            *
//* After being hidden, the controls in the MenuBar cannot be accessed by the  *
//* Tab/ShiftTab sequence i.e. NextControl() / PrevControl() BUT they can      *
//* still be accessed and made visible again via the controls' respective      *
//* hotkeys (if any). If a valid hotkey for any member of the MenuBar is       *
//* detected, the MenuBar is made visible (see ShowMenuBar()) and the input    *
//* focus is moved to that control.                                            *
//*                                                                            *
//* Additionally, if caller specified an auxilliary hotkey for accessing the   *
//* MenuBar when defining the DialogMenuwin group, then detection of that      *
//* hotkey will toggle the visibility/invisibility of the MenuBar, and when    *
//* MenuBar becomes visible via the auxilliary hotkey, the first (lowest index)*
//* control in the MenuBar will get the input focus.                           *
//*                                                                            *
//* Note on input focus:                                                       *
//*  If this method is called while a member of the specified MenuBar has the  *
//*  input focus, we must move the focus to a control that is not a member of  *
//*  the group. At least one active control that DOES NOT belong to the MenuBar*
//*  must be defined for this dialog in order to make the MenuBar invisible,   *
//*  because input focus must be somewhere after the specified MenuBar         *
//*  disappears. Thus, we force the focus out of the MenuBar to another active *
//*  control before we hide the MenuBar.                                       *
//*  By enforcing this action we ensure that there is at least one active      *
//*  control outside the MenuBar. There is no guarantee that the application   *
//*  will be happy with our chosen focus position, so it is better if the      *
//*  application moves the focus out of the MenuBar BEFORE calling this        *
//*  method.                                                                   *
//*                                                                            *
//*  If the application wants to hide a MenuBar containing all the active      *
//*  controls in the dialog, there is a work-around. Define a dummy active     *
//*  Pushbutton control which is active, but does nothing. Such a Pushbutton   *
//*  can be effectively invisible if it is one character wide, and that        *
//*  character is a SPACE of the same color as the dialog background. The      *
//*  application would then move the focus to that control before hiding the   *
//*  MenuBar. The user would never know the extra Pushbutton control existed   *
//*                                                                            *
//* Input  : mbIndex: index of any member control in a MenuBar                 *
//*                                                                            *
//* Returns: OK if successful                                                  *
//*          ERR if:                                                           *
//*            1. mbIndex is not a valid control index                         *
//*            2. specified control is not a dctMENUWIN control                *
//*            3. specified control is not a member of a MenuBar               *
//*            4. all active controls in the dialog are members of the MenuBar *
//******************************************************************************

short NcDialog::HideMenuBar ( short mbIndex )
{
short    result = ERR ;

   //* Verify control type and if control is a member of a MenuBar.        *
   if (    mbIndex <= this->lastCtrl && this->dCtrl[mbIndex]->type == dctMENUWIN 
        && this->dCtrl[mbIndex]->groupCode != ZERO )
   {
      //* Move the input focus away from our MenuBar because An invisible  *
      //* control cannot have input focus.                                 *
      if ( this->dCtrl[this->currCtrl]->groupCode == this->dCtrl[mbIndex]->groupCode )
      {
         //* Move the input focus to the FIRST member of the group, then to    *
         //* the control before that. If this puts us outside the MenuBar, we  *
         //* can hide it. Else all active controls belong to the MenuBar, so   *
         //* we can't hide it.                                                 *
         this->emwFocus2GroupEnd ( false ) ;
         this->PrevControl () ;
         if ( this->dCtrl[this->currCtrl]->groupCode != this->dCtrl[mbIndex]->groupCode )
            result = OK ;
      }
      else
         result = OK ;

      //* Make each control in the MenuBar invisible *
      for ( short cI = ZERO ; cI <= this->lastCtrl && result == OK ; cI++ )
      {
         if ( this->dCtrl[cI]->groupCode == this->dCtrl[mbIndex]->groupCode )
         {
            DialogMenuwin* cpt = (DialogMenuwin*)this->dCtrl[cI] ;
            cpt->grpVisible = false ;     // MenuBar member is invisible
            cpt->active = false ;         // make the control inactive (cannot receive focus)
            //* Overwrite the space occupied by the control. It will not be    *
            //* refreshed because it is tagged as 'invisible'.                 *
            short offY = cpt->tulY - this->dulY,
                  offX = cpt->tulX - this->dulX ;
            for ( short i = ZERO ; i < cpt->tCols ; i++ )
               this->WriteChar ( offY, offX++, L' ', this->iColor ) ;
         }
      }
      this->RefreshWin () ;               // update the display
   }
   return result ;

}  //* End HideMenuBar() *

//************************
//*     ShowMenuBar      *
//************************
//******************************************************************************
//* For a MenuBar (group of DialogMenuwin controls) previously hidden by a     *
//* call to  HideMenuBar(), make the MenuBar visible.                          *
//* This method simulates an access via the (optional) auxilliary hotkey       *
//* specified when the MenuBar was established.                                *
//*                                                                            *
//* This is the public-access version of this method, called directly by the   *
//* application.                                                               *
//*                                                                            *
//* Note: This method will do nothing if:                                      *
//*       1. mbIndex is not a valid control index                              *
//*       2. the specified control is not a dctMENUWIN control                 *
//*       3. the specified control is not a member of a MenuBar                *
//*       4. If setFocus==true AND referenced MenuBar is already visible,      *
//*          input focus is simply moved to the control specified by mbIndex.  *
//*                                                                            *
//* Input  : mbIndex : index of any member control in a MenuBar                *
//*          setFocus: if 'true', immediately move the input focus to the      *
//*                      control indicated by mbIndex.                         *
//*                    if 'false', input focus is not changed                  *
//*                                                                            *
//* Returns: index of control with input focus.                                *
//******************************************************************************

short NcDialog::ShowMenuBar ( short mbIndex, bool setFocus )
{
short    focusControl = this->currCtrl ;

   //* Verify control index from caller, then make MeunBar visible *
   if ( mbIndex >= ZERO && mbIndex <= this->lastCtrl )
      focusControl = this->ShowMenuBar ( mbIndex ) ;

   //* Set input focus? *
   if ( setFocus != false )
   {
      //* Set the input focus *
      while ( this->currCtrl > focusControl )
         this->PrevControl () ;
      while ( this->currCtrl < focusControl )
         this->NextControl () ;
   }
   return this->currCtrl ;   

}  //* End ShowMenuBar() *

//**************************
//*      ShowMenuBar       *
//**************************
//******************************************************************************
//* Make a MenuBar(DialogMenuwin control group) visible.                       *
//* This is a private method, called only from within NcDialog class.          *
//*                                                                            *
//* The control that is RECOMMENDED to receive the input focus on return is    *
//* determined by the method used to access the MenuBar.                       *
//*  1. If user presses a hotkey associated with one of the individual controls*
//*     in the MenuBar, focus goes to that control (see IsHotkey()).           *
//*  2. If user presses the auxilliary hotkey (if any) specified when the      *
//*     MenuBar was established, the focus goes to the first (lowest index)    *
//*     control of the MenuBar.                                                *
//*                                                                            *
//* Input  : mbIndex: index of any member control in a MenuBar                 *
//*                                                                            *
//* Returns: index of control that SHOULD receive input focus i.e. mbIndex     *
//*                (focus is not changed in this method)                       *
//*            (if mbIndex does not reference a control which is a member )    *
//*            (of a MenuBar, then index of current control is returned   )    *
//******************************************************************************

short NcDialog::ShowMenuBar ( short mbIndex )
{
   //* Return value: index of control to receive input focus            *
   //* (in case of bad parameter, index of current control is returned) *
   short focusControl = this->currCtrl ;

   //* Verify control type and whether it is a member of a MenuBar      *
   if (    this->dCtrl[mbIndex]->type == dctMENUWIN 
        && this->dCtrl[mbIndex]->groupCode != ZERO )
   {
      //* This control is recommended to receive the input focus *
      focusControl = mbIndex ;

      //* If the specified control's group is currently hidden *
      DialogMenuwin* cp = (DialogMenuwin*)this->dCtrl[mbIndex] ;
      if ( cp->grpVisible == false )
      {
         //* Return all group members to the active, non-hidden state *
         for ( short cI = ZERO ; cI <= this->lastCtrl ; cI++ )
         {
            if ( this->dCtrl[cI]->groupCode == cp->groupCode )
            {
               DialogMenuwin* cpt = (DialogMenuwin*)this->dCtrl[cI] ;
               cpt->grpVisible = cpt->active = true ; // make control visible and active
               cpt->RedrawControl ( false ) ;         // update display
            }
         }
      }
   }
   return focusControl ;

}  //* End ShowMenuBar() *

//*************************
//*    MenuBarVisible     *
//*************************
//******************************************************************************
//* Returns current state of a MenuBar, visible or invisible.                  *
//*                                                                            *
//* Input  : mbIndex : index of any member control in a MenuBar                *
//* Returns: 'true' if specified MenuBar is visible                            *
//*          'false' if MenuBar is invisible OR                                *
//*            1. mbIndex is not a valid control index                         *
//*            2. specified control is not a dctMENUWIN control                *
//*            3. specified control is not a member of a MenuBar               *
//******************************************************************************

bool NcDialog::MenuBarVisible ( short mbIndex )
{
   bool visible = false ;
   if (    mbIndex >= ZERO && mbIndex <= this->lastCtrl 
        && this->dCtrl[mbIndex]->type == dctMENUWIN )
   {
      DialogMenuwin* cp = (DialogMenuwin*)this->dCtrl[mbIndex] ;
      if ( cp->groupCode > ZERO )
         visible = cp->grpVisible ;
   }
   return visible ;

}  //* End MenuBarVisible() *

//************************
//*  SetActiveMenuItems  *
//************************
//******************************************************************************
//* Set which menu items in a dctMENUWIN control are active (user-selectable). *
//* These changes are allowed only for a control that currently does not have  *
//* the input focus.                                                           *
//*                                                                            *
//* A dctMENUWIN control has a fixed number of menu items determined at the    *
//* time the control is instantiated; however, some menu items may have no     *
//* meaning or must be disallowed in certain contexts. Menu items may therefore*
//* be activated or deactivated as necessary, and the item's color attribute   *
//* can be adjusted to visually indicate the item's current status.            *
//*                                                                            *
//* Note on beautification: If the 'active' menu color uses a 'reverse'        *
//* attribute, then the 'inactive' color should also use a 'reverse' attribute *
//* to avoid confusion with the color of the highlighted item. For an example  *
//* of using this method, please see Dialog1 test application, Test10.         *
//*                                                                            *
//* This method can also be used to change the color attribute for each menu   *
//* item with-or-without affecting the active/inactive status of each item.    *
//*                                                                            *
//* Input  : cIndex : index of dctMenuwin control                              *
//*        : aFlags : an array of boolean values, one for each menu item       *
//*                   'true' == active, 'false' == inactive                    *
//*                   Note: At least one menu item must be 'active'.           *
//*                         (To deactivate an entire menu, see ControlActive())*
//*          aColors: (optional, NULL pointer by default)                      *
//*                   if specified, pointer to an array of color attributes,   *
//*                   one for each menu item.                                  *
//*          hide   : (optional, false by default)                             *
//*                   if 'true' hide (make invisible) all inactive menu items  *
//*                   [NOTE: 'hide' option is not currently implemented]       *
//*                                                                            *
//* Returns: OK if successful                                                  *
//*          ERR if:                                                           *
//*              1. cIndex is not a valid control index                        *
//*              2. cIndex does not reference a control of dctMENUWIN type     *
//*              3. target control has input focus                             *
//******************************************************************************

short NcDialog::SetActiveMenuItems ( short cIndex, const bool aFlags[], 
                                     const attr_t* aColors, bool hide )
{
   short status = ERR ;

   //* Verify that target is in fact a menu-win control AND *
   //* that it does not currently have the input focus.     *
   if ( cIndex >= ZERO && cIndex <= this->lastCtrl && 
        this->dCtrl[cIndex]->type == dctMENUWIN && cIndex != this->currCtrl )
   {
      //* Get access to the control's methods and data members *
      DialogMenuwin* cp = (DialogMenuwin*)this->dCtrl[cIndex] ;
      bool someActive = false ;  // caller must have at least one active item
      for ( short i = ZERO ; i < cp->bItems ; i++ )
      {
         cp->bActive[i] = aFlags[i] ;
         if ( cp->bActive[i] != false )
            someActive = true ;
         if ( aColors != NULL )
            cp->bColor[i] = aColors[i] ;
      }
      if ( ! someActive )
         cp->bActive[ZERO] = true ;
      status = OK ;
   }
   return status ;

}  //* End SetActiveMenuItems() *

//************************
//*  SetActiveMenuItems  *
//************************
//******************************************************************************
//* Set new text strings for a menu. Optionally, also set the active/inactive  *
//* flags and color attributes for the menu items.                             *
//*                                                                            *
//* Important Notes:                                                           *
//*   a) There must be exactly the same number of replacement-text strings as  *
//*      the existing number of items.                                         *
//*   b) Each replacement string must contain exactly the same number of       *
//*      _columns_ (not necessarily the same number of characters) as the      *
//*      string it replaces.                                                   *
//*                                                                            *
//* Input  : cIndex : index of dctMENUWIN control                              *
//*          mnuText: array of display strings, one for each menu item         *
//*        : aFlags : (optional, NULL pointer by default)                      *
//*                   an array of boolean values, one for each menu item       *
//*                   'true' == active, 'false' == inactive                    *
//*                   Note: At least one menu item must be 'active'.           *
//*                         (To deactivate an entire menu, see ControlActive())*
//*          aColors: (optional, NULL pointer by default)                      *
//*                   if specified, pointer to an array of color attributes,   *
//*                   one for each menu item.                                  *
//* Returns: OK if successful                                                  *
//*          ERR if:                                                           *
//*              1. cIndex is not a valid control index                        *
//*              2. cIndex does not reference a control of dctMENUWIN type     *
//*              3. target control has input focus                             *
//******************************************************************************
//* Programmer's Note: Because application input is notoriously unreliable     *
//* we do our best to avoid array overruns, but if caller screwed up badly,    *
//* we're going to crash and burn while parsing the text.                      *
//******************************************************************************

short NcDialog::SetActiveMenuItems ( short cIndex, const char* mnuText, 
                                     const bool* aFlags, const attr_t* aColors )
{
   short status = ERR ;

   //* Verify that target is in fact a menu-win control AND *
   //* that it does not currently have the input focus, AND *
   //* that caller is actually pointing at SOMETHING.       *
   if ( (cIndex >= ZERO) && (cIndex <= this->lastCtrl) && 
        (this->dCtrl[cIndex]->type == dctMENUWIN && cIndex != this->currCtrl) &&
        (mnuText != NULL) )
   {
      status = OK ;                 // provisional success

      //* Get access to the control's methods and data members *
      //* and create a temporary buffer.                       *
      DialogMenuwin* cp = (DialogMenuwin*)this->dCtrl[cIndex] ;
      wchar_t tmpStrings[cp->bItems][gsALLOCDFLT] ;

      //* Parse the text strings ensuring that they are the same number of *
      //* columns as the existing text.                                    *
      gString gsOld, gsNew ;
      short tndx = ZERO ;
      for ( short indx = ZERO ; indx < cp->bItems ; ++indx )
      {
         gsOld = cp->bText[indx] ;
         gsNew = &mnuText[tndx] ;
         tndx += gsNew.gschars() ;        // index the next string
         if ( (gsNew.gscols()) == (gsOld.gscols()) )
         {
            gsNew.copy( tmpStrings[indx], gsALLOCDFLT ) ;
         }
         else
         {
            status = ERR ;
            break ;
         }
      }
      if ( status == OK )
      {
         //* The formatted strings overwrite the existing strings,*
         //* and the text pointer array is reinitialized.         *
         char* dString = (cp->bText[ZERO]) ;
         tndx = ZERO ;
         short ubytes ;
         for ( short indx = ZERO ; indx < cp->bItems ; ++indx )
         {
            gsNew = tmpStrings[indx] ;
            ubytes = gsNew.utfbytes() ;
            gsNew.copy( dString, ubytes ) ;
            cp->bText[indx] = dString ;
            dString += ubytes ;
         }
      }

      //* If we are to modify color attributes or active/inactive flags *
      if ( aFlags != NULL || aColors != NULL )
         status = this->SetActiveMenuItems ( cIndex, aFlags, aColors ) ;
   }
   return status ;

}  //* End SetActiveMenuItems() *

//************************
//* SetMenuwinBorderAttr *
//************************
//******************************************************************************
//* Set the border colors for the Menuwin control when it is in the expanded   *
//* state. When the menu is expanded and has the input focus, the border will  *
//* be drawn in the 'focus' color. When the menu is expanded but a sub-menu    *
//* has the input focus, the border will be drawn in the 'nonfocus' color.     *
//*            (Note that the border is not visible when )                     *
//*            (the control is in the collapsed state.   )                     *
//*                                                                            *
//* The Menuwin control's border colors (both with and without focus) are set  *
//* when the control is instantiated. However, with certain color schemes,     *
//* custom border color attributes, or swapped focus/non-focus attributes can  *
//* more easily be interpreted by the user.                                    *
//*                                                                            *
//* Input  : cIndex : index of dctMENUWIN control                              *
//*          swap   : exchange the focus and non-focus border colors           *
//*                   if 'true',  swap the focus and non-focus attributes      *
//*                               ('fAttr' and 'nfAttr' parameters ignored)    *
//*                   if 'false', use the border attributes specified by the   *
//*                               'fAttr' and 'nfAttr' parameters              *
//*          fAttr  : (optional, ZERO by default) border color attribute       *
//*                     when control has input focus                           *
//*                     (ignored when 'swap' is set)                           *
//*          nfAttr : (optional, ZERO by default) border color attribute       *
//*                     when control does not have input focus                 *
//*                     (ignored when 'swap' is set)                           *
//*                                                                            *
//* Returns: OK if successful                                                  *
//*          ERR if:                                                           *
//*              1. cIndex is not a valid control index                        *
//*              2. cIndex does not reference a control of dctMENUWIN type     *
//*              3. target control has input focus                             *
//******************************************************************************

short NcDialog::SetMenuwinBorderAttr ( short cIndex, bool swap, 
                                       attr_t fAttr, attr_t nfAttr )
{
   short status = ERR ;

   //* Verify that target is in fact a menu-win control AND *
   //* that it does not currently have the input focus.     *
   if ( (cIndex >= ZERO) && (cIndex <= this->lastCtrl) && 
        (this->dCtrl[cIndex]->type == dctMENUWIN && cIndex != this->currCtrl) )
   {
      status = OK ;                 // provisional success

      //* Get access to the control's methods and data members *
      DialogMenuwin* cp = (DialogMenuwin*)this->dCtrl[cIndex] ;

      //* If color-attribute swap *
      if ( swap )
      {
         attr_t swapAttr = cp->fBorder ;
         cp->fBorder = cp->nBorder ;
         cp->nBorder = swapAttr ;
      }

      //* If new color attributes *
      else
      {
         cp->fBorder = fAttr ;
         cp->nBorder = nfAttr ;
      }
   }
   return status ;

}  //* End SetMenuwinBorderAttr() *

//************************
//*                      *
//************************
//********************************************************************************
//*                                                                              *
//*                                                                              *
//*                                                                              *
//* Input  :                                                                     *
//*                                                                              *
//* Returns:                                                                     *
//********************************************************************************


