//********************************************************************************
//* File       : NcdControlPB.cpp                                                *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2006-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in NcDialog.hpp            *
//* Date       : 12-Feb-2021                                                     *
//* Version    : (see NcDialogVersion string in NcDialog.cpp)                    *
//*                                                                              *
//* Description: Contains the methods of the DialogPushbutton class              *
//*              and the NcDialog class EditPushbutton() method.                 *
//*              See NcDialog.hpp for the definition of this class.              *
//*                                                                              *
//*                                                                              *
//* Development Tools: See NcDialog.cpp.                                         *
//********************************************************************************
//* Version History (most recent first):                                         *
//*   See version history in NcDialog.cpp.                                       *
//********************************************************************************

//*****************
//* Include Files *
//*****************

#include "GlobalDef.hpp"               //* General definitions

#ifndef NCURSES_INCLUDED
#include "NCurses.hpp"
#endif

#ifndef NCDIALOG_INCLUDED
#include "NcDialog.hpp"
#endif


      //*************************************************
      //** THIS SECTION IMPLEMENTS THE METHODS OF THE  **
      //**           DialogPushbutton CLASS.           **
      //** - - - - - - - - - - - - - - - - - - - - - - **
      //** See below for public NcDialog-class methods **
      //**  related to DialogPushbutton-class access.  **
      //*************************************************

//*************************
//*   DialogPushbutton    *
//*************************
//******************************************************************************
//* Constructor for DialogPushbutton class object.                             *
//*                                                                            *
//* Input  : pointer to initialization structure                               *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

DialogPushbutton::DialogPushbutton ( InitCtrl* iPtr )
{
   this->type   = dctPUSHBUTTON ;   // set control type
   this->ulY    = iPtr->ulY ;       // control's offset within the dialog
   this->ulX    = iPtr->ulX ;
   this->lines  = iPtr->lines ;     // control's dimensions
   this->cols   = iPtr->cols ;
   this->nColor = iPtr->nColor ;    // non-focus color
   this->fColor = iPtr->fColor ;    // focus color
   this->active = iPtr->active ;    // can control receive input focus?

   //* Copy the display text to data member, wLabel and *
   //* initialize the bHotkey. (allow multi-line label) *
   this->InitHotkey ( iPtr->dispText, false ) ;

   //* Default values for all other data members *
   this->labY = this->labX = ZERO ; // pushbuttons do not have labels
   this->groupCode = ZERO ;         // pushbuttons are not grouped (not used)
   this->bStyle = (ncLineType)ZERO ;// by default, control has no border
   this->rtlContent = false ;       // left-to-right language content by default

   //* Instantiate the underlying window object *
   wPtr = new NcWindow ( this->lines, this->cols, this->ulY, this->ulX ) ;
   
}  //* End DialogPushbutton() *

//*************************
//*   ~DialogPushbutton   *
//*************************
//******************************************************************************
//* Destructor for DialogPushbutton class object.                              *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

DialogPushbutton::~DialogPushbutton ( void )
{
   //* Close the underlying window *
   delete this->wPtr ;

}  //* End ~DialogPushbutton() *

//*************************
//*     OpenControl       *
//*************************
//******************************************************************************
//* Draw a dialog button control i.e. open the window previously               *
//* instantiated by a call to DialogPushbutton().                              *
//*                                                                            *
//* Input  : hasFocus : optional boolean, false by default, determines         *
//*                     whether to draw the object in nColor or fColor         *
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
//******************************************************************************

short DialogPushbutton::OpenControl ( bool hasFocus )
{
short    result ;

   if ( (result = this->wPtr->OpenWindow ()) == OK )
   {
      this->RedrawControl ( hasFocus ) ;
   }
   return result ;

}  //* End OpenControl() *

//*************************
//*     RedrawControl     *
//*************************
//******************************************************************************
//* Redraw the data for a Pushbutton control.                                  *
//* (refreshes the control's window)                                           *
//*                                                                            *
//* Input  : hasFocus: if true, control has focus                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void DialogPushbutton::RedrawControl ( bool hasFocus )
{

   this->RedrawControl ( hasFocus, true ) ;

}  //* End RedrawControl()

//*************************
//*     RedrawControl     *
//*************************
//******************************************************************************
//* Redraw the data for a Pushbutton control.                                  *
//*                                                                            *
//* NOTE: Called directly only by members of the DialogPushbutton class.       *
//*                                                                            *
//* Input  : hasFocus: if true, control has focus                              *
//*          refresh : if true, refresh display                                *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void DialogPushbutton::RedrawControl ( bool hasFocus, bool refresh )
{
   attr_t color = hasFocus ? fColor : nColor ;

   this->wPtr->SetInteriorColor ( color ) ;
   this->wPtr->ClearWin () ;
   winPos wp( ZERO, ZERO ) ;
   if ( this->bStyle != ZERO )
   {
      this->wPtr->DrawBorder ( color, NULL, this->bStyle ) ;
      ++wp.ypos ;                   // move text inside the border
      ++wp.xpos ;
   }

   //* RTL data starts from right edge of control *
   if ( this->rtlContent != false )
      wp.xpos = this->cols - (this->bStyle == ZERO ? 1 : 2) ;

   this->wPtr->WriteParagraph ( wp, this->wLabel, color, false, this->rtlContent ) ;
   if ( this->bHotkey.hotkey != false )
   {
      short xpos = this->rtlContent ? (this->cols - this->bHotkey.xoffset - 1) : 
                                      this->bHotkey.xoffset ;
      this->wPtr->WriteChar ( this->bHotkey.yoffset, xpos, 
                              this->bHotkey.hotchar, color | ncuATTR ) ;
   }
   if ( refresh != false )
      this->wPtr->RefreshWin () ;

}  //* End RedrawControl()

//*************************
//*    SetOutputFormat    *
//*************************
//******************************************************************************
//* Set the text data output format: RTL (right-to-left), or                   *
//* LTR (left-to-right). Does not refresh display.                             *
//* Note that the default (unmodified) output format is LTR.                   *
//*                                                                            *
//* Input  : rtlFormat : if 'true',  then set as RTL format                    *
//*                      if 'false', then set as LTR format                    *
//*                                                                            *
//* Returns: OK  if successful                                                 *
//*          ERR if data format cannot be changed (currently not used)         *
//******************************************************************************
//* Programmer's Note: When redrawing the control's contents, we assume that   *
//* control does not have focus. This could be incorrect, but since we don't   *
//* update the display, the user will never know it.                           *
//******************************************************************************

short DialogPushbutton::SetOutputFormat ( bool rtlFormat )
{

   this->rtlContent = rtlFormat ;
   this->RedrawControl ( false, false ) ; // force data update
   return OK ;

}  //* End SetOutputFormat() *



      //*************************************************
      //**     THIS SECTION IMPLEMENTS THE PUBLIC,     **
      //**    NcDialog-class METHODS FOR ACCESSING     **
      //**    DialogPushbutton-class FUNCTIONALITY.    **
      //*************************************************

//*************************
//*    EditPushbutton     *
//*************************
//******************************************************************************
//* If control with input focus == dctPUSHBUTTON, call this method to get      *
//* user's key input. Returns when a Pushbutton has been been pressed (nckENTER*
//* or nckSPACE), when Pushbutton IS READY to lose focus (nckTAB or nckSTAB)   *
//* -OR- has lost focus due to a hotkey press.                                 *
//* The nckESC (Escape key) is a special case. We don't know what it means in  *
//* this context (probably user panic), but caller may want to handle it.      *
//*                                                                            *
//* Input  : uiInfo class (by reference) - initial values ignored              *
//*                                                                            *
//* Returns: index of control that currently has the input focus               *
//*           (See note in NcDialog.hpp about interpretation )                 *
//*           (of values returned in the uiInfo-class object.)                 *
//******************************************************************************
//* Note that the interpretation of navigation keys:                           *
//* nckTAB, nckRIGHT and nckDOWN cause a move to the next control,             *
//* while nckSTAB, nckLEFT and nckUP cause a move to the previous control.     *
//* While this is entirely logical, it may not be entirely intuitive to the    *
//* user, since the physical layout of the controls on the screen may not      *
//* coincide with these directions. This is a minor, but annoying design issue *
//* to be kept in mind when positioning your controls within the dialog window.*
//******************************************************************************

short NcDialog::EditPushbutton ( uiInfo& info )
{
   bool  done = false ;                // loop control

   //* Protect ourselves from application calling incorrect method *
   if ( this->dCtrl[this->currCtrl]->type != dctPUSHBUTTON )
      done = true ;

   //**************************
   //* Interact with the user *
   //**************************
   while ( ! done )
   {
      //* 1) Get user input.                                  *
      //* 2) Check for toggle of Insert/Overstrike mode.      *
      //* 3) Check for terminal-resize event.                 *
      //* 4) Access callback method, if specified.            *
      //* 5) If input has been handled, return to top of loop.*
      this->GetKeyInput ( info.wk ) ;

      if ( (info.wk.type == wktFUNKEY)
            && (info.wk.key == nckINSERT || info.wk.key == nckRESIZE) )
      {
         if ( info.wk.key == nckINSERT )
            this->ToggleIns () ;
         else
            this->TermResized () ;

         if ( ExternalControlUpdate != NULL )
            ExternalControlUpdate ( this->currCtrl, info.wk, false ) ;
         continue ;
      }

      //* Request to move to next control in list *
      if ( (info.wk.type == wktFUNKEY) && 
           (info.wk.key == nckTAB || info.wk.key == nckDOWN || info.wk.key == nckRIGHT) )
      {
         info.ctrlType = dctPUSHBUTTON ;     // 
         info.ctrlIndex = this->currCtrl ;   // control with input focus
         info.hasFocus = true ;              // control has focus
         info.dataMod = false ;              // pushbutton not pressed
         info.isSel = false ;                // not pressed
         info.keyIn = nckTAB ;               // move forward to next control
         info.selMember = MAX_DIALOG_CONTROLS ; // don't care
         info.viaHotkey = false ;            // control did not get focus via hotkey
         done = true ;                       // exit the input loop
      }

      //* Request to move to previous control in list *
      else if ( (info.wk.type == wktFUNKEY) && 
                (info.wk.key == nckSTAB || info.wk.key == nckUP || info.wk.key == nckLEFT) )
      {
         info.ctrlType = dctPUSHBUTTON ;     // 
         info.ctrlIndex = this->currCtrl ;   // control with input focus
         info.hasFocus = true ;              // control has focus
         info.dataMod = false ;              // pushbutton not pressed
         info.isSel = false ;                // not pressed
         info.keyIn = nckSTAB ;              // move backward to previous control
         info.selMember = MAX_DIALOG_CONTROLS ; // don't care
         info.viaHotkey = false ;            // control did not get focus via hotkey
         done = true ;                       // exit the input loop
      }
      else if (   (info.wk.type == wktPRINT && info.wk.key == SPACE)
               || (info.wk.type == wktFUNKEY && 
                   (info.wk.key == nckENTER || info.wk.key == nckpENTER)) )
      {
         //* Pushbutton was pressed *
         info.ctrlType = dctPUSHBUTTON ;     // 
         info.ctrlIndex = this->currCtrl ;   // control with input focus
         info.hasFocus = true ;              // control has focus
         info.dataMod = true ;               // pushbutton has been pressed
         info.isSel = true ;                 // pressed
         info.keyIn = ZERO ;                 // don't care
         info.selMember = MAX_DIALOG_CONTROLS ; // don't care
         info.viaHotkey = false ;            // control did not get focus via hotkey
         done = true ;                 // exit the input loop
      }
      else if ( info.wk.type == wktFUNKEY && info.wk.key == nckESC )
      {
         //* User is trying (unsuccessfully) to tell us something. *
         info.ctrlType = dctPUSHBUTTON ;     // 
         info.ctrlIndex = this->currCtrl ;   // control with input focus
         info.hasFocus = true ;              // control has focus
         info.dataMod = false ;              // pushbutton not pressed
         info.isSel = false ;                // not pressed
         info.keyIn = nckTAB ;               // return input to caller
         info.selMember = MAX_DIALOG_CONTROLS ; // don't care
         info.viaHotkey = false ;            // control did not get focus via hotkey
         done = true ;                 // exit the input loop
      }
      else     // test for a possible hotkey
      {
         //* Scan the controls for one whose hotkey matches the input.*
         //* If match found, returns control's index, else -1.        *
         short hotIndex = this->IsHotkey ( info.wk ) ;
         if ( hotIndex >= ZERO )
         {
            //* If user has pressed THIS button via hotkey *
            //* treat it an an Enter key                   *
            if ( hotIndex == this->currCtrl )
            {
               info.ctrlType = dctPUSHBUTTON ;
               info.ctrlIndex = this->currCtrl ;// control with input focus
               info.hasFocus = true ;        // control has focus
               info.dataMod = true ;         // pushbutton has been pressed
               info.isSel = true ;           // pressed
               info.keyIn = ZERO ;           // don't care
               info.selMember = MAX_DIALOG_CONTROLS ; // don't care
               info.viaHotkey = false ;      // control did not get focus via hotkey
               done = true ;                 // exit the input loop
            }
            else
            {
               //* Control activated via hotkey was not this control *
               info.ctrlType = dctPUSHBUTTON ;
               info.ctrlIndex = this->currCtrl ;// control with input focus
               info.hasFocus = true ;        // this may be reset by ChangedFocusViaHotkey()
               info.dataMod = false ;        // pushbutton not pressed
               info.isSel = false ;          // not pressed
               info.keyIn = ZERO ;           // direction focus moves n/a
               info.selMember = MAX_DIALOG_CONTROLS ; // don't care
               info.viaHotkey = false ;      // this may be set by ChangedFocusViaHotkey()
               done = true ;                 // exit the input loop

               //* Make the indicated control the current/active control *
               this->ChangedFocusViaHotkey ( info.wk.key, hotIndex, info ) ;
               // on return, new control has the focus, and the info.h_XX
               // variables adjusted
            }
         }
         else
         {
            // key input value is not valid in this context, ignore it
         }
      }

      //* If caller has specified a callback method, do it now.*
      if ( ExternalControlUpdate != NULL )
         ExternalControlUpdate ( this->currCtrl, info.wk, false ) ;
   }  // while(!done)

   return this->currCtrl; 

}  //* End EditPushbutton() *

//*************************
//*   SetPushbuttonText   *
//*************************
//******************************************************************************
//* Set the display text and optionally the color attributes for a             *
//* dctPUSHBUTTON control, and refresh the control's display.                  *
//*                                                                            *
//* This is useful for changing the text each time the button is pressed,      *
//* for example:  'ENABLE' vs. 'DISABLE'.                                      *
//* NOTE: If you specify a 'hotkey' in the text string for the control,        *
//*       good style would indicate that a given control should always have    *
//*       the same hotkey (this is not enforced, but is highly recommended).   *
//*                                                                            *
//* Input  : cIndex: index into array of dialog controls                       *
//*          pbText: pointer to new text data                                  *
//*          fAttr : (optional) if specified, replaces the color attribute for *
//*                  the control when it has input focus                       *
//*          nAttr : (optional) if specified, replaces the color attribute for *
//*                  the control when it does not have input focus             *
//*                                                                            *
//* Returns: OK if successful,                                                 *
//*          ERR if cIndex does not reference a dctPUSHBUTTON control          *
//******************************************************************************
//* Programmer's Note: Any previously-assigned hotkey will be discarded by the *
//* call to InitHotkey(). If it is not replaced by the call, then the control  *
//* will have NO hotkey and will therefore not respond to mouse input. For this*
//* reason, we save the prevously-assigned _default_ hotkey (if any), and if a *
//* new hotkey is not assigned, we restore the default hotkey.                 *
//* We DO NOT manipulate user-supplied hotkeys.                                *
//******************************************************************************

short NcDialog::SetPushbuttonText ( short cIndex, const char* pbText,
                                    attr_t fAttr, attr_t nAttr )
{
short    success = OK ;

   if ( cIndex <= this->lastCtrl && this->dCtrl[cIndex]->type == dctPUSHBUTTON )
   {
      DialogPushbutton* cp = (DialogPushbutton*)(this->dCtrl[cIndex]) ;
      wchar_t dflthk = NULLCHAR ;
      if ( cp->bHotkey.hotdflt != false ) // if an existing _default_ hotkey
         dflthk = cp->bHotkey.hotchar ;   // save it
      cp->InitHotkey ( pbText, false ) ;
      if ( (cp->bHotkey.hotkey == false) && (dflthk != NULLCHAR) )
      {
         cp->bHotkey.hotchar = dflthk ;
         cp->bHotkey.hotdflt = true ;
      }
      if ( fAttr != attrDFLT )
         cp->fColor = fAttr ;
      if ( nAttr != attrDFLT )
         cp->nColor = nAttr ;
      cp->RedrawControl ( bool(cIndex == currCtrl ? true : false) ) ;
   }
   else
      success = ERR ;
   return success ;

}  //* End SetPushbuttonText() *

//**************************
//* EnablePushbuttonBorder *
//**************************
//******************************************************************************
//* Enable display of a border around a dctPUSHBUTTON control.                 *
//*  - By default, pushbutton controls have no border.                         *
//*  - Pushbutton controls defined with less than three (3) display lines      *
//*    or less than three (3) columns NEVER have a border.                     *
//*  - A border for pushbutton controls defined with >= 3 display lines/columns*
//*    MAY be specified.                                                       *
//*  - Display text for the control should reflect whether the border is       *
//*    activated. With border, the display text is automagically offset to     *
//*    avoid overwriting the border.                                           *
//*    Example: For a 3-line, 8-column control with "OK" centered in it:       *
//*             a) no border  : const char* txt = "        \n   OK   " ;       *
//*             b) with border: const char* txt = "  OK  " ;                   *
//*  - Aesthetically, if the border is to be enabled, it should happen before  *
//*    the first call to RefreshWin().                                         *
//*  - See test application Dialog1, Test07 for further explanation.           *
//*                                                                            *
//* Input  : cIndex: index into array of dialog controls                       *
//*          style : (optional, 'ncltSINGLE' by default)                       *
//*                  If specified, border style is a member of enum ncLineType.*
//*                                                                            *
//* Returns: OK if successful,                                                 *
//*          ERR if cIndex does not reference a dctPUSHBUTTON control          *
//*              OR if control is too small to have a border                   *
//******************************************************************************

short NcDialog::EnablePushbuttonBorder ( short cIndex, ncLineType style )
{
   short success = ERR ;

   if ( cIndex <= this->lastCtrl && this->dCtrl[cIndex]->type == dctPUSHBUTTON )
   {
      DialogPushbutton* cp = (DialogPushbutton*)(this->dCtrl[cIndex]) ;
      if ( cp->lines >= 3 && cp->cols >= 3 )
      {
         cp->bStyle = style ;
         if ( cp->bHotkey.hotkey != false )
         {  //* If a hotkey defined, adjust its display position.*
            ++cp->bHotkey.yoffset ;
            ++cp->bHotkey.xoffset ;
         }
         cp->RedrawControl ( bool(cIndex == this->currCtrl) ) ;
         success = OK ;
      }
   }
   return success ;

}  //* End EnablePushbuttonBorder() *

//************************
//*                      *
//************************
//********************************************************************************
//*                                                                              *
//*                                                                              *
//*                                                                              *
//* Input  :                                                                     *
//*                                                                              *
//* Returns:                                                                     *
//********************************************************************************

