//********************************************************************************
//* File       : NcdControlRB.cpp                                                *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2006-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in NcDialog.hpp            *
//* Date       : 23-Mar-2021                                                     *
//* Version    : (see NcDialogVersion string in NcDialog.cpp)                    *
//*                                                                              *
//* Description: Contains the methods of the DialogRadiobutton class             *
//*              and the public NcDialog-class methods for accessing             *
//*              the DialogRadiobutton object's functionality.                   *
//*              See NcDialog.hpp for the definition of this class.              *
//*                                                                              *
//*              The NcDialog::DisplayRadiobuttonTypes() method is also          *
//*              located here.                                                   *
//*                                                                              *
//* Development Tools: See NcDialog.cpp.                                         *
//********************************************************************************
//* Version History (most recent first):                                         *
//*   See version history in NcDialog.cpp.                                       *
//********************************************************************************

//*****************
//* Include Files *
//*****************

#include "GlobalDef.hpp"               //* General definitions

#ifndef NCURSES_INCLUDED
#include "NCurses.hpp"
#endif

#ifndef NCDIALOG_INCLUDED
#include "NcDialog.hpp"
#endif

//*********************
//* Local Definitions *
//*********************
class RB_DisplayString
{
public:
   short    rbSelOffset ;     //* Offset into string to position selection char
   short    rbColumns ;       //* Display columns used
   wchar_t  rbString[MAX_RADIO_WIDTH] ;//* Display string (without selection char)
   wchar_t  rbSelchar ;       //* Selection character
} ;

// CZONE - Replace wcsDIAMOND with a 1-column diamond.
//* Radio Button display data for each member of RBType *
RB_DisplayString RB_Strings[rbtTYPES] = 
{
   { 0, 1, { L" " }, wcsSDIAMOND },          //* rbtS1
   { 1, 3, { L"< >" }, wcsSDIAMOND },        //* rbtS3a
   { 1, 3, { L"[ ]" }, wcsSDIAMOND },        //* rbtS3s
   { 1, 3, { L"( )" }, wcsSDIAMOND },        //* rbtS3p
   { 2, 5, { L"<   >" }, wcsSDIAMOND },      //* rbtS5a
   { 2, 5, { L"[   ]" }, wcsSDIAMOND },      //* rbtS5s
   { 2, 5, { L"(   )" }, wcsSDIAMOND },      //* rbtS5p
   { 0, 1, { L" " }, L'#' },                 //* rbtC1
   { 0, 2, { L"  " }, L'#' },                //* rbtC2
   { 1, 3, { L"   " }, L'#' },               //* rbtC3
   { 1, 4, { L"    " }, L'#' },              //* rbtC4
   { 2, 5, { L"     " }, L'#' },             //* rbtC5
   { 2, 6, { L"      " }, L'#' },            //* rbtC6
} ;


      //*************************************************
      //** THIS SECTION IMPLEMENTS THE METHODS OF THE  **
      //**          DialogRadiobutton CLASS.           **
      //** - - - - - - - - - - - - - - - - - - - - - - **
      //** See below for public NcDialog-class methods **
      //**   related to DialogRadiobutton-class access.**
      //*************************************************

//*************************
//*   DialogRadiobutton   *
//*************************
//******************************************************************************
//* Constructor for DialogRadiobutton class object.                            *
//*                                                                            *
//*                                                                            *
//* Input  : pointer to initialization structure                               *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

DialogRadiobutton::DialogRadiobutton ( InitCtrl* iPtr )
{
   this->type    = dctRADIOBUTTON ;    // set control type
   if ( iPtr->rbSubtype >= rbtS1 && iPtr->rbSubtype < rbtTYPES )
      this->subtype = iPtr->rbSubtype ;// remember the display format
   else
      this->subtype = rbtS3a ;         // set default display format
   this->ulY     = iPtr->ulY ;         // position of control in parent window
   this->ulX     = iPtr->ulX ;
   this->lines   = 1 ;                 // radio buttons have only one line
   this->nColor  = iPtr->nColor ;      // control background color
   this->fColor  = iPtr->fColor ;      // color of 'select' character
   this->labY    = iPtr->labY ;        // label offsets
   this->labX    = iPtr->labX ;
   this->selected = iPtr->rbSelect ;   // initial state of 'selected' flag
   this->active  = iPtr->active ;      // whether control can be edited by user

   //* Default values for all other data members *
   this->groupCode = ZERO ;            // initially not a member of an exclusive-OR group
   this->rtlContent = false ;          // left-to-right language content by default

   //* Copy the label text to data member, wLabel *
   //* and initialize bHotkey.                    *
   this->InitHotkey ( iPtr->label, false ) ;

   //* Set default display string and 'select' character for the button type.  *
   //*      (Note that we are using an enum as an index, so be wise.)          *
   wcsncpy ( this->wText, RB_Strings[this->subtype].rbString, MAX_RADIO_WIDTH ) ;
   this->selChar = RB_Strings[this->subtype].rbSelchar ;
   this->cols = RB_Strings[this->subtype].rbColumns ;

   //* For Custom radio buttons, get caller's display string *
   if ( this->subtype == rbtC1 || this->subtype == rbtC2 || 
        this->subtype == rbtC3 || this->subtype == rbtC4 || 
        this->subtype == rbtC5 || this->subtype == rbtC6 )
   {
      gString gs( iPtr->dispText ) ;
      short tLen = gs.gschars() - 1,
            tWid = gs.gscols() ;
      //* If caller's custom string is the length it's supposed to be, *
      //* save the 'select' character, then write the string WITHOUT   *
      //* the 'select' character.                                      *
      // Programmer's Note: This is a really good place for the application 
      // to screw us with bad data, so beware.
      if ( this->subtype == rbtC1 && tLen == 1 && tWid == 1 )
         this->selChar = *gs.gstr() ;
      else if ( this->subtype == rbtC2 && tLen == 1 && tWid == 2 )
         this->selChar = *gs.gstr() ;
      else if ( this->subtype == rbtC3 && tLen == 3 && tWid == 3 )
      {
         this->selChar = gs.gstr()[1] ;
         wchar_t a = gs.gstr()[0], b = gs.gstr()[2] ;
         gs.compose( "%C %C", &a, &b ) ;
         gs.copy( this->wText, MAX_RADIO_WIDTH ) ;
      }
      else if ( this->subtype == rbtC4 && tLen == 3 && tWid == 4 )
      {
         wchar_t a = gs.gstr()[0], b = gs.gstr()[2] ;
         gs.shiftChars ( -1 ) ;
         gs.limitChars ( 1 ) ;
         if ( gs.gscols() == 2 )
         {
            this->selChar = *gs.gstr() ;
            gs.compose( "%C  %C", &a, &b ) ;
            gs.copy( this->wText, MAX_RADIO_WIDTH ) ;
         }
      }
      else if ( this->subtype == rbtC5 && tWid == 5 )
      {
         if ( tLen == 5 )     // 5, 1-column characters
         {
            wchar_t a = gs.gstr()[0], b = gs.gstr()[1], c = gs.gstr()[2], 
                    d = gs.gstr()[3], e = gs.gstr()[4] ;
            this->selChar = c ;
            gs.compose( "%C%C %C%C", &a, &b, &d, &e ) ;
            gs.copy( this->wText, MAX_RADIO_WIDTH ) ;
         }
         else if ( tLen == 3 )   // 2-column, 1-column, 2-column 
         {
            wchar_t a = gs.gstr()[0], b = gs.gstr()[1], c = gs.gstr()[2] ;
            gs.compose( "%C", &b ) ;
            if ( gs.gscols() == 1 )
            {
               this->selChar = b ;
               gs.compose( "%C %C", &a, &c ) ;
               gs.copy( this->wText, MAX_RADIO_WIDTH ) ;
            }
         }
      }
      else if ( this->subtype == rbtC6 && tLen == 3 && tWid == 6 )
      {
         this->selChar = gs.gstr()[1] ;
         wchar_t a = gs.gstr()[0], b = gs.gstr()[2] ;
         gs.compose( "%C  %C", &a, &b ) ;
         gs.copy( this->wText, MAX_RADIO_WIDTH ) ;
      }
   }
   
   //* Instantiate the underlying window object *
   wPtr = new NcWindow ( lines, cols, ulY, ulX ) ;
   
}  //* End DialogRadiobutton() *

//*************************
//*  ~DialogRadiobutton   *
//*************************
//******************************************************************************
//* Destructor for DialogRadiobutton class object.                             *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

DialogRadiobutton::~DialogRadiobutton ( void )
{
   //* Close the underlying window *
   delete this->wPtr ;

}  //* End ~DialogRadiobutton() *

//*************************
//*     OpenControl       *
//*************************
//******************************************************************************
//* Draw a dialog radio button control i.e. open the window previously         *
//* instantiated by a call to DialogRadiobutton().                             *
//*                                                                            *
//* Input  : hasFocus : optional boolean, false by default, determines         *
//*                     whether to draw the object in nColor or fColor         *
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
//******************************************************************************

short DialogRadiobutton::OpenControl ( bool hasFocus )
{
short    result ;

   if ( (result = wPtr->OpenWindow ()) == OK )
   {
      this->RedrawControl ( hasFocus ) ;     // display radio button text
   }
   return result ;

}  //* End OpenControl() *

//*************************
//*     RedrawControl     *
//*************************
//******************************************************************************
//* Redraw the data in a Radio Button window. Because the color selection is   *
//* rather complex, we isolate this code sequence here.                        *
//* (refreshes the control's window)                                           *
//*                                                                            *
//* Input  : hasFocus: if true, button has focus                               *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note: see notes in RB_Strings[] definitions and in the        *
//* DisplayAltCharacterSet() method in NcDialog.cpp.                           *
//*                                                                            *
//******************************************************************************

void DialogRadiobutton::RedrawControl ( bool hasFocus )
{
attr_t   color = hasFocus ? this->fColor : this->nColor ;

   this->wPtr->WriteString ( ZERO, ZERO, this->wText, color ) ;
   if ( this->selected )                  // if radio button is 'selected'
      wPtr->WriteChar ( ZERO, RB_Strings[this->subtype].rbSelOffset, this->selChar, color ) ;

   //* For radiobutton subtype 'rbtC1' only:, if character is     *
   //* "wcsFISHEYE" (0x25C9) or wcsDIAMONDS (0x25C8), then        *
   //* for the 'reset' state, substitute wcsWHITE_CIRCLE (0x25CB) *
   //* or wcsWDIAMOND (0x25C7), respectively.                     *
   else if ( (this->subtype == rbtC1) &&
             ((this->selChar == wcsFISHEYE) || (this->selChar == wcsDIAMONDS)) )
   {
      wPtr->WriteChar ( ZERO, RB_Strings[this->subtype].rbSelOffset, 
                        (this->selChar == wcsFISHEYE ? wcsWCIRCLE : wcsWDIAMOND), color ) ;
   }

   this->wPtr->RefreshWin () ;            // refresh the control's underlying window
   
}  //* End RedrawControl()

//*************************
//*    SetOutputFormat    *
//*************************
//******************************************************************************
//* Set the text data output format: RTL (right-to-left), or                   *
//* LTR (left-to-right). Does not refresh display.                             *
//* Note that the default (unmodified) output format is LTR.                   *
//*                                                                            *
//* Input  : rtlFormat : if 'true',  then set as RTL format                    *
//*                      if 'false', then set as LTR format                    *
//*                                                                            *
//* Returns: ERR: request is not applicable to this control type               *
//******************************************************************************

short DialogRadiobutton::SetOutputFormat ( bool rtlFormat )
{
   //* Does not apply to dctRADIOBUTTON controls. *
   return ERR ;

}  //* End SetOutputFormat() *



      //*************************************************
      //**     THIS SECTION IMPLEMENTS THE PUBLIC,     **
      //**    NcDialog-class METHODS FOR ACCESSING     **
      //**    DialogRadiobutton-class FUNCTIONALITY.   **
      //*************************************************

//*************************
//*    EditRadiobutton    *
//*************************
//******************************************************************************
//* If control with input focus == dctRADIOBUTTON, call this method to get     *
//* user's key input. Allows user to edit the state of an independent Radio    *
//* Button or the selection within a Radio Button group. Returns when the      *
//* 'selected' state of the button or group has changed (nckENTER or nckSPACE),*
//* when button or group IS READY to lose the input focus (nckTAB or nckSTAB)  *
//* ('selected' state may or may not have changed) -OR- button or group has    *
//* lost focus due to a hotkey press.                                          *
//*                                                                            *
//* Input  : uiInfo class (by reference) - initial values ignored              *
//*                                                                            *
//* Returns: index of control that currently has the input focus               *
//*           (See note in NcDialog.hpp about interpretation )                 *
//*           (of values returned in the uiInfo-class object.)                 *
//******************************************************************************

short NcDialog::EditRadiobutton ( uiInfo& info )
{
   //* Protect ourselves from application calling incorrect method *
   if ( this->dCtrl[this->currCtrl]->type == dctRADIOBUTTON )
   {
      DialogRadiobutton* cp = (DialogRadiobutton*)this->dCtrl[this->currCtrl] ;

      if ( cp->groupCode == ZERO )              // individual radio button
         this->EditSingleRadio ( info ) ;
      else                                      // button is member of a group
         this->EditGroupRadio ( info ) ;
   }
   else                  // control is not a radio button i.e. application error
   {
      //* Do what we can to minimize the damage *
      info.ctrlType = dCtrl[currCtrl]->type ;
      info.ctrlIndex = this->currCtrl ;
      info.keyIn = nckTAB ;               // move on to next control
      info.hasFocus = true ;              // control has focus
      info.dataMod = false ;              // no data modified
      info.selMember = MAX_DIALOG_CONTROLS ; // don't care
      info.isSel = false ;                // don't care
      info.viaHotkey = false ;            // invalidate any hotkey data
   }
   
   return this->currCtrl; 

}  //* End EditRadiobutton() *

//*************************
//*    EditSingleRadio    *
//*************************
//******************************************************************************
//* User interface for editing an independent Radio Button.                    *
//*                                                                            *
//*                                                                            *
//* Input  : uiInfo class (by reference) - initial values ignored              *
//*           (see note in NcDialog.hpp, about values returned)                *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void NcDialog::EditSingleRadio ( uiInfo& info )
{
   //* Get a pointer to the control object that has input focus *
   DialogRadiobutton *cp = (DialogRadiobutton*)this->dCtrl[this->currCtrl] ;

   //**************************
   //* Interact with the user *
   //**************************
   bool  done = false ;          // loop control
   while ( ! done )
   {
      //* 1) Get user input.                                  *
      //* 2) Check for toggle of Insert/Overstrike mode.      *
      //* 3) Check for terminal-resize event.                 *
      //* 4) Access callback method, if specified.            *
      //* 5) If input has been handled, return to top of loop.*
      this->GetKeyInput ( info.wk ) ;

      if ( (info.wk.type == wktFUNKEY)
            && (info.wk.key == nckINSERT || info.wk.key == nckRESIZE) )
      {
         if ( info.wk.key == nckINSERT )
            this->ToggleIns () ;
         else
            this->TermResized () ;

         if ( ExternalControlUpdate != NULL )
            ExternalControlUpdate ( this->currCtrl, info.wk, false ) ;
         continue ;
      }

      //* Request to move to next control in list *
      if ( (info.wk.type == wktFUNKEY) && 
           (info.wk.key == nckTAB || info.wk.key == nckDOWN || 
            info.wk.key == nckRIGHT || info.wk.key == nckESC) )
      {
         info.ctrlType = dctRADIOBUTTON ;    // 
         info.ctrlIndex = this->currCtrl ;   // control with input focus
         info.hasFocus = true ;              // focus was not lost
         info.dataMod = false ;              // no change in state
         info.keyIn = nckTAB ;               // focus to move forward to next control
         info.selMember = MAX_DIALOG_CONTROLS ; // not a group member
         info.isSel = cp->selected ;         // current state of flag
         info.viaHotkey = false ;            // control did not get focus via hotkey
         done = true ;                       // exit the input loop
      }

      //* Request to move to previous control in list *
      else if ( (info.wk.type == wktFUNKEY) && 
                (info.wk.key == nckSTAB || info.wk.key == nckUP || info.wk.key == nckLEFT) )
      {
         info.ctrlType = dctRADIOBUTTON ;    // 
         info.ctrlIndex = this->currCtrl ;   // control with input focus
         info.hasFocus = true ;              // focus was not lost
         info.dataMod = false ;              // no change in state
         info.keyIn = nckSTAB ;              // focus to move backward to previous control
         info.selMember = MAX_DIALOG_CONTROLS ; // not a group member
         info.isSel = cp->selected ;         // current state of flag
         info.viaHotkey = false ;            // control did not get focus via hotkey
         done = true ;                       // exit the input loop
      }

      //* Change of state: nckENTER sets, SPACE toggles *
      else if (   (info.wk.type == wktPRINT && info.wk.key == SPACE)
               || (info.wk.type == wktFUNKEY && 
                   (info.wk.key == nckENTER || info.wk.key == nckpENTER)) )
      {
         info.ctrlType = dctRADIOBUTTON ;    // 
         info.ctrlIndex = this->currCtrl ;   // control with input focus
         info.hasFocus = true ;              // focus was not lost
         bool origState = cp->selected ;
         if ( info.wk.key == nckENTER || info.wk.key == nckpENTER )  // set the 'selected' state
            cp->selected = true ;            // update internal tracking
         else                                // toggle the 'selected' state
            cp->selected = cp->selected ? false : true ;
         info.isSel = cp->selected ;         // current state of flag
         info.dataMod = ((cp->selected==origState) ? false : true) ; // state change?
         info.keyIn = nckTAB ;               // focus to move forward to next ontrol
         info.selMember = MAX_DIALOG_CONTROLS ; // not a group member
         info.viaHotkey = false ;            // control did not get focus via hotkey
         done = true ;                       // exit the input loop
      }
      else     // test for a possible hotkey
      {
         //* Scan the controls for one whose hotkey matches the input.*
         //* If match found, returns control's index, else -1.        *
         short hotIndex = this->IsHotkey ( info.wk ) ;
         if ( hotIndex >= ZERO )
         {
            //* If user has 'selected' THIS button via hotkey *
            //* treat it an an Enter key                      *
            if ( hotIndex == this->currCtrl )
            {
               info.ctrlType = dctRADIOBUTTON ; // 
               info.ctrlIndex = this->currCtrl ;// control with input focus
               info.hasFocus = true ;           // focus was not lost
               //* If selection was via a mouse event, then toggle state. *
               if ( info.wk.mevent.conv != false )
               {
                  info.dataMod = true ; // data changed
                  cp->selected = cp->selected ? false : true ;
                  info.keyIn = nckTAB ; // direction focus moves (forces a refresh)
               }
               //* If selection was via application-defined *
               //* hotkey, then set the state.              *
               else
               {
                  info.dataMod = (cp->selected ? false : true ) ; // data changed??
                  cp->selected = true ;      // radio buttons are 'selected' via hotkey
                  info.keyIn = ZERO ;        // direction focus moves n/a
               }
               info.isSel = cp->selected ;      // current state of flag
               info.selMember = MAX_DIALOG_CONTROLS ; // not a group member
               info.viaHotkey = false ;         // control did not get focus via hotkey
               done = true ;                    // exit the input loop
            }
            else
            {
               //* (this radio button not selected or toggled) *
               info.ctrlType = dctRADIOBUTTON ; // 
               info.ctrlIndex = this->currCtrl ;// control with input focus
               info.hasFocus = true ;           // this may be reset by ChangedFocusViaHotkey()
               info.dataMod = false ;           // no change in state
               info.keyIn = nckTAB ;            // focus to move forward to next control
               info.selMember = MAX_DIALOG_CONTROLS ; // not a group member
               info.isSel = cp->selected ;      // current state of flag
               info.viaHotkey = false ;         // this may be set by ChangedFocusViaHotkey()
               done = true ;                    // exit the input loop

               //* Make the indicated control the current/active control *
               this->ChangedFocusViaHotkey ( info.wk.key, hotIndex, info ) ;
               // on return, new control has the focus, and the info.h_XX
               // variables adjusted
            }
         }
         else
         {
            // key input value is not valid in this context, ignore it
         }
      }
      //* If caller has specified a callback method, do it now.*
      if ( ExternalControlUpdate != NULL )
         ExternalControlUpdate ( this->currCtrl, info.wk, false ) ;
   }  // while(!done)
}  //* End EditSingleRadio() *

//*************************
//*    EditGroupRadio     *
//*************************
//******************************************************************************
//* User interface for editing the members of a Radio Button group.            *
//*                                                                            *
//* Input  : uiInfo class (by reference) - initial values ignored              *
//*           (see note in NcDialog.hpp, about values returned)                *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void NcDialog::EditGroupRadio ( uiInfo& info )
{
   //* Get a pointer to the control object that has input focus *
   DialogRadiobutton *cpt = (DialogRadiobutton*)this->dCtrl[this->currCtrl] ;

   // Remember which button of the group was "selected" on entry *
   short origSelMember = this->GetRbGroupSelection ( this->currCtrl ) ;
   short origGroupCode = cpt->groupCode ;

   //**************************
   //* Interact with the user *
   //**************************
   bool  done = false ;          // loop control
   while ( ! done )
   {
      //* 1) Get user input.                                  *
      //* 2) Check for toggle of Insert/Overstrike mode.      *
      //* 3) Check for terminal-resize event.                 *
      //* 4) Access callback method, if specified.            *
      //* 5) If input has been handled, return to top of loop.*
      this->GetKeyInput ( info.wk ) ;

      if ( (info.wk.type == wktFUNKEY)
            && (info.wk.key == nckINSERT || info.wk.key == nckRESIZE) )
      {
         if ( info.wk.key == nckINSERT )
            this->ToggleIns () ;
         else
            this->TermResized () ;

         if ( ExternalControlUpdate != NULL )
            ExternalControlUpdate ( this->currCtrl, info.wk, false ) ;
         continue ;
      }
      
      //* Request to move to next control in list *
      if ( (info.wk.type == wktFUNKEY) && 
           (info.wk.key == nckTAB || info.wk.key == nckDOWN || info.wk.key == nckRIGHT) )
      {
         //* Determine whether next control is a member of this button group *
         if (   ((this->currCtrl < this->lastCtrl) 
                && (this->dCtrl[this->currCtrl+1]->type == dctRADIOBUTTON))
             || ((this->currCtrl == this->lastCtrl) 
                && (this->dCtrl[ZERO]->type == dctRADIOBUTTON)) )
         {
            cpt = (DialogRadiobutton*)this->dCtrl[this->currCtrl==this->lastCtrl ? 
                                                  ZERO : this->currCtrl+1] ;
            if ( cpt->groupCode == origGroupCode )
            {
               this->NextControl () ;     // advance to next member of this button group
            }
            else
            {
               //* Otherwise the next control is not a member of the group,*
               //* Return to caller with no change of state in the group   *
               info.ctrlType = dctRADIOBUTTON ; // 
               info.ctrlIndex = this->currCtrl ;// control with input focus
               info.hasFocus = true ;           // focus was not lost
               info.dataMod = false ;           // no change in selected member
               info.keyIn = nckTAB ;            // focus to move forward out of group
               info.selMember = origSelMember ; // 'selected' member of the group
               info.isSel = false ;             // don't care
               info.viaHotkey = false ;         // control did not get focus via hotkey
               done = true ;              // returning to caller
            }
         }
         else
         {
            //* Otherwise the next control is not a radio button.     *
            //* Return to caller with no change of state in the group *
            info.ctrlType = dctRADIOBUTTON ;    // 
            info.ctrlIndex = this->currCtrl ;   // control with input focus
            info.hasFocus = true ;              // focus was not lost
            info.dataMod = false ;              // no change in selected member
            info.keyIn = nckTAB ;               // focus to move forward out of group
            info.selMember = origSelMember ;    // 'selected' member of the group
            info.isSel = false ;                // don't care
            info.viaHotkey = false ;            // control did not get focus via hotkey
            done = true ;                 // returning to caller
         }
      }

      //* Request to move to previous control in list *
      else if ( (info.wk.type == wktFUNKEY) && 
                (info.wk.key == nckSTAB || info.wk.key == nckUP || info.wk.key == nckLEFT) )
      {
         //* Determine whether previous control is a member of this button group *
         if (   ((this->currCtrl > ZERO) 
                && (this->dCtrl[this->currCtrl-1]->type == dctRADIOBUTTON))
             || ((this->currCtrl == ZERO) 
                && (this->dCtrl[this->lastCtrl]->type == dctRADIOBUTTON)) )
         {
            cpt = (DialogRadiobutton*)this->dCtrl[this->currCtrl==ZERO ? 
                                            this->lastCtrl : this->currCtrl-1] ;
            if ( cpt->groupCode == origGroupCode )
            {
               this->PrevControl () ;     // move to previous member of this button group
            }
            else
            {
               //* Otherwise the next control is not a member of the group,*
               //* Return to caller with no change of state in the group   *
               info.ctrlType = dctRADIOBUTTON ; // 
               info.ctrlIndex = this->currCtrl ;// control with input focus
               info.hasFocus = true ;           // focus was not lost
               info.dataMod = false ;           // no change in selected member
               info.keyIn = nckSTAB ;           // focus to move backward out of group
               info.selMember = origSelMember ; // 'selected' member of the group
               info.isSel = false ;             // don't care
               info.viaHotkey = false ;         // control did not get focus via hotkey
               done = true ;              // returning to caller
            }
         }
         else
         {
            //* Otherwise the next control is not a radio button.     *
            //* Return to caller with no change of state in the group *
            info.ctrlType = dctRADIOBUTTON ;    // 
            info.ctrlIndex = this->currCtrl ;   // control with input focus
            info.hasFocus = true ;              // focus was not lost
            info.dataMod = false ;              // no change in selected member
            info.keyIn = nckSTAB ;              // focus to move backward out of group
            info.selMember = origSelMember ;    // 'selected' member of the group
            info.isSel = false ;                // don't care
            info.viaHotkey = false ;            // control did not get focus via hotkey
            done = true ;                 // returning to caller
         }
      }
      
      //* Activate radio button that has focus *
      else if (   (info.wk.type == wktPRINT && info.wk.key == SPACE)
               || (info.wk.type == wktFUNKEY && 
                   (info.wk.key == nckENTER || info.wk.key == nckpENTER)) )
      {
         //* Set control as 'selected', then update and redraw other    *
         //* members of the group, moving focus to last (highest index) *
         //* member of the group (prepares group to lose focus)         *
         //* (Members of an XOR group may not be de-selected.)          *
         //* if new 'selected' group member is different from           *
         //* previous 'selected' member, indicate the change            *
         info.ctrlType = dctRADIOBUTTON ;       // 
         info.hasFocus = true ;                 // focus was not lost
         //* 'true' if selection changed *
         info.dataMod = ((this->currCtrl==origSelMember) ? false : true) ;
         info.selMember = this->currCtrl ;      // 'selected' member of the group
         cpt = (DialogRadiobutton*)this->dCtrl[this->currCtrl] ;
         cpt->selected = true ;                 // set active member as 'selected'
         info.keyIn = nckENTER ;                // focus to move forward out of group
         info.isSel = false ;                   // don't care
         info.viaHotkey = false ;               // control did not get focus via hotkey
         this->RadioXorUpdate ( this->currCtrl ) ;// update all members of the group
         // NOTE: Last member of group now has the input focus
         info.ctrlIndex = this->currCtrl ;      // control with input focus
         done = true ;                          // exit the input loop
      }
      else if ( info.wk.type == wktFUNKEY && info.wk.key == nckESC ) // abort selection
      {
         //* return all group members to initial state *
         this->RadioXorUpdate ( origSelMember ) ;

         info.ctrlType = dctRADIOBUTTON ;       // 
         info.ctrlIndex = this->currCtrl ;      // control with input focus
         info.hasFocus = true ;                 // focus was not lost
         info.dataMod = false ;                 // no change in selected member
         info.keyIn = nckTAB ;                  // focus to move forward out of group
         info.selMember = origSelMember ;       // 'selected' member of the group
         info.isSel = false ;                   // don't care
         info.viaHotkey = false ;               // control did not get focus via hotkey
         done = true ;                          // returning to caller
      }
      else     // test for a possible hotkey
      {
         //* Scan the controls for one whose hotkey matches the input.*
         //* If match found, returns control's index, else -1.        *
         short hotIndex = this->IsHotkey ( info.wk ) ;
         if ( hotIndex >= ZERO )
         {
            //* We are currently still within the radio button group.*
            //* If the hotkey references a member of this group,     *
            //* treat it as an Enter key                             *
            bool  hotGroup = true ;    // being overly cautious in this test
            if ( this->dCtrl[hotIndex]->type != dctRADIOBUTTON )
               hotGroup = false ;
            else
            {
               cpt = (DialogRadiobutton*)this->dCtrl[hotIndex] ;
               if ( cpt->groupCode != origGroupCode )
                  hotGroup = false ;
            }
            if ( hotGroup != false )
            {
               info.ctrlType = dctRADIOBUTTON ; // 
               info.hasFocus = true ;           // focus was not lost
               info.dataMod = ((hotIndex==origSelMember) ? false : true) ; // true if selection changed
               info.selMember = hotIndex ;      // 'selected' member of the group
               info.isSel = false ;             // don't care
               info.keyIn = nckENTER ;          // focus to move forward out of group
               info.viaHotkey = false ;         // control did not get focus via hotkey
               cpt = (DialogRadiobutton*)this->dCtrl[hotIndex] ;
               cpt->selected = true ;           // set active member as 'selected'
               this->RadioXorUpdate ( hotIndex ) ; // update all members of the group
               // NOTE: Last member of group now has the input focus
               info.ctrlIndex = this->currCtrl ;// control with input focus
               done = true ;                    // exit the input loop
            }
            else
            {
               //* Hotkey-selected control is not a member of this group *
               info.ctrlType = dctRADIOBUTTON ; // 
               info.ctrlIndex = this->currCtrl ;// control with input focus
               info.hasFocus = true ;           // this may be reset by ChangedFocusViaHotkey()
               info.dataMod = false ;           // no change in selected member
               info.keyIn = ZERO ;              // direction focus moves n/a
               info.selMember = origSelMember ; // 'selected' member of the group
               info.isSel = false ;             // don't care
               info.viaHotkey = false ;         // this may be set by ChangedFocusViaHotkey()
               done = true ;                 // exit the edit loop

               //* Make the indicated control the current/active control *
               this->ChangedFocusViaHotkey ( info.wk.key, hotIndex, info ) ;
               // on return, new control has the focus, and the info.h_XX
               // variables adjusted
            }
         }
         else
         {
            // key input value is not valid in this context, ignore it
         }
      }
      //* If caller has specified a callback method, do it now.*
      if ( ExternalControlUpdate != NULL )
         ExternalControlUpdate ( this->currCtrl, info.wk, false ) ;
   }     // while()

}  //* End EditGroupRadio() *

//*************************
//*   GroupRadiobuttons   *
//*************************
//******************************************************************************
//* Establish a group of DialogRadiobutton class objects as an exclusive-OR    *
//* group. Within this group, exactly ONE radio button may be 'selected' by    *
//* the user at any time.                                                      *
//*                                                                            *
//* Parameter is an array of index numbers for the  DialogRadiobutton objects  *
//* in the group. List is terminated by a -1.                                  *
//*   Example: short List[] = { 4, 5, 6, -1 }                                  *
//*                                                                            *
//* Programmer's Note: If specified indices are not consecutive, anomolous     *
//* display behavior may result.                                               *
//*                                                                            *
//* Input  : array of control index numbers (-1 ends list)                     *
//*                                                                            *
//* Returns: OK if successful                                                  *
//*          ERR returned if:                                                  *
//*              1. invalid index number in list                               *
//*              2. referenced control is not a dctRADIOBUTTON object          *
//******************************************************************************

short NcDialog::GroupRadiobuttons ( short indexList[] )
{
DialogRadiobutton* cp ;
short    lIndex = ZERO,
         status = ERR ;

   while ( indexList[lIndex] >= ZERO && lIndex <= this->lastCtrl )
   {
      if ( this->dCtrl[indexList[lIndex]]->type == dctRADIOBUTTON )
      {
         cp = (DialogRadiobutton*)dCtrl[indexList[lIndex]] ;
         cp->groupCode = this->cgGroupCode ;
         ++lIndex ;
         status = OK ;
      }
      else
      {
         status = ERR ;
         break ;
      }
   }

   if ( status == OK )
      ++this->cgGroupCode ;
   return status ;

}  //* End GroupRadiobuttons() *

//************************
//* SetRadiobuttonState  *
//************************
//******************************************************************************
//* Set or reset the 'selected' flag of the specified, independent Radio       *
//* Button control, and refresh the control's display.                         *
//*                                                                            *
//* Input  : cIndex: index into array of dialog controls                       *
//*          select: new state of 'selected' flag                              *
//*                                                                            *
//* Returns: OK  if successful                                                 *
//*          ERR if:                                                           *
//*              a. specified control is a member of a radiobutton group       *
//*              b. invalid control index                                      *
//*              c. referenced control is not a dctRADIOBUTTON object          *
//******************************************************************************

short NcDialog::SetRadiobuttonState ( short cIndex, bool select )
{
short    success = ERR ;

   if ( cIndex <= this->lastCtrl && this->dCtrl[cIndex]->type == dctRADIOBUTTON )
   {
      char gCode = this->dCtrl[cIndex]->groupCode ;
      if ( gCode == ZERO )    // control not member of a radiobutton group
      {
         DialogRadiobutton* cp = (DialogRadiobutton*)(this->dCtrl[cIndex]) ;
         cp->selected = select ;
         cp->RedrawControl ( bool(cIndex == this->currCtrl ? true : false) ) ;
         success = OK ;
      }
   }
   return success ;

}  //* End SetRadiobuttonState() *

//*************************
//*  GetRadiobuttonState  *
//*************************
//******************************************************************************
//* Return the current state of the 'selected' flag of the specified Radio     *
//* Button control.                                                            *
//*                                                                            *
//* Input  : cIndex: index into array of dialog controls                       *
//*          select: receives current state of 'selected' flag                 *
//*                  (by reference)                                            *
//*                                                                            *
//* Returns: OK  if successful                                                 *
//*          ERR if:                                                           *
//*              a. invalid control index                                      *
//*              b. referenced control is not a dctRADIOBUTTON object          *
//******************************************************************************

short NcDialog::GetRadiobuttonState ( short cIndex, bool& select )
{
short    success = OK ;

   if ( cIndex <= this->lastCtrl && this->dCtrl[cIndex]->type == dctRADIOBUTTON )
   {
      DialogRadiobutton* cp = (DialogRadiobutton*)(this->dCtrl[cIndex]) ;
      select = cp->selected ;
   }
   else
      success = ERR ;
   return success ;

}  //* End GetRadiobuttonState() *

//*************************
//*     RadioXorUpdate    *
//*************************
//******************************************************************************
//* Update the 'selected' status of all members of the radio button group      *
//* of which the specified button is a member. It is assumed that the          *
//* specified control IS a dctRADIOBUTTON and that it has just been            *
//* 'selected', because EditRadiobutton() does not allow de-selection of an    *
//* exclusive-OR group member.                                                 *
//*                                                                            *
//* Radio button groups are identified by the groupCode variable.              *
//*                                                                            *
//* If specified control is a member of a radio button group, then on return,  *
//* focus will be on the highest-index control in the group. Else, focus       *
//* unchanged.                                                                 *
//*                                                                            *
//* Input  : rbIndex: index of radio button whose state has just changed       *
//*          updateFocus: (optional, true by default)                          *
//*                   if true, place focus on last member of group             *
//*                   if false, focus is unchanged                             *
//*                                                                            *
//* Returns: OK                                                                *
//******************************************************************************

short NcDialog::RadioXorUpdate ( short rbIndex, bool updateFocus )
{
short    largestIndex = rbIndex,
         status = OK ;

   //* If specified control is member of an XOR group *
   if ( this->dCtrl[rbIndex]->groupCode != ZERO )
   {
      char gCode = this->dCtrl[rbIndex]->groupCode ; // group we're working with
      
      //* Scan all controls in the dialog *
      // Programmer's Note: We COULD begin the loop with i==(rbIndex+1),
      // but that would require tricker tests. Simple is better than Tricky,
      // and this is a user interface, so we have plenty of time.
      for ( short i = ZERO ; i <= this->lastCtrl ; i++ )
      {
         if ( this->dCtrl[i]->groupCode == gCode )
         {
            //* Determine highest control index in the group *
            largestIndex = i > largestIndex ? i : largestIndex ;

            DialogRadiobutton* cpt = (DialogRadiobutton*)this->dCtrl[i] ;
            if ( i == rbIndex )           // if this is the new 'selected' control
               cpt->selected = true ;     // (not technically necessary, but safer)
            else                          // another member of the group
               cpt->selected = false ;    // de-select
            cpt->RedrawControl ( (bool)(i == this->currCtrl) ) ;
         }
      }
      //* Advance the focus to the last control in the the XOR group.*
      if ( updateFocus != false )
      {
         while ( this->currCtrl < largestIndex )
            this->NextControl () ;
      }
   }
   return status ;

}  //* End RadioXorUpdate() *

//*************************
//*  GetRbGroupSelection  *
//*************************
//******************************************************************************
//* For the radio-button group of which the specified radio button control     *
//* is a member, determine which button in the group is currently 'selected'.  *
//* Note that exactly one member of a radio button group is 'selected'.        *
//*                                                                            *
//* Input  : cIndex : index of any dctRADIOBUTTON in the target button group   *
//*                                                                            *
//* Returns: index of selected member of group                                 *
//*          returns ERR if specified control is not a member of a             *
//*           radio-button group                                               *
//*            OR specified control is not of type dctRADIOBUTTON              *
//******************************************************************************

short NcDialog::GetRbGroupSelection ( short cIndex )
{
short    selIndex = ERR ;

   if ( cIndex <= this->lastCtrl && this->dCtrl[cIndex]->type == dctRADIOBUTTON )
   {
      char gCode = this->dCtrl[cIndex]->groupCode ;
      if ( gCode != ZERO )    // control must be a member of a radio button group
      {
         //* Scan the control list for the member of this group *
         //* whose 'selected' bit is set.                       *
         for ( short i = ZERO ; i <= this->lastCtrl ; i++ )
         {
            if ( this->dCtrl[i]->type == dctRADIOBUTTON )
            {
               DialogRadiobutton* cpt = (DialogRadiobutton*)this->dCtrl[i] ;
               if ( cpt->groupCode == gCode && cpt->selected != false )
               {
                  selIndex = i ;
                  break ;
               }
            }
         }
      }
   }
   return selIndex ;

}  //* End GetRbGroupSelection() *

//*************************
//*  SetRbGroupSelection  *
//*************************
//******************************************************************************
//* Set the 'selected' control for a Radio Button group. The previously-       *
//* 'selected' group member will be reset.                                     *
//*                                                                            *
//*                                                                            *
//* Input  : cIndex : index of control to be set as 'selected'                 *
//*                                                                            *
//* Returns: index of 'selected' group member                                  *
//*          returns ERR if:                                                   *
//*           1. specified control is not of type dctRADIOBUTTON               *
//*           2. specified control is not a member of a radiobutton group      *
//******************************************************************************

short NcDialog::SetRbGroupSelection ( short cIndex )
{
short    selIndex = ERR ;

   if ( cIndex <= this->lastCtrl && this->dCtrl[cIndex]->type == dctRADIOBUTTON )
   {
      char gCode = this->dCtrl[cIndex]->groupCode ;
      if ( gCode != ZERO )    // control must be a member of a radio button group
      {  //* Set the control as 'selected' *
         DialogRadiobutton* cpt = (DialogRadiobutton*)this->dCtrl[cIndex] ;
         cpt->selected = true ;

         //* Update the group tracking variables for new 'selected' *
         //* member and re-display the Radio Button                 *
         this->RadioXorUpdate ( cIndex, false ) ;
         selIndex = this->currCtrl ;   // return index of 'selected' member 
      }
   }
   return selIndex ;

}  //* End SetRbGroupSelection() *

//*****************************
//*  DisplayRadiobuttonTypes  *
//*****************************
//******************************************************************************
//* Software-development tool. Displays the standard formats available for     *
//* radio buttons.                                                             *
//*                                                                            *
//* Input  : wPos : screen coordinates for display                             *
//*          baseColor: display color for all but 'select' character           *
//*          selColor : display color for 'select' character                   *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void NcDialog::DisplayRadiobuttonTypes ( winPos wPos, attr_t baseColor, attr_t selColor )
{
#if ENABLE_DEVELOPMENT_METHODS != 0
const short dLines = 18, dCols = 46 ;
const wchar_t cjkSHI = L'是',        // U+662F ('shi' ) == 'yes'
              cjkLBRACKET = L'【',   // U+3010 (2-column left bracket)
              cjkRBRACKET = L'】',   // U+3011 (2-column right bracket)
              wcsMEDSQUARE = L'◼' ; // U+25FC (medium square)

   //* Save parent dialog, just in case *
   this->CaptureDialogDisplayData () ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dLines,         // number of display lines
                       dCols,          // number of display columns
                       wPos.ypos,      // Y offset from upper-left of terminal 
                       wPos.xpos,      // X offset from upper-left of terminal
                       " Display Radio Button Types ", // dialog title
                       ncltSINGLE,     // border line-style
                       baseColor,      // border color attribute
                       baseColor,      // interior color attribute
                       NULL            // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow ()) == OK )
   {
      //* Display the data and then wait for a keypress *
      winPos   wp( 2, 0 ) ;

      //* rbtS1 *
      wp = dp->WriteString ( wp.ypos, 2, RB_Strings[rbtS1].rbString, baseColor ) ;
      dp->WriteChar ( wp.ypos, 2+RB_Strings[rbtS1].rbSelOffset, RB_Strings[rbtS1].rbSelchar, selColor ) ;
      dp->WriteString ( wp.ypos, wp.xpos, "     rbtS1  - Standard, 1 columns wide", iColor ) ;
      ++wp.ypos ;

      //* rbtS3a *
      wp = dp->WriteString ( wp.ypos, 2, RB_Strings[rbtS3a].rbString, baseColor ) ;
      dp->WriteChar ( wp.ypos, 2+RB_Strings[rbtS3a].rbSelOffset, RB_Strings[rbtS3a].rbSelchar, selColor ) ;
      dp->WriteString ( wp.ypos, wp.xpos, "   rbtS3a - Standard, 3 columns wide", iColor ) ;
      ++wp.ypos ;

      //* rbtS3s *
      wp = dp->WriteString ( wp.ypos, 2, RB_Strings[rbtS3s].rbString, baseColor ) ;
      dp->WriteChar ( wp.ypos, 2+RB_Strings[rbtS3s].rbSelOffset, RB_Strings[rbtS3s].rbSelchar, selColor ) ;
      dp->WriteString ( wp.ypos, wp.xpos, "   rbtS3s - Standard, 3 columns wide", iColor ) ;
      ++wp.ypos ;

      //* rbtS3p *
      wp = dp->WriteString ( wp.ypos, 2, RB_Strings[rbtS3p].rbString, baseColor ) ;
      dp->WriteChar ( wp.ypos, 2+RB_Strings[rbtS3p].rbSelOffset, RB_Strings[rbtS3p].rbSelchar, selColor ) ;
      dp->WriteString ( wp.ypos, wp.xpos, "   rbtS3p - Standard, 3 columns wide", iColor ) ;
      ++wp.ypos ;

      //* rbtS5a *
      wp = dp->WriteString ( wp.ypos, 2, RB_Strings[rbtS5a].rbString, baseColor ) ;
      dp->WriteChar ( wp.ypos, 2+RB_Strings[rbtS5a].rbSelOffset, RB_Strings[rbtS5a].rbSelchar, selColor ) ;
      dp->WriteString ( wp.ypos, wp.xpos, " rbtS5a - Standard, 5 columns wide", iColor ) ;
      ++wp.ypos ;

      //* rbtS5s *
      wp = dp->WriteString ( wp.ypos, 2, RB_Strings[rbtS5s].rbString, baseColor ) ;
      dp->WriteChar ( wp.ypos, 2+RB_Strings[rbtS5s].rbSelOffset, RB_Strings[rbtS5s].rbSelchar, selColor ) ;
      dp->WriteString ( wp.ypos, wp.xpos, " rbtS5s - Standard, 5 columns wide", iColor ) ;
      ++wp.ypos ;

      //* rbtS5p *
      wp = dp->WriteString ( wp.ypos, 2, RB_Strings[rbtS5p].rbString, baseColor ) ;
      dp->WriteChar ( wp.ypos, 2+RB_Strings[rbtS5p].rbSelOffset, RB_Strings[rbtS5p].rbSelchar, selColor ) ;
      dp->WriteString ( wp.ypos, wp.xpos, " rbtS5p - Standard, 5 columns wide", iColor ) ;
      ++wp.ypos ;

      //* rbtC1 *
      wp = dp->WriteString ( wp.ypos, 2, " ", baseColor ) ;
      dp->WriteChar ( wp.ypos, 2+RB_Strings[rbtC1].rbSelOffset, wcsMEDSQUARE, selColor ) ;
      dp->WriteString ( wp.ypos, wp.xpos, "     rbtC1  - Custom (example), 1 columns", iColor ) ;
      ++wp.ypos ;

      //* rbtC2 (character is 'yes', shi) *
      wp = dp->WriteString ( wp.ypos, 2, "  ", baseColor ) ;
      dp->WriteChar ( wp.ypos, 2+RB_Strings[rbtC1].rbSelOffset, cjkSHI, selColor | ncrATTR ) ;
      dp->WriteString ( wp.ypos, wp.xpos, "    rbtC2  - Custom (example), 2 columns", iColor ) ;
      ++wp.ypos ;

      //* rbtC3 *
      gString gs( "%C %C", &wcsSCAN7, &wcsSCAN7 ) ;
      wp = dp->WriteString ( wp.ypos, 2, gs, baseColor ) ;
      dp->WriteChar ( wp.ypos, 2+RB_Strings[rbtC3].rbSelOffset, wcsMEDSQUARE, selColor ) ;
      dp->WriteString ( wp.ypos, wp.xpos, "   rbtC3  - Custom (example), 3 columns", iColor ) ;
      ++wp.ypos ;

      //* rbtC4 (character is 'yes', shi) *
      wp = dp->WriteString ( wp.ypos, 2, L"[  ]", baseColor ) ;
      dp->WriteChar ( wp.ypos, 2+RB_Strings[rbtC3].rbSelOffset, cjkSHI, selColor | ncrATTR ) ;
      dp->WriteString ( wp.ypos, wp.xpos, "  rbtC4  - Custom (example), 4 columns", iColor ) ;
      ++wp.ypos ;

      //* rbtC5 *
      gs.compose( "%C%C %C%C", &wcsSCAN3, &wcsSCAN7, &wcsSCAN7, &wcsSCAN3 ) ;
      wp = dp->WriteString ( wp.ypos, 2, gs, baseColor ) ;
      dp->WriteChar ( wp.ypos, 2+RB_Strings[rbtC5].rbSelOffset, wcsMEDSQUARE, selColor ) ;
      dp->WriteString ( wp.ypos, wp.xpos, " rbtC5  - Custom (example), 5 columns", iColor ) ;
      ++wp.ypos ;

      //* rbtC6 *
      gs.compose( "%C  %C", &cjkLBRACKET, &cjkRBRACKET ) ;
      wp = dp->WriteString ( wp.ypos, 2, gs, baseColor ) ;
      dp->WriteChar ( wp.ypos, 2+RB_Strings[rbtC6].rbSelOffset, cjkSHI, selColor | ncrATTR ) ;
      dp->WriteString ( wp, "rbtC6  - Custom (example), 6 columns", iColor ) ;

      dp->WriteString ( dLines-2, dCols/2-7, " Press Any Key ", nc.reS, true ) ;

      nckPause() ;
   }
   if ( dp != NULL )                      // close the window
      delete ( dp ) ;
   
   this->RefreshWin () ;                  // restore parent dialog
   
#endif   // ENABLE_DEVELOPMENT_METHODS
}  //* End DisplayRadiobuttonTypes() *

//*************************
//*                       *
//*************************
//********************************************************************************
//*                                                                              *
//*                                                                              *
//*                                                                              *
//* Input  :                                                                     *
//*                                                                              *
//* Returns:                                                                     *
//********************************************************************************

