//********************************************************************************
//* File       : NcdControlSP.cpp                                                *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2006-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in NcDialog.hpp            *
//* Date       : 19-May-2021                                                     *
//* Version    : (see NcDialogVersion string in NcDialog.cpp)                    *
//*                                                                              *
//* Description: Contains the methods of the DialogSpinner class                 *
//*              and the public NcDialog-class methods for accessing             *
//*              the DialogSpinner object's functionality.                       *
//*              See NcDialog.hpp for the definition of this class.              *
//*                                                                              *
//*                                                                              *
//* Development Tools: See NcDialog.cpp.                                         *
//********************************************************************************
//* Version History (most recent first):                                         *
//*   See version history in NcDialog.cpp.                                       *
//********************************************************************************

//*****************
//* Include Files *
//*****************

#include "GlobalDef.hpp"               //* General definitions

#ifndef NCURSES_INCLUDED
#include "NCurses.hpp"
#endif

#ifndef NCDIALOG_INCLUDED
#include "NcDialog.hpp"
#endif


//******************************
//* Local Definitions and Data *
//******************************


      //*************************************************
      //** THIS SECTION IMPLEMENTS THE METHODS OF THE  **
      //**            DialogSpinner CLASS.             **
      //** - - - - - - - - - - - - - - - - - - - - - - **
      //** See below for public NcDialog-class methods **
      //**    related to DialogSpinner-class access.   **
      //*************************************************

//************************
//*    DialogSpinner     *
//************************
//******************************************************************************
//* Constructor for DialogSpinner class object.                                *
//*                                                                            *
//*                                                                            *
//* Input  : pointer to initialization structure                               *
//*                                                                            *
//* Returns: implicitly returns a pointer to DialogSpinner object              *
//******************************************************************************
//* Notes:                                                                     *
//* ======                                                                     *
//* - Because all data members of the dspinData class are public, there is no  *
//*   control over the the values the user sends us; however:                  *
//*   a) the maximum integer values are limited by the size of 'signed int'    *
//*      on the target system.                                                 *
//*   b) minimum value must be <= than maximum value.                          *
//*     - For dspinHEX1 and dspinHEX2 the minimum value is                     *
//*       restricted to: >= ZERO.                                              *
//*   c) minimum <= initial <= maximum                                         *
//*   d) width of the control is limited to MAX_SPINNER_WIDTH (including the   *
//*      indicator character). This is because the width limit for 'swprintf'  *
//*      C-library function format specification is sixteen (16).              *
//*                                                                            *
//******************************************************************************

DialogSpinner::DialogSpinner ( InitCtrl* iPtr )
{
   this->type    = dctSPINNER ;     // set control type
   this->ulY     = iPtr->ulY ;      // transfer the initialization data
   this->ulX     = iPtr->ulX ;
   this->lines   = 1 ;              // display lines == 1
   //* Display columns, including indicator, if any. (nominal range check)  *
   //* For numeric field width, subtract width of indicator character.      *
   this->cols    = ((iPtr->cols > ZERO) && (iPtr->cols < MAX_SPINNER_WIDTH)) ?
                   iPtr->cols : MAX_SPINNER_WIDTH ;
   this->nColor  = iPtr->nColor ;   // color of border when control does not have focus
   this->fColor  = iPtr->fColor ;   // color of border when control has focus
   this->labY    = iPtr->labY ;     // label offsets
   this->labX    = iPtr->labX ;
   this->active = iPtr->active ;    // indicates whether control can be selected

   //* Copy the remaining parameters, and be sure that the values make sense *
   if ( iPtr->spinData != NULL )
   {
      this->fmtCode = iPtr->spinData->dsFormat ;// formatting code (member of enum dspinFormat)
      this->minVal = iPtr->spinData->minValue ; // min value
      this->maxVal = iPtr->spinData->maxValue ; // max value
      this->curVal = iPtr->spinData->iniValue ; // initial display value
      this->iChar  = iPtr->spinData->indiChar ; // indicator character
      this->iColor = iPtr->spinData->indiAttr ; // indicator color attribute

      //* Range check the range *
      if ( this->maxVal < this->minVal )
      {
         short tmp = this->minVal ;
         this->minVal = this->maxVal ;
         this->maxVal = tmp ;
      }

      //* Range checking - min hex value must be >= ZERO *
      if ( ((this->fmtCode == dspinHEX1) || (this->fmtCode == dspinHEX2))
           && (this->minVal < ZERO) )
         this->minVal = ZERO ;
   }
   else     // prevent catastrophy
   {
      this->iColor = nc.bw ;
      this->minVal = this->curVal = ZERO ;
      this->maxVal = 1 ;
      this->fmtCode = dspinINTEGER ;
   }
   if ( this->minVal > this->maxVal )        // compare the data-range parameters
   {
      int   tmp = this->minVal ;
      this->minVal = this->maxVal ;
      this->maxVal = tmp ;
   }
   if ( this->curVal < this->minVal || this->curVal > this->maxVal )
      this->curVal = this->minVal ;

   gString gs ;                  // get width of indicator character
   gs.append( this->iChar ) ;
   int valWidth = this->cols - gs.gscols() ; // width of numeric field

   switch ( this->fmtCode )            // create display data formatting string
   {
      case dspinDEC1:                     // one decimal place
         if ( iPtr->spinData->leadZero )  // with leading zeroes
            swprintf ( this->fmtSpec, MAX_SPINNER_WIDTH, L"%%0%d.%dlf", valWidth, 1 ) ;
         else                             // without leading zeroes
            swprintf ( this->fmtSpec, MAX_SPINNER_WIDTH, L"%%%d.%dlf", valWidth, 1 ) ;
         break ;
      case dspinDEC2:                     // two decimal places
         if ( iPtr->spinData->leadZero )
            swprintf ( this->fmtSpec, MAX_SPINNER_WIDTH, L"%%0%d.%dlf", valWidth, 2 ) ;
         else
            swprintf ( this->fmtSpec, MAX_SPINNER_WIDTH, L"%%%d.%dlf", valWidth, 2 ) ;
         break ;
      case dspinDEC3:                     // three decimal places
         if ( iPtr->spinData->leadZero )
            swprintf ( this->fmtSpec, MAX_SPINNER_WIDTH, L"%%0%d.%dlf", valWidth, 3 ) ;
         else
            swprintf ( this->fmtSpec, MAX_SPINNER_WIDTH, L"%%%d.%dlf", valWidth, 3 ) ;
         break ;
      case dspinHEX1:                     // hexadecimal (0-9,A-F)
         if ( iPtr->spinData->leadZero )  // with leading zeroes
            swprintf ( this->fmtSpec, MAX_SPINNER_WIDTH, L"%%0%dX", valWidth ) ;
         else                             // without leading zeroes
            swprintf ( this->fmtSpec, MAX_SPINNER_WIDTH, L"%%%dX", valWidth ) ;
         break ;
      case dspinHEX2:                     // hexadecimal (0-9,a-f)
         if ( iPtr->spinData->leadZero )  // with leading zeroes
            swprintf ( this->fmtSpec, MAX_SPINNER_WIDTH, L"%%0%dx", valWidth ) ;
         else                             // without leading zeroes
            swprintf ( this->fmtSpec, MAX_SPINNER_WIDTH, L"%%%dx", valWidth ) ;
         break ;
      case dspinINTEGER:                  // integer value
      default:                            // or unsupported option
         if ( iPtr->spinData->leadZero )  // with leading zeroes
            swprintf ( this->fmtSpec, MAX_SPINNER_WIDTH, L"%%0%dd", valWidth ) ;
         else                             // without leading zeroes
            swprintf ( this->fmtSpec, MAX_SPINNER_WIDTH, L"%%%dd", valWidth ) ;
         break ;
   }

   //* Copy the label text to data member, wLabel *
   //* and initialize bHotkey.                    *
   this->InitHotkey ( iPtr->label, false ) ;

   //* Default values for all other data members *
   this->groupCode = ZERO ;   // spinner controls are not grouped (not used)
   this->rtlContent = false ; // left-to-right language content by default

   //* Instantiate the underlying window object *
   wPtr = new NcWindow ( this->lines, this->cols, this->ulY, this->ulX ) ;
   
}  //* End DialogSpinner() *

//************************
//*   ~DialogSpinner     *
//************************
//******************************************************************************
//* Destructor for DialogSpinner class object.                                 *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

DialogSpinner::~DialogSpinner ( void )
{
   //* Close the underlying windows *
   delete this->wPtr ;

}  //* End ~DialogSpinner() *

//************************
//*    OpenControl       *
//************************
//******************************************************************************
//* Draw a dialog spinner control i.e. open the window previously              *
//* instantiated by a call to DialogSpinner().                                 *
//*                                                                            *
//* Input  : hasFocus : optional boolean, false by default, determines         *
//*                     whether to draw the object in nColor or fColor         *
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
//******************************************************************************

short DialogSpinner::OpenControl ( bool hasFocus )
{
short    result ;

   if ( (result = this->wPtr->OpenWindow ()) == OK )
   {
      this->RedrawControl ( hasFocus ) ;  // draw the control
   }
   return result ;

}  //* End OpenControl() *

//************************
//*   RedrawControl      *
//************************
//******************************************************************************
//* Redraw the data for a Spinner control.                                     *
//* (refreshes the control's window)                                           *
//*                                                                            *
//* Input  : hasFocus: if true, control has focus                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void DialogSpinner::RedrawControl ( bool hasFocus )
{
int   color = hasFocus ? fColor : nColor ;

   this->wPtr->SetInteriorColor ( color ) ;  // clear the window
   this->wPtr->ClearWin () ;

   //* Draw the up/down indicator character (if any) *
   gString gs ;                  // get width of indicator character
   gs.append( this->iChar ) ;
   short x = this->cols - gs.gscols() ;   // X offset for indicator
   if ( gs.gscols() > ZERO )
      this->wPtr->WriteChar ( ZERO, x, this->iChar, this->iColor ) ;

   //* Format the display text *
   if ( (this->fmtCode == dspinINTEGER) ||
        (this->fmtCode == dspinHEX1)    ||
        (this->fmtCode == dspinHEX2) )
      swprintf ( this->wText, MAX_SPINNER_WIDTH, this->fmtSpec, this->curVal ) ;
   else
   {
      double dVal ;
      if ( this->fmtCode == dspinDEC1 )
         dVal = (double)this->curVal / 10 ;
      else if ( this->fmtCode == dspinDEC2 )
         dVal = (double)this->curVal / 100 ;
      else 
         dVal = (double)this->curVal / 1000 ;
      swprintf ( this->wText, MAX_SPINNER_WIDTH, this->fmtSpec, dVal ) ;
   }

   //* If display string is too wide for the field, display error indication *
   gs = this->wText ;
   if ( gs.gscols() > x )
   {
      short i = ZERO ;
      for ( ; i < x ; ++i )
         this->wText[i] = L'#' ;
      this->wText[i] = NULLCHAR ;
   }
   this->wPtr->WriteString ( ZERO, ZERO, this->wText, color ) ; // draw text
   this->wPtr->RefreshWin () ;

}  //* End RedrawControl()

//*************************
//*    SetOutputFormat    *
//*************************
//******************************************************************************
//* Set the text data output format: RTL (right-to-left), or                   *
//* LTR (left-to-right). Does not refresh display.                             *
//* Note that the default (unmodified) output format is LTR.                   *
//*                                                                            *
//* Input  : rtlFormat : if 'true',  then set as RTL format                    *
//*                      if 'false', then set as LTR format                    *
//*                                                                            *
//* Returns: ERR: request is not applicable to this control type               *
//******************************************************************************

short DialogSpinner::SetOutputFormat ( bool rtlFormat )
{
   //* Does not apply to dctSPINNER controls. *
   return ERR ;

}  //* End SetOutputFormat() *



      //*************************************************
      //**     THIS SECTION IMPLEMENTS THE PUBLIC,     **
      //**    NcDialog-class METHODS FOR ACCESSING     **
      //**    DialogSpinner-class FUNCTIONALITY.       **
      //*************************************************

//************************
//*     EditSpinner      *
//************************
//******************************************************************************
//* Public method for editing text in dctSPINNER controls.                     *
//*                                                                            *
//* User input keys:                                                           *
//* -------------------------------------------------------------------------- *
//*           up arrow:increase by 1               down arrow:decrease by 1    *
//*   Shift + up arrow:increase by 10      Shift + down arrow:decrease by 10   *
//* Control + up arrow:increase by 100   Control + down arrow:decrease by 100  *
//*     Alt + up arrow:increase by 1000      Alt + down arrow:decrease by 1000 *
//*        Page Up key:increase by 10%          Page Down key:decrease by 10%  *
//*          Home key :maximum value                  End key:minimum value    *
//*                                                                            *
//* Input  : uiInfo class (by reference) - initial values ignored              *
//*          (see note in NcDialog.hpp, about values returned)                 *
//*                                                                            *
//* Returns: index of control that currently has the input focus               *
//******************************************************************************

short NcDialog::EditSpinner ( uiInfo& info )
{
DialogSpinner* cp ;
short    origValue ;
bool     done = false ;    // loop control

   //* Be sure control is actually a spinner control *
   if ( this->dCtrl[this->currCtrl]->type == dctSPINNER )
   {
      cp = (DialogSpinner*)this->dCtrl[this->currCtrl] ;
      origValue = cp->curVal ;      // save current value
   }
   //* Control with focus is not a spinner, i.e. application error    *
   else
   {
      //* Do what we can to minimize the damage *
      info.ctrlType = this->dCtrl[this->currCtrl]->type ;
      info.ctrlIndex = this->currCtrl ;
      info.keyIn = nckTAB ;               // move on to next control
      info.hasFocus = true ;              // control has focus
      info.dataMod = false ;              // no data modified
      info.selMember = MAX_DIALOG_CONTROLS ; // don't care
      info.isSel = false ;                // don't care
      info.viaHotkey = false ;            // invalidate any hotkey data
      done = true ;  
   }

   //**************************
   //* Interact with the user *
   //**************************
   while ( ! done )
   {
      //* 1) Get user input.                                  *
      //* 2) Check for toggle of Insert/Overstrike mode.      *
      //* 3) Check for terminal-resize event.                 *
      //* 4) Access callback method, if specified.            *
      //* 5) If input has been handled, return to top of loop.*
      this->GetKeyInput ( info.wk ) ;

      if ( (info.wk.type == wktFUNKEY)
            && (info.wk.key == nckINSERT || info.wk.key == nckRESIZE) )
      {
         if ( info.wk.key == nckINSERT )
            this->ToggleIns () ;
         else
            this->TermResized () ;

         if ( ExternalControlUpdate != NULL )
            ExternalControlUpdate ( this->currCtrl, info.wk, false ) ;
         continue ;
      }

      //* Transform Enter key and Space key (not used here) to Tab key.*
      if ( info.wk.type == wktFUNKEY && 
           (info.wk.key == nckENTER || info.wk.key == nckpENTER) )
      {
         info.wk.key = nckTAB ;
      }
      else if ( info.wk.type == wktPRINT && info.wk.key == nckSPACE )
      {
         info.wk.type = wktFUNKEY ;
         info.wk.key = nckTAB ;
      }

      //* Done scrolling. Move to next control *
      if ( (info.wk.type == wktFUNKEY) && 
           (info.wk.key == nckTAB || info.wk.key == nckRIGHT) )
      {
         info.ctrlType = dctSPINNER ;        // control type
         info.ctrlIndex = this->currCtrl ;   // control with input focus
         info.hasFocus = true ;              // focus was not lost
         info.keyIn = nckTAB ;               // focus to move forward to next control
         info.dataMod = true ;               // indicate value change (but see below)
         info.selMember = cp->curVal ;       // currently-displayed value
         info.isSel = false ;                // don't care
         info.viaHotkey = false ;            // control did not get focus via hotkey
         done = true ;                 // returning to caller
      }
      //* Done scrolling. Move to previous control *
      else if ( (info.wk.type == wktFUNKEY) && 
                (info.wk.key == nckSTAB || info.wk.key == nckLEFT) )
      {
         info.ctrlType = dctSPINNER ;        // control type
         info.ctrlIndex = this->currCtrl ;   // control with input focus
         info.hasFocus = true ;              // focus was not lost
         info.keyIn = nckSTAB ;              // focus to move backward to previous control
         info.dataMod = true ;               // indicate value change (but see below)
         info.selMember = cp->curVal ;       // currently-displayed value
         info.isSel = false ;                // don't care
         info.viaHotkey = false ;            // control did not get focus via hotkey
         done = true ;                 // returning to caller
      }
      //* Display-value adjustment keys *
      else if (   ((info.wk.type == wktPRINT) &&
                   (info.wk.key == '+' || info.wk.key == '=' || 
                    info.wk.key == '-' || info.wk.key == '_' )
                  )
               ||
                  ((info.wk.type == wktFUNKEY) &&
                   (   info.wk.key == nckUP   || info.wk.key == nckDOWN
                    || info.wk.key == nckSUP  || info.wk.key == nckSDOWN
                    || info.wk.key == nckCUP  || info.wk.key == nckCDOWN
                    || info.wk.key == nckAUP  || info.wk.key == nckADOWN
                    || info.wk.key == nckPGUP || info.wk.key == nckPGDOWN
                    || info.wk.key == nckHOME || info.wk.key == nckEND)
                  )
              )
      {
         bool  displayUpdate = false ;
         switch ( info.wk.key )
         {
            //* Increase display value by one *
            case nckUP:
            case '+':
            case '=':
               if ( cp->curVal < cp->maxVal )
               {
                  ++cp->curVal ;
                  displayUpdate = true ;
               }
               break ;
            //* Decrease display value by one *
            case nckDOWN:
            case '-':
            case '_':
               if ( cp->curVal > cp->minVal )
               {
                  --cp->curVal ;
                  displayUpdate = true ;
               }
               break ;
            //* Increase display value by ten (Shift+nckUP) *
            case nckSUP:
               if ( cp->curVal < cp->maxVal )
               {
                  cp->curVal += 10 ;
                  if ( cp->curVal > cp->maxVal )
                     cp->curVal = cp->maxVal ;
                  displayUpdate = true ;
               }
               break ;
            //* Decrease display value by ten (Shift+nckDOWN) *
            case nckSDOWN:
               if ( cp->curVal > cp->minVal )
               {
                  cp->curVal -= 10 ;
                  if ( cp->curVal < cp->minVal )
                     cp->curVal = cp->minVal ;
                  displayUpdate = true ;
               }
               break ;
            //* Increase display value by one hundred (Ctrl+nckUP) *
            case nckCUP:
               if ( cp->curVal < cp->maxVal )
               {
                  cp->curVal += 100 ;
                  if ( cp->curVal > cp->maxVal )
                     cp->curVal = cp->maxVal ;
                  displayUpdate = true ;
               }
               break ;
            //* Decrease display value by one hundred (Ctrl+nckDOWN) *
            case nckCDOWN:
               if ( cp->curVal > cp->minVal )
               {
                  cp->curVal -= 100 ;
                  if ( cp->curVal < cp->minVal )
                     cp->curVal = cp->minVal ;
                  displayUpdate = true ;
               }
               break ;
            //* Increase display value by one thousand (Alt+nckUP) *
            case nckAUP:
               if ( cp->curVal < cp->maxVal )
               {
                  cp->curVal += 1000 ;
                  if ( cp->curVal > cp->maxVal )
                     cp->curVal = cp->maxVal ;
                  displayUpdate = true ;
               }
               break ;
            //* Decrease display value by one thousand (Alt+nckDOWN) *
            case nckADOWN:
               if ( cp->curVal > cp->minVal )
               {
                  cp->curVal -= 1000 ;
                  if ( cp->curVal < cp->minVal )
                     cp->curVal = cp->minVal ;
                  displayUpdate = true ;
               }
               break ;
            //* Increase display value by 10 percent of total range *
            case nckPGUP:
               if ( cp->curVal < cp->maxVal )
               {
                  int totrange = cp->maxVal + abs(cp->minVal),
                      tenpct = totrange / 10 ;
                  if ( (cp->curVal + tenpct) <= cp->maxVal )
                     cp->curVal += tenpct ;
                  else
                     cp->curVal = cp->maxVal ;
                  displayUpdate = true ;
               }
               break ;
            //* Decrease display value by 10 percent of total range *
            case nckPGDOWN:
               if ( cp->curVal > cp->minVal )
               {
                  int totrange = cp->maxVal + abs(cp->minVal),
                      tenpct = totrange / 10 ;
                  if ( (cp->curVal - tenpct) >= cp->minVal )
                     cp->curVal -= tenpct ;
                  else
                     cp->curVal = cp->minVal ;
                  displayUpdate = true ;
               }
               break ;
            //* Set display value to maximum *
            case nckHOME:
               if ( cp->curVal < cp->maxVal )
               {
                  cp->curVal = cp->maxVal ;
                  displayUpdate = true ;
               }
               break ;
            //* Set display value to minimum *
            case nckEND:
               if ( cp->curVal > cp->minVal )
               {
                  cp->curVal = cp->minVal ;
                  displayUpdate = true ;
               }
               break ;
         }
         if ( displayUpdate != false )    // update the display
            cp->RedrawControl ( true ) ;
      }     // display-value adjustment keys

      else if ( info.wk.type == wktFUNKEY && info.wk.key == nckESC ) // abort selection
      {
         //* return display value to initial state *
         if ( cp->curVal != origValue )
         {
            cp->curVal = origValue ;
            cp->RedrawControl ( true ) ;
            if ( ExternalControlUpdate != NULL )// in case caller is tracking the activity
               ExternalControlUpdate ( this->currCtrl, info.wk, false ) ;
         }

         info.ctrlType = dctSPINNER ;        // control type
         info.ctrlIndex = this->currCtrl ;   // control with input focus
         info.hasFocus = true ;              // focus was not lost
         info.keyIn = nckTAB ;               // focus to move forward to next control
         info.dataMod = false ;              // data unchanged
         info.selMember = cp->curVal ;       // currently-displayed value
         info.isSel = false ;                // don't care
         info.viaHotkey = false ;            // control did not get focus via hotkey
         done = true ;                    // returning to caller
      }
      else     // test for a possible hotkey
      {
         //* Scan the controls for one whose hotkey matches the input.*
         //* If match found, returns control's index, else -1.        *
         short hotIndex = this->IsHotkey ( info.wk ) ;
         if ( hotIndex >= ZERO )
         {
            //* Ignore hotkey selection of this control,                    *
            //* else, make the indicated control the current/active control *
            if ( hotIndex != currCtrl )
            {
               //* Indicate the results of our edit *
               info.ctrlType = dctSPINNER ;     // control type
               info.ctrlIndex = this->currCtrl ;// control with input focus
               info.hasFocus = true ;           // this may be reset by ChangedFocusViaHotkey()
               info.keyIn = ZERO ;              // indicate no shift in focus necessary
               info.dataMod = true ;            // indicate selection change (but see below)
               info.selMember = cp->curVal ;    // currently-displayed value
               info.isSel = false ;             // don't care
               info.viaHotkey = false ;         // this may be set by ChangedFocusViaHotkey()
               this->ChangedFocusViaHotkey ( info.wk.key, hotIndex, info ) ;
               done = true ;
            }
         }
         else
         {
            // key input value is not valid in this context, ignore it
         }
      }

      //* If caller has specified a callback method, do it now.*
      if ( ExternalControlUpdate != NULL )
         ExternalControlUpdate ( this->currCtrl, info.wk, false ) ;
   }  // while(!done)

   //* If displayed value has not changed, reset dataMod flag *
   if ( cp->curVal == origValue )
      info.dataMod = false ;

   return this->currCtrl; 

}  //* End EditSpinner() *

//************************
//*   GetSpinnerValue    *
//************************
//******************************************************************************
//* Return the currently-displayed value for the specified dctSPINNER control. *
//*                                                                            *
//* Input  : cIndex : index into list of controls defined for this dialog      *
//*          spValue: (by reference, initial value ignored)                    *
//*                   receives currently-displayed value                       *
//*                   - See notes on interpretation of Spinner data values     *
//*                     in the dspinData-class definition.                     *
//*                                                                            *
//* Returns: OK if successful                                                  *
//*          OR ERR if specified control is not of type dctSPINNER             *
//******************************************************************************

short NcDialog::GetSpinnerValue ( short cIndex, int& spValue )
{
   short status = ERR ;
   if ( cIndex >= ZERO && cIndex <= lastCtrl && dCtrl[cIndex]->type == dctSPINNER )
   {
      DialogSpinner* cp = (DialogSpinner*)(dCtrl[cIndex]) ;
      spValue = cp->curVal ;
      status = OK ;
   }
   return status ;

}  //* End GetSpinnerValue() *

//************************
//*   SetSpinnerValue    *
//************************
//******************************************************************************
//* Set a new display value for specified dctSPINNER control and update display*
//*                                                                            *
//* IMPORTANT NOTE: Please carefully review the notes on dctSPINNTER-control   *
//*                 value encoding in the definition of the dspinData class.   *
//*                                                                            *
//* Input  : cIndex : index into list of controls defined for this dialog      *
//*          spValue: new 'current' value for the control                      *
//*                                                                            *
//* Returns: OK if successful                                                  *
//*          OR ERR if spValue is outside the defined range for this control   *
//*                 if specified control currently has the input focus         *
//*                 if specified control is not of type dctSPINNER             *
//******************************************************************************

short NcDialog::SetSpinnerValue ( short cIndex, int spValue )
{
   short status = ERR ;
   if ( cIndex >= ZERO && cIndex <= this->lastCtrl && 
        (this->dCtrl[cIndex])->type == dctSPINNER && cIndex != this->currCtrl )
   {
      DialogSpinner* cp = (DialogSpinner*)(dCtrl[cIndex]) ;
      if ( spValue >= cp->minVal && spValue <= cp->maxVal )
      {
         cp->curVal = spValue ;
         cp->RedrawControl ( false ) ;
         status = OK ;
      }
   }
   return status ;

}  //* End SetSpinnerValue() *

//************************
//*   SetSpinnerLimits   *
//************************
//******************************************************************************
//* Specify a new minimum value, maximum value and current value for a         *
//* dctSPINNER control.                                                        *
//*                                                                            *
//* IMPORTANT NOTE: Please carefully review the notes on dctSPINNTER-control   *
//*                 value encoding in the definition of the dspinData class.   *
//*                                                                            *
//* Input  : cIndex: index into list of controls defined for this dialog       *
//*          minValue : new minimum data value for the control                 *
//*          maxValue : new maximum data value for the control                 *
//*          setValue : new currently-displayed data value                     *
//*                                                                            *
//* Returns: OK if successful                                                  *
//*          OR ERR if spValue is outside the minValue to maxValue range       *
//*                 if specified control currently has the input focus         *
//*                 if specified control is not of type dctSPINNER             *
//******************************************************************************
//* Notes:                                                                     *
//* ======                                                                     *
//* Parameter validation is the same as for the DialogSpinner constructor:     *
//*   a) the maximum integer values are limited by the size of 'signed int'    *
//*      on the target system.                                                 *
//*   b) minimum value must be <= than maximum value.                          *
//*     - For dspinHEX1 and dspinHEX2 the minimum value is                     *
//*       restricted to: >= ZERO.                                              *
//*   c) minimum <= initial <= maximum                                         *
//******************************************************************************

short NcDialog::SetSpinnerLimits ( short cIndex, int minValue, 
                                   int maxValue, int setValue )
{
   short status = ERR ;
   if ( cIndex >= ZERO && cIndex <= this->lastCtrl && 
        (this->dCtrl[cIndex])->type == dctSPINNER && cIndex != this->currCtrl )
   {
      if ( minValue > maxValue )  // compare the data-range parameters
      {                           // swap min and max if necessary
         int  tmp = minValue ;
         minValue = maxValue ;
         maxValue = tmp ;
      }

      DialogSpinner* cp = (DialogSpinner*)(dCtrl[cIndex]) ;
      if ( ((cp->fmtCode == dspinHEX1) || (cp->fmtCode == dspinHEX2))
           && (minValue < ZERO) )
         minValue = ZERO ;

      if ( setValue >= minValue && setValue <= maxValue )
      {
         cp->minVal = minValue ;
         cp->maxVal = maxValue ;
         cp->curVal = setValue ;
         cp->RedrawControl ( false ) ;
         status = OK ;
      }
   }
   return status ;

}  //* End SetSpinnerLimits() *

//************************
//*                      *
//************************
//********************************************************************************
//*                                                                              *
//*                                                                              *
//*                                                                              *
//* Input :                                                                      *
//*                                                                              *
//* Returns:                                                                     *
//********************************************************************************

