//********************************************************************************
//* File       : NcdControlTB.cpp                                                *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2006-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in NcDialog.hpp            *
//* Date       : 26-Jul-2025                                                     *
//* Version    : (see NcDialogVersion string in NcDialog.cpp)                    *
//*                                                                              *
//* Description: Contains the methods of the DialogTextbox class                 *
//*              and the public NcDialog-class methods for accessing             *
//*              the DialogTextbox object's functionality.                       *
//*              See NcDialog.hpp for the definition of this class.              *
//*                                                                              *
//*                                                                              *
//* Development Tools: See NcDialog.cpp.                                         *
//********************************************************************************
//* Version History (most recent first):                                         *
//*   See version history in NcDialog.cpp.                                       *
//********************************************************************************
//* Notes on use of GNOME/Wayland clipboard with NcDialog textbox controls:      *
//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -                *
//* To avoid tying ourselves to external libraries, we do not directly           *
//* access the Wayland (system) clipboard from inside the NcDialog class.        *
//* Instead, we interface with the external third-party "wl-clipboard"           *
//* utilities "wl-copy" and "wl-paste. See documentation for details.            *
//*                                                                              *
//* These utilities (if installed) may be accessed through the WaylandCB class   *
//* which will move data between the system clipboard and the NcDialog Local     *
//* Clipboard.                                                                   *
//* If the "wl-clipboard" utilities are not installed or are otherwise           *
//* unavailable, the CTRL+C, CTRL+V, CTRL+X clipboard commands will access only  *
//* the NcDialog Local Clipboard.                                                *
//*                                                                              *
//* Note on the terminal emulator's clipboard interface:                         *
//* - - - - - - - - - - - - - - - - - - - - - - - - - -                          *
//* The terminal emulator's copy-and-paste menu items or the equivalent          *
//* shortcut keys CTRL+SHIFT+C and CTRL+SHIFT+V as well as the mouse-only        *
//* console copy-and-paste are functional, but are invisible to console          *
//* applications, and so are potentially dangerous. See documentation for        *
//* additional information on this hidden clipboard interface.                   *
//*                                                                              *
//********************************************************************************


//*****************
//* Include Files *
//*****************

#include "GlobalDef.hpp"               //* General definitions

#ifndef NCURSES_INCLUDED
#include "NCurses.hpp"
#endif

#ifndef NCDIALOG_INCLUDED
#include "NcDialog.hpp"
#endif

//*********************
//* Local Definitions *
//*********************
#define DEBUG_ET (0)          // Set to 1 to debug singleline controls, else 0
#define DEBUG_ETM (0)         // Set to 1 to debug multiline controls, else 0

//* For multi-line textbox editing, 'head' text i.e. *
//* un-displayed text before the displayed text.     *
#define mlHead wText

const short appendCOLS = 2 ;           //* space needed for appending text

//* Default clipboard access key assignments.              *
//* Note that all of these keycodes are of type wktFUNKEY, *
//* that is, they are classified as 'function keys' by     *
//* the ncurses system library.                            *
enum cbAccess : wchar_t {
                         caSELECT_RIGHT  = nckSRIGHT,
                         caSELECT_LEFT   = nckSLEFT,
                         caSELECT_ALL    = nckC_A,
                         caCOPY_SELECTED = nckC_C,
                         caCUT_SELECTED  = nckC_X,
                         caPASTE         = nckC_V
                        } ;


//*********************
//*    Local Data     *
//*********************
#if DEBUG_ET != 0 || DEBUG_ETM != 0 // For debugging the EditTextbox() methods only
NcDialog* etPtr      = NULL ;
attr_t    etColor    = nc.blR ;
gString   gsetb ;
const short etROWS   = 25,
            etCols   = 28,
            etBaseY  = 2,
            etBaseX  = 0,
            etStatY  = 6 ;
short       etRows   = etROWS ;
#endif   // DEBUG_ET || DEBUG_ETM

//* Non-member pointer to WaylandCB class object for access to the *
//* GNOME/Wayland clipboard. This must be a non-member method for  *
//* all child dialogs to inherit access to the system clipboard.   *
static WaylandCB *wcbClip = NULL ;

//* For scanning text data. List of Linux/UNIX shell "special" characters.*
const short   linuxSpecialCount = 11 ;
const wchar_t linuxSpecialChars[linuxSpecialCount] = 
{ L'\\', L'\'', L'"', L'?', L':', L';', L'&', L'>', L'<', L'|', L'*' } ;


      //*************************************************
      //** THIS SECTION IMPLEMENTS THE METHODS OF THE  **
      //**            DialogTextbox CLASS.             **
      //** - - - - - - - - - - - - - - - - - - - - - - **
      //** See below for public NcDialog-class methods **
      //**    related to DialogTextbox-class access.   **
      //*************************************************

//*************************
//*     DialogTextbox     *
//*************************
//******************************************************************************
//* Constructor for DialogTextbox class object.                                *
//*                                                                            *
//*                                                                            *
//* Input  : pointer to initialization structure                               *
//*                                                                            *
//* Returns: constructors implicitly return a pointer to the object            *
//******************************************************************************

DialogTextbox::DialogTextbox ( InitCtrl* iPtr )
{
   //* Transfer the initialization data *
   this->type  = dctTEXTBOX ;       // set control type
   this->ulY   = iPtr->ulY ;        // control's offset within the dialog
   this->ulX   = iPtr->ulX ;
   //* Range check caller's 'lines' and 'cols' values and adjust if necessary.*
   if ( iPtr->cols < 1 )
      iPtr->cols = MIN_TB_SHIFT_WIDTH ;
   else if ( iPtr->cols > MAX_DIALOG_WIDTH )
      iPtr->cols = MAX_DIALOG_WIDTH ;
   this->cols = iPtr->cols ;        // Diaplay columns
   if ( iPtr->lines < 1 )
      iPtr->lines = 1 ;
   //* This ensures that the control's text will fit into a gString object.*
   while ( (iPtr->lines > 1) && 
           (iPtr->lines * this->cols) >= gsALLOCDFLT )
      --iPtr->lines ;
   this->lines = iPtr->lines ;      // Display lines
   //* Color of text when control DOES NOT have focus (disallow underlining) *
   this->nColor = iPtr->nColor & ~ncuATTR ;
   //* Color of text when control DOES have focus (disallow underlining) *
   this->fColor = iPtr->fColor & ~ncuATTR ;
   this->filter = iPtr->filter ;    // text filter code
   this->labY   = iPtr->labY ;      // label offsets
   this->labX   = iPtr->labX ;
   this->hzShift = this->cols < MIN_TB_SHIFT_WIDTH ? false : true ; // Extended text?
   this->active = iPtr->active ;    // can control receive input focus?

   //* Audible alert for invalid key input is initially disabled. *
   this->audibleAlert = false ;

   //* Copy the label text to data member, wLabel *
   //* and initialize bHotkey.                    *
   this->InitHotkey ( iPtr->label, false ) ;

   //* If a mult-line control, then allocate text storage.*
   this->mlText = NULL ;
   this->mlTail = NULL ;
   if ( this->lines > 1 )
   {
      this->mlText = new multiText[this->lines] ;
      this->mlTail = new wchar_t[gsALLOCDFLT] ;
   }

   //* Assume that no display text was specified. Initializes:               *
   //* 'wText' ('mlHead'), 'leftIndex', 'tbIndex', 'tbCursor' and 'mlText[]' *
   this->tbClear () ;

   //* Filter, format and store the display text (if any). *
   if ( iPtr->dispText != NULL && iPtr->dispText[ZERO] != NULLCHAR )
   {
      gString gs( iPtr->dispText ) ;
      this->tbSetText ( gs ) ;
   }

   //* Default values for all other data members                          *
   //* Note: The TbSelect 'select' member is initialized by instantiation.*
   this->rightJust = false ;     // input is left justified by default
   this->groupCode = ZERO ;      // text-box controls are not grouped (not used)
   this->rtlContent = false ;    // left-to-right language content by default

   //* Instantiate the underlying window object *
   this->wPtr = new NcWindow ( this->lines, this->cols, this->ulY, this->ulX ) ;

}  //* End DialogTextbox() *

//*************************
//*    ~DialogTextbox     *
//*************************
//******************************************************************************
//* Destructor for DialogTextbox class object.                                 *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

DialogTextbox::~DialogTextbox ( void )
{
   //* Delete dynamically-allocated data *
   if ( this->mlText != NULL )
      delete [] this->mlText ;
   if ( this->mlTail != NULL )
      delete [] this->mlTail ;

   //* Close the underlying window *
   delete this->wPtr ;

}  //* End ~DialogTextbox() *

//*************************
//*     OpenControl       *
//*************************
//******************************************************************************
//* Draw a dialog text box control i.e. open the window previously             *
//* instantiated by a call to DialogTextbox().                                 *
//*                                                                            *
//* Input  : hasFocus : optional boolean, false by default, determines         *
//*                     whether to draw the object in nColor or fColor         *
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
//******************************************************************************

short DialogTextbox::OpenControl ( bool hasFocus )
{
   short result ;

   //* Initialize the control's underlying window *
   if ( (result = this->wPtr->OpenWindow ()) == OK )
   {  //* Draw control on the screen with text data (if any) *
      this->RedrawControl ( hasFocus ) ;
   }
   return result ;

}  //* End OpenControl() *

//*************************
//*    RedrawControl     *
//*************************
//******************************************************************************
//* Redraw the data for a Textbox control.                                     *
//* Note that if data are longer than control width, the WriteString() method  *
//* will truncate the output at edge of control.                               *
//* (refreshes control window's display)                                       *
//*                                                                            *
//* Input  : hasFocus: if true, control has input focus                        *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void DialogTextbox::RedrawControl ( bool hasFocus )
{

   this->RedrawControl ( hasFocus, true ) ;

}  //* End RedrawControl()

//*************************
//*     RedrawControl     *
//*************************
//******************************************************************************
//* Redraw the control's data and optionally refresh the display.              *
//*                                                                            *
//* NOTE: Called directly only by members of the DialogTextbox class.          *
//*                                                                            *
//* Input  : hasFocus: if true, control has focus                              *
//*          refresh : (optional, 'true' by default) refresh display           *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void DialogTextbox::RedrawControl ( bool hasFocus, bool refresh )
{
   attr_t   dColor = hasFocus ? this->fColor : this->nColor ;

   this->wPtr->SetInteriorColor ( dColor ) ; // set control's interior color

   //* Right-justified text applies only to single-line controls.*
   if ( this->rightJust != false && this->lines == 1 )
   {
      this->wPtr->ClearWin () ;     // clear existing data (if any)
      //* For right-justified 'selection', it's all or nothing.*
      if ( this->select.active )
         dColor |= ncuATTR ;
      this->wPtr->WriteString ( this->tbCursor.ypos, this->leftIndex, 
                                this->wText, dColor ) ;
   }

   //* Left-justified text or RTL languages *
   else
   {
      this->wPtr->ClearWin () ;     // clear existing data (if any)

      //* Single-line controls *
      if ( this->lines == 1 )
      {  //* Redisplay currently-visible text in specified color *
         short index = hasFocus ? this->leftIndex : ZERO,
               xpos  = this->rtlContent ? (this->cols - 1) : ZERO ;

         if ( this->select.active )
         {
            this->tbWriteColorString ( this->tbCursor.ypos, xpos, 
                                       &this->wText[index], dColor ) ;
         }
         else
            this->wPtr->WriteString ( this->tbCursor.ypos, xpos, 
                                      &this->wText[index], dColor, 
                                      false, this->rtlContent ) ;

         if ( hasFocus )      // set the visible cursor
            this->wPtr->SetCursor ( this->tbCursor ) ;
      }
      else  // (this->lines > 1)
      {
         //* If control does not have focus, then display from top of text.*
         //* This is done by fudging the 'leftIndex' member and            *
         //* reformatting the data.                                        *
         short oldLeftIndex = this->leftIndex ;
         winPos oldTbIndex  = this->tbIndex,
                oldTbCursor = this->tbCursor ;
         gString gs ;
         if ( hasFocus == false && this->leftIndex > ZERO )
         {
            this->tbGetText ( gs ) ;
            this->leftIndex = ZERO ;
            this->mlFmtDisplay ( gs ) ;
         }

         short xoff = this->rtlContent ? (this->cols - 1) : ZERO ;
         if ( this->select.active )
         {
            for ( short lIndex = ZERO ; lIndex < this->lines ; lIndex++ )
            {
               this->tbWriteColorString ( lIndex, xoff, 
                                          this->mlText[lIndex].ln, dColor ) ;
            }
         }
         else
         {
            for ( short lIndex = ZERO ; lIndex < this->lines ; lIndex++ )
               this->wPtr->WriteString ( lIndex, xoff, 
                                         this->mlText[lIndex].ln, dColor,
                                         false, this->rtlContent ) ;
         }

         if ( hasFocus )      // set the visible cursor
            this->wPtr->SetCursor ( this->tbCursor ) ;

         if ( hasFocus == false && oldLeftIndex > ZERO )
         {  //* Restore true stats *
            this->leftIndex = oldLeftIndex ;
            this->mlFmtDisplay ( gs ) ;
            this->tbIndex = oldTbIndex ;
            this->tbCursor = oldTbCursor ;
         }
      }
   }
   if ( refresh != false )
      this->wPtr->RefreshWin () ;              // update the display

}  //* End RedrawControl()

//*************************
//*     tbSelectChar      *
//*************************
//******************************************************************************
//* User is selecting text for a subsequent Copy/Cut/Paste operation.          *
//* The direction the cursor is moving is indicated by 'rightward'.            *
//*                                                                            *
//* The actual direction of selection is not about left or right movement, but *
//* about the head or tail of the data.                                        *
//*      LTR: rightward is toward tail of data                                 *
//*           leftward is toward head of data                                  *
//*      RTL: leftward is toward tail of data                                  *
//*           rightward is toward head of data                                 *
//*                                                                            *
//* Input  : rightward : direction of cursor movement                          *
//*                      'true' : user is selecting toward the right           *
//*                      'false': user is selecting toward the left            *
//*                                                                            *
//* Returns: OK  if selection successful                                       *
//*          ERR if no characters to be selected in indicated direction        *
//******************************************************************************

short DialogTextbox::tbSelectChar ( bool rightward )
{
   gString gs ;
   this->tbGetText ( gs, false ) ;  // get a copy of our data
   short wChars = gs.gschars() - 1, // number of characters (not incl. null)
                                    // (note: for mltext, this includes newlines)
         iPoint,                    // insertion point (position of cursor)
         nloff = ZERO,              // offset to compensate for newlines in the stream
         status = OK ;              // return value
   bool  ml = (this->lines > 1),    // 'true' if multi-line control
         rtl = this->rtlContent,    // 'true' if RTL data
         toTail = bool((rightward && !rtl) || (!rightward && rtl)) ; // direction of movement

   if ( ml )                        // iPoint calculated _with_ newlines
   {
      iPoint = this->mlIndex2Ip ( gs ) ;
      if ( gs.gstr()[iPoint] == nckNEWLINE )
      {
         if ( rightward )
         { this->mlScrollRight () ; iPoint += rtl ? (-1) : 1 ; }
         else
         { this->mlScrollLeft () ; iPoint += rtl ? 1 : (-1) ;  }
      }
      nloff += this->tbIndex.ypos ;
   }
   else                             // iPoint is simple X offset
      iPoint = this->tbIndex.xpos ;

   //* If moving toward the tail of the data, and we have somewhere to go *
   if ( toTail && iPoint < wChars )
   {
      if ( ! this->select.active )  // if beginning a new selection
      {
         this->select.active = this->select.sel[iPoint - nloff] = true ;
         this->select.base = this->tbIndex ;
         this->select.count = 1 ;
         this->select.fwd = true ;
      }
      else                          // select (or deselect) character to tail
      {
         if ( ! this->select.sel[iPoint - nloff] )
         {
            if ( ! this->select.sel[iPoint - nloff + 1] )
            {
               this->select.sel[iPoint - nloff] = true ;      // select
               if ( (++this->select.count) == 1 )
                  this->select.fwd = true ;
            }
            else
            {
               this->select.sel[iPoint - nloff + 1] = false ; // deselect
               --this->select.count ;
            }
         }
         else
         {
            this->select.sel[iPoint - nloff] = false ;        // deselect
            --this->select.count ;
         }
      }

      if ( rtl )                 // RTL data
      {
         //* Move the cursor one position to the left *
         if ( ml )   this->mlScrollLeft () ;
         else        this->slScrollLeft () ;
      }
      else                       // LTR data
      {
         //* Move the cursor one position to the right *
         if ( ml )   this->mlScrollRight () ;
         else        this->slScrollRight () ;
      }
   }

   //* If moving toward the head of the data *
   else if ( ! toTail )
   {
      if ( ! this->select.active )  // if beginning a new selection
      {
         this->select.active = this->select.sel[iPoint - nloff] = true ;
         this->select.base = this->tbIndex ;
         this->select.count = 1 ;
         this->select.fwd = false ;
      }
      else if ( iPoint > ZERO )     // select (or deselect) character to head
      {
         if ( ! this->select.sel[iPoint - nloff] )
         {
            if ( ! this->select.sel[iPoint - nloff - 1] )
            {
               this->select.sel[iPoint - nloff] = true ;      // select
               if ( (++this->select.count) == 1 )
                  this->select.fwd = false ;
            }
            else
            {
               this->select.sel[iPoint - nloff - 1] = false ; // deselect
               --this->select.count ;
               //* Special Case: if we have just deselected the head character,*
               //* then all characters are now deselected, so cancel selection.*
               if ( iPoint == 1 )
                  this->select.reset() ;
            }
         }
         else
         {
            this->select.sel[iPoint - nloff] = false ;        // deselect
            --this->select.count ;
         }
      }
      else
      {  //* Special Case: if iPoint is at head of string AND *
         //*               if selection already in progress   *
         if ( this->select.sel[iPoint - nloff + 1] )
         {
            if ( ! this->select.sel[iPoint - nloff] )
            {
               this->select.sel[iPoint - nloff] = true ;   // select
               ++this->select.count ;
            }
            else              // already fully selected
               status = ERR ; // not really an error, just a warning
         }
         else
         {
            this->select.sel[iPoint - nloff] = false ;     // deselect
            //* All characters are now deselected, so cancel selection.*
            this->select.reset() ;
         }
      }

      //* If not already at head of data *
      if ( iPoint > ZERO )
      {
         if ( rtl )                 // RTL data
         {
            //* Move the cursor one position to the right *
            if ( ml )
            {
               short oldY = this->tbIndex.ypos ;
               this->mlScrollRight () ;
               //* If we're sitting on a line terminator, reference a real character *
               if ( (this->tbIndex.ypos != oldY) && (this->tbIndex.xpos > ZERO) )
                  this->mlScrollRight () ;
            }
            else
               this->slScrollRight () ;
         }
         else                       // LTR data
         {
            //* Move the cursor one position to the left *
            if ( ml )
            {
               short oldY = this->tbIndex.ypos ;
               this->mlScrollLeft () ;
               //* If we're sitting on a line terminator, reference a real character *
               if ( (this->tbIndex.ypos != oldY) && (this->tbIndex.xpos > ZERO) )
                  this->mlScrollLeft () ;
            }
            else
               this->slScrollLeft () ;
         }
      }
   }

   //* Otherwise, nowhere to go in indicated direction *
   else
      status = ERR ;

   //* Update the display *
   if ( status == OK )
      this->RedrawControl ( true, true ) ;

   return status ;

}  //* End tbSelectChar() *

//*************************
//*     tbSelectText      *
//*************************
//******************************************************************************
//* For the Textbox control which is currently under edit:                     *
//*  a) 'select all' text data in the control for use with a subsequent        *
//*     Copy/Cut/Paste operation.                                              *
//*     Text that has been selected is visually indicated by the ncuATTR       *
//*     (underline) color attribute.                                           *
//*     -- Note that for Copy or Cut operations, a 'select all' is for visual  *
//*        effect only. These commands operate on ALL source text by default.  *
//*     -- For Paste operations, the data are INSERTED into the target at the  *
//*        current cursor position by default. However if there is selected    *
//*        text in the target control, then the selected text will be REPLACED *
//*        by the pasted data.                                                 *
//*  b) 'deselect' any currently-selected text data in the source control.     *
//*     -- Selection is automatically cleared when the call to EditTextbox()   *
//*        returns to caller, OR when user presses a non-selection key.        *
//*                                                                            *
//* Call this method only to select ALL or to deselect ALL text in the control.*
//* Selection of source text on a character-by-character basis is done directly*
//* by the user through tbSelectChar().                                        *
//*  a) nckSLEFT  (SHIFT + LeftArrow) key, or                                  *
//*  b) nclSRIGHT (SHIFT + RightArrow) key                                     *
//*  c) If mouse support is enabled, then the SHIFT key + ScrollWheel Up/Down  *
//*     are equivalent to selection keycodes.                                  *
//*                                                                            *
//* See the SetTextboxReservedKeys() method for defining the key combinations  *
//* to be used for Select operations and Copy/Cut/Paste operations.            *
//*                                                                            *
//*                                                                            *
//* Input  : selectAll: 'true'  select all text                                *
//*                     'false' deselect all text                              *
//*                                                                            *
//* Returns: OK if successful                                                  *
//*          or ERR if:                                                        *
//*             a) control contains no text data                               *
//******************************************************************************

short DialogTextbox::tbSelectText ( bool selectAll )
{
   short status = ERR ;

   if ( (this->lines == 1 && *this->wText != nckNULLCHAR) ||
        (this->lines > 1 && *this->mlText[ZERO].ln != nckNULLCHAR) )
   {
      //* Select all text data:                          *
      //* Activate selection and set all selection flags *
      if ( selectAll != false )
      {
         gString gs ;
         short cnt = (this->tbGetText ( gs )) - 1 ;
         this->select.set( cnt ) ;

         //* Set the cursor at the 'append' position (except for right- *
         //* justified fields, which have a fixed cursor position).     *
         if ( ! this->rightJust )
            this->tbSetCursor ( tbcpAPPEND ) ;
      }

      //* Cancel selection in progress:                     *
      //* Deactivate selection and reset all selection data *
      else
         this->select.reset() ; 

      this->RedrawControl ( true, true ) ;   // make changes visible
      status = OK ;                          // and report success
   }
   return status ;

}  //* End tbSelectText() *

//*************************
//*  tbWriteColorString   *
//*************************
//******************************************************************************
//* Called by RedrawControl(). If there is 'selected' text in the display area *
//* then write one character at a time using the appropriate color attribute.  *
//*                                                                            *
//* Input  : y     : line on which to write                                    *
//*          x     : x offset at which to begin write                          *
//*          wPtr  : pointer to source text                                    *
//*          dColor: base color attribute                                      *
//*                                                                            *
//* Returns: OK                                                                *
//******************************************************************************

short DialogTextbox::tbWriteColorString ( short y, short x, 
                                          const wchar_t* srcPtr, attr_t dColor )
{
   winPos wp( y, x ) ;
   attr_t uColor = dColor | ncuATTR ;
   gString gs( srcPtr ) ;
   gs.limitCols( this->cols ) ;
   int wchars ;
   const wchar_t* wptr = gs.gstr( wchars ) ;
   int j = ZERO ;
   if ( y > ZERO )      // adjust flag index for multi-line controls
   {
      gString gsm ;
      for ( short k = ZERO ; k < y ; k++ )
      {
         gsm = this->mlText[k].ln ;
         j += gsm.gschars() - 1 ;
      }
   }
   for ( int i = ZERO ; i < wchars ; i++, j++ )
      wp = this->wPtr->WriteChar ( wp, wptr[i], 
                                   (this->select.sel[j] ? uColor : dColor), 
                                   false, this->rtlContent ) ;
   return OK ;

}  //* End tbWriteColorString() *

//*************************
//*    SetOutputFormat    *
//*************************
//******************************************************************************
//* Set the text data output format: RTL (right-to-left), or                   *
//* LTR (left-to-right). Does not refresh display.                             *
//* Note that the default (unmodified) output format is LTR.                   *
//*                                                                            *
//* Input  : rtlFormat : if 'true',  then set as RTL format                    *
//*                      if 'false', then set as LTR format                    *
//*                                                                            *
//* Returns: OK  if successful                                                 *
//*          ERR if data format cannot be changed                              *
//*              a) if control is configured for right-justified input         *
//******************************************************************************

short DialogTextbox::SetOutputFormat ( bool rtlFormat )
{
   short status = ERR ;
   if ( this->rightJust == false )
   {
      gString gs ;                     // temp storage for multi-line data
      if ( this->lines > 1 )
         this->tbGetText ( gs ) ;
      this->rtlContent = rtlFormat ;
      this->tbIndex  = { ZERO, ZERO } ;// start at top of data
      this->leftIndex = ZERO ;
      if ( this->lines > 1 )           // re-format multi-line data
         this->mlFmtDisplay ( gs ) ;
      this->tbSynchCursor ( false ) ;  // calculate cursor position
      this->RedrawControl ( false, false ) ; // redraw, but do not re-display the data
      status = OK ;
   }
   return status ;

}  //* End SetOutputFormat() *

//*************************
//*     slInsertChar      *
//*************************
//******************************************************************************
//* Single-line Textboxes: Insert a character before character under cursor,   *
//* and cursor steps forward to next character.                                *
//*                                                                            *
//* Input  : wk   : character to be inserted                                   *
//*                 (printing character, input filter has been applied)        *
//*                                                                            *
//* Returns: 'true' if new character inserted                                  *
//*          'false' if character NOT inserted                                 *
//******************************************************************************
//* Programmer's Note: During testing, we have found that Arabic (at least)    *
//* can have up to three (3) modifier characters associated with a printing    *
//* character. This means that if an IME adds a character to the string, it    *
//* could consist of as many as 4, wchar_t characters. Thus, if the buffer     *
//* becomes filled before all the associated pieces are input, then the last   *
//* character in the string may be incomplete. This is unlikely, but possible. *
//******************************************************************************

bool DialogTextbox::slInsertChar ( const wkeyCode& wk )
{
   bool status = false ;                           // return value
   gString gsText( this->wText ) ;                 // get a copy of data
   if ( gsText.gschars() < (gsALLOCDFLT - 2) )     // if enough freespace
   {
      status = true ;                              // provisional success
      gsText.insert( wk.key, this->tbIndex.xpos) ; // insert the new character

      //* For fixed-width fields, test for overflow *
      if ( ! this->hzShift &&
           ((gsText.gscols()) > this->cols) )
         status = false ;

      if ( status )
      {  //* Save the new data, update display *
         //* and move cursor to next character *
         gsText.copy( this->wText, gsALLOCDFLT ) ;
         this->RedrawControl ( true ) ;
         this->slScrollNext () ;
      }
   }
   return status ;

}  //* End slInsertChar() *

//*************************
//*     slReplaceChar     *
//*************************
//******************************************************************************
//* Single-line Textboxes: Replace the character under the cursor, and cursor  *
//* steps to next character.                                                   *
//*                                                                            *
//* Input  : wk   : character to be inserted                                   *
//*                 (printing character, input filter has been applied)        *
//*                                                                            *
//* Returns: 'true' if new character replaced existing character               *
//*          'false' if character NOT replaced (see notes)                     *
//******************************************************************************
//* In most cases, we replace the indexed character.                           *
//* Exception: If indexed character is the NULLCHAR:                           *
//*  1) We append if there is space available.                                 *
//*  2) If no space available:                                                 *
//*     a) If !hzShift, replace last character in field.                       *
//*     b) Else overflow; return failure to replace.                           *
//*                                                                            *
//* Programmer's Note:                                                         *
//* If horizontal shift is disabled AND the field is full, then replacing      *
//* a one-column character with a two-column character would cause the last    *
//* character in the string to not be displayed. We prevent this by refusing   *
//* to over-fill a fixed-width field.                                          *
//*                                                                            *
//* Programmer's Note: During testing, we have found that Arabic (at least)    *
//* can have up to three (3) modifier characters associated with a printing    *
//* character. This means that if an IME adds a character to the string, it    *
//* could consist of as many as 4, wchar_t characters.                         *
//* In this case, replacing an existing character becomes less than            *
//* straightforward:                                                           *
//*   a) If the existing character consists of more elements than the new      *
//*      character, then artifacts may be left behind.                         *
//*   b) If the existing character consists of fewer elements than the new     *
//*      character, then all or part of the following character may be trashed.*
//* For this reason, it is strongly recommended that if Arabic or another      *
//* written language which uses 'modifier' characters is being input, then     *
//* the application should set 'Insert Mode' and prevent the user from         *
//* toggling to 'Overstrike Mode'. This can be done in the callback method     *
//* for the user-input loop.                                                   *
//******************************************************************************

bool DialogTextbox::slReplaceChar ( const wkeyCode& wk )
{
   bool status = false ;                           // return value
   gString gsText( this->wText ) ;                 // get a copy of data
   if ( gsText.gschars() < (gsALLOCDFLT - 2) )     // if enough freespace
   {
      status = true ;         // provisional success

      bool replace = true ;   // see note above
      if ( this->tbIndex.xpos >= (gsText.gschars() - 1) )
      {
         if ( ! this->hzShift )
         {
            if ( ((gsText.gscols()) >= this->cols) && (this->tbIndex.xpos > ZERO) )
               --this->tbIndex.xpos ;
            else
               replace = false ;
         }
         else
            replace = false ;
      }
      if ( replace )
         gsText.replace( this->wText[this->tbIndex.xpos], wk.key, this->tbIndex.xpos ) ;
      else
         gsText.append( wk.key ) ;

      //* For fixed-width field, test for overflow (see note) *
      if ( ! this->hzShift )
      {
         if ( (gsText.gscols()) > this->cols )
            status = false ;
      }

      if ( status != false )
      {
         gsText.copy(this->wText, gsALLOCDFLT ) ;  // save the modified data

         //* If cursor is on last column of field, shift next character into   *
         //* view. This is in case a one-column character has been replaced    *
         //* by a two-column character which we want to be visible.            *
         if ( (this->hzShift != false) &&
              (((this->rtlContent == false) && 
               (this->tbCursor.xpos == (this->cols - 1))) ||
               ((this->rtlContent != false) && (this->tbCursor.xpos == ZERO))) )
            ++this->leftIndex ;

         //* Update display and move cursor to next character. *
         this->RedrawControl ( true ) ;
         this->slScrollNext () ;
      }
   }
   return status ;

}  //* End slReplaceChar() *

//*************************
//*     slDeleteChar      *
//*************************
//******************************************************************************
//* Single-line Textboxes: Delete a character from the text string: either the *
//* character under the cursor OR the character which preceeds it.                                 *
//*                                                                            *
//* Input  : wk   : command key                                                *
//*                  nckDELETE: delete character under cursor                  *
//*                  nckBKSP  : delete character which preceeds the character  *
//*                             under the cursor                               *
//*                                                                            *
//* Returns: 'true'  if character deleted, else 'false'                        *
//******************************************************************************
//* Programmer's Note: There is a judgement call regarding what happens when   *
//* the user presses the Backspace key within an RTL field. In the author's    *
//* mindset, the Backspace key clearly goes LEFT, but it's possible that users *
//* of RTL languages, will expect it to go right -- and that is how we have    *
//* implemented it for RTL.                                                    *
//*                                                                            *
//* Programmer's Note: During testing, we have found that Arabic (at least)    *
//* can have up to three (3) modifier characters associated with a printing    *
//* character. This means that if we delete a printing character, we must also *
//* delete its associated modifier characters (if any).                        *
//*               CURRENTLY, THIS IS _NOT_ DONE.                               *
//******************************************************************************

bool DialogTextbox::slDeleteChar ( const wkeyCode& wk )
{
   bool status = false ;

   //* If we have data to be deleted *
   if ( ((wk.key == nckDELETE) && 
         (this->wText[this->tbIndex.xpos] != nckNULLCHAR)) ||
        ((wk.key == nckBKSP) && (this->tbIndex.xpos > ZERO)) )
   {
      gString gsText( this->wText ) ;

      //* Place cursor on character to be deleted. *
      if ( wk.key == nckBKSP )
      {
         this->slScrollPrev () ;
         if ( ((this->rtlContent == false) && 
               (this->tbIndex.xpos > ZERO) && (this->tbCursor.xpos == ZERO)) 
              ||
              ((this->rtlContent != false) && 
               (this->tbIndex.xpos > ZERO) && 
               (this->tbCursor.xpos == (this->cols - 1))) )
         {  //* This provides a visual cue to user *
            // NOTE: WE WOULD LIKE TO BRING IN A WHOLE WORD, IT'S MORE FRIENDLY.
            this->slScrollPrev () ;
            this->slScrollNext () ;
         }
      }
      else if ( (wk.key == nckDELETE) &&
                ((this->rtlContent == false && 
                  (this->tbCursor.xpos >= (this->cols - 2))) || 
                 (this->rtlContent != false && 
                  (this->tbCursor.xpos == ZERO))) )
      {  //* This provides a visual cue to user *
         // NOTE: WE WOULD LIKE TO BRING IN A WHOLE WORD, IT'S MORE FRIENDLY.
         this->slScrollNext () ;
         this->slScrollPrev () ;
      }

      //* Get characters that preceed the target character.*
      if ( this->tbIndex.xpos > ZERO )
         gsText.limitChars( this->tbIndex.xpos ) ;
      else
         gsText.clear() ;
      //* Append the characters that follow the target *
      //* character, and save to our data member.      *
      gsText.append( &this->wText[this->tbIndex.xpos + 1] ) ;
      gsText.copy( this->wText, gsALLOCDFLT ) ;

      //* Update display (cursor is already in position). *
      this->RedrawControl ( true ) ;

      status = true ;                            // return the good news
   }
   return status ;

}  //* End slDeleteChar() *

//*************************
//*     slScrollRight     *
//*************************
//******************************************************************************
//* Single-line Text boxes: Move cursor one character to the right.            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void DialogTextbox::slScrollRight ( void )
{
   //* LTR languages *
   if ( this->rtlContent == false )
   {
      gString gsText( this->wText ) ;
      if ( (this->tbIndex.xpos < (gsText.gschars() - 1))
           && ((this->hzShift != false) ||
               ((this->hzShift == false) && 
                ((this->tbIndex.xpos < (gsText.gschars() - 2)) ||
                ((gsText.gscols()) < this->cols)))) )
      {
         ++this->tbIndex.xpos ;        // reference the next charater
         this->tbCursor.xpos = ZERO ;  // reset cursor
         int wIndex = this->leftIndex,
             oldlindex = this->leftIndex,
             charCount ;
         const short* cPtr = gsText.gscols( charCount ) ;
         while ( wIndex < this->tbIndex.xpos )
         {  //* Advance the cursor position *
            this->tbCursor.xpos += cPtr[wIndex++] ;

            if ( this->tbCursor.xpos >= this->cols )
            {  //* Shift a character out the left to make room *
               this->tbCursor.xpos -= cPtr[this->leftIndex++] ;
            }
         }
         if ( this->leftIndex == oldlindex )
            this->wPtr->SetCursor ( this->tbCursor ) ; // position the cursor
         else
         {
            //* Erase old text and write new text *
            // NOTE: we rely on the WriteString() method to stop writing 
            //       characters when right edge of control is reached.
            this->wPtr->ClearWin () ;
            this->wPtr->WriteString ( this->tbCursor.ypos, ZERO, 
                                      &this->wText[this->leftIndex], this->fColor ) ;
         }
         this->wPtr->RefreshWin () ;                // update the display
      }
   }

   //* RTL languages *
   else
   {
      if ( this->tbIndex.xpos > ZERO )
      {
         --this->tbIndex.xpos ;           // reference the previous charater
   
         //* Determine whether the referenced character is currently visible, *
         //* and if not, shift it into view.                                  *
         if ( this->tbIndex.xpos < this->leftIndex )
         {
            --this->leftIndex ;
            this->RedrawControl ( true ) ;
         }
         this->tbSynchCursor ( true ) ;   // position the cursor
      }
   }

}  //* End slScrollRight() *

//*************************
//*     slScrollLeft      *
//*************************
//******************************************************************************
//* Single-line Textboxes: Move cursor one character to the left.              *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void DialogTextbox::slScrollLeft ( void )
{
   //* LTR languages *
   if ( this->rtlContent == false )
   {
      if ( this->tbIndex.xpos > ZERO )
      {
         gString gsText( this->wText ) ;
         int charCount ;
         const short* cPtr = gsText.gscols( charCount ) ;
         --this->tbIndex.xpos ;        // reference the previous charater
         if ( (this->tbCursor.xpos -= cPtr[this->tbIndex.xpos]) < ZERO )
         {  //* Shift next character in from left.*
            --this->leftIndex ;
            this->tbCursor.xpos = ZERO ;
            //* Erase old text and write new text *
            this->wPtr->ClearWin () ;
            this->wPtr->WriteString ( this->tbCursor.ypos, ZERO, 
                                      &this->wText[this->leftIndex], this->fColor ) ;
         }
         this->wPtr->SetCursor ( this->tbCursor ) ; // position the cursor
         this->wPtr->RefreshWin () ;                // update the display
      }
   }

   //* RTL languages *
   else
   {
      gString gsText( this->wText ) ;
      if ( (this->tbIndex.xpos < (gsText.gschars() - 1))
           && ((this->hzShift != false) ||
               ((this->hzShift == false) && 
                ((this->tbIndex.xpos < (gsText.gschars() - 2)) ||
                ((gsText.gscols()) < this->cols)))) )
      {
         ++this->tbIndex.xpos ;           // reference the next charater
   
         //* Determine whether the referenced character is currently visible, *
         //* and if not, shift it into view.                                  *
         int charCount,
             colCount = ZERO,
             cIndex = this->leftIndex ;
         const short* cPtr = gsText.gscols( charCount ) ;
         while ( cIndex < this->tbIndex.xpos )
            colCount += cPtr[cIndex++] ;
         if ( colCount >= this->cols )
         {
            ++this->leftIndex ;
            this->RedrawControl ( true ) ;
         }
         this->tbSynchCursor ( true  ) ;  // position the cursor
      }
   }

}  //* End slScrollLeft() *

//*************************
//*     slScrollHome      *
//*************************
//******************************************************************************
//* Single-line Textboxes: Scroll until we reach the HOME position             *
//* (reference top-of-data).                                                   *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void DialogTextbox::slScrollHome ( void )
{
   while ( this->tbIndex.xpos > ZERO )
      this->slScrollPrev () ;

}  //* End slScrollHome() *

//*************************
//*     slScrollEnd       *
//*************************
//******************************************************************************
//* Single-line Textboxes: Scroll until we reach the 'append' position         *
//* (reference end-of-data).                                                   *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void DialogTextbox::slScrollEnd ( void )
{
   //* Horizontal shift is enabled, OR field is not full, then go to 'append' *
   //* position, (index the null terminator). Else, index last printing char. *
   gString gsText( this->wText ) ;
   short loopy = gsText.gschars() - 1 ;   // loop counter
   if ( (! this->hzShift) && (gsText.gscols() == this->cols) )
      --loopy ;

   while ( this->tbIndex.xpos < loopy )
      this->slScrollNext () ;

}  //* End slScrollEnd() *

//*************************
//*     slScrollNext      *
//*************************
//******************************************************************************
//* Single-line Textboxes: Move cursor to next character.                      *
//* For LTR data, this is character to right.                                  *
//* For RTL data, this is character to left.                                   *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if already at end-of-data, else 'false'                    *
//******************************************************************************

bool DialogTextbox::slScrollNext ( void )
{
   bool tod = true ;
   if ( this->wText[this->tbIndex.xpos] != nckNULLCHAR )
   {
      if ( this->rtlContent == false )    // LTR
         this->slScrollRight () ;
      else                                // RTL
         this->slScrollLeft () ;
      tod = false ;
   }
   return tod ;

}  //* End slScrollNext() *

//*************************
//*     slScrollPrev      *
//*************************
//******************************************************************************
//* Single-line Textboxes: Move cursor to previous character.                  *
//* For LTR data, this is character to left.                                   *
//* For RTL data, this is character to right.                                  *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if already at top-of-data, else 'false'                    *
//******************************************************************************

bool DialogTextbox::slScrollPrev ( void )
{
   bool eod = true ;
   if ( this->tbIndex.xpos > ZERO || this->leftIndex > ZERO )
   {
      if ( this->rtlContent == false )    // LTR
         this->slScrollLeft () ;
      else                                // RTL
         this->slScrollRight () ;
      eod = false ;
   }
   return eod ;

}  //* End slScrollPrev() *

//*************************
//*     rjInsertChar      *
//*************************
//******************************************************************************
//* Single-line, right-justified Textboxes:                                    *
//* Insert a character at cursor position, shifting any existing data leftward.*
//*                                                                            *
//* Input  : wk   : character to be inserted                                   *
//*                 (printing character, input filter has been applied)        *
//*                                                                            *
//* Returns: 'true'  if character successfully inserted,                       *
//*          'false' if buffer is full (no action taken)                       *
//******************************************************************************

bool DialogTextbox::rjInsertChar ( const wkeyCode& wk )
{
   bool status = false ;            // return value

   //* Determine number of columns required by new character *
   //* and number of free columns available.                 *
   gString gsText ;
   gsText.append( wk.key ) ;
   short neededCols = gsText.gscols() ;
   gsText = this->wText ;
   short availableCols = this->cols - gsText.gscols() ;
   if ( neededCols <= availableCols )
   {
      gsText.append( wk.key ) ;        // append the new character
      gsText.copy( this->wText, gsALLOCDFLT ) ; // save the data
      this->tbSynchCursor ( false ) ;  // calculate positioning of text
      this->RedrawControl ( true ) ;   // make the changes visible
      status = true ;                  // return the good news
   }
   return status ;

}  //* End rjInsertChar() *

//*************************
//*     rjDeleteChar      *
//*************************
//******************************************************************************
//* Single-line, right-justified Textboxes:                                    *
//* Delete the character under the cursor (rightmost displayed character), and *
//* shift all other data to the right to fill vacated space.                   *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true'  if character successfully deleted                         *
//*          'false' if buffer is empty (no action taken)                      *
//******************************************************************************

bool DialogTextbox::rjDeleteChar ( void )
{
   bool status = false ;            // return value
   gString gsText( this->wText ) ;
   if ( gsText.gschars() > 1 )
   {
      gsText.limitChars( gsText.gschars() - 2 ) ; // delete last character
      gsText.copy( this->wText, gsALLOCDFLT ) ; // save the data
      this->tbSynchCursor ( false ) ;  // calculate positioning of text
      this->RedrawControl ( true ) ;   // make the changes visible
      status = true ;                  // return the good news
   }
   return status ;

}  //* End rjDeleteChar() *

//*************************
//*     mlScrollRight     *
//*************************
//******************************************************************************
//* Multi-line Textboxes: Move cursor one character to the right.              *
//*                                                                            *
//* Input  : forceLTR: (optional, false by default)                            *
//*                    If 'true', then we have been called by mlScrollLeft()   *
//*                     AND the data are RTL. (see note in mlScrollLeft())     *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void DialogTextbox::mlScrollRight ( bool forceLTR )
{
   //* LTR languages AND not already at end of data stream *
   if ( ((this->rtlContent == false) && !(this->mlEoD ())) || forceLTR )
   {
      gString gs( this->mlText[this->tbIndex.ypos].ln ) ; // text for this line
      int lCols = gs.gscols (),        // occupied columns on this line
          cCols = ZERO,                // columns from left to current position
          lChars ;                     // characters on this line
      const short* colPtr = gs.gscols( lChars ) ;
      for ( short i = ZERO ; i < lChars && i <= this->tbIndex.xpos ; i++ )
         cCols += colPtr[i] ;

      //* If not yet at end of current line, move right by one character, *
      //* to reference either a visible character or the nullchar.        *
      if ( (cCols < lCols) || 
           ((cCols == lCols && lCols < this->cols) &&
            (this->tbIndex.xpos < (lChars - 1))) )
      {
         ++this->tbIndex.xpos ;
         this->tbSynchCursor ( true ) ;   // update cursor position
      }

      //* At end of data for current line. If next line is visible, *
      //* go to first character (or 'append') on next line.         *
      else if (this->tbIndex.ypos < (this->lines - 1) )
      {
         ++this->tbIndex.ypos ;
         this->tbIndex.xpos = ZERO ;
         this->tbSynchCursor ( true ) ;   // update cursor position
      }

      //* Cursor is at end of last display line.           *
      //* Move data from first display line into 'mlHead', *
      //* and scroll all other lines upward. Then bring    *
      //* in a line of data from 'mlTail'.                 *
      else
//OLD      else if ( this->hzShift != false )
      {
         gString gs( this->mlText[ZERO].ln ) ;  // existing line 0 data
         short firstChars = gs.gschars() - 1 ;  // number of characters on 1st line
         this->tbGetText ( gs ) ;               // copy of all data
         this->leftIndex += firstChars ;        // new count of head characters
         this->mlFmtDisplay ( gs ) ;            // reformat the data
         this->tbIndex.xpos = ZERO ;            // cursor to beginning of line
         this->tbSynchCursor ( false ) ;        // update cursor position
         this->RedrawControl ( true ) ;         // update the display
      }
   }     // LTR languages

   //* RTL languages AND not already at top of data stream *
   else if ( (this->rtlContent != false) && !(this->mlToD ()) )
   {
      //* This is a trick to save codespace. See note in 'mlScrollLeft' header.*
      this->mlScrollLeft ( true ) ;
   }     // RTL languages

}  //* End mlScrollRight() *

//*************************
//*     mlScrollLeft      *
//*************************
//******************************************************************************
//* Multi-line Textboxes: Move cursor one character to the left.               *
//*                                                                            *
//* Input  : forceLTR: (optional, false by default)                            *
//*                    If 'true', then we have been called by mlScrollRight()  *
//*                     AND the data are RTL. (see note below)                 *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note: We play a trick with cursor movement for RTL data:      *
//* The 'scroll-left' code for RTL data is the same code as 'scroll-right'     *
//* for LTR data, AND the 'scroll-right' code for RTL data is the same code as *
//* 'scroll-left' for LTR data. The trick works because all cursor movement is *
//* actually done by adjusting 'tbIndex' and THEN calculating cursor position. *
//******************************************************************************

void  DialogTextbox::mlScrollLeft ( bool forceLTR )
{
   //* LTR languages AND not already at top of data stream *
   if ( ((this->rtlContent == false) && !(this->mlToD ())) || forceLTR )
   {
      //* If not at left edge of current line, move left by one character.*
      if ( this->tbIndex.xpos > ZERO )
      {
         --this->tbIndex.xpos ;
         this->tbSynchCursor ( true ) ;   // update cursor position
      }
      //* Else if not on first display line,      *
      //* move to last character of previous line.*
      else if ( this->tbIndex.ypos > ZERO )
      {
         --this->tbIndex.ypos ;
         for ( this->tbIndex.xpos = ZERO ;  // index the NULLCHAR
               (this->mlText[this->tbIndex.ypos].ln[this->tbIndex.xpos] != NULLCHAR) ;
               ++this->tbIndex.xpos ) ;
         this->tbSynchCursor ( true ) ;   // update cursor position
      }
      else if ( (this->tbIndex.ypos == ZERO) && (this->leftIndex > ZERO) )
      {  //* Shift a character from the 'head' data into view,    *
         //* and reformat the text. Cursor position is unchanged. *
         gString gs ;
         this->tbGetText ( gs ) ;
         --this->leftIndex ;
         this->mlFmtDisplay ( gs ) ;
         this->RedrawControl ( true ) ;   // update the display
      }
   }     // LTR languages

   //* RTL languages AND not already at end of data stream *
   else if ( (this->rtlContent != false) && !(this->mlEoD ()) )
   {
      //* This is a trick to save codespace. See note in method header.*
      this->mlScrollRight ( true ) ;
   }     // RTL languages

}  //* End mlScrollLeft() *

//*************************
//*      mlScrollUp       *
//*************************
//******************************************************************************
//* Multi-line Textboxes: Move cursor one line up.                             *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Ideally, we will find the corresponding column on the new line; however,   *
//* if that column is not on a character boundary, we position on the column   *
//* of the previous character.                                                 *
//******************************************************************************

void DialogTextbox::mlScrollUp ( void )
{
   if ( !(this->mlToD ()) )
   {
      //* Move cursor to beginning of current line.*
      winPos oldCursor = this->tbCursor ;    // remember old cursor position
      while ( this->tbIndex.xpos > ZERO )
         this->mlScrollPrev () ;

      //* If cursor is not on the first display line, just move cursor.*
      if ( this->tbIndex.ypos > ZERO )
      {
         //* Move cursor through previous line until matching 'xpos'.*
         do
         {
            if ( (this->mlScrollPrev()) != false )
               break ;
         }
         while ( (this->tbIndex.xpos > ZERO) && 
                 (((this->rtlContent == false) && 
                   (this->tbCursor.xpos > oldCursor.xpos))
                  ||
                  ((this->rtlContent != false) && 
                   (this->tbCursor.xpos < oldCursor.xpos))) ) ;
      }

      //* Else, cursor is on first line. If undisplayed *
      //* data in 'mlHead', bring in a lineful.         *
      else if ( this->leftIndex > ZERO )
      {
         gString gs( this->mlText[ZERO].ln ) ;
         short lChars = gs.gschars() - 1,
               mChars = ZERO ;
         for ( ; mChars < lChars && !(this->mlToD ()) ; mChars++ )
            this->mlScrollPrev () ;
         do
         {
            if ( (this->mlScrollNext()) != false )
               break ;
         }
         while ( !(this->mlEoL ()) &&
                 (((this->rtlContent == false) && 
                   (this->tbCursor.xpos < oldCursor.xpos))
                  ||
                  ((this->rtlContent != false) &&
                   (this->tbCursor.xpos > oldCursor.xpos))) ) ;
      }
   }

}  //* End mlScrollUp() *

//*************************
//*     mlScrollDown      *
//*************************
//******************************************************************************
//* Multi-line Textboxes: Move cursor one line down.                           *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* - If there is visible data on the next line corresponding to the X offset  *
//*   of the current line, then ideally, we will find the  corresponding       *
//*   column on the new line; however, if that column is not on a character    *
//*   boundary, we position on the column  of the previous character.          *
//* - If the next line's data are narrower than current X offset, then we go   *
//*   to 'append' position of next line.                                       *
//* If already on last display line, then shift in a line of data from tail    *
//* (if any), then set cursor according to above criteria.                     *
//******************************************************************************

void DialogTextbox::mlScrollDown ( void )
{
   if ( !(this->mlEoD ()) )
   {
      winPos oldCursor = this->tbCursor ;    // remember old cursor position

      //* Move cursor to 'append' position of current line *
      while ( !(this->mlEoL ()) )
         this->mlScrollNext () ;

      //* Move cursor to equivalent offset into next line *
      if ( !(this->mlEoD ()) )
      {
         do
         {
            if ( (this->mlScrollNext()) != false )
               break ;
         }
         while ( !(this->mlEoL ()) && 
                 (((this->rtlContent == false) && 
                   (this->tbCursor.xpos < oldCursor.xpos))
                  ||
                  ((this->rtlContent != false) && 
                   (this->tbCursor.xpos > oldCursor.xpos))) ) ;
         //* If we overshot, then go back one character.*
         if ( (this->rtlContent == false && this->tbCursor.xpos > oldCursor.xpos) ||
              (this->rtlContent != false && this->tbCursor.xpos < oldCursor.xpos) )
            this->mlScrollPrev () ;
      }
   }

}  //* End mlScrollDown() *

//*************************
//*     mlScrollHome      *
//*************************
//******************************************************************************
//* Multi-line Textboxes: Move cursor to 'home' position.                      *
//* This is a 3-stage operation:                                               *
//* 1) If cursor is not at beginning of current line, then set cursor to       *
//*    beginning of current line.                                              *
//* 2) If cursor is at beginning of current line, BUT current line is not the  *
//*    first line, then set cursor to beginning of first line.                 *
//* 3) If cursor is at beginning of first line, BUT not at top-of-data, then   *
//*    go to top-of-data.                                                      *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void DialogTextbox::mlScrollHome ( void )
{
   //* If not already at beginning of current line, go there.*
   if ( this->tbIndex.xpos > ZERO )
      this->tbIndex.xpos = ZERO ;

   //* Else if current line is not first line, go to beginning of first line.*
   else if ( this->tbIndex.ypos > ZERO )
      this->tbIndex.ypos = ZERO ;

   //* Else, if not already at top-of-data, go there.*
   else if ( this->leftIndex > ZERO )
   {
      gString gs ;
      this->tbGetText ( gs ) ;
      this->leftIndex = ZERO ;
      this->mlFmtDisplay ( gs ) ;
      this->RedrawControl ( true ) ;   // update the display
   }
   this->tbSynchCursor ( true ) ;      // update cursor position

}  //* End mlScrollHome() *

//*************************
//*      mlScrollEnd      *
//*************************
//******************************************************************************
//* Multi-line Textboxes: Move cursor to 'append' position.                    *
//* This is a 3-stage operation:                                               *
//* 1) If cursor is not at end of current line, then move cursor to end of     *
//*    current line.                                                           *
//* 2) If cursor is at end of current line, BUT current line is not the last   *
//*    occupied line, then set cursor to end of last occupied line.            *
//*    This may, or may not also be end-of-data.                               *
//* 3) If cursor is at end of last occupied line, BUT not at end-of-data,      *
//*    then go to end-of-data.                                                 *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void DialogTextbox::mlScrollEnd ( void )
{
   if ( !(this->mlEoD ()) )
   {
      //* If not at end of current line, go there.*
      if ( !(this->mlEoL ()) )
      {
         while ( !(this->mlEoL ()) )
            this->mlScrollNext () ;
      }

      //* We are at EoL but not at EoD; therefore go to end *
      //* of line which includes the 'append' position, or  *
      //* end of last visible line, whichever comes first.  *
      else if ( (*this->mlTail != NULLCHAR) && 
                (this->tbIndex.ypos < (this->lines - 1)) )
      {
         while ( this->tbIndex.ypos < (this->lines - 1) )
            this->mlScrollNext () ;
         while ( !(this->mlEoL ()) )
            this->mlScrollNext () ;
      }
      else
      {
         while ( !(this->mlEoD ()) )
            this->mlScrollNext () ;
      }
   }

}  //* End mlScrollEnd() *

//*************************
//*     mlScrollNext      *
//*************************
//******************************************************************************
//* Multi-line Textboxes: Move cursor to next character.                       *
//* For LTR data, this is character to right.                                  *
//* For RTL data, this is character to left.                                   *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if already at end-of-data, else 'false'                    *
//******************************************************************************

bool DialogTextbox::mlScrollNext ( void )
{
   bool eod = true ;
   if ( !(this->mlEoD ()) )
   {
      if ( this->rtlContent == false )    // LTR
         this->mlScrollRight () ;
      else                                // RTL
         this->mlScrollLeft () ;
      eod = false ;
   }
   return eod ;

}  //* End mlScrollNext() *

//*************************
//*     mlScrollPrev      *
//*************************
//******************************************************************************
//* Multi-line Textboxes: Move cursor to previous character.                   *
//* For LTR data, this is character to left.                                   *
//* For RTL data, this is character to right.                                  *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if already at top-of-data, else 'false'                    *
//******************************************************************************

bool DialogTextbox::mlScrollPrev ( void )
{
   bool tod = true ;
   if ( !(this->mlToD ()) )
   {
      if ( this->rtlContent == false )    // LTR
         this->mlScrollLeft () ;
      else                                // RTL
         this->mlScrollRight () ;
      tod = false ;
   }  
   return tod ;

}  //* End mlScrollPrev() *

//*************************
//*      mlIndex2Ip       *
//*************************
//******************************************************************************
//* Multi-line Textboxes: Calculate the insertion point in the text stream     *
//* based on 'tbIndex' member. Corresponds to the character under the cursor.  *
//*                                                                            *
//* Input  : gsText : source text (source text must contain NEWLINE characters *
//*                   to correctly calculate linebreaks)                       *
//*                                                                            *
//* Returns: offset into gsText indicating character under cursor:             *
//*          gsText.gstr()[iPoint] == character under cursor                   *
//******************************************************************************

short DialogTextbox::mlIndex2Ip ( const gString& gsText )
{
   const wchar_t* wPtr = gsText.gstr() ;
   short wIndex = ZERO, 
         lIndex = ZERO,
         iPoint = ZERO ;

   //* Step over the 'head' data (if any) *
   if ( this->leftIndex > ZERO )
      lIndex = wIndex += this->leftIndex + 1 ;

   //* Reference first character of line where cursor currently located *
   for ( short yIndex = ZERO ; yIndex < this->tbIndex.ypos ; yIndex++ )
   {
      while ( wPtr[wIndex] != nckNULLCHAR )
      {
         ++lIndex ;
         if ( wPtr[wIndex++] == nckNEWLINE )
            break ;
      }
   }
   //* Reference character under cursor *
   iPoint = lIndex + this->tbIndex.xpos ;
   
   return iPoint ;

}  //* End mlIndex2Ip() *

//*************************
//*      mlIp2Index       *
//*************************
//******************************************************************************
//* Multi-line Textboxes: Calculate new 'tbIndex' and 'tbCursor' based on      *
//* specified insertion point.                                                 *
//*                                                                            *
//* Input  : iPoint  : indicates the character in the text stream which        *
//*                    should be under the cursor.                             *
//*                                                                            *
//* Returns: nothing : 'tbIndex' member is adjusted and                        *
//*                    'tbCursor' member is recalculated                       *
//******************************************************************************

void DialogTextbox::mlIp2Index ( short iPoint )
{
   gString gsText ;
   this->tbGetText ( gsText, false ) ;    // get a copy of text (with newlines)
   int   wIndex = ZERO,                   // source index
         ip = ZERO,                       // new insertion point
         charCount ;                      // source characters
   short y = ZERO, x = ZERO ;             // Y/X indices
   const wchar_t* wPtr = gsText.gstr( charCount ) ;  // reference the text
   wchar_t wChar ;                        // character under analysis

   //* Step over the 'head' data (if any) *
   if ( this->leftIndex > ZERO )
      ip = wIndex += this->leftIndex + 1 ;

   while ( ip < iPoint && wIndex < charCount )
   {
      wChar = wPtr[wIndex++] ;
      ++x ;
      if ( ++ip <= iPoint )
      {
         if ( wChar == nckNEWLINE )
         {  //* End of current line, but not end of data *
            if ( y < (this->lines - 1) )
            {
               ++y ;                // reference next display line
               x = ZERO ;           // reset the horizontal index
            }
            else        // end of display area
               break ;  // (remaining data belongs to tail, which we ignore here)
         }
      }
   }
   this->tbIndex = { y, x } ;       // update character index
   this->tbSynchCursor ( true ) ;   // update cursor position

}  //* End mlIp2Index() *

//*************************
//*     mlInsertChar      *
//*************************
//******************************************************************************
//* Multi-line Textboxes: Insert a character at the cursor position.           *
//*  - Character under cursor and all characters following it are shifted one  *
//*    position toward end-of-data.                                            *
//*  - New character is inserted at current cursor position.                   *
//*  - Cursor is positioned on the character FOLLOWING the inserted character. *
//*  - Text data for the control are reformatted to update word wrapping.      *
//*  - Display is refreshed.                                                   *
//*                                                                            *
//* Input  : wk  : keytype/keycode of character to be inserted                 *
//*                (input filter has already been applied)                     *
//*                                                                            *
//* Returns: 'true'  if character successfully inserted,                       *
//*          'false' if buffer is full  (data are not modified)                *
//******************************************************************************

bool DialogTextbox::mlInsertChar ( const wkeyCode& wk )
{
   gString gsText ;
   bool status = false ;         // return value

   //* Get a copy of existing data (with newline delimiters) *
   this->tbGetText ( gsText, false ) ;
   if ( (gsText.gschars()) < (gsALLOCDFLT - 2) ) // prevent potential buffer overrun
   {
      gsText = this->mlText[this->tbIndex.ypos].ln ; // current line's text
      gsText.insert( wk.key, this->tbIndex.xpos) ;
      gsText.copy( this->mlText[this->tbIndex.ypos].ln, mtLENGTH ) ;
      this->tbGetText ( gsText ) ;           // get a copy of the modified data
      this->mlFmtDisplay ( gsText, true ) ;  // reformat data with cursor tracking
      this->RedrawControl ( true ) ;         // update the display
      this->mlScrollNext () ;                // scroll to next character
      status = true ;         // provisional success
   }
   return status ;

}  //* End mlInsertChar() *

//*************************
//*     mlReplaceChar     *
//*************************
//******************************************************************************
//* Multi-line Textboxes: Replace the character under the cursor.              *
//*  - New character is inserted at current cursor position, overwriting the   *
//*    existing character.                                                     *
//*  - Cursor is positioned AFTER the inserted character.                      *
//*  - Display is refreshed.                                                   *
//*                                                                            *
//* Input  : wk  : keytype/keycode of character to replace existing character  *
//*                (input filter has already been applied)                     *
//*                                                                            *
//* Returns: 'true' if new character replaced existing character               *
//*          'false' if character NOT replaced (unlikely)                      *
//******************************************************************************
//* In most cases, we replace the indexed character.                           *
//* Exception: If indexed character is the NULLCHAR:                           *
//*  1) We append if there is space available.                                 *
//*  2) If no space available:                                                 *
//*     a) If !hzShift, replace last character in field.                       *
//*     b) Else overflow; return failure to replace.                           *
//*                                                                            *
//* Programmer's Note:                                                         *
//* If horizontal shift is disabled AND the field is full, then replacing      *
//* a one-column character with a two-column character would cause the last    *
//* character in the string to not be displayed. We prevent this by refusing   *
//* to over-fill a fixed-width field.                                          *
//*                                                                            *
//******************************************************************************

bool DialogTextbox::mlReplaceChar ( const wkeyCode& wk )
{
   gString gsText ;
   bool status = false ;         // return value

   //* Get a copy of existing data (with newline delimiters) *
   this->tbGetText ( gsText, false ) ;
   if ( (gsText.gschars()) < (gsALLOCDFLT - 2) ) // prevent potential buffer overrun
   {
      status = true ;         // provisional success

      //* Get insertion point (index of character under cursor) *
      short iPoint = this->mlIndex2Ip ( gsText ) ;

      //* If referencing the NULL terminator, append instead of replace *
      bool replace = (iPoint < (gsText.gschars() - 1)) ;
      if ( replace )
         gsText.replace( gsText.gstr()[iPoint], wk.key, iPoint ) ;
      else
         gsText.append( wk.key ) ;

      //* Strip the newlines from the data *
      while ( (gsText.erase( L'\n' )) >= ZERO ) ;

      //* Save the new data *
      this->mlFmtDisplay ( gsText, true ) ; // reformat data with cursor tracking
      this->RedrawControl ( true ) ;      // update the display
      this->mlScrollNext () ;             // advance cursor
   }
   return status ;

}  //* End mlReplaceChar() *

//*************************
//*     mlDeleteChar      *
//*************************
//******************************************************************************
//* Multi-line Textboxes: Delete a character.                                  *
//*  1) If nckDELETE key, delete character under cursor.                       *
//*     Cursor' will reference the character that followed deleted character.  *
//*                                                                            *
//*  2) If nckBKSP key, then delete character BEFORE the referenced character. *
//*     Cursor will reference the same character (incl. NULLCHAR) it did       *
//*     before the deletion.                                                   *
//*                                                                            *
//* Input  : wk     : command character, either nckBKSP or nckDELETE           *
//*                                                                            *
//* Returns: 'true'  if character successfully deleted                         *
//*          'false' if no character available for deletion                    *
//*                  - for nckDELETE, if cursor references end-of-data.        *
//*                  - for nckBKSP, if cursor references top-of-data.          *
//******************************************************************************

bool DialogTextbox::mlDeleteChar ( const wkeyCode& wk )
{
   bool status = false ;
   if ( (wk.key == nckBKSP && !(this->mlToD ())) ||
        (wk.key == nckDELETE && !(this->mlEoD ())) )
   {
      short oldY = this->tbIndex.ypos ;   // remember which line we're on

      //* If backspace key, then reference the character to be deleted. *
      if ( wk.key == nckBKSP )
      {
         //* If not at beginning of line, move to previous character. *
         if ( this->tbIndex.xpos > ZERO )
            this->mlScrollPrev () ;

         //* Else, if at top left corner, then character to be deleted is not  *
         //* currently in view. Scroll in a word from 'mlHead'. We know that   *
         //* mlHead contains data because we verified !top-of-data above.      *
         else if ( this->tbIndex.ypos == ZERO )
         {
            gString gsHead( this->mlHead ) ;
            int headChars, i ;
            const wchar_t* wPtr = gsHead.gstr( headChars ) ;
            for ( i = headChars - 2 ; i >= ZERO ; )
            {
               this->mlScrollPrev () ;
               if ( i > ZERO && wPtr[--i] == nckSPACE )
                  break ;
            }
            while ( ++i < (headChars - 2) )
               this->mlScrollNext () ;
         }
         //* Else, scroll to end of previous line.                         *
         //* If end-of-line marker is referenced, go to previous character.*
         else
         {
            this->mlScrollPrev () ;
            if ( this->mlText[this->tbIndex.ypos].ln[this->tbIndex.xpos] == nckNULLCHAR )
               this->mlScrollPrev () ;
            --oldY ;    // we're now on the line above
         }
      }

      //* If delete key, and end-of-line marker is *
      //* referenced, go to start of next line.    *
      else
      {
         if ( this->mlText[this->tbIndex.ypos].ln[this->tbIndex.xpos] == nckNULLCHAR )
            this->mlScrollNext () ;
      }

      //* Get the data Before and After the target character. *
      gString gsBefore( this->mlText[this->tbIndex.ypos].ln, this->tbIndex.xpos ),
              gsAfter( &this->mlText[this->tbIndex.ypos].ln[this->tbIndex.xpos + 1] ) ;
      if ( this->tbIndex.xpos > ZERO )
         gsBefore.append( gsAfter.gstr() ) ;
      else
         gsBefore = gsAfter ;
      gsBefore.copy( this->mlText[this->tbIndex.ypos].ln, mtLENGTH ) ;
      this->tbGetText ( gsBefore ) ;         // get a copy of the modified data
      this->mlFmtDisplay ( gsBefore, true ) ;// reformat data with cursor tracking
      this->RedrawControl ( true ) ;         // update the display
      this->tbSynchCursor ( true ) ;         // update cursor position (to be safe)

      //* There is a certain messiness in automagic word-wrapping.    *
      //* Compensate if we have just crossed an end-of-line boundary, *
      //* BUT we are not on the end-of-line marker.                   *
      if ( (this->tbIndex.ypos < oldY) &&
           (this->mlText[this->tbIndex.ypos].ln[this->tbIndex.xpos] != nckNULLCHAR) )
         this->mlScrollPrev () ;
      status = true ;
   }
   return status ;

}  //* End mlDeleteChar() *

//*************************
//*         mlToD         *
//*************************
//******************************************************************************
//* Determine whether index is at Top-of-Data.                                 *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if index is at top-of-data, else 'false'                   *
//******************************************************************************

bool DialogTextbox::mlToD ( void )
{
   bool tod = false ;
   if ( this->tbIndex.ypos == ZERO &&     // line ZERO
        this->tbIndex.xpos == ZERO &&     // leftmost column
        this->leftIndex == ZERO )         // no 'head' data
      tod = true ;
   return tod ;

}  //* End mlToD() *

//*************************
//*         mlEoD         *
//*************************
//******************************************************************************
//* Determine whether index is at End-of-Data.                                 *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if index is at end-of-data, else 'false'                   *
//******************************************************************************

bool DialogTextbox::mlEoD ( void )
{
   bool eod = false ;
   if ( *this->mlTail == nckNULLCHAR )    // if no 'tail' data
   {
      if ( ((this->tbIndex.ypos == (this->lines - 1)) ||
           ((this->tbIndex.ypos < (this->lines - 1)) &&
            (*this->mlText[this->tbIndex.ypos + 1].ln == nckNULLCHAR))) &&
            (this->mlText[this->tbIndex.ypos].ln[this->tbIndex.xpos] 
                                                               == nckNULLCHAR) )
         eod = true ;
   }
   return eod ;

}  //* End mlEoD() *

//*************************
//*         mlEoL         *
//*************************
//******************************************************************************
//* Determine whether cursor references end of current line.                   *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if index is at end-of-line, else 'false'                   *
//******************************************************************************

bool DialogTextbox::mlEoL ( void )
{
   bool eol = false ;
   if ( this->mlText[this->tbIndex.ypos].ln[this->tbIndex.xpos] == nckNULLCHAR )
      eol = true ;
   else if ( (this->mlText[this->tbIndex.ypos].ln[this->tbIndex.xpos + 1] 
              == nckNULLCHAR) )
   {
      #if 0    // DISABLED
      // NOTE: This is untested because we don't currently allow a line 
      //       to be completely filled.
      gString gs( this->mlText[this->tbIndex.ypos].ln ) ;
      if ( gs.gscols() == this->cols )
         eol = true ;
      #endif   // DISABLED
   }
   return eol ;

}  //* End mlEoL() *

//*************************
//*     mlFmtDisplay      *
//*************************
//******************************************************************************
//* Multi-line Textboxes: Format the text stream, parsing it in discrete       *
//*                       display lines.                                       *
//*                                                                            *
//* Input  : gsText : source text to be formatted and displayed                *
//*          ipCalc : (optional, 'false' by default)                           *
//*                   if 'false', then caller will calculate new iPoint        *
//*                   if 'true',  then calculate new tbIndex and tbCursor      *
//*                               (requires gsText to have hard linebreaks)    *
//*          hBreak : (optional, 'false' by default)                           *
//*                   if 'false', then ignore hard line breaks                 *
//*                   if 'true',  then use hard line breaks if present         *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* - Displayed text is stored in the 'mlText' array.                          *
//* - Un-displayed text that preceeds the displayed text is stored in          *
//*   'mlHead' (aka 'wText') member.                                           *
//* - Un-displayed text that follows the displayed text is stored in           *
//*   'mlTail' member.                                                         *
//* -The combined size of of text in these three storage areas may be up to    *
//*  gsALLOCDFLT (minus space for linebreaks). If during editing the text grows*
//*  beyond this size, the excess will be truncated. (This is rather unlikely, *
//*  but possible.)                                                            *
//*                                                                            *
//* Like single-line textbox controls, user may scroll the cursor to to bring  *
//* the desired section of the text into view.                                 *
//*                                                                            *
//* The beginning of the DISPLAYED text is indicated by the 'leftIndex' data   *
//* member. If leftIndex==ZERO, then there is no valid data in 'mlHead'.       *
//*                                                                            *
//*  mlHead       ->   [gsALLOCDFLT]  un-displayed head data                   *
//*                +-  line 0                                                  *
//*                |   line 1                                                  *
//*  mlText[].ln  -|   line 2                                                  *
//*                |   ...           data for 'n' display lines stored here    *
//*                |   line n - 1                                              *
//*                +-  line n                                                  *
//*  mlTail       ->   [gsALLOCDFLT]  un-displayed tail data                   *
//*                                                                            *
//* Note on line breaks:                                                       *
//*  - A 'soft' line break is one which automagically breaks a line at the     *
//*    beginning of a word that would otherwise extend beyond the display area.*
//*    If the space character before the word will not fit on the current line,*
//*    it is wrapped with the word that follows it.                            *
//*  - A 'hard' line break is indicated by a newline character ('\n') in the   *
//*    text stream.                                                            *
//*  - By default, this method ignores hard breaks. Also, if hard break use is *
//*    specified but would mess up the display, we disregard them.             *
//*    Although at this time (May, 2014) we can't think of a reason to use     *
//*    hard breaks here, it is available through optional 'hBreak' parameter.  *
//*                                                                            *
//* Note on index and cursor tracking:                                         *
//*  - The 'tbIndex' member references the character under the cursor.         *
//*  - The 'tbCursor' member specifies the cursor position of that character.  *
//*  - Both 'tbIndex' and 'tbCursor' are affected by the reformatting, so      *
//*    these members must be updated to reflect any change in position.        *
//*    Either the caller can ask us to do it here ( ipCalc != false ), or      *
//*    caller must do it on return.                                            *
//*    If 'ipCalc' specifies, then we find the offset into the text stream     *
//*    which corresponds to 'tbIndex' before and after formatting and update   *
//*    'tbIndex' if necessary. The value of 'tbCursor' is then caculated       *
//*    directly from 'tbIndex'.                                                *
//******************************************************************************

void DialogTextbox::mlFmtDisplay ( const gString& gsText, bool ipCalc, bool hBreak )
{
   #define DEBUG_SOFTBREAK (0)   // for debugging this method only

   //* If we have data *
   if ( gsText.gschars() > 1 )
   {
      //* Calculate text-stream offset to character under cursor *
      short iPoint = ZERO ;
      if ( ipCalc )
      {
         gString gsipCalc ;
         this->tbGetText ( gsipCalc, false ) ;
         iPoint = this->mlIndex2Ip ( gsipCalc ) ;
      }

      //* Parse the text string into individual display lines.*
      int   wCount,                          // number of source characters
            wIndex = ZERO,                   // index into source data
            spIndex ;                        // word-break index (potential line break)
      wchar_t wChar ;                        // character being inspected
      short   wCols ;                        // columns written to line
      const wchar_t* wPtr = gsText.gstr() ;  // pointer to text data
      const short*   wCol = gsText.gscols( wCount ) ; // pointer to column data

      #if DEBUG_SOFTBREAK != 0 && DEBUG_ETM != 0
      winPos etp( 1, 1 ) ;
      //* If debugging window has been instantiated, which it  *
      //* has not been when method called from the constructor.*
      if ( etPtr != NULL )
         etPtr->ClearWin () ; // clear debugging window
      #endif   // DEBUG_SOFTBREAK && DEBUG_ETM

      //* Clear the character buffers *
      for ( short lIndex = ZERO ; lIndex < this->lines ; lIndex++ )
         *this->mlText[lIndex].ln = nckNULLCHAR ;
      *this->mlHead = nckNULLCHAR ;
      *this->mlTail = nckNULLCHAR ;

      //* If we have 'head' data, store it first. *
      if ( this->leftIndex > ZERO )
      {
         short i = ZERO ;
         do { this->mlHead[i++] = wPtr[wIndex++] ; }
         while ( i < this->leftIndex && wPtr[wIndex] != nckNEWLINE && 
                 wPtr[wIndex] != nckNULLCHAR && i < (gsALLOCDFLT - 1) ) ;
         this->mlHead[i] = nckNULLCHAR ;
         if ( wPtr[wIndex] == nckNEWLINE )
            ++wIndex ;
      }

      //* For each display line in the control *
      for ( short lIndex = ZERO ; lIndex < this->lines ; lIndex++ )
      {
         wCols = ZERO ;          // reset column accumulator
         spIndex = ZERO ;        // reset word-break index

         //* Copy source to target-line buffer *
         for ( short tIndex = ZERO ; wIndex < wCount ; tIndex++ )
         {
            if ( wPtr[wIndex] == nckSPACE )  // remember most recent space
               spIndex = tIndex ;

            //* Check whether the following character will fill or over-fill   *
            //* the current line.                                              *
            //* - If not on last line, do a 'soft' line break.                 *
            //* - If on last line, truncate displayed text at edge of field.   *
            wCols += wCol[wIndex] ;
            if ( wCols >= this->cols )
            {
               #if DEBUG_SOFTBREAK != 0 && DEBUG_ETM != 0
               if ( etPtr != NULL )
               {
                  gsetb.compose( "b%hd: sp:%02hd ti:%02hd wi:%03hd %3hd\n", 
                                 &lIndex, &spIndex, &tIndex, &wIndex, &iPoint ) ;
                  etp = etPtr->WriteParagraph ( etp, gsetb, etColor, true ) ;
               }
               #endif   // DEBUG_SOFTBREAK && DEBUG_ETM

               //* If we arrive here, it means that any 'hard' breaks *
               //* provided cannot be trusted, so ignore them.        *
               hBreak = false ;

               bool bumpIp = false ;   // 'true' if break inserted before iPoint

               //* If we have a previous word-break on this line *
               //* OR if the word break is the current character *
               if ( spIndex > ZERO && spIndex <= tIndex )
               {  //* Break the line AFTER the space character unless the *
                  //* space character is beyond the field, in which case  *
                  //* break ON the space character.                       *
                  if ( spIndex < tIndex )
                     ++spIndex ;
                  this->mlText[lIndex].ln[spIndex] = NULLCHAR ;
                  //* Rewind the source index *
                  if ( wPtr[wIndex -= tIndex - spIndex] == nckNEWLINE )
                     --wIndex ;

                  //* If we are inserting a break on the same line *
                  //* as the cursor but AHEAD of the iPoint.       *
                  if ( (ipCalc != false) && (lIndex == this->tbIndex.ypos) &&
                       (spIndex <= this->tbIndex.xpos) )
                  { bumpIp = true ; }
               }

               //* Else, line is filled BUT without a word break, *
               //* so we are forced to break mid-word.            *
               else
               {
                  this->mlText[lIndex].ln[tIndex] = NULLCHAR ;

                  if ( ipCalc && (lIndex == this->tbIndex.ypos) && 
                       (tIndex == this->tbIndex.xpos) )
                  { bumpIp = true ; }
               }

               if ( bumpIp != false )
               {
                  //* Advance iPoint by one character to stay synchronized.*
                  ++iPoint ;

                  //* If we have just pushed the iPoint out of the display *
                  //* area, scroll up by a line to make iPoint visible.    *
                  if ( lIndex == (this->lines - 1) )
                  {
                     gString gsh( this->mlHead ),
                             gs0( this->mlText[ZERO].ln ) ;

                     //* If mlHead not empty, then advance was unnecessary, *
                     //* else keep it for newline inserted at end of mlHead.*
                     if ( gsh.gschars() > 1 )
                        --iPoint ;

                     gsh.append( gs0.gstr() ) ;
                     this->leftIndex = gsh.gschars() - 1 ;
                     gsh.copy( this->mlHead, gsALLOCDFLT ) ;
                     for ( short i = 1 ; i < this->lines ; i++ )
                     {
                        gsh = this->mlText[i].ln ;
                        gsh.copy( this->mlText[i - 1].ln, mtLENGTH ) ;
                        *this->mlText[i].ln = nckNULLCHAR ;
                     }
                     --lIndex ;        // process last line again
                  }
               }

               #if DEBUG_SOFTBREAK != 0 && DEBUG_ETM != 0
               if ( etPtr != NULL )
               {
                  gsetb.compose( "    sp:%02hd ti:%02hd wi:%03hd %3hd\n", 
                                 &spIndex, &tIndex, &wIndex, &iPoint ) ;
                  etp = etPtr->WriteParagraph ( etp, gsetb, etColor, true ) ;
               }
               #endif   // DEBUG_SOFTBREAK && DEBUG_ETM

               break ;              // end of current line
            }

            //* Write source to target:                           *
            //* - If null terminator, write it to end the line.   *
            //* - If VALID newline, write null to end the line.   *
            //* - If INVALID newline, ignore it.                  *
            //* - Else write source character to target.          *
            wChar = wPtr[wIndex++] ;
            if ( wChar == nckNULLCHAR || (wChar == nckNEWLINE && hBreak) )
            {
               this->mlText[lIndex].ln[tIndex] = NULLCHAR ;
               break ;
            }
            else if ( wChar != nckNEWLINE )
               this->mlText[lIndex].ln[tIndex] = wChar ;
            else  // ignore the newline character
               --tIndex ;
         }
         if ( wChar == nckNULLCHAR )   // all source data has been scanned
            break ;
      }

      //* If we have text that did not fit into the display area, store it *
      //* (without newlines). We trust that string is null-terminated.     *
      if ( (this->hzShift != false) && (wIndex < (wCount - 1)) )
      {
         for ( short i = ZERO ; wIndex < wCount ; wIndex++ )
         {
            if ( wPtr[wIndex] != nckNEWLINE )
               this->mlTail[i++] = wPtr[wIndex] ;
         }
      }

      //* Recalculate the index and cursor position.*
      if ( ipCalc )
      {
         this->mlIp2Index ( iPoint ) ;
      }

      #if DEBUG_SOFTBREAK != 0 && DEBUG_ETM != 0
      if ( etPtr != NULL )
      {
         //etPtr->WriteString ( etp, " Press a key...", nc.rebl, true ) ;
         //nckPause();
         //sleep ( 3 ) ;
      }
      #endif   // DEBUG_SOFTBREAK && DEBUG_ETM
   }

   #undef DEBUG_SOFTBREAK
}  //* End mlFmtDisplay() *

//*************************
//*      tbSetCursor      *
//*************************
//******************************************************************************
//* Configure the way the target control accepts and displays text input       *
//* during editing. OR set the cursor position over the specified character of *
//* the text string.                                                           *
//*                     See notes in SetTextboxCursor().                       *
//*                     -------------------------------                        *
//*                                                                            *
//* Input  : xoffset: member of enum tbCurPos -OR- character offset            *
//*          yoffset: (optional, ZERO by default - currently unused)           *
//*                                                                            *
//* Returns: OK if successful                                                  *
//*            - For input configuration, change is immediate.                 *
//*            - For cursor positioning, the change becomes visible the next   *
//*              time the control receives input focus.                        *
//*                                                                            *
//*          ERR if: - specified configuration not applicable to target control*
//*                    - tbcpHOME, tbcpAPPEND and char offset specification    *
//*                      do not apply if configured for tbcpRIGHTJUST          *
//*                    - tbcpRIGHTJUST applies only to single-line controls    *
//*                  - specified character offset is out-of-range              *
//*                    - if xoffset < ZERO, then no action                     *
//*                    - if xoffset > EOD, then cursor set at 'append'         *
//******************************************************************************

short DialogTextbox::tbSetCursor ( short xoffset, short yoffset )
{
   short    status = OK ;     // return value

   //* If setting cursor position to 'home' *
   if ( (xoffset == tbcpHOME || xoffset == ZERO) && (this->rightJust == false) )
   {  //* Single-line controls *
      if ( this->lines == 1 )
         this->leftIndex = this->tbIndex.ypos = this->tbIndex.xpos = ZERO ;

      //* Multi-line controls *
      else
      {
         if ( this->leftIndex > ZERO )
         {  //* Reveal un-displayed 'head' data *
            gString gs ;
            this->tbGetText ( gs ) ;
            this->leftIndex = ZERO ;
            this->mlFmtDisplay ( gs ) ;
         }
         this->tbIndex.ypos = this->tbIndex.xpos = ZERO ;
      }
      this->tbSynchCursor ( false ) ;  // calculate cursor position
   }  // tbcpHOME

   //* If setting cursor position to 'append' *
   else if ( (xoffset == tbcpAPPEND) && (this->rightJust == false) )
   {  //* Single-line control with text data (both LTR and RTL) *
      if ( (this->lines == 1) && (*this->wText != nckNULLCHAR) )
      {
         //* Get statistics on existing text data *
         gString gs( this->wText ) ;
         int colCount = gs.gscols() ;  // total columns of text data
         int charCount ;               // total characters of text data (incl. NULL)
         const short* colArray = gs.gscols(charCount) ; // array of column widths

         //* If append position is visible without a text shift *
         if ( colCount <= (this->cols - appendCOLS) )
         {
            this->leftIndex = ZERO ;         // reference first character of string
            this->tbIndex.xpos = charCount - 1 ; // reference NULL terminator
         }

         //* Else appending new characters requires a text shift *
         else if ( this->hzShift != false ) // if text shifting enabled
         {  //* Set cursor at append position and scan backward to locate   *
            //* leftmost displayed character.                               *
            //* (reference NULLCHAR character)                              *
            this->tbIndex.xpos = this->leftIndex = charCount - 1 ;
            short scanCols = appendCOLS ;       // count of columns scanned
            while ( scanCols < this->cols && this->leftIndex > ZERO )
            {
               scanCols += colArray[--this->leftIndex] ;
            }
            if ( scanCols > this->cols )  // if mid-character
               ++this->leftIndex ;
         }
         else     // text shifting disabled, AND append space not available
         {        // go as far to right as possible
            this->leftIndex = ZERO ;   // reference first character of string
            this->tbIndex.xpos = colCount < this->cols ? 
                              (charCount - 1) : (charCount - 2) ;
         }
      }

      //* Multi-line control with text data *
      else if ( (this->lines > 1) && (*this->mlText[ZERO].ln != nckNULLCHAR) )
      {
         if ( *this->mlTail != nckNULLCHAR )
         {  //* Reveal un-displayed 'tail' data *
            // Programmer's Note: We hate this inefficient loop; however, 
            // soft linebreaks make it impossible to determine ahead of time
            // what will fit into the display area, so we must loop until 
            // 'mlTail' is empty (max 2-3 iterations).
            gString gs ;
            short tailChars ;
            do
            {
               gs = this->mlTail ;
               tailChars = gs.gschars() - 1 ;
               this->tbGetText ( gs ) ;
               this->leftIndex += tailChars ;
               this->mlFmtDisplay ( gs ) ;
            }
            while ( tailChars > ZERO ) ;
         }

         //* Determine the position limits *
         short lastY = ZERO,     // index of last non-empty line
               lastX = ZERO ;    // index of last char on lastY
         while ( (lastY < (this->lines - 1)) && 
                 (*this->mlText[lastY+1].ln != NULLCHAR) )
            ++lastY ;
         while ( this->mlText[lastY].ln[lastX] != NULLCHAR )
            ++lastX ;
         this->tbIndex = { lastY, lastX } ;
      }
      this->tbSynchCursor ( false ) ;  // calculate cursor position
   }  // tbcpAPPEND

   //* If configuring target for right-justified input *
   //*      (valid only for single-line controls)      *
   else if ( xoffset == tbcpRIGHTJUST && this->lines == 1 )
   {
      this->rightJust = true ;      // configure control for right justification
      this->rtlContent = false ;    // reset flag for RTL editing
      this->tbIndex.ypos = ZERO ;   // always line zero

      //** Right-justified text MAY NOT grow beyond width of control **
      this->hzShift = false ;       // limit input to control width
      gString gs( this->wText ) ;   
      short colCount = gs.gscols() ; // total columns of text data
      if ( colCount > this->cols )  // truncate any existing text to width of control
      {
         gs.limitCols( this->cols ) ;
         gs.copy( this->wText, gsALLOCDFLT ) ;
      }
      this->tbSynchCursor ( false ) ;  // calculate cursor position
      this->RedrawControl ( false ) ;  // update the display
   }  // tbcpRIGHTJUST

   //* If specifying a character offset (not valid for right-justified fields) *
   else if ( (xoffset > ZERO) && (this->rightJust == false) )
   {  //* Single-line controls *
      if ( this->lines == 1 )
      {
         //* Get statistics on existing text data *
         gString gs( this->wText ) ;
         int colCount = gs.gscols() ;  // total columns of text data
         int charCount ;               // total characters of text data (incl. NULL)
         const short* colArray = gs.gscols(charCount) ;  // array of column widths

         //* If offset is within valid range *
         if ( xoffset < (charCount - 1) )
         {
            short cursCols = ZERO ;    // count of columns scanned
            short i = -1 ;
            do
            { cursCols += colArray[++i] ; }
            while ( i < xoffset ) ;
            //* If target position is visible without a text shift *
            if ( cursCols <= this->cols )
            {
               this->leftIndex = ZERO ;
               this->tbIndex.xpos = i ;
            }
            //* Else set cursor position at left edge of field and if       *
            //* necessary, shift data to the right until field is filled    *
            //* with printing characters.                                   *
            else if ( this->hzShift != false )
            {
               this->leftIndex = this->tbIndex.xpos = i ;
               short leftCols = cursCols ;
               while ( (colCount - leftCols) < (this->cols - 1) &&
                       (this->leftIndex > ZERO) )
               {
                  leftCols -= colArray[--this->leftIndex] ;
               }
            }
            else
            {  //* Horizontal shift is diabled but target character is not  *
               //* accessible, (this should never happen, but...)           *
               status = ERR ;
            }
            if ( status == OK )
               this->tbSynchCursor ( false ) ;  // calculate cursor position
         }
         else                       // invalid xoffset
         {  //* Set cursor at 'append' position and return error condition. *
            this->tbSetCursor ( tbcpAPPEND ) ;  // (recursive call)
            status = ERR ;
         }
      }

      //* Multi-line controls (this->lines > 1) *
      else
      {  //* Start the search from the top-of-data. This is a recursive call.  *
         this->tbSetCursor ( tbcpHOME ) ;

         gString gs ;                     // work buffer
         this->tbGetText ( gs, false ) ;  // get a copy of text (with newlines)
         const wchar_t* wPtr = gs.gstr() ;// pointer to source text
         short newY = ZERO, newX = ZERO,  // new Y/X position
               sBase = ZERO ;             // source data base index
         bool npFound = false ;           // 'true' if new position found

         for ( short wi = ZERO ; wPtr[wi] != nckNULLCHAR ; wi++ )
         {
            if ( wPtr[wi] != nckNEWLINE )
            {  //* If target found OR target is beyond end-of-data *
               if ( (sBase + newX) == xoffset )
               {
                  npFound = true ;
                  break ;
               }
               else              // advance to next character
                  ++newX ;
            }
            else                 // end of current line
            {
               sBase += (newX) ;       // characters scanned so far
               ++newY ;                // next display line
               newX = ZERO ;           // beginning of line

               if ( (newY == this->lines) && (*this->mlTail != nckNULLCHAR) )
               {  //* Target not found in displayed data. Shift in a line   *
                  //* of data, if any, from mlTail.                         *
                  newY = ZERO ;  // reset line index
                  gString gsf( this->mlText[0].ln ) ;
                  short leftShift = gsf.gschars() - 1 ;  // not incl. nullchar
                  this->tbGetText ( gs ) ;      // get current contents
                  this->leftIndex += leftShift ;// shift out first line
                  sBase = wi = this->leftIndex ;// reference 1st displayed character
                  this->mlFmtDisplay ( gs ) ;   // reformat the data
                  this->tbGetText ( gs, false );// refresh our copy of data
               }
            }
         }
         if ( npFound )
         {
            this->tbIndex = { newY, newX } ; // set specified index
            this->tbSynchCursor ( false ) ;  // calculate cursor position
         }
         else
         {  //* Set cursor at 'append' position and return error condition. *
            this->tbIndex = { newY, newX } ;    // avoid trashing the index
            this->tbSetCursor ( tbcpAPPEND ) ;  // (recursive call)
            status = ERR ;
         }
      }
   }
   else        // invalid parameter - no modifications made
      status = ERR ;
   return status ;

}  //* End tbSetCursor() *

//*************************
//*     tbSynchCursor     *
//*************************
//******************************************************************************
//* Calculate cursor position (this->tbCursor) based on current values of      *
//* this->leftIndex and this->tbIndex, and optionally update the display.      *
//*                                                                            *
//* Note: Additional updates for right-justified control:                      *
//*       - this->tbIndex.xpos references (rightmost) last printing character  *
//*       - this->leftIndex references column for start of string              *
//*                                                                            *
//* Input  : refresh : if 'true', then after data update, refresh the display  *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void DialogTextbox::tbSynchCursor ( bool refresh )
{
   //* tbIndex.ypos is assumed to be correct *
   this->tbCursor.ypos = this->tbIndex.ypos ;

   if ( this->lines == 1 )
   {
      //* Get an array of column counts for each character. *
      gString gsText( this->wText ) ;
      int charCount ;
      const short* cPtr = gsText.gscols( charCount ) ;

      if ( this->rightJust != false )
      {  //* Get width of last printing character. *
         if ( charCount > 1 )
            this->tbCursor.xpos = this->cols - cPtr[charCount - 2] ;
         else     // no data in field
            this->tbCursor.xpos = this->cols - 1 ;

         //* Refrence last printing character *
         this->tbIndex.xpos = gsText.gschars() > 1 ? gsText.gschars() - 2 : ZERO ;

         //* Calculate leftmost occupied column *
         this->leftIndex = this->cols - gsText.gscols() ;
      }
      else  // LTR or RTL configured controls
      {
         short cIndex = this->leftIndex,     // index into array of columns
               cCols = ZERO ;                // column accumulator
   
         while ( cIndex < this->tbIndex.xpos )
            cCols += cPtr[cIndex++] ;
   
         if ( this->rtlContent == false )    // LTR
            this->tbCursor.xpos = cCols ;
         else                                // RTL
            this->tbCursor.xpos = this->cols - cCols - 1 ;
      }
   }
   else  // (this->lines > 1)
   {
      //* Get a copy of text for current line AND a list of   *
      //* columns-per-character, and update the column offset.*
      gString gsText( this->mlText[this->tbIndex.ypos].ln ) ;
      int  lChars ;
      const short* lColptr = gsText.gscols( lChars ) ;
      //* Initialize column offset *
      this->tbCursor.xpos = this->rtlContent ? (this->cols - 1) : ZERO ;
      for ( int i = ZERO ; i < this->tbIndex.xpos && i < lChars ; i++ )
      {
         if ( this->rtlContent == false )    // LTR languages
         {
            if ( (this->tbCursor.xpos += lColptr[i]) >= this->cols )
            {
               this->tbCursor.xpos -= lColptr[i] ;
               if ( this->tbIndex.xpos > ZERO )    // (this should always be true)
                  --this->tbIndex.xpos ;
               break ;
            }
         }
         else                                // RTL languages
         {
            if ( (this->tbCursor.xpos -= lColptr[i]) >= this->cols )
            {
               this->tbCursor.xpos += lColptr[i] ;
               if ( this->tbIndex.xpos > ZERO )    // (this should always be true)
                  --this->tbIndex.xpos ;
               break ;
            }
         }
      }
   }
   if ( refresh != false )
   {
      this->wPtr->SetCursor ( this->tbCursor ) ;
      this->wPtr->RefreshWin () ;
   }

}  //* End tbSynchCursor() *

//*************************
//*     tbGetText         *
//*************************
//******************************************************************************
//* Get a copy of text data.                                                   *
//* - For single-line controls, this is 'wText'.                               *
//* - For multi-line controls, this is a concatenation of the text for any     *
//*   non-displayed head data ('mlHead'), the text for each display line of    *
//*   the 'mlText' array and any non-displayed tail data ('mlTail') -- all     *
//*   optionally separated by NEWLINE characters.                              *
//*                                                                            *
//* Input  : gsTrg     : receives text data                                    *
//*          strip     : (optional, true by default)                           *
//*                      if 'true', do not include the NEWLINE separators      *
//*                      if 'false', include NEWLINE separators                *
//*                                                                            *
//* Returns: number of characters in string including the NULLCHAR             *
//******************************************************************************

short DialogTextbox::tbGetText ( gString& gsTrg, bool strip )
{
   gsTrg.clear() ;            // clear caller's buffer
   if ( this->lines == 1 )
      gsTrg = this->wText ;
   else if ( this->mlText != NULL && this->mlTail != NULL )
   {
      if ( this->leftIndex > ZERO )                   // get head data
      {
         gsTrg = this->mlHead ;
         if ( !strip )
            gsTrg.append( L"\n" ) ;
      }
      for ( short i = ZERO ; i < this->lines ; i++ )  // get visible data
      {
         if ( *this->mlText[i].ln != NULLCHAR )
         {
            if ( i > ZERO && !strip )
               gsTrg.append( L"\n" ) ;
            gsTrg.append( this->mlText[i].ln ) ;
         }
      }
      if ( *this->mlTail != nckNULLCHAR )
      {
         if ( !strip )
            gsTrg.append( L"\n" ) ;
         gsTrg.append( this->mlTail ) ;
      }
   }
   return ( gsTrg.gschars() ) ;

}  //* End tbGetText() *

//*************************
//*     tbSetText         *
//*************************
//******************************************************************************
//* Format the display data for the text box and store it in our data members. *
//* Called by: DialogTextbox constructor                                       *
//*            SetTextboxText method                                           *
//*            DisplayTextboxMessage method ('srcFilter' will be 'false')      *
//*                                                                            *
//* For control defined with a single display line:                            *
//*  1) Length of text may be up to gsALLOCDFLT unless hzShift == false, in    *
//*     which case width of text is limited to the width of the control.       *
//*     If source data is too long, it will be silently truncated.             *
//*  2) All control characters (0x01 - 0x1F) will be silently stripped from    *
//*     the string before filtering is applied.                                *
//*  3) Textbox filtering will be applied if srcFilter != false.               *
//*  4) Formatted data will be stored in this->wText member.                   *
//*                                                                            *
//* For control defined with multiple display lines:                           *
//*  1) Length/width of text data                                              *
//*       a) If hzShift!=false : length of text may be up to gsALLOCDFLT.      *
//*       b) If hzShift==false : width of text may be up to                    *
//*          (display lines * display columns).                                *
//*     If source data is too long (or too wide), it will be silently          *
//*     truncated.                                                             *
//*  2) All control characters (0x01 - 0x1F) will be silently stripped from    *
//*     the string before filtering is applied.                                *
//*  3) Textbox filtering will be applied if srcFilter != false.               *
//*  4) Text data will be formatted with 'soft' line breaks at the beginning   *
//*     of words for the best fit in the control.                              *
//*  5) Formatted data will be stored in this->mlText array                    *
//*     (and if hzShift!=false, overflow will be this->wText).                 *
//*                                                                            *
//* Input  : srcText   : source text data                                      *
//*          srcFilter : (optional, 'true' by default)                         *
//*                      if 'true', apply specified text filter (this->filter) *
//*                      if 'false', then do not apply the filter              *
//*                                  (display-only, non-editable text)         *
//*                                                                            *
//* Returns: number of characters in string including the NULLCHAR             *
//*          OR ERR if data filter detected an invalid character               *
//*                 (data members will be cleared)                             *
//******************************************************************************

short DialogTextbox::tbSetText ( const gString& srcText, bool srcFilter )
{
   //* Re-initialize our data members *
   this->tbClear () ;
   short status = 1 ;   // default return value == one character i.e. NULLCHAR

   if ( srcText.gschars() > 1 )           // if we have source data
   {
      wchar_t  workText[gsALLOCDFLT] ;    // work buffers
      gString gsText( srcText ) ;

      //* If horizontal shift is disabled, limit text to size of visible area. *
      if ( this->hzShift == false )
      {
         if ( this->lines == 1 )    // single-line controls
            gsText.limitCols( this->cols ) ;
//         else                       // Multi-line controls
//            gsText.limitCols( this->lines * this->cols ) ;
      }

      //* Strip control characters from the text. *
      int wCount, dIndex = ZERO ;
      const wchar_t* wPtr = gsText.gstr( wCount ) ;
      for ( int sIndex = ZERO ; sIndex < wCount ; sIndex++ )
      {
         if ( (wPtr[sIndex] >= nckSPACE) || (wPtr[sIndex] == NULLCHAR) )
            workText[dIndex++] = wPtr[sIndex] ;
      }
      gsText = workText ;           // clean source data
      status = gsText.gschars() ;   // return value == character count

      //* Apply the previously-established text filter. *
      //*  ( Filter is not applied if data are sent )   *
      //*  ( from DisplayTextboxMessage().          )   *
      //* Then store data to appropriate data member.   *
      if ( srcFilter != false )
      {
         if ( (this->tbApplyFilter ( gsText )) == false )
            status = ERR ;
      }
      if ( status != ERR )
      {
         if ( this->lines == 1 )
            gsText.copy( this->wText, gsALLOCDFLT ) ;
         else
            this->mlFmtDisplay ( gsText ) ;
      }
   }
   return status ;

}  //* End of tbSetText() *

//*************************
//*        tbClear        *
//*************************
//******************************************************************************
//* Erase all display data and reset cursor position.                          *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: OK                                                                *
//******************************************************************************

short DialogTextbox::tbClear ( void )
{
   *this->wText = NULLCHAR ;
   if ( this->mlText != NULL )
   {
      for ( short lIndex = ZERO ; lIndex < this->lines ; lIndex++ )
         *this->mlText[lIndex].ln = NULLCHAR ;
   }
   if ( this->mlTail != NULL )
      *this->mlTail = nckNULLCHAR ;
   this->leftIndex = ZERO ;            //* reset cursor tracking
   this->tbCursor = { ZERO, ZERO } ;
   this->tbIndex  = { ZERO, ZERO } ;
   if ( this->rightJust )
      this->tbCursor.xpos = this->cols - 1 ;
   return OK ;

}  //* End tbClear() *

//*************************
//*   tbDisplayMessage    *
//*************************
//******************************************************************************
//* Display a multi-color message, but do not store the data.                  *
//* The next call to RedrawControl() will restore the previously-displayed     *
//* text (if any).                                                             *
//*                                                                            *
//* Called only through the NcDialog DisplayTextboxMessage() method.           *
//*                                                                            *
//* Input  : uData : dtbmData class object containing the text and color       *
//*                  data to be displayed.                                     *
//*                                                                            *
//* Returns: OK if successful                                                  *
//*          ERR if message too long for control (msg displayed-but-truncated) *
//******************************************************************************

short DialogTextbox::tbDisplayMessage ( dtbmData& uData )
{
attr_t   monoColor,           // if monochrome text, paint with this color
         colorFlag ;          // color code (not necesssarily a color attribute)
short    result = OK ;        // return value
bool     multiColor = false ; // true if multi-color data received
winPos   wPos( ZERO, (uData.rtlText ? this->cols - 1 : ZERO) ) ; // start position

   //* Set color attribute for single-color text display, *
   //* or set flag for multi-color text display.          *
   colorFlag = *uData.GetColorPtr() ;
   if ( colorFlag == dtbmFcolor )
      monoColor = this->fColor ;       // default Focus color
   else
   {
      monoColor = this->nColor ;       // default Non-focus color
      if ( colorFlag != dtbmNFcolor )  // else flag is a real color attribute
         multiColor = true ;
   }

   this->wPtr->ClearWin () ;           // clear the display area

   //* Get a copy of the text and limit text to size of control.*
   gString gs( uData.GetTextPtr() ) ;
   short maxWidth = this->lines * this->cols ;
   if ( ((gs.gscols()) > maxWidth) )
   {
      gs.limitCols( maxWidth ) ;
      result = ERR ;
   }

   //* Single-color output *
   if ( multiColor == false )
   {
      if ( this->lines == 1 )    // (WriteString() ignores newline characters.)
         this->wPtr->WriteString ( wPos, gs, monoColor, false, uData.rtlText ) ;
      else
         this->wPtr->WriteParagraph ( wPos, gs, monoColor, false, uData.rtlText ) ;
   }

   //* Multi-color output *
   else
   {
      int charCount ;
      const wchar_t* txtPtr = gs.gstr( charCount ) ;  // point to text
      --charCount ;                                   // ignore null terminator
      const attr_t*  atrPtr = uData.GetColorPtr() ;   // point to color data
      int dIndex = ZERO ;                             // txtPtr and atrPtr index
      

      //* For each line of data, create a multi-color output string.*
      while ( dIndex < charCount && wPos.ypos < this->lines )
      {
         while ( txtPtr[dIndex] != nckNULLCHAR )
         {
            if ( txtPtr[dIndex] != nckNEWLINE )
               wPos = this->wPtr->WriteChar ( wPos, txtPtr[dIndex], atrPtr[dIndex],
                                              false, uData.rtlText ) ;
            else if ( this->lines > 1 )
               wPos = { short(wPos.ypos + 1), 
                        short(uData.rtlText ? this->cols - 1 : ZERO) } ;
            ++dIndex ;
         }
      }
   }

   if ( uData.refreshControl != false )
      this->RefreshControl () ;
   return result ;

}  //* End tbDisplayMessage() *

//*************************
//*    tbApplyFilter      *
//*************************
//******************************************************************************
//* Selector for text box data filtering methods.                              *
//*                                                                            *
//* These filters are based on the 'wide character' test functions declared    *
//* in wctype.h. The exceptions are the filters that restrict data to the      *
//* ASCII character set.                                                       *
//*                                                                            *
//* NOTE: Caller should pre-filter the following, before calling this          *
//*       method: nckENTER, nckTAB, nckSTAB, nckESC, nckNEWLINE, etc.          *
//*       These characters are always consumed by the text-box editing         *
//*       methods, and should therefore not be seen here.                      *
//*       NULLCHAR string terminator should also be handled by caller.         *
//*                                                                            *
//* Input  : wch   : pointer to character code to be tested                    *
//*                                                                            *
//* Returns: true if wch is member of specified group, else false              *
//******************************************************************************

bool DialogTextbox::tbApplyFilter ( wchar_t* wch )
{
   bool  goodChar ;

   switch ( this->filter )
   {
      case tbPrint:     goodChar = this->tbIsPrint ( wch ) ;                  break ;
      case tbPrintUpper:goodChar = this->tbIsPrint ( wch, 1 ) ;               break ;
      case tbPrintLower:goodChar = this->tbIsPrint ( wch, -1 ) ;              break ;
      case tbAsciiPrint:goodChar = this->tbIsAsciiPrint ( *wch ) ;            break ;
      case tbAlNum:     goodChar = this->tbIsAlNum ( *wch ) ;                 break ;
      case tbAlNumSp:   goodChar = this->tbIsAlNum ( *wch, true ) ;           break ;
      case tbAlpha:     goodChar = this->tbIsAlpha ( *wch ) ;                 break ;
      case tbAlphaSp:   goodChar = this->tbIsAlpha ( *wch, true ) ;           break ;
      case tbNumeric:   goodChar = this->tbIsNumeric ( *wch ) ;               break ;
      case tbNumber:    goodChar = this->tbIsNumeric ( *wch, false ) ;        break ;
      case tbHexNum:    goodChar = this->tbIsHexNum ( wch ) ;                 break ;
      case tbHexNumUp:  goodChar = this->tbIsHexNum ( wch, true ) ;           break ;
      case tbLower:     goodChar = this->tbIsLower ( wch ) ;                  break ;
      case tbLowerSp:   goodChar = this->tbIsLower ( wch, true ) ;            break ;
      case tbUpper:     goodChar = this->tbIsUpper ( wch ) ;                  break ;
      case tbUpperSp:   goodChar = this->tbIsUpper ( wch, true ) ;            break ;
      case tbURL:       goodChar = this->tbIsURL ( wch ) ;                    break ;
      case tbFileName:  goodChar = this->tbIsFileName ( *wch ) ;              break ;
      case tbFileLinux: goodChar = this->tbIsFileName ( *wch, false, true ) ; break ;
      case tbPathName:  goodChar = this->tbIsFileName ( *wch, true ) ;        break ;
      case tbPathLinux: goodChar = this->tbIsFileName ( *wch, true, true ) ;  break ;
      default:          goodChar = false ;                                    break ;
   }
   return goodChar ;

}  //* End tbApplyFilter() *

//*************************
//*    tbApplyFilter      *
//*************************
//******************************************************************************
//* Test text against the text filter specified for this Text Box.             *
//* PRIVATE METHOD.                                                            *
//*                                                                            *
//* Input  : gStr  : character string to be tested                             *
//*                                                                            *
//* Returns: true if all characters are members of specified group, else false *
//******************************************************************************

bool DialogTextbox::tbApplyFilter ( gString& gStr )
{
   bool  goodData = true ;          // return value

   //* Get a working copy of the wchar_t data *
   short    wChars = gStr.gschars() ;
   wchar_t  wSrc[gsALLOCDFLT] ;
   gStr.copy ( wSrc, wChars ) ;
   --wChars ;                       // don't test the NULLCHAR character

   //* Apply specifed data filter to each character in the array *
   for ( short i = ZERO ; i < wChars ; i++ )
   {
      //* Must be a printing character that passes the filter. *
      if ( (wSrc[i] < nckSPACE) || 
           ((this->tbApplyFilter ( &wSrc[i] )) == false) )
      {  //* Filter detected an invalid character, abort the operation *
         goodData = false ;
         break ;
      }
   }
   //* If data are valid, copy to caller's object *
   if ( goodData != false )
      gStr = wSrc ;

   return goodData ;

}  //* End tbApplyFilter() *

//*************************
//*      tbIsPrint        *
//*************************
//******************************************************************************
//* Validate character for printable wchar_t character.                        *
//*                                                                            *
//* Input  : wch      : pointer to wchar_t character to test                   *
//*          adjCase: (optional, ZERO by default)                              *
//*                   if 1,  then force lowercase characters to uppercase      *
//*                   if -1, then force uppercase characters to lowercase      *
//*                                                                            *
//* Returns: true if meets criteria, else false                                *
//******************************************************************************

bool DialogTextbox::tbIsPrint ( wchar_t* wch, short adjCase )
{
bool  goodChar = false ;

   if ( (iswprint ( *wch )) != ZERO )
   {
      goodChar = true ;
      if ( adjCase != ZERO )
      {
         wchar_t nwch = *wch ;
         if ( adjCase > ZERO && ((this->tbIsUpper ( &nwch, true )) != false) )
            *wch = nwch ;
         else if ( adjCase < ZERO && ((this->tbIsLower ( &nwch, true )) != false) )
            *wch = nwch ;
      }
   }
   return goodChar ;

}  //* End tbIsPrint() *

//*************************
//*    tbIsAsciiPrint     *
//*************************
//******************************************************************************
//* Validate character for printable ASCII as user input in a text box.        *
//*                                                                            *
//* Input  : wch : wchar_t character to test                                   *
//*                                                                            *
//* Returns: true if meets criteria, else false                                *
//******************************************************************************

bool DialogTextbox::tbIsAsciiPrint ( wchar_t wch )
{
bool  goodChar = false ;

   if ( wch >= SPACE && wch <= TILDE )
      goodChar = true ;
   return goodChar ;
   
}  //* End tbIsAsciiPrint() *

//*************************
//*     tbIsAlNum         *
//*************************
//******************************************************************************
//* Validate character for wchar_t alpha-numeric characters and PERIOD.        *
//*                                                                            *
//* Input  : wch : wchar_t character to test                                   *
//*          inclSpace: if true, allow SPACE characters                        *
//*                     (default==false)                                       *
//*                                                                            *
//* Returns: true if meets criteria, else false                                *
//******************************************************************************
//* Programmer's Note: See iswalpha(), iswalnum() for info on non-ASCII        *
//* characters which may be accepted as alphanumeric characters.               *
//*                                                                            *
//******************************************************************************

bool DialogTextbox::tbIsAlNum ( wchar_t wch, bool inclSpace )
{
bool  goodChar = false ;

   if ( (iswalnum ( wch )) != ZERO || wch == PERIOD )
      goodChar = true ;
   else if ( inclSpace != false && wch == SPACE )
      goodChar = true ;
   return goodChar ;

}  //* End tbIsAlNum() *

//*************************
//*      tbIsAlpha        *
//*************************
//******************************************************************************
//* Validate character for upper and lower case alpha.                         *
//*                                                                            *
//* Input  : wch : wchar_t character to test                                   *
//*          inclSpace: if true, allow SPACE characters                        *
//*                     (default==false)                                       *
//*                                                                            *
//* Returns: true if meets criteria, else false                                *
//******************************************************************************
//* iswalpha() uses a wint_t 'wide' character as an argument.                  *
//* So far as we can tell, this is no different from the wchar_t type.         *
//* The wide character test functions are declared in wctype.h.                *
//******************************************************************************

bool DialogTextbox::tbIsAlpha ( wchar_t wch, bool inclSpace )
{
bool  goodChar = false ;

   if ( (iswalpha ( wch )) != ZERO )
      goodChar = true ;
   else if ( inclSpace != false && wch == SPACE )
      goodChar = true ;
   return goodChar ;

}  //* End tbIsAlpha() *

//*************************
//*     tbIsNumeric       *
//*************************
//******************************************************************************
//* Validate character for number only, or for numeric characters.             *
//*                                                                            *
//* Input  : wch      : wchar_t character to test                              *
//*          inclSign : if true, allow PLUS, MINUS, PERIOD                     *
//*                     else '0' through '9' only                              *
//* Returns: true if meets criteria, else false                                *
//******************************************************************************

bool DialogTextbox::tbIsNumeric ( wchar_t wch, bool inclSign )
{
bool  goodChar = false ;

   if ( wch >= '0' && wch <= '9' )
      goodChar = true ;
   else if ( inclSign != false )
   {
      //* Plus or Minus must be first non-space character, *
      //* so check for it on exit from text box.           *
      if ( wch == '.' || wch == '+' || wch == '-' )
         goodChar = true ;
   }
   return goodChar ;

}  //* End tbIsNumeric() *

//*************************
//*     tbIsHexNum        *
//*************************
//******************************************************************************
//* Validate character for hexadecimal numbers '0'-'9' and 'A'-'F'.            *
//* Note that 'a'-'f' are accepted but forced to uppercase for display IF      *
//* forceUpper flag is true.                                                   *
//*                                                                            *
//* Input  : wch : pointer to wchar_t character to test                        *
//*          forceUpper: (optional, 'false' by default)                        *
//*                      if true convert lowercase 'a'-'f' to uppercase 'A'-'F'*
//*                      else, no conversion                                   *
//*                                                                            *
//* Returns: true if meets criteria, else false                                *
//******************************************************************************

bool DialogTextbox::tbIsHexNum ( wchar_t* wch, bool forceUpper )
{
bool  goodChar = false ;

   if ( (*wch >= '0' && *wch <= '9') || (*wch >= 'A' && *wch <= 'F') ||
        (*wch >= 'a' && *wch <= 'f' ) )
   {
      goodChar = true ;
      if ( (forceUpper != false) && (*wch >= 'a' && *wch <= 'f' ) )
      {
         *wch = towupper ( *wch ) ;
         if ( (iswupper ( *wch )) == false )
            goodChar = false ;
      }
   }
   return goodChar ;

}  //* End tbIsHexNum() *

//*************************
//*      tbIsLower        *
//*************************
//******************************************************************************
//* Validate character for wide-character lower case. If possible, convert     *
//* upper-case characters to lower case.                                       *
//*                                                                            *
//* Input  : wch      : pointer to wchar_t character to test                   *
//*          inclSpace: if true, allow SPACE characters                        *
//*                     (default==false)                                       *
//*                                                                            *
//* Returns: true if meets criteria, else false                                *
//******************************************************************************
//* iswlower() and towlower() use a wint_t 'wide' character as an argument.    *
//* So far as we can tell, this is no different from the wchar_t type.         *
//* The wide character test functions are declared in wctype.h.                *
//******************************************************************************

bool DialogTextbox::tbIsLower ( wchar_t* wch, bool inclSpace )
{
bool  goodChar = false ;

   if ( (iswlower ( *wch )) != false )    // if already lower case
      goodChar = true ;
   else if ( inclSpace != false && *wch == SPACE ) // allow whitespace character
      goodChar = true ;
   else                                   // try to convert to lower case
   {
      *wch = towlower ( *wch ) ;
      if ( (iswlower ( *wch )) != false )
         goodChar = true ;
   }
   return goodChar ;

}  //* End tbIsLower() *

//*************************
//*      tbIsUpper        *
//*************************
//******************************************************************************
//* Validate character for wide-character upper case. If possible, convert     *
//* lower-case characters to upper case.                                       *
//*                                                                            *
//* Input  : wch      : pointer to wchar_t character to test                   *
//*          inclSpace: if true, allow SPACE characters                        *
//*                     (default==false)                                       *
//*                                                                            *
//* Returns: true if meets criteria, else false                                *
//******************************************************************************
//* iswupper() and towupper() use a wint_t 'wide' character as an argument.    *
//* So far as we can tell, this is no different from the wchar_t type.         *
//* The wide character test functions are declared in wctype.h.                *
//******************************************************************************

bool DialogTextbox::tbIsUpper ( wchar_t* wch, bool inclSpace )
{
bool  goodChar = false ;

   if ( (iswupper ( *wch )) != false )    // if already upper case
      goodChar = true ;
   else if ( inclSpace != false && *wch == SPACE ) // alow whitespace character
      goodChar = true ;
   else                                   // try to convert to upper case
   {
      *wch = towupper ( *wch ) ;
      if ( (iswupper ( *wch )) != false )
         goodChar = true ;
   }
   return goodChar ;

}  //* End tbIsUpper() *

//*************************
//*       tbIsURL         *
//*************************
//******************************************************************************
//* Validate character for a URL specification.                                *
//*                                                                            *
//* Important Note: URLs contain percent-encoded (%nn) data which we cannot    *
//*                 test for on a character-by-character basis. What this      *
//*                 means is that a string could contain all valid URL         *
//*                 characters and still not be a valid URL string.            *
//*                                                                            *
//* Input  : wch      : pointer to wchar_t character (or %-encoded sequence)   *
//*                     to be tested                                           *
//*                                                                            *
//* Returns: true if meets criteria, else false                                *
//******************************************************************************
//* Programmer's Note: As of this writing (28 June 2011) the keepers of all    *
//* things netly are still trying to decide how to allow non-ASCII characters  *
//* in URL specifications. When they figure it out, we will update this method.*
//*                                                                            *
//* As of 2005, URI (Uniform Resource Identifier) string may contain the       *
//* following characters:                                                      *
//* Reserved Characters  : ! * ' ( ) ; : @ & = + $ , / ? # [ ]                 *
//* Unreserved Characters: ABCDEFGHIJKLMNOPQRSTUVWXYZ                          *
//*                        abcdefghijklmnopqrstuvwxyz                          *
//*                        0123456789 - _ . ~                                  *
//* Percent-encoded Characters (from Reserved Character list):                 *
//* !   *   '   (   )   ;   :   @   &   =   +   $   ,   /   ?   #   [   ]      *
//* %21 %2A %27 %28 %29 %3B %3A %40 %26 %3D %2B %24 %2C %2F %3F %23 %5B %5D    *
//*                                                                            *
//* Also: %   (space)   | however the space character is usually encoded as    *
//*       %25 %20       | a plus character ( + )                               *
//*                                                                            *
//* Non-ASCII characters: the character is encoded in UTF-8 format, and each   *
//* byte of the character is percent-encoded. In essense, this means that any  *
//* character could be encoded into a URI query, so in theory, any character   *
//* could be used within a URL specification. That they are not used, says     *
//* something about the complexity of the internet addressing system, and also *
//* says something about the English-centric nabobs that control it.           *
//*                                                                            *
//* "The generic URI syntax mandates that new URI schemes that provide for     *
//*  the representation of character data in a URI must, in effect, represent  *
//*  characters from the unreserved set without translation, and should        *
//*  convert all other characters to bytes according to UTF-8, and then        *
//*  percent-encode those values. This requirement was introduced in January   *
//*  2005 with the publication of RFC 3986." http://tools.ietf.org/html/rfc3986*
//*                                                                            *
//*  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -   *
//*  The bottom line is that we allow any printing character in a URI string;  *
//*  with the caveat that it must be properly encoded before used to address   *
//*  an object in the specific scheme (http, gopher, ftp, https, etc.).        *
//*          NOTE: We DO NOT perform URL scheme encoding here.                 *
//******************************************************************************

bool DialogTextbox::tbIsURL ( wchar_t* wch )
{
bool  goodChar = false ;

   #if 1    // Unrestricted
   if ( (this->tbIsPrint ( wch )) != false )
      goodChar = true ;
   #else    // Strict interpretation - not fully implemented
   if ( (this->tbIsAlNum ( *wch, true )) != false )
      goodChar = true ;
   //* Unreserved Characters *
   else if ( (*wch >= 'a' && *wch <= 'z') || (*wch >= 'A' && *wch <= 'Z') || 
             (*wch >= '0' && *wch <= '9') ||
              *wch == '-' || *wch == '_' || *wch == '.' || *wch == '~' )
   {
      goodChar = true ;
   }
   //* Reserved Characters *
   else if ( *wch == '!' || *wch == '*' || *wch == '\'' || *wch == '(' || 
             *wch == ')' || *wch == ';' || *wch == ':'  || *wch == '@' || 
             *wch == '&' || *wch == '=' || *wch == '+'  || *wch == '$' || 
             *wch == ','  || *wch == '/' || *wch == '?' || *wch == '#' || 
             *wch == '[' || *wch == ']' )
   {
      goodChar = true ;
   }
   //* %-encoded character sequences *
   else if ( *wch == '%' )
   {
      // TEST THE SEQUENCE HERE
   }
   #endif   // Strict interpretation - not fully implemented

   return goodChar ;

}  //* End tbIsURL() *

//*************************
//*      Encode_URI       *
//*************************
//******************************************************************************
//* Encode a raw, unencoded URI/URL sequence using the RFC 3986 standard or    *
//* its successors.                                                            *
//*  NOT YET IMPLEMENTED                                                       *
//*                                                                            *
//* Input  : gStr   : (by reference) data string to be encoded                 *
//*                                                                            *
//* Returns: 'true' if all characters match tbURL filter criteria, else 'false'*
//******************************************************************************

bool  Encode_URI ( gString& gStr )
{
   return false ;
}  //* End Encode_URI() *

//*************************
//*      Decode_URI       *
//*************************
//******************************************************************************
//* Decode an encoded URI/URL sequence into its raw text form using the        *
//* RFC 3986 standard or its successors.                                       *
//*  NOT YET IMPLEMENTED                                                       *
//*                                                                            *
//* Input  : gStr   : (by reference) data string to be decoded                 *
//*                                                                            *
//* Returns: 'true' if all characters match tbURL filter criteria, else 'false'*
//******************************************************************************

bool  Decode_URI ( gString& gStr )
{
   return false ;
}  //* End Decode_URI() *

//*************************
//*     tbIsFileName      *
//*************************
//******************************************************************************
//* Validate character for filename or directory name. If inclSlash, then also *
//* allow forward slash as a pathname (but not as a filename) character.       *
//*                                                                            *
//* This method allows all alphanumeric characters for all alphabets and all   *
//* number systems, SPACE, PERIOD, and non-alnum characters that have proven   *
//* to be valid filename characters under Fedora Linux.                        *
//* (For Windoze system pathnames, we would also need to accept the backslash) *
//* (character, '\\' but currently we don't.                                 ) *
//*                                                                            *
//* Input  : wch      : wchar_t character to test                              *
//*          inclSlash: true for tbPathName and tbPathSpecial filters          *
//*                     false for tbFileName and tbFileSpecial filters         *
//*          linuxSpec: true for tbFileSpecial and tbPathSpecial filters       *
//*                     false for tbFileName and tbPathName filters            *
//*                                                                            *
//* Returns: true if meets criteria, else false                                *
//******************************************************************************
//* Linux / UNIX: Valid filename characters                                    *
//* ---------------------------------------                                    *
//* Under Linux all characters except '\0' and '/' are valid filename          *
//* characters. However, some operating systems and filesystems will not       *
//* accept all valid Linux filename characters. While we are not going to      *
//* force the ultra-limited POSIX (global compatibility) standard on anyone,   *
//* the user must be aware of interoperability issues.                         *
//*                                                                            *
//* Filenames that must travel across the web should be aware of the following.*
//*  #      <     $     +   %      >     !     `                               *
//*  &      *     ‘     |   {      ?     "     =                               *
//*  }      /     :     \   @      whitespace                                  *
//*                                                                            *
//* The filenames '.' and '..' are reserved, but we rely on the user to know   *
//* that.                                                                      *
//*                                                                            *
//* Linux/UNIX Reserved Characters                                             *
//* ------------------------------                                             *
//* Avoid using the following characters in file names:                        *
//*  '   "   ?   :   ;   \   &   >   <   |   *                                 *
//*                                                                            *
//* All of these characters are allowed when the 'linuxSpec' flag is set, but  *
//* but use these characters with caution when naming files.                   *
//*  a) Some of these characters are reserved characters under VFAT/NTFS       *
//*     filesystems (see below).                                               *
//*                                                                            *
//* The shell program will require that certain 'special' characters have to   *
//* be 'escaped' in order to perform operations on filenames that contain      *
//* those characters.                                                          *
//*                                                                            *
//* These characters are interpreted as instructions to the 'bash' shell or    *
//* other shell programs UNLESS they are "escaped" i.e. preceeded by the '\'   *
//* escape character when sent to the shell.                                   *
//*                                                                            *
//* VFAT (Virtual FAT) and NTFS Reserved Characters                            *
//* -----------------------------------------------                            *
//* Note that while Linux filesystems allow all UTF-8 characters in filenames, *
//* under VFAT and NTFS filesystems (often used for data backup), certain      *
//* characters are reserved:  0x00-0x1F 0x7F " * / : < > ? \ |                 *
//*  a) Only a moron would use a control character in a filename, so           *
//*     0x00-0x1F 0x7F are always rejected by this method.                     *
//*  b) The Linux cp (copy) and mv (rename) commands will refuse to operate    *
//*     on filenames with the reserved characters when the target filesystem   *
//*     is VFAT or NTFS. Therefore, the 'linuxSpec' flag should not be set if  *
//*     the resulting filename will be stored on those filesystems.            *
//******************************************************************************

bool DialogTextbox::tbIsFileName ( wchar_t wch, bool inclSlash, bool linuxSpec )
{
   const short   restrictedFnCount = 12 ;
   const wchar_t restrictedFnChars[restrictedFnCount] = 
   { L'/', L'<', L'>', L'|', L':', L';', L'&', L'*', L'?', L'"', L'`', L'\\' } ;

   bool  goodChar = false ;

   if ( (this->tbIsAlNum ( wch, true )) != false )
      goodChar = true ;

   else if ( (iswpunct ( wch )) != false )
   {
      if ( (inclSlash && wch == L'/') )   // pathnames may include forward slash
         goodChar = true ;

      //* Reject VFAT/NTFS reserved characters and *
      //* "special" shell-comand characters.       *
      else if ( ! linuxSpec )
      {
         goodChar = true ;
         for ( short i = ZERO ; i < restrictedFnCount ; ++i )
         {
            if ( wch == restrictedFnChars[i] )
            {
               goodChar = false ;
               break ;
            }
         }
      }

      //* Allow "special" shell-command characters. (see not above) *
      else if ( linuxSpec )
      {
         for ( short i = ZERO ; i < linuxSpecialCount ; ++i )
         {
            if ( wch == linuxSpecialChars[i] )
            {
               goodChar = true ;
               break ;
            }
         }

         if ( ! goodChar && (wch != L'/' && wch != L'`') )
            goodChar = true ;
      }
   }
   return goodChar ;

}  //* End tbIsFileName() *

#if DEBUG_ET != 0 || DEBUG_ETM != 0
//*************************
//*        tbAlert        *
//*************************
//******************************************************************************
//* For debugging DialogTextbox class. Make some noise.                        *
//* NOTE: 'Terminal Bell' must be enabled in the console window.               *
//*                                                                            *
//* Input  : count : (optional, 1 by default) number of boinks                 *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void DialogTextbox::tbAlert ( short boinks )
{
   if ( etPtr != NULL )
   {
      if ( boinks < ZERO || boinks > 100 )
         boinks = 1 ;
      while ( boinks-- > ZERO )
      { etPtr->UserAlert () ; usleep ( 150000 ) ; }
   }
}  //* End tbAlert()
#endif   // DEBUG_ET || DEBUG_ETM


      //****************************************************
      //** THIS SECTION IMPLEMENTS THE PUBLIC and PRIVATE **
      //**      NcDialog-class METHODS FOR ACCESSING      **
      //**       DialogTextbox-class FUNCTIONALITY.       **
      //****************************************************

   //****************************************************
   //** Local Clipboard: This clipboard definition is  **
   //** directly accessed only by NcDialog methods:    **
   //**  1) GetLocalClipboard()                        **
   //**  2) SetLocalClipboard()                        **
   //**  3) Tb2LocalClipboard()                        **
   //**  4) LocalClipboard2Tb()                        **
   //**  5) GetLocalClipboard_Bytes()                  **
   //**  6) GetLocalClipboard_Chars()                  **
   //** The DialogTextbox class knows nothing about    **
   //** this object.                                   **
   //**                                                **
   //****************************************************
   class TextboxLocalClipboard
   {
      // NOTE: Even through member methods are declared 'public', they
      //       are accessed only by NcDialog member methods.
      public:
      TextboxLocalClipboard ( void )         //* Constructor
      {
         this->tblcClear () ;
      }

      short tblcStore ( const gString& srcText, short srcIndex )
      {  //* Store caller's data (if any)
         this->tbClipboard = srcText ;
         this->tbsrc = srcIndex ;
         return this->tbClipboard.gschars() ; // return character count
      }
      short tblcRetrieve ( gString& trgText )
      {
         trgText = this->tbClipboard ;
         return trgText.gschars() ;          // return character count
      }
      short tblcGetSource ( void )
      {
         return this->tbsrc ;                // return source index
      }
      short tblcClear ( void )
      {
         this->tbClipboard.clear() ;         // empty string
         this->tbsrc = MAX_DIALOG_CONTROLS ; // no valid source control
         return OK ;
      }
      friend class NcDialog ;    //* Allow NcDialog class access to all members

      private:
      gString tbClipboard ;      //* Local clipboard buffer
      short tbsrc ;              //* Index of source Textbox control

   } static tbLocalClipboard ;   //* Instance of local clipboard

//*************************
//*    etbTb2Clipboard    *
//*************************
//******************************************************************************
//* Private Method                                                             *
//* --------------                                                             *
//* Copy the contents of the Textbox which is currently under edit to the      *
//* Textbox Local Clipboard buffer, and if currently connected to the system   *
//* clipboard, copy the local clipboard data to the system clipboard.          *
//*  a) If no source text has been 'selected', then assume that all source     *
//*     text is to be copied.                                                  *
//*  b) If text has been 'selected' then copy only the selection.              *
//*  c) If the 'cutSel' flag is set, then after copying the data to the        *
//*     Local Clipboard, delete the selected source text (or all source text   *
//*     if no selection).                                                      *
//*                                                                            *
//* Input  : cutSel : (optional, 'false' by default)                           *
//*                   if 'true', then after copy, delete the 'selected'        *
//*                   data in the source control                               *
//*                                                                            *
//* Returns: OK if successful                                                  *
//*          OR ERR if:                                                        *
//*             a) Textbox under edit is currently empty                       *
//******************************************************************************

short NcDialog::etbTb2Clipboard ( bool cutSel )
{
   gString gsText ;                 // source data
   short srcChars,                  // number of source characters
         status = ERR ;             // return value

   if ( (srcChars = this->GetTextboxText ( this->currCtrl, gsText )) > 1 )
   {
      --srcChars ;      // don't count the NULLCHAR

      //* Access to the Textbox methods and data *
      DialogTextbox* cp = (DialogTextbox*)this->dCtrl[this->currCtrl] ;

      //* No selection is equivalent to ALL selected.*
      if ( cp->select.active == false || cp->select.count == ZERO )
         cp->tbSelectText ( true ) ;

      //* Find and isolate the selection. *
// NOTE: THIS WORKS, BUT WE COULD UPDATE TO USE OUR STORED 'base' and 'count'
      short selIndex = ZERO, selCount = srcChars ;
      while ( (cp->select.sel[selIndex] == false) && 
              (selIndex < srcChars) )
         ++selIndex ;
      short i = selIndex ;
      while ( (cp->select.sel[i] != false) && (i <= srcChars) )
         ++i ;
      selCount = i - selIndex ;

      //* If we have more than an empty string *
      if ( selCount > ZERO )
      {
         gString gsClip = &gsText.gstr()[selIndex] ;
         gsClip.limitChars( selCount ) ;
         tbLocalClipboard.tblcStore ( gsClip, this->currCtrl ) ;
         status = OK ;

         //* If request was to 'cut' the selection, delete 'selected' data. *
         if ( cutSel != false )
         {
            if ( cp->select.count == srcChars )
            {  // Programmer's Note: This includes all Textboxes configured
               // for right-justified input.
               cp->select.reset() ;
               cp->tbClear () ;     // delete all existing text
            }
            else
               this->etbDeleteSelection ( cp ) ;
         }
         cp->select.reset() ;       // cancel the selection

         //* Update the display *
         cp->RedrawControl ( true, true ) ;

         //* If Wayland Clipboard is active and connected,     *
         //* copy the local clipboard data to system clipboard.*
         this->wcbLocalCb2SystemCb () ;
      }
   }
   return status ;

}  //* End etbTb2Clipboard() *

//*************************
//*    etbClipboard2Tb    *
//*************************
//********************************************************************************
//* Private Method                                                               *
//* --------------                                                               *
//* If currently connected to the system clipboard, copy the contents of the     *
//* system clipboard to the Textbox Local Clipboard buffer.                      *
//* Then add to (or replace) text in the Textbox which is currently under        *
//* edit with the contents of the Textbox Local Clipboard.                       *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: OK if successful                                                    *
//*          OR ERR if:                                                          *
//*             a) one or more characters in the local Textbox clipboard buffer  *
//*                does not match the filtering criterion of the target          *
//*             b) Local Textbox Clipboard is currently empty                    *
//*             c) combined length of existing and new text (after optional      *
//*                text deletion) would overflow the control's buffer            *
//********************************************************************************

short NcDialog::etbClipboard2Tb ( void )
{
   short status = ERR ;                               // return value

   //* If Wayland Clipboard is active and connected,     *
   //* copy the system clipboard data to local clipboard.*
   this->wcbSystemCb2LocalCb () ;

   //* Verify that the clipboard contains more than a NULLCHAR.*
   if ( (this->GetLocalClipboard_Chars ()) > 1 )
   {
      DialogTextbox* cp = (DialogTextbox*)this->dCtrl[this->currCtrl] ;

      //* Prefilter the source data according to control's criteria.*
      gString  cbText ;             // contents of local clipboard
      this->GetLocalClipboard ( cbText ) ;
      if ( (this->VerifyTbText ( this->currCtrl, cbText, cp->filter )) == OK )
      {
         int cbChars = cbText.gschars() - 1 ;
         gString  tbText ;          // current contents of textbox control
         int tbChars = this->GetTextboxText ( this->currCtrl, tbText ) ;

         //* Test for potential buffer overflow, and *
         //* if overflow, abort without processing.  *
         int totalChars = tbChars + cbChars ;
         if ( cp->select.active )
            totalChars -= cp->select.count ;
         if ( (!cp->rightJust) && (totalChars < gsALLOCDFLT) )
         {
            //* If replacing existing selection in target, *
            //* delete the selection.                      *
            if ( cp->select.active )
            {
               if ( (cp->select.count + 1) == tbChars )
               {
                  cp->select.reset() ;
                  cp->tbClear () ;     // delete all existing text
               }
               else
                  this->etbDeleteSelection ( cp ) ;
            }

            //* Insert clipboard data at current cursor position.*
            wkeyCode wk( ZERO, wktPRINT ) ;
            const wchar_t* wPtr = cbText.gstr() ;
            for ( int cbi = ZERO ; cbi < cbChars ; cbi++ )
            {
               wk.key = wPtr[cbi] ;
               if ( cp->lines == 1 )
                  cp->slInsertChar ( wk ) ;
               else
                  cp->mlInsertChar ( wk ) ;
            }
            status = OK ;
         }

         //* Data for right-justified fields is limited *
         //* to width of field.                         *
         else if ( cp->rightJust != false )
         {
            short totalCols = (cbText.gscols()) 
                              + (cp->select.active ? ZERO : (tbText.gscols())) ;
            if ( totalCols <= cp->cols )
            {
               //* If replacing existing data *
               if ( cp->select.active )
               {
                  cp->select.reset() ;
                  cp->tbClear () ;     // delete all existing text
                  tbText.clear() ;
               }

               //* Write the new data *
               wkeyCode wk( ZERO, wktPRINT ) ;
               const wchar_t* wPtr = cbText.gstr( cbChars ) ;
               for ( short cbi = ZERO ; cbi < cbChars ; cbi++ )
               {
                  wk.key = wPtr[cbi] ;
                  cp->rjInsertChar ( wk ) ;
               }
               status = OK ;
            }
         }
      }
   }
   return status ;
   
}  //* End etbClipboard2Tb() *

//*************************
//*  etbDeleteSelection   *
//*************************
//******************************************************************************
//* PRIVATE METHOD:                                                            *
//* Delete the 'selected' data in the target Textbox control.                  *
//* Called by etbTb2Clipboard() if 'Cut' selection.                            *
//* Called by etbClipboard2Tb() if 'Paste' replaces selection                  *
//* Called by: EditSlText(), EditMlText() to process Delete key when           *
//*            there is 'selected' text.                                       *
//*                                                                            *
//* Input  : cp  : pointer to DialogTextbox object                             *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void NcDialog::etbDeleteSelection ( DialogTextbox* cp )
{
   //* Move cursor to base-point of selection, then        *
   //* delete characters in the direction of the selection.*
   wkeyCode wkt = { nckDELETE, wktFUNKEY } ;
   if ( cp->lines == 1 )         // single-line controls
   {
      if ( cp->rtlContent )
      {
         while ( cp->tbIndex.xpos > cp->select.base.xpos )
            cp->slScrollRight () ;
         while ( cp->tbIndex.xpos < cp->select.base.xpos )
            cp->slScrollLeft () ;
         if ( ! cp->select.fwd )
         {
            cp->slScrollLeft () ;
            wkt.key = nckBKSP ;
         }
      }
      else  // LTR Content
      {
         while ( cp->tbIndex.xpos > cp->select.base.xpos )
            cp->slScrollLeft () ;
         while ( cp->tbIndex.xpos < cp->select.base.xpos )
            cp->slScrollRight () ;
         if ( ! cp->select.fwd )
         {
            cp->slScrollRight () ;
            wkt.key = nckBKSP ;
         }
      }
      for ( short i = cp->select.count ; i > ZERO ; i-- )
         cp->slDeleteChar ( wkt ) ;
   }
   else                          // multi-line controls
   {
      while ( cp->tbIndex.ypos > cp->select.base.ypos )
         cp->mlScrollUp () ;
      while ( cp->tbIndex.ypos < cp->select.base.ypos )
         cp->mlScrollDown () ;
      if ( cp->rtlContent )
      {
         while ( cp->tbIndex.xpos > cp->select.base.xpos )
            cp->mlScrollRight () ;
         while ( cp->tbIndex.xpos < cp->select.base.xpos )
            cp->mlScrollLeft () ;
         if ( ! cp->select.fwd )
         {
            cp->mlScrollLeft () ;
            wkt.key = nckBKSP ;
         }
      }
      else  // LTR Content
      {
         while ( cp->tbIndex.xpos > cp->select.base.xpos )
            cp->mlScrollLeft () ;
         while ( cp->tbIndex.xpos < cp->select.base.xpos )
            cp->mlScrollRight () ;
         if ( cp->select.fwd == false )
         {
            cp->mlScrollRight () ;
            wkt.key = nckBKSP ;
         }
      }
      for ( short i = cp->select.count ; i > ZERO ; i-- )
         cp->mlDeleteChar ( wkt ) ;
   }
   cp->select.reset() ;       // cancel the selection

}  //* End etbDeleteSelection() *

//**************************
//* etbProcessReservedKeys *
//**************************
//******************************************************************************
//* PRIVATE METHOD: Called by the EditXxText() group.                          *
//* Process a member of the list of reserved keys set by                       *
//* SetTextboxReservedKeys().                                                  *
//*                                                                            *
//* Input  : cp  : pointer to DialogTextbox object under edit                  *
//*          wk  : user's key input,                                           *
//*                one of the members of the reserved-key list ('resKeys')     *
//*                                                                            *
//* Returns: OK                                                                *
//******************************************************************************

short NcDialog::etbProcessReservedKeys ( DialogTextbox* cp, const wkeyCode& wk )
{
   bool boink = false, flag ;
   if ( (flag = (wk == this->resKeys.selRightKey)) ||
        (wk == this->resKeys.selLeftKey) )
   {
      if ( (cp->tbSelectChar ( flag )) != OK )
         boink = true ;          // boink the user for invalid selection
   }
   else if ( wk == this->resKeys.selAllKey )
   {
      if ( (cp->tbSelectText ( true )) != OK ) 
         boink = true ;          // boink the user for trying to select air
   }
   else if ( (flag = (wk == this->resKeys.cutKey)) ||
             (wk == this->resKeys.copyKey) )
   {
      if ( (this->etbTb2Clipboard ( flag )) != OK )
         boink = true ;          // boink the user for trying to copy air
   }
   else if ( wk == this->resKeys.pasteKey )
   {
      if ( (this->etbClipboard2Tb ()) != OK )
         boink = true ;          // boink the user for trying to paste air
   }

   if ( boink && (cp->audibleAlert) )
      this->UserAlert () ;

   return OK ;

}  //* End etbProcessReservedKeys() *

//*************************
//*   SetLocalClipboard   *
//*************************
//******************************************************************************
//* Copy the specified text data to the Textbox Local Clipboard buffer.        *
//*                                                                            *
//* Input  : srcText: text to be stored                                        *
//*                                                                            *
//* Returns: number of characters stored (including NULLCHAR)                  *
//******************************************************************************
//* Note that source data are not associated with a textbox control.           *
//******************************************************************************

short NcDialog::SetLocalClipboard ( const gString& srcText )
{

   return ( tbLocalClipboard.tblcStore ( srcText, MAX_DIALOG_CONTROLS ) ) ;

}  //* End SetLocalClipboard() *

//*************************
//*   GetLocalClipboard   *
//*************************
//******************************************************************************
//* Retrieve a copy of the current contents of the Textbox Local Clipboard     *
//* buffer.                                                                    *
//*                                                                            *
//* Input  : trgText: receives the text data                                   *
//*                                                                            *
//* Returns: number of characters retrieved (including NULLCHAR)               *
//******************************************************************************

short NcDialog::GetLocalClipboard ( gString& trgText )
{

   tbLocalClipboard.tblcRetrieve ( trgText ) ;
   return trgText.gschars() ;

}  //* End GetLocalClipboard() *

//***************************
//* GetLocalClipboard_Bytes *
//***************************
//******************************************************************************
//* Returns the number of data bytes currently in the local clipboard buffer.  *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: number of data bytes (including NULLCHAR)                         *
//******************************************************************************

short NcDialog::GetLocalClipboard_Bytes ( void )
{

   return ( tbLocalClipboard.tbClipboard.utfbytes() ) ;

}  //* End GetLocalClipboard_Bytes() *

//***************************
//* GetLocalClipboard_Chars *
//***************************
//******************************************************************************
//* Returns the number of characters currently in the local clipboard buffer.  *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: number of (wchar_t) characters (including NULLCHAR)               *
//******************************************************************************

short NcDialog::GetLocalClipboard_Chars ( void )
{

   return ( tbLocalClipboard.tbClipboard.gschars() ) ;

}  //* End GetLocalClipboard_Chars() *

//*************************
//*     EditTextbox       *
//*************************
//******************************************************************************
//* If control with input focus == dctTEXTBOX, call this method to get user's  *
//* key input. Allows user to edit the text in the text box control.           *
//* Returns when editing is complete (nckENTER), when Textbox IS READY to lose *
//* focus (nckTAB or nckSTAB), or when Textbox has lost focus due to a hotkey  *
//* press. The nckESC (Escape) key causes any changes to the text to be        *
//* discarded and a return to caller as if the nckTAB key had been pressed.    *
//*                                                                            *
//* Input  : uiInfo class (by reference) - initial values ignored              *
//*                                                                            *
//* Returns: index of control that currently has the input focus               *
//*           (See note in NcDialog.hpp about interpretation )                 *
//*           (of values returned in the uiInfo-class object.)                 *
//******************************************************************************
//* Public method for editing text in dctTEXTBOX controls.                     *
//* Here, we determine whether the text box is set up for left-justified,      *
//* right-justified or RTL input, as well as whether the target is a           *
//* single-line or multi-line control and call the appropriate private method. *
//*                                                                            *
//* If debugging flag(s) at top of module are set, then the debugging window   *
//* for textbox editing is opened and closed here.                             *
//*                                                                            *
//* Programmer's Note: When the cursor becomes visible, or is moved or the     *
//* terminal window gets focus, the cursor will blink approximately 10 times   *
//* before stablizing as a solid block. This is apparently a 'feature', of the *
//* terminal emulator (GNOME 3.2.1), and we have little control over it.       *
//******************************************************************************

short NcDialog::EditTextbox ( uiInfo& info )
{
   //* If control with focus is actually a text box *
   if ( this->dCtrl[this->currCtrl]->type == dctTEXTBOX )
   {
      DialogTextbox* cp = (DialogTextbox*)this->dCtrl[this->currCtrl] ;

      #if DEBUG_ET != 0 || DEBUG_ETM != 0    // For debugging only
      // NOTE: This opens an auxilliary dialog for display of debugging information.
      //       Dialog is designed specifically to work with Dialog1, Test01.
      //       Also works with the cut-and-paste test application: Dialogx.
      etRows = etROWS ;                // base window height
      short sy, sx ;                   // auto-expand the window
      nc.ScreenDimensions ( sy, sx ) ;
      sx = (sy - 1) - (etBaseY + etROWS) ;
      if ( sx > ZERO )
         etRows += sx ;
      etColor = nc.blR ;               // message color
      InitNcDialog dInit( etRows,      // number of display lines
                          etCols,      // number of display columns
                          etBaseY,     // Y offset from upper-left of terminal
                          etBaseX,     // X offset from upper-left of terminal 
                          "  Debug  EditTextbox  ", // dialog title
                          ncltSINGLE,  // border line-style
                          nc.rebl,     // border color attribute
                          etColor,     // interior color attribute
                          NULL         // pointer to list of control definitions
                        ) ;
      etPtr = new NcDialog( dInit ) ;
      if ( (etPtr->OpenWindow ()) == OK  )
      { /* Nothing to do here */  }
      #endif   // DEBUG_ET || DEBUG_ETM

      if ( cp->rightJust != false )
         this->EditRjText ( cp, info ) ;  // right-justified input (single-line only)
      else if ( this->dCtrl[this->currCtrl]->lines == 1 )
         this->EditSlText ( cp, info ) ;  // edit data in single-line controls
      else  // (lines > 1)
         this->EditMlText ( cp, info ) ;  // edit data in multi-line controls

      #if DEBUG_ET != 0 || DEBUG_ETM != 0    // For debugging only
      #if 0    // display return data
      winPos dui( this->dulY, (this->dulX + this->dCols - 3) ) ;
      this->Dump_uiInfo ( dui, info ) ;
      #endif   // display return data
      if ( etPtr != NULL )
         delete ( etPtr ) ;
      etPtr = NULL ;
      #endif   // DEBUG_ET || DEBUG_ETM
   }
   else                  // control is not a text box i.e. application error
   {
      //* Do what we can to minimize the damage *
      info.ctrlType = this->dCtrl[this->currCtrl]->type ;
      info.ctrlIndex = this->currCtrl ;
      info.keyIn = nckTAB ;               // move on to next control
      info.hasFocus = true ;              // control has focus
      info.dataMod = false ;              // no data modified
      info.selMember = MAX_DIALOG_CONTROLS ; // don't care
      info.isSel = false ;                // don't care
      info.viaHotkey = false ;            // invalidate any hotkey data
   }

   return this->currCtrl; 

}  //* End EditTextbox() *

//*************************
//*      EditSlText       *
//*************************
//******************************************************************************
//* User interface for editing text in single-line dctTEXTBOX controls.        *
//*                                                                            *
//*                                                                            *
//* Input  : pointer to DialogTextbox object                                   *
//*        : uiInfo class (by reference) - initial values ignored              *
//*           (see note in NcDialog.hpp, about values returned)                *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note: Hotkey support for accessing dialog controls is         *
//* limited while inside a text box. Because most key input is valid for the   *
//* text box itself, only control-key combinations nckC_A <= hotkey <= nckC_Z  *
//* are recognized as hotkeys.                                                 *
//*                                                                            *
//* Tracking cursor position and character substitution within the text edit   *
//* poses some problems for multi-column characters in the text string.        *
//* Our decision is that we will allow the gString class to tell us what we    *
//* need to know, in order to keep UTF-8 and wchar_t conversions, byte counts, *
//* column counts and other stuff encapsulated. This make the  editing routine *
//* rather inefficient in terms of CPU cycles; however, most of our time is    *
//* spent waiting for user input, so performance is not a big issue here.      *
//*                                                                            *
//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *
//* Programmer's Note: For text data that is wider than the display field:     *
//*  IMPORTANT NOTE: When shifting the display text, there are possible        *
//*                  complications if the display-field width is less than     *
//*                  three (3) characters (3-6 columns). Testing for this and  *
//*                  compensating for it are complex issues.                   *
//*                - We have implemented a compromise which disallows text     *
//*                  shifting for left-justified or RTL text in controls of    *
//*                  less than MIN_TB_SHIFT_WIDTH columns.                     *
//*                - Text shifting is not necessary in right-justified fields  *
//*                  because data are not allowed to be wider than the field.  *
//*                                                                            *
//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *
//*  Editing text in left-to-right (LTR) and right-to-left (RTL) fields:       *
//*  -------------------------------------------------------------------       *
//*  Define 'Base Position':                                                   *
//*    For LTR data this is the left edge of the field.                        *
//*    For RTL data this is the right edge of the field.                       *
//*  Define 'Append Position':                                                 *
//*    The first unoccupied column following the existing text.                *
//*                                                                            *
//*  Note on RTL languages:                                                    *
//*    According to Wikipedia (27 May, 2014), the following are the the major  *
//*    RTL (right-to-left) written languages. Arabic, Aramaic,                 *
//*    Southern Balochi, Bakthiari, Sorani, Dhivehi, Persian, Gilaki, Hebrew,  *
//*    Kurdish, Mazanderani, Western Punjabi, Pashto, Sindhi, Uyghur, Urdu,    *
//*    Yiddish.                                                                *
//*    Note that some languages can be written top-to-bottom (LTR or RTL), but *
//*    those formats are not directly supported.                               *
//*                                                                            *
//*  For text data that is less-than-or-equal-to the width of the control, the *
//*  first character in the string is always displayed at the base position    *
//*  of the field.                                                             *
//*  Note: Shifting of text data within the field is enabled/disabled by the   *
//*        'hzShift' member of the DialogTextbox class. If text shift is       *
//*        disabled, text may not grow beyond the width of the field, so edit  *
//*        is relatively simple.                                               *
//*                                                                            *
//*  For text data that is greater than width of control, text shift must be   *
//*  enabled, and life within the world of text editing becomes more complex.  *
//*   a) If control does not have focus (is not being edited), first character *
//*      of string is displayed in base position and display of string         *
//*      is truncated at right edge of the control.                            *
//*      EXCEPTION: If application wants tail of string displayed when control *
//*                 does not have the input focus, the application will call   *
//*                 DisplayTextboxTail() method. This does not affect edit.    *
//*   b) During edit, the text string is shifted as necessary to keep the      *
//*      character under the cursor in the visible field. There are are some   *
//*      rules about how to do this:                                           *
//*      - HOME key detected: first character is displayed in base position,   *
//*        and cursor references that character.                               *
//*                                                                            *
//*      - END key detected: text data is shifted so that there are appendCOLS *
//*        empty columns at the trailing edge of the control, and cursor       *
//*        references the 'append' position i.e. the first empty column.       *
//*                                                                            *
//*      - LEFT ARROW key detected: if the next character to left of current   *
//*        cursor position is visible, simply move the cursor left by the      *
//*        number of columns occupied by the target character.                 *
//*          - Note that for LTR text, the cursor is ALWAYS positioned at the  *
//*            left column of a two-column character.                          *
//*          - Note that for RTL text, the cursor is ALWAYS positioned at the  *
//*            right column of a two-column character.                         *
//*        If the next character to the left of cursor is not currently        *
//*        visible, shift the entire string to the right by the number of      *
//*        columns occupied by the target character, one column for single-    *
//*        column characters and two columns for 2-column characters. Cursor   *
//*        references the target character.                                    *
//*                                                                            *
//*      - RIGHT ARROW key detected: if the next character to the right of     *
//*        current cursor position is visible, simply move the cursor right    *
//*        by the number of columns occupied by the character currently under  *
//*        the cursor.                                                         *
//*          - Note that for LTR text, the cursor is ALWAYS positioned at the  *
//*            left column of a two-column character.                          *
//*          - Note that for RTL text, the cursor is ALWAYS positioned at the  *
//*            right column of a two-column character.                         *
//*        If the next character to the right of cursor is not currently       *
//*        visible, shift the entire string to the left by the number of       *
//*        columns occupied by the target character, one column for single-    *
//*        column characters and two columns for 2-column characters. Cursor   *
//*        references the target character.                                    *
//*        Note that if the width of the character being shifted OUT is        *
//*        different from the width of the character being shifted in, then    *
//*        some adjustment must be made to compensate.                         *
//*        If there is no character to the right, then rules for END key apply.*
//*                                                                            *
//*      - Append a character:                                                 *
//*        Appending a character to the end-of-data requires free space within *
//*        the display field. The amount of space needed depends on the width  *
//*        of the character to be appended. Because we can't know this ahead   *
//*        of time, we ensure that there are at least appendCOLS columns       *
//*        of free space in the field following the existing text before       *
//*        allowing user to append a new character. See appendCOLS constant.   *
//*                                                                            *
//*      - Insert a character:                                                 *
//*        The character under the cursor and all characters following it are  *
//*        shifted toward end-of-data to make room for the new character.      *
//*        Depending on the amount of free space following the displayed data, *
//*        one or more characters at the opposite edge of field MAY need to be *
//*        shifted out of view to make room for display of new character.      *
//*        Additionally, the cursor is advanced by one character after         *
//*        insertion is complete.                                              *
//*        This means that an insertion at the trailing edge of the field may  *
//*        also require that a character be shifted out on the from the        *
//*        opposite edge to bring the character now under cursor into view.    *
//*                                                                            *
//*      - Replace a character:                                                *
//*        The character under the cursor is replaced by the new character.    *
//*        If the new character is not the same width as the replaced          *
//*        character, the position of all following characters will need to be *
//*        adjusted. If there is no free space following the displayed data,   *
//*        these adjustments may entail either shifting character(s) out of    *
//*        view from the opposite edge OR shifting character(s) (or free space)*
//*        into view from the near edge. Additionally, the cursor is           *
//*        advanced by one character after the replacement is complete. This   *
//*        means that a replacement at the trailing edge of the field may also *
//*        require that a character be shifted out from the opposite edge to   *
//*        bring the character now under cursor into view.                     *
//*                                                                            *
//*      - Delete a character:                                                 *
//*        This can take two forms:                                            *
//*          1) Delete the character under the cursor (nckDELETE key)          *
//*          2) Delete the character before the cursor (nckBKSP key)           *
//*        Either way, the target character is removed from the string and all *
//*        data that follows it is shifted to fill the void. Characters        *
//*        (or free space) are shifted into view as necessary to re-fill the   *
//*        field.                                                              *
//*                                                                            *
//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *
//* Moving about in a text string that may or may not have multi-column        *
//* characters is a rather complex task. For instance:                         *
//* 1) Characters may require 0, 1 or 2 display columns. Can we assume that no *
//*    character will require more than 2 columns? Well, probably.             *
//*    What other assumptions are we allowed to make?                          *
//*                                                                            *
//* 2) It is known that some characters such as non-printing meta-characters   *
//*    and control codes require ZERO columns, while most Western characters   *
//*    require 1 column, and Chinese/Japanese/Korean (CJK) characters typically*
//*    require two (2) columns.                                                *
//*    As a working hypothesis then, we can assume that any character a user   *
//*    might enter will require 0, 1, or 2 columns; thus, if the cursor is     *
//*    just beyond the end of the existing string, we need to be sure that at  *
//*    least 2 columns (appendCOLS) of space are available to display whatever *
//*    character the user may enter (unless the field is one column wide).     *
//*    (We don't allow non-printing control characters in a text box, but some *
//*    zero-column meta-characters (character-modification codes) exist.       *
//*                                                                            *
//* 3) If the field width is 1 or 2 columns only, this will present problems   *
//*    of its own as a special case. Note that an application programmer may   *
//*    define a field 1 column wide, but would then need to limit user input   *
//*    to characters that require only one column to display; since a 1-column *
//*    field is not wide enough to display a 2-column character. While this is *
//*    really not our problem, we need to keep it in mind, so we can gracefully*
//*    display a question mark or '#' or similar if it happens.                *
//*    A three-column field should give us no trouble in our calculations, but *
//*    a 2-column field could cause some < ZERO or divide-by-zero issues.      *
//*    To alleviate some of these worries, we have implemented the             *
//*    DialogTextbox class data member 'hzShift' which disallows horizontal    *
//*    shifting of text for for controls of less than MIN_TB_SHIFT_WIDTH.      *
//*    Thus, shift calculations always have enough columns to work with so     *
//*    negative distances and divide-by-zero don't happen there.               *
//*                                                                            *
//* 4) Overstrike mode: User may overwrite a 1-column character with a         *
//*    2-column character, or overwrite a 2-column character with a 1-column   *
//*    character, thus playing merry hell with our column-width data.          *
//*                                                                            *
//* 5) Insert mode: User may insert a character into an existing string, so    *
//*    how many columns must the following data be shifted, and how many       *
//*    columns (if any) must the cursor be advanced? Interesting problems....  *
//*                                                                            *
//* 6) Deletion: User may delete characters randomly from anywhere in the      *
//*    existing text data. How do we shift the remaining data into the vacated *
//*    space? It's easy for ASCII since 1 column == 1 character. What are the  *
//*    issues if character is not a 1-column character? Let's not even dream   *
//*    about deletion of a zero-column meta-character. If a displayed character*
//*    includes meta-characters, then they must be deleted as a group.         *
//*    This issue is worrisome.                                                *
//*                                                                            *
//* 7) Because the text data string may require more columns than are available*
//*    in the control's display area, we need to be able to shift the text     *
//*    left and right as user scrolls through the data or enters new/additional*
//*    text.                                                                   *
//*                                                                            *
//* 8) The call to SetTextboxCursor() must transparently place the cursor on   *
//*    the leading column of a two-column character, or scrolling left and     *
//*    right could produce some odd results that the user really doesn't want  *
//*    to see.                                                                 *
//*                                                                            *
//*                                                                            *
//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *
//* Processing keycodes reserved by the application for Selection or           *
//* Copy/Cut/Paste operations.                                                 *
//* 1) If we are on the Local Clipboard only, then there is no need for a      *
//*    callback method to process clipboard requests.                          *
//*                                                                            *
//* 2) If application has successfully initialized access to the system        *
//*    clipboard through the embedded WaylandCB interface, then no further     *
//*    action () e.g. a callback) is required at the application level.        *
//*    Communication among the Textbox controls, the local NcDialog clipboard  *
//*    and the Wayland system clipboard is handled automatically.              *
//*                                                                            *
//* 3) If application is accessing the system clipboard independently, then    *
//*    access through the embedded WaylandCB interface should remain disabled  *
//*    to avoid potential resource conflicts.                                  *
//*    a) Of course, we don't know if the application is directly accessing    *
//*       the system clipboard, but if it is, then we must call the external   *
//*       update method in the proper sequence.                                *
//*       - For Copy and Cut operations, we must move the data from the source *
//*         Textbox to the the Local Clipboard BEFORE calling the external     *
//*         update method, so the application can (if it wants) transfer the   *
//*         data to the system clipboard.                                      *
//*       - For Paste operations, we must call the external update method      *
//*         BEFORE the operation, so the application can (if it wants)         *
//*         retrieve data from the system clipboard and place it in the        *
//*         Local Clipboard so we can THEN copy it from the Local Clipboard    *
//*         to the target Textbox.                                             *
//*    b) For selection operations, everything is handled internally so timing *
//*       of the call to the external update method is not critical.           *
//*                                                                            *
//******************************************************************************

void NcDialog::EditSlText ( DialogTextbox* cp, uiInfo& info )
{
   //* Cursor state (visibility) on entry *
   ncCurVis oldState = nc.GetCursorState () ;

   //* Make a backup copy of the existing text data for later comparison.*
   gString textBackup, textNew ;
   cp->tbGetText ( textBackup ) ;

   //* Re-calculate cursor position. It 'should' be ok, but be safe.  *
   cp->tbSynchCursor ( true ) ;
   nc.SetCursorState ( nccvNORMAL ) ;     // make cursor visible

   //**************************
   //* Interact with the user *
   //**************************
   bool  done = false,           // loop control
         rkey = false ;          // 'reserved-key-detected' flag
   do
   {
      #if DEBUG_ET != 0    // For debugging only
      this->ET_Stat ( cp ) ;
      #endif   // DEBUG_ET

      //* 1) Get user input.                                  *
      //* 2) Check for toggle of Insert/Overstrike mode.      *
      //* 3) Check for terminal-resize event.                 *
      //* 4) Check for application-specific 'reserved' keys   *
      //* 5) Access callback method, if specified.            *
      //* 6) If key input handled, return to top of loop.     *
      this->etbGetUserText ( cp, info ) ;

      if ( ((info.wk.type == wktFUNKEY)
            && (info.wk.key == nckINSERT || info.wk.key == nckRESIZE))
           || (rkey = this->resKeys.isReservedKey ( info.wk )) )
      {
         if ( info.wk.type == wktFUNKEY && info.wk.key == nckINSERT )
            this->ToggleIns () ;
         else if ( info.wk.type == wktFUNKEY && info.wk.key == nckRESIZE )
            this->TermResized () ;

         //* Process a keycode reserved by the application *
         //* for Selection, Copy, Cut or Paste operations. *
         //*   (see notes EditSlText() in method header)   *
         if ( (rkey != false) )
         {
            //* If WaylandCB is active, process any reserved keycode.*
            if ( ((wcbClip != NULL) && (wcbClip->wcbIsConnected ())) ||
                 //* or if no callback method specified, *
                 (ExternalControlUpdate == NULL) ||
                 //* or if copy/cut operation (not paste)*
                 (info.wk != this->resKeys.pasteKey) )
            {
               this->etbProcessReservedKeys ( cp, info.wk ) ;
               rkey = false ;       // reset the reserved-keys flag
            }
         }

         //* If a callback method has been defined *
         if ( ExternalControlUpdate != NULL )
         {
            ExternalControlUpdate ( this->currCtrl, info.wk, false ) ;
            cp->wPtr->SetCursor ( cp->tbCursor ) ; // in case cursor was moved
            cp->wPtr->RefreshWin () ;              // update the display

            //* If unprocessed reserved keycode, i.e. a Paste operation, *
            //* (with WaylandCB inactive), process it now.               *
            if ( rkey != false )
            {
               this->etbProcessReservedKeys ( cp, info.wk ) ;
               rkey = false ;       // reset the reserved-keys flag
            }
         }
         continue ;
      }

      //* If a selection operation is in progress, but we          *
      //* have a non-selection key, cancel the selection.          *
      //* Exception: If DEL key, allow selected text to be deleted.*
      if ( cp->select.active &&
           ((info.wk.key != nckDELETE) || (info.wk.type != wktFUNKEY)) )
      {
         cp->select.reset() ;
         cp->RedrawControl ( true, true ) ;
      }

      //* Apply text filter to input. *
      if ( info.wk.type == wktPRINT && 
           ((cp->tbApplyFilter ( &info.wk.key )) != false) )
      {
         if ( this->overStrike != false )    //** overstrike-mode input **
         {
            if ( ((cp->slReplaceChar ( info.wk )) == false) && (cp->audibleAlert) )
               this->UserAlert () ;
         }
         else                                //** insert-mode input **
         {
            if ( ((cp->slInsertChar ( info.wk )) == false) && (cp->audibleAlert) )
               this->UserAlert () ;
         }
      }
      else
      {  //* If:                                                               *
         //* 1) user scrolling through data using cursor keys,      --OR--     *
         //* 2) leaving edit via nckTAB, nckSTAB, nckESC, nckENTER, --OR--     *
         //* 3) erasing previous input via nckDELETE, nckBKSP,      --OR--     *
         //* 4) performing 'cut', 'copy', 'paste' (if defined),     --OR--     *
         //* 5) pressing a 'hotkey' (Ctrl+key) for another control, --OR--     *
         //* 6) pressing some other function key (F01 - F12,                   *
         //*    Ctrl+F01 - Ctrl+F12, Alt+F01 - Alt+F12, etc.)                  *
         //* -- then it is not a key to be added to the text.                  *
         //* It is also possible (though unlikely) that an un-handled Escape   *
         //* sequence was captured or that system returned an error.           *

         //* ENTER key is transformed to TAB key.                         *
         if ( info.wk.type == wktFUNKEY && 
              (info.wk.key == nckENTER || info.wk.key == nckpENTER) )
            info.wk.key = nckTAB ;

         //* Backspace or Delete erases a character *
         if ( (info.wk.type == wktFUNKEY) && 
              (info.wk.key == nckBKSP || info.wk.key == nckDELETE) ) 
         {
            if ( !this->disableIns || !this->overStrike )
            {
               //* If 'selected' text, then delete key deletes the selection.*
               if ( (cp->select.active) && (info.wk.key == nckDELETE) )
               {
                  this->etbDeleteSelection ( cp ) ;
                  cp->RedrawControl ( true, true ) ;
               }
               else if ( !(cp->slDeleteChar ( info.wk )) && (cp->audibleAlert) )
                  this->UserAlert () ;
            }
            else if ( cp->audibleAlert )
               this->UserAlert () ;
         }

         //* Scrolling through data *
         else if ( info.wk.type == wktFUNKEY && info.wk.key == nckLEFT )
            cp->slScrollLeft () ;
         else if ( info.wk.type == wktFUNKEY && info.wk.key == nckRIGHT )
            cp->slScrollRight () ;
         else if ( info.wk.type == wktFUNKEY && info.wk.key == nckHOME )
            cp->slScrollHome () ;
         else if ( info.wk.type == wktFUNKEY && info.wk.key == nckEND )
            cp->slScrollEnd () ;

         //* Else if edit complete, return the results.*
         else if ( info.wk.type == wktFUNKEY && 
                   (info.wk.key == nckTAB || info.wk.key == nckSTAB || 
                    info.wk.key == nckUP  || info.wk.key == nckDOWN || 
                    info.wk.key == nckESC) )
         {
            info.ctrlType = dctTEXTBOX ;        // this is a text box control
            info.ctrlIndex = this->currCtrl ;   // control with input focus
            info.hasFocus = true ;              // focus was not lost
            info.selMember = MAX_DIALOG_CONTROLS ; // don't care
            info.isSel = false ;                // don't care
            info.viaHotkey = false ;            // control did not get focus via hotkey
            if (info.wk.key == nckSTAB || info.wk.key == nckUP)
               info.keyIn = nckSTAB ;           // focus to move backward to previous control
            else  // (key==nckTAB || key==nckDOWN || key==nckESC )
               info.keyIn = nckTAB ;            // focus to move forward to next control

            //* If ESC, then discard any edits by restoring original text.*
            if ( info.wk.key == nckESC )
            {
               cp->tbSetText ( textBackup ) ;
               info.dataMod = false ;
            }
            //* Else, determine whether data have been modified.*
            else
            {
               textNew = cp->wText ;
               info.dataMod = bool(textNew != textBackup) ;
            }
            done = true ;                       // exit the input loop
         }

         //* Check for hotkey (CTRL+['A' ... 'Z']) OR *
         //* 'default' hotkey if mouse enabled        *
         else if ( (info.wk.type == wktFUNKEY && 
                    (info.wk.key >= nckC_A && info.wk.key <= nckC_Z)) ||
                   ((this->meMouseEnabled ()) && 
                    (info.wk.mevent.conv != false)) )
         {
            //* Scan the controls for one whose hotkey matches the input.*
            //* If match found, returns control's index, else -1.        *
            //* Ignore hotkey selection of this text box.                *
            short hotIndex = this->IsHotkey ( info.wk ) ;
            if ( hotIndex >= ZERO && hotIndex != this->currCtrl )
            {  //* Make the indicated control the current/active control *
               //* and indicate the results of our edit.                 *
               info.ctrlType = dctTEXTBOX ;     // this is a text box control
               info.ctrlIndex = this->currCtrl ;// control with input focus
               info.hasFocus = true ;           // this may be reset by ChangedFocusViaHotkey()
               info.keyIn = ZERO ;              // indicate no shift in focus necessary
               info.selMember = MAX_DIALOG_CONTROLS ; // don't care
               info.isSel = false ;             // don't care
               info.viaHotkey = false ;         // this may be set by ChangedFocusViaHotkey()
               //* Determine whether data have been modified. *
               textNew = cp->wText ;
               info.dataMod = bool(textNew != textBackup) ;
               done = true ;                    // exit the input loop
               this->ChangedFocusViaHotkey ( info.wk.key, hotIndex, info ) ;
               // on return, new control has the focus, and the info.h_XX
               // variables have been adjusted for that control
            }
            else if ( hotIndex < ZERO )
            {
               //* Key input value is not valid in this context, ignore it. *
               //* If audible alert enabled for invalid key data.           *
               if ( cp->audibleAlert && (info.wk.type != wktMOUSE) )
                  this->UserAlert () ;
            }
         }
         else
         {
            //* Key input value is not valid in this context, ignore it. *
            //* If audible alert enabled for invalid key data.           *
            if ( cp->audibleAlert && (info.wk.type != wktMOUSE) )
               this->UserAlert () ;
         }
      }
      if ( ExternalControlUpdate != NULL )
      {
         ExternalControlUpdate ( this->currCtrl, info.wk, false ) ;
         cp->wPtr->SetCursor ( cp->tbCursor ) ; // in case cursor was moved
         cp->wPtr->RefreshWin () ;              // update the display
      }
   }
   while ( ! done ) ;

   nc.SetCursorState ( oldState ) ;          // restore cursor invisibility

}  //* End EditSlText() *

//*************************
//*      EditMlText       *
//*************************
//******************************************************************************
//* User interface for editing text in multi-line dctTEXTBOX controls.         *
//*                                                                            *
//*                                                                            *
//* Input  : cp  : pointer to DialogTextbox object                             *
//*        : info: uiInfo class (by reference) - initial values ignored        *
//*                 (see note in NcDialog.hpp, about values returned)          *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Notes on multi-line text editing:                                          *
//* ---------------------------------                                          *
//* 1) Multi-line text is a single text stream, optionally broken into segments*
//*    by line-break ('\n') characters for ease of formatting. The stream is   *
//*    displayed on successive lines of the control.                           *
//*    a) Soft line breaks:                                                    *
//*       - Calculated at SPACE characters (0x20).                             *
//*       - If practical, the SPACE character is left on the current line, and *
//*         the following text is moved down to the next line.                 *
//*       - However, if the last word on the newly-formatted current line      *
//*         occupies the last column of the current line, then the SPACE       *
//*         character and the following text are moved down to the next line   *
//*         together.                                                          *
//*       - If a line ends with a multi-column character or if the first       *
//*         character moved to the next line is a multi-column character, then *
//*         an inaccessible empty column may end the current line.             *
//*                   (internationalization can be a pain :-)                  *
//*                                                                            *
//* 2) Line wrapping occurs:                                                   *
//*    - When the new text shifts existing text out of view, OR                *
//*    - When the curser goes beyond the right edge of the current line.       *
//*                                                                            *
//* 3) 'Insert' and 'Overstrike' modes:                                        *
//*    a) 'Insert Mode': As new characters are entered, existing text under,   *
//*        or to the right of cursor is shifted to the right/down to make room *
//*        for the new character.                                              *
//*    b) 'Overstrike Mode': When a new character is entered, it replaces the  *
//*       character under the cursor. If old and new characters occupy         *
//*       a different number of columns, existing text to the right of cursor  *
//*       is shifted to compensate.                                            *
//*                                                                            *
//* 4) Note that TAB and SHIFT+TAB characters are not recognized as valid      *
//*    display characters, but instead indicate that user has finished the     *
//*    edit and is moving to next/previous control. The ENTER key is           *
//*    interpreted as a TAB character.                                         *
//*                                                                            *
//* 5) Special characters may be defined for 'cut', 'copy' and 'paste' of text *
//*    in a textbox under edit.                                                *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//******************************************************************************

void NcDialog::EditMlText ( DialogTextbox* cp, uiInfo& info )
{
   //* Cursor state (visibility) on entry *
   ncCurVis oldState = nc.GetCursorState () ;

   //* Make a backup copy of the existing text data for later comparison.*
   gString textBackup, textNew ;
   cp->tbGetText ( textBackup ) ;

   //* Shift the text as necessary to make cursor position visible *
   cp->RedrawControl ( true ) ;

   nc.SetCursorState ( nccvNORMAL ) ;     // make cursor visible
   cp->wPtr->SetCursor ( cp->tbCursor ) ; // set cursor position
   cp->wPtr->RefreshWin () ;              // refresh the display

   //**************************
   //* Interact with the user *
   //**************************
   bool  dataModified = false,   // true if char inserted/replaced/deleted
         rkey = false,           // 'reserved-key-detected' flag
         done = false ;          // loop control
   do
   {
      #if DEBUG_ETM != 0   // For debugging only
      this->ET_Stat ( cp ) ;
      #endif   // DEBUG_ETM

      //* 1) Get user input.                                  *
      //* 2) Check for toggle of Insert/Overstrike mode.      *
      //* 3) Check for terminal-resize event.                 *
      //* 4) Check for 'reserved' keys (application specific) *
      //* 5) Access callback method, if specified.            *
      //* 6) If key input handled, return to top of loop.     *
      this->etbGetUserText ( cp, info ) ;
      if ( ((info.wk.type == wktFUNKEY)
            && (info.wk.key == nckINSERT || info.wk.key == nckRESIZE))
           || (rkey = this->resKeys.isReservedKey ( info.wk )) )
      {
         if ( info.wk.type == wktFUNKEY && info.wk.key == nckINSERT )
            this->ToggleIns () ;
         else if ( info.wk.type == wktFUNKEY && info.wk.key == nckRESIZE )
            this->TermResized () ;

         //* Process a keycode reserved by the application *
         //* for Selection, Copy, Cut or Paste operations. *
         //*   (see notes EditSlText() in method header)   *
         if ( (rkey != false) )
         {
            //* If WaylandCB is active, process any reserved keycode.*
            if ( ((wcbClip != NULL) && (wcbClip->wcbIsConnected ())) ||
                 //* or if no callback method specified, *
                 (ExternalControlUpdate == NULL) ||
                 //* or if copy/cut operation (not paste)*
                 (info.wk != this->resKeys.pasteKey) )
            {
               this->etbProcessReservedKeys ( cp, info.wk ) ;
               rkey = false ;       // reset the reserved-keys flag
            }
         }

         //* If a callback method has been defined *
         if ( ExternalControlUpdate != NULL )
         {
            ExternalControlUpdate ( this->currCtrl, info.wk, false ) ;
            cp->wPtr->SetCursor ( cp->tbCursor ) ; // in case cursor was moved
            cp->wPtr->RefreshWin () ;              // update the display

            //* If unprocessed reserved keycode, i.e. a Paste operation, *
            //* (with WaylandCB inactive), process it now.               *
            if ( rkey != false )
            {
               this->etbProcessReservedKeys ( cp, info.wk ) ;
               rkey = false ;       // reset the reserved-keys flag
            }
         }
         continue ;
      }

      //* If a selection operation is in progress, but we          *
      //* have a non-selection key, cancel the selection.          *
      //* Exception: If DEL key, allow selected text to be deleted.*
      if ( cp->select.active &&
           ((info.wk.key != nckDELETE) || (info.wk.type != wktFUNKEY)) )
      {
         cp->select.reset() ;
         cp->RedrawControl ( true, true ) ;
      }

      //* Apply text filter to input. *
      if ( info.wk.type == wktPRINT && 
           ((cp->tbApplyFilter ( &info.wk.key )) != false) )
      {
         if ( this->overStrike != false )    //** overstrike-mode input **
         {
            if ( (cp->mlReplaceChar ( info.wk )) != false )
               dataModified = true ;
            else if ( cp->audibleAlert )
               this->UserAlert () ;
         }
         else                                //** insert-mode input **
         {
            if ( (cp->mlInsertChar ( info.wk )) != false )
               dataModified = true ;
            else if ( cp->audibleAlert )
               this->UserAlert () ;
         }
      }
      else
      {  //* If:                                                               *
         //* 1) user scrolling through data using cursor keys,      --OR--     *
         //* 2) leaving edit via nckTAB, nckSTAB, nckESC, nckENTER, --OR--     *
         //* 3) erasing previous input via nckDELETE, nckBKSP,      --OR--     *
         //* 4) performing 'cut', 'copy', 'paste' (if defined),     --OR--     *
         //* 5) pressing a 'hotkey' (Ctrl+key) for another control, --OR--     *
         //* 6) pressing some other function key (F01 - F12,                   *
         //*    Ctrl+F01 - Ctrl+F12, Alt+F01 - Alt+F12, etc.)                  *
         //* -- then it is not a key to be added to the text.                  *
         //* It is also possible (though unlikely) that an un-handled Escape   *
         //* sequence was captured or that system returned an error.           *

         //* ENTER key is transformed to TAB key.                     *
         //* If at end-of-data, AND if there have been no edits, then *
         //* Down Arrow is transformed to TAB key.                    *
         //* If at top-of-data, AND if there have been no edits, then *
         //* Up Arrow is transformed to STAB key.                     *
         if ( info.wk.type == wktFUNKEY )
         {
            if ( info.wk.key == nckENTER || info.wk.key == nckpENTER )
               info.wk.key = nckTAB ;
            else if ( (info.wk.key == nckDOWN) && !dataModified && (cp->mlEoD ()) )
               info.wk.key = nckTAB ;
            else if ( (info.wk.key == nckUP) && !dataModified && (cp->mlToD ()) )
               info.wk.key = nckSTAB ;
         }

         //* Backspace or Delete erases a character *
         if ( (info.wk.type == wktFUNKEY) && 
              (info.wk.key == nckBKSP || info.wk.key == nckDELETE) ) 
         {
            if ( !this->disableIns || !this->overStrike )
            {
               //* If 'selected' text, then delete key deletes the selection.*
               if ( (cp->select.active) && (info.wk.key == nckDELETE) )
               {
                  this->etbDeleteSelection ( cp ) ;
                  cp->RedrawControl ( true, true ) ;
               }
               else if ( (cp->mlDeleteChar ( info.wk )) != false )
                  dataModified = true ;
               else if ( cp->audibleAlert )
                  this->UserAlert () ;
            }
            else if ( cp->audibleAlert )
               this->UserAlert () ;
         }

         //* Scrolling merrily along... *
         else if ( info.wk.type == wktFUNKEY && info.wk.key == nckLEFT )
            cp->mlScrollLeft () ;
         else if ( info.wk.type == wktFUNKEY && info.wk.key == nckRIGHT )
            cp->mlScrollRight () ;
         else if ( info.wk.type == wktFUNKEY && info.wk.key == nckDOWN )
            cp->mlScrollDown () ;
         else if ( info.wk.type == wktFUNKEY && info.wk.key == nckUP )
            cp->mlScrollUp () ;
         else if ( info.wk.type == wktFUNKEY && info.wk.key == nckHOME )
            cp->mlScrollHome () ;
         else if ( info.wk.type == wktFUNKEY && info.wk.key == nckEND )
            cp->mlScrollEnd () ;

#if 0    // EXPERIMENTAL
         // - EditMlText(implement soft enter key Alt+Enter
         // - If this works, it can join wktPRINT, above.
         // - Unfortunately, the reformatting discards our newline character 
         // - UNLESS we set the 'hard-break' flag. This would require adding
         // - a parameter to the method which calls nkFmtDisplay(), e.g.
         // - mlInsertNewline (). Unfortunately, the line break would be 
         // - discarded when the next character is entered.

         //* Line break key - Alt+Enter key (aka: Alt+Ctrl+j) *
         else if ( info.wk.type == wktEXTEND && info.wk.key == nckAC_J )
         {
            info.wk.key = NEWLINE ;
            if ( (cp->mlInsertChar ( info.wk )) != false )
               dataModified = true ;
            else if ( cp->audibleAlert )
               this->UserAlert () ;
         }
#endif   // EXPERIMENTAL

         //* Edit complete, return the results.*
         else if ( (info.wk.type == wktFUNKEY) && 
                   ((info.wk.key == nckTAB)  || 
                    (info.wk.key == nckSTAB) ||
                    (info.wk.key == nckESC)) )
         {
            info.ctrlType = dctTEXTBOX ;        // this is a text box control
            info.ctrlIndex = this->currCtrl ;   // control with input focus
            info.hasFocus = true ;              // focus was not lost
            info.selMember = MAX_DIALOG_CONTROLS ; // don't care
            info.isSel = false ;                // don't care
            info.viaHotkey = false ;            // control did not get focus via hotkey
            //* Determine whether focus will move forward or back.*
            info.keyIn = info.wk.key == nckSTAB ? nckSTAB : nckTAB ;

            //* If ESC, then discard any edits by restoring original text.*
            if ( info.wk.key == nckESC )
            {
               cp->tbSetText ( textBackup ) ;
               info.dataMod = false ;
            }
            //* Else, determine whether data have been modified.*
            else
            {
               cp->tbGetText ( textNew ) ;
               info.dataMod = bool(textNew != textBackup) ;
            }
            done = true ;                       // exit the input loop
         }

         //* check for hotkey (CTRL+['A' ... 'Z']) OR *
         //* 'default' hotkey if mouse enabled        *
         else if ( (info.wk.type == wktFUNKEY && 
                    (info.wk.key >= nckC_A && info.wk.key <= nckC_Z)) ||
                   ((this->meMouseEnabled ()) && 
                    (info.wk.mevent.conv != false)) )
         {
            //* Scan the controls for one whose hotkey matches the input.*
            //* If match found, returns control's index, else -1.        *
            //* Ignore hotkey selection of this text box.                *
            short hotIndex = this->IsHotkey ( info.wk ) ;
            if ( hotIndex >= ZERO && hotIndex != this->currCtrl )
            {  //* Make the indicated control the current/active control *
               //* and indicate the results of our edit.                 *
               info.ctrlType = dctTEXTBOX ;     // this is a text box control
               info.ctrlIndex = this->currCtrl ;// control with input focus
               info.hasFocus = true ;           // this may be reset by ChangedFocusViaHotkey()
               info.keyIn = ZERO ;              // indicate no shift in focus necessary
               info.selMember = MAX_DIALOG_CONTROLS ; // don't care
               info.isSel = false ;             // don't care
               info.viaHotkey = false ;         // this may be set by ChangedFocusViaHotkey()
               //* Determine whether data have been modified. *
               cp->tbGetText ( textNew ) ;
               info.dataMod = bool(textNew != textBackup) ;
               done = true ;                    // exit the input loop
               this->ChangedFocusViaHotkey ( info.wk.key, hotIndex, info ) ;
               // on return, new control has the focus, and the info.h_XX
               // variables have been adjusted for that control
            }
            else if ( hotIndex < ZERO )
            {
               //* Key input value is not valid in this context, ignore it. *
               //* If audible alert enabled for invalid key data.           *
               if ( cp->audibleAlert && (info.wk.type != wktMOUSE) )
                  this->UserAlert () ;
            }
         }

         else
         {
            //* Key input value is not valid in this context, ignore it. *
            //* If audible alert enabled for invalid key data.           *
            if ( cp->audibleAlert && (info.wk.type != wktMOUSE) )
               this->UserAlert () ;
         }
      }

      //* If caller has specified a callback method, do it now.*
      if ( ExternalControlUpdate != NULL )
      {
         ExternalControlUpdate ( this->currCtrl, info.wk, false ) ;
         cp->wPtr->SetCursor ( cp->tbCursor ) ; // in case cursor was moved
         cp->wPtr->RefreshWin () ;              // update the display
      }
   }
   while ( ! done ) ;

   nc.SetCursorState ( oldState ) ;          // restore cursor invisibility

}  //* End EditMlText() *

//*************************
//*      EditRjText       *
//*************************
//******************************************************************************
//* User interface for editing right-justified text in a dctTEXTBOX control.   *
//*                                                                            *
//* Input  : cp  : pointer to DialogTextbox object                             *
//*        : info: uiInfo class (by reference) - initial values ignored        *
//*                 (see note in NcDialog.hpp, about values returned)          *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note: Hotkey support for accessing dialog controls is         *
//* limited while inside a text box. Because most key input is valid for the   *
//* text box itself, only control-key combinations nckC_A <= hotkey <= nckC_Z  *
//* are recognized as hotkeys.                                                 *
//*                                                                            *
//* Programmer's Note: For esthetic reasons, the overall design of text box    *
//* editing says that right-justified text should not grow beyond the width of *
//* the field, and this is enforced by the state of the DialogTextbox::hzShift *
//* flag.                                                                      *
//******************************************************************************

void NcDialog::EditRjText ( DialogTextbox* cp, uiInfo& info )
{
   //* Cursor state (visibility) on entry *
   ncCurVis oldState = nc.GetCursorState () ;

   //* Make a backup copy of the existing text data for later comparison.*
   gString textBackup, textNew ;
   cp->tbGetText ( textBackup ) ;

   //* Re-calculate cursor position. It 'should' be ok, but be safe.  *
   cp->tbSynchCursor ( true ) ;
   nc.SetCursorState ( nccvNORMAL ) ;     // make cursor visible

   //**************************
   //* Interact with the user *
   //**************************
   bool  done = false,           // loop control
         rkey = false ;          // 'reserved-key-detected' flag
   do
   {
      #if DEBUG_ET != 0    // For debugging only
      this->ET_Stat ( cp ) ;
      #endif   // DEBUG_ET

      //* 1) Get user input.                                  *
      //* 2) Check for toggle of Insert/Overstrike mode.      *
      //* 3) Check for terminal-resize event.                 *
      //* 4) Check for 'reserved' keys (application specific) *
      //* 5) Access callback method, if specified.            *
      //* 6) If key input handled, return to top of loop.     *
      this->etbGetUserText ( cp, info ) ;
      if ( ((info.wk.type == wktFUNKEY)
            && (info.wk.key == nckINSERT || info.wk.key == nckRESIZE))
           || (rkey = this->resKeys.isReservedKey ( info.wk )) )
      {
         if ( info.wk.type == wktFUNKEY && info.wk.key == nckINSERT )
            this->ToggleIns () ;
         else if ( info.wk.type == wktFUNKEY && info.wk.key == nckRESIZE )
            this->TermResized () ;

         //* Process a keycode reserved by the application *
         //* for Selection/Copy/Cut/Paste operations.      *
         else if ( rkey != false )
         {
            //* Character-by-character selection does not  *
            //* apply to right-justified fields.           *
            if ( (info.wk == this->resKeys.selLeftKey) ||
                 (info.wk == this->resKeys.selRightKey) )
               rkey = false ;

            if ( (rkey != false) )
            {
               //* If WaylandCB is active, process any reserved keycode.*
               if ( ((wcbClip != NULL) && (wcbClip->wcbIsConnected ())) ||
                    //* or if no callback method specified, *
                    (ExternalControlUpdate == NULL) ||
                    //* or if copy/cut operation (not paste)*
                    (info.wk != this->resKeys.pasteKey) )
               {
                  this->etbProcessReservedKeys ( cp, info.wk ) ;
                  rkey = false ;       // reset the reserved-keys flag
               }
            }
         }

         if ( ExternalControlUpdate != NULL )
         {
            ExternalControlUpdate ( this->currCtrl, info.wk, false ) ;
            cp->wPtr->SetCursor ( cp->tbCursor ) ; // in case cursor was moved
            cp->wPtr->RefreshWin () ;              // update the display

            //* If unprocessed reserved keycode, i.e. a Paste operation, *
            //* (with WaylandCB inactive), process it now.               *
            if ( rkey != false )
            {
               this->etbProcessReservedKeys ( cp, info.wk ) ;
               rkey = false ;       // reset the reserved-keys flag
            }
         }
         continue ;
      }

      //* If a selection operation is in progress, but we          *
      //* have a non-selection key, cancel the selection.          *
      //* Exception: If DEL key, allow selected text to be deleted.*
      if ( cp->select.active &&
           ((info.wk.key != nckDELETE) || (info.wk.type != wktFUNKEY)) )
      {
         cp->select.reset() ;
         cp->RedrawControl ( true, true ) ;
      }

      //* Apply text filter to input. *
      if ( info.wk.type == wktPRINT && 
           ((cp->tbApplyFilter ( &info.wk.key )) != false) )
      {
         if ( ((cp->rjInsertChar ( info.wk )) == false) && (cp->audibleAlert) )
            this->UserAlert () ;
      }
      else
      {  //* If:                                                               *
         //* 1) leaving edit via nckTAB, nckSTAB, nckESC, nckENTER, --OR--     *
         //* 2) erasing previous input via nckDELETE, nckBKSP,      --OR--     *
         //* 3) performing 'cut', 'copy', 'paste' (if defined),     --OR--     *
         //* 4) pressing a 'hotkey' (Ctrl+key) for another control, --OR--     *
         //* 5) pressing some other function key (F01 - F12,                   *
         //*    Ctrl+F01 - Ctrl+F12, Alt+F01 - Alt+F12, etc.)                  *
         //* -- then it is not a key to be added to the text.                  *
         //* It is also possible (though unlikely) that an un-handled Escape   *
         //* sequence was captured or that system returned an error.           *

         //* ENTER key is transformed to TAB key.                         *
         if ( info.wk.type == wktFUNKEY && 
              (info.wk.key == nckENTER || info.wk.key == nckpENTER) )
            info.wk.key = nckTAB ;

         //* Backspace or Delete erases a character *
         if ( (info.wk.type == wktFUNKEY) && 
              (info.wk.key == nckBKSP || info.wk.key == nckDELETE) ) 
         {
            //* If 'selected' text, then delete key deletes the selection.  *
            //* Note: For right-justified data, selection is all or nothing.*
            if ( (cp->select.active) && (info.wk.key == nckDELETE) )
            {
               cp->tbClear () ;
               cp->RedrawControl ( true, true ) ;
            }
            else if ( !(cp->rjDeleteChar ()) && (cp->audibleAlert) )
               this->UserAlert () ;
         }

         //* Else if edit complete, return the results.*
         else if ( info.wk.type == wktFUNKEY && 
                   (info.wk.key == nckTAB || info.wk.key == nckSTAB || 
                    info.wk.key == nckUP  || info.wk.key == nckDOWN || 
                    info.wk.key == nckESC) )
         {
            info.ctrlType = dctTEXTBOX ;        // this is a text box control
            info.ctrlIndex = this->currCtrl ;   // control with input focus
            info.hasFocus = true ;              // focus was not lost
            info.selMember = MAX_DIALOG_CONTROLS ; // don't care
            info.isSel = false ;                // don't care
            info.viaHotkey = false ;            // control did not get focus via hotkey
            if (info.wk.key == nckSTAB || info.wk.key == nckUP)
               info.keyIn = nckSTAB ;           // focus to move backward to previous control
            else  // (key==nckTAB || key==nckDOWN || key==nckESC )
               info.keyIn = nckTAB ;            // focus to move forward to next control

            //* If ESC, then discard any edits by restoring original text.*
            if ( info.wk.key == nckESC )
            {
               cp->tbSetText ( textBackup ) ;
               info.dataMod = false ;
            }
            //* Else, determine whether data have been modified.*
            else
            {
               textNew = cp->wText ;
               info.dataMod = bool(textNew != textBackup) ;
            }
            done = true ;                       // exit the input loop
         }

         //* check for hotkey (CTRL+['A' ... 'Z']) OR *
         //* 'default' hotkey if mouse enabled        *
         else if ( (info.wk.type == wktFUNKEY && 
                    (info.wk.key >= nckC_A && info.wk.key <= nckC_Z)) ||
                   ((this->meMouseEnabled ()) && 
                    (info.wk.mevent.conv != false)) )
         {
            //* Scan the controls for one whose hotkey matches the input.*
            //* If match found, returns control's index, else -1.        *
            //* Ignore hotkey selection of this text box.                *
            short hotIndex = this->IsHotkey ( info.wk ) ;
            if ( hotIndex >= ZERO && hotIndex != this->currCtrl )
            {  //* Make the indicated control the current/active control *
               //* and indicate the results of our edit.                 *
               info.ctrlType = dctTEXTBOX ;     // this is a text box control
               info.ctrlIndex = this->currCtrl ;// control with input focus
               info.hasFocus = true ;           // this may be reset by ChangedFocusViaHotkey()
               info.keyIn = ZERO ;              // indicate no shift in focus necessary
               info.selMember = MAX_DIALOG_CONTROLS ; // don't care
               info.isSel = false ;             // don't care
               info.viaHotkey = false ;         // this may be set by ChangedFocusViaHotkey()
               //* Determine whether data have been modified. *
               textNew = cp->wText ;
               info.dataMod = bool(textNew != textBackup) ;
               done = true ;                    // exit the input loop
               this->ChangedFocusViaHotkey ( info.wk.key, hotIndex, info ) ;
               // on return, new control has the focus, and the info.h_XX
               // variables have been adjusted for that control
            }
            else if ( hotIndex < ZERO )
            {
               //* Key input value is not valid in this context, ignore it. *
               //* If audible alert enabled for invalid key data.           *
               if ( cp->audibleAlert && (info.wk.type != wktMOUSE) )
                  this->UserAlert () ;
            }
         }
         else
         {
            //* Key input value is not valid in this context, ignore it. *
            //* If audible alert enabled for invalid key data.           *
            if ( cp->audibleAlert && (info.wk.type != wktMOUSE) )
               this->UserAlert () ;
         }
      }
      if ( ExternalControlUpdate != NULL )
      {
         ExternalControlUpdate ( this->currCtrl, info.wk, false ) ;
         cp->wPtr->SetCursor ( cp->tbCursor ) ; // in case cursor was moved
         cp->wPtr->RefreshWin () ;              // update the display
      }
   }
   while ( ! done ) ;

   nc.SetCursorState ( oldState ) ;          // restore cursor invisibility

}  //* End EditRjText() *

//*************************
//*    etbGetUserText     *
//*************************
//******************************************************************************
//* PRIVATE METHOD: For the textbox editing methods, encapsulate the call      *
//* to GetKeyInput(). Called ONLY from inside the EditXxText() methods.        *
//*                                                                            *
//* Input  : pointer to DialogTextbox object                                   *
//*        : uiInfo class (by reference) - wk member gets key data             *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Note on running in a multithreaded environment:                            *
//* - If the cursor (insertion-point marker) is invisible, we don't care where *
//*   it is until we need to use it. However, during textbox editing, the      *
//*   cursor is VISIBLE, so we need to take extra precautions.                 *
//* - In a single-threaded environment, the visible cursor will remain where   *
//*   it is placed during edit; however, in a multi-threaded application,      *
//*   Forces of Evil (another thread) may--and probably will--move the cursor. *
//* - For this reason, while the cursor is visible, we continually position    *
//*   (re-position) it while we wait for user input.                           *
//* - The alternative would be to leave the actual cursor invisible and to     *
//*   display a cursor-like image at the correct position. We may do this in   *
//*   a future release if it can be done elegantly.                            *
//*                                                                            *
//******************************************************************************

void NcDialog::etbGetUserText ( DialogTextbox* cp, uiInfo& info )
{
   do
   {
      this->SetCursor ( cp->tbCursor ) ;
      cp->RefreshControl () ;
      chrono::duration<short, std::milli>aMoment( 50 ) ;
      this_thread::sleep_for( aMoment ) ;
   }
   while ( (this->KeyPeek ( info.wk )) == wktERR ) ;
   this->GetKeyInput ( info.wk ) ;

}  //* End etbGetUserText() *

//*************************
//*  SetTextboxInputMode  *
//*************************
//******************************************************************************
//* Set DialogTextbox control's editing mode to 'insert' or 'overstrike'.      *
//* 'insert' mode: When user enters a  character, the character under cursor   *
//*                and all characters to the right of it are shifted one       *
//*                position to the right to make room for the new character.   *
//* 'overstrike' mode: When user enters a character, the character under the   *
//*                cursor is replaced with the new character.                  *
//* Note: insert/overstrike flag is ignored for textbox controls which are     *
//*       configured for right-justified input (see enum tbCurPos).            *
//*                                                                            *
//* Input  : overstrike : 'true'  to set overstrike mode,                      *
//*                       'false' to set insert mode                           *
//*          disable    : (optional, 'false' by default)                       *
//*                       if 'true', then the 'Insert' key is disabled to      *
//*                       prevent the user from changing the state of the      *
//*                       insert/overstrike flag.                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void NcDialog::SetTextboxInputMode ( bool overstrike, bool disable )
{

   this->overStrike = overstrike ;
   this->disableIns = disable ;

}  //* End SetTextboxInputMode() *

//*************************
//*     IsOverstrike      *
//*************************
//******************************************************************************
//* Returns the current state of the OverStrike flag.                          *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: current state of overStrike flag                                  *
//******************************************************************************

bool NcDialog::IsOverstrike ( void )
{

   return overStrike ;
   
}  //* End IsOverstrike() *

//*************************
//*       ToggleIns       *
//*************************
//******************************************************************************
//* PRIVATE METHOD.                                                            *
//* Toggle the state of the 'overStrike' flag in response to user pressing     *
//* 'INSERT' key.                                                              *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void NcDialog::ToggleIns ( void )
{

   if ( this->disableIns == false )
      this->overStrike = this->overStrike == false ? true : false ;
 
}  //* End ToggleIns() *

//*************************
//*    SetTextboxText     *
//*************************
//******************************************************************************
//* Replace current textbox data, if any, with specified data.                 *
//* Internally, all data are handled as wchar_t type ('wide') characters, and  *
//* the limits described below refer to wchar_t storage.                       *
//*                                                                            *
//* Text data provided may be longer than control width up to gsALLOCDFLT.     *
//* If source data > gsALLOCDFLT, it will be silently truncated to gsALLOCDFLT.*
//*                                                                            *
//* For single-line textbox controls:                                          *
//* ---------------------------------                                          *
//* - If horizontal shifting is disabled, text will be silently truncated      *
//*   to the width of control.                                                 *
//*                                                                            *
//* For multi-line textbox controls:                                           *
//* --------------------------------                                           *
//* - If horozontal shifting is disabled, text will be silently truncated      *
//*   to the size of the control, which is calculated as:                      *
//*                 (display lines * display columns).                         *
//*                                                                            *
//* Applying the specified text filter:                                        *
//* -----------------------------------                                        *
//* 1) All non-printing control characters (0x01 - 0x1F) will be silently      *
//*    stripped from the display text BEFORE the specified text filter         *
//*    is applied.                                                             *
//* 2) If the text filter encounters an invalid character, the control will    *
//*    contain a NULL string on return.                                        *
//*                                                                            *
//* Input  : cIndex : index of target dctTEXTBOX control                       *
//*          uPtr   : pointer to new display data in UTF-8 format              *
//*               OR                                                           *
//*          wPtr   : pointer to new display data in wchar_t format            *
//*               OR                                                           *
//*          gsData : gString object (by reference) containing the new text    *
//*                                                                            *
//* Returns: number of characters in string including the NULLCHAR             *
//*          OR ERR if data filter detected an invalid character               *
//*              OR specified control currently has input focus                *
//*              OR specified control is not of type dctTEXTBOX                *
//******************************************************************************

short NcDialog::SetTextboxText ( short cIndex, const gString& gsData )
{
short    success = ERR ;      // return value

   if ( cIndex <= this->lastCtrl && dCtrl[cIndex]->type == dctTEXTBOX &&
        cIndex != this->currCtrl )
   {
      DialogTextbox* cp = (DialogTextbox*)dCtrl[cIndex] ; // pointer to control
      success = cp->tbSetText ( gsData ) ;

      //* Update the display to reflect the changes *
      cp->RedrawControl ( (bool)(cIndex == this->currCtrl) ) ;
   }
   return success ;

}  //* End SetTextboxText() *

short NcDialog::SetTextboxText ( short cIndex, const char* uPtr )
{
   short success = ERR ;   // return value
   gString gs( uPtr, gsDFLTBYTES) ;
   success = this->SetTextboxText ( cIndex, gs ) ;
   return success ;

}  //* End SetTextboxText() *

short NcDialog::SetTextboxText ( short cIndex, const wchar_t* wPtr )
{
   short success = ERR ;   // return value
   gString gs( wPtr, gsALLOCDFLT ) ;
   success = this->SetTextboxText ( cIndex, gs ) ;
   return success ;

}  //* End SetTextboxText() *

//*************************
//*    GetTextboxText     *
//*************************
//******************************************************************************
//* Returns a copy of the specified text box's text data.                      *
//* It is the caller's responsibility to provide a buffer large enough to hold *
//* the text data.                                                             *
//*               - gString object, or                                         *
//*               - wchar_t 'wide' characters: gsALLOCDFLT                     *
//*               - UTF-8 (byte encoded) data: gsDFLTBYTES                     *
//*                                                                            *
//* Input  : cIndex : index of target dctTEXTBOX control                       *
//*          uPtr   : pointer to UTF-8 target buffer                           *
//*               OR                                                           *
//*          wPtr   : pointer to wchar_t target buffer                         *
//*               OR                                                           *
//*          gsData : gString object (by reference) to receive text            *
//*                                                                            *
//* Returns: if target is a UTF-8 buffer: returns the number of bytes in       *
//*           string including NULLCHAR                                        *
//*          if target is a wchar_t buffer or gString object: returns the      *
//*           number of characters in the string including NULLCHAR            *
//*          OR ERR if specified control is not of type dctTEXTBOX             *
//******************************************************************************

short NcDialog::GetTextboxText ( short cIndex, gString& gsData )
{
   //* If caller's index number references a Textbox control *
   short charCount = ERR ;
   if ( (cIndex >= ZERO) && (cIndex <= this->lastCtrl) && 
        (this->dCtrl[cIndex])->type == dctTEXTBOX )
   {
      DialogTextbox* cp = (DialogTextbox*)(dCtrl[cIndex]) ; // Pointer to control
      charCount = cp->tbGetText ( gsData ) ;                // retrieve the data
   }
   return charCount ;

}  //* End GetTextboxText() *

short NcDialog::GetTextboxText ( short cIndex, wchar_t wPtr[gsALLOCDFLT] )
{
   gString gs ;
   short charCount = this->GetTextboxText ( cIndex, gs ) ;
   if ( charCount != ERR )
      gs.copy ( wPtr, charCount ) ;
   else
      *wPtr = nckNULLCHAR ;
   return charCount ;

}  //* End GetTextboxText() *

short NcDialog::GetTextboxText ( short cIndex, char* uPtr )
{
   short byteCount = ERR ;
   gString gs ;
   if ( (this->GetTextboxText ( cIndex, gs )) != ERR )
   {
      byteCount = gs.utfbytes() ;
      gs.copy ( uPtr, byteCount ) ;
   }
   else
      *uPtr = nckNULLCHAR ;
   return byteCount ;

}  //* End GetTextboxText() *

//************************
//*  FixedWidthTextbox   *
//************************
//******************************************************************************
//* Set specified dctTEXTBOX control as a fixed-width field.                   *
//* OR                                                                         *
//* Reset specified control from a fixed-width field to an extended-text field.*
//*                                                                            *
//* Specified control:                                                         *
//*  a) must be defined as a single-line control                               *
//*  b) must be defined with >= MIN_TB_SHIFT_WIDTH columns.                    *
//*  c) must be configured for LTR or RTL input, (NOT right-justified input)   *
//*  d) must not currently have the input focus                                *
//*                                                                            *
//* - Extended text is text that may be wider than the display field width and *
//*   and will be horizontally shifted as necessary for visibility during      *
//*   editing.                                                                 *
//*   - When extended text data is enabled, the text may expand during         *
//*     editing to gsALLOCDFLT characters (approx. gsALLOCDFLT*2 columns) even *
//*     though all the text may not be simultaneously visible.                 *
//* - In a fixed-width field, the text may never be wider than the field.      *
//*   That is, the number of text columns may never exceed the number of       *
//*   columns defined for the control.                                         *
//* - All Textbox controls defined with >= MIN_TB_SHIFT_WIDTH columns are      *
//*   extended-text fields by default. Extended text may be disabled or        *
//*   re-enabled for controls that meet the above criteria.                    *
//*   - IMPORTANT NOTE: When a control is set to fixed-width, any existing     *
//*     text is truncated to fit the width of the field.                       *
//* - Textbox controls which are configured for right-justified input are      *
//*   always fixed-width fields.                                               *
//* - Textbox controls defined with < MIN_TB_SHIFT_WIDTH columns, are always   *
//*   fixed-width fields.                                                      *
//*                                                                            *
//* Input  : cIndex : index of target dctTEXTBOX control                       *
//*          enable : if 'true' allow EditTextbox() method to shift data       *
//*                    left/right for text longer than control width.          *
//*                   if 'false' display data cannot be shifted in             *
//*                    window (string width is limited to field width)         *
//*                                                                            *
//* Returns: OK if successful                                                  *
//*          ERR if:                                                           *
//*          1. target control currently has input focus                       *
//*          2. width of target control is less than MIN_TB_SHIFT_WIDTH        *
//*          3. target control is configured for right-justified input         *
//*          4. target control is a multi-line Textbox                         *
//*          5. specified control is not of type dctTEXTBOX                    *
//******************************************************************************
//* Programmer's Note: When disabling horizontal shift, any existing data are  *
//* truncated to the width of the field. The user should not be surprised by   *
//* this, but users are such simple creatures.                                 *
//*                                                                            *
//******************************************************************************

short NcDialog::FixedWidthTextbox ( short cIndex, bool enable )
{
short status = ERR ;

   //* If caller's index number references a Textbox control *
   if ( ((cIndex >= ZERO) && (cIndex <= this->lastCtrl)) && 
        (this->dCtrl[cIndex])->type == dctTEXTBOX && (cIndex != this->currCtrl) )
   {
      DialogTextbox* cp = (DialogTextbox*)(dCtrl[cIndex]) ; // Pointer to control
      if ( (cp->lines == 1) &&
           (cp->cols >= MIN_TB_SHIFT_WIDTH) &&
           (cp->rightJust == false) )
      {
         cp->hzShift = enable ;
         if ( cp->hzShift == false && *cp->wText != NULLCHAR )
         {
            gString tText ;
            this->GetTextboxText ( cIndex, tText ) ;
            tText.limitCols( (cp->cols * cp->lines) ) ;
            this->SetTextboxText ( cIndex, tText ) ;
         }
         status = OK ;
      }
   }   
   return status ;

}  //* End FixedWidthTextbox() *

//*************************
//*   SetTextboxCursor    *
//*************************
//******************************************************************************
//* Configure the way the target control accepts and displays text input       *
//* during editing.                                                            *
//*   OR                                                                       *
//* Set the cursor position over the specified character of the text string.   *
//*                                                                            *
//* Configuring the dctTEXTBOX control for editing style:                      *
//* -----------------------------------------------------                      *
//*  - Configuration style is specified by specifying a member of              *
//*    enum tbCurPos as the 'offset' parameter.                                *
//*    - tbcpHOME  : Set the cursor on the first character of the text data.   *
//*                  - For controls previously configured as LTR               *
//*                    (left-to-right), this will be the top left corner.      *
//*                    Note: LTR is the default editing style.                 *
//*                  - For controls previously configured as RTL               *
//*                    (right-to-left), this will be the top right corner.     *
//*                  - Not applicable if control configured as right-justified *
//*                  - Ignored if control contains no data.                    *
//*    - tbcpAPPEND: Set the cursor to the 'append' position just past the     *
//*                  last character of text data, so that the next character   *
//*                  entered will be appended to the end of the existing text. *
//*                  - Not applicable if control configured as right-justified *
//*                  - Ignored if control contains no data.                    *
//*    - tbcpRIGHTJUST: Configure the target control for right-justified       *
//*                  editing. The cursor is fixed at the right edge of the     *
//*                  field, and as text is entered, existing data are shifted  *
//*                  left to make room for the new character.                  *
//*                  - Note: Right-justified editing is available ONLY for     *
//*                          single-line controls.                             *
//*                  - Note: Right-justified text cannot grow beyond the width *
//*                          of the control. If existing data are wider than   *
//*                          control when this parameter is specified, then    *
//*                          data will be truncated to fit the field.          *
//*                                                                            *
//* Setting a specific cursor position:                                        *
//* ---------------------------------------                                    *
//*  - Set the cursor at the specified offset in the text data.                *
//*  - Note that because text data may contain multi-column characters, the    *
//*    actual cursor position MAY NOT be specified directly. The cursor        *
//*    position specified is interpreted as the character offset into the text *
//*    string and the cursor will be set over that character.                  *
//*  - Does not immediately modify position of the visible cursor. The change  *
//*    becomes visible when control receives input focus.                      *
//*                                                                            *
//*                                                                            *
//* Input  : cIndex : index number of text box control                         *
//*          offset : member of enum tbCurPos (see above)                      *
//*                   -OR- character offset at which to set the cursor         *
//*                        Range ZERO <= offset < number of text characters    *
//*                        in data (not incl NULLCHAR)                         *
//*                                                                            *
//* Returns: OK if successful,                                                 *
//*          ERR if:                                                           *
//*          1. specified position is out of range                             *
//*             - if 'offset' > End-Of-Data, then cursor set at 'append'       *
//*          2. specified member of enum tbCurPos is not applicable (see above)*
//*          3  specified text box currently has input focus                   *
//*          4. specified control is not of type dctTEXTBOX                    *
//******************************************************************************
//* Programmer's Note:                                                         *
//* Under most circumstances, the members of enum tbCurPos should be           *
//* sufficient for cursor positioning, and the cursor position will not        *
//* need to be explicity set because its position is internally tracked        *
//* during user edit of text data. However, this method could be used to       *
//* protect part of a text string from user modification, or to direct         *
//* user's attention to a specific point in the text.                          *
//******************************************************************************

short NcDialog::SetTextboxCursor ( short cIndex, short offset )
{
   short    success = ERR ;      // return value

   if ( cIndex >= ZERO && cIndex <= this->lastCtrl && 
        dCtrl[cIndex]->type == dctTEXTBOX && cIndex != this->currCtrl )
   {
      DialogTextbox* cp = (DialogTextbox*)dCtrl[cIndex] ; // pointer to control
      success = cp->tbSetCursor ( offset ) ;
   }
   return success ;

}  //* End SetTextboxCursor() *

//**************************
//* SetTextboxReservedKeys *
//**************************
//********************************************************************************
//* Specifiy a list of keycodes reserved by the application for use in           *
//* 'selecting' text, and for Copy/Cut/Paste operations to the Textbox Local     *
//* clipboard from within the EditTextbox() method.                              *
//* Clipboard operations will be echoed to the system clipboard if               *
//* communication with the system clipboard has been established.                *
//*                                                                              *
//* The user will be able to select, copy, cut and paste text ONLY if you        *
//* have defined this set of keycodes.                                           *
//*                                                                              *
//* Specifying Keycodes:                                                         *
//* --------------------                                                         *
//* 1) To provide the application designer with maximum flexibility, we have     *
//*    made the design decision not to pre-define Copy/Cut/Paste keycode         *
//*    definitions; however, it is likely that users will want this              *
//*    functionality, so consider defining this set of keycodes during your      *
//*    startup sequence.                                                         *
//*                                                                              *
//* 2) Although any key not dedicated to other uses may be selected as one of    *
//*    the clipboard access keys, typically, an application will use the         *
//*    conventional keycode groups to avoid confusing the user:                  *
//*    a) Universal convention specifies that the CTRL+X (cut), CTRL+C (copy)    *
//*       and CTRL+V (paste) should be used for clipboard access whenever        *
//*       possible. In addition CTRL+A (select All) is conventionally used to    *
//*       select all text in the field.                                          *
//*       -- If these keys are unavaliable or are unacceptable to you, then      *
//*          the ALT+X, ALT+C, ALT+V and ALT+A are the recommended substitutes.  *
//*       -- Note that character-by-character selection keycodes are             *
//*          pre-defined to allow integration with mouse input:                  *
//*          rkeyList.selRightKey = {nckSRIGHT, wktFUNKEY} ;                     *
//*                               SHIFT+RightArrow (select toward the right)     *
//*              and                                                             *
//*          rkeyList.selLeftKey  = {nckSLEFT, wktFUNKEY} ;                      *
//*                               SHIFT+LeftArrow (select toward the left).      *
//*          These two definitions are automatically inserted into the list      *
//*          in case you forget them.                                            *
//*       -- Note on mouse input: The character-by-character selection keys      *
//*          are generated by the SHIFT key + the mouse ScrollWheel up/down.     *
//*                                                                              *
//*    b) For Linux/UNIX console applications, SHIFT+CTRL+C (copy) and           *
//*       SHIFT+CTRL+V (paste) are defined to avoid conflict with shell command  *
//*       codes. Your application will never see these keycodes because they     *
//*       are captured by the terminal emulator, so don't use them in you list.  *
//*       There is no defined console equivalent to the 'Cut' command keycode.   *
//*                                                                              *
//*    c) Keycodes are defined in two parts the key 'type' and the 'keycode'     *
//*       itself. See definition of the wkeyCode class for details.              *
//*       -- If you know the key(s) you want to use, but are not sure of the     *
//*          key's 'type' and 'keycode', then please use the Dialog2 test        *
//*          application, Test 05 which returns the key type and NcDialog key    *
//*          const definition for each keyboard key pressed.                     *
//*                                                                              *
//* 3) It is assumed that the keycode data specified through this method are     *
//*    valid. Keycode validation IS NOT performed here.                          *
//*                                                                              *
//* 4) To release a previously-specified list of reserved keys, call with        *
//*    the following sequence:                                                   *
//*       rkeyList.reset() ;                                                     *
//*       dp->SetTextboxReservedKeys ( rkeyList ) ;                              *
//*                                                                              *
//* 5) Note that if you specify 'hotkeys' for user-interface control access,     *
//*    be sure that they are different from the keycodes specified for           *
//*    clipboard access. Within the EditTextbox() method, clipboard-access       *
//*    keycodes will mask any conflicting hotkeys.                               *
//* --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  --   *
//*                                                                              *
//* Input  : rkeyList : (by reference)                                           *
//*                     an initialized list of keycodes to be reserved           *
//*                     -- to re-enable any previously-reserved keycodes         *
//*                        call with rkeyList.reserve == false                   *
//*                                                                              *
//* Returns: OK if successful, else ERR                                          *
//********************************************************************************
//* Clipboard Access Implementation Notes:                                       *
//* --------------------------------------                                       *
//* 1) If defined these keycodes are handled at the top of the EditTextbox()     *
//*    method and provide application access to the Textbox Local Clipboard.     *
//*    This is a LOCAL, STATIC clipboard buffer for use among dctTEXTBOX         *
//*    objects within an application, but this local clipboard buffer is not     *
//*    visible to other applications or processes.                               *
//*    a) Direct application access to the Textbox Local Clipboard is through    *
//*       following methods:                                                     *
//*          SetLocalClipboard()                                                 *
//*          GetLocalClipboard()                                                 *
//*          GetLocalClipboard_Chars()                                           *
//*          GetLocalClipboard_Bytes()                                           *
//*    b) For proper operation, these methods should ONLY be called from         *
//*       within a callback method established by the application during         *
//*       startup. See EstablishCallback() method for further information.       *
//*                                                                              *
//* 2) The NcDialog class knows NOTHING about the system (X-server) clipboard.   *
//*    a) This was done by design in order to keep the NcDialog class            *
//*       independent of external libraries. The X-server is massively complex   *
//*       and direct application access to the X-server is mostly unnecessary    *
//*       for console applications, so in our view, unnecessarily forcing the    *
//*       application into a dependency on X would be a significant design       *
//*       error.                                                                 *
//*    b) Thus, access to the X-clipboard, if needed, must be handled at the     *
//*       application level.                                                     *
//*    c) The application code must handle any moving of text data between       *
//*       the Textbox Local Clipboard and the X-clipboard.                       *
//*       -- This task is facilitated by the tbXClip class and tbXClip link      *
//*          library bundled with the NcDialog API distribution.                 *
//*       -- Please see the NcDialog API documentation and the tbXClip class     *
//*          definition for details on binding the X-clipboard into your         *
//*          application.                                                        *
//*       -- See also the 'Dialogx' test application for a working model of      *
//*          integrating the tbXClip functionality into your application.        *
//*                                                                              *
//* 3) Most modern GNU/Linux systems are transitioning from the X-windows        *
//*    system to Wayland. For this reason, the NcDialogAPI no longer supports    *
//*    the tbXClip class which was designed as the interface to the X-windows    *
//*    clipboard.                                                                *
//*    a) Access to the Wayland clipboard is through the WaylandCB class in      *
//*       combination with the external "wl-clipboard" utilities by Sergey       *
//*       Bugaev, et. al. This has led to a few changes in the way clipboard     *
//*       access is handled.                                                     *
//*         i) The set of reserved keycodes is now set to default values         *
//*            during NcDialog instantiation.                                    *
//*        ii) The set of reserved keycodes is ALSO set to default values        *
//*            by the WaylandCB-class default constructor.                       *
//*       iii) Custom values for the reserved keycodes may be set using the      *
//*            WaylandCB-class initialization constructor.                       *
//*                                                                              *
//********************************************************************************

short NcDialog::SetTextboxReservedKeys ( reservedKeys& rkeyList )
{
   if ( rkeyList.reserve != false )
   {
      //* Enforce definitions for 'select-leftward' and 'select-rightward' *
      if ( ! (rkeyList.selLeftKey.type == wktFUNKEY && 
              rkeyList.selLeftKey.key == caSELECT_LEFT) )
      {
         rkeyList.selLeftKey.type = wktFUNKEY ;
         rkeyList.selLeftKey.key = caSELECT_LEFT ;
      }
      if ( ! (rkeyList.selRightKey.type == wktFUNKEY && 
              rkeyList.selRightKey.key == caSELECT_RIGHT) )
      {
         rkeyList.selRightKey.type = wktFUNKEY ;
         rkeyList.selRightKey.key = caSELECT_RIGHT ;
      }
      this->resKeys = rkeyList ;       // save a local copy
   }
   else
      this->resKeys.reserve = false ;  // release reserved keys
   return OK ;

}  //* End SetTextboxReservedKeys() *

//**************************
//* SetTextboxReservedKeys *
//**************************
//********************************************************************************
//* Set the default list of keycodes reserved by the application for use in      *
//* 'selecting' text, and for Copy/Cut/Paste operations to the Textbox Local     *
//* clipboard from within the EditTextbox() method.                              *
//* Clipboard operations will be echoed to the system clipboard if               *
//* communication with the system clipboard has been established.                *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void NcDialog::SetTextboxReservedKeys ( void )
{
   //* Set the keycodes to be used for user access   *
   //* to the clipboard for select, copy, cut, paste.*
   wkeyCode copy( caCOPY_SELECTED, wktFUNKEY ),
             cut( caCUT_SELECTED, wktFUNKEY ),
           paste( caPASTE, wktFUNKEY ),
          selAll( caSELECT_ALL, wktFUNKEY ),
         selLeft( caSELECT_LEFT, wktFUNKEY ),
        selRight( caSELECT_RIGHT, wktFUNKEY ) ;
   reservedKeys rk( copy, cut, paste, selAll, selLeft, selRight, true ) ;
   this->SetTextboxReservedKeys ( rk ) ;

}  //* End SetTextboxReservedKeys() *

//**************************
//* SetTextboxReservedKeys *
//**************************
//********************************************************************************
//* For Development and Testing Only                                             *
//* --------------------------------                                             *
//* Returns a copy of the currently-defined "reserved Keycodes" used for         *
//* copy, cut, paste, and text selection within a Textbox control.               *
//* This provides verification that the keycodes initialized during instantiation*
//* or through a call to SetTextboxReservedKeys() have been initialized correctly*
//*                                                                              *
//* Input  : rkeyList : (by reference) receives a copy of current settings       *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void NcDialog::GetTextboxReservedKeys ( reservedKeys& rkeyList )
{
   rkeyList.copyKey     = this->resKeys.copyKey ;     // copy-to-local-clipboard
   rkeyList.cutKey      = this->resKeys.cutKey ;      // cut-to-local-clipboard
   rkeyList.pasteKey    = this->resKeys.pasteKey ;    // paste-from-local-clipboard
   rkeyList.selAllKey   = this->resKeys.selAllKey ;   // select-all-text in Textbox under edit
   rkeyList.selLeftKey  = this->resKeys.selLeftKey ;  // select-from-cursor leftward
   rkeyList.selRightKey = this->resKeys.selRightKey ; // select-from-cursor rightward
   rkeyList.reserve     = this->resKeys.reserve ;     // set/release flag

}  //* End GetTextboxReservedKeys() *

//*************************
//*  DisplayTextboxTail   *
//*************************
//******************************************************************************
//* Re-display the contents of the specified dctTEXTBOX control so that the    *
//* rightmost character is visible. This is useful for display of path/filename*
//* strings or other text where it is advantageous for the user to see the     *
//* the tail of the string. This method does not modify the contents of the    *
//* textbox in any way; it _temporarily_ displays the text with the last       *
//* character at the lower right corner of the field for user convenience.     *
//* - Note that if entire string is already visible in the control,            *
//*   then no action will be taken.                                            *
//*                                                                            *
//* Internally, this method calls 'DisplayTextboxMessage' (see below), but     *
//* does the shift calculation for you.                                        *
//*                                                                            *
//* Input  : cIndex : index of target dctTEXTBOX control                       *
//*                                                                            *
//* Returns: OK if successful                                                  *
//*          ERR if:                                                           *
//*          1. target control currently has input focus                       *
//*          2. target control is configured for right-justified input         *
//*          3. target control is configured with horizontal shift disabled    *
//*          4. specified control is not of type dctTEXTBOX                    *
//******************************************************************************

short NcDialog::DisplayTextboxTail ( short cIndex )
{
   short result = ERR ;
   if ( (cIndex >= ZERO) && (cIndex <= this->lastCtrl) && 
        ((this->dCtrl[cIndex])->type == dctTEXTBOX) && 
        (cIndex != this->currCtrl) )
   {
      //* Get a pointer to the control's private methods and data *
      DialogTextbox* cp = (DialogTextbox*)(dCtrl[cIndex]) ;
      if ( cp->rightJust == false && cp->hzShift != false )
      {
         gString gs ;
         this->GetTextboxText ( cIndex, gs ) ;
         short ctrlCols = (this->dCtrl[cIndex])->lines * (this->dCtrl[cIndex])->cols,
               txtCols  = gs.gscols() ;

         //* If number of text columns > number of control columns, *
         //* i.e. tail is not displayed, then display the tail.     *
         if ( ctrlCols < txtCols )
         {  //* Beginning with the last printing character, count columns *
            //* until we have enough to fill the field. 
            int txtChars ;
            txtCols = ZERO ;
            const short* colPtr = gs.gscols( txtChars ) ;
            short sIndex = txtChars - 2 ;    // reference last printing character
            while ( sIndex >= ZERO )
            {
               if ( (txtCols += colPtr[sIndex]) >= ctrlCols )
               {
                  if ( txtCols > ctrlCols )
                     ++sIndex ;
                  break ;
               }
               --sIndex ;
            }

            dtbmData uData ;
            //* Single-line controls *
            if ( (this->dCtrl[cIndex])->lines == 1 )
               uData = &gs.gstr()[sIndex] ;

            //* Multi-line controls *
            else
            {  //* We must format the string with appropriate hard linebreaks.*
               //* Because the linebreaks may waste up to one column per      *
               //* display line, we need to verify that the formatted data    *
               //* will still fit in the field. If text columns for a line is *
               //* less than columns available for that line, then we have a  *
               //* wasted column. We may do it that way later, but for now,   *
               //* the easy way is to shift out the first cp->lines columns.  *
               gs.shiftCols( -(cp->lines) ) ;
               const wchar_t* wPtr = gs.gstr() ;
               short lineCols = ZERO ;
               for ( short tIndex = ZERO ; sIndex < txtChars ; tIndex++ )
               {
                  lineCols += colPtr[sIndex] ;
                  if ( (uData.textData[tIndex] = wPtr[sIndex++]) == nckNULLCHAR )
                     break ;
                  if ( (lineCols == cp->cols) || 
                       ((lineCols + colPtr[sIndex]) > cp->cols) )
                  {  //* Insert a linebreak *
                     uData.textData[++tIndex] = nckNEWLINE ;
                     lineCols = ZERO ;
                  }
               }
            }
            this->DisplayTextboxMessage ( cIndex, uData ) ;
         }
      }
   }
   return result ;

}  //* End DisplayTextboxTail() *

//*************************
//* DisplayTextboxMessage *
//*************************
//******************************************************************************
//* A Text Box control may also be used as a message display area, as well     *
//* as a text-editing control. Use this method to display a single-color or    *
//* multi-color message in a Text Box. The message is not saved, merely        *
//* written to the display area. Any existing text data in the control         *
//* is unchanged, but is temporarily obscured.                                 *
//*                                                                            *
//* Notes:                                                                     *
//*  dtbmData class for passing data to this method.                           *
//*  See also the description of initialization constructors for the class.    *
//*  a) 'textData' member gets the text to be displayed (UTF-8 format).        *
//*     - The text input filter defined for the target control is ignored      *
//*       by this method.                                                      *
//*     - Because automatic line wrap is not performed by this method, it is   *
//*       recommended that for multi-line target controls. line breaks ('\n')  *
//*       be used to pre-format the data.                                      *
//*              (Line breaks are ignored for single-line controls.)           *
//*     - To erase the message and redisplay the contents (if any) of the      *
//*       control object, call this method with 'textData' set to an empty     *
//*       string ( "" ).                                                       *
//*  b) 'colorData' member specifies the color attribute(s) of the text        *
//*     - If colorData[0] == dtbmNFcolor, (default), then text will be         *
//*       displayed in the non-focus color defined for the text box.           *
//*     - If colorData[0] == dtbmFcolor, then text will be displayed in        *
//*       the focus color defined for the text box.                            *
//*     - Otherwise an array of color attributes is expected, and a bit of     *
//*       experimentation may be needed to achieve a beautious result.         *
//*       - Each character in textData (including any newline characters)      *
//*         should have a corresponding color attribute.                       *
//*       - Note that if the text is to be 'centered', AND if colorData[0]     *
//*         DOES NOT contain one of the special attributes described above,    *
//*         then the 'colorData' array must contain at least as many           *
//*         attributes as the number of character columns defined for the      *
//*         control (plus newlines). In general, this is:                      *
//*                 attributeCount = lines * columns + lines ;                 *
//*  c) 'refreshControl' member ('true' by default) indicates whether the      *
//*     display should be refreshed immediately.                               *
//*  d) 'centered' member ('false' by default).                                *
//*     - If 'false', the text is written as provided.                         *
//*     - If 'true', then each line of text will be horizontally centered      *
//*                  in the field.                                             *
//*  e) 'rtlText' member ('false' by default)                                  *
//*     - If 'false', the text is processed as LTR (left-to-right)             *
//*     - If 'true', the text processed text as an RTL (right-to-left) language*
//*                  (Arabic, Hebrew, Persian, Urdu, etc.)                     *
//*                                                                            *
//* Input  : cIndex : index of target dctTEXTBOX control                       *
//*          uData  : initialized dtbmData class object                        *
//*                                                                            *
//* Returns: OK if successful                                                  *
//*          ERR if:                                                           *
//*          1. message too long for control (message displayed-but-truncated) *
//*          2. target control currently has input focus                       *
//*          3. specified control is not of type dctTEXTBOX                    *
//******************************************************************************

short NcDialog::DisplayTextboxMessage ( short cIndex, dtbmData& uData )
{
short    result = ERR ;

   //* If caller's index number references a Textbox control, and *
   //* the control DOES NOT currently have the input focus.       *
   if ( cIndex >= ZERO && cIndex <= this->lastCtrl && 
        (this->dCtrl[cIndex])->type == dctTEXTBOX && cIndex != this->currCtrl )
   {
      //* Get a pointer to the control's private methods and data *
      DialogTextbox* cp = (DialogTextbox*)(dCtrl[cIndex]) ;

      //* If we have data to display *
      if ( *uData.GetTextPtr() != NULLCHAR )
      {
         //* If specified, horizontally center the message in the field.*
         if ( uData.centered != false )
         {
            gString gsa( uData.textData ),      // full text
                    gsb,                        // substring text
                    gsc,                        // padded substring text
                    gsd ;                       // concatenated substrings
            const wchar_t* wPtr = gsa.gstr() ;  // access to full text
            wchar_t wBuff[gsALLOCDFLT] ;        // temp work buffer
            wchar_t pBuff[gsALLOCDFLT] ;        // temp padding buffer
            short boxCols = cp->cols,           // width of control
                  wIndex = ZERO,                // wPtr index
                  wBase,                        // wPtr base index
                  sIndex,                       // substring index
                  padLen ;                      // width of padding string

            for ( short lineCount = ZERO ; 
                  (lineCount < cp->lines) && (wPtr[wIndex] != nckNULLCHAR) ; 
                  lineCount++ )
            {
               //* Isolate substring for this line *
               wBase = wIndex ;
               while ( wPtr[wIndex] != nckNEWLINE && wPtr[wIndex] != nckNULLCHAR )
                  ++wIndex ;
               for ( sIndex = ZERO ; wBase < wIndex ; sIndex++, wBase++ )
                  wBuff[sIndex] = wPtr[wBase] ;
               wBuff[sIndex] = nckNULLCHAR ;

               //* Create padding string *
               gsb = wBuff ;
               padLen = (boxCols - gsb.gscols()) / 2 ;
               if ( padLen > ZERO )
               {
                  short i ;
                  for ( i = ZERO ; i < padLen ; i++ )
                     pBuff[i] = nckSPACE ;
                  pBuff[i] = nckNULLCHAR ;
                  //* Pad both ends of caller's text *
                  gsc.compose( L"%S%S%S ", pBuff, wBuff, pBuff ) ;
               }
               else     // no padding needed for this line
                  gsc = gsb ;

               //* Discard excess padding and append data to accumulator *
               gsc.limitCols( boxCols ) ;
               if ( wPtr[wIndex] == nckNEWLINE )
               {
                  gsc.append( L"\n" ) ;
                  ++wIndex ;
               }
               gsd.append( gsc.gstr() ) ;
            }
            //* Copy formatted data to caller's object *
            gsd.copy( uData.textData, gsALLOCDFLT ) ;
         }
         result = cp->tbDisplayMessage ( uData ) ;
      }

      //* Empty string. Redraw the original text in non-focus color *
      else
      {
         cp->RedrawControl ( false ) ;
         result = OK ;
      }
   }
   return result ;

}  //* End DisplayTextboxMessage() *

//************************
//*     VerifyTbText     *
//************************
//********************************************************************************
//* Verify that a text string meets the specified filtering criteria             *
//* for strings created/used in EditText().                                      *
//* Use before pre-loading a text box with string data or to apply a             *
//* secondary filter to edited data. (target text box is not modified.)          *
//*                                                                              *
//* Input  : cIndex : index of target dctTEXTBOX control                         *
//*          uStr  : text data to be verified in UTF-8 format                    *
//*                  (Recommend that uStr be at least gsDFLTBYTES in size)       *
//*              OR                                                              *
//*          wStr  : text data to be verified in wchar_t format                  *
//*                  (Recommend that wStr be at least gsALLOCDFLT in size)       *
//*              OR                                                              *
//*          gStr  : text data to be verified contained in a gString object      *
//*          filter: member of enum tbChars, indicates which filter to apply     *
//*                                                                              *
//* Returns: OK  if all characters match filter criteria                         *
//*              Note: Some filters such as upper/lower case will adjust         *
//*                    the provided source data to match filter criteria.        *
//*          ERR if data filter detected an invalid character                    *
//*                (original data unchanged)                                     *
//*              OR specified control is not of type dctTEXTBOX                  *
//********************************************************************************

short NcDialog::VerifyTbText ( short cIndex, gString& gStr, tbChars filter )
{
   short goodData = false ;

   //* If caller's index number references a Textbox control *
   if ( ((cIndex >= ZERO) && (cIndex <= this->lastCtrl)) && 
        (this->dCtrl[cIndex])->type == dctTEXTBOX )
   {
      DialogTextbox* cp = (DialogTextbox*)(dCtrl[cIndex]) ; // Pointer to control
      tbChars savedFilter = cp->filter ;  // temporarily substitute specified filter
      cp->filter = filter ;

      //* Apply the text filter for this control object *
      goodData = ((cp->tbApplyFilter ( gStr )) == false) ? ERR : OK ;

      cp->filter = savedFilter ;    // restore control's original filter code
   }
   return goodData ;

}  //* End VerifyTbText() *

short NcDialog::VerifyTbText ( short cIndex, char* uStr, tbChars filter )
{
   gString gs( uStr ) ;             // convert from UTF-8 data
   short status ;
   if ( (status = this->VerifyTbText ( cIndex, gs, filter )) != false )
      gs.copy( uStr, (gsDFLTBYTES) ) ;
   return status ;
}  //* End VerifyTbText() *

short NcDialog::VerifyTbText ( short cIndex, wchar_t* wStr, tbChars filter )
{
   gString gs( wStr ) ;             // convert from wchar_t data
   short status ;
   if ( (status = this->VerifyTbText ( cIndex, gs, filter )) != false )
      gs.copy( wStr, gsALLOCDFLT ) ;
   return status ;
}  //* End VerifyTbText() *

//*************************
//*     TextboxAlert      *
//*************************
//********************************************************************************
//* Enable or disable audible alert (beep) for invalid characters received       *
//* during edit of Textbox data. An invalid character in this context is any     *
//* keycode that is not recognized OR that has been filtered out by the          *
//* active input filter associated with the control.                             *
//*                                                                              *
//* Audible alert is initially disabled, and may be enabled after instantiation  *
//* of the control. The alert mechanism is a call to the UserAlert() method      *
//* (with no parameters) as described elsewhere.                                 *
//*                                                                              *
//* Input  : cIndex : index of target dctTEXTBOX control                         *
//*          enable : 'true'  to enable audible alerts                           *
//*                   'false' to disable audible alerts                          *
//*                                                                              *
//* Returns: OK if successful                                                    *
//*          ERR if specified control is not of type dctTEXTBOX                  *
//********************************************************************************

short NcDialog::TextboxAlert ( short cIndex, bool enable )
{
   short status = ERR ;

   //* If caller's index number references a Textbox control.*
   if ( cIndex >= ZERO && cIndex <= this->lastCtrl && 
        (this->dCtrl[cIndex])->type == dctTEXTBOX )
   {
      //* Get a pointer to the control's private methods and data *
      DialogTextbox* cp = (DialogTextbox*)(dCtrl[cIndex]) ;
      cp->audibleAlert = enable ;
      status = OK ;
   }
   return status ;

}  //* End TextboxAlert() *

//*****************************
//*  EscapeLinuxSpecialChars  *
//*****************************
//******************************************************************************
//* For string data which contain special Linux reserved characters which      *
//* are interpreted by the shell program as shell instructions, place the      *
//* escape character (backslash '\') before each special character so the      *
//* shell will interpret the character as plain text. The Linux special        *
//* characters are:  \   '   "   ?   :   ;   &   >   <   |   *                 *
//*                                                                            *
//* Used primarily when passing filenames or path specifications to the shell  *
//* or to invoke a shell program, ensuring that the shell does not             *
//* misinterpret the received command.                                         *
//*                                                                            *
//* Any special characters which have _already_ been escaped will be ignored.  *
//*                                                                            *
//* Input  : pathSpec: (by reference) data to be scanned                       *
//*                      (usually a filename or path specification)            *
//*                                                                            *
//* Returns: number of characters 'escaped'                                    *
//******************************************************************************
//* Special Case Testing for '\' as the special character:                     *
//* 1) If the incomming data contains a '\', we must test whether:             *
//*    a) The '\' itself has already been escaped,                             *
//*    b) The '\' is escaping some other special character,                    *
//*    c) The '\' is a special character which needs to be escaped.            *
//* 2) If data contains an escaped special character, but caller intended      *
//*    for it to be two SEPERATE special characters, caller will be            *
//*    disappointed that we could not read his/her/its mind. (sorry about that)*
//******************************************************************************

short NcDialog::EscapeLinuxSpecialChars ( gString& pathSpec )
{
   const wchar_t bSLASH = L'\\' ;
   short indx,                      // text index
         specChars = ZERO ;         // return value

   for ( short specIndx = ZERO ; specIndx < linuxSpecialCount ; ++specIndx )
   {
      indx = ZERO ;
      do
      {
         if ( (indx = pathSpec.find( linuxSpecialChars[specIndx], indx )) >= ZERO )
         {
            //* Special character == '\'. (see note above) *
            if ( (pathSpec.gstr()[indx]) == bSLASH )
            {
               bool insert = true ;
               //* If escape character is itself escaped *
               if ( (indx > ZERO) && ((pathSpec.gstr()[indx - 1]) == bSLASH) )
               { ++indx ; insert = false ; }
               //* If '\' is being used as an escape character *
               for ( short i = ZERO ; i < linuxSpecialCount && insert ; ++i )
               {
                  if ( (pathSpec.gstr()[indx + 1]) == linuxSpecialChars[i] )
                  { indx += 2 ; insert = false ; }
               }
               if ( insert )  // escape the escape character
               {
                  pathSpec.insert( bSLASH, indx ) ;
                  ++specChars ;
                  indx += 2 ;
               }
            }
            else
            {
               if ( indx == ZERO )       // first character in array
               {
                  pathSpec.insert( bSLASH, indx ) ;
                  ++specChars ;
                  indx += 2 ;
               }
               else // (indx > ZERO), second or later character in array
               {
                  if ( (pathSpec.gstr()[indx - 1]) != bSLASH )
                  {
                     pathSpec.insert( bSLASH, indx ) ;
                     ++specChars ;
                     indx += 2 ;
                  }
                  else
                     ++indx ;
               }
            }
         }
      }
      while ( indx >= ZERO ) ;
   }
   return specChars ;

}  //* End EscapeLinuxSpecialChars() *



//******************************************************************************
//* Passthrough methods to WaylandCB class implementation.                     *
//* See WaylandCB.hpp for a detailed description of each called method.        *
//******************************************************************************

//* WaylandCB Destructor: Delete temp files and delete WaylandCB object.*
//* Returns: 'true' if successful, else 'false' (no class object)       *
bool NcDialog::wcbDisable ( void )
{
   bool status = false ;
   if ( wcbClip != NULL )
   {
      delete wcbClip ;
      wcbClip = NULL ;
      status = true ;
   }
   return status ;

}  //* End wcbDisable() *

//* Default constructor (default reserve-key map used)      *
//* If enable called when object has already been           *
//* instantiated, simply returns current status.            *
//* Returns: 'true' if connection established, else 'false' *
bool NcDialog::wcbEnable ( void )
{
   bool status = false ;
   if ( wcbClip == NULL )
   {  //* Create the clipboard interface object.*
      if ( (wcbClip = new WaylandCB ()) != NULL )
      {
         status = wcbClip->wcbIsConnected () ; // test the connection
         this->SetTextboxReservedKeys () ;     // set default clipboard-access keys
      }
   }
   //* Else, object previously instantiated, *
   //* so just return connection status.     *
   else
      status = this->wcbIsConnected() ;
   return status ;

}  //* End wcbEnable() *

//* Constructor with caller-supplied reserve-key definitions. *
//* If connection already established, then only key          *
//* definitions are updated.                                  *
//* Returns: 'true' if connection established, else 'false'   *
bool NcDialog::wcbEnable ( reservedKeys& rk )
{
   bool status = false ;

   if ( wcbClip == NULL )
   {  //* Create the clipboard interface object.*
      if ( (wcbClip = new WaylandCB ()) != NULL )
      {
         status = wcbClip->wcbIsConnected () ;  // test the connection
         this->SetTextboxReservedKeys ( rk ) ;  // set clipboard-access keys
      }
   }
   //* Clipboard connection already established *
   else
   {
      status = wcbClip->wcbIsConnected () ;  // test the connection
      this->SetTextboxReservedKeys ( rk ) ;  // set clipboard-access keys
   }
   return status ;

}  //* End wcbEnable() *

//* Get the contents of the clipboard.                                         *
//* Returns: for wchar_t target, number of characters read (incl.NULLCHAR)     *
//*          for char target or gString target, number of bytes read           *
//*           (incl. NULLCHAR)                                                 *
//*          returns wcbsNOINIT if WaylandCB not instantiated                  *
//*          returns wcbsNOCONNECT if error i.e. not connected to clipboard    *
//*          returns wcbsNOINSTALL if 'wl-paste' utility not installed         *
short NcDialog::wcbGet ( char* trg, bool primary, short len )
{
   short byteCount = wcbsNOINIT ;
   if ( wcbClip != NULL )
      byteCount = wcbClip->wcbGet ( trg, primary, len ) ;
   return byteCount ;

}  //* End wcbGet() *

short NcDialog::wcbGet ( gString& trg, bool primary, short len )
{
   short byteCount = wcbsNOINIT ;
   if ( wcbClip != NULL )
      byteCount = wcbClip->wcbGet ( trg, primary, len ) ;
   return byteCount ;

}  //* End wcbGet() *

short NcDialog::wcbGet ( wchar_t* trg, bool primary, short len )
{
   short charCount = wcbsNOINIT ;
   if ( wcbClip != NULL )
      charCount = wcbClip->wcbGet ( trg, primary, len ) ;
   return charCount ;

}  //* End wcbGet() *

//* Set the contents of the clipboard.                                           *
//* Returns: for char source, or gString source, number of bytes written         *
//*             (incl. NULLCHAR)                                                 *
//*          for wchar_t source, number of characters written (incl.NULLCHAR)    *
//*          returns wcbsNOINIT if WaylandCB not instantiated                    *
//*          returns wcbsNOCONNECT if error i.e. not connected to clipboard      *
//*          returns wcbsNOINSTALL if 'wl-copy' utility not installed            *
short NcDialog::wcbSet ( const char* csrc, bool primary, short cnt )
{
   short byteCount = wcbsNOINIT ;
   if ( wcbClip != NULL )
      byteCount = wcbClip->wcbSet ( csrc, primary, cnt ) ;
   return byteCount ;

}  //* End wcbSet() *

short NcDialog::wcbSet ( const gString& csrc, bool primary, short cnt )
{
   short byteCount = wcbsNOINIT ;
   if ( wcbClip != NULL )
      byteCount = wcbClip->wcbSet ( csrc, primary, cnt ) ;
   return byteCount ;

}  //* End wcbSet() *

short NcDialog::wcbSet ( const wchar_t* csrc, bool primary, short cnt )
{
   short charCount = wcbsNOINIT ;
   if ( wcbClip != NULL )
      charCount = wcbClip->wcbSet ( csrc, primary, cnt ) ;
   return charCount ;

}  //* End wcbSet() *

//* Clear the clipboard. (note that 2nd optional parameter "useCmd" is ignored)*
//* Returns: 'true' if successful                                              *
//*          'false' if not connected, or other error                          *
bool NcDialog::wcbClear ( bool primary )
{
   bool status = false ;
   if ( wcbClip != NULL )
      status = wcbClip->wcbClear ( primary ) ;
   return status ;

}  //* End wcbClear() *

//* Report number of data bytes on clipboard.                                  *
//* Returns: bytes of data on the clipboard incl. NULLCHAR                     *
//*          returns wcbsNOINIT if WaylandCB not instantiated                  *
//*          returns wcbsNOCONNECT if error i.e. not connected to clipboard    *
//*          returns wcbsNOINSTALL if 'wl-copy' utility not installed          *
int NcDialog::wcbBytes ( bool primary )
{
   int cbBytes = wcbsNOINIT ;
   if ( wcbClip != NULL )
      cbBytes = wcbClip->wcbBytes ( primary ) ;
   return cbBytes ;

}  //* End wcbBytes() *

//* Report available data formats.                                             *
//* Returns: for wchar_t target, number of characters read (incl.NULLCHAR)     *
//*          for char target or gString target, number of bytes read           *
//*           (incl. NULLCHAR)                                                 *
//*          returns wcbsNOINIT if WaylandCB not instantiated                  *
//*          returns wcbsNOCONNECT if error i.e. not connected to clipboard    *
//*          returns wcbsNOINSTALL if 'wl-copy' utility not installed          *
short NcDialog::wcbTypes ( gString& trg, short cnt, bool primary )
{
   short cbBytes = wcbsNOINIT ;
   if ( wcbClip != NULL )
      cbBytes = wcbClip->wcbTypes ( trg, cnt, primary ) ;
   return cbBytes ;

}  //* End wcbTypes() *

short NcDialog::wcbTypes ( char* trg, short cnt, bool primary )
{
   short cbBytes = wcbsNOINIT ;
   if ( wcbClip != NULL )
      cbBytes = wcbClip->wcbTypes ( trg, cnt, primary ) ;
   return cbBytes ;

}  //* End wcbTypes() *

short NcDialog::wcbTypes ( wchar_t* trg, short cnt, bool primary )
{
   short cbChars = wcbsNOINIT ;
   if ( wcbClip != NULL )
      cbChars = wcbClip->wcbTypes ( trg, cnt, primary ) ;
   return cbChars ;

}  //* End wcbTypes() *

//* Report whether connection established.                                     *
//* Returns: 'true' if connected to clipboard, else 'false'                    *
bool NcDialog::wcbIsConnected ( void )
{
   bool status = false ;
   if ( wcbClip != NULL )
      status = wcbClip->wcbIsConnected () ;
   return status ;

}  //* End wcbIsConnected() *

#if 0    // EXPERIMENTAL
//* For Debugging Only: Report whether the WaylandCB-class *
//* object has been instantiated.                          *
bool wcbIsInstantiated ( void ) ;
//* For Debugging Only:                                                        *
//* Report whether the WaylandCB-class object has been assigned to wcbClip.    *
bool NcDialog::wcbIsInstantiated ( void )
{
   bool status = false ;
   if ( wcbClip != NULL )
      status = true ;
   return status ;
}  //* End wcbIsInstantiated() *
#endif   // EXPERIMENTAL

//* Test the connection.                                                       *
//* Returns: Member of enum wcbStatus:                                         *
wcbStatus NcDialog::wcbTest ( const char* testdata )
{
   wcbStatus status = wcbsNOINIT ;
   if ( wcbClip != NULL )
      status = wcbClip->wcbTest ( testdata ) ;
   return status ;

}  //* End wcbTest() *

//* Delete local data and reconnect with Wayland clipboard.                    *
//* Returns: 'true' if connection re-established, else 'false'                 *
bool NcDialog::wcbReinit ( void )
{
   bool status = false ;
   if ( wcbClip != NULL )
      status = wcbClip->wcbReinit () ;
   return status ;

}  //* End wcbReinit() *

//* Report WaylandCB class version.                                            *
//* Returns: pointer to const string of the format: "x.y.zz"                   *
const char* NcDialog::wcbVersion ( void )
{
   const char* ptr = NULL ;
   if ( wcbClip != NULL )
      ptr = wcbClip->wcbVersion () ;
   return ( ptr ) ;

}  //* End wcbVersion() *

//* Report both WaylandCB version and wl-clipboard version.                    *
//* This is a pass-through to the wcbVersion(gString&) method.                 *
//* The two version-number strings are returned in a gString object, and are   *
//* separated by a newline character ('\n') as shown:                          *
//*    "x.y.zz\na.b.c"                                                         *
//* where "x.y.zz" is the WaylandCB version, and "a.b.c" is the reported       *
//* wl-copy/wl-paste version. Example:  "0.0.04\n2.2.1"                        *
//*                                                                            *
//* Input  : gsVersion : (by reference) receives concatenated version strings  *
//*                                                                            *
//* Returns: ZERO if successful, or                                            *
//*             wcbsNOCONNECT if error i.e. not connected to clipboard         *
//*             wcbsNOINSTALL if 'wl-copy' utility not installed               *
short NcDialog::wcbVersion ( gString& gsVersion )
{
   short status = wcbsNOINSTALL ;
   if ( wcbClip != NULL )
      status = wcbClip->wcbVersion ( gsVersion ) ;
   return status ;

}  //* End wcbVersion() *

//*************************
//*     wcbUserAlert      *
//*************************
//********************************************************************************
//* Information dialog reporting status of clipboard connection.                 *
//*                                                                              *
//* Default Layout:                                                              *
//* Position: The dialog is centered in the parent dialog window.                *
//* Size (x): The dialog is the width of the widest text string (+4 columns),    *
//*           but not less than 32 columns and not wider than parent dialog.     *
//* Size (y): The dialog is the height sufficient to display all text strings    *
//*           (+3 rows), but not less than 4 rows and not greater than parent    *
//*           dialog.                                                            *
//* Color   : By default, color attribute of the dialog is the same as interior  *
//*           of parent dialog; however, the attribute may be specified directly.*
//* Text    : Default text depends on the 'warnLevel' argument.                  *
//*           Note that if custom text is specified, then the connectivity       *
//*           test is not performed, because it is assumed that caller has       *
//*           already performed a call to wcbTest() (or another form of          *
//*           diagnostic) and has set the display text accordingly.              *
//*                                                                              *
//* Input  : warnLevel : indicates the type of warning to be displayed           *
//*                      wcbsACTIVE:    always report status of connection test  *
//*                      wcbsNOINSTALL: always report if not connected           *
//*                      wcbsNOCONNECT: report if 'wl-clipboard' utilities       *
//*                              installed, but unable to establish connection   *
//*          gsptr     : (optional, NULL pointer by default)                     *
//*                      If specified, pointer to a gString object containing    *
//*                      the text to be displayed, with display lines separated  *
//*                      by NEWLINE characters.                                  *
//*                      The first line is assumed to be the dialog title.       *
//*          color     : (optional, 0 by default)                                *
//*                      If specified, the dialog border and interior color.     *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* Programmer's Notes;                                                          *
//* 1) If caller does something stupid like sending us an empty string,          *
//*    he/she/it will get an empty dialog. (It's only fair :-)                   *
//* 2) If caller sends only enough text for a title, then there will be no       *
//*    body text.                                                                *
//* 3) If height and/or width of source text exceeds the dimensions of the       *
//*    parent dialog, data will be truncated.                                    *
//*                                                                              *
//*                                                                              *
//********************************************************************************

void NcDialog::wcbUserAlert ( wcbStatus warnLevel, const gString* gsptr, attr_t color )
{
   const char* const uaTestData = "NcDialogAPI system clipboard test." ;
   const short activeMsgCnt = 4 ;
   const char* activeMsg[activeMsgCnt + 1] =
   { //123456789-123456789-123456789-123456789-1 (min 46 columns)
      " System Clipboard  ",
      " Connection between application and",
      " system clipboard established and active.",
      " ",
      NULL
   } ;
   const short noconnMsgCnt = 4 ;
   const char* noconnMsg[noconnMsgCnt + 1] =
   { //123456789-123456789-123456789-123456789 (min 44 columns)
      "  System Clipboard  ",
      " Unable to establish connection with",
      " system clipboard: communication error.",
      " ",
      NULL
   } ;
   const short noinstMsgCnt = 5 ;
   const char* noinstMsg[noinstMsgCnt + 1] =
   { //123456789-123456789-123456789-123456789-12 (min 46 columns)
      "  System Clipboard  ",
      "Unable to establish connection with system",
      "clipboard: Necessary external utilities",
      "   \"wl-copy\" and \"wl-paste\" not found.",
      " ",
      NULL
   } ;
   const short noinitMsgCnt = 4 ;
   const char* noinitMsg[noinitMsgCnt + 1] = 
   { //123456789-123456789-123456789-123456789-1 (min 45 columns)
      "  System Clipboard  ",
      "Application interface to system clipboard",
      "has not yet been established.            ",
      " ",
      NULL
   } ;


   genDialog gd ;                   // dialog definition
   char* msgArray = NULL ;          // converted row data
   const char** ptrArray = NULL ;   // pointers to line data
   short  dlgRows = 6,              // dialog rows
          dlgCols = 46 ;            // dialog column
   attr_t bColor = (color == attr_t(0)) ? this->iColor : color ; // dialog color
   bool report = false ;

   //* If caller did not provide message text, then perform connection test.*
   if ( gsptr == NULL )
   {
      //* Test the connection *
      wcbStatus wcbstatus = this->wcbTest ( uaTestData ) ;

      //* If connection error OR if caller specified forced report *
      if ( wcbstatus != wcbsACTIVE || warnLevel == wcbsACTIVE )
      {
         if ( wcbstatus == wcbsACTIVE )
            gd.msgList = activeMsg ;
         else if ( wcbstatus == wcbsNOCONNECT )
         { gd.msgList = noconnMsg ; dlgCols -= 2 ; }
         else if ( wcbstatus == wcbsNOINSTALL )
         { gd.msgList = noinstMsg ; ++dlgRows ; }
         else  // (wcbstatus == wcbsNOINIT)
         { gd.msgList = noinitMsg ;/* ++dlgRows ;*/ }

         gd.dColor  = bColor ;
         gd.pfColor = (gd.dColor != nc.reR && gd.dColor != nc.reS) ? nc.reR : nc.grbk ;
         gd.dLines  = dlgRows ;
         gd.dCols   = dlgCols ;
         report = true ;
      }
   }

   //* Else, assume that caller knows what he/she is doing (always dangerous).*
   else
   {
      gString gs ;                              // text analysis
      dlgRows = 3 ;                             // initial sub-dialog rows
      dlgCols = 32 ;                            // initial sub-dialog columns
      int lineCnt = ZERO,                       // data rows (incl. title)
          wIndx = ZERO,                         // 'wbuff' index
          wChars ;                              // number of source characters
      const wchar_t* wptr = gsptr->gstr( wChars ) ; // pointer to source data
      const int MAX_DROWS  = this->dLines - 2 ; // max sub-dialog rows
      const int MAX_DCOLS  = this->dCols - 2 ;  // max sub-dialog columns
      const int MAX_RCOLS  = MAX_DCOLS - 4 ;    // max data columns per row
      const int ITEM_BYTES = this->dCols ;      // max data bytes per row
      int msgIndx = ZERO,                       // 'magArray' index
          ptrIndx = ZERO ;                      // 'ptrArray' index

      //* Allocate text and pointer arrays for display text.*
      msgArray = new char[ITEM_BYTES * this->dLines] ;
      ptrArray = new const char*[this->dLines] ;
      wchar_t wbuff[gsALLOCMIN] ;

      //* Parse the raw data into rows.*
      for ( short w = ZERO ; w < wChars ; ++w )
      {
         if ( (wptr[w] != nckNEWLINE) && (wptr[w] != nckNULLCHAR) )
            wbuff[wIndx++] = wptr[w] ;
         else
         {
            wbuff[wIndx] = wbuff[wIndx+1] = NULLCHAR ; // terminate the substring
            wIndx = ZERO ;                // reset the index
            gs = wbuff ;                  // copy to char target
            if ( (gs.gscols()) > MAX_RCOLS ) // truncate to max width
               gs.limitCols( MAX_RCOLS ) ;
            gs.copy( &msgArray[msgIndx], ITEM_BYTES ) ;
            ptrArray[ptrIndx++] = &msgArray[msgIndx] ; // point to head of line data
            msgIndx += gs.utfbytes() ;    // step over the saved data
            if ( ((gs.gscols()) + 4) > dlgCols ) // track widest display string
               dlgCols = (gs.gscols()) + 4 ;
            ++lineCnt ;
            if ( (wptr[w] != NULLCHAR) && (wptr[w + 1] == NULLCHAR) )
               ++w ;    // null terminator found, don't include empty row
         }
      }
      ptrArray[ptrIndx] = NULL ;          // indicate end-of-array
      dlgRows += lineCnt ;                // set sub-dialog height

      if ( dlgRows > MAX_DROWS )          // range testing
      { dlgRows = MAX_DROWS ; ptrArray[dlgRows - 2] = NULL ; }
      if ( dlgCols > MAX_DCOLS )
         dlgCols = MAX_DCOLS ;

      gd.msgList = ptrArray ;             // point to array of pointers
      gd.dLines  = dlgRows ;              // dialog rows
      gd.dCols   = dlgCols ;              // dialog columns
      gd.dColor  = bColor ;
      gd.pfColor = (gd.dColor != nc.reR && gd.dColor != nc.reS) ? nc.reR : nc.grbk ;
      report = true ;
   }

   //* If connection error OR if caller specified forced report *
   if ( report != false )
   {
      //* Save parent-dialog display.*
      this->SetDialogObscured () ;

      //* Report the results.*
      this->InfoDialog ( gd ) ;

      //* Restore the parent dialog.*
      this->RefreshWin () ;
   }

   //* Release dynamically-allocated data *
   if ( msgArray != NULL ) { delete [] msgArray ; }
   if ( ptrArray != NULL ) { delete [] ptrArray ; }

}  //* End wcbUserAlert() *

//*************************
//*  wcbLocalCb2SystemCb  *
//*************************
//********************************************************************************
//* Private Method                                                               *
//* --------------                                                               *
//* If the Wayland Clipboard interface is active, copy the contents of the       *
//* local clipboard to the system clipboard.                                     *
//*                                                                              *
//*                                                                              *
//* Input  : none (data to be copied is on the local clipboard)                  *
//*                                                                              *
//* Returns: 'true' if data copied to system clipboard, else 'false'             *
//********************************************************************************

bool NcDialog::wcbLocalCb2SystemCb ( void )
{
   bool status = false ;

   if ( (wcbClip != NULL) && (wcbClip->wcbIsConnected ()) )
   {
      gString gsClip ;

      //* Get contents of local clipboard.*
      if ( (this->GetLocalClipboard ( gsClip )) > ZERO )
      {
         //* Set contents of system clipboard.*
         if ( (this->wcbSet ( gsClip )) > ZERO )
            status = true ;
         else     //* Alert user that transfer failed.*
         {
            // InfoDialog(sys call failed)
         }
      }
   }
   return status ;
}

//*************************
//*  wcbSystemCb2LocalCb  *
//*************************
//********************************************************************************
//* Private Method                                                               *
//* --------------                                                               *
//* If the Wayland Clipboard interface is active, copy the contents of the       *
//* system clipboard to the local clipboard.                                     *
//*                                                                              *
//*                                                                              *
//* Input  : none (data to be copied is on the local clipboard)                  *
//*                                                                              *
//* Returns: 'true' if data copied to system clipboard, else 'false'             *
//********************************************************************************

bool NcDialog::wcbSystemCb2LocalCb ( void )
{
   bool status = false ;

   if ( (wcbClip != NULL) && (wcbClip->wcbIsConnected ()) )
   {
      gString gsClip ;

      //* Get contents of system clipboard.*
      if ( (this->wcbGet ( gsClip )) > ZERO )
      {
         //* Set contents of local clipboard.  *
         this->SetLocalClipboard ( gsClip ) ;
         status = true ;
      }
      else
      {
         // InfoDialog(sys call failed)
      }
   }
   return status ;
}


//******************************************************************************
//*             dtbmData class implementation (see NcDialog.hpp)               *
//******************************************************************************

//*************************
//*       dtbmData        *
//*************************
//******************************************************************************
//* This constructor initializes all data members to default values:           *
//*  - text data is initialized to a null string                               *
//*  - text color is control's non-focus color (dtbmNFcolor)                   *
//*  - 'refreshControl' == 'true' and 'centered' == ZERO                       *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: implicitly returns pointer to object                              *
//******************************************************************************

dtbmData::dtbmData ( void )
{

   wchar_t wch = nckNULLCHAR ;
   attr_t attr = dtbmNFcolor ;
   this->init ( &wch, &attr, true, ZERO, false ) ;

}  //* End dtbmData() *

//*************************
//*       dtbmData        *
//*************************
//******************************************************************************
//* This constructor initializes text data (other fields to default values).   *
//*                                                                            *
//* Input  : tData : pointer to display text (UTF-8 OR wchar_t)                *
//*                  max length==gsALLOCMIN*4 (UTF-8) or gsALLOCDFLT (wchar_t) *
//*          NOTE: text color is control's non-focus color (dtbmNFcolor)       *
//*                'refreshControl' == 'true' and 'centered' == ZERO           *
//*                                                                            *
//* Returns: implicitly returns pointer to object                              *
//******************************************************************************

dtbmData::dtbmData ( const char* tData )
{

   gString gs( tData ) ;
   attr_t attr = dtbmNFcolor ;
   this->init ( gs.gstr(), &attr, true, ZERO, false ) ;

}  //* End dtbmData() *

dtbmData::dtbmData ( const wchar_t* tData )
{

   attr_t attr = dtbmNFcolor ;
   this->init ( tData, &attr, true, ZERO, false ) ;

}  //* End dtbmData() *

//*************************
//*       dtbmData        *
//*************************
//******************************************************************************
//* This constructor initializes both text data and color attribute data.      *
//* Input  : tData : pointer to display text (UTF-8 OR wchar_t)                *
//*          cData : array of color attributes, one for each character         *
//*                  OR cData[0]==dtbmNFcolor || cData[0]==dtbmFcolor          *
//*         refresh: (optional, 'true' by default)                             *
//*                  refresh display i.e. make text visible immediately        *
//*         center : (optional, ZERO by default)                               *
//*                  If specified, the message will be centered in the target  *
//*                  field, completely filling the field.                      *
//*                  if > ZERO, specifies width of target textbox control      *
//*                   AND optionally, number of elements in 'cData'            *
//*         rtl    : (optional, 'false' by default)                            *
//*                  if 'true', then process text as an RTL language           *
//*                             (Arabic, Hebrew, Persian, Urdu, etc.)          *
//*                                                                            *
//* IMPORTANT NOTE: If cData[0] != dtbmNFcolor && cData[0] != dtbmFcolor,      *
//* then your color data buffer must have at least as many elements as the     *
//* number of characters (not byte count) in tData. ALSO, if the 'center'      *
//* parameter is specified, the length of the tData string is dynamically      *
//* adjusted to fill the target field, so the number of color attributes in    *
//* the 'cData' array must be >= 'center'. Otherwise, a system memory access   *
//* violation, an application crash and the ridicule of your peers will be     *
//* the likely results. To be safe, if you are not using one of the defined    *
//* default color attributes, always use a color buffer of at least            *
//* MAX_DIALOG_WIDTH.       BEE YEE CAREFUL!                                   *
//*                                                                            *
//* Returns: implicitly returns pointer to object                              *
//******************************************************************************

dtbmData::dtbmData ( const wchar_t* tData, const attr_t* cData, 
                     bool refresh, short center, bool rtl )
{

   this->init ( tData, cData, refresh, center, rtl ) ;

}  //* End dtbmData() *

dtbmData::dtbmData ( const char* tData, const attr_t cData[gsALLOCDFLT], 
                     bool refresh, short center, bool rtl )
{

   gString gs( tData ) ;
   this->init ( gs.gstr(), cData, refresh, center, rtl ) ;

}  //* End dtbmData() *

//*************************
//*         init          *
//*************************
//******************************************************************************
//* Initialize our data members. Called by constructor.                        *
//*                                                                            *
//* Input  : tData : pointer to wchar_t display text                           *
//*          cData : array of color attributes, one for each character         *
//*                  OR cData[0]==dtbmNFcolor || cData[0]==dtbmFcolor          *
//*         refresh: if 'true, refresh display (make text visible immediately) *
//*                  if 'false', do not refresh display                        *
//*         center : if non-ZERO, specifies width of target textbox control    *
//*                   AND optionally, number of elements in 'cData'            *
//*         rtl    : if 'true',  then process text as an RTL language          *
//*                  if 'false', then process text as an LTR language          *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void dtbmData::init ( const wchar_t* tData, const attr_t* cData, 
                      bool refresh, short center, bool rtl )
{
   gString gs( tData ) ;
   gs.copy( this->textData, gsALLOCDFLT ) ;
   if ( *cData == dtbmNFcolor || *cData == dtbmFcolor )
      *this->colorData = *cData ;
   else                    // copy caller's color array
   {
      short attrCount = center > ZERO ? center : gs.gschars(),
            i ;
      for ( i = ZERO ; i < attrCount ; i++ )
         this->colorData[i] = cData[i] ;
      while ( i < gsALLOCDFLT ) // be thorough, fill with default color attribute
         this->colorData[i++] = attr_t(ZERO) ;
   }
   this->refreshControl = refresh ;
   this->centered = bool(center > ZERO) ;
   this->rtlText = rtl ;

}  //* End init() *

//*************************
//*      operator =       *
//*************************
//******************************************************************************
//* Assign a new text string. (UTF-8 OR wchar_t)                               *
//* (color attributes,'refresh' flag and 'centered' flag are unchanged)        *
//*                                                                            *
//* Input  : tData : pointer to new display text (UTF-8 OR  wchar_t)           *
//*                  max length == gsDFLTBYTES (UTF-8) or gsALLOCDFLT (wchar_t)*
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
                                                          
void dtbmData::operator = ( const char *tData )
{
   gString gs( tData ) ;
   gs.copy( this->textData, gsALLOCDFLT ) ;
}  //* End operator=() *

void dtbmData::operator = ( const wchar_t *tData )
{
   gString gs( tData ) ;
   gs.copy( this->textData, gsALLOCDFLT ) ;
}  //* End operator=() *
//******************************************************************************
//*                 End of dtbmData class implementation                       *
//******************************************************************************



#if DEBUG_ET != 0 || DEBUG_ETM != 0
//*************************
//*       ET_Stat         *
//*************************
//******************************************************************************
//* Write diagnostic data to the debugging window.                             *
//*  - if 'gsPtr' != NULL, display its data instead of stat data               *
//*                                                                            *
//* Input  : cp    : handle to dctTEXTBOX object currently under edit          *
//*          gsPtr : (optional, NULL pointer by default) ad-hoc data           *
//*                  if specified, display this data INSTEAD OF the stat data. *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note: RTL text data are displayed backward by this method,    *
//* but this is not critical in this context. Only the stat data need to be    *
//* correct because "It's Only a Northern Song." (Paul MacCartney)             *
//*                                                                            *
//******************************************************************************

void NcDialog::ET_Stat ( DialogTextbox* cp, gString* gsPtr )
{
   //* If ad-hoc data provided *
   if ( gsPtr != NULL )
   {
      winPos wp( 1, 1 ) ;
      for ( short i = 1 ; i < etStatY ; i++ )
         etPtr->ClearLine ( i ) ;
      etPtr->WriteParagraph ( wp, *gsPtr, etColor, true ) ;
   }
   //* Else display edit status info *
   else
   {
      //* Clear the status display area *
      winPos wp( etStatY, 1 ) ;
      for ( short i = etStatY ; i < (etRows -1) ; i++ )
         etPtr->ClearLine ( i ) ;

      const attr_t ndColor = nc.rebl ;
      gString gs ;
      short redge = etCols - 1 ;
      wchar_t wChar ;
      bool overflow = false ;

      if ( cp->select.active != false )
      {  //* If we have a selection in progress, report that first *
         wchar_t dirChar = cp->select.fwd ? wcsRARROW : wcsLARROW ;
         gs.compose( L"Select %02hd:%02hd %hd %C\n", 
                     &cp->select.base.ypos, &cp->select.base.xpos, 
                     &cp->select.count, &dirChar ) ;
         wp = etPtr->WriteParagraph ( wp, gs, nc.cybl ) ;
      }
      else  ++wp.ypos ;

      //* Single-line textbox *
      if ( cp->lines == 1 )
      {
         cp->tbGetText ( gs ) ;
         short totalChars = gs.gschars(),
               iPoint = cp->tbIndex.xpos ;
         wchar_t ipChar = gs.gstr()[iPoint], 
                 ipDisp = ipChar ;
         if ( ipChar == nckNEWLINE )   // make non-printing characters visible
            ipDisp = wcsLARROW ;
         else if ( ipChar == nckNULLCHAR )
            ipDisp = wcsDARROW ;
         gsetb.compose( L"Index  %02hd:%02hd lI:%02hd\n"
                         "Cursor %02hd:%02hd ip'%C' 0x%04X\n"
                         "-    -    -    -    -    -\n"
                         "rows:%02hd cols:%02hd totCh:%3hd\n",
                        &cp->tbIndex.ypos, &cp->tbIndex.xpos, &cp->leftIndex,
                        &cp->tbCursor.ypos, &cp->tbCursor.xpos, &ipDisp, &ipChar,
                        &cp->lines, &cp->cols, &totalChars ) ;
         wp = etPtr->WriteParagraph ( wp, gsetb, etColor ) ;

         //* Report the head data. (right-justified fields have no head data) *
         if ( cp->leftIndex > ZERO && cp->rightJust == false )
         {
            for ( short hIndex = ZERO ; hIndex < cp->leftIndex ; )
            {
               wChar = cp->wText[hIndex++] ;
               if ( wChar == nckSPACE )      // make space characters visible
                  wChar = wcsBLOCKm ;
               if ( wp.xpos >= redge )
               { ++wp.ypos ; wp.xpos = 1 ; }
               if ( wp.ypos >= (etRows - 1) )
               { overflow = true ; break ; }
               wp = etPtr->WriteChar ( wp, wChar, ndColor ) ;
            }
         }

         if ( ! overflow )
         {
            // Report the displayed data. *
            gs = cp->rightJust == false ? &cp->wText[cp->leftIndex] :
                                          cp->wText ;
            gString gstail = gs ;
            gs.limitCols( cp->cols ) ;
            gstail.shiftChars( -((gs.gschars()) - 1) ) ;
            while ( gs.gschars() > 1 )
            {
               wChar = *gs.gstr() ;
               gs.shiftChars( -1 ) ;
               if ( wp.xpos >= redge )
               { ++wp.ypos ; wp.xpos = 1 ; }
               if ( wp.ypos >= (etRows - 1) )
               { overflow = true ; break ; }
               wp = etPtr->WriteChar ( wp, wChar, etColor ) ;
            }

            if ( ! overflow )
            {
               //* Report the tail data. (right-justified fields have no tail data) *
               while ( gstail.gschars() > 1 )
               {
                  wChar = *gstail.gstr() ;
                  gstail.shiftChars( -1 ) ;
                  if ( wChar == nckSPACE )      // make space characters visible
                     wChar = wcsBLOCKm ;
                  if ( wp.xpos >= redge )
                  { ++wp.ypos ; wp.xpos = 1 ; }
                  if ( wp.ypos >= (etRows - 1) )
                  { overflow = true ; break ; }
                  wp = etPtr->WriteChar ( wp, wChar, ndColor ) ;
               }
            }
         }
      }

      //* Multi-line textbox *
      else
      {
         cp->tbGetText ( gs, false ) ;
         short totalChars = gs.gschars(),
               iPoint = cp->mlIndex2Ip ( gs ) ;    // iPoint with newlines
         wchar_t ipChar = gs.gstr()[iPoint], 
                 ipDisp = ipChar ;
         if ( ipChar == nckNEWLINE )   // make non-printing characters visible
            ipDisp = wcsLARROW ;
         else if ( ipChar == nckNULLCHAR )
            ipDisp = wcsDARROW ;
         gsetb.compose( L"Index  %02hd:%02hd lI:%02hd iP%3hd\n"
                         "Cursor %02hd:%02hd ip'%C' 0x%04X\n"
                         "-    -    -    -    -    -\n"
                         "rows:%02hd cols:%02hd totCh:%3hd\n",
                        &cp->tbIndex.ypos, &cp->tbIndex.xpos, 
                        &cp->leftIndex, &iPoint, 
                        &cp->tbCursor.ypos, &cp->tbCursor.xpos, &ipDisp, &ipChar,
                        &cp->lines, &cp->cols, &totalChars ) ;
         wp = etPtr->WriteParagraph ( wp, gsetb, etColor ) ;

         //* Report the head data. *
         if ( cp->leftIndex > ZERO )
         {
            for ( short hIndex = ZERO ; cp->mlHead[hIndex] != nckNULLCHAR ; )
            {
               wChar = cp->mlHead[hIndex++] ;
               if ( wChar == nckSPACE )      // make space characters visible
                  wChar = wcsBLOCKm ;
               if ( wp.xpos >= redge )
               { ++wp.ypos ; wp.xpos = 1 ; }
               if ( wp.ypos >= (etRows - 1) )
               { overflow = true ; break ; }
               wp = etPtr->WriteChar ( wp, wChar, ndColor ) ;
            }
         }

         if ( ! overflow )
         {
            // Report the displayed data. *
            for ( short i = ZERO ; i < cp->lines && !overflow ; i++ )
            {
               if ( *cp->mlText[i].ln == nckNULLCHAR )
                  break ;
               for ( short j = ZERO ; cp->mlText[i].ln[j] != nckNULLCHAR ; )
               {
                  wChar = cp->mlText[i].ln[j++] ;
                  if ( wp.xpos >= redge )
                  {
                     wp.xpos = 1 ;
                     if ( ++wp.ypos >= (etRows - 1) )
                     { overflow = true ; break ; }
                  }
                  wp = etPtr->WriteChar ( wp, wChar, etColor ) ;
               }
            }

            if ( ! overflow )
            {
               //* Report the tail data. *
               if ( *cp->mlTail != nckNULLCHAR && ! overflow )
               {
                  for ( short tIndex = ZERO ; 
                        cp->mlTail[tIndex] != nckNULLCHAR ; )
                  {
                     wChar = cp->mlTail[tIndex++] ;
                     if ( wChar == nckSPACE )      // make space characters visible
                        wChar = wcsBLOCKm ;
                     if ( wp.xpos >= redge )
                     {
                        wp.xpos = 1 ;
                        if ( ++wp.ypos >= (etRows - 1) )
                        { overflow = true ; break ; }
                     }
                     wp = etPtr->WriteChar ( wp, wChar, ndColor ) ;
                  }
               }
            }
         }
      }
      etPtr->RefreshWin () ;
   }
   //* Because textbox control is under edit, we must restore cursor position. *
   cp->wPtr->SetCursor ( cp->tbCursor ) ;
   cp->wPtr->RefreshWin () ;

}  //* End ET_Stat() *
#endif   // DEBUG_ET || DEBUG_ETM

//*************************
//*                       *
//*************************
//********************************************************************************
//*                                                                              *
//*                                                                              *
//*                                                                              *
//* Input  :                                                                     *
//*                                                                              *
//* Returns:                                                                     *
//********************************************************************************

