//********************************************************************************
//* File       : NcdKey.cpp                                                      *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2006-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in NcDialog.hpp            *
//* Date       : 30-Aug-2021                                                     *
//* Version    : (see NcDialogVersion string in NcDialog.cpp)                    *
//*                                                                              *
//* Description: This module contains contains:                                  *
//*   a) The application level GetKeyInput() method which handles both key       *
//*      input and mouse event translation.                                      *
//*                                                                              *
//*   b) The NcDialog-class extensions to the NcWindow-class mouse               *
//*      configuration group of methods.                                         *
//*      Because mouse events arrive in the NcDialog class through the           *
//*      key-input stream, our local GetKeyInput() method transforms the         *
//*      mouse data before it is returned to the other NcDialog-class members    *
//*      or to the application.                                                  *
//*                                                                              *
//*   c) This module also contains the NcDialog-class UserAlert() methods.       *
//*      While the NCurses class also includes a UserAlert() method, it is a     *
//*      simple, parameter-less beep. The NcDialog-class implementation is       *
//*      somewhat more sophisticated, yet still annoying enough to get           *
//*      the user's attention.                                                   *
//*                                                                              *
//*   d) Also included in this module are the shell-access methods:              *
//*          Shell_Out()                                                         *
//*          Pseudo_Shell()                                                      *
//*      These methods cause the dialog window to temporarily go to sleep        *
//*      so the user has access to the command line.                             *
//*                                                                              *
//*                                                                              *
//* Development Tools: See NcDialog.cpp.                                         *
//********************************************************************************
//* Technical notes on mouse support.                                            *
//* 1) Meese are cantankerous beasts. What the terminal receives from the        *
//*    system desktop (GNOME/KDE/etc.) is heavily filtered, and what we          *
//*    receive from the ncurses library is not much, although it's probably      *
//*    the best it can do on a multi-platform, multi-hardware target base.       *
//*    The good news is that mouse support on a text-based application is        *
//*    really unnecessary and even counter-productive in terms of efficiency;    *
//*    thus, what we are able to do here will probably never be used anyway.     *
//*                                                                              *
//* 2) All ncurses mouse functions are accessed through the NCurses class.       *
//*    NCurses does its job rather well, but doesn't make many intelligent       *
//*    decisions about the data it gathers. That is its job -- its level of      *
//*    abstraction.                                                              *
//*                                                                              *
//* 3) Our job here in the NcDialog class is to make the semi-raw mouse-event    *
//*    data meaningful on an application level.                                  *
//*    See the meTransformEvent() method for details.                            *
//*                                                                              *
//*                                                                              *
//********************************************************************************

//*****************
//* Include Files *
//*****************

#include "GlobalDef.hpp"               //* General definitions

#ifndef NCURSES_INCLUDED
#include "NCurses.hpp"
#endif

#ifndef NCDIALOG_INCLUDED
#include "NcDialog.hpp"
#endif

//******************************
//* Local Definitions and Data *
//******************************
//* To avoid complicated definition checking for the ncurses  *
//* NCURSES_MOUSE_VERSION constant, we define equivalents to  *
//* accomodate any undefined macros.                          *
//* A test for a Button5 event will always fail unless        *
//* Button5 is defined.                                       *
//* NOTE: If the ncurses.h definitions for mouse version > 1  *
//*       change, this may break.                             *
static const mmask_t 
   Button5_Released = 
      NCURSES_MOUSE_VERSION <= 1 ? 0x00000000 : 0x00100000,
   Button5_Pressed = 
      NCURSES_MOUSE_VERSION <= 1 ? 0x00000000 : 0x00200000,
   Button5_Clicked = 
      NCURSES_MOUSE_VERSION <= 1 ? 0x00000000 : 0x00400000,
   Button5_Double_Clicked = 
      NCURSES_MOUSE_VERSION <= 1 ? 0x00000000 : 0x00800000,
   Button5_Triple_Clicked = 
      NCURSES_MOUSE_VERSION <= 1 ? 0x00000000 : 0x01000000 ;

//* Mouse event types which are converted to keycodes. *
//* Defining them here means that we can easily add    *
//* more convertible types later.                      *
// Programmer's Note: It would be more friendly to let the application choose 
// which event type do what, but that is for a later release.
static const short convertTYPES = 4 ;
static short convTypes[convertTYPES] =
{
   metB1_S,          // Button 1 - single click
   metB1_D,          // Button 1 - double click
   metSW_D,          // Scroll-wheel - scroll down
   metSW_U           // Scroll-wheel - scroll up
} ;


//*************************
//*     GetKeyInput       *
//*************************
//******************************************************************************
//* Get a keycode and keytype from the ncursesw input stream.                  *
//* This includes all supported international character codes as well as       *
//* captured-and-translated escape sequences representing function keys,       *
//* cursor keys and other special-purpose key combinations.                    *
//*                                                                            *
//* If mouse input has been enabled, mouse events will also be returned in     *
//* caller's wkeyCode-class object's 'mevent' data member.                     *
//*                                                                            *
//* Input  : wKey ( by reference, initial values ignored)                      *
//*          receives the key input data                                       *
//*                                                                            *
//* Returns: member of wkeyType                                                *
//******************************************************************************
//* Programmer's Note: This method overrides the definition of the             *
//* NcWindow-class method of the same name but calls it directly.              *
//*                                                                            *
//******************************************************************************

wkeyType NcDialog::GetKeyInput ( wkeyCode& wKey )
{

   NcWindow::GetKeyInput ( wKey ) ; // call the obscured method
   if ( wKey.type == wktMOUSE )
   {  //* Mouse event captured. Transform coordinates to dialog-relative, *
      //* and optionally convert mouse event to keycode (if possible).    *
      this->meTransformEvent ( wKey ) ;
   }
   return wKey.type ;

}  //* End GetKeyInput() *

      //*************************************************
      //**   This Section Implements the Methods for   **
      //**      NcDialog-class Mouse Support.          **
      //** - - - - - - - - - - - - - - - - - - - - - - **
      //**                                             **
      //**                                             **
      //*************************************************

//*************************
//* meAssignDefaultHotkey *
//*************************
//******************************************************************************
//* PRIVATE METHOD.                                                            *
//* If specified control has no associated hotkey, then assign a default       *
//* hotkey, so mouse events can push the input focus to targeted control.      *
//*                                                                            *
//* Input  : cIndex: index into array of dialog controls                       *
//*                                                                            *
//* Returns: OK  if successful, else ERR                                       *
//******************************************************************************
//* Programmer's Note: This code will not appear in the control label, and     *
//* will not be visible to the application. (see notes in module header)       *
//* If mouse input is not enabled, then this value will remain unreferenced.   *
//*                                                                            *
//* Note: These are not actual characters, just useful codes; so if you have a *
//* need to print it for debugging purposes, 'wchar_t' is a 32-bit integer,    *
//* so use a "%X" specifier.                                                   *
//*                                                                            *
//* Codes are uppercase alpha with bit 7 set:                                  *
//*    'A' == 0x00000031                                                       *
//*  code  == 0x000000B1                                                       *
//* 'Real' (user-supplied) hotkeys are actual keycodes.                        *
//*                                                                            *
//******************************************************************************

short NcDialog::meAssignDefaultHotkey ( short cIndex )
{
   short status = ERR ;
   if ( cIndex >= ZERO && cIndex <= this->lastCtrl && 
        (this->dCtrl[cIndex]->bHotkey.hotkey == false) )
   {
      this->dCtrl[cIndex]->bHotkey.hotchar = wchar_t((L'A' | 0x00080) + cIndex) ;
      this->dCtrl[cIndex]->bHotkey.hotdflt = true ;
      status = OK ;
   }
   return status ;

}  //* End meAssignDefaultHotkey() *

//**************************
//*    meTransformEvent    *
//**************************
//******************************************************************************
//* PRIVATE METHOD.                                                            *
//* 1) Transform screen-oriented mouse Y/X coordinates to dialog-relative      *
//*    coordinates.                                                            *
//* 2) Translate the event mask into a member of enum meTypes and set/reset    *
//*    modifier-key flags.                                                     *
//* 3) Determine which, if any, control in the dialog is under the mouse       *
//*    pointer.                                                                *
//* 4) If the 'meAuto' flag is set, then if possible, convert the mouse event  *
//*    to a keytype/keycode pair. The mouse-event data are still available in  *
//*    the wk.mevent member.                                                   *
//*                                                                            *
//* Input  : wk    : (by reference) wkeyCode object containing the mouse event *
//*                                                                            *
//* Returns: if a control is under the mouse pointer:                          *
//*            returns control index                                           *
//*          if no control is under the mouse pointer:                         *
//*            returns MAX_DIALOG_CONTROLS                                     *
//*          ERR if mouse pointer coordinates are outside the dialog window    *
//*              or if 'wk' parameter does not contain a mouse event           *
//******************************************************************************
//* Notes on transforming a mouse event to a control object's hotkey:          *
//* - For controls with a fixed size, a simple calculation for whether the     *
//*   mouse pointer lies within its scope will suffice.                        *
//* - However, for controls that can change size, we need to consider the      *
//*   control's state. Currently, there are two control types that can change  *
//*   size based on state: DialogMenuwin and DialogDropdown.                   *
//* - If either a DialogMenuwin or DialogDropdown is in the expanded state,    *
//*   then it may be obscuring other controls. If so, then the event must be   *
//*   directed to the expanded control, NOT to the obscured controls beneath   *
//*   it. For this reason, we must pre-test all controls of this type to       *
//*   determine whether any are in an expanded state, and further determine    *
//*   whether the event occurred within that expanded area.                    *
//* - In the 'collapsed' state, these controls will respond to mouse events    *
//*   much like fixed-size controls, but with perhaps some special processing. *
//*                                                                            *
//* - DialogDropdown class:                                                    *
//*   - If a Dropdown control is expanded, then it is being edited and all     *
//*     other controls are, by definition, collapsed.                          *
//*   - If a Dropdown control has input focus, then                            *
//*     - a hotkey will expand a collapsed control                             *
//*     - a hotkey referencing the current control will be ignored by an       *
//*       expanded control                                                     *
//*     - A ScrollWheel event will move the highlight in the specified         *
//*       direction.                                                           *
//*                                                                            *
//* - DialogMenuwin class:                                                     *
//*   - An expanded Menuwin control may or may not have the input focus;       *
//*     however, if ANY Menuwin control is in an expanded state, then it, or   *
//*     one of its sub-menus DOES have input focus and is being edited.        *
//*   - The Menuwin control being edited AND any member of its group may be in *
//*     an expanded state, and thus may be obscuring one or more other         *
//*     controls. This is tedious to test for (see meIsObscured()), but we     *
//*     must do it.                                                            *
//*   - Note that collapsed context-menu Menuwin objects are invisible and     *
//*     inactive, and are therefore not sensitive to mouse events. This is     *
//*     somewhat different from a GUI, where a right click will open a context *
//*     menu. For us, the problem would be WHICH context menu to open, so that *
//*     scenario must be handled at a higher (application) level.              *
//*   - Note also that expanded Menuwin controls may extend OUTSIDE (below)    *
//*     the boundaries of the dialog window, and this would have to be handled *
//*     as a special case except that as detailed above, we would in that case *
//*     be transmitting a hotkey code which would be ignored anyway.           *
//*     Instead a click within an expanded Menuwin highlights and selects that *
//*     item.                                                                  *
//*   - A ScrollWheel event within an expanded Menuwin will move the highlight *
//*     in the specified direction.                                            *
//*   - It is particularly difficult to handle events that affect expanded     *
//*     Menuwin controls -- expecially members of a Menuwin group. We simply   *
//*     don't have enough information to smoothly process (with one keycode)   *
//*     all requests the user may make. The user interface gets ckunky in some *
//*     such situations. Sorry about that...                                   *
//*                                                                            *
//* - Logic:                                                                   *
//*   a) An event outside the dialog boundaries is ignored.                    *
//*   b) An event inside the dialog boundaries but not over an active control  *
//*      will be translated to dialog-relative Y/X coordinates BUT will not be *
//*      translated to a keycode UNLESS: one or more controls is in an         *
//*      expanded state.                                                       *
//*         In that case, an nckESC will be returned which will cause all      *
//*         expanded controls to be collapsed and focus will move to the       *
//*         next control (or if focus is on a member of a control group,       *
//*         then focus moves out of the control group).                        *
//*   c) Only an 'active' control may receive the input focus.                 *
//*      A control must be 'active' or its position is considered as empty     *
//*      space in the dialog.                                                  *
//*   d) All active controls are visible, though not all visible controls are  *
//*      active, but see the special case of a hidden MenuBar.                 *
//*   e) All expanded controls are active, and with one exception, an expanded *
//*      control will have the input focus.                                    *
//*      EXCEPTION: A Menuwin control which is expanded but does not have      *
//*                 input focus, is an ancestor of the Menuwin control which   *
//*                 does have focus.                                           *
//*   f) All controls (except an expanded Menuwin) have a rectangular shape.   *
//*   g) As a design decision, we will return only one keycode. This means     *
//*      that even though we COULD induce more natural results on some         *
//*      operations by pushing one or more extra keycodes into the input queue,*
//*      we don't. This could cause the user a bit of minor head scratching,   *
//*      but he/she/it will get used to it. [We may revisit this later.]       *
//* These rules make our testing easier, but if the rules change, the tests    *
//* may need to be adjusted. For instance, if new control is defined with      *
//* expanded/collapsed states.                                                 *
//*                                                                            *
//* NOTE: We cannot provide full, GUI-like mouse support here. It would require*
//*       a significant rewrite of the NcDialog class to make it more mouse    *
//*       oriented, which we believe would be a huge waste of time for a       *
//*       text-based environment.                                              *
//*     - In a GUI environment, clicking on a particular item within a control *
//*       that contains a list of items selects that item.                     *
//*       For us this is DialogScrollbox and DialogScrollext controls, plus    *
//*       EXPANDED DialogDropdown and DialogMenuwin controls.                  *
//*     - Also, in a GUI environment, there are scroll widgets on scrolling    *
//*       controls that the mouse could use to scroll (view) the undisplayed   *
//*       items in the control; however, this implementation does not support  *
//*       such widgets. However, we can and do capture ScrollWheel events for  *
//*       scrolling through lists and adjusting Spinners.                      *
//*                                                                            *
//* NOTE: Testing of this functionality is done in Test #4 of the Dialog4      *
//*       test application.                                                    *
//*                                                                            *
//* -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -  *
//* The Alchemy Behind the Transformations:                                    *
//*                                                                            *
//* 1) Transform the screen-oriented Y/X coordinates into dialog-window-       *
//*    relative coordinates. This is fairly simple:                            *
//*    a) If the event happend outside the writeable area of the terminal      *
//*       window, we never see it.                                             *
//*    b) If the event happened inside the writeable area of the terminal      *
//*       window, but OUTSIDE our dialog window, we do no coordinate           *
//*       translation because it's not our event --UNLESS-- the event belongs  *
//*       to a Menuwin control that has expanded beyond the lower border.      *
//*    c) If the event occurred within our dialog window (or the extended      *
//*       Menuwin), then the wkeyCode.mevent.ypos and wkeyCode.mevent.xpos get *
//*       coordinates relative to the upper-left corner of the dialog.         *
//*                                                                            *
//* 2) Determine whether the event is of a type we care about.                 *
//*    a) Currently, we act ONLY on:                                           *
//*           Button #1, single click                                          *
//*           Button #1, double click                                          *
//*           ScrollWheel, scroll up                                           *
//*           ScrollWheel, scroll down                                         *
//*       The 'modifier keys' (SHIFT, CTRL, ALT) may or may not be referenced. *
//*       See the 'convTypes[] array at the top of this module.                *
//*    b) In future, we hope to make the list user-configurable, but that      *
//*       will require a much more sophisticated transformation algorithm.     *
//*                                                                            *
//* 3) Determine whether the event occurred above a dialog control object      *
//*    or above empty space in the dialog window.                              *
//*    a) If the event is over empty dialog space, then it is probably         *
//*       meaningless to the application because it is not directly associated *
//*       with any particular control object.                                  *
//*       - Exception: If a control is currently in the 'expanded' state, then *
//*         we return an nckESC which will collapse the control and move the   *
//*         focus forward.                                                     *
//*    b) If the event is over a dialog control object, then certain actions   *
//*       can be deduced, and we can convert the mouse event into a keycode,   *
//*       based on the control type (and state, if any), so that the           *
//*       application (and indeed, the rest of the NcDialog class) need know   *
//*       nothing about the mouse input.                                       *
//*       See below for interpretation of events for each control type.        *
//*       - Note that the target control must be 'active', else it will be     *
//*         interpreted as empty dialog-window space.                          *
//*       - If there are expanded controls, AND a hotkey event occurs outside  *
//*         any expanded control, we signal an nckESC to collapse all controls.*
//*       - If there are expanded controls, AND we have a hotkey event for a   *
//*         DIFFERENT expanded control (Menuwin only), then we would like to   *
//*         collapse all submenus back to the menu the hotkey belongs to.      *
//*         (submenus don't have hotkeys)                                      *
//*       - Note that we cannot identify a 'drag' event because the ncursesw   *
//*         library does not report mouse movements that are not associated    *
//*         with an specific event. Too bad  :-(                               *
//*    c) To be able to convert mouse-click events, it is necessary that each  *
//*       control be associated with a hotkey.                                 *
//*       - We could force the application programmer to define hotkeys for    *
//*         all controls when the mouse input is enabled; however, this would  *
//*         be an unreasonable demand on the application programmer.           *
//*       - If the application specified a hotkey for the control we use it.   *
//*       - Otherwise, we quietly provide a default (and hidden) hotkey for    *
//*         the control.                                                       *
//*         - These keycodes do not interfere with the application hotkey      *
//*           definitions, neither the ASCII alpha or equivalent ASCII control *
//*           codes. These default hokey codes are assigned in the NcDialog    *
//*           constructor via a call to meAssignDefaultHotkey().               *
//*    d) Happily, ScrollWheel events are reported by the ncurses library.     *
//*       The ncurses implementation is kludgy, but effective (at least on     *
//*       our test machines).                                                  *
//*       machines).                                                           *
//*       - Pushbutton, Radiobutton, Billboard, collapsed Menuwin,, collapsed  *
//*         Dropdown and Textbox controls: ScrollWheel events are meaningless, *
//*         and therefore are not converted to keycodes.                       *
//*       - Scrollbox, Scrollext, expanded Menuwin, expanded Dropdown, Spinner *
//*         and Slider controls: ScrollWheel events are converted to navigation*
//*         keys. These scroll the highlight for for scrolling controls, or    *
//*         in the case of Spinners and Sliders, increment/decrement the value.*
//*         - The actual navigation key mapped depends on the states of the    *
//*           modifier keys (CTRL, ALT, SHIFT). Note that the SHIFT key is not *
//*           reported for click events by the ncursesw library in Gnome or    *
//*           KDE or Xwin terminal windows, but all three modifier keys are    *
//*           reported with ScrollWheel events.                                *
//*                                                                            *
//*                                                                            *
//*                  Mapping mouse events to dialog controls.                  *
//*                  ========================================                  *
//*                                                                            *
//* Mouse event types are referenced by a member of enum meTypes AFTER they    *
//* are translated from the raw bit-mask codes.                                *
//*                                                                            *
//*                                                                            *
//* DialogPushbutton (type dctPUSHBUTTON):                                     *
//* --------------------------------------                                     *
//*   metB1_S : single click of left button                                    *
//*   metB1_D : double click of left button                                    *
//*   This is equivalent to the user 'pressing' the button.                    *
//*    a) If the target control IS NOT already the control with input focus,   *
//*       convert the mouse event to the hotkey associated with the target     *
//*       control. This will give the control focus AND automatically indicates*
//*       that the button has been pressed.                                    *
//*    b) If the target control IS already the control with input focus,       *
//*       just convert the mouse event to an nckENTER key.                     *
//*   All other mouse event types are ignored for Pushbutton controls.         *
//*                                                                            *
//* DialogRadiobutton (type dctRADIOBUTTON):                                   *
//* ----------------------------------------                                   *
//*   metB1_S : single click of left button                                    *
//*   metB1_D : double click of left button                                    *
//*   This is equivalent to the user either 'toggling' or 'selecting' the      *
//*   button (depending on whether the button is grouped).                     *
//*   a) If the target control IS NOT already the control with input focus,    *
//*      convert the mouse event to the hotkey associated with the target      *
//*      control. This will give the control focus AND automatically indicates *
//*      that the button has been 'selected' (set).                            *
//*   b) If the target control IS already the control with input focus, then   *
//*      the mouse event is converted to an nckSPACE key.                      *
//*      - If the control is an independent Radiobutton, then nckSPACE         *
//*        _toggles_ the state of the button.                                  *
//*      - If the control is a member of an XOR Radiobutton group, then        *
//*        nckSPACE _sets_ the state of the button.                            *
//*   All other mouse event types are ignored for Radiobutton controls.        *
//*                                                                            *
//* DialogTextbox (type dctTEXTBOX):                                           *
//* --------------------------------                                           *
//*   metB1_S : single click of left button (no modifier keys)                 *
//*   The action is dependent on whether the control currently has focus.      *
//*   a) If the target control IS NOT already the control with input focus,    *
//*      then convert the mouse event to the hotkey associated with the target *
//*      control. This will give the control focus.                            *
//*   b) If the target control IS already the control with input focus, then   *
//*      the position of the click within the control indicates where the      *
//*      insertion point (text cursor) should be located.                      *
//*      NOTE: Specifying the insertion point via mouse is currently not       *
//*            implemented due to the technical difficulty involved in         *
//*            returning multiple keystrokes in response to a mouse event.     *
//*            We will try to overcome this problem in a future release.       *
//*                                                                            *
//*   metB1_S : single click of left button (with CTRL modifier key)           *
//*   metB1_D : double click of left button                                    *
//*   The action is dependent on whether the control currently has focus.      *
//*   a) If the target control _is not_ the control with input focus,          *
//*      then the event has the same effect as a single-click                  *
//*      with no modifier key.                                                 *
//*   b) If the target control _is_ the control with input                     *
//*      focus, then convert the event to the nckENTER key which will complete *
//*      the edit and move the input focus to the next control.                *
//*                                                                            *
//*   metSW_U : Scroll-wheel - scroll up                                       *
//*   metSW_D : Scroll-wheel - scroll down                                     *
//*   This is equivalent to pressing the Right or Left navigation keys.        *
//*   a) If the target control _is not_ the control with input focus,          *
//*      these events are ignored.                                             *
//*   b) If the target control _is_ the control with input focus:              *
//*      metSW_U is converted to the nckRIGHT navigation key, which will       *
//*              move the highlight one character toward the end of the        *
//*              data, if any.                                                 *
//*              (If 'SHIFT' modifier key, then converted to nckSRIGHT.)       *
//*      metSW_D is converted to the nckLEFT navigation key, which will        *
//*              move the highlight one character toward the beginning         *
//*              of the data, if any.                                          *
//*              (If 'SHIFT' modifier key, then converted to nckSLEFT.)        *
//*      The data will be scrolled as necessary to bring the target            *
//*      character into view.                                                  *
//*      Note that the sense of Right and Left are reversed if the control     *
//*      contains RTL language text.                                           *
//*                                                                            *
//*   All other mouse event types are ignored for Textbox controls.            *
//*                                                                            *
//* DialogScrollbox (type dctSCROLLBOX):                                       *
//* DialogScrollext (type dctSCROLLEXT):                                       *
//* ------------------------------------                                       *
//*   metB1_S : single click of left button (with or without modifier keys)    *
//*   metB1_D : double click of left button (with or without modifier keys)    *
//*   The action is dependent on whether the control currently has focus.      *
//*   a) If the target control IS NOT already the control with input focus,    *
//*      then a click event anywhere on or within the control's borders will   *
//*      be converted to the hotkey associated with the target control.        *
//*      This will give the control focus.                                     *
//*   b) If the target control IS already the control with input focus, then:  *
//*      - if the event occurred in the control's top border, then             *
//*        the event is converted to the nckUP navigation key, which           *
//*        will move the highlight upward to the previous line item, if any.   *
//*      - if the event occurred in the control's bottom border _or_           *
//*        in the left or right border, then the event is converted to the     *
//*        nckDOWN navigation key, which will move the highlight downward      *
//*        to the next line item, if any.                                      *
//*      - if the event occurs on a line item within the control, then         *
//*        the highlight is moved to that line item, and the event is          *
//*        converted to the nckENTER key which 'selects' that line item.       *
//*      The data will be scrolled as necessary to bring the target line item  *
//*      into view.                                                            *
//*                                                                            *
//*   metSW_U : Scroll-wheel - scroll up                                       *
//*   metSW_D : Scroll-wheel - scroll down                                     *
//*   This is equivalent to pressing the Up or Down navigation keys.           *
//*   a) If the target control IS NOT already the control with input focus,    *
//*      then these events are ignored.                                        *
//*   b) If the target control IS already the control with input focus, then   *
//*      metSW_U is converted to the nckDOWN navigation key, which will move   *
//*              the highlight downward to the next line item, if any          *
//*      metSW_D is converted to the nckUP navigation key, which will move     *
//*              the highlight upward to the previous line item, if any        *
//*      The data will be scrolled as necessary to bring the target line item  *
//*      into view.                                                            *
//*                                                                            *
//* DialogDropdown (type dctDROPDOWN):                                         *
//* ----------------------------------                                         *
//*   metB1_S : single click of left button (with or without modifier keys)    *
//*   metB1_D : double click of left button (with or without modifier keys)    *
//*   The action is dependent on whether the control currently has focus AND   *
//*   whether the control is in the 'collapsed' or 'expanded' state.           *
//*   a) If the target control _is not_ the control with input focus,          *
//*      OR if the control has focus but is in the collapsed state, then       *
//*      convert the mouse event to the hotkey associated with the target      *
//*      control. This will give the control focus AND automatically           *
//*      indicates that the control is to be immediately 'expanded'.           *
//*   b) Otherwise, the control _does_ have focus AND is in 'expanded' state.  *
//*      - if the event occurred in the control's top border, then             *
//*        the event is converted to the nckUP navigation key, which           *
//*        will move the highlight upward to the previous line item, if any.   *
//*      - if the event occurred in the control's bottom border _or_           *
//*        in the left or right border, then the event is converted to the     *
//*        nckDOWN navigation key, which will move the highlight downward      *
//*        to the next line item, if any.                                      *
//*      - if the event occurs on a line item within the control, then         *
//*        the highlight is moved to that line item, and the event is          *
//*        converted to the nckENTER key which 'selects' that line item.       *
//*        This selection returns the control to the 'collapsed' state.        *
//*      The data will be scrolled as necessary to bring the target line item  *
//*      into view.                                                            *
//*                                                                            *
//*   metSW_U : Scroll-wheel - scroll up                                       *
//*   metSW_D : Scroll-wheel - scroll down                                     *
//*   This is equivalent to pressing the Up or Down navigation keys.           *
//*    a) If the target control _is not_ the control with input focus,         *
//*       OR if the control is in the collapsed state, then these events       *
//*       are ignored.                                                         *
//*    b) If the target control _is_ the control with input focus, AND is in   *
//*       the expanded state:                                                  *
//*       metSW_U is converted to the nckDOWN navigation key, which will       *
//*               move the highlight downward to the next line item, if any.   *
//*       metSW_D is converted to the nckUP navigation key, which will         *
//*               move the highlight upward to the previous line item, if any. *
//*       The data will be scrolled as necessary to bring the target line      *
//*       item into view.                                                      *
//*                                                                            *
//* DialogMenuwin (type dctMENUWIN):                                           *
//* --------------------------------                                           *
//*   metB1_S : single click of left button (with or without modifier keys)    *
//*   metB1_D : double click of left button (with or without modifier keys)    *
//*   The action is dependent on whether the control currently has focus,      *
//*   whether the control is in the 'collapsed' or 'expanded' state, and       *
//*   whether the control is a top-level menu or a submenu.                    *
//*                                                                            *
//*   a) If the target control is in the _collapsed_ state, then               *
//*      convert the mouse event to the hotkey associated with the target      *
//*      control. This will give the control focus AND automatically           *
//*      indicates that the control is to be immediately 'expanded'.           *
//*   b) If the target control is in the _expanded_ state, AND                 *
//*      has input focus:                                                      *
//*      - if the event occurred in the title area, then the event is ignored. *
//*      - if the event occurred in the control's top border, then             *
//*        the event is converted to the nckUP navigation key, which           *
//*        will move the highlight upward to the previous line item, if any,   *
//*        or will wrap around to the last item.                               *
//*      - if the event occurred in the control's bottom border, _or_ in the   *
//*        left or right borders, then the event is converted to the nckDOWN   *
//*        navigation key, which will move the highlight downward to the next  *
//*        line item, if any, or will wrap around to the first item.           *
//*      - if the event occurs on a line item within the control, then         *
//*        the highlight is moved to that line item, and the event is          *
//*        converted to the nckENTER key which 'selects' that line item.       *
//*        This selection returns the control to the 'collapsed' state.        *
//*   c) If the target control is in the _expanded_ state, BUT                 *
//*      _does not_ have focus, then:                                          *
//*      the target control is the parent of the submenu which does have       *
//*      focus. The event is converted into the nckLEFT key, which will        *
//*      collapse the submenu which has focus. This may, or may not return     *
//*      focus to the target control (depending on how many levels of          *
//*      submenu are expanded).                                                *
//*                                                                            *
//*   metSW_U : Scroll-wheel - scroll up                                       *
//*   metSW_D : Scroll-wheel - scroll down                                     *
//*   This is equivalent to pressing the Up or Down navigation keys.           *
//*   a) If the target control _is not_ the control with input focus, _or_ if  *
//*      the control has focus but is in the collapsed state, then these       *
//*      events are ignored.                                                   *
//*   b) If the target control _is_ the control with input focus, _and_        *
//*      the control is in the expanded state:                                 *
//*      metSW_U is converted to the nckDOWN navigation key, which will        *
//*              move the highlight downward to the next line item, if any,    *
//*              and otherwise will wrap around to the first line item.        *
//*      metSW_D is converted to the nckUP navigation key, which will          *
//*              move the highlight upward to the previous line item, if       *
//*              any, and otherwise will wrap around to the last line item.    *
//*                                                                            *
//*   All other mouse event types are ignored for Menuwin controls.            *
//*                                                                            *
//* DialogSpinner (type dctSPINNER):                                           *
//* --------------------------------                                           *
//*   metB1_S : single click of left button (no modifier keys)                 *
//*   The action is dependent on whether the control currently has focus.      *
//*   a) If the target control _is not_ the control with input focus,          *
//*      then convert the mouse event to the hotkey associated with the        *
//*      target control. This will give the control focus.                     *
//*   b) If the target control is _already_ the control with                   *
//*      input focus, then the event is ignored.                               *
//*                                                                            *
//*   metB1_S : single click of left button (with CTRL modifier key)           *
//*   metB1_D : double click of left button                                    *
//*   The action is dependent on whether the control currently has focus.      *
//*   a) If the target control _is not_ the control with input focus,          *
//*      then the event has the same effect as a single-click                  *
//*      with no modifier key.                                                 *
//*   b) If the target control _is_ the control with input                     *
//*      focus, then convert the event to the nckENTER key which will complete *
//*      the edit and move the input focus to the next control.                *
//*                                                                            *
//*   metSW_U : Scroll-wheel - scroll up                                       *
//*   metSW_D : Scroll-wheel - scroll down                                     *
//*   This is equivalent to pressing the Up or Down navigation keys.           *
//*    a) If the target control _is not_ the control with input focus,         *
//*       then these events are ignored.                                       *
//*    b) If the target control is _already_ the control with input            *
//*       focus:                                                               *
//*       metSW_U is converted to the nckUP navigation key, which will         *
//*               increase the value in the control by one(1).                 *
//*          - If the SHIFT key is held down when this event is received,      *
//*            the event is converted to the nckSUP (SHIFT+UpArrow)            *
//*            key, which will increase the value in the control by ten(10).   *
//*          - If the CTRL key is held down when this event is received,       *
//*            the event is converted to the nckCUP (CTRL+UpArrow)             *
//*            key, which will increase the value in the control by 100.       *
//*          - If the ALT key is held down when this event is received,        *
//*            the event is converted to the nckAUP (ALT+UpArrow)              *
//*            key, which will increase the value in the control by 1000.      *
//*       metSW_D is converted to the nckDOWN navigation key, which will       *
//*               decrease the value in the control by one(1).                 *
//*          - If the SHIFT key is held down when this event is received,      *
//*            the event is converted to the nckSDOWN (SHIFT+DownArrow)        *
//*            key, which will decrease the value in the control by ten(10).   *
//*          - If the CTRL key is held down when this event is received,       *
//*            the event is converted to the nckCDOWN (CTRL+DownArrow)         *
//*            key, which will decrease the value in the control by 100.       *
//*          - If the ALT key is held down when this event is received,        *
//*            the event is converted to the nckADOWN (ALT+DownArrow)          *
//*            key, which will decrease the value in the control by 1000.      *
//*                                                                            *
//*  All other mouse event types are ignored for Spinner controls.             *
//*                                                                            *
//* DialogSlider (type dctSLIDER):                                             *
//* ------------------------------                                             *
//*   metB1_S : single click of left button (no modifier keys)                 *
//*   The action is dependent on whether the control currently has focus.      *
//*   a) If the target control _is not_ the control with input focus,          *
//*      then convert the mouse event to the hotkey associated with the        *
//*      target control. This will give the control focus.                     *
//*   b) If the target control is _already_ the control with                   *
//*      input focus, then the event is (currently) ignored.                   *
//*      (In future, we MAY allow this event to change the Slider value.)      *
//*                                                                            *
//*   metB1_S : single click of left button (with CTRL modifier key)           *
//*   metB1_D : double click of left button                                    *
//*   The action is dependent on whether the control currently has focus.      *
//*   a) If the target control _is not_ the control with input focus,          *
//*      then the event has the same effect as a single-click                  *
//*      with no modifier key.                                                 *
//*   b) If the target control _is_ the control with input                     *
//*      focus, then convert the event to the nckENTER key which will complete *
//*      the edit and move the input focus to the next control.                *
//*                                                                            *
//*   metSW_U : Scroll-wheel - scroll up                                       *
//*   metSW_D : Scroll-wheel - scroll down                                     *
//*   This is logically equivalent to pressing the Up or Down navigation keys, *
//*   but with certain variations in interpretation depending on the the       *
//*   target control's "Input Mode" i.e. the direction in which the slider     *
//*   bar grows.                                                               *
//*    a) If the target control is _already_ the control with input            *
//*       focus, the the Input Mode determies the interpretation of            *
//*       scroll-wheel events.                                                 *
//*       Mode 1: Slider growth direction is upward.                           *
//*        metSW_U is converted to the nckUP navigation key, which will        *
//*                increase the value in the control by one(1) step.           *
//*           - If the SHIFT key is held down when this event is received,     *
//*             the event is converted to the nckPGUP (Page-Up) key, which     *
//*             will increase the value in the control by ten percent.         *
//*           - If the CTRL key or ALT key is held down when this event is     *
//*             received, the event is converted to the nckHOME (Home) key     *
//*             which will set the control to its maximum value.               *
//*        metSW_D is converted to the nckDOWN navigation key, which will      *
//*                decrease the value in the control by one(1) step.           *
//*           - If the SHIFT key is held down when this event is received,     *
//*             the event is converted to the nckPGDOWN (Page-Down) key,       *
//*             which will decrease the value in the control by ten percent.   *
//*           - If the CTRL key or ALT key is held down when this event is     *
//*             received, the event is converted to the nckEND (End) key       *
//*             which will set the control to its minimum value.               *
//*                                                                            *
//*       Mode 2: Slider growth direction is downward.                         *
//*        metSW_D is converted to the nckDOWN navigation key, which will      *
//*                increase the value in the control by one(1) step.           *
//*           - If the SHIFT key is held down when this event is received,     *
//*             the event is converted to the nckPGDOWN (Page-Down) key,       *
//*             which will increase the value in the control by ten percent.   *
//*           - If the CTRL key or ALT key is held down when this event is     *
//*             received, the event is converted to the nckEND (End) key       *
//*             which will set the control to its maximum value.               *
//*        metSW_U is converted to the nckUP navigation key, which will        *
//*                decrease the value in the control by one(1) step.           *
//*           - If the SHIFT key is held down when this event is received,     *
//*             the event is converted to the nckPGUP (Page-Up) key, which     *
//*             will decrease the value in the control by ten percent.         *
//*           - If the CTRL key or ALT key is held down when this event is     *
//*             received, the event is converted to the nckHOME (Home) key     *
//*             which will set the control to its minimum value.               *
//*                                                                            *
//*       Mode 3: Slider growth direction is to the right.                     *
//*        metSW_U is converted to the nckRIGHT navigation key, which will     *
//*                increase the value in the control by one(1) step.           *
//*           - If the SHIFT key is held down when this event is received,     *
//*             the event is converted to the nckPGUP (Page-Up) key, which     *
//*             will increase the value in the control by ten percent.         *
//*           - If the CTRL key or ALT key is held down when this event is     *
//*             received, the event is converted to the nckHOME (Home) key     *
//*             which will set the control to its maximum value.               *
//*        metSW_D is converted to the nckLEFT navigation key, which will      *
//*                decrease the value in the control by one(1) step.           *
//*           - If the SHIFT key is held down when this event is received,     *
//*             the event is converted to the nckPGDOWN (Page-Down) key,       *
//*             which will decrease the value in the control by ten percent.   *
//*           - If the CTRL key or ALT key is held down when this event is     *
//*             received, the event is converted to the nckEND (End) key       *
//*             which will set the control to its minimum value.               *
//*                                                                            *
//*       Mode 4: Slider growth direction is to the left.                      *
//*        metSW_D is converted to the nckLEFT navigation key, which will      *
//*                increase the value in the control by one(1) step.           *
//*           - If the SHIFT key is held down when this event is received,     *
//*             the event is converted to the nckPGDOWN (Page-Down) key,       *
//*             which will increase the value in the control by ten percent.   *
//*           - If the CTRL key or ALT key is held down when this event is     *
//*             received, the event is converted to the nckEND (End) key       *
//*             which will set the control to its maximum value.               *
//*        metSW_U is converted to the nckRIGHT navigation key, which will     *
//*                decrease the value in the control by one(1) step.           *
//*           - If the SHIFT key is held down when this event is received,     *
//*             the event is converted to the nckPGUP (Page-Up) key, which     *
//*             will decrease the value in the control by ten percent.         *
//*           - If the CTRL key or ALT key is held down when this event is     *
//*             received, the event is converted to the nckHOME (Home) key     *
//*             which will set the control to its minimum value.               *
//*                                                                            *
//*    b) If the target control _is not_ the control with input focus,         *
//*       then these events are ignored.                                       *
//*                                                                            *
//* DialogBillboard (type dctBILLBOARD):                                       *
//* ------------------------------------                                       *
//*   Billboard controls are never active; they are read-only controls from    *
//*   the user's point of view. Thus, they cannot receive the input focus,     *
//*   and cannot be associated with mouse events.                              *
//*                                                                            *
//*   Even if you mistakenly set a Billboard control as 'active' in your       *
//*   application, input to that control will be ignored.                      *
//*      Programmer's Note: We actually do move the focus to an ACTIVE         *
//*      Billboard control for purposes of testing; however, an application    *
//*      should never activate a Billboard, so we don't include this factoid   *
//*      in the public documentation.                                          *
//*                                                                            *
//*   -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -   *
//* Programmer's Note: Special processing when user clicks an un-expanded      *
//* menu which is in the same MenuBar group as the currently-expanded menu:    *
//* - After sending the signal to collapse all menus, (ESC key), then insert   *
//*   the target menu's hotkey into the data stream.                           *
//*   - If the menu with focus is in the same group as the target, we can      *
//*     simply insert the target's hotkey into the stream.                     *
//*   - However, if the expanded menu with focus is a sub-menu, we have        *
//*     trouble because sub-menus are not group members.                       *
//*   - Because sub-menus do not have titles, we can distinguish between       *
//*     parent and sub-menu; however, sub-menus don't know who their parent is.*
//*     Therefore, if the menu with focus is a sub-menu AND if the target is a *
//*     member of a MenuBar group, we have to identify the parent through a    *
//*     scan for an expanded menu which is of the same group as the target.    *
//*     - If found, we can insert its hotkey into the stream.                  *
//*       If not found, then expanded parent and target are not in the same    *
//*       group, and there is no need to push a keycode into the stream.       *
//*                                                                            *
//*                                                                            *
//******************************************************************************

short NcDialog::meTransformEvent ( wkeyCode& wk )
{
   short status = ERR ;

   //* Convert mouse-event-type bitmask value to enum value *
   if ( wk.type == wktMOUSE )
   {
      wk.mevent.meType = metNONE ;     // initialize the event type
      mmask_t  eType = wk.mevent.eventType &
                       ~(BUTTON_CTRL | BUTTON_SHIFT | BUTTON_ALT) ; 
      if ( eType == BUTTON1_CLICKED )
         wk.mevent.meType = metB1_S ;
      else if ( eType == BUTTON1_DOUBLE_CLICKED )
         wk.mevent.meType = metB1_D ;
      else if ( eType == BUTTON1_TRIPLE_CLICKED )
         wk.mevent.meType = metB1_T ;
      else if ( eType == BUTTON1_PRESSED )
         wk.mevent.meType = metB1_P ;
      else if ( eType == BUTTON1_RELEASED )
         wk.mevent.meType = metB1_R ;

      if ( eType == BUTTON2_CLICKED )
         wk.mevent.meType = metB2_S ;
      else if ( eType == BUTTON2_DOUBLE_CLICKED )
         wk.mevent.meType = metB2_D ;
      else if ( eType == BUTTON2_TRIPLE_CLICKED )
         wk.mevent.meType = metB2_T ;
      else if ( eType == BUTTON2_PRESSED )
         wk.mevent.meType = metB2_P ;
      else if ( eType == BUTTON2_RELEASED )
         wk.mevent.meType = metB2_R ;

      if ( eType == BUTTON3_CLICKED )
         wk.mevent.meType = metB3_S ;
      else if ( eType == BUTTON3_DOUBLE_CLICKED )
         wk.mevent.meType = metB3_D ;
      else if ( eType == BUTTON3_TRIPLE_CLICKED )
         wk.mevent.meType = metB3_T ;
      else if ( eType == BUTTON3_PRESSED )
         wk.mevent.meType = metB3_P ;
      else if ( eType == BUTTON3_RELEASED )
         wk.mevent.meType = metB3_R ;

      if ( eType == BUTTON4_CLICKED )
         wk.mevent.meType = metB4_S ;
      else if ( eType == BUTTON4_DOUBLE_CLICKED )
         wk.mevent.meType = metB4_D ;
      else if ( eType == BUTTON4_TRIPLE_CLICKED )
         wk.mevent.meType = metB4_T ;
      else if ( eType == BUTTON4_PRESSED )
      {
         mmask_t emt ;
         this->meGetEventTypes ( emt ) ;
         if ( (emt & REPORT_SCROLL_WHEEL) == REPORT_SCROLL_WHEEL )
            wk.mevent.meType = metSW_D ;
         else
            wk.mevent.meType = metB4_P ;
      }
      else if ( eType == BUTTON4_RELEASED )
         wk.mevent.meType = metB4_R ;

      else if ( eType == REPORT_MOUSE_POSITION )
      {
         // Programmer's Note: It is not currently possible to request a 
         // position report, so this is a spurious event _even though_
         // actual position reports do sometimes get through.
         wk.mevent.meType = metPOS ;
      }

      else if ( NCURSES_MOUSE_VERSION > 1 )
      {
         if ( (eType & Button5_Clicked) != ZERO )
            wk.mevent.meType = metB5_S ;
         else if ( (eType & Button5_Double_Clicked) != ZERO )
            wk.mevent.meType = metB5_D ;
         else if ( (eType & Button5_Triple_Clicked) != ZERO )
            wk.mevent.meType = metB5_T ;
         else if ( (eType & Button5_Pressed) != ZERO )
         {
            mmask_t emt ;
            this->meGetEventTypes ( emt ) ;
            if ( (emt & REPORT_SCROLL_WHEEL) == REPORT_SCROLL_WHEEL )
               wk.mevent.meType = metSW_U ;
            else
               wk.mevent.meType = metB5_P ;
         }
         else if ( (eType & Button5_Released) != ZERO )
            wk.mevent.meType = metB5_R ;
      }

      if ( wk.mevent.meType == metNONE )
      {  //* No event identified. If coordinates specified, *
         //* assume position report, else report no event.  *
         if ( (wk.mevent.sypos > ZERO) && (wk.mevent.sxpos > ZERO) )
            wk.mevent.meType = metPOS ;
      }
   }

   //* Initialize modifier-key flags *
   // Programmer's Note: Shift+Click events are probably captured by
   // the terminal but Shift+ScrollWheel events MAY BE visible.
   if ( (wk.mevent.eventType & BUTTON_CTRL)  != ZERO ) wk.mevent.cKey = true ;
   if ( (wk.mevent.eventType & BUTTON_SHIFT) != ZERO ) wk.mevent.sKey = true ;
   if ( (wk.mevent.eventType & BUTTON_ALT)   != ZERO ) wk.mevent.aKey = true ;
   if ( (wk.mevent.eventType & REPORT_MOUSE_POSITION) != ZERO ) wk.mevent.mPos = true ;

   bool isEvent      = false,    // 'true' if a mouse event detected
        dialogEvent  = false,    // 'true' if the event belongs to us
        convertEvent = false ;   // 'true' if event type is one of the events
                                 //        which we convert to keycodes

   //* If the coordinates are >= to top left of dialog *
   if ( wk.type == wktMOUSE && 
        wk.mevent.sypos >= this->dulY && wk.mevent.sxpos >= this->dulX )
   {
      isEvent = true ;

      //* If X coordinate is <= right edge of dialog *
      if ( wk.mevent.sxpos < (this->dulX + this->dCols) )
      {
         //* If Y coordinate is <= bottom edge of dialog
         if ( wk.mevent.sypos < (this->dulY + this->dLines) )
            dialogEvent = true ;

         //* Test for event within an expanded Menuwin control *
         //* which extends below the dialog window.            *
         else
         {
            for ( short ci = ZERO ; ci <= this->lastCtrl ; ci++ )
            {
               if ( this->dCtrl[ci]->type == dctMENUWIN && 
                    this->dCtrl[ci]->active != false )
               {
                  DialogMenuwin* cp = (DialogMenuwin*)(this->dCtrl[ci]) ;
                  if ( (cp->expanded != false) && 
                       ((cp->ulY + cp->lines) > (this->dulY + this->dLines)) &&
                       (wk.mevent.sxpos >= cp->ulX && 
                        wk.mevent.sxpos < (cp->ulX + cp->cols)) &&
                       (wk.mevent.sypos < (cp->ulY + cp->lines))
                     )
                  {
                     dialogEvent = true ;
                     break ;
                  }
               }
            }
         }
      }

      if ( dialogEvent != false )
      {
         //* Pointer is within dialog boundaries, but not yet *
         //* tested for association with a control object.    *
         status = MAX_DIALOG_CONTROLS ;
         wk.mevent.cIndex = -1 ;

         //* Convert coordinates from terminal-window-relative *
         //* to dialog-window-relative.                        *
         wk.mevent.ypos = wk.mevent.sypos - this->dulY ;
         wk.mevent.xpos = wk.mevent.sxpos - this->dulX ;
         wk.mevent.zpos = ZERO ; // z-dimension not currently supported by ncurses

         //* Test whether this event is on our list of events to be converted.*
         //*              (see note at the top of this module)                *
         for ( short i = ZERO ; i < convertTYPES ; i++ )
         {
            if ( wk.mevent.meType == convTypes[i] )
            {
               convertEvent = true ;
               break ;
            }
         }
      }
   }

   //* IF: a) we have a mouse event, and                *
   //*     b) the event belongs to us, and              *
   //*     c) this is a convertible event, and          *
   //*     d) event conversion is enabled,              *
   //* THEN: test for association with a control object.*
   if ( isEvent && dialogEvent && convertEvent && this->meAuto != false )
   {
      short cyt,           // control's Top line
            cyb,           // control's Bottom line
            cxl,           // control's Left column
            cxr,           // control's Right column
            expCtrl = -1,  // index of any control in an expanded state
            multiY ;       // y-offset within multi-item control with focus
      bool hasFocus,       // true if target control has input focus
           trgFound = false,// true if event associated with a target control
           expMWin = false;// true if non-focus, expanded Menuwin control

      //* Pre-scan for controls that may be 'expanded' and therefore *
      //* may be obscuring other controls. (see notes above)         *
      for ( short ci = ZERO ; 
            wk.mevent.cIndex < ZERO && ci <= this->lastCtrl ; ci++ )
      {
         //* Does control under test have focus? *
         hasFocus = bool(ci == this->currCtrl ) ;

         if ( this->dCtrl[ci]->active != false )
         {
            if ( this->dCtrl[ci]->type == dctDROPDOWN )
            {
               DialogDropdown* cp = (DialogDropdown*)(this->dCtrl[ci]) ;
               if ( cp->expanded != false )
               {  //* Potentially-obscures other controls in the window *
                  expCtrl = ci ;
                  cyt = cp->ulY - this->dulY ;
                  cyb = cyt + cp->lines - 1 ;
                  cxl = cp->ulX - this->dulX ;
                  cxr = cxl + cp->cols - 1 ;
                  if ( wk.mevent.ypos >= cyt && wk.mevent.ypos <= cyb &&
                       wk.mevent.xpos >= cxl && wk.mevent.xpos <= cxr )
                  {  //* Event references the control under edit.*
                     wk.mevent.cIndex = ci ; // target identified
                     trgFound = true ;
                     multiY = wk.mevent.ypos - cyt ; // indicates vertical offset
                     break ;
                  }
               }
            }
            else if ( this->dCtrl[ci]->type == dctMENUWIN )
            {
               DialogMenuwin* cp = (DialogMenuwin*)(this->dCtrl[ci]) ;
               if ( cp->expanded != false )
               {  //* Potentially-obscures other controls in the window *
                  expCtrl = ci ;
                  expMWin = true ;
                  cyt = cp->tulY - this->dulY ;
                  cyb = cyt + cp->lines ;    // height of title + body
                  if ( cp->tCols == ZERO )   // submenus have no title
                     --cyb ;
                  cxl = cp->ulX - this->dulX ;
                  cxr = cxl + cp->cols - 1 ;
                  //* Event in title or in menu body?            *
                  //* Test title line for menus that have titles,*
                  //* Else test for menu body only.              *
                  if ( (cp->tCols > ZERO && 
                         (wk.mevent.ypos == cyt) && (wk.mevent.xpos >= cxl) &&
                         (wk.mevent.xpos < (cxl + cp->tCols)))
                        ||
                       (cp->tCols == ZERO && wk.mevent.ypos == cyt)
                        ||
                        (wk.mevent.ypos > cyt && wk.mevent.ypos <= cyb &&
                         wk.mevent.xpos >= cxl && wk.mevent.xpos <= cxr) )
                  {
                     wk.mevent.cIndex = ci ; // target identified
                     trgFound = true ;
                     multiY = wk.mevent.ypos - cyt ;  // indicates vertical offset
                     if ( hasFocus != false )
                     {  //* Event references the control under edit.*
                        expMWin = false ;
                     }
                     break ;
                  }
               }
            }
         }
      }

      //* If target control not already identified, then *
      //* scan for a control object under the pointer.   *
      for ( short ci = ZERO ; 
            trgFound == false && ci <= this->lastCtrl ; ci++ )
      {
         if ( this->dCtrl[ci]->active != false )
         {
            //* Does control under test have focus? *
            hasFocus = bool(ci == this->currCtrl ) ;

            cyt = this->dCtrl[ci]->ulY - this->dulY ;    // control's top Y
            if ( (this->dCtrl[ci]->type == dctMENUWIN) &&
                 (*this->dCtrl[ci]->wLabel != NULLCHAR) )
               --cyt ;  // (include the Menuwin label)
            cxl = this->dCtrl[ci]->ulX - this->dulX ;    // control's left X

            //* Test top and left boundaries *
            if ( wk.mevent.ypos >= cyt && wk.mevent.xpos >= cxl )
            {
               cyb = cyt + this->dCtrl[ci]->lines - 1 ;  // control's bottom Y
               cxr = cxl + this->dCtrl[ci]->cols - 1 ;   // control's right X

               //* Special tests for expanding controls *
               if ( this->dCtrl[ci]->type == dctDROPDOWN )
               {
                  //* Control is in a collapsed state; expanded Dropdowns   *
                  //* already identified above.                             *
                  //* 'x' extents are known, but find 'y' extents.          *
                  DialogDropdown* cp = (DialogDropdown*)(this->dCtrl[ci]) ;
                  cyt = cp->tulY - this->dulY ;
                  cyb = cyt + cp->tLines - 1 ;
                  if ( (wk.mevent.ypos >= cyt) && (wk.mevent.ypos <= cyb) && 
                       (wk.mevent.xpos <= cxr) )
                  {
                     wk.mevent.cIndex = ci ; // target identified
                     trgFound = true ;
                     break ;
                  }
               }
               else if ( this->dCtrl[ci]->type == dctMENUWIN )
               {
                  //* Control may be in either a collapsed or expanded      *
                  //* state, but if expanded AND the target, it has already *
                  //* been identified, and if it is expanded and NOT the    *
                  //* target, then it has been eliminated -- so test only   *
                  //* collapsed controls (i.e. the title area):             *
                  //*   a) an active, independent menu                      *
                  //*   b) an active member of a VISIBLE MenuBar group      *
                  //* Also, verify Target NOT obscured by expanded control. *
                  DialogMenuwin* cp = (DialogMenuwin*)(this->dCtrl[ci]) ;
                  if ( (cp->expanded == false) && ((cp->groupCode == ZERO) || 
                       (cp->groupCode > ZERO && cp->grpVisible != false)) )
                  {
                     cxr = cxl + cp->tCols - 1 ; // right extent of menu title
                     if ( (wk.mevent.ypos == cyt) && (wk.mevent.xpos <= cxr)
                          && ((expCtrl < ZERO) || !(this->meIsObscured ( wk ))) )
                     {
                        wk.mevent.cIndex = ci ; // target identified
                        trgFound = true ;
                        break ;
                     }
                  }
               }

               //* Test for fixed-size controls *
               else
               {
                  if ( (wk.mevent.ypos <= cyb) && (wk.mevent.xpos <= cxr) && 
                       (! expMWin || !(this->meIsObscured( wk ))) )
                  {
                     wk.mevent.cIndex = ci ; // target identified
                     trgFound = true ;

                     //* For multi-item controls *
                     if ( this->dCtrl[ci]->type == dctSCROLLBOX || 
                          this->dCtrl[ci]->type == dctSCROLLEXT )
                     {
                        multiY = wk.mevent.ypos - cyt ;
                     }
                     break ;
                  }
               }  // fixed-size controls
            }     // top & left match
         }        // 'active'
      }           // for(;;)

      //* If a target control has been identified, AND               *
      //* if one or more Menuwin controls are expanded, AND          *
      //* if there is a click event that doesn't belong to any of    *
      //* them, then collapse them all. If click event targets a     *
      //* collapsed member of the same MenuBar group, target found.  *
      // Programmer's Note: This is a bloody-tricky test sequence, beware!
      //                    See notes in method header.
      if ( (trgFound != false ) && (expCtrl >= ZERO) &&
           (this->dCtrl[expCtrl]->type == dctMENUWIN) &&
           (wk.mevent.cIndex != this->currCtrl) &&
           (wk.mevent.meType != metSW_U) && (wk.mevent.meType != metSW_D) )
      {
         bool collapseAll = true ;     // assume that we will collapse
         if ( this->dCtrl[wk.mevent.cIndex]->type == dctMENUWIN )
         {  //* Access the data of target control *
            DialogMenuwin* cp = (DialogMenuwin*)(this->dCtrl[wk.mevent.cIndex]) ;

            //* If target is already expanded, just close the top sub-menu. *
            if ( cp->expanded != false )
               collapseAll = false ;

            //* If the (collapsed) target menu is a member of a MenuBar group *
            else if ( cp->groupCode != ZERO )
            {
               bool recode = false ;

               //* If expanded menu and target menu belong to the SAME group *
               if ( (this->dCtrl[expCtrl]->groupCode != ZERO) &&
                      (this->dCtrl[expCtrl]->groupCode == cp->groupCode) )
                  recode = true ;

               //* If the expanded menu with focus is a sub-menu,   *
               //* find the group code (if any) of its parent menu. *
               else
               {  //* Access the data of expanded control with focus. *
                  DialogMenuwin* cpx = (DialogMenuwin*)(this->dCtrl[expCtrl]) ;
                  if ( cpx->tCols == ZERO )  // indicates a sub-menu
                  {  //* Scan the array of controls to get index of parent. *
                     short idx,                       // step index
                           tidx = wk.mevent.cIndex ;  // target index
                     char  tgrp = cp->groupCode ;     // target group code
                     for ( idx = ZERO ; idx <= this->lastCtrl ; ++idx )
                     {
                        if ( (idx != tidx) && (this->dCtrl[idx]->type == dctMENUWIN) &&
                             (this->dCtrl[idx]->groupCode == tgrp) )
                        {  //* We have located another member of the target's  *
                           //* MenuBar group. Test whether it is expanded.     *
                           //* This test works because only one group-member   *
                           //* menu may be expanded at a given moment.         *
                           if ( ((DialogMenuwin*)(this->dCtrl[idx]))->expanded )
                           {
                              recode = true ;
                              break ;
                           }
                        }
                     }
                  }
               }
               if ( recode )
               {  //* Convert to standard hotkey keycode and *
                  //* push the keycode into the input queue. *
                  wkeyCode wktmp = wk ;
                  this->meRecodeHotkey ( wktmp ) ;
                  this->UngetKeyInput ( wktmp ) ;
               }
            }
         }
         if ( collapseAll != false )
         {
            wk.key = nckESC ;
            wk.type = wktFUNKEY ;
            wk.mevent.conv = true ;
            trgFound = false ;
         }
      }

      //* If a convertible event is associated with a control object *
      // Programmer's Note: We split these tests more than necessary 
      // to make future enhancements easier.
      if ( trgFound != false )
      {
         //* Get target control type and index, and     *
         //* determine if target has focus.             *
         //* (return value is index of target control.) *
         short cIndex = status = wk.mevent.cIndex ;
         DCType cType = this->dCtrl[cIndex]->type ;
         hasFocus = bool(cIndex == this->currCtrl ) ;
         wk.mevent.conv = true ;    // conversion is provisionally complete

         //* If target is a Menuwin control *
         if ( cType == dctMENUWIN )
         {
            DialogMenuwin* cp = (DialogMenuwin*)(this->dCtrl[cIndex]) ;

            //* Handle ScrollWheel events:                           *
            //* Valid ONLY if control is expanded AND has focus.     *
            if ( wk.mevent.meType == metSW_U || wk.mevent.meType == metSW_D )
            {
               if ( hasFocus && cp->expanded )
               {
                  wk.type = wktFUNKEY ;   // navigation keys are function keys
                  if ( wk.mevent.meType == metSW_U )  // ScrollWheel UP
                     wk.key = nckDOWN ;
                  else                                // ScrollWheel DOWN
                     wk.key = nckUP ;
               }
               else  wk.mevent.conv = false ;   // no conversion
            }

            //* Handle Click events: *
            else
            {
               //* If click event is inside an expanded Menuwin control *
               //* that has focus                                       *
               if ( hasFocus && cp->expanded )
               {
                  if ( cp->tCols > ZERO && multiY == ZERO )
                  { //* Currently, a click in the title of *
                    //* an expanded Menuwin does nothing.  *
                  }
                  else
                  {
                    //* Set for ZERO == top border *
                    if ( cp->tCols > ZERO && multiY > ZERO )
                       --multiY ;

                     wk.type = wktFUNKEY ;   // navigation keys are function keys
                     //* Click in top border *
                     if ( multiY == ZERO )
                     {
                        wk.key = nckUP ;
                     }
                     //* Click in bottom border OR in left or right border *
                     else if ( (multiY == (this->dCtrl[cIndex]->lines - 1)) ||
                               (wk.mevent.xpos == cxl) || (wk.mevent.xpos == cxr) )
                     {
                        wk.key = nckDOWN ;
                     }
                     //* Click on line item *
                     else
                     {
                        short selItem = 
                        this->meIdentifyClickItem ( cp->wPtr, (multiY - 1) ) ;
                        if ( (selItem >= ZERO) && (selItem < cp->bItems) )
                        {  //* Move the highlight to the indicated item.*
                           while ( cp->bIndex < selItem )
                              cp->bIndex = cp->wPtr->ScrollData ( nckDOWN ) ;
                           while ( cp->bIndex > selItem )
                              cp->bIndex = cp->wPtr->ScrollData ( nckUP ) ;
                           wk.key = nckENTER ;
                        }
                        else  // this is unlikey, but be safe
                        {
                           wk.type = wktMOUSE ;
                           wk.mevent.conv = false ;
                        }
                     }
                  }
               }

               //* If Menuwin is expanded, but does not have focus,     *
               //* then collapse the submenu which DOES have focus.     *
               else if ( cp->expanded )
               {
                  wk.type = wktFUNKEY ;
                  wk.key  = nckLEFT ;
               }

               //* Menuwin is in the collapsed state (with or without   *
               //* focus). Hotkey (click) expands it.                   *
               else
                  this->meRecodeHotkey ( wk ) ;
            }
         }

         else if ( cType == dctDROPDOWN )
         {
            //* Access to the 'expanded' flag *
            DialogDropdown* cp = (DialogDropdown*)(this->dCtrl[cIndex]) ;

            //* If the event type is ScrollWheel, AND if target   *
            //* control currently has focus, AND if control is    *
            //* expanded, then convert to navigation key.         *
            if ( wk.mevent.meType == metSW_U || wk.mevent.meType == metSW_D )
            {
               if ( hasFocus && cp->expanded )
               {
                  wk.type = wktFUNKEY ;   // navigation keys are function keys
                  if ( wk.mevent.meType == metSW_U )  // ScrollWheel UP
                     wk.key = nckDOWN ;
                  else                                // ScrollWheel DOWN
                     wk.key = nckUP ;
               }
               else  wk.mevent.conv = false ;   // no conversion
            }

            //* If the event type is Click, AND if control has    *
            //* focus AND if control is expanded, then determine  *
            //* WHERE the click occurred:                         *
            //* a) if top border, scroll highlight up             *
            //* b) if bottom border, or unconditioned single-click*
            //*    in data area, scroll highlight down            *
            //* c) if double-click (or CTRL + single-click) in    *
            //*    data area (or its left/right margins), scroll  *
            //*    to the indicated item and 'select' the item.   *
            else if ( hasFocus && cp->expanded )
            {
               wk.type = wktFUNKEY ;   // navigation keys are function keys
               //* Click in top border *
               if ( multiY == ZERO )
               {
                  wk.key = nckUP ;
               }
               //* Click in bottom border OR in left or right border *
               else if ( (multiY == (this->dCtrl[cIndex]->lines - 1)) ||
                         (wk.mevent.xpos == cxl) || (wk.mevent.xpos == cxr) )
               {
                  wk.key = nckDOWN ;
               }
               //* Click on line item *
               else
               {
                  DialogDropdown* cp = (DialogDropdown*)(dCtrl[cIndex]) ;
                  short selItem = 
                     this->meIdentifyClickItem ( cp->wPtr, (multiY - 1) ) ;
                  if ( (selItem >= ZERO) && (selItem < cp->bItems) )
                  {  //* Move the highlight to the indicated item.*
                     while ( cp->bIndex < selItem )
                        cp->bIndex = cp->wPtr->ScrollData ( nckDOWN ) ;
                     while ( cp->bIndex > selItem )
                        cp->bIndex = cp->wPtr->ScrollData ( nckUP ) ;
                     wk.key = nckENTER ;
                  }
                  else  // this is unlikey, but be safe
                  {
                     wk.type = wktMOUSE ;
                     wk.mevent.conv = false ;
                  }
               }
            }

            //* Otherwise, target does not have focus, AND/OR is  *
            //* not expanded, so hotkey (click) event will give   *
            //* it focus AND expand it.                           *
            else
               this->meRecodeHotkey ( wk ) ;
         }

         //* Target is neither an expanded control, nor obscured  *
         //* by an expanded control.                              *
         else
         {
            //* Binary controls are 'selected' via hotkey (click)    *
            //* regardless of whether the control currently has      *
            //* focus. ScrollWheel events ignored for these controls.*
            if ( cType == dctPUSHBUTTON || cType == dctRADIOBUTTON )
            {
               if ( wk.mevent.meType != metSW_U && wk.mevent.meType != metSW_D )
               {
                  //* Convert the stored hotkey code to user input code.*
                  this->meRecodeHotkey ( wk ) ;
               }
               else  wk.mevent.conv = false ;   // no conversion
            }

            //* Textbox (and Billboard) controls receive focus       *
            //* via hotkey (click). Textbox controls interpret       *
            //* ScrollWheel events as Left/Right navigation keys.    *
            else if ( cType == dctTEXTBOX || cType == dctBILLBOARD )
            {
               if ( wk.mevent.meType == metSW_U || wk.mevent.meType == metSW_D )
               {
                  if ( cType == dctTEXTBOX && hasFocus )
                  {
                     wk.type = wktFUNKEY ;   // navigation keys are function keys
                     if ( wk.mevent.meType == metSW_U )  // ScrollWheel UP
                     {
                        if ( wk.mevent.sKey )   wk.key = nckSRIGHT ;
                        else                    wk.key = nckRIGHT ;
                     }
                     else                                // ScrollWheel DOWN
                     {
                        if ( wk.mevent.sKey )   wk.key = nckSLEFT ;
                        else                    wk.key = nckLEFT ;
                     }
                  }
                  else  wk.mevent.conv = false ;   // no conversion
               }
               else if ( (cType == dctTEXTBOX) && (hasFocus != false) &&
                         ((wk.mevent.meType == metB1_D) ||
                          ((wk.mevent.meType == metB1_S) && wk.mevent.cKey)) )
               {  //* Edit is terminated by a double-click or *
                  //* CTRL + single-click.                    *
                  wk.type = wktFUNKEY ;
                  wk.key  = nckENTER ;
               }
               else     // single-click events (Textbox or Billboard)
               {        // with or without focus
                  this->meRecodeHotkey ( wk ) ;
               }

            }

            if ( cType == dctSCROLLBOX || cType == dctSCROLLEXT )
            {
               //* If target control currently has focus AND event   *
               //* is ScrollWheel, then convert to navigation key.   *
               if ( wk.mevent.meType == metSW_U || wk.mevent.meType == metSW_D )
               {
                  if ( hasFocus )
                  {
                     wk.type = wktFUNKEY ;   // navigation keys are function keys
                     if ( wk.mevent.meType == metSW_U )  // ScrollWheel UP
                        wk.key = nckDOWN ;
                     else                                // ScrollWheel DOWN
                        wk.key = nckUP ;
                  }
                  else  wk.mevent.conv = false ;   // no conversion
               }

               //* If the event type is Click, AND if control has    *
               //* focus, then determine WHERE the click occurred:   *
               //* a) if top border, scroll highlight up             *
               //* b) if bottom border, or unconditioned single-click*
               //*    in data area, scroll highlight down            *
               //* c) if double-click (or CTRL + single-click) in    *
               //*    data area (or its left/right margins), scroll  *
               //*    to the indicated item and 'select' the item.   *
               else if ( hasFocus )
               {
                  wk.type = wktFUNKEY ;   // navigation keys are function keys
                  //* Click in top border *
                  if ( multiY == ZERO )
                  {
                     wk.key = nckUP ;
                  }
                  //* Click in bottom border OR in left or right border *
                  else if ( (multiY == (this->dCtrl[cIndex]->lines - 1)) ||
                            (wk.mevent.xpos == cxl) || (wk.mevent.xpos == cxr) )
                  {
                     wk.key = nckDOWN ;
                  }
                  //* Click on line item *
                  else
                  {  // Programmer's Note: Although both control types have the 
                     // same data-member names, their memory layout is different,
                     // so we must handle them separately.
                     if ( cType == dctSCROLLBOX )
                     {
                        DialogScrollbox* cp = (DialogScrollbox*)(dCtrl[cIndex]) ;
                        short selItem = 
                           this->meIdentifyClickItem ( cp->wPtr, (multiY - 1) ) ;
                        if ( (selItem >= ZERO) && (selItem < cp->bItems) )
                        {  //* Move the highlight to the indicated item.*
                           while ( cp->bIndex < selItem )
                              cp->bIndex = cp->wPtr->ScrollData ( nckDOWN ) ;
                           while ( cp->bIndex > selItem )
                              cp->bIndex = cp->wPtr->ScrollData ( nckUP ) ;
                           wk.key = nckENTER ;  // 'select' key
                        }
                        else  // this is unlikey, but be safe
                        {
                           wk.type = wktMOUSE ;
                           wk.mevent.conv = false ;
                        }
                     }
                     else     // cType == dctSCROLLEXT
                     {
                        DialogScrollext* cp = (DialogScrollext*)(dCtrl[cIndex]) ;
                        short selItem = 
                           this->meIdentifyClickItem ( cp->wPtr, (multiY - 1) ) ;
                        if ( (selItem >= ZERO) && (selItem < cp->bItems) )
                        {  //* Move the highlight to the indicated item.*
                           while ( cp->bIndex < selItem )
                              cp->bIndex = cp->wPtr->ScrollData ( nckDOWN ) ;
                           while ( cp->bIndex > selItem )
                              cp->bIndex = cp->wPtr->ScrollData ( nckUP ) ;
                           wk.key = nckENTER ;  // 'select' key
                        }
                        else  // this is unlikey, but be safe
                        {
                           wk.type = wktMOUSE ;
                           wk.mevent.conv = false ;
                        }
                     }
                  }
               }

               //* Otherwise, target does not have focus, so hotkey  *
               //* (click) event will give it focus.                 *
               else
                  this->meRecodeHotkey ( wk ) ;
            }


            //* Spinner controls receive focus via hotkey (click)    *
            //* OR if they already have focus can be adjusted via    *
            //* ScrollWheel events.                                  *
            else if ( cType == dctSPINNER )
            {
               if ( (wk.mevent.meType == metSW_U ||
                     wk.mevent.meType == metSW_D) )
               {
                  if ( hasFocus )
                  {
                     wk.type = wktFUNKEY ;   // navigation keys are function keys

                     if ( wk.mevent.meType == metSW_U )  // ScrollWheel UP
                     {
                        if ( wk.mevent.sKey )            // ScrollWheel Up + Shift
                           wk.key = nckSUP ;
                        else if ( wk.mevent.cKey )       // ScrollWheel Up + Control
                           wk.key = nckCUP ;
                        else if ( wk.mevent.aKey )       // ScrollWheel Up + Alt
                           wk.key = nckAUP ;
                        else                             // ScrollWheel Up (no modifiers)
                           wk.key = nckUP ;
                     }
                     else                                // ScrollWheel DOWN
                     {
                        if ( wk.mevent.sKey )            // ScrollWheel Down + Shift
                           wk.key = nckSDOWN ;
                        else if ( wk.mevent.cKey )       // ScrollWheel Down + Control
                           wk.key = nckCDOWN ;
                        else if ( wk.mevent.aKey )       // ScrollWheel Down + Alt
                           wk.key = nckADOWN ;
                        else                             // ScrollWheel Down (no modifiers)
                           wk.key = nckDOWN ;
                     }
                  }
                  else
                  { /* Ignore ScrollWheel events over non-focus Spinner */ }
               }

               //* For a Spinner that has focus, a double-click or *
               //* CTRL + single-click terminates the edit         *
               else if ( hasFocus && 
                         ((wk.mevent.meType == metB1_D) ||
                          ((wk.mevent.meType == metB1_S) &&
                           (wk.mevent.cKey != false))) )
               {
                  wk.type = wktFUNKEY ;
                  wk.key  = nckENTER ;
               }
               else
                  this->meRecodeHotkey ( wk ) ;
            }

            //* Slider controls receive focus via hotkey (click)     *
            //* OR if they already have focus can be adjusted via    *
            //* ScrollWheel events.                                  *
            else if ( cType == dctSLIDER )
            {
               if ( (wk.mevent.meType == metSW_U ||
                     wk.mevent.meType == metSW_D) )
               {
                  if ( hasFocus )
                  {
                     wk.type = wktFUNKEY ;   // navigation keys are function keys

                     if ( wk.mevent.meType == metSW_U )  // ScrollWheel UP
                     {
                        if ( wk.mevent.sKey )            // ScrollWheel Up + Shift
                           wk.key = nckPGUP ;            //  becomes PageUp key
                        else if ( wk.mevent.cKey ||      // ScrollWheel Up + Control
                                  wk.mevent.aKey )       // ScrollWheel Up + Alt
                           wk.key = nckHOME ;            //  become Home key
                        else                             // ScrollWheel Up (no modifiers)
                           wk.key = nckUP ;              //  becomes UpArrow key
                     }
                     else                                // ScrollWheel DOWN
                     {
                        if ( wk.mevent.sKey )            // ScrollWheel Down + Shift
                           wk.key = nckPGDOWN ;          //  becomes PageDown key
                        else if ( wk.mevent.cKey ||      // ScrollWheel Down + Control
                                  wk.mevent.aKey )       // ScrollWheel Down + Alt
                           wk.key = nckEND ;             //  become End key
                        else                             // ScrollWheel Down (no modifiers)
                           wk.key = nckDOWN ;            //  becomes DownArrow key
                     }
                  }
                  else
                  { /* Ignore ScrollWheel events over non-focus Slider */ }
               }

               //* For a Slider that has focus, a double-click or  *
               //* CTRL + single-click terminates the edit         *
               else if ( hasFocus && 
                         ((wk.mevent.meType == metB1_D) ||
                          ((wk.mevent.meType == metB1_S) &&
                           (wk.mevent.cKey != false))) )
               {
                  wk.type = wktFUNKEY ;
                  wk.key  = nckENTER ;
               }
               else
                  this->meRecodeHotkey ( wk ) ;
            }
         }
      }

      //* Event is not associated with a control object.       *
      //* If there is an expanded control in the dialog, then  *
      //* a Click event is a signal to collapse it and move on.*
      else if ( expCtrl >= ZERO )
      {
         if ( wk.mevent.meType != metSW_U && wk.mevent.meType != metSW_D )
         {
            wk.key = nckESC ;
            wk.type = wktFUNKEY ;
            wk.mevent.conv = true ;
         }
         else  wk.mevent.conv = false ;   // no conversion
      }
   }              // isEvent && dialogEvent && convertEvent && meAuto
   return status ;

}  //* End meTransformEvent() *

//**************************
//*     meRecodeHotkey     *
//**************************
//******************************************************************************
//* PRIVATE METHOD: Called only by meTransformEvent.                           *
//* Extract a control's stored hotkey character and convert it to an ASCII     *
//* control code OR the 'default hotkey' pseudocharacter.                      *
//*                                                                            *
//* Input  : wk : key/mouse data received from input stream                    *
//*               (wk.mevent.cIndex == target control)                         *
//*                                                                            *
//* Returns: nothing                                                           *
//*          (wk.type and wk.key received the recoded hotkey data)             *
//******************************************************************************

void NcDialog::meRecodeHotkey ( wkeyCode& wk )
{
   //* Get the stored code *
   wchar_t hk = this->dCtrl[wk.mevent.cIndex]->bHotkey.hotchar ;

   //* If a user-defined code, convert to ASCII control code.*
   if ( this->dCtrl[wk.mevent.cIndex]->bHotkey.hotkey != false )
   {
      wk.type = wktFUNKEY ;
      wk.key  = (hk & 0x001F) ;
   }
   //* Else, is a non-character, 'default hotkey' *
   else
   {
      wk.type = wktMOUSE ;
      wk.key  = hk ;
   }

}  //* End meRecodeHotkey() *

//*************************
//*     meIsObscured      *
//*************************
//******************************************************************************
//* Determine whether the specified event position targets a control object    *
//* which is obscured by an expanded Menuwin object.                           *
//*                                                                            *
//* Called ONLY by meTransformEvent(), and ONLY IF:                            *
//* 1) Mouse event occurred above an active control object                     *
//* 2) One or more Menuwin controls are currently in an 'expanded' state       *
//* 3) Mouse event does not lie within the expanded Menuwin control which      *
//*    has the input focus.                                                    *
//*                                                                            *
//*         This is an expensive test. Don't call it unnecessarily!            *
//*                                                                            *
//* Input  : trgPoint : mouse-event data                                       *
//*                                                                            *
//* Returns: 'true' if specified point lies within an expanded Menuwin object  *
//*          else 'false'                                                      *
//******************************************************************************
//* For each 'expanded' Menuwin control, determine whether the specified       *
//* mouse event lies within its space. If so, then the targeted control which  *
//* lies beneath is not accessible via mouse event.                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//******************************************************************************

bool NcDialog::meIsObscured ( wkeyCode trgPoint )
{
   short cyt,        // control's Top line
         cyb,        // control's Bottom line
         cxl,        // control's Left column
         cxr ;       // control's Right column
   bool obs = false ;// return value
   for ( short ci = ZERO ; ci <= this->lastCtrl ; ci++ )
   {
      if ( this->dCtrl[ci]->type == dctMENUWIN )
      {
         DialogMenuwin* cp = (DialogMenuwin*)(this->dCtrl[ci]) ;
         if ( cp->expanded != false )
         {
            cyt = cp->tulY - this->dulY ;
            cyb = cyt + cp->lines ;
            if ( cp->tCols == ZERO )   // submenus have no title
               --cyb ;
            cxl = cp->ulX - this->dulX ;
            if ( trgPoint.mevent.ypos == cyt && cp->tCols > ZERO )
               cxr = cxl + cp->tCols - 1 ;
            else
               cxr = cxl + cp->cols - 1 ;
            if ( trgPoint.mevent.ypos >= cyt && trgPoint.mevent.ypos <= cyb &&
                 trgPoint.mevent.xpos >= cxl && trgPoint.mevent.xpos <= cxr )
            {
               obs = true ;
               break ;
            }
         }
      }
   }
   return obs ;

}  //* End meIsObscured() *

//** - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -**
//** - - - - - - - - - - End Mouse-support Methods - - - - - - - - - - - - - -**
//** - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -**

//*************************
//*      UserAlert        *
//*************************
//******************************************************************************
//* Sound an audible alert to attract user's attention.                        *
//* If setup parameters are provided the alert will follow the specified ping  *
//* sequence until the terminal count is reached OR until user presses any key.*
//* If no parameters are provided, then a single ping will be generated,       *
//* followed by an immediate return.                                           *
//*                                                                            *
//* Note that if user presses a key during the sequence, the key-input buffer  *
//* will be flushed before returning; however, a caffinated user may manage to *
//* press a key we don't see, so caller should check the key-input buffer on   *
//* return.                                                                    *
//*                                                                            *
//* Input  : pingParms : (optional, NULL pointer by default)                   *
//*                      If specified, provides parameters for a ping sequence.*
//*                      If not specified, a single ping is sounded.           *
//*                                                                            *
//* Returns: OK  if specified operation is supported by the system             *
//*          ERR if alert not supported                                        *
//******************************************************************************
//* Note that we support only the 'beep', not the 'flash' functionality;       *
//* however, the way the ncurses implements these is that if the one which is  *
//* called is not supported by the system, then the other will be called.      *
//* If neither is supported (unlikely), then nothing happens.                  *
//*                                                                            *
//* NOTE: A very small value for 'pInterval' may not be recognized. See range  *
//*       checking below which stays within the capabilities of most systems.  *
//*                                                                            *
//******************************************************************************

short NcDialog::UserAlert ( AudibleAlert* pingParms )
{
   short status = ERR ;    // return value
   if ( pingParms != NULL )
   {
      const short intervalRESOLUTION = 50 ;     // milliseconds
      chrono::duration<short, std::milli>delayResolution( 100 ) ;

      //* Range checking *
      if ( pingParms->pCount <= ZERO )    pingParms->pCount = 1 ;
      if ( pingParms->pInterval < minUAINTERVAL )
         pingParms->pInterval = minUAINTERVAL ;
      if ( pingParms->pRepeat < ZERO )    pingParms->pRepeat = ZERO ;
      if ( pingParms->pDelay < ZERO )     pingParms->pDelay = ZERO ;

      short repeat = pingParms->pRepeat >= 1 ? pingParms->pRepeat : 1 ;
      int interval = pingParms->pInterval * intervalRESOLUTION ;
      short iIndex = ZERO ;
      wkeyCode wk ;
      chrono::duration<short, std::milli>intervalTimer( interval ) ;

      //* Outer loop is sequence repeat count *
      for ( ; repeat > ZERO ; repeat-- )
      {
         //* Inner loop executes the sequence or pattern *
         iIndex = ZERO ;
         for ( short i = pingParms->pCount ; i > ZERO ; i-- )
         {
            status = nc.UserAlert () ;       // make some noise

            if ( i > 1 )      // if not on last iteration, take a nap
            {
               if ( pingParms->pPattern != NULL )
               {
                  int pat = pingParms->pPattern[iIndex++] * intervalRESOLUTION ;
                  chrono::duration<short, std::milli>patternTimer( pat ) ;
                  this_thread::sleep_for( patternTimer ) ;
               }
               else if ( pingParms->pInterval > ZERO )
                  this_thread::sleep_for( intervalTimer ) ;
            }
         }        // inner for(;;)

         //* Inter-sequence delay *
         for ( short i = pingParms->pDelay ; i > ZERO ; i-- )
         {
            this_thread::sleep_for( delayResolution ) ;
         }

         //* Test for user interference *
         if ( ((repeat > 1) || pingParms->pRepeat == -1) &&
              ((this->KeyPeek ( wk )) != wktERR) )
         {
            this->FlushKeyInputStream () ;
            break ;        // break out of outer loop
         }

         //* If waiting for a keypress, keep the loop going.*
         if ( pingParms->pRepeat == -1 )
            ++repeat ;
      }           // outer for(;;)
   }
   else
      status = nc.UserAlert () ;
   return status ;

}  //* End UserAlert() *

//*************************
//*      UserAlert        *
//*************************
//******************************************************************************
//* Sound an audible alert to attract user's attention.                        *
//*                                                                            *
//* This is a simplified version of UserAlert which does not require full      *
//* initialization. Instead of receiving the parameters in an AudibleAlert     *
//* object, this method takes only a ping count and an inter-ping delay.       *
//*                                                                            *
//*                                                                            *
//* Input  : count    : number of pings                                        *
//*          interval : (optional) delay between pings                         *
//*                     - if specified, then this is the inter-ping delay      *
//*                       in 20ths of a second                                 *
//*                     - otherwise delay is set to approximately 0.2 seconds  *
//*                       which is a short, but reliable interval              *
//*                                                                            *
//* Returns: OK  if specified operation is supported by the system             *
//*          ERR if alert not supported                                        *
//******************************************************************************

short NcDialog::UserAlert ( short count, short interval )
{
   if ( count < 1 )                          // range checking
      count = 1 ;
   if ( interval < minUAINTERVAL )
      interval = minUAINTERVAL * 2 ;

   AudibleAlert aa = { count, interval } ;   // initialization

   short status = this->UserAlert ( &aa ) ;  // call the mother ship

   return status ;

}  //* End UserAlert() *

//*************************
//*       ShellOut        *
//*************************
//******************************************************************************
//* Go to command shell.                                                       *
//*   1) save the dialog display data                                          *
//*   2) put the NCurses Engine to sleep                                       *
//*   3) clear terminal screen if specified                                    *
//*   4) display 'exit-to-return' message if specified                         *
//*   5) invoke 'bash' or the specified built-in or utility                    *
//*   6) on return:                                                            *
//*      a) display 'press-Enter-key' message if specified                     *
//*      b) pause for Enter key if specified                                   *
//*      c) wake the NCurses Engine                                            *
//*      d) restore the dialog window                                          *
//*                                                                            *
//* Important Note:                                                            *
//*      This method should be called ONLY from the primary dialog,            *
//*      AND there should be no sub-dialogs open.                              *
//*      If invoked from a sub-dialog, then the primary dialog's               *
//*      display data will not be properly saved.                              *
//*                                                                            *
//* Input  : option   : member of enum soOptions (optional, soE by default)    *
//*          shellCmd : (optional, NULL pointer by default)                    *
//*                     - if specified, invoke the specified shell builtin     *
//*                       command, shell program or console utility program    *
//*                     - if NULL pointer, then invoke a copy of 'bash' shell  *
//*                       NOTE: If your default shell IS NOT 'bash', then you  *
//*                       should use this parameter to specify your shell      *
//*                       program in 'shellCmd'.                               *
//*          textColor: (optional, NULL pointer i.e. Black text by default)    *
//*                     ANSI escape sequence which specifies the color of text *
//*                     written to standard output.                            *
//*                     (If your terminal does not support ANSI colors,)       *
//*                     (this will have no effect.                     )       *
//*                                                                            *
//* Returns: exit code of child process                                        *
//*           Ideally, this is the exit code of the last utility run in the    *
//*           child process, but this cannot be guaranteed.                    *
//******************************************************************************
//* ANSI Colors (partial list):                                                *
//* =========== Foreground  Background   ============ Foreground  Background   *
//* Black       \033[0;30m  \033[0;40m   Dark Gray    \033[1;30m  \033[1;40m   *
//* Red         \033[0;31m  \033[0;41m   Bold Red     \033[1;31m  \033[1;41m   *
//* Green       \033[0;32m  \033[0;42m   Bold Green   \033[1;32m  \033[1;42m   *
//* Yellow      \033[0;33m  \033[0;43m   Bold Yellow  \033[1;33m  \033[1;43m   *
//* Blue        \033[0;34m  \033[0;44m   Bold Blue    \033[1;34m  \033[1;44m   *
//* Purple      \033[0;35m  \033[0;45m   Bold Purple  \033[1;35m  \033[1;45m   *
//* Cyan        \033[0;36m  \033[0;46m   Bold Cyan    \033[1;36m  \033[1;46m   *
//* Light Gray  \033[0;37m  \033[0;47m   White        \033[1;37m  \033[1;47m   *
//*                                                                            *
//* We assume that the existing foreground/background is Black text on a White *
//* background. If this is not the case, then this method could cause color    *
//* artifacts. (Querying termcap for the actual color is too much work. :)     *
//*                                                                            *
//* Although 'textColor' may specify both foreground and background color, a   *
//* non-default background color may not be handled uniformly by the terminal. *
//* For this reason, it is recommended that only a foreground color be         *
//* specified.                                                                 *
//*                                                                            *
//*  Note that if 'shellCmd' specifies a shell builtin command, and if         *
//* 'textColor' is specified, then the command will _probably_ use the         *
//*  specified color, but this cannot be guaranteed.                           *
//*         dp->ShellOut ( "ls", true, "\033[1;31m" ) ;                        *
//*         dp->ShellOut ( "cat Makefile", true, "\033[1;31m" ) ;              *
//*  Console utilities are more independent and will _probably not_ use the    *
//*  specified color.                                                          *
//*         dp->ShellOut ( "less Makefile", true, "\033[1;31m" ) ;             *
//*                                                                            *
//* Note that if the 'shellCmd' is not specified OR is the name of a shell     *
//* program, then a pause should not be specified. It would do no harm, but    *
//* would make you look foolish.                                               *
//*                                                                            *
//* Note that under rare circumstances, when you exit the dialog application,  *
//* the cursor will retain the color (if any) you specified with 'textColor'.  *
//*                                                                            *
//******************************************************************************

short NcDialog::ShellOut ( soOptions option, const char* shellCmd, const char* textColor )
{
   const char* shellName = "bash" ;          // default external command
   const char* txtAttr   = "\033[0;34m" ;    // blue text on default background
   const char* fgndDflt  = "\033[0;30m" ;    // default (black) foreground

   short status = ZERO ;            // return value

   this->SetDialogObscured () ;
   this->Hibernate () ;

   gString gs( shellName ) ;        // set the external command string
   if ( shellCmd != NULL )
      gs = shellCmd ;

   //* Determine:
   //*  a) whether to display an initial message
   //*  b) whether to pause before return from shell
   //*  c) whether to display a message on pause
   //*  d) whether to clear screen before launch of shell command
   //* Default: soE
   //* (Note: additional options may be added at a later time.)
   bool emsg  = true,      // entry message
        pause = false,     // pause after execution
        pmsg  = true,      // pause message
        clear = false ;    // clear screen
   switch ( option )
   {
      case soX:
         emsg = pmsg = false  ;
         break ;
      case soP:
         emsg = false ;
         // (fall through)
      case soEP:
         pause = true ;
         break ;
      case soPN:
      case soEPN:
         pause = true ;
         pmsg  = false ;
         break ;
      case soCX:
         clear = true ;
         emsg  = false ;
         break ;
      case soCE:
         clear = true ;
         break ;
      case soCP:
         clear = true ;
         pause = true ;
         emsg  = false ;
         break ;
      case soCEP:
         clear = true ;
         pause = true ;
         break ;
      case soCPN:
         clear = true ;
         pause = true ;
         pmsg  = false ;
         emsg  = false ;
         break ;
      case soCEPN:
         clear = true ;
         pause = true ;
         pmsg  = false ;
         break ;
      case soE:
      default:
         break ;
   } ;

   if ( textColor != NULL )         // if specified, use caller's text color
      wcout << " \n" << textColor << flush ;
   else if ( emsg )                 // if color not specified AND message, use Blue
      wcout << " \n" << txtAttr << flush ;

   if ( clear )                     // if screen to be cleared
      system ( "clear" ) ;
   else if ( emsg )                 // else, just insert some space
      wcout << '\n' << flush ;
   if ( emsg )                      // if 'type-Exit-to-return' message specified
   {
      wcout << "*********************\n"
               "    Command Shell    \n"
               "Type 'exit' to return\n"
               "*********************" ;
      if ( textColor == NULL )      // if color not specified
         wcout << fgndDflt ;        // return to (assumed) default color
      wcout << endl ;
   }

   #if 0    // DEBUGGING ONLY
   system ( "ps" ) ;
   #endif   // DEBUGGING ONLY

   //* Invoke the external application or shell program *
   status = system ( gs.ustr() ) ;

   if ( pause )
   {
      if ( pmsg )
      {
         if ( textColor == NULL )
            wcout << txtAttr ;
         wcout << "\nPress ENTER key..." << flush ;
      }
      getch();
   }

   if ( textColor != NULL || emsg || (pause && pmsg) )
      wcout << fgndDflt ;           // return to (assumed) default color
   wcout << endl ;

   this->Wake () ;
   this->RefreshWin () ;

   return status ;

}  //* End ShellOut() *

//*************************
//*      PseudoShell      *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  : ?                                                                 *
//*                                                                            *
//* Returns: ?                                                                 *
//******************************************************************************

void NcDialog::PseudoShell ( void )
{
}  //* End PseudoShell() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

