//******************************************************************************
//* File       : ColorTest.cpp                                                 *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2013-2025 Mahlon R. Smith, The Software Samurai *
//*                  GNU GPL copyright notice located in NCurses.hpp           *
//* Date       : 21-Mar-2025                                                   *
//* Version    : (see below)                                                   *
//*                                                                            *
//* Description: ColorTest is a simple class definition for exercising the     *
//*              NCurses/NcDialog color-mapping methods.                       *
//*                                                                            *
//* Development Tools: See NcDialog.cpp.                                       *
//******************************************************************************
//* Version History (most recent first):                                       *
//*                                                                            *
//* v: 0.00.01 01-Jun-2013 Transfer functionality from Dialog2, Test04.        *
//*                        Move some controls to allow for a smaller dialog    *
//*                         window, and enhance support for >8 RGB registers.  *
//******************************************************************************
//* Programmer's Note:                                                         *
//* Setting non-default values for Color Pair ZERO, can have unexpected        *
//* results. Please see NCurses-class source code, StartColorEngine() method   *
//* for additional notes or setting color-pair values.                         *
//******************************************************************************

#include "ColorTest.hpp"

//****************
//** Local Data **
//****************
static const short MAX_PATH = gsDFLTBYTES ;  // This is the POSIX PATH_MAX
typedef struct tm Tm ;                       // Linux Time structure

//* Constant text *
//* RGB descriptons for selecting the RGB color set to be modified *
static const short rgbDescWIDTH = 18 ;
static const char rgbDescText[ncCOLOR_MAX * 2 - 1][rgbDescWIDTH - 1] = 
{
   " Modify BLACK   ", "      ----      ",
   " Modify RED     ", "      ----      ",
   " Modify GREEN   ", "      ----      ",
   " Modify BROWN   ", "      ----      ",
   " Modify BLUE    ", "      ----      ",
   " Modify MAGENTA ", "      ----      ",
   " Modify CYAN    ", "      ----      ",
   " Modify GREY    ", "      ----      ",
   " Mod BRT BLACK  ", "      ----      ",
   " Mod BRT RED    ", "      ----      ",
   " Mod BRT GREEN  ", "      ----      ",
   " Mod BRT BROWN  ", "      ----      ",
   " Mod BRT BLUE   ", "      ----      ",
   " Mod BRT MAGENTA", "      ----      ",
   " Mod BRT CYAN   ", "      ----      ",
   " Mod BRT GREY   ",
} ;
static attr_t rgbDescAttr[ncCOLOR_MAX * 2 - 1] ; 
static bool rgbDescFlags[ncCOLOR_MAX * 2 - 1] =
{
   true, false, true, false, true, false, true, false, 
   true, false, true, false, true, false, true, false, 
   true, false, true, false, true, false, true, false, 
   true, false, true, false, true, false, true 
} ;
//* 'Set Default Background' menu data *
static const short bkgMenuWIDTH = 18, bkgMenuITEMS = 17 ;
static const char bkgMenuText[bkgMenuITEMS][bkgMenuWIDTH - 1] = 
{
   " -1 Terminal    ",
   " 00-Black       ",
   " 01-Red         ",
   " 02-Green       ",
   " 03-Brown       ",
   " 04-Blue        ",
   " 05-Magenta     ",
   " 06-Cyan        ",
   " 07-Grey        ",
   " 08-BrightBlack ",
   " 09-BrightRed   ",
   " 10-BrightGreen ",
   " 11-BrightBrown ",
   " 12-BrightBlue  ",
   " 13-Brt Magenta ",
   " 14-BrightCyan  ",
   " 15-BrightGrey  ",
} ;
static attr_t bkgMenuAttr[bkgMenuITEMS] ; 


static const short dialogROWS = ctdROWS ; // display lines
static const short dialogCOLS = ctdCOLS ; // display columns
static const short sampleGROUP = 8 ;      // number of color-pair samples per group
static dspinData dsData ;                 // data for the ctMprSP spinner control
static winPos cinfoPos( 20, 2 ) ;         // positioning of controls within dialog


static InitCtrl ic[ctControlsDEFINED] =   // control initialization structures
{
   {  //* 'NEXT GROUP' pushbutton - - - - - - - - - - - - - - - - -    ctNxtPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 5),        // ulY:       upper left corner in Y
      short(dialogCOLS / 2 - 18),   // ulX:       upper left corner in X
      1,                            // lines:     control lines
      16,                           // cols:      control columns
      " Next SampleGrp ",           // dispText:  
      nc.gyR,                       // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[ctPrvPB],                 // nextCtrl:  link in next structure
   },
   {  //* 'PREV GROUP' pushbutton - - - - - - - - - - - - - - - - -    ctPrvPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[ctNxtPB].ulY,              // ulY:       upper left corner in Y
      short(dialogCOLS / 2 + 2),    // ulX:       upper left corner in X
      1,                            // lines:     control lines
      16,                           // cols:      control columns
      " Prev SampleGrp ",           // dispText:  
      nc.gyR,                       // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[ctMprSP],                 // nextCtrl:  link in next structure
   },
   {  //* 'PAIR SELECT' spinner - - - - - - - - - - - - - - - - - -    ctMprSP *
      dctSPINNER,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[ctNxtPB].ulY + 2),   // ulY:       upper left corner in Y
      short(ic[ctNxtPB].ulX + 6),   // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      4,                            // cols:      control columns
      NULL,                         // dispText:  (n/a)
      nc.gyR,                       // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Select Color Pair",          // label:     
      1,                            // labY:      
      -6,                           // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      &dsData,                      // spinData:  spinner init
      true,                         // active:    allow control to gain focus
      &ic[ctMprPB],                 // nextCtrl:  link in next structure
   },
   {  //* 'MODIFY PAIR' pushbutton  - - - - - - - - - - - - - - - -    ctMprPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[ctMprSP].ulY),       // ulY:       upper left corner in Y
      short(ic[ctMprSP].ulX + 15),  // ulX:       upper left corner in X
      1,                            // lines:     control lines
      13,                           // cols:      control columns
      " MODIFY PAIR ",              // dispText:  
      nc.gyR,                       // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[ctRgbMW],                 // nextCtrl:  link in next structure
   },
   {  //* 'RGB selector' Menuwin  - - - - - - - - - - - - - - - - - -  ctRgbMW *
      dctMENUWIN,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      1,                            // ulY: upper left corner in Y
      25,                           // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      rgbDescWIDTH,                 // cols:      control columns, 'expanded' state
      (const char*)rgbDescText,     // dispText:  array of menu items
      nc.reG,        // nColor: non-focus border color / focus 'collapsed' color
      nc.gyR,        // fColor: focus border color / non-focus 'collapsed' color
      tbPrint,                      // filter:    (n/a)
      " Modify RGB Set ",           // label:     label text == 'collapsed state' text
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      short(ncCOLOR_MAX * 2 - 1),   // scrItems:  number of menu items
      ZERO,                         // scrSel:    (n/a)
      (attr_t*)rgbDescAttr,         // scrColor:  array of color attributes for menu items
      NULL,                         // spinData:  (n/a)
      true,                         // active:    initially inactive
      &ic[ctBkgMW],                 // nextCtrl:  link in next structure
   },
   {  //* 'Background' Menuwin    - - - - - - - - - - - - - - - - - -  ctBkgMW *
      dctMENUWIN,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[ctRgbMW].ulY + 2),   // ulY:       upper left corner in Y
      short(ic[ctRgbMW].ulX),       // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      bkgMenuWIDTH,                 // cols:      control columns, 'expanded' state
      (const char*)bkgMenuText,     // dispText:  array of menu items
      nc.reG,        // nColor: non-focus border color / focus 'collapsed' color
      nc.gyR,        // fColor: focus border color / non-focus 'collapsed' color
      tbPrint,                      // filter:    (n/a)
      " Set Dflt Bkgnd ",           // label:     label text == 'collapsed state' text
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      bkgMenuITEMS,                 // scrItems:  number of menu items
      ZERO,                         // scrSel:    (n/a)
      (attr_t*)bkgMenuAttr,         // scrColor:  array of color attributes for menu items
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[ctnppPB],                 // nextCtrl:  link in next structure
   },
   {  //* 'NEXT PAIR PAGE' pushbutton   - - - - - - - - - - - - - -    ctnppPB *
      dctPUSHBUTTON,                // type:
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[ctBkgMW].ulY + 2),   // ulY:       upper left corner in Y
      short(ic[ctBkgMW].ulX),       // ulX:       upper left corner in X
      1,                            // lines:     control lines
      short(rgbDescWIDTH - 2),      // cols:      control columns
      " NEXT PAIR PAGE ",           // dispText:  
      nc.grR,                       // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[ctRstPB],                 // nextCtrl:  link in next structure
   },
   {  //* 'RESET ALL' pushbutton  - - - - - - - - - - - - - - - - -    ctRstPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[ctnppPB].ulY + 2),   // ulY:       upper left corner in Y
      ic[ctnppPB].ulX,              // ulX:       upper left corner in X
      1,                            // lines:     control lines
      short(rgbDescWIDTH - 2),      // cols:      control columns
      "   RESET ALL    ",           // dispText:  
      nc.grR,                       // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[ctRddPB],                 // nextCtrl:  link in next structure
   },
   {  //* 'READ' pushbutton - - - - - - - - - - - - - - - - - - - -    ctRddPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[ctRstPB].ulY + 2),   // ulY:       upper left corner in Y
      ic[ctRstPB].ulX,              // ulX:       upper left corner in X
      1,                            // lines:     control lines
      short(rgbDescWIDTH - 2),      // cols:      control columns
      " READ FROM FILE ",           // dispText:  
      nc.grR,                       // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[ctSavPB],                 // nextCtrl:  link in next structure
   },
   {  //* 'SAVE' pushbutton     - - - - - - - - - - - - - - - - - -    ctSavPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[ctRddPB].ulY + 2),   // ulY:       upper left corner in Y
      ic[ctRddPB].ulX,              // ulX:       upper left corner in X
      1,                            // lines:     control lines
      short(rgbDescWIDTH - 2),      // cols:      control columns
      "  SAVE TO FILE  ",           // dispText:  
      nc.grR,                       // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[ctDonePB],                // nextCtrl:  link in next structure
   },
   {  //* 'DONE' pushbutton - - - - - - - - - - - - - - - - - - -     ctDonePB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[ctSavPB].ulY + 2),   // ulY:       upper left corner in Y
      short(ic[ctSavPB].ulX + 5),   // ulX:       upper left corner in X
      1,                            // lines:     control lines
      8,                            // cols:      control columns
      "  DONE  ",                   // dispText:  
      nc.gyR,                       // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   },
} ;   // InitCtrl array


//*************************
//*      ~ColorTest       *
//*************************
//******************************************************************************
//* Destructor.                                                                *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

ColorTest::~ColorTest ( void )
{

   if ( this->dp != NULL )                // close the window
      delete ( this->dp ) ;

}  //* End ~ColorTest() 

//*************************
//*       ColorTest       *
//*************************
//******************************************************************************
//* Constructor.                                                               *
//*                                                                            *
//* Input  : tLines     : number of display line in terminal window            *
//*          tCols      : number of display columns in terminal window         *
//*          minY       : first available display line                         *
//*                                                                            *
//* Returns: implicitly return class reference                                 *
//******************************************************************************

ColorTest::ColorTest ( short tLines, short tCols, short minY )
{
   //* Initialize data members *
   this->termRows = tLines ;
   this->termCols = tCols ;
   this->minULY   = minY ;
   this->dp = NULL ;
   this->dColor = nc.cyR ;
   this->firstSample = ZERO ;
   this->firstPIndex = ZERO ;
   this->ctOpen = false ;

   //* Initialize remainder of control-definition array.           *
   //* (colors become available after NCurses engine instantiated) *
   ic[ctDonePB].nColor = nc.gyR ;
   ic[ctDonePB].fColor = nc.reG ;
   ic[ctRgbMW].nColor = nc.reG ;
   ic[ctRgbMW].fColor = nc.gyR ;
   ic[ctBkgMW].nColor = nc.reG ;
   ic[ctBkgMW].fColor = nc.gyR ;
   ic[ctNxtPB].nColor = nc.gyR ;
   ic[ctNxtPB].fColor = nc.reG ;
   ic[ctPrvPB].nColor = nc.gyR ;
   ic[ctPrvPB].fColor = nc.reG ;
   ic[ctMprSP].nColor = nc.gyR ;
   ic[ctMprSP].fColor = nc.bw  ;
   ic[ctMprPB].nColor = nc.gyR ;
   ic[ctMprPB].fColor = nc.reG ;
   ic[ctnppPB].nColor = nc.grR ;
   ic[ctnppPB].fColor = nc.reG ;
   ic[ctRstPB].nColor = nc.grR ;
   ic[ctRstPB].fColor = nc.reG ;
   ic[ctRddPB].nColor = nc.grR ;
   ic[ctRddPB].fColor = nc.reG ;
   ic[ctSavPB].nColor = nc.grR ;
   ic[ctSavPB].fColor = nc.reG ;

   //* Get a copy of the terminal's color-map data *
   this->ctGetCMap () ;
   this->origCM = this->colorMap ;  // save a copy of the original settings

   //* Adjust the number of items in the RGB selector *
   //* array and in the background selector menu.     *
   if ( this->colorMap.rgbCount < ncCOLOR_MAX )
   {
      ic[ctRgbMW].scrItems = this->colorMap.rgbCount * 2 ;
      ic[ctBkgMW].scrItems = this->colorMap.rgbCount + 1 ;
   }

   //* Adjust colors for menu-window data *
   //* RGB descriptions *
   short items = (ncCOLOR_MAX * 2 - 1) ;
   for ( short i = ZERO ; i < items ; i++ )
   {
      if ( (i % 2) > ZERO )         // dividing spaces
         rgbDescAttr[i] = nc.blR ;
      else                          // selectable items
      {
         switch ( i )
         {
            case 0:  rgbDescAttr[i] = nc.bwR ;  break ;  // Black
            case 2:  rgbDescAttr[i] = nc.reR ;  break ;  // Red
            case 4:  rgbDescAttr[i] = nc.grR ;  break ;  // Green
            case 6:  rgbDescAttr[i] = nc.brR ;  break ;  // Brown
            case 8:  rgbDescAttr[i] = nc.blR ;  break ;  // Blue
            case 10: rgbDescAttr[i] = nc.maR ;  break ;  // Magenta
            case 12: rgbDescAttr[i] = nc.cyR ;  break ;  // Cyan
            case 14: rgbDescAttr[i] = nc.gyR ;  break ;  // Grey
            case 16: rgbDescAttr[i] = nc.bbk | ncrATTR ;  break ;  // Bright Black
            case 18: rgbDescAttr[i] = nc.bre | ncrATTR ;  break ;  // Bright Red
            case 20: rgbDescAttr[i] = nc.bgr | ncrATTR ;  break ;  // Bright Green
            case 22: rgbDescAttr[i] = nc.bbr | ncrATTR ;  break ;  // Bright Brown (yellow)
            case 24: rgbDescAttr[i] = nc.bbl | ncrATTR ;  break ;  // Bright Blue
            case 26: rgbDescAttr[i] = nc.bma | ncrATTR ;  break ;  // Bright Magenta
            case 28: rgbDescAttr[i] = nc.bcy | ncrATTR ;  break ;  // Bright Cyan
            case 30: rgbDescAttr[i] = nc.gy ;             break ;  // Bright Grey (nc.bgy)
         }
      }
   }

   //* Background Color descriptions *
   //* (background 8-15 currently not supported by ncurses library)
   bkgMenuAttr[0] = nc.bw ;
   bkgMenuAttr[1] = (nc.bk & ~ncrATTR) ;
   bkgMenuAttr[2] = nc.re ;
   bkgMenuAttr[3] = nc.gr ;
   bkgMenuAttr[4] = nc.br ;
   bkgMenuAttr[5] = nc.bl ;
   bkgMenuAttr[6] = nc.ma ;
   bkgMenuAttr[7] = nc.cy ;
   bkgMenuAttr[8] = nc.gy ; 
   bkgMenuAttr[9]  = nc.bbk ;
   bkgMenuAttr[10] = nc.bre ;
   bkgMenuAttr[11] = nc.bgr ;
   bkgMenuAttr[12] = nc.bbr ; // (yellow)
   bkgMenuAttr[13] = nc.bbl ;
   bkgMenuAttr[14] = nc.bma ;
   bkgMenuAttr[15] = nc.bcy ;
   bkgMenuAttr[16] = nc.gy ;  // (nc.bgy == invisible)

   //* Initialize the dctSPINNER control data *
   dsData.minValue = ZERO ;
   dsData.maxValue = this->colorMap.pairCount - 1 ;
   dsData.iniValue = ZERO ;
   dsData.dsFormat = dspinINTEGER ;
   dsData.indiAttr = nc.grG ;


   if ( (this->ctOpen = this->ctOpenDialog ()) != false )
   {

   }  // OpenWindow()

}  //* End ColorTest() 

//*************************
//*     ctOpenDialog      *
//*************************
//******************************************************************************
//* Open the dialog window.                                                    *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if dialog window opened successfully, else 'false'         *
//******************************************************************************

bool ColorTest::ctOpenDialog ( void )
{
   short ctrY    = this->termRows/2 + 1,     // dialog center in Y
         ctrX    = this->termCols/2,         // dialog center in X
         //* Upper left corner in Y (cannot obscure status window) *
         ulY     = (ctrY - dialogROWS/2) >= this->minULY ? 
                   (ctrY - dialogROWS/2) : this->minULY,
         ulX     = ctrX - dialogCOLS / 2 ;   // upper left corner in X
   bool  success = false ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       "    Exercise the NcDialog Color Map     ", // dialog title
                       ncltSINGLE,     // border line-style
                       this->dColor,   // border color attribute
                       this->dColor,   // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   this->dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (this->dp->OpenWindow()) == OK )
   {
      //* Set the 'inactive' items in the RGB selector menu.*
      this->dp->SetActiveMenuItems ( ctRgbMW, rgbDescFlags ) ;

      //* Report system capabilities *
      bool  colorCapable = nc.ColorText_Available () ;
      const wchar_t* coText = colorCapable ? L"Yes" : L"No " ;
      const char* const sysTitle = " System Info " ;
      gString gsOut ;
      cinfoPos = { short(ic[ctDonePB].ulY + 2), ic[ctRgbMW].ulX } ; 
      this->dp->DrawBox ( cinfoPos.ypos, cinfoPos.xpos, 7, 19, 
                          nc.brcy, sysTitle, ncltDUAL ) ;
      gsOut.compose( L"                 \n"
                      "Color Output :%S\n"
                      "RGB Registers:%3hd\n"
                      "maxColorPairs:%3hd\n"
                      "                 ", 
                     coText, &this->colorMap.rgbCount, &this->colorMap.pairCount ) ;
      this->dp->WriteParagraph ( cinfoPos.ypos + 1, cinfoPos.xpos + 1, 
                                 gsOut, this->dColor | ncbATTR ) ;

      this->ctDisplayRGBvalues () ;    // display RGB register values

      this->ctDisplayColorPairs () ;   // display color-pair values

      //* Display sample text strings for each color-modification attribute *
      this->ctDisplaySampleText () ;

      this->dp->PrevControl () ;       // 'Done' button has initial focus

      this->dp->RefreshWin () ;        // make everything visible
      success = true ;
   }
   return success ;

}  //* End ctOpenDialog() *

//*************************
//*    ctDialogOpened     *
//*************************
//******************************************************************************
//* Satisfy caller's curiosity.                                                *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if dialog opened successfully, else 'false'                *
//******************************************************************************

bool ColorTest::ctDialogOpened ( void )
{
   return this->ctOpen ;

}  //* End ctDialogOpened() *

//*************************
//*      ctInteract       *
//*************************
//******************************************************************************
//* User interactive experience.                                               *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void ColorTest::ctInteract ( void )
{
   if ( this->ctOpen )
   {
      //* File read/write message position *
      winPos mp( short(ic[ctDonePB].ulY + 9), ic[ctRgbMW].ulX ) ;
      //* Track user's selections *
      uiInfo   Info ;                     // user interface data returned here
      short    icIndex = ctDonePB ;       // index of control with input focus
      bool     done = false ;             // loop control
      while ( ! done )
      {
         //*******************************************
         //* If focus is currently on a Pushbutton   *
         //*******************************************
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {  //* Get user input *
            icIndex = this->dp->EditPushbutton ( Info ) ;

            //* If a Pushbutton was pressed *
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == ctDonePB )
               {  //* We're done *
                  done = true ;
               }

               //* If user wants to see samples for the next group of pairs *
               else if ( Info.ctrlIndex == ctNxtPB )
               {
                  if ( (this->firstSample + sampleGROUP) <= 
                       (this->colorMap.pairCount - sampleGROUP) )
                  {
                     this->firstSample += sampleGROUP ;
                     this->ctDisplaySampleText () ;
                     this->dp->RefreshWin () ;
                  }
               }

               //* If user wants to see samples for previous group of pairs *
               else if ( Info.ctrlIndex == ctPrvPB )
               {
                  if ( this->firstSample > ZERO )
                  {
                     this->firstSample -= sampleGROUP ;
                     this->ctDisplaySampleText () ;
                     this->dp->RefreshWin () ;
                  }
               }

               //* If user wants to modify one of the color pairs *
               else if ( Info.ctrlIndex == ctMprPB )
               {
                  //* Fade the pushbutton so as not to distract user.*
                  this->dp->SetPushbuttonText ( ctMprPB, 
                     ic[ctMprPB].dispText, ic[ctMprPB].nColor, ic[ctMprPB].nColor ) ;
                  this->dp->SetDialogObscured () ; // save parent-window display
                  bool status = this->ctModifyColorPair () ;
                  this->dp->RefreshWin () ;  // restore parent-window display
                  this->dp->SetPushbuttonText ( ctMprPB, 
                     ic[ctMprPB].dispText, ic[ctMprPB].fColor, ic[ctMprPB].nColor ) ;
                  if ( status != false )
                  {
                     this->ctDisplayRGBvalues () ;
                     this->ctDisplayColorPairs () ;
                     this->dp->RefreshWin () ;  // make changes visible
                  }
               }

               //* If user wants to see alternate page of Pair Table *
               else if ( Info.ctrlIndex == ctnppPB )
               {
                  if ( this->colorMap.pairCount > ncSUPPORTED_PAIRS )
                  {
                     this->firstPIndex = 
                        (this->firstPIndex == ZERO) ? ctptROWS : ZERO ;
                     this->ctDisplayColorPairs () ;
                     this->dp->RefreshWin () ;  // make changes visible
                  }
                  Info.keyIn = nckTAB ;   // allow input focus to advance
               }

               //* If user wants to reset all data to default values *
               else if ( Info.ctrlIndex == ctRstPB )
               {
                  nc.SetColorMap ( &this->origCM, this->origCM.canModifyPairs, 
                                   this->origCM.canModifyRGB ) ;
                  this->colorMap = this->origCM ;
                  this->ctDisplayRGBvalues () ;
                  this->ctDisplayColorPairs () ;
                  this->dp->RefreshWin () ;
                  Info.keyIn = nckTAB ;   // allow input focus to advance
               }

               //* If user wants to set color map from a file *
               else if ( Info.ctrlIndex == ctRddPB )
               {  //* !! Call To Non-member Method !! *
                  short status = File2ColorMap ( this->colorMap, cmapFilename ) ;
                  if ( status == OK )
                  {
                     this->dp->WriteParagraph ( mp, 
                                                "    File Read      \n"
                                                "   Successfully    ", 
                                                (this->dColor & !ncrATTR), true );
                     sleep ( 2 ) ;
                     nc.SetColorMap ( &this->colorMap, 
                                      this->colorMap.canModifyPairs, 
                                      this->colorMap.canModifyRGB ) ;
                     this->ctDisplayColorPairs () ;
                     this->dp->RefreshWin () ;
                  }
                  else     // error reading file
                  {
                     this->dp->WriteParagraph ( mp, 
                                                "     File Read     \n"
                                                "       Error!      ", 
                                                (this->dColor & !ncrATTR), true );
                     sleep ( 3 ) ;
                  }
                  this->dp->WriteParagraph ( mp, 
                                             "                   \n"
                                             "                   ", 
                                             this->dColor, true );
                  Info.keyIn = nckTAB ;   // allow input focus to advance
               }

               //* If user wants to save the current settings to a file *
               else if ( Info.ctrlIndex == ctSavPB )
               {
                  gString gs ;
                  gs.compose( L"  Saving data to   \n"
                               "   %-15s ", cmapFilename ) ;
                  this->dp->WriteParagraph ( mp, gs, (this->dColor & !ncrATTR), true );
                  sleep ( 3 ) ;
                  //* !! Call To Non-member Method !! *
                  short status = ColorMap2File ( this->colorMap, cmapFilename ) ;
                  if ( status != OK )
                  {
                     this->dp->WriteParagraph ( mp, 
                                                "    File Write     \n"
                                                "       Error!      ", 
                                                (this->dColor & !ncrATTR), true );
                     sleep ( 3 ) ;
                  }
                  this->dp->WriteParagraph ( mp, 
                                             "                   \n"
                                             "                   ", 
                                             this->dColor, true );
                  Info.keyIn = nckTAB ;   // allow input focus to advance
               }
            }
         }
         //**************************************
         //* If focus is currently on a Menuwin *
         //**************************************
         if ( ic[icIndex].type == dctMENUWIN )
         {
            icIndex = this->dp->EditMenuwin ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == ctRgbMW )
               {
                  this->dp->SetDialogObscured () ; // save parent-window display
                  short rIndex = Info.selMember / 2 ;
                  bool status = this->ctModifyRGB ( rIndex ) ;
                  this->dp->RefreshWin () ;     // restore parent-window display
                  if ( status != false )
                  {
                     this->ctDisplayRGBvalues () ;
                     this->dp->RefreshWin () ;
                  }
               }
               else if ( Info.ctrlIndex == ctBkgMW )
               {
                  short newBkgnd = ncbcDEFAULT ;   // (note this trick with enums)
//                  NcBaseColors newBkgnd = ncbcDEFAULT ;
                  switch ( Info.selMember )
                  {
                     case 0:  newBkgnd = ncbcBW ;  break ;
                     case 1:  newBkgnd = ncbcBK ;  break ;
                     case 2:  newBkgnd = ncbcRE ;  break ;
                     case 3:  newBkgnd = ncbcGR ;  break ;
                     case 4:  newBkgnd = ncbcBR ;  break ;
                     case 5:  newBkgnd = ncbcBL ;  break ;
                     case 6:  newBkgnd = ncbcMA ;  break ;
                     case 7:  newBkgnd = ncbcCY ;  break ;
                     case 8:  newBkgnd = ncbcGY ;  break ;
                     // Note: The ncurses library currently supports only
                     //       eight (8) background colors, even though the
                     //       Gnome terminal now supports 16 or more.
                     case 9:  newBkgnd = ncecBK ;  break ;
                     case 10: newBkgnd = ncecRE ;  break ;
                     case 11: newBkgnd = ncecGR ;  break ;
                     case 12: newBkgnd = ncecBR ;  break ;
                     case 13: newBkgnd = ncecBL ;  break ;
                     case 14: newBkgnd = ncecMA ;  break ;
                     case 15: newBkgnd = ncecCY ;  break ;
                     case 16: newBkgnd = ncecGY ;  break ;
                  }
                  nc.SetDefaultBackground ( (NcBaseColors)(newBkgnd) ) ;
                  this->ctGetCMap () ;    // get the modified color map
                  this->ctDisplayColorPairs () ;
                  this->dp->RefreshWin () ;
               }
            }
         }
         //**************************************
         //* If focus is currently on a Spinner *
         //**************************************
         if ( ic[icIndex].type == dctSPINNER )
         {
            icIndex = this->dp->EditSpinner ( Info ) ;
         }
         //* Move along, nothing to see here *
         if ( done == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = this->dp->PrevControl () ; 
            else if ( Info.keyIn != ZERO )
               icIndex = this->dp->NextControl () ;
         }
      }  // while()
   }
   else
      { /* Caller is an idiot */ }

}  //* End ctInteract() *

//*************************
//*  ctDisplayRGBvalues   *
//*************************
//******************************************************************************
//* Display the contents of the RGB register sets and a sample color for each. *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note:                                                         *
//* - Because ncurses output routines use the color pairs, AND because the     *
//*   user of this test (YOU!) may have redefined the color pairs, selecting   *
//*   the color attribute for the RGB samples is not straightforward. For each *
//*   RGB register set, we need to find a color pair that has the appropriate  *
//*   foreground or background to accurately display the sample color.         *
//*                                                                            *
//* - The ncurses library references only the first eight (8) RGB register     *
//*   sets; however, most modern GNU/Linux terminal emulators support either   *
//*   8 or 16 register sets.                                                   *
//******************************************************************************

void ColorTest::ctDisplayRGBvalues ( void )
{
const wchar_t* colorNames[] = 
{
   L"BK-", L"RE-", L"GR-", L"BR-", L"BL-", L"MA-", L"CY-", L"GY-",
   L"bBK", L"bRE", L"bGR", L"bBR", L"bBL", L"bMA", L"bCY", L"bGY",
} ;
attr_t colorSample[] = 
{
   nc.bwR, nc.reR, nc.grR, nc.brR, nc.blR, nc.maR, nc.cyR, nc.gyR,
   nc.bbk | ncrATTR, nc.bre | ncrATTR, nc.bgr | ncrATTR, nc.bbr | ncrATTR,
   nc.bbl | ncrATTR, nc.bma | ncrATTR, nc.bcy | ncrATTR, nc.bgy
} ;
const wchar_t *blankSample = L"                      ",
              *unmapSample = L"(register set unused) ", 
              blockSample[] = 
              { wcsBLOCKs, wcsBLOCKs, wcsBLOCKs, wcsBLOCKs, wcsBLOCKs, wcsBLOCKs, 
                wcsBLOCKs, wcsBLOCKs, wcsBLOCKs, wcsBLOCKs, wcsBLOCKs, wcsBLOCKs, 
                wcsBLOCKs, wcsBLOCKs, wcsBLOCKs, wcsBLOCKs, wcsBLOCKs, wcsBLOCKs, 
                wcsBLOCKs, wcsBLOCKs, wcsBLOCKs, wcsBLOCKs, NULLCHAR } ;

   attr_t pColor = nc.blR ;
   winPos wp( 2, 1 ) ;
   short boxRows = 18 ;
   if ( this->colorMap.rgbCount > ncbcCOLORS )
   { --wp.ypos ; boxRows = boxRows * 2 - 2 ; }

   this->dp->DrawBox ( wp.ypos, wp.xpos, boxRows, 24, 
                       pColor | ncbATTR, "RGB:     r    g    b", ncltSINGLE ) ;
   ++wp.ypos ;
   ++wp.xpos ;
   gString gsOut ;
   for ( short i = ZERO ; i < this->colorMap.rgbCount && i < ncCOLOR_MAX ; i++ )
   {
      attr_t sAttr = colorSample[i] ;
      const wchar_t* sPtr = blankSample ;

// BUG! - THE EXAMPLE COLORS FOR REGISTERS 8-15 IS NOT BEING RENDERED CORRECTLY 
//        TO THE DISPLAY.
      //* Play find-the-attribute (see note above) *
      if ( (i == ZERO && this->colorMap.pairMap[i].fgnd > ZERO) || 
           ((i > ZERO) && this->colorMap.pairMap[i].fgnd != i) )
      {  
         short j = ZERO ;
         while ( j < this->colorMap.pairCount )
         {
            if ( this->colorMap.pairMap[j].bkgnd == i )
               { sAttr = (i << 8) | ncrATTR ; break ; }
            else if ( this->colorMap.pairMap[j].fgnd == i )
               { sPtr = blockSample ; sAttr = (i << 8) | ncuATTR ; break ; }
            ++j ;
         }
         if ( j >= this->colorMap.pairCount )
         {
            sAttr = pColor ;
            sPtr = unmapSample ;
         }
      }
      gsOut.compose( L"%S%02hd  %04hd %04hd %04hd ", colorNames[i], &i, 
                     &this->colorMap.rgbMap[i].r, &this->colorMap.rgbMap[i].g, 
                     &this->colorMap.rgbMap[i].b ) ;
      this->dp->WriteString ( wp.ypos++, wp.xpos, gsOut, pColor ) ;
      this->dp->WriteString ( wp.ypos++, wp.xpos, sPtr, sAttr ) ;
   }

}  //* End ctDisplayRGBvalues()  *

//*************************
//*  ctDisplayColorPairs  *
//*************************
//******************************************************************************
//* Display the color-pair configuration.                                      *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void ColorTest::ctDisplayColorPairs ( void )
{
   short wLines = this->colorMap.pairCount <= ctptROWS ? 
                  this->colorMap.pairCount / 2 : 32 ;
   winPos wp( 2, 86 ) ;
   attr_t wColor = nc.blR ;
   gString gsOut ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, 
      L"PAIR: FGND / BKGND    PAIR: FGND / BKGND", wColor | ncbATTR | ncuATTR ) ;
   for ( short i = this->firstPIndex ; i < (wLines + this->firstPIndex) ; i++ )
   {
      short j = i + wLines ;
      gsOut.compose( L" %02hd   %04hd   %04hd     %3hd   %04hd   %04hd ", 
                     &i, &this->colorMap.pairMap[i].fgnd, 
                     &this->colorMap.pairMap[i].bkgnd, 
                     &j, 
                     &this->colorMap.pairMap[j].fgnd, 
                     &this->colorMap.pairMap[j].bkgnd ) ;
      this->dp->WriteString ( wp.ypos++, wp.xpos, gsOut, wColor ) ;
   }

}  //* End ctDisplayColorPairs() *

//*************************
//*  ctDisplaySampleText  *
//*************************
//******************************************************************************
//* Display text samples for eight (8) color-pairs using each of its           *
//* color-modification attributes.                                             *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void ColorTest::ctDisplaySampleText ( void )
{
   winPos wp( 2, 44 ) ;             // base display position
//* Display data for the first eight (8) defined color pairs *
const short basicSTRINGS = 48,
            scolHEIGHT = basicSTRINGS / 2,
            attrTYPES = 6,
            c1X = wp.xpos ;
const wchar_t* basicStrings[basicSTRINGS] =
{
   L" Bk-Text Normal    ", L" Bk-Text Bold      ", L" Bk-Text Underline ",
   L" Bk-Text Dimmed    ", L" Bk-Text Reverse   ", L" Bk-Text Standout  ",

   L" Re-Text Normal    ", L" Re-Text Bold      ", L" Re-Text Underline ",
   L" Re-Text Dimmed    ", L" Re-Text Reverse   ", L" Re-Text Standout  ",

   L" Gr-Text Normal    ", L" Gr-Text Bold      ", L" Gr-Text Underline ",
   L" Gr-Text Dimmed    ", L" Gr-Text Reverse   ", L" Gr-Text Standout  ",

   L" Br-Text Normal    ", L" Br-Text Bold      ", L" Br-Text Underline ",
   L" Br-Text Dimmed    ", L" Br-Text Reverse   ", L" Br-Text Standout  ",

   L" Bl-Text Normal    ", L" Bl-Text Bold      ", L" Bl-Text Underline ",
   L" Bl-Text Dimmed    ", L" Bl-Text Reverse   ", L" Bl-Text Standout  ",

   L" Ma-Text Normal    ", L" Ma-Text Bold      ", L" Ma-Text Underline ",
   L" Ma-Text Dimmed    ", L" Ma-Text Reverse   ", L" Ma-Text Standout  ",

   L" Cy-Text Normal    ", L" Cy-Text Bold      ", L" Cy-Text Underline ",
   L" Cy-Text Dimmed    ", L" Cy-Text Reverse   ", L" Cy-Text Standout  ",

   L" Gy-Text Normal    ", L" Gy-Text Bold      ", L" Gy-Text Underline ",
   L" Gy-Text Dimmed    ", L" Gy-Text Reverse   ", L" Gy-Text Standout  ",
} ;
#if 1    // NEW
const wchar_t* custStrings[attrTYPES] = 
{
   L"-Text Normal    ", L"-Text Bold      ", L"-Text Underline ",
   L"-Text Dimmed    ", L"-Text Reverse   ", L"-Text Standout  ",
} ;
#else    // OLD
const wchar_t* custStrings[attrTYPES] = 
{
   L" %02hd-Text Normal    ", L" %02hd-Text Bold      ", L" %02hd-Text Underline ",
   L" %02hd-Text Dimmed    ", L" %02hd-Text Reverse   ", L" %02hd-Text Standout  ",
} ;
#endif   // OLD
const attr_t attrMods[attrTYPES] = 
{ ZERO, ncbATTR, ncuATTR, ncdATTR, ncrATTR, ncsATTR } ;

   attr_t wColor = nc.blR, attrOut ;
   short pairIndex, attrIndex ;
   gString gsOut ;

   if ( this->firstSample == ZERO )
   {  //* Display data for the basic 8 pairs *
      pairIndex = -1 ;
      for ( short i = ZERO ; i < scolHEIGHT ; i++, ++attrIndex )
      {
         if ( !(i % attrTYPES) )
         {
            ++pairIndex ;
            short p1 = pairIndex, p2 = p1 + 4 ;
            gsOut.compose( L"      PAIR %02hd              PAIR %02hd      ", 
                           &p1, &p2 ) ;
            this->dp->WriteString ( wp.ypos++, c1X, gsOut, wColor | ncbATTR ) ;
            attrIndex = ZERO ;
         }
         attrOut = (pairIndex << 8) | attrMods[attrIndex] ;
         wp = this->dp->WriteString ( wp.ypos, (wp.xpos = c1X), 
                                      basicStrings[i], attrOut ) ;
         attrOut = ((pairIndex + 4) << 8) | attrMods[attrIndex] ;
         this->dp->WriteString ( wp.ypos++, (wp.xpos + 2), 
                                 basicStrings[i+scolHEIGHT], attrOut ) ;
      }
   }
   else
   {  //* Display data for custom (> 8th) pairs *
      pairIndex = this->firstSample - 1 ;
      short p1, p2 ;
      for ( short i = ZERO ; i < scolHEIGHT ; i++, ++attrIndex )
      {
         if ( !(i % attrTYPES) )
         {
            ++pairIndex ;
            p1 = pairIndex ;
            p2 = p1 + 4 ;
            if ( p1 < 100 && p2 < 100 )
               gsOut.compose( L"      PAIR %02hd              PAIR %02hd      ", &p1, &p2 ) ;
            else if ( p1 < 100 && p2 >= 100 )
               gsOut.compose( L"      PAIR %02hd             PAIR %3hd     ", &p1, &p2 ) ;
            else
               gsOut.compose( L"      PAIR %3hd             PAIR %3hd     ", &p1, &p2 ) ;
            this->dp->WriteString ( wp.ypos++, c1X, gsOut, wColor | ncbATTR ) ;
            attrIndex = ZERO ;
         }
         attrOut = (p1 << 8) | attrMods[attrIndex] ;
         if ( p1 < 100 )
            gsOut.compose( L" %02hd%S", &p1, custStrings[i % attrTYPES] ) ;
         else
            gsOut.compose( L"%03hd%S", &p1, custStrings[i % attrTYPES] ) ;
         wp = this->dp->WriteString ( wp.ypos, (wp.xpos = c1X), gsOut, attrOut ) ;
         attrOut = ((p2) << 8) | attrMods[attrIndex] ;
         if ( p2 < 100 )
            gsOut.compose( L" %02hd%S", &p2, custStrings[i % attrTYPES] ) ;
         else
            gsOut.compose( L"%03hd%S", &p2, custStrings[i % attrTYPES] ) ;
         this->dp->WriteString ( wp.ypos++, (wp.xpos + 2), gsOut, attrOut ) ;
      }
   }

}  //* End ctDisplaySampleText() *

//*************************
//*   ctModifyColorPair   *
//*************************
//******************************************************************************
//* Display the values of a foreground/background color pair and allow user to *
//* interactively change the values. Screen colors will be updated on return.  *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if successful, else 'false'                                *
//******************************************************************************

bool ColorTest::ctModifyColorPair ( void )
{
   const short dlgROWS = 8 ;           // display lines
   const short dlgCOLS = 40 ;          // display columns
   winPos dpos = this->dp->GetDialogPosition () ;// parent dialog position
   //* Our dialog positon (depends on number of RGB registers reported) *
   const short ulY = dpos.ypos + cinfoPos.ypos + 
                     ((this->colorMap.rgbCount <= 8) ? 4 : 7),
               ulX = dpos.xpos + 2 ;
   const attr_t bColor = nc.blR ;      // dialog background color
   int pairNum ;
   this->dp->GetSpinnerValue ( ctMprSP, pairNum ) ; // pair to be modified
   bool status = true ;

//* Initialization data for spinner controls *
dspinData spinData1( -1, (colorMap.rgbCount - 1), this->colorMap.pairMap[pairNum].fgnd, 
                     dspinINTEGER, nc.grG ),
          spinData2( -1, (colorMap.rgbCount - 1), this->colorMap.pairMap[pairNum].bkgnd, 
                     dspinINTEGER, nc.grG ) ;

enum mcpControls : short { donePB = ZERO, fSP, bSP, controlsDEFINED } ;
InitCtrl dic[controlsDEFINED] =   // array of dialog control initialization objects
{
   {  //* 'DONE' pushbutton - - - - - - - - - - - - - - - - - - - -     donePB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dlgROWS - 3),           // ulY:       upper left corner in Y
      short(dlgCOLS / 2 - 4),       // ulX:       upper left corner in X
      1,                            // lines:     control lines
      8,                            // cols:      control columns
      "  DONE  ",                   // dispText:  
      nc.gyR,                       // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &dic[fSP],                    // nextCtrl:  link in next structure
   },
   {  //* 'FGND'  Spinner - - - - - - - - - - - - - - - - - - - - - -      fSP *
      dctSPINNER,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      2,                            // ulY:       upper left corner in Y
      13,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      6,                            // cols:      control columns
      NULL,                         // dispText:  (n/a)
      nc.gyR,                       // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Foreground",                 // label:     
      -1,                           // labY:      
      -2,                           // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      &spinData1,                   // spinData:  spinner init
      true,                         // active:    allow control to gain focus
      &dic[bSP],                    // nextCtrl:  link in next structure
   },
   {  //* 'BKGND' Spinner - - - - - - - - - - - - - - - - - - - - - -      bSP *
      dctSPINNER,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      dic[fSP].ulY,                 // ulY:       upper left corner in Y
      short(dic[fSP].ulX + 11),     // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      6,                            // cols:      control columns
      NULL,                         // dispText:  (n/a)
      nc.gyR,                       // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Background",                 // label:     
      -1,                           // labY:      
      -2,                           // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      &spinData2,                   // spinData:  spinner init
      true,                         // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   },
} ;

   //* If terminal does not support color-pair     *
   //* modification, disable the unneeded controls.*
   if ( this->colorMap.canModifyPairs == false )
      dic[donePB].nextCtrl = NULL ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dlgROWS,        // number of display lines
                       dlgCOLS,        // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       "  Modify Color-Pair Map  ", // dialog title
                       ncltSINGLE,     // border line-style
                       (bColor | ncbATTR), // border color attribute
                       bColor,         // interior color attribute
                       dic             // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* ddp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (ddp->OpenWindow()) == OK )
   {
      NcBaseColors origFg = this->colorMap.pairMap[pairNum].fgnd,
                   origBg = this->colorMap.pairMap[pairNum].bkgnd ;
      if ( this->colorMap.canModifyPairs != false )
      {
         gString gsOut ;
         gsOut.compose( L"Pair: %02hd\n"
                         "  Current: (%03hd)      (%03hd)", 
                        &pairNum, &origFg, &origBg ) ;
         ddp->WriteParagraph ( 2, 2, gsOut, bColor ) ;
         gsOut.compose( L"[%C1 = Up/Dn  %C10 = Shift+Up/Dn]", 
                        &wcsPLMINUS, &wcsPLMINUS ) ;
         ddp->WriteString ( (dlgROWS - 2), 4, gsOut, bColor ) ;
         //* When setting pair zero, user may be surprised, so give warning.*
         if ( pairNum == ZERO )
         {
            ddp->RefreshWin () ;
            ddp->SetDialogObscured () ;
            ddp->WriteParagraph ( 2, 1, 
                             L" Note that setting non-default values \n"
                              "for pair 0 may have surprising results", 
                             (bColor & ~ncrATTR), true ) ;
            sleep ( 3 ) ;
            ddp->RefreshWin () ;
         }
      }
      else
      {
         ddp->WriteParagraph ( 2, 1, 
                             L" Sorry, this terminal does not support\n"
                              "     modification of color pairs.     ", 
                             (bColor & ~ncrATTR) ) ;
      }
      ddp->RefreshWin () ;

      uiInfo Info ;                 // user interface data returned here
      short  icIndex = ZERO ;       // index of control with input focus
      bool   done = false ;         // loop control
      while ( ! done )
      {
         //*******************************************
         //* If focus is currently on a Pushbutton   *
         //*******************************************
         if ( dic[icIndex].type == dctPUSHBUTTON )
         {
            icIndex = ddp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false && Info.ctrlIndex == donePB )
            {
               if ( this->colorMap.canModifyPairs != false )
               {
                  // (looks odd because of data-type conversions)
                  NcBaseColors newFg, newBg ;
                  int iVal ;
                  ddp->GetSpinnerValue ( fSP, iVal ) ;
                  newFg = NcBaseColors(iVal) ;
                  ddp->GetSpinnerValue ( bSP, iVal ) ;
                  newBg = NcBaseColors(iVal) ;
                  if ( newFg != origFg || newBg != origBg )
                  {
                     this->colorMap.pairMap[pairNum].fgnd = newFg ;
                     this->colorMap.pairMap[pairNum].bkgnd = newBg ;
                     if ( (nc.SetColorMap ( pairNum, 
                           this->colorMap.pairMap[pairNum] )) != OK )
                     {
                        ddp->WriteString ( (dlgROWS - 3), 1, 
                                          " Error, unable to modify color pair!  ", 
                                          (bColor & ~ncrATTR), true ) ;
                        sleep ( 3 ) ;
                        status = false ;
                     }
                     //* Get an updated copy of the map *
                     this->ctGetCMap () ;
//                     nc.GetColorMap ( this->colorMap ) ;
                  }
               }
               done = true ;                 // and we're done
            }
         }
         //**************************************
         //* If focus is currently on a Spinner *
         //**************************************
         if ( dic[icIndex].type == dctSPINNER )
         {
            icIndex = ddp->EditSpinner ( Info ) ;
         }
         if ( done == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = ddp->PrevControl () ; 
            else if ( Info.keyIn != ZERO )
               icIndex = ddp->NextControl () ;
         }
      }  // while()
   }
   else     // this is unlikely under our controlled circumstances
      status = false ;

   if ( ddp != NULL )                        // close the window
      delete ( ddp ) ;

   return status ;

}  //* End ctModifyColorPair() *

//*************************
//*      ctModifyRGB      *
//*************************
//******************************************************************************
//* Display the values of an RGB register set and allow user to interactively  *
//* change the register values. Screen colors will be updated for visual       *
//* feedback.                                                                  *
//*                                                                            *
//*                                                                            *
//* Input  : regSet  : index into colorMap.rgbMap[] for register set to modify *
//*                                                                            *
//* Returns: 'true' if successful, else 'false'                                *
//******************************************************************************
//* Programmer's Note: If the ncurses library reports that we may not modify   *
//* RGB registers, but we try to do it anyway, the library writes an error     *
//* message to the terminal window, and trashes our display. So if it says     *
//* that you can't do it, then DON'T do it.                                    *
//******************************************************************************

bool ColorTest::ctModifyRGB ( short regSet )
{
#define DEBUG_ModifyRGB (0)      // for debugging only
const short dlgROWS = 8 ;           // display lines
const short dlgCOLS = 40 ;          // display columns
winPos dpos = this->dp->GetDialogPosition () ;// parent dialog position
//* Our dialog positon (depends on number of RGB registers reported) *
const short ulY = dpos.ypos + cinfoPos.ypos + 
                  ((this->colorMap.rgbCount <= 8) ? 4 : 7),
            ulX = dpos.xpos + 2 ;
attr_t bColor = nc.blR ;            // dialog background color
bool status = true ;

//* Initialization data for spinner controls *
dspinData spinData1( 0, maxRGBvalue, this->colorMap.rgbMap[regSet].r, dspinINTEGER, nc.grG ),
          spinData2( 0, maxRGBvalue, this->colorMap.rgbMap[regSet].g, dspinINTEGER, nc.grG ),
          spinData3( 0, maxRGBvalue, this->colorMap.rgbMap[regSet].b, dspinINTEGER, nc.grG ) ;

enum mrgbControls : short { donePB = ZERO, rSP, gSP, bSP, controlsDEFINED } ;
InitCtrl dic[controlsDEFINED] =  // array of dialog control initialization objects
{
   {  //* 'DONE' pushbutton - - - - - - - - - - - - - - - - - - - -     donePB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dlgROWS - 3),           // ulY:       upper left corner in Y
      short(dlgCOLS / 2 - 4),       // ulX:       upper left corner in X
      1,                            // lines:     control lines
      8,                            // cols:      control columns
      "  DONE  ",                   // dispText:  
      nc.gyR,                       // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &dic[rSP],                    // nextCtrl:  link in next structure
   },
   {  //* 'RED'   Spinner - - - - - - - - - - - - - - - - - - - - - -      rSP *
      dctSPINNER,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      2,                            // ulY:       upper left corner in Y
      15,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      6,                            // cols:      control columns
      NULL,                         // dispText:  (n/a)
      nc.gyR,                       // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Red",                        // label:     
      -1,                           // labY:      
      2,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      &spinData1,                   // spinData:  spinner init
      true,                         // active:    allow control to gain focus
      &dic[gSP],                     // nextCtrl:  link in next structure
   },
   {  //* 'GREEN' Spinner - - - - - - - - - - - - - - - - - - - - - -      gSP *
      dctSPINNER,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      dic[rSP].ulY,                 // ulY:       upper left corner in Y
      short(dic[rSP].ulX + 8),      // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      6,                            // cols:      control columns
      NULL,                         // dispText:  (n/a)
      nc.gyR,                       // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Green",                      // label:     
      -1,                           // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      &spinData2,                   // spinData:  spinner init
      true,                         // active:    allow control to gain focus
      &dic[bSP],                    // nextCtrl:  link in next structure
   },
   {  //* 'BLUE'  Spinner - - - - - - - - - - - - - - - - - - - - - -      bSP *
      dctSPINNER,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      dic[rSP].ulY,                 // ulY:       upper left corner in Y
      short(dic[gSP].ulX + 8),      // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      6,                            // cols:      control columns
      NULL,                         // dispText:  (n/a)
      nc.gyR,                       // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Blue",                       // label:     
      -1,                           // labY:      
      1,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      &spinData3,                   // spinData:  spinner init
      true,                         // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   },
} ;

   //* If terminal (or ncurses) does not support RGB register modification *
   #if DEBUG_ModifyRGB != 0   // for debugging only
   this->colorMap.canModifyRGB = true ;
   #endif   // for debugging only
   if ( ! this->colorMap.canModifyRGB )
      dic[donePB].nextCtrl = NULL ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dlgROWS,        // number of display lines
                       dlgCOLS,        // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       "  Modify RGB Register Set  ", // dialog title
                       ncltSINGLE,     // border line-style
                       (bColor | ncbATTR), // border color attribute
                       bColor,         // interior color attribute
                       dic             // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* ddp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (ddp->OpenWindow()) == OK )
   {
      if ( this->colorMap.canModifyRGB != false )
      {
         gString gsOut ;
         gsOut.compose( L"RGB Set: %02hd\n"
                         "   Current:  (%03hd)   (%03hd)   (%03hd)", 
                        &regSet, &this->colorMap.rgbMap[regSet].r, 
                        &this->colorMap.rgbMap[regSet].g, 
                        &this->colorMap.rgbMap[regSet].b ) ;
         ddp->WriteParagraph ( 2, 2, gsOut, bColor ) ;
         gsOut.compose( L"[%C1 =Up/Dn   ]\n[%C10=S+Up/Dn ]", &wcsPLMINUS, &wcsPLMINUS ) ;
         ddp->WriteParagraph ( (dlgROWS - 3), 1, gsOut, bColor ) ;
         gsOut.compose( L"[%C100=C+Up/Dn]\n[  Home/End  ]", &wcsPLMINUS, &wcsPLMINUS ) ;
         ddp->WriteParagraph ( (dlgROWS - 3), 25, gsOut, bColor ) ;
      }
      else
      {
         ddp->WriteParagraph ( 2, 1, 
                               L" Sorry, this terminal does not support\n"
                                "    modification of RGB registers.    ", 
                               (bColor & ~ncrATTR) ) ;
      }
      ddp->RefreshWin () ;

      uiInfo Info ;                 // user interface data returned here
      short  icIndex = ZERO ;       // index of control with input focus
      bool   changes = false,       // 'true' if registers modified
             done = false ;         // loop control
      while ( ! done )
      {
         //*******************************************
         //* If focus is currently on a Pushbutton   *
         //*******************************************
         if ( dic[icIndex].type == dctPUSHBUTTON )
         {
            icIndex = ddp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false && Info.ctrlIndex == donePB )
            {  //* If something has changed, update the color map *
               if ( changes != false )
                  this->ctGetCMap () ;
               done = true ;                 // and we're done
            }
         }
         //**************************************
         //* If focus is currently on a Spinner *
         //**************************************
         else if ( dic[icIndex].type == dctSPINNER )
         {
            icIndex = ddp->EditSpinner ( Info ) ;
            if ( Info.dataMod != false )
            {
               int newVal ;
               ddp->GetSpinnerValue ( Info.ctrlIndex, newVal ) ;
               switch ( Info.ctrlIndex )
               {
                  case rSP: this->colorMap.rgbMap[regSet].r = newVal ;   break ;
                  case gSP: this->colorMap.rgbMap[regSet].g = newVal ;   break ;
                  case bSP: this->colorMap.rgbMap[regSet].b = newVal ;   break ;
               }
               short result ;
               #if DEBUG_ModifyRGB != 0   // for debugging only
               result = OK  ;
               #else                      // production build
               result = nc.SetColorMap ( regSet, this->colorMap.rgbMap[regSet] );
               #endif                     // production build
               ddp->RefreshWin () ;    // make the changes visible
               if ( result == OK )
               {
                  changes = true ;
                  ddp->WriteString ( 4, 1, "    RGB register values modified.     ", 
                                     (bColor & ~ncrATTR), true ) ;
                  sleep ( 2 ) ;
               }
               else
               {
                  ddp->WriteString ( 4, 1, "     Error setting RGB registers      ", 
                                     (bColor & ~ncrATTR), true ) ;
                  status = false ;
                  sleep ( 3 ) ;
               }
// BUG! - SETTING THE GLOBAL REGISTER ERASES THE PARENT DIALOG. NEED TO REFRESH!
               ddp->ClearLine ( 4 ) ;
            }
         }
         if ( done == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = ddp->PrevControl () ; 
            else if ( Info.keyIn != ZERO )
               icIndex = ddp->NextControl () ;
         }
      }  // while()
   }
   else     // this is unlikely under our controlled circumstances
      status = false ;

   if ( ddp != NULL )                        // close the window
      delete ( ddp ) ;

   return status ;

#undef DEBUG_ModifyRGB
}  //* End ctModifyRGB() *

//*************************
//*      ctGetCMap        *
//*************************
//******************************************************************************
//* Refresh our copy of the color map: this->colorMap                          *
//* The call to GetColorMap() is isolated here to facilitate debugging of the  *
//* dialog, i.e. we can conveniently lie about the number of RGB registers     *
//* reported to exercise the full functionality of the dialog.                 *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void ColorTest::ctGetCMap ( void )
{

   nc.GetColorMap ( this->colorMap ) ;

}  //* End ctGetCMap() *


//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *
//* - - - - - - - - - - - - Non-member Methods- - - - - - - - - - - - - - - -  *
//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *

//*************************
//*     ColorMap2File     *
//*************************
//******************************************************************************
//* Write the contents of an NcColorMap object to a text file.                 *
//* If file does not exist, it will be created. If file already exists, the    *
//* new data will be appended.                                                 *
//*                                                                            *
//* Input  : cm   : (by reference) data to be written                          *
//*          fPath: pointer to full path/filename of target file               *
//*                                                                            *
//* Returns: OK if data successfully written, else ERR                         *
//******************************************************************************

short ColorMap2File ( const NcColorMap& cm, const char* fPath )
{
const wchar_t* colorNames[] = 
{
#if 1    // NEW
   L"BK-", L"RE-", L"GR-", L"BR-", L"BL-", L"MA-", L"CY-", L"GY-",
   L"bBK", L"bRE", L"bGR", L"bBR", L"bBL", L"bMA", L"bCY", L"bGY",
#else    // OLD
   L"BK-", L"RE-", L"GR-", L"BR-", L"BL-", L"MA-", L"CY-", L"GY-",
   L"BBK", L"BRE", L"BGR", L"BBR", L"BBL", L"BMA", L"BCY", L"BGY",
#endif   // OLD
} ;
const wchar_t* fileHeader = 
   L"NcDialog Color Map Settings -created %02hu/%02hu/%04hu %02hu:%02hu:%02hu\n"
    "(this is a generated file - use care if editing manually)\n"
    "~--------------------------------------------------------" ;
const char * rgbHeader  = "\n RGB:      r    g    b \n"
                          "~---------------------" ;
const char * clrHeader  = "\nPAIR:  FGND / BKGND\n"
                          "~---------------------" ;
   USHORT date = ZERO,  month = ZERO,   year = ZERO, 
          hours = ZERO, minutes = ZERO, seconds = ZERO ;
   short status = ERR ;

   //* Get the system time code *
   time_t epoch ;
   if ( (epoch = time ( NULL )) >= 0 )
   {
      //* Decode the system time *
      Tm    tm ;                    // Linux time structure
      if ( (localtime_r ( &epoch, &tm )) != NULL )
      {
         //* Translate *
         date    = tm.tm_mday ;        // today's date
         month   = tm.tm_mon + 1 ;     // month
         year    = tm.tm_year + 1900 ; // year
         hours   = tm.tm_hour ;        // hour
         minutes = tm.tm_min ;         // minutes
         seconds = tm.tm_sec ;         // seconds
      }
   }

   //* Create an output file (or append to existing file) *
   ofstream ofs ( fPath, ofstream::out | ofstream::app ) ;
   if ( ofs.is_open() )                         // if output file open
   {
      //* Write the file header *
      gString  gsOut ;              // formatting buffer
      gsOut.compose( fileHeader, &month, &date, &year, &hours, &minutes, &seconds ) ;
      ofs << gsOut.ustr() << endl ;

      //* Write the array sizes and modification flags *
      gsOut.compose( L"rgbCount : %3hd", &cm.rgbCount ) ;
      ofs << gsOut.ustr() << endl ;
      gsOut.compose( L"pairCount: %3hd", &cm.pairCount ) ;
      ofs << gsOut.ustr() << endl ;
      gsOut.compose( L"supColors: %3hd", &cm.supColors ) ;
      ofs << gsOut.ustr() << endl ;
      gsOut.compose( L"rgbMods  : %hhd", &cm.canModifyRGB ) ;
      ofs << gsOut.ustr() << endl ;
      gsOut.compose( L"pairMods : %hhd", &cm.canModifyPairs ) ;
      ofs << gsOut.ustr() << endl ;

      //* Write the RGB data header *
      ofs << rgbHeader << endl ;

      // Write the RGB data *
      for ( short i = ZERO ; i < cm.rgbCount ; i++ )
      {
         gsOut.compose( L"%S-%02hd  %04hd %04hd %04hd", colorNames[i], &i, 
                        &cm.rgbMap[i].r, &cm.rgbMap[i].g, &cm.rgbMap[i].b ) ;
         ofs << gsOut.ustr() << endl ;
      }

      //* Write the Color-pair header *
      ofs << clrHeader << endl ;

      //* Write the Color-pair data *
      for ( short i = ZERO ; i < cm.pairCount ; i++ )
      {
         gsOut.compose( L" %03hd   %04hd   %04hd", 
                        &i, &cm.pairMap[i].fgnd, &cm.pairMap[i].bkgnd ) ;
         ofs << gsOut.ustr() << endl ;
      }

      ofs << "\n---end map---\n" << endl << flush ;
      ofs.close() ;                 // close the file
      status = OK ;                 // return the good news
   }
   return status ;

}  //* End ColorMap2File() *

//*************************
//*     File2ColorMap     *
//*************************
//******************************************************************************
//* Read the contents of a Color Map text file and save it to an NcColorMap    *
//* object.                                                                    *
//*                                                                            *
//* Input  : cm   : (by reference) data to be written                          *
//*          fPath: pointer to full path/filename of target file               *
//*                                                                            *
//* Returns: OK if data object successfully initialized, else ERR              *
//******************************************************************************
//* Programmer's Note: This is a brain-dead method with hardly any error       *
//* checking. If you have a real-life need to read the color map from a file,  *
//* we recommend a more robust implementation (and don't used sscanf).         *
//******************************************************************************

short File2ColorMap ( NcColorMap& cm, const char* fPath )
{
   short status = ERR ;

   //* Open the specified file *
   ifstream ifs ( fPath, ifstream::in ) ;
   if ( ifs.is_open() )             // if input file open
   {
      char  inp[MAX_PATH] ;
      status = OK ;                 // hope for the best

      //* Step over header information *
      do { ifs.getline ( inp, MAX_PATH ) ; } while ( *inp != TILDE && ifs.good() ) ;

      //* Get array sizes *
      ifs.getline ( inp, MAX_PATH ) ;
      if ( *inp == 'r' && inp[3] == 'C' )
      {
         if ( (sscanf ( &inp[10], " %hd", &cm.rgbCount )) != 1 )
            status = ERR ;
      }
      else
         status = ERR ;
      ifs.getline ( inp, MAX_PATH ) ;
      if ( *inp == 'p' && inp[4] == 'C' )
      {
         if ( (sscanf ( &inp[10], " %hd", &cm.pairCount )) != 1 )
            status = ERR ;
      }
      else
         status = ERR ;

      //* Get and discard 'supColors' *
      ifs.getline ( inp, MAX_PATH ) ;

      //* Get modification flags *
      ifs.getline ( inp, MAX_PATH ) ;
      if ( *inp == 'r' && inp[3] == 'M' )
      {
         short tmp ;
         if ( (sscanf ( &inp[10], " %hd", &tmp )) == 1 )
            cm.canModifyRGB = tmp == ZERO ? false : true ;
         else
            status = ERR ;
      }
      else
         status = ERR ;
      ifs.getline ( inp, MAX_PATH ) ;
      if ( *inp == 'p' && inp[4] == 'M' )
      {
         short tmp ;
         if ( (sscanf ( &inp[10], " %hd", &tmp )) == 1 )
            cm.canModifyPairs = tmp == ZERO ? false : true ;
         else
            status = ERR ;
      }
      else
         status = ERR ;

      if ( status == OK )
      {
         //* Step over header information *
         do { ifs.getline ( inp, MAX_PATH ) ; } while ( *inp != TILDE && ifs.good() ) ;

         //* Get RGB register set data *
         short index = ZERO ;
         do
         {
            ifs.getline ( inp, MAX_PATH ) ;
            if ( index < cm.rgbCount )
            {
               if ( (sscanf ( &inp[6], " %hd %hd %hd", 
                              &cm.rgbMap[index].r, 
                              &cm.rgbMap[index].g, 
                              &cm.rgbMap[index].b )) != 3 )
                  status = ERR ;
               ++index ;
            }
         }
         while ( ifs.gcount() == 23 && ifs.good() && (index < cm.rgbCount) ) ;
   
         //* Step over header information *
         do { ifs.getline ( inp, MAX_PATH ) ; } while ( *inp != TILDE && ifs.good() ) ;
   
         //* Get Color Pair data *
         index = ZERO ;
         do
         {
            ifs.getline ( inp, MAX_PATH ) ;
            if ( index < cm.pairCount )
            {
               short p, f, b ;
               if ( (sscanf ( inp, " %hd %hd %hd", 
                              &p, &f, &b )) == 3 )
               {
                  cm.pairMap[index].fgnd = (NcBaseColors)f ; 
                  cm.pairMap[index].bkgnd = (NcBaseColors)b ;
               }
               else
                  status = ERR ;
               ++index ;
            }
         }
         while ( ifs.gcount() == 19 && ifs.good() && (index < cm.pairCount) ) ;
      }
      ifs.close() ;                 // close the file
   }
   return status ;

}  //* End File2ColorMap() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

