//******************************************************************************
//* File       : ChartTest.cpp                                                 *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2020-2025 Mahlon R. Smith, The Software Samurai *
//*                  GNU GPL copyright notice located in NcDialog.hpp          *
//* Date       : 21-Mar-2025                                                   *
//* Version    : (see below)                                                   *
//*                                                                            *
//* Description: Class definition for testing of the Chart class.              *
//*              This class provides a dialog window in which all functionality*
//*              of the Chart class can be tested and demonstrated.            *
//*              This test is called as a menu item within the Dialog4         *
//*              application, Test10.                                          *
//*                                                                            *
//* Development Tools: See NcDialog.cpp.                                       *
//******************************************************************************
//* Version History (most recent first):                                       *
//* v: 0.00.03 21-Aug-2021                                                     *
//*   - Convert the file-access methods to static, non-member methods.         *
//*     This brings the file-access definitions into local scope to avoid      *
//*     conflicting definitions among the various independent tests in the     *
//*     "ExpandTest" group.                                                    *
//*                                                                            *
//* v: 0.00.02 01-Jul-2021                                                     *
//*   - Original code moved into ChartTest.cpp.                                *
//*   - Refine layout.                                                         *
//*   - Define user-interface controls.                                        *
//*   - Enhanced range checking and error checking.                            *
//*                                                                            *
//* v: 0.00.01 01-Mar-2021                                                     *
//*   - Created as a method in the Exercalc application.                       *
//******************************************************************************

//****************
//* Header Files *
//****************
#include "ChartTest.hpp"

//***************************
//* File-access definitions *
//***************************
//* Local definitions for file type encoded in the "st_mode" *
//* element of the stat64{} structure                        *
enum fmFType : short
{ fmDIR_TYPE,   fmREG_TYPE,  fmLINK_TYPE, fmCHDEV_TYPE, 
  fmBKDEV_TYPE, fmFIFO_TYPE, fmSOCK_TYPE, fmUNKNOWN_TYPE, fmTYPES } ;

//* The following c-type structures defined in the standard header files 
//* need to be C++ data types: stat{} dirent{}, etc.
typedef struct stat64 FileStats ;
//typedef struct dirent64 deStats ;
typedef struct tm Tm ;
//* Size of buffer to hold filename (POSIX) *
const short MAX_FNAME = 255 ;

//* Class for reporting the system local date/time.                            *
//* ---------------------------------------------------------------------------*
#define DECODE_TZ (0)   // if non-zero, decode time-zone fields of localTime
const int64_t maxETIME32 = 0x07FFFFFFF,      // largest 32-bit time_t
              minETIME32 = -maxETIME32 - 1 ; // smallest 32-bit time_t
const short ltFMT_LEN = 16 ;                 // length of time string buffers
const short ltISO_LEN = 24 ;
//* Important note about the year 2038 bug:                                    *
//* Because 'time_t' is defined as a signed, 32-bit value on 32-bit systems,   *
//* time reporting will break down for date/time values                        *
//*                     >= 19 Jan 2038 at 11:14:08                             *
//* For this reason and for others, we have defined our own local-time class.  *
//* This class uses a signed, 64-bit value for storing the epoch time.         *
//* It is hoped that all Linux systems will have been updated long before this *
//* becomes an issue; however, even now (2013), some applications are using    *
//* date/time manipulations that exceed the 32-bit limit. BEWARE!!             *
//* NOTE: time arithmetic should always be performed on signed 64-bit values.  *
//*                                                                            *
class localTime
{
   public:
   localTime ( void ) { this->reset() ; }
   void reset ( void )
   {  //* Set to epoch date: Thursday, 01 Jan. 1970 @ 08:00:00
      this->sysepoch = ZERO ;
      this->epoch = ZERO ;
      this->day = 4 ;
      this->year = 1970 ;
      this->month = this->date = 1 ;
      this->hours = 8 ;
      this->minutes = this->seconds = ZERO ;
      this->julian = ZERO ;
      *this->iso = NULLCHAR ;
      #if DECODE_TZ != 0   // time-zone data
      *this->timezone = NULLCHAR ;
      *this->utc_zone  = NULLCHAR ;
      this->gmtoffset = ZERO ;
      this->dst       = false ;
      #endif   // DECODE_TZ
   }
   int64_t  epoch ;        //* Seconds since the epoch (01-Jan-1970)
   time_t   sysepoch ;     //* system epoch time (see note)
   uint16_t day ;          //* Day of the week (0 == Sun, ..., 6 == Sat)
   uint16_t date ;         //* Day of the month (1-[28|29|30|31])
   uint16_t month ;        //* Month of the year (1 == Jan, ... 12 == Dec)
   uint16_t year ;         //* Year (four digit year)
   uint16_t hours ;        //* Time of day, hour (0-23)
   uint16_t minutes ;      //* Time of day, minutes (0-59)
   uint16_t seconds ;      //* Time of day, seconds (0-59)
   uint16_t julian ;       //* Completed days since beginning of year
   char iso[ltISO_LEN] ;   //* String representation: ISO 8601-1:2019 "yyyy-mm-ddThh:mm:ss"

   #if DECODE_TZ != 0   // time-zone data
   // See DecodeEpochTime method for more information.
   char     timezone[ltFMT_LEN] ; //* string representing the local time zone (useless)
   char     utc_zone[ltFMT_LEN] ; //* string representing the UTC offset for local time zone
   short    gmtoffset ;    //* Expressed as (signed) number of seconds offset from UTC+0:00
   bool     dst ;          //* 'true' if daylight savings time in effect,
                           //* 'false if not DST or info not available
   #endif   // DECODE_TZ
} ;

//* Class definition for human-readable file stats.                            *
class tnFName
{
   public:
   ~tnFName ( void ) {}       //* Destructor
   tnFName ( void )           //* Constructor
   { this->ReInit () ; }
   void ReInit(void)
   {
      fName[0] = NULLCHAR ;
      fType    = fmTYPES ;
      fBytes   = ZERO ;
      modTime.day = modTime.date = modTime.month = modTime.year 
                  = modTime.hours = modTime.minutes = modTime.seconds = ZERO ;
      readAcc  = false ;
      writeAcc = false ;
      // NOTE: data member, rawStats is left unitialized
   }
   char        fName[MAX_FNAME] ; //* Filename string
   fmFType     fType ;     //* File type (enum fmFType)
   UINT64      fBytes ;    //* File size (in bytes)
   localTime   modTime ;   //* Date/Time file last modified (human readable)
   FileStats   rawStats ;  //* Copy of system's "stat" structure (all file info)
   bool        readAcc ;   //* 'true' if user has read access to file
   bool        writeAcc ;  //* 'true' if user has write access to file 
} ;


//**********************
//** Local Prototypes **
//**********************
static short ctControlUpdate ( const short currIndex, 
                               const wkeyCode wkey, bool firstTime ) ;
static bool TargetExists ( const gString& trgPath, fmFType& fType ) ;
static bool GetFileStats ( const gString& trgPath, tnFName& fStats ) ;


//****************
//** Local Data **
//****************
static const short 
   dROWS      = 34,        // dialog rows
   dCOLS      = 122,       // dialog columns
   CHART_YOFF = 1,         // offset for internal chart area
   CHART_XOFF = 53,
   CHART_ROWS = dROWS / 2, // default internal-chart dimensions
   CHART_COLS = 67,
   CDLG_Y     = 0,         // offset from parent dialog origin to child dialog
   CDLG_X     = 44,
   CDLG_ROWS  = dROWS,     // child dialog dimensions
   CDLG_COLS  = 86,
   CDLG_FOOT  = 12 ;

static const wchar_t spongeBob = 0x25AA ; // (▪) black-small-square
static const wchar_t wDiamond  = 0x25C6 ; // (◆) black diamond

static const int32_t Data1Count = 200 ;
static const double  Data1[Data1Count] // All data are positive values
{
    1.0,  1.5,  2.0,  2.5,  3.0,  3.5,  4.0,  4.5,  5.0,  5.5, // increase
    6.0,  6.5,  7.0,  7.5,  8.0,  8.5,  9.0,  9.5, 10.0, 10.5,
   11.0, 11.5, 12.0, 12.5, 13.0, 13.5, 14.0, 14.5, 15.0, 15.5,
   16.0, 16.5, 17.0, 17.5, 18.0, 18.5, 19.0, 19.5, 20.0, 20.5,

   20.3, 20.1, 19.9, 19.7, 19.5, 19.3, 19.1, 18.9, 18.7, 18.5, // decrease
   18.3, 18.1, 17.9, 17.7, 17.5, 17.3, 17.1, 16.9, 16.7, 16.5,

   15.0, 15.1, 15.2, 15.3, 15.4, 15.5, 15.6, 15.4, 15.2, 15.0, // fluctuate
   14.8, 14.6, 14.4, 14.2, 14.0, 14.5, 15.0, 15.5, 15.3, 15.1,

   15.0, 14.0, 15.0, 14.8, 14.6, 14.4, 14.2, 14.0, 13.8, 13.6, // decrease
   13.4, 13.2, 13.0, 12.8, 12.6, 12.4, 12.2, 12.0, 11.8, 11.6,
   11.4, 11.2, 11.0, 10.8, 10.6, 10.4, 10.2, 10.0,  9.8,  9.6,
    9.4,  9.2,  9.0,  8.8,  8.6,  8.4,  8.2,  8.0,  7.8,  7.6,
    7.4,  7.2,  7.0,  6.8,  6.6,  6.4,  6.2,  6.0,  6.8,  6.6,
    6.4,  6.2,  6.0,  5.8,  5.6,  5.4,  5.2,  5.0,  4.8,  4.6,
    4.4,  4.2,  4.0,  3.8,  3.6,  3.4,  3.2,  3.0,  2.8,  2.6,
    2.4,  2.2,  2.0,  1.8,  1.6,  1.4,  1.2,  1.0,  0.8,  0.6,
    
    0.4,  0.2,  0.0,  0.5,  1.0,  1.5,  2.0,  2.5,  3.0,  2.5, // step up
    2.0,  2.5,  3.0,  3.5,  4.0,  4.5,  5.0,  4.5,  4.0,  4.5,
    5.0,  5.5,  6.0,  6.5,  7.0,  7.5,  8.0,  8.5,  9.0,  9.5,
   10.0, 11.0, 12.0, 13.0, 13.5, 14.0, 14.5, 15.0, 15.5, 14.0,
} ;

//* Multi-color array. Initialized during instantiation. *
static attr_t Attr1[Data1Count] ;

static const int32_t Data2Count = 200 ;
static const double  Data2[Data2Count] // Mixed positive and negative data values
{
    // first page, wander about
    5.0,  5.5,  6.0,  6.5,  5.5,  4.5,  4.0,  2.0,  1.5,  1.0,
    0.2,  0.0, -0.3, -0.7, -1.1, -1.6, -2.2, -2.8, -3.4, -4.0,
   -4.8, -5.1, -5.5, -6.0, -6.2, -6.4, -6.8, -7.4, -7.5, -7.5,
   -7.4, -7.2, -6.9, -6.5, -6.0, -5.4, -4.7, -3.9, -2.0, -1.0,
    0.1,  1.3,  2.4,  3.5,  4.6,  4.0,  3.0,  2.0,  1.0,  0.5,
   -1.5, -2.0, -2.5, -3.0, -2.4, -2.1, -1.6, -1.1, -0.8, -0.4,
    // increase from zero to max and back to zero
    0.0,  0.2,  0.4,  0.7,  0.9,  1.1,  1.4,  1.6,  1.8,  2.1,
    2.3,  2.5,  2.8,  3.0,  3.2,  3.5,  3.7,  3.9,  4.2,  4.4,
    4.6,  4.9,  5.1,  5.3,  5.6,  5.8,  6.0,  6.3,  6.5,  6.7,
    7.0,  7.2,  7.4,  7.5,  7.4,  7.2,  7.0,  6.7,  6.5,  6.3,
    6.0,  5.8,  5.6,  5.3,  5.1,  4.9,  4.6,  4.4,  4.2,  3.9,
    3.7,  3.5,  3.2,  3.0,  2.8,  2.5,  2.3,  2.1,  1.8,  1.6,
    1.4,  1.1,  0.9,  0.7,  0.4,  0.2,  0.0, -0.2, -0.4, -0.7,
   // decrease from zero to min and back to zero
   -0.9, -1.1, -1.4, -1.6, -1.8, -2.1, -2.3, -2.5, -2.8, -3.0,
   -3.2, -3.5, -3.7, -3.9, -4.2, -4.4, -4.6, -4.9, -5.1, -5.3,
   -5.6, -5.8, -6.0, -6.3, -6.5, -6.7, -7.0, -7.2, -7.4, -7.5,
   -7.4, -7.2, -7.0, -6.7, -6.5, -6.3, -6.0, -5.8, -5.6, -5.3,
   -5.1, -4.9, -4.6, -4.4, -4.2, -3.9, -3.7, -3.5, -3.2, -3.0,
   -2.8, -2.5, -2.3, -2.1, -1.8, -1.6, -1.4, -1.1, -0.9, -0.7,
   -0.4, -0.2,  0.0,  0.5,  1.0,  1.5,  2.0,  1.5,  2.0,  2.5,
} ;

static const int32_t IntCount = 50 ;
static const float FloatData[IntCount]
{
   15.0,  12.0,   9.0,   6.0,   3.0,   1.0,   0.0,  -2.0,  -4.0,  -6.0,
   -8.0, -10.0, -12.0, -14.0, -15.0, -14.5, -14.0, -13.0, -12.5, -12.0,
  -11.5, -11.0, -10.5, -10.0,  -9.0,  -8.5,  -7.0,  -6.0,  -5.0,  -4.0,
   -3.0,  -2.0,  -1.0,  -0.0,   1.0,   2.5,   4.0,   5.5,   7.0,   8.5,
   10.0,  11.0,  11.5,  12.0,  12.5,  13.0,  13.5,  14.0,  14.5,  15.0,
} ;
static const char sByteData[IntCount]
{
   15,  12,   9,   6,   3,   1,   0,  -2,  -4,  -6,
   -8, -10, -12, -14, -15, -14, -14, -13, -12, -12,
  -11, -11, -10, -10,  -9,  -8,  -7,  -6,  -5,  -4,
   -3,  -2,  -1,  -0,   1,   2,   4,   5,   7,   8,
   10,  11,  11,  12,  12,  13,  13,  14,  14,  15,
}  ;
static const unsigned char uByteData[IntCount]
{
    0,   2,   4,   6,   8,  10,  12,  14,  16,  18,
   20,  22,  24,  26,  28,  30,  32,  31,  30,  29,
   28,  27,  26,  25,  24,  23,  22,  21,  20,  19,
   18,  17,  16,  15,  14,  13,  12,  11,  10,   9,
    8,   7,   6,   5,   4,   3,   4,   3,   2,   1,
}  ;
static const short sShortData[IntCount]
{
   15,  12,   9,   6,   3,   1,   0,  -2,  -4,  -6,
   -8, -10, -12, -14, -15, -14, -14, -13, -12, -12,
  -11, -11, -10, -10,  -9,  -8,  -7,  -6,  -5,  -4,
   -3,  -2,  -1,  -0,   1,   2,   4,   5,   7,   8,
   10,  11,  12,  11,  12,  13,  14,  13,  14,  15,
}  ;
static const unsigned short uShortData[IntCount]
{
    0,   2,   4,   6,   8,  10,  12,  14,  16,  18,
   20,  22,  24,  26,  28,  30,  32,  31,  30,  29,
   28,  27,  26,  25,  24,  23,  22,  21,  20,  19,
   18,  17,  16,  15,  14,  13,  12,  11,  10,   9,
    8,   7,   6,   5,   4,   3,   4,   3,   2,   1,
}  ;
static const int sIntData[IntCount]
{
  150,  120,   90,   60,   30,   10,    0,  -20,  -40,  -60,
  -80, -100, -120, -140, -150, -140, -135, -125, -120, -115,
 -110, -105, -100,  -80,  -70,  -60,  -50,  -40,  -30,  -20,
  -10,   -5,    0,    5,   10,   20,   30,   40,   45,   50,
   55,   60,   65,   70,   75,   80,   85,   90,  100,  115,
}  ;
static const unsigned int uIntData[IntCount]
{
    0,  20,  40,  60,  80, 100, 120, 140, 160, 180,
  200, 220, 240, 260, 280, 300, 320, 310, 300, 290,
  280, 270, 260, 250, 240, 230, 220, 210, 200, 190,
  180, 170, 160, 150, 140, 130, 120, 110, 105, 100,
   90,  80,  70,  60,  55,  50,  45,  40,  35,  30,
}  ;
static const long int sLongData[IntCount]
{
  150,  200,  300,  400,  500,  550,  600,   700,  800,  900,
 1000,  850,  700,  550,  400,  250,  100,   -50, -200, -350,
 -500, -650, -800, -950, -975, -985, -995, -1000, -900, -800,
 -750, -650, -500, -400, -200, -100,    0,    50,  100,  150,
  200,  250,  300,  350,  400,  450,  500,   550,  600,  700,
}  ;
static const unsigned long int uLongData[IntCount]
{
     0,  200,  400,  600,  800, 1000, 1200, 1400, 1600, 1800,
  2000, 2200, 2400, 2600, 2800, 3000, 3200, 3100, 3000, 2900,
  2800, 2700, 2600, 2500, 2400, 2300, 2200, 2100, 2000, 1900,
  1800, 1700, 1600, 1500, 1400, 1300, 1200, 1100, 1050, 1000,
   900,  800,  700,  600,  550,  500,  450,  400,  350,  300,
}  ;
static const long long int sLongLongData[IntCount]
{
  150,  200,  300,  400,  500,  550,  600,   700,  800,  900,
 1000,  850,  700,  550,  400,  250,  100,   -50, -200, -350,
 -500, -650, -800, -950, -975, -985, -995, -1000, -900, -800,
 -750, -650, -500, -400, -200, -100,    0,    50,  100,  150,
  200,  250,  300,  350,  400,  450,  500,   550,  600,  700,
}  ;
static const unsigned long long int uLongLongData[IntCount]
{
     0,  200,  400,  600,  800, 1000, 1200, 1400, 1600, 1800,
  2000, 2200, 2400, 2600, 2800, 3000, 3200, 3100, 3000, 2900,
  2800, 2700, 2600, 2500, 2400, 2300, 2200, 2100, 2000, 1900,
  1800, 1700, 1600, 1500, 1400, 1300, 1200, 1100, 1050, 1000,
   900,  800,  700,  600,  550,  500,  450,  400,  350,  300,
}  ;
//* Coordinate pairs for use by Cartesian charts *
static const int32_t CartDataCount = 408 ;    // Two hundred four X/Y Pairs
static const double CartData1[CartDataCount]  // Range in X: -20.0 through 20.0
{                                      // Range in Y: ?
   //* First and Second Quadrants *
     1.0,  1.0,     // 00
     1.0,  2.0,     // 01
     2.0,  2.0,     // 02
     3.0,  0.0,     // 03
     4.0,  0.0,     // 04
     4.0,  1.0,     // 05
     4.0,  3.0,     // 06
     5.0,  2.0,     // 07
     5.0,  4.0,     // 08
     6.0, -1.0,     // 09
     6.0,  0.0,     // 10
     6.0,  2.0,     // 11
     7.0,  1.0,     // 12
     7.0,  3.0,     // 13
     7.0,  5.0,     // 14
     8.0, -2.0,     // 15
     8.0,  0.0,     // 16
     8.0,  3.0,     // 17
     8.0,  6.0,     // 18
     9.0,  1.0,     // 19
     9.0,  4.0,     // 20
    10.0, -3.0,     // 21
    10.0,  0.0,     // 22
    10.0,  4.0,     // 23
    10.0,  7.0,     // 24
    11.0,  1.0,     // 25
    11.0,  3.0,     // 26
    11.0,  5.0,     // 27
    11.0,  8.0,     // 28
    12.0, -4.0,     // 29
    12.0,  0.0,     // 30
    12.0,  5.0,     // 31
    12.0,  6.0,     // 32
    13.0,  2.0,     // 33
    13.0,  4.0,     // 34
    13.0,  7.0,     // 35
    14.0, -5.0,     // 36
    14.0,  0.0,     // 37
    14.0,  4.0,     // 38
    14.0,  6.0,     // 39
    15.0,  2.0,     // 40
    15.0,  5.0,     // 41
    15.0,  7.0,     // 42
    16.0, -6.0,     // 43
    16.0,  4.0,     // 44
    16.0,  7.0,     // 45
    17.0,  0.0,     // 46
    17.0,  2.0,     // 47
    17.0,  5.0,     // 48
    17.0,  7.0,     // 49
    18.0,  3.0,     // 50
    18.0,  6.0,     // 51
    18.0,  7.0,     // 52
    19.0,  0.0,     // 53
    19.0,  2.0,     // 54
    20.0, -8.0,     // 55
    20.0,  7.0,     // 56
    19.0, -2.0,     // 57
    18.0, -3.0,     // 58
    18.0, -6.0,     // 59
    17.0, -2.0,     // 60
    17.0, -5.0,     // 61
    17.0, -7.0,     // 62
    16.0, -7.0,     // 63
    16.0, -4.0,     // 64
    16.0,  6.0,     // 65
    15.0, -2.0,     // 66
    15.0, -5.0,     // 67
    15.0, -7.0,     // 68
    14.0,  5.0,     // 69
    14.0, -4.0,     // 70
    14.0, -6.0,     // 71
    13.0, -2.0,     // 72
    13.0, -4.0,     // 73
    13.0, -8.0,     // 74
    12.0, -6.0,     // 75
    12.0, -5.0,     // 76
    12.0,  4.0,     // 77
    11.0, -1.0,     // 78
    11.0, -3.0,     // 79
    11.0, -5.0,     // 80
    11.0, -8.0,     // 81
    10.0, -7.0,     // 82
    10.0, -4.0,     // 83
    10.0,  3.0,     // 84
     9.0, -1.0,     // 85
     9.0, -4.0,     // 86
     8.0, -6.0,     // 87
     8.0, -3.0,     // 88
     8.0,  2.0,     // 89
     7.0, -1.0,     // 90
     7.0, -3.0,     // 91
     7.0, -5.0,     // 92
     6.0, -2.0,     // 93
     6.0,  1.0,     // 94
     5.0, -2.0,     // 95
     5.0, -4.0,     // 96
     4.0, -3.0,     // 97
     4.0, -1.0,     // 98
     2.0, -1.0,     // 99
     2.0, -2.0,     //100
     1.0, -1.0,     //101

   //* Third and Fourth Quadrants *
    -1.0, -1.0,     //102
    -1.0, -2.0,     //103
    -2.0, -2.0,     //104
    -3.0,  0.0,     //105
    -4.0,  0.0,     //106
    -4.0, -1.0,     //107
    -4.0, -3.0,     //108
    -5.0, -2.0,     //109
    -5.0, -4.0,     //110
    -6.0,  1.0,     //111
    -6.0, -0.0,     //112
    -6.0, -2.0,     //113
    -7.0, -1.0,     //114
    -7.0, -3.0,     //115
    -7.0, -5.0,     //116
    -8.0,  2.0,     //117
    -8.0, -0.0,     //118
    -8.0, -3.0,     //119
    -8.0, -6.0,     //120
    -9.0, -1.0,     //121
    -9.0, -4.0,     //122
   -10.0,  3.0,     //123
   -10.0, -0.0,     //124
   -10.0, -4.0,     //125
   -10.0, -7.0,     //126
   -11.0, -1.0,     //127
   -11.0, -3.0,     //128
   -11.0, -5.0,     //129
   -11.0, -8.0,     //130
   -12.0,  4.0,     //131
   -12.0, -0.0,     //132
   -12.0, -5.0,     //133
   -12.0, -6.0,     //134
   -13.0, -2.0,     //135
   -13.0, -4.0,     //136
   -13.0, -7.0,     //137
   -14.0,  5.0,     //138
   -14.0, -0.0,     //139
   -14.0, -4.0,     //140
   -14.0, -6.0,     //141
   -15.0, -2.0,     //142
   -15.0, -5.0,     //143
   -15.0, -7.0,     //144
   -16.0,  6.0,     //145
   -16.0, -4.0,     //146
   -16.0, -7.0,     //147
   -17.0, -0.0,     //148
   -17.0, -2.0,     //149
   -17.0, -5.0,     //150
   -17.0, -7.0,     //151
   -18.0, -3.0,     //152
   -18.0, -6.0,     //153
   -18.0, -7.0,     //154
   -19.0, -0.0,     //155
   -19.0, -2.0,     //156
   -20.0,  8.0,     //157
   -20.0, -7.0,     //158
   -19.0,  2.0,     //159
   -18.0,  3.0,     //160
   -18.0,  6.0,     //161
   -17.0,  2.0,     //162
   -17.0,  5.0,     //163
   -17.0,  7.0,     //164
   -16.0,  7.0,     //165
   -16.0,  4.0,     //166
   -16.0, -6.0,     //167
   -15.0,  2.0,     //168
   -15.0,  5.0,     //169
   -15.0,  7.0,     //170
   -14.0, -5.0,     //171
   -14.0,  4.0,     //172
   -14.0,  6.0,     //173
   -13.0,  2.0,     //174
   -13.0,  4.0,     //175
   -13.0,  8.0,     //176
   -12.0,  6.0,     //177
   -12.0,  5.0,     //178
   -12.0, -4.0,     //179
   -11.0,  1.0,     //180
   -11.0,  3.0,     //181
   -11.0,  5.0,     //182
   -11.0,  8.0,     //183
   -10.0,  7.0,     //184
   -10.0,  4.0,     //185
   -10.0, -3.0,     //186
    -9.0,  1.0,     //187
    -9.0,  4.0,     //188
    -8.0,  6.0,     //189
    -8.0,  3.0,     //190
    -8.0, -2.0,     //191
    -7.0,  1.0,     //192
    -7.0,  3.0,     //193
    -7.0,  5.0,     //194
    -6.0,  2.0,     //195
    -6.0, -1.0,     //196
    -5.0,  2.0,     //197
    -5.0,  4.0,     //198
    -4.0,  3.0,     //199
    -4.0,  1.0,     //200
    -2.0,  1.0,     //201
    -2.0,  2.0,     //202
    -1.0,  1.0,     //203
} ;
//* These data bracket the endpoints of the X and Y axes.*
// Programmer's Note: These values were empirically constructed to test the 
// limits of the overlay using the default (25/67) chart-area dimensions. 
// They form approximations of diamond shapes. If alternate chart dimensions or 
// borders are used, the datapoint layout will be less pretty but should still 
// produce a valid test.
static const int32_t CartOverlayCount = 92 ;
static const double CartOverlayData[CartOverlayCount]
{
     0.0,   9.0,    // 00 (off-scale above, should be discarded)
     1.5,   8.0,    // 01
     2.0,   7.0,    // 02
     2.5,   6.0,    // 03
     2.0,   4.5,    // 04
     1.5,   3.0,    // 05
    -1.0,   8.0,    // 06
    -1.5,   7.0,    // 07
    -2.0,   6.0,    // 08
    -1.5,   4.0,    // 09
    -1.0,   3.0,    // 10

     0.0,  -9.5,    // 11 (off-scale below, should be discarded)
     1.5,  -8.0,    // 12
     2.0,  -7.0,    // 13
     2.5,  -6.0,    // 14
     2.0,  -5.0,    // 15
     1.5,  -4.0,    // 16
    -1.0,  -8.0,    // 17
    -1.5,  -7.0,    // 18
    -2.0,  -6.0,    // 19
    -1.5,  -5.0,    // 20
    -1.0,  -4.0,    // 21

    21.5,   0.0,    // 22 (off-scale right, should be discarded)
    21.0,   1.0,    // 23
    20.0,   2.5,    // 24
    19.5,   3.5,    // 25
    19.0,   2.5,    // 26
    18.0,   1.0,    // 27
    17.5,   0.0,    // 28
    18.0,  -1.5,    // 29
    19.0,  -2.5,    // 30
    19.5,  -3.5,    // 31
    20.0,  -2.5,    // 32
    21.0,  -1.5,    // 33

   -21.0,   0.0,    // 34 (off-scale left, should be discarded)
   -20.0,   1.0,    // 35
   -19.5,   2.5,    // 36
   -19.0,   3.5,    // 37
   -18.5,   2.5,    // 38
   -18.0,   1.0,    // 39
   -17.0,   0.0,    // 40
   -18.0,  -1.5,    // 41
   -18.5,  -2.5,    // 42
   -19.0,  -3.5,    // 43
   -19.5,  -2.5,    // 44
   -20.0,  -1.5,    // 45
} ;

static const char* const horizLabel = "Horizontal-axis Label" ;
static const char* const vertLabel  = "Vertical-axis Label" ;
static const char* const headText   = 
   "  This chart displays some random dataset in a way which users \n"
   "  might find interesting. -- (users are such simple creatures)" ;
static const char* const footText   =
   "     These data were recorded by people in white lab coats and  \n"
   "          bad haircuts, so you know it must be accurate.        " ;
static const char* const marginText =
   "These data represent\n"
   "the number of cicadas\n"
   "(genus Magicicada)\n"
   "which emerged during\n"
   "each 24-hour period of\n"
   "the 2021 cycle.\n"
   "One might say that we\n"
   "could not possibly have\n"
   "counted them all,\n"
   "individually, but in fact\n"
   "we did accomplish this\n"
   "feat simply by counting\n"
   "their legs and dividing\n"
   "by six.\n"
   " --Hemiptera Harry, PhD" ;

// Shift Instructions inset from right edge of display area *
static const short siINSET = 33 ;
static const short siHEIGHT = 7 ;   // rows in ShiftInstr message
static const wchar_t* ShiftInstrDflt =
  L"───────────────────────────────\n"
   "RightArrow :+1,  LeftArrow :-1\n"
   "Shift+Right:+5,  Shift+Left:-5\n"
   "Ctrl+Right :+10, Ctrl+Left :-10\n"
   "PageDown, PageUp, Home, End,\n"
   "  OR: Ctrl+R to reload dataset.\n"
   "(any other key to close chart)\n" ;
static const wchar_t* ShiftInstrCust = 
  L"───────────────────────────────\n"
   "'f' first page  'l' last page\n"
   "'n' next page   'p' prev page\n"
   "'a' : +1  'b' : +5  'c' : +10\n"
   "'A' : -1  'B' : -5  'C' : -10\n"
   " OR: Ctrl+R to reload dataset.\n"
   "(any other key to close chart)\n" ;
static const char* ChartTestTitle = "  Chart-class Formats and Options  " ;
static const char* ParentDlgTitle = "  Display Chart in Parent Dialog  " ;
static const char* ChildDlgTitle  = "  Local Chart Dialog Window  " ;

//*************************************
//** Test-dialog control information **
//*************************************
enum ctControls : short    // Uniquely name each dialog control
{
   ctLaunchPB = ZERO,      // 'Launch Chart' Pushbutton
   ctClosePB,              // 'Close' Pushbutton
   ctHelpPB,               // 'Info Help' Pushbutton
   ctStatTB,               // 'Status Message' Textbox
   ctTypeDD,               // 'Chart Type' Dropdown
   ctRowSP,                // 'Chart Rows' Spinner
   ctColSP,                // 'Chart Cols' Spinner
   ctFootSP,               // 'Footer Rows' Spinner
   ctMarginSP,             // "Margin Columns' Spinner
   ctBorderSP,             // 'Border Style' Spinner
   ctGridSP,               // 'Grid Style' Spinner
   ctHlineSP,              // 'Horizontal Divider' Spinner
   ctBdivSP,               // 'Bar Divisions' Spinner
   ctIndyRB,               // 'Child Dialog' Radiobutton
   ctHtextRB,              // 'Set Header Text' Radiobutton
   ctFtextRB,              // 'Set Footer Text' Radiobutton
   ctMtextRB,              // 'Set Margin Text' Radiobutton
   ctTtextRB,              // 'Set Title Text' Radiobutton
   ctHbarsRB,              // 'Horizontal Bars' Radiobutton
   ctMultiRB,              // 'Multi-color Bars' Radiobutton
   ctSpecSE,               // 'Special Tests' Scrollext
   ctDataDD,               // 'Data Type' Dropdown
   ctCONTROLS              // number of controls defined
} ;


//* Color scheme for dctSCROLLEXT and dctDROPDOWN and dctMENUWIN controls *
static attr_t monoColor[2] = { attrDFLT, nc.bw } ;

//* Chart-type Scrollbox *
static const short typeDDrows = 11,
            typeDDcols = 16,
            typeDDitems = 9 ;
static const char typeDDdata[typeDDitems][typeDDcols - 1] = 
{
   "ctLowerLeft   ",
   "ctLowerRight  ",
   "ctUpperLeft   ",
   "ctUpperRight  ",
   "ctLowerCenter ",
   "ctUpperCenter ",
   "ctCenterLeft  ",
   "ctCenterRight ",
   "ctCartesian   ",
} ;

//* Special-tests Scrollext *
enum SpecTest : short
{
   stSpaceBars,         // space bars at every 2nd column/row
   stAudibleAlert,      // beep for invalid shift keypress
   stCustomKeys,        // custom data-shift UI keys
   stShiftInstr,        // display data-shift instructions
   stMarginBdr,         // draw border around margin area
   stHeaderLine,        // draw a divider as last line of header
   stFooterLine,        // draw a divider as first line of footer
   stAltNegColor,       // alternate color for negative values
   stReportStats,       // report stats on the dataset

   stOverlay,           // Cartesian: overlay 2nd dataset
   stAltPointChar,      // Cartesian: alternate datapoint character
   stInvertPairs,       // Cartesian: Invert data pairs (Y/X vs. X/Y)

   stStressHead,        // Perform stress-test sequence for Header area
   stStressFoot,        // Perform stress-test sequence for Footer area
   stStressMargin,      // Perform stress-test sequence for Margin area

   stITEMS              // number of items in Scrollext control
} ;
static const short 
   specHEIGHT = 7,                     // height of Scrollext control
   specWIDTH  = 39,                    // width of Scrollext control
   specSRC    = specWIDTH + 4 ;        // length of specTextSrc[] text
static char specTextSrc[stITEMS][specSRC]
{
   "_ Space Between Data Bars            ",
   "_ Audible Alert for Invalid UI Keys  ",
   "_ Use Alternate UI Key Definitions   ",
   "_ Display UI Instructions            ",
   "_ Draw Border Around Margin Area     ",
   "_ Draw Line Under Header             ",
   "_ Draw Line Over Footer              ",
   "_ Alt Color for Negative Values      ",
   "_ Report Statistics On Dataset       ",
   "_ Overlay a 2nd Dataset   (Cartesian)",
   "_ Alt Datapoint Character (Cartesian)",
   "_ Invert Data Pairs       (Cartesian)",
   "_ Stress Test for Header Area        ",
   "_ Stress Test for Footer Area        ",
   "_ Stress Test for Margin Area        ",
} ;
static char* specText[stITEMS]
{
   specTextSrc[stSpaceBars],
   specTextSrc[stAudibleAlert],
   specTextSrc[stCustomKeys],
   specTextSrc[stShiftInstr],
   specTextSrc[stMarginBdr],
   specTextSrc[stHeaderLine],
   specTextSrc[stFooterLine],
   specTextSrc[stAltNegColor],
   specTextSrc[stReportStats],
   specTextSrc[stOverlay],
   specTextSrc[stAltPointChar],
   specTextSrc[stInvertPairs],
   specTextSrc[stStressHead],
   specTextSrc[stStressFoot],
   specTextSrc[stStressMargin],
} ;
static bool specFlags[stITEMS] ; // item-selection flags, initialized in Setup()
static attr_t specAttr[stITEMS] ;// attributes initialized in Setup()
static ssetData setSpecTest( (const char**)specText, specAttr, stITEMS, 
                             ZERO, false ) ;

//* Data-set selection Dropdown *
static const short
   datasetROWS  = 15,
   datasetCOLS  = 34,
   datasetITEMS = 13 ;
static const char datasetData[datasetITEMS][datasetCOLS + 1]
{
   " 200 positive doubles           ",
   " 200 pos/neg  doubles           ",
   "  50 pos/neg  floats            ",
   "  50 integers, signed           ",
   "  50 integers, unsigned         ",
   "  50 long ints, signed          ",
   "  50 long ints, unsigned        ",
   "  50 long-long ints, signed     ",
   "  50 long-long ints, unsigned   ",
   "  50 short ints, signed         ",
   "  50 short ints, unsigned       ",
   "  50 bytes, signed              ",
   "  50 bytes, unsigned            ",
} ;

//* Row and Column spinners *
static const short
   minHEIGHT  = 10,        // chart area minimum rows
   maxHEIGHT  = 32,        // chart area maximum rows
//   dfltHEIGHT = CHART_ROWS,// chart area default rows
   dfltHEIGHT = 25,        // chart area default rows (enough to display shift instructions)
   minWIDTH   = 12,        // chart area minimum columns
   maxWIDTH   = 80,        // chart area maximum columns
   dfltWIDTH  = CHART_COLS,// chart area default columns
   spinWIDTH  = 6 ;        // width of spinner controls
static dspinData rspinData(minHEIGHT, maxHEIGHT, dfltHEIGHT, dspinINTEGER, nc.brgr ) ;
static dspinData cspinData(minWIDTH, maxWIDTH, dfltWIDTH, dspinINTEGER, nc.brgr ) ;

//* Free Footer Space spinner *
static const short
   minFOOT  = ZERO,     // no space below axis label
   maxFOOT  = 17,       // space for data-shift instructions, etc.
   dfltFOOT = 9,        // (enough to display shift instructions)
   textFOOT = 3 ;       // two text lines + 1
static dspinData fspinData(minFOOT, maxFOOT, dfltFOOT, dspinINTEGER, nc.brgr ) ;

//* Free Margin Space spinner *
static const short
   minMARGIN  = ZERO,
   maxMARGIN  = maxWIDTH / 2 - 4,
   dfltMARGIN = 1 ;
static dspinData mspinData(minMARGIN, maxMARGIN, dfltMARGIN, dspinINTEGER, nc.brgr ) ;

//* Border-style spinner *
static const short
   minBORDER = 0,
   maxBORDER = 9 ;
static dspinData bspinData(minBORDER, maxBORDER, minBORDER, dspinINTEGER, nc.brgr ) ;

//* Grid-style spinner *
static const short
   minGRID = 1,
   maxGRID = 2 ;
static dspinData gspinData(minGRID, maxGRID, ZERO, dspinINTEGER, nc.brgr ) ;

//* Horizontal-line-offset spinner *
static const short
   minHLINE = -1,          // no line
   maxHLINE = maxFOOT ;    // out-of-range
static dspinData hspinData(minHLINE, maxHLINE, minHLINE, dspinINTEGER, nc.brgr ) ;

//* Bar-width spinner *
static const short
   minBDIV = ZERO,
   maxBDIV = cellDIVS ;
static dspinData wspinData(minBDIV, maxBDIV, maxBDIV, dspinINTEGER, nc.brgr ) ;

//* Dummy declaration for control-object definitions.   *
//* See the bottom of this module for actual definition.*
extern InitCtrl ChartTestIC[ctCONTROLS] ;

//* Data access for Callback method *
static NcDialog *ctcuPtr = NULL ;
static InitCtrl *cticPtr = NULL ;
static ChartTest *ctPtr  = NULL ;
static attr_t    ctDlgAttr = ZERO ;
static attr_t    ctEmpAttr = ZERO ;


//*************************
//*       ChartTest       *
//*************************
//******************************************************************************
//* Constructor: Exercise the options of the Chart class.                      *
//* The 'dp' member will be initialized by caller.                             *
//*                                                                            *
//* Input  : wpOrig : (by reference) origin (upper left corner) for test dialog*
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

ChartTest::ChartTest ( const winPos& wpOrig )
{
   //* Initialize our data members.                               *
   //* The 'dp' and 'cdef.dPtr' members will be initialized below.*
   this->Setup ( wpOrig ) ;

   //* Define the dialog object *
   InitNcDialog dInit( dROWS,          // number of display lines
                       dCOLS,          // number of display columns
                       this->wpOrig.ypos, // Y offset from upper-left of terminal 
                       this->wpOrig.xpos, // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltDUAL,       // border line-style
                       this->hColor,   // border color attribute
                       this->dColor,   // interior color attribute
                       this->icptr     // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   this->dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (this->dlgOpen = bool(this->dp->OpenWindow()) == OK) )
   {
      //* Set Chart-object target dialog pointer *
      this->cdef.dPtr = this->dp ;

      //* Complete setup of Special-tests Scrollext control *
      this->dp->SetScrollextText ( ctSpecSE, setSpecTest ) ;

      //* Set dialog title *
      this->dp->SetDialogTitle ( ChartTestTitle, this->tColor ) ;

      //* Welcome message *
      this->statMsg ( "Yer' now off the Charts, Matey.\n"
                      "Here, there be dragons!", true, true ) ;

      this->dp->RefreshWin () ;     // make everything visible

      //* Establish a callback method.                *
      ctcuPtr = this->dp ;          // access to dialog
      cticPtr = this->icptr ;       // access to control definitions
      ctPtr = this ;                // access to application methods
      ctDlgAttr = this->dColor ;    // dialog color
      ctEmpAttr = this->hColor ;    // dialog border color
      this->dp->EstablishCallback ( &ctControlUpdate ) ;


      //****************************************
      //* Interact with user (filthy beasts :) *
      //****************************************
      uiInfo Info ;                 // user interface data returned here
      short  icIndex = ZERO ;       // index of control with input focus
      bool   done = false ;         // loop control
      while ( ! done )
      {
         if ( this->icptr[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey != false )
               Info.HotData2Primary () ;
            else
               icIndex = this->dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == ctLaunchPB )
               {
                  this->BlastOff () ;
               }
               else if ( Info.ctrlIndex == ctHelpPB )
               {
                  this->Cry4Help () ;
               }
               else if ( Info.ctrlIndex == ctClosePB) 
               {
                  done = true ;
               }
            }
         }

         else if ( this->icptr[icIndex].type == dctSCROLLBOX )
         {
            if ( Info.viaHotkey != false )
               Info.viaHotkey = false ;
            icIndex = this->dp->EditScrollbox ( Info ) ;
            if ( Info.dataMod != false )
            {
            }
         }

         else if ( this->icptr[icIndex].type == dctSCROLLEXT )
         {
            if ( Info.viaHotkey )         // arrived via hotkey
               Info.viaHotkey = false ;   // ignore hotkey data
            icIndex = this->dp->EditScrollext ( Info ) ;
            if ( Info.dataMod != false )
            {
               //* If not leaving control via hotkey, retain *
               //* focus after explicit item selection.      *
               //* Note that user selection has been recorded*
               //* by the callback method.                   *
               if ( ! Info.viaHotkey &&
                    ((Info.keyIn == nckSPACE) || (Info.keyIn == nckENTER) ||
                     (Info.keyIn == nckpENTER)) )
               {
                  this->dp->PrevControl () ;
                  Info.keyIn = nckTAB ;
               }
            }
         }

         else if ( this->icptr[icIndex].type == dctDROPDOWN )
         {
            //* If selection of control via hotkey, *
            //* Dropdown will expand immediately.   *
            icIndex = this->dp->EditDropdown ( Info ) ;
            if ( Info.dataMod != false )
            {
            }
         }

         else if ( this->icptr[icIndex].type == dctSPINNER )
         {
            icIndex = this->dp->EditSpinner ( Info ) ;
            if ( Info.dataMod != false )
            {
            }
         }

         else if ( this->icptr[icIndex].type == dctRADIOBUTTON )
         {
            if ( Info.viaHotkey != false )
               Info.HotData2Primary () ;
            else
               icIndex = this->dp->EditRadiobutton ( Info ) ;
            if ( Info.dataMod != false )
            {
            }
         }

         else if ( this->icptr[icIndex].type == dctTEXTBOX )
         {
            Info.viaHotkey = false ;
            icIndex = this->dp->EditTextbox ( Info ) ;
            if ( Info.dataMod != false )
            {
            }
         }

         //* Move input focus to next/previous control.*
         if ( done == false && Info.viaHotkey == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = this->dp->PrevControl () ; 
            else
               icIndex = this->dp->NextControl () ;
         }
      }  // while()
   }

   if ( this->dp != NULL )             // close the window
      delete ( this->dp ) ;
   ctcuPtr = NULL ;                    // reset callback pointers
   cticPtr = NULL ;
   ctPtr   = NULL ;
   ctDlgAttr = ZERO ;
   ctEmpAttr = ZERO ;

}  //* End ChartTest() *

//*************************
//*         Setup         *
//*************************
//******************************************************************************
//* Called by ChartTest() method to initialize the test data.                  *
//*                                                                            *
//* Input  : wpo : (by reference) position of dialog window                    *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void ChartTest::Setup ( const winPos& wpo )
{
   //* Color Attributes *
   this->dColor   = nc.blR ;                 // dialog base color
   this->tColor   = this->dColor | ncbATTR ; // dialog title color
   this->bColor   = nc.gybl | ncbATTR ;      // dialog border color
   this->hColor   = nc.brbl | ncbATTR ;      // dialog highlight color
   this->chbColor = nc.br ;                  // chart base color, grid color, bar color
   this->chnColor = nc.bl ;                  // chart negative-values color
   this->cdbColor = nc.grbk ;                // child dialog border color
   this->pbnColor = nc.gyR ;                 // Pushbutton colors
   this->pbfColor = nc.reG ;
   this->tbnColor = nc.bbrbl ;               // Textbox colors
   this->tbfColor = nc.bkgr ;
   this->ddnColor = this->dColor ;           // Scrollext and Dropdown colors
   this->ddfColor = nc.rebl ;
   monoColor[1]   = nc.cyma ;                // list-item color attributes
   this->spnColor = nc.cyma ;                // Spinner colors
   this->spfColor = nc.bw ;
   rspinData.indiAttr = cspinData.indiAttr = fspinData.indiAttr = 
   mspinData.indiAttr = bspinData.indiAttr = gspinData.indiAttr = 
   hspinData.indiAttr = wspinData.indiAttr = nc.brgr ;
   this->rbnColor = this->spnColor ;         // Radiobutton colors
   this->rbfColor = this->spfColor ;

   this->wpOrig = { wpo.ypos, wpo.xpos } ;   // main dialog origin
   this->dlgOpen  = false ;                  // dialog is not yet open

   //* Set initial parameters for creating the Chart object.*
   this->cdef.dPtr     = this->dp ;          // dialog pointer
   this->cdef.ulY      = CHART_YOFF ;        // Y origin
   this->cdef.ulX      = CHART_XOFF ;        // X origin
   this->cdef.rows     = CHART_ROWS ;        // rows
   this->cdef.cols     = CHART_COLS ;        // columns
   this->cdef.yOffset  = 1 ;                 // Y offset
   this->cdef.xOffset  = 1 ;                 // X offset
   this->cdef.footRows = dfltFOOT ;          // footer rows
   this->cdef.barWidth = cellDIVS ;          // bar width (divisions)
   this->cdef.barSpace = ZERO ;              // bar spacing (0 or 1)
   this->cdef.chType   = ctLowerLeft ;       // chart type (see below)
   this->cdef.borderColor = this->chbColor ; // border color
   this->cdef.titleColor  = this->chbColor ; // title color
   this->cdef.baseColor   = this->chbColor ; // base color
   this->cdef.textColor   = this->chnColor ; // text color
   this->cdef.gridColor   = this->chbColor ; // grid color
   this->cdef.barColor    = this->chbColor ; // bar color (all values, or positive for centered axes)
   this->cdef.negColor    = this->chnColor ; // bar color (negative values for centered axes)
   this->cdef.dataCount   = Data1Count ;     // data element count
   this->cdef.dataOffset  = ZERO ;           // offset into data array
   this->cdef.dataPtr     = Data1 ;          // data array
   this->cdef.dataType    = idtDouble ;      // type of data referenced by 'dataPtr'
   this->cdef.attrPtr     = NULL ;           // bar-color array
   this->cdef.cartChar    = dblDiamond ;     // Cartesian datapoint character
   this->cdef.borderStyle = ncltSINGLE ;     // border style (line type)
   this->cdef.gridStyle   = ncltSINGLE ;     // grid style (see below)
   this->cdef.vaxisLabel  = vertLabel ;      // vertical-axis label
   this->cdef.haxisLabel  = horizLabel ;     // horizontal-axis label
   this->cdef.titleText   = NULL ;           // dialog title text
   this->cdef.headText    = NULL ;           // header text
   this->cdef.footText    = NULL ;           // footer text
   this->cdef.marginText  = NULL ;           // margin text
   this->cdef.horizBars   = false ;          // 'true' if horizontal bars, 'false' if vertical bars
                                             // (for Cartesian chart, indicates pair order)
   this->cdef.drawBorder  = false ;          // 'true' if area border, 'false' if no border
   this->icptr = ChartTestIC ;               // access to InitCtrl array

   //* Initialize the color attributes for the 'InitCtrl' array.*
   this->icptr[ctLaunchPB].nColor = this->pbnColor ;
   this->icptr[ctLaunchPB].fColor = this->pbfColor ;
   this->icptr[ctClosePB].nColor  = this->pbnColor ;
   this->icptr[ctClosePB].fColor  = this->pbfColor ;
   this->icptr[ctHelpPB].nColor   = this->pbnColor ;
   this->icptr[ctHelpPB].fColor   = nc.brgr ;
   this->icptr[ctStatTB].nColor   = this->tbnColor ;
   this->icptr[ctStatTB].fColor   = this->tbfColor ;
   this->icptr[ctTypeDD].nColor   = this->ddnColor ;
   this->icptr[ctTypeDD].fColor   = this->ddfColor ;
   this->icptr[ctRowSP].nColor    = this->spnColor ;
   this->icptr[ctRowSP].fColor    = this->spfColor ;
   this->icptr[ctColSP].nColor    = this->spnColor ;
   this->icptr[ctColSP].fColor    = this->spfColor ;
   this->icptr[ctFootSP].nColor   = this->spnColor ;
   this->icptr[ctFootSP].fColor   = this->spfColor ;
   this->icptr[ctMarginSP].nColor = this->spnColor ;
   this->icptr[ctMarginSP].fColor = this->spfColor ;
   this->icptr[ctBorderSP].nColor = this->spnColor ;
   this->icptr[ctBorderSP].fColor = this->spfColor ;
   this->icptr[ctGridSP].nColor   = this->spnColor ;
   this->icptr[ctGridSP].fColor   = this->spfColor ;
   this->icptr[ctHlineSP].nColor  = this->rbnColor ;
   this->icptr[ctHlineSP].fColor  = this->rbfColor ;
   this->icptr[ctBdivSP].nColor   = this->spnColor ;
   this->icptr[ctBdivSP].fColor   = this->spfColor ;
   this->icptr[ctIndyRB].nColor   = this->rbnColor ;
   this->icptr[ctIndyRB].fColor   = this->rbfColor ;
   this->icptr[ctHtextRB].nColor  = this->rbnColor ;
   this->icptr[ctHtextRB].fColor  = this->rbfColor ;
   this->icptr[ctFtextRB].nColor  = this->rbnColor ;
   this->icptr[ctFtextRB].fColor  = this->rbfColor ;
   this->icptr[ctMtextRB].nColor  = this->rbnColor ;
   this->icptr[ctMtextRB].fColor  = this->rbfColor ;
   this->icptr[ctTtextRB].nColor  = this->rbnColor ;
   this->icptr[ctTtextRB].fColor  = this->rbfColor ;
   this->icptr[ctHbarsRB].nColor  = this->rbnColor ;
   this->icptr[ctHbarsRB].fColor  = this->rbfColor ;
   this->icptr[ctMultiRB].nColor  = this->rbnColor ;
   this->icptr[ctMultiRB].fColor  = this->rbfColor ;
   this->icptr[ctSpecSE].nColor   = this->ddnColor ;
   this->icptr[ctSpecSE].fColor   = this->ddfColor ;
   this->icptr[ctDataDD].nColor   = this->ddnColor ;
   this->icptr[ctDataDD].fColor   = this->ddfColor ;

   //* Initialize attribute array and static array of *
   //* booleans for ctSpecSE Scrollext control.       *
   for ( short i = ZERO ; i < stITEMS ; ++i )
      specAttr[i] = monoColor[1] ;
   specFlags[stSpaceBars]     = false ;
   specFlags[stAudibleAlert]  = true ;
   specFlags[stCustomKeys]    = false ;
   specFlags[stShiftInstr]    = true ;
   specFlags[stMarginBdr]     = true ;
   specFlags[stHeaderLine]    = false ;
   specFlags[stFooterLine]    = false ;
   specFlags[stAltNegColor]   = true ;
   specFlags[stReportStats]   = false ;
   specFlags[stOverlay]       = false ;
   specFlags[stAltPointChar]  = false ;
   specFlags[stInvertPairs]   = false ;
   specFlags[stStressHead]    = false ;
   specFlags[stStressFoot]    = false ;
   specFlags[stStressMargin]  = false ;


   //* Initialize the repeating color pattern for multi-color bars *
   for ( short i = ZERO ; i < Data1Count ; )
   {
      Attr1[i++] = nc.br ;
      Attr1[i++] = nc.cy ;
      Attr1[i++] = nc.br ;
      Attr1[i++] = nc.ma ;
      Attr1[i++] = nc.br ;
      Attr1[i++] = nc.bl ;
      Attr1[i++] = nc.br ;
      Attr1[i++] = nc.bl ;
      Attr1[i++] = nc.br ;
      Attr1[i++] = nc.bl ;
   }

   //* Get the filespec of the current working directory (CWD), *
   //* Initialize the 'helpPath' and 'helpUrl' members. *
   this->InitHelpPath () ;

}  //* End Setup() *

//*************************
//*       BlastOff        *
//*************************
//******************************************************************************
//* Set the configuration parameters and instantiate the Chart-class object.   *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void ChartTest::BlastOff ( void )
{
   //* Enable or disable auto-refresh to test each algorithm *
   #define AUTO_REFRESH (0)

   Chart *cp = NULL ;      // pointer to Chart instance
   gString cVersion,       // Chart-class version
           gs ;            // text formatting
   wkeyCode wk ;           // user key input
   winPos  wpOffset ;      // Y/X offset into footer area
   int   spinVal ;         // spinner values
   short trgRows, trgCols ;// receives dimensions of the target area
   bool  rbState ;         // state of target Radiobutton control

   //* Update the configuration parameters *
   this->SetChartDefinition () ;

   //* If Chart is to be drawn in the main dialog, clear the target area.*
   //* If Chart is to be drawn in a child dialog, save the parent dialog.*
   if ( this->cdef.dPtr != NULL )
      this->ClearTargetArea () ;
   else
      this->dp->SetDialogObscured () ;

   #if AUTO_REFRESH == 0
   //* Delay display refresh until after post-instantiation adjustments     *
   //* This would be the normal sequence, and will avoid multiple refreshes *
   //* which would appear as screen flickering. If no adjustments are       *
   //* needed, then refresh could occur immediately, as shown below.        *
   cp = new Chart( cdef ) ;

   #else
   //* Instantiate the object and refresh the display immediately.*
   cp = new Chart( this->cdef, true ) ;

   #endif   // AUTO_REFRESH

   //* Perform specified post-instantiation setup *
   if ( ! specFlags[stAudibleAlert] )  // silence the warning beep
      cp->AudibleShift( false ) ;
   if ( specFlags[stAltPointChar] )    // alternate Cartesian character
      cp->SetCartesianChar ( spongeBob ) ;
   //* Apply a secondary dataset (for Cartesian charts only) *
   if ( specFlags[stOverlay] && (cdef.chType == ctCartesian) )
   {
      cp->OverlayCartesianDataset ( CartOverlayData, CartOverlayCount, this->chnColor, 
         idtDouble, (this->cdef.cartChar == dblDiamond ? spongeBob : dblDiamond ) ) ;
   }

   //* If user specified margin text _with_ margin border, *
   //* this requires an explicit post-instantiation call.  *
   this->dp->GetRadiobuttonState ( ctMtextRB, rbState ) ;
   if ( rbState && specFlags[stMarginBdr] )
      cp->DrawMarginText ( marginText, this->cdef.textColor, true ) ;

   //* Get the Chart-class version number. (displayed below) *
   cVersion.compose( " \nChart v:%s", cp->GetVersion() ) ;

   //* Make the chart visible *
   cp->RefreshDisplay() ;

   //* Exercise the post-instantiation methods for writing   *
   //* Header, Footer and Margin text.                       *
   //* Default header/footer/margin text (if any) are written*
   //* during instantiation, and the data-shift instructions *
   //* are written to the footer below. However, stress-     *
   //* testing the positioning and potential truncation of   *
   //* text written to these areas must be done through the  *
   //* post-instantiation write methods.                     *
   //* Determine which, if any stress test has been selected.*
   wchar_t targetArea = L'x' ;
   if ( specFlags[stStressHead] )         targetArea = L'H' ;
   else if ( specFlags[stStressFoot] )    targetArea = L'F' ;
   else if ( specFlags[stStressMargin] )  targetArea = L'M' ;

   if ( targetArea != L'x' )     // if a stress test is active
   {
      //* For footer test, disable display of data-shift *
      //* instructions to avoid conflict with test.      *
      if ( targetArea == L'F' )
      {
         specFlags[stShiftInstr] = false ;
         gs = specTextSrc[stShiftInstr] ;
         gs.shiftChars( -1 ) ;
         gs.insert( nckSPACE ) ;
         gs.copy( specTextSrc[stShiftInstr], specSRC ) ;
         this->dp->RefreshScrollextText ( ctSpecSE, false ) ;
      }

      //* Perform the test *
      this->StressedHead2Foot ( cp, targetArea ) ;
   }

   //* If header is not being stress-tested, AND *
   //* if header dividing line was specified, OR *
   //* if statistics report was specified...     *
   if ( (targetArea != L'H') && 
        (specFlags[stHeaderLine] || specFlags[stReportStats]) )
   {
      //* If header text is specified, write text with    *
      //* divider line, else draw divider with no text.   *
      //* Note: the divider will be drawn in border color.*
      this->dp->GetRadiobuttonState ( ctHtextRB, rbState ) ;
      if ( specFlags[stReportStats] )
         this->DisplayStats ( cp ) ;
      else
      {
         gs = (rbState ? headText : "") ;
         cp->DrawHeaderText( gs.ustr(), this->cdef.textColor, true, true ) ;
      }
   }
   //* If footer is not being stress-tested, AND *
   //* if footer dividing line was specified...  *
   if ( (targetArea != L'F') && (specFlags[stFooterLine]) )
   {
      //* If footer text is specified, write text with    *
      //* divider line, else draw divider with no text.   *
      //* Note: the divider will be drawn in border color.*
      this->dp->GetRadiobuttonState ( ctFtextRB, rbState ) ;
      cp->DrawFooterText( (rbState ? footText : ""), 
                          this->cdef.textColor, true, true ) ;
   }

   //* Display data-shift instructions at offset from footer position.*
   //* If insufficient space, the instructions will not be written.   *
   if ( specFlags[stShiftInstr] && this->cdef.chType != ctCartesian )
   {
      cp->GetFooterDimensions ( trgRows, trgCols ) ;
      gs = (specFlags[stCustomKeys] ? ShiftInstrCust : ShiftInstrDflt) ;
      if ( (trgRows >= siHEIGHT + 3) ||
           (!specFlags[stFooterLine] && (trgRows >= (siHEIGHT + 2))) )
      {
         wpOffset = { short(specFlags[stFooterLine] ? 3 : 2), 
                      short(trgCols - siINSET) } ;
         cp->Add2FooterText ( wpOffset, gs.ustr(), this->chnColor, true ) ;
      }
   }

   //* Add horizontal divider line to footer area *
   this->dp->GetSpinnerValue ( ctHlineSP, spinVal ) ;
   if ( spinVal > minHLINE )     // if positive offset specified
   {
      cp->GetFooterDimensions ( trgRows, trgCols ) ;
      if ( trgRows > ZERO )
      {
         if ( !(cp->DrawDivider ( 
                     (this->cdef.drawBorder ? this->cdef.borderStyle : ncltSINGLE),
                     spinVal, this->cdef.borderColor, true )) )
         {
            //* If Chart object created its own dialog, save it before     *
            //* writing to the application's main dialog. (restored below).*
            if ( this->cdef.dPtr == NULL )
               cp->ProtectDialog () ;
            this->statMsg ( " \nDivider-line Offset Out-of-Range!", true, true, 1 ) ;
            if ( this->cdef.dPtr == NULL )
            { this->dp->SetDialogObscured () ; cp->RestoreDialog () ; }
         }
      }
      else
      {
         //* If Chart object created its own dialog, save it before     *
         //* writing to the application's main dialog. (restored below).*
         if ( this->cdef.dPtr == NULL )
            cp->ProtectDialog () ;
         this->statMsg ( " \nFooter Area Rows == ZERO.", true, true, 1 ) ;
         if ( this->cdef.dPtr == NULL )
         { this->dp->SetDialogObscured () ; cp->RestoreDialog () ; }
      }
   }

   //* Call data-shift method with specified keycode definitions.  *
   //* -- Call will return when user presses a key which is not    *
   //*    processed by the called method.                          *
   //* -- If Cartesian chart type OR if all data are currently     *
   //*    displayed, call returns immediately without user         *
   //*    interaction.                                             *
   //* -- The ALT+SHIFT+P (screen capture) is handled locally.     *
   //* -- The CTRL+R key (reload dataset) is handled locally.      *

   //* Create a working copy of the dataset parameters.*
   const void *currData  = this->cdef.dataPtr ;
   int32_t     currCount = this->cdef.dataCount ;
   idataType   currType  = this->cdef.dataType ;
   bool        currHzb   = this->cdef.horizBars ;

   if ( ! specFlags[stCustomKeys] )
   {
      while ( true )
      {
         cp->ShiftData ( wk ) ;     // call user-interface method

         //* If all data is currently displayed,  *
         //* nullchar is returned. Get user input.*
         if ( wk.key == nckNULLCHAR )
            this->dp->GetKeyInput ( wk ) ;

         //* CTRL+R : Reload dataset.                        *
         //* Switch between two datasets to demonstrate that *
         //* reload is working, and if active display stats  *
         //* on the dataset.                                 *
         if ( (wk.type == wktFUNKEY) && (wk.key == nckC_R) )
         {
            this->SwapDatasets ( currData, currCount, currType, currHzb ) ;
            cp->ReloadDataset ( currData, currCount, currType, currHzb ) ;
            this->DisplayStats ( cp ) ;
         }

         //* ALT+SHIFT+P : Perform screen capture *
         else if ( (wk.type == wktEXTEND) && (wk.key == nckAS_P) )
            this->ScreenCapture ( cp ) ;

         //* Else, terminate the loop. *
         else
            break ;
      }
   }     // default definitions
   else
   {
      const short sDEFS = 10 ;
      ShiftDef sdef[sDEFS] =
      { // wk.key wk.type    sb enum      shftcnt
         { {L'f', wktPRINT}, sbFirstPage, ZERO },
         { {L'l', wktPRINT}, sbLastPage,  ZERO  },
         { {L'n', wktPRINT}, sbNextPage,  ZERO  },
         { {L'p', wktPRINT}, sbPrevPage,  ZERO  },
         { {L'a', wktPRINT}, sbNoShift,      1  },
         { {L'A', wktPRINT}, sbNoShift,     -1  },
         { {L'b', wktPRINT}, sbNoShift,      5  },
         { {L'B', wktPRINT}, sbNoShift,     -5  },
         { {L'c', wktPRINT}, sbNoShift,     10  },
         { {L'C', wktPRINT}, sbNoShift,    -10  },
      } ;

      while ( true )
      {
         cp->ShiftData ( wk, sDEFS, sdef ) ; // call user-interface method

         //* If all data is currently displayed,  *
         //* nullchar is returned. Get user input.*
         if ( wk.key == nckNULLCHAR )
            this->dp->GetKeyInput ( wk ) ;

         //* CTRL+R : Reload dataset.                        *
         //* Switch between two datasets to demonstrate that *
         //* reload is working, and if active display stats  *
         //* on the dataset.                                 *
         if ( (wk.type == wktFUNKEY) && (wk.key == nckC_R) )
         {
            this->SwapDatasets ( currData, currCount, currType, currHzb ) ;
            cp->ReloadDataset ( currData, currCount, currType, currHzb ) ;
            this->DisplayStats ( cp ) ;
         }

         //* ALT+SHIFT+P : Perform screen capture *
         else if ( (wk.type == wktEXTEND) && (wk.key == nckAS_P) )
            this->ScreenCapture ( cp ) ;

         //* Else, terminate the loop. *
         else
            break ;
      }
   }     // custom definitions

   //* Delete the Chart object *
   delete cp ;
   cp = NULL ;

   //* If Chart was drawn in the main dialog, *
   //* return the area to the dialog color.   *
   //* Else, restore saved display data.      *
   if ( this->cdef.dPtr != NULL )
   {
      this->ClearTargetArea ( true ) ;
      //* If dynamic title change, restore original title text *
      if ( this->cdef.titleText != NULL )
      {
         this->dp->SetDialogTitle ( ChartTestTitle, this->tColor ) ;
         this->dp->RefreshWin () ;
      }
   }
   else
      this->dp->RefreshWin () ;
   this->statMsg ( "" ) ;        // clear the status message

   //* Report the Chart-class version *
   if ( cVersion.gschars() > 1 )
      this->statMsg ( cVersion, true, true ) ;

   #undef AUTO_REFRESH
}  //* End BlastOff() *

//*************************
//*  SetChartDefinition   *
//*************************
//******************************************************************************
//* Scan the dialog control objects and update the Chart-class instantiation   *
//* parameters contained in the 'cdef' member.                                 *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note: In some contexts, certain data-member values are        *
//* range-checked to avoid logical inconsistencies. If the retrieved value is  *
//* out-of-range, a default value is assigned to the data member and the       *
//* source control is set to match that value. It is assumed that focus is     *
//* currently on the "Launch" Pushbutton so there will be no conflict when     *
//* setting the target control's value.                                        *
//******************************************************************************

void ChartTest::SetChartDefinition ( void )
{
   gString gs ;            // status messages
   int   spinVal ;         // spinner values
   short ctrlIndx ;        // index of selected member of list controls
   bool  rbState ;         // radiobutton state

   //* Chart Type *
   ctrlIndx = this->dp->GetDropdownSelect ( ctTypeDD ) ;
   switch ( ctrlIndx )
   {
      case 0: this->cdef.chType = ctLowerLeft ;    break ;
      case 1: this->cdef.chType = ctLowerRight ;   break ;
      case 2: this->cdef.chType = ctUpperLeft ;    break ;
      case 3: this->cdef.chType = ctUpperRight ;   break ;
      case 4: this->cdef.chType = ctLowerCenter ;  break ;
      case 5: this->cdef.chType = ctUpperCenter ;  break ;
      case 6: this->cdef.chType = ctCenterLeft ;   break ;
      case 7: this->cdef.chType = ctCenterRight ;  break ;
      case 8:
         this->cdef.chType    = ctCartesian ;
         this->cdef.dataPtr   = CartData1 ;
         this->cdef.dataCount = CartDataCount ;
         this->cdef.dataType  = idtDouble ;
         break ;
   } ;

   //* Data Type (Cartesian data type set above) *
   if ( this->cdef.chType != ctCartesian )
   {
      ctrlIndx = this->dp->GetDropdownSelect ( ctDataDD ) ;
      switch ( ctrlIndx )
      {
         case 0:     // 200 positive doubles
            this->cdef.dataPtr   = Data1 ;
            this->cdef.dataCount = Data1Count ;
            this->cdef.dataType  = idtDouble ;
            break ;
         case 1:     // 200 positive/negative doubles
            this->cdef.dataPtr   = Data2 ;
            this->cdef.dataCount = Data2Count ;
            this->cdef.dataType  = idtDouble ;
            break ;
         case 2:     // 50 positive/negative floats
            this->cdef.dataPtr   = FloatData ;
            this->cdef.dataCount = IntCount ;
            this->cdef.dataType  = idtFloat ;
            break ;
         case 3:     // 50 integers (signed)
            this->cdef.dataPtr   = sIntData ;
            this->cdef.dataCount = IntCount ;
            this->cdef.dataType  = idtInt_s ;
            break ;
         case 4:     // 50 integers (unsigned)
            this->cdef.dataPtr   = uIntData ;
            this->cdef.dataCount = IntCount ;
            this->cdef.dataType  = idtInt_u ;
            break ;
         case 5:     // 50 long integers (signed)
            this->cdef.dataPtr   = sLongData ;
            this->cdef.dataCount = IntCount ;
            this->cdef.dataType  = idtLong_s ;
            break ;
         case 6:     // 50 long integers (unsigned)
            this->cdef.dataPtr   = uLongData ;
            this->cdef.dataCount = IntCount ;
            this->cdef.dataType  = idtLong_u ;
            break ;
         case 7:     // 50 long-long integers (signed)
            this->cdef.dataPtr   = sLongLongData ;
            this->cdef.dataCount = IntCount ;
            this->cdef.dataType  = idtLongLong_s ;
            break ;
         case 8:     // 50 long-long integers (unsigned)
            this->cdef.dataPtr   = uLongLongData ;
            this->cdef.dataCount = IntCount ;
            this->cdef.dataType  = idtLongLong_u ;
            break ;
         case 9:     // 50 short integers (signed)
            this->cdef.dataPtr   = sShortData ;
            this->cdef.dataCount = IntCount ;
            this->cdef.dataType  = idtShort_s ;
            break ;
         case 10:    // 50 short integers (unsigned)
            this->cdef.dataPtr   = uShortData ;
            this->cdef.dataCount = IntCount ;
            this->cdef.dataType  = idtShort_u ;
            break ;
         case 11:    // 50 bytes (signed)
            this->cdef.dataPtr   = sByteData ;
            this->cdef.dataCount = IntCount ;
            this->cdef.dataType  = idtByte_s ;
            break ;
         case 12:    // 50 bytes (unsigned)
            this->cdef.dataPtr   = uByteData ;
            this->cdef.dataCount = IntCount ;
            this->cdef.dataType  = idtByte_u ;
            break ;
      } ;
   }     // data type

   //* Cartesian data pairs may be used ONLY with Cartesian charts *
   if ( (this->cdef.chType != ctCartesian) && (this->cdef.dataPtr == CartData1) )
   {
      this->cdef.dataPtr = Data1 ;
      this->cdef.dataCount = Data1Count ;
      this->cdef.dataType  = idtDouble ;
      this->dp->SetDropdownSelect ( ctTypeDD, ZERO ) ;
   }

   //* Chart-area height *
   this->dp->GetSpinnerValue ( ctRowSP, spinVal ) ;
   this->cdef.rows = short(spinVal) ;

   //* Chart-area width *
   this->dp->GetSpinnerValue ( ctColSP, spinVal ) ;
   this->cdef.cols = short(spinVal) ;

   //* Chart-area footer *
   this->dp->GetSpinnerValue ( ctFootSP, spinVal ) ;
   this->cdef.footRows = spinVal ;

   //* Chart-area margin (left) *
   this->dp->GetSpinnerValue ( ctMarginSP, spinVal ) ;
   this->cdef.xOffset = spinVal ;

   //* Bar divisions (if zero(0), display bar tips only) *
   this->dp->GetSpinnerValue ( ctBdivSP, spinVal ) ;
   this->cdef.barWidth = short(spinVal) ;

   //* Chart-area border.                         *
   //* Note that the border flag will be ignored  *
   //* if chart is drawn into a child dialog.     *
   this->dp->GetSpinnerValue ( ctBorderSP, spinVal ) ;
   this->cdef.drawBorder = true ;
   switch ( spinVal )
   {
      case 0:        // no border
         this->cdef.borderStyle = ncltSINGLE ;
         this->cdef.drawBorder = false ;
         break ;
      case 1:        // ncltSINGLE
         this->cdef.borderStyle = ncltSINGLE ;     break ;
      case 2:        // ncltDUAL
         this->cdef.borderStyle = ncltDUAL ;       break ;
      case 3:        // ncltSINGLEBOLD
         this->cdef.borderStyle = ncltSINGLEBOLD ; break ;
      case 4:        // ncltDASH2
         this->cdef.borderStyle = ncltDASH2 ;      break ;
      case 5:        // ncltDASH2BOLD
         this->cdef.borderStyle = ncltDASH2BOLD ;  break ;
      case 6:        // ncltDASH3
         this->cdef.borderStyle = ncltDASH3 ;      break ;
      case 7:        // ncltDASH3BOLD
         this->cdef.borderStyle = ncltDASH3BOLD ;  break ;
      case 8:        // ncltDASH4
         this->cdef.borderStyle = ncltDASH4 ;      break ;
      case 9:        // ncltDASH4BOLD
         this->cdef.borderStyle = ncltDASH4BOLD ;  break ;
   } ;

   //* Axis grid format *
   this->dp->GetSpinnerValue ( ctGridSP, spinVal ) ;
   this->cdef.gridStyle = (spinVal == 2 ? ncltDUAL : ncltSINGLE) ;

   //* Main dialog vs. child dialog *
   this->dp->GetRadiobuttonState ( ctIndyRB, rbState ) ;
   if ( rbState )       // set parameters for child dialog
   {
      //* Indicate that Chart object should generate a dialog window*
      cdef.dPtr  = NULL ;
      cdef.ulY   = this->wpOrig.ypos + CDLG_Y ; // terminal offset in Y
      cdef.ulX   = this->wpOrig.xpos + CDLG_X ; // terminal offset in X
      cdef.rows  += 2 ;// child dialog rows    (display area == rows - 2
      cdef.cols  += 2 ;// child dialog columns (display area == cols - 2
      //* Chart area begins at offset 1,1 in the child dialog.*
      //* 'cdef.rows', 'cdef.cols' && 'cdef.footRows members initialized above *
      this->cdef.borderColor = this->cdbColor ;    // child dialog border color

      //* Report chart-area position and dimensions *
      gs.compose( "Child Dialog Dimensions:\n"
                  "ulY:%hd ulX:%hd rows:%hd cols:%hd", 
                  &this->cdef.ulY, &this->cdef.ulX, 
                  &this->cdef.rows, &this->cdef.cols ) ;
   }
   else                 // set parameters for main dialog
   {
      //* Indicate that Chart object will be drawn in the main dialog *
      cdef.dPtr  = this->dp ;
      this->cdef.ulY      = CHART_YOFF ;        // Y origin
      this->cdef.ulX      = CHART_XOFF ;        // X origin
      //* 'cdef.rows', 'cdef.cols' && 'cdef.footRows members initialized above *
      this->cdef.borderColor = this->chbColor ; // chart-area border color

      // Range limit chart-area dimensions when drawing chart in main dialog.*
      if ( this->cdef.cols > dfltWIDTH )
      {
         this->cdef.cols = dfltWIDTH ;
         this->dp->SetSpinnerValue ( ctColSP, this->cdef.cols ) ;
      }

      //* Report chart-area position and dimensions *
      gs.compose( "Chart Area Dimensions:\n"
                  "ulY:%hd ulX:%hd rows:%hd cols:%hd", 
                  &this->cdef.ulY, &this->cdef.ulX, 
                  &this->cdef.rows, &this->cdef.cols ) ;
   }
   this->statMsg ( gs ) ;        // display dimensions

   //* Display Header Text *
   this->dp->GetRadiobuttonState ( ctHtextRB, rbState ) ;
   //* If header text is to be displayed, size the header *
   //* appropriately for the type of test to be performed.*
   if ( rbState )
   {
      if ( specFlags[stHeaderLine] || specFlags[stReportStats] )
      {
         this->cdef.headText = NULL ;  // header text will be written post-instantiation
         this->cdef.yOffset = 4 ;      // three header rows
      }
      else
      {
         this->cdef.headText = headText ; // specify the header text
         this->cdef.yOffset = 3 ;         // two header rows
      }
   }
   else if ( specFlags[stHeaderLine] || 
             specFlags[stStressHead] || specFlags[stReportStats] )
   {
      this->cdef.headText = NULL ;     // no header text
      this->cdef.yOffset = 4 ;         // three header rows
   }
   else
   {
      this->cdef.headText = NULL ;     // no header text
      this->cdef.yOffset = 1 ;         // no header (zero header rows)
   }

   //* Display Margin Text *
   this->dp->GetRadiobuttonState ( ctMtextRB, rbState ) ;
   this->cdef.marginText = (rbState ? marginText : NULL) ;

   //* Display Footer Text *
   this->dp->GetRadiobuttonState ( ctFtextRB, rbState ) ;
   //* If footer text is to be displayed, WITHOUT *
   //* divider line, then draw the footer text    *
   //* during instantiation. Otherwise, footer    *
   //* text plus divider line will be drawn       *
   //* post-instantiation.                        *
   if ( rbState && !specFlags[stFooterLine] )
      this->cdef.footText = footText ;
   else
      this->cdef.footText = NULL ;

   //* Dialog title *
   this->dp->GetRadiobuttonState ( ctTtextRB, rbState ) ;
   if ( rbState )
   {
      if ( cdef.dPtr == NULL )
         this->cdef.titleText = ChildDlgTitle ;
      else
         this->cdef.titleText = ParentDlgTitle ;
   }
   else
      this->cdef.titleText = NULL ;

   //* Horizontal vs. Vertical bars *
   this->dp->GetRadiobuttonState ( ctHbarsRB, rbState ) ;
   this->cdef.horizBars = rbState ;

   //* Multi-color Bars *
   this->dp->GetRadiobuttonState ( ctMultiRB, rbState ) ;
   this->cdef.attrPtr = (rbState ? Attr1 : NULL) ;

   //* Scan 'Special Tests' Flags *
   //* -------------------------- *
   //* Space Between Bars *
   this->cdef.barSpace = (specFlags[stSpaceBars] ? 1 : 0) ;
   //* Alternate color attribute for negative values *
   this->cdef.negColor = (specFlags[stAltNegColor] ? this->chnColor : this->chbColor) ;
   //* Alternate Data-point Character (Cartesian) *
   this->cdef.cartChar = (specFlags[stAltPointChar] ? spongeBob : dblDiamond) ;
   //* X/Y Pairs vs. Y/X Pairs *
   if ( this->cdef.chType == ctCartesian )
      this->cdef.horizBars = specFlags[stInvertPairs] ;
}  //* End SetChartDefinition() *

//*************************
//*    ClearTargetArea    *
//*************************
//******************************************************************************
//* Clear the area of the dialog which will be used to draw the chart.         *
//* This is the area defined for the Chart object PLUS the debug output area.  *
//*                                                                            *
//* The color attribute used is the terminal default (nc.bw) which is assumed  *
//* to have a white or off-white background. If it does not, then the Chart    *
//* may be rendered with visual artifacts.                                     *
//*                                                                            *
//* Input  : restore : (optional, 'false' by default)                          *
//*                    if 'false', clear the area to a white background        *
//*                    if 'true',  clear the area to the original dialog color *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void ChartTest::ClearTargetArea ( bool restore )
{
   gString gs( " " ) ;
   gs.padCols ( CHART_COLS ) ;
   gs.append( L'\n' ) ;
   winPos wp( CHART_YOFF, CHART_XOFF ) ;
   attr_t clearColor = (restore ? this->dColor : nc.bw) ;

   while ( wp.ypos < (dROWS - 1) )
      wp = this->dp->WriteParagraph ( wp, gs, clearColor ) ;
   this->dp->RefreshWin () ;

}  //* End ClearTargetArea() *

//*************************
//*       Cry4Help        *
//*************************
//******************************************************************************
//* Private Method                                                             *
//* --------------                                                             *
//* Ask the user whether the '.info' or '.html' format of the document should  *
//* be displayed. Based on user's selection, open the application Help file.   *
//*  a) Shell out and invoke the 'info' documentation reader.                  *
//*  b) Invoke the default browser with help in HTML format.                   *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Notes:                                                                     *
//* a) This method and the methods is calls are a simplified version of the    *
//*    Exercalc application's algorithm for invoking external help.            *
//*    For this application, the target help file(s) almost certainly exist,   *
//*    but this sequence demonstrates how to validate the target file before   *
//*    attempting to access it.                                                *
//* b) If HTML format is requested, but HTML file is not found, the '.info'    *
//*    document is invoked instead.                                            *
//* c) Invoking the HTML document with 'xdg-open' opens the document at the    *
//*    top by default. To automagically scan to the target chapter, it is      *
//*    necessary to construct the full, absolute filespec with the chapter     *
//*    name appended. Example:                                                 *
//*    /home/sam/SoftwareDesign/NcDialog/Texinfo/ncdialogapi.html#Chart-Widget *
//*    The prefix "file:///" is added to indicate that the target file is a    *
//*    LOCAL file, NOT a remote URL.                                           *
//*                                                                            *
//******************************************************************************

void ChartTest::Cry4Help ( void )
{
   const short cryROWS = 6,
               cryCOLS = 38,
               cryYPOS = (this->wpOrig.ypos + this->icptr[ctTypeDD].ulY),
               cryXPOS = (this->wpOrig.xpos + this->icptr[ctTypeDD].ulX) ;
   enum cry : short { cryInfoPB = ZERO, cryHtmlPB, cryAbortPB, cryITEMS } ;

   attr_t bAttr = nc.brgr,
          dAttr = nc.bkgr ;
   bool htmlFormat = false ;


   InitCtrl ic[cryITEMS] =
   {
   {  //* 'INFO' pushbutton  - - - - - - - - - - - - - - - - - - - - cryInfoPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(cryROWS - 2),           // ulY:       upper left corner in Y
      4,                            // ulX:       upper left corner in X
      1,                            // lines:     control lines
      8,                            // cols:      control columns
      "  INFO  ",                   // dispText:  
      this->pbnColor,               // nColor:    non-focus color
      this->pbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[cryHtmlPB]                // nextCtrl:  link in next structure
   },
   {  //* 'HTML' pushbutton  - - - - - - - - - - - - - - - - - - - - cryHtmlPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[cryInfoPB].ulY,            // ulY:       upper left corner in Y
      short(cryCOLS / 2 - 4),       // ulX:       upper left corner in X
      1,                            // lines:     control lines
      8,                            // cols:      control columns
      "  HTML  ",                   // dispText:  
      this->pbnColor,               // nColor:    non-focus color
      this->pbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[cryAbortPB]               // nextCtrl:  link in next structure
   },
   {  //* 'CANCEL' pushbutton  - - - - - - - - - - - - - - - - - -  cryAbortPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[cryHtmlPB].ulY,            // ulY:       upper left corner in Y
      short(cryCOLS - 12),          // ulX:       upper left corner in X
      1,                            // lines:     control lines
      8,                            // cols:      control columns
      " CANCEL ",                   // dispText:  
      this->pbnColor,               // nColor:    non-focus color
      this->pbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
   } ;

   //* Define the dialog object *
   InitNcDialog dInit( cryROWS,        // number of display lines
                       cryCOLS,        // number of display columns
                       cryYPOS,        // Y offset from upper-left of terminal 
                       cryXPOS,        // X offset from upper-left of terminal 
                       "  Launch Application Help  ", // dialog title
                       ncltDUAL,       // border line-style
                       bAttr,          // border color attribute
                       dAttr,          // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Save the parent dialog *
   this->dp->SetDialogObscured () ;

   //* Instantiate the dialog window *
   NcDialog *dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow ()) == OK )
   {
      winPos wp( 2, 2 ) ;
      dp->WriteParagraph ( wp, "Please select Help document format.", dAttr, true ) ;

      //* Ask user which format to use *
      uiInfo Info ;                 // user interface data returned here
      short  icIndex = ZERO ;       // index of control with input focus
      bool   done = false,          // loop control
             abort = false ;        // 'true' if user aborts

      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey != false )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == cryInfoPB )
                  htmlFormat = false ;
               else if ( Info.ctrlIndex == cryHtmlPB )
                  htmlFormat = true ;
               else if ( Info.ctrlIndex == cryAbortPB) 
                  abort = true ;
               done = true ;
            }
         }

         //* Move input focus to next/previous control.*
         if ( done == false && Info.viaHotkey == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }  // while()

      //* If user has made a selection *
      if ( ! abort )
      {
         if ( htmlFormat && (*this->helpUrl != NULLCHAR) )  // help (HTML format)
         {
            pid_t fpid = vfork () ;                // create the child process

            if ( fpid == ZERO)
            {
               execlp ( "xdg-open", "xdg-open", this->helpUrl, NULL ) ;
               //* In case the exec call fails: child process MUST NOT return.*
               _exit ( 0 ) ;
            }
            else
            {  //* The parent process continues execution here *
               if ( fpid > ZERO )      // child successfully launched
               { /* Currently, nothing to be done. */ }
            }
         }
         else if ( *this->helpPath != NULLCHAR )            // help (info format)
         {
            gString gs( "info -f \"%s\" -n \"Chart Widget\"", this->helpPath ) ;
            this->dp->ShellOut ( soX, gs.ustr() ) ;
         }
         else                    // Error: help file not found
         {
            gString gs( "  Sorry, Help file was not found.  \n"
                        "  \"../Texinfo/ncdialogapi.info\"" ) ;
            dp->WriteParagraph ( wp, gs, dAttr, true ) ;
            this->dp->UserAlert ( 2 ) ;
            chrono::duration<short>aWhile( 3 ) ;
            this_thread::sleep_for( aWhile ) ;
         }
      }  // (!abort)
   }     // dialog opened
   if ( dp != NULL )             // close the window
      delete ( dp ) ;

   //* Restore parent dialog *
   this->dp->RefreshWin () ;

}  //* End Cry4Help() *

//****************************
//* LaunchDefaultApplication *
//****************************
//******************************************************************************
//* Launch the system default application for handling the specified file type.*
//*                                                                            *
//* Input  : targFile : filespec of file to be processed by the external       *
//*                     application                                            *
//*          redirect : (optional, 'true' by default)                          *
//*                     if 'true',  redirect stdout and stderr to temp files   *
//*                     if 'false', allow stdout and stderr to write to the    *
//*                                 terminal window                            *
//*                                                                            *
//* Returns: a) the child process ID                                           *
//*          b) -1 if child process not created                                *
//******************************************************************************
//* Notes:                                                                     *
//* ------                                                                     *
//* 1) fork a new process: vfork()                                             *
//* 2) Redirection:                                                            *
//*    a) It is possible that the called application will write to the terminal*
//*       window, either during startup or with some later status message.     *
//*    b) If the called application is a console application, this is almost   *
//*       certain; while for external GUI applications, it is still likely to  *
//*       some extent.                                                         *
//*    c) If the parent application has relinquished control of stdout and     *
//*       stderr. i.e. the NCurses engine has been put into hibernation mode,  *
//*       then the called application can display messages without conflict.   *
//*    d) If, however, the parent application is still in active control of    *
//*       stdout and stderr, then resource conflicts are likely to occur.      *
//*       In this case, it is necessary to redirect the called application's   *
//*       output. Our solution is to redirect stdout and stderr to temporary   *
//*       files by default. These files will persist until the parent          *
//*       application closes. (To prevent this, call the method with           *
//*       'redirect' set to 'false'.) The Taggit-class destructor will delete  *
//*       these temporary files during application shutdown.                   *
//* 3) Use the new process to launch the application:                          *
//*    a) This method uses the 'xdg-open' system utility (shell script) which  *
//*       launches the application associated with the file type of 'targFile'.*
//*    b) execlp searches the path looking for 'xdg-open'                      *
//*       Because we are fairly safe in assuming that 'xdg-open' actually IS   *
//*       on the path, the first argument is the name of the script (no path). *
//*    c) The second argument is argv[0] i.e. the script name again.           *
//*    d) The third argument is argv[1] i.e. the filespec passed in by caller. *
//*    e) The fourth argument is argv[2] i.e. a null pointer                   *
//*    f) Note that xdg-open passes only the specified filespec to the target  *
//*       application, and cannot pass additional configuration arguments.     *
//*       To invoke the target application will additional setup/configuration *
//*       parameters, please see the LaunchExternalApplication() method, below.*
//*    g) The child process is not directly associated with the terminal       *
//*       window from which it was launched, so the window may be closed       *
//*       asynchronously.                                                      *
//*    h) The exec functions do not return. The child process terminates when  *
//*       the external application exits.                                      *
//* 4) primary process returns to caller                                       *
//*                                                                            *
//* Programmer's Note: This sequence assumes that multi-threading is not       *
//* active. For multi-threaded applications, use pthread_fork() instead.       *
//******************************************************************************

int ChartTest::LaunchDefaultApplication ( const char* targFile, bool redirect )
{
   #define ENABLE_REDIRECT (0)
   #if ENABLE_REDIRECT != 0
   gString stdoutTemp, stderrTemp ;       // stdout target file
   this->CreateTempname ( stdoutTemp ) ;  // stderr target file
   this->CreateTempname ( stderrTemp ) ;  // create the temp filenames
   //int soDesc = ZERO,                     // stdout target descriptor
   //    seDesc ;                           // stderr target descriptor
   #endif   // ENABLE_REDIRECT

   pid_t fpid = vfork () ;                // create the child process

   if ( fpid == ZERO)
   {
      // Programmer's Note: This is a generalized launch method, and can do 
      // considerably more than this application requires. Because this 
      //* application calls this method ONLY for opening the HTML help file 
      // in a browser, there is no need to use redirect.
      #if ENABLE_REDIRECT != 0
      //* The child process (if created) executes here.                  *
      //* Create temp files as targets for redirecting stdout and stderr.*
      if ( redirect )
      {
         soDesc = open ( stdoutTemp.ustr(), O_WRONLY | O_CREAT | O_TRUNC, 
                           S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH ) ;
         seDesc = open ( stderrTemp.ustr(), O_WRONLY | O_CREAT | O_TRUNC, 
                           S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH ) ;
         dup2 ( soDesc, STDOUT_FILENO) ;
         dup2 ( seDesc, STDERR_FILENO) ;
         close ( soDesc ) ;
         close ( seDesc ) ;
      }
      #endif   // ENABLE_REDIRECT

      execlp ( "xdg-open", "xdg-open", targFile, NULL ) ;

      //* In case the exec call fails: child process MUST NOT return.*
      _exit ( 0 ) ;
   }
   else
   {  //* The parent process continues execution here *
      if ( fpid > ZERO )      // child successfully launched
      {
         /* Currently, nothing to be done. */
      }
   }
   return fpid ;

}  //* End LaunchDefaultApplication() *

//*************************
//*        statMsg        *
//*************************
//******************************************************************************
//* Private Method                                                             *
//* --------------                                                             *
//* Write a status message to the status textbox control.                      *
//*                                                                            *
//* Input  : msg   : text to be written                                        *
//*          bold  : (optional, 'false' by default)                            *
//*                  if 'false', display text in standard color                *
//*                  if 'true',  display text in bold color                    *
//*          center: (optional, 'true' by default)                             *
//*                  if 'true',  center the message in the textbox control     *
//*                  if 'false', display the message as written                *
//*          warn  : (optional, ZERO by default)                               *
//*                  if > ZERO, flash the message and beep specified count     *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void ChartTest::statMsg ( gString& msg, bool boldAttr, bool center, short warn )
{
   const attr_t focusAttr  = dtbmFcolor ;
   const attr_t nfocusAttr = dtbmNFcolor ;

   dtbmData uData( msg.gstr(), (boldAttr ? &focusAttr : &nfocusAttr), true, center ) ;

   //* If specified, flash the message *
   if ( warn > ZERO )
   {
      //* Save the specified color attribute*
      const attr_t tmpAttr = uData.colorData[ZERO] ;
      //* Flash the message using the alternate color *
      uData.colorData[0] = (boldAttr ? nfocusAttr : focusAttr) ;
      this->dp->DisplayTextboxMessage ( ctStatTB, uData ) ;
      this->dp->UserAlert ( warn ) ;         // make some noise
      chrono::duration<short>aWhile( 1 ) ;   // pause for a moment
      this_thread::sleep_for( aWhile ) ;
      uData.colorData[0] = tmpAttr ;         // restore the color attribute
   }
   this->dp->DisplayTextboxMessage ( ctStatTB, uData ) ;

}  //* End statMsg() *

void ChartTest::statMsg ( const char* msg, bool boldAttr, bool center, short warn )
{
   gString gs( msg ) ;
   this->statMsg ( gs, boldAttr, center, warn ) ;

}  //* End statMsg() *

//*************************
//*     GetDimensions     *
//*************************
//******************************************************************************
//* Returns the rows/columns needed to open the main dialog.                   *
//* If dialog does not open, caller can retrive and report the terminal-window *
//* dimensions required for this test.                                         *
//*                                                                            *
//* Input  : rowsNeeded : (by reference) receives number of rows required      *
//*          colsNeeded : (by reference) receives number of columns required   *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void ChartTest::GetDimensions ( short& rowsNeeded, short& colsNeeded )
{

   rowsNeeded = dROWS ;
   colsNeeded = dCOLS ;

}  //* End GetDimensions() *

//*************************
//*     InitHelpPath      *
//*************************
//******************************************************************************
//* Initialize the 'helpPath' and 'helpUrl' members.                           *
//* These point to the NcDialog API documentation, in info-reader format and   *
//* HTML format, respectively. If file not found, then the member is           *
//* initialized to the NULL string.                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void ChartTest::InitHelpPath ( void )
{
   // Programmer's Note: POSIX defines 'PATH_MAX' as 4096 bytes, but unfortunately,
   // the compiler defines the default buffer size for getcwd() as 4100 bytes.
   // We love the GNU compiler, but sometimes the developers fail to think clearly.
   const short MAX_PATH = PATH_MAX + 16 ; //* Size of buffer to hold path/filename

   const char* infoSpec  = "Texinfo/ncdialogapi.info" ;
   const char* htmlSpec  = "Texinfo/ncdialogapi.html" ;
   const char* urlPrefix = "file:///" ;
   const char* urlOffset = "#Chart-Widget" ;

   gString gsBase,               // text formatting
           gs ;
   char cwd[MAX_PATH] ;          // current working directory

   //* Initialize the member variables *
   *this->helpPath = NULLCHAR ;
   *this->helpUrl  = NULLCHAR ;

   //* Get the current working directory *
   if ( (getcwd ( cwd, MAX_PATH )) == cwd )
   {
      gsBase = cwd ;
      short indx = gsBase.findlast( L'/' ) ;
      if ( indx > ZERO )
      {
         fmFType ft ;
         gsBase.limitChars( indx + 1 ) ;

         //* Construct path to '.info' document *
         gs.compose( "%S%s", gsBase.gstr(), infoSpec ) ;
         if ( (TargetExists ( gs, ft )) && (ft == fmREG_TYPE) )
            gs.copy( this->helpPath, gsDFLTBYTES ) ;

         //* Construct path to '.html' document *
         gs.compose( "%S%s", gsBase.gstr(), htmlSpec ) ;
         if ( (TargetExists ( gs, ft )) && (ft == fmREG_TYPE) )
         {
            gs.insert( urlPrefix ) ;
            gs.append( urlOffset ) ;
            gs.copy( this->helpUrl, gsDFLTBYTES ) ;
         }
      }
   }

}  //* End InitHelpPath() *

//*************************
//*     DisplayStats      *
//*************************
//******************************************************************************
//* If flag is set for display of stats on dataset, retrieve the stats and     *
//* display them in the header area.                                           *
//*                                                                            *
//* Input  : cp  : pointer to an active Chart-class object                     *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void ChartTest::DisplayStats ( Chart *cp )
{
   if ( specFlags[stReportStats] )
   {
      gString gs ;      // text formatting

      //* Get statistics on current dataset *
      ChartStats cStats ;
      cp->GetStats ( cStats ) ;

      //* Format and display statistics for Cartesian charts.*
      if ( this->cdef.chType == ctCartesian )
      {
         gs.compose( "    Statistics: minX:%-6.2lf maxX:%5.2lf meanX:%.2lf medianX:%.2lf\n"
                     "     pairs:%-3d  minY:%-6.2lf maxY:%5.2lf meanY:%.2lf medianY:%.2lf",
                     &cStats.cartRange.minValX,  &cStats.cartRange.maxValX, 
                     &cStats.cartRange.meanValX, &cStats.cartRange.medianValX, 
                     &cStats.dataItems,
                     &cStats.cartRange.minValY,  &cStats.cartRange.maxValY, 
                     &cStats.cartRange.meanValY, &cStats.cartRange.medianValY ) ;
      }

      //* Format and display the statistics for non-Cartesian chart types.*
      else
      {
         gs.compose( "     Statistics: dataItems:%d  minValue:%.2lf  maxValue:%.2lf\n"
                     "                 meanValue(avg):%.2lf  medianValue:%.2lf",
                     &cStats.dataItems, &cStats.minValue, &cStats.maxValue,
                     &cStats.meanValue, &cStats.medianValue ) ;
      }
      cp->DrawHeaderText( gs.ustr(), this->cdef.textColor, true, true ) ;
   }

}  //* End DisplayStats() *

//*************************
//*     SwapDatasets      *
//*************************
//******************************************************************************
//* Switch 'cdef' values between two similar datasets.                         *
//* Called within the BlastOff() loop to test the Chart-class                  *
//* ReloadDataset() method.                                                    *
//*                                                                            *
//* Notes:                                                                     *
//* -- There are two datasets for doubles: Data1 and Data2                     *
//* -- For the integer types, switch between the signed and unsigned datasets. *
//*    Note that all integer datasets have the same number of elements.        *
//* -- There is only one single-precision floating-point dataset, so the swap  *
//*    is to Data1 (doubles).                                                  *
//* -- 'currHzb' is modified ONLY for Cartesian charts. Used to specify        *
//*    X/Y pairs (false) vs. Y/X pairs (true).                                 *
//*                                                                            *
//* Input  : currData  : pointer to current dataset                            *
//*                      receives pointer to alternate dataset                 *
//*          currCount : number of items in current dataset                    *
//*                      receives number of items in alternate dataset         *
//*          currType  : data type of current dataset                          *
//*                      receives data type of alternate dataset               *
//*          currHzb   : bar orientation for current dataset                   *
//*                      ('false' == vertical, 'true' == horizontal)           *
//*                                                                            *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void ChartTest::SwapDatasets ( const void *& currData, int32_t& currCount, 
                               idataType& currType, bool& currHzb )
{
   if ( (const double*)currData == Data1 )
   { currData = (const void*)Data2 ; currCount = Data2Count ; }
   else if ( ((const double*)currData == Data2) || ((const float*)currData == FloatData) )
   { currData = (const void*)Data1 ; currCount = Data1Count ; currType = idtDouble ; }
   else if ( (const char*)currData == sByteData )
   { currData = (const void*)uByteData ; currType = idtByte_u ; }
   else if ( (const unsigned char*)currData == uByteData )
   { currData = (const void*)sByteData ; currType = idtByte_s ; }
   else if ( (const short*)currData == sShortData )
   { currData = (const void*)uShortData ; currType = idtShort_u ; }
   else if ( (const unsigned short*)currData == uShortData )
   { currData = (const void*)sShortData ; currType = idtShort_s ; }
   else if ( (const int*)currData == sIntData )
   { currData = (const void*)uIntData ; currType = idtInt_u ; }
   else if ( (const unsigned int*)currData == uIntData )
   { currData = (const void*)sIntData ; currType = idtInt_s ; }
   else if ( (const long*)currData == sLongData )
   { currData = (const void*)uLongData ; currType = idtLong_u ; }
   else if ( (const unsigned long*)currData == uLongData )
   { currData = (const void*)sLongData ; currType = idtLong_s ; }
   else if ( (const long long*)currData == sLongLongData )
   { currData = (const void*)uLongLongData ; currType = idtLongLong_u ; }
   else if ( (const unsigned long long*)currData == uLongLongData )
   { currData = (const void*)sLongLongData ; currType = idtLongLong_s ; }
   //* For Cartesian data, switch between X/Y pairs and Y/X pairs *
   else if ( (const double*)currData == CartData1 )
      currHzb = currHzb ? false : true ;

   //* In case caller is stupid *
   else
   { currData = (const void*)Data1 ; currCount = Data1Count ; currType = idtDouble ; }

}  //* End SwapDatasets() *

//*************************
//*   StressedHead2Foot   *
//*************************
//******************************************************************************
//* Perform various positioning and overflow tests of writes to the Header     *
//* area, Footer area and Margin area.                                         *
//*                                                                            *
//* Input  : cp         : pointer to an active Chart-class object              *
//*          targetArea : indicates which area is the target of the test:      *
//*                       'H' == Header                                        *
//*                       'F' == Footer                                        *
//*                       'M' == MARGIN                                        *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void ChartTest::StressedHead2Foot ( Chart *cp, wchar_t targetArea )
{
   const short HEADER_TESTS = 6 ;      // number of header tests
   const short FOOTER_TESTS = 5 ;      // number of footer tests
   const short MARGIN_TESTS = 7 ;      // number of margin tests

   const char* starWarts = "Once upon a time, in a galaxy far, far away,\n"
                           "                Star Wars!" ;
   const char* princessLaid1 = "Princess Leia" ;
   const char* princessLaid2 = " is so " ;
   const char* princessLaid3 = "HOT!" ;
   const char* theFarce    = "There is a disturbance in the Force!  " ;
   const char* wheatGrass  = "Luke, your father is a vegetarian!" ;
   const char* whatU_Say   = "A vegetable? What happened?  " ;
   const char* jediJumble  = "No, a vegetarian, you ...\n"
                             "wretched Jedi nit!" ;
   const char* way2goTico  = "            Ah, balance has been restored to the Force!\n"
                             "                    Tico, you've done it again!" ;
   const char* Marplotter1 = "Definition of \"marplotter\":\n"
                             "One who, by meddling, mars or\n"
                             "frustrates the designs or plans\n"
                             "of others.";
   const char* Marplotter2 = "Synonyms:\n" ;
   const char* Marplotter3 = " Douchbag," ;
   const char* Marplotter4 = " Duckhead," ;
   const char* Marplotter5 = " Uranus." ;
   const char* Marplotter6 = "Example:\n" ;
   const char* Marplotter7 = "\"The Franklins, and other\n"
                             "marplotters in the Potomac Army,\n"
                             "menace to resign if Hooker is put\n"
                             "in command. The sooner the better\n"
                             "for the army to get rid of such\n"
                             "trash. But the imbeciles and the\n"
                             "intriguers in power think not so;\n"
                             "and all may remain as it was, and\n"
                             "a new slaughter of our heroes may\n"
                             "loom in the future.\"\n"
                             " -- Adam Gurowski (1863)" ;

   gString gs ;
   winPos  wp ;
   chrono::duration<short, std::milli>aMoment( 400 ) ;
   const attr_t c1 = nc.blgr | ncbATTR ;
   const attr_t c2 = nc.regr | ncbATTR ;
   const attr_t c3 = nc.magr ;
   const attr_t c4 = nc.bkgr ;
   short statusBeeps ;

   if ( targetArea == L'H' )     // test header messages
   {
      short  hdrRows, hdrCols ;
      cp->GetHeaderDimensions ( hdrRows, hdrCols ) ;
      for ( short i = 1 ; i <= HEADER_TESTS ; ++i )
      {
         statusBeeps = 1 ;

         switch ( i )
         {
            case 1:     // Test #1 - clear the area and write a message on first line
               {
               short indx ;
               gs = starWarts ;
               gs.limitChars( (indx = gs.find( L'\n') ) ) ;
               gs.padCols( hdrCols, L' ', true ) ;
               if ( !(cp->DrawHeaderText ( gs.ustr(), c1, false, true )) )
                  statusBeeps = 3 ;
               }
               break ;
            case 2:     // Test #2 - add text, write to second line
               wp = { 1, 0 } ;
               wp = cp->Add2HeaderText ( wp, theFarce, c4, true ) ;
               if ( (wp.ypos == ZERO) && (wp.xpos == ZERO) )
                  statusBeeps = 3 ;
               break ;
            case 3:     // Test #3 violates the right edge
               wp = cp->Add2HeaderText ( wp, wheatGrass, c2, true ) ;
               if ( (wp.ypos == ZERO) && (wp.xpos == ZERO) )
                  statusBeeps = 3 ;

               //* If third line not available, abort remaining tests *
               if ( ! specFlags[stHeaderLine] )
                  i = HEADER_TESTS - 1 ;
               break ;
            case 4:     // Test #4 writes a partial 3rd line
               wp = { short(hdrRows - 1), ZERO } ;
               wp = cp->Add2HeaderText ( wp, whatU_Say, c1, true ) ;
               if ( (wp.ypos == ZERO) && (wp.xpos == ZERO) )
                  statusBeeps = 3 ;
               break ;
            case 5:     // Test #5 violates the lower edge
               wp = cp->Add2HeaderText ( wp, jediJumble, c2, true ) ;
               if ( (wp.ypos == ZERO) && (wp.xpos == ZERO) )
                  statusBeeps = 3 ;
               break ;
            case 6:     // Test #6 restores balance to the Force
               if ( !(cp->DrawHeaderText ( way2goTico, c1, true, true )) )
                  statusBeeps = 3 ;
               break ;
            default:    break ;
         } ;
         this->dp->UserAlert ( statusBeeps ) ;
         this_thread::sleep_for( aMoment ) ;
      }
   }     // header

   else if ( targetArea == L'F' )   // test footer messages
   {
      //* Get dimension of footer area *
      short  ftrRows, ftrCols ;
      cp->GetFooterDimensions ( ftrRows, ftrCols ) ;

      for ( short i = 1 ; i <= FOOTER_TESTS ; ++i )
      {
         statusBeeps = 1 ;

         switch ( i )
         {
            case 1:     // clear the footer area and write a standard message
               if ( !(cp->DrawFooterText ( starWarts, c1, false, true )) )
                  statusBeeps = 3 ;
               break ;
            case 2:     // add multi-colored text to existing message
               wp = { 2, 0 } ;
               wp = cp->Add2FooterText ( wp, princessLaid1, c1, true ) ;
               if ( (wp.ypos == ZERO) && (wp.xpos == ZERO) )
                  statusBeeps = 3 ;
               else
               {
                  wp = cp->Add2FooterText ( wp, princessLaid2, c2, true ) ;
                  if ( (wp.ypos == ZERO) && (wp.xpos == ZERO) )
                     statusBeeps = 3 ;
                  else
                  {
                     wp = cp->Add2FooterText ( wp, princessLaid3, c3, true ) ;
                     if ( (wp.ypos == ZERO) && (wp.xpos == ZERO) )
                        statusBeeps = 3 ;
                  }
               }
               break ;
            case 3:     // overrun the right edge of footer area
                        // Note: footer area s/b at least 5, else 
                        // bottom edge may also be overrun.
               ++wp.ypos ; wp.xpos = ftrCols / 2 ;
               wp = cp->Add2FooterText ( wp, starWarts, c1, true ) ;
               if ( (wp.ypos == ZERO) && (wp.xpos == ZERO) )
                  statusBeeps = 3 ;
               break ;
            case 4:  // send a negative offset (error if non-zero position returned)
               wp = { -1, -5 } ;
               wp = cp->Add2FooterText ( wp, "Off in the Weeds!", c3, true ) ;
               if ( (wp.ypos != ZERO) || (wp.xpos != ZERO) )
                  statusBeeps = 3 ;
               else
                  continue ;     // back to top of loop
               break ;
            case 5:     // overrun the bottom edge of footer area
               wp = { short(ftrRows - 1), 2 } ;
               wp = cp->Add2FooterText ( wp, starWarts, c2, true ) ;
               if ( (wp.ypos == ZERO) && (wp.xpos == ZERO) )
                  statusBeeps = 3 ;
               break ;
            default:
               break ;
         } ;
         this->dp->UserAlert ( statusBeeps ) ;
         this_thread::sleep_for( aMoment ) ;
      }
   }     // footer

   else if ( targetArea == L'M' )   // test margin messages
   {
      //* Get dimension of margin area *
      short  marRows, marCols ;
      cp->GetMarginDimensions ( marRows, marCols ) ;

      for ( short i = 1 ; i <= MARGIN_TESTS ; ++i )
      {
         statusBeeps = 1 ;
         switch ( i )
         {
            case 1:
               if ( !(cp->DrawMarginText ( Marplotter1, c1, true, true )) )
               {
                  statusBeeps = 3 ;
                  i = MARGIN_TESTS ;   // do not perform remaining tests
               }
               break ;
            case 2:
               wp = { 6, 0 } ;
               wp = cp->Add2MarginText ( wp, Marplotter2, c1 | ncuATTR, true ) ;
               if ( (wp.ypos == ZERO) && (wp.xpos == ZERO) )
                  statusBeeps = 3 ;
               break ;
            case 3:
               wp = cp->Add2MarginText ( wp, Marplotter3, c2, true ) ;
               if ( (wp.ypos == ZERO) && (wp.xpos == ZERO) )
                  statusBeeps = 3 ;
               break ;
            case 4:
               wp = cp->Add2MarginText ( wp, Marplotter4, c4, true ) ;
               if ( (wp.ypos == ZERO) && (wp.xpos == ZERO) )
                  statusBeeps = 3 ;
               break ;
            case 5:
               wp = cp->Add2MarginText ( wp, Marplotter5, c1, true ) ;
               if ( (wp.ypos == ZERO) && (wp.xpos == ZERO) )
                  statusBeeps = 3 ;
               break ;
            case 6:
               wp = { short(wp.ypos + 2), 0 } ;
               wp = cp->Add2MarginText ( wp, Marplotter6, c4 | ncuATTR, true ) ;
               if ( (wp.ypos == ZERO) && (wp.xpos == ZERO) )
                  statusBeeps = 3 ;
               break ;
            case 7:
               wp = cp->Add2MarginText ( wp, Marplotter7, c1, true ) ;
               if ( (wp.ypos == ZERO) && (wp.xpos == ZERO) )
                  statusBeeps = 3 ;
               break ;
            default:
               break ;
         } ;
         this->dp->UserAlert ( statusBeeps ) ;
         this_thread::sleep_for( aMoment ) ;
      }
   }     // margin

}  //* End StressedHead2Foot() *

//**************************
//*     ScreenCapture      *
//**************************
//******************************************************************************
//* For debugging only: Capture screenshot of the target dialog.               *
//* For Chart objects which contain scrollable data, screenshots may be        *
//* captured during the call to the ShiftData() method. However, for Cartesian *
//* charts and for other chart types where all of the data are currently       *
//* displayed, this method may be called after instantiation (and refresh),    *
//* OR after the ShiftData() method returns.                                   *
//*                                                                            *
//* Input  : chartPtr : pointer the Chart object                               *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void ChartTest::ScreenCapture ( Chart *chartPtr )
{
   AudibleAlert aa( 2,     // pings in sequence
                    2,     // delay between pings in 20ths of a second
                    3,     // repeat count
                    4 ) ;  // delay between sequences in 10ths of a second

   if ( chartPtr != NULL ) // be safe
   {
      //* Plain-text capture *
      chartPtr->CaptureDialog ( "./capturedlg.txt" ) ;

      //* HTML capture *
      chartPtr->CaptureDialog ( "./capturedlg.html", true, false, 
                                "infodoc-styles.css", 4, false, nc.blR ) ;

      //* Make a joyful noise.*
      this->dp->UserAlert ( &aa ) ;
   }
}  //* End ScreenCapture() *


//** - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  **
//** - - - - - - - - NON-MEMBER METHODS FOR THIS MODULE  - - - - - - - - - -  **
//** - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  **

//*************************
//*    ctControlUpdate    *
//*************************
//******************************************************************************
//* NON-MEMBER METHOD:                                                         *
//* This is a callback method for manually updating the dialog controls.       *
//*                                                                            *
//*  For this test, the following are updated by the callback method:          *
//*   a) Border Style Spinner                                                  *
//*   b) Grid Style Spinner                                                    *
//*   c) Horizontal-line Offset Spinner                                        *
//*   d) Bar Width Spinner                                                     *
//*   e) Selection bullets for items in the "Special Test" Scrollext control*
//*   f) Generate default Chart object (debugging only)                        *
//*                                                                            *
//* Input  : currIndex: index of control that currently has focus              *
//*          wkey     : user's key input data                                  *
//*          firstTime: the EstablishCallback() method calls this method once  *
//*                     with firstTime==true, to perform any required          *
//*                     initialization. Subsequently, the NcDialog class       *
//*                     always calls with firstTime==false.                    *
//* Returns: OK                                                                *
//******************************************************************************
//* Notes: Need to convert this callback method to a member method.            *
//******************************************************************************

static short ctControlUpdate ( const short currIndex, 
                               const wkeyCode wkey, bool firstTime )
{
   const char *clearDesc = "           " ;
   const char *bdrStyles[10]
   {
      "No Border",
      "Single-line",
      "Dual-line",
      "Single Bold",
      "Dash2",
      "Dash2 Bold",
      "Dash3",
      "Dash3 Bold",
      "Dash4",
      "Dash4 Bold",
   } ;
   const char *grdStyles[3]
   {
      "---dummy---",
      "Single-line",
      "Double-line",
   } ;
   const char *barWidth[9]
   {
      "BarTipsOnly",
      "Min. Width",
      " ",
      " ",
      "Half Width",
      " ",
      " ",
      " ",
      "Full Width",
   } ;
   const char *noHLine = "No Line" ;
   static winPos bdrdescPos ;
   static winPos grddescPos ;
   static winPos hldescPos ;
   static winPos bwdescPos ;
   gString gs ;
   int    descIndx ;

   if ( firstTime  )
   {
      //** Border-style Description **
      bdrdescPos.ypos = cticPtr[ctBorderSP].ulY ;
      bdrdescPos.xpos = cticPtr[ctBorderSP].ulX + cticPtr[ctBorderSP].cols + 1 ;
      ctcuPtr->GetSpinnerValue ( ctBorderSP, descIndx ) ;
      ctcuPtr->WriteString ( bdrdescPos, bdrStyles[descIndx], ctEmpAttr ) ;

      //** Grid-style Description **
      grddescPos.ypos = cticPtr[ctGridSP].ulY ;
      grddescPos.xpos = cticPtr[ctGridSP].ulX + cticPtr[ctGridSP].cols + 1 ;
      ctcuPtr->GetSpinnerValue ( ctGridSP, descIndx ) ;
      ctcuPtr->WriteString ( grddescPos, grdStyles[descIndx], ctEmpAttr ) ;

      //** Horizontal-line-position Description
      hldescPos.ypos = cticPtr[ctHlineSP].ulY ;
      hldescPos.xpos = cticPtr[ctHlineSP].ulX + cticPtr[ctHlineSP].cols + 1 ;
      ctcuPtr->GetSpinnerValue ( ctHlineSP, descIndx ) ;
      if ( descIndx == minHLINE )
         ctcuPtr->WriteString ( hldescPos, noHLine, ctEmpAttr ) ;
      else
         ctcuPtr->WriteString ( hldescPos, clearDesc, ctDlgAttr ) ;

      //* Bar-width Descripiton *
      bwdescPos.ypos = cticPtr[ctBdivSP].ulY ;
      bwdescPos.xpos = cticPtr[ctBdivSP].ulX + cticPtr[ctBdivSP].cols + 1 ;
      ctcuPtr->GetSpinnerValue ( ctBdivSP, descIndx ) ;
      ctcuPtr->WriteString ( bwdescPos, barWidth[descIndx], ctEmpAttr ) ;

      //** Initialize Special-test Descriptions **
      for ( short i = ZERO ; i < stITEMS ; ++i )
      {
         gs = specTextSrc[i] ;
         gs.shiftChars( -1 ) ;
         gs.insert( specFlags[i] ? wDiamond : nckSPACE ) ;
         gs.copy( specTextSrc[i], specSRC ) ;
      }
      ctcuPtr->RefreshScrollextText ( ctSpecSE, false ) ;

      ctcuPtr->RefreshWin () ;
   }  // (firstTime)

   if ( currIndex == ctBorderSP )
   {
      ctcuPtr->WriteString ( bdrdescPos, clearDesc, ctDlgAttr ) ;
      ctcuPtr->GetSpinnerValue ( ctBorderSP, descIndx ) ;
      ctcuPtr->WriteString ( bdrdescPos, bdrStyles[descIndx], ctEmpAttr, true ) ;
   }
   else if ( currIndex == ctGridSP )
   {
      ctcuPtr->WriteString ( grddescPos, clearDesc, ctDlgAttr ) ;
      ctcuPtr->GetSpinnerValue ( ctGridSP, descIndx ) ;
      ctcuPtr->WriteString ( grddescPos, grdStyles[descIndx], ctEmpAttr, true ) ;
   }
   else if ( currIndex == ctHlineSP )
   {
      ctcuPtr->GetSpinnerValue ( ctHlineSP, descIndx ) ;
      if ( descIndx == minHLINE )
         ctcuPtr->WriteString ( hldescPos, noHLine, ctEmpAttr ) ;
      else
         ctcuPtr->WriteString ( hldescPos, clearDesc, ctDlgAttr ) ;
      ctcuPtr->RefreshWin () ;
   }
   else if ( currIndex == ctBdivSP )
   {
      ctcuPtr->WriteString ( bwdescPos, clearDesc, ctDlgAttr ) ;
      ctcuPtr->GetSpinnerValue ( ctBdivSP, descIndx ) ;
      ctcuPtr->WriteString ( bwdescPos, barWidth[descIndx], ctEmpAttr, true ) ;
   }
   else if ( currIndex == ctSpecSE )
   {
      if ( (wkey.type == wktPRINT && wkey.key == nckSPACE) ||
           (wkey.type == wktFUNKEY && (wkey.key == nckENTER || wkey.key == nckpENTER)) )
      {
         descIndx = ctcuPtr->GetScrollextSelect ( ctSpecSE ) ;
         wchar_t wch ;
         if ( specFlags[descIndx] )    // if flag is currently set, reset it
         { specFlags[descIndx] = false ; wch = nckSPACE ; }
         else                          // else, set the flag
         { specFlags[descIndx] = true ; wch = wDiamond ; }
         gs = specTextSrc[descIndx] ;
         gs.shiftChars( -1 ) ;
         gs.insert( wch ) ;
         gs.copy( specTextSrc[descIndx], specSRC ) ;

         //* Important Note: Three items: stStressHead, stStressFoot *
         //* and stStressMargin form an XOR group.                   *
         //*    Head  Foot  Margin                                   *
         //*     -     -     -         all tests disabled            *
         //*     x     -     -         Header test enabled           *
         //*     -     x     -         Footer test enabled           *
         //*     -     -     x         Margin test enabled           *
         //* If one of these has just been set, reset the other two. *
         if ( specFlags[descIndx] && ((descIndx == stStressHead) || 
              (descIndx == stStressFoot) || (descIndx == stStressMargin)) )
         {
            wch = nckSPACE ;
            if ( descIndx != stStressHead )
            {
               gs = specTextSrc[stStressHead] ;
               gs.shiftChars( -1 ) ;
               gs.insert( wch ) ;
               gs.copy( specTextSrc[stStressHead], specSRC ) ;
               specFlags[stStressHead] = false ;
            }
            if ( descIndx != stStressFoot )
            {
               gs = specTextSrc[stStressFoot] ;
               gs.shiftChars( -1 ) ;
               gs.insert( wch ) ;
               gs.copy( specTextSrc[stStressFoot], specSRC ) ;
               specFlags[stStressFoot] = false ;
            }
            if ( descIndx != stStressMargin )
            {
               gs = specTextSrc[stStressMargin] ;
               gs.shiftChars( -1 ) ;
               gs.insert( wch ) ;
               gs.copy( specTextSrc[stStressMargin], specSRC ) ;
               specFlags[stStressMargin] = false ;
            }
         }
         //* Refresh the control's display data *
         ctcuPtr->RefreshScrollextText ( ctSpecSE, true ) ;
      }
   }

   //* Generate a default Chart object and capture screenshots *
   if ( (wkey.type == wktEXTEND) && (wkey.key == nckAS_P) )
   {
      ctcuPtr->SetDialogObscured () ;
      ChartDef dfltDef ;
      //if ( ct != ctLowerLeft )
      //   dfltDef.chType = ct ;
      Chart *cp = new Chart( dfltDef, true ) ;
      ctPtr->ScreenCapture ( cp ) ;
      nckPause();
      delete cp ;
      ctcuPtr->RefreshWin () ;
   }

   return OK ;

}  //* End pbControlUpdate() *

//*************************
//*     TargetExists      *
//*************************
//******************************************************************************
//* NON-MEMBER METHOD:                                                         *
//* Determine whether specified file exists and return its filetype.           *
//*                                                                            *
//* Input  : trgPath : filespec of target file                                 *
//*          fType   : (by reference)                                          *
//*                  : if target exists, initialized to target fileType        *
//*                    else fmUNKNOWN_TYPE                                     *
//*                                                                            *
//* Returns: 'true' if target exists, else 'false'                             *
//******************************************************************************
//* Programmer's Note: This method is grossly inefficient, but this            *
//* application requires simplicity of design over speed.                      *
//******************************************************************************

static bool TargetExists ( const gString& trgPath, fmFType& fType )
{
   tnFName fs ;               // file stats
   bool exists = false ;      // return value

   if ( (GetFileStats ( trgPath, fs )) )
   {
      fType = fs.fType ;
      exists = true ;
   }
   return exists ;

}  //* End TargetExists() *

//*************************
//*     GetFileStats      *
//*************************
//******************************************************************************
//* NON-MEMBER METHOD:                                                         *
//* Perform a 'stat' ('lstat') on the file specified by the full path string.  *
//*                                                                            *
//* Input  : trgPath: full path/filename specification                         *
//*          fStats : (by reference) receives the stat data for the file       *
//*                                                                            *
//* Returns: 'true'  if file exists, else 'false'                              *
//******************************************************************************

static bool GetFileStats ( const gString& trgPath, tnFName& fStats )
{
   bool exists = false ;            // return value

   //* Provides an orderly crash in case of failure *
   fStats.modTime.reset() ;
   fStats.fBytes = ZERO ;
   fStats.fType = fmUNKNOWN_TYPE ;
   fStats.readAcc = fStats.writeAcc = false ;

   if ( (lstat64 ( trgPath.ustr(), &fStats.rawStats )) == OK )
   {
      //* Extract filename from caller's string *
      short indx = trgPath.findlast( L'/' ) ;
      if ( indx < ZERO )   indx = ZERO ;
      else                 ++indx ;
      gString fName( &trgPath.gstr()[indx] ) ;
      fName.copy( fStats.fName, MAX_FNAME ) ;

      fStats.fBytes = fStats.rawStats.st_size ;    // number of bytes in file

      //* Decode the filetype *
      UINT mode = fStats.rawStats.st_mode ;
      if ( (S_ISREG(mode)) != false )        fStats.fType = fmREG_TYPE ;
      else if ( (S_ISDIR(mode)) != false )   fStats.fType = fmDIR_TYPE ;
      else if ( (S_ISLNK(mode)) != false )   fStats.fType = fmLINK_TYPE ;
      else if ( (S_ISCHR(mode)) != false )   fStats.fType = fmCHDEV_TYPE ;
      else if ( (S_ISBLK(mode)) != false )   fStats.fType = fmBKDEV_TYPE ;
      else if ( (S_ISFIFO(mode)) != false )  fStats.fType = fmFIFO_TYPE ;
      else if ( (S_ISSOCK(mode)) != false )  fStats.fType = fmSOCK_TYPE ;

      //* Decode the mod time *
      fStats.modTime.epoch = (int64_t)fStats.rawStats.st_mtime ;
      fStats.modTime.sysepoch = (time_t)fStats.rawStats.st_mtime ; // (possible narrowing)
      Tm bdt ;       // receives broken-down time
      if ( (localtime_r ( &fStats.modTime.sysepoch, &bdt )) != NULL )
      {
         //* Translate to localTime format (timezone data not decoded) *
         fStats.modTime.date    = bdt.tm_mday ;         // today's date
         fStats.modTime.month   = bdt.tm_mon + 1 ;      // month
         fStats.modTime.year    = bdt.tm_year + 1900 ;  // year
         fStats.modTime.hours   = bdt.tm_hour ;         // hour
         fStats.modTime.minutes = bdt.tm_min ;          // minutes
         fStats.modTime.seconds = bdt.tm_sec ;          // seconds
         fStats.modTime.day     = bdt.tm_wday ;         // day-of-week (0 == Sunday)
         fStats.modTime.julian  = bdt.tm_yday ;         // Julian date (0 == Jan.01)
      }

      //* Initialize the read-access and write-access flags *
      fStats.readAcc = bool((access ( trgPath.ustr(), R_OK )) == ZERO) ;
      if ( fStats.readAcc != false && fStats.fType == fmDIR_TYPE )
         fStats.readAcc = bool((access ( trgPath.ustr(), X_OK )) == ZERO) ;
      fStats.writeAcc = bool((access ( trgPath.ustr(), W_OK )) == ZERO) ;

      exists = true ;
   }
   return exists ;

}  //* End GetFileStats() *


//********************************************************
//* Initialization of dialog control objects.            *
//* Color attributes initialized by CharTest constructor.*
//********************************************************
InitCtrl ChartTestIC[ctCONTROLS] =
{
   {  //* 'LAUNCH' pushbutton  - - - - - - - - - - - - - - - - - -  ctLaunchPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      2,                            // ulY:       upper left corner in Y
      4,                            // ulX:       upper left corner in X
      1,                            // lines:     control lines
      10,                           // cols:      control columns
      "  ^LAUNCH  ",                // dispText:  
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ChartTestIC[ctClosePB]       // nextCtrl:  link in next structure
   },
   {  //* 'CLOSE' pushbutton  - - - - - - - - - - - - - - - - - - -  ctClosePB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ChartTestIC[ctLaunchPB].ulY,  // ulY:       upper left corner in Y
      19,                           // ulX:       upper left corner in X
      1,                            // lines:     control lines
      9,                            // cols:      control columns
      "  CL^OSE  ",                 // dispText:  
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ChartTestIC[ctHelpPB]        // nextCtrl:  link in next structure
   },
   {  //* 'HELP' pushbutton  - - - - - - - - - - - - - - - - - - - -  ctHelpPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ChartTestIC[ctLaunchPB].ulY,  // ulY:       upper left corner in Y
      34,                           // ulX:       upper left corner in X
      1,                            // lines:     control lines
      8,                            // cols:      control columns
      "  HEL^P  ",                  // dispText:  
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ChartTestIC[ctStatTB]        // nextCtrl:  link in next structure
   },
   {  //* 'Status Msg' textbox  - - - - - - - - - - - - - - - - -  ctStatTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ChartTestIC[ctLaunchPB].ulY + 1), // ulY: upper left corner in Y
      ChartTestIC[ctLaunchPB].ulX,  // ulX:       upper left corner in X
      2,                            // lines:     (n/a)
      38,                           // cols:      control columns
      NULL,                         // dispText:  (value of t3indexO3)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      NULL,                         // label:     
      ZERO,                         // labY:      
      -12,                          // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    cannot gain focus
      &ChartTestIC[ctTypeDD]        // nextCtrl:  link in next structure
   },
   {  //* "Chart Type' Dropdown   - - - - - - - - - - - - - -      ctTypeDD *
      dctDROPDOWN,                  // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      short(ChartTestIC[ctStatTB].ulY + 4), // ulY: upper left corner in Y
      ChartTestIC[ctStatTB].ulX,    // ulX: upper left corner in X
      typeDDrows,                   // lines:     control lines
      typeDDcols,                   // cols:      control columns
      (char*)&typeDDdata,           // dispText:  n/a
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Chart ^Type",                // label:
      -1,                           // labY:      offset from control's ulY
      ZERO,                         // labX       offset from control's ulX
      ddBoxDOWN,                    // exType:    expands downward
      typeDDitems,                  // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    index of initial highlighted element
      monoColor,                    // scrColor:  single-color data display
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ChartTestIC[ctRowSP]         // nextCtrl:  link in next structure
   },
   {  //* 'ROWS' spinner  - - - - - - - - - - - - - - - - - - - -   ctRowSP *
      dctSPINNER,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ChartTestIC[ctTypeDD].ulY,    // ulY: upper left corner in Y
      short(ChartTestIC[ctTypeDD].ulX + 19), // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      spinWIDTH,                    // cols:      control columns
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Chart Area ^Rows",           // label:     
      -1,                           // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      &rspinData,                   // spinData:  spinner init
      true,                         // active:    allow control to gain focus
      &ChartTestIC[ctColSP]         // nextCtrl:  link in next structure
   },
   {  //* 'COLS' spinner  - - - - - - - - - - - - - - - - - - - -   ctColSP *
      dctSPINNER,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ChartTestIC[ctRowSP].ulY + 2), // ulY: upper left corner in Y
      ChartTestIC[ctRowSP].ulX,     // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      spinWIDTH,                    // cols:      control columns
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Chart Area ^Columns",        // label:     
      -1,                           // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      &cspinData,                   // spinData:  spinner init
      true,                         // active:    allow control to gain focus
      &ChartTestIC[ctFootSP],       // nextCtrl:  link in next structure
   },
   {  //* 'FOOTER ROWS' Spinner  - - - - - - - - - - - - - - - - -    ctFootSP *
      dctSPINNER,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ChartTestIC[ctColSP].ulY + 2), // ulY: upper left corner in Y
      ChartTestIC[ctColSP].ulX,     // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      spinWIDTH,                    // cols:      control columns
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "^Footer Area Rows",          // label:
      -1,                           // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      &fspinData,                   // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ChartTestIC[ctMarginSP]      // nextCtrl:  link in next structure
   },
   {  //* 'MARGIN COLUMNS' Spinner  - - - - - - - - - - - - - - -   ctMarginSP *
      dctSPINNER,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ChartTestIC[ctFootSP].ulY + 2), // ulY: upper left corner in Y
      ChartTestIC[ctFootSP].ulX,    // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      spinWIDTH,                    // cols:      control columns
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "M^argin Area Cols",          // label:
      -1,                           // labY:      label offset Y
      ZERO,                         // labX       label offset X
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      &mspinData,                   // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ChartTestIC[ctBorderSP]      // nextCtrl:  link in next structure
   },
   {  //* 'BORDER STYLE' spinner  - - - - - - - - - - - - - - - -   ctBorderSP *
      dctSPINNER,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ChartTestIC[ctMarginSP].ulY + 2), // ulY: upper left corner in Y
      ChartTestIC[ctMarginSP].ulX,  // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      spinWIDTH,                    // cols:      control columns
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "^Border Style",              // label:     
      -1,                           // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      &bspinData,                   // spinData:  spinner init
      true,                         // active:    allow control to gain focus
      &ChartTestIC[ctGridSP]        // nextCtrl:  link in next structure
   },
   {  //* 'GRID STYLE' spinner  - - - - - - - - - - - - - - - - - -   ctGridSP *
      dctSPINNER,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ChartTestIC[ctBorderSP].ulY + 2), // ulY: upper left corner in Y
      ChartTestIC[ctBorderSP].ulX,  // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      spinWIDTH,                    // cols:      control columns
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "^Grid Style",                // label:     
      -1,                           // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      &gspinData,                   // spinData:  spinner init
      true,                         // active:    allow control to gain focus
      &ChartTestIC[ctHlineSP]       // nextCtrl:  link in next structure
   },
   {  //* 'HORIZONTAL LINE' spinner - - - - - - - - - - - - - - - -  ctHlineSP *
      dctSPINNER,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ChartTestIC[ctGridSP].ulY + 2), // ulY: upper left corner in Y
      ChartTestIC[ctGridSP].ulX,    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      spinWIDTH,                    // cols:      control columns
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Hori^z Line Offset",         // label:     
      -1,                           // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      &hspinData,                   // spinData:  spinner init
      true,                         // active:    allow control to gain focus
      &ChartTestIC[ctBdivSP]        // nextCtrl:  link in next structure
   },
   {  //* 'BAR WIDTH' spinner - - - - - - - - - - - - - - - - - - -   ctBdivSP *
      dctSPINNER,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ChartTestIC[ctHlineSP].ulY + 2), // ulY: upper left corner in Y
      ChartTestIC[ctHlineSP].ulX,   // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      spinWIDTH,                    // cols:      control columns
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Bar ^Width (divs)",          // label:     
      -1,                           // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      &wspinData,                   // spinData:  spinner init
      true,                         // active:    allow control to gain focus
      &ChartTestIC[ctIndyRB]        // nextCtrl:  link in next structure
   },
   {  //* 'INDEPENDENT DIALOG' radiobutton  - - - - - - - - - - - -   ctIndyRB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: square, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ChartTestIC[ctTypeDD].ulY + 4), // ulY: upper left corner in Y
      short(ChartTestIC[ctTypeDD].ulX - 2), // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "I^ndependent Dlg",           // label:
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      ZERO,                         // scrItems:  (n/a)
      ZERO,                         // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ChartTestIC[ctHtextRB]       // nextCtrl:  link in next structure
   },
   {  //* 'SET HEADER TEXT' radiobutton - - - - - - - - - - - - - -  ctHtextRB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: square, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ChartTestIC[ctIndyRB].ulY + 1), // ulY: upper left corner in Y
      ChartTestIC[ctIndyRB].ulX,    // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "^Header Text",               // label:
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      ZERO,                         // scrItems:  (n/a)
      ZERO,                         // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ChartTestIC[ctFtextRB]       // nextCtrl:  link in next structure
   },
   {  //* 'SET FOOTER TEXT' radiobutton   - - - - - - - - - - - - -  ctFtextRB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: square, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ChartTestIC[ctHtextRB].ulY + 1), // ulY: upper left corner in Y
      ChartTestIC[ctHtextRB].ulX,   // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Footer Text",                // label:
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      ZERO,                         // scrItems:  (n/a)
      ZERO,                         // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ChartTestIC[ctMtextRB]       // nextCtrl:  link in next structure
   },
   {  //* 'SET MARGIN TEXT' radiobutton   - - - - - - - - - - - - -  ctMtextRB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: square, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ChartTestIC[ctFtextRB].ulY + 1), // ulY: upper left corner in Y
      ChartTestIC[ctFtextRB].ulX,   // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Margin Text",                // label:
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      ZERO,                         // scrItems:  (n/a)
      ZERO,                         // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ChartTestIC[ctTtextRB]       // nextCtrl:  link in next structure
   },
   {  //* 'SET TITLE TEXT' radiobutton - - - - - - - - - - - - - -   ctTtextRB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: square, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ChartTestIC[ctMtextRB].ulY + 1), // ulY: upper left corner in Y
      ChartTestIC[ctMtextRB].ulX,   // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Title Text",                 // label:
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      ZERO,                         // scrItems:  (n/a)
      ZERO,                         // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ChartTestIC[ctHbarsRB]       // nextCtrl:  link in next structure
   },
   {  //* 'HORIZ BARS' radiobutton   - - - - - - - - - - - - - - -   ctHbarsRB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: square, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ChartTestIC[ctTtextRB].ulY + 1), // ulY: upper left corner in Y
      ChartTestIC[ctTtextRB].ulX,   // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Horizontal Bars",            // label:
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      ZERO,                         // scrItems:  (n/a)
      ZERO,                         // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ChartTestIC[ctMultiRB]       // nextCtrl:  link in next structure
   },
   {  //* 'MULTI-COLOR BARS' radiobutton   - - - - - - - - - - - -   ctMultiRB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: square, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ChartTestIC[ctHbarsRB].ulY + 1), // ulY: upper left corner in Y
      ChartTestIC[ctHbarsRB].ulX,   // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Multicolor Bars",            // label:
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      ZERO,                         // scrItems:  (n/a)
      ZERO,                         // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ChartTestIC[ctSpecSE]        // nextCtrl:  link in next structure
   },
   {  //* 'SPECIAL TESTS' Scrollext - - - - - - - - - - - - - - - -   ctSpecSE *
      dctSCROLLEXT,                 // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      short(ChartTestIC[ctBdivSP].ulY + 1), // ulY: upper left corner in Y
      ChartTestIC[ctMultiRB].ulX,   // ulX: upper left corner in X
      specHEIGHT,                   // lines:     control lines
      specWIDTH,                    // cols:      control columns
      NULL,                         // dispText:  n/a
      nc.bw,                        // nColor:    non-focus border color
      nc.bw,                        // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "^Special Tests",             // label:     
      -1,                           // labY:      offset from control's ulY
      1,                            // labX       offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      ZERO,                         // scrItems:  (n/a)
      ZERO,                         // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ChartTestIC[ctDataDD]        // nextCtrl:  link in next structure
   },
   { //* 'DATA SET' Dropdown   - - - - - - - - - - - - - - - - -   ctDataDD *
      dctDROPDOWN,                  // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      short(ChartTestIC[ctSpecSE].ulY + ChartTestIC[ctSpecSE].lines), // ulY:
      ChartTestIC[ctSpecSE].ulX,    // ulX:       upper left corner in X
      datasetROWS,                  // lines:     control lines
      datasetCOLS,                  // cols:      control columns
      (const char*)datasetData,     // dispText:  items displayed
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Select ^Data Set",           // label:     
      3,                            // labY:      offset from control's ulY
      1,                            // labX       offset from control's ulX
      ddBoxUP,                      // exType:    expands downward
      datasetITEMS,                 // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    index of initial highlighted element
      monoColor,                    // scrColor:  single-color data display
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
} ;

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

