//********************************************************************************
//* File       : ClInfoTest.cpp                                                  *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2021-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in NcDialog.hpp            *
//* Date       : 15-Jul-2025                                                     *
//* Version    : (see below)                                                     *
//*                                                                              *
//* Description: Class definition for testing of the wcbUserAlert() method and   *
//*              other methods related to the NcDialog access to the WaylandCB   *
//*              class.                                                          *
//*              This test is called as a menu item within the Dialog4           *
//*              application, Test10.                                            *
//*                                                                              *
//* Development Tools: See NcDialog.cpp.                                         *
//********************************************************************************
//* Version History (most recent first):                                         *
//*                                                                              *
//* v: 0.00.01 11-Jul-2025                                                       *
//*   - Base data was created and verified in the Dialog4 sandbox.               *
//*     The enclosing dialog is based on a variation of the PinwheelTest dialog. *
//*     The user-interface loop is based on the Sandbox loop.                    *
//*                                                                              *
//*                                                                              *
//********************************************************************************

//****************
//* Header Files *
//****************
#include "ClInfoTest.hpp"

//**********************
//** Local Prototypes **
//**********************


//****************
//** Local Data **
//****************
static const char* wTrunc =      // test data for truncation of width
"  Width Truncation Test (78 columns max)  \n"
"123456789-123456789-123456789-123456789-123456789-123456789-123456789-123456789-\n"
"I will happily pay you Tuesday for a hamburger today. I will happily pay you Tuesday for a hamburger today.\n"
"abcdefghijklmnopqrstuvwzyz-abcdefghijklmnopqrstuvwzyz-abcdefghijklmnopqrstuvwzyz-\n" ;
static const char* hTrunc =      // test data for truncation of height
" Height Truncation Test (22 lines max) \n"
"Data Line ------------------------- #1\nData Line ------------------------- #2\n"
"Data Line ------------------------- #3\nData Line ------------------------- #4\n"
"Data Line ------------------------- #5\nData Line ------------------------- #6\n"
"Data Line ------------------------- #7\nData Line ------------------------- #8\n"
"Data Line ------------------------- #9\nData Line ------------------------- #10\n"
"Data Line ------------------------- #11\nData Line ------------------------- #12\n"
"Data Line ------------------------- #13\nData Line ------------------------- #14\n"
"Data Line ------------------------- #15\nData Line ------------------------- #16\n"
"Data Line ------------------------- #17\nData Line ------------------------- #18\n"
"Data Line ------------------------- #19\nData Line ------------------------- #20\n"
"Data Line ------------------------- #21\nData Line ------------------------- #22\n" ;

enum cliControls : short
{
   cliOpenPB = ZERO,             // 'Open Dialog' Pushbutton
   cliExitPB,                    // 'Done' Pushbutton
   //* XOR Group - Message Type *
   cliStatRB,                    // Report connection status
   cliInfoRB,                    // Standard user-specified message
   cliWMsgRB,                    // Width-truncated message
   cliHMsgRB,                    // Height-truncated message
   //* XOR Group - Color Attribute *
   cliReddRB,                    // Red color attribute
   cliCyanRB,                    // Cyan color attribute
   cliInheRB,                    // Inherited color attribute

   cliConnRB,                    // Connect to or disconnect from clipboard interface
   cliKeysRB,                    // Use alternate keycodes for copy/cut/paste
   cliTextTB,
   cliCTRLS                      // number of dialog controls
} ;

//* Define a copy of the default set of Copy/Cut/Paste/Select keycodes. *
//* These are defined at the top of NcdControlTB.cpp.                   *
static wkeyCode sCopy( nckC_C, wktFUNKEY ),
                 sCut( nckC_X, wktFUNKEY ),
               sPaste( nckC_V, wktFUNKEY ),
              sSelAll( nckC_A, wktFUNKEY ),
             sSelLeft( nckSLEFT, wktFUNKEY ),
            sSelRight( nckSRIGHT, wktFUNKEY ) ;
reservedKeys stdKeys( sCopy, sCut, sPaste, sSelAll, sSelLeft, sSelRight, true ) ;
//* Define an alternate set of Copy/Cut/Paste/Select keycodes. *
//* (This is the recommended alternate set.)                   *
static wkeyCode aCopy( nckA_C, wktEXTEND ),
                 aCut( nckA_X, wktEXTEND ),
               aPaste( nckA_V, wktEXTEND ),
              aSelAll( nckA_A, wktEXTEND ),
             aSelLeft( nckSRIGHT, wktEXTEND ),
            aSelRight( nckSRIGHT, wktEXTEND ) ;
static reservedKeys altKeys( aCopy, aCut, aPaste, aSelAll, aSelLeft, aSelRight, true ) ;


//*************************
//*      ClInfoTest       *
//*************************
//********************************************************************************
//* Exercise the functionality of the wcbUserAlert() method.                     *
//*                                                                              *
//* Input  : wpOrig : (by reference) origin (upper left corner) for test dialog  *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

ClInfoTest::ClInfoTest ( const winPos& wpOrig )
{
   const short dialogROWS  = 24 ;   // display lines
   const short dialogCOLS  = 80 ;   // display columns
   const short COL2        = 41 ;   // column 2 offset

   gString gsMsg ;                  // message sent to sub-dialog
   this->dColor = nc.blR ;          // dialog background color
   this->bColor = nc.brbl ;         // dialog border color
   this->fColor = nc.reG ;         // controls' focus color
   attr_t idAttr ;                  // color attribute for info dialog
   short ulY = wpOrig.ypos,         // upper left corner in Y
         ulX = wpOrig.xpos,         // upper left corner in X
         gmIndex,                   // index of message group selection
         gcIndex ;                  // index of color attribute selection
   bool  conn = false,              // status of CB connection
         connState,                 // temp state of 'cliConnRB'
         keys = false,              // status of reserved keycode set
         keysState ;                // temp state of 'cliKeysRB'

InitCtrl ic[cliCTRLS] =             // array of dialog control initialization objects
{
   {  //* 'OPEN' pushbutton - - - - - - - - - - - - - - - - - - - -   cliOpenPB  *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 2),        // ulY:       upper left corner in Y
      short(dialogCOLS / 2 - 10),   // ulX:       upper left corner in X
      1,                            // lines:     control lines
      8,                            // cols:      control columns
      "  ^OPEN  ",                  // dispText:  
      nc.gyR,                       // nColor:    non-focus color
      this->fColor,                 // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[cliExitPB]                // nextCtrl:  link in next structure
   },

   {  //* 'EXIT' pushbutton - - - - - - - - - - - - - - - - - - - -   cliExitPB  *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[cliOpenPB].ulY,            // ulY:       upper left corner in Y
      short(ic[cliOpenPB].ulX + ic[cliOpenPB].cols + 2), // ulX: upper left corner in X
      1,                            // lines:     control lines
      8,                            // cols:      control columns
      "  EXI^T  ",                  // dispText:  
      nc.gyR,                       // nColor:    non-focus color
      this->fColor,                 // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[cliStatRB]                // nextCtrl:  link in next structure
   },

   {  //* Message Type XOR Group  - - - - - - - - - - - - - - - - - -  cliStatRB *
      dctRADIOBUTTON,               // type:      
      rbtS5s,                       // rbSubtype: angle, 3 chars wide
      true,                         // rbSelect:  default selection
      4,                            // ulY:       upper left corner in Y
      3,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->dColor,                 // nColor:    non-focus color
      this->fColor,                 // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "^Status of Connection",      // label:
      0,                            // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      ZERO,                         // scrItems:  (n/a)
      ZERO,                         // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[cliInfoRB]                // nextCtrl:  link in next structure
   },

   {  //* Message Type XOR Group  - - - - - - - - - - - - - - - - - -  cliInfoRB *
      dctRADIOBUTTON,               // type:      
      rbtS5s,                       // rbSubtype: angle, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[cliStatRB].ulY + 1), // ulY:       upper left corner in Y
      3,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->dColor,                 // nColor:    non-focus color
      this->fColor,                 // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "^Detailed Info Report",      // label:
      0,                            // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      ZERO,                         // scrItems:  (n/a)
      ZERO,                         // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[cliWMsgRB]                // nextCtrl:  link in next structure
   },

   {  //* Message Type XOR Group  - - - - - - - - - - - - - - - - - -  cliWMsgRB *
      dctRADIOBUTTON,               // type:      
      rbtS5s,                       // rbSubtype: angle, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[cliInfoRB].ulY + 1), // ulY:       upper left corner in Y
      3,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->dColor,                 // nColor:    non-focus color
      this->fColor,                 // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "^Width Truncation Test",     // label:
      0,                            // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      ZERO,                         // scrItems:  (n/a)
      ZERO,                         // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[cliHMsgRB]                // nextCtrl:  link in next structure
   },

   {  //* Message Type XOR Group  - - - - - - - - - - - - - - - - - -  cliHMsgRB *
      dctRADIOBUTTON,               // type:      
      rbtS5s,                       // rbSubtype: angle, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[cliWMsgRB].ulY + 1), // ulY:       upper left corner in Y
      3,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->dColor,                 // nColor:    non-focus color
      this->fColor,                 // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "^Height Truncation Test",    // label:
      0,                            // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      ZERO,                         // scrItems:  (n/a)
      ZERO,                         // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[cliReddRB]                // nextCtrl:  link in next structure
   },

   {  //* Message Type XOR Group  - - - - - - - - - - - - - - - - - -  cliReddRB *
      dctRADIOBUTTON,               // type:      
      rbtS5s,                       // rbSubtype: angle, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[cliHMsgRB].ulY + 3), // ulY:       upper left corner in Y
      3,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->dColor,                 // nColor:    non-focus color
      this->fColor,                 // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "^Red Background Attribute",  // label:
      0,                            // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      ZERO,                         // scrItems:  (n/a)
      ZERO,                         // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[cliCyanRB]                // nextCtrl:  link in next structure
   },

   {  //* Message Type XOR Group  - - - - - - - - - - - - - - - - - -  cliCyanRB *
      dctRADIOBUTTON,               // type:      
      rbtS5s,                       // rbSubtype: angle, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[cliReddRB].ulY + 1), // ulY:       upper left corner in Y
      3,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->dColor,                 // nColor:    non-focus color
      this->fColor,                 // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "C^yan Background Attribute", // label:
      0,                            // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      ZERO,                         // scrItems:  (n/a)
      ZERO,                         // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[cliInheRB]                // nextCtrl:  link in next structure
   },

   {  //* Message Type XOR Group  - - - - - - - - - - - - - - - - - -  cliInheRB *
      dctRADIOBUTTON,               // type:      
      rbtS5s,                       // rbSubtype: angle, 3 chars wide
      true,                         // rbSelect:  default selection
      short(ic[cliCyanRB].ulY + 1), // ulY:       upper left corner in Y
      3,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->dColor,                 // nColor:    non-focus color
      this->fColor,                 // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Inherit Color Attri^butes",   // label:
      0,                            // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      ZERO,                         // scrItems:  (n/a)
      ZERO,                         // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[cliConnRB]                // nextCtrl:  link in next structure
   },

   {  //* 'CONNECT' Radiobutton - - - - - - - - - - - - - - - - - - -  cliConnRB *
      dctRADIOBUTTON,               // type:      
      rbtS5s,                       // rbSubtype: angle, 3 chars wide
      conn,                         // rbSelect:  default selection
      short(ic[cliInheRB].ulY + 3), // ulY:       upper left corner in Y
      3,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->dColor,                 // nColor:    non-focus color
      this->fColor,                 // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "^Enable WaylandCB Interface",// label:
      0,                            // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      ZERO,                         // scrItems:  (n/a)
      ZERO,                         // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[cliKeysRB]                // nextCtrl:  link in next structure
   },

   {  //* 'KEYCODES' Radiobutton  - - - - - - - - - - - - - - - - - -  cliKeysRB *
      dctRADIOBUTTON,               // type:      
      rbtS5s,                       // rbSubtype: angle, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[cliConnRB].ulY + 1), // ulY:       upper left corner in Y
      3,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->dColor,                 // nColor:    non-focus color
      this->fColor,                 // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Alternate U/I ^Keycodes",    // label:
      0,                            // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      ZERO,                         // scrItems:  (n/a)
      ZERO,                         // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[cliTextTB]                // nextCtrl:  link in next structure
   },

   {  //* 'TEXT' Textbox  - - - - - - - - - - - - - - - - - - - - - -  cliTextTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[cliStatRB].ulY),     // ulY:       upper left corner in Y
      COL2,                         // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      36,                           // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      fColor,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:
      0,                            // labY:      
      0,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      ZERO,                         // scrItems:  (n/a)
      ZERO,                         // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
} ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       "  NcDialog Interface to WaylandCB Class  ", // dialog title
                       ncltDUAL,       // border line-style
                       this->bColor,   // border color attribute
                       this->dColor,   // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   this->dp = new NcDialog ( dInit ) ;
   
   //* Enable clipboard interface and update *
   //* state of connection radiobutton.      *
   this->dp->SetRadiobuttonState ( cliConnRB, (conn = this->dp->wcbEnable ()) ) ;

   //* Initialize default set of "reserved keycodes" *
   //* for copy/cut/paste/select and enable auditory *
   //* alert for invalid keycodes.                   *
   this->dp->SetTextboxReservedKeys() ;
   this->dp->TextboxAlert ( cliTextTB, true ) ;

   //* Open the dialog window *
   if ( (this->dp->OpenWindow()) == OK )
   {
      //* Write dialog static text *
      this->wp = { short(ic[cliStatRB].ulY - 1), ic[cliStatRB].ulX } ;
      this->dp->WriteParagraph ( wp,         
         "Message Type for Info Dialog   \n\n\n\n\n\n"
         "Info Dialog Color Attributes   \n\n\n\n\n"
         "Configure Clipboard Interface  \n\n\n", (this->bColor | ncuATTR) ) ;
      this->wp.xpos = COL2 ;
      this->wp = this->dp->WriteParagraph ( wp, 
         "Test Clipbord Functionality         \n\n", (this->bColor | ncuATTR) ) ;
      gsMsg.compose ( "Copy  keycode   : xxxxx\n"
                      "Cut   keycode   : xxxxx\n"
                      "Paste keycode   : xxxxx\n"
                      "Select rightward: xxxxx\n"
                      "Select leftward : xxxxx\n"
                      "Select all      : ") ;
      this->wp = this->dp->WriteParagraph ( wp, gsMsg, this->bColor ) ;
      this->wp.ypos -= 5 ;    // set position for keycode report
      this->setReservedKeycodes ( keys, false ) ;

      //* Create Radiobutton groups *
      short XorGroup[] = { cliStatRB, cliInfoRB, cliWMsgRB, cliHMsgRB, -1 } ;
      this->dp->GroupRadiobuttons ( XorGroup ) ;
      XorGroup[0] = cliReddRB ; XorGroup[1] = cliCyanRB ; 
      XorGroup[2] = cliInheRB ; XorGroup[3] = -1 ;
      this->dp->GroupRadiobuttons ( XorGroup ) ;

      this->dp->RefreshWin () ;     // make everything visible

      //* Interact with user *
      uiInfo Info ;                 // user interface data returned here
      short  icIndex = ZERO ;       // index of control with input focus
      bool   done = false ;         // loop control
      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )         // arrived via hotkey
               Info.HotData2Primary () ;  // unpack the data
            else
               icIndex = this->dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == cliExitPB )
                  done = true ;
               else if ( Info.ctrlIndex == cliOpenPB )
               {
                  gString *gsPtr = NULL ;
                  gmIndex = this->dp->GetRbGroupSelection ( cliStatRB ) ; // get message type
                  gcIndex = this->dp->GetRbGroupSelection ( cliReddRB ) ; // get message type
                  switch ( gmIndex )
                  {
                     case cliInfoRB:            // Report status and software versions
                        {                       // (user-provided message)
                        gString gsver, gsval ;
                        int cbbytes ;
                        conn = this->dp->wcbIsConnected () ; // get WCB status
                        this->dp->wcbVersion ( gsver ) ;     // get concatenated version strings
                        if ( conn )
                        {
                           gsver.insert( "WaylandCB version:    " ) ;
                           gsver.insert( "wl-clipboard version: ", (gsver.after( NEWLINE )) ) ;
                           cbbytes = this->dp->wcbBytes () ;
                           gsval.formatInt( cbbytes, 5, true );
                           gsver.append( "\nclipboard contents: %S bytes", gsval.gstr() ) ;
                        }
                        gsMsg.compose( "  Clipboard Status  \n"
                                       "System Clipboard Connected: %s\n"
                                       "%S\n",
                                       (conn ? "yes" : "no"), gsver.gstr() ) ;
                        gsPtr = &gsMsg ;
                        }
                        break ;
                     case cliWMsgRB: gsMsg = wTrunc ; gsPtr = &gsMsg ; break ; // width test
                     case cliHMsgRB: gsMsg = hTrunc ; gsPtr = &gsMsg ; break ; // height test
                     case cliStatRB:   break ;  // Report current connection status
                     default:          break ;  // (should never happed)
                  } ;
                  switch ( gcIndex )
                  {
                     case cliReddRB: idAttr = nc.reR ; break ;
                     case cliCyanRB: idAttr = nc.cyR ; break ;
                     case cliInheRB: idAttr = ZERO ;   break ;
                     default:                          break ; // (should never happed)
                  }
                  this->dp->wcbUserAlert ( wcbsACTIVE, gsPtr, idAttr ) ;
               }
            }
         }
         else if ( ic[icIndex].type == dctTEXTBOX )
         {
            if ( Info.viaHotkey )         // arrived via hotkey
               Info.viaHotkey = false ;   // ignore hotkey data
            icIndex = this->dp->EditTextbox ( Info ) ;
         }
         else if ( ic[icIndex].type == dctRADIOBUTTON )
         {
            if ( Info.viaHotkey )         // arrived via hotkey
               Info.HotData2Primary () ;  // unpack the data
            else
               icIndex = this->dp->EditRadiobutton ( Info ) ;
         }

         //* Move input focus to next/previous control.*
         if ( done == false && Info.viaHotkey == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = this->dp->PrevControl () ; 
            else
               icIndex = this->dp->NextControl () ;
         }

         //* If state of connection RB has changed, *
         //* enable/disable clipboard connection.   *
         this->dp->GetRadiobuttonState ( cliConnRB, connState ) ;
         if ( connState != conn )
         {
            if ( connState )
            {
               //* Get keycode configuration RB. *
               bool altKeycodes ;
               this->dp->GetRadiobuttonState( cliKeysRB, altKeycodes ) ;
               if ( altKeycodes ) { this->dp->wcbEnable ( altKeys ) ; }
               else               { this->dp->wcbEnable () ; }
            }
            else { this->dp->wcbDisable () ; }
            conn = this->dp->wcbIsConnected() ;
         }

         //* If reserved-keycode set has been toggled, *
         //* update the keycode group and display the  *
         //* current keycode-group definitions.        *
         this->dp->GetRadiobuttonState ( cliKeysRB, keysState ) ;
         if ( keysState != keys )
         {
            this->setReservedKeycodes ( (keys = keysState) ) ;
         }
      }  // while()
   }     // dialog opened

   this->dp->wcbDisable () ;                 // close system clipboard connection
   if ( this->dp != NULL )                   // close the window
      delete ( this->dp ) ;

}  //* End ClInfoTest() *

//**************************
//* reportReservedKeycodes *
//**************************
//********************************************************************************
//* Get a copy of the keycodes currently reserved for clipboard operations       *
//* within a Textbox control and report the keycodes.                            *
//*                                                                              *
//* Note: For simplicity, the test assumes the reserved keycodes SHOULD BE       *
//*       either the default set, or the locally-defined alternate set.          *
//* Note also that the test is rather kludgy, but hey, It's Only a Northern Song.*
//*                                                                              *
//* Input  : altKeycodes : expected keycode group                                *
//*                        reset : default keycode group                         *
//*                        set   : alternate keycode group                       *
//*          update      : (optional, default: set)                              *
//*                        set   : update reserved keycodes before reporting     *
//*                        reset : report only                                   *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void ClInfoTest::setReservedKeycodes ( bool altKeycodes, bool update )
{
   const short keyCnt = 6 ;
   const char* stdKeyNames[keyCnt] = 
   {
      "Ctrl+C",
      "Ctrl+X",
      "Ctrl+V",
      "Ctrl+A",
      "Shift+RArrow",
      "Shift+LArrow",
   } ;
   const char* altKeyNames[keyCnt] = 
   {
      "Alt+C ",
      "Alt+X ",
      "Alt+V ",
      "Alt+A ",
      "Shift+RArrow",
      "Shift+LArrow",
   } ;
   const char* keyError = "error!" ;
   const char** dataPtr = stdKeyNames ;   // pointer to display-data array
   reservedKeys rk ;                      // copy of current reserved-key configuration
   gString gs ;                           // text formatting
   bool match[keyCnt] = { true, true, true, true, true, true } ;

   //* Update the reserved-keycode list *
   if ( update )
   {
      if ( altKeycodes )                  // alternate keycode set
         this->dp->SetTextboxReservedKeys ( altKeys ) ;
      else                                // standard keycode set
         this->dp->SetTextboxReservedKeys () ;
   }

   this->dp->GetTextboxReservedKeys ( rk ) ; // get a copy of the current data
   if ( altKeycodes )                        // alternate set
   {  //* Compare expected values with actual values *
      if ( (rk.copyKey.type != altKeys.copyKey.type) || 
           (rk.copyKey.key != altKeys.copyKey.key) )         { match[0] = false ; }
      if ( (rk.cutKey.type != altKeys.cutKey.type) || 
           (rk.cutKey.key != altKeys.cutKey.key) )           { match[1] = false ; }
      if ( (rk.pasteKey.type != altKeys.pasteKey.type) || 
           (rk.pasteKey.key != altKeys.pasteKey.key) )       { match[2] = false ; }
      if ( (rk.selAllKey.type != altKeys.selAllKey.type) || 
           (rk.selAllKey.key != altKeys.selAllKey.key) )     { match[3] = false ; }
      if ( (rk.selLeftKey.type != altKeys.selLeftKey.type) || 
           (rk.selLeftKey.key != altKeys.selLeftKey.key) )   { match[4] = false ; }
      if ( (rk.selRightKey.type != altKeys.selRightKey.type) || 
           (rk.selRightKey.key != altKeys.selRightKey.key) ) { match[5] = false ; }
      dataPtr = altKeyNames ;
   }
   else                                      // default set
   {  //* Compare expected values with actual values *
      if ( (rk.copyKey.type != stdKeys.copyKey.type) || 
           (rk.copyKey.key != stdKeys.copyKey.key) )         { match[0] = false ; }
      if ( (rk.cutKey.type != stdKeys.cutKey.type) || 
           (rk.cutKey.key != stdKeys.cutKey.key) )           { match[1] = false ; }
      if ( (rk.pasteKey.type != stdKeys.pasteKey.type) || 
           (rk.pasteKey.key != stdKeys.pasteKey.key) )       { match[2] = false ; }
      if ( (rk.selAllKey.type != stdKeys.selAllKey.type) || 
           (rk.selAllKey.key != stdKeys.selAllKey.key) )     { match[3] = false ; }
      if ( (rk.selLeftKey.type != stdKeys.selLeftKey.type) || 
           (rk.selLeftKey.key != stdKeys.selLeftKey.key) )   { match[4] = false ; }
      if ( (rk.selRightKey.type != stdKeys.selRightKey.type) || 
           (rk.selRightKey.key != stdKeys.selRightKey.key) ) { match[5] = false ; }
   }

   //* Report status of reserved keycode group *
   for ( short k = ZERO ; k < keyCnt ; ++k )
      gs.append( "%s\n", match[k] ? dataPtr[k] : keyError ) ;
   this->dp->WriteParagraph ( this->wp, gs, this->dColor ) ;
   this->dp->RefreshWin() ;

}  //* End setReservedKeycodes() *


//** - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  **
//** - - - - - - - - NON-MEMBER METHODS FOR THIS MODULE  - - - - - - - - - -  **
//** - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  **


//*************************
//*                       *
//*************************
//********************************************************************************
//*                                                                              *
//*                                                                              *
//* Input  :                                                                     *
//*                                                                              *
//* Returns:                                                                     *
//********************************************************************************

