//********************************************************************************
//* File       : ExpandTest.cpp                                                  *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2021-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in NcDialog.hpp            *
//* Date       : 15-Jul-2025                                                     *
//* Version    : (see below)                                                     *
//*                                                                              *
//* Description: Class definition for the expandTest class.                      *
//*              This class is a framework for launching miscellaneous tests     *
//*              related to the NcDialog API.                                    *
//*              It is instantiated by the Dialog4 application, Test10.          *
//*                                                                              *
//* Development Tools: See NcDialog.cpp.                                         *
//********************************************************************************
//* Version History (most recent first):                                         *
//*                                                                              *
//* v:0.00.02 11-Jul-2025                                                        *
//*   - Add a test of the wcbUserAlert() and wcbVersion(gString&) methods.       *
//*                                                                              *
//* v: 0.00.01 10-May-2021                                                       *
//*   - First effort: Provides a mechanism for expanding the number of tests     *
//*     which can be launched within the application.                            *
//*   - Add a comprehensive test for the Pinwheel class functionality.           *
//*     See PinwheelTest.cpp.                                                    *
//*                                                                              *
//********************************************************************************

#include "ExpandTest.hpp"


//**********************
//** Local Prototypes **
//**********************


//****************
//** Local Data **
//****************

//* Keep ddData and enum testNumber in synch *
static const short ddITEMS = 15, ddWIDTH = 34 ;
static const char ddData[ddITEMS][ddWIDTH + 2] = 
{
   " Pinwheel class                   ",
   " Chart class                      ",
   " Slider Controls                  ",
   " Clipboard Info Dialog            ",
   " 04 ----------------------------- ",
   " 05 ----------------------------- ",
   " 06 ----------------------------- ",
   " 07 ----------------------------- ",
   " 08 ----------------------------- ",
   " 09 ----------------------------- ",
   " 10 ----------------------------- ",
   " 11 ----------------------------- ",
   " 12 ----------------------------- ",
   " 13 ----------------------------- ",
   " 14 ----------------------------- ",
} ;
enum testNumber : short
{
   tn_Pinwheel = ZERO,        // Pinwheel class
   tn_Chart,                  // Chart class
   tn_Slider,                 // dctSlider class
   tn_Clipinfo,               // clipboard info dialog
   tn_4,                      // currently unused
   tn_5, tn_6, tn_7, tn_8, tn_9,
   tn_10, tn_11, tn_12, tn_13, tn_14, 
   
   tn_TESTS_DEFINED
} ;

//* Audible alert sequence in case of serious error *
static AudibleAlert aa( 3, 3, 2, 5 ) ;

//* Color scheme for dctDROPDOWN control *
static attr_t monoColor[2] = { attrDFLT, nc.bw } ;

//* Control definitions for prompt dialog *
static InitCtrl promptIc[extCONTROLS_DEFINED] = 
{
   { //* 'SELECT' Dropdown - - - - - - - - - - - - - - - - - - - - - -   extDD *
      dctDROPDOWN,                  // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      3,                            // ulY:       upper left corner in Y
      2,                            // ulX: upper left corner in X
      short(tselROWS - 5),          // lines:     control lines
      short(ddWIDTH + 2),           // cols:      control columns
      (char*)&ddData,               // dispText:  items displayed
      nc.bw,                        // nColor:    non-focus border color
      nc.bw,                        // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     
      ZERO,                         // labY:      offset from control's ulY
      ZERO,                         // labX       offset from control's ulX
      ddBoxDOWN,                    // exType:    (n/a)
      ddITEMS,                      // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    index of initial highlighted element
      monoColor,                    // scrColor:  single-color data display
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &promptIc[extPB],             // nextCtrl:  link in next structure
   },
   {  //* 'DONE' Pushbutton  - - - - - - - - - - - - - - - - - - - - -   extPB *
   dctPUSHBUTTON,                // type:      
   rbtTYPES,                     // rbSubtype: (n/a)
   false,                        // rbSelect:  (n/a)
   short(tselROWS - 2),          // ulY:       upper left corner in Y
   short(tselCOLS / 2 - 4),      // ulX:       upper left corner in X
   1,                            // lines:     control lines
   8,                            // cols:      control columns
   "  DONE  ",                   // dispText:  
   nc.gyR,                       // nColor:    non-focus color
   nc.reG,                       // fColor:    focus color
   tbPrint,                      // filter:    (n/a)
   NULL,                         // label:     (n/a)
   ZERO,                         // labY:      (n/a)
   ZERO,                         // labX       (n/a)
   ddBoxTYPES,                   // exType:    (n/a)
   1,                            // scrItems:  (n/a)
   1,                            // scrSel:    (n/a)
   NULL,                         // scrColor:  (n/a)
   NULL,                         // spinData:  (n/a)
   true,                         // active:    allow control to gain focus
   NULL                          // nextCtrl:  link in next structure
   },
} ;

//*************************
//*      ~expandTest      *
//*************************
//******************************************************************************
//* Destructor.                                                                *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

expandTest::~expandTest ( void )
{

   if ( this->sdp != NULL )               // close the window
      delete ( this->sdp ) ;

}  //* End ~expandTest()

//*************************
//*      expandTest       *
//*************************
//********************************************************************************
//* Constructor.                                                                 *
//*                                                                              *
//* Input  : tLines     : number of display line in terminal window              *
//*          tCols      : number of display columns in terminal window           *
//*          minY       : first available display line                           *
//*                                                                              *
//* Returns: implicitly return class reference                                   *
//********************************************************************************

expandTest::expandTest ( short tRows, short tCols, short minY )
{
   //* Initialize data members *
   this->termRows = tRows ;
   this->termCols = tCols ;
   this->minULY   = minY ;
   this->dRows    = 1 ;                      // dialog dimensions
   this->dCols    = 1 ;
   this->icPtr    = NULL ;                   // control-definition pointer
   this->sdp      = NULL ;                   // object pointer
   this->dColor   = nc.blR ;                 // dialog base color
   this->tColor   = this->dColor | ncbATTR ; // dialog title color
   this->bColor   = nc.gybl | ncbATTR ;      // dialog border color
   this->hColor   = nc.brbl | ncbATTR ;      // dialog highlight color
   this->etOpen   = false ;                  // not yet opened

   //* Initialize remainder of control-definition array.           *
   //* (colors become available after NCurses engine instantiated) *
   promptIc[extPB].nColor = nc.gyR ;
   promptIc[extPB].fColor = nc.reG ;
   promptIc[extDD].nColor = nc.gyR ;
   promptIc[extDD].fColor = nc.rebl ;
   monoColor[1] = nc.bkG ;

   //* Open the prompt dialog. If dialog opens caller may continue.*
   if ( (this->etOpen = this->etOpenPrompt ()) != false )
   {

   }  // OpenWindow()

}  //* End expandTest() 

//*************************
//*     etOpenPrompt      *
//*************************
//******************************************************************************
//* Open the prompt dialog window.                                             *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if dialog window opened successfully, else 'false'         *
//******************************************************************************

bool expandTest::etOpenPrompt ( void )
{
   this->icPtr = promptIc ;
   this->dRows = tselROWS ;
   this->dCols = tselCOLS ;

   short ulY     = this->minULY + 1,         // upper left corner in Y
         ulX     = 40 - (this->dCols / 2) ;  // upper left corner in X
   bool  success = false ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( this->dRows,    // number of display lines
                       this->dCols,    // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       this->bColor,   // border color attribute
                       this->dColor,   // interior color attribute
                       this->icPtr     // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   this->sdp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (this->sdp->OpenWindow()) == OK )
   {
      this->sdp->SetDialogTitle ( "  Select a Test Item  ", this->tColor ) ;
      this->sdp->WriteParagraph ( 1, 3, 
                       "Use up/down arrow keys to highlight\n"
                       "desired test, then press ENTER key.", this->hColor ) ;

      #if 0    // ONLY FOR HIGHLIGHTING A TEST UNDER DEVELOPMENT
      this->sdp->PrevControl () ;
      this->sdp->SetDropdownSelect ( extDD, 3 ) ;
      this->sdp->NextControl () ;
      #endif   // DEVELOPMENT ONLY

      this->sdp->RefreshWin () ;
      success = true ;
   }
   return success ;

}  //* End etOpenPrompt() *

//*************************
//*    etDialogOpened     *
//*************************
//******************************************************************************
//* Satisfy caller's curiosity.                                                *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if dialog opened successfully, else 'false'                *
//******************************************************************************

bool expandTest::etDialogOpened ( void )
{

   return this->etOpen ;

}  //* End etDialogOpened() *

//*************************
//*      etInteract       *
//*************************
//******************************************************************************
//* User interactive experience. Calls the appropriate user-interface method.  *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void expandTest::etInteract ( void )
{
   if ( this->etOpen )
   {
      winPos wpOrig( this->icPtr->ulY, 1 ) ;
      uiInfo info ;
      short icIndex = ZERO ;
      bool done = false ;

      while ( ! done )
      {
         if ( icIndex == extPB )
         {
            this->sdp->EditPushbutton ( info ) ;
            if ( info.dataMod != false )
               done = true ;
            else
               icIndex = this->sdp->NextControl () ;
         }

         else if ( icIndex == extDD )
         {
            info.viaHotkey = true ;    // force the dropdown to open immediately
            this->sdp->EditDropdown ( info ) ;
            icIndex = this->sdp->NextControl () ;

            //* If user has made an active selection *
            if ( info.keyIn == nckENTER || info.keyIn == nckSPACE )
            {
               this->sdp->HideWin () ;       // hide the selection dialog
               if ( info.selMember == tn_Pinwheel )
               {
                  //* Create the PinwheelTest object. *
                  //* Returns when test is complete.  *
                  PinwheelTest *pwt = new PinwheelTest ( wpOrig ) ;
                  if ( pwt != NULL )
                  {
                     if ( ! pwt->isOpen() )
                     {  //* Most likely cause of dialog not opening is that    *
                        //* the terminal window is too small. Give user a clue.*                                    *
                        this->sdp->UserAlert ( &aa ) ;
                     }
                     delete pwt ;
                  }
               }
               else if ( info.selMember == tn_Chart )
               {
                  //* Create the ChartTest object.    *
                  //* Returns when test is complete.  *
                  ChartTest *cht = new ChartTest ( wpOrig ) ;
                  if ( cht != NULL )
                  {
                     if ( ! cht->isOpen() )
                     {  //* Most likely cause of dialog not opening is that    *
                        //* the terminal window is too small. Give user a clue.*
                        short r, c ;
                        cht->GetDimensions ( r, c ) ;
                        r += wpOrig.ypos ;
                        c += wpOrig.xpos ;
                        gString gs( " This test required that the terminal window "
                                    "be at least %hd Rows by %hd Columns. ",
                                    &r, &c ) ;
                        nc.WriteString ( wpOrig, gs.gstr(), nc.reR ) ;
                        this->sdp->UserAlert ( &aa ) ;
                        chrono::duration<short>aWhile( 2 ) ;
                        this_thread::sleep_for( aWhile ) ;
                        nc.ClearLine ( wpOrig.ypos ) ;
                     }
                     delete cht ;
                  }
               }
               else if ( info.selMember == tn_Slider )
               {
                  //* Create the SliderTest object.   *
                  //* Returns when test is complete.  *
                  SliderTest *slide = new SliderTest ( wpOrig ) ;
                  if ( slide != NULL )
                  {
                     if ( ! slide->isOpen() )
                     {  //* Most likely cause of dialog not opening is that    *
                        //* the terminal window is too small. Give user a clue.*
                        short r, c ;
                        slide->GetDimensions ( r, c ) ;
                        r += wpOrig.ypos ;
                        c += wpOrig.xpos ;
                        gString gs( " This test required that the terminal window "
                                    "be at least %hd Rows by %hd Columns. ",
                                    &r, &c ) ;
                        nc.WriteString ( wpOrig, gs.gstr(), nc.reR ) ;
                        this->sdp->UserAlert ( &aa ) ;
                        chrono::duration<short>aWhile( 2 ) ;
                        this_thread::sleep_for( aWhile ) ;
                        nc.ClearLine ( wpOrig.ypos ) ;
                     }
                     delete slide ;
                  }
               }
               else if ( info.selMember == tn_Clipinfo )
               {
                  //* Create the CInfoTest object.    *
                  //* Returns when test is complete.  *
                  ClInfoTest *clinfo = new ClInfoTest ( wpOrig ) ;
                  if ( clinfo != NULL )
                  {
                     delete clinfo ;
                  }
               }
               else if ( false )
               {
                  // CALL NEW TESTS FROM HERE...
               }
               this->sdp->RefreshWin () ;    // restore selection dialog
            }
         }
      }  // while()
   }

}  //* End etInteract() *


//** - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  **
//** - - - - - - - - NON-MEMBER METHODS FOR THIS MODULE  - - - - - - - - - -  **
//** - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  **


//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

