//******************************************************************************
//* File       : RTL_ContentTest.cpp                                           *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2014-2025 Mahlon R. Smith, The Software Samurai *
//*                  GNU GPL copyright notice located in NcDialog.hpp          *
//* Date       : 23-Dec-2020                                                   *
//* Version    : (see below)                                                   *
//*                                                                            *
//* Description: Class definition for the RTL_ContentTest class.               *
//*              This class exercises functionality related to                 *
//*              RTL(right-to-left) text formatting and output.                *
//*              It is instantiated by the Dialog4 application, Test06.        *
//*              See notes below.                                              *
//*                                                                            *
//*              Note that this module uses multi-threading. If your system    *
//*              does not support multithreading, you can disable the secondary*
//*              thread using the ENABLE_RT_MULTITHREAD definition in          *
//*              RTL_ContentTest.hpp.                                          *
//*                                                                            *
//* Developed using GNU G++ (Gcc v: 4.8.0)                                     *
//*  under Fedora Release 16, Kernel Linux 3.6.11-4.fc16.i686.PAE              *
//******************************************************************************
//* Version History (most recent first):                                       *
//*                                                                            *
//* v: 0.00.02 05-Jul-2015                                                     *
//*   - Add conditional compile in rtInteract() to enable/disable mouse        *
//*     interface for testing. This is a good test because all control types   *
//*     are represented.                                                       *
//*                                                                            *
//* v: 0.00.01 24-Jun-2013                                                     *
//*   - Layout based on BillboardTest-class.                                   *
//******************************************************************************
//* Programmer's Notes:                                                        *
//* - The following NcDialog-class functionality must be aware of              *
//*   RTL(right-to-left) text data output.                                     *
//*  1) Dialog window title                                                    *
//*     a) initial title created during instantiation                          *
//*     b) created by SetDialogTitle()                                         *
//*  2) Dialog control labels for all defined control types                    *
//*     a) with optional 'hotkey' definitions                                  *
//*     b) external labels (drawn in the parent dialog window)                 *
//*     c) embedded labels used as control titles                              *
//*  3) Text contents (display data) for all control types which               *
//*     contain text.                                                          *
//*     a) dctSPINNER controls contain only ASCII numeric data,                *
//*        and are not affected by RTL output.                                 *
//*     b) dctRADIOBUTTON controls contain only symbols or single-             *
//*        character data and are not affected by RTL output.                  *
//*     c) All other control types must be aware of RTL display                *
//*        data.                                                               *
//*  4) Of particular interest are the control types whose data can be         *
//*     dynamically updated. As of this writing (NcDialog version 0.0.25),     *
//*     the following control types contain dynamically modifiable data:       *
//*     dctPUSHBUTTON, dctTEXTBOX, dctBILLBOARD, dctSCROLLEXT and dctMENUWIN.  *
//*     Simple tests for all these types are performed in this module.         *
//*     The most interesting test done here is the dynamic switching between   *
//*     RTL and LTR languages in the dctBILLBOARD. See rtDailyThought() method *
//*     for additional information.                                            *
//*                                                                            *
//* - The language used in this test is Yiddish (Hebrew alphapet), which is    *
//*   one of the major RTL (right-to-left) written languages.                  *
//*                                                                            *
//*                                                                            *
//******************************************************************************

#include "RTL_ContentTest.hpp"

//****************
//** Local Data **
//****************
static const short dialogROWS = rtROWS ;  // display lines
static const short dialogCOLS = rtCOLS ;  // display columns

#if ENABLE_RT_MULTITHREAD != 0
//* Module-scope access to the dialog for the non-member callback method *
static NcDialog* rtcuPtr = NULL ;
static short rtControlUpdate ( const short currIndex, 
                               const wkeyCode wkey, bool firstTime ) ;
#endif   // ENABLE_RT_MULTITHREAD

static dspinData dsData( -99, 99, 34, dspinINTEGER ) ;

static attr_t monoColorB[2]  =  { attrDFLT, nc.brR } ; // menuwin data colors
static attr_t monoColorG[2]  =  { attrDFLT, nc.grR } ; // scrollbox data colors

static const char* DialogTitle[] =     // Dialog titles: standard and alternate
{
   "רעכט צו לינקס שפּראַך טעסטינג  "
   "  )tseT noitingoceR egaugnaL LTR(  ",
   "  בייַטנ לויט דער ריי טיטל"
   "  )eltiT etanretlA( ",
} ;
static const char* pbData[] =       // contents of 'Done' pushbutton
{
   " פאַרטיק\n "
   " EN^OD ",
   " RTLtlA \n"
   "  en^oD ",

} ;
static const char* labelText[] =    // control labels (with hotkey designated)
{
   //* Standard Radiobutton 'A' *
   "^A נאָרמאַל ראַדיאָ קנעפּל",
   //* Custom Radiobutton 'B' *
   "^B מנהג ראַדיאָ קנעפּל",
   //* Single-line Textbox 'C' *
   "^C איין שורה טעקסט קעסטל",
   //* Multi-line Textbox 'D' *
   "^D קייפל שורה טעקסט קעסטל",
   //* Billboard control 'R' *
   "בילבאָרד\n"
   "לייענען בלויז )ylnO dae^R(",
   //* Main Menu 'S' (submenu has no label): "Today's Specials" *
   " הייַנט ס ספּעשאַלז )slaicep^S( ",
   //* Scroll Box control '?' *
   " אַוויאַנס )snai^va( ",
   //* Scroll Ext control '?' *
   " מאַמאַלז )s^lammam( ",
   //* Dropdown control '?' *
   "ון בייַ די גאָרטן\n"
   ")ooz eht ta nu^f(",
   //* Spinner control ("ideal chest size") *
   "ידעאַל קאַסטן גרייס\n"
   ")^z("
//   "ידעאַל קאַסטן גרייס )^z(",
} ;
//* Flag controls position of labels in rtScrSB and rtScrSE. *
//* Normally '1' Set to '0' for testing only.                *
#define embeddedLABELS (1)

//* Contents of single-line textbox: "Wash your hands before dinner." *
const char* sltbText = "וואַשן דיין הענט איידער מיטאָג." ;

//* Contents of multi-line textbox: *
//* "The kettle breaks,             *
//*  the kettle doesn't break.      *
//*  It's the kettle's choice."     *
//* From: http://hellopoetry.com/words/362023/pots/poems/
const char* mltbText = 
"האַקן אַ טשײַניק             "
"ניט האַקן מיר קיין טשײַניק "
"אַ ברירה פון די טשײַניק " ;

//* Contents of Billboard: "" *
#if ENABLE_RT_MULTITHREAD == 0
static const char* bbmdData = " )delbasid gnidaerhtitlum( " ;
#else
static const char* bbmdData = NULL ;
#endif   // ENABLE_RT_MULTITHREAD

//* Contents of Main Menu:       *
//*   Baked fish with onions   A *
//*   Beef brisket with garlic B *
//*   Sauteed peas and carrots C *
//*   Green beans              D *
//*   Fruit salad              E *
//*   Mutton chops             F *
//*   Drink Menu               G *
static const short mmITEMS = 7, mmWIDTH = 32 ;
static const char mmData[mmITEMS][mmWIDTH + 27] = 
{
" באַקעד פיש מיט אַניאַנז       ^A",
" רינדערנס בריסקעט מיט קנאָבל ^B",
" סאָטייד פּיז און קעראַץ       ^C",
" שטריקל בינז                ^D",
" פרוכט סאַלאַט                ^E",
" שעפּסנפלייש טשאַפּס           ^F",
">סעלעקציע פון ​​דרינקס        ^G",
} ;

//* Contents of Submenu: *
//*   Beer                       *
//*   Wine                       *
//*   Tea                        *
//*   Coffee                     *
static const short smITEMS = 4, smWIDTH = 18 ;
static const char smData[smITEMS][smWIDTH + 8] = 
{
   " ביר     )ree^b(",
   " ווייַן   )eni^w(",
   " טיי      )ae^t(",
   " קאַווע )eeffo^c(",
} ;

//* Data for rtScrSB, rtScrSE and rtDrpDD *
static const short ddITEMS = 16, sbITEMS = 8, sbWIDTH = 21 ;
static const char sbData[ddITEMS][sbWIDTH + 10] = 
{  //* (8 avians and 8 mammals) *
   " גאַנדז )esoog(     ",
   " הינדל )nekcihc(   ",
   " קאַטשקע )kcud(     ",
   " טויב )noegip(     ",
   " שפּערל )worraps(   ",
   " מאַגפּיע )eipgam(   ",
   " רויטהעלדזל )nibor(",
   " טויבן )evod(      ",
   " קו )woc(          ",
   " פערד )esroh(      ",
   " שעפּס )peehs(      ",
   " כאַזער )goh(       ",
   " ציג )taog(        ",
   " קיניגל )tibbar(   ",
   " מוס )esoom(       ",
   " מויז )esuom(      ",
} ;
static const char* seData[] = 
{
   sbData[8],  sbData[9],  sbData[10], sbData[11], 
   sbData[12], sbData[13], sbData[14], sbData[15],  
} ;
static attr_t seColors[sbITEMS] = 
{ nc.bw, nc.bw, nc.bw, nc.bw, nc.bw, nc.bw, nc.bw, nc.bw } ;



static InitCtrl ic[rtControlsDEFINED] = 
{
   {  //* 'DONE' pushbutton - - - - - - - - - - - - - - - - - - - -   rtDonePB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 7),        // ulY:       upper left corner in Y
      short(dialogCOLS / 2 - 4),    // ulX:       upper left corner in X
      2,                            // lines:     control lines
      8,                            // cols:      control columns
      pbData[0],                    // dispText:  
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[rtStdRB]                  // nextCtrl:  link in next structure
   },
   {  //* 'STD' radiobutton  - - - - - - - - - - - - - - - - - -       rtStdRB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      true,                         // rbSelect:  default selection
      2,                            // ulY:       upper left corner in Y
      24,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      labelText[0],                 // label:
      ZERO,                         // labY:      
      -2,                           // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[rtCstRB],                 // nextCtrl:  link in next structure
   },
   {  //* 'CUSTOM' radiobutton   - - - - - - - - - - - - - - - -       rtCstRB *
      dctRADIOBUTTON,               // type:      
      rbtC5,                        // rbSubtype: standard, 3 chars wide
      true,                         // rbSelect:  default selection
      short(ic[rtStdRB].ulY + 2),   // ulY:       upper left corner in Y
      ic[rtStdRB].ulX,              // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      "{ * }",                      // dispText:  contents if 'set'
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      labelText[1],                 // label:
      ZERO,                         // labY:      
      -2,                           // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[rtSinTB]                  // nextCtrl:  link in next structure
   },
   {  //* 'SINGLE-LINE' textbox  - - - - - - - - - - - - - - - - - -   rtSinTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[rtCstRB].ulY + 3),   // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      27,                           // cols:      control columns
      sltbText,                     // dispText:  (value of t3indexO3)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      labelText[2],                 // label:     
      -1,                           // labY:      
      26,                           // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[rtMulTB]                  // nextCtrl:  link in next structure
   },
   {  //* 'MULTI-LINE' textbox   - - - - - - - - - - - - - - - - - -   rtMulTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[rtSinTB].ulY + 3),   // ulY:       upper left corner in Y
      ic[rtSinTB].ulX,              // ulX:       upper left corner in X
      4,                            // lines:     (n/a)
      27,                           // cols:      control columns
      mltbText,                     // dispText:  (value of t3indexO3)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      labelText[3],                 // label:     
      -1,                           // labY:      
      26,                           // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[rtBilBB]                  // nextCtrl:  link in next structure
   },
   {  //* 'BILLBOARD'      - - - - - - - - - - - - - - - - - - -       rtBilBB *
      dctBILLBOARD,                 // type:
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[rtMulTB].ulY + 7),   // ulY:       upper left corner in Y
      ic[rtMulTB].ulX,              // ulX:       upper left corner in X
      5,                            // lines:     control lines
      ic[rtMulTB].cols,             // cols:      control columns
      bbmdData,                     // dispText:  initial display text
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      labelText[4],                 // label:     
      -2,                           // labY:      label offset Y
      ic[rtMulTB].labX,             // labX       label offset X
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  if specified, attribute array
      NULL,                         // spinData:  (n/a)
      false,                        // active:    view-only
      &ic[rtMwnMW]                  // nextCtrl:  link in next structure
   },      
   {  //* 'Main Menu' Menuwin  - - - - - - - - - - - - - - - - -       rtMwnMW *
      dctMENUWIN,                   // type:
      rbtTYPES,               // rbSubtype: (n/a)
      false,                  // rbSelect:  (n/a)
      ic[rtStdRB].ulY,        // ulY:       upper left corner in Y
      short(ic[rtDonePB].ulX - 2), // ulX:  upper left corner in X
      1,                      // lines:     (n/a)
      mmWIDTH,                // cols:      control columns, 'expanded' state
      (const char*)&mmData,   // dispText:  array of menu items
      nc.bw,         // nColor: non-focus border color / focus 'collapsed' color
      nc.bw,         // fColor: focus border color / non-focus 'collapsed' color
      tbPrint,                // filter:    (n/a)
      labelText[5],           // label:     label text == 'collapsed state' text
      ZERO,                   // labY:      (n/a)
      ZERO,                   // labX       (n/a)
      ddBoxTYPES,             // exType:    (n/a)
      mmITEMS,                // scrItems:  number of menu items
      ZERO,                   // scrSel:    (n/a)
      monoColorB,             // scrColor:  array of color attributes for menu items
      NULL,                   // spinData:  (n/a)
      true,                   // active:    allow control to gain focus
      &ic[rtMwsMW]            // nextCtrl:  link in next structure
   },
   {  //* 'Sub-Menu' Menuwin - - - - - - - - - - - - - - - - - -       rtMwsMW *
      dctMENUWIN,                   // type:
      rbtTYPES,               // rbSubtype: (n/a)
      false,                  // rbSelect:  (n/a)
      short(ic[rtMwnMW].ulY + 9), // ulY: upper left corner in Y
      short(ic[rtMwnMW].ulX + 27),// ulX: upper left corner in X
      1,                      // lines:     (n/a)
      smWIDTH,                // cols:      control columns, 'expanded' state
      (const char*)&smData,   // dispText:  array of menu items
      nc.bw,         // nColor: non-focus border color / focus 'collapsed' color
      nc.bw,         // fColor: focus border color / non-focus 'collapsed' color
      tbPrint,                // filter:    (n/a)
      NULL,                   // label:     submenu has no label
      ZERO,                   // labY:      (n/a)
      ZERO,                   // labX       (n/a)
      ddBoxTYPES,             // exType:    (n/a)
      smITEMS,                // scrItems:  number of menu items
      ZERO,                   // scrSel:    (n/a)
      monoColorB,             // scrColor:  array of color attributes for menu items
      NULL,                   // spinData:  (n/a)
      false,                  // active:    submenu initially inactive
      &ic[rtScrSB]            // nextCtrl:  link in next structure
   },
   {  //* 'SCROLLBOX'      - - - - - - - - - - - - - - - - - - -       rtScrSB *
      dctSCROLLBOX,           // type:
      rbtTYPES,               // rbSubtype: (na)
      false,                  // rbSelect:  (n/a)
      short(ic[rtMwnMW].ulY + 2), // ulY: upper left corner in Y
      ic[rtMwnMW].ulX,        // ulX: upper left corner in X
      7,                      // lines:     control lines
      sbWIDTH,                // cols:      control columns
      (char*)&sbData,         // dispText:  n/a
      nc.bw,                  // nColor:    non-focus border color
      nc.bw,                  // fColor:    focus border color
      tbPrint,                // filter:    (n/a)
      labelText[6],           // label:
      #if embeddedLABELS != 0 // label embedded within control
      ZERO,                   // labY:      offset from control's ulY
      ZERO,                   // labX       offset from control's ulX
      #else                   // label is located outside the control
      -1,                     // labY:      offset from control's ulY
      sbWIDTH,                // labX       offset from control's ulX
      #endif                  // embeddedLABELS
      ddBoxTYPES,             // exType:    (n/a)
      sbITEMS,                // scrItems:  number of elements in text/color arrays
      ZERO,                   // scrSel:    index of initial highlighted element
      monoColorG,             // scrColor:  (n/a)
      NULL,                   // spinData:  (n/a)
      true,                   // active:    allow control to gain focus
      &ic[rtScrSE]            // nextCtrl:  link in next structure
   },
   {  //* 'SCROLLEXT'      - - - - - - - - - - - - - - - - - - -       rtScrSE *
      dctSCROLLEXT,                 // type:
      rbtTYPES,               // rbSubtype: (na)
      false,                  // rbSelect:  (n/a)
      ic[rtScrSB].ulY,        // ulY: upper left corner in Y
      short(ic[rtScrSB].ulX + ic[rtScrSB].cols + 2),// ulX: upper left corner in X
      7,                      // lines:     control lines
      sbWIDTH,                // cols:      control columns
      NULL,                   // dispText:  n/a
      nc.bw,                  // nColor:    non-focus border color
      nc.bw,                  // fColor:    focus border color
      tbPrint,                // filter:    (n/a)
      labelText[7],           // label:     
      #if embeddedLABELS != 0 // label embedded within control
      ZERO,                   // labY:      offset from control's ulY
      ZERO,                   // labX       offset from control's ulX
      #else                   // label is located outside the control
      -1,                     // labY:      offset from control's ulY
      sbWIDTH,                // labX       offset from control's ulX
      #endif                  // embeddedLABELS
      ddBoxTYPES,             // exType:    (n/a)
      sbITEMS,                // scrItems:  number of elements in text/color arrays
      ZERO,                   // scrSel:    index of initial highlighted element
      monoColorG,             // scrColor:  (n/a)
      NULL,                   // spinData:  (n/a)
      true,                   // active:    allow control to gain focus
      &ic[rtDrpDD]            // nextCtrl:  link in next structure
   },
   {  //* 'DROPDOWN;       - - - - - - - - - - - - - - - - - - -       rtDrpDD *
      dctDROPDOWN,            // type:
      rbtTYPES,               // rbSubtype: (na)
      false,                  // rbSelect:  (n/a)
      short(ic[rtScrSB].ulY + ic[rtScrSB].lines + 3),// ulY: upper left corner in Y
      ic[rtScrSB].ulX,        // ulX:       upper left corner in X
      ddITEMS / 2,            // lines:     control lines
      sbWIDTH,                // cols:      control columns
      (char*)&sbData,         // dispText:  n/a
      nc.bw,                  // nColor:    non-focus border color
      nc.bw,                  // fColor:    focus border color
      tbPrint,                // filter:    (n/a)
      labelText[8],           // label:     
      -2,                     // labY:      offset from control's ulY
      short(sbWIDTH - 1),     // labX       offset from control's ulX
      ddBoxDOWN,              // exType:    (n/a)
      ddITEMS,                // scrItems:  number of elements in text/color arrays
      ZERO,                   // scrSel:    index of initial highlighted element
      monoColorG,             // scrColor:  single-color data display
      NULL,                   // spinData:  (n/a)
      true,                   // active:    allow control to gain focus
      &ic[rtSpnSP]            // nextCtrl:  link in next structure
   },
   {  //* 'SPINNER'        - - - - - - - - - - - - - - - - - - -       rtSpnSP *
      dctSPINNER,                   // type:
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[rtDrpDD].ulY,              // ulY:       upper left corner in Y
      ic[rtScrSE].ulX,              // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      13,                           // cols:      control columns
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      labelText[9],                 // label:     
      -1,                           // labY:      
      16,                           // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      &dsData,                      // spinData:  spinner init
      true,                         // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   },      
} ;



//*************************
//*   ~RTL_ContentTest    *
//*************************
//******************************************************************************
//* Destructor.                                                                *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

RTL_ContentTest::~RTL_ContentTest ( void )
{

   if ( this->dp != NULL )          // close the window
      delete ( this->dp ) ;

   #if ENABLE_RT_MULTITHREAD != 0
   rtcuPtr = NULL ;                 // neutralize our callback pointer
   #endif   // ENABLE_RT_MULTITHREAD

}  //* End ~RTL_ContentTest() 

//*************************
//*    RTL_ContentTest    *
//*************************
//******************************************************************************
//* Constructor.                                                               *
//*                                                                            *
//* Input  : tLines     : number of display line in terminal window            *
//*          tCols      : number of display columns in terminal window         *
//*          minY       : first available display line                         *
//*                                                                            *
//* Returns: implicitly return class reference                                 *
//******************************************************************************

RTL_ContentTest::RTL_ContentTest ( short tLines, short tCols, short minY )
{
   //* Initialize data members *
   this->termRows = tLines ;
   this->termCols = tCols ;
   this->minULY   = minY ;
   this->dp       = NULL ;
   this->dColor   = nc.gybl | ncbATTR ;
   this->bColor   = this->dColor ;
   this->rtOpen   = false ;

   //* Initialize remainder of control-definition array.           *
   //* (colors become available after NCurses engine instantiated) *
   ic[rtDonePB].nColor = nc.gyR ;
   ic[rtDonePB].fColor = nc.gyre | ncbATTR ;
   ic[rtStdRB].nColor = ic[rtCstRB].nColor = nc.brbl ;
   ic[rtStdRB].fColor = ic[rtCstRB].fColor = nc.rebl | ncbATTR ;
   ic[rtSinTB].nColor = ic[rtMulTB].nColor = nc.bwB ;
   ic[rtSinTB].fColor = ic[rtMulTB].fColor = nc.gyre | ncbATTR ;
   ic[rtBilBB].nColor = ic[rtBilBB].fColor = nc.brma | ncbATTR ;
   ic[rtMwnMW].nColor = ic[rtMwsMW].nColor = nc.br ;
   ic[rtMwnMW].fColor = ic[rtMwsMW].fColor = monoColorB[1] = nc.brR ;
   ic[rtScrSB].nColor = ic[rtScrSE].nColor = ic[rtDrpDD].nColor = nc.gygr | ncbATTR ;
   ic[rtScrSB].fColor = ic[rtScrSE].fColor = ic[rtDrpDD].fColor = nc.gyre | ncbATTR ;
   ic[rtSpnSP].nColor = nc.bw ; ic[rtSpnSP].fColor = nc.gyre | ncbATTR ;
   for ( short i = ZERO ; i < sbITEMS ; i++ )
      seColors[i] = nc.bw ;

   //* Initialize the dctSPINNER control color *
   dsData.indiAttr = nc.brgr ;

   if ( (this->rtOpen = this->rtOpenDialog ()) != false )
   {

   }  // OpenWindow()

}  //* End RTL_ContentTest() 

//*************************
//*     rtOpenDialog      *
//*************************
//******************************************************************************
//* Open the dialog window.                                                    *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if dialog window opened successfully, else 'false'         *
//******************************************************************************
//* Note that the compiler will not allow us to use a member method as the     *
//* callback target (unless we use the -fpermissive flag), so we must give the *
//* non-member method access to our dialog. Although this is a dangerous       *
//* thing, our target method promises to behave itself :-)                     *
//******************************************************************************

bool RTL_ContentTest::rtOpenDialog ( void )
{
   short ctrY    = this->termRows/2,         // dialog center in Y
         ctrX    = this->termCols/2,         // dialog center in X
         //* Upper left corner in Y (cannot obscure status window) *
         ulY     = (ctrY - dialogROWS/2) >= this->minULY ? 
                   (ctrY - dialogROWS/2) : this->minULY,
         ulX     = ctrX - dialogCOLS / 2 ;   // upper left corner in X
   bool  success = false ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       DialogTitle[0], // dialog title
                       ncltDUAL,       // border line-style
                       this->bColor,   // border color attribute
                       this->dColor,   // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   this->dp = new NcDialog ( dInit ) ;

   //* Configure for RTL dialog title and control labels. *
   dp->DrawLabelsAsRTL () ;

   //* Configure controls with text contents as RTL data. *
   this->dp->DrawContentsAsRTL ( rtSinTB ) ;
   this->dp->DrawContentsAsRTL ( rtMulTB ) ;
   this->dp->DrawContentsAsRTL ( rtBilBB ) ;
   this->dp->DrawContentsAsRTL ( rtMwnMW ) ;
   this->dp->DrawContentsAsRTL ( rtMwsMW ) ;
   this->dp->DrawContentsAsRTL ( rtScrSB ) ;
   this->dp->DrawContentsAsRTL ( rtScrSE ) ;
   this->dp->DrawContentsAsRTL ( rtDrpDD ) ;

   //* Open the dialog window *
   if ( (this->dp->OpenWindow()) == OK )
   {
      //* Attach sub-menu to main menu control *
      short smList[] = { MAX_DIALOG_CONTROLS, MAX_DIALOG_CONTROLS, 
                         MAX_DIALOG_CONTROLS, MAX_DIALOG_CONTROLS, 
                         MAX_DIALOG_CONTROLS, MAX_DIALOG_CONTROLS, 
                         rtMwsMW, -1 } ;
      if ( (this->dp->AttachMenuwinSubmenus ( rtMwnMW, smList )) != OK )
         dp->UserAlert () ;

      //* Initialize dctSCROLLEXT contents *
      ssetData ssd( seData, seColors, sbITEMS, 2, true ) ;
      this->dp->SetScrollextText ( rtScrSE, ssd ) ;

      //* Print dialog window's static text *
      //* Divide window horizontally *
      winPos wp( (ic[rtDonePB].ulY + 2), 1 ) ;
      LineDef  lDef(ncltHORIZ, ncltSINGLE, wp.ypos++, 
                    ZERO, dialogCOLS, this->dColor) ;
      this->dp->DrawLine ( lDef ) ;
      this->dp->WriteParagraph ( wp, 
        "Test of control-object labels and contents, and dialog title written as RTL\n"
        "(right-to-left) text. However hotkeys, if specified, must still be designated\n"
        "using ASCII 'A'-'Z' or 'a'-'z'.             (Radio buttons toggle test modes)",
        this->dColor ) ;
      wp = { short(ic[rtDonePB].ulY - 3), short(ic[rtDonePB].ulX + 10) } ;
      this->dp->DrawBox ( wp.ypos++, wp.xpos, 5, 32, this->dColor,
                          " טיטל פֿאַר אַ באָקס  ", ncltSINGLE, true ) ;
      // Translation: "Sorry about the rudimentary Yiddish,
      //               I should have listened to my grandmother."
      this->dp->WriteParagraph ( wp.ypos, (wp.xpos + 30),
      "נעבעכדיק וועגן די רודאַמענטערי \nייִדיש, איך זאָל האָבן \nאיינגעהערט צו מיין גראַמאַ.",
      this->dColor, false, true ) ;

      this->dp->WriteParagraph ( ic[rtStdRB].ulY + 1, ic[rtStdRB].ulX - 23,
                                 "(display alt dlg title)\n\n"
                                 "(toggle RTL/LTR)", ic[rtStdRB].nColor ) ;

      this->dp->WriteString ( (ic[rtBilBB].ulY + ic[rtBilBB].lines), ic[rtBilBB].ulX, 
                              "(ALT+P to pause Billboard)", 
                              this->dColor & ~ncbATTR ) ;
      dp->RefreshWin () ;           // make everything visible

      #if ENABLE_RT_MULTITHREAD != 0
      //* Establish a callback method.                *
      //* Used only to pause/unpause secondary thread.*
      rtcuPtr = this->dp ;
      this->dp->EstablishCallback ( &rtControlUpdate ) ;
      #endif   // ENABLE_RT_MULTITHREAD

      success = true ;
   }
   return success ;

}  //* End rtOpenDialog() *

//*************************
//*    rtDialogOpened     *
//*************************
//******************************************************************************
//* Satisfy caller's curiosity.                                                *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if dialog opened successfully, else 'false'                *
//******************************************************************************

bool RTL_ContentTest::rtDialogOpened ( void )
{
   return this->rtOpen ;

}  //* End rtDialogOpened() *

//*************************
//*      rtInteract       *
//*************************
//******************************************************************************
//* User interactive experience.                                               *
//* (Not quite "California Adventure"(tm), but still cool.)                    *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void RTL_ContentTest::rtInteract ( void )
{
   #define RTL_ENABLE_MOUSE (1)  // Enable/disable the 'stable' mouse for testing

   if ( this->rtOpen )
   {
      #if RTL_ENABLE_MOUSE != 0
      // Note that because we have multiple threads running, it takes a bit
      // longer than the default interval to see the double-click.
      dp->meEnableStableMouse ( 240 ) ;
      dp->WriteParagraph ( 1, (rtCOLS - 14), 
                           "[mouse input]\n"
                           "[  enabled  ]", nc.brma, true ) ;
      #endif   // RTL_ENABLE_MOUSE

      //* Track user's selections *
      uiInfo Info ;                 // user interface data returned here
      short  icIndex = ZERO ;       // index of control with input focus
      bool   done = false ;         // loop control

      #if ENABLE_RT_MULTITHREAD != 0
      //* Note the odd syntax required to declare a member method *
      //* as the target of a new thread.                          *
      //* NOTE: Because the ncurses library is not thread safe,   *
      //*       screen artifacts may sometimes appear. We are     *
      //*       working on this problem from a higher level, and  *
      //*       hope to have it fixed soon....                    *
      thread DTthread( &RTL_ContentTest::rtDailyThought, this, &done ) ;
      #endif   // ENABLE_RT_MULTITHREAD

      while ( ! done )
      {
         //*******************************************
         //* If focus is currently on a Pushbutton   *
         //*******************************************
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey == false )
               icIndex = this->dp->EditPushbutton ( Info ) ;
            else
               Info.HotData2Primary () ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == rtDonePB )
               {
                  done = true ;
                  #if ENABLE_RT_MULTITHREAD != 0
                  DTthread.join() ;    // wait for secondary thread to return
                  #endif   // ENABLE_RT_MULTITHREAD
               }
            }
         }
         //*******************************************
         //* If focus is currently on a Radio button *
         //*******************************************
         else if ( ic[icIndex].type == dctRADIOBUTTON )
         {
            if ( Info.viaHotkey == false )
               icIndex = this->dp->EditRadiobutton ( Info ) ;
            else
               Info.HotData2Primary () ;
            if ( Info.dataMod != false )
            {  //* For thorough testing of LTR/RTL conversions, we use    *
               //* these radiobutton controls to toggle data conversions. *
               if ( Info.ctrlIndex == rtCstRB )
               {  //* Toggle label/title flag between RTL and LTR. *
                  //* Note: Affects ONLY dialog title.             *
                  dp->DrawLabelsAsRTL ( Info.isSel ) ;

                  //* Controls which contain dynamically updated data can be   *
                  //* (but generally shouldn't be) toggled between LTR and RTL.*
                  //* Note that we could also toggle the Dropdown control, but *
                  //* we don't because it is controlled by another thread.     *
                  dp->DrawContentsAsRTL ( rtDonePB,Info.isSel ) ;
                  dp->DrawContentsAsRTL ( rtSinTB, Info.isSel ) ;
                  dp->DrawContentsAsRTL ( rtMulTB, Info.isSel ) ;
                  dp->DrawContentsAsRTL ( rtMwnMW, Info.isSel ) ;
                  dp->DrawContentsAsRTL ( rtMwsMW, Info.isSel ) ;
                  dp->DrawContentsAsRTL ( rtScrSE, Info.isSel ) ;
                  dp->RefreshWin () ;     // make the changes visible
               }
               //* Test explicit set of dialog title and pushbutton text. *
               bool std ;
               dp->GetRadiobuttonState ( rtStdRB, std ) ;
               dp->SetDialogTitle ( std ? DialogTitle[0] : DialogTitle[1],
                                    std ? attrDFLT : nc.grR ) ;
               dp->SetPushbuttonText ( rtDonePB, 
                                       std ? pbData[0] : pbData[1],
                                       attrDFLT, 
                                       std ? ic[rtDonePB].nColor : nc.gygr ) ;
               dp->RefreshWin () ;
            }
         }
         //**********************************************
         //* If focus is currently on a TextBox control *
         //**********************************************
         else if ( ic[icIndex].type == dctTEXTBOX )
         {
            if ( Info.viaHotkey != false )
               Info.viaHotkey = false ;
            icIndex = this->dp->EditTextbox ( Info ) ;
            if ( Info.dataMod != false )
            { /* Do stuff here */ }
         }
         //**********************************************
         //* If focus is currently on a Menuwin control *
         //**********************************************
         else if ( ic[icIndex].type == dctMENUWIN )
         {  //* If via hotkey, expand immediately. *
            icIndex = this->dp->EditMenuwin ( Info ) ;
            if ( Info.dataMod != false )
            { /* Do stuff here */ }
         }
         //************************************************
         //* If focus is currently on a Scrollbox control *
         //************************************************
         else if ( ic[icIndex].type == dctSCROLLBOX )
         {
            if ( Info.viaHotkey != false )
               Info.viaHotkey = false ;
            icIndex = this->dp->EditScrollbox ( Info ) ;
            if ( Info.dataMod != false )
            { /* Do stuff here */ }
         }
         //************************************************
         //* If focus is currently on a Scrollext control *
         //************************************************
         else if ( ic[icIndex].type == dctSCROLLEXT )
         {
            if ( Info.viaHotkey != false )
               Info.viaHotkey = false ;
            icIndex = this->dp->EditScrollext ( Info ) ;
            if ( Info.dataMod != false )
            { /* Do stuff here */ }
         }
         //***********************************************
         //* If focus is currently on a Dropdown control *
         //***********************************************
         else if ( ic[icIndex].type == dctDROPDOWN )
         {  //* If via hotkey, expand immediately. *
            icIndex = this->dp->EditDropdown ( Info ) ;
            if ( Info.dataMod != false )
            { /* Do stuff here */ }
         }
         //**********************************************
         //* If focus is currently on a Spinner control *
         //**********************************************
         else if ( ic[icIndex].type == dctSPINNER )
         {
            if ( Info.viaHotkey != false )
               Info.viaHotkey = false ;
            icIndex = this->dp->EditSpinner ( Info ) ;
            if ( Info.dataMod != false )
            { /* Do stuff here */ }
         }
         //************************************************
         //* If focus is currently on a Billboard control *
         //************************************************
         else if ( ic[icIndex].type == dctBILLBOARD )
         {
            //* We should never arrive here because Billboard controls     *
            //* are not directly editable AND are always set as 'inactive'.*
            //*            If we DO get here, sound an alert.              *
            AudibleAlert aa( 2, 5, 3, 5 ) ;
            dp->UserAlert ( &aa ) ;
            if ( Info.viaHotkey != false )
               Info.viaHotkey = false ;
         }
         else
         { /* no other control types defined for this method */ }

         //* Move input focus to next/previous control.*
         if ( done == false && Info.viaHotkey == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = this->dp->PrevControl () ; 
            else
               icIndex = this->dp->NextControl () ;
         }
      }  // while()

      #if RTL_ENABLE_MOUSE != 0
      dp->meDisableMouse();
      #endif   // RTL_ENABLE_MOUSE
   }
   else
      { /* Caller is an idiot */ }

}  //* End rtInteract() *

#if ENABLE_RT_MULTITHREAD != 0
//*************************
//*    rtDailyThought     *
//*************************
//******************************************************************************
//* Launched by rtInteract().                                                  *
//* A secondary thread starts here and scrolls through the Thought-for-the-Day *
//* until main thread signals that we should come home                         *
//*                                                                            *
//* Note that the text of bbData[] is partly in Yiddish and partly in English. *
//* As we scroll through the data, we reconfigure the Billboard control for    *
//* the appropriate LTR or RTL output format to match the data being displayed.*
//* During the transition, the control is blanked, so we skip putting the      *
//* thread to sleep at that point which makes for a smoother scroll.           *
//*                                                                            *
//* Note that in the real world, the semaphore 'quittingTime', should have an  *
//* access lock, but this is a test program, and timing is not critical.       *
//*                                                                            *
//* Input  : quittingTime : (pointer to boolean)                               *
//*                         when flag changes to 'true', then rejoin the       *
//*                         main thread                                        *
//* Returns: nothing                                                           *
//******************************************************************************

static bool rtDT_Pause = false ;
void RTL_ContentTest::rtDailyThought ( bool* quittingTime )
{
const short MAX_nIndex = 29, nIndex_Switch = 15 ;
const char* bbData[MAX_nIndex] = 
{
"'אונדזער יונגערמאַן         ",
"מענטש ביינגז וועט אָפֿט -אָפֿט-",
"אַנטוישן אונדז.             ",
"אָבער יעדער איצט און        ",
"דעריבער זיי וועט יבערראַשן  ",
"אונדז אין וועגן אַז         ",
"מאַכן אַלע די אנדערע         ",
"מיסט ווערטיק.              ",
"זיין אַז יבערראַשן. '        ",
"-- 伦马 - ווייכווארג שמואל ",
"            . .            ",
"            . .            ",
"            . .            ",
"            . .            ",
"            . .            ",
"Our fellow human beings    ",
"will often -often-         ",
"disappoint us.             ",
"But every now and then,    ",
"they will surprise us      ",
"in ways that make all the  ",
"other garbage worthwhile.  ",
"Be that surprise.          ",
"-- 马伦 - Software Sam     ",
"            . .            ",
"            . .            ",
"            . .            ",
"            . .            ",
"            . .            ",
} ;

   short nIndex = ZERO ;
   do
   {
      if ( ! rtDT_Pause )
      {
         this->dp->Append2Billboard ( rtBilBB, bbData[nIndex++] ) ;
         if ( nIndex == nIndex_Switch )
            this->dp->DrawContentsAsRTL ( rtBilBB, false ) ;
         if ( nIndex >= MAX_nIndex )
         {
            nIndex = ZERO ;
            this->dp->DrawContentsAsRTL ( rtBilBB ) ;
         }
      }
      if ( nIndex != ZERO && nIndex != nIndex_Switch )
      {
         chrono::duration<short, std::milli>aMoment( 950 ) ;
         this_thread::sleep_for( aMoment ) ;
      }
   }
   while ( *quittingTime == false ) ;

}  //* End rtDailyThought() *
#endif   // ENABLE_RT_MULTITHREAD

//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *
//* - - - - - - - - - - - - Non-member Methods- - - - - - - - - - - - - - - -  *
//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *

//*************************
//*    rtControlUpdate    *
//*************************
//******************************************************************************
//* This is a callback method for manually updating the dialog controls.       *
//*                                                                            *
//*  For this test, the following are updated by the callback method:          *
//*   1. Pause/Un-pause the auto-scroll thread.                                *
//*                                                                            *
//* Input  : currIndex: index of control that currently has focus              *
//*          wkey     : user's key input data                                  *
//*          firstTime: the EstablishCallback() method calls this method once  *
//*                     with firstTime==true, to perform any required          *
//*                     initialization. Subsequently, the NcDialog class       *
//*                     always calls with firstTime==false.                    *
//* Returns: OK                                                                *
//******************************************************************************
//* Notes:                                                                     *
//******************************************************************************

#if ENABLE_RT_MULTITHREAD != 0
static short rtControlUpdate ( const short currIndex, 
                               const wkeyCode wkey, bool firstTime )
{
   if ( !firstTime && (wkey.type == wktEXTEND && wkey.key == 'p') )
   { // Convenient debugging signal: stop/start the auto-scroll thread
      rtDT_Pause = rtDT_Pause ? false : true ;
   }

   return OK ;

}  //* End rtControlUpdate() *
#endif   // ENABLE_RT_MULTITHREAD

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************


