//******************************************************************************
//* File       : ShellTest.hpp                                                 *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2015-2025 Mahlon R. Smith, The Software Samurai *
//*                  GNU GPL copyright notice located in NcDialog.hpp          *
//* Date       : 02-Jan-2020                                                   *
//*                                                                            *
//* Description: Class definition for the ShellTest class.                     *
//*              This class allows you to play with the options for the        *
//*              NcDialog ShellOut() and PseudoShell() methods.                *
//*              It is instantiated by the Dialog4 application, Test08.        *
//*                                                                            *
//* Development Tools: See NcDialog.cpp.                                       *
//******************************************************************************
//* Version History (most recent first):                                       *
//*                                                                            *
//* v: 0.00.01 14-Apr-2015                                                     *
//*   - Layout based on ThreadTest-class.                                      *
//******************************************************************************
//* Programmer's Notes:                                                        *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//******************************************************************************

#include "ShellTest.hpp"

//****************
//** Local Data **
//****************
static const short dialogROWS = soROWS ;  // display lines
static const short dialogCOLS = soCOLS ;  // display columns

static const short ddITEMS = 9 ;
static const short ddWIDTH = 36 ;
static const char  ddText[ddITEMS][ddWIDTH] =
{
   "\\033[0;30m Black                  ",
   "\\033[0;31m Red                    ",
   "\\033[0;32m Green                  ",
   "\\033[1;31m Bold Red               ",
   "\\033[0;34m Blue                   ",
   "\\033[0;35m Purple                 ",
   "\\033[0;36m Cyan                   ",
   "\\033[0;37m Gray                   ",
   "\\033[0;44m\\033[1;33m Blue/Yellow ",
} ;
static const char* esSeq[]
{
   "\033[0;30m", "\033[0;31m", "\033[0;32m", "\033[1;31m", "\033[0;34m", 
   "\033[0;35m", "\033[0;36m", "\033[0;37m", "\033[0;44m\033[1;33m" 
} ;
static attr_t ddAttr[ddITEMS] = 
{
   nc.bw, nc.re, nc.gr, nc.brB, nc.bl, nc.ma, nc.cy, nc.gy 
} ;
static const char* sampleCmd = "ls *.cpp" ;
static const char* rbLabel[] = 
{
   "soX      .     .     .     .",
   "soE      .     Y     .     .",
   "soP      .     .     Y     Y",
   "soPN     .     .     Y     .",
   "soEP     .     Y     Y     Y",
   "soEPN    .     Y     Y     .",
   "soCX     Y     .     .     .",
   "soCE     Y     Y     .     .",
   "soCP     Y     .     Y     Y",
   "soCPN    Y     .     Y     .",
   "soCEP    Y     Y     Y     Y",
   "soCEPN   Y     Y     Y     .",
} ;

static InitCtrl ic[soControlsDEFINED] = 
{
   {  //* 'SHELL' pushbutton  - - - - - - - - - - - - - - - - - - -  soShellPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 5),        // ulY:       upper left corner in Y
      24,                           // ulX:       upper left corner in X
      4,                            // lines:     control lines
      9,                            // cols:      control columns
      " Go to \n"                   // dispText:  
      " ^Shell ",
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[soDonePB],                // nextCtrl:  link in next structure
   },
   {  //* 'DONE' pushbutton - - - - - - - - - - - - - - - - - - - -   soDonePB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[soShellPB].ulY,            // ulY:       upper left corner in Y
      short(dialogCOLS / 2 - 4),    // ulX:       upper left corner in X
      3,                            // lines:     control lines
      8,                            // cols:      control columns
      " ^Done ",                    // dispText:  
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[soShellRB],               // nextCtrl:  link in next structure
   },

   {  //* 'ShellOut' Radiobutton - - - - - - - - - - - - - - - - - - soShellRB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      true,                         // rbSelect:  default selection
      2,                            // ulY:       upper left corner in Y
      4,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "ShellOut Testing",           // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[soPseudoRB],              // nextCtrl:  link in next structure
   },
   {  //* 'PseudoShell' Radiobutton  - - - - - - - - - - - - - - -  soPseudoRB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[soShellRB].ulY + 1), // ulY:       upper left corner in Y
      ic[soShellRB].ulX,            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "PseudoShell Testing "        // label:
      "(not yet implemented)",
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[soX_RB],                  // nextCtrl:  link in next structure
   },

   {  //* 'soX' Radiobutton  - - - - - - - - - - - - - - - - - - - -    soX_RB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[soPseudoRB].ulY + 4),// ulY:       upper left corner in Y
      ic[soPseudoRB].ulX,           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      rbLabel[0],                   // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[soE_RB],                  // nextCtrl:  link in next structure
   },
   {  //* 'soE' (default) Radiobutton  - - - - - - - - - - - - - - -    soE_RB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      true,                         // rbSelect:  default selection
      short(ic[soX_RB].ulY + 1),    // ulY:       upper left corner in Y
      ic[soX_RB].ulX,               // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      rbLabel[1],                   // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[soP_RB],                  // nextCtrl:  link in next structure
   },
   {  //* 'soP'    Radiobutton   - - - - - - - - - - - - - - - - - -    soP_RB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[soE_RB].ulY + 1),    // ulY:       upper left corner in Y
      ic[soE_RB].ulX,               // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      rbLabel[2],                   // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[soPN_RB],                 // nextCtrl:  link in next structure
   },
   {  //* 'soPN'   Radiobutton   - - - - - - - - - - - - - - - - - -   soPN_RB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[soP_RB].ulY + 1),    // ulY:       upper left corner in Y
      ic[soP_RB].ulX,               // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      rbLabel[3],                   // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[soEP_RB],                 // nextCtrl:  link in next structure
   },
   {  //* 'soEP'   Radiobutton   - - - - - - - - - - - - - - - - - -   soEP_RB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[soPN_RB].ulY + 1),   // ulY:       upper left corner in Y
      ic[soPN_RB].ulX,              // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      rbLabel[4],                   // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[soEPN_RB],                // nextCtrl:  link in next structure
   },
   {  //* 'soEPN'  Radiobutton   - - - - - - - - - - - - - - - - - -  soEPN_RB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[soEP_RB].ulY + 1),   // ulY:       upper left corner in Y
      ic[soEP_RB].ulX,              // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      rbLabel[5],                   // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[soCX_RB],                 // nextCtrl:  link in next structure
   },
   {  //* 'soCX'   Radiobutton   - - - - - - - - - - - - - - - - - -   soCX_RB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[soEPN_RB].ulY + 1),  // ulY:       upper left corner in Y
      ic[soEPN_RB].ulX,             // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      rbLabel[6],                   // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[soCE_RB],                 // nextCtrl:  link in next structure
   },
   {  //* 'soCE'   Radiobutton   - - - - - - - - - - - - - - - - - -   soCE_RB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[soCX_RB].ulY + 1),   // ulY:       upper left corner in Y
      ic[soCX_RB].ulX,              // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      rbLabel[7],                   // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[soCP_RB],                 // nextCtrl:  link in next structure
   },
   {  //* 'soCP'   Radiobutton   - - - - - - - - - - - - - - - - - -   soCP_RB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[soCE_RB].ulY + 1),   // ulY:       upper left corner in Y
      ic[soCE_RB].ulX,              // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      rbLabel[8],                   // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[soCPN_RB],                // nextCtrl:  link in next structure
   },
   {  //* 'soCPN'  Radiobutton   - - - - - - - - - - - - - - - - - -  soCPN_RB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[soCP_RB].ulY + 1),   // ulY:       upper left corner in Y
      ic[soCP_RB].ulX,               // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      rbLabel[9],                   // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[soCEP_RB],                // nextCtrl:  link in next structure
   },
   {  //* 'soCEP'  Radiobutton   - - - - - - - - - - - - - - - - - -  soCEP_RB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[soCPN_RB].ulY + 1),  // ulY:       upper left corner in Y
      ic[soCPN_RB].ulX,             // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      rbLabel[10],                  // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[soCEPN_RB],               // nextCtrl:  link in next structure
   },
   {  //* 'soCEPN' Radiobutton   - - - - - - - - - - - - - - - - - - soCEPN_RB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[soCEP_RB].ulY + 1),  // ulY:       upper left corner in Y
      ic[soCEP_RB].ulX,             // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      rbLabel[11],                  // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[soColorDD]                // nextCtrl:  link in next structure
   },

   { //* 'Color' Dropdown  - - - - - - - - - - - - - - - - - - - -   soColorDD *
      dctDROPDOWN,                  // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      short(ic[soX_RB].ulY - 2),    // ulY:       upper left corner in Y
      short(ic[soX_RB].ulX + 38),   // ulX:       upper left corner in X
      ddITEMS + 2,                  // lines:     control lines
      ddWIDTH + 1,                  // cols:      control columns
      (char*)&ddText,               // dispText:  n/a
      nc.bw,                        // nColor:    non-focus border color
      nc.bw,                        // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "Color of Shell Text",        // label:     
      -1,                           // labY:      offset from control's ulY
      ZERO,                         // labX       offset from control's ulX
      ddBoxDOWN,                    // exType:    (n/a)
      ddITEMS,                      // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    index of initial highlighted element
      ddAttr,                       // scrColor:  multi-color data display
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[soCmdTB]                  // nextCtrl:  link in next structure
   },
   {  //* 'Command' Textbox  - - - - - - - - - - - - - - - - - - - -   soCmdTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[soCEPN_RB].ulY + 2), // ulY:       upper left corner in Y
      ic[soCEPN_RB].ulX,            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      35,                           // cols:      control columns
      sampleCmd,                    // dispText:
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Enter an Alternate Command", // label:     
      1,                            // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[soCmdRB]                  // nextCtrl:  link in next structure
   },
   {  //* 'soCmdRB' Radiobutton    - - - - - - - - - - - - - - - - -   soCmdRB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      ic[soCmdTB].ulY,              // ulY:       upper left corner in Y
      short(ic[soCmdTB].ulX + ic[soCmdTB].cols + 3), // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Execute Alternate Command",  // label:
      1,                            // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
} ;


//*************************
//*      ~ShellTest       *
//*************************
//******************************************************************************
//* Destructor.                                                                *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

ShellTest::~ShellTest ( void )
{

   if ( this->dp != NULL )          // close the window
      delete ( this->dp ) ;

}  //* End ~ShellTest() 

//*************************
//*       ShellTest       *
//*************************
//******************************************************************************
//* Constructor.                                                               *
//*                                                                            *
//* Input  : tLines     : number of display line in terminal window            *
//*          tCols      : number of display columns in terminal window         *
//*          minY       : first available display line                         *
//*                                                                            *
//* Returns: implicitly return class reference                                 *
//******************************************************************************

ShellTest::ShellTest ( short tLines, short tCols, short minY )
{
   //* Initialize data members *
   this->termRows = tLines ;
   this->termCols = tCols ;
   this->minULY   = minY ;
   this->dp = NULL ;
   this->dColor = nc.blR ;
   this->bColor = nc.grbl ;
   this->soDone = this->soOpen = false ;

   //* Initialize the colors of the control-definition array.      *
   //* (colors become available after NCurses engine instantiated) *
   ic[soDonePB].nColor = ic[soShellPB].nColor = nc.gyR ;
   ic[soDonePB].fColor = ic[soShellPB].fColor = nc.gyre | ncbATTR ;
   ic[soShellRB].nColor = ic[soPseudoRB].nColor = nc.gyR ;
   ic[soShellRB].fColor = ic[soPseudoRB].fColor = nc.reB ;

   ic[soX_RB].nColor   = ic[soE_RB].nColor   = ic[soP_RB].nColor    = 
   ic[soPN_RB].nColor  = ic[soEP_RB].nColor  = ic[soEPN_RB].nColor  = 
   ic[soCX_RB].nColor  = ic[soCE_RB].nColor  = ic[soCP_RB].nColor   = 
   ic[soCPN_RB].nColor = ic[soCEP_RB].nColor = ic[soCEPN_RB].nColor = nc.brR ;
   ic[soX_RB].fColor   = ic[soE_RB].fColor   = ic[soP_RB].fColor    = 
   ic[soPN_RB].fColor  = ic[soEP_RB].fColor  = ic[soEPN_RB].fColor  = 
   ic[soCX_RB].fColor  = ic[soCE_RB].fColor  = ic[soCP_RB].fColor   = 
   ic[soCPN_RB].fColor = ic[soCEP_RB].fColor = ic[soCEPN_RB].fColor = nc.reB ;

   ic[soColorDD].nColor = nc.gyR ; ic[soColorDD].fColor = nc.grR ;
   ddAttr[0] = nc.bw ;
   ddAttr[1] = nc.re ;
   ddAttr[2] = nc.gr ;
   ddAttr[3] = nc.reB ;
   ddAttr[4] = nc.bl ;
   ddAttr[5] = nc.ma ;
   ddAttr[6] = nc.cy ;
   ddAttr[7] = nc.gy ;
   ddAttr[8] = nc.brbl ;

   ic[soCmdTB].nColor = nc.gyR ; ic[soCmdTB].fColor = nc.gygr | ncbATTR ;
   ic[soCmdRB].nColor = nc.cyR ; ic[soCmdRB].fColor = nc.reR ;

   if ( (this->soOpen = this->soOpenDialog ()) != false )
   {
      /* Nothing to do here */
   }  // OpenWindow()

}  //* End ShellTest() 

//*************************
//*     soOpenDialog      *
//*************************
//******************************************************************************
//* Open the dialog window.                                                    *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if dialog window opened successfully, else 'false'         *
//******************************************************************************

bool ShellTest::soOpenDialog ( void )
{
   short ctrY    = this->termRows/2,         // dialog center in Y
         ctrX    = this->termCols/2,         // dialog center in X
         //* Upper left corner in Y (cannot obscure status window) *
         ulY     = (ctrY - dialogROWS/2) >= this->minULY ? 
                   (ctrY - dialogROWS/2) : this->minULY,
         ulX     = ctrX - dialogCOLS / 2 ;   // upper left corner in X
   bool  success = false ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       "  Access Command-line Shell  ", // dialog title
                       ncltDUAL,       // border line-style
                       this->bColor,   // border color attribute
                       this->dColor,   // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   this->dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (this->dp->OpenWindow()) == OK )
   {  //* Put borders around 'Done' and 'Go-to-Shell' Pushbuttons *
      this->dp->EnablePushbuttonBorder ( soDonePB ) ;
      this->dp->EnablePushbuttonBorder ( soShellPB ) ;

      //* Create Radiobutton groups *
      short XorGroup[] = { soX_RB, soE_RB, soP_RB, soPN_RB, soEP_RB, soEPN_RB, 
                           soCX_RB, soCE_RB, soCP_RB, soCPN_RB, soCEP_RB, soCEPN_RB,
                          -1 } ;
      dp->GroupRadiobuttons ( XorGroup ) ;
      XorGroup[0] = soShellRB ; XorGroup[1] = soPseudoRB ;XorGroup[2] = -1 ;
      dp->GroupRadiobuttons ( XorGroup ) ;

      //* Write static text *
      winPos wp( ic[soX_RB].ulY - 2, ic[soX_RB].ulX + ic[soX_RB].labX) ;
      wp = 
      this->dp->WriteParagraph ( wp,
                                 "       CLEAR  EXIT       PAUSE\n", dColor ) ;
      this->dp->WriteParagraph ( wp,
                                 "OPTION SCREEN MSG  PAUSE  MSG ", dColor | ncuATTR ) ;

      //* You got some 'splainin to do, Lucy! *
      wp = { 10, 70 } ;
      this->dp->DrawBox ( 9, 68, 12, 55, dColor, NULL, ncltDUAL ) ;
      this->dp->WriteParagraph ( wp, 
               "Access to the command shell is a valuable resource  \n"
               "which allows the user to return to the command line \n"
               "to perform tasks that are beyond the scope of the   \n"
               "currently-running application. The application is   \n"
               "temporarily put to sleep, which returns control to  \n"
               "the terminal window.\n"
               "\n"
               "This test demonstrates some of the useful formatting\n"
               "options which can be specified when accessing the   \n"
               "shell program.\n",
               dColor ) ;

      this->dp->RefreshWin () ;     // make everything visible

      success = true ;
   }
   return success ;

}  //* End soOpenDialog() *

//*************************
//*    soDialogOpened     *
//*************************
//******************************************************************************
//* Satisfy caller's curiosity.                                                *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if dialog opened successfully, else 'false'                *
//******************************************************************************

bool ShellTest::soDialogOpened ( void )
{
   return this->soOpen ;

}  //* End soDialogOpened() *

//*************************
//*      soInteract       *
//*************************
//******************************************************************************
//* Talk with dirty, smelly, nose-picking users. (Be sure to shower afterward.)*
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//*                                                                            *
//******************************************************************************

void ShellTest::soInteract ( void )
{
   if ( this->soOpen )
   {
      //* Track user's selections *
      uiInfo Info ;                 // user interface data returned here
      short  icIndex = ZERO ;       // index of control with input focus
      this->soDone = false ;        // loop control

      while ( ! this->soDone )
      {
         //*******************************************
         //* If focus is currently on a Pushbutton   *
         //*******************************************
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey == false )
               icIndex = this->dp->EditPushbutton ( Info ) ;
            else
               Info.HotData2Primary () ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == soDonePB )
               {
                  this->soDone = true ;
               }
               else if ( Info.ctrlIndex == soShellPB )
               {
                  short oindex = this->dp->GetRbGroupSelection ( soE_RB ) ;
                  soOptions option ;
                  switch ( oindex )
                  {
                     case soX_RB:    option = soX ;         break ;
                     case soP_RB:    option = soP ;         break ;
                     case soPN_RB:   option = soPN ;        break ;
                     case soEP_RB:   option = soEP ;        break ;
                     case soEPN_RB:  option = soEPN ;       break ;
                     case soCX_RB:   option = soCX ;        break ;
                     case soCE_RB:   option = soCE;         break ;
                     case soCP_RB:   option = soCP ;        break ;
                     case soCPN_RB:  option = soCPN ;       break ;
                     case soCEP_RB:  option = soCEP ;       break ;
                     case soCEPN_RB: option = soCEPN ;      break ;
                     case soE_RB: default: option = soE ;   break ;
                  }
                  gString gsCmd ;
                  const char* cmdPtr = NULL ;
                  const char* atrPtr = NULL ;
                  bool cmdState ;
                  this->dp->GetRadiobuttonState ( soCmdRB, cmdState ) ;
                  if ( cmdState != false )
                  {
                     this->dp->GetTextboxText ( soCmdTB, gsCmd ) ;
                     cmdPtr = gsCmd.ustr() ;
                  }
                  short aindex = this->dp->GetDropdownSelect ( soColorDD ) ;
                  if ( aindex > ZERO )
                     atrPtr = esSeq[aindex] ;
                  this->dp->ShellOut ( option, cmdPtr, atrPtr ) ;
                  this->dp->PrevControl () ;    // focus stays on this button
               }
            }
         }
         //**************************************************
         //* If focus is currently on a Radiobutton control *
         //**************************************************
         else if ( ic[icIndex].type == dctRADIOBUTTON )
         {
            if ( Info.viaHotkey != false )
               Info.viaHotkey = false ;
            icIndex = this->dp->EditRadiobutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == soShellRB )
               {
                  /* do nothing at this time */
               }
               else if ( Info.ctrlIndex == soPseudoRB )
               {
/* TEMP - NOT YET IMPLEMENTED */ this->dp->SetRbGroupSelection ( soShellRB ) ;
               }
            }
         }
         //**********************************************
         //* If focus is currently on a TextBox control *
         //**********************************************
         else if ( ic[icIndex].type == dctTEXTBOX )
         {
            if ( Info.viaHotkey != false )
               Info.viaHotkey = false ;
            icIndex = this->dp->EditTextbox ( Info ) ;
         }
         //***********************************************
         //* If focus is currently on a DropDown control *
         //***********************************************
         else if ( ic[icIndex].type == dctDROPDOWN )
         {
            if ( Info.viaHotkey != false )
               Info.viaHotkey = false ;
            icIndex = this->dp->EditDropdown ( Info ) ;
         }
         else
         { /* no other control types defined for this method */ }

         //* Move input focus to next/previous control.*
         if ( this->soDone == false && Info.viaHotkey == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = this->dp->PrevControl () ; 
            else
               icIndex = this->dp->NextControl () ;
         }
      }  // while( !done )
   }
   else
      { /* Caller is an idiot */ }

}  //* End soInteract() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************


