//******************************************************************************
//* File       : ThreadTest.cpp                                                *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2014-2025 Mahlon R. Smith, The Software Samurai *
//* Date       : 02-Jan-2020                                                   *
//* Version    : (see below)                                                   *
//*                                                                            *
//* Description: Class definition for the ThreadTestclass.                     *
//*              This class performs stress testing in an attempt to compromise*
//*              the NcWindow/NcDialog input and output thread safety.         *
//*              It is instantiated by the Dialog4 application, Test07.        *
//*              See notes below.                                              *
//*                                                                            *
//*              Note that this module uses multi-threading. The stress test   *
//*              is specifically designed to test I/O interaction among the    *
//*              execution threads. If your system does not support            *
//*              multithreading, then you can disable the secondary threads    *
//*              using the ENABLE_TT_MULTITHREAD definition in ThreadTest.hpp. *
//*              However, this will make the test rather useless.              *
//*                                                                            *
//* Developed using GNU G++ (Gcc v: 4.8.0)                                     *
//*  under Fedora Release 16, Kernel Linux 3.6.11-4.fc16.i686.PAE              *
//******************************************************************************
//* Version History (most recent first):                                       *
//*                                                                            *
//* v: 0.00.01 08-Jul-2014                                                     *
//*   - Layout based on BillboardTest-class.                                   *
//******************************************************************************
//* Programmer's Notes:                                                        *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//******************************************************************************

#include "ThreadTest.hpp"

//****************
//** Local Data **
//****************
static const short dialogROWS = ttROWS ;  // display lines
static const short dialogCOLS = ttCOLS ;  // display columns

static dspinData dsData( 0, 64, 1, dspinINTEGER ) ; // thread execution speed

static InitCtrl ic[ttControlsDEFINED] = 
{
   {  //* 'DONE' pushbutton - - - - - - - - - - - - - - - - - - - -   ttDonePB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 5),        // ulY:       upper left corner in Y
      4,                            // ulX:       upper left corner in X
      3,                            // lines:     control lines
      8,                            // cols:      control columns
      "        \n"                  // dispText:  
      "  Done  \n"
      "        ",
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[ttInputTB],               // nextCtrl:  link in next structure
   },

   {  //* Multi-line textbox   - - - - - - - - - - - - - - - - - -   ttInputTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[ttDonePB].ulY,             // ulY:       upper left corner in Y
      short(dialogCOLS / 2 - 17),   // ulX:       upper left corner in X
      3,                            // lines:     (n/a)
      34,                           // cols:      control columns
      "Edit text while you're waiting! " // dispText:
      "Cursor may blink erratically but should remain where you put it.",
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Text Editing During Chaos",  // label:     
      3,                            // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[ttSpeedSP],               // nextCtrl:  link in next structure
   },

   {  //* 'Execution-speed' Spinner  - - - - - - - - - - - - - -     ttSpeedSP *
      dctSPINNER,                   // type:
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[ttInputTB].ulY + 2), // ulY:       upper left corner in Y
      short(dialogCOLS - 18),       // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      8,                            // cols:      control columns
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Execution Speed\n"           // label:     
      "for Autobots\n\n"
      "(zero to pause)",
      -2,                           // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      &dsData,                      // spinData:  spinner init
      true,                         // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   },      
} ;


//*************************
//*      ~ThreadTest      *
//*************************
//******************************************************************************
//* Destructor.                                                                *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

ThreadTest::~ThreadTest ( void )
{

   if ( this->dp != NULL )          // close the window
      delete ( this->dp ) ;

}  //* End ~ThreadTest() 

//*************************
//*      ThreadTest       *
//*************************
//******************************************************************************
//* Constructor.                                                               *
//*                                                                            *
//* Input  : tLines     : number of display line in terminal window            *
//*          tCols      : number of display columns in terminal window         *
//*          minY       : first available display line                         *
//*                                                                            *
//* Returns: implicitly return class reference                                 *
//******************************************************************************

ThreadTest::ThreadTest ( short tLines, short tCols, short minY )
{
   //* Initialize data members *
   this->termRows = tLines ;
   this->termCols = tCols ;
   this->minULY   = minY ;
   this->dp = NULL ;
   this->dColor = nc.gybl | ncbATTR ;
   this->bColor = this->dColor ;
   this->abY = 4 ;
   this->abX = 1 ;
   this->abInc = ZERO ;  // (will be set below in ttOpenDialog())
   this->ttDone = this->ttOpen = false ;
   //* Initialize output colors and output modes *
   attr_t colorList[] = 
   {
      nc.rebl | ncbATTR, nc.grbl | ncbATTR, nc.brbl | ncbATTR, 
      nc.cybl | ncbATTR, nc.mabl | ncbATTR, 
      nc.bw, nc.bw, nc.bw, nc.bw // spares - to avoid accidents
   } ;
   ttModes modeList[] = 
   {
      ttChar,     // 1st thread outputs characters
      ttStrn,     // 2nd thread outputs strings
      ttChar,     // 3rd thread outputs characters
      ttPara,     // 4th thread outputs paragraphs
      ttStrn,     // 5th thread outputs strings
      ttChar, ttChar, ttChar, ttChar,  // spares
   } ;
   for ( short i = ZERO ; i < abTHREADS ; i++ )
   {
      this->abColor[i] = colorList[i] ;
      this->abMode[i]  = modeList[i] ;
   }

   //* Initialize remainder of control-definition array.           *
   //* (colors become available after NCurses engine instantiated) *
   ic[ttDonePB].nColor = nc.gyR ;
   ic[ttDonePB].fColor = nc.gyre | ncbATTR ;
   ic[ttInputTB].nColor = nc.blgy ;
   ic[ttInputTB].fColor = nc.gybr | ncbATTR ;
   ic[ttSpeedSP].nColor = nc.bw ; ic[ttSpeedSP].fColor = nc.gyre | ncbATTR ;

   //* Initialize the dctSPINNER control color *
   dsData.indiAttr = nc.brgr ;

   if ( (this->ttOpen = this->ttOpenDialog ()) != false )
   {
      /* Nothing to do here */
   }  // OpenWindow()

}  //* End ThreadTest() 

//*************************
//*     ttOpenDialog      *
//*************************
//******************************************************************************
//* Open the dialog window.                                                    *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if dialog window opened successfully, else 'false'         *
//******************************************************************************
//* Note that the compiler will not allow us to use a member method as the     *
//* callback target (unless we use the -fpermissive flag), so we must give the *
//* non-member method access to our dialog. Although this is a dangerous       *
//* thing, our target method promises to behave itself :-)                     *
//******************************************************************************

bool ThreadTest::ttOpenDialog ( void )
{
   short ctrY    = this->termRows/2,         // dialog center in Y
         ctrX    = this->termCols/2,         // dialog center in X
         //* Upper left corner in Y (cannot obscure status window) *
         ulY     = (ctrY - dialogROWS/2) >= this->minULY ? 
                   (ctrY - dialogROWS/2) : this->minULY,
         ulX     = ctrX - dialogCOLS / 2 ;   // upper left corner in X
   bool  success = false ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       "  Pull On a Thread: " // dialog title
                       "Thread-safety Stress Test  ",
                       ncltDUAL,       // border line-style
                       this->bColor,   // border color attribute
                       this->dColor,   // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   this->dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (this->dp->OpenWindow()) == OK )
   {
      //* Divide window horizontally *
      LineDef  lDef(ncltHORIZ, ncltSINGLE, 3, ZERO, dialogCOLS, this->dColor) ;
      this->dp->DrawLine ( lDef ) ;
      lDef.startY = dialogROWS - 6 ;
      this->dp->DrawLine ( lDef ) ;

      //* Divide window vertically *
      const char* ColHead[] = 
      {
         "       Autobot 01       ",
         "       Autobot 02       ",
         "       Autobot 03       ",
         "       Autobot 04       ",
         "       Autobot 05       ",
      } ;
      winPos wp( 2, 1 ) ;
      short abi = ZERO ;
      wp = this->dp->WriteString ( wp, ColHead[abi], this->abColor[abi] ) ;
      this->abInc = wp.xpos ;       // remember column width
      lDef.type = ncltVERT ;
      lDef.startY = wp.ypos + 1 ;
      lDef.startX = wp.xpos++ ;
      lDef.length = dialogROWS - 8 ;
      this->dp->DrawLine ( lDef ) ;
      for ( ++abi ; abi < abTHREADS ; abi++ )
      {
         wp = this->dp->WriteString ( wp, ColHead[abi], this->abColor[abi] ) ;
         if ( abi < 4 )
         {
            lDef.startX = wp.xpos++ ;
            this->dp->DrawLine ( lDef ) ;
         }
      }

      this->dp->RefreshWin () ;     // make everything visible

      //* Launch our minions into the application space. *
      //*   "Cry havoc and let slip the dogs of war!"    *
      #if ENABLE_TT_MULTITHREAD != 0
      //* Note the odd syntax required to declare a      *
      //* member method as the target of a new thread.   *
      winPos abPos( 4, 1 ) ;
      for ( short t = ZERO ; t < abTHREADS ; t++ )
      {
         this->autobots[t] = thread( &ThreadTest::ttAutoPilot, this, 
                                     this->abColor[t], abPos, this->abMode[t] ) ;
         abPos.xpos += this->abInc ;

         //* Stagger the launch for a more dramatic effect *
         chrono::duration<short, std::milli>aMoment( 750 ) ;
         this_thread::sleep_for( aMoment ) ;
      }
      #endif   // ENABLE_TT_MULTITHREAD

      success = true ;
   }
   return success ;

}  //* End ttOpenDialog() *

//*************************
//*    ttDialogOpened     *
//*************************
//******************************************************************************
//* Satisfy caller's curiosity.                                                *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if dialog opened successfully, else 'false'                *
//******************************************************************************

bool ThreadTest::ttDialogOpened ( void )
{
   return this->ttOpen ;

}  //* End ttDialogOpened() *

//*************************
//*      ttInteract       *
//*************************
//******************************************************************************
//* Talk with dirty, smelly, nose-picking users. (Be sure to shower afterward.)*
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Note on Autobot execution speed. This is the spinner value used to         *
//* determine the length of the pause between writes:                          *
//*                                                                            *
//* Actual output speed is limited by the speed of the video hardware, the     *
//* CPU speed and the efficiency of the hardware/software multi-threading      *
//* support. On the test machine Intel(tm) dual-core, 2.26GHz with generic     *
//* video hardware, the visible update speed maxes out (and actually starts to *
//* slow down) at spinner values above 48 for 5 simultaneous output threads    *
//* plus the main user-interface thread. The slow-down can be attributed to    *
//* the additional overhead of swapping out the active threads.                *
//*                                                                            *
//* At this setting, our CPU is loafing along at about 15%, and we still have  *
//* plenty of video bandwidth available to concurrently run a video under      *
//* VLC Media Player(tm).                                                      *
//*                                                                            *
//* The bottleneck is most likely the I/O bandwidth of the single-threaded     *
//* 'ncurses' system library; however, if you have sexier hardware, you could  *
//* try increasing the Autobots' upper speed limit and see what happens.       *
//*             (see initialization in ThreadTest constructor ).               *
//*                                                                            *
//* In any case, at the maximum setting AND while holding down the TAB key to  *
//* continually redraw the control objects, we get no output artifacts,        *
//* indicating that our thread safety implementation is basically sound.       *
//* If you can reliably produce on-screen artifacts, please contact the author *
//* with your test scenario.                                                   *
//******************************************************************************

void ThreadTest::ttInteract ( void )
{
   if ( this->ttOpen )
   {
      //* Track user's selections *
      uiInfo Info ;                 // user interface data returned here
      short  icIndex = ZERO ;       // index of control with input focus
      this->ttDone = false ;        // loop control

      while ( ! this->ttDone )
      {
         //*******************************************
         //* If focus is currently on a Pushbutton   *
         //*******************************************
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey == false )
               icIndex = this->dp->EditPushbutton ( Info ) ;
            else
               Info.HotData2Primary () ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == ttDonePB )
               {
                  this->ttDone = true ;
                  #if ENABLE_TT_MULTITHREAD != 0
                  //* Wait for Autobot threads to return *
                  for ( short t = ZERO ; t < abTHREADS ; t++ )
                     this->autobots[t].join() ;
                  #endif   // ENABLE_TT_MULTITHREAD
               }
            }
         }
         //**********************************************
         //* If focus is currently on a TextBox control *
         //**********************************************
         else if ( ic[icIndex].type == dctTEXTBOX )
         {
            if ( Info.viaHotkey != false )
               Info.viaHotkey = false ;
            icIndex = this->dp->EditTextbox ( Info ) ;
         }
         //**********************************************
         //* If focus is currently on a Spinner control *
         //**********************************************
         else if ( ic[icIndex].type == dctSPINNER )
         {
            if ( Info.viaHotkey != false )
               Info.viaHotkey = false ;
            icIndex = this->dp->EditSpinner ( Info ) ;
         }
         else
         { /* no other control types defined for this method */ }

         //* Move input focus to next/previous control.*
         if ( this->ttDone == false && Info.viaHotkey == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = this->dp->PrevControl () ; 
            else
               icIndex = this->dp->NextControl () ;
         }
      }  // while( !done )
   }
   else
      { /* Caller is an idiot */ }

}  //* End ttInteract() *

#if ENABLE_TT_MULTITHREAD != 0
//*************************
//*      ttAutoPilot      *
//*************************
//******************************************************************************
//* Launched by ttInteract().                                                  *
//* The secondary threads start here. Each thread has a specific display area  *
//* in which to write AND a specific color attribute for the text written.     *
//* If text of a different color shows up in the thread's display space, it    *
//* indicates that thread safety still needs work.                             *
//* Execute until main thread signals that we should come home.                *
//*                                                                            *
//* Note that in the real world, the semaphore 'ttDone', should have an        *
//* access lock, but this is a test program, and timing is not critical.       *
//*                                                                            *
//* Input  : abColor      : color in which to draw the output                  *
//*          basePos      : Y/X position for base of display area              *
//*          oMode        : Output Mode:                                       *
//*                         1 == WriteChar()                                   *
//*                         2 == WriteString()                                 *
//*                         3 == WriteParagraph()                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void ThreadTest::ttAutoPilot ( attr_t abColor, const winPos& basePos, short oMode )
{
   const int maxSNOOZE_INTERVAL = 1500 ;     // (milliseconds)
   const short lastDISPLINE = ttROWS - 7 ;   // display area limit
   const short asLINES = 29 ;                // number of story lines
   const char* AutobotStory[asLINES] =       // output data
   { //123456789+123456789+1234
      "The car pulled away,    \n",
      "crushing her heart under\n",
      "its wheels as it went.  \n",
      "Sarah hadn't gone two   \n",
      "meters toward the house \n",
      "when her knees gave way,\n",
      "and she sat/fell in slow\n",
      "motion on the sidewalk. \n",
      "                        \n",
      "It had been a tough week\n",
      "for love, and the       \n",
      "strained goodbye-smile  \n",
      "had finished her        \n",
      "reserves.               \n",
      "                        \n",
      "Sarah was no stranger   \n",
      "to heartbreak, but the  \n",
      "wisdom gained from past \n",
      "breakups, had not yet   \n",
      "had time to inform the  \n",
      "present one -- and she  \n",
      "sat gasping like an     \n",
      "overly-energetic        \n",
      "goldfish who has leaped \n",
      "its way to freedom, and \n",
      "is now quietly learning \n",
      "regret on the carpet.   \n",
      " - - - - - - - - - - -  \n",
      "                        \n",
   } ;

   winPos wp = basePos ;            // start position for output
   wp.ypos += oMode - 1 ;
   int spVal, snooze, asIndex = ZERO ;
   do
   {
      this->dp->GetSpinnerValue ( ttSpeedSP, spVal ) ;
      snooze = maxSNOOZE_INTERVAL / (spVal > ZERO ? spVal : 1) ;
      if ( spVal > ZERO )
      {
         switch ( oMode )
         {
            case ttPara:   // OUTPUT MODE: WriteParagraph()
               wp = this->dp->WriteParagraph ( wp, AutobotStory[asIndex++], 
                                               abColor, true ) ;
               if ( wp.ypos > lastDISPLINE )
                  wp = basePos ;
               break ;
            case ttStrn:   // OUTPUT MODE: WriteString()
               this->dp->WriteString ( wp, AutobotStory[asIndex++], abColor, true ) ;
               if ( ++wp.ypos > lastDISPLINE )
                  wp = basePos ;
               break ;
            case ttChar:   // OUTPUT MODE: WriteChar()
            default:       // (prevent stupid errors)
               for ( short i = ZERO ; AutobotStory[asIndex][i] != nckNEWLINE ; i++ )
                  wp = this->dp->WriteChar ( wp, AutobotStory[asIndex][i], abColor ) ;
               ++asIndex ;
               if ( ++wp.ypos > lastDISPLINE )
                  wp = basePos ;
               else
                  wp.xpos = basePos.xpos ;
               this->dp->RefreshWin () ;
               break ;
         }
         if ( asIndex >= asLINES )
            asIndex = ZERO ;
      }
      chrono::duration<short, std::milli>aMoment( snooze ) ;
      this_thread::sleep_for( aMoment ) ;
   }
   while ( this->ttDone == false ) ;

}  //* End ttAutoPilot() *
#endif   // ENABLE_TT_MULTITHREAD

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************


