//******************************************************************************
//* File       : Dialogx.hpp                                                   *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2019-2025 Mahlon R. Smith, The Software Samurai *
//*                 GNU GPL copyright notice located below                     *
//* Date       : 19-Mar-2025                                                   *
//* Version    : (see AppVersion string)                                       *
//*                                                                            *
//* Description: Class definition and miscellaneous constant definitions       *
//*              for the Dialogx application used to experimentally access     *
//*              the Wayland clipboards.                                       *
//*                                                                            *
//******************************************************************************
//* Copyright Notice:                                                          *
//* This program is free software: you can redistribute it and/or modify it    *
//* under the terms of the GNU General Public License as published by the Free *
//* Software Foundation, either version 3 of the License, or (at your option)  *
//* any later version.                                                         *
//*                                                                            *
//* This program is distributed in the hope that it will be useful, but        *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
//* for more details.                                                          *
//*                                                                            *
//* You should have received a copy of the GNU General Public License along    *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.            *
//*                                                                            *
//*         Full text of the GPL License may be found in the Texinfo           *
//*         documentation for this program under 'Copyright Notice'.           *
//******************************************************************************

//****************
//* Header Files *
//****************
#include "GlobalDef.hpp"   //* NcDialog API family of classes
#include <fstream>         //* C++ file I/O
#include <limits.h>        //* for realpath() and PATH_MAX
#include <dirent.h>        //* Read directory entries
#include <sys/stat.h>      //* Define stat(), lstat(), etc.

//***************
//* Definitions *
//***************

//* The following c-type structures defined in the standard header files 
//* need to be C++ data types: stat{} dirent{}, etc.
//typedef struct stat64 FileStats ;
//typedef struct dirent64 deStats ;
//typedef struct tm Tm ;
//typedef struct passwd PwdInfo ;
//typedef struct group GrpInfo ;
//typedef struct utimbuf uTime ;

//* Application-specific clipboard access key assignments  *
//* Note that all of these keycodes are of type wktFUNKEY, *
//* that is, they are classified as 'function keys' by     *
//* the ncurses system library.                            *
enum cbAccess : wchar_t {
                         caSELECT_RIGHT  = nckSRIGHT,
                         caSELECT_LEFT   = nckSLEFT,
                         caSELECT_ALL    = nckC_A,
                         caCOPY_SELECTED = nckC_C,
                         caCUT_SELECTED  = nckC_X,
                         caPASTE         = nckC_V
                        } ;

//***********************************************
//* Constant Data:                              *
//***********************************************

//* Application version string. Keep it current! *
const char* const AppVersion = "0.0.08" ;
const char* const crYears = "2019-2025" ;
const char* const AppTitle = "Dialogx - v:" ;
const char* const titleTemplate = "%s%s (c)%s The Software Samurai  " ;

const short MAX_PATH  = PATH_MAX ;  //* Size of buffer to hold path/filename
const short MIN_ROWS = 32 ;         //* Minimum lines needed for dialog window
const short MIN_COLS = 132 ;        //* Minimum columns needed for dialog window
const short tbHEIGHT = 9 ;          //* multi-line textbox rows
const short tbWIDTH = 36 ;          //* textbox columns

//* Root directory path. (during processing, this is often a special case) *
const wchar_t* const ROOT_PATH = L"/" ;
const wchar_t fSLASH = L'/' ;
const wchar_t wNEWLINE = L'\n' ;

//* Gather and process command-line arguments *
const short LN_NAME = 32 ;    // size of 'altLocale' buffer
class commArgs
{
   public:
   commArgs ( int argc, char** argv, char** argenv ) :
               argCount(argc), argList(argv), envList(argenv)
   {
      *this->appPath   = NULLCHAR ;
      this->diagPause  = ZERO ;        // no pause
      *this->altLocale = NULLCHAR ;    // default locale
      this->mouseFlag  = true ;        // enabled by default
      this->bRtl = this->bRj = 
      this->verFlag = this->helpFlag = false ;
   }
   short    argCount ;           // application's 'argc'
   char**   argList ;            // application's 'argv'
   char**   envList ;            // application's 'argenv'
   char     appPath[MAX_PATH] ;  // directory of application executable
   char     altLocale[LN_NAME] ; // Name of alternate locale 
   short    diagPause ;          // -p option: >0 == pause after start-up diagnostics
                                 // (undocumented) >1 == display verbose start-up diagnostics
   bool     bRtl ;               // if 'true', configure for testing RTL input
   bool     bRj ;                // if 'true', configure for testing right-justified input
   bool     mouseFlag ;          // -m option: 'true' if enable mouse support
   bool     verFlag ;            // --version: 'true' if display command-line version
   bool     helpFlag ;           // -H option: 'true' if display command-line help
} ;


//********************************
//* Application class definition *
//********************************

class Dialogx
{
   public:

   virtual ~Dialogx ( void ) ;            //* destructor
   Dialogx ( commArgs& clArgs ) ;         //* constructor

   short OpenDialog ( const commArgs& ca ) ; //* Open the main dialog window
   void  UserInterface ( void ) ;         //* Interact with the user.
   void  ClearClipboard ( void ) ;        //* Clear the active clipboard
   short GetCbBytes ( short srcCb ) ;     //* Returns bytes stored on specified clipboard
   void  ReportFormats ( void ) ;         //* Report available MIME types and cb bytes
   void  TestConnection ( void ) ;        //* Test the connection with the Wayland clipboard
   void  ReinitWaylandCB ( void ) ;       //* Reinitialize WaylandCB class
   void  WheresWayland ( void ) ;         //* Give warning msg if no Wayland connection

   //* Utility methods *
   bool GetCommandLineArgs ( commArgs& ca ) ;
   void DiagMsg ( const char* msg, attr_t color, bool newLine = true ) ;
   void DisplayVersion ( void ) ;
   void DisplayHelp ( void ) ;

   //****************
   //* Data Members *
   //****************
   NcDialog *dPtr ;           //* Pointer to application dialog
   InitCtrl *sic;             //* Pointer to an array of control objects
   char appPath[MAX_PATH] ;   //* Application path (current working directory)
   WaylandCB *wclip ;         //* Pointer to Wayland interface class
   bool    wcbConnect ;       //* 'true' if Wayland clipboard connection established
   winPos  suPos ;            //* Cursor position for start-up messages
   attr_t  dColor ;           //* Color attribute for menu, prompt, help
   attr_t  hColor ;           //* Highlight (emphasis) color
   attr_t  eColor ;           //* Error-message color
   attr_t  pbfColor ;         //* Pushbutton focus color
   attr_t  pbnColor ;         //* Pushbutton non-focus color
   attr_t  tbfColor ;         //* Textbox focus color
   attr_t  tbnColor ;         //* Textbox non-focus color
   attr_t  rbfColor ;         //* Radiobutton focus color
   attr_t  rbnColor ;         //* Radiobutton non-focus color
   attr_t  bbfColor ;         //* Billboard focus color
   attr_t  bbnColor ;         //* Billboard non-focus color

   //* Size of terminal window (set when NCurses Engine starts)  *
   //* Modified when terminal window is resized.                 *
   short termRows,
         termCols,
         dlgRows,
         dlgCols ;
} ;