//******************************************************************************
//* File       : Wayclip.hpp                                                   *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2019-2025 Mahlon R. Smith, The Software Samurai *
//*                 GNU GPL copyright notice located below                     *
//* Date       : 19-Mar-2025                                                   *
//* Version    : (see AppVersion string)                                       *
//*                                                                            *
//* Description: Class definition and miscellaneous constant definitions       *
//*              for the Dialogx application used to experimentally access     *
//*              the Wayland clipboards.                                       *
//*                                                                            *
//******************************************************************************
//* Copyright Notice:                                                          *
//* This program is free software: you can redistribute it and/or modify it    *
//* under the terms of the GNU General Public License as published by the Free *
//* Software Foundation, either version 3 of the License, or (at your option)  *
//* any later version.                                                         *
//*                                                                            *
//* This program is distributed in the hope that it will be useful, but        *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
//* for more details.                                                          *
//*                                                                            *
//* You should have received a copy of the GNU General Public License along    *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.            *
//*                                                                            *
//*         Full text of the GPL License may be found in the Texinfo           *
//*         documentation for this program under 'Copyright Notice'.           *
//******************************************************************************

//****************
//* Header Files *
//****************
#include <iostream>        //* Standard I/O definitions
#include <sstream>         //* Definitions for iostringstream classes
#include <iomanip>         //* Output formatting specifications
#include <cstring>         //* String manipulation from C
#include <unistd.h>        //* UNIX system interface
#include <csignal>         //* For capture of SIGINT (CTRL+C)
#include <cctype>          //* Character testing
#include <cmath>           //* Math library
#include <cstdlib>         //* Misc. functionality
#include <ctime>           //* Access to system time
#include <thread>          //* std::thread definition (and POSIX thread defs)
#include <condition_variable> //* Additional info for threads
#include <chrono>          //* Timers for temporarily putting threads to sleep
#include <mutex>           //* For access locks on critical data
#include <locale>          //* Locale handling for character encoding, etc. 
#include <langinfo.h>      //* Support for world languages identification/conversion
#include <fstream>         //* C++ file I/O
#include <limits.h>        //* for realpath() and PATH_MAX
#include <dirent.h>        //* Read directory entries
#include <sys/stat.h>      //* Define stat(), lstat(), etc.

#include "WaylandCB.hpp"   //* Access to Wayland clipboard
#include "gString.hpp"     //* gString class definition (text formatting)

//***************
//* Definitions *
//***************

using namespace std ;                  //* Scope quailfier


//** Definitions **
#define ZERO         0
#define SPACE        (' ')
#define DASH         ('-')
#define COMMA        (',')
#define PERIOD       ('.')
#define COLON        (':')
#define TILDE        ('~')
#define UNDERSCORE   ('_')
#define CARET        ('^')
#define TAB          ('\t')
#define NEWLINE      ('\n')
#define NULLCHAR     ('\0')
#define ESC          (0x1B)

//* Indices for ANSI color-attribute sequences *
enum attr_t : short
{
   attrBK = ZERO, // black-on-white
   attrBR,        // brown-on-white
   attrBRr,       // white-on-brown
   attrRE,        // red-on-white
   attrBL,        // blue-on-white
   attrGR,        // green-on-white
   attrCOLORS
} ;

//* ANSI escape sequence for various text color attributes.*
//* see enum attr_t above.                                 *
//* ANSI Colors (partial list):                                                *
//* =========== Foreground  Background   ============ Foreground  Background   *
//* Black       \033[0;30m  \033[0;40m   Dark Gray    \033[1;30m  \033[1;40m   *
//* Red         \033[0;31m  \033[0;41m   Bold Red     \033[1;31m  \033[1;41m   *
//* Green       \033[0;32m  \033[0;42m   Bold Green   \033[1;32m  \033[1;42m   *
//* Yellow      \033[0;33m  \033[0;43m   Bold Yellow  \033[1;33m  \033[1;43m   *
//* Blue        \033[0;34m  \033[0;44m   Bold Blue    \033[1;34m  \033[1;44m   *
//* Purple      \033[0;35m  \033[0;45m   Bold Purple  \033[1;35m  \033[1;45m   *
//* Cyan        \033[0;36m  \033[0;46m   Bold Cyan    \033[1;36m  \033[1;46m   *
//* Light Gray  \033[0;37m  \033[0;47m   White        \033[1;37m  \033[1;47m   *
const wchar_t* const Ansi[attrCOLORS] = 
{
   L"\033[0;30m",    // black-on-white
   L"\033[0;33m",    // brown-on-white
   L"\033[0;43m\033[1;37m", // white-on-brown
   L"\033[1;31m",    // bold-red-on-white
   L"\033[0;34m",    // blue-on-white
   L"\033[0;32m",    // green-on-white
} ;

//* Text format identifier. Specifies the internal  *
//* format used to communicate with WaylandCB class.*
enum txtFmt : short { txtfmtUTF8 = ZERO, txtfmtUTF32 = 1, txtfmtGSTRING = 2 } ;

//***********************************************
//* Constant Data:                              *
//***********************************************

//* Application version string. Keep it current! *
const char* const AppVersion = "0.0.05" ;
const char* const crYears = "2019-2025" ;
const char* const AppTitle = "Wayclip - v:" ;
const char* const titleTemplate = "%s%s (c)%s The Software Samurai  " ;

const short MAX_PATH  = PATH_MAX ;  //* Size of buffer to hold path/filename

//* Root directory path. (during processing, this is often a special case) *
const wchar_t* const ROOT_PATH = L"/" ;
const wchar_t fSLASH = L'/' ;
const wchar_t wNEWLINE = L'\n' ;

//* Gather and process command-line arguments *
class commArgs
{
   public:
   commArgs ( int argc, char** argv, char** argenv ) :
               argCount(argc), argList(argv), envList(argenv)
   {
      *this->appPath   = NULLCHAR ;
      this->verFlag = this->helpFlag = false ;
   }
   short    argCount ;           // application's 'argc'
   char**   argList ;            // application's 'argv'
   char**   envList ;            // application's 'argenv'
   char     appPath[MAX_PATH] ;  // directory of application executable
   bool     primeFlag ;          // 'true' == Primary, 'false' == Regular clipboard
   bool     verFlag ;            // --version: 'true' if display command-line version
   bool     helpFlag ;           // -H option: 'true' if display command-line help
} ;


//********************************
//* Application class definition *
//********************************

class Wayclip
{
   public:

   virtual ~Wayclip ( void ) ;            //* destructor
   Wayclip ( commArgs& clArgs ) ;         //* constructor

   void  UserInterface ( void ) ;         //* Interact with the user.
   void  SetClipboard ( bool primary ) ;  //* Copy test data to clipboard
   void  GetClipboard ( bool primary ) ;  //* Retrieve test data from clipboard
   void  ClearClipboard ( bool primary ); //* Clear the clipboard
   short GetCbBytes ( bool primary ) ;    //* Returns bytes stored on specified clipboard
   short GetCbChars ( bool primary ) ;    //* Returns characters stored on specified clipboard
   void  SpecifyActiveCb ( void ) ;       //* Specify the active clipboard (Regular/Primary)
   void  SpecifyTestText ( void ) ;       //* Specify test data
   void  SpecifyInternalFormat ( void ) ; //* Specify internal text formatting
   void  ReportFormats ( void ) ;         //* Report available MIME types and cb bytes
   void  TestConnection ( void ) ;        //* Test the connection with the Wayland clipboard
   void  TestBufferingLimits ( void ) ;   //* Test the maximum data size for communications
   void  ReinitWaylandCB ( void ) ;       //* Reinitialize WaylandCB class
   void  WheresWayland ( bool verbose = false ) ; //* Give warning msg if no Wayland connection

   //* Utility methods *
   bool GetCommandLineArgs ( commArgs& ca ) ;
   void WriteParagraph ( const wchar_t* msg, attr_t bColor, attr_t eColor, 
                         bool clear = false ) ;
   void DisplayTitle ( bool clear = false, bool menu = false ) ;
   void DisplayMenu ( bool clear = false ) ;
   void DisplayVersion ( void ) ;
   void DisplayHelp ( void ) ;

   //****************
   //* Data Members *
   //****************
   char appPath[MAX_PATH] ;   //* Application path (current working directory)
   WaylandCB *wclip ;         //* Pointer to Wayland interface class
   gString  testData ;        //* Test data to be written to clipboard
   txtFmt  txtfmt ;           //* Internal communications format
   const locale* ioLocale ;   //* Application locale (language and text processing)
   bool    wcbPrimary ;       //* 'true'==Primary, 'false'==Regular
   bool    wcbConnect ;       //* 'true' if Wayland clipboard connection established
   attr_t  apnColor ;         //* Application 'normal' color
   attr_t  aphColor ;         //* Application 'highlight' color
   attr_t  aprColor ;         //* Application 'reverse' color
   attr_t  apeColor ;         //* Application 'error message' color
   attr_t  stnColor ;         //* Status 'normal' color
   attr_t  sthColor ;         //* Status 'highlight' color

} ;