//********************************************************************************
//* File       : KeymapAuto.cpp                                                  *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2019-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in Keymap.hpp              *
//* Date       : 01-Jun-2024                                                     *
//* Version    : (see Keymap.cpp)                                                *
//*                                                                              *
//* Description:                                                                 *
//* The Keymap utility interactively maps the keycode associated with each       *
//* key on the keyboard. The application is based on the NcDialog API and is     *
//* designed to analyze the keycode definitions and key escape sequences visible *
//* within the terminal window environment.                                      *
//*                                                                              *
//* This module contain the template and code for generating a C++ header file   *
//* which defines the keycodes as tested. If the generated header file contains  *
//* keycodes which do not match our static keycode list (keyDesc keyGroup[])     *
//* then those keycodes may require manual adjustment.                           *
//*                                                                              *
//* Development Tools: See NcDialog.cpp.                                         *
//********************************************************************************
//* Version History (see Keymap.cpp)                                             *
//*                                                                              *
//********************************************************************************

//****************
//* Header Files *
//****************
//* All necessary information for:                  *
//* NCurses, NcWindow, NcDialog and gString classes.*
#include "GlobalDef.hpp"
#include "Keymap.hpp"            //* Application-class definition

//***************
//* Definitions *
//***************
//* Class for reporting the system local date/time.                            *
//* ---------------------------------------------------------------------------*
#define DECODE_TZ (1)   // if non-zero, decode time-zone fields of localTime
const int64_t maxETIME32 = 0x07FFFFFFF,      // largest 32-bit time_t
              minETIME32 = -maxETIME32 - 1 ; // smallest 32-bit time_t
const short ltFMT_LEN = 16 ;                 // length of time string buffers
//* Important note about the year 2038 bug:                                    *
//* Because 'time_t' is defined as a signed, 32-bit value on 32-bit systems,   *
//* time reporting will break down for date/time values                        *
//*                     >= 19 Jan 2038 at 11:14:08                             *
//* For this reason and for others, we have defined our own local-time class.  *
//* This class uses a signed, 64-bit value for storing the epoch time.         *
//* It is hoped that all Linux systems will have been updated long before this *
//* becomes an issue; however, even now (2013), some applications are using    *
//* date/time manipulations that exceed the 32-bit limit. BEWARE!!             *
//* NOTE: time arithmetic should always be performed on signed 64-bit values.  *
//*                                                                            *
class localTime
{
   public:
   localTime ( void ) { this->reset() ; }
   void reset ( void )
   {  //* Set to epoch date: Thursday, 01 Jan. 1970 @ 08:00:00
      this->sysepoch = ZERO ;
      this->epoch = ZERO ;
      this->day = 4 ;
      this->year = 1970 ;
      this->month = this->date = 1 ;
      this->hours = 8 ;
      this->minutes = this->seconds = ZERO ;
      this->julian = ZERO ;
      #if DECODE_TZ != 0   // time-zone data
      *this->timezone = NULLCHAR ;
      *this->utc_zone  = NULLCHAR ;
      this->gmtoffset = ZERO ;
      this->dst       = false ;
      #endif   // DECODE_TZ
   }
   int64_t  epoch ;        //* Seconds since the epoch (01-Jan-1970)
   time_t   sysepoch ;     //* system epoch time (see note)
   USHORT   day ;          //* Day of the week (0 == Sun, ..., 6 == Sat)
   USHORT   date ;         //* Day of the month (1-[28|29|30|31])
   USHORT   month ;        //* Month of the year (1 == Jan, ... 12 == Dec)
   USHORT   year ;         //* Year (four digit year)
   USHORT   hours ;        //* Time of day, hour (0-23)
   USHORT   minutes ;      //* Time of day, minutes (0-59)
   USHORT   seconds ;      //* Time of day, seconds (0-59)
   USHORT   julian ;       //* Completed days since beginning of year

   #if DECODE_TZ != 0   // time-zone data
   // See DecodeEpochTime method for more information.
   char     timezone[ltFMT_LEN] ; //* string representing the local time zone (useless)
   char     utc_zone[ltFMT_LEN] ; //* string representing the UTC offset for local time zone
   short    gmtoffset ;    //* Expressed as (signed) number of seconds offset from UTC+0:00
   bool     dst ;          //* 'true' if daylight savings time in effect,
                           //* 'false if not DST or info not available
   #endif   // DECODE_TZ
} ;


//***************
//* Prototypes  *
//***************
static bool GetLocalTime ( localTime& ft ) ;
static void FormatDateString ( const localTime& lt, gString& str ) ;

//**********
//*  Data  *
//**********
static char klVersion[VERBUF] = "\0" ; //* Current version of o-s kernel (Fedora)
static char ncVersion[VERBUF] = "\0" ; //* Current version of ncurses C-language library
static char gtVersion[VERBUF] = "\0" ; //* Current version of gnometerm
static char knVersion[VERBUF] = "\0" ; //* Current version of KDE konsole
static char xtVersion[VERBUF] = "\0" ; //* Current version of XTerm


const char* const Header1 = 
"//******************************************************************************\n"
"//* File       : NCursesKeyDef_auto.hpp                                        *\n"
"//* Author     : Mahlon R. Smith                                               *\n"
"//*              Copyright (c) 2019-%04hu Mahlon R. Smith, The Software Samurai *\n"
"//*                 GNU GPL copyright notice below                             *\n" ;

const char* const Header2 = 
"//* Created    : %S                                           *\n" ;

const char* const Header3 = 
"//*                                                                            *\n"
"//* Description: This is an automatically-generated file created by the        *\n"
"//*              NcDialogAPI utility \"keymap\".                                 *\n"
"//* This file defines the group of keycodes which are system dependent, and    *\n"
"//* vary according to OS distribution, terminal-emulator version, keyboard     *\n"
"//* hardware and other factors.                                                *\n"
"//*                                                                            *\n"
"//* A copy of this file, using default keycode definitions is embedded within  *\n"
"//* the NcDialogAPI header file: \"NCursesKeyDef.hpp\" which defines the         *\n"
"//* keyboard keycodes available within the terminal window (gnome-terminal,    *\n"
"//* konsole, xterm, etc.).                                                     *\n"
"//* The default key map was generated on a clean Fedora system with            *\n"
"//* U.S. English keyboard (and numeric keypad)                                 *\n"
"//*                                                                            *\n" ;

const char* const Header4kl = "//*       kernel-release   : v:%s" ;
const char* const Header4nc = "//*       ncurses C library: v:%s" ;
const char* const Header4gt = "//*       GNOME terminal   : v:%s" ;
const char* const Header4kn = "//*       KDE Konsole      : v:%s" ;
const char* const Header4xt = "//*       XTerm            : v:%s" ;

const char* const Header5 = 
"//*                                                                            *\n"
"//* For systems which report non-default keycode definitions, a customized     *\n"
"//* version of this file may be included within the \"NCursesKeyDef.hpp\"        *\n"
"//* header file. The \"#include\" statement is under a conditional compilation   *\n"
"//* flag as shown below which allows for easily switching between default and  *\n"
"//* custom keycodes.                                                           *\n"
"//*                                                                            *\n"
"//*    //** Conditional Compile:                                               *\n"
"//*    //** (1) Use default keycode definitions.                               *\n"
"//*    //** (0) Use auto-generated custom keycode definitions.                 *\n"
"//*    #define DEFAULT_KEY_DEFINITIONS (1)                                     *\n"
"//*                                                                            *\n"
"//*    #if DEFAULT_KEY_DEFINITIONS == 0    // Custom Keycode Definitions       *\n"
"//*    #include \"NCursesKeyDef_auto.hpp\"                                       *\n"
"//*    #else    // DEFAULT_KEYCODE_DEFINITIONS==(1)                            *\n"
"//*    (default key definitions located here....)                              *\n"
"//*    #endif   // DEFAULT_KEYCODE_DEFINITIONS                                 *\n"
"//*                                                                            *\n"
"//* ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ---- ----*\n"
"//* This file may be generated with all default values by invoking \"keymap\"    *\n"
"//* with the 'g' option:   keymap -g                                           *\n"
"//*                                                                            *\n"
"//* Alternatively, the file may be generated after all keycodes have been      *\n"
"//* interactively tested.                                                      *\n"
"//*                                                                            *\n"
"//* Note: While the file may also be generated using data from incomplete      *\n"
"//* testing of the keycodes, the resulting keycode definitions may not be      *\n"
"//* reliable.                                                                  *\n"
"//*                                                                            *\n"
"//* If a key combination remains untested, either because it was captured by   *\n"
"//* the system or because the test was skipped, then the default keycode and   *\n"
"//* escape sequence for that record will be used.                              *\n"
"//* If such errors occur during testing please consult the \"keymap\" log file   *\n"
"//* for details; then, disable as many of the system \"accelerator keys\" as     *\n"
"//* possible and run the tests again.                                          *\n"
"//*                                                                            *\n"
"//******************************************************************************\n"
"\n" ;
const char* const Header6 = 
"\n"
"//*********** %S key group (%hd keycodes) ***********\n"
"//            Key Name       Keycode    Key Esc Sequence (hex)  Key Combination\n"
"//            -----------    --------   ----------------------  ---------------------------\n" ;

const char* recordTemplate = "const wchar_t %-12S = 0x%04X ;  // %-21S (%S)\n" ;

//*************************
//*  GenerateHeaderFile   *
//*************************
//********************************************************************************
//* Create a C++ header file to be #include'd in NCursesKeyDef.hpp.              *
//* This file incorporates the results of keycode testing.                       *
//*                                                                              *
//* Input  : genDflt: (optional, 'false' by default)                             *
//*                   'false' generate table using test results                  *
//*                   'true'  generate table using default values                *
//*                                                                              *
//* Returns: 'true' if file successfully created, else 'false'                   *
//********************************************************************************

bool Keymap::GenerateHeaderFile ( bool genDflt )
{
   gString gsOut ;               // data formatting
   short kIndex,                 // sub-sequence index
         kCount ;                // sub-sequence count
   bool status = false ;         // return value

   //* If version numbers not already captured, do it now.*
   if ( *ncVersion == NULLCHAR )
      this->GetVersions () ;

   if ( this->yxPos.xpos != ZERO )
      this->WriteParagraph ( L"\n", this->menuAttr, true ) ;

   //* If target file already exists, rename it *
   gsOut.compose( "%s/%s", this->appPath, hdrName ) ;
   if ( (this->TargetExists ( gsOut )) )
      this->RenameAsBackup ( gsOut ) ;

   //* Open a new file *
   ofstream ofs( hdrName, (ofstream::out | ofstream::trunc) ) ;
   if ( ofs.is_open() )
   {
      status = true ;

      //* Get current local time and format it for output *
      localTime lt ;
      GetLocalTime ( lt ) ;
      gString gsdate ;
      FormatDateString ( lt, gsdate ) ;
      gsOut.compose( Header1, &lt.year ) ;
      ofs << gsOut.ustr() ;
      gsOut.compose( Header2, gsdate.gstr() ) ;
      ofs << gsOut.ustr() ;
      ofs << Header3 ;

      gsOut.compose( Header4kl, klVersion ) ;      // kernel version
      gsOut.padCols( 79 ) ;
      gsOut.append( L"*\n" ) ;
      ofs << gsOut.ustr() ;
      gsOut.compose( Header4nc, ncVersion ) ;      // ncurses library version
      gsOut.padCols( 79 ) ;
      gsOut.append( L"*\n" ) ;
      ofs << gsOut.ustr() ;
      gsOut.compose( Header4gt, gtVersion ) ;      // gnome-terminal version
      gsOut.padCols( 79 ) ;
      gsOut.append( L"*\n" ) ;
      ofs << gsOut.ustr() ;
      gsOut.compose( Header4kn, knVersion ) ;      // KDE Konsole version
      gsOut.padCols( 79 ) ;
      gsOut.append( L"*\n" ) ;
      ofs << gsOut.ustr() ;
      gsOut.compose( Header4xt, xtVersion ) ;      // XTerm version
      gsOut.padCols( 79 ) ;
      gsOut.append( L"*\n" ) ;
      ofs << gsOut.ustr() ;

      ofs << Header5 ;
      ofs.flush() ;

      //* Navigation Keys *
      kIndex = this->nvIndex ;
      kCount = this->nvCount ;
      gsOut.compose( Header6, L"Navigation", &kCount ) ;
      ofs << gsOut.ustr() ;
      for ( short k = ZERO ; k < kCount ; ++k, ++kIndex )
      {
         if ( genDflt )
         {
            gsOut.compose( recordTemplate, 
                           this->keyLookup[kIndex].name,
                           &this->keyLookup[kIndex].code,
                           this->keyLookup[kIndex].seq, 
                           this->keyLookup[kIndex].desc ) ;
         }
         else
         {
            gsOut.compose( recordTemplate, 
                           this->keyLookup[kIndex].name,
                           ((this->keyLookup[kIndex].kcode > ZERO) ?
                              &this->keyLookup[kIndex].kcode :
                              &this->keyLookup[kIndex].code),
                           ((*this->keyLookup[kIndex].kseq != NULLCHAR) ?
                              this->keyLookup[kIndex].kseq :
                              this->keyLookup[kIndex].seq),
                           this->keyLookup[kIndex].desc ) ;
         }
         ofs << gsOut.ustr() ;
      }
      ofs.flush() ;

      //* Function Keys *
      kIndex = this->fnIndex ;
      kCount = this->fnCount ;
      gsOut.compose( Header6, L"Function", &kCount ) ;
      ofs << gsOut.ustr() ;
      for ( short k = ZERO ; k < kCount ; ++k, ++kIndex )
      {
         if ( genDflt )
         {
            gsOut.compose( recordTemplate, 
                           this->keyLookup[kIndex].name,
                           &this->keyLookup[kIndex].code,
                           this->keyLookup[kIndex].seq, 
                           this->keyLookup[kIndex].desc ) ;
         }
         else
         {
            gsOut.compose( recordTemplate, 
                           this->keyLookup[kIndex].name,
                           ((this->keyLookup[kIndex].kcode > ZERO) ?
                              &this->keyLookup[kIndex].kcode :
                              &this->keyLookup[kIndex].code),
                           ((*this->keyLookup[kIndex].kseq != NULLCHAR) ?
                              this->keyLookup[kIndex].kseq :
                              this->keyLookup[kIndex].seq),
                           this->keyLookup[kIndex].desc ) ;
         }
         ofs << gsOut.ustr() ;
      }
      ofs.flush() ;

      //* Numeric Keypad Keys *
      kIndex = this->kpIndex ;
      kCount = this->kpCount ;
      gsOut.compose( Header6, L"Keypad", &kCount ) ;
      ofs << gsOut.ustr() ;
      for ( short k = ZERO ; k < kCount ; ++k, ++kIndex )
      {
         if ( genDflt )
         {
            gsOut.compose( recordTemplate, 
                           this->keyLookup[kIndex].name,
                           &this->keyLookup[kIndex].code,
                           this->keyLookup[kIndex].seq, 
                           this->keyLookup[kIndex].desc ) ;
         }
         else
         {
            gsOut.compose( recordTemplate, 
                           this->keyLookup[kIndex].name,
                           ((this->keyLookup[kIndex].kcode > ZERO) ?
                              &this->keyLookup[kIndex].kcode :
                              &this->keyLookup[kIndex].code),
                           ((*this->keyLookup[kIndex].kseq != NULLCHAR) ?
                              this->keyLookup[kIndex].kseq :
                              this->keyLookup[kIndex].seq),
                           this->keyLookup[kIndex].desc ) ;
         }
         ofs << gsOut.ustr() ;
      }
      ofs << "\n//** End " << hdrName << " **\n" ;
      ofs.flush() ;
      ofs.close() ;
      this->hdrGenerated = true ;
   }

   this->WriteParagraph ( L" ", this->menuAttr, true ) ;
   gsOut.compose ( "Header file \"%s\" ", hdrName ) ;
   attr_t attr = (this->okAttr | ncrATTR) ;
   if ( status != false )
      gsOut.append( "created in current directory.\n" ) ;
   else
   {
      gsOut.insert( "Error! - " ) ;
      gsOut.append( "not created.\n" ) ;
      attr = (this->errAttr | ncrATTR) ;
   }
   this->WriteParagraph ( gsOut, attr, true ) ;

   #if 0    // For Debugging Only - display the captured version numbers
   gsOut.compose( " kernel-release  v:%s\n"
                  " ncurses library v:%s\n"
                  " gnome-terminal  v:%s\n"
                  " KDE Konsole     v:%s\n"
                  " XTerm           v:%s\n",
                  klVersion, ncVersion, gtVersion, knVersion, xtVersion ) ;
   this->WriteParagraph ( gsOut, nc.bl, true ) ;
   #endif   // For Debugging Only

   return status ;

}  //* End GenerateHeaderFile() *

//*************************
//*  GenerateTransTable   *
//*************************
//********************************************************************************
//* Generate the TransTable2[] translation table which is used by the            *
//* Dialog2 test/demonstration app which is part of the NcDialogAPI              *
//* distribution. This translation table matches the keycode received to its     *
//* human-readable name for interactively testing individual keycodes.           *
//*                                                                              *
//* Input  : dflt  : (optional, 'false' by default)                              *
//*          if 'false', reference captured test keycodes ( 'kcode' member )     *
//*          if 'true',  reference default keycodes ( 'code' member )            *
//*                                                                              *
//* Returns: 'true' if file successfully created, else 'false'                   *
//********************************************************************************
//* Notes:                                                                       *
//* ------                                                                       *
//* The validity of this table depends on the validity of the captured keycode   * 
//* data.                                                                        *
//* 1) The size of the table is: 0x0250 - 0x0102 + 1 == 0x014F == 335 (decimal)  *
//*    records.                                                                  *
//* 2) Each record is an 11-byte array (10 chars max + nullchar).                *
//* 3) The table is laid out in five columns, plus a sixth column which is a     *
//*    comment indicating the keycode range for that row.                        *
//* 4) Because efficiency is not critical, we do not sort the table, but         *
//*    instead scan the table 325 times searching for the next keycode.          *
//* 5) If a keycode is not defined for use by the NcDialog API, then a default   *
//*    text string will be used for that record in the form: "key_xnnn" where    *
//*    "nnn" is the keycode number in hexadecimal.                               *
//*                                                                              *
//* ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---    *
//* Programmer's Note: The previous sizes of the table were:                     *
//*  0x24B - 0x0102 + 1 == 0x014A == 330 (decimal) records.                      *
//*  0x246 - 0x0102 + 1 == 0x0145 == 325 (decimal) records.                      *
//* These expansions were made to accomodate keycode definitions that were       *
//* expanded by Wayland/ncurses/gnome/fedora/witches? to fall outside the table  *
//* range.                                                                       *
//* We hope that the Witches of Wayland have now finished their little jokes.    *
//*                                                                              *
//********************************************************************************

bool Keymap::GenerateTransTable ( bool dflt )
{
   const wchar_t firstREC = 0x0102,       // first test record
                 lastREC  = 0x0250 ;      // last test record
   const short TRECS = lastREC - firstREC + 0x0001 ;
   const short trecWIDTH = 32 ;
   class tt2Records
   {
      public:
      wchar_t r[trecWIDTH] ;
   } ;
   tt2Records *ttPtr = new tt2Records[TRECS] ;

   gString gsOut ;               // data formatting
   wchar_t trgCode = firstREC ;  // keycode being scanned for
   short   srcIndex ;            // index into this->keyLookup[] array
   bool    goodRec,              // 'true' if record copied
           status = false ;      // return value

   for ( short trgIndex = ZERO ; trgIndex < TRECS ; ++trgIndex )
   {
      goodRec = false ;
      for ( srcIndex = ZERO ; srcIndex < this->klCount ; ++srcIndex )
      {
         if ( (dflt  && (this->keyLookup[srcIndex].code == trgCode)) ||
              (!dflt && (this->keyLookup[srcIndex].kcode == trgCode)) )
         {
            //* Get the keycode name *
            gsOut = this->keyLookup[srcIndex].name ;

            //* If keycode is among the group of duplications between keypad  *
            //* and and main keyboard, then report the keyname for the main   *
            //* keyboard. This ASSUMES that the listed keycodes actually do   *
            //* share a keycode with the main keyboard equivalents. This      *
            //* assumption may bite us on the fanny at some point.            *
            if ( (gsOut.find( L"nckp", ZERO, true )) >= ZERO )
            {
               switch ( trgCode )
               {
                  case nckpINSERT:
                  case nckpAINSERT:
                  case nckpDELETE:
                  case nckpADEL:
                  case nckpCDEL:
                  //case nckpACDEL:
                  case nckpCPGUP:
                  case nckpCPGDN:
                     gsOut.erase( L'p', ZERO, true ) ;
                     break ;
                  default:
                     break ;
               } ;
            }
            gsOut.copy( ttPtr[trgIndex].r, trecWIDTH ) ;
            goodRec = true ;
            break ;
         }
      }
      if ( ! goodRec )
      {
         // special case: KEY_F0 is not associated with an actual key
         if ( trgCode == KEY_F0 )
            gsOut = "KEY_F0" ;
         // special case: KEY_RESIZE is not associated with an actual key
         else if ( trgCode == KEY_RESIZE )
            gsOut = "nckRESIZE" ;
         // special case: KEY_MOUSE is not associated with an actual key
         else if ( trgCode == KEY_MOUSE )
            gsOut = "KEY_MOUSE" ;
         else
            gsOut.compose( "key_x%03X", &trgCode ) ;
         gsOut.copy( ttPtr[trgIndex].r, trecWIDTH ) ;
      }
      ++trgCode ;    // next target keycode
   }
   --trgCode ;       // index the last element in the array

   //* Write the records to a file *
   ofstream ofs( tt2Name, (ofstream::out | ofstream::trunc) ) ;
   if ( ofs.is_open() )
   {
      wchar_t startCode = firstREC,
              endCode   = startCode + 0x0004 ;
              gsOut.compose( "TransTable2 Array - Records 0x%04X - 0x%04X\n",
                             &firstREC, &trgCode ) ;
      ofs << gsOut.ustr() ;

      gsOut.compose( "Generated by Keymap v:%s , NcDialogAPI v:%s , ncursesw link library v:%s\n",
                     AppVersion, dPtr->Get_NcDialog_Version (), nc.Get_nclibrary_Version () ) ;
      ofs << gsOut.ustr() << endl ;

      for ( short trgIndex = ZERO ; trgIndex < TRECS ; )
      {
         gsOut.compose( "   \"%-10S\", \"%-10S\", \"%-10S\", \"%-10S\", \"%-10S\", //0x%03X-0x%03X", 
                        ttPtr[trgIndex].r, 
                        ttPtr[trgIndex + 1].r, 
                        ttPtr[trgIndex + 2].r, 
                        ttPtr[trgIndex + 3].r, 
                        ttPtr[trgIndex + 4].r, 
                        &startCode, &endCode ) ;
         ofs << gsOut.ustr() << endl ;
         trgIndex += 5 ;
         startCode += 5 ;
         endCode += 5 ;
      }
      ofs << "\nEND TABLE" << endl ;
      ofs.close() ;
      status = true ;
   }
   
   //* Delete the local allocation *
   delete [] ttPtr ;

   this->WriteParagraph ( L" ", this->menuAttr, true ) ;
   gsOut.compose ( "Translation Table \"%s\" ", tt2Name ) ;
   attr_t attr = (this->okAttr | ncrATTR) ;
   if ( status != false )
      gsOut.append( "created in current directory.\n" ) ;
   else
   {
      gsOut.insert( "Error! - " ) ;
      gsOut.append( "not created.\n" ) ;
      attr = (this->errAttr | ncrATTR) ;
   }
   this->WriteParagraph ( gsOut, attr, true ) ;

   return status ;

}  //* End GenerateTransTable() *

//*************************
//*      ViewLogFile      *
//*************************
//********************************************************************************
//* View the session log file.                                                   *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void Keymap::ViewLogFile ( void )
{

   //* Put the NCurses Engine into hibernation mode *
   nc.Hibernate () ;

   gString gsCmd( "less -Rc %S", this->logPath.gstr() ) ;
   system ( gsCmd.ustr() ) ;

   //* Wake the NCurses Engine and refresh the display *
   nc.Wake () ;

   this->WriteParagraph ( L"\n Do you want to save the log file "
                           "to the current directory? (y/n):", this->menuAttr ) ;

   wkeyCode wk ;
   nc.GetKeyInput ( wk ) ;
   if ( wk.type == wktPRINT )
   {
      gsCmd.compose( "%C", &wk.key ) ;
      this->WriteParagraph ( gsCmd, this->menuAttr ) ;
      
      if ( (wk.key == 'y') || (wk.key == 'Y') )
      {
         this->WriteParagraph ( L" OK\n Log will be saved on exit.\n\n ", 
                                this->menuAttr ) ;
         this->logSave = true ;
      }
      else
         this->InstructionPrompt ( true ) ;
   }
   else
      this->InstructionPrompt ( true ) ;

}  //* End ViewLogFile() *

//*************************
//*      SaveLogFile      *
//*************************
//********************************************************************************
//* Save the session log file.                                                   *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void Keymap::SaveLogFile ( void )
{
   gString trgPath( "%s/%s", this->appPath, logName ),
           gsCmd( "cp --preserve=all -d \"%S\" \"%S\" 1>/dev/null 2>/dev/null",
                  this->logPath.gstr(), trgPath.gstr() ),
           gsOut ;               // text formatting

   if ( this->ofsLog.is_open() ) // close the file
      this->ofsLog.close() ;

   if ( (system ( gsCmd.ustr() )) == ZERO )
      gsOut.compose( "\n Log file saved to \"%s\".\n", logName ) ;
   else
      gsOut = L"\n Error saving log file.\n" ;
   this->WriteParagraph ( gsOut, this->menuAttr ) ;
   chrono::duration<short>aMoment( 1 ) ;
   this_thread::sleep_for( aMoment ) ;

}  //* End SaveLogFile() *

//*************************
//*    ApplicationHelp    *
//*************************
//********************************************************************************
//* Display application help.                                                    *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void Keymap::ApplicationHelp ( void )
{
   const char* const infoHelp = "../Texinfo/ncdialogapi.info" ;
   gString gsCmd ;

   //* Verify that we can access the help file *
   if ( (this->TargetExists ( infoHelp )) )
   {
      //* Put the NCurses Engine into hibernation mode *
      nc.Hibernate () ;

      //* Invoke the 'info' reader to display Keymap Help *
      gsCmd.compose( "info -f '%s' -n 'Keymap App'", infoHelp ) ;
      system ( gsCmd.ustr() ) ;

      //* Wake the NCurses Engine and refresh the display *
      nc.Wake () ;
   }
   else     // unable to locate help file
   {
      gsCmd.compose( "\n Sorry, unable to find the help file."
                     "\n File expected at: %s\n ", infoHelp ) ;
      this->WriteParagraph ( gsCmd, this->menuAttr ) ;
   }

}  //* ApplicationHelp() *

//************************
//*    GetLocalTime      *
//************************
//********************************************************************************
//* Get the system timecode and convert it to localTime format.                  *
//*                                                                              *
//* Input  : ft  : (by reference, initial values ignored)                        *
//*                on return, contains decoded date/time                         *
//*                                                                              *
//* Returns: true if successful, false if system call fails                      *
//********************************************************************************
//* Programmer's Note: Currently we do not decode the timezone and DST fields.   *
//*                                                                              *
//* -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  - -  *
//* Note that we do a bit of defensive programming in anticipation of the        *
//* dreaded year 2038 overflow of the 32-bit time_t type.                        *
//********************************************************************************

static bool GetLocalTime ( localTime& ft )
{
   bool success = false ;                       // return value
  
   ft.reset() ;                                 // set to default values
   if ( (time ( &ft.sysepoch )) != (-1) )
   {
      if ( ft.sysepoch >= ZERO )                // reported time is AFTER the epoch
      {
         ft.epoch = (int64_t)ft.sysepoch ;      // promote to 64-bit
         //* Decode the system time *
         Tm    tm ;                             // Linux time structure
         if ( (localtime_r ( &ft.sysepoch, &tm )) != NULL )
         {
            //* Translate to localTime format *
            ft.julian   = tm.tm_yday ;          // Julian date
            ft.day      = tm.tm_wday ;          // 0 == Sunday ... 6 == Saturday
            ft.date     = tm.tm_mday ;          // today's date
            ft.month    = tm.tm_mon + 1 ;       // month
            ft.year     = tm.tm_year + 1900 ;   // year
            ft.hours    = tm.tm_hour ;          // hour
            ft.minutes  = tm.tm_min ;           // minutes
            ft.seconds  = tm.tm_sec ;           // seconds
            success = true ;
         }
      }
      else
      {  /* SYSTEM ERROR - time_t OVERFLOW */ }
   }
   return success ;

}  //* End GetLocalTime() *

//*************************
//*   FormatDateString    *
//*************************
//********************************************************************************
//* Produce a formatted output string based on data in a localTime object.       *
//*   Format:  "dd Mmm yyyy hh:mm:ss"                                            *
//*                                                                              *
//* Input  : lt  : initialized localTime structure (by reference)                *
//*                Note: 'day', the day-of-the-week field is ignored.            *
//*          str : buffer to receive formatted data                              *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

static void FormatDateString ( const localTime& lt, gString& str )
{
//   //* Month-name strings for file info formatting *
//   const wchar_t wMonthString[13][4] =
//   {
//      { L"   " },
//      { L"Jan" }, { L"Feb" }, { L"Mar" }, { L"Apr" }, { L"May" }, { L"Jun" },
//      { L"Jul" }, { L"Aug" }, { L"Sep" }, { L"Oct" }, { L"Nov" }, { L"Dec" }
//   } ;

   str.compose( L"%04hu-%02hu-%02huT%02hu:%02hu:%02hu", 
                &lt.year, &lt.month, &lt.date,
                &lt.hours, &lt.minutes, &lt.seconds ) ;
//   str.compose( L"%02hu %S %04hu %02hu:%02hu:%02hu",
//                &ft.date, wMonthString[ft.month], &ft.year, 
//                &ft.hours, &ft.minutes, &ft.seconds ) ;

}  //* End FormatDateString() *

//*************************
//*    RenameAsBackup     *
//*************************
//********************************************************************************
//* If target file exists, rename it as a backup file (append tilde '~').        *
//* If the backup already exists, it will be overwritten.                        * 
//*                                                                              *
//* Input  : srcPath : full path/filename specification of source                *
//*                                                                              *
//* Returns: 'true' on success, else 'false'                                     *
//********************************************************************************

bool Keymap::RenameAsBackup ( const gString& srcPath )
{
   gString trgPath ;

   trgPath.compose( "%S~", srcPath.gstr() ) ;
   this->DeleteFile ( trgPath ) ;
   return ( (this->RenameFile ( srcPath, trgPath )) ) ;

}  //* End RenameAsBackup() *

//*************************
//*      RenameFile       *
//*************************
//********************************************************************************
//* Rename the specified file.                                                   *
//*                                                                              *
//* Input  : srcPath : full pathfilename specification of source                 *
//*          trgPath : full pathfilename specification of target                 *
//*                                                                              *
//* Returns: 'true' on success, else 'false'                                     *
//********************************************************************************

bool Keymap::RenameFile ( const gString& srcPath, const gString& trgPath )
{
   bool status = true ;

   if ( (rename ( srcPath.ustr(), trgPath.ustr() )) != ZERO )
      status = false ;
   
   return status ;

}  //* End RenameFile() *

//*************************
//*     TargetExists      *
//*************************
//********************************************************************************
//* Stat the target to see if it exists.                                         *
//*                                                                              *
//* Input  : trgPath : full path/filename specification of target                *
//*                                                                              *
//* Returns: 'true' if target exists, else 'false'                               *
//********************************************************************************

bool Keymap::TargetExists ( const gString& trgPath )
{
   FileStats fstat ;          // receives file stats
   bool status = true ;       // return value

   if ( (stat64 ( trgPath.ustr(), &fstat )) != ZERO )
      status = false ;

   return status ;

}  //* End TargetExists()

//*************************
//*      DeleteFile       *
//*************************
//********************************************************************************
//* Delete the specified file.                                                   *
//*                                                                              *
//* Input  : trgPath : full path/filename specification of target                *
//*                                                                              *
//* Returns: 'true' if target deleted OR if target does not exist                *
//*          'false' if target not deleted                                       *
//********************************************************************************

bool Keymap::DeleteFile ( const gString& trgPath )
{
   bool status = true ;       // return value

   if ( (this->TargetExists ( trgPath )) )
   {
      if ( (unlink ( trgPath.ustr() )) != ZERO )
         status = false ;
   }
   return status ;

}  //* End DeleteFile() *

//*************************
//*      GetVersions      *
//*************************
//********************************************************************************
//* Get version numbers for the following tools used for capture of keycodes,    *
//* and initialize the following member variables:                               *
//*  1) ncVersion          ncurses library version                               *
//*  2) gtVersion          gnome-terminal version                                *
//*                                                                              *
//*                                                                              *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* Programmer's Note: Technically, we should construct the command using a      *
//* buffer of at least MAX_PATH * 2 bytes to accomodate two max path lengths;    *
//* however, we "know" that both the temp directory and our development directory*
//* paths are much shorter than the maximum length.                              *
//********************************************************************************

void Keymap::GetVersions ( void )
{
   const char* const fdTemplate = "uname -r 1>\"%S\" 2>&1" ;
   const char* const gtTemplate = "gnome-terminal --version 1>\"%S\" 2>&1" ;
   const char* const knTemplate = "konsole --version 1>\"%S\" 2>&1" ;
   const char* const xtTemplate = "xterm -version 1>\"%S\" 2>&1" ;
   const char* knName = "konsole " ;
   const char* xtName = "XTerm(" ;
   const char* unk    = "(unknown)" ;
   gString tmpName, gsCmd, gs ;
   ifstream ifs ;
   short indx ;
   this->CreateTempname ( tmpName ) ;     // create a file in the temp directory

   gs = nc.Get_nclibrary_Version () ;     // get ncurses library verison
   gs.copy( ncVersion, VERBUF ) ;

   //* Query kernel-release (Fedora) version and redirect to temp file *
   gsCmd.compose( fdTemplate, tmpName.gstr() ) ;
   system ( gsCmd.ustr() ) ;
   gs = unk ;
   ifs.open( tmpName.ustr(), ifstream::in ) ;
   if ( ifs.is_open() )
   {
      this->GetLine ( ifs, gs ) ;
      ifs.close() ;
      if ( (gs.gschars()) < 7 )
         gs = unk ;
   }
   gs.copy( klVersion, VERBUF ) ;

   //* Query gnome-terminal version and redirect to temp file *
   gsCmd.compose( gtTemplate, tmpName.gstr() ) ;
   system ( gsCmd.ustr() ) ;
   gs = unk ;
   ifs.open( tmpName.ustr(), ifstream::in ) ;
   if ( ifs.is_open() )
   {
      this->GetLine ( ifs, gs ) ;
      ifs.close() ;
      if ( (indx = gs.after( L"# GNOME Terminal " )) > ZERO )
      {
         gs.shiftChars( -(indx) ) ;// isolate the version number
         indx = gs.find( L' ' ) ;
         gs.limitChars( indx ) ;
      }
      else
         gs = unk ;
   }
   gs.copy( gtVersion, VERBUF ) ;

   //* Query KDE Konsole version and redirect to temp file. *
   //* Note that konsole cannot be launched or queried from *
   //* within gnometerm without generating a warning        *
   //* message, so step over the warning-message line.      *
   gsCmd.compose( knTemplate, tmpName.gstr() ) ;
   system ( gsCmd.ustr() ) ;
   gs = unk ;
   ifs.open( tmpName.ustr(), ifstream::in ) ;
   if ( ifs.is_open() )
   {
      this->GetLine ( ifs, gs ) ;
      if ( (indx = gs.after( knName )) < ZERO )
      {
         this->GetLine ( ifs, gs ) ;
         indx = gs.after( knName ) ;
      }
      if ( (indx > ZERO) &&
           ((gs.gstr()[indx] >= L'0') && (gs.gstr()[indx] <= L'9')) )
      {
         gs.shiftChars( -(indx) ) ;       // isolate the version number
         gs.strip() ;                     // discard leading and trailing whitespace
      }
      else
         gs = unk ;
      ifs.close() ;
   }
   gs.copy( knVersion, VERBUF ) ;

   //* Query XTerm version and redirect to temp file. *
   gsCmd.compose( xtTemplate, tmpName.gstr() ) ;
   system ( gsCmd.ustr() ) ;
   gs = unk ;
   ifs.open( tmpName.ustr(), ifstream::in ) ;
   if ( ifs.is_open() )
   {
      this->GetLine ( ifs, gs ) ;
      ifs.close() ;
      if ( ((indx = gs.after( xtName )) > ZERO) &&
           ((gs.gstr()[indx] >= L'0') && (gs.gstr()[indx] <= L'9')) )
      {
         gs.shiftChars( -(indx) ) ;       // isolate the version number
         gs.erase( L")", ZERO, false, true ) ;
         gs.strip() ;                     // discard leading and trailing whitespace
      }
      else
         gs = unk ;
   }
   gs.copy( xtVersion, VERBUF ) ;

   this->DeleteFile ( tmpName ) ;         // delete the temp file

}  //* End GetVersions() *

//*************************
//*                       *
//*************************
//********************************************************************************
//*                                                                              *
//*                                                                              *
//*                                                                              *
//* Input  :                                                                     *
//*                                                                              *
//* Returns:                                                                     *
//********************************************************************************

