//******************************************************************************
//* File       : NcDialog.hpp                                                  *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2006-2015 Mahlon R. Smith, The Software Samurai *
//*                 GNU GPL copyright notice below                             *
//* Date       : 28-Jul-2015                                                   *
//* Version    : (see NcDialogVersion string in NcDialog.cpp)                  *
//*                                                                            *
//* Description: This class is derived from the NcWindow class and implements  *
//* a window class specifically for dialog windows.                            *
//* The various types of user-input controls available for the dialog are also *
//* defined here.                                                              *
//*                                                                            *
//*                                                                            *
//* Development Tools: See NcDialog.cpp.                                       *
//******************************************************************************
//* Version History:                                                           *
//*   See version history in NcDialog.cpp.                                     *
//*                                                                            *
//******************************************************************************
//* Copyright Notice:                                                          *
//* This program is free software: you can redistribute it and/or modify it    *
//* under the terms of the GNU General Public License as published by the Free *
//* Software Foundation, either version 3 of the License, or (at your option)  *
//* any later version.                                                         *
//*                                                                            *
//* This program is distributed in the hope that it will be useful, but        *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
//* for more details.                                                          *
//*                                                                            *
//* You should have received a copy of the GNU General Public License along    *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.            *
//*                                                                            *
//*         Full text of the GPL License may be found in the Texinfo           *
//*         documentation for this program under 'Copyright Notice'.           *
//******************************************************************************
//* Programmer's Notes:                                                        *
//* 1) Please note that some of the most important methods, the 'WriteString'  *
//*    group, the 'WriteChar' group, and 'GetKeyInput' group are inherited     *
//*    from the NcWindow, parent class. See NcWindow.hpp for complete          *
//*    descriptions of these and other inherited methods.                      *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//******************************************************************************

#ifndef NCDIALOG_INCLUDED
#define NCDIALOG_INCLUDED

//* Include the definition of the base class *
#include "NcWindow.hpp"


//***************
//* Definitions *
//***************

//* This flag controls compilation of certain member and non-member methods    *
//* that are useful only during application development.                       *
//* Setting this flag to '0' saves approximately 90 KBytes of codespace, and   *
//* thus is hardly worth the effort and inconvenience of disabling the code.   *
#define ENABLE_DEVELOPMENT_METHODS (1)

const short MAX_DIALOG_CONTROLS = 24 ; //* Maximum number of interactive control objects in dialog
const short MAX_DIALOG_WIDTH    = 132 ;//* Maximum display columns in a dialog window
const short MAX_LABEL_CHARS     = 128 ;//* Maximum characters in a control label string
const short DDBOX_MIN_LINES     = 3 ;  //* Minimum size of a dctDROPDOWN control
const short MAX_SPINNER_WIDTH   = 16 ; //* Maximum width of a spinner control (incl. indicator)
const short MAX_RADIO_WIDTH     = 8 ;  //* Maximum characters in a radio button display string
const short MIN_TB_SHIFT_WIDTH  = 5 ;  //* For Text Box controls, the minimum control 
                                       //* width for edit of text that is longer than
                                       //* the display field. See ExtendedTextboxData().
const attr_t attrDFLT = (-1) ;         //* Dummy (placeholder) color attribute indicating 
                                       //* that a default value should be used


//* Types of controls supported within an NcDialog class object *
enum DCType : short 
   { dctPUSHBUTTON, dctTEXTBOX, dctBILLBOARD, dctRADIOBUTTON, dctSCROLLBOX, 
     dctDROPDOWN, dctMENUWIN, dctSCROLLEXT, dctSPINNER, 
     dctTYPES
   } ;

//* Radio Button subtypes. *
//* To display samples of each radio-button type, open a dialog window and call*
//* DisplayRadiobuttonTypes() method or see Dialog1 test application, Test #5. *
enum RBType : short
{  //* Standard types use a Diamond as the selector character *
   rbtS1 = 0,  // single-column with no brackets
   rbtS3a,     // three-column with angle brackets
   rbtS3s,     // three-column with square brackets
   rbtS3p,     // three-column with parentheses brackets
   rbtS5a,     // five-column with angle brackets
   rbtS5s,     // five-column with square brackets
   rbtS5p,     // five-column with parentheses brackets
   rbtC1,      // single-column custom
   rbtC2,      // two-column custom (for 2-column selector character)
   rbtC3,      // three-column custom (one-column selector character)
   rbtC4,      // four-column custom (for 2-column selector character)
   rbtC5,      // five-column custom (2-1-2 column, or 1-1-1-1-1 column)
   rbtC6,      // six-column custom (3, 2-column characters)
   rbtTYPES
} ;

//* For dctDROPDOWN controls, when the control expands for user selection, *
//* the following types of expansion are available: DOWN from the anchor   *
//* point (default), UP from the anchor point, or CENTERed vertically on   *
//* the anchor point. See SetDropdownExpansion() method.                   *
enum ddBoxExpansionType : short { ddBoxDOWN, ddBoxUP, ddBoxCENTER, ddBoxTYPES } ;

//* Options for valid input in text boxes.                                 *
//* NOTE: These filters are based on UTF-8 encoded characters. Unless      *
//*       specified as ASCII-only, filters include the entire UTF-8 set.   *
enum tbChars : short
{
   tbPrint,          //* All printing characters
   tbPrintUpper,     //* All printing chars AND force lowercase to uppercase
   tbPrintLower,     //* All printing chars AND force uppercase to lowercase
   tbAsciiPrint,     //* All 7-bit printing characters from SPACE (0x20) through TILDE (0x7E)
   tbAlNum,          //* Alphanumeric characters (all alphabets && numeric systems) incl. PERIOD 
   tbAlNumSp,        //* Alphanumeric characters (all alpha && num) incl. PERIOD and SPACE
   tbAlpha,          //* Upper and lower case alphabetic characters (all alphabets)
   tbAlphaSp,        //* Upper and lower case alpha (all alphabets) incl. SPACE
   tbNumeric,        //* Numbers '0' through '9' plus PERIOD, PLUS, MINUS (ASCII)
   tbNumber,         //* Numbers only '0' through '9' (ASCII)
   tbHexNum,         //* Numbers '0' through '9' and 'A' through 'F' ('a' - 'f') (ASCII)
   tbHexNumUp,       //* Same as tbHexNum except force lower case to upper case
   tbLower,          //* Lower case alpha (forces upper to lower)
   tbLowerSp,        //* Lower case alpha incl. SPACE (forces upper to lower)
   tbUpper,          //* Upper case alpha (forces lower to upper)
   tbUpperSp,        //* Upper case alpha incl. SPACE (forces lower to upper)
   tbURL,            //* Characters valid in a web address
   tbFileName,       //* Characters valid in filenames and directory names (not incl. '/')
   tbPathName        //* Same as tbFileName but includes '/'
} ;

//* Display format code for DialogSpinner control. See definition of           *
//* dspinData class below.                                                     *
enum dspinFormat : short { dspinINTEGER, dspinDEC1, dspinDEC2, dspinDEC3 } ;
 
//* For initialization of dctSPINNER class controls during instantiation of an *
//* NcDialog object, the 'spinData' member of the InitCtrl structure must      *
//* point to an initialized instance of a dspinData class object.              *
//* Notes on interpretation of display data values:                            *
//*  1) Both positive and negative values are supported, BUT when declaring    *
//*     the width of the control, be sure to take into account the space       *
//*     required for display of the minus sign. If the display field is too    *
//*     narrow for the value, the field will be filled with the '#' character. *
//*  2) The minValue, maxValue, and iniValue integers will be interpreted      *
//*     diferently, depending upon the formatting code provided in the         *
//*     dsFormat field:                                                        *
//*       dsFormat==dspinINTEGER, values are displayed as integers             *
//*       dsFormat==dspinDEC1,    values are interpreted as decimal values     *
//*                               where value == value/10, and are             *
//*                               displayed with one (1) decimal place.        *
//*                                ex: iniValue of 15 displayed as 1.5         *
//*       dsFormat==dspinDEC2,    values are interpreted as decimal values     *
//*                               where value == value/100, and are            *
//*                               displayed with two (2) decimal places.       *
//*                                ex: iniValue of 156 displayed as 1.56       *
//*       dsFormat==dspinDEC3,    values are interpreted as decimal values     *
//*                               where value == value/1000, and are           *
//*                               displayed with three (3) decimal places.     *
//*                                ex: iniValue of 1567 displayed as 1.567     *
//* These interpretations also hold for the NcDialog methods, GetSpinnerValue()*
//* SetSpinnerValue() and SetSpinnerLimits. Thus, even if the value is         *
//* displayed as a decimal value, it is passed, retrieved, and stored as an    *
//* integer. So if you use the data value in your application, do the proper   *
//* conversion:                                                                *
//* For example: if a dctSPINNER control is instantiated with a format of      *
//* dspinDEC3, then when you retrieve the display value for use in your        *
//* application, you would do the following:                                   *
//*   int iValue ;                                                             *
//*   dlgPtr->GetSpinnerValue ( index, iValue );                               *
//*   double spValue = (double)iValue /= 1000.0;                               *
//* spValue will then be equivalent to what is displayed by the control object.*
class dspinData
{
   public:
   int    minValue ;       //*  minimum display value
   int    maxValue ;       //*  maximum display value
   int    iniValue ;       //*  initial display value
   attr_t indiAttr ;       //*  color attribute for up/down indicator
   dspinFormat dsFormat ;  //*  format code (member of enum dspinFormat)
} ;

//* Initialization data for a dialog control object.                           *
//* A singly-linked list of InitCtrl objects is passed to the NcDialog         *
//* constructor, one object for each control defined within the dialog         *
//* (see the InitNcDialog class definition below)                              *
//* Programmer's Note: The NcDialog instantiation process will modify the data *
//* in this object ONLY if an invalid value is found. This is important because*
//* the application will need to reference these data during the dialog's      *
//* user input loop.                                                           *
class InitCtrl
{
   public:
   //* Control type from enum DCType
   DCType      type ;

   //* For radio buttons, button sub-type. For all other control types: unused
   RBType      rbSubtype ;

   //* For radio buttons, initial state of 'selected'. For all other types: unused
   bool        rbSelect ;

   //* Position of control in Y relative to upper-left corner of dialog window
   short       ulY ;

   //* Position of control in X relative to upper-left corner of dialog window
   short       ulX ;

   //* Display lines required by control: dctPUSHBUTTON, dctTEXTBOX, 
   //* dctBILLBOARD, dctSCROLLBOX, dctSCROLLEXT, dctDROPDOWN.
   //* Note: Minimum value varies according to control type.
   //*       Maximum value varies according to control type.
   //*        For dctTEXTBOX controls, maximum is calculated based on the total 
   //*        number of characters that can be stored in the control's buffer.
   //*       For dctRADIOBUTTON, dctSPINNER: always == 1
   //*       For dctMENUWIN: unused
   short       lines ;

   //* Display columns required by control.
   //* Note: Minimum value varies according to control type.
   //*       Maximum value is MAX_DIALOG_WIDTH.
   //* For decMENUWIN controls, this is the width in the expanded state.
   //* For dctRADIOBUTTON controls: unused (rbSubtype determines width of control)
   short       cols ;

   //* Text data displayed within control (if any)
   //* Note that the dispText member is a multi-purpose pointer.
   //* For dctPUSHBUTTON, dctTEXTBOX, 'custom' dctRADIOBUTTON and
   //* optionally for dctBILLBOARD controls:
   //*   points to a UTF-8 character string (or is a NULL pointer)
   //* For dctSCROLLBOX, dctDROPDOWN, dctMENUWIN controls, and
   //* optionally for dctSCROLLEXT controls:
   //*   points to a two-dimensional UTF-8 character array,
   //*   (OR to an array of const char pointers),
   //*   one string per display item (requires a cast to const char*),
   //* For dctSPINNER, and standard dctRADIOBUTTON: unused
   const char* dispText ;

   //* For controls with borders, border color while control does not have 
   //* input focus: dctSCROLLBOX, dctSCROLLEXT, dctDROPDOWN,
   //* For borderless controls, display text color while control does not have
   //* input focus: dctPUSHBUTTON, dctTEXTBOX, dctBILLBOARD, dctRADIOBUTTON, 
   //* dctSPINNER, and dctMENUWIN (when in collapsed state)
   attr_t      nColor ;

   //* For controls with borders, border color while control has input focus: 
   //*   dctSCROLLBOX, dctSCROLLEXT, dctDROPDOWN,
   //* For borderless controls, display text color while control has input focus: 
   //*   dctPUSHBUTTON, dctTEXTBOX, dctBILLBOARD, dctRADIOBUTTON, dctSPINNER,
   //*   and dctMENUWIN (when in collapsed state)
   attr_t      fColor ;

   //* Text input filter for dctTEXTBOX controls, all other types: unused
   tbChars     filter ;

   //* Control's label string. Set to NULL pointer for no label.
   //* For dctMENUWIN, this is text displayed while in collapsed state, AND the
   //*   width of the text determines the width of control in collapsed state
   //* For dctPUSHBUTTON controls: unused (pushbuttons have no label)
   //* Note that except for controls which define their labels in the 'Title'
   //* position (labY==ZERO && labX==ZERO), multi-line labels are supported.
   const char* label ;

   //* For controls with a label specified: Y/X offset for label display  
   //* relative to ulY/ulX specified for the control i.e. ulY+labY / ulX+labX
   short       labY ;
   short       labX ;

   //* For Dropdown controls: direction of expansion. For all other types: unused
   ddBoxExpansionType exType ;

   //* Number of items in the display list: 
   //* dctSCROLLBOX, dctDROPDOWN, and dctMENUWIN (in expanded state).
   //* Optional for dctSCROLLEXT controls.
   //* For all other types: unused.
   short       scrItems ;
 
   //* Index of initially-highlighted item in display list:
   //* dctSCROLLBOX, dctDROPDOWN, and dctMENUWIN (in expanded state).
   //* Optional for dctSCROLLEXT controls.
   //* For all other types: unused.
   short       scrSel ;
 
   //* For controls with a list of display items, scrColor points to an array 
   //* of color attributes for the display list: dctSCROLLBOX, dctDROPDOWN, 
   //* and dctMENUWIN (in expanded state). 
   //* Optional for dctSCROLLEXT and dctBILLBOARD controls.
   //* Notes:
   //* For an array of display items of all the same color, scrColor has two (2) 
   //* elements: scrColor[0] == attrDFLT  AND  scrColor[1] == color attribute 
   //* for all items. (does not apply to dctSCROLLEXT controls)
   //* For an array of display items, each item having its own color, 
   //* scrColor[0] == color attribute for item 0, scrColor[1] == color attribute 
   //* for item 1, etc.
   //* For all other types: unused.
   attr_t*     scrColor ;

   //* For dctSPINNER controls only: pointer to a dspinData class object
   //* for control initialization.
   const dspinData*  spinData ;

   //* Control-is-active flag: If 'true' control can receive input focus;
   //* if 'false' control is view-only. NOTE: First control in list must be 
   //* marked as 'active'.to ensure that at least one control in the dialog 
   //* window can receive the input focus.
   bool        active ;

   //* Pointer to next instance in the singly-linked list.
   //* If end of list, set to NULL pointer.
   InitCtrl* nextCtrl ;
} ;

//* Initialization data passed to the NcDialog constructor. *
class InitNcDialog
{
   public:
   //* Default constructor *
   InitNcDialog( void )
   {
      dLines = dColumns = dYoffset = dXoffset = ZERO ;
      dTitle = NULL ;
      borderStyle = ncltSINGLE ;
      borderColor = interiorColor = ZERO ;
      ctrlPtr = NULL ;
   }
   //* Full-initialization constructor *
   InitNcDialog( short l, short c, short y, short x, 
                 const char* t, ncLineType s, attr_t b, attr_t i, InitCtrl* p ) :
      dLines(l), dColumns(c), dYoffset(y), dXoffset(x), dTitle(t), 
      borderStyle(s), borderColor(b), interiorColor(i), ctrlPtr(p) { }
   short    dLines ;       //* Display lines for dialog window
   short    dColumns ;     //* Display columns for dialog window
   short    dYoffset ;     //* Offset in Y from upper-left corner of terminal window
   short    dXoffset ;     //* Offset in X from upper-left corner of terminal window
   const char* dTitle ;    //* Title to be displayed in first dialog line (NULL if none)
   ncLineType borderStyle ;//* Dialog window's border style: 
                           //* (ncltSINGLE(default), ncltSINGLEBOLD, ncltDUAL, etc.)
   attr_t   borderColor ;  //* Dialog window's border color attribute
   attr_t   interiorColor ;//* Dialog window's interior color attribute
   InitCtrl* ctrlPtr ;     //* Pointer to head of control initializer list
} ;

//* Cursor positions for call to NcDialog::SetTextboxCursor().     *
//* For further information, see note in method declaration below. *
enum tbCurPos : short
{
   tbcpHOME = -5, //* Cursor positioned at beginning of text data i.e. offset ZERO
   tbcpAPPEND,    //* Cursor positioned at 'append' position (first space after text)
   tbcpRIGHTJUST, //* Cursor is fixed at the right edge of text box and input grows
                  //* toward the left.
} ;

//* Type definition for callback function for external update of dialog        *
//* controls. If mainline code needs access to one or more controls in the     *
//* dialog on a more-or-less real-time basis, this is the way to get that      *
//* access. If a callback method is established for an open dialog, then the   *
//* edit methods for each control will call the callback method every time     *
//* key input from the user is processed.                                      *
//* See prototype for EstablishCallback() method below for more details.       *
typedef short (*CUPTR)( const short currentIndex, const wkeyCode wkey, bool firstTime ) ;


//* Dialog save-window class definition. When a dialog window is partially or 
//* completely obscured by another object, it is often necessary to save the 
//* display data for that window. Saving the display data for a window is not 
//* always necessary. If the only data in the window are controls, then the 
//* dialog can simply be refreshed via a call to RefreshWin(). If, however, the 
//* application has written data directly into the dialog (comments, warnings, 
//* etc that the NcDialog class cannot track, then it is necessary to call the 
//* SetDialogObscured() method to explicitly save all the display data for the 
//* dialog. Then, a call to RefreshWin() will completely restore the visible 
//* window. 
//* Note that when an obscured dialog is refreshed, the saved display data are 
//* discarded, on the assumption that the information in the controls will 
//* change. Thus, the SetDialogObscured() method must be called each time the 
//* dialog is about to be obscured.
//   Programmer's Note: In a future version of this class, we may be able to 
//    automagically track when a dialog is being obscured, and save its 
//    display data without application intervention, but that is a major 
//    undertaking, both technically and in utilization of memory.
class dSaveWin
{
   public:
   //* Constructor *
   dSaveWin ( short rows, short cols, short yOffset = ZERO, short xOffset = ZERO )
   {
      this->swPtr = new SaveWin ( rows, cols, yOffset, xOffset ) ;
      this->nextNode = this->prevNode = NULL ;
   }
   //* Destructor *
   ~dSaveWin ( void )
   {
      if ( this->swPtr != NULL ) delete this->swPtr ;
   }
   protected:
   SaveWin*  swPtr ;       // pointer to saved text and color-attribute data
   dSaveWin* nextNode ;    // next node in linked list
   dSaveWin* prevNode ;    // previous node in linked list
   friend class NcDialog ; // Allow NcDialog class access to all members
} ;

//* User-Interface Information                             *
//* Data passed among the edit routines for the various    *
//* control types and back to the application's input loop.*
//* (see notes for individual edit routines below)         *
class uiInfo
{
   public:
   uiInfo()
   {
      ctrlType    = dctTYPES ;
      ctrlIndex   = MAX_DIALOG_CONTROLS ;
      hasFocus    = false ;
      dataMod     = false ;
      keyIn       = ZERO ;
      wk.key      = ZERO ;
      wk.type     = wktERR ;
      selMember   = MAX_DIALOG_CONTROLS ;
      isSel       = false ;
      viaHotkey   = false ; 
   }
   //* Move the hotkey data and invalidate the h_xx fields *
   void HotData2Primary ( void )
   {
      ctrlType    = h_ctrlType ;
      ctrlIndex   = h_ctrlIndex ;
      hasFocus    = h_hasFocus ;
      dataMod     = h_dataMod ;
      keyIn       = h_keyIn ;
      wk          = h_wk ;
      selMember   = h_selMember ;
      isSel       = h_isSel ;
      viaHotkey = false ;
   }

   short ctrlType ;     // type of control edited (enum DCType)
   short ctrlIndex ;    // index of edited control in control list
   bool  hasFocus ;     // true if control has input focus on return from edit
   bool  dataMod ;      // true if control's data modified during edit
   wchar_t keyIn ;      // last key input received (or ZERO if n/a)
   wkeyCode wk ;        // actual last key pressed (generally unused)
   short selMember ;    // for controls that have more than one member:
                        //  i.e. Scrollbox, Scrollext, Radiobutton Group, 
                        //  Dropdown, and Menuwin, this is the
                        //  currently-selected member of the group
   bool  isSel ;        // for controls that have binary data i.e. Pushbuttons
                        //  and independent Radio Buttons, current state:
                        //     Radio Button: true if 'selected'
                        //     Pushbutton  : true if 'pressed'

   bool  viaHotkey ;    // true if a valid hotkey was detected during edit, 
                        // causing the control being edited to lose focus

         // NOTE: The following fields contain valid data only if the viaHotkey 
         //       flag != false on return from edit routine
   short h_ctrlType ;   // type of control accessed via hotkey (enum dcTYPE)
   short h_ctrlIndex ;  // index of control that received focus via hotkey
   bool  h_hasFocus ;   // true if the control has input focus
   bool  h_dataMod ;    // true if control's data modified
   wchar_t h_keyIn ;    // last key input received (or ZERO if n/a)
   wkeyCode h_wk ;      // actual last key pressed (generally unused)
   short h_selMember ;  // for controls that have more than one member:
                        //  i.e. Scroll Box, Radio Button Group, and Drop-Down,
                        //  this is the currently-selected member of the group
   bool  h_isSel ;      // for controls that have binary data i.e. Pushbuttons
                        //  and independent Radio Buttons, current state:
                        //     Radio Button: true if 'selected'
                        //     Pushbutton  : true if 'pressed'

   friend class NcDialog ;    //* Allow NcDialog class access to all data
                              //* Because all data are public, everyone has access
} ;

//* Used internally by the NcDialog class to manage hotkey access to controls. *
//* For accessing controls via 'hot key', embed a caret ( '^' ) into the       *
//* control's label string just before the character to be used as a hotkey.   *
//* IMPORTANT NOTE:                                                            *
//*   Valid hotkeys are 'A' <= hotkey <= 'Z'  OR 'a' <= hotkey <= 'z'          *
//*   This restriction is necessary to allow a single, control key sequence to *
//*   invoke the hotkey functionality: Ctrl+A, Ctrl+b, etc.                    *
//* This character will display as underlined, and will be the hotkey for the  *
//* specified control. The character will be displayed in the upper/lower case *
//* provided but the user input for hotkeys is case-insensitive: For example:  *
//* 'A' (0x41), 'a' (0x61), and CTRL+A (0x01) are processed as the same key.   *
//* IMPORTANT NOTE:                                                            *
//*   User selection of a control via hotkey may be by any of the three (3)    *
//*   keycode: uppercase alpha, lowercase alpha or Ctrl+alphakey. However,     *
//*   try to avoid using hotkeys whose Ctrl+x versions have established        *
//*   meanings because those Ctrl+key combinations will not be interpreted as  *
//*   valid hotkeys. For instance, under Linux, CTRL+M and CTRL+J are both     *
//*   interpreted as the ENTER key, and CTRL+I is interpreted as the Tab key,  *
//*   so using 'M' 'm' 'J' 'j' 'I' 'i' as hotkeys could cause problems due to  *
//*   lack of symmetry with the other 3-keycode groups.                        *
//* This functionality is available for the label strings of all control types.*
class dcHotkey
{
   public:
   dcHotkey ( void )
   { this->reset() ; }
   void reset ( void )
   {
      this->hotchar = NULLCHAR ;
      this->yoffset = this->xoffset = ZERO ;
      this->hotkey = this->hotdflt = false ;
   }
   wchar_t  hotchar ;      //* Hotkey character 
   short    yoffset ;      //* Y offset into display string for hotkey display
   short    xoffset ;      //* X offset into display string for hotkey display
   bool     hotkey ;       //* 'true' if hotkey specified
   bool     hotdflt ;      //* 'true' if default hotkey assigned
} ;

//* Structure for passing data to the SetScrollextText() method *
class ssetData
{
   public:
   //* Default constructor *
   ssetData ( const char** dt = NULL, const attr_t* dc = NULL, int di = ZERO,
              int hi = ZERO, bool hs = true ) :
               dispText(dt), dispColor(dc), dispItems(di),
               hlIndex(hi), hlShow(hs) {}
   const char** dispText ;  //* Pointer to an array of display string pointers
   const attr_t* dispColor ;//* Pointer to an array of color attributes 
                            //*  (one for each display string)
   int      dispItems ;     //* Number of display items in above arrays
   int      hlIndex ;       //* Index of display item initially highlighted
   bool     hlShow ;        //* Show/hide the highlight for current item
} ;

//* For passing connect-point data to the ConnectControl2Border() method.      *
//* Connecion types for joining control borders to the dialog window border    *
class cdConnect
{
   public:
   cdConnect()
   {
      ul2Left     = false ;
      ul2Top      = false ;
      ll2Left     = false ;
      ll2Bot      = false ;
      ur2Right    = false ;
      ur2Top      = false ;
      lr2Right    = false ;
      lr2Bot      = false ;
      connection  = false ;
   }
   bool  ul2Left ;
   bool  ul2Top ;
   bool  ll2Left ;
   bool  ll2Bot ;
   bool  ur2Right ;
   bool  ur2Top ;
   bool  lr2Right ;
   bool  lr2Bot ;
   bool  connection ;
} ;

//* The dtbmData class is used for passing text and color parameters to the    *
//* DisplayTextboxMessage() method.                                            *
//* (see additional explanation at declaration of method prototype below)      *
// Programmer's Note: method implementations for this class are in the         *
// NcdControlTB.cpp source module.                                             *
enum dtbmColors : attr_t 
{
  dtbmNFcolor = (attrDFLT - 1), // use non-focus color defined for the Text Box
  dtbmFcolor  = (attrDFLT - 2)  // use focus color defined for the Text Box
} ;
class dtbmData
{
   public:
   //* This constructor initializes all data members to default values:        *
   //*  - text data is initialized to a null string                            *
   //*  - text color is control's non-focus color (dtbmNFcolor)                *
   //*  - 'refreshControl' == 'true' and 'centered' == ZERO                    *
   dtbmData ( void ) ;

   //* This constructor initializes text data (other fields to default values).*
   //* Input  : tData : pointer to display text (UTF-8 OR wchar_t)             *
   //*                  max length==gsMAXBYTES (UTF-8) or gsMAXCHARS (wchar_t) *
   //*          NOTE: text color is control's non-focus color (dtbmNFcolor)    *
   //*                'refreshControl' == 'true' and 'centered' == ZERO        *
   dtbmData ( const char* tData ) ;
   dtbmData ( const wchar_t* tData ) ;

   //* This constructor initializes both text data and color attribute data.   *
   //* Input  : tData : pointer to display text (UTF-8 OR wchar_t)             *
   //*          cData : array of color attributes, one for each character      *
   //*                  OR cData[0]==dtbmNFcolor || cData[0]==dtbmFcolor       *
   //*         refresh: (optional, 'true' by default)                          *
   //*                  refresh display i.e. make text visible immediately     *
   //*         center : (optional, 'false' by default)                         *
   //*                  If specified, the message will be centered in the      *
   //*                  target field, completely filling the field.            *
   //*         rtl    : (optional, 'false' by default)                         *
   //*                  if 'true', then process text as an RTL language        *
   //*                             (Arabic, Hebrew, Persian, Urdu, etc.)       *
   //*                                                                         *
   //* IMPORTANT NOTE: If cData[0] != dtbmNFcolor && cData[0] != dtbmFcolor,   *
   //* then your color data buffer must have at least as many elements as the  *
   //* number of characters (not byte count) in tData. ALSO, if the 'center'   *
   //* parameter is specified, the length of the tData string is dynamically   *
   //* adjusted to fill the target field, so the number of color attributes in *
   //* the 'cData' array must be >= 'center'. Otherwise, a system memory access*
   //* violation, an application crash and the ridicule of your peers will be  *
   //* the likely results. To be safe, if you are not using one of the defined *
   //* default color attributes, always use a color buffer of gsMAXCHARS.      *
   //*                         BEE YEE CAREFUL!                                *
   //*                                                                         *
   //* Returns: implicitly returns pointer to object                           *
   dtbmData ( const char* tData, const attr_t cData[gsMAXCHARS], 
              bool refresh = true, short center = ZERO, bool rtl = false ) ;
   dtbmData ( const wchar_t* tData, const attr_t cData[gsMAXCHARS], 
              bool refresh = true, short center = ZERO, bool rtl = false ) ;

   //* Assign a new text string. (UTF-8 OR wchar_t)                            *
   //* (color attributes, 'refresh', 'centered' and 'rtl' flags are unchanged) *
   void operator = ( const char *tData ) ;
   void operator = ( const wchar_t *tData ) ;

   //* Returns a pointer to the const text data (wchar_t 'wide' format)        *
   const wchar_t* GetTextPtr ( void ) { return this->textData ; }

   //* Returns a pointer to the const color attribute data                     *
   const attr_t* GetColorPtr ( void ) { return this->colorData ; }

   //* Data members *
   wchar_t textData[gsMAXCHARS] ;   // wchar_t (wide) text data
   attr_t colorData[gsMAXCHARS] ;   // color attribute data
   bool   refreshControl ;    // if 'true' (default) refresh display after update
   bool   centered ;          // if 'true' adjust message to be centered in textbox
   bool   rtlText ;           // if 'true' process data as RTL (right-to-left)

   private:
   void init ( const wchar_t* tData, const attr_t* cData, 
               bool refresh, short center, bool rtl ) ;
} ;

//* Use this class to generate a simple message dialog.                        *
//* Used as input data for a call to InfoDialog() and DecisionDialog() methods.*
const short MIN_genDialog_LINES = 4 ;     // minimum number of dialog lines
const short MIN_genDialog_COLS  = 15 ;    // minimum number of dialog columns
class genDialog
{
   public:
   //* Full-initialization constructor                       *
   //* (Note: constructor method is located in NcDialog.cpp) *
   genDialog( const char** msgList, attr_t dColor, short dLines, short dCols, 
              short yoffset=(-1), short xoffset=(-1), const attr_t* msgAttr=NULL,
              bool rtl=false, attr_t pnColor=attrDFLT, attr_t pfColr=attrDFLT, 
              const char* yes_text=NULL, const char* no_text=NULL ) ;

   //* Default constructor *
   genDialog( void )
   {  // default message strings
      static const char* dfltList[] = { "unitialized", NULL } ;

      this->msgList  = dfltList ;            // pointer to list of message strings
      this->dColor   = ZERO ;                // base color attribute for dialog
      this->pnColor  = attrDFLT ;            // pushbutton non-focus color
      this->pfColor  = attrDFLT ;            // pushbutton focus color
      this->dLines   = MIN_genDialog_LINES ; // dialog lines
      this->dCols    = MIN_genDialog_COLS ;  // dialog columns
      this->dUL      = { -1, -1 } ;          // (optional) Y/X offset
      this->msgAttr  = NULL ;                // (optional) pointer to attribute array
      this->rtl      = false ;               // (optional) direction of text flow
   }

   protected:
   //* Array of pointers to messages to be displayed on each line of the dialog*
   //*  The first message in the list is interpreted as the dialog title.      *
   //*  A message consisting of a single space, " " denotes an empty line      *
   //*  A NULL pointer terminates the list                                     *
   const char** msgList ;

   attr_t   dColor ;       // base color attribute for the dialog
   attr_t   pnColor ;      // color attribute for pushbutton without focus
   attr_t   pfColor ;      // color attribute for pushbutton with focus

   short    dLines ;       // number of display lines in the dialog
                           // (<= number of lines in parent dialog)
   short    dCols ;        // number of display columns in the dialog
                           // (<= number of columns in parent dialog)
   winPos   dUL ;          // Y/X offset from upper left corner of parent dialog
                           // (optional, default = centered in parent dialog
   bool     rtl ;          // Optional, default = 'false' i.e. LTR language)

   //* List of color attributes (optional, NULL by default)                    *
   //* If specified, this is a pointer to an array of color attributes, one    *
   //* for each message line in msgList.                                       *
   //* If NULL pointer, then all messages are written in the dColor attribute. *
   const attr_t* msgAttr ;

   friend class NcDialog ; // Allow NcDialog class access to all members
} ;


class NcDialog ;        //* Dummy declaration

//* Basic dialog control. This is an abstract class and cannot be instantiated.*
class DialogControl
{
   protected:
   //* Virtual method makes this an abstract class *
   virtual short OpenControl ( bool focus=false ) = 0 ; //* Make the control visible
   virtual void  RedrawControl ( bool hasFocus ) = 0 ;  //* Draw control with/without input focus
   virtual short SetOutputFormat ( bool rtlFormat ) = 0;//* Set data format as RTL OR LTR output
   void     InitHotkey ( const char* src, bool stripNL = true ) ; //* Hotkey initialazation
   void     DrawLabel ( NcDialog* dlgPtr, attr_t tColor, bool rtlText ) ; //* Draw control's label
   void     RefreshControl ( void ) ;  //* Refresh (update display) for the control
   NcWindow*   wPtr ;                  //* Pointer to control's underlying window
   DCType      type ;                  //* Control type from enum DCType
   attr_t      nColor ;                //* Text color if control does not have focus
   attr_t      fColor ;                //* Text color if control has focus
   short       ulY ;                   //* Upper left corner in Y for control display
   short       ulX ;                   //* Upper left corner in X for control display
   short       lines ;                 //* Display lines
   short       cols ;                  //* Display columns
   wchar_t     wLabel[MAX_LABEL_CHARS] ; //* Control label
   short       labY ;                   //* Offset from control's ulY for control label
   short       labX ;                   //* Offset from control's ulX for control label
   char        groupCode ;              //* If control is member of a group (0==none)
   dcHotkey    bHotkey ;                //* Data for optional hotkey (see definition above)
   bool        rtlContent ;             //* If true, labels and contents RTL, else LTR
   bool        active ;                 //* Active: if true, control can receive focus
   friend class NcDialog ;              //* Allow NcDialog class access to all members
} ;

//* Pushbutton control (all methods and data are private) *
class DialogPushbutton : protected DialogControl
{
   protected:
   DialogPushbutton ( InitCtrl* iPtr ) ;     //* Constructor
   virtual ~DialogPushbutton ( void ) ;      //* Destructor
   short    OpenControl ( bool hasFocus = false ) ;  //* Make control visible
   void     RedrawControl ( bool hasFocus ) ;//* Redraw and refresh
   void     RedrawControl ( bool hasFocus, bool refresh ) ;
   short    SetOutputFormat ( bool rtlFormat ) ; //* Set data format as RTL OR LTR output
   ncLineType bStyle ;                       //* Border style (default == none)
   friend class NcDialog ;                   //* Allow NcDialog class access to all data
} ;

//* Within a DialogTextbox control, tracks text selection *
//* for Copy/Cut/Paste operations.                        *
class TbSelect
{
   public:
   TbSelect ( void ) { this->reset(); }   // Constructor
   ~TbSelect ( void ) { }                 // Destructor
   void reset ( void )
   {
      this->active = this->fwd = false ;
      this->base.ypos = this->base.xpos = this->count = ZERO ;
      short i = ZERO ;
      while ( i < gsMAXCHARS )
         this->sel[i++] = false ;
   }
   void set ( short cnt )
   {
      this->active = this->fwd = true ;
      this->base.ypos = this->base.xpos = ZERO ;
      this->count = cnt ;
      short i = ZERO ;
      while ( i < this->count )
         this->sel[i++] = true ;
   }
   winPos base ;              // Y/X offset of first character selected
   short  count ;             // number of selected characters
   bool   fwd ;               // direction of selection ('true'==toward tail)
   bool   sel[gsMAXCHARS] ;   // element 'true' if corresponding char selected
   bool   active ;            // 'true' if a selection sequence in progress
} ;

//* Used to hold a list of reserved keycodes, optionally specified at the      *
//* application level. If specified, these keycodes give the user access to    *
//* the Textbox Local Clipboard while inside the EditTextbox() method.         *
//* The user may then 'select' text and Copy/Cut/Paste text in the Textbox     *
//* control which is currently under edit. With application support, the user  *
//* optionally also has access to the X clipboard.                             *
//* Please see SetTextboxReservedKeys() for details.                           *
class reservedKeys
{
   public:
   reservedKeys ( void ) { this->reset () ; }   // constructor
   void reset ( void )   { this->reserve = false ; } // release reserved keys
   bool isReservedKey ( wkeyCode wk )  // scan the list for a match
   {
      bool matched = false ;
      if ( this->reserve )    // if reserve keys defined
      {
         if      ( wk == this->selRightKey ) matched = true ;
         else if ( wk == this->selLeftKey )  matched = true ;
         else if ( wk == this->selAllKey )   matched = true ;
         else if ( wk == this->copyKey )     matched = true ;
         else if ( wk == this->pasteKey )    matched = true ;
         else if ( wk == this->cutKey )      matched = true ;
      }
      return matched ;
   }
   //* Public data members *
   wkeyCode copyKey ;      // key for copy-to-local-clipboard
   wkeyCode cutKey ;       // key for cut-to-local-clipboard
   wkeyCode pasteKey ;     // key for paste-from-local-clipboard
   wkeyCode selAllKey ;    // key for select-all-text in Textbox under edit
   wkeyCode selLeftKey ;   // key for select-from-cursor lefward
                           // fixed as: nckSLEFT (SHIFT + LeftArrow)
   wkeyCode selRightKey ;  // key for select-from-cursor rightward
                           // fixed as: nckSRIGHT (SHIFT + RightArrow)
   bool reserve ;          // 'true' to reserve list of keycodes
                           // 'false' to release all reserved keycodes
} ;

//* Data definition for dctBILLBOARD controls and multi-line dctTEXTBOX controls*
const short mtLENGTH = MAX_DIALOG_WIDTH + 1 ;
class multiText
{
   public:
   wchar_t  ln[mtLENGTH+8] ;  // text data

   private:
   void reset ( void )
   { *this->ln = NULLCHAR ; this->lnChars = this->lnCols = ZERO ; }
   short    lnChars ;      // character count (used only for formatting output)
   short    lnCols ;       // column count    (used only for formatting output)
   friend class DialogTextbox ;
   friend class DialogBillboard ;
} ;

//* Textbox control (all methods and data are private) *
class DialogTextbox : protected DialogControl
{
   protected:
   DialogTextbox ( InitCtrl* iPtr ) ;        //* Constructor
   virtual ~DialogTextbox ( void ) ;         //* Destructor
   short    OpenControl ( bool hasFocus = false ) ;  //* Make control visible
   void     RedrawControl ( bool hasFocus ) ; //* Redraw and refresh
   void     RedrawControl ( bool hasFocus, bool refresh ) ;
   short    SetOutputFormat ( bool rtlFormat ) ; //* Set data format as RTL OR LTR output

   //* Group of methods to support single-line textbox edit.*
   bool     slInsertChar  ( const wkeyCode& wk ) ;
   bool     slReplaceChar ( const wkeyCode& wk ) ;
   bool     slDeleteChar  ( const wkeyCode& wk ) ;
   void     slScrollRight ( void ) ;
   void     slScrollLeft  ( void ) ;
   void     slScrollHome  ( void ) ;
   void     slScrollEnd   ( void ) ;
   bool     slScrollNext  ( void ) ;
   bool     slScrollPrev  ( void ) ;

   //* Group of methods to support single-line, right-justified textbox edit.*
   bool     rjInsertChar  ( const wkeyCode& wk ) ;
   bool     rjDeleteChar  ( void ) ;

   //* Group of methods to support multi-line textbox edit.*
   bool     mlInsertChar  ( const wkeyCode& wk ) ;
   bool     mlReplaceChar ( const wkeyCode& wk ) ;
   bool     mlDeleteChar  ( const wkeyCode& wk ) ;
   void     mlScrollRight ( bool forceLTR = false ) ;
   void     mlScrollLeft  ( bool forceLTR = false ) ;
   void     mlScrollUp    ( void ) ;
   void     mlScrollDown  ( void ) ;
   void     mlScrollHome  ( void ) ;
   void     mlScrollEnd   ( void ) ;
   bool     mlScrollNext  ( void ) ;
   bool     mlScrollPrev  ( void ) ;
   short    mlIndex2Ip    ( const gString& gsText ) ;
   void     mlIp2Index    ( short iPoint ) ;
   bool     mlToD         ( void ) ;
   bool     mlEoD         ( void ) ;
   bool     mlEoL         ( void ) ;
   void     mlFmtDisplay  ( const gString& gsText, 
                            bool ipCalc = false, bool hBreak = false ) ;

   //* Group of methods used by all textbox controls *
   short    tbSetCursor ( short xoffset, short yoffset = ZERO ) ;
   void     tbSynchCursor ( bool refresh ) ;
   short    tbGetText ( gString& gsTrg, bool strip = true ) ;
   short    tbSetText ( const gString& srcText, bool srcFilter = true ) ;
   short    tbClear ( void ) ;                  //* Reset data members
   short    tbDisplayMessage ( dtbmData& data ) ; //* Display (but do not store) a message
   short    tbSelectText ( bool selectAll ) ;
   short    tbSelectChar ( bool rightward ) ;
   short    tbWriteColorString ( short y, short x, //* Write multi-colored text
                                 const wchar_t* srcPtr, attr_t dColor ) ;
   bool     tbApplyFilter ( wchar_t* wch ) ;    //* Character filtering group
   bool     tbApplyFilter ( gString& gsText ) ;
   bool     tbIsPrint ( wchar_t* wch, short adjCase = ZERO ) ;
   bool     tbIsAsciiPrint ( wchar_t wch ) ;
   bool     tbIsAlNum ( wchar_t wch, bool inclSpace = false ) ;
   bool     tbIsAlpha ( wchar_t wch, bool inclSpace = false ) ;
   bool     tbIsNumeric ( wchar_t wch, bool inclSign = true ) ;
   bool     tbIsHexNum ( wchar_t* wch, bool forceUpper = false ) ;
   bool     tbIsLower ( wchar_t* wch, bool inclSpace = false ) ;
   bool     tbIsUpper ( wchar_t* wch, bool inclSpace = false ) ;
   bool     tbIsURL ( wchar_t* wch ) ;
   bool     tbIsFileName ( wchar_t wch, bool inclSlash = false ) ;
   void     tbAlert ( short boinks = 1 ) ;      //* For debugging only

   //* Data members specific to single-line controls *
   wchar_t  wText[gsMAXCHARS] ;  //* For single-line: text data string
                                 //* For multi-line : un-displayed leading text
   bool     rightJust ;          //* 'true' if text input is right justified
   bool     hzShift ;            //* If 'true' (default), then display 
                                 //* text may be wider than control width.
                                 //*  Exception: Always 'false' if control's
                                 //*  width < MIN_TB_SHIFT_WIDTH columns.

   //* Data members specific to multi-line controls *
   multiText* mlText ;           //* Text data array
   wchar_t*   mlTail ;           //* Pointer to un-displayed trailing text

   //* Data members that apply to all textbox controls *
   winPos   tbCursor ;           //* Cursor position
   winPos   tbIndex ;            //* Index of character under cursor
   tbChars  filter ;             //* Text input filter (legal input characters)
   short    leftIndex ;          //* For LTR and RTL configurations, indicates
                                 //* first visible character. For right-justified
                                 //* configuration, leftmost occupied column.
   bool     audibleAlert ;       //* enable audible alert for invalid key input
   TbSelect select ;             //* Used only for user text selection
   friend class NcDialog ;       //* Allow NcDialog class access to all data
} ;

//* Billboard control (all methods and data are private) *
class DialogBillboard : protected DialogControl
{
   protected:
   DialogBillboard ( InitCtrl* iPtr ) ;   //* Constructor
   virtual ~DialogBillboard ( void ) ;    //* Destructor
   short    OpenControl ( bool hasFocus = false ) ;  //* Make control visible
   void     RedrawControl ( bool hasFocus ) ; //* Redraw and refresh
   void     RedrawControl ( bool hasFocus, bool refresh ) ;
   short    SetOutputFormat ( bool rtlFormat ) ; //* Set data format as RTL OR LTR output

   //* Replace exising data (all lines) *
   short    bbSetText ( const gString& srcText, 
                        bool hasFocus, const attr_t* srcAttr = NULL ) ;
   short    bbSetText ( const wchar_t* srcText, 
                        bool hasFocus, const attr_t* srcAttr = NULL ) ;
   short    bbSetText ( const char* srcText, 
                        bool hasFocus, const attr_t* srcAttr = NULL ) ;
   short    bbFormatLine ( const wchar_t* srcText, gString& gs, 
                           short* bStrip = NULL ) ;

   //* Add a line to existing data (potentially destroying an existing line)   *
   //* OR replace a single, existing line, OR append data to an existing line. *
   short    bbAddText ( const wchar_t* srcText, short& byteOffset, bool hasFocus, 
                        attr_t srcAttr = attrDFLT, 
                        short rowIndex = (-1), bool append = false ) ;
   short    bbNextFreeLine ( void ) ;
   short    bbNextFreeLine ( short& ctrlLines ) ;

   //* Get copy of existing text data for specified line.*
   short    bbGetText ( multiText* rowText, short rowIndex ) ;
   short    bbDisplayLines ( void ) { return this->lines ; }

   //* Erase text and color attribute data.*
   short    bbClear ( bool clearAll = true, const attr_t* srcAttr = NULL ) ;

   //* Modify color attributes for a single line or the entire list.*
   void     bbSetColor ( const attr_t* attrList, bool hasFocus ) ;
   void     bbSetColor ( short trgLine, attr_t trgAttr, bool hasFocus ) ;
   const attr_t* bbGetColor ( short& lineCount ) ;

   multiText* mlText ;     //* Text data array
   attr_t*    mlAttr ;     //* Array of color attributes
   short      maxWChars ;  //* Max capacity of mlText array
   bool       altAttr ;    //* 'true' if non-default color attributes specified
   bool       sclAttr ;    //* 'true' if mlAttr is scrolled with the text (unused)

   friend class NcDialog ; //* Allow NcDialog class access to all data
} ;

//* Radio Button control (all methods and data are private) *
class DialogRadiobutton : protected DialogControl
{
   protected:
   DialogRadiobutton ( InitCtrl* iPtr ) ; //* Constructor
   virtual ~DialogRadiobutton ( void ) ;  //* Destructor
   short    OpenControl ( bool hasFocus = false ) ; //* Make control visible
   void     RedrawControl ( bool hasFocus ) ; //* Draw with/without 'selected' character
   short    SetOutputFormat ( bool rtlFormat ) ; //* Set data format as RTL OR LTR output
   wchar_t  wText[MAX_RADIO_WIDTH] ;//* Control display text
   RBType   subtype ;               //* Display format for radio button
   wchar_t  selChar ;               //* Character indicating button is 'selected'
                                    //* 'groupCode' defines an exclusive-OR group (0==none)
   bool     selected ;              //* true if selected, else false
   friend class NcDialog ;          //* Allow NcDialog class access to all data
} ;

//* Scrollbox control (all methods and data are private) *
class DialogScrollbox : protected DialogControl
{
   protected:
   DialogScrollbox ( InitCtrl* iPtr ) ;   //* Constructor
   virtual ~DialogScrollbox ( void ) ;    //* Destructor
   short    OpenControl ( bool hasFocus = false ) ; //* Make control visible
   void     RedrawControl ( bool hasFocus ) ; //* Redraw entire control
   short    SetOutputFormat ( bool rtlFormat ) ; //* Set data format as RTL OR LTR output
   void     RefreshControl ( void ) ;     //* Refresh display of control

   NcWindow* bPtr ;           //* Pointer to control's outer (border) window
   char**   bText ;           //* Pointer to array of display-string pointers
                              //* (also points to head of dynamic-allocation block)
   attr_t*  bColor ;          //* Pointer to array of color attributes
   short    bItems ;          //* Number of elements in bText/bColor arrays
   short    bIndex ;          //* Index of currently-highlighted element
   bool     sdPaint ;         //* true if selection data have been previously displayed
   cdConnect bConnect ;       //* Codes for connecting border of control to dialog border
                     //* Programmer's Note: If both labY and labX are initialized to ZERO,
                     //* the label will be drawn as a title in the scroll box rather than as 
                     //* a label outside the scroll box. 
   friend class NcDialog ;    //* Allow NcDialog class access to all data
} ;

//* Dropdown control (all methods and data are private) *
class DialogDropdown : protected DialogControl
{
   protected:
   DialogDropdown ( InitCtrl* iPtr, short dialogUY, short dialogLI ) ; //* Constructor
   virtual ~DialogDropdown ( void ) ;   //* Destructor
   short    OpenControl ( bool hasFocus = false ) ;  //* Make control visible
   void     RedrawControl ( bool hasFocus ) ; //* Redraw entire control
   short    SetOutputFormat ( bool rtlFormat ) ; //* Set data format as RTL OR LTR output
   void     RefreshControl ( void ) ;   //* Refresh display of control

   //* 'Expanded' control is a Scroll Box       *
   //* This group of data variables is identical*
   //* to that of a DialogScrollbox class.      *
   NcWindow* bPtr ;     //* Pointer to control's outer (border) window
   char**   bText ;     //* Pointer to array of display-string pointers
                        //* (also points to head of dynamic-allocation block)
   attr_t*  bColor ;    //* Pointer to array of color attributes
   short    bItems ;    //* Number of elements in bText/bColor arrays
   short    bIndex ;    //* Index of currently-highlighted element

   //* Additional data for the 'collapsed' control,    *
   //* and switching between 'expanded' and 'collapsed'*
   ddBoxExpansionType ddxp ; //* Direction of expansion
   bool     expanded ;  //* True if control is expanded for user selection
   bool     sdPaint ;   //* True if selection data have been previously displayed

   NcWindow* tPtr ;     //* Pointer to text display window (displayed when control is collapsed)
   short    tLines ;    //* lines in collapsed control window
   short    tCols ;     //* Columns in collapsed control window
   short    tulY ;      //* Position of 'collapsed' control window
   short    tulX ;
   wchar_t  tcChar ;    //* Special character marking control as a drop-down
   attr_t   tcAttr ;    //* Special character color attribute (currently unused)
   short    tcYpos ;    //* Y offset for display of special character
   short    tcXpos ;    //* X offset for display of special character
   short    typos ;     //* Y offset for string display
   short    txpos ;     //* X offset for string display
   friend class NcDialog ;                   //* Allow NcDialog class access to all data
} ;

//* Menuwin control (all methods and data are private) *
class DialogMenuwin : protected DialogControl
{
   protected:
   DialogMenuwin ( InitCtrl* iPtr, short dialogUY, short dialogLI ) ; //* Constructor
   virtual ~DialogMenuwin ( void ) ;          //* Destructor
   short    OpenControl ( bool hasFocus = false ) ;  //* Make control visible
   void     RedrawControl ( bool hasFocus ) ; //* Redraw entire control
   void     RedrawControl ( bool hasFocus, bool initHilite, bool refresh = true ) ;
   short    SetOutputFormat ( bool rtlFormat ) ; //* Set data format as RTL OR LTR output
   void     RefreshControl ( void ) ;         //* Refresh display of control
   short    ScrollMenuData ( wkeyCode& wk ) ; //* Scroll through menu items

   NcWindow* bPtr ;     //* Pointer to control's outer (border) window
   char**   bText ;     //* Pointer to array of display-string pointers
                        //* (also points to head of dynamic-allocation block)
   attr_t*  bColor ;    //* Pointer to array of color attributes
   bool*    bActive ;   //* Pointer to array of flags indicating active items (initially all)
   short    bItems ;    //* Number of elements in bText/bColor arrays
   short    bIndex ;    //* Index of currently-highlighted element
   short*   subMenu ;   //* Pointer to array of sub-menu handles
                        //* groupCode defines the Menu-bar group (0==none)
                        //* Note: wLabel[] is the text displayed in 'collapsed' control
   bool     hideItems ; //* If 'true' hide inactive menu items

   //* Additional data for the 'collapsed' control,    *
   //* and switching between 'expanded' and 'collapsed'*
   bool     expanded ;  //* 'true' if control is expanded for user selection
   //bool     sdPaint ;   //* not currently used
   bool     grpVisible ;//* if groupCode > ZERO, then:
                        //*  'true' if is group member AND group is visible
   wchar_t  grpHotkey ; //* if groupCode > ZERO, then:
                        //* optional auxilliary hotkey for accessing a menu group
   NcWindow* tPtr ;     //* Pointer to text display window (displayed when control is collapsed)
   short    tLines ;    //* lines in collapsed control window
   short    tCols ;     //* Columns in collapsed control window
   short    tulY ;      //* Position of 'collapsed' control window
   short    tulX ;
   attr_t   tcAttr ;    //* Special character color attribute
   short    typos ;     //* Y offset for string display
   short    txpos ;     //* X offset for string display
   friend class NcDialog ;    //* Allow NcDialog class access to all data
} ;

//* Scrollext control (all methods and data are private) *
class DialogScrollext : protected DialogControl
{
   protected:
   DialogScrollext ( InitCtrl* iPtr ) ;      //* Constructor
   virtual ~DialogScrollext ( void ) ;       //* Destructor
   short    OpenControl ( bool hasFocus = false ) ;  //* Make control visible
   void     RedrawControl ( bool hasFocus ) ;//* Redraw entire control (existing data)
   void     RedrawControl ( bool hasFocus,   //* Redraw entire control (new or old data)
                            bool newData, bool refresh = true ) ;
   short    SetOutputFormat ( bool rtlFormat ) ; //* Set data format as RTL OR LTR output
   void     RefreshControl ( void ) ;        //* Refresh display of control

   NcWindow* bPtr ;     //* Pointer to control's outer (border) window
   const char**  bText ; //* Pointer to array of display-string pointers
   const attr_t* bColor ;//* Pointer to array of color attributes, one per display line
   int      bItems ;    //* Number of elements in bText/bColor arrays
   int      bIndex ;    //* Index of currently-highlighted element
   cdConnect bConnect ; //* Codes for connecting border of control to dialog border
   //bool     sdPaint ;   //* not currently used
   bool     hlShow ;    //* If true, current item is highlighted
               //* Programmer's Note: If both labY and labX are initialized to ZERO,
               //* the label will be drawn as a title in the scroll box rather than as 
               //* a label outside the scroll box. 
   friend class NcDialog ;                   //* Allow NcDialog class access to all data
} ;

//* Spinner control (all methods and data are private) *
class DialogSpinner : protected DialogControl
{
   protected:
   DialogSpinner ( InitCtrl* iPtr ) ;        //* Constructor
   virtual ~DialogSpinner ( void ) ;         //* Destructor
   short    OpenControl ( bool hasFocus = false ) ;  //* Make control visible
   void     RedrawControl ( bool hasFocus ) ;//* Redraw entire control
   short    SetOutputFormat ( bool rtlFormat ) ; //* Set data format as RTL OR LTR output
   wchar_t  wText[MAX_SPINNER_WIDTH] ;       //* Display-text buffer
   wchar_t  fmtSpec[MAX_SPINNER_WIDTH] ;     //* Formatting string for display output
   dspinFormat fmtCode ;                     //* Formatting code
   int      minVal ;                         //* Min spinner value
   int      maxVal ;                         //* Max spinner value
   int      curVal ;                         //* Current spinner value
   attr_t   iColor ;                         //* color attribute for spin indicator
   friend class NcDialog ;                   //* Allow NcDialog class access to all data
} ;

//* Parameters for invoking 'UserAlert' method. *
const short minUAINTERVAL = 2 ;
class AudibleAlert
{
   public:
   //* Default constructor - parameters set for a single ping *
   AudibleAlert ( void )
   {
      pCount = 1 ;
      pInterval = minUAINTERVAL ;
      pRepeat = ZERO ;
      pPattern = NULL ;
   }
   //* Initialization constructor - specify ping sequence *
   AudibleAlert ( short c, short i = minUAINTERVAL, short r = ZERO, 
                  short d = ZERO, const short* p = NULL )
   {
      pCount    = c ;
      pInterval = i ;
      pRepeat   = r ;
      pDelay    = d ;
      pPattern  = p ;
   }
   short pCount ;    //* number of pings  (range: 1 - 32,768)
   short pInterval ; //* delay (in twentieths of a second) between pings
                     //* (range: minUAINTERVAL - 32,768)
   short pRepeat ;   //* number of times to repeat pCount/pInterval sequence
                     //* (range: 1 - 32,768. A value of -1 indicates that    )
                     //* (sequence should be repeated until a key is pressed.)
   short pDelay ;    //* delay (in tenths of a second) between repeats of sequence
                     //* (range: 0 - 32,768)

   //* If specified, replaces pInterval with a sequence of interval values     *
   //* (in tenths of a second). pCount indicates the length of the array.      *
   const short* pPattern ;
} ;

//* Options for invoking the 'ShellOut' method. See below. *
enum soOptions : short
{
   soX,        // No message prompts, no pause
   soE,        // Initial message: 'type-Exit-to-return' (default)
   soP,        // Pause after execution with message: 'press-Enter-to-exit'
   soPN,       // Pause after execution with No messages
   soEP,       // Both soE and soP options
   soEPN,      // Both soE and soPN options
   soCX,       // Clear screen, then soX option
   soCE,       // Clear screen, then  soE option
   soCP,       // Clear screen, then  soP option
   soCPN,      // Clear screen, then soPN option
   soCEP,      // Clear screen, then soEP option
   soCEPN,     // Clear screen, then soEPN option

   soOPTIONS   // number of options
} ;



//*****************************
//* NcDialog Class Definition *
//*****************************
class NcDialog : public NcWindow
{
public:

/*****
 * Public methods inherited from the NcWindow class:
 * This is a list of some of the more important inherited methods.
 * Please see NcWindow.hpp for complete list and descriptions.

 winPos WriteString ( short startY, short startX, const char* uStr, 
                      attr_t cAttr, bool refresh=false, rtl = false ) ;
 winPos WriteString ( const winPos& startYX, const char* uStr, 
                      attr_t cAttr, bool refresh=false, rtl = false ) ;
 winPos WriteString ( short startY, short startX, const wchar_t* wStr, 
                      attr_t cAttr, bool refresh=false, rtl = false ) ;
 winPos WriteString ( const winPos& startYX, const wchar_t* wStr, 
                      attr_t cAttr, bool refresh=false, rtl = false ) ;
 winPos WriteString ( short startY, short startX, const gString& gStr, 
                      attr_t cAttr, bool refresh=false, rtl = false ) ;
 winPos WriteString ( const winPos& startYX, const gString& gStr, 
                      attr_t cAttr, bool refresh=false, rtl = false ) ;
 winPos WriteParagraph ( short startY, short startX, const gString& gStr, 
                         attr_t cAttr, bool refresh = false, rtl = false ) ;
 winPos WriteParagraph ( const winPos& startYX, const gString& gStr, 
                         attr_t cAttr, bool refresh = false, rtl = false ) ;
 winPos WriteChar ( short startY, short startX, const char* uChar, 
                    attr_t cAttr, bool refresh = false, rtl = false ) ;
 winPos WriteChar ( const winPos& startYX, const char* uChar, 
                    attr_t cAttr, bool refresh = false, rtl = false ) ;
 winPos WriteChar ( short startY, short startX, wchar_t wChar, 
                    attr_t cAttr, bool refresh=false, rtl = false ) ;
 winPos WriteChar ( const winPos& startYX, wchar_t wChar, 
                    attr_t cAttr, bool refresh=false, rtl = false ) ;
 short meEnableMouse ( mmask_t eventTypes, short interval = ncmiDFLT ) ;
 short meEnableStableMouse ( short dclick = ncmiNONE, bool swheel = true ) ;
 short meDisableMouse ( void ) ;
 bool  meMouseEnabled ( void ) ;
 short meSetEventTypes ( mmask_t eventTypes ) ;
 short meGetEventTypes ( mmask_t& eventTypes ) ;
 short meSetClickInterval ( short waitMax, short* oldMax = NULL ) ;
 short meGetClickInterval ( short& waitMax ) ;
 short meFlushEventQueue ( void ) ;
 short meAutoConvert ( bool enable ) ;
 short meAutoFilter ( bool filter ) ;
 const char* Get_NcWindow_Version ( void ) ;

 * Pass-through methods to the NCurses class:
 wkeyType GetKeyInput ( wkeyCode& wKey ) ; // overriden by NcDialog method (see above)
 wkeyType KeyPeek ( wkeyCode& wKey ) ;
 short UngetKeyInput ( wkeyCode& wKey ) ;
 short FlushKeyInputStream ( void ) ;
 void  ScreenDimensions ( short& scrLines, short& scrColumns ) ;
 void  Hibernate ( void ) ;
 void  Wake ( void ) ;
 const char* Get_NCurses_Version ( void ) ;
 const char* Get_nclibrary_Version ( void ) ;
 *****/

//* Destructor for an NcDialog object.                                         *
//* Release all resources held by the dialog window and its associated control *
//* objects. Erase dialog from the display.                                    *
//*                                                                            *
//* Input  : none                                                              *
//* Returns: nothing                                                           *
virtual ~NcDialog () ;                    //* Destructor
   
//* Constructor for an NcDialog object.                                        *
//*                                                                            *
//* Input  : dDef: a fully initialized InitNcDialog class object (by reference)*
//*                See definition of InitNcDialog class.                       *
//* Returns: implicitly returns a pointer to NcDialog object                   *
NcDialog ( InitNcDialog& init ) ;

//* Allocate additional resources for a previously-instantiatiated dialog      *
//* and draw the dialog in the terminal window.                                *
//*                                                                            *
//* Input  : none                                                              *
//* Returns: OK if successful, else ERR                                        *
short OpenWindow ( void ) ;

//* Refresh display of dialog window and all its controls. Used to update      *
//* display with changes made to the dialog or its controls.                   *
//* If the dialog had previously been marked as 'obscured', (see HideWin()),   *
//* re-draw the dialog image and discard the saved image data.                 *
//*                                                                            *
//* Input  : none                                                              *
//* Returns: nothing                                                           *
void  RefreshWin ( void ) ;

//* Erase (overwrite with spaces) contents of the window using the window's    *
//* background color, but do not refresh display.                              *
//*                                                                            *
//* Input  : clearAll: (optional, false by default)                            *
//*                    if 'false' clear interior, leaving border               *
//*                    if 'true' clear entire window including border          *
//* Returns: OK                                                                *
short ClearWin ( bool clearAll = false ) ;

//* Clear the specified line of the dialog using the defined background color. *
//*                                                                            *
//* Input  : lIndex  : index of line to be cleared (zero-based)                *
//*          clearAll: (optional, 'false' by default)                          *
//*                    if 'true', clear entire line                            *
//*                    if 'false' do not clear border area                     *
//*          xpos : (optional, by default, for LTR: one(1), for RTL: columns-2)*
//*                 starting column (zero-based)                               *
//*                 (ignored if clearAll != false)                             *
//*          rtl  : (optional, 'false' by default)                             *
//*                 if 'false', then clear from 'xpos' toward right            *
//*                 if 'true',  then clear from 'xpos' toward left             *
//*                 (ignored if clearAll != false)                             *
//* Returns: OK if successful, ERR if invalid line index                       *
short ClearLine ( short lIndex, bool clearAll = false, short xpos = 1, bool rtl = false ) ;

//* Clear the specified line of the dialog using an alternate color attribute. *
//*                                                                            *
//* Input  : lIndex  : index of line to be cleared                             *
//*          altColor: color attribute for the target line                     *
//*          clearAll: (optional, 'false' by default)                          *
//*                    if 'true', clear entire line                            *
//*                    if 'false' do not clear border area                     *
//* Returns: OK if successful, ERR if invalid line index                       *
short ClearLine ( short lIndex, attr_t altColor, bool clearAll = false ) ;

//* Clear the specified rectangular area of the dialog window.                 *
//* Optionally use an alternate color attribute and/or an alternate fill       *
//* character.                                                                 *
//*                                                                            *
//* Input  : lIndex   : index of top line of target area                       *
//*          cIndex   : index of left column of target area                    *
//*          lCount   : number of lines in target area                         *
//*          cCount   : number of columns in target area                       *
//*          fillColor: (optional, dialog window's background color by default)*
//*                     alternate color attribute for the target area          *
//*          fillChar : (optional, SPACE character by default)                 *
//*                     alternate character to write into each character cell  *
//*                     NOTE: Specified character must be a single-column      *
//*                           character. A multi-column character              *
//*                           specification will be ignored.                   *
//* Returns: OK if successful, ERR if parameter(s) out-of-range                *
short ClearArea ( short lIndex, short cIndex, short lCount, short cCount, 
                  attr_t fillColor = attr_t(-1), wchar_t fillChar = nckSPACE ) ;

//* Hide the dialog window from view, that is, save its display data, then     *
//* erase it from the screen.                                                  *
//* If display data have not already been saved by a previous call to          *
//* SetDialogObscured(), data will be saved before erase.                      *
//* If application has already saved the display data through a previous call  *
//* to SetDialogObscured(), just erase the dialog's display data.              *
//* A call to RefreshWin() will make window visible again and release the      *
//* saved data.                                                                *
//*                                                                            *
//* Input  : blankingColor: (optional) specifies a background color to be      *
//*                         used for erasing the dialog window                 *
//*                         default==use parent window's background color      *
//* Returns: OK                                                                *
short HideWin ( attr_t blankingColor = ZERO ) ;
   
//* Capture the dialog window display data and set the winObscured flag.       *
//* Application should call this method in anticipation of the dialog being    *
//* wholly or partially obscured by another object.                            *
//* Note: you must call this method BEFORE the dialog is actually obscured.    *
//* Similar to HideWin() except that the dialog is not erased.                 *
//*                                                                            *
//* Input  : none                                                              *
//* Returns: OK if successful                                                  *
//*          ERR if window data has already been saved                         *
short SetDialogObscured ( void ) ;

//* Move the dialog window from its current position in the terminal window    *
//* to the specified position (Y/X coordinates of upper left corner of         *
//* dialog window).                                                            *
//*                                                                            *
//* Input  : ulYX    : Y/X offset of new position                              *
//*          relative: (optional, false by default)                            *
//*                    if 'true', coordinates are interpreted as an offset     *
//*                     from current dialog position.                          *
//*                    if 'false', coordinates are interpreded as an offset    *
//*                     from upper left of terminal window.                    *
//* Returns: OK if successful,                                                 *
//*           On successful return, the ulYX parameter has been updated        *
//*           with the previous coordinates of the dialog window.              *
//*          ERR if:                                                           *
//*           1. window data has already been saved (see SetDialogObscured()). *
//*           2. if the new position would place all or part of the dialog     *
//*              outside the terminal window's display area                    *
//*           3. memory-allocation error                                       *
short MoveWin ( winPos& ulYX, bool relative = false ) ;
   
//* Report the dialog window's current position as an offset from the upper    *
//* left corner of the terminal window.                                        *
//*                                                                            *
//* Input  : none                                                              *
//* Returns: Y/X window position                                               *
winPos GetDialogPosition ( void ) ;
   
//* Report the dialog window's dimensions as the number of display lines and   *
//* display columns.                                                           *
//*                                                                            *
//* Input  : dLines (by reference): on return, number of display lines         *
//*          dCols  (by reference): on return, number of display columns       *
//* Returns: nothing                                                           *
void  GetDialogDimensions ( short& dLines, short& dColumns ) ;

//* Get a keycode and keytype from the ncursesw input stream.                  *
//* This includes all supported international character codes as well as       *
//* captured-and-translated escape sequences representing function keys,       *
//* cursor keys and other special-purpose key combinations.                    *
//*                                                                            *
//* If mouse input has been enabled, mouse events will also be returned in     *
//* caller's wkeyCode-class object's 'mevent' data member.                     *
//*         See below for additional information on mouse support.             *
//*                                                                            *
//* Input  : wKey ( by reference, initial values ignored)                      *
//*          receives the key input data                                       *
//* Returns: member of wkeyType                                                *
wkeyType GetKeyInput ( wkeyCode& wKey ) ;


               //*****************************************
               //** dctPUSHBUTTON Control Access        **
               //*****************************************
//* If control with input focus == dctPUSHBUTTON, call this method to get      *
//* user's key input. Returns when a Pushbutton has been been pressed (nckENTER*
//* or nckSPACE), when Pushbutton IS READY to lose focus (nckTAB or nckSTAB)   *
//* -OR- has lost focus due to a hotkey press.                                 *
//* The nckESC (Escape key) is a special case. We don't know what it means in  *
//* this context (probably user panic), but caller may want to handle it.      *
//*                                                                            *
//* Input  : uiInfo class (by reference) - initial values ignored              *
//* Returns: index of control that currently has the input focus               *
//*           (See note in NcDialog.hpp about interpretation )                 *
//*           (of values returned in the uiInfo-class object.)                 *
short EditPushbutton ( uiInfo& info ) ;

//* Set the display text and optionally the color attributes for a             *
//* dctPUSHBUTTON control, and refresh the control's display.                  *
//*                                                                            *
//* This is useful for changing the text each time the button is pressed,      *
//* for example:  'ENABLE' vs. 'DISABLE'.                                      *
//* NOTE: If you specify a 'hotkey' in the text string for the control,        *
//*       good style would indicate that a given control should always have    *
//*       the same hotkey (this is not enforced, but is highly recommended).   *
//*                                                                            *
//* Input  : cIndex: index into array of dialog controls                       *
//*          pbText: pointer to new text data                                  *
//*          fAttr : (optional) if specified, replaces the color attribute for *
//*                  the control when it has input focus                       *
//*          nAttr : (optional) if specified, replaces the color attribute for *
//*                  the control when it does not have input focus             *
//* Returns: OK if successful,                                                 *
//*          ERR if cIndex does not reference a dctPUSHBUTTON control          *
short SetPushbuttonText ( short ctrlIndex, const char* pbText,
                          attr_t fAttr = attrDFLT, attr_t nAttr = attrDFLT ) ;

//* Enable display of a border around a dctPUSHBUTTON control.                 *
//*  - By default, pushbutton controls have no border.                         *
//*  - Pushbutton controls defined with less than three (3) display lines      *
//*    or less than three (3) columns NEVER have a border.                     *
//*  - A border for pushbutton controls defined with >= 3 display lines/columns*
//*    MAY be specified.                                                       *
//*  - Display text for the control should reflect whether the border is       *
//*    activated. With border, the display text is automagically offset to     *
//*    avoid overwriting the border.                                           *
//*    Example: For a 3-line, 8-column control with "OK" centered in it:       *
//*             a) no border  : const char* txt = "        \n   OK   " ;       *
//*             b) with border: const char* txt = "  OK  " ;                   *
//*  - Aesthetically, if the border is to be enabled, it should happen before  *
//*    the first call to RefreshWin().                                         *
//*  - See test application Dialog1, Test07 for further explanation.           *
//*                                                                            *
//* Input  : cIndex: index into array of dialog controls                       *
//*          style : (optional, 'ncltSINGLE' by default)                       *
//*                  If specified, border style is a member of enum ncLineType.*
//* Returns: OK if successful,                                                 *
//*          ERR if cIndex does not reference a dctPUSHBUTTON control          *
//*              OR if control is too small to have a border                   *
short EnablePushbuttonBorder ( short cIndex, ncLineType style = ncltSINGLE ) ;


               //*****************************************
               //** dctTEXTBOX Control Access           **
               //*****************************************
//* If control with input focus == dctTEXTBOX, call this method to get user's  *
//* key input. Allows user to edit the text in the text box control.           *
//* Returns when editing is complete (nckENTER), when Textbox IS READY to lose *
//* focus (nckTAB or nckSTAB), or when Textbox has lost focus due to a hotkey  *
//* press. The nckESC (Escape) key causes any changes to the text to be        *
//* discarded and a return to caller as if the nckTAB key had been pressed.    *
//*                                                                            *
//* Input  : uiInfo class (by reference) - initial values ignored              *
//* Returns: index of control that currently has the input focus               *
//*           (See note in NcDialog.hpp about interpretation )                 *
//*           (of values returned in the uiInfo-class object.)                 *
short EditTextbox ( uiInfo& info ) ;

//* Replace current textbox data, if any, with specified data.                 *
//* Internally, all data are handled as wchar_t type ('wide') characters, and  *
//* the limits described below refer to wchar_t storage.                       *
//*                                                                            *
//* Text data provided may be longer than control width up to gsMAXCHARS.      *
//* If source data > gsMAXCHARS, it will be silently truncated to gsMAXCHARS.  *
//*                                                                            *
//* For single-line textbox controls:                                          *
//* ---------------------------------                                          *
//* - If horizontal shifting is disabled, text will be silently truncated      *
//*   to the width of control.                                                 *
//*                                                                            *
//* For multi-line textbox controls:                                           *
//* --------------------------------                                           *
//* - If horozontal shifting is disabled, text will be silently truncated      *
//*   to the size of the control, which is calculated as:                      *
//*                 (display lines * display columns).                         *
//*                                                                            *
//* Applying the specified text filter:                                        *
//* -----------------------------------                                        *
//* 1) All non-printing control characters (0x01 - 0x1F) will be silently      *
//*    stripped from the display text BEFORE the specified text filter         *
//*    is applied.                                                             *
//* 2) If the text filter encounters an invalid character, the control will    *
//*    contain a NULL string on return.                                        *
//*                                                                            *
//* Input  : cIndex : index of target dctTEXTBOX control                       *
//*          uPtr   : pointer to new display data in UTF-8 format              *
//*               OR                                                           *
//*          wPtr   : pointer to new display data in wchar_t format            *
//*               OR                                                           *
//*          gsData : gString object (by reference) containing the new text    *
//*                                                                            *
//* Returns: number of characters in string including the NULLCHAR             *
//*          OR ERR if data filter detected an invalid character               *
//*              OR specified control currently has input focus                *
//*              OR specified control is not of type dctTEXTBOX                *
short SetTextboxText ( short cIndex, const char* uPtr ) ;
short SetTextboxText ( short cIndex, const wchar_t* wPtr ) ;
short SetTextboxText ( short cIndex, const gString& gsData ) ;

//* Returns a copy of the specified text box's text data.                      *
//* It is the caller's responsibility to provide a buffer large enough to hold *
//* the text data:                                                             *
//*               - gString object, or                                         *
//*               - wchar_t 'wide' characters: gsMAXCHARS                      *
//*               - UTF-8 (byte encoded) data: gsMAXBYTES                      *
//*                                                                            *
//* Input  : cIndex : index of target dctTEXTBOX control                       *
//*          uPtr   : pointer to UTF-8 target buffer                           *
//*               OR                                                           *
//*          wPtr   : pointer to wchar_t target buffer                         *
//*               OR                                                           *
//*          gsData : gString object (by reference) to receive text            *
//* Returns: if target is a UTF-8 buffer: returns the number of bytes in       *
//*           string including NULLCHAR                                        *
//*          if target is a wchar_t buffer or gString object: returns the      *
//*           number of characters in the string including NULLCHAR            *
//*          OR ERR if specified control is not of type dctTEXTBOX             *
short GetTextboxText ( short cIndex, char uPtr[gsMAXBYTES] ) ;
short GetTextboxText ( short cIndex, wchar_t wPtr[gsMAXCHARS] ) ;
short GetTextboxText ( short cIndex, gString& gsData ) ;

//* Enable or disable editing of extended text data within a dctTEXTBOX        *
//* control. Extended text is text that is wider than the display field        *
//* width and will be horizontally shifted as necessary for visibility         *
//* during editing.                                                            *
//* - When extended text data is enabled, the text may expand during           *
//*   editing to gsMAXCHARS characters (approx. gsMAXCHARS*2 columns).         *
//* - When extended text data is disabled, the width of the text is            *
//*   limited to the number of display columns specified for the control.      *
//* Default Settings:                                                          *
//* - For textbox controls which are configured for left-justified input or    *
//*   for right-to-left (RTL) input,                                           *
//*   AND which have a control width >= MIN_TB_SHIFT_WIDTH columns:            *
//*   - Extended text is enabled by default.                                   *
//*   - Extended text may be explicitly disabled or reenabled.                 *
//* - For textbox controls which are configured for right-justified input      *
//*   OR which have a control width < MIN_TB_SHIFT_WIDTH columns               *
//*   - Extended text is always disabled.                                      *
//*                                                                            *
//* Input  : cIndex : index of target dctTEXTBOX control                       *
//*          enable : if 'true' allow EditTextbox() method to shift data       *
//*                    left/right for text longer than control width.          *
//*                   if 'false' display data cannot be shifted in             *
//*                    window (string length is limited to field width)        *
//* Returns: OK if successful                                                  *
//*          ERR if:                                                           *
//*          1. target control currently has input focus                       *
//*          2. width of target control is less than MIN_TB_SHIFT_WIDTH        *
//*          3. target control is configured for right-justified input         *
//*          4. specified control is not of type dctTEXTBOX                    *
short ExtendedTextboxData ( short cIndex, bool enable ) ;

//* Re-display the contents of the specified dctTEXTBOX control so that the    *
//* rightmost character is visible. This is useful for display of path/filename*
//* strings or other text where it is advantageous for the user to see the     *
//* the tail of the string. This method does not modify the contents of the    *
//* textbox in any way; it _temporarily_ displays the text with the last       *
//* character at the lower right corner of the field for user convenience.     *
//* - Note that if entire string is already visible in the control,            *
//*   then no action will be taken.                                            *
//*                                                                            *
//* Internally, this method calls 'DisplayTextboxMessage' (see below), but     *
//* does the shift calculation for you.                                        *
//*                                                                            *
//* Input  : cIndex : index of target dctTEXTBOX control                       *
//* Returns: OK if successful                                                  *
//*          ERR if:                                                           *
//*          1. target control currently has input focus                       *
//*          2. target control is configured for right-justified input         *
//*          3. target control is configured with horizontal shift disabled    *
//*          4. specified control is not of type dctTEXTBOX                    *
short DisplayTextboxTail ( short cIndex ) ;

//* A Text Box control may also be used as a message display area, as well     *
//* as a text-editing control. Use this method to display a single-color or    *
//* multi-color message in a Text Box. The message is not saved, merely        *
//* written to the display area. Any existing text data in the control         *
//* is unchanged, but is temporarily obscured.                                 *
//*                                                                            *
//* Notes:                                                                     *
//*  dtbmData class for passing data to this method.                           *
//*  See also the description of initialization constructors for the class.    *
//*  a) 'textData' member gets the text to be displayed (UTF-8 format).        *
//*     - The text input filter defined for the target control is ignored      *
//*       by this method.                                                      *
//*     - Because automatic line wrap is not performed by this method, it is   *
//*       recommended that for multi-line target controls. line breaks ('\n')  *
//*       be used to pre-format the data.                                      *
//*              (Line breaks are ignored for single-line controls.)           *
//*     - To erase the message and redisplay the contents (if any) of the      *
//*       control object, call this method with 'textData' set to an empty     *
//*       string ( "" ).                                                       *
//*  b) 'colorData' member specifies the color attribute(s) of the text        *
//*     - If colorData[0] == dtbmNFcolor, (default), then text will be         *
//*       displayed in the non-focus color defined for the text box.           *
//*     - If colorData[0] == dtbmFcolor, then text will be displayed in        *
//*       the focus color defined for the text box.                            *
//*     - Otherwise an array of color attributes is expected, and a bit of     *
//*       experimentation may be needed to achieve a beautious result.         *
//*       - Each character in textData (including any newline characters)      *
//*         should have a corresponding color attribute.                       *
//*       - Note that if the text is to be 'centered', AND if colorData[0]     *
//*         DOES NOT contain one of the special attributes described above,    *
//*         then the 'colorData' array must contain at least as many           *
//*         attributes as the number of character columns defined for the      *
//*         control (plus newlines). In general, this is:                      *
//*                 attributeCount = lines * columns + lines ;                 *
//*  c) 'refreshControl' member ('true' by default) indicates whether the      *
//*     display should be refreshed immediately.                               *
//*  d) 'centered' member ('false' by default).                                *
//*     - If 'false', the text is written as provided.                         *
//*     - If 'true', then each line of text will be horizontally centered      *
//*                  in the field.                                             *
//*  e) 'rtlText' member ('false' by default)                                  *
//*     - If 'false', the text is processed as LTR (left-to-right)             *
//*     - If 'true', the text processed text as an RTL (right-to-left) language*
//*                  (Arabic, Hebrew, Persian, Urdu, etc.)                     *
//*                                                                            *
//* Input  : cIndex : index of target dctTEXTBOX control                       *
//*          uData  : initialized dtbmData class object                        *
//*                                                                            *
//* Returns: OK if successful                                                  *
//*          ERR if:                                                           *
//*          1. message too long for control (message displayed-but-truncated) *
//*          2. target control currently has input focus                       *
//*          3. specified control is not of type dctTEXTBOX                    *
short DisplayTextboxMessage ( short cIndex, dtbmData& uData ) ;

//* Verify that a text string meets the specified filtering criteria           *
//* for strings created/used in EditText().                                    *
//* Use before pre-loading a text box with string data or to apply a           *
//* secondary filter to edited data. (target text box is not modified.)        *
//*                                                                            *
//* Input  : cIndex : index of target dctTEXTBOX control                       *
//*          uStr  : text data to be verified in UTF-8 format                  *
//*                  (Recommend that uStr be at least gsMAXBYTES)              *
//*              OR                                                            *
//*          wStr  : text data to be verified in wchar_t format                *
//*                  (Recommend that wStr be at least gsMAXCHARS in size)      *
//*              OR                                                            *
//*          gStr  : text data to be verified contained in a gString object    *
//*          filter: member of enum tbChars, indicates which filter to apply   *
//* Returns: OK  if all characters match filter criteria                       *
//*              Note: Some filters such as upper/lower case will adjust       *
//*                    the provided source data to match filter criteria.      *
//*          ERR if data filter detected an invalid character                  *
//*                (original data unchanged)                                   *
//*              OR specified control is not of type dctTEXTBOX                *
short VerifyTbText ( short cIndex, char* uStr, tbChars filter ) ;
short VerifyTbText ( short cIndex, wchar_t* wStr, tbChars filter ) ;
short VerifyTbText ( short cIndex, gString& gStr, tbChars filter ) ;

//* Enable or disable audible alert (beep) for invalid characters received     *
//* during edit of Textbox data. An invalid character in this context is any   *
//* keycode that is not recognized OR that has been filtered out by the        *
//* active input filter associated with the control.                           *
//* Audible alert is initially disabled, and may be enabled after instantiation*
//* of the control. The alert mechanism is a call to the UserAlert() method    *
//* (with no parameters) as described elsewhere.                               *
//*                                                                            *
//* Input  : cIndex : index of target dctTEXTBOX control                       *
//*          enable : 'true'  to enable audible alerts                         *
//*                   'false' to disable audible alerts                        *
//* Returns: OK if successful                                                  *
//*          ERR if specified control is not of type dctTEXTBOX                *
short TextboxAlert ( short cIndex, bool enable ) ;

//* Encode a raw, unencoded URI/URL sequence using the RFC 3986 standard or    *
//* its successors.                                                            *
//*  NOT YET IMPLEMENTED                                                       *
//* Input  : gStr   : (by reference) data string to be encoded                 *
//* Returns: 'true' if all characters match tbURL filter criteria, else 'false'*
bool  Encode_URI ( gString& gStr ) ;

//* Decode an encoded URI/URL sequence into its raw text form using the        *
//* RFC 3986 standard or its successors.                                       *
//*  NOT YET IMPLEMENTED                                                       *
//* Input  : gStr   : (by reference) data string to be decoded                 *
//* Returns: 'true' if all characters match tbURL filter criteria, else 'false'*
bool  Decode_URI ( gString& gStr ) ;

//* Configure the way the target control accepts and displays text input       *
//* during editing.                                                            *
//*   OR                                                                       *
//* Set the cursor position over the specified character of the text string.   *
//*                                                                            *
//* Configuring the dctTEXTBOX control for editing style:                      *
//* -----------------------------------------------------                      *
//*  - Configuration style is specified by specifying a member of              *
//*    enum tbCurPos as the 'offset' parameter.                                *
//*    - tbcpHOME  : Set the cursor on the first character of the text data.   *
//*                  - For controls previously configured as LTR               *
//*                    (left-to-right), this will be the top left corner.      *
//*                    Note: LTR is the default editing style.                 *
//*                  - For controls previously configured as RTL               *
//*                    (right-to-left), this will be the top right corner.     *
//*                  - Not applicable if control configured as right-justified *
//*                  - Ignored if control contains no data.                    *
//*    - tbcpAPPEND: Set the cursor to the 'append' position just past the     *
//*                  last character of text data, so that the next character   *
//*                  entered will be appended to the end of the existing text. *
//*                  - Not applicable if control configured as right-justified *
//*                  - Ignored if control contains no data.                    *
//*    - tbcpRIGHTJUST: Configure the target control for right-justified       *
//*                  editing. The cursor is fixed at the right edge of the     *
//*                  field, and as text is entered, existing data are shifted  *
//*                  left to make room for the new character.                  *
//*                  - Note: Right-justified editing is available ONLY for     *
//*                          single-line controls.                             *
//*                  - Note: Right-justified text cannot grow beyond the width *
//*                          of the control. If existing data are wider than   *
//*                          control when this parameter is specified, then    *
//*                          data will be truncated to fit the field.          *
//*                                                                            *
//* Setting a specific cursor position:                                        *
//* ---------------------------------------                                    *
//*  - Set the cursor at the specified offset in the text data.                *
//*  - Note that because text data may contain multi-column characters, the    *
//*    actual cursor position MAY NOT be specified directly. The cursor        *
//*    position specified is interpreted as the character offset into the text *
//*    string and the cursor will be set over that character.                  *
//*  - Does not immediately modify position of the visible cursor. The change  *
//*    becomes visible when control receives input focus.                      *
//*                                                                            *
//* Input  : cIndex : index number of text box control                         *
//*          offset : member of enum tbCurPos (see above)                      *
//*                   -OR- character offset at which to set the cursor         *
//*                        Range ZERO <= offset < number of text characters    *
//*                        in data (not incl NULLCHAR)                         *
//* Returns: OK if successful,                                                 *
//*          ERR if:                                                           *
//*          1. specified position is out of range                             *
//*             - if 'offset' > End-Of-Data, then cursor set at 'append'       *
//*          2. specified member of enum tbCurPos is not applicable (see above)*
//*          3  specified text box currently has input focus                   *
//*          4. specified control is not of type dctTEXTBOX                    *
short SetTextboxCursor ( short cIndex, short offset ) ;

//* Specifiy a list of keycodes reserved by the application for use in         *
//* 'selecting' text, and for Copy/Cut/Paste operations to the Textbox Local   *
//* clipboard from within the EditTextbox() method.                            *
//*                                                                            *
//* The user will be able to select, copy, cut and paste text ONLY if you      *
//* have defined this set of keycodes.                                         *
//*                                                                            *
//* Specifying Keycodes:                                                       *
//* --------------------                                                       *
//* 1) To provide the application designer with maximum flexibility, we have   *
//*    made the design decision not to pre-define Copy/Cut/Paste keycode       *
//*    definitions; however, it is likely that users will want this            *
//*    functionality, so consider defining this set of keycodes during your    *
//*    startup sequence.                                                       *
//*                                                                            *
//* 2) Although any key not dedicated to other uses may be selected as one of  *
//*    the clipboard access keys, typically, an application will use the       *
//*    conventional keycode groups to avoid confusing the user:                *
//*    a) Universal convention specifies that the CTRL+X (cut), CTRL+C (copy)  *
//*       and CTRL+V (paste) should be used for clipboard access whenever      *
//*       possible. In addition CTRL+A (select All) is conventionally used to  *
//*       select all text in the field.                                        *
//*       -- If these keys are unavaliable or are unacceptable to you, then    *
//*          the ALT+X, ALT+C, ALT+V and ALT+A are the recommended substitutes.*
//*       -- Note that character-by-character selection keycodes are           *
//*          pre-defined to allow integration with mouse input:                *
//*          rkeyList.selRight = {nckSRIGHT, wktFUNKEY} ;                      *
//*                              SHIFT+RightArrow (select toward the right)    *
//*              and                                                           *
//*          rkeyList.selLeft  = {nckSLEFT, wktFUNKEY} ;                       *
//*                              SHIFT+LeftArrow (select toward the left).     *
//*          These two definitions are automatically inserted into the list    *
//*          in case you forget them.                                          *
//*       -- Note on mouse input: The character-by-character selection keys    *
//*          are generated by the SHIFT key + the mouse ScrollWheel up/down.   *
//*                                                                            *
//*    b) For Linux/UNIX console applications, SHIFT+CTRL+C (copy) and         *
//*       SHIFT+CTRL+V (paste) are defined to avoid conflict with shell command*
//*       codes. Your application will never see these keycodes because they   *
//*       are captured by the terminal emulator, so don't use them in you list.*
//*       There is no defined console equivalent to the 'Cut' command keycode. *
//*                                                                            *
//*    c) Keycodes are defined in two parts the key 'type' and the 'keycode'   *
//*       itself. See definition of the wkeyCode class for details.            *
//*       -- If you know the key(s) you want to use, but are not sure of the   *
//*          key's 'type' and 'keycode', then please use the Dialog2 test      *
//*          application, Test 05 which returns the key type and NcDialog key  *
//*          const definition for each keyboard key pressed.                   *
//*                                                                            *
//* 3) It is assumed that the keycode data specified through this method are   *
//*    valid. Keycode validation IS NOT performed here.                        *
//*                                                                            *
//* 4) To release a previously-specified list of reserved keys, call with      *
//*    the following sequence:                                                 *
//*       rkeyList.reset() ;                                                   *
//*       dp->SetTextboxReservedKeys ( rkeyList ) ;                            *
//*                                                                            *
//* 5) Note that if you specify 'hotkeys' for user-interface control access,   *
//*    be sure that they are different from the keycodes specified for         *
//*    clipboard access. Within the EditTextbox() method, clipboard-access     *
//*    keycodes will mask any conflicting hotkeys.                             *
//* --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  -- *
//*                                                                            *
//* Input  : rkeyList : (by reference)                                         *
//*                     an initialized list of keycodes to be reserved         *
//*                     -- to re-enable any previously-reserved keycodes       *
//*                        call with rkeyList.reserve == false                 *
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
short SetTextboxReservedKeys ( reservedKeys& rkeyList ) ;


      //** - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - **
      //**          Textbox Local Clipboard method group.              **
      //** The Local Clipboard is a static local buffer used for       **
      //** transfer of data:                                           **
      //**  a) internally, among Textbox controls within an application**
      //**  b) by the application to pass data between the X-server    **
      //**    (system) clipboard and the NcDialog API.                 **
      //** The Textbox control which currently has input focus is      **
      //** either the source of data to the Local Clipboard, OR is the **
      //** target of data from the Local Clipboard.                    **
      //**   Clipboard access is a complex topic: please see further   **
      //**   discussion in the NcDialog API documentation, and study   **
      //*    the Dialogx test application.                             **
      //** - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - **

//* Copy the specified text data to the Textbox Local Clipboard buffer.        *
//*                                                                            *
//* Input  : srcText: text to be stored                                        *
//* Returns: number of characters stored (including NULLCHAR)                  *
short SetLocalClipboard ( const gString& srcText ) ;

//* Retrieve a copy of the current contents of the Textbox Local Clipboard     *
//* buffer.                                                                    *
//*                                                                            *
//* Input  : trgText: receives the text data                                   *
//* Returns: number of characters retrieved (including NULLCHAR)               *
short GetLocalClipboard ( gString& trgText ) ;

//* Returns the number of data bytes currently in the local clipboard buffer.  *
//*                                                                            *
//* Input  : none                                                              *
//* Returns: number of data bytes (including NULLCHAR)                         *
short GetLocalClipboard_Bytes ( void ) ;

//* Returns the number of characters currently in the local clipboard buffer.  *
//*                                                                            *
//* Input  : none                                                              *
//* Returns: number of (wchar_t) characters (including NULLCHAR)               *
short GetLocalClipboard_Chars ( void ) ;
      //** - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - **
      //**         End Textbox Local Clipboard method group.           **
      //** - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - **


//* Set DialogTextbox control's editing mode to 'insert' or 'overstrike'.      *
//* 'insert' mode: When user enters a  character, the character under cursor   *
//*                and all characters to the right of it are shifted one       *
//*                position to the right to make room for the new character.   *
//* 'overstrike' mode: When user enters a character, the character under the   *
//*                cursor is replaced with the new character.                  *
//* Note: insert/overstrike flag is ignored for textbox controls which are     *
//*       configured for right-justified input (see enum tbCurPos).            *
//*                                                                            *
//* Input  : overstrike : 'true'  to set overstrike mode,                      *
//*                       'false' to set insert mode                           *
//*          disable    : (optional, 'false' by default)                       *
//*                       if 'true', then the 'Insert' key is disabled to      *
//*                       prevent the user from changing the state of the      *
//*                       insert/overstrike flag.                              *
//* Returns: nothing                                                           *
void  SetTextboxInputMode ( bool overstrike, bool disable = false ) ;
bool  IsOverstrike ( void ) ;   // returns current state of overstrike flag


               //*****************************************
               //** dctBILLBOARD Control Access         **
               //*****************************************
//* If control with input focus == dctBILLBOARD, call this method to get       *
//* user's key input. Data displayed in this control cannot be directly edited.*
//* Currently this method is simply a stub that waits for key input indicating *
//* that the input focus is moving to another control; thus, it is recommended *
//* that dctBILLBOARD controls be instantiated as 'inactive'.                  *
//*                                                                            *
//* Input  : uiInfo class (by reference) - initial values ignored              *
//* Returns: index of control that currently has the input focus               *
//*           (See note in NcDialog.hpp about interpretation )                 *
//*           (of values returned in the uiInfo-class object.)                 *
short EditBillboard ( uiInfo& info ) ;

//* Replace current display data, if any, for a dctBILLBOARD control with      *
//* specified data. The data is a single, null-terminated character array with *
//* each line's data optionally separated by an NEWLINE ('\n') character.      *
//*  NOTE: Automatic inter-word line break calculation is performed, but is    *
//*        rudimentary, so if a line needs to break in a certain place, be     *
//*        safe and indicate the breakpoint with a NEWLINE character.          *
//*                                                                            *
//* Internally, all data are handled as wchar_t type ('wide') characters, and  *
//* the limits described below refer to wchar_t storage.                       *
//* The limits are based on the amount of display area available in the target *
//* control.                                                                   *
//* 1) The text stored for, and displayed on a single line is limited by the   *
//*    number of display columns i.e. the width of the control.                *
//* 2) If source data for any line of the control is wider than the control,   *
//*    the excess will be wrapped to the next line.                            *
//* 3) If the number of delimited lines of data is greater than the number of  *
//*    display lines defined for the control, OR if undelimited text is too    *
//*    long for the control, then the excess data will not be processed.       *
//* 4) All non-printing characters (Ctrl+A - Ctrl+Z, etc.) including newline   *
//*    characters will be silently stripped from the display data.             *
//*                                                                            *
//* Input  : cIndex : index of target dctBILLBOARD control                     *
//*          uPtr   : pointer to new display data in UTF-8 format              *
//*               OR                                                           *
//*          wPtr   : pointer to new display data in wchar_t format            *
//*               OR                                                           *
//*          gsData : gString object (by reference) containing the new data    *
//*          cPtr   : (optional, NULL pointer by default)                      *
//*                   if specified, points to an array of color attributes,    *
//*                   ONE VALUE FOR EACH DISPLAY LINE defined for the control  *
//* Returns: Note on return value (sourceIndex)                                *
//*          The value returned is the number of characters or bytes processed.*
//*          If display area is not large enough to display all the source     *
//*          data, then the value returned may be interpreted as the index of  *
//*          the first UNPROCESSED character or byte.                          *
//*            For UTF-8 source data                                           *
//*               index of first unprocessed source byte (if any)              *
//*               Example: uPtr[sourceIndex] would be the first byte of the    *
//*                        first unprocessed character.                        *
//*            For wchar_t source data                                         *
//*               index of first unprocessed source character (if any)         *
//*               Example: wPtr[sourceIndex] would be the first unprocessed    *
//*                        character.                                          *
//*            For gString object source data                                  *
//*               index of first unprocessed source character (if any)         *
//*               Example: gString::gstr()[sourceIndex] would be the first     *
//*                        unprocessed character
//*          OR ERR if specified control is not of type dctBILLBOARD           *
short SetBillboard ( short cIndex, const char* uPtr, 
                     const attr_t* cPtr = NULL ) ;
short SetBillboard ( short cIndex, const wchar_t* wPtr, 
                     const attr_t* cPtr = NULL ) ;
short SetBillboard ( short cIndex, const gString& gsData, 
                     const attr_t* cPtr = NULL ) ;

//* Returns a copy of the specified dctBILLBOARD display data.                 *
//* The data are copied to an array of multiText-class objects, one for each   *
//* line of the control. The data returned are 'wide' (wchar_t) characters.    *
//* Example of conversion from wchar_t to UTF-8:                               *
//*   short lineCount = 5 ;                                                    *
//*   multiText billboardData[lineCount];    // wchar_t data (source)          *
//*   char utfData[lineCount][mtLENGTH * 4]; // UTF-8 data (target)            *
//*   gString gs ;                           // conversion tool                *
//*   for ( short i = ZERO ; i < lineCount ; i++ )                             *
//*   {                                                                        *
//*      gs = billboardData[i].ln ;                                            *
//*      gs.copy( utfData[i], (mtLENGTH*4) ) ;                                 *
//*   }                                                                        *
//*                                                                            *
//* Input  : cIndex : index of target dctBILLBOARD control                     *
//*          mtPtr  : pointer to an array of multiText-class objects to        *
//*                   receive the text data                                    *
//* Returns: OK if successful,                                                 *
//*          OR ERR if specified control is not of type dctBILLBOARD           *
short GetBillboard ( short cIndex, multiText* mtPtr ) ;

//* Returns a copy of the display data for the specified line of a             *
//* dctBILLBOARD control.                                                      *
//*                                                                            *
//* Input  : cIndex : index of target dctBILLBOARD control                     *
//*          gsData : (by reference) receives the text data                    *
//*          lIndex : index of control line                                    *
//* Returns: OK if successful,                                                 *
//*          OR ERR if specified control is not of type dctBILLBOARD           *
short GetBillboard ( short cIndex, gString& gsData, short lIndex ) ;

//* Append a single line of text data to the currently-displayed data (if any).*
//* If all display lines are already occupied, then scroll all lines up by one,*
//* discarding data on first line, and append the new data at the last line.   *
//*                                                                            *
//* Input  : cIndex : index of target dctBILLBOARD control                     *
//*          uPtr   : pointer to new display data in UTF-8 format              *
//*               OR                                                           *
//*          wPtr   : pointer to new display data in wchar_t format            *
//*               OR                                                           *
//*          gsData : gString object (by reference) containing the new data    *
//*          cAttr  : (optional, attrDFLT by default)                          *
//*                   if specified, text will be written using the specified   *
//*                   color attribute. Options are: a member of enum dtbmColors*
//*                   or one of the color attributes defined in NCurses.hpp.   *
//*          lIndex : (optional, (-1) by default)                              *
//*                   if specified, value is interpreted as an index of        *
//*                   the control line whose data is to be replaced by         *
//*                   the new data. If not a valid line index, then it         *
//*                   will be ignored.                                         *
//*          attach : (optional, false by default)                             *
//*                   if 'true', then                                          *
//*                      for the target line, append new text to existing text *
//*                      (if any), shifting the data to the left as necessary  *
//*                      to fully display the new text on the target display   *
//*                      line                                                  *
//*                      NOTE: If new data is too wide for the field, i.e.     *
//*                      an application error, then new data will be           *
//*                      truncated to fit the field AND subsequent calls on    *
//*                      the same data stream may be incorrectly indexed.      *
//*                   if 'false', then                                         *
//*                      replace existing text (if any) on target line,        *
//*                      TRUNCATING the result if necessary to fit on the      *
//*                      target display line                                   *
//* Returns: number of characters or bytes processed                           *
//*           For note on return value, see 'SetBillboard' above.              *
//*          OR ERR if specified control is not of type dctBILLBOARD           *
short Append2Billboard ( short cIndex, const char* uPtr, attr_t cAttr = attrDFLT, 
                         short lIndex = -1, bool attach = false ) ;
short Append2Billboard ( short cIndex, const wchar_t* wPtr, attr_t cAttr = attrDFLT, 
                         short lIndex = -1, bool attach = false ) ;
short Append2Billboard ( short cIndex, const gString& gsData, attr_t cAttr = attrDFLT, 
                         short lIndex = -1, bool attach = false ) ;

//* Insert a single line of text data at the top of the currently-displayed    *
//* data (if any). Any existing data are scrolled DOWN by one line and the new *
//* text is inserted on the first display line of the control.                 *
//*                                                                            *
//* Input  : cIndex : index of target dctBILLBOARD control                     *
//*          uPtr   : pointer to new display data in UTF-8 format              *
//*               OR                                                           *
//*          wPtr   : pointer to new display data in wchar_t format            *
//*               OR                                                           *
//*          gsData : gString object (by reference) containing the new data    *
//*          cAttr  : (optional, attrDFLT by default)                          *
//*                   if specified, text will be written using the specified   *
//*                   color attribute. Options are: a member of enum dtbmColors*
//*                   or one of the color attributes defined in NCurses.hpp.   *
//*                                                                            *
//* Returns: OK if successful                                                  *
//*          OR ERR if specified control is not of type dctBILLBOARD           *
short Insert2Billboard ( short cIndex, const char* uPtr,
                         attr_t cAttr = attrDFLT ) ;
short Insert2Billboard ( short cIndex, const wchar_t* wPtr, 
                         attr_t cAttr = attrDFLT ) ;
short Insert2Billboard ( short cIndex, const gString& gsData, 
                         attr_t cAttr = attrDFLT ) ;

//* Erase the text and optionally any 'alternate-color' data from the control. *
//*                                                                            *
//* Input  : cIndex : index of target dctBILLBOARD control                     *
//*          clearAll: (optional, 'true' by default)                           *
//*                    - if 'true',  then the color attribute for each line    *
//*                      will be set to the default color                      *
//*                    - if 'false', then any previously-specified             *
//*                      'alternate' color attributes will be retained         *
//*          cPtr    : (optional, NULL pointer by default)                     *
//*                    if specified, points to an array of color attributes,   *
//*                    ONE VALUE FOR EACH DISPLAY LINE defined for the control,*
//*                    to reinitialize attribute array  (overrides 'clearAll') *
//* Returns: OK if successful                                                  *
//*          OR ERR if specified control is not of type dctBILLBOARD           *
short ClearBillboard ( short cIndex, 
                       bool clearAll = true, const attr_t* cPtr = NULL ) ;

//* Returns a const pointer to the color-attribute array for the specified     *
//* dctBILLBOARD control AND the number of items in the array.                 *
//*                Values may not be modified directly.                        *
//*            Please see SetBillboardColors method, below.                    *
//*                                                                            *
//* Input  : cIndex   : index of target dctBILLBOARD control                   *
//*          lineCount: (by reference, initial value ignored)                  *
//*                     receives the number of items in the referenced array   *
//*                                                                            *
//* Returns: pointer to color-attribute array                                  *
//*          OR NULL pointer if specified control is not of type dctBILLBOARD  *
const attr_t* GetBillboardColors ( short cIndex, short& lineCount ) ;

//* Establish an alternate color attribute map for the specified dctBILLBOARD  *
//* display data. Text data (if any) are not modified.                         *
//*                                                                            *
//* Input  : cIndex : index of target dctBILLBOARD control                     *
//*          cPtr   : (optional, NULL pointer by default)                      *
//*                   - if new map not specified, then the map is set to the   *
//*                     default color determined by whether the control        *
//*                     currently has the input focus                          *
//*                   - if specified, points to an array of color attributes,  *
//*                     ONE VALUE FOR EACH DISPLAY LINE defined for the control*
//* Returns: OK if successful                                                  *
//*          OR ERR if specified control is not of type dctBILLBOARD           *
short SetBillboardColors ( short cIndex, const attr_t* cPtr = NULL ) ;

//* Modify the color attribute for the specified display line of a             *
//* dctBILLBOARD control. Text data are not modified.                          *
//*                                                                            *
//* Input  : cIndex : index of target dctBILLBOARD control                     *
//*          lIndex : index of target line                                     *
//*          cAttr  : new color attribute for line                             *
//* Returns: OK if successful                                                  *
//*          OR ERR if lIndex is out-of-range                                  *
//*                 if specified control is not of type dctBILLBOARD           *
short SetBillboardColors ( short cIndex, short lIndex, attr_t cAttr ) ;


               //*****************************************
               //** dctRADIOBUTTON Control Access       **
               //*****************************************
//* If control with input focus == dctRADIOBUTTON, call this method to get     *
//* user's key input. Allows user to edit the state of an independent Radio    *
//* Button or the selection within a Radio Button group. Returns when the      *
//* 'selected' state of the button or group has changed (nckENTER or nckSPACE),*
//* when button or group IS READY to lose the input focus (nckTAB or nckSTAB)  *
//* ('selected' state may or may not have changed) -OR- button or group has    *
//* lost focus due to a hotkey press.                                          *
//*                                                                            *
//* Input  : uiInfo class (by reference) - initial values ignored              *
//* Returns: index of control that currently has the input focus               *
//*           (See note in NcDialog.hpp about interpretation )                 *
//*           (of values returned in the uiInfo-class object.)                 *
short EditRadiobutton ( uiInfo& info ) ;

//* Set or reset the 'selected' flag of the specified, independent Radio       *
//* Button control, and refresh the control's display.                         *
//*                                                                            *
//* Input  : cIndex: index into array of dialog controls                       *
//*          select: new state of 'selected' flag                              *
//*                                                                            *
//* Returns: OK  if successful                                                 *
//*          ERR if: a. specified control is a member of a radiobutton group   *
//*                  b. invalid control index                                  *
//*                  c. referenced control is not a dctRADIOBUTTON object      *
short SetRadiobuttonState ( short ctrlIndex, bool select ) ;

//* Return the current state of the 'selected' flag of the specified Radio     *
//* Button control.                                                            *
//*                                                                            *
//* Input  : cIndex: index into array of dialog controls                       *
//*          select: receives current state of 'selected' flag                 *
//*                  (by reference)                                            *
//*                                                                            *
//* Returns: OK  if successful                                                 *
//*          ERR if: a. invalid control index                                  *
//*                  b. referenced control is not a dctRADIOBUTTON object      *
short GetRadiobuttonState ( short ctrlIndex, bool& select ) ;

//* Establish a group of DialogRadiobutton class objects as an exclusive-OR    *
//* group. Within this group, exactly ONE radio button may be 'selected' by    *
//* the user at any time.                                                      *
//* Parameter is an array of index numbers for the  DialogRadiobutton objects  *
//* in the group. List is terminated by a -1.                                  *
//*   Example: short List[] = { 4, 5, 6, -1 }                                  *
//* Programmer's Note: If specified indices are not consecutive, anomolous     *
//* display behavior may result.                                               *
//*                                                                            *
//* Input  : array of control index numbers (-1 ends list)                     *
//* Returns: OK if successful                                                  *
//*          ERR returned if:                                                  *
//*              1. invalid index number in list                               *
//*              2. referenced control is not a dctRADIOBUTTON object          *
short GroupRadiobuttons ( short indexList[] ) ;

//* For the radio-button group of which the specified radio button control     *
//* is a member, determine which button in the group is currently 'selected'.  *
//* Note that exactly one member of a radio button group is 'selected'.        *
//*                                                                            *
//* Input  : cIndex : index of any dctRADIOBUTTON in the target button group   *
//* Returns: index of selected member of group                                 *
//*          returns ERR if specified control is not a member of a             *
//*           radio-button group                                               *
//*            OR specified control is not of type dctRADIOBUTTON              *
short GetRbGroupSelection ( short cIndex ) ;

//* Set the 'selected' control for a Radio Button group. The previously-       *
//* 'selected' group member will be reset.                                     *
//*                                                                            *
//* Input  : cIndex : index of control to be set as 'selected'                 *
//* Returns: index of 'selected' group member                                  *
//*          returns ERR if:                                                   *
//*           1. specified control is not of type dctRADIOBUTTON               *
//*           2. specified control is not a member of a radio-button group     *
short SetRbGroupSelection ( short cIndex ) ;


               //*****************************************
               //** dctSCROLLBOX Control Access         **
               //*****************************************
//* If control with input focus == dctSCROLLBOX, call this method to get user's*
//* key input. Allows user to edit which item in the control is 'selected'.    *
//* Returns to caller when user has finished selection (selection may or may   *
//* not have changed), -OR- when control has lost focus due to a hotkey press. *
//*                                                                            *
//* Input  : uiInfo class (by reference) - initial values ignored              *
//* Returns: index of control that currently has the input focus               *
//*           (See note in NcDialog.hpp about interpretation )                 *
//*           (of values returned in the uiInfo-class object.)                 *
//*           (highlight IS visible on return)                                 *
short EditScrollbox ( uiInfo& info ) ;

//* Returns the index of the highlighted item in the specified dctSCROLLBOX    *
//* control.                                                                   *
//*                                                                            *
//* Input  : cIndex : index number of source control                           *
//* Returns: index of 'selected' (highlighted) item                            *
//*          or ERR if:                                                        *
//*            a) specified control is not of type dctSCROLLBOX                *
short GetScrollboxSelect ( short cIndex ) ;

//* Set 'selected' (highlighted) item in specified dctSCROLLBOX control.       *
//*                                                                            *
//* Input  : cIndex : index number of target control                           *
//*          selMember : index of item to be set as 'selected'                 *
//* Returns: index of 'selected' member                                        *
//*          or ERR if:                                                        *
//*            a) invalid item index specified                                 *
//*            b) specified control currently has the input focus              *
//*            c) specified control is not of type dctSCROLLBOX                *
short SetScrollboxSelect ( short cIndex, short selMember ) ;

//* Connect the border of a dialog control object to the dialog window border. *
//* Visual connection does not alter the behavior of the control in any way;   *
//* it is just visually pleasing at times to integrate display of a control    *
//* and the dialog border.                                                     *
//* Note that not all controls have borders AND that not all controls are      *
//* allowed to extend into the parent window's border. Currently, only         *
//* dctSCROLLBOX and dctSCROLLEXT controls may be connected to the parent      *
//* dialog's border.                                                           *
//*                                                                            *
//* Input  : cIndex: index of target control                                   *
//*          connPoint: flags indicating which connections to make             *
//* Returns: OK if successful, or                                              *
//*          ERR if:                                                           *
//*           1. invalid control index                                         *
//*           2. connection is not supported for target control type           *
short ConnectControl2Border ( short cIndex, cdConnect& connectPoints ) ;


               //*****************************************
               //** dctSCROLLEXT Control Access         **
               //*****************************************
//* If control with input focus == dctSCROLLEXT, call this method to get user's*
//* key input. Allows user to edit which item in the control is 'selected'.    *
//* Returns to caller when user has finished selection (selection may or may   *
//* not have changed), -OR- when control has lost focus due to a hotkey press. *
//*                                                                            *
//* Input  : uiInfo class (by reference) - initial values ignored              *
//* Returns: index of control that currently has the input focus               *
//*           (See note in NcDialog.hpp about interpretation )                 *
//*           (of values returned in the uiInfo-class object.)                 *
//*           (highlight IS visible on return)                                 *
short EditScrollext ( uiInfo& info ) ;

//* This method is similar to the EditScrollext() method above, except that the*
//* highlight is not displayed for the current data item and the user cannot   *
//* 'select' a data item. This is useful for scrolling through informational   *
//* message text rather than through selectable items.                         *
//*                                                                            *
//* Input  : uiInfo class (by reference) - initial values ignored              *
//* Returns: index of control that currently has the input focus               *
//*           (See note in NcDialog.hpp about interpretation )                 *
//*           (of values returned in the uiInfo-class object.)                 *
//*           (highlight IS NOT visible on return)                             *
short ViewScrollext ( uiInfo& info ) ;

//* Specify the display text and associated color attributes for a             *
//* dctSCROLLEXT control. Unlike the display data for other control types, the *
//* display data for dctSCROLLEXT control is external to the NcDialog's memory *
//* space. This is done to allow the application to dynamically update the     *
//* display text and color attribute data.                                     *
//*                ("You'll thank me later." - Adrian Monk)                    *
//* NOTE: It is the application's responsibility to ensure that each item's    *
//*       text string width (number of display columns) exactly fits the width *
//*       of the control window i.e. the 'cols' specified during               *
//*       instantiation, minus 2 (for left and right border columns).          *
//* NOTE: If there are fewer data items than display lines, the unused lines   *
//*       will be filled with the background color of the first display item.  *
//*                                                                            *
//* Input  : cIndex: index of target dctSCROLLEXT control                      *
//*          displayData: an initialized ssetData class object                 *
//*           dispText : pointer to an array of pointers to text strings       *
//*           dispColor: pointer to an array of color attribute codes (int)    *
//*           dispItems: number of items in the display arrays                 *
//*           hlIndex  : specifies the item that is initially highlighted      *
//*           hlShow   : if true, show the highlight, else, hide it            *
//*                      Highlight remains in the specified state until        *
//*                      changed by another call to SetScrollextText() or until*
//*                      RefreshScrollextText() or EditScrollext() is called.  *
//* Returns: OK if successful, else ERR                                        *
short SetScrollextText ( short cIndex, ssetData& displayData ) ;

//* Refresh the dctSCROLLEXT display data previously specified via a call      *
//* to SetScrollextText(). Call this method if the text data and/or color      *
//* attribute data have been modified BUT the number of items in the arrays    *
//* have not. This method provides a much faster display update that to call   *
//* SetScrollextText() again.                                                  *
//*                                                                            *
//* Input : cIndex : index of target dctSCROLLEXT control                      *
//*         hlShow : (optional, default==true): if true, highlight             *
//*                  the current data item, else current data item is          *
//*                  drawn without highlight. Highlight remains in the         *
//*                  specified state until changed by another call to          *
//*                  RefreshScrollextText() or until SetScrollextText() or     *
//*                  EditScrollext() is called.                                *
//* Returns: OK if successful, else ERR                                        *
short RefreshScrollextText ( short cIndex, bool hlShow = true ) ;
   
//* Move the highlight within a dctSCROLLEXT control.                          *
//* Operates as if the user had pressed the specified scrolling key.           *
//*                                                                            *
//* Input  : cIndex == index of target dctSCROLLEXT control                    *
//*          sKey   == key code for one of the valid scrolling keys:           *
//*                nckUP, nckDOWN, nckPGUP, nckPGDOWN, nckHOME, nckEND         *
//*                (all other key values ignored)                              *
//*          iIndex == (optional: default == -1): if specified,                *
//*                indicates the index of the data item to be                  *
//*                highlighted (sKey will be ignored)                          *
//*                                                                            *
//* Returns: index of data item which is currently highlighted                 *
//*          or ERR if specified control is not of type dctSCROLLEXT           *
//*            invalid iIndex specified                                        *
short MoveScrollextHighlight ( short cIndex, wchar_t scrollKey, 
                               short itemIndex = -1 ) ;

//* Returns the index of the highlighted item in the specified dctSCROLLEXT    *
//* control.                                                                   *
//*                                                                            *
//* Input  : cIndex : index number of source control                           *
//* Returns: index of 'selected' (highlighted) item                            *
//*          or ERR if:                                                        *
//*            a) specified control is not of type dctSCROLLEXT                *
short GetScrollextSelect ( short cIndex ) ;

//* Set 'selected' (highlighted) item in specified dctSCROLLEXT control.       *
//*                                                                            *
//* Input  : cIndex : index number of target control                           *
//*          selMember : index of item to be set as 'selected'                 *
//* Returns: index of 'selected' member                                        *
//*          or ERR if:                                                        *
//*            a) invalid item index specified                                 *
//*            b) specified control currently has the input focus              *
//*            c) specified control is not of type dctSCROLLEXT                *
short SetScrollextSelect ( short cIndex, short selMember ) ;


               //*****************************************
               //** dctDROPDOWN Control Access          **
               //*****************************************
//* User interface for editing the position of the highlight within a          *
//* dctDROPDOWN control.                                                       *
//* On entry, the control is in its 'collapsed' form i.e. only the currently-  *
//* 'selected' data item is visible. We make the remaining data items visible  *
//* at the correct moment, and then make them invisible again before returning *
//* to caller. To put it another way, the data from which the user makes a     *
//* selection are only visible while inside this method.                       *
//*                                                                            *
//* Input  : uiInfo class (by reference) - initial values ignored              *
//*           EXCEPT: If info.viaHotkey != false, expand the control           *
//*                   immediately on entry.                                    *
//* Returns: index of control that currently has the input focus               *
//*           (See note in NcDialog.hpp about interpretation )                 *
//*           (of values returned in the uiInfo-class object.)                 *
short EditDropdown ( uiInfo& info ) ;

//* Returns the index of the highlighted item in the specified dctDROPDOWN     *
//* control.                                                                   *
//*                                                                            *
//* Input  : cIndex : index number of source control                           *
//* Returns: index of 'selected' (highlighted) item                            *
//*          or ERR if:                                                        *
//*            a) specified control is not of type dctDROPDOWN                 *
short GetDropdownSelect ( short cIndex ) ;

//* Set 'selected' (highlighted) item in specified dctDROPDOWN control.        *
//*                                                                            *
//* Input  : cIndex : index number of target control                           *
//*          selMember : index of item to be set as 'selected'                 *
//* Returns: index of 'selected' member                                        *
//*          or ERR if:                                                        *
//*            a) invalid item index specified                                 *
//*            b) specified control currently has the input focus              *
//*            c) specified control is not of type dctDROPDOWN                 *
short SetDropdownSelect ( short cIndex, short selMember ) ;


               //*****************************************
               //** dctMENUWIN Control Access           **
               //*****************************************
//* If control with input focus is of type dctMENUWIN, call this method to get *
//* user's key input.                                                          *
//* - For a menu which is a member of a Menuwin group known as a 'Menu Bar':   *
//*   returns when the user has selected an item from any member control in    *
//*   the Menu Bar OR when user has exited the Menu Bar without making a       *
//*   selection.                                                               *
//* - For a menu which is not a member of a Menuwin group:                     *
//*   returns when the user has selected an item from the menu OR when user    *
//*   has exited the menu without making a selection.                          *
//*                                                                            *
//* Input  : info: uiInfo class (by reference) - initial values ignored        *
//*                EXCEPT: If info.viaHotkey != false, expand the control      *
//*                        immediately on entry.                               *
//* Returns: index of control that currently has the input focus               *
//*           (See note in NcDialog.hpp about interpretation )                 *
//*           (of values returned in the uiInfo-class object.)                 *
short EditMenuwin ( uiInfo& info ) ;

//* For a context menu (control of type dctMENUWIN which is inactive and       *
//* invisible), call this method to:                                           *
//*  1. activate the menu (make it accessible to user input)                   *
//*  2. make the menu visible                                                  *
//*  3. get user's key input                                                   *
//*  4. hide the menu                                                          *
//*  5. deactivate the menu (make it not accessible to user input)             *
//*                                                                            *
//* Input  : cIndex : index of a context (normally invisible) Menuwin control  *
//*          info   : uiInfo class (by reference) - initial values ignored     *
//* Returns: index of control that currently has the input focus               *
//*           (See note in NcDialog.hpp about interpretation )                 *
//*           (of values returned in the uiInfo-class object.)                 *
short EditMenuwin ( short contextMenuIndex, uiInfo& info ) ;

//* Establish a MenuBar (a group of DialogMenuwin class objects).              *
//* The controls of a MenuBar may be edited as a group when EditMenuwin() is   *
//* called, returning when a selection is made in ANY member control of the    *
//* MenuBar's group.                                                           *
//*                                                                            *
//* The list of group members is passed as an array of index numbers for the   *
//* DialogMenuwin objects. List is terminated by a negative 1 (-1).            *
//*   Example: short List[] = { 4, 5, 6, -1 } ;                                *
//* NOTE: Sub-menus and other context menus MAY NOT be members of a MenuBar.   *
//*       Only visible, top-level menus, may be members of a MenuBar.          *
//*                                                                            *
//* NOTE: If specified indices are not consecutive, anomolous display behavior *
//*       may result, so members of a group should have consecutive indices.   *
//*                                                                            *
//* Input  : indexList: array of control index numbers (-1 ends list)          *
//*          auxHotkey: (optional, default==ZERO): if a value is specified,    *
//*                     it is an auxilliary hotkey that toggles the MenuBar's  *
//*                     visibility/invisibility                                *
//*                     Note that this key will not be recognized while in the *
//*                     EditXxx() method for an 'expanded' control because the *
//*                     expanded control may obscure the MenuBar's position.   *
//*                                                                            *
//* Returns: OK if successful                                                  *
//*          or ERR if:                                                        *
//*          1. if index list contains an invalid index number                 *
//*          2. an index in the list references a control which is not a       *
//*             DialogMenuwin object                                           *
//*          3. an index in the list references a sub-menu or context menu     *
short GroupMenuwinControls ( const short indexList[], wchar_t auxHotkey=ZERO ) ;

//* Establish a connection between display elements in a dctMENUWIN control    *
//* and context menus that will act as submenus for those elements.            *
//* Note: A context menu is a dctMENUWIN control that has been defined with    *
//*       a NULL string label, and is therefore invisible when in its          *
//*       collapsed state i.e. does not have input focus.                      *
//*                                                                            *
//* Input  : pMenu   : index of parent menu of type dctMENUWIN to              *
//*                      which sub-menus will be attached.                     *
//*          smList[]: array of index numbers for the DialogMenuwin            *
//*                      context menu objects to be associated with            *
//*                      each data item in the parent menu.                    *
//* Returns: OK if successful OR                                               *
//*          ERR if:                                                           *
//*           1. invalid index number                                          *
//*           2. any referenced control is not of dctMENUWIN type              *
//*           3. any specified sub-menu not defined as a context menu          *
short AttachMenuwinSubmenus ( short parentMenu, const short submenuList[] ) ;

//* Make a MenuBar (DialogMenuwin control group) invisible.                    *
//* Grouping must have been previously established through a call to           *
//* GroupMenuwinControls().                                                    *
//* After being hidden, the controls in the MenuBar cannot be accessed by the  *
//* Tab/ShiftTab sequence i.e. NextControl() / PrevControl() BUT they can      *
//* still be accessed and made visible again via the controls' respective      *
//* hotkeys (if any). If a valid hotkey for any member of the MenuBar is       *
//* detected, the MenuBar is made visible (see ShowMenuBar()) and the input    *
//* focus is moved to that control.                                            *
//* Additionally, if caller specified an auxilliary hotkey for accessing the   *
//* MenuBar when defining the DialogMenuwin group, then detection of that      *
//* hotkey will toggle the visibility/invisibility of the MenuBar, and when    *
//* MenuBar becomes visible via the auxilliary hotkey, the first (lowest index)*
//* control in the MenuBar will get the input focus.                           *
//*                                                                            *
//* Note on input focus:                                                       *
//*  If this method is called while a member of the specified MenuBar has the  *
//*  input focus, we must move the focus to a control that is not a member of  *
//*  member. At least one active control that DOES NOT belong to the MenuBar   *
//*  must be defined for this dialog in order to make the MenuBar invisible.   *
//*                                                                            *
//* Input  : mbIndex: index of any member control in a MenuBar                 *
//* Returns: OK if successful                                                  *
//*          ERR if:                                                           *
//*            1. mbIndex is not a valid control index                         *
//*            2. specified control is not a dctMENUWIN control                *
//*            3. specified control is not a member of a MenuBar               *
//*            4. all active controls in the dialog are members of the MenuBar *
short HideMenuBar ( short mbIndex ) ;

//* For a MenuBar (group of DialogMenuwin controls) previously hidden by a     *
//* call to  HideMenuBar(), make the MenuBar visible.                          *
//* This method simulates an access via the (optional) auxilliary hotkey       *
//* specified when the MenuBar was established.                                *
//* Note: This method will do nothing if:                                      *
//*       1. mbIndex is not a valid control index                              *
//*       2. the specified control is not a dctMENUWIN control                 *
//*       3. the specified control is not a member of a MenuBar                *
//* Note: If setFocus==true AND referenced MenuBar is already visible,         *
//*       input focus is simply moved to the control specified by mbIndex.     *
//*                                                                            *
//* Input  : mbIndex : index of any member control in a MenuBar                *
//*          setFocus: if 'true', immediately move the input focus to the      *
//*                      control indicated by mbIndex.                         *
//*                    if 'false', input focus is not changed                  *
//* Returns: index of control with input focus.                                *
short ShowMenuBar ( short mbIndex, bool setFocus ) ;

//* Returns current state of a MenuBar, visible or invisible.                  *
//*                                                                            *
//* Input  : mbIndex : index of any member control in a MenuBar                *
//* Returns: 'true' if specified MenuBar is visible                            *
//*          'false' if MenuBar is invisible OR                                *
//*            1. mbIndex is not a valid control index                         *
//*            2. specified control is not a dctMENUWIN control                *
//*            3. specified control is not a member of a MenuBar               *
bool  MenuBarVisible ( short mbIndex ) ;

//* Set the position of a dctMENUWIN control which acts as a context menu.     *
//* Context menus are DialogMenuwin class objects that are invisible when in   *
//* the 'collapsed' state, which is any time they are not being edited by the  *
//* user. In a GUI application, this would be the menu that opens when the     *
//* right mouse button is pressed. A context menu may be opened anywhere       *
//* within the dialog window. A context menu is created by instantiating a     *
//* dctMENUWIN object with a zero-length label.                                *
//*                                                                            *
//* Input  : cIndex: index of control to be positioned                         *
//*          offY  : Y offset from upper left of parent dialog                 *
//*          offX  : X offset from upper left of parent dialog                 *
//* Returns: OK if successful                                                  *
//*          ERR if error:                                                     *
//*           1. specified control is not a context menu,                      *
//*           2. the menu is already expanded (visible),                       *
//*           3. specified offset puts menu on or beyond dialog border.        *
short PositionContextMenu ( short cIndex, short offY, short offX ) ;

//* Set which menu items in a dctMENUWIN control are active (user-selectable). *
//* These changes are allowed only for a control that currently does not have  *
//* the input focus.                                                           *
//* A dctMENUWIN control has a fixed number of menu items determined at the    *
//* time the control is instantiated; however, some menu items may have no     *
//* meaning or must be disallowed in certain contexts. Menu items may therefore*
//* be activated or deactivated as necessary, and the item's color attribute   *
//* can be adjusted to visually indicate the item's current status.            *
//* Note on beautification: If the 'active' menu color uses a 'reverse'        *
//* attribute, then the 'inactive' color should also use a 'reverse' attribute *
//* to avoid confusion with the color of the highlighted item. For an example  *
//* of using this method, please see Dialog1 test application, Test10.         *
//* This method can also be used to change the color attribute for each menu   *
//* item with-or-without affecting the active/inactive status of each item.    *
//*                                                                            *
//* Input  : cIndex : index of dctMENUWIN control                              *
//*        : aFlags : an array of boolean values, one for each menu item       *
//*                   'true' == active, 'false' == inactive                    *
//*                   Note: At least one menu item must be 'active'.           *
//*                         (To deactivate an entire menu, see ControlActive())*
//*          aColors: (optional, NULL pointer by default)                      *
//*                   if specified, pointer to an array of color attributes,   *
//*                   one for each menu item.                                  *
//*          hide   : (optional, false by default)                             *
//*                   if 'true' hide (make invisible) all inactive menu items  *
//*                   [NOTE: 'hide' option is not currently implemented]       *
//* Returns: OK if successful                                                  *
//*          ERR if:                                                           *
//*              1. cIndex is not a valid control index                        *
//*              2. cIndex does not reference a control of dctMENUWIN type     *
//*              3. target control has input focus                             *
short SetActiveMenuItems ( short cIndex, const bool aFlags[], 
                           const attr_t* aColors = NULL, bool hide = false ) ;


               //*****************************************
               //** dctSPINNER Control Access           **
               //*****************************************
//* If control with input focus == dctSPINNER, call this method to get user's  *
//* key input. Allows user to adjust the value in a Spinner control. Returns   *
//* when the control IS READY to lose the input focus (displayed value may or  *
//* may not have changed) -OR- control has lost focus due to a hotkey press.   *
//*                                                                            *
//* User input keys:                                                           *
//* -------------------------------------------------------------------------- *
//*           up arrow:increase by 1               down arrow:decrease by 1    *
//*   Shift + up arrow:increase by 10      Shift + down arrow:decrease by 10   *
//* Control + up arrow:increase by 100   Control + down arrow:decrease by 10   *
//*     Alt + up arrow:increase by 1000      Alt + down arrow:decrease by 1000 *
//*        Page Up key:increase by 10%          Page Down key:decrease by 10%  *
//*          Home key :maximum value                  End key:minimum value    *
//*                                                                            *
//* Input  : uiInfo class (by reference) - initial values ignored              *
//* Returns: index of control that currently has the input focus               *
//*           (See note in NcDialog.hpp about interpretation )                 *
//*           (of values returned in the uiInfo-class object.)                 *
short EditSpinner ( uiInfo& info ) ;

//* Return the currently-displayed value for the specified dctSPINNER control. *
//*                                                                            *
//* Input  : cIndex : index into list of controls defined for this dialog      *
//*          spValue: (by reference, initial value ignored)                    *
//*                   receives currently-displayed value                       *
//*                   - See notes on interpretation of the returned value      *
//*                     in the dspinData-class definition above.               *
//* Returns: OK if successful                                                  *
//*          OR ERR if specified control is not of type dctSPINNER             *
short GetSpinnerValue ( short cIndex, int& spValue ) ;

//* Set a new display value for specified dctSPINNER control and update display*
//* IMPORTANT NOTE: Please carefully review the notes on dctSPINNTER-control   *
//*                 value encoding in the definition of the dspinData class.   *
//*                                                                            *
//* Input  : cIndex : index into list of controls defined for this dialog      *
//*          spValue: new 'current' value for the control                      *
//* Returns: OK if successful                                                  *
//*          OR ERR if spValue is outside the defined range for this control   *
//*                 if specified control currently has the input focus         *
//*                 if specified control is not of type dctSPINNER             *
short SetSpinnerValue ( short cIndex, int setValue ) ;

//* Specify a new minimum value, maximum value and current value for a         *
//* dctSPINNER control.                                                        *
//* IMPORTANT NOTE: Please carefully review the notes on dctSPINNTER-control   *
//*                 value encoding in the definition of the dspinData class.   *
//*                                                                            *
//* Input  : cIndex: index into list of controls defined for this dialog       *
//*          minValue : new minimum data value for the control                 *
//*          maxValue : new maximum data value for the control                 *
//*          setValue : new currently-displayed data value                     *
//* Returns: OK if successful                                                  *
//*          OR ERR if spValue is outside the minValue to maxValue range       *
//*                 if specified control currently has the input focus         *
//*                 if specified control is not of type dctSPINNER             *
short SetSpinnerLimits ( short cIndex, int minValue, int maxValue, int setValue ) ;


               //*****************************************
               //** Dialog-control Utility Methods      **
               //*****************************************
//* Shift focus to next control in list.                                       *
//* Note that only controls that are marked as Active may receive the input    *
//* focus.                                                                     *
//*                                                                            *
//* Input  : none                                                              *
//* Returns: index of control which currently has the input focus              *
short NextControl ( void ) ;
   
//* Shift focus to previous control in list.                                   *
//* Note that only controls that are marked as Active may receive the input    *
//* focus.                                                                     *
//*                                                                            *
//* Input  : none                                                              *
//* Returns: index of control which currently has the input focus              *
short PrevControl ( void ) ;
   
//* Return the index of the control that currently has the input focus.        *
//*                                                                            *
//* Input  : none                                                              *
//* Returns: index of control which currently has the input focus              *
short GetCurrControl ( void ) ;
   
//* Establish (or disable) a callback method that allows the mainline code     *
//* to dynamically update the controls within the dialog window.               *
//* The external method handles situations that the dialog code cannot.        *
//* For example:                                                               *
//*  - secondary validation of text data in a text box                         *
//*  - manual update of non-active controls i.e. display-only controls that    *
//*    user can view, but cannot access                                        *
//*                                                                            *
//* Input  : valid pointer to caller's method of the correct type              *
//*          OR NULL pointer to disable existing callback                      *
//* Returns: OK                                                                *
//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -    *
//* Notes:                                                                     *
//* - Dialog must be instantiated AND opened before calling this method        *
//*   since the callback will be performed immediately after it is             *
//*   established.                                                             *
//* - The EstablishCallback() method calls the specified method once with      *
//*   the parameter firstTime==true, to perform any required initialization.   *
//*   (wkey parameter will be meaningless for first call)                      *
//*   Subsequently, the NcDialog class always calls with firstTime==false.     *
//* - See the macro CUPTR above for a definition of the callback method.       *
//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -    *
short EstablishCallback ( CUPTR methodname ) ;

//* Activate or deactivate the specified control.                              *
//*   a. If active, control can receive input focus.                           *
//*   b. If inactive, control cannot receive input focus.                      *
//*                                                                            *
//* On instantiation, each control may be declared Active or Inactive.         *
//* During application execution, however, circumstances may arise where it    *
//* does not make logical sense to allow the user to access a given control.   *
//* Under these circumstances, the control could be deactivated, that is,      *
//* it would still be visible in the dialog, but would not be able to          *
//* obtain the input focus.                                                    *
//*                                                                            *
//* Input  : cIndex   : index of control to activate/deactivate                *
//*          activate : if true, activate the control                          *
//*                     if false, de-activate the control                      *
//*                                                                            *
//* Returns: OK if successful                                                  *
//*          ERR if invalid index OR target control currently has input focus  *
short ControlActive ( short cIndex, bool activate ) ;

//* Create a centered title in line ZERO of the dialog window.                 *
//*                                                                            *
//* Input  : dTitle : title string, character columns <= dialog width-4        *
//*                     (truncated if necessary to fit available display area) *
//*                     title may be in UTF-8, wchar_t or gString format       *
//*          cAttr  : (optional, default == dialog's border color)             *
//*                     alternate color attribute for title text               *
//*                                                                            *
//* Returns: Y/X position for start of string                                  *
winPos SetDialogTitle ( const char* dTitle, attr_t cAttr=attrDFLT ) ;
winPos SetDialogTitle ( const wchar_t* dTitle, attr_t cAttr=attrDFLT ) ;
winPos SetDialogTitle ( gString& dTitle, attr_t cAttr=attrDFLT ) ;

//* Specify that when the dialog is opened, all control labels and the dialog  *
//* title should be interpreted as RTL (right-to-left) language text.          *
//* - By default, all control labels and the dialog title (if specified) are   *
//*   interpreted as LTR (left-to-right) language text.                        *
//* - To draw labels and title as RTL language text, this method must be       *
//*   called AFTER dialog instantiation but BEFORE the call to OpenWindow().   *
//* - NOTE: Normally, it would make no sense to call with rtlFormat==false     *
//*   because this would have no effect on labels or title which have already  *
//*   been written into the dialog's display space. However, calling with      *
//*   rtlFormat==false would allow any SUBSEQUENT calls to SetDialogTitle()    *
//*   to be interpreted as LTR text.                                           *
//*                                                                            *
//* Input  : rtlFormat : (optional, 'true' by default) draw all labels and     *
//*                      title as RTL                                          *
//*                                                                            *
//* Returns: OK                                                                *
short DrawLabelsAsRTL ( bool rtlFormat = true ) ;

//* Enable (or disable) drawing/redrawing the contents of the control object   *
//* as RTL-language text. Does not refresh display.                            *
//* - By default, contents of all controls are interpreted as                  *
//*   LTR (left-to-right) language text.                                       *
//* - For RTL languages, it is strongly recommended, though not enforced, that *
//*   this method be called BEFORE the first time dialog display is refreshed  *
//*   (see error conditions below).                                            *
//* - Normally, there will be no need to call this method a second time for a  *
//*   particular control; however, for some control types, it may be           *
//*   theoretically useful to toggle between RTL and LTR data. Toggling        *
//*   between RTL and LTR formatting is supported for controls whose contents  *
//*   can be dynamically updated:                                              *
//*    dctPUSHBUTTON, dctTEXTBOX, dctBILLBOARD, dctSCROLLEXT and dctMENUWIN.   *
//* - Please refer to the Dialog4 test application, Test06 for examples.       *
//*                                                                            *
//* Input  : cIndex   : index of control for which to enable/disable RTL       *
//*                     content formatting                                     *
//*          rtlFormat: (optional, true by default)                            *
//*                     if true, output control contents as RTL text           *
//*                     if false, output control contents as LTR text          *
//*                                                                            *
//* Returns: OK if successful                                                  *
//*          ERR if invalid index OR if specified format cannot be applied     *
//*              to target control.                                            *
//*              a) Not applicable for dctSPINNER and dctRADIOBUTTON controls  *
//*                 (spinners contain only ASCII numeric text)                 *
//*                 (radio buttons contain only single-character symbols)      *
//*              b) For controls with static contents format may only be set   *
//*                 before display is drawn for the first time.                *
//*              c) For dctTEXTBOX controls: formatting may not be changed if  *
//*                 control currently has the input focus.                     *
//*              d) For dctMENUWIN controls: formatting may not be changed if  *
//*                 control is currently in the 'expanded' state.              *
short DrawContentsAsRTL ( short cIndex, bool rtlFormat = true ) ;

//* Generic information dialog. May be used to display any list of constant    *
//* strings, then waits for user to press Enter (or Space or ESC)              *
//* For reasons of beauty, messages begin at offset 2 in X and SHOULD end at   *
//* dialog-width minus 2.                                                      *
//*                                                                            *
//* Input  : gd : initialized genDialog class object (by reference)            *
//* Returns: nothing                                                           *
void  InfoDialog ( genDialog& gd ) ;

//* Generic decision dialog. May be used to display any list of constant       *
//* strings, then waits for user to select either the 'YES' or 'NO' pushbutton *
//* (or ESC == 'NO').                                                          *
//* For reasons of beauty, messages begin at offset 2 in X and SHOULD end at   *
//* dialog-width minus 2.                                                      *
//* Note that the question posed by the dialog should be written in such a way *
//* that can be answered with either a 'Yes' or 'No' response.                 *
//*                                                                            *
//* Input  : gd : initialized genDialog class object (by reference)            *
//* Returns: 'true' if user selects 'YES', else 'false'                        *
bool  DecisionDialog ( genDialog& gd ) ;

//* Sound an audible alert to attract user's attention.                        *
//* If setup parameters are provided the alert will follow the specified ping  *
//* sequence until the terminal count is reached OR until user presses any key.*
//* If no parameters are provided, then a single ping will be generated,       *
//* followed by an immediate return.                                           *
//*                                                                            *
//* Note that if user presses a key during the sequence, the key-input buffer  *
//* will be flushed before returning; however, a caffinated user may manage to *
//* press a key we don't see, so caller should check the key-input buffer on   *
//* return.                                                                    *
//*                                                                            *
//* Input  : pingParms : (optional, NULL pointer by default)                   *
//*                      If specified, provides parameters for a ping sequence.*
//*                      If not specified, a single ping is sounded.           *
//* Returns: OK  if specified operation is supported by the system             *
//*          ERR if alert not supported                                        *
short UserAlert ( AudibleAlert* pingParms = NULL ) ;

//* Sound an audible alert to attract user's attention.                        *
//*                                                                            *
//* This is a simplified version of UserAlert which does not require full      *
//* initialization. Instead of receiving the parameters in an AudibleAlert     *
//* object, this method takes only a ping count and an inter-ping delay.       *
//*                                                                            *
//* Input  : count    : number of pings                                        *
//*          interval : (optional) delay between pings                         *
//*                     - if specified, then this is the inter-ping delay      *
//*                       in 20ths of a second                                 *
//*                     - otherwise delay is set to approximately 0.2 seconds  *
//*                       which is a short, but reliable interval              *
//* Returns: OK  if specified operation is supported by the system             *
//*          ERR if alert not supported                                        *
short UserAlert ( short count, short interval = (minUAINTERVAL * 2) ) ;

//* Go to command shell.                                                       *
//*   1) save the dialog display data                                          *
//*   2) put the NCurses Engine to sleep                                       *
//*   3) clear terminal screen if specified                                    *
//*   4) display 'exit-to-return' message if specified                         *
//*   5) invoke 'bash' or the specified built-in or utility                    *
//*   6) on return:                                                            *
//*      a) display 'press-Enter-key' message if specified                     *
//*      b) pause for Enter key if specified                                   *
//*      c) wake the NCurses Engine                                            *
//*      d) restore the dialog window                                          *
//*                                                                            *
//* Important Note:                                                            *
//*      This method should be called ONLY from the primary dialog,            *
//*      AND there should be no sub-dialogs open.                              *
//*      If invoked from a sub-dialog, then the primary dialog's               *
//*      display data will not be properly saved.                              *
//*                                                                            *
//* Input  : option   : member of enum soOptions (optional, soE by default)    *
//*          shellCmd : (optional, NULL pointer by default)                    *
//*                     - if specified, invoke the specified shell builtin     *
//*                       command, shell program or console utility program    *
//*                     - if NULL pointer, then invoke a copy of 'bash' shell  *
//*                       NOTE: If your default shell IS NOT 'bash', then you  *
//*                       should use this parameter to specify your shell      *
//*                       program in 'shellCmd'.                               *
//*          textColor: (optional, NULL pointer i.e. Black text by default)    *
//*                     ANSI escape sequence which specifies the color of text *
//*                     written to standard output.                            *
//*                     (If your terminal does not support ANSI colors,)       *
//*                     (this will have no effect.                     )       *
//*                                                                            *
//* Returns: nothing                                                           *
void  ShellOut ( soOptions option = soE, const char* shellCmd = NULL, 
                 const char* textColor = NULL ) ;

// NOT YET IMPLEMENTED!
void  PseudoShell ( void ) ;


// OBSOLETE OBSOLETE OBSOLETE OBSOLETE OBSOLETE OBSOLETE OBSOLETE OBSOLETE 
#if 0    // OBSOLETE - MOVED TO NcwKey.cpp
//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *
//* Programmer's Note: The author believes that mouse input for a text-based   *
//* dialog is nearly useless. In addition, the ncurses C-language library's    *
//* mouse support functions suffer from multiple issues, primarily timing      *
//* artifacts. Still, it's not so bad, so please have a go; and as always,     *
//* constructive feedback on our implementation is welcome.                    *
//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *
//* Notes on mouse events:                                                     *
//* 1) Do not mix NcDialog-class mouse methods with the simple NCurses-class   *
//*    methods. The NCurses class knows nothing about the higher-level use of  *
//*    the mouse interface, and confusion may result.                          *
//* 2) A 'click' event will be reported ONLY if the 'press' and 'release'      *
//*    sub-events occur within the same character cell (row/column position),  *
//*    AND the 'press/release' sequence occurs within the specified interval.  *
//* 3) The ncurses library provides no mechanism for directly requesting the   *
//*    current mouse-pointer position; thus, position is only available when   *
//*    in the company of a button event.                                       *
//* 4) Detection of conditioning keys, 'Control', 'Shift' and 'Alt' in         *
//*    combination with a mouse event is not entirely straightforward because  *
//*    the GUI desktop will capture many mouse events that have associated     *
//*    conditioning key, AND/OR the xterm mouse driver (or other driver) may   *
//*    not report all combinations accurately. You can test your system's      *
//*    capabilities using the Dialog4 test application, Test03 (NCurses) or    *
//*    Test04 (NcDialog).                                                      *
//* 5) Please note that NcDialog does not access the mouse driver directly,    *
//*    but simply uses the available ncurses library primitives. For           *
//*    sophisticated mouse support, your application would need to access the  *
//*    mouse driver directly, which is beyond the scope of this project.       *
//*                                                                            *
//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *

//* Enable reporting of mouse events.                                          *
//* Mouse events are reported within the 'GetKeyInput' keyboard input stream.  *
//*                                                                            *
//* Input  : eventTypes : bit mask of event types to be enabled                *
//*                       Bit definitions are the BUTTONxxxx group of          *
//*                       macros defined in ncurses.h (yes, it's clunky,       *
//*                       but you only have to define the mask once).          *
//*                       NOTE:                                                *
//*                       If no mouse-event types are specified,               *
//*                       (eventTypes==ZERO), then the following mask is set   *
//*                       as the default:                                      *
//*                          eventTypes = BUTTON1_CLICKED |                    *
//*                                       BUTTON1_DOUBLE_CLICKED |             *
//*                                       REPORT_SCROLL_WHEEL;                 *
//*          interval  : (optional, ncmiDFLT by default)                       *
//*                      the maximum interval to wait between the start of an  *
//*                      event and the end of the event in thousandths (0.001) *
//*                      of a second. Specify an interval OR a member of       *
//*                      enum ncmInterval.                                     *
//* Returns: OK  if mouse-event reporting enabled AND the system supports all  *
//*              requested event types                                         *
//*          ERR if mouse could not be enabled                                 *
//*           OR if optionally-specified 'interval' value is out of range      *
//*           OR one or more of the requested event types is not supported by  *
//*              the system (remaing event types will still be reported)       *
//*              In this case, the meGetEventTypes() method should be called   *
//*              to determine which events will be reported.                   *
short meEnableMouse ( mmask_t eventTypes, short interval = ncmiDFLT ) ;

//* Enable enhanced reporting of mouse events.                                 *
//* Mouse events are reported within the 'GetKeyInput' keyboard input stream.  *
//*                                                                            *
//* This form of enabling the mouse interface is different from the more       *
//* general meEnableMouse() method above. Whereas meEnableMouse() will continue*
//* even if the system does not fully support the requested configuration,     *
//* this method requires that all necessary parameters be fully configured.    *
//* If the system cannot be configured as required, then the mouse interface   *
//* is disabled and ERR is returned.                                           *
//*                                                                            *
//* The purpose of this configuration method is to reduce the unfortunate      *
//* timing artifacts inherited from the xterm mouse driver and from the        *
//* ncurses(w) C-language library. It is hoped that this will reduce the loss  *
//* of valid mouse-event data and improve overall mouse performance.           *
//*                                                                            *
//* Enable synthesized timing for mouse events, AND restrict the mouse         *
//* interface to event types:                                                  *
//*      metB1_S : Button 1 - single click                                     *
//*      metB1_D : Button 1 - double click                                     *
//*      metSW_U : Scroll-wheel - scroll up   (see 'swheel' parameter)         *
//*      metSW_D : Scroll-wheel - scroll down                                  *
//*      with and without modifier keys (CTRL, SHIFT, ALT).                    *
//*                                                                            *
//* Input  : dclick : (optional, ncmiSTABLE by default)                        *
//*                   This parameter indicates the time to wait after the      *
//*                   first click event is received to determine whether       *
//*                   the event is a double-click.                             *
//*                   Specified in thousandths (0.001) of a second OR a        *
//*                   member of enum ncmInterval.                              *
//*          swheel : (optional, 'true' by default)                            *
//*                   'true'  : ScrollWheel events are reported                *
//*                   'false' : ScrollWheel events ARE NOT reported            *
//* Returns: OK  if mouse-event reporting enabled and the system supports all  *
//*              required mouse event types and specified timing parameters.   *
//*          ERR if:                                                           *
//*            a) mouse could not be enabled                                   *
//*            b) if optionally-specified 'interval' value is out of range     *
//*            c) one or more required event types not supported by the system *
//*            d) the required timing interval could not be established        *
//*            If ERR, then mouse interface will be disabled.                  *
short meEnableStableMouse ( short dclick = ncmiNONE, bool swheel = true ) ;

//* Disable reporting of mouse events.                                         *
//*                                                                            *
//* Input  : none                                                              *
//* Returns: OK  if mouse events successfully disabled, else ERR               *
short meDisableMouse ( void ) ;

//* Reports whether mouse input is currently enabled.                          *
//*                                                                            *
//* Input  : none                                                              *
//* Returns: 'true'  if mouse-event reporting enabled, else, 'false'           *
bool  meMouseEnabled ( void ) ;

//* Returns the mouse-event-type mask for currently-enabled event types.       *
//* Generally useful only if meEnableMouse() method returns an error.          *
//*                                                                            *
//* Input  : eventTypes : (by reference) receives the event mask               *
//* Returns: OK  if successful                                                 *
//*          ERR if mouse-event reporting not enabled                          *
short meGetEventTypes ( mmask_t& eventTypes ) ;

//* Modify the type(s) of mouse events which will be reported.                 *
//*                                                                            *
//* Input  : eventTypes : bit mask of event types to be enabled                *
//* Returns: OK  if ALL requested event-types are supported by the system      *
//*          ERR if mouse-event reporting not enabled                          *
//*           OR no event types specified                                      *
//*           OR one or more of the requested event types is not supported by  *
//*              the system (remaing event types will still be reported)       *
short meSetEventTypes ( mmask_t eventTypes ) ;

//* Get the current maximum mouse-click interval in thousandths of a second.   *
//*                                                                            *
//* Input  : waitMax   : (by reference, initial value ignored)                 *
//*                      receives current interval                             *
//* Returns: OK  if successtul                                                 *
//*          ERR if mouse-event input disabled                                 *
short meGetClickInterval ( short& waitMax ) ;

//* Specify the maximum interval between a button-press event and a following  *
//* button-release event that will result in a mouse 'click' being reported.   *
//* Note: this interval applies to single-, double- and triple-click events.   *
//* If the interval between press and release is greater than this value, then *
//* a 'click' event will not be reported.                                      *
//* Note that this inverval is optionally set when mouse-event reporting is    *
//* is enabled through the meEnableMouse method.                               *
//*                                                                            *
//* Input  : waitMax   : interval in thousandths (0.001) of a second           *
//*                      specify an interval OR a member of enum ncmInterval   *
//*          oldWait   : (optional, NULL pointer by default)                   *
//*                      pointer to variable to receive previous interval value*
//* Returns: OK  if successtul                                                 *
//*          ERR if value out-of-range OR mouse-event input disabled           *
short meSetClickInterval ( short waitMax, short* oldMax = NULL ) ;

//* Remove any mouse events waiting in the mouse-event queue.                  *
//*                                                                            *
//* Input  : none                                                              *
//* Returns: OK  if successful                                                 *
//*          ERR if mouse-event reporting not enabled                          *
short meFlushEventQueue ( void ) ;

//* Within an NcDialog window, mouse event are, by default, automatically and  *
//* intelligently converted to keycodes (when possible). This eliminates the   *
//* need for the application to handle mouse events manually. In general, the  *
//* less the application needs to know about mouse events, the better.         *
//* There may be circumstances, however, when you need to temporarily disable  *
//* this automatic conversion, but use caution: It means that the edit routines*
//* for the dialog controls will return from the edit with the unprocessed     *
//* mouse event, and your user-input loop needs to know what to do with it.    *
//*                                                                            *
//* Input  : enable : 'true'  to enable auto-conversion (default setting)      *
//*                   'false' to disable auto-conversion                       *
//* Returns: OK  if successful                                                 *
//*          ERR if mouse-event reporting not enabled                          *
short meAutoConvert ( bool enable ) ;

//* Temporarily disable filtering of spurious (unrequested) mouse events.      *
//* Filtering is enabled by default. Should REMAIN enabled except for testing. *
//*                                                                            *
//* Input  : filter : 'true' enables filtering                                 *
//*                   'false disables filtering                                *
//* Returns: OK if successful, or ERR if mouse-event input disabled            *
short meAutoFilter ( bool filter ) ;

#endif   // OBSOLETE - MOVED TO NcwKey.cpp
// OBSOLETE OBSOLETE OBSOLETE OBSOLETE OBSOLETE OBSOLETE OBSOLETE OBSOLETE 
//* Returns version string for the NcDialog class.  *
//* Note that this is ALSO the NcDialog API version.*
const char* Get_NcDialog_Version ( void ) ;


            //*******************************************
            //** Methods for development and debugging **
            //** Conditionally compiled: Please see    **
            //** ENABLE_DEVELOPMENT_METHODS flag above.**
            //*******************************************

//* Displays the contents of a uiInfo-class object returned from one of the    *
//* control editing routines.                                                  *
//*                                                                            *
//* Input  : wPos : screen coordinates for display                             *
//*          info : uiInfo-class object                                        *
//*          moose: (optional, 'false' by default)                             *
//*                 if 'true', then display mouse-event data (if any)          *
//* Returns: nothing                                                           *
void  Dump_uiInfo ( winPos wPos, uiInfo info, bool moose = false ) ;

//* Capture the dialog's display data, text and color attributes to a file.    *
//*                                                                            *
//* Input  : fPath       : full path/filename for data to be saved             *
//*          fhtml       : (optional, false by default)                        *
//*                        if 'false', then save as text data                  *
//*                                    (viewable in text editor)               *
//*                        if 'true',  then save as HTML data                  *
//*                                    (viewable in web browser)               *
//*          timeStamp   : (optional, default == true)                         *
//*                        if 'true', then write the timestamp record at the   *
//*                                   top of the output file.                  *
//*                        if 'false, then do not write the timestamp record   *
//*          stylePath   : (optional, HTML output only)                        *
//*                        if specified, path/filename to CSS definition file  *
//*                        (path will be written into the HTML <head></head>)  *
//*                        default == "screenshot-styles.css"                  *
//*          cellsPerLine: (optional, HTML output only)                        *
//*                        range: 1-40,  default == 4                          *
//*                        if specified, the number of <td></td> cells written *
//*                        per output line                                     *
//*          lineComments: (optional, HTML output only, default == true)       *
//*                        if 'true', then the raw text of each line is written*
//*                                   as a comment for easy searching          *
//*                        if 'false', then do not write the comment lines     *
//*          simpleColor : (optional, HTML output only, default == ZERO)       *
//*                        interior color of the of the dialog being captured  *
//*                        Referenced ONLY when simple HTML capture has been   *
//*                        specified (see docs for details)                    *
//*                                                                            *
//* Returns: OK if successful                                                  *
//*          ERR  : a) if parameter out-of-range                               *
//*                 b) if unable to write data to specified file               *
//*                 c) if development methods are disabled                     *
short CaptureDialog ( const char* fPath, bool fhtml = false, bool timeStamp = true, 
                      const char* stylePath = NULL, short cellsPerLine = 4,
                      bool lineComments = true, attr_t simpleColor = ZERO ) ;

//* Displays the standard formats available for radio buttons.                 *
//*                                                                            *
//* Input  : wPos : screen coordinates for display                             *
//*          baseColor: display color for all but 'select' character           *
//*          selColor : display color for 'select' character                   *
//* Returns: nothing                                                           *
void  DisplayRadiobuttonTypes ( winPos wPos, attr_t baseColor, attr_t selColor ) ;

//* Display the characters of the Alternate Character Set (ACS).               *
//* The ACS is accessed by setting the ncaATTR bit of the color attribute used *
//* to display the character. The ACS is used for drawing lines in the window  *
//* or displaying certain special characters. The individual characters may be *
//* accessed through the acsXXX group of constants in NCurses.hpp.             *
//* Example of drawing a line-intersect character:                             *
//*       char intersection = acsINSECT ;                                      *
//*       this->WriteChar ( 2, 2, &intersection, nc.bw | ncaATTR ) ;           *
//*                                                                            *
//* Input  : wPos     : screen coordinates for display                         *
//*        : baseColor: display color dialog background                        *
//* Returns: nothing                                                           *
void  DisplayAltCharacterSet ( winPos wPos, attr_t baseColor ) ;

//* Display the 'wide' equivalents to the Alternate Character Set (ACS).       *
//* These characters are used for drawing lines in the window or for displaying*
//* certain special characters. The individual characters may be accessed      *
//* through the wcsXXXX group of constants in NCurses.hpp.                     *
//* NOTE: The ncaATTR color-attribute bit IS NOT USED to display these         *
//*       characters.                                                          *
//* Example of drawing a line-intersect character:                             *
//*       wchar_t intersection = wcsINSECT ;                                   *
//*       dp->WriteChar ( 2, 2, intersection, nc.bw ) ;                        *
//*                                                                            *
//* Input  : wPos     : screen coordinates for display                         *
//*        : baseColor: display color dialog background                        *
//* Returns: nothing                                                           *
void  DisplayWideCharacterSet ( winPos wPos, attr_t baseColor ) ;

//* Display the line-drawing characters defined within the wcsXXXX group of    *
//* constants in NCurses.hpp.                                                  *
//* These characters are used for drawing lines in the window.                 *
//* See also the LineDef class definition in NcWindow.hpp.                     *
//*                                                                            *
//* Input  : wPos     : screen coordinates for display                         *
//*        : baseColor: display color dialog background                        *
//* Returns: nothing                                                           *
void  DisplayLineDrawingSet ( winPos wPos, attr_t baseColor ) ;

//* Display the line-drawing characters defined within the wcsXXXX group of    *
//* constants in NCurses.hpp.                                                  *
//* These characters are used for drawing lines in the window.                 *
//* See also the LineDef class definition in NcWindow.hpp.                     *
//*                                                                            *
//* Input  : wPos     : screen coordinates for display                         *
//*        : baseColor: display color dialog background                        *
//* Returns: nothing                                                           *
void  DisplayLineDrawingExamples ( winPos wPos, attr_t baseColor ) ;

//* Displays a debugging message along the last line of the dialog.            *
//* Useful for development, but don't use it in final application.             *
//*                                                                            *
//* Input  : msg   : message to be displayed                                   *
//*                  if message == "%", clear the message area                 *
//*          pause : (optional, ZERO by default): wait before return           *
//*                  interpreted as number of seconds to wait                  *
//*                  pause >= 30 indicates wait for keypress                   *
//*          erase : (optional, false by default): if true, erase the message  *
//*                  after the pause                                           *
//* Returns: nothing                                                           *
void  DebugMsg ( const char* msg, short pause = ZERO, bool erase = false ) ; 

//********
//* DATA *
//********
private:
short       dLines ;       //* Number of lines in the dialog
short       dCols ;        //* Number of columns in the dialog
short       dulY,          //* Coordinates for upper-lefthand corner of dialog
            dulX ;
ncLineType  bStyle ;       //* Dialog window's border line-style
attr_t      bColor ;       //* Dialog window's border color
attr_t      iColor ;       //* Dialog window's interior color
DialogControl* dCtrl[MAX_DIALOG_CONTROLS] ; //* Pointers to dialog control data
CUPTR       ExternalControlUpdate ; //* Pointer to callback method (if any)
dSaveWin*   dswData ;      //* Pointer to structure containing dialog's 
                           //* saved display data (if any)
reservedKeys resKeys ;     //* For dctTEXTBOX controls, keys reserved for
                           //* clipboard access during edit
wchar_t     dTitle[MAX_LABEL_CHARS] ; // Dialog title string (if any)
short       currCtrl ;     //* Index of control that has focus
short       lastCtrl ;     //* Index of last control in list
char        cgGroupCode ;  //* Code letter for next control grouping to be established
bool        overStrike ;   //* If true, text input to dialog is in 
                           //* overstrike mode, else insert mode
bool        disableIns ;   //* If true, disable user access to 'Insert' key
                           //* during Textbox edits.
bool        rtlLabels ;    //* If true, draw labels and dialog title as RTL
bool        winObscured ;  //* If true, dialog is wholly or partially obscured 
                           //* by another object


//*******************
//* Private methods *
//*******************
void  RedrawCurrControl ( bool focus ) ;
short RedrawWindow ( void ) ;
void  EditSlText ( DialogTextbox* cp, uiInfo& info ) ;   // single-line-control input
void  EditMlText ( DialogTextbox* cp, uiInfo& info ) ;   // multi-line-control input
void  EditRjText ( DialogTextbox* cp, uiInfo& info ) ;   // right-justified input
void  etbGetUserText ( DialogTextbox* cp, uiInfo& info );// textbox user input
short etbTb2LocalClipboard ( bool cutSel = false ) ;     // copy from textbox to local clibboard
short etbLocalClipboard2Tb ( void ) ;                    // copy from local clipboard to textbox
void  etbDeleteSelection ( DialogTextbox* cp ) ;         // delete 'selected' text from textbox
short etbProcessReservedKeys ( DialogTextbox* cp,        // process reserved clipboard keys
                               const wkeyCode& wk ) ;
void  ToggleIns ( void ) ;                               // toggle insert/overstrike
void  TermResized ( void ) ;                             // if term win resized, refresh
void  EditSingleRadio ( uiInfo& info ) ;
void  EditGroupRadio ( uiInfo& info ) ;
short EditScrollext ( uiInfo& info, bool withHighlight ) ;
void  emwEditMultilevel ( short refList[MAX_DIALOG_CONTROLS + 2], uiInfo& info ) ;
void  emwEditSingle ( DialogMenuwin* cp, uiInfo& info ) ;
bool  emwEditCollapsed ( uiInfo& info ) ;
short emwSubmenuCount ( DialogMenuwin* cp ) ;
bool  emwIsAuxHotkey ( wkeyCode wk, short& ahkIndex ) ;
short emwFocus2GroupEnd ( bool last ) ;
bool  ToggleMenuBar ( short mbIndex ) ;
short RadioXorUpdate ( short rbIndex, bool updateFocus = true ) ;
short IsHotkey ( wkeyCode wKey ) ;
short GetActiveCtrl ( short startIndex, bool forward ) ;
short ChangedFocusViaHotkey ( wchar_t keyIn, short hotIndex, uiInfo& info ) ;
short AutoPositionX ( InitCtrl* p ) ;
short AutoPositionY ( InitCtrl* p ) ;
void  CaptureDialogDisplayData ( void ) ;
short CaptureDialog ( SaveWin& sw ) ;
void  ccapSimple ( DialogPushbutton* cpt, wchar_t* txtData, attr_t* atrData, short charsinX ) ;
void  ccapMulti  ( DialogScrollbox* cpt, wchar_t* txtData, attr_t* atrData, short charsinX ) ;
void  ccapMTit ( DialogMenuwin* cpt, wchar_t* txtData, attr_t* atrData ) ;
short ccapFile ( SaveWin& sw, const wchar_t* txtData, const attr_t* atrData, 
                 const char* fPath, bool fhtml, bool tStamp, 
                 const char* sPath, short cpl,  bool lComment, attr_t sbg ) ;
short ccapDocs ( const attr_t bkgnd ) ;
short meAssignDefaultHotkey ( short cIndex ) ;
short meTransformEvent ( wkeyCode& wk ) ;
void  meRecodeHotkey ( wkeyCode& wk ) ;
bool  meIsObscured ( wkeyCode trgPoint ) ;
short ShowMenuBar ( short mbIndex ) ;
short ShowWin ( void ) ;
bool  GenericDialog ( genDialog& gd, bool decide ) ;
//* For debugging only *
void  ET_Stat ( DialogTextbox* cp, gString* gsPtr = NULL ) ;

} ;

#endif   // NCDIALOG_INCLUDED

