//******************************************************************************
//* File       : SrcProf.cpp                                                   *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 1998-2015 Mahlon R. Smith, The Software Samurai *
//*                  GNU GPL copyright notice located in SrcProf.hpp           *
//* Date       : 30-Oct-2015                                                   *
//* Version    : (see AppVersion string in SrcProf.hpp)                        *
//*                                                                            *
//* Description: Count the source data lines and comments in program source    *
//* files and perform simple statistical analysis on the data.                 *
//*                                                                            *
//* The author believes strongly that consistent formatting, use of            *
//* white-space and above all, meaningful comments are the foundation of       *
//* maintainable code. For this reason we would like to gently encourage all   *
//* you UNIX / K&R dinosaurs to come into the 21st century.                    *
//*                                                                            *
//* The purpose of comments is not so much to describe what you are doing,     *
//* but WHY you are doing it. This gives the future maintenance programmer     *
//* (probably YOU) a clue about the arcanly beautiful, but altogether          *
//* incomprehensible code you wrote to make the project a soaring success      *
//* the first time around.                                                     *
//*                                                                            *
//*      "A computer language is not just a way of getting a computer          *
//*       to perform operations but rather … it is a novel formal              *
//*       medium for expressing ideas about methodology. Thus, programs        *
//*       must be written for people to read, and only incidentally for        *
//*       machines to execute."                                                *
//*       -- Harold Abelson, Gerald Jay Sussman with Julie Sussman             *
//*          Structure and Interpretation of Computer Programs                 *
//*          MIT Press, 1985                                                   *
//*                                                                            *
//******************************************************************************
//* Copyright Notice:                                                          *
//* This program is free software: you can redistribute it and/or modify it    *
//* under the terms of the GNU General Public License as published by the Free *
//* Software Foundation, either version 3 of the License, or (at your option)  *
//* any later version.                                                         *
//*                                                                            *
//* This program is distributed in the hope that it will be useful, but        *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
//* for more details.                                                          *
//*                                                                            *
//* You should have received a copy of the GNU General Public License along    *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.            *
//*                                                                            *
//*         Full text of the GPL License may be found in the Texinfo           *
//*         documentation for this program under 'Copyright Notice'.           *
//******************************************************************************
//* Version History (most recent first):                                       *
//*                                                                            *
//* v: 0.0.18 07-Oct-2015                                                      *
//*   - Restructure GetCommandLineArgs() method to make it more robust.        *
//*       a) allow combining options into a single token                       *
//*       b) pre-validate all specified source filenames for existence and type*
//*       c) allow all four methods of filename specification to coexist:      *
//*          direct name specification, '-A' option, '-L' option, and          *
//*          '-D' option.                                                      *
//*          Note, however, that this opens the possibility of duplicate names,*
//*          and at this time, we do not check for duplicate filenames.        *
//*   - Restructure startup code to take advantage of pre-validated source     *
//*     files.                                                                 *
//*   - Verify that we are running in a UTF-8 aware locale in both Text Mode   *
//*     and Interactive Mode.                                                  *
//*   - Allow source files which have no filename extension _IF_ the first     *
//*     line of the file identifies it as a shell script.                      *
//*   - Complete the scan method for Perl source files: '.pl', '.pm', '.t.'.   *
//*   - Add CSS (Cascading Style Sheets to the 'C' language comment family.    *
//*   - Implement optional ANSI color output for Text Mode display data.       *
//*     ANSI color output is _enabled_ by default. Although this was requested *
//*     by multiple users, we aren't sure that enabling color by default is    *
//*     the right choice because ANSI escape sequences look like crap if       *
//*     stdout is redirected to a file. In a future release, we MAY decide to  *
//*     disable ANSI output by default.                                        *
//*   - Source file types not yet fully supported. Stubs are provided which    *
//*     currently just count the total lines, and leave other parameters at    *
//*     zero.                                                                  *
//*     - PHP                                                                  *
//*     - Ruby                                                                 *
//*     - SQL                                                                  *
//*                                                                            *
//* v: 0.0.17 06-Aug-2015                                                      *
//*   - Automatically expand the Interactive Mode dialog vertically to fill    *
//*     the terminal window.                                                   *
//*   - Minor changes to output formatting for beautification.                 *
//*   - Put all NcDialog access under a conditional compile flag, so           *
//*     application can be built as a pure command-line utility. NcDialog      *
//*     header files are still #include'd, but are unreferenced. The           *
//*     'lncursesw' and NcDialog.a libraries will not be linked.               *
//*     With Interactive Mode and Text Mode:                                   *
//*       a) Set #define COMM_LINE_ONLY (0)                                    *
//*       b) gmake clean                                                       *
//*       c) gmake                                                             *
//*     With Text Mode only (Interactive Mode disabled):                       *
//*       a) Set #define COMM_LINE_ONLY (1)                                    *
//*       b) gmake -f Make_textonly clean                                      *
//*       b) gmake -f Make_textonly                                            *
//*   - Create temporary files in the system's 'tmp' directory.                *
//*   - Simplify the imDebug() method.                                         *
//*   - Update copyright notices and dates.                                    *
//*   - Update HTML stats output from v3.2Final to HTML 5.0.                   *
//*   - Add scan of Texinfo (.texi') source and another scan for HTML mark-up. *
//*     Maintainability of mark-up languages is not really applicable, but     *
//*     it's handy for all the docs we create in these formats.                *
//*   - Update Texinfo docs.                                                   *
//*                                                                            *
//* v: 0.0.16 05-Dec-2013                                                      *
//*   - Update integer formatting in fixed-width fields to use the             *
//*     gString-class formatInteger() method. This affects only the            *
//*     AnalyzeFileList() method, and it eliminates the FormatInteger.hpp      *
//*     header file (and removes some bugs).                                   *
//*   - Create a scripting-language threshold group in ProfData class.         *
//*   - Update dialog colors to support 16-color terminal emulation.           *
//*                                                                            *
//* v: 0.0.15 03-Jun-2012                                                      *
//*   - Port from 'C' to 'C++', and from DOS to Linux/UNIX.                    *
//*   - Convert from text-only display to an optional NcDialog-class interface.*
//*                                                                            *
//* v: 0.0.14 07-Aug-2001                                                      *
//*   - Added a Help screen.                                                   *
//*   - Fixed formatting problem when input file was of an unknown type.       *
//*                                                                            *
//* v: 0.0.13 17-Jul-2001                                                      *
//*   - Implemented scan of multiple files, listed in a .TXT file.             *
//*                                                                            *
//* v: 0.0.12 12-Oct-2000                                                      *
//*   - Fixed major bugs in C and CPP scan.                                    *
//*                                                                            *
//* v: 0.0.11 04-May-2000                                                      *
//*   - Add support for .S07 HC11 and .S33 HC12 assembler files.               *
//*                                                                            *
//* v: 0.0.10 11-Sep-1998 Created as a DOS command-line utility.               *
//******************************************************************************
//* Programmer's Notes:                                                        *
//* - Most string data used here are 'wide' characters. This was done for more *
//*   consistent output to the display. Although Linux speaks UTF-8, some      *
//*   C-library functions do not. Filenames are intrinsically multi-lingual    *
//*   and column alignment of display output requires conversion to wchar_t    *
//*   (wide) characters.                                                       *
//* - We use the gString class for conversion between UTF-8 and wchar_t        *
//*   strings - in fact, we over-use it. There is a very good reason for this: *
//*   the gString class is our invention, and we wished to stress test it as   *
//*   thoroughly as possible before inflicting it on the world at large.       *
//*   Of all the code written for the NcDialog class family, (and there's a    *
//*   lot if it) gString may well be the most useful.                          *
//* - Speaking of the NcDialog class, the use of it in this application is     *
//*   quite simplistic, but it proves the usefulness of wrapping the ncurses   *
//*   primitives in a warm and fuzzy C++ environment. In our opinion the C     *
//*   language is dying, and although we resisted the code bloat of C++ for as *
//*   long as possible, the self-documenting maintainability, modularization,  *
//*   and general robustness of C++ is undeniable.                             *
//*                                                                            *
//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *
//*                                                                            *
//* Clarity Index ==                                                           *
//*              100.00 - ((CommentLines+MixedLines) / SourceLines) * 100.0 ;  *
//*                                                                            *
//******************************************************************************
//* TO-DO LIST:                                                                *
//* ===========                                                                *
//* -- Capture of filenames for HTML varients: XHTML, XHT, XML                 *
//* -- Ruby                                                                    *
//* -- PHP                                                                     *
//* -- SQL                                                                     *
//* -- Update docs:                                                            *
//*    --                                                                      *
//* -- Add '-e=fname' option to 'e'xclude a file from processing.              *
//*    Useful in conjunction with '-a' or '-d=dirname' options.                *
//*    (This may be more trouble than it's worth.)                             *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//******************************************************************************

//****************
//* Header Files *
//****************
#include "SrcProf.hpp"

//****************
//* Local Data   *
//****************


//*************************
//*         main          *
//*************************
//******************************************************************************
//* Program entry point.                                                       *
//*                                                                            *
//* Command-line Usage:  See GetCommandLineArgs() below for command-line       *
//*                      argument processing.                                  *
//*                                                                            *
//* Returns: 0 if all command-line argument are valid                          *
//*         -1 if one or more arguments invalid or if one or more specified    *
//*            source files not found or inaccessible                          *
//******************************************************************************

int main ( int argc, char* argv[], char* argenv[] )
{
   //* Gather our entry-sequence data *
   commArgs clArgs( argc, argv, argenv ) ;

   //* Create the application class object *
   SrcProf sProf( clArgs ) ;
   bool validArgs = sProf.ValidCommArgs() ;

   return (validArgs ? ZERO : (-1) ) ;

}  //* End main() *

//*************************
//*     ~SrcProf          *
//*************************
//******************************************************************************
//* Destructor. Return all resources to the system.                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

SrcProf::~SrcProf ( void )
{
   //* Release any dynamic memory allocation *
   //* attached to our internal data members *
   this->pdRelease () ;
   this->tnfRelease () ;

   //* If we created any temp files, remove them now *
   this->spfDeleteTempFiles ( true ) ;

}  //* End ~SrcProf() *

//*************************
//*      SrcProf          *
//*************************
//******************************************************************************
//* Default constructor.                                                       *
//*                                                                            *
//* Input  : commArgs class object (by reference)                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

SrcProf::SrcProf ( commArgs& ca )
{
   //* Set the locale to the environment's choice i.e. UTF-8 encoding.*
   //* This affects all input and output streams.                     *
   std::locale::global(std::locale(""));
   #if 0    // DEBUGGING ONLY
   wcout << L"ioLocale:" << std::locale().name().c_str() << endl ;
   #endif   // DEBUGGING ONLY

   //* Initialize our data members.                             *
   //* (pd members are initialized by the ProfData constructor) *
   this->dRows = this->dCols = ZERO ;
   this->tnfPtr = NULL ;
   this->tnfCount = ZERO ;
   this->emCount = ZERO ;

   //* Get path of current-working directory *
   this->spfGetCWD ( this->cwDir ) ;

   //* Get path of user's temp-file directory, and create *
   //* a filename for the list of files to be scanned.    *
   this->spfCreateTemppath () ;

   //* User may have specified one or more command-line arguments *
   this->validArgs = this->GetCommandLineArgs ( ca ) ;

   //* If valid user input, but not a cry for help *
   if ( this->validArgs && !ca.helpFlag && !ca.verFlag ) 
   {
      //* If alternate sort option specified *
      this->pd.fileSort = ca.sortOpt ;

      //* If alternate threshold values specified *
      if ( ca.threshFlag != false )
      {
         this->pd.goodpctC = ca.goodC ;
         this->pd.fairpctC = ca.fairC ;
         this->pd.goodpctA = ca.goodA ;
         this->pd.fairpctA = ca.fairA ;
         if ( ca.goodS >= ZERO )
         {  //* Set script/markup thresholds *
            this->pd.goodpctS = ca.goodS ;
            this->pd.fairpctS = ca.fairS ;
         }
         else
         {  //* Default calculation for script/markup thresholds *
            this->pd.goodpctS = ca.fairC ;
            this->pd.fairpctS = this->pd.goodpctS / 2 ;
         }
      }

      //* If source files have been specified, allocate *
      //* a work space, copy the filespec data to it.   *
      if ( ca.fnCount > ZERO )
      {
         //* Allocate memory for storage of profile data:  *
         //*   1) file-data objects    (this->pd.fileData) *
         //*   2) display strings      (this->pd.dispData) *
         this->pdAllocate ( ca.fnCount ) ;

         ifstream ifs( this->lstFile.ustr(), ifstream::in ) ;
         if ( ifs.is_open() )
         {
            for ( UINT i = ZERO ; i < this->pd.fileCount ; ++ i )
            {
               if ( (this->spfReadLine ( ifs, this->pd.fileData[i].fPath )) != false )
               {
                  this->spfExtractFilename ( this->pd.fileData[i].fName, 
                                             this->pd.fileData[i].fPath ) ;
               }
            }
            ifs.close() ;
         }
         else     // list file not found (this is unlikely) *
         {
            this->pdRelease () ;
         }
      }

      //* Analyze the source-file data, and format data for display *
      if ( this->pd.fileCount > ZERO )
        this->AnalyzeFileList () ;

      //* Text Mode - display results of analysis.*
      if ( ca.textFlag )
      {
         this->TextMode ( ca.ansiFlag ) ;

         //* Write data to log file if specified.                    *
         //* Filename extension determines '.txt' or '.html' output. *
         //* If file exists, new data will be appended.              *
         if ( ca.logFlag != false )
         {
            gString logFile ;
            this->spfRealpath ( ca.logFile, logFile ) ;
            bool isHtml = bool((logFile.find( ".html" )) > ZERO ),
                 exists = this->spfTargetExists ( logFile ) ;
            if ( (this->WriteLogFile ( logFile, isHtml, exists )) != osNONE )
            {
               wcout << L"** Statistical data saved to file: '"
                     << ca.logFile << L"' **" << endl ;
            }
            else
            {
               wcout << L"ERROR: Unable to save statistical data to: '"
                     << ca.logFile << L"'" << endl ;
            }
         }
      }

      //* Interactive Mode - display results of initial analysis, *
      //* and the interact with user for additional analysis.     *
      else
      {
         #if COMM_LINE_ONLY == 0    // if Interactive Mode enabled
         //* If specified, this is the filename or path/filename for log file.*
         if ( ca.logFlag != false )
            this->spfExtractFilename ( this->logName, ca.logFile ) ;

         this->InteractiveMode () ;
   
         //* If errors occurred, report them.     *
         //* Informational messages begin with an *
         //* astrisk, else is error message.      *
         if ( this->emCount > ZERO )
            this->DisplayAppMessages () ;
         #endif   // COMM_LINE_ONLY
      }
   }  // valid user input

   //* Request for version/copyright message *
   else if ( ca.verFlag )
   {
      this->DisplayAppVersion () ;
   }
   //* Explicit or implicit cry for help *
   else if ( ca.helpFlag )
   {
      this->DisplayCommandLineHelp () ;
   }
   //* Display list of error messages (if any) *
   else
   {
      if ( this->emCount > ZERO )
      {
         this->DisplayAppTitle () ;
         this->DisplayAppMessages () ;
      }
   }

}  //* End SrcProf() *

//*************************
//*       TextMode        *
//*************************
//******************************************************************************
//* Display analytical results (or error messages) in text-only mode.          *
//*                                                                            *
//*                                                                            *
//* Input  : ansiOutput : if 'true' embed ANSI color codes into display data   *
//*                       if 'false' display data using terminal default color *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Note on ANSI color output:                                                 *
//* The 'maintainability' of each source file may be indicated using ANSI      *
//* color escape sequences. Some users will think this is cool, and others     *
//* will just find it annoying. For this reason, the data are displayed in     *
//* default colors by default.                                                 *
//*                                                                            *
//* We have three(3) levels of maintainability:                                *
//*   GOOD  - Green text    : \033[0;32m                                       *
//*   FAIR  - Purple text   : \033[0;35m                                       *
//*   POOR  - Red text      : \033[1;31m  (this is BOLD red)                   *
//*                                                                            *
//* Headers and summary data will always be displayed using the terminal       *
//* default color (usually black)                                              *
//*   Default : black text :  \033[0;30m                                       *
//*                                                                            *
//* Background color used is always the default (usually off-white).           *
//*                                                                            *
//* ANSI Colors (partial list):                                                *
//* =========== Foreground  Background   ============ Foreground  Background   *
//* Black       \033[0;30m  \033[0;40m   Dark Gray    \033[1;30m  \033[1;40m   *
//* Red         \033[0;31m  \033[0;41m   Bold Red     \033[1;31m  \033[1;41m   *
//* Green       \033[0;32m  \033[0;42m   Bold Green   \033[1;32m  \033[1;42m   *
//* Yellow      \033[0;33m  \033[0;43m   Bold Yellow  \033[1;33m  \033[1;43m   *
//* Blue        \033[0;34m  \033[0;44m   Bold Blue    \033[1;34m  \033[1;44m   *
//* Purple      \033[0;35m  \033[0;45m   Bold Purple  \033[1;35m  \033[1;45m   *
//* Cyan        \033[0;36m  \033[0;46m   Bold Cyan    \033[1;36m  \033[1;46m   *
//* Light Gray  \033[0;37m  \033[0;47m   White        \033[1;37m  \033[1;47m   *
//******************************************************************************

void SrcProf::TextMode ( bool ansiOutput )
{
   const wchar_t *blackTEXT  = L"\033[0;30m",
                 *greenTEXT  = L"\033[0;32m",
                 *purpleTEXT = L"\033[0;35m",
                 *redTEXT    = L"\033[1;31m" ;
   this->DisplayAppTitle () ;

   if ( this->pd.fileCount > ZERO )
   {
      if ( ansiOutput )
         wcout << blackTEXT ;
      wcout << statHead << endl ;
      for ( UINT i = ZERO ; i < this->pd.fileCount ; i++ )
      {
         if ( ansiOutput )
         {
            if ( this->pd.fileData[i].sfType == sffA )   // assembly-language group
            {
               if ( this->pd.fileData[i].maintIndex >= this->pd.goodpctA )
                  wcout << greenTEXT ;
               else if ( this->pd.fileData[i].maintIndex >= this->pd.fairpctA )
                  wcout << purpleTEXT ;
               else
                  wcout << redTEXT ;
            }
            else if ( this->pd.fileData[i].sfType == sffC ||   // C, and C-like languages
                      this->pd.fileData[i].sfType == sffVB )   // Visual Basic
            {
               if ( this->pd.fileData[i].maintIndex >= this->pd.goodpctC )
                  wcout << greenTEXT ;
               else if ( this->pd.fileData[i].maintIndex >= this->pd.fairpctC )
                  wcout << purpleTEXT ;
               else
                  wcout << redTEXT ;
            }
            else                             // scripting and mark-up languages
            {
               if ( this->pd.fileData[i].maintIndex >= this->pd.goodpctS )
                  wcout << greenTEXT ;
               else if ( this->pd.fileData[i].maintIndex >= this->pd.fairpctS )
                  wcout << purpleTEXT ;
               else
                  wcout << redTEXT ;
            }
         }
         wcout << this->pd.dispData[i].gstr() << endl ;
      }
      if ( ansiOutput )
         wcout << blackTEXT ;
      wcout << statSummary1 << endl ;
      wcout << this->pd.summaryData.gstr() << endl << endl ;

      #if 0    // FOR DEBUG ONLY - CHARACTER AND BYTE COUNTS NOT REPORTED AT THIS TIME
      wcout << L"   * Total Characters: " << this->pd.tFileChars
            << L"  Total Bytes: " << this->pd.tFileBytes << endl ;
      #endif   // FOR DEBUG ONLY - CHARACTER AND BYTE COUNTS NOT REPORTED AT THIS TIME
   }
   else
   {
      if ( ansiOutput )
         wcout << blackTEXT ;
      wcout << L"No Source Files Specified\n" ;
      wcout << L"Type: srcprof --help\n" << endl ;
   }
}  //* End TextMode() *

//*************************
//*      pdAllocate       *
//*************************
//******************************************************************************
//* Allocate dynamic memory and attach it to our 'pd' data member              *
//*                                                                            *
//* Data member affected are: this->pd.fileCount                               *
//*                           this->pd.fileData                                *
//*                           this->pd.dispData                                *
//*                           this->pd.fileCount                               *
//* Any previous allocation for these members is released.                     *
//* Note that this->pd.listStrings and this->pd.listCount ARE NOT affected.    *
//*                                                                            *
//* Input  : itemCount: number of elements to allocate for each array          *
//*                                                                            *
//* Returns: OK  if allocation is successful                                   *
//*              (any existing allocation will be released)                    *
//*          ERR if itemCount parameter == ZERO                                *
//*              (existing allocation will be undisturbed)                     *
//******************************************************************************

short SrcProf::pdAllocate ( UINT itemCount )
{
   short status = ERR ;

   if ( itemCount > ZERO )
   {
      //* Release any current allocation for these members *
      if ( this->pd.fileData != NULL )
         delete [] this->pd.fileData ;
      if ( this->pd.dispData != NULL )
         delete [] this->pd.dispData ;
      this->pd.fileCount = itemCount ;
      this->pd.fileData = new FileData[this->pd.fileCount] ;
      this->pd.dispData = new gString[this->pd.fileCount] ;
      status = OK ;
   }
   return status ;
   
}  //* End pdAllocate() *

//***************************
//*       pdRelease         *
//***************************
//******************************************************************************
//* Release any previous memory allocation attached to our 'pd' data member.   *
//*                                                                            *
//* All dynamic data attached to the following pointers is released.           *
//*                    this->pd.listStrings                                    *
//*                    this->pd.fileData                                       *
//*                    this->pd.dispData                                       *
//* All data members in the 'pd' object are reset EXCEPT:                      *
//*                goodpctC, fairpctC, goodpctA, fairpctA                      *
//*                goodAttr, fairAttr, poorAttr, dfltAttr                      *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void  SrcProf::pdRelease ( void )
{
   if ( this->pd.fileData != NULL )
      delete [] this->pd.fileData ;
   if ( this->pd.dispData != NULL )
      delete [] this->pd.dispData ;
   this->pd.reinit() ;     // reset pointers and counters

}  //* End pdRelease() *

//*************************
//*     tnfAllocate       *
//*************************
//******************************************************************************
//* Allocate dynamic memory and attach it to our 'tn' data member              *
//*                                                                            *
//* Data member affected are: this->tnfPtr                                     *
//*                           this->tnfCount                                   *
//*                                                                            *
//* Input  : itemCount: number of elements to allocate for the array           *
//*                                                                            *
//* Returns: OK  if allocation is successful                                   *
//*              (any existing allocation will be released)                    *
//*          ERR if itemCount parameter == ZERO                                *
//*              (existing allocation will be undisturbed)                     *
//******************************************************************************

short SrcProf::tnfAllocate ( UINT itemCount )
{
   short status = ERR ;

   if ( itemCount > ZERO )
   {  //* Release any existing allocation *
      if ( this->tnfPtr != NULL )
         this->tnfRelease () ;
      this->tnfPtr = new tnFName[itemCount] ;
      this->tnfCount = itemCount ;
   }
   return status ;

}  //* End tnfAllocate() *

//*************************
//*      tnfRelease       *
//*************************
//******************************************************************************
//* Release any previous memory allocation attached to our 'tnfPtr' data member*
//*                                                                            *
//* Members affected are: this-tnfPtr                                          *
//*                       this->tnfCount                                       *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void  SrcProf::tnfRelease ( void )
{
   if ( this->tnfPtr != NULL )
   {
      delete [] this->tnfPtr ;
      this->tnfPtr = NULL ;
      this->tnfCount = ZERO ;
   }

}  //* End tnfRelease() *

//*************************
//*     WriteLogFile      *
//*************************
//******************************************************************************
//* Save currently-displayed statistics to a file.                             *
//*  It is assumed that there is valid data to be written.                     *
//*                                                                            *
//* 1) Write a header line                                                     *
//* 2) Write path of directory where files were found                          *
//* 3) Write a date/timestamp for when data were added to the file             *
//* 4) Write the column headings.                                              *
//* 5) Copy the display data to the file                                       *
//*                                                                            *
//* Input  : fPath : path/filename for target file                             *
//*          htmout: if 'true' output as HTML, if 'false' output as text       *
//*          append: if 'true', append data to existing target file (if any)   *
//*                  if 'false', truncate existing target file (if any)        *
//*                                                                            *
//* Returns: osTEXT if text-format data written successfully                   *
//*          osHTML if HTML-format data written successfully                   *
//*          osNONE if operation cancelled or error writing to file            *
//******************************************************************************

OutSave SrcProf::WriteLogFile ( gString& fPath, bool htmout, bool append )
{
//* Shared output for both text and HTML *
static const char* hdr1 = "# Source Profiler Analytical Data" ;
static const char* hdr2 = "# for directory: '" ;
static const char* hdrDelim = 
   "-  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  - " ;

//* Header, <head></head> information for HTML output. *
static const char* htmlHead = 
"<!DOCTYPE HTML>\n"
"<html lang=\"en\">\n"
"<head>\n"
"<meta charset=\"utf-8\" />\n"
"<title>SrcProf Analytical Data</title>\n"
"<style type=\"text/css\">\n"
".container\n"
"{\n"
"  margin-left:  1em;\n"
"  margin-top: 0;\n"
"  margin-bottom: 1em;\n"
"  margin-right:  auto;\n"
"  max-width: 45.5em;\n"
"  padding: 15px 15px 15px 15px;\n"
"  background-color:#CCFFFF;\n"
"  color:black;\n"
"  font-family:monospace;\n"
"  white-space: pre;\n"
"  font-size:24px;\n"
"}\n"
".tstamp { font-style: italic; }\n"
".chead { font-weight: bold; }\n"
".cgood { color: green; font-weight: bold; }\n"
".cfair { color: magenta; font-weight: bold; }\n"
".cpoor { color: red; font-weight: bold; }\n"
".csumm { font-weight: bold; }\n"
"</style>\n"
"\n"
"\n"
"<!-- Created using SourceProfiler ('srcprof')\n"
"     Copyright (c) 1998-2015 The Software Samurai\n"
"     Released under GNU GPL version 3. -->\n"
"</head>\n"
"\n"
"<body>\n"
"<div class=\"container\">" ;

//* HTML Timestamp template *
static const char* tStampTemplate = 
"<div class=\"tstamp\"> %s\n"
" %s%S'\n"
" # Created: %04hd-%02hd-%02hd @ %02hd:%02hd:%02hd \n"
" # ---------------------------------\n"
"</div>\n" ;

//* HTML end-of-document sequence *
static const char* htmlTail = 
"\n</div>"
"\n</body>"
"\n</html>\n" ;
static const char* htmlEOR    = "<!-- end-of-record -->" ;// end-of-data marker

//* Timestamp for Text-only output *
static const char* ttStampTemplate = 
"%s\n"
"%s%S'\n"
"# Created: %04hd-%02hd-%02hd @ %02hd:%02hd:%02hd \n"
"# ---------------------------------\n" ;

   gString gsOut, srcDir ;          // data formatting
   ofstream ofs ;                   // output stream
   OutSave status = osNONE ;        // return value

   bool tExist = this->spfTargetExists ( fPath ) ;

   //* Get current system date/time.                 *
   //* (if system call fails, zeros will be written) *
   localTime lt ;
   GetLocalTime ( lt ) ;

   //****************************
   //*    Write to HTML file    *
   //****************************
   if ( htmout != false )
   {
      //* If target exists AND we are to append to it, then *
      //* rename the existing file and copy its contents,   *
      //* except for the end sequence, to the new filename. *                       
      bool bkStatus = true,         // status of backup file (if any)
           bkExist = false ;        // 'true' if backup file created
      gString gsBackup ;
      if ( tExist && append )
      {  //* Create backup-file name *
         gsBackup = fPath ;
         gsBackup.append ( L"~" ) ;

         //* Discard any previous backup file *
         if ( this->spfTargetExists ( gsBackup ) )
            bkStatus = this->spfDeleteFile ( gsBackup ) ;

         //* Rename the existing target *
         if ( (bkStatus != false) && 
              ((this->spfRenameFile ( fPath, gsBackup )) == OK ) )
         {
            bkExist = true ;     // remember that we created a backup file
            //* Open the backup and target files *
            ifstream ifs( gsBackup.ustr(), ifstream::in ) ;
            ofs.open( fPath.ustr(), ofstream::out | ofstream::trunc ) ;

            //* Copy backup file contents to target file except EOD marker. *
            if ( ifs.is_open() && ofs.is_open() )
            {
               char  tmp[gsMAXBYTES] ; // UTF-8 line input from file
               bool  done = false ;    // loop control
               while ( ! done )
               {
                  //* Read a source line *
                  ifs.getline ( tmp, gsMAXBYTES, NEWLINE ) ;
      
                  if ( ifs.good() )
                  {
                     if ( (strncmp ( tmp, htmlEOR, gsMAXBYTES)) != ZERO )
                        ofs << tmp << endl ;
                     else
                     {
                        //* Separate new data from existing data *
                        if ( tExist && append )
                           ofs << hdrDelim << endl ;
                        done = true ;
                     }
                  }  // good
                  else                    // end of file (or read error)
                     done = true ;
               }     // while()
            }
            else
            {  //* Error, quit before we make things worse *
               bkStatus = false ;
            }
            //* Close both files *
            if ( ifs.is_open() )
               ifs.close() ;
            if ( ofs.is_open() )
               ofs.close() ;
         }
      }

      //* Write new data to the file *
      if ( bkStatus != false )
      {
         //* Create the file with header in application's invocation directory *
         if ( append != false )
            ofs.open( fPath.ustr(), ofstream::out | ofstream::app ) ;
         else
            ofs.open( fPath.ustr(), ofstream::out | ofstream::trunc ) ;
         if ( ofs.is_open() )             // if input file open
         {
            //* Write the HTML head and open the body section *
            if ( ! append || ! tExist )
               ofs << htmlHead ;

            this->spfExtractFilename ( srcDir, this->cwDir.ustr() ) ;
            gsOut.compose( tStampTemplate, hdr1, hdr2, srcDir.gstr(), 
                           &lt.year, &lt.month, &lt.date,
                           &lt.hours, &lt.minutes, &lt.seconds ) ;
            ofs << gsOut.ustr() << endl ;

            //* Write the column headings *
            gsOut.compose( "<span class=\"chead\">%S\n%S</span>", statHead, statHead2 ) ;
            ofs << gsOut.ustr() << endl ;

            //* Copy caller's data to file (in living color) *
            for ( UINT i = ZERO ; i < this->pd.fileCount ; ++i )
            {
               if ( this->pd.fileData[i].sfType == sffA )   // assembly-language group
               {
                  if ( this->pd.fileData[i].maintIndex >= this->pd.goodpctA )
                     ofs << "<span class=\"cgood\">" ;
                  else if ( this->pd.fileData[i].maintIndex >= this->pd.fairpctA )
                     ofs << "<span class=\"cfair\">" ;
                  else
                     ofs << "<span class=\"cpoor\">" ;
               }
               else if ( this->pd.fileData[i].sfType == sffC ||   // C, and C-like languages
                         this->pd.fileData[i].sfType == sffVB )   // Visual Basic
               {
                  if ( this->pd.fileData[i].maintIndex >= this->pd.goodpctC )
                     ofs << "<span class=\"cgood\">" ;
                  else if ( this->pd.fileData[i].maintIndex >= this->pd.fairpctC )
                     ofs << "<span class=\"cfair\">" ;
                  else
                     ofs << "<span class=\"cpoor\">" ;
               }
               else                             // scripting and mark-up languages
               {
                  if ( this->pd.fileData[i].maintIndex >= this->pd.goodpctS )
                     ofs << "<span class=\"cgood\">" ;
                  else if ( this->pd.fileData[i].maintIndex >= this->pd.fairpctS )
                     ofs << "<span class=\"cfair\">" ;
                  else
                     ofs << "<span class=\"cpoor\">" ;
               }

               //* Display text + end_span *
               ofs << this->pd.dispData[i].ustr() << "</span>" << endl ;
            }

            //* Write the summary data to file *
            ofs << "<span class=\"csumm\">"
                << statSummary1 << "\n"
                << this->pd.summaryData.ustr() << "\n</span>" << endl ;

            //* Write the HTML tail and declare success *
            ofs << "\n\n" << htmlEOR << htmlTail << endl ;
            status = osHTML ;
         }
         ofs.close() ;        // close the file
      }
      //* If we made a backup AND if write was successful, delete the backup.*
      if ( status == osHTML && bkExist )
         this->spfDeleteFile ( gsBackup ) ;
   }


   //****************************
   //* Write to plain text file *
   //****************************
   else
   {
      //* Open the target file and create the data header. *
      if ( append != false )
         ofs.open( fPath.ustr(), ofstream::out | ofstream::app ) ;
      else
         ofs.open( fPath.ustr(), ofstream::out | ofstream::trunc ) ;
      if ( ofs.is_open() )             // if input file open
      {
         //* Separate new data from existing data *
         if ( tExist && append )
         {
            ofs << "\n" << hdrDelim << endl ;
         }
         //* Format and write the timestamp */
         // Programmer's Note: The directory specified in the header is
         // _probably_ the same directory where at least some of the source 
         // files are located; however, this cannot be guaranteed because the 
         // source files could be in many different directories.
         this->spfExtractFilename ( srcDir, this->cwDir.ustr() ) ;
         gsOut.compose( ttStampTemplate, hdr1, hdr2, srcDir.gstr(), 
                        &lt.year, &lt.month, &lt.date,
                        &lt.hours, &lt.minutes, &lt.seconds ) ;
         ofs << gsOut.ustr() << endl ;

         //* Write the column headings *
         gsOut.compose( "%S\n%S", statHead, statHead2 ) ;
         ofs << gsOut.ustr() << endl ;

         //* Copy caller's data to file *
         for ( UINT i = ZERO ; i < this->pd.fileCount ; ++i )
            ofs << this->pd.dispData[i].ustr() << endl ;

         //* Write the summary data to file *
         ofs << statSummary1 << "\n"
             << this->pd.summaryData.ustr() << endl ;

         status = osTEXT ;    // return the good news
      }
      ofs.close() ;        // close the file
   }     // text output

   return status ;

}  //* End WriteLogFile() *

//*************************
//*  GetCommandLineArgs   *
//*************************
//******************************************************************************
//* Capture user's command-line arguments.                                     *
//*                                                                            *
//* All specified source filenames (if any) are verified to exist and to be    *
//* of a supported file type.                                                  *
//*                                                                            *
//*                                                                            *
//* Valid Arguments: (see DisplayCommandLineHelp())                            *
//*                                                                            *
//* Input  : commArgs class object (by reference)                              *
//*                                                                            *
//* Returns: 'true' if all arguments are valid                                 *
//*          'false' if invalid argument(s)                                    *
//******************************************************************************
//* Undocumented options:                                                      *
//*   none at this time                                                        *
//*                                                                            *
//*                                                                            *
//******************************************************************************

bool SrcProf::GetCommandLineArgs ( commArgs& ca )
{
   const char EQUAL = '=' ;
   gString gs,                   // formatted output
           fnPath ;              // decoded path/filename
   bool status = true ;          // return value

   if ( ca.argCount > 1 )
   {
      //* Open a temporary file to hold the filenames to be processed.*
      ofstream ofs( this->lstFile.ustr(), ofstream::out | ofstream::trunc ) ;

      short j = ZERO ;           // for multiple arguments in same token
      bool multiarg = false ;

      for ( short i = 1 ; (i < ca.argCount) || (multiarg != false) ; i++ )
      {
         if ( multiarg != false )   // if additional switches in same argument
            --i ;
         else
            j = ZERO ;

         //* If a command-line switch OR continuing previous switch argument *
         if ( ca.argList[i][j] == DASH || multiarg != false )
         {
            multiarg = false ;
            ++j ;

            if ( ca.argList[i][j] == DASH ) // (step over double dash)
               ++j ;

            char argLetter = ca.argList[i][j] ;

            //**********************
            //**    Text Mode     **
            //**********************
            if ( argLetter == 't' || argLetter == 'T' )
            {
               ca.textFlag = true ;
               if ( (status != false) && ca.argList[i][j + 1] != NULLCHAR )
                  multiarg = true ;
            }

            //**********************
            //** Interactive Mode **
            //**********************
            else if ( argLetter == 'i' || argLetter == 'I' )
            {  // Programmer's Note: If built with text-mode only, the Interactive 
               // flag is accepted, but ignored.
               #if COMM_LINE_ONLY == 0    // Interactive Mode enabled
               ca.textFlag = false ;
               #endif   // COMM_LINE_ONLY
               if ( (status != false) && ca.argList[i][j + 1] != NULLCHAR )
                  multiarg = true ;
            }

            //***********************************************************
            //** Profile all source files in current working directory **
            //***********************************************************
            else if ( argLetter == 'a' || argLetter == 'A' )
            {
               //* Because source is CWD, we assume that it exists *
               DIR* dirPtr = opendir ( this->cwDir.ustr() ) ;
               if ( dirPtr != NULL )
               {
                  deStats   destat, *sptr ;  // entry read from directory file
                  bool  showFile ;           // true if file to be included in list
                  while ( (readdir64_r ( dirPtr, &destat, &sptr )) == ZERO )
                  {
                     // NOTE: This compensates for a bug in the return value of readdir64_r.
                     //       If sptr == NULL, it means end-of-list.
                     if ( sptr == NULL )
                        break ;

                     //* DO NOT include 'current dir' or 'parent' ('..') dir,  *
                     //* but DO included 'hidden' files, i.e. filenames whose  *
                     //* first character is a '.' (PERIOD).                    *
                     showFile = true ;
                     if ( (destat.d_name[ZERO] == PERIOD) &&
                          ((destat.d_name[1] == PERIOD) || (destat.d_name[1] == NULLCHAR)) )
                        showFile = false ;

                     //* If this is a file we may want to store in the list *
                     if ( showFile != false )
                     {
                        this->spfCatPathFilename ( fnPath, this->cwDir, destat.d_name ) ;
                        if ( (this->AddFileToList ( ofs, fnPath )) != false )
                           ++ca.fnCount ;
                     }
                  }  // while()
                  closedir ( dirPtr ) ;      // close the directory
               }     // opendir()

               if ( (status != false) && ca.argList[i][j + 1] != NULLCHAR )
                  multiarg = true ;
            }

            //********************************************************
            //** Profile all source files in specified in list file **
            //********************************************************
            else if ( argLetter == 'l' || argLetter == 'L' )
            {
               if ( (ca.argList[i][j+1] == EQUAL) && (ca.argList[i][j+2] != NULLCHAR) )
               {
                  gs = &ca.argList[i][j+2] ;
                  ifstream ifs( gs.ustr(), ifstream::in ) ;
                  if ( ifs.is_open() )
                  {
                     while ( (this->spfReadLine ( ifs, gs )) != false )
                     {
                        if ( (gs.gschars() > 1) && (*gs.gstr() != HASH) )
                        {
                        if ( (this->AddFileToList ( ofs, gs )) != false )
                           ++ca.fnCount ;
                        }
                     }
                     ifs.close() ;
                  }
                  else
                  {  //* Generate error mesage: list-file-not-found *
                     if ( this->emCount < emMAX )
                     {
                        gs.compose( badFileTemplate[bftListNotFound], &ca.argList[i][j+2] ) ;
                        gs.copy( this->ErrorMsg[this->emCount++], minTERMCOLS ) ;
                     }
                     status = false ;
                  }
               }
               else     // List flag found, but no filename associated with it
               {  //* Generate an error mesage *
                  if ( this->emCount < emMAX )
                  {
                     gs.compose( badFileTemplate[bftListNotSpecified], &argLetter ) ;
                     gs.copy( this->ErrorMsg[this->emCount++], minTERMCOLS ) ;
                  }
                  status = false ;
               }
            }

            //************************************************************
            //** Profile all source files in specified target directory **
            //************************************************************
            else if ( argLetter == 'd' || argLetter == 'D' )
            {
               if ( (ca.argList[i][j+1] == EQUAL) && (ca.argList[i][j+2] != NULLCHAR) )
               {
                  gs = &ca.argList[i][j+2] ;
                  gString trgDir ;
                  if ( ((this->spfRealpath ( gs.ustr(), trgDir )) != false) &&
                       ((this->spfTargetExists ( trgDir, true )) != false) )
                  {
                     DIR* dirPtr = opendir ( trgDir.ustr() ) ;
                     if ( dirPtr != NULL )
                     {
                        deStats   destat, *sptr ;  // entry read from directory file
                        bool  showFile ;           // true if file to be included in list
                        while ( (readdir64_r ( dirPtr, &destat, &sptr )) == ZERO )
                        {
                           if ( sptr == NULL )
                              break ;
      
                           //* DO NOT include 'current dir' or 'parent' ('..') dir,  *
                           //* but DO included 'hidden' files, i.e. filenames whose  *
                           //* first character is a '.' (PERIOD).                    *
                           showFile = true ;
                           if ( (destat.d_name[ZERO] == PERIOD) &&
                                ((destat.d_name[1] == PERIOD) || (destat.d_name[1] == NULLCHAR)) )
                              showFile = false ;
      
                           //* If this is a file we may want to store in the list *
                           if ( showFile != false )
                           {
                              this->spfCatPathFilename ( fnPath, trgDir, destat.d_name ) ;
                              if ( (this->AddFileToList ( ofs, fnPath )) != false )
                                 ++ca.fnCount ;
                           }
                        }  // while()
                        closedir ( dirPtr ) ;      // close the directory
                     }     // opendir()

                     else  // Unable to read target directory (unlikely) *
                     {  //* Generate an error message *
                        if ( this->emCount < emMAX )
                        {
                           gs.compose( badFileTemplate[bftDirNotFound], trgDir.ustr() ) ;
                           gs.copy( this->ErrorMsg[this->emCount++], minTERMCOLS ) ;
                        }
                     }
                  }
                  else
                  {
                     if ( this->emCount < emMAX )
                     {
                        gs.compose( badFileTemplate[bftDirNotFound], gs.ustr() ) ;
                        gs.copy( this->ErrorMsg[this->emCount++], minTERMCOLS ) ;
                     }
                     status = false ;
                  }
               }
               else     // Directory flag found, but no associated dirname
               {  //* Generate an error mesage *
                  if ( this->emCount < emMAX )
                  {
                     gs.compose( badFileTemplate[bftDirNotSpecified], &argLetter ) ;
                     gs.copy( this->ErrorMsg[this->emCount++], minTERMCOLS ) ;
                  }
                  status = false ;
               }
            }

            //*************************
            //** Specify sort option **
            //*************************
            else if ( argLetter == 's' || argLetter == 'S' )
            {
               bool sstatus = true ;
               if ( (ca.argList[i][j+1] == EQUAL) && (ca.argList[i][j+2] != NULLCHAR) )
               {
                  char subArg = ca.argList[i][j+2] ;
                  if ( subArg != NULLCHAR )
                  {
                     switch ( subArg )
                     {
                        case 'e':   case 'E':   ca.sortOpt = soExtension ;    break ;
                        case 'n':   case 'N':   ca.sortOpt = soFilename ;     break ;
                        case 'm':   case 'M':   ca.sortOpt = soMaintain ;     break ;
                        case 's':   case 'S':   ca.sortOpt = soSourcelines ;  break ;
                        default: sstatus = false ; break ;
                     }
                  }
                  else
                     sstatus = false ;
               }
               else
                  sstatus = false ;

               if ( sstatus == false )
               {  //* Generate an error mesage *
                  if ( this->emCount < emMAX )
                  {
                     gs.compose( badFileTemplate[bftBadSort], &argLetter ) ;
                     gs.copy( this->ErrorMsg[this->emCount++], minTERMCOLS ) ;
                  }
                  status = false ;
               }
            }

            //*****************************************
            //** Disable Text-mode ANSI-color output.**
            //*****************************************
            else if ( argLetter == 'c' || argLetter == 'C' )
            {
               ca.ansiFlag = false ;
               if ( (status != false) && ca.argList[i][j + 1] != NULLCHAR )
                  multiarg = true ;
            }

            //**********************************************
            //** Write a copy of output to specified file.**
            //**********************************************
            else if ( argLetter == 'w' || argLetter == 'W' )
            {
               if ( (ca.argList[i][j+1] == EQUAL) && (ca.argList[i][j+2] != NULLCHAR) )
               {
                  gs = &ca.argList[i][j+2] ;
                  gs.copy( ca.logFile, gsMAXBYTES ) ;
                  ca.logFlag = true ;
               }
               else     // Log-file flag found, but no associated filename
               {  //* Generate an error mesage *
                  if ( this->emCount < emMAX )
                  {
                     gs.compose( badFileTemplate[bftLogNotSpecified], &argLetter ) ;
                     gs.copy( this->ErrorMsg[this->emCount++], minTERMCOLS ) ;
                  }
                  status = false ;
               }
            }

            //***********************************
            //** Set maintainability thresholds *
            //***********************************
            else if ( argLetter == 'm' || argLetter == 'M' )
            {
               //* We need four (4) numeric arguments, comma-separated *
               //* with no spaces and in the range 0-100.              *
               if ( (ca.argList[i][j+1] == EQUAL) && (ca.argList[i][j+2] != NULLCHAR) )
               {
                  short ghv = -1, fhv = -1, gav = -1, fav = -1 ;
                  gs = &ca.argList[i][j+2]  ;
                  int conv = swscanf ( gs.gstr(), L"%hd,%hd,%hd,%hd", 
                                       &ghv, &fhv, &gav, &fav ) ;
                  if ( conv == 4 && (ghv >= 0 && ghv <= 100) && 
                                    (fhv >= 0 && fhv <= 100) && 
                                    (gav >= 0 && gav <= 100) && 
                                    (fav >= 0 && fav <= 100) )
                  {
                     ca.goodC = (double)ghv ;
                     ca.fairC = (double)fhv ;
                     ca.goodA = (double)gav ;
                     ca.fairA = (double)fav ;
                     ca.threshFlag = true ;

                     //* Scan for optional 5th and 6th parameters representing *
                     //* goodScript and fairScript, respectively.              *
                     short commaCount = ZERO,
                           commaIndex = ZERO,
                           gsv = -1, fsv = -1 ;
                     do
                     {
                        if ( (commaIndex = gs.find( L',', commaIndex )) > ZERO )
                        {
                           ++commaCount ;
                           ++commaIndex ;
                        }
                        else
                           break ;
                     }
                     while ( commaCount < 4 ) ;
                     if ( commaCount == 4 )
                     {
                        conv = swscanf ( &gs.gstr()[commaIndex], L"%hd,%hd", 
                                         &gsv, &fsv ) ;
                        if ( conv == 2 && (gsv >= 0 && gsv <= 100) && 
                                          (fsv >= 0 && fsv <= 100) )
                        {
                           ca.goodS = (double)gsv ;
                           ca.fairS = (double)fsv ;
                        }
                     }
                  }
                  else
                  {
                     if ( this->emCount < emMAX )
                     {
                        gs.compose( badFileTemplate[bftThreshold] ) ;
                        gs.copy( this->ErrorMsg[this->emCount++], minTERMCOLS ) ;
                     }
                     status = false ;
                  }
               }
               else
               {
                  if ( this->emCount < emMAX )
                  {
                     gs.compose( badFileTemplate[bftThreshold] ) ;
                     gs.copy( this->ErrorMsg[this->emCount++], minTERMCOLS ) ;
                  }
                  status = false ;
               }
            }

            else if ( argLetter == 'h' || argLetter == 'H' )  // request for help
            {  //* A cry for help overrides everything else   *
               //* on the command line except Version.        *
               //* (see below)
               ca.helpFlag = true ;
            }

            else if ( (strncmp ( ca.argList[i], "--version", 10 )) == ZERO )
            {  //* Most Linux console apps respond to a       *
               //* '--version' request in a cannonical manner.*
               //* Overrides everything else on comand line.  *
               this->emCount = ZERO ;
               ca.reset() ;
               ca.helpFlag = ca.verFlag = true ;
               break ;
            }

            else           // invalid switch
            {  //* Generate an error mesage *
               if ( this->emCount < emMAX )
               {
                  gs.compose( badFileTemplate[bftUnrecognized], &argLetter ) ;
                  gs.copy( this->ErrorMsg[this->emCount++], minTERMCOLS ) ;
               }
               status = false ;
            }
         }
         else
         {  //* Interpret argument as a filename. *
            if ( (this->AddFileToList ( ofs, ca.argList[i], true )) != false )
               ++ca.fnCount ;
            else
               status = false ;
         }
      }     // for(;;)

      //*********************************************
      //* If we have an open output file, close it. *
      //*********************************************
      if ( ofs.is_open() )
         ofs.close() ;
      else if ( !ca.verFlag && !ca.helpFlag )
      {  //* Temp file creation error. (this is unlikely) *
         ca.fnCount = ZERO ;   // report that no filenames were specified
         gs = badFileTemplate[bftBadTempfile] ;
         gs.copy( this->ErrorMsg[this->emCount++], minTERMCOLS ) ;
         status = false ;
      }

      if ( ca.helpFlag && !ca.verFlag )
      {  //* Help overrides everything except Version *
         this->emCount = ZERO ;
         ca.reset() ;
         ca.helpFlag = true ;
      }
   }        // if(ca.argCount>1)
   return status ;
   
}  //* End GetCommandLineArgs() *

//*************************
//*    DisplayAppTitle    *
//*************************
//******************************************************************************
//* Display the application's title (Text Mode)                                *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void SrcProf::DisplayAppTitle ( void )
{
   gString gsOut( "\n%S%S%S\n", AppTitle1, AppVersion, AppTitle2 ) ;
   wcout << gsOut.gstr() << AppTitle3 << endl ;

}  //* End DisplayAppTitle() *

//*************************
//*  DisplayAppVersion    *
//*************************
//******************************************************************************
//* Display the application's title, version and copyright info.               *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note: We play tricks with our title strings to make this      *
//* message look canonical. If the strings are changed, this message may get   *
//* ugly. That's why Trix are for kids....                                     *
//******************************************************************************

void SrcProf::DisplayAppVersion ( void )
{
   const wchar_t* freeSoftware = 
   L"License GPLv3+: GNU GPL version 3 or later <http://gnu.org/licenses/gpl.html>\n"
    "This is free software: you are free to change and redistribute it.\n"
    "There is NO WARRANTY, to the extent permitted by law.\n" ;

   gString gsOut( "\nSourceProfiler (srcprof) %S\nCopyright (C) %S\n", 
                  AppVersion, &AppTitle2[4] ) ;
   wcout << gsOut.gstr() ;
   wcout << freeSoftware << endl ;
   
}  //* End DisplayAppVersion() *

//*************************
//*  DisplayAppMessages   *
//*************************
//******************************************************************************
//* Display error/info message list (Text Mode)                                *
//* a) Called on startup to display error messages.                            *
//* b) Called after interactive-mode exit to display informational or error    *
//*    messages.                                                               *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void SrcProf::DisplayAppMessages ( void )
{
   if ( this->emCount > ZERO )
   {
      for ( short i = ZERO ; i < this->emCount ; i++ )
         wcout << this->ErrorMsg[i] << endl ;
      if ( this->ErrorMsg[ZERO][ZERO] == '*' )
         wcout << endl ;
      else
         wcout << L"Type: srcprof --help\n" << endl ;
   }

}  //* End DisplayAppMessages() *

//**************************
//* DisplayCommandLineHelp *
//**************************
//******************************************************************************
//* Display the brief version of command-line help. The NCurses Engine is not  *
//* running at this time, so output is through std::wcout.                     *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void SrcProf::DisplayCommandLineHelp ( void )
{
   this->DisplayAppTitle () ;

            //23456789-123456789-123456789-123456789-123456789-123456789-123456789-123456789-
   wcout << L" Usage  : srcprof [OPTIONS] [FILENAMES]\n"
         << L"          Arguments may be specified in any order.\n"
         << L"          Any reasonable number of source filenames may be specified.\n"
         << L" EXAMPLES: srcprof GuiCode.cpp GuiHeader.hpp Socket.asm\n"
         #if COMM_LINE_ONLY != 0    // if Text Mode only
         << L"           srcprof -cs=n *.[hc]pp\n"
         << L"           srcprof -a -d=config\n"
         #else    // if Interactive Mode enabled
         << L"           srcprof -i *.[hc]pp -s=n\n"
         << L"           srcprof -ta\n"
         << L" -i  Start the application in interactive mode (default).\n"
         << L" -t  Start the application in text-only mode.\n"
         #endif   // COMM_LINE_ONLY
         << L" -a  Profile all source code files in current directory.\n"
         << L" -l  Specify the name of a plain text file containing a list of source\n"
         << L"     files to be analyzed, one filename or path/filename per line.\n"
         << L"        EXAMPLE:  srcprof -l=source_file_list.txt\n"
         << L" -d  Profile all source code files in the specified directory.\n"
         << L"        EXAMPLE:  srcprof -d=./src/videoConvert\n"
         << L" -s  Sort option with sub-option [e | n | m | s] Extension (default),\n"
         << L"     Name, Maintainability index, Source code lines. EXAMPLE: -s=n\n"
         << L" -c  For Text Mode only: Use ANSI color codes for data output.\n"
         << L" -w  For Text Mode only: Write a copy of the output to the specified file.\n"
         << L"     Filename extension indicates whether output is plain-text or HTML.\n"
         << L"        EXAMPLES: srcprof -w=srcprof_log.html   srcprof -w=srcprof_log.txt\n"
         << L" -m  Maintainability thresholds. Specify alternate thresholds for code\n"
         << L"     evaluation. The values (range 0-100%) are interpreted as\n"
         << L"     'good high-level', 'fair high-level', 'good assembler' and\n"
         << L"     'fair assembler'.  EXAMPLE: -m=40,25,55,45  (default values)\n"
         << L" -h --help  Command-line Help\n" 
         << L" --version  Display version number and copyright notice.\n" << endl ;

}  //* End DisplayCommandLineHelp() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

