//******************************************************************************
//* File       : SrcProf.hpp                                                   *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 1998-2015 Mahlon R. Smith, The Software Samurai *
//*                  GNU GPL copyright notice located in SrcProf.hpp           *
//* Date       : 30-Oct-2015                                                   *
//* Version    : (see AppVersion string below)                                 *
//*                                                                            *
//* Description: Definitions and data for Source Profiler, a simple utility    *
//* for analysis of source code files.                                         *
//*                                                                            *
//******************************************************************************
//* Copyright Notice:                                                          *
//* This program is free software: you can redistribute it and/or modify it    *
//* under the terms of the GNU General Public License as published by the Free *
//* Software Foundation, either version 3 of the License, or (at your option)  *
//* any later version.                                                         *
//*                                                                            *
//* This program is distributed in the hope that it will be useful, but        *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
//* for more details.                                                          *
//*                                                                            *
//* You should have received a copy of the GNU General Public License along    *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.            *
//*                                                                            *
//*         Full text of the GPL License may be found in the Texinfo           *
//*         documentation for this program under 'Copyright Notice'.           *
//******************************************************************************
//* Version History (see SrcProf.cpp)                                          *
//*                                                                            *
//******************************************************************************


//* Set this definition to '0' to build for both            *
//* interactive (NcDialog) and text-mode interfaces,        *
//* and build with 'gmake'.                                 *
//* Set this definition to '1' for text-mode-only build.    *
//* and build with 'gmake -f Make_textonly'                 *
#define COMM_LINE_ONLY (1)

//****************
//* Header Files *
//****************
//* All necessary information for:                  *
//* NCurses, NcWindow, NcDialog and gString classes.*
#include "GlobalDef.hpp"
//* Access to file i/o and other system functions.  *
#include "SrcProf_File.hpp"


//***************
//* Definitions *
//***************

//* Minimum terminal window size for this application *
const short minTERMLINES = 25 ;
const short minTERMCOLS  = 80 ;

const wchar_t CR = 0x000D ;         // carriage-return character
const wchar_t STAR = (L'*') ;       // asterisk
const wchar_t SLASH = (L'/') ;      // forward slash
const wchar_t BSLASH = (L'\\') ;    // back slash
const wchar_t SGLQUOTE = (L'\'') ;  // single quote
const wchar_t DBLQUOTE = (L'"') ;   // double quotation mark
const wchar_t HASH = (L'#') ;       // hash mark, pound/number sign
const short emMAX = 16 ;      // maximum number of errors reported


//* Source file types based on filename extension. *
//* Keep synchronized with fnExtensions[].         *
enum sfType : short   {
                        //* sffC family *
                        sftH = ZERO,      // .h
                        sftC,             // .c
                        sftHPP,           // .hpp
                        sftCPP,           // .cpp
                        sftJAVA,          // .java
                        sftJS,            // .js      (Javascript)
                        sftAS,            // .as      (Adobe Flash ActionScript)
                        sftM,             // .m       (Objective-C)
                        sftMM,            // .mm      (Objective-C)
                        sftCS,            // .cs      (C#)
                        sftCSS,           // .css     (Cascading Style Sheet)

                        //* sffA family *
                        sftASM,           // .asm     (Intel x86, etc. assembler)
                        sftINC,           // .inc     (Intel x86, etc. assembler)
                        sftM51,           // .m51     (Intel 8051 assembler)
                        sftS07,           // .s07     (Motorola HC11 assembler)
                        sftS33,           // .s33     (Motorola HC12 assembler)

                        //* sffPY family *
                        sftPY,            // .py      Python
                        sftPYW,           // .pyw     Python Windoze (no console)
                        sftSH,            // .sh      Bourne, Bash, C-shell, Tee-shell, etc.

                        //* sffPE family *
                        sftPL,            // .pl      Perl
                        sftPM,            // .pm      Perl
                        sftT,             // .t       Perl

                        //* sffPH family *
                        sftPHP,           // .php     PHP
                        sftPHTML,         // .phtml   PHP
                        sftPHP3,          // .php3    PHP
                        sftPHP4,          // .php4    PHP
                        sftPHP5,          // .php5    PHP
                        sftPHPS,          // .phps    PHP

                        //* sffRU family *
                        sftRB,            // .rb      Ruby
                        sftRBW,           // .rbw     Ruby

                        //* sffVB family *
                        sftVB,            // .vb      Visual Basic
                        sftVBS,           // .vbs     Visual Basic

                        //* sffSQ family *
                        sftSQL,           // .sql     PL/SQL

                        //* Texinfo family *
                        sftTEXI,          // .texi    Texinfo markup
                        sftTEXINFO,       // .texinfo

                        //* HTML family *
                        sftHTML,          // .html    HTML markup
                        sftHTM,           // .htm
                        sftXHTML,         // .xhtml
                        sftXHT,           // .xht
                        sftXML,           // .xml

                        sftMAX     // number of supported filename extensions
                      } ;

//* Source-file families: members share syntax for source code comments *
enum sfFamily : short { 
                        sffC,    // C: *.h, *.c, *.hpp, *.cpp. *.css
                        sffA,    // ASSEMBLER: *.inc, *.asm, *.M51, *.S07, *.S33
                        sffPY,   // Python: *.py, *.pyw
                        sffPE,   // Perl: *.pl, *.pm, *.t
                        sffPH,   // PHP: *.php, *.phtml, *.php3, *.php3, *.php5, *.phps
                        sffRU,   // Ruby: *.rb, *.rbw
                        sffVB,   // Visual Basic: *.vb, *.vbs
                        sffSQ,   // PL/SQL: *.sql
                        sffTX,   // Texinfo: *.texi*
                        sffHT,   // HTML: *.html *.htm, *.xhtml, *.xht, *.xml
                        sffUnk   // unsupported filename extension
                      } ;

//* Sort options for display of analytical data *
enum SortOption : short { soExtension, // sort by filename extension, sub-sort by filename
                          soFilename,  // sort by filename
                          soMaintain,  // sort by maintainability index
                          soSourcelines// sort by number of source code lines
                        } ;

//* For analytical data saved to a file, file format *
enum OutSave : short { osNONE = ZERO, osTEXT, osHTML } ;

//* Holds captured command-line arguments *
class commArgs
{
   public:
   commArgs ( int argc, char** argv, char** argenv ) :
               argCount(argc), argList(argv), envList(argenv)
   {
      this->reset() ;
   }
   void reset ( void )
   {
      fnCount = ZERO ;
      sortOpt = soExtension ;    // default sort option
      textFlag = logFlag = threshFlag = verFlag = helpFlag = false ;
      ansiFlag = true ;          // Text Mode color output by default
      goodC = fairC = goodA = fairA = goodS = fairS = (double)(-1.0) ;
      #if COMM_LINE_ONLY != 0    // Interactive Mode disabled
      textFlag = true ;
      #endif   // COMM_LINE_ONLY
   }
   short    argCount ;     // command-line arguments
   char**   argList ;      // list of argument strings
   char**   envList ;      // pointer to terminal environment
   char     logFile[gsMAXBYTES] ; // name of output log file (see logFlag)
   USHORT   fnCount ;      // number of path/filenames specified
   double   goodC ;        // threshold for 'good-high-level'
   double   fairC ;        // threshold for 'fair-high-level'
   double   goodA ;        // threshold for 'good-assembler'
   double   fairA ;        // threshold for 'fair-assembler'
   double   goodS ;        // threshold for 'good-script/markup'
   double   fairS ;        // threshold for 'fair-script/markup'
   SortOption sortOpt ;    // sort option
   bool     textFlag ;     // 'true' if text-only, 'false' if interactive dialog
   bool     ansiFlag ;     // 'true' if text-mode ANSI color attributes
   bool     logFlag ;      // 'true' if text-mode write-output-to-log-file
   bool     threshFlag ;   // 'true' if alternate maintainability thresholds specified
   bool     verFlag ;      // /true' if application version request (overrides all others)
   bool     helpFlag ;     // 'true' if command-line help (overrides all except verFlag)
} ;

//* These classes hold the data for profiled source files *
const short maxFNAMECOLS = 30 ;  // filename field column width
class FileData
{
   public:
   FileData ( void )
   {
      lineCount = sourceLines = commentLines = mixedLines = blankLines = ZERO ;
      fileChars = fileBytes = ZERO ;
      sfType = sffUnk ;
   }
   gString  fName ;        //* source filename
   gString  fPath ;        //* full path/filename specification
   UINT     lineCount ;    //* Lines read from source file
   UINT     sourceLines ;  //* Source code lines detected
   UINT     commentLines ; //* Comment lines detected
   UINT     mixedLines ;   //* Source lines that also have comments
   UINT     blankLines ;   //* Whitespace (empty) lines detected
   UINT     fileChars ;    //* Number of characters in file (not reported)
   UINT     fileBytes ;    //* Number of bytes in file      (not reported)
   double   maintIndex ;   //* Maintainability index
   sfFamily sfType ;       //* Source file family
} ;
class ProfData
{
   public:
   ProfData ( void )
   {
      this->reinit() ;           // initialize the data members

      this->fileSort = soExtension ;   // set the default sort option

      //* Set the threshold values for 'good', 'fair' and 'poor' code quality. *
      //* These values are admittedly arbitrary, but they are my personal goals*
      //* and frankly, I wish everyone would be a little more like me. :-)     *
      this->goodpctC = 40.0 ;
      this->fairpctC = 25.0 ;
      this->goodpctA = 55.0 ;
      this->fairpctA = 45.0 ;
      // Note: Currently, scripting languages are calculated as a percentage 
      //       of the target for high-level languages. We may adjust these 
      //       values after analyzing a statistically-significant number of scripts.
      this->goodpctS = this->fairpctC ;
      this->fairpctS = 10.0 ;
      // Note: NCurses Engine is not running at this time, 
      //       so we can't set color attribute values.
      this->goodAttr = this->fairAttr = this->poorAttr = this->dfltAttr = ZERO ;
   }

   void reinit ( void )
   {
      this->fileData = NULL ;
      this->fileCount = ZERO ;
      this->dispData = NULL ;
      this->tLineCount = this->tSourceLines = this->tCommentLines 
                       = this->tMixedLines = this->tBlankLines = ZERO ;
      this->tFileChars = this->tFileBytes = ZERO ;
      this->tMaintIndex = 0.0 ;
      this->summaryData.clear() ;
   }

   FileData*   fileData ;  //* Data for individual source files
   UINT        fileCount ; //* Total number of source files to be profiled
   SortOption  fileSort ;  //* Sorting option for fileData array

   gString*    dispData ;  //* Display data for each source file profiled
   attr_t      goodAttr ;  //* Color attribute for 'good' results
   attr_t      fairAttr ;  //* Color attribute for 'fair' results
   attr_t      poorAttr ;  //* Color attribute for 'poor' results
   attr_t      dfltAttr ;  //* Default color attribute (summary display)
   //* 'C' language family *
   double      goodpctC ;  //* 'good' results if >= this percentage
   double      fairpctC ;  //* 'fair' results if >= this percentage
                           //* 'poor' results if < this percentage
   //* Assembly language family *
   double      goodpctA ;  //* 'good' results if >= this percentage
   double      fairpctA ;  //* 'fair' results if >= this percentage
                           //* 'poor' results if < this percentage
   //* Scripting language family *
   double      goodpctS ;  //* 'good' results if >= this percentage
   double      fairpctS ;  //* 'fair' results if >= this percentage
                           //* 'poor' results if < this percentage

   //* Accumulators for all source files profiled *
   UINT     tLineCount ;   //* Line Count
   UINT     tSourceLines ; //* Source Lines
   UINT     tCommentLines ;//* Comment Lines
   UINT     tMixedLines ;  //* Mixed Lines (source code and comments)
   UINT     tBlankLines ;  //* Empty White-space Lines
   UINT     tFileChars ;   //* Character count (not reported)
   UINT     tFileBytes ;   //* Byte count      (not reported)
   double   tMaintIndex ;  //* Maintainability Index
   gString  summaryData ;  //* Display data for accumulated totals
} ;


//***********************************************
//* Constant Data:                              *
//***********************************************

//* Application version string. Keep it current! *
const wchar_t* const AppVersion = L"0.0.18" ;

//* Application title *
const wchar_t* const AppTitle1 = L"Source Profiler v:" ;
const wchar_t* const AppTitle2 = L" (c)1998-2015 The Software Samurai" ;
const wchar_t* const AppTitle3 = // (this string == minTERMCOLS)
L"------------------------------------------------------------------------------" ;

//* Column headings for statistics display *
const wchar_t* const statHead = 
   L"                               TOTAL  SOURCE  COMMENT SRC AND  BLANK MAINTAIN \n"
    "         SOURCE FILES          LINES   LINES    LINES COMMENT  LINES    INDEX " ;
const wchar_t* const statHead2 = 
   L"------------------------------ -----  ------  ------- -------  -----  -------" ;
const char* const statSummary1 = 
   "  * Summary Data       *      ------  ------  ------- -------  -----  -------" ;
const wchar_t* const statSummaryT =
   L"  * Scanned Files:%4u *        " ;

//* Templates for creating temporary files *
const wchar_t* const spSelFName = L"spSEL%02hd~" ;
const wchar_t* const spSelFormat = L"0x%08X " ;

//* Default filenames for saving user's statistical data *
const char* const spSaveTextStats = "srcprof_log.txt" ;
const char* const spSaveHtmlStats = "srcprof_log.html" ;

//* Date-time stamp for output files *
const wchar_t* const spDateTimeHdr = 
   L"// Created: %04hu-%02hu-%02hu @ %02hu:%02hu:%02hu" ;
const char* const spDateTimeHdrB = 
   "// ------------------------------" ;

//* Supported source code filename extensions.*
//* Keep synchronized with enum sfType.       *
const wchar_t* const fnExtensions[sftMAX] = 
{
   L".h", L".c", L".hpp", L".cpp", L".java", L".js", L".as",   // sffC family
   L".m", L".mm", L".cs", L".css",
   L".asm", L".inc", L".m51", L".s07", L".s33",                // sffA family
   L".py", L".pyw", L".sh",                                    // sffPY family (incl. no-extension scripts)
   L".pl", L".pm", L".t",                                      // sffPE family
   L".php", L".phtml", L".php3", L".php4", L".php5", L"phps",  // sffPH family
   L".rb", L".rbw",                                            // sffRU family
   L".vb", L".vbs",                                            // sffVB family
   L".sql",                                                    // sffSQ family
   L".texi", L".texinfo",                                      // sffTX family
   L".html", L".htm", L".xhtml", L".xht", L".xml",             // sffHT family
} ;

//* Error reporting templates and enum for accessing them *
const wchar_t* const badFileTemplate[] = 
{  //123456789+123456789+123456789+123456789+123456789+123456789+123456789+123456789|
   L"Source file '%s' not found, not a regular file or no read access.",      // bftNotFound
   L"'%S' has an unsupported filename extension.",                            // bftUnsupported
   L"'-%c' is an unrecognized option.",                                       // bftUnrecognized
   L"'-%c' option requires name of list file.",                               // bftListNotSpecified
   L"'-%c' option requires name of source directory.",                        // bftDirNotSpecified
   L"List file '%s' was not found or no read access.",                        // bftListNotFound
   L"List file '%s' contains no valid filenames or has format errors.",       // bftListErrors
   L"Specified source directory '%s' was not found.",                         // bftDirNotFound
   L"Specified source directory '%s' contains no valid source files.",        // bftEmptyDir
   L"'-%c' option requires name of log file.",                                // bftLogNotSpecified
   L"'-%c' option requires a sub-option of [e | n | p | s].",                 // bftBadSort
   L"'-M' option requires four (4) arguments, range: 0 to 100.",              // bftThreshold
   L"ERROR! Unable to initialize NCurses Engine.",                            // bftiEngineA
   L"       ncurses v5.9 or higher must be installed on your system.",        // bftiEngineB
   L"ERROR! The terminal's default locale: '%s'",                             // bftiLocaleA
   L"        may not support UTF-8 encoded text. For a complete listing",     // bftiLocaleB
   L"        of locales supported by your terminal, type: locale -a",         // bftiLocaleC
   L"ERROR! Terminal window must be at least %02hd lines x %02hd columns.",   // bftiTermsize
   L"ERROR! Unable to create temporary file(s).",                             // bftBadTempfile
} ;
enum errorType : short  // (see badFileTemplate array above)
{
   bftNotFound, bftUnsupported, bftUnrecognized,   // general user errors
   bftListNotSpecified, bftDirNotSpecified, 
   bftListNotFound, bftListErrors, bftDirNotFound, bftEmptyDir, bftLogNotSpecified, 
   bftBadSort, bftThreshold,
   bftEngineA, bftEngineB,                        // system or environment errors
   bftLocaleA, bftLocaleB, bftLocaleC,
   bftTermsize, bftBadTempfile
} ;


//********************************
//* Application class definition *
//********************************
class SrcProf
{
   public:
   virtual ~SrcProf () ;            // destructor
   SrcProf ( commArgs& ca ) ;       // constructor
   bool  ValidCommArgs ( void )     // returns 'true' if user's data valid
   { return this->validArgs ; }


   private:
   //*** Methods for start-up, memory management ***
   //***-----------------------------------------***
   //* Interpret user's command options and gather specified source-filenames. *
   bool  GetCommandLineArgs ( commArgs& ca ) ;
   //* Display the application's title (Text Mode)                             *
   void  DisplayAppTitle ( void ) ;
   //* Display the application's title, version and copyright info.            *
   void  DisplayAppVersion ( void ) ;
   //* Display command-line options *
   void  DisplayCommandLineHelp ( void ) ;
   //* Display error/info message list (Text Mode)                             *
   void  DisplayAppMessages ( void ) ;
   //* Allocate dynamic memory and attach it to our 'pd' data member           *
   short pdAllocate ( UINT itemCount ) ;
   //* Release previously-allocated memory attached to our 'pd' data member.   *
   void  pdRelease ( void ) ;
   //* Allocate dynamic memory and attach it to our 'tnfPtr' data member.      *
   short tnfAllocate ( UINT itemCount ) ;
   //* Release previously-allocated memory attached to our 'tnfPtr' data member*
   void  tnfRelease ( void ) ;
   //* Display analytical results in text-only mode.                           *
   void  TextMode ( bool ansiOutput ) ;
   //* Save currently-displayed statistics to a file.                          *
   OutSave WriteLogFile ( gString& fName, bool htmout, bool append ) ;

   //*** Methods for Interactive Mode            ***
   //***-----------------------------------------***
   //* Start the NCurses Engine.                                               *
   void  InteractiveMode ( void ) ;
   //* Open the interactive dialog window and display analytical results.      *
   void  imOpenDialog ( void ) ;
   //* Interactive selection of files to be analyzed                           *
   int   imSelect ( NcDialog* dp, ssetData& ssd, short cIndex ) ;
   //* Create, format and display a list of interesting files in CWD.          *
   void  imSelectRead ( NcDialog* dp, ssetData& ssd, short cIndex ) ;
   //* Save 'selected' source code filenames to a temporary file.              *
   short imSelectSave ( ssetData& ssd, UINT selectCount ) ;
   //* Analyze the source code files in the list file                          *
   short imSelectRefresh ( ssetData& ssd ) ;
   //* Format the statistical data for display in the dialog window.           *
   void  imFormatStatData ( ssetData& ssd ) ;
   //* Sort directory entries according to whether user has selected them.     *
   void  imSortDirEntriesBySelected ( ssetData& ssd ) ;
   //* Create a list of source file names in CWD for display in dialog window, *
   UINT  imReadCWD ( ssetData& ssd ) ;
   //* Allocate dynamic memory for display data && attach it to ssetData object*
   short imAllocate ( ssetData& ssd, UINT itemCount ) ;
   //* Release dynamic memory allocation attached to ssetData class object.    *
   void  imRelease ( ssetData& ssd ) ;
   //* Save currently-displayed statistics to a file.                          *
   OutSave imSave2File ( const winPos& ctr ) ;
   //* Allow user to select the sort option for display of statistical data.   *
   bool  imSortOption ( ssetData& ssd, winPos& ctr ) ;
   //* Display application help in a child dialog.                             *
   void  HelpDialog ( short ctrY, short ctrX, 
                      const char* ncdVersion, const char* nclVersion ) ;
   //* Display critical data - for debugging only! (requires >= 132 columns)   *
   NcDialog* imDebug ( const ssetData& ssd, 
                       bool stayOpen = false, const char* msg = NULL ) ;
   //* Display elements of the tnfPtr array - for debugging only               *
   NcDialog* imDump_tnfPtr ( bool stayOpen = false, const char* msg = NULL ) ;
   //* Display elements of the fileData array - for debugging only             *
   NcDialog* imDump_fileData ( bool stayOpen = false, const char* msg = NULL ) ;

   //*** Methods to create and read list files   ***
   //***-----------------------------------------***
   //* If a request to generate a list of source files to process              *
   short CreateFileList ( const char* dirPath ) ;
   short CreateFileList ( const gString& dPath, int scanCount = -1, bool append = false ) ;
   //* Add one path/filename to the list of files to be analyzed.              *
   bool  AddFileToList ( ofstream& ofs, const gString& srcFile, bool errorMsg = false ) ;
   bool  AddFileToList ( ofstream& ofs, const char* srcFile, bool errorMsg = false ) ;
   //* Check for existence, accessibility of each file in the source file list.*
   UINT  ValidateFileList ( void ) ;
   //* Create a date/timestamp for the temporary file *
   bool  GetLocalTime ( localTime& ft ) ;

   //*** Methods for analyzing source code files ***
   //***-----------------------------------------***
   //* Scan all files in the list, collecting statistical data on each.        *
   bool  AnalyzeFileList ( void ) ;
   //* Scan the specified file and collect statistical data on it.             *
   bool  AnalyzeFile ( FileData& fd ) ;
   bool  af_sffC ( FileData& fd ) ;
   bool  af_sffA ( FileData& fd ) ;
   bool  af_sffPY ( FileData& fd ) ;
   bool  af_sffPE ( FileData& fd ) ;
   bool  af_sffPH ( FileData& fd ) ;
   bool  af_sffRU ( FileData& fd ) ;
   bool  af_sffVB ( FileData& fd ) ;
   bool  af_sffSQ ( FileData& fd ) ;
   bool  af_sffTX ( FileData& fd ) ;
   bool  af_sffHT ( FileData& fd ) ;
   //* Scan through the source line until non-whitespace found (or EOL).       *
   void  SkipWhiteSpace ( const wchar_t* inp, USHORT& index ) ;
   //* A C-style comment is open, scan source line until closed (or EOL).      *
   bool  Scan4CommentClose ( const wchar_t* inp, USHORT& index ) ;
   //* Scan though the source statement on the line until comment (or EOL).    *
   bool  Scan2SourceClose ( const wchar_t* inp, USHORT& index ) ;
   //* Scan though source statement on the line until end of constant (or EOL).*
   bool  Scan2ConstClose ( const wchar_t* inp, USHORT& index ) ;
   //* Extract source filename from the path/filename and format for output.   *
   void  FormatFilename ( gString& gsName, const gString& fPath ) ;
   //* Determine which parser to use in analyzing the source code file.        *
   sfFamily GetFileFamily ( const char* fPath ) ;
   //* Convert the analytical data to display data.                            *
   void  CreateDisplayStrings ( void ) ;
   //* Sort the array of FileData-class objects.                               *
   void  SortFileData ( void ) ;
   void  Compare_Extension ( FileData* fdPtr, UINT fdCount ) ;
   void  Compare_Filename ( FileData* fdPtr, UINT fdCount ) ;
   void  Compare_Maintainability ( FileData* fdPtr, UINT fdCount ) ;
   void  Compare_Sourcelines ( FileData* fdPtr, UINT fdCount ) ;

   //*** Methods for file system and dir entries ***
   //***-----------------------------------------***
   //* Create a list of files-of-interest found in the specified directory.    *
   UINT  spfReadDirectory ( const gString& dPath, bool inclDir = false ) ;
   //* Sort entries according to file type, then by filename.                  *
   void  spfSortDirEntriesByName ( tnFName* tPtr, UINT tCnt ) ;
   //* Sort entries by filename extension                                      *
   void  spfSortDirEntriesByExtension ( UINT fCount ) ;
   //* Get the path for the current working directory (CWD).                   *
   short spfGetCWD ( gString& dPath ) ;
   //* Create a unique path/directory name for application's temporary files.  *
   bool  spfCreateTemppath ( void ) ;
   //* Create a unique path/filename for a temporary file.                     *
   bool  spfCreateTempname ( gString& tmpPath ) ;
   //* Change the current working directory to parent of current CWD.          *
   short spfCdParent ( void ) ;
   //* Change the current working directory to specified subdirectory of CWD.  *
   short spfCdChild ( const char* dirName ) ; 
   //* Concatenate a path string with a filename string.                       *
   short spfCatPathFilename ( gString& pgs, const gString& wPath, const char* uFile ) ;
   short spfCatPathFilename ( gString& pgs, const gString& wPath, const wchar_t* wFile ) ;
   //* Extract the path from the path/filename provided.                       *
   void  spfExtractPathname ( gString& ePath, const gString& fPath ) ;
   //* Extract the filename from the path/filename provided.                   *
   void  spfExtractFilename ( gString& eName, const gString& fPath ) ;
   void  spfExtractFilename ( gString& gsName, const char* fPath ) ;
   //* Extract the filename extension (including the '.') from  path/filename  *
   void  spfExtractFileExtension ( gString& eExt, const gString& fPath ) ;
   void  spfExtractFileExtension ( gString& gsExt, const char* fName ) ;
   //* Convert specified path/filename to full path/filename.                  *
   bool  spfRealpath ( const char* rawPath, gString& realPath) ;
   //* 'stat' the specified path/filename to see if it exists.                 *
   bool  spfTargetExists ( const gString& fPath, bool isDir = false ) ;
   //* Convert GNU/UNIX epoch time code to local time format.                  *
   void  spfFormatEpochTime ( ULONG eTime, localTime& ftTime ) ;
   //* Delete a file.                                                          *
   bool  spfDeleteFile ( const gString& trgPath ) ;
   //* Rename a file.                                                          *
   short spfRenameFile ( const gString& srcPath, const gString& trgPath ) ;
   //* Read one line of text from the specified input stream.                  *
   bool  spfReadLine ( ifstream& ifs, gString& gs ) ;
   //* Delete the application's temporary files.                               *
   void  spfDeleteTempFiles ( bool all ) ;

   //**********
   //** DATA **
   //**********
   short    dRows ;        //* Interactive-Mode display rows
   short    dCols ;        //* Interactive-mode display columns
   ProfData pd ;           //* Analysis statistics and display strings
   tnFName* tnfPtr ;       //* Array of file descriptions from dir file
   UINT     tnfCount ;     //* Number of elements in 'tn' array
   gString  cwDir ;        //* Current-working-directory on start-up
   gString  tmpDir ;       //* Path to any temporary files
   gString  lstFile ;      //* Path/filename of list file
   gString  logName ;      //* Filename of output log
   wchar_t  ErrorMsg[emMAX][gsMAXCHARS] ; //* Formatted error message strings
   short    emCount ;      //* Number of error messages in queue
   bool     validArgs ;    //* 'true' if valid user input
} ;


